/*--------------------------------------------------------------------------+
$Id: StateflowBuilderTest.java 26285 2010-02-18 11:22:54Z juergens $
|                                                                          |
| Copyright 2005-2010 Technische Universitaet Muenchen                     |
|                                                                          |
| Licensed under the Apache License, Version 2.0 (the "License");          |
| you may not use this file except in compliance with the License.         |
| You may obtain a copy of the License at                                  |
|                                                                          |
|    http://www.apache.org/licenses/LICENSE-2.0                            |
|                                                                          |
| Unless required by applicable law or agreed to in writing, software      |
| distributed under the License is distributed on an "AS IS" BASIS,        |
| WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. |
| See the License for the specific language governing permissions and      |
| limitations under the License.                                           |
+--------------------------------------------------------------------------*/
package edu.tum.cs.simulink.builder;

import java.io.FileNotFoundException;
import java.util.HashSet;

import edu.tum.cs.commons.collections.CollectionUtils;
import edu.tum.cs.simulink.model.SimulinkConstants;
import edu.tum.cs.simulink.model.SimulinkModel;
import edu.tum.cs.simulink.model.stateflow.StateflowBlock;
import edu.tum.cs.simulink.model.stateflow.StateflowChart;
import edu.tum.cs.simulink.model.stateflow.StateflowData;
import edu.tum.cs.simulink.model.stateflow.StateflowEvent;
import edu.tum.cs.simulink.model.stateflow.StateflowJunction;
import edu.tum.cs.simulink.model.stateflow.StateflowMachine;
import edu.tum.cs.simulink.model.stateflow.StateflowModelTest;
import edu.tum.cs.simulink.model.stateflow.StateflowNodeBase;
import edu.tum.cs.simulink.model.stateflow.StateflowState;
import edu.tum.cs.simulink.model.stateflow.StateflowTarget;
import edu.tum.cs.simulink.model.stateflow.StateflowTransition;
import edu.tum.cs.simulink.testutils.SimulinkTestBase;

/**
 * Test class for the Stateflow builder. This tests if the model is
 * <em>built</em> properly. Correct implementation of the model itself, e.g.
 * deep cloning, element removal, is tested in {@link StateflowModelTest}.
 * <p>
 * The test specified here are far from complete but meant as basis to
 * conveniently add more tests.
 * 
 * @author deissenb
 * @author $Author: juergens $
 * @version $Rev: 26285 $
 * @levd.rating GREEN Hash: CFBB43C9E51762BBEE65E89846DB710B
 */
public class StateflowBuilderTest extends SimulinkTestBase {

	/** Block under test. */
	private StateflowBlock block;

	/** Load model and set block. */
	@Override
	public void setUp() throws FileNotFoundException,
			SimulinkModelBuildingException {
		SimulinkModel model = loadModel("model02.mdl");
		block = (StateflowBlock) model.getBlock("model02/Stateflow");
	}

	/** Test if {@link StateflowMachine} was built correctly. */
	public void testMachine() {
		assertSame(block.getChart().getMachine(), block.getModel()
				.getStateflowMachine());

		StateflowMachine machine = block.getChart().getMachine();
		assertEquals(1, machine.getCharts().size());

		StateflowChart chart = machine.getCharts().iterator().next();
		assertEquals("Stateflow", chart.getName());

		assertEquals(1, machine.getTargets().size());
		StateflowTarget target = machine.getTargets().iterator().next();
		assertEquals("85", target.getStateflowId());
		assertEquals("sfun", target.getParameter(SimulinkConstants.PARAM_name));
		assertEquals("Default Simulink S-Function Target.", target
				.getParameter("description"));
	}

	/** Test if {@link StateflowChart} was built correctly. */
	public void testChart() {
		StateflowChart chart = block.getChart();
		assertEquals("Stateflow", chart.getName());
		assertEquals(2, chart.getNodes().size());

		StateflowState assemblyPoint = (StateflowState) getStateflowNode("12",
				chart);
		assertEquals("AssemblyPoint", assemblyPoint.getLabel());

		StateflowState hotelStateflow = (StateflowState) getStateflowNode("7",
				chart);
		assertEquals("HotelStateflow", hotelStateflow.getLabel());

		HashSet<String> names = new HashSet<String>();
		for (StateflowEvent event : chart.getEvents()) {
			names.add(event.getName());
		}
		assertTrue(names.containsAll(CollectionUtils.asHashSet("Allclear",
				"checkin", "clk", "E", "Firealarm")));

		names.clear();

		for (StateflowData data : chart.getData()) {
			names.add(data.getName());
		}
		assertTrue(names.containsAll(CollectionUtils.asHashSet("b", "c1",
				"guest", "rich", "smoking")));
	}

	/** Test if {@link StateflowState}s were built correctly. */
	public void testState() {
		StateflowChart chart = block.getChart();
		StateflowState assemblyPoint = (StateflowState) getStateflowNode("12",
				chart);
		assertEquals("AssemblyPoint", assemblyPoint.getLabel());
		assertTrue(assemblyPoint.getNodes().isEmpty());

		StateflowState hotelStateflow = (StateflowState) getStateflowNode("7",
				chart);
		assertEquals("HotelStateflow", hotelStateflow.getLabel());

		assertEquals(13, hotelStateflow.getNodes().size());

		// Junctions
		StateflowJunction junction = (StateflowJunction) getStateflowNode("21",
				hotelStateflow);
		assertEquals("HISTORY_JUNCTION", junction.getType());

		junction = (StateflowJunction) getStateflowNode("18", hotelStateflow);
		assertEquals("CONNECTIVE_JUNCTION", junction.getType());

		junction = (StateflowJunction) getStateflowNode("19", hotelStateflow);
		assertEquals("CONNECTIVE_JUNCTION", junction.getType());

		junction = (StateflowJunction) getStateflowNode("20", hotelStateflow);
		assertEquals("CONNECTIVE_JUNCTION", junction.getType());

		// States
		StateflowState state = (StateflowState) getStateflowNode("9",
				hotelStateflow);
		assertEquals("FamilySuite", state.getLabel());
		assertEquals("OR_STATE", state
				.getParameter(SimulinkConstants.PARAM_type));

		state = (StateflowState) getStateflowNode("4", hotelStateflow);
		assertEquals("SmokingRoom", state.getLabel());

		state = (StateflowState) getStateflowNode("3", hotelStateflow);
		assertEquals(
				"NonSmokingRoom\\n entry: b=unpack();\\n during: b=callhome();\\n exit: b=pack();",
				state.getLabel());

		state = (StateflowState) getStateflowNode("16", hotelStateflow);
		assertEquals("y=unpack", state.getLabel());
		assertEquals("FUNC_STATE", state
				.getParameter(SimulinkConstants.PARAM_type));

		state = (StateflowState) getStateflowNode("14", hotelStateflow);
		assertEquals("y=pack", state.getLabel());
		assertEquals("FUNC_STATE", state
				.getParameter(SimulinkConstants.PARAM_type));
		assertEquals("1", state.getParameter("truthTable.isTruthTable"));
	}

	/** Test if {@link StateflowTransition}s were built correctly. */
	public void testTransition() {
		StateflowChart chart = block.getChart();

		// AssemblyPoint
		StateflowState assemblyPoint = (StateflowState) getStateflowNode("12",
				chart);
		assertEquals(1, assemblyPoint.getInTransitions().size());
		StateflowTransition inTransition = assemblyPoint.getInTransitions()
				.iterator().next();
		assertEquals("7", inTransition.getSrc().getStateflowId());
		assertEquals("12", inTransition.getDst().getStateflowId());
		assertEquals("Firealarm", inTransition.getLabel());

		assertEquals(1, assemblyPoint.getOutTransitions().size());
		StateflowTransition outTransition = assemblyPoint.getOutTransitions()
				.iterator().next();
		assertEquals("12", outTransition.getSrc().getStateflowId());
		assertEquals("7", outTransition.getDst().getStateflowId());
		assertEquals("Allclear", outTransition.getLabel());

		// Hotel Stateflow
		StateflowState hotelStateflow = (StateflowState) getStateflowNode("7",
				chart);
		assertEquals(2, hotelStateflow.getInTransitions().size());

		StateflowTransition transition = getTransitionWithSrc(hotelStateflow,
				null);
		assertNotNull(transition);

		transition = getTransitionWithSrc(hotelStateflow, assemblyPoint);
		assertEquals("Allclear", transition.getLabel());

		assertEquals(1, hotelStateflow.getOutTransitions().size());
		transition = getTransitionWithDst(hotelStateflow, assemblyPoint);
		assertEquals("Firealarm", transition.getLabel());

		// Transitions to Junctions
		StateflowState wait = (StateflowState) getStateflowNode("17",
				hotelStateflow);
		assertEquals(1, wait.getOutTransitions().size());
		transition = wait.getOutTransitions().iterator().next();
		assertEquals("checkin", transition.getLabel());
		StateflowJunction junction = (StateflowJunction) transition.getDst();
		assertEquals("19", junction.getStateflowId());

		assertSame(transition, junction.getInTransitions().iterator().next());

		assertEquals(2, junction.getOutTransitions().size());
	}

	/** Get the outgoing transition of a node with a specified destination. */
	private StateflowTransition getTransitionWithDst(StateflowNodeBase node,
			StateflowNodeBase dst) {
		for (StateflowTransition transition : node.getOutTransitions()) {
			if (transition.getDst() == dst) {
				return transition;
			}
		}
		return null;
	}

	/** Get the incoming transition of a node with a specified source. */
	private StateflowTransition getTransitionWithSrc(StateflowNodeBase node,
			StateflowNodeBase src) {
		for (StateflowTransition transition : node.getInTransitions()) {
			if (transition.getSrc() == src) {
				return transition;
			}
		}
		return null;
	}

}