/*
 *  @(#)DefaultIssueState.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.pmti.v1.defimpl;

import net.sourceforge.groboutils.pmti.v1.IIssueState;
import net.sourceforge.groboutils.pmti.v1.IAttributeSet;

 

/**
 * Describes the state of an Issue.  Issue states may have additional
 * information associated with them, and as such uses the Attribute methodology
 * in the same way an Issue does.  All implementations of <tt>IIssueState</tt>
 * must be immutable, unless they are also instances of
 * <tt>IEditableIssueState</tt>.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:51:58 $
 * @since      July 12, 2002
 */
public class DefaultIssueState implements IIssueState
{
    private String name;
    private String desc;
    private boolean open;
    private IAttributeSet attribs;
    
    public DefaultIssueState( String n, String d, boolean o, IAttributeSet s )
    {
        if (n == null || d == null || s == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        
        this.name = n;
        this.desc = d;
        this.open = o;
        this.attribs = s;
    }
    
    
    /**
     * Returns the short name of the state.
     *
     * @return the short readable name of the state.
     */
    public String getName()
    {
        return this.name;
    }
    
    
    /**
     * Retrieves a long, human-readable, description of the state.
     *
     * @return the state's description.
     */
    public String getDescription()
    {
        return this.desc;
    }
    
    
    /**
     * A broad category for the state - it means that the issue has not been
     * resolved yet, and the code is still open for changes based on this
     * issue.
     * <P>
     * <tt>isOpen()</tt> must always return the opposite of
     * <tt>isClosed()</tt>; that is, the following code:
     * <PRE>
     *      isOpen() == !isClosed()
     * </PRE>
     * <i>must always</i> evaluate to <tt>true</tt>.
     */
    public boolean isOpen()
    {
        return this.open;
    }
    
    
    /**
     * A broad category for the state - it means that the issue has been
     * resolved, and the code is no longer open for changes based on this
     * issue.
     * <P>
     * <tt>isClosed()</tt> must always return the opposite of
     * <tt>isOpen()</tt>; that is, the following code:
     * <PRE>
     *      isOpen() == !isClosed()
     * </PRE>
     * <i>must always</i> evaluate to <tt>true</tt>.
     */
    public boolean isClosed()
    {
        return !isOpen();
    }
    
    
    /**
     * Returns a list of all attributes associated with this state.  All
     * states of a particular type should have the same set of issues.  If
     * the problem tracker does not have attributes associated with an issue
     * state, then this must still return a non-<tt>null</tt>, but the set
     * will be empty.
     *
     * @return the set of tracker-specific and issue type-specific attributes
     *      and values associated with this issue.  Can never return
     *      <tt>null</tt>.
     */
    public IAttributeSet getAttributes()
    {
        return this.attribs;
    }
}

