/* ========================================================================= *
 *                                                                           *
 *                               OpenMesh                                    *
 *           Copyright (c) 2001-2025, RWTH-Aachen University                 *
 *           Department of Computer Graphics and Multimedia                  *
 *                          All rights reserved.                             *
 *                            www.openmesh.org                               *
 *                                                                           *
 *---------------------------------------------------------------------------*
 * This file is part of OpenMesh.                                            *
 *---------------------------------------------------------------------------*
 *                                                                           *
 * Redistribution and use in source and binary forms, with or without        *
 * modification, are permitted provided that the following conditions        *
 * are met:                                                                  *
 *                                                                           *
 * 1. Redistributions of source code must retain the above copyright notice, *
 *    this list of conditions and the following disclaimer.                  *
 *                                                                           *
 * 2. Redistributions in binary form must reproduce the above copyright      *
 *    notice, this list of conditions and the following disclaimer in the    *
 *    documentation and/or other materials provided with the distribution.   *
 *                                                                           *
 * 3. Neither the name of the copyright holder nor the names of its          *
 *    contributors may be used to endorse or promote products derived from   *
 *    this software without specific prior written permission.               *
 *                                                                           *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS       *
 * "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED *
 * TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A           *
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER *
 * OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,  *
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,       *
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR        *
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF    *
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING      *
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS        *
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.              *
 *                                                                           *
 * ========================================================================= */

//=============================================================================
//
//  Helper Functions for generating a random number between 0.0 and 1.0 with
//  a garantueed resolution
//
//=============================================================================


//== INCLUDES =================================================================


#include <OpenMesh/Core/Utils/RandomNumberGenerator.hh>


//== NAMESPACES ===============================================================


namespace OpenMesh {


//== IMPLEMENTATION ===========================================================

RandomNumberGenerator::RandomNumberGenerator(const size_t _resolution) :
  resolution_(_resolution),
  iterations_(1),
  maxNum_(RAND_MAX + 1.0)
{
  double tmp = double(resolution_);
  while (tmp > (double(RAND_MAX) + 1.0) ) {
    iterations_++;
    tmp /= (double(RAND_MAX) + 1.0);
  }

  for ( unsigned int i = 0 ; i < iterations_ - 1; ++i ) {
    maxNum_ *= (RAND_MAX + 1.0);
  }
}

//-----------------------------------------------------------------------------

double RandomNumberGenerator::getRand() const {
  double randNum = 0.0;
  for ( unsigned int i = 0 ; i < iterations_; ++i ) {
    randNum *= (RAND_MAX + 1.0);
    randNum += rand();
  }

  return randNum / maxNum_;
}

double RandomNumberGenerator::resolution() const {
  return maxNum_;
}

//=============================================================================
} // namespace OpenMesh
//=============================================================================
