#
# This file is part of pyasn1-modules software.
#
# Created by Russ Housley
# Copyright (c) 2019, Vigil Security, LLC
# License: https://www.pysnmp.com/pyasn1/license.html
#
import sys
import unittest

from pyasn1.codec.der.decoder import decode as der_decoder
from pyasn1.codec.der.encoder import encode as der_encoder

from pyasn1_modules import pem, rfc5280, rfc8360


class CertExtnsTestCase(unittest.TestCase):
    extns_pem_text = """\
MIJM7TCCRkUGCCsGAQUFBwEcBIJGNzCCRjMwgkXJBAIAATCCRcEDAgACMAoDAgAF
AwQFBRwAMAoDBAMFHCgDAgEEMAoDAwINdAMDAg14AwMCDYwwCgMDAw2oAwMDDbAD
AwIN8AMEAhcTMAMEAxcTOAMEAxdTQAMEBBdTcAMEBRdp4DAMAwQDF2poAwQDF2pw
AwQEF2zQAwMAF20DBAMXbwAwDAMEBBdvEAMEBhdvADAMAwQEF29QAwQHF28AMAwD
BAYXb8ADBAMXb/ADBAUX74ADBAMX+egwDAMEARf8QgMEABf8RDAMAwQAF/xHAwQA
F/xIMAwDBAAX/EsDBAAX/EwDBAAX/E8DAwIYhAMEARjrFgMCABkwCwMDARtuAwQC
G24QMAkDAgAfAwMAH84wDAMEAx/PCAMEAx/PMDAKAwQCH888AwIFADAJAwIAJQMD
ACVuMAkDAwQlcAMCASQDBAIr4XADBAIr+6ADAwMtCAMEBS0rQAMEBS044AMEBi1B
QAMDAS1CAwQHLUuAAwMELVADAwUtgAMEAi36GDAKAwIBLgMEBi6VgDAKAwQELpXQ
AwIALgMDATEMAwMBMjwDAwAydzAJAwIAMwMDADNOAwMAM1AwCQMDATNSAwICMAME
BTSQQAMCADUwCgMDAjYkAwMANiYDAgA5AwQCPQ7gMAoDAgE+AwQGPggAMAwDBAU+
CGADBAU+DEAwDAMEBz4MgAMEBT4YQDAMAwQHPhiAAwQGPj2AMAsDAwE+PgMEBT5E
ADAMAwQGPkRAAwQFPkTAMAoDAwA+RQMDAT5wMAsDAwA+cwMEBT51ADALAwQGPnVA
AwMAPoYwCwMEBz6HgAMDAD6KAwQGPowAMAwDBAc+jIADBAU+rQAwDAMEBj6tQAME
Bj7BAAMEBT7BgDALAwQGPsHAAwMBPugwCwMDAD7rAwQFPvAAAwQFPvBAMAwDBAc+
8IADBAc+8QAwDAMEBT7xoAMEBz77AAMDAj78MAwDBAE/jSIDBAE/jSQDBAQ/jgAD
BAVAEMADBAVAHAADBAdAKwADBAZAQUADBAZAicADBAJA7+wDBAVA/SADBAZBEoAD
BARBEtADBAVBJ0ADBAZBY4ADBAVCT2ADBARCVEADBAZCdoADBAVCzUADBAVC1GAD
BAJC+MwwDAMEBUMWIAMEAkMWOAMEAEOePgMEBUPRgAMEBERGwAMEBkUGAAMEBEVe
QAMEBEVecAMEBUWsYAMEBkXCQDAJAwIATQMDAk3YMAwDBAVN3CADBAdQQwAwDAME
BFBDkAMEBVBHADAMAwQEUEcwAwQFUEhAMAwDBARQSHADBAVQS4AwDAMEBlBLwAME
BFBWADAMAwQFUFYgAwQGUFcAMAsDBAVQV2ADAwNQUDALAwQEUFgQAwMAUF4wDAME
BFBfEAMEBlDwgDALAwQEUPDQAwMBUPQwCwMEBFD2EAMDA1DwMAwDBARQ+BADBAZQ
+AAwDAMEBFD4UAMEBlD5ADAMAwQEUPlQAwQFUPoAMAsDBARQ+jADAwJRADALAwQG
UQRAAwMBUQgwDAMEB1EKgAMEBVEVQDAMAwQEURVwAwQGURYAMAwDBAVRFmADBARR
F8AwDAMEBVEX4AMEBlEaADAMAwQEURpQAwQFUR1AMAsDBARRHXADAwJRQDALAwMD
UUgDBAVRW8AwCwMEBFFb8AMDBlGAMAsDAwBRwQMEB1JlADALAwQGUmXAAwMHUgAw
DAMEB1KAgAMEB1KBADALAwMBUoIDBAZSlwAwCwMEBVKXYAMDAlKYMAsDAwFSngME
B1LJADALAwMBUsoDBANTjxAwCwMEBVOPIAMDAlQgMAsDAwBUJQMEBVTNQDALAwQH
VM2AAwMAVOgwDAMEB1TpgAMEBVf3gDAMAwQDV/eoAwQFV/9AMAwDBAdX/4ADBARZ
0OAwCwMDAFnRAwQEWe8gMAwDBAZZ70ADBAVb4QAwDAMEAVvhIgMEAlvrADALAwQB
W+sGAwMBXMgwCwMDAlzMAwQFXbNAAwQDXbNoAwQCXbN4MAwDBAdds4ADBABeGhow
DAMEAl4aHAMEAF4aHjAMAwQFXhogAwQCXhooAwQAXhotMAwDBAReGjADBABeGkIw
DAMEAl4aRAMEAF4aSDAMAwQBXhpKAwQCXhpQMAwDBAFeGlYDBAJeZwAwDAMEA15n
CAMEBF6+wAMEBF6+8DAMAwQHXr+AAwQGX6OAMAsDBARfo9ADAwBfqDAMAwQFX6kg
AwQCX7W4MAoDBAZftcADAgVAAwQFYAmAAwQEYH2QAwMAYoADAwNlOAMEAmfhyAME
AmfniAMEAmfxbAMEAmf7pAMEAmf8WAMEA2glIAMEAmiZVAMEAWjoJAMEBmjpQAME
Amjz2AMEAmj0CDAMAwQBa5aiAwQCa5awAwQEa6GgAwQFa7WAAwQAa79EMAkDAgBt
AwMCbegwCwMEBG3sEAMDAm3wMAkDAwBt9QMCAWwwCwMDAHHLAwQGccuAAwMBdMoD
BAJ9PkgDAwCAAAMDAIAHAwMAgBAwCgMDAIAnAwMBgCgDAwCALTAKAwMAgEEDAwSA
QAMDAYBWAwMAgF0DAwCAYgMEB4B0gAMDAIB8AwMAgH8DAwGAgjAKAwMAgIsDAwCA
jgMDAICoAwMAgLADAwGAsgMDAIDHAwMAgMwDAwCA1gMDAIDoAwMAgOoDAwCA8AMD
AIDzAwMAgPYwCgMDAIELAwMBgQwDAwCBEAMDAIEUAwMBgRoDAwCBHwMDAIFDMAoD
AwCBRQMDAIFGAwMAgUkDAwCBWDAKAwMBgWYDAwCBaAMDAIF9AwMAgYEDAwCBhAMD
AYGOAwMAgakDAwCBrzAKAwMAgbEDAwCBsjAKAwMAgbUDAwGBuAMDAIG7AwMBgcID
AwCBxwMDAIHOAwMAgdADAwGB1gMDAIHZMAoDAwCB6QMDAIHqMAoDAwSB8AMDAIHy
AwMAgfcDAwCCAAMDAYIYAwMAghwDAwCCIAMDAIIlAwMAgiswCgMDAII7AwMAgjwD
AwGCQgMDAIJJAwMAgksDAwGCTjAKAwMBglIDAwCCVAMDAYJYAwMBglwDAwCCYgMD
AIJkAwMAgmgDAwCCcAMDAIJzAwMAgngDAwCCfQMDAIKFAwMAgogwCgMDAYKKAwMB
gpQDAwCCnwMDAIKhAwQHgqSAAwMAgrQDAwCCtzAKAwMAgrkDAwCCugMDAIK8AwMA
gr4DAwGCwAMDAILGAwMAgswDAwCCzgMDAYLQAwMAgt8wCgMDAILhAwMAgvQDAwCC
9gMEA4L4MDAMAwQBgvg6AwQGgvgAMAwDBAKC+EQDBAKC+GgDAwCC+wMDAIL/AwMA
gwEDAwCDYQMDAINjAwMAg2YDAwCDbzAKAwMBg3IDAwGDdAMDAIOCAwMAg5gDAwGD
mjAKAwMAg58DAwCDoDAKAwMCg6QDAwCDpgMDAIOpMAoDAwCDrQMDAYOwAwMAg7QD
AwCDvAMDAIPNAwMAg88DAwCD0wMDAIPcAwMAg+AwCgMDAIPjAwMAg+QDAwCD5wMD
AIPqAwMAg+0DAwCD9gMDAIP7AwMAg/4wCgMDBoRAAwMAhE4DAwCEkjAKAwMAhJUD
AwCElgMDAISZAwMAhJswCgMDAISlAwMBhKgDAwCEqwMDAISwAwMAhLQwCgMDAIS5
AwMChLgwCgMDAITDAwMAhMQDAwCExwMDAITjMAoDAwCE5QMDA4TgAwMAhPQDAwCE
/AMDAoYAAwMAhhEDAwCGEwMDAIYVAwMAhhkwCgMDAIYbAwMAhhwDAwCGHjAKAwMB
hiIDAwGGJAMDAIYvAwMAhjYwCgMDAYY6AwMBhjwDAwCGTAMDAIZRAwMAhlMwCgMD
AYZaAwMAhm4DAwCGdwMDAIaCAwMAhooDAwGGjjAKAwMAhpEDAwKGkAMDAIaXAwMA
hpswCgMDAIadAwMAhp4DAwCGqQMDAIarAwMAhrAwCgMDAIa3AwMAhrgDAwCGvAMD
AIa/AwMAhssDAwCGzjAKAwMChtQDAwCG1jAKAwMAhtsDAwCG3jAKAwMAhuEDAwKG
4AMDAIbvMAoDAwCG9QMDA4bwAwMAhvkDAwCG/wMEAodUNAMEAodUlAMDAIdaAwMA
h8QDBAeIjwAwCwMEBIiQEAMDAIiQAwMAiJQwCgMDAIibAwMBiJwDBAeInoAwCgMD
AIijAwMAiKQwCgMDAIipAwMAiKoDAwGIrAMDAIjHAwMAiMkDAwCIzgMDAIjhAwMB
iOYDAwCI8wMDAIj/AwMAiREwCgMDAIkhAwMAiSIwCgMDAIkrAwMAiSwDAwCJLwMD
AIkyMAoDAwCJNwMDAIk4AwMAiTowCgMDAok8AwMAiT4wCgMDAIlJAwMAiUoDAwCJ
XQMDAIlgAwMAiWUDAwCJaQMDAIlsAwMBiXgDAwCJgQMDAImFAwMAiYoDAwCJnAMD
AImjAwMAia4DAwCJvzAKAwMAicEDAwKJwAMDAYnMAwMAidADAwCJ1TAKAwMAidkD
AwCJ2jAKAwMAid0DAwCJ4AMDAIniAwMAifgDAwGJ+gMDAIn9MAoDAwCKAwMDAIoE
AwMAigYDAwCKDjAKAwMAihUDAwCKFjAKAwMAiiUDAwCKJgMDAYooAwMAijADAwGK
PgMDAIpCAwMAikYDAwCKUQMDAIpgAwMAimQwCgMDAYpmAwMAimgDAwCKagMDAIp8
AwQDioCIMAoDAwCKgwMDAIqEAwMAioYwCgMDAIq7AwMGioADAwCKwzAKAwMBisYD
AwGKyAMDAIrLMAoDAwCKzQMDAIrOMAoDAwCK1wMDAIrYMAoDAwCK3QMDAIrkMAoD
AwCK5wMDAYroMAoDAwKK9AMDAIr2AwMCivgDAwCK/TAKAwMAiwEDAwCLBDAKAwMB
iwYDAwCLCDAKAwMBiwoDAwGLGDAKAwMAixsDAwCLHgMDAIstAwMAiy8DAwCLMgMD
AIs2AwMAizowCwMDAIs/AwQHi0AAAwMAi0IDAwGLSgMDAItPAwMAi1MwCgMDAItZ
AwMAi1wwCgMDBYtgAwMAi2IDAwCLZDAKAwMAi2kDAwCLeDAKAwMBi3oDAwCLfAMD
AIuAAwMAi4UDBAWLisADAwCLjQMDAIuPAwMAi5EDAwCLlQMDAIuZMAoDAwKLnAMD
AIueAwMAi6ADAwCLojAKAwMCi6QDAwCLpgMDAIuuAwMBi7IDAwCLuAMDAIu/AwMA
i94wCgMDAIxNAwMAjE4wCgMDAoxUAwMAjFYwCgMDAIxdAwMAjF4DAwCMYQMDAIxp
AwMBjJYDAwCMpAMDAIymAwMAjLUDAwCMywMDAIznMAkDAgCNAwMBjVQwCgMDAI1X
AwMBjWQDAwCNaQMDAI1sAwMAjXEDAwCNcwMDAI19AwMAjYIwCgMDAY2GAwMAjYow
CgMDAI2PAwMAjZQDAwCNowMEB42kgDAKAwMAjakDAwCNrDAKAwMAja8DAwCNsAMD
AI3AAwQAjcEUAwQCjcFsAwQBjcHWAwMAjcIDAwCNxAMDAY3IAwMAjcsDAwCN0DAL
AwMBjeIDBAON4oADBACN4oswCwMEBI3ikAMDAI3kAwMAjecDAwCN7QMDAI3xAwMB
jfQwCgMDAI35AwMAjfoDAwGN/AMDAI3/AwQDjlsIAwQDjlt4AwQDjluYAwQHjpoA
AwQDjuogAwMAjvcDAwCPKQMDAI8vMAoDAwGPMgMDAY80AwMAj0EwCwMEB49cgAMD
AY9cAwMAj2EDAwCPYzAKAwMAj3UDAwCPdgMDAI95AwMAj34wCgMDAI+BAwMAj4ID
AwCPoTAKAwMAj6MDAwCPpAMDAI+nAwMAj6kwCgMDBI+wAwMAj7QDAwCPxAMDAI/N
AwMAj9IDAwCP2QMDAY/gMAoDAwCP6QMDAI/qAwMAj+0DAwCP7wMDAI/1AwMAj/wD
AwCQAgMDAJAEAwMBkBQDAwCQGAMDAJAbAwMAkCADAwCQKTAKAwMAkCsDAwCQLAME
ApAwPAMDAJA2AwMBkDgwCgMDAJA/AwMBkEADAwCQTAMDAJBSMAsDAwKQVAMEB5BW
AAMDAJBXAwMAkF8DAwCQYgMDAJB3AwMAkHoDAwCQfAMDAJB/AwMAkJEDAwGQpAMD
AJCtMAoDAwSQsAMDAZC0AwMAkMEDAwCQyAMDAJDMAwMAkM4DBAaQ0AAwCwMEB5DQ
gAMDAZDQAwMAkPgwCQMCAJEDAwCSADAKAwMBkgIDAwCSBAMDAJITAwMAkhUDAwCS
MAMDAJIyAwMAkjQwCgMDAJI7AwMAkjwDAwGSQgMDAJJGAwQBkkdeAwMAkkgDAwCS
SwMDAJJNAwMBklADAwCSVwMEA5JY6AMDAJJaAwMAkmEwCgMDApJkAwMAkm4DAwCS
cDAKAwMAkncDAwCSeAMDAJJ8AwMAkoUDAwCSiAMDAJKMAwMBkp4wCgMDAJKhAwMA
kqIDAwCSqQMDAZKsMAoDAwCSrwMDAZKwAwMAkrMDAwCSuQMDAJK8MAoDAwCSvwMD
AJLCMAoDAwGS0gMDAZLUAwMAktgwCgMDAJLbAwMAktwDAwCS4DAKAwMAkuMDAwCS
5AMDAJLqAwMAkvEwCgMDAJL3AwMBkvgDAwCS+zAJAwMAkv0DAgCSAwMAkwcwCgMD
ApMMAwMAkw4wCgMDAJMbAwMAkx4DAwGTIAMDAZMsAwMAkzQDAwCTNgMDAJM8MAoD
AwCTQwMDAJNEAwMAk0swCgMDAJNNAwMAk04wCgMDAZNSAwMAk1QwCgMDAZNWAwMB
k1gDAwCTWzAKAwMAk10DAwCTYDAKAwMBk2IDAwCTZAMDAJNmMAoDAwCTbwMDAJNw
AwMAk3cDAwGTegMDAJN9AwMAk38DBAeTh4ADAwGTjjAKAwMAk5MDAwCTmAMDAJOc
MAoDAwCToQMDApOgAwMAk6cwCgMDAJOrAwMBk6wDAwCTrwMDAZO0AwMAk7gDAwCT
ugMDAZO8AwMAk8EDAwGTxAMDAJPJAwMAk8wDAwCT0jAKAwMAk9UDAwOT0AMDAJPc
MAoDAwKT5AMDAZPsAwMAk/MwCgMDAZP6AwMAk/wDAwGUAgMDAJQGAwMAlDYDAwCU
OAMEAJQ7cwMDAJQ8AwMAlD8DBAKUQDgDAwCURQMDAJRHAwQFlEzgAwMAlE8wCgMD
AJRRAwMClFADAwCUWAMDAZRuMAoDAwGUdgMDApR4MAoDAwCUhwMDAJSIAwMAlIoD
AwCUjAMDAJSPAwMAlJQDAwCUlwMDAJSgAwMAlKkDAwCUsAMDAJS1AwMAlLkDAwCU
uzAKAwMClMQDAwCUxgMDAJTIMAoDAwCU+wMDAZT8AwMAlQADAwCVAwMDAJUbAwMA
lTEDAwCVOwMDAJU+AwMAlVEDAwCVWgMDAJVtAwMAlX4wCgMDApWEAwMAlYYwCgMD
AJWLAwMAlYwwCgMDAZWSAwMAlZQwCgMDAJWZAwMBlZwDAwCVqgMDAJWsMAoDAwCV
sQMDAZXEMAoDAwOVyAMDAZXoMAwDBAOV6ggDBAeV6gAwCgMDAJXrAwMDlfAwCgMD
AJX5AwMClfgDAwGV/gMDAJZqAwQClmvIAwMAlnADAwCWgAMEApaBCAMDAJaEAwMA
lowwCgMDAJaRAwMAlpIDAwCWrwMDAJayAwMBlswwCgMDAJbVAwMAltYDAwCW2QMD
AJbjAwMBluwDAwCW8QMDAJb0AwMAlvsDAwCW/jAJAwIAlwMDAJdkMAsDAwGXagME
BZdqgDAMAwQGl2rAAwQEl2rgAwMAl3MDAwCXeAMDAJd/AwMAl4EDAwCXhTAKAwMA
l4cDAwCXiAMDAZecAwMAl6oDAwCXrTAKAwMAl68DAwGXsAMDAJe0MAoDAwGXtgMD
AJe4AwMAl7sDAwCXvTAKAwMAl9MDAwCX1DAJAwMDl9gDAgOQAwMAmEIDAwCYRwMD
AJhJMAoDAwCYTQMDAJhOAwMAmFEwCgMDA5hYAwMAmFowCgMDAJhdAwMAmGADAwCY
aQMDAZhyAwMAmIYDAwCYjwMDAJiWAwMAmJgDAwCZAQMDAJkFAwMAmQ8DAwCZEQMD
AJkTAwMAmVgwCgMDAplcAwMAmWIDAwCZZDAKAwMCmWwDAwCZbgMDAJlwMAsDAwOa
CAMEBJoIIAMEBpoIQAMDAZoOAwMAmiADAwCbBAMDAJstMAoDAwGbNgMDAJs4AwMA
m0IDAwCbSQMDAJtpMAoDAwCbgwMDAZuEAwMBm4gDBAabikADAwCbjAMDAJuRAwMA
m54DAwCbuQMDAJvAAwMAm8YDAwCbygMDAJvMAwMAm88wCgMDAJvRAwMAm9IDAwCb
3zAKAwMAm+MDAwCb5AMDAJvnAwMAm/UwCgMDAJv5AwMAm/oDAwCb/QMEBZv+IAMD
AJwKAwMAnA4wCgMDAJwRAwMAnBIDAwCcGQMDAJwcAwMAnCMDAwCcKwMDAZwwMAoD
AwCcMwMDAJw0AwMAnDYDAwCcOgMDAJw9AwMAnEMDAwCcUwMDAJxqMAoDAwGccgMD
AJx0AwMAnHYDAwCchQMDAJyHAwMAnJQDAwCcljAKAwMAnRcDAwGdHDAKAwMAnVMD
AwCdVAMDAJ1YAwMAnV4DAwGdYAMDAJ1jAwQDnXjgMAsDBAKdeOwDAwCdeAMDAZ18
AwMAnYEDAwCdiAMDAJ2KAwMAnYwDAwCdkDAKAwMAnZ0DAwWdgDAKAwMAnaEDAwCd
pDAKAwMAnacDAwGdqDAKAwMAnasDAwGdrAMDAJ2xAwMBnbQDAwCdugMDAJ2+AwMA
ncEDAwCdyAMDAJ3LAwMAneQDAwCd5wMDAZ3sAwMAnfMDAwCd9wMDAJ35AwMCniQw
CgMDAJ4pAwMCnigDAwGeLjAKAwMAnjEDAwCeMgMDAJ46AwMAnjwDAwCeQAMDAZ5C
AwMAnksDAwCeWgMDAJ5cAwMAnl4DAwCeYwMDAJ5mAwMAnmkwCgMDAJ5tAwMAnm4D
AwCecAMDAJ53AwMCnnwDAwCegQMDAJ6DAwMAnoUDBASejBADBAaejEADAwCejwME
BJ6SgDAKAwMCnpQDAwCelgMDAJ6YAwMAnpwDAwCeojAKAwMBnqYDAwGeqDAKAwMC
nqwDAwGetDAKAwMBnr4DAwGexAMDAJ7YAwMAntoDAwCe3AMDAJ7fMAoDAwCe4QMD
Ap7gMAoDAwGe5gMDAJ7qMAoDAwOe+AMDAJ76MAoDAwCe/wMDAJ8AMAoDAwCfBwMD
AJ8IAwMAnwwDAwCfDwMDAJ8UAwMBnxYDAwCfGTAKAwMAnx0DAwWfAAMDAJ8iAwMA
nyYDAwCfLgMDAZ8yAwMAnzoDAwCfPAMDAJ9FAwMAn0gDAwCfUQMDAJ9UAwMAn1YD
AwGfXAMDAJ9fAwMBn2QwCgMDAJ9nAwMAn2gDAwCfawMDAJ9tAwMAn3IDBAOfdcAw
DAMEAZ91ygMEBJ91wAMDAJ96AwMAn4IDAwCfhgMEBZ+HgAMDAJ+QMAoDAwGfkgMD
AZ+UMAoDAwCflwMDAJ+YAwMAn5oDAwCfnAMDAJ+gAwMAn6IwCgMDAJ+nAwMAn6gD
AwGfqgMDAJ+tMAoDAwCfswMDAJ+0AwMAn74wCgMDAJ/BAwMCn8ADAwCfxQMDAJ/I
AwMAn80DAwCf0gMDAJ/VAwMBn9gDAwCf2wMDAJ/gAwMAn+gDAwGf7AMDAJ/vAwMA
n/EDBAOf8ggDBAaf8kADBASf8uADAwGf9DAJAwMAn/0DAgWAMAoDAwCgBQMDAKAG
AwMBoAgDBAKgE1wDBAKgE7QDBAOgFGADBAKgFGwDBASgFJADBAGgFNYDBACgFOUD
BAGgFPgDAwCgJgMDAKAoMAoDAwKgLAMDBqAAMAoDAwGgQgMDAKBEAwMBoEYDAwCg
SwMDAKBOAwMAoFADAwCgVQMDAKBcMAoDAwCgYQMDAKBkMAoDAwCgZwMDAKBoAwMA
oHIDAwCgtAMEBKDKEAMDAaDSMAoDAwCg1QMDAKDWMAoDAwOg2AMDAaDcAwMAoOQD
BAag50ADBACg7hUDBAKg7iQDBAKg7jQDBAGg7jwDBAKg7mADBASg7nAwCgMDAaEC
AwMBoQQDAwGhCAMDAKEMAwMAoREDAwChFAMDAKEXAwMAoRsDAwChHgMDAKElMAoD
AwChKQMDAKEqAwMAoTAwCgMDAqE0AwMAoTYDAwChOwMDAKE+AwMAoUMwCgMDAaFG
AwMAoUoDAwChTAMDAKFOMAoDAwChUwMDAKFcMAoDAwOhaAMDAKFqMAoDAwGhbgMD
AKFwAwMAoXQDAwChhgMDAKGTAwMAoZwDAwChngMDAKHKAwMAodoDAwCh5jAKAwMA
ofsDAwCh/AMDAKILAwQDogzIAwMAog0DAwCiFQMDAKIXMAoDAwCiGQMDAKIaAwMA
oiYDAwCiVgMDAKKFAwQHoo6AAwQBotiKAwQCotn4AwQDotpYAwQCotqwAwQDotzw
MAwDBAOi3lgDBACi3loDBAKi9DQDBAKi9cwDBAKi+MQDBAKi+tgDBAOi+yADBAKi
/8QDAwCjAQMDAKMDAwMAowUDAwCjCQMDAKMiMAoDAwGjPgMDAaN0AwMAo3cwCgMD
AqOcAwMEo6ADAwCj8jAJAwICpAMDAKQoAwMApDAwCgMDAKQ7AwMBpDwDAwCkUQME
B6RdgDAKAwMBpH4DAwSkgAMDAKSxAwMApNcDAwClTgMEBKVU0AMDAKVyAwMApcAD
AwCl2gMDAKXeAwMDpggDBAemMYADAwCmVwMDAKdRMAsDBAGnVkIDAwOnUAMDAKdi
AwQHp2SAAwMAp28DBAWnoAADAwCnqAMDAKesAwMAp8sDAwCn6QMDAKgBAwMAqIsD
BAeolQADBAGolfgDAwCouwMEAaj1xAMDBakgAwQHqZQAAwQEqgpwAwQHqhGAMAwD
BACqJcsDBACqJc4DAwCqPAMEB6plgAMEB6pmgAMEB6qFAAMDAarsAwMAqv8wCgMD
BKsQAwMBqyADBAesUIADBAOsZ1gDBAOs8QAwDAMEA6zxKAMEA6zxMAMEA6zxQAME
AqzxVAMEBKz/AAMEBKz/YAMEA6z/iAMEBKz/wAMEBKz/4AMEA6z/+AMEBq3UwAME
Aq3WyAMEA63qgAMEBK30kAMEBq35AAMEA63/kAMEBq6MAAMEB69ugDAKAwIEsAME
A7BvMDAMAwQAsG85AwQEsHOgMAwDBAOwc7gDBAewegAwCgMEBrB6wAMCALAwCgMC
AbIDBASy7kAwCgMEBbLuYAMCALIDAwC06jAKAwIAuQMEArkIYDAMAwQDuQhoAwQC
uRTYMAwDBAW5FOADBAO5HkAwDAMEArkeTAMEAbkm0DAMAwQCuSbUAwQCuVrwMAwD
BAO5WvgDBAO5ZCAwDAMEArlkLAMEALlpBjAMAwQDuWkIAwQFuZIAMAwDBAK5kiQD
BAK5qWgwDAMEBLmpcAMEBLmrwDAMAwQCuavUAwQCubAwMAwDBAO5sDgDBAS5soAw
CgMEArmylAMCAbgwCgMCArwDBAe8gwAwCwMDAryEAwQEvNGAMAwDBAO80ZgDBAa8
1gAwCgMEBLzWUAMCALwwDAMEAsAFHAMEAMAFHgMEAMAFJAMEAMAFMgMEAMAFOzAM
AwQAwAU9AwQAwAU+AwQAwAVhAwQAwAWOAwQAwAWRAwQAwAWiAwQAwAXvAwQAwAX+
AwQAwAwBAwQAwAwvAwQAwAw2AwQBwAxIAwQAwAxNAwQAwAxRAwQAwAxgAwQAwAxj
MAwDBAbADMADBADADMIwDAMEAMAM2QMEAMAM2jAMAwQAwAznAwQAwAzoAwQAwAzr
AwQAwAz3AwMAwA8wDAMEAMAQewMEAMAQpjAMAwQAwBC3AwQAwBDKAwQAwBLDMAwD
BALAGhwDBATAGiAwDAMEAMAaaQMEAcAabDAMAwQAwBpvAwQAwBqAMAwDBADAGoUD
BAPAGoAwDAMEAcAamgMEAcAawAMEAMAa5wMEAMAa6gMEAsAa7DAMAwQCwBx8AwQC
wByAAwQAwB8OAwQAwB8XAwQBwB8aAwQAwB8fAwQAwB8oAwQAwB8+AwQAwB9mMAwD
BAHAH6YDBALAH6gDBADAH9MDBADAH+cDBADAH/wwDAMEAMAhDwMEAMAhEAMEAMAh
JDAMAwQAwCFXAwQEwCFgMAwDBADAIXEDBADAIXIwDAMEAcAhdgMEB8AhAAMEAcAh
gjAMAwQAwCGPAwQBwCGQMAwDBADAIZMDBADAIaYwDAMEAMAhqQMEAsAhqDAMAwQE
wCGwAwQAwCG2MAwDBAbAIcADBAPAIeAwDAMEAMAh6QMEBMAh4AMEAMAh/gMEAMAi
EwMEAMAiMgMEAMAiawMEAMAidAMEAMAiszALAwMAwCMDBALAIxAwDAMEAMAjPwME
AMAjSAMEAMAjWgMEAMAjXgMEAMAjbDAMAwQBwCOCAwQBwCOEAwQAwCOKAwQAwCOS
MAwDBADAI5UDBAHAI5gDBADAI6wwDAMEAMAjtwMEAMAjwDAMAwQAwCPFAwQAwCPG
MAwDBADAI80DBATAI8ADBADAI+UwDAMEBMAj8AMEAMAj9AMEAcAj9jAKAwMCwCQD
AwDAJjAMAwQAwChFAwQAwChQAwQAwCjkMAwDBADAKWcDBADAKYgwDAMEAsApjAME
AcApkAMEAMApkzAMAwQAwCmVAwQAwCmgAwQAwCnSAwQAwCnYAwQAwCnaAwQAwCnj
AwQAwCoBMAwDBAHAKioDBATAKiADBADAKjUwDAMEAMAqPwMEAcAqQAMEAMAqVwME
AMAqYQMEAMAqZAMEAMAqZjAMAwQAwCpxAwQAwCqEAwQAwCqPMAwDBALAKrQDBAHA
KsgDBADAKv0wDAMEAcArogMEAsArqDAMAwQGwCvAAwQAwCvEAwQAwCvSAwQAwCvU
AwQAwCvqMAsDAwLALAMEAsAsQDAMAwQAwCxHAwQAwCxaMAwDBATALPADBALALPgw
DAMEAsAv9AMEAcAv+AMEAMAwHwMEAMAwawMEAMAwkQMEAMAw4AMEAMAw5wMDAMAx
AwQEwDMAMAsDAwLANAMEAMA0MgMEAMA0mDAMAwQAwDSfAwQBwDSgMAwDBADANN0D
BAXANMAwDAMEAMA0/QMEAMA0/jAMAwQAwDVnAwQAwDVoMAwDBADANh8DBADANjQw
DAMEAcA2NgMEAMA2UAMEAMA2aAMEAMA2cTAMAwQAwDZzAwQAwDZ4AwQAwDZ6MAwD
BADANn0DBADANoADBADANoQDBADANoswDAMEAMA2jQMEAcA23AMEAMA24QMEAcA2
9AMEAMA2/gMEAMA3VAMEAMA3WQMEAMA3ZQMEAMA3aQMEAMA3bQMEAMA3cwMEAMA3
gQMEAMA3hAMEAMA3vAMEAMA3wQMEAMA3xQMEAMA31AMEAMA39DAMAwQCwDocAwQC
wDogMAwDBADAOikDBAHAOlgDBADAOsUDBADAOtowDAMEAcA64gMEAcA65AMEAMBA
HAMEAcBALAMEAsBAZAMEAMBAfQMEAMBAyjAMAwQAwEEzAwQAwEFGMAwDBALAQVwD
BADAQV4DBADAQWAwDAMEAMBBgwMEAMBBhAMEAMBBizAMAwQEwEGQAwQAwEGSAwQA
wEGZMAwDBADAQbcDBAHAQcQwDAMEAMBB2wMEAMBB5AMDAMBCMAwDBADAQwMDBADA
QwQDBADAQycDBADAQysDBADAQy8DBADAQzIDBADAQzQDBADAQzcDBADAQzoDBADA
Q0wDBADAQ08DBADAQ1cDBADAQ14wDAMEAsBDZAMEAMBDaAMEAMBDhwMEAMBDpwME
AMBDqjAMAwQAwEO9AwQAwEPQAwQAwEPaAwQBwEPcAwQAwEPfAwQAwEP5MAsDAwLA
RAMEAsBEEAMEAMBEFzAMAwQAwEQfAwQCwEQwMAwDBALAREwDBALARGgwDAMEAMBE
lwMEAcBEmDAMAwQAwESlAwQAwESqAwQAwESuAwQAwESwAwQAwES2AwQAwES6AwQA
wETRMAwDBADARNMDBADARNgDBADARN0DBADAROADBADAROYwDAMEAcBE+gMEAMBE
/AMEAMBE/jALAwMBwEYDBAPARnAwDAMEAMBGhQMEAMBGhgMEAMBGiDAMAwQCwEaM
AwQFwEaAAwQDwEbAAwQAwEbyAwMAwEcwDAMEAMBJEwMEAMBJFDAMAwQBwEkiAwQA
wEksAwQAwEniAwQAwEnlMAwDBAHATAYDBALATCAwDAMEAMBMewMEAMBMhDAMAwQB
wEyGAwQAwEysAwQAwEywMAwDBADATPEDBADATPgDBADATQsDBAHATXIDBALATXgw
DAMEAsBNhAMEAMBNigMEAcBNjAMEAcBQFDAMAwQAwFAfAwQAwFAqAwQAwFAuAwQA
wFAzAwQAwFE7MAwDBADAUT0DBADAUT4DBADAUW0DBADAUXkDBADAUXsDBADAUaAw
DAMEAcBRtgMEAMBRuAMEAMBRwgMEAMBR5gMEAMBR6gMEAMBSeQMEAMBSfAMEAMBS
fwMEAMBSmTAMAwQAwFKdAwQAwFKeAwQAwFLWAwQBwFLcAwQAwFLxMAsDAwDAUwME
AMBTZAMEAMBTZgMEAMBToAMEAMBTpQMEAMBTyAMEAMBTygMEAMBT2AMEAMBT3zAM
AwQAwFPlAwQAwFPmAwQAwFQFAwQAwFQNAwQAwFQPAwQAwFQbAwQAwFQeAwQBwFQg
AwQAwFQ+MAwDBADAVEsDBADAVFQDBADAVFcwDAMEAcBUWgMEAsBUYDAMAwQAwFRl
AwQBwFRsMAwDBADAVH8DBADAVJwDBADAVKYDBADAVK0wDAMEBMBUsAMEBMBUwAME
AMBU1AMEAcBU3DAMAwQBwFTiAwQBwFTkMAwDBADAVPUDBAPAVPADBADAVgsDBADA
Vg4DBADAVhIDBADAVhkDBADAVhsDBADAVlkwDAMEAMBWfQMEB8BWAAMEAMBWhjAM
AwQAwFaJAwQAwFaKAwQAwFajAwQBwFamAwQAwFapAwQAwFb+AwMAwFcDBADAWAED
BADAWAQwDAMEAMBYCQMEAMBYCgMEAMBYETAMAwQAwFgXAwQAwFgYMAwDBADAWFMD
BADAWFQDBADAWFYwDAMEAMBYYQMEAMBYYgMEAMBYbAMEAMBYdgMEAMBYewMEAMBY
gAMEAMBYggMEAMBYhQMEAMBYxAMEAMBYzAMEAcBY7jAMAwQBwFj6AwQAwFj+AwMA
wFkDBAHAW4wDBADAW7EDBADAW7oDBADAW70DBADAW78DBADAW8cDBADAW8kDBADA
W9MwDAMEAcBb1gMEAcBb6DAMAwQCwFvsAwQDwFvwAwQAwFxWAwQAwFxeMAwDBAPA
XGgDBADAXGoDBAHAXGwDBADAXHQwDAMEAMBcfQMEAsBciDAMAwQAwFyNAwQBwFyY
MAwDBADAXJsDBADAXJwDBADAXNgDAwDAXQMEAMBeGAMEAMBeHDAMAwQAwF45AwQA
wF46MAwDBADAXkMDBADAXkQDBADAXkwDBADAXk4wDAMEAMBebwMEAcBedDAMAwQC
wF6cAwQCwF6gAwQAwF6sMAwDBADAXq8DBAPAXsADBADAXtQDBADAXt0DBADAXuID
BADAXukDBADAXusDBADAXu8DAwDAYgMEAMBkEjAMAwQAwGQXAwQBwGQYAwQAwGQ0
AwQAwGQ9AwQAwGQ/AwQAwGROAwQCwGRgMAwDBAHAZGYDBAHAZIQwDAMEAMBkhwME
AMBkjAMEAMBkkAMEAMBkmgMEAMBlAQMEAMBlBAMEAMBlCAMEAMBlCwMEAMBlHAME
AMBlIgMEAMBlSzAMAwQAwGVRAwQAwGVaMAwDBADAZW8DBADAZXIDBADAZXYDBADA
ZYkwDAMEAMBloQMEAMBlqAMEAMBlqgMEAMBlsDAMAwQAwGWzAwQAwGW0AwQAwGXA
MAwDBADAZcUDBADAZcYDBADAZfwDBADAZgEwDAMEAcBmBgMEAMBmCDAMAwQAwGYR
AwQBwGZQAwQAwGZZAwQAwGZfMAwDBAHAZpIDBAHAZrADBADAZtYDBAHAZuAwDAME
AMBm4wMEAcBm5AMEAMBnAgMEAMBnBwMEAMBnDgMEAMBnFAMEAMBnFzAMAwQAwGcb
AwQAwGcoMAwDBADAZ1UDBADAZ3QwDAMEAMBniQMEAsBniAMEAMBnkwMEAMBoFwME
AcBoHDAMAwQAwGgjAwQBwGgkAwQAwGgpAwQAwGgwAwQAwGg1MAwDBADAaDcDBADA
aDoDBADAaEgDBADAaE0DBADAaFIDBADAaIwDBADAaI4DBADAaJMDBAHAaJowDAME
AMBopwMEAMBoqAMEAMBo7gMEAMBo9QMEAMBo+AMEAMBo+wMEAMBpSwMDAMBqAwQA
wGsCAwQBwGsEMAwDBADAawsDBAHAawwwDAMEAMBrMwMEAMBrZAMEAMBrbgMEAMBr
cjAMAwQBwGt6AwQBwGuAAwQAwGuEAwQAwGuoAwQAwGuuMAwDBADAa7EDBADAa7ID
BADAa7swDAMEA8BryAMEAcBr6DAMAwQAwGvrAwQAwGvsMAwDBADAbBcDBAHAbDAw
DAMEAMBsMwMEAMBsXAMEAcBsZDAMAwQAwGxrAwQAwGxsMAwDBAHAbHIDBAHAbHgw
DAMEAMBsfQMEAMBsfjAMAwQHwGyAAwQAwGyuMAwDBADAbMMDBADAbNYDBADAbOoD
BADAbO4wCwMDAMBtAwQAwG0sMAwDBAHAbS4DBADAbUowDAMEAsBtTAMEAsBtWDAM
AwQBwG1eAwQAwG1iMAwDBALAbWQDBADAbWYwDAMEAMBtaQMEA8BtcDAMAwQAwG15
AwQBwG3wMAsDBADAbfMDAwHAbAMEAMBvIQMEAMBvJwMEAMBvLDAMAwQAwG8vAwQA
wG8wAwQAwG9YAwQAwG9lMAwDBADAb2cDBADAb2gDBAHAb3wDBADAb38DBADAb/ww
DAMEAcBwHgMEAMBwIAMEAMBwLQMEAMBwMQMEAMBwPTAMAwQBwHBGAwQEwHBAMAwD
BAHAcGIDBADAcGQDBADAcMwDBADAcM4DBADAcNAwDAMEAMBw1QMEAMBw1gMEAMBw
9wMEAMBw/jAKAwMAwHEDAwDAdgMDAMB5MAwDBADAegEDBADAeoIwDAMEAMB6jQME
AMB6kjAMAwQAwHqXAwQAwHqqAwQAwHrWMAwDBAPAetgDBADAeuowDAMEAcB67gME
AMB68gMEAMB6/jAMAwQAwHwZAwQAwHwcAwQAwHwgAwQAwHwnAwQAwHwuAwQAwHxw
MAwDBADAfHMDBADAfHQDBADAfJswDAMEAcB8qgMEAsB82AMEAMB86zAMAwQAwHzt
AwQBwHz0MAwDBADAfPcDBADAfPgwDAMEAcB8+gMEAMB8/gMDAMB9MAwDBADAfgED
BADAfkAwDAMEAMCBAQMEAcCBPAMEAMCBUAMEAMCBVwMEAMCBYgMDAMCCAwQAwIMU
MAwDBADAgxkDBADAgxoDBADAg08DBADAg1kDBADAg2ADBADAg2wDBADAg4QwDAME
AMCECQMEAsCECAMEAMCEIgMEAMCENQMEAMCENwMEAMCEYwMEAMCE7wMEAcCE9AME
AMCE/AMEAMCFDwMEAMCFHAMEAMCFIAMEAMCFJDAMAwQAwIU1AwQDwIUwAwQAwIU6
AwQAwIVAAwQCwIVsAwQAwIV5AwQAwIWDAwQAwIX0AwMAwIYwDAMEAMCHBwMEAcCH
JAMEAMCHLjAMAwQAwIczAwQBwIc0AwQAwIc/AwQAwIdCAwQAwIdEAwQAwIdSAwQA
wIdkAwQAwIeBAwQAwIeFAwQAwIePMAwDBADAh5EDBADAh6gDBADAh68DBADAh7sD
BADAh9sDBADAh+EwDAMEAMCH5wMEAMCH6jAMAwQAwIf9AwQAwIf+AwQAwIgHAwQA
wIgJAwQBwIgSAwQAwIgXMAwDBADAiB0DBAXAiAADBAHAiCgDBADAiDEwDAMEAMCI
MwMEAMCINAMEAMCIPTAMAwQAwIhHAwQAwIhmMAwDBAHAiJoDBADAiJwwDAMEAMCK
AQMEAMCKCAMEAMCKVjAMAwQAwIppAwQBwIp0MAwDBADAipsDBADAip4DBADAircD
BADAisADBADAiswwDAMEAsCK5AMEAMCK6AMEAMCK+AMEAMCLTgMEAcCMAgMEBsCQ
ADAMAwQBwJBKAwQBwJBMAwQHwJEAAwQCwJHgAwQAwJJ1AwQAwJJ3MAwDBADAknsD
BADAkn4DBADAkoQwDAMEAcCShgMEAcCSjAMEAMCSmDAMAwQAwJKjAwQAwJKqMAwD
BALAkqwDBADAkrYDBADAkrkDBADAkrsDBADAksEDBADAkswwDAMEAMCS4wMEAMCS
5DAMAwQAwJLpAwQAwJLqAwQBwJLuAwQAwJLyAwQAwJMXAwQAwJMiAwQAwJMkAwQA
wJMqAwQCwJNMAwQAwJONAwQAwJOWAwQAwJObAwQAwJPUMAwDBADAk9cDBAHAk9gD
BADAk9sDBADAk+QDBADAk/cDBADAk/swDAMEAMCUIQMEAMCUXAMEAMCUZwMEAcCU
pjAMAwQAwJSxAwQAwJS6MAwDBAbAlMADBADAlMIwDAMEAcCUxgMEAMCU2AMEAMCU
3AMEAMCVAwMEAMCVBQMEAMCVDwMEAMCVEwMEAMCVGzAMAwQAwJUdAwQBwJUgAwQA
wJUjAwQAwJUpAwQAwJU5MAwDBADAlTsDBADAlTwwDAMEAMCVTQMEBMCVQAMEAMCV
YgMEAMCVZAMEAMCVZgMEAcCVbjAMAwQAwJV1AwQBwJV4AwQAwJV+AwQAwJXjAwQA
wJXoAwQAwJXuAwQAwJYUMAwDBAHAljoDBADAlkgwDAMEAMCWSwMEAMCWTDAMAwQB
wJZOAwQAwJZUAwQAwJZZAwQAwJZcAwQAwJZeAwQAwJZoAwQAwJZqAwQAwJZ8AwQA
wJaMAwQAwJaSMAwDBADAlrEDBADAlrgwDAMEAsCWvAMEAMCWvjAMAwQGwJbAAwQA
wJbGMAwDBADAlssDBAHAlswDBAHAltADBADAlt8wDAMEAMCW4wMEAcCW6AMEAMCW
7gMEAMCW+AMEAMCW/AMEAMCW/gMEAMCYBgMEAMCYDgMEAMCYEQMEAcCYGgMEAMCY
KgMEAMCYLDAMAwQAwJgvAwQCwJgwAwQAwJg2MAwDBADAmD0DBAbAmAADBADAmEQD
BADAmFIDBADAmGIwDAMEAMCYbwMEAMCYcAMEAMCYegMEAMCYfAMEAMCYjQMEAMCY
lwMEAMCYnAMEAcCYpgMEAMCYrgMEAsCYuAMEAMCY8QMEAMCY9DAMAwQAwJj9AwQA
wJj+AwQBwJkCAwQAwJkNAwQBwJkSAwQAwJlZAwQAwJl0AwQAwJl/AwQAwJmZAwQA
wJmmAwQAwJmoAwQAwJmrMAwDBADAma0DBADAmbYDBAHAmbwDBADAmcIDBADAmdUw
DAMEAMCbAQMEAMCbBgMEAMCchAMEAMCcogMEAMCcpwMEAMCc0gMEAMCc1QMEAMCc
2QMEAMCc4wMEAMCc7wMEAMCc+DAMAwQAwJ0BAwQCwJ0AMAwDBAPAnQgDBAHAnRAD
BADAnYEwDAMEAMCdpQMEAMCdrAMEAMCdrgMEAMCdsAMEAMCduQMEAMCduwMEAMCd
vQMEAcCfJgMEAMCfRgMEAMCfSQMEAMCfTQMEAcCfVAMEAMCfWgMEAMCfXzAMAwQA
wJ9jAwQDwJ9gAwQAwJ9pAwQBwJ9sAwQAwJ92MAwDBADAn3kDBADAn3oDBADAoAoD
BADAoA8wDAMEAMCgFQMEA8CgEAMEAMCgGwMEAMCgIQMEAMCgJQMEAMCgQQMEAMCg
QzAMAwQAwKBfAwQAwKBgAwQAwKBqMAwDBADAoG0DBADAoG4DBADAoHsDBADAoH4D
BAHAoI4DBADAoJgDBADAoJwDBADAoKADBADAoKwwDAMEAMCgsQMEAsCgsAMEAcCg
wgMEAcCg4DAMAwQAwKDnAwQBwKDoMAwDBADAoPUDBAHAoPgwDAMEAMCg+wMEAMCg
/AMEAcChBgMEAcChQAMDAMCiMAwDBAXAoyADBAXAo4ADAwLApDAMAwQAwKsBAwQB
wKsEMAwDBAfAq4ADBAPAq8ADBADArOgDBADArP0wDAMEAMCtAQMEAMCtBAMEA8Ct
gDAMAwQGwK5AAwQAwK5EAwQBwK8OAwQEwK8gAwMAwLAwDAMEBMC7EAMEAcC7GAME
AMC8CjAMAwQAwLw/AwQBwLxAAwQAwLxFAwQAwLxgAwQAwLxpAwQBwLx0MAwDBADA
vHkDBADAvHoDBADAvH0DBADAvH8DBADAvIEDBADAvIQDBADAvIgDBADAvJEwDAME
AMC8nQMEAMC8ngMEAMC8uwMEAMC8vTAMAwQAwLzpAwQBwLzsMAwDBAHAvPIDBADA
vPgDBADAvQEDBALAvQgDBADAvQ4DBADAvRcDBADAvSkwDAMEAMC9MwMEAMC9NAME
AMC9NwMEAMC9QjAMAwQAwL1FAwQAwL1GAwQAwL1JAwQAwL1MAwQAwL13AwQAwL2X
AwQAwL2aAwQAwL2dAwQAwL2gMAwDBAHAvaYDBADAvaoDBAHAvcoDBADAvfsDBADA
viwDBAHAvjoDBADAvkADBADAvkMDBADAvkUDBADAvl8wDAMEAMC+gQMEAMC+hDAM
AwQAwL6tAwQAwL6uAwQBwL62MAwDBAHAvr4DBAHAvsAwDAMEAMC+yQMEAMC+yjAM
AwQAwL7pAwQAwL7sAwQAwL7wAwQAwL7yMAwDBADAvvcDBAHAvvgDAwDAwgMEAMDD
AQMEAMDDCAMEAcDDKgMEAMDDSAMEAMDDYjAMAwQAwMNpAwQAwMNqAwQAwMNuMAwD
BALAw3QDBADAw3YwDAMEAcDDhgMEAMDDlDAMAwQAwMO3AwQAwMO4AwQAwMPDAwQA
wMPsMAwDBADAxAEDBALAxJgDBATAxxADBADAy1ADBAHAy2wDBADAy+MwDAMEAMDO
TQMEAcDOUAMEAMDOVjAMAwQBwM7eAwQAwM7iAwQAwM8OAwQAwM8fMAwDBADAz40D
BADAz44DBADAz8QDBADA50MDBADA51IwDAMEAMDuAQMEAMDuCgMEAMD1mAMEAMD1
qQMEAMD14TAMAwQAwPcBAwQAwPcKMAwDBADA+z0DBAHA+0ADBADA++IDBADA++Yw
CgMCAMEDBADBEdYwDAMEA8ER2AMEA8ET4DAMAwQCwRPsAwQBwSmQMAwDBALBKZQD
BAbBUgAwDAMEBcFSYAMEBcFSwDAKAwMAwVMDAwDBXjAMAwQHwV+AAwQAwWwWAwQC
wWwYMAwDBADBbB0DBAHBbNQwDAMEAMFs1wMEAsFs+DALAwMAwW0DBAHBbUAwDAME
AsFtRAMEA8FuYDALAwQBwW5qAwMEwWADAwDBcTAMAwQFwXIgAwQFwXJAMAwDBAXB
cqADBAXBcwAwDAMEB8FzgAMEBcF0AAMEBsF0gAMEBsF1ADALAwQFwXVgAwMAwXYw
CwMEB8F3gAMDAcF4MAsDAwLBfAMEAMG8BjALAwQDwbwIAwMAwbwwDAMEAcG9QgME
B8G9ADAMAwQAwb2BAwQAwcIAMAwDBAHBwgYDBAXBwgADBAXBwmAwDAMEAMHCgQME
BcHCgDAMAwQGwcLAAwQBwd3YMAsDBADB3dsDAwDB4gMEBsHjQDAMAwQAweOBAwQF
wgbAMAwDBADCBuEDBAbCCQAwDAMEAcIJQgMEAcIJUDAMAwQCwglUAwQAwiO+MAwD
BAbCI8ADBAXCT0AwCwMEB8JPgAMDAMJaMAsDAwLCXAMEBcLBAAMEBsLBQDAMAwQF
wsHgAwQGwsyAMAoDAwDCzQMDAMLeMAwDBAXC32ADBAXC34AwDAMEBsLfwAMEBMMY
QDAMAwQDwxhYAwQGwxiAMAwDBAXDGOADBAHDJ9gwCwMEAsMn3AMDAMMqMAwDBAXD
KyADBALDgAAwDAMEA8OACAMEBcOmwDALAwMAw6cDBAXDp4AwDAMEBMOnsAMEBsPK
ADAMAwQFw8pgAwQDw+pwMAwDBALD6nwDBAPD6qAwDAMEAMPqqQMEAMPquDAMAwQB
w+q6AwQCw+r4MAsDAwDD6wMEBcP2ADAKAwQGw/ZAAwICwAMEAMQBAzAMAwQAxAEF
AwQAxAEGAwQCxAFAMAwDBADEAUUDBADEAUYDBALEAgQwDAMEAcQDQgMEA8QDQAME
AMQDWwMEBcQPIAMEAsYLAAMEAMYRTQMEAMYRdTAMAwQCxhG0AwQAxhG2AwQAxhYz
MAwDBADGFl0DBAHGFmAwDAMEBcYkIAMEAMYkLgMEAsYtdAMEAMYzDDAMAwQAxjOP
AwQAxjOSAwQCxjQsAwQCxjccAwQDxllYAwQAxmOUAwQAxmPeAwQFxmlgMAwDBATG
hVADBAHGhVQDBADGhYwDBADGhc4DBADGheIwDAMEAMaHiQMEAMaHigMEAMaHpwME
AMaToDAMAwQAxpSxAwQAxpSyMAwDBAHGtJYDBAHGtJgDBADGzsUwDAMEA8bOyAME
AMbOygMEB8bwgAMEAccr9gMEAccw5gMDAMc1AwQDx1jQAwQEx1sQMAwDBAPHZwgD
BADHZwwDBAXH9wADBAPH9zgDBAXH+oADBADKAE0DBATLn1ADBAPLvjgDBALMCwAD
AwDMEgMEAcwwIAMEAMzh2gMEAM3JNwMEAM3TUwMEAc3c2AMEAM6nIQMEAs6+3AME
Bc7DIAMEBc784AMEBs9ZQAMEBc+WoAMEAs+u2AMEBc+yQAMEBs+0wAMEBM+9wDAM
AwQEz+VwAwQAz+V0MAwDBAHP5XYDBADP5XgwDAMEAc/legMEB8/lAAMEAtBSSAME
BtEqwAMEBdGigAMEBdHOAAMEAdHOJgMEBNHVMAMEBdH64DAMAwQA0fvDAwQB0fvE
AwQB0fv8AwQH1AAAMAwDBAXUAKADBAHUCOQwDAMEA9QI6AMEANQI8DAMAwQB1Ajy
AwQB1Aj8MAsDAwDUCQMEBdQMwDALAwMA1A0DBAXUFoAwDAMEBtQWwAMEBtQxADAM
AwQF1DFgAwQH1DQAMAwDBAXUNKADBAbUPAAwCwMEBdQ8YAMDBtQAMAwDBAfUQIAD
BATURYAwDAMEBdRFoAMEBtRVgDAMAwQF1FXgAwQF1FhAMAwDBAfUWIADBAHUXGQw
DAMEA9RcaAMEB9RfADALAwQF1F+gAwMF1EAwDAMEBdRgIAMEBtRkADAMAwQF1GRg
AwQF1GeAMAwDBAbUZ8ADBATUdSAwDAMEBtR1QAMEBdR6wDALAwMA1HsDBAfUgQAw
CgMDAdSCAwMA1NgwDAMEB9TZgAMEBtU3ADAMAwQH1TeAAwQG1YMAMAwDBAXVg2AD
BAXViEAwDAMEB9WIgAMEBtWTADAMAwQF1ZNgAwQF1ZZAAwQF1ZaAMAwDBAXVluAD
BAbVmAAwDAMEBdWYYAMEBdWaADAMAwQF1ZpgAwQF1Z6AMAwDBAbVnsADBAHVnwww
DAMEBNWfEAMEAtWfgDAMAwQD1Z+IAwQH1awAMAwDBAXVrKADBAXVs4AwDAMEBtWz
wAMEBdW1wDALAwMB1bYDBAXVwQAwDAMEBtXBQAMEBtXUgDAKAwMA1dUDAwDV9jAM
AwQF1fcgAwQH1f8AMAoDBAXV/6ADAgHUAwQB2C58AwQA2GPeAwQE2J5gAwQE2KxA
AwQG2NWAAwQF2PGAMAoDAgDZAwQE2Q5AMAwDBAXZDmADBAXZFMAwDAMEBNkU8AME
BNkVYDAMAwQH2RWAAwQH2R0AMAwDBATZHZADBATZHcAwCwMEBdkd4AMDAtkwMAsD
AwPZOAMEBdlAQDAMAwQE2UBwAwQG2U0AMAwDBATZTVADBAbZTgAwCwMEBNlOUAMD
ANl0MAsDBATZdRADAwDZijALAwMC2YwDBAPZk7AwDAMEBtmTwAMEBNmqgDALAwQF
2aqgAwMB2bAwCwMDANmzAwQE2ceAMAoDBAXZx6ADAgHYAwQC3J7EMGQEAgACMF4D
BQAgAQAFMA0DBAEgAQYDBQEgAQf4MA0DBQAgAQf7AwQCIAEIAwQCIAEUMAwDBAEg
ARoDBAEgAUADBAEgAUYwDAMEASABSgMEASABTAMEBCABUAMEBiADAAMDBCoAMIIG
oAYIKwYBBQUHAR0EggaSMIIGjqCCBoowggaGAgEHAgEcAgIAiQICAOAwCAICAPgC
AgD7AgIBBQICAR4CAgEgAgIBJgICAXcCAgF6AgICAQICAgUwCAICAhACAgIRAgIC
IAICAikCAgIvAgICNQICAk4CAgJRAgICnTAIAgICpwICAqgwCAICArcCAgK5MAgC
AgLFAgICxgICAsgCAgLPMAgCAgL4AgIC+QICAvwCAgL+MAgCAgMGAgIDDwICAxIw
CAICAxUCAgMWMAgCAgRNAgIEsAICBLMCAgS1AgIEvTAIAgIE0gICBNMCAgTZAgIE
4AICBOUCAgTpMAgCAgTzAgIE+wICBP8CAgUKAgIFETAIAgIFEwICBR0CAgUmAgIF
PjAIAgIFSAICBUkCAgYLMAgCAgZ1AgIGdgICBn8CAgaQMAgCAgarAgIGvgICBsEC
AgbEMAgCAgbKAgIGywICBs0CAgbUAgIG2DAIAgIG2gICBtwCAgbfAgIG5DAIAgIG
6gICBusCAgbuAgIG8AICBvQCAgcpMAgCAgcrAgIHLQICBzEwCAICBzkCAgc6MAgC
Agc9AgIHPjAIAgIHVQICB28wCAICB4ECAgeDAgIHhgICB4owCAICB48CAgejMAgC
AgeoAgIHqgICB68CAgfUAgIH3DAIAgIH4AICB+EwCAICB+oCAgftAgIH9DAIAgIH
9gICB/gCAgf7AgIH/QICB/8CAggBMAgCAggJAgIIWDAIAgIIYwICCGQwCAICCH4C
AgjhMAgCAgjmAgIJSQICCUwwCAICCVMCAgm4AgIJvjAIAgIJ4QICCeICAgntMAgC
AgnyAgIJ8wICChIwCAICChkCAgo2AgIKUwICClcCAgp7AgIKzjAIAgIK1QICCwYw
CAICCw4CAgs/AgILTwICC2UCAgtpAgIL8jAIAgIMCwICDCUCAgxPMAgCAgxSAgIM
hzAIAgIMiQICDRkwCAICDVQCAg1XAgIOKAICDwMwCAICD00CAg9OAgIQNDAIAgIR
NQICEU4wCAICEWkCAhFqAgIRrDAIAgIR7AICEe0CAhNuAgIT4TAIAgIVAQICFZ8w
CAICFaECAhX/AgIXswICF8UCAhgYAgIYsAICGQwwCAICGgACAho4MAgCAho6AgIa
3jAIAgIa4AICGv8CAh+dMAgCAiAAAgIhSzAIAgIhTQICIkEwCAICIkMCAiOoMAgC
AiOqAgIj/wICLE0CAi2MAgIvDjAIAgIwAAICMKYwCAICMKgCAjELMAgCAjENAgIz
pzAIAgIzqQICM/8CAjY3MAgCAjwAAgI8JjAIAgI8KAICPHIwCAICPHQCAj1ZMAgC
Aj1bAgI9uzAIAgI9vQICPdAwCAICPdICAj3ZMAgCAj3bAgI+WzAIAgI+XQICPrkw
CAICPrsCAj9VMAgCAj9XAgI/mzAIAgI/nQICP/8CAkksAgJK6gICS7ACAkvHMAgC
AlAAAgJQAzAIAgJQBQICUXkwCAICUXsCAlG/MAgCAlHBAgJSCjAIAgJSDAICUp8w
CAICUqECAlL5MAgCAlL7AgJTFjAIAgJTGAICUx0CAlMfMAgCAlMhAgJTjjAIAgJT
kAICU8swCAICU80CAlP/AgJWXAICWGMCAlibAgJayjAIAgJgAAICYJ8wCAICYKEC
AmC0MAgCAmC2AgJg0zAIAgJg1QICYOAwCAICYOICAmECMAgCAmEEAgJhHjAIAgJh
IAICYS0wCAICYS8CAmGaMAgCAmGcAgJiSjAIAgJiTAICYqEwCAICYqMCAmMRAgJj
EzAIAgJjFQICY8YwCAICY8gCAmPfMAgCAmPhAgJj5zAIAgJj6QICY/8CAmUYMAgC
AnAAAgJwCjAIAgJwDAICcBkwCAICcBsCAnDwMAgCAnDyAgJxojAIAgJxpAICcpkC
AnKbMAgCAnKdAgJy8zAIAgJy9QICczYwCAICczgCAnNnMAgCAnNpAgJzgjAIAgJz
hAICc60wCAICc68CAnPpMAgCAnPrAgJz/zAIAgJ4AAICeK8wCAICeLECAnkDAgJ5
BTAIAgJ5GAICeVgwCAICeVoCAnoMMAgCAnoOAgJ7gjAIAgJ7hAICe/8wCgIDAIQA
AgMAi/8wCgIDAJgAAgMAm/8wCgIDAKAAAgMApY0wCgIDAKWPAgMAr/8wCgIDALgA
AgMAy/8wCgIDANwAAgMA4/8wCgIDAOgAAgMA7/8wCgIDAPIAAgMA8/8wCgIDAPuM
AgMA++8wCgIDAwAAAgMDNZs=
"""

    def setUp(self):
        self.asn1Spec = rfc5280.Extensions()

    def testDerCodec(self):
        substrate = pem.readBase64fromText(self.extns_pem_text)
        asn1Object, rest = der_decoder(substrate, asn1Spec=self.asn1Spec)

        self.assertFalse(rest)
        self.assertTrue(asn1Object.prettyPrint())
        self.assertEqual(substrate, der_encoder(asn1Object))

        oids = []
        for extn in asn1Object:
            oids.append(extn["extnID"])
            extn_value, rest = der_decoder(
                extn["extnValue"], rfc5280.certificateExtensionsMap[extn["extnID"]]
            )

            self.assertFalse(rest)
            self.assertTrue(extn_value.prettyPrint())
            self.assertEqual(extn["extnValue"], der_encoder(extn_value))

        self.assertIn(rfc8360.id_pe_ipAddrBlocks_v2, oids)
        self.assertIn(rfc8360.id_pe_autonomousSysIds_v2, oids)


suite = unittest.TestLoader().loadTestsFromModule(sys.modules[__name__])

if __name__ == "__main__":
    result = unittest.TextTestRunner(verbosity=2).run(suite)
    sys.exit(not result.wasSuccessful())
