% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/principal_components.R, R/utils_pca_efa.R
\name{principal_components}
\alias{principal_components}
\alias{closest_component}
\alias{rotated_data}
\alias{predict.parameters_efa}
\title{Principal Component Analysis (PCA)}
\usage{
principal_components(
  x,
  n = "auto",
  rotation = "none",
  sort = FALSE,
  threshold = NULL,
  standardize = TRUE,
  ...
)

closest_component(x)

rotated_data(x)

\method{predict}{parameters_efa}(object, newdata = NULL, names = NULL, keep_na = TRUE, ...)
}
\arguments{
\item{x}{A data frame or a statistical model.}

\item{n}{Number of components to extract. If \code{n="all"}, then \code{n} is set as the number of variables minus 1 (\code{ncol(x)-1}). If \code{n="auto"} (default) or \code{n=NULL}, the number of components is selected through \code{\link{n_factors}} resp. \code{\link{n_components}}. In \code{\link{reduce_parameters}}, can also be \code{"max"}, in which case it will select all the components that are maximally pseudo-loaded (i.e., correlated) by at least one variable.}

\item{rotation}{If not \code{"none"}, the PCA / FA will be computed using the \pkg{psych} package. Possible options include \code{"varimax"}, \code{"quartimax"}, \code{"promax"}, \code{"oblimin"}, \code{"simplimax"}, or \code{"cluster"} (and more). See \code{\link[psych]{fa}} for details.}

\item{sort}{Sort the loadings.}

\item{threshold}{A value between 0 and 1 indicates which (absolute) values from the loadings should be removed. An integer higher than 1 indicates the n strongest loadings to retain. Can also be \code{"max"}, in which case it will only display the maximum loading per variable (the most simple structure).}

\item{standardize}{A logical value indicating whether the variables should be standardized (centered and scaled) to have unit variance before the analysis takes place (in general, such scaling is advisable).}

\item{...}{Arguments passed to or from other methods.}

\item{object}{An object of class \code{parameters_pca} or \code{parameters_efa}}

\item{newdata}{An optional data frame in which to look for variables with which to predict. If omitted, the fitted values are used.}

\item{names}{Optional character vector to name columns of the returned data frame.}

\item{keep_na}{Logical, if \code{TRUE}, predictions also return observations with missing values from the original data, hence the number of rows of predicted data and original data is equal.}
}
\value{
A data frame of loadings.
}
\description{
This function performs a principal component analysis (PCA) and returns the loadings as a data frame.
}
\details{
\subsection{Complexity}{
   Complexity represents the number of latent components needed to account
   for the observed variables. Whereas a perfect simple structure solution
   has a complexity of 1 in that each item would only load on one factor,
   a solution with evenly distributed items has a complexity greater than 1
   (\cite{Hofman, 1978; Pettersson and Turkheimer, 2010}) .
 }
 \subsection{Uniqueness}{
   Uniqueness represents the variance that is 'unique' to the variable and
   not shared with other variables. It is equal to \code{1 – communality}
   (variance that is shared with other variables). A uniqueness of \code{0.20}
   suggests that 20\% or that variable's variance is not shared with other
   variables in the overall factor model. The greater 'uniqueness' the lower
   the relevance of the variable in the factor model.
 }
 \subsection{MSA}{
   MSA represents the Kaiser-Meyer-Olkin Measure of Sampling Adequacy
   (\cite{Kaiser and Rice, 1974}) for each item. It indicates whether there
   is enough data for each factor give reliable results for the PCA. The
   value should be > 0.6, and desirable values are > 0.8
   (\cite{Tabachnick and Fidell, 2013}).
 }
 \subsection{PCA or FA?}{
 There is a simplified rule of thumb that may help do decide whether to run
 a factor analysis or a principal component analysis:
 \itemize{
   \item Run \emph{factor analysis} if you assume or wish to test a theoretical model of latent factors causing observed variables.
   \item Run \emph{principal component analysis} If you want to simply reduce your correlated observed variables to a smaller set of important independent composite variables.
 }
 (Source: \href{https://stats.stackexchange.com/q/1576/54740}{CrossValidated})
 }
 \subsection{Computing Item Scores}{
   Use \code{\link{get_scores}} to compute scores for the "subscales" represented
   by the extracted principal components. \code{get_scores()} takes the results
   from \code{principal_components()} and extracts the variables for each
   component found by the PCA. Then, for each of these "subscales", row means
   are calculated (which equals adding up the single items and dividing by
   the number of items). This results in a sum score for each component from
   the PCA, which is on the same scale as the original, single items that were
   used to compute the PCA.

   One can also use \code{predict()} to back-predict scores for each component,
   to which one can provide \code{newdata} or a vector of \code{names} for the
   components.
 }
}
\note{
There is a \code{summary()}-method that prints the Eigenvalues and (explained) variance for each extracted component. \code{closest_component()} will return a numeric vector with the assigned component index for each column from the original data frame. \code{rotated_data()} will return the rotated data, including missing values, so it matches the original data frame. There is also a \href{https://easystats.github.io/see/articles/parameters.html}{\code{plot()}-method} implemented in the \href{https://easystats.github.io/see/}{\pkg{see}-package}.
}
\examples{
\donttest{
library(parameters)
if (require("psych")) {
  principal_components(mtcars[, 1:7], n = "all", threshold = 0.2)
  principal_components(mtcars[, 1:7],
    n = 2, rotation = "oblimin",
    threshold = "max", sort = TRUE
  )
  principal_components(mtcars[, 1:7], n = 2, threshold = 2, sort = TRUE)

  pca <- principal_components(mtcars[, 1:5], n = 2, rotation = "varimax")
  pca # Print loadings
  summary(pca) # Print information about the factors
  predict(pca, names = c("Component1", "Component2")) # Back-predict scores

  # which variables from the original data belong to which extracted component?
  closest_component(pca)
}

# Automated number of components
principal_components(mtcars[, 1:4], n = "auto")
}
}
\references{
\itemize{
  \item Kaiser, H.F. and Rice. J. (1974). Little jiffy, mark iv. Educational and Psychological Measurement, 34(1):111–117
  \item Hofmann, R. (1978). Complexity and simplicity as objective indices descriptive of factor solutions. Multivariate Behavioral Research, 13:2, 247-250, \doi{10.1207/s15327906mbr1302_9}
  \item Pettersson, E., & Turkheimer, E. (2010). Item selection, evaluation, and simple structure in personality data. Journal of research in personality, 44(4), 407-420, \doi{10.1016/j.jrp.2010.03.002}
  \item Tabachnick, B. G., and Fidell, L. S. (2013). Using multivariate statistics (6th ed.). Boston: Pearson Education.
}
}
\seealso{
\code{\link[performance]{check_itemscale}} to compute various measures of internal consistencies applied to the (sub)scales (i.e. components) extracted from the PCA. Use \code{\link{get_scores}} to compute scores for each subscale.
}
