#
# = sample/test_restriction_enzyme_long.rb - Benchmark tests for Bio::RestrictionEnzyme::Analysis.cut for long sequences
#
# Copyright::   Copyright (C) 2011
#               Naohisa Goto <ng@bioruby.org>
# License::     The Ruby License
#
# Acknowledgements: The idea of the test is based on the issue report
# https://github.com/bioruby/bioruby/issues/10
# posted by ray1729 (https://github.com/ray1729).
#

require 'test/unit'
require 'benchmark'
require 'bio'

entry = Bio::TogoWS::REST.entry('ncbi-nucleotide', 'BA000007.2')
EcoliO157H7Seq = Bio::GenBank.new(entry).naseq.freeze

module TestRestrictionEnzymeAnalysisCutLong

  # dummy benchmarch class
  class DummyBench
    def report(str); yield; end
  end

  module HelperMethods
    def _truncate_cut_ranges(cut_ranges, len)
      limit = len - 1
      ret = cut_ranges.collect do |a|
        if a[0] > limit || a[2] > limit then
          nil
        else
          a.collect { |pos| pos > limit ? limit : pos }
        end
      end
      ret.compact!
      if last_a = ret[-1] then
        last_a[1] = limit
        last_a[3] = limit
      end
      ret
    end

    def _collect_cut_ranges(cuts)
      cuts.collect do |f|
                [ f.p_left, f.p_right, f.c_left, f.c_right ]
      end
    end

    def _test_by_size(len, bench = DummyBench.new)
      cuts = nil
      bench.report("#{self.class::TestLabel} #{len}") {
        cuts = _cut(self.class::SampleSequence[0, len])
      }
      cut_ranges = _collect_cut_ranges(cuts)
      expected = _truncate_cut_ranges(self.class::SampleCutRanges, len)
      assert_equal(expected, cut_ranges)
    end

    def test_10k_to_100k
      $stderr.print "\n"
      Benchmark.bm(26) do |bench|
        10_000.step(100_000, 10_000) do |len|
          _test_by_size(len, bench)
        end
      end
    end

    def test_100k_to_1M
      $stderr.print "\n"
      Benchmark.bm(26) do |bench|
        100_000.step(1_000_000, 100_000) do |len|
          _test_by_size(len, bench)
        end
      end
    end

    def test_1M_to_5M_and_whole
      $stderr.print "\n"
      Benchmark.bm(26) do |bench|
        1_000_000.step(5_000_000, 1_000_000) do |len|
          _test_by_size(len, bench)
        end
        _test_by_size(self.class::SampleSequence.length, bench)
      end
    end if defined? Bio::RestrictionEnzyme::SortedNumArray

    def disabled_test_whole
      cuts = _cut(self.class::SampleSequence)
      cut_ranges = _collect_cut_ranges(cuts)
      cut_ranges.each do |a|
        $stderr.print "        [ ", a.join(", "), " ], \n"
      end
      assert_equal(self.class::SampleCutRanges, cut_ranges)
    end
  end #module HelperMethods


  class TestEcoliO157H7_BstEII < Test::Unit::TestCase

    include HelperMethods

    TestLabel = 'BstEII'

    SampleSequence = EcoliO157H7Seq

    SampleCutRanges = BstEII_WHOLE =
      [ [ 0, 79, 0, 84 ], 
        [ 80, 4612, 85, 4617 ], 
        [ 4613, 13483, 4618, 13488 ], 
        [ 13484, 15984, 13489, 15989 ], 
        [ 15985, 21462, 15990, 21467 ], 
        [ 21463, 27326, 21468, 27331 ], 
        [ 27327, 30943, 27332, 30948 ], 
        [ 30944, 34888, 30949, 34893 ], 
        [ 34889, 35077, 34894, 35082 ], 
        [ 35078, 35310, 35083, 35315 ], 
        [ 35311, 36254, 35316, 36259 ], 
        [ 36255, 41885, 36260, 41890 ], 
        [ 41886, 43070, 41891, 43075 ], 
        [ 43071, 45689, 43076, 45694 ], 
        [ 45690, 52325, 45695, 52330 ], 
        [ 52326, 55703, 52331, 55708 ], 
        [ 55704, 58828, 55709, 58833 ], 
        [ 58829, 59178, 58834, 59183 ], 
        [ 59179, 72610, 59184, 72615 ], 
        [ 72611, 72739, 72616, 72744 ], 
        [ 72740, 73099, 72745, 73104 ], 
        [ 73100, 75123, 73105, 75128 ], 
        [ 75124, 77366, 75129, 77371 ], 
        [ 77367, 77810, 77372, 77815 ], 
        [ 77811, 78740, 77816, 78745 ], 
        [ 78741, 79717, 78746, 79722 ], 
        [ 79718, 82250, 79723, 82255 ], 
        [ 82251, 84604, 82256, 84609 ], 
        [ 84605, 95491, 84610, 95496 ], 
        [ 95492, 95785, 95497, 95790 ], 
        [ 95786, 95794, 95791, 95799 ], 
        [ 95795, 96335, 95800, 96340 ], 
        [ 96336, 102044, 96341, 102049 ], 
        [ 102045, 102541, 102050, 102546 ], 
        [ 102542, 103192, 102547, 103197 ], 
        [ 103193, 104722, 103198, 104727 ], 
        [ 104723, 110883, 104728, 110888 ], 
        [ 110884, 120090, 110889, 120095 ], 
        [ 120091, 120657, 120096, 120662 ], 
        [ 120658, 128308, 120663, 128313 ], 
        [ 128309, 138305, 128314, 138310 ], 
        [ 138306, 141147, 138311, 141152 ], 
        [ 141148, 143724, 141153, 143729 ], 
        [ 143725, 143838, 143730, 143843 ], 
        [ 143839, 144303, 143844, 144308 ], 
        [ 144304, 148199, 144309, 148204 ], 
        [ 148200, 149577, 148205, 149582 ], 
        [ 149578, 149731, 149583, 149736 ], 
        [ 149732, 156115, 149737, 156120 ], 
        [ 156116, 161126, 156121, 161131 ], 
        [ 161127, 162856, 161132, 162861 ], 
        [ 162857, 170693, 162862, 170698 ], 
        [ 170694, 170944, 170699, 170949 ], 
        [ 170945, 171201, 170950, 171206 ], 
        [ 171202, 173241, 171207, 173246 ], 
        [ 173242, 177283, 173247, 177288 ], 
        [ 177284, 178177, 177289, 178182 ], 
        [ 178178, 178781, 178183, 178786 ], 
        [ 178782, 181610, 178787, 181615 ], 
        [ 181611, 181706, 181616, 181711 ], 
        [ 181707, 185661, 181712, 185666 ], 
        [ 185662, 193407, 185667, 193412 ], 
        [ 193408, 195511, 193413, 195516 ], 
        [ 195512, 195754, 195517, 195759 ], 
        [ 195755, 197247, 195760, 197252 ], 
        [ 197248, 200659, 197253, 200664 ], 
        [ 200660, 201820, 200665, 201825 ], 
        [ 201821, 202300, 201826, 202305 ], 
        [ 202301, 202686, 202306, 202691 ], 
        [ 202687, 206289, 202692, 206294 ], 
        [ 206290, 206466, 206295, 206471 ], 
        [ 206467, 207011, 206472, 207016 ], 
        [ 207012, 208159, 207017, 208164 ], 
        [ 208160, 209976, 208165, 209981 ], 
        [ 209977, 210078, 209982, 210083 ], 
        [ 210079, 211485, 210084, 211490 ], 
        [ 211486, 212377, 211491, 212382 ], 
        [ 212378, 213569, 212383, 213574 ], 
        [ 213570, 216005, 213575, 216010 ], 
        [ 216006, 220098, 216011, 220103 ], 
        [ 220099, 224063, 220104, 224068 ], 
        [ 224064, 228604, 224069, 228609 ], 
        [ 228605, 239993, 228610, 239998 ], 
        [ 239994, 247914, 239999, 247919 ], 
        [ 247915, 251579, 247920, 251584 ], 
        [ 251580, 257092, 251585, 257097 ], 
        [ 257093, 261621, 257098, 261626 ], 
        [ 261622, 263030, 261627, 263035 ], 
        [ 263031, 265084, 263036, 265089 ], 
        [ 265085, 265243, 265090, 265248 ], 
        [ 265244, 265534, 265249, 265539 ], 
        [ 265535, 266117, 265540, 266122 ], 
        [ 266118, 274428, 266123, 274433 ], 
        [ 274429, 282285, 274434, 282290 ], 
        [ 282286, 286948, 282291, 286953 ], 
        [ 286949, 292547, 286954, 292552 ], 
        [ 292548, 297678, 292553, 297683 ], 
        [ 297679, 308161, 297684, 308166 ], 
        [ 308162, 308706, 308167, 308711 ], 
        [ 308707, 313482, 308712, 313487 ], 
        [ 313483, 337118, 313488, 337123 ], 
        [ 337119, 337935, 337124, 337940 ], 
        [ 337936, 338781, 337941, 338786 ], 
        [ 338782, 339493, 338787, 339498 ], 
        [ 339494, 341025, 339499, 341030 ], 
        [ 341026, 344424, 341031, 344429 ], 
        [ 344425, 348384, 344430, 348389 ], 
        [ 348385, 354781, 348390, 354786 ], 
        [ 354782, 356692, 354787, 356697 ], 
        [ 356693, 357008, 356698, 357013 ], 
        [ 357009, 357305, 357014, 357310 ], 
        [ 357306, 357328, 357311, 357333 ], 
        [ 357329, 358126, 357334, 358131 ], 
        [ 358127, 359472, 358132, 359477 ], 
        [ 359473, 362160, 359478, 362165 ], 
        [ 362161, 365395, 362166, 365400 ], 
        [ 365396, 365704, 365401, 365709 ], 
        [ 365705, 381746, 365710, 381751 ], 
        [ 381747, 381994, 381752, 381999 ], 
        [ 381995, 383335, 382000, 383340 ], 
        [ 383336, 385141, 383341, 385146 ], 
        [ 385142, 390171, 385147, 390176 ], 
        [ 390172, 392764, 390177, 392769 ], 
        [ 392765, 394338, 392770, 394343 ], 
        [ 394339, 394686, 394344, 394691 ], 
        [ 394687, 398703, 394692, 398708 ], 
        [ 398704, 404095, 398709, 404100 ], 
        [ 404096, 408361, 404101, 408366 ], 
        [ 408362, 413032, 408367, 413037 ], 
        [ 413033, 414563, 413038, 414568 ], 
        [ 414564, 416901, 414569, 416906 ], 
        [ 416902, 417419, 416907, 417424 ], 
        [ 417420, 421777, 417425, 421782 ], 
        [ 421778, 423748, 421783, 423753 ], 
        [ 423749, 431903, 423754, 431908 ], 
        [ 431904, 440000, 431909, 440005 ], 
        [ 440001, 448040, 440006, 448045 ], 
        [ 448041, 452994, 448046, 452999 ], 
        [ 452995, 453075, 453000, 453080 ], 
        [ 453076, 454950, 453081, 454955 ], 
        [ 454951, 455888, 454956, 455893 ], 
        [ 455889, 460160, 455894, 460165 ], 
        [ 460161, 463076, 460166, 463081 ], 
        [ 463077, 465003, 463082, 465008 ], 
        [ 465004, 466828, 465009, 466833 ], 
        [ 466829, 467686, 466834, 467691 ], 
        [ 467687, 468596, 467692, 468601 ], 
        [ 468597, 479953, 468602, 479958 ], 
        [ 479954, 480538, 479959, 480543 ], 
        [ 480539, 482869, 480544, 482874 ], 
        [ 482870, 489378, 482875, 489383 ], 
        [ 489379, 492241, 489384, 492246 ], 
        [ 492242, 495406, 492247, 495411 ], 
        [ 495407, 495712, 495412, 495717 ], 
        [ 495713, 497829, 495718, 497834 ], 
        [ 497830, 501698, 497835, 501703 ], 
        [ 501699, 504565, 501704, 504570 ], 
        [ 504566, 505105, 504571, 505110 ], 
        [ 505106, 508452, 505111, 508457 ], 
        [ 508453, 515947, 508458, 515952 ], 
        [ 515948, 519141, 515953, 519146 ], 
        [ 519142, 519398, 519147, 519403 ], 
        [ 519399, 521386, 519404, 521391 ], 
        [ 521387, 526115, 521392, 526120 ], 
        [ 526116, 526729, 526121, 526734 ], 
        [ 526730, 527018, 526735, 527023 ], 
        [ 527019, 528059, 527024, 528064 ], 
        [ 528060, 532689, 528065, 532694 ], 
        [ 532690, 534702, 532695, 534707 ], 
        [ 534703, 535272, 534708, 535277 ], 
        [ 535273, 538668, 535278, 538673 ], 
        [ 538669, 543939, 538674, 543944 ], 
        [ 543940, 547429, 543945, 547434 ], 
        [ 547430, 553890, 547435, 553895 ], 
        [ 553891, 554678, 553896, 554683 ], 
        [ 554679, 555452, 554684, 555457 ], 
        [ 555453, 556296, 555458, 556301 ], 
        [ 556297, 559341, 556302, 559346 ], 
        [ 559342, 559991, 559347, 559996 ], 
        [ 559992, 563242, 559997, 563247 ], 
        [ 563243, 576432, 563248, 576437 ], 
        [ 576433, 582431, 576438, 582436 ], 
        [ 582432, 582959, 582437, 582964 ], 
        [ 582960, 583475, 582965, 583480 ], 
        [ 583476, 583589, 583481, 583594 ], 
        [ 583590, 583670, 583595, 583675 ], 
        [ 583671, 583901, 583676, 583906 ], 
        [ 583902, 584198, 583907, 584203 ], 
        [ 584199, 584633, 584204, 584638 ], 
        [ 584634, 585704, 584639, 585709 ], 
        [ 585705, 585746, 585710, 585751 ], 
        [ 585747, 586175, 585752, 586180 ], 
        [ 586176, 586301, 586181, 586306 ], 
        [ 586302, 586643, 586307, 586648 ], 
        [ 586644, 586775, 586649, 586780 ], 
        [ 586776, 587072, 586781, 587077 ], 
        [ 587073, 587214, 587078, 587219 ], 
        [ 587215, 587540, 587220, 587545 ], 
        [ 587541, 587969, 587546, 587974 ], 
        [ 587970, 588095, 587975, 588100 ], 
        [ 588096, 588437, 588101, 588442 ], 
        [ 588438, 588569, 588443, 588574 ], 
        [ 588570, 589008, 588575, 589013 ], 
        [ 589009, 589166, 589014, 589171 ], 
        [ 589167, 590366, 589172, 590371 ], 
        [ 590367, 590792, 590372, 590797 ], 
        [ 590793, 591077, 590798, 591082 ], 
        [ 591078, 591263, 591083, 591268 ], 
        [ 591264, 591863, 591269, 591868 ], 
        [ 591864, 592058, 591869, 592063 ], 
        [ 592059, 592160, 592064, 592165 ], 
        [ 592161, 592568, 592166, 592573 ], 
        [ 592569, 592760, 592574, 592765 ], 
        [ 592761, 593060, 592766, 593065 ], 
        [ 593061, 593186, 593066, 593191 ], 
        [ 593187, 593366, 593192, 593371 ], 
        [ 593367, 593957, 593372, 593962 ], 
        [ 593958, 594827, 593963, 594832 ], 
        [ 594828, 594980, 594833, 594985 ], 
        [ 594981, 595649, 594986, 595654 ], 
        [ 595650, 595893, 595655, 595898 ], 
        [ 595894, 596057, 595899, 596062 ], 
        [ 596058, 596159, 596063, 596164 ], 
        [ 596160, 596351, 596165, 596356 ], 
        [ 596352, 596660, 596357, 596665 ], 
        [ 596661, 596960, 596666, 596965 ], 
        [ 596961, 597102, 596966, 597107 ], 
        [ 597103, 597155, 597108, 597160 ], 
        [ 597156, 597257, 597161, 597262 ], 
        [ 597258, 599957, 597263, 599962 ], 
        [ 599958, 611038, 599963, 611043 ], 
        [ 611039, 612202, 611044, 612207 ], 
        [ 612203, 614051, 612208, 614056 ], 
        [ 614052, 614134, 614057, 614139 ], 
        [ 614135, 614787, 614140, 614792 ], 
        [ 614788, 616272, 614793, 616277 ], 
        [ 616273, 617737, 616278, 617742 ], 
        [ 617738, 627339, 617743, 627344 ], 
        [ 627340, 628902, 627345, 628907 ], 
        [ 628903, 636523, 628908, 636528 ], 
        [ 636524, 637529, 636529, 637534 ], 
        [ 637530, 647713, 637535, 647718 ], 
        [ 647714, 648684, 647719, 648689 ], 
        [ 648685, 653543, 648690, 653548 ], 
        [ 653544, 659030, 653549, 659035 ], 
        [ 659031, 662241, 659036, 662246 ], 
        [ 662242, 671781, 662247, 671786 ], 
        [ 671782, 672048, 671787, 672053 ], 
        [ 672049, 673788, 672054, 673793 ], 
        [ 673789, 674707, 673794, 674712 ], 
        [ 674708, 674998, 674713, 675003 ], 
        [ 674999, 675157, 675004, 675162 ], 
        [ 675158, 688595, 675163, 688600 ], 
        [ 688596, 693309, 688601, 693314 ], 
        [ 693310, 697406, 693315, 697411 ], 
        [ 697407, 702676, 697412, 702681 ], 
        [ 702677, 707382, 702682, 707387 ], 
        [ 707383, 708604, 707388, 708609 ], 
        [ 708605, 710046, 708610, 710051 ], 
        [ 710047, 711630, 710052, 711635 ], 
        [ 711631, 711696, 711636, 711701 ], 
        [ 711697, 712329, 711702, 712334 ], 
        [ 712330, 716461, 712335, 716466 ], 
        [ 716462, 720238, 716467, 720243 ], 
        [ 720239, 720374, 720244, 720379 ], 
        [ 720375, 724200, 720380, 724205 ], 
        [ 724201, 725687, 724206, 725692 ], 
        [ 725688, 730067, 725693, 730072 ], 
        [ 730068, 730574, 730073, 730579 ], 
        [ 730575, 730699, 730580, 730704 ], 
        [ 730700, 732726, 730705, 732731 ], 
        [ 732727, 738597, 732732, 738602 ], 
        [ 738598, 743326, 738603, 743331 ], 
        [ 743327, 744992, 743332, 744997 ], 
        [ 744993, 745843, 744998, 745848 ], 
        [ 745844, 751518, 745849, 751523 ], 
        [ 751519, 752431, 751524, 752436 ], 
        [ 752432, 752549, 752437, 752554 ], 
        [ 752550, 766036, 752555, 766041 ], 
        [ 766037, 768968, 766042, 768973 ], 
        [ 768969, 770151, 768974, 770156 ], 
        [ 770152, 771158, 770157, 771163 ], 
        [ 771159, 771405, 771164, 771410 ], 
        [ 771406, 781958, 771411, 781963 ], 
        [ 781959, 784226, 781964, 784231 ], 
        [ 784227, 786945, 784232, 786950 ], 
        [ 786946, 787203, 786951, 787208 ], 
        [ 787204, 789251, 787209, 789256 ], 
        [ 789252, 791218, 789257, 791223 ], 
        [ 791219, 793716, 791224, 793721 ], 
        [ 793717, 795003, 793722, 795008 ], 
        [ 795004, 795521, 795009, 795526 ], 
        [ 795522, 804514, 795527, 804519 ], 
        [ 804515, 805238, 804520, 805243 ], 
        [ 805239, 805887, 805244, 805892 ], 
        [ 805888, 808461, 805893, 808466 ], 
        [ 808462, 809805, 808467, 809810 ], 
        [ 809806, 810086, 809811, 810091 ], 
        [ 810087, 810726, 810092, 810731 ], 
        [ 810727, 820111, 810732, 820116 ], 
        [ 820112, 821326, 820117, 821331 ], 
        [ 821327, 821647, 821332, 821652 ], 
        [ 821648, 824277, 821653, 824282 ], 
        [ 824278, 825750, 824283, 825755 ], 
        [ 825751, 828770, 825756, 828775 ], 
        [ 828771, 828924, 828776, 828929 ], 
        [ 828925, 830194, 828930, 830199 ], 
        [ 830195, 830786, 830200, 830791 ], 
        [ 830787, 832788, 830792, 832793 ], 
        [ 832789, 833306, 832794, 833311 ], 
        [ 833307, 835656, 833312, 835661 ], 
        [ 835657, 841180, 835662, 841185 ], 
        [ 841181, 842112, 841186, 842117 ], 
        [ 842113, 843973, 842118, 843978 ], 
        [ 843974, 843990, 843979, 843995 ], 
        [ 843991, 852882, 843996, 852887 ], 
        [ 852883, 854392, 852888, 854397 ], 
        [ 854393, 857721, 854398, 857726 ], 
        [ 857722, 857961, 857727, 857966 ], 
        [ 857962, 862783, 857967, 862788 ], 
        [ 862784, 878953, 862789, 878958 ], 
        [ 878954, 885194, 878959, 885199 ], 
        [ 885195, 886313, 885200, 886318 ], 
        [ 886314, 886460, 886319, 886465 ], 
        [ 886461, 890233, 886466, 890238 ], 
        [ 890234, 890346, 890239, 890351 ], 
        [ 890347, 890379, 890352, 890384 ], 
        [ 890380, 899676, 890385, 899681 ], 
        [ 899677, 903962, 899682, 903967 ], 
        [ 903963, 904236, 903968, 904241 ], 
        [ 904237, 908130, 904242, 908135 ], 
        [ 908131, 916611, 908136, 916616 ], 
        [ 916612, 916803, 916617, 916808 ], 
        [ 916804, 920531, 916809, 920536 ], 
        [ 920532, 928505, 920537, 928510 ], 
        [ 928506, 936947, 928511, 936952 ], 
        [ 936948, 937240, 936953, 937245 ], 
        [ 937241, 939698, 937246, 939703 ], 
        [ 939699, 939711, 939704, 939716 ], 
        [ 939712, 941642, 939717, 941647 ], 
        [ 941643, 949052, 941648, 949057 ], 
        [ 949053, 949800, 949058, 949805 ], 
        [ 949801, 951412, 949806, 951417 ], 
        [ 951413, 951810, 951418, 951815 ], 
        [ 951811, 952386, 951816, 952391 ], 
        [ 952387, 953295, 952392, 953300 ], 
        [ 953296, 953894, 953301, 953899 ], 
        [ 953895, 958753, 953900, 958758 ], 
        [ 958754, 964476, 958759, 964481 ], 
        [ 964477, 967468, 964482, 967473 ], 
        [ 967469, 969631, 967474, 969636 ], 
        [ 969632, 970966, 969637, 970971 ], 
        [ 970967, 971138, 970972, 971143 ], 
        [ 971139, 974185, 971144, 974190 ], 
        [ 974186, 974365, 974191, 974370 ], 
        [ 974366, 975256, 974371, 975261 ], 
        [ 975257, 976794, 975262, 976799 ], 
        [ 976795, 987406, 976800, 987411 ], 
        [ 987407, 988132, 987412, 988137 ], 
        [ 988133, 992809, 988138, 992814 ], 
        [ 992810, 1000225, 992815, 1000230 ], 
        [ 1000226, 1001626, 1000231, 1001631 ], 
        [ 1001627, 1007354, 1001632, 1007359 ], 
        [ 1007355, 1011910, 1007360, 1011915 ], 
        [ 1011911, 1012377, 1011916, 1012382 ], 
        [ 1012378, 1017328, 1012383, 1017333 ], 
        [ 1017329, 1020891, 1017334, 1020896 ], 
        [ 1020892, 1021340, 1020897, 1021345 ], 
        [ 1021341, 1024845, 1021346, 1024850 ], 
        [ 1024846, 1025853, 1024851, 1025858 ], 
        [ 1025854, 1030691, 1025859, 1030696 ], 
        [ 1030692, 1032676, 1030697, 1032681 ], 
        [ 1032677, 1037847, 1032682, 1037852 ], 
        [ 1037848, 1039473, 1037853, 1039478 ], 
        [ 1039474, 1044241, 1039479, 1044246 ], 
        [ 1044242, 1045920, 1044247, 1045925 ], 
        [ 1045921, 1053286, 1045926, 1053291 ], 
        [ 1053287, 1053309, 1053292, 1053314 ], 
        [ 1053310, 1054643, 1053315, 1054648 ], 
        [ 1054644, 1056527, 1054649, 1056532 ], 
        [ 1056528, 1058682, 1056533, 1058687 ], 
        [ 1058683, 1059297, 1058688, 1059302 ], 
        [ 1059298, 1060416, 1059303, 1060421 ], 
        [ 1060417, 1064234, 1060422, 1064239 ], 
        [ 1064235, 1064848, 1064240, 1064853 ], 
        [ 1064849, 1065434, 1064854, 1065439 ], 
        [ 1065435, 1075642, 1065440, 1075647 ], 
        [ 1075643, 1076325, 1075648, 1076330 ], 
        [ 1076326, 1076534, 1076331, 1076539 ], 
        [ 1076535, 1078866, 1076540, 1078871 ], 
        [ 1078867, 1080537, 1078872, 1080542 ], 
        [ 1080538, 1082144, 1080543, 1082149 ], 
        [ 1082145, 1085746, 1082150, 1085751 ], 
        [ 1085747, 1087087, 1085752, 1087092 ], 
        [ 1087088, 1088273, 1087093, 1088278 ], 
        [ 1088274, 1093062, 1088279, 1093067 ], 
        [ 1093063, 1096867, 1093068, 1096872 ], 
        [ 1096868, 1102488, 1096873, 1102493 ], 
        [ 1102489, 1106371, 1102494, 1106376 ], 
        [ 1106372, 1108123, 1106377, 1108128 ], 
        [ 1108124, 1113311, 1108129, 1113316 ], 
        [ 1113312, 1114557, 1113317, 1114562 ], 
        [ 1114558, 1120566, 1114563, 1120571 ], 
        [ 1120567, 1121004, 1120572, 1121009 ], 
        [ 1121005, 1122501, 1121010, 1122506 ], 
        [ 1122502, 1130582, 1122507, 1130587 ], 
        [ 1130583, 1132170, 1130588, 1132175 ], 
        [ 1132171, 1140126, 1132176, 1140131 ], 
        [ 1140127, 1143361, 1140132, 1143366 ], 
        [ 1143362, 1149205, 1143367, 1149210 ], 
        [ 1149206, 1149331, 1149211, 1149336 ], 
        [ 1149332, 1156272, 1149337, 1156277 ], 
        [ 1156273, 1161624, 1156278, 1161629 ], 
        [ 1161625, 1171353, 1161630, 1171358 ], 
        [ 1171354, 1171934, 1171359, 1171939 ], 
        [ 1171935, 1172114, 1171940, 1172119 ], 
        [ 1172115, 1185368, 1172120, 1185373 ], 
        [ 1185369, 1193993, 1185374, 1193998 ], 
        [ 1193994, 1194272, 1193999, 1194277 ], 
        [ 1194273, 1197920, 1194278, 1197925 ], 
        [ 1197921, 1200373, 1197926, 1200378 ], 
        [ 1200374, 1200597, 1200379, 1200602 ], 
        [ 1200598, 1200714, 1200603, 1200719 ], 
        [ 1200715, 1203674, 1200720, 1203679 ], 
        [ 1203675, 1204865, 1203680, 1204870 ], 
        [ 1204866, 1205330, 1204871, 1205335 ], 
        [ 1205331, 1210727, 1205336, 1210732 ], 
        [ 1210728, 1211881, 1210733, 1211886 ], 
        [ 1211882, 1214283, 1211887, 1214288 ], 
        [ 1214284, 1216981, 1214289, 1216986 ], 
        [ 1216982, 1223522, 1216987, 1223527 ], 
        [ 1223523, 1228205, 1223528, 1228210 ], 
        [ 1228206, 1236067, 1228211, 1236072 ], 
        [ 1236068, 1236265, 1236073, 1236270 ], 
        [ 1236266, 1239969, 1236271, 1239974 ], 
        [ 1239970, 1240641, 1239975, 1240646 ], 
        [ 1240642, 1244738, 1240647, 1244743 ], 
        [ 1244739, 1244821, 1244744, 1244826 ], 
        [ 1244822, 1272971, 1244827, 1272976 ], 
        [ 1272972, 1276524, 1272977, 1276529 ], 
        [ 1276525, 1290344, 1276530, 1290349 ], 
        [ 1290345, 1292253, 1290350, 1292258 ], 
        [ 1292254, 1293482, 1292259, 1293487 ], 
        [ 1293483, 1295919, 1293488, 1295924 ], 
        [ 1295920, 1302834, 1295925, 1302839 ], 
        [ 1302835, 1303464, 1302840, 1303469 ], 
        [ 1303465, 1309308, 1303470, 1309313 ], 
        [ 1309309, 1311482, 1309314, 1311487 ], 
        [ 1311483, 1312493, 1311488, 1312498 ], 
        [ 1312494, 1316488, 1312499, 1316493 ], 
        [ 1316489, 1318127, 1316494, 1318132 ], 
        [ 1318128, 1325643, 1318133, 1325648 ], 
        [ 1325644, 1328313, 1325649, 1328318 ], 
        [ 1328314, 1345348, 1328319, 1345353 ], 
        [ 1345349, 1347480, 1345354, 1347485 ], 
        [ 1347481, 1348458, 1347486, 1348463 ], 
        [ 1348459, 1350595, 1348464, 1350600 ], 
        [ 1350596, 1350770, 1350601, 1350775 ], 
        [ 1350771, 1351954, 1350776, 1351959 ], 
        [ 1351955, 1356474, 1351960, 1356479 ], 
        [ 1356475, 1362756, 1356480, 1362761 ], 
        [ 1362757, 1368544, 1362762, 1368549 ], 
        [ 1368545, 1377993, 1368550, 1377998 ], 
        [ 1377994, 1379610, 1377999, 1379615 ], 
        [ 1379611, 1391551, 1379616, 1391556 ], 
        [ 1391552, 1395841, 1391557, 1395846 ], 
        [ 1395842, 1401721, 1395847, 1401726 ], 
        [ 1401722, 1406871, 1401727, 1406876 ], 
        [ 1406872, 1411041, 1406877, 1411046 ], 
        [ 1411042, 1417851, 1411047, 1417856 ], 
        [ 1417852, 1419058, 1417857, 1419063 ], 
        [ 1419059, 1428120, 1419064, 1428125 ], 
        [ 1428121, 1428584, 1428126, 1428589 ], 
        [ 1428585, 1430700, 1428590, 1430705 ], 
        [ 1430701, 1438278, 1430706, 1438283 ], 
        [ 1438279, 1443084, 1438284, 1443089 ], 
        [ 1443085, 1444668, 1443090, 1444673 ], 
        [ 1444669, 1444866, 1444674, 1444871 ], 
        [ 1444867, 1444914, 1444872, 1444919 ], 
        [ 1444915, 1445093, 1444920, 1445098 ], 
        [ 1445094, 1446216, 1445099, 1446221 ], 
        [ 1446217, 1448518, 1446222, 1448523 ], 
        [ 1448519, 1452860, 1448524, 1452865 ], 
        [ 1452861, 1454246, 1452866, 1454251 ], 
        [ 1454247, 1455414, 1454252, 1455419 ], 
        [ 1455415, 1460976, 1455420, 1460981 ], 
        [ 1460977, 1461164, 1460982, 1461169 ], 
        [ 1461165, 1463675, 1461170, 1463680 ], 
        [ 1463676, 1465339, 1463681, 1465344 ], 
        [ 1465340, 1469872, 1465345, 1469877 ], 
        [ 1469873, 1471479, 1469878, 1471484 ], 
        [ 1471480, 1472745, 1471485, 1472750 ], 
        [ 1472746, 1479208, 1472751, 1479213 ], 
        [ 1479209, 1480831, 1479214, 1480836 ], 
        [ 1480832, 1485359, 1480837, 1485364 ], 
        [ 1485360, 1485530, 1485365, 1485535 ], 
        [ 1485531, 1486004, 1485536, 1486009 ], 
        [ 1486005, 1487314, 1486010, 1487319 ], 
        [ 1487315, 1491008, 1487320, 1491013 ], 
        [ 1491009, 1492068, 1491014, 1492073 ], 
        [ 1492069, 1493001, 1492074, 1493006 ], 
        [ 1493002, 1495524, 1493007, 1495529 ], 
        [ 1495525, 1498599, 1495530, 1498604 ], 
        [ 1498600, 1499384, 1498605, 1499389 ], 
        [ 1499385, 1500494, 1499390, 1500499 ], 
        [ 1500495, 1504828, 1500500, 1504833 ], 
        [ 1504829, 1509790, 1504834, 1509795 ], 
        [ 1509791, 1512050, 1509796, 1512055 ], 
        [ 1512051, 1514922, 1512056, 1514927 ], 
        [ 1514923, 1515140, 1514928, 1515145 ], 
        [ 1515141, 1515194, 1515146, 1515199 ], 
        [ 1515195, 1515647, 1515200, 1515652 ], 
        [ 1515648, 1516602, 1515653, 1516607 ], 
        [ 1516603, 1517689, 1516608, 1517694 ], 
        [ 1517690, 1519324, 1517695, 1519329 ], 
        [ 1519325, 1524288, 1519330, 1524293 ], 
        [ 1524289, 1524809, 1524294, 1524814 ], 
        [ 1524810, 1525934, 1524815, 1525939 ], 
        [ 1525935, 1526325, 1525940, 1526330 ], 
        [ 1526326, 1527046, 1526331, 1527051 ], 
        [ 1527047, 1528800, 1527052, 1528805 ], 
        [ 1528801, 1529067, 1528806, 1529072 ], 
        [ 1529068, 1529127, 1529073, 1529132 ], 
        [ 1529128, 1536262, 1529133, 1536267 ], 
        [ 1536263, 1543858, 1536268, 1543863 ], 
        [ 1543859, 1554015, 1543864, 1554020 ], 
        [ 1554016, 1555315, 1554021, 1555320 ], 
        [ 1555316, 1558476, 1555321, 1558481 ], 
        [ 1558477, 1560403, 1558482, 1560408 ], 
        [ 1560404, 1564152, 1560409, 1564157 ], 
        [ 1564153, 1565868, 1564158, 1565873 ], 
        [ 1565869, 1566075, 1565874, 1566080 ], 
        [ 1566076, 1572715, 1566081, 1572720 ], 
        [ 1572716, 1575566, 1572721, 1575571 ], 
        [ 1575567, 1575840, 1575572, 1575845 ], 
        [ 1575841, 1575957, 1575846, 1575962 ], 
        [ 1575958, 1578588, 1575963, 1578593 ], 
        [ 1578589, 1587557, 1578594, 1587562 ], 
        [ 1587558, 1588891, 1587563, 1588896 ], 
        [ 1588892, 1597227, 1588897, 1597232 ], 
        [ 1597228, 1597262, 1597233, 1597267 ], 
        [ 1597263, 1606974, 1597268, 1606979 ], 
        [ 1606975, 1613512, 1606980, 1613517 ], 
        [ 1613513, 1613900, 1613518, 1613905 ], 
        [ 1613901, 1614931, 1613906, 1614936 ], 
        [ 1614932, 1620971, 1614937, 1620976 ], 
        [ 1620972, 1625931, 1620977, 1625936 ], 
        [ 1625932, 1635578, 1625937, 1635583 ], 
        [ 1635579, 1636949, 1635584, 1636954 ], 
        [ 1636950, 1642076, 1636955, 1642081 ], 
        [ 1642077, 1643227, 1642082, 1643232 ], 
        [ 1643228, 1643451, 1643233, 1643456 ], 
        [ 1643452, 1643568, 1643457, 1643573 ], 
        [ 1643569, 1651406, 1643574, 1651411 ], 
        [ 1651407, 1651474, 1651412, 1651479 ], 
        [ 1651475, 1660688, 1651480, 1660693 ], 
        [ 1660689, 1665846, 1660694, 1665851 ], 
        [ 1665847, 1667026, 1665852, 1667031 ], 
        [ 1667027, 1675465, 1667032, 1675470 ], 
        [ 1675466, 1679164, 1675471, 1679169 ], 
        [ 1679165, 1681962, 1679170, 1681967 ], 
        [ 1681963, 1688016, 1681968, 1688021 ], 
        [ 1688017, 1690659, 1688022, 1690664 ], 
        [ 1690660, 1692872, 1690665, 1692877 ], 
        [ 1692873, 1697102, 1692878, 1697107 ], 
        [ 1697103, 1698132, 1697108, 1698137 ], 
        [ 1698133, 1703429, 1698138, 1703434 ], 
        [ 1703430, 1706057, 1703435, 1706062 ], 
        [ 1706058, 1708683, 1706063, 1708688 ], 
        [ 1708684, 1720884, 1708689, 1720889 ], 
        [ 1720885, 1721218, 1720890, 1721223 ], 
        [ 1721219, 1725289, 1721224, 1725294 ], 
        [ 1725290, 1726495, 1725295, 1726500 ], 
        [ 1726496, 1728646, 1726501, 1728651 ], 
        [ 1728647, 1729060, 1728652, 1729065 ], 
        [ 1729061, 1732801, 1729066, 1732806 ], 
        [ 1732802, 1733308, 1732807, 1733313 ], 
        [ 1733309, 1734471, 1733314, 1734476 ], 
        [ 1734472, 1740942, 1734477, 1740947 ], 
        [ 1740943, 1744762, 1740948, 1744767 ], 
        [ 1744763, 1746379, 1744768, 1746384 ], 
        [ 1746380, 1747144, 1746385, 1747149 ], 
        [ 1747145, 1753062, 1747150, 1753067 ], 
        [ 1753063, 1754367, 1753068, 1754372 ], 
        [ 1754368, 1763444, 1754373, 1763449 ], 
        [ 1763445, 1777420, 1763450, 1777425 ], 
        [ 1777421, 1782626, 1777426, 1782631 ], 
        [ 1782627, 1784342, 1782632, 1784347 ], 
        [ 1784343, 1784549, 1784348, 1784554 ], 
        [ 1784550, 1791189, 1784555, 1791194 ], 
        [ 1791190, 1793878, 1791195, 1793883 ], 
        [ 1793879, 1794152, 1793884, 1794157 ], 
        [ 1794153, 1794269, 1794158, 1794274 ], 
        [ 1794270, 1794972, 1794275, 1794977 ], 
        [ 1794973, 1796163, 1794978, 1796168 ], 
        [ 1796164, 1802296, 1796169, 1802301 ], 
        [ 1802297, 1805729, 1802302, 1805734 ], 
        [ 1805730, 1806305, 1805735, 1806310 ], 
        [ 1806306, 1810512, 1806311, 1810517 ], 
        [ 1810513, 1816402, 1810518, 1816407 ], 
        [ 1816403, 1826227, 1816408, 1826232 ], 
        [ 1826228, 1826701, 1826233, 1826706 ], 
        [ 1826702, 1827720, 1826707, 1827725 ], 
        [ 1827721, 1836707, 1827726, 1836712 ], 
        [ 1836708, 1836926, 1836713, 1836931 ], 
        [ 1836927, 1838667, 1836932, 1838672 ], 
        [ 1838668, 1843220, 1838673, 1843225 ], 
        [ 1843221, 1843829, 1843226, 1843834 ], 
        [ 1843830, 1846577, 1843835, 1846582 ], 
        [ 1846578, 1849125, 1846583, 1849130 ], 
        [ 1849126, 1850237, 1849131, 1850242 ], 
        [ 1850238, 1851708, 1850243, 1851713 ], 
        [ 1851709, 1853436, 1851714, 1853441 ], 
        [ 1853437, 1853475, 1853442, 1853480 ], 
        [ 1853476, 1853493, 1853481, 1853498 ], 
        [ 1853494, 1854900, 1853499, 1854905 ], 
        [ 1854901, 1861797, 1854906, 1861802 ], 
        [ 1861798, 1862267, 1861803, 1862272 ], 
        [ 1862268, 1866445, 1862273, 1866450 ], 
        [ 1866446, 1866700, 1866451, 1866705 ], 
        [ 1866701, 1870143, 1866706, 1870148 ], 
        [ 1870144, 1870675, 1870149, 1870680 ], 
        [ 1870676, 1881704, 1870681, 1881709 ], 
        [ 1881705, 1882659, 1881710, 1882664 ], 
        [ 1882660, 1884008, 1882665, 1884013 ], 
        [ 1884009, 1885076, 1884014, 1885081 ], 
        [ 1885077, 1897857, 1885082, 1897862 ], 
        [ 1897858, 1931549, 1897863, 1931554 ], 
        [ 1931550, 1931660, 1931555, 1931665 ], 
        [ 1931661, 1936680, 1931666, 1936685 ], 
        [ 1936681, 1938835, 1936686, 1938840 ], 
        [ 1938836, 1939367, 1938841, 1939372 ], 
        [ 1939368, 1944718, 1939373, 1944723 ], 
        [ 1944719, 1949924, 1944724, 1949929 ], 
        [ 1949925, 1951640, 1949930, 1951645 ], 
        [ 1951641, 1951847, 1951646, 1951852 ], 
        [ 1951848, 1958495, 1951853, 1958500 ], 
        [ 1958496, 1961184, 1958501, 1961189 ], 
        [ 1961185, 1961458, 1961190, 1961463 ], 
        [ 1961459, 1963223, 1961464, 1963228 ], 
        [ 1963224, 1964535, 1963229, 1964540 ], 
        [ 1964536, 1964578, 1964541, 1964583 ], 
        [ 1964579, 1965726, 1964584, 1965731 ], 
        [ 1965727, 1975723, 1965732, 1975728 ], 
        [ 1975724, 1983495, 1975729, 1983500 ], 
        [ 1983496, 1989041, 1983501, 1989046 ], 
        [ 1989042, 1991939, 1989047, 1991944 ], 
        [ 1991940, 1994134, 1991945, 1994139 ], 
        [ 1994135, 2006390, 1994140, 2006395 ], 
        [ 2006391, 2006681, 2006396, 2006686 ], 
        [ 2006682, 2012753, 2006687, 2012758 ], 
        [ 2012754, 2020299, 2012759, 2020304 ], 
        [ 2020300, 2021594, 2020305, 2021599 ], 
        [ 2021595, 2035653, 2021600, 2035658 ], 
        [ 2035654, 2043961, 2035659, 2043966 ], 
        [ 2043962, 2044411, 2043967, 2044416 ], 
        [ 2044412, 2045320, 2044417, 2045325 ], 
        [ 2045321, 2046593, 2045326, 2046598 ], 
        [ 2046594, 2058014, 2046599, 2058019 ], 
        [ 2058015, 2058262, 2058020, 2058267 ], 
        [ 2058263, 2061616, 2058268, 2061621 ], 
        [ 2061617, 2067334, 2061622, 2067339 ], 
        [ 2067335, 2069059, 2067340, 2069064 ], 
        [ 2069060, 2073142, 2069065, 2073147 ], 
        [ 2073143, 2074555, 2073148, 2074560 ], 
        [ 2074556, 2074634, 2074561, 2074639 ], 
        [ 2074635, 2076422, 2074640, 2076427 ], 
        [ 2076423, 2081937, 2076428, 2081942 ], 
        [ 2081938, 2082042, 2081943, 2082047 ], 
        [ 2082043, 2082408, 2082048, 2082413 ], 
        [ 2082409, 2094661, 2082414, 2094666 ], 
        [ 2094662, 2105556, 2094667, 2105561 ], 
        [ 2105557, 2106153, 2105562, 2106158 ], 
        [ 2106154, 2113282, 2106159, 2113287 ], 
        [ 2113283, 2114197, 2113288, 2114202 ], 
        [ 2114198, 2124245, 2114203, 2124250 ], 
        [ 2124246, 2126629, 2124251, 2126634 ], 
        [ 2126630, 2127367, 2126635, 2127372 ], 
        [ 2127368, 2131854, 2127373, 2131859 ], 
        [ 2131855, 2138481, 2131860, 2138486 ], 
        [ 2138482, 2140084, 2138487, 2140089 ], 
        [ 2140085, 2151397, 2140090, 2151402 ], 
        [ 2151398, 2154116, 2151403, 2154121 ], 
        [ 2154117, 2164531, 2154122, 2164536 ], 
        [ 2164532, 2164999, 2164537, 2165004 ], 
        [ 2165000, 2166190, 2165005, 2166195 ], 
        [ 2166191, 2168535, 2166196, 2168540 ], 
        [ 2168536, 2168652, 2168541, 2168657 ], 
        [ 2168653, 2168876, 2168658, 2168881 ], 
        [ 2168877, 2175197, 2168882, 2175202 ], 
        [ 2175198, 2176568, 2175203, 2176573 ], 
        [ 2176569, 2185419, 2176574, 2185424 ], 
        [ 2185420, 2198074, 2185425, 2198079 ], 
        [ 2198075, 2205716, 2198080, 2205721 ], 
        [ 2205717, 2206482, 2205722, 2206487 ], 
        [ 2206483, 2214819, 2206488, 2214824 ], 
        [ 2214820, 2215255, 2214825, 2215260 ], 
        [ 2215256, 2216910, 2215261, 2216915 ], 
        [ 2216911, 2219477, 2216916, 2219482 ], 
        [ 2219478, 2219751, 2219483, 2219756 ], 
        [ 2219752, 2222602, 2219757, 2222607 ], 
        [ 2222603, 2224016, 2222608, 2224021 ], 
        [ 2224017, 2229253, 2224022, 2229258 ], 
        [ 2229254, 2229460, 2229259, 2229465 ], 
        [ 2229461, 2231176, 2229466, 2231181 ], 
        [ 2231177, 2236382, 2231182, 2236387 ], 
        [ 2236383, 2245581, 2236388, 2245586 ], 
        [ 2245582, 2245719, 2245587, 2245724 ], 
        [ 2245720, 2245761, 2245725, 2245766 ], 
        [ 2245762, 2249902, 2245767, 2249907 ], 
        [ 2249903, 2254722, 2249908, 2254727 ], 
        [ 2254723, 2262668, 2254728, 2262673 ], 
        [ 2262669, 2276333, 2262674, 2276338 ], 
        [ 2276334, 2278349, 2276339, 2278354 ], 
        [ 2278350, 2278595, 2278355, 2278600 ], 
        [ 2278596, 2282039, 2278601, 2282044 ], 
        [ 2282040, 2309292, 2282045, 2309297 ], 
        [ 2309293, 2309737, 2309298, 2309742 ], 
        [ 2309738, 2314845, 2309743, 2314850 ], 
        [ 2314846, 2315016, 2314851, 2315021 ], 
        [ 2315017, 2320047, 2315022, 2320052 ], 
        [ 2320048, 2320645, 2320053, 2320650 ], 
        [ 2320646, 2330437, 2320651, 2330442 ], 
        [ 2330438, 2338082, 2330443, 2338087 ], 
        [ 2338083, 2345465, 2338088, 2345470 ], 
        [ 2345466, 2347233, 2345471, 2347238 ], 
        [ 2347234, 2348720, 2347239, 2348725 ], 
        [ 2348721, 2351324, 2348726, 2351329 ], 
        [ 2351325, 2352448, 2351330, 2352453 ], 
        [ 2352449, 2353999, 2352454, 2354004 ], 
        [ 2354000, 2359046, 2354005, 2359051 ], 
        [ 2359047, 2361149, 2359052, 2361154 ], 
        [ 2361150, 2374039, 2361155, 2374044 ], 
        [ 2374040, 2385349, 2374045, 2385354 ], 
        [ 2385350, 2388585, 2385355, 2388590 ], 
        [ 2388586, 2391734, 2388591, 2391739 ], 
        [ 2391735, 2392141, 2391740, 2392146 ], 
        [ 2392142, 2393939, 2392147, 2393944 ], 
        [ 2393940, 2395026, 2393945, 2395031 ], 
        [ 2395027, 2395860, 2395032, 2395865 ], 
        [ 2395861, 2398211, 2395866, 2398216 ], 
        [ 2398212, 2398326, 2398217, 2398331 ], 
        [ 2398327, 2402303, 2398332, 2402308 ], 
        [ 2402304, 2408154, 2402309, 2408159 ], 
        [ 2408155, 2409936, 2408160, 2409941 ], 
        [ 2409937, 2410353, 2409942, 2410358 ], 
        [ 2410354, 2411021, 2410359, 2411026 ], 
        [ 2411022, 2419571, 2411027, 2419576 ], 
        [ 2419572, 2424488, 2419577, 2424493 ], 
        [ 2424489, 2427895, 2424494, 2427900 ], 
        [ 2427896, 2433794, 2427901, 2433799 ], 
        [ 2433795, 2434280, 2433800, 2434285 ], 
        [ 2434281, 2436129, 2434286, 2436134 ], 
        [ 2436130, 2446339, 2436135, 2446344 ], 
        [ 2446340, 2446355, 2446345, 2446360 ], 
        [ 2446356, 2447550, 2446361, 2447555 ], 
        [ 2447551, 2456375, 2447556, 2456380 ], 
        [ 2456376, 2459685, 2456381, 2459690 ], 
        [ 2459686, 2467707, 2459691, 2467712 ], 
        [ 2467708, 2489626, 2467713, 2489631 ], 
        [ 2489627, 2490030, 2489632, 2490035 ], 
        [ 2490031, 2494181, 2490036, 2494186 ], 
        [ 2494182, 2494578, 2494187, 2494583 ], 
        [ 2494579, 2498330, 2494584, 2498335 ], 
        [ 2498331, 2501619, 2498336, 2501624 ], 
        [ 2501620, 2502774, 2501625, 2502779 ], 
        [ 2502775, 2505440, 2502780, 2505445 ], 
        [ 2505441, 2507840, 2505446, 2507845 ], 
        [ 2507841, 2513953, 2507846, 2513958 ], 
        [ 2513954, 2518482, 2513959, 2518487 ], 
        [ 2518483, 2518510, 2518488, 2518515 ], 
        [ 2518511, 2519154, 2518516, 2519159 ], 
        [ 2519155, 2521663, 2519160, 2521668 ], 
        [ 2521664, 2522690, 2521669, 2522695 ], 
        [ 2522691, 2535156, 2522696, 2535161 ], 
        [ 2535157, 2536302, 2535162, 2536307 ], 
        [ 2536303, 2539683, 2536308, 2539688 ], 
        [ 2539684, 2540838, 2539689, 2540843 ], 
        [ 2540839, 2542542, 2540844, 2542547 ], 
        [ 2542543, 2549711, 2542548, 2549716 ], 
        [ 2549712, 2549979, 2549717, 2549984 ], 
        [ 2549980, 2550376, 2549985, 2550381 ], 
        [ 2550377, 2550442, 2550382, 2550447 ], 
        [ 2550443, 2552498, 2550448, 2552503 ], 
        [ 2552499, 2556237, 2552504, 2556242 ], 
        [ 2556238, 2561281, 2556243, 2561286 ], 
        [ 2561282, 2562381, 2561287, 2562386 ], 
        [ 2562382, 2571576, 2562387, 2571581 ], 
        [ 2571577, 2573918, 2571582, 2573923 ], 
        [ 2573919, 2575854, 2573924, 2575859 ], 
        [ 2575855, 2579045, 2575860, 2579050 ], 
        [ 2579046, 2588728, 2579051, 2588733 ], 
        [ 2588729, 2591930, 2588734, 2591935 ], 
        [ 2591931, 2601304, 2591936, 2601309 ], 
        [ 2601305, 2614812, 2601310, 2614817 ], 
        [ 2614813, 2614839, 2614818, 2614844 ], 
        [ 2614840, 2622328, 2614845, 2622333 ], 
        [ 2622329, 2627903, 2622334, 2627908 ], 
        [ 2627904, 2648431, 2627909, 2648436 ], 
        [ 2648432, 2651846, 2648437, 2651851 ], 
        [ 2651847, 2660586, 2651852, 2660591 ], 
        [ 2660587, 2663434, 2660592, 2663439 ], 
        [ 2663435, 2674481, 2663440, 2674486 ], 
        [ 2674482, 2674949, 2674487, 2674954 ], 
        [ 2674950, 2676096, 2674955, 2676101 ], 
        [ 2676097, 2676139, 2676102, 2676144 ], 
        [ 2676140, 2678485, 2676145, 2678490 ], 
        [ 2678486, 2678602, 2678491, 2678607 ], 
        [ 2678603, 2678826, 2678608, 2678831 ], 
        [ 2678827, 2681278, 2678832, 2681283 ], 
        [ 2681279, 2684926, 2681284, 2684931 ], 
        [ 2684927, 2685205, 2684932, 2685210 ], 
        [ 2685206, 2692493, 2685211, 2692498 ], 
        [ 2692494, 2714956, 2692499, 2714961 ], 
        [ 2714957, 2716135, 2714962, 2716140 ], 
        [ 2716136, 2716930, 2716141, 2716935 ], 
        [ 2716931, 2717350, 2716936, 2717355 ], 
        [ 2717351, 2717774, 2717356, 2717779 ], 
        [ 2717775, 2718590, 2717780, 2718595 ], 
        [ 2718591, 2720379, 2718596, 2720384 ], 
        [ 2720380, 2723960, 2720385, 2723965 ], 
        [ 2723961, 2725658, 2723966, 2725663 ], 
        [ 2725659, 2731980, 2725664, 2731985 ], 
        [ 2731981, 2738172, 2731986, 2738177 ], 
        [ 2738173, 2748433, 2738178, 2748438 ], 
        [ 2748434, 2748596, 2748439, 2748601 ], 
        [ 2748597, 2749729, 2748602, 2749734 ], 
        [ 2749730, 2752944, 2749735, 2752949 ], 
        [ 2752945, 2753953, 2752950, 2753958 ], 
        [ 2753954, 2760599, 2753959, 2760604 ], 
        [ 2760600, 2761236, 2760605, 2761241 ], 
        [ 2761237, 2763346, 2761242, 2763351 ], 
        [ 2763347, 2764218, 2763352, 2764223 ], 
        [ 2764219, 2773348, 2764224, 2773353 ], 
        [ 2773349, 2779707, 2773354, 2779712 ], 
        [ 2779708, 2792111, 2779713, 2792116 ], 
        [ 2792112, 2794418, 2792117, 2794423 ], 
        [ 2794419, 2795818, 2794424, 2795823 ], 
        [ 2795819, 2796261, 2795824, 2796266 ], 
        [ 2796262, 2798929, 2796267, 2798934 ], 
        [ 2798930, 2799454, 2798935, 2799459 ], 
        [ 2799455, 2813416, 2799460, 2813421 ], 
        [ 2813417, 2813479, 2813422, 2813484 ], 
        [ 2813480, 2814107, 2813485, 2814112 ], 
        [ 2814108, 2825357, 2814113, 2825362 ], 
        [ 2825358, 2826820, 2825363, 2826825 ], 
        [ 2826821, 2828096, 2826826, 2828101 ], 
        [ 2828097, 2830088, 2828102, 2830093 ], 
        [ 2830089, 2834601, 2830094, 2834606 ], 
        [ 2834602, 2836621, 2834607, 2836626 ], 
        [ 2836622, 2836884, 2836627, 2836889 ], 
        [ 2836885, 2837913, 2836890, 2837918 ], 
        [ 2837914, 2840393, 2837919, 2840398 ], 
        [ 2840394, 2843391, 2840399, 2843396 ], 
        [ 2843392, 2843629, 2843397, 2843634 ], 
        [ 2843630, 2844665, 2843635, 2844670 ], 
        [ 2844666, 2847821, 2844671, 2847826 ], 
        [ 2847822, 2849601, 2847827, 2849606 ], 
        [ 2849602, 2853189, 2849607, 2853194 ], 
        [ 2853190, 2860428, 2853195, 2860433 ], 
        [ 2860429, 2862152, 2860434, 2862157 ], 
        [ 2862153, 2862729, 2862158, 2862734 ], 
        [ 2862730, 2869033, 2862735, 2869038 ], 
        [ 2869034, 2869157, 2869039, 2869162 ], 
        [ 2869158, 2882082, 2869163, 2882087 ], 
        [ 2882083, 2894091, 2882088, 2894096 ], 
        [ 2894092, 2895090, 2894097, 2895095 ], 
        [ 2895091, 2900119, 2895096, 2900124 ], 
        [ 2900120, 2900555, 2900125, 2900560 ], 
        [ 2900556, 2902167, 2900561, 2902172 ], 
        [ 2902168, 2902210, 2902173, 2902215 ], 
        [ 2902211, 2904556, 2902216, 2904561 ], 
        [ 2904557, 2904673, 2904562, 2904678 ], 
        [ 2904674, 2904897, 2904679, 2904902 ], 
        [ 2904898, 2907797, 2904903, 2907802 ], 
        [ 2907798, 2910456, 2907803, 2910461 ], 
        [ 2910457, 2911608, 2910462, 2911613 ], 
        [ 2911609, 2914744, 2911614, 2914749 ], 
        [ 2914745, 2914779, 2914750, 2914784 ], 
        [ 2914780, 2918124, 2914785, 2918129 ], 
        [ 2918125, 2921020, 2918130, 2921025 ], 
        [ 2921021, 2921458, 2921026, 2921463 ], 
        [ 2921459, 2926580, 2921464, 2926585 ], 
        [ 2926581, 2930570, 2926586, 2930575 ], 
        [ 2930571, 2934873, 2930576, 2934878 ], 
        [ 2934874, 2942883, 2934879, 2942888 ], 
        [ 2942884, 2950823, 2942889, 2950828 ], 
        [ 2950824, 2952204, 2950829, 2952209 ], 
        [ 2952205, 2954108, 2952210, 2954113 ], 
        [ 2954109, 2958788, 2954114, 2958793 ], 
        [ 2958789, 2962338, 2958794, 2962343 ], 
        [ 2962339, 2962634, 2962344, 2962639 ], 
        [ 2962635, 2963567, 2962640, 2963572 ], 
        [ 2963568, 2965512, 2963573, 2965517 ], 
        [ 2965513, 2965715, 2965518, 2965720 ], 
        [ 2965716, 2969537, 2965721, 2969542 ], 
        [ 2969538, 2969667, 2969543, 2969672 ], 
        [ 2969668, 2971097, 2969673, 2971102 ], 
        [ 2971098, 2971329, 2971103, 2971334 ], 
        [ 2971330, 2971874, 2971335, 2971879 ], 
        [ 2971875, 2972098, 2971880, 2972103 ], 
        [ 2972099, 2978342, 2972104, 2978347 ], 
        [ 2978343, 2984060, 2978348, 2984065 ], 
        [ 2984061, 2988924, 2984066, 2988929 ], 
        [ 2988925, 2994739, 2988930, 2994744 ], 
        [ 2994740, 3002088, 2994745, 3002093 ], 
        [ 3002089, 3009887, 3002094, 3009892 ], 
        [ 3009888, 3014827, 3009893, 3014832 ], 
        [ 3014828, 3020885, 3014833, 3020890 ], 
        [ 3020886, 3022261, 3020891, 3022266 ], 
        [ 3022262, 3029543, 3022267, 3029548 ], 
        [ 3029544, 3030265, 3029549, 3030270 ], 
        [ 3030266, 3032363, 3030271, 3032368 ], 
        [ 3032364, 3033161, 3032369, 3033166 ], 
        [ 3033162, 3042175, 3033167, 3042180 ], 
        [ 3042176, 3042389, 3042181, 3042394 ], 
        [ 3042390, 3049663, 3042395, 3049668 ], 
        [ 3049664, 3050210, 3049669, 3050215 ], 
        [ 3050211, 3051389, 3050216, 3051394 ], 
        [ 3051390, 3052128, 3051395, 3052133 ], 
        [ 3052129, 3052883, 3052134, 3052888 ], 
        [ 3052884, 3054679, 3052889, 3054684 ], 
        [ 3054680, 3055955, 3054685, 3055960 ], 
        [ 3055956, 3056024, 3055961, 3056029 ], 
        [ 3056025, 3062859, 3056030, 3062864 ], 
        [ 3062860, 3063276, 3062865, 3063281 ], 
        [ 3063277, 3064101, 3063282, 3064106 ], 
        [ 3064102, 3065575, 3064107, 3065580 ], 
        [ 3065576, 3065710, 3065581, 3065715 ], 
        [ 3065711, 3066590, 3065716, 3066595 ], 
        [ 3066591, 3075292, 3066596, 3075297 ], 
        [ 3075293, 3076853, 3075298, 3076858 ], 
        [ 3076854, 3080566, 3076859, 3080571 ], 
        [ 3080567, 3080707, 3080572, 3080712 ], 
        [ 3080708, 3080747, 3080713, 3080752 ], 
        [ 3080748, 3080916, 3080753, 3080921 ], 
        [ 3080917, 3082200, 3080922, 3082205 ], 
        [ 3082201, 3085986, 3082206, 3085991 ], 
        [ 3085987, 3086111, 3085992, 3086116 ], 
        [ 3086112, 3088509, 3086117, 3088514 ], 
        [ 3088510, 3091108, 3088515, 3091113 ], 
        [ 3091109, 3094298, 3091114, 3094303 ], 
        [ 3094299, 3098254, 3094304, 3098259 ], 
        [ 3098255, 3100958, 3098260, 3100963 ], 
        [ 3100959, 3101111, 3100964, 3101116 ], 
        [ 3101112, 3101710, 3101117, 3101715 ], 
        [ 3101711, 3107070, 3101716, 3107075 ], 
        [ 3107071, 3108550, 3107076, 3108555 ], 
        [ 3108551, 3116752, 3108556, 3116757 ], 
        [ 3116753, 3119656, 3116758, 3119661 ], 
        [ 3119657, 3120436, 3119662, 3120441 ], 
        [ 3120437, 3122936, 3120442, 3122941 ], 
        [ 3122937, 3125746, 3122942, 3125751 ], 
        [ 3125747, 3126385, 3125752, 3126390 ], 
        [ 3126386, 3129525, 3126391, 3129530 ], 
        [ 3129526, 3129570, 3129531, 3129575 ], 
        [ 3129571, 3129752, 3129576, 3129757 ], 
        [ 3129753, 3132349, 3129758, 3132354 ], 
        [ 3132350, 3139904, 3132355, 3139909 ], 
        [ 3139905, 3143832, 3139910, 3143837 ], 
        [ 3143833, 3144573, 3143838, 3144578 ], 
        [ 3144574, 3148252, 3144579, 3148257 ], 
        [ 3148253, 3150053, 3148258, 3150058 ], 
        [ 3150054, 3152136, 3150059, 3152141 ], 
        [ 3152137, 3161578, 3152142, 3161583 ], 
        [ 3161579, 3164826, 3161584, 3164831 ], 
        [ 3164827, 3164943, 3164832, 3164948 ], 
        [ 3164944, 3167563, 3164949, 3167568 ], 
        [ 3167564, 3168258, 3167569, 3168263 ], 
        [ 3168259, 3170295, 3168264, 3170300 ], 
        [ 3170296, 3172533, 3170301, 3172538 ], 
        [ 3172534, 3176792, 3172539, 3176797 ], 
        [ 3176793, 3177637, 3176798, 3177642 ], 
        [ 3177638, 3178306, 3177643, 3178311 ], 
        [ 3178307, 3186857, 3178312, 3186862 ], 
        [ 3186858, 3196386, 3186863, 3196391 ], 
        [ 3196387, 3197840, 3196392, 3197845 ], 
        [ 3197841, 3210828, 3197846, 3210833 ], 
        [ 3210829, 3212953, 3210834, 3212958 ], 
        [ 3212954, 3226292, 3212959, 3226297 ], 
        [ 3226293, 3226762, 3226298, 3226767 ], 
        [ 3226763, 3230485, 3226768, 3230490 ], 
        [ 3230486, 3233743, 3230491, 3233748 ], 
        [ 3233744, 3234567, 3233749, 3234572 ], 
        [ 3234568, 3235194, 3234573, 3235199 ], 
        [ 3235195, 3235499, 3235200, 3235504 ], 
        [ 3235500, 3243889, 3235505, 3243894 ], 
        [ 3243890, 3259251, 3243895, 3259256 ], 
        [ 3259252, 3262432, 3259257, 3262437 ], 
        [ 3262433, 3262672, 3262438, 3262677 ], 
        [ 3262673, 3266686, 3262678, 3266691 ], 
        [ 3266687, 3269046, 3266692, 3269051 ], 
        [ 3269047, 3269620, 3269052, 3269625 ], 
        [ 3269621, 3272036, 3269626, 3272041 ], 
        [ 3272037, 3272516, 3272042, 3272521 ], 
        [ 3272517, 3272796, 3272522, 3272801 ], 
        [ 3272797, 3278978, 3272802, 3278983 ], 
        [ 3278979, 3279467, 3278984, 3279472 ], 
        [ 3279468, 3280283, 3279473, 3280288 ], 
        [ 3280284, 3282531, 3280289, 3282536 ], 
        [ 3282532, 3286305, 3282537, 3286310 ], 
        [ 3286306, 3286606, 3286311, 3286611 ], 
        [ 3286607, 3287302, 3286612, 3287307 ], 
        [ 3287303, 3290776, 3287308, 3290781 ], 
        [ 3290777, 3294305, 3290782, 3294310 ], 
        [ 3294306, 3298010, 3294311, 3298015 ], 
        [ 3298011, 3298799, 3298016, 3298804 ], 
        [ 3298800, 3299744, 3298805, 3299749 ], 
        [ 3299745, 3302402, 3299750, 3302407 ], 
        [ 3302403, 3314750, 3302408, 3314755 ], 
        [ 3314751, 3314777, 3314756, 3314782 ], 
        [ 3314778, 3315543, 3314783, 3315548 ], 
        [ 3315544, 3319495, 3315549, 3319500 ], 
        [ 3319496, 3321744, 3319501, 3321749 ], 
        [ 3321745, 3324685, 3321750, 3324690 ], 
        [ 3324686, 3331241, 3324691, 3331246 ], 
        [ 3331242, 3345624, 3331247, 3345629 ], 
        [ 3345625, 3349729, 3345630, 3349734 ], 
        [ 3349730, 3350579, 3349735, 3350584 ], 
        [ 3350580, 3353994, 3350585, 3353999 ], 
        [ 3353995, 3355238, 3354000, 3355243 ], 
        [ 3355239, 3360120, 3355244, 3360125 ], 
        [ 3360121, 3368828, 3360126, 3368833 ], 
        [ 3368829, 3376335, 3368834, 3376340 ], 
        [ 3376336, 3380722, 3376341, 3380727 ], 
        [ 3380723, 3381309, 3380728, 3381314 ], 
        [ 3381310, 3382080, 3381315, 3382085 ], 
        [ 3382081, 3384674, 3382086, 3384679 ], 
        [ 3384675, 3385188, 3384680, 3385193 ], 
        [ 3385189, 3388713, 3385194, 3388718 ], 
        [ 3388714, 3392079, 3388719, 3392084 ], 
        [ 3392080, 3394131, 3392085, 3394136 ], 
        [ 3394132, 3394381, 3394137, 3394386 ], 
        [ 3394382, 3395362, 3394387, 3395367 ], 
        [ 3395363, 3396615, 3395368, 3396620 ], 
        [ 3396616, 3399453, 3396621, 3399458 ], 
        [ 3399454, 3400585, 3399459, 3400590 ], 
        [ 3400586, 3405494, 3400591, 3405499 ], 
        [ 3405495, 3413040, 3405500, 3413045 ], 
        [ 3413041, 3413955, 3413046, 3413960 ], 
        [ 3413956, 3413965, 3413961, 3413970 ], 
        [ 3413966, 3415450, 3413971, 3415455 ], 
        [ 3415451, 3415543, 3415456, 3415548 ], 
        [ 3415544, 3415693, 3415549, 3415698 ], 
        [ 3415694, 3416866, 3415699, 3416871 ], 
        [ 3416867, 3421885, 3416872, 3421890 ], 
        [ 3421886, 3425142, 3421891, 3425147 ], 
        [ 3425143, 3437815, 3425148, 3437820 ], 
        [ 3437816, 3440231, 3437821, 3440236 ], 
        [ 3440232, 3442385, 3440237, 3442390 ], 
        [ 3442386, 3449766, 3442391, 3449771 ], 
        [ 3449767, 3455799, 3449772, 3455804 ], 
        [ 3455800, 3473047, 3455805, 3473052 ], 
        [ 3473048, 3483375, 3473053, 3483380 ], 
        [ 3483376, 3505500, 3483381, 3505505 ], 
        [ 3505501, 3514082, 3505506, 3514087 ], 
        [ 3514083, 3522092, 3514088, 3522097 ], 
        [ 3522093, 3522111, 3522098, 3522116 ], 
        [ 3522112, 3522767, 3522117, 3522772 ], 
        [ 3522768, 3531627, 3522773, 3531632 ], 
        [ 3531628, 3537780, 3531633, 3537785 ], 
        [ 3537781, 3540644, 3537786, 3540649 ], 
        [ 3540645, 3543017, 3540650, 3543022 ], 
        [ 3543018, 3543842, 3543023, 3543847 ], 
        [ 3543843, 3545546, 3543848, 3545551 ], 
        [ 3545547, 3547077, 3545552, 3547082 ], 
        [ 3547078, 3547149, 3547083, 3547154 ], 
        [ 3547150, 3548533, 3547155, 3548538 ], 
        [ 3548534, 3549829, 3548539, 3549834 ], 
        [ 3549830, 3549935, 3549835, 3549940 ], 
        [ 3549936, 3550203, 3549941, 3550208 ], 
        [ 3550204, 3550227, 3550209, 3550232 ], 
        [ 3550228, 3551480, 3550233, 3551485 ], 
        [ 3551481, 3552210, 3551486, 3552215 ], 
        [ 3552211, 3562847, 3552216, 3562852 ], 
        [ 3562848, 3563068, 3562853, 3563073 ], 
        [ 3563069, 3563973, 3563074, 3563978 ], 
        [ 3563974, 3565130, 3563979, 3565135 ], 
        [ 3565131, 3565565, 3565136, 3565570 ], 
        [ 3565566, 3566132, 3565571, 3566137 ], 
        [ 3566133, 3567018, 3566138, 3567023 ], 
        [ 3567019, 3567627, 3567024, 3567632 ], 
        [ 3567628, 3570838, 3567633, 3570843 ], 
        [ 3570839, 3574124, 3570844, 3574129 ], 
        [ 3574125, 3575098, 3574130, 3575103 ], 
        [ 3575099, 3575586, 3575104, 3575591 ], 
        [ 3575587, 3576602, 3575592, 3576607 ], 
        [ 3576603, 3590383, 3576608, 3590388 ], 
        [ 3590384, 3594697, 3590389, 3594702 ], 
        [ 3594698, 3597307, 3594703, 3597312 ], 
        [ 3597308, 3598837, 3597313, 3598842 ], 
        [ 3598838, 3599839, 3598843, 3599844 ], 
        [ 3599840, 3603578, 3599845, 3603583 ], 
        [ 3603579, 3609322, 3603584, 3609327 ], 
        [ 3609323, 3614549, 3609328, 3614554 ], 
        [ 3614550, 3618664, 3614555, 3618669 ], 
        [ 3618665, 3619899, 3618670, 3619904 ], 
        [ 3619900, 3624824, 3619905, 3624829 ], 
        [ 3624825, 3628159, 3624830, 3628164 ], 
        [ 3628160, 3628423, 3628165, 3628428 ], 
        [ 3628424, 3630285, 3628429, 3630290 ], 
        [ 3630286, 3630486, 3630291, 3630491 ], 
        [ 3630487, 3634237, 3630492, 3634242 ], 
        [ 3634238, 3639209, 3634243, 3639214 ], 
        [ 3639210, 3652188, 3639215, 3652193 ], 
        [ 3652189, 3659027, 3652194, 3659032 ], 
        [ 3659028, 3659079, 3659033, 3659084 ], 
        [ 3659080, 3659207, 3659085, 3659212 ], 
        [ 3659208, 3667507, 3659213, 3667512 ], 
        [ 3667508, 3667741, 3667513, 3667746 ], 
        [ 3667742, 3668429, 3667747, 3668434 ], 
        [ 3668430, 3670470, 3668435, 3670475 ], 
        [ 3670471, 3673148, 3670476, 3673153 ], 
        [ 3673149, 3674083, 3673154, 3674088 ], 
        [ 3674084, 3678214, 3674089, 3678219 ], 
        [ 3678215, 3680677, 3678220, 3680682 ], 
        [ 3680678, 3684205, 3680683, 3684210 ], 
        [ 3684206, 3690753, 3684211, 3690758 ], 
        [ 3690754, 3698350, 3690759, 3698355 ], 
        [ 3698351, 3699287, 3698356, 3699292 ], 
        [ 3699288, 3702298, 3699293, 3702303 ], 
        [ 3702299, 3706224, 3702304, 3706229 ], 
        [ 3706225, 3706948, 3706230, 3706953 ], 
        [ 3706949, 3709050, 3706954, 3709055 ], 
        [ 3709051, 3725109, 3709056, 3725114 ], 
        [ 3725110, 3731514, 3725115, 3731519 ], 
        [ 3731515, 3736239, 3731520, 3736244 ], 
        [ 3736240, 3736361, 3736245, 3736366 ], 
        [ 3736362, 3738698, 3736367, 3738703 ], 
        [ 3738699, 3745350, 3738704, 3745355 ], 
        [ 3745351, 3747516, 3745356, 3747521 ], 
        [ 3747517, 3748248, 3747522, 3748253 ], 
        [ 3748249, 3750382, 3748254, 3750387 ], 
        [ 3750383, 3761090, 3750388, 3761095 ], 
        [ 3761091, 3762084, 3761096, 3762089 ], 
        [ 3762085, 3762616, 3762090, 3762621 ], 
        [ 3762617, 3769672, 3762622, 3769677 ], 
        [ 3769673, 3769809, 3769678, 3769814 ], 
        [ 3769810, 3769962, 3769815, 3769967 ], 
        [ 3769963, 3774893, 3769968, 3774898 ], 
        [ 3774894, 3776322, 3774899, 3776327 ], 
        [ 3776323, 3778089, 3776328, 3778094 ], 
        [ 3778090, 3784214, 3778095, 3784219 ], 
        [ 3784215, 3799015, 3784220, 3799020 ], 
        [ 3799016, 3806065, 3799021, 3806070 ], 
        [ 3806066, 3807768, 3806071, 3807773 ], 
        [ 3807769, 3816682, 3807774, 3816687 ], 
        [ 3816683, 3817044, 3816688, 3817049 ], 
        [ 3817045, 3817768, 3817050, 3817773 ], 
        [ 3817769, 3822167, 3817774, 3822172 ], 
        [ 3822168, 3824032, 3822173, 3824037 ], 
        [ 3824033, 3828542, 3824038, 3828547 ], 
        [ 3828543, 3832118, 3828548, 3832123 ], 
        [ 3832119, 3832453, 3832124, 3832458 ], 
        [ 3832454, 3837573, 3832459, 3837578 ], 
        [ 3837574, 3842658, 3837579, 3842663 ], 
        [ 3842659, 3844599, 3842664, 3844604 ], 
        [ 3844600, 3844788, 3844605, 3844793 ], 
        [ 3844789, 3850826, 3844794, 3850831 ], 
        [ 3850827, 3855395, 3850832, 3855400 ], 
        [ 3855396, 3860530, 3855401, 3860535 ], 
        [ 3860531, 3873056, 3860536, 3873061 ], 
        [ 3873057, 3880197, 3873062, 3880202 ], 
        [ 3880198, 3882820, 3880203, 3882825 ], 
        [ 3882821, 3883310, 3882826, 3883315 ], 
        [ 3883311, 3885744, 3883316, 3885749 ], 
        [ 3885745, 3891218, 3885750, 3891223 ], 
        [ 3891219, 3891354, 3891224, 3891359 ], 
        [ 3891355, 3894419, 3891360, 3894424 ], 
        [ 3894420, 3900500, 3894425, 3900505 ], 
        [ 3900501, 3907313, 3900506, 3907318 ], 
        [ 3907314, 3908919, 3907319, 3908924 ], 
        [ 3908920, 3909807, 3908925, 3909812 ], 
        [ 3909808, 3909977, 3909813, 3909982 ], 
        [ 3909978, 3930585, 3909983, 3930590 ], 
        [ 3930586, 3933363, 3930591, 3933368 ], 
        [ 3933364, 3935176, 3933369, 3935181 ], 
        [ 3935177, 3936871, 3935182, 3936876 ], 
        [ 3936872, 3945198, 3936877, 3945203 ], 
        [ 3945199, 3946390, 3945204, 3946395 ], 
        [ 3946391, 3946986, 3946396, 3946991 ], 
        [ 3946987, 3956408, 3946992, 3956413 ], 
        [ 3956409, 3958333, 3956414, 3958338 ], 
        [ 3958334, 3959031, 3958339, 3959036 ], 
        [ 3959032, 3960932, 3959037, 3960937 ], 
        [ 3960933, 3964190, 3960938, 3964195 ], 
        [ 3964191, 3969413, 3964196, 3969418 ], 
        [ 3969414, 3972146, 3969419, 3972151 ], 
        [ 3972147, 3972344, 3972152, 3972349 ], 
        [ 3972345, 3978065, 3972350, 3978070 ], 
        [ 3978066, 3981977, 3978071, 3981982 ], 
        [ 3981978, 3984768, 3981983, 3984773 ], 
        [ 3984769, 3984918, 3984774, 3984923 ], 
        [ 3984919, 3985704, 3984924, 3985709 ], 
        [ 3985705, 3995454, 3985710, 3995459 ], 
        [ 3995455, 3997410, 3995460, 3997415 ], 
        [ 3997411, 4000982, 3997416, 4000987 ], 
        [ 4000983, 4002514, 4000988, 4002519 ], 
        [ 4002515, 4005689, 4002520, 4005694 ], 
        [ 4005690, 4016078, 4005695, 4016083 ], 
        [ 4016079, 4017237, 4016084, 4017242 ], 
        [ 4017238, 4018412, 4017243, 4018417 ], 
        [ 4018413, 4018717, 4018418, 4018722 ], 
        [ 4018718, 4023281, 4018723, 4023286 ], 
        [ 4023282, 4032076, 4023287, 4032081 ], 
        [ 4032077, 4041164, 4032082, 4041169 ], 
        [ 4041165, 4041295, 4041170, 4041300 ], 
        [ 4041296, 4041920, 4041301, 4041925 ], 
        [ 4041921, 4046747, 4041926, 4046752 ], 
        [ 4046748, 4052606, 4046753, 4052611 ], 
        [ 4052607, 4054469, 4052612, 4054474 ], 
        [ 4054470, 4054629, 4054475, 4054634 ], 
        [ 4054630, 4054968, 4054635, 4054973 ], 
        [ 4054969, 4055333, 4054974, 4055338 ], 
        [ 4055334, 4056058, 4055339, 4056063 ], 
        [ 4056059, 4059084, 4056064, 4059089 ], 
        [ 4059085, 4062508, 4059090, 4062513 ], 
        [ 4062509, 4065383, 4062514, 4065388 ], 
        [ 4065384, 4065643, 4065389, 4065648 ], 
        [ 4065644, 4069949, 4065649, 4069954 ], 
        [ 4069950, 4078343, 4069955, 4078348 ], 
        [ 4078344, 4083896, 4078349, 4083901 ], 
        [ 4083897, 4085884, 4083902, 4085889 ], 
        [ 4085885, 4090152, 4085890, 4090157 ], 
        [ 4090153, 4093908, 4090158, 4093913 ], 
        [ 4093909, 4094118, 4093914, 4094123 ], 
        [ 4094119, 4095249, 4094124, 4095254 ], 
        [ 4095250, 4097811, 4095255, 4097816 ], 
        [ 4097812, 4101896, 4097817, 4101901 ], 
        [ 4101897, 4107088, 4101902, 4107093 ], 
        [ 4107089, 4107551, 4107094, 4107556 ], 
        [ 4107552, 4107580, 4107557, 4107585 ], 
        [ 4107581, 4109180, 4107586, 4109185 ], 
        [ 4109181, 4110537, 4109186, 4110542 ], 
        [ 4110538, 4116491, 4110543, 4116496 ], 
        [ 4116492, 4117909, 4116497, 4117914 ], 
        [ 4117910, 4118469, 4117915, 4118474 ], 
        [ 4118470, 4123661, 4118475, 4123666 ], 
        [ 4123662, 4123865, 4123667, 4123870 ], 
        [ 4123866, 4125127, 4123871, 4125132 ], 
        [ 4125128, 4129187, 4125133, 4129192 ], 
        [ 4129188, 4132972, 4129193, 4132977 ], 
        [ 4132973, 4134272, 4132978, 4134277 ], 
        [ 4134273, 4135470, 4134278, 4135475 ], 
        [ 4135471, 4136837, 4135476, 4136842 ], 
        [ 4136838, 4146292, 4136843, 4146297 ], 
        [ 4146293, 4146443, 4146298, 4146448 ], 
        [ 4146444, 4148935, 4146449, 4148940 ], 
        [ 4148936, 4162265, 4148941, 4162270 ], 
        [ 4162266, 4164781, 4162271, 4164786 ], 
        [ 4164782, 4170898, 4164787, 4170903 ], 
        [ 4170899, 4175870, 4170904, 4175875 ], 
        [ 4175871, 4175960, 4175876, 4175965 ], 
        [ 4175961, 4180869, 4175966, 4180874 ], 
        [ 4180870, 4181479, 4180875, 4181484 ], 
        [ 4181480, 4187785, 4181485, 4187790 ], 
        [ 4187786, 4192423, 4187791, 4192428 ], 
        [ 4192424, 4192625, 4192429, 4192630 ], 
        [ 4192626, 4194747, 4192631, 4194752 ], 
        [ 4194748, 4195600, 4194753, 4195605 ], 
        [ 4195601, 4195842, 4195606, 4195847 ], 
        [ 4195843, 4197634, 4195848, 4197639 ], 
        [ 4197635, 4198036, 4197640, 4198041 ], 
        [ 4198037, 4205660, 4198042, 4205665 ], 
        [ 4205661, 4213510, 4205666, 4213515 ], 
        [ 4213511, 4217238, 4213516, 4217243 ], 
        [ 4217239, 4221217, 4217244, 4221222 ], 
        [ 4221218, 4225159, 4221223, 4225164 ], 
        [ 4225160, 4226183, 4225165, 4226188 ], 
        [ 4226184, 4228901, 4226189, 4228906 ], 
        [ 4228902, 4229774, 4228907, 4229779 ], 
        [ 4229775, 4230559, 4229780, 4230564 ], 
        [ 4230560, 4235572, 4230565, 4235577 ], 
        [ 4235573, 4242029, 4235578, 4242034 ], 
        [ 4242030, 4242765, 4242035, 4242770 ], 
        [ 4242766, 4244054, 4242771, 4244059 ], 
        [ 4244055, 4245334, 4244060, 4245339 ], 
        [ 4245335, 4249019, 4245340, 4249024 ], 
        [ 4249020, 4249100, 4249025, 4249105 ], 
        [ 4249101, 4256705, 4249106, 4256710 ], 
        [ 4256706, 4259509, 4256711, 4259514 ], 
        [ 4259510, 4261410, 4259515, 4261415 ], 
        [ 4261411, 4270525, 4261416, 4270530 ], 
        [ 4270526, 4274109, 4270531, 4274114 ], 
        [ 4274110, 4281849, 4274115, 4281854 ], 
        [ 4281850, 4284762, 4281855, 4284767 ], 
        [ 4284763, 4303042, 4284768, 4303047 ], 
        [ 4303043, 4303258, 4303048, 4303263 ], 
        [ 4303259, 4306632, 4303264, 4306637 ], 
        [ 4306633, 4312216, 4306638, 4312221 ], 
        [ 4312217, 4314890, 4312222, 4314895 ], 
        [ 4314891, 4316460, 4314896, 4316465 ], 
        [ 4316461, 4316626, 4316466, 4316631 ], 
        [ 4316627, 4318092, 4316632, 4318097 ], 
        [ 4318093, 4318604, 4318098, 4318609 ], 
        [ 4318605, 4318772, 4318610, 4318777 ], 
        [ 4318773, 4321424, 4318778, 4321429 ], 
        [ 4321425, 4321489, 4321430, 4321494 ], 
        [ 4321490, 4327535, 4321495, 4327540 ], 
        [ 4327536, 4329548, 4327541, 4329553 ], 
        [ 4329549, 4331509, 4329554, 4331514 ], 
        [ 4331510, 4332147, 4331515, 4332152 ], 
        [ 4332148, 4334445, 4332153, 4334450 ], 
        [ 4334446, 4338820, 4334451, 4338825 ], 
        [ 4338821, 4339739, 4338826, 4339744 ], 
        [ 4339740, 4343682, 4339745, 4343687 ], 
        [ 4343683, 4348264, 4343688, 4348269 ], 
        [ 4348265, 4352206, 4348270, 4352211 ], 
        [ 4352207, 4356621, 4352212, 4356626 ], 
        [ 4356622, 4366184, 4356627, 4366189 ], 
        [ 4366185, 4374134, 4366190, 4374139 ], 
        [ 4374135, 4374749, 4374140, 4374754 ], 
        [ 4374750, 4388049, 4374755, 4388054 ], 
        [ 4388050, 4388548, 4388055, 4388553 ], 
        [ 4388549, 4395214, 4388554, 4395219 ], 
        [ 4395215, 4397725, 4395220, 4397730 ], 
        [ 4397726, 4401917, 4397731, 4401922 ], 
        [ 4401918, 4401962, 4401923, 4401967 ], 
        [ 4401963, 4404470, 4401968, 4404475 ], 
        [ 4404471, 4404603, 4404476, 4404608 ], 
        [ 4404604, 4404677, 4404609, 4404682 ], 
        [ 4404678, 4407845, 4404683, 4407850 ], 
        [ 4407846, 4415693, 4407851, 4415698 ], 
        [ 4415694, 4415957, 4415699, 4415962 ], 
        [ 4415958, 4426334, 4415963, 4426339 ], 
        [ 4426335, 4427143, 4426340, 4427148 ], 
        [ 4427144, 4432989, 4427149, 4432994 ], 
        [ 4432990, 4433640, 4432995, 4433645 ], 
        [ 4433641, 4435236, 4433646, 4435241 ], 
        [ 4435237, 4450981, 4435242, 4450986 ], 
        [ 4450982, 4452121, 4450987, 4452126 ], 
        [ 4452122, 4454417, 4452127, 4454422 ], 
        [ 4454418, 4455164, 4454423, 4455169 ], 
        [ 4455165, 4459226, 4455170, 4459231 ], 
        [ 4459227, 4462863, 4459232, 4462868 ], 
        [ 4462864, 4469714, 4462869, 4469719 ], 
        [ 4469715, 4471008, 4469720, 4471013 ], 
        [ 4471009, 4473114, 4471014, 4473119 ], 
        [ 4473115, 4477852, 4473120, 4477857 ], 
        [ 4477853, 4477874, 4477858, 4477879 ], 
        [ 4477875, 4482921, 4477880, 4482926 ], 
        [ 4482922, 4489809, 4482927, 4489814 ], 
        [ 4489810, 4490912, 4489815, 4490917 ], 
        [ 4490913, 4491974, 4490918, 4491979 ], 
        [ 4491975, 4492157, 4491980, 4492162 ], 
        [ 4492158, 4493614, 4492163, 4493619 ], 
        [ 4493615, 4496829, 4493620, 4496834 ], 
        [ 4496830, 4497697, 4496835, 4497702 ], 
        [ 4497698, 4499157, 4497703, 4499162 ], 
        [ 4499158, 4502248, 4499163, 4502253 ], 
        [ 4502249, 4504493, 4502254, 4504498 ], 
        [ 4504494, 4505336, 4504499, 4505341 ], 
        [ 4505337, 4506680, 4505342, 4506685 ], 
        [ 4506681, 4506961, 4506686, 4506966 ], 
        [ 4506962, 4507601, 4506967, 4507606 ], 
        [ 4507602, 4513351, 4507607, 4513356 ], 
        [ 4513352, 4516356, 4513357, 4516361 ], 
        [ 4516357, 4520650, 4516362, 4520655 ], 
        [ 4520651, 4528820, 4520656, 4528825 ], 
        [ 4528821, 4535971, 4528826, 4535976 ], 
        [ 4535972, 4540172, 4535977, 4540177 ], 
        [ 4540173, 4551230, 4540178, 4551235 ], 
        [ 4551231, 4552997, 4551236, 4553002 ], 
        [ 4552998, 4555491, 4553003, 4555496 ], 
        [ 4555492, 4558033, 4555497, 4558038 ], 
        [ 4558034, 4562123, 4558039, 4562128 ], 
        [ 4562124, 4563100, 4562129, 4563105 ], 
        [ 4563101, 4564639, 4563106, 4564644 ], 
        [ 4564640, 4566006, 4564645, 4566011 ], 
        [ 4566007, 4575457, 4566012, 4575462 ], 
        [ 4575458, 4575809, 4575463, 4575814 ], 
        [ 4575810, 4576253, 4575815, 4576258 ], 
        [ 4576254, 4579647, 4576259, 4579652 ], 
        [ 4579648, 4588823, 4579653, 4588828 ], 
        [ 4588824, 4589254, 4588829, 4589259 ], 
        [ 4589255, 4598670, 4589260, 4598675 ], 
        [ 4598671, 4601105, 4598676, 4601110 ], 
        [ 4601106, 4602566, 4601111, 4602571 ], 
        [ 4602567, 4612068, 4602572, 4612073 ], 
        [ 4612069, 4615603, 4612074, 4615608 ], 
        [ 4615604, 4627887, 4615609, 4627892 ], 
        [ 4627888, 4631394, 4627893, 4631399 ], 
        [ 4631395, 4631631, 4631400, 4631636 ], 
        [ 4631632, 4635963, 4631637, 4635968 ], 
        [ 4635964, 4641129, 4635969, 4641134 ], 
        [ 4641130, 4642980, 4641135, 4642985 ], 
        [ 4642981, 4643635, 4642986, 4643640 ], 
        [ 4643636, 4644147, 4643641, 4644152 ], 
        [ 4644148, 4649332, 4644153, 4649337 ], 
        [ 4649333, 4649464, 4649338, 4649469 ], 
        [ 4649465, 4656366, 4649470, 4656371 ], 
        [ 4656367, 4656864, 4656372, 4656869 ], 
        [ 4656865, 4656933, 4656870, 4656938 ], 
        [ 4656934, 4660056, 4656939, 4660061 ], 
        [ 4660057, 4665881, 4660062, 4665886 ], 
        [ 4665882, 4668837, 4665887, 4668842 ], 
        [ 4668838, 4672873, 4668843, 4672878 ], 
        [ 4672874, 4681462, 4672879, 4681467 ], 
        [ 4681463, 4696368, 4681468, 4696373 ], 
        [ 4696369, 4699474, 4696374, 4699479 ], 
        [ 4699475, 4704523, 4699480, 4704528 ], 
        [ 4704524, 4706008, 4704529, 4706013 ], 
        [ 4706009, 4706510, 4706014, 4706515 ], 
        [ 4706511, 4711295, 4706516, 4711300 ], 
        [ 4711296, 4711543, 4711301, 4711548 ], 
        [ 4711544, 4711935, 4711549, 4711940 ], 
        [ 4711936, 4712790, 4711941, 4712795 ], 
        [ 4712791, 4713126, 4712796, 4713131 ], 
        [ 4713127, 4713730, 4713132, 4713735 ], 
        [ 4713731, 4717619, 4713736, 4717624 ], 
        [ 4717620, 4724224, 4717625, 4724229 ], 
        [ 4724225, 4725868, 4724230, 4725873 ], 
        [ 4725869, 4727653, 4725874, 4727658 ], 
        [ 4727654, 4729069, 4727659, 4729074 ], 
        [ 4729070, 4730833, 4729075, 4730838 ], 
        [ 4730834, 4733099, 4730839, 4733104 ], 
        [ 4733100, 4733576, 4733105, 4733581 ], 
        [ 4733577, 4736754, 4733582, 4736759 ], 
        [ 4736755, 4741684, 4736760, 4741689 ], 
        [ 4741685, 4744830, 4741690, 4744835 ], 
        [ 4744831, 4746768, 4744836, 4746773 ], 
        [ 4746769, 4749037, 4746774, 4749042 ], 
        [ 4749038, 4749801, 4749043, 4749806 ], 
        [ 4749802, 4749864, 4749807, 4749869 ], 
        [ 4749865, 4750966, 4749870, 4750971 ], 
        [ 4750967, 4752965, 4750972, 4752970 ], 
        [ 4752966, 4754237, 4752971, 4754242 ], 
        [ 4754238, 4757191, 4754243, 4757196 ], 
        [ 4757192, 4762052, 4757197, 4762057 ], 
        [ 4762053, 4764164, 4762058, 4764169 ], 
        [ 4764165, 4766341, 4764170, 4766346 ], 
        [ 4766342, 4767519, 4766347, 4767524 ], 
        [ 4767520, 4769451, 4767525, 4769456 ], 
        [ 4769452, 4770366, 4769457, 4770371 ], 
        [ 4770367, 4774504, 4770372, 4774509 ], 
        [ 4774505, 4779310, 4774510, 4779315 ], 
        [ 4779311, 4784713, 4779316, 4784718 ], 
        [ 4784714, 4784960, 4784719, 4784965 ], 
        [ 4784961, 4789181, 4784966, 4789186 ], 
        [ 4789182, 4792894, 4789187, 4792899 ], 
        [ 4792895, 4804321, 4792900, 4804326 ], 
        [ 4804322, 4807780, 4804327, 4807785 ], 
        [ 4807781, 4808367, 4807786, 4808372 ], 
        [ 4808368, 4811025, 4808373, 4811030 ], 
        [ 4811026, 4813062, 4811031, 4813067 ], 
        [ 4813063, 4822160, 4813068, 4822165 ], 
        [ 4822161, 4833156, 4822166, 4833161 ], 
        [ 4833157, 4839621, 4833162, 4839626 ], 
        [ 4839622, 4853316, 4839627, 4853321 ], 
        [ 4853317, 4862268, 4853322, 4862273 ], 
        [ 4862269, 4862689, 4862274, 4862694 ], 
        [ 4862690, 4863453, 4862695, 4863458 ], 
        [ 4863454, 4863657, 4863459, 4863662 ], 
        [ 4863658, 4867215, 4863663, 4867220 ], 
        [ 4867216, 4867943, 4867221, 4867948 ], 
        [ 4867944, 4870367, 4867949, 4870372 ], 
        [ 4870368, 4871260, 4870373, 4871265 ], 
        [ 4871261, 4871925, 4871266, 4871930 ], 
        [ 4871926, 4872824, 4871931, 4872829 ], 
        [ 4872825, 4879935, 4872830, 4879940 ], 
        [ 4879936, 4881593, 4879941, 4881598 ], 
        [ 4881594, 4882087, 4881599, 4882092 ], 
        [ 4882088, 4889351, 4882093, 4889356 ], 
        [ 4889352, 4890443, 4889357, 4890448 ], 
        [ 4890444, 4898485, 4890449, 4898490 ], 
        [ 4898486, 4901057, 4898491, 4901062 ], 
        [ 4901058, 4904245, 4901063, 4904250 ], 
        [ 4904246, 4904668, 4904251, 4904673 ], 
        [ 4904669, 4904984, 4904674, 4904989 ], 
        [ 4904985, 4914224, 4904990, 4914229 ], 
        [ 4914225, 4916537, 4914230, 4916542 ], 
        [ 4916538, 4919908, 4916543, 4919913 ], 
        [ 4919909, 4926663, 4919914, 4926668 ], 
        [ 4926664, 4929329, 4926669, 4929334 ], 
        [ 4929330, 4930673, 4929335, 4930678 ], 
        [ 4930674, 4930954, 4930679, 4930959 ], 
        [ 4930955, 4931594, 4930960, 4931599 ], 
        [ 4931595, 4937282, 4931600, 4937287 ], 
        [ 4937283, 4939667, 4937288, 4939672 ], 
        [ 4939668, 4941837, 4939673, 4941842 ], 
        [ 4941838, 4947030, 4941843, 4947035 ], 
        [ 4947031, 4951071, 4947036, 4951076 ], 
        [ 4951072, 4953999, 4951077, 4954004 ], 
        [ 4954000, 4955481, 4954005, 4955486 ], 
        [ 4955482, 4959224, 4955487, 4959229 ], 
        [ 4959225, 4974624, 4959230, 4974629 ], 
        [ 4974625, 4977429, 4974630, 4977434 ], 
        [ 4977430, 4984448, 4977435, 4984453 ], 
        [ 4984449, 4986670, 4984454, 4986675 ], 
        [ 4986671, 4992038, 4986676, 4992043 ], 
        [ 4992039, 4993811, 4992044, 4993816 ], 
        [ 4993812, 4995631, 4993817, 4995636 ], 
        [ 4995632, 4996624, 4995637, 4996629 ], 
        [ 4996625, 4996668, 4996630, 4996673 ], 
        [ 4996669, 4998818, 4996674, 4998823 ], 
        [ 4998819, 5004186, 4998824, 5004191 ], 
        [ 5004187, 5013598, 5004192, 5013603 ], 
        [ 5013599, 5016180, 5013604, 5016185 ], 
        [ 5016181, 5018455, 5016186, 5018460 ], 
        [ 5018456, 5026770, 5018461, 5026775 ], 
        [ 5026771, 5028841, 5026776, 5028846 ], 
        [ 5028842, 5031862, 5028847, 5031867 ], 
        [ 5031863, 5036331, 5031868, 5036336 ], 
        [ 5036332, 5037861, 5036337, 5037866 ], 
        [ 5037862, 5038887, 5037867, 5038892 ], 
        [ 5038888, 5040440, 5038893, 5040445 ], 
        [ 5040441, 5042902, 5040446, 5042907 ], 
        [ 5042903, 5044827, 5042908, 5044832 ], 
        [ 5044828, 5050524, 5044833, 5050529 ], 
        [ 5050525, 5053866, 5050530, 5053871 ], 
        [ 5053867, 5054707, 5053872, 5054712 ], 
        [ 5054708, 5055021, 5054713, 5055026 ], 
        [ 5055022, 5057873, 5055027, 5057878 ], 
        [ 5057874, 5059734, 5057879, 5059739 ], 
        [ 5059735, 5061548, 5059740, 5061553 ], 
        [ 5061549, 5063342, 5061554, 5063347 ], 
        [ 5063343, 5064119, 5063348, 5064124 ], 
        [ 5064120, 5064638, 5064125, 5064643 ], 
        [ 5064639, 5068774, 5064644, 5068779 ], 
        [ 5068775, 5069157, 5068780, 5069162 ], 
        [ 5069158, 5069375, 5069163, 5069380 ], 
        [ 5069376, 5071533, 5069381, 5071538 ], 
        [ 5071534, 5072259, 5071539, 5072264 ], 
        [ 5072260, 5072332, 5072265, 5072337 ], 
        [ 5072333, 5074288, 5072338, 5074293 ], 
        [ 5074289, 5087508, 5074294, 5087513 ], 
        [ 5087509, 5088409, 5087514, 5088414 ], 
        [ 5088410, 5093963, 5088415, 5093968 ], 
        [ 5093964, 5098261, 5093969, 5098266 ], 
        [ 5098262, 5116037, 5098267, 5116042 ], 
        [ 5116038, 5116647, 5116043, 5116652 ], 
        [ 5116648, 5119282, 5116653, 5119287 ], 
        [ 5119283, 5132940, 5119288, 5132945 ], 
        [ 5132941, 5133405, 5132946, 5133410 ], 
        [ 5133406, 5134558, 5133411, 5134563 ], 
        [ 5134559, 5138432, 5134564, 5138437 ], 
        [ 5138433, 5138944, 5138438, 5138949 ], 
        [ 5138945, 5139157, 5138950, 5139162 ], 
        [ 5139158, 5139587, 5139163, 5139592 ], 
        [ 5139588, 5142617, 5139593, 5142622 ], 
        [ 5142618, 5148183, 5142623, 5148188 ], 
        [ 5148184, 5148672, 5148189, 5148677 ], 
        [ 5148673, 5150053, 5148678, 5150058 ], 
        [ 5150054, 5151087, 5150059, 5151092 ], 
        [ 5151088, 5153217, 5151093, 5153222 ], 
        [ 5153218, 5154383, 5153223, 5154388 ], 
        [ 5154384, 5155016, 5154389, 5155021 ], 
        [ 5155017, 5156599, 5155022, 5156604 ], 
        [ 5156600, 5157802, 5156605, 5157807 ], 
        [ 5157803, 5157970, 5157808, 5157975 ], 
        [ 5157971, 5160625, 5157976, 5160630 ], 
        [ 5160626, 5162852, 5160631, 5162857 ], 
        [ 5162853, 5164824, 5162858, 5164829 ], 
        [ 5164825, 5171077, 5164830, 5171082 ], 
        [ 5171078, 5180591, 5171083, 5180596 ], 
        [ 5180592, 5188235, 5180597, 5188240 ], 
        [ 5188236, 5194013, 5188241, 5194018 ], 
        [ 5194014, 5200363, 5194019, 5200368 ], 
        [ 5200364, 5200380, 5200369, 5200385 ], 
        [ 5200381, 5212416, 5200386, 5212421 ], 
        [ 5212417, 5214189, 5212422, 5214194 ], 
        [ 5214190, 5218448, 5214195, 5218453 ], 
        [ 5218449, 5221514, 5218454, 5221519 ], 
        [ 5221515, 5222405, 5221520, 5222410 ], 
        [ 5222406, 5223121, 5222411, 5223126 ], 
        [ 5223122, 5225062, 5223127, 5225067 ], 
        [ 5225063, 5227034, 5225068, 5227039 ], 
        [ 5227035, 5237062, 5227040, 5237067 ], 
        [ 5237063, 5238549, 5237068, 5238554 ], 
        [ 5238550, 5239941, 5238555, 5239946 ], 
        [ 5239942, 5241160, 5239947, 5241165 ], 
        [ 5241161, 5245009, 5241166, 5245014 ], 
        [ 5245010, 5245420, 5245015, 5245425 ], 
        [ 5245421, 5246665, 5245426, 5246670 ], 
        [ 5246666, 5246882, 5246671, 5246887 ], 
        [ 5246883, 5252321, 5246888, 5252326 ], 
        [ 5252322, 5261182, 5252327, 5261187 ], 
        [ 5261183, 5273111, 5261188, 5273116 ], 
        [ 5273112, 5273132, 5273117, 5273137 ], 
        [ 5273133, 5273680, 5273138, 5273685 ], 
        [ 5273681, 5274282, 5273686, 5274287 ], 
        [ 5274283, 5277485, 5274288, 5277490 ], 
        [ 5277486, 5278602, 5277491, 5278607 ], 
        [ 5278603, 5286668, 5278608, 5286673 ], 
        [ 5286669, 5288844, 5286674, 5288849 ], 
        [ 5288845, 5295426, 5288850, 5295431 ], 
        [ 5295427, 5299331, 5295432, 5299336 ], 
        [ 5299332, 5299740, 5299337, 5299745 ], 
        [ 5299741, 5302074, 5299746, 5302079 ], 
        [ 5302075, 5304927, 5302080, 5304932 ], 
        [ 5304928, 5316196, 5304933, 5316201 ], 
        [ 5316197, 5321884, 5316202, 5321889 ], 
        [ 5321885, 5327443, 5321890, 5327448 ], 
        [ 5327444, 5333009, 5327449, 5333014 ], 
        [ 5333010, 5335558, 5333015, 5335563 ], 
        [ 5335559, 5337934, 5335564, 5337939 ], 
        [ 5337935, 5340296, 5337940, 5340301 ], 
        [ 5340297, 5342510, 5340302, 5342515 ], 
        [ 5342511, 5343834, 5342516, 5343839 ], 
        [ 5343835, 5359283, 5343840, 5359288 ], 
        [ 5359284, 5362544, 5359289, 5362549 ], 
        [ 5362545, 5375729, 5362550, 5375734 ], 
        [ 5375730, 5375790, 5375735, 5375795 ], 
        [ 5375791, 5377253, 5375796, 5377258 ], 
        [ 5377254, 5378696, 5377259, 5378701 ], 
        [ 5378697, 5382060, 5378702, 5382065 ], 
        [ 5382061, 5388224, 5382066, 5388229 ], 
        [ 5388225, 5391633, 5388230, 5391638 ], 
        [ 5391634, 5401417, 5391639, 5401422 ], 
        [ 5401418, 5406537, 5401423, 5406542 ], 
        [ 5406538, 5408637, 5406543, 5408642 ], 
        [ 5408638, 5417270, 5408643, 5417275 ], 
        [ 5417271, 5419353, 5417276, 5419358 ], 
        [ 5419354, 5420144, 5419359, 5420149 ], 
        [ 5420145, 5420216, 5420150, 5420221 ], 
        [ 5420217, 5420740, 5420222, 5420745 ], 
        [ 5420741, 5427477, 5420746, 5427482 ], 
        [ 5427478, 5429323, 5427483, 5429328 ], 
        [ 5429324, 5441634, 5429329, 5441639 ], 
        [ 5441635, 5448663, 5441640, 5448668 ], 
        [ 5448664, 5452231, 5448669, 5452236 ], 
        [ 5452232, 5458274, 5452237, 5458279 ], 
        [ 5458275, 5459524, 5458280, 5459529 ], 
        [ 5459525, 5468509, 5459530, 5468514 ], 
        [ 5468510, 5469773, 5468515, 5469778 ], 
        [ 5469774, 5475379, 5469779, 5475384 ], 
        [ 5475380, 5476063, 5475385, 5476068 ], 
        [ 5476064, 5477860, 5476069, 5477865 ], 
        [ 5477861, 5478124, 5477866, 5478129 ], 
        [ 5478125, 5478577, 5478130, 5478582 ], 
        [ 5478578, 5479176, 5478583, 5479181 ], 
        [ 5479177, 5483012, 5479182, 5483017 ], 
        [ 5483013, 5483809, 5483018, 5483814 ], 
        [ 5483810, 5495234, 5483815, 5495239 ], 
        [ 5495235, 5498449, 5495240, 5498449 ]
      ]

    def _cut(seq)
      cuts = Bio::RestrictionEnzyme::Analysis.cut(seq, "BstEII",
                                                  {:view_ranges => true})
    end

    def test_BstEII_edge_cases
      (13481..13492).each do |len|
        _test_by_size(len)
      end
    end

  end # class TestEcoliO157H7_BstEII

  class TestEcoliO157H7_3enzymes < Test::Unit::TestCase
    include HelperMethods

    TestLabel = 'SacI+EcoRI+BstEII'

    SampleSequence = EcoliO157H7Seq

    
    SampleCutRanges = The3Enzymes_WHOLE =
      [ [ 0, 79, 0, 84 ], 
        [ 80, 3858, 85, 3862 ], 
        [ 3859, 4612, 3863, 4617 ], 
        [ 4613, 5619, 4618, 5623 ], 
        [ 5620, 7472, 5624, 7468 ], 
        [ 7473, 12905, 7469, 12909 ], 
        [ 12906, 13483, 12910, 13488 ], 
        [ 13484, 14551, 13489, 14547 ], 
        [ 14552, 15984, 14548, 15989 ], 
        [ 15985, 20045, 15990, 20049 ], 
        [ 20046, 21462, 20050, 21467 ], 
        [ 21463, 27326, 21468, 27331 ], 
        [ 27327, 30943, 27332, 30948 ], 
        [ 30944, 34888, 30949, 34893 ], 
        [ 34889, 35077, 34894, 35082 ], 
        [ 35078, 35310, 35083, 35315 ], 
        [ 35311, 36254, 35316, 36259 ], 
        [ 36255, 36648, 36260, 36652 ], 
        [ 36649, 36918, 36653, 36922 ], 
        [ 36919, 41885, 36923, 41890 ], 
        [ 41886, 43070, 41891, 43075 ], 
        [ 43071, 45689, 43076, 45694 ], 
        [ 45690, 48588, 45695, 48584 ], 
        [ 48589, 52325, 48585, 52330 ], 
        [ 52326, 54650, 52331, 54654 ], 
        [ 54651, 54728, 54655, 54732 ], 
        [ 54729, 55703, 54733, 55708 ], 
        [ 55704, 58828, 55709, 58833 ], 
        [ 58829, 59178, 58834, 59183 ], 
        [ 59179, 59800, 59184, 59796 ], 
        [ 59801, 61256, 59797, 61260 ], 
        [ 61257, 72610, 61261, 72615 ], 
        [ 72611, 72739, 72616, 72744 ], 
        [ 72740, 73099, 72745, 73104 ], 
        [ 73100, 75123, 73105, 75128 ], 
        [ 75124, 77366, 75129, 77371 ], 
        [ 77367, 77810, 77372, 77815 ], 
        [ 77811, 78740, 77816, 78745 ], 
        [ 78741, 79717, 78746, 79722 ], 
        [ 79718, 82250, 79723, 82255 ], 
        [ 82251, 84604, 82256, 84609 ], 
        [ 84605, 95491, 84610, 95496 ], 
        [ 95492, 95785, 95497, 95790 ], 
        [ 95786, 95794, 95791, 95799 ], 
        [ 95795, 96335, 95800, 96340 ], 
        [ 96336, 96489, 96341, 96493 ], 
        [ 96490, 101464, 96494, 101468 ], 
        [ 101465, 102044, 101469, 102049 ], 
        [ 102045, 102541, 102050, 102546 ], 
        [ 102542, 103192, 102547, 103197 ], 
        [ 103193, 103397, 103198, 103393 ], 
        [ 103398, 104722, 103394, 104727 ], 
        [ 104723, 106365, 104728, 106369 ], 
        [ 106366, 106896, 106370, 106900 ], 
        [ 106897, 107735, 106901, 107739 ], 
        [ 107736, 110020, 107740, 110024 ], 
        [ 110021, 110883, 110025, 110888 ], 
        [ 110884, 112524, 110889, 112528 ], 
        [ 112525, 113324, 112529, 113328 ], 
        [ 113325, 115867, 113329, 115871 ], 
        [ 115868, 117723, 115872, 117727 ], 
        [ 117724, 118742, 117728, 118738 ], 
        [ 118743, 120090, 118739, 120095 ], 
        [ 120091, 120657, 120096, 120662 ], 
        [ 120658, 128060, 120663, 128064 ], 
        [ 128061, 128308, 128065, 128313 ], 
        [ 128309, 136112, 128314, 136116 ], 
        [ 136113, 138305, 136117, 138310 ], 
        [ 138306, 138996, 138311, 139000 ], 
        [ 138997, 139146, 139001, 139142 ], 
        [ 139147, 141147, 139143, 141152 ], 
        [ 141148, 143724, 141153, 143729 ], 
        [ 143725, 143838, 143730, 143843 ], 
        [ 143839, 144303, 143844, 144308 ], 
        [ 144304, 148199, 144309, 148204 ], 
        [ 148200, 149577, 148205, 149582 ], 
        [ 149578, 149731, 149583, 149736 ], 
        [ 149732, 152137, 149737, 152141 ], 
        [ 152138, 156115, 152142, 156120 ], 
        [ 156116, 161126, 156121, 161131 ], 
        [ 161127, 162856, 161132, 162861 ], 
        [ 162857, 168965, 162862, 168961 ], 
        [ 168966, 170693, 168962, 170698 ], 
        [ 170694, 170944, 170699, 170949 ], 
        [ 170945, 171201, 170950, 171206 ], 
        [ 171202, 173241, 171207, 173246 ], 
        [ 173242, 177283, 173247, 177288 ], 
        [ 177284, 178048, 177289, 178052 ], 
        [ 178049, 178177, 178053, 178182 ], 
        [ 178178, 178781, 178183, 178786 ], 
        [ 178782, 181610, 178787, 181615 ], 
        [ 181611, 181706, 181616, 181711 ], 
        [ 181707, 185355, 181712, 185351 ], 
        [ 185356, 185661, 185352, 185666 ], 
        [ 185662, 193407, 185667, 193412 ], 
        [ 193408, 194141, 193413, 194145 ], 
        [ 194142, 194876, 194146, 194880 ], 
        [ 194877, 195511, 194881, 195516 ], 
        [ 195512, 195754, 195517, 195759 ], 
        [ 195755, 197005, 195760, 197009 ], 
        [ 197006, 197247, 197010, 197252 ], 
        [ 197248, 200659, 197253, 200664 ], 
        [ 200660, 201820, 200665, 201825 ], 
        [ 201821, 202300, 201826, 202305 ], 
        [ 202301, 202686, 202306, 202691 ], 
        [ 202687, 206289, 202692, 206294 ], 
        [ 206290, 206466, 206295, 206471 ], 
        [ 206467, 207011, 206472, 207016 ], 
        [ 207012, 208159, 207017, 208164 ], 
        [ 208160, 209976, 208165, 209981 ], 
        [ 209977, 210078, 209982, 210083 ], 
        [ 210079, 211485, 210084, 211490 ], 
        [ 211486, 212377, 211491, 212382 ], 
        [ 212378, 213569, 212383, 213574 ], 
        [ 213570, 214316, 213575, 214312 ], 
        [ 214317, 216005, 214313, 216010 ], 
        [ 216006, 217226, 216011, 217222 ], 
        [ 217227, 220098, 217223, 220103 ], 
        [ 220099, 221476, 220104, 221480 ], 
        [ 221477, 221641, 221481, 221645 ], 
        [ 221642, 224063, 221646, 224068 ], 
        [ 224064, 227774, 224069, 227778 ], 
        [ 227775, 228604, 227779, 228609 ], 
        [ 228605, 229453, 228610, 229449 ], 
        [ 229454, 229931, 229450, 229935 ], 
        [ 229932, 232247, 229936, 232251 ], 
        [ 232248, 235221, 232252, 235225 ], 
        [ 235222, 237291, 235226, 237295 ], 
        [ 237292, 239035, 237296, 239039 ], 
        [ 239036, 239993, 239040, 239998 ], 
        [ 239994, 240624, 239999, 240628 ], 
        [ 240625, 240887, 240629, 240891 ], 
        [ 240888, 242089, 240892, 242093 ], 
        [ 242090, 243880, 242094, 243884 ], 
        [ 243881, 245321, 243885, 245325 ], 
        [ 245322, 247914, 245326, 247919 ], 
        [ 247915, 251579, 247920, 251584 ], 
        [ 251580, 257092, 251585, 257097 ], 
        [ 257093, 259887, 257098, 259891 ], 
        [ 259888, 260535, 259892, 260539 ], 
        [ 260536, 261621, 260540, 261626 ], 
        [ 261622, 263030, 261627, 263035 ], 
        [ 263031, 264258, 263036, 264262 ], 
        [ 264259, 265004, 264263, 265008 ], 
        [ 265005, 265084, 265009, 265089 ], 
        [ 265085, 265243, 265090, 265248 ], 
        [ 265244, 265534, 265249, 265539 ], 
        [ 265535, 266117, 265540, 266122 ], 
        [ 266118, 274428, 266123, 274433 ], 
        [ 274429, 275235, 274434, 275231 ], 
        [ 275236, 276946, 275232, 276950 ], 
        [ 276947, 277457, 276951, 277461 ], 
        [ 277458, 279137, 277462, 279133 ], 
        [ 279138, 282285, 279134, 282290 ], 
        [ 282286, 286948, 282291, 286953 ], 
        [ 286949, 288342, 286954, 288338 ], 
        [ 288343, 289897, 288339, 289901 ], 
        [ 289898, 292547, 289902, 292552 ], 
        [ 292548, 297678, 292553, 297683 ], 
        [ 297679, 303902, 297684, 303906 ], 
        [ 303903, 304580, 303907, 304584 ], 
        [ 304581, 307362, 304585, 307366 ], 
        [ 307363, 307931, 307367, 307935 ], 
        [ 307932, 308161, 307936, 308166 ], 
        [ 308162, 308706, 308167, 308711 ], 
        [ 308707, 313482, 308712, 313487 ], 
        [ 313483, 316025, 313488, 316021 ], 
        [ 316026, 324159, 316022, 324163 ], 
        [ 324160, 326130, 324164, 326134 ], 
        [ 326131, 331620, 326135, 331624 ], 
        [ 331621, 336338, 331625, 336342 ], 
        [ 336339, 336873, 336343, 336877 ], 
        [ 336874, 337118, 336878, 337123 ], 
        [ 337119, 337935, 337124, 337940 ], 
        [ 337936, 338781, 337941, 338786 ], 
        [ 338782, 339493, 338787, 339498 ], 
        [ 339494, 341025, 339499, 341030 ], 
        [ 341026, 343919, 341031, 343923 ], 
        [ 343920, 344424, 343924, 344429 ], 
        [ 344425, 348384, 344430, 348389 ], 
        [ 348385, 348408, 348390, 348404 ], 
        [ 348409, 353417, 348405, 353413 ], 
        [ 353418, 354781, 353414, 354786 ], 
        [ 354782, 356692, 354787, 356697 ], 
        [ 356693, 357008, 356698, 357013 ], 
        [ 357009, 357305, 357014, 357310 ], 
        [ 357306, 357328, 357311, 357333 ], 
        [ 357329, 358126, 357334, 358131 ], 
        [ 358127, 359472, 358132, 359477 ], 
        [ 359473, 362160, 359478, 362165 ], 
        [ 362161, 365395, 362166, 365400 ], 
        [ 365396, 365704, 365401, 365709 ], 
        [ 365705, 368663, 365710, 368667 ], 
        [ 368664, 368841, 368668, 368845 ], 
        [ 368842, 370589, 368846, 370593 ], 
        [ 370590, 371148, 370594, 371152 ], 
        [ 371149, 373639, 371153, 373643 ], 
        [ 373640, 377393, 373644, 377397 ], 
        [ 377394, 381068, 377398, 381072 ], 
        [ 381069, 381692, 381073, 381688 ], 
        [ 381693, 381746, 381689, 381751 ], 
        [ 381747, 381994, 381752, 381999 ], 
        [ 381995, 383335, 382000, 383340 ], 
        [ 383336, 385141, 383341, 385146 ], 
        [ 385142, 389399, 385147, 389403 ], 
        [ 389400, 390171, 389404, 390176 ], 
        [ 390172, 392340, 390177, 392344 ], 
        [ 392341, 392764, 392345, 392769 ], 
        [ 392765, 394338, 392770, 394343 ], 
        [ 394339, 394686, 394344, 394691 ], 
        [ 394687, 397592, 394692, 397596 ], 
        [ 397593, 398703, 397597, 398708 ], 
        [ 398704, 404095, 398709, 404100 ], 
        [ 404096, 408361, 404101, 408366 ], 
        [ 408362, 409029, 408367, 409025 ], 
        [ 409030, 413032, 409026, 413037 ], 
        [ 413033, 414563, 413038, 414568 ], 
        [ 414564, 416901, 414569, 416906 ], 
        [ 416902, 417419, 416907, 417424 ], 
        [ 417420, 420057, 417425, 420061 ], 
        [ 420058, 421129, 420062, 421125 ], 
        [ 421130, 421777, 421126, 421782 ], 
        [ 421778, 423748, 421783, 423753 ], 
        [ 423749, 431903, 423754, 431908 ], 
        [ 431904, 432852, 431909, 432848 ], 
        [ 432853, 440000, 432849, 440005 ], 
        [ 440001, 440754, 440006, 440750 ], 
        [ 440755, 444226, 440751, 444222 ], 
        [ 444227, 448040, 444223, 448045 ], 
        [ 448041, 452994, 448046, 452999 ], 
        [ 452995, 453075, 453000, 453080 ], 
        [ 453076, 454654, 453081, 454658 ], 
        [ 454655, 454950, 454659, 454955 ], 
        [ 454951, 455888, 454956, 455893 ], 
        [ 455889, 460160, 455894, 460165 ], 
        [ 460161, 462319, 460166, 462323 ], 
        [ 462320, 462650, 462324, 462654 ], 
        [ 462651, 463076, 462655, 463081 ], 
        [ 463077, 465003, 463082, 465008 ], 
        [ 465004, 466828, 465009, 466833 ], 
        [ 466829, 467686, 466834, 467691 ], 
        [ 467687, 468596, 467692, 468601 ], 
        [ 468597, 475083, 468602, 475087 ], 
        [ 475084, 479953, 475088, 479958 ], 
        [ 479954, 480538, 479959, 480543 ], 
        [ 480539, 482480, 480544, 482484 ], 
        [ 482481, 482869, 482485, 482874 ], 
        [ 482870, 483410, 482875, 483414 ], 
        [ 483411, 489378, 483415, 489383 ], 
        [ 489379, 492112, 489384, 492116 ], 
        [ 492113, 492241, 492117, 492246 ], 
        [ 492242, 493791, 492247, 493795 ], 
        [ 493792, 495406, 493796, 495411 ], 
        [ 495407, 495712, 495412, 495717 ], 
        [ 495713, 497829, 495718, 497834 ], 
        [ 497830, 501698, 497835, 501703 ], 
        [ 501699, 503304, 501704, 503308 ], 
        [ 503305, 504565, 503309, 504570 ], 
        [ 504566, 505105, 504571, 505110 ], 
        [ 505106, 508452, 505111, 508457 ], 
        [ 508453, 514353, 508458, 514357 ], 
        [ 514354, 515947, 514358, 515952 ], 
        [ 515948, 519141, 515953, 519146 ], 
        [ 519142, 519398, 519147, 519403 ], 
        [ 519399, 519662, 519404, 519666 ], 
        [ 519663, 521386, 519667, 521391 ], 
        [ 521387, 521935, 521392, 521939 ], 
        [ 521936, 523114, 521940, 523118 ], 
        [ 523115, 524176, 523119, 524180 ], 
        [ 524177, 524521, 524181, 524525 ], 
        [ 524522, 524936, 524526, 524932 ], 
        [ 524937, 526115, 524933, 526120 ], 
        [ 526116, 526729, 526121, 526734 ], 
        [ 526730, 527018, 526735, 527023 ], 
        [ 527019, 528059, 527024, 528064 ], 
        [ 528060, 532689, 528065, 532694 ], 
        [ 532690, 534193, 532695, 534189 ], 
        [ 534194, 534702, 534190, 534707 ], 
        [ 534703, 535272, 534708, 535277 ], 
        [ 535273, 538638, 535278, 538642 ], 
        [ 538639, 538668, 538643, 538673 ], 
        [ 538669, 543939, 538674, 543944 ], 
        [ 543940, 547429, 543945, 547434 ], 
        [ 547430, 547624, 547435, 547628 ], 
        [ 547625, 550898, 547629, 550902 ], 
        [ 550899, 553890, 550903, 553895 ], 
        [ 553891, 554678, 553896, 554683 ], 
        [ 554679, 555452, 554684, 555457 ], 
        [ 555453, 556296, 555458, 556301 ], 
        [ 556297, 557116, 556302, 557120 ], 
        [ 557117, 559341, 557121, 559346 ], 
        [ 559342, 559991, 559347, 559996 ], 
        [ 559992, 563242, 559997, 563247 ], 
        [ 563243, 563390, 563248, 563394 ], 
        [ 563391, 566071, 563395, 566075 ], 
        [ 566072, 566857, 566076, 566861 ], 
        [ 566858, 571925, 566862, 571929 ], 
        [ 571926, 576432, 571930, 576437 ], 
        [ 576433, 582431, 576438, 582436 ], 
        [ 582432, 582959, 582437, 582964 ], 
        [ 582960, 583475, 582965, 583480 ], 
        [ 583476, 583589, 583481, 583594 ], 
        [ 583590, 583670, 583595, 583675 ], 
        [ 583671, 583901, 583676, 583906 ], 
        [ 583902, 584198, 583907, 584203 ], 
        [ 584199, 584633, 584204, 584638 ], 
        [ 584634, 585704, 584639, 585709 ], 
        [ 585705, 585746, 585710, 585751 ], 
        [ 585747, 586175, 585752, 586180 ], 
        [ 586176, 586301, 586181, 586306 ], 
        [ 586302, 586643, 586307, 586648 ], 
        [ 586644, 586775, 586649, 586780 ], 
        [ 586776, 587072, 586781, 587077 ], 
        [ 587073, 587214, 587078, 587219 ], 
        [ 587215, 587540, 587220, 587545 ], 
        [ 587541, 587969, 587546, 587974 ], 
        [ 587970, 588095, 587975, 588100 ], 
        [ 588096, 588437, 588101, 588442 ], 
        [ 588438, 588569, 588443, 588574 ], 
        [ 588570, 589008, 588575, 589013 ], 
        [ 589009, 589166, 589014, 589171 ], 
        [ 589167, 590366, 589172, 590371 ], 
        [ 590367, 590792, 590372, 590797 ], 
        [ 590793, 591077, 590798, 591082 ], 
        [ 591078, 591263, 591083, 591268 ], 
        [ 591264, 591863, 591269, 591868 ], 
        [ 591864, 592058, 591869, 592063 ], 
        [ 592059, 592160, 592064, 592165 ], 
        [ 592161, 592568, 592166, 592573 ], 
        [ 592569, 592760, 592574, 592765 ], 
        [ 592761, 593060, 592766, 593065 ], 
        [ 593061, 593186, 593066, 593191 ], 
        [ 593187, 593366, 593192, 593371 ], 
        [ 593367, 593957, 593372, 593962 ], 
        [ 593958, 594827, 593963, 594832 ], 
        [ 594828, 594980, 594833, 594985 ], 
        [ 594981, 595649, 594986, 595654 ], 
        [ 595650, 595893, 595655, 595898 ], 
        [ 595894, 596057, 595899, 596062 ], 
        [ 596058, 596159, 596063, 596164 ], 
        [ 596160, 596351, 596165, 596356 ], 
        [ 596352, 596660, 596357, 596665 ], 
        [ 596661, 596960, 596666, 596965 ], 
        [ 596961, 597102, 596966, 597107 ], 
        [ 597103, 597155, 597108, 597160 ], 
        [ 597156, 597257, 597161, 597262 ], 
        [ 597258, 599957, 597263, 599962 ], 
        [ 599958, 604182, 599963, 604186 ], 
        [ 604183, 611038, 604187, 611043 ], 
        [ 611039, 612202, 611044, 612207 ], 
        [ 612203, 614051, 612208, 614056 ], 
        [ 614052, 614134, 614057, 614139 ], 
        [ 614135, 614787, 614140, 614792 ], 
        [ 614788, 616272, 614793, 616277 ], 
        [ 616273, 616867, 616278, 616871 ], 
        [ 616868, 617737, 616872, 617742 ], 
        [ 617738, 627339, 617743, 627344 ], 
        [ 627340, 628902, 627345, 628907 ], 
        [ 628903, 629142, 628908, 629146 ], 
        [ 629143, 629458, 629147, 629454 ], 
        [ 629459, 636523, 629455, 636528 ], 
        [ 636524, 637529, 636529, 637534 ], 
        [ 637530, 639061, 637535, 639065 ], 
        [ 639062, 647713, 639066, 647718 ], 
        [ 647714, 648684, 647719, 648689 ], 
        [ 648685, 652752, 648690, 652748 ], 
        [ 652753, 653543, 652749, 653548 ], 
        [ 653544, 654406, 653549, 654410 ], 
        [ 654407, 658188, 654411, 658192 ], 
        [ 658189, 659030, 658193, 659035 ], 
        [ 659031, 662241, 659036, 662246 ], 
        [ 662242, 670896, 662247, 670900 ], 
        [ 670897, 671781, 670901, 671786 ], 
        [ 671782, 672048, 671787, 672053 ], 
        [ 672049, 673788, 672054, 673793 ], 
        [ 673789, 674707, 673794, 674712 ], 
        [ 674708, 674998, 674713, 675003 ], 
        [ 674999, 675157, 675004, 675162 ], 
        [ 675158, 688595, 675163, 688600 ], 
        [ 688596, 693309, 688601, 693314 ], 
        [ 693310, 693523, 693315, 693527 ], 
        [ 693524, 696514, 693528, 696518 ], 
        [ 696515, 697406, 696519, 697411 ], 
        [ 697407, 702676, 697412, 702681 ], 
        [ 702677, 707208, 702682, 707212 ], 
        [ 707209, 707382, 707213, 707387 ], 
        [ 707383, 708604, 707388, 708609 ], 
        [ 708605, 710046, 708610, 710051 ], 
        [ 710047, 711630, 710052, 711635 ], 
        [ 711631, 711696, 711636, 711701 ], 
        [ 711697, 712329, 711702, 712334 ], 
        [ 712330, 714099, 712335, 714103 ], 
        [ 714100, 716461, 714104, 716466 ], 
        [ 716462, 720238, 716467, 720243 ], 
        [ 720239, 720374, 720244, 720379 ], 
        [ 720375, 720471, 720380, 720475 ], 
        [ 720472, 721463, 720476, 721467 ], 
        [ 721464, 723143, 721468, 723147 ], 
        [ 723144, 723348, 723148, 723352 ], 
        [ 723349, 724200, 723353, 724205 ], 
        [ 724201, 725464, 724206, 725468 ], 
        [ 725465, 725687, 725469, 725692 ], 
        [ 725688, 729467, 725693, 729471 ], 
        [ 729468, 730067, 729472, 730072 ], 
        [ 730068, 730406, 730073, 730410 ], 
        [ 730407, 730574, 730411, 730579 ], 
        [ 730575, 730699, 730580, 730704 ], 
        [ 730700, 732726, 730705, 732731 ], 
        [ 732727, 734363, 732732, 734359 ], 
        [ 734364, 738597, 734360, 738602 ], 
        [ 738598, 738869, 738603, 738873 ], 
        [ 738870, 739571, 738874, 739575 ], 
        [ 739572, 742040, 739576, 742044 ], 
        [ 742041, 742350, 742045, 742346 ], 
        [ 742351, 743326, 742347, 743331 ], 
        [ 743327, 743557, 743332, 743561 ], 
        [ 743558, 743966, 743562, 743970 ], 
        [ 743967, 744992, 743971, 744997 ], 
        [ 744993, 745843, 744998, 745848 ], 
        [ 745844, 751518, 745849, 751523 ], 
        [ 751519, 752431, 751524, 752436 ], 
        [ 752432, 752549, 752437, 752554 ], 
        [ 752550, 758622, 752555, 758626 ], 
        [ 758623, 760978, 758627, 760982 ], 
        [ 760979, 761319, 760983, 761315 ], 
        [ 761320, 766036, 761316, 766041 ], 
        [ 766037, 768968, 766042, 768973 ], 
        [ 768969, 770151, 768974, 770156 ], 
        [ 770152, 771158, 770157, 771163 ], 
        [ 771159, 771405, 771164, 771410 ], 
        [ 771406, 771429, 771411, 771433 ], 
        [ 771430, 774384, 771434, 774388 ], 
        [ 774385, 781958, 774389, 781963 ], 
        [ 781959, 784226, 781964, 784231 ], 
        [ 784227, 784572, 784232, 784576 ], 
        [ 784573, 784806, 784577, 784810 ], 
        [ 784807, 786886, 784811, 786890 ], 
        [ 786887, 786945, 786891, 786950 ], 
        [ 786946, 787203, 786951, 787208 ], 
        [ 787204, 789251, 787209, 789256 ], 
        [ 789252, 791218, 789257, 791223 ], 
        [ 791219, 793716, 791224, 793721 ], 
        [ 793717, 795003, 793722, 795008 ], 
        [ 795004, 795521, 795009, 795526 ], 
        [ 795522, 800659, 795527, 800663 ], 
        [ 800660, 802360, 800664, 802364 ], 
        [ 802361, 804514, 802365, 804519 ], 
        [ 804515, 805238, 804520, 805243 ], 
        [ 805239, 805887, 805244, 805892 ], 
        [ 805888, 807288, 805893, 807292 ], 
        [ 807289, 808461, 807293, 808466 ], 
        [ 808462, 808692, 808467, 808696 ], 
        [ 808693, 809805, 808697, 809810 ], 
        [ 809806, 810086, 809811, 810091 ], 
        [ 810087, 810726, 810092, 810731 ], 
        [ 810727, 813170, 810732, 813174 ], 
        [ 813171, 813863, 813175, 813867 ], 
        [ 813864, 820111, 813868, 820116 ], 
        [ 820112, 821326, 820117, 821331 ], 
        [ 821327, 821647, 821332, 821652 ], 
        [ 821648, 824277, 821653, 824282 ], 
        [ 824278, 825750, 824283, 825755 ], 
        [ 825751, 828770, 825756, 828775 ], 
        [ 828771, 828924, 828776, 828929 ], 
        [ 828925, 830194, 828930, 830199 ], 
        [ 830195, 830786, 830200, 830791 ], 
        [ 830787, 831245, 830792, 831241 ], 
        [ 831246, 832788, 831242, 832793 ], 
        [ 832789, 833306, 832794, 833311 ], 
        [ 833307, 835264, 833312, 835260 ], 
        [ 835265, 835656, 835261, 835661 ], 
        [ 835657, 841180, 835662, 841185 ], 
        [ 841181, 842112, 841186, 842117 ], 
        [ 842113, 842524, 842118, 842528 ], 
        [ 842525, 843973, 842529, 843978 ], 
        [ 843974, 843990, 843979, 843995 ], 
        [ 843991, 851267, 843996, 851271 ], 
        [ 851268, 852882, 851272, 852887 ], 
        [ 852883, 854392, 852888, 854397 ], 
        [ 854393, 857721, 854398, 857726 ], 
        [ 857722, 857961, 857727, 857966 ], 
        [ 857962, 859112, 857967, 859116 ], 
        [ 859113, 862783, 859117, 862788 ], 
        [ 862784, 869922, 862789, 869926 ], 
        [ 869923, 878953, 869927, 878958 ], 
        [ 878954, 885194, 878959, 885199 ], 
        [ 885195, 886313, 885200, 886318 ], 
        [ 886314, 886460, 886319, 886465 ], 
        [ 886461, 888041, 886466, 888045 ], 
        [ 888042, 890161, 888046, 890165 ], 
        [ 890162, 890233, 890166, 890238 ], 
        [ 890234, 890346, 890239, 890351 ], 
        [ 890347, 890379, 890352, 890384 ], 
        [ 890380, 895074, 890385, 895078 ], 
        [ 895075, 897876, 895079, 897880 ], 
        [ 897877, 897943, 897881, 897947 ], 
        [ 897944, 899572, 897948, 899576 ], 
        [ 899573, 899676, 899577, 899681 ], 
        [ 899677, 903962, 899682, 903967 ], 
        [ 903963, 904236, 903968, 904241 ], 
        [ 904237, 908130, 904242, 908135 ], 
        [ 908131, 912131, 908136, 912127 ], 
        [ 912132, 916611, 912128, 916616 ], 
        [ 916612, 916803, 916617, 916808 ], 
        [ 916804, 920531, 916809, 920536 ], 
        [ 920532, 923592, 920537, 923596 ], 
        [ 923593, 927519, 923597, 927523 ], 
        [ 927520, 928181, 927524, 928185 ], 
        [ 928182, 928505, 928186, 928510 ], 
        [ 928506, 928644, 928511, 928640 ], 
        [ 928645, 934935, 928641, 934931 ], 
        [ 934936, 935911, 934932, 935915 ], 
        [ 935912, 936947, 935916, 936952 ], 
        [ 936948, 937240, 936953, 937245 ], 
        [ 937241, 939698, 937246, 939703 ], 
        [ 939699, 939711, 939704, 939716 ], 
        [ 939712, 941642, 939717, 941647 ], 
        [ 941643, 949052, 941648, 949057 ], 
        [ 949053, 949800, 949058, 949805 ], 
        [ 949801, 949851, 949806, 949855 ], 
        [ 949852, 951412, 949856, 951417 ], 
        [ 951413, 951810, 951418, 951815 ], 
        [ 951811, 952386, 951816, 952391 ], 
        [ 952387, 953295, 952392, 953300 ], 
        [ 953296, 953894, 953301, 953899 ], 
        [ 953895, 955768, 953900, 955772 ], 
        [ 955769, 955953, 955773, 955957 ], 
        [ 955954, 956045, 955958, 956041 ], 
        [ 956046, 958753, 956042, 958758 ], 
        [ 958754, 964476, 958759, 964481 ], 
        [ 964477, 967468, 964482, 967473 ], 
        [ 967469, 969625, 967474, 969629 ], 
        [ 969626, 969631, 969630, 969636 ], 
        [ 969632, 970966, 969637, 970971 ], 
        [ 970967, 971138, 970972, 971143 ], 
        [ 971139, 974185, 971144, 974190 ], 
        [ 974186, 974365, 974191, 974370 ], 
        [ 974366, 975256, 974371, 975261 ], 
        [ 975257, 976794, 975262, 976799 ], 
        [ 976795, 980694, 976800, 980698 ], 
        [ 980695, 987406, 980699, 987411 ], 
        [ 987407, 988132, 987412, 988137 ], 
        [ 988133, 992809, 988138, 992814 ], 
        [ 992810, 996307, 992815, 996311 ], 
        [ 996308, 999121, 996312, 999125 ], 
        [ 999122, 1000225, 999126, 1000230 ], 
        [ 1000226, 1001626, 1000231, 1001631 ], 
        [ 1001627, 1005050, 1001632, 1005054 ], 
        [ 1005051, 1007354, 1005055, 1007359 ], 
        [ 1007355, 1011910, 1007360, 1011915 ], 
        [ 1011911, 1012377, 1011916, 1012382 ], 
        [ 1012378, 1015175, 1012383, 1015179 ], 
        [ 1015176, 1017328, 1015180, 1017333 ], 
        [ 1017329, 1020891, 1017334, 1020896 ], 
        [ 1020892, 1021340, 1020897, 1021345 ], 
        [ 1021341, 1024845, 1021346, 1024850 ], 
        [ 1024846, 1025853, 1024851, 1025858 ], 
        [ 1025854, 1030691, 1025859, 1030696 ], 
        [ 1030692, 1032676, 1030697, 1032681 ], 
        [ 1032677, 1037847, 1032682, 1037852 ], 
        [ 1037848, 1039473, 1037853, 1039478 ], 
        [ 1039474, 1039858, 1039479, 1039854 ], 
        [ 1039859, 1044241, 1039855, 1044246 ], 
        [ 1044242, 1045920, 1044247, 1045925 ], 
        [ 1045921, 1049805, 1045926, 1049809 ], 
        [ 1049806, 1050388, 1049810, 1050392 ], 
        [ 1050389, 1053286, 1050393, 1053291 ], 
        [ 1053287, 1053309, 1053292, 1053314 ], 
        [ 1053310, 1054643, 1053315, 1054648 ], 
        [ 1054644, 1056252, 1054649, 1056256 ], 
        [ 1056253, 1056527, 1056257, 1056532 ], 
        [ 1056528, 1056947, 1056533, 1056943 ], 
        [ 1056948, 1058682, 1056944, 1058687 ], 
        [ 1058683, 1059297, 1058688, 1059302 ], 
        [ 1059298, 1060416, 1059303, 1060421 ], 
        [ 1060417, 1061894, 1060422, 1061898 ], 
        [ 1061895, 1064234, 1061899, 1064239 ], 
        [ 1064235, 1064848, 1064240, 1064853 ], 
        [ 1064849, 1065434, 1064854, 1065439 ], 
        [ 1065435, 1075046, 1065440, 1075042 ], 
        [ 1075047, 1075642, 1075043, 1075647 ], 
        [ 1075643, 1076325, 1075648, 1076330 ], 
        [ 1076326, 1076534, 1076331, 1076539 ], 
        [ 1076535, 1078135, 1076540, 1078139 ], 
        [ 1078136, 1078866, 1078140, 1078871 ], 
        [ 1078867, 1079914, 1078872, 1079910 ], 
        [ 1079915, 1080537, 1079911, 1080542 ], 
        [ 1080538, 1082144, 1080543, 1082149 ], 
        [ 1082145, 1083416, 1082150, 1083420 ], 
        [ 1083417, 1085746, 1083421, 1085751 ], 
        [ 1085747, 1087087, 1085752, 1087092 ], 
        [ 1087088, 1088273, 1087093, 1088278 ], 
        [ 1088274, 1092093, 1088279, 1092097 ], 
        [ 1092094, 1093062, 1092098, 1093067 ], 
        [ 1093063, 1096867, 1093068, 1096872 ], 
        [ 1096868, 1097288, 1096873, 1097292 ], 
        [ 1097289, 1102488, 1097293, 1102493 ], 
        [ 1102489, 1102751, 1102494, 1102747 ], 
        [ 1102752, 1104366, 1102748, 1104362 ], 
        [ 1104367, 1106371, 1104363, 1106376 ], 
        [ 1106372, 1108123, 1106377, 1108128 ], 
        [ 1108124, 1112263, 1108129, 1112259 ], 
        [ 1112264, 1113311, 1112260, 1113316 ], 
        [ 1113312, 1114557, 1113317, 1114562 ], 
        [ 1114558, 1117715, 1114563, 1117719 ], 
        [ 1117716, 1118552, 1117720, 1118556 ], 
        [ 1118553, 1120566, 1118557, 1120571 ], 
        [ 1120567, 1121004, 1120572, 1121009 ], 
        [ 1121005, 1121076, 1121010, 1121080 ], 
        [ 1121077, 1121609, 1121081, 1121613 ], 
        [ 1121610, 1121694, 1121614, 1121698 ], 
        [ 1121695, 1122501, 1121699, 1122506 ], 
        [ 1122502, 1130582, 1122507, 1130587 ], 
        [ 1130583, 1132170, 1130588, 1132175 ], 
        [ 1132171, 1135259, 1132176, 1135263 ], 
        [ 1135260, 1136119, 1135264, 1136123 ], 
        [ 1136120, 1137316, 1136124, 1137320 ], 
        [ 1137317, 1140126, 1137321, 1140131 ], 
        [ 1140127, 1142998, 1140132, 1143002 ], 
        [ 1142999, 1143361, 1143003, 1143366 ], 
        [ 1143362, 1143637, 1143367, 1143633 ], 
        [ 1143638, 1143644, 1143634, 1143648 ], 
        [ 1143645, 1146618, 1143649, 1146622 ], 
        [ 1146619, 1149205, 1146623, 1149210 ], 
        [ 1149206, 1149331, 1149211, 1149336 ], 
        [ 1149332, 1152263, 1149337, 1152259 ], 
        [ 1152264, 1152809, 1152260, 1152805 ], 
        [ 1152810, 1154382, 1152806, 1154386 ], 
        [ 1154383, 1156272, 1154387, 1156277 ], 
        [ 1156273, 1159968, 1156278, 1159972 ], 
        [ 1159969, 1161624, 1159973, 1161629 ], 
        [ 1161625, 1163044, 1161630, 1163048 ], 
        [ 1163045, 1164030, 1163049, 1164026 ], 
        [ 1164031, 1166628, 1164027, 1166632 ], 
        [ 1166629, 1167897, 1166633, 1167901 ], 
        [ 1167898, 1171353, 1167902, 1171358 ], 
        [ 1171354, 1171934, 1171359, 1171939 ], 
        [ 1171935, 1172114, 1171940, 1172119 ], 
        [ 1172115, 1173373, 1172120, 1173369 ], 
        [ 1173374, 1175421, 1173370, 1175417 ], 
        [ 1175422, 1179850, 1175418, 1179854 ], 
        [ 1179851, 1181462, 1179855, 1181466 ], 
        [ 1181463, 1185368, 1181467, 1185373 ], 
        [ 1185369, 1193993, 1185374, 1193998 ], 
        [ 1193994, 1194272, 1193999, 1194277 ], 
        [ 1194273, 1195941, 1194278, 1195945 ], 
        [ 1195942, 1197920, 1195946, 1197925 ], 
        [ 1197921, 1199011, 1197926, 1199007 ], 
        [ 1199012, 1200079, 1199008, 1200075 ], 
        [ 1200080, 1200373, 1200076, 1200378 ], 
        [ 1200374, 1200419, 1200379, 1200423 ], 
        [ 1200420, 1200597, 1200424, 1200602 ], 
        [ 1200598, 1200714, 1200603, 1200719 ], 
        [ 1200715, 1203674, 1200720, 1203679 ], 
        [ 1203675, 1204865, 1203680, 1204870 ], 
        [ 1204866, 1205330, 1204871, 1205335 ], 
        [ 1205331, 1206317, 1205336, 1206321 ], 
        [ 1206318, 1210727, 1206322, 1210732 ], 
        [ 1210728, 1211881, 1210733, 1211886 ], 
        [ 1211882, 1214283, 1211887, 1214288 ], 
        [ 1214284, 1215542, 1214289, 1215546 ], 
        [ 1215543, 1216981, 1215547, 1216986 ], 
        [ 1216982, 1220754, 1216987, 1220750 ], 
        [ 1220755, 1221554, 1220751, 1221550 ], 
        [ 1221555, 1222528, 1221551, 1222532 ], 
        [ 1222529, 1223522, 1222533, 1223527 ], 
        [ 1223523, 1223938, 1223528, 1223942 ], 
        [ 1223939, 1226908, 1223943, 1226912 ], 
        [ 1226909, 1227062, 1226913, 1227066 ], 
        [ 1227063, 1228205, 1227067, 1228210 ], 
        [ 1228206, 1229934, 1228211, 1229938 ], 
        [ 1229935, 1236067, 1229939, 1236072 ], 
        [ 1236068, 1236265, 1236073, 1236270 ], 
        [ 1236266, 1239904, 1236271, 1239908 ], 
        [ 1239905, 1239969, 1239909, 1239974 ], 
        [ 1239970, 1240641, 1239975, 1240646 ], 
        [ 1240642, 1244451, 1240647, 1244447 ], 
        [ 1244452, 1244738, 1244448, 1244743 ], 
        [ 1244739, 1244821, 1244744, 1244826 ], 
        [ 1244822, 1247339, 1244827, 1247343 ], 
        [ 1247340, 1248889, 1247344, 1248885 ], 
        [ 1248890, 1250671, 1248886, 1250675 ], 
        [ 1250672, 1254643, 1250676, 1254647 ], 
        [ 1254644, 1255112, 1254648, 1255108 ], 
        [ 1255113, 1257527, 1255109, 1257523 ], 
        [ 1257528, 1264487, 1257524, 1264491 ], 
        [ 1264488, 1269317, 1264492, 1269321 ], 
        [ 1269318, 1272971, 1269322, 1272976 ], 
        [ 1272972, 1276524, 1272977, 1276529 ], 
        [ 1276525, 1281881, 1276530, 1281877 ], 
        [ 1281882, 1281933, 1281878, 1281937 ], 
        [ 1281934, 1287297, 1281938, 1287301 ], 
        [ 1287298, 1287557, 1287302, 1287561 ], 
        [ 1287558, 1290344, 1287562, 1290349 ], 
        [ 1290345, 1292253, 1290350, 1292258 ], 
        [ 1292254, 1293482, 1292259, 1293487 ], 
        [ 1293483, 1295919, 1293488, 1295924 ], 
        [ 1295920, 1302576, 1295925, 1302580 ], 
        [ 1302577, 1302834, 1302581, 1302839 ], 
        [ 1302835, 1302920, 1302840, 1302924 ], 
        [ 1302921, 1303091, 1302925, 1303095 ], 
        [ 1303092, 1303464, 1303096, 1303469 ], 
        [ 1303465, 1306801, 1303470, 1306805 ], 
        [ 1306802, 1307555, 1306806, 1307559 ], 
        [ 1307556, 1309308, 1307560, 1309313 ], 
        [ 1309309, 1311482, 1309314, 1311487 ], 
        [ 1311483, 1312493, 1311488, 1312498 ], 
        [ 1312494, 1316488, 1312499, 1316493 ], 
        [ 1316489, 1318127, 1316494, 1318132 ], 
        [ 1318128, 1325643, 1318133, 1325648 ], 
        [ 1325644, 1328313, 1325649, 1328318 ], 
        [ 1328314, 1329159, 1328319, 1329155 ], 
        [ 1329160, 1332129, 1329156, 1332133 ], 
        [ 1332130, 1332245, 1332134, 1332249 ], 
        [ 1332246, 1332269, 1332250, 1332265 ], 
        [ 1332270, 1332837, 1332266, 1332841 ], 
        [ 1332838, 1334227, 1332842, 1334223 ], 
        [ 1334228, 1345348, 1334224, 1345353 ], 
        [ 1345349, 1346662, 1345354, 1346666 ], 
        [ 1346663, 1347480, 1346667, 1347485 ], 
        [ 1347481, 1348458, 1347486, 1348463 ], 
        [ 1348459, 1350595, 1348464, 1350600 ], 
        [ 1350596, 1350770, 1350601, 1350775 ], 
        [ 1350771, 1351954, 1350776, 1351959 ], 
        [ 1351955, 1356474, 1351960, 1356479 ], 
        [ 1356475, 1359703, 1356480, 1359699 ], 
        [ 1359704, 1361176, 1359700, 1361180 ], 
        [ 1361177, 1362756, 1361181, 1362761 ], 
        [ 1362757, 1368544, 1362762, 1368549 ], 
        [ 1368545, 1370442, 1368550, 1370438 ], 
        [ 1370443, 1373317, 1370439, 1373321 ], 
        [ 1373318, 1374484, 1373322, 1374488 ], 
        [ 1374485, 1377993, 1374489, 1377998 ], 
        [ 1377994, 1379610, 1377999, 1379615 ], 
        [ 1379611, 1380792, 1379616, 1380796 ], 
        [ 1380793, 1381209, 1380797, 1381205 ], 
        [ 1381210, 1386893, 1381206, 1386897 ], 
        [ 1386894, 1391551, 1386898, 1391556 ], 
        [ 1391552, 1393799, 1391557, 1393803 ], 
        [ 1393800, 1395841, 1393804, 1395846 ], 
        [ 1395842, 1397239, 1395847, 1397243 ], 
        [ 1397240, 1401721, 1397244, 1401726 ], 
        [ 1401722, 1403822, 1401727, 1403818 ], 
        [ 1403823, 1406871, 1403819, 1406876 ], 
        [ 1406872, 1407926, 1406877, 1407922 ], 
        [ 1407927, 1408482, 1407923, 1408486 ], 
        [ 1408483, 1409484, 1408487, 1409480 ], 
        [ 1409485, 1410252, 1409481, 1410256 ], 
        [ 1410253, 1411041, 1410257, 1411046 ], 
        [ 1411042, 1417851, 1411047, 1417856 ], 
        [ 1417852, 1419058, 1417857, 1419063 ], 
        [ 1419059, 1419370, 1419064, 1419366 ], 
        [ 1419371, 1419429, 1419367, 1419433 ], 
        [ 1419430, 1426518, 1419434, 1426522 ], 
        [ 1426519, 1428120, 1426523, 1428125 ], 
        [ 1428121, 1428584, 1428126, 1428589 ], 
        [ 1428585, 1430135, 1428590, 1430139 ], 
        [ 1430136, 1430700, 1430140, 1430705 ], 
        [ 1430701, 1436904, 1430706, 1436908 ], 
        [ 1436905, 1438278, 1436909, 1438283 ], 
        [ 1438279, 1441717, 1438284, 1441721 ], 
        [ 1441718, 1443084, 1441722, 1443089 ], 
        [ 1443085, 1444668, 1443090, 1444673 ], 
        [ 1444669, 1444866, 1444674, 1444871 ], 
        [ 1444867, 1444914, 1444872, 1444919 ], 
        [ 1444915, 1445093, 1444920, 1445098 ], 
        [ 1445094, 1446216, 1445099, 1446221 ], 
        [ 1446217, 1448333, 1446222, 1448329 ], 
        [ 1448334, 1448518, 1448330, 1448523 ], 
        [ 1448519, 1449362, 1448524, 1449358 ], 
        [ 1449363, 1449444, 1449359, 1449440 ], 
        [ 1449445, 1452860, 1449441, 1452865 ], 
        [ 1452861, 1454246, 1452866, 1454251 ], 
        [ 1454247, 1455021, 1454252, 1455017 ], 
        [ 1455022, 1455414, 1455018, 1455419 ], 
        [ 1455415, 1460976, 1455420, 1460981 ], 
        [ 1460977, 1461164, 1460982, 1461169 ], 
        [ 1461165, 1461294, 1461170, 1461298 ], 
        [ 1461295, 1463675, 1461299, 1463680 ], 
        [ 1463676, 1463710, 1463681, 1463714 ], 
        [ 1463711, 1465339, 1463715, 1465344 ], 
        [ 1465340, 1469872, 1465345, 1469877 ], 
        [ 1469873, 1471479, 1469878, 1471484 ], 
        [ 1471480, 1471922, 1471485, 1471926 ], 
        [ 1471923, 1472450, 1471927, 1472454 ], 
        [ 1472451, 1472745, 1472455, 1472750 ], 
        [ 1472746, 1479208, 1472751, 1479213 ], 
        [ 1479209, 1480831, 1479214, 1480836 ], 
        [ 1480832, 1483483, 1480837, 1483479 ], 
        [ 1483484, 1485359, 1483480, 1485364 ], 
        [ 1485360, 1485530, 1485365, 1485535 ], 
        [ 1485531, 1485675, 1485536, 1485679 ], 
        [ 1485676, 1486004, 1485680, 1486009 ], 
        [ 1486005, 1487314, 1486010, 1487319 ], 
        [ 1487315, 1491008, 1487320, 1491013 ], 
        [ 1491009, 1492068, 1491014, 1492073 ], 
        [ 1492069, 1492190, 1492074, 1492194 ], 
        [ 1492191, 1493001, 1492195, 1493006 ], 
        [ 1493002, 1495524, 1493007, 1495529 ], 
        [ 1495525, 1498599, 1495530, 1498604 ], 
        [ 1498600, 1499384, 1498605, 1499389 ], 
        [ 1499385, 1500494, 1499390, 1500499 ], 
        [ 1500495, 1504828, 1500500, 1504833 ], 
        [ 1504829, 1506224, 1504834, 1506228 ], 
        [ 1506225, 1506798, 1506229, 1506802 ], 
        [ 1506799, 1508452, 1506803, 1508456 ], 
        [ 1508453, 1509790, 1508457, 1509795 ], 
        [ 1509791, 1512050, 1509796, 1512055 ], 
        [ 1512051, 1514922, 1512056, 1514927 ], 
        [ 1514923, 1515140, 1514928, 1515145 ], 
        [ 1515141, 1515194, 1515146, 1515199 ], 
        [ 1515195, 1515647, 1515200, 1515652 ], 
        [ 1515648, 1516602, 1515653, 1516607 ], 
        [ 1516603, 1517689, 1516608, 1517694 ], 
        [ 1517690, 1519324, 1517695, 1519329 ], 
        [ 1519325, 1524288, 1519330, 1524293 ], 
        [ 1524289, 1524809, 1524294, 1524814 ], 
        [ 1524810, 1524838, 1524815, 1524842 ], 
        [ 1524839, 1525934, 1524843, 1525939 ], 
        [ 1525935, 1526325, 1525940, 1526330 ], 
        [ 1526326, 1527046, 1526331, 1527051 ], 
        [ 1527047, 1527437, 1527052, 1527441 ], 
        [ 1527438, 1528800, 1527442, 1528805 ], 
        [ 1528801, 1529067, 1528806, 1529072 ], 
        [ 1529068, 1529127, 1529073, 1529132 ], 
        [ 1529128, 1532112, 1529133, 1532108 ], 
        [ 1532113, 1533431, 1532109, 1533435 ], 
        [ 1533432, 1536262, 1533436, 1536267 ], 
        [ 1536263, 1543858, 1536268, 1543863 ], 
        [ 1543859, 1547883, 1543864, 1547887 ], 
        [ 1547884, 1550923, 1547888, 1550927 ], 
        [ 1550924, 1550941, 1550928, 1550945 ], 
        [ 1550942, 1551904, 1550946, 1551900 ], 
        [ 1551905, 1554015, 1551901, 1554020 ], 
        [ 1554016, 1554903, 1554021, 1554907 ], 
        [ 1554904, 1555315, 1554908, 1555320 ], 
        [ 1555316, 1558476, 1555321, 1558481 ], 
        [ 1558477, 1560403, 1558482, 1560408 ], 
        [ 1560404, 1564152, 1560409, 1564157 ], 
        [ 1564153, 1565868, 1564158, 1565873 ], 
        [ 1565869, 1566075, 1565874, 1566080 ], 
        [ 1566076, 1572396, 1566081, 1572392 ], 
        [ 1572397, 1572715, 1572393, 1572720 ], 
        [ 1572716, 1573897, 1572721, 1573901 ], 
        [ 1573898, 1575566, 1573902, 1575571 ], 
        [ 1575567, 1575840, 1575572, 1575845 ], 
        [ 1575841, 1575957, 1575846, 1575962 ], 
        [ 1575958, 1577744, 1575963, 1577748 ], 
        [ 1577745, 1578588, 1577749, 1578593 ], 
        [ 1578589, 1580138, 1578594, 1580134 ], 
        [ 1580139, 1582760, 1580135, 1582764 ], 
        [ 1582761, 1587557, 1582765, 1587562 ], 
        [ 1587558, 1588891, 1587563, 1588896 ], 
        [ 1588892, 1590824, 1588897, 1590828 ], 
        [ 1590825, 1591786, 1590829, 1591790 ], 
        [ 1591787, 1597227, 1591791, 1597232 ], 
        [ 1597228, 1597262, 1597233, 1597267 ], 
        [ 1597263, 1606974, 1597268, 1606979 ], 
        [ 1606975, 1608871, 1606980, 1608875 ], 
        [ 1608872, 1613363, 1608876, 1613367 ], 
        [ 1613364, 1613512, 1613368, 1613517 ], 
        [ 1613513, 1613900, 1613518, 1613905 ], 
        [ 1613901, 1614931, 1613906, 1614936 ], 
        [ 1614932, 1616478, 1614937, 1616482 ], 
        [ 1616479, 1620090, 1616483, 1620094 ], 
        [ 1620091, 1620971, 1620095, 1620976 ], 
        [ 1620972, 1625931, 1620977, 1625936 ], 
        [ 1625932, 1635578, 1625937, 1635583 ], 
        [ 1635579, 1636949, 1635584, 1636954 ], 
        [ 1636950, 1642076, 1636955, 1642081 ], 
        [ 1642077, 1643227, 1642082, 1643232 ], 
        [ 1643228, 1643451, 1643233, 1643456 ], 
        [ 1643452, 1643568, 1643457, 1643573 ], 
        [ 1643569, 1651406, 1643574, 1651411 ], 
        [ 1651407, 1651474, 1651412, 1651479 ], 
        [ 1651475, 1656665, 1651480, 1656669 ], 
        [ 1656666, 1660688, 1656670, 1660693 ], 
        [ 1660689, 1662867, 1660694, 1662871 ], 
        [ 1662868, 1665846, 1662872, 1665851 ], 
        [ 1665847, 1667026, 1665852, 1667031 ], 
        [ 1667027, 1669021, 1667032, 1669025 ], 
        [ 1669022, 1669975, 1669026, 1669979 ], 
        [ 1669976, 1675465, 1669980, 1675470 ], 
        [ 1675466, 1679164, 1675471, 1679169 ], 
        [ 1679165, 1681962, 1679170, 1681967 ], 
        [ 1681963, 1688016, 1681968, 1688021 ], 
        [ 1688017, 1690659, 1688022, 1690664 ], 
        [ 1690660, 1692872, 1690665, 1692877 ], 
        [ 1692873, 1697102, 1692878, 1697107 ], 
        [ 1697103, 1698132, 1697108, 1698137 ], 
        [ 1698133, 1698208, 1698138, 1698212 ], 
        [ 1698209, 1703429, 1698213, 1703434 ], 
        [ 1703430, 1705101, 1703435, 1705105 ], 
        [ 1705102, 1705881, 1705106, 1705885 ], 
        [ 1705882, 1706057, 1705886, 1706062 ], 
        [ 1706058, 1708138, 1706063, 1708142 ], 
        [ 1708139, 1708683, 1708143, 1708688 ], 
        [ 1708684, 1712200, 1708689, 1712204 ], 
        [ 1712201, 1720884, 1712205, 1720889 ], 
        [ 1720885, 1721218, 1720890, 1721223 ], 
        [ 1721219, 1725289, 1721224, 1725294 ], 
        [ 1725290, 1726034, 1725295, 1726038 ], 
        [ 1726035, 1726495, 1726039, 1726500 ], 
        [ 1726496, 1728646, 1726501, 1728651 ], 
        [ 1728647, 1729060, 1728652, 1729065 ], 
        [ 1729061, 1732801, 1729066, 1732806 ], 
        [ 1732802, 1733308, 1732807, 1733313 ], 
        [ 1733309, 1734471, 1733314, 1734476 ], 
        [ 1734472, 1738054, 1734477, 1738058 ], 
        [ 1738055, 1738256, 1738059, 1738260 ], 
        [ 1738257, 1740942, 1738261, 1740947 ], 
        [ 1740943, 1744762, 1740948, 1744767 ], 
        [ 1744763, 1746379, 1744768, 1746384 ], 
        [ 1746380, 1746672, 1746385, 1746668 ], 
        [ 1746673, 1747144, 1746669, 1747149 ], 
        [ 1747145, 1751310, 1747150, 1751306 ], 
        [ 1751311, 1753062, 1751307, 1753067 ], 
        [ 1753063, 1754367, 1753068, 1754372 ], 
        [ 1754368, 1758860, 1754373, 1758864 ], 
        [ 1758861, 1763444, 1758865, 1763449 ], 
        [ 1763445, 1769087, 1763450, 1769091 ], 
        [ 1769088, 1769105, 1769092, 1769109 ], 
        [ 1769106, 1769947, 1769110, 1769951 ], 
        [ 1769948, 1772285, 1769952, 1772289 ], 
        [ 1772286, 1774404, 1772290, 1774408 ], 
        [ 1774405, 1776003, 1774409, 1776007 ], 
        [ 1776004, 1777420, 1776008, 1777425 ], 
        [ 1777421, 1782626, 1777426, 1782631 ], 
        [ 1782627, 1784342, 1782632, 1784347 ], 
        [ 1784343, 1784549, 1784348, 1784554 ], 
        [ 1784550, 1785362, 1784555, 1785366 ], 
        [ 1785363, 1791189, 1785367, 1791194 ], 
        [ 1791190, 1792209, 1791195, 1792213 ], 
        [ 1792210, 1793878, 1792214, 1793883 ], 
        [ 1793879, 1794152, 1793884, 1794157 ], 
        [ 1794153, 1794269, 1794158, 1794274 ], 
        [ 1794270, 1794972, 1794275, 1794977 ], 
        [ 1794973, 1796163, 1794978, 1796168 ], 
        [ 1796164, 1800687, 1796169, 1800683 ], 
        [ 1800688, 1802296, 1800684, 1802301 ], 
        [ 1802297, 1804730, 1802302, 1804734 ], 
        [ 1804731, 1805422, 1804735, 1805426 ], 
        [ 1805423, 1805729, 1805427, 1805734 ], 
        [ 1805730, 1806305, 1805735, 1806310 ], 
        [ 1806306, 1806755, 1806311, 1806759 ], 
        [ 1806756, 1806879, 1806760, 1806875 ], 
        [ 1806880, 1810512, 1806876, 1810517 ], 
        [ 1810513, 1816402, 1810518, 1816407 ], 
        [ 1816403, 1826227, 1816408, 1826232 ], 
        [ 1826228, 1826701, 1826233, 1826706 ], 
        [ 1826702, 1827720, 1826707, 1827725 ], 
        [ 1827721, 1833845, 1827726, 1833849 ], 
        [ 1833846, 1836707, 1833850, 1836712 ], 
        [ 1836708, 1836926, 1836713, 1836931 ], 
        [ 1836927, 1838667, 1836932, 1838672 ], 
        [ 1838668, 1838935, 1838673, 1838939 ], 
        [ 1838936, 1842071, 1838940, 1842075 ], 
        [ 1842072, 1842664, 1842076, 1842668 ], 
        [ 1842665, 1843220, 1842669, 1843225 ], 
        [ 1843221, 1843829, 1843226, 1843834 ], 
        [ 1843830, 1845044, 1843835, 1845048 ], 
        [ 1845045, 1846577, 1845049, 1846582 ], 
        [ 1846578, 1848717, 1846583, 1848721 ], 
        [ 1848718, 1849125, 1848722, 1849130 ], 
        [ 1849126, 1850237, 1849131, 1850242 ], 
        [ 1850238, 1851708, 1850243, 1851713 ], 
        [ 1851709, 1853436, 1851714, 1853441 ], 
        [ 1853437, 1853475, 1853442, 1853480 ], 
        [ 1853476, 1853493, 1853481, 1853498 ], 
        [ 1853494, 1854900, 1853499, 1854905 ], 
        [ 1854901, 1854987, 1854906, 1854991 ], 
        [ 1854988, 1861797, 1854992, 1861802 ], 
        [ 1861798, 1862267, 1861803, 1862272 ], 
        [ 1862268, 1866445, 1862273, 1866450 ], 
        [ 1866446, 1866700, 1866451, 1866705 ], 
        [ 1866701, 1870035, 1866706, 1870039 ], 
        [ 1870036, 1870143, 1870040, 1870148 ], 
        [ 1870144, 1870675, 1870149, 1870680 ], 
        [ 1870676, 1871498, 1870681, 1871502 ], 
        [ 1871499, 1873369, 1871503, 1873373 ], 
        [ 1873370, 1877824, 1873374, 1877828 ], 
        [ 1877825, 1880920, 1877829, 1880916 ], 
        [ 1880921, 1881704, 1880917, 1881709 ], 
        [ 1881705, 1882659, 1881710, 1882664 ], 
        [ 1882660, 1884008, 1882665, 1884013 ], 
        [ 1884009, 1885076, 1884014, 1885081 ], 
        [ 1885077, 1888274, 1885082, 1888270 ], 
        [ 1888275, 1897857, 1888271, 1897862 ], 
        [ 1897858, 1907112, 1897863, 1907116 ], 
        [ 1907113, 1907770, 1907117, 1907774 ], 
        [ 1907771, 1912119, 1907775, 1912123 ], 
        [ 1912120, 1913812, 1912124, 1913816 ], 
        [ 1913813, 1913938, 1913817, 1913942 ], 
        [ 1913939, 1922086, 1913943, 1922090 ], 
        [ 1922087, 1924874, 1922091, 1924878 ], 
        [ 1924875, 1929326, 1924879, 1929322 ], 
        [ 1929327, 1931549, 1929323, 1931554 ], 
        [ 1931550, 1931660, 1931555, 1931665 ], 
        [ 1931661, 1936680, 1931666, 1936685 ], 
        [ 1936681, 1938633, 1936686, 1938637 ], 
        [ 1938634, 1938835, 1938638, 1938840 ], 
        [ 1938836, 1939367, 1938841, 1939372 ], 
        [ 1939368, 1941696, 1939373, 1941700 ], 
        [ 1941697, 1943301, 1941701, 1943305 ], 
        [ 1943302, 1944718, 1943306, 1944723 ], 
        [ 1944719, 1949924, 1944724, 1949929 ], 
        [ 1949925, 1951640, 1949930, 1951645 ], 
        [ 1951641, 1951847, 1951646, 1951852 ], 
        [ 1951848, 1952660, 1951853, 1952664 ], 
        [ 1952661, 1953731, 1952665, 1953735 ], 
        [ 1953732, 1958495, 1953736, 1958500 ], 
        [ 1958496, 1959515, 1958501, 1959519 ], 
        [ 1959516, 1961184, 1959520, 1961189 ], 
        [ 1961185, 1961458, 1961190, 1961463 ], 
        [ 1961459, 1963223, 1961464, 1963228 ], 
        [ 1963224, 1964535, 1963229, 1964540 ], 
        [ 1964536, 1964578, 1964541, 1964583 ], 
        [ 1964579, 1965726, 1964584, 1965731 ], 
        [ 1965727, 1967936, 1965732, 1967940 ], 
        [ 1967937, 1970524, 1967941, 1970528 ], 
        [ 1970525, 1971307, 1970529, 1971303 ], 
        [ 1971308, 1971952, 1971304, 1971956 ], 
        [ 1971953, 1971961, 1971957, 1971965 ], 
        [ 1971962, 1974876, 1971966, 1974880 ], 
        [ 1974877, 1975723, 1974881, 1975728 ], 
        [ 1975724, 1977141, 1975729, 1977137 ], 
        [ 1977142, 1983495, 1977138, 1983500 ], 
        [ 1983496, 1985266, 1983501, 1985270 ], 
        [ 1985267, 1989041, 1985271, 1989046 ], 
        [ 1989042, 1991504, 1989047, 1991508 ], 
        [ 1991505, 1991939, 1991509, 1991944 ], 
        [ 1991940, 1994134, 1991945, 1994139 ], 
        [ 1994135, 1997719, 1994140, 1997723 ], 
        [ 1997720, 1997804, 1997724, 1997800 ], 
        [ 1997805, 2002856, 1997801, 2002860 ], 
        [ 2002857, 2006390, 2002861, 2006395 ], 
        [ 2006391, 2006681, 2006396, 2006686 ], 
        [ 2006682, 2012753, 2006687, 2012758 ], 
        [ 2012754, 2020299, 2012759, 2020304 ], 
        [ 2020300, 2021594, 2020305, 2021599 ], 
        [ 2021595, 2023023, 2021600, 2023027 ], 
        [ 2023024, 2026978, 2023028, 2026974 ], 
        [ 2026979, 2035653, 2026975, 2035658 ], 
        [ 2035654, 2035969, 2035659, 2035973 ], 
        [ 2035970, 2038524, 2035974, 2038528 ], 
        [ 2038525, 2040783, 2038529, 2040787 ], 
        [ 2040784, 2042430, 2040788, 2042434 ], 
        [ 2042431, 2043961, 2042435, 2043966 ], 
        [ 2043962, 2044411, 2043967, 2044416 ], 
        [ 2044412, 2045320, 2044417, 2045325 ], 
        [ 2045321, 2045652, 2045326, 2045648 ], 
        [ 2045653, 2046593, 2045649, 2046598 ], 
        [ 2046594, 2058014, 2046599, 2058019 ], 
        [ 2058015, 2058163, 2058020, 2058167 ], 
        [ 2058164, 2058262, 2058168, 2058267 ], 
        [ 2058263, 2059250, 2058268, 2059246 ], 
        [ 2059251, 2061616, 2059247, 2061621 ], 
        [ 2061617, 2067334, 2061622, 2067339 ], 
        [ 2067335, 2069059, 2067340, 2069064 ], 
        [ 2069060, 2073142, 2069065, 2073147 ], 
        [ 2073143, 2074555, 2073148, 2074560 ], 
        [ 2074556, 2074634, 2074561, 2074639 ], 
        [ 2074635, 2076234, 2074640, 2076238 ], 
        [ 2076235, 2076422, 2076239, 2076427 ], 
        [ 2076423, 2080648, 2076428, 2080652 ], 
        [ 2080649, 2081937, 2080653, 2081942 ], 
        [ 2081938, 2082042, 2081943, 2082047 ], 
        [ 2082043, 2082408, 2082048, 2082413 ], 
        [ 2082409, 2086395, 2082414, 2086399 ], 
        [ 2086396, 2088062, 2086400, 2088066 ], 
        [ 2088063, 2088207, 2088067, 2088211 ], 
        [ 2088208, 2092297, 2088212, 2092301 ], 
        [ 2092298, 2093453, 2092302, 2093449 ], 
        [ 2093454, 2094661, 2093450, 2094666 ], 
        [ 2094662, 2098386, 2094667, 2098390 ], 
        [ 2098387, 2105404, 2098391, 2105408 ], 
        [ 2105405, 2105556, 2105409, 2105561 ], 
        [ 2105557, 2106153, 2105562, 2106158 ], 
        [ 2106154, 2107284, 2106159, 2107288 ], 
        [ 2107285, 2110444, 2107289, 2110448 ], 
        [ 2110445, 2110523, 2110449, 2110527 ], 
        [ 2110524, 2113282, 2110528, 2113287 ], 
        [ 2113283, 2114197, 2113288, 2114202 ], 
        [ 2114198, 2119617, 2114203, 2119621 ], 
        [ 2119618, 2124245, 2119622, 2124250 ], 
        [ 2124246, 2126629, 2124251, 2126634 ], 
        [ 2126630, 2127367, 2126635, 2127372 ], 
        [ 2127368, 2131057, 2127373, 2131061 ], 
        [ 2131058, 2131854, 2131062, 2131859 ], 
        [ 2131855, 2134278, 2131860, 2134282 ], 
        [ 2134279, 2134456, 2134283, 2134460 ], 
        [ 2134457, 2137761, 2134461, 2137765 ], 
        [ 2137762, 2138481, 2137766, 2138486 ], 
        [ 2138482, 2139541, 2138487, 2139545 ], 
        [ 2139542, 2140084, 2139546, 2140089 ], 
        [ 2140085, 2151397, 2140090, 2151402 ], 
        [ 2151398, 2154116, 2151403, 2154121 ], 
        [ 2154117, 2158877, 2154122, 2158881 ], 
        [ 2158878, 2158886, 2158882, 2158890 ], 
        [ 2158887, 2160314, 2158891, 2160318 ], 
        [ 2160315, 2164531, 2160319, 2164536 ], 
        [ 2164532, 2164999, 2164537, 2165004 ], 
        [ 2165000, 2166190, 2165005, 2166195 ], 
        [ 2166191, 2168535, 2166196, 2168540 ], 
        [ 2168536, 2168652, 2168541, 2168657 ], 
        [ 2168653, 2168876, 2168658, 2168881 ], 
        [ 2168877, 2169179, 2168882, 2169175 ], 
        [ 2169180, 2170247, 2169176, 2170243 ], 
        [ 2170248, 2175197, 2170244, 2175202 ], 
        [ 2175198, 2176568, 2175203, 2176573 ], 
        [ 2176569, 2185419, 2176574, 2185424 ], 
        [ 2185420, 2187124, 2185425, 2187128 ], 
        [ 2187125, 2188632, 2187129, 2188636 ], 
        [ 2188633, 2194633, 2188637, 2194637 ], 
        [ 2194634, 2196521, 2194638, 2196525 ], 
        [ 2196522, 2197400, 2196526, 2197396 ], 
        [ 2197401, 2198074, 2197397, 2198079 ], 
        [ 2198075, 2200597, 2198080, 2200601 ], 
        [ 2200598, 2202376, 2200602, 2202380 ], 
        [ 2202377, 2203542, 2202381, 2203546 ], 
        [ 2203543, 2205716, 2203547, 2205721 ], 
        [ 2205717, 2206482, 2205722, 2206487 ], 
        [ 2206483, 2209774, 2206488, 2209778 ], 
        [ 2209775, 2214819, 2209779, 2214824 ], 
        [ 2214820, 2215255, 2214825, 2215260 ], 
        [ 2215256, 2216910, 2215261, 2216915 ], 
        [ 2216911, 2219477, 2216916, 2219482 ], 
        [ 2219478, 2219751, 2219483, 2219756 ], 
        [ 2219752, 2221421, 2219757, 2221425 ], 
        [ 2221422, 2222602, 2221426, 2222607 ], 
        [ 2222603, 2222929, 2222608, 2222925 ], 
        [ 2222930, 2224016, 2222926, 2224021 ], 
        [ 2224017, 2228441, 2224022, 2228445 ], 
        [ 2228442, 2229253, 2228446, 2229258 ], 
        [ 2229254, 2229460, 2229259, 2229465 ], 
        [ 2229461, 2231176, 2229466, 2231181 ], 
        [ 2231177, 2236382, 2231182, 2236387 ], 
        [ 2236383, 2238087, 2236388, 2238091 ], 
        [ 2238088, 2239596, 2238092, 2239600 ], 
        [ 2239597, 2242924, 2239601, 2242920 ], 
        [ 2242925, 2243879, 2242921, 2243883 ], 
        [ 2243880, 2243897, 2243884, 2243901 ], 
        [ 2243898, 2245581, 2243902, 2245586 ], 
        [ 2245582, 2245719, 2245587, 2245724 ], 
        [ 2245720, 2245761, 2245725, 2245766 ], 
        [ 2245762, 2249902, 2245767, 2249907 ], 
        [ 2249903, 2254722, 2249908, 2254727 ], 
        [ 2254723, 2254803, 2254728, 2254807 ], 
        [ 2254804, 2257149, 2254808, 2257153 ], 
        [ 2257150, 2262668, 2257154, 2262673 ], 
        [ 2262669, 2267507, 2262674, 2267503 ], 
        [ 2267508, 2270370, 2267504, 2270366 ], 
        [ 2270371, 2276333, 2270367, 2276338 ], 
        [ 2276334, 2277071, 2276339, 2277075 ], 
        [ 2277072, 2278349, 2277076, 2278354 ], 
        [ 2278350, 2278595, 2278355, 2278600 ], 
        [ 2278596, 2281303, 2278601, 2281307 ], 
        [ 2281304, 2282039, 2281308, 2282044 ], 
        [ 2282040, 2286122, 2282045, 2286126 ], 
        [ 2286123, 2295986, 2286127, 2295990 ], 
        [ 2295987, 2296911, 2295991, 2296907 ], 
        [ 2296912, 2300006, 2296908, 2300010 ], 
        [ 2300007, 2309292, 2300011, 2309297 ], 
        [ 2309293, 2309737, 2309298, 2309742 ], 
        [ 2309738, 2312320, 2309743, 2312324 ], 
        [ 2312321, 2314845, 2312325, 2314850 ], 
        [ 2314846, 2315016, 2314851, 2315021 ], 
        [ 2315017, 2320047, 2315022, 2320052 ], 
        [ 2320048, 2320645, 2320053, 2320650 ], 
        [ 2320646, 2326925, 2320651, 2326921 ], 
        [ 2326926, 2330437, 2326922, 2330442 ], 
        [ 2330438, 2335656, 2330443, 2335660 ], 
        [ 2335657, 2338082, 2335661, 2338087 ], 
        [ 2338083, 2343729, 2338088, 2343725 ], 
        [ 2343730, 2345465, 2343726, 2345470 ], 
        [ 2345466, 2345517, 2345471, 2345521 ], 
        [ 2345518, 2347136, 2345522, 2347140 ], 
        [ 2347137, 2347233, 2347141, 2347238 ], 
        [ 2347234, 2348720, 2347239, 2348725 ], 
        [ 2348721, 2351324, 2348726, 2351329 ], 
        [ 2351325, 2352448, 2351330, 2352453 ], 
        [ 2352449, 2353999, 2352454, 2354004 ], 
        [ 2354000, 2354134, 2354005, 2354138 ], 
        [ 2354135, 2359046, 2354139, 2359051 ], 
        [ 2359047, 2361149, 2359052, 2361154 ], 
        [ 2361150, 2374039, 2361155, 2374044 ], 
        [ 2374040, 2382502, 2374045, 2382506 ], 
        [ 2382503, 2385349, 2382507, 2385354 ], 
        [ 2385350, 2388585, 2385355, 2388590 ], 
        [ 2388586, 2391734, 2388591, 2391739 ], 
        [ 2391735, 2392141, 2391740, 2392146 ], 
        [ 2392142, 2393939, 2392147, 2393944 ], 
        [ 2393940, 2395026, 2393945, 2395031 ], 
        [ 2395027, 2395860, 2395032, 2395865 ], 
        [ 2395861, 2398211, 2395866, 2398216 ], 
        [ 2398212, 2398326, 2398217, 2398331 ], 
        [ 2398327, 2400696, 2398332, 2400700 ], 
        [ 2400697, 2402303, 2400701, 2402308 ], 
        [ 2402304, 2405335, 2402309, 2405339 ], 
        [ 2405336, 2408154, 2405340, 2408159 ], 
        [ 2408155, 2409936, 2408160, 2409941 ], 
        [ 2409937, 2410353, 2409942, 2410358 ], 
        [ 2410354, 2411021, 2410359, 2411026 ], 
        [ 2411022, 2414614, 2411027, 2414618 ], 
        [ 2414615, 2419571, 2414619, 2419576 ], 
        [ 2419572, 2421744, 2419577, 2421748 ], 
        [ 2421745, 2424488, 2421749, 2424493 ], 
        [ 2424489, 2427895, 2424494, 2427900 ], 
        [ 2427896, 2430604, 2427901, 2430600 ], 
        [ 2430605, 2433794, 2430601, 2433799 ], 
        [ 2433795, 2434280, 2433800, 2434285 ], 
        [ 2434281, 2435465, 2434286, 2435461 ], 
        [ 2435466, 2436129, 2435462, 2436134 ], 
        [ 2436130, 2446339, 2436135, 2446344 ], 
        [ 2446340, 2446355, 2446345, 2446360 ], 
        [ 2446356, 2447550, 2446361, 2447555 ], 
        [ 2447551, 2447589, 2447556, 2447593 ], 
        [ 2447590, 2451279, 2447594, 2451283 ], 
        [ 2451280, 2456375, 2451284, 2456380 ], 
        [ 2456376, 2458676, 2456381, 2458672 ], 
        [ 2458677, 2459075, 2458673, 2459079 ], 
        [ 2459076, 2459685, 2459080, 2459690 ], 
        [ 2459686, 2467707, 2459691, 2467712 ], 
        [ 2467708, 2474920, 2467713, 2474924 ], 
        [ 2474921, 2483809, 2474925, 2483813 ], 
        [ 2483810, 2487345, 2483814, 2487349 ], 
        [ 2487346, 2489626, 2487350, 2489631 ], 
        [ 2489627, 2490030, 2489632, 2490035 ], 
        [ 2490031, 2493086, 2490036, 2493090 ], 
        [ 2493087, 2494181, 2493091, 2494186 ], 
        [ 2494182, 2494578, 2494187, 2494583 ], 
        [ 2494579, 2498330, 2494584, 2498335 ], 
        [ 2498331, 2501619, 2498336, 2501624 ], 
        [ 2501620, 2502148, 2501625, 2502152 ], 
        [ 2502149, 2502774, 2502153, 2502779 ], 
        [ 2502775, 2503405, 2502780, 2503409 ], 
        [ 2503406, 2505440, 2503410, 2505445 ], 
        [ 2505441, 2507840, 2505446, 2507845 ], 
        [ 2507841, 2513737, 2507846, 2513741 ], 
        [ 2513738, 2513953, 2513742, 2513958 ], 
        [ 2513954, 2516708, 2513959, 2516712 ], 
        [ 2516709, 2518482, 2516713, 2518487 ], 
        [ 2518483, 2518510, 2518488, 2518515 ], 
        [ 2518511, 2519154, 2518516, 2519159 ], 
        [ 2519155, 2521663, 2519160, 2521668 ], 
        [ 2521664, 2522690, 2521669, 2522695 ], 
        [ 2522691, 2533188, 2522696, 2533184 ], 
        [ 2533189, 2535156, 2533185, 2535161 ], 
        [ 2535157, 2536302, 2535162, 2536307 ], 
        [ 2536303, 2536525, 2536308, 2536521 ], 
        [ 2536526, 2539683, 2536522, 2539688 ], 
        [ 2539684, 2540838, 2539689, 2540843 ], 
        [ 2540839, 2542188, 2540844, 2542192 ], 
        [ 2542189, 2542542, 2542193, 2542547 ], 
        [ 2542543, 2543529, 2542548, 2543533 ], 
        [ 2543530, 2543833, 2543534, 2543829 ], 
        [ 2543834, 2549711, 2543830, 2549716 ], 
        [ 2549712, 2549979, 2549717, 2549984 ], 
        [ 2549980, 2550376, 2549985, 2550381 ], 
        [ 2550377, 2550442, 2550382, 2550447 ], 
        [ 2550443, 2552498, 2550448, 2552503 ], 
        [ 2552499, 2556237, 2552504, 2556242 ], 
        [ 2556238, 2557871, 2556243, 2557875 ], 
        [ 2557872, 2561281, 2557876, 2561286 ], 
        [ 2561282, 2562381, 2561287, 2562386 ], 
        [ 2562382, 2571576, 2562387, 2571581 ], 
        [ 2571577, 2573918, 2571582, 2573923 ], 
        [ 2573919, 2575854, 2573924, 2575859 ], 
        [ 2575855, 2575961, 2575860, 2575965 ], 
        [ 2575962, 2576170, 2575966, 2576166 ], 
        [ 2576171, 2579045, 2576167, 2579050 ], 
        [ 2579046, 2587793, 2579051, 2587797 ], 
        [ 2587794, 2588693, 2587798, 2588689 ], 
        [ 2588694, 2588728, 2588690, 2588733 ], 
        [ 2588729, 2589643, 2588734, 2589647 ], 
        [ 2589644, 2591930, 2589648, 2591935 ], 
        [ 2591931, 2592492, 2591936, 2592496 ], 
        [ 2592493, 2595383, 2592497, 2595387 ], 
        [ 2595384, 2601304, 2595388, 2601309 ], 
        [ 2601305, 2610396, 2601310, 2610400 ], 
        [ 2610397, 2613914, 2610401, 2613918 ], 
        [ 2613915, 2614812, 2613919, 2614817 ], 
        [ 2614813, 2614839, 2614818, 2614844 ], 
        [ 2614840, 2622328, 2614845, 2622333 ], 
        [ 2622329, 2624051, 2622334, 2624055 ], 
        [ 2624052, 2627903, 2624056, 2627908 ], 
        [ 2627904, 2633758, 2627909, 2633762 ], 
        [ 2633759, 2640020, 2633763, 2640024 ], 
        [ 2640021, 2648431, 2640025, 2648436 ], 
        [ 2648432, 2651846, 2648437, 2651851 ], 
        [ 2651847, 2658412, 2651852, 2658416 ], 
        [ 2658413, 2660586, 2658417, 2660591 ], 
        [ 2660587, 2660643, 2660592, 2660647 ], 
        [ 2660644, 2662994, 2660648, 2662998 ], 
        [ 2662995, 2663434, 2662999, 2663439 ], 
        [ 2663435, 2664288, 2663440, 2664292 ], 
        [ 2664289, 2666257, 2664293, 2666261 ], 
        [ 2666258, 2668305, 2666262, 2668301 ], 
        [ 2668306, 2668367, 2668302, 2668363 ], 
        [ 2668368, 2669373, 2668364, 2669377 ], 
        [ 2669374, 2674481, 2669378, 2674486 ], 
        [ 2674482, 2674949, 2674487, 2674954 ], 
        [ 2674950, 2676096, 2674955, 2676101 ], 
        [ 2676097, 2676139, 2676102, 2676144 ], 
        [ 2676140, 2678485, 2676145, 2678490 ], 
        [ 2678486, 2678602, 2678491, 2678607 ], 
        [ 2678603, 2678826, 2678608, 2678831 ], 
        [ 2678827, 2679129, 2678832, 2679125 ], 
        [ 2679130, 2680196, 2679126, 2680192 ], 
        [ 2680197, 2681278, 2680193, 2681283 ], 
        [ 2681279, 2683258, 2681284, 2683262 ], 
        [ 2683259, 2684926, 2683263, 2684931 ], 
        [ 2684927, 2685205, 2684932, 2685210 ], 
        [ 2685206, 2691894, 2685211, 2691898 ], 
        [ 2691895, 2692493, 2691899, 2692498 ], 
        [ 2692494, 2695364, 2692499, 2695368 ], 
        [ 2695365, 2696872, 2695369, 2696876 ], 
        [ 2696873, 2701148, 2696877, 2701152 ], 
        [ 2701149, 2701165, 2701153, 2701169 ], 
        [ 2701166, 2703577, 2701170, 2703581 ], 
        [ 2703578, 2709273, 2703582, 2709269 ], 
        [ 2709274, 2710251, 2709270, 2710255 ], 
        [ 2710252, 2714956, 2710256, 2714961 ], 
        [ 2714957, 2716135, 2714962, 2716140 ], 
        [ 2716136, 2716265, 2716141, 2716261 ], 
        [ 2716266, 2716930, 2716262, 2716935 ], 
        [ 2716931, 2717350, 2716936, 2717355 ], 
        [ 2717351, 2717774, 2717356, 2717779 ], 
        [ 2717775, 2718590, 2717780, 2718595 ], 
        [ 2718591, 2720379, 2718596, 2720384 ], 
        [ 2720380, 2721786, 2720385, 2721790 ], 
        [ 2721787, 2723960, 2721791, 2723965 ], 
        [ 2723961, 2725032, 2723966, 2725036 ], 
        [ 2725033, 2725167, 2725037, 2725171 ], 
        [ 2725168, 2725658, 2725172, 2725663 ], 
        [ 2725659, 2731050, 2725664, 2731054 ], 
        [ 2731051, 2731980, 2731055, 2731985 ], 
        [ 2731981, 2738172, 2731986, 2738177 ], 
        [ 2738173, 2738501, 2738178, 2738505 ], 
        [ 2738502, 2741368, 2738506, 2741372 ], 
        [ 2741369, 2741523, 2741373, 2741527 ], 
        [ 2741524, 2743821, 2741528, 2743817 ], 
        [ 2743822, 2743903, 2743818, 2743899 ], 
        [ 2743904, 2746696, 2743900, 2746700 ], 
        [ 2746697, 2748433, 2746701, 2748438 ], 
        [ 2748434, 2748596, 2748439, 2748601 ], 
        [ 2748597, 2749549, 2748602, 2749553 ], 
        [ 2749550, 2749729, 2749554, 2749734 ], 
        [ 2749730, 2750963, 2749735, 2750967 ], 
        [ 2750964, 2751613, 2750968, 2751609 ], 
        [ 2751614, 2752944, 2751610, 2752949 ], 
        [ 2752945, 2753953, 2752950, 2753958 ], 
        [ 2753954, 2754044, 2753959, 2754040 ], 
        [ 2754045, 2755376, 2754041, 2755380 ], 
        [ 2755377, 2760599, 2755381, 2760604 ], 
        [ 2760600, 2761236, 2760605, 2761241 ], 
        [ 2761237, 2763346, 2761242, 2763351 ], 
        [ 2763347, 2764218, 2763352, 2764223 ], 
        [ 2764219, 2767629, 2764224, 2767633 ], 
        [ 2767630, 2771002, 2767634, 2771006 ], 
        [ 2771003, 2773348, 2771007, 2773353 ], 
        [ 2773349, 2778556, 2773354, 2778560 ], 
        [ 2778557, 2779108, 2778561, 2779112 ], 
        [ 2779109, 2779707, 2779113, 2779712 ], 
        [ 2779708, 2780639, 2779713, 2780643 ], 
        [ 2780640, 2781795, 2780644, 2781791 ], 
        [ 2781796, 2782076, 2781792, 2782080 ], 
        [ 2782077, 2788770, 2782081, 2788774 ], 
        [ 2788771, 2792111, 2788775, 2792116 ], 
        [ 2792112, 2794418, 2792117, 2794423 ], 
        [ 2794419, 2795818, 2794424, 2795823 ], 
        [ 2795819, 2796261, 2795824, 2796266 ], 
        [ 2796262, 2798929, 2796267, 2798934 ], 
        [ 2798930, 2799454, 2798935, 2799459 ], 
        [ 2799455, 2811616, 2799460, 2811620 ], 
        [ 2811617, 2813416, 2811621, 2813421 ], 
        [ 2813417, 2813479, 2813422, 2813484 ], 
        [ 2813480, 2814107, 2813485, 2814112 ], 
        [ 2814108, 2818929, 2814113, 2818933 ], 
        [ 2818930, 2825357, 2818934, 2825362 ], 
        [ 2825358, 2826820, 2825363, 2826825 ], 
        [ 2826821, 2828096, 2826826, 2828101 ], 
        [ 2828097, 2828932, 2828102, 2828936 ], 
        [ 2828933, 2830088, 2828937, 2830093 ], 
        [ 2830089, 2830687, 2830094, 2830683 ], 
        [ 2830688, 2834601, 2830684, 2834606 ], 
        [ 2834602, 2836621, 2834607, 2836626 ], 
        [ 2836622, 2836884, 2836627, 2836889 ], 
        [ 2836885, 2837913, 2836890, 2837918 ], 
        [ 2837914, 2840393, 2837919, 2840398 ], 
        [ 2840394, 2841225, 2840399, 2841229 ], 
        [ 2841226, 2843391, 2841230, 2843396 ], 
        [ 2843392, 2843629, 2843397, 2843634 ], 
        [ 2843630, 2844665, 2843635, 2844670 ], 
        [ 2844666, 2845263, 2844671, 2845267 ], 
        [ 2845264, 2847821, 2845268, 2847826 ], 
        [ 2847822, 2848450, 2847827, 2848454 ], 
        [ 2848451, 2849601, 2848455, 2849606 ], 
        [ 2849602, 2853189, 2849607, 2853194 ], 
        [ 2853190, 2853798, 2853195, 2853802 ], 
        [ 2853799, 2860428, 2853803, 2860433 ], 
        [ 2860429, 2862152, 2860434, 2862157 ], 
        [ 2862153, 2862729, 2862158, 2862734 ], 
        [ 2862730, 2869033, 2862735, 2869038 ], 
        [ 2869034, 2869157, 2869039, 2869162 ], 
        [ 2869158, 2872699, 2869163, 2872703 ], 
        [ 2872700, 2882082, 2872704, 2882087 ], 
        [ 2882083, 2888775, 2882088, 2888779 ], 
        [ 2888776, 2894091, 2888780, 2894096 ], 
        [ 2894092, 2895090, 2894097, 2895095 ], 
        [ 2895091, 2895718, 2895096, 2895722 ], 
        [ 2895719, 2896669, 2895723, 2896665 ], 
        [ 2896670, 2900119, 2896666, 2900124 ], 
        [ 2900120, 2900555, 2900125, 2900560 ], 
        [ 2900556, 2902167, 2900561, 2902172 ], 
        [ 2902168, 2902210, 2902173, 2902215 ], 
        [ 2902211, 2904556, 2902216, 2904561 ], 
        [ 2904557, 2904673, 2904562, 2904678 ], 
        [ 2904674, 2904897, 2904679, 2904902 ], 
        [ 2904898, 2905200, 2904903, 2905196 ], 
        [ 2905201, 2906268, 2905197, 2906264 ], 
        [ 2906269, 2907797, 2906265, 2907802 ], 
        [ 2907798, 2910456, 2907803, 2910461 ], 
        [ 2910457, 2911608, 2910462, 2911613 ], 
        [ 2911609, 2914587, 2911614, 2914591 ], 
        [ 2914588, 2914744, 2914592, 2914749 ], 
        [ 2914745, 2914779, 2914750, 2914784 ], 
        [ 2914780, 2918124, 2914785, 2918129 ], 
        [ 2918125, 2921020, 2918130, 2921025 ], 
        [ 2921021, 2921458, 2921026, 2921463 ], 
        [ 2921459, 2923338, 2921464, 2923342 ], 
        [ 2923339, 2926580, 2923343, 2926585 ], 
        [ 2926581, 2930570, 2926586, 2930575 ], 
        [ 2930571, 2931452, 2930576, 2931456 ], 
        [ 2931453, 2934873, 2931457, 2934878 ], 
        [ 2934874, 2940294, 2934879, 2940298 ], 
        [ 2940295, 2942883, 2940299, 2942888 ], 
        [ 2942884, 2943246, 2942889, 2943250 ], 
        [ 2943247, 2946109, 2943251, 2946105 ], 
        [ 2946110, 2950823, 2946106, 2950828 ], 
        [ 2950824, 2952204, 2950829, 2952209 ], 
        [ 2952205, 2952842, 2952210, 2952846 ], 
        [ 2952843, 2954108, 2952847, 2954113 ], 
        [ 2954109, 2958788, 2954114, 2958793 ], 
        [ 2958789, 2959652, 2958794, 2959656 ], 
        [ 2959653, 2962338, 2959657, 2962343 ], 
        [ 2962339, 2962634, 2962344, 2962639 ], 
        [ 2962635, 2963567, 2962640, 2963572 ], 
        [ 2963568, 2965512, 2963573, 2965517 ], 
        [ 2965513, 2965542, 2965518, 2965546 ], 
        [ 2965543, 2965715, 2965547, 2965720 ], 
        [ 2965716, 2969537, 2965721, 2969542 ], 
        [ 2969538, 2969667, 2969543, 2969672 ], 
        [ 2969668, 2971097, 2969673, 2971102 ], 
        [ 2971098, 2971329, 2971103, 2971334 ], 
        [ 2971330, 2971441, 2971335, 2971437 ], 
        [ 2971442, 2971588, 2971438, 2971584 ], 
        [ 2971589, 2971874, 2971585, 2971879 ], 
        [ 2971875, 2972098, 2971880, 2972103 ], 
        [ 2972099, 2974355, 2972104, 2974359 ], 
        [ 2974356, 2978342, 2974360, 2978347 ], 
        [ 2978343, 2984060, 2978348, 2984065 ], 
        [ 2984061, 2985968, 2984066, 2985972 ], 
        [ 2985969, 2987338, 2985973, 2987342 ], 
        [ 2987339, 2988924, 2987343, 2988929 ], 
        [ 2988925, 2991267, 2988930, 2991271 ], 
        [ 2991268, 2994739, 2991272, 2994744 ], 
        [ 2994740, 3002088, 2994745, 3002093 ], 
        [ 3002089, 3009887, 3002094, 3009892 ], 
        [ 3009888, 3014827, 3009893, 3014832 ], 
        [ 3014828, 3017608, 3014833, 3017612 ], 
        [ 3017609, 3020196, 3017613, 3020192 ], 
        [ 3020197, 3020885, 3020193, 3020890 ], 
        [ 3020886, 3022261, 3020891, 3022266 ], 
        [ 3022262, 3029543, 3022267, 3029548 ], 
        [ 3029544, 3030265, 3029549, 3030270 ], 
        [ 3030266, 3032363, 3030271, 3032368 ], 
        [ 3032364, 3033161, 3032369, 3033166 ], 
        [ 3033162, 3042175, 3033167, 3042180 ], 
        [ 3042176, 3042389, 3042181, 3042394 ], 
        [ 3042390, 3043311, 3042395, 3043315 ], 
        [ 3043312, 3045343, 3043316, 3045347 ], 
        [ 3045344, 3049663, 3045348, 3049668 ], 
        [ 3049664, 3050210, 3049669, 3050215 ], 
        [ 3050211, 3051389, 3050216, 3051394 ], 
        [ 3051390, 3052128, 3051395, 3052133 ], 
        [ 3052129, 3052883, 3052134, 3052888 ], 
        [ 3052884, 3054679, 3052889, 3054684 ], 
        [ 3054680, 3055955, 3054685, 3055960 ], 
        [ 3055956, 3056024, 3055961, 3056029 ], 
        [ 3056025, 3062859, 3056030, 3062864 ], 
        [ 3062860, 3063276, 3062865, 3063281 ], 
        [ 3063277, 3064101, 3063282, 3064106 ], 
        [ 3064102, 3065135, 3064107, 3065131 ], 
        [ 3065136, 3065575, 3065132, 3065580 ], 
        [ 3065576, 3065710, 3065581, 3065715 ], 
        [ 3065711, 3066590, 3065716, 3066595 ], 
        [ 3066591, 3070621, 3066596, 3070625 ], 
        [ 3070622, 3075292, 3070626, 3075297 ], 
        [ 3075293, 3076853, 3075298, 3076858 ], 
        [ 3076854, 3078695, 3076859, 3078699 ], 
        [ 3078696, 3080566, 3078700, 3080571 ], 
        [ 3080567, 3080707, 3080572, 3080712 ], 
        [ 3080708, 3080747, 3080713, 3080752 ], 
        [ 3080748, 3080916, 3080753, 3080921 ], 
        [ 3080917, 3082200, 3080922, 3082205 ], 
        [ 3082201, 3085986, 3082206, 3085991 ], 
        [ 3085987, 3086111, 3085992, 3086116 ], 
        [ 3086112, 3088509, 3086117, 3088514 ], 
        [ 3088510, 3090889, 3088515, 3090893 ], 
        [ 3090890, 3091108, 3090894, 3091113 ], 
        [ 3091109, 3094298, 3091114, 3094303 ], 
        [ 3094299, 3095542, 3094304, 3095546 ], 
        [ 3095543, 3098254, 3095547, 3098259 ], 
        [ 3098255, 3100958, 3098260, 3100963 ], 
        [ 3100959, 3101111, 3100964, 3101116 ], 
        [ 3101112, 3101710, 3101117, 3101715 ], 
        [ 3101711, 3102184, 3101716, 3102180 ], 
        [ 3102185, 3107070, 3102181, 3107075 ], 
        [ 3107071, 3108550, 3107076, 3108555 ], 
        [ 3108551, 3109638, 3108556, 3109642 ], 
        [ 3109639, 3110010, 3109643, 3110014 ], 
        [ 3110011, 3110596, 3110015, 3110592 ], 
        [ 3110597, 3114705, 3110593, 3114709 ], 
        [ 3114706, 3116752, 3114710, 3116757 ], 
        [ 3116753, 3118943, 3116758, 3118947 ], 
        [ 3118944, 3118963, 3118948, 3118967 ], 
        [ 3118964, 3119656, 3118968, 3119661 ], 
        [ 3119657, 3120436, 3119662, 3120441 ], 
        [ 3120437, 3122936, 3120442, 3122941 ], 
        [ 3122937, 3125746, 3122942, 3125751 ], 
        [ 3125747, 3126385, 3125752, 3126390 ], 
        [ 3126386, 3126792, 3126391, 3126796 ], 
        [ 3126793, 3129487, 3126797, 3129491 ], 
        [ 3129488, 3129525, 3129492, 3129530 ], 
        [ 3129526, 3129570, 3129531, 3129575 ], 
        [ 3129571, 3129752, 3129576, 3129757 ], 
        [ 3129753, 3132349, 3129758, 3132354 ], 
        [ 3132350, 3133454, 3132355, 3133458 ], 
        [ 3133455, 3135493, 3133459, 3135497 ], 
        [ 3135494, 3139904, 3135498, 3139909 ], 
        [ 3139905, 3143832, 3139910, 3143837 ], 
        [ 3143833, 3144573, 3143838, 3144578 ], 
        [ 3144574, 3146787, 3144579, 3146783 ], 
        [ 3146788, 3148252, 3146784, 3148257 ], 
        [ 3148253, 3149799, 3148258, 3149803 ], 
        [ 3149800, 3150053, 3149804, 3150058 ], 
        [ 3150054, 3152136, 3150059, 3152141 ], 
        [ 3152137, 3159529, 3152142, 3159533 ], 
        [ 3159530, 3161578, 3159534, 3161583 ], 
        [ 3161579, 3162165, 3161584, 3162169 ], 
        [ 3162166, 3162816, 3162170, 3162820 ], 
        [ 3162817, 3164826, 3162821, 3164831 ], 
        [ 3164827, 3164943, 3164832, 3164948 ], 
        [ 3164944, 3167365, 3164949, 3167369 ], 
        [ 3167366, 3167563, 3167370, 3167568 ], 
        [ 3167564, 3168258, 3167569, 3168263 ], 
        [ 3168259, 3170295, 3168264, 3170300 ], 
        [ 3170296, 3172533, 3170301, 3172538 ], 
        [ 3172534, 3176792, 3172539, 3176797 ], 
        [ 3176793, 3177637, 3176798, 3177642 ], 
        [ 3177638, 3178306, 3177643, 3178311 ], 
        [ 3178307, 3180657, 3178312, 3180661 ], 
        [ 3180658, 3181485, 3180662, 3181489 ], 
        [ 3181486, 3182629, 3181490, 3182633 ], 
        [ 3182630, 3186857, 3182634, 3186862 ], 
        [ 3186858, 3190084, 3186863, 3190088 ], 
        [ 3190085, 3194127, 3190089, 3194131 ], 
        [ 3194128, 3196386, 3194132, 3196391 ], 
        [ 3196387, 3196711, 3196392, 3196715 ], 
        [ 3196712, 3197840, 3196716, 3197845 ], 
        [ 3197841, 3201756, 3197846, 3201760 ], 
        [ 3201757, 3208054, 3201761, 3208058 ], 
        [ 3208055, 3210828, 3208059, 3210833 ], 
        [ 3210829, 3212953, 3210834, 3212958 ], 
        [ 3212954, 3224927, 3212959, 3224931 ], 
        [ 3224928, 3226292, 3224932, 3226297 ], 
        [ 3226293, 3226762, 3226298, 3226767 ], 
        [ 3226763, 3228540, 3226768, 3228544 ], 
        [ 3228541, 3229224, 3228545, 3229228 ], 
        [ 3229225, 3230485, 3229229, 3230490 ], 
        [ 3230486, 3233260, 3230491, 3233264 ], 
        [ 3233261, 3233743, 3233265, 3233748 ], 
        [ 3233744, 3234567, 3233749, 3234572 ], 
        [ 3234568, 3235194, 3234573, 3235199 ], 
        [ 3235195, 3235499, 3235200, 3235504 ], 
        [ 3235500, 3243889, 3235505, 3243894 ], 
        [ 3243890, 3245053, 3243895, 3245057 ], 
        [ 3245054, 3245941, 3245058, 3245945 ], 
        [ 3245942, 3255842, 3245946, 3255838 ], 
        [ 3255843, 3259251, 3255839, 3259256 ], 
        [ 3259252, 3260019, 3259257, 3260023 ], 
        [ 3260020, 3262432, 3260024, 3262437 ], 
        [ 3262433, 3262672, 3262438, 3262677 ], 
        [ 3262673, 3266221, 3262678, 3266225 ], 
        [ 3266222, 3266686, 3266226, 3266691 ], 
        [ 3266687, 3269046, 3266692, 3269051 ], 
        [ 3269047, 3269620, 3269052, 3269625 ], 
        [ 3269621, 3272036, 3269626, 3272041 ], 
        [ 3272037, 3272516, 3272042, 3272521 ], 
        [ 3272517, 3272796, 3272522, 3272801 ], 
        [ 3272797, 3278978, 3272802, 3278983 ], 
        [ 3278979, 3279467, 3278984, 3279472 ], 
        [ 3279468, 3280283, 3279473, 3280288 ], 
        [ 3280284, 3282531, 3280289, 3282536 ], 
        [ 3282532, 3283074, 3282537, 3283078 ], 
        [ 3283075, 3283850, 3283079, 3283854 ], 
        [ 3283851, 3286305, 3283855, 3286310 ], 
        [ 3286306, 3286606, 3286311, 3286611 ], 
        [ 3286607, 3287302, 3286612, 3287307 ], 
        [ 3287303, 3290573, 3287308, 3290569 ], 
        [ 3290574, 3290776, 3290570, 3290781 ], 
        [ 3290777, 3294305, 3290782, 3294310 ], 
        [ 3294306, 3295201, 3294311, 3295205 ], 
        [ 3295202, 3298010, 3295206, 3298015 ], 
        [ 3298011, 3298799, 3298016, 3298804 ], 
        [ 3298800, 3299744, 3298805, 3299749 ], 
        [ 3299745, 3302402, 3299750, 3302407 ], 
        [ 3302403, 3308399, 3302408, 3308403 ], 
        [ 3308400, 3313629, 3308404, 3313633 ], 
        [ 3313630, 3314750, 3313634, 3314755 ], 
        [ 3314751, 3314777, 3314756, 3314782 ], 
        [ 3314778, 3315543, 3314783, 3315548 ], 
        [ 3315544, 3317191, 3315549, 3317195 ], 
        [ 3317192, 3319495, 3317196, 3319500 ], 
        [ 3319496, 3321744, 3319501, 3321749 ], 
        [ 3321745, 3324685, 3321750, 3324690 ], 
        [ 3324686, 3331190, 3324691, 3331194 ], 
        [ 3331191, 3331241, 3331195, 3331246 ], 
        [ 3331242, 3334147, 3331247, 3334151 ], 
        [ 3334148, 3334546, 3334152, 3334550 ], 
        [ 3334547, 3339219, 3334551, 3339215 ], 
        [ 3339220, 3339452, 3339216, 3339456 ], 
        [ 3339453, 3345624, 3339457, 3345629 ], 
        [ 3345625, 3349642, 3345630, 3349646 ], 
        [ 3349643, 3349729, 3349647, 3349734 ], 
        [ 3349730, 3350579, 3349735, 3350584 ], 
        [ 3350580, 3351752, 3350585, 3351756 ], 
        [ 3351753, 3351958, 3351757, 3351962 ], 
        [ 3351959, 3353994, 3351963, 3353999 ], 
        [ 3353995, 3355238, 3354000, 3355243 ], 
        [ 3355239, 3356216, 3355244, 3356220 ], 
        [ 3356217, 3360120, 3356221, 3360125 ], 
        [ 3360121, 3360807, 3360126, 3360811 ], 
        [ 3360808, 3362361, 3360812, 3362365 ], 
        [ 3362362, 3363476, 3362366, 3363480 ], 
        [ 3363477, 3364333, 3363481, 3364337 ], 
        [ 3364334, 3368828, 3364338, 3368833 ], 
        [ 3368829, 3370235, 3368834, 3370239 ], 
        [ 3370236, 3371627, 3370240, 3371631 ], 
        [ 3371628, 3375963, 3371632, 3375967 ], 
        [ 3375964, 3376335, 3375968, 3376340 ], 
        [ 3376336, 3380722, 3376341, 3380727 ], 
        [ 3380723, 3381309, 3380728, 3381314 ], 
        [ 3381310, 3382080, 3381315, 3382085 ], 
        [ 3382081, 3384674, 3382086, 3384679 ], 
        [ 3384675, 3385188, 3384680, 3385193 ], 
        [ 3385189, 3388713, 3385194, 3388718 ], 
        [ 3388714, 3392079, 3388719, 3392084 ], 
        [ 3392080, 3394131, 3392085, 3394136 ], 
        [ 3394132, 3394381, 3394137, 3394386 ], 
        [ 3394382, 3395362, 3394387, 3395367 ], 
        [ 3395363, 3395419, 3395368, 3395423 ], 
        [ 3395420, 3396615, 3395424, 3396620 ], 
        [ 3396616, 3399453, 3396621, 3399458 ], 
        [ 3399454, 3400585, 3399459, 3400590 ], 
        [ 3400586, 3401981, 3400591, 3401977 ], 
        [ 3401982, 3405494, 3401978, 3405499 ], 
        [ 3405495, 3408302, 3405500, 3408306 ], 
        [ 3408303, 3413040, 3408307, 3413045 ], 
        [ 3413041, 3413955, 3413046, 3413960 ], 
        [ 3413956, 3413965, 3413961, 3413970 ], 
        [ 3413966, 3414900, 3413971, 3414904 ], 
        [ 3414901, 3415450, 3414905, 3415455 ], 
        [ 3415451, 3415543, 3415456, 3415548 ], 
        [ 3415544, 3415693, 3415549, 3415698 ], 
        [ 3415694, 3416832, 3415699, 3416828 ], 
        [ 3416833, 3416866, 3416829, 3416871 ], 
        [ 3416867, 3420715, 3416872, 3420719 ], 
        [ 3420716, 3421885, 3420720, 3421890 ], 
        [ 3421886, 3425027, 3421891, 3425031 ], 
        [ 3425028, 3425142, 3425032, 3425147 ], 
        [ 3425143, 3429164, 3425148, 3429168 ], 
        [ 3429165, 3431958, 3429169, 3431962 ], 
        [ 3431959, 3437815, 3431963, 3437820 ], 
        [ 3437816, 3439239, 3437821, 3439243 ], 
        [ 3439240, 3440231, 3439244, 3440236 ], 
        [ 3440232, 3440415, 3440237, 3440411 ], 
        [ 3440416, 3442385, 3440412, 3442390 ], 
        [ 3442386, 3448532, 3442391, 3448536 ], 
        [ 3448533, 3449018, 3448537, 3449014 ], 
        [ 3449019, 3449766, 3449015, 3449771 ], 
        [ 3449767, 3450597, 3449772, 3450601 ], 
        [ 3450598, 3451510, 3450602, 3451514 ], 
        [ 3451511, 3455799, 3451515, 3455804 ], 
        [ 3455800, 3457236, 3455805, 3457240 ], 
        [ 3457237, 3463488, 3457241, 3463492 ], 
        [ 3463489, 3473047, 3463493, 3473052 ], 
        [ 3473048, 3479777, 3473053, 3479773 ], 
        [ 3479778, 3483375, 3479774, 3483380 ], 
        [ 3483376, 3484793, 3483381, 3484797 ], 
        [ 3484794, 3486395, 3484798, 3486399 ], 
        [ 3486396, 3490246, 3486400, 3490242 ], 
        [ 3490247, 3494295, 3490243, 3494299 ], 
        [ 3494296, 3495597, 3494300, 3495601 ], 
        [ 3495598, 3496721, 3495602, 3496725 ], 
        [ 3496722, 3499522, 3496726, 3499526 ], 
        [ 3499523, 3503195, 3499527, 3503199 ], 
        [ 3503196, 3505500, 3503200, 3505505 ], 
        [ 3505501, 3510463, 3505506, 3510467 ], 
        [ 3510464, 3514082, 3510468, 3514087 ], 
        [ 3514083, 3520619, 3514088, 3520623 ], 
        [ 3520620, 3521049, 3520624, 3521053 ], 
        [ 3521050, 3522092, 3521054, 3522097 ], 
        [ 3522093, 3522111, 3522098, 3522116 ], 
        [ 3522112, 3522311, 3522117, 3522315 ], 
        [ 3522312, 3522767, 3522316, 3522772 ], 
        [ 3522768, 3531627, 3522773, 3531632 ], 
        [ 3531628, 3537780, 3531633, 3537785 ], 
        [ 3537781, 3538989, 3537786, 3538993 ], 
        [ 3538990, 3540393, 3538994, 3540397 ], 
        [ 3540394, 3540644, 3540398, 3540649 ], 
        [ 3540645, 3543017, 3540650, 3543022 ], 
        [ 3543018, 3543842, 3543023, 3543847 ], 
        [ 3543843, 3545546, 3543848, 3545551 ], 
        [ 3545547, 3546913, 3545552, 3546917 ], 
        [ 3546914, 3547077, 3546918, 3547082 ], 
        [ 3547078, 3547149, 3547083, 3547154 ], 
        [ 3547150, 3548533, 3547155, 3548538 ], 
        [ 3548534, 3549829, 3548539, 3549834 ], 
        [ 3549830, 3549935, 3549835, 3549940 ], 
        [ 3549936, 3550203, 3549941, 3550208 ], 
        [ 3550204, 3550227, 3550209, 3550232 ], 
        [ 3550228, 3551480, 3550233, 3551485 ], 
        [ 3551481, 3552210, 3551486, 3552215 ], 
        [ 3552211, 3561260, 3552216, 3561264 ], 
        [ 3561261, 3562847, 3561265, 3562852 ], 
        [ 3562848, 3563068, 3562853, 3563073 ], 
        [ 3563069, 3563973, 3563074, 3563978 ], 
        [ 3563974, 3565130, 3563979, 3565135 ], 
        [ 3565131, 3565565, 3565136, 3565570 ], 
        [ 3565566, 3566132, 3565571, 3566137 ], 
        [ 3566133, 3567018, 3566138, 3567023 ], 
        [ 3567019, 3567627, 3567024, 3567632 ], 
        [ 3567628, 3570838, 3567633, 3570843 ], 
        [ 3570839, 3574124, 3570844, 3574129 ], 
        [ 3574125, 3575098, 3574130, 3575103 ], 
        [ 3575099, 3575586, 3575104, 3575591 ], 
        [ 3575587, 3576602, 3575592, 3576607 ], 
        [ 3576603, 3590383, 3576608, 3590388 ], 
        [ 3590384, 3594697, 3590389, 3594702 ], 
        [ 3594698, 3596771, 3594703, 3596775 ], 
        [ 3596772, 3597307, 3596776, 3597312 ], 
        [ 3597308, 3598184, 3597313, 3598188 ], 
        [ 3598185, 3598837, 3598189, 3598842 ], 
        [ 3598838, 3599839, 3598843, 3599844 ], 
        [ 3599840, 3603578, 3599845, 3603583 ], 
        [ 3603579, 3605362, 3603584, 3605366 ], 
        [ 3605363, 3609322, 3605367, 3609327 ], 
        [ 3609323, 3614549, 3609328, 3614554 ], 
        [ 3614550, 3618664, 3614555, 3618669 ], 
        [ 3618665, 3619899, 3618670, 3619904 ], 
        [ 3619900, 3624824, 3619905, 3624829 ], 
        [ 3624825, 3627309, 3624830, 3627313 ], 
        [ 3627310, 3628159, 3627314, 3628164 ], 
        [ 3628160, 3628423, 3628165, 3628428 ], 
        [ 3628424, 3629521, 3628429, 3629525 ], 
        [ 3629522, 3629584, 3629526, 3629588 ], 
        [ 3629585, 3629833, 3629589, 3629837 ], 
        [ 3629834, 3630285, 3629838, 3630290 ], 
        [ 3630286, 3630486, 3630291, 3630491 ], 
        [ 3630487, 3634237, 3630492, 3634242 ], 
        [ 3634238, 3639209, 3634243, 3639214 ], 
        [ 3639210, 3639649, 3639215, 3639645 ], 
        [ 3639650, 3647161, 3639646, 3647165 ], 
        [ 3647162, 3648227, 3647166, 3648231 ], 
        [ 3648228, 3652188, 3648232, 3652193 ], 
        [ 3652189, 3659027, 3652194, 3659032 ], 
        [ 3659028, 3659079, 3659033, 3659084 ], 
        [ 3659080, 3659207, 3659085, 3659212 ], 
        [ 3659208, 3660377, 3659213, 3660381 ], 
        [ 3660378, 3667507, 3660382, 3667512 ], 
        [ 3667508, 3667741, 3667513, 3667746 ], 
        [ 3667742, 3668429, 3667747, 3668434 ], 
        [ 3668430, 3670470, 3668435, 3670475 ], 
        [ 3670471, 3673148, 3670476, 3673153 ], 
        [ 3673149, 3674083, 3673154, 3674088 ], 
        [ 3674084, 3678214, 3674089, 3678219 ], 
        [ 3678215, 3680677, 3678220, 3680682 ], 
        [ 3680678, 3684205, 3680683, 3684210 ], 
        [ 3684206, 3688887, 3684211, 3688883 ], 
        [ 3688888, 3690753, 3688884, 3690758 ], 
        [ 3690754, 3698350, 3690759, 3698355 ], 
        [ 3698351, 3699287, 3698356, 3699292 ], 
        [ 3699288, 3700655, 3699293, 3700659 ], 
        [ 3700656, 3702298, 3700660, 3702303 ], 
        [ 3702299, 3706224, 3702304, 3706229 ], 
        [ 3706225, 3706948, 3706230, 3706953 ], 
        [ 3706949, 3709050, 3706954, 3709055 ], 
        [ 3709051, 3710401, 3709056, 3710405 ], 
        [ 3710402, 3713151, 3710406, 3713155 ], 
        [ 3713152, 3714587, 3713156, 3714591 ], 
        [ 3714588, 3714794, 3714592, 3714798 ], 
        [ 3714795, 3725109, 3714799, 3725114 ], 
        [ 3725110, 3725822, 3725115, 3725818 ], 
        [ 3725823, 3726413, 3725819, 3726409 ], 
        [ 3726414, 3731356, 3726410, 3731352 ], 
        [ 3731357, 3731514, 3731353, 3731519 ], 
        [ 3731515, 3736239, 3731520, 3736244 ], 
        [ 3736240, 3736361, 3736245, 3736366 ], 
        [ 3736362, 3738698, 3736367, 3738703 ], 
        [ 3738699, 3742314, 3738704, 3742318 ], 
        [ 3742315, 3745350, 3742319, 3745355 ], 
        [ 3745351, 3747516, 3745356, 3747521 ], 
        [ 3747517, 3747787, 3747522, 3747791 ], 
        [ 3747788, 3748248, 3747792, 3748253 ], 
        [ 3748249, 3750382, 3748254, 3750387 ], 
        [ 3750383, 3752531, 3750388, 3752527 ], 
        [ 3752532, 3753792, 3752528, 3753788 ], 
        [ 3753793, 3755999, 3753789, 3756003 ], 
        [ 3756000, 3761090, 3756004, 3761095 ], 
        [ 3761091, 3762084, 3761096, 3762089 ], 
        [ 3762085, 3762616, 3762090, 3762621 ], 
        [ 3762617, 3769672, 3762622, 3769677 ], 
        [ 3769673, 3769809, 3769678, 3769814 ], 
        [ 3769810, 3769962, 3769815, 3769967 ], 
        [ 3769963, 3770651, 3769968, 3770655 ], 
        [ 3770652, 3773164, 3770656, 3773168 ], 
        [ 3773165, 3774893, 3773169, 3774898 ], 
        [ 3774894, 3776296, 3774899, 3776300 ], 
        [ 3776297, 3776322, 3776301, 3776327 ], 
        [ 3776323, 3778089, 3776328, 3778094 ], 
        [ 3778090, 3783192, 3778095, 3783188 ], 
        [ 3783193, 3784214, 3783189, 3784219 ], 
        [ 3784215, 3795080, 3784220, 3795084 ], 
        [ 3795081, 3799015, 3795085, 3799020 ], 
        [ 3799016, 3805176, 3799021, 3805180 ], 
        [ 3805177, 3806065, 3805181, 3806070 ], 
        [ 3806066, 3807768, 3806071, 3807773 ], 
        [ 3807769, 3808991, 3807774, 3808995 ], 
        [ 3808992, 3809153, 3808996, 3809157 ], 
        [ 3809154, 3809882, 3809158, 3809886 ], 
        [ 3809883, 3813249, 3809887, 3813253 ], 
        [ 3813250, 3816682, 3813254, 3816687 ], 
        [ 3816683, 3817044, 3816688, 3817049 ], 
        [ 3817045, 3817768, 3817050, 3817773 ], 
        [ 3817769, 3821121, 3817774, 3821125 ], 
        [ 3821122, 3822167, 3821126, 3822172 ], 
        [ 3822168, 3824032, 3822173, 3824037 ], 
        [ 3824033, 3827315, 3824038, 3827311 ], 
        [ 3827316, 3828542, 3827312, 3828547 ], 
        [ 3828543, 3829917, 3828548, 3829921 ], 
        [ 3829918, 3832118, 3829922, 3832123 ], 
        [ 3832119, 3832453, 3832124, 3832458 ], 
        [ 3832454, 3837573, 3832459, 3837578 ], 
        [ 3837574, 3842658, 3837579, 3842663 ], 
        [ 3842659, 3843673, 3842664, 3843677 ], 
        [ 3843674, 3844593, 3843678, 3844597 ], 
        [ 3844594, 3844599, 3844598, 3844604 ], 
        [ 3844600, 3844788, 3844605, 3844793 ], 
        [ 3844789, 3850826, 3844794, 3850831 ], 
        [ 3850827, 3854490, 3850832, 3854494 ], 
        [ 3854491, 3855395, 3854495, 3855400 ], 
        [ 3855396, 3855755, 3855401, 3855751 ], 
        [ 3855756, 3860530, 3855752, 3860535 ], 
        [ 3860531, 3860887, 3860536, 3860883 ], 
        [ 3860888, 3862913, 3860884, 3862917 ], 
        [ 3862914, 3872850, 3862918, 3872854 ], 
        [ 3872851, 3873056, 3872855, 3873061 ], 
        [ 3873057, 3875254, 3873062, 3875258 ], 
        [ 3875255, 3880197, 3875259, 3880202 ], 
        [ 3880198, 3882479, 3880203, 3882483 ], 
        [ 3882480, 3882820, 3882484, 3882825 ], 
        [ 3882821, 3883310, 3882826, 3883315 ], 
        [ 3883311, 3885744, 3883316, 3885749 ], 
        [ 3885745, 3890942, 3885750, 3890946 ], 
        [ 3890943, 3891218, 3890947, 3891223 ], 
        [ 3891219, 3891354, 3891224, 3891359 ], 
        [ 3891355, 3893483, 3891360, 3893487 ], 
        [ 3893484, 3894419, 3893488, 3894424 ], 
        [ 3894420, 3900500, 3894425, 3900505 ], 
        [ 3900501, 3901493, 3900506, 3901489 ], 
        [ 3901494, 3903140, 3901490, 3903136 ], 
        [ 3903141, 3907313, 3903137, 3907318 ], 
        [ 3907314, 3908755, 3907319, 3908759 ], 
        [ 3908756, 3908919, 3908760, 3908924 ], 
        [ 3908920, 3909807, 3908925, 3909812 ], 
        [ 3909808, 3909977, 3909813, 3909982 ], 
        [ 3909978, 3911959, 3909983, 3911955 ], 
        [ 3911960, 3914009, 3911956, 3914005 ], 
        [ 3914010, 3922759, 3914006, 3922763 ], 
        [ 3922760, 3930585, 3922764, 3930590 ], 
        [ 3930586, 3931083, 3930591, 3931087 ], 
        [ 3931084, 3933248, 3931088, 3933252 ], 
        [ 3933249, 3933363, 3933253, 3933368 ], 
        [ 3933364, 3935176, 3933369, 3935181 ], 
        [ 3935177, 3936871, 3935182, 3936876 ], 
        [ 3936872, 3937525, 3936877, 3937529 ], 
        [ 3937526, 3945198, 3937530, 3945203 ], 
        [ 3945199, 3946390, 3945204, 3946395 ], 
        [ 3946391, 3946986, 3946396, 3946991 ], 
        [ 3946987, 3952348, 3946992, 3952344 ], 
        [ 3952349, 3956408, 3952345, 3956413 ], 
        [ 3956409, 3958333, 3956414, 3958338 ], 
        [ 3958334, 3959031, 3958339, 3959036 ], 
        [ 3959032, 3960932, 3959037, 3960937 ], 
        [ 3960933, 3964190, 3960938, 3964195 ], 
        [ 3964191, 3969413, 3964196, 3969418 ], 
        [ 3969414, 3972146, 3969419, 3972151 ], 
        [ 3972147, 3972344, 3972152, 3972349 ], 
        [ 3972345, 3978065, 3972350, 3978070 ], 
        [ 3978066, 3981977, 3978071, 3981982 ], 
        [ 3981978, 3984768, 3981983, 3984773 ], 
        [ 3984769, 3984918, 3984774, 3984923 ], 
        [ 3984919, 3985704, 3984924, 3985709 ], 
        [ 3985705, 3986544, 3985710, 3986548 ], 
        [ 3986545, 3995454, 3986549, 3995459 ], 
        [ 3995455, 3997410, 3995460, 3997415 ], 
        [ 3997411, 4000982, 3997416, 4000987 ], 
        [ 4000983, 4002297, 4000988, 4002301 ], 
        [ 4002298, 4002514, 4002302, 4002519 ], 
        [ 4002515, 4005689, 4002520, 4005694 ], 
        [ 4005690, 4008445, 4005695, 4008449 ], 
        [ 4008446, 4008591, 4008450, 4008595 ], 
        [ 4008592, 4016078, 4008596, 4016083 ], 
        [ 4016079, 4017237, 4016084, 4017242 ], 
        [ 4017238, 4018412, 4017243, 4018417 ], 
        [ 4018413, 4018717, 4018418, 4018722 ], 
        [ 4018718, 4018835, 4018723, 4018839 ], 
        [ 4018836, 4021930, 4018840, 4021934 ], 
        [ 4021931, 4023281, 4021935, 4023286 ], 
        [ 4023282, 4032076, 4023287, 4032081 ], 
        [ 4032077, 4041164, 4032082, 4041169 ], 
        [ 4041165, 4041295, 4041170, 4041300 ], 
        [ 4041296, 4041920, 4041301, 4041925 ], 
        [ 4041921, 4046747, 4041926, 4046752 ], 
        [ 4046748, 4049001, 4046753, 4049005 ], 
        [ 4049002, 4050345, 4049006, 4050349 ], 
        [ 4050346, 4052606, 4050350, 4052611 ], 
        [ 4052607, 4054469, 4052612, 4054474 ], 
        [ 4054470, 4054629, 4054475, 4054634 ], 
        [ 4054630, 4054968, 4054635, 4054973 ], 
        [ 4054969, 4055333, 4054974, 4055338 ], 
        [ 4055334, 4056058, 4055339, 4056063 ], 
        [ 4056059, 4057967, 4056064, 4057963 ], 
        [ 4057968, 4059084, 4057964, 4059089 ], 
        [ 4059085, 4062508, 4059090, 4062513 ], 
        [ 4062509, 4065383, 4062514, 4065388 ], 
        [ 4065384, 4065588, 4065389, 4065592 ], 
        [ 4065589, 4065643, 4065593, 4065648 ], 
        [ 4065644, 4065670, 4065649, 4065674 ], 
        [ 4065671, 4069949, 4065675, 4069954 ], 
        [ 4069950, 4073053, 4069955, 4073057 ], 
        [ 4073054, 4076608, 4073058, 4076612 ], 
        [ 4076609, 4078343, 4076613, 4078348 ], 
        [ 4078344, 4083896, 4078349, 4083901 ], 
        [ 4083897, 4085884, 4083902, 4085889 ], 
        [ 4085885, 4090152, 4085890, 4090157 ], 
        [ 4090153, 4093908, 4090158, 4093913 ], 
        [ 4093909, 4094118, 4093914, 4094123 ], 
        [ 4094119, 4095249, 4094124, 4095254 ], 
        [ 4095250, 4095382, 4095255, 4095386 ], 
        [ 4095383, 4095456, 4095387, 4095460 ], 
        [ 4095457, 4097811, 4095461, 4097816 ], 
        [ 4097812, 4101896, 4097817, 4101901 ], 
        [ 4101897, 4102182, 4101902, 4102186 ], 
        [ 4102183, 4104214, 4102187, 4104218 ], 
        [ 4104215, 4107088, 4104219, 4107093 ], 
        [ 4107089, 4107551, 4107094, 4107556 ], 
        [ 4107552, 4107580, 4107557, 4107585 ], 
        [ 4107581, 4107671, 4107586, 4107675 ], 
        [ 4107672, 4109180, 4107676, 4109185 ], 
        [ 4109181, 4110537, 4109186, 4110542 ], 
        [ 4110538, 4116188, 4110543, 4116192 ], 
        [ 4116189, 4116491, 4116193, 4116496 ], 
        [ 4116492, 4117909, 4116497, 4117914 ], 
        [ 4117910, 4118469, 4117915, 4118474 ], 
        [ 4118470, 4123661, 4118475, 4123666 ], 
        [ 4123662, 4123865, 4123667, 4123870 ], 
        [ 4123866, 4125127, 4123871, 4125132 ], 
        [ 4125128, 4125135, 4125133, 4125139 ], 
        [ 4125136, 4129187, 4125140, 4129192 ], 
        [ 4129188, 4132972, 4129193, 4132977 ], 
        [ 4132973, 4134272, 4132978, 4134277 ], 
        [ 4134273, 4135470, 4134278, 4135475 ], 
        [ 4135471, 4136837, 4135476, 4136842 ], 
        [ 4136838, 4146292, 4136843, 4146297 ], 
        [ 4146293, 4146443, 4146298, 4146448 ], 
        [ 4146444, 4148935, 4146449, 4148940 ], 
        [ 4148936, 4150723, 4148941, 4150727 ], 
        [ 4150724, 4156926, 4150728, 4156930 ], 
        [ 4156927, 4160939, 4156931, 4160943 ], 
        [ 4160940, 4161425, 4160944, 4161421 ], 
        [ 4161426, 4162265, 4161422, 4162270 ], 
        [ 4162266, 4163096, 4162271, 4163100 ], 
        [ 4163097, 4163991, 4163101, 4163995 ], 
        [ 4163992, 4164781, 4163996, 4164786 ], 
        [ 4164782, 4167637, 4164787, 4167633 ], 
        [ 4167638, 4170898, 4167634, 4170903 ], 
        [ 4170899, 4175178, 4170904, 4175174 ], 
        [ 4175179, 4175563, 4175175, 4175567 ], 
        [ 4175564, 4175870, 4175568, 4175875 ], 
        [ 4175871, 4175960, 4175876, 4175965 ], 
        [ 4175961, 4178040, 4175966, 4178044 ], 
        [ 4178041, 4179795, 4178045, 4179799 ], 
        [ 4179796, 4180869, 4179800, 4180874 ], 
        [ 4180870, 4181479, 4180875, 4181484 ], 
        [ 4181480, 4184808, 4181485, 4184812 ], 
        [ 4184809, 4185316, 4184813, 4185320 ], 
        [ 4185317, 4187692, 4185321, 4187696 ], 
        [ 4187693, 4187785, 4187697, 4187790 ], 
        [ 4187786, 4188293, 4187791, 4188289 ], 
        [ 4188294, 4189984, 4188290, 4189980 ], 
        [ 4189985, 4191368, 4189981, 4191372 ], 
        [ 4191369, 4191989, 4191373, 4191993 ], 
        [ 4191990, 4192187, 4191994, 4192191 ], 
        [ 4192188, 4192423, 4192192, 4192428 ], 
        [ 4192424, 4192625, 4192429, 4192630 ], 
        [ 4192626, 4194747, 4192631, 4194752 ], 
        [ 4194748, 4195600, 4194753, 4195605 ], 
        [ 4195601, 4195842, 4195606, 4195847 ], 
        [ 4195843, 4196615, 4195848, 4196619 ], 
        [ 4196616, 4197634, 4196620, 4197639 ], 
        [ 4197635, 4198036, 4197640, 4198041 ], 
        [ 4198037, 4198447, 4198042, 4198451 ], 
        [ 4198448, 4198840, 4198452, 4198836 ], 
        [ 4198841, 4203137, 4198837, 4203141 ], 
        [ 4203138, 4203155, 4203142, 4203159 ], 
        [ 4203156, 4203840, 4203160, 4203844 ], 
        [ 4203841, 4205660, 4203845, 4205665 ], 
        [ 4205661, 4213510, 4205666, 4213515 ], 
        [ 4213511, 4217238, 4213516, 4217243 ], 
        [ 4217239, 4219123, 4217244, 4219127 ], 
        [ 4219124, 4221217, 4219128, 4221222 ], 
        [ 4221218, 4225159, 4221223, 4225164 ], 
        [ 4225160, 4226183, 4225165, 4226188 ], 
        [ 4226184, 4228901, 4226189, 4228906 ], 
        [ 4228902, 4229490, 4228907, 4229494 ], 
        [ 4229491, 4229774, 4229495, 4229779 ], 
        [ 4229775, 4230559, 4229780, 4230564 ], 
        [ 4230560, 4231316, 4230565, 4231312 ], 
        [ 4231317, 4235572, 4231313, 4235577 ], 
        [ 4235573, 4242029, 4235578, 4242034 ], 
        [ 4242030, 4242765, 4242035, 4242770 ], 
        [ 4242766, 4242869, 4242771, 4242865 ], 
        [ 4242870, 4244054, 4242866, 4244059 ], 
        [ 4244055, 4245120, 4244060, 4245116 ], 
        [ 4245121, 4245334, 4245117, 4245339 ], 
        [ 4245335, 4245565, 4245340, 4245569 ], 
        [ 4245566, 4247741, 4245570, 4247745 ], 
        [ 4247742, 4249019, 4247746, 4249024 ], 
        [ 4249020, 4249100, 4249025, 4249105 ], 
        [ 4249101, 4256705, 4249106, 4256710 ], 
        [ 4256706, 4258303, 4256711, 4258307 ], 
        [ 4258304, 4259509, 4258308, 4259514 ], 
        [ 4259510, 4261410, 4259515, 4261415 ], 
        [ 4261411, 4269352, 4261416, 4269356 ], 
        [ 4269353, 4270525, 4269357, 4270530 ], 
        [ 4270526, 4274109, 4270531, 4274114 ], 
        [ 4274110, 4275137, 4274115, 4275141 ], 
        [ 4275138, 4281838, 4275142, 4281842 ], 
        [ 4281839, 4281849, 4281843, 4281854 ], 
        [ 4281850, 4283260, 4281855, 4283264 ], 
        [ 4283261, 4284055, 4283265, 4284059 ], 
        [ 4284056, 4284552, 4284060, 4284556 ], 
        [ 4284553, 4284762, 4284557, 4284767 ], 
        [ 4284763, 4287988, 4284768, 4287992 ], 
        [ 4287989, 4294545, 4287993, 4294549 ], 
        [ 4294546, 4303042, 4294550, 4303047 ], 
        [ 4303043, 4303258, 4303048, 4303263 ], 
        [ 4303259, 4303753, 4303264, 4303757 ], 
        [ 4303754, 4306632, 4303758, 4306637 ], 
        [ 4306633, 4310014, 4306638, 4310018 ], 
        [ 4310015, 4312216, 4310019, 4312221 ], 
        [ 4312217, 4314890, 4312222, 4314895 ], 
        [ 4314891, 4316460, 4314896, 4316465 ], 
        [ 4316461, 4316626, 4316466, 4316631 ], 
        [ 4316627, 4318092, 4316632, 4318097 ], 
        [ 4318093, 4318604, 4318098, 4318609 ], 
        [ 4318605, 4318772, 4318610, 4318777 ], 
        [ 4318773, 4321424, 4318778, 4321429 ], 
        [ 4321425, 4321489, 4321430, 4321494 ], 
        [ 4321490, 4324980, 4321495, 4324984 ], 
        [ 4324981, 4327535, 4324985, 4327540 ], 
        [ 4327536, 4329548, 4327541, 4329553 ], 
        [ 4329549, 4331509, 4329554, 4331514 ], 
        [ 4331510, 4332147, 4331515, 4332152 ], 
        [ 4332148, 4334445, 4332153, 4334450 ], 
        [ 4334446, 4338820, 4334451, 4338825 ], 
        [ 4338821, 4339739, 4338826, 4339744 ], 
        [ 4339740, 4343682, 4339745, 4343687 ], 
        [ 4343683, 4348264, 4343688, 4348269 ], 
        [ 4348265, 4351770, 4348270, 4351774 ], 
        [ 4351771, 4352206, 4351775, 4352211 ], 
        [ 4352207, 4356621, 4352212, 4356626 ], 
        [ 4356622, 4362927, 4356627, 4362923 ], 
        [ 4362928, 4366184, 4362924, 4366189 ], 
        [ 4366185, 4374134, 4366190, 4374139 ], 
        [ 4374135, 4374749, 4374140, 4374754 ], 
        [ 4374750, 4386564, 4374755, 4386568 ], 
        [ 4386565, 4386786, 4386569, 4386790 ], 
        [ 4386787, 4388049, 4386791, 4388054 ], 
        [ 4388050, 4388548, 4388055, 4388553 ], 
        [ 4388549, 4395145, 4388554, 4395141 ], 
        [ 4395146, 4395214, 4395142, 4395219 ], 
        [ 4395215, 4396301, 4395220, 4396297 ], 
        [ 4396302, 4397725, 4396298, 4397730 ], 
        [ 4397726, 4400829, 4397731, 4400833 ], 
        [ 4400830, 4401917, 4400834, 4401922 ], 
        [ 4401918, 4401962, 4401923, 4401967 ], 
        [ 4401963, 4404470, 4401968, 4404475 ], 
        [ 4404471, 4404603, 4404476, 4404608 ], 
        [ 4404604, 4404677, 4404609, 4404682 ], 
        [ 4404678, 4407845, 4404683, 4407850 ], 
        [ 4407846, 4408723, 4407851, 4408727 ], 
        [ 4408724, 4414102, 4408728, 4414106 ], 
        [ 4414103, 4415693, 4414107, 4415698 ], 
        [ 4415694, 4415957, 4415699, 4415962 ], 
        [ 4415958, 4418305, 4415963, 4418309 ], 
        [ 4418306, 4426334, 4418310, 4426339 ], 
        [ 4426335, 4427143, 4426340, 4427148 ], 
        [ 4427144, 4432989, 4427149, 4432994 ], 
        [ 4432990, 4433640, 4432995, 4433645 ], 
        [ 4433641, 4435236, 4433646, 4435241 ], 
        [ 4435237, 4436966, 4435242, 4436962 ], 
        [ 4436967, 4447677, 4436963, 4447681 ], 
        [ 4447678, 4449373, 4447682, 4449377 ], 
        [ 4449374, 4450981, 4449378, 4450986 ], 
        [ 4450982, 4452121, 4450987, 4452126 ], 
        [ 4452122, 4453471, 4452127, 4453475 ], 
        [ 4453472, 4454417, 4453476, 4454422 ], 
        [ 4454418, 4455164, 4454423, 4455169 ], 
        [ 4455165, 4459226, 4455170, 4459231 ], 
        [ 4459227, 4460531, 4459232, 4460535 ], 
        [ 4460532, 4462863, 4460536, 4462868 ], 
        [ 4462864, 4469714, 4462869, 4469719 ], 
        [ 4469715, 4469976, 4469720, 4469980 ], 
        [ 4469977, 4471008, 4469981, 4471013 ], 
        [ 4471009, 4473114, 4471014, 4473119 ], 
        [ 4473115, 4477852, 4473120, 4477857 ], 
        [ 4477853, 4477874, 4477858, 4477879 ], 
        [ 4477875, 4482921, 4477880, 4482926 ], 
        [ 4482922, 4489809, 4482927, 4489814 ], 
        [ 4489810, 4490912, 4489815, 4490917 ], 
        [ 4490913, 4491974, 4490918, 4491979 ], 
        [ 4491975, 4492157, 4491980, 4492162 ], 
        [ 4492158, 4493614, 4492163, 4493619 ], 
        [ 4493615, 4496829, 4493620, 4496834 ], 
        [ 4496830, 4497697, 4496835, 4497702 ], 
        [ 4497698, 4499157, 4497703, 4499162 ], 
        [ 4499158, 4502248, 4499163, 4502253 ], 
        [ 4502249, 4504493, 4502254, 4504498 ], 
        [ 4504494, 4505336, 4504499, 4505341 ], 
        [ 4505337, 4505567, 4505342, 4505571 ], 
        [ 4505568, 4506680, 4505572, 4506685 ], 
        [ 4506681, 4506961, 4506686, 4506966 ], 
        [ 4506962, 4507601, 4506967, 4507606 ], 
        [ 4507602, 4509484, 4507607, 4509488 ], 
        [ 4509485, 4509536, 4509489, 4509532 ], 
        [ 4509537, 4513351, 4509533, 4513356 ], 
        [ 4513352, 4516356, 4513357, 4516361 ], 
        [ 4516357, 4520650, 4516362, 4520655 ], 
        [ 4520651, 4522948, 4520656, 4522952 ], 
        [ 4522949, 4527592, 4522953, 4527596 ], 
        [ 4527593, 4528820, 4527597, 4528825 ], 
        [ 4528821, 4533216, 4528826, 4533212 ], 
        [ 4533217, 4535971, 4533213, 4535976 ], 
        [ 4535972, 4536320, 4535977, 4536324 ], 
        [ 4536321, 4539874, 4536325, 4539878 ], 
        [ 4539875, 4540172, 4539879, 4540177 ], 
        [ 4540173, 4541756, 4540178, 4541752 ], 
        [ 4541757, 4543862, 4541753, 4543858 ], 
        [ 4543863, 4545833, 4543859, 4545829 ], 
        [ 4545834, 4551230, 4545830, 4551235 ], 
        [ 4551231, 4552997, 4551236, 4553002 ], 
        [ 4552998, 4555491, 4553003, 4555496 ], 
        [ 4555492, 4557748, 4555497, 4557744 ], 
        [ 4557749, 4558033, 4557745, 4558038 ], 
        [ 4558034, 4561276, 4558039, 4561280 ], 
        [ 4561277, 4562123, 4561281, 4562128 ], 
        [ 4562124, 4562284, 4562129, 4562288 ], 
        [ 4562285, 4563100, 4562289, 4563105 ], 
        [ 4563101, 4564382, 4563106, 4564386 ], 
        [ 4564383, 4564639, 4564387, 4564644 ], 
        [ 4564640, 4566006, 4564645, 4566011 ], 
        [ 4566007, 4575457, 4566012, 4575462 ], 
        [ 4575458, 4575809, 4575463, 4575814 ], 
        [ 4575810, 4576253, 4575815, 4576258 ], 
        [ 4576254, 4579647, 4576259, 4579652 ], 
        [ 4579648, 4582407, 4579653, 4582411 ], 
        [ 4582408, 4588823, 4582412, 4588828 ], 
        [ 4588824, 4589254, 4588829, 4589259 ], 
        [ 4589255, 4589933, 4589260, 4589929 ], 
        [ 4589934, 4591334, 4589930, 4591338 ], 
        [ 4591335, 4598670, 4591339, 4598675 ], 
        [ 4598671, 4599152, 4598676, 4599156 ], 
        [ 4599153, 4600707, 4599157, 4600703 ], 
        [ 4600708, 4601105, 4600704, 4601110 ], 
        [ 4601106, 4601445, 4601111, 4601449 ], 
        [ 4601446, 4602566, 4601450, 4602571 ], 
        [ 4602567, 4606392, 4602572, 4606396 ], 
        [ 4606393, 4607895, 4606397, 4607899 ], 
        [ 4607896, 4612068, 4607900, 4612073 ], 
        [ 4612069, 4615603, 4612074, 4615608 ], 
        [ 4615604, 4618026, 4615609, 4618030 ], 
        [ 4618027, 4621061, 4618031, 4621065 ], 
        [ 4621062, 4627887, 4621066, 4627892 ], 
        [ 4627888, 4631394, 4627893, 4631399 ], 
        [ 4631395, 4631631, 4631400, 4631636 ], 
        [ 4631632, 4635963, 4631637, 4635968 ], 
        [ 4635964, 4641129, 4635969, 4641134 ], 
        [ 4641130, 4642188, 4641135, 4642192 ], 
        [ 4642189, 4642980, 4642193, 4642985 ], 
        [ 4642981, 4643635, 4642986, 4643640 ], 
        [ 4643636, 4644147, 4643641, 4644152 ], 
        [ 4644148, 4644814, 4644153, 4644818 ], 
        [ 4644815, 4649017, 4644819, 4649021 ], 
        [ 4649018, 4649332, 4649022, 4649337 ], 
        [ 4649333, 4649464, 4649338, 4649469 ], 
        [ 4649465, 4650584, 4649470, 4650588 ], 
        [ 4650585, 4653836, 4650589, 4653840 ], 
        [ 4653837, 4654913, 4653841, 4654917 ], 
        [ 4654914, 4656366, 4654918, 4656371 ], 
        [ 4656367, 4656864, 4656372, 4656869 ], 
        [ 4656865, 4656933, 4656870, 4656938 ], 
        [ 4656934, 4660056, 4656939, 4660061 ], 
        [ 4660057, 4665881, 4660062, 4665886 ], 
        [ 4665882, 4666341, 4665887, 4666345 ], 
        [ 4666342, 4668837, 4666346, 4668842 ], 
        [ 4668838, 4669896, 4668843, 4669900 ], 
        [ 4669897, 4670841, 4669901, 4670845 ], 
        [ 4670842, 4672873, 4670846, 4672878 ], 
        [ 4672874, 4681462, 4672879, 4681467 ], 
        [ 4681463, 4682181, 4681468, 4682185 ], 
        [ 4682182, 4691924, 4682186, 4691928 ], 
        [ 4691925, 4696368, 4691929, 4696373 ], 
        [ 4696369, 4699474, 4696374, 4699479 ], 
        [ 4699475, 4702530, 4699480, 4702534 ], 
        [ 4702531, 4704523, 4702535, 4704528 ], 
        [ 4704524, 4704899, 4704529, 4704903 ], 
        [ 4704900, 4706008, 4704904, 4706013 ], 
        [ 4706009, 4706120, 4706014, 4706124 ], 
        [ 4706121, 4706510, 4706125, 4706515 ], 
        [ 4706511, 4710334, 4706516, 4710330 ], 
        [ 4710335, 4710765, 4710331, 4710769 ], 
        [ 4710766, 4711295, 4710770, 4711300 ], 
        [ 4711296, 4711543, 4711301, 4711548 ], 
        [ 4711544, 4711935, 4711549, 4711940 ], 
        [ 4711936, 4712790, 4711941, 4712795 ], 
        [ 4712791, 4713126, 4712796, 4713131 ], 
        [ 4713127, 4713287, 4713132, 4713291 ], 
        [ 4713288, 4713730, 4713292, 4713735 ], 
        [ 4713731, 4717619, 4713736, 4717624 ], 
        [ 4717620, 4718789, 4717625, 4718785 ], 
        [ 4718790, 4722736, 4718786, 4722740 ], 
        [ 4722737, 4724224, 4722741, 4724229 ], 
        [ 4724225, 4725868, 4724230, 4725873 ], 
        [ 4725869, 4727653, 4725874, 4727658 ], 
        [ 4727654, 4729069, 4727659, 4729074 ], 
        [ 4729070, 4730833, 4729075, 4730838 ], 
        [ 4730834, 4733099, 4730839, 4733104 ], 
        [ 4733100, 4733576, 4733105, 4733581 ], 
        [ 4733577, 4735011, 4733582, 4735015 ], 
        [ 4735012, 4735924, 4735016, 4735928 ], 
        [ 4735925, 4736754, 4735929, 4736759 ], 
        [ 4736755, 4737511, 4736760, 4737507 ], 
        [ 4737512, 4737991, 4737508, 4737995 ], 
        [ 4737992, 4740307, 4737996, 4740311 ], 
        [ 4740308, 4741684, 4740312, 4741689 ], 
        [ 4741685, 4744830, 4741690, 4744835 ], 
        [ 4744831, 4746768, 4744836, 4746773 ], 
        [ 4746769, 4749037, 4746774, 4749042 ], 
        [ 4749038, 4749801, 4749043, 4749806 ], 
        [ 4749802, 4749864, 4749807, 4749869 ], 
        [ 4749865, 4750090, 4749870, 4750094 ], 
        [ 4750091, 4750966, 4750095, 4750971 ], 
        [ 4750967, 4751355, 4750972, 4751359 ], 
        [ 4751356, 4751493, 4751360, 4751497 ], 
        [ 4751494, 4752345, 4751498, 4752349 ], 
        [ 4752346, 4752965, 4752350, 4752970 ], 
        [ 4752966, 4753059, 4752971, 4753063 ], 
        [ 4753060, 4754115, 4753064, 4754119 ], 
        [ 4754116, 4754237, 4754120, 4754242 ], 
        [ 4754238, 4757191, 4754243, 4757196 ], 
        [ 4757192, 4761010, 4757197, 4761014 ], 
        [ 4761011, 4762052, 4761015, 4762057 ], 
        [ 4762053, 4762604, 4762058, 4762600 ], 
        [ 4762605, 4764164, 4762601, 4764169 ], 
        [ 4764165, 4766341, 4764170, 4766346 ], 
        [ 4766342, 4767519, 4766347, 4767524 ], 
        [ 4767520, 4769451, 4767525, 4769456 ], 
        [ 4769452, 4770366, 4769457, 4770371 ], 
        [ 4770367, 4774504, 4770372, 4774509 ], 
        [ 4774505, 4777167, 4774510, 4777171 ], 
        [ 4777168, 4778234, 4777172, 4778238 ], 
        [ 4778235, 4779310, 4778239, 4779315 ], 
        [ 4779311, 4784157, 4779316, 4784161 ], 
        [ 4784158, 4784713, 4784162, 4784718 ], 
        [ 4784714, 4784960, 4784719, 4784965 ], 
        [ 4784961, 4789181, 4784966, 4789186 ], 
        [ 4789182, 4792300, 4789187, 4792304 ], 
        [ 4792301, 4792894, 4792305, 4792899 ], 
        [ 4792895, 4804321, 4792900, 4804326 ], 
        [ 4804322, 4807780, 4804327, 4807785 ], 
        [ 4807781, 4808367, 4807786, 4808372 ], 
        [ 4808368, 4811025, 4808373, 4811030 ], 
        [ 4811026, 4811936, 4811031, 4811940 ], 
        [ 4811937, 4812883, 4811941, 4812887 ], 
        [ 4812884, 4813062, 4812888, 4813067 ], 
        [ 4813063, 4813117, 4813068, 4813121 ], 
        [ 4813118, 4822160, 4813122, 4822165 ], 
        [ 4822161, 4830957, 4822166, 4830961 ], 
        [ 4830958, 4831410, 4830962, 4831414 ], 
        [ 4831411, 4832326, 4831415, 4832330 ], 
        [ 4832327, 4833156, 4832331, 4833161 ], 
        [ 4833157, 4834005, 4833162, 4834001 ], 
        [ 4834006, 4834485, 4834002, 4834489 ], 
        [ 4834486, 4839621, 4834490, 4839626 ], 
        [ 4839622, 4844592, 4839627, 4844588 ], 
        [ 4844593, 4853316, 4844589, 4853321 ], 
        [ 4853317, 4853434, 4853322, 4853438 ], 
        [ 4853435, 4853704, 4853439, 4853708 ], 
        [ 4853705, 4862268, 4853709, 4862273 ], 
        [ 4862269, 4862689, 4862274, 4862694 ], 
        [ 4862690, 4863453, 4862695, 4863458 ], 
        [ 4863454, 4863657, 4863459, 4863662 ], 
        [ 4863658, 4865285, 4863663, 4865289 ], 
        [ 4865286, 4867215, 4865290, 4867220 ], 
        [ 4867216, 4867943, 4867221, 4867948 ], 
        [ 4867944, 4870367, 4867949, 4870372 ], 
        [ 4870368, 4871260, 4870373, 4871265 ], 
        [ 4871261, 4871925, 4871266, 4871930 ], 
        [ 4871926, 4872824, 4871931, 4872829 ], 
        [ 4872825, 4877259, 4872830, 4877263 ], 
        [ 4877260, 4877520, 4877264, 4877524 ], 
        [ 4877521, 4879935, 4877525, 4879940 ], 
        [ 4879936, 4880330, 4879941, 4880334 ], 
        [ 4880331, 4881593, 4880335, 4881598 ], 
        [ 4881594, 4882087, 4881599, 4882092 ], 
        [ 4882088, 4889351, 4882093, 4889356 ], 
        [ 4889352, 4890443, 4889357, 4890448 ], 
        [ 4890444, 4892812, 4890449, 4892816 ], 
        [ 4892813, 4893056, 4892817, 4893052 ], 
        [ 4893057, 4893088, 4893053, 4893092 ], 
        [ 4893089, 4897454, 4893093, 4897458 ], 
        [ 4897455, 4898485, 4897459, 4898490 ], 
        [ 4898486, 4901057, 4898491, 4901062 ], 
        [ 4901058, 4904245, 4901063, 4904250 ], 
        [ 4904246, 4904281, 4904251, 4904285 ], 
        [ 4904282, 4904668, 4904286, 4904673 ], 
        [ 4904669, 4904984, 4904674, 4904989 ], 
        [ 4904985, 4907736, 4904990, 4907740 ], 
        [ 4907737, 4908072, 4907741, 4908076 ], 
        [ 4908073, 4908096, 4908077, 4908100 ], 
        [ 4908097, 4908561, 4908101, 4908565 ], 
        [ 4908562, 4913327, 4908566, 4913323 ], 
        [ 4913328, 4914224, 4913324, 4914229 ], 
        [ 4914225, 4916537, 4914230, 4916542 ], 
        [ 4916538, 4918099, 4916543, 4918103 ], 
        [ 4918100, 4919908, 4918104, 4919913 ], 
        [ 4919909, 4926663, 4919914, 4926668 ], 
        [ 4926664, 4929329, 4926669, 4929334 ], 
        [ 4929330, 4929560, 4929335, 4929564 ], 
        [ 4929561, 4930673, 4929565, 4930678 ], 
        [ 4930674, 4930954, 4930679, 4930959 ], 
        [ 4930955, 4931594, 4930960, 4931599 ], 
        [ 4931595, 4932928, 4931600, 4932932 ], 
        [ 4932929, 4937282, 4932933, 4937287 ], 
        [ 4937283, 4939667, 4937288, 4939672 ], 
        [ 4939668, 4940551, 4939673, 4940555 ], 
        [ 4940552, 4940908, 4940556, 4940912 ], 
        [ 4940909, 4941837, 4940913, 4941842 ], 
        [ 4941838, 4946673, 4941843, 4946677 ], 
        [ 4946674, 4947030, 4946678, 4947035 ], 
        [ 4947031, 4948839, 4947036, 4948843 ], 
        [ 4948840, 4951071, 4948844, 4951076 ], 
        [ 4951072, 4953999, 4951077, 4954004 ], 
        [ 4954000, 4955481, 4954005, 4955486 ], 
        [ 4955482, 4959224, 4955487, 4959229 ], 
        [ 4959225, 4967229, 4959230, 4967233 ], 
        [ 4967230, 4972603, 4967234, 4972607 ], 
        [ 4972604, 4974094, 4972608, 4974098 ], 
        [ 4974095, 4974624, 4974099, 4974629 ], 
        [ 4974625, 4975683, 4974630, 4975687 ], 
        [ 4975684, 4976599, 4975688, 4976603 ], 
        [ 4976600, 4977429, 4976604, 4977434 ], 
        [ 4977430, 4978186, 4977435, 4978182 ], 
        [ 4978187, 4978666, 4978183, 4978670 ], 
        [ 4978667, 4984448, 4978671, 4984453 ], 
        [ 4984449, 4986670, 4984454, 4986675 ], 
        [ 4986671, 4987571, 4986676, 4987575 ], 
        [ 4987572, 4989614, 4987576, 4989618 ], 
        [ 4989615, 4990698, 4989619, 4990702 ], 
        [ 4990699, 4992038, 4990703, 4992043 ], 
        [ 4992039, 4993566, 4992044, 4993570 ], 
        [ 4993567, 4993811, 4993571, 4993816 ], 
        [ 4993812, 4994767, 4993817, 4994771 ], 
        [ 4994768, 4995631, 4994772, 4995636 ], 
        [ 4995632, 4996624, 4995637, 4996629 ], 
        [ 4996625, 4996668, 4996630, 4996673 ], 
        [ 4996669, 4997203, 4996674, 4997207 ], 
        [ 4997204, 4998818, 4997208, 4998823 ], 
        [ 4998819, 5004186, 4998824, 5004191 ], 
        [ 5004187, 5004432, 5004192, 5004436 ], 
        [ 5004433, 5008758, 5004437, 5008754 ], 
        [ 5008759, 5013598, 5008755, 5013603 ], 
        [ 5013599, 5016180, 5013604, 5016185 ], 
        [ 5016181, 5016267, 5016186, 5016271 ], 
        [ 5016268, 5017625, 5016272, 5017629 ], 
        [ 5017626, 5018455, 5017630, 5018460 ], 
        [ 5018456, 5019212, 5018461, 5019208 ], 
        [ 5019213, 5019692, 5019209, 5019696 ], 
        [ 5019693, 5026663, 5019697, 5026667 ], 
        [ 5026664, 5026770, 5026668, 5026775 ], 
        [ 5026771, 5028841, 5026776, 5028846 ], 
        [ 5028842, 5030486, 5028847, 5030490 ], 
        [ 5030487, 5031862, 5030491, 5031867 ], 
        [ 5031863, 5036331, 5031868, 5036336 ], 
        [ 5036332, 5037861, 5036337, 5037866 ], 
        [ 5037862, 5038887, 5037867, 5038892 ], 
        [ 5038888, 5040440, 5038893, 5040445 ], 
        [ 5040441, 5042902, 5040446, 5042907 ], 
        [ 5042903, 5044827, 5042908, 5044832 ], 
        [ 5044828, 5050524, 5044833, 5050529 ], 
        [ 5050525, 5053866, 5050530, 5053871 ], 
        [ 5053867, 5054707, 5053872, 5054712 ], 
        [ 5054708, 5055021, 5054713, 5055026 ], 
        [ 5055022, 5057873, 5055027, 5057878 ], 
        [ 5057874, 5059653, 5057879, 5059657 ], 
        [ 5059654, 5059734, 5059658, 5059739 ], 
        [ 5059735, 5061548, 5059740, 5061553 ], 
        [ 5061549, 5063342, 5061554, 5063347 ], 
        [ 5063343, 5064119, 5063348, 5064124 ], 
        [ 5064120, 5064638, 5064125, 5064643 ], 
        [ 5064639, 5066316, 5064644, 5066320 ], 
        [ 5066317, 5068774, 5066321, 5068779 ], 
        [ 5068775, 5069157, 5068780, 5069162 ], 
        [ 5069158, 5069375, 5069163, 5069380 ], 
        [ 5069376, 5071533, 5069381, 5071538 ], 
        [ 5071534, 5072259, 5071539, 5072264 ], 
        [ 5072260, 5072332, 5072265, 5072337 ], 
        [ 5072333, 5074288, 5072338, 5074293 ], 
        [ 5074289, 5078241, 5074294, 5078237 ], 
        [ 5078242, 5084929, 5078238, 5084933 ], 
        [ 5084930, 5087508, 5084934, 5087513 ], 
        [ 5087509, 5088409, 5087514, 5088414 ], 
        [ 5088410, 5093963, 5088415, 5093968 ], 
        [ 5093964, 5097754, 5093969, 5097750 ], 
        [ 5097755, 5098261, 5097751, 5098266 ], 
        [ 5098262, 5100144, 5098267, 5100148 ], 
        [ 5100145, 5102713, 5100149, 5102709 ], 
        [ 5102714, 5105308, 5102710, 5105312 ], 
        [ 5105309, 5110195, 5105313, 5110199 ], 
        [ 5110196, 5116037, 5110200, 5116042 ], 
        [ 5116038, 5116647, 5116043, 5116652 ], 
        [ 5116648, 5119282, 5116653, 5119287 ], 
        [ 5119283, 5121619, 5119288, 5121623 ], 
        [ 5121620, 5122889, 5121624, 5122893 ], 
        [ 5122890, 5125691, 5122894, 5125695 ], 
        [ 5125692, 5125943, 5125696, 5125947 ], 
        [ 5125944, 5132940, 5125948, 5132945 ], 
        [ 5132941, 5133068, 5132946, 5133072 ], 
        [ 5133069, 5133405, 5133073, 5133410 ], 
        [ 5133406, 5134558, 5133411, 5134563 ], 
        [ 5134559, 5138091, 5134564, 5138095 ], 
        [ 5138092, 5138432, 5138096, 5138437 ], 
        [ 5138433, 5138944, 5138438, 5138949 ], 
        [ 5138945, 5139157, 5138950, 5139162 ], 
        [ 5139158, 5139587, 5139163, 5139592 ], 
        [ 5139588, 5142617, 5139593, 5142622 ], 
        [ 5142618, 5148183, 5142623, 5148188 ], 
        [ 5148184, 5148672, 5148189, 5148677 ], 
        [ 5148673, 5150053, 5148678, 5150058 ], 
        [ 5150054, 5151087, 5150059, 5151092 ], 
        [ 5151088, 5153217, 5151093, 5153222 ], 
        [ 5153218, 5154383, 5153223, 5154388 ], 
        [ 5154384, 5154947, 5154389, 5154951 ], 
        [ 5154948, 5155016, 5154952, 5155021 ], 
        [ 5155017, 5156599, 5155022, 5156604 ], 
        [ 5156600, 5157802, 5156605, 5157807 ], 
        [ 5157803, 5157970, 5157808, 5157975 ], 
        [ 5157971, 5160625, 5157976, 5160630 ], 
        [ 5160626, 5162852, 5160631, 5162857 ], 
        [ 5162853, 5164824, 5162858, 5164829 ], 
        [ 5164825, 5171077, 5164830, 5171082 ], 
        [ 5171078, 5176566, 5171083, 5176570 ], 
        [ 5176567, 5180104, 5176571, 5180108 ], 
        [ 5180105, 5180591, 5180109, 5180596 ], 
        [ 5180592, 5184142, 5180597, 5184138 ], 
        [ 5184143, 5188235, 5184139, 5188240 ], 
        [ 5188236, 5190430, 5188241, 5190434 ], 
        [ 5190431, 5194013, 5190435, 5194018 ], 
        [ 5194014, 5194634, 5194019, 5194638 ], 
        [ 5194635, 5199338, 5194639, 5199342 ], 
        [ 5199339, 5200363, 5199343, 5200368 ], 
        [ 5200364, 5200380, 5200369, 5200385 ], 
        [ 5200381, 5207919, 5200386, 5207923 ], 
        [ 5207920, 5210484, 5207924, 5210480 ], 
        [ 5210485, 5211327, 5210481, 5211331 ], 
        [ 5211328, 5212227, 5211332, 5212231 ], 
        [ 5212228, 5212416, 5212232, 5212421 ], 
        [ 5212417, 5214189, 5212422, 5214194 ], 
        [ 5214190, 5218448, 5214195, 5218453 ], 
        [ 5218449, 5221514, 5218454, 5221519 ], 
        [ 5221515, 5222166, 5221520, 5222170 ], 
        [ 5222167, 5222235, 5222171, 5222239 ], 
        [ 5222236, 5222405, 5222240, 5222410 ], 
        [ 5222406, 5223121, 5222411, 5223126 ], 
        [ 5223122, 5225062, 5223127, 5225067 ], 
        [ 5225063, 5227034, 5225068, 5227039 ], 
        [ 5227035, 5230060, 5227040, 5230064 ], 
        [ 5230061, 5235911, 5230065, 5235915 ], 
        [ 5235912, 5237062, 5235916, 5237067 ], 
        [ 5237063, 5238549, 5237068, 5238554 ], 
        [ 5238550, 5239941, 5238555, 5239946 ], 
        [ 5239942, 5241160, 5239947, 5241165 ], 
        [ 5241161, 5243887, 5241166, 5243891 ], 
        [ 5243888, 5244364, 5243892, 5244368 ], 
        [ 5244365, 5245009, 5244369, 5245014 ], 
        [ 5245010, 5245420, 5245015, 5245425 ], 
        [ 5245421, 5246665, 5245426, 5246670 ], 
        [ 5246666, 5246882, 5246671, 5246887 ], 
        [ 5246883, 5252321, 5246888, 5252326 ], 
        [ 5252322, 5258113, 5252327, 5258117 ], 
        [ 5258114, 5259300, 5258118, 5259304 ], 
        [ 5259301, 5261182, 5259305, 5261187 ], 
        [ 5261183, 5261507, 5261188, 5261503 ], 
        [ 5261508, 5268860, 5261504, 5268864 ], 
        [ 5268861, 5269941, 5268865, 5269945 ], 
        [ 5269942, 5270945, 5269946, 5270949 ], 
        [ 5270946, 5271167, 5270950, 5271171 ], 
        [ 5271168, 5273111, 5271172, 5273116 ], 
        [ 5273112, 5273132, 5273117, 5273137 ], 
        [ 5273133, 5273680, 5273138, 5273685 ], 
        [ 5273681, 5274282, 5273686, 5274287 ], 
        [ 5274283, 5277485, 5274288, 5277490 ], 
        [ 5277486, 5278602, 5277491, 5278607 ], 
        [ 5278603, 5279312, 5278608, 5279316 ], 
        [ 5279313, 5279965, 5279317, 5279969 ], 
        [ 5279966, 5286668, 5279970, 5286673 ], 
        [ 5286669, 5288844, 5286674, 5288849 ], 
        [ 5288845, 5292505, 5288850, 5292509 ], 
        [ 5292506, 5295145, 5292510, 5295149 ], 
        [ 5295146, 5295426, 5295150, 5295431 ], 
        [ 5295427, 5296411, 5295432, 5296407 ], 
        [ 5296412, 5299331, 5296408, 5299336 ], 
        [ 5299332, 5299740, 5299337, 5299745 ], 
        [ 5299741, 5302074, 5299746, 5302079 ], 
        [ 5302075, 5304927, 5302080, 5304932 ], 
        [ 5304928, 5309929, 5304933, 5309925 ], 
        [ 5309930, 5311027, 5309926, 5311023 ], 
        [ 5311028, 5313142, 5311024, 5313138 ], 
        [ 5313143, 5313612, 5313139, 5313616 ], 
        [ 5313613, 5316196, 5313617, 5316201 ], 
        [ 5316197, 5320424, 5316202, 5320428 ], 
        [ 5320425, 5321884, 5320429, 5321889 ], 
        [ 5321885, 5327443, 5321890, 5327448 ], 
        [ 5327444, 5333009, 5327449, 5333014 ], 
        [ 5333010, 5335558, 5333015, 5335563 ], 
        [ 5335559, 5337934, 5335564, 5337939 ], 
        [ 5337935, 5340296, 5337940, 5340301 ], 
        [ 5340297, 5341322, 5340302, 5341326 ], 
        [ 5341323, 5342510, 5341327, 5342515 ], 
        [ 5342511, 5343834, 5342516, 5343839 ], 
        [ 5343835, 5351733, 5343840, 5351737 ], 
        [ 5351734, 5355838, 5351738, 5355842 ], 
        [ 5355839, 5359283, 5355843, 5359288 ], 
        [ 5359284, 5362430, 5359289, 5362434 ], 
        [ 5362431, 5362544, 5362435, 5362549 ], 
        [ 5362545, 5364468, 5362550, 5364472 ], 
        [ 5364469, 5369245, 5364473, 5369241 ], 
        [ 5369246, 5375729, 5369242, 5375734 ], 
        [ 5375730, 5375790, 5375735, 5375795 ], 
        [ 5375791, 5377253, 5375796, 5377258 ], 
        [ 5377254, 5378696, 5377259, 5378701 ], 
        [ 5378697, 5382060, 5378702, 5382065 ], 
        [ 5382061, 5388003, 5382066, 5388007 ], 
        [ 5388004, 5388224, 5388008, 5388229 ], 
        [ 5388225, 5389155, 5388230, 5389159 ], 
        [ 5389156, 5391633, 5389160, 5391638 ], 
        [ 5391634, 5399712, 5391639, 5399716 ], 
        [ 5399713, 5401417, 5399717, 5401422 ], 
        [ 5401418, 5406297, 5401423, 5406301 ], 
        [ 5406298, 5406537, 5406302, 5406542 ], 
        [ 5406538, 5406850, 5406543, 5406854 ], 
        [ 5406851, 5408637, 5406855, 5408642 ], 
        [ 5408638, 5410093, 5408643, 5410097 ], 
        [ 5410094, 5417270, 5410098, 5417275 ], 
        [ 5417271, 5419353, 5417276, 5419358 ], 
        [ 5419354, 5420144, 5419359, 5420149 ], 
        [ 5420145, 5420216, 5420150, 5420221 ], 
        [ 5420217, 5420740, 5420222, 5420745 ], 
        [ 5420741, 5423081, 5420746, 5423085 ], 
        [ 5423082, 5424075, 5423086, 5424079 ], 
        [ 5424076, 5425137, 5424080, 5425141 ], 
        [ 5425138, 5427477, 5425142, 5427482 ], 
        [ 5427478, 5429323, 5427483, 5429328 ], 
        [ 5429324, 5429631, 5429329, 5429635 ], 
        [ 5429632, 5432849, 5429636, 5432853 ], 
        [ 5432850, 5432874, 5432854, 5432878 ], 
        [ 5432875, 5440584, 5432879, 5440588 ], 
        [ 5440585, 5440833, 5440589, 5440837 ], 
        [ 5440834, 5441634, 5440838, 5441639 ], 
        [ 5441635, 5446602, 5441640, 5446606 ], 
        [ 5446603, 5448663, 5446607, 5448668 ], 
        [ 5448664, 5452231, 5448669, 5452236 ], 
        [ 5452232, 5453350, 5452237, 5453354 ], 
        [ 5453351, 5455164, 5453355, 5455168 ], 
        [ 5455165, 5458274, 5455169, 5458279 ], 
        [ 5458275, 5459524, 5458280, 5459529 ], 
        [ 5459525, 5468509, 5459530, 5468514 ], 
        [ 5468510, 5469773, 5468515, 5469778 ], 
        [ 5469774, 5474040, 5469779, 5474044 ], 
        [ 5474041, 5475379, 5474045, 5475384 ], 
        [ 5475380, 5476063, 5475385, 5476068 ], 
        [ 5476064, 5477860, 5476069, 5477865 ], 
        [ 5477861, 5478124, 5477866, 5478129 ], 
        [ 5478125, 5478577, 5478130, 5478582 ], 
        [ 5478578, 5479176, 5478583, 5479181 ], 
        [ 5479177, 5483012, 5479182, 5483017 ], 
        [ 5483013, 5483809, 5483018, 5483814 ], 
        [ 5483810, 5490967, 5483815, 5490963 ], 
        [ 5490968, 5491739, 5490964, 5491743 ], 
        [ 5491740, 5495234, 5491744, 5495239 ], 
        [ 5495235, 5498449, 5495240, 5498449 ]
      ]

    def _cut(seq)
      cuts = Bio::RestrictionEnzyme::Analysis.cut(seq,
                                                  "SacI", "EcoRI", "BstEII",
                                                  {:view_ranges => true})
    end
  end #TestEcoliO157H7_3enzymes


end #module TestRestrictionEnzymeAnalysisCutLong
