/* Copyright (c) 2018, 2019 Evan Nemerson <evan@nemerson.com>
 * Copyright (c) 2020 Christopher Moore <moore@free.fr>
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use, copy,
 * modify, merge, publish, distribute, sublicense, and/or sell copies
 * of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#define SIMDE_TESTS_CURRENT_ISAX avx512f
#include <test/x86/test-avx512.h>
#include <simde/x86/avx512f.h>

static int
test_simde_mm512_abs_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(-1095158286), INT32_C( -133595553), INT32_C( -941949577), INT32_C(-1117722052),
                            INT32_C(-1053667317), INT32_C( -662420643), INT32_C( 2095193825), INT32_C( -799061081),
                            INT32_C(  347912513), INT32_C( -439299809), INT32_C( 2053030698), INT32_C( -277514113),
                            INT32_C( 1476262970), INT32_C( 1955038119), INT32_C(  -77085072), INT32_C( 1014069144)),
      simde_mm512_set_epi32(INT32_C( 1095158286), INT32_C(  133595553), INT32_C(  941949577), INT32_C( 1117722052),
                            INT32_C( 1053667317), INT32_C(  662420643), INT32_C( 2095193825), INT32_C(  799061081),
                            INT32_C(  347912513), INT32_C(  439299809), INT32_C( 2053030698), INT32_C(  277514113),
                            INT32_C( 1476262970), INT32_C( 1955038119), INT32_C(   77085072), INT32_C( 1014069144)) },
    { simde_mm512_set_epi32(INT32_C( 1865049709), INT32_C( -408997463), INT32_C( 1771073477), INT32_C( 1463780468),
                            INT32_C(  268019741), INT32_C(-1725054429), INT32_C( 1408597864), INT32_C( 1576985133),
                            INT32_C(  170783936), INT32_C(  836522882), INT32_C( 1364040350), INT32_C(  563663058),
                            INT32_C(-1491438903), INT32_C( -873504608), INT32_C( 1431273511), INT32_C( -164765086)),
      simde_mm512_set_epi32(INT32_C( 1865049709), INT32_C(  408997463), INT32_C( 1771073477), INT32_C( 1463780468),
                            INT32_C(  268019741), INT32_C( 1725054429), INT32_C( 1408597864), INT32_C( 1576985133),
                            INT32_C(  170783936), INT32_C(  836522882), INT32_C( 1364040350), INT32_C(  563663058),
                            INT32_C( 1491438903), INT32_C(  873504608), INT32_C( 1431273511), INT32_C(  164765086)) },
    { simde_mm512_set_epi32(INT32_C( 1505063340), INT32_C(  -79208486), INT32_C( -115790145), INT32_C( 1137793635),
                            INT32_C( -719063760), INT32_C( -465633360), INT32_C( 1417132608), INT32_C( 1715322300),
                            INT32_C( 1194443989), INT32_C( 1598244723), INT32_C( -360509626), INT32_C( -844528776),
                            INT32_C( -291907566), INT32_C( -980752736), INT32_C(  701363552), INT32_C( 1148036152)),
      simde_mm512_set_epi32(INT32_C( 1505063340), INT32_C(   79208486), INT32_C(  115790145), INT32_C( 1137793635),
                            INT32_C(  719063760), INT32_C(  465633360), INT32_C( 1417132608), INT32_C( 1715322300),
                            INT32_C( 1194443989), INT32_C( 1598244723), INT32_C(  360509626), INT32_C(  844528776),
                            INT32_C(  291907566), INT32_C(  980752736), INT32_C(  701363552), INT32_C( 1148036152)) },
    { simde_mm512_set_epi32(INT32_C(-1538804784), INT32_C(  -43683957), INT32_C(  -70380459), INT32_C(  259050545),
                            INT32_C(-1140217223), INT32_C(  -24242506), INT32_C(-1281378925), INT32_C( -426768587),
                            INT32_C(-1825251144), INT32_C( -975195895), INT32_C(  758020113), INT32_C(   -3401471),
                            INT32_C(  154668063), INT32_C( -827616009), INT32_C(  793625070), INT32_C( -735990247)),
      simde_mm512_set_epi32(INT32_C( 1538804784), INT32_C(   43683957), INT32_C(   70380459), INT32_C(  259050545),
                            INT32_C( 1140217223), INT32_C(   24242506), INT32_C( 1281378925), INT32_C(  426768587),
                            INT32_C( 1825251144), INT32_C(  975195895), INT32_C(  758020113), INT32_C(    3401471),
                            INT32_C(  154668063), INT32_C(  827616009), INT32_C(  793625070), INT32_C(  735990247)) },
    { simde_mm512_set_epi32(INT32_C( -919197120), INT32_C( 1902742720), INT32_C(  576001152), INT32_C(  772608991),
                            INT32_C( 1373611304), INT32_C(  156079462), INT32_C(  392030686), INT32_C( 1159450969),
                            INT32_C( 1376625025), INT32_C( -701917672), INT32_C( 1911493359), INT32_C( -115817480),
                            INT32_C( -875216623), INT32_C( 1333681477), INT32_C(-1067533891), INT32_C( 1671330781)),
      simde_mm512_set_epi32(INT32_C(  919197120), INT32_C( 1902742720), INT32_C(  576001152), INT32_C(  772608991),
                            INT32_C( 1373611304), INT32_C(  156079462), INT32_C(  392030686), INT32_C( 1159450969),
                            INT32_C( 1376625025), INT32_C(  701917672), INT32_C( 1911493359), INT32_C(  115817480),
                            INT32_C(  875216623), INT32_C( 1333681477), INT32_C( 1067533891), INT32_C( 1671330781)) },
    { simde_mm512_set_epi32(INT32_C(-1168385947), INT32_C(-1671882855), INT32_C(-1182456995), INT32_C(-1803534861),
                            INT32_C(  443878759), INT32_C(  702169153), INT32_C(-1879742181), INT32_C( 1627978919),
                            INT32_C(  583873330), INT32_C( -857098109), INT32_C(  710347808), INT32_C( 1707849385),
                            INT32_C( 1863512780), INT32_C( -371421167), INT32_C( 1902179408), INT32_C(-1189025654)),
      simde_mm512_set_epi32(INT32_C( 1168385947), INT32_C( 1671882855), INT32_C( 1182456995), INT32_C( 1803534861),
                            INT32_C(  443878759), INT32_C(  702169153), INT32_C( 1879742181), INT32_C( 1627978919),
                            INT32_C(  583873330), INT32_C(  857098109), INT32_C(  710347808), INT32_C( 1707849385),
                            INT32_C( 1863512780), INT32_C(  371421167), INT32_C( 1902179408), INT32_C( 1189025654)) },
    { simde_mm512_set_epi32(INT32_C(    7990856), INT32_C(-1991291137), INT32_C( 1404443548), INT32_C(-1023849862),
                            INT32_C( 2054941409), INT32_C(-1604088325), INT32_C(  721271909), INT32_C(-1622295089),
                            INT32_C( 1869222605), INT32_C(-1583998423), INT32_C( -801626928), INT32_C( -940395766),
                            INT32_C( 1108931720), INT32_C( -471669445), INT32_C( 1204289475), INT32_C( -752679106)),
      simde_mm512_set_epi32(INT32_C(    7990856), INT32_C( 1991291137), INT32_C( 1404443548), INT32_C( 1023849862),
                            INT32_C( 2054941409), INT32_C( 1604088325), INT32_C(  721271909), INT32_C( 1622295089),
                            INT32_C( 1869222605), INT32_C( 1583998423), INT32_C(  801626928), INT32_C(  940395766),
                            INT32_C( 1108931720), INT32_C(  471669445), INT32_C( 1204289475), INT32_C(  752679106)) },
    { simde_mm512_set_epi32(INT32_C( 1399806844), INT32_C( 1131841699), INT32_C( -346937782), INT32_C(  567816154),
                            INT32_C(-1589012616), INT32_C(-2005496894), INT32_C( 1401681986), INT32_C(  423760716),
                            INT32_C(  431684101), INT32_C(  852583616), INT32_C(-1369299290), INT32_C( -663899319),
                            INT32_C( 1580470265), INT32_C(  298083241), INT32_C( -630373638), INT32_C(-1937828661)),
      simde_mm512_set_epi32(INT32_C( 1399806844), INT32_C( 1131841699), INT32_C(  346937782), INT32_C(  567816154),
                            INT32_C( 1589012616), INT32_C( 2005496894), INT32_C( 1401681986), INT32_C(  423760716),
                            INT32_C(  431684101), INT32_C(  852583616), INT32_C( 1369299290), INT32_C(  663899319),
                            INT32_C( 1580470265), INT32_C(  298083241), INT32_C(  630373638), INT32_C( 1937828661)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_abs_epi32(test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_abs_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(  114710097), INT32_C( 1837246098), INT32_C(-1399577225), INT32_C(-1388127606),
                            INT32_C( 1116027725), INT32_C( -871797325), INT32_C(-1979326643), INT32_C( 1477004857),
                            INT32_C( 1670723749), INT32_C(-1006052339), INT32_C( 1863789116), INT32_C( -690396684),
                            INT32_C( -629773535), INT32_C(  667046523), INT32_C( 1317445565), INT32_C( -732937024)),
      UINT16_C(28888),
      simde_mm512_set_epi32(INT32_C(-1877548571), INT32_C( -399920351), INT32_C(   15525797), INT32_C(   18165921),
                            INT32_C( 2085930596), INT32_C( 1662282658), INT32_C(-1842752263), INT32_C(  499820912),
                            INT32_C( 1419797765), INT32_C(  -44818966), INT32_C( 1761152620), INT32_C(-1288657930),
                            INT32_C(  894078020), INT32_C(-1369431563), INT32_C( -103362440), INT32_C(-1559726025)),
      simde_mm512_set_epi32(INT32_C(  114710097), INT32_C(  399920351), INT32_C(   15525797), INT32_C(   18165921),
                            INT32_C( 1116027725), INT32_C( -871797325), INT32_C(-1979326643), INT32_C( 1477004857),
                            INT32_C( 1419797765), INT32_C(   44818966), INT32_C( 1863789116), INT32_C( 1288657930),
                            INT32_C(  894078020), INT32_C(  667046523), INT32_C( 1317445565), INT32_C( -732937024)) },
    { simde_mm512_set_epi32(INT32_C( 1505436737), INT32_C(  342253548), INT32_C( 1435179252), INT32_C( 1326766533),
                            INT32_C(  156769011), INT32_C(  343760696), INT32_C(  611303965), INT32_C( 1457341409),
                            INT32_C(  165452421), INT32_C(-1824090116), INT32_C( -184738383), INT32_C( -191086464),
                            INT32_C( -117301127), INT32_C(-1526262537), INT32_C( -208138847), INT32_C(  807348777)),
      UINT16_C(61134),
      simde_mm512_set_epi32(INT32_C( 1427056174), INT32_C( 2097896620), INT32_C( 1813263538), INT32_C( 1909821993),
                            INT32_C( 1439822042), INT32_C(-1049213292), INT32_C( 1557133349), INT32_C( 1168931268),
                            INT32_C( -810546774), INT32_C(-1283013132), INT32_C(  654302587), INT32_C(  314275905),
                            INT32_C(-1091094079), INT32_C( -114174508), INT32_C(  407580338), INT32_C( 1906809805)),
      simde_mm512_set_epi32(INT32_C( 1427056174), INT32_C( 2097896620), INT32_C( 1813263538), INT32_C( 1326766533),
                            INT32_C( 1439822042), INT32_C( 1049213292), INT32_C( 1557133349), INT32_C( 1457341409),
                            INT32_C(  810546774), INT32_C( 1283013132), INT32_C( -184738383), INT32_C( -191086464),
                            INT32_C( 1091094079), INT32_C(  114174508), INT32_C(  407580338), INT32_C(  807348777)) },
    { simde_mm512_set_epi32(INT32_C(  905172649), INT32_C(-1044778809), INT32_C(-1938215986), INT32_C(-1138753169),
                            INT32_C(-1689961651), INT32_C(  890456168), INT32_C( 1382435241), INT32_C( -803845344),
                            INT32_C(  430838507), INT32_C( 1075259040), INT32_C(-1956785379), INT32_C(-1586468297),
                            INT32_C(  622055688), INT32_C(-1127740382), INT32_C(  466514910), INT32_C(-1745879628)),
      UINT16_C(30570),
      simde_mm512_set_epi32(INT32_C( -310045086), INT32_C(  560822999), INT32_C( -680371476), INT32_C( 1838395052),
                            INT32_C(-1152635838), INT32_C( -481448106), INT32_C(  871399876), INT32_C( -939960538),
                            INT32_C( -898000986), INT32_C( -641497176), INT32_C(  657638908), INT32_C(-1796735419),
                            INT32_C(-1032150818), INT32_C(  151713087), INT32_C( 1554707006), INT32_C( -318690470)),
      simde_mm512_set_epi32(INT32_C(  905172649), INT32_C(  560822999), INT32_C(  680371476), INT32_C( 1838395052),
                            INT32_C(-1689961651), INT32_C(  481448106), INT32_C(  871399876), INT32_C(  939960538),
                            INT32_C(  430838507), INT32_C(  641497176), INT32_C(  657638908), INT32_C(-1586468297),
                            INT32_C( 1032150818), INT32_C(-1127740382), INT32_C( 1554707006), INT32_C(-1745879628)) },
    { simde_mm512_set_epi32(INT32_C(-1675700291), INT32_C(  -85412591), INT32_C(-1865493216), INT32_C(-1122257925),
                            INT32_C(  955620837), INT32_C( -725693586), INT32_C( 1056307491), INT32_C( 1924019839),
                            INT32_C(-2012466116), INT32_C(-1808881746), INT32_C( -887453452), INT32_C(  160221724),
                            INT32_C( -886018282), INT32_C( 1222780200), INT32_C( 1877396684), INT32_C(  283360472)),
      UINT16_C(28339),
      simde_mm512_set_epi32(INT32_C(-1238615237), INT32_C(  583893938), INT32_C( -594441984), INT32_C( 1561597956),
                            INT32_C(  174377227), INT32_C(  319460903), INT32_C(-1295208114), INT32_C(  659707887),
                            INT32_C( 1117898731), INT32_C( -209622907), INT32_C(-1431480123), INT32_C(-2058827609),
                            INT32_C(-1519596795), INT32_C(   24332922), INT32_C( -338106630), INT32_C(-1565374776)),
      simde_mm512_set_epi32(INT32_C(-1675700291), INT32_C(  583893938), INT32_C(  594441984), INT32_C(-1122257925),
                            INT32_C(  174377227), INT32_C(  319460903), INT32_C( 1295208114), INT32_C( 1924019839),
                            INT32_C( 1117898731), INT32_C(-1808881746), INT32_C( 1431480123), INT32_C( 2058827609),
                            INT32_C( -886018282), INT32_C( 1222780200), INT32_C(  338106630), INT32_C( 1565374776)) },
    { simde_mm512_set_epi32(INT32_C(  178377352), INT32_C( -324510384), INT32_C(  446946466), INT32_C(-1323398690),
                            INT32_C( -720979875), INT32_C( -512216094), INT32_C( 1145272930), INT32_C( -706074883),
                            INT32_C(-1863795060), INT32_C( -525595897), INT32_C( 1357119557), INT32_C(  837734387),
                            INT32_C( -607392699), INT32_C( -498581669), INT32_C(-2108693629), INT32_C( -476969927)),
      UINT16_C(42507),
      simde_mm512_set_epi32(INT32_C(   -5472621), INT32_C( -263868960), INT32_C(-1867831731), INT32_C(  955254216),
                            INT32_C( 1990179011), INT32_C(-1729740457), INT32_C( 1711933869), INT32_C(-1566075058),
                            INT32_C( -550106516), INT32_C(-1087591249), INT32_C(  919917002), INT32_C(-1410389997),
                            INT32_C( -188117230), INT32_C( 1025569327), INT32_C(-1456210246), INT32_C( -254945819)),
      simde_mm512_set_epi32(INT32_C(    5472621), INT32_C( -324510384), INT32_C( 1867831731), INT32_C(-1323398690),
                            INT32_C( -720979875), INT32_C( 1729740457), INT32_C( 1711933869), INT32_C( -706074883),
                            INT32_C(-1863795060), INT32_C( -525595897), INT32_C( 1357119557), INT32_C(  837734387),
                            INT32_C(  188117230), INT32_C( -498581669), INT32_C( 1456210246), INT32_C(  254945819)) },
    { simde_mm512_set_epi32(INT32_C(-1007934437), INT32_C(  201253136), INT32_C( 2123754123), INT32_C( 1034305262),
                            INT32_C( 2139323878), INT32_C( -545410429), INT32_C(-1549231865), INT32_C( 1779895500),
                            INT32_C( 1932853973), INT32_C( 2135732954), INT32_C( 1232725518), INT32_C(  339564914),
                            INT32_C( -113030707), INT32_C(-1715459937), INT32_C( -492435091), INT32_C(-1720946495)),
      UINT16_C(49758),
      simde_mm512_set_epi32(INT32_C(  348473993), INT32_C(-1624874318), INT32_C(  361690252), INT32_C(  165927413),
                            INT32_C(-1864332117), INT32_C( -524477604), INT32_C(  481484649), INT32_C(-1499715490),
                            INT32_C(-1683117466), INT32_C(-2055457330), INT32_C( -850617531), INT32_C(-2081246973),
                            INT32_C( 1276057415), INT32_C( 1619064589), INT32_C(-1536816688), INT32_C( 2060578085)),
      simde_mm512_set_epi32(INT32_C(  348473993), INT32_C( 1624874318), INT32_C( 2123754123), INT32_C( 1034305262),
                            INT32_C( 2139323878), INT32_C( -545410429), INT32_C(  481484649), INT32_C( 1779895500),
                            INT32_C( 1932853973), INT32_C( 2055457330), INT32_C( 1232725518), INT32_C( 2081246973),
                            INT32_C( 1276057415), INT32_C( 1619064589), INT32_C( 1536816688), INT32_C(-1720946495)) },
    { simde_mm512_set_epi32(INT32_C(  860828042), INT32_C( 1459856596), INT32_C(-1901530659), INT32_C( 1296141157),
                            INT32_C(  778663095), INT32_C(-1872048536), INT32_C(-1115787645), INT32_C(-1142406643),
                            INT32_C( 1518955242), INT32_C( -174688543), INT32_C( 1537062129), INT32_C( -974095643),
                            INT32_C(  125816377), INT32_C(-1032428044), INT32_C( -374455538), INT32_C( -648832583)),
      UINT16_C(41340),
      simde_mm512_set_epi32(INT32_C( 1553986008), INT32_C( -808715903), INT32_C(-2114331727), INT32_C(  878797396),
                            INT32_C( 1547560130), INT32_C( -931453209), INT32_C(  639671594), INT32_C(  734358771),
                            INT32_C(-1802430748), INT32_C(   38083245), INT32_C(  636500349), INT32_C( 2020438947),
                            INT32_C(   89083218), INT32_C( 2041918986), INT32_C(-2068453500), INT32_C( 1772569863)),
      simde_mm512_set_epi32(INT32_C( 1553986008), INT32_C( 1459856596), INT32_C( 2114331727), INT32_C( 1296141157),
                            INT32_C(  778663095), INT32_C(-1872048536), INT32_C(-1115787645), INT32_C(  734358771),
                            INT32_C( 1518955242), INT32_C(   38083245), INT32_C(  636500349), INT32_C( 2020438947),
                            INT32_C(   89083218), INT32_C( 2041918986), INT32_C( -374455538), INT32_C( -648832583)) },
    { simde_mm512_set_epi32(INT32_C(-1208548961), INT32_C( 1705109710), INT32_C( -159097588), INT32_C( -879037423),
                            INT32_C( 2121552533), INT32_C(  595529007), INT32_C( -405863552), INT32_C( 1431630584),
                            INT32_C( -616000216), INT32_C(  444327364), INT32_C(  613413664), INT32_C(-2128463203),
                            INT32_C(  939927077), INT32_C(-1255659348), INT32_C(-1631544337), INT32_C(-1727626838)),
      UINT16_C(49163),
      simde_mm512_set_epi32(INT32_C(  895846723), INT32_C(  449272422), INT32_C( 1127330699), INT32_C(-1084895433),
                            INT32_C( -399265722), INT32_C(  697840482), INT32_C( -598276089), INT32_C(  -50403840),
                            INT32_C( 1970006978), INT32_C( 1602141812), INT32_C(-1773480652), INT32_C(  740913018),
                            INT32_C( 1668822994), INT32_C(  698152405), INT32_C( 1772335922), INT32_C(  847772835)),
      simde_mm512_set_epi32(INT32_C(  895846723), INT32_C(  449272422), INT32_C( -159097588), INT32_C( -879037423),
                            INT32_C( 2121552533), INT32_C(  595529007), INT32_C( -405863552), INT32_C( 1431630584),
                            INT32_C( -616000216), INT32_C(  444327364), INT32_C(  613413664), INT32_C(-2128463203),
                            INT32_C( 1668822994), INT32_C(-1255659348), INT32_C( 1772335922), INT32_C(  847772835)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_abs_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_abs_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(17600),
      simde_mm512_set_epi32(INT32_C(  393115914), INT32_C(   -9604904), INT32_C(  114710097), INT32_C( 1837246098),
                            INT32_C(-1399577225), INT32_C(-1388127606), INT32_C( 1116027725), INT32_C( -871797325),
                            INT32_C(-1979326643), INT32_C( 1477004857), INT32_C( 1670723749), INT32_C(-1006052339),
                            INT32_C( 1863789116), INT32_C( -690396684), INT32_C( -629773535), INT32_C(  667046523)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(    9604904), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1388127606), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1979326643), INT32_C( 1477004857), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(30775),
      simde_mm512_set_epi32(INT32_C( -208138847), INT32_C(  807348777), INT32_C(-1877548571), INT32_C( -399920351),
                            INT32_C(   15525797), INT32_C(   18165921), INT32_C( 2085930596), INT32_C( 1662282658),
                            INT32_C(-1842752263), INT32_C(  499820912), INT32_C( 1419797765), INT32_C(  -44818966),
                            INT32_C( 1761152620), INT32_C(-1288657930), INT32_C(  894078020), INT32_C(-1369431563)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  807348777), INT32_C( 1877548571), INT32_C(  399920351),
                            INT32_C(   15525797), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( 1419797765), INT32_C(   44818966),
                            INT32_C(          0), INT32_C( 1288657930), INT32_C(  894078020), INT32_C( 1369431563)) },
    { UINT16_C( 5367),
      simde_mm512_set_epi32(INT32_C(  407580338), INT32_C( 1906809805), INT32_C( -849801752), INT32_C(-1965822258),
                            INT32_C( 1505436737), INT32_C(  342253548), INT32_C( 1435179252), INT32_C( 1326766533),
                            INT32_C(  156769011), INT32_C(  343760696), INT32_C(  611303965), INT32_C( 1457341409),
                            INT32_C(  165452421), INT32_C(-1824090116), INT32_C( -184738383), INT32_C( -191086464)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C( 1965822258),
                            INT32_C(          0), INT32_C(  342253548), INT32_C(          0), INT32_C(          0),
                            INT32_C(  156769011), INT32_C(  343760696), INT32_C(  611303965), INT32_C( 1457341409),
                            INT32_C(          0), INT32_C( 1824090116), INT32_C(  184738383), INT32_C(  191086464)) },
    { UINT16_C(54740),
      simde_mm512_set_epi32(INT32_C(  622055688), INT32_C(-1127740382), INT32_C(  466514910), INT32_C(-1745879628),
                            INT32_C( 1427056174), INT32_C( 2097896620), INT32_C( 1813263538), INT32_C( 1909821993),
                            INT32_C( 1439822042), INT32_C(-1049213292), INT32_C( 1557133349), INT32_C( 1168931268),
                            INT32_C( -810546774), INT32_C(-1283013132), INT32_C(  654302587), INT32_C(  314275905)),
      simde_mm512_set_epi32(INT32_C(  622055688), INT32_C( 1127740382), INT32_C(          0), INT32_C( 1745879628),
                            INT32_C(          0), INT32_C( 2097896620), INT32_C(          0), INT32_C( 1909821993),
                            INT32_C( 1439822042), INT32_C( 1049213292), INT32_C(          0), INT32_C( 1168931268),
                            INT32_C(          0), INT32_C( 1283013132), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(27191),
      simde_mm512_set_epi32(INT32_C(-1032150818), INT32_C(  151713087), INT32_C( 1554707006), INT32_C( -318690470),
                            INT32_C(  788893537), INT32_C( -230394006), INT32_C(  905172649), INT32_C(-1044778809),
                            INT32_C(-1938215986), INT32_C(-1138753169), INT32_C(-1689961651), INT32_C(  890456168),
                            INT32_C( 1382435241), INT32_C( -803845344), INT32_C(  430838507), INT32_C( 1075259040)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  151713087), INT32_C( 1554707006), INT32_C(          0),
                            INT32_C(  788893537), INT32_C(          0), INT32_C(  905172649), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( 1689961651), INT32_C(  890456168),
                            INT32_C(          0), INT32_C(  803845344), INT32_C(  430838507), INT32_C( 1075259040)) },
    { UINT16_C(65093),
      simde_mm512_set_epi32(INT32_C( -887453452), INT32_C(  160221724), INT32_C( -886018282), INT32_C( 1222780200),
                            INT32_C( 1877396684), INT32_C(  283360472), INT32_C( -310045086), INT32_C(  560822999),
                            INT32_C( -680371476), INT32_C( 1838395052), INT32_C(-1152635838), INT32_C( -481448106),
                            INT32_C(  871399876), INT32_C( -939960538), INT32_C( -898000986), INT32_C( -641497176)),
      simde_mm512_set_epi32(INT32_C(  887453452), INT32_C(  160221724), INT32_C(  886018282), INT32_C( 1222780200),
                            INT32_C( 1877396684), INT32_C(  283360472), INT32_C(  310045086), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1838395052), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(  939960538), INT32_C(          0), INT32_C(  641497176)) },
    { UINT16_C(42926),
      simde_mm512_set_epi32(INT32_C(-1431480123), INT32_C(-2058827609), INT32_C(-1519596795), INT32_C(   24332922),
                            INT32_C( -338106630), INT32_C(-1565374776), INT32_C(-1426452996), INT32_C( -680300877),
                            INT32_C(-1675700291), INT32_C(  -85412591), INT32_C(-1865493216), INT32_C(-1122257925),
                            INT32_C(  955620837), INT32_C( -725693586), INT32_C( 1056307491), INT32_C( 1924019839)),
      simde_mm512_set_epi32(INT32_C( 1431480123), INT32_C(          0), INT32_C( 1519596795), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1565374776), INT32_C( 1426452996), INT32_C(  680300877),
                            INT32_C( 1675700291), INT32_C(          0), INT32_C( 1865493216), INT32_C(          0),
                            INT32_C(  955620837), INT32_C(  725693586), INT32_C( 1056307491), INT32_C(          0)) },
    { UINT16_C(26757),
      simde_mm512_set_epi32(INT32_C(-1863795060), INT32_C( -525595897), INT32_C( 1357119557), INT32_C(  837734387),
                            INT32_C( -607392699), INT32_C( -498581669), INT32_C(-2108693629), INT32_C( -476969927),
                            INT32_C(-1238615237), INT32_C(  583893938), INT32_C( -594441984), INT32_C( 1561597956),
                            INT32_C(  174377227), INT32_C(  319460903), INT32_C(-1295208114), INT32_C(  659707887)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  525595897), INT32_C( 1357119557), INT32_C(          0),
                            INT32_C(  607392699), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1238615237), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(  319460903), INT32_C(          0), INT32_C(  659707887)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_abs_epi32(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_abs_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-4703669018152042913), INT64_C(-4045642624518788548),
                            INT64_C(-4525466663746518179), INT64_C( 8998788960652053415),
                            INT64_C( 1494272869059842335), INT64_C( 8817699709611505791),
                            INT64_C( 6340501178400867239), INT64_C( -331077862235736168)),
      simde_mm512_set_epi64(INT64_C( 4703669018152042913), INT64_C( 4045642624518788548),
                            INT64_C( 4525466663746518179), INT64_C( 8998788960652053415),
                            INT64_C( 1494272869059842335), INT64_C( 8817699709611505791),
                            INT64_C( 6340501178400867239), INT64_C(  331077862235736168)) },
    { simde_mm512_set_epi64(INT64_C( 8010327509455286697), INT64_C( 7606702663991788660),
                            INT64_C( 1151136024847303203), INT64_C( 6049881760672440877),
                            INT64_C(  733511420638679938), INT64_C( 5858508694238056658),
                            INT64_C(-6405681308945653600), INT64_C( 6147272925506298466)),
      simde_mm512_set_epi64(INT64_C( 8010327509455286697), INT64_C( 7606702663991788660),
                            INT64_C( 1151136024847303203), INT64_C( 6049881760672440877),
                            INT64_C(  733511420638679938), INT64_C( 5858508694238056658),
                            INT64_C( 6405681308945653600), INT64_C( 6147272925506298466)) },
    { simde_mm512_set_epi64(INT64_C( 6464197827924287450), INT64_C( -497314884836304285),
                            INT64_C(-3088355329109459024), INT64_C( 6086538207170510268),
                            INT64_C( 5130097871257028467), INT64_C(-1548377050112752776),
                            INT64_C(-1253733446110746976), INT64_C( 3012333519594431544)),
      simde_mm512_set_epi64(INT64_C( 6464197827924287450), INT64_C(  497314884836304285),
                            INT64_C( 3088355329109459024), INT64_C( 6086538207170510268),
                            INT64_C( 5130097871257028467), INT64_C( 1548377050112752776),
                            INT64_C( 1253733446110746976), INT64_C( 3012333519594431544)) },
    { simde_mm512_set_epi64(INT64_C(-6609116217957060725), INT64_C( -302281769423418319),
                            INT64_C(-4897195678850214218), INT64_C(-5503480572790438091),
                            INT64_C(-7839393967146815223), INT64_C( 3255671599336790273),
                            INT64_C(  664294275788018935), INT64_C( 3408593724494687769)),
      simde_mm512_set_epi64(INT64_C( 6609116217957060725), INT64_C(  302281769423418319),
                            INT64_C( 4897195678850214218), INT64_C( 5503480572790438091),
                            INT64_C( 7839393967146815223), INT64_C( 3255671599336790273),
                            INT64_C(  664294275788018935), INT64_C( 3408593724494687769)) },
    { simde_mm512_set_epi64(INT64_C(-3947921567074644800), INT64_C( 2473906111070933983),
                            INT64_C( 5899615628251993446), INT64_C( 1683758976557896025),
                            INT64_C( 5912559464823232024), INT64_C( 8209801467605337080),
                            INT64_C(-3759026771366879931), INT64_C(-4585023147545297955)),
      simde_mm512_set_epi64(INT64_C( 3947921567074644800), INT64_C( 2473906111070933983),
                            INT64_C( 5899615628251993446), INT64_C( 1683758976557896025),
                            INT64_C( 5912559464823232024), INT64_C( 8209801467605337080),
                            INT64_C( 3759026771366879931), INT64_C( 4585023147545297955)) },
    { simde_mm512_set_epi64(INT64_C(-5018179428847904871), INT64_C(-5078614119960003085),
                            INT64_C( 1906444753996234817), INT64_C(-8073431190678733657),
                            INT64_C( 2507716860794484867), INT64_C( 3050920605853136553),
                            INT64_C( 8003726449701589009), INT64_C( 8169798351590582410)),
      simde_mm512_set_epi64(INT64_C( 5018179428847904871), INT64_C( 5078614119960003085),
                            INT64_C( 1906444753996234817), INT64_C( 8073431190678733657),
                            INT64_C( 2507716860794484867), INT64_C( 3050920605853136553),
                            INT64_C( 8003726449701589009), INT64_C( 8169798351590582410)) },
    { simde_mm512_set_epi64(INT64_C(   34320467490721535), INT64_C( 6032039111009323642),
                            INT64_C( 8825906149542039035), INT64_C( 3097839263351160271),
                            INT64_C( 8028249960129894953), INT64_C(-3442961435998375158),
                            INT64_C( 4762825474720326971), INT64_C( 5172383913584297790)),
      simde_mm512_set_epi64(INT64_C(   34320467490721535), INT64_C( 6032039111009323642),
                            INT64_C( 8825906149542039035), INT64_C( 3097839263351160271),
                            INT64_C( 8028249960129894953), INT64_C( 3442961435998375158),
                            INT64_C( 4762825474720326971), INT64_C( 5172383913584297790)) },
    { simde_mm512_set_epi64(INT64_C( 6012124616828815523), INT64_C(-1490086426868961318),
                            INT64_C(-6824757216361935934), INT64_C( 6020178289686090572),
                            INT64_C( 1854069096850744512), INT64_C(-5881095665354951863),
                            INT64_C( 6788068100773536681), INT64_C(-2707434157113404213)),
      simde_mm512_set_epi64(INT64_C( 6012124616828815523), INT64_C( 1490086426868961318),
                            INT64_C( 6824757216361935934), INT64_C( 6020178289686090572),
                            INT64_C( 1854069096850744512), INT64_C( 5881095665354951863),
                            INT64_C( 6788068100773536681), INT64_C( 2707434157113404213)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_abs_epi64(test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_abs_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(  492676116973233810), INT64_C(-6011138406694593910),
                            INT64_C( 4793302583727451571), INT64_C(-8501143198309462471),
                            INT64_C( 7175703865894427661), INT64_C( 8004913303465320948),
                            INT64_C(-2704856736044264837), INT64_C( 5658385619497272512)),
      UINT8_C(216),
      simde_mm512_set_epi64(INT64_C(-1717644828534315099), INT64_C(   78022038682650212),
                            INT64_C( 7139449655270167801), INT64_C( 2146714472316691717),
                            INT64_C( -192495991449383316), INT64_C(-5534743664186979260),
                            INT64_C(-5881663773003558792), INT64_C(-6698972267701962486)),
      simde_mm512_set_epi64(INT64_C( 1717644828534315099), INT64_C(   78022038682650212),
                            INT64_C( 4793302583727451571), INT64_C( 2146714472316691717),
                            INT64_C(  192495991449383316), INT64_C( 8004913303465320948),
                            INT64_C(-2704856736044264837), INT64_C( 5658385619497272512)) },
    { simde_mm512_set_epi64(INT64_C( 1469967797035145460), INT64_C( 5698418868819073779),
                            INT64_C( 1476440947581501981), INT64_C( 6259233690927012485),
                            INT64_C(-7834407389066617423), INT64_C( -820710109410615175),
                            INT64_C(-6555247677438161503), INT64_C( 3467536596098015717)),
      UINT8_C( 65),
      simde_mm512_set_epi64(INT64_C( 7787907596649075241), INT64_C( 6183988585695692436),
                            INT64_C( 6687836810634885572), INT64_C(-3481271883196348940),
                            INT64_C( 2810208213167470657), INT64_C(-4686213381983447596),
                            INT64_C( 1750544224109435853), INT64_C(-3649870730594357554)),
      simde_mm512_set_epi64(INT64_C( 1469967797035145460), INT64_C( 6183988585695692436),
                            INT64_C( 1476440947581501981), INT64_C( 6259233690927012485),
                            INT64_C(-7834407389066617423), INT64_C( -820710109410615175),
                            INT64_C(-6555247677438161503), INT64_C( 3649870730594357554)) },
    { simde_mm512_set_epi64(INT64_C(-8324574269298179729), INT64_C(-7258330021648709528),
                            INT64_C( 5937514152424000288), INT64_C( 1850437298497726112),
                            INT64_C(-8404329205387466185), INT64_C( 2671708839418006562),
                            INT64_C( 2003666284095471028), INT64_C( 6129159598982782124)),
      UINT8_C(199),
      simde_mm512_set_epi64(INT64_C( 7895846628610550850), INT64_C(-2067803869119741500),
                            INT64_C(-4037099766843598938), INT64_C(-2755209390738717188),
                            INT64_C(-7716919860907040546), INT64_C(  651602748594909758),
                            INT64_C(-1368765145407975583), INT64_C( -989534720059255127)),
      simde_mm512_set_epi64(INT64_C( 7895846628610550850), INT64_C( 2067803869119741500),
                            INT64_C( 5937514152424000288), INT64_C( 1850437298497726112),
                            INT64_C(-8404329205387466185), INT64_C(  651602748594909758),
                            INT64_C( 1368765145407975583), INT64_C(  989534720059255127)) },
    { simde_mm512_set_epi64(INT64_C(-4820061084596199963), INT64_C(-3116830217730655965),
                            INT64_C( 8263602287642686524), INT64_C(-7769087937993864972),
                            INT64_C(  688147068097687318), INT64_C( 5251800971073735884),
                            INT64_C( 1217023964204045922), INT64_C( 2408716443164236524)),
      UINT8_C( 32),
      simde_mm512_set_epi64(INT64_C(  748944487451629095), INT64_C(-5562876490484131857),
                            INT64_C( 4801338493970245765), INT64_C(-6148160310922917721),
                            INT64_C(-6526618537607083398), INT64_C(-1452156915681179960),
                            INT64_C(-6126568963486552397), INT64_C(-7197077943533128431)),
      simde_mm512_set_epi64(INT64_C(-4820061084596199963), INT64_C(-3116830217730655965),
                            INT64_C( 4801338493970245765), INT64_C(-7769087937993864972),
                            INT64_C(  688147068097687318), INT64_C( 5251800971073735884),
                            INT64_C( 1217023964204045922), INT64_C( 2408716443164236524)) },
    { simde_mm512_set_epi64(INT64_C(-3096584980416416798), INT64_C( 4918909782932989693),
                            INT64_C(-8004938825376986361), INT64_C( 5828784114914742259),
                            INT64_C(-2608731774237786277), INT64_C(-9056770170020559815),
                            INT64_C(-5319811934658395214), INT64_C(-2553108879087757308)),
      UINT8_C(222),
      simde_mm512_set_epi64(INT64_C(-7429178691671160403), INT64_C(-6726241153446442388),
                            INT64_C(-4671168844950875702), INT64_C(-6057578907613688046),
                            INT64_C( 4404786722084486842), INT64_C(-1094983952222664046),
                            INT64_C( 5287971478839612040), INT64_C(-1393761486045455198)),
      simde_mm512_set_epi64(INT64_C( 7429178691671160403), INT64_C( 6726241153446442388),
                            INT64_C(-8004938825376986361), INT64_C( 6057578907613688046),
                            INT64_C( 4404786722084486842), INT64_C( 1094983952222664046),
                            INT64_C( 5287971478839612040), INT64_C(-2553108879087757308)) },
    { simde_mm512_set_epi64(INT64_C(-2342519952706594553), INT64_C( 7644592964730421973),
                            INT64_C( 9172903191652197902), INT64_C( 1458420204680989133),
                            INT64_C(-7367844323210688147), INT64_C(-7391408909901332845),
                            INT64_C(-1133308551202396595), INT64_C( 4102785619076298947)),
      UINT8_C(230),
      simde_mm512_set_epi64(INT64_C( 2067960823776290910), INT64_C(-7228934469556881970),
                            INT64_C(-3653374474835545853), INT64_C( 5480624866862364429),
                            INT64_C(-6600577412846457563), INT64_C( 2529415530022027870),
                            INT64_C(-4329045443225919216), INT64_C( 9121454504064466670)),
      simde_mm512_set_epi64(INT64_C( 2067960823776290910), INT64_C( 7228934469556881970),
                            INT64_C( 3653374474835545853), INT64_C( 1458420204680989133),
                            INT64_C(-7367844323210688147), INT64_C( 2529415530022027870),
                            INT64_C( 4329045443225919216), INT64_C( 4102785619076298947)) },
    { simde_mm512_set_epi64(INT64_C(-4792271441403297267), INT64_C( 6523863092598044385),
                            INT64_C( 6601631579296004837), INT64_C(  540377227778745844),
                            INT64_C(-1608274285869950535), INT64_C( 1496684406111625906),
                            INT64_C( 1553447803787926005), INT64_C(-8007245467626955940)),
      UINT8_C(104),
      simde_mm512_set_epi64(INT64_C( 3154046907468289764), INT64_C(  163566292437055869),
                            INT64_C( 8677719201018760530), INT64_C( 8769975268177995652),
                            INT64_C( 7613129594859420923), INT64_C(-1998576254813523574),
                            INT64_C( 6270036339063321053), INT64_C( 5566883881093264567)),
      simde_mm512_set_epi64(INT64_C(-4792271441403297267), INT64_C(  163566292437055869),
                            INT64_C( 8677719201018760530), INT64_C(  540377227778745844),
                            INT64_C( 7613129594859420923), INT64_C( 1496684406111625906),
                            INT64_C( 1553447803787926005), INT64_C(-8007245467626955940)) },
    { simde_mm512_set_epi64(INT64_C( 6148806541912347944), INT64_C( 1908371497711301408),
                            INT64_C(-9141679846684482011), INT64_C(-5393015831913260049),
                            INT64_C(-7420100767347904040), INT64_C(-3473408352959472719),
                            INT64_C( 3774406077177521346), INT64_C(-4000561069769581270)),
      UINT8_C(128),
      simde_mm512_set_epi64(INT64_C( 8461115545003933300), INT64_C(-7617041399687843974),
                            INT64_C( 7167540182740956629), INT64_C( 7612124823363779747),
                            INT64_C(-1891905030773424117), INT64_C(-5190678261404669746),
                            INT64_C( -683318933916552175), INT64_C( 9111998746576489775)),
      simde_mm512_set_epi64(INT64_C( 8461115545003933300), INT64_C( 1908371497711301408),
                            INT64_C(-9141679846684482011), INT64_C(-5393015831913260049),
                            INT64_C(-7420100767347904040), INT64_C(-3473408352959472719),
                            INT64_C( 3774406077177521346), INT64_C(-4000561069769581270)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_abs_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_abs_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C(192),
      simde_mm512_set_epi64(INT64_C(  -41252748446509487), INT64_C( 7890911908509001079),
                            INT64_C(-5961962669328745651), INT64_C(-3744340997299642547),
                            INT64_C( 6343687558518880421), INT64_C(-4320961892205516228),
                            INT64_C(-2965231175381652703), INT64_C( 2864943002512957373)),
      simde_mm512_set_epi64(INT64_C(   41252748446509487), INT64_C( 7890911908509001079),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C( 10),
      simde_mm512_set_epi64(INT64_C(-8064009705201487071), INT64_C(   66682790377500833),
                            INT64_C( 8959003693208071074), INT64_C(-7914560703715169936),
                            INT64_C( 6097984971859041770), INT64_C( 7564092909171024886),
                            INT64_C( 3840035858897969653), INT64_C( -443938296699520969)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 6097984971859041770), INT64_C(                   0),
                            INT64_C( 3840035858897969653), INT64_C(                   0)) },
    { UINT8_C( 41),
      simde_mm512_set_epi64(INT64_C(-8443142306353437631), INT64_C( 1469967797035145460),
                            INT64_C( 5698418868819073779), INT64_C( 1476440947581501981),
                            INT64_C( 6259233690927012485), INT64_C(-7834407389066617423),
                            INT64_C( -820710109410615175), INT64_C(-6555247677438161503)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 5698418868819073779), INT64_C(                   0),
                            INT64_C( 6259233690927012485), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 6555247677438161503)) },
    { UINT8_C(232),
      simde_mm512_set_epi64(INT64_C( 6129159598982782124), INT64_C( 7787907596649075241),
                            INT64_C( 6183988585695692436), INT64_C( 6687836810634885572),
                            INT64_C(-3481271883196348940), INT64_C( 2810208213167470657),
                            INT64_C(-4686213381983447596), INT64_C( 1750544224109435853)),
      simde_mm512_set_epi64(INT64_C( 6129159598982782124), INT64_C( 7787907596649075241),
                            INT64_C( 6183988585695692436), INT64_C(                   0),
                            INT64_C( 3481271883196348940), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(180),
      simde_mm512_set_epi64(INT64_C( -989534720059255127), INT64_C(-4487290813852079154),
                            INT64_C(-4890907616466355379), INT64_C( 3824480121463916969),
                            INT64_C(-3452489463091031317), INT64_C( 4618202413866537757),
                            INT64_C(-6813829451133759224), INT64_C(-4843608058602032162)),
      simde_mm512_set_epi64(INT64_C(  989534720059255127), INT64_C(                   0),
                            INT64_C( 4890907616466355379), INT64_C( 3824480121463916969),
                            INT64_C(                   0), INT64_C( 4618202413866537757),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C( 97),
      simde_mm512_set_epi64(INT64_C(-1331633504094684457), INT64_C(-2922173236712853844),
                            INT64_C(-4950533224594034858), INT64_C( 3742633972513462054),
                            INT64_C(-3856884862992283736), INT64_C( 2824537604935384645),
                            INT64_C(-4433054007697935041), INT64_C( 6677415749608352602)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C( 2922173236712853844),
                            INT64_C( 4950533224594034858), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 6677415749608352602)) },
    { UINT8_C(216),
      simde_mm512_set_epi64(INT64_C(-2921870015535851587), INT64_C( -366844282582149856),
                            INT64_C(-4820061084596199963), INT64_C(-3116830217730655965),
                            INT64_C( 8263602287642686524), INT64_C(-7769087937993864972),
                            INT64_C(  688147068097687318), INT64_C( 5251800971073735884)),
      simde_mm512_set_epi64(INT64_C( 2921870015535851587), INT64_C(  366844282582149856),
                            INT64_C(                   0), INT64_C( 3116830217730655965),
                            INT64_C( 8263602287642686524), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(252),
      simde_mm512_set_epi64(INT64_C(-5319811934658395214), INT64_C(-2553108879087757308),
                            INT64_C(  748944487451629095), INT64_C(-5562876490484131857),
                            INT64_C( 4801338493970245765), INT64_C(-6148160310922917721),
                            INT64_C(-6526618537607083398), INT64_C(-1452156915681179960)),
      simde_mm512_set_epi64(INT64_C( 5319811934658395214), INT64_C( 2553108879087757308),
                            INT64_C(  748944487451629095), INT64_C( 5562876490484131857),
                            INT64_C( 4801338493970245765), INT64_C( 6148160310922917721),
                            INT64_C(                   0), INT64_C(                   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_abs_epi64(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_abs_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   747.74), SIMDE_FLOAT32_C(  -874.37), SIMDE_FLOAT32_C(   751.90), SIMDE_FLOAT32_C(  -592.77),
                         SIMDE_FLOAT32_C(  -708.81), SIMDE_FLOAT32_C(   252.42), SIMDE_FLOAT32_C(  -787.46), SIMDE_FLOAT32_C(  -882.47),
                         SIMDE_FLOAT32_C(  -140.56), SIMDE_FLOAT32_C(  -558.99), SIMDE_FLOAT32_C(   240.08), SIMDE_FLOAT32_C(  -481.72),
                         SIMDE_FLOAT32_C(   489.35), SIMDE_FLOAT32_C(   686.76), SIMDE_FLOAT32_C(  -206.54), SIMDE_FLOAT32_C(   728.61)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   747.74), SIMDE_FLOAT32_C(   874.37), SIMDE_FLOAT32_C(   751.90), SIMDE_FLOAT32_C(   592.77),
                         SIMDE_FLOAT32_C(   708.81), SIMDE_FLOAT32_C(   252.42), SIMDE_FLOAT32_C(   787.46), SIMDE_FLOAT32_C(   882.47),
                         SIMDE_FLOAT32_C(   140.56), SIMDE_FLOAT32_C(   558.99), SIMDE_FLOAT32_C(   240.08), SIMDE_FLOAT32_C(   481.72),
                         SIMDE_FLOAT32_C(   489.35), SIMDE_FLOAT32_C(   686.76), SIMDE_FLOAT32_C(   206.54), SIMDE_FLOAT32_C(   728.61)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(    76.28), SIMDE_FLOAT32_C(  -319.23), SIMDE_FLOAT32_C(   655.09), SIMDE_FLOAT32_C(   773.21),
                         SIMDE_FLOAT32_C(  -928.32), SIMDE_FLOAT32_C(   -25.13), SIMDE_FLOAT32_C(  -847.53), SIMDE_FLOAT32_C(   859.40),
                         SIMDE_FLOAT32_C(   388.54), SIMDE_FLOAT32_C(  -184.67), SIMDE_FLOAT32_C(   102.38), SIMDE_FLOAT32_C(   833.56),
                         SIMDE_FLOAT32_C(  -722.29), SIMDE_FLOAT32_C(  -441.84), SIMDE_FLOAT32_C(  -821.42), SIMDE_FLOAT32_C(  -761.98)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    76.28), SIMDE_FLOAT32_C(   319.23), SIMDE_FLOAT32_C(   655.09), SIMDE_FLOAT32_C(   773.21),
                         SIMDE_FLOAT32_C(   928.32), SIMDE_FLOAT32_C(    25.13), SIMDE_FLOAT32_C(   847.53), SIMDE_FLOAT32_C(   859.40),
                         SIMDE_FLOAT32_C(   388.54), SIMDE_FLOAT32_C(   184.67), SIMDE_FLOAT32_C(   102.38), SIMDE_FLOAT32_C(   833.56),
                         SIMDE_FLOAT32_C(   722.29), SIMDE_FLOAT32_C(   441.84), SIMDE_FLOAT32_C(   821.42), SIMDE_FLOAT32_C(   761.98)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -645.61), SIMDE_FLOAT32_C(   594.77), SIMDE_FLOAT32_C(  -171.69), SIMDE_FLOAT32_C(   108.08),
                         SIMDE_FLOAT32_C(    -7.24), SIMDE_FLOAT32_C(   885.82), SIMDE_FLOAT32_C(   296.84), SIMDE_FLOAT32_C(  -408.70),
                         SIMDE_FLOAT32_C(   -40.31), SIMDE_FLOAT32_C(   866.84), SIMDE_FLOAT32_C(  -660.11), SIMDE_FLOAT32_C(   121.17),
                         SIMDE_FLOAT32_C(   988.31), SIMDE_FLOAT32_C(  -622.26), SIMDE_FLOAT32_C(   206.00), SIMDE_FLOAT32_C(   520.48)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   645.61), SIMDE_FLOAT32_C(   594.77), SIMDE_FLOAT32_C(   171.69), SIMDE_FLOAT32_C(   108.08),
                         SIMDE_FLOAT32_C(     7.24), SIMDE_FLOAT32_C(   885.82), SIMDE_FLOAT32_C(   296.84), SIMDE_FLOAT32_C(   408.70),
                         SIMDE_FLOAT32_C(    40.31), SIMDE_FLOAT32_C(   866.84), SIMDE_FLOAT32_C(   660.11), SIMDE_FLOAT32_C(   121.17),
                         SIMDE_FLOAT32_C(   988.31), SIMDE_FLOAT32_C(   622.26), SIMDE_FLOAT32_C(   206.00), SIMDE_FLOAT32_C(   520.48)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   530.77), SIMDE_FLOAT32_C(   966.97), SIMDE_FLOAT32_C(   -63.51), SIMDE_FLOAT32_C(   360.07),
                         SIMDE_FLOAT32_C(  -846.61), SIMDE_FLOAT32_C(  -749.79), SIMDE_FLOAT32_C(   510.77), SIMDE_FLOAT32_C(  -104.12),
                         SIMDE_FLOAT32_C(  -838.06), SIMDE_FLOAT32_C(  -901.25), SIMDE_FLOAT32_C(   -89.58), SIMDE_FLOAT32_C(   539.88),
                         SIMDE_FLOAT32_C(    88.35), SIMDE_FLOAT32_C(   773.77), SIMDE_FLOAT32_C(  -729.20), SIMDE_FLOAT32_C(  -254.72)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   530.77), SIMDE_FLOAT32_C(   966.97), SIMDE_FLOAT32_C(    63.51), SIMDE_FLOAT32_C(   360.07),
                         SIMDE_FLOAT32_C(   846.61), SIMDE_FLOAT32_C(   749.79), SIMDE_FLOAT32_C(   510.77), SIMDE_FLOAT32_C(   104.12),
                         SIMDE_FLOAT32_C(   838.06), SIMDE_FLOAT32_C(   901.25), SIMDE_FLOAT32_C(    89.58), SIMDE_FLOAT32_C(   539.88),
                         SIMDE_FLOAT32_C(    88.35), SIMDE_FLOAT32_C(   773.77), SIMDE_FLOAT32_C(   729.20), SIMDE_FLOAT32_C(   254.72)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -814.45), SIMDE_FLOAT32_C(  -377.80), SIMDE_FLOAT32_C(   640.68), SIMDE_FLOAT32_C(   778.00),
                         SIMDE_FLOAT32_C(   377.67), SIMDE_FLOAT32_C(  -489.06), SIMDE_FLOAT32_C(   933.74), SIMDE_FLOAT32_C(  -749.41),
                         SIMDE_FLOAT32_C(   193.12), SIMDE_FLOAT32_C(  -423.37), SIMDE_FLOAT32_C(  -194.06), SIMDE_FLOAT32_C(  -118.88),
                         SIMDE_FLOAT32_C(   -77.74), SIMDE_FLOAT32_C(  -506.16), SIMDE_FLOAT32_C(  -617.33), SIMDE_FLOAT32_C(  -947.60)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   814.45), SIMDE_FLOAT32_C(   377.80), SIMDE_FLOAT32_C(   640.68), SIMDE_FLOAT32_C(   778.00),
                         SIMDE_FLOAT32_C(   377.67), SIMDE_FLOAT32_C(   489.06), SIMDE_FLOAT32_C(   933.74), SIMDE_FLOAT32_C(   749.41),
                         SIMDE_FLOAT32_C(   193.12), SIMDE_FLOAT32_C(   423.37), SIMDE_FLOAT32_C(   194.06), SIMDE_FLOAT32_C(   118.88),
                         SIMDE_FLOAT32_C(    77.74), SIMDE_FLOAT32_C(   506.16), SIMDE_FLOAT32_C(   617.33), SIMDE_FLOAT32_C(   947.60)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   525.11), SIMDE_FLOAT32_C(   299.00), SIMDE_FLOAT32_C(   814.48), SIMDE_FLOAT32_C(   676.51),
                         SIMDE_FLOAT32_C(  -481.76), SIMDE_FLOAT32_C(   528.75), SIMDE_FLOAT32_C(  -375.20), SIMDE_FLOAT32_C(   146.55),
                         SIMDE_FLOAT32_C(   199.14), SIMDE_FLOAT32_C(  -505.05), SIMDE_FLOAT32_C(   833.96), SIMDE_FLOAT32_C(  -388.48),
                         SIMDE_FLOAT32_C(  -212.57), SIMDE_FLOAT32_C(   943.89), SIMDE_FLOAT32_C(   651.63), SIMDE_FLOAT32_C(   695.54)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   525.11), SIMDE_FLOAT32_C(   299.00), SIMDE_FLOAT32_C(   814.48), SIMDE_FLOAT32_C(   676.51),
                         SIMDE_FLOAT32_C(   481.76), SIMDE_FLOAT32_C(   528.75), SIMDE_FLOAT32_C(   375.20), SIMDE_FLOAT32_C(   146.55),
                         SIMDE_FLOAT32_C(   199.14), SIMDE_FLOAT32_C(   505.05), SIMDE_FLOAT32_C(   833.96), SIMDE_FLOAT32_C(   388.48),
                         SIMDE_FLOAT32_C(   212.57), SIMDE_FLOAT32_C(   943.89), SIMDE_FLOAT32_C(   651.63), SIMDE_FLOAT32_C(   695.54)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   -13.73), SIMDE_FLOAT32_C(  -546.29), SIMDE_FLOAT32_C(  -787.44), SIMDE_FLOAT32_C(  -104.88),
                         SIMDE_FLOAT32_C(   979.47), SIMDE_FLOAT32_C(  -744.23), SIMDE_FLOAT32_C(   836.15), SIMDE_FLOAT32_C(   495.73),
                         SIMDE_FLOAT32_C(  -301.39), SIMDE_FLOAT32_C(   262.00), SIMDE_FLOAT32_C(   466.22), SIMDE_FLOAT32_C(   536.10),
                         SIMDE_FLOAT32_C(  -613.16), SIMDE_FLOAT32_C(  -393.36), SIMDE_FLOAT32_C(   -56.94), SIMDE_FLOAT32_C(   670.22)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    13.73), SIMDE_FLOAT32_C(   546.29), SIMDE_FLOAT32_C(   787.44), SIMDE_FLOAT32_C(   104.88),
                         SIMDE_FLOAT32_C(   979.47), SIMDE_FLOAT32_C(   744.23), SIMDE_FLOAT32_C(   836.15), SIMDE_FLOAT32_C(   495.73),
                         SIMDE_FLOAT32_C(   301.39), SIMDE_FLOAT32_C(   262.00), SIMDE_FLOAT32_C(   466.22), SIMDE_FLOAT32_C(   536.10),
                         SIMDE_FLOAT32_C(   613.16), SIMDE_FLOAT32_C(   393.36), SIMDE_FLOAT32_C(    56.94), SIMDE_FLOAT32_C(   670.22)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   493.79), SIMDE_FLOAT32_C(   -29.44), SIMDE_FLOAT32_C(  -941.83), SIMDE_FLOAT32_C(  -567.95),
                         SIMDE_FLOAT32_C(   535.05), SIMDE_FLOAT32_C(    43.85), SIMDE_FLOAT32_C(  -963.94), SIMDE_FLOAT32_C(   235.87),
                         SIMDE_FLOAT32_C(   143.93), SIMDE_FLOAT32_C(  -236.80), SIMDE_FLOAT32_C(   550.36), SIMDE_FLOAT32_C(    -8.58),
                         SIMDE_FLOAT32_C(   374.16), SIMDE_FLOAT32_C(   714.91), SIMDE_FLOAT32_C(  -355.51), SIMDE_FLOAT32_C(  -520.52)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   493.79), SIMDE_FLOAT32_C(    29.44), SIMDE_FLOAT32_C(   941.83), SIMDE_FLOAT32_C(   567.95),
                         SIMDE_FLOAT32_C(   535.05), SIMDE_FLOAT32_C(    43.85), SIMDE_FLOAT32_C(   963.94), SIMDE_FLOAT32_C(   235.87),
                         SIMDE_FLOAT32_C(   143.93), SIMDE_FLOAT32_C(   236.80), SIMDE_FLOAT32_C(   550.36), SIMDE_FLOAT32_C(     8.58),
                         SIMDE_FLOAT32_C(   374.16), SIMDE_FLOAT32_C(   714.91), SIMDE_FLOAT32_C(   355.51), SIMDE_FLOAT32_C(   520.52)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_abs_ps(test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_abs_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -319.23), SIMDE_FLOAT32_C(   773.21), SIMDE_FLOAT32_C(   -25.13), SIMDE_FLOAT32_C(   859.40),
                         SIMDE_FLOAT32_C(  -184.67), SIMDE_FLOAT32_C(   833.56), SIMDE_FLOAT32_C(  -441.84), SIMDE_FLOAT32_C(  -761.98),
                         SIMDE_FLOAT32_C(  -874.37), SIMDE_FLOAT32_C(  -592.77), SIMDE_FLOAT32_C(   252.42), SIMDE_FLOAT32_C(  -882.47),
                         SIMDE_FLOAT32_C(  -558.99), SIMDE_FLOAT32_C(  -481.72), SIMDE_FLOAT32_C(   686.76), SIMDE_FLOAT32_C(   728.61)),
      UINT16_C(15540),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    76.28), SIMDE_FLOAT32_C(   655.09), SIMDE_FLOAT32_C(  -928.32), SIMDE_FLOAT32_C(  -847.53),
                         SIMDE_FLOAT32_C(   388.54), SIMDE_FLOAT32_C(   102.38), SIMDE_FLOAT32_C(  -722.29), SIMDE_FLOAT32_C(  -821.42),
                         SIMDE_FLOAT32_C(   747.74), SIMDE_FLOAT32_C(   751.90), SIMDE_FLOAT32_C(  -708.81), SIMDE_FLOAT32_C(  -787.46),
                         SIMDE_FLOAT32_C(  -140.56), SIMDE_FLOAT32_C(   240.08), SIMDE_FLOAT32_C(   489.35), SIMDE_FLOAT32_C(  -206.54)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -319.23), SIMDE_FLOAT32_C(   773.21), SIMDE_FLOAT32_C(   928.32), SIMDE_FLOAT32_C(   847.53),
                         SIMDE_FLOAT32_C(   388.54), SIMDE_FLOAT32_C(   102.38), SIMDE_FLOAT32_C(  -441.84), SIMDE_FLOAT32_C(  -761.98),
                         SIMDE_FLOAT32_C(   747.74), SIMDE_FLOAT32_C(  -592.77), SIMDE_FLOAT32_C(   708.81), SIMDE_FLOAT32_C(   787.46),
                         SIMDE_FLOAT32_C(  -558.99), SIMDE_FLOAT32_C(   240.08), SIMDE_FLOAT32_C(   686.76), SIMDE_FLOAT32_C(   728.61)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -947.60), SIMDE_FLOAT32_C(   966.97), SIMDE_FLOAT32_C(   360.07), SIMDE_FLOAT32_C(  -749.79),
                         SIMDE_FLOAT32_C(  -104.12), SIMDE_FLOAT32_C(  -901.25), SIMDE_FLOAT32_C(   539.88), SIMDE_FLOAT32_C(   773.77),
                         SIMDE_FLOAT32_C(  -254.72), SIMDE_FLOAT32_C(   594.77), SIMDE_FLOAT32_C(   108.08), SIMDE_FLOAT32_C(   885.82),
                         SIMDE_FLOAT32_C(  -408.70), SIMDE_FLOAT32_C(   866.84), SIMDE_FLOAT32_C(   121.17), SIMDE_FLOAT32_C(  -622.26)),
      UINT16_C( 6415),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -617.33), SIMDE_FLOAT32_C(   530.77), SIMDE_FLOAT32_C(   -63.51), SIMDE_FLOAT32_C(  -846.61),
                         SIMDE_FLOAT32_C(   510.77), SIMDE_FLOAT32_C(  -838.06), SIMDE_FLOAT32_C(   -89.58), SIMDE_FLOAT32_C(    88.35),
                         SIMDE_FLOAT32_C(  -729.20), SIMDE_FLOAT32_C(  -645.61), SIMDE_FLOAT32_C(  -171.69), SIMDE_FLOAT32_C(    -7.24),
                         SIMDE_FLOAT32_C(   296.84), SIMDE_FLOAT32_C(   -40.31), SIMDE_FLOAT32_C(  -660.11), SIMDE_FLOAT32_C(   988.31)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -947.60), SIMDE_FLOAT32_C(   966.97), SIMDE_FLOAT32_C(   360.07), SIMDE_FLOAT32_C(   846.61),
                         SIMDE_FLOAT32_C(   510.77), SIMDE_FLOAT32_C(  -901.25), SIMDE_FLOAT32_C(   539.88), SIMDE_FLOAT32_C(    88.35),
                         SIMDE_FLOAT32_C(  -254.72), SIMDE_FLOAT32_C(   594.77), SIMDE_FLOAT32_C(   108.08), SIMDE_FLOAT32_C(   885.82),
                         SIMDE_FLOAT32_C(   296.84), SIMDE_FLOAT32_C(    40.31), SIMDE_FLOAT32_C(   660.11), SIMDE_FLOAT32_C(   988.31)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -393.36), SIMDE_FLOAT32_C(   670.22), SIMDE_FLOAT32_C(   299.00), SIMDE_FLOAT32_C(   676.51),
                         SIMDE_FLOAT32_C(   528.75), SIMDE_FLOAT32_C(   146.55), SIMDE_FLOAT32_C(  -505.05), SIMDE_FLOAT32_C(  -388.48),
                         SIMDE_FLOAT32_C(   943.89), SIMDE_FLOAT32_C(   695.54), SIMDE_FLOAT32_C(  -377.80), SIMDE_FLOAT32_C(   778.00),
                         SIMDE_FLOAT32_C(  -489.06), SIMDE_FLOAT32_C(  -749.41), SIMDE_FLOAT32_C(  -423.37), SIMDE_FLOAT32_C(  -118.88)),
      UINT16_C( 1525),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -613.16), SIMDE_FLOAT32_C(   -56.94), SIMDE_FLOAT32_C(   525.11), SIMDE_FLOAT32_C(   814.48),
                         SIMDE_FLOAT32_C(  -481.76), SIMDE_FLOAT32_C(  -375.20), SIMDE_FLOAT32_C(   199.14), SIMDE_FLOAT32_C(   833.96),
                         SIMDE_FLOAT32_C(  -212.57), SIMDE_FLOAT32_C(   651.63), SIMDE_FLOAT32_C(  -814.45), SIMDE_FLOAT32_C(   640.68),
                         SIMDE_FLOAT32_C(   377.67), SIMDE_FLOAT32_C(   933.74), SIMDE_FLOAT32_C(   193.12), SIMDE_FLOAT32_C(  -194.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -393.36), SIMDE_FLOAT32_C(   670.22), SIMDE_FLOAT32_C(   299.00), SIMDE_FLOAT32_C(   676.51),
                         SIMDE_FLOAT32_C(   528.75), SIMDE_FLOAT32_C(   375.20), SIMDE_FLOAT32_C(  -505.05), SIMDE_FLOAT32_C(   833.96),
                         SIMDE_FLOAT32_C(   212.57), SIMDE_FLOAT32_C(   651.63), SIMDE_FLOAT32_C(   814.45), SIMDE_FLOAT32_C(   640.68),
                         SIMDE_FLOAT32_C(  -489.06), SIMDE_FLOAT32_C(   933.74), SIMDE_FLOAT32_C(  -423.37), SIMDE_FLOAT32_C(   194.06)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(    61.51), SIMDE_FLOAT32_C(  -643.69), SIMDE_FLOAT32_C(   -16.59), SIMDE_FLOAT32_C(   -29.44),
                         SIMDE_FLOAT32_C(  -567.95), SIMDE_FLOAT32_C(    43.85), SIMDE_FLOAT32_C(   235.87), SIMDE_FLOAT32_C(  -236.80),
                         SIMDE_FLOAT32_C(    -8.58), SIMDE_FLOAT32_C(   714.91), SIMDE_FLOAT32_C(  -520.52), SIMDE_FLOAT32_C(  -546.29),
                         SIMDE_FLOAT32_C(  -104.88), SIMDE_FLOAT32_C(  -744.23), SIMDE_FLOAT32_C(   495.73), SIMDE_FLOAT32_C(   262.00)),
      UINT16_C(29879),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   515.30), SIMDE_FLOAT32_C(   896.28), SIMDE_FLOAT32_C(   660.35), SIMDE_FLOAT32_C(   493.79),
                         SIMDE_FLOAT32_C(  -941.83), SIMDE_FLOAT32_C(   535.05), SIMDE_FLOAT32_C(  -963.94), SIMDE_FLOAT32_C(   143.93),
                         SIMDE_FLOAT32_C(   550.36), SIMDE_FLOAT32_C(   374.16), SIMDE_FLOAT32_C(  -355.51), SIMDE_FLOAT32_C(   -13.73),
                         SIMDE_FLOAT32_C(  -787.44), SIMDE_FLOAT32_C(   979.47), SIMDE_FLOAT32_C(   836.15), SIMDE_FLOAT32_C(  -301.39)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    61.51), SIMDE_FLOAT32_C(   896.28), SIMDE_FLOAT32_C(   660.35), SIMDE_FLOAT32_C(   493.79),
                         SIMDE_FLOAT32_C(  -567.95), SIMDE_FLOAT32_C(   535.05), SIMDE_FLOAT32_C(   235.87), SIMDE_FLOAT32_C(  -236.80),
                         SIMDE_FLOAT32_C(   550.36), SIMDE_FLOAT32_C(   714.91), SIMDE_FLOAT32_C(   355.51), SIMDE_FLOAT32_C(    13.73),
                         SIMDE_FLOAT32_C(  -104.88), SIMDE_FLOAT32_C(   979.47), SIMDE_FLOAT32_C(   836.15), SIMDE_FLOAT32_C(   301.39)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   148.99), SIMDE_FLOAT32_C(  -963.65), SIMDE_FLOAT32_C(   149.45), SIMDE_FLOAT32_C(  -850.34),
                         SIMDE_FLOAT32_C(  -524.37), SIMDE_FLOAT32_C(  -513.69), SIMDE_FLOAT32_C(    22.08), SIMDE_FLOAT32_C(   488.53),
                         SIMDE_FLOAT32_C(   770.65), SIMDE_FLOAT32_C(   491.66), SIMDE_FLOAT32_C(    89.59), SIMDE_FLOAT32_C(   924.64),
                         SIMDE_FLOAT32_C(  -763.40), SIMDE_FLOAT32_C(  -404.62), SIMDE_FLOAT32_C(  -957.75), SIMDE_FLOAT32_C(   281.78)),
      UINT16_C(44157),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -979.51), SIMDE_FLOAT32_C(  -129.70), SIMDE_FLOAT32_C(  -587.42), SIMDE_FLOAT32_C(    94.97),
                         SIMDE_FLOAT32_C(  -887.16), SIMDE_FLOAT32_C(  -189.75), SIMDE_FLOAT32_C(   881.78), SIMDE_FLOAT32_C(  -152.81),
                         SIMDE_FLOAT32_C(   943.19), SIMDE_FLOAT32_C(  -229.02), SIMDE_FLOAT32_C(  -577.41), SIMDE_FLOAT32_C(  -719.96),
                         SIMDE_FLOAT32_C(   770.58), SIMDE_FLOAT32_C(  -153.52), SIMDE_FLOAT32_C(  -991.64), SIMDE_FLOAT32_C(   -53.48)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   979.51), SIMDE_FLOAT32_C(  -963.65), SIMDE_FLOAT32_C(   587.42), SIMDE_FLOAT32_C(  -850.34),
                         SIMDE_FLOAT32_C(   887.16), SIMDE_FLOAT32_C(   189.75), SIMDE_FLOAT32_C(    22.08), SIMDE_FLOAT32_C(   488.53),
                         SIMDE_FLOAT32_C(   770.65), SIMDE_FLOAT32_C(   229.02), SIMDE_FLOAT32_C(   577.41), SIMDE_FLOAT32_C(   719.96),
                         SIMDE_FLOAT32_C(   770.58), SIMDE_FLOAT32_C(   153.52), SIMDE_FLOAT32_C(  -957.75), SIMDE_FLOAT32_C(    53.48)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   631.94), SIMDE_FLOAT32_C(  -409.79), SIMDE_FLOAT32_C(   668.07), SIMDE_FLOAT32_C(   542.88),
                         SIMDE_FLOAT32_C(  -896.06), SIMDE_FLOAT32_C(   248.80), SIMDE_FLOAT32_C(   200.01), SIMDE_FLOAT32_C(   669.33),
                         SIMDE_FLOAT32_C(  -642.07), SIMDE_FLOAT32_C(  -212.55), SIMDE_FLOAT32_C(  -356.51), SIMDE_FLOAT32_C(  -440.95),
                         SIMDE_FLOAT32_C(  -982.52), SIMDE_FLOAT32_C(  -842.67), SIMDE_FLOAT32_C(  -420.59), SIMDE_FLOAT32_C(  -949.02)),
      UINT16_C(15240),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   966.19), SIMDE_FLOAT32_C(   529.24), SIMDE_FLOAT32_C(  -544.06), SIMDE_FLOAT32_C(  -881.83),
                         SIMDE_FLOAT32_C(  -242.38), SIMDE_FLOAT32_C(  -380.44), SIMDE_FLOAT32_C(  -752.70), SIMDE_FLOAT32_C(  -160.45),
                         SIMDE_FLOAT32_C(   773.41), SIMDE_FLOAT32_C(  -474.98), SIMDE_FLOAT32_C(   573.78), SIMDE_FLOAT32_C(  -190.69),
                         SIMDE_FLOAT32_C(  -743.99), SIMDE_FLOAT32_C(  -698.61), SIMDE_FLOAT32_C(  -633.81), SIMDE_FLOAT32_C(   938.50)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   631.94), SIMDE_FLOAT32_C(  -409.79), SIMDE_FLOAT32_C(   544.06), SIMDE_FLOAT32_C(   881.83),
                         SIMDE_FLOAT32_C(   242.38), SIMDE_FLOAT32_C(   248.80), SIMDE_FLOAT32_C(   752.70), SIMDE_FLOAT32_C(   160.45),
                         SIMDE_FLOAT32_C(   773.41), SIMDE_FLOAT32_C(  -212.55), SIMDE_FLOAT32_C(  -356.51), SIMDE_FLOAT32_C(  -440.95),
                         SIMDE_FLOAT32_C(   743.99), SIMDE_FLOAT32_C(  -842.67), SIMDE_FLOAT32_C(  -420.59), SIMDE_FLOAT32_C(  -949.02)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   156.92), SIMDE_FLOAT32_C(  -736.34), SIMDE_FLOAT32_C(   166.92), SIMDE_FLOAT32_C(   300.41),
                         SIMDE_FLOAT32_C(  -295.98), SIMDE_FLOAT32_C(  -702.22), SIMDE_FLOAT32_C(  -740.49), SIMDE_FLOAT32_C(   -80.99),
                         SIMDE_FLOAT32_C(  -785.06), SIMDE_FLOAT32_C(    87.65), SIMDE_FLOAT32_C(  -482.52), SIMDE_FLOAT32_C(  -681.02),
                         SIMDE_FLOAT32_C(   764.25), SIMDE_FLOAT32_C(   305.46), SIMDE_FLOAT32_C(   526.44), SIMDE_FLOAT32_C(   369.20)),
      UINT16_C(49024),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   809.64), SIMDE_FLOAT32_C(  -790.72), SIMDE_FLOAT32_C(   295.53), SIMDE_FLOAT32_C(  -856.33),
                         SIMDE_FLOAT32_C(   237.04), SIMDE_FLOAT32_C(  -607.75), SIMDE_FLOAT32_C(  -732.96), SIMDE_FLOAT32_C(  -497.56),
                         SIMDE_FLOAT32_C(  -918.03), SIMDE_FLOAT32_C(   488.66), SIMDE_FLOAT32_C(  -523.80), SIMDE_FLOAT32_C(  -224.58),
                         SIMDE_FLOAT32_C(   298.04), SIMDE_FLOAT32_C(   606.61), SIMDE_FLOAT32_C(  -852.36), SIMDE_FLOAT32_C(  -314.42)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   809.64), SIMDE_FLOAT32_C(  -736.34), SIMDE_FLOAT32_C(   295.53), SIMDE_FLOAT32_C(   856.33),
                         SIMDE_FLOAT32_C(   237.04), SIMDE_FLOAT32_C(   607.75), SIMDE_FLOAT32_C(   732.96), SIMDE_FLOAT32_C(   497.56),
                         SIMDE_FLOAT32_C(   918.03), SIMDE_FLOAT32_C(    87.65), SIMDE_FLOAT32_C(  -482.52), SIMDE_FLOAT32_C(  -681.02),
                         SIMDE_FLOAT32_C(   764.25), SIMDE_FLOAT32_C(   305.46), SIMDE_FLOAT32_C(   526.44), SIMDE_FLOAT32_C(   369.20)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -897.44), SIMDE_FLOAT32_C(   113.95), SIMDE_FLOAT32_C(   654.01), SIMDE_FLOAT32_C(   620.23),
                         SIMDE_FLOAT32_C(   623.09), SIMDE_FLOAT32_C(  -407.46), SIMDE_FLOAT32_C(  -763.16), SIMDE_FLOAT32_C(  -768.89),
                         SIMDE_FLOAT32_C(   966.30), SIMDE_FLOAT32_C(   863.50), SIMDE_FLOAT32_C(   709.25), SIMDE_FLOAT32_C(   348.50),
                         SIMDE_FLOAT32_C(  -816.66), SIMDE_FLOAT32_C(  -662.92), SIMDE_FLOAT32_C(   913.50), SIMDE_FLOAT32_C(   301.72)),
      UINT16_C(64661),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -54.30), SIMDE_FLOAT32_C(  -771.33), SIMDE_FLOAT32_C(   -34.80), SIMDE_FLOAT32_C(   -55.97),
                         SIMDE_FLOAT32_C(  -654.29), SIMDE_FLOAT32_C(   768.64), SIMDE_FLOAT32_C(  -409.48), SIMDE_FLOAT32_C(   859.32),
                         SIMDE_FLOAT32_C(  -160.39), SIMDE_FLOAT32_C(  -988.34), SIMDE_FLOAT32_C(  -518.87), SIMDE_FLOAT32_C(  -778.28),
                         SIMDE_FLOAT32_C(   357.12), SIMDE_FLOAT32_C(   449.29), SIMDE_FLOAT32_C(   -46.50), SIMDE_FLOAT32_C(    93.99)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    54.30), SIMDE_FLOAT32_C(   771.33), SIMDE_FLOAT32_C(    34.80), SIMDE_FLOAT32_C(    55.97),
                         SIMDE_FLOAT32_C(   654.29), SIMDE_FLOAT32_C(   768.64), SIMDE_FLOAT32_C(  -763.16), SIMDE_FLOAT32_C(  -768.89),
                         SIMDE_FLOAT32_C(   160.39), SIMDE_FLOAT32_C(   863.50), SIMDE_FLOAT32_C(   709.25), SIMDE_FLOAT32_C(   778.28),
                         SIMDE_FLOAT32_C(  -816.66), SIMDE_FLOAT32_C(   449.29), SIMDE_FLOAT32_C(   913.50), SIMDE_FLOAT32_C(    93.99)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_abs_ps(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_abs_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -140.56), SIMDE_FLOAT64_C( -558.99),
                         SIMDE_FLOAT64_C(  240.08), SIMDE_FLOAT64_C( -481.72),
                         SIMDE_FLOAT64_C(  489.35), SIMDE_FLOAT64_C(  686.76),
                         SIMDE_FLOAT64_C( -206.54), SIMDE_FLOAT64_C(  728.61)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  140.56), SIMDE_FLOAT64_C(  558.99),
                         SIMDE_FLOAT64_C(  240.08), SIMDE_FLOAT64_C(  481.72),
                         SIMDE_FLOAT64_C(  489.35), SIMDE_FLOAT64_C(  686.76),
                         SIMDE_FLOAT64_C(  206.54), SIMDE_FLOAT64_C(  728.61)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  747.74), SIMDE_FLOAT64_C( -874.37),
                         SIMDE_FLOAT64_C(  751.90), SIMDE_FLOAT64_C( -592.77),
                         SIMDE_FLOAT64_C( -708.81), SIMDE_FLOAT64_C(  252.42),
                         SIMDE_FLOAT64_C( -787.46), SIMDE_FLOAT64_C( -882.47)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  747.74), SIMDE_FLOAT64_C(  874.37),
                         SIMDE_FLOAT64_C(  751.90), SIMDE_FLOAT64_C(  592.77),
                         SIMDE_FLOAT64_C(  708.81), SIMDE_FLOAT64_C(  252.42),
                         SIMDE_FLOAT64_C(  787.46), SIMDE_FLOAT64_C(  882.47)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  388.54), SIMDE_FLOAT64_C( -184.67),
                         SIMDE_FLOAT64_C(  102.38), SIMDE_FLOAT64_C(  833.56),
                         SIMDE_FLOAT64_C( -722.29), SIMDE_FLOAT64_C( -441.84),
                         SIMDE_FLOAT64_C( -821.42), SIMDE_FLOAT64_C( -761.98)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  388.54), SIMDE_FLOAT64_C(  184.67),
                         SIMDE_FLOAT64_C(  102.38), SIMDE_FLOAT64_C(  833.56),
                         SIMDE_FLOAT64_C(  722.29), SIMDE_FLOAT64_C(  441.84),
                         SIMDE_FLOAT64_C(  821.42), SIMDE_FLOAT64_C(  761.98)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(   76.28), SIMDE_FLOAT64_C( -319.23),
                         SIMDE_FLOAT64_C(  655.09), SIMDE_FLOAT64_C(  773.21),
                         SIMDE_FLOAT64_C( -928.32), SIMDE_FLOAT64_C(  -25.13),
                         SIMDE_FLOAT64_C( -847.53), SIMDE_FLOAT64_C(  859.40)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   76.28), SIMDE_FLOAT64_C(  319.23),
                         SIMDE_FLOAT64_C(  655.09), SIMDE_FLOAT64_C(  773.21),
                         SIMDE_FLOAT64_C(  928.32), SIMDE_FLOAT64_C(   25.13),
                         SIMDE_FLOAT64_C(  847.53), SIMDE_FLOAT64_C(  859.40)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  -40.31), SIMDE_FLOAT64_C(  866.84),
                         SIMDE_FLOAT64_C( -660.11), SIMDE_FLOAT64_C(  121.17),
                         SIMDE_FLOAT64_C(  988.31), SIMDE_FLOAT64_C( -622.26),
                         SIMDE_FLOAT64_C(  206.00), SIMDE_FLOAT64_C(  520.48)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   40.31), SIMDE_FLOAT64_C(  866.84),
                         SIMDE_FLOAT64_C(  660.11), SIMDE_FLOAT64_C(  121.17),
                         SIMDE_FLOAT64_C(  988.31), SIMDE_FLOAT64_C(  622.26),
                         SIMDE_FLOAT64_C(  206.00), SIMDE_FLOAT64_C(  520.48)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -645.61), SIMDE_FLOAT64_C(  594.77),
                         SIMDE_FLOAT64_C( -171.69), SIMDE_FLOAT64_C(  108.08),
                         SIMDE_FLOAT64_C(   -7.24), SIMDE_FLOAT64_C(  885.82),
                         SIMDE_FLOAT64_C(  296.84), SIMDE_FLOAT64_C( -408.70)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  645.61), SIMDE_FLOAT64_C(  594.77),
                         SIMDE_FLOAT64_C(  171.69), SIMDE_FLOAT64_C(  108.08),
                         SIMDE_FLOAT64_C(    7.24), SIMDE_FLOAT64_C(  885.82),
                         SIMDE_FLOAT64_C(  296.84), SIMDE_FLOAT64_C(  408.70)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -838.06), SIMDE_FLOAT64_C( -901.25),
                         SIMDE_FLOAT64_C(  -89.58), SIMDE_FLOAT64_C(  539.88),
                         SIMDE_FLOAT64_C(   88.35), SIMDE_FLOAT64_C(  773.77),
                         SIMDE_FLOAT64_C( -729.20), SIMDE_FLOAT64_C( -254.72)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  838.06), SIMDE_FLOAT64_C(  901.25),
                         SIMDE_FLOAT64_C(   89.58), SIMDE_FLOAT64_C(  539.88),
                         SIMDE_FLOAT64_C(   88.35), SIMDE_FLOAT64_C(  773.77),
                         SIMDE_FLOAT64_C(  729.20), SIMDE_FLOAT64_C(  254.72)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  530.77), SIMDE_FLOAT64_C(  966.97),
                         SIMDE_FLOAT64_C(  -63.51), SIMDE_FLOAT64_C(  360.07),
                         SIMDE_FLOAT64_C( -846.61), SIMDE_FLOAT64_C( -749.79),
                         SIMDE_FLOAT64_C(  510.77), SIMDE_FLOAT64_C( -104.12)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  530.77), SIMDE_FLOAT64_C(  966.97),
                         SIMDE_FLOAT64_C(   63.51), SIMDE_FLOAT64_C(  360.07),
                         SIMDE_FLOAT64_C(  846.61), SIMDE_FLOAT64_C(  749.79),
                         SIMDE_FLOAT64_C(  510.77), SIMDE_FLOAT64_C(  104.12)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_abs_pd(test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_abs_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -874.37), SIMDE_FLOAT64_C( -592.77),
                         SIMDE_FLOAT64_C(  252.42), SIMDE_FLOAT64_C( -882.47),
                         SIMDE_FLOAT64_C( -558.99), SIMDE_FLOAT64_C( -481.72),
                         SIMDE_FLOAT64_C(  686.76), SIMDE_FLOAT64_C(  728.61)),
      UINT8_C( 67),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  747.74), SIMDE_FLOAT64_C(  751.90),
                         SIMDE_FLOAT64_C( -708.81), SIMDE_FLOAT64_C( -787.46),
                         SIMDE_FLOAT64_C( -140.56), SIMDE_FLOAT64_C(  240.08),
                         SIMDE_FLOAT64_C(  489.35), SIMDE_FLOAT64_C( -206.54)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -874.37), SIMDE_FLOAT64_C(  751.90),
                         SIMDE_FLOAT64_C(  252.42), SIMDE_FLOAT64_C( -882.47),
                         SIMDE_FLOAT64_C( -558.99), SIMDE_FLOAT64_C( -481.72),
                         SIMDE_FLOAT64_C(  489.35), SIMDE_FLOAT64_C(  206.54)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(   76.28), SIMDE_FLOAT64_C(  655.09),
                         SIMDE_FLOAT64_C( -928.32), SIMDE_FLOAT64_C( -847.53),
                         SIMDE_FLOAT64_C(  388.54), SIMDE_FLOAT64_C(  102.38),
                         SIMDE_FLOAT64_C( -722.29), SIMDE_FLOAT64_C( -821.42)),
      UINT8_C(153),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  520.48), SIMDE_FLOAT64_C( -319.23),
                         SIMDE_FLOAT64_C(  773.21), SIMDE_FLOAT64_C(  -25.13),
                         SIMDE_FLOAT64_C(  859.40), SIMDE_FLOAT64_C( -184.67),
                         SIMDE_FLOAT64_C(  833.56), SIMDE_FLOAT64_C( -441.84)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  520.48), SIMDE_FLOAT64_C(  655.09),
                         SIMDE_FLOAT64_C( -928.32), SIMDE_FLOAT64_C(   25.13),
                         SIMDE_FLOAT64_C(  859.40), SIMDE_FLOAT64_C(  102.38),
                         SIMDE_FLOAT64_C( -722.29), SIMDE_FLOAT64_C(  441.84)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -254.72), SIMDE_FLOAT64_C(  594.77),
                         SIMDE_FLOAT64_C(  108.08), SIMDE_FLOAT64_C(  885.82),
                         SIMDE_FLOAT64_C( -408.70), SIMDE_FLOAT64_C(  866.84),
                         SIMDE_FLOAT64_C(  121.17), SIMDE_FLOAT64_C( -622.26)),
      UINT8_C( 41),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -729.20), SIMDE_FLOAT64_C( -645.61),
                         SIMDE_FLOAT64_C( -171.69), SIMDE_FLOAT64_C(   -7.24),
                         SIMDE_FLOAT64_C(  296.84), SIMDE_FLOAT64_C(  -40.31),
                         SIMDE_FLOAT64_C( -660.11), SIMDE_FLOAT64_C(  988.31)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -254.72), SIMDE_FLOAT64_C(  594.77),
                         SIMDE_FLOAT64_C(  171.69), SIMDE_FLOAT64_C(  885.82),
                         SIMDE_FLOAT64_C(  296.84), SIMDE_FLOAT64_C(  866.84),
                         SIMDE_FLOAT64_C(  121.17), SIMDE_FLOAT64_C(  988.31)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -617.33), SIMDE_FLOAT64_C(  530.77),
                         SIMDE_FLOAT64_C(  -63.51), SIMDE_FLOAT64_C( -846.61),
                         SIMDE_FLOAT64_C(  510.77), SIMDE_FLOAT64_C( -838.06),
                         SIMDE_FLOAT64_C(  -89.58), SIMDE_FLOAT64_C(   88.35)),
      UINT8_C(208),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -506.16), SIMDE_FLOAT64_C( -947.60),
                         SIMDE_FLOAT64_C(  966.97), SIMDE_FLOAT64_C(  360.07),
                         SIMDE_FLOAT64_C( -749.79), SIMDE_FLOAT64_C( -104.12),
                         SIMDE_FLOAT64_C( -901.25), SIMDE_FLOAT64_C(  539.88)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  506.16), SIMDE_FLOAT64_C(  947.60),
                         SIMDE_FLOAT64_C(  -63.51), SIMDE_FLOAT64_C(  360.07),
                         SIMDE_FLOAT64_C(  510.77), SIMDE_FLOAT64_C( -838.06),
                         SIMDE_FLOAT64_C(  -89.58), SIMDE_FLOAT64_C(   88.35)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  943.89), SIMDE_FLOAT64_C(  695.54),
                         SIMDE_FLOAT64_C( -377.80), SIMDE_FLOAT64_C(  778.00),
                         SIMDE_FLOAT64_C( -489.06), SIMDE_FLOAT64_C( -749.41),
                         SIMDE_FLOAT64_C( -423.37), SIMDE_FLOAT64_C( -118.88)),
      UINT8_C( 52),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -212.57), SIMDE_FLOAT64_C(  651.63),
                         SIMDE_FLOAT64_C( -814.45), SIMDE_FLOAT64_C(  640.68),
                         SIMDE_FLOAT64_C(  377.67), SIMDE_FLOAT64_C(  933.74),
                         SIMDE_FLOAT64_C(  193.12), SIMDE_FLOAT64_C( -194.06)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  943.89), SIMDE_FLOAT64_C(  695.54),
                         SIMDE_FLOAT64_C(  814.45), SIMDE_FLOAT64_C(  640.68),
                         SIMDE_FLOAT64_C( -489.06), SIMDE_FLOAT64_C(  933.74),
                         SIMDE_FLOAT64_C( -423.37), SIMDE_FLOAT64_C( -118.88)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -613.16), SIMDE_FLOAT64_C(  -56.94),
                         SIMDE_FLOAT64_C(  525.11), SIMDE_FLOAT64_C(  814.48),
                         SIMDE_FLOAT64_C( -481.76), SIMDE_FLOAT64_C( -375.20),
                         SIMDE_FLOAT64_C(  199.14), SIMDE_FLOAT64_C(  833.96)),
      UINT8_C(108),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  536.10), SIMDE_FLOAT64_C( -393.36),
                         SIMDE_FLOAT64_C(  670.22), SIMDE_FLOAT64_C(  299.00),
                         SIMDE_FLOAT64_C(  676.51), SIMDE_FLOAT64_C(  528.75),
                         SIMDE_FLOAT64_C(  146.55), SIMDE_FLOAT64_C( -505.05)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -613.16), SIMDE_FLOAT64_C(  393.36),
                         SIMDE_FLOAT64_C(  670.22), SIMDE_FLOAT64_C(  814.48),
                         SIMDE_FLOAT64_C(  676.51), SIMDE_FLOAT64_C(  528.75),
                         SIMDE_FLOAT64_C(  199.14), SIMDE_FLOAT64_C(  833.96)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(   -8.58), SIMDE_FLOAT64_C(  714.91),
                         SIMDE_FLOAT64_C( -520.52), SIMDE_FLOAT64_C( -546.29),
                         SIMDE_FLOAT64_C( -104.88), SIMDE_FLOAT64_C( -744.23),
                         SIMDE_FLOAT64_C(  495.73), SIMDE_FLOAT64_C(  262.00)),
      UINT8_C(147),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  550.36), SIMDE_FLOAT64_C(  374.16),
                         SIMDE_FLOAT64_C( -355.51), SIMDE_FLOAT64_C(  -13.73),
                         SIMDE_FLOAT64_C( -787.44), SIMDE_FLOAT64_C(  979.47),
                         SIMDE_FLOAT64_C(  836.15), SIMDE_FLOAT64_C( -301.39)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  550.36), SIMDE_FLOAT64_C(  714.91),
                         SIMDE_FLOAT64_C( -520.52), SIMDE_FLOAT64_C(   13.73),
                         SIMDE_FLOAT64_C( -104.88), SIMDE_FLOAT64_C( -744.23),
                         SIMDE_FLOAT64_C(  836.15), SIMDE_FLOAT64_C(  301.39)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  515.30), SIMDE_FLOAT64_C(  896.28),
                         SIMDE_FLOAT64_C(  660.35), SIMDE_FLOAT64_C(  493.79),
                         SIMDE_FLOAT64_C( -941.83), SIMDE_FLOAT64_C(  535.05),
                         SIMDE_FLOAT64_C( -963.94), SIMDE_FLOAT64_C(  143.93)),
      UINT8_C( 75),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  520.16), SIMDE_FLOAT64_C(   61.51),
                         SIMDE_FLOAT64_C( -643.69), SIMDE_FLOAT64_C(  -16.59),
                         SIMDE_FLOAT64_C(  -29.44), SIMDE_FLOAT64_C( -567.95),
                         SIMDE_FLOAT64_C(   43.85), SIMDE_FLOAT64_C(  235.87)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  515.30), SIMDE_FLOAT64_C(   61.51),
                         SIMDE_FLOAT64_C(  660.35), SIMDE_FLOAT64_C(  493.79),
                         SIMDE_FLOAT64_C(   29.44), SIMDE_FLOAT64_C(  535.05),
                         SIMDE_FLOAT64_C(   43.85), SIMDE_FLOAT64_C(  235.87)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_abs_pd(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_add_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  struct {
    int32_t a[16];
    int32_t b[16];
    int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(  2023322181), -INT32_C(   153083711),  INT32_C(  1496228679), -INT32_C(  1879098037),  INT32_C(   556222349), -INT32_C(  1406744359),  INT32_C(   328840924),  INT32_C(   558351974),
         INT32_C(   363441491),  INT32_C(   353077710),  INT32_C(  2003712300),  INT32_C(    50752886), -INT32_C(  1926943052), -INT32_C(   767996938),  INT32_C(  1759891970), -INT32_C(   326488680) },
      { -INT32_C(  1476316198), -INT32_C(   390198084), -INT32_C(  1151325115),  INT32_C(  1321166490), -INT32_C(  1965235052),  INT32_C(   828118319), -INT32_C(  2137373976),  INT32_C(  1231823983),
         INT32_C(   334654807), -INT32_C(  1090801543),  INT32_C(  1987730396),  INT32_C(  1455765954),  INT32_C(  1289855261), -INT32_C(  1635893834), -INT32_C(   299952001), -INT32_C(  1825010884) },
      {  INT32_C(   795328917), -INT32_C(   543281795),  INT32_C(   344903564), -INT32_C(   557931547), -INT32_C(  1409012703), -INT32_C(   578626040), -INT32_C(  1808533052),  INT32_C(  1790175957),
         INT32_C(   698096298), -INT32_C(   737723833), -INT32_C(   303524600),  INT32_C(  1506518840), -INT32_C(   637087791),  INT32_C(  1891076524),  INT32_C(  1459939969),  INT32_C(  2143467732) } },
    { {  INT32_C(  1923492601), -INT32_C(  1238261286), -INT32_C(  1087525891),  INT32_C(    18215652),  INT32_C(  1229846163), -INT32_C(  1276589260),  INT32_C(   530712547), -INT32_C(  1951212910),
        -INT32_C(   553756668), -INT32_C(   141283334), -INT32_C(  1095319078),  INT32_C(  1186974643), -INT32_C(   158331710), -INT32_C(  1146521384),  INT32_C(   299584383),  INT32_C(   698191141) },
      { -INT32_C(   536372250), -INT32_C(  1529373494),  INT32_C(   291671389),  INT32_C(   441917784),  INT32_C(    84993837), -INT32_C(   557729185),  INT32_C(   737188869),  INT32_C(   257199401),
        -INT32_C(   219194328),  INT32_C(  1435944696), -INT32_C(  1402537901),  INT32_C(  1187429913),  INT32_C(    88922021), -INT32_C(  1763504751), -INT32_C(   759049303), -INT32_C(  1998449056) },
      {  INT32_C(  1387120351),  INT32_C(  1527332516), -INT32_C(   795854502),  INT32_C(   460133436),  INT32_C(  1314840000), -INT32_C(  1834318445),  INT32_C(  1267901416), -INT32_C(  1694013509),
        -INT32_C(   772950996),  INT32_C(  1294661362),  INT32_C(  1797110317), -INT32_C(  1920562740), -INT32_C(    69409689),  INT32_C(  1384941161), -INT32_C(   459464920), -INT32_C(  1300257915) } },
    { {  INT32_C(  1786433906), -INT32_C(   339799912),  INT32_C(   563553800), -INT32_C(  1989648668), -INT32_C(   963726283),  INT32_C(  1784443585), -INT32_C(  1506009531), -INT32_C(  1506927052),
        -INT32_C(  2012173840), -INT32_C(  1032597575), -INT32_C(   639431691), -INT32_C(  1637659799), -INT32_C(  1067126273), -INT32_C(  1456816029),  INT32_C(   307193822),  INT32_C(  1975025029) },
      { -INT32_C(   520239066), -INT32_C(  1918733928), -INT32_C(   446200452), -INT32_C(   796669231),  INT32_C(   529655739), -INT32_C(  2033665113), -INT32_C(  1466427614), -INT32_C(  1155706476),
        -INT32_C(  1315235047),  INT32_C(   138362252), -INT32_C(  1813141822),  INT32_C(   728002672), -INT32_C(    28641961), -INT32_C(   746319184), -INT32_C(  1099227863), -INT32_C(  2022074258) },
      {  INT32_C(  1266194840),  INT32_C(  2036433456),  INT32_C(   117353348),  INT32_C(  1508649397), -INT32_C(   434070544), -INT32_C(   249221528),  INT32_C(  1322530151),  INT32_C(  1632333768),
         INT32_C(   967558409), -INT32_C(   894235323),  INT32_C(  1842393783), -INT32_C(   909657127), -INT32_C(  1095768234),  INT32_C(  2091832083), -INT32_C(   792034041), -INT32_C(    47049229) } },
    { {  INT32_C(  1060705459),  INT32_C(   323450961), -INT32_C(  1901644770), -INT32_C(    71758940), -INT32_C(  1325792256),  INT32_C(  1082359318),  INT32_C(   167706267),  INT32_C(  1251047319),
        -INT32_C(   594883957),  INT32_C(  1626329410), -INT32_C(  1427204602), -INT32_C(  1582913631), -INT32_C(  1034772309), -INT32_C(  1174219490),  INT32_C(  1807941844),  INT32_C(    45438071) },
      {  INT32_C(  1625177886),  INT32_C(   398511377),  INT32_C(    96579172),  INT32_C(    27748182),  INT32_C(   650377479), -INT32_C(  1562327602),  INT32_C(  1007526853),  INT32_C(   373212152),
         INT32_C(   326573058),  INT32_C(  1311389674),  INT32_C(  1012133094),  INT32_C(  1530788435), -INT32_C(  1031732749), -INT32_C(  1939578426), -INT32_C(    53972476),  INT32_C(   923993909) },
      { -INT32_C(  1609083951),  INT32_C(   721962338), -INT32_C(  1805065598), -INT32_C(    44010758), -INT32_C(   675414777), -INT32_C(   479968284),  INT32_C(  1175233120),  INT32_C(  1624259471),
        -INT32_C(   268310899), -INT32_C(  1357248212), -INT32_C(   415071508), -INT32_C(    52125196), -INT32_C(  2066505058),  INT32_C(  1181169380),  INT32_C(  1753969368),  INT32_C(   969431980) } },
    { {  INT32_C(   223054371), -INT32_C(  1487178303), -INT32_C(  1243369631), -INT32_C(  1659887191), -INT32_C(   396390110), -INT32_C(   160119822),  INT32_C(  1794325813),  INT32_C(  1738671684),
         INT32_C(  1366683024), -INT32_C(   990261150),  INT32_C(   695852159),  INT32_C(   533105149),  INT32_C(   201860378),  INT32_C(   503479528), -INT32_C(    41355847), -INT32_C(  1956304133) },
      {  INT32_C(  2061359639),  INT32_C(   708761258), -INT32_C(  1336690766),  INT32_C(  1523521856),  INT32_C(   644273982),  INT32_C(   222586964),  INT32_C(  1493945694),  INT32_C(   266694903),
        -INT32_C(   192298422),  INT32_C(  1243531160), -INT32_C(  1090883202), -INT32_C(   937899382), -INT32_C(   168853855),  INT32_C(  1141060582), -INT32_C(   123859456), -INT32_C(   939031682) },
      { -INT32_C(  2010553286), -INT32_C(   778417045),  INT32_C(  1714906899), -INT32_C(   136365335),  INT32_C(   247883872),  INT32_C(    62467142), -INT32_C(  1006695789),  INT32_C(  2005366587),
         INT32_C(  1174384602),  INT32_C(   253270010), -INT32_C(   395031043), -INT32_C(   404794233),  INT32_C(    33006523),  INT32_C(  1644540110), -INT32_C(   165215303),  INT32_C(  1399631481) } },
    { { -INT32_C(   574844859), -INT32_C(   718808233), -INT32_C(   678223284), -INT32_C(  1918915604),  INT32_C(   260279849), -INT32_C(  1034647870),  INT32_C(   314241684), -INT32_C(  1160068747),
        -INT32_C(  1466460591), -INT32_C(  1099055503), -INT32_C(   862646048), -INT32_C(   463850309), -INT32_C(  2047550013), -INT32_C(   146323357), -INT32_C(  1358364102),  INT32_C(   359261123) },
      { -INT32_C(   339935111), -INT32_C(  1616299074),  INT32_C(   124468811),  INT32_C(   904643954),  INT32_C(    96133026),  INT32_C(  1643905575), -INT32_C(   955251452),  INT32_C(  1658616296),
         INT32_C(   944609913),  INT32_C(   551024341), -INT32_C(  1507376588), -INT32_C(  1428417784),  INT32_C(   447780594), -INT32_C(  1669616488), -INT32_C(  1704686414),  INT32_C(  2147237893) },
      { -INT32_C(   914779970),  INT32_C(  1959859989), -INT32_C(   553754473), -INT32_C(  1014271650),  INT32_C(   356412875),  INT32_C(   609257705), -INT32_C(   641009768),  INT32_C(   498547549),
        -INT32_C(   521850678), -INT32_C(   548031162),  INT32_C(  1924944660), -INT32_C(  1892268093), -INT32_C(  1599769419), -INT32_C(  1815939845),  INT32_C(  1231916780), -INT32_C(  1788468280) } },
    { { -INT32_C(  1346942502),  INT32_C(  1943047743), -INT32_C(   669321264), -INT32_C(    41683446),  INT32_C(   622277516), -INT32_C(  1849584929),  INT32_C(   606872862),  INT32_C(  1084434534),
        -INT32_C(  1309648270), -INT32_C(  1205485336), -INT32_C(  1030668361), -INT32_C(  1044442059),  INT32_C(   652662343), -INT32_C(  2017941400),  INT32_C(   866903245),  INT32_C(  2121551372) },
      { -INT32_C(  1875876696), -INT32_C(   616016604), -INT32_C(   912402028),  INT32_C(   881482989), -INT32_C(  1688506062), -INT32_C(   433974503),  INT32_C(    52088311), -INT32_C(  1014854117),
         INT32_C(   374584050), -INT32_C(  1678664953),  INT32_C(  1650757493),  INT32_C(   513273579),  INT32_C(  2025452127), -INT32_C(    60826875), -INT32_C(  1006667352), -INT32_C(   108625657) },
      {  INT32_C(  1072148098),  INT32_C(  1327031139), -INT32_C(  1581723292),  INT32_C(   839799543), -INT32_C(  1066228546),  INT32_C(  2011407864),  INT32_C(   658961173),  INT32_C(    69580417),
        -INT32_C(   935064220),  INT32_C(  1410817007),  INT32_C(   620089132), -INT32_C(   531168480), -INT32_C(  1616852826), -INT32_C(  2078768275), -INT32_C(   139764107),  INT32_C(  2012925715) } },
    { {  INT32_C(   974117171), -INT32_C(   371916684),  INT32_C(  2068593039), -INT32_C(  2019957976), -INT32_C(   637513003), -INT32_C(   707371219), -INT32_C(   543631912), -INT32_C(  1965547945),
         INT32_C(  1808132087),  INT32_C(  2002098919), -INT32_C(    51207724),  INT32_C(  1501793156),  INT32_C(   171148253), -INT32_C(  1159788062),  INT32_C(   899250142), -INT32_C(  1933545067) },
      {  INT32_C(  1089963352), -INT32_C(   206091233),  INT32_C(  1911532013),  INT32_C(   298480436), -INT32_C(   652476938), -INT32_C(   443287034),  INT32_C(   102378865), -INT32_C(   141370722),
         INT32_C(  2134346079), -INT32_C(  1015877930), -INT32_C(   885693801), -INT32_C(   874709035),  INT32_C(    61143037),  INT32_C(  1659386097),  INT32_C(    57148261),  INT32_C(  1039858397) },
      {  INT32_C(  2064080523), -INT32_C(   578007917), -INT32_C(   314842244), -INT32_C(  1721477540), -INT32_C(  1289989941), -INT32_C(  1150658253), -INT32_C(   441253047), -INT32_C(  2106918667),
        -INT32_C(   352489130),  INT32_C(   986220989), -INT32_C(   936901525),  INT32_C(   627084121),  INT32_C(   232291290),  INT32_C(   499598035),  INT32_C(   956398403), -INT32_C(   893686670) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_add_epi32(a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mask_add_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int32_t src[16];
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { { -INT32_C(  1958532398), -INT32_C(   733777631), -INT32_C(    61793464),  INT32_C(  1915617450),  INT32_C(   759754662),  INT32_C(   196190852), -INT32_C(    77082310), -INT32_C(   259049954),
        -INT32_C(  2105748895), -INT32_C(  1940471997), -INT32_C(  2071418662),  INT32_C(  1324791464),  INT32_C(   695943077),  INT32_C(   456404449),  INT32_C(   471309310),  INT32_C(   856467154) },
      UINT16_C(34936),
      {  INT32_C(  1554501556),  INT32_C(   715824517), -INT32_C(  1118624036),  INT32_C(  1373210451), -INT32_C(   177344477),  INT32_C(   237533590),  INT32_C(  1254743298), -INT32_C(  1653861911),
         INT32_C(  1727599584), -INT32_C(  1919900495),  INT32_C(   491513034), -INT32_C(  1905384341), -INT32_C(  1434199276),  INT32_C(  1454943060),  INT32_C(  1923121545),  INT32_C(  1930431890) },
      { -INT32_C(   774305504),  INT32_C(  2002741677), -INT32_C(  1181439411), -INT32_C(   498662706),  INT32_C(   848088029), -INT32_C(     7846794),  INT32_C(  1483876805),  INT32_C(  1456177718),
         INT32_C(   958964875),  INT32_C(  1538295565), -INT32_C(    15448783),  INT32_C(   635525959), -INT32_C(  1655214810),  INT32_C(  1989992369),  INT32_C(  1053691400),  INT32_C(   479566224) },
      { -INT32_C(  1958532398), -INT32_C(   733777631), -INT32_C(    61793464),  INT32_C(   874547745),  INT32_C(   670743552),  INT32_C(   229686796), -INT32_C(  1556347193), -INT32_C(   259049954),
        -INT32_C(  2105748895), -INT32_C(  1940471997), -INT32_C(  2071418662), -INT32_C(  1269858382),  INT32_C(   695943077),  INT32_C(   456404449),  INT32_C(   471309310), -INT32_C(  1884969182) } },
    { {  INT32_C(  1263910205),  INT32_C(  1973814596), -INT32_C(  1837843894),  INT32_C(  1018647829),  INT32_C(  1943604930), -INT32_C(   152472083),  INT32_C(   338999428), -INT32_C(  1892628143),
        -INT32_C(   891648634), -INT32_C(   717258613),  INT32_C(  1332196154), -INT32_C(   896852472),  INT32_C(   440230956),  INT32_C(  1578117082),  INT32_C(   812795103), -INT32_C(  1799380210) },
      UINT16_C(39208),
      {  INT32_C(  1520693330),  INT32_C(   992292367),  INT32_C(  1834312339),  INT32_C(  1758160265), -INT32_C(  1197916758),  INT32_C(   155998432),  INT32_C(   196914162), -INT32_C(  1671477942),
         INT32_C(  1173750326), -INT32_C(  1015014608),  INT32_C(   120641150), -INT32_C(   445580485),  INT32_C(   429721913),  INT32_C(  1394797153), -INT32_C(   547364971),  INT32_C(  1518059044) },
      { -INT32_C(   257985856),  INT32_C(   196354189),  INT32_C(   823387382),  INT32_C(   420971488), -INT32_C(   315444084),  INT32_C(   876696990),  INT32_C(  1477681204),  INT32_C(   515084126),
        -INT32_C(  1911664127),  INT32_C(  1754972786), -INT32_C(  2019906137), -INT32_C(  1130319568),  INT32_C(    78238309),  INT32_C(  1530456615), -INT32_C(   390837366), -INT32_C(   620337190) },
      {  INT32_C(  1263910205),  INT32_C(  1973814596), -INT32_C(  1837843894), -INT32_C(  2115835543),  INT32_C(  1943604930),  INT32_C(  1032695422),  INT32_C(   338999428), -INT32_C(  1892628143),
        -INT32_C(   737913801), -INT32_C(   717258613),  INT32_C(  1332196154), -INT32_C(  1575900053),  INT32_C(   507960222),  INT32_C(  1578117082),  INT32_C(   812795103),  INT32_C(   897721854) } },
    { {  INT32_C(   745149881),  INT32_C(  2123629783), -INT32_C(   519754063),  INT32_C(  1167959519), -INT32_C(  1622587784), -INT32_C(  1141145295), -INT32_C(  1482379316), -INT32_C(   813520362),
        -INT32_C(  1745097537), -INT32_C(  1592422160), -INT32_C(  1635640386),  INT32_C(   954408896), -INT32_C(  1747440538),  INT32_C(  2035471277), -INT32_C(  1742670206),  INT32_C(  1617404833) },
      UINT16_C(25487),
      { -INT32_C(   397433816),  INT32_C(   690041539),  INT32_C(   197196126),  INT32_C(  1317344204), -INT32_C(  1427725047), -INT32_C(   670347960), -INT32_C(  1554513232),  INT32_C(   928348431),
        -INT32_C(   517954531),  INT32_C(   889864663),  INT32_C(   104975162), -INT32_C(   413874466), -INT32_C(  1265485205),  INT32_C(   948739463), -INT32_C(  1344543585),  INT32_C(  2078683229) },
      { -INT32_C(  1403255083), -INT32_C(  2115934649),  INT32_C(   260514353),  INT32_C(  1425529832), -INT32_C(  1660385003), -INT32_C(   975858650), -INT32_C(   713772936), -INT32_C(  1236247583),
        -INT32_C(  1453151135),  INT32_C(  1143620371),  INT32_C(  1314173542), -INT32_C(  1549644915), -INT32_C(    96425260), -INT32_C(  1228991170), -INT32_C(  1500760891), -INT32_C(   262349681) },
      { -INT32_C(  1800688899), -INT32_C(  1425893110),  INT32_C(   457710479), -INT32_C(  1552093260), -INT32_C(  1622587784), -INT32_C(  1141145295), -INT32_C(  1482379316), -INT32_C(   307899152),
        -INT32_C(  1971105666),  INT32_C(  2033485034), -INT32_C(  1635640386),  INT32_C(   954408896), -INT32_C(  1747440538), -INT32_C(   280251707),  INT32_C(  1449662820),  INT32_C(  1617404833) } },
    { { -INT32_C(  1667645815),  INT32_C(  1759560706),  INT32_C(    62272630),  INT32_C(  1403410815),  INT32_C(  1112401411), -INT32_C(  1040708101), -INT32_C(   798522303), -INT32_C(   356465567),
        -INT32_C(  2071569790), -INT32_C(  1796446690),  INT32_C(   446145435), -INT32_C(     9552132),  INT32_C(   541178660),  INT32_C(   165755592),  INT32_C(   534333630), -INT32_C(  1895196148) },
      UINT16_C(36852),
      { -INT32_C(  1616167517), -INT32_C(  1600251525), -INT32_C(  1648303915), -INT32_C(   660102886), -INT32_C(   151486231),  INT32_C(   243597594), -INT32_C(  2027906927),  INT32_C(   991479448),
         INT32_C(  2145043204), -INT32_C(  1306560035),  INT32_C(  1934614361),  INT32_C(  1783363200), -INT32_C(  1855962249), -INT32_C(   694098619),  INT32_C(   375242877), -INT32_C(  1957595769) },
      {  INT32_C(   352988216), -INT32_C(    20501851), -INT32_C(  1972300023), -INT32_C(  2064335859),  INT32_C(  1159091200), -INT32_C(  1239697863), -INT32_C(    36931466),  INT32_C(   629677805),
        -INT32_C(   281308342), -INT32_C(   957545795),  INT32_C(   659578393),  INT32_C(   447431706), -INT32_C(   782253672), -INT32_C(   293045641), -INT32_C(   538225422), -INT32_C(  1140493198) },
      { -INT32_C(  1667645815),  INT32_C(  1759560706),  INT32_C(   674363358),  INT32_C(  1403410815),  INT32_C(  1007604969), -INT32_C(   996100269), -INT32_C(  2064838393),  INT32_C(  1621157253),
         INT32_C(  1863734862),  INT32_C(  2030861466), -INT32_C(  1700774542), -INT32_C(  2064172390),  INT32_C(   541178660),  INT32_C(   165755592),  INT32_C(   534333630),  INT32_C(  1196878329) } },
    { { -INT32_C(   995409913),  INT32_C(  1552586818),  INT32_C(   293854198), -INT32_C(  1205129697),  INT32_C(  1737067504), -INT32_C(   128642811), -INT32_C(   656981658), -INT32_C(  1131029323),
         INT32_C(  1602240540), -INT32_C(   809825575),  INT32_C(    98582245),  INT32_C(  1555893356), -INT32_C(  1664858473), -INT32_C(  1097590440),  INT32_C(   261516378),  INT32_C(  1707813704) },
      UINT16_C(19308),
      {  INT32_C(   692123069), -INT32_C(  1735983871), -INT32_C(  1674294716), -INT32_C(  1101346461),  INT32_C(  2110648373), -INT32_C(  1998415588),  INT32_C(   986556132), -INT32_C(   495525595),
        -INT32_C(   687032618), -INT32_C(   126905676),  INT32_C(  1066706140), -INT32_C(  1560416659), -INT32_C(    98579490),  INT32_C(  1216479844), -INT32_C(   830255192),  INT32_C(   129038641) },
      {  INT32_C(  1675607215),  INT32_C(   710626894),  INT32_C(  1600843762), -INT32_C(  1140758563), -INT32_C(  1766448846), -INT32_C(   874563293), -INT32_C(  1181130104),  INT32_C(   180439643),
         INT32_C(  1433313286), -INT32_C(   511718930), -INT32_C(  1774130759), -INT32_C(  2091761071), -INT32_C(  2045114013), -INT32_C(   900597438), -INT32_C(  1232802981),  INT32_C(  1002456373) },
      { -INT32_C(   995409913),  INT32_C(  1552586818), -INT32_C(    73450954),  INT32_C(  2052862272),  INT32_C(  1737067504),  INT32_C(  1421988415), -INT32_C(   194573972), -INT32_C(  1131029323),
         INT32_C(   746280668), -INT32_C(   638624606),  INT32_C(    98582245),  INT32_C(   642789566), -INT32_C(  1664858473), -INT32_C(  1097590440), -INT32_C(  2063058173),  INT32_C(  1707813704) } },
    { { -INT32_C(   745525531), -INT32_C(  1313599240),  INT32_C(  1246230009), -INT32_C(  1697736137), -INT32_C(   450828125),  INT32_C(  1018130913), -INT32_C(  1846398116),  INT32_C(  1573761656),
        -INT32_C(   651076127),  INT32_C(  1737155949),  INT32_C(   296866266),  INT32_C(   246120299),  INT32_C(  1223936871), -INT32_C(  1719360707),  INT32_C(  1328248534),  INT32_C(   179107881) },
      UINT16_C(56661),
      { -INT32_C(  1431315650), -INT32_C(  1028105637),  INT32_C(  1661709350),  INT32_C(   637308751),  INT32_C(   796141318),  INT32_C(  1966678303), -INT32_C(  1053287170), -INT32_C(   950050167),
        -INT32_C(  1737421251), -INT32_C(  1906627992),  INT32_C(   636577494), -INT32_C(    78975243),  INT32_C(   891993877), -INT32_C(   559258656),  INT32_C(   144761471), -INT32_C(  2117009596) },
      {  INT32_C(  1964654861),  INT32_C(  1090811243), -INT32_C(   798558757), -INT32_C(   104025629),  INT32_C(  1345255024), -INT32_C(   651241382), -INT32_C(    18690374), -INT32_C(   629165363),
         INT32_C(  1599117811), -INT32_C(   375368690),  INT32_C(   767166281),  INT32_C(   673613496), -INT32_C(   696757124), -INT32_C(   424630740),  INT32_C(  1122275957),  INT32_C(   924672836) },
      {  INT32_C(   533339211), -INT32_C(  1313599240),  INT32_C(   863150593), -INT32_C(  1697736137),  INT32_C(  2141396342),  INT32_C(  1018130913), -INT32_C(  1071977544),  INT32_C(  1573761656),
        -INT32_C(   138303440),  INT32_C(  1737155949),  INT32_C(  1403743775),  INT32_C(   594638253),  INT32_C(   195236753), -INT32_C(  1719360707),  INT32_C(  1267037428), -INT32_C(  1192336760) } },
    { {  INT32_C(   194407933),  INT32_C(   183842753), -INT32_C(   164122818), -INT32_C(  1323410123), -INT32_C(   578251087), -INT32_C(  1312606148),  INT32_C(   250914762),  INT32_C(   138744075),
         INT32_C(  1058266238),  INT32_C(  1363740691), -INT32_C(   330858057), -INT32_C(  1868667426),  INT32_C(   929900283),  INT32_C(   686371166), -INT32_C(   482943528),  INT32_C(  1827372014) },
      UINT16_C(65367),
      {  INT32_C(  1420493429),  INT32_C(  1659128167), -INT32_C(   845524625),  INT32_C(  1542816642),  INT32_C(  1312697184), -INT32_C(    21353817),  INT32_C(   812213545), -INT32_C(   806411175),
        -INT32_C(  1910269145),  INT32_C(  1425082340), -INT32_C(   618558632),  INT32_C(  1849038606), -INT32_C(   373525438), -INT32_C(   941066594),  INT32_C(   888689115), -INT32_C(  1677465739) },
      {  INT32_C(  1730881154), -INT32_C(  2034557907), -INT32_C(  1251877721),  INT32_C(   908302323), -INT32_C(  1440751861),  INT32_C(   812713813), -INT32_C(   832280232), -INT32_C(   748001199),
        -INT32_C(  1137011314),  INT32_C(  1480783281), -INT32_C(   988961838),  INT32_C(  1174089786),  INT32_C(  1693391631),  INT32_C(  2073321762),  INT32_C(   457832906), -INT32_C(   269503647) },
      { -INT32_C(  1143592713), -INT32_C(   375429740), -INT32_C(  2097402346), -INT32_C(  1323410123), -INT32_C(   128054677), -INT32_C(  1312606148), -INT32_C(    20066687),  INT32_C(   138744075),
         INT32_C(  1247686837), -INT32_C(  1389101675), -INT32_C(  1607520470), -INT32_C(  1271838904),  INT32_C(  1319866193),  INT32_C(  1132255168),  INT32_C(  1346522021), -INT32_C(  1946969386) } },
    { { -INT32_C(    89446071), -INT32_C(   246158049), -INT32_C(   894017392), -INT32_C(  1609518447), -INT32_C(   284819507),  INT32_C(   728406368), -INT32_C(   213470318), -INT32_C(  1327286937),
         INT32_C(  2125106783),  INT32_C(   208665980), -INT32_C(   271112866), -INT32_C(  1534072873),  INT32_C(  1200919782), -INT32_C(  1066205650),  INT32_C(   431274162),  INT32_C(  1305057262) },
      UINT16_C(29477),
      {  INT32_C(   935232863),  INT32_C(  1390103916),  INT32_C(   278491106),  INT32_C(   550505326), -INT32_C(  1304853308),  INT32_C(  1107231259), -INT32_C(   421344651),  INT32_C(  1672843268),
        -INT32_C(  2120584427), -INT32_C(  1546357055),  INT32_C(  1404268005),  INT32_C(  1030980473),  INT32_C(   602909704),  INT32_C(   610594478), -INT32_C(  1140176968), -INT32_C(   316686121) },
      { -INT32_C(   194069965),  INT32_C(   362234416),  INT32_C(   694766256), -INT32_C(   697901874),  INT32_C(   939087241), -INT32_C(    77898173),  INT32_C(  2092394149),  INT32_C(  1500108326),
        -INT32_C(  1068574576), -INT32_C(   891886310), -INT32_C(    17613008), -INT32_C(  1529587429), -INT32_C(   237187666), -INT32_C(   789825749), -INT32_C(  1018322019),  INT32_C(   169719418) },
      {  INT32_C(   741162898), -INT32_C(   246158049),  INT32_C(   973257362), -INT32_C(  1609518447), -INT32_C(   284819507),  INT32_C(  1029333086), -INT32_C(   213470318), -INT32_C(  1327286937),
         INT32_C(  1105808293),  INT32_C(  1856723931), -INT32_C(   271112866), -INT32_C(  1534072873),  INT32_C(   365722038), -INT32_C(   179231271),  INT32_C(  2136468309),  INT32_C(  1305057262) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi32(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_add_epi32(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_maskz_add_epi32 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask16 k;
    const int32_t a[16];
    const int32_t b[16];
    const int32_t r[16];
  } test_vec[] = {
    { UINT16_C(52979),
      {  INT32_C(  1318030952), -INT32_C(   938502652), -INT32_C(  1205717630),  INT32_C(  1648076236),  INT32_C(  1874746093),  INT32_C(   507402795), -INT32_C(   271937240), -INT32_C(   581761675),
        -INT32_C(  1758731373), -INT32_C(    77579399),  INT32_C(  1018397296),  INT32_C(   345959975),  INT32_C(  1954766153), -INT32_C(   527253065), -INT32_C(   925934509), -INT32_C(   190504095) },
      {  INT32_C(  2139869190),  INT32_C(  2071653131), -INT32_C(  1799934611),  INT32_C(  1688819227),  INT32_C(  1792552115),  INT32_C(  1095396078),  INT32_C(   654908102),  INT32_C(  1125887549),
        -INT32_C(  1966954626),  INT32_C(      343186), -INT32_C(  2070626967), -INT32_C(   957793005),  INT32_C(  1479590250), -INT32_C(   224822484), -INT32_C(   770006379), -INT32_C(   837470896) },
      { -INT32_C(   837067154),  INT32_C(  1133150479),  INT32_C(           0),  INT32_C(           0), -INT32_C(   627669088),  INT32_C(  1602798873),  INT32_C(   382970862),  INT32_C(   544125874),
         INT32_C(           0), -INT32_C(    77236213), -INT32_C(  1052229671), -INT32_C(   611833030),  INT32_C(           0),  INT32_C(           0), -INT32_C(  1695940888), -INT32_C(  1027974991) } },
    { UINT16_C(55260),
      {  INT32_C(   771752731), -INT32_C(  1410012863), -INT32_C(   687659861),  INT32_C(   885628063), -INT32_C(  1912151234), -INT32_C(   178513127),  INT32_C(    90486258),  INT32_C(   780260115),
         INT32_C(   408715991),  INT32_C(   381898859),  INT32_C(   351127156), -INT32_C(  1605847198), -INT32_C(  1288810598),  INT32_C(  1571392106),  INT32_C(  1382157631), -INT32_C(  1199512351) },
      { -INT32_C(   774841242), -INT32_C(  1578593492), -INT32_C(  1145711271),  INT32_C(   660340108), -INT32_C(  1210414772),  INT32_C(  1393853203),  INT32_C(  1923446417), -INT32_C(  1070979494),
         INT32_C(   798161410), -INT32_C(   422544755), -INT32_C(   593394353), -INT32_C(   821822334), -INT32_C(  1735991931), -INT32_C(   219440543),  INT32_C(  1801752848), -INT32_C(  1188327753) },
      {  INT32_C(           0),  INT32_C(           0), -INT32_C(  1833371132),  INT32_C(  1545968171),  INT32_C(  1172401290),  INT32_C(           0),  INT32_C(  2013932675), -INT32_C(   290719379),
         INT32_C(  1206877401), -INT32_C(    40645896), -INT32_C(   242267197),  INT32_C(           0),  INT32_C(  1270164767),  INT32_C(           0), -INT32_C(  1111056817),  INT32_C(  1907127192) } },
    { UINT16_C(48520),
      { -INT32_C(  1067213763),  INT32_C(   495937176), -INT32_C(  1531636413), -INT32_C(  1080647249), -INT32_C(   383059406),  INT32_C(   279074440),  INT32_C(  1260751635), -INT32_C(  2116935613),
         INT32_C(  1413559740), -INT32_C(   562966373), -INT32_C(  1803343899), -INT32_C(    95217208), -INT32_C(  1662812652), -INT32_C(   408058412),  INT32_C(  1412616720), -INT32_C(  1344994061) },
      { -INT32_C(   737929671), -INT32_C(   877431322),  INT32_C(  1683961500),  INT32_C(  1667150415),  INT32_C(    67125552), -INT32_C(   672354873),  INT32_C(  1915428479),  INT32_C(  1545732131),
        -INT32_C(    63887850),  INT32_C(   952624283),  INT32_C(  1771841050),  INT32_C(   164494297),  INT32_C(    51301692), -INT32_C(   103024006),  INT32_C(   996935192),  INT32_C(   496537095) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(   586503166),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   571203482),
         INT32_C(  1349671890),  INT32_C(           0), -INT32_C(    31502849),  INT32_C(    69277089), -INT32_C(  1611510960), -INT32_C(   511082418),  INT32_C(           0), -INT32_C(   848456966) } },
    { UINT16_C(51636),
      { -INT32_C(   516938744),  INT32_C(  1542126879),  INT32_C(  1147140298),  INT32_C(   188627698),  INT32_C(  1195813440), -INT32_C(   328868296),  INT32_C(  1413185447),  INT32_C(  1746649952),
         INT32_C(   105467111),  INT32_C(   341914697),  INT32_C(   525910060),  INT32_C(   992646906),  INT32_C(  2021814336), -INT32_C(   161159345), -INT32_C(   951345050),  INT32_C(  1244620387) },
      { -INT32_C(    61834830), -INT32_C(   653217363),  INT32_C(  1828218994),  INT32_C(  1067918079), -INT32_C(   491246957),  INT32_C(  2027428881),  INT32_C(   524231612),  INT32_C(  1013542538),
        -INT32_C(  1808221721), -INT32_C(   579975061), -INT32_C(  1337366863),  INT32_C(   485486985),  INT32_C(   754886427), -INT32_C(  2136680764), -INT32_C(  2069830662),  INT32_C(   968886610) },
      {  INT32_C(           0),  INT32_C(           0), -INT32_C(  1319608004),  INT32_C(           0),  INT32_C(   704566483),  INT32_C(  1698560585),  INT32_C(           0), -INT32_C(  1534774806),
        -INT32_C(  1702754610),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1478133891),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1273791584), -INT32_C(  2081460299) } },
    { UINT16_C(63939),
      {  INT32_C(   732058786),  INT32_C(  1615303237), -INT32_C(    41073351),  INT32_C(   377368860),  INT32_C(  1738153493), -INT32_C(   358589913), -INT32_C(  1793561005),  INT32_C(  1300702122),
        -INT32_C(  1116198280), -INT32_C(   182533956),  INT32_C(   569617157), -INT32_C(   248024612), -INT32_C(  1235693169),  INT32_C(  2141321516),  INT32_C(   303348071),  INT32_C(  1432329437) },
      { -INT32_C(  2112694330), -INT32_C(  1653133161), -INT32_C(  1195480357),  INT32_C(  1789523675), -INT32_C(   215940409),  INT32_C(  1651753723),  INT32_C(  1484031867), -INT32_C(   374484189),
         INT32_C(  1114357931), -INT32_C(   857742352),  INT32_C(   696557133),  INT32_C(  1536372116),  INT32_C(   709866543), -INT32_C(   225590666),  INT32_C(  1833566537), -INT32_C(  2141783851) },
      { -INT32_C(  1380635544), -INT32_C(    37829924),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   309529138),  INT32_C(   926217933),
        -INT32_C(     1840349),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1288347504), -INT32_C(   525826626),  INT32_C(  1915730850),  INT32_C(  2136914608), -INT32_C(   709454414) } },
    { UINT16_C(49848),
      { -INT32_C(   736364480),  INT32_C(  1429188390), -INT32_C(   629113245), -INT32_C(  1966338752), -INT32_C(   470346226), -INT32_C(   966570738), -INT32_C(  1267784177),  INT32_C(   145220552),
         INT32_C(    48022236), -INT32_C(   715715727), -INT32_C(   894445686), -INT32_C(   212567068), -INT32_C(  1596568687), -INT32_C(  1469695335),  INT32_C(   677238112), -INT32_C(  1792015175) },
      {  INT32_C(   949423302), -INT32_C(  1592922601), -INT32_C(  1435714362), -INT32_C(   929185737), -INT32_C(  1519881204),  INT32_C(   239980462),  INT32_C(  1563863716),  INT32_C(  1978820270),
        -INT32_C(  1985115790),  INT32_C(  1043053176), -INT32_C(  1377265802),  INT32_C(  1668646487), -INT32_C(  1475813638),  INT32_C(  1370904237), -INT32_C(  1347425280), -INT32_C(  1004232366) },
      {  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1399442807), -INT32_C(  1990227430), -INT32_C(   726590276),  INT32_C(           0),  INT32_C(  2124040822),
         INT32_C(           0),  INT32_C(   327337449),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   670187168),  INT32_C(  1498719755) } },
    { UINT16_C(53547),
      {  INT32_C(  1739507983),  INT32_C(  1237984079), -INT32_C(  1359883519),  INT32_C(   687908649), -INT32_C(   438784109),  INT32_C(  2074737744),  INT32_C(  1478424525),  INT32_C(  2136604527),
         INT32_C(   417728457),  INT32_C(   744665131), -INT32_C(  1394912381), -INT32_C(  1898521605), -INT32_C(   629887350),  INT32_C(  2018909611), -INT32_C(  2066648044),  INT32_C(  1023617652) },
      {  INT32_C(  1565911346),  INT32_C(   495564697),  INT32_C(   113861643), -INT32_C(   913006785), -INT32_C(   106690482),  INT32_C(   980548134), -INT32_C(   490847634), -INT32_C(  1625308819),
         INT32_C(  1157395882),  INT32_C(   929137964),  INT32_C(   691874538),  INT32_C(   418632394),  INT32_C(     1152986), -INT32_C(    12877167),  INT32_C(   853735877),  INT32_C(  1708196283) },
      { -INT32_C(   989547967),  INT32_C(  1733548776),  INT32_C(           0), -INT32_C(   225098136),  INT32_C(           0), -INT32_C(  1239681418),  INT32_C(           0),  INT32_C(           0),
         INT32_C(  1575124339),  INT32_C(           0),  INT32_C(           0),  INT32_C(           0), -INT32_C(   628734364),  INT32_C(           0), -INT32_C(  1212912167), -INT32_C(  1563153361) } },
    { UINT16_C(52598),
      { -INT32_C(    10086347), -INT32_C(  1005102614), -INT32_C(  2117785360),  INT32_C(  1870659754), -INT32_C(  1264491783), -INT32_C(   635800988), -INT32_C(  1837251777),  INT32_C(    63854798),
         INT32_C(  1510093936),  INT32_C(  2099124621), -INT32_C(   335617215), -INT32_C(   581206045),  INT32_C(  1167195361), -INT32_C(  1373590673),  INT32_C(  1027644783),  INT32_C(  1698697205) },
      { -INT32_C(   775994813), -INT32_C(  1672552869), -INT32_C(  1517859391), -INT32_C(  1383931188),  INT32_C(  1324553183),  INT32_C(   788272063), -INT32_C(  1502921296), -INT32_C(  1895060660),
         INT32_C(  1214303213),  INT32_C(  1793372073), -INT32_C(   938513412),  INT32_C(   762679630),  INT32_C(  1685809317),  INT32_C(   747796347),  INT32_C(    13827508), -INT32_C(  1785668184) },
      {  INT32_C(           0),  INT32_C(  1617311813),  INT32_C(   659322545),  INT32_C(           0),  INT32_C(    60061400),  INT32_C(   152471075),  INT32_C(   954794223),  INT32_C(           0),
        -INT32_C(  1570570147),  INT32_C(           0), -INT32_C(  1274130627),  INT32_C(   181473585),  INT32_C(           0),  INT32_C(           0),  INT32_C(  1041472291), -INT32_C(    86970979) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi32(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi32(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_add_epi32(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i32x16(r, simde_mm512_loadu_epi32(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_add_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t a[8];
    const int64_t b[8];
    const int64_t r[8];
  } test_vec[] = {
    { {  INT64_C( 5896694048212443049),  INT64_C( 9000266092577364175),  INT64_C( 7152103947590144860),  INT64_C( 2989496802120002433),
         INT64_C( 7710631622698424498), -INT64_C( 3633641352504339518),  INT64_C( 4274662313843579209),  INT64_C( 6826149357438294289) },
      { -INT64_C(  164363539517042510), -INT64_C( 8691483022440823005),  INT64_C( 9220214710722807807), -INT64_C( 2401321110146592095),
        -INT64_C( 1183069234730910884),  INT64_C( 5562197665816815723),  INT64_C( 1124204932795639468), -INT64_C( 3326459772972193332) },
      {  INT64_C( 5732330508695400539),  INT64_C(  308783070136541170), -INT64_C( 2074425415396598949),  INT64_C(  588175691973410338),
         INT64_C( 6527562387967513614),  INT64_C( 1928556313312476205),  INT64_C( 5398867246639218677),  INT64_C( 3499689584466100957) } },
    { {  INT64_C( 5873253456280027845), -INT64_C( 7547985786885765724),  INT64_C(  958785414761629392),  INT64_C( 8879370812030102515),
         INT64_C( 4762149369024389598),  INT64_C(  798391932315570322),  INT64_C(  147097132267652539), -INT64_C( 3786220049007964093) },
      { -INT64_C( 6906651495143342010), -INT64_C( 8287694440800363594),  INT64_C( 2452371479376447222),  INT64_C( 6726846324779217826),
         INT64_C( 5025569198072523023),  INT64_C(  225235015271111619), -INT64_C( 2487938372584494983),  INT64_C( 3745242421369017476) },
      { -INT64_C( 1033398038863314165),  INT64_C( 2611063846023422298),  INT64_C( 3411156894138076614), -INT64_C( 2840526936900231275),
        -INT64_C( 8659025506612638995),  INT64_C( 1023626947586681941), -INT64_C( 2340841240316842444), -INT64_C(   40977627638946617) } },
    { {  INT64_C( 7387045378804169392),  INT64_C( 7348215347083393770), -INT64_C( 7432287296260602942), -INT64_C( 4746462990122596405),
         INT64_C( 7669772552352133735),  INT64_C( 5083821277037292091),  INT64_C( 5407731889132030559), -INT64_C( 3084302269135830938) },
      { -INT64_C( 1461330460425637939), -INT64_C( 3061426891990558023),  INT64_C( 5599758734307477482),  INT64_C( 1862788523933954198),
        -INT64_C( 4557890179386853341),  INT64_C( 6666876053459161657), -INT64_C( 4937214972124475832),  INT64_C( 6111833508638834029) },
      {  INT64_C( 5925714918378531453),  INT64_C( 4286788455092835747), -INT64_C( 1832528561953125460), -INT64_C( 2883674466188642207),
         INT64_C( 3111882372965280394), -INT64_C( 6696046743213097868),  INT64_C(  470516917007554727),  INT64_C( 3027531239503003091) } },
    { { -INT64_C( 7991663547628636080), -INT64_C( 2555292973839346502),  INT64_C( 4212139769629200532),  INT64_C( 1966319092590916547),
         INT64_C( 1506042142180667901), -INT64_C( 9075093079022557283),  INT64_C( 7143746535270586651), -INT64_C( 2897889499141433630) },
      {  INT64_C( 6953298079720946194), -INT64_C( 6437157297342791622),  INT64_C( 8555627167819425208), -INT64_C( 4217080419303877945),
        -INT64_C(  916288211658955227),  INT64_C( 3576356706803505520), -INT64_C( 6218269451284303702), -INT64_C( 1742958193093650601) },
      { -INT64_C( 1038365467907689886), -INT64_C( 8992450271182138124), -INT64_C( 5678977136260925876), -INT64_C( 2250761326712961398),
         INT64_C(  589753930521712674), -INT64_C( 5498736372219051763),  INT64_C(  925477083986282949), -INT64_C( 4640847692235084231) } },
    { {  INT64_C( 2312342974665588586), -INT64_C( 6729576343545367823), -INT64_C( 4578026214523853331), -INT64_C( 1074221180203122067),
        -INT64_C( 1195656230424156519),  INT64_C( 3385005156404397150), -INT64_C( 2575086539621213671),  INT64_C( 4660983342689947190) },
      {  INT64_C( 5690001192450114569),  INT64_C( 6765706558176579445), -INT64_C( 8375529455621185160), -INT64_C(  280638300551000014),
         INT64_C(  791220201005032380), -INT64_C( 5337991249511014582),  INT64_C( 2478776332018633862),  INT64_C( 7142732816633802545) },
      {  INT64_C( 8002344167115703155),  INT64_C(   36130214631211622),  INT64_C( 5493188403564513125), -INT64_C( 1354859480754122081),
        -INT64_C(  404436029419124139), -INT64_C( 1952986093106617432), -INT64_C(   96310207602579809), -INT64_C( 6643027914385801881) } },
    { {  INT64_C( 7590546826509362360),  INT64_C( 4799960603843565481), -INT64_C( 3764863488869189202), -INT64_C( 8485326154395304909),
         INT64_C(  125025846558150196),  INT64_C( 4919203572335817541),  INT64_C( 1811753159855661758),  INT64_C( 6393760326532469855) },
      {  INT64_C(  421764692607537793),  INT64_C(  501672283606598428), -INT64_C( 2545232539499374162),  INT64_C( 6202803407104615064),
         INT64_C( 4980991260009414746),  INT64_C( 2385761506151573452), -INT64_C( 7628987825040033081),  INT64_C( 5850290225876708869) },
      {  INT64_C( 8012311519116900153),  INT64_C( 5301632887450163909), -INT64_C( 6310096028368563364), -INT64_C( 2282522747290689845),
         INT64_C( 5106017106567564942),  INT64_C( 7304965078487390993), -INT64_C( 5817234665184371323), -INT64_C( 6202693521300372892) } },
    { {  INT64_C( 3861145535682141991),  INT64_C( 4704120286579625139),  INT64_C( 7310649930581147103),  INT64_C( 6132617560052451027),
        -INT64_C( 4220933801323952434), -INT64_C(  467755223424977465), -INT64_C( 9153765608270723279),  INT64_C( 8400169494660134417) },
      {  INT64_C( 4306311459952605676),  INT64_C( 1432426031515283149),  INT64_C( 1311843823099622919), -INT64_C( 3392084749394608174),
         INT64_C( 8992722739203577885),  INT64_C( 5779599678188505408),  INT64_C( 5119810430763850234),  INT64_C( 6804001435340987831) },
      {  INT64_C( 8167456995634747667),  INT64_C( 6136546318094908288),  INT64_C( 8622493753680770022),  INT64_C( 2740532810657842853),
         INT64_C( 4771788937879625451),  INT64_C( 5311844454763527943), -INT64_C( 4033955177506873045), -INT64_C( 3242573143708429368) } },
    { {  INT64_C(  896142439321910083),  INT64_C( 1197503498379252485), -INT64_C( 7856220743107108291), -INT64_C( 6406762567310591882),
        -INT64_C( 4058014011976186410),  INT64_C( 9080299469053222364), -INT64_C( 7078487466013880490),  INT64_C( 7199966683762914017) },
      {  INT64_C( 7780449457883481456),  INT64_C( 1824347912971095698), -INT64_C( 8415522727832944271),  INT64_C( 7418198203865008897),
         INT64_C( 4556395623730444353),  INT64_C( 7889010207409543840), -INT64_C( 3090529460147599642), -INT64_C( 1719435354305139514) },
      {  INT64_C( 8676591897205391539),  INT64_C( 3021851411350348183),  INT64_C( 2175000602769499054),  INT64_C( 1011435636554417015),
         INT64_C(  498381611754257943), -INT64_C( 1477434397246785412),  INT64_C( 8277727147548071484),  INT64_C( 5480531329457774503) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_add_epi64(a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_mask_add_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const int64_t src[16];
    const simde__mmask8 k;
    const int64_t a[16];
    const int64_t b[16];
    const int64_t r[16];
  } test_vec[] = {
    { {  INT64_C( 1637095571448452370), -INT64_C( 5010656489973054228),  INT64_C( 8096577095910365922),  INT64_C( 3926524403043278656),
         INT64_C( 1692383872749537703), -INT64_C( 7546382928320257262), -INT64_C( 6602394267959061769),  INT64_C( 2598636899144412341) },
      UINT8_C( 63),
      {  INT64_C( 3064278465490095078), -INT64_C( 1676615221711466009), -INT64_C( 5333405411065419087), -INT64_C( 3810230114928306775),
        -INT64_C( 1075017760364328478),  INT64_C( 2095704811519734998), -INT64_C(  299103093840977638),  INT64_C( 5108483185182444596) },
      { -INT64_C( 4648200900296301693), -INT64_C( 4629136759825157063), -INT64_C( 5066623773317061022),  INT64_C(  536973459407932105),
         INT64_C( 2486008889004565721), -INT64_C( 3554876755438703545),  INT64_C( 1436140625484484016),  INT64_C( 5713013447801749692) },
      { -INT64_C( 1583922434806206615), -INT64_C( 6305751981536623072),  INT64_C( 8046714889327071507), -INT64_C( 3273256655520374670),
         INT64_C( 1410991128640237243), -INT64_C( 1459171943918968547), -INT64_C( 6602394267959061769),  INT64_C( 2598636899144412341) } },
    { {  INT64_C( 3991194155833482583),  INT64_C( 7365863369617845245), -INT64_C( 6217348007288128678),  INT64_C(  148675600489051978),
        -INT64_C( 5748943111581392624), -INT64_C( 9084794923389396527),  INT64_C( 6675703621262608398), -INT64_C( 3278006165881122860) },
      UINT8_C( 79),
      {  INT64_C( 7078729567351001797),  INT64_C( 3976127268296180429),  INT64_C( 6764870419675162927), -INT64_C( 6394845513855835965),
        -INT64_C( 4382478565492243517), -INT64_C( 1011318967947184367), -INT64_C( 6311831277423214532),  INT64_C( 4236157129718335039) },
      {  INT64_C( 4311691048566315805),  INT64_C(  693258357862808300), -INT64_C( 5061911316372677582),  INT64_C( 5814609134873172224),
        -INT64_C( 2890120277031405697),  INT64_C(  273070111211249652),  INT64_C( 6337650268323962303), -INT64_C(  956874791454847436) },
      { -INT64_C( 7056323457792234014),  INT64_C( 4669385626158988729),  INT64_C( 1702959103302485345), -INT64_C(  580236378982663741),
        -INT64_C( 5748943111581392624), -INT64_C( 9084794923389396527),  INT64_C(   25818990900747771), -INT64_C( 3278006165881122860) } },
    { {  INT64_C( 5402490335443754038),  INT64_C( 7004459312563912287),  INT64_C( 6873494867043635124), -INT64_C( 1746693303777676963),
         INT64_C( 4412405986682822043), -INT64_C( 3277034903515019135),  INT64_C(  971442364987875570),  INT64_C( 6704628126445290098) },
      UINT8_C(252),
      { -INT64_C( 3199500174101950700),  INT64_C( 7068896874256776325), -INT64_C(  840732006067128670), -INT64_C( 7451465598208935429),
         INT64_C(  990872770473652578),  INT64_C( 1777037797882114565), -INT64_C( 3158904769779877244),  INT64_C( 6189642379913322441) },
      { -INT64_C( 2014230672746244489),  INT64_C( 1316129223197016245),  INT64_C( 6358081634684124815),  INT64_C( 2925524125942721361),
        -INT64_C( 7645647755206468574),  INT64_C( 8364597264550793588),  INT64_C( 2245635740289228099),  INT64_C( 9124008468664275140) },
      {  INT64_C( 5402490335443754038),  INT64_C( 7004459312563912287),  INT64_C( 5517349628616996145), -INT64_C( 4525941472266214068),
        -INT64_C( 6654774984732815996), -INT64_C( 8305109011276643463), -INT64_C(  913269029490649145), -INT64_C( 3133093225131954035) } },
    { {  INT64_C( 7876626396527707865),  INT64_C( 6327703798935457910), -INT64_C( 8444156093278868254),  INT64_C(  792525990600389412),
         INT64_C( 6542343655737491300), -INT64_C( 6733297332257473758),  INT64_C( 3495113324412254258), -INT64_C( 8894133035806391978) },
      UINT8_C( 48),
      { -INT64_C( 1618640895730195884),  INT64_C(  566130083197796387), -INT64_C( 3091365637900741985), -INT64_C( 8802714067975954187),
         INT64_C( 8931894081495034460), -INT64_C( 8463108217014804938), -INT64_C( 2811541516088205358), -INT64_C( 4054272745087766267) },
      {  INT64_C( 6129898402509662270),  INT64_C(  565315231888848484), -INT64_C( 8016080185148496634), -INT64_C( 3365171251436437734),
         INT64_C( 5232753838442094123), -INT64_C( 1806946338783921745),  INT64_C( 6678716485601335700),  INT64_C( 2537267084449117649) },
      {  INT64_C( 7876626396527707865),  INT64_C( 6327703798935457910), -INT64_C( 8444156093278868254),  INT64_C(  792525990600389412),
        -INT64_C( 4282096153772423033),  INT64_C( 8176689517910824933),  INT64_C( 3495113324412254258), -INT64_C( 8894133035806391978) } },
    { { -INT64_C( 4217327386109371060),  INT64_C( 1462146507223994500),  INT64_C( 9029403535350110895), -INT64_C( 6164557771088777128),
         INT64_C( 7967243682726010805), -INT64_C( 9152970505335981211),  INT64_C( 7521223655988276535),  INT64_C( 1078941248321503985) },
      UINT8_C( 10),
      { -INT64_C( 6444823229810484523), -INT64_C( 7166643799492954826),  INT64_C( 1160825679683284586),  INT64_C( 4107978185158323148),
        -INT64_C( 8042316938503522478),  INT64_C( 4355947116441623144),  INT64_C(  124837676903243996), -INT64_C( 1113239454258551314) },
      {  INT64_C( 5394206117329760241),  INT64_C(  790827237554372843), -INT64_C( 3320718750563147595), -INT64_C( 3521057494574767212),
        -INT64_C( 3689301451095683169),  INT64_C( 4102642388072787639), -INT64_C( 6298270799792855837),  INT64_C(  908597294068841711) },
      { -INT64_C( 4217327386109371060), -INT64_C( 6375816561938581983),  INT64_C( 9029403535350110895),  INT64_C(  586920690583555936),
         INT64_C( 7967243682726010805), -INT64_C( 9152970505335981211),  INT64_C( 7521223655988276535),  INT64_C( 1078941248321503985) } },
    { {  INT64_C( 7311693701301843659),  INT64_C( 7494898546895421768),  INT64_C( 2349409172957636062),  INT64_C( 4322479761028576388),
         INT64_C( 3265778120923777598), -INT64_C( 5310310381393437343), -INT64_C( 4003064257566966751),  INT64_C( 2693634056535957430) },
      UINT8_C( 63),
      {  INT64_C(   74681461099467337), -INT64_C( 5086377914583683253), -INT64_C( 8273458662043960522), -INT64_C(   39800438883330947),
         INT64_C( 3679636505814865579), -INT64_C( 5866531736128853600), -INT64_C( 3073049960134569313),  INT64_C( 1395686423709339305) },
      {  INT64_C( 6281452445510075920),  INT64_C( 3045217899379926812),  INT64_C( 7460303757460924507),  INT64_C( 1845390670211485473),
         INT64_C( 1096976101920587563), -INT64_C( 7954793774127551260),  INT64_C(  392601397348307534), -INT64_C( 8539621634010629797) },
      {  INT64_C( 6356133906609543257), -INT64_C( 2041160015203756441), -INT64_C(  813154904583036015),  INT64_C( 1805590231328154526),
         INT64_C( 4776612607735453142),  INT64_C( 4625418563453146756), -INT64_C( 4003064257566966751),  INT64_C( 2693634056535957430) } },
    { { -INT64_C( 5625659159720783894), -INT64_C( 4262733505137438704),  INT64_C( 4771074415986154316), -INT64_C( 2710563408861215365),
        -INT64_C( 9137340262048543309),  INT64_C( 6372485775011303733), -INT64_C(  224123893461729351),  INT64_C( 7083941961317845637) },
      UINT8_C(  4),
      {  INT64_C( 7269643312887620103),  INT64_C( 4329870181778099646),  INT64_C( 2564722579906344530),  INT64_C( 7190335853134220430),
        -INT64_C(  968852038973637098),  INT64_C( 1853343154121473663), -INT64_C( 4838903194234096357), -INT64_C(  824357888695620912) },
      {  INT64_C( 1274425862000582536),  INT64_C( 9189953907530268329),  INT64_C( 5306942928662607291), -INT64_C(  321439533223302985),
        -INT64_C(  340471119033620572), -INT64_C( 3077940849910492058),  INT64_C( 4642198055108443306), -INT64_C( 8432040435859988082) },
      { -INT64_C( 5625659159720783894), -INT64_C( 4262733505137438704),  INT64_C( 7871665508568951821), -INT64_C( 2710563408861215365),
        -INT64_C( 9137340262048543309),  INT64_C( 6372485775011303733), -INT64_C(  224123893461729351),  INT64_C( 7083941961317845637) } },
    { { -INT64_C( 9127382355256823033),  INT64_C( 6974267907656827098),  INT64_C( 9068262761557100815),  INT64_C( 1459580854064754385),
        -INT64_C( 2177275983803055828), -INT64_C( 5361079444635839613), -INT64_C( 2408539542357402585), -INT64_C( 5262782123028966956) },
      UINT8_C(216),
      { -INT64_C( 2761901989156618652), -INT64_C( 7396259151174703979),  INT64_C( 1620878075755917699), -INT64_C( 4915584061870677991),
        -INT64_C(  219395007845324972),  INT64_C( 3208968296463365233), -INT64_C( 3812486535545803012),  INT64_C( 7117239981973485491) },
      { -INT64_C( 6444525492333861076),  INT64_C( 5168757207706484966), -INT64_C( 7509645842022035381),  INT64_C( 3857445270331687960),
         INT64_C( 6839094782695310862), -INT64_C( 1825179838618698216),  INT64_C( 7833075129166066744), -INT64_C( 3860117335376243408) },
      { -INT64_C( 9127382355256823033),  INT64_C( 6974267907656827098),  INT64_C( 9068262761557100815), -INT64_C( 1058138791538990031),
         INT64_C( 6619699774849985890), -INT64_C( 5361079444635839613),  INT64_C( 4020588593620263732),  INT64_C( 3257122646597242083) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i src = simde_mm512_loadu_epi64(test_vec[i].src);
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_mask_add_epi64(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_maskz_add_epi64 (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const int64_t a[16];
    const int64_t b[16];
    const int64_t r[16];
  } test_vec[] = {
    { UINT8_C(165),
      {  INT64_C( 6299320458837796671), -INT64_C( 3196421240547742572), -INT64_C( 9151855083952004989), -INT64_C( 2652966953870515301),
        -INT64_C( 3361856595458879637), -INT64_C( 8765515588673012554),  INT64_C( 4218943347121949634),  INT64_C( 8056360307695763285) },
      { -INT64_C( 6084423613766652800), -INT64_C(  986696027690857020),  INT64_C(  716507424025936408),  INT64_C( 2755580261000000714),
         INT64_C( 4185659851829194101), -INT64_C( 1359153785955268607),  INT64_C( 8651579458846990930),  INT64_C( 5266260289850313545) },
      {  INT64_C(  214896845071143871),  INT64_C(                   0), -INT64_C( 8435347659926068581),  INT64_C(                   0),
         INT64_C(                   0),  INT64_C( 8322074699081270455),  INT64_C(                   0), -INT64_C( 5124123476163474786) } },
    { UINT8_C( 48),
      {  INT64_C( 1327964625155044601), -INT64_C( 1979941431104987422), -INT64_C( 6388061835839239302),  INT64_C( 6198577468949612625),
         INT64_C( 7878256497849969529), -INT64_C( 8253649976125538866),  INT64_C( 7274427282076993456), -INT64_C( 5985215513423679939) },
      { -INT64_C( 5628578266044451862),  INT64_C( 6350840359232373634), -INT64_C(  993721339898183746), -INT64_C( 7573227544723558906),
        -INT64_C( 7078269819051780816),  INT64_C( 8967324078724744818), -INT64_C( 5423879114017925356), -INT64_C(  850101963731351568) },
      {  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),  INT64_C(                   0),
         INT64_C(  799986678798188713),  INT64_C(  713674102599205952),  INT64_C(                   0),  INT64_C(                   0) } },
    { UINT8_C(169),
      { -INT64_C( 3269505804180125889),  INT64_C( 4898802782180457107), -INT64_C( 4122299440839867048), -INT64_C( 2805040416254433388),
        -INT64_C( 3881074597838727547),  INT64_C( 1644288571922952801), -INT64_C(   91554778652228748),  INT64_C( 5302276918373401890) },
      { -INT64_C( 6606572555650556850),  INT64_C( 2829390529692828527),  INT64_C( 1381995888231790022),  INT64_C( 6568329687495316506),
        -INT64_C( 1303457298250678015), -INT64_C( 2013848872050549965),  INT64_C( 1771957535492024468), -INT64_C( 3657734556536641579) },
      {  INT64_C( 8570665713878868877),  INT64_C(                   0),  INT64_C(                   0),  INT64_C( 3763289271240883118),
         INT64_C(                   0), -INT64_C(  369560300127597164),  INT64_C(                   0),  INT64_C( 1644542361836760311) } },
    { UINT8_C(222),
      {  INT64_C( 1693716282863260189), -INT64_C( 7181241992025315484), -INT64_C( 2718570591168046034), -INT64_C( 9033248451413530712),
        -INT64_C( 5937343786860347514),  INT64_C( 1858518704354021561), -INT64_C( 4687457667859782492), -INT64_C( 7792311420757763850) },
      { -INT64_C( 5252692508087571419), -INT64_C(  611453451093374081), -INT64_C( 3394024332202210286), -INT64_C( 5460606234653922919),
         INT64_C( 9036821187608596148),  INT64_C( 1013709022150741447), -INT64_C( 5106768477839482762),  INT64_C(  561708961651182727) },
      {  INT64_C(                   0), -INT64_C( 7792695443118689565), -INT64_C( 6112594923370256320),  INT64_C( 3952889387642097985),
         INT64_C( 3099477400748248634),  INT64_C(                   0),  INT64_C( 8652517928010286362), -INT64_C( 7230602459106581123) } },
    { UINT8_C(229),
      {  INT64_C( 8138391701483141613),  INT64_C( 4406625028354607943), -INT64_C( 1993379839983388751),  INT64_C( 2662541310383647862),
         INT64_C( 9046393778122708729),  INT64_C( 2568271637353789258), -INT64_C( 8121881179064237364), -INT64_C( 5039088444989734475) },
      { -INT64_C( 1722519523622035611),  INT64_C( 7561249774353008216), -INT64_C( 2405460785354645258),  INT64_C( 3464354200514345880),
        -INT64_C( 6718838163239081926), -INT64_C(  275183546372714198), -INT64_C( 6250246341167154373),  INT64_C( 2512751206208769253) },
      {  INT64_C( 6415872177861106002),  INT64_C(                   0), -INT64_C( 4398840625338034009),  INT64_C(                   0),
         INT64_C(                   0),  INT64_C( 2293088090981075060),  INT64_C( 4074616553478159879), -INT64_C( 2526337238780965222) } },
    { UINT8_C(254),
      { -INT64_C( 3622607429175870549), -INT64_C( 6274694410419404970), -INT64_C( 7031979866514108454), -INT64_C( 6667111988167567258),
         INT64_C(   12644015949398435), -INT64_C( 6358628794173882517),  INT64_C( 4907679902253939692), -INT64_C( 1348765095626235475) },
      { -INT64_C( 3412187665191306502), -INT64_C( 2311003071927724424),  INT64_C( 6505582346217724197),  INT64_C( 4346333461565343769),
        -INT64_C( 8000778778988929343), -INT64_C( 5787191995171151651), -INT64_C( 7069248972678558756), -INT64_C( 1332976243435314173) },
      {  INT64_C(                   0), -INT64_C( 8585697482347129394), -INT64_C(  526397520296384257), -INT64_C( 2320778526602223489),
        -INT64_C( 7988134763039530908),  INT64_C( 6300923284364517448), -INT64_C( 2161569070424619064), -INT64_C( 2681741339061549648) } },
    { UINT8_C(239),
      { -INT64_C( 1385535232953346975),  INT64_C( 5696251178006254957),  INT64_C( 6906112230749870041), -INT64_C(  166219096561869968),
        -INT64_C( 4862855913802450804), -INT64_C(  444736920620238273), -INT64_C( 8760446760531417455), -INT64_C(  334961341082568769) },
      {  INT64_C( 2935809197118471858),  INT64_C( 3902790899556199184),  INT64_C( 6467643616834876965), -INT64_C( 5327742948472452442),
        -INT64_C( 4008634985254182324), -INT64_C( 5837191191359649246), -INT64_C( 7634820792522817257),  INT64_C( 1904947663936929972) },
      {  INT64_C( 1550273964165124883), -INT64_C( 8847701996147097475), -INT64_C( 5072988226124804610), -INT64_C( 5493962045034322410),
         INT64_C(                   0), -INT64_C( 6281928111979887519),  INT64_C( 2051476520655316904),  INT64_C( 1569986322854361203) } },
    { UINT8_C( 94),
      { -INT64_C( 5133576159156088793), -INT64_C( 3958400705177220649), -INT64_C( 8271053347050896680), -INT64_C( 8784986448452653061),
        -INT64_C( 2149372564095095867),  INT64_C( 3728957796702186606), -INT64_C( 4321223872130680659), -INT64_C( 7079217880864431396) },
      {  INT64_C( 4154637502148371899),  INT64_C( 2033637388041814953), -INT64_C( 5191631281194602905), -INT64_C( 5010619628260266496),
         INT64_C( 7826456547109668761),  INT64_C( 2465062992106081707),  INT64_C( 7649721765552376983),  INT64_C( 7524593379129367732) },
      {  INT64_C(                   0), -INT64_C( 1924763317135405696),  INT64_C( 4984059445464052031),  INT64_C( 4651137996996632059),
         INT64_C( 5677083983014572894),  INT64_C(                   0),  INT64_C( 3328497893421696324),  INT64_C(                   0) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512i a = simde_mm512_loadu_epi64(test_vec[i].a);
    simde__m512i b = simde_mm512_loadu_epi64(test_vec[i].b);
    simde__m512i r = simde_mm512_maskz_add_epi64(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_i64x8(r, simde_mm512_loadu_epi64(test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_add_ps (SIMDE_MUNIT_TEST_ARGS) {
  struct {
    simde_float32 a[16];
    simde_float32 b[16];
    simde_float32 r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   667.91), SIMDE_FLOAT32_C(   656.58), SIMDE_FLOAT32_C(  -596.78), SIMDE_FLOAT32_C(  -487.31),
        SIMDE_FLOAT32_C(  -822.62), SIMDE_FLOAT32_C(   812.31), SIMDE_FLOAT32_C(    89.92), SIMDE_FLOAT32_C(  -844.98),
        SIMDE_FLOAT32_C(  -326.84), SIMDE_FLOAT32_C(  -869.81), SIMDE_FLOAT32_C(  -327.10), SIMDE_FLOAT32_C(  -520.41),
        SIMDE_FLOAT32_C(   590.39), SIMDE_FLOAT32_C(   190.09), SIMDE_FLOAT32_C(  -999.49), SIMDE_FLOAT32_C(   279.05) },
      { SIMDE_FLOAT32_C(   510.55), SIMDE_FLOAT32_C(   821.50), SIMDE_FLOAT32_C(  -282.52), SIMDE_FLOAT32_C(   624.18),
        SIMDE_FLOAT32_C(   410.62), SIMDE_FLOAT32_C(  -938.89), SIMDE_FLOAT32_C(    71.79), SIMDE_FLOAT32_C(   376.91),
        SIMDE_FLOAT32_C(   674.13), SIMDE_FLOAT32_C(    85.78), SIMDE_FLOAT32_C(   -18.27), SIMDE_FLOAT32_C(   115.82),
        SIMDE_FLOAT32_C(  -281.68), SIMDE_FLOAT32_C(  -193.16), SIMDE_FLOAT32_C(  -673.77), SIMDE_FLOAT32_C(  -613.77) },
      { SIMDE_FLOAT32_C(  1178.46), SIMDE_FLOAT32_C(  1478.08), SIMDE_FLOAT32_C(  -879.31), SIMDE_FLOAT32_C(   136.87),
        SIMDE_FLOAT32_C(  -412.00), SIMDE_FLOAT32_C(  -126.58), SIMDE_FLOAT32_C(   161.71), SIMDE_FLOAT32_C(  -468.07),
        SIMDE_FLOAT32_C(   347.28), SIMDE_FLOAT32_C(  -784.02), SIMDE_FLOAT32_C(  -345.37), SIMDE_FLOAT32_C(  -404.59),
        SIMDE_FLOAT32_C(   308.71), SIMDE_FLOAT32_C(    -3.07), SIMDE_FLOAT32_C( -1673.26), SIMDE_FLOAT32_C(  -334.72) } },
    { { SIMDE_FLOAT32_C(  -536.58), SIMDE_FLOAT32_C(  -270.56), SIMDE_FLOAT32_C(  -101.08), SIMDE_FLOAT32_C(  -359.21),
        SIMDE_FLOAT32_C(  -458.24), SIMDE_FLOAT32_C(   988.84), SIMDE_FLOAT32_C(  -204.19), SIMDE_FLOAT32_C(   214.91),
        SIMDE_FLOAT32_C(  -880.97), SIMDE_FLOAT32_C(   468.71), SIMDE_FLOAT32_C(   694.51), SIMDE_FLOAT32_C(   709.42),
        SIMDE_FLOAT32_C(  -341.20), SIMDE_FLOAT32_C(   695.02), SIMDE_FLOAT32_C(   -11.52), SIMDE_FLOAT32_C(  -830.65) },
      { SIMDE_FLOAT32_C(   516.52), SIMDE_FLOAT32_C(   705.95), SIMDE_FLOAT32_C(   793.53), SIMDE_FLOAT32_C(   -72.87),
        SIMDE_FLOAT32_C(   767.06), SIMDE_FLOAT32_C(  -134.68), SIMDE_FLOAT32_C(  -695.95), SIMDE_FLOAT32_C(   441.19),
        SIMDE_FLOAT32_C(   951.11), SIMDE_FLOAT32_C(   285.78), SIMDE_FLOAT32_C(  -442.99), SIMDE_FLOAT32_C(  -330.57),
        SIMDE_FLOAT32_C(  -907.38), SIMDE_FLOAT32_C(  -116.76), SIMDE_FLOAT32_C(    55.65), SIMDE_FLOAT32_C(  -443.96) },
      { SIMDE_FLOAT32_C(   -20.07), SIMDE_FLOAT32_C(   435.40), SIMDE_FLOAT32_C(   692.45), SIMDE_FLOAT32_C(  -432.07),
        SIMDE_FLOAT32_C(   308.82), SIMDE_FLOAT32_C(   854.16), SIMDE_FLOAT32_C(  -900.14), SIMDE_FLOAT32_C(   656.10),
        SIMDE_FLOAT32_C(    70.14), SIMDE_FLOAT32_C(   754.49), SIMDE_FLOAT32_C(   251.51), SIMDE_FLOAT32_C(   378.85),
        SIMDE_FLOAT32_C( -1248.58), SIMDE_FLOAT32_C(   578.25), SIMDE_FLOAT32_C(    44.13), SIMDE_FLOAT32_C( -1274.61) } },
    { { SIMDE_FLOAT32_C(   612.68), SIMDE_FLOAT32_C(   954.57), SIMDE_FLOAT32_C(   196.83), SIMDE_FLOAT32_C(  -845.56),
        SIMDE_FLOAT32_C(   943.41), SIMDE_FLOAT32_C(   992.64), SIMDE_FLOAT32_C(   369.35), SIMDE_FLOAT32_C(  -937.56),
        SIMDE_FLOAT32_C(   461.35), SIMDE_FLOAT32_C(    63.86), SIMDE_FLOAT32_C(   771.86), SIMDE_FLOAT32_C(  -879.85),
        SIMDE_FLOAT32_C(  -241.12), SIMDE_FLOAT32_C(  -239.67), SIMDE_FLOAT32_C(  -710.49), SIMDE_FLOAT32_C(  -724.61) },
      { SIMDE_FLOAT32_C(  -533.71), SIMDE_FLOAT32_C(  -916.96), SIMDE_FLOAT32_C(   202.53), SIMDE_FLOAT32_C(  -766.65),
        SIMDE_FLOAT32_C(   -51.64), SIMDE_FLOAT32_C(   506.57), SIMDE_FLOAT32_C(   674.54), SIMDE_FLOAT32_C(  -100.53),
        SIMDE_FLOAT32_C(  -207.65), SIMDE_FLOAT32_C(  -768.46), SIMDE_FLOAT32_C(   568.90), SIMDE_FLOAT32_C(  -115.03),
        SIMDE_FLOAT32_C(   114.78), SIMDE_FLOAT32_C(  -375.45), SIMDE_FLOAT32_C(   441.01), SIMDE_FLOAT32_C(  -272.54) },
      { SIMDE_FLOAT32_C(    78.97), SIMDE_FLOAT32_C(    37.61), SIMDE_FLOAT32_C(   399.35), SIMDE_FLOAT32_C( -1612.21),
        SIMDE_FLOAT32_C(   891.77), SIMDE_FLOAT32_C(  1499.21), SIMDE_FLOAT32_C(  1043.89), SIMDE_FLOAT32_C( -1038.09),
        SIMDE_FLOAT32_C(   253.70), SIMDE_FLOAT32_C(  -704.60), SIMDE_FLOAT32_C(  1340.75), SIMDE_FLOAT32_C(  -994.87),
        SIMDE_FLOAT32_C(  -126.35), SIMDE_FLOAT32_C(  -615.12), SIMDE_FLOAT32_C(  -269.48), SIMDE_FLOAT32_C(  -997.15) } },
    { { SIMDE_FLOAT32_C(  -420.88), SIMDE_FLOAT32_C(  -362.16), SIMDE_FLOAT32_C(  -118.10), SIMDE_FLOAT32_C(  -477.47),
        SIMDE_FLOAT32_C(  -369.52), SIMDE_FLOAT32_C(  -748.75), SIMDE_FLOAT32_C(  -415.03), SIMDE_FLOAT32_C(  -908.17),
        SIMDE_FLOAT32_C(   315.11), SIMDE_FLOAT32_C(  -643.17), SIMDE_FLOAT32_C(  -788.02), SIMDE_FLOAT32_C(  -926.02),
        SIMDE_FLOAT32_C(   117.16), SIMDE_FLOAT32_C(  -498.52), SIMDE_FLOAT32_C(  -650.63), SIMDE_FLOAT32_C(   583.45) },
      { SIMDE_FLOAT32_C(  -415.48), SIMDE_FLOAT32_C(   551.90), SIMDE_FLOAT32_C(   816.80), SIMDE_FLOAT32_C(   532.88),
        SIMDE_FLOAT32_C(    58.47), SIMDE_FLOAT32_C(   491.34), SIMDE_FLOAT32_C(  -567.65), SIMDE_FLOAT32_C(   850.83),
        SIMDE_FLOAT32_C(   722.88), SIMDE_FLOAT32_C(  -998.75), SIMDE_FLOAT32_C(  -264.20), SIMDE_FLOAT32_C(  -162.34),
        SIMDE_FLOAT32_C(  -374.20), SIMDE_FLOAT32_C(  -823.19), SIMDE_FLOAT32_C(   565.12), SIMDE_FLOAT32_C(   204.92) },
      { SIMDE_FLOAT32_C(  -836.35), SIMDE_FLOAT32_C(   189.74), SIMDE_FLOAT32_C(   698.70), SIMDE_FLOAT32_C(    55.42),
        SIMDE_FLOAT32_C(  -311.05), SIMDE_FLOAT32_C(  -257.41), SIMDE_FLOAT32_C(  -982.68), SIMDE_FLOAT32_C(   -57.35),
        SIMDE_FLOAT32_C(  1037.99), SIMDE_FLOAT32_C( -1641.92), SIMDE_FLOAT32_C( -1052.22), SIMDE_FLOAT32_C( -1088.36),
        SIMDE_FLOAT32_C(  -257.04), SIMDE_FLOAT32_C( -1321.70), SIMDE_FLOAT32_C(   -85.51), SIMDE_FLOAT32_C(   788.38) } },
    { { SIMDE_FLOAT32_C(  -185.35), SIMDE_FLOAT32_C(  -552.99), SIMDE_FLOAT32_C(   727.46), SIMDE_FLOAT32_C(   445.13),
        SIMDE_FLOAT32_C(  -301.74), SIMDE_FLOAT32_C(  -687.57), SIMDE_FLOAT32_C(   536.96), SIMDE_FLOAT32_C(  -986.63),
        SIMDE_FLOAT32_C(  -330.75), SIMDE_FLOAT32_C(   748.93), SIMDE_FLOAT32_C(  -912.65), SIMDE_FLOAT32_C(   786.42),
        SIMDE_FLOAT32_C(  -749.58), SIMDE_FLOAT32_C(  -563.28), SIMDE_FLOAT32_C(   369.87), SIMDE_FLOAT32_C(  -165.06) },
      { SIMDE_FLOAT32_C(   988.62), SIMDE_FLOAT32_C(   186.67), SIMDE_FLOAT32_C(  -632.17), SIMDE_FLOAT32_C(    47.10),
        SIMDE_FLOAT32_C(  -321.99), SIMDE_FLOAT32_C(  -199.82), SIMDE_FLOAT32_C(  -102.08), SIMDE_FLOAT32_C(  -599.11),
        SIMDE_FLOAT32_C(  -198.57), SIMDE_FLOAT32_C(   633.73), SIMDE_FLOAT32_C(   238.55), SIMDE_FLOAT32_C(   427.23),
        SIMDE_FLOAT32_C(   810.54), SIMDE_FLOAT32_C(  -196.33), SIMDE_FLOAT32_C(  -367.85), SIMDE_FLOAT32_C(  -374.81) },
      { SIMDE_FLOAT32_C(   803.28), SIMDE_FLOAT32_C(  -366.32), SIMDE_FLOAT32_C(    95.28), SIMDE_FLOAT32_C(   492.23),
        SIMDE_FLOAT32_C(  -623.73), SIMDE_FLOAT32_C(  -887.39), SIMDE_FLOAT32_C(   434.88), SIMDE_FLOAT32_C( -1585.74),
        SIMDE_FLOAT32_C(  -529.32), SIMDE_FLOAT32_C(  1382.66), SIMDE_FLOAT32_C(  -674.10), SIMDE_FLOAT32_C(  1213.65),
        SIMDE_FLOAT32_C(    60.96), SIMDE_FLOAT32_C(  -759.61), SIMDE_FLOAT32_C(     2.02), SIMDE_FLOAT32_C(  -539.87) } },
    { { SIMDE_FLOAT32_C(   250.68), SIMDE_FLOAT32_C(  -640.39), SIMDE_FLOAT32_C(  -929.68), SIMDE_FLOAT32_C(   948.94),
        SIMDE_FLOAT32_C(  -327.96), SIMDE_FLOAT32_C(   607.27), SIMDE_FLOAT32_C(   962.31), SIMDE_FLOAT32_C(   341.29),
        SIMDE_FLOAT32_C(   356.21), SIMDE_FLOAT32_C(  -950.34), SIMDE_FLOAT32_C(   127.71), SIMDE_FLOAT32_C(   606.63),
        SIMDE_FLOAT32_C(  -513.62), SIMDE_FLOAT32_C(  -502.42), SIMDE_FLOAT32_C(  -558.43), SIMDE_FLOAT32_C(  -524.99) },
      { SIMDE_FLOAT32_C(   684.24), SIMDE_FLOAT32_C(  -190.61), SIMDE_FLOAT32_C(   522.10), SIMDE_FLOAT32_C(  -637.75),
        SIMDE_FLOAT32_C(   609.57), SIMDE_FLOAT32_C(  -579.97), SIMDE_FLOAT32_C(  -236.86), SIMDE_FLOAT32_C(  -589.00),
        SIMDE_FLOAT32_C(  -946.25), SIMDE_FLOAT32_C(  -998.31), SIMDE_FLOAT32_C(   838.23), SIMDE_FLOAT32_C(   864.29),
        SIMDE_FLOAT32_C(  -194.64), SIMDE_FLOAT32_C(  -529.61), SIMDE_FLOAT32_C(  -510.52), SIMDE_FLOAT32_C(  -943.96) },
      { SIMDE_FLOAT32_C(   934.92), SIMDE_FLOAT32_C(  -831.00), SIMDE_FLOAT32_C(  -407.58), SIMDE_FLOAT32_C(   311.19),
        SIMDE_FLOAT32_C(   281.61), SIMDE_FLOAT32_C(    27.30), SIMDE_FLOAT32_C(   725.45), SIMDE_FLOAT32_C(  -247.71),
        SIMDE_FLOAT32_C(  -590.04), SIMDE_FLOAT32_C( -1948.65), SIMDE_FLOAT32_C(   965.94), SIMDE_FLOAT32_C(  1470.92),
        SIMDE_FLOAT32_C(  -708.26), SIMDE_FLOAT32_C( -1032.04), SIMDE_FLOAT32_C( -1068.95), SIMDE_FLOAT32_C( -1468.95) } },
    { { SIMDE_FLOAT32_C(  -170.00), SIMDE_FLOAT32_C(  -440.20), SIMDE_FLOAT32_C(  -995.02), SIMDE_FLOAT32_C(   502.03),
        SIMDE_FLOAT32_C(  -832.92), SIMDE_FLOAT32_C(   967.29), SIMDE_FLOAT32_C(  -156.68), SIMDE_FLOAT32_C(   523.28),
        SIMDE_FLOAT32_C(  -983.05), SIMDE_FLOAT32_C(   971.03), SIMDE_FLOAT32_C(   129.91), SIMDE_FLOAT32_C(  -496.67),
        SIMDE_FLOAT32_C(  -531.39), SIMDE_FLOAT32_C(   571.48), SIMDE_FLOAT32_C(   -21.66), SIMDE_FLOAT32_C(  -847.15) },
      { SIMDE_FLOAT32_C(  -619.13), SIMDE_FLOAT32_C(  -499.55), SIMDE_FLOAT32_C(  -484.90), SIMDE_FLOAT32_C(   990.45),
        SIMDE_FLOAT32_C(   -79.53), SIMDE_FLOAT32_C(   278.24), SIMDE_FLOAT32_C(  -598.55), SIMDE_FLOAT32_C(   -25.77),
        SIMDE_FLOAT32_C(   279.93), SIMDE_FLOAT32_C(  -760.32), SIMDE_FLOAT32_C(  -161.48), SIMDE_FLOAT32_C(  -914.71),
        SIMDE_FLOAT32_C(  -289.93), SIMDE_FLOAT32_C(   328.00), SIMDE_FLOAT32_C(  -858.67), SIMDE_FLOAT32_C(   540.06) },
      { SIMDE_FLOAT32_C(  -789.13), SIMDE_FLOAT32_C(  -939.75), SIMDE_FLOAT32_C( -1479.92), SIMDE_FLOAT32_C(  1492.48),
        SIMDE_FLOAT32_C(  -912.45), SIMDE_FLOAT32_C(  1245.53), SIMDE_FLOAT32_C(  -755.23), SIMDE_FLOAT32_C(   497.51),
        SIMDE_FLOAT32_C(  -703.12), SIMDE_FLOAT32_C(   210.71), SIMDE_FLOAT32_C(   -31.57), SIMDE_FLOAT32_C( -1411.38),
        SIMDE_FLOAT32_C(  -821.33), SIMDE_FLOAT32_C(   899.48), SIMDE_FLOAT32_C(  -880.33), SIMDE_FLOAT32_C(  -307.09) } },
    { { SIMDE_FLOAT32_C(   887.80), SIMDE_FLOAT32_C(  -853.69), SIMDE_FLOAT32_C(    42.10), SIMDE_FLOAT32_C(  -945.12),
        SIMDE_FLOAT32_C(  -886.40), SIMDE_FLOAT32_C(   885.42), SIMDE_FLOAT32_C(   578.16), SIMDE_FLOAT32_C(  -869.46),
        SIMDE_FLOAT32_C(   856.45), SIMDE_FLOAT32_C(  -291.93), SIMDE_FLOAT32_C(  -366.12), SIMDE_FLOAT32_C(  -674.94),
        SIMDE_FLOAT32_C(  -720.45), SIMDE_FLOAT32_C(   612.22), SIMDE_FLOAT32_C(  -522.09), SIMDE_FLOAT32_C(  -339.57) },
      { SIMDE_FLOAT32_C(  -887.34), SIMDE_FLOAT32_C(    -6.99), SIMDE_FLOAT32_C(  -349.13), SIMDE_FLOAT32_C(    33.14),
        SIMDE_FLOAT32_C(  -728.74), SIMDE_FLOAT32_C(    52.32), SIMDE_FLOAT32_C(  -992.63), SIMDE_FLOAT32_C(   551.19),
        SIMDE_FLOAT32_C(   292.00), SIMDE_FLOAT32_C(  -154.11), SIMDE_FLOAT32_C(   636.48), SIMDE_FLOAT32_C(  -997.93),
        SIMDE_FLOAT32_C(  -826.11), SIMDE_FLOAT32_C(   777.81), SIMDE_FLOAT32_C(   542.14), SIMDE_FLOAT32_C(  -938.31) },
      { SIMDE_FLOAT32_C(     0.46), SIMDE_FLOAT32_C(  -860.68), SIMDE_FLOAT32_C(  -307.03), SIMDE_FLOAT32_C(  -911.99),
        SIMDE_FLOAT32_C( -1615.15), SIMDE_FLOAT32_C(   937.74), SIMDE_FLOAT32_C(  -414.47), SIMDE_FLOAT32_C(  -318.27),
        SIMDE_FLOAT32_C(  1148.46), SIMDE_FLOAT32_C(  -446.04), SIMDE_FLOAT32_C(   270.35), SIMDE_FLOAT32_C( -1672.87),
        SIMDE_FLOAT32_C( -1546.56), SIMDE_FLOAT32_C(  1390.02), SIMDE_FLOAT32_C(    20.05), SIMDE_FLOAT32_C( -1277.88) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m512 b = simde_mm512_loadu_ps(test_vec[i].b);
    simde__m512 r = simde_mm512_add_ps(a, b);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_add_ps (SIMDE_MUNIT_TEST_ARGS) {
  struct {
    simde_float32 src[16];
    simde__mmask16 k;
    simde_float32 a[16];
    simde_float32 b[16];
    simde_float32 r[16];
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   460.46), SIMDE_FLOAT32_C(  -331.63), SIMDE_FLOAT32_C(   681.04), SIMDE_FLOAT32_C(  -156.99),
        SIMDE_FLOAT32_C(  -543.60), SIMDE_FLOAT32_C(    94.06), SIMDE_FLOAT32_C(   923.51), SIMDE_FLOAT32_C(  -881.03),
        SIMDE_FLOAT32_C(  -751.28), SIMDE_FLOAT32_C(  -565.86), SIMDE_FLOAT32_C(  -825.81), SIMDE_FLOAT32_C(   899.74),
        SIMDE_FLOAT32_C(   180.53), SIMDE_FLOAT32_C(  -308.94), SIMDE_FLOAT32_C(   105.08), SIMDE_FLOAT32_C(  -812.17) },
      UINT16_C(46254),
      { SIMDE_FLOAT32_C(   159.98), SIMDE_FLOAT32_C(   930.16), SIMDE_FLOAT32_C(  -647.50), SIMDE_FLOAT32_C(  -273.39),
        SIMDE_FLOAT32_C(  -580.12), SIMDE_FLOAT32_C(  -662.69), SIMDE_FLOAT32_C(  -636.46), SIMDE_FLOAT32_C(   880.33),
        SIMDE_FLOAT32_C(     5.69), SIMDE_FLOAT32_C(  -955.42), SIMDE_FLOAT32_C(  -276.66), SIMDE_FLOAT32_C(   462.09),
        SIMDE_FLOAT32_C(   138.64), SIMDE_FLOAT32_C(  -353.15), SIMDE_FLOAT32_C(   581.06), SIMDE_FLOAT32_C(   387.36) },
      { SIMDE_FLOAT32_C(    80.99), SIMDE_FLOAT32_C(   755.25), SIMDE_FLOAT32_C(   287.10), SIMDE_FLOAT32_C(  -738.48),
        SIMDE_FLOAT32_C(  -553.70), SIMDE_FLOAT32_C(  -607.83), SIMDE_FLOAT32_C(  -550.66), SIMDE_FLOAT32_C(   141.56),
        SIMDE_FLOAT32_C(  -486.72), SIMDE_FLOAT32_C(   158.46), SIMDE_FLOAT32_C(    11.82), SIMDE_FLOAT32_C(  -371.24),
        SIMDE_FLOAT32_C(   774.24), SIMDE_FLOAT32_C(  -643.60), SIMDE_FLOAT32_C(   412.35), SIMDE_FLOAT32_C(   -65.78) },
      { SIMDE_FLOAT32_C(   460.46), SIMDE_FLOAT32_C(  1685.41), SIMDE_FLOAT32_C(  -360.40), SIMDE_FLOAT32_C( -1011.87),
        SIMDE_FLOAT32_C(  -543.60), SIMDE_FLOAT32_C( -1270.51), SIMDE_FLOAT32_C(   923.51), SIMDE_FLOAT32_C(  1021.89),
        SIMDE_FLOAT32_C(  -751.28), SIMDE_FLOAT32_C(  -565.86), SIMDE_FLOAT32_C(  -264.84), SIMDE_FLOAT32_C(   899.74),
        SIMDE_FLOAT32_C(   912.88), SIMDE_FLOAT32_C(  -996.76), SIMDE_FLOAT32_C(   105.08), SIMDE_FLOAT32_C(   321.58) } },
    { { SIMDE_FLOAT32_C(  -713.44), SIMDE_FLOAT32_C(   764.86), SIMDE_FLOAT32_C(   660.83), SIMDE_FLOAT32_C(  -293.56),
        SIMDE_FLOAT32_C(  -897.83), SIMDE_FLOAT32_C(  -975.63), SIMDE_FLOAT32_C(  -413.23), SIMDE_FLOAT32_C(   107.86),
        SIMDE_FLOAT32_C(  -931.06), SIMDE_FLOAT32_C(   310.12), SIMDE_FLOAT32_C(  -430.05), SIMDE_FLOAT32_C(   207.58),
        SIMDE_FLOAT32_C(   956.96), SIMDE_FLOAT32_C(  -848.99), SIMDE_FLOAT32_C(  -405.06), SIMDE_FLOAT32_C(    37.95) },
      UINT16_C(56892),
      { SIMDE_FLOAT32_C(   -92.73), SIMDE_FLOAT32_C(  -494.07), SIMDE_FLOAT32_C(   416.25), SIMDE_FLOAT32_C(  -318.49),
        SIMDE_FLOAT32_C(  -137.67), SIMDE_FLOAT32_C(  -171.40), SIMDE_FLOAT32_C(   615.73), SIMDE_FLOAT32_C(   148.89),
        SIMDE_FLOAT32_C(  -406.54), SIMDE_FLOAT32_C(   276.56), SIMDE_FLOAT32_C(   855.33), SIMDE_FLOAT32_C(  -304.37),
        SIMDE_FLOAT32_C(   300.92), SIMDE_FLOAT32_C(  -557.90), SIMDE_FLOAT32_C(   803.48), SIMDE_FLOAT32_C(   369.86) },
      { SIMDE_FLOAT32_C(   752.21), SIMDE_FLOAT32_C(  -626.57), SIMDE_FLOAT32_C(  -422.55), SIMDE_FLOAT32_C(   709.18),
        SIMDE_FLOAT32_C(  -475.56), SIMDE_FLOAT32_C(   172.39), SIMDE_FLOAT32_C(  -252.87), SIMDE_FLOAT32_C(  -569.31),
        SIMDE_FLOAT32_C(    54.43), SIMDE_FLOAT32_C(  -953.40), SIMDE_FLOAT32_C(  -216.76), SIMDE_FLOAT32_C(   328.64),
        SIMDE_FLOAT32_C(   795.42), SIMDE_FLOAT32_C(  -722.64), SIMDE_FLOAT32_C(  -883.86), SIMDE_FLOAT32_C(  -297.31) },
      { SIMDE_FLOAT32_C(  -713.44), SIMDE_FLOAT32_C(   764.86), SIMDE_FLOAT32_C(    -6.30), SIMDE_FLOAT32_C(   390.69),
        SIMDE_FLOAT32_C(  -613.23), SIMDE_FLOAT32_C(     0.99), SIMDE_FLOAT32_C(  -413.23), SIMDE_FLOAT32_C(   107.86),
        SIMDE_FLOAT32_C(  -931.06), SIMDE_FLOAT32_C(  -676.84), SIMDE_FLOAT32_C(   638.57), SIMDE_FLOAT32_C(    24.27),
        SIMDE_FLOAT32_C(  1096.34), SIMDE_FLOAT32_C(  -848.99), SIMDE_FLOAT32_C(   -80.38), SIMDE_FLOAT32_C(    72.56) } },
    { { SIMDE_FLOAT32_C(  -216.71), SIMDE_FLOAT32_C(   532.39), SIMDE_FLOAT32_C(   384.21), SIMDE_FLOAT32_C(   645.62),
        SIMDE_FLOAT32_C(  -639.01), SIMDE_FLOAT32_C(    -0.07), SIMDE_FLOAT32_C(  -205.49), SIMDE_FLOAT32_C(   -45.55),
        SIMDE_FLOAT32_C(  -723.51), SIMDE_FLOAT32_C(  -350.17), SIMDE_FLOAT32_C(   650.08), SIMDE_FLOAT32_C(   577.41),
        SIMDE_FLOAT32_C(    91.93), SIMDE_FLOAT32_C(   453.57), SIMDE_FLOAT32_C(   -52.73), SIMDE_FLOAT32_C(  -155.86) },
      UINT16_C( 2131),
      { SIMDE_FLOAT32_C(   347.05), SIMDE_FLOAT32_C(   565.37), SIMDE_FLOAT32_C(    80.17), SIMDE_FLOAT32_C(   142.47),
        SIMDE_FLOAT32_C(   842.73), SIMDE_FLOAT32_C(   196.31), SIMDE_FLOAT32_C(   845.17), SIMDE_FLOAT32_C(  -373.98),
        SIMDE_FLOAT32_C(  -271.30), SIMDE_FLOAT32_C(   229.37), SIMDE_FLOAT32_C(  -728.36), SIMDE_FLOAT32_C(    89.69),
        SIMDE_FLOAT32_C(  -770.69), SIMDE_FLOAT32_C(    66.14), SIMDE_FLOAT32_C(  -955.85), SIMDE_FLOAT32_C(  -494.20) },
      { SIMDE_FLOAT32_C(   715.97), SIMDE_FLOAT32_C(   694.23), SIMDE_FLOAT32_C(  -916.79), SIMDE_FLOAT32_C(  -192.10),
        SIMDE_FLOAT32_C(   147.79), SIMDE_FLOAT32_C(    30.48), SIMDE_FLOAT32_C(   652.05), SIMDE_FLOAT32_C(   -25.21),
        SIMDE_FLOAT32_C(  -444.80), SIMDE_FLOAT32_C(  -794.64), SIMDE_FLOAT32_C(   326.23), SIMDE_FLOAT32_C(   252.31),
        SIMDE_FLOAT32_C(   505.81), SIMDE_FLOAT32_C(  -891.64), SIMDE_FLOAT32_C(     3.84), SIMDE_FLOAT32_C(  -147.13) },
      { SIMDE_FLOAT32_C(  1063.03), SIMDE_FLOAT32_C(  1259.60), SIMDE_FLOAT32_C(   384.21), SIMDE_FLOAT32_C(   645.62),
        SIMDE_FLOAT32_C(   990.52), SIMDE_FLOAT32_C(    -0.07), SIMDE_FLOAT32_C(  1497.21), SIMDE_FLOAT32_C(   -45.55),
        SIMDE_FLOAT32_C(  -723.51), SIMDE_FLOAT32_C(  -350.17), SIMDE_FLOAT32_C(   650.08), SIMDE_FLOAT32_C(   342.00),
        SIMDE_FLOAT32_C(    91.93), SIMDE_FLOAT32_C(   453.57), SIMDE_FLOAT32_C(   -52.73), SIMDE_FLOAT32_C(  -155.86) } },
    { { SIMDE_FLOAT32_C(   673.73), SIMDE_FLOAT32_C(  -915.98), SIMDE_FLOAT32_C(   995.34), SIMDE_FLOAT32_C(   516.46),
        SIMDE_FLOAT32_C(   280.33), SIMDE_FLOAT32_C(   840.51), SIMDE_FLOAT32_C(  -857.52), SIMDE_FLOAT32_C(  -990.97),
        SIMDE_FLOAT32_C(    69.88), SIMDE_FLOAT32_C(  -585.88), SIMDE_FLOAT32_C(    98.72), SIMDE_FLOAT32_C(   299.19),
        SIMDE_FLOAT32_C(   480.27), SIMDE_FLOAT32_C(   142.87), SIMDE_FLOAT32_C(   804.99), SIMDE_FLOAT32_C(   196.24) },
      UINT16_C(47953),
      { SIMDE_FLOAT32_C(   861.56), SIMDE_FLOAT32_C(  -714.09), SIMDE_FLOAT32_C(  -273.80), SIMDE_FLOAT32_C(   367.37),
        SIMDE_FLOAT32_C(  -605.73), SIMDE_FLOAT32_C(   730.04), SIMDE_FLOAT32_C(  -779.76), SIMDE_FLOAT32_C(  -932.00),
        SIMDE_FLOAT32_C(   814.06), SIMDE_FLOAT32_C(  -784.42), SIMDE_FLOAT32_C(   584.46), SIMDE_FLOAT32_C(    94.39),
        SIMDE_FLOAT32_C(  -943.91), SIMDE_FLOAT32_C(   726.95), SIMDE_FLOAT32_C(   103.42), SIMDE_FLOAT32_C(   125.97) },
      { SIMDE_FLOAT32_C(  -858.93), SIMDE_FLOAT32_C(  -797.86), SIMDE_FLOAT32_C(  -574.84), SIMDE_FLOAT32_C(   621.33),
        SIMDE_FLOAT32_C(   345.01), SIMDE_FLOAT32_C(  -769.85), SIMDE_FLOAT32_C(  -182.42), SIMDE_FLOAT32_C(  -817.89),
        SIMDE_FLOAT32_C(  -881.65), SIMDE_FLOAT32_C(  -178.28), SIMDE_FLOAT32_C(  -833.00), SIMDE_FLOAT32_C(    37.03),
        SIMDE_FLOAT32_C(  -522.09), SIMDE_FLOAT32_C(   126.68), SIMDE_FLOAT32_C(  -489.08), SIMDE_FLOAT32_C(  -660.53) },
      { SIMDE_FLOAT32_C(     2.62), SIMDE_FLOAT32_C(  -915.98), SIMDE_FLOAT32_C(   995.34), SIMDE_FLOAT32_C(   516.46),
        SIMDE_FLOAT32_C(  -260.72), SIMDE_FLOAT32_C(   840.51), SIMDE_FLOAT32_C(  -962.19), SIMDE_FLOAT32_C(  -990.97),
        SIMDE_FLOAT32_C(   -67.59), SIMDE_FLOAT32_C(  -962.70), SIMDE_FLOAT32_C(    98.72), SIMDE_FLOAT32_C(   131.42),
        SIMDE_FLOAT32_C( -1466.00), SIMDE_FLOAT32_C(   853.63), SIMDE_FLOAT32_C(   804.99), SIMDE_FLOAT32_C(  -534.56) } },
    { { SIMDE_FLOAT32_C(   412.59), SIMDE_FLOAT32_C(   237.12), SIMDE_FLOAT32_C(   706.84), SIMDE_FLOAT32_C(   806.86),
        SIMDE_FLOAT32_C(   -32.84), SIMDE_FLOAT32_C(   927.07), SIMDE_FLOAT32_C(   874.86), SIMDE_FLOAT32_C(  -218.78),
        SIMDE_FLOAT32_C(  -857.35), SIMDE_FLOAT32_C(   459.32), SIMDE_FLOAT32_C(   875.61), SIMDE_FLOAT32_C(  -801.26),
        SIMDE_FLOAT32_C(   186.27), SIMDE_FLOAT32_C(   -20.97), SIMDE_FLOAT32_C(   324.71), SIMDE_FLOAT32_C(   327.34) },
      UINT16_C(16785),
      { SIMDE_FLOAT32_C(  -412.03), SIMDE_FLOAT32_C(  -124.71), SIMDE_FLOAT32_C(   135.41), SIMDE_FLOAT32_C(    65.88),
        SIMDE_FLOAT32_C(  -998.03), SIMDE_FLOAT32_C(   646.33), SIMDE_FLOAT32_C(   405.35), SIMDE_FLOAT32_C(   414.56),
        SIMDE_FLOAT32_C(  -116.55), SIMDE_FLOAT32_C(   112.18), SIMDE_FLOAT32_C(   221.42), SIMDE_FLOAT32_C(   850.61),
        SIMDE_FLOAT32_C(    39.26), SIMDE_FLOAT32_C(    96.28), SIMDE_FLOAT32_C(  -368.17), SIMDE_FLOAT32_C(   181.91) },
      { SIMDE_FLOAT32_C(  -444.40), SIMDE_FLOAT32_C(  -492.56), SIMDE_FLOAT32_C(   380.65), SIMDE_FLOAT32_C(   741.87),
        SIMDE_FLOAT32_C(   486.46), SIMDE_FLOAT32_C(  -294.64), SIMDE_FLOAT32_C(    69.20), SIMDE_FLOAT32_C(  -332.37),
        SIMDE_FLOAT32_C(  -544.77), SIMDE_FLOAT32_C(  -982.12), SIMDE_FLOAT32_C(   193.82), SIMDE_FLOAT32_C(  -564.75),
        SIMDE_FLOAT32_C(   784.12), SIMDE_FLOAT32_C(   902.11), SIMDE_FLOAT32_C(  -466.37), SIMDE_FLOAT32_C(  -627.91) },
      { SIMDE_FLOAT32_C(  -856.43), SIMDE_FLOAT32_C(   237.12), SIMDE_FLOAT32_C(   706.84), SIMDE_FLOAT32_C(   806.86),
        SIMDE_FLOAT32_C(  -511.57), SIMDE_FLOAT32_C(   927.07), SIMDE_FLOAT32_C(   874.86), SIMDE_FLOAT32_C(    82.19),
        SIMDE_FLOAT32_C(  -661.32), SIMDE_FLOAT32_C(   459.32), SIMDE_FLOAT32_C(   875.61), SIMDE_FLOAT32_C(  -801.26),
        SIMDE_FLOAT32_C(   186.27), SIMDE_FLOAT32_C(   -20.97), SIMDE_FLOAT32_C(  -834.55), SIMDE_FLOAT32_C(   327.34) } },
    { { SIMDE_FLOAT32_C(  -222.60), SIMDE_FLOAT32_C(   669.04), SIMDE_FLOAT32_C(   437.97), SIMDE_FLOAT32_C(  -220.63),
        SIMDE_FLOAT32_C(   315.37), SIMDE_FLOAT32_C(  -156.68), SIMDE_FLOAT32_C(  -806.07), SIMDE_FLOAT32_C(  -801.18),
        SIMDE_FLOAT32_C(   955.50), SIMDE_FLOAT32_C(   415.35), SIMDE_FLOAT32_C(  -950.57), SIMDE_FLOAT32_C(    -5.24),
        SIMDE_FLOAT32_C(  -488.38), SIMDE_FLOAT32_C(  -318.75), SIMDE_FLOAT32_C(  -823.33), SIMDE_FLOAT32_C(    67.22) },
      UINT16_C(17154),
      { SIMDE_FLOAT32_C(   896.17), SIMDE_FLOAT32_C(  -463.40), SIMDE_FLOAT32_C(   153.15), SIMDE_FLOAT32_C(   680.29),
        SIMDE_FLOAT32_C(  -561.29), SIMDE_FLOAT32_C(   686.78), SIMDE_FLOAT32_C(  -947.62), SIMDE_FLOAT32_C(   216.11),
        SIMDE_FLOAT32_C(   355.82), SIMDE_FLOAT32_C(   490.35), SIMDE_FLOAT32_C(   995.48), SIMDE_FLOAT32_C(  -328.82),
        SIMDE_FLOAT32_C(  -666.33), SIMDE_FLOAT32_C(  -810.59), SIMDE_FLOAT32_C(  -130.00), SIMDE_FLOAT32_C(  -710.83) },
      { SIMDE_FLOAT32_C(   604.75), SIMDE_FLOAT32_C(   -80.58), SIMDE_FLOAT32_C(   283.92), SIMDE_FLOAT32_C(  -883.63),
        SIMDE_FLOAT32_C(   600.68), SIMDE_FLOAT32_C(   460.59), SIMDE_FLOAT32_C(   183.59), SIMDE_FLOAT32_C(  -210.63),
        SIMDE_FLOAT32_C(    17.91), SIMDE_FLOAT32_C(   992.68), SIMDE_FLOAT32_C(   464.52), SIMDE_FLOAT32_C(   280.58),
        SIMDE_FLOAT32_C(   870.97), SIMDE_FLOAT32_C(  -192.70), SIMDE_FLOAT32_C(   998.48), SIMDE_FLOAT32_C(   767.14) },
      { SIMDE_FLOAT32_C(  -222.60), SIMDE_FLOAT32_C(  -543.97), SIMDE_FLOAT32_C(   437.97), SIMDE_FLOAT32_C(  -220.63),
        SIMDE_FLOAT32_C(   315.37), SIMDE_FLOAT32_C(  -156.68), SIMDE_FLOAT32_C(  -806.07), SIMDE_FLOAT32_C(  -801.18),
        SIMDE_FLOAT32_C(   373.72), SIMDE_FLOAT32_C(  1483.03), SIMDE_FLOAT32_C(  -950.57), SIMDE_FLOAT32_C(    -5.24),
        SIMDE_FLOAT32_C(  -488.38), SIMDE_FLOAT32_C(  -318.75), SIMDE_FLOAT32_C(   868.48), SIMDE_FLOAT32_C(    67.22) } },
    { { SIMDE_FLOAT32_C(   343.91), SIMDE_FLOAT32_C(   151.64), SIMDE_FLOAT32_C(   447.43), SIMDE_FLOAT32_C(   782.62),
        SIMDE_FLOAT32_C(  -161.58), SIMDE_FLOAT32_C(   499.81), SIMDE_FLOAT32_C(    -1.27), SIMDE_FLOAT32_C(  -805.77),
        SIMDE_FLOAT32_C(    -9.84), SIMDE_FLOAT32_C(    -5.79), SIMDE_FLOAT32_C(  -134.58), SIMDE_FLOAT32_C(   323.82),
        SIMDE_FLOAT32_C(   183.61), SIMDE_FLOAT32_C(   735.41), SIMDE_FLOAT32_C(   612.99), SIMDE_FLOAT32_C(  -211.63) },
      UINT16_C(55098),
      { SIMDE_FLOAT32_C(  -918.99), SIMDE_FLOAT32_C(  -490.60), SIMDE_FLOAT32_C(  -344.01), SIMDE_FLOAT32_C(   951.99),
        SIMDE_FLOAT32_C(   316.70), SIMDE_FLOAT32_C(  -345.53), SIMDE_FLOAT32_C(   719.12), SIMDE_FLOAT32_C(  -339.39),
        SIMDE_FLOAT32_C(   806.11), SIMDE_FLOAT32_C(   166.55), SIMDE_FLOAT32_C(  -556.77), SIMDE_FLOAT32_C(  -355.47),
        SIMDE_FLOAT32_C(  -333.64), SIMDE_FLOAT32_C(   441.96), SIMDE_FLOAT32_C(  -161.24), SIMDE_FLOAT32_C(   656.52) },
      { SIMDE_FLOAT32_C(  -563.83), SIMDE_FLOAT32_C(   704.18), SIMDE_FLOAT32_C(   -19.66), SIMDE_FLOAT32_C(   619.78),
        SIMDE_FLOAT32_C(   439.59), SIMDE_FLOAT32_C(  -406.67), SIMDE_FLOAT32_C(  -591.85), SIMDE_FLOAT32_C(  -905.57),
        SIMDE_FLOAT32_C(   490.24), SIMDE_FLOAT32_C(   312.88), SIMDE_FLOAT32_C(  -650.06), SIMDE_FLOAT32_C(   847.74),
        SIMDE_FLOAT32_C(   401.22), SIMDE_FLOAT32_C(   394.82), SIMDE_FLOAT32_C(   223.15), SIMDE_FLOAT32_C(   482.23) },
      { SIMDE_FLOAT32_C(   343.91), SIMDE_FLOAT32_C(   213.57), SIMDE_FLOAT32_C(   447.43), SIMDE_FLOAT32_C(  1571.77),
        SIMDE_FLOAT32_C(   756.29), SIMDE_FLOAT32_C(  -752.20), SIMDE_FLOAT32_C(    -1.27), SIMDE_FLOAT32_C(  -805.77),
        SIMDE_FLOAT32_C(  1296.35), SIMDE_FLOAT32_C(   479.44), SIMDE_FLOAT32_C( -1206.83), SIMDE_FLOAT32_C(   323.82),
        SIMDE_FLOAT32_C(    67.58), SIMDE_FLOAT32_C(   735.41), SIMDE_FLOAT32_C(    61.91), SIMDE_FLOAT32_C(  1138.75) } },
    { { SIMDE_FLOAT32_C(   904.21), SIMDE_FLOAT32_C(   879.14), SIMDE_FLOAT32_C(   434.21), SIMDE_FLOAT32_C(   220.91),
        SIMDE_FLOAT32_C(  -466.39), SIMDE_FLOAT32_C(   153.34), SIMDE_FLOAT32_C(   881.52), SIMDE_FLOAT32_C(  -660.28),
        SIMDE_FLOAT32_C(  -680.11), SIMDE_FLOAT32_C(  -675.25), SIMDE_FLOAT32_C(   -15.75), SIMDE_FLOAT32_C(   -13.75),
        SIMDE_FLOAT32_C(   766.71), SIMDE_FLOAT32_C(   823.02), SIMDE_FLOAT32_C(  -357.23), SIMDE_FLOAT32_C(  -797.13) },
      UINT16_C(62059),
      { SIMDE_FLOAT32_C(   543.68), SIMDE_FLOAT32_C(   411.16), SIMDE_FLOAT32_C(   554.42), SIMDE_FLOAT32_C(   -55.10),
        SIMDE_FLOAT32_C(  -194.03), SIMDE_FLOAT32_C(  -222.43), SIMDE_FLOAT32_C(  -572.87), SIMDE_FLOAT32_C(  -289.81),
        SIMDE_FLOAT32_C(  -343.29), SIMDE_FLOAT32_C(   861.34), SIMDE_FLOAT32_C(   931.10), SIMDE_FLOAT32_C(   190.32),
        SIMDE_FLOAT32_C(    14.68), SIMDE_FLOAT32_C(   812.62), SIMDE_FLOAT32_C(   530.05), SIMDE_FLOAT32_C(   334.57) },
      { SIMDE_FLOAT32_C(  -862.62), SIMDE_FLOAT32_C(  -485.70), SIMDE_FLOAT32_C(  -679.18), SIMDE_FLOAT32_C(   904.08),
        SIMDE_FLOAT32_C(  -662.68), SIMDE_FLOAT32_C(   -36.41), SIMDE_FLOAT32_C(  -893.04), SIMDE_FLOAT32_C(   864.51),
        SIMDE_FLOAT32_C(  -413.30), SIMDE_FLOAT32_C(   929.61), SIMDE_FLOAT32_C(  -168.70), SIMDE_FLOAT32_C(  -196.86),
        SIMDE_FLOAT32_C(  -839.59), SIMDE_FLOAT32_C(   892.52), SIMDE_FLOAT32_C(  -490.18), SIMDE_FLOAT32_C(   704.10) },
      { SIMDE_FLOAT32_C(  -318.94), SIMDE_FLOAT32_C(   -74.54), SIMDE_FLOAT32_C(   434.21), SIMDE_FLOAT32_C(   848.98),
        SIMDE_FLOAT32_C(  -466.39), SIMDE_FLOAT32_C(  -258.84), SIMDE_FLOAT32_C( -1465.91), SIMDE_FLOAT32_C(  -660.28),
        SIMDE_FLOAT32_C(  -680.11), SIMDE_FLOAT32_C(  1790.95), SIMDE_FLOAT32_C(   -15.75), SIMDE_FLOAT32_C(   -13.75),
        SIMDE_FLOAT32_C(  -824.91), SIMDE_FLOAT32_C(  1705.14), SIMDE_FLOAT32_C(    39.87), SIMDE_FLOAT32_C(  1038.67) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 src = simde_mm512_loadu_ps(test_vec[i].src);
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m512 b = simde_mm512_loadu_ps(test_vec[i].b);
    simde__m512 r = simde_mm512_mask_add_ps(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_add_ps (SIMDE_MUNIT_TEST_ARGS) {
  struct {
    simde__mmask16 k;
    simde_float32 a[16];
    simde_float32 b[16];
    simde_float32 r[16];
  } test_vec[] = {
    { UINT16_C( 7629),
      { SIMDE_FLOAT32_C(   914.45), SIMDE_FLOAT32_C(   855.01), SIMDE_FLOAT32_C(    38.80), SIMDE_FLOAT32_C(   492.72),
        SIMDE_FLOAT32_C(   597.40), SIMDE_FLOAT32_C(   253.75), SIMDE_FLOAT32_C(   616.90), SIMDE_FLOAT32_C(   108.68),
        SIMDE_FLOAT32_C(   217.16), SIMDE_FLOAT32_C(   439.38), SIMDE_FLOAT32_C(   724.30), SIMDE_FLOAT32_C(   474.66),
        SIMDE_FLOAT32_C(   870.80), SIMDE_FLOAT32_C(   -46.25), SIMDE_FLOAT32_C(  -743.93), SIMDE_FLOAT32_C(   176.79) },
      { SIMDE_FLOAT32_C(  -872.85), SIMDE_FLOAT32_C(   805.82), SIMDE_FLOAT32_C(   350.81), SIMDE_FLOAT32_C(  -515.94),
        SIMDE_FLOAT32_C(  -720.47), SIMDE_FLOAT32_C(   570.49), SIMDE_FLOAT32_C(   295.95), SIMDE_FLOAT32_C(   265.48),
        SIMDE_FLOAT32_C(   175.46), SIMDE_FLOAT32_C(  -217.20), SIMDE_FLOAT32_C(  -845.54), SIMDE_FLOAT32_C(   857.16),
        SIMDE_FLOAT32_C(   138.12), SIMDE_FLOAT32_C(  -599.93), SIMDE_FLOAT32_C(   503.35), SIMDE_FLOAT32_C(    52.57) },
      { SIMDE_FLOAT32_C(    41.60), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   389.61), SIMDE_FLOAT32_C(   -23.22),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   912.85), SIMDE_FLOAT32_C(   374.16),
        SIMDE_FLOAT32_C(   392.62), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -121.24), SIMDE_FLOAT32_C(  1331.82),
        SIMDE_FLOAT32_C(  1008.92), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(26184),
      { SIMDE_FLOAT32_C(   601.56), SIMDE_FLOAT32_C(  -314.52), SIMDE_FLOAT32_C(  -512.28), SIMDE_FLOAT32_C(   472.36),
        SIMDE_FLOAT32_C(   639.23), SIMDE_FLOAT32_C(  -256.21), SIMDE_FLOAT32_C(  -350.85), SIMDE_FLOAT32_C(   766.38),
        SIMDE_FLOAT32_C(  -450.39), SIMDE_FLOAT32_C(   999.96), SIMDE_FLOAT32_C(  -749.56), SIMDE_FLOAT32_C(  -170.85),
        SIMDE_FLOAT32_C(   570.45), SIMDE_FLOAT32_C(   546.39), SIMDE_FLOAT32_C(  -905.38), SIMDE_FLOAT32_C(  -254.09) },
      { SIMDE_FLOAT32_C(  -670.81), SIMDE_FLOAT32_C(  -750.92), SIMDE_FLOAT32_C(  -396.93), SIMDE_FLOAT32_C(   467.31),
        SIMDE_FLOAT32_C(  -350.85), SIMDE_FLOAT32_C(  -893.58), SIMDE_FLOAT32_C(  -480.12), SIMDE_FLOAT32_C(   -95.76),
        SIMDE_FLOAT32_C(  -351.43), SIMDE_FLOAT32_C(    65.16), SIMDE_FLOAT32_C(  -243.28), SIMDE_FLOAT32_C(  -555.53),
        SIMDE_FLOAT32_C(   227.35), SIMDE_FLOAT32_C(   717.89), SIMDE_FLOAT32_C(   457.53), SIMDE_FLOAT32_C(  -171.09) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   939.67),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -830.97), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  1065.13), SIMDE_FLOAT32_C(  -992.84), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  1264.28), SIMDE_FLOAT32_C(  -447.85), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(38268),
      { SIMDE_FLOAT32_C(   -49.61), SIMDE_FLOAT32_C(  -940.59), SIMDE_FLOAT32_C(  -932.20), SIMDE_FLOAT32_C(  -479.16),
        SIMDE_FLOAT32_C(   605.80), SIMDE_FLOAT32_C(  -837.58), SIMDE_FLOAT32_C(   266.75), SIMDE_FLOAT32_C(   934.99),
        SIMDE_FLOAT32_C(  -588.49), SIMDE_FLOAT32_C(   869.82), SIMDE_FLOAT32_C(   402.30), SIMDE_FLOAT32_C(    60.66),
        SIMDE_FLOAT32_C(   976.24), SIMDE_FLOAT32_C(   922.17), SIMDE_FLOAT32_C(   964.89), SIMDE_FLOAT32_C(  -375.20) },
      { SIMDE_FLOAT32_C(   -12.67), SIMDE_FLOAT32_C(  -278.39), SIMDE_FLOAT32_C(    69.27), SIMDE_FLOAT32_C(  -785.32),
        SIMDE_FLOAT32_C(  -560.49), SIMDE_FLOAT32_C(  -473.20), SIMDE_FLOAT32_C(    43.59), SIMDE_FLOAT32_C(  -157.12),
        SIMDE_FLOAT32_C(  -527.94), SIMDE_FLOAT32_C(   344.87), SIMDE_FLOAT32_C(  -114.53), SIMDE_FLOAT32_C(   161.10),
        SIMDE_FLOAT32_C(  -704.71), SIMDE_FLOAT32_C(  -305.55), SIMDE_FLOAT32_C(  -600.24), SIMDE_FLOAT32_C(   245.68) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -862.92), SIMDE_FLOAT32_C( -1264.48),
        SIMDE_FLOAT32_C(    45.31), SIMDE_FLOAT32_C( -1310.77), SIMDE_FLOAT32_C(   310.34), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C( -1116.44), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   287.77), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   271.52), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -129.52) } },
    { UINT16_C(37369),
      { SIMDE_FLOAT32_C(   -96.91), SIMDE_FLOAT32_C(   696.94), SIMDE_FLOAT32_C(  -897.85), SIMDE_FLOAT32_C(  -120.68),
        SIMDE_FLOAT32_C(   619.12), SIMDE_FLOAT32_C(  -932.96), SIMDE_FLOAT32_C(   504.13), SIMDE_FLOAT32_C(  -393.55),
        SIMDE_FLOAT32_C(  -211.35), SIMDE_FLOAT32_C(  -426.60), SIMDE_FLOAT32_C(  -178.87), SIMDE_FLOAT32_C(   228.16),
        SIMDE_FLOAT32_C(   100.20), SIMDE_FLOAT32_C(   864.72), SIMDE_FLOAT32_C(  -928.97), SIMDE_FLOAT32_C(   572.26) },
      { SIMDE_FLOAT32_C(   209.59), SIMDE_FLOAT32_C(   -43.49), SIMDE_FLOAT32_C(  -266.64), SIMDE_FLOAT32_C(   504.88),
        SIMDE_FLOAT32_C(   650.96), SIMDE_FLOAT32_C(   133.12), SIMDE_FLOAT32_C(  -249.44), SIMDE_FLOAT32_C(  -595.18),
        SIMDE_FLOAT32_C(   600.68), SIMDE_FLOAT32_C(  -482.93), SIMDE_FLOAT32_C(  -235.52), SIMDE_FLOAT32_C(  -769.33),
        SIMDE_FLOAT32_C(   550.34), SIMDE_FLOAT32_C(    59.13), SIMDE_FLOAT32_C(   272.16), SIMDE_FLOAT32_C(  -546.58) },
      { SIMDE_FLOAT32_C(   112.67), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   384.20),
        SIMDE_FLOAT32_C(  1270.07), SIMDE_FLOAT32_C(  -799.84), SIMDE_FLOAT32_C(   254.68), SIMDE_FLOAT32_C(  -988.73),
        SIMDE_FLOAT32_C(   389.33), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   650.54), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    25.68) } },
    { UINT16_C(17887),
      { SIMDE_FLOAT32_C(   410.27), SIMDE_FLOAT32_C(   802.77), SIMDE_FLOAT32_C(   458.15), SIMDE_FLOAT32_C(  -489.53),
        SIMDE_FLOAT32_C(   667.49), SIMDE_FLOAT32_C(   529.19), SIMDE_FLOAT32_C(  -917.27), SIMDE_FLOAT32_C(  -122.92),
        SIMDE_FLOAT32_C(  -514.30), SIMDE_FLOAT32_C(  -183.91), SIMDE_FLOAT32_C(  -618.04), SIMDE_FLOAT32_C(  -863.35),
        SIMDE_FLOAT32_C(   949.21), SIMDE_FLOAT32_C(   132.51), SIMDE_FLOAT32_C(  -458.53), SIMDE_FLOAT32_C(   549.89) },
      { SIMDE_FLOAT32_C(   649.59), SIMDE_FLOAT32_C(   305.95), SIMDE_FLOAT32_C(   780.56), SIMDE_FLOAT32_C(   199.92),
        SIMDE_FLOAT32_C(  -634.93), SIMDE_FLOAT32_C(    52.72), SIMDE_FLOAT32_C(   653.35), SIMDE_FLOAT32_C(   121.14),
        SIMDE_FLOAT32_C(  -572.98), SIMDE_FLOAT32_C(   -13.91), SIMDE_FLOAT32_C(   496.32), SIMDE_FLOAT32_C(   868.36),
        SIMDE_FLOAT32_C(   822.96), SIMDE_FLOAT32_C(  -522.04), SIMDE_FLOAT32_C(  -901.64), SIMDE_FLOAT32_C(   233.23) },
      { SIMDE_FLOAT32_C(  1059.85), SIMDE_FLOAT32_C(  1108.71), SIMDE_FLOAT32_C(  1238.71), SIMDE_FLOAT32_C(  -289.60),
        SIMDE_FLOAT32_C(    32.56), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -263.92), SIMDE_FLOAT32_C(    -1.78),
        SIMDE_FLOAT32_C( -1087.28), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -121.72), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C( -1360.17), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C(52085),
      { SIMDE_FLOAT32_C(  -357.48), SIMDE_FLOAT32_C(   207.25), SIMDE_FLOAT32_C(   708.05), SIMDE_FLOAT32_C(  -408.26),
        SIMDE_FLOAT32_C(  -660.23), SIMDE_FLOAT32_C(  -750.48), SIMDE_FLOAT32_C(  -858.37), SIMDE_FLOAT32_C(   989.35),
        SIMDE_FLOAT32_C(   555.47), SIMDE_FLOAT32_C(   922.19), SIMDE_FLOAT32_C(   189.28), SIMDE_FLOAT32_C(   920.54),
        SIMDE_FLOAT32_C(   -25.09), SIMDE_FLOAT32_C(  -157.38), SIMDE_FLOAT32_C(    41.68), SIMDE_FLOAT32_C(   401.93) },
      { SIMDE_FLOAT32_C(   828.72), SIMDE_FLOAT32_C(  -462.00), SIMDE_FLOAT32_C(   270.29), SIMDE_FLOAT32_C(   651.68),
        SIMDE_FLOAT32_C(    15.96), SIMDE_FLOAT32_C(   368.65), SIMDE_FLOAT32_C(  -115.09), SIMDE_FLOAT32_C(   296.68),
        SIMDE_FLOAT32_C(   -74.83), SIMDE_FLOAT32_C(  -371.39), SIMDE_FLOAT32_C(   244.89), SIMDE_FLOAT32_C(  -989.13),
        SIMDE_FLOAT32_C(  -544.95), SIMDE_FLOAT32_C(  -929.81), SIMDE_FLOAT32_C(   582.27), SIMDE_FLOAT32_C(    97.57) },
      { SIMDE_FLOAT32_C(   471.24), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   978.34), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(  -644.28), SIMDE_FLOAT32_C(  -381.83), SIMDE_FLOAT32_C(  -973.46), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   480.63), SIMDE_FLOAT32_C(   550.80), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -68.59),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   623.95), SIMDE_FLOAT32_C(   499.50) } },
    { UINT16_C(30320),
      { SIMDE_FLOAT32_C(   753.13), SIMDE_FLOAT32_C(  -204.17), SIMDE_FLOAT32_C(    15.84), SIMDE_FLOAT32_C(  -271.97),
        SIMDE_FLOAT32_C(   638.45), SIMDE_FLOAT32_C(  -942.48), SIMDE_FLOAT32_C(  -870.04), SIMDE_FLOAT32_C(   467.17),
        SIMDE_FLOAT32_C(  -404.47), SIMDE_FLOAT32_C(   400.26), SIMDE_FLOAT32_C(   118.85), SIMDE_FLOAT32_C(   611.49),
        SIMDE_FLOAT32_C(  -231.09), SIMDE_FLOAT32_C(  -996.24), SIMDE_FLOAT32_C(   -91.83), SIMDE_FLOAT32_C(   694.08) },
      { SIMDE_FLOAT32_C(  -367.63), SIMDE_FLOAT32_C(  -846.94), SIMDE_FLOAT32_C(   704.95), SIMDE_FLOAT32_C(    87.42),
        SIMDE_FLOAT32_C(  -776.75), SIMDE_FLOAT32_C(   287.22), SIMDE_FLOAT32_C(  -815.01), SIMDE_FLOAT32_C(   500.69),
        SIMDE_FLOAT32_C(  -422.46), SIMDE_FLOAT32_C(   874.30), SIMDE_FLOAT32_C(   117.89), SIMDE_FLOAT32_C(  -882.62),
        SIMDE_FLOAT32_C(   705.23), SIMDE_FLOAT32_C(  -275.56), SIMDE_FLOAT32_C(   212.68), SIMDE_FLOAT32_C(   458.36) },
      { SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(  -138.30), SIMDE_FLOAT32_C(  -655.26), SIMDE_FLOAT32_C( -1685.05), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  1274.55), SIMDE_FLOAT32_C(   236.74), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(   474.14), SIMDE_FLOAT32_C( -1271.80), SIMDE_FLOAT32_C(   120.85), SIMDE_FLOAT32_C(     0.00) } },
    { UINT16_C( 6201),
      { SIMDE_FLOAT32_C(   716.62), SIMDE_FLOAT32_C(   744.75), SIMDE_FLOAT32_C(  -506.94), SIMDE_FLOAT32_C(  -514.47),
        SIMDE_FLOAT32_C(   748.50), SIMDE_FLOAT32_C(   401.23), SIMDE_FLOAT32_C(  -820.39), SIMDE_FLOAT32_C(  -619.12),
        SIMDE_FLOAT32_C(   554.30), SIMDE_FLOAT32_C(   884.56), SIMDE_FLOAT32_C(   468.30), SIMDE_FLOAT32_C(   777.54),
        SIMDE_FLOAT32_C(   171.78), SIMDE_FLOAT32_C(   653.28), SIMDE_FLOAT32_C(   278.23), SIMDE_FLOAT32_C(   749.31) },
      { SIMDE_FLOAT32_C(   527.58), SIMDE_FLOAT32_C(  -603.88), SIMDE_FLOAT32_C(   866.69), SIMDE_FLOAT32_C(   232.81),
        SIMDE_FLOAT32_C(   120.56), SIMDE_FLOAT32_C(    79.37), SIMDE_FLOAT32_C(  -308.83), SIMDE_FLOAT32_C(  -359.16),
        SIMDE_FLOAT32_C(   307.90), SIMDE_FLOAT32_C(  -122.44), SIMDE_FLOAT32_C(   799.56), SIMDE_FLOAT32_C(   593.95),
        SIMDE_FLOAT32_C(   193.92), SIMDE_FLOAT32_C(  -574.54), SIMDE_FLOAT32_C(  -524.47), SIMDE_FLOAT32_C(   -89.46) },
      { SIMDE_FLOAT32_C(  1244.20), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -281.66),
        SIMDE_FLOAT32_C(   869.07), SIMDE_FLOAT32_C(   480.61), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
        SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  1371.50),
        SIMDE_FLOAT32_C(   365.70), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m512 b = simde_mm512_loadu_ps(test_vec[i].b);
    simde__m512 r = simde_mm512_maskz_add_ps(test_vec[i].k, a, b);
    simde_test_x86_assert_equal_f32x16(r, simde_mm512_loadu_ps(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_add_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 a[8];
    const simde_float64 b[8];
    const simde_float64 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(  -775.47), SIMDE_FLOAT64_C(     0.19), SIMDE_FLOAT64_C(  -757.09), SIMDE_FLOAT64_C(  -410.73),
        SIMDE_FLOAT64_C(  -780.15), SIMDE_FLOAT64_C(  -708.07), SIMDE_FLOAT64_C(   367.35), SIMDE_FLOAT64_C(  -968.32) },
      { SIMDE_FLOAT64_C(   820.33), SIMDE_FLOAT64_C(   850.82), SIMDE_FLOAT64_C(   596.13), SIMDE_FLOAT64_C(  -878.99),
        SIMDE_FLOAT64_C(  -603.61), SIMDE_FLOAT64_C(   533.64), SIMDE_FLOAT64_C(  -994.37), SIMDE_FLOAT64_C(   846.35) },
      { SIMDE_FLOAT64_C(    44.87), SIMDE_FLOAT64_C(   851.01), SIMDE_FLOAT64_C(  -160.96), SIMDE_FLOAT64_C( -1289.72),
        SIMDE_FLOAT64_C( -1383.75), SIMDE_FLOAT64_C(  -174.43), SIMDE_FLOAT64_C(  -627.02), SIMDE_FLOAT64_C(  -121.96) } },
    { { SIMDE_FLOAT64_C(  -503.42), SIMDE_FLOAT64_C(  -250.77), SIMDE_FLOAT64_C(  -532.42), SIMDE_FLOAT64_C(   815.06),
        SIMDE_FLOAT64_C(  -419.11), SIMDE_FLOAT64_C(   224.41), SIMDE_FLOAT64_C(   -34.26), SIMDE_FLOAT64_C(  -803.36) },
      { SIMDE_FLOAT64_C(  -331.10), SIMDE_FLOAT64_C(  -474.33), SIMDE_FLOAT64_C(   866.30), SIMDE_FLOAT64_C(   560.33),
        SIMDE_FLOAT64_C(   467.15), SIMDE_FLOAT64_C(   279.38), SIMDE_FLOAT64_C(  -475.96), SIMDE_FLOAT64_C(   691.69) },
      { SIMDE_FLOAT64_C(  -834.52), SIMDE_FLOAT64_C(  -725.11), SIMDE_FLOAT64_C(   333.88), SIMDE_FLOAT64_C(  1375.40),
        SIMDE_FLOAT64_C(    48.04), SIMDE_FLOAT64_C(   503.79), SIMDE_FLOAT64_C(  -510.22), SIMDE_FLOAT64_C(  -111.67) } },
    { { SIMDE_FLOAT64_C(  -720.44), SIMDE_FLOAT64_C(  -233.05), SIMDE_FLOAT64_C(  -719.04), SIMDE_FLOAT64_C(  -500.58),
        SIMDE_FLOAT64_C(    58.88), SIMDE_FLOAT64_C(   648.31), SIMDE_FLOAT64_C(  -468.90), SIMDE_FLOAT64_C(  -120.79) },
      { SIMDE_FLOAT64_C(   499.13), SIMDE_FLOAT64_C(  -872.76), SIMDE_FLOAT64_C(     0.22), SIMDE_FLOAT64_C(   895.52),
        SIMDE_FLOAT64_C(   660.88), SIMDE_FLOAT64_C(     5.85), SIMDE_FLOAT64_C(   741.88), SIMDE_FLOAT64_C(  -842.54) },
      { SIMDE_FLOAT64_C(  -221.31), SIMDE_FLOAT64_C( -1105.81), SIMDE_FLOAT64_C(  -718.83), SIMDE_FLOAT64_C(   394.94),
        SIMDE_FLOAT64_C(   719.76), SIMDE_FLOAT64_C(   654.16), SIMDE_FLOAT64_C(   272.98), SIMDE_FLOAT64_C(  -963.33) } },
    { { SIMDE_FLOAT64_C(   755.08), SIMDE_FLOAT64_C(  -790.54), SIMDE_FLOAT64_C(   972.53), SIMDE_FLOAT64_C(  -664.03),
        SIMDE_FLOAT64_C(   433.87), SIMDE_FLOAT64_C(   -61.74), SIMDE_FLOAT64_C(  -467.39), SIMDE_FLOAT64_C(  -897.23) },
      { SIMDE_FLOAT64_C(   463.93), SIMDE_FLOAT64_C(  -601.09), SIMDE_FLOAT64_C(   663.10), SIMDE_FLOAT64_C(   -68.92),
        SIMDE_FLOAT64_C(   678.29), SIMDE_FLOAT64_C(  -812.86), SIMDE_FLOAT64_C(  -377.23), SIMDE_FLOAT64_C(   957.85) },
      { SIMDE_FLOAT64_C(  1219.01), SIMDE_FLOAT64_C( -1391.63), SIMDE_FLOAT64_C(  1635.63), SIMDE_FLOAT64_C(  -732.95),
        SIMDE_FLOAT64_C(  1112.16), SIMDE_FLOAT64_C(  -874.59), SIMDE_FLOAT64_C(  -844.62), SIMDE_FLOAT64_C(    60.62) } },
    { { SIMDE_FLOAT64_C(   -45.90), SIMDE_FLOAT64_C(   -96.28), SIMDE_FLOAT64_C(  -542.73), SIMDE_FLOAT64_C(  -987.02),
        SIMDE_FLOAT64_C(  -447.97), SIMDE_FLOAT64_C(   -11.63), SIMDE_FLOAT64_C(  -107.82), SIMDE_FLOAT64_C(  -948.84) },
      { SIMDE_FLOAT64_C(   115.60), SIMDE_FLOAT64_C(   892.40), SIMDE_FLOAT64_C(   946.68), SIMDE_FLOAT64_C(  -223.52),
        SIMDE_FLOAT64_C(  -101.75), SIMDE_FLOAT64_C(   688.56), SIMDE_FLOAT64_C(   -66.05), SIMDE_FLOAT64_C(  -346.67) },
      { SIMDE_FLOAT64_C(    69.70), SIMDE_FLOAT64_C(   796.12), SIMDE_FLOAT64_C(   403.95), SIMDE_FLOAT64_C( -1210.54),
        SIMDE_FLOAT64_C(  -549.72), SIMDE_FLOAT64_C(   676.92), SIMDE_FLOAT64_C(  -173.87), SIMDE_FLOAT64_C( -1295.52) } },
    { { SIMDE_FLOAT64_C(   898.01), SIMDE_FLOAT64_C(   -93.53), SIMDE_FLOAT64_C(   -10.70), SIMDE_FLOAT64_C(   331.89),
        SIMDE_FLOAT64_C(   844.74), SIMDE_FLOAT64_C(   521.91), SIMDE_FLOAT64_C(   434.66), SIMDE_FLOAT64_C(   308.66) },
      { SIMDE_FLOAT64_C(   920.82), SIMDE_FLOAT64_C(    97.76), SIMDE_FLOAT64_C(  -760.25), SIMDE_FLOAT64_C(   599.10),
        SIMDE_FLOAT64_C(   284.91), SIMDE_FLOAT64_C(  -137.49), SIMDE_FLOAT64_C(   556.96), SIMDE_FLOAT64_C(  -761.00) },
      { SIMDE_FLOAT64_C(  1818.83), SIMDE_FLOAT64_C(     4.23), SIMDE_FLOAT64_C(  -770.96), SIMDE_FLOAT64_C(   930.99),
        SIMDE_FLOAT64_C(  1129.64), SIMDE_FLOAT64_C(   384.42), SIMDE_FLOAT64_C(   991.61), SIMDE_FLOAT64_C(  -452.33) } },
    { { SIMDE_FLOAT64_C(   766.23), SIMDE_FLOAT64_C(  -985.78), SIMDE_FLOAT64_C(  -748.02), SIMDE_FLOAT64_C(  -681.74),
        SIMDE_FLOAT64_C(     2.59), SIMDE_FLOAT64_C(   144.16), SIMDE_FLOAT64_C(  -630.58), SIMDE_FLOAT64_C(  -881.80) },
      { SIMDE_FLOAT64_C(    36.57), SIMDE_FLOAT64_C(  -683.90), SIMDE_FLOAT64_C(  -105.32), SIMDE_FLOAT64_C(   934.82),
        SIMDE_FLOAT64_C(  -995.35), SIMDE_FLOAT64_C(   828.63), SIMDE_FLOAT64_C(  -411.86), SIMDE_FLOAT64_C(   902.67) },
      { SIMDE_FLOAT64_C(   802.80), SIMDE_FLOAT64_C( -1669.68), SIMDE_FLOAT64_C(  -853.34), SIMDE_FLOAT64_C(   253.08),
        SIMDE_FLOAT64_C(  -992.76), SIMDE_FLOAT64_C(   972.79), SIMDE_FLOAT64_C( -1042.44), SIMDE_FLOAT64_C(    20.86) } },
    { { SIMDE_FLOAT64_C(  -264.90), SIMDE_FLOAT64_C(   577.44), SIMDE_FLOAT64_C(   234.56), SIMDE_FLOAT64_C(  -420.17),
        SIMDE_FLOAT64_C(    99.35), SIMDE_FLOAT64_C(  -330.78), SIMDE_FLOAT64_C(   888.50), SIMDE_FLOAT64_C(    20.17) },
      { SIMDE_FLOAT64_C(   766.98), SIMDE_FLOAT64_C(  -871.76), SIMDE_FLOAT64_C(  -380.73), SIMDE_FLOAT64_C(    51.88),
        SIMDE_FLOAT64_C(    -9.24), SIMDE_FLOAT64_C(  -823.77), SIMDE_FLOAT64_C(   290.89), SIMDE_FLOAT64_C(  -243.01) },
      { SIMDE_FLOAT64_C(   502.08), SIMDE_FLOAT64_C(  -294.31), SIMDE_FLOAT64_C(  -146.17), SIMDE_FLOAT64_C(  -368.28),
        SIMDE_FLOAT64_C(    90.11), SIMDE_FLOAT64_C( -1154.55), SIMDE_FLOAT64_C(  1179.39), SIMDE_FLOAT64_C(  -222.84) } }
  };


  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m512d b = simde_mm512_loadu_pd(test_vec[i].b);
    simde__m512d r = simde_mm512_add_pd(a, b);
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_add_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 src[8];
    const simde__mmask8 k;
    const simde_float64 a[8];
    const simde_float64 b[8];
    const simde_float64 r[8];
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(  -241.95), SIMDE_FLOAT64_C(   348.31), SIMDE_FLOAT64_C(  -125.04), SIMDE_FLOAT64_C(  -245.69),
        SIMDE_FLOAT64_C(  -588.93), SIMDE_FLOAT64_C(  -276.58), SIMDE_FLOAT64_C(  -867.91), SIMDE_FLOAT64_C(   -10.44) },
      UINT8_C( 17),
      { SIMDE_FLOAT64_C(   296.41), SIMDE_FLOAT64_C(  -576.00), SIMDE_FLOAT64_C(   367.45), SIMDE_FLOAT64_C(  -294.17),
        SIMDE_FLOAT64_C(  -916.48), SIMDE_FLOAT64_C(   266.60), SIMDE_FLOAT64_C(   288.43), SIMDE_FLOAT64_C(   471.22) },
      { SIMDE_FLOAT64_C(  -995.36), SIMDE_FLOAT64_C(   468.56), SIMDE_FLOAT64_C(    50.02), SIMDE_FLOAT64_C(   -51.51),
        SIMDE_FLOAT64_C(   997.69), SIMDE_FLOAT64_C(   806.19), SIMDE_FLOAT64_C(  -145.36), SIMDE_FLOAT64_C(   877.33) },
      { SIMDE_FLOAT64_C(  -698.96), SIMDE_FLOAT64_C(   348.31), SIMDE_FLOAT64_C(  -125.04), SIMDE_FLOAT64_C(  -245.69),
        SIMDE_FLOAT64_C(    81.21), SIMDE_FLOAT64_C(  -276.58), SIMDE_FLOAT64_C(  -867.91), SIMDE_FLOAT64_C(   -10.44) } },
    { { SIMDE_FLOAT64_C(  -303.10), SIMDE_FLOAT64_C(  -675.79), SIMDE_FLOAT64_C(   770.76), SIMDE_FLOAT64_C(   600.76),
        SIMDE_FLOAT64_C(  -105.79), SIMDE_FLOAT64_C(  -257.88), SIMDE_FLOAT64_C(  -641.18), SIMDE_FLOAT64_C(  -757.48) },
      UINT8_C(183),
      { SIMDE_FLOAT64_C(   113.13), SIMDE_FLOAT64_C(  -346.41), SIMDE_FLOAT64_C(  -659.51), SIMDE_FLOAT64_C(   245.22),
        SIMDE_FLOAT64_C(   643.14), SIMDE_FLOAT64_C(    43.25), SIMDE_FLOAT64_C(  -458.37), SIMDE_FLOAT64_C(  -932.86) },
      { SIMDE_FLOAT64_C(  -589.30), SIMDE_FLOAT64_C(   247.46), SIMDE_FLOAT64_C(  -849.33), SIMDE_FLOAT64_C(   677.31),
        SIMDE_FLOAT64_C(  -464.11), SIMDE_FLOAT64_C(   621.89), SIMDE_FLOAT64_C(   681.94), SIMDE_FLOAT64_C(  -995.54) },
      { SIMDE_FLOAT64_C(  -476.17), SIMDE_FLOAT64_C(   -98.95), SIMDE_FLOAT64_C( -1508.84), SIMDE_FLOAT64_C(   600.76),
        SIMDE_FLOAT64_C(   179.04), SIMDE_FLOAT64_C(   665.14), SIMDE_FLOAT64_C(  -641.18), SIMDE_FLOAT64_C( -1928.40) } },
    { { SIMDE_FLOAT64_C(  -328.10), SIMDE_FLOAT64_C(  -369.57), SIMDE_FLOAT64_C(  -997.86), SIMDE_FLOAT64_C(  -521.91),
        SIMDE_FLOAT64_C(   485.07), SIMDE_FLOAT64_C(   879.48), SIMDE_FLOAT64_C(   175.00), SIMDE_FLOAT64_C(   809.28) },
      UINT8_C( 91),
      { SIMDE_FLOAT64_C(  -224.24), SIMDE_FLOAT64_C(  -296.51), SIMDE_FLOAT64_C(  -607.64), SIMDE_FLOAT64_C(   134.57),
        SIMDE_FLOAT64_C(   -53.99), SIMDE_FLOAT64_C(  -990.57), SIMDE_FLOAT64_C(  -752.30), SIMDE_FLOAT64_C(   599.60) },
      { SIMDE_FLOAT64_C(  -650.08), SIMDE_FLOAT64_C(   492.93), SIMDE_FLOAT64_C(   242.74), SIMDE_FLOAT64_C(   393.17),
        SIMDE_FLOAT64_C(  -965.44), SIMDE_FLOAT64_C(   309.89), SIMDE_FLOAT64_C(   803.88), SIMDE_FLOAT64_C(   282.02) },
      { SIMDE_FLOAT64_C(  -874.32), SIMDE_FLOAT64_C(   196.42), SIMDE_FLOAT64_C(  -997.86), SIMDE_FLOAT64_C(   527.75),
        SIMDE_FLOAT64_C( -1019.43), SIMDE_FLOAT64_C(   879.48), SIMDE_FLOAT64_C(    51.58), SIMDE_FLOAT64_C(   809.28) } },
    { { SIMDE_FLOAT64_C(   460.56), SIMDE_FLOAT64_C(   481.18), SIMDE_FLOAT64_C(   817.91), SIMDE_FLOAT64_C(    82.44),
        SIMDE_FLOAT64_C(   163.12), SIMDE_FLOAT64_C(   822.36), SIMDE_FLOAT64_C(   754.35), SIMDE_FLOAT64_C(   793.56) },
      UINT8_C( 35),
      { SIMDE_FLOAT64_C(  -767.56), SIMDE_FLOAT64_C(   278.63), SIMDE_FLOAT64_C(   703.98), SIMDE_FLOAT64_C(   407.44),
        SIMDE_FLOAT64_C(    87.91), SIMDE_FLOAT64_C(   354.22), SIMDE_FLOAT64_C(  -816.81), SIMDE_FLOAT64_C(   791.41) },
      { SIMDE_FLOAT64_C(   746.58), SIMDE_FLOAT64_C(   317.77), SIMDE_FLOAT64_C(  -262.58), SIMDE_FLOAT64_C(   756.01),
        SIMDE_FLOAT64_C(   565.47), SIMDE_FLOAT64_C(  -662.99), SIMDE_FLOAT64_C(  -894.07), SIMDE_FLOAT64_C(    58.40) },
      { SIMDE_FLOAT64_C(   -20.98), SIMDE_FLOAT64_C(   596.40), SIMDE_FLOAT64_C(   817.91), SIMDE_FLOAT64_C(    82.44),
        SIMDE_FLOAT64_C(   163.12), SIMDE_FLOAT64_C(  -308.76), SIMDE_FLOAT64_C(   754.35), SIMDE_FLOAT64_C(   793.56) } },
    { { SIMDE_FLOAT64_C(   579.76), SIMDE_FLOAT64_C(   499.11), SIMDE_FLOAT64_C(    92.96), SIMDE_FLOAT64_C(  -110.35),
        SIMDE_FLOAT64_C(   302.99), SIMDE_FLOAT64_C(  -625.02), SIMDE_FLOAT64_C(  -649.80), SIMDE_FLOAT64_C(  -215.83) },
      UINT8_C(  3),
      { SIMDE_FLOAT64_C(   432.65), SIMDE_FLOAT64_C(   947.29), SIMDE_FLOAT64_C(  -984.75), SIMDE_FLOAT64_C(   186.99),
        SIMDE_FLOAT64_C(   740.85), SIMDE_FLOAT64_C(   839.76), SIMDE_FLOAT64_C(   419.43), SIMDE_FLOAT64_C(    19.48) },
      { SIMDE_FLOAT64_C(   543.74), SIMDE_FLOAT64_C(  -173.13), SIMDE_FLOAT64_C(  -892.61), SIMDE_FLOAT64_C(  -102.04),
        SIMDE_FLOAT64_C(    10.06), SIMDE_FLOAT64_C(   898.80), SIMDE_FLOAT64_C(  -355.45), SIMDE_FLOAT64_C(  -672.17) },
      { SIMDE_FLOAT64_C(   976.39), SIMDE_FLOAT64_C(   774.16), SIMDE_FLOAT64_C(    92.96), SIMDE_FLOAT64_C(  -110.35),
        SIMDE_FLOAT64_C(   302.99), SIMDE_FLOAT64_C(  -625.02), SIMDE_FLOAT64_C(  -649.80), SIMDE_FLOAT64_C(  -215.83) } },
    { { SIMDE_FLOAT64_C(  -363.79), SIMDE_FLOAT64_C(  -599.44), SIMDE_FLOAT64_C(   893.30), SIMDE_FLOAT64_C(   -26.77),
        SIMDE_FLOAT64_C(  -493.51), SIMDE_FLOAT64_C(   -48.30), SIMDE_FLOAT64_C(  -447.01), SIMDE_FLOAT64_C(  -994.40) },
      UINT8_C( 89),
      { SIMDE_FLOAT64_C(   442.63), SIMDE_FLOAT64_C(   308.59), SIMDE_FLOAT64_C(  -580.36), SIMDE_FLOAT64_C(   792.84),
        SIMDE_FLOAT64_C(  -907.24), SIMDE_FLOAT64_C(  -387.48), SIMDE_FLOAT64_C(   225.48), SIMDE_FLOAT64_C(  -959.95) },
      { SIMDE_FLOAT64_C(  -372.23), SIMDE_FLOAT64_C(  -587.52), SIMDE_FLOAT64_C(   780.90), SIMDE_FLOAT64_C(  -532.47),
        SIMDE_FLOAT64_C(   831.91), SIMDE_FLOAT64_C(  -199.62), SIMDE_FLOAT64_C(  -988.73), SIMDE_FLOAT64_C(  -341.22) },
      { SIMDE_FLOAT64_C(    70.40), SIMDE_FLOAT64_C(  -599.44), SIMDE_FLOAT64_C(   893.30), SIMDE_FLOAT64_C(   260.36),
        SIMDE_FLOAT64_C(   -75.33), SIMDE_FLOAT64_C(   -48.30), SIMDE_FLOAT64_C(  -763.25), SIMDE_FLOAT64_C(  -994.40) } },
    { { SIMDE_FLOAT64_C(   -92.23), SIMDE_FLOAT64_C(   -90.77), SIMDE_FLOAT64_C(   668.84), SIMDE_FLOAT64_C(  -193.43),
        SIMDE_FLOAT64_C(   553.78), SIMDE_FLOAT64_C(   996.67), SIMDE_FLOAT64_C(   442.78), SIMDE_FLOAT64_C(   954.34) },
      UINT8_C(200),
      { SIMDE_FLOAT64_C(  -583.99), SIMDE_FLOAT64_C(  -539.17), SIMDE_FLOAT64_C(  -158.32), SIMDE_FLOAT64_C(   -31.00),
        SIMDE_FLOAT64_C(  -533.56), SIMDE_FLOAT64_C(  -113.65), SIMDE_FLOAT64_C(  -588.37), SIMDE_FLOAT64_C(   775.02) },
      { SIMDE_FLOAT64_C(   305.99), SIMDE_FLOAT64_C(  -795.53), SIMDE_FLOAT64_C(   867.78), SIMDE_FLOAT64_C(   918.51),
        SIMDE_FLOAT64_C(   429.95), SIMDE_FLOAT64_C(   907.83), SIMDE_FLOAT64_C(  -453.72), SIMDE_FLOAT64_C(   842.43) },
      { SIMDE_FLOAT64_C(   -92.23), SIMDE_FLOAT64_C(   -90.77), SIMDE_FLOAT64_C(   668.84), SIMDE_FLOAT64_C(   887.51),
        SIMDE_FLOAT64_C(   553.78), SIMDE_FLOAT64_C(   996.67), SIMDE_FLOAT64_C( -1042.09), SIMDE_FLOAT64_C(  1617.45) } },
    { { SIMDE_FLOAT64_C(   688.73), SIMDE_FLOAT64_C(    13.81), SIMDE_FLOAT64_C(   674.34), SIMDE_FLOAT64_C(  -510.89),
        SIMDE_FLOAT64_C(    25.08), SIMDE_FLOAT64_C(  -666.88), SIMDE_FLOAT64_C(   396.88), SIMDE_FLOAT64_C(   934.31) },
      UINT8_C(155),
      { SIMDE_FLOAT64_C(  -796.55), SIMDE_FLOAT64_C(   488.09), SIMDE_FLOAT64_C(   998.63), SIMDE_FLOAT64_C(   646.24),
        SIMDE_FLOAT64_C(   442.43), SIMDE_FLOAT64_C(   888.61), SIMDE_FLOAT64_C(  -937.75), SIMDE_FLOAT64_C(   903.26) },
      { SIMDE_FLOAT64_C(  -269.71), SIMDE_FLOAT64_C(    31.25), SIMDE_FLOAT64_C(  -630.30), SIMDE_FLOAT64_C(   616.64),
        SIMDE_FLOAT64_C(   442.88), SIMDE_FLOAT64_C(  -855.28), SIMDE_FLOAT64_C(   -77.38), SIMDE_FLOAT64_C(   647.35) },
      { SIMDE_FLOAT64_C( -1066.26), SIMDE_FLOAT64_C(   519.34), SIMDE_FLOAT64_C(   674.34), SIMDE_FLOAT64_C(  1262.87),
        SIMDE_FLOAT64_C(   885.31), SIMDE_FLOAT64_C(  -666.88), SIMDE_FLOAT64_C(   396.88), SIMDE_FLOAT64_C(  1550.61) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d src = simde_mm512_loadu_pd(test_vec[i].src);
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m512d b = simde_mm512_loadu_pd(test_vec[i].b);
    simde__m512d r = simde_mm512_mask_add_pd(src, test_vec[i].k, a, b);
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_add_pd (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde__mmask8 k;
    const simde_float64 a[8];
    const simde_float64 b[8];
    const simde_float64 r[8];
  } test_vec[] = {
    { UINT8_C(141),
      { SIMDE_FLOAT64_C(   539.39), SIMDE_FLOAT64_C(   127.65), SIMDE_FLOAT64_C(  -419.83), SIMDE_FLOAT64_C(  -509.25),
        SIMDE_FLOAT64_C(   614.81), SIMDE_FLOAT64_C(  -356.87), SIMDE_FLOAT64_C(  -437.81), SIMDE_FLOAT64_C(   217.95) },
      { SIMDE_FLOAT64_C(   -60.15), SIMDE_FLOAT64_C(  -699.30), SIMDE_FLOAT64_C(   963.74), SIMDE_FLOAT64_C(   851.36),
        SIMDE_FLOAT64_C(   773.07), SIMDE_FLOAT64_C(  -457.96), SIMDE_FLOAT64_C(  -310.92), SIMDE_FLOAT64_C(   852.62) },
      { SIMDE_FLOAT64_C(   479.24), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   543.91), SIMDE_FLOAT64_C(   342.11),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  1070.58) } },
    { UINT8_C(123),
      { SIMDE_FLOAT64_C(   902.15), SIMDE_FLOAT64_C(   661.09), SIMDE_FLOAT64_C(  -493.90), SIMDE_FLOAT64_C(   433.62),
        SIMDE_FLOAT64_C(  -884.72), SIMDE_FLOAT64_C(  -690.47), SIMDE_FLOAT64_C(  -391.44), SIMDE_FLOAT64_C(   -97.69) },
      { SIMDE_FLOAT64_C(  -732.29), SIMDE_FLOAT64_C(   446.84), SIMDE_FLOAT64_C(  -990.19), SIMDE_FLOAT64_C(   216.62),
        SIMDE_FLOAT64_C(  -720.09), SIMDE_FLOAT64_C(    35.61), SIMDE_FLOAT64_C(  -243.99), SIMDE_FLOAT64_C(   407.56) },
      { SIMDE_FLOAT64_C(   169.86), SIMDE_FLOAT64_C(  1107.93), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   650.24),
        SIMDE_FLOAT64_C( -1604.81), SIMDE_FLOAT64_C(  -654.86), SIMDE_FLOAT64_C(  -635.42), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C(175),
      { SIMDE_FLOAT64_C(   246.76), SIMDE_FLOAT64_C(    22.37), SIMDE_FLOAT64_C(  -741.09), SIMDE_FLOAT64_C(   808.94),
        SIMDE_FLOAT64_C(  -759.68), SIMDE_FLOAT64_C(   198.75), SIMDE_FLOAT64_C(  -890.36), SIMDE_FLOAT64_C(  -795.93) },
      { SIMDE_FLOAT64_C(    50.12), SIMDE_FLOAT64_C(   882.71), SIMDE_FLOAT64_C(  -253.90), SIMDE_FLOAT64_C(   739.19),
        SIMDE_FLOAT64_C(   735.33), SIMDE_FLOAT64_C(   572.27), SIMDE_FLOAT64_C(   641.34), SIMDE_FLOAT64_C(   396.42) },
      { SIMDE_FLOAT64_C(   296.87), SIMDE_FLOAT64_C(   905.08), SIMDE_FLOAT64_C(  -994.99), SIMDE_FLOAT64_C(  1548.14),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   771.02), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -399.51) } },
    { UINT8_C( 10),
      { SIMDE_FLOAT64_C(    74.96), SIMDE_FLOAT64_C(   511.70), SIMDE_FLOAT64_C(  -612.10), SIMDE_FLOAT64_C(   683.53),
        SIMDE_FLOAT64_C(  -585.99), SIMDE_FLOAT64_C(  -344.39), SIMDE_FLOAT64_C(   130.37), SIMDE_FLOAT64_C(  -576.18) },
      { SIMDE_FLOAT64_C(   872.23), SIMDE_FLOAT64_C(   410.28), SIMDE_FLOAT64_C(   459.43), SIMDE_FLOAT64_C(  -371.75),
        SIMDE_FLOAT64_C(  -182.16), SIMDE_FLOAT64_C(    75.20), SIMDE_FLOAT64_C(   875.00), SIMDE_FLOAT64_C(   840.21) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   921.98), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   311.77),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C( 10),
      { SIMDE_FLOAT64_C(   683.95), SIMDE_FLOAT64_C(  -919.47), SIMDE_FLOAT64_C(  -467.14), SIMDE_FLOAT64_C(   793.59),
        SIMDE_FLOAT64_C(  -715.40), SIMDE_FLOAT64_C(   582.98), SIMDE_FLOAT64_C(   676.29), SIMDE_FLOAT64_C(    30.70) },
      { SIMDE_FLOAT64_C(   322.17), SIMDE_FLOAT64_C(   411.62), SIMDE_FLOAT64_C(  -397.03), SIMDE_FLOAT64_C(   -36.48),
        SIMDE_FLOAT64_C(  -191.96), SIMDE_FLOAT64_C(  -318.66), SIMDE_FLOAT64_C(  -961.52), SIMDE_FLOAT64_C(  -680.25) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -507.84), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   757.10),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C(199),
      { SIMDE_FLOAT64_C(   722.01), SIMDE_FLOAT64_C(  -266.24), SIMDE_FLOAT64_C(   724.85), SIMDE_FLOAT64_C(  -147.62),
        SIMDE_FLOAT64_C(   157.58), SIMDE_FLOAT64_C(   597.08), SIMDE_FLOAT64_C(  -737.35), SIMDE_FLOAT64_C(  -383.00) },
      { SIMDE_FLOAT64_C(  -774.68), SIMDE_FLOAT64_C(    80.49), SIMDE_FLOAT64_C(   692.21), SIMDE_FLOAT64_C(  -899.67),
        SIMDE_FLOAT64_C(   -79.30), SIMDE_FLOAT64_C(    26.32), SIMDE_FLOAT64_C(   784.27), SIMDE_FLOAT64_C(     1.24) },
      { SIMDE_FLOAT64_C(   -52.67), SIMDE_FLOAT64_C(  -185.75), SIMDE_FLOAT64_C(  1417.06), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(    46.93), SIMDE_FLOAT64_C(  -381.76) } },
    { UINT8_C(108),
      { SIMDE_FLOAT64_C(   577.86), SIMDE_FLOAT64_C(   285.84), SIMDE_FLOAT64_C(   142.16), SIMDE_FLOAT64_C(   254.16),
        SIMDE_FLOAT64_C(  -683.46), SIMDE_FLOAT64_C(  -535.67), SIMDE_FLOAT64_C(  -334.22), SIMDE_FLOAT64_C(   -80.49) },
      { SIMDE_FLOAT64_C(   427.85), SIMDE_FLOAT64_C(   473.82), SIMDE_FLOAT64_C(   600.85), SIMDE_FLOAT64_C(   466.33),
        SIMDE_FLOAT64_C(   793.57), SIMDE_FLOAT64_C(  -329.91), SIMDE_FLOAT64_C(   188.34), SIMDE_FLOAT64_C(  -472.67) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(   743.02), SIMDE_FLOAT64_C(   720.48),
        SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(  -865.57), SIMDE_FLOAT64_C(  -145.89), SIMDE_FLOAT64_C(     0.00) } },
    { UINT8_C( 48),
      { SIMDE_FLOAT64_C(  -959.29), SIMDE_FLOAT64_C(   684.90), SIMDE_FLOAT64_C(   992.02), SIMDE_FLOAT64_C(  -696.63),
        SIMDE_FLOAT64_C(  -698.09), SIMDE_FLOAT64_C(  -782.66), SIMDE_FLOAT64_C(   383.86), SIMDE_FLOAT64_C(   994.11) },
      { SIMDE_FLOAT64_C(  -682.33), SIMDE_FLOAT64_C(  -695.44), SIMDE_FLOAT64_C(    20.43), SIMDE_FLOAT64_C(  -898.06),
        SIMDE_FLOAT64_C(   305.80), SIMDE_FLOAT64_C(  -420.39), SIMDE_FLOAT64_C(   679.80), SIMDE_FLOAT64_C(  -408.37) },
      { SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00),
        SIMDE_FLOAT64_C(  -392.30), SIMDE_FLOAT64_C( -1203.04), SIMDE_FLOAT64_C(     0.00), SIMDE_FLOAT64_C(     0.00) } }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m512d b = simde_mm512_loadu_pd(test_vec[i].b);
    simde__m512d r = simde_mm512_maskz_add_pd(test_vec[i].k, a, b);
    #if defined(__EMSCRIPTEN__)
    (void) r;
    #else
    simde_test_x86_assert_equal_f64x8(r, simde_mm512_loadu_pd(test_vec[i].r), 1);
    #endif
  }

  return 0;
}

static int
test_simde_mm512_and_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 2103907232), INT32_C(-1995421302), INT32_C( 1328084931), INT32_C( -379562245),
                            INT32_C( 1144599747), INT32_C(-1418414219), INT32_C( 1379143176), INT32_C(-2075387410),
                            INT32_C(-1152868472), INT32_C( 1425101887), INT32_C(-1626225579), INT32_C( 2014677400),
                            INT32_C( 1108777022), INT32_C(  877429697), INT32_C( 1558575022), INT32_C(  651318578)),
      simde_mm512_set_epi32(INT32_C( -803993576), INT32_C(  163813867), INT32_C(-1017884831), INT32_C(-1258992440),
                            INT32_C( 1182354404), INT32_C(-1425047283), INT32_C( 1708628167), INT32_C(-2051115159),
                            INT32_C(  143410800), INT32_C(-1041180848), INT32_C(-1023702255), INT32_C( -240255997),
                            INT32_C(  978107452), INT32_C(-1343783755), INT32_C( -880122145), INT32_C( 1183808039)),
      simde_mm512_set_epi32(INT32_C( 1342439424), INT32_C(  150997386), INT32_C( 1124091713), INT32_C(-1604303672),
                            INT32_C( 1144586432), INT32_C(-1425768187), INT32_C( 1075054592), INT32_C(-2079583896),
                            INT32_C(  134743552), INT32_C( 1089491984), INT32_C(-2112782319), INT32_C( 1879410688),
                            INT32_C(   33855548), INT32_C(  608436353), INT32_C( 1216374414), INT32_C(  109203490)) },
    { simde_mm512_set_epi32(INT32_C( 1231278072), INT32_C(  832865002), INT32_C(-1694490420), INT32_C( -466764866),
                            INT32_C( 1702721236), INT32_C( 2092439418), INT32_C(  611933067), INT32_C(-1184445720),
                            INT32_C(-1689922195), INT32_C(-1140532352), INT32_C(  158346875), INT32_C( 1089717474),
                            INT32_C( 1230253618), INT32_C( 1504968720), INT32_C(  653725328), INT32_C( 1822881045)),
      simde_mm512_set_epi32(INT32_C(-1605590998), INT32_C( 1810066171), INT32_C(  -10167007), INT32_C( 1359569321),
                            INT32_C(-1430119757), INT32_C(  256064867), INT32_C(  788618356), INT32_C( -392789188),
                            INT32_C( -613873031), INT32_C(  773158597), INT32_C( -194106681), INT32_C(  165465349),
                            INT32_C(-1346434037), INT32_C(-1412186885), INT32_C( -632750822), INT32_C( 1966832804)),
      simde_mm512_set_epi32(INT32_C(    4227112), INT32_C(  564134634), INT32_C(-1694498816), INT32_C( 1074340264),
                            INT32_C(  541069968), INT32_C(  201332066), INT32_C(  604067840), INT32_C(-1475968472),
                            INT32_C(-1690238871), INT32_C(  738480768), INT32_C(    6302275), INT32_C(   13681664),
                            INT32_C(  152308738), INT32_C(  160432144), INT32_C(   37751824), INT32_C( 1679979524)) },
    { simde_mm512_set_epi32(INT32_C(-1888865381), INT32_C(   15541452), INT32_C( -670396349), INT32_C(-1090081489),
                            INT32_C(  116025329), INT32_C(  130963716), INT32_C(  230354364), INT32_C( 1174065929),
                            INT32_C( 1971493681), INT32_C(-1343257591), INT32_C(-1419733408), INT32_C(  659096905),
                            INT32_C(  183711411), INT32_C( -298263182), INT32_C(-1286938208), INT32_C(  537660993)),
      simde_mm512_set_epi32(INT32_C( 1616282233), INT32_C(-2132390836), INT32_C( -153917613), INT32_C(  736513734),
                            INT32_C(-1311057544), INT32_C(  505476811), INT32_C( 1767695145), INT32_C(  157469724),
                            INT32_C( -371725260), INT32_C( 1996701751), INT32_C(-1377678442), INT32_C( -132601652),
                            INT32_C( -903948497), INT32_C( -794660034), INT32_C(  173952757), INT32_C( 1507611872)),
      simde_mm512_set_epi32(INT32_C(    4325913), INT32_C(   14943308), INT32_C( -805175229), INT32_C(  721816582),
                            INT32_C(   13255024), INT32_C(  100683776), INT32_C(  152625448), INT32_C(   23248904),
                            INT32_C( 1635944496), INT32_C(  654508033), INT32_C(-1453325824), INT32_C(  537395272),
                            INT32_C(  168957475), INT32_C(-1071623886), INT32_C(   38420640), INT32_C(     787520)) },
    { simde_mm512_set_epi32(INT32_C( -748350470), INT32_C( 1755197901), INT32_C( 1090059253), INT32_C( 1329426651),
                            INT32_C(-1604442789), INT32_C(  878047098), INT32_C(-1682276633), INT32_C(  -78811559),
                            INT32_C( -973139496), INT32_C(   91517188), INT32_C(   37440120), INT32_C(  829401648),
                            INT32_C(  860279707), INT32_C(  218373799), INT32_C( -362113249), INT32_C( -694086277)),
      simde_mm512_set_epi32(INT32_C(  892006932), INT32_C(-1158010747), INT32_C( -700127305), INT32_C(-1058033333),
                            INT32_C( 2135825335), INT32_C( -510043422), INT32_C( -139544800), INT32_C(   95748631),
                            INT32_C( 1671238992), INT32_C( 1305915968), INT32_C( -582790199), INT32_C( 1758236157),
                            INT32_C(  986405639), INT32_C( 1121234438), INT32_C( -731393712), INT32_C( -784753228)),
      simde_mm512_set_epi32(INT32_C(  287314448), INT32_C(  681189509), INT32_C( 1077994421), INT32_C( 1076703307),
                            INT32_C(  541982995), INT32_C(  538004066), INT32_C(-1817561568), INT32_C(   17105425),
                            INT32_C( 1100812624), INT32_C(   89403392), INT32_C(     213576), INT32_C(  541886512),
                            INT32_C(  843207427), INT32_C(     265734), INT32_C(-1067282160), INT32_C( -803138256)) },
    { simde_mm512_set_epi32(INT32_C( 1347900829), INT32_C(-1792354715), INT32_C( -371177698), INT32_C(  255088013),
                            INT32_C( 1961231505), INT32_C( -659343095), INT32_C( 1620234692), INT32_C(  843561067),
                            INT32_C( 1265300992), INT32_C(-1675104490), INT32_C( -873664156), INT32_C(-2045109653),
                            INT32_C( 2057630636), INT32_C(  335188274), INT32_C( 1272591061), INT32_C( -327494197)),
      simde_mm512_set_epi32(INT32_C(-2095740678), INT32_C(-1857753563), INT32_C(-1236342636), INT32_C( 1439297909),
                            INT32_C( -576201057), INT32_C( 1488873085), INT32_C(-1369304746), INT32_C(  567848046),
                            INT32_C( 1335236564), INT32_C( -942680632), INT32_C(-1512916560), INT32_C( -697747292),
                            INT32_C(-1779695782), INT32_C(  491327584), INT32_C( -563681080), INT32_C( 1527319596)),
      simde_mm512_set_epi32(INT32_C(    1376408), INT32_C(-1862220251), INT32_C(-1606402028), INT32_C(   83906821),
                            INT32_C( 1420166289), INT32_C( 1488068617), INT32_C(  537006404), INT32_C(  541108330),
                            INT32_C( 1258427904), INT32_C(-2079865600), INT32_C(-2118078176), INT32_C(-2046289888),
                            INT32_C(  278983432), INT32_C(  289935392), INT32_C( 1245847744), INT32_C( 1208484872)) },
    { simde_mm512_set_epi32(INT32_C(  131205926), INT32_C( 2061955170), INT32_C(   37003574), INT32_C( 1649229141),
                            INT32_C(  612060260), INT32_C(-1402263233), INT32_C( -513572270), INT32_C(  701923816),
                            INT32_C(  511549547), INT32_C(  969083331), INT32_C( 1364542630), INT32_C( -822209230),
                            INT32_C(-1549704264), INT32_C(-1157339218), INT32_C(-2025137124), INT32_C(-1631723043)),
      simde_mm512_set_epi32(INT32_C(  424825857), INT32_C(  434716327), INT32_C( 1663095683), INT32_C( 1730428966),
                            INT32_C( -252943126), INT32_C(  373990324), INT32_C( 2100741912), INT32_C(-1005385937),
                            INT32_C(-1470990839), INT32_C(-1200692008), INT32_C(-2098508971), INT32_C(  488824783),
                            INT32_C( 1647236603), INT32_C(-1221154150), INT32_C(-1012750616), INT32_C( 1849794587)),
      simde_mm512_set_epi32(INT32_C(   22151168), INT32_C(  417345570), INT32_C(   35684610), INT32_C( 1644429316),
                            INT32_C(  543704160), INT32_C(   71967028), INT32_C( 1629651472), INT32_C(    1179944),
                            INT32_C(  139468809), INT32_C(  943849664), INT32_C(    4276228), INT32_C(  203560194),
                            INT32_C(  572539320), INT32_C(-1291574646), INT32_C(-2096987128), INT32_C(  234987545)) },
    { simde_mm512_set_epi32(INT32_C(-2080136983), INT32_C( -623547588), INT32_C( 1015056564), INT32_C(-1333355305),
                            INT32_C( 1925062912), INT32_C(  330138155), INT32_C( -444195598), INT32_C(  874806560),
                            INT32_C(  319126943), INT32_C(  475403370), INT32_C( -873396634), INT32_C(-1835948135),
                            INT32_C(-1570208244), INT32_C( -244400530), INT32_C( 1126824505), INT32_C( 1036340167)),
      simde_mm512_set_epi32(INT32_C(-1989715102), INT32_C( 1785805415), INT32_C( 1739507553), INT32_C(-1932540399),
                            INT32_C( -641363523), INT32_C(   72499486), INT32_C(-2005610208), INT32_C( -497730582),
                            INT32_C(  312611747), INT32_C(  793357988), INT32_C(  607421194), INT32_C(-1375787856),
                            INT32_C( 1799684145), INT32_C( -551549754), INT32_C( 1249951615), INT32_C( 1447553529)),
      simde_mm512_set_epi32(INT32_C(-2147286944), INT32_C( 1246833188), INT32_C(  612401184), INT32_C(-2138668015),
                            INT32_C( 1350828288), INT32_C(         10), INT32_C(-2147217376), INT32_C(  537147680),
                            INT32_C(  301994371), INT32_C(  205521952), INT32_C(    3146242), INT32_C(-2137971568),
                            INT32_C(  574619648), INT32_C( -786430906), INT32_C( 1107343929), INT32_C(  340085185)) },
    { simde_mm512_set_epi32(INT32_C(  423295425), INT32_C( -460615607), INT32_C( 1208771148), INT32_C(-2128303155),
                            INT32_C( -738338972), INT32_C( 2110676823), INT32_C(-1405320678), INT32_C(-2007459833),
                            INT32_C( 1043638626), INT32_C( -542891463), INT32_C(  629803756), INT32_C(-1216921331),
                            INT32_C( -301860714), INT32_C(  317296385), INT32_C( 1833800187), INT32_C( -645353377)),
      simde_mm512_set_epi32(INT32_C( -257851255), INT32_C( -479522767), INT32_C(  745275629), INT32_C(-1783480446),
                            INT32_C(-1431666964), INT32_C( -422291816), INT32_C( -309252994), INT32_C(-2112946871),
                            INT32_C(-2123521230), INT32_C( -883174176), INT32_C( -332498316), INT32_C(  142889340),
                            INT32_C( 1231002435), INT32_C( 1681334055), INT32_C(  138372594), INT32_C( -878588971)),
      simde_mm512_set_epi32(INT32_C(  270565505), INT32_C( -536145919), INT32_C(  135004236), INT32_C(-2128598144),
                            INT32_C(-2102897052), INT32_C( 1690588688), INT32_C(-1408237542), INT32_C(-2146920447),
                            INT32_C(    2400546), INT32_C( -889192416), INT32_C(  604637796), INT32_C(     278796),
                            INT32_C( 1208064002), INT32_C(    2166529), INT32_C(  135070194), INT32_C( -914325419)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_and_epi32(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_and_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 2103907232), INT32_C(-1995421302), INT32_C( 1328084931), INT32_C( -379562245),
                            INT32_C( 1144599747), INT32_C(-1418414219), INT32_C( 1379143176), INT32_C(-2075387410),
                            INT32_C(-1152868472), INT32_C( 1425101887), INT32_C(-1626225579), INT32_C( 2014677400),
                            INT32_C( 1108777022), INT32_C(  877429697), INT32_C( 1558575022), INT32_C(  651318578)),
      UINT16_C(31271),
      simde_mm512_set_epi32(INT32_C( 1822881045), INT32_C( -803993576), INT32_C(  163813867), INT32_C(-1017884831),
                            INT32_C(-1258992440), INT32_C( 1182354404), INT32_C(-1425047283), INT32_C( 1708628167),
                            INT32_C(-2051115159), INT32_C(  143410800), INT32_C(-1041180848), INT32_C(-1023702255),
                            INT32_C( -240255997), INT32_C(  978107452), INT32_C(-1343783755), INT32_C( -880122145)),
      simde_mm512_set_epi32(INT32_C( 1966832804), INT32_C( 1231278072), INT32_C(  832865002), INT32_C(-1694490420),
                            INT32_C( -466764866), INT32_C( 1702721236), INT32_C( 2092439418), INT32_C(  611933067),
                            INT32_C(-1184445720), INT32_C(-1689922195), INT32_C(-1140532352), INT32_C(  158346875),
                            INT32_C( 1089717474), INT32_C( 1230253618), INT32_C( 1504968720), INT32_C(  653725328)),
      simde_mm512_set_epi32(INT32_C( 2103907232), INT32_C( 1073743896), INT32_C(   25198826), INT32_C(-2097151936),
                            INT32_C(-1541076856), INT32_C(-1418414219), INT32_C(  671613192), INT32_C(-2075387410),
                            INT32_C(-1152868472), INT32_C( 1425101887), INT32_C(-2147429632), INT32_C( 2014677400),
                            INT32_C( 1108777022), INT32_C(  138685488), INT32_C(  161742864), INT32_C(   42075792)) },
    { simde_mm512_set_epi32(INT32_C(  537660993), INT32_C(-1605590998), INT32_C( 1810066171), INT32_C(  -10167007),
                            INT32_C( 1359569321), INT32_C(-1430119757), INT32_C(  256064867), INT32_C(  788618356),
                            INT32_C( -392789188), INT32_C( -613873031), INT32_C(  773158597), INT32_C( -194106681),
                            INT32_C(  165465349), INT32_C(-1346434037), INT32_C(-1412186885), INT32_C( -632750822)),
      UINT16_C(57760),
      simde_mm512_set_epi32(INT32_C(  173952757), INT32_C( 1507611872), INT32_C(-1888865381), INT32_C(   15541452),
                            INT32_C( -670396349), INT32_C(-1090081489), INT32_C(  116025329), INT32_C(  130963716),
                            INT32_C(  230354364), INT32_C( 1174065929), INT32_C( 1971493681), INT32_C(-1343257591),
                            INT32_C(-1419733408), INT32_C(  659096905), INT32_C(  183711411), INT32_C( -298263182)),
      simde_mm512_set_epi32(INT32_C( -362113249), INT32_C( -694086277), INT32_C( 1616282233), INT32_C(-2132390836),
                            INT32_C( -153917613), INT32_C(  736513734), INT32_C(-1311057544), INT32_C(  505476811),
                            INT32_C( 1767695145), INT32_C(  157469724), INT32_C( -371725260), INT32_C( 1996701751),
                            INT32_C(-1377678442), INT32_C( -132601652), INT32_C( -903948497), INT32_C( -794660034)),
      simde_mm512_set_epi32(INT32_C(  172623381), INT32_C( 1350571104), INT32_C(    4325913), INT32_C(  -10167007),
                            INT32_C( 1359569321), INT32_C(-1430119757), INT32_C(  256064867), INT32_C(  100683776),
                            INT32_C(  152625448), INT32_C( -613873031), INT32_C( 1635944496), INT32_C( -194106681),
                            INT32_C(  165465349), INT32_C(-1346434037), INT32_C(-1412186885), INT32_C( -632750822)) },
    { simde_mm512_set_epi32(INT32_C( -731393712), INT32_C( -784753228), INT32_C( -748350470), INT32_C( 1755197901),
                            INT32_C( 1090059253), INT32_C( 1329426651), INT32_C(-1604442789), INT32_C(  878047098),
                            INT32_C(-1682276633), INT32_C(  -78811559), INT32_C( -973139496), INT32_C(   91517188),
                            INT32_C(   37440120), INT32_C(  829401648), INT32_C(  860279707), INT32_C(  218373799)),
      UINT16_C(44550),
      simde_mm512_set_epi32(INT32_C(  335188274), INT32_C( 1272591061), INT32_C( -327494197), INT32_C(  892006932),
                            INT32_C(-1158010747), INT32_C( -700127305), INT32_C(-1058033333), INT32_C( 2135825335),
                            INT32_C( -510043422), INT32_C( -139544800), INT32_C(   95748631), INT32_C( 1671238992),
                            INT32_C( 1305915968), INT32_C( -582790199), INT32_C( 1758236157), INT32_C(  986405639)),
      simde_mm512_set_epi32(INT32_C(  491327584), INT32_C( -563681080), INT32_C( 1527319596), INT32_C( 1347900829),
                            INT32_C(-1792354715), INT32_C( -371177698), INT32_C(  255088013), INT32_C( 1961231505),
                            INT32_C( -659343095), INT32_C( 1620234692), INT32_C(  843561067), INT32_C( 1265300992),
                            INT32_C(-1675104490), INT32_C( -873664156), INT32_C(-2045109653), INT32_C( 2057630636)),
      simde_mm512_set_epi32(INT32_C(  289935392), INT32_C( -784753228), INT32_C( 1208484872), INT32_C( 1755197901),
                            INT32_C(-1876295675), INT32_C(-1069529322), INT32_C(    2363657), INT32_C(  878047098),
                            INT32_C(-1682276633), INT32_C(  -78811559), INT32_C( -973139496), INT32_C(   91517188),
                            INT32_C(   37440120), INT32_C( -918531776), INT32_C(     526441), INT32_C(  218373799)) },
    { simde_mm512_set_epi32(INT32_C(-1157339218), INT32_C(-2025137124), INT32_C(-1631723043), INT32_C(-2095740678),
                            INT32_C(-1857753563), INT32_C(-1236342636), INT32_C( 1439297909), INT32_C( -576201057),
                            INT32_C( 1488873085), INT32_C(-1369304746), INT32_C(  567848046), INT32_C( 1335236564),
                            INT32_C( -942680632), INT32_C(-1512916560), INT32_C( -697747292), INT32_C(-1779695782)),
      UINT16_C(25528),
      simde_mm512_set_epi32(INT32_C( 1647236603), INT32_C(-1221154150), INT32_C(-1012750616), INT32_C( 1849794587),
                            INT32_C(  131205926), INT32_C( 2061955170), INT32_C(   37003574), INT32_C( 1649229141),
                            INT32_C(  612060260), INT32_C(-1402263233), INT32_C( -513572270), INT32_C(  701923816),
                            INT32_C(  511549547), INT32_C(  969083331), INT32_C( 1364542630), INT32_C( -822209230)),
      simde_mm512_set_epi32(INT32_C(-1570208244), INT32_C( -244400530), INT32_C( 1126824505), INT32_C( 1036340167),
                            INT32_C(  424825857), INT32_C(  434716327), INT32_C( 1663095683), INT32_C( 1730428966),
                            INT32_C( -252943126), INT32_C(  373990324), INT32_C( 2100741912), INT32_C(-1005385937),
                            INT32_C(-1470990839), INT32_C(-1200692008), INT32_C(-2098508971), INT32_C(  488824783)),
      simde_mm512_set_epi32(INT32_C(-1157339218), INT32_C(-1322866166), INT32_C( 1126212136), INT32_C(-2095740678),
                            INT32_C(-1857753563), INT32_C(-1236342636), INT32_C(   35684610), INT32_C( 1644429316),
                            INT32_C(  543704160), INT32_C(-1369304746), INT32_C( 1629651472), INT32_C(    1179944),
                            INT32_C(  139468809), INT32_C(-1512916560), INT32_C( -697747292), INT32_C(-1779695782)) },
    { simde_mm512_set_epi32(INT32_C( 1799684145), INT32_C( -551549754), INT32_C( 1249951615), INT32_C( 1447553529),
                            INT32_C(-2080136983), INT32_C( -623547588), INT32_C( 1015056564), INT32_C(-1333355305),
                            INT32_C( 1925062912), INT32_C(  330138155), INT32_C( -444195598), INT32_C(  874806560),
                            INT32_C(  319126943), INT32_C(  475403370), INT32_C( -873396634), INT32_C(-1835948135)),
      UINT16_C( 9392),
      simde_mm512_set_epi32(INT32_C(-1216921331), INT32_C( -301860714), INT32_C(  317296385), INT32_C( 1833800187),
                            INT32_C( -645353377), INT32_C(-1989715102), INT32_C( 1785805415), INT32_C( 1739507553),
                            INT32_C(-1932540399), INT32_C( -641363523), INT32_C(   72499486), INT32_C(-2005610208),
                            INT32_C( -497730582), INT32_C(  312611747), INT32_C(  793357988), INT32_C(  607421194)),
      simde_mm512_set_epi32(INT32_C(  142889340), INT32_C( 1231002435), INT32_C( 1681334055), INT32_C(  138372594),
                            INT32_C( -878588971), INT32_C(  423295425), INT32_C( -460615607), INT32_C( 1208771148),
                            INT32_C(-2128303155), INT32_C( -738338972), INT32_C( 2110676823), INT32_C(-1405320678),
                            INT32_C(-2007459833), INT32_C( 1043638626), INT32_C( -542891463), INT32_C(  629803756)),
      simde_mm512_set_epi32(INT32_C( 1799684145), INT32_C( -551549754), INT32_C(    2166529), INT32_C( 1447553529),
                            INT32_C(-2080136983), INT32_C(  153246016), INT32_C( 1015056564), INT32_C(-1333355305),
                            INT32_C(-2147188223), INT32_C(  330138155), INT32_C(   71450902), INT32_C(-2009825280),
                            INT32_C(  319126943), INT32_C(  475403370), INT32_C( -873396634), INT32_C(-1835948135)) },
    { simde_mm512_set_epi32(INT32_C(  861635987), INT32_C( 1823839521), INT32_C( 1391000031), INT32_C(   73229946),
                            INT32_C(  -53693878), INT32_C( -257851255), INT32_C( -479522767), INT32_C(  745275629),
                            INT32_C(-1783480446), INT32_C(-1431666964), INT32_C( -422291816), INT32_C( -309252994),
                            INT32_C(-2112946871), INT32_C(-2123521230), INT32_C( -883174176), INT32_C( -332498316)),
      UINT16_C(31381),
      simde_mm512_set_epi32(INT32_C( -410707923), INT32_C( -804790801), INT32_C( -675940069), INT32_C(  717543141),
                            INT32_C( 1610339352), INT32_C(  785451213), INT32_C(  -67248356), INT32_C(-1147482606),
                            INT32_C(  877778312), INT32_C( 1833609670), INT32_C( 1105011960), INT32_C(-1909564752),
                            INT32_C(-2137129603), INT32_C(-1991115340), INT32_C( -941377596), INT32_C(-1151664921)),
      simde_mm512_set_epi32(INT32_C( -605598510), INT32_C( 1332169075), INT32_C(  829771204), INT32_C(  806631323),
                            INT32_C(  -62111889), INT32_C( 1452741835), INT32_C(  921236435), INT32_C(-1348081811),
                            INT32_C( -883327193), INT32_C(-1324808596), INT32_C(-2119312832), INT32_C(-1371509978),
                            INT32_C(-1887676953), INT32_C(   42335263), INT32_C(  818544934), INT32_C(-1864687690)),
      simde_mm512_set_epi32(INT32_C(  861635987), INT32_C( 1074219363), INT32_C(  288704768), INT32_C(  537138817),
                            INT32_C( 1548227592), INT32_C( -257851255), INT32_C(  854119184), INT32_C(  745275629),
                            INT32_C(    5330176), INT32_C(-1431666964), INT32_C( -422291816), INT32_C(-1912583648),
                            INT32_C(-2112946871), INT32_C(     130068), INT32_C( -883174176), INT32_C(-1873142618)) },
    { simde_mm512_set_epi32(INT32_C( 1456151906), INT32_C( -346366427), INT32_C(  534496658), INT32_C( 1981510934),
                            INT32_C( -935678271), INT32_C( 1523008579), INT32_C(   -6105095), INT32_C( 2115600842),
                            INT32_C( -420343454), INT32_C(  652783640), INT32_C( -871055383), INT32_C(  142253075),
                            INT32_C(  557825344), INT32_C(  707825888), INT32_C(  944883191), INT32_C( 1704858885)),
      UINT16_C(19039),
      simde_mm512_set_epi32(INT32_C( 1893303454), INT32_C( 1567616976), INT32_C( 1190892677), INT32_C( 1594451864),
                            INT32_C(-1033342432), INT32_C( -738674203), INT32_C(-1847547828), INT32_C( 1893640833),
                            INT32_C(   26320713), INT32_C( 1830669951), INT32_C( 1304924639), INT32_C( -277717409),
                            INT32_C(-1566722863), INT32_C( 1534951086), INT32_C( -925669609), INT32_C(  359322092)),
      simde_mm512_set_epi32(INT32_C( 1160904262), INT32_C(  732990033), INT32_C(  138388028), INT32_C( 1168180194),
                            INT32_C( 1057944486), INT32_C( 1875512725), INT32_C( -910818137), INT32_C( -743685110),
                            INT32_C( -947775444), INT32_C( 1458691146), INT32_C( 1273454073), INT32_C(  927814838),
                            INT32_C(-1840329583), INT32_C( -789758267), INT32_C( -870216121), INT32_C(  449327093)),
      simde_mm512_set_epi32(INT32_C( 1456151906), INT32_C(  153127504), INT32_C(  534496658), INT32_C( 1981510934),
                            INT32_C(   34107936), INT32_C( 1523008579), INT32_C(-2120220668), INT32_C( 2115600842),
                            INT32_C( -420343454), INT32_C( 1142017098), INT32_C( -871055383), INT32_C(  658527254),
                            INT32_C(-2113108847), INT32_C( 1349336708), INT32_C( -939457017), INT32_C(  273154532)) },
    { simde_mm512_set_epi32(INT32_C(-1055194531), INT32_C( 1846727705), INT32_C(  335680535), INT32_C( -610713755),
                            INT32_C(  944256620), INT32_C(  697979892), INT32_C(   49552843), INT32_C( -460412596),
                            INT32_C(-2060335241), INT32_C(  135497979), INT32_C( -331098630), INT32_C( -140680021),
                            INT32_C(-1676162464), INT32_C(  626483741), INT32_C(  170885439), INT32_C(  230851400)),
      UINT16_C(10528),
      simde_mm512_set_epi32(INT32_C(-1447071985), INT32_C( -310464227), INT32_C( -679161042), INT32_C( -527096592),
                            INT32_C( 1833269922), INT32_C(-1164990327), INT32_C( 1955493691), INT32_C( 1263046717),
                            INT32_C( 1122048689), INT32_C( 2074234443), INT32_C(  723669938), INT32_C(  284884896),
                            INT32_C( -458264538), INT32_C(  514387150), INT32_C(-1369468153), INT32_C( 1859652102)),
      simde_mm512_set_epi32(INT32_C(-1315612420), INT32_C( -330188185), INT32_C(     890374), INT32_C( -958458643),
                            INT32_C( 2023887571), INT32_C(  226435011), INT32_C(    1929100), INT32_C(-1975437469),
                            INT32_C(  871389437), INT32_C(-2041059805), INT32_C(  636566673), INT32_C( -485819928),
                            INT32_C(  904524629), INT32_C(-1963695561), INT32_C( -297278672), INT32_C(-1666020030)),
      simde_mm512_set_epi32(INT32_C(-1055194531), INT32_C( 1846727705), INT32_C(     299526), INT32_C( -610713755),
                            INT32_C( 1744835202), INT32_C(  697979892), INT32_C(   49552843), INT32_C(  171970593),
                            INT32_C(-2060335241), INT32_C(  135497979), INT32_C(  555749520), INT32_C( -140680021),
                            INT32_C(-1676162464), INT32_C(  626483741), INT32_C(  170885439), INT32_C(  230851400)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_and_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_and_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(21810),
      simde_mm512_set_epi32(INT32_C( 1183808039), INT32_C( 2103907232), INT32_C(-1995421302), INT32_C( 1328084931),
                            INT32_C( -379562245), INT32_C( 1144599747), INT32_C(-1418414219), INT32_C( 1379143176),
                            INT32_C(-2075387410), INT32_C(-1152868472), INT32_C( 1425101887), INT32_C(-1626225579),
                            INT32_C( 2014677400), INT32_C( 1108777022), INT32_C(  877429697), INT32_C( 1558575022)),
      simde_mm512_set_epi32(INT32_C( 1822881045), INT32_C( -803993576), INT32_C(  163813867), INT32_C(-1017884831),
                            INT32_C(-1258992440), INT32_C( 1182354404), INT32_C(-1425047283), INT32_C( 1708628167),
                            INT32_C(-2051115159), INT32_C(  143410800), INT32_C(-1041180848), INT32_C(-1023702255),
                            INT32_C( -240255997), INT32_C(  978107452), INT32_C(-1343783755), INT32_C( -880122145)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C( 1342439424), INT32_C(          0), INT32_C( 1124091713),
                            INT32_C(          0), INT32_C( 1144586432), INT32_C(          0), INT32_C( 1075054592),
                            INT32_C(          0), INT32_C(          0), INT32_C( 1089491984), INT32_C(-2112782319),
                            INT32_C(          0), INT32_C(          0), INT32_C(  608436353), INT32_C(          0)) },
    { UINT16_C( 3728),
      simde_mm512_set_epi32(INT32_C( -632750822), INT32_C( 1966832804), INT32_C( 1231278072), INT32_C(  832865002),
                            INT32_C(-1694490420), INT32_C( -466764866), INT32_C( 1702721236), INT32_C( 2092439418),
                            INT32_C(  611933067), INT32_C(-1184445720), INT32_C(-1689922195), INT32_C(-1140532352),
                            INT32_C(  158346875), INT32_C( 1089717474), INT32_C( 1230253618), INT32_C( 1504968720)),
      simde_mm512_set_epi32(INT32_C(-1286938208), INT32_C(  537660993), INT32_C(-1605590998), INT32_C( 1810066171),
                            INT32_C(  -10167007), INT32_C( 1359569321), INT32_C(-1430119757), INT32_C(  256064867),
                            INT32_C(  788618356), INT32_C( -392789188), INT32_C( -613873031), INT32_C(  773158597),
                            INT32_C( -194106681), INT32_C(  165465349), INT32_C(-1346434037), INT32_C(-1412186885)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(-1694498816), INT32_C( 1074340264), INT32_C(  541069968), INT32_C(          0),
                            INT32_C(  604067840), INT32_C(          0), INT32_C(          0), INT32_C(  738480768),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(56690),
      simde_mm512_set_epi32(INT32_C( -794660034), INT32_C(  173952757), INT32_C( 1507611872), INT32_C(-1888865381),
                            INT32_C(   15541452), INT32_C( -670396349), INT32_C(-1090081489), INT32_C(  116025329),
                            INT32_C(  130963716), INT32_C(  230354364), INT32_C( 1174065929), INT32_C( 1971493681),
                            INT32_C(-1343257591), INT32_C(-1419733408), INT32_C(  659096905), INT32_C(  183711411)),
      simde_mm512_set_epi32(INT32_C(  218373799), INT32_C( -362113249), INT32_C( -694086277), INT32_C( 1616282233),
                            INT32_C(-2132390836), INT32_C( -153917613), INT32_C(  736513734), INT32_C(-1311057544),
                            INT32_C(  505476811), INT32_C( 1767695145), INT32_C(  157469724), INT32_C( -371725260),
                            INT32_C( 1996701751), INT32_C(-1377678442), INT32_C( -132601652), INT32_C( -903948497)),
      simde_mm512_set_epi32(INT32_C(       4646), INT32_C(  172623381), INT32_C(          0), INT32_C(    4325913),
                            INT32_C(   14943308), INT32_C( -805175229), INT32_C(          0), INT32_C(   13255024),
                            INT32_C(          0), INT32_C(  152625448), INT32_C(   23248904), INT32_C( 1635944496),
                            INT32_C(          0), INT32_C(          0), INT32_C(  537395272), INT32_C(          0)) },
    { UINT16_C(54171),
      simde_mm512_set_epi32(INT32_C(  986405639), INT32_C( 1121234438), INT32_C( -731393712), INT32_C( -784753228),
                            INT32_C( -748350470), INT32_C( 1755197901), INT32_C( 1090059253), INT32_C( 1329426651),
                            INT32_C(-1604442789), INT32_C(  878047098), INT32_C(-1682276633), INT32_C(  -78811559),
                            INT32_C( -973139496), INT32_C(   91517188), INT32_C(   37440120), INT32_C(  829401648)),
      simde_mm512_set_epi32(INT32_C( 2057630636), INT32_C(  335188274), INT32_C( 1272591061), INT32_C( -327494197),
                            INT32_C(  892006932), INT32_C(-1158010747), INT32_C( -700127305), INT32_C(-1058033333),
                            INT32_C( 2135825335), INT32_C( -510043422), INT32_C( -139544800), INT32_C(   95748631),
                            INT32_C( 1671238992), INT32_C( 1305915968), INT32_C( -582790199), INT32_C( 1758236157)),
      simde_mm512_set_epi32(INT32_C(  981488388), INT32_C(   47218690), INT32_C(          0), INT32_C(-1070033536),
                            INT32_C(          0), INT32_C(          0), INT32_C( 1077994421), INT32_C( 1076703307),
                            INT32_C(  541982995), INT32_C(          0), INT32_C(          0), INT32_C(   17105425),
                            INT32_C( 1100812624), INT32_C(          0), INT32_C(     213576), INT32_C(  541886512)) },
    { UINT16_C( 6763),
      simde_mm512_set_epi32(INT32_C( -697747292), INT32_C(-1779695782), INT32_C(  491327584), INT32_C( -563681080),
                            INT32_C( 1527319596), INT32_C( 1347900829), INT32_C(-1792354715), INT32_C( -371177698),
                            INT32_C(  255088013), INT32_C( 1961231505), INT32_C( -659343095), INT32_C( 1620234692),
                            INT32_C(  843561067), INT32_C( 1265300992), INT32_C(-1675104490), INT32_C( -873664156)),
      simde_mm512_set_epi32(INT32_C( -822209230), INT32_C(-1549704264), INT32_C(-1157339218), INT32_C(-2025137124),
                            INT32_C(-1631723043), INT32_C(-2095740678), INT32_C(-1857753563), INT32_C(-1236342636),
                            INT32_C( 1439297909), INT32_C( -576201057), INT32_C( 1488873085), INT32_C(-1369304746),
                            INT32_C(  567848046), INT32_C( 1335236564), INT32_C( -942680632), INT32_C(-1512916560)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(-2042443768),
                            INT32_C(  436797452), INT32_C(          0), INT32_C(-1862220251), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1420166289), INT32_C( 1488068617), INT32_C(          0),
                            INT32_C(  541108330), INT32_C(          0), INT32_C(-2079865600), INT32_C(-2118078176)) },
    { UINT16_C(17574),
      simde_mm512_set_epi32(INT32_C(-2098508971), INT32_C(  488824783), INT32_C( 1647236603), INT32_C(-1221154150),
                            INT32_C(-1012750616), INT32_C( 1849794587), INT32_C(  131205926), INT32_C( 2061955170),
                            INT32_C(   37003574), INT32_C( 1649229141), INT32_C(  612060260), INT32_C(-1402263233),
                            INT32_C( -513572270), INT32_C(  701923816), INT32_C(  511549547), INT32_C(  969083331)),
      simde_mm512_set_epi32(INT32_C( -873396634), INT32_C(-1835948135), INT32_C(-1570208244), INT32_C( -244400530),
                            INT32_C( 1126824505), INT32_C( 1036340167), INT32_C(  424825857), INT32_C(  434716327),
                            INT32_C( 1663095683), INT32_C( 1730428966), INT32_C( -252943126), INT32_C(  373990324),
                            INT32_C( 2100741912), INT32_C(-1005385937), INT32_C(-1470990839), INT32_C(-1200692008)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  268470153), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(  742457347), INT32_C(          0), INT32_C(          0),
                            INT32_C(   35684610), INT32_C(          0), INT32_C(  543704160), INT32_C(          0),
                            INT32_C(          0), INT32_C(    1179944), INT32_C(  139468809), INT32_C(          0)) },
    { UINT16_C( 5226),
      simde_mm512_set_epi32(INT32_C(  793357988), INT32_C(  607421194), INT32_C(-1375787856), INT32_C( 1799684145),
                            INT32_C( -551549754), INT32_C( 1249951615), INT32_C( 1447553529), INT32_C(-2080136983),
                            INT32_C( -623547588), INT32_C( 1015056564), INT32_C(-1333355305), INT32_C( 1925062912),
                            INT32_C(  330138155), INT32_C( -444195598), INT32_C(  874806560), INT32_C(  319126943)),
      simde_mm512_set_epi32(INT32_C( -542891463), INT32_C(  629803756), INT32_C(-1216921331), INT32_C( -301860714),
                            INT32_C(  317296385), INT32_C( 1833800187), INT32_C( -645353377), INT32_C(-1989715102),
                            INT32_C( 1785805415), INT32_C( 1739507553), INT32_C(-1932540399), INT32_C( -641363523),
                            INT32_C(   72499486), INT32_C(-2005610208), INT32_C( -497730582), INT32_C(  312611747)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C( 1778450448),
                            INT32_C(          0), INT32_C( 1207996795), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(  612401184), INT32_C(-2138668015), INT32_C(          0),
                            INT32_C(         10), INT32_C(          0), INT32_C(  537147680), INT32_C(          0)) },
    { UINT16_C(43362),
      simde_mm512_set_epi32(INT32_C(-2123521230), INT32_C( -883174176), INT32_C( -332498316), INT32_C(  142889340),
                            INT32_C( 1231002435), INT32_C( 1681334055), INT32_C(  138372594), INT32_C( -878588971),
                            INT32_C(  423295425), INT32_C( -460615607), INT32_C( 1208771148), INT32_C(-2128303155),
                            INT32_C( -738338972), INT32_C( 2110676823), INT32_C(-1405320678), INT32_C(-2007459833)),
      simde_mm512_set_epi32(INT32_C( -941377596), INT32_C(-1151664921), INT32_C( -656770411), INT32_C(  861635987),
                            INT32_C( 1823839521), INT32_C( 1391000031), INT32_C(   73229946), INT32_C(  -53693878),
                            INT32_C( -257851255), INT32_C( -479522767), INT32_C(  745275629), INT32_C(-1783480446),
                            INT32_C(-1431666964), INT32_C( -422291816), INT32_C( -309252994), INT32_C(-2112946871)),
      simde_mm512_set_epi32(INT32_C(-2124307712), INT32_C(          0), INT32_C( -938837484), INT32_C(          0),
                            INT32_C( 1209372929), INT32_C(          0), INT32_C(          0), INT32_C( -931102144),
                            INT32_C(          0), INT32_C( -536145919), INT32_C(  135004236), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(-1408237542), INT32_C(          0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_and_epi32(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_and_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C( 9036212757557430666), INT64_C( 5704081348870821627),
                            INT64_C( 4916018483251427189), INT64_C( 5923374839641151982),
                            INT64_C(-4951532382404389825), INT64_C(-6984585675708986984),
                            INT64_C( 4762161048923702209), INT64_C( 6694028748503799090)),
      simde_mm512_set_epi64(INT64_C(-3453126114950276629), INT64_C(-4371782057203512120),
                            INT64_C( 5078173500331491597), INT64_C( 7338502100533278569),
                            INT64_C(  615944699146983248), INT64_C(-4396767702011741181),
                            INT64_C( 4200939521265073333), INT64_C(-3780095828076561881)),
      simde_mm512_set_epi64(INT64_C( 5765733423092074890), INT64_C( 4827937147730281672),
                            INT64_C( 4915961295754526981), INT64_C( 4617324316270006632),
                            INT64_C(  578719150276367376), INT64_C(-9074330961792628736),
                            INT64_C(  145408472056594561), INT64_C( 5224288327930368034)) },
    { simde_mm512_set_epi64(INT64_C( 5288299052354798314), INT64_C(-7277780933457101890),
                            INT64_C( 7313132024917137274), INT64_C( 2628232513216498408),
                            INT64_C(-7258160557155099776), INT64_C(  680094650638517474),
                            INT64_C( 5283899056600645648), INT64_C( 2807728906149754133)),
      simde_mm512_set_epi64(INT64_C(-6895960825351935237), INT64_C(  -43666961203633751),
                            INT64_C(-6142317585422402205), INT64_C( 3387090051947463484),
                            INT64_C(-2636564591268235579), INT64_C( -833681846664639227),
                            INT64_C(-5782890152253473541), INT64_C(-2717644085040284508)),
      simde_mm512_set_epi64(INT64_C(   18155308360663786), INT64_C(-7277816996756381272),
                            INT64_C( 2323877817609098594), INT64_C( 2594451620184359464),
                            INT64_C(-7259520672634482048), INT64_C(   27068065029080064),
                            INT64_C(  654161048765464592), INT64_C(  162142851124327428)) },
    { simde_mm512_set_epi64(INT64_C(-8112615037926038324), INT64_C(-2879330391107916497),
                            INT64_C(  498324993693604100), INT64_C(  989364461044945673),
                            INT64_C( 8467500887117366281), INT64_C(-6097708555739527863),
                            INT64_C(  789034506143718770), INT64_C(-5527357514795184575)),
      simde_mm512_set_epi64(INT64_C( 6941879334003428428), INT64_C( -661071113376870714),
                            INT64_C(-5630949274148604213), INT64_C( 7592192837230447644),
                            INT64_C(-1596547832800395209), INT64_C(-5917083848631867188),
                            INT64_C(-3882429228383046850), INT64_C(  747121403871646944)),
      simde_mm512_set_epi64(INT64_C(   18579654875284556), INT64_C(-3458201275382494202),
                            INT64_C(   56929894688378880), INT64_C(  655521307720597512),
                            INT64_C( 7026328109045710849), INT64_C(-6241986883974856632),
                            INT64_C(  725666832763081010), INT64_C(  165015392292176960)) },
    { simde_mm512_set_epi64(INT64_C(-3214140792841031219), INT64_C( 4681768843666616539),
                            INT64_C(-6891029306179981446), INT64_C(-7225323117343838631),
                            INT64_C(-4179602309674405628), INT64_C(  160804091787717168),
                            INT64_C( 3694873207195836071), INT64_C(-1555264558302423685)),
      simde_mm512_set_epi64(INT64_C( 3831140603882252421), INT64_C(-3007023874774683317),
                            INT64_C( 9173299967578168034), INT64_C( -599340352231112169),
                            INT64_C( 7177916815745921600), INT64_C(-2503064843376095747),
                            INT64_C( 4236579961216216582), INT64_C(-3141312070029828684)),
      simde_mm512_set_epi64(INT64_C( 1234006158509482117), INT64_C( 4629950784542158923),
                            INT64_C( 2327799239051135586), INT64_C(-7806367493009374703),
                            INT64_C( 4727954219193348096), INT64_C(     917302477097008),
                            INT64_C( 3621548322709573126), INT64_C(-4583941969312410320)) },
    { simde_mm512_set_epi64(INT64_C( 5789189981308900965), INT64_C(-1594196073659476595),
                            INT64_C( 8423425177495484681), INT64_C( 6958855014828193899),
                            INT64_C( 5434426382856220438), INT64_C(-3752358975457584533),
                            INT64_C( 8837456289202868530), INT64_C( 5465736992144414155)),
      simde_mm512_set_epi64(INT64_C(-9001137670469652955), INT64_C(-5310051186831134347),
                            INT64_C(-2474764694246758787), INT64_C(-5881119101759738770),
                            INT64_C( 5734797378155697608), INT64_C(-6497927143179601756),
                            INT64_C(-7643735180027817888), INT64_C(-2420991802446640084)),
      simde_mm512_set_epi64(INT64_C(    5911629778699813), INT64_C(-6899444174404169467),
                            INT64_C( 6099567767624753161), INT64_C( 2306424943463671914),
                            INT64_C( 5404906694268929280), INT64_C(-9097076494042654688),
                            INT64_C( 1198224716855775264), INT64_C( 5350875317483865096)) },
    { simde_mm512_set_epi64(INT64_C(  563525163273351266), INT64_C(  158929141814345045),
                            INT64_C( 2628778802773961023), INT64_C(-2205776103080558104),
                            INT64_C( 2197088575617698243), INT64_C( 5860665973320586546),
                            INT64_C(-6655929129214122066), INT64_C(-8697897714832252451)),
      simde_mm512_set_epi64(INT64_C( 1824613162744888999), INT64_C( 7142941570334212134),
                            INT64_C(-1086382453544016972), INT64_C( 9022617812666091311),
                            INT64_C(-6317857543126326056), INT64_C(-9013027400318787633),
                            INT64_C( 7074827341732948634), INT64_C(-4349730772874059749)),
      simde_mm512_set_epi64(INT64_C(   95138542545547298), INT64_C(  153264234564943876),
                            INT64_C( 2335191585971118388), INT64_C( 6999299776119439656),
                            INT64_C(  599013974410920128), INT64_C(   18366259613799682),
                            INT64_C( 2459037658077471370), INT64_C(-9006491134657978343)) },
    { simde_mm512_set_epi64(INT64_C(-8934120309513688260), INT64_C( 4359634748931742935),
                            INT64_C( 8268082250112664107), INT64_C(-1907805565562356448),
                            INT64_C( 1370639783932859498), INT64_C(-3751209977007462503),
                            INT64_C(-6743993051839021458), INT64_C( 4839674398342728647)),
      simde_mm512_set_epi64(INT64_C(-8545761289661498777), INT64_C( 7471128053642413585),
                            INT64_C(-2754635356059844322), INT64_C(-8614030248086520854),
                            INT64_C( 1342657230503784100), INT64_C( 2608854166046450864),
                            INT64_C( 7729584549648139462), INT64_C( 5368501309454936569)),
      simde_mm512_set_epi64(INT64_C(-9222527198360950236), INT64_C( 2630243059467977745),
                            INT64_C( 5801763319471669258), INT64_C(-9222228406785787616),
                            INT64_C( 1297055947226612768), INT64_C(   13513008652297360),
                            INT64_C( 2467972599307568198), INT64_C( 4756005960819231169)) },
    { simde_mm512_set_epi64(INT64_C( 1818040010755772489), INT64_C( 5191632551175039949),
                            INT64_C(-3171141735991582889), INT64_C(-6035806350115039225),
                            INT64_C( 4482393771264451129), INT64_C( 2704986537996009741),
                            INT64_C(-1296481894259912959), INT64_C( 7876111834213298271)),
      simde_mm512_set_epi64(INT64_C(-1107462703642111951), INT64_C( 3200934455572316034),
                            INT64_C(-6148962785270933864), INT64_C(-1328231493238063799),
                            INT64_C(-9120454231799900960), INT64_C(-1428069393052184196),
                            INT64_C( 5287115201302699815), INT64_C(  594305769309064149)),
      simde_mm512_set_epi64(INT64_C( 1162069999159545857), INT64_C(  579838780607835008),
                            INT64_C(-9031874063504222704), INT64_C(-6048334185741379583),
                            INT64_C(   10310269968318496), INT64_C( 2596899559745798412),
                            INT64_C( 5188595380067045121), INT64_C(  580122069275017301)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_and_epi64(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_and_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C( 9036212757557430666), INT64_C( 5704081348870821627),
                            INT64_C( 4916018483251427189), INT64_C( 5923374839641151982),
                            INT64_C(-4951532382404389825), INT64_C(-6984585675708986984),
                            INT64_C( 4762161048923702209), INT64_C( 6694028748503799090)),
      UINT8_C( 39),
      simde_mm512_set_epi64(INT64_C( 7829214476264278040), INT64_C(  703575204673376097),
                            INT64_C(-5407331354528887836), INT64_C(-6120531474030028601),
                            INT64_C(-8809472528091429264), INT64_C(-4471837688110281967),
                            INT64_C(-1031891648804766660), INT64_C(-5771507277206231329)),
      simde_mm512_set_epi64(INT64_C( 8447482571111256056), INT64_C( 3577127948173451468),
                            INT64_C(-2004739832689101100), INT64_C( 8986958869783206795),
                            INT64_C(-5087155628682128019), INT64_C(-4898549151711613317),
                            INT64_C( 4680300913939983922), INT64_C( 6463791434556706448)),
      simde_mm512_set_epi64(INT64_C( 9036212757557430666), INT64_C( 5704081348870821627),
                            INT64_C(-6618874695993703740), INT64_C( 5923374839641151982),
                            INT64_C(-4951532382404389825), INT64_C(-9223140039893972463),
                            INT64_C( 4657219002685467696), INT64_C(  694680311283451536)) },
    { simde_mm512_set_epi64(INT64_C( 2309236383959261226), INT64_C( 7774175012325743905),
                            INT64_C( 5839305773204773555), INT64_C( 1099790230208207988),
                            INT64_C(-1687016713001301383), INT64_C( 3320690892837104327),
                            INT64_C(  710668265524759563), INT64_C(-6065296483252896486)),
      UINT8_C(160),
      simde_mm512_set_epi64(INT64_C(  747121403871646944), INT64_C(-8112615037926038324),
                            INT64_C(-2879330391107916497), INT64_C(  498324993693604100),
                            INT64_C(  989364461044945673), INT64_C( 8467500887117366281),
                            INT64_C(-6097708555739527863), INT64_C(  789034506143718770)),
      simde_mm512_set_epi64(INT64_C(-1555264558302423685), INT64_C( 6941879334003428428),
                            INT64_C( -661071113376870714), INT64_C(-5630949274148604213),
                            INT64_C( 7592192837230447644), INT64_C(-1596547832800395209),
                            INT64_C(-5917083848631867188), INT64_C(-3882429228383046850)),
      simde_mm512_set_epi64(INT64_C(  741411777270518880), INT64_C( 7774175012325743905),
                            INT64_C(-3458201275382494202), INT64_C( 1099790230208207988),
                            INT64_C(-1687016713001301383), INT64_C( 3320690892837104327),
                            INT64_C(  710668265524759563), INT64_C(-6065296483252896486)) },
    { simde_mm512_set_epi64(INT64_C(-3141312070029828684), INT64_C(-3214140792841031219),
                            INT64_C( 4681768843666616539), INT64_C(-6891029306179981446),
                            INT64_C(-7225323117343838631), INT64_C(-4179602309674405628),
                            INT64_C(  160804091787717168), INT64_C( 3694873207195836071)),
      UINT8_C(  6),
      simde_mm512_set_epi64(INT64_C( 1439622676105278165), INT64_C(-1406576864852774380),
                            INT64_C(-4973618283186690121), INT64_C(-4544218561177052233),
                            INT64_C(-2190619812874504416), INT64_C(  411237240453010768),
                            INT64_C( 5608866377596359625), INT64_C( 7551566793946127111)),
      simde_mm512_set_epi64(INT64_C( 2110235908633979080), INT64_C( 6559787716707833245),
                            INT64_C(-7698104879832611042), INT64_C( 1095594675397854353),
                            INT64_C(-2831857028248186428), INT64_C( 3623067196209165824),
                            INT64_C(-7194518998511455900), INT64_C(-8783679074311277652)),
      simde_mm512_set_epi64(INT64_C(-3141312070029828684), INT64_C(-3214140792841031219),
                            INT64_C( 4681768843666616539), INT64_C(-6891029306179981446),
                            INT64_C(-7225323117343838631), INT64_C(    1407388893058048),
                            INT64_C(  866558102575534400), INT64_C( 3694873207195836071)) },
    { simde_mm512_set_epi64(INT64_C(-4970734089418384356), INT64_C(-7008197103615375110),
                            INT64_C(-7978990794053850988), INT64_C( 6181737452074950303),
                            INT64_C( 6394661210895290710), INT64_C( 2438888788002740180),
                            INT64_C(-4048782482230560336), INT64_C(-2996801797497290918)),
      UINT8_C(184),
      simde_mm512_set_epi64(INT64_C( 7074827341732948634), INT64_C(-4349730772874059749),
                            INT64_C(  563525163273351266), INT64_C(  158929141814345045),
                            INT64_C( 2628778802773961023), INT64_C(-2205776103080558104),
                            INT64_C( 2197088575617698243), INT64_C( 5860665973320586546)),
      simde_mm512_set_epi64(INT64_C(-6743993051839021458), INT64_C( 4839674398342728647),
                            INT64_C( 1824613162744888999), INT64_C( 7142941570334212134),
                            INT64_C(-1086382453544016972), INT64_C( 9022617812666091311),
                            INT64_C(-6317857543126326056), INT64_C(-9013027400318787633)),
      simde_mm512_set_epi64(INT64_C( 2461362369224681994), INT64_C(-7008197103615375110),
                            INT64_C(   95138542545547298), INT64_C(  153264234564943876),
                            INT64_C( 2335191585971118388), INT64_C( 2438888788002740180),
                            INT64_C(-4048782482230560336), INT64_C(-2996801797497290918)) },
    { simde_mm512_set_epi64(INT64_C( 7729584549648139462), INT64_C( 5368501309454936569),
                            INT64_C(-8934120309513688260), INT64_C( 4359634748931742935),
                            INT64_C( 8268082250112664107), INT64_C(-1907805565562356448),
                            INT64_C( 1370639783932859498), INT64_C(-3751209977007462503)),
      UINT8_C(176),
      simde_mm512_set_epi64(INT64_C(-5226637314456684394), INT64_C( 1362777598547825147),
                            INT64_C(-2771771646272906398), INT64_C( 7669975856184215393),
                            INT64_C(-8300197808250187331), INT64_C(  311382923636166944),
                            INT64_C(-2137736571596434525), INT64_C( 3407446613087781642)),
      simde_mm512_set_epi64(INT64_C(  613705043478027075), INT64_C( 7221274780014437874),
                            INT64_C(-3773510896647996991), INT64_C(-1978328966883417524),
                            INT64_C(-9140992443141990556), INT64_C( 9065287930099827226),
                            INT64_C(-8621974329724982942), INT64_C(-2331701078232790292)),
      simde_mm512_set_epi64(INT64_C(    1197420910319618), INT64_C( 5368501309454936569),
                            INT64_C(-3926997772353251008), INT64_C( 6917828375051256384),
                            INT64_C( 8268082250112664107), INT64_C(-1907805565562356448),
                            INT64_C( 1370639783932859498), INT64_C(-3751209977007462503)) },
    { simde_mm512_set_epi64(INT64_C( 3700698387045520673), INT64_C( 5974299641953216122),
                            INT64_C( -230613445968297847), INT64_C(-2059534601207152403),
                            INT64_C(-7659990185762193684), INT64_C(-1813729535102735234),
                            INT64_C(-9075037706959084750), INT64_C(-3793204198629279116)),
      UINT8_C(149),
      simde_mm512_set_epi64(INT64_C(-1763977094002909713), INT64_C(-2903140489693440283),
                            INT64_C( 6916354853087283405), INT64_C( -288829486582280686),
                            INT64_C( 3770029145011694022), INT64_C( 4745990232274262704),
                            INT64_C(-9178901749894611532), INT64_C(-4043185984863798041)),
      simde_mm512_set_epi64(INT64_C(-2601025793624159885), INT64_C( 3563840185149175707),
                            INT64_C( -266768530495040309), INT64_C( 3956680363155515245),
                            INT64_C(-3793861402632321428), INT64_C(-9102379300509684954),
                            INT64_C(-8107510778505593825), INT64_C( 3515623724266758070)),
      simde_mm512_set_epi64(INT64_C(-4358068267243845277), INT64_C( 5974299641953216122),
                            INT64_C( -230613445968297847), INT64_C( 3668413965043549696),
                            INT64_C(-7659990185762193684), INT64_C(  111745843014421024),
                            INT64_C(-9075037706959084750), INT64_C(   54607264595188902)) },
    { simde_mm512_set_epi64(INT64_C( 6254124818226667045), INT64_C( 2295645667912807702),
                            INT64_C(-4018707571999816637), INT64_C(  -26221181248372278),
                            INT64_C(-1805361387364896744), INT64_C(-3741154382847501293),
                            INT64_C( 2395841610067775712), INT64_C( 4058242405589980421)),
      UINT8_C( 95),
      simde_mm512_set_epi64(INT64_C( 8131676417901457360), INT64_C( 5114845102355343256),
                            INT64_C(-4438171947452810779), INT64_C(-7935157497162192255),
                            INT64_C(  113046603373071999), INT64_C( 5604608652266856031),
                            INT64_C(-6729023456945537362), INT64_C(-3975720697196785172)),
      simde_mm512_set_epi64(INT64_C( 4986045839810005585), INT64_C(  594372055586112482),
                            INT64_C( 4543836970229042581), INT64_C(-3911934107467365366),
                            INT64_C(-4070664534473188278), INT64_C( 5469443597420811446),
                            INT64_C(-7904155369341108539), INT64_C(-3737549779697651723)),
      simde_mm512_set_epi64(INT64_C( 6254124818226667045), INT64_C(   16747779451847552),
                            INT64_C(-4018707571999816637), INT64_C(-9106278428011921408),
                            INT64_C(  108112814837713994), INT64_C( 5316237509498262550),
                            INT64_C(-9075733389403931004), INT64_C(-4034937163739561500)) },
    { simde_mm512_set_epi64(INT64_C(-4532025999716330471), INT64_C( 1441736923413036901),
                            INT64_C( 4055551302629479412), INT64_C(  212827843943377228),
                            INT64_C(-8849072478755780357), INT64_C(-1422057783446117205),
                            INT64_C(-7199062965036293603), INT64_C(  733947372098454344)),
      UINT8_C( 32),
      simde_mm512_set_epi64(INT64_C(-6215126846548299491), INT64_C(-2916974460339411728),
                            INT64_C( 7873834362860447881), INT64_C( 8398781451642376253),
                            INT64_C( 4819162425848909387), INT64_C( 3108138717093232544),
                            INT64_C(-1968231203112162098), INT64_C(-5881820928188872186)),
      simde_mm512_set_epi64(INT64_C(-5650512314146637209), INT64_C(    3824130547717357),
                            INT64_C( 8692530928452313027), INT64_C(    8285423730243427),
                            INT64_C( 3742589136248759843), INT64_C( 2734033046067673576),
                            INT64_C( 3884903702312804919), INT64_C(-1276802171409363646)),
      simde_mm512_set_epi64(INT64_C(-4532025999716330471), INT64_C( 1441736923413036901),
                            INT64_C( 7494010129634756737), INT64_C(  212827843943377228),
                            INT64_C(-8849072478755780357), INT64_C(-1422057783446117205),
                            INT64_C(-7199062965036293603), INT64_C(  733947372098454344)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_and_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_and_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C( 50),
      simde_mm512_set_epi64(INT64_C( 5084416814350799776), INT64_C(-8570269232503654461),
                            INT64_C(-1630207427926739773), INT64_C(-6092042681407238648),
                            INT64_C(-8913721049338044536), INT64_C( 6120766000801629269),
                            INT64_C( 8652973546099087422), INT64_C( 3768531854712764334)),
      simde_mm512_set_epi64(INT64_C( 7829214476264278040), INT64_C(  703575204673376097),
                            INT64_C(-5407331354528887836), INT64_C(-6120531474030028601),
                            INT64_C(-8809472528091429264), INT64_C(-4471837688110281967),
                            INT64_C(-1031891648804766660), INT64_C(-5771507277206231329)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(-6890431802948124480), INT64_C(-6123627733767157760),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 8072007440746715196), INT64_C(                   0)) },
    { UINT8_C(144),
      simde_mm512_set_epi64(INT64_C(-2717644085040284508), INT64_C( 5288299052354798314),
                            INT64_C(-7277780933457101890), INT64_C( 7313132024917137274),
                            INT64_C( 2628232513216498408), INT64_C(-7258160557155099776),
                            INT64_C(  680094650638517474), INT64_C( 5283899056600645648)),
      simde_mm512_set_epi64(INT64_C(-5527357514795184575), INT64_C(-6895960825351935237),
                            INT64_C(  -43666961203633751), INT64_C(-6142317585422402205),
                            INT64_C( 3387090051947463484), INT64_C(-2636564591268235579),
                            INT64_C( -833681846664639227), INT64_C(-5782890152253473541)),
      simde_mm512_set_epi64(INT64_C(-7905821755195257856), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 2323877817609098594),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(114),
      simde_mm512_set_epi64(INT64_C(-3413038857294295307), INT64_C( 6475143687707440027),
                            INT64_C(   66750031696924739), INT64_C(-4681864345113958415),
                            INT64_C(  562484877412986300), INT64_C( 5042574770374351665),
                            INT64_C(-5769247420573510048), INT64_C( 2830799652053530291)),
      simde_mm512_set_epi64(INT64_C(  937908328941131551), INT64_C(-2981077858701114759),
                            INT64_C(-9158548898769049773), INT64_C( 3163302403568753016),
                            INT64_C( 2171006373899068201), INT64_C(  676327318613388340),
                            INT64_C( 8575768723328224150), INT64_C( -569519755344554193)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C( 5800658722606940697),
                            INT64_C(   64181022643847235), INT64_C( 3100178613413757296),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 2811090599545930240), INT64_C(                   0)) },
    { UINT8_C(155),
      simde_mm512_set_epi64(INT64_C( 4236579961216216582), INT64_C(-3141312070029828684),
                            INT64_C(-3214140792841031219), INT64_C( 4681768843666616539),
                            INT64_C(-6891029306179981446), INT64_C(-7225323117343838631),
                            INT64_C(-4179602309674405628), INT64_C(  160804091787717168)),
      simde_mm512_set_epi64(INT64_C( 8837456289202868530), INT64_C( 5465736992144414155),
                            INT64_C( 3831140603882252421), INT64_C(-3007023874774683317),
                            INT64_C( 9173299967578168034), INT64_C( -599340352231112169),
                            INT64_C( 7177916815745921600), INT64_C(-2503064843376095747)),
      simde_mm512_set_epi64(INT64_C( 4215460527910977538), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 4629950784542158923),
                            INT64_C( 2327799239051135586), INT64_C(                   0),
                            INT64_C( 4727954219193348096), INT64_C(     917302477097008)) },
    { UINT8_C(107),
      simde_mm512_set_epi64(INT64_C(-2996801797497290918), INT64_C( 2110235908633979080),
                            INT64_C( 6559787716707833245), INT64_C(-7698104879832611042),
                            INT64_C( 1095594675397854353), INT64_C(-2831857028248186428),
                            INT64_C( 3623067196209165824), INT64_C(-7194518998511455900)),
      simde_mm512_set_epi64(INT64_C(-3531361750574079048), INT64_C(-4970734089418384356),
                            INT64_C(-7008197103615375110), INT64_C(-7978990794053850988),
                            INT64_C( 6181737452074950303), INT64_C( 6394661210895290710),
                            INT64_C( 2438888788002740180), INT64_C(-4048782482230560336)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C( 1801439990639675400),
                            INT64_C( 1876030771317506200), INT64_C(                   0),
                            INT64_C(  360377053526492305), INT64_C(                   0),
                            INT64_C( 2324042582201603584), INT64_C(-8932954729898528480)) },
    { UINT8_C(166),
      simde_mm512_set_epi64(INT64_C(-9013027400318787633), INT64_C( 7074827341732948634),
                            INT64_C(-4349730772874059749), INT64_C(  563525163273351266),
                            INT64_C(  158929141814345045), INT64_C( 2628778802773961023),
                            INT64_C(-2205776103080558104), INT64_C( 2197088575617698243)),
      simde_mm512_set_epi64(INT64_C(-3751209977007462503), INT64_C(-6743993051839021458),
                            INT64_C( 4839674398342728647), INT64_C( 1824613162744888999),
                            INT64_C( 7142941570334212134), INT64_C(-1086382453544016972),
                            INT64_C( 9022617812666091311), INT64_C(-6317857543126326056)),
      simde_mm512_set_epi64(INT64_C(-9015922487669520503), INT64_C(                   0),
                            INT64_C( 4837044293220761603), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 2335191585971118388),
                            INT64_C( 6999299776119439656), INT64_C(                   0)) },
    { UINT8_C(106),
      simde_mm512_set_epi64(INT64_C( 3407446613087781642), INT64_C(-5908963845954273231),
                            INT64_C(-2368888154296893569), INT64_C( 6217195068479217897),
                            INT64_C(-2678116496944625484), INT64_C(-5726717426998042368),
                            INT64_C( 1417932582737550578), INT64_C( 3757265565845388703)),
      simde_mm512_set_epi64(INT64_C(-2331701078232790292), INT64_C(-5226637314456684394),
                            INT64_C( 1362777598547825147), INT64_C(-2771771646272906398),
                            INT64_C( 7669975856184215393), INT64_C(-8300197808250187331),
                            INT64_C(  311382923636166944), INT64_C(-2137736571596434525)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-6523745533444423664),
                            INT64_C( 1306048291191951739), INT64_C(                   0),
                            INT64_C( 5355107766639820832), INT64_C(                   0),
                            INT64_C(         45097422880), INT64_C(                   0)) },
    { UINT8_C( 98),
      simde_mm512_set_epi64(INT64_C(-9120454231799900960), INT64_C(-1428069393052184196),
                            INT64_C( 5287115201302699815), INT64_C(  594305769309064149),
                            INT64_C( 1818040010755772489), INT64_C( 5191632551175039949),
                            INT64_C(-3171141735991582889), INT64_C(-6035806350115039225)),
      simde_mm512_set_epi64(INT64_C(-4043185984863798041), INT64_C(-2820807435363842669),
                            INT64_C( 7833331097238305247), INT64_C(  314520227399119434),
                            INT64_C(-1107462703642111951), INT64_C( 3200934455572316034),
                            INT64_C(-6148962785270933864), INT64_C(-1328231493238063799)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-4032276290038922992),
                            INT64_C( 5194217179798575367), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(-9031874063504222704), INT64_C(                   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_and_epi64(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_and_si512(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1762603276), INT32_C(-1316946536), INT32_C( -409636803), INT32_C(-1096492450),
                            INT32_C( 1487241173), INT32_C(-1940071138), INT32_C( 1116126146), INT32_C( -916337722),
                            INT32_C(   52488417), INT32_C( 1044081507), INT32_C(-1035184013), INT32_C(-1384518181),
                            INT32_C(-1687535487), INT32_C(-1003450758), INT32_C(  131212491), INT32_C(-1389162000)),
      simde_mm512_set_epi32(INT32_C(-1226890085), INT32_C(  294090078), INT32_C( 1993383303), INT32_C( 2117895966),
                            INT32_C( -579088274), INT32_C( 1336927221), INT32_C(   43647960), INT32_C( 1458493074),
                            INT32_C( -558283956), INT32_C(  428402022), INT32_C(  974860288), INT32_C( -624769004),
                            INT32_C(-1594949458), INT32_C(  -55819000), INT32_C(  423209449), INT32_C( -449605661)),
      simde_mm512_set_epi32(INT32_C(  537862152), INT32_C(  293631256), INT32_C( 1720717317), INT32_C( 1042579998),
                            INT32_C( 1478578244), INT32_C(  202162452), INT32_C(   42075072), INT32_C( 1080083074),
                            INT32_C(   35668032), INT32_C(  403202402), INT32_C(   34082816), INT32_C(-2009020400),
                            INT32_C(-2140536704), INT32_C(-1004534776), INT32_C(   17834185), INT32_C(-1523383840)) },
    { simde_mm512_set_epi32(INT32_C(-1226448376), INT32_C( 1927095485), INT32_C( 1303264402), INT32_C(  926300607),
                            INT32_C( 2116494628), INT32_C(-1100758331), INT32_C(  853434891), INT32_C( 1856581548),
                            INT32_C( 1730450085), INT32_C(-1233336711), INT32_C(  410802607), INT32_C(-1458345357),
                            INT32_C(-1414257532), INT32_C(-1908909941), INT32_C(  216004947), INT32_C(-1153616267)),
      simde_mm512_set_epi32(INT32_C(-1170743204), INT32_C( 1817377482), INT32_C(  934231558), INT32_C( -128119768),
                            INT32_C(  674694491), INT32_C( 1902405145), INT32_C(-1438313883), INT32_C( -508947384),
                            INT32_C( -334819615), INT32_C(-1499616800), INT32_C( -300643115), INT32_C(-1687918613),
                            INT32_C(  204696129), INT32_C( -790552335), INT32_C( -837803722), INT32_C(-1569322126)),
      simde_mm512_set_epi32(INT32_C(-1306140664), INT32_C( 1615863944), INT32_C(   95304706), INT32_C(  806619176),
                            INT32_C(  673645824), INT32_C(  811614721), INT32_C(  574882305), INT32_C( 1621623816),
                            INT32_C( 1677722273), INT32_C(-1508081568), INT32_C(  135530629), INT32_C(-1996463517),
                            INT32_C(  137371648), INT32_C(-2145386367), INT32_C(  202383634), INT32_C(-1573649808)) },
    { simde_mm512_set_epi32(INT32_C( -967914791), INT32_C( 1028004547), INT32_C( 1106145634), INT32_C(-2126623640),
                            INT32_C(   -6485699), INT32_C( -256904631), INT32_C(-1220204919), INT32_C(  917835787),
                            INT32_C( 1623427491), INT32_C( -222464855), INT32_C( -472879958), INT32_C( -762982604),
                            INT32_C(-1085405824), INT32_C( 1812599478), INT32_C( -613988136), INT32_C(  621554720)),
      simde_mm512_set_epi32(INT32_C( 1377438428), INT32_C( 1201451322), INT32_C(  619734582), INT32_C(-1992526637),
                            INT32_C( -868585296), INT32_C( 1439924174), INT32_C( 1656419868), INT32_C(-1390170089),
                            INT32_C( 1058692615), INT32_C( -836785520), INT32_C(-1830636694), INT32_C(  554982917),
                            INT32_C(-1514534388), INT32_C( -747673601), INT32_C( -692811287), INT32_C(-1799571639)),
      simde_mm512_set_epi32(INT32_C( 1107954392), INT32_C(   84152322), INT32_C(   14704674), INT32_C(-2126756800),
                            INT32_C( -870842320), INT32_C( 1350791752), INT32_C(  570435080), INT32_C(  606146563),
                            INT32_C(  537004035), INT32_C(-1038278528), INT32_C(-2101338070), INT32_C(     282116),
                            INT32_C(-1526071296), INT32_C( 1074397878), INT32_C( -769390392), INT32_C(   67904512)) },
    { simde_mm512_set_epi32(INT32_C(-1436950998), INT32_C(-1742059387), INT32_C( 1377677769), INT32_C(-2097193192),
                            INT32_C( 1556973207), INT32_C(   58040738), INT32_C(-1875805492), INT32_C( -452882923),
                            INT32_C(-2070651162), INT32_C(-1417594324), INT32_C( -990171302), INT32_C(  444234765),
                            INT32_C( -651701039), INT32_C( -296257488), INT32_C( 1302666953), INT32_C( 1243668562)),
      simde_mm512_set_epi32(INT32_C( -228023402), INT32_C( 1737651280), INT32_C(  890037909), INT32_C(  822465192),
                            INT32_C( 1525557148), INT32_C( 1672658803), INT32_C( 1808682106), INT32_C( 1316739447),
                            INT32_C(  903813947), INT32_C(  221590740), INT32_C( 1668581990), INT32_C(-1092503304),
                            INT32_C( 1369460064), INT32_C( 1353181098), INT32_C(  652356799), INT32_C( -684439573)),
      simde_mm512_set_epi32(INT32_C(-1572306430), INT32_C(     147456), INT32_C(  269262977), INT32_C(     348680),
                            INT32_C( 1489766036), INT32_C(   53518626), INT32_C(      18504), INT32_C( 1140951061),
                            INT32_C(   76814882), INT32_C(  151072772), INT32_C( 1081084482), INT32_C(  442513416),
                            INT32_C( 1361069120), INT32_C( 1074225184), INT32_C(   77597833), INT32_C( 1109409858)) },
    { simde_mm512_set_epi32(INT32_C(-1043054173), INT32_C( -396216896), INT32_C(-1145802326), INT32_C( -804000246),
                            INT32_C( -145399860), INT32_C( -890427310), INT32_C( -401401997), INT32_C(  802016776),
                            INT32_C( 1929893502), INT32_C(   73827769), INT32_C(-1971097644), INT32_C(-1831682098),
                            INT32_C(  546355465), INT32_C( -199725455), INT32_C(  931867413), INT32_C(-1496909535)),
      simde_mm512_set_epi32(INT32_C(-1796636811), INT32_C(-1576316556), INT32_C( 1080356179), INT32_C(-1830141457),
                            INT32_C(-1444813077), INT32_C( 1282909316), INT32_C(  814589845), INT32_C(  563073613),
                            INT32_C( -161574330), INT32_C( 1115054069), INT32_C(-1922096352), INT32_C( 1283172543),
                            INT32_C( 1028016376), INT32_C( 1652445236), INT32_C( 1602581177), INT32_C(-1986713581)),
      simde_mm512_set_epi32(INT32_C(-2134884063), INT32_C(-1610477248), INT32_C(    2384130), INT32_C(-1878900726),
                            INT32_C(-1589557560), INT32_C( 1214586880), INT32_C(  536938257), INT32_C(  562938376),
                            INT32_C( 1913033286), INT32_C(    6686129), INT32_C(-2013060352), INT32_C(    5417102),
                            INT32_C(  536872968), INT32_C( 1612202032), INT32_C(  394338833), INT32_C(-2138822655)) },
    { simde_mm512_set_epi32(INT32_C(  213329535), INT32_C( -522060385), INT32_C( -710729699), INT32_C(  911515198),
                            INT32_C(-1475915599), INT32_C(-1846311235), INT32_C(-1624654725), INT32_C( -496488954),
                            INT32_C(-2105881976), INT32_C( -863113580), INT32_C( -870973395), INT32_C(-2135017149),
                            INT32_C( 1179500895), INT32_C(  102238134), INT32_C( 1890546920), INT32_C( 1651955955)),
      simde_mm512_set_epi32(INT32_C(-1460720620), INT32_C(-1283988079), INT32_C( 2139823103), INT32_C(-2058406982),
                            INT32_C( -677653135), INT32_C(  526832430), INT32_C(  918576849), INT32_C(-1987609349),
                            INT32_C( -819905099), INT32_C( 2043707434), INT32_C( 1005516756), INT32_C(  646673888),
                            INT32_C( -792085599), INT32_C(  923333390), INT32_C(  549762390), INT32_C( 1063027034)),
      simde_mm512_set_epi32(INT32_C(  145171476), INT32_C(-1604196975), INT32_C( 1434648605), INT32_C(   71569978),
                            INT32_C(-2147266511), INT32_C(  291668524), INT32_C(  369106001), INT32_C(-2147472382),
                            INT32_C(-2111829888), INT32_C( 1216380928), INT32_C(  134610948), INT32_C(    9056576),
                            INT32_C( 1078558977), INT32_C(  101188358), INT32_C(  545535040), INT32_C(  575963218)) },
    { simde_mm512_set_epi32(INT32_C(-1614227898), INT32_C(-1072924213), INT32_C(-2048516742), INT32_C(-1735505047),
                            INT32_C(  409846045), INT32_C( -501166301), INT32_C(  385735082), INT32_C(-1379445210),
                            INT32_C( 1301699864), INT32_C( -237316746), INT32_C( -173549926), INT32_C(-1638681430),
                            INT32_C( 1204990643), INT32_C( -623938106), INT32_C(  621663116), INT32_C(-2139715294)),
      simde_mm512_set_epi32(INT32_C( 1168648208), INT32_C(  679514223), INT32_C(-1255159953), INT32_C(-2016174737),
                            INT32_C( -817087094), INT32_C( 1605116212), INT32_C(  684814447), INT32_C( 1274003485),
                            INT32_C( 1881744290), INT32_C(  579021373), INT32_C( -658206082), INT32_C( 1152351107),
                            INT32_C( -539739024), INT32_C( 1438387923), INT32_C( -569943597), INT32_C(  -79238784)),
      simde_mm512_set_epi32(INT32_C(   92798976), INT32_C(       4171), INT32_C(-2061105814), INT32_C(-2138961559),
                            INT32_C(  139198728), INT32_C( 1109393696), INT32_C(   13717546), INT32_C(  164052996),
                            INT32_C( 1073742080), INT32_C(  545390644), INT32_C( -796618214), INT32_C(   67317890),
                            INT32_C( 1204826160), INT32_C( 1351353538), INT32_C(   67457408), INT32_C(-2142861056)) },
    { simde_mm512_set_epi32(INT32_C( -593800358), INT32_C( -124181915), INT32_C( 2110561848), INT32_C( 1255401496),
                            INT32_C( -282522813), INT32_C( -286538666), INT32_C(-2011412362), INT32_C(-1839527164),
                            INT32_C(-1330408299), INT32_C( 1769934774), INT32_C( -358481155), INT32_C( -123958768),
                            INT32_C( 1676106379), INT32_C(-1305862521), INT32_C( 1797940107), INT32_C(  653525737)),
      simde_mm512_set_epi32(INT32_C(-1432835313), INT32_C( 1661538833), INT32_C( 1372337273), INT32_C(-1604084834),
                            INT32_C( -921184393), INT32_C(-1395990480), INT32_C( 1258870002), INT32_C( -947895097),
                            INT32_C(-1351881935), INT32_C(   99634026), INT32_C( 2033361976), INT32_C( 1231716550),
                            INT32_C( -228173591), INT32_C(-1552770129), INT32_C( -338049103), INT32_C(-1393391283)),
      simde_mm512_set_epi32(INT32_C(-2003303670), INT32_C( 1611203073), INT32_C( 1372329016), INT32_C(    4425752),
                            INT32_C( -922680509), INT32_C(-1395998704), INT32_C(  134758514), INT32_C(-2113929212),
                            INT32_C(-1608286191), INT32_C(   24120098), INT32_C( 1747058744), INT32_C( 1208516608),
                            INT32_C( 1650874505), INT32_C(-1574823289), INT32_C( 1795703681), INT32_C(  619708489)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_and_si512(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_andnot_si512(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( -335330897), INT32_C( 1860840666), INT32_C( -837102383), INT32_C( 1544121603),
                            INT32_C(  -31451516), INT32_C(  294501250), INT32_C( 1844141610), INT32_C(  711066163),
                            INT32_C( 1032767823), INT32_C(  466876164), INT32_C( 1432923079), INT32_C( -137339965),
                            INT32_C(-1216294439), INT32_C( 1110388055), INT32_C(  754234048), INT32_C( -712839683)),
      simde_mm512_set_epi32(INT32_C(  645969883), INT32_C(   45407696), INT32_C(-1431929981), INT32_C(-1744968675),
                            INT32_C( 1491740298), INT32_C( -669732847), INT32_C(-1678703719), INT32_C(-1110558488),
                            INT32_C(-1993251114), INT32_C( 1910816421), INT32_C( 2013403088), INT32_C(  882585036),
                            INT32_C( 1733706468), INT32_C( 1763057771), INT32_C(  303070795), INT32_C( -805966849)),
      simde_mm512_set_epi32(INT32_C(   41989712), INT32_C(    1364224), INT32_C(  547629826), INT32_C(-2081128420),
                            INT32_C(   13246474), INT32_C( -938475503), INT32_C(-1844444783), INT32_C(-1785972536),
                            INT32_C(-2144262512), INT32_C( 1612974241), INT32_C(  671224848), INT32_C(     730124),
                            INT32_C( 1079378468), INT32_C(  688914472), INT32_C(  302009355), INT32_C(  175440386)) },
    { simde_mm512_set_epi32(INT32_C( 1267266514), INT32_C(-1810114077), INT32_C(-1631053656), INT32_C(-1291259659),
                            INT32_C(-1797405973), INT32_C(-2052491824), INT32_C(  218690610), INT32_C(  434694077),
                            INT32_C(  322569513), INT32_C( -492306370), INT32_C( 1714124310), INT32_C(  757183592),
                            INT32_C( 1904845371), INT32_C( 1921390915), INT32_C( 1219016836), INT32_C( -491589854)),
      simde_mm512_set_epi32(INT32_C( -843887215), INT32_C(-1144045392), INT32_C( 1523671305), INT32_C( -687015924),
                            INT32_C( -651771268), INT32_C(-1812069901), INT32_C(  132880464), INT32_C( 1912329512),
                            INT32_C( -208209918), INT32_C(-1079631083), INT32_C( -134611197), INT32_C(-1062410635),
                            INT32_C( -896925558), INT32_C( -559765979), INT32_C( 1912148196), INT32_C( -437846049)),
      simde_mm512_set_epi32(INT32_C(-2077029375), INT32_C(  734265360), INT32_C( 1074880769), INT32_C( 1141114888),
                            INT32_C( 1226965012), INT32_C(  307626019), INT32_C(   48433216), INT32_C( 1611862016),
                            INT32_C( -528156670), INT32_C(  486941441), INT32_C(-1848604415), INT32_C(-1064549867),
                            INT32_C(-1979580288), INT32_C(-1944017372), INT32_C(  827392096), INT32_C(   88346845)) },
    { simde_mm512_set_epi32(INT32_C(  451034606), INT32_C(  160382101), INT32_C(-1268862602), INT32_C(  782115678),
                            INT32_C(-1160318793), INT32_C( -575355195), INT32_C( 1432838242), INT32_C(-2114154695),
                            INT32_C(-1020410376), INT32_C( -714076046), INT32_C(-1407849113), INT32_C(  996241684),
                            INT32_C(  481606881), INT32_C(-1834956523), INT32_C(  493396975), INT32_C(-1084672800)),
      simde_mm512_set_epi32(INT32_C( 1458493934), INT32_C( 1051105030), INT32_C( -836083742), INT32_C( 1407748874),
                            INT32_C(-1387312486), INT32_C(  776481471), INT32_C(  275093143), INT32_C( -137438390),
                            INT32_C( 1860284960), INT32_C(  540502552), INT32_C( 1411461258), INT32_C( 1517918194),
                            INT32_C( -266161178), INT32_C( 1269265702), INT32_C(  809771495), INT32_C(-1968711037)),
      simde_mm512_set_epi32(INT32_C( 1141686272), INT32_C(  908100354), INT32_C( 1243631232), INT32_C( 1365280768),
                            INT32_C(   84478472), INT32_C(  575154234), INT32_C(      38549), INT32_C( 1979863106),
                            INT32_C(  750793216), INT32_C(  537356808), INT32_C( 1344278664), INT32_C( 1075349218),
                            INT32_C( -536739066), INT32_C( 1225208866), INT32_C(  537133056), INT32_C(   10930691)) },
    { simde_mm512_set_epi32(INT32_C(-1562592645), INT32_C(  -32255724), INT32_C( -923416118), INT32_C(-2134713284),
                            INT32_C(-1313323965), INT32_C(-1729518909), INT32_C( 1286411285), INT32_C( -376910154),
                            INT32_C(-1786193108), INT32_C(-2035089818), INT32_C( 1552020826), INT32_C(  726998554),
                            INT32_C( 1864619074), INT32_C( 1828024315), INT32_C( -824341738), INT32_C(-1420030579)),
      simde_mm512_set_epi32(INT32_C( 1087836695), INT32_C(-2094233976), INT32_C( 1148487684), INT32_C(-1514127182),
                            INT32_C( -524459384), INT32_C(  725104708), INT32_C( 1787286694), INT32_C(-1533684832),
                            INT32_C(   46575098), INT32_C( 2086853653), INT32_C(  815292575), INT32_C(-1270435744),
                            INT32_C( 2014177347), INT32_C( 1099600134), INT32_C( -622983952), INT32_C(  822011154)),
      simde_mm512_set_epi32(INT32_C( 1073938436), INT32_C(   19662472), INT32_C(   67108868), INT32_C(  620757122),
                            INT32_C( 1074078344), INT32_C(  588257284), INT32_C(  570605730), INT32_C(   68489472),
                            INT32_C(   38151378), INT32_C( 2017460241), INT32_C(  538443909), INT32_C(-1811767200),
                            INT32_C(  269232129), INT32_C(   17469444), INT32_C(  268567776), INT32_C(  279109650)) },
    { simde_mm512_set_epi32(INT32_C(-1657115762), INT32_C( 1585840022), INT32_C(-1070898703), INT32_C( 1022031619),
                            INT32_C(-1380717315), INT32_C( 1086658406), INT32_C( -124039065), INT32_C(-1974944947),
                            INT32_C( 2044249149), INT32_C( 1638783653), INT32_C( 1466240446), INT32_C(-1803146403),
                            INT32_C( 1060682707), INT32_C(-1592428518), INT32_C(  156586666), INT32_C( -266957088)),
      simde_mm512_set_epi32(INT32_C( -703454581), INT32_C(  797686885), INT32_C( 1723425278), INT32_C( -158454369),
                            INT32_C(-1043830066), INT32_C(  709622512), INT32_C(-2136296570), INT32_C( -863350926),
                            INT32_C( 1844461284), INT32_C(  -21472306), INT32_C(-1932483198), INT32_C(-1320584016),
                            INT32_C( -370591173), INT32_C( -330170023), INT32_C( -975385097), INT32_C( -654562432)),
      simde_mm512_set_epi32(INT32_C( 1107296257), INT32_C(  554303585), INT32_C(  646971406), INT32_C(-1039923044),
                            INT32_C( 1078460930), INT32_C(  705357968), INT32_C(    2139008), INT32_C( 1149387826),
                            INT32_C(   69221056), INT32_C(-1643118262), INT32_C(-2003787776), INT32_C(  558453920),
                            INT32_C(-1061093336), INT32_C( 1279394113), INT32_C( -997683883), INT32_C(  149430528)) },
    { simde_mm512_set_epi32(INT32_C(  962558787), INT32_C(-1212292378), INT32_C(-1698562444), INT32_C(-1456708578),
                            INT32_C( 1605522258), INT32_C(-1389853810), INT32_C(  605095260), INT32_C(  449573803),
                            INT32_C(-1932095036), INT32_C( 1214045264), INT32_C(-1966228541), INT32_C(  484352026),
                            INT32_C(-1251622562), INT32_C(   97048183), INT32_C( 1801957969), INT32_C(   39148591)),
      simde_mm512_set_epi32(INT32_C( 1144673524), INT32_C(-1837539909), INT32_C(-1995926176), INT32_C( -775830454),
                            INT32_C( 1197039500), INT32_C(  605086417), INT32_C(-1681915928), INT32_C(-1694227594),
                            INT32_C(  250277648), INT32_C( 1517650405), INT32_C( -529860796), INT32_C(  319331129),
                            INT32_C( 1337610221), INT32_C( -515158609), INT32_C(-1958759875), INT32_C(  480005412)),
      simde_mm512_set_epi32(INT32_C( 1142949044), INT32_C(    4201753), INT32_C(   17301760), INT32_C( 1354858560),
                            INT32_C(    4787340), INT32_C(    1073233), INT32_C(-1683031392), INT32_C(-2130444204),
                            INT32_C(   36204048), INT32_C(  304152997), INT32_C( 1612858372), INT32_C(   50338593),
                            INT32_C( 1251610273), INT32_C( -536671864), INT32_C(-2145910740), INT32_C(  478675200)) },
    { simde_mm512_set_epi32(INT32_C(  477799556), INT32_C(  718106947), INT32_C( -702434720), INT32_C(  911156446),
                            INT32_C(  692922531), INT32_C( -634559193), INT32_C( -541024501), INT32_C(    6957260),
                            INT32_C(  891904501), INT32_C( 1674261328), INT32_C(  463285837), INT32_C(  465636281),
                            INT32_C( -567453998), INT32_C( -675807734), INT32_C( 1242869264), INT32_C(-2003535835)),
      simde_mm512_set_epi32(INT32_C( -440269466), INT32_C( 1069561863), INT32_C( -850138274), INT32_C( 1324108467),
                            INT32_C(  996083706), INT32_C(-1741332408), INT32_C(-1720688024), INT32_C( -195389802),
                            INT32_C( -122163269), INT32_C(-1678986062), INT32_C( -261742027), INT32_C(  147621305),
                            INT32_C( 1928957095), INT32_C(  647911914), INT32_C(-1231783784), INT32_C(-1597793099)),
      simde_mm512_set_epi32(INT32_C( -511704734), INT32_C(  352323588), INT32_C(  156387614), INT32_C( 1218464289),
                            INT32_C(  303171416), INT32_C(    1048648), INT32_C(    3170400), INT32_C( -200239598),
                            INT32_C( -929657334), INT32_C(-1742437214), INT32_C( -530448336), INT32_C(     820224),
                            INT32_C(  550537253), INT32_C(  537395680), INT32_C(-1266659192), INT32_C(  541295760)) },
    { simde_mm512_set_epi32(INT32_C(-1322452749), INT32_C(-1191485380), INT32_C(   61071601), INT32_C( -255981709),
                            INT32_C( 1745472557), INT32_C( 1521357726), INT32_C(-1111842070), INT32_C( 1783291089),
                            INT32_C(  718609371), INT32_C( -553071779), INT32_C(-1373014967), INT32_C(  751334079),
                            INT32_C( -828271800), INT32_C(-1578484948), INT32_C(-1597074675), INT32_C(  393018558)),
      simde_mm512_set_epi32(INT32_C(-1722624236), INT32_C( -955857282), INT32_C( 1790216473), INT32_C( -762838785),
                            INT32_C( -108799681), INT32_C( -975838651), INT32_C( 1961237228), INT32_C(   52752901),
                            INT32_C(-1440122977), INT32_C(-1167835972), INT32_C( 1345250484), INT32_C( 2101674065),
                            INT32_C( -149671798), INT32_C(  738167968), INT32_C( -764040824), INT32_C( -514982245)),
      simde_mm512_set_epi32(INT32_C(  139593476), INT32_C( 1191478850), INT32_C( 1746143496), INT32_C(   33554572),
                            INT32_C(-1853746926), INT32_C(-2058231743), INT32_C( 1078204420), INT32_C(   19145220),
                            INT32_C(-2144836604), INT32_C(  543437984), INT32_C( 1342603444), INT32_C( 1359282240),
                            INT32_C(  823402626), INT32_C(  169182336), INT32_C( 1378953344), INT32_C( -536804863)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_andnot_si512(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_andnot_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1786505147), INT32_C(  366806262), INT32_C(-1595474360), INT32_C( -741125130),
                            INT32_C(  623580589), INT32_C( 1819639708), INT32_C(-1998267151), INT32_C(   54696203),
                            INT32_C( 1230356730), INT32_C( -528215990), INT32_C(-1085976265), INT32_C(  -88891472),
                            INT32_C(  263402075), INT32_C( 2072408294), INT32_C( 1041613232), INT32_C(-1299496488)),
      simde_mm512_set_epi32(INT32_C(  634085978), INT32_C( 1880231468), INT32_C( 1457044755), INT32_C( -852481645),
                            INT32_C( 1274177455), INT32_C( -223885439), INT32_C(  582318482), INT32_C( 1515067636),
                            INT32_C(-1348943060), INT32_C(   88850487), INT32_C(-1067534176), INT32_C( 1770437803),
                            INT32_C(-1608407464), INT32_C(-1335624696), INT32_C( 1637100454), INT32_C(-1783030263)),
      simde_mm512_set_epi32(INT32_C(   92274752), INT32_C( 1610746376), INT32_C( 1444413715), INT32_C(  203433985),
                            INT32_C( 1255170562), INT32_C(-1836941311), INT32_C(  571547906), INT32_C( 1477181684),
                            INT32_C(-1501035772), INT32_C(   88850485), INT32_C( 1075488896), INT32_C(   17047563),
                            INT32_C(-1610578944), INT32_C(-2141090808), INT32_C( 1098918406), INT32_C(   87039489)) },
    { simde_mm512_set_epi32(INT32_C(  923576423), INT32_C(-1078925154), INT32_C( -430557576), INT32_C(-1684773193),
                            INT32_C(-1179101215), INT32_C(-1985153431), INT32_C(  584718967), INT32_C( -112765469),
                            INT32_C( 1515864234), INT32_C(-1118210252), INT32_C(  931438007), INT32_C( -352031421),
                            INT32_C( 1134370188), INT32_C( 1556623900), INT32_C(   57329867), INT32_C(  254759017)),
      simde_mm512_set_epi32(INT32_C( -300442460), INT32_C( -893266841), INT32_C(-1015236925), INT32_C(  262163323),
                            INT32_C( 2095940386), INT32_C( 1009617335), INT32_C(  458760718), INT32_C(-1732442867),
                            INT32_C( 1273945161), INT32_C(  946706622), INT32_C( 1469023509), INT32_C(-2064451876),
                            INT32_C( -256697390), INT32_C(-1934774398), INT32_C(  433298181), INT32_C( -530351918)),
      simde_mm512_set_epi32(INT32_C( -938273664), INT32_C( 1078006881), INT32_C(   19431555), INT32_C(   69208392),
                            INT32_C( 1145405442), INT32_C(  872483222), INT32_C(  419438600), INT32_C(   12058636),
                            INT32_C(   27565121), INT32_C(    2394250), INT32_C( 1074462720), INT32_C(   83006108),
                            INT32_C(-1339948462), INT32_C(-2145022590), INT32_C(  412160772), INT32_C( -532666222)) },
    { simde_mm512_set_epi32(INT32_C(  835311518), INT32_C(  593132209), INT32_C(-1205845883), INT32_C(-2103435972),
                            INT32_C(  331121937), INT32_C(-1122763027), INT32_C(  -11044623), INT32_C( 1217358106),
                            INT32_C(  899389553), INT32_C(   61750829), INT32_C(-1644418892), INT32_C( 1179256254),
                            INT32_C( -236468269), INT32_C( -666751062), INT32_C( -733547571), INT32_C( 2125570021)),
      simde_mm512_set_epi32(INT32_C(-1443754597), INT32_C( 1972174992), INT32_C(-2074962423), INT32_C( -531291976),
                            INT32_C( 1382830722), INT32_C( -282269267), INT32_C( 1453780297), INT32_C(  363272438),
                            INT32_C( 1819778130), INT32_C(-1488646809), INT32_C( 1000774887), INT32_C( 2075973242),
                            INT32_C(  251762527), INT32_C(  254090322), INT32_C( -106442053), INT32_C(-1147166459)),
      simde_mm512_set_epi32(INT32_C(-2009987071), INT32_C( 1418002432), INT32_C(   72521224), INT32_C( 1616191616),
                            INT32_C( 1077957250), INT32_C( 1110180096), INT32_C(   10519816), INT32_C(  354423012),
                            INT32_C( 1214514178), INT32_C(-1539243710), INT32_C(  570594371), INT32_C(  968151616),
                            INT32_C(  234885132), INT32_C(  119869520), INT32_C(  698417202), INT32_C(-2129780736)) },
    { simde_mm512_set_epi32(INT32_C( 1259282838), INT32_C( -167567006), INT32_C( 1470440257), INT32_C(-1702928569),
                            INT32_C(-1493129242), INT32_C( -361616020), INT32_C( 1148861436), INT32_C(-2140586026),
                            INT32_C(-1901343726), INT32_C( 1258604211), INT32_C( 1382183555), INT32_C(  464481172),
                            INT32_C(   87817013), INT32_C(  -25672201), INT32_C(-1647580547), INT32_C( -833959607)),
      simde_mm512_set_epi32(INT32_C( -711482206), INT32_C(-1110405208), INT32_C(  -55795162), INT32_C(-1789106875),
                            INT32_C(-1077987504), INT32_C( 2002242576), INT32_C(  879044440), INT32_C(  728498187),
                            INT32_C( -580810324), INT32_C(-1054241155), INT32_C(  416673383), INT32_C( 1924176623),
                            INT32_C( 1323235160), INT32_C(  659292758), INT32_C(-2101310960), INT32_C( 1303315999)),
      simde_mm512_set_epi32(INT32_C(-1802468320), INT32_C(  164662920), INT32_C(-1475837914), INT32_C(   83894272),
                            INT32_C(  415174672), INT32_C(  352698384), INT32_C(  805635072), INT32_C(  721682441),
                            INT32_C( 1363149228), INT32_C(-2144794548), INT32_C(  144018532), INT32_C( 1611698283),
                            INT32_C( 1254360136), INT32_C(   17039872), INT32_C(   33554432), INT32_C(   27592214)) },
    { simde_mm512_set_epi32(INT32_C( 1317706320), INT32_C( 1095937634), INT32_C(-2042379654), INT32_C( -425062813),
                            INT32_C(-1422676870), INT32_C(-1972727484), INT32_C( 1448617643), INT32_C( 1446030445),
                            INT32_C(-1203372071), INT32_C( 1257548767), INT32_C(   95515950), INT32_C(  288075556),
                            INT32_C( -562902724), INT32_C( 1866018725), INT32_C( -140491543), INT32_C( -853598261)),
      simde_mm512_set_epi32(INT32_C(-1862602245), INT32_C( 1299263323), INT32_C(-1100697239), INT32_C(-1165132701),
                            INT32_C(-1312528679), INT32_C(-2057483334), INT32_C(-2116201571), INT32_C(-1004874347),
                            INT32_C( -792865239), INT32_C(  167838662), INT32_C(-1158285246), INT32_C(  788705850),
                            INT32_C(-1470598876), INT32_C( -300747724), INT32_C( -732019428), INT32_C(-1060860437)),
      simde_mm512_set_epi32(INT32_C(-1871683157), INT32_C(  203489561), INT32_C(  941895937), INT32_C(  403009536),
                            INT32_C(  281297537), INT32_C(   85281466), INT32_C(-2121969388), INT32_C(-2146807408),
                            INT32_C( 1085800480), INT32_C(      65536), INT32_C(-1169913792), INT32_C(  771756058),
                            INT32_C(  537407488), INT32_C(-2147299312), INT32_C(    6160660), INT32_C(   12615712)) },
    { simde_mm512_set_epi32(INT32_C(  782435122), INT32_C( 1862046610), INT32_C( 2063073020), INT32_C(-2039040635),
                            INT32_C( 1210624813), INT32_C( 1482889596), INT32_C(-1693737823), INT32_C( -742414353),
                            INT32_C(  769657412), INT32_C(-1049696640), INT32_C(  237587070), INT32_C( 1546361918),
                            INT32_C( -364413489), INT32_C(-1858108224), INT32_C(-1524047519), INT32_C( -892082969)),
      simde_mm512_set_epi32(INT32_C( 1276319466), INT32_C( -348382036), INT32_C(  -54124638), INT32_C(-1613416797),
                            INT32_C( -277896350), INT32_C(-1555914365), INT32_C( 1602672291), INT32_C(  612591504),
                            INT32_C(-1670560036), INT32_C( 2118020891), INT32_C(-1204159467), INT32_C(  299945581),
                            INT32_C( 1470077526), INT32_C(-1901456818), INT32_C( 1982811443), INT32_C(  366998615)),
      simde_mm512_set_epi32(INT32_C( 1074795720), INT32_C(-2130703316), INT32_C(-2080374526), INT32_C(  427885090),
                            INT32_C(-1488519102), INT32_C(-1560239997), INT32_C( 1149518338), INT32_C(  603996176),
                            INT32_C(-1878178664), INT32_C( 1041238299), INT32_C(-1341082623), INT32_C(   29377089),
                            INT32_C(  362316304), INT32_C(  243274254), INT32_C( 1376193554), INT32_C(  353112080)) },
    { simde_mm512_set_epi32(INT32_C( -664438730), INT32_C( 1158162569), INT32_C(-1048438639), INT32_C(  819552403),
                            INT32_C(  486427093), INT32_C(-1267830843), INT32_C( 1178270581), INT32_C(-1348447676),
                            INT32_C( -981472284), INT32_C( 1962298807), INT32_C( -393093452), INT32_C(-1754911100),
                            INT32_C(-1506604227), INT32_C( -220324223), INT32_C(  856278899), INT32_C(   15706156)),
      simde_mm512_set_epi32(INT32_C( -689282393), INT32_C( -261985647), INT32_C(-1390325708), INT32_C(-1552766747),
                            INT32_C(-1576064212), INT32_C( -185898645), INT32_C(-1798232738), INT32_C( -401409831),
                            INT32_C( 1975803231), INT32_C( 1826250001), INT32_C(-1038398890), INT32_C( -306355124),
                            INT32_C(-1154269982), INT32_C( -209110535), INT32_C(-2033491342), INT32_C( -971905248)),
      simde_mm512_set_epi32(INT32_C(  109707905), INT32_C(-1335737840), INT32_C(  740376612), INT32_C(-2094888860),
                            INT32_C(-1576984024), INT32_C( 1082196010), INT32_C(-1866398710), INT32_C( 1074964633),
                            INT32_C(  813700123), INT32_C(  134746112), INT32_C(   34209858), INT32_C( 1754873928),
                            INT32_C(  419443906), INT32_C(   16851320), INT32_C(-2067652608), INT32_C( -972011776)) },
    { simde_mm512_set_epi32(INT32_C(-1519344071), INT32_C( 1556822852), INT32_C(-1382496853), INT32_C( -624683333),
                            INT32_C( 1477411394), INT32_C( -704833096), INT32_C(-1957423151), INT32_C( -471773069),
                            INT32_C( 1263493389), INT32_C( 2117955521), INT32_C(-1143959230), INT32_C( -832581030),
                            INT32_C(-1273834890), INT32_C( -392148704), INT32_C( 1764655366), INT32_C( -721713055)),
      simde_mm512_set_epi32(INT32_C(-1396008954), INT32_C( -651865449), INT32_C(  452267102), INT32_C( -741136221),
                            INT32_C( 1539744858), INT32_C(-2014766256), INT32_C(-1095604449), INT32_C(-1527666044),
                            INT32_C( -826073132), INT32_C(   -8340331), INT32_C( 1447376741), INT32_C( 1608478316),
                            INT32_C( 1253487795), INT32_C( 2056029052), INT32_C( -880457902), INT32_C( -691872315)),
      simde_mm512_set_epi32(INT32_C(  143267846), INT32_C(-2128330605), INT32_C(  308611156), INT32_C(   18032640),
                            INT32_C(   62923800), INT32_C(   33555008), INT32_C(  883056654), INT32_C(   68201092),
                            INT32_C(-2071978288), INT32_C(-2122280940), INT32_C( 1141188133), INT32_C(  293612580),
                            INT32_C( 1252271233), INT32_C(  302809692), INT32_C(-2105457072), INT32_C(   33576324)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_andnot_epi32(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_andnot_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(-1056724565), INT32_C( 1525326722), INT32_C( -860629095), INT32_C( 1674345138),
                            INT32_C( -780517906), INT32_C(-1953060088), INT32_C(-1307294727), INT32_C(-1463687440),
                            INT32_C( -675695615), INT32_C( 1308561010), INT32_C(  639253006), INT32_C( -651243687),
                            INT32_C( -612977662), INT32_C(  549809264), INT32_C(  644920842), INT32_C( 1882016760)),
      UINT16_C(49595),
      simde_mm512_set_epi32(INT32_C( 1189938329), INT32_C( 1797835672), INT32_C(  297527233), INT32_C( 1889709809),
                            INT32_C( -409509393), INT32_C(  994472936), INT32_C( -666015338), INT32_C( -260985600),
                            INT32_C(  750956055), INT32_C( 1716384261), INT32_C(-1960820967), INT32_C(  548004165),
                            INT32_C( 1158678479), INT32_C( 1692264694), INT32_C(  789910754), INT32_C(-1468927401)),
      simde_mm512_set_epi32(INT32_C( -428021677), INT32_C( 2072345032), INT32_C(-1760567882), INT32_C( -446864592),
                            INT32_C( 1299362117), INT32_C( 1402678741), INT32_C( -667918085), INT32_C( 1296019907),
                            INT32_C(-1260791809), INT32_C( 1231406525), INT32_C( -414651973), INT32_C( 1577314987),
                            INT32_C( 1453608195), INT32_C( 1199168765), INT32_C( 1764761558), INT32_C(-1053144882)),
      simde_mm512_set_epi32(INT32_C(-1609504702), INT32_C(  277164096), INT32_C( -860629095), INT32_C( 1674345138),
                            INT32_C( -780517906), INT32_C(-1953060088), INT32_C(-1307294727), INT32_C(  219025603),
                            INT32_C(-1877388824), INT32_C( 1308561010), INT32_C( 1682489506), INT32_C( 1577192106),
                            INT32_C(  312756224), INT32_C(  549809264), INT32_C( 1075841812), INT32_C( 1091176584)) },
    { simde_mm512_set_epi32(INT32_C( -922226792), INT32_C( 1177794317), INT32_C(-1987384202), INT32_C(  817250921),
                            INT32_C( 1296662639), INT32_C(   64131133), INT32_C(-1048693918), INT32_C( 1748498328),
                            INT32_C( -392119279), INT32_C(-1074948281), INT32_C( 1219088991), INT32_C(  346956559),
                            INT32_C( -778487174), INT32_C( 2030262893), INT32_C( -325938509), INT32_C( 2088865417)),
      UINT16_C(43842),
      simde_mm512_set_epi32(INT32_C( -911190750), INT32_C( -150954698), INT32_C(-2108244068), INT32_C( -219871492),
                            INT32_C(  954142226), INT32_C( -657696450), INT32_C(  -40171606), INT32_C(  523799369),
                            INT32_C(-1984820679), INT32_C( -352318109), INT32_C( 1527484465), INT32_C( 1078897849),
                            INT32_C( -979432773), INT32_C( -222789591), INT32_C( -127333602), INT32_C( 1547833861)),
      simde_mm512_set_epi32(INT32_C( 1706771302), INT32_C(-1876132949), INT32_C( -300867745), INT32_C(-1574226708),
                            INT32_C(  909541228), INT32_C(-1473521559), INT32_C(-2035272090), INT32_C( -843632177),
                            INT32_C(-1617888467), INT32_C( -960934829), INT32_C( -805571508), INT32_C( -811280081),
                            INT32_C(-1033748670), INT32_C(-1374688928), INT32_C( -924697051), INT32_C( -396703151)),
      simde_mm512_set_epi32(INT32_C(  604701252), INT32_C( 1177794317), INT32_C( 1812004931), INT32_C(  817250921),
                            INT32_C(  102789484), INT32_C(   64131133), INT32_C(   35663940), INT32_C(-1064882042),
                            INT32_C( -392119279), INT32_C(   79249424), INT32_C( 1219088991), INT32_C(  346956559),
                            INT32_C( -778487174), INT32_C( 2030262893), INT32_C(    8533025), INT32_C( 2088865417)) },
    { simde_mm512_set_epi32(INT32_C(-1406718947), INT32_C(  276558393), INT32_C(  154803470), INT32_C( 1010355861),
                            INT32_C( -906943422), INT32_C(-1458735792), INT32_C( -135902673), INT32_C( 2125322250),
                            INT32_C(  668612521), INT32_C( 2134097324), INT32_C( 1431164540), INT32_C(-1097880462),
                            INT32_C( 1895279922), INT32_C( -455917584), INT32_C(-1635623774), INT32_C( 1646110584)),
      UINT16_C(61721),
      simde_mm512_set_epi32(INT32_C(-1147100012), INT32_C( -529153170), INT32_C(-1710107397), INT32_C( 1085126684),
                            INT32_C( -365628842), INT32_C( 1126939173), INT32_C(-1962930746), INT32_C(-2032518388),
                            INT32_C( -893793955), INT32_C(-1793978656), INT32_C(  353794556), INT32_C(  484459160),
                            INT32_C( 1795576890), INT32_C(-1800969495), INT32_C(  570832120), INT32_C( -805110645)),
      simde_mm512_set_epi32(INT32_C(-1152323073), INT32_C(-1880366011), INT32_C( 1623795528), INT32_C(  779718762),
                            INT32_C( -950308445), INT32_C(  601329882), INT32_C( 1983067756), INT32_C( 1014514692),
                            INT32_C(  192697146), INT32_C( 1393627685), INT32_C( -618845734), INT32_C(-1526656596),
                            INT32_C( -668243521), INT32_C(  858775967), INT32_C( -874197170), INT32_C(-1013451033)),
      simde_mm512_set_epi32(INT32_C(    5263723), INT32_C(  260711425), INT32_C( 1623728896), INT32_C(  776994914),
                            INT32_C( -906943422), INT32_C(-1458735792), INT32_C( -135902673), INT32_C(  941638656),
                            INT32_C(  668612521), INT32_C( 2134097324), INT32_C( 1431164540), INT32_C(-1593765596),
                            INT32_C(-1876350587), INT32_C( -455917584), INT32_C(-1635623774), INT32_C(   60097124)) },
    { simde_mm512_set_epi32(INT32_C(   73765979), INT32_C( 1196192749), INT32_C( -212227718), INT32_C(-1980699203),
                            INT32_C(  -37222007), INT32_C(-1986328859), INT32_C( 1483201456), INT32_C(  129080387),
                            INT32_C( -259597220), INT32_C(-1814466623), INT32_C( 1536667113), INT32_C( 1702406736),
                            INT32_C( 1032855403), INT32_C( -907220805), INT32_C( -744099936), INT32_C( -484286001)),
      UINT16_C(60398),
      simde_mm512_set_epi32(INT32_C( 2131878120), INT32_C( -709717494), INT32_C(  677603870), INT32_C( 1110837767),
                            INT32_C(  137332416), INT32_C( 1049147481), INT32_C( -429123521), INT32_C(  562109282),
                            INT32_C( -475857832), INT32_C(-1750530864), INT32_C(-1098694184), INT32_C(-1278646805),
                            INT32_C(  274075622), INT32_C(  310096866), INT32_C( 1944249360), INT32_C(-1457965117)),
      simde_mm512_set_epi32(INT32_C(-1770120574), INT32_C(-1267999916), INT32_C(  920660290), INT32_C( 1218524275),
                            INT32_C( -813719782), INT32_C(   17574100), INT32_C( 1228269274), INT32_C( -540460196),
                            INT32_C( -544630186), INT32_C( -973323962), INT32_C( -900762472), INT32_C( 1800691074),
                            INT32_C( -934840396), INT32_C(-2024059127), INT32_C( 2050139755), INT32_C(-1648520849)),
      simde_mm512_set_epi32(INT32_C(-2140268030), INT32_C(  541673812), INT32_C(  377487680), INT32_C(-1980699203),
                            INT32_C( -951052006), INT32_C(-1986328859), INT32_C(  152168128), INT32_C( -565698532),
                            INT32_C(  470352390), INT32_C( 1079263494), INT32_C( 1078735872), INT32_C( 1702406736),
                            INT32_C( -939429872), INT32_C(-2063578103), INT32_C(  135266923), INT32_C( -484286001)) },
    { simde_mm512_set_epi32(INT32_C(  359551557), INT32_C(  851518101), INT32_C( 1700885885), INT32_C( 1144006274),
                            INT32_C(  718077661), INT32_C( 1054313754), INT32_C(   65647391), INT32_C(-1867262731),
                            INT32_C(  208941224), INT32_C(  989467762), INT32_C(-1763663368), INT32_C(  732190820),
                            INT32_C( -780985117), INT32_C(-1786203682), INT32_C( -893464048), INT32_C(-1930046056)),
      UINT16_C( 5280),
      simde_mm512_set_epi32(INT32_C( 2082802710), INT32_C(  398405458), INT32_C( -610997258), INT32_C(  830342728),
                            INT32_C( -327286830), INT32_C( 1285368273), INT32_C(-1636339073), INT32_C( 1467021210),
                            INT32_C( -637556884), INT32_C( 1464578281), INT32_C(  -78771124), INT32_C(-1194071193),
                            INT32_C(-1454776494), INT32_C(  224158188), INT32_C( 1578376173), INT32_C( 2022699384)),
      simde_mm512_set_epi32(INT32_C(-1580866758), INT32_C( 1705729088), INT32_C(-1204463345), INT32_C(  806420788),
                            INT32_C(-1410408996), INT32_C(  863225653), INT32_C(-2071560363), INT32_C( 1819484417),
                            INT32_C( -246595685), INT32_C(  243263522), INT32_C( 2052176477), INT32_C(  253176681),
                            INT32_C( 1676258794), INT32_C(-1129907739), INT32_C(  395133900), INT32_C(  -86934818)),
      simde_mm512_set_epi32(INT32_C(  359551557), INT32_C(  851518101), INT32_C( 1700885885), INT32_C(      65844),
                            INT32_C(  718077661), INT32_C(  862111268), INT32_C(   65647391), INT32_C(-1867262731),
                            INT32_C(  536877203), INT32_C(  989467762), INT32_C(    1159697), INT32_C(  732190820),
                            INT32_C( -780985117), INT32_C(-1786203682), INT32_C( -893464048), INT32_C(-1930046056)) },
    { simde_mm512_set_epi32(INT32_C( -763717484), INT32_C(-1454287993), INT32_C( -815713015), INT32_C( -381645662),
                            INT32_C( 1143121149), INT32_C(-2120634980), INT32_C( -259357121), INT32_C( -593579957),
                            INT32_C(-1529041977), INT32_C(-2065541499), INT32_C( 1009471119), INT32_C(  674532491),
                            INT32_C( -605291509), INT32_C( -802607554), INT32_C( -850350011), INT32_C(  732847081)),
      UINT16_C(41568),
      simde_mm512_set_epi32(INT32_C( 1295870302), INT32_C(  336570348), INT32_C(-1662536141), INT32_C(-1054381248),
                            INT32_C( 1593114303), INT32_C(-1017054773), INT32_C(-1409414000), INT32_C(  227338784),
                            INT32_C( 1117509139), INT32_C( 1937140770), INT32_C( 1843080524), INT32_C(  775622876),
                            INT32_C(  903821795), INT32_C(-1108923393), INT32_C( -348808591), INT32_C(  691553406)),
      simde_mm512_set_epi32(INT32_C( -957741997), INT32_C( -389978329), INT32_C(-1992364300), INT32_C(-1194120095),
                            INT32_C( 1460280679), INT32_C( -461012902), INT32_C(  191451119), INT32_C(  395863574),
                            INT32_C( 2007897293), INT32_C(  647995187), INT32_C( 1812181798), INT32_C(-1288356108),
                            INT32_C(-1946740515), INT32_C(-1688294491), INT32_C( -146679692), INT32_C( -960173252)),
      simde_mm512_set_epi32(INT32_C(-2101214207), INT32_C(-1454287993), INT32_C(   18368708), INT32_C( -381645662),
                            INT32_C( 1143121149), INT32_C(-2120634980), INT32_C(      82799), INT32_C( -593579957),
                            INT32_C(-1529041977), INT32_C(   76124945), INT32_C(      37410), INT32_C(  674532491),
                            INT32_C( -605291509), INT32_C( -802607554), INT32_C( -850350011), INT32_C(  732847081)) },
    { simde_mm512_set_epi32(INT32_C(-1543080560), INT32_C(  326946931), INT32_C(  691349892), INT32_C( 1226829378),
                            INT32_C( 1127061143), INT32_C( 1548237043), INT32_C(-1885371906), INT32_C(  673215002),
                            INT32_C(   -2545554), INT32_C(-1367277302), INT32_C( -227991301), INT32_C(  746457208),
                            INT32_C(-1737407854), INT32_C( 1988034150), INT32_C( -605858038), INT32_C( -752579769)),
      UINT16_C(24718),
      simde_mm512_set_epi32(INT32_C( 1517976828), INT32_C(  453076709), INT32_C( 1155311084), INT32_C(-1730593997),
                            INT32_C( 2009897302), INT32_C( -813354987), INT32_C( 1160389453), INT32_C(-1543844644),
                            INT32_C( -908777016), INT32_C(  107061968), INT32_C(-1889800585), INT32_C(-1309816398),
                            INT32_C( 1760607631), INT32_C(-1373730647), INT32_C( 1475928392), INT32_C(-1415204909)),
      simde_mm512_set_epi32(INT32_C(  901302066), INT32_C(  236605933), INT32_C( 1144123725), INT32_C(  765559000),
                            INT32_C( -272466037), INT32_C(  489940181), INT32_C( 1285546635), INT32_C(  894611583),
                            INT32_C(-1280504231), INT32_C( -511809158), INT32_C(  517714821), INT32_C( -458114298),
                            INT32_C(-1583011646), INT32_C( 2050708057), INT32_C(-1873361568), INT32_C( 1295393304)),
      simde_mm512_set_epi32(INT32_C(-1543080560), INT32_C(   68817160), INT32_C(    2183169), INT32_C( 1226829378),
                            INT32_C( 1127061143), INT32_C( 1548237043), INT32_C(-1885371906), INT32_C(  673215002),
                            INT32_C(  841483793), INT32_C(-1367277302), INT32_C( -227991301), INT32_C(  746457208),
                            INT32_C(-2130378688), INT32_C( 1344361040), INT32_C(-2147089376), INT32_C( -752579769)) },
    { simde_mm512_set_epi32(INT32_C( -203532895), INT32_C(-1671983312), INT32_C( -485765980), INT32_C(-1920770849),
                            INT32_C(  -87193791), INT32_C( 1659979037), INT32_C(-1337410362), INT32_C( 1209029675),
                            INT32_C(  587197109), INT32_C( -530755740), INT32_C(  281664792), INT32_C(  -47077792),
                            INT32_C( -945013045), INT32_C( -166692659), INT32_C( 1790118115), INT32_C(  689330771)),
      UINT16_C( 7519),
      simde_mm512_set_epi32(INT32_C( -384323470), INT32_C(  473195364), INT32_C(  206146438), INT32_C(-1217279332),
                            INT32_C(-1088463893), INT32_C(  970520784), INT32_C( -929499045), INT32_C(-1086034653),
                            INT32_C(-1051759609), INT32_C(-1753508816), INT32_C( 1464082608), INT32_C(  492133710),
                            INT32_C( 1610388137), INT32_C(-2026322187), INT32_C(-1721391979), INT32_C(  466414066)),
      simde_mm512_set_epi32(INT32_C( 1039275088), INT32_C( -195464931), INT32_C(-1467895249), INT32_C( 1829711637),
                            INT32_C( 2006708634), INT32_C(  837542220), INT32_C( -759309790), INT32_C( -498075629),
                            INT32_C(  922280800), INT32_C(  925077084), INT32_C( 1941328295), INT32_C(   27280850),
                            INT32_C( -499921640), INT32_C(  738410205), INT32_C(  972641353), INT32_C( 1011602801)),
      simde_mm512_set_epi32(INT32_C( -203532895), INT32_C(-1671983312), INT32_C( -485765980), INT32_C( 1208886529),
                            INT32_C( 1082171408), INT32_C(    2294028), INT32_C(-1337410362), INT32_C( 1074499600),
                            INT32_C(  587197109), INT32_C(  536873548), INT32_C(  281664792), INT32_C(   10485904),
                            INT32_C(-1610396400), INT32_C(  671299592), INT32_C(  546852936), INT32_C(  604181505)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_andnot_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_andnot_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(56303),
      simde_mm512_set_epi32(INT32_C(  684353163), INT32_C( -624296854), INT32_C(-1626870831), INT32_C( 1693659819),
                            INT32_C( 1814966119), INT32_C( 1428960968), INT32_C( 1709146671), INT32_C(-1269736679),
                            INT32_C( -399245267), INT32_C(  128121852), INT32_C(  623395494), INT32_C( 1817163956),
                            INT32_C( 1991308671), INT32_C( -978886098), INT32_C( 1436967950), INT32_C(  227176170)),
      simde_mm512_set_epi32(INT32_C( -155316348), INT32_C( 1821995326), INT32_C(-1956349521), INT32_C( 2078645861),
                            INT32_C(-2002962850), INT32_C( 1961273418), INT32_C( 1026886280), INT32_C( 1852456749),
                            INT32_C( 1549356853), INT32_C(  905982506), INT32_C( -562722910), INT32_C( 1231420121),
                            INT32_C(  786944005), INT32_C(-1682464667), INT32_C(   12357782), INT32_C(  913777965)),
      simde_mm512_set_epi32(INT32_C( -701232892), INT32_C(  605028628), INT32_C(          0), INT32_C(  453282884),
                            INT32_C(-2137976808), INT32_C(          0), INT32_C(  404752512), INT32_C( 1244275748),
                            INT32_C(  340348688), INT32_C(  805306370), INT32_C( -631929600), INT32_C(          0),
                            INT32_C(  138870784), INT32_C(  437289025), INT32_C(    1609872), INT32_C(  846528773)) },
    { UINT16_C(56200),
      simde_mm512_set_epi32(INT32_C( -452164103), INT32_C( 1890508390), INT32_C( 1258638805), INT32_C( -750109723),
                            INT32_C( -513503890), INT32_C( -379667747), INT32_C(-1651966538), INT32_C(  418163645),
                            INT32_C(-1484633406), INT32_C(  128570401), INT32_C(-1432905388), INT32_C(-1460529893),
                            INT32_C( -808466332), INT32_C(-1300168003), INT32_C(  153276923), INT32_C( -912847520)),
      simde_mm512_set_epi32(INT32_C( 1849401350), INT32_C(-2046167065), INT32_C(-1772087293), INT32_C(  763578781),
                            INT32_C(  -59556630), INT32_C( -574235850), INT32_C(-1931079616), INT32_C(  856557360),
                            INT32_C( 1798494574), INT32_C( -255236934), INT32_C( -498039931), INT32_C( 1916101155),
                            INT32_C( 1291737736), INT32_C(-1818740725), INT32_C( 1042711156), INT32_C(  770521823)),
      simde_mm512_set_epi32(INT32_C(  171122694), INT32_C(-2046746239), INT32_C(          0), INT32_C(  746668056),
                            INT32_C(  471019648), INT32_C(          0), INT32_C(    6684736), INT32_C(  587334656),
                            INT32_C( 1211142444), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(    3150472), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(29534),
      simde_mm512_set_epi32(INT32_C(-1569526022), INT32_C( -566961257), INT32_C( -454262297), INT32_C(-2011970966),
                            INT32_C( 1729229439), INT32_C(  515441803), INT32_C( 1629075756), INT32_C( -633945234),
                            INT32_C(-1517000454), INT32_C(-2129179491), INT32_C(-1082415130), INT32_C( -643068488),
                            INT32_C(-1177678851), INT32_C(  811665360), INT32_C(-1120986687), INT32_C( 1945770944)),
      simde_mm512_set_epi32(INT32_C( 1206445472), INT32_C( 1685117563), INT32_C( -105634979), INT32_C(  300875900),
                            INT32_C( 1292473590), INT32_C( -154568093), INT32_C( -725481309), INT32_C( 1537059805),
                            INT32_C(-1299234249), INT32_C( 1342055246), INT32_C( 1121196977), INT32_C( -936323200),
                            INT32_C(  284920534), INT32_C( -501374627), INT32_C(  523356394), INT32_C( 2082914622)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  541065320), INT32_C(  420487704), INT32_C(  300679188),
                            INT32_C(          0), INT32_C(          0), INT32_C(-1799225213), INT32_C(   25766033),
                            INT32_C(          0), INT32_C( 1323836226), INT32_C(          0), INT32_C(    1069568),
                            INT32_C(    3244034), INT32_C(-1038311411), INT32_C(   34652202), INT32_C(          0)) },
    { UINT16_C(18467),
      simde_mm512_set_epi32(INT32_C(-1126901666), INT32_C(-1540993522), INT32_C( -310394649), INT32_C( 1569425965),
                            INT32_C( 1860055197), INT32_C( 1022884520), INT32_C(  886587779), INT32_C(   -7751100),
                            INT32_C(  725782952), INT32_C( 1524528742), INT32_C(-1901622691), INT32_C( -205155472),
                            INT32_C( 1297212229), INT32_C(-1562315637), INT32_C(-1561800150), INT32_C( 1969817622)),
      simde_mm512_set_epi32(INT32_C( 1691822441), INT32_C( -747576101), INT32_C(  526461787), INT32_C(-1551035253),
                            INT32_C( -494445545), INT32_C(  601243904), INT32_C( 1621282220), INT32_C(   87983768),
                            INT32_C( 1749180883), INT32_C(  653596692), INT32_C( 1933605299), INT32_C( 2110990238),
                            INT32_C( 1287872496), INT32_C( -947101027), INT32_C(-1469323630), INT32_C( -103698146)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C( 1397792977), INT32_C(          0), INT32_C(          0),
                            INT32_C(-2147398654), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( 1900048802), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(  134423696), INT32_C(-2003783416)) },
    { UINT16_C(10499),
      simde_mm512_set_epi32(INT32_C(-1800578563), INT32_C( 1189147870), INT32_C( -918534753), INT32_C(-2046784432),
                            INT32_C( 2146267513), INT32_C( 1185116678), INT32_C(  743422455), INT32_C( -958735431),
                            INT32_C(-1272492795), INT32_C(-1993475811), INT32_C( -901911405), INT32_C( -444376352),
                            INT32_C( 1645484254), INT32_C( 1890851846), INT32_C(  632187417), INT32_C( 2142729898)),
      simde_mm512_set_epi32(INT32_C( -752859034), INT32_C( -661272677), INT32_C( 1736074301), INT32_C( 1246429845),
                            INT32_C(-1327059157), INT32_C(-1760626525), INT32_C(  693999571), INT32_C(  179503183),
                            INT32_C(-1261277577), INT32_C( 2014601419), INT32_C(   45385261), INT32_C( 1333239387),
                            INT32_C( 1950214560), INT32_C( 2050540474), INT32_C(  -73887902), INT32_C(-1586317941)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(  641343520), INT32_C(          0),
                            INT32_C(-2147316222), INT32_C(          0), INT32_C(          0), INT32_C(  136380486),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( -636451486), INT32_C(-2143256319)) },
    { UINT16_C(17481),
      simde_mm512_set_epi32(INT32_C( -441498123), INT32_C(  324738064), INT32_C(  -27713047), INT32_C(  322022433),
                            INT32_C( -623687192), INT32_C(  441486000), INT32_C(-1091397610), INT32_C(  486920838),
                            INT32_C(  727930899), INT32_C(  134578624), INT32_C( -229821250), INT32_C(-1459771681),
                            INT32_C(  786852212), INT32_C(-1562273484), INT32_C(  592450244), INT32_C( -391708168)),
      simde_mm512_set_epi32(INT32_C(  792156312), INT32_C(  407601311), INT32_C(-1255558455), INT32_C( 1648353396),
                            INT32_C(-1874603621), INT32_C(-1962724996), INT32_C(-1379808132), INT32_C(-1917277067),
                            INT32_C( -327375348), INT32_C( -266290190), INT32_C( -446684576), INT32_C( -218289365),
                            INT32_C( 1659849163), INT32_C(  313080914), INT32_C(  914897986), INT32_C( -690088867)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  134250639), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(-2130497204), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( -266323406), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1074331787), INT32_C(          0), INT32_C(          0), INT32_C(  374871045)) },
    { UINT16_C(25655),
      simde_mm512_set_epi32(INT32_C(    7734189), INT32_C(-1107618186), INT32_C( 1291997837), INT32_C( -657618671),
                            INT32_C( -523204184), INT32_C(  197247571), INT32_C(-1924672781), INT32_C( 1367953812),
                            INT32_C( 1671605226), INT32_C( -667696065), INT32_C(  734579404), INT32_C(  -25998720),
                            INT32_C( -791898275), INT32_C(-1848361166), INT32_C(  302446873), INT32_C(-1290034089)),
      simde_mm512_set_epi32(INT32_C(-2140777278), INT32_C( 1356458144), INT32_C(  990615850), INT32_C(  122581591),
                            INT32_C( 1842174798), INT32_C( 1633161914), INT32_C( 1487544794), INT32_C( 1680890315),
                            INT32_C(-1051319145), INT32_C( 1671869354), INT32_C( -657093416), INT32_C(   76483879),
                            INT32_C(  897241075), INT32_C(-1385812547), INT32_C(  518745683), INT32_C( 1278998383)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C( 1073801344), INT32_C(  839485730), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1612189864), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( -803929072), INT32_C(    9176359),
                            INT32_C(          0), INT32_C(  740425869), INT32_C(  216598082), INT32_C( 1277186856)) },
    { UINT16_C( 9319),
      simde_mm512_set_epi32(INT32_C(  359510622), INT32_C( 1667719225), INT32_C(  630674948), INT32_C(  610105763),
                            INT32_C(   20744378), INT32_C(-1334671422), INT32_C( 1934181344), INT32_C( -207473635),
                            INT32_C(  -12247390), INT32_C(  935971775), INT32_C( -814870615), INT32_C(  272416728),
                            INT32_C(-2094904434), INT32_C(  118285194), INT32_C( 1770668331), INT32_C(-1463910375)),
      simde_mm512_set_epi32(INT32_C(  399098366), INT32_C(-1713281213), INT32_C( 2124618772), INT32_C(-1052563089),
                            INT32_C( 1851869047), INT32_C( 2020277970), INT32_C(-1035589842), INT32_C(-1789987668),
                            INT32_C(  733487930), INT32_C( -497440680), INT32_C(-1951336884), INT32_C(-1752937795),
                            INT32_C(-1263292061), INT32_C(     242422), INT32_C( 1531342059), INT32_C( -447099781)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C( 1512048656), INT32_C(          0),
                            INT32_C(          0), INT32_C( 1208514576), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(-1072684992), INT32_C(    9498692), INT32_C(          0),
                            INT32_C(          0), INT32_C(     200820), INT32_C(  306457792), INT32_C( 1161907298)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_andnot_epi32(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_andnot_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(  207721957124820559), INT64_C( 7800065217939756514),
                            INT64_C(-3924116943760495845), INT64_C(-4670511705337769443),
                            INT64_C( 8681164262815197674), INT64_C(-1748050366477277388),
                            INT64_C( 6521714148432251392), INT64_C( -441034964778727222)),
      simde_mm512_set_epi64(INT64_C( 1906622887772594248), INT64_C(-6188571553716009650),
                            INT64_C(  264280323541139152), INT64_C( 1584607577489802492),
                            INT64_C( 1109502791419861047), INT64_C( 7178327557348084990),
                            INT64_C( 1395553581907359924), INT64_C(-6533533183118051768)),
      simde_mm512_set_epi64(INT64_C( 1735017709269196800), INT64_C(-9079113909020786676),
                            INT64_C(  153194412888495296), INT64_C(   58731659148920032),
                            INT64_C(  505544490090532885), INT64_C(     565705191721162),
                            INT64_C(   98516792007995572), INT64_C(  293888628881438208)) },
    { simde_mm512_set_epi64(INT64_C(-6724768156717290819), INT64_C(-5250906792133082841),
                            INT64_C( 7101763469273509464), INT64_C(-6606445878350250265),
                            INT64_C( -992513133092468415), INT64_C( 6991549638647222936),
                            INT64_C(-6702609966967541799), INT64_C( 6463119549714578187)),
      simde_mm512_set_epi64(INT64_C( -887590357697526439), INT64_C( 7877951253899372304),
                            INT64_C(-2913340636735054567), INT64_C(-9221955371178601401),
                            INT64_C(-8332586207462320569), INT64_C(-6065729331557277752),
                            INT64_C(-5495188752553836498), INT64_C(-4545091831756868823)),
      simde_mm512_set_epi64(INT64_C( 5837268749247317312), INT64_C( 5211811456626140688),
                            INT64_C(-7705320311288082175), INT64_C(    1134704830971904),
                            INT64_C(  883853889565267462), INT64_C(-8444247080808460992),
                            INT64_C( 1226140460016631846), INT64_C(-9201976626802327520)) },
    { simde_mm512_set_epi64(INT64_C(-8081018174907924542), INT64_C( 6936249846129023242),
                            INT64_C(-1059210610078769383), INT64_C( 1593162574725548027),
                            INT64_C( 2194029932784271057), INT64_C( 2297742112014824027),
                            INT64_C( 6872936620014531062), INT64_C(-4458741002964204726)),
      simde_mm512_set_epi64(INT64_C( 7389599045220123111), INT64_C(-4734617337151831127),
                            INT64_C(-3688698012661984630), INT64_C( 2942411497108224949),
                            INT64_C( 3088165388972230068), INT64_C(-8598989874996476457),
                            INT64_C(-2925060949778391940), INT64_C( 1600057734617632940)),
      simde_mm512_set_epi64(INT64_C( 6918937665425915941), INT64_C(-7059040014775614815),
                            INT64_C(  901582149085413506), INT64_C( 2936772469119858692),
                            INT64_C( 2344435893274706212), INT64_C(-9221049868269222524),
                            INT64_C(-9221110889849910264), INT64_C( 1450299817904312484)) },
    { simde_mm512_set_epi64(INT64_C(-2851531746227363368), INT64_C( 2067892326136395565),
                            INT64_C( 5955544350840259834), INT64_C(-9215158447496033102),
                            INT64_C(-6496129397571023850), INT64_C( 6580537045822776099),
                            INT64_C(-1881492268188536860), INT64_C( 6477581622128112348)),
      simde_mm512_set_epi64(INT64_C( 4736931688263401886), INT64_C( -422510099501192510),
                            INT64_C( 3904035851984069712), INT64_C(-1269778779692298262),
                            INT64_C( 7103388094266435672), INT64_C( 8538164081108009860),
                            INT64_C( 7657481289221491954), INT64_C(-6346831563088898420)),
      simde_mm512_set_epi64(INT64_C(  112770318310899718), INT64_C(-2161639582911543102),
                            INT64_C( 2596786860877701120), INT64_C( 7953586499903062856),
                            INT64_C( 4756998866794012744), INT64_C( 2606779805598826628),
                            INT64_C(  721778983603339282), INT64_C(-6482086895067069440)) },
    { simde_mm512_set_epi64(INT64_C( -821005629772787069), INT64_C(-4647973389902912809),
                            INT64_C( 6459900742609080709), INT64_C(   -1266809698382208),
                            INT64_C(  701020828809534395), INT64_C(-8547290149729742964),
                            INT64_C( -440779604644636577), INT64_C(-3509307452635316669)),
      simde_mm512_set_epi64(INT64_C( 8999318376500703433), INT64_C( 1719097867730734351),
                            INT64_C(  360091487853740826), INT64_C(-6254537314592943558),
                            INT64_C( -632347399973673450), INT64_C( 2614451855333869078),
                            INT64_C( 6887846494654494209), INT64_C( 6275950466702179569)),
      simde_mm512_set_epi64(INT64_C(  604608525006544968), INT64_C(   36136703980768520),
                            INT64_C(  313352018360009242), INT64_C(        137573240890),
                            INT64_C( -720169941136284668), INT64_C( 2596327487390613522),
                            INT64_C(  438118704866436608), INT64_C( 1157566394459521200)) },
    { simde_mm512_set_epi64(INT64_C(-5483950330033170066), INT64_C(-4153699507396814554),
                            INT64_C( 1686943364333831141), INT64_C(-6155572369391990976),
                            INT64_C(-2338197867102969548), INT64_C( 4970317907692585902),
                            INT64_C( -659027381808082615), INT64_C(-8301976371410819309)),
      simde_mm512_set_epi64(INT64_C(-5922203424268985599), INT64_C( 1802271341012641429),
                            INT64_C(-7199161640250473305), INT64_C( 4184910176757162424),
                            INT64_C(-5885970898589897236), INT64_C( 5320604596895707800),
                            INT64_C(-7049806138053003152), INT64_C( 7856069210784274088)),
      simde_mm512_set_epi64(INT64_C(  869198318683570689), INT64_C( 1801690747234690705),
                            INT64_C(-8640647776843037694), INT64_C( 1153141544681808056),
                            INT64_C( 2328590264702274760), INT64_C(  649785191505621008),
                            INT64_C(  585473076492838960), INT64_C( 6991285376398659752)) },
    { simde_mm512_set_epi64(INT64_C(  772369500911491951), INT64_C(-3487181344595680581),
                            INT64_C(-6776954808191866646), INT64_C( 1437133779275187040),
                            INT64_C(-3742444221385296201), INT64_C( 3619551202282748987),
                            INT64_C(-5676058734881350704), INT64_C( 3034639668798379519)),
      simde_mm512_set_epi64(INT64_C( 7799576852730631653), INT64_C(-4611614721990756478),
                            INT64_C( 4179897201710999091), INT64_C(-6554042946408561565),
                            INT64_C( 7858455943023474684), INT64_C(-4868663260305658784),
                            INT64_C(-6563387696243649675), INT64_C( -252761203575600938)),
      simde_mm512_set_epi64(INT64_C( 7207351508714783872), INT64_C(      71283551638784),
                            INT64_C( 1873656161226589713), INT64_C(-6626408997484215293),
                            INT64_C( 2382069952524845384), INT64_C(-8339540561327800256),
                            INT64_C(  342318686555209765), INT64_C(-3142946274104309760)) },
    { simde_mm512_set_epi64(INT64_C(-6272776462503295319), INT64_C(-8894851852280934479),
                            INT64_C( 6828037840473322695), INT64_C( -784763491569829334),
                            INT64_C(-6956613286547242208), INT64_C(-7641604144835014945),
                            INT64_C( 4137535773895137731), INT64_C( 3122415965305276610)),
      simde_mm512_set_epi64(INT64_C( 5967240469174938071), INT64_C( 2271146860082105533),
                            INT64_C( 2488999494207974941), INT64_C(-7245269557183082373),
                            INT64_C(-6094983942162054282), INT64_C( 5272800144124782830),
                            INT64_C(-1112016268759137335), INT64_C( 3873297534982922048)),
      simde_mm512_set_epi64(INT64_C( 5912488079989451094), INT64_C( 1945726568221376524),
                            INT64_C( 2308274862648494616), INT64_C(  747597780979417169),
                            INT64_C( 2308728562190385238), INT64_C( 5191575370143047712),
                            INT64_C(-4571152522202316280), INT64_C( 1477377559112455936)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_andnot_epi64(test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_andnot_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-8706055201876274534), INT64_C(-2974526497282267924),
                            INT64_C(-5064099105424399850), INT64_C( 4173762680971677425),
                            INT64_C( 5058953897646810163), INT64_C( 3129329827313761969),
                            INT64_C(-7680990319456213473), INT64_C( 3095613893972693568)),
      UINT8_C(148),
      simde_mm512_set_epi64(INT64_C( -438459145642420823), INT64_C( 2788318060387771818),
                            INT64_C(-6405634033298828022), INT64_C( 5697280571633296693),
                            INT64_C(-4038706177987584167), INT64_C( 7050984609072161968),
                            INT64_C( 2749018709708772273), INT64_C(-2265592192997989021)),
      simde_mm512_set_epi64(INT64_C(-1049270424665539045), INT64_C(-5614406584732574076),
                            INT64_C(  631202638299991092), INT64_C(-8590255914187036925),
                            INT64_C(-6307315262773811693), INT64_C(-1209843912248425712),
                            INT64_C(-3872834841544228683), INT64_C(-3085083838104197908)),
      simde_mm512_set_epi64(INT64_C(    4565176708278802), INT64_C(-2974526497282267924),
                            INT64_C(-5064099105424399850), INT64_C(-9166793916060532222),
                            INT64_C( 5058953897646810163), INT64_C(-8203935250787499264),
                            INT64_C(-7680990319456213473), INT64_C( 3095613893972693568)) },
    { simde_mm512_set_epi64(INT64_C(-4842938149095873389), INT64_C( -846085209911123390),
                            INT64_C(  902030110892207375), INT64_C(-8179884512098486778),
                            INT64_C( 7136180633023633249), INT64_C(-7202514001649392691),
                            INT64_C(-4512985345247872566), INT64_C( 6280820093975482096)),
      UINT8_C( 88),
      simde_mm512_set_epi64(INT64_C(-5899542268894168412), INT64_C( 5687678929880926481),
                            INT64_C(  754471637334648472), INT64_C( 1530269878614188173),
                            INT64_C(-3481843836368626596), INT64_C( 7214537798473258692),
                            INT64_C( 3186147264512503626), INT64_C(-2220217993706522327)),
      simde_mm512_set_epi64(INT64_C( 4741426381855247639), INT64_C(-6093431436741802321),
                            INT64_C(-7277776184535270866), INT64_C(-5890238516652006119),
                            INT64_C(-8031043717190201593), INT64_C( 7604814614465185239),
                            INT64_C( 6712821644684838579), INT64_C(-1700024539209227072)),
      simde_mm512_set_epi64(INT64_C(-4842938149095873389), INT64_C(-6845110636993050962),
                            INT64_C(  902030110892207375), INT64_C(-6178618704551475952),
                            INT64_C( 1152921504676217603), INT64_C(-7202514001649392691),
                            INT64_C(-4512985345247872566), INT64_C( 6280820093975482096)) },
    { simde_mm512_set_epi64(INT64_C( -647905387169688868), INT64_C(-8461625299591442725),
                            INT64_C(-4959110866452894415), INT64_C(-6046186632754619075),
                            INT64_C(-1792277330244185216), INT64_C( 7899374623587606112),
                            INT64_C(-2530906147097710338), INT64_C(-3452464982464189359)),
      UINT8_C(234),
      simde_mm512_set_epi64(INT64_C( 1092825191169264761), INT64_C(  518154175979275913),
                            INT64_C(-2540128939765803497), INT64_C( 7206989642204137224),
                            INT64_C( 5053971549089664110), INT64_C(  275130895293265200),
                            INT64_C( 5870095287105445532), INT64_C( 3766077764635497461)),
      simde_mm512_set_epi64(INT64_C( 4726923138274336458), INT64_C( 3036293318033390010),
                            INT64_C( 3265833753663381966), INT64_C(-5548402770380826836),
                            INT64_C(-1910939043053590920), INT64_C(-2803972634053834044),
                            INT64_C( 8571307896088376800), INT64_C(-2906367800591944553)),
      simde_mm512_set_epi64(INT64_C( 4652501007819903618), INT64_C( 2883153157893175602),
                            INT64_C( 2395932937578488264), INT64_C(-6046186632754619075),
                            INT64_C(-6820513618777071088), INT64_C( 7899374623587606112),
                            INT64_C( 2774537390188929376), INT64_C(-3452464982464189359)) },
    { simde_mm512_set_epi64(INT64_C( 1235103765186305905), INT64_C( 8251648155281492223),
                            INT64_C( 6607793927948629202), INT64_C(-4956133557414585628),
                            INT64_C( -962568210701922461), INT64_C( 7520783669412628517),
                            INT64_C( 4493695514722238610), INT64_C( 6191552237626999876)),
      UINT8_C(175),
      simde_mm512_set_epi64(INT64_C(-1999731829913464848), INT64_C( 7072204574593617968),
                            INT64_C( -329416891633690006), INT64_C( 4219653511875682573),
                            INT64_C(-5631405021388401918), INT64_C( -157450572284011331),
                            INT64_C(-6448890677231800514), INT64_C(-7780641104162742337)),
      simde_mm512_set_epi64(INT64_C(  261057906798578959), INT64_C(-4964336716206621793),
                            INT64_C(-2469501117696455323), INT64_C( 2339328587648411167),
                            INT64_C( 8220620103791574591), INT64_C(  273538927111600315),
                            INT64_C(-3298288074488883789), INT64_C(-8357787233131660724)),
      simde_mm512_set_epi64(INT64_C(  252325274594050063), INT64_C( 8251648155281492223),
                            INT64_C(  329344140649481477), INT64_C(-4956133557414585628),
                            INT64_C( 4757067868831771709), INT64_C(  147282005282398210),
                            INT64_C( 5780933484690985089), INT64_C(  577059746971148352)) },
    { simde_mm512_set_epi64(INT64_C(-4285851555602414983), INT64_C(-8492982904341423564),
                            INT64_C(-2837093742585682248), INT64_C(  267283033869441308),
                            INT64_C( 4311088349833897908), INT64_C( -647706517356585524),
                            INT64_C(-3770716194274572842), INT64_C(-8566807519504738391)),
      UINT8_C( 75),
      simde_mm512_set_epi64(INT64_C(-6282230583383062251), INT64_C(-7841791912404359359),
                            INT64_C(-7579575622870303941), INT64_C(-2922061146712111361),
                            INT64_C( 4606944383693507801), INT64_C(-6882069134795290712),
                            INT64_C(-4540648442557822523), INT64_C( 8626282944079879495)),
      simde_mm512_set_epi64(INT64_C(-1823698107073259294), INT64_C( 8029233569224881686),
                            INT64_C(   46900467487790247), INT64_C( 8663098726891022114),
                            INT64_C( 2596646339415618602), INT64_C( 7059567741718714192),
                            INT64_C( 7446336952031093968), INT64_C(   16931348739669095)),
      simde_mm512_set_epi64(INT64_C(-4285851555602414983), INT64_C( 7800656914580246550),
                            INT64_C(-2837093742585682248), INT64_C(  267283033869441308),
                            INT64_C(         88250757154), INT64_C( -647706517356585524),
                            INT64_C( 2810971851134903312), INT64_C(    2252181026775072)) },
    { simde_mm512_set_epi64(INT64_C( 2037127205197222183), INT64_C( 3451898891201360501),
                            INT64_C( 1455211247092394628), INT64_C( 2206658725580708086),
                            INT64_C( 5349364315141837270), INT64_C( 7849256443344717184),
                            INT64_C( 4856719246957022704), INT64_C(-4923001172558722698)),
      UINT8_C(149),
      simde_mm512_set_epi64(INT64_C( 6411014556179012579), INT64_C(-8290562023531042118),
                            INT64_C( 3513406971994598159), INT64_C(  170515694744852127),
                            INT64_C( 7762613428125762288), INT64_C( 4486051683696872920),
                            INT64_C(-3347799382542858009), INT64_C( 7877354972766519961)),
      simde_mm512_set_epi64(INT64_C( 2384233607786009160), INT64_C( 7136321197786935066),
                            INT64_C(-2775012291419678803), INT64_C( 1447324989515017380),
                            INT64_C(-5436087904826886612), INT64_C( 7888585058472078205),
                            INT64_C(-7864278168616859201), INT64_C( 8559884086409161720)),
      simde_mm512_set_epi64(INT64_C( 2379589521848270856), INT64_C( 3451898891201360501),
                            INT64_C( 1455211247092394628), INT64_C( 1441191875528796192),
                            INT64_C( 5349364315141837270), INT64_C( 4699579053875929637),
                            INT64_C( 4856719246957022704), INT64_C( 1335881482333858144)) },
    { simde_mm512_set_epi64(INT64_C( -626073311570320561), INT64_C( 4678237318537021585),
                            INT64_C( 7326175960335696621), INT64_C( 2614088339478761539),
                            INT64_C(-3404519381245739218), INT64_C( 8481274767690754747),
                            INT64_C(-4945537623263429760), INT64_C( 5945167030889147721)),
      UINT8_C(209),
      simde_mm512_set_epi64(INT64_C( 1396956538408270925), INT64_C(  433531675836732237),
                            INT64_C(-2740776246441943234), INT64_C(  627773489989817177),
                            INT64_C( 2334235533617502306), INT64_C( 5200994462656867787),
                            INT64_C( 6058971438237170661), INT64_C(-1718043134590880356)),
      simde_mm512_set_epi64(INT64_C( 6582702301060698834), INT64_C(-6620728110496909408),
                            INT64_C(-2674893574601157335), INT64_C(-3191892667818640289),
                            INT64_C(-1755995440120031315), INT64_C(-9164966479234216120),
                            INT64_C( -811539623059483440), INT64_C(-8790398035654865383)),
      simde_mm512_set_epi64(INT64_C( 5194902496598033042), INT64_C(-6910140186789469024),
                            INT64_C( 7326175960335696621), INT64_C(-3242571914706752506),
                            INT64_C(-3404519381245739218), INT64_C( 8481274767690754747),
                            INT64_C(-4945537623263429760), INT64_C(  432964590381304321)) },
    { simde_mm512_set_epi64(INT64_C(-6743158443935274483), INT64_C( -109319504177728220),
                            INT64_C(-4028288193005214442), INT64_C(  132288430860812468),
                            INT64_C(  917336920958928215), INT64_C(-8592087087533075804),
                            INT64_C( -911564553413882344), INT64_C(-5778334739542351628)),
      UINT8_C(132),
      simde_mm512_set_epi64(INT64_C(-8373098054511418162), INT64_C( 7896680406183363835),
                            INT64_C( 4931162839211744539), INT64_C(-7345169465412510410),
                            INT64_C(-7349547769362151281), INT64_C(    1089692206936889),
                            INT64_C( 6524506004040415129), INT64_C( 6226593529101379713)),
      simde_mm512_set_epi64(INT64_C( 3458147115787789114), INT64_C( 7210094384770191006),
                            INT64_C( 7088560670460655534), INT64_C( -803268445524244375),
                            INT64_C( 4723424603414443741), INT64_C( 1370109689785890561),
                            INT64_C(-4376650697011830162), INT64_C( -620804834547376669)),
      simde_mm512_set_epi64(INT64_C( 2607874799996928816), INT64_C( -109319504177728220),
                            INT64_C(-4028288193005214442), INT64_C(  132288430860812468),
                            INT64_C(  917336920958928215), INT64_C( 1369094837600650240),
                            INT64_C( -911564553413882344), INT64_C(-5778334739542351628)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_andnot_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_andnot_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C(  7),
      simde_mm512_set_epi64(INT64_C(-2016264017930850215), INT64_C( 6207900603916400351),
                            INT64_C( 7392720324711365837), INT64_C( 8770333430120422633),
                            INT64_C(  490532205378570002), INT64_C(-6106476949393880649),
                            INT64_C(-1854090463849988422), INT64_C( 2161894352221900559)),
      simde_mm512_set_epi64(INT64_C( 2471053143203888378), INT64_C( 4307108638624930374),
                            INT64_C( 8813537095665060151), INT64_C( -722272124812023485),
                            INT64_C( -967288076808354317), INT64_C(-6013850093851417513),
                            INT64_C( 3331958923341291108), INT64_C( -281534168919433716)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(  327214808695383104),
                            INT64_C(  592518518720905284), INT64_C(-2299289876066988032)) },
    { UINT8_C( 76),
      simde_mm512_set_epi64(INT64_C(-7188491746248886702), INT64_C( 3795103503776882624),
                            INT64_C( 8025930014425820340), INT64_C(-7929605366413196523),
                            INT64_C( 5924420044782879602), INT64_C(-3302350069387149227),
                            INT64_C(-1821341009738891830), INT64_C(-6812922588519498817)),
      simde_mm512_set_epi64(INT64_C(-1266328346505933550), INT64_C( 1669938728598205410),
                            INT64_C(-7350359895777029108), INT64_C( 9139543262716722238),
                            INT64_C(-9200593584210926828), INT64_C(-3449434666635797941),
                            INT64_C( 4314658246940308870), INT64_C( -478133805478226079)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(  217316721059520546),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(-9205313376526131196), INT64_C(       5910164412938),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(148),
      simde_mm512_set_epi64(INT64_C( 2173045647004856331), INT64_C(-9109531323294262314),
                            INT64_C(-2493109132018654878), INT64_C( 6270825741977490200),
                            INT64_C( 8719769943602297687), INT64_C(-4201021528893071940),
                            INT64_C( 9011627797455533120), INT64_C( 6620301637478416060)),
      simde_mm512_set_epi64(INT64_C(-4851330938418837166), INT64_C( 8567660546009495156),
                            INT64_C(-2946935282469126440), INT64_C(-3944680176869437518),
                            INT64_C(-3189291857021003507), INT64_C(  852944387991302704),
                            INT64_C( 5948575888921546761), INT64_C( 4930911444432807162)),
      simde_mm512_set_epi64(INT64_C(-6880224560885528240), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(-8628461452576158558),
                            INT64_C(                   0), INT64_C(  739720922782507520),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C( 97),
      simde_mm512_set_epi64(INT64_C( -924406031683798297), INT64_C( 1082742291630099615),
                            INT64_C( 3950666752159487194), INT64_C( 8443851551588188807),
                            INT64_C( 5838662214875022266), INT64_C(-6073322957639126750),
                            INT64_C( 1174103819847041898), INT64_C(  693926700598930845)),
      simde_mm512_set_epi64(INT64_C(  917406711858321823), INT64_C(-2954398701286057389),
                            INT64_C(  580508427727522845), INT64_C(-4656281121400174897),
                            INT64_C(-3028496641912979897), INT64_C( 6357018899588818011),
                            INT64_C( 5102737467710367164), INT64_C( 1099306012957445482)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-3388587049163943360),
                            INT64_C(  579945122294155269), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(  450505469795237986)) },
    { UINT8_C(210),
      simde_mm512_set_epi64(INT64_C( 4586733821042914141), INT64_C(-6562128612845469564),
                            INT64_C(-5088971089241108691), INT64_C( 4584509013736167571),
                            INT64_C( 7541158438725419821), INT64_C(-6577447853347647248),
                            INT64_C( 8000393737083977627), INT64_C(-3838210298295657456)),
      simde_mm512_set_epi64(INT64_C(-3810154219907114893), INT64_C( 1390546034528663938),
                            INT64_C(-7278948997228835946), INT64_C(-6400015342302035742),
                            INT64_C(-5025729231272531675), INT64_C( -727304839347940122),
                            INT64_C( 5841837551579279726), INT64_C(-6256756974903097514)),
      simde_mm512_set_epi64(INT64_C(-4604784503990056926), INT64_C( 1369116277674653954),
                            INT64_C(                   0), INT64_C(-9214222637876019104),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 1157530966609723492), INT64_C(                   0)) },
    { UINT8_C(171),
      simde_mm512_set_epi64(INT64_C( 2614947921582018787), INT64_C(-4851561713766233132),
                            INT64_C(-9121795968209612126), INT64_C(  784084589312935430),
                            INT64_C( 3206750945776122646), INT64_C( 2956179786298753960),
                            INT64_C( 5449808455866424595), INT64_C(  314020808054955060)),
      simde_mm512_set_epi64(INT64_C(  420924716680581769), INT64_C(  634178498505834615),
                            INT64_C(-2861544115657502554), INT64_C(-7045300656768620560),
                            INT64_C( 3724569018417139461), INT64_C( 7684038547017787602),
                            INT64_C( 4661447160348399809), INT64_C( 8780209518656646828)),
      simde_mm512_set_epi64(INT64_C(  114072716522619400), INT64_C(                   0),
                            INT64_C( 6341349786890797060), INT64_C(                   0),
                            INT64_C( 1382694151414203393), INT64_C(                   0),
                            INT64_C(    4574054841401536), INT64_C( 8755017506026431112)) },
    { UINT8_C(225),
      simde_mm512_set_epi64(INT64_C(-3697729744057786539), INT64_C(-2459882991819182775),
                            INT64_C( 6065837030945349572), INT64_C( 8437722782224197038),
                            INT64_C( 1700648554253726454), INT64_C(-4293199790864835662),
                            INT64_C( 6581402203822969825), INT64_C(-6231169800047978744)),
      simde_mm512_set_epi64(INT64_C( 8326587265612039337), INT64_C( 6780517041864519531),
                            INT64_C(-7817226648374121699), INT64_C(-3500732471169369834),
                            INT64_C( 1796671772602068213), INT64_C( 1885612779837593615),
                            INT64_C(-6040660189943903948), INT64_C(  675381603587673544)),
      simde_mm512_set_epi64(INT64_C( 3675211075874242728), INT64_C(  144396663190979106),
                            INT64_C(-8970746854625959911), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(   25068767419582656)) },
    { UINT8_C(182),
      simde_mm512_set_epi64(INT64_C(-3172486234888138881), INT64_C( 3556874573334620913),
                            INT64_C(-7174772828994546158), INT64_C( -768272060832782008),
                            INT64_C(-7948383401788128664), INT64_C(-3962825949835743119),
                            INT64_C(-4177466042331622142), INT64_C(-4344904134560657490)),
      simde_mm512_set_epi64(INT64_C( 3131865100191000199), INT64_C( 3277342092864256055),
                            INT64_C( 2638156770812089616), INT64_C(-5499406567603861656),
                            INT64_C( 5836973950118592576), INT64_C(-4232123399129603430),
                            INT64_C( 8656431254350139121), INT64_C(-8853511068983619849)),
      simde_mm512_set_epi64(INT64_C( 2884148896870883456), INT64_C(                   0),
                            INT64_C( 2346526704673489152), INT64_C(  191440763308049952),
                            INT64_C(                   0), INT64_C(  307450048046744202),
                            INT64_C( 4044604204229069553), INT64_C(                   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_andnot_epi64(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_blend_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(12684),
      simde_mm512_set_epi32(INT32_C( 2139597246), INT32_C(-2035467821), INT32_C(-1381016544), INT32_C( -293624181),
                            INT32_C( 1610331725), INT32_C(  134146865), INT32_C(  837546022), INT32_C(-1561535917),
                            INT32_C(  522767958), INT32_C(  350987310), INT32_C( 1200416302), INT32_C( 1035712103),
                            INT32_C( -697441354), INT32_C(-1209277994), INT32_C( 1905768221), INT32_C(  941422574)),
      simde_mm512_set_epi32(INT32_C( -854245863), INT32_C(   77819890), INT32_C( -597706880), INT32_C(  -28687979),
                            INT32_C( 1384494246), INT32_C(-1885694903), INT32_C(  479641666), INT32_C(  436747778),
                            INT32_C( 1142933685), INT32_C(  -36150185), INT32_C(  234764144), INT32_C( -925516387),
                            INT32_C( 1528722995), INT32_C( 1957265461), INT32_C( -558613563), INT32_C(  535737103)),
      simde_mm512_set_epi32(INT32_C( 2139597246), INT32_C(-2035467821), INT32_C( -597706880), INT32_C(  -28687979),
                            INT32_C( 1610331725), INT32_C(  134146865), INT32_C(  837546022), INT32_C(  436747778),
                            INT32_C( 1142933685), INT32_C(  350987310), INT32_C( 1200416302), INT32_C( 1035712103),
                            INT32_C( 1528722995), INT32_C( 1957265461), INT32_C( 1905768221), INT32_C(  941422574)) },
    { UINT16_C(12889),
      simde_mm512_set_epi32(INT32_C(-1009854213), INT32_C(   19207470), INT32_C( 2053019824), INT32_C( 1679893080),
                            INT32_C( -761309092), INT32_C(-1797634461), INT32_C( 1499461014), INT32_C( -349931656),
                            INT32_C( 1308377490), INT32_C(-1862232386), INT32_C( -706282442), INT32_C( 1752887042),
                            INT32_C( 1045610342), INT32_C(   88096217), INT32_C(-1144289151), INT32_C( 1410502196)),
      simde_mm512_set_epi32(INT32_C( 1437842356), INT32_C(-1817562257), INT32_C(  808924311), INT32_C( 1765692072),
                            INT32_C(-1346910557), INT32_C(  -92284700), INT32_C( 1533217965), INT32_C( 1732689820),
                            INT32_C(-1401128233), INT32_C( -762168473), INT32_C(   97276971), INT32_C( 2145432631),
                            INT32_C(-1561525899), INT32_C(-2005427238), INT32_C( -455460474), INT32_C( -933959435)),
      simde_mm512_set_epi32(INT32_C(-1009854213), INT32_C(   19207470), INT32_C(  808924311), INT32_C( 1765692072),
                            INT32_C( -761309092), INT32_C(-1797634461), INT32_C( 1533217965), INT32_C( -349931656),
                            INT32_C( 1308377490), INT32_C( -762168473), INT32_C( -706282442), INT32_C( 2145432631),
                            INT32_C(-1561525899), INT32_C(   88096217), INT32_C(-1144289151), INT32_C( -933959435)) },
    { UINT16_C(18390),
      simde_mm512_set_epi32(INT32_C(  191788932), INT32_C(  410937469), INT32_C(  218604234), INT32_C( -632545043),
                            INT32_C(  246082482), INT32_C( 2029197195), INT32_C( 1188965621), INT32_C( -844747875),
                            INT32_C(  989502056), INT32_C( 2119540790), INT32_C(-1766179858), INT32_C(-1109416221),
                            INT32_C(-1963025204), INT32_C( -932958949), INT32_C(   47867627), INT32_C( -567270366)),
      simde_mm512_set_epi32(INT32_C(  558500028), INT32_C( 1400907983), INT32_C(-1581715774), INT32_C(  -29022872),
                            INT32_C( 1773849857), INT32_C( -311975417), INT32_C( 1183760637), INT32_C(-1160252785),
                            INT32_C( 2107838031), INT32_C( 1909470743), INT32_C(-2018375211), INT32_C(  267812095),
                            INT32_C( -225335539), INT32_C( -871226308), INT32_C( -872412082), INT32_C( 1435481672)),
      simde_mm512_set_epi32(INT32_C(  191788932), INT32_C( 1400907983), INT32_C(  218604234), INT32_C( -632545043),
                            INT32_C(  246082482), INT32_C( -311975417), INT32_C( 1183760637), INT32_C(-1160252785),
                            INT32_C( 2107838031), INT32_C( 1909470743), INT32_C(-1766179858), INT32_C(  267812095),
                            INT32_C(-1963025204), INT32_C( -871226308), INT32_C( -872412082), INT32_C( -567270366)) },
    { UINT16_C(46409),
      simde_mm512_set_epi32(INT32_C( -239336968), INT32_C( 1154172094), INT32_C( 1382102779), INT32_C(-1946237060),
                            INT32_C( -995869857), INT32_C( 1050338310), INT32_C(-2047829467), INT32_C( -881484106),
                            INT32_C(  -67227529), INT32_C( -510303256), INT32_C( -387177060), INT32_C( 2007995362),
                            INT32_C( 1596393504), INT32_C( 1716443052), INT32_C( 1056333857), INT32_C( -879795312)),
      simde_mm512_set_epi32(INT32_C(-1321156942), INT32_C(-1763902924), INT32_C( -145319736), INT32_C(  356975558),
                            INT32_C(  109934631), INT32_C( 1326272066), INT32_C(  723198088), INT32_C(-1812908400),
                            INT32_C( 1665386649), INT32_C( 1770577849), INT32_C(-2015166919), INT32_C(-1565649496),
                            INT32_C( 1045296779), INT32_C( 1401153164), INT32_C( -294475079), INT32_C(  378377774)),
      simde_mm512_set_epi32(INT32_C(-1321156942), INT32_C( 1154172094), INT32_C( -145319736), INT32_C(  356975558),
                            INT32_C( -995869857), INT32_C( 1326272066), INT32_C(-2047829467), INT32_C(-1812908400),
                            INT32_C(  -67227529), INT32_C( 1770577849), INT32_C( -387177060), INT32_C( 2007995362),
                            INT32_C( 1045296779), INT32_C( 1716443052), INT32_C( 1056333857), INT32_C(  378377774)) },
    { UINT16_C(35033),
      simde_mm512_set_epi32(INT32_C(  576121858), INT32_C(  -83274089), INT32_C( 1081604364), INT32_C( 1853977291),
                            INT32_C(-1408149319), INT32_C(-1793071292), INT32_C( -580417531), INT32_C( 1708989591),
                            INT32_C(-1803428364), INT32_C(-1884594628), INT32_C(-1049896819), INT32_C( 1351777033),
                            INT32_C( -543435799), INT32_C(   45073785), INT32_C(  310971883), INT32_C(  586295496)),
      simde_mm512_set_epi32(INT32_C(  274706498), INT32_C( 1339140311), INT32_C(  773365916), INT32_C( -407443831),
                            INT32_C(  -44487881), INT32_C( -363465063), INT32_C( -778555208), INT32_C( -640038352),
                            INT32_C( -428291654), INT32_C(-1080717955), INT32_C(  906661653), INT32_C( 1353353955),
                            INT32_C(-1311936279), INT32_C( 1168846380), INT32_C(  -71806717), INT32_C(  617275327)),
      simde_mm512_set_epi32(INT32_C(  274706498), INT32_C(  -83274089), INT32_C( 1081604364), INT32_C( 1853977291),
                            INT32_C(  -44487881), INT32_C(-1793071292), INT32_C( -580417531), INT32_C( 1708989591),
                            INT32_C( -428291654), INT32_C(-1080717955), INT32_C(-1049896819), INT32_C( 1353353955),
                            INT32_C(-1311936279), INT32_C(   45073785), INT32_C(  310971883), INT32_C(  617275327)) },
    { UINT16_C(62826),
      simde_mm512_set_epi32(INT32_C( -943712419), INT32_C( -981833223), INT32_C( 2020022414), INT32_C(  630972788),
                            INT32_C( 1615502534), INT32_C(  991949979), INT32_C(  601817641), INT32_C(-2063962607),
                            INT32_C(-1545145030), INT32_C( 1626575612), INT32_C(-1511315708), INT32_C( 1422623346),
                            INT32_C( 1496301111), INT32_C(-1751918881), INT32_C(  333195983), INT32_C( 1655699275)),
      simde_mm512_set_epi32(INT32_C(-1770653828), INT32_C( -674401292), INT32_C(-2023667251), INT32_C( 1038799540),
                            INT32_C(-1877506849), INT32_C(  791301479), INT32_C(-2115975814), INT32_C( 1430860109),
                            INT32_C(-2123570597), INT32_C(-2010985064), INT32_C( 1367050649), INT32_C( -268988786),
                            INT32_C( 1975120887), INT32_C(   83320183), INT32_C( 2120549505), INT32_C(  163195572)),
      simde_mm512_set_epi32(INT32_C(-1770653828), INT32_C( -674401292), INT32_C(-2023667251), INT32_C( 1038799540),
                            INT32_C( 1615502534), INT32_C(  791301479), INT32_C(  601817641), INT32_C( 1430860109),
                            INT32_C(-1545145030), INT32_C(-2010985064), INT32_C( 1367050649), INT32_C( 1422623346),
                            INT32_C( 1975120887), INT32_C(-1751918881), INT32_C( 2120549505), INT32_C( 1655699275)) },
    { UINT16_C( 2185),
      simde_mm512_set_epi32(INT32_C( 1990735603), INT32_C( -226564964), INT32_C( 2030923034), INT32_C( 1100474296),
                            INT32_C( 1370205128), INT32_C( 1609607637), INT32_C( -586788969), INT32_C( 1072075481),
                            INT32_C( -723316478), INT32_C(-1331909036), INT32_C(-1524508879), INT32_C( -832646284),
                            INT32_C(-1636241911), INT32_C(  215718784), INT32_C(  478814832), INT32_C( 1231303010)),
      simde_mm512_set_epi32(INT32_C(   96871414), INT32_C(  245175863), INT32_C( 1584772139), INT32_C( 1604795180),
                            INT32_C(-1972440261), INT32_C(-1471456371), INT32_C( 1108051273), INT32_C(   -4141330),
                            INT32_C(-1064630270), INT32_C( -911670021), INT32_C(-1808660435), INT32_C(-2035294308),
                            INT32_C(-2014140232), INT32_C(-1194657062), INT32_C(-1732550793), INT32_C(-1643523135)),
      simde_mm512_set_epi32(INT32_C( 1990735603), INT32_C( -226564964), INT32_C( 2030923034), INT32_C( 1100474296),
                            INT32_C(-1972440261), INT32_C( 1609607637), INT32_C( -586788969), INT32_C( 1072075481),
                            INT32_C(-1064630270), INT32_C(-1331909036), INT32_C(-1524508879), INT32_C( -832646284),
                            INT32_C(-2014140232), INT32_C(  215718784), INT32_C(  478814832), INT32_C(-1643523135)) },
    { UINT16_C(11244),
      simde_mm512_set_epi32(INT32_C( 1605800253), INT32_C( 1825375434), INT32_C( -280209407), INT32_C( 1616462798),
                            INT32_C( 1380939760), INT32_C( 1670822362), INT32_C(-1872387325), INT32_C(  451486273),
                            INT32_C( -978012107), INT32_C( -718128180), INT32_C(-1625787118), INT32_C( -879101117),
                            INT32_C(  173600397), INT32_C( 1426384314), INT32_C( -517748272), INT32_C(   76760759)),
      simde_mm512_set_epi32(INT32_C(  805228357), INT32_C(  980137697), INT32_C( -555168446), INT32_C(-2016549382),
                            INT32_C(  927860791), INT32_C(-1515288559), INT32_C( -918296563), INT32_C(-1858382028),
                            INT32_C( 1365882699), INT32_C(  616589376), INT32_C(-1573056329), INT32_C( 1014781400),
                            INT32_C( -260360112), INT32_C( -953114112), INT32_C( 1191757764), INT32_C( -706360509)),
      simde_mm512_set_epi32(INT32_C( 1605800253), INT32_C( 1825375434), INT32_C( -555168446), INT32_C( 1616462798),
                            INT32_C(  927860791), INT32_C( 1670822362), INT32_C( -918296563), INT32_C(-1858382028),
                            INT32_C( 1365882699), INT32_C(  616589376), INT32_C(-1573056329), INT32_C( -879101117),
                            INT32_C( -260360112), INT32_C( -953114112), INT32_C( -517748272), INT32_C(   76760759)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_blend_epi32(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_blend_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C(140),
      simde_mm512_set_epi64(INT64_C(-8742267720341431264), INT64_C(-1261106253099452851),
                            INT64_C(  576156398873473062), INT64_C(-6706745694521602474),
                            INT64_C( 1507479018961430062), INT64_C( 4448349614053909430),
                            INT64_C(-5193809434096716003), INT64_C( 4043379170809950035)),
      simde_mm512_set_epi64(INT64_C(  334233886225577856), INT64_C( -123213930208840538),
                            INT64_C(-8098997938139250622), INT64_C( 1875817424253601973),
                            INT64_C( -155263862084585616), INT64_C(-3975062612548356557),
                            INT64_C( 8406391148321717189), INT64_C( 2300973338778380734)),
      simde_mm512_set_epi64(INT64_C(  334233886225577856), INT64_C(-1261106253099452851),
                            INT64_C(  576156398873473062), INT64_C(-6706745694521602474),
                            INT64_C( -155263862084585616), INT64_C(-3975062612548356557),
                            INT64_C(-5193809434096716003), INT64_C( 4043379170809950035)) },
    { UINT8_C( 25),
      simde_mm512_set_epi64(INT64_C( 8817653003799568984), INT64_C(-3269797649790122397),
                            INT64_C( 6440136020702033784), INT64_C( 5619438532805301950),
                            INT64_C(-3033459988376129790), INT64_C( 4490862223337471449),
                            INT64_C(-4914684479302103500), INT64_C(  357175739365339737)),
      simde_mm512_set_epi64(INT64_C( 3474303462450025128), INT64_C(-5784936788749461276),
                            INT64_C( 6585121019047362460), INT64_C(-6017799934704469145),
                            INT64_C(  417801411244373047), INT64_C(-6706702665772459046),
                            INT64_C(-1956187837089650443), INT64_C(-4337290818543610578)),
      simde_mm512_set_epi64(INT64_C( 8817653003799568984), INT64_C(-3269797649790122397),
                            INT64_C( 6440136020702033784), INT64_C(-6017799934704469145),
                            INT64_C(  417801411244373047), INT64_C( 4490862223337471449),
                            INT64_C(-4914684479302103500), INT64_C(-4337290818543610578)) },
    { UINT8_C(111),
      simde_mm512_set_epi64(INT64_C(-2716760272685831246), INT64_C( 8715335590848900341),
                            INT64_C(-3628164495500993944), INT64_C( 9103358378116791278),
                            INT64_C(-4764906384514966324), INT64_C(-4007028174417664277),
                            INT64_C(-2436407666547579589), INT64_C( 1334270375494925236)),
      simde_mm512_set_epi64(INT64_C( -124652284302144255), INT64_C(-1339924211987201795),
                            INT64_C(-4983247764560081329), INT64_C( 8201114396130413013),
                            INT64_C( 1150244193567876877), INT64_C(-3741888496852267954),
                            INT64_C( 6165346835439187844), INT64_C( 1764962990274618058)),
      simde_mm512_set_epi64(INT64_C(-2716760272685831246), INT64_C(-1339924211987201795),
                            INT64_C(-4983247764560081329), INT64_C( 9103358378116791278),
                            INT64_C( 1150244193567876877), INT64_C(-3741888496852267954),
                            INT64_C( 6165346835439187844), INT64_C( 1764962990274618058)) },
    { UINT8_C(194),
      simde_mm512_set_epi64(INT64_C(-4277228465836858362), INT64_C(-8795360585136628042),
                            INT64_C( -288740034661227544), INT64_C(-1662912808453434398),
                            INT64_C( 6856457892943288236), INT64_C( 4536919372887712656),
                            INT64_C(-1781208167188155063), INT64_C( 2398739356475992271)),
      simde_mm512_set_epi64(INT64_C(  472165646169099842), INT64_C( 3106112138971788944),
                            INT64_C( 7152781194420608953), INT64_C(-8655076010356763224),
                            INT64_C( 4489515481820292748), INT64_C(-1264760833413638610),
                            INT64_C(-1027944449129626434), INT64_C( 5936086237864445820)),
      simde_mm512_set_epi64(INT64_C(  472165646169099842), INT64_C( 3106112138971788944),
                            INT64_C( -288740034661227544), INT64_C(-1662912808453434398),
                            INT64_C( 6856457892943288236), INT64_C( 4536919372887712656),
                            INT64_C(-1027944449129626434), INT64_C( 2398739356475992271)) },
    { UINT8_C(198),
      simde_mm512_set_epi64(INT64_C(-7701182554821916667), INT64_C( 7340054405040954868),
                            INT64_C(-8094272290232215411), INT64_C( 5805838151970444265),
                            INT64_C(  193590432792907243), INT64_C( 2518119983696523684),
                            INT64_C(-7781788212556415310), INT64_C(-7575905367749125944)),
      simde_mm512_set_epi64(INT64_C(-1561070555307167560), INT64_C(-2748943786159060550),
                            INT64_C(-4641648272018338027), INT64_C( 5812610979620286697),
                            INT64_C( 5020156980371149059), INT64_C( 2651177342668827650),
                            INT64_C( -357659487777588980), INT64_C( 7962771835258493113)),
      simde_mm512_set_epi64(INT64_C(-1561070555307167560), INT64_C(-2748943786159060550),
                            INT64_C(-8094272290232215411), INT64_C( 5805838151970444265),
                            INT64_C(  193590432792907243), INT64_C( 2651177342668827650),
                            INT64_C( -357659487777588980), INT64_C(-7575905367749125944)) },
    { UINT8_C( 55),
      simde_mm512_set_epi64(INT64_C( 2584787088481873425), INT64_C(-6636347369800363268),
                            INT64_C(-6491051538368462222), INT64_C( 6426564339256514271),
                            INT64_C( 1431065851799271243), INT64_C(-5052971989011532438),
                            INT64_C( 1179855426247829719), INT64_C( 3321581320948606601)),
      simde_mm512_set_epi64(INT64_C(-9088046918826118835), INT64_C(-9120666262578213480),
                            INT64_C( 5871437833456553614), INT64_C( 8483079615394831735),
                            INT64_C( 9107690773687184052), INT64_C(-4053213973120914951),
                            INT64_C( 8675930205947945332), INT64_C( 6938530551127078043)),
      simde_mm512_set_epi64(INT64_C( 2584787088481873425), INT64_C(-6636347369800363268),
                            INT64_C( 5871437833456553614), INT64_C( 8483079615394831735),
                            INT64_C( 1431065851799271243), INT64_C(-4053213973120914951),
                            INT64_C( 8675930205947945332), INT64_C( 6938530551127078043)) },
    { UINT8_C(103),
      simde_mm512_set_epi64(INT64_C( 4604529133310120194), INT64_C(-5720505748096428239),
                            INT64_C(-3576188556257202679), INT64_C(  926505122891702896),
                            INT64_C( 5288406162053320871), INT64_C(-5908713324082235524),
                            INT64_C(-2896531491248846387), INT64_C( 4461610053817304287)),
      simde_mm512_set_epi64(INT64_C(  -17786873681606654), INT64_C(-3915592922452326355),
                            INT64_C(-8741522488314124104), INT64_C(-5131013008663027849),
                            INT64_C(-7058878113053657357), INT64_C( -973089108768494310),
                            INT64_C( 4726501112778828744), INT64_C( 6913212164015017879)),
      simde_mm512_set_epi64(INT64_C( 4604529133310120194), INT64_C(-3915592922452326355),
                            INT64_C(-8741522488314124104), INT64_C(  926505122891702896),
                            INT64_C( 5288406162053320871), INT64_C( -973089108768494310),
                            INT64_C( 4726501112778828744), INT64_C( 6913212164015017879)) },
    { UINT8_C( 73),
      simde_mm512_set_epi64(INT64_C(-4200530011080213556), INT64_C(-6982702498652226749),
                            INT64_C(  745608029114000826), INT64_C(-2223711895723751753),
                            INT64_C( 4918324162995104748), INT64_C(  416059555292452407),
                            INT64_C( 6806544510221761324), INT64_C(-8471566411485193331)),
      simde_mm512_set_epi64(INT64_C( 5866421522993801280), INT64_C(-6756225486806034984),
                            INT64_C(-1118238162881043968), INT64_C( 5118560624722692931),
                            INT64_C( 6896859572368901322), INT64_C(-1203490237480090674),
                            INT64_C( 5931091108616911322), INT64_C(-8041842325868436927)),
      simde_mm512_set_epi64(INT64_C(-4200530011080213556), INT64_C(-6756225486806034984),
                            INT64_C(  745608029114000826), INT64_C(-2223711895723751753),
                            INT64_C( 6896859572368901322), INT64_C(  416059555292452407),
                            INT64_C( 6806544510221761324), INT64_C(-8041842325868436927)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_blend_epi64(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_blend_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512 a;
    simde__m512 b;
    simde__m512 r;
  } test_vec[8] = {
    { UINT16_C(28658),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   986.64), SIMDE_FLOAT32_C(   121.90), SIMDE_FLOAT32_C(  -796.62), SIMDE_FLOAT32_C(   983.17),
                         SIMDE_FLOAT32_C(   569.02), SIMDE_FLOAT32_C(   -88.58), SIMDE_FLOAT32_C(  -750.53), SIMDE_FLOAT32_C(    52.16),
                         SIMDE_FLOAT32_C(   863.27), SIMDE_FLOAT32_C(  -937.53), SIMDE_FLOAT32_C(   272.85), SIMDE_FLOAT32_C(  -836.56),
                         SIMDE_FLOAT32_C(  -517.71), SIMDE_FLOAT32_C(   436.89), SIMDE_FLOAT32_C(  -561.62), SIMDE_FLOAT32_C(  -796.29)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   721.67), SIMDE_FLOAT32_C(  -355.29), SIMDE_FLOAT32_C(  -776.65), SIMDE_FLOAT32_C(  -467.78),
                         SIMDE_FLOAT32_C(  -890.68), SIMDE_FLOAT32_C(  -288.13), SIMDE_FLOAT32_C(   739.88), SIMDE_FLOAT32_C(    -3.67),
                         SIMDE_FLOAT32_C(   356.91), SIMDE_FLOAT32_C(  -250.13), SIMDE_FLOAT32_C(  -609.99), SIMDE_FLOAT32_C(  -756.57),
                         SIMDE_FLOAT32_C(  -441.01), SIMDE_FLOAT32_C(   675.23), SIMDE_FLOAT32_C(  -112.56), SIMDE_FLOAT32_C(   752.66)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   986.64), SIMDE_FLOAT32_C(  -355.29), SIMDE_FLOAT32_C(  -776.65), SIMDE_FLOAT32_C(   983.17),
                         SIMDE_FLOAT32_C(  -890.68), SIMDE_FLOAT32_C(  -288.13), SIMDE_FLOAT32_C(   739.88), SIMDE_FLOAT32_C(    -3.67),
                         SIMDE_FLOAT32_C(   356.91), SIMDE_FLOAT32_C(  -250.13), SIMDE_FLOAT32_C(  -609.99), SIMDE_FLOAT32_C(  -756.57),
                         SIMDE_FLOAT32_C(  -517.71), SIMDE_FLOAT32_C(   436.89), SIMDE_FLOAT32_C(  -112.56), SIMDE_FLOAT32_C(  -796.29)) },
    { UINT16_C(13167),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -177.79), SIMDE_FLOAT32_C(   957.03), SIMDE_FLOAT32_C(  -193.15), SIMDE_FLOAT32_C(   645.09),
                         SIMDE_FLOAT32_C(    -0.96), SIMDE_FLOAT32_C(    66.15), SIMDE_FLOAT32_C(   565.09), SIMDE_FLOAT32_C(  -991.06),
                         SIMDE_FLOAT32_C(  -217.74), SIMDE_FLOAT32_C(   162.91), SIMDE_FLOAT32_C(   837.05), SIMDE_FLOAT32_C(   132.83),
                         SIMDE_FLOAT32_C(  -183.75), SIMDE_FLOAT32_C(  -958.98), SIMDE_FLOAT32_C(  -343.18), SIMDE_FLOAT32_C(  -412.04)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -623.32), SIMDE_FLOAT32_C(   372.80), SIMDE_FLOAT32_C(  -286.04), SIMDE_FLOAT32_C(   347.55),
                         SIMDE_FLOAT32_C(  -954.70), SIMDE_FLOAT32_C(   272.86), SIMDE_FLOAT32_C(   787.91), SIMDE_FLOAT32_C(   529.75),
                         SIMDE_FLOAT32_C(   -43.99), SIMDE_FLOAT32_C(   645.49), SIMDE_FLOAT32_C(  -301.76), SIMDE_FLOAT32_C(  -390.74),
                         SIMDE_FLOAT32_C(   671.11), SIMDE_FLOAT32_C(  -513.10), SIMDE_FLOAT32_C(   467.15), SIMDE_FLOAT32_C(  -961.27)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -177.79), SIMDE_FLOAT32_C(   957.03), SIMDE_FLOAT32_C(  -286.04), SIMDE_FLOAT32_C(   347.55),
                         SIMDE_FLOAT32_C(    -0.96), SIMDE_FLOAT32_C(    66.15), SIMDE_FLOAT32_C(   787.91), SIMDE_FLOAT32_C(   529.75),
                         SIMDE_FLOAT32_C(  -217.74), SIMDE_FLOAT32_C(   645.49), SIMDE_FLOAT32_C(  -301.76), SIMDE_FLOAT32_C(   132.83),
                         SIMDE_FLOAT32_C(   671.11), SIMDE_FLOAT32_C(  -513.10), SIMDE_FLOAT32_C(   467.15), SIMDE_FLOAT32_C(  -961.27)) },
    { UINT16_C(10447),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   986.49), SIMDE_FLOAT32_C(   854.73), SIMDE_FLOAT32_C(   459.72), SIMDE_FLOAT32_C(  -110.83),
                         SIMDE_FLOAT32_C(  -875.29), SIMDE_FLOAT32_C(   594.30), SIMDE_FLOAT32_C(  -331.55), SIMDE_FLOAT32_C(  -808.64),
                         SIMDE_FLOAT32_C(   705.45), SIMDE_FLOAT32_C(   -55.08), SIMDE_FLOAT32_C(   606.63), SIMDE_FLOAT32_C(   -13.01),
                         SIMDE_FLOAT32_C(   483.39), SIMDE_FLOAT32_C(   565.56), SIMDE_FLOAT32_C(   735.84), SIMDE_FLOAT32_C(  -855.34)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   263.46), SIMDE_FLOAT32_C(  -173.99), SIMDE_FLOAT32_C(  -448.77), SIMDE_FLOAT32_C(   -18.46),
                         SIMDE_FLOAT32_C(    60.12), SIMDE_FLOAT32_C(   895.07), SIMDE_FLOAT32_C(   593.75), SIMDE_FLOAT32_C(  -910.69),
                         SIMDE_FLOAT32_C(  -898.20), SIMDE_FLOAT32_C(  -885.41), SIMDE_FLOAT32_C(  -446.34), SIMDE_FLOAT32_C(  -539.23),
                         SIMDE_FLOAT32_C(   177.56), SIMDE_FLOAT32_C(    85.90), SIMDE_FLOAT32_C(  -977.71), SIMDE_FLOAT32_C(   589.01)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   986.49), SIMDE_FLOAT32_C(   854.73), SIMDE_FLOAT32_C(  -448.77), SIMDE_FLOAT32_C(  -110.83),
                         SIMDE_FLOAT32_C(    60.12), SIMDE_FLOAT32_C(   594.30), SIMDE_FLOAT32_C(  -331.55), SIMDE_FLOAT32_C(  -808.64),
                         SIMDE_FLOAT32_C(  -898.20), SIMDE_FLOAT32_C(  -885.41), SIMDE_FLOAT32_C(   606.63), SIMDE_FLOAT32_C(   -13.01),
                         SIMDE_FLOAT32_C(   177.56), SIMDE_FLOAT32_C(    85.90), SIMDE_FLOAT32_C(  -977.71), SIMDE_FLOAT32_C(   589.01)) },
    { UINT16_C(64052),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -833.77), SIMDE_FLOAT32_C(  -382.41), SIMDE_FLOAT32_C(   155.80), SIMDE_FLOAT32_C(  -175.51),
                         SIMDE_FLOAT32_C(   270.94), SIMDE_FLOAT32_C(  -347.54), SIMDE_FLOAT32_C(  -823.80), SIMDE_FLOAT32_C(  -462.55),
                         SIMDE_FLOAT32_C(    93.71), SIMDE_FLOAT32_C(  -510.90), SIMDE_FLOAT32_C(   589.53), SIMDE_FLOAT32_C(   762.37),
                         SIMDE_FLOAT32_C(   -64.95), SIMDE_FLOAT32_C(  -200.72), SIMDE_FLOAT32_C(   590.31), SIMDE_FLOAT32_C(   904.10)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   932.33), SIMDE_FLOAT32_C(  -948.81), SIMDE_FLOAT32_C(  -663.23), SIMDE_FLOAT32_C(  -224.49),
                         SIMDE_FLOAT32_C(    61.61), SIMDE_FLOAT32_C(  -513.25), SIMDE_FLOAT32_C(   862.87), SIMDE_FLOAT32_C(   888.55),
                         SIMDE_FLOAT32_C(  -356.41), SIMDE_FLOAT32_C(   536.26), SIMDE_FLOAT32_C(    46.41), SIMDE_FLOAT32_C(   968.69),
                         SIMDE_FLOAT32_C(   819.71), SIMDE_FLOAT32_C(  -256.62), SIMDE_FLOAT32_C(  -508.11), SIMDE_FLOAT32_C(   806.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   932.33), SIMDE_FLOAT32_C(  -948.81), SIMDE_FLOAT32_C(  -663.23), SIMDE_FLOAT32_C(  -224.49),
                         SIMDE_FLOAT32_C(    61.61), SIMDE_FLOAT32_C(  -347.54), SIMDE_FLOAT32_C(   862.87), SIMDE_FLOAT32_C(  -462.55),
                         SIMDE_FLOAT32_C(    93.71), SIMDE_FLOAT32_C(  -510.90), SIMDE_FLOAT32_C(    46.41), SIMDE_FLOAT32_C(   968.69),
                         SIMDE_FLOAT32_C(   -64.95), SIMDE_FLOAT32_C(  -256.62), SIMDE_FLOAT32_C(   590.31), SIMDE_FLOAT32_C(   904.10)) },
    { UINT16_C(43223),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   810.27), SIMDE_FLOAT32_C(   830.75), SIMDE_FLOAT32_C(   701.96), SIMDE_FLOAT32_C(   496.75),
                         SIMDE_FLOAT32_C(  -369.80), SIMDE_FLOAT32_C(  -455.71), SIMDE_FLOAT32_C(  -712.56), SIMDE_FLOAT32_C(   961.22),
                         SIMDE_FLOAT32_C(  -136.67), SIMDE_FLOAT32_C(   165.04), SIMDE_FLOAT32_C(  -204.19), SIMDE_FLOAT32_C(   122.42),
                         SIMDE_FLOAT32_C(  -370.53), SIMDE_FLOAT32_C(  -979.01), SIMDE_FLOAT32_C(  -726.98), SIMDE_FLOAT32_C(   156.30)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -639.87), SIMDE_FLOAT32_C(   979.28), SIMDE_FLOAT32_C(   637.46), SIMDE_FLOAT32_C(   800.56),
                         SIMDE_FLOAT32_C(  -577.80), SIMDE_FLOAT32_C(   389.08), SIMDE_FLOAT32_C(   966.56), SIMDE_FLOAT32_C(  -731.72),
                         SIMDE_FLOAT32_C(  -496.34), SIMDE_FLOAT32_C(   344.28), SIMDE_FLOAT32_C(   729.72), SIMDE_FLOAT32_C(   160.21),
                         SIMDE_FLOAT32_C(   511.10), SIMDE_FLOAT32_C(   746.94), SIMDE_FLOAT32_C(  -855.19), SIMDE_FLOAT32_C(   203.47)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -639.87), SIMDE_FLOAT32_C(   830.75), SIMDE_FLOAT32_C(   637.46), SIMDE_FLOAT32_C(   496.75),
                         SIMDE_FLOAT32_C(  -577.80), SIMDE_FLOAT32_C(  -455.71), SIMDE_FLOAT32_C(  -712.56), SIMDE_FLOAT32_C(   961.22),
                         SIMDE_FLOAT32_C(  -496.34), SIMDE_FLOAT32_C(   344.28), SIMDE_FLOAT32_C(  -204.19), SIMDE_FLOAT32_C(   160.21),
                         SIMDE_FLOAT32_C(  -370.53), SIMDE_FLOAT32_C(   746.94), SIMDE_FLOAT32_C(  -855.19), SIMDE_FLOAT32_C(   203.47)) },
    { UINT16_C(29684),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -516.27), SIMDE_FLOAT32_C(  -631.52), SIMDE_FLOAT32_C(  -333.70), SIMDE_FLOAT32_C(    63.56),
                         SIMDE_FLOAT32_C(   874.74), SIMDE_FLOAT32_C(  -961.20), SIMDE_FLOAT32_C(  -924.01), SIMDE_FLOAT32_C(   542.80),
                         SIMDE_FLOAT32_C(  -706.18), SIMDE_FLOAT32_C(  -538.09), SIMDE_FLOAT32_C(    38.89), SIMDE_FLOAT32_C(  -242.57),
                         SIMDE_FLOAT32_C(  -337.54), SIMDE_FLOAT32_C(   184.20), SIMDE_FLOAT32_C(  -229.00), SIMDE_FLOAT32_C(  -133.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    57.66), SIMDE_FLOAT32_C(   125.72), SIMDE_FLOAT32_C(    14.67), SIMDE_FLOAT32_C(    11.14),
                         SIMDE_FLOAT32_C(  -363.42), SIMDE_FLOAT32_C(   -80.26), SIMDE_FLOAT32_C(   -12.54), SIMDE_FLOAT32_C(   560.55),
                         SIMDE_FLOAT32_C(   -59.35), SIMDE_FLOAT32_C(  -247.72), SIMDE_FLOAT32_C(  -719.76), SIMDE_FLOAT32_C(   280.49),
                         SIMDE_FLOAT32_C(   296.24), SIMDE_FLOAT32_C(  -303.23), SIMDE_FLOAT32_C(  -844.84), SIMDE_FLOAT32_C(   452.16)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -516.27), SIMDE_FLOAT32_C(   125.72), SIMDE_FLOAT32_C(    14.67), SIMDE_FLOAT32_C(    11.14),
                         SIMDE_FLOAT32_C(   874.74), SIMDE_FLOAT32_C(  -961.20), SIMDE_FLOAT32_C(   -12.54), SIMDE_FLOAT32_C(   560.55),
                         SIMDE_FLOAT32_C(   -59.35), SIMDE_FLOAT32_C(  -247.72), SIMDE_FLOAT32_C(  -719.76), SIMDE_FLOAT32_C(   280.49),
                         SIMDE_FLOAT32_C(  -337.54), SIMDE_FLOAT32_C(  -303.23), SIMDE_FLOAT32_C(  -229.00), SIMDE_FLOAT32_C(  -133.06)) },
    { UINT16_C( 5687),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -252.71), SIMDE_FLOAT32_C(   314.80), SIMDE_FLOAT32_C(   998.07), SIMDE_FLOAT32_C(   575.47),
                         SIMDE_FLOAT32_C(    52.24), SIMDE_FLOAT32_C(   443.69), SIMDE_FLOAT32_C(   234.67), SIMDE_FLOAT32_C(   894.50),
                         SIMDE_FLOAT32_C(  -487.55), SIMDE_FLOAT32_C(  -250.47), SIMDE_FLOAT32_C(  -500.78), SIMDE_FLOAT32_C(   379.78),
                         SIMDE_FLOAT32_C(   612.27), SIMDE_FLOAT32_C(  -899.55), SIMDE_FLOAT32_C(  -426.63), SIMDE_FLOAT32_C(   359.38)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -262.03), SIMDE_FLOAT32_C(    81.51), SIMDE_FLOAT32_C(  -484.02), SIMDE_FLOAT32_C(   504.24),
                         SIMDE_FLOAT32_C(   157.78), SIMDE_FLOAT32_C(    62.09), SIMDE_FLOAT32_C(   193.22), SIMDE_FLOAT32_C(   -72.99),
                         SIMDE_FLOAT32_C(   -54.28), SIMDE_FLOAT32_C(  -361.95), SIMDE_FLOAT32_C(   726.76), SIMDE_FLOAT32_C(   663.18),
                         SIMDE_FLOAT32_C(   290.10), SIMDE_FLOAT32_C(   238.07), SIMDE_FLOAT32_C(  -777.03), SIMDE_FLOAT32_C(   227.93)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -252.71), SIMDE_FLOAT32_C(   314.80), SIMDE_FLOAT32_C(   998.07), SIMDE_FLOAT32_C(   504.24),
                         SIMDE_FLOAT32_C(    52.24), SIMDE_FLOAT32_C(    62.09), SIMDE_FLOAT32_C(   193.22), SIMDE_FLOAT32_C(   894.50),
                         SIMDE_FLOAT32_C(  -487.55), SIMDE_FLOAT32_C(  -250.47), SIMDE_FLOAT32_C(   726.76), SIMDE_FLOAT32_C(   663.18),
                         SIMDE_FLOAT32_C(   612.27), SIMDE_FLOAT32_C(   238.07), SIMDE_FLOAT32_C(  -777.03), SIMDE_FLOAT32_C(   227.93)) },
    { UINT16_C(46817),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    60.97), SIMDE_FLOAT32_C(   294.39), SIMDE_FLOAT32_C(   134.62), SIMDE_FLOAT32_C(  -712.88),
                         SIMDE_FLOAT32_C(  -527.46), SIMDE_FLOAT32_C(   556.17), SIMDE_FLOAT32_C(   671.08), SIMDE_FLOAT32_C(  -149.99),
                         SIMDE_FLOAT32_C(  -247.28), SIMDE_FLOAT32_C(  -221.96), SIMDE_FLOAT32_C(  -789.76), SIMDE_FLOAT32_C(   665.60),
                         SIMDE_FLOAT32_C(   590.64), SIMDE_FLOAT32_C(  -335.79), SIMDE_FLOAT32_C(  -964.26), SIMDE_FLOAT32_C(   137.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   741.48), SIMDE_FLOAT32_C(  -567.93), SIMDE_FLOAT32_C(   572.38), SIMDE_FLOAT32_C(  -363.96),
                         SIMDE_FLOAT32_C(   267.49), SIMDE_FLOAT32_C(   878.76), SIMDE_FLOAT32_C(  -445.04), SIMDE_FLOAT32_C(  -252.24),
                         SIMDE_FLOAT32_C(   869.52), SIMDE_FLOAT32_C(  -356.95), SIMDE_FLOAT32_C(   128.10), SIMDE_FLOAT32_C(   544.58),
                         SIMDE_FLOAT32_C(   242.93), SIMDE_FLOAT32_C(  -919.16), SIMDE_FLOAT32_C(   758.90), SIMDE_FLOAT32_C(  -466.75)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   741.48), SIMDE_FLOAT32_C(   294.39), SIMDE_FLOAT32_C(   572.38), SIMDE_FLOAT32_C(  -363.96),
                         SIMDE_FLOAT32_C(  -527.46), SIMDE_FLOAT32_C(   878.76), SIMDE_FLOAT32_C(  -445.04), SIMDE_FLOAT32_C(  -149.99),
                         SIMDE_FLOAT32_C(   869.52), SIMDE_FLOAT32_C(  -356.95), SIMDE_FLOAT32_C(   128.10), SIMDE_FLOAT32_C(   665.60),
                         SIMDE_FLOAT32_C(   590.64), SIMDE_FLOAT32_C(  -335.79), SIMDE_FLOAT32_C(  -964.26), SIMDE_FLOAT32_C(  -466.75)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_blend_ps(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_blend_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512d a;
    simde__m512d b;
    simde__m512d r;
  } test_vec[8] = {
    { UINT8_C(211),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  863.27), SIMDE_FLOAT64_C( -937.53),
                         SIMDE_FLOAT64_C(  272.85), SIMDE_FLOAT64_C( -836.56),
                         SIMDE_FLOAT64_C( -517.71), SIMDE_FLOAT64_C(  436.89),
                         SIMDE_FLOAT64_C( -561.62), SIMDE_FLOAT64_C( -796.29)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  356.91), SIMDE_FLOAT64_C( -250.13),
                         SIMDE_FLOAT64_C( -609.99), SIMDE_FLOAT64_C( -756.57),
                         SIMDE_FLOAT64_C( -441.01), SIMDE_FLOAT64_C(  675.23),
                         SIMDE_FLOAT64_C( -112.56), SIMDE_FLOAT64_C(  752.66)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  356.91), SIMDE_FLOAT64_C( -250.13),
                         SIMDE_FLOAT64_C(  272.85), SIMDE_FLOAT64_C( -756.57),
                         SIMDE_FLOAT64_C( -517.71), SIMDE_FLOAT64_C(  436.89),
                         SIMDE_FLOAT64_C( -112.56), SIMDE_FLOAT64_C(  752.66)) },
    { UINT8_C( 25),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  721.67), SIMDE_FLOAT64_C( -355.29),
                         SIMDE_FLOAT64_C( -776.65), SIMDE_FLOAT64_C( -467.78),
                         SIMDE_FLOAT64_C( -890.68), SIMDE_FLOAT64_C( -288.13),
                         SIMDE_FLOAT64_C(  739.88), SIMDE_FLOAT64_C(   -3.67)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -963.76), SIMDE_FLOAT64_C(  986.64),
                         SIMDE_FLOAT64_C(  121.90), SIMDE_FLOAT64_C( -796.62),
                         SIMDE_FLOAT64_C(  983.17), SIMDE_FLOAT64_C(  569.02),
                         SIMDE_FLOAT64_C(  -88.58), SIMDE_FLOAT64_C( -750.53)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  721.67), SIMDE_FLOAT64_C( -355.29),
                         SIMDE_FLOAT64_C( -776.65), SIMDE_FLOAT64_C( -796.62),
                         SIMDE_FLOAT64_C(  983.17), SIMDE_FLOAT64_C( -288.13),
                         SIMDE_FLOAT64_C(  739.88), SIMDE_FLOAT64_C( -750.53)) },
    { UINT8_C( 46),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -217.74), SIMDE_FLOAT64_C(  162.91),
                         SIMDE_FLOAT64_C(  837.05), SIMDE_FLOAT64_C(  132.83),
                         SIMDE_FLOAT64_C( -183.75), SIMDE_FLOAT64_C( -958.98),
                         SIMDE_FLOAT64_C( -343.18), SIMDE_FLOAT64_C( -412.04)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -43.99), SIMDE_FLOAT64_C(  645.49),
                         SIMDE_FLOAT64_C( -301.76), SIMDE_FLOAT64_C( -390.74),
                         SIMDE_FLOAT64_C(  671.11), SIMDE_FLOAT64_C( -513.10),
                         SIMDE_FLOAT64_C(  467.15), SIMDE_FLOAT64_C( -961.27)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -217.74), SIMDE_FLOAT64_C(  162.91),
                         SIMDE_FLOAT64_C( -301.76), SIMDE_FLOAT64_C(  132.83),
                         SIMDE_FLOAT64_C(  671.11), SIMDE_FLOAT64_C( -513.10),
                         SIMDE_FLOAT64_C(  467.15), SIMDE_FLOAT64_C( -412.04)) },
    { UINT8_C(180),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -623.32), SIMDE_FLOAT64_C(  372.80),
                         SIMDE_FLOAT64_C( -286.04), SIMDE_FLOAT64_C(  347.55),
                         SIMDE_FLOAT64_C( -954.70), SIMDE_FLOAT64_C(  272.86),
                         SIMDE_FLOAT64_C(  787.91), SIMDE_FLOAT64_C(  529.75)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  153.63), SIMDE_FLOAT64_C( -177.79),
                         SIMDE_FLOAT64_C(  957.03), SIMDE_FLOAT64_C( -193.15),
                         SIMDE_FLOAT64_C(  645.09), SIMDE_FLOAT64_C(   -0.96),
                         SIMDE_FLOAT64_C(   66.15), SIMDE_FLOAT64_C(  565.09)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  153.63), SIMDE_FLOAT64_C(  372.80),
                         SIMDE_FLOAT64_C(  957.03), SIMDE_FLOAT64_C( -193.15),
                         SIMDE_FLOAT64_C( -954.70), SIMDE_FLOAT64_C(   -0.96),
                         SIMDE_FLOAT64_C(  787.91), SIMDE_FLOAT64_C(  529.75)) },
    { UINT8_C(125),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  705.45), SIMDE_FLOAT64_C(  -55.08),
                         SIMDE_FLOAT64_C(  606.63), SIMDE_FLOAT64_C(  -13.01),
                         SIMDE_FLOAT64_C(  483.39), SIMDE_FLOAT64_C(  565.56),
                         SIMDE_FLOAT64_C(  735.84), SIMDE_FLOAT64_C( -855.34)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -898.20), SIMDE_FLOAT64_C( -885.41),
                         SIMDE_FLOAT64_C( -446.34), SIMDE_FLOAT64_C( -539.23),
                         SIMDE_FLOAT64_C(  177.56), SIMDE_FLOAT64_C(   85.90),
                         SIMDE_FLOAT64_C( -977.71), SIMDE_FLOAT64_C(  589.01)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  705.45), SIMDE_FLOAT64_C( -885.41),
                         SIMDE_FLOAT64_C( -446.34), SIMDE_FLOAT64_C( -539.23),
                         SIMDE_FLOAT64_C(  177.56), SIMDE_FLOAT64_C(   85.90),
                         SIMDE_FLOAT64_C(  735.84), SIMDE_FLOAT64_C(  589.01)) },
    { UINT8_C(188),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  263.46), SIMDE_FLOAT64_C( -173.99),
                         SIMDE_FLOAT64_C( -448.77), SIMDE_FLOAT64_C(  -18.46),
                         SIMDE_FLOAT64_C(   60.12), SIMDE_FLOAT64_C(  895.07),
                         SIMDE_FLOAT64_C(  593.75), SIMDE_FLOAT64_C( -910.69)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -347.65), SIMDE_FLOAT64_C(  986.49),
                         SIMDE_FLOAT64_C(  854.73), SIMDE_FLOAT64_C(  459.72),
                         SIMDE_FLOAT64_C( -110.83), SIMDE_FLOAT64_C( -875.29),
                         SIMDE_FLOAT64_C(  594.30), SIMDE_FLOAT64_C( -331.55)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -347.65), SIMDE_FLOAT64_C( -173.99),
                         SIMDE_FLOAT64_C(  854.73), SIMDE_FLOAT64_C(  459.72),
                         SIMDE_FLOAT64_C( -110.83), SIMDE_FLOAT64_C( -875.29),
                         SIMDE_FLOAT64_C(  593.75), SIMDE_FLOAT64_C( -910.69)) },
    { UINT8_C(190),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   93.71), SIMDE_FLOAT64_C( -510.90),
                         SIMDE_FLOAT64_C(  589.53), SIMDE_FLOAT64_C(  762.37),
                         SIMDE_FLOAT64_C(  -64.95), SIMDE_FLOAT64_C( -200.72),
                         SIMDE_FLOAT64_C(  590.31), SIMDE_FLOAT64_C(  904.10)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -356.41), SIMDE_FLOAT64_C(  536.26),
                         SIMDE_FLOAT64_C(   46.41), SIMDE_FLOAT64_C(  968.69),
                         SIMDE_FLOAT64_C(  819.71), SIMDE_FLOAT64_C( -256.62),
                         SIMDE_FLOAT64_C( -508.11), SIMDE_FLOAT64_C(  806.88)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -356.41), SIMDE_FLOAT64_C( -510.90),
                         SIMDE_FLOAT64_C(   46.41), SIMDE_FLOAT64_C(  968.69),
                         SIMDE_FLOAT64_C(  819.71), SIMDE_FLOAT64_C( -256.62),
                         SIMDE_FLOAT64_C( -508.11), SIMDE_FLOAT64_C(  904.10)) },
    { UINT8_C(178),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  932.33), SIMDE_FLOAT64_C( -948.81),
                         SIMDE_FLOAT64_C( -663.23), SIMDE_FLOAT64_C( -224.49),
                         SIMDE_FLOAT64_C(   61.61), SIMDE_FLOAT64_C( -513.25),
                         SIMDE_FLOAT64_C(  862.87), SIMDE_FLOAT64_C(  888.55)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  178.62), SIMDE_FLOAT64_C( -833.77),
                         SIMDE_FLOAT64_C( -382.41), SIMDE_FLOAT64_C(  155.80),
                         SIMDE_FLOAT64_C( -175.51), SIMDE_FLOAT64_C(  270.94),
                         SIMDE_FLOAT64_C( -347.54), SIMDE_FLOAT64_C( -823.80)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  178.62), SIMDE_FLOAT64_C( -948.81),
                         SIMDE_FLOAT64_C( -382.41), SIMDE_FLOAT64_C(  155.80),
                         SIMDE_FLOAT64_C(   61.61), SIMDE_FLOAT64_C( -513.25),
                         SIMDE_FLOAT64_C( -347.54), SIMDE_FLOAT64_C(  888.55)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_blend_pd(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_broadcast_f32x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   241.63), SIMDE_FLOAT32_C(   962.32), SIMDE_FLOAT32_C(  -223.53), SIMDE_FLOAT32_C(  -221.69)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   241.63), SIMDE_FLOAT32_C(   962.32), SIMDE_FLOAT32_C(  -223.53), SIMDE_FLOAT32_C(  -221.69),
                         SIMDE_FLOAT32_C(   241.63), SIMDE_FLOAT32_C(   962.32), SIMDE_FLOAT32_C(  -223.53), SIMDE_FLOAT32_C(  -221.69),
                         SIMDE_FLOAT32_C(   241.63), SIMDE_FLOAT32_C(   962.32), SIMDE_FLOAT32_C(  -223.53), SIMDE_FLOAT32_C(  -221.69),
                         SIMDE_FLOAT32_C(   241.63), SIMDE_FLOAT32_C(   962.32), SIMDE_FLOAT32_C(  -223.53), SIMDE_FLOAT32_C(  -221.69)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   115.71), SIMDE_FLOAT32_C(  -206.04), SIMDE_FLOAT32_C(  -581.48), SIMDE_FLOAT32_C(   670.36)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   115.71), SIMDE_FLOAT32_C(  -206.04), SIMDE_FLOAT32_C(  -581.48), SIMDE_FLOAT32_C(   670.36),
                         SIMDE_FLOAT32_C(   115.71), SIMDE_FLOAT32_C(  -206.04), SIMDE_FLOAT32_C(  -581.48), SIMDE_FLOAT32_C(   670.36),
                         SIMDE_FLOAT32_C(   115.71), SIMDE_FLOAT32_C(  -206.04), SIMDE_FLOAT32_C(  -581.48), SIMDE_FLOAT32_C(   670.36),
                         SIMDE_FLOAT32_C(   115.71), SIMDE_FLOAT32_C(  -206.04), SIMDE_FLOAT32_C(  -581.48), SIMDE_FLOAT32_C(   670.36)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   297.45), SIMDE_FLOAT32_C(   193.39), SIMDE_FLOAT32_C(  -163.24), SIMDE_FLOAT32_C(  -775.87)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   297.45), SIMDE_FLOAT32_C(   193.39), SIMDE_FLOAT32_C(  -163.24), SIMDE_FLOAT32_C(  -775.87),
                         SIMDE_FLOAT32_C(   297.45), SIMDE_FLOAT32_C(   193.39), SIMDE_FLOAT32_C(  -163.24), SIMDE_FLOAT32_C(  -775.87),
                         SIMDE_FLOAT32_C(   297.45), SIMDE_FLOAT32_C(   193.39), SIMDE_FLOAT32_C(  -163.24), SIMDE_FLOAT32_C(  -775.87),
                         SIMDE_FLOAT32_C(   297.45), SIMDE_FLOAT32_C(   193.39), SIMDE_FLOAT32_C(  -163.24), SIMDE_FLOAT32_C(  -775.87)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -675.37), SIMDE_FLOAT32_C(   853.20), SIMDE_FLOAT32_C(  -377.67), SIMDE_FLOAT32_C(   233.14)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -675.37), SIMDE_FLOAT32_C(   853.20), SIMDE_FLOAT32_C(  -377.67), SIMDE_FLOAT32_C(   233.14),
                         SIMDE_FLOAT32_C(  -675.37), SIMDE_FLOAT32_C(   853.20), SIMDE_FLOAT32_C(  -377.67), SIMDE_FLOAT32_C(   233.14),
                         SIMDE_FLOAT32_C(  -675.37), SIMDE_FLOAT32_C(   853.20), SIMDE_FLOAT32_C(  -377.67), SIMDE_FLOAT32_C(   233.14),
                         SIMDE_FLOAT32_C(  -675.37), SIMDE_FLOAT32_C(   853.20), SIMDE_FLOAT32_C(  -377.67), SIMDE_FLOAT32_C(   233.14)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -156.08), SIMDE_FLOAT32_C(  -209.26), SIMDE_FLOAT32_C(    48.51), SIMDE_FLOAT32_C(  -627.76)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -156.08), SIMDE_FLOAT32_C(  -209.26), SIMDE_FLOAT32_C(    48.51), SIMDE_FLOAT32_C(  -627.76),
                         SIMDE_FLOAT32_C(  -156.08), SIMDE_FLOAT32_C(  -209.26), SIMDE_FLOAT32_C(    48.51), SIMDE_FLOAT32_C(  -627.76),
                         SIMDE_FLOAT32_C(  -156.08), SIMDE_FLOAT32_C(  -209.26), SIMDE_FLOAT32_C(    48.51), SIMDE_FLOAT32_C(  -627.76),
                         SIMDE_FLOAT32_C(  -156.08), SIMDE_FLOAT32_C(  -209.26), SIMDE_FLOAT32_C(    48.51), SIMDE_FLOAT32_C(  -627.76)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   108.40), SIMDE_FLOAT32_C(   970.37), SIMDE_FLOAT32_C(   934.72), SIMDE_FLOAT32_C(  -932.81)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   108.40), SIMDE_FLOAT32_C(   970.37), SIMDE_FLOAT32_C(   934.72), SIMDE_FLOAT32_C(  -932.81),
                         SIMDE_FLOAT32_C(   108.40), SIMDE_FLOAT32_C(   970.37), SIMDE_FLOAT32_C(   934.72), SIMDE_FLOAT32_C(  -932.81),
                         SIMDE_FLOAT32_C(   108.40), SIMDE_FLOAT32_C(   970.37), SIMDE_FLOAT32_C(   934.72), SIMDE_FLOAT32_C(  -932.81),
                         SIMDE_FLOAT32_C(   108.40), SIMDE_FLOAT32_C(   970.37), SIMDE_FLOAT32_C(   934.72), SIMDE_FLOAT32_C(  -932.81)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   690.58), SIMDE_FLOAT32_C(   836.42), SIMDE_FLOAT32_C(  -952.66), SIMDE_FLOAT32_C(    22.35)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   690.58), SIMDE_FLOAT32_C(   836.42), SIMDE_FLOAT32_C(  -952.66), SIMDE_FLOAT32_C(    22.35),
                         SIMDE_FLOAT32_C(   690.58), SIMDE_FLOAT32_C(   836.42), SIMDE_FLOAT32_C(  -952.66), SIMDE_FLOAT32_C(    22.35),
                         SIMDE_FLOAT32_C(   690.58), SIMDE_FLOAT32_C(   836.42), SIMDE_FLOAT32_C(  -952.66), SIMDE_FLOAT32_C(    22.35),
                         SIMDE_FLOAT32_C(   690.58), SIMDE_FLOAT32_C(   836.42), SIMDE_FLOAT32_C(  -952.66), SIMDE_FLOAT32_C(    22.35)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   740.10), SIMDE_FLOAT32_C(   159.65), SIMDE_FLOAT32_C(   -65.49), SIMDE_FLOAT32_C(   946.83)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   740.10), SIMDE_FLOAT32_C(   159.65), SIMDE_FLOAT32_C(   -65.49), SIMDE_FLOAT32_C(   946.83),
                         SIMDE_FLOAT32_C(   740.10), SIMDE_FLOAT32_C(   159.65), SIMDE_FLOAT32_C(   -65.49), SIMDE_FLOAT32_C(   946.83),
                         SIMDE_FLOAT32_C(   740.10), SIMDE_FLOAT32_C(   159.65), SIMDE_FLOAT32_C(   -65.49), SIMDE_FLOAT32_C(   946.83),
                         SIMDE_FLOAT32_C(   740.10), SIMDE_FLOAT32_C(   159.65), SIMDE_FLOAT32_C(   -65.49), SIMDE_FLOAT32_C(   946.83)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_broadcast_f32x4(test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcast_f32x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -476.82), SIMDE_FLOAT32_C(   687.27), SIMDE_FLOAT32_C(   239.12), SIMDE_FLOAT32_C(  -622.96),
                         SIMDE_FLOAT32_C(   479.82), SIMDE_FLOAT32_C(  -652.18), SIMDE_FLOAT32_C(   585.66), SIMDE_FLOAT32_C(  -840.39),
                         SIMDE_FLOAT32_C(  -680.47), SIMDE_FLOAT32_C(  -211.69), SIMDE_FLOAT32_C(   879.50), SIMDE_FLOAT32_C(   245.88),
                         SIMDE_FLOAT32_C(   689.68), SIMDE_FLOAT32_C(   107.64), SIMDE_FLOAT32_C(  -872.56), SIMDE_FLOAT32_C(  -586.10)),
      UINT16_C(63721),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   337.98), SIMDE_FLOAT32_C(  -931.30), SIMDE_FLOAT32_C(   -93.71), SIMDE_FLOAT32_C(   492.43)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   337.98), SIMDE_FLOAT32_C(  -931.30), SIMDE_FLOAT32_C(   -93.71), SIMDE_FLOAT32_C(   492.43),
                         SIMDE_FLOAT32_C(   337.98), SIMDE_FLOAT32_C(  -652.18), SIMDE_FLOAT32_C(   585.66), SIMDE_FLOAT32_C(  -840.39),
                         SIMDE_FLOAT32_C(   337.98), SIMDE_FLOAT32_C(  -931.30), SIMDE_FLOAT32_C(   -93.71), SIMDE_FLOAT32_C(   245.88),
                         SIMDE_FLOAT32_C(   337.98), SIMDE_FLOAT32_C(   107.64), SIMDE_FLOAT32_C(  -872.56), SIMDE_FLOAT32_C(   492.43)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   792.34), SIMDE_FLOAT32_C(  -828.98), SIMDE_FLOAT32_C(   152.82), SIMDE_FLOAT32_C(   261.49),
                         SIMDE_FLOAT32_C(  -674.96), SIMDE_FLOAT32_C(  -626.70), SIMDE_FLOAT32_C(  -365.50), SIMDE_FLOAT32_C(   522.39),
                         SIMDE_FLOAT32_C(   659.15), SIMDE_FLOAT32_C(   204.13), SIMDE_FLOAT32_C(   487.20), SIMDE_FLOAT32_C(   790.92),
                         SIMDE_FLOAT32_C(  -372.23), SIMDE_FLOAT32_C(  -362.18), SIMDE_FLOAT32_C(   725.62), SIMDE_FLOAT32_C(   817.00)),
      UINT16_C(44067),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -858.39), SIMDE_FLOAT32_C(   608.18), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(  -779.98)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -858.39), SIMDE_FLOAT32_C(  -828.98), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(   261.49),
                         SIMDE_FLOAT32_C(  -858.39), SIMDE_FLOAT32_C(   608.18), SIMDE_FLOAT32_C(  -365.50), SIMDE_FLOAT32_C(   522.39),
                         SIMDE_FLOAT32_C(   659.15), SIMDE_FLOAT32_C(   204.13), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(   790.92),
                         SIMDE_FLOAT32_C(  -372.23), SIMDE_FLOAT32_C(  -362.18), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(  -779.98)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   433.05), SIMDE_FLOAT32_C(   405.44), SIMDE_FLOAT32_C(   652.04), SIMDE_FLOAT32_C(  -453.75),
                         SIMDE_FLOAT32_C(    56.24), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(  -127.57), SIMDE_FLOAT32_C(  -230.83),
                         SIMDE_FLOAT32_C(  -815.89), SIMDE_FLOAT32_C(   351.22), SIMDE_FLOAT32_C(  -739.81), SIMDE_FLOAT32_C(  -104.33),
                         SIMDE_FLOAT32_C(   331.38), SIMDE_FLOAT32_C(   749.42), SIMDE_FLOAT32_C(   151.95), SIMDE_FLOAT32_C(   -25.90)),
      UINT16_C(12331),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -159.95), SIMDE_FLOAT32_C(  -519.57), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -690.93)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   433.05), SIMDE_FLOAT32_C(   405.44), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -690.93),
                         SIMDE_FLOAT32_C(    56.24), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(  -127.57), SIMDE_FLOAT32_C(  -230.83),
                         SIMDE_FLOAT32_C(  -815.89), SIMDE_FLOAT32_C(   351.22), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -104.33),
                         SIMDE_FLOAT32_C(  -159.95), SIMDE_FLOAT32_C(   749.42), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -690.93)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   417.50), SIMDE_FLOAT32_C(   245.21), SIMDE_FLOAT32_C(   960.01), SIMDE_FLOAT32_C(  -303.61),
                         SIMDE_FLOAT32_C(  -550.57), SIMDE_FLOAT32_C(   665.98), SIMDE_FLOAT32_C(  -521.00), SIMDE_FLOAT32_C(   239.39),
                         SIMDE_FLOAT32_C(   798.32), SIMDE_FLOAT32_C(   251.37), SIMDE_FLOAT32_C(  -596.78), SIMDE_FLOAT32_C(   840.69),
                         SIMDE_FLOAT32_C(  -684.92), SIMDE_FLOAT32_C(    87.08), SIMDE_FLOAT32_C(   734.84), SIMDE_FLOAT32_C(  -854.89)),
      UINT16_C(52021),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -116.62), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(   229.99), SIMDE_FLOAT32_C(  -771.72)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -116.62), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(   960.01), SIMDE_FLOAT32_C(  -303.61),
                         SIMDE_FLOAT32_C(  -116.62), SIMDE_FLOAT32_C(   665.98), SIMDE_FLOAT32_C(   229.99), SIMDE_FLOAT32_C(  -771.72),
                         SIMDE_FLOAT32_C(   798.32), SIMDE_FLOAT32_C(   251.37), SIMDE_FLOAT32_C(   229.99), SIMDE_FLOAT32_C(  -771.72),
                         SIMDE_FLOAT32_C(  -684.92), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(   734.84), SIMDE_FLOAT32_C(  -771.72)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -616.45), SIMDE_FLOAT32_C(   914.70), SIMDE_FLOAT32_C(  -963.67), SIMDE_FLOAT32_C(  -935.61),
                         SIMDE_FLOAT32_C(   106.52), SIMDE_FLOAT32_C(   367.48), SIMDE_FLOAT32_C(   -10.30), SIMDE_FLOAT32_C(   543.55),
                         SIMDE_FLOAT32_C(   142.17), SIMDE_FLOAT32_C(  -844.51), SIMDE_FLOAT32_C(  -959.58), SIMDE_FLOAT32_C(   913.58),
                         SIMDE_FLOAT32_C(  -227.61), SIMDE_FLOAT32_C(  -979.09), SIMDE_FLOAT32_C(  -746.95), SIMDE_FLOAT32_C(   363.67)),
      UINT16_C(46395),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -319.95), SIMDE_FLOAT32_C(  -241.48), SIMDE_FLOAT32_C(  -416.05), SIMDE_FLOAT32_C(  -700.83)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -319.95), SIMDE_FLOAT32_C(   914.70), SIMDE_FLOAT32_C(  -416.05), SIMDE_FLOAT32_C(  -700.83),
                         SIMDE_FLOAT32_C(   106.52), SIMDE_FLOAT32_C(  -241.48), SIMDE_FLOAT32_C(   -10.30), SIMDE_FLOAT32_C(  -700.83),
                         SIMDE_FLOAT32_C(   142.17), SIMDE_FLOAT32_C(  -844.51), SIMDE_FLOAT32_C(  -416.05), SIMDE_FLOAT32_C(  -700.83),
                         SIMDE_FLOAT32_C(  -319.95), SIMDE_FLOAT32_C(  -979.09), SIMDE_FLOAT32_C(  -416.05), SIMDE_FLOAT32_C(  -700.83)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   810.23), SIMDE_FLOAT32_C(  -571.66), SIMDE_FLOAT32_C(  -313.94), SIMDE_FLOAT32_C(   812.08),
                         SIMDE_FLOAT32_C(   905.89), SIMDE_FLOAT32_C(    95.84), SIMDE_FLOAT32_C(  -942.64), SIMDE_FLOAT32_C(   490.95),
                         SIMDE_FLOAT32_C(   432.01), SIMDE_FLOAT32_C(  -989.57), SIMDE_FLOAT32_C(  -908.07), SIMDE_FLOAT32_C(   843.06),
                         SIMDE_FLOAT32_C(  -567.12), SIMDE_FLOAT32_C(   561.55), SIMDE_FLOAT32_C(  -316.58), SIMDE_FLOAT32_C(  -224.94)),
      UINT16_C(28510),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   608.47), SIMDE_FLOAT32_C(   502.71), SIMDE_FLOAT32_C(   524.73), SIMDE_FLOAT32_C(  -206.66)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   810.23), SIMDE_FLOAT32_C(   502.71), SIMDE_FLOAT32_C(   524.73), SIMDE_FLOAT32_C(   812.08),
                         SIMDE_FLOAT32_C(   608.47), SIMDE_FLOAT32_C(   502.71), SIMDE_FLOAT32_C(   524.73), SIMDE_FLOAT32_C(  -206.66),
                         SIMDE_FLOAT32_C(   432.01), SIMDE_FLOAT32_C(   502.71), SIMDE_FLOAT32_C(  -908.07), SIMDE_FLOAT32_C(  -206.66),
                         SIMDE_FLOAT32_C(   608.47), SIMDE_FLOAT32_C(   502.71), SIMDE_FLOAT32_C(   524.73), SIMDE_FLOAT32_C(  -224.94)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -770.27), SIMDE_FLOAT32_C(  -598.61), SIMDE_FLOAT32_C(   672.88), SIMDE_FLOAT32_C(  -504.06),
                         SIMDE_FLOAT32_C(   481.78), SIMDE_FLOAT32_C(  -154.88), SIMDE_FLOAT32_C(  -363.51), SIMDE_FLOAT32_C(  -643.93),
                         SIMDE_FLOAT32_C(  -973.84), SIMDE_FLOAT32_C(  -599.20), SIMDE_FLOAT32_C(   230.44), SIMDE_FLOAT32_C(  -713.35),
                         SIMDE_FLOAT32_C(  -554.88), SIMDE_FLOAT32_C(  -858.98), SIMDE_FLOAT32_C(   -21.09), SIMDE_FLOAT32_C(  -441.11)),
      UINT16_C( 6749),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   687.60), SIMDE_FLOAT32_C(   681.66), SIMDE_FLOAT32_C(  -362.35), SIMDE_FLOAT32_C(  -482.20)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -770.27), SIMDE_FLOAT32_C(  -598.61), SIMDE_FLOAT32_C(   672.88), SIMDE_FLOAT32_C(  -482.20),
                         SIMDE_FLOAT32_C(   687.60), SIMDE_FLOAT32_C(  -154.88), SIMDE_FLOAT32_C(  -362.35), SIMDE_FLOAT32_C(  -643.93),
                         SIMDE_FLOAT32_C(  -973.84), SIMDE_FLOAT32_C(   681.66), SIMDE_FLOAT32_C(   230.44), SIMDE_FLOAT32_C(  -482.20),
                         SIMDE_FLOAT32_C(   687.60), SIMDE_FLOAT32_C(   681.66), SIMDE_FLOAT32_C(   -21.09), SIMDE_FLOAT32_C(  -482.20)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -454.36), SIMDE_FLOAT32_C(  -172.69), SIMDE_FLOAT32_C(   256.23), SIMDE_FLOAT32_C(   682.27),
                         SIMDE_FLOAT32_C(   -43.91), SIMDE_FLOAT32_C(  -300.48), SIMDE_FLOAT32_C(   916.93), SIMDE_FLOAT32_C(  -592.77),
                         SIMDE_FLOAT32_C(   939.83), SIMDE_FLOAT32_C(  -553.88), SIMDE_FLOAT32_C(  -796.09), SIMDE_FLOAT32_C(  -515.91),
                         SIMDE_FLOAT32_C(   623.85), SIMDE_FLOAT32_C(   359.37), SIMDE_FLOAT32_C(  -557.79), SIMDE_FLOAT32_C(   595.65)),
      UINT16_C( 8287),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -705.53), SIMDE_FLOAT32_C(   238.42), SIMDE_FLOAT32_C(   504.37), SIMDE_FLOAT32_C(   296.48)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -454.36), SIMDE_FLOAT32_C(  -172.69), SIMDE_FLOAT32_C(   504.37), SIMDE_FLOAT32_C(   682.27),
                         SIMDE_FLOAT32_C(   -43.91), SIMDE_FLOAT32_C(  -300.48), SIMDE_FLOAT32_C(   916.93), SIMDE_FLOAT32_C(  -592.77),
                         SIMDE_FLOAT32_C(   939.83), SIMDE_FLOAT32_C(   238.42), SIMDE_FLOAT32_C(  -796.09), SIMDE_FLOAT32_C(   296.48),
                         SIMDE_FLOAT32_C(  -705.53), SIMDE_FLOAT32_C(   238.42), SIMDE_FLOAT32_C(   504.37), SIMDE_FLOAT32_C(   296.48)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_broadcast_f32x4(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcast_f32x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { UINT16_C(12860),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   -93.71), SIMDE_FLOAT32_C(   137.99), SIMDE_FLOAT32_C(   492.43), SIMDE_FLOAT32_C(   420.83)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   492.43), SIMDE_FLOAT32_C(   420.83),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   492.43), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   492.43), SIMDE_FLOAT32_C(   420.83),
                         SIMDE_FLOAT32_C(   -93.71), SIMDE_FLOAT32_C(   137.99), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(63770),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -652.18), SIMDE_FLOAT32_C(  -872.56), SIMDE_FLOAT32_C(   585.66), SIMDE_FLOAT32_C(  -586.10)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -652.18), SIMDE_FLOAT32_C(  -872.56), SIMDE_FLOAT32_C(   585.66), SIMDE_FLOAT32_C(  -586.10),
                         SIMDE_FLOAT32_C(  -652.18), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -586.10),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -586.10),
                         SIMDE_FLOAT32_C(  -652.18), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   585.66), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(26030),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   700.39), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(   708.98), SIMDE_FLOAT32_C(  -779.98)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(   708.98), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -779.98),
                         SIMDE_FLOAT32_C(   700.39), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   708.98), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   700.39), SIMDE_FLOAT32_C(   129.78), SIMDE_FLOAT32_C(   708.98), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(41122),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -362.18), SIMDE_FLOAT32_C(  -626.70), SIMDE_FLOAT32_C(   725.62), SIMDE_FLOAT32_C(  -365.50)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -362.18), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   725.62), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -362.18), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   725.62), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   725.62), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(49851),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -519.57), SIMDE_FLOAT32_C(  -632.83), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -181.94)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -519.57), SIMDE_FLOAT32_C(  -632.83), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -519.57), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -181.94),
                         SIMDE_FLOAT32_C(  -519.57), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -66.62), SIMDE_FLOAT32_C(  -181.94)) },
    { UINT16_C(41826),
      simde_mm_set_ps(SIMDE_FLOAT32_C(    56.24), SIMDE_FLOAT32_C(   749.42), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(   151.95)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(    56.24), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(   151.95),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   749.42), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   506.86), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(19285),
      simde_mm_set_ps(SIMDE_FLOAT32_C(    57.37), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(   347.13), SIMDE_FLOAT32_C(   229.99)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(    57.37), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   347.13), SIMDE_FLOAT32_C(   229.99),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   229.99),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   -17.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   229.99)) },
    { UINT16_C(48133),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -684.92), SIMDE_FLOAT32_C(  -550.57), SIMDE_FLOAT32_C(    87.08), SIMDE_FLOAT32_C(   665.98)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -684.92), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(    87.08), SIMDE_FLOAT32_C(   665.98),
                         SIMDE_FLOAT32_C(  -684.92), SIMDE_FLOAT32_C(  -550.57), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -550.57), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   665.98)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_maskz_broadcast_f32x4(test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_broadcast_f64x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  241.63), SIMDE_FLOAT64_C(  962.32),
                         SIMDE_FLOAT64_C( -223.53), SIMDE_FLOAT64_C( -221.69)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  241.63), SIMDE_FLOAT64_C(  962.32),
                         SIMDE_FLOAT64_C( -223.53), SIMDE_FLOAT64_C( -221.69),
                         SIMDE_FLOAT64_C(  241.63), SIMDE_FLOAT64_C(  962.32),
                         SIMDE_FLOAT64_C( -223.53), SIMDE_FLOAT64_C( -221.69)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  115.71), SIMDE_FLOAT64_C( -206.04),
                         SIMDE_FLOAT64_C( -581.48), SIMDE_FLOAT64_C(  670.36)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  115.71), SIMDE_FLOAT64_C( -206.04),
                         SIMDE_FLOAT64_C( -581.48), SIMDE_FLOAT64_C(  670.36),
                         SIMDE_FLOAT64_C(  115.71), SIMDE_FLOAT64_C( -206.04),
                         SIMDE_FLOAT64_C( -581.48), SIMDE_FLOAT64_C(  670.36)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  297.45), SIMDE_FLOAT64_C(  193.39),
                         SIMDE_FLOAT64_C( -163.24), SIMDE_FLOAT64_C( -775.87)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  297.45), SIMDE_FLOAT64_C(  193.39),
                         SIMDE_FLOAT64_C( -163.24), SIMDE_FLOAT64_C( -775.87),
                         SIMDE_FLOAT64_C(  297.45), SIMDE_FLOAT64_C(  193.39),
                         SIMDE_FLOAT64_C( -163.24), SIMDE_FLOAT64_C( -775.87)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -675.37), SIMDE_FLOAT64_C(  853.20),
                         SIMDE_FLOAT64_C( -377.67), SIMDE_FLOAT64_C(  233.14)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -675.37), SIMDE_FLOAT64_C(  853.20),
                         SIMDE_FLOAT64_C( -377.67), SIMDE_FLOAT64_C(  233.14),
                         SIMDE_FLOAT64_C( -675.37), SIMDE_FLOAT64_C(  853.20),
                         SIMDE_FLOAT64_C( -377.67), SIMDE_FLOAT64_C(  233.14)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -156.08), SIMDE_FLOAT64_C( -209.26),
                         SIMDE_FLOAT64_C(   48.51), SIMDE_FLOAT64_C( -627.76)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -156.08), SIMDE_FLOAT64_C( -209.26),
                         SIMDE_FLOAT64_C(   48.51), SIMDE_FLOAT64_C( -627.76),
                         SIMDE_FLOAT64_C( -156.08), SIMDE_FLOAT64_C( -209.26),
                         SIMDE_FLOAT64_C(   48.51), SIMDE_FLOAT64_C( -627.76)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  108.40), SIMDE_FLOAT64_C(  970.37),
                         SIMDE_FLOAT64_C(  934.72), SIMDE_FLOAT64_C( -932.81)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  108.40), SIMDE_FLOAT64_C(  970.37),
                         SIMDE_FLOAT64_C(  934.72), SIMDE_FLOAT64_C( -932.81),
                         SIMDE_FLOAT64_C(  108.40), SIMDE_FLOAT64_C(  970.37),
                         SIMDE_FLOAT64_C(  934.72), SIMDE_FLOAT64_C( -932.81)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  690.58), SIMDE_FLOAT64_C(  836.42),
                         SIMDE_FLOAT64_C( -952.66), SIMDE_FLOAT64_C(   22.35)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  690.58), SIMDE_FLOAT64_C(  836.42),
                         SIMDE_FLOAT64_C( -952.66), SIMDE_FLOAT64_C(   22.35),
                         SIMDE_FLOAT64_C(  690.58), SIMDE_FLOAT64_C(  836.42),
                         SIMDE_FLOAT64_C( -952.66), SIMDE_FLOAT64_C(   22.35)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  740.10), SIMDE_FLOAT64_C(  159.65),
                         SIMDE_FLOAT64_C(  -65.49), SIMDE_FLOAT64_C(  946.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  740.10), SIMDE_FLOAT64_C(  159.65),
                         SIMDE_FLOAT64_C(  -65.49), SIMDE_FLOAT64_C(  946.83),
                         SIMDE_FLOAT64_C(  740.10), SIMDE_FLOAT64_C(  159.65),
                         SIMDE_FLOAT64_C(  -65.49), SIMDE_FLOAT64_C(  946.83)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_broadcast_f64x4(test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcast_f64x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m256d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -396.88), SIMDE_FLOAT64_C(  354.04),
                         SIMDE_FLOAT64_C(  268.06), SIMDE_FLOAT64_C( -972.10),
                         SIMDE_FLOAT64_C( -213.85), SIMDE_FLOAT64_C( -574.68),
                         SIMDE_FLOAT64_C(  137.99), SIMDE_FLOAT64_C(  420.83)),
      UINT8_C( 60),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  337.98), SIMDE_FLOAT64_C( -931.30),
                         SIMDE_FLOAT64_C(  -93.71), SIMDE_FLOAT64_C(  492.43)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -396.88), SIMDE_FLOAT64_C(  354.04),
                         SIMDE_FLOAT64_C(  -93.71), SIMDE_FLOAT64_C(  492.43),
                         SIMDE_FLOAT64_C(  337.98), SIMDE_FLOAT64_C( -931.30),
                         SIMDE_FLOAT64_C(  137.99), SIMDE_FLOAT64_C(  420.83)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -680.47), SIMDE_FLOAT64_C( -211.69),
                         SIMDE_FLOAT64_C(  879.50), SIMDE_FLOAT64_C(  245.88),
                         SIMDE_FLOAT64_C(  689.68), SIMDE_FLOAT64_C(  107.64),
                         SIMDE_FLOAT64_C( -872.56), SIMDE_FLOAT64_C( -586.10)),
      UINT8_C( 26),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -622.96), SIMDE_FLOAT64_C(  479.82),
                         SIMDE_FLOAT64_C( -652.18), SIMDE_FLOAT64_C(  585.66)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -680.47), SIMDE_FLOAT64_C( -211.69),
                         SIMDE_FLOAT64_C(  879.50), SIMDE_FLOAT64_C(  585.66),
                         SIMDE_FLOAT64_C( -622.96), SIMDE_FLOAT64_C(  107.64),
                         SIMDE_FLOAT64_C( -652.18), SIMDE_FLOAT64_C( -586.10)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  408.02), SIMDE_FLOAT64_C(  662.99),
                         SIMDE_FLOAT64_C( -491.44), SIMDE_FLOAT64_C( -586.97),
                         SIMDE_FLOAT64_C( -858.39), SIMDE_FLOAT64_C(  608.18),
                         SIMDE_FLOAT64_C(  129.78), SIMDE_FLOAT64_C( -779.98)),
      UINT8_C(174),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  469.58), SIMDE_FLOAT64_C( -229.18),
                         SIMDE_FLOAT64_C(  700.39), SIMDE_FLOAT64_C(  708.98)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  469.58), SIMDE_FLOAT64_C(  662.99),
                         SIMDE_FLOAT64_C(  700.39), SIMDE_FLOAT64_C( -586.97),
                         SIMDE_FLOAT64_C(  469.58), SIMDE_FLOAT64_C( -229.18),
                         SIMDE_FLOAT64_C(  700.39), SIMDE_FLOAT64_C( -779.98)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -491.65), SIMDE_FLOAT64_C(  792.34),
                         SIMDE_FLOAT64_C( -828.98), SIMDE_FLOAT64_C(  152.82),
                         SIMDE_FLOAT64_C(  261.49), SIMDE_FLOAT64_C( -674.96),
                         SIMDE_FLOAT64_C( -626.70), SIMDE_FLOAT64_C( -365.50)),
      UINT8_C(162),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  790.92), SIMDE_FLOAT64_C( -372.23),
                         SIMDE_FLOAT64_C( -362.18), SIMDE_FLOAT64_C(  725.62)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  790.92), SIMDE_FLOAT64_C(  792.34),
                         SIMDE_FLOAT64_C( -362.18), SIMDE_FLOAT64_C(  152.82),
                         SIMDE_FLOAT64_C(  261.49), SIMDE_FLOAT64_C( -674.96),
                         SIMDE_FLOAT64_C( -362.18), SIMDE_FLOAT64_C( -365.50)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -230.83), SIMDE_FLOAT64_C( -480.11),
                         SIMDE_FLOAT64_C(  511.94), SIMDE_FLOAT64_C(  614.74),
                         SIMDE_FLOAT64_C(  794.95), SIMDE_FLOAT64_C( -331.37),
                         SIMDE_FLOAT64_C( -632.83), SIMDE_FLOAT64_C( -181.94)),
      UINT8_C(187),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  125.71), SIMDE_FLOAT64_C( -159.95),
                         SIMDE_FLOAT64_C( -519.57), SIMDE_FLOAT64_C(  -66.62)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  125.71), SIMDE_FLOAT64_C( -480.11),
                         SIMDE_FLOAT64_C( -519.57), SIMDE_FLOAT64_C(  -66.62),
                         SIMDE_FLOAT64_C(  125.71), SIMDE_FLOAT64_C( -331.37),
                         SIMDE_FLOAT64_C( -519.57), SIMDE_FLOAT64_C(  -66.62)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  870.57), SIMDE_FLOAT64_C( -815.89),
                         SIMDE_FLOAT64_C(  351.22), SIMDE_FLOAT64_C( -739.81),
                         SIMDE_FLOAT64_C( -104.33), SIMDE_FLOAT64_C(  331.38),
                         SIMDE_FLOAT64_C(  749.42), SIMDE_FLOAT64_C(  151.95)),
      UINT8_C( 98),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  652.04), SIMDE_FLOAT64_C( -453.75),
                         SIMDE_FLOAT64_C(   56.24), SIMDE_FLOAT64_C(  506.86)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  870.57), SIMDE_FLOAT64_C( -453.75),
                         SIMDE_FLOAT64_C(   56.24), SIMDE_FLOAT64_C( -739.81),
                         SIMDE_FLOAT64_C( -104.33), SIMDE_FLOAT64_C(  331.38),
                         SIMDE_FLOAT64_C(   56.24), SIMDE_FLOAT64_C(  151.95)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -854.89), SIMDE_FLOAT64_C(  376.16),
                         SIMDE_FLOAT64_C( -846.26), SIMDE_FLOAT64_C(  817.65),
                         SIMDE_FLOAT64_C( -403.95), SIMDE_FLOAT64_C( -116.62),
                         SIMDE_FLOAT64_C(  -17.97), SIMDE_FLOAT64_C(  229.99)),
      UINT8_C( 85),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  -65.83), SIMDE_FLOAT64_C( -494.87),
                         SIMDE_FLOAT64_C(   57.37), SIMDE_FLOAT64_C(  347.13)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -854.89), SIMDE_FLOAT64_C( -494.87),
                         SIMDE_FLOAT64_C( -846.26), SIMDE_FLOAT64_C(  347.13),
                         SIMDE_FLOAT64_C( -403.95), SIMDE_FLOAT64_C( -494.87),
                         SIMDE_FLOAT64_C(  -17.97), SIMDE_FLOAT64_C(  347.13)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -700.83), SIMDE_FLOAT64_C( -289.50),
                         SIMDE_FLOAT64_C(  417.50), SIMDE_FLOAT64_C(  245.21),
                         SIMDE_FLOAT64_C(  960.01), SIMDE_FLOAT64_C( -303.61),
                         SIMDE_FLOAT64_C( -550.57), SIMDE_FLOAT64_C(  665.98)),
      UINT8_C(  5),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -596.78), SIMDE_FLOAT64_C(  840.69),
                         SIMDE_FLOAT64_C( -684.92), SIMDE_FLOAT64_C(   87.08)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -700.83), SIMDE_FLOAT64_C( -289.50),
                         SIMDE_FLOAT64_C(  417.50), SIMDE_FLOAT64_C(  245.21),
                         SIMDE_FLOAT64_C(  960.01), SIMDE_FLOAT64_C(  840.69),
                         SIMDE_FLOAT64_C( -550.57), SIMDE_FLOAT64_C(   87.08)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_broadcast_f64x4(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcast_f64x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m256d a;
    simde__m512d r;
  } test_vec[8] = {
    { UINT8_C( 25),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  -93.71), SIMDE_FLOAT64_C(  137.99),
                         SIMDE_FLOAT64_C(  492.43), SIMDE_FLOAT64_C(  420.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  420.83),
                         SIMDE_FLOAT64_C(  -93.71), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  420.83)) },
    { UINT8_C(223),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  354.04), SIMDE_FLOAT64_C( -261.67),
                         SIMDE_FLOAT64_C(  268.06), SIMDE_FLOAT64_C(  648.56)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  354.04), SIMDE_FLOAT64_C( -261.67),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  648.56),
                         SIMDE_FLOAT64_C(  354.04), SIMDE_FLOAT64_C( -261.67),
                         SIMDE_FLOAT64_C(  268.06), SIMDE_FLOAT64_C(  648.56)) },
    { UINT8_C(191),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  107.64), SIMDE_FLOAT64_C( -652.18),
                         SIMDE_FLOAT64_C( -872.56), SIMDE_FLOAT64_C(  585.66)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  107.64), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -872.56), SIMDE_FLOAT64_C(  585.66),
                         SIMDE_FLOAT64_C(  107.64), SIMDE_FLOAT64_C( -652.18),
                         SIMDE_FLOAT64_C( -872.56), SIMDE_FLOAT64_C(  585.66)) },
    { UINT8_C( 77),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -476.82), SIMDE_FLOAT64_C( -211.69),
                         SIMDE_FLOAT64_C(  687.27), SIMDE_FLOAT64_C(  879.50)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -211.69),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -476.82), SIMDE_FLOAT64_C( -211.69),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  879.50)) },
    { UINT8_C(216),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -229.18), SIMDE_FLOAT64_C(  608.18),
                         SIMDE_FLOAT64_C(  700.39), SIMDE_FLOAT64_C(  129.78)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -229.18), SIMDE_FLOAT64_C(  608.18),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  129.78),
                         SIMDE_FLOAT64_C( -229.18), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(196),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  408.02), SIMDE_FLOAT64_C( -213.85),
                         SIMDE_FLOAT64_C(  662.99), SIMDE_FLOAT64_C(  346.52)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  408.02), SIMDE_FLOAT64_C( -213.85),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -213.85),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(125),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  261.49), SIMDE_FLOAT64_C( -372.23),
                         SIMDE_FLOAT64_C( -674.96), SIMDE_FLOAT64_C( -362.18)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -372.23),
                         SIMDE_FLOAT64_C( -674.96), SIMDE_FLOAT64_C( -362.18),
                         SIMDE_FLOAT64_C(  261.49), SIMDE_FLOAT64_C( -372.23),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -362.18)) },
    { UINT8_C( 95),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  475.10), SIMDE_FLOAT64_C( -491.65),
                         SIMDE_FLOAT64_C(  659.15), SIMDE_FLOAT64_C(  792.34)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -491.65),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  792.34),
                         SIMDE_FLOAT64_C(  475.10), SIMDE_FLOAT64_C( -491.65),
                         SIMDE_FLOAT64_C(  659.15), SIMDE_FLOAT64_C(  792.34)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_maskz_broadcast_f64x4(test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_broadcast_i32x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi32(INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853)),
      simde_mm512_set_epi32(INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853),
                            INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853),
                            INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853),
                            INT32_C( 1322912216), INT32_C( -192131569), INT32_C(  457247766), INT32_C( 1585478853)) },
    { simde_mm_set_epi32(INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165)),
      simde_mm512_set_epi32(INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165),
                            INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165),
                            INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165),
                            INT32_C(  455358584), INT32_C( -549958328), INT32_C( 1779282555), INT32_C(-1938144165)) },
    { simde_mm_set_epi32(INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792)),
      simde_mm512_set_epi32(INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792),
                            INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792),
                            INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792),
                            INT32_C(   35244693), INT32_C( -163894097), INT32_C(  -32854349), INT32_C(-1300832792)) },
    { simde_mm_set_epi32(INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300)),
      simde_mm512_set_epi32(INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300),
                            INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300),
                            INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300),
                            INT32_C( 1137728540), INT32_C( 1602744474), INT32_C( -610393021), INT32_C(-1810116300)) },
    { simde_mm_set_epi32(INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562)),
      simde_mm512_set_epi32(INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562),
                            INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562),
                            INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562),
                            INT32_C(-1023450780), INT32_C(  840494259), INT32_C(-1087383364), INT32_C(-1604779562)) },
    { simde_mm_set_epi32(INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438)),
      simde_mm512_set_epi32(INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438),
                            INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438),
                            INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438),
                            INT32_C( 1284866833), INT32_C(   27132707), INT32_C(-1597877982), INT32_C(-1252321438)) },
    { simde_mm_set_epi32(INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347)),
      simde_mm512_set_epi32(INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347),
                            INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347),
                            INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347),
                            INT32_C( -165954025), INT32_C(  878840386), INT32_C( -802596544), INT32_C( 1574139347)) },
    { simde_mm_set_epi32(INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769)),
      simde_mm512_set_epi32(INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769),
                            INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769),
                            INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769),
                            INT32_C( -602275056), INT32_C(-1823359312), INT32_C( 1232365699), INT32_C(  345237769)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcast_i32x4(test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcast_i32x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1479802474), INT32_C(  587294539), INT32_C( -174751528), INT32_C( 1465222154),
                            INT32_C( 1625882140), INT32_C(-1283973275), INT32_C(  567394727), INT32_C( 1808136008),
                            INT32_C(  324921956), INT32_C(-1888780980), INT32_C( -262803011), INT32_C( 2131227345),
                            INT32_C( -161180317), INT32_C( -569391310), INT32_C(  471197581), INT32_C( 2029664703)),
      UINT16_C(12686),
      simde_mm_set_epi32(INT32_C(-1929654026), INT32_C(-1217014324), INT32_C(  230292224), INT32_C( 1361651453)),
      simde_mm512_set_epi32(INT32_C( 1479802474), INT32_C(  587294539), INT32_C(  230292224), INT32_C( 1361651453),
                            INT32_C( 1625882140), INT32_C(-1283973275), INT32_C(  567394727), INT32_C( 1361651453),
                            INT32_C(-1929654026), INT32_C(-1888780980), INT32_C( -262803011), INT32_C( 2131227345),
                            INT32_C(-1929654026), INT32_C(-1217014324), INT32_C(  230292224), INT32_C( 2029664703)) },
    { simde_mm512_set_epi32(INT32_C( 1958214116), INT32_C( 2124258263), INT32_C(-1603442041), INT32_C(-1137458903),
                            INT32_C( -291704812), INT32_C( -523349105), INT32_C( -769676631), INT32_C(  359038153),
                            INT32_C( -860324016), INT32_C(  142975746), INT32_C( 1871956670), INT32_C(-2122929741),
                            INT32_C( 1007202856), INT32_C(-1693638626), INT32_C(-1497430440), INT32_C(  766142674)),
      UINT16_C( 3460),
      simde_mm_set_epi32(INT32_C(-1801778632), INT32_C(  793094568), INT32_C(  739597071), INT32_C( 1855829690)),
      simde_mm512_set_epi32(INT32_C( 1958214116), INT32_C( 2124258263), INT32_C(-1603442041), INT32_C(-1137458903),
                            INT32_C(-1801778632), INT32_C(  793094568), INT32_C( -769676631), INT32_C( 1855829690),
                            INT32_C(-1801778632), INT32_C(  142975746), INT32_C( 1871956670), INT32_C(-2122929741),
                            INT32_C( 1007202856), INT32_C(  793094568), INT32_C(-1497430440), INT32_C(  766142674)) },
    { simde_mm512_set_epi32(INT32_C( -491998875), INT32_C( -465346847), INT32_C( 1096008422), INT32_C( -151618100),
                            INT32_C( -483382033), INT32_C(-1500806456), INT32_C(  175505846), INT32_C( -698441328),
                            INT32_C( -515513970), INT32_C( 1679973349), INT32_C(-1523347194), INT32_C(   91392241),
                            INT32_C( -561919749), INT32_C( -634254878), INT32_C( -625316172), INT32_C(  -17019235)),
      UINT16_C(25030),
      simde_mm_set_epi32(INT32_C( -839244820), INT32_C(-1678825378), INT32_C(  464598558), INT32_C(-1198702193)),
      simde_mm512_set_epi32(INT32_C( -491998875), INT32_C(-1678825378), INT32_C(  464598558), INT32_C( -151618100),
                            INT32_C( -483382033), INT32_C(-1500806456), INT32_C(  175505846), INT32_C(-1198702193),
                            INT32_C( -839244820), INT32_C(-1678825378), INT32_C(-1523347194), INT32_C(   91392241),
                            INT32_C( -561919749), INT32_C(-1678825378), INT32_C(  464598558), INT32_C(  -17019235)) },
    { simde_mm512_set_epi32(INT32_C( 1319681857), INT32_C(  649867282), INT32_C(-1955467744), INT32_C(-1687114005),
                            INT32_C(-1950655074), INT32_C(-2040429697), INT32_C( 1764915437), INT32_C(  813475409),
                            INT32_C(-1622276195), INT32_C(  614665853), INT32_C( -661145222), INT32_C(  -43416876),
                            INT32_C(  954392932), INT32_C(-1003825870), INT32_C( -858676034), INT32_C( 1589986539)),
      UINT16_C(29308),
      simde_mm_set_epi32(INT32_C(-1945617369), INT32_C( -313192838), INT32_C( -614227976), INT32_C(  -73637500)),
      simde_mm512_set_epi32(INT32_C( 1319681857), INT32_C( -313192838), INT32_C( -614227976), INT32_C(  -73637500),
                            INT32_C(-1950655074), INT32_C(-2040429697), INT32_C( -614227976), INT32_C(  813475409),
                            INT32_C(-1622276195), INT32_C( -313192838), INT32_C( -614227976), INT32_C(  -73637500),
                            INT32_C(-1945617369), INT32_C( -313192838), INT32_C( -858676034), INT32_C( 1589986539)) },
    { simde_mm512_set_epi32(INT32_C(  482652005), INT32_C( 1083073699), INT32_C( -547163888), INT32_C(-1439583577),
                            INT32_C( -836573741), INT32_C(-2032318592), INT32_C( 1307381638), INT32_C( 2027662416),
                            INT32_C( 2001285861), INT32_C( 1074543972), INT32_C(-2107097596), INT32_C(-2025611729),
                            INT32_C(  962055101), INT32_C( 1886777199), INT32_C( 1689643613), INT32_C(-1874481648)),
      UINT16_C(45428),
      simde_mm_set_epi32(INT32_C(  110278011), INT32_C(-1940227644), INT32_C(-1803195700), INT32_C( 1287862649)),
      simde_mm512_set_epi32(INT32_C(  110278011), INT32_C( 1083073699), INT32_C(-1803195700), INT32_C( 1287862649),
                            INT32_C( -836573741), INT32_C(-2032318592), INT32_C( 1307381638), INT32_C( 1287862649),
                            INT32_C( 2001285861), INT32_C(-1940227644), INT32_C(-1803195700), INT32_C( 1287862649),
                            INT32_C(  962055101), INT32_C(-1940227644), INT32_C( 1689643613), INT32_C(-1874481648)) },
    { simde_mm512_set_epi32(INT32_C(  485695865), INT32_C( 1704586743), INT32_C(-1227241134), INT32_C(  279727823),
                            INT32_C( -480355834), INT32_C( 1374909005), INT32_C(-1706379633), INT32_C( 1300025155),
                            INT32_C( 1901096153), INT32_C(-1845297076), INT32_C(  188971064), INT32_C( 1903842318),
                            INT32_C(-1221674473), INT32_C(-1332164211), INT32_C(   23564349), INT32_C(-2098316192)),
      UINT16_C(21964),
      simde_mm_set_epi32(INT32_C(-1820692848), INT32_C( -830585945), INT32_C( 1667959054), INT32_C(-1758734041)),
      simde_mm512_set_epi32(INT32_C(  485695865), INT32_C( -830585945), INT32_C(-1227241134), INT32_C(-1758734041),
                            INT32_C( -480355834), INT32_C( -830585945), INT32_C(-1706379633), INT32_C(-1758734041),
                            INT32_C(-1820692848), INT32_C( -830585945), INT32_C(  188971064), INT32_C( 1903842318),
                            INT32_C(-1820692848), INT32_C( -830585945), INT32_C(   23564349), INT32_C(-2098316192)) },
    { simde_mm512_set_epi32(INT32_C(-1876069406), INT32_C( 1820341222), INT32_C(  987166931), INT32_C(-1021572249),
                            INT32_C(-1046533173), INT32_C(-1808511518), INT32_C( -283777637), INT32_C( -168486656),
                            INT32_C( 1250903497), INT32_C( 1175614584), INT32_C(  204391673), INT32_C( -667659280),
                            INT32_C( 2035348040), INT32_C( -596829354), INT32_C(-1607289004), INT32_C( -670488239)),
      UINT16_C(31159),
      simde_mm_set_epi32(INT32_C(-1492076939), INT32_C( 1502879171), INT32_C( 1497885207), INT32_C(-1325620059)),
      simde_mm512_set_epi32(INT32_C(-1876069406), INT32_C( 1502879171), INT32_C( 1497885207), INT32_C(-1325620059),
                            INT32_C(-1492076939), INT32_C(-1808511518), INT32_C( -283777637), INT32_C(-1325620059),
                            INT32_C(-1492076939), INT32_C( 1175614584), INT32_C( 1497885207), INT32_C(-1325620059),
                            INT32_C( 2035348040), INT32_C( 1502879171), INT32_C( 1497885207), INT32_C(-1325620059)) },
    { simde_mm512_set_epi32(INT32_C(-1346174896), INT32_C( 1223712250), INT32_C( 2029339086), INT32_C( 2108949315),
                            INT32_C(-1822742445), INT32_C( -343433299), INT32_C(-1626119528), INT32_C( 1735301543),
                            INT32_C(  766111295), INT32_C(  -80424103), INT32_C( 1232059506), INT32_C(-1681875170),
                            INT32_C( 1819208351), INT32_C( -734074357), INT32_C(   61937468), INT32_C(-1403575087)),
      UINT16_C(37926),
      simde_mm_set_epi32(INT32_C( 1656599178), INT32_C( 1293315993), INT32_C( -728433677), INT32_C( -125533424)),
      simde_mm512_set_epi32(INT32_C( 1656599178), INT32_C( 1223712250), INT32_C( 2029339086), INT32_C( -125533424),
                            INT32_C(-1822742445), INT32_C( 1293315993), INT32_C(-1626119528), INT32_C( 1735301543),
                            INT32_C(  766111295), INT32_C(  -80424103), INT32_C( -728433677), INT32_C(-1681875170),
                            INT32_C( 1819208351), INT32_C( 1293315993), INT32_C( -728433677), INT32_C(-1403575087)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_broadcast_i32x4(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcast_i32x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(57503),
      simde_mm_set_epi32(INT32_C(  913371223), INT32_C( 1946242675), INT32_C(-1851162974), INT32_C(-1090004303)),
      simde_mm512_set_epi32(INT32_C(  913371223), INT32_C( 1946242675), INT32_C(-1851162974), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  913371223), INT32_C(          0), INT32_C(          0), INT32_C(-1090004303),
                            INT32_C(  913371223), INT32_C( 1946242675), INT32_C(-1851162974), INT32_C(-1090004303)) },
    { UINT16_C( 9830),
      simde_mm_set_epi32(INT32_C( -754702866), INT32_C(   59910169), INT32_C(-1421684089), INT32_C( 1688249563)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(-1421684089), INT32_C(          0),
                            INT32_C(          0), INT32_C(   59910169), INT32_C(-1421684089), INT32_C(          0),
                            INT32_C(          0), INT32_C(   59910169), INT32_C(-1421684089), INT32_C(          0),
                            INT32_C(          0), INT32_C(   59910169), INT32_C(-1421684089), INT32_C(          0)) },
    { UINT16_C(54973),
      simde_mm_set_epi32(INT32_C( 1295192258), INT32_C( 2064350366), INT32_C(-1387191485), INT32_C( 1585557386)),
      simde_mm512_set_epi32(INT32_C( 1295192258), INT32_C( 2064350366), INT32_C(          0), INT32_C( 1585557386),
                            INT32_C(          0), INT32_C( 2064350366), INT32_C(-1387191485), INT32_C(          0),
                            INT32_C( 1295192258), INT32_C(          0), INT32_C(-1387191485), INT32_C( 1585557386),
                            INT32_C( 1295192258), INT32_C( 2064350366), INT32_C(          0), INT32_C( 1585557386)) },
    { UINT16_C( 2571),
      simde_mm_set_epi32(INT32_C(  273665101), INT32_C( -889778981), INT32_C(  888851167), INT32_C(  342766140)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  273665101), INT32_C(          0), INT32_C(  888851167), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  273665101), INT32_C(          0), INT32_C(  888851167), INT32_C(  342766140)) },
    { UINT16_C(34156),
      simde_mm_set_epi32(INT32_C(  809684493), INT32_C( -666403540), INT32_C(-1117073828), INT32_C(-1916337185)),
      simde_mm512_set_epi32(INT32_C(  809684493), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( -666403540), INT32_C(          0), INT32_C(-1916337185),
                            INT32_C(          0), INT32_C( -666403540), INT32_C(-1117073828), INT32_C(          0),
                            INT32_C(  809684493), INT32_C( -666403540), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C( 6544),
      simde_mm_set_epi32(INT32_C( 1692879261), INT32_C( -671588299), INT32_C( -258764942), INT32_C(-1633977409)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(-1633977409),
                            INT32_C( 1692879261), INT32_C(          0), INT32_C(          0), INT32_C(-1633977409),
                            INT32_C( 1692879261), INT32_C(          0), INT32_C(          0), INT32_C(-1633977409),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(45909),
      simde_mm_set_epi32(INT32_C(  472486650), INT32_C( 1238366490), INT32_C(-1084360471), INT32_C(  686181072)),
      simde_mm512_set_epi32(INT32_C(  472486650), INT32_C(          0), INT32_C(-1084360471), INT32_C(  686181072),
                            INT32_C(          0), INT32_C(          0), INT32_C(-1084360471), INT32_C(  686181072),
                            INT32_C(          0), INT32_C( 1238366490), INT32_C(          0), INT32_C(  686181072),
                            INT32_C(          0), INT32_C( 1238366490), INT32_C(          0), INT32_C(  686181072)) },
    { UINT16_C(56653),
      simde_mm_set_epi32(INT32_C( 1655322598), INT32_C( -841418169), INT32_C( -643403227), INT32_C(-1868778842)),
      simde_mm512_set_epi32(INT32_C( 1655322598), INT32_C( -841418169), INT32_C(          0), INT32_C(-1868778842),
                            INT32_C( 1655322598), INT32_C( -841418169), INT32_C(          0), INT32_C(-1868778842),
                            INT32_C(          0), INT32_C( -841418169), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1655322598), INT32_C( -841418169), INT32_C(          0), INT32_C(-1868778842)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_broadcast_i32x4(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_broadcast_i64x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm256_set_epi64x(INT64_C( 2067253863170152603), INT64_C( 7322969156688688496),
                             INT64_C(-3040413397780943697), INT64_C( -347515311309491350)),
      simde_mm512_set_epi64(INT64_C( 2067253863170152603), INT64_C( 7322969156688688496),
                            INT64_C(-3040413397780943697), INT64_C( -347515311309491350),
                            INT64_C( 2067253863170152603), INT64_C( 7322969156688688496),
                            INT64_C(-3040413397780943697), INT64_C( -347515311309491350)) },
    { simde_mm256_set_epi64x(INT64_C(-8775907405261856642), INT64_C( 2994184764454707691),
                             INT64_C( 5740004668815682638), INT64_C(-6479861669953478300)),
      simde_mm512_set_epi64(INT64_C(-8775907405261856642), INT64_C( 2994184764454707691),
                            INT64_C( 5740004668815682638), INT64_C(-6479861669953478300),
                            INT64_C(-8775907405261856642), INT64_C( 2994184764454707691),
                            INT64_C( 5740004668815682638), INT64_C(-6479861669953478300)) },
    { simde_mm256_set_epi64x(INT64_C(-1508734178901937051), INT64_C(-9017252864562564261),
                             INT64_C( -273279204292504060), INT64_C(  619750219118375084)),
      simde_mm512_set_epi64(INT64_C(-1508734178901937051), INT64_C(-9017252864562564261),
                            INT64_C( -273279204292504060), INT64_C(  619750219118375084),
                            INT64_C(-1508734178901937051), INT64_C(-9017252864562564261),
                            INT64_C( -273279204292504060), INT64_C(  619750219118375084)) },
    { simde_mm256_set_epi64x(INT64_C( 5726987144774798582), INT64_C(-5242976599564634972),
                             INT64_C(-2397121704692329659), INT64_C( 8619348224440898856)),
      simde_mm512_set_epi64(INT64_C( 5726987144774798582), INT64_C(-5242976599564634972),
                            INT64_C(-2397121704692329659), INT64_C( 8619348224440898856),
                            INT64_C( 5726987144774798582), INT64_C(-5242976599564634972),
                            INT64_C(-2397121704692329659), INT64_C( 8619348224440898856)) },
    { simde_mm256_set_epi64x(INT64_C( 3770039990400590046), INT64_C(-4228023324121815234),
                             INT64_C(-2554402032947045809), INT64_C(-5734730006803594733)),
      simde_mm512_set_epi64(INT64_C( 3770039990400590046), INT64_C(-4228023324121815234),
                            INT64_C(-2554402032947045809), INT64_C(-5734730006803594733),
                            INT64_C( 3770039990400590046), INT64_C(-4228023324121815234),
                            INT64_C(-2554402032947045809), INT64_C(-5734730006803594733)) },
    { simde_mm256_set_epi64x(INT64_C(-7969300362390541280), INT64_C( 5131273406597805369),
                             INT64_C( 3164578103377175393), INT64_C( -896289702737256643)),
      simde_mm512_set_epi64(INT64_C(-7969300362390541280), INT64_C( 5131273406597805369),
                            INT64_C( 3164578103377175393), INT64_C( -896289702737256643),
                            INT64_C(-7969300362390541280), INT64_C( 5131273406597805369),
                            INT64_C( 3164578103377175393), INT64_C( -896289702737256643)) },
    { simde_mm256_set_epi64x(INT64_C( 6358202424481672256), INT64_C(-2088789378195753898),
                             INT64_C(-3832720361616382569), INT64_C(-1395499602347228816)),
      simde_mm512_set_epi64(INT64_C( 6358202424481672256), INT64_C(-2088789378195753898),
                            INT64_C(-3832720361616382569), INT64_C(-1395499602347228816),
                            INT64_C( 6358202424481672256), INT64_C(-2088789378195753898),
                            INT64_C(-3832720361616382569), INT64_C(-1395499602347228816)) },
    { simde_mm256_set_epi64x(INT64_C(-7005415045902450329), INT64_C(  454800303112400674),
                             INT64_C(  120562593220559221), INT64_C(-9183341893829321065)),
      simde_mm512_set_epi64(INT64_C(-7005415045902450329), INT64_C(  454800303112400674),
                            INT64_C(  120562593220559221), INT64_C(-9183341893829321065),
                            INT64_C(-7005415045902450329), INT64_C(  454800303112400674),
                            INT64_C(  120562593220559221), INT64_C(-9183341893829321065)) },
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcast_i64x4(test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcast_i64x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m256i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-6314317108894035774), INT64_C( 8866317312363406147),
                            INT64_C( 6809917121524389565), INT64_C(-3241424127607560167),
                            INT64_C(-6106086665810303781), INT64_C(  633642393017577559),
                            INT64_C( 8359048641648361122), INT64_C(-4681532830833057633)),
      UINT8_C( 60),
      simde_mm256_set_epi64x(INT64_C( 3477568421141904684), INT64_C(-4797795556098898977),
                             INT64_C( 3208117008747973709), INT64_C(-3821571623174354209)),
      simde_mm512_set_epi64(INT64_C(-6314317108894035774), INT64_C( 8866317312363406147),
                            INT64_C( 3208117008747973709), INT64_C(-3821571623174354209),
                            INT64_C( 3477568421141904684), INT64_C(-4797795556098898977),
                            INT64_C( 8359048641648361122), INT64_C(-4681532830833057633)) },
    { simde_mm512_set_epi64(INT64_C( 1306125493676423142), INT64_C(-3613863514463636955),
                            INT64_C(-8026344006176744115), INT64_C( 2029314710784964890),
                            INT64_C(-4657292759333975344), INT64_C( 4825522705097247133),
                            INT64_C(-2884449776545067150), INT64_C(-7017879531382302320)),
      UINT8_C(  0),
      simde_mm256_set_epi64x(INT64_C(-4405193415265233332), INT64_C( 7250935849068321562),
                             INT64_C(-6027293339582699304), INT64_C(-3733599027822978693)),
      simde_mm512_set_epi64(INT64_C( 1306125493676423142), INT64_C(-3613863514463636955),
                            INT64_C(-8026344006176744115), INT64_C( 2029314710784964890),
                            INT64_C(-4657292759333975344), INT64_C( 4825522705097247133),
                            INT64_C(-2884449776545067150), INT64_C(-7017879531382302320)) },
    { simde_mm512_set_epi64(INT64_C( 4688717956956220153), INT64_C(-1915316091557446787),
                            INT64_C( 1577347929723399506), INT64_C(-7813885322626023749),
                            INT64_C(-6811547529988353683), INT64_C( 2997984888778655645),
                            INT64_C( 3443124806434765346), INT64_C( 5852240145563215278)),
      UINT8_C(246),
      simde_mm256_set_epi64x(INT64_C(-1891210360757244537), INT64_C( 6167039147883013727),
                             INT64_C( 3386552444698298512), INT64_C( 7545310155849572514)),
      simde_mm512_set_epi64(INT64_C(-1891210360757244537), INT64_C( 6167039147883013727),
                            INT64_C( 3386552444698298512), INT64_C( 7545310155849572514),
                            INT64_C(-6811547529988353683), INT64_C( 6167039147883013727),
                            INT64_C( 3386552444698298512), INT64_C( 5852240145563215278)) },
    { simde_mm512_set_epi64(INT64_C( 5038277295705077786), INT64_C(-8704670477732479640),
                            INT64_C(-4548397220420700343), INT64_C( 8046739269734052975),
                            INT64_C( 7094379553694909752), INT64_C( 4795143479989329521),
                            INT64_C(-4501545483124413586), INT64_C(-3553418787378740418)),
      UINT8_C( 91),
      simde_mm256_set_epi64x(INT64_C(-1193819960890806229), INT64_C( 1698145641448748604),
                             INT64_C(-5983907472113043464), INT64_C( 2399871967268573321)),
      simde_mm512_set_epi64(INT64_C( 5038277295705077786), INT64_C( 1698145641448748604),
                            INT64_C(-4548397220420700343), INT64_C( 2399871967268573321),
                            INT64_C(-1193819960890806229), INT64_C( 4795143479989329521),
                            INT64_C(-5983907472113043464), INT64_C( 2399871967268573321)) },
    { simde_mm512_set_epi64(INT64_C(-7015430497800685262), INT64_C( 6395476272833483099),
                            INT64_C(-7658177893206805688), INT64_C( 8616202346974378134),
                            INT64_C( 4658965153462790469), INT64_C(-8694270525310808014),
                            INT64_C(-6021620893121233714), INT64_C(-2734912706905093379)),
      UINT8_C(200),
      simde_mm256_set_epi64x(INT64_C(-1469383970610000896), INT64_C( 2906056864364420569),
                             INT64_C(-8420208282727167471), INT64_C(-2445653243165948933)),
      simde_mm512_set_epi64(INT64_C(-1469383970610000896), INT64_C( 2906056864364420569),
                            INT64_C(-7658177893206805688), INT64_C( 8616202346974378134),
                            INT64_C(-1469383970610000896), INT64_C(-8694270525310808014),
                            INT64_C(-6021620893121233714), INT64_C(-2734912706905093379)) },
    { simde_mm512_set_epi64(INT64_C(  -14573144697473529), INT64_C( 8194534140513027918),
                            INT64_C( 2864848388614962181), INT64_C(-8899252041456864412),
                            INT64_C( 6379752944219310901), INT64_C(-1860193003353627344),
                            INT64_C(-6904865090556452860), INT64_C( 3719036040063860682)),
      UINT8_C(242),
      simde_mm256_set_epi64x(INT64_C(-5869124324801971655), INT64_C( 4548184433513821860),
                             INT64_C( -866976878921007676), INT64_C( 2203520398864570966)),
      simde_mm512_set_epi64(INT64_C(-5869124324801971655), INT64_C( 4548184433513821860),
                            INT64_C( -866976878921007676), INT64_C( 2203520398864570966),
                            INT64_C( 6379752944219310901), INT64_C(-1860193003353627344),
                            INT64_C( -866976878921007676), INT64_C( 3719036040063860682)) },
    { simde_mm512_set_epi64(INT64_C( 6933317985964373307), INT64_C(-7912084547370987750),
                            INT64_C( 1434122569595023374), INT64_C(  372849821895528123),
                            INT64_C( -797096709674116855), INT64_C( 7124042714150240897),
                            INT64_C(  192820077199458500), INT64_C( 2333974304098521090)),
      UINT8_C(243),
      simde_mm256_set_epi64x(INT64_C( 9216760499566437432), INT64_C(-3611239802138142732),
                             INT64_C(-4586686018735308980), INT64_C(-4383556822793463465)),
      simde_mm512_set_epi64(INT64_C( 9216760499566437432), INT64_C(-3611239802138142732),
                            INT64_C(-4586686018735308980), INT64_C(-4383556822793463465),
                            INT64_C( -797096709674116855), INT64_C( 7124042714150240897),
                            INT64_C(-4586686018735308980), INT64_C(-4383556822793463465)) },
    { simde_mm512_set_epi64(INT64_C(-1447537183271280169), INT64_C( 3992622506060288146),
                            INT64_C(-4043997837551953925), INT64_C( 6303477149728220498),
                            INT64_C( 7148655265583700891), INT64_C(-2780283900793463061),
                            INT64_C( 3296623181868458839), INT64_C( 3808941703531633947)),
      UINT8_C(191),
      simde_mm256_set_epi64x(INT64_C( 4775871390633368548), INT64_C( 1184569154591270183),
                             INT64_C(-1750343127516454914), INT64_C( 3950749388527391085)),
      simde_mm512_set_epi64(INT64_C( 4775871390633368548), INT64_C( 3992622506060288146),
                            INT64_C(-1750343127516454914), INT64_C( 3950749388527391085),
                            INT64_C( 4775871390633368548), INT64_C( 1184569154591270183),
                            INT64_C(-1750343127516454914), INT64_C( 3950749388527391085)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_broadcast_i64x4(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcast_i64x4(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m256i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C( 81),
      simde_mm256_set_epi64x(INT64_C(-3226888659503117201), INT64_C( 7490209482650655404),
                             INT64_C(-9179276487306987344), INT64_C( 7055682156038845095)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C( 7490209482650655404),
                            INT64_C(                   0), INT64_C( 7055682156038845095),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 7055682156038845095)) },
    { UINT8_C(230),
      simde_mm256_set_epi64x(INT64_C( 6952848743567724070), INT64_C( 6398498157984007660),
                             INT64_C(-7276216502972313781), INT64_C( 4842545408380684085)),
      simde_mm512_set_epi64(INT64_C( 6952848743567724070), INT64_C( 6398498157984007660),
                            INT64_C(-7276216502972313781), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 6398498157984007660),
                            INT64_C(-7276216502972313781), INT64_C(                   0)) },
    { UINT8_C(115),
      simde_mm256_set_epi64x(INT64_C( -147426939517817059), INT64_C(-3374766540151601501),
                             INT64_C( 9013437962204473886), INT64_C( 2290211861166994880)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-3374766540151601501),
                            INT64_C( 9013437962204473886), INT64_C( 2290211861166994880),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 9013437962204473886), INT64_C( 2290211861166994880)) },
    { UINT8_C(102),
      simde_mm256_set_epi64x(INT64_C(-8700458333795307779), INT64_C(-9147297996573979024),
                             INT64_C(-3649385965919135635), INT64_C( 1818037113458506686)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-9147297996573979024),
                            INT64_C(-3649385965919135635), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(-9147297996573979024),
                            INT64_C(-3649385965919135635), INT64_C(                   0)) },
    { UINT8_C( 59),
      simde_mm256_set_epi64x(INT64_C( 8763762661767364639), INT64_C(-7194784414741958081),
                             INT64_C(-1605849263772874289), INT64_C(-2187551180549076287)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(-1605849263772874289), INT64_C(-2187551180549076287),
                            INT64_C( 8763762661767364639), INT64_C(                   0),
                            INT64_C(-1605849263772874289), INT64_C(-2187551180549076287)) },
    { UINT8_C(119),
      simde_mm256_set_epi64x(INT64_C( 3282428208913039389), INT64_C(-2887297167729747289),
                             INT64_C( 6938672003976555894), INT64_C(-3765766577293323049)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-2887297167729747289),
                            INT64_C( 6938672003976555894), INT64_C(-3765766577293323049),
                            INT64_C(                   0), INT64_C(-2887297167729747289),
                            INT64_C( 6938672003976555894), INT64_C(-3765766577293323049)) },
    { UINT8_C( 25),
      simde_mm256_set_epi64x(INT64_C(-4802008903577488206), INT64_C(-3983516919532966210),
                             INT64_C(-4702094198572773446), INT64_C( -958715043139892800)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( -958715043139892800),
                            INT64_C(-4802008903577488206), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( -958715043139892800)) },
    { UINT8_C(207),
      simde_mm256_set_epi64x(INT64_C( 2289318697780797186), INT64_C(-4515948424499803858),
                             INT64_C( 7316310196690749623), INT64_C( 4937967944726422430)),
      simde_mm512_set_epi64(INT64_C( 2289318697780797186), INT64_C(-4515948424499803858),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 2289318697780797186), INT64_C(-4515948424499803858),
                            INT64_C( 7316310196690749623), INT64_C( 4937967944726422430)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_broadcast_i64x4(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_broadcastd_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi32(INT32_C(-1051270324), INT32_C(-1977183446), INT32_C( -548195640), INT32_C(-1363461466)),
      simde_mm512_set_epi32(INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466),
                            INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466),
                            INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466),
                            INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466), INT32_C(-1363461466)) },
    { simde_mm_set_epi32(INT32_C(  979094891), INT32_C(  416506319), INT32_C( 2123490297), INT32_C(  200388421)),
      simde_mm512_set_epi32(INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421),
                            INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421),
                            INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421),
                            INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421), INT32_C(  200388421)) },
    { simde_mm_set_epi32(INT32_C( 1927260635), INT32_C( 1201458882), INT32_C(-1448742498), INT32_C(-1111904220)),
      simde_mm512_set_epi32(INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220),
                            INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220),
                            INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220),
                            INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220), INT32_C(-1111904220)) },
    { simde_mm_set_epi32(INT32_C( -976455818), INT32_C(  542613123), INT32_C(  -15911923), INT32_C( -562895064)),
      simde_mm512_set_epi32(INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064),
                            INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064),
                            INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064),
                            INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064), INT32_C( -562895064)) },
    { simde_mm_set_epi32(INT32_C(  836747087), INT32_C(-1431045412), INT32_C(-1356396683), INT32_C( 1489138473)),
      simde_mm512_set_epi32(INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473),
                            INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473),
                            INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473),
                            INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473), INT32_C( 1489138473)) },
    { simde_mm_set_epi32(INT32_C(-1783426961), INT32_C( -263517415), INT32_C(-1697630001), INT32_C( 2025142863)),
      simde_mm512_set_epi32(INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863),
                            INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863),
                            INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863),
                            INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863), INT32_C( 2025142863)) },
    { simde_mm_set_epi32(INT32_C(  300619496), INT32_C( -659754204), INT32_C(-1019736463), INT32_C( 1022872166)),
      simde_mm512_set_epi32(INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166),
                            INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166),
                            INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166),
                            INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166), INT32_C( 1022872166)) },
    { simde_mm_set_epi32(INT32_C( -274893610), INT32_C(  171227717), INT32_C( 1187872667), INT32_C( -590903223)),
      simde_mm512_set_epi32(INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223),
                            INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223),
                            INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223),
                            INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223), INT32_C( -590903223)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcastd_epi32(test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcastd_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1638944021), INT32_C( -385149059), INT32_C(  852916680), INT32_C(-1839015366),
                            INT32_C( 1146921463), INT32_C(  765234486), INT32_C( -388218844), INT32_C(-1402803832),
                            INT32_C( 1245942358), INT32_C( 2001202713), INT32_C(  868062804), INT32_C(-1988191751),
                            INT32_C(  807099340), INT32_C(  -38350755), INT32_C( -443928971), INT32_C( -432554813)),
      UINT16_C(24594),
      simde_mm_set_epi32(INT32_C( -255909174), INT32_C(-1302917278), INT32_C(  327520540), INT32_C(  176606543)),
      simde_mm512_set_epi32(INT32_C( 1638944021), INT32_C(  176606543), INT32_C(  176606543), INT32_C(-1839015366),
                            INT32_C( 1146921463), INT32_C(  765234486), INT32_C( -388218844), INT32_C(-1402803832),
                            INT32_C( 1245942358), INT32_C( 2001202713), INT32_C(  868062804), INT32_C(  176606543),
                            INT32_C(  807099340), INT32_C(  -38350755), INT32_C(  176606543), INT32_C( -432554813)) },
    { simde_mm512_set_epi32(INT32_C( -115460801), INT32_C( 1889676725), INT32_C(    2912775), INT32_C(-1289469215),
                            INT32_C( 1033489041), INT32_C(  147853139), INT32_C(  706073024), INT32_C( -130092746),
                            INT32_C( -799642653), INT32_C(-1439962375), INT32_C(-1798405841), INT32_C( 1190396108),
                            INT32_C(-1013986568), INT32_C(  994541610), INT32_C(-1127995400), INT32_C( 1108325476)),
      UINT16_C(40849),
      simde_mm_set_epi32(INT32_C(  250706831), INT32_C( -936079925), INT32_C(-1129184131), INT32_C(  803417186)),
      simde_mm512_set_epi32(INT32_C(  803417186), INT32_C( 1889676725), INT32_C(    2912775), INT32_C(  803417186),
                            INT32_C(  803417186), INT32_C(  803417186), INT32_C(  803417186), INT32_C(  803417186),
                            INT32_C(  803417186), INT32_C(-1439962375), INT32_C(-1798405841), INT32_C(  803417186),
                            INT32_C(-1013986568), INT32_C(  994541610), INT32_C(-1127995400), INT32_C(  803417186)) },
    { simde_mm512_set_epi32(INT32_C(  357625867), INT32_C( -157238200), INT32_C(  909767636), INT32_C( 1422277073),
                            INT32_C( 2123935701), INT32_C(-1040550911), INT32_C(  686758291), INT32_C(-2090356905),
                            INT32_C( -362358815), INT32_C( -482453842), INT32_C(  117787421), INT32_C( 1300554279),
                            INT32_C(-1085613264), INT32_C( -109297466), INT32_C(-1230203271), INT32_C(-1731521429)),
      UINT16_C(53728),
      simde_mm_set_epi32(INT32_C( -707786971), INT32_C( 1712040202), INT32_C(-2012675757), INT32_C(-1396559749)),
      simde_mm512_set_epi32(INT32_C(-1396559749), INT32_C(-1396559749), INT32_C(  909767636), INT32_C(-1396559749),
                            INT32_C( 2123935701), INT32_C(-1040550911), INT32_C(  686758291), INT32_C(-1396559749),
                            INT32_C(-1396559749), INT32_C(-1396559749), INT32_C(-1396559749), INT32_C( 1300554279),
                            INT32_C(-1085613264), INT32_C( -109297466), INT32_C(-1230203271), INT32_C(-1731521429)) },
    { simde_mm512_set_epi32(INT32_C( 2041534605), INT32_C( 1255681923), INT32_C( 1220121473), INT32_C( 1819952522),
                            INT32_C(-1737362693), INT32_C(  712438877), INT32_C(-1234448370), INT32_C(  217554028),
                            INT32_C(-1878093154), INT32_C( -741869417), INT32_C(  943666007), INT32_C(  622675686),
                            INT32_C( -269910912), INT32_C(  137195559), INT32_C(  469574756), INT32_C( 1490101689)),
      UINT16_C(50038),
      simde_mm_set_epi32(INT32_C( -272719467), INT32_C( -594597983), INT32_C( -820913821), INT32_C(  345700481)),
      simde_mm512_set_epi32(INT32_C(  345700481), INT32_C(  345700481), INT32_C( 1220121473), INT32_C( 1819952522),
                            INT32_C(-1737362693), INT32_C(  712438877), INT32_C(  345700481), INT32_C(  345700481),
                            INT32_C(-1878093154), INT32_C(  345700481), INT32_C(  345700481), INT32_C(  345700481),
                            INT32_C( -269910912), INT32_C(  345700481), INT32_C(  345700481), INT32_C( 1490101689)) },
    { simde_mm512_set_epi32(INT32_C(  605201121), INT32_C(    2188130), INT32_C( -956406632), INT32_C(-1144421408),
                            INT32_C(-2008693903), INT32_C( 1823632430), INT32_C( 2043624683), INT32_C(  457225971),
                            INT32_C( 1484257119), INT32_C(  719932227), INT32_C( 1722430058), INT32_C(  916001650),
                            INT32_C(  553469699), INT32_C(-2003831430), INT32_C(-1834906502), INT32_C(  225358926)),
      UINT16_C(22657),
      simde_mm_set_epi32(INT32_C(  290541765), INT32_C( -479926223), INT32_C( 2079119915), INT32_C( -331512500)),
      simde_mm512_set_epi32(INT32_C(  605201121), INT32_C( -331512500), INT32_C( -956406632), INT32_C( -331512500),
                            INT32_C( -331512500), INT32_C( 1823632430), INT32_C( 2043624683), INT32_C(  457225971),
                            INT32_C( -331512500), INT32_C(  719932227), INT32_C( 1722430058), INT32_C(  916001650),
                            INT32_C(  553469699), INT32_C(-2003831430), INT32_C(-1834906502), INT32_C( -331512500)) },
    { simde_mm512_set_epi32(INT32_C( -545987817), INT32_C(-1146550995), INT32_C(  963048631), INT32_C( -701605919),
                            INT32_C(  432096480), INT32_C(-2030393254), INT32_C(-1236899565), INT32_C(-1697034971),
                            INT32_C( -998012960), INT32_C(-1579141793), INT32_C( 1664269708), INT32_C( -667117157),
                            INT32_C( -708117814), INT32_C(   85211107), INT32_C(  909670673), INT32_C( 1616737139)),
      UINT16_C( 4531),
      simde_mm_set_epi32(INT32_C( -503580732), INT32_C(-1790221512), INT32_C(-1663970343), INT32_C( 1633501790)),
      simde_mm512_set_epi32(INT32_C( -545987817), INT32_C(-1146550995), INT32_C(  963048631), INT32_C( 1633501790),
                            INT32_C(  432096480), INT32_C(-2030393254), INT32_C(-1236899565), INT32_C( 1633501790),
                            INT32_C( 1633501790), INT32_C(-1579141793), INT32_C( 1633501790), INT32_C( 1633501790),
                            INT32_C( -708117814), INT32_C(   85211107), INT32_C( 1633501790), INT32_C( 1633501790)) },
    { simde_mm512_set_epi32(INT32_C(-1668661089), INT32_C( 1895031925), INT32_C( 2107029353), INT32_C(-1915428586),
                            INT32_C(  963718296), INT32_C( 1878898594), INT32_C( -403168746), INT32_C(  502390291),
                            INT32_C( 1855826407), INT32_C(-1442018177), INT32_C( -244961355), INT32_C( 1777042193),
                            INT32_C(  373997996), INT32_C( -684064874), INT32_C(  930695451), INT32_C(-1073438864)),
      UINT16_C(53861),
      simde_mm_set_epi32(INT32_C( 1599859635), INT32_C(  543659234), INT32_C(-1222091200), INT32_C(  817594139)),
      simde_mm512_set_epi32(INT32_C(  817594139), INT32_C(  817594139), INT32_C( 2107029353), INT32_C(  817594139),
                            INT32_C(  963718296), INT32_C( 1878898594), INT32_C(  817594139), INT32_C(  502390291),
                            INT32_C( 1855826407), INT32_C(  817594139), INT32_C(  817594139), INT32_C( 1777042193),
                            INT32_C(  373997996), INT32_C(  817594139), INT32_C(  930695451), INT32_C(  817594139)) },
    { simde_mm512_set_epi32(INT32_C( -831807470), INT32_C( -591553083), INT32_C( -492649784), INT32_C(-1394371521),
                            INT32_C(-1760655625), INT32_C( 2135736563), INT32_C(-2075134444), INT32_C( -933317766),
                            INT32_C( -731013025), INT32_C(-2091361347), INT32_C( 1562364760), INT32_C( -612070110),
                            INT32_C( 1365385309), INT32_C( -121237183), INT32_C( 1543044931), INT32_C(-1490381593)),
      UINT16_C(20921),
      simde_mm_set_epi32(INT32_C(-1466503600), INT32_C(  824864478), INT32_C(-1491396230), INT32_C(-1907140086)),
      simde_mm512_set_epi32(INT32_C( -831807470), INT32_C(-1907140086), INT32_C( -492649784), INT32_C(-1907140086),
                            INT32_C(-1760655625), INT32_C( 2135736563), INT32_C(-2075134444), INT32_C(-1907140086),
                            INT32_C(-1907140086), INT32_C(-2091361347), INT32_C(-1907140086), INT32_C(-1907140086),
                            INT32_C(-1907140086), INT32_C( -121237183), INT32_C( 1543044931), INT32_C(-1907140086)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_broadcastd_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcastd_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(21274),
      simde_mm_set_epi32(INT32_C( 1459257075), INT32_C(  587801532), INT32_C( 1631678564), INT32_C(  715337051)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  715337051), INT32_C(          0), INT32_C(  715337051),
                            INT32_C(          0), INT32_C(          0), INT32_C(  715337051), INT32_C(  715337051),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(  715337051),
                            INT32_C(  715337051), INT32_C(          0), INT32_C(  715337051), INT32_C(          0)) },
    { UINT16_C(59357),
      simde_mm_set_epi32(INT32_C(-2022546688), INT32_C( 2145084340), INT32_C(   29275255), INT32_C( -827125259)),
      simde_mm512_set_epi32(INT32_C( -827125259), INT32_C( -827125259), INT32_C( -827125259), INT32_C(          0),
                            INT32_C(          0), INT32_C( -827125259), INT32_C( -827125259), INT32_C( -827125259),
                            INT32_C( -827125259), INT32_C( -827125259), INT32_C(          0), INT32_C( -827125259),
                            INT32_C( -827125259), INT32_C( -827125259), INT32_C(          0), INT32_C( -827125259)) },
    { UINT16_C(34446),
      simde_mm_set_epi32(INT32_C(  973425906), INT32_C( -935954345), INT32_C( 1285315081), INT32_C( 2142489532)),
      simde_mm512_set_epi32(INT32_C( 2142489532), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( 2142489532), INT32_C( 2142489532), INT32_C(          0),
                            INT32_C( 2142489532), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 2142489532), INT32_C( 2142489532), INT32_C( 2142489532), INT32_C(          0)) },
    { UINT16_C(33955),
      simde_mm_set_epi32(INT32_C(-1114656122), INT32_C( 1221674060), INT32_C( -740975665), INT32_C( 2132760332)),
      simde_mm512_set_epi32(INT32_C( 2132760332), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C( 2132760332), INT32_C(          0), INT32_C(          0),
                            INT32_C( 2132760332), INT32_C(          0), INT32_C( 2132760332), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C( 2132760332), INT32_C( 2132760332)) },
    { UINT16_C(52572),
      simde_mm_set_epi32(INT32_C( -724774954), INT32_C( -166426332), INT32_C(-1571631693), INT32_C( -124417294)),
      simde_mm512_set_epi32(INT32_C( -124417294), INT32_C( -124417294), INT32_C(          0), INT32_C(          0),
                            INT32_C( -124417294), INT32_C( -124417294), INT32_C(          0), INT32_C( -124417294),
                            INT32_C(          0), INT32_C( -124417294), INT32_C(          0), INT32_C( -124417294),
                            INT32_C( -124417294), INT32_C( -124417294), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(38931),
      simde_mm_set_epi32(INT32_C(-1992244525), INT32_C( -292982508), INT32_C( -691380397), INT32_C(-1292068161)),
      simde_mm512_set_epi32(INT32_C(-1292068161), INT32_C(          0), INT32_C(          0), INT32_C(-1292068161),
                            INT32_C(-1292068161), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(-1292068161),
                            INT32_C(          0), INT32_C(          0), INT32_C(-1292068161), INT32_C(-1292068161)) },
    { UINT16_C(32377),
      simde_mm_set_epi32(INT32_C( -766689829), INT32_C(-1724046912), INT32_C( 1799018744), INT32_C(  623047724)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  623047724), INT32_C(  623047724), INT32_C(  623047724),
                            INT32_C(  623047724), INT32_C(  623047724), INT32_C(  623047724), INT32_C(          0),
                            INT32_C(          0), INT32_C(  623047724), INT32_C(  623047724), INT32_C(  623047724),
                            INT32_C(  623047724), INT32_C(          0), INT32_C(          0), INT32_C(  623047724)) },
    { UINT16_C(18782),
      simde_mm_set_epi32(INT32_C(-2020669200), INT32_C( -170583969), INT32_C( -628885190), INT32_C(  818636447)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(  818636447), INT32_C(          0), INT32_C(          0),
                            INT32_C(  818636447), INT32_C(          0), INT32_C(          0), INT32_C(  818636447),
                            INT32_C(          0), INT32_C(  818636447), INT32_C(          0), INT32_C(  818636447),
                            INT32_C(  818636447), INT32_C(  818636447), INT32_C(  818636447), INT32_C(          0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_broadcastd_epi32(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_broadcastq_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi64x(INT64_C(-4515171658517540054), INT64_C(-2354482342678283610)),
      simde_mm512_set_epi64(INT64_C(-2354482342678283610), INT64_C(-2354482342678283610),
                            INT64_C(-2354482342678283610), INT64_C(-2354482342678283610),
                            INT64_C(-2354482342678283610), INT64_C(-2354482342678283610),
                            INT64_C(-2354482342678283610), INT64_C(-2354482342678283610)) },
    { simde_mm_set_epi64x(INT64_C( 4205180536942191055), INT64_C( 9120321379188715333)),
      simde_mm512_set_epi64(INT64_C( 9120321379188715333), INT64_C( 9120321379188715333),
                            INT64_C( 9120321379188715333), INT64_C( 9120321379188715333),
                            INT64_C( 9120321379188715333), INT64_C( 9120321379188715333),
                            INT64_C( 9120321379188715333), INT64_C( 9120321379188715333)) },
    { simde_mm_set_epi64x(INT64_C( 8277521399394651842), INT64_C(-6222301646052282332)),
      simde_mm512_set_epi64(INT64_C(-6222301646052282332), INT64_C(-6222301646052282332),
                            INT64_C(-6222301646052282332), INT64_C(-6222301646052282332),
                            INT64_C(-6222301646052282332), INT64_C(-6222301646052282332),
                            INT64_C(-6222301646052282332), INT64_C(-6222301646052282332)) },
    { simde_mm_set_epi64x(INT64_C(-4193845803756315005), INT64_C(  -68341185169397976)),
      simde_mm512_set_epi64(INT64_C(  -68341185169397976), INT64_C(  -68341185169397976),
                            INT64_C(  -68341185169397976), INT64_C(  -68341185169397976),
                            INT64_C(  -68341185169397976), INT64_C(  -68341185169397976),
                            INT64_C(  -68341185169397976), INT64_C(  -68341185169397976)) },
    { simde_mm_set_epi64x(INT64_C( 3593801376552188636), INT64_C(-5825679392398740695)),
      simde_mm512_set_epi64(INT64_C(-5825679392398740695), INT64_C(-5825679392398740695),
                            INT64_C(-5825679392398740695), INT64_C(-5825679392398740695),
                            INT64_C(-5825679392398740695), INT64_C(-5825679392398740695),
                            INT64_C(-5825679392398740695), INT64_C(-5825679392398740695)) },
    { simde_mm_set_epi64x(INT64_C(-7659760468268217575), INT64_C(-7291265332978304433)),
      simde_mm512_set_epi64(INT64_C(-7291265332978304433), INT64_C(-7291265332978304433),
                            INT64_C(-7291265332978304433), INT64_C(-7291265332978304433),
                            INT64_C(-7291265332978304433), INT64_C(-7291265332978304433),
                            INT64_C(-7291265332978304433), INT64_C(-7291265332978304433)) },
    { simde_mm_set_epi64x(INT64_C( 1291150907495215908), INT64_C(-4379734758100841882)),
      simde_mm512_set_epi64(INT64_C(-4379734758100841882), INT64_C(-4379734758100841882),
                            INT64_C(-4379734758100841882), INT64_C(-4379734758100841882),
                            INT64_C(-4379734758100841882), INT64_C(-4379734758100841882),
                            INT64_C(-4379734758100841882), INT64_C(-4379734758100841882)) },
    { simde_mm_set_epi64x(INT64_C(-1180659064658150843), INT64_C( 5101874260281362505)),
      simde_mm512_set_epi64(INT64_C( 5101874260281362505), INT64_C( 5101874260281362505),
                            INT64_C( 5101874260281362505), INT64_C( 5101874260281362505),
                            INT64_C( 5101874260281362505), INT64_C( 5101874260281362505),
                            INT64_C( 5101874260281362505), INT64_C( 5101874260281362505)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_broadcastq_epi64(test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcastq_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C( 7039210974079555453), INT64_C( 3663249249268849210),
                            INT64_C( 4925990175430708534), INT64_C(-1667387235778762360),
                            INT64_C( 5351281682312326681), INT64_C( 3728301356360833529),
                            INT64_C( 3466465274179801181), INT64_C(-1906660408329519933)),
      UINT8_C( 18),
      simde_mm_set_epi64x(INT64_C(-5595987098075819748), INT64_C(  758519329013942001)),
      simde_mm512_set_epi64(INT64_C( 7039210974079555453), INT64_C( 3663249249268849210),
                            INT64_C( 4925990175430708534), INT64_C(  758519329013942001),
                            INT64_C( 5351281682312326681), INT64_C( 3728301356360833529),
                            INT64_C(  758519329013942001), INT64_C(-1906660408329519933)) },
    { simde_mm512_set_epi64(INT64_C( 8116099733890298375), INT64_C(-5538228106590303599),
                            INT64_C(  635024397322015168), INT64_C( -558744086021510173),
                            INT64_C(-6184591305598926545), INT64_C( 5112712356426664696),
                            INT64_C( 4271523692628158456), INT64_C( 4760221676782691018)),
      UINT8_C( 63),
      simde_mm_set_epi64x(INT64_C(-4849808913003762590), INT64_C( 7901381612815228817)),
      simde_mm512_set_epi64(INT64_C( 8116099733890298375), INT64_C(-5538228106590303599),
                            INT64_C( 7901381612815228817), INT64_C( 7901381612815228817),
                            INT64_C( 7901381612815228817), INT64_C( 7901381612815228817),
                            INT64_C( 7901381612815228817), INT64_C( 7901381612815228817)) },
    { simde_mm512_set_epi64(INT64_C( 3907422245001509329), INT64_C( 9122234377856250881),
                            INT64_C( 2949604402306461527), INT64_C(-1556319256029800786),
                            INT64_C(  505893122375737895), INT64_C(-4662673460798144314),
                            INT64_C(-5283682813813779349), INT64_C( 1076777643387686347)),
      UINT8_C( 72),
      simde_mm_set_epi64x(INT64_C(-5998178448496319999), INT64_C( 7927410529462710283)),
      simde_mm512_set_epi64(INT64_C( 3907422245001509329), INT64_C( 7927410529462710283),
                            INT64_C( 2949604402306461527), INT64_C(-1556319256029800786),
                            INT64_C( 7927410529462710283), INT64_C(-4662673460798144314),
                            INT64_C(-5283682813813779349), INT64_C( 1076777643387686347)) },
    { simde_mm512_set_epi64(INT64_C( 7816636564820325115), INT64_C( 3059901680174485518),
                            INT64_C(  934387437789942430), INT64_C(-3186304882973920425),
                            INT64_C( 2674371711409421440), INT64_C(  589250439531013220),
                            INT64_C( 6399938025556543269), INT64_C( 7353156679309525331)),
      UINT8_C(129),
      simde_mm_set_epi64x(INT64_C(-3132262719190613130), INT64_C( 8768324363382960003)),
      simde_mm512_set_epi64(INT64_C( 8768324363382960003), INT64_C( 3059901680174485518),
                            INT64_C(  934387437789942430), INT64_C(-3186304882973920425),
                            INT64_C( 2674371711409421440), INT64_C(  589250439531013220),
                            INT64_C( 6399938025556543269), INT64_C( 8768324363382960003)) },
    { simde_mm512_set_epi64(INT64_C(-8627274619235963858), INT64_C( 8777301179240593139),
                            INT64_C( 6374835785680112451), INT64_C( 7397780769673384818),
                            INT64_C( 2377134258823099770), INT64_C(-7880863417082399666),
                            INT64_C(-1171321188047181919), INT64_C(-3525798013683697535)),
      UINT8_C(224),
      simde_mm_set_epi64x(INT64_C(-5077429793204296991), INT64_C(    9397950127957144)),
      simde_mm512_set_epi64(INT64_C(    9397950127957144), INT64_C(    9397950127957144),
                            INT64_C(    9397950127957144), INT64_C( 7397780769673384818),
                            INT64_C( 2377134258823099770), INT64_C(-7880863417082399666),
                            INT64_C(-1171321188047181919), INT64_C(-3525798013683697535)) },
    { simde_mm512_set_epi64(INT64_C(-8720472620890953453), INT64_C(-7288709697316354080),
                            INT64_C(-6782362355017532020), INT64_C(-2865246368328647990),
                            INT64_C(  365978918730627345), INT64_C( 6943833138524147909),
                            INT64_C(-2061267430198683093), INT64_C(-1423835345422209809)),
      UINT8_C(224),
      simde_mm_set_epi64x(INT64_C(-2344999814881016531), INT64_C( 4136262378195933153)),
      simde_mm512_set_epi64(INT64_C( 4136262378195933153), INT64_C( 4136262378195933153),
                            INT64_C( 4136262378195933153), INT64_C(-2865246368328647990),
                            INT64_C(  365978918730627345), INT64_C( 6943833138524147909),
                            INT64_C(-2061267430198683093), INT64_C(-1423835345422209809)) },
    { simde_mm512_set_epi64(INT64_C(-1731596578336940525), INT64_C( 7970713727971134591),
                            INT64_C(-1052101006731803887), INT64_C( 1606309165200441238),
                            INT64_C( 3997306527802498928), INT64_C(-2162862772330994888),
                            INT64_C(-7146698203065400738), INT64_C(-4784261768320577101)),
      UINT8_C(162),
      simde_mm_set_epi64x(INT64_C( 8139100144857954153), INT64_C(-8226703133729805160)),
      simde_mm512_set_epi64(INT64_C(-8226703133729805160), INT64_C( 7970713727971134591),
                            INT64_C(-8226703133729805160), INT64_C( 1606309165200441238),
                            INT64_C( 3997306527802498928), INT64_C(-2162862772330994888),
                            INT64_C(-8226703133729805160), INT64_C(-4784261768320577101)) },
    { simde_mm512_set_epi64(INT64_C(-4008569278181826465), INT64_C(-8982328587921142952),
                            INT64_C(-2628821103943737251), INT64_C( -520709734501122237),
                            INT64_C(-6401140198895522893), INT64_C( 2334998633271287360),
                            INT64_C( 3511540092651127844), INT64_C( 3766647997225123999)),
      UINT8_C( 20),
      simde_mm_set_epi64x(INT64_C(-2115914707760868289), INT64_C(-7561958326757703437)),
      simde_mm512_set_epi64(INT64_C(-4008569278181826465), INT64_C(-8982328587921142952),
                            INT64_C(-2628821103943737251), INT64_C(-7561958326757703437),
                            INT64_C(-6401140198895522893), INT64_C(-7561958326757703437),
                            INT64_C( 3511540092651127844), INT64_C( 3766647997225123999)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_broadcastq_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcastq_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C( 26),
      simde_mm_set_epi64x(INT64_C( 2524588358110376036), INT64_C( 3072349241054123220)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 3072349241054123220),
                            INT64_C( 3072349241054123220), INT64_C(                   0),
                            INT64_C( 3072349241054123220), INT64_C(                   0)) },
    { UINT8_C(243),
      simde_mm_set_epi64x(INT64_C(  125736266274902517), INT64_C( 4529119523676940253)),
      simde_mm512_set_epi64(INT64_C( 4529119523676940253), INT64_C( 4529119523676940253),
                            INT64_C( 4529119523676940253), INT64_C( 4529119523676940253),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 4529119523676940253), INT64_C( 4529119523676940253)) },
    { UINT8_C(180),
      simde_mm_set_epi64x(INT64_C( 9201922475629043961), INT64_C(-5256397243355602176)),
      simde_mm512_set_epi64(INT64_C(-5256397243355602176), INT64_C(                   0),
                            INT64_C(-5256397243355602176), INT64_C(-5256397243355602176),
                            INT64_C(                   0), INT64_C(-5256397243355602176),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(  9),
      simde_mm_set_epi64x(INT64_C( 4382010425855345827), INT64_C( 4180832434708183127)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 4180832434708183127), INT64_C(                   0),
                            INT64_C(                   0), INT64_C( 4180832434708183127)) },
    { UINT8_C( 12),
      simde_mm_set_epi64x(INT64_C(-1597707644585397626), INT64_C( 5247050137625533391)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 5247050137625533391), INT64_C( 5247050137625533391),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(150),
      simde_mm_set_epi64x(INT64_C(-3112884720261363420), INT64_C(-6750106718621562126)),
      simde_mm512_set_epi64(INT64_C(-6750106718621562126), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(-6750106718621562126),
                            INT64_C(                   0), INT64_C(-6750106718621562126),
                            INT64_C(-6750106718621562126), INT64_C(                   0)) },
    { UINT8_C( 19),
      simde_mm_set_epi64x(INT64_C(-1258350286556471469), INT64_C(-5549390491787734701)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(-5549390491787734701),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(-5549390491787734701), INT64_C(-5549390491787734701)) },
    { UINT8_C(211),
      simde_mm_set_epi64x(INT64_C( 7726726670994043948), INT64_C( 5635717459582615161)),
      simde_mm512_set_epi64(INT64_C( 5635717459582615161), INT64_C( 5635717459582615161),
                            INT64_C(                   0), INT64_C( 5635717459582615161),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C( 5635717459582615161), INT64_C( 5635717459582615161)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_broadcastq_epi64(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_broadcastss_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   104.48), SIMDE_FLOAT32_C(   410.97), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   631.04)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   668.18), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -627.99), SIMDE_FLOAT32_C(  -347.00)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00), SIMDE_FLOAT32_C(  -347.00)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(  -776.74), SIMDE_FLOAT32_C(   643.82), SIMDE_FLOAT32_C(  -941.79)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79),
                         SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79),
                         SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79),
                         SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79), SIMDE_FLOAT32_C(  -941.79)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -401.52), SIMDE_FLOAT32_C(   338.53), SIMDE_FLOAT32_C(  -725.48), SIMDE_FLOAT32_C(   387.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(   387.06)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   119.04), SIMDE_FLOAT32_C(   263.81), SIMDE_FLOAT32_C(   717.18), SIMDE_FLOAT32_C(  -996.30)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30),
                         SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30),
                         SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30),
                         SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(  -996.30)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -278.68), SIMDE_FLOAT32_C(   120.15), SIMDE_FLOAT32_C(   751.98), SIMDE_FLOAT32_C(   536.33)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33),
                         SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33),
                         SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33),
                         SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(   536.33)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -143.65), SIMDE_FLOAT32_C(   810.77), SIMDE_FLOAT32_C(  -448.76), SIMDE_FLOAT32_C(   234.43)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43),
                         SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43),
                         SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43),
                         SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   234.43)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   -42.20), SIMDE_FLOAT32_C(  -923.83), SIMDE_FLOAT32_C(   357.03), SIMDE_FLOAT32_C(  -933.51)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51),
                         SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51),
                         SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51),
                         SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -933.51)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_broadcastss_ps(test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcastss_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 src;
    simde__mmask16 k;
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -227.30), SIMDE_FLOAT32_C(   999.04), SIMDE_FLOAT32_C(   956.07), SIMDE_FLOAT32_C(  -270.40),
                         SIMDE_FLOAT32_C(   132.00), SIMDE_FLOAT32_C(   480.19), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -927.52), SIMDE_FLOAT32_C(   -67.87), SIMDE_FLOAT32_C(   891.86), SIMDE_FLOAT32_C(  -870.50),
                         SIMDE_FLOAT32_C(   932.69), SIMDE_FLOAT32_C(   244.86), SIMDE_FLOAT32_C(  -621.59), SIMDE_FLOAT32_C(    36.25)),
      UINT16_C(30253),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   134.75), SIMDE_FLOAT32_C(   871.12), SIMDE_FLOAT32_C(   104.48), SIMDE_FLOAT32_C(   548.32)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -227.30), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   548.32),
                         SIMDE_FLOAT32_C(   132.00), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(  -347.00),
                         SIMDE_FLOAT32_C(  -927.52), SIMDE_FLOAT32_C(   -67.87), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(  -870.50),
                         SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(  -621.59), SIMDE_FLOAT32_C(   548.32)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -996.30), SIMDE_FLOAT32_C(   389.31), SIMDE_FLOAT32_C(   670.35), SIMDE_FLOAT32_C(   396.13),
                         SIMDE_FLOAT32_C(  -971.67), SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(   275.37), SIMDE_FLOAT32_C(   338.53),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(    29.64), SIMDE_FLOAT32_C(   199.34), SIMDE_FLOAT32_C(  -686.40),
                         SIMDE_FLOAT32_C(   717.18), SIMDE_FLOAT32_C(   416.06), SIMDE_FLOAT32_C(   645.78), SIMDE_FLOAT32_C(  -990.79)),
      UINT16_C(37933),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   442.28), SIMDE_FLOAT32_C(   811.14), SIMDE_FLOAT32_C(  -767.79), SIMDE_FLOAT32_C(   236.12)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(   389.31), SIMDE_FLOAT32_C(   670.35), SIMDE_FLOAT32_C(   236.12),
                         SIMDE_FLOAT32_C(  -971.67), SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(   275.37), SIMDE_FLOAT32_C(   338.53),
                         SIMDE_FLOAT32_C(   387.06), SIMDE_FLOAT32_C(    29.64), SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(  -686.40),
                         SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(   236.12), SIMDE_FLOAT32_C(   645.78), SIMDE_FLOAT32_C(   236.12)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   810.77), SIMDE_FLOAT32_C(   234.43), SIMDE_FLOAT32_C(   101.39), SIMDE_FLOAT32_C(  -366.10),
                         SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   -55.56), SIMDE_FLOAT32_C(   896.89), SIMDE_FLOAT32_C(   697.60),
                         SIMDE_FLOAT32_C(   120.15), SIMDE_FLOAT32_C(   536.33), SIMDE_FLOAT32_C(  -156.71), SIMDE_FLOAT32_C(  -331.13),
                         SIMDE_FLOAT32_C(  -143.65), SIMDE_FLOAT32_C(  -448.76), SIMDE_FLOAT32_C(  -628.22), SIMDE_FLOAT32_C(   318.72)),
      UINT16_C(19701),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -248.84), SIMDE_FLOAT32_C(   566.99), SIMDE_FLOAT32_C(  -650.08), SIMDE_FLOAT32_C(  -460.40)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   810.77), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(   101.39), SIMDE_FLOAT32_C(  -366.10),
                         SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(   896.89), SIMDE_FLOAT32_C(   697.60),
                         SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -460.40),
                         SIMDE_FLOAT32_C(  -143.65), SIMDE_FLOAT32_C(  -460.40), SIMDE_FLOAT32_C(  -628.22), SIMDE_FLOAT32_C(  -460.40)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   779.71), SIMDE_FLOAT32_C(   196.66), SIMDE_FLOAT32_C(    -0.50), SIMDE_FLOAT32_C(  -273.06),
                         SIMDE_FLOAT32_C(   429.50), SIMDE_FLOAT32_C(   650.80), SIMDE_FLOAT32_C(   509.10), SIMDE_FLOAT32_C(   709.57),
                         SIMDE_FLOAT32_C(  -561.64), SIMDE_FLOAT32_C(  -923.83), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -304.13),
                         SIMDE_FLOAT32_C(   728.72), SIMDE_FLOAT32_C(  -511.49), SIMDE_FLOAT32_C(   144.42), SIMDE_FLOAT32_C(   848.91)),
      UINT16_C(27468),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   949.19), SIMDE_FLOAT32_C(  -102.63), SIMDE_FLOAT32_C(    87.04), SIMDE_FLOAT32_C(   914.16)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   779.71), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(  -273.06),
                         SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   650.80), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   914.16),
                         SIMDE_FLOAT32_C(  -561.64), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(  -933.51), SIMDE_FLOAT32_C(  -304.13),
                         SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   914.16), SIMDE_FLOAT32_C(   144.42), SIMDE_FLOAT32_C(   848.91)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -960.76), SIMDE_FLOAT32_C(  -613.57), SIMDE_FLOAT32_C(   864.92), SIMDE_FLOAT32_C(   278.02),
                         SIMDE_FLOAT32_C(   573.37), SIMDE_FLOAT32_C(   393.40), SIMDE_FLOAT32_C(  -782.91), SIMDE_FLOAT32_C(  -933.90),
                         SIMDE_FLOAT32_C(  -291.87), SIMDE_FLOAT32_C(   382.75), SIMDE_FLOAT32_C(   -62.73), SIMDE_FLOAT32_C(   163.52),
                         SIMDE_FLOAT32_C(    87.09), SIMDE_FLOAT32_C(  -486.60), SIMDE_FLOAT32_C(  -157.79), SIMDE_FLOAT32_C(  -247.69)),
      UINT16_C(56353),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   -97.06), SIMDE_FLOAT32_C(    -2.41), SIMDE_FLOAT32_C(   418.81), SIMDE_FLOAT32_C(  -141.42)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(   864.92), SIMDE_FLOAT32_C(  -141.42),
                         SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(  -782.91), SIMDE_FLOAT32_C(  -933.90),
                         SIMDE_FLOAT32_C(  -291.87), SIMDE_FLOAT32_C(   382.75), SIMDE_FLOAT32_C(  -141.42), SIMDE_FLOAT32_C(   163.52),
                         SIMDE_FLOAT32_C(    87.09), SIMDE_FLOAT32_C(  -486.60), SIMDE_FLOAT32_C(  -157.79), SIMDE_FLOAT32_C(  -141.42)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -490.98), SIMDE_FLOAT32_C(  -718.54), SIMDE_FLOAT32_C(  -172.08), SIMDE_FLOAT32_C(   476.19),
                         SIMDE_FLOAT32_C(  -825.45), SIMDE_FLOAT32_C(  -528.02), SIMDE_FLOAT32_C(  -604.26), SIMDE_FLOAT32_C(  -201.78),
                         SIMDE_FLOAT32_C(  -105.47), SIMDE_FLOAT32_C(   619.70), SIMDE_FLOAT32_C(   603.28), SIMDE_FLOAT32_C(  -553.28),
                         SIMDE_FLOAT32_C(   787.83), SIMDE_FLOAT32_C(  -945.21), SIMDE_FLOAT32_C(  -786.09), SIMDE_FLOAT32_C(   628.77)),
      UINT16_C(51486),
      simde_mm_set_ps(SIMDE_FLOAT32_C(    54.48), SIMDE_FLOAT32_C(   679.92), SIMDE_FLOAT32_C(  -550.45), SIMDE_FLOAT32_C(  -482.87)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -172.08), SIMDE_FLOAT32_C(   476.19),
                         SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -528.02), SIMDE_FLOAT32_C(  -604.26), SIMDE_FLOAT32_C(  -482.87),
                         SIMDE_FLOAT32_C(  -105.47), SIMDE_FLOAT32_C(   619.70), SIMDE_FLOAT32_C(   603.28), SIMDE_FLOAT32_C(  -482.87),
                         SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(  -482.87), SIMDE_FLOAT32_C(   628.77)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -124.63), SIMDE_FLOAT32_C(  -948.04), SIMDE_FLOAT32_C(  -344.11), SIMDE_FLOAT32_C(  -424.86),
                         SIMDE_FLOAT32_C(   640.76), SIMDE_FLOAT32_C(  -243.42), SIMDE_FLOAT32_C(   962.71), SIMDE_FLOAT32_C(   314.11),
                         SIMDE_FLOAT32_C(   599.88), SIMDE_FLOAT32_C(  -844.53), SIMDE_FLOAT32_C(  -530.48), SIMDE_FLOAT32_C(   563.54),
                         SIMDE_FLOAT32_C(   165.16), SIMDE_FLOAT32_C(   384.17), SIMDE_FLOAT32_C(   149.22), SIMDE_FLOAT32_C(   712.14)),
      UINT16_C(53759),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -284.59), SIMDE_FLOAT32_C(  -286.48), SIMDE_FLOAT32_C(  -340.65), SIMDE_FLOAT32_C(   563.88)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(  -344.11), SIMDE_FLOAT32_C(   563.88),
                         SIMDE_FLOAT32_C(   640.76), SIMDE_FLOAT32_C(  -243.42), SIMDE_FLOAT32_C(   962.71), SIMDE_FLOAT32_C(   563.88),
                         SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88),
                         SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88), SIMDE_FLOAT32_C(   563.88)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -434.52), SIMDE_FLOAT32_C(   678.42), SIMDE_FLOAT32_C(   -65.20), SIMDE_FLOAT32_C(  -319.19),
                         SIMDE_FLOAT32_C(   664.97), SIMDE_FLOAT32_C(     9.01), SIMDE_FLOAT32_C(  -334.08), SIMDE_FLOAT32_C(  -870.44),
                         SIMDE_FLOAT32_C(   269.08), SIMDE_FLOAT32_C(  -345.75), SIMDE_FLOAT32_C(  -732.77), SIMDE_FLOAT32_C(   374.12),
                         SIMDE_FLOAT32_C(  -491.24), SIMDE_FLOAT32_C(   525.54), SIMDE_FLOAT32_C(  -178.26), SIMDE_FLOAT32_C(  -733.62)),
      UINT16_C(50870),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -300.63), SIMDE_FLOAT32_C(  -396.75), SIMDE_FLOAT32_C(   745.02), SIMDE_FLOAT32_C(   369.43)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   -65.20), SIMDE_FLOAT32_C(  -319.19),
                         SIMDE_FLOAT32_C(   664.97), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(  -870.44),
                         SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(  -345.75), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   369.43),
                         SIMDE_FLOAT32_C(  -491.24), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(   369.43), SIMDE_FLOAT32_C(  -733.62)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_mask_broadcastss_ps(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcastss_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { UINT16_C(25371),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   104.48), SIMDE_FLOAT32_C(   410.97), SIMDE_FLOAT32_C(   548.32), SIMDE_FLOAT32_C(   631.04)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   631.04),
                         SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   631.04), SIMDE_FLOAT32_C(   631.04)) },
    { UINT16_C(49342),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -559.08), SIMDE_FLOAT32_C(   480.19), SIMDE_FLOAT32_C(   668.18), SIMDE_FLOAT32_C(  -107.97)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -107.97),
                         SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(  -107.97), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(24820),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   811.14), SIMDE_FLOAT32_C(  -333.00), SIMDE_FLOAT32_C(  -767.79), SIMDE_FLOAT32_C(   825.12)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(   825.12),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   825.12), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(45881),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -610.29), SIMDE_FLOAT32_C(  -971.67), SIMDE_FLOAT32_C(   997.86), SIMDE_FLOAT32_C(   528.69)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(   528.69),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(   528.69),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(   528.69),
                         SIMDE_FLOAT32_C(   528.69), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   528.69)) },
    { UINT16_C(28771),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -248.84), SIMDE_FLOAT32_C(   102.57), SIMDE_FLOAT32_C(   566.99), SIMDE_FLOAT32_C(   900.54)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(   900.54),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   900.54), SIMDE_FLOAT32_C(   900.54)) },
    { UINT16_C(61611),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   318.72), SIMDE_FLOAT32_C(  -366.10), SIMDE_FLOAT32_C(   625.17), SIMDE_FLOAT32_C(   855.96)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   855.96),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   855.96), SIMDE_FLOAT32_C(   855.96)) },
    { UINT16_C(55548),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   324.92), SIMDE_FLOAT32_C(  -304.13), SIMDE_FLOAT32_C(   949.19), SIMDE_FLOAT32_C(   617.60)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(   617.60),
                         SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60),
                         SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00)) },
    { UINT16_C(15841),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   144.42), SIMDE_FLOAT32_C(    -0.50), SIMDE_FLOAT32_C(   848.91), SIMDE_FLOAT32_C(  -273.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(  -273.06),
                         SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -273.06),
                         SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(  -273.06), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(  -273.06)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_maskz_broadcastss_ps(test_vec[i].k, test_vec[i].a);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_broadcastsd_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C(  842.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(  842.49),
                         SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(  842.49),
                         SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(  842.49),
                         SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(  842.49)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  -93.61), SIMDE_FLOAT64_C( -903.55)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -903.55), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C( -903.55), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C( -903.55), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C( -903.55), SIMDE_FLOAT64_C( -903.55)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -219.93), SIMDE_FLOAT64_C( -754.32)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -754.32), SIMDE_FLOAT64_C( -754.32),
                         SIMDE_FLOAT64_C( -754.32), SIMDE_FLOAT64_C( -754.32),
                         SIMDE_FLOAT64_C( -754.32), SIMDE_FLOAT64_C( -754.32),
                         SIMDE_FLOAT64_C( -754.32), SIMDE_FLOAT64_C( -754.32)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  137.51), SIMDE_FLOAT64_C(  527.47)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  527.47), SIMDE_FLOAT64_C(  527.47),
                         SIMDE_FLOAT64_C(  527.47), SIMDE_FLOAT64_C(  527.47),
                         SIMDE_FLOAT64_C(  527.47), SIMDE_FLOAT64_C(  527.47),
                         SIMDE_FLOAT64_C(  527.47), SIMDE_FLOAT64_C(  527.47)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  596.82), SIMDE_FLOAT64_C(  365.41)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  365.41),
                         SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  365.41),
                         SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  365.41),
                         SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  365.41)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -229.59), SIMDE_FLOAT64_C( -642.88)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C( -642.88),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C( -642.88),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C( -642.88),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C( -642.88)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  308.04), SIMDE_FLOAT64_C( -958.64)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C( -958.64),
                         SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C( -958.64),
                         SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C( -958.64),
                         SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C( -958.64)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  991.16), SIMDE_FLOAT64_C( -172.14)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -172.14), SIMDE_FLOAT64_C( -172.14),
                         SIMDE_FLOAT64_C( -172.14), SIMDE_FLOAT64_C( -172.14),
                         SIMDE_FLOAT64_C( -172.14), SIMDE_FLOAT64_C( -172.14),
                         SIMDE_FLOAT64_C( -172.14), SIMDE_FLOAT64_C( -172.14)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_broadcastsd_pd(test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_mask_broadcastsd_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d src;
    simde__mmask8 k;
    simde__m128d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -746.75), SIMDE_FLOAT64_C(  634.39),
                         SIMDE_FLOAT64_C( -651.68), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C(  689.73), SIMDE_FLOAT64_C(  178.89),
                         SIMDE_FLOAT64_C( -342.04), SIMDE_FLOAT64_C( -292.58)),
      UINT8_C(162),
      simde_mm_set_pd(SIMDE_FLOAT64_C(   63.63), SIMDE_FLOAT64_C(  912.41)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C(  634.39),
                         SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C( -903.55),
                         SIMDE_FLOAT64_C(  689.73), SIMDE_FLOAT64_C(  178.89),
                         SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C( -292.58)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -287.03), SIMDE_FLOAT64_C(  851.62),
                         SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(  137.51),
                         SIMDE_FLOAT64_C( -457.60), SIMDE_FLOAT64_C(  815.46),
                         SIMDE_FLOAT64_C(  365.41), SIMDE_FLOAT64_C(  250.27)),
      UINT8_C( 66),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -149.06), SIMDE_FLOAT64_C( -899.78)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -287.03), SIMDE_FLOAT64_C( -899.78),
                         SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(  137.51),
                         SIMDE_FLOAT64_C( -457.60), SIMDE_FLOAT64_C(  815.46),
                         SIMDE_FLOAT64_C( -899.78), SIMDE_FLOAT64_C(  250.27)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C(  523.36),
                         SIMDE_FLOAT64_C( -361.34), SIMDE_FLOAT64_C( -153.87),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C(  573.19),
                         SIMDE_FLOAT64_C(  308.04), SIMDE_FLOAT64_C(  -38.88)),
      UINT8_C(115),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -758.12), SIMDE_FLOAT64_C(   12.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -958.64), SIMDE_FLOAT64_C(   12.83),
                         SIMDE_FLOAT64_C(   12.83), SIMDE_FLOAT64_C(   12.83),
                         SIMDE_FLOAT64_C( -642.88), SIMDE_FLOAT64_C(  573.19),
                         SIMDE_FLOAT64_C(   12.83), SIMDE_FLOAT64_C(   12.83)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -221.94), SIMDE_FLOAT64_C(  499.58),
                         SIMDE_FLOAT64_C(   49.04), SIMDE_FLOAT64_C( -205.69),
                         SIMDE_FLOAT64_C(  991.16), SIMDE_FLOAT64_C( -984.94),
                         SIMDE_FLOAT64_C(  224.44), SIMDE_FLOAT64_C(  644.01)),
      UINT8_C(  4),
      simde_mm_set_pd(SIMDE_FLOAT64_C(   53.80), SIMDE_FLOAT64_C( -691.82)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -221.94), SIMDE_FLOAT64_C(  499.58),
                         SIMDE_FLOAT64_C(   49.04), SIMDE_FLOAT64_C( -205.69),
                         SIMDE_FLOAT64_C(  991.16), SIMDE_FLOAT64_C( -691.82),
                         SIMDE_FLOAT64_C(  224.44), SIMDE_FLOAT64_C(  644.01)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -225.42), SIMDE_FLOAT64_C( -144.08),
                         SIMDE_FLOAT64_C( -549.59), SIMDE_FLOAT64_C(  465.78),
                         SIMDE_FLOAT64_C( -316.69), SIMDE_FLOAT64_C( -133.94),
                         SIMDE_FLOAT64_C( -646.50), SIMDE_FLOAT64_C(  160.17)),
      UINT8_C(172),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  841.59), SIMDE_FLOAT64_C(  843.47)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  843.47), SIMDE_FLOAT64_C( -144.08),
                         SIMDE_FLOAT64_C(  843.47), SIMDE_FLOAT64_C(  465.78),
                         SIMDE_FLOAT64_C(  843.47), SIMDE_FLOAT64_C(  843.47),
                         SIMDE_FLOAT64_C( -646.50), SIMDE_FLOAT64_C(  160.17)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -560.31), SIMDE_FLOAT64_C( -882.49),
                         SIMDE_FLOAT64_C(  -54.78), SIMDE_FLOAT64_C( -896.38),
                         SIMDE_FLOAT64_C(  607.65), SIMDE_FLOAT64_C( -296.43),
                         SIMDE_FLOAT64_C(  124.51), SIMDE_FLOAT64_C( -913.38)),
      UINT8_C(201),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  625.18), SIMDE_FLOAT64_C(   54.43)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   54.43), SIMDE_FLOAT64_C(   54.43),
                         SIMDE_FLOAT64_C(  -54.78), SIMDE_FLOAT64_C( -896.38),
                         SIMDE_FLOAT64_C(   54.43), SIMDE_FLOAT64_C( -296.43),
                         SIMDE_FLOAT64_C(  124.51), SIMDE_FLOAT64_C(   54.43)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  349.28), SIMDE_FLOAT64_C( -559.74),
                         SIMDE_FLOAT64_C( -116.49), SIMDE_FLOAT64_C(  342.49),
                         SIMDE_FLOAT64_C( -608.07), SIMDE_FLOAT64_C(  778.83),
                         SIMDE_FLOAT64_C( -284.17), SIMDE_FLOAT64_C( -113.81)),
      UINT8_C(234),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  312.83), SIMDE_FLOAT64_C(  -27.64)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  -27.64), SIMDE_FLOAT64_C(  -27.64),
                         SIMDE_FLOAT64_C(  -27.64), SIMDE_FLOAT64_C(  342.49),
                         SIMDE_FLOAT64_C(  -27.64), SIMDE_FLOAT64_C(  778.83),
                         SIMDE_FLOAT64_C(  -27.64), SIMDE_FLOAT64_C( -113.81)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  364.84), SIMDE_FLOAT64_C(   86.19),
                         SIMDE_FLOAT64_C( -699.29), SIMDE_FLOAT64_C(  244.26),
                         SIMDE_FLOAT64_C( -206.27), SIMDE_FLOAT64_C( -921.17),
                         SIMDE_FLOAT64_C(  483.42), SIMDE_FLOAT64_C( -935.00)),
      UINT8_C( 12),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -328.98), SIMDE_FLOAT64_C(  803.91)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  364.84), SIMDE_FLOAT64_C(   86.19),
                         SIMDE_FLOAT64_C( -699.29), SIMDE_FLOAT64_C(  244.26),
                         SIMDE_FLOAT64_C(  803.91), SIMDE_FLOAT64_C(  803.91),
                         SIMDE_FLOAT64_C(  483.42), SIMDE_FLOAT64_C( -935.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_mask_broadcastsd_pd(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_maskz_broadcastsd_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m128d a;
    simde__m512d r;
  } test_vec[8] = {
    { UINT8_C(128),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  912.41), SIMDE_FLOAT64_C(  842.49)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  842.49), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(  2),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -651.68), SIMDE_FLOAT64_C(  -93.61)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  -93.61), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(216),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  881.57), SIMDE_FLOAT64_C( -899.78)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -899.78), SIMDE_FLOAT64_C( -899.78),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C( -899.78),
                         SIMDE_FLOAT64_C( -899.78), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) },
    { UINT8_C(183),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -292.94), SIMDE_FLOAT64_C(  765.97)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(  765.97),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  765.97),
                         SIMDE_FLOAT64_C(  765.97), SIMDE_FLOAT64_C(  765.97)) },
    { UINT8_C(169),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -758.12), SIMDE_FLOAT64_C(  593.03)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  593.03), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  593.03), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  593.03), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(  593.03)) },
    { UINT8_C(243),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  523.36), SIMDE_FLOAT64_C(  761.91)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  761.91), SIMDE_FLOAT64_C(  761.91),
                         SIMDE_FLOAT64_C(  761.91), SIMDE_FLOAT64_C(  761.91),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  761.91), SIMDE_FLOAT64_C(  761.91)) },
    { UINT8_C(109),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -984.94), SIMDE_FLOAT64_C(   53.80)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   53.80),
                         SIMDE_FLOAT64_C(   53.80), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(   53.80), SIMDE_FLOAT64_C(   53.80),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(   53.80)) },
    { UINT8_C(168),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  644.01), SIMDE_FLOAT64_C(  499.58)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  499.58), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  499.58), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  499.58), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_maskz_broadcastsd_pd(test_vec[i].k, test_vec[i].a);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_castpd512_pd128(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d a;
    simde__m128d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -503.58), SIMDE_FLOAT64_C(  409.99),
                         SIMDE_FLOAT64_C(  882.79), SIMDE_FLOAT64_C(  967.41),
                         SIMDE_FLOAT64_C(  644.16), SIMDE_FLOAT64_C(   22.93),
                         SIMDE_FLOAT64_C(  629.49), SIMDE_FLOAT64_C(  -19.16)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  629.49), SIMDE_FLOAT64_C(  -19.16)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -371.76), SIMDE_FLOAT64_C( -903.70),
                         SIMDE_FLOAT64_C( -183.89), SIMDE_FLOAT64_C(   -1.20),
                         SIMDE_FLOAT64_C(  -62.86), SIMDE_FLOAT64_C(  444.19),
                         SIMDE_FLOAT64_C(  672.79), SIMDE_FLOAT64_C(  471.18)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  672.79), SIMDE_FLOAT64_C(  471.18)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -771.58), SIMDE_FLOAT64_C(  494.03),
                         SIMDE_FLOAT64_C(  138.84), SIMDE_FLOAT64_C( -195.59),
                         SIMDE_FLOAT64_C( -760.58), SIMDE_FLOAT64_C( -773.64),
                         SIMDE_FLOAT64_C(  870.25), SIMDE_FLOAT64_C(  314.25)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  870.25), SIMDE_FLOAT64_C(  314.25)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(   -2.05), SIMDE_FLOAT64_C( -418.62),
                         SIMDE_FLOAT64_C( -507.96), SIMDE_FLOAT64_C( -261.33),
                         SIMDE_FLOAT64_C( -683.39), SIMDE_FLOAT64_C(  -27.96),
                         SIMDE_FLOAT64_C(  534.55), SIMDE_FLOAT64_C(  185.44)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  534.55), SIMDE_FLOAT64_C(  185.44)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  596.68), SIMDE_FLOAT64_C( -902.04),
                         SIMDE_FLOAT64_C(  847.76), SIMDE_FLOAT64_C(  134.78),
                         SIMDE_FLOAT64_C(  116.62), SIMDE_FLOAT64_C(    7.30),
                         SIMDE_FLOAT64_C(  382.57), SIMDE_FLOAT64_C(  434.75)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  382.57), SIMDE_FLOAT64_C(  434.75)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  688.01), SIMDE_FLOAT64_C(  829.27),
                         SIMDE_FLOAT64_C( -970.03), SIMDE_FLOAT64_C(  528.04),
                         SIMDE_FLOAT64_C( -386.08), SIMDE_FLOAT64_C(  603.97),
                         SIMDE_FLOAT64_C( -132.68), SIMDE_FLOAT64_C(  463.66)),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -132.68), SIMDE_FLOAT64_C(  463.66)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  235.78), SIMDE_FLOAT64_C( -244.37),
                         SIMDE_FLOAT64_C(  846.69), SIMDE_FLOAT64_C(  794.12),
                         SIMDE_FLOAT64_C(  672.11), SIMDE_FLOAT64_C( -445.78),
                         SIMDE_FLOAT64_C(  396.01), SIMDE_FLOAT64_C( -284.10)),
      simde_mm_set_pd(SIMDE_FLOAT64_C(  396.01), SIMDE_FLOAT64_C( -284.10)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  603.46), SIMDE_FLOAT64_C( -988.15),
                         SIMDE_FLOAT64_C( -342.31), SIMDE_FLOAT64_C( -282.11),
                         SIMDE_FLOAT64_C(  379.72), SIMDE_FLOAT64_C( -680.63),
                         SIMDE_FLOAT64_C( -541.51), SIMDE_FLOAT64_C(  -40.90)),
      simde_mm_set_pd(SIMDE_FLOAT64_C( -541.51), SIMDE_FLOAT64_C(  -40.90)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128d r = simde_mm512_castpd512_pd128(test_vec[i].a);
    simde_assert_m128d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_castpd512_pd256(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d a;
    simde__m256d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  414.39), SIMDE_FLOAT64_C(  -15.63),
                         SIMDE_FLOAT64_C(  546.05), SIMDE_FLOAT64_C( -960.01),
                         SIMDE_FLOAT64_C( -752.11), SIMDE_FLOAT64_C( -702.83),
                         SIMDE_FLOAT64_C( -642.16), SIMDE_FLOAT64_C(  617.65)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -752.11), SIMDE_FLOAT64_C( -702.83),
                         SIMDE_FLOAT64_C( -642.16), SIMDE_FLOAT64_C(  617.65)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -964.21), SIMDE_FLOAT64_C( -388.73),
                         SIMDE_FLOAT64_C(  322.52), SIMDE_FLOAT64_C(  699.65),
                         SIMDE_FLOAT64_C(  390.02), SIMDE_FLOAT64_C( -490.64),
                         SIMDE_FLOAT64_C( -297.41), SIMDE_FLOAT64_C(  563.23)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  390.02), SIMDE_FLOAT64_C( -490.64),
                         SIMDE_FLOAT64_C( -297.41), SIMDE_FLOAT64_C(  563.23)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  920.55), SIMDE_FLOAT64_C(   39.56),
                         SIMDE_FLOAT64_C(  750.72), SIMDE_FLOAT64_C(  440.72),
                         SIMDE_FLOAT64_C( -246.45), SIMDE_FLOAT64_C(  981.88),
                         SIMDE_FLOAT64_C( -299.81), SIMDE_FLOAT64_C(  -98.82)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -246.45), SIMDE_FLOAT64_C(  981.88),
                         SIMDE_FLOAT64_C( -299.81), SIMDE_FLOAT64_C(  -98.82)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -471.70), SIMDE_FLOAT64_C( -359.51),
                         SIMDE_FLOAT64_C(  724.77), SIMDE_FLOAT64_C( -820.10),
                         SIMDE_FLOAT64_C(  777.44), SIMDE_FLOAT64_C( -220.31),
                         SIMDE_FLOAT64_C(  567.27), SIMDE_FLOAT64_C(  977.08)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  777.44), SIMDE_FLOAT64_C( -220.31),
                         SIMDE_FLOAT64_C(  567.27), SIMDE_FLOAT64_C(  977.08)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  438.55), SIMDE_FLOAT64_C( -230.11),
                         SIMDE_FLOAT64_C(  759.60), SIMDE_FLOAT64_C(  615.37),
                         SIMDE_FLOAT64_C( -356.18), SIMDE_FLOAT64_C(  777.70),
                         SIMDE_FLOAT64_C( -385.32), SIMDE_FLOAT64_C( -478.17)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C( -356.18), SIMDE_FLOAT64_C(  777.70),
                         SIMDE_FLOAT64_C( -385.32), SIMDE_FLOAT64_C( -478.17)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -337.84), SIMDE_FLOAT64_C( -274.76),
                         SIMDE_FLOAT64_C(  865.32), SIMDE_FLOAT64_C( -257.44),
                         SIMDE_FLOAT64_C(  325.76), SIMDE_FLOAT64_C(  -67.76),
                         SIMDE_FLOAT64_C( -238.87), SIMDE_FLOAT64_C( -384.04)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  325.76), SIMDE_FLOAT64_C(  -67.76),
                         SIMDE_FLOAT64_C( -238.87), SIMDE_FLOAT64_C( -384.04)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  120.04), SIMDE_FLOAT64_C(  416.14),
                         SIMDE_FLOAT64_C(  299.72), SIMDE_FLOAT64_C(  991.84),
                         SIMDE_FLOAT64_C(  325.55), SIMDE_FLOAT64_C(  685.94),
                         SIMDE_FLOAT64_C( -182.11), SIMDE_FLOAT64_C(  943.15)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  325.55), SIMDE_FLOAT64_C(  685.94),
                         SIMDE_FLOAT64_C( -182.11), SIMDE_FLOAT64_C(  943.15)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  556.40), SIMDE_FLOAT64_C( -775.19),
                         SIMDE_FLOAT64_C( -814.91), SIMDE_FLOAT64_C(   73.06),
                         SIMDE_FLOAT64_C(  317.95), SIMDE_FLOAT64_C( -552.86),
                         SIMDE_FLOAT64_C( -767.41), SIMDE_FLOAT64_C(  130.66)),
      simde_mm256_set_pd(SIMDE_FLOAT64_C(  317.95), SIMDE_FLOAT64_C( -552.86),
                         SIMDE_FLOAT64_C( -767.41), SIMDE_FLOAT64_C(  130.66)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256d r = simde_mm512_castpd512_pd256(test_vec[i].a);
    simde_assert_m256d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_castpd128_pd512(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  605.48), SIMDE_FLOAT64_C(  349.95)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  605.48), SIMDE_FLOAT64_C(  349.95)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  125.01), SIMDE_FLOAT64_C(  -52.81)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  125.01), SIMDE_FLOAT64_C(  -52.81)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -137.83), SIMDE_FLOAT64_C( -367.46)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -137.83), SIMDE_FLOAT64_C( -367.46)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -706.70), SIMDE_FLOAT64_C(  -54.57)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -706.70), SIMDE_FLOAT64_C(  -54.57)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -477.44), SIMDE_FLOAT64_C( -678.80)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -477.44), SIMDE_FLOAT64_C( -678.80)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C( -750.01), SIMDE_FLOAT64_C(  943.83)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -750.01), SIMDE_FLOAT64_C(  943.83)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  329.32), SIMDE_FLOAT64_C(  600.02)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  329.32), SIMDE_FLOAT64_C(  600.02)) },
    { simde_mm_set_pd(SIMDE_FLOAT64_C(  495.92), SIMDE_FLOAT64_C(  855.97)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  495.92), SIMDE_FLOAT64_C(  855.97)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_castpd128_pd512(test_vec[i].a);
    r=simde_mm512_maskz_mov_pd(UINT8_C(3), r);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_castpd256_pd512(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256d a;
    simde__m512d r;
  } test_vec[8] = {
        { simde_mm256_set_pd(SIMDE_FLOAT64_C( -610.76), SIMDE_FLOAT64_C( -445.99),
                         SIMDE_FLOAT64_C( -292.19), SIMDE_FLOAT64_C(  327.18)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -610.76), SIMDE_FLOAT64_C( -445.99),
                         SIMDE_FLOAT64_C( -292.19), SIMDE_FLOAT64_C(  327.18)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  498.95), SIMDE_FLOAT64_C(  765.72),
                         SIMDE_FLOAT64_C( -978.89), SIMDE_FLOAT64_C( -427.55)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  498.95), SIMDE_FLOAT64_C(  765.72),
                         SIMDE_FLOAT64_C( -978.89), SIMDE_FLOAT64_C( -427.55)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -991.71), SIMDE_FLOAT64_C(  -66.46),
                         SIMDE_FLOAT64_C(  -87.69), SIMDE_FLOAT64_C(  225.73)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -991.71), SIMDE_FLOAT64_C(  -66.46),
                         SIMDE_FLOAT64_C(  -87.69), SIMDE_FLOAT64_C(  225.73)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -983.02), SIMDE_FLOAT64_C(  768.15),
                         SIMDE_FLOAT64_C(  423.32), SIMDE_FLOAT64_C( -601.96)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -983.02), SIMDE_FLOAT64_C(  768.15),
                         SIMDE_FLOAT64_C(  423.32), SIMDE_FLOAT64_C( -601.96)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  269.59), SIMDE_FLOAT64_C(  -12.98),
                         SIMDE_FLOAT64_C(  833.22), SIMDE_FLOAT64_C( -167.34)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  269.59), SIMDE_FLOAT64_C(  -12.98),
                         SIMDE_FLOAT64_C(  833.22), SIMDE_FLOAT64_C( -167.34)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -976.56), SIMDE_FLOAT64_C( -631.12),
                         SIMDE_FLOAT64_C( -613.64), SIMDE_FLOAT64_C( -574.89)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -976.56), SIMDE_FLOAT64_C( -631.12),
                         SIMDE_FLOAT64_C( -613.64), SIMDE_FLOAT64_C( -574.89)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C(  900.29), SIMDE_FLOAT64_C(  197.75),
                         SIMDE_FLOAT64_C( -976.11), SIMDE_FLOAT64_C( -958.45)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(  900.29), SIMDE_FLOAT64_C(  197.75),
                         SIMDE_FLOAT64_C( -976.11), SIMDE_FLOAT64_C( -958.45)) },
    { simde_mm256_set_pd(SIMDE_FLOAT64_C( -467.55), SIMDE_FLOAT64_C( -302.23),
                         SIMDE_FLOAT64_C(  207.74), SIMDE_FLOAT64_C( -217.89)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C(    0.00), SIMDE_FLOAT64_C(    0.00),
                         SIMDE_FLOAT64_C( -467.55), SIMDE_FLOAT64_C( -302.23),
                         SIMDE_FLOAT64_C(  207.74), SIMDE_FLOAT64_C( -217.89)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512d r = simde_mm512_castpd256_pd512(test_vec[i].a);
    r=simde_mm512_maskz_mov_pd(UINT8_C(15), r);
    simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_castps512_ps128(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 a;
    simde__m128 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   764.32), SIMDE_FLOAT32_C(  -951.31), SIMDE_FLOAT32_C(   179.06), SIMDE_FLOAT32_C(  -437.66),
                         SIMDE_FLOAT32_C(   402.64), SIMDE_FLOAT32_C(   734.29), SIMDE_FLOAT32_C(   267.34), SIMDE_FLOAT32_C(   208.00),
                         SIMDE_FLOAT32_C(   587.55), SIMDE_FLOAT32_C(  -635.96), SIMDE_FLOAT32_C(  -958.84), SIMDE_FLOAT32_C(  -271.31),
                         SIMDE_FLOAT32_C(   711.90), SIMDE_FLOAT32_C(   959.36), SIMDE_FLOAT32_C(   988.07), SIMDE_FLOAT32_C(  -648.23)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   711.90), SIMDE_FLOAT32_C(   959.36), SIMDE_FLOAT32_C(   988.07), SIMDE_FLOAT32_C(  -648.23)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   193.72), SIMDE_FLOAT32_C(  -347.47), SIMDE_FLOAT32_C(   437.71), SIMDE_FLOAT32_C(  -756.77),
                         SIMDE_FLOAT32_C(  -591.14), SIMDE_FLOAT32_C(  -431.84), SIMDE_FLOAT32_C(  -564.12), SIMDE_FLOAT32_C(   252.26),
                         SIMDE_FLOAT32_C(  -784.50), SIMDE_FLOAT32_C(   149.30), SIMDE_FLOAT32_C(  -246.02), SIMDE_FLOAT32_C(   -60.58),
                         SIMDE_FLOAT32_C(   384.90), SIMDE_FLOAT32_C(   156.04), SIMDE_FLOAT32_C(  -595.88), SIMDE_FLOAT32_C(  -574.45)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   384.90), SIMDE_FLOAT32_C(   156.04), SIMDE_FLOAT32_C(  -595.88), SIMDE_FLOAT32_C(  -574.45)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -425.80), SIMDE_FLOAT32_C(   886.22), SIMDE_FLOAT32_C(  -478.46), SIMDE_FLOAT32_C(  -993.15),
                         SIMDE_FLOAT32_C(  -912.29), SIMDE_FLOAT32_C(  -564.52), SIMDE_FLOAT32_C(   -25.07), SIMDE_FLOAT32_C(  -469.67),
                         SIMDE_FLOAT32_C(   828.84), SIMDE_FLOAT32_C(  -525.49), SIMDE_FLOAT32_C(  -521.07), SIMDE_FLOAT32_C(   132.36),
                         SIMDE_FLOAT32_C(   368.34), SIMDE_FLOAT32_C(   996.83), SIMDE_FLOAT32_C(   514.03), SIMDE_FLOAT32_C(   350.58)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   368.34), SIMDE_FLOAT32_C(   996.83), SIMDE_FLOAT32_C(   514.03), SIMDE_FLOAT32_C(   350.58)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -806.26), SIMDE_FLOAT32_C(  -958.10), SIMDE_FLOAT32_C(   -61.10), SIMDE_FLOAT32_C(   698.78),
                         SIMDE_FLOAT32_C(   893.49), SIMDE_FLOAT32_C(  -774.73), SIMDE_FLOAT32_C(  -469.54), SIMDE_FLOAT32_C(    88.88),
                         SIMDE_FLOAT32_C(  -797.01), SIMDE_FLOAT32_C(  -143.15), SIMDE_FLOAT32_C(  -201.38), SIMDE_FLOAT32_C(  -439.08),
                         SIMDE_FLOAT32_C(   122.21), SIMDE_FLOAT32_C(   251.76), SIMDE_FLOAT32_C(   853.77), SIMDE_FLOAT32_C(  -306.48)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   122.21), SIMDE_FLOAT32_C(   251.76), SIMDE_FLOAT32_C(   853.77), SIMDE_FLOAT32_C(  -306.48)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   995.27), SIMDE_FLOAT32_C(   762.38), SIMDE_FLOAT32_C(   849.07), SIMDE_FLOAT32_C(   533.84),
                         SIMDE_FLOAT32_C(   128.75), SIMDE_FLOAT32_C(   581.04), SIMDE_FLOAT32_C(   680.92), SIMDE_FLOAT32_C(    97.54),
                         SIMDE_FLOAT32_C(   183.54), SIMDE_FLOAT32_C(    89.55), SIMDE_FLOAT32_C(  -801.20), SIMDE_FLOAT32_C(  -458.50),
                         SIMDE_FLOAT32_C(  -944.26), SIMDE_FLOAT32_C(  -648.24), SIMDE_FLOAT32_C(  -216.16), SIMDE_FLOAT32_C(   273.51)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(  -944.26), SIMDE_FLOAT32_C(  -648.24), SIMDE_FLOAT32_C(  -216.16), SIMDE_FLOAT32_C(   273.51)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   563.30), SIMDE_FLOAT32_C(  -488.59), SIMDE_FLOAT32_C(   -41.61), SIMDE_FLOAT32_C(  -745.38),
                         SIMDE_FLOAT32_C(  -156.54), SIMDE_FLOAT32_C(  -651.09), SIMDE_FLOAT32_C(   393.55), SIMDE_FLOAT32_C(  -610.90),
                         SIMDE_FLOAT32_C(   656.44), SIMDE_FLOAT32_C(   254.16), SIMDE_FLOAT32_C(   -27.43), SIMDE_FLOAT32_C(  -862.48),
                         SIMDE_FLOAT32_C(   639.71), SIMDE_FLOAT32_C(  -674.10), SIMDE_FLOAT32_C(  -151.15), SIMDE_FLOAT32_C(  -736.28)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   639.71), SIMDE_FLOAT32_C(  -674.10), SIMDE_FLOAT32_C(  -151.15), SIMDE_FLOAT32_C(  -736.28)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -997.30), SIMDE_FLOAT32_C(   958.88), SIMDE_FLOAT32_C(   414.33), SIMDE_FLOAT32_C(  -486.71),
                         SIMDE_FLOAT32_C(   347.81), SIMDE_FLOAT32_C(   -73.64), SIMDE_FLOAT32_C(   301.45), SIMDE_FLOAT32_C(  -935.75),
                         SIMDE_FLOAT32_C(   554.43), SIMDE_FLOAT32_C(   239.85), SIMDE_FLOAT32_C(  -172.14), SIMDE_FLOAT32_C(  -200.20),
                         SIMDE_FLOAT32_C(   383.84), SIMDE_FLOAT32_C(    -4.79), SIMDE_FLOAT32_C(   659.24), SIMDE_FLOAT32_C(   178.95)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   383.84), SIMDE_FLOAT32_C(    -4.79), SIMDE_FLOAT32_C(   659.24), SIMDE_FLOAT32_C(   178.95)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   282.49), SIMDE_FLOAT32_C(  -287.66), SIMDE_FLOAT32_C(  -300.74), SIMDE_FLOAT32_C(  -282.12),
                         SIMDE_FLOAT32_C(   228.49), SIMDE_FLOAT32_C(   194.42), SIMDE_FLOAT32_C(   232.99), SIMDE_FLOAT32_C(   312.60),
                         SIMDE_FLOAT32_C(    59.18), SIMDE_FLOAT32_C(   591.94), SIMDE_FLOAT32_C(   490.41), SIMDE_FLOAT32_C(   768.05),
                         SIMDE_FLOAT32_C(   628.85), SIMDE_FLOAT32_C(   497.59), SIMDE_FLOAT32_C(   700.20), SIMDE_FLOAT32_C(  -694.51)),
      simde_mm_set_ps(SIMDE_FLOAT32_C(   628.85), SIMDE_FLOAT32_C(   497.59), SIMDE_FLOAT32_C(   700.20), SIMDE_FLOAT32_C(  -694.51)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128 r = simde_mm512_castps512_ps128(test_vec[i].a);
    simde_assert_m128_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_castps512_ps256(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 a;
    simde__m256 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   516.61), SIMDE_FLOAT32_C(   494.30), SIMDE_FLOAT32_C(   266.21), SIMDE_FLOAT32_C(   450.63),
                         SIMDE_FLOAT32_C(  -862.95), SIMDE_FLOAT32_C(  -528.18), SIMDE_FLOAT32_C(   206.23), SIMDE_FLOAT32_C(  -212.40),
                         SIMDE_FLOAT32_C(   805.40), SIMDE_FLOAT32_C(  -902.72), SIMDE_FLOAT32_C(  -631.10), SIMDE_FLOAT32_C(  -480.24),
                         SIMDE_FLOAT32_C(   577.49), SIMDE_FLOAT32_C(   490.33), SIMDE_FLOAT32_C(  -227.96), SIMDE_FLOAT32_C(   938.93)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(   805.40), SIMDE_FLOAT32_C(  -902.72),
                         SIMDE_FLOAT32_C(  -631.10), SIMDE_FLOAT32_C(  -480.24),
                         SIMDE_FLOAT32_C(   577.49), SIMDE_FLOAT32_C(   490.33),
                         SIMDE_FLOAT32_C(  -227.96), SIMDE_FLOAT32_C(   938.93)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   205.80), SIMDE_FLOAT32_C(  -896.03), SIMDE_FLOAT32_C(  -533.62), SIMDE_FLOAT32_C(  -495.13),
                         SIMDE_FLOAT32_C(   100.81), SIMDE_FLOAT32_C(   789.28), SIMDE_FLOAT32_C(   976.66), SIMDE_FLOAT32_C(  -300.54),
                         SIMDE_FLOAT32_C(  -430.09), SIMDE_FLOAT32_C(   725.00), SIMDE_FLOAT32_C(    80.84), SIMDE_FLOAT32_C(   -71.63),
                         SIMDE_FLOAT32_C(   804.31), SIMDE_FLOAT32_C(   798.70), SIMDE_FLOAT32_C(   251.20), SIMDE_FLOAT32_C(   -32.38)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(  -430.09), SIMDE_FLOAT32_C(   725.00),
                         SIMDE_FLOAT32_C(    80.84), SIMDE_FLOAT32_C(   -71.63),
                         SIMDE_FLOAT32_C(   804.31), SIMDE_FLOAT32_C(   798.70),
                         SIMDE_FLOAT32_C(   251.20), SIMDE_FLOAT32_C(   -32.38)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   635.87), SIMDE_FLOAT32_C(    20.88), SIMDE_FLOAT32_C(   462.09), SIMDE_FLOAT32_C(   342.91),
                         SIMDE_FLOAT32_C(  -645.79), SIMDE_FLOAT32_C(   103.74), SIMDE_FLOAT32_C(  -376.41), SIMDE_FLOAT32_C(   439.99),
                         SIMDE_FLOAT32_C(   895.02), SIMDE_FLOAT32_C(   204.80), SIMDE_FLOAT32_C(     1.77), SIMDE_FLOAT32_C(   909.62),
                         SIMDE_FLOAT32_C(  -446.42), SIMDE_FLOAT32_C(     7.95), SIMDE_FLOAT32_C(  -614.77), SIMDE_FLOAT32_C(   315.09)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(   895.02), SIMDE_FLOAT32_C(   204.80),
                         SIMDE_FLOAT32_C(     1.77), SIMDE_FLOAT32_C(   909.62),
                         SIMDE_FLOAT32_C(  -446.42), SIMDE_FLOAT32_C(     7.95),
                         SIMDE_FLOAT32_C(  -614.77), SIMDE_FLOAT32_C(   315.09)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -372.65), SIMDE_FLOAT32_C(   316.64), SIMDE_FLOAT32_C(   -78.40), SIMDE_FLOAT32_C(   677.31),
                         SIMDE_FLOAT32_C(   299.82), SIMDE_FLOAT32_C(   452.58), SIMDE_FLOAT32_C(   876.09), SIMDE_FLOAT32_C(   746.57),
                         SIMDE_FLOAT32_C(   916.69), SIMDE_FLOAT32_C(  -811.99), SIMDE_FLOAT32_C(   637.74), SIMDE_FLOAT32_C(   -83.23),
                         SIMDE_FLOAT32_C(   289.79), SIMDE_FLOAT32_C(  -689.87), SIMDE_FLOAT32_C(   493.07), SIMDE_FLOAT32_C(   108.56)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(   916.69), SIMDE_FLOAT32_C(  -811.99),
                         SIMDE_FLOAT32_C(   637.74), SIMDE_FLOAT32_C(   -83.23),
                         SIMDE_FLOAT32_C(   289.79), SIMDE_FLOAT32_C(  -689.87),
                         SIMDE_FLOAT32_C(   493.07), SIMDE_FLOAT32_C(   108.56)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -296.96), SIMDE_FLOAT32_C(   756.35), SIMDE_FLOAT32_C(  -395.61), SIMDE_FLOAT32_C(  -521.80),
                         SIMDE_FLOAT32_C(  -721.04), SIMDE_FLOAT32_C(   -68.70), SIMDE_FLOAT32_C(  -334.12), SIMDE_FLOAT32_C(   -56.72),
                         SIMDE_FLOAT32_C(    59.02), SIMDE_FLOAT32_C(   -66.01), SIMDE_FLOAT32_C(  -767.54), SIMDE_FLOAT32_C(   502.66),
                         SIMDE_FLOAT32_C(  -591.60), SIMDE_FLOAT32_C(  -826.33), SIMDE_FLOAT32_C(  -334.73), SIMDE_FLOAT32_C(   -13.18)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(    59.02), SIMDE_FLOAT32_C(   -66.01),
                         SIMDE_FLOAT32_C(  -767.54), SIMDE_FLOAT32_C(   502.66),
                         SIMDE_FLOAT32_C(  -591.60), SIMDE_FLOAT32_C(  -826.33),
                         SIMDE_FLOAT32_C(  -334.73), SIMDE_FLOAT32_C(   -13.18)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(    -2.88), SIMDE_FLOAT32_C(  -966.98), SIMDE_FLOAT32_C(   854.50), SIMDE_FLOAT32_C(   668.99),
                         SIMDE_FLOAT32_C(   259.81), SIMDE_FLOAT32_C(   -77.24), SIMDE_FLOAT32_C(  -462.32), SIMDE_FLOAT32_C(   688.88),
                         SIMDE_FLOAT32_C(   754.80), SIMDE_FLOAT32_C(   986.78), SIMDE_FLOAT32_C(   961.39), SIMDE_FLOAT32_C(   850.00),
                         SIMDE_FLOAT32_C(   721.06), SIMDE_FLOAT32_C(   -70.04), SIMDE_FLOAT32_C(   136.98), SIMDE_FLOAT32_C(   949.65)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(   754.80), SIMDE_FLOAT32_C(   986.78),
                         SIMDE_FLOAT32_C(   961.39), SIMDE_FLOAT32_C(   850.00),
                         SIMDE_FLOAT32_C(   721.06), SIMDE_FLOAT32_C(   -70.04),
                         SIMDE_FLOAT32_C(   136.98), SIMDE_FLOAT32_C(   949.65)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   841.78), SIMDE_FLOAT32_C(  -315.64), SIMDE_FLOAT32_C(   747.52), SIMDE_FLOAT32_C(   946.88),
                         SIMDE_FLOAT32_C(   871.78), SIMDE_FLOAT32_C(   775.22), SIMDE_FLOAT32_C(  -159.38), SIMDE_FLOAT32_C(   270.48),
                         SIMDE_FLOAT32_C(  -276.64), SIMDE_FLOAT32_C(   602.71), SIMDE_FLOAT32_C(  -825.17), SIMDE_FLOAT32_C(  -882.05),
                         SIMDE_FLOAT32_C(   -12.61), SIMDE_FLOAT32_C(  -809.78), SIMDE_FLOAT32_C(  -114.18), SIMDE_FLOAT32_C(   260.44)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(  -276.64), SIMDE_FLOAT32_C(   602.71),
                         SIMDE_FLOAT32_C(  -825.17), SIMDE_FLOAT32_C(  -882.05),
                         SIMDE_FLOAT32_C(   -12.61), SIMDE_FLOAT32_C(  -809.78),
                         SIMDE_FLOAT32_C(  -114.18), SIMDE_FLOAT32_C(   260.44)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   258.67), SIMDE_FLOAT32_C(   265.23), SIMDE_FLOAT32_C(   761.29), SIMDE_FLOAT32_C(  -932.57),
                         SIMDE_FLOAT32_C(   366.86), SIMDE_FLOAT32_C(  -726.22), SIMDE_FLOAT32_C(   179.78), SIMDE_FLOAT32_C(   965.96),
                         SIMDE_FLOAT32_C(  -420.83), SIMDE_FLOAT32_C(  -371.51), SIMDE_FLOAT32_C(   392.50), SIMDE_FLOAT32_C(   925.30),
                         SIMDE_FLOAT32_C(   990.85), SIMDE_FLOAT32_C(   -34.78), SIMDE_FLOAT32_C(   -24.58), SIMDE_FLOAT32_C(  -150.35)),
      simde_mm256_set_ps(SIMDE_FLOAT32_C(  -420.83), SIMDE_FLOAT32_C(  -371.51),
                         SIMDE_FLOAT32_C(   392.50), SIMDE_FLOAT32_C(   925.30),
                         SIMDE_FLOAT32_C(   990.85), SIMDE_FLOAT32_C(   -34.78),
                         SIMDE_FLOAT32_C(   -24.58), SIMDE_FLOAT32_C(  -150.35)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256 r = simde_mm512_castps512_ps256(test_vec[i].a);
    simde_assert_m256_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_castps_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -475.09), SIMDE_FLOAT32_C(   736.84), SIMDE_FLOAT32_C(  -702.97), SIMDE_FLOAT32_C(  -433.94),
                         SIMDE_FLOAT32_C(   854.93), SIMDE_FLOAT32_C(  -157.11), SIMDE_FLOAT32_C(    43.61), SIMDE_FLOAT32_C(  -310.79),
                         SIMDE_FLOAT32_C(  -893.63), SIMDE_FLOAT32_C(    37.00), SIMDE_FLOAT32_C(   245.96), SIMDE_FLOAT32_C(   381.92),
                         SIMDE_FLOAT32_C(  -702.80), SIMDE_FLOAT32_C(   931.47), SIMDE_FLOAT32_C(   429.22), SIMDE_FLOAT32_C(  -639.06)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -475.09), SIMDE_FLOAT32_C(   736.84), SIMDE_FLOAT32_C(  -702.97), SIMDE_FLOAT32_C(  -433.94),
                         SIMDE_FLOAT32_C(   854.93), SIMDE_FLOAT32_C(  -157.11), SIMDE_FLOAT32_C(    43.61), SIMDE_FLOAT32_C(  -310.79),
                         SIMDE_FLOAT32_C(  -893.63), SIMDE_FLOAT32_C(    37.00), SIMDE_FLOAT32_C(   245.96), SIMDE_FLOAT32_C(   381.92),
                         SIMDE_FLOAT32_C(  -702.80), SIMDE_FLOAT32_C(   931.47), SIMDE_FLOAT32_C(   429.22), SIMDE_FLOAT32_C(  -639.06)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -147.89), SIMDE_FLOAT32_C(   347.19), SIMDE_FLOAT32_C(   203.54), SIMDE_FLOAT32_C(  -988.99),
                         SIMDE_FLOAT32_C(   306.65), SIMDE_FLOAT32_C(   641.36), SIMDE_FLOAT32_C(  -965.49), SIMDE_FLOAT32_C(  -587.11),
                         SIMDE_FLOAT32_C(   724.40), SIMDE_FLOAT32_C(  -636.19), SIMDE_FLOAT32_C(  -368.06), SIMDE_FLOAT32_C(   887.68),
                         SIMDE_FLOAT32_C(  -970.41), SIMDE_FLOAT32_C(  -566.45), SIMDE_FLOAT32_C(   791.35), SIMDE_FLOAT32_C(  -284.64)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -147.89), SIMDE_FLOAT32_C(   347.19), SIMDE_FLOAT32_C(   203.54), SIMDE_FLOAT32_C(  -988.99),
                         SIMDE_FLOAT32_C(   306.65), SIMDE_FLOAT32_C(   641.36), SIMDE_FLOAT32_C(  -965.49), SIMDE_FLOAT32_C(  -587.11),
                         SIMDE_FLOAT32_C(   724.40), SIMDE_FLOAT32_C(  -636.19), SIMDE_FLOAT32_C(  -368.06), SIMDE_FLOAT32_C(   887.68),
                         SIMDE_FLOAT32_C(  -970.41), SIMDE_FLOAT32_C(  -566.45), SIMDE_FLOAT32_C(   791.35), SIMDE_FLOAT32_C(  -284.64)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   -73.77), SIMDE_FLOAT32_C(  -499.31), SIMDE_FLOAT32_C(   113.82), SIMDE_FLOAT32_C(  -847.95),
                         SIMDE_FLOAT32_C(  -593.73), SIMDE_FLOAT32_C(   978.06), SIMDE_FLOAT32_C(   354.00), SIMDE_FLOAT32_C(  -120.89),
                         SIMDE_FLOAT32_C(  -554.86), SIMDE_FLOAT32_C(  -105.83), SIMDE_FLOAT32_C(   247.41), SIMDE_FLOAT32_C(   728.79),
                         SIMDE_FLOAT32_C(  -359.78), SIMDE_FLOAT32_C(  -375.68), SIMDE_FLOAT32_C(  -456.84), SIMDE_FLOAT32_C(  -859.51)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -73.77), SIMDE_FLOAT32_C(  -499.31), SIMDE_FLOAT32_C(   113.82), SIMDE_FLOAT32_C(  -847.95),
                         SIMDE_FLOAT32_C(  -593.73), SIMDE_FLOAT32_C(   978.06), SIMDE_FLOAT32_C(   354.00), SIMDE_FLOAT32_C(  -120.89),
                         SIMDE_FLOAT32_C(  -554.86), SIMDE_FLOAT32_C(  -105.83), SIMDE_FLOAT32_C(   247.41), SIMDE_FLOAT32_C(   728.79),
                         SIMDE_FLOAT32_C(  -359.78), SIMDE_FLOAT32_C(  -375.68), SIMDE_FLOAT32_C(  -456.84), SIMDE_FLOAT32_C(  -859.51)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -892.63), SIMDE_FLOAT32_C(   554.75), SIMDE_FLOAT32_C(   786.31), SIMDE_FLOAT32_C(   695.92),
                         SIMDE_FLOAT32_C(  -629.78), SIMDE_FLOAT32_C(   376.00), SIMDE_FLOAT32_C(   490.47), SIMDE_FLOAT32_C(  -658.22),
                         SIMDE_FLOAT32_C(  -993.58), SIMDE_FLOAT32_C(   931.16), SIMDE_FLOAT32_C(   896.06), SIMDE_FLOAT32_C(  -621.22),
                         SIMDE_FLOAT32_C(   936.87), SIMDE_FLOAT32_C(  -330.96), SIMDE_FLOAT32_C(   683.22), SIMDE_FLOAT32_C(   273.63)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -892.63), SIMDE_FLOAT32_C(   554.75), SIMDE_FLOAT32_C(   786.31), SIMDE_FLOAT32_C(   695.92),
                         SIMDE_FLOAT32_C(  -629.78), SIMDE_FLOAT32_C(   376.00), SIMDE_FLOAT32_C(   490.47), SIMDE_FLOAT32_C(  -658.22),
                         SIMDE_FLOAT32_C(  -993.58), SIMDE_FLOAT32_C(   931.16), SIMDE_FLOAT32_C(   896.06), SIMDE_FLOAT32_C(  -621.22),
                         SIMDE_FLOAT32_C(   936.87), SIMDE_FLOAT32_C(  -330.96), SIMDE_FLOAT32_C(   683.22), SIMDE_FLOAT32_C(   273.63)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -464.48), SIMDE_FLOAT32_C(   711.99), SIMDE_FLOAT32_C(   146.80), SIMDE_FLOAT32_C(  -790.19),
                         SIMDE_FLOAT32_C(  -196.31), SIMDE_FLOAT32_C(   -19.08), SIMDE_FLOAT32_C(   390.50), SIMDE_FLOAT32_C(  -693.04),
                         SIMDE_FLOAT32_C(   121.41), SIMDE_FLOAT32_C(  -677.25), SIMDE_FLOAT32_C(   840.59), SIMDE_FLOAT32_C(   653.82),
                         SIMDE_FLOAT32_C(  -341.60), SIMDE_FLOAT32_C(  -152.67), SIMDE_FLOAT32_C(  -910.13), SIMDE_FLOAT32_C(   648.18)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -464.48), SIMDE_FLOAT32_C(   711.99), SIMDE_FLOAT32_C(   146.80), SIMDE_FLOAT32_C(  -790.19),
                         SIMDE_FLOAT32_C(  -196.31), SIMDE_FLOAT32_C(   -19.08), SIMDE_FLOAT32_C(   390.50), SIMDE_FLOAT32_C(  -693.04),
                         SIMDE_FLOAT32_C(   121.41), SIMDE_FLOAT32_C(  -677.25), SIMDE_FLOAT32_C(   840.59), SIMDE_FLOAT32_C(   653.82),
                         SIMDE_FLOAT32_C(  -341.60), SIMDE_FLOAT32_C(  -152.67), SIMDE_FLOAT32_C(  -910.13), SIMDE_FLOAT32_C(   648.18)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   610.99), SIMDE_FLOAT32_C(   354.13), SIMDE_FLOAT32_C(  -805.98), SIMDE_FLOAT32_C(   470.53),
                         SIMDE_FLOAT32_C(   848.62), SIMDE_FLOAT32_C(  -853.14), SIMDE_FLOAT32_C(   898.19), SIMDE_FLOAT32_C(  -186.52),
                         SIMDE_FLOAT32_C(  -954.10), SIMDE_FLOAT32_C(   887.52), SIMDE_FLOAT32_C(   607.34), SIMDE_FLOAT32_C(    42.34),
                         SIMDE_FLOAT32_C(  -842.20), SIMDE_FLOAT32_C(   627.26), SIMDE_FLOAT32_C(   417.38), SIMDE_FLOAT32_C(  -300.32)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   610.99), SIMDE_FLOAT32_C(   354.13), SIMDE_FLOAT32_C(  -805.98), SIMDE_FLOAT32_C(   470.53),
                         SIMDE_FLOAT32_C(   848.62), SIMDE_FLOAT32_C(  -853.14), SIMDE_FLOAT32_C(   898.19), SIMDE_FLOAT32_C(  -186.52),
                         SIMDE_FLOAT32_C(  -954.10), SIMDE_FLOAT32_C(   887.52), SIMDE_FLOAT32_C(   607.34), SIMDE_FLOAT32_C(    42.34),
                         SIMDE_FLOAT32_C(  -842.20), SIMDE_FLOAT32_C(   627.26), SIMDE_FLOAT32_C(   417.38), SIMDE_FLOAT32_C(  -300.32)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(   -91.92), SIMDE_FLOAT32_C(  -128.53), SIMDE_FLOAT32_C(  -157.39), SIMDE_FLOAT32_C(   922.31),
                         SIMDE_FLOAT32_C(    68.36), SIMDE_FLOAT32_C(  -842.81), SIMDE_FLOAT32_C(    60.56), SIMDE_FLOAT32_C(   689.02),
                         SIMDE_FLOAT32_C(   -70.29), SIMDE_FLOAT32_C(  -753.33), SIMDE_FLOAT32_C(  -999.31), SIMDE_FLOAT32_C(    72.69),
                         SIMDE_FLOAT32_C(    47.99), SIMDE_FLOAT32_C(   903.18), SIMDE_FLOAT32_C(    -5.05), SIMDE_FLOAT32_C(  -435.35)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(   -91.92), SIMDE_FLOAT32_C(  -128.53), SIMDE_FLOAT32_C(  -157.39), SIMDE_FLOAT32_C(   922.31),
                         SIMDE_FLOAT32_C(    68.36), SIMDE_FLOAT32_C(  -842.81), SIMDE_FLOAT32_C(    60.56), SIMDE_FLOAT32_C(   689.02),
                         SIMDE_FLOAT32_C(   -70.29), SIMDE_FLOAT32_C(  -753.33), SIMDE_FLOAT32_C(  -999.31), SIMDE_FLOAT32_C(    72.69),
                         SIMDE_FLOAT32_C(    47.99), SIMDE_FLOAT32_C(   903.18), SIMDE_FLOAT32_C(    -5.05), SIMDE_FLOAT32_C(  -435.35)) },
    { simde_mm512_set_ps(SIMDE_FLOAT32_C(  -687.13), SIMDE_FLOAT32_C(  -562.12), SIMDE_FLOAT32_C(  -564.90), SIMDE_FLOAT32_C(  -948.64),
                         SIMDE_FLOAT32_C(   287.72), SIMDE_FLOAT32_C(   -13.16), SIMDE_FLOAT32_C(   706.69), SIMDE_FLOAT32_C(    62.48),
                         SIMDE_FLOAT32_C(    15.33), SIMDE_FLOAT32_C(  -938.52), SIMDE_FLOAT32_C(  -548.66), SIMDE_FLOAT32_C(   893.31),
                         SIMDE_FLOAT32_C(  -166.31), SIMDE_FLOAT32_C(     3.21), SIMDE_FLOAT32_C(  -121.90), SIMDE_FLOAT32_C(  -237.87)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(  -687.13), SIMDE_FLOAT32_C(  -562.12), SIMDE_FLOAT32_C(  -564.90), SIMDE_FLOAT32_C(  -948.64),
                         SIMDE_FLOAT32_C(   287.72), SIMDE_FLOAT32_C(   -13.16), SIMDE_FLOAT32_C(   706.69), SIMDE_FLOAT32_C(    62.48),
                         SIMDE_FLOAT32_C(    15.33), SIMDE_FLOAT32_C(  -938.52), SIMDE_FLOAT32_C(  -548.66), SIMDE_FLOAT32_C(   893.31),
                         SIMDE_FLOAT32_C(  -166.31), SIMDE_FLOAT32_C(     3.21), SIMDE_FLOAT32_C(  -121.90), SIMDE_FLOAT32_C(  -237.87)) }
  };
  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
  simde__m512 r = simde_mm512_castpd_ps(simde_mm512_castps_pd(test_vec[i].a));
  simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_castsi512_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1318886849), INT32_C( -963615992), INT32_C( 1168255165), INT32_C(-1910220907),
                            INT32_C( -362848940), INT32_C( 1228894571), INT32_C( -604141281), INT32_C( -310556576),
                            INT32_C( -195291453), INT32_C(-1762187610), INT32_C(  785462248), INT32_C(-1654799886),
                            INT32_C(  443157727), INT32_C(  291457152), INT32_C( -673220379), INT32_C( 1588648097)),
      simde_mm512_set_epi32(INT32_C( 1318886849), INT32_C( -963615992), INT32_C( 1168255165), INT32_C(-1910220907),
                            INT32_C( -362848940), INT32_C( 1228894571), INT32_C( -604141281), INT32_C( -310556576),
                            INT32_C( -195291453), INT32_C(-1762187610), INT32_C(  785462248), INT32_C(-1654799886),
                            INT32_C(  443157727), INT32_C(  291457152), INT32_C( -673220379), INT32_C( 1588648097)) },
    { simde_mm512_set_epi32(INT32_C(  388365668), INT32_C( 1922479678), INT32_C(-1025869072), INT32_C( 1053113602),
                            INT32_C( -948060413), INT32_C( 1300260897), INT32_C(  723694906), INT32_C( 1579982945),
                            INT32_C(  551168292), INT32_C( 2117629385), INT32_C( 1278333888), INT32_C(-1219831740),
                            INT32_C( 1012175805), INT32_C(-2142193359), INT32_C(-1234937614), INT32_C(  296489222)),
      simde_mm512_set_epi32(INT32_C(  388365668), INT32_C( 1922479678), INT32_C(-1025869072), INT32_C( 1053113602),
                            INT32_C( -948060413), INT32_C( 1300260897), INT32_C(  723694906), INT32_C( 1579982945),
                            INT32_C(  551168292), INT32_C( 2117629385), INT32_C( 1278333888), INT32_C(-1219831740),
                            INT32_C( 1012175805), INT32_C(-2142193359), INT32_C(-1234937614), INT32_C(  296489222)) },
    { simde_mm512_set_epi32(INT32_C( 1558252976), INT32_C( -896391163), INT32_C( -282653026), INT32_C( -261252112),
                            INT32_C(-1886684518), INT32_C( -792082029), INT32_C(-1821569388), INT32_C(  452082945),
                            INT32_C( 1996286767), INT32_C( -816391022), INT32_C(   52318481), INT32_C( -732357064),
                            INT32_C(-1730294171), INT32_C( 1249707888), INT32_C( 1882801706), INT32_C( -348126448)),
      simde_mm512_set_epi32(INT32_C( 1558252976), INT32_C( -896391163), INT32_C( -282653026), INT32_C( -261252112),
                            INT32_C(-1886684518), INT32_C( -792082029), INT32_C(-1821569388), INT32_C(  452082945),
                            INT32_C( 1996286767), INT32_C( -816391022), INT32_C(   52318481), INT32_C( -732357064),
                            INT32_C(-1730294171), INT32_C( 1249707888), INT32_C( 1882801706), INT32_C( -348126448)) },
    { simde_mm512_set_epi32(INT32_C(-1306007403), INT32_C(  -14823844), INT32_C(   68140915), INT32_C(-1279691583),
                            INT32_C( -366833324), INT32_C( 1029809788), INT32_C(  398169753), INT32_C( 1644631001),
                            INT32_C(-1495643320), INT32_C( -349662299), INT32_C(  337000079), INT32_C(-1599869755),
                            INT32_C( -326525511), INT32_C( 2145777358), INT32_C( -671246537), INT32_C(-1483766126)),
      simde_mm512_set_epi32(INT32_C(-1306007403), INT32_C(  -14823844), INT32_C(   68140915), INT32_C(-1279691583),
                            INT32_C( -366833324), INT32_C( 1029809788), INT32_C(  398169753), INT32_C( 1644631001),
                            INT32_C(-1495643320), INT32_C( -349662299), INT32_C(  337000079), INT32_C(-1599869755),
                            INT32_C( -326525511), INT32_C( 2145777358), INT32_C( -671246537), INT32_C(-1483766126)) },
    { simde_mm512_set_epi32(INT32_C( 1306718049), INT32_C( -481786167), INT32_C( 1036658403), INT32_C(-1275842496),
                            INT32_C(-1384128337), INT32_C( 1110346848), INT32_C( 1633808121), INT32_C(  818679772),
                            INT32_C(  528120539), INT32_C( 1454269875), INT32_C(-1309905851), INT32_C(  195834431),
                            INT32_C( 1710057355), INT32_C(-1286641516), INT32_C( -825070396), INT32_C(-1314661204)),
      simde_mm512_set_epi32(INT32_C( 1306718049), INT32_C( -481786167), INT32_C( 1036658403), INT32_C(-1275842496),
                            INT32_C(-1384128337), INT32_C( 1110346848), INT32_C( 1633808121), INT32_C(  818679772),
                            INT32_C(  528120539), INT32_C( 1454269875), INT32_C(-1309905851), INT32_C(  195834431),
                            INT32_C( 1710057355), INT32_C(-1286641516), INT32_C( -825070396), INT32_C(-1314661204)) },
    { simde_mm512_set_epi32(INT32_C(  897666230), INT32_C(-1247200387), INT32_C(-1594960612), INT32_C(  784358708),
                            INT32_C( -639715103), INT32_C( 1722588202), INT32_C( -957042730), INT32_C( 1748493888),
                            INT32_C(-1424605467), INT32_C( 1715503797), INT32_C(-1877902145), INT32_C( 1433291829),
                            INT32_C(-1672795293), INT32_C( -125523250), INT32_C(  817060741), INT32_C(-1940108057)),
      simde_mm512_set_epi32(INT32_C(  897666230), INT32_C(-1247200387), INT32_C(-1594960612), INT32_C(  784358708),
                            INT32_C( -639715103), INT32_C( 1722588202), INT32_C( -957042730), INT32_C( 1748493888),
                            INT32_C(-1424605467), INT32_C( 1715503797), INT32_C(-1877902145), INT32_C( 1433291829),
                            INT32_C(-1672795293), INT32_C( -125523250), INT32_C(  817060741), INT32_C(-1940108057)) },
    { simde_mm512_set_epi32(INT32_C( -771897847), INT32_C(   51088448), INT32_C(  731748986), INT32_C(-2010019631),
                            INT32_C(-1805256328), INT32_C( -982668321), INT32_C(-1986983933), INT32_C( -806537017),
                            INT32_C( 2103732941), INT32_C(-1950119891), INT32_C( -262758582), INT32_C( -704905824),
                            INT32_C( -298698020), INT32_C( -290664422), INT32_C( -169474404), INT32_C( -381923585)),
      simde_mm512_set_epi32(INT32_C( -771897847), INT32_C(   51088448), INT32_C(  731748986), INT32_C(-2010019631),
                            INT32_C(-1805256328), INT32_C( -982668321), INT32_C(-1986983933), INT32_C( -806537017),
                            INT32_C( 2103732941), INT32_C(-1950119891), INT32_C( -262758582), INT32_C( -704905824),
                            INT32_C( -298698020), INT32_C( -290664422), INT32_C( -169474404), INT32_C( -381923585)) },
    { simde_mm512_set_epi32(INT32_C(-1424890095), INT32_C( -662922774), INT32_C( 1172732731), INT32_C(  220940559),
                            INT32_C(  736508018), INT32_C(-1201335155), INT32_C( -747976663), INT32_C( 1864271935),
                            INT32_C( 1893933430), INT32_C(  214467364), INT32_C(-1230640603), INT32_C(-1529352277),
                            INT32_C(  301681975), INT32_C(  -86355089), INT32_C( 1945634979), INT32_C( 1310017249)),
      simde_mm512_set_epi32(INT32_C(-1424890095), INT32_C( -662922774), INT32_C( 1172732731), INT32_C(  220940559),
                            INT32_C(  736508018), INT32_C(-1201335155), INT32_C( -747976663), INT32_C( 1864271935),
                            INT32_C( 1893933430), INT32_C(  214467364), INT32_C(-1230640603), INT32_C(-1529352277),
                            INT32_C(  301681975), INT32_C(  -86355089), INT32_C( 1945634979), INT32_C( 1310017249)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_castps_si512(simde_mm512_castsi512_ps(test_vec[i].a));
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_castpd_ps(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512d a;
    simde__m512d r;
  } test_vec[8] = {
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -628.24), SIMDE_FLOAT64_C( -135.47),
                         SIMDE_FLOAT64_C( -682.87), SIMDE_FLOAT64_C(  278.82),
                         SIMDE_FLOAT64_C(   71.00), SIMDE_FLOAT64_C(  284.08),
                         SIMDE_FLOAT64_C( -576.74), SIMDE_FLOAT64_C( -655.56)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -628.24), SIMDE_FLOAT64_C( -135.47),
                         SIMDE_FLOAT64_C( -682.87), SIMDE_FLOAT64_C(  278.82),
                         SIMDE_FLOAT64_C(   71.00), SIMDE_FLOAT64_C(  284.08),
                         SIMDE_FLOAT64_C( -576.74), SIMDE_FLOAT64_C( -655.56)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  730.25), SIMDE_FLOAT64_C(  427.27),
                         SIMDE_FLOAT64_C( -482.13), SIMDE_FLOAT64_C( -725.95),
                         SIMDE_FLOAT64_C(  102.42), SIMDE_FLOAT64_C( -673.41),
                         SIMDE_FLOAT64_C(  528.31), SIMDE_FLOAT64_C( -870.66)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  730.25), SIMDE_FLOAT64_C(  427.27),
                         SIMDE_FLOAT64_C( -482.13), SIMDE_FLOAT64_C( -725.95),
                         SIMDE_FLOAT64_C(  102.42), SIMDE_FLOAT64_C( -673.41),
                         SIMDE_FLOAT64_C(  528.31), SIMDE_FLOAT64_C( -870.66)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -250.54), SIMDE_FLOAT64_C( -740.40),
                         SIMDE_FLOAT64_C(  -94.04), SIMDE_FLOAT64_C( -104.23),
                         SIMDE_FLOAT64_C( -857.17), SIMDE_FLOAT64_C( -236.14),
                         SIMDE_FLOAT64_C( -926.88), SIMDE_FLOAT64_C(  849.79)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -250.54), SIMDE_FLOAT64_C( -740.40),
                         SIMDE_FLOAT64_C(  -94.04), SIMDE_FLOAT64_C( -104.23),
                         SIMDE_FLOAT64_C( -857.17), SIMDE_FLOAT64_C( -236.14),
                         SIMDE_FLOAT64_C( -926.88), SIMDE_FLOAT64_C(  849.79)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(   72.78), SIMDE_FLOAT64_C(  -73.84),
                         SIMDE_FLOAT64_C(  -20.96), SIMDE_FLOAT64_C( -723.10),
                         SIMDE_FLOAT64_C( -895.17), SIMDE_FLOAT64_C(  -12.07),
                         SIMDE_FLOAT64_C( -741.57), SIMDE_FLOAT64_C(  210.75)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(   72.78), SIMDE_FLOAT64_C(  -73.84),
                         SIMDE_FLOAT64_C(  -20.96), SIMDE_FLOAT64_C( -723.10),
                         SIMDE_FLOAT64_C( -895.17), SIMDE_FLOAT64_C(  -12.07),
                         SIMDE_FLOAT64_C( -741.57), SIMDE_FLOAT64_C(  210.75)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C( -460.57), SIMDE_FLOAT64_C( -774.17),
                         SIMDE_FLOAT64_C(  356.36), SIMDE_FLOAT64_C(  996.28),
                         SIMDE_FLOAT64_C(    2.33), SIMDE_FLOAT64_C( -321.17),
                         SIMDE_FLOAT64_C(  407.13), SIMDE_FLOAT64_C(  410.39)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C( -460.57), SIMDE_FLOAT64_C( -774.17),
                         SIMDE_FLOAT64_C(  356.36), SIMDE_FLOAT64_C(  996.28),
                         SIMDE_FLOAT64_C(    2.33), SIMDE_FLOAT64_C( -321.17),
                         SIMDE_FLOAT64_C(  407.13), SIMDE_FLOAT64_C(  410.39)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  913.16), SIMDE_FLOAT64_C( -861.99),
                         SIMDE_FLOAT64_C( -778.41), SIMDE_FLOAT64_C( -877.17),
                         SIMDE_FLOAT64_C( -312.46), SIMDE_FLOAT64_C(  942.54),
                         SIMDE_FLOAT64_C(  824.90), SIMDE_FLOAT64_C( -524.63)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  913.16), SIMDE_FLOAT64_C( -861.99),
                         SIMDE_FLOAT64_C( -778.41), SIMDE_FLOAT64_C( -877.17),
                         SIMDE_FLOAT64_C( -312.46), SIMDE_FLOAT64_C(  942.54),
                         SIMDE_FLOAT64_C(  824.90), SIMDE_FLOAT64_C( -524.63)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  993.37), SIMDE_FLOAT64_C( -877.52),
                         SIMDE_FLOAT64_C( -379.85), SIMDE_FLOAT64_C( -359.06),
                         SIMDE_FLOAT64_C(   56.95), SIMDE_FLOAT64_C( -974.33),
                         SIMDE_FLOAT64_C(  222.57), SIMDE_FLOAT64_C(  341.02)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  993.37), SIMDE_FLOAT64_C( -877.52),
                         SIMDE_FLOAT64_C( -379.85), SIMDE_FLOAT64_C( -359.06),
                         SIMDE_FLOAT64_C(   56.95), SIMDE_FLOAT64_C( -974.33),
                         SIMDE_FLOAT64_C(  222.57), SIMDE_FLOAT64_C(  341.02)) },
    { simde_mm512_set_pd(SIMDE_FLOAT64_C(  935.31), SIMDE_FLOAT64_C(  -34.65),
                         SIMDE_FLOAT64_C( -660.60), SIMDE_FLOAT64_C(  275.30),
                         SIMDE_FLOAT64_C( -994.69), SIMDE_FLOAT64_C( -110.75),
                         SIMDE_FLOAT64_C( -596.29), SIMDE_FLOAT64_C(  897.05)),
      simde_mm512_set_pd(SIMDE_FLOAT64_C(  935.31), SIMDE_FLOAT64_C(  -34.65),
                         SIMDE_FLOAT64_C( -660.60), SIMDE_FLOAT64_C(  275.30),
                         SIMDE_FLOAT64_C( -994.69), SIMDE_FLOAT64_C( -110.75),
                         SIMDE_FLOAT64_C( -596.29), SIMDE_FLOAT64_C(  897.05)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
  simde__m512d r = simde_mm512_castps_pd(simde_mm512_castpd_ps(test_vec[i].a));
  simde_assert_m512d_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_castpd_si512(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-2932283473482861325), INT64_C(-7551244776617231168),
                            INT64_C(-7807931705287408960), INT64_C(-5709657210396534887),
                            INT64_C(-5733822187822983321), INT64_C(-6449026762131886574),
                            INT64_C(-4469461462291751367), INT64_C(-1966061413581863839)),
      simde_mm512_set_epi64(INT64_C(-2932283473482861325), INT64_C(-7551244776617231168),
                            INT64_C(-7807931705287408960), INT64_C(-5709657210396534887),
                            INT64_C(-5733822187822983321), INT64_C(-6449026762131886574),
                            INT64_C(-4469461462291751367), INT64_C(-1966061413581863839)) },
    { simde_mm512_set_epi64(INT64_C( 1279499818540525091), INT64_C( 7861921123166401201),
                            INT64_C(-7776868620274070112), INT64_C( 1282257894426558327),
                            INT64_C( 7845161632627040500), INT64_C(-5240804517978550546),
                            INT64_C( 5302531798719229176), INT64_C(-5843726612220411524)),
      simde_mm512_set_epi64(INT64_C( 1279499818540525091), INT64_C( 7861921123166401201),
                            INT64_C(-7776868620274070112), INT64_C( 1282257894426558327),
                            INT64_C( 7845161632627040500), INT64_C(-5240804517978550546),
                            INT64_C( 5302531798719229176), INT64_C(-5843726612220411524)) },
    { simde_mm512_set_epi64(INT64_C( 8264653255264008622), INT64_C(  162552689407469387),
                            INT64_C(-6727945509653055258), INT64_C(-8051656683653588284),
                            INT64_C(-3673194952315675544), INT64_C(-6715795272606770523),
                            INT64_C( 4713570617590173150), INT64_C(-1460297597223426823)),
      simde_mm512_set_epi64(INT64_C( 8264653255264008622), INT64_C(  162552689407469387),
                            INT64_C(-6727945509653055258), INT64_C(-8051656683653588284),
                            INT64_C(-3673194952315675544), INT64_C(-6715795272606770523),
                            INT64_C( 4713570617590173150), INT64_C(-1460297597223426823)) },
    { simde_mm512_set_epi64(INT64_C( 8052472118589501494), INT64_C(-9124867990213523455),
                            INT64_C( 8236047422487698023), INT64_C( 5945811080038499438),
                            INT64_C( 6518955141271267147), INT64_C( 2853045589014014838),
                            INT64_C( 2581043198697401036), INT64_C( 3427873189039658818)),
      simde_mm512_set_epi64(INT64_C( 8052472118589501494), INT64_C(-9124867990213523455),
                            INT64_C( 8236047422487698023), INT64_C( 5945811080038499438),
                            INT64_C( 6518955141271267147), INT64_C( 2853045589014014838),
                            INT64_C( 2581043198697401036), INT64_C( 3427873189039658818)) },
    { simde_mm512_set_epi64(INT64_C(-3824716377022324243), INT64_C( 2673669294054155970),
                            INT64_C(-3539125802254885907), INT64_C(-2325147789690797111),
                            INT64_C(-8959458229056338743), INT64_C( 8174617038106487054),
                            INT64_C( 2599971863369021204), INT64_C(-4237128038265732285)),
      simde_mm512_set_epi64(INT64_C(-3824716377022324243), INT64_C( 2673669294054155970),
                            INT64_C(-3539125802254885907), INT64_C(-2325147789690797111),
                            INT64_C(-8959458229056338743), INT64_C( 8174617038106487054),
                            INT64_C( 2599971863369021204), INT64_C(-4237128038265732285)) },
    { simde_mm512_set_epi64(INT64_C(  467193351903391999), INT64_C( 4360504692705357109),
                            INT64_C( 9013482442495432840), INT64_C(-2009950226490426066),
                            INT64_C(  483507342486254018), INT64_C(  183419223247078674),
                            INT64_C( 2287275646029180906), INT64_C(-3986540516598251529)),
      simde_mm512_set_epi64(INT64_C(  467193351903391999), INT64_C( 4360504692705357109),
                            INT64_C( 9013482442495432840), INT64_C(-2009950226490426066),
                            INT64_C(  483507342486254018), INT64_C(  183419223247078674),
                            INT64_C( 2287275646029180906), INT64_C(-3986540516598251529)) },
    { simde_mm512_set_epi64(INT64_C(-3077645007853633163), INT64_C(-1572779564739653348),
                            INT64_C(-9063178373150191806), INT64_C(-7245631614198864513),
                            INT64_C(-8737364776584512923), INT64_C(-2892458838054920588),
                            INT64_C(  471547968378055740), INT64_C(  775209234293426283)),
      simde_mm512_set_epi64(INT64_C(-3077645007853633163), INT64_C(-1572779564739653348),
                            INT64_C(-9063178373150191806), INT64_C(-7245631614198864513),
                            INT64_C(-8737364776584512923), INT64_C(-2892458838054920588),
                            INT64_C(  471547968378055740), INT64_C(  775209234293426283)) },
    { simde_mm512_set_epi64(INT64_C(-6041628496323364468), INT64_C(-9137870096902224131),
                            INT64_C(-1573113398564050163), INT64_C( 8854240990059664398),
                            INT64_C(-2186298020810705372), INT64_C(-2128579590201805820),
                            INT64_C( 2972790510556186969), INT64_C(-3293272700939969964)),
      simde_mm512_set_epi64(INT64_C(-6041628496323364468), INT64_C(-9137870096902224131),
                            INT64_C(-1573113398564050163), INT64_C( 8854240990059664398),
                            INT64_C(-2186298020810705372), INT64_C(-2128579590201805820),
                            INT64_C( 2972790510556186969), INT64_C(-3293272700939969964)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_castpd_si512(simde_mm512_castsi512_pd(test_vec[i].a));
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_castsi512_pd(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-2932283473482861325), INT64_C(-7551244776617231168),
                            INT64_C(-7807931705287408960), INT64_C(-5709657210396534887),
                            INT64_C(-5733822187822983321), INT64_C(-6449026762131886574),
                            INT64_C(-4469461462291751367), INT64_C(-1966061413581863839)),
      simde_mm512_set_epi64(INT64_C(-2932283473482861325), INT64_C(-7551244776617231168),
                            INT64_C(-7807931705287408960), INT64_C(-5709657210396534887),
                            INT64_C(-5733822187822983321), INT64_C(-6449026762131886574),
                            INT64_C(-4469461462291751367), INT64_C(-1966061413581863839)) },
    { simde_mm512_set_epi64(INT64_C( 1279499818540525091), INT64_C( 7861921123166401201),
                            INT64_C(-7776868620274070112), INT64_C( 1282257894426558327),
                            INT64_C( 7845161632627040500), INT64_C(-5240804517978550546),
                            INT64_C( 5302531798719229176), INT64_C(-5843726612220411524)),
      simde_mm512_set_epi64(INT64_C( 1279499818540525091), INT64_C( 7861921123166401201),
                            INT64_C(-7776868620274070112), INT64_C( 1282257894426558327),
                            INT64_C( 7845161632627040500), INT64_C(-5240804517978550546),
                            INT64_C( 5302531798719229176), INT64_C(-5843726612220411524)) },
    { simde_mm512_set_epi64(INT64_C( 8264653255264008622), INT64_C(  162552689407469387),
                            INT64_C(-6727945509653055258), INT64_C(-8051656683653588284),
                            INT64_C(-3673194952315675544), INT64_C(-6715795272606770523),
                            INT64_C( 4713570617590173150), INT64_C(-1460297597223426823)),
      simde_mm512_set_epi64(INT64_C( 8264653255264008622), INT64_C(  162552689407469387),
                            INT64_C(-6727945509653055258), INT64_C(-8051656683653588284),
                            INT64_C(-3673194952315675544), INT64_C(-6715795272606770523),
                            INT64_C( 4713570617590173150), INT64_C(-1460297597223426823)) },
    { simde_mm512_set_epi64(INT64_C( 8052472118589501494), INT64_C(-9124867990213523455),
                            INT64_C( 8236047422487698023), INT64_C( 5945811080038499438),
                            INT64_C( 6518955141271267147), INT64_C( 2853045589014014838),
                            INT64_C( 2581043198697401036), INT64_C( 3427873189039658818)),
      simde_mm512_set_epi64(INT64_C( 8052472118589501494), INT64_C(-9124867990213523455),
                            INT64_C( 8236047422487698023), INT64_C( 5945811080038499438),
                            INT64_C( 6518955141271267147), INT64_C( 2853045589014014838),
                            INT64_C( 2581043198697401036), INT64_C( 3427873189039658818)) },
    { simde_mm512_set_epi64(INT64_C(-3824716377022324243), INT64_C( 2673669294054155970),
                            INT64_C(-3539125802254885907), INT64_C(-2325147789690797111),
                            INT64_C(-8959458229056338743), INT64_C( 8174617038106487054),
                            INT64_C( 2599971863369021204), INT64_C(-4237128038265732285)),
      simde_mm512_set_epi64(INT64_C(-3824716377022324243), INT64_C( 2673669294054155970),
                            INT64_C(-3539125802254885907), INT64_C(-2325147789690797111),
                            INT64_C(-8959458229056338743), INT64_C( 8174617038106487054),
                            INT64_C( 2599971863369021204), INT64_C(-4237128038265732285)) },
    { simde_mm512_set_epi64(INT64_C(  467193351903391999), INT64_C( 4360504692705357109),
                            INT64_C( 9013482442495432840), INT64_C(-2009950226490426066),
                            INT64_C(  483507342486254018), INT64_C(  183419223247078674),
                            INT64_C( 2287275646029180906), INT64_C(-3986540516598251529)),
      simde_mm512_set_epi64(INT64_C(  467193351903391999), INT64_C( 4360504692705357109),
                            INT64_C( 9013482442495432840), INT64_C(-2009950226490426066),
                            INT64_C(  483507342486254018), INT64_C(  183419223247078674),
                            INT64_C( 2287275646029180906), INT64_C(-3986540516598251529)) },
    { simde_mm512_set_epi64(INT64_C(-3077645007853633163), INT64_C(-1572779564739653348),
                            INT64_C(-9063178373150191806), INT64_C(-7245631614198864513),
                            INT64_C(-8737364776584512923), INT64_C(-2892458838054920588),
                            INT64_C(  471547968378055740), INT64_C(  775209234293426283)),
      simde_mm512_set_epi64(INT64_C(-3077645007853633163), INT64_C(-1572779564739653348),
                            INT64_C(-9063178373150191806), INT64_C(-7245631614198864513),
                            INT64_C(-8737364776584512923), INT64_C(-2892458838054920588),
                            INT64_C(  471547968378055740), INT64_C(  775209234293426283)) },
    { simde_mm512_set_epi64(INT64_C(-6041628496323364468), INT64_C(-9137870096902224131),
                            INT64_C(-1573113398564050163), INT64_C( 8854240990059664398),
                            INT64_C(-2186298020810705372), INT64_C(-2128579590201805820),
                            INT64_C( 2972790510556186969), INT64_C(-3293272700939969964)),
      simde_mm512_set_epi64(INT64_C(-6041628496323364468), INT64_C(-9137870096902224131),
                            INT64_C(-1573113398564050163), INT64_C( 8854240990059664398),
                            INT64_C(-2186298020810705372), INT64_C(-2128579590201805820),
                            INT64_C( 2972790510556186969), INT64_C(-3293272700939969964)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_castpd_si512(simde_mm512_castsi512_pd(test_vec[i].a));
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_castsi128_si512(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi32(INT32_C(-1668834023), INT32_C(-1352312258), INT32_C(  556637397), INT32_C( -245835434)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(-1668834023), INT32_C(-1352312258), INT32_C(  556637397), INT32_C( -245835434)) },
    { simde_mm_set_epi32(INT32_C( 1624943139), INT32_C( -461116292), INT32_C(-2119913042), INT32_C( -900874486)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1624943139), INT32_C( -461116292), INT32_C(-2119913042), INT32_C( -900874486)) },
    { simde_mm_set_epi32(INT32_C( -483029339), INT32_C( 1367342572), INT32_C( 1717714414), INT32_C( -425758846)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( -483029339), INT32_C( 1367342572), INT32_C( 1717714414), INT32_C( -425758846)) },
    { simde_mm_set_epi32(INT32_C( -911211301), INT32_C(  327710477), INT32_C( -375455356), INT32_C(-1057936365)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( -911211301), INT32_C(  327710477), INT32_C( -375455356), INT32_C(-1057936365)) },
    { simde_mm_set_epi32(INT32_C(  275440871), INT32_C(  641107064), INT32_C( 1055988271), INT32_C(  449906430)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  275440871), INT32_C(  641107064), INT32_C( 1055988271), INT32_C(  449906430)) },
    { simde_mm_set_epi32(INT32_C( -222973014), INT32_C( -356035719), INT32_C( -568690768), INT32_C( 1550764475)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( -222973014), INT32_C( -356035719), INT32_C( -568690768), INT32_C( 1550764475)) },
    { simde_mm_set_epi32(INT32_C( 1024653484), INT32_C(-1114765609), INT32_C(  501755487), INT32_C( 1011153625)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1024653484), INT32_C(-1114765609), INT32_C(  501755487), INT32_C( 1011153625)) },
    { simde_mm_set_epi32(INT32_C( 1112995048), INT32_C( 1174048979), INT32_C(  979704874), INT32_C( -581442043)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 1112995048), INT32_C( 1174048979), INT32_C(  979704874), INT32_C( -581442043)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_castsi128_si512(test_vec[i].a);
    r=simde_mm512_maskz_mov_epi32(UINT8_C(15), r);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_castsi256_si512(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C( -627018310), INT32_C( -732773372), INT32_C(-1935004141), INT32_C( 1864732488),
                            INT32_C(  140289699), INT32_C(-1570899663), INT32_C(-1630998993), INT32_C(  818347323)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( -627018310), INT32_C( -732773372), INT32_C(-1935004141), INT32_C( 1864732488),
                            INT32_C(  140289699), INT32_C(-1570899663), INT32_C(-1630998993), INT32_C(  818347323)) },
    { simde_mm256_set_epi32(INT32_C( -208321892), INT32_C(-1091642607), INT32_C(  600758994), INT32_C(  251226426),
                            INT32_C( -263819406), INT32_C(-1881495622), INT32_C( 1833100173), INT32_C(-1639467184)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( -208321892), INT32_C(-1091642607), INT32_C(  600758994), INT32_C(  251226426),
                            INT32_C( -263819406), INT32_C(-1881495622), INT32_C( 1833100173), INT32_C(-1639467184)) },
    { simde_mm256_set_epi32(INT32_C(-1754784817), INT32_C( -579924690), INT32_C(  661573432), INT32_C(-1181370591),
                            INT32_C(   60095626), INT32_C(  956226137), INT32_C( -117371256), INT32_C(-1481124874)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(-1754784817), INT32_C( -579924690), INT32_C(  661573432), INT32_C(-1181370591),
                            INT32_C(   60095626), INT32_C(  956226137), INT32_C( -117371256), INT32_C(-1481124874)) },
    { simde_mm256_set_epi32(INT32_C( 2059950650), INT32_C(  100086839), INT32_C( 1040622382), INT32_C(  188824341),
                            INT32_C( 1391404675), INT32_C(-2134538567), INT32_C( 1420659092), INT32_C( 1719736439)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( 2059950650), INT32_C(  100086839), INT32_C( 1040622382), INT32_C(  188824341),
                            INT32_C( 1391404675), INT32_C(-2134538567), INT32_C( 1420659092), INT32_C( 1719736439)) },
    { simde_mm256_set_epi32(INT32_C(  -72277944), INT32_C(    -175328), INT32_C( 1214469435), INT32_C( -481210186),
                            INT32_C( 1755907779), INT32_C( 1941862792), INT32_C(  -53966731), INT32_C(  177508362)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  -72277944), INT32_C(    -175328), INT32_C( 1214469435), INT32_C( -481210186),
                            INT32_C( 1755907779), INT32_C( 1941862792), INT32_C(  -53966731), INT32_C(  177508362)) },
    { simde_mm256_set_epi32(INT32_C( -579195101), INT32_C(  394175316), INT32_C( 2062674518), INT32_C(-2146728292),
                            INT32_C(   67273180), INT32_C(-1275684351), INT32_C(  245156584), INT32_C(  382170563)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C( -579195101), INT32_C(  394175316), INT32_C( 2062674518), INT32_C(-2146728292),
                            INT32_C(   67273180), INT32_C(-1275684351), INT32_C(  245156584), INT32_C(  382170563)) },
    { simde_mm256_set_epi32(INT32_C(-1725634605), INT32_C(-1992977299), INT32_C( -692965094), INT32_C( 1390281908),
                            INT32_C( 1056328531), INT32_C(  448471511), INT32_C(  676963201), INT32_C( -471721601)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(-1725634605), INT32_C(-1992977299), INT32_C( -692965094), INT32_C( 1390281908),
                            INT32_C( 1056328531), INT32_C(  448471511), INT32_C(  676963201), INT32_C( -471721601)) },
    { simde_mm256_set_epi32(INT32_C(  868040614), INT32_C( -210101311), INT32_C( 1711471891), INT32_C(  -87087404),
                            INT32_C(-1569971706), INT32_C(-1573190685), INT32_C( 1472943366), INT32_C(-1700287741)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(  868040614), INT32_C( -210101311), INT32_C( 1711471891), INT32_C(  -87087404),
                            INT32_C(-1569971706), INT32_C(-1573190685), INT32_C( 1472943366), INT32_C(-1700287741)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_castsi256_si512(test_vec[i].a);
    r=simde_mm512_maskz_mov_epi32(UINT8_C(255), r);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_castps_si512(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1318886849), INT32_C( -963615992), INT32_C( 1168255165), INT32_C(-1910220907),
                            INT32_C( -362848940), INT32_C( 1228894571), INT32_C( -604141281), INT32_C( -310556576),
                            INT32_C( -195291453), INT32_C(-1762187610), INT32_C(  785462248), INT32_C(-1654799886),
                            INT32_C(  443157727), INT32_C(  291457152), INT32_C( -673220379), INT32_C( 1588648097)),
      simde_mm512_set_epi32(INT32_C( 1318886849), INT32_C( -963615992), INT32_C( 1168255165), INT32_C(-1910220907),
                            INT32_C( -362848940), INT32_C( 1228894571), INT32_C( -604141281), INT32_C( -310556576),
                            INT32_C( -195291453), INT32_C(-1762187610), INT32_C(  785462248), INT32_C(-1654799886),
                            INT32_C(  443157727), INT32_C(  291457152), INT32_C( -673220379), INT32_C( 1588648097)) },
    { simde_mm512_set_epi32(INT32_C(  388365668), INT32_C( 1922479678), INT32_C(-1025869072), INT32_C( 1053113602),
                            INT32_C( -948060413), INT32_C( 1300260897), INT32_C(  723694906), INT32_C( 1579982945),
                            INT32_C(  551168292), INT32_C( 2117629385), INT32_C( 1278333888), INT32_C(-1219831740),
                            INT32_C( 1012175805), INT32_C(-2142193359), INT32_C(-1234937614), INT32_C(  296489222)),
      simde_mm512_set_epi32(INT32_C(  388365668), INT32_C( 1922479678), INT32_C(-1025869072), INT32_C( 1053113602),
                            INT32_C( -948060413), INT32_C( 1300260897), INT32_C(  723694906), INT32_C( 1579982945),
                            INT32_C(  551168292), INT32_C( 2117629385), INT32_C( 1278333888), INT32_C(-1219831740),
                            INT32_C( 1012175805), INT32_C(-2142193359), INT32_C(-1234937614), INT32_C(  296489222)) },
    { simde_mm512_set_epi32(INT32_C( 1558252976), INT32_C( -896391163), INT32_C( -282653026), INT32_C( -261252112),
                            INT32_C(-1886684518), INT32_C( -792082029), INT32_C(-1821569388), INT32_C(  452082945),
                            INT32_C( 1996286767), INT32_C( -816391022), INT32_C(   52318481), INT32_C( -732357064),
                            INT32_C(-1730294171), INT32_C( 1249707888), INT32_C( 1882801706), INT32_C( -348126448)),
      simde_mm512_set_epi32(INT32_C( 1558252976), INT32_C( -896391163), INT32_C( -282653026), INT32_C( -261252112),
                            INT32_C(-1886684518), INT32_C( -792082029), INT32_C(-1821569388), INT32_C(  452082945),
                            INT32_C( 1996286767), INT32_C( -816391022), INT32_C(   52318481), INT32_C( -732357064),
                            INT32_C(-1730294171), INT32_C( 1249707888), INT32_C( 1882801706), INT32_C( -348126448)) },
    { simde_mm512_set_epi32(INT32_C(-1306007403), INT32_C(  -14823844), INT32_C(   68140915), INT32_C(-1279691583),
                            INT32_C( -366833324), INT32_C( 1029809788), INT32_C(  398169753), INT32_C( 1644631001),
                            INT32_C(-1495643320), INT32_C( -349662299), INT32_C(  337000079), INT32_C(-1599869755),
                            INT32_C( -326525511), INT32_C( 2145777358), INT32_C( -671246537), INT32_C(-1483766126)),
      simde_mm512_set_epi32(INT32_C(-1306007403), INT32_C(  -14823844), INT32_C(   68140915), INT32_C(-1279691583),
                            INT32_C( -366833324), INT32_C( 1029809788), INT32_C(  398169753), INT32_C( 1644631001),
                            INT32_C(-1495643320), INT32_C( -349662299), INT32_C(  337000079), INT32_C(-1599869755),
                            INT32_C( -326525511), INT32_C( 2145777358), INT32_C( -671246537), INT32_C(-1483766126)) },
    { simde_mm512_set_epi32(INT32_C( 1306718049), INT32_C( -481786167), INT32_C( 1036658403), INT32_C(-1275842496),
                            INT32_C(-1384128337), INT32_C( 1110346848), INT32_C( 1633808121), INT32_C(  818679772),
                            INT32_C(  528120539), INT32_C( 1454269875), INT32_C(-1309905851), INT32_C(  195834431),
                            INT32_C( 1710057355), INT32_C(-1286641516), INT32_C( -825070396), INT32_C(-1314661204)),
      simde_mm512_set_epi32(INT32_C( 1306718049), INT32_C( -481786167), INT32_C( 1036658403), INT32_C(-1275842496),
                            INT32_C(-1384128337), INT32_C( 1110346848), INT32_C( 1633808121), INT32_C(  818679772),
                            INT32_C(  528120539), INT32_C( 1454269875), INT32_C(-1309905851), INT32_C(  195834431),
                            INT32_C( 1710057355), INT32_C(-1286641516), INT32_C( -825070396), INT32_C(-1314661204)) },
    { simde_mm512_set_epi32(INT32_C(  897666230), INT32_C(-1247200387), INT32_C(-1594960612), INT32_C(  784358708),
                            INT32_C( -639715103), INT32_C( 1722588202), INT32_C( -957042730), INT32_C( 1748493888),
                            INT32_C(-1424605467), INT32_C( 1715503797), INT32_C(-1877902145), INT32_C( 1433291829),
                            INT32_C(-1672795293), INT32_C( -125523250), INT32_C(  817060741), INT32_C(-1940108057)),
      simde_mm512_set_epi32(INT32_C(  897666230), INT32_C(-1247200387), INT32_C(-1594960612), INT32_C(  784358708),
                            INT32_C( -639715103), INT32_C( 1722588202), INT32_C( -957042730), INT32_C( 1748493888),
                            INT32_C(-1424605467), INT32_C( 1715503797), INT32_C(-1877902145), INT32_C( 1433291829),
                            INT32_C(-1672795293), INT32_C( -125523250), INT32_C(  817060741), INT32_C(-1940108057)) },
    { simde_mm512_set_epi32(INT32_C( -771897847), INT32_C(   51088448), INT32_C(  731748986), INT32_C(-2010019631),
                            INT32_C(-1805256328), INT32_C( -982668321), INT32_C(-1986983933), INT32_C( -806537017),
                            INT32_C( 2103732941), INT32_C(-1950119891), INT32_C( -262758582), INT32_C( -704905824),
                            INT32_C( -298698020), INT32_C( -290664422), INT32_C( -169474404), INT32_C( -381923585)),
      simde_mm512_set_epi32(INT32_C( -771897847), INT32_C(   51088448), INT32_C(  731748986), INT32_C(-2010019631),
                            INT32_C(-1805256328), INT32_C( -982668321), INT32_C(-1986983933), INT32_C( -806537017),
                            INT32_C( 2103732941), INT32_C(-1950119891), INT32_C( -262758582), INT32_C( -704905824),
                            INT32_C( -298698020), INT32_C( -290664422), INT32_C( -169474404), INT32_C( -381923585)) },
    { simde_mm512_set_epi32(INT32_C(-1424890095), INT32_C( -662922774), INT32_C( 1172732731), INT32_C(  220940559),
                            INT32_C(  736508018), INT32_C(-1201335155), INT32_C( -747976663), INT32_C( 1864271935),
                            INT32_C( 1893933430), INT32_C(  214467364), INT32_C(-1230640603), INT32_C(-1529352277),
                            INT32_C(  301681975), INT32_C(  -86355089), INT32_C( 1945634979), INT32_C( 1310017249)),
      simde_mm512_set_epi32(INT32_C(-1424890095), INT32_C( -662922774), INT32_C( 1172732731), INT32_C(  220940559),
                            INT32_C(  736508018), INT32_C(-1201335155), INT32_C( -747976663), INT32_C( 1864271935),
                            INT32_C( 1893933430), INT32_C(  214467364), INT32_C(-1230640603), INT32_C(-1529352277),
                            INT32_C(  301681975), INT32_C(  -86355089), INT32_C( 1945634979), INT32_C( 1310017249)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_castps_si512(simde_mm512_castsi512_ps(test_vec[i].a));
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_castps128_ps512(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128 a;
    simde__m512 r;
  } test_vec[8] = {
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -124.41), SIMDE_FLOAT32_C(   994.42), SIMDE_FLOAT32_C(  -888.56), SIMDE_FLOAT32_C(  -241.67)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -124.41), SIMDE_FLOAT32_C(   994.42), SIMDE_FLOAT32_C(  -888.56), SIMDE_FLOAT32_C(  -241.67)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -472.63), SIMDE_FLOAT32_C(  -923.61), SIMDE_FLOAT32_C(  -185.31), SIMDE_FLOAT32_C(   531.85)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -472.63), SIMDE_FLOAT32_C(  -923.61), SIMDE_FLOAT32_C(  -185.31), SIMDE_FLOAT32_C(   531.85)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -636.73), SIMDE_FLOAT32_C(  -810.98), SIMDE_FLOAT32_C(  -952.52), SIMDE_FLOAT32_C(   896.08)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -636.73), SIMDE_FLOAT32_C(  -810.98), SIMDE_FLOAT32_C(  -952.52), SIMDE_FLOAT32_C(   896.08)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   142.17), SIMDE_FLOAT32_C(  -205.03), SIMDE_FLOAT32_C(  -269.17), SIMDE_FLOAT32_C(  -426.17)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   142.17), SIMDE_FLOAT32_C(  -205.03), SIMDE_FLOAT32_C(  -269.17), SIMDE_FLOAT32_C(  -426.17)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   149.14), SIMDE_FLOAT32_C(  -672.83), SIMDE_FLOAT32_C(  -162.76), SIMDE_FLOAT32_C(  -647.13)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   149.14), SIMDE_FLOAT32_C(  -672.83), SIMDE_FLOAT32_C(  -162.76), SIMDE_FLOAT32_C(  -647.13)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -915.60), SIMDE_FLOAT32_C(    75.79), SIMDE_FLOAT32_C(   831.71), SIMDE_FLOAT32_C(   144.92)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -915.60), SIMDE_FLOAT32_C(    75.79), SIMDE_FLOAT32_C(   831.71), SIMDE_FLOAT32_C(   144.92)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(  -748.69), SIMDE_FLOAT32_C(  -492.84), SIMDE_FLOAT32_C(  -132.48), SIMDE_FLOAT32_C(  -765.52)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -748.69), SIMDE_FLOAT32_C(  -492.84), SIMDE_FLOAT32_C(  -132.48), SIMDE_FLOAT32_C(  -765.52)) },
    { simde_mm_set_ps(SIMDE_FLOAT32_C(   346.51), SIMDE_FLOAT32_C(   297.54), SIMDE_FLOAT32_C(   196.93), SIMDE_FLOAT32_C(  -729.98)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   346.51), SIMDE_FLOAT32_C(   297.54), SIMDE_FLOAT32_C(   196.93), SIMDE_FLOAT32_C(  -729.98)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_castps128_ps512(test_vec[i].a);
    r=simde_mm512_maskz_mov_ps(UINT8_C(15), r);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_castps256_ps512(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256 a;
    simde__m512 r;
  } test_vec[8] = {
     { simde_mm256_set_ps(SIMDE_FLOAT32_C(  -144.51), SIMDE_FLOAT32_C(   522.85),
                         SIMDE_FLOAT32_C(   259.94), SIMDE_FLOAT32_C(   889.02),
                         SIMDE_FLOAT32_C(    47.86), SIMDE_FLOAT32_C(   181.02),
                         SIMDE_FLOAT32_C(   389.89), SIMDE_FLOAT32_C(   -27.36)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -144.51), SIMDE_FLOAT32_C(   522.85), SIMDE_FLOAT32_C(   259.94), SIMDE_FLOAT32_C(   889.02),
                         SIMDE_FLOAT32_C(    47.86), SIMDE_FLOAT32_C(   181.02), SIMDE_FLOAT32_C(   389.89), SIMDE_FLOAT32_C(   -27.36)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(  -477.70), SIMDE_FLOAT32_C(  -298.40),
                         SIMDE_FLOAT32_C(  -730.62), SIMDE_FLOAT32_C(   379.73),
                         SIMDE_FLOAT32_C(   223.22), SIMDE_FLOAT32_C(  -917.34),
                         SIMDE_FLOAT32_C(  -268.46), SIMDE_FLOAT32_C(  -361.61)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -477.70), SIMDE_FLOAT32_C(  -298.40), SIMDE_FLOAT32_C(  -730.62), SIMDE_FLOAT32_C(   379.73),
                         SIMDE_FLOAT32_C(   223.22), SIMDE_FLOAT32_C(  -917.34), SIMDE_FLOAT32_C(  -268.46), SIMDE_FLOAT32_C(  -361.61)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(   995.68), SIMDE_FLOAT32_C(   377.56),
                         SIMDE_FLOAT32_C(  -263.10), SIMDE_FLOAT32_C(   880.58),
                         SIMDE_FLOAT32_C(    61.47), SIMDE_FLOAT32_C(   993.95),
                         SIMDE_FLOAT32_C(   -45.95), SIMDE_FLOAT32_C(  -445.12)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   995.68), SIMDE_FLOAT32_C(   377.56), SIMDE_FLOAT32_C(  -263.10), SIMDE_FLOAT32_C(   880.58),
                         SIMDE_FLOAT32_C(    61.47), SIMDE_FLOAT32_C(   993.95), SIMDE_FLOAT32_C(   -45.95), SIMDE_FLOAT32_C(  -445.12)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(   391.67), SIMDE_FLOAT32_C(  -695.88),
                         SIMDE_FLOAT32_C(   312.22), SIMDE_FLOAT32_C(   389.64),
                         SIMDE_FLOAT32_C(    81.09), SIMDE_FLOAT32_C(   745.89),
                         SIMDE_FLOAT32_C(  -178.17), SIMDE_FLOAT32_C(  -269.92)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   391.67), SIMDE_FLOAT32_C(  -695.88), SIMDE_FLOAT32_C(   312.22), SIMDE_FLOAT32_C(   389.64),
                         SIMDE_FLOAT32_C(    81.09), SIMDE_FLOAT32_C(   745.89), SIMDE_FLOAT32_C(  -178.17), SIMDE_FLOAT32_C(  -269.92)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(   498.47), SIMDE_FLOAT32_C(  -985.97),
                         SIMDE_FLOAT32_C(   802.71), SIMDE_FLOAT32_C(   -28.60),
                         SIMDE_FLOAT32_C(  -781.89), SIMDE_FLOAT32_C(  -410.06),
                         SIMDE_FLOAT32_C(  -602.74), SIMDE_FLOAT32_C(  -353.26)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   498.47), SIMDE_FLOAT32_C(  -985.97), SIMDE_FLOAT32_C(   802.71), SIMDE_FLOAT32_C(   -28.60),
                         SIMDE_FLOAT32_C(  -781.89), SIMDE_FLOAT32_C(  -410.06), SIMDE_FLOAT32_C(  -602.74), SIMDE_FLOAT32_C(  -353.26)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(   265.34), SIMDE_FLOAT32_C(  -587.26),
                         SIMDE_FLOAT32_C(  -801.19), SIMDE_FLOAT32_C(    -4.56),
                         SIMDE_FLOAT32_C(  -900.49), SIMDE_FLOAT32_C(   391.40),
                         SIMDE_FLOAT32_C(  -416.15), SIMDE_FLOAT32_C(  -286.27)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   265.34), SIMDE_FLOAT32_C(  -587.26), SIMDE_FLOAT32_C(  -801.19), SIMDE_FLOAT32_C(    -4.56),
                         SIMDE_FLOAT32_C(  -900.49), SIMDE_FLOAT32_C(   391.40), SIMDE_FLOAT32_C(  -416.15), SIMDE_FLOAT32_C(  -286.27)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(   167.27), SIMDE_FLOAT32_C(  -483.42),
                         SIMDE_FLOAT32_C(   759.82), SIMDE_FLOAT32_C(  -560.80),
                         SIMDE_FLOAT32_C(  -968.24), SIMDE_FLOAT32_C(  -534.87),
                         SIMDE_FLOAT32_C(  -698.82), SIMDE_FLOAT32_C(  -382.24)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(   167.27), SIMDE_FLOAT32_C(  -483.42), SIMDE_FLOAT32_C(   759.82), SIMDE_FLOAT32_C(  -560.80),
                         SIMDE_FLOAT32_C(  -968.24), SIMDE_FLOAT32_C(  -534.87), SIMDE_FLOAT32_C(  -698.82), SIMDE_FLOAT32_C(  -382.24)) },
    { simde_mm256_set_ps(SIMDE_FLOAT32_C(  -792.23), SIMDE_FLOAT32_C(   994.18),
                         SIMDE_FLOAT32_C(  -632.41), SIMDE_FLOAT32_C(   872.87),
                         SIMDE_FLOAT32_C(  -406.41), SIMDE_FLOAT32_C(   989.00),
                         SIMDE_FLOAT32_C(   781.89), SIMDE_FLOAT32_C(   609.26)),
      simde_mm512_set_ps(SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00), SIMDE_FLOAT32_C(     0.00),
                         SIMDE_FLOAT32_C(  -792.23), SIMDE_FLOAT32_C(   994.18), SIMDE_FLOAT32_C(  -632.41), SIMDE_FLOAT32_C(   872.87),
                         SIMDE_FLOAT32_C(  -406.41), SIMDE_FLOAT32_C(   989.00), SIMDE_FLOAT32_C(   781.89), SIMDE_FLOAT32_C(   609.26)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512 r = simde_mm512_castps256_ps512(test_vec[i].a);
    r=simde_mm512_maskz_mov_ps(UINT8_C(255), r);
    simde_assert_m512_close(r, test_vec[i].r, 1);
  }

  return 0;
}

static int
test_simde_mm512_castsi512_si128(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1658575222), INT32_C(-1117261553), INT32_C(-1839997259), INT32_C( -299852262),
                            INT32_C( -856912374), INT32_C( 2142936567), INT32_C( -954684084), INT32_C( 1657017766),
                            INT32_C( -348123015), INT32_C( -101609698), INT32_C(  554693435), INT32_C(-1533582435),
                            INT32_C( -400642696), INT32_C( 1776297700), INT32_C( 1392524409), INT32_C(  398537104)),
      simde_mm_set_epi32(INT32_C( -400642696), INT32_C( 1776297700), INT32_C( 1392524409), INT32_C(  398537104)) },
    { simde_mm512_set_epi32(INT32_C(  776372316), INT32_C(  831674978), INT32_C(   55416914), INT32_C(  243273534),
                            INT32_C( -792877249), INT32_C( -771344813), INT32_C( -310546031), INT32_C(-2107743643),
                            INT32_C(-2024392408), INT32_C(-1627000847), INT32_C( 2079816508), INT32_C(  371701407),
                            INT32_C(-1936589526), INT32_C(-1276131185), INT32_C(-2091159003), INT32_C(  424652427)),
      simde_mm_set_epi32(INT32_C(-1936589526), INT32_C(-1276131185), INT32_C(-2091159003), INT32_C(  424652427)) },
    { simde_mm512_set_epi32(INT32_C( 1173201274), INT32_C( 2136038971), INT32_C( 1938580455), INT32_C( 1481503254),
                            INT32_C( -900987494), INT32_C(  381103974), INT32_C( -844512348), INT32_C(  585017509),
                            INT32_C(  711603246), INT32_C(-1519792959), INT32_C( 1933920466), INT32_C(-1698769431),
                            INT32_C(-1782976050), INT32_C(-1748575141), INT32_C(-1460360861), INT32_C( -334769088)),
      simde_mm_set_epi32(INT32_C(-1782976050), INT32_C(-1748575141), INT32_C(-1460360861), INT32_C( -334769088)) },
    { simde_mm512_set_epi32(INT32_C( 1804169520), INT32_C(-1929398298), INT32_C( 1527159056), INT32_C(  779300869),
                            INT32_C(-1718101462), INT32_C( -324959902), INT32_C(  356284621), INT32_C(  259973358),
                            INT32_C(  -85521098), INT32_C( 1509969945), INT32_C( -146300380), INT32_C(-1326146169),
                            INT32_C( -222527512), INT32_C( 2102498956), INT32_C(   21770579), INT32_C(  799947456)),
      simde_mm_set_epi32(INT32_C( -222527512), INT32_C( 2102498956), INT32_C(   21770579), INT32_C(  799947456)) },
    { simde_mm512_set_epi32(INT32_C(   21446459), INT32_C( -874117397), INT32_C(-1237626592), INT32_C(-1776987073),
                            INT32_C(-1868633440), INT32_C( 1756862535), INT32_C(  992623093), INT32_C( 2049114875),
                            INT32_C(-1780158792), INT32_C( -396995291), INT32_C( 1019180338), INT32_C(-1532238028),
                            INT32_C( 1367863842), INT32_C(-1092466184), INT32_C( 1828179807), INT32_C( -163821277)),
      simde_mm_set_epi32(INT32_C( 1367863842), INT32_C(-1092466184), INT32_C( 1828179807), INT32_C( -163821277)) },
    { simde_mm512_set_epi32(INT32_C(  430786946), INT32_C( -784387301), INT32_C(-1254472927), INT32_C(-1743072668),
                            INT32_C(  651286019), INT32_C(-1388487554), INT32_C(  757390683), INT32_C( 1947676255),
                            INT32_C(  969518281), INT32_C(  -72978072), INT32_C(  760758773), INT32_C( 1573392247),
                            INT32_C(  865448841), INT32_C( 1831180971), INT32_C(-1639739355), INT32_C(   75957511)),
      simde_mm_set_epi32(INT32_C(  865448841), INT32_C( 1831180971), INT32_C(-1639739355), INT32_C(   75957511)) },
    { simde_mm512_set_epi32(INT32_C( 2143921197), INT32_C( -566618658), INT32_C( -232811985), INT32_C( -646802339),
                            INT32_C(-1178364835), INT32_C( 1276977735), INT32_C(  767086016), INT32_C( -313594533),
                            INT32_C(-1508762742), INT32_C(   64124631), INT32_C(-1825241034), INT32_C(  688272909),
                            INT32_C(  217977439), INT32_C(  193086834), INT32_C( -334037547), INT32_C(  -96078706)),
      simde_mm_set_epi32(INT32_C(  217977439), INT32_C(  193086834), INT32_C( -334037547), INT32_C(  -96078706)) },
    { simde_mm512_set_epi32(INT32_C(-1446801805), INT32_C( 1701962052), INT32_C( 2110825600), INT32_C( 1003088425),
                            INT32_C(-1369466226), INT32_C( -588712121), INT32_C(-1911969632), INT32_C( 1166459977),
                            INT32_C(-1912318103), INT32_C( -539596217), INT32_C(-1472577858), INT32_C( -435893517),
                            INT32_C( 1340330175), INT32_C( 1762005905), INT32_C(-1553204174), INT32_C( -839776889)),
      simde_mm_set_epi32(INT32_C( 1340330175), INT32_C( 1762005905), INT32_C(-1553204174), INT32_C( -839776889)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_castsi512_si128(test_vec[i].a);
    simde_assert_m128i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_castsi512_si256(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1313023361), INT32_C( -284143420), INT32_C(  762022716), INT32_C( -109476439),
                            INT32_C( -377196873), INT32_C(  289021876), INT32_C(-1835156104), INT32_C(-2032178077),
                            INT32_C(-1821500948), INT32_C(  676317044), INT32_C( 1400280404), INT32_C(-1176819357),
                            INT32_C( -771867753), INT32_C(-1499090744), INT32_C(  845761822), INT32_C( 1051767340)),
      simde_mm256_set_epi32(INT32_C(-1821500948), INT32_C(  676317044), INT32_C( 1400280404), INT32_C(-1176819357),
                            INT32_C( -771867753), INT32_C(-1499090744), INT32_C(  845761822), INT32_C( 1051767340)) },
    { simde_mm512_set_epi32(INT32_C(  584849517), INT32_C( 1957310042), INT32_C( 1025237717), INT32_C( -771085485),
                            INT32_C(  578687068), INT32_C(   23449370), INT32_C(-1748604352), INT32_C( 1528872918),
                            INT32_C(  331610618), INT32_C( 1589449489), INT32_C(-1723663783), INT32_C( -274890341),
                            INT32_C( 1614271470), INT32_C(-1087921179), INT32_C(  318041946), INT32_C(-1241918784)),
      simde_mm256_set_epi32(INT32_C(  331610618), INT32_C( 1589449489), INT32_C(-1723663783), INT32_C( -274890341),
                            INT32_C( 1614271470), INT32_C(-1087921179), INT32_C(  318041946), INT32_C(-1241918784)) },
    { simde_mm512_set_epi32(INT32_C(-1852610712), INT32_C( 1069895831), INT32_C(-1772412775), INT32_C( 1674180593),
                            INT32_C( 1091612022), INT32_C( 2100501546), INT32_C(  179298125), INT32_C( -696375983),
                            INT32_C(  697782088), INT32_C(  360978520), INT32_C( -569372444), INT32_C(-1688535258),
                            INT32_C(  857372618), INT32_C( 1720533351), INT32_C(  174096645), INT32_C(  -27592678)),
      simde_mm256_set_epi32(INT32_C(  697782088), INT32_C(  360978520), INT32_C( -569372444), INT32_C(-1688535258),
                            INT32_C(  857372618), INT32_C( 1720533351), INT32_C(  174096645), INT32_C(  -27592678)) },
    { simde_mm512_set_epi32(INT32_C( 1588790835), INT32_C( 1641509494), INT32_C(-1255607211), INT32_C(  325546446),
                            INT32_C( -545478700), INT32_C( 1405876195), INT32_C(-1090548279), INT32_C(-1934286399),
                            INT32_C(-1145918593), INT32_C( -667441949), INT32_C(  804839434), INT32_C( 1326022329),
                            INT32_C(  307496614), INT32_C(-1108181850), INT32_C( 1980353111), INT32_C( 2112031083)),
      simde_mm256_set_epi32(INT32_C(-1145918593), INT32_C( -667441949), INT32_C(  804839434), INT32_C( 1326022329),
                            INT32_C(  307496614), INT32_C(-1108181850), INT32_C( 1980353111), INT32_C( 2112031083)) },
    { simde_mm512_set_epi32(INT32_C(-1042067236), INT32_C(-1304341967), INT32_C(  157202478), INT32_C(  335492980),
                            INT32_C( 1721471627), INT32_C( -970374929), INT32_C(-1215103856), INT32_C(  804236438),
                            INT32_C(  763203116), INT32_C( 1650455375), INT32_C(  191870396), INT32_C(-1509651897),
                            INT32_C( 1356692638), INT32_C(-1388072975), INT32_C(-1822839305), INT32_C(  511824893)),
      simde_mm256_set_epi32(INT32_C(  763203116), INT32_C( 1650455375), INT32_C(  191870396), INT32_C(-1509651897),
                            INT32_C( 1356692638), INT32_C(-1388072975), INT32_C(-1822839305), INT32_C(  511824893)) },
    { simde_mm512_set_epi32(INT32_C( -321038022), INT32_C( 1580565478), INT32_C( 1887212628), INT32_C( -321448210),
                            INT32_C( 1646995861), INT32_C(  223122732), INT32_C(-1757872907), INT32_C(-1745139814),
                            INT32_C( 1574697064), INT32_C( 1593370323), INT32_C( 1019611869), INT32_C( 1148772416),
                            INT32_C(   41212650), INT32_C( 1180707726), INT32_C(-1448405590), INT32_C( -480086797)),
      simde_mm256_set_epi32(INT32_C( 1574697064), INT32_C( 1593370323), INT32_C( 1019611869), INT32_C( 1148772416),
                            INT32_C(   41212650), INT32_C( 1180707726), INT32_C(-1448405590), INT32_C( -480086797)) },
    { simde_mm512_set_epi32(INT32_C(-2123925649), INT32_C(-1300783990), INT32_C( 1491522954), INT32_C(-1497225833),
                            INT32_C(  264593112), INT32_C(-1577295575), INT32_C( 1721102182), INT32_C( -406919611),
                            INT32_C( -248074414), INT32_C(-1644442090), INT32_C(-1586232899), INT32_C(  -62204050),
                            INT32_C(  356135618), INT32_C(-1239722051), INT32_C( 1765867002), INT32_C(-2007643147)),
      simde_mm256_set_epi32(INT32_C( -248074414), INT32_C(-1644442090), INT32_C(-1586232899), INT32_C(  -62204050),
                            INT32_C(  356135618), INT32_C(-1239722051), INT32_C( 1765867002), INT32_C(-2007643147)) },
    { simde_mm512_set_epi32(INT32_C( 1712658433), INT32_C( 1782026188), INT32_C( 1026238577), INT32_C(  999831740),
                            INT32_C( 1890262543), INT32_C( -282267234), INT32_C( -314081065), INT32_C( 1258071145),
                            INT32_C(  -96847849), INT32_C(  -33628993), INT32_C( -700755553), INT32_C( 1395877145),
                            INT32_C( 1943621426), INT32_C( 1606414493), INT32_C(  234520793), INT32_C( 1932887352)),
      simde_mm256_set_epi32(INT32_C(  -96847849), INT32_C(  -33628993), INT32_C( -700755553), INT32_C( 1395877145),
                            INT32_C( 1943621426), INT32_C( 1606414493), INT32_C(  234520793), INT32_C( 1932887352)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_castsi512_si256(test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cmpgt_epi32_mask(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__mmask16 r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( -126651070), INT32_C( 1757388710), INT32_C(  617530196), INT32_C(  407807901),
                            INT32_C( 1271989524), INT32_C( 1251214807), INT32_C(-1247045111), INT32_C(-1024057759),
                            INT32_C(   50729453), INT32_C(  464444874), INT32_C( 1840702207), INT32_C( 1916050591),
                            INT32_C(  484601458), INT32_C( -782065931), INT32_C(-1485735658), INT32_C(-1326388993)),
      simde_mm512_set_epi32(INT32_C(  111072774), INT32_C( 1757388710), INT32_C( 1496897687), INT32_C(  407807901),
                            INT32_C( 1271989524), INT32_C( 1496985365), INT32_C(-1247045111), INT32_C(-1225014979),
                            INT32_C(   50729453), INT32_C(  464444874), INT32_C(  924537351), INT32_C( 1916050591),
                            INT32_C(  484601458), INT32_C( -782065931), INT32_C(-1485735658), INT32_C(-1326388993)),
      UINT16_C(  288) },
    { simde_mm512_set_epi32(INT32_C( 2106044062), INT32_C( 1752498924), INT32_C(-1086695378), INT32_C(  627787891),
                            INT32_C(-1783053554), INT32_C(-1485517848), INT32_C( 1105114322), INT32_C(-1862707588),
                            INT32_C(  823946037), INT32_C(-2030244995), INT32_C( -219180660), INT32_C(  810910725),
                            INT32_C( -642105946), INT32_C(  760774613), INT32_C(  -62603432), INT32_C(-2064446807)),
      simde_mm512_set_epi32(INT32_C( 2106044062), INT32_C( 1752498924), INT32_C( -582421212), INT32_C( 1649238471),
                            INT32_C( 1446053889), INT32_C(-1485517848), INT32_C( 1105114322), INT32_C(-1862707588),
                            INT32_C( -846383385), INT32_C(-2030244995), INT32_C( -905258415), INT32_C(  810910725),
                            INT32_C(-1668595380), INT32_C( -760772652), INT32_C( 2145797270), INT32_C(   57887151)),
      UINT16_C(  172) },
    { simde_mm512_set_epi32(INT32_C(  948728954), INT32_C(  965445469), INT32_C( -298261731), INT32_C( 1889741023),
                            INT32_C(  101476677), INT32_C( -598834633), INT32_C( 1592735604), INT32_C(  428243294),
                            INT32_C(-2001034764), INT32_C( -639043872), INT32_C(  567427880), INT32_C(-1305749494),
                            INT32_C( -204185535), INT32_C( -550643286), INT32_C( -170363385), INT32_C( 1483518213)),
      simde_mm512_set_epi32(INT32_C(  948728954), INT32_C(  965445469), INT32_C(  364841947), INT32_C(-1221758106),
                            INT32_C(  101476677), INT32_C( -598834633), INT32_C( 1592735604), INT32_C(-1456245493),
                            INT32_C(-2001034764), INT32_C( -639043872), INT32_C(  567427880), INT32_C(-1305749494),
                            INT32_C( -204185535), INT32_C(  830345587), INT32_C( -170363385), INT32_C( -603563929)),
      UINT16_C( 4353) },
    { simde_mm512_set_epi32(INT32_C( -163413000), INT32_C( -831194762), INT32_C( -664019578), INT32_C( 2031024026),
                            INT32_C( 1912388774), INT32_C(  982200166), INT32_C(  596130243), INT32_C(  446035443),
                            INT32_C( 1373006598), INT32_C(-1540837035), INT32_C( 1581631435), INT32_C(-2083299381),
                            INT32_C( 1992847454), INT32_C(  448258110), INT32_C(  875345838), INT32_C( 1612926819)),
      simde_mm512_set_epi32(INT32_C(   81573630), INT32_C( -831194762), INT32_C( -664019578), INT32_C( 2031024026),
                            INT32_C( 1912388774), INT32_C(-2072470454), INT32_C(  596130243), INT32_C( 1961646011),
                            INT32_C( 1373006598), INT32_C( -197223193), INT32_C( 1581631435), INT32_C(-2083299381),
                            INT32_C( 1457480410), INT32_C( 1181119535), INT32_C( 1263228451), INT32_C(-1998542716)),
      UINT16_C( 1033) },
    { simde_mm512_set_epi32(INT32_C( 1436278246), INT32_C(   99684976), INT32_C( 1345577484), INT32_C(  166701508),
                            INT32_C( -780731111), INT32_C( -840749601), INT32_C( 1523342039), INT32_C( 1058674665),
                            INT32_C( -523908416), INT32_C( 1659465207), INT32_C(-1927062215), INT32_C(-1156760340),
                            INT32_C(  715569317), INT32_C(-1515814414), INT32_C( 1243253180), INT32_C( 2080215882)),
      simde_mm512_set_epi32(INT32_C(  432908742), INT32_C(   99684976), INT32_C(  -14330157), INT32_C(-1223154556),
                            INT32_C( -780731111), INT32_C(  696697372), INT32_C( 1523342039), INT32_C( 1058674665),
                            INT32_C( -523908416), INT32_C( 1659465207), INT32_C(-1927062215), INT32_C(-1156760340),
                            INT32_C( -171262349), INT32_C(-1515814414), INT32_C(-1234169573), INT32_C(-1847568101)),
      UINT16_C(45067) },
    { simde_mm512_set_epi32(INT32_C( 1399825551), INT32_C(-1064541474), INT32_C( 2112452992), INT32_C(  575137303),
                            INT32_C( -979898374), INT32_C(-1476679333), INT32_C( 1320423852), INT32_C( 1767893242),
                            INT32_C( -389599783), INT32_C(-1459729991), INT32_C(  995424065), INT32_C( -522129019),
                            INT32_C( -466751981), INT32_C( 1371238810), INT32_C( 1006677155), INT32_C( 1609037982)),
      simde_mm512_set_epi32(INT32_C( 1399825551), INT32_C(-1064541474), INT32_C( 2112452992), INT32_C(  134645750),
                            INT32_C(  500192289), INT32_C( 1600988950), INT32_C( 1320423852), INT32_C( 1198845893),
                            INT32_C( -389599783), INT32_C( 1504468794), INT32_C(  995424065), INT32_C(-2123865443),
                            INT32_C( -466751981), INT32_C(-1711282630), INT32_C( 1006677155), INT32_C( 1609037982)),
      UINT16_C( 4372) },
    { simde_mm512_set_epi32(INT32_C(-1862774816), INT32_C(   28374488), INT32_C(  250156705), INT32_C( -932694837),
                            INT32_C(-2079251566), INT32_C( -246439183), INT32_C( -875109534), INT32_C( 1740046060),
                            INT32_C( 1735819269), INT32_C( 1371885292), INT32_C( -914870851), INT32_C( -473073032),
                            INT32_C( -580976455), INT32_C( 1688786028), INT32_C(  637430498), INT32_C(-1740972685)),
      simde_mm512_set_epi32(INT32_C(-1862774816), INT32_C(   28374488), INT32_C(  580744870), INT32_C( -666445473),
                            INT32_C( -129274908), INT32_C( -928751425), INT32_C( -388443661), INT32_C( 1740046060),
                            INT32_C(-1909361652), INT32_C( 1371885292), INT32_C(  857928163), INT32_C(   37075976),
                            INT32_C( -580976455), INT32_C(-1545948444), INT32_C(  637430498), INT32_C(-1740972685)),
      UINT16_C( 1156) },
    { simde_mm512_set_epi32(INT32_C(-1890406982), INT32_C( 2110791016), INT32_C( 1083476771), INT32_C( -620691621),
                            INT32_C(  543588207), INT32_C( -227503647), INT32_C( -759273149), INT32_C(  775085710),
                            INT32_C( 1404885802), INT32_C(-1395233065), INT32_C(  832528180), INT32_C( 1065959566),
                            INT32_C(-2083201484), INT32_C(  937916550), INT32_C( -710457746), INT32_C( -246147415)),
      simde_mm512_set_epi32(INT32_C(   84669207), INT32_C(  470641840), INT32_C( 1083476771), INT32_C( -620691621),
                            INT32_C( 1099959895), INT32_C( -961354454), INT32_C(-1751384146), INT32_C(  775085710),
                            INT32_C( 1075765582), INT32_C(  834655006), INT32_C(  832528180), INT32_C( 1065959566),
                            INT32_C(  954342416), INT32_C(  937916550), INT32_C(-1946395018), INT32_C(  757651617)),
      UINT16_C(18050) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask16 r = simde_mm512_cmpgt_epi32_mask(test_vec[i].a, test_vec[i].b);
    simde_assert_mmask16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cmpgt_epi32_mask(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__mmask16 r;
  } test_vec[8] = {
       { UINT16_C(12249),
      simde_mm512_set_epi32(INT32_C(-1151856667), INT32_C(  -49918748), INT32_C(-1709830250), INT32_C( 1750293451),
                            INT32_C(-1728641738), INT32_C(   79295022), INT32_C(  308064941), INT32_C( 1216157597),
                            INT32_C(  320231148), INT32_C( -697875804), INT32_C(  990066955), INT32_C(-2091005609),
                            INT32_C( 1037816180), INT32_C( -748290940), INT32_C( 1166526776), INT32_C(-1428331975)),
      simde_mm512_set_epi32(INT32_C( 1846695950), INT32_C(  884046092), INT32_C( -392734818), INT32_C(-1618937497),
                            INT32_C(  670851975), INT32_C(-1269946840), INT32_C(-1917256160), INT32_C(  228203505),
                            INT32_C( 1263965918), INT32_C(-2053175323), INT32_C(-1206891799), INT32_C( -371464947),
                            INT32_C(  -99745007), INT32_C( -847464628), INT32_C(  -93605380), INT32_C(-1859612096)),
      UINT16_C( 1865) },
    { UINT16_C(47912),
      simde_mm512_set_epi32(INT32_C(  238725197), INT32_C( 1521340392), INT32_C(-2077410041), INT32_C( 2110592657),
                            INT32_C(  630925822), INT32_C(  689275449), INT32_C(-1970822997), INT32_C(-1836727953),
                            INT32_C(  237271984), INT32_C( -578417637), INT32_C( -768235708), INT32_C( 1362514984),
                            INT32_C( 2019276284), INT32_C(-1411239380), INT32_C( 2052618114), INT32_C(-1238213534)),
      simde_mm512_set_epi32(INT32_C( -669654436), INT32_C( -822780196), INT32_C(  205688995), INT32_C( 1680146061),
                            INT32_C(  393599682), INT32_C(-1451941808), INT32_C(  947305201), INT32_C(  -75999449),
                            INT32_C( -484166756), INT32_C(  833555969), INT32_C( -549302423), INT32_C( 1610578173),
                            INT32_C(-2070337581), INT32_C(  664412106), INT32_C(  255732930), INT32_C( 1319359183)),
      UINT16_C(38920) },
    { UINT16_C(35968),
      simde_mm512_set_epi32(INT32_C( -359339347), INT32_C( -666198998), INT32_C(  830421084), INT32_C(-2087460228),
                            INT32_C(-1475104408), INT32_C(  721289147), INT32_C( 1281161083), INT32_C(  852871551),
                            INT32_C(-1589170839), INT32_C( -746357372), INT32_C( -154169474), INT32_C( -148954645),
                            INT32_C(-1357418925), INT32_C(-2112116028), INT32_C(  174617048), INT32_C(   -4103177)),
      simde_mm512_set_epi32(INT32_C( -871886017), INT32_C(-1688556984), INT32_C(  524690471), INT32_C( -124192434),
                            INT32_C( 1423100629), INT32_C( -161239972), INT32_C( -396308843), INT32_C( 1070701282),
                            INT32_C(-1826016016), INT32_C(  425347357), INT32_C(  366897524), INT32_C( -401692546),
                            INT32_C( -812557761), INT32_C( 1614519786), INT32_C(-1648390428), INT32_C( 1830061179)),
      UINT16_C(33920) },
    { UINT16_C(16809),
      simde_mm512_set_epi32(INT32_C(-1280324837), INT32_C( -161347329), INT32_C(  735858798), INT32_C( -674499230),
                            INT32_C(  -21391979), INT32_C( -381504266), INT32_C( 1528317100), INT32_C(  430345703),
                            INT32_C( -679679907), INT32_C(  515475896), INT32_C( -731085275), INT32_C( 1308429395),
                            INT32_C(  582932299), INT32_C(  489160586), INT32_C( 1760701165), INT32_C(  118948129)),
      simde_mm512_set_epi32(INT32_C( 1240889055), INT32_C(  888122014), INT32_C(-1469013917), INT32_C( 1209972337),
                            INT32_C( -691361230), INT32_C(  361393556), INT32_C(-1167116782), INT32_C( 1467757879),
                            INT32_C( 2124803699), INT32_C( -154870634), INT32_C(-1603618479), INT32_C(-2032203238),
                            INT32_C( 2111773805), INT32_C( -496949166), INT32_C( 1844580247), INT32_C(-2053814402)),
      UINT16_C(   33) },
    { UINT16_C(44464),
      simde_mm512_set_epi32(INT32_C( 1072149321), INT32_C( 1813169024), INT32_C(-1284365076), INT32_C(-1623700182),
                            INT32_C(  530512850), INT32_C(  116537892), INT32_C(  258206492), INT32_C(  690441736),
                            INT32_C( 1005371642), INT32_C( 1116924342), INT32_C( 1297564984), INT32_C( -835039581),
                            INT32_C( 1286263864), INT32_C(-1749149234), INT32_C(  558298824), INT32_C( 1064688827)),
      simde_mm512_set_epi32(INT32_C( 1323805616), INT32_C(-1558886902), INT32_C( 1778691088), INT32_C(  108147743),
                            INT32_C( 1106435712), INT32_C( -967535450), INT32_C(  600280311), INT32_C(  109364043),
                            INT32_C(  423389578), INT32_C( 1225761441), INT32_C( -440804681), INT32_C( -707540326),
                            INT32_C(-1898655855), INT32_C(-1268681648), INT32_C(-1360056367), INT32_C( -275254487)),
      UINT16_C( 1440) },
    { UINT16_C(23993),
      simde_mm512_set_epi32(INT32_C(-2038065128), INT32_C( -446679229), INT32_C(   78082001), INT32_C(  379830516),
                            INT32_C(-1929569644), INT32_C( 1595859976), INT32_C(  320798226), INT32_C( -738570818),
                            INT32_C( -165441023), INT32_C( -172594873), INT32_C(  912601062), INT32_C(  -56802863),
                            INT32_C(  503255814), INT32_C( 2046199592), INT32_C( -622599746), INT32_C( 1337235103)),
      simde_mm512_set_epi32(INT32_C(-1519343201), INT32_C( -448055921), INT32_C(-1909251875), INT32_C( -347447915),
                            INT32_C(  397553753), INT32_C(  713040821), INT32_C(-1458903601), INT32_C(  -45886582),
                            INT32_C( 1230465483), INT32_C( -828483015), INT32_C( -699493978), INT32_C(-1811052070),
                            INT32_C( 1577065087), INT32_C( -109599940), INT32_C(-1093577090), INT32_C(-1788879767)),
      UINT16_C(21553) },
    { UINT16_C(10358),
      simde_mm512_set_epi32(INT32_C(  648390363), INT32_C(  -30837841), INT32_C(-1635592815), INT32_C( -694389961),
                            INT32_C( -883952626), INT32_C( -761345991), INT32_C(  346040825), INT32_C(-1780780575),
                            INT32_C( 1510717568), INT32_C(-1185143236), INT32_C( 2143540932), INT32_C(  880567806),
                            INT32_C(-1670993371), INT32_C(-1942419167), INT32_C(-1196759463), INT32_C( 1386099146)),
      simde_mm512_set_epi32(INT32_C(-1614031176), INT32_C(  414071648), INT32_C(-1152911954), INT32_C(  424701353),
                            INT32_C( 1739922394), INT32_C( -506382165), INT32_C(  257126844), INT32_C( 1724223193),
                            INT32_C( 1096709845), INT32_C(-1643231112), INT32_C(-1639890652), INT32_C( -403971200),
                            INT32_C( 1318667734), INT32_C(  206062573), INT32_C(  -18472190), INT32_C(   -1701112)),
      UINT16_C(  112) },
    { UINT16_C(35023),
      simde_mm512_set_epi32(INT32_C(  228305355), INT32_C(-1904004735), INT32_C(  118523411), INT32_C( 1661507666),
                            INT32_C(-1400326500), INT32_C(   63010183), INT32_C(   62197704), INT32_C( -635599967),
                            INT32_C( 1677709284), INT32_C(-1294080152), INT32_C( -900737233), INT32_C(-1991940005),
                            INT32_C( -240404149), INT32_C(-1448242105), INT32_C(-1972665039), INT32_C( 1511694245)),
      simde_mm512_set_epi32(INT32_C(-1506289043), INT32_C(   82234507), INT32_C( -557930538), INT32_C( -911612825),
                            INT32_C( 1352158017), INT32_C( -554125937), INT32_C( -727289650), INT32_C(-1102664191),
                            INT32_C( 1941639559), INT32_C(-2124299952), INT32_C( -385431179), INT32_C(  112242864),
                            INT32_C(  -66697069), INT32_C( 1379403470), INT32_C(-1996504296), INT32_C(  658235880)),
      UINT16_C(32835) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask16 r = simde_mm512_mask_cmpgt_epi32_mask(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_mmask16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cmpgt_epi64_mask(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__mmask8 r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-3344943500899736927), INT64_C( -508674271294480923),
                            INT64_C( 4367550852745697236), INT64_C(-1765523250257788813),
                            INT64_C(-6325172456788566604), INT64_C( 3340966423446181237),
                            INT64_C( 4899669249714194025), INT64_C(-3109134868060088778)),
      simde_mm512_set_epi64(INT64_C(-3344943500899736927), INT64_C( -115747659216396604),
                            INT64_C( 7173930031566073597), INT64_C(-1765523250257788813),
                            INT64_C(-4525526016698522721), INT64_C( 3340966423446181237),
                            INT64_C( 4899669249714194025), INT64_C(-3109134868060088778)),
      UINT8_C(  0) },
    { simde_mm512_set_epi64(INT64_C(  161300839730502102), INT64_C(-4154011080047619988),
                            INT64_C( 3510183776865199207), INT64_C( 5188039415407364902),
                            INT64_C(-8649541106015277925), INT64_C( 2036065042708484429),
                            INT64_C(-7714656548902029220), INT64_C(-9105101012109136347)),
      simde_mm512_set_epi64(INT64_C(  161300839730502102), INT64_C(-4154011080047619988),
                            INT64_C( 6120426103009778754), INT64_C( -248023738166197182),
                            INT64_C(-8649541106015277925), INT64_C( 2693047687180833180),
                            INT64_C(-7714656548902029220), INT64_C(-9105101012109136347)),
      UINT8_C( 16) },
    { simde_mm512_set_epi64(INT64_C(-2825253727352691686), INT64_C( 4405965118825132522),
                            INT64_C(-6791426899562410985), INT64_C(-4409900925389880930),
                            INT64_C( 1845845856613597884), INT64_C(-4842241423465696621),
                            INT64_C(  163081221433998591), INT64_C( 4482804709675222173)),
      simde_mm512_set_epi64(INT64_C( 3285810068291760082), INT64_C( 4405965118825132522),
                            INT64_C(-6791426899562410985), INT64_C(-4943963491966669187),
                            INT64_C( 1845845856613597884), INT64_C( 7384036385676540465),
                            INT64_C(-7313503223753260102), INT64_C( 5128036791088991318)),
      UINT8_C( 18) },
    { simde_mm512_set_epi64(INT64_C(-1633105180711142836), INT64_C(-4287337651200520652),
                            INT64_C( 8346707004388378871), INT64_C(-5848595418894109542),
                            INT64_C(-7300386321370732776), INT64_C( -648586863376006844),
                            INT64_C(-3473939784680689044), INT64_C(-1628617817613399979)),
      simde_mm512_set_epi64(INT64_C( 1934898870952452550), INT64_C(-4287337651200520652),
                            INT64_C( 1557479703737443505), INT64_C(-5848595418894109542),
                            INT64_C(-7179299072208562799), INT64_C( -648586863376006844),
                            INT64_C(-3473939784680689044), INT64_C( 2817575692173645704)),
      UINT8_C( 32) },
    { simde_mm512_set_epi64(INT64_C(-8952123954418726140), INT64_C( 5461301954902244462),
                            INT64_C(-5820184907423972656), INT64_C(  420402622060248705),
                            INT64_C(-1664441445637860283), INT64_C(-9088734991256809986),
                            INT64_C( 5606803261787264235), INT64_C( 3392608019150722653)),
      simde_mm512_set_epi64(INT64_C(-8952123954418726140), INT64_C(-6318099565586317695),
                            INT64_C(-5820184907423972656), INT64_C(  420402622060248705),
                            INT64_C(-1664441445637860283), INT64_C( 6565206217411025613),
                            INT64_C( 8598198622090956400), INT64_C(-7576266643160730964)),
      UINT8_C( 65) },
    { simde_mm512_set_epi64(INT64_C(-3313522622815895345), INT64_C(-6452175545498154090),
                            INT64_C( -937049212555566038), INT64_C(-4143019958444030865),
                            INT64_C(-3410665359562609619), INT64_C(  966786109195223540),
                            INT64_C( 7283097367839393163), INT64_C(-2640534975929709368)),
      simde_mm512_set_epi64(INT64_C(-3313522622815895345), INT64_C(-6452175545498154090),
                            INT64_C( 7057508826094118763), INT64_C(-2466255848420720587),
                            INT64_C(-3410665359562609619), INT64_C(-7091282311083875172),
                            INT64_C(-5778676633446214654), INT64_C(-2640534975929709368)),
      UINT8_C(  6) },
    { simde_mm512_set_epi64(INT64_C( 7946101066156420330), INT64_C(-1199223599247032864),
                            INT64_C(-1997073553979895023), INT64_C(-2305098272308636911),
                            INT64_C( -630363562210498119), INT64_C( 4426020973322885294),
                            INT64_C( 8782098874831326668), INT64_C(-6058337867533474769)),
      simde_mm512_set_epi64(INT64_C( 7946101066156420330), INT64_C(-1199223599247032864),
                            INT64_C(-1997073553979895023), INT64_C(-2305098272308636911),
                            INT64_C( -630363562210498119), INT64_C( 8629524505567702841),
                            INT64_C( 8782098874831326668), INT64_C( 2660246489815857132)),
      UINT8_C(  0) },
    { simde_mm512_set_epi64(INT64_C(-2815932903868980343), INT64_C(  791308056982133256),
                            INT64_C( 8277712790583824674), INT64_C(-3943050990178000322),
                            INT64_C(-2127265598488665647), INT64_C( 4379715049649431166),
                            INT64_C(-9154071905230416728), INT64_C(-2123362159730266714)),
      simde_mm512_set_epi64(INT64_C(-2815932903868980343), INT64_C(  791308056982133256),
                            INT64_C(-6685750631550937327), INT64_C( 1585978438239301211),
                            INT64_C( 3432556139556266760), INT64_C( 4379715049649431166),
                            INT64_C(-9154071905230416728), INT64_C(-1483875325616410698)),
      UINT8_C( 32) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask8 r = simde_mm512_cmpgt_epi64_mask(test_vec[i].a, test_vec[i].b);
    simde_assert_mmask8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cmpgt_epi64_mask(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__mmask8 r;
  } test_vec[8] = {
    { UINT8_C( 16),
      simde_mm512_set_epi64(INT64_C( 2255026789087372129), INT64_C( 6954636019969939696),
                            INT64_C( 8135587588110756767), INT64_C(-6775895683000468083),
                            INT64_C( -900701109459786534), INT64_C(-7915280239032503360),
                            INT64_C(-1216817989725562739), INT64_C(-8432176550710264558)),
      simde_mm512_set_epi64(INT64_C( 2255026789087372129), INT64_C( 6954636019969939696),
                            INT64_C( 8135587588110756767), INT64_C(-6775895683000468083),
                            INT64_C( -900701109459786534), INT64_C(-3598046066747317833),
                            INT64_C(-1709941778912207388), INT64_C( 3169788859761289772)),
      UINT8_C(  0) },
    { UINT8_C(  6),
      simde_mm512_set_epi64(INT64_C(-2239217399172416071), INT64_C(-1788368623206610532),
                            INT64_C(-8621353419023287056), INT64_C( 2167646577764527484),
                            INT64_C( 7373916114077140533), INT64_C( 4679914315089846813),
                            INT64_C(-4785189148228967071), INT64_C(-2291884851836211058)),
      simde_mm512_set_epi64(INT64_C(-4674113889822188979), INT64_C( 6851357122574510655),
                            INT64_C(-8621353419023287056), INT64_C( 2167646577764527484),
                            INT64_C( 7373916114077140533), INT64_C(-2091630176064440564),
                            INT64_C(-4166997073722787613), INT64_C(-2291884851836211058)),
      UINT8_C(  4) },
    { UINT8_C(231),
      simde_mm512_set_epi64(INT64_C( -437845083503418422), INT64_C( 7030863114044209442),
                            INT64_C( 8238064857893579595), INT64_C( 3062172269146343786),
                            INT64_C( 7457006241836305381), INT64_C(-9078752323516671886),
                            INT64_C(-6382075143273833301), INT64_C( 3840898770164583597)),
      simde_mm512_set_epi64(INT64_C(-4268408126209392137), INT64_C( 7030863114044209442),
                            INT64_C(  663353489862938549), INT64_C( 3062172269146343786),
                            INT64_C( 7457006241836305381), INT64_C( 8174310593560152615),
                            INT64_C(-6382075143273833301), INT64_C(-4495103935185291795)),
      UINT8_C(161) },
    { UINT8_C( 60),
      simde_mm512_set_epi64(INT64_C( 3543184366849060052), INT64_C( 8101296544771348510),
                            INT64_C( 1359772700119148960), INT64_C(-8357828074665392254),
                            INT64_C(-5672294839872616078), INT64_C(-2918525673450782654),
                            INT64_C(-6303315662009814438), INT64_C( 4773615511108508590)),
      simde_mm512_set_epi64(INT64_C( 3543184366849060052), INT64_C(  286276641590586651),
                            INT64_C( 1359772700119148960), INT64_C(-3217204137928962858),
                            INT64_C(-5672294839872616078), INT64_C(-2918525673450782654),
                            INT64_C(-2554453706959743566), INT64_C(-6197005744039272430)),
      UINT8_C(  0) },
    { UINT8_C( 97),
      simde_mm512_set_epi64(INT64_C(-4278296701436995238), INT64_C( 3569507405853529045),
                            INT64_C(-3380367559374400304), INT64_C(-4948363566435325304),
                            INT64_C(-6678298576976263631), INT64_C( 8848650777417470336),
                            INT64_C( 6320411494008491541), INT64_C( 2280208700508329072)),
      simde_mm512_set_epi64(INT64_C(  326944370261152484), INT64_C( 3569507405853529045),
                            INT64_C(  715678757448860576), INT64_C(-4948363566435325304),
                            INT64_C(-6678298576976263631), INT64_C(-5367013526541491012),
                            INT64_C( 1008601224594483315), INT64_C( 2280208700508329072)),
      UINT8_C(  0) },
    { UINT8_C(153),
      simde_mm512_set_epi64(INT64_C( 8361426666750729591), INT64_C(-6668359429543518025),
                            INT64_C( 2952092805333509636), INT64_C( 8284871946243647248),
                            INT64_C(-8896262213455925533), INT64_C( 3194469353298560173),
                            INT64_C( 5466230282228711049), INT64_C(-1091365868294702661)),
      simde_mm512_set_epi64(INT64_C(-8667260419906723988), INT64_C(-6668359429543518025),
                            INT64_C( 2952092805333509636), INT64_C( 8284871946243647248),
                            INT64_C( 3185065043241333471), INT64_C( -134870333477219304),
                            INT64_C( 5466230282228711049), INT64_C(-4571723861926798973)),
      UINT8_C(129) },
    { UINT8_C( 60),
      simde_mm512_set_epi64(INT64_C(-5632979726637184794), INT64_C( 3790754159972080576),
                            INT64_C(-7842038005332057398), INT64_C(-1292705499011984897),
                            INT64_C( 7597886654367336733), INT64_C( 1457057381762531412),
                            INT64_C(-1572264173383359920), INT64_C(-8716209376375056305)),
      simde_mm512_set_epi64(INT64_C(-5632979726637184794), INT64_C( 3790754159972080576),
                            INT64_C( 1913605115921194336), INT64_C(-6143563121944184390),
                            INT64_C( 7597886654367336733), INT64_C( 1457057381762531412),
                            INT64_C( 7253226870637562008), INT64_C(-6283001429373579825)),
      UINT8_C( 16) },
    { UINT8_C( 88),
      simde_mm512_set_epi64(INT64_C(-2374777447002601129), INT64_C(-5785141086360428669),
                            INT64_C( 6450311718709789609), INT64_C( 4609381622161693926),
                            INT64_C( -638886780002324864), INT64_C(-5739159461288227194),
                            INT64_C(-4392084870376418631), INT64_C( 2798977638636065147)),
      simde_mm512_set_epi64(INT64_C(  753500986908300233), INT64_C(-5785141086360428669),
                            INT64_C( 6450311718709789609), INT64_C(-4648819914956469219),
                            INT64_C(-8767820380557260648), INT64_C(-5739159461288227194),
                            INT64_C( 2360822030941279123), INT64_C(-6092063218708168180)),
      UINT8_C( 24) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask8 r = simde_mm512_mask_cmpgt_epi64_mask(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_mmask8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cmpeq_epi32_mask(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__mmask16 r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( 1955445938), INT32_C( 1791143901), INT32_C(-1554982337), INT32_C(-1864115653),
                            INT32_C(-1774796435), INT32_C( 1168347531), INT32_C(  660969508), INT32_C( 1153796239),
                            INT32_C(  609464964), INT32_C( 1687040663), INT32_C( -477087011), INT32_C(  309017072),
                            INT32_C(-2144421691), INT32_C(  682838279), INT32_C( 1954361304), INT32_C( 1694661655)),
      simde_mm512_set_epi32(INT32_C( 1955445938), INT32_C( 1791143901), INT32_C(-1554982337), INT32_C(-1021004514),
                            INT32_C(-1774796435), INT32_C( 1168347531), INT32_C( 1662960443), INT32_C( 1153796239),
                            INT32_C(  609464964), INT32_C(  428287860), INT32_C(  954212709), INT32_C(  309017072),
                            INT32_C(-2144421691), INT32_C( 1263291650), INT32_C( 1954361304), INT32_C( 1694661655)),
      UINT16_C(60827) },
    { simde_mm512_set_epi32(INT32_C(-1966343749), INT32_C(  200215975), INT32_C(-1192030414), INT32_C(  122433675),
                            INT32_C( 2024442800), INT32_C(-1612503082), INT32_C( -352764842), INT32_C( -964919402),
                            INT32_C(  198779956), INT32_C(-1052322954), INT32_C(-2053476283), INT32_C( 1410443780),
                            INT32_C( -220052733), INT32_C( 1401656692), INT32_C(  344284159), INT32_C(  358488145)),
      simde_mm512_set_epi32(INT32_C(-1966343749), INT32_C(  200215975), INT32_C( 1606502639), INT32_C(  122433675),
                            INT32_C( 2024442800), INT32_C( 1676122341), INT32_C( 1013297923), INT32_C( 1764819720),
                            INT32_C(-1006160158), INT32_C(  500282446), INT32_C(-2053476283), INT32_C( 1410443780),
                            INT32_C( -891361692), INT32_C( 1401656692), INT32_C(-1666424273), INT32_C(  358488145)),
      UINT16_C(55349) },
    { simde_mm512_set_epi32(INT32_C( 2029514541), INT32_C(-1405121342), INT32_C( -922299669), INT32_C(-1157312309),
                            INT32_C(  348700274), INT32_C( 1888848128), INT32_C( -909115111), INT32_C(-1689020830),
                            INT32_C( -310128293), INT32_C(-2105034529), INT32_C( 1894199289), INT32_C( -520350707),
                            INT32_C( 2077151614), INT32_C(  947639177), INT32_C(  972348424), INT32_C(  688864219)),
      simde_mm512_set_epi32(INT32_C( 2029514541), INT32_C(-1405121342), INT32_C( -922299669), INT32_C(-1157312309),
                            INT32_C( -582496350), INT32_C(  321618555), INT32_C( -707023911), INT32_C(-1689020830),
                            INT32_C(-1223923200), INT32_C( -293312096), INT32_C( 1894199289), INT32_C(  -89856908),
                            INT32_C( 2077151614), INT32_C(  947639177), INT32_C(  352633301), INT32_C( -580477735)),
      UINT16_C(61740) },
    { simde_mm512_set_epi32(INT32_C(   63224893), INT32_C( -945243939), INT32_C( 1472177104), INT32_C(-1518639341),
                            INT32_C( 1244274955), INT32_C(-1053463974), INT32_C(  207788622), INT32_C( -375726536),
                            INT32_C( -219606791), INT32_C( -871332353), INT32_C( 2036105492), INT32_C( 1465626684),
                            INT32_C(  930923741), INT32_C(-1996202276), INT32_C(  336972805), INT32_C(-1729161210)),
      simde_mm512_set_epi32(INT32_C( 1739683658), INT32_C( -945243939), INT32_C( 1472177104), INT32_C( -959740920),
                            INT32_C( 1244274955), INT32_C(  236525659), INT32_C(  207788622), INT32_C( 1159372556),
                            INT32_C( -632841040), INT32_C( -871332353), INT32_C( 2036105492), INT32_C(-1821161401),
                            INT32_C(  626098659), INT32_C(-1099705903), INT32_C( 1892226777), INT32_C(  183269504)),
      UINT16_C(27232) },
    { simde_mm512_set_epi32(INT32_C( 1660264790), INT32_C( -188014963), INT32_C( 1960568786), INT32_C(  630575470),
                            INT32_C(-1560285386), INT32_C(-1080983958), INT32_C( -186614663), INT32_C(-1365084922),
                            INT32_C( 1687374482), INT32_C( 2091712477), INT32_C( 1770300152), INT32_C( 1222615684),
                            INT32_C(  987382002), INT32_C( -869689297), INT32_C( 1381156346), INT32_C(  352829646)),
      simde_mm512_set_epi32(INT32_C( 1495188549), INT32_C( -188014963), INT32_C( -879412194), INT32_C( 1172150075),
                            INT32_C( 1163780404), INT32_C(-1080983958), INT32_C( -186614663), INT32_C(-1365084922),
                            INT32_C( 1196042729), INT32_C( 2091712477), INT32_C( 1770300152), INT32_C( 1222615684),
                            INT32_C(  987382002), INT32_C( -453542339), INT32_C(-1460537486), INT32_C( 1311735715)),
      UINT16_C(18296) },
    { simde_mm512_set_epi32(INT32_C( -830898164), INT32_C( 2065530031), INT32_C( 1849339474), INT32_C( -161498764),
                            INT32_C(  726295410), INT32_C(-1366062470), INT32_C(  110025501), INT32_C(-2061598845),
                            INT32_C(-1911113344), INT32_C(-2008355607), INT32_C( 1140427951), INT32_C( 1963231912),
                            INT32_C( 1593065931), INT32_C( 1712671682), INT32_C(-2139143015), INT32_C(  330252777)),
      simde_mm512_set_epi32(INT32_C( -830898164), INT32_C(-2132024757), INT32_C( 1102342058), INT32_C( -161498764),
                            INT32_C(-2106128090), INT32_C(  648329890), INT32_C(-1284054768), INT32_C(-2061598845),
                            INT32_C(-1911113344), INT32_C(-2008355607), INT32_C( 1715485148), INT32_C(  155412419),
                            INT32_C( 1273550758), INT32_C( 1712671682), INT32_C(-1857983881), INT32_C( 1633779150)),
      UINT16_C(37316) },
    { simde_mm512_set_epi32(INT32_C( 1956746364), INT32_C( 1930323834), INT32_C(  923874794), INT32_C(  121318212),
                            INT32_C(-1375858452), INT32_C( -462992597), INT32_C( 1495829546), INT32_C(  697040437),
                            INT32_C(  727111035), INT32_C(-2061427382), INT32_C( -815432287), INT32_C(  913775211),
                            INT32_C(-1333809472), INT32_C(  114048073), INT32_C( 1312920985), INT32_C(-1819914035)),
      simde_mm512_set_epi32(INT32_C( 1659376087), INT32_C( 1930323834), INT32_C(  923874794), INT32_C(  121318212),
                            INT32_C(-1375858452), INT32_C(-1480844812), INT32_C(-1803673478), INT32_C(  697040437),
                            INT32_C( 1793922150), INT32_C(  391658500), INT32_C( -815432287), INT32_C(  913775211),
                            INT32_C(-1333809472), INT32_C( -169359358), INT32_C(  140424991), INT32_C(-1819914035)),
      UINT16_C(31033) },
    { simde_mm512_set_epi32(INT32_C( 1334496661), INT32_C(-1765072906), INT32_C(-1980138391), INT32_C(-1150536116),
                            INT32_C( -711226926), INT32_C( 1955166809), INT32_C( 1418224832), INT32_C( 1791996583),
                            INT32_C(-1305868646), INT32_C( -507537618), INT32_C(  272749509), INT32_C(-1826072492),
                            INT32_C( -629068596), INT32_C(-2142583585), INT32_C( 2048200365), INT32_C(-1377550438)),
      simde_mm512_set_epi32(INT32_C( 1334496661), INT32_C(-1765072906), INT32_C( -890751438), INT32_C(-1150536116),
                            INT32_C(  307879329), INT32_C( 1955166809), INT32_C(-1884386825), INT32_C( 1791996583),
                            INT32_C( 1128431085), INT32_C( -507537618), INT32_C(  272749509), INT32_C( 1579228324),
                            INT32_C( 1577134581), INT32_C(-2142583585), INT32_C( 1998674783), INT32_C(-1377550438)),
      UINT16_C(54629) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask16 r = simde_mm512_cmpeq_epi32_mask(test_vec[i].a, test_vec[i].b);
    simde_assert_mmask16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cmpeq_epi32_mask(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m512i b;
    simde__mmask16 r;
  } test_vec[8] = {
    { UINT16_C(15798),
      simde_mm512_set_epi32(INT32_C(-1396783922), INT32_C( 2147469122), INT32_C(  245941047), INT32_C(-1608794680),
                            INT32_C( 1508622706), INT32_C( -820009589), INT32_C(-2056933337), INT32_C( 1399160559),
                            INT32_C( -781782717), INT32_C( -745938688), INT32_C( 1376412239), INT32_C(  358147007),
                            INT32_C(-2141927311), INT32_C(  -13921241), INT32_C(  993767039), INT32_C( 1399506469)),
      simde_mm512_set_epi32(INT32_C(-1396783922), INT32_C( 1246457300), INT32_C( 1479047358), INT32_C(  -46329110),
                            INT32_C( 1508622706), INT32_C( -820009589), INT32_C( 2080173402), INT32_C( 1937046285),
                            INT32_C( 1642337112), INT32_C(  717149471), INT32_C(  913865239), INT32_C(  358147007),
                            INT32_C( 1535690363), INT32_C(  -13921241), INT32_C(  993767039), INT32_C(  520030741)),
      UINT16_C( 3094) },
    { UINT16_C(11237),
      simde_mm512_set_epi32(INT32_C( -503773363), INT32_C(-1842967576), INT32_C(  214407733), INT32_C( 2136243689),
                            INT32_C(  619563347), INT32_C(-2046647578), INT32_C( -882198373), INT32_C( -832110086),
                            INT32_C(  828288790), INT32_C( 1819046419), INT32_C(  292064966), INT32_C( -747926472),
                            INT32_C( -881463995), INT32_C(-1469434386), INT32_C( -207196114), INT32_C(-1865668915)),
      simde_mm512_set_epi32(INT32_C(  261150369), INT32_C(-1842967576), INT32_C(-1148601855), INT32_C( 2136243689),
                            INT32_C(-2013121518), INT32_C(-2046647578), INT32_C(-1816537753), INT32_C( -832110086),
                            INT32_C(  828288790), INT32_C( 1801925150), INT32_C(  292064966), INT32_C(-1755078036),
                            INT32_C(-1190065823), INT32_C(  764421376), INT32_C( -207196114), INT32_C(-1865668915)),
      UINT16_C(  417) },
    { UINT16_C(   61),
      simde_mm512_set_epi32(INT32_C(  219790698), INT32_C(  346516102), INT32_C(-2082495343), INT32_C(  775700745),
                            INT32_C( -630025741), INT32_C( 1914162819), INT32_C( -226095226), INT32_C( -350619769),
                            INT32_C( 1825330894), INT32_C(  -79420124), INT32_C(  939662489), INT32_C(  667606641),
                            INT32_C(-1935137328), INT32_C(  247120548), INT32_C( -894861328), INT32_C(-1690188311)),
      simde_mm512_set_epi32(INT32_C(  400967600), INT32_C(  346516102), INT32_C(-2082495343), INT32_C(  775700745),
                            INT32_C( -630025741), INT32_C( 1914162819), INT32_C(-1771983812), INT32_C( -350619769),
                            INT32_C( 1825330894), INT32_C(-1887033010), INT32_C(  939662489), INT32_C( -440942769),
                            INT32_C(-1935137328), INT32_C(  247120548), INT32_C(-1352163534), INT32_C(-1690188311)),
      UINT16_C(   45) },
    { UINT16_C(40799),
      simde_mm512_set_epi32(INT32_C(-1103261115), INT32_C(-1530092257), INT32_C( -178076517), INT32_C( 1725180044),
                            INT32_C( -619562554), INT32_C(-2028225116), INT32_C( -516465044), INT32_C( -790492311),
                            INT32_C(  770588317), INT32_C(  966229539), INT32_C( 1696981823), INT32_C( 1846986452),
                            INT32_C(  201003704), INT32_C(  -88144749), INT32_C( -459260562), INT32_C(-1462493092)),
      simde_mm512_set_epi32(INT32_C(-1103261115), INT32_C(-1530092257), INT32_C( 1272329003), INT32_C( 1725180044),
                            INT32_C(-1714282865), INT32_C( 1964019033), INT32_C(   -6888948), INT32_C( -794446809),
                            INT32_C(  770588317), INT32_C(  966229539), INT32_C(-1587543669), INT32_C( 1846986452),
                            INT32_C(  201003704), INT32_C(  -88144749), INT32_C(-1549459108), INT32_C(-1462493092)),
      UINT16_C(36957) },
    { UINT16_C(18708),
      simde_mm512_set_epi32(INT32_C( 1654168369), INT32_C(-1358646009), INT32_C(  945188582), INT32_C( 1242452940),
                            INT32_C(-2068238117), INT32_C(  613827224), INT32_C( 1766050173), INT32_C(  788865946),
                            INT32_C( -226150288), INT32_C(   20626714), INT32_C(-1790747056), INT32_C(-1510999017),
                            INT32_C(-2059568770), INT32_C(  525242273), INT32_C(-1970979230), INT32_C( -983788353)),
      simde_mm512_set_epi32(INT32_C(-1802152524), INT32_C(-1358646009), INT32_C( 1400410557), INT32_C( 1242452940),
                            INT32_C(-2068238117), INT32_C(-1745049433), INT32_C(-1272787498), INT32_C(   36641197),
                            INT32_C( -226150288), INT32_C(-2087200149), INT32_C( -530182364), INT32_C(-1510999017),
                            INT32_C(-2082577633), INT32_C(  525242273), INT32_C(-1970979230), INT32_C( -983788353)),
      UINT16_C(18452) },
    { UINT16_C(48938),
      simde_mm512_set_epi32(INT32_C(-1802886705), INT32_C(  505130099), INT32_C( 1294359394), INT32_C(  564426410),
                            INT32_C( -813242663), INT32_C(-1097324530), INT32_C( 1599346411), INT32_C(-1815738445),
                            INT32_C( 2114996332), INT32_C( 2143192037), INT32_C(  342894910), INT32_C( 1933006347),
                            INT32_C(  215936041), INT32_C( 2138148935), INT32_C(-1975112588), INT32_C(-1313889253)),
      simde_mm512_set_epi32(INT32_C( 1272515820), INT32_C(-1571014987), INT32_C( 1294359394), INT32_C(  564426410),
                            INT32_C( -305474417), INT32_C(-2099686495), INT32_C(  217917259), INT32_C( 1770631752),
                            INT32_C( 2114996332), INT32_C( 2143192037), INT32_C(  -26985081), INT32_C(  603877714),
                            INT32_C( 1592556524), INT32_C(  420570241), INT32_C(-1975112588), INT32_C(-1313889253)),
      UINT16_C(12290) },
    { UINT16_C(14127),
      simde_mm512_set_epi32(INT32_C(  452796731), INT32_C( -256668338), INT32_C(-1710549095), INT32_C( 1982965424),
                            INT32_C( 1184306045), INT32_C( -221254467), INT32_C( 1420239721), INT32_C( 2028887361),
                            INT32_C(-1950932361), INT32_C( 1650853943), INT32_C(  239751123), INT32_C( 1018010808),
                            INT32_C( -248946240), INT32_C(  701510715), INT32_C(  824235240), INT32_C( 1829156606)),
      simde_mm512_set_epi32(INT32_C(  452796731), INT32_C( 1031814185), INT32_C(-1710549095), INT32_C(  406415467),
                            INT32_C( 1184306045), INT32_C( -221254467), INT32_C(  419739010), INT32_C( 1708161231),
                            INT32_C(-1950932361), INT32_C( 1650853943), INT32_C(  239751123), INT32_C( 1018010808),
                            INT32_C( -248946240), INT32_C(  701510715), INT32_C(-1571248435), INT32_C( 1829156606)),
      UINT16_C( 9261) },
    { UINT16_C(22801),
      simde_mm512_set_epi32(INT32_C( 1869800572), INT32_C(  184060195), INT32_C(   81710208), INT32_C( -451284065),
                            INT32_C(  397153235), INT32_C(  120564446), INT32_C(-2128920097), INT32_C( 1498011427),
                            INT32_C( -602736654), INT32_C( -931955343), INT32_C(  270436915), INT32_C( -984637478),
                            INT32_C( 2080482721), INT32_C( 1599947836), INT32_C(  374268618), INT32_C(  202341051)),
      simde_mm512_set_epi32(INT32_C( 1869800572), INT32_C(  350721255), INT32_C( 1725621650), INT32_C( 2020045509),
                            INT32_C(  397153235), INT32_C( 2059505832), INT32_C(-2128920097), INT32_C( 1498011427),
                            INT32_C(  884679844), INT32_C( -931955343), INT32_C(-1565261303), INT32_C( -984637478),
                            INT32_C( 1047792745), INT32_C(  969830078), INT32_C(  374268618), INT32_C(  202341051)),
      UINT16_C( 2321) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask16 r = simde_mm512_mask_cmpeq_epi32_mask(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_mmask16(HEDLEY_STATIC_CAST(uint16_t, r), ==, HEDLEY_STATIC_CAST(uint16_t, test_vec[i].r));
  }

  return 0;
}

static int
test_simde_mm512_cmpeq_epi64_mask(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m512i b;
    simde__mmask8 r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C( 1145569124203592220), INT64_C( 8866992319046943109),
                            INT64_C( 1920152028348566704), INT64_C( 5434169962120345100),
                            INT64_C( 2279810443797316081), INT64_C( 8202334326145056493),
                            INT64_C(-3178237508368099649), INT64_C(  691539695110831332)),
      simde_mm512_set_epi64(INT64_C( 1145569124203592220), INT64_C( 7456258988741005047),
                            INT64_C( 1920152028348566704), INT64_C(-5531344059509930758),
                            INT64_C( 2279810443797316081), INT64_C( 4212838215119313762),
                            INT64_C(-3178237508368099649), INT64_C(  805234941373423218)),
      UINT8_C(170) },
    { simde_mm512_set_epi64(INT64_C(-9153030146845322943), INT64_C(-3269802522838954453),
                            INT64_C( 4057416377680772191), INT64_C(-4770808841142629569),
                            INT64_C(-3341852712217874901), INT64_C( 2807150139607841402),
                            INT64_C(-4019938614639236980), INT64_C(-5612844059017155926)),
      simde_mm512_set_epi64(INT64_C(-9153030146845322943), INT64_C(-3269802522838954453),
                            INT64_C( 4057416377680772191), INT64_C( -940603917401247915),
                            INT64_C(-3341852712217874901), INT64_C( 2807150139607841402),
                            INT64_C(-4019938614639236980), INT64_C(-5612844059017155926)),
      UINT8_C(239) },
    { simde_mm512_set_epi64(INT64_C(-6535160640888369935), INT64_C( 4320507985166557084),
                            INT64_C( 5472888856009482484), INT64_C(-3128657528300165720),
                            INT64_C( 7430914891859325397), INT64_C( -863913734109164798),
                            INT64_C(-7794735185814972844), INT64_C( 6163895309000776850)),
      simde_mm512_set_epi64(INT64_C(-4220461743444256802), INT64_C( 2616373937292152846),
                            INT64_C( 5472888856009482484), INT64_C(-3128657528300165720),
                            INT64_C(-4693544589206901195), INT64_C( -863913734109164798),
                            INT64_C(-7794735185814972844), INT64_C(-7914074467628930001)),
      UINT8_C( 54) },
    { simde_mm512_set_epi64(INT64_C(-2366835550617683621), INT64_C(-7526131353484035546),
                            INT64_C(  792273323175818479), INT64_C(-3293855504021481788),
                            INT64_C( 5766970185621377329), INT64_C( 8555682923177627503),
                            INT64_C( 6363802824474944219), INT64_C(-2294667848278645794)),
      simde_mm512_set_epi64(INT64_C(-2869910567061155737), INT64_C(-7526131353484035546),
                            INT64_C(  792273323175818479), INT64_C( 7002007300200137801),
                            INT64_C( 5766970185621377329), INT64_C( 8555682923177627503),
                            INT64_C( 6363802824474944219), INT64_C( 1235305386817146646)),
      UINT8_C(110) },
    { simde_mm512_set_epi64(INT64_C( 1781453983767744725), INT64_C( 8717105342239974860),
                            INT64_C(-6555437070120516632), INT64_C( -945434448901737124),
                            INT64_C( 2467182069214891728), INT64_C( 6380355612148977321),
                            INT64_C(-8948278762783625779), INT64_C( 4377995125356963906)),
      simde_mm512_set_epi64(INT64_C( 1781453983767744725), INT64_C( 8717105342239974860),
                            INT64_C(-1574265126641933862), INT64_C(-3609136820784425910),
                            INT64_C( 8413977304256335681), INT64_C(-4046135395677773903),
                            INT64_C(-8948278762783625779), INT64_C( 4377995125356963906)),
      UINT8_C(195) },
    { simde_mm512_set_epi64(INT64_C( 3163831140256245302), INT64_C(-7476767975543057611),
                            INT64_C( 4783231484571490676), INT64_C( 4356333140933542181),
                            INT64_C(-5734470250608567849), INT64_C( 5428089484060124354),
                            INT64_C( 6915844972668556704), INT64_C(-4128418267655054219)),
      simde_mm512_set_epi64(INT64_C( 3163831140256245302), INT64_C(-7945608864951271413),
                            INT64_C(-3728561425301803734), INT64_C( 6344562138072151085),
                            INT64_C(-5734470250608567849), INT64_C( 5868624435741359252),
                            INT64_C( 6915844972668556704), INT64_C(-4128418267655054219)),
      UINT8_C(139) },
    { simde_mm512_set_epi64(INT64_C(-8545152605640787948), INT64_C(-5234822620280611494),
                            INT64_C(-2932089901585751375), INT64_C( 4017618173912988951),
                            INT64_C(-8696436452927061736), INT64_C( 3602642025812661939),
                            INT64_C( 5777136991119584953), INT64_C(-5473038481952171581)),
      simde_mm512_set_epi64(INT64_C(-8545152605640787948), INT64_C(-7808252369899371159),
                            INT64_C(-2932089901585751375), INT64_C(  -26139149052321087),
                            INT64_C(-6986660196527912755), INT64_C( 3602642025812661939),
                            INT64_C( 5777136991119584953), INT64_C(-5473038481952171581)),
      UINT8_C(167) },
    { simde_mm512_set_epi64(INT64_C( 7780170108497689334), INT64_C(-8001282944915881932),
                            INT64_C(  382835809361431399), INT64_C(-5014881555296189468),
                            INT64_C(-1844642447215154571), INT64_C(-5452282829002750089),
                            INT64_C(-7793611854809744260), INT64_C(  938166230586687295)),
      simde_mm512_set_epi64(INT64_C( 7780170108497689334), INT64_C( 1739290942316187796),
                            INT64_C( 7815402837606564081), INT64_C(-5014881555296189468),
                            INT64_C(-1844642447215154571), INT64_C( -253128228754997390),
                            INT64_C(-7793611854809744260), INT64_C(  938166230586687295)),
      UINT8_C(155) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask8 r = simde_mm512_cmpeq_epi64_mask(test_vec[i].a, test_vec[i].b);
    simde_assert_mmask8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cmpeq_epi64_mask(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m512i b;
    simde__mmask8 r;
  } test_vec[8] = {
       { UINT8_C( 90),
      simde_mm512_set_epi64(INT64_C( 7722926897436765530), INT64_C( 7338279138551748064),
                            INT64_C( 8433308126101200079), INT64_C(-4390305748733976547),
                            INT64_C(-1482589068035252753), INT64_C(-5002011091694476743),
                            INT64_C( 5809674310022718254), INT64_C( 7620249298233351482)),
      simde_mm512_set_epi64(INT64_C( 7722926897436765530), INT64_C( 7338279138551748064),
                            INT64_C(-2656726859984743367), INT64_C(-4390305748733976547),
                            INT64_C(-1482589068035252753), INT64_C(-5269390469191050553),
                            INT64_C( 5809674310022718254), INT64_C( 7620249298233351482)),
      UINT8_C( 90) },
    { UINT8_C(178),
      simde_mm512_set_epi64(INT64_C(-8806453660480970182), INT64_C(-2722914847628644365),
                            INT64_C( 2171146127585219679), INT64_C(-7200523266853707115),
                            INT64_C( 8505301695237968355), INT64_C( 6373940775215479358),
                            INT64_C( 8209357864908427195), INT64_C( -995665125730760835)),
      simde_mm512_set_epi64(INT64_C(-8806453660480970182), INT64_C(-2722914847628644365),
                            INT64_C(-1180134256156200317), INT64_C(-7200523266853707115),
                            INT64_C(-1610604796376715795), INT64_C( 5419019224867820225),
                            INT64_C( 8209357864908427195), INT64_C( -995665125730760835)),
      UINT8_C(146) },
    { UINT8_C(171),
      simde_mm512_set_epi64(INT64_C(-6245801519083893310), INT64_C(-7866373458730819532),
                            INT64_C(-5627757407772356197), INT64_C(-2425546480980122794),
                            INT64_C(-8451301604567613199), INT64_C( 1369383717682198649),
                            INT64_C( -532343328754521574), INT64_C(-1062878680437210584)),
      simde_mm512_set_epi64(INT64_C(-9163399881020056955), INT64_C(-2992244142829238392),
                            INT64_C( -213476403626539965), INT64_C(-8591297333400286921),
                            INT64_C(-8451301604567613199), INT64_C(-8139768780594538635),
                            INT64_C(-4714070518405120331), INT64_C(-1062878680437210584)),
      UINT8_C(  9) },
    { UINT8_C( 28),
      simde_mm512_set_epi64(INT64_C( 7845956693704412298), INT64_C(-5781930833336581955),
                            INT64_C( 2851517750261041799), INT64_C(-5814293521236182366),
                            INT64_C( 2292150971239308783), INT64_C( 2594053186857735013),
                            INT64_C( 7307535341641173075), INT64_C(-4427478291595527940)),
      simde_mm512_set_epi64(INT64_C(  536264388241191871), INT64_C(-5781930833336581955),
                            INT64_C( 2851517750261041799), INT64_C( 1349842462394812975),
                            INT64_C( 2292150971239308783), INT64_C( 2594053186857735013),
                            INT64_C( 7307535341641173075), INT64_C(-4427478291595527940)),
      UINT8_C( 12) },
    { UINT8_C(248),
      simde_mm512_set_epi64(INT64_C( 4900816215694077255), INT64_C(-2732029741423656661),
                            INT64_C( 1082977534221618055), INT64_C(-3092044493389993636),
                            INT64_C(-4299277917890019767), INT64_C(-2055775203132417874),
                            INT64_C( -778633101599852237), INT64_C( -563223173848121636)),
      simde_mm512_set_epi64(INT64_C( 7049304296219110648), INT64_C(-2732029741423656661),
                            INT64_C( 7088083428992159722), INT64_C(-3092044493389993636),
                            INT64_C(-4299277917890019767), INT64_C( 4225506809727089751),
                            INT64_C( -778633101599852237), INT64_C( -563223173848121636)),
      UINT8_C( 88) },
    { UINT8_C(171),
      simde_mm512_set_epi64(INT64_C(-1412821155990992029), INT64_C( 4454576651901490962),
                            INT64_C(-7284760734604447652), INT64_C(-7443130466673006479),
                            INT64_C(  320054597637804434), INT64_C(-8860872372305530355),
                            INT64_C(-8428145646879978292), INT64_C(-6547252853189215611)),
      simde_mm512_set_epi64(INT64_C(-1412821155990992029), INT64_C(-2354123670646573707),
                            INT64_C( 4506838144989822528), INT64_C(-7443130466673006479),
                            INT64_C(-5147543239321546686), INT64_C(-8860872372305530355),
                            INT64_C(-8428145646879978292), INT64_C(-6547252853189215611)),
      UINT8_C(131) },
    { UINT8_C( 29),
      simde_mm512_set_epi64(INT64_C( 5675137803130124480), INT64_C( 1211541157654985046),
                            INT64_C( 8724633375562564314), INT64_C(-2760658800846254598),
                            INT64_C(-6714474269646576270), INT64_C( 3484180661422871715),
                            INT64_C( 1469796163712815354), INT64_C(-2336393240308600160)),
      simde_mm512_set_epi64(INT64_C( 5675137803130124480), INT64_C( 1211541157654985046),
                            INT64_C(-8867413355151838495), INT64_C(-8867147959443474315),
                            INT64_C(-6714474269646576270), INT64_C( 3484180661422871715),
                            INT64_C(-7735267815657951749), INT64_C(  413036036281601883)),
      UINT8_C( 12) },
    { UINT8_C(211),
      simde_mm512_set_epi64(INT64_C(-6713502673628263139), INT64_C( 1559753162601267291),
                            INT64_C( 5045660940436454371), INT64_C( 7013290440433503154),
                            INT64_C(-8475145246816690249), INT64_C(-6834826688677600633),
                            INT64_C(-2109099044497919348), INT64_C( 1351143524438105934)),
      simde_mm512_set_epi64(INT64_C( 5625319538109918668), INT64_C( 1559753162601267291),
                            INT64_C( 5045660940436454371), INT64_C(-4654386914804892920),
                            INT64_C( 2407237530895996207), INT64_C(-6834826688677600633),
                            INT64_C( 4684210505965066200), INT64_C( 1351143524438105934)),
      UINT8_C( 65) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__mmask8 r = simde_mm512_mask_cmpeq_epi64_mask(test_vec[i].k, test_vec[i].a, test_vec[i].b);
    simde_assert_mmask8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cvtepi8_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C(   6), INT8_C(  33), INT8_C( 124), INT8_C(-128),
                        INT8_C(  38), INT8_C(  59), INT8_C( -37), INT8_C( 121),
                        INT8_C(  67), INT8_C( 112), INT8_C(  62), INT8_C(-113),
                        INT8_C(-121), INT8_C( -89), INT8_C(  63), INT8_C( -72)),
      simde_mm512_set_epi32(INT32_C(          6), INT32_C(         33), INT32_C(        124), INT32_C(       -128),
                            INT32_C(         38), INT32_C(         59), INT32_C(        -37), INT32_C(        121),
                            INT32_C(         67), INT32_C(        112), INT32_C(         62), INT32_C(       -113),
                            INT32_C(       -121), INT32_C(        -89), INT32_C(         63), INT32_C(        -72)) },
    { simde_mm_set_epi8(INT8_C( -94), INT8_C( 125), INT8_C( 105), INT8_C( -92),
                        INT8_C(-107), INT8_C( -69), INT8_C(  25), INT8_C(-125),
                        INT8_C( -11), INT8_C( -44), INT8_C(  60), INT8_C( -96),
                        INT8_C( 119), INT8_C( -40), INT8_C(  29), INT8_C(   9)),
      simde_mm512_set_epi32(INT32_C(        -94), INT32_C(        125), INT32_C(        105), INT32_C(        -92),
                            INT32_C(       -107), INT32_C(        -69), INT32_C(         25), INT32_C(       -125),
                            INT32_C(        -11), INT32_C(        -44), INT32_C(         60), INT32_C(        -96),
                            INT32_C(        119), INT32_C(        -40), INT32_C(         29), INT32_C(          9)) },
    { simde_mm_set_epi8(INT8_C(  47), INT8_C(  -9), INT8_C(-127), INT8_C(  66),
                        INT8_C( 126), INT8_C( -95), INT8_C(  99), INT8_C( -25),
                        INT8_C( -19), INT8_C(  84), INT8_C( -96), INT8_C( -77),
                        INT8_C( -34), INT8_C(-108), INT8_C( -33), INT8_C(  62)),
      simde_mm512_set_epi32(INT32_C(         47), INT32_C(         -9), INT32_C(       -127), INT32_C(         66),
                            INT32_C(        126), INT32_C(        -95), INT32_C(         99), INT32_C(        -25),
                            INT32_C(        -19), INT32_C(         84), INT32_C(        -96), INT32_C(        -77),
                            INT32_C(        -34), INT32_C(       -108), INT32_C(        -33), INT32_C(         62)) },
    { simde_mm_set_epi8(INT8_C(  93), INT8_C(-106), INT8_C(  61), INT8_C(  64),
                        INT8_C( -98), INT8_C(  31), INT8_C( -75), INT8_C( -23),
                        INT8_C( 113), INT8_C( -94), INT8_C( -70), INT8_C( -73),
                        INT8_C(  66), INT8_C( -84), INT8_C(  92), INT8_C(-113)),
      simde_mm512_set_epi32(INT32_C(         93), INT32_C(       -106), INT32_C(         61), INT32_C(         64),
                            INT32_C(        -98), INT32_C(         31), INT32_C(        -75), INT32_C(        -23),
                            INT32_C(        113), INT32_C(        -94), INT32_C(        -70), INT32_C(        -73),
                            INT32_C(         66), INT32_C(        -84), INT32_C(         92), INT32_C(       -113)) },
    { simde_mm_set_epi8(INT8_C( -92), INT8_C( -17), INT8_C( -42), INT8_C( 103),
                        INT8_C( 111), INT8_C( -23), INT8_C(  14), INT8_C(-122),
                        INT8_C( -61), INT8_C(   1), INT8_C(  16), INT8_C(-124),
                        INT8_C( -62), INT8_C(  59), INT8_C( 106), INT8_C(-105)),
      simde_mm512_set_epi32(INT32_C(        -92), INT32_C(        -17), INT32_C(        -42), INT32_C(        103),
                            INT32_C(        111), INT32_C(        -23), INT32_C(         14), INT32_C(       -122),
                            INT32_C(        -61), INT32_C(          1), INT32_C(         16), INT32_C(       -124),
                            INT32_C(        -62), INT32_C(         59), INT32_C(        106), INT32_C(       -105)) },
    { simde_mm_set_epi8(INT8_C(  71), INT8_C(  82), INT8_C( -83), INT8_C( 118),
                        INT8_C( 127), INT8_C(  -5), INT8_C(  43), INT8_C( -48),
                        INT8_C(  67), INT8_C(-117), INT8_C( -15), INT8_C( 105),
                        INT8_C( -88), INT8_C( 127), INT8_C(  85), INT8_C(-110)),
      simde_mm512_set_epi32(INT32_C(         71), INT32_C(         82), INT32_C(        -83), INT32_C(        118),
                            INT32_C(        127), INT32_C(         -5), INT32_C(         43), INT32_C(        -48),
                            INT32_C(         67), INT32_C(       -117), INT32_C(        -15), INT32_C(        105),
                            INT32_C(        -88), INT32_C(        127), INT32_C(         85), INT32_C(       -110)) },
    { simde_mm_set_epi8(INT8_C(   8), INT8_C(  35), INT8_C( -44), INT8_C(  18),
                        INT8_C(   3), INT8_C(  -7), INT8_C(  34), INT8_C(  98),
                        INT8_C(  43), INT8_C(  97), INT8_C(-127), INT8_C( 121),
                        INT8_C( -47), INT8_C( -94), INT8_C( -74), INT8_C( -59)),
      simde_mm512_set_epi32(INT32_C(          8), INT32_C(         35), INT32_C(        -44), INT32_C(         18),
                            INT32_C(          3), INT32_C(         -7), INT32_C(         34), INT32_C(         98),
                            INT32_C(         43), INT32_C(         97), INT32_C(       -127), INT32_C(        121),
                            INT32_C(        -47), INT32_C(        -94), INT32_C(        -74), INT32_C(        -59)) },
    { simde_mm_set_epi8(INT8_C(  79), INT8_C(  98), INT8_C(  70), INT8_C( -29),
                        INT8_C(  61), INT8_C( 100), INT8_C( -92), INT8_C(  10),
                        INT8_C(-107), INT8_C( -13), INT8_C(  83), INT8_C(-102),
                        INT8_C( -54), INT8_C(  80), INT8_C(  43), INT8_C(  56)),
      simde_mm512_set_epi32(INT32_C(         79), INT32_C(         98), INT32_C(         70), INT32_C(        -29),
                            INT32_C(         61), INT32_C(        100), INT32_C(        -92), INT32_C(         10),
                            INT32_C(       -107), INT32_C(        -13), INT32_C(         83), INT32_C(       -102),
                            INT32_C(        -54), INT32_C(         80), INT32_C(         43), INT32_C(         56)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_cvtepi8_epi32(test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

#if !defined(SIMDE_NATIVE_ALIASES_TESTING)
static int
test_simde_mm512_cmp_ps_mask (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 a[16];
    const simde_float32 b[16];
    const int imm8;
    const simde__mmask16 r;
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(   662.74), SIMDE_FLOAT32_C(  -296.50), SIMDE_FLOAT32_C(    51.89), SIMDE_FLOAT32_C(  -877.47),
        SIMDE_FLOAT32_C(   872.85), SIMDE_FLOAT32_C(  -965.85), SIMDE_FLOAT32_C(  -688.43), SIMDE_FLOAT32_C(   580.42),
        SIMDE_FLOAT32_C(   433.23), SIMDE_FLOAT32_C(   579.01), SIMDE_FLOAT32_C(   251.51), SIMDE_FLOAT32_C(  -341.82),
        SIMDE_FLOAT32_C(  -712.66), SIMDE_FLOAT32_C(  -533.80), SIMDE_FLOAT32_C(   415.61), SIMDE_FLOAT32_C(   350.44) },
      { SIMDE_FLOAT32_C(  -243.73), SIMDE_FLOAT32_C(  -899.82), SIMDE_FLOAT32_C(  -876.74), SIMDE_FLOAT32_C(   548.84),
        SIMDE_FLOAT32_C(   112.62), SIMDE_FLOAT32_C(   228.46), SIMDE_FLOAT32_C(   592.01), SIMDE_FLOAT32_C(  -633.78),
        SIMDE_FLOAT32_C(   405.27), SIMDE_FLOAT32_C(   447.83), SIMDE_FLOAT32_C(  -706.03), SIMDE_FLOAT32_C(   628.46),
        SIMDE_FLOAT32_C(  -379.69), SIMDE_FLOAT32_C(   592.13), SIMDE_FLOAT32_C(   228.45), SIMDE_FLOAT32_C(  -716.95) },
       INT32_C(           0),
      UINT16_C(    0) },
    { { SIMDE_FLOAT32_C(  -704.37), SIMDE_FLOAT32_C(  -719.66), SIMDE_FLOAT32_C(  -594.42), SIMDE_FLOAT32_C(  -831.52),
        SIMDE_FLOAT32_C(  -685.51), SIMDE_FLOAT32_C(  -282.86), SIMDE_FLOAT32_C(   748.90), SIMDE_FLOAT32_C(   747.73),
        SIMDE_FLOAT32_C(  -703.85), SIMDE_FLOAT32_C(     0.41), SIMDE_FLOAT32_C(  -594.10), SIMDE_FLOAT32_C(  -416.51),
        SIMDE_FLOAT32_C(   466.61), SIMDE_FLOAT32_C(   821.52), SIMDE_FLOAT32_C(   933.94), SIMDE_FLOAT32_C(  -777.12) },
      { SIMDE_FLOAT32_C(   921.70), SIMDE_FLOAT32_C(  -942.80), SIMDE_FLOAT32_C(   771.72), SIMDE_FLOAT32_C(    34.32),
        SIMDE_FLOAT32_C(   285.66), SIMDE_FLOAT32_C(   363.73), SIMDE_FLOAT32_C(   400.54), SIMDE_FLOAT32_C(  -309.07),
        SIMDE_FLOAT32_C(  -188.44), SIMDE_FLOAT32_C(   694.50), SIMDE_FLOAT32_C(  -680.61), SIMDE_FLOAT32_C(   431.87),
        SIMDE_FLOAT32_C(   286.63), SIMDE_FLOAT32_C(   547.84), SIMDE_FLOAT32_C(   714.92), SIMDE_FLOAT32_C(   582.26) },
       INT32_C(           1),
      UINT16_C(35645) },
    { { SIMDE_FLOAT32_C(   828.18), SIMDE_FLOAT32_C(  -879.50), SIMDE_FLOAT32_C(   750.74), SIMDE_FLOAT32_C(  -857.33),
        SIMDE_FLOAT32_C(  -162.36), SIMDE_FLOAT32_C(   499.63), SIMDE_FLOAT32_C(   890.40), SIMDE_FLOAT32_C(   133.79),
        SIMDE_FLOAT32_C(  -499.96), SIMDE_FLOAT32_C(  -703.70), SIMDE_FLOAT32_C(   717.28), SIMDE_FLOAT32_C(   966.65),
        SIMDE_FLOAT32_C(  -882.18), SIMDE_FLOAT32_C(   651.22), SIMDE_FLOAT32_C(  -810.47), SIMDE_FLOAT32_C(  -960.48) },
      { SIMDE_FLOAT32_C(   708.42), SIMDE_FLOAT32_C(   961.25), SIMDE_FLOAT32_C(    73.84), SIMDE_FLOAT32_C(    -5.91),
        SIMDE_FLOAT32_C(   324.99), SIMDE_FLOAT32_C(  -525.63), SIMDE_FLOAT32_C(   685.02), SIMDE_FLOAT32_C(  -863.45),
        SIMDE_FLOAT32_C(  -831.12), SIMDE_FLOAT32_C(  -995.59), SIMDE_FLOAT32_C(   568.42), SIMDE_FLOAT32_C(   455.51),
        SIMDE_FLOAT32_C(   552.25), SIMDE_FLOAT32_C(   283.34), SIMDE_FLOAT32_C(    37.76), SIMDE_FLOAT32_C(   380.43) },
       INT32_C(           2),
      UINT16_C(53274) },
    { { SIMDE_FLOAT32_C(   403.83), SIMDE_FLOAT32_C(  -211.50), SIMDE_FLOAT32_C(   523.10), SIMDE_FLOAT32_C(  -758.53),
        SIMDE_FLOAT32_C(  -711.87), SIMDE_FLOAT32_C(   413.50), SIMDE_FLOAT32_C(   375.26), SIMDE_FLOAT32_C(  -211.83),
        SIMDE_FLOAT32_C(   709.80), SIMDE_FLOAT32_C(    92.55), SIMDE_FLOAT32_C(  -245.18), SIMDE_FLOAT32_C(   827.62),
        SIMDE_FLOAT32_C(  -256.23), SIMDE_FLOAT32_C(   -55.64), SIMDE_FLOAT32_C(   867.14), SIMDE_FLOAT32_C(  -547.81) },
      { SIMDE_FLOAT32_C(   -94.39), SIMDE_FLOAT32_C(   -59.02), SIMDE_FLOAT32_C(   446.28), SIMDE_FLOAT32_C(  -769.41),
        SIMDE_FLOAT32_C(   415.35), SIMDE_FLOAT32_C(   131.30), SIMDE_FLOAT32_C(  -632.86), SIMDE_FLOAT32_C(   584.23),
        SIMDE_FLOAT32_C(   135.71), SIMDE_FLOAT32_C(   935.56), SIMDE_FLOAT32_C(    39.74), SIMDE_FLOAT32_C(  -312.04),
        SIMDE_FLOAT32_C(   218.89), SIMDE_FLOAT32_C(  -922.50), SIMDE_FLOAT32_C(  -931.62), SIMDE_FLOAT32_C(  -377.28) },
       INT32_C(           3),
      UINT16_C(    0) },
    { { SIMDE_FLOAT32_C(  -134.00), SIMDE_FLOAT32_C(   591.48), SIMDE_FLOAT32_C(  -135.81), SIMDE_FLOAT32_C(   154.13),
        SIMDE_FLOAT32_C(     4.98), SIMDE_FLOAT32_C(  -760.55), SIMDE_FLOAT32_C(   942.30), SIMDE_FLOAT32_C(  -285.22),
        SIMDE_FLOAT32_C(   332.00), SIMDE_FLOAT32_C(  -302.88), SIMDE_FLOAT32_C(  -457.60), SIMDE_FLOAT32_C(  -924.23),
        SIMDE_FLOAT32_C(   641.48), SIMDE_FLOAT32_C(  -590.45), SIMDE_FLOAT32_C(  -472.04), SIMDE_FLOAT32_C(  -452.91) },
      { SIMDE_FLOAT32_C(   350.53), SIMDE_FLOAT32_C(   974.23), SIMDE_FLOAT32_C(  -222.32), SIMDE_FLOAT32_C(  -234.12),
        SIMDE_FLOAT32_C(   105.53), SIMDE_FLOAT32_C(   144.82), SIMDE_FLOAT32_C(  -649.88), SIMDE_FLOAT32_C(  -758.76),
        SIMDE_FLOAT32_C(    80.38), SIMDE_FLOAT32_C(   389.85), SIMDE_FLOAT32_C(   -70.81), SIMDE_FLOAT32_C(  -700.73),
        SIMDE_FLOAT32_C(   467.35), SIMDE_FLOAT32_C(    -2.42), SIMDE_FLOAT32_C(   -78.01), SIMDE_FLOAT32_C(  -666.65) },
       INT32_C(           4),
           UINT16_MAX },
    { { SIMDE_FLOAT32_C(  -410.94), SIMDE_FLOAT32_C(   786.18), SIMDE_FLOAT32_C(   487.47), SIMDE_FLOAT32_C(   594.03),
        SIMDE_FLOAT32_C(  -974.37), SIMDE_FLOAT32_C(   429.77), SIMDE_FLOAT32_C(  -691.18), SIMDE_FLOAT32_C(   357.63),
        SIMDE_FLOAT32_C(  -873.11), SIMDE_FLOAT32_C(  -148.78), SIMDE_FLOAT32_C(   433.40), SIMDE_FLOAT32_C(   768.37),
        SIMDE_FLOAT32_C(   260.77), SIMDE_FLOAT32_C(   961.36), SIMDE_FLOAT32_C(  -684.54), SIMDE_FLOAT32_C(  -388.70) },
      { SIMDE_FLOAT32_C(   935.59), SIMDE_FLOAT32_C(    93.14), SIMDE_FLOAT32_C(   377.18), SIMDE_FLOAT32_C(    41.12),
        SIMDE_FLOAT32_C(  -762.04), SIMDE_FLOAT32_C(   727.30), SIMDE_FLOAT32_C(   282.36), SIMDE_FLOAT32_C(   318.33),
        SIMDE_FLOAT32_C(   117.15), SIMDE_FLOAT32_C(  -788.45), SIMDE_FLOAT32_C(   617.60), SIMDE_FLOAT32_C(  -415.50),
        SIMDE_FLOAT32_C(   209.13), SIMDE_FLOAT32_C(  -460.41), SIMDE_FLOAT32_C(   -82.16), SIMDE_FLOAT32_C(   798.18) },
       INT32_C(           5),
      UINT16_C(14990) },
    { { SIMDE_FLOAT32_C(  -674.23), SIMDE_FLOAT32_C(  -594.68), SIMDE_FLOAT32_C(   392.22), SIMDE_FLOAT32_C(  -648.60),
        SIMDE_FLOAT32_C(   835.09), SIMDE_FLOAT32_C(   701.04), SIMDE_FLOAT32_C(   709.03), SIMDE_FLOAT32_C(   961.98),
        SIMDE_FLOAT32_C(  -447.74), SIMDE_FLOAT32_C(   142.43), SIMDE_FLOAT32_C(   730.35), SIMDE_FLOAT32_C(   813.03),
        SIMDE_FLOAT32_C(   103.79), SIMDE_FLOAT32_C(  -954.19), SIMDE_FLOAT32_C(  -575.67), SIMDE_FLOAT32_C(    39.38) },
      { SIMDE_FLOAT32_C(   138.95), SIMDE_FLOAT32_C(   801.51), SIMDE_FLOAT32_C(  -919.50), SIMDE_FLOAT32_C(   376.91),
        SIMDE_FLOAT32_C(   528.81), SIMDE_FLOAT32_C(   362.86), SIMDE_FLOAT32_C(  -304.76), SIMDE_FLOAT32_C(  -354.04),
        SIMDE_FLOAT32_C(   574.42), SIMDE_FLOAT32_C(  -687.16), SIMDE_FLOAT32_C(   230.46), SIMDE_FLOAT32_C(  -216.45),
        SIMDE_FLOAT32_C(  -147.57), SIMDE_FLOAT32_C(  -851.70), SIMDE_FLOAT32_C(  -418.27), SIMDE_FLOAT32_C(   178.19) },
       INT32_C(           6),
      UINT16_C( 7924) },
    { { SIMDE_FLOAT32_C(  -446.38), SIMDE_FLOAT32_C(   973.95), SIMDE_FLOAT32_C(   529.59), SIMDE_FLOAT32_C(  -611.29),
        SIMDE_FLOAT32_C(   674.99), SIMDE_FLOAT32_C(   238.63), SIMDE_FLOAT32_C(  -649.31), SIMDE_FLOAT32_C(  -772.76),
        SIMDE_FLOAT32_C(  -618.94), SIMDE_FLOAT32_C(  -918.96), SIMDE_FLOAT32_C(  -959.73), SIMDE_FLOAT32_C(   484.85),
        SIMDE_FLOAT32_C(  -873.15), SIMDE_FLOAT32_C(  -535.40), SIMDE_FLOAT32_C(  -475.77), SIMDE_FLOAT32_C(   265.80) },
      { SIMDE_FLOAT32_C(  -733.89), SIMDE_FLOAT32_C(  -395.27), SIMDE_FLOAT32_C(  -357.30), SIMDE_FLOAT32_C(   794.92),
        SIMDE_FLOAT32_C(   967.60), SIMDE_FLOAT32_C(   337.94), SIMDE_FLOAT32_C(  -559.13), SIMDE_FLOAT32_C(   542.02),
        SIMDE_FLOAT32_C(   650.78), SIMDE_FLOAT32_C(   671.33), SIMDE_FLOAT32_C(  -674.44), SIMDE_FLOAT32_C(  -496.79),
        SIMDE_FLOAT32_C(   819.63), SIMDE_FLOAT32_C(   -92.71), SIMDE_FLOAT32_C(   681.40), SIMDE_FLOAT32_C(  -626.75) },
       INT32_C(           7),
           UINT16_MAX },
    { { SIMDE_FLOAT32_C(  -118.76), SIMDE_FLOAT32_C(   210.99), SIMDE_FLOAT32_C(  -238.04), SIMDE_FLOAT32_C(  -443.77),
        SIMDE_FLOAT32_C(  -550.38), SIMDE_FLOAT32_C(   112.65), SIMDE_FLOAT32_C(  -216.52), SIMDE_FLOAT32_C(  -169.32),
        SIMDE_FLOAT32_C(   193.68), SIMDE_FLOAT32_C(  -176.25), SIMDE_FLOAT32_C(  -684.48), SIMDE_FLOAT32_C(   320.53),
        SIMDE_FLOAT32_C(   288.35), SIMDE_FLOAT32_C(  -160.25), SIMDE_FLOAT32_C(  -413.67), SIMDE_FLOAT32_C(   554.45) },
      { SIMDE_FLOAT32_C(   444.49), SIMDE_FLOAT32_C(   229.03), SIMDE_FLOAT32_C(   349.37), SIMDE_FLOAT32_C(   412.09),
        SIMDE_FLOAT32_C(  -433.02), SIMDE_FLOAT32_C(   790.25), SIMDE_FLOAT32_C(   -45.90), SIMDE_FLOAT32_C(  -782.24),
        SIMDE_FLOAT32_C(   461.58), SIMDE_FLOAT32_C(   279.66), SIMDE_FLOAT32_C(  -279.03), SIMDE_FLOAT32_C(   281.21),
        SIMDE_FLOAT32_C(  -813.04), SIMDE_FLOAT32_C(  -597.63), SIMDE_FLOAT32_C(   654.46), SIMDE_FLOAT32_C(    68.20) },
       INT32_C(           8),
      UINT16_C(    0) },
    { { SIMDE_FLOAT32_C(   613.36), SIMDE_FLOAT32_C(  -583.58), SIMDE_FLOAT32_C(   624.43), SIMDE_FLOAT32_C(  -937.02),
        SIMDE_FLOAT32_C(   529.07), SIMDE_FLOAT32_C(  -592.09), SIMDE_FLOAT32_C(  -106.35), SIMDE_FLOAT32_C(  -277.25),
        SIMDE_FLOAT32_C(   231.66), SIMDE_FLOAT32_C(   209.18), SIMDE_FLOAT32_C(  -956.71), SIMDE_FLOAT32_C(  -480.00),
        SIMDE_FLOAT32_C(  -951.07), SIMDE_FLOAT32_C(  -370.38), SIMDE_FLOAT32_C(  -925.54), SIMDE_FLOAT32_C(   493.42) },
      { SIMDE_FLOAT32_C(   858.65), SIMDE_FLOAT32_C(   423.83), SIMDE_FLOAT32_C(   -94.50), SIMDE_FLOAT32_C(  -574.37),
        SIMDE_FLOAT32_C(   214.07), SIMDE_FLOAT32_C(   859.61), SIMDE_FLOAT32_C(  -356.61), SIMDE_FLOAT32_C(  -324.35),
        SIMDE_FLOAT32_C(   139.27), SIMDE_FLOAT32_C(   364.36), SIMDE_FLOAT32_C(   956.86), SIMDE_FLOAT32_C(   326.23),
        SIMDE_FLOAT32_C(   766.72), SIMDE_FLOAT32_C(   611.33), SIMDE_FLOAT32_C(  -605.57), SIMDE_FLOAT32_C(   380.08) },
       INT32_C(           9),
      UINT16_C(32299) },
    { { SIMDE_FLOAT32_C(  -972.25), SIMDE_FLOAT32_C(  -981.14), SIMDE_FLOAT32_C(   443.06), SIMDE_FLOAT32_C(   556.82),
        SIMDE_FLOAT32_C(  -573.23), SIMDE_FLOAT32_C(  -663.28), SIMDE_FLOAT32_C(  -720.43), SIMDE_FLOAT32_C(   658.42),
        SIMDE_FLOAT32_C(   545.89), SIMDE_FLOAT32_C(  -677.14), SIMDE_FLOAT32_C(  -821.57), SIMDE_FLOAT32_C(   594.83),
        SIMDE_FLOAT32_C(   -47.52), SIMDE_FLOAT32_C(  -747.12), SIMDE_FLOAT32_C(    88.25), SIMDE_FLOAT32_C(  -188.87) },
      { SIMDE_FLOAT32_C(   676.71), SIMDE_FLOAT32_C(   993.75), SIMDE_FLOAT32_C(   236.76), SIMDE_FLOAT32_C(  -109.21),
        SIMDE_FLOAT32_C(   853.36), SIMDE_FLOAT32_C(   880.15), SIMDE_FLOAT32_C(   566.44), SIMDE_FLOAT32_C(    -7.37),
        SIMDE_FLOAT32_C(   244.51), SIMDE_FLOAT32_C(   523.30), SIMDE_FLOAT32_C(  -681.15), SIMDE_FLOAT32_C(    11.24),
        SIMDE_FLOAT32_C(   134.63), SIMDE_FLOAT32_C(  -286.72), SIMDE_FLOAT32_C(  -608.68), SIMDE_FLOAT32_C(   162.38) },
       INT32_C(          10),
      UINT16_C(46707) },
    { { SIMDE_FLOAT32_C(  -267.86), SIMDE_FLOAT32_C(   834.38), SIMDE_FLOAT32_C(  -280.80), SIMDE_FLOAT32_C(   158.91),
        SIMDE_FLOAT32_C(  -828.90), SIMDE_FLOAT32_C(    -1.23), SIMDE_FLOAT32_C(  -182.67), SIMDE_FLOAT32_C(   716.99),
        SIMDE_FLOAT32_C(   321.63), SIMDE_FLOAT32_C(    -4.24), SIMDE_FLOAT32_C(   311.82), SIMDE_FLOAT32_C(  -725.89),
        SIMDE_FLOAT32_C(   248.64), SIMDE_FLOAT32_C(  -599.93), SIMDE_FLOAT32_C(    85.24), SIMDE_FLOAT32_C(   -74.64) },
      { SIMDE_FLOAT32_C(  -606.18), SIMDE_FLOAT32_C(  -677.99), SIMDE_FLOAT32_C(   816.14), SIMDE_FLOAT32_C(  -752.82),
        SIMDE_FLOAT32_C(  -797.84), SIMDE_FLOAT32_C(   382.58), SIMDE_FLOAT32_C(   239.80), SIMDE_FLOAT32_C(   446.68),
        SIMDE_FLOAT32_C(   -94.12), SIMDE_FLOAT32_C(   558.66), SIMDE_FLOAT32_C(  -542.09), SIMDE_FLOAT32_C(  -959.49),
        SIMDE_FLOAT32_C(  -728.06), SIMDE_FLOAT32_C(  -150.77), SIMDE_FLOAT32_C(   202.89), SIMDE_FLOAT32_C(     4.08) },
       INT32_C(          11),
      UINT16_C(    0) },
    { { SIMDE_FLOAT32_C(  -316.38), SIMDE_FLOAT32_C(   922.09), SIMDE_FLOAT32_C(  -837.02), SIMDE_FLOAT32_C(  -145.29),
        SIMDE_FLOAT32_C(   -79.14), SIMDE_FLOAT32_C(   -19.68), SIMDE_FLOAT32_C(  -428.29), SIMDE_FLOAT32_C(  -757.51),
        SIMDE_FLOAT32_C(   976.07), SIMDE_FLOAT32_C(   883.53), SIMDE_FLOAT32_C(  -483.40), SIMDE_FLOAT32_C(   224.72),
        SIMDE_FLOAT32_C(  -716.41), SIMDE_FLOAT32_C(   601.84), SIMDE_FLOAT32_C(  -849.93), SIMDE_FLOAT32_C(  -322.59) },
      { SIMDE_FLOAT32_C(   923.85), SIMDE_FLOAT32_C(   966.22), SIMDE_FLOAT32_C(   -75.41), SIMDE_FLOAT32_C(  -873.98),
        SIMDE_FLOAT32_C(   348.80), SIMDE_FLOAT32_C(  -835.61), SIMDE_FLOAT32_C(   572.69), SIMDE_FLOAT32_C(  -745.32),
        SIMDE_FLOAT32_C(   723.05), SIMDE_FLOAT32_C(  -969.40), SIMDE_FLOAT32_C(  -704.81), SIMDE_FLOAT32_C(   994.98),
        SIMDE_FLOAT32_C(  -120.16), SIMDE_FLOAT32_C(   498.08), SIMDE_FLOAT32_C(    -0.94), SIMDE_FLOAT32_C(   563.45) },
       INT32_C(          12),
           UINT16_MAX },
    { { SIMDE_FLOAT32_C(   420.16), SIMDE_FLOAT32_C(   162.04), SIMDE_FLOAT32_C(  -581.83), SIMDE_FLOAT32_C(  -658.98),
        SIMDE_FLOAT32_C(  -857.64), SIMDE_FLOAT32_C(   -10.13), SIMDE_FLOAT32_C(  -416.49), SIMDE_FLOAT32_C(  -881.56),
        SIMDE_FLOAT32_C(  -126.60), SIMDE_FLOAT32_C(   100.10), SIMDE_FLOAT32_C(   343.15), SIMDE_FLOAT32_C(   156.99),
        SIMDE_FLOAT32_C(  -298.05), SIMDE_FLOAT32_C(   493.23), SIMDE_FLOAT32_C(   834.41), SIMDE_FLOAT32_C(  -374.20) },
      { SIMDE_FLOAT32_C(   459.44), SIMDE_FLOAT32_C(  -241.01), SIMDE_FLOAT32_C(  -248.18), SIMDE_FLOAT32_C(  -191.76),
        SIMDE_FLOAT32_C(   -76.61), SIMDE_FLOAT32_C(  -675.49), SIMDE_FLOAT32_C(    62.92), SIMDE_FLOAT32_C(  -353.57),
        SIMDE_FLOAT32_C(  -644.89), SIMDE_FLOAT32_C(   358.11), SIMDE_FLOAT32_C(  -358.58), SIMDE_FLOAT32_C(   234.95),
        SIMDE_FLOAT32_C(  -143.82), SIMDE_FLOAT32_C(   640.48), SIMDE_FLOAT32_C(  -201.60), SIMDE_FLOAT32_C(  -723.66) },
       INT32_C(          13),
      UINT16_C(50466) },
    { { SIMDE_FLOAT32_C(  -197.48), SIMDE_FLOAT32_C(   216.57), SIMDE_FLOAT32_C(  -382.64), SIMDE_FLOAT32_C(   -55.12),
        SIMDE_FLOAT32_C(  -793.56), SIMDE_FLOAT32_C(   200.87), SIMDE_FLOAT32_C(    63.32), SIMDE_FLOAT32_C(    79.84),
        SIMDE_FLOAT32_C(  -699.03), SIMDE_FLOAT32_C(  -593.53), SIMDE_FLOAT32_C(  -763.16), SIMDE_FLOAT32_C(     2.92),
        SIMDE_FLOAT32_C(   899.70), SIMDE_FLOAT32_C(  -928.76), SIMDE_FLOAT32_C(   628.72), SIMDE_FLOAT32_C(   359.14) },
      { SIMDE_FLOAT32_C(  -169.76), SIMDE_FLOAT32_C(  -619.46), SIMDE_FLOAT32_C(  -832.62), SIMDE_FLOAT32_C(   753.63),
        SIMDE_FLOAT32_C(  -294.96), SIMDE_FLOAT32_C(   230.30), SIMDE_FLOAT32_C(  -599.94), SIMDE_FLOAT32_C(    60.15),
        SIMDE_FLOAT32_C(  -411.60), SIMDE_FLOAT32_C(    41.48), SIMDE_FLOAT32_C(  -704.90), SIMDE_FLOAT32_C(   444.59),
        SIMDE_FLOAT32_C(  -318.04), SIMDE_FLOAT32_C(    93.51), SIMDE_FLOAT32_C(   720.93), SIMDE_FLOAT32_C(   484.48) },
       INT32_C(          14),
      UINT16_C( 4294) },
    { { SIMDE_FLOAT32_C(  -689.92), SIMDE_FLOAT32_C(  -661.71), SIMDE_FLOAT32_C(  -570.64), SIMDE_FLOAT32_C(  -483.48),
        SIMDE_FLOAT32_C(   539.16), SIMDE_FLOAT32_C(   492.68), SIMDE_FLOAT32_C(   596.36), SIMDE_FLOAT32_C(   840.13),
        SIMDE_FLOAT32_C(   899.15), SIMDE_FLOAT32_C(   833.20), SIMDE_FLOAT32_C(  -156.95), SIMDE_FLOAT32_C(   798.85),
        SIMDE_FLOAT32_C(   904.44), SIMDE_FLOAT32_C(  -528.23), SIMDE_FLOAT32_C(   157.99), SIMDE_FLOAT32_C(  -265.32) },
      { SIMDE_FLOAT32_C(  -147.69), SIMDE_FLOAT32_C(   325.37), SIMDE_FLOAT32_C(  -511.69), SIMDE_FLOAT32_C(   557.35),
        SIMDE_FLOAT32_C(  -444.33), SIMDE_FLOAT32_C(  -111.63), SIMDE_FLOAT32_C(  -382.50), SIMDE_FLOAT32_C(   144.07),
        SIMDE_FLOAT32_C(   929.85), SIMDE_FLOAT32_C(   -87.39), SIMDE_FLOAT32_C(  -411.34), SIMDE_FLOAT32_C(  -388.19),
        SIMDE_FLOAT32_C(  -993.88), SIMDE_FLOAT32_C(  -690.41), SIMDE_FLOAT32_C(  -903.71), SIMDE_FLOAT32_C(  -683.81) },
       INT32_C(          15),
           UINT16_MAX },
    { { SIMDE_FLOAT32_C(  -352.12), SIMDE_FLOAT32_C(  -474.35), SIMDE_FLOAT32_C(  -167.29), SIMDE_FLOAT32_C(  -812.96),
        SIMDE_FLOAT32_C(  -981.67), SIMDE_FLOAT32_C(  -570.92), SIMDE_FLOAT32_C(  -972.83), SIMDE_FLOAT32_C(   917.49),
        SIMDE_FLOAT32_C(  -737.72), SIMDE_FLOAT32_C(  -129.78), SIMDE_FLOAT32_C(   716.34), SIMDE_FLOAT32_C(  -833.28),
        SIMDE_FLOAT32_C(   341.99), SIMDE_FLOAT32_C(  -125.67), SIMDE_FLOAT32_C(   -98.59), SIMDE_FLOAT32_C(  -805.70) },
      { SIMDE_FLOAT32_C(  -800.30), SIMDE_FLOAT32_C(   389.72), SIMDE_FLOAT32_C(   751.65), SIMDE_FLOAT32_C(  -244.63),
        SIMDE_FLOAT32_C(  -721.91), SIMDE_FLOAT32_C(  -630.84), SIMDE_FLOAT32_C(   899.44), SIMDE_FLOAT32_C(  -792.06),
        SIMDE_FLOAT32_C(   281.76), SIMDE_FLOAT32_C(  -511.90), SIMDE_FLOAT32_C(  -180.26), SIMDE_FLOAT32_C(   287.88),
        SIMDE_FLOAT32_C(  -202.31), SIMDE_FLOAT32_C(   -83.97), SIMDE_FLOAT32_C(   604.08), SIMDE_FLOAT32_C(   445.56) },
       INT32_C(          16),
      UINT16_C(    0) },
    { { SIMDE_FLOAT32_C(   441.68), SIMDE_FLOAT32_C(  -563.21), SIMDE_FLOAT32_C(   632.60), SIMDE_FLOAT32_C(   460.01),
        SIMDE_FLOAT32_C(  -134.13), SIMDE_FLOAT32_C(   659.78), SIMDE_FLOAT32_C(   377.50), SIMDE_FLOAT32_C(   128.15),
        SIMDE_FLOAT32_C(  -470.00), SIMDE_FLOAT32_C(    93.84), SIMDE_FLOAT32_C(   294.87), SIMDE_FLOAT32_C(   871.99),
        SIMDE_FLOAT32_C(   968.16), SIMDE_FLOAT32_C(  -803.72), SIMDE_FLOAT32_C(  -933.71), SIMDE_FLOAT32_C(  -832.14) },
      { SIMDE_FLOAT32_C(   586.00), SIMDE_FLOAT32_C(   817.94), SIMDE_FLOAT32_C(   -76.77), SIMDE_FLOAT32_C(   864.08),
        SIMDE_FLOAT32_C(  -812.90), SIMDE_FLOAT32_C(  -177.33), SIMDE_FLOAT32_C(  -927.98), SIMDE_FLOAT32_C(   468.86),
        SIMDE_FLOAT32_C(   310.77), SIMDE_FLOAT32_C(  -108.24), SIMDE_FLOAT32_C(  -243.26), SIMDE_FLOAT32_C(  -891.55),
        SIMDE_FLOAT32_C(   807.79), SIMDE_FLOAT32_C(  -639.18), SIMDE_FLOAT32_C(   554.02), SIMDE_FLOAT32_C(   249.47) },
       INT32_C(          17),
      UINT16_C(57739) },
    { { SIMDE_FLOAT32_C(  -202.39), SIMDE_FLOAT32_C(   186.62), SIMDE_FLOAT32_C(  -290.52), SIMDE_FLOAT32_C(   663.47),
        SIMDE_FLOAT32_C(  -153.60), SIMDE_FLOAT32_C(  -913.02), SIMDE_FLOAT32_C(  -208.38), SIMDE_FLOAT32_C(   376.40),
        SIMDE_FLOAT32_C(   180.82), SIMDE_FLOAT32_C(  -913.51), SIMDE_FLOAT32_C(   248.39), SIMDE_FLOAT32_C(   148.98),
        SIMDE_FLOAT32_C(  -717.23), SIMDE_FLOAT32_C(   314.68), SIMDE_FLOAT32_C(   316.85), SIMDE_FLOAT32_C(   868.77) },
      { SIMDE_FLOAT32_C(   132.62), SIMDE_FLOAT32_C(  -759.92), SIMDE_FLOAT32_C(   732.85), SIMDE_FLOAT32_C(   319.72),
        SIMDE_FLOAT32_C(    62.75), SIMDE_FLOAT32_C(   804.87), SIMDE_FLOAT32_C(  -211.42), SIMDE_FLOAT32_C(  -626.48),
        SIMDE_FLOAT32_C(  -303.37), SIMDE_FLOAT32_C(   545.32), SIMDE_FLOAT32_C(  -518.02), SIMDE_FLOAT32_C(  -495.58),
        SIMDE_FLOAT32_C(   906.14), SIMDE_FLOAT32_C(  -964.00), SIMDE_FLOAT32_C(   753.90), SIMDE_FLOAT32_C(  -296.25) },
       INT32_C(          18),
      UINT16_C(21045) },
    { { SIMDE_FLOAT32_C(   222.62), SIMDE_FLOAT32_C(  -536.62), SIMDE_FLOAT32_C(  -632.78), SIMDE_FLOAT32_C(  -930.98),
        SIMDE_FLOAT32_C(  -449.64), SIMDE_FLOAT32_C(   158.84), SIMDE_FLOAT32_C(   445.42), SIMDE_FLOAT32_C(   731.18),
        SIMDE_FLOAT32_C(   245.34), SIMDE_FLOAT32_C(  -306.20), SIMDE_FLOAT32_C(  -119.84), SIMDE_FLOAT32_C(   528.11),
        SIMDE_FLOAT32_C(  -991.52), SIMDE_FLOAT32_C(  -802.99), SIMDE_FLOAT32_C(   396.88), SIMDE_FLOAT32_C(   141.10) },
      { SIMDE_FLOAT32_C(  -562.91), SIMDE_FLOAT32_C(   129.73), SIMDE_FLOAT32_C(  -539.18), SIMDE_FLOAT32_C(   499.84),
        SIMDE_FLOAT32_C(   -65.40), SIMDE_FLOAT32_C(   249.40), SIMDE_FLOAT32_C(   873.36), SIMDE_FLOAT32_C(   631.23),
        SIMDE_FLOAT32_C(  -205.28), SIMDE_FLOAT32_C(  -644.66), SIMDE_FLOAT32_C(  -864.35), SIMDE_FLOAT32_C(  -299.14),
        SIMDE_FLOAT32_C(  -608.67), SIMDE_FLOAT32_C(   889.55), SIMDE_FLOAT32_C(   404.61), SIMDE_FLOAT32_C(   613.95) },
       INT32_C(          19),
      UINT16_C(    0) },
    { { SIMDE_FLOAT32_C(  -647.07), SIMDE_FLOAT32_C(   771.83), SIMDE_FLOAT32_C(   682.97), SIMDE_FLOAT32_C(   -96.71),
        SIMDE_FLOAT32_C(   -69.32), SIMDE_FLOAT32_C(   128.39), SIMDE_FLOAT32_C(  -365.53), SIMDE_FLOAT32_C(  -823.99),
        SIMDE_FLOAT32_C(   822.19), SIMDE_FLOAT32_C(   514.63), SIMDE_FLOAT32_C(   704.12), SIMDE_FLOAT32_C(   830.67),
        SIMDE_FLOAT32_C(   711.64), SIMDE_FLOAT32_C(   101.00), SIMDE_FLOAT32_C(   -28.22), SIMDE_FLOAT32_C(  -851.27) },
      { SIMDE_FLOAT32_C(  -769.27), SIMDE_FLOAT32_C(   432.60), SIMDE_FLOAT32_C(   648.57), SIMDE_FLOAT32_C(   165.32),
        SIMDE_FLOAT32_C(  -318.00), SIMDE_FLOAT32_C(   521.93), SIMDE_FLOAT32_C(  -203.45), SIMDE_FLOAT32_C(   476.73),
        SIMDE_FLOAT32_C(   877.27), SIMDE_FLOAT32_C(   -67.79), SIMDE_FLOAT32_C(  -822.41), SIMDE_FLOAT32_C(  -731.40),
        SIMDE_FLOAT32_C(  -178.24), SIMDE_FLOAT32_C(   582.20), SIMDE_FLOAT32_C(   882.55), SIMDE_FLOAT32_C(   174.68) },
       INT32_C(          20),
           UINT16_MAX },
    { { SIMDE_FLOAT32_C(   354.04), SIMDE_FLOAT32_C(   565.52), SIMDE_FLOAT32_C(  -922.02), SIMDE_FLOAT32_C(  -715.29),
        SIMDE_FLOAT32_C(  -306.09), SIMDE_FLOAT32_C(  -287.55), SIMDE_FLOAT32_C(  -539.27), SIMDE_FLOAT32_C(  -483.90),
        SIMDE_FLOAT32_C(  -772.92), SIMDE_FLOAT32_C(  -835.15), SIMDE_FLOAT32_C(  -653.23), SIMDE_FLOAT32_C(   938.73),
        SIMDE_FLOAT32_C(   265.85), SIMDE_FLOAT32_C(   318.55), SIMDE_FLOAT32_C(  -912.54), SIMDE_FLOAT32_C(   496.58) },
      { SIMDE_FLOAT32_C(  -248.85), SIMDE_FLOAT32_C(   736.03), SIMDE_FLOAT32_C(  -338.10), SIMDE_FLOAT32_C(   433.16),
        SIMDE_FLOAT32_C(   257.97), SIMDE_FLOAT32_C(   458.45), SIMDE_FLOAT32_C(   -90.12), SIMDE_FLOAT32_C(   135.24),
        SIMDE_FLOAT32_C(  -609.34), SIMDE_FLOAT32_C(    87.47), SIMDE_FLOAT32_C(   403.84), SIMDE_FLOAT32_C(   212.42),
        SIMDE_FLOAT32_C(  -330.33), SIMDE_FLOAT32_C(   286.39), SIMDE_FLOAT32_C(  -612.90), SIMDE_FLOAT32_C(  -976.29) },
       INT32_C(          21),
      UINT16_C(47105) },
    { { SIMDE_FLOAT32_C(  -148.09), SIMDE_FLOAT32_C(  -534.92), SIMDE_FLOAT32_C(  -691.57), SIMDE_FLOAT32_C(   545.82),
        SIMDE_FLOAT32_C(   177.53), SIMDE_FLOAT32_C(  -230.85), SIMDE_FLOAT32_C(  -938.08), SIMDE_FLOAT32_C(   404.61),
        SIMDE_FLOAT32_C(   -65.99), SIMDE_FLOAT32_C(  -591.31), SIMDE_FLOAT32_C(   343.34), SIMDE_FLOAT32_C(  -800.14),
        SIMDE_FLOAT32_C(   727.24), SIMDE_FLOAT32_C(   430.80), SIMDE_FLOAT32_C(   696.43), SIMDE_FLOAT32_C(  -521.61) },
      { SIMDE_FLOAT32_C(   166.83), SIMDE_FLOAT32_C(  -641.66), SIMDE_FLOAT32_C(   911.55), SIMDE_FLOAT32_C(  -575.20),
        SIMDE_FLOAT32_C(   816.79), SIMDE_FLOAT32_C(  -178.57), SIMDE_FLOAT32_C(   560.03), SIMDE_FLOAT32_C(  -792.55),
        SIMDE_FLOAT32_C(   908.90), SIMDE_FLOAT32_C(   -36.13), SIMDE_FLOAT32_C(   419.87), SIMDE_FLOAT32_C(  -421.42),
        SIMDE_FLOAT32_C(  -749.74), SIMDE_FLOAT32_C(   806.97), SIMDE_FLOAT32_C(  -397.71), SIMDE_FLOAT32_C(   102.17) },
       INT32_C(          22),
      UINT16_C(20618) },
    { { SIMDE_FLOAT32_C(  -727.95), SIMDE_FLOAT32_C(   -89.29), SIMDE_FLOAT32_C(  -352.01), SIMDE_FLOAT32_C(   449.58),
        SIMDE_FLOAT32_C(   679.87), SIMDE_FLOAT32_C(  -290.09), SIMDE_FLOAT32_C(  -145.82), SIMDE_FLOAT32_C(  -386.13),
        SIMDE_FLOAT32_C(   118.59), SIMDE_FLOAT32_C(  -802.48), SIMDE_FLOAT32_C(  -186.27), SIMDE_FLOAT32_C(  -154.16),
        SIMDE_FLOAT32_C(   628.32), SIMDE_FLOAT32_C(  -489.83), SIMDE_FLOAT32_C(   324.23), SIMDE_FLOAT32_C(  -204.85) },
      { SIMDE_FLOAT32_C(  -131.50), SIMDE_FLOAT32_C(   235.78), SIMDE_FLOAT32_C(   219.94), SIMDE_FLOAT32_C(  -314.71),
        SIMDE_FLOAT32_C(  -942.79), SIMDE_FLOAT32_C(  -220.02), SIMDE_FLOAT32_C(  -107.26), SIMDE_FLOAT32_C(   966.11),
        SIMDE_FLOAT32_C(   743.85), SIMDE_FLOAT32_C(  -687.39), SIMDE_FLOAT32_C(  -455.31), SIMDE_FLOAT32_C(   994.11),
        SIMDE_FLOAT32_C(  -880.42), SIMDE_FLOAT32_C(   146.98), SIMDE_FLOAT32_C(    96.28), SIMDE_FLOAT32_C(  -608.37) },
       INT32_C(          23),
           UINT16_MAX },
    { { SIMDE_FLOAT32_C(  -942.31), SIMDE_FLOAT32_C(   744.27), SIMDE_FLOAT32_C(   841.21), SIMDE_FLOAT32_C(   737.56),
        SIMDE_FLOAT32_C(  -545.82), SIMDE_FLOAT32_C(  -304.61), SIMDE_FLOAT32_C(  -648.57), SIMDE_FLOAT32_C(   572.77),
        SIMDE_FLOAT32_C(  -107.09), SIMDE_FLOAT32_C(   165.16), SIMDE_FLOAT32_C(  -581.39), SIMDE_FLOAT32_C(  -478.77),
        SIMDE_FLOAT32_C(   675.33), SIMDE_FLOAT32_C(   742.84), SIMDE_FLOAT32_C(   316.38), SIMDE_FLOAT32_C(  -456.17) },
      { SIMDE_FLOAT32_C(   -21.39), SIMDE_FLOAT32_C(  -463.68), SIMDE_FLOAT32_C(   229.13), SIMDE_FLOAT32_C(    35.82),
        SIMDE_FLOAT32_C(   316.30), SIMDE_FLOAT32_C(  -878.13), SIMDE_FLOAT32_C(     1.93), SIMDE_FLOAT32_C(    60.15),
        SIMDE_FLOAT32_C(  -565.52), SIMDE_FLOAT32_C(   546.62), SIMDE_FLOAT32_C(    54.26), SIMDE_FLOAT32_C(  -445.94),
        SIMDE_FLOAT32_C(  -306.41), SIMDE_FLOAT32_C(  -849.45), SIMDE_FLOAT32_C(   -54.31), SIMDE_FLOAT32_C(  -248.72) },
       INT32_C(          24),
      UINT16_C(    0) },
    { { SIMDE_FLOAT32_C(   894.82), SIMDE_FLOAT32_C(  -213.10), SIMDE_FLOAT32_C(  -511.16), SIMDE_FLOAT32_C(  -651.01),
        SIMDE_FLOAT32_C(   482.29), SIMDE_FLOAT32_C(  -159.73), SIMDE_FLOAT32_C(   921.76), SIMDE_FLOAT32_C(  -624.79),
        SIMDE_FLOAT32_C(  -994.57), SIMDE_FLOAT32_C(  -659.63), SIMDE_FLOAT32_C(  -103.56), SIMDE_FLOAT32_C(   680.76),
        SIMDE_FLOAT32_C(  -916.80), SIMDE_FLOAT32_C(  -787.19), SIMDE_FLOAT32_C(  -775.40), SIMDE_FLOAT32_C(    61.82) },
      { SIMDE_FLOAT32_C(  -250.87), SIMDE_FLOAT32_C(   453.72), SIMDE_FLOAT32_C(  -902.36), SIMDE_FLOAT32_C(  -934.56),
        SIMDE_FLOAT32_C(   575.59), SIMDE_FLOAT32_C(    99.57), SIMDE_FLOAT32_C(   125.59), SIMDE_FLOAT32_C(  -989.93),
        SIMDE_FLOAT32_C(  -353.81), SIMDE_FLOAT32_C(  -820.15), SIMDE_FLOAT32_C(  -435.87), SIMDE_FLOAT32_C(   339.78),
        SIMDE_FLOAT32_C(  -669.60), SIMDE_FLOAT32_C(   509.82), SIMDE_FLOAT32_C(  -908.94), SIMDE_FLOAT32_C(  -774.79) },
       INT32_C(          25),
      UINT16_C(12594) },
    { { SIMDE_FLOAT32_C(  -703.28), SIMDE_FLOAT32_C(  -420.10), SIMDE_FLOAT32_C(  -425.79), SIMDE_FLOAT32_C(   779.02),
        SIMDE_FLOAT32_C(   420.16), SIMDE_FLOAT32_C(  -504.03), SIMDE_FLOAT32_C(  -845.78), SIMDE_FLOAT32_C(   425.59),
        SIMDE_FLOAT32_C(  -163.66), SIMDE_FLOAT32_C(    50.66), SIMDE_FLOAT32_C(   106.36), SIMDE_FLOAT32_C(   -80.46),
        SIMDE_FLOAT32_C(   263.47), SIMDE_FLOAT32_C(   330.95), SIMDE_FLOAT32_C(   981.36), SIMDE_FLOAT32_C(  -987.39) },
      { SIMDE_FLOAT32_C(  -215.33), SIMDE_FLOAT32_C(  -921.00), SIMDE_FLOAT32_C(  -921.96), SIMDE_FLOAT32_C(  -639.74),
        SIMDE_FLOAT32_C(   178.56), SIMDE_FLOAT32_C(   203.63), SIMDE_FLOAT32_C(  -629.67), SIMDE_FLOAT32_C(   824.75),
        SIMDE_FLOAT32_C(   383.48), SIMDE_FLOAT32_C(   -65.54), SIMDE_FLOAT32_C(   164.53), SIMDE_FLOAT32_C(   713.88),
        SIMDE_FLOAT32_C(  -555.72), SIMDE_FLOAT32_C(   255.59), SIMDE_FLOAT32_C(   939.10), SIMDE_FLOAT32_C(  -258.99) },
       INT32_C(          26),
      UINT16_C(36321) },
    { { SIMDE_FLOAT32_C(   835.48), SIMDE_FLOAT32_C(  -486.70), SIMDE_FLOAT32_C(  -479.98), SIMDE_FLOAT32_C(   255.65),
        SIMDE_FLOAT32_C(     9.27), SIMDE_FLOAT32_C(  -325.76), SIMDE_FLOAT32_C(  -318.76), SIMDE_FLOAT32_C(   845.61),
        SIMDE_FLOAT32_C(   724.90), SIMDE_FLOAT32_C(   787.60), SIMDE_FLOAT32_C(  -234.85), SIMDE_FLOAT32_C(   -11.62),
        SIMDE_FLOAT32_C(   118.55), SIMDE_FLOAT32_C(  -253.49), SIMDE_FLOAT32_C(     0.98), SIMDE_FLOAT32_C(   903.23) },
      { SIMDE_FLOAT32_C(  -174.49), SIMDE_FLOAT32_C(    79.03), SIMDE_FLOAT32_C(  -736.51), SIMDE_FLOAT32_C(  -995.93),
        SIMDE_FLOAT32_C(  -717.34), SIMDE_FLOAT32_C(  -366.18), SIMDE_FLOAT32_C(   828.82), SIMDE_FLOAT32_C(   666.15),
        SIMDE_FLOAT32_C(   568.29), SIMDE_FLOAT32_C(    -6.65), SIMDE_FLOAT32_C(   380.03), SIMDE_FLOAT32_C(  -987.43),
        SIMDE_FLOAT32_C(  -751.06), SIMDE_FLOAT32_C(   319.13), SIMDE_FLOAT32_C(  -246.43), SIMDE_FLOAT32_C(  -915.58) },
       INT32_C(          27),
      UINT16_C(    0) },
    { { SIMDE_FLOAT32_C(   832.43), SIMDE_FLOAT32_C(   273.60), SIMDE_FLOAT32_C(   340.07), SIMDE_FLOAT32_C(  -158.30),
        SIMDE_FLOAT32_C(   947.84), SIMDE_FLOAT32_C(  -978.69), SIMDE_FLOAT32_C(  -312.69), SIMDE_FLOAT32_C(   672.74),
        SIMDE_FLOAT32_C(   808.91), SIMDE_FLOAT32_C(   452.46), SIMDE_FLOAT32_C(  -338.88), SIMDE_FLOAT32_C(   -72.53),
        SIMDE_FLOAT32_C(  -801.03), SIMDE_FLOAT32_C(   662.10), SIMDE_FLOAT32_C(  -169.31), SIMDE_FLOAT32_C(    24.48) },
      { SIMDE_FLOAT32_C(  -258.87), SIMDE_FLOAT32_C(    94.18), SIMDE_FLOAT32_C(    28.55), SIMDE_FLOAT32_C(    23.79),
        SIMDE_FLOAT32_C(   728.01), SIMDE_FLOAT32_C(  -142.63), SIMDE_FLOAT32_C(  -310.06), SIMDE_FLOAT32_C(   296.29),
        SIMDE_FLOAT32_C(   850.72), SIMDE_FLOAT32_C(  -930.03), SIMDE_FLOAT32_C(   308.86), SIMDE_FLOAT32_C(  -900.34),
        SIMDE_FLOAT32_C(   389.10), SIMDE_FLOAT32_C(  -937.56), SIMDE_FLOAT32_C(  -815.92), SIMDE_FLOAT32_C(   221.53) },
       INT32_C(          28),
           UINT16_MAX },
    { { SIMDE_FLOAT32_C(   336.03), SIMDE_FLOAT32_C(   524.16), SIMDE_FLOAT32_C(  -936.77), SIMDE_FLOAT32_C(   283.87),
        SIMDE_FLOAT32_C(   545.47), SIMDE_FLOAT32_C(  -249.46), SIMDE_FLOAT32_C(   -43.38), SIMDE_FLOAT32_C(   354.38),
        SIMDE_FLOAT32_C(  -797.00), SIMDE_FLOAT32_C(   617.74), SIMDE_FLOAT32_C(  -718.15), SIMDE_FLOAT32_C(  -598.03),
        SIMDE_FLOAT32_C(   279.84), SIMDE_FLOAT32_C(   112.54), SIMDE_FLOAT32_C(   426.45), SIMDE_FLOAT32_C(  -979.03) },
      { SIMDE_FLOAT32_C(  -793.27), SIMDE_FLOAT32_C(  -545.00), SIMDE_FLOAT32_C(    44.77), SIMDE_FLOAT32_C(   934.73),
        SIMDE_FLOAT32_C(   312.37), SIMDE_FLOAT32_C(   734.71), SIMDE_FLOAT32_C(   231.03), SIMDE_FLOAT32_C(   163.09),
        SIMDE_FLOAT32_C(   804.68), SIMDE_FLOAT32_C(  -460.11), SIMDE_FLOAT32_C(   262.76), SIMDE_FLOAT32_C(   193.77),
        SIMDE_FLOAT32_C(  -397.67), SIMDE_FLOAT32_C(   446.84), SIMDE_FLOAT32_C(  -584.70), SIMDE_FLOAT32_C(   938.36) },
       INT32_C(          29),
      UINT16_C(21139) },
    { { SIMDE_FLOAT32_C(   -29.00), SIMDE_FLOAT32_C(  -521.48), SIMDE_FLOAT32_C(   222.23), SIMDE_FLOAT32_C(  -483.53),
        SIMDE_FLOAT32_C(   229.06), SIMDE_FLOAT32_C(  -821.16), SIMDE_FLOAT32_C(   870.85), SIMDE_FLOAT32_C(   432.06),
        SIMDE_FLOAT32_C(   796.58), SIMDE_FLOAT32_C(  -847.30), SIMDE_FLOAT32_C(   834.03), SIMDE_FLOAT32_C(    76.42),
        SIMDE_FLOAT32_C(   265.24), SIMDE_FLOAT32_C(   260.47), SIMDE_FLOAT32_C(    97.39), SIMDE_FLOAT32_C(   471.97) },
      { SIMDE_FLOAT32_C(   715.47), SIMDE_FLOAT32_C(  -857.84), SIMDE_FLOAT32_C(   406.70), SIMDE_FLOAT32_C(    27.84),
        SIMDE_FLOAT32_C(   876.87), SIMDE_FLOAT32_C(  -362.27), SIMDE_FLOAT32_C(  -809.06), SIMDE_FLOAT32_C(   681.54),
        SIMDE_FLOAT32_C(   177.62), SIMDE_FLOAT32_C(   453.69), SIMDE_FLOAT32_C(  -124.69), SIMDE_FLOAT32_C(   779.94),
        SIMDE_FLOAT32_C(   -99.47), SIMDE_FLOAT32_C(   290.61), SIMDE_FLOAT32_C(   718.30), SIMDE_FLOAT32_C(   871.53) },
       INT32_C(          30),
      UINT16_C( 5442) },
    { { SIMDE_FLOAT32_C(   769.14), SIMDE_FLOAT32_C(   -59.47), SIMDE_FLOAT32_C(  -612.01), SIMDE_FLOAT32_C(    -1.80),
        SIMDE_FLOAT32_C(   119.37), SIMDE_FLOAT32_C(  -741.16), SIMDE_FLOAT32_C(  -569.75), SIMDE_FLOAT32_C(   -84.05),
        SIMDE_FLOAT32_C(  -588.46), SIMDE_FLOAT32_C(  -735.72), SIMDE_FLOAT32_C(   992.37), SIMDE_FLOAT32_C(   676.78),
        SIMDE_FLOAT32_C(   524.75), SIMDE_FLOAT32_C(    89.76), SIMDE_FLOAT32_C(   148.75), SIMDE_FLOAT32_C(   240.22) },
      { SIMDE_FLOAT32_C(   231.92), SIMDE_FLOAT32_C(  -444.55), SIMDE_FLOAT32_C(  -731.94), SIMDE_FLOAT32_C(   108.79),
        SIMDE_FLOAT32_C(   193.18), SIMDE_FLOAT32_C(  -541.00), SIMDE_FLOAT32_C(  -209.67), SIMDE_FLOAT32_C(  -629.20),
        SIMDE_FLOAT32_C(   912.69), SIMDE_FLOAT32_C(   665.64), SIMDE_FLOAT32_C(  -849.26), SIMDE_FLOAT32_C(  -186.78),
        SIMDE_FLOAT32_C(   -43.74), SIMDE_FLOAT32_C(   869.04), SIMDE_FLOAT32_C(  -315.25), SIMDE_FLOAT32_C(  -274.61) },
       INT32_C(          31),
           UINT16_MAX }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m512 b = simde_mm512_loadu_ps(test_vec[i].b);
    simde__mmask16 r = simde_mm512_cmp_ps_mask(a, b, test_vec[i].imm8);
    simde_assert_mmask16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cmp_pd_mask (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 a[8];
    const simde_float64 b[8];
    const int imm8;
    const simde__mmask8 r;
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(  -889.13), SIMDE_FLOAT64_C(   346.35), SIMDE_FLOAT64_C(  -842.69), SIMDE_FLOAT64_C(   879.16),
        SIMDE_FLOAT64_C(    37.28), SIMDE_FLOAT64_C(   607.79), SIMDE_FLOAT64_C(  -858.34), SIMDE_FLOAT64_C(  -122.77) },
      { SIMDE_FLOAT64_C(   597.22), SIMDE_FLOAT64_C(  -446.39), SIMDE_FLOAT64_C(  -495.07), SIMDE_FLOAT64_C(  -701.44),
        SIMDE_FLOAT64_C(   913.94), SIMDE_FLOAT64_C(   514.01), SIMDE_FLOAT64_C(  -970.90), SIMDE_FLOAT64_C(    91.95) },
       INT32_C(          18),
      UINT8_C(149) },
    { { SIMDE_FLOAT64_C(  -229.78), SIMDE_FLOAT64_C(   109.42), SIMDE_FLOAT64_C(   986.52), SIMDE_FLOAT64_C(   450.97),
        SIMDE_FLOAT64_C(  -621.15), SIMDE_FLOAT64_C(   366.22), SIMDE_FLOAT64_C(  -999.97), SIMDE_FLOAT64_C(  -551.44) },
      { SIMDE_FLOAT64_C(   -43.21), SIMDE_FLOAT64_C(  -236.56), SIMDE_FLOAT64_C(    73.66), SIMDE_FLOAT64_C(    21.45),
        SIMDE_FLOAT64_C(   426.81), SIMDE_FLOAT64_C(  -684.87), SIMDE_FLOAT64_C(  -547.62), SIMDE_FLOAT64_C(   194.20) },
       INT32_C(          30),
      UINT8_C( 46) },
    { { SIMDE_FLOAT64_C(   465.94), SIMDE_FLOAT64_C(  -899.85), SIMDE_FLOAT64_C(   236.88), SIMDE_FLOAT64_C(  -744.20),
        SIMDE_FLOAT64_C(   213.84), SIMDE_FLOAT64_C(    84.61), SIMDE_FLOAT64_C(    -4.00), SIMDE_FLOAT64_C(   791.14) },
      { SIMDE_FLOAT64_C(   691.24), SIMDE_FLOAT64_C(  -392.69), SIMDE_FLOAT64_C(    37.26), SIMDE_FLOAT64_C(   209.16),
        SIMDE_FLOAT64_C(  -604.04), SIMDE_FLOAT64_C(  -124.25), SIMDE_FLOAT64_C(  -288.59), SIMDE_FLOAT64_C(  -412.86) },
       INT32_C(          21),
      UINT8_C(244) },
    { { SIMDE_FLOAT64_C(  -618.86), SIMDE_FLOAT64_C(   797.13), SIMDE_FLOAT64_C(  -583.56), SIMDE_FLOAT64_C(    46.88),
        SIMDE_FLOAT64_C(   -89.41), SIMDE_FLOAT64_C(  -683.29), SIMDE_FLOAT64_C(    20.57), SIMDE_FLOAT64_C(  -213.31) },
      { SIMDE_FLOAT64_C(   887.10), SIMDE_FLOAT64_C(  -441.79), SIMDE_FLOAT64_C(   836.33), SIMDE_FLOAT64_C(   135.59),
        SIMDE_FLOAT64_C(   918.70), SIMDE_FLOAT64_C(   512.23), SIMDE_FLOAT64_C(  -895.63), SIMDE_FLOAT64_C(  -900.96) },
       INT32_C(          31),
         UINT8_MAX },
    { { SIMDE_FLOAT64_C(  -989.35), SIMDE_FLOAT64_C(   -86.98), SIMDE_FLOAT64_C(   193.68), SIMDE_FLOAT64_C(  -742.71),
        SIMDE_FLOAT64_C(  -727.59), SIMDE_FLOAT64_C(  -646.86), SIMDE_FLOAT64_C(   183.87), SIMDE_FLOAT64_C(   287.33) },
      { SIMDE_FLOAT64_C(  -774.81), SIMDE_FLOAT64_C(  -242.40), SIMDE_FLOAT64_C(    53.99), SIMDE_FLOAT64_C(  -593.99),
        SIMDE_FLOAT64_C(   779.72), SIMDE_FLOAT64_C(   806.29), SIMDE_FLOAT64_C(  -734.86), SIMDE_FLOAT64_C(  -839.78) },
       INT32_C(           6),
      UINT8_C(198) },
    { { SIMDE_FLOAT64_C(  -467.13), SIMDE_FLOAT64_C(   942.12), SIMDE_FLOAT64_C(   248.01), SIMDE_FLOAT64_C(   325.07),
        SIMDE_FLOAT64_C(  -486.56), SIMDE_FLOAT64_C(   428.42), SIMDE_FLOAT64_C(   503.39), SIMDE_FLOAT64_C(   520.75) },
      { SIMDE_FLOAT64_C(   191.14), SIMDE_FLOAT64_C(   441.43), SIMDE_FLOAT64_C(  -872.87), SIMDE_FLOAT64_C(  -283.89),
        SIMDE_FLOAT64_C(   651.45), SIMDE_FLOAT64_C(   971.81), SIMDE_FLOAT64_C(  -736.72), SIMDE_FLOAT64_C(   -71.12) },
       INT32_C(          17),
      UINT8_C( 49) },
    { { SIMDE_FLOAT64_C(   768.97), SIMDE_FLOAT64_C(    83.87), SIMDE_FLOAT64_C(  -412.86), SIMDE_FLOAT64_C(   997.28),
        SIMDE_FLOAT64_C(  -659.21), SIMDE_FLOAT64_C(   650.14), SIMDE_FLOAT64_C(   927.00), SIMDE_FLOAT64_C(    40.06) },
      { SIMDE_FLOAT64_C(  -783.94), SIMDE_FLOAT64_C(   289.86), SIMDE_FLOAT64_C(  -325.98), SIMDE_FLOAT64_C(  -693.23),
        SIMDE_FLOAT64_C(   823.83), SIMDE_FLOAT64_C(    81.84), SIMDE_FLOAT64_C(  -557.12), SIMDE_FLOAT64_C(   458.20) },
       INT32_C(          28),
         UINT8_MAX },
    { { SIMDE_FLOAT64_C(   728.01), SIMDE_FLOAT64_C(  -387.72), SIMDE_FLOAT64_C(  -341.65), SIMDE_FLOAT64_C(   -84.21),
        SIMDE_FLOAT64_C(   640.32), SIMDE_FLOAT64_C(  -112.91), SIMDE_FLOAT64_C(   308.09), SIMDE_FLOAT64_C(    20.16) },
      { SIMDE_FLOAT64_C(   745.10), SIMDE_FLOAT64_C(   919.13), SIMDE_FLOAT64_C(  -195.91), SIMDE_FLOAT64_C(  -612.27),
        SIMDE_FLOAT64_C(  -399.85), SIMDE_FLOAT64_C(  -354.18), SIMDE_FLOAT64_C(  -752.13), SIMDE_FLOAT64_C(   868.12) },
       INT32_C(          29),
      UINT8_C(120) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m512d b = simde_mm512_loadu_pd(test_vec[i].b);
    simde__mmask8 r = simde_mm512_cmp_pd_mask(a, b, test_vec[i].imm8);
    simde_assert_mmask8(r, ==, test_vec[i].r);
  }

  return 0;
}

#endif /* !defined(SIMDE_NATIVE_ALIASES_TESTING */

#if !defined(SIMDE_BUG_GCC_96174)
static int
test_simde_mm512_cmplt_ps_mask (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 a[16];
    const simde_float32 b[16];
    const simde__mmask16 r;
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -679.30), SIMDE_FLOAT32_C(   966.54), SIMDE_FLOAT32_C(    -8.95), SIMDE_FLOAT32_C(   958.36),
        SIMDE_FLOAT32_C(  -725.44), SIMDE_FLOAT32_C(  -760.28), SIMDE_FLOAT32_C(   751.05), SIMDE_FLOAT32_C(   763.86),
        SIMDE_FLOAT32_C(  -137.17), SIMDE_FLOAT32_C(  -526.42), SIMDE_FLOAT32_C(  -580.26), SIMDE_FLOAT32_C(    72.73),
        SIMDE_FLOAT32_C(   848.96), SIMDE_FLOAT32_C(  -167.99), SIMDE_FLOAT32_C(    95.30), SIMDE_FLOAT32_C(   277.78) },
      { SIMDE_FLOAT32_C(   425.87), SIMDE_FLOAT32_C(  -693.81), SIMDE_FLOAT32_C(   225.64), SIMDE_FLOAT32_C(  -374.81),
        SIMDE_FLOAT32_C(  -490.07), SIMDE_FLOAT32_C(    62.38), SIMDE_FLOAT32_C(   630.88), SIMDE_FLOAT32_C(   308.80),
        SIMDE_FLOAT32_C(  -539.84), SIMDE_FLOAT32_C(  -683.39), SIMDE_FLOAT32_C(  -735.32), SIMDE_FLOAT32_C(  -999.26),
        SIMDE_FLOAT32_C(   848.96), SIMDE_FLOAT32_C(   579.92), SIMDE_FLOAT32_C(   118.33), SIMDE_FLOAT32_C(  -830.34) },
      UINT16_C(24629) },
    { { SIMDE_FLOAT32_C(  -833.10), SIMDE_FLOAT32_C(   667.53), SIMDE_FLOAT32_C(  -768.58), SIMDE_FLOAT32_C(    27.81),
        SIMDE_FLOAT32_C(   969.40), SIMDE_FLOAT32_C(  -884.80), SIMDE_FLOAT32_C(  -758.63), SIMDE_FLOAT32_C(  -724.10),
        SIMDE_FLOAT32_C(  -716.35), SIMDE_FLOAT32_C(  -476.71), SIMDE_FLOAT32_C(   419.04), SIMDE_FLOAT32_C(   832.05),
        SIMDE_FLOAT32_C(   151.35), SIMDE_FLOAT32_C(  -175.30), SIMDE_FLOAT32_C(    66.61), SIMDE_FLOAT32_C(   351.20) },
      { SIMDE_FLOAT32_C(   118.05), SIMDE_FLOAT32_C(  -502.75), SIMDE_FLOAT32_C(  -814.79), SIMDE_FLOAT32_C(   929.98),
        SIMDE_FLOAT32_C(   432.78), SIMDE_FLOAT32_C(  -886.46), SIMDE_FLOAT32_C(   577.10), SIMDE_FLOAT32_C(  -862.12),
        SIMDE_FLOAT32_C(   136.94), SIMDE_FLOAT32_C(   908.37), SIMDE_FLOAT32_C(  -807.53), SIMDE_FLOAT32_C(  -626.26),
        SIMDE_FLOAT32_C(    93.44), SIMDE_FLOAT32_C(   143.01), SIMDE_FLOAT32_C(   933.29), SIMDE_FLOAT32_C(   260.34) },
      UINT16_C(25417) },
    { { SIMDE_FLOAT32_C(   397.82), SIMDE_FLOAT32_C(    82.73), SIMDE_FLOAT32_C(  -728.93), SIMDE_FLOAT32_C(  -716.45),
        SIMDE_FLOAT32_C(   278.34), SIMDE_FLOAT32_C(  -422.65), SIMDE_FLOAT32_C(  -540.28), SIMDE_FLOAT32_C(   265.15),
        SIMDE_FLOAT32_C(   279.24), SIMDE_FLOAT32_C(  -171.08), SIMDE_FLOAT32_C(  -468.61), SIMDE_FLOAT32_C(   443.34),
        SIMDE_FLOAT32_C(   751.73), SIMDE_FLOAT32_C(  -744.43), SIMDE_FLOAT32_C(   566.91), SIMDE_FLOAT32_C(  -904.35) },
      { SIMDE_FLOAT32_C(  -692.80), SIMDE_FLOAT32_C(    82.73), SIMDE_FLOAT32_C(   507.25), SIMDE_FLOAT32_C(  -716.45),
        SIMDE_FLOAT32_C(  -871.32), SIMDE_FLOAT32_C(   909.91), SIMDE_FLOAT32_C(  -907.02), SIMDE_FLOAT32_C(  -102.77),
        SIMDE_FLOAT32_C(   677.37), SIMDE_FLOAT32_C(  -171.08), SIMDE_FLOAT32_C(  -468.61), SIMDE_FLOAT32_C(  -257.08),
        SIMDE_FLOAT32_C(   751.73), SIMDE_FLOAT32_C(   841.70), SIMDE_FLOAT32_C(   271.27), SIMDE_FLOAT32_C(   149.55) },
      UINT16_C(41252) },
    { { SIMDE_FLOAT32_C(  -351.59), SIMDE_FLOAT32_C(  -757.31), SIMDE_FLOAT32_C(  -739.49), SIMDE_FLOAT32_C(   354.82),
        SIMDE_FLOAT32_C(   779.77), SIMDE_FLOAT32_C(   796.84), SIMDE_FLOAT32_C(   253.65), SIMDE_FLOAT32_C(  -980.02),
        SIMDE_FLOAT32_C(  -824.56), SIMDE_FLOAT32_C(  -806.24), SIMDE_FLOAT32_C(   218.91), SIMDE_FLOAT32_C(   807.03),
        SIMDE_FLOAT32_C(  -499.44), SIMDE_FLOAT32_C(   683.75), SIMDE_FLOAT32_C(   242.90), SIMDE_FLOAT32_C(   681.31) },
      { SIMDE_FLOAT32_C(   698.06), SIMDE_FLOAT32_C(   143.17), SIMDE_FLOAT32_C(   645.90), SIMDE_FLOAT32_C(   354.82),
        SIMDE_FLOAT32_C(   561.25), SIMDE_FLOAT32_C(  -928.28), SIMDE_FLOAT32_C(   482.94), SIMDE_FLOAT32_C(    28.55),
        SIMDE_FLOAT32_C(   701.67), SIMDE_FLOAT32_C(   834.16), SIMDE_FLOAT32_C(   386.75), SIMDE_FLOAT32_C(   807.03),
        SIMDE_FLOAT32_C(   558.03), SIMDE_FLOAT32_C(  -756.03), SIMDE_FLOAT32_C(   930.12), SIMDE_FLOAT32_C(  -793.56) },
      UINT16_C(22471) },
    { { SIMDE_FLOAT32_C(   434.87), SIMDE_FLOAT32_C(  -355.05), SIMDE_FLOAT32_C(  -653.48), SIMDE_FLOAT32_C(   594.11),
        SIMDE_FLOAT32_C(   799.49), SIMDE_FLOAT32_C(   264.31), SIMDE_FLOAT32_C(    -8.19), SIMDE_FLOAT32_C(  -922.96),
        SIMDE_FLOAT32_C(   308.23), SIMDE_FLOAT32_C(  -871.48), SIMDE_FLOAT32_C(   543.66), SIMDE_FLOAT32_C(   721.18),
        SIMDE_FLOAT32_C(  -314.45), SIMDE_FLOAT32_C(   897.43), SIMDE_FLOAT32_C(   646.34), SIMDE_FLOAT32_C(  -691.19) },
      { SIMDE_FLOAT32_C(  -506.84), SIMDE_FLOAT32_C(  -355.05), SIMDE_FLOAT32_C(    70.02), SIMDE_FLOAT32_C(  -186.22),
        SIMDE_FLOAT32_C(   745.56), SIMDE_FLOAT32_C(  -329.15), SIMDE_FLOAT32_C(  -306.53), SIMDE_FLOAT32_C(  -665.08),
        SIMDE_FLOAT32_C(   -81.67), SIMDE_FLOAT32_C(   690.25), SIMDE_FLOAT32_C(  -343.01), SIMDE_FLOAT32_C(   742.59),
        SIMDE_FLOAT32_C(  -989.44), SIMDE_FLOAT32_C(   198.45), SIMDE_FLOAT32_C(   334.24), SIMDE_FLOAT32_C(   445.42) },
      UINT16_C(35460) },
    { { SIMDE_FLOAT32_C(    72.70), SIMDE_FLOAT32_C(  -926.98), SIMDE_FLOAT32_C(   386.60), SIMDE_FLOAT32_C(  -166.44),
        SIMDE_FLOAT32_C(  -372.12), SIMDE_FLOAT32_C(   156.01), SIMDE_FLOAT32_C(  -432.45), SIMDE_FLOAT32_C(  -171.34),
        SIMDE_FLOAT32_C(  -100.09), SIMDE_FLOAT32_C(   220.75), SIMDE_FLOAT32_C(  -427.23), SIMDE_FLOAT32_C(  -735.37),
        SIMDE_FLOAT32_C(   440.82), SIMDE_FLOAT32_C(  -646.62), SIMDE_FLOAT32_C(   895.12), SIMDE_FLOAT32_C(   585.45) },
      { SIMDE_FLOAT32_C(   -15.73), SIMDE_FLOAT32_C(   536.94), SIMDE_FLOAT32_C(  -374.81), SIMDE_FLOAT32_C(   158.91),
        SIMDE_FLOAT32_C(   525.00), SIMDE_FLOAT32_C(   478.37), SIMDE_FLOAT32_C(  -432.45), SIMDE_FLOAT32_C(  -483.69),
        SIMDE_FLOAT32_C(   887.57), SIMDE_FLOAT32_C(   220.75), SIMDE_FLOAT32_C(   709.30), SIMDE_FLOAT32_C(   187.04),
        SIMDE_FLOAT32_C(  -436.07), SIMDE_FLOAT32_C(   329.70), SIMDE_FLOAT32_C(    57.53), SIMDE_FLOAT32_C(   636.63) },
      UINT16_C(44346) },
    { { SIMDE_FLOAT32_C(  -715.67), SIMDE_FLOAT32_C(  -253.10), SIMDE_FLOAT32_C(   805.99), SIMDE_FLOAT32_C(   896.48),
        SIMDE_FLOAT32_C(  -683.44), SIMDE_FLOAT32_C(  -642.77), SIMDE_FLOAT32_C(  -746.45), SIMDE_FLOAT32_C(   318.24),
        SIMDE_FLOAT32_C(  -949.63), SIMDE_FLOAT32_C(  -203.63), SIMDE_FLOAT32_C(  -894.66), SIMDE_FLOAT32_C(   648.89),
        SIMDE_FLOAT32_C(   110.40), SIMDE_FLOAT32_C(   662.12), SIMDE_FLOAT32_C(   821.38), SIMDE_FLOAT32_C(   820.81) },
      { SIMDE_FLOAT32_C(   147.48), SIMDE_FLOAT32_C(   715.61), SIMDE_FLOAT32_C(  -594.01), SIMDE_FLOAT32_C(   128.99),
        SIMDE_FLOAT32_C(   847.91), SIMDE_FLOAT32_C(  -246.50), SIMDE_FLOAT32_C(  -172.62), SIMDE_FLOAT32_C(   927.56),
        SIMDE_FLOAT32_C(  -949.63), SIMDE_FLOAT32_C(  -193.40), SIMDE_FLOAT32_C(   284.28), SIMDE_FLOAT32_C(   354.14),
        SIMDE_FLOAT32_C(  -296.72), SIMDE_FLOAT32_C(   320.79), SIMDE_FLOAT32_C(   108.95), SIMDE_FLOAT32_C(   -12.38) },
      UINT16_C( 1779) },
    { { SIMDE_FLOAT32_C(   372.34), SIMDE_FLOAT32_C(   943.17), SIMDE_FLOAT32_C(  -546.38), SIMDE_FLOAT32_C(  -534.61),
        SIMDE_FLOAT32_C(  -390.69), SIMDE_FLOAT32_C(   249.11), SIMDE_FLOAT32_C(   492.46), SIMDE_FLOAT32_C(    83.28),
        SIMDE_FLOAT32_C(   -13.87), SIMDE_FLOAT32_C(   563.95), SIMDE_FLOAT32_C(    27.19), SIMDE_FLOAT32_C(    69.48),
        SIMDE_FLOAT32_C(  -499.31), SIMDE_FLOAT32_C(   588.53), SIMDE_FLOAT32_C(   881.11), SIMDE_FLOAT32_C(  -291.35) },
      { SIMDE_FLOAT32_C(   896.28), SIMDE_FLOAT32_C(  -328.16), SIMDE_FLOAT32_C(   -58.67), SIMDE_FLOAT32_C(  -222.16),
        SIMDE_FLOAT32_C(   369.25), SIMDE_FLOAT32_C(   249.11), SIMDE_FLOAT32_C(    39.79), SIMDE_FLOAT32_C(   257.60),
        SIMDE_FLOAT32_C(   -13.87), SIMDE_FLOAT32_C(   385.43), SIMDE_FLOAT32_C(   657.69), SIMDE_FLOAT32_C(   261.33),
        SIMDE_FLOAT32_C(  -197.63), SIMDE_FLOAT32_C(  -362.80), SIMDE_FLOAT32_C(   -10.34), SIMDE_FLOAT32_C(  -825.29) },
      UINT16_C( 7325) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m512 b = simde_mm512_loadu_ps(test_vec[i].b);
    simde__mmask16 r = simde_mm512_cmplt_ps_mask(a, b);
    simde_assert_mmask16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cmpeq_ps_mask (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float32 a[16];
    const simde_float32 b[16];
    const simde__mmask16 r;
  } test_vec[] = {
    { { SIMDE_FLOAT32_C(  -330.05), SIMDE_FLOAT32_C(   847.28), SIMDE_FLOAT32_C(    61.79), SIMDE_FLOAT32_C(   748.75),
        SIMDE_FLOAT32_C(  -125.94), SIMDE_FLOAT32_C(  -287.83), SIMDE_FLOAT32_C(  -156.45), SIMDE_FLOAT32_C(  -904.26),
        SIMDE_FLOAT32_C(   393.62), SIMDE_FLOAT32_C(   694.71), SIMDE_FLOAT32_C(   345.37), SIMDE_FLOAT32_C(   245.98),
        SIMDE_FLOAT32_C(  -522.67), SIMDE_FLOAT32_C(   140.34), SIMDE_FLOAT32_C(  -555.38), SIMDE_FLOAT32_C(   596.45) },
      { SIMDE_FLOAT32_C(  -330.05), SIMDE_FLOAT32_C(  -812.74), SIMDE_FLOAT32_C(    61.79), SIMDE_FLOAT32_C(  -304.55),
        SIMDE_FLOAT32_C(    95.53), SIMDE_FLOAT32_C(  -287.83), SIMDE_FLOAT32_C(  -156.45), SIMDE_FLOAT32_C(   699.14),
        SIMDE_FLOAT32_C(   676.85), SIMDE_FLOAT32_C(   694.71), SIMDE_FLOAT32_C(   345.37), SIMDE_FLOAT32_C(   245.98),
        SIMDE_FLOAT32_C(  -161.51), SIMDE_FLOAT32_C(   140.34), SIMDE_FLOAT32_C(  -399.39), SIMDE_FLOAT32_C(   596.45) },
      UINT16_C(44645) },
    { { SIMDE_FLOAT32_C(  -717.84), SIMDE_FLOAT32_C(   512.02), SIMDE_FLOAT32_C(    98.80), SIMDE_FLOAT32_C(  -966.72),
        SIMDE_FLOAT32_C(   -60.71), SIMDE_FLOAT32_C(  -584.27), SIMDE_FLOAT32_C(   204.10), SIMDE_FLOAT32_C(   295.96),
        SIMDE_FLOAT32_C(   -70.24), SIMDE_FLOAT32_C(   661.77), SIMDE_FLOAT32_C(   894.04), SIMDE_FLOAT32_C(   352.28),
        SIMDE_FLOAT32_C(   620.44), SIMDE_FLOAT32_C(   936.22), SIMDE_FLOAT32_C(   428.81), SIMDE_FLOAT32_C(   543.55) },
      { SIMDE_FLOAT32_C(  -717.84), SIMDE_FLOAT32_C(   313.36), SIMDE_FLOAT32_C(  -806.61), SIMDE_FLOAT32_C(   690.40),
        SIMDE_FLOAT32_C(   646.60), SIMDE_FLOAT32_C(  -584.27), SIMDE_FLOAT32_C(   204.10), SIMDE_FLOAT32_C(   460.04),
        SIMDE_FLOAT32_C(   733.57), SIMDE_FLOAT32_C(   661.77), SIMDE_FLOAT32_C(   894.04), SIMDE_FLOAT32_C(  -252.47),
        SIMDE_FLOAT32_C(  -340.15), SIMDE_FLOAT32_C(   936.22), SIMDE_FLOAT32_C(   428.81), SIMDE_FLOAT32_C(   543.55) },
      UINT16_C(58977) },
    { { SIMDE_FLOAT32_C(   375.91), SIMDE_FLOAT32_C(   -19.72), SIMDE_FLOAT32_C(   336.05), SIMDE_FLOAT32_C(  -540.20),
        SIMDE_FLOAT32_C(  -665.47), SIMDE_FLOAT32_C(  -492.58), SIMDE_FLOAT32_C(    15.30), SIMDE_FLOAT32_C(   126.92),
        SIMDE_FLOAT32_C(   767.58), SIMDE_FLOAT32_C(   861.15), SIMDE_FLOAT32_C(   -58.47), SIMDE_FLOAT32_C(  -387.52),
        SIMDE_FLOAT32_C(   800.70), SIMDE_FLOAT32_C(  -537.31), SIMDE_FLOAT32_C(  -644.51), SIMDE_FLOAT32_C(  -955.44) },
      { SIMDE_FLOAT32_C(   375.91), SIMDE_FLOAT32_C(  -356.80), SIMDE_FLOAT32_C(   336.05), SIMDE_FLOAT32_C(   -10.02),
        SIMDE_FLOAT32_C(   -64.34), SIMDE_FLOAT32_C(   408.76), SIMDE_FLOAT32_C(  -734.89), SIMDE_FLOAT32_C(   126.92),
        SIMDE_FLOAT32_C(    10.21), SIMDE_FLOAT32_C(   861.15), SIMDE_FLOAT32_C(   153.18), SIMDE_FLOAT32_C(   569.21),
        SIMDE_FLOAT32_C(   321.66), SIMDE_FLOAT32_C(  -537.31), SIMDE_FLOAT32_C(   613.36), SIMDE_FLOAT32_C(  -776.54) },
      UINT16_C( 8837) },
    { { SIMDE_FLOAT32_C(  -842.85), SIMDE_FLOAT32_C(  -336.15), SIMDE_FLOAT32_C(  -966.61), SIMDE_FLOAT32_C(   670.20),
        SIMDE_FLOAT32_C(   783.55), SIMDE_FLOAT32_C(   477.19), SIMDE_FLOAT32_C(  -864.95), SIMDE_FLOAT32_C(  -372.20),
        SIMDE_FLOAT32_C(   -94.30), SIMDE_FLOAT32_C(  -879.40), SIMDE_FLOAT32_C(  -161.82), SIMDE_FLOAT32_C(   100.12),
        SIMDE_FLOAT32_C(   850.32), SIMDE_FLOAT32_C(   476.49), SIMDE_FLOAT32_C(  -174.44), SIMDE_FLOAT32_C(   934.13) },
      { SIMDE_FLOAT32_C(   404.40), SIMDE_FLOAT32_C(  -570.57), SIMDE_FLOAT32_C(   -86.01), SIMDE_FLOAT32_C(   670.20),
        SIMDE_FLOAT32_C(   312.45), SIMDE_FLOAT32_C(   381.45), SIMDE_FLOAT32_C(  -864.95), SIMDE_FLOAT32_C(  -372.20),
        SIMDE_FLOAT32_C(   -94.30), SIMDE_FLOAT32_C(  -879.40), SIMDE_FLOAT32_C(  -161.82), SIMDE_FLOAT32_C(   100.12),
        SIMDE_FLOAT32_C(   850.32), SIMDE_FLOAT32_C(   386.54), SIMDE_FLOAT32_C(   295.11), SIMDE_FLOAT32_C(  -992.52) },
      UINT16_C( 8136) },
    { { SIMDE_FLOAT32_C(   877.31), SIMDE_FLOAT32_C(   884.47), SIMDE_FLOAT32_C(  -380.38), SIMDE_FLOAT32_C(  -700.84),
        SIMDE_FLOAT32_C(   945.89), SIMDE_FLOAT32_C(   280.68), SIMDE_FLOAT32_C(   832.06), SIMDE_FLOAT32_C(   359.22),
        SIMDE_FLOAT32_C(   586.70), SIMDE_FLOAT32_C(   448.55), SIMDE_FLOAT32_C(   510.98), SIMDE_FLOAT32_C(  -325.00),
        SIMDE_FLOAT32_C(   847.13), SIMDE_FLOAT32_C(  -548.42), SIMDE_FLOAT32_C(  -663.23), SIMDE_FLOAT32_C(   110.33) },
      { SIMDE_FLOAT32_C(   877.31), SIMDE_FLOAT32_C(   884.47), SIMDE_FLOAT32_C(  -380.38), SIMDE_FLOAT32_C(  -700.84),
        SIMDE_FLOAT32_C(   945.89), SIMDE_FLOAT32_C(   280.68), SIMDE_FLOAT32_C(   832.06), SIMDE_FLOAT32_C(   359.22),
        SIMDE_FLOAT32_C(  -806.36), SIMDE_FLOAT32_C(  -673.67), SIMDE_FLOAT32_C(   510.98), SIMDE_FLOAT32_C(  -346.39),
        SIMDE_FLOAT32_C(   789.45), SIMDE_FLOAT32_C(  -548.42), SIMDE_FLOAT32_C(   989.10), SIMDE_FLOAT32_C(  -487.94) },
      UINT16_C( 9471) },
    { { SIMDE_FLOAT32_C(  -787.05), SIMDE_FLOAT32_C(   806.72), SIMDE_FLOAT32_C(   520.29), SIMDE_FLOAT32_C(  -321.05),
        SIMDE_FLOAT32_C(  -366.95), SIMDE_FLOAT32_C(  -748.89), SIMDE_FLOAT32_C(   687.71), SIMDE_FLOAT32_C(  -416.88),
        SIMDE_FLOAT32_C(  -561.92), SIMDE_FLOAT32_C(  -926.01), SIMDE_FLOAT32_C(   843.79), SIMDE_FLOAT32_C(   849.56),
        SIMDE_FLOAT32_C(   -51.86), SIMDE_FLOAT32_C(  -481.78), SIMDE_FLOAT32_C(   491.33), SIMDE_FLOAT32_C(  -936.26) },
      { SIMDE_FLOAT32_C(  -787.05), SIMDE_FLOAT32_C(   806.72), SIMDE_FLOAT32_C(   299.54), SIMDE_FLOAT32_C(   884.74),
        SIMDE_FLOAT32_C(  -278.71), SIMDE_FLOAT32_C(  -748.89), SIMDE_FLOAT32_C(   570.30), SIMDE_FLOAT32_C(  -416.88),
        SIMDE_FLOAT32_C(  -561.92), SIMDE_FLOAT32_C(    59.09), SIMDE_FLOAT32_C(   843.79), SIMDE_FLOAT32_C(   849.56),
        SIMDE_FLOAT32_C(  -136.84), SIMDE_FLOAT32_C(  -481.78), SIMDE_FLOAT32_C(   491.33), SIMDE_FLOAT32_C(  -936.26) },
      UINT16_C(60835) },
    { { SIMDE_FLOAT32_C(  -837.49), SIMDE_FLOAT32_C(   -79.02), SIMDE_FLOAT32_C(  -844.39), SIMDE_FLOAT32_C(  -973.47),
        SIMDE_FLOAT32_C(  -499.80), SIMDE_FLOAT32_C(   961.14), SIMDE_FLOAT32_C(   336.59), SIMDE_FLOAT32_C(  -368.95),
        SIMDE_FLOAT32_C(   727.99), SIMDE_FLOAT32_C(  -900.81), SIMDE_FLOAT32_C(   655.07), SIMDE_FLOAT32_C(  -624.42),
        SIMDE_FLOAT32_C(   244.09), SIMDE_FLOAT32_C(   360.96), SIMDE_FLOAT32_C(  -837.70), SIMDE_FLOAT32_C(  -929.19) },
      { SIMDE_FLOAT32_C(  -837.49), SIMDE_FLOAT32_C(   -79.02), SIMDE_FLOAT32_C(  -169.54), SIMDE_FLOAT32_C(   100.98),
        SIMDE_FLOAT32_C(  -499.80), SIMDE_FLOAT32_C(   961.14), SIMDE_FLOAT32_C(  -254.87), SIMDE_FLOAT32_C(   592.42),
        SIMDE_FLOAT32_C(   312.40), SIMDE_FLOAT32_C(   958.12), SIMDE_FLOAT32_C(  -284.13), SIMDE_FLOAT32_C(  -624.42),
        SIMDE_FLOAT32_C(  -196.30), SIMDE_FLOAT32_C(   360.96), SIMDE_FLOAT32_C(  -837.70), SIMDE_FLOAT32_C(  -975.45) },
      UINT16_C(26675) },
    { { SIMDE_FLOAT32_C(   928.69), SIMDE_FLOAT32_C(    -3.95), SIMDE_FLOAT32_C(  -214.33), SIMDE_FLOAT32_C(  -971.80),
        SIMDE_FLOAT32_C(  -780.70), SIMDE_FLOAT32_C(   950.39), SIMDE_FLOAT32_C(  -857.68), SIMDE_FLOAT32_C(  -246.08),
        SIMDE_FLOAT32_C(   789.62), SIMDE_FLOAT32_C(  -840.89), SIMDE_FLOAT32_C(   194.42), SIMDE_FLOAT32_C(  -873.48),
        SIMDE_FLOAT32_C(  -365.78), SIMDE_FLOAT32_C(  -117.81), SIMDE_FLOAT32_C(   601.86), SIMDE_FLOAT32_C(   913.26) },
      { SIMDE_FLOAT32_C(   928.69), SIMDE_FLOAT32_C(    -3.95), SIMDE_FLOAT32_C(  -214.33), SIMDE_FLOAT32_C(   377.34),
        SIMDE_FLOAT32_C(  -525.21), SIMDE_FLOAT32_C(  -436.16), SIMDE_FLOAT32_C(   186.25), SIMDE_FLOAT32_C(  -246.08),
        SIMDE_FLOAT32_C(   623.36), SIMDE_FLOAT32_C(  -840.89), SIMDE_FLOAT32_C(   194.42), SIMDE_FLOAT32_C(  -873.48),
        SIMDE_FLOAT32_C(  -679.52), SIMDE_FLOAT32_C(   447.41), SIMDE_FLOAT32_C(  -608.79), SIMDE_FLOAT32_C(   721.43) },
      UINT16_C( 3719) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512 a = simde_mm512_loadu_ps(test_vec[i].a);
    simde__m512 b = simde_mm512_loadu_ps(test_vec[i].b);
    simde__mmask16 r = simde_mm512_cmpeq_ps_mask(a, b);
    simde_assert_mmask16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cmplt_pd_mask (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 a[8];
    const simde_float64 b[8];
    const simde__mmask8 r;
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   159.59), SIMDE_FLOAT64_C(   210.36), SIMDE_FLOAT64_C(  -469.27), SIMDE_FLOAT64_C(  -961.30),
        SIMDE_FLOAT64_C(  -565.87), SIMDE_FLOAT64_C(  -556.86), SIMDE_FLOAT64_C(   785.14), SIMDE_FLOAT64_C(   -76.60) },
      { SIMDE_FLOAT64_C(   549.87), SIMDE_FLOAT64_C(  -373.87), SIMDE_FLOAT64_C(   375.69), SIMDE_FLOAT64_C(   255.43),
        SIMDE_FLOAT64_C(  -924.84), SIMDE_FLOAT64_C(   236.64), SIMDE_FLOAT64_C(  -838.91), SIMDE_FLOAT64_C(   432.31) },
      UINT8_C(173) },
    { { SIMDE_FLOAT64_C(   -86.33), SIMDE_FLOAT64_C(   998.88), SIMDE_FLOAT64_C(   169.13), SIMDE_FLOAT64_C(   558.70),
        SIMDE_FLOAT64_C(   146.37), SIMDE_FLOAT64_C(    90.58), SIMDE_FLOAT64_C(   405.41), SIMDE_FLOAT64_C(  -319.04) },
      { SIMDE_FLOAT64_C(  -110.18), SIMDE_FLOAT64_C(   182.04), SIMDE_FLOAT64_C(  -496.16), SIMDE_FLOAT64_C(  -883.07),
        SIMDE_FLOAT64_C(   321.39), SIMDE_FLOAT64_C(  -344.51), SIMDE_FLOAT64_C(   -99.97), SIMDE_FLOAT64_C(  -263.70) },
      UINT8_C(144) },
    { { SIMDE_FLOAT64_C(    29.15), SIMDE_FLOAT64_C(   -41.18), SIMDE_FLOAT64_C(  -110.04), SIMDE_FLOAT64_C(   548.12),
        SIMDE_FLOAT64_C(   271.03), SIMDE_FLOAT64_C(  -770.85), SIMDE_FLOAT64_C(   346.58), SIMDE_FLOAT64_C(   912.24) },
      { SIMDE_FLOAT64_C(    79.16), SIMDE_FLOAT64_C(   358.77), SIMDE_FLOAT64_C(  -231.91), SIMDE_FLOAT64_C(   206.83),
        SIMDE_FLOAT64_C(   115.65), SIMDE_FLOAT64_C(  -336.81), SIMDE_FLOAT64_C(  -732.53), SIMDE_FLOAT64_C(   334.52) },
      UINT8_C( 35) },
    { { SIMDE_FLOAT64_C(   256.18), SIMDE_FLOAT64_C(  -459.33), SIMDE_FLOAT64_C(   101.00), SIMDE_FLOAT64_C(  -417.14),
        SIMDE_FLOAT64_C(  -900.86), SIMDE_FLOAT64_C(  -806.81), SIMDE_FLOAT64_C(    -5.42), SIMDE_FLOAT64_C(   857.96) },
      { SIMDE_FLOAT64_C(  -232.59), SIMDE_FLOAT64_C(   931.14), SIMDE_FLOAT64_C(  -321.87), SIMDE_FLOAT64_C(   407.35),
        SIMDE_FLOAT64_C(   262.90), SIMDE_FLOAT64_C(   592.56), SIMDE_FLOAT64_C(  -812.34), SIMDE_FLOAT64_C(   950.75) },
      UINT8_C(186) },
    { { SIMDE_FLOAT64_C(  -662.53), SIMDE_FLOAT64_C(   872.08), SIMDE_FLOAT64_C(  -996.83), SIMDE_FLOAT64_C(   245.09),
        SIMDE_FLOAT64_C(  -755.15), SIMDE_FLOAT64_C(   154.86), SIMDE_FLOAT64_C(   690.61), SIMDE_FLOAT64_C(  -850.32) },
      { SIMDE_FLOAT64_C(   718.59), SIMDE_FLOAT64_C(  -644.78), SIMDE_FLOAT64_C(  -744.92), SIMDE_FLOAT64_C(   162.05),
        SIMDE_FLOAT64_C(  -429.20), SIMDE_FLOAT64_C(   382.77), SIMDE_FLOAT64_C(  -712.41), SIMDE_FLOAT64_C(   553.41) },
      UINT8_C(181) },
    { { SIMDE_FLOAT64_C(  -767.88), SIMDE_FLOAT64_C(   220.93), SIMDE_FLOAT64_C(  -852.88), SIMDE_FLOAT64_C(  -422.20),
        SIMDE_FLOAT64_C(    24.06), SIMDE_FLOAT64_C(   396.29), SIMDE_FLOAT64_C(   393.46), SIMDE_FLOAT64_C(   825.11) },
      { SIMDE_FLOAT64_C(  -326.63), SIMDE_FLOAT64_C(   260.49), SIMDE_FLOAT64_C(    21.96), SIMDE_FLOAT64_C(  -870.80),
        SIMDE_FLOAT64_C(   390.98), SIMDE_FLOAT64_C(  -810.50), SIMDE_FLOAT64_C(   -47.31), SIMDE_FLOAT64_C(   928.47) },
      UINT8_C(151) },
    { { SIMDE_FLOAT64_C(   764.04), SIMDE_FLOAT64_C(  -755.85), SIMDE_FLOAT64_C(   350.20), SIMDE_FLOAT64_C(  -122.92),
        SIMDE_FLOAT64_C(    41.32), SIMDE_FLOAT64_C(   468.91), SIMDE_FLOAT64_C(   941.23), SIMDE_FLOAT64_C(  -826.92) },
      { SIMDE_FLOAT64_C(   -79.39), SIMDE_FLOAT64_C(  -301.22), SIMDE_FLOAT64_C(  -613.48), SIMDE_FLOAT64_C(  -831.83),
        SIMDE_FLOAT64_C(  -533.10), SIMDE_FLOAT64_C(   168.63), SIMDE_FLOAT64_C(   232.01), SIMDE_FLOAT64_C(  -589.49) },
      UINT8_C(130) },
    { { SIMDE_FLOAT64_C(   431.35), SIMDE_FLOAT64_C(  -312.15), SIMDE_FLOAT64_C(  -300.41), SIMDE_FLOAT64_C(  -919.37),
        SIMDE_FLOAT64_C(    97.60), SIMDE_FLOAT64_C(   323.36), SIMDE_FLOAT64_C(   650.47), SIMDE_FLOAT64_C(   378.00) },
      { SIMDE_FLOAT64_C(  -942.80), SIMDE_FLOAT64_C(   278.12), SIMDE_FLOAT64_C(   437.54), SIMDE_FLOAT64_C(  -207.26),
        SIMDE_FLOAT64_C(   628.37), SIMDE_FLOAT64_C(  -977.34), SIMDE_FLOAT64_C(   -73.78), SIMDE_FLOAT64_C(   -44.83) },
      UINT8_C( 30) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m512d b = simde_mm512_loadu_pd(test_vec[i].b);
    simde__mmask8 r = simde_mm512_cmplt_pd_mask(a, b);
    simde_assert_mmask8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cmpeq_pd_mask (SIMDE_MUNIT_TEST_ARGS) {
  static const struct {
    const simde_float64 a[8];
    const simde_float64 b[8];
    const simde__mmask8 r;
  } test_vec[] = {
    { { SIMDE_FLOAT64_C(   271.69), SIMDE_FLOAT64_C(   217.70), SIMDE_FLOAT64_C(   925.72), SIMDE_FLOAT64_C(   520.03),
        SIMDE_FLOAT64_C(   127.68), SIMDE_FLOAT64_C(   -63.68), SIMDE_FLOAT64_C(  -338.02), SIMDE_FLOAT64_C(   823.86) },
      { SIMDE_FLOAT64_C(   690.45), SIMDE_FLOAT64_C(   347.44), SIMDE_FLOAT64_C(  -649.53), SIMDE_FLOAT64_C(  -641.60),
        SIMDE_FLOAT64_C(   658.05), SIMDE_FLOAT64_C(  -212.84), SIMDE_FLOAT64_C(   796.21), SIMDE_FLOAT64_C(   -36.15) },
      UINT8_C(  0) },
    { { SIMDE_FLOAT64_C(   417.99), SIMDE_FLOAT64_C(  -883.29), SIMDE_FLOAT64_C(   571.34), SIMDE_FLOAT64_C(   535.08),
        SIMDE_FLOAT64_C(  -923.74), SIMDE_FLOAT64_C(   773.69), SIMDE_FLOAT64_C(  -589.26), SIMDE_FLOAT64_C(   350.94) },
      { SIMDE_FLOAT64_C(   179.83), SIMDE_FLOAT64_C(   445.85), SIMDE_FLOAT64_C(  -677.60), SIMDE_FLOAT64_C(  -480.10),
        SIMDE_FLOAT64_C(  -974.87), SIMDE_FLOAT64_C(  -558.93), SIMDE_FLOAT64_C(    47.77), SIMDE_FLOAT64_C(   924.57) },
      UINT8_C(  0) },
    { { SIMDE_FLOAT64_C(  -695.12), SIMDE_FLOAT64_C(  -819.01), SIMDE_FLOAT64_C(   861.37), SIMDE_FLOAT64_C(  -968.92),
        SIMDE_FLOAT64_C(  -642.09), SIMDE_FLOAT64_C(   475.36), SIMDE_FLOAT64_C(  -653.40), SIMDE_FLOAT64_C(   274.91) },
      { SIMDE_FLOAT64_C(   408.72), SIMDE_FLOAT64_C(  -646.21), SIMDE_FLOAT64_C(  -843.45), SIMDE_FLOAT64_C(   107.84),
        SIMDE_FLOAT64_C(   465.38), SIMDE_FLOAT64_C(  -336.34), SIMDE_FLOAT64_C(  -820.42), SIMDE_FLOAT64_C(  -749.33) },
      UINT8_C(  0) },
    { { SIMDE_FLOAT64_C(    -9.72), SIMDE_FLOAT64_C(   643.44), SIMDE_FLOAT64_C(   336.27), SIMDE_FLOAT64_C(  -313.97),
        SIMDE_FLOAT64_C(  -863.83), SIMDE_FLOAT64_C(  -448.10), SIMDE_FLOAT64_C(   771.84), SIMDE_FLOAT64_C(   249.27) },
      { SIMDE_FLOAT64_C(  -506.33), SIMDE_FLOAT64_C(    28.98), SIMDE_FLOAT64_C(  -919.42), SIMDE_FLOAT64_C(  -710.08),
        SIMDE_FLOAT64_C(  -376.38), SIMDE_FLOAT64_C(   181.22), SIMDE_FLOAT64_C(  -315.61), SIMDE_FLOAT64_C(  -521.71) },
      UINT8_C(  0) },
    { { SIMDE_FLOAT64_C(  -309.90), SIMDE_FLOAT64_C(  -566.85), SIMDE_FLOAT64_C(   953.96), SIMDE_FLOAT64_C(  -760.71),
        SIMDE_FLOAT64_C(   715.80), SIMDE_FLOAT64_C(   511.82), SIMDE_FLOAT64_C(   185.57), SIMDE_FLOAT64_C(   958.96) },
      { SIMDE_FLOAT64_C(  -823.31), SIMDE_FLOAT64_C(   653.67), SIMDE_FLOAT64_C(   300.89), SIMDE_FLOAT64_C(  -999.35),
        SIMDE_FLOAT64_C(  -123.69), SIMDE_FLOAT64_C(  -935.82), SIMDE_FLOAT64_C(  -283.75), SIMDE_FLOAT64_C(  -911.67) },
      UINT8_C(  0) },
    { { SIMDE_FLOAT64_C(  -370.17), SIMDE_FLOAT64_C(   581.24), SIMDE_FLOAT64_C(   903.15), SIMDE_FLOAT64_C(  -702.97),
        SIMDE_FLOAT64_C(  -784.81), SIMDE_FLOAT64_C(  -282.51), SIMDE_FLOAT64_C(  -162.91), SIMDE_FLOAT64_C(   -67.74) },
      { SIMDE_FLOAT64_C(  -458.51), SIMDE_FLOAT64_C(  -138.00), SIMDE_FLOAT64_C(   634.22), SIMDE_FLOAT64_C(  -641.32),
        SIMDE_FLOAT64_C(  -700.95), SIMDE_FLOAT64_C(  -830.62), SIMDE_FLOAT64_C(  -270.15), SIMDE_FLOAT64_C(  -342.52) },
      UINT8_C(  0) },
    { { SIMDE_FLOAT64_C(  -741.30), SIMDE_FLOAT64_C(  -961.63), SIMDE_FLOAT64_C(  -159.42), SIMDE_FLOAT64_C(   596.72),
        SIMDE_FLOAT64_C(  -872.26), SIMDE_FLOAT64_C(   -77.79), SIMDE_FLOAT64_C(   608.69), SIMDE_FLOAT64_C(   181.91) },
      { SIMDE_FLOAT64_C(  -693.78), SIMDE_FLOAT64_C(  -430.90), SIMDE_FLOAT64_C(  -141.87), SIMDE_FLOAT64_C(  -384.25),
        SIMDE_FLOAT64_C(   -74.70), SIMDE_FLOAT64_C(   434.70), SIMDE_FLOAT64_C(    -4.99), SIMDE_FLOAT64_C(   104.05) },
      UINT8_C(  0) },
    { { SIMDE_FLOAT64_C(   817.79), SIMDE_FLOAT64_C(   652.33), SIMDE_FLOAT64_C(  -345.32), SIMDE_FLOAT64_C(   150.71),
        SIMDE_FLOAT64_C(   939.32), SIMDE_FLOAT64_C(  -867.25), SIMDE_FLOAT64_C(   158.96), SIMDE_FLOAT64_C(  -396.12) },
      { SIMDE_FLOAT64_C(   363.34), SIMDE_FLOAT64_C(   571.53), SIMDE_FLOAT64_C(  -232.25), SIMDE_FLOAT64_C(   496.58),
        SIMDE_FLOAT64_C(    40.81), SIMDE_FLOAT64_C(   -69.57), SIMDE_FLOAT64_C(   792.81), SIMDE_FLOAT64_C(   833.83) },
      UINT8_C(  0) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])) ; i++) {
    simde__m512d a = simde_mm512_loadu_pd(test_vec[i].a);
    simde__m512d b = simde_mm512_loadu_pd(test_vec[i].b);
    simde__mmask8 r = simde_mm512_cmpeq_pd_mask(a, b);
    simde_assert_mmask8(r, ==, test_vec[i].r);
  }

  return 0;
}

#endif /* !defined(SIMDE_BUG_GCC_96174) */

static int
test_simde_mm512_mask_cvtepi8_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( -948891544), INT32_C( 1432337912), INT32_C( -157601137), INT32_C(  557886841),
                            INT32_C(  405458672), INT32_C( 1108283966), INT32_C(  671693090), INT32_C( -656229219),
                            INT32_C( 1844651109), INT32_C(  722363475), INT32_C( 1034392721), INT32_C( -352112508),
                            INT32_C( -454251458), INT32_C( -470551525), INT32_C( -884105703), INT32_C( 1305270065)),
      UINT16_C(30779),
      simde_mm_set_epi8(INT8_C(-116), INT8_C(  40), INT8_C(  78), INT8_C(  27),
                        INT8_C( -49), INT8_C(  94), INT8_C(  11), INT8_C(  59),
                        INT8_C(-125), INT8_C(  49), INT8_C(  61), INT8_C(-117),
                        INT8_C( -52), INT8_C(-102), INT8_C(  15), INT8_C(  39)),
      simde_mm512_set_epi32(INT32_C( -948891544), INT32_C(         40), INT32_C(         78), INT32_C(         27),
                            INT32_C(        -49), INT32_C( 1108283966), INT32_C(  671693090), INT32_C( -656229219),
                            INT32_C( 1844651109), INT32_C(  722363475), INT32_C(         61), INT32_C(       -117),
                            INT32_C(        -52), INT32_C( -470551525), INT32_C(         15), INT32_C(         39)) },
    { simde_mm512_set_epi32(INT32_C(-1678459053), INT32_C( -751001872), INT32_C(  948848242), INT32_C(-1255138969),
                            INT32_C(  843494992), INT32_C( 1568180449), INT32_C( 1114219711), INT32_C(  -91183620),
                            INT32_C(  -74671399), INT32_C( -726320383), INT32_C(  339500370), INT32_C(  976271619),
                            INT32_C( -404720858), INT32_C(-1856521809), INT32_C(  491307301), INT32_C(  992626281)),
      UINT16_C(28152),
      simde_mm_set_epi8(INT8_C(  95), INT8_C( -75), INT8_C(  81), INT8_C(  24),
                        INT8_C(  84), INT8_C(   1), INT8_C(  61), INT8_C( -53),
                        INT8_C( -76), INT8_C(  20), INT8_C( -59), INT8_C(  -3),
                        INT8_C(  37), INT8_C(-113), INT8_C(-116), INT8_C(   6)),
      simde_mm512_set_epi32(INT32_C(-1678459053), INT32_C(        -75), INT32_C(         81), INT32_C(-1255138969),
                            INT32_C(         84), INT32_C(          1), INT32_C( 1114219711), INT32_C(        -53),
                            INT32_C(        -76), INT32_C(         20), INT32_C(        -59), INT32_C(         -3),
                            INT32_C(         37), INT32_C(-1856521809), INT32_C(  491307301), INT32_C(  992626281)) },
    { simde_mm512_set_epi32(INT32_C( 1482305192), INT32_C( 1996708987), INT32_C(  723535956), INT32_C(-1670036300),
                            INT32_C( -363335449), INT32_C(  -47930977), INT32_C( -403738658), INT32_C( 1182057510),
                            INT32_C(  831714400), INT32_C(-1744656678), INT32_C(-1981557460), INT32_C( -724476623),
                            INT32_C(  939131668), INT32_C(  353771966), INT32_C( 1625236055), INT32_C(-2058304981)),
      UINT16_C(35313),
      simde_mm_set_epi8(INT8_C(  81), INT8_C(  92), INT8_C( -16), INT8_C( -77),
                        INT8_C( -10), INT8_C(  31), INT8_C(  48), INT8_C(  46),
                        INT8_C( 122), INT8_C( -46), INT8_C( 100), INT8_C( -42),
                        INT8_C( -95), INT8_C( -43), INT8_C(  68), INT8_C(-104)),
      simde_mm512_set_epi32(INT32_C(         81), INT32_C( 1996708987), INT32_C(  723535956), INT32_C(-1670036300),
                            INT32_C(        -10), INT32_C(  -47930977), INT32_C( -403738658), INT32_C(         46),
                            INT32_C(        122), INT32_C(        -46), INT32_C(        100), INT32_C(        -42),
                            INT32_C(  939131668), INT32_C(  353771966), INT32_C( 1625236055), INT32_C(       -104)) },
    { simde_mm512_set_epi32(INT32_C( 1830936869), INT32_C(-1741742044), INT32_C(-1737557360), INT32_C( -273240771),
                            INT32_C(-1961147279), INT32_C( -229097469), INT32_C( -763894872), INT32_C( 1529380840),
                            INT32_C( -665264196), INT32_C(  639643484), INT32_C(  240439150), INT32_C( 1588063308),
                            INT32_C(  586997217), INT32_C( 1018879786), INT32_C( 1848501586), INT32_C( -979270594)),
      UINT16_C(33265),
      simde_mm_set_epi8(INT8_C(-123), INT8_C(  90), INT8_C(  -2), INT8_C(  97),
                        INT8_C(  68), INT8_C(  78), INT8_C(  77), INT8_C(  55),
                        INT8_C(-105), INT8_C(-116), INT8_C(  28), INT8_C( -62),
                        INT8_C( -88), INT8_C(  -8), INT8_C( 115), INT8_C(-111)),
      simde_mm512_set_epi32(INT32_C(       -123), INT32_C(-1741742044), INT32_C(-1737557360), INT32_C( -273240771),
                            INT32_C(-1961147279), INT32_C( -229097469), INT32_C( -763894872), INT32_C(         55),
                            INT32_C(       -105), INT32_C(       -116), INT32_C(         28), INT32_C(        -62),
                            INT32_C(  586997217), INT32_C( 1018879786), INT32_C( 1848501586), INT32_C(       -111)) },
    { simde_mm512_set_epi32(INT32_C( 1041391151), INT32_C( 1093398186), INT32_C(-1592148298), INT32_C( 1164412431),
                            INT32_C(-1767544990), INT32_C(  389847058), INT32_C(  388979962), INT32_C(  625503476),
                            INT32_C(-1034617064), INT32_C(  622175081), INT32_C(  228405175), INT32_C( 1183542116),
                            INT32_C( 1822156655), INT32_C(  232229273), INT32_C(-1158516102), INT32_C( 1768636180)),
      UINT16_C(48999),
      simde_mm_set_epi8(INT8_C(  97), INT8_C( -30), INT8_C( 124), INT8_C(  25),
                        INT8_C( -97), INT8_C(-128), INT8_C(  84), INT8_C( -30),
                        INT8_C( -42), INT8_C(  87), INT8_C(  15), INT8_C(-126),
                        INT8_C(  19), INT8_C( 104), INT8_C(  47), INT8_C( -47)),
      simde_mm512_set_epi32(INT32_C(         97), INT32_C( 1093398186), INT32_C(        124), INT32_C(         25),
                            INT32_C(        -97), INT32_C(       -128), INT32_C(         84), INT32_C(        -30),
                            INT32_C(-1034617064), INT32_C(         87), INT32_C(         15), INT32_C( 1183542116),
                            INT32_C( 1822156655), INT32_C(        104), INT32_C(         47), INT32_C(        -47)) },
    { simde_mm512_set_epi32(INT32_C(   95296887), INT32_C(-1039650207), INT32_C(  851568188), INT32_C(-1463556566),
                            INT32_C( -376745735), INT32_C(  437204996), INT32_C( -892096867), INT32_C(-1877695621),
                            INT32_C(  434885689), INT32_C(-1210422431), INT32_C(  933725665), INT32_C( -669851252),
                            INT32_C(  149185919), INT32_C(-1625981030), INT32_C( -573589938), INT32_C(  178183647)),
      UINT16_C(16017),
      simde_mm_set_epi8(INT8_C(-121), INT8_C( -49), INT8_C( 114), INT8_C(-112),
                        INT8_C( -43), INT8_C( -25), INT8_C( 119), INT8_C( 103),
                        INT8_C(  40), INT8_C(-112), INT8_C(  -8), INT8_C( -89),
                        INT8_C(  73), INT8_C( -54), INT8_C(  83), INT8_C( -40)),
      simde_mm512_set_epi32(INT32_C(   95296887), INT32_C(-1039650207), INT32_C(        114), INT32_C(       -112),
                            INT32_C(        -43), INT32_C(        -25), INT32_C(        119), INT32_C(-1877695621),
                            INT32_C(         40), INT32_C(-1210422431), INT32_C(  933725665), INT32_C(        -89),
                            INT32_C(  149185919), INT32_C(-1625981030), INT32_C( -573589938), INT32_C(        -40)) },
    { simde_mm512_set_epi32(INT32_C( -739285659), INT32_C(   81885987), INT32_C(  847528238), INT32_C(-2089319861),
                            INT32_C(-1629855246), INT32_C( 2048738507), INT32_C( 1195062920), INT32_C(  399815228),
                            INT32_C(   83479215), INT32_C(  685330423), INT32_C( -319423423), INT32_C(  771854533),
                            INT32_C(  -58748336), INT32_C(-1621219518), INT32_C(-1185941810), INT32_C(-2120224468)),
      UINT16_C(27674),
      simde_mm_set_epi8(INT8_C(  70), INT8_C( -19), INT8_C( -25), INT8_C( -59),
                        INT8_C(  64), INT8_C( 110), INT8_C( -46), INT8_C( -94),
                        INT8_C(  69), INT8_C(   2), INT8_C(  44), INT8_C( -87),
                        INT8_C( -72), INT8_C(-120), INT8_C( -53), INT8_C( -22)),
      simde_mm512_set_epi32(INT32_C( -739285659), INT32_C(        -19), INT32_C(        -25), INT32_C(-2089319861),
                            INT32_C(         64), INT32_C(        110), INT32_C( 1195062920), INT32_C(  399815228),
                            INT32_C(   83479215), INT32_C(  685330423), INT32_C( -319423423), INT32_C(        -87),
                            INT32_C(        -72), INT32_C(-1621219518), INT32_C(        -53), INT32_C(-2120224468)) },
    { simde_mm512_set_epi32(INT32_C(  812929568), INT32_C( -558329566), INT32_C( -820382393), INT32_C(-2033989853),
                            INT32_C(-1691272889), INT32_C(-1250227005), INT32_C(  926032561), INT32_C( -492473735),
                            INT32_C(  131989417), INT32_C( -707025949), INT32_C(  705079402), INT32_C( 2061631189),
                            INT32_C(  860537414), INT32_C(-1872373827), INT32_C(  166627023), INT32_C( 1034543202)),
      UINT16_C(10417),
      simde_mm_set_epi8(INT8_C(  61), INT8_C( 124), INT8_C(  65), INT8_C(-124),
                        INT8_C( -83), INT8_C(  86), INT8_C(  53), INT8_C(   9),
                        INT8_C( -15), INT8_C(-117), INT8_C( -19), INT8_C(  84),
                        INT8_C(-107), INT8_C( 116), INT8_C(  83), INT8_C( -43)),
      simde_mm512_set_epi32(INT32_C(  812929568), INT32_C( -558329566), INT32_C(         65), INT32_C(-2033989853),
                            INT32_C(        -83), INT32_C(-1250227005), INT32_C(  926032561), INT32_C( -492473735),
                            INT32_C(        -15), INT32_C( -707025949), INT32_C(        -19), INT32_C(         84),
                            INT32_C(  860537414), INT32_C(-1872373827), INT32_C(  166627023), INT32_C(        -43)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_cvtepi8_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_cvtepi8_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT16_C(55089),
      simde_mm_set_epi8(INT8_C(  61), INT8_C( -89), INT8_C(-108), INT8_C(-111),
                        INT8_C( -21), INT8_C(   3), INT8_C(  48), INT8_C(-124),
                        INT8_C( -28), INT8_C( -20), INT8_C( -84), INT8_C(  62),
                        INT8_C( -29), INT8_C( -13), INT8_C( -12), INT8_C(  27)),
      simde_mm512_set_epi32(INT32_C(         61), INT32_C(        -89), INT32_C(          0), INT32_C(       -111),
                            INT32_C(          0), INT32_C(          3), INT32_C(         48), INT32_C(       -124),
                            INT32_C(          0), INT32_C(          0), INT32_C(        -84), INT32_C(         62),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(         27)) },
    { UINT16_C(25683),
      simde_mm_set_epi8(INT8_C(  24), INT8_C(  42), INT8_C( -50), INT8_C( -16),
                        INT8_C(  66), INT8_C(  15), INT8_C(  18), INT8_C(  62),
                        INT8_C(  40), INT8_C(   9), INT8_C(  57), INT8_C(  34),
                        INT8_C( -40), INT8_C( -30), INT8_C( -68), INT8_C( -99)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(         42), INT32_C(        -50), INT32_C(          0),
                            INT32_C(          0), INT32_C(         15), INT32_C(          0), INT32_C(          0),
                            INT32_C(          0), INT32_C(          9), INT32_C(          0), INT32_C(         34),
                            INT32_C(          0), INT32_C(          0), INT32_C(        -68), INT32_C(        -99)) },
    { UINT16_C(44409),
      simde_mm_set_epi8(INT8_C( 121), INT8_C(  62), INT8_C( -84), INT8_C(  58),
                        INT8_C( -34), INT8_C(  31), INT8_C( 120), INT8_C(  59),
                        INT8_C( -57), INT8_C( 113), INT8_C(  16), INT8_C( 104),
                        INT8_C(  85), INT8_C(  95), INT8_C( -67), INT8_C(  -8)),
      simde_mm512_set_epi32(INT32_C(        121), INT32_C(          0), INT32_C(        -84), INT32_C(          0),
                            INT32_C(        -34), INT32_C(         31), INT32_C(          0), INT32_C(         59),
                            INT32_C(          0), INT32_C(        113), INT32_C(         16), INT32_C(        104),
                            INT32_C(         85), INT32_C(          0), INT32_C(          0), INT32_C(         -8)) },
    { UINT16_C( 3879),
      simde_mm_set_epi8(INT8_C(  29), INT8_C(  72), INT8_C( -63), INT8_C(  37),
                        INT8_C(  59), INT8_C(  42), INT8_C(  70), INT8_C( 105),
                        INT8_C(-116), INT8_C(  40), INT8_C(  78), INT8_C(  27),
                        INT8_C( -49), INT8_C(  94), INT8_C(  11), INT8_C(  59)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0),
                            INT32_C(         59), INT32_C(         42), INT32_C(         70), INT32_C(        105),
                            INT32_C(          0), INT32_C(          0), INT32_C(         78), INT32_C(          0),
                            INT32_C(          0), INT32_C(         94), INT32_C(         11), INT32_C(         59)) },
    { UINT16_C(47535),
      simde_mm_set_epi8(INT8_C(  -5), INT8_C(-116), INT8_C(-102), INT8_C( -39),
                        INT8_C( -44), INT8_C( -75), INT8_C(  59), INT8_C(   1),
                        INT8_C(  20), INT8_C(  60), INT8_C(  93), INT8_C(  82),
                        INT8_C(  58), INT8_C(  48), INT8_C( -71), INT8_C(   3)),
      simde_mm512_set_epi32(INT32_C(         -5), INT32_C(          0), INT32_C(       -102), INT32_C(        -39),
                            INT32_C(        -44), INT32_C(          0), INT32_C(          0), INT32_C(          1),
                            INT32_C(         20), INT32_C(          0), INT32_C(         93), INT32_C(          0),
                            INT32_C(         58), INT32_C(         48), INT32_C(        -71), INT32_C(          3)) },
    { UINT16_C(42492),
      simde_mm_set_epi8(INT8_C(  56), INT8_C(-114), INT8_C(  70), INT8_C( 114),
                        INT8_C( -75), INT8_C(  48), INT8_C(  25), INT8_C( 103),
                        INT8_C(  50), INT8_C(  70), INT8_C( -74), INT8_C(  80),
                        INT8_C(  93), INT8_C( 120), INT8_C(-120), INT8_C( -31)),
      simde_mm512_set_epi32(INT32_C(         56), INT32_C(          0), INT32_C(         70), INT32_C(          0),
                            INT32_C(          0), INT32_C(         48), INT32_C(          0), INT32_C(        103),
                            INT32_C(         50), INT32_C(         70), INT32_C(        -74), INT32_C(         80),
                            INT32_C(         93), INT32_C(        120), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(40688),
      simde_mm_set_epi8(INT8_C( -76), INT8_C(  20), INT8_C( -59), INT8_C(  -3),
                        INT8_C(  37), INT8_C(-113), INT8_C(-116), INT8_C(   6),
                        INT8_C(-128), INT8_C(   9), INT8_C( -96), INT8_C(  -4),
                        INT8_C( -38), INT8_C(-121), INT8_C( 109), INT8_C(  -8)),
      simde_mm512_set_epi32(INT32_C(        -76), INT32_C(          0), INT32_C(          0), INT32_C(         -3),
                            INT32_C(         37), INT32_C(       -113), INT32_C(       -116), INT32_C(          0),
                            INT32_C(       -128), INT32_C(          9), INT32_C(        -96), INT32_C(         -4),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT16_C(15819),
      simde_mm_set_epi8(INT8_C(  55), INT8_C(  -6), INT8_C(   3), INT8_C(  20),
                        INT8_C(  21), INT8_C(  22), INT8_C(  33), INT8_C( -66),
                        INT8_C(  96), INT8_C( -33), INT8_C(  34), INT8_C(  87),
                        INT8_C(-123), INT8_C(  80), INT8_C( -62), INT8_C(  43)),
      simde_mm512_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          3), INT32_C(         20),
                            INT32_C(         21), INT32_C(         22), INT32_C(          0), INT32_C(        -66),
                            INT32_C(         96), INT32_C(        -33), INT32_C(          0), INT32_C(          0),
                            INT32_C(       -123), INT32_C(          0), INT32_C(        -62), INT32_C(         43)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_cvtepi8_epi32(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cvtepi8_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C(  40), INT8_C( -85), INT8_C(  94), INT8_C(  35),
                        INT8_C( -54), INT8_C( -71), INT8_C(-106), INT8_C(-127),
                        INT8_C(  11), INT8_C( 105), INT8_C(  37), INT8_C(-105),
                        INT8_C(   6), INT8_C( -65), INT8_C(  17), INT8_C(  29)),
      simde_mm512_set_epi64(INT64_C(                  11), INT64_C(                 105),
                            INT64_C(                  37), INT64_C(                -105),
                            INT64_C(                   6), INT64_C(                 -65),
                            INT64_C(                  17), INT64_C(                  29)) },
    { simde_mm_set_epi8(INT8_C(  65), INT8_C( -98), INT8_C( -80), INT8_C( -71),
                        INT8_C( -32), INT8_C(  61), INT8_C( -91), INT8_C(   2),
                        INT8_C(  62), INT8_C(  86), INT8_C( -39), INT8_C( -20),
                        INT8_C(  65), INT8_C( -87), INT8_C( 116), INT8_C(-112)),
      simde_mm512_set_epi64(INT64_C(                  62), INT64_C(                  86),
                            INT64_C(                 -39), INT64_C(                 -20),
                            INT64_C(                  65), INT64_C(                 -87),
                            INT64_C(                 116), INT64_C(                -112)) },
    { simde_mm_set_epi8(INT8_C( -35), INT8_C(  91), INT8_C(  95), INT8_C( -91),
                        INT8_C(-120), INT8_C( -72), INT8_C(   6), INT8_C( 127),
                        INT8_C( -64), INT8_C(  43), INT8_C( -93), INT8_C(   8),
                        INT8_C( 105), INT8_C( -16), INT8_C(  39), INT8_C( 125)),
      simde_mm512_set_epi64(INT64_C(                 -64), INT64_C(                  43),
                            INT64_C(                 -93), INT64_C(                   8),
                            INT64_C(                 105), INT64_C(                 -16),
                            INT64_C(                  39), INT64_C(                 125)) },
    { simde_mm_set_epi8(INT8_C( 108), INT8_C( 105), INT8_C(  98), INT8_C( -57),
                        INT8_C( -42), INT8_C( -18), INT8_C( -55), INT8_C(  -1),
                        INT8_C( -97), INT8_C( -26), INT8_C( -21), INT8_C(-119),
                        INT8_C(  95), INT8_C(  83), INT8_C( -95), INT8_C(  86)),
      simde_mm512_set_epi64(INT64_C(                 -97), INT64_C(                 -26),
                            INT64_C(                 -21), INT64_C(                -119),
                            INT64_C(                  95), INT64_C(                  83),
                            INT64_C(                 -95), INT64_C(                  86)) },
    { simde_mm_set_epi8(INT8_C(  63), INT8_C(-119), INT8_C(  65), INT8_C( 108),
                        INT8_C( -17), INT8_C( -16), INT8_C(  -4), INT8_C(  16),
                        INT8_C(-117), INT8_C( -62), INT8_C( -96), INT8_C(   5),
                        INT8_C( 116), INT8_C( -58), INT8_C( -56), INT8_C(-121)),
      simde_mm512_set_epi64(INT64_C(                -117), INT64_C(                 -62),
                            INT64_C(                 -96), INT64_C(                   5),
                            INT64_C(                 116), INT64_C(                 -58),
                            INT64_C(                 -56), INT64_C(                -121)) },
    { simde_mm_set_epi8(INT8_C( -69), INT8_C(  98), INT8_C(  52), INT8_C( -73),
                        INT8_C(  71), INT8_C(  44), INT8_C( -92), INT8_C(  54),
                        INT8_C( 126), INT8_C(  88), INT8_C(  32), INT8_C(-122),
                        INT8_C( -69), INT8_C( -55), INT8_C(-123), INT8_C(  79)),
      simde_mm512_set_epi64(INT64_C(                 126), INT64_C(                  88),
                            INT64_C(                  32), INT64_C(                -122),
                            INT64_C(                 -69), INT64_C(                 -55),
                            INT64_C(                -123), INT64_C(                  79)) },
    { simde_mm_set_epi8(INT8_C(-110), INT8_C( -17), INT8_C( -48), INT8_C( -64),
                        INT8_C(   9), INT8_C(  96), INT8_C( 113), INT8_C( -55),
                        INT8_C( -45), INT8_C(  -9), INT8_C( 104), INT8_C( -61),
                        INT8_C( 127), INT8_C( 121), INT8_C(  70), INT8_C( -22)),
      simde_mm512_set_epi64(INT64_C(                 -45), INT64_C(                  -9),
                            INT64_C(                 104), INT64_C(                 -61),
                            INT64_C(                 127), INT64_C(                 121),
                            INT64_C(                  70), INT64_C(                 -22)) },
    { simde_mm_set_epi8(INT8_C(  68), INT8_C( -24), INT8_C(  90), INT8_C( -28),
                        INT8_C(  55), INT8_C( -48), INT8_C(  13), INT8_C(  95),
                        INT8_C(  10), INT8_C( -72), INT8_C( 109), INT8_C( -27),
                        INT8_C(  94), INT8_C( 121), INT8_C(  33), INT8_C(  52)),
      simde_mm512_set_epi64(INT64_C(                  10), INT64_C(                 -72),
                            INT64_C(                 109), INT64_C(                 -27),
                            INT64_C(                  94), INT64_C(                 121),
                            INT64_C(                  33), INT64_C(                  52)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_cvtepi8_epi64(test_vec[i].a);
    #if defined(__EMSCRIPTEN__)
    (void) r;
    #else
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
    #endif
  }

  return 0;
}

static int
test_simde_mm512_mask_cvtepi8_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i src;
    simde__mmask8 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-4075458147498607112), INT64_C( -676891728669528711),
                            INT64_C( 1741431737227874878), INT64_C( 2884899858137922717),
                            INT64_C( 7922716186407494739), INT64_C( 4442682911858307204),
                            INT64_C(-1950995152445901797), INT64_C(-3797205079286819023)),
      UINT8_C( 59),
      simde_mm_set_epi8(INT8_C( -49), INT8_C(  94), INT8_C(  11), INT8_C(  59),
                        INT8_C(-125), INT8_C(  49), INT8_C(  61), INT8_C(-117),
                        INT8_C( -52), INT8_C(-102), INT8_C(  15), INT8_C(  39),
                        INT8_C( 121), INT8_C(  62), INT8_C( -84), INT8_C(  58)),
      simde_mm512_set_epi64(INT64_C(-4075458147498607112), INT64_C( -676891728669528711),
                            INT64_C(                  15), INT64_C(                  39),
                            INT64_C(                 121), INT64_C( 4442682911858307204),
                            INT64_C(                 -84), INT64_C(                  58)) },
    { simde_mm512_set_epi64(INT64_C(-3225528478525929870), INT64_C(-5390780822946662832),
                            INT64_C( 6735283743795815615), INT64_C( -391630661610595623),
                            INT64_C(-3119522291063693998), INT64_C( 4193054679508218662),
                            INT64_C(-7973700453474451163), INT64_C( 4263297416396557851)),
      UINT8_C( 83),
      simde_mm_set_epi8(INT8_C( -76), INT8_C(  20), INT8_C( -59), INT8_C(  -3),
                        INT8_C(  37), INT8_C(-113), INT8_C(-116), INT8_C(   6),
                        INT8_C(-128), INT8_C(   9), INT8_C( -96), INT8_C(  -4),
                        INT8_C( -38), INT8_C(-121), INT8_C( 109), INT8_C(  -8)),
      simde_mm512_set_epi64(INT64_C(-3225528478525929870), INT64_C(                   9),
                            INT64_C( 6735283743795815615), INT64_C(                  -4),
                            INT64_C(-3119522291063693998), INT64_C( 4193054679508218662),
                            INT64_C(                 109), INT64_C(                  -8)) },
    { simde_mm512_set_epi64(INT64_C( 3107563271125025972), INT64_C(-1560513866685439585),
                            INT64_C(-1734044331058871258), INT64_C( 3572186150162573018),
                            INT64_C(-8510724482274337487), INT64_C( 4033539801051701694),
                            INT64_C( 6980335706741719595), INT64_C( 6896507569318215115)),
      UINT8_C(123),
      simde_mm_set_epi8(INT8_C( -95), INT8_C( -43), INT8_C(  68), INT8_C(-104),
                        INT8_C(  48), INT8_C( 123), INT8_C( 124), INT8_C(   4),
                        INT8_C(  -4), INT8_C(  21), INT8_C(-119), INT8_C( -15),
                        INT8_C(  88), INT8_C(  90), INT8_C(  46), INT8_C( -88)),
      simde_mm512_set_epi64(INT64_C( 3107563271125025972), INT64_C(                  21),
                            INT64_C(                -119), INT64_C(                 -15),
                            INT64_C(                  88), INT64_C( 4033539801051701694),
                            INT64_C(                  46), INT64_C(                 -88)) },
    { simde_mm512_set_epi64(INT64_C(-1173560173045005199), INT64_C( -983966133420301400),
                            INT64_C( 6568640694558711740), INT64_C( 2747247845119938414),
                            INT64_C( 6820679972424572385), INT64_C( 4376055361273980242),
                            INT64_C(-4205935173799448397), INT64_C( -711797239914011434)),
      UINT8_C(144),
      simde_mm_set_epi8(INT8_C( -38), INT8_C( 103), INT8_C( 119), INT8_C(  -1),
                        INT8_C(  70), INT8_C(  39), INT8_C(-127), INT8_C( -15),
                        INT8_C( 109), INT8_C(  33), INT8_C( -31), INT8_C(  37),
                        INT8_C(-104), INT8_C(  47), INT8_C(  32), INT8_C(  36)),
      simde_mm512_set_epi64(INT64_C(                 109), INT64_C( -983966133420301400),
                            INT64_C( 6568640694558711740), INT64_C(                  37),
                            INT64_C( 6820679972424572385), INT64_C( 4376055361273980242),
                            INT64_C(-4205935173799448397), INT64_C( -711797239914011434)) },
    { simde_mm512_set_epi64(INT64_C(-7591547925868799982), INT64_C( 1670656216214826228),
                            INT64_C(-4443646453141363863), INT64_C(  980992758045698916),
                            INT64_C( 7826103241645984153), INT64_C(-4975788768210764012),
                            INT64_C(-8837471625049912009), INT64_C(-7526609254858853487)),
      UINT8_C( 15),
      simde_mm_set_epi8(INT8_C(-111), INT8_C( 121), INT8_C( -65), INT8_C( 103),
                        INT8_C(  62), INT8_C(  18), INT8_C(  94), INT8_C(  47),
                        INT8_C(  65), INT8_C(  43), INT8_C( -18), INT8_C( -86),
                        INT8_C( -95), INT8_C(  25), INT8_C( -66), INT8_C( -74)),
      simde_mm512_set_epi64(INT64_C(-7591547925868799982), INT64_C( 1670656216214826228),
                            INT64_C(-4443646453141363863), INT64_C(  980992758045698916),
                            INT64_C(                 -95), INT64_C(                  25),
                            INT64_C(                 -66), INT64_C(                 -74)) },
    { simde_mm512_set_epi64(INT64_C( 1877781162870681245), INT64_C(-8064641283602525127),
                            INT64_C(-5198724754556090911), INT64_C(-2876989220375468673),
                            INT64_C(-6983535344045017522), INT64_C(  765292938189241369),
                            INT64_C(-6953464491424673918), INT64_C( 1398420262929112913)),
      UINT8_C(249),
      simde_mm_set_epi8(INT8_C(   5), INT8_C( -82), INT8_C(  29), INT8_C( 119),
                        INT8_C( -62), INT8_C(   8), INT8_C(  50), INT8_C(  97),
                        INT8_C(  50), INT8_C( -63), INT8_C( -26), INT8_C(  60),
                        INT8_C( -88), INT8_C( -61), INT8_C( -26), INT8_C(  42)),
      simde_mm512_set_epi64(INT64_C(                  50), INT64_C(                 -63),
                            INT64_C(                 -26), INT64_C(                  60),
                            INT64_C(                 -88), INT64_C(  765292938189241369),
                            INT64_C(-6953464491424673918), INT64_C(                  42)) },
    { simde_mm512_set_epi64(INT64_C( 5132756158462079548), INT64_C(  358540499006083063),
                            INT64_C(-1371913154589519675), INT64_C( -252322179140671678),
                            INT64_C(-5093581286734302932), INT64_C(-8660577592020600985),
                            INT64_C( 2923109555544675288), INT64_C( 9023002677975531153)),
      UINT8_C(203),
      simde_mm_set_epi8(INT8_C(   4), INT8_C( -31), INT8_C( 123), INT8_C(  35),
                        INT8_C(  50), INT8_C(-124), INT8_C(  65), INT8_C(  46),
                        INT8_C(-125), INT8_C( 119), INT8_C(-126), INT8_C(  75),
                        INT8_C( -98), INT8_C( -38), INT8_C(  97), INT8_C( -14)),
      simde_mm512_set_epi64(INT64_C(                -125), INT64_C(                 119),
                            INT64_C(-1371913154589519675), INT64_C( -252322179140671678),
                            INT64_C(                 -98), INT64_C(-8660577592020600985),
                            INT64_C(                  97), INT64_C(                 -14)) },
    { simde_mm512_set_epi64(INT64_C(-2115158585831981143), INT64_C(-3036653327673284502),
                            INT64_C( 8854638534029132358), INT64_C(-8041784352684734769),
                            INT64_C( 4443329220079118277), INT64_C( 4642879860249865385),
                            INT64_C(-5149641967024239455), INT64_C(-8739960639361948315)),
      UINT8_C(177),
      simde_mm_set_epi8(INT8_C( -49), INT8_C(  25), INT8_C( -11), INT8_C(  71),
                        INT8_C(-122), INT8_C( -61), INT8_C( -57), INT8_C(  35),
                        INT8_C(-101), INT8_C(  49), INT8_C(  57), INT8_C(  71),
                        INT8_C( -75), INT8_C( 123), INT8_C(  12), INT8_C( -61)),
      simde_mm512_set_epi64(INT64_C(                -101), INT64_C(-3036653327673284502),
                            INT64_C(                  57), INT64_C(                  71),
                            INT64_C( 4443329220079118277), INT64_C( 4642879860249865385),
                            INT64_C(-5149641967024239455), INT64_C(                 -61)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_mask_cvtepi8_epi64(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_cvtepi8_epi64(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m128i a;
    simde__m512i r;
  } test_vec[8] = {
    { UINT8_C( 49),
      simde_mm_set_epi8(INT8_C( -21), INT8_C(   3), INT8_C(  48), INT8_C(-124),
                        INT8_C( -28), INT8_C( -20), INT8_C( -84), INT8_C(  62),
                        INT8_C( -29), INT8_C( -13), INT8_C( -12), INT8_C(  27),
                        INT8_C( -53), INT8_C(  77), INT8_C( -98), INT8_C(  25)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                 -12), INT64_C(                  27),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                  25)) },
    { UINT8_C(145),
      simde_mm_set_epi8(INT8_C(  40), INT8_C(   9), INT8_C(  57), INT8_C(  34),
                        INT8_C( -40), INT8_C( -30), INT8_C( -68), INT8_C( -99),
                        INT8_C( 109), INT8_C( -13), INT8_C(  36), INT8_C( 101),
                        INT8_C(  43), INT8_C(  14), INT8_C( 100), INT8_C(  83)),
      simde_mm512_set_epi64(INT64_C(                 109), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                 101),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                  83)) },
    { UINT8_C( 62),
      simde_mm_set_epi8(INT8_C(  85), INT8_C(  95), INT8_C( -67), INT8_C(  -8),
                        INT8_C( -10), INT8_C(-101), INT8_C(  50), INT8_C(-113),
                        INT8_C(  33), INT8_C(  64), INT8_C( -83), INT8_C( 121),
                        INT8_C(  24), INT8_C(  42), INT8_C( -50), INT8_C( -16)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                 -83), INT64_C(                 121),
                            INT64_C(                  24), INT64_C(                  42),
                            INT64_C(                 -50), INT64_C(                   0)) },
    { UINT8_C(104),
      simde_mm_set_epi8(INT8_C(-125), INT8_C(  49), INT8_C(  61), INT8_C(-117),
                        INT8_C( -52), INT8_C(-102), INT8_C(  15), INT8_C(  39),
                        INT8_C( 121), INT8_C(  62), INT8_C( -84), INT8_C(  58),
                        INT8_C( -34), INT8_C(  31), INT8_C( 120), INT8_C(  59)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                  62),
                            INT64_C(                 -84), INT64_C(                   0),
                            INT64_C(                 -34), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C( 59),
      simde_mm_set_epi8(INT8_C(-111), INT8_C(  87), INT8_C( -71), INT8_C( -81),
                        INT8_C(  29), INT8_C(  72), INT8_C( -63), INT8_C(  37),
                        INT8_C(  59), INT8_C(  42), INT8_C(  70), INT8_C( 105),
                        INT8_C(-116), INT8_C(  40), INT8_C(  78), INT8_C(  27)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                  70), INT64_C(                 105),
                            INT64_C(                -116), INT64_C(                   0),
                            INT64_C(                  78), INT64_C(                  27)) },
    { UINT8_C( 38),
      simde_mm_set_epi8(INT8_C(  -5), INT8_C(-116), INT8_C(-102), INT8_C( -39),
                        INT8_C( -44), INT8_C( -75), INT8_C(  59), INT8_C(   1),
                        INT8_C(  20), INT8_C(  60), INT8_C(  93), INT8_C(  82),
                        INT8_C(  58), INT8_C(  48), INT8_C( -71), INT8_C(   3)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                  93), INT64_C(                   0),
                            INT64_C(                   0), INT64_C(                  48),
                            INT64_C(                 -71), INT64_C(                   0)) },
    { UINT8_C(252),
      simde_mm_set_epi8(INT8_C( -75), INT8_C(  48), INT8_C(  25), INT8_C( 103),
                        INT8_C(  50), INT8_C(  70), INT8_C( -74), INT8_C(  80),
                        INT8_C(  93), INT8_C( 120), INT8_C(-120), INT8_C( -31),
                        INT8_C(  66), INT8_C( 105), INT8_C( -92), INT8_C( -65)),
      simde_mm512_set_epi64(INT64_C(                  93), INT64_C(                 120),
                            INT64_C(                -120), INT64_C(                 -31),
                            INT64_C(                  66), INT64_C(                 105),
                            INT64_C(                   0), INT64_C(                   0)) },
    { UINT8_C(114),
      simde_mm_set_epi8(INT8_C(-128), INT8_C(   9), INT8_C( -96), INT8_C(  -4),
                        INT8_C( -38), INT8_C(-121), INT8_C( 109), INT8_C(  -8),
                        INT8_C(-101), INT8_C( -12), INT8_C( -65), INT8_C(  83),
                        INT8_C( -45), INT8_C(  60), INT8_C( -98), INT8_C( -16)),
      simde_mm512_set_epi64(INT64_C(                   0), INT64_C(                 -12),
                            INT64_C(                 -65), INT64_C(                  83),
                            INT64_C(                   0), INT64_C(                   0),
                            INT64_C(                 -98), INT64_C(                   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m512i r = simde_mm512_maskz_cvtepi8_epi64(test_vec[i].k, test_vec[i].a);
    simde_assert_m512i_i64(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cvtepi32_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(         -2), INT32_C(         -4), INT32_C( -120451969), INT32_C(      14509),
                            INT32_C(      -1510), INT32_C(      98804), INT32_C(       1802), INT32_C(     -32352),
                            INT32_C(      14540), INT32_C(         -2), INT32_C(        222), INT32_C(        152),
                            INT32_C(     -48720), INT32_C(     250746), INT32_C(         15), INT32_C(         -2)),
      simde_mm_set_epi8(INT8_C(  -2), INT8_C(  -4), INT8_C( 127), INT8_C( -83),
                        INT8_C(  26), INT8_C( -12), INT8_C(  10), INT8_C( -96),
                        INT8_C( -52), INT8_C(  -2), INT8_C( -34), INT8_C(-104),
                        INT8_C( -80), INT8_C( 122), INT8_C(  15), INT8_C(  -2)) },
    { simde_mm512_set_epi32(INT32_C(   -2537924), INT32_C(          0), INT32_C(       3842), INT32_C(    -439330),
                            INT32_C(      39001), INT32_C(         -1), INT32_C(   32480192), INT32_C(          0),
                            INT32_C(          4), INT32_C(  -11661865), INT32_C(          0), INT32_C(          2),
                            INT32_C(      63322), INT32_C(        -33), INT32_C(      14448), INT32_C(          2)),
      simde_mm_set_epi8(INT8_C(  60), INT8_C(   0), INT8_C(   2), INT8_C( -34),
                        INT8_C(  89), INT8_C(  -1), INT8_C( -64), INT8_C(   0),
                        INT8_C(   4), INT8_C( -41), INT8_C(   0), INT8_C(   2),
                        INT8_C(  90), INT8_C( -33), INT8_C( 112), INT8_C(   2)) },
    { simde_mm512_set_epi32(INT32_C(        -18), INT32_C(   -2011647), INT32_C(   -7768922), INT32_C( -921783558),
                            INT32_C(   -1941821), INT32_C(          0), INT32_C(    -647690), INT32_C(      -5119),
                            INT32_C(         -1), INT32_C(        343), INT32_C(  113610714), INT32_C(          3),
                            INT32_C(      38353), INT32_C(        246), INT32_C(  -10559231), INT32_C(       8543)),
      simde_mm_set_epi8(INT8_C( -18), INT8_C(   1), INT8_C( -90), INT8_C(  -6),
                        INT8_C( -61), INT8_C(   0), INT8_C( -10), INT8_C(   1),
                        INT8_C(  -1), INT8_C(  87), INT8_C( -38), INT8_C(   3),
                        INT8_C( -47), INT8_C( -10), INT8_C(   1), INT8_C(  95)) },
    { simde_mm512_set_epi32(INT32_C( -177879544), INT32_C(         -4), INT32_C(         -1), INT32_C(      -1874),
                            INT32_C(         -4), INT32_C( -469598096), INT32_C(      -1647), INT32_C(     129553),
                            INT32_C(    9710669), INT32_C(          3), INT32_C(    1473856), INT32_C(  134714256),
                            INT32_C(   70527996), INT32_C(    2339642), INT32_C(     148218), INT32_C(  -32767248)),
      simde_mm_set_epi8(INT8_C(   8), INT8_C(  -4), INT8_C(  -1), INT8_C( -82),
                        INT8_C(  -4), INT8_C( 112), INT8_C(-111), INT8_C(  17),
                        INT8_C(  77), INT8_C(   3), INT8_C(  64), INT8_C(-112),
                        INT8_C(  -4), INT8_C(  58), INT8_C(  -6), INT8_C( -16)) },
    { simde_mm512_set_epi32(INT32_C(   -1418204), INT32_C(    -122943), INT32_C(     799821), INT32_C(        -51),
                            INT32_C(     207931), INT32_C(  -11778782), INT32_C(      66993), INT32_C(     -15812),
                            INT32_C(   55345677), INT32_C( -194873886), INT32_C(      -3955), INT32_C(        -22),
                            INT32_C(       1761), INT32_C(        751), INT32_C(         19), INT32_C(         -4)),
      simde_mm_set_epi8(INT8_C(  36), INT8_C( -63), INT8_C(  77), INT8_C( -51),
                        INT8_C(  59), INT8_C(  34), INT8_C( -79), INT8_C(  60),
                        INT8_C(  13), INT8_C( -30), INT8_C(-115), INT8_C( -22),
                        INT8_C( -31), INT8_C( -17), INT8_C(  19), INT8_C(  -4)) },
    { simde_mm512_set_epi32(INT32_C(        -17), INT32_C(        -26), INT32_C(     854285), INT32_C(         51),
                            INT32_C(  -60746537), INT32_C(    3687234), INT32_C(   52848365), INT32_C(   26958727),
                            INT32_C(          2), INT32_C(        104), INT32_C(    4725058), INT32_C(     -56297),
                            INT32_C(       5336), INT32_C(     443041), INT32_C(        -35), INT32_C(     229612)),
      simde_mm_set_epi8(INT8_C( -17), INT8_C( -26), INT8_C(  13), INT8_C(  51),
                        INT8_C( -41), INT8_C(  66), INT8_C( -19), INT8_C(-121),
                        INT8_C(   2), INT8_C( 104), INT8_C(  66), INT8_C(  23),
                        INT8_C( -40), INT8_C( -95), INT8_C( -35), INT8_C( -20)) },
    { simde_mm512_set_epi32(INT32_C(        -27), INT32_C(  127397292), INT32_C(   29325489), INT32_C(        691),
                            INT32_C(       -978), INT32_C(    -559751), INT32_C(   -3037707), INT32_C(     189833),
                            INT32_C(         11), INT32_C(   -4085970), INT32_C(      -3499), INT32_C(     -16323),
                            INT32_C(     732682), INT32_C(     108115), INT32_C(   29565452), INT32_C( -145574324)),
      simde_mm_set_epi8(INT8_C( -27), INT8_C( -84), INT8_C( -79), INT8_C( -77),
                        INT8_C(  46), INT8_C( 121), INT8_C( -11), INT8_C(-119),
                        INT8_C(  11), INT8_C(  46), INT8_C(  85), INT8_C(  61),
                        INT8_C(  10), INT8_C(  83), INT8_C(  12), INT8_C(  76)) },
    { simde_mm512_set_epi32(INT32_C(        -14), INT32_C(    6208981), INT32_C(  133763173), INT32_C(  -30227251),
                            INT32_C(  -17898651), INT32_C( -197203605), INT32_C(         -4), INT32_C(         13),
                            INT32_C(   -1312564), INT32_C(         -3), INT32_C(    5632807), INT32_C(       2549),
                            INT32_C(         -3), INT32_C(      -2772), INT32_C(      -1504), INT32_C(          1)),
      simde_mm_set_epi8(INT8_C( -14), INT8_C( -43), INT8_C( 101), INT8_C( -51),
                        INT8_C( 101), INT8_C( 107), INT8_C(  -4), INT8_C(  13),
                        INT8_C( -52), INT8_C(  -3), INT8_C(  39), INT8_C( -11),
                        INT8_C(  -3), INT8_C(  44), INT8_C(  32), INT8_C(   1)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_cvtepi32_epi8(test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cvtepi32_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C(-115), INT8_C(  38), INT8_C( -52), INT8_C( -72),
                        INT8_C( -15), INT8_C( -73), INT8_C(  99), INT8_C(  21),
                        INT8_C(  43), INT8_C( -43), INT8_C( -79), INT8_C( -14),
                        INT8_C( -64), INT8_C(  13), INT8_C( -52), INT8_C(  21)),
      UINT16_C(59249),
      simde_mm512_set_epi32(INT32_C( -633053809), INT32_C( 1045231202), INT32_C( 1582941645), INT32_C(  118733775),
                            INT32_C( -376070904), INT32_C(-1346059078), INT32_C(-1785775774), INT32_C( 1344364263),
                            INT32_C(-1284394530), INT32_C(  845981746), INT32_C( 1995633378), INT32_C(   86515524),
                            INT32_C( 1684133172), INT32_C(  134426622), INT32_C( 1324231526), INT32_C(-1677357037)),
      simde_mm_set_epi8(INT8_C(-113), INT8_C(  98), INT8_C( -51), INT8_C( -72),
                        INT8_C( -15), INT8_C( -70), INT8_C(  98), INT8_C( -25),
                        INT8_C(  43), INT8_C(  50), INT8_C( -30), INT8_C(  68),
                        INT8_C( -64), INT8_C(  13), INT8_C( -52), INT8_C(  19)) },
    { simde_mm_set_epi8(INT8_C(  11), INT8_C( -83), INT8_C(-118), INT8_C( -45),
                        INT8_C( 127), INT8_C( -97), INT8_C(  58), INT8_C( -28),
                        INT8_C( 119), INT8_C( 125), INT8_C( 108), INT8_C( 121),
                        INT8_C( -56), INT8_C(  31), INT8_C( 101), INT8_C(  30)),
      UINT16_C(15281),
      simde_mm512_set_epi32(INT32_C(-1863151852), INT32_C( 1841626176), INT32_C(  128694508), INT32_C(-1952059086),
                            INT32_C(   35559102), INT32_C(  170089620), INT32_C(  592079099), INT32_C(  986767838),
                            INT32_C(-2078459331), INT32_C(-1259484799), INT32_C( -745194505), INT32_C( -806534005),
                            INT32_C(  178571728), INT32_C(  724303028), INT32_C(-1558698351), INT32_C( 1087124862)),
      simde_mm_set_epi8(INT8_C(  11), INT8_C( -83), INT8_C( -20), INT8_C(  50),
                        INT8_C( -66), INT8_C( -97), INT8_C(  -5), INT8_C( -34),
                        INT8_C(  61), INT8_C( 125), INT8_C(  -9), INT8_C(-117),
                        INT8_C( -56), INT8_C(  31), INT8_C( 101), INT8_C( 126)) },
    { simde_mm_set_epi8(INT8_C( -37), INT8_C(  17), INT8_C( -46), INT8_C(  62),
                        INT8_C(-122), INT8_C( -48), INT8_C(  95), INT8_C(  94),
                        INT8_C(-120), INT8_C(  70), INT8_C(  86), INT8_C( -34),
                        INT8_C( 103), INT8_C(-121), INT8_C(-125), INT8_C( -37)),
      UINT16_C(60508),
      simde_mm512_set_epi32(INT32_C(  785975787), INT32_C(-1640844720), INT32_C(  801487669), INT32_C( -230277812),
                            INT32_C( 1785836931), INT32_C(-1252913343), INT32_C( -593649464), INT32_C(  826028222),
                            INT32_C( 1348935036), INT32_C( 1700017282), INT32_C(  975497373), INT32_C(  680935328),
                            INT32_C(  926666753), INT32_C(-1896106060), INT32_C(  828740304), INT32_C(  854544945)),
      simde_mm_set_epi8(INT8_C( -21), INT8_C(  80), INT8_C(  53), INT8_C(  62),
                        INT8_C(-125), INT8_C(  65), INT8_C(  95), INT8_C(  94),
                        INT8_C(-120), INT8_C(-126), INT8_C(  86), INT8_C( -96),
                        INT8_C(   1), INT8_C( -76), INT8_C(-125), INT8_C( -37)) },
    { simde_mm_set_epi8(INT8_C(  10), INT8_C(  72), INT8_C(  14), INT8_C(  71),
                        INT8_C(  38), INT8_C( 127), INT8_C(  12), INT8_C( 109),
                        INT8_C(  37), INT8_C(   0), INT8_C(-100), INT8_C( 103),
                        INT8_C(  94), INT8_C(-124), INT8_C(  98), INT8_C( 121)),
      UINT16_C(27626),
      simde_mm512_set_epi32(INT32_C(-2129288013), INT32_C( -640424446), INT32_C( -475582201), INT32_C(-1084971703),
                            INT32_C(-1025326436), INT32_C(  476614498), INT32_C(-2026838640), INT32_C( 1547453534),
                            INT32_C(  608354747), INT32_C( -419005782), INT32_C( -987109050), INT32_C(-1602281775),
                            INT32_C(-1047324466), INT32_C( 1650730666), INT32_C(-1634562157), INT32_C( 1609673966)),
      simde_mm_set_epi8(INT8_C(  10), INT8_C(   2), INT8_C(   7), INT8_C(  71),
                        INT8_C(-100), INT8_C( 127), INT8_C(-112), INT8_C(  94),
                        INT8_C( -69), INT8_C( -86), INT8_C(  70), INT8_C( 103),
                        INT8_C( -50), INT8_C(-124), INT8_C(-109), INT8_C( 121)) },
    { simde_mm_set_epi8(INT8_C(-121), INT8_C(  49), INT8_C( -49), INT8_C(-115),
                        INT8_C(  46), INT8_C(  78), INT8_C( -24), INT8_C( 108),
                        INT8_C(   6), INT8_C( -70), INT8_C( -66), INT8_C( -81),
                        INT8_C( -83), INT8_C(-127), INT8_C( -30), INT8_C( -93)),
      UINT16_C(65313),
      simde_mm512_set_epi32(INT32_C( -866938802), INT32_C(-1684051229), INT32_C( 1750567699), INT32_C( -890678355),
                            INT32_C( 1248284911), INT32_C(  685959243), INT32_C(-1971795404), INT32_C( -380779864),
                            INT32_C(  768594520), INT32_C( 1934584708), INT32_C( 1771009293), INT32_C(  977292129),
                            INT32_C(  116806954), INT32_C( 1249795808), INT32_C( 1848907251), INT32_C(-1141949480)),
      simde_mm_set_epi8(INT8_C(  78), INT8_C( -29), INT8_C(  19), INT8_C( -83),
                        INT8_C( -17), INT8_C(  75), INT8_C(  52), INT8_C( -88),
                        INT8_C(   6), INT8_C( -70), INT8_C(  13), INT8_C( -81),
                        INT8_C( -83), INT8_C(-127), INT8_C( -30), INT8_C( -40)) },
    { simde_mm_set_epi8(INT8_C( -22), INT8_C( 123), INT8_C(-115), INT8_C(-106),
                        INT8_C( -17), INT8_C(  46), INT8_C(-125), INT8_C( -54),
                        INT8_C( 109), INT8_C( -45), INT8_C(  71), INT8_C(  92),
                        INT8_C(   4), INT8_C(  62), INT8_C( 125), INT8_C(  87)),
      UINT16_C(61443),
      simde_mm512_set_epi32(INT32_C(-1334588139), INT32_C(-1182979414), INT32_C(  652824334), INT32_C(-1886834903),
                            INT32_C(-2077275827), INT32_C(-2124149498), INT32_C( 1711430883), INT32_C( 1063836903),
                            INT32_C(  196832116), INT32_C( 1645984078), INT32_C( -706180494), INT32_C(-2114013608),
                            INT32_C( -409538379), INT32_C( -238667028), INT32_C( -203770260), INT32_C(  668929896)),
      simde_mm_set_epi8(INT8_C(  21), INT8_C( -86), INT8_C(  14), INT8_C(  41),
                        INT8_C( -17), INT8_C(  46), INT8_C(-125), INT8_C( -54),
                        INT8_C( 109), INT8_C( -45), INT8_C(  71), INT8_C(  92),
                        INT8_C(   4), INT8_C(  62), INT8_C( 108), INT8_C( 104)) },
    { simde_mm_set_epi8(INT8_C( -86), INT8_C(  70), INT8_C(  52), INT8_C( 118),
                        INT8_C(   9), INT8_C(  39), INT8_C( 114), INT8_C( -16),
                        INT8_C( 117), INT8_C(  23), INT8_C(  56), INT8_C( 116),
                        INT8_C(-103), INT8_C( -20), INT8_C( -54), INT8_C( -40)),
      UINT16_C(17932),
      simde_mm512_set_epi32(INT32_C(-1632449466), INT32_C( -201574937), INT32_C( -969151873), INT32_C( 1717523001),
                            INT32_C(  328447052), INT32_C( -807008823), INT32_C(-1137627317), INT32_C(  257941943),
                            INT32_C( -575132239), INT32_C(-1243310715), INT32_C( 1780403357), INT32_C(-1221616339),
                            INT32_C(-1111710057), INT32_C( -345601306), INT32_C( 1878290192), INT32_C( 1400816063)),
      simde_mm_set_epi8(INT8_C( -86), INT8_C( -25), INT8_C(  52), INT8_C( 118),
                        INT8_C(   9), INT8_C( -55), INT8_C(  75), INT8_C( -16),
                        INT8_C( 117), INT8_C(  23), INT8_C(  56), INT8_C( 116),
                        INT8_C(-105), INT8_C( -26), INT8_C( -54), INT8_C( -40)) },
    { simde_mm_set_epi8(INT8_C(-111), INT8_C(-107), INT8_C( -18), INT8_C( -11),
                        INT8_C( -19), INT8_C(  72), INT8_C( -61), INT8_C( -36),
                        INT8_C( 112), INT8_C( -89), INT8_C(  40), INT8_C( -85),
                        INT8_C( -85), INT8_C( -24), INT8_C(   1), INT8_C( 118)),
      UINT16_C( 5127),
      simde_mm512_set_epi32(INT32_C( 1198543284), INT32_C( -953548547), INT32_C(-1654400435), INT32_C(-1584606457),
                            INT32_C(-1412039433), INT32_C(  978591462), INT32_C(  932668344), INT32_C(    4811681),
                            INT32_C( 1470607276), INT32_C( -870107132), INT32_C(-1548582602), INT32_C(-1629317483),
                            INT32_C( 1240536520), INT32_C( 1712711286), INT32_C(  -33733649), INT32_C( -824647896)),
      simde_mm_set_epi8(INT8_C(-111), INT8_C(-107), INT8_C( -18), INT8_C(   7),
                        INT8_C( -19), INT8_C( -26), INT8_C( -61), INT8_C( -36),
                        INT8_C( 112), INT8_C( -89), INT8_C(  40), INT8_C( -85),
                        INT8_C( -85), INT8_C( 118), INT8_C( -17), INT8_C(  40)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_mask_cvtepi32_epi8(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_cvtepi32_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { UINT16_C(30170),
      simde_mm512_set_epi32(INT32_C(-1490536007), INT32_C( 1066535041), INT32_C( -550061645), INT32_C(-1270000019),
                            INT32_C( 1151371971), INT32_C(-1698501418), INT32_C(  316796751), INT32_C( 1981586156),
                            INT32_C(-1946547448), INT32_C( 1308377999), INT32_C( -844439772), INT32_C(   66448910),
                            INT32_C( -858584361), INT32_C(  937214420), INT32_C(-1606259759), INT32_C(-1809752141)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(-127), INT8_C( -77), INT8_C( 109),
                        INT8_C(   0), INT8_C( -42), INT8_C(   0), INT8_C( -20),
                        INT8_C(   8), INT8_C(-113), INT8_C(   0), INT8_C(  14),
                        INT8_C( -41), INT8_C(   0), INT8_C( -47), INT8_C(   0)) },
    { UINT16_C(62388),
      simde_mm512_set_epi32(INT32_C( -211999323), INT32_C(-1844444789), INT32_C(-1795946402), INT32_C( 1448672032),
                            INT32_C( -934434797), INT32_C(-2083148260), INT32_C(-1634027507), INT32_C(-1634157891),
                            INT32_C(  293018136), INT32_C( -844643479), INT32_C(-1645027193), INT32_C(-1269262821),
                            INT32_C(  471304631), INT32_C(-1979771737), INT32_C( 1002540732), INT32_C( -224819409)),
      simde_mm_set_epi8(INT8_C( -91), INT8_C(-117), INT8_C(  94), INT8_C(  32),
                        INT8_C(   0), INT8_C(   0), INT8_C(  13), INT8_C( -67),
                        INT8_C(  24), INT8_C(   0), INT8_C(-121), INT8_C(  27),
                        INT8_C(   0), INT8_C( -89), INT8_C(   0), INT8_C(   0)) },
    { UINT16_C(62171),
      simde_mm512_set_epi32(INT32_C(   52699594), INT32_C(  488650932), INT32_C( 1760732830), INT32_C( 2055656791),
                            INT32_C(  901653669), INT32_C(-1110809377), INT32_C( -921351138), INT32_C(-1112269588),
                            INT32_C( -380695850), INT32_C(-1321417051), INT32_C( -488851354), INT32_C( 1504345267),
                            INT32_C( -892751803), INT32_C( 1370474856), INT32_C(-1824202240), INT32_C( 1071081537)),
      simde_mm_set_epi8(INT8_C( -54), INT8_C( -76), INT8_C( -98), INT8_C(  87),
                        INT8_C(   0), INT8_C(   0), INT8_C(  30), INT8_C(   0),
                        INT8_C( -42), INT8_C( -91), INT8_C(   0), INT8_C( -77),
                        INT8_C(  69), INT8_C(   0), INT8_C(   0), INT8_C(  65)) },
    { UINT16_C(25207),
      simde_mm512_set_epi32(INT32_C(-1432829131), INT32_C(  749437314), INT32_C( 2054705796), INT32_C( -391985181),
                            INT32_C( 1552532172), INT32_C(-1395437371), INT32_C( 2050638296), INT32_C(  899300612),
                            INT32_C( -439013837), INT32_C( 1227975380), INT32_C( 1163662343), INT32_C( 1510671711),
                            INT32_C(  284422502), INT32_C( -115052064), INT32_C(-1371764485), INT32_C( 1680575393)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(-126), INT8_C(-124), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C( -40), INT8_C(   0),
                        INT8_C(   0), INT8_C( -44), INT8_C(   7), INT8_C(  95),
                        INT8_C(   0), INT8_C( -32), INT8_C(  -5), INT8_C( -95)) },
    { UINT16_C(24449),
      simde_mm512_set_epi32(INT32_C( 1421206868), INT32_C( 2098367545), INT32_C(-1824932601), INT32_C( -645288155),
                            INT32_C(-1709048480), INT32_C( -754994889), INT32_C( 1910543063), INT32_C( 1017349806),
                            INT32_C(-1269450519), INT32_C( -918081531), INT32_C( -891095185), INT32_C(  463720262),
                            INT32_C(-1217679150), INT32_C(   15502381), INT32_C( 1778046630), INT32_C( 1937097954)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(  57), INT8_C(   0), INT8_C(  37),
                        INT8_C(  96), INT8_C(  55), INT8_C( -41), INT8_C( -82),
                        INT8_C( -23), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C( -30)) },
    { UINT16_C(29387),
      simde_mm512_set_epi32(INT32_C( 1987003656), INT32_C( 1660112052), INT32_C(  352562636), INT32_C(-1628125955),
                            INT32_C( -306345670), INT32_C(-1188776876), INT32_C( 1380454403), INT32_C(-1671272217),
                            INT32_C(   81322544), INT32_C(-1759657246), INT32_C(  996228951), INT32_C(-1018298216),
                            INT32_C(  368633517), INT32_C(   88316290), INT32_C( 1098979766), INT32_C( -528467658)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C( -76), INT8_C( -52), INT8_C(  -3),
                        INT8_C(   0), INT8_C(   0), INT8_C(   3), INT8_C(   0),
                        INT8_C(  48), INT8_C( -30), INT8_C(   0), INT8_C(   0),
                        INT8_C( -83), INT8_C(   0), INT8_C( -74), INT8_C(  54)) },
    { UINT16_C( 4173),
      simde_mm512_set_epi32(INT32_C( -494557689), INT32_C( 1662324959), INT32_C( 1347000947), INT32_C( -162261965),
                            INT32_C(-1860865781), INT32_C(-1254000245), INT32_C(  -64373227), INT32_C( -128416414),
                            INT32_C(  679962295), INT32_C( -855169599), INT32_C(-1117560264), INT32_C(-1737150948),
                            INT32_C( -703356518), INT32_C(-1187698017), INT32_C(  283483033), INT32_C(  528640345)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(  51),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C( -63), INT8_C(   0), INT8_C(   0),
                        INT8_C(-102), INT8_C( -97), INT8_C(   0), INT8_C(  89)) },
    { UINT16_C(42283),
      simde_mm512_set_epi32(INT32_C( -988318647), INT32_C(-1013856804), INT32_C( -265791763), INT32_C(  863100255),
                            INT32_C( 1939230730), INT32_C( -592540076), INT32_C( -177529906), INT32_C(  212115003),
                            INT32_C( -240452355), INT32_C(-2076064537), INT32_C(-1929718803), INT32_C( -295261474),
                            INT32_C( -375082091), INT32_C(-1476895591), INT32_C(  282049218), INT32_C( 1800963183)),
      simde_mm_set_epi8(INT8_C(  73), INT8_C(   0), INT8_C( -19), INT8_C(   0),
                        INT8_C(   0), INT8_C(  84), INT8_C(   0), INT8_C(  59),
                        INT8_C(   0), INT8_C(   0), INT8_C( -19), INT8_C(   0),
                        INT8_C(-107), INT8_C(   0), INT8_C( -62), INT8_C( 111)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_maskz_cvtepi32_epi8(test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cvtepi32_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(-1332198505), INT32_C(  940882866), INT32_C(-1991926811), INT32_C(  366788767),
                            INT32_C(-1143068695), INT32_C(-1032919886), INT32_C(-2128096317), INT32_C( -268982741),
                            INT32_C(-2068381358), INT32_C(  616587314), INT32_C( -881395152), INT32_C( -231234031),
                            INT32_C( -227030663), INT32_C(-2069337736), INT32_C( -657801206), INT32_C( -481592217)),
      simde_mm256_set_epi16(INT16_C( 17303), INT16_C(-17486), INT16_C(-25627), INT16_C(-16225),
                            INT16_C( 10217), INT16_C( -6990), INT16_C(-11325), INT16_C(-22997),
                            INT16_C(   338), INT16_C( 24626), INT16_C( -1488), INT16_C(-23023),
                            INT16_C(-13959), INT16_C( 27000), INT16_C(-16374), INT16_C( 31847)) },
    { simde_mm512_set_epi32(INT32_C(-2105440453), INT32_C( -920655637), INT32_C(-1308443450), INT32_C( 1530912720),
                            INT32_C(  327449420), INT32_C(-1587559441), INT32_C( 1636779725), INT32_C( 1337676952),
                            INT32_C(  437538013), INT32_C( -547064950), INT32_C(-1989988035), INT32_C( 1891070193),
                            INT32_C(  268816178), INT32_C(-2114746214), INT32_C(-1830004348), INT32_C( -326386089)),
      simde_mm256_set_epi16(INT16_C(-30917), INT16_C( -5909), INT16_C(-17210), INT16_C( -8240),
                            INT16_C( 31564), INT16_C(-15377), INT16_C( 18125), INT16_C( 21656),
                            INT16_C( 19677), INT16_C( 29578), INT16_C( 12605), INT16_C( 28913),
                            INT16_C(-12494), INT16_C(-30566), INT16_C( 22916), INT16_C(-16809)) },
    { simde_mm512_set_epi32(INT32_C(   65778579), INT32_C(-1223552012), INT32_C( 2126519569), INT32_C( -313585586),
                            INT32_C(  176014611), INT32_C( -579188851), INT32_C( 1255563377), INT32_C( 2103012622),
                            INT32_C(  632398769), INT32_C( -371050877), INT32_C( 1353038446), INT32_C(  825386533),
                            INT32_C(-1874850751), INT32_C(-1002389366), INT32_C(  190340527), INT32_C(-1797342038)),
      simde_mm256_set_epi16(INT16_C(-19565), INT16_C(  5108), INT16_C(  7441), INT16_C(  4174),
                            INT16_C(-15085), INT16_C( 18317), INT16_C( 24689), INT16_C( 27918),
                            INT16_C(-23631), INT16_C( 13955), INT16_C(-17810), INT16_C( 26149),
                            INT16_C(  3137), INT16_C(-16246), INT16_C( 23983), INT16_C(-17238)) },
    { simde_mm512_set_epi32(INT32_C( 1791900902), INT32_C(-1811007005), INT32_C(-1697094173), INT32_C(  380010296),
                            INT32_C(-1383570112), INT32_C( 1657316314), INT32_C(  607692203), INT32_C( 1114614249),
                            INT32_C(-2038136105), INT32_C( -866829407), INT32_C(  438125461), INT32_C(-1743999841),
                            INT32_C( 1594785360), INT32_C( 1281130339), INT32_C( -934532021), INT32_C(  860713686)),
      simde_mm256_set_epi16(INT16_C( 15590), INT16_C( 14819), INT16_C( 26083), INT16_C( 32568),
                            INT16_C( 25920), INT16_C(-23590), INT16_C(-23125), INT16_C(-22039),
                            INT16_C(-32041), INT16_C( 15265), INT16_C( 17301), INT16_C(-21345),
                            INT16_C( 32336), INT16_C( 32611), INT16_C( 11339), INT16_C( 29398)) },
    { simde_mm512_set_epi32(INT32_C( 2051345376), INT32_C( 1984352737), INT32_C( -556704403), INT32_C(-1837075026),
                            INT32_C(-1368462428), INT32_C(-1008395565), INT32_C( -364003348), INT32_C( 1194517560),
                            INT32_C( -724894707), INT32_C( 1417788368), INT32_C(  813255612), INT32_C(-1030021276),
                            INT32_C(  619302066), INT32_C( -403277800), INT32_C(-1431725263), INT32_C( -983683884)),
      simde_mm256_set_epi16(INT16_C(  3040), INT16_C(-11807), INT16_C( 23917), INT16_C( 30126),
                            INT16_C( -5212), INT16_C(  6867), INT16_C(-16404), INT16_C( -7112),
                            INT16_C( -1011), INT16_C(-17456), INT16_C( 19388), INT16_C(  8036),
                            INT16_C(-13134), INT16_C( 30744), INT16_C(-25807), INT16_C( 11476)) },
    { simde_mm512_set_epi32(INT32_C(-1122798844), INT32_C(-1762786870), INT32_C( -798997093), INT32_C(-1700624092),
                            INT32_C(  274386357), INT32_C(-1124458887), INT32_C(-1997652654), INT32_C( -871223206),
                            INT32_C(-1283482560), INT32_C(-1963416707), INT32_C( 1626011520), INT32_C(-1795267365),
                            INT32_C( 1814240343), INT32_C(  563972607), INT32_C(-1989715395), INT32_C( 1806536177)),
      simde_mm256_set_epi16(INT16_C( 29444), INT16_C(   458), INT16_C( 17819), INT16_C(-30428),
                            INT16_C(-12875), INT16_C(  7801), INT16_C( 15698), INT16_C( 12378),
                            INT16_C(-25536), INT16_C(-23683), INT16_C( -2176), INT16_C( 25819),
                            INT16_C(  7255), INT16_C(-30209), INT16_C( 23101), INT16_C(-29199)) },
    { simde_mm512_set_epi32(INT32_C(  630973837), INT32_C(  113536673), INT32_C( -518601343), INT32_C(-1286957458),
                            INT32_C( 1161343316), INT32_C(-1166902161), INT32_C( -814068470), INT32_C(  979467752),
                            INT32_C(-1209279266), INT32_C(  299486195), INT32_C(-1121450290), INT32_C(-1207879975),
                            INT32_C(-1157080602), INT32_C( 1406950891), INT32_C(-1364074288), INT32_C(  597374251)),
      simde_mm256_set_epi16(INT16_C( -6771), INT16_C( 28321), INT16_C(-14975), INT16_C(-27026),
                            INT16_C(-20140), INT16_C( 31855), INT16_C( 19722), INT16_C( 32232),
                            INT16_C( -8994), INT16_C(-13325), INT16_C(  1742), INT16_C( 14041),
                            INT16_C( 23014), INT16_C( 24043), INT16_C( -7984), INT16_C( 13611)) },
    { simde_mm512_set_epi32(INT32_C(-1559893444), INT32_C(-1654530433), INT32_C(  517502824), INT32_C( -829443487),
                            INT32_C( 1010937703), INT32_C(  846882690), INT32_C( -969138942), INT32_C( -844620432),
                            INT32_C(-1194851807), INT32_C( -113524179), INT32_C(-1017791786), INT32_C(  -30012988),
                            INT32_C(-1341538338), INT32_C( -898929280), INT32_C( 1716675353), INT32_C( -910502857)),
      simde_mm256_set_epi16(INT16_C( -5572), INT16_C( -8577), INT16_C( 30568), INT16_C(-19871),
                            INT16_C(-20633), INT16_C( 26498), INT16_C(  7426), INT16_C(  7536),
                            INT16_C(   545), INT16_C(-15827), INT16_C(-17706), INT16_C(  2500),
                            INT16_C(-16418), INT16_C( 28032), INT16_C( 25369), INT16_C(-11209)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_cvtepi32_epi16(test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cvtepi32_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C(-24546), INT16_C( -2793), INT16_C( 30898), INT16_C(-16727),
                            INT16_C(  8944), INT16_C(-21339), INT16_C( 15242), INT16_C( 19130),
                            INT16_C(  6628), INT16_C(   311), INT16_C( 10740), INT16_C(-29133),
                            INT16_C(-29367), INT16_C(  5394), INT16_C( -3185), INT16_C(  6093)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C( -192240909), INT32_C( 1754400453), INT32_C( 2092363074), INT32_C( -216902206),
                            INT32_C( 1710225239), INT32_C(-1888469291), INT32_C( 1486952867), INT32_C( -799781754),
                            INT32_C(-1362821820), INT32_C(-1788704779), INT32_C(-1168048073), INT32_C(  865982914),
                            INT32_C(-1536735064), INT32_C( 1323733132), INT32_C(-1675904112), INT32_C( -962936689)),
      simde_mm256_set_epi16(INT16_C(-24546), INT16_C( -2793), INT16_C( 30898), INT16_C(-16727),
                            INT16_C(  8944), INT16_C(-21339), INT16_C( 15242), INT16_C( 19130),
                            INT16_C(  6628), INT16_C(   311), INT16_C( 10740), INT16_C(-29133),
                            INT16_C(-29367), INT16_C(  5394), INT16_C( -3185), INT16_C(  6093)) },
    { simde_mm256_set_epi16(INT16_C(   479), INT16_C( 20906), INT16_C( 31339), INT16_C(-25789),
                            INT16_C( 26024), INT16_C( 23257), INT16_C( 14038), INT16_C(-13152),
                            INT16_C(  6837), INT16_C( -4529), INT16_C(-16234), INT16_C(-29803),
                            INT16_C(   345), INT16_C(  3529), INT16_C( -4208), INT16_C(-14200)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(-1401550594), INT32_C( -392755156), INT32_C(-1796819596), INT32_C(-1026487242),
                            INT32_C(  635705585), INT32_C( 1947566486), INT32_C( 1579971968), INT32_C(  671552800),
                            INT32_C(  -53163704), INT32_C(   93283209), INT32_C( 1507494084), INT32_C( 2043988513),
                            INT32_C( 1862942047), INT32_C( -239955513), INT32_C( -505978631), INT32_C( 1742275249)),
      simde_mm256_set_epi16(INT16_C(   479), INT16_C( 20906), INT16_C( 31339), INT16_C(-25789),
                            INT16_C( 26024), INT16_C( 23257), INT16_C( 14038), INT16_C(-13152),
                            INT16_C(  6837), INT16_C( -4529), INT16_C(-16234), INT16_C(-29803),
                            INT16_C(   345), INT16_C(  3529), INT16_C( -4208), INT16_C(-14200)) },
    { simde_mm256_set_epi16(INT16_C( -3427), INT16_C( 17836), INT16_C(  7353), INT16_C(-29365),
                            INT16_C( -9227), INT16_C( 28467), INT16_C(-13660), INT16_C( 28564),
                            INT16_C(  3740), INT16_C( 26153), INT16_C(-27156), INT16_C(  8245),
                            INT16_C( 28488), INT16_C( -5847), INT16_C(-18032), INT16_C(-28087)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(  484462136), INT32_C(-1869216478), INT32_C(-1224423681), INT32_C(  259010490),
                            INT32_C(  271695042), INT32_C( -508100150), INT32_C(  326629519), INT32_C(-2075728301),
                            INT32_C( 1430269846), INT32_C( -557150729), INT32_C(-1603145004), INT32_C(-1926846046),
                            INT32_C( 1515815293), INT32_C(-1074983379), INT32_C(-1337974349), INT32_C( 1536893374)),
      simde_mm256_set_epi16(INT16_C( -3427), INT16_C( 17836), INT16_C(  7353), INT16_C(-29365),
                            INT16_C( -9227), INT16_C( 28467), INT16_C(-13660), INT16_C( 28564),
                            INT16_C(  3740), INT16_C( 26153), INT16_C(-27156), INT16_C(  8245),
                            INT16_C( 28488), INT16_C( -5847), INT16_C(-18032), INT16_C(-28087)) },
    { simde_mm256_set_epi16(INT16_C( 11942), INT16_C(  3200), INT16_C( 26207), INT16_C( 28891),
                            INT16_C(-21120), INT16_C( 23935), INT16_C( 23697), INT16_C( -6686),
                            INT16_C(  4973), INT16_C(-12874), INT16_C( 21523), INT16_C( 18403),
                            INT16_C(-30279), INT16_C( 22300), INT16_C(-10277), INT16_C( 31923)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(   99021516), INT32_C( -725988672), INT32_C(-1021115053), INT32_C( 1709403708),
                            INT32_C(   -4910667), INT32_C( 1604667568), INT32_C(-1909621033), INT32_C( 1075339790),
                            INT32_C( -834463428), INT32_C(-1922935907), INT32_C(  358822882), INT32_C(-1580688075),
                            INT32_C(  770718488), INT32_C( 1450946493), INT32_C(-1709682666), INT32_C( 1943195809)),
      simde_mm256_set_epi16(INT16_C( 11942), INT16_C(  3200), INT16_C( 26207), INT16_C( 28891),
                            INT16_C(-21120), INT16_C( 23935), INT16_C( 23697), INT16_C( -6686),
                            INT16_C(  4973), INT16_C(-12874), INT16_C( 21523), INT16_C( 18403),
                            INT16_C(-30279), INT16_C( 22300), INT16_C(-10277), INT16_C( 31923)) },
    { simde_mm256_set_epi16(INT16_C( 23594), INT16_C(-12459), INT16_C(-31007), INT16_C( 27189),
                            INT16_C(  9973), INT16_C(-17207), INT16_C( 29928), INT16_C(-22888),
                            INT16_C( -6347), INT16_C(-15381), INT16_C( 11924), INT16_C( 27730),
                            INT16_C( 32118), INT16_C(-13020), INT16_C(-15209), INT16_C(-12363)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(  663660472), INT32_C( 1693978555), INT32_C(-1956550652), INT32_C(  443334302),
                            INT32_C(-1616544448), INT32_C( 1550532992), INT32_C( -641247424), INT32_C( -628774246),
                            INT32_C( -100015796), INT32_C(-1083334846), INT32_C( 1289981083), INT32_C(-1401758632),
                            INT32_C( 1185649877), INT32_C(-1803875103), INT32_C(-1965714593), INT32_C(-1155378946)),
      simde_mm256_set_epi16(INT16_C( 23594), INT16_C(-12459), INT16_C(-31007), INT16_C( 27189),
                            INT16_C(  9973), INT16_C(-17207), INT16_C( 29928), INT16_C(-22888),
                            INT16_C( -6347), INT16_C(-15381), INT16_C( 11924), INT16_C( 27730),
                            INT16_C( 32118), INT16_C(-13020), INT16_C(-15209), INT16_C(-12363)) },
    { simde_mm256_set_epi16(INT16_C(-27164), INT16_C( 24479), INT16_C( -5741), INT16_C( 22801),
                            INT16_C(  4156), INT16_C( 16807), INT16_C(-14947), INT16_C(-16390),
                            INT16_C(-10040), INT16_C(-23097), INT16_C(-17671), INT16_C(  7407),
                            INT16_C( 10396), INT16_C( 18504), INT16_C(  -151), INT16_C( 25224)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(-1299509468), INT32_C(  625075243), INT32_C(  366130068), INT32_C(  879215932),
                            INT32_C(-1266430196), INT32_C(  614354711), INT32_C( -106069824), INT32_C( 1616883515),
                            INT32_C(   76033746), INT32_C( 1717724968), INT32_C(-1376284635), INT32_C( 1121289971),
                            INT32_C(-2088495749), INT32_C(  336749198), INT32_C( -155113738), INT32_C(-1830992577)),
      simde_mm256_set_epi16(INT16_C(-27164), INT16_C( 24479), INT16_C( -5741), INT16_C( 22801),
                            INT16_C(  4156), INT16_C( 16807), INT16_C(-14947), INT16_C(-16390),
                            INT16_C(-10040), INT16_C(-23097), INT16_C(-17671), INT16_C(  7407),
                            INT16_C( 10396), INT16_C( 18504), INT16_C(  -151), INT16_C( 25224)) },
    { simde_mm256_set_epi16(INT16_C( 12625), INT16_C(-27479), INT16_C(  8247), INT16_C(-17103),
                            INT16_C( 25662), INT16_C( 20506), INT16_C( 11066), INT16_C(  6659),
                            INT16_C( 26083), INT16_C( 21192), INT16_C( 16228), INT16_C(-11727),
                            INT16_C( 26864), INT16_C(-24696), INT16_C(-30487), INT16_C( 29529)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(   34084284), INT32_C(  409350881), INT32_C(-1729303858), INT32_C( -262004167),
                            INT32_C(  146178005), INT32_C(-2107356944), INT32_C(  395917776), INT32_C( 2107797030),
                            INT32_C( -440548267), INT32_C( -370689036), INT32_C(  806714643), INT32_C( 1905229024),
                            INT32_C( -272848474), INT32_C(-1969451680), INT32_C(-1920837116), INT32_C( 1649878440)),
      simde_mm256_set_epi16(INT16_C( 12625), INT16_C(-27479), INT16_C(  8247), INT16_C(-17103),
                            INT16_C( 25662), INT16_C( 20506), INT16_C( 11066), INT16_C(  6659),
                            INT16_C( 26083), INT16_C( 21192), INT16_C( 16228), INT16_C(-11727),
                            INT16_C( 26864), INT16_C(-24696), INT16_C(-30487), INT16_C( 29529)) },
    { simde_mm256_set_epi16(INT16_C(-32242), INT16_C(  1477), INT16_C( 29883), INT16_C(-20158),
                            INT16_C( 25891), INT16_C(  4875), INT16_C(-10189), INT16_C( -5617),
                            INT16_C(  6706), INT16_C( 24335), INT16_C(  9342), INT16_C(-13786),
                            INT16_C(  3512), INT16_C( 16237), INT16_C( -8714), INT16_C( 16153)),
      UINT16_C(    0),
      simde_mm512_set_epi32(INT32_C(  195376036), INT32_C( -495064786), INT32_C(  -53371424), INT32_C( 1311701561),
                            INT32_C(-2117744500), INT32_C(-1179511086), INT32_C(-1867959349), INT32_C(  475635859),
                            INT32_C(  904842846), INT32_C(  684217125), INT32_C( 1793883204), INT32_C(-1966946957),
                            INT32_C(-2097971976), INT32_C( -962037344), INT32_C(  848042064), INT32_C( -776821492)),
      simde_mm256_set_epi16(INT16_C(-32242), INT16_C(  1477), INT16_C( 29883), INT16_C(-20158),
                            INT16_C( 25891), INT16_C(  4875), INT16_C(-10189), INT16_C( -5617),
                            INT16_C(  6706), INT16_C( 24335), INT16_C(  9342), INT16_C(-13786),
                            INT16_C(  3512), INT16_C( 16237), INT16_C( -8714), INT16_C( 16153)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_mask_cvtepi32_epi16(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_cvtepi32_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { UINT16_C(62448),
      simde_mm512_set_epi32(INT32_C( 1733443347), INT32_C( 1269945573), INT32_C( 1675340693), INT32_C( -372773929),
                            INT32_C( 2035462260), INT32_C(  166743347), INT32_C(-2017260412), INT32_C( -999942581),
                            INT32_C( 1336596551), INT32_C( 1310130923), INT32_C(-2130894500), INT32_C(   79178551),
                            INT32_C( -564203638), INT32_C( -457882615), INT32_C(  841119524), INT32_C(-1991554176)),
      simde_mm256_set_epi16(INT16_C( 16147), INT16_C(-11035), INT16_C(-21611), INT16_C( -5161),
                            INT16_C(     0), INT16_C(     0), INT16_C(  3204), INT16_C(  5707),
                            INT16_C(-10169), INT16_C(   747), INT16_C(  8540), INT16_C( 11063),
                            INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(44660),
      simde_mm512_set_epi32(INT32_C(-1218417254), INT32_C( 1692990026), INT32_C(-1386090603), INT32_C( 1397806664),
                            INT32_C( -575365446), INT32_C(-1171498312), INT32_C( 1924925658), INT32_C( -627760657),
                            INT32_C(-1356569050), INT32_C(-1868805570), INT32_C( -261243738), INT32_C( 1743833717),
                            INT32_C(-1242914880), INT32_C( 1254989546), INT32_C( 1994544999), INT32_C( 1112335649)),
      simde_mm256_set_epi16(INT16_C( 28058), INT16_C(     0), INT16_C( -4203), INT16_C(     0),
                            INT16_C(-24902), INT16_C( 23224), INT16_C(  2266), INT16_C(     0),
                            INT16_C(     0), INT16_C( 19006), INT16_C(-17242), INT16_C(-13707),
                            INT16_C(     0), INT16_C(-24854), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(31128),
      simde_mm512_set_epi32(INT32_C(   69575138), INT32_C( 1919952173), INT32_C( 1401785497), INT32_C( 1100669410),
                            INT32_C(-1630704561), INT32_C(-1855467473), INT32_C(   23912175), INT32_C(-1425116453),
                            INT32_C( 1420355947), INT32_C( 1598566097), INT32_C( 2042011572), INT32_C(  547568041),
                            INT32_C( 2024698419), INT32_C(  280574247), INT32_C( 1759265883), INT32_C(  938304495)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C(  9517), INT16_C(-29543), INT16_C( -7710),
                            INT16_C( 27727), INT16_C(     0), INT16_C(     0), INT16_C( 29403),
                            INT16_C( -5781), INT16_C(     0), INT16_C(     0), INT16_C( 14761),
                            INT16_C( 29235), INT16_C(     0), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(19693),
      simde_mm512_set_epi32(INT32_C(-1905579249), INT32_C( 1462851665), INT32_C(-1254373510), INT32_C(  100048907),
                            INT32_C(  516278640), INT32_C(-1425769185), INT32_C(   -7461592), INT32_C( 1494577436),
                            INT32_C(  502504372), INT32_C( 1089107848), INT32_C(-1216995523), INT32_C(  461811592),
                            INT32_C( -365493071), INT32_C(  361540743), INT32_C(-2121833283), INT32_C(  157473702)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C( 22609), INT16_C(     0), INT16_C(     0),
                            INT16_C(-13968), INT16_C( 32031), INT16_C(     0), INT16_C(     0),
                            INT16_C(-25676), INT16_C( 30600), INT16_C(  7997), INT16_C(     0),
                            INT16_C(  1201), INT16_C(-21369), INT16_C(     0), INT16_C( -9306)) },
    { UINT16_C(38458),
      simde_mm512_set_epi32(INT32_C( 1275507295), INT32_C( 1283962881), INT32_C( 1982699930), INT32_C(-1010602497),
                            INT32_C(  376304743), INT32_C( 1749590467), INT32_C( 1162888686), INT32_C( 1275140125),
                            INT32_C(-1086608184), INT32_C(-1843574089), INT32_C( 1348635921), INT32_C(-1517373547),
                            INT32_C(  605500023), INT32_C(  634159381), INT32_C( 1101094426), INT32_C( 1261269548)),
      simde_mm256_set_epi16(INT16_C(-19873), INT16_C(     0), INT16_C(     0), INT16_C( 28159),
                            INT16_C(     0), INT16_C(-24125), INT16_C( 17902), INT16_C(     0),
                            INT16_C(     0), INT16_C(     0), INT16_C(-29423), INT16_C(-18539),
                            INT16_C( 12919), INT16_C(     0), INT16_C( 24090), INT16_C(     0)) },
    { UINT16_C(65367),
      simde_mm512_set_epi32(INT32_C( 1934570464), INT32_C( 1255340091), INT32_C(  198090707), INT32_C(-1547945733),
                            INT32_C( -765577646), INT32_C(  117063114), INT32_C(-1248867899), INT32_C( 1778355257),
                            INT32_C( -663218462), INT32_C( 1146911732), INT32_C(  833410001), INT32_C(  831168945),
                            INT32_C(  809078758), INT32_C( 2051643150), INT32_C( 1911602886), INT32_C( 2000896017)),
      simde_mm256_set_epi16(INT16_C( 13280), INT16_C( -1989), INT16_C(-24621), INT16_C( 14587),
                            INT16_C( 13906), INT16_C( 15818), INT16_C(-13883), INT16_C(-29639),
                            INT16_C(     0), INT16_C( 31732), INT16_C(     0), INT16_C(-24143),
                            INT16_C(     0), INT16_C(-26866), INT16_C(-16698), INT16_C( 16401)) },
    { UINT16_C(62783),
      simde_mm512_set_epi32(INT32_C( 1627565128), INT32_C(-1096299878), INT32_C(  406409532), INT32_C( -810620223),
                            INT32_C(-1765599290), INT32_C(  940004452), INT32_C(    4015514), INT32_C(  569486595),
                            INT32_C( 1803479161), INT32_C(  923486692), INT32_C(  979134652), INT32_C(  -47423892),
                            INT32_C(  359951146), INT32_C( 1353906358), INT32_C( -927313372), INT32_C( -280751214)),
      simde_mm256_set_epi16(INT16_C(-21432), INT16_C(-13670), INT16_C( 20796), INT16_C( -5439),
                            INT16_C(     0), INT16_C( 21604), INT16_C(     0), INT16_C(-21245),
                            INT16_C(     0), INT16_C(     0), INT16_C( 26812), INT16_C( 24172),
                            INT16_C( 27434), INT16_C( -1866), INT16_C( 21028), INT16_C(  5010)) },
    { UINT16_C(44985),
      simde_mm512_set_epi32(INT32_C(  592495083), INT32_C( 1070090610), INT32_C( 1700726886), INT32_C( -811687128),
                            INT32_C( 1766271711), INT32_C(  793626986), INT32_C(-1804033847), INT32_C(-1014866300),
                            INT32_C(-1598413920), INT32_C( 2119648754), INT32_C( 2102514823), INT32_C(   24213809),
                            INT32_C( 1119164534), INT32_C(  583559347), INT32_C(-2107244695), INT32_C( -582680808)),
      simde_mm256_set_epi16(INT16_C(-15893), INT16_C(     0), INT16_C(  2150), INT16_C(     0),
                            INT16_C( 10975), INT16_C(-13974), INT16_C(-24375), INT16_C( 24196),
                            INT16_C(  9120), INT16_C(     0), INT16_C(-11129), INT16_C( 31025),
                            INT16_C(  6262), INT16_C(     0), INT16_C(     0), INT16_C(  -232)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_maskz_cvtepi32_epi16(test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cvtepi64_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(              273955), INT64_C(               -4013),
                            INT64_C(            -7033556), INT64_C(      -1383025729160),
                            INT64_C(          -218214744), INT64_C(   -9402863842296753),
                            INT64_C(                   0), INT64_C(          -240066712)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  35), INT8_C(  83), INT8_C(  44), INT8_C( 120),
                        INT8_C( -88), INT8_C(  79), INT8_C(   0), INT8_C( 104)) },
    { simde_mm512_set_epi64(INT64_C(               50833), INT64_C(           484208564),
                            INT64_C(                -124), INT64_C(         -8907018595),
                            INT64_C(         -1053135968), INT64_C(    2128258677497261),
                            INT64_C(                   3), INT64_C(  263107913893504060)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-111), INT8_C( -76), INT8_C(-124), INT8_C( -99),
                        INT8_C( -96), INT8_C( -83), INT8_C(   3), INT8_C(  60)) },
    { simde_mm512_set_epi64(INT64_C(    6119961081599912), INT64_C(                 949),
                            INT64_C(  761714638418543261), INT64_C(         -7281666562),
                            INT64_C(         -3399190417), INT64_C(                 231),
                            INT64_C(-1217801394263696454), INT64_C(                  43)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -88), INT8_C( -75), INT8_C( -99), INT8_C(  -2),
                        INT8_C( 111), INT8_C( -25), INT8_C( -70), INT8_C(  43)) },
    { simde_mm512_set_epi64(INT64_C(     -74999030828832), INT64_C(       3805264232880),
                            INT64_C(              321542), INT64_C(        -18067967511),
                            INT64_C(      21602392348127), INT64_C(          -482358781),
                            INT64_C(              -12189), INT64_C(   16807900958735709)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -32), INT8_C( -80), INT8_C(   6), INT8_C( -23),
                        INT8_C( -33), INT8_C(   3), INT8_C(  99), INT8_C(  93)) },
    { simde_mm512_set_epi64(INT64_C(               11997), INT64_C(                 192),
                            INT64_C(        -32039837063), INT64_C(               39788),
                            INT64_C(           684970521), INT64_C(     361409660761858),
                            INT64_C(   27655177518327113), INT64_C(         -7050752136)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -35), INT8_C( -64), INT8_C( 121), INT8_C( 108),
                        INT8_C(  25), INT8_C(   2), INT8_C(  73), INT8_C( 120)) },
    { simde_mm512_set_epi64(INT64_C(    -131967536383739), INT64_C(                 -32),
                            INT64_C(                 -54), INT64_C(                -245),
                            INT64_C(              -42658), INT64_C(                   3),
                            INT64_C(         26934708458), INT64_C(               -6255)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   5), INT8_C( -32), INT8_C( -54), INT8_C(  11),
                        INT8_C(  94), INT8_C(   3), INT8_C( -22), INT8_C(-111)) },
    { simde_mm512_set_epi64(INT64_C(                   0), INT64_C(-4069220464223700234),
                            INT64_C(                   0), INT64_C(-8071151262900075656),
                            INT64_C(                  -1), INT64_C(          4132460747),
                            INT64_C(                 -27), INT64_C(            -7116923)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C( -10), INT8_C(   0), INT8_C( 120),
                        INT8_C(  -1), INT8_C( -53), INT8_C( -27), INT8_C(-123)) },
    { simde_mm512_set_epi64(INT64_C(         54634445793), INT64_C(         63569905620),
                            INT64_C(            20981054), INT64_C(             2614918),
                            INT64_C(         11917164823), INT64_C(   15401551242937960),
                            INT64_C(                  -1), INT64_C(        366397165244)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -31), INT8_C( -44), INT8_C(  62), INT8_C(-122),
                        INT8_C(  23), INT8_C( 104), INT8_C(  -1), INT8_C( -68)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_cvtepi64_epi8(test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cvtepi64_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C( -29), INT8_C(  40), INT8_C(-125), INT8_C( 109),
                        INT8_C(  49), INT8_C(-123), INT8_C(  61), INT8_C(  93),
                        INT8_C( 119), INT8_C(  82), INT8_C( -47), INT8_C(-123),
                        INT8_C(  96), INT8_C( 115), INT8_C(  88), INT8_C( 113)),
      UINT8_C(223),
      simde_mm512_set_epi64(INT64_C( 6687541384263213551), INT64_C( 5724651445127999262),
                            INT64_C(  515917234952190154), INT64_C(-2078513286835465824),
                            INT64_C(-5678095839466519103), INT64_C(-5250598357465102176),
                            INT64_C( 5628372052312524003), INT64_C(-6534496578513864639)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -17), INT8_C(  30), INT8_C( -47), INT8_C( -96),
                        INT8_C( -63), INT8_C( -96), INT8_C( -29), INT8_C(  65)) },
    { simde_mm_set_epi8(INT8_C( -14), INT8_C(-119), INT8_C( -22), INT8_C(  58),
                        INT8_C(  54), INT8_C( 111), INT8_C(  42), INT8_C( -77),
                        INT8_C( -18), INT8_C( -70), INT8_C( -80), INT8_C(  46),
                        INT8_C( 106), INT8_C( -96), INT8_C(  15), INT8_C(  95)),
      UINT8_C( 47),
      simde_mm512_set_epi64(INT64_C(-7199926593972874140), INT64_C( 3582798934829117823),
                            INT64_C(-3995728052429757855), INT64_C( 5657566593790284700),
                            INT64_C( 6773484901462632925), INT64_C( 8241942592423347061),
                            INT64_C( 2229820159646878746), INT64_C( 3896347912103466998)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -18), INT8_C( -70), INT8_C(  97), INT8_C(  46),
                        INT8_C( -35), INT8_C( 117), INT8_C(  26), INT8_C( -10)) },
    { simde_mm_set_epi8(INT8_C( -79), INT8_C( -49), INT8_C(   7), INT8_C(-126),
                        INT8_C( -28), INT8_C( 123), INT8_C( -96), INT8_C( 124),
                        INT8_C(  91), INT8_C( -99), INT8_C( -80), INT8_C( -88),
                        INT8_C( 116), INT8_C( -89), INT8_C( 114), INT8_C(  45)),
      UINT8_C( 32),
      simde_mm512_set_epi64(INT64_C( 1872773192980738689), INT64_C(-1477397926013066747),
                            INT64_C(-6194316386860074693), INT64_C(-4822857680519278981),
                            INT64_C(-5283405123335993403), INT64_C(-6429150440483390464),
                            INT64_C(-4929379111042447395), INT64_C(-6405006500483831845)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  91), INT8_C( -99), INT8_C(  59), INT8_C( -88),
                        INT8_C( 116), INT8_C( -89), INT8_C( 114), INT8_C(  45)) },
    { simde_mm_set_epi8(INT8_C( 114), INT8_C( -31), INT8_C(  47), INT8_C(-111),
                        INT8_C(  33), INT8_C( -14), INT8_C( 122), INT8_C( 111),
                        INT8_C( -83), INT8_C(  91), INT8_C( -60), INT8_C(  45),
                        INT8_C(  37), INT8_C( -88), INT8_C( 104), INT8_C(  99)),
      UINT8_C(249),
      simde_mm512_set_epi64(INT64_C(-7653330178277967370), INT64_C(-5020495147285141275),
                            INT64_C( 6758764557968224023), INT64_C(-1851791773179197952),
                            INT64_C(-6103777279451617061), INT64_C(  206815346516353622),
                            INT64_C( 6813004940767614351), INT64_C(-7563904436720933700)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -10), INT8_C( -27), INT8_C(  23), INT8_C(   0),
                        INT8_C( -37), INT8_C( -88), INT8_C( 104), INT8_C( -68)) },
    { simde_mm_set_epi8(INT8_C( -67), INT8_C(  -1), INT8_C( -46), INT8_C( -84),
                        INT8_C(  39), INT8_C(  39), INT8_C(-119), INT8_C(  65),
                        INT8_C(  46), INT8_C( -71), INT8_C(  52), INT8_C( -94),
                        INT8_C( -35), INT8_C(  40), INT8_C(  43), INT8_C(  49)),
      UINT8_C(  7),
      simde_mm512_set_epi64(INT64_C(-3757474529630882229), INT64_C(-5520552524494149273),
                            INT64_C( 8098657770465110034), INT64_C(-3376247133280868335),
                            INT64_C(-6999648475839917999), INT64_C(-6263768072863701828),
                            INT64_C( 8573253492598977185), INT64_C(-5991441569081448700)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  46), INT8_C( -71), INT8_C(  52), INT8_C( -94),
                        INT8_C( -35), INT8_C( -68), INT8_C( -95), INT8_C(   4)) },
    { simde_mm_set_epi8(INT8_C(  -9), INT8_C(-118), INT8_C(  65), INT8_C( -57),
                        INT8_C(  27), INT8_C(  91), INT8_C( -23), INT8_C(  52),
                        INT8_C(   2), INT8_C( -28), INT8_C(  97), INT8_C( -21),
                        INT8_C(  27), INT8_C( -30), INT8_C( 122), INT8_C(-109)),
      UINT8_C(149),
      simde_mm512_set_epi64(INT64_C( 5458614153698880625), INT64_C( -778938361127636026),
                            INT64_C( 8050145995564739343), INT64_C( 1111000858517683962),
                            INT64_C( 3114364286144520444), INT64_C( 6655688875273101631),
                            INT64_C(-7614107199841735371), INT64_C( 2031290604527978571)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 113), INT8_C( -28), INT8_C(  97), INT8_C(  -6),
                        INT8_C(  27), INT8_C(  63), INT8_C( 122), INT8_C(  75)) },
    { simde_mm_set_epi8(INT8_C( -60), INT8_C(  62), INT8_C( 101), INT8_C( -30),
                        INT8_C( -93), INT8_C( -50), INT8_C(  37), INT8_C(  31),
                        INT8_C(  27), INT8_C( -48), INT8_C(  69), INT8_C(  44),
                        INT8_C( -43), INT8_C(  17), INT8_C( -35), INT8_C( 115)),
      UINT8_C(249),
      simde_mm512_set_epi64(INT64_C( 3482415301107112997), INT64_C(-1066915963483370707),
                            INT64_C(  975879680641503817), INT64_C( 1992911285604122710),
                            INT64_C(-3358914405152476651), INT64_C(-2231921767905544800),
                            INT64_C(-2421993418742613205), INT64_C( 7643326833070961923)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  37), INT8_C(  45), INT8_C(  73), INT8_C(  86),
                        INT8_C(  21), INT8_C(  17), INT8_C( -35), INT8_C(   3)) },
    { simde_mm_set_epi8(INT8_C(   6), INT8_C( -97), INT8_C(  51), INT8_C( 125),
                        INT8_C( -36), INT8_C(  69), INT8_C(-124), INT8_C(-125),
                        INT8_C(  22), INT8_C( 105), INT8_C( -62), INT8_C( -46),
                        INT8_C(  52), INT8_C(  71), INT8_C(  89), INT8_C(  35)),
      UINT8_C(239),
      simde_mm512_set_epi64(INT64_C( 1040334403932540361), INT64_C(-6095052890563426812),
                            INT64_C( 3754572277322521777), INT64_C(-6367682946633100387),
                            INT64_C( 1442431276846498456), INT64_C( 7207346769540727007),
                            INT64_C(  308866448566430857), INT64_C( -679722980524780734)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -55), INT8_C(   4), INT8_C( -79), INT8_C( -46),
                        INT8_C(-104), INT8_C( -33), INT8_C(-119), INT8_C(  66)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_mask_cvtepi64_epi8(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_cvtepi64_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { UINT8_C(113),
      simde_mm512_set_epi64(INT64_C(  515917234952190154), INT64_C(-2078513286835465824),
                            INT64_C(-5678095839466519103), INT64_C(-5250598357465102176),
                            INT64_C( 5628372052312524003), INT64_C(-6534496578513864639),
                            INT64_C(-8550316027439185043), INT64_C( 3568325751404155269)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C( -96), INT8_C( -63), INT8_C( -96),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(-123)) },
    { UINT8_C( 30),
      simde_mm512_set_epi64(INT64_C( 6773484901462632925), INT64_C( 8241942592423347061),
                            INT64_C( 2229820159646878746), INT64_C( 3896347912103466998),
                            INT64_C(-5906398137596384710), INT64_C( 3922400752755716142),
                            INT64_C( 7683157866547440500), INT64_C( -739819664570184664)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C( -10),
                        INT8_C(  58), INT8_C(  46), INT8_C( 116), INT8_C(   0)) },
    { UINT8_C(156),
      simde_mm512_set_epi64(INT64_C(-4929379111042447395), INT64_C(-6405006500483831845),
                            INT64_C( 2215750266367444866), INT64_C(-1982814754953121624),
                            INT64_C( 8405812769728018664), INT64_C(-2496168041650804570),
                            INT64_C( 1451057929838678657), INT64_C(-5774632452127213865)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -35), INT8_C(   0), INT8_C(   0), INT8_C( -88),
                        INT8_C( -24), INT8_C( -90), INT8_C(   0), INT8_C(   0)) },
    { UINT8_C(  0),
      simde_mm512_set_epi64(INT64_C( 8742888725914136465), INT64_C( 2446152167684031533),
                            INT64_C( 2713533550337813835), INT64_C(-1117540861446505879),
                            INT64_C( -858039058779513328), INT64_C(-6425799683333831555),
                            INT64_C( -116444347105438968), INT64_C( 8292331728128119228)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0)) },
    { UINT8_C(188),
      simde_mm512_set_epi64(INT64_C(-2510709300292622144), INT64_C( 9148774531133054531),
                            INT64_C( 7313609285459244174), INT64_C( 6211078615791902001),
                            INT64_C(-8527066713273202969), INT64_C( 5633907317463236982),
                            INT64_C( 4554513983759755237), INT64_C(-7822939952516719363)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -64), INT8_C(   0), INT8_C(-114), INT8_C(  49),
                        INT8_C( -25), INT8_C( 118), INT8_C(   0), INT8_C(   0)) },
    { UINT8_C(162),
      simde_mm512_set_epi64(INT64_C( 7323954049511077461), INT64_C( 7928490467780798750),
                            INT64_C(-3252469668491018036), INT64_C(  357209688360787999),
                            INT64_C(-1907656262787903328), INT64_C(-2036415786530496605),
                            INT64_C( 1036996715160227079), INT64_C(-4755851044646844095)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  85), INT8_C(   0), INT8_C( -52), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   7), INT8_C(   0)) },
    { UINT8_C(186),
      simde_mm512_set_epi64(INT64_C( -208925926019796527), INT64_C( 5822768373978481092),
                            INT64_C(-3878688622084013609), INT64_C(-3481162437083752438),
                            INT64_C(-1839007639418773611), INT64_C( -609602476141975244),
                            INT64_C(  208399145178921619), INT64_C(-3757474529630882229)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -47), INT8_C(   0), INT8_C( -41), INT8_C(  10),
                        INT8_C(-107), INT8_C(   0), INT8_C(-109), INT8_C(   0)) },
    { UINT8_C(100),
      simde_mm512_set_epi64(INT64_C( 3568880194756499921), INT64_C( 7170640089353194818),
                            INT64_C( 3704198613752651513), INT64_C(-4305892169634405089),
                            INT64_C( 2004177893035466099), INT64_C( 5458614153698880625),
                            INT64_C( -778938361127636026), INT64_C( 8050145995564739343)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(  66), INT8_C(  -7), INT8_C(   0),
                        INT8_C(   0), INT8_C( 113), INT8_C(   0), INT8_C(   0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_maskz_cvtepi64_epi8(test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cvtepi64_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm512_set_epi64(INT64_C(-7076619543590638545), INT64_C(-1547640767609811155),
                            INT64_C(-3757295450136412488), INT64_C( 7215428799648851281),
                            INT64_C( 4617967402188816469), INT64_C(-6436617106056712861),
                            INT64_C(-5810949596740302593), INT64_C(-6127930763767189365)),
      simde_mm_set_epi16(INT16_C( 27695), INT16_C( 20269), INT16_C( 32440), INT16_C( 25937),
                         INT16_C( 16469), INT16_C( 28003), INT16_C( 18687), INT16_C( -3957)) },
    { simde_mm512_set_epi64(INT64_C(-5381921416793630443), INT64_C( -135676082428955899),
                            INT64_C( 5552873466219372969), INT64_C( 6395720488802354930),
                            INT64_C( 7752411237949515847), INT64_C( 7000778850177503560),
                            INT64_C(-2818020559339799438), INT64_C(-2377752498793556325)),
      simde_mm_set_epi16(INT16_C( -9963), INT16_C( 13061), INT16_C(-24151), INT16_C(-30990),
                         INT16_C( 10311), INT16_C(-11960), INT16_C(-17294), INT16_C(-10597)) },
    { simde_mm512_set_epi64(INT64_C( 2102322971158047948), INT64_C(-4534059143765607616),
                            INT64_C(-8156186454528956670), INT64_C(-1110322353084003886),
                            INT64_C( 2722224959827855869), INT64_C(-8203319889631878316),
                            INT64_C( 3055433368131497869), INT64_C(-5348785296051142577)),
      simde_mm_set_epi16(INT16_C( -4916), INT16_C( 15168), INT16_C( -6398), INT16_C(-30254),
                         INT16_C(-13827), INT16_C( 25428), INT16_C( 28557), INT16_C(    79)) },
    { simde_mm512_set_epi64(INT64_C( 1873487555889602710), INT64_C(-5277671716035001257),
                            INT64_C(-1735965719996896598), INT64_C(-3435103322406338318),
                            INT64_C( 3655189638093457022), INT64_C( 2841010088304043730),
                            INT64_C( 5827184599862175253), INT64_C( -971579224938582414)),
      simde_mm_set_epi16(INT16_C(  8342), INT16_C(-31657), INT16_C( 27306), INT16_C(-16142),
                         INT16_C(-14722), INT16_C( -6446), INT16_C( 26133), INT16_C( -4494)) },
    { simde_mm512_set_epi64(INT64_C(-7259266739541946745), INT64_C(-8120641422716522112),
                            INT64_C( 3643097831469212363), INT64_C(-8930268807776982308),
                            INT64_C( 8468988530628615446), INT64_C( 8901878832922162004),
                            INT64_C( 3527022727868595852), INT64_C( 3854384042855807476)),
      simde_mm_set_epi16(INT16_C( 20103), INT16_C(-20096), INT16_C( -7477), INT16_C(-21796),
                         INT16_C( -8938), INT16_C( -3244), INT16_C( 16012), INT16_C(-17932)) },
    { simde_mm512_set_epi64(INT64_C( 7662135416143755844), INT64_C(-4685369386623572014),
                            INT64_C( 2116014329396479379), INT64_C(-3403348481015692825),
                            INT64_C( 6159842858110289023), INT64_C(  -32959868996567131),
                            INT64_C( 5549180600650677482), INT64_C( 3533643669382248608)),
      simde_mm_set_epi16(INT16_C( 14916), INT16_C( -6190), INT16_C( 17811), INT16_C(-15897),
                         INT16_C(  1151), INT16_C(-16475), INT16_C(-17174), INT16_C( 14496)) },
    { simde_mm512_set_epi64(INT64_C(-7461888410031105124), INT64_C(-5633250430711017511),
                            INT64_C( 2303316420630814828), INT64_C( 8431273119974737498),
                            INT64_C( 6792217696133583430), INT64_C(-1977946393686863567),
                            INT64_C(    7761100783468144), INT64_C( 1660710717369297622)),
      simde_mm_set_epi16(INT16_C( -9316), INT16_C( 20441), INT16_C(  8300), INT16_C(-14758),
                         INT16_C(  4678), INT16_C(-28367), INT16_C(-16784), INT16_C( 10966)) },
    { simde_mm512_set_epi64(INT64_C(  797114543045129949), INT64_C(-2395988650566541266),
                            INT64_C( 3385029089061196001), INT64_C(-8666680959957475234),
                            INT64_C(-2790900178180541201), INT64_C( 9019487904069949528),
                            INT64_C( 4196811514711324420), INT64_C( -189237280820252155)),
      simde_mm_set_epi16(INT16_C(-18723), INT16_C(  8238), INT16_C( 20705), INT16_C( -8098),
                         INT16_C( 29935), INT16_C(-26536), INT16_C( 14084), INT16_C(-19963)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_cvtepi64_epi16(test_vec[i].a);
    simde_assert_m128i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cvtepi64_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi16(INT16_C( -7384), INT16_C(-31891), INT16_C( 12677), INT16_C( 15709),
                         INT16_C( 30546), INT16_C(-11899), INT16_C( 24691), INT16_C( 22641)),
      UINT8_C(223),
      simde_mm512_set_epi64(INT64_C( 6687541384263213551), INT64_C( 5724651445127999262),
                            INT64_C(  515917234952190154), INT64_C(-2078513286835465824),
                            INT64_C(-5678095839466519103), INT64_C(-5250598357465102176),
                            INT64_C( 5628372052312524003), INT64_C(-6534496578513864639)),
      simde_mm_set_epi16(INT16_C(-24081), INT16_C( -4322), INT16_C( 12677), INT16_C(  4512),
                         INT16_C(-31295), INT16_C(  2208), INT16_C(  1251), INT16_C( -1983)) },
    { simde_mm_set_epi16(INT16_C( -3447), INT16_C( -5574), INT16_C( 13935), INT16_C( 10931),
                         INT16_C( -4422), INT16_C(-20434), INT16_C( 27296), INT16_C(  3935)),
      UINT8_C( 47),
      simde_mm512_set_epi64(INT64_C(-7199926593972874140), INT64_C( 3582798934829117823),
                            INT64_C(-3995728052429757855), INT64_C( 5657566593790284700),
                            INT64_C( 6773484901462632925), INT64_C( 8241942592423347061),
                            INT64_C( 2229820159646878746), INT64_C( 3896347912103466998)),
      simde_mm_set_epi16(INT16_C( -3447), INT16_C( -5574), INT16_C( 25185), INT16_C( 10931),
                         INT16_C( -5667), INT16_C( 17269), INT16_C(-13286), INT16_C(  6134)) },
    { simde_mm_set_epi16(INT16_C(-20017), INT16_C(  1922), INT16_C( -7045), INT16_C(-24452),
                         INT16_C( 23453), INT16_C(-20312), INT16_C( 29863), INT16_C( 29229)),
      UINT8_C( 32),
      simde_mm512_set_epi64(INT64_C( 1872773192980738689), INT64_C(-1477397926013066747),
                            INT64_C(-6194316386860074693), INT64_C(-4822857680519278981),
                            INT64_C(-5283405123335993403), INT64_C(-6429150440483390464),
                            INT64_C(-4929379111042447395), INT64_C(-6405006500483831845)),
      simde_mm_set_epi16(INT16_C(-20017), INT16_C(  1922), INT16_C( -1733), INT16_C(-24452),
                         INT16_C( 23453), INT16_C(-20312), INT16_C( 29863), INT16_C( 29229)) },
    { simde_mm_set_epi16(INT16_C( 29409), INT16_C( 12177), INT16_C(  8690), INT16_C( 31343),
                         INT16_C(-21157), INT16_C(-15315), INT16_C(  9640), INT16_C( 26723)),
      UINT8_C(249),
      simde_mm512_set_epi64(INT64_C(-7653330178277967370), INT64_C(-5020495147285141275),
                            INT64_C( 6758764557968224023), INT64_C(-1851791773179197952),
                            INT64_C(-6103777279451617061), INT64_C(  206815346516353622),
                            INT64_C( 6813004940767614351), INT64_C(-7563904436720933700)),
      simde_mm_set_epi16(INT16_C( -1546), INT16_C( 10469), INT16_C( 12055), INT16_C(-14848),
                         INT16_C(-22309), INT16_C(-15315), INT16_C(  9640), INT16_C( 25788)) },
    { simde_mm_set_epi16(INT16_C(-16897), INT16_C(-11604), INT16_C( 10023), INT16_C(-30399),
                         INT16_C( 11961), INT16_C( 13474), INT16_C( -8920), INT16_C( 11057)),
      UINT8_C(  7),
      simde_mm512_set_epi64(INT64_C(-3757474529630882229), INT64_C(-5520552524494149273),
                            INT64_C( 8098657770465110034), INT64_C(-3376247133280868335),
                            INT64_C(-6999648475839917999), INT64_C(-6263768072863701828),
                            INT64_C( 8573253492598977185), INT64_C(-5991441569081448700)),
      simde_mm_set_epi16(INT16_C(-16897), INT16_C(-11604), INT16_C( 10023), INT16_C(-30399),
                         INT16_C( 11961), INT16_C(-23364), INT16_C( 12961), INT16_C(  9988)) },
    { simde_mm_set_epi16(INT16_C( -2166), INT16_C( 16839), INT16_C(  7003), INT16_C( -5836),
                         INT16_C(   740), INT16_C( 25067), INT16_C(  7138), INT16_C( 31379)),
      UINT8_C(149),
      simde_mm512_set_epi64(INT64_C( 5458614153698880625), INT64_C( -778938361127636026),
                            INT64_C( 8050145995564739343), INT64_C( 1111000858517683962),
                            INT64_C( 3114364286144520444), INT64_C( 6655688875273101631),
                            INT64_C(-7614107199841735371), INT64_C( 2031290604527978571)),
      simde_mm_set_epi16(INT16_C( 11377), INT16_C( 16839), INT16_C(  7003), INT16_C(-16646),
                         INT16_C(   740), INT16_C(  8511), INT16_C(  7138), INT16_C(-30645)) },
    { simde_mm_set_epi16(INT16_C(-15298), INT16_C( 26082), INT16_C(-23602), INT16_C(  9503),
                         INT16_C(  7120), INT16_C( 17708), INT16_C(-10991), INT16_C( -8845)),
      UINT8_C(249),
      simde_mm512_set_epi64(INT64_C( 3482415301107112997), INT64_C(-1066915963483370707),
                            INT64_C(  975879680641503817), INT64_C( 1992911285604122710),
                            INT64_C(-3358914405152476651), INT64_C(-2231921767905544800),
                            INT64_C(-2421993418742613205), INT64_C( 7643326833070961923)),
      simde_mm_set_epi16(INT16_C( -9179), INT16_C( 19245), INT16_C(-28087), INT16_C(    86),
                         INT16_C(-19947), INT16_C( 17708), INT16_C(-10991), INT16_C( -2813)) },
    { simde_mm_set_epi16(INT16_C(  1695), INT16_C( 13181), INT16_C( -9147), INT16_C(-31613),
                         INT16_C(  5737), INT16_C(-15662), INT16_C( 13383), INT16_C( 22819)),
      UINT8_C(239),
      simde_mm512_set_epi64(INT64_C( 1040334403932540361), INT64_C(-6095052890563426812),
                            INT64_C( 3754572277322521777), INT64_C(-6367682946633100387),
                            INT64_C( 1442431276846498456), INT64_C( 7207346769540727007),
                            INT64_C(  308866448566430857), INT64_C( -679722980524780734)),
      simde_mm_set_epi16(INT16_C( 19913), INT16_C(-14844), INT16_C(-25423), INT16_C(-31613),
                         INT16_C( 17048), INT16_C(-16161), INT16_C(-21367), INT16_C(-27838)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_mask_cvtepi64_epi16(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_cvtepi64_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { UINT8_C(113),
      simde_mm512_set_epi64(INT64_C(  515917234952190154), INT64_C(-2078513286835465824),
                            INT64_C(-5678095839466519103), INT64_C(-5250598357465102176),
                            INT64_C( 5628372052312524003), INT64_C(-6534496578513864639),
                            INT64_C(-8550316027439185043), INT64_C( 3568325751404155269)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C(  4512), INT16_C(-31295), INT16_C(  2208),
                         INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(-11899)) },
    { UINT8_C( 30),
      simde_mm512_set_epi64(INT64_C( 6773484901462632925), INT64_C( 8241942592423347061),
                            INT64_C( 2229820159646878746), INT64_C( 3896347912103466998),
                            INT64_C(-5906398137596384710), INT64_C( 3922400752755716142),
                            INT64_C( 7683157866547440500), INT64_C( -739819664570184664)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(  6134),
                         INT16_C( -5574), INT16_C(-20434), INT16_C( -5260), INT16_C(     0)) },
    { UINT8_C(156),
      simde_mm512_set_epi64(INT64_C(-4929379111042447395), INT64_C(-6405006500483831845),
                            INT64_C( 2215750266367444866), INT64_C(-1982814754953121624),
                            INT64_C( 8405812769728018664), INT64_C(-2496168041650804570),
                            INT64_C( 1451057929838678657), INT64_C(-5774632452127213865)),
      simde_mm_set_epi16(INT16_C(-30755), INT16_C(     0), INT16_C(     0), INT16_C(-20312),
                         INT16_C(-18200), INT16_C(-22362), INT16_C(     0), INT16_C(     0)) },
    { UINT8_C(  0),
      simde_mm512_set_epi64(INT64_C( 8742888725914136465), INT64_C( 2446152167684031533),
                            INT64_C( 2713533550337813835), INT64_C(-1117540861446505879),
                            INT64_C( -858039058779513328), INT64_C(-6425799683333831555),
                            INT64_C( -116444347105438968), INT64_C( 8292331728128119228)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(     0),
                         INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(     0)) },
    { UINT8_C(188),
      simde_mm512_set_epi64(INT64_C(-2510709300292622144), INT64_C( 9148774531133054531),
                            INT64_C( 7313609285459244174), INT64_C( 6211078615791902001),
                            INT64_C(-8527066713273202969), INT64_C( 5633907317463236982),
                            INT64_C( 4554513983759755237), INT64_C(-7822939952516719363)),
      simde_mm_set_epi16(INT16_C( -5952), INT16_C(     0), INT16_C( -2930), INT16_C(  7473),
                         INT16_C(  1767), INT16_C(-16010), INT16_C(     0), INT16_C(     0)) },
    { UINT8_C(162),
      simde_mm512_set_epi64(INT64_C( 7323954049511077461), INT64_C( 7928490467780798750),
                            INT64_C(-3252469668491018036), INT64_C(  357209688360787999),
                            INT64_C(-1907656262787903328), INT64_C(-2036415786530496605),
                            INT64_C( 1036996715160227079), INT64_C(-4755851044646844095)),
      simde_mm_set_epi16(INT16_C( 13909), INT16_C(     0), INT16_C( 16588), INT16_C(     0),
                         INT16_C(     0), INT16_C(     0), INT16_C( 19719), INT16_C(     0)) },
    { UINT8_C(186),
      simde_mm512_set_epi64(INT64_C( -208925926019796527), INT64_C( 5822768373978481092),
                            INT64_C(-3878688622084013609), INT64_C(-3481162437083752438),
                            INT64_C(-1839007639418773611), INT64_C( -609602476141975244),
                            INT64_C(  208399145178921619), INT64_C(-3757474529630882229)),
      simde_mm_set_epi16(INT16_C( 29137), INT16_C(     0), INT16_C( 16855), INT16_C(-26614),
                         INT16_C( -8299), INT16_C(     0), INT16_C( 31379), INT16_C(     0)) },
    { UINT8_C(100),
      simde_mm512_set_epi64(INT64_C( 3568880194756499921), INT64_C( 7170640089353194818),
                            INT64_C( 3704198613752651513), INT64_C(-4305892169634405089),
                            INT64_C( 2004177893035466099), INT64_C( 5458614153698880625),
                            INT64_C( -778938361127636026), INT64_C( 8050145995564739343)),
      simde_mm_set_epi16(INT16_C(     0), INT16_C(-29374), INT16_C(-24839), INT16_C(     0),
                         INT16_C(     0), INT16_C( 11377), INT16_C(     0), INT16_C(     0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_maskz_cvtepi64_epi16(test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cvtsepi32_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C(  699026811), INT32_C( -375318237), INT32_C(  900674930), INT32_C( -532379219),
                            INT32_C(  259624037), INT32_C( -680802854), INT32_C( 1547540196), INT32_C( -396867814),
                            INT32_C( -680186334), INT32_C( -646546417), INT32_C( 1050185959), INT32_C(  210164141),
                            INT32_C( -363246440), INT32_C( -713541510), INT32_C(-2055712079), INT32_C( 1173262878)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C( 127)) },
    { simde_mm512_set_epi32(INT32_C( -577654055), INT32_C( 1305263210), INT32_C(-1667480872), INT32_C(-1872423369),
                            INT32_C( 1597611840), INT32_C( 1615412737), INT32_C( -741695305), INT32_C(-2095777141),
                            INT32_C(-1892042790), INT32_C(  882313442), INT32_C( -646755171), INT32_C(  911535864),
                            INT32_C( 1767815726), INT32_C( 1917649804), INT32_C(-2128748387), INT32_C( -486061354)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(-128),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(-128)) },
    { simde_mm512_set_epi32(INT32_C( 1571463905), INT32_C(  -29910853), INT32_C( -395190633), INT32_C(-1244489996),
                            INT32_C( -209986706), INT32_C( 1448112069), INT32_C(  665030078), INT32_C( -422459827),
                            INT32_C(  944452940), INT32_C( -118153072), INT32_C(-1276500342), INT32_C( -169030554),
                            INT32_C( -426966016), INT32_C(-1701128651), INT32_C( 2062590869), INT32_C( -398665788)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C(-128), INT8_C( 127), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C(-128)) },
    { simde_mm512_set_epi32(INT32_C(-1419625975), INT32_C(-1694295907), INT32_C( 1865288725), INT32_C(-1595213023),
                            INT32_C( 1905686792), INT32_C(  988618194), INT32_C(  317245247), INT32_C( 1735742912),
                            INT32_C(-1998627679), INT32_C( 1116885338), INT32_C(-1515593887), INT32_C( 2092259348),
                            INT32_C(-1378434205), INT32_C(-1489378877), INT32_C(-2045535759), INT32_C( -659443643)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C(-128)) },
    { simde_mm512_set_epi32(INT32_C(-2129053891), INT32_C( 1544504796), INT32_C( -507196434), INT32_C( 1930810889),
                            INT32_C(  483646719), INT32_C(-1089855428), INT32_C( 2145343900), INT32_C( -881615810),
                            INT32_C( 1970504702), INT32_C( 1583273127), INT32_C( 1163729707), INT32_C(-2126388173),
                            INT32_C( -806581107), INT32_C(-1698532218), INT32_C(-1392314532), INT32_C(-1512591869)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C(-128),
                        INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C(-128)) },
    { simde_mm512_set_epi32(INT32_C( 1328977658), INT32_C(  842060052), INT32_C(-1244029168), INT32_C(  509188382),
                            INT32_C(  982259454), INT32_C(-1037867093), INT32_C(  686186344), INT32_C(  324188035),
                            INT32_C( 1498892321), INT32_C(  508595369), INT32_C( -602085962), INT32_C(-1205094693),
                            INT32_C( -758631374), INT32_C(-1994460640), INT32_C( 1443985396), INT32_C(-1807079691)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(-128),
                        INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C(-128)) },
    { simde_mm512_set_epi32(INT32_C( 1080310926), INT32_C(-1355901593), INT32_C(  256636159), INT32_C( 1796880446),
                            INT32_C(-1357394192), INT32_C( 1762335377), INT32_C( -735301194), INT32_C(  230956482),
                            INT32_C( 1868400485), INT32_C(  -33472709), INT32_C( 1961785078), INT32_C( 1853410726),
                            INT32_C(-1472588796), INT32_C( 1430279086), INT32_C( -445655125), INT32_C( 1987885572)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127)) },
    { simde_mm512_set_epi32(INT32_C(  270394269), INT32_C( -407998192), INT32_C(-1084410926), INT32_C(-1333891767),
                            INT32_C( 1497122895), INT32_C( -215247173), INT32_C(-1835983824), INT32_C(  -84201182),
                            INT32_C(-1074778144), INT32_C( 1293769723), INT32_C( 1950241896), INT32_C(-1359838077),
                            INT32_C(  595655233), INT32_C( 2095576879), INT32_C(-1995468186), INT32_C(-1602668019)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C(-128), INT8_C( 127), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(-128)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_cvtsepi32_epi8(test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cvtsepi32_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C(   4), INT8_C( 110), INT8_C(   8), INT8_C( 124),
                        INT8_C( -85), INT8_C( -65), INT8_C( -62), INT8_C(-114),
                        INT8_C( 110), INT8_C(  33), INT8_C( -28), INT8_C( 106),
                        INT8_C( 107), INT8_C( 102), INT8_C(  89), INT8_C( 124)),
      UINT16_C(12710),
      simde_mm512_set_epi32(INT32_C(-1797606071), INT32_C(  433393018), INT32_C(  712958933), INT32_C( -930490644),
                            INT32_C(  760893683), INT32_C(-2027734617), INT32_C( 1683947105), INT32_C( -424320007),
                            INT32_C(  107722959), INT32_C(  -13745640), INT32_C(-1276316442), INT32_C(-1722135079),
                            INT32_C( 1405989540), INT32_C(-1209232739), INT32_C(  265793764), INT32_C(-1369604571)),
      simde_mm_set_epi8(INT8_C(   4), INT8_C( 110), INT8_C( 127), INT8_C(-128),
                        INT8_C( -85), INT8_C( -65), INT8_C( -62), INT8_C(-128),
                        INT8_C( 127), INT8_C(  33), INT8_C(-128), INT8_C( 106),
                        INT8_C( 107), INT8_C(-128), INT8_C( 127), INT8_C( 124)) },
    { simde_mm_set_epi8(INT8_C(-103), INT8_C(-117), INT8_C(  21), INT8_C( -66),
                        INT8_C(   4), INT8_C( 101), INT8_C(-112), INT8_C( -12),
                        INT8_C(  -5), INT8_C( -50), INT8_C(  26), INT8_C( -76),
                        INT8_C( -83), INT8_C( -93), INT8_C(  76), INT8_C( -53)),
      UINT16_C(12983),
      simde_mm512_set_epi32(INT32_C( -521071455), INT32_C( -376568333), INT32_C(-1920414563), INT32_C(-1600156196),
                            INT32_C( -699126682), INT32_C(  420932189), INT32_C(  368322579), INT32_C(  983236120),
                            INT32_C(  564176453), INT32_C( -258554655), INT32_C(-1141343253), INT32_C(  -63087550),
                            INT32_C( 1360677357), INT32_C(-1430308733), INT32_C( -736696329), INT32_C(  172239397)),
      simde_mm_set_epi8(INT8_C(-103), INT8_C(-117), INT8_C(-128), INT8_C(-128),
                        INT8_C(   4), INT8_C( 101), INT8_C( 127), INT8_C( -12),
                        INT8_C( 127), INT8_C( -50), INT8_C(-128), INT8_C(-128),
                        INT8_C( -83), INT8_C(-128), INT8_C(-128), INT8_C( 127)) },
    { simde_mm_set_epi8(INT8_C( 103), INT8_C( -24), INT8_C( -57), INT8_C( -16),
                        INT8_C( -92), INT8_C( -81), INT8_C(-124), INT8_C(  85),
                        INT8_C(  75), INT8_C(  77), INT8_C(  63), INT8_C(  75),
                        INT8_C( 108), INT8_C( -84), INT8_C( -66), INT8_C( -39)),
      UINT16_C(58066),
      simde_mm512_set_epi32(INT32_C(  123157162), INT32_C(  -87325004), INT32_C(  -74930983), INT32_C( 1838315906),
                            INT32_C(  -81455038), INT32_C( -220592487), INT32_C( 1565085824), INT32_C(   60698648),
                            INT32_C( 1617237598), INT32_C( 1724869908), INT32_C( 1766262170), INT32_C(-1382435444),
                            INT32_C(  -17747961), INT32_C( -126136440), INT32_C( -844801591), INT32_C(-1631191639)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C( -16),
                        INT8_C( -92), INT8_C( -81), INT8_C( 127), INT8_C(  85),
                        INT8_C( 127), INT8_C( 127), INT8_C(  63), INT8_C(-128),
                        INT8_C( 108), INT8_C( -84), INT8_C(-128), INT8_C( -39)) },
    { simde_mm_set_epi8(INT8_C(  26), INT8_C(-116), INT8_C( -21), INT8_C( -72),
                        INT8_C(  76), INT8_C( -52), INT8_C( -71), INT8_C( 116),
                        INT8_C(  41), INT8_C(-107), INT8_C( -34), INT8_C(  88),
                        INT8_C(  29), INT8_C( 112), INT8_C(  89), INT8_C( 127)),
      UINT16_C(63967),
      simde_mm512_set_epi32(INT32_C(-1851657392), INT32_C(  834971347), INT32_C(   96523257), INT32_C( 1786925287),
                            INT32_C( -210779015), INT32_C( 1982975326), INT32_C( 1450447838), INT32_C( 2089131393),
                            INT32_C(-1680682639), INT32_C( -651728462), INT32_C(-1976749038), INT32_C(-1119382179),
                            INT32_C(  674486709), INT32_C(  -51017687), INT32_C(  -85403955), INT32_C( -771014813)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C( 127), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C( -52), INT8_C( -71), INT8_C( 127),
                        INT8_C(-128), INT8_C(-128), INT8_C( -34), INT8_C(-128),
                        INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C(-128)) },
    { simde_mm_set_epi8(INT8_C( -70), INT8_C( -89), INT8_C(  47), INT8_C(  24),
                        INT8_C( 102), INT8_C(  -2), INT8_C(  77), INT8_C(  93),
                        INT8_C(   8), INT8_C(-121), INT8_C(-101), INT8_C(  37),
                        INT8_C(  18), INT8_C(-107), INT8_C(  62), INT8_C( -51)),
      UINT16_C(26024),
      simde_mm512_set_epi32(INT32_C( 1802290864), INT32_C(-1214702907), INT32_C(  821368379), INT32_C( 1594494278),
                            INT32_C(  958208413), INT32_C(-1231281503), INT32_C(  509306793), INT32_C( 1685930534),
                            INT32_C( -983619151), INT32_C(  293070963), INT32_C( 1991494863), INT32_C( 1867270897),
                            INT32_C(-1968548850), INT32_C(-1694953429), INT32_C(  342647810), INT32_C(  475783090)),
      simde_mm_set_epi8(INT8_C( -70), INT8_C(-128), INT8_C( 127), INT8_C(  24),
                        INT8_C( 102), INT8_C(-128), INT8_C(  77), INT8_C( 127),
                        INT8_C(-128), INT8_C(-121), INT8_C( 127), INT8_C(  37),
                        INT8_C(-128), INT8_C(-107), INT8_C(  62), INT8_C( -51)) },
    { simde_mm_set_epi8(INT8_C(  39), INT8_C( 109), INT8_C( 102), INT8_C( -61),
                        INT8_C( -24), INT8_C(  15), INT8_C( 101), INT8_C( -27),
                        INT8_C(  60), INT8_C(-116), INT8_C(  88), INT8_C(   9),
                        INT8_C(  98), INT8_C( -87), INT8_C( -42), INT8_C(  93)),
      UINT16_C(31919),
      simde_mm512_set_epi32(INT32_C(-2040024665), INT32_C(  961434729), INT32_C(-1271404275), INT32_C( 1259406574),
                            INT32_C(-1600364807), INT32_C( 1013354479), INT32_C(  720414379), INT32_C( -600382162),
                            INT32_C( 1552024491), INT32_C(-1608825765), INT32_C( 1791475411), INT32_C(  664867443),
                            INT32_C(  674611871), INT32_C( 2098484402), INT32_C(-1985583060), INT32_C(-2005775341)),
      simde_mm_set_epi8(INT8_C(  39), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C( 101), INT8_C( -27),
                        INT8_C( 127), INT8_C(-116), INT8_C( 127), INT8_C(   9),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(-128)) },
    { simde_mm_set_epi8(INT8_C( -81), INT8_C(  41), INT8_C(  25), INT8_C(  -5),
                        INT8_C( -64), INT8_C(  70), INT8_C( 124), INT8_C( -34),
                        INT8_C(  -2), INT8_C(-101), INT8_C(  45), INT8_C( -46),
                        INT8_C(  80), INT8_C(  30), INT8_C( -59), INT8_C(-110)),
      UINT16_C(11111),
      simde_mm512_set_epi32(INT32_C( -897513851), INT32_C(  631520322), INT32_C(  634536018), INT32_C(-2012793591),
                            INT32_C( 1627606334), INT32_C(-1721387558), INT32_C(-1230866277), INT32_C(-1212419278),
                            INT32_C(  577821894), INT32_C( 1429584547), INT32_C(-2040986931), INT32_C( 2040118421),
                            INT32_C(  936070741), INT32_C(  920738578), INT32_C(-1305390023), INT32_C(-1363026646)),
      simde_mm_set_epi8(INT8_C( -81), INT8_C(  41), INT8_C( 127), INT8_C(  -5),
                        INT8_C( 127), INT8_C(  70), INT8_C(-128), INT8_C(-128),
                        INT8_C(  -2), INT8_C( 127), INT8_C(-128), INT8_C( -46),
                        INT8_C(  80), INT8_C( 127), INT8_C(-128), INT8_C(-128)) },
    { simde_mm_set_epi8(INT8_C(-107), INT8_C( -52), INT8_C(  65), INT8_C(  94),
                        INT8_C( -32), INT8_C( -76), INT8_C(  70), INT8_C( -85),
                        INT8_C( -67), INT8_C(  72), INT8_C(  36), INT8_C( -31),
                        INT8_C(  47), INT8_C( 126), INT8_C(  14), INT8_C(-117)),
      UINT16_C(53720),
      simde_mm512_set_epi32(INT32_C( -215671891), INT32_C( 1165933857), INT32_C(  229060973), INT32_C(-1525363693),
                            INT32_C(-1670687960), INT32_C(  159553420), INT32_C( -802431529), INT32_C(-1862869598),
                            INT32_C(-1575511501), INT32_C( 1180622586), INT32_C( 1144679036), INT32_C( 1714786815),
                            INT32_C( 1463480444), INT32_C( 1916349718), INT32_C( -744963447), INT32_C(-1790859765)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C( 127), INT8_C(  65), INT8_C(-128),
                        INT8_C( -32), INT8_C( -76), INT8_C(  70), INT8_C(-128),
                        INT8_C(-128), INT8_C( 127), INT8_C(  36), INT8_C( 127),
                        INT8_C( 127), INT8_C( 126), INT8_C(  14), INT8_C(-117)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_mask_cvtsepi32_epi8(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_cvtsepi32_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { UINT16_C(22908),
      simde_mm512_set_epi32(INT32_C(  760893683), INT32_C(-2027734617), INT32_C( 1683947105), INT32_C( -424320007),
                            INT32_C(  107722959), INT32_C(  -13745640), INT32_C(-1276316442), INT32_C(-1722135079),
                            INT32_C( 1405989540), INT32_C(-1209232739), INT32_C(  265793764), INT32_C(-1369604571),
                            INT32_C( -598522698), INT32_C( 1529360806), INT32_C(   74319996), INT32_C(-1413496178)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(-128), INT8_C(   0), INT8_C(-128),
                        INT8_C( 127), INT8_C(   0), INT8_C(   0), INT8_C(-128),
                        INT8_C(   0), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C(-128), INT8_C( 127), INT8_C(   0), INT8_C(   0)) },
    { UINT16_C(55020),
      simde_mm512_set_epi32(INT32_C(  564176453), INT32_C( -258554655), INT32_C(-1141343253), INT32_C(  -63087550),
                            INT32_C( 1360677357), INT32_C(-1430308733), INT32_C( -736696329), INT32_C(  172239397),
                            INT32_C(  696452905), INT32_C( 1625371319), INT32_C(-1718938178), INT32_C(   73765108),
                            INT32_C(  -70378828), INT32_C(-1381806901), INT32_C(-1797606071), INT32_C(  433393018)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(-128), INT8_C(   0), INT8_C(-128),
                        INT8_C(   0), INT8_C(-128), INT8_C(-128), INT8_C(   0),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C(   0), INT8_C(   0)) },
    { UINT16_C(65048),
      simde_mm512_set_epi32(INT32_C(  -17747961), INT32_C( -126136440), INT32_C( -844801591), INT32_C(-1631191639),
                            INT32_C( 1959138923), INT32_C( 1255989970), INT32_C( 1743308784), INT32_C(-1532001195),
                            INT32_C( 1263353675), INT32_C( 1823260377), INT32_C( -521071455), INT32_C( -376568333),
                            INT32_C(-1920414563), INT32_C(-1600156196), INT32_C( -699126682), INT32_C(  420932189)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(-128),
                        INT8_C(-128), INT8_C(   0), INT8_C(   0), INT8_C(   0)) },
    { UINT16_C(46476),
      simde_mm512_set_epi32(INT32_C( 1491415646), INT32_C( -327353889), INT32_C(  445443000), INT32_C( 1288485236),
                            INT32_C(  697687640), INT32_C(  493902207), INT32_C(  123157162), INT32_C(  -87325004),
                            INT32_C(  -74930983), INT32_C( 1838315906), INT32_C(  -81455038), INT32_C( -220592487),
                            INT32_C( 1565085824), INT32_C(   60698648), INT32_C( 1617237598), INT32_C( 1724869908)),
      simde_mm_set_epi8(INT8_C( 127), INT8_C(   0), INT8_C( 127), INT8_C( 127),
                        INT8_C(   0), INT8_C( 127), INT8_C(   0), INT8_C(-128),
                        INT8_C(-128), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C( 127), INT8_C(   0), INT8_C(   0)) },
    { UINT16_C(16227),
      simde_mm512_set_epi32(INT32_C(  143104805), INT32_C(  311770829), INT32_C(-1851657392), INT32_C(  834971347),
                            INT32_C(   96523257), INT32_C( 1786925287), INT32_C( -210779015), INT32_C( 1982975326),
                            INT32_C( 1450447838), INT32_C( 2089131393), INT32_C(-1680682639), INT32_C( -651728462),
                            INT32_C(-1976749038), INT32_C(-1119382179), INT32_C(  674486709), INT32_C(  -51017687)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(-128), INT8_C( 127),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C(   0), INT8_C( 127), INT8_C(-128), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C( 127), INT8_C(-128)) },
    { UINT16_C(19805),
      simde_mm512_set_epi32(INT32_C(  821368379), INT32_C( 1594494278), INT32_C(  958208413), INT32_C(-1231281503),
                            INT32_C(  509306793), INT32_C( 1685930534), INT32_C( -983619151), INT32_C(  293070963),
                            INT32_C( 1991494863), INT32_C( 1867270897), INT32_C(-1968548850), INT32_C(-1694953429),
                            INT32_C(  342647810), INT32_C(  475783090), INT32_C(-1901860575), INT32_C(-1695455832)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C( 127), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C( 127), INT8_C(   0), INT8_C( 127),
                        INT8_C(   0), INT8_C( 127), INT8_C(   0), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C(   0), INT8_C(-128)) },
    { UINT16_C( 6853),
      simde_mm512_set_epi32(INT32_C(  720414379), INT32_C( -600382162), INT32_C( 1552024491), INT32_C(-1608825765),
                            INT32_C( 1791475411), INT32_C(  664867443), INT32_C(  674611871), INT32_C( 2098484402),
                            INT32_C(-1985583060), INT32_C(-2005775341), INT32_C(-1904713988), INT32_C(  -21726033),
                            INT32_C(  661481155), INT32_C( -401644059), INT32_C( 1015830537), INT32_C( 1655297629)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(-128),
                        INT8_C( 127), INT8_C(   0), INT8_C( 127), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(-128), INT8_C(   0), INT8_C( 127)) },
    { UINT16_C(36847),
      simde_mm512_set_epi32(INT32_C(-2040986931), INT32_C( 2040118421), INT32_C(  936070741), INT32_C(  920738578),
                            INT32_C(-1305390023), INT32_C(-1363026646), INT32_C(  965440518), INT32_C(-1619252377),
                            INT32_C(-1356260869), INT32_C(-1069122338), INT32_C(  -23384622), INT32_C( 1344193938),
                            INT32_C(-2040024665), INT32_C(  961434729), INT32_C(-1271404275), INT32_C( 1259406574)),
      simde_mm_set_epi8(INT8_C(-128), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C(   0),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_maskz_cvtsepi32_epi8(test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cvtepi64_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
      { simde_mm512_set_epi64(INT64_C( 3636972464222704255), INT64_C(-8637898597608066092),
                            INT64_C(-7117920839692631501), INT64_C(  779376322937895778),
                            INT64_C( 8625341086991316813), INT64_C( -619898035156678588),
                            INT64_C( 9102644088317736936), INT64_C( 6748691650780068291)),
      simde_mm256_set_epi32(INT32_C( -874573185), INT32_C(-1279659052), INT32_C( 1791312435), INT32_C( -351257758),
                            INT32_C(   78837581), INT32_C( 2038827076), INT32_C(  150796264), INT32_C( -547096125)) },
    { simde_mm512_set_epi64(INT64_C(-3188691098302098823), INT64_C(-2239829205413177691),
                            INT64_C(-3686324874761745207), INT64_C( 7163879657873596880),
                            INT64_C(-3722186173215781182), INT64_C( -688687161986027520),
                            INT64_C( -414049911340659363), INT64_C(-5760455419255020630)),
      simde_mm256_set_epi32(INT32_C(  725668473), INT32_C( 1842173605), INT32_C(-1323053879), INT32_C( -852237872),
                            INT32_C( 2107264706), INT32_C( -864849920), INT32_C(-1555952291), INT32_C(  -19618902)) },
    { simde_mm512_set_epi64(INT64_C(  954928457282272691), INT64_C( 6743095391785628647),
                            INT64_C(-4113183023138858226), INT64_C(-4216200208371002899),
                            INT64_C( 4862978039430736617), INT64_C(-8133245958833822305),
                            INT64_C(  422975927762384643), INT64_C(-2594689062389743569)),
      simde_mm256_set_epi32(INT32_C( 1513668019), INT32_C( 1240773607), INT32_C( 1101273870), INT32_C(-1239526931),
                            INT32_C(  -20223255), INT32_C( 1758733727), INT32_C(-2100201725), INT32_C(-2052390865)) },
    { simde_mm512_set_epi64(INT64_C( 1235418251988608652), INT64_C(-1697784138139039522),
                            INT64_C(-7844197282489602934), INT64_C( 8524715380160522696),
                            INT64_C( 2488078747749094790), INT64_C(  668997188312749400),
                            INT64_C( 5402489328274860019), INT64_C( 3255809389037965161)),
      simde_mm256_set_epi32(INT32_C( -892295540), INT32_C( 1424885982), INT32_C( -987796342), INT32_C( -305731128),
                            INT32_C( 1288213894), INT32_C( -182594216), INT32_C(-1565956109), INT32_C(-1443016855)) },
    { simde_mm512_set_epi64(INT64_C( 8211940214269115513), INT64_C(-1467524992070371089),
                            INT64_C( 7395547920392152513), INT64_C( 5720449131594366621),
                            INT64_C(-4149254972258420537), INT64_C(-2405712178673606191),
                            INT64_C(-3007967901025062583), INT64_C( 5561908781648105862)),
      simde_mm256_set_epi32(INT32_C(-1136558983), INT32_C(-2069677841), INT32_C( 2052744641), INT32_C( 1281343133),
                            INT32_C(  537098439), INT32_C(  -28040751), INT32_C(-2086686391), INT32_C(-1824337530)) },
    { simde_mm512_set_epi64(INT64_C(  380636487945477944), INT64_C( 4631524987567251326),
                            INT64_C( 2584519444747767308), INT64_C(  816258266807653524),
                            INT64_C(-1187714861805569615), INT64_C(-4379934582538880391),
                            INT64_C(-4279816909313388105), INT64_C(-2848166400408167703)),
      simde_mm256_set_epi32(INT32_C( 2089475896), INT32_C( -196436098), INT32_C( -616051188), INT32_C(-1509174124),
                            INT32_C( 1067481521), INT32_C(  -61719943), INT32_C( 1219121591), INT32_C(-1032332567)) },
    { simde_mm512_set_epi64(INT64_C(-7496005764999359846), INT64_C(-8729150684960000613),
                            INT64_C( -490843546754529526), INT64_C(-4780112060131952140),
                            INT64_C( 3884257663950773205), INT64_C(-6711214101553505106),
                            INT64_C(-7275075039360044698), INT64_C( -663449602597245794)),
      simde_mm256_set_epi32(INT32_C( -420556134), INT32_C( 2014967195), INT32_C(  325535498), INT32_C( 1296447988),
                            INT32_C(   65002453), INT32_C(  156212398), INT32_C( -190697114), INT32_C(  -19845986)) },
    { simde_mm512_set_epi64(INT64_C(-3417201024126416880), INT64_C(-6294431374261990633),
                            INT64_C(-8960361343509742295), INT64_C(-4865505548535460298),
                            INT64_C( 9141410768359557548), INT64_C(  154894143831828371),
                            INT64_C(-9081419987100634516), INT64_C(-5334041650922884862)),
      simde_mm256_set_epi32(INT32_C(  249038864), INT32_C(-1307948265), INT32_C(   83213609), INT32_C( 1859784246),
                            INT32_C(  436242860), INT32_C(  887252883), INT32_C(  482079340), INT32_C(  597324034)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_cvtepi64_epi32(test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cvtepi64_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi32(INT32_C(  262030369), INT32_C(  472381389), INT32_C( 1820292769), INT32_C( 1183923792),
                            INT32_C(-1208341183), INT32_C(-1257970130), INT32_C(  847221831), INT32_C(  575663548)),
      UINT8_C( 10),
      simde_mm512_set_epi64(INT64_C(-7966868535937805932), INT64_C( 5299008656095408483),
                            INT64_C(-5590285311753850045), INT64_C( 7137964584137341662),
                            INT64_C(-6434834176816605387), INT64_C(-6333271890954486833),
                            INT64_C( 3282767373978865693), INT64_C( 4967684130096997469)),
      simde_mm256_set_epi32(INT32_C(  262030369), INT32_C(  472381389), INT32_C( 1820292769), INT32_C( 1183923792),
                            INT32_C( -833775819), INT32_C(-1257970130), INT32_C(-1505718243), INT32_C(  575663548)) },
    { simde_mm256_set_epi32(INT32_C(-1816200207), INT32_C(  105537387), INT32_C( 1059695632), INT32_C(  611774133),
                            INT32_C( 1637079250), INT32_C( -358018783), INT32_C( 1261913751), INT32_C( -501053537)),
      UINT8_C( 45),
      simde_mm512_set_epi64(INT64_C(-1802035461684441265), INT64_C( 1464385524221546353),
                            INT64_C( 2980222649966367123), INT64_C( 8099038506776411426),
                            INT64_C(  905145040348223898), INT64_C(-2812330085540872295),
                            INT64_C( 9037878101616659294), INT64_C(   45028518261476449)),
      simde_mm256_set_epi32(INT32_C(-1816200207), INT32_C(  105537387), INT32_C( -758017645), INT32_C(  611774133),
                            INT32_C( 1479121306), INT32_C(   70336409), INT32_C( 1261913751), INT32_C( -474166175)) },
    { simde_mm256_set_epi32(INT32_C( 1160949832), INT32_C(  790671825), INT32_C( 1609551524), INT32_C( 1396807876),
                            INT32_C(-1711874318), INT32_C(-1489515049), INT32_C( 1928745488), INT32_C( 1062430982)),
      UINT8_C( 64),
      simde_mm512_set_epi64(INT64_C(-9005930620427138564), INT64_C(-2261531819996514214),
                            INT64_C(-1908590373704801150), INT64_C(-8335280881430651030),
                            INT64_C(-7726592629601528093), INT64_C(  557495353260737980),
                            INT64_C( 8816092406723833767), INT64_C( 1087345366514775540)),
      simde_mm256_set_epi32(INT32_C( 1160949832), INT32_C(-1260554150), INT32_C( 1609551524), INT32_C( 1396807876),
                            INT32_C(-1711874318), INT32_C(-1489515049), INT32_C( 1928745488), INT32_C( 1062430982)) },
    { simde_mm256_set_epi32(INT32_C( -921525147), INT32_C( 1970381634), INT32_C(-1500476692), INT32_C(-1962686032),
                            INT32_C( -498932433), INT32_C( 1322625139), INT32_C( 2144788778), INT32_C(  686983052)),
      UINT8_C( 43),
      simde_mm512_set_epi64(INT64_C(  339079297327618442), INT64_C( -530434709005010771),
                            INT64_C( 3893900520561141682), INT64_C( 6261327879404285016),
                            INT64_C(-4792365276005063244), INT64_C(  139860979584315670),
                            INT64_C( 1098126303565167243), INT64_C(-6673986251972702615)),
      simde_mm256_set_epi32(INT32_C( -921525147), INT32_C( 1970381634), INT32_C(-2034145358), INT32_C(-1962686032),
                            INT32_C(  701310388), INT32_C( 1322625139), INT32_C(  310622859), INT32_C(  928222825)) },
    { simde_mm256_set_epi32(INT32_C( 1215918679), INT32_C( 1464765810), INT32_C( -508860334), INT32_C(  370237422),
                            INT32_C( 2136016247), INT32_C( -652356139), INT32_C(  878885802), INT32_C( 1025777926)),
      UINT8_C(125),
      simde_mm512_set_epi64(INT64_C( 1292112428628168131), INT64_C( 3996200026511053278),
                            INT64_C(-1183464275819018712), INT64_C( 7009328331734920985),
                            INT64_C( 1248957650094672934), INT64_C(-3647207441389629252),
                            INT64_C( 5572165573184689325), INT64_C( 4950883391543296907)),
      simde_mm256_set_epi32(INT32_C( 1215918679), INT32_C( 1469649374), INT32_C( -344898008), INT32_C(  254029593),
                            INT32_C(-1855566810), INT32_C( -506507076), INT32_C(  878885802), INT32_C(-1252932725)) },
    { simde_mm256_set_epi32(INT32_C(  352792784), INT32_C(-1695146469), INT32_C(-1794219354), INT32_C(  767902659),
                            INT32_C(-1756346957), INT32_C( 1345243832), INT32_C( -845357564), INT32_C(  382403555)),
      UINT8_C(222),
      simde_mm512_set_epi64(INT64_C( 8625819089946530671), INT64_C( 1260476417816797254),
                            INT64_C( 6695601616901705840), INT64_C(-3241707820207032037),
                            INT64_C( 2724457343546367531), INT64_C( -510428823621027398),
                            INT64_C(-1373332270819176988), INT64_C( 8119817203721462534)),
      simde_mm256_set_epi32(INT32_C(-1056189585), INT32_C(-1413865402), INT32_C(-1794219354), INT32_C(-1359726309),
                            INT32_C( -406249941), INT32_C( 1962064314), INT32_C( 1866337764), INT32_C(  382403555)) },
    { simde_mm256_set_epi32(INT32_C(  294146054), INT32_C( 1809922858), INT32_C(   26511597), INT32_C( -735526333),
                            INT32_C( -676610806), INT32_C( -291478152), INT32_C(-1583953722), INT32_C(-1538734134)),
      UINT8_C( 12),
      simde_mm512_set_epi64(INT64_C(  972095241651198809), INT64_C( 2070126427207517639),
                            INT64_C( 2191096336057470646), INT64_C(  728519017029928955),
                            INT64_C(-8870108936041189470), INT64_C(-3116006297117081948),
                            INT64_C( 1324483146372913362), INT64_C(-3117177062817132908)),
      simde_mm256_set_epi32(INT32_C(  294146054), INT32_C( 1809922858), INT32_C(   26511597), INT32_C( -735526333),
                            INT32_C(  608120738), INT32_C(  913620644), INT32_C(-1583953722), INT32_C(-1538734134)) },
    { simde_mm256_set_epi32(INT32_C( 1667809857), INT32_C( 1891439881), INT32_C(  486588760), INT32_C( 1276336231),
                            INT32_C(-1279253460), INT32_C(  -53345180), INT32_C( -873798020), INT32_C(  302688693)),
      UINT8_C(103),
      simde_mm512_set_epi64(INT64_C(-3227093025788146904), INT64_C(  779144960212675434),
                            INT64_C( -398606095670776166), INT64_C(-3328435871470452771),
                            INT64_C( 6940472738149975738), INT64_C( 2141675531954249559),
                            INT64_C(  283583591498109395), INT64_C(  106915262196291640)),
      simde_mm256_set_epi32(INT32_C( 1667809857), INT32_C(-1778177174), INT32_C(  775745178), INT32_C( 1276336231),
                            INT32_C(-1279253460), INT32_C(-1706598569), INT32_C(  593482195), INT32_C( 1346836536)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_mask_cvtepi64_epi32(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_cvtepi64_epi32(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { UINT8_C(188),
      simde_mm512_set_epi64(INT64_C(-6434834176816605387), INT64_C(-6333271890954486833),
                            INT64_C( 3282767373978865693), INT64_C( 4967684130096997469),
                            INT64_C(-6489573770131520479), INT64_C( 2028862618814346913),
                            INT64_C( 5084913970682932545), INT64_C(-5402940566847646649)),
      simde_mm256_set_epi32(INT32_C( -833775819), INT32_C(          0), INT32_C(-1505718243), INT32_C( 2059854941),
                            INT32_C(  262030369), INT32_C( 1820292769), INT32_C(          0), INT32_C(          0)) },
    { UINT8_C(222),
      simde_mm512_set_epi64(INT64_C( 7219165944822561777), INT64_C(  453279626729991184),
                            INT64_C( 2627549895410833618), INT64_C(-1537678963076807017),
                            INT64_C(-2152008552520089786), INT64_C(-2364159369550373812),
                            INT64_C( 2326515355522650400), INT64_C(-6353467178539342600)),
      simde_mm256_set_epi32(INT32_C(-1816200207), INT32_C( 1059695632), INT32_C(          0), INT32_C( 1261913751),
                            INT32_C(-1854931130), INT32_C( 1233771596), INT32_C(-1301589728), INT32_C(          0)) },
    { UINT8_C( 97),
      simde_mm512_set_epi64(INT64_C( 4563106325822562891), INT64_C(-7952143840967488800),
                            INT64_C( 2986747956576379061), INT64_C(-3255660993180233572),
                            INT64_C(-9034716586009708386), INT64_C( 6352777639726979248),
                            INT64_C(  302092578477375039), INT64_C(-6186666752246548174)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(  340953824), INT32_C(  693887157), INT32_C(          0),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(   10484018)) },
    { UINT8_C( 16),
      simde_mm512_set_epi64(INT64_C( 6284948560106856688), INT64_C(-3714600103307337967),
                            INT64_C(-1222298779301376683), INT64_C(-7941271492122704167),
                            INT64_C(-8122323343088436160), INT64_C( 4986241561527366097),
                            INT64_C( 6912971158203766980), INT64_C(-7352444207866851881)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(  253167321),
                            INT32_C(          0), INT32_C(          0), INT32_C(          0), INT32_C(          0)) },
    { UINT8_C( 51),
      simde_mm512_set_epi64(INT64_C( 3986686679770014763), INT64_C(-3957920366836210878),
                            INT64_C(-6444498318217983568), INT64_C(-2142898481326086029),
                            INT64_C( 9211797659024787340), INT64_C(-9005930620427138564),
                            INT64_C(-2261531819996514214), INT64_C(-1908590373704801150)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(-1962686032), INT32_C( 1322625139),
                            INT32_C(          0), INT32_C(          0), INT32_C(-1260554150), INT32_C( -778541950)) },
    { UINT8_C(221),
      simde_mm512_set_epi64(INT64_C( 3774785777534509318), INT64_C(  339079297327618442),
                            INT64_C( -530434709005010771), INT64_C( 3893900520561141682),
                            INT64_C( 6261327879404285016), INT64_C(-4792365276005063244),
                            INT64_C(  139860979584315670), INT64_C( 1098126303565167243)),
      simde_mm256_set_epi32(INT32_C( 1025777926), INT32_C(  199678346), INT32_C(          0), INT32_C(-2034145358),
                            INT32_C( 1243901016), INT32_C(  701310388), INT32_C(          0), INT32_C(  310622859)) },
    { UINT8_C(213),
      simde_mm512_set_epi64(INT64_C( 7009328331734920985), INT64_C( 1248957650094672934),
                            INT64_C(-3647207441389629252), INT64_C( 5572165573184689325),
                            INT64_C( 4950883391543296907), INT64_C(-3417026514343333289),
                            INT64_C( 6291121254035056722), INT64_C( 1590157621381367159)),
      simde_mm256_set_epi32(INT32_C(  254029593), INT32_C(-1855566810), INT32_C(          0), INT32_C( -212495187),
                            INT32_C(          0), INT32_C( 1215918679), INT32_C(          0), INT32_C( 2136016247)) },
    { UINT8_C( 40),
      simde_mm512_set_epi64(INT64_C( 8119817203721462534), INT64_C( 4481656528138875088),
                            INT64_C(-7280598643784129882), INT64_C( 3298116809455060403),
                            INT64_C( 5777778267035328004), INT64_C( 1642410762899980648),
                            INT64_C( 7944294506037598912), INT64_C( 6312096001936293250)),
      simde_mm256_set_epi32(INT32_C(          0), INT32_C(          0), INT32_C(-1794219354), INT32_C(          0),
                            INT32_C( -845357564), INT32_C(          0), INT32_C(          0), INT32_C(          0)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_maskz_cvtepi64_epi32(test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i32(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cvtsepi32_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm512_set_epi32(INT32_C( -633826313), INT32_C( -624858207), INT32_C( 1490386470), INT32_C(-2098903851),
                            INT32_C(-1539984349), INT32_C( 1958536651), INT32_C(-1468703883), INT32_C( -330293651),
                            INT32_C( 2111698546), INT32_C(-1712476271), INT32_C( 1928035775), INT32_C(  815855626),
                            INT32_C(-1671100889), INT32_C(-1198252639), INT32_C(-1543874579), INT32_C(  564524240)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(-32768), INT16_C( 32767), INT16_C( 32767),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767)) },
    { simde_mm512_set_epi32(INT32_C( 1797360798), INT32_C(  205569425), INT32_C( 1156136221), INT32_C( 1772840561),
                            INT32_C( 2126360842), INT32_C( 1592304637), INT32_C( 1062735606), INT32_C( -473740091),
                            INT32_C( 1443682851), INT32_C( -765250611), INT32_C( -738272039), INT32_C( 1732082686),
                            INT32_C( -881631866), INT32_C(  -99399017), INT32_C(  252546807), INT32_C(-1695736037)),
      simde_mm256_set_epi16(INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767),
                            INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768)) },
    { simde_mm512_set_epi32(INT32_C( -260564878), INT32_C(  427961207), INT32_C(-1056239301), INT32_C( 1225809734),
                            INT32_C(  213123881), INT32_C(  379025644), INT32_C( 1973103858), INT32_C(-2053346996),
                            INT32_C(-1954883006), INT32_C(  667947760), INT32_C(  184139306), INT32_C(  168675179),
                            INT32_C(-1651207657), INT32_C(-1932451475), INT32_C(  207773420), INT32_C(-1587464239)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C( 32767),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-32768), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                            INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768)) },
    { simde_mm512_set_epi32(INT32_C( -301891179), INT32_C( 1118988231), INT32_C(  -81115331), INT32_C(   95289597),
                            INT32_C(   79711365), INT32_C( -252265140), INT32_C(-1901421271), INT32_C(  -86214221),
                            INT32_C(  494361026), INT32_C( 1393707090), INT32_C( 1367330444), INT32_C( 1529415750),
                            INT32_C( -219576728), INT32_C(  785923707), INT32_C(-1615246269), INT32_C( -860551446)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C( 32767),
                            INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                            INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C(-32768)) },
    { simde_mm512_set_epi32(INT32_C( -668182583), INT32_C(  778563571), INT32_C( 1803655402), INT32_C(  668307371),
                            INT32_C(  166146078), INT32_C( 2089905810), INT32_C(     540370), INT32_C( -610320208),
                            INT32_C( -644851790), INT32_C( 1731401690), INT32_C( 1162165084), INT32_C( -702531425),
                            INT32_C( 1412040872), INT32_C( -506509249), INT32_C(-1429255709), INT32_C( 1954724351)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-32768), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767)) },
    { simde_mm512_set_epi32(INT32_C(  895849603), INT32_C( 1425609255), INT32_C(  649927889), INT32_C( 1727611756),
                            INT32_C( 1297009704), INT32_C(-1218308449), INT32_C(-1305018130), INT32_C( 1967276695),
                            INT32_C( -413179024), INT32_C(-1726322338), INT32_C( 1840183236), INT32_C(  284256212),
                            INT32_C( 1788242258), INT32_C(  806002384), INT32_C( -365421949), INT32_C(-2081268803)),
      simde_mm256_set_epi16(INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                            INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767),
                            INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C( 32767),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(-32768), INT16_C(-32768)) },
    { simde_mm512_set_epi32(INT32_C( -850026299), INT32_C(  763115562), INT32_C( -698858672), INT32_C(-2063362395),
                            INT32_C(-1349430084), INT32_C(  963716730), INT32_C( 1798713564), INT32_C(-1893101926),
                            INT32_C(  662898969), INT32_C(  188387073), INT32_C( -332183490), INT32_C(-1234385790),
                            INT32_C(  839752488), INT32_C( 1421488669), INT32_C(  544371590), INT32_C(  832408505)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C(-32768),
                            INT16_C(-32768), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(-32768), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767)) },
    { simde_mm512_set_epi32(INT32_C( 1070222476), INT32_C( 1830989582), INT32_C(  732250943), INT32_C(-1460385002),
                            INT32_C(-1298211555), INT32_C(-1342658291), INT32_C(  646581591), INT32_C(-2104719353),
                            INT32_C( 1887679149), INT32_C( 1571180495), INT32_C( 1241438931), INT32_C(   97459949),
                            INT32_C( 1558770394), INT32_C(-1780666235), INT32_C( 1621650638), INT32_C( -918750867)),
      simde_mm256_set_epi16(INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C( 32767),
                            INT16_C( 32767), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_cvtsepi32_epi16(test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cvtsepi32_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m256i src;
    simde__mmask16 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { simde_mm256_set_epi16(INT16_C(  4055), INT16_C(-20252), INT16_C(-20899), INT16_C( 32293),
                            INT16_C( -9133), INT16_C( 17590), INT16_C( 23336), INT16_C( 12710),
                            INT16_C(  1134), INT16_C(  2172), INT16_C(-21569), INT16_C(-15730),
                            INT16_C( 28193), INT16_C( -7062), INT16_C( 27494), INT16_C( 22908)),
      UINT16_C(37533),
      simde_mm512_set_epi32(INT32_C(-1718938178), INT32_C(   73765108), INT32_C(  -70378828), INT32_C(-1381806901),
                            INT32_C(-1797606071), INT32_C(  433393018), INT32_C(  712958933), INT32_C( -930490644),
                            INT32_C(  760893683), INT32_C(-2027734617), INT32_C( 1683947105), INT32_C( -424320007),
                            INT32_C(  107722959), INT32_C(  -13745640), INT32_C(-1276316442), INT32_C(-1722135079)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C(-20252), INT16_C(-20899), INT16_C(-32768),
                            INT16_C( -9133), INT16_C( 17590), INT16_C( 32767), INT16_C( 12710),
                            INT16_C( 32767), INT16_C(  2172), INT16_C(-21569), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(-32768), INT16_C( 27494), INT16_C(-32768)) },
    { simde_mm256_set_epi16(INT16_C(-17416), INT16_C( 31723), INT16_C(  -963), INT16_C( 23618),
                            INT16_C( 20762), INT16_C( 18925), INT16_C(-21825), INT16_C( 14467),
                            INT16_C(-11242), INT16_C( -6153), INT16_C(  2628), INT16_C( 10789),
                            INT16_C( 10627), INT16_C(  1833), INT16_C( 24801), INT16_C( 12983)),
      UINT16_C(50401),
      simde_mm512_set_epi32(INT32_C( -844801591), INT32_C(-1631191639), INT32_C( 1959138923), INT32_C( 1255989970),
                            INT32_C( 1743308784), INT32_C(-1532001195), INT32_C( 1263353675), INT32_C( 1823260377),
                            INT32_C( -521071455), INT32_C( -376568333), INT32_C(-1920414563), INT32_C(-1600156196),
                            INT32_C( -699126682), INT32_C(  420932189), INT32_C(  368322579), INT32_C(  983236120)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C(-32768), INT16_C(  -963), INT16_C( 23618),
                            INT16_C( 20762), INT16_C(-32768), INT16_C(-21825), INT16_C( 14467),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C( 10789),
                            INT16_C( 10627), INT16_C(  1833), INT16_C( 24801), INT16_C( 32767)) },
    { simde_mm256_set_epi16(INT16_C( 23881), INT16_C( 20608), INT16_C(   926), INT16_C( 12312),
                            INT16_C( 24677), INT16_C(  5726), INT16_C( 26319), INT16_C( 27924),
                            INT16_C( 26951), INT16_C(  1434), INT16_C(-21095), INT16_C(-19060),
                            INT16_C(  -271), INT16_C( 12295), INT16_C( -1925), INT16_C( 20360)),
      UINT16_C( 1689),
      simde_mm512_set_epi32(INT32_C(-1976749038), INT32_C(-1119382179), INT32_C(  674486709), INT32_C(  -51017687),
                            INT32_C(  -85403955), INT32_C( -771014813), INT32_C( 1491415646), INT32_C( -327353889),
                            INT32_C(  445443000), INT32_C( 1288485236), INT32_C(  697687640), INT32_C(  493902207),
                            INT32_C(  123157162), INT32_C(  -87325004), INT32_C(  -74930983), INT32_C( 1838315906)),
      simde_mm256_set_epi16(INT16_C( 23881), INT16_C( 20608), INT16_C(   926), INT16_C( 12312),
                            INT16_C( 24677), INT16_C(-32768), INT16_C( 32767), INT16_C( 27924),
                            INT16_C( 32767), INT16_C(  1434), INT16_C(-21095), INT16_C( 32767),
                            INT16_C( 32767), INT16_C( 12295), INT16_C( -1925), INT16_C( 32767)) },
    { simde_mm256_set_epi16(INT16_C(  1472), INT16_C(-11271), INT16_C( 27266), INT16_C( 20711),
                            INT16_C( -3217), INT16_C(-15239), INT16_C( 30257), INT16_C(-12962),
                            INT16_C( 22132), INT16_C(  5086), INT16_C( 31877), INT16_C(-25215),
                            INT16_C(-25646), INT16_C(-11919), INT16_C( -9945), INT16_C( 27058)),
      UINT16_C(42707),
      simde_mm512_set_epi32(INT32_C(  509306793), INT32_C( 1685930534), INT32_C( -983619151), INT32_C(  293070963),
                            INT32_C( 1991494863), INT32_C( 1867270897), INT32_C(-1968548850), INT32_C(-1694953429),
                            INT32_C(  342647810), INT32_C(  475783090), INT32_C(-1901860575), INT32_C(-1695455832),
                            INT32_C(-1163448552), INT32_C( 1727941981), INT32_C(  143104805), INT32_C(  311770829)),
      simde_mm256_set_epi16(INT16_C( 32767), INT16_C(-11271), INT16_C(-32768), INT16_C( 20711),
                            INT16_C( -3217), INT16_C( 32767), INT16_C(-32768), INT16_C(-12962),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 31877), INT16_C(-32768),
                            INT16_C(-25646), INT16_C(-11919), INT16_C( 32767), INT16_C( 32767)) },
    { simde_mm256_set_epi16(INT16_C( 15500), INT16_C( 22537), INT16_C( 25257), INT16_C(-10659),
                            INT16_C( 27500), INT16_C(-14672), INT16_C(-18535), INT16_C(  6853),
                            INT16_C( 12533), INT16_C(  5691), INT16_C( 24330), INT16_C(  3398),
                            INT16_C( 14621), INT16_C(  6557), INT16_C(-18788), INT16_C(  8865)),
      UINT16_C(26085),
      simde_mm512_set_epi32(INT32_C(-1271404275), INT32_C( 1259406574), INT32_C(-1600364807), INT32_C( 1013354479),
                            INT32_C(  720414379), INT32_C( -600382162), INT32_C( 1552024491), INT32_C(-1608825765),
                            INT32_C( 1791475411), INT32_C(  664867443), INT32_C(  674611871), INT32_C( 2098484402),
                            INT32_C(-1985583060), INT32_C(-2005775341), INT32_C(-1904713988), INT32_C(  -21726033)),
      simde_mm256_set_epi16(INT16_C( 15500), INT16_C( 32767), INT16_C(-32768), INT16_C(-10659),
                            INT16_C( 27500), INT16_C(-32768), INT16_C(-18535), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(  3398),
                            INT16_C( 14621), INT16_C(-32768), INT16_C(-18788), INT16_C(-32768)) },
    { simde_mm256_set_epi16(INT16_C( 14731), INT16_C( 29702), INT16_C(-24708), INT16_C( 11111),
                            INT16_C(-20695), INT16_C(  6651), INT16_C(-16314), INT16_C( 31966),
                            INT16_C(  -357), INT16_C( 11730), INT16_C( 20510), INT16_C(-14958),
                            INT16_C(-31129), INT16_C(-20057), INT16_C( 14670), INT16_C( 21609)),
      UINT16_C(56618),
      simde_mm512_set_epi32(INT32_C(-1119345439), INT32_C(  796790411), INT32_C( -897513851), INT32_C(  631520322),
                            INT32_C(  634536018), INT32_C(-2012793591), INT32_C( 1627606334), INT32_C(-1721387558),
                            INT32_C(-1230866277), INT32_C(-1212419278), INT32_C(  577821894), INT32_C( 1429584547),
                            INT32_C(-2040986931), INT32_C( 2040118421), INT32_C(  936070741), INT32_C(  920738578)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C( 32767), INT16_C(-24708), INT16_C( 32767),
                            INT16_C( 32767), INT16_C(-32768), INT16_C(-16314), INT16_C(-32768),
                            INT16_C(  -357), INT16_C( 11730), INT16_C( 32767), INT16_C(-14958),
                            INT16_C(-32768), INT16_C(-20057), INT16_C( 32767), INT16_C( 21609)) },
    { simde_mm256_set_epi16(INT16_C( 22330), INT16_C( -3972), INT16_C( 29241), INT16_C( 11542),
                            INT16_C(-11368), INT16_C(-15735), INT16_C(-27327), INT16_C(-23029),
                            INT16_C( -8383), INT16_C(-32273), INT16_C(  3895), INT16_C(-11816),
                            INT16_C(-27188), INT16_C( 16734), INT16_C( -8012), INT16_C( 18091)),
      UINT16_C(37375),
      simde_mm512_set_epi32(INT32_C( -221392236), INT32_C( -711063988), INT32_C( 2133711027), INT32_C(  509873922),
                            INT32_C( 1729615213), INT32_C( -831792372), INT32_C( -215671891), INT32_C( 1165933857),
                            INT32_C(  229060973), INT32_C(-1525363693), INT32_C(-1670687960), INT32_C(  159553420),
                            INT32_C( -802431529), INT32_C(-1862869598), INT32_C(-1575511501), INT32_C( 1180622586)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C( -3972), INT16_C( 29241), INT16_C( 32767),
                            INT16_C(-11368), INT16_C(-15735), INT16_C(-27327), INT16_C( 32767),
                            INT16_C( 32767), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C( 32767)) },
    { simde_mm256_set_epi16(INT16_C(  -456), INT16_C( -6277), INT16_C(-13154), INT16_C( 20424),
                            INT16_C(  7701), INT16_C(  9153), INT16_C( 18208), INT16_C(  1720),
                            INT16_C(-14211), INT16_C( 26958), INT16_C( 27767), INT16_C(-15259),
                            INT16_C(-16868), INT16_C(-31112), INT16_C( -6815), INT16_C( 21503)),
      UINT16_C(29475),
      simde_mm512_set_epi32(INT32_C(  441964111), INT32_C(-1041336788), INT32_C( 1240459905), INT32_C( 1190379131),
                            INT32_C(  660048661), INT32_C(  992314379), INT32_C(   13012539), INT32_C(-1751743734),
                            INT32_C(-1323762798), INT32_C(  640110239), INT32_C( 1866432721), INT32_C(-1867947321),
                            INT32_C(-2066921506), INT32_C(  704072659), INT32_C(  929476148), INT32_C(-1455589491)),
      simde_mm256_set_epi16(INT16_C(  -456), INT16_C(-32768), INT16_C( 32767), INT16_C( 32767),
                            INT16_C(  7701), INT16_C(  9153), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-14211), INT16_C( 26958), INT16_C( 32767), INT16_C(-15259),
                            INT16_C(-16868), INT16_C(-31112), INT16_C( 32767), INT16_C(-32768)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_mask_cvtsepi32_epi16(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_cvtsepi32_epi16(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask16 k;
    simde__m512i a;
    simde__m256i r;
  } test_vec[8] = {
    { UINT16_C(22908),
      simde_mm512_set_epi32(INT32_C(  760893683), INT32_C(-2027734617), INT32_C( 1683947105), INT32_C( -424320007),
                            INT32_C(  107722959), INT32_C(  -13745640), INT32_C(-1276316442), INT32_C(-1722135079),
                            INT32_C( 1405989540), INT32_C(-1209232739), INT32_C(  265793764), INT32_C(-1369604571),
                            INT32_C( -598522698), INT32_C( 1529360806), INT32_C(   74319996), INT32_C(-1413496178)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C(-32768), INT16_C(     0), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(     0), INT16_C(     0), INT16_C(-32768),
                            INT16_C(     0), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-32768), INT16_C( 32767), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(55020),
      simde_mm512_set_epi32(INT32_C(  564176453), INT32_C( -258554655), INT32_C(-1141343253), INT32_C(  -63087550),
                            INT32_C( 1360677357), INT32_C(-1430308733), INT32_C( -736696329), INT32_C(  172239397),
                            INT32_C(  696452905), INT32_C( 1625371319), INT32_C(-1718938178), INT32_C(   73765108),
                            INT32_C(  -70378828), INT32_C(-1381806901), INT32_C(-1797606071), INT32_C(  433393018)),
      simde_mm256_set_epi16(INT16_C( 32767), INT16_C(-32768), INT16_C(     0), INT16_C(-32768),
                            INT16_C(     0), INT16_C(-32768), INT16_C(-32768), INT16_C(     0),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(-32768), INT16_C(     0),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(65048),
      simde_mm512_set_epi32(INT32_C(  -17747961), INT32_C( -126136440), INT32_C( -844801591), INT32_C(-1631191639),
                            INT32_C( 1959138923), INT32_C( 1255989970), INT32_C( 1743308784), INT32_C(-1532001195),
                            INT32_C( 1263353675), INT32_C( 1823260377), INT32_C( -521071455), INT32_C( -376568333),
                            INT32_C(-1920414563), INT32_C(-1600156196), INT32_C( -699126682), INT32_C(  420932189)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C( 32767), INT16_C(     0),
                            INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(-32768),
                            INT16_C(-32768), INT16_C(     0), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(46476),
      simde_mm512_set_epi32(INT32_C( 1491415646), INT32_C( -327353889), INT32_C(  445443000), INT32_C( 1288485236),
                            INT32_C(  697687640), INT32_C(  493902207), INT32_C(  123157162), INT32_C(  -87325004),
                            INT32_C(  -74930983), INT32_C( 1838315906), INT32_C(  -81455038), INT32_C( -220592487),
                            INT32_C( 1565085824), INT32_C(   60698648), INT32_C( 1617237598), INT32_C( 1724869908)),
      simde_mm256_set_epi16(INT16_C( 32767), INT16_C(     0), INT16_C( 32767), INT16_C( 32767),
                            INT16_C(     0), INT16_C( 32767), INT16_C(     0), INT16_C(-32768),
                            INT16_C(-32768), INT16_C(     0), INT16_C(     0), INT16_C(     0),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(     0), INT16_C(     0)) },
    { UINT16_C(16227),
      simde_mm512_set_epi32(INT32_C(  143104805), INT32_C(  311770829), INT32_C(-1851657392), INT32_C(  834971347),
                            INT32_C(   96523257), INT32_C( 1786925287), INT32_C( -210779015), INT32_C( 1982975326),
                            INT32_C( 1450447838), INT32_C( 2089131393), INT32_C(-1680682639), INT32_C( -651728462),
                            INT32_C(-1976749038), INT32_C(-1119382179), INT32_C(  674486709), INT32_C(  -51017687)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(-32768), INT16_C( 32767),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(-32768), INT16_C( 32767),
                            INT16_C(     0), INT16_C( 32767), INT16_C(-32768), INT16_C(     0),
                            INT16_C(     0), INT16_C(     0), INT16_C( 32767), INT16_C(-32768)) },
    { UINT16_C(19805),
      simde_mm512_set_epi32(INT32_C(  821368379), INT32_C( 1594494278), INT32_C(  958208413), INT32_C(-1231281503),
                            INT32_C(  509306793), INT32_C( 1685930534), INT32_C( -983619151), INT32_C(  293070963),
                            INT32_C( 1991494863), INT32_C( 1867270897), INT32_C(-1968548850), INT32_C(-1694953429),
                            INT32_C(  342647810), INT32_C(  475783090), INT32_C(-1901860575), INT32_C(-1695455832)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C( 32767), INT16_C(     0), INT16_C(     0),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(     0), INT16_C( 32767),
                            INT16_C(     0), INT16_C( 32767), INT16_C(     0), INT16_C(-32768),
                            INT16_C( 32767), INT16_C( 32767), INT16_C(     0), INT16_C(-32768)) },
    { UINT16_C( 6853),
      simde_mm512_set_epi32(INT32_C(  720414379), INT32_C( -600382162), INT32_C( 1552024491), INT32_C(-1608825765),
                            INT32_C( 1791475411), INT32_C(  664867443), INT32_C(  674611871), INT32_C( 2098484402),
                            INT32_C(-1985583060), INT32_C(-2005775341), INT32_C(-1904713988), INT32_C(  -21726033),
                            INT32_C(  661481155), INT32_C( -401644059), INT32_C( 1015830537), INT32_C( 1655297629)),
      simde_mm256_set_epi16(INT16_C(     0), INT16_C(     0), INT16_C(     0), INT16_C(-32768),
                            INT16_C( 32767), INT16_C(     0), INT16_C( 32767), INT16_C(     0),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(     0), INT16_C(     0),
                            INT16_C(     0), INT16_C(-32768), INT16_C(     0), INT16_C( 32767)) },
    { UINT16_C(36847),
      simde_mm512_set_epi32(INT32_C(-2040986931), INT32_C( 2040118421), INT32_C(  936070741), INT32_C(  920738578),
                            INT32_C(-1305390023), INT32_C(-1363026646), INT32_C(  965440518), INT32_C(-1619252377),
                            INT32_C(-1356260869), INT32_C(-1069122338), INT32_C(  -23384622), INT32_C( 1344193938),
                            INT32_C(-2040024665), INT32_C(  961434729), INT32_C(-1271404275), INT32_C( 1259406574)),
      simde_mm256_set_epi16(INT16_C(-32768), INT16_C(     0), INT16_C(     0), INT16_C(     0),
                            INT16_C(-32768), INT16_C(-32768), INT16_C( 32767), INT16_C(-32768),
                            INT16_C(-32768), INT16_C(-32768), INT16_C(-32768), INT16_C(     0),
                            INT16_C(-32768), INT16_C( 32767), INT16_C(-32768), INT16_C( 32767)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m256i r = simde_mm512_maskz_cvtsepi32_epi16(test_vec[i].k, test_vec[i].a);
    simde_assert_m256i_i16(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_cvtsepi64_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
     { simde_mm512_set_epi64(INT64_C( 5688878986790062607), INT64_C( -507006338933993777),
                            INT64_C( 2731700857838766689), INT64_C( 5038766546414012764),
                            INT64_C( 8031668245477288096), INT64_C( 8558843731862564067),
                            INT64_C( 1445459848667085535), INT64_C(  -83609451038325995)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C(-128)) },
    { simde_mm512_set_epi64(INT64_C( 8541199063845314870), INT64_C( 6183904058691300593),
                            INT64_C( 4987917255037318931), INT64_C(-6461883516210235787),
                            INT64_C(-2290177652468594952), INT64_C( -292256871616436513),
                            INT64_C( 8655434451305349549), INT64_C( 2098144080369590904)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C(-128),
                        INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C( 127)) },
    { simde_mm512_set_epi64(INT64_C( 8921116119188857611), INT64_C(-7689618306950712122),
                            INT64_C( 1518593000980919604), INT64_C( 4285824591643310377),
                            INT64_C(-5934398452121416225), INT64_C( 3925581736765257012),
                            INT64_C(-1280589966661740296), INT64_C(-6610380819072272984)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C(-128)) },
    { simde_mm512_set_epi64(INT64_C(-7117386000751069153), INT64_C(-4042166984900833962),
                            INT64_C( 5586053217269340515), INT64_C(-7584975210127513398),
                            INT64_C( 3657404296828616096), INT64_C(-2770219948143520502),
                            INT64_C( 4495347402256698165), INT64_C( -398794429961863290)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C(-128)) },
    { simde_mm512_set_epi64(INT64_C(-7309298589786727464), INT64_C( 6351016797136664888),
                            INT64_C( -796932205202576292), INT64_C( 5974868289408841942),
                            INT64_C(-5246045528720698174), INT64_C( 5897717133320969396),
                            INT64_C( 8855007060554502516), INT64_C( 1795078486534561817)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C( 127), INT8_C(-128), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C( 127), INT8_C( 127)) },
    { simde_mm512_set_epi64(INT64_C(-8235362769944622173), INT64_C(-1394979073669202107),
                            INT64_C(-1990341901794640861), INT64_C(-5381553921365904442),
                            INT64_C( 2258222102802041167), INT64_C( 8110919934642332525),
                            INT64_C( 3859146607814864086), INT64_C( 5989226152769229183)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C( 127)) },
    { simde_mm512_set_epi64(INT64_C( 7694422639508462949), INT64_C(-2072064725478552758),
                            INT64_C( 3296208651842008367), INT64_C(-5674066605955291159),
                            INT64_C( 5400731755389512823), INT64_C( 4001845542455980591),
                            INT64_C( 1263947732552642640), INT64_C( 1152590716152552703)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C(-128),
                        INT8_C( 127), INT8_C( 127), INT8_C( 127), INT8_C( 127)) },
    { simde_mm512_set_epi64(INT64_C(-5372614012128921165), INT64_C(-2041172040005057172),
                            INT64_C(-8233280709947200170), INT64_C( 1449409854914395254),
                            INT64_C( 6224671476337807114), INT64_C( 4022680139615958951),
                            INT64_C( -277976409085756360), INT64_C(-4710983618327201696)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C(-128), INT8_C(-128), INT8_C( 127),
                        INT8_C( 127), INT8_C( 127), INT8_C(-128), INT8_C(-128)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_cvtsepi64_epi8(test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_mask_cvtsepi64_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__m128i src;
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { simde_mm_set_epi8(INT8_C( -98), INT8_C(-118), INT8_C( -55), INT8_C(  84),
                        INT8_C( -38), INT8_C(-100), INT8_C(  69), INT8_C( -11),
                        INT8_C( -76), INT8_C( -44), INT8_C( 111), INT8_C(  57),
                        INT8_C(-108), INT8_C( 117), INT8_C( -94), INT8_C(  99)),
      UINT8_C( 89),
      simde_mm512_set_epi64(INT64_C(-6464916445333598184), INT64_C(-9074050598773963479),
                            INT64_C(-6773163771856001287), INT64_C( 8528895860955669022),
                            INT64_C(  743395091751495893), INT64_C(-5753646205421077345),
                            INT64_C( 6781795189594641427), INT64_C( 7308036127418939455)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( -76), INT8_C(-128), INT8_C( 111), INT8_C( 127),
                        INT8_C( 127), INT8_C( 117), INT8_C( -94), INT8_C( 127)) },
    { simde_mm_set_epi8(INT8_C( -29), INT8_C( -88), INT8_C(  59), INT8_C( -11),
                        INT8_C(-100), INT8_C(-112), INT8_C(  21), INT8_C( -50),
                        INT8_C(  53), INT8_C( -68), INT8_C(-121), INT8_C(  94),
                        INT8_C(-100), INT8_C( -25), INT8_C( 102), INT8_C( 120)),
      UINT8_C(108),
      simde_mm512_set_epi64(INT64_C(-8224370586313845882), INT64_C(-1993121323347227003),
                            INT64_C( 8831696545371448920), INT64_C( 2185815767663877612),
                            INT64_C( -270570670623534210), INT64_C(-7835605414239696560),
                            INT64_C(-1637430132815230795), INT64_C( 7232010829548987022)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  53), INT8_C(-128), INT8_C( 127), INT8_C(  94),
                        INT8_C(-128), INT8_C(-128), INT8_C( 102), INT8_C( 120)) },
    { simde_mm_set_epi8(INT8_C( -93), INT8_C( -72), INT8_C(   3), INT8_C( -86),
                        INT8_C(-106), INT8_C(  54), INT8_C( -93), INT8_C(  77),
                        INT8_C( 111), INT8_C(-126), INT8_C(  23), INT8_C( -40),
                        INT8_C(  -4), INT8_C(  -3), INT8_C( -74), INT8_C(   6)),
      UINT8_C(212),
      simde_mm512_set_epi64(INT64_C(-3956629285800542954), INT64_C( 7768661847448918660),
                            INT64_C( 8414129281940348180), INT64_C(-3282225164507364365),
                            INT64_C( 3972788221445796445), INT64_C(-1066584669561441235),
                            INT64_C( 2182845185502488635), INT64_C(-2591044625591562731)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-128), INT8_C( 127), INT8_C(  23), INT8_C(-128),
                        INT8_C(  -4), INT8_C(-128), INT8_C( -74), INT8_C(   6)) },
    { simde_mm_set_epi8(INT8_C(  95), INT8_C( 126), INT8_C( 102), INT8_C(  -2),
                        INT8_C(  37), INT8_C( -94), INT8_C(  95), INT8_C( -94),
                        INT8_C(-125), INT8_C(  -9), INT8_C( -95), INT8_C(  41),
                        INT8_C(  -4), INT8_C( -36), INT8_C( -88), INT8_C(  79)),
      UINT8_C( 80),
      simde_mm512_set_epi64(INT64_C( 5062048906197334415), INT64_C( -115992959878725320),
                            INT64_C( 1136786795492082678), INT64_C(-1948109654186813959),
                            INT64_C(  120681793319816518), INT64_C(-3982161075345546564),
                            INT64_C(-4623140196798581694), INT64_C(-1633613586256931500)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(-125), INT8_C(-128), INT8_C( -95), INT8_C(-128),
                        INT8_C(  -4), INT8_C( -36), INT8_C( -88), INT8_C(  79)) },
    { simde_mm_set_epi8(INT8_C( -59), INT8_C( -75), INT8_C( -72), INT8_C( -42),
                        INT8_C( -55), INT8_C(-119), INT8_C( -60), INT8_C(  67),
                        INT8_C( 109), INT8_C( -73), INT8_C(  91), INT8_C(  51),
                        INT8_C(  78), INT8_C(  83), INT8_C(-115), INT8_C(  62)),
      UINT8_C(249),
      simde_mm512_set_epi64(INT64_C( 5851174439164188465), INT64_C(-6319779881941429071),
                            INT64_C( 1879763181654775792), INT64_C( 4044922446109520806),
                            INT64_C( 1345671618220988839), INT64_C(-8367274893382946539),
                            INT64_C( 4386554571925238747), INT64_C(-2885812676818812761)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C(-128), INT8_C( 127), INT8_C( 127),
                        INT8_C( 127), INT8_C(  83), INT8_C(-115), INT8_C(-128)) },
    { simde_mm_set_epi8(INT8_C(  79), INT8_C(  39), INT8_C(  96), INT8_C(  42),
                        INT8_C( -85), INT8_C( 114), INT8_C(  97), INT8_C( -62),
                        INT8_C(   8), INT8_C(  38), INT8_C(  93), INT8_C( 127),
                        INT8_C(-112), INT8_C( 122), INT8_C(-125), INT8_C( -69)),
      UINT8_C( 46),
      simde_mm512_set_epi64(INT64_C(-5543414139022216695), INT64_C(-7351051862345725962),
                            INT64_C( 7765938089573807601), INT64_C(-8520500957815241015),
                            INT64_C(-4114690540063739335), INT64_C( 2127377446843006162),
                            INT64_C( 4629741254325605495), INT64_C( 6283226406328232321)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   8), INT8_C(  38), INT8_C( 127), INT8_C( 127),
                        INT8_C(-128), INT8_C( 127), INT8_C( 127), INT8_C( -69)) },
    { simde_mm_set_epi8(INT8_C( -81), INT8_C(-127), INT8_C( -86), INT8_C(  87),
                        INT8_C(  54), INT8_C(  74), INT8_C(  -3), INT8_C(  90),
                        INT8_C( -47), INT8_C( -23), INT8_C(  73), INT8_C(  26),
                        INT8_C(  65), INT8_C(  65), INT8_C(  86), INT8_C(-128)),
      UINT8_C(244),
      simde_mm512_set_epi64(INT64_C( 3168330803816226018), INT64_C(-5665190690683619181),
                            INT64_C(-6479275574786056503), INT64_C(-4854337280374936929),
                            INT64_C( 9110007504327473973), INT64_C( 2306966881245130829),
                            INT64_C( 3371747741827789856), INT64_C( 4883552887250537067)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C( 127), INT8_C(-128), INT8_C(-128), INT8_C(-128),
                        INT8_C(  65), INT8_C( 127), INT8_C(  86), INT8_C(-128)) },
    { simde_mm_set_epi8(INT8_C( -36), INT8_C( -15), INT8_C(  94), INT8_C(  96),
                        INT8_C(  41), INT8_C(  71), INT8_C(-125), INT8_C(  50),
                        INT8_C(  -5), INT8_C( -42), INT8_C(   6), INT8_C(  31),
                        INT8_C( -82), INT8_C(  -3), INT8_C( -84), INT8_C( 118)),
      UINT8_C( 84),
      simde_mm512_set_epi64(INT64_C(-5802305007366524132), INT64_C( 6368102336095963395),
                            INT64_C(   26715019440636716), INT64_C( 3396530625665467524),
                            INT64_C( 4010180268457941346), INT64_C( 5473964637387956828),
                            INT64_C( 6527063328512873018), INT64_C( 7600981363267795639)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(  -5), INT8_C( 127), INT8_C(   6), INT8_C( 127),
                        INT8_C( -82), INT8_C( 127), INT8_C( -84), INT8_C( 118)) }
  };

  for (size_t i = 0 ; i < (sizeof(test_vec) / sizeof(test_vec[0])); i++) {
    simde__m128i r = simde_mm512_mask_cvtsepi64_epi8(test_vec[i].src, test_vec[i].k, test_vec[i].a);
    simde_assert_m128i_i8(r, ==, test_vec[i].r);
  }

  return 0;
}

static int
test_simde_mm512_maskz_cvtsepi64_epi8(SIMDE_MUNIT_TEST_ARGS) {
  const struct {
    simde__mmask8 k;
    simde__m512i a;
    simde__m128i r;
  } test_vec[8] = {
    { UINT8_C( 99),
      simde_mm512_set_epi64(INT64_C(-6773163771856001287), INT64_C( 8528895860955669022),
                            INT64_C(  743395091751495893), INT64_C(-5753646205421077345),
                            INT64_C( 6781795189594641427), INT64_C( 7308036127418939455),
                            INT64_C(  616165601606027604), INT64_C(-2694201555471274183)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C( 127), INT8_C( 127), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C( 127), INT8_C(-128)) },
    { UINT8_C( 41),
      simde_mm512_set_epi64(INT64_C( -270570670623534210), INT64_C(-7835605414239696560),
                            INT64_C(-1637430132815230795), INT64_C( 7232010829548987022),
                            INT64_C(  564423766678453237), INT64_C(-7165203031737464994),
                            INT64_C(-7140626015800786754), INT64_C(-2499691201975648498)),
      simde_mm_set_epi8(INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(   0), INT8_C(   0),
                        INT8_C(   0), INT8_C(   0), INT8_C(-128), INT8_C(   0),
                        INT8_C( 127), INT8_C(   0), INT8_C(   0), INT8_C(-128)) },
    { UINT8_C(236),
      simde_mm512_set_epi64(INT64_C( 2182845185502488635), INT64_C(-2591044625591562731),
                            INT64_C(  634126652413576106), INT64_C(-7622725766299379