/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <assert.h>
#include <pthread.h>
#include <semaphore.h>
#include <unistd.h>

#include <sys/select.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <sys/types.h>

#include <iostream>

#define Uses_SCIM_BACKEND
#define Uses_SCIM_IMENGINE
#define Uses_SCIM_IMENGINE_MODULE
#define Uses_SCIM_PANEL_CLIENT

#include <scim.h>

#include "scim-bridge-agent-event-server.h"
#include "scim-bridge-agent-output.h"
#include "scim-bridge-agent-panel-listener.h"
#include "scim-bridge-agent-panel-listener-protected.h"
#include "scim-bridge-agent-protected.h"
#include "scim-bridge-display.h"
#include "scim-bridge-path.h"

#include "event/scim-bridge-agent-panel-pending-event.h"

using std::endl;
using std::ifstream;
using std::ofstream;
using std::vector;

using namespace scim;

/* Class definition */
class ScimBridgeAgentPanelListenerImpl: public ScimBridgeAgentPanelListener
{

    public:

        ScimBridgeAgentPanelListenerImpl (const String &config_name, const ScimBridgeDisplay *new_display, ScimBridgeAgentProtected *new_agent);
        ~ScimBridgeAgentPanelListenerImpl ();

        void read_and_dispatch ();
        void prepare (scim_bridge_imcontext_id_t imcontext_id);
        void send ();

		retval_t launch ();

        /* The following functions are virtually protected */
        void initialize ();

        retval_t create_lockfile ();
        retval_t destroy_lockfile ();
        retval_t update_lockfile ();

        void focus_in (const String &factory_uuid);
        void focus_out ();

        void update_screen ();
        void update_cursor_location (int x, int y);
        void update_factory_info (const PanelFactoryInfo &factory_info);

        void turn_on ();
        void turn_off ();

        void set_aux_string (const WideString &str, const AttributeList &attrs);
        void show_aux_string ();
        void hide_aux_string ();

        void set_lookup_table (const LookupTable &table);
        void show_lookup_table ();
        void hide_lookup_table ();

        void set_preedit_cursor_position (int cursor_pos);
        void set_preedit_string (const WideString &str, const AttributeList &attrs);
        void show_preedit ();
        void hide_preedit ();

        void start_helper (const String &helper_uuid);
        void stop_helper (const String &helper_uuid);
        void send_helper_event (const String &helper_uuid, const Transaction &trans);

        void register_properties (const PropertyList &properties);
        void update_property (const Property &property);

        void show_factory_menu (const vector<PanelFactoryInfo> &menu);
        void show_help (const String &string);

        void register_input_context (scim_bridge_imcontext_id_t id, const String &factory_uuid);
        void remove_input_context (scim_bridge_imcontext_id_t id);

        /* The following functions are virtually private */
        int get_panel_fd ();

        void panel_error_occured ();
        void panel_event_occured ();
        void panel_listener_exited ();
        void read_event ();
        bool is_active ();

    protected:

        void do_close_event_client ();

    private:

        ScimBridgeAgentProtected *agent;

        String scim_config_name;

        ScimBridgeDisplay *display;

        PanelClient scim_panel_client;

        int panel_fd;

        scim_bridge_imcontext_id_t prepared_imcontext_id;

        pthread_t panel_listener_thread;

        sem_t semaphore;

		bool launched;

        void connect_panel_client ();

        /* Panel related callbacks */
        void slot_reload_config (scim_bridge_imcontext_id_t id);
        void slot_exit (scim_bridge_imcontext_id_t id);
        void slot_update_lookup_table_page_size (scim_bridge_imcontext_id_t id, int page_size);
        void slot_lookup_table_page_up (scim_bridge_imcontext_id_t id);
        void slot_lookup_table_page_down (scim_bridge_imcontext_id_t id);
        void slot_trigger_property (scim_bridge_imcontext_id_t id, const scim::String &property);
        void slot_process_helper_event (scim_bridge_imcontext_id_t id, const String &target_uuid, const String &helper_uuid, const Transaction &trans);
        void slot_move_preedit_caret (scim_bridge_imcontext_id_t id, int caret_pos);
        void slot_select_candidate (scim_bridge_imcontext_id_t id, int cand_index);
        void slot_process_key_event (scim_bridge_imcontext_id_t id, const KeyEvent &key_event);
        void slot_commit (scim_bridge_imcontext_id_t id, const WideString &wstr);
        void slot_forward_key_event (scim_bridge_imcontext_id_t id, const KeyEvent &key_event);
        void slot_request_help (scim_bridge_imcontext_id_t id);
        void slot_request_factory_menu (scim_bridge_imcontext_id_t id);
        void slot_change_factory (scim_bridge_imcontext_id_t id, const String &uuid);

};

/* Helper functions */
void *run_panel_listener (void *arg)
{
    ScimBridgeAgentPanelListenerImpl *panel_listener = static_cast<ScimBridgeAgentPanelListenerImpl*> (arg);

    timeval previous_time;
    gettimeofday (&previous_time, NULL);

    while (panel_listener->is_active ()) {
        int panel_fd = panel_listener->get_panel_fd ();

        timeval timeout;
        timeout.tv_sec = 60 * 60;
        timeout.tv_usec = 0;

        fd_set read_set;
        FD_ZERO (&read_set);
        FD_SET (panel_fd, &read_set);

        fd_set error_set;
        FD_ZERO (&error_set);
        FD_SET (panel_fd, &error_set);
        const int ret = select (panel_fd + 1, &read_set, NULL, &error_set, &timeout);

        timeval current_time;
        gettimeofday (&current_time, NULL);

        if ((current_time.tv_sec - previous_time.tv_sec) + (current_time.tv_usec - previous_time.tv_usec) / 1000000 >= 60 * 60) {
            panel_listener->update_lockfile ();
            previous_time = current_time;
        }

        if (ret > 0 && panel_listener->is_active ()) {
            if (FD_ISSET (panel_fd, &error_set)) {
                panel_listener->panel_error_occured ();
            } else {
                panel_listener->panel_event_occured ();
            }
        }
    }

    panel_listener->panel_listener_exited ();
    return NULL;
}


/* Implementations */
ScimBridgeAgentPanelListener *ScimBridgeAgentPanelListener::alloc (const String &config_name, const ScimBridgeDisplay *new_display, ScimBridgeAgentProtected *new_agent)
{
    ScimBridgeAgentPanelListenerImpl *panel_listener = new ScimBridgeAgentPanelListenerImpl (config_name, new_display, new_agent);
    if (panel_listener->create_lockfile ()) {
        delete panel_listener;
        return NULL;
    } else {
        panel_listener->initialize ();
        return panel_listener;
    }
}


ScimBridgeAgentPanelListenerImpl::ScimBridgeAgentPanelListenerImpl (const String &config_name, const ScimBridgeDisplay *new_display, ScimBridgeAgentProtected *new_agent):
agent (new_agent), scim_config_name (config_name), panel_fd (-1), prepared_imcontext_id (-1), launched (false)
{
    display = scim_bridge_alloc_display ();
    scim_bridge_copy_display (display, new_display);

    sem_init (&semaphore, 0, 0);
}


ScimBridgeAgentPanelListenerImpl::~ScimBridgeAgentPanelListenerImpl ()
{
    scim_bridge_free_display (display);
    sem_destroy (&semaphore);
}


void ScimBridgeAgentPanelListenerImpl::initialize ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_SCIM, 4, "Initializing panel listener...");

    scim_panel_client.signal_connect_reload_config (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_reload_config));
    scim_panel_client.signal_connect_exit (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_exit));
    scim_panel_client.signal_connect_update_lookup_table_page_size (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_update_lookup_table_page_size));
    scim_panel_client.signal_connect_lookup_table_page_up (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_lookup_table_page_up));
    scim_panel_client.signal_connect_lookup_table_page_down (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_lookup_table_page_down));
    scim_panel_client.signal_connect_trigger_property (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_trigger_property));
    scim_panel_client.signal_connect_process_helper_event (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_process_helper_event));
    scim_panel_client.signal_connect_move_preedit_caret (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_move_preedit_caret));
    scim_panel_client.signal_connect_select_candidate (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_select_candidate));
    scim_panel_client.signal_connect_process_key_event (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_process_key_event));
    scim_panel_client.signal_connect_commit_string (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_commit));
    scim_panel_client.signal_connect_forward_key_event (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_forward_key_event));
    scim_panel_client.signal_connect_request_help (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_request_help));
    scim_panel_client.signal_connect_request_factory_menu (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_request_factory_menu));
    scim_panel_client.signal_connect_change_factory (slot (this, &ScimBridgeAgentPanelListenerImpl::slot_change_factory));
}


retval_t ScimBridgeAgentPanelListenerImpl::launch ()
{
	assert (!launched);
	launched = true;

    connect_panel_client ();
    
	if (pthread_create (&panel_listener_thread, NULL, &run_panel_listener, this)) {
        scim_bridge_psyslogln (SYSLOG_CRITICAL, "Cannot invoke the panel listener thread");
        return RETVAL_FAILED;
    } else {
		return RETVAL_SUCCEEDED;
	}
}


retval_t ScimBridgeAgentPanelListenerImpl::create_lockfile ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Creating the lockfile...");

    const char *lockfile_path = scim_bridge_path_get_lockfile ();
    ifstream input_stream (lockfile_path);

    pid_t pid = 0;
    input_stream >> pid;

    input_stream.close ();

    if (input_stream.good ()) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Pinging for the process: pid = %d", pid);
        if (kill (pid, 0) == 0) {
            scim_bridge_psyslogln (SYSLOG_CRITICAL, "Another agent is running. I'll exit.");
            return RETVAL_FAILED;
        }
    }

    if (update_lockfile ()) {
        return RETVAL_FAILED;
    } else {
        return RETVAL_SUCCEEDED;
    }
}


retval_t ScimBridgeAgentPanelListenerImpl::update_lockfile ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 1, "Updating the lockfile...");

    const char *lockfile_path = scim_bridge_path_get_lockfile ();
    ofstream output_stream (lockfile_path);

    output_stream << getpid () << endl;
    output_stream.close ();

    if (!output_stream.good ()) {
        scim_bridge_perrorln ("Cannot update the lockfile");
        return RETVAL_FAILED;
    } else {
        return RETVAL_SUCCEEDED;
    }
}


retval_t ScimBridgeAgentPanelListenerImpl::destroy_lockfile ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Destroying the lockfile...");

    const char *lockfile_path = scim_bridge_path_get_lockfile ();
    unlink (lockfile_path);

    return RETVAL_SUCCEEDED;
}


void ScimBridgeAgentPanelListenerImpl::panel_listener_exited ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 5, "Panel listener exited");
    panel_fd = -2;
}


void ScimBridgeAgentPanelListenerImpl::connect_panel_client ()
{
    const char *display_name = scim_bridge_display_get_name (display);
    if (display_name == NULL) {
        scim_bridge_psyslogln (SYSLOG_ERROR, "Cannot get the variable, 'DISPLAY'");
        panel_fd = -1;
    }
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_SCIM, 3, "'DISPLAY' = '%s'", display_name);

    if (scim_panel_client.open_connection (scim_config_name, String (display_name)) >= 0) {
        panel_fd = scim_panel_client.get_connection_number ();
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_SCIM, 3, "Panel fd = %d", panel_fd);
    } else {
        scim_bridge_psyslogln (SYSLOG_ERROR, "Failed to open the panel socket");
        panel_fd = -1;
    }
}


void ScimBridgeAgentPanelListenerImpl::panel_error_occured ()
{
    scim_bridge_psyslogln (SYSLOG_WARNING, "Panel client has been downed...");

    shutdown (panel_fd, SHUT_RDWR);
    scim_panel_client.close_connection ();
    panel_fd = -1;

    connect_panel_client ();
}


void ScimBridgeAgentPanelListenerImpl::panel_event_occured ()
{
    get_event_server ()->push_event (new ScimBridgeAgentPanelPendingEvent ());
    sem_wait (&semaphore);
}


void ScimBridgeAgentPanelListenerImpl::read_and_dispatch ()
{
    fd_set read_set;
    FD_ZERO (&read_set);
    FD_SET (panel_fd, &read_set);

    timeval timeout;
    timeout.tv_sec = 0;
    timeout.tv_usec = 0;

    if (select (panel_fd + 1, &read_set, NULL, NULL, &timeout) > 0) {
        if (!scim_panel_client.filter_event ()) panel_error_occured ();
    }
    sem_post (&semaphore);
}


int ScimBridgeAgentPanelListenerImpl::get_panel_fd ()
{
    return panel_fd;
}


bool ScimBridgeAgentPanelListenerImpl::is_active ()
{
    return get_event_client_id () != -2;
}


void ScimBridgeAgentPanelListenerImpl::do_close_event_client ()
{
    const int tmp_fd = panel_fd;
    panel_fd = -1;
    shutdown (tmp_fd, SHUT_RDWR);
    scim_panel_client.close_connection ();

    while (panel_fd != -2) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_AGENT, 3, "Killing panel listener...");
        sem_post (&semaphore);
        pthread_kill (panel_listener_thread, SIGNOTIFY);
        usleep (50);
    }
    sem_post (&semaphore);
    pthread_join (panel_listener_thread, NULL);

    destroy_lockfile ();
}


void ScimBridgeAgentPanelListenerImpl::prepare (scim_bridge_imcontext_id_t imcontext_id)
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_SCIM, 5, "panel prepare");
    if (prepared_imcontext_id != -1) {
        scim_bridge_perrorln ("Panel client has been already prepared");
        return;
    }

    prepared_imcontext_id = imcontext_id;
    scim_panel_client.prepare (prepared_imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::send ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_SCIM, 5, "panel send");
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
    } else {
        scim_panel_client.send ();
        prepared_imcontext_id = -1;
    }
}


void ScimBridgeAgentPanelListenerImpl::focus_in (const String &factory_uuid)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.focus_in (prepared_imcontext_id, factory_uuid);
}


void ScimBridgeAgentPanelListenerImpl::focus_out ()
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.focus_out (prepared_imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::update_screen ()
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.update_screen (prepared_imcontext_id, scim_bridge_display_get_screen_number (display));
}


void ScimBridgeAgentPanelListenerImpl::update_cursor_location (int x, int y)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.update_spot_location (prepared_imcontext_id, x, y);
}


void ScimBridgeAgentPanelListenerImpl::update_factory_info (const PanelFactoryInfo &factory_info)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.update_factory_info (prepared_imcontext_id, factory_info);
}


void ScimBridgeAgentPanelListenerImpl::turn_on ()
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.turn_on (prepared_imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::turn_off ()
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.turn_off (prepared_imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::show_aux_string ()
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.show_aux_string (prepared_imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::hide_aux_string ()
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.hide_aux_string (prepared_imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::set_aux_string (const WideString &str, const AttributeList &attrs)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.update_aux_string (prepared_imcontext_id, str, attrs);
}


void ScimBridgeAgentPanelListenerImpl::set_lookup_table (const LookupTable &table)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.update_lookup_table (prepared_imcontext_id, table);
}


void ScimBridgeAgentPanelListenerImpl::show_lookup_table ()
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.show_lookup_table (prepared_imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::hide_lookup_table ()
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.hide_lookup_table (prepared_imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::set_preedit_cursor_position (int cursor_pos)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.update_preedit_caret (prepared_imcontext_id, cursor_pos);
}


void ScimBridgeAgentPanelListenerImpl::set_preedit_string (const WideString &str, const AttributeList &attrs)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.update_preedit_string (prepared_imcontext_id, str, attrs);
}


void ScimBridgeAgentPanelListenerImpl::show_preedit ()
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.show_preedit_string (prepared_imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::hide_preedit ()
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.hide_preedit_string (prepared_imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::register_properties (const PropertyList &properties)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.register_properties (prepared_imcontext_id, properties);
}


void ScimBridgeAgentPanelListenerImpl::update_property (const Property &property)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.update_property (prepared_imcontext_id, property);
}


void ScimBridgeAgentPanelListenerImpl::start_helper (const String &helper_uuid)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.start_helper (prepared_imcontext_id, helper_uuid);
}


void ScimBridgeAgentPanelListenerImpl::stop_helper (const String &helper_uuid)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.stop_helper (prepared_imcontext_id, helper_uuid);
}


void ScimBridgeAgentPanelListenerImpl::send_helper_event (const String &helper_uuid, const Transaction &trans)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.send_helper_event (prepared_imcontext_id, helper_uuid, trans);
}


void ScimBridgeAgentPanelListenerImpl::show_factory_menu (const vector<PanelFactoryInfo> &menu)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.show_factory_menu (prepared_imcontext_id, menu);
}


void ScimBridgeAgentPanelListenerImpl::show_help (const scim::String &string)
{
    if (prepared_imcontext_id == -1) {
        scim_bridge_perrorln ("Panel client has not yet been prepared");
        return;
    }
    scim_panel_client.show_help (prepared_imcontext_id, string);
}


void ScimBridgeAgentPanelListenerImpl::register_input_context (scim_bridge_imcontext_id_t id, const String &factory_uuid)
{
    scim_panel_client.register_input_context (id, factory_uuid);
}


void ScimBridgeAgentPanelListenerImpl::remove_input_context (scim_bridge_imcontext_id_t id)
{
    scim_panel_client.remove_input_context (id);
}


/* Slot functions */
void ScimBridgeAgentPanelListenerImpl::slot_reload_config (scim_bridge_imcontext_id_t imcontext_id)
{
    agent->reload_config ();
}


void ScimBridgeAgentPanelListenerImpl::slot_exit (scim_bridge_imcontext_id_t imcontext_id)
{
    scim_bridge_perrorln ("FIXME: not implemented yet: ScimBridgeAgentPanelListenerImpl::slot_exit ()");
    scim_bridge_psyslogln (SYSLOG_WARNING, "FIXME: not implemented yet: ScimBridgeAgentPanelListenerImpl::slot_exit ()");
}


void ScimBridgeAgentPanelListenerImpl::slot_update_lookup_table_page_size (scim_bridge_imcontext_id_t imcontext_id, int page_size)
{
    agent->update_lookup_table_page_size (imcontext_id, page_size);
}


void ScimBridgeAgentPanelListenerImpl::slot_lookup_table_page_up (scim_bridge_imcontext_id_t imcontext_id)
{
    agent->lookup_table_page_up (imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::slot_lookup_table_page_down (scim_bridge_imcontext_id_t imcontext_id)
{
    agent->lookup_table_page_down (imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::slot_trigger_property (scim_bridge_imcontext_id_t imcontext_id, const scim::String &property)
{
    agent->trigger_property (imcontext_id, property);
}


void ScimBridgeAgentPanelListenerImpl::slot_process_helper_event (scim_bridge_imcontext_id_t imcontext_id, const String &target_uuid, const String &helper_uuid, const Transaction &trans)
{
    agent->process_helper_event (imcontext_id, target_uuid, helper_uuid, trans);
}


void ScimBridgeAgentPanelListenerImpl::slot_move_preedit_caret (scim_bridge_imcontext_id_t imcontext_id, int caret_pos)
{
    agent->panel_move_preedit_caret (imcontext_id, caret_pos);
}


void ScimBridgeAgentPanelListenerImpl::slot_select_candidate (scim_bridge_imcontext_id_t imcontext_id, int cand_index)
{
    agent->lookup_table_select_candidate (imcontext_id, cand_index);
}


void ScimBridgeAgentPanelListenerImpl::slot_process_key_event (scim_bridge_imcontext_id_t imcontext_id, const KeyEvent &key_event)
{
    agent->panel_process_key_event (imcontext_id, key_event);
}


void ScimBridgeAgentPanelListenerImpl::slot_commit (scim_bridge_imcontext_id_t imcontext_id, const WideString &wstr)
{
    agent->panel_commit (imcontext_id, wstr);
}


void ScimBridgeAgentPanelListenerImpl::slot_forward_key_event (scim_bridge_imcontext_id_t imcontext_id, const KeyEvent &key_event)
{
    agent->panel_forward_key_event (imcontext_id, key_event);
}


void ScimBridgeAgentPanelListenerImpl::slot_request_help (scim_bridge_imcontext_id_t imcontext_id)
{
    agent->panel_request_help (imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::slot_request_factory_menu (scim_bridge_imcontext_id_t imcontext_id)
{
    agent->panel_request_factory_menu (imcontext_id);
}


void ScimBridgeAgentPanelListenerImpl::slot_change_factory (scim_bridge_imcontext_id_t imcontext_id, const String &uuid)
{
    agent->panel_change_factory (imcontext_id, uuid);
}
