/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

#include <alloca.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>

#include <sys/socket.h>
#include <sys/types.h>
#include <sys/un.h>

#include "scim-bridge-client.h"
#include "scim-bridge-client-imcontext.h"
#include "scim-bridge-client-protected.h"
#include "scim-bridge-exception.h"
#include "scim-bridge-messenger.h"
#include "scim-bridge-output.h"
#include "scim-bridge-path.h"
#include "scim-bridge-string.h"

/* Private data type */
typedef enum _key_event_status_t
{
    KEY_EVENT_PENDING,
    KEY_EVENT_IGNORED,
    KEY_EVENT_CONSUMED
} key_event_status_t;

/* Private variables */
static key_event_status_t key_event_status;

static scim_bridge_imcontext_id_t pending_imcontext_id;

static ScimBridgeMessenger *messenger;

static boolean active = FALSE;

static int socket_fd;

/* Helper Functions */
static retval_t launch_agent ()
{
    scim_bridge_exception_clear ();

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 1, "Invoking the agent...");
    if (system ("scim-bridge-agent") != 0) {
        scim_bridge_exception_occured (SYSTEM_EXCEPTION, "Failed to invoking the agent: %s", strerror (errno));
        scim_bridge_exception_push_stack ("launch_agent ()");
        return RETVAL_FAILED;
    } else {
        return RETVAL_SUCCEEDED;
    }
}


static retval_t open_connection ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 8, "open_connection ()");
    scim_bridge_exception_clear ();

    int i;
    for (i = 0; i < 5; ++i) {
        socket_fd = socket (PF_UNIX, SOCK_STREAM, 0);
        if (socket_fd < 0) {
            scim_bridge_exception_occured (IO_EXCEPTION, "Failed to create the message socket: %s", strerror (errno));
            scim_bridge_exception_push_stack ("open_connection ()");
            return RETVAL_FAILED;
        }

        struct sockaddr_un socket_addr;
        memset (&socket_addr, 0, sizeof (struct sockaddr_un));
        socket_addr.sun_family = AF_UNIX;
        strcpy (socket_addr.sun_path, scim_bridge_path_get_socket ());
    
        if (connect (socket_fd, (struct sockaddr*)&socket_addr, sizeof (socket_addr.sun_family) + strlen (socket_addr.sun_path)) != 0) {
            if (i == 0 && launch_agent ()) {
                scim_bridge_exception_push_stack ("open_connection ()");
                return RETVAL_FAILED;
            } else {
                scim_bridge_exception_occured (IO_EXCEPTION, "Failed to connect the message socket: %s", strerror (errno));
                scim_bridge_exception_push_stack ("open_connection ()");
                socket_fd = -1;
                usleep (500000);
            }
        } else {
            return RETVAL_SUCCEEDED;
        }
    }

    return RETVAL_FAILED;
}


static void connection_lost ()
{
    scim_bridge_client_finalize ();
    scim_bridge_client_connection_lost ();
}


/* Message Hundlers */
static retval_t received_message_unknown (const ScimBridgeMessage *message)
{
    const char *message_header = scim_bridge_message_get_header (message);
    scim_bridge_perrorln ("Unknown message received: %s", message_header);

    return RETVAL_SUCCEEDED;
}


static retval_t received_message_imcontext_allocated (const ScimBridgeMessage *message)
{
    const char *ic_id_str = scim_bridge_message_get_argument (message, 0);

    int ic_id;
    if (scim_bridge_string_to_int (&ic_id, ic_id_str)) {
        scim_bridge_perrorln ("Invalid arguments for the message: imcontext_allocated (%s)", ic_id_str);
        pending_imcontext_id = -2;
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'imcontext_allocated' message:ic_id = %d", ic_id);
        if (ic_id < 0) {
            pending_imcontext_id = -2;
        } else {
            pending_imcontext_id = ic_id;
        }
    }

    return RETVAL_SUCCEEDED;
}


static retval_t received_message_imcontext_freed (const ScimBridgeMessage *message)
{
    const char *ic_id_str = scim_bridge_message_get_argument (message, 0);

    int ic_id;
    if (scim_bridge_string_to_int (&ic_id, ic_id_str)) {
        scim_bridge_perrorln ("Invalid arguments for the message: imcontext_freed (%s)", ic_id_str);
        pending_imcontext_id = -2;
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'imcontext_freed' message:ic_id = %d", ic_id);
        if (ic_id == pending_imcontext_id) {
            pending_imcontext_id = -1;
        } else {
            pending_imcontext_id = -2;
        }
    }

    return RETVAL_SUCCEEDED;
}


static retval_t received_message_key_event_handled (const ScimBridgeMessage *message)
{
    const char *consumed_str = scim_bridge_message_get_argument (message, 0);

    boolean consumed;
    if (scim_bridge_string_to_boolean (&consumed, consumed_str)) {
        scim_bridge_perrorln ("Invalid arguments for the message: key_event_handled (%s)", consumed_str);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'key_event_handled' message: consumed = %s", consumed ? "TRUE":"FALSE");
        if (consumed) {
            key_event_status = KEY_EVENT_CONSUMED;
        } else {
            key_event_status = KEY_EVENT_IGNORED;
        }
    }

    return RETVAL_SUCCEEDED;
}


static retval_t received_message_commit (const ScimBridgeMessage *message)
{
    const char *ic_id_str = scim_bridge_message_get_argument (message, 0);
    const char *string = scim_bridge_message_get_argument (message, 1);

    scim_bridge_imcontext_id_t ic_id;
    if (scim_bridge_string_to_int (&ic_id, ic_id_str)) {
        scim_bridge_perrorln ("Invalid arguments for the message: commit (%s, %s)", ic_id_str, string);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'commit' message: ic_id = %d, string = \"%s\"", ic_id, string);

        ScimBridgeClientIMContext *imcontext = scim_bridge_client_find_imcontext (ic_id);
        if (imcontext != NULL) {
            scim_bridge_client_imcontext_set_commit_string (imcontext, string);
            scim_bridge_client_imcontext_commit (imcontext);
        } else {
            scim_bridge_perrorln ("No such imcontext: id = %d", ic_id);
        }
    }

    return RETVAL_SUCCEEDED;
}


static retval_t received_message_set_preedit_shown (const ScimBridgeMessage *message)
{
    const char *ic_id_str = scim_bridge_message_get_argument (message, 0);
    const char *shown_str = scim_bridge_message_get_argument (message, 1);

    scim_bridge_imcontext_id_t ic_id;
    boolean shown;
    if (scim_bridge_string_to_int (&ic_id, ic_id_str) || scim_bridge_string_to_boolean (&shown, shown_str)) {
        scim_bridge_perrorln ("Invalid arguments for the message: set_preedit_shown (%s, %s)", ic_id_str, shown_str);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'set preedit shown' message: ic_id = %u, shown = %s", ic_id, shown ? "TRUE":"FALSE");

        ScimBridgeClientIMContext *imcontext = scim_bridge_client_find_imcontext (ic_id);
        if (imcontext != NULL) {
            scim_bridge_client_imcontext_set_preedit_shown (imcontext, shown);
        } else {
            scim_bridge_perrorln ("No such imcontext: id = %d", ic_id);
        }
    }

    return RETVAL_SUCCEEDED;
}


static retval_t received_message_set_preedit_string (const ScimBridgeMessage *message)
{
    const char *ic_id_str = scim_bridge_message_get_argument (message, 0);
    const char *preedit_string = scim_bridge_message_get_argument (message, 1);

    scim_bridge_imcontext_id_t ic_id;
    if (scim_bridge_string_to_int (&ic_id, ic_id_str)) {
        scim_bridge_perrorln ("Invalid arguments for the message: set_preedit_string (%s, %s)", ic_id_str, preedit_string);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'set preedit string' message: ic_id =%u, preedit_string = \"%s\"", ic_id, preedit_string);

        ScimBridgeClientIMContext *imcontext = scim_bridge_client_find_imcontext (ic_id);

        if (imcontext != NULL) {
            scim_bridge_client_imcontext_set_preedit_string (imcontext, preedit_string);
        } else {
            scim_bridge_perrorln ("No such imcontext: id = %d", ic_id);
        }
    }

    return RETVAL_SUCCEEDED;
}


static retval_t received_message_set_preedit_attributes (const ScimBridgeMessage *message)
{
    const char *ic_id_str = scim_bridge_message_get_argument (message, 0);

    scim_bridge_imcontext_id_t ic_id;
    if (scim_bridge_string_to_int (&ic_id, ic_id_str) || scim_bridge_message_get_argument_count (message) % 4 != 1) {
        scim_bridge_perrorln ("Invalid arguments for the message: set_preedit_attributes (%s,...)", ic_id_str);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'set preedit attributes' message: ic_id =%u", ic_id);

        const size_t attribute_count = (scim_bridge_message_get_argument_count (message) - 1) / 4;
        ScimBridgeAttribute **attributes = alloca (sizeof (ScimBridgeAttribute*) * attribute_count);

        int i;
        for (i = 0; i < attribute_count; ++i) {
            attributes[i] = scim_bridge_alloc_attribute ();
            ScimBridgeAttribute *attribute = attributes[i];

            const char *attribute_begin_str = scim_bridge_message_get_argument (message, i * 4 + 1);
            const char *attribute_end_str = scim_bridge_message_get_argument (message, i * 4 + 2);
            const char *attribute_type_str = scim_bridge_message_get_argument (message, i * 4 + 3);
            const char *attribute_value_str = scim_bridge_message_get_argument (message, i * 4 + 4);

            unsigned int attribute_begin;
            unsigned int attribute_end;
            if (scim_bridge_string_to_uint (&attribute_begin, attribute_begin_str) || scim_bridge_string_to_uint (&attribute_end, attribute_end_str)) {
                scim_bridge_perrorln ("Invalid range for an attribute: begin = \"%s\", end = \"%s\"", attribute_begin_str, attribute_end_str);
                scim_bridge_attribute_set_begin (attribute, 0);
                scim_bridge_attribute_set_end (attribute, 0);
                scim_bridge_attribute_set_type (attribute, ATTRIBUTE_NONE);
                scim_bridge_attribute_set_value (attribute, SCIM_BRIDGE_ATTRIBUTE_DECORATE_NONE);
                continue;
            }

            scim_bridge_attribute_set_begin (attribute, attribute_begin);
            scim_bridge_attribute_set_end (attribute, attribute_end);

            if (strcmp (attribute_type_str, SCIM_BRIDGE_MESSAGE_NONE) == 0) {
                scim_bridge_attribute_set_type (attribute, ATTRIBUTE_NONE);
            } else if (strcmp (attribute_type_str, SCIM_BRIDGE_MESSAGE_DECORATE) == 0) {
                scim_bridge_attribute_set_type (attribute, ATTRIBUTE_DECORATE);

                if (strcmp (attribute_value_str, SCIM_BRIDGE_MESSAGE_HIGHLIGHT) == 0) {
                    scim_bridge_attribute_set_value (attribute, SCIM_BRIDGE_ATTRIBUTE_DECORATE_HIGHLIGHT);
                } else if (strcmp (attribute_value_str, SCIM_BRIDGE_MESSAGE_UNDERLINE) == 0) {
                    scim_bridge_attribute_set_value (attribute, SCIM_BRIDGE_ATTRIBUTE_DECORATE_UNDERLINE);
                } else if (strcmp (attribute_value_str, SCIM_BRIDGE_MESSAGE_REVERSE) == 0) {
                    scim_bridge_attribute_set_value (attribute, SCIM_BRIDGE_ATTRIBUTE_DECORATE_REVERSE);
                } else {
                    scim_bridge_perrorln ("Unknown decoration for the attribute: %s", attribute_value_str);
                    scim_bridge_attribute_set_type (attribute, ATTRIBUTE_NONE);
                    scim_bridge_attribute_set_value (attribute, SCIM_BRIDGE_ATTRIBUTE_DECORATE_NONE);
                }

            } else if (strcmp (attribute_type_str, SCIM_BRIDGE_MESSAGE_FOREGROUND) == 0
                || strcmp (attribute_type_str, SCIM_BRIDGE_MESSAGE_BACKGROUND) == 0) {

                if (strcmp (attribute_type_str, SCIM_BRIDGE_MESSAGE_FOREGROUND) == 0) {
                    scim_bridge_attribute_set_type (attribute, ATTRIBUTE_FOREGROUND);
                } else {
                    scim_bridge_attribute_set_type (attribute, ATTRIBUTE_BACKGROUND);
                }

                unsigned int color_val = 0x1000000;
                if (strncmp (attribute_value_str, SCIM_BRIDGE_MESSAGE_COLOR, strlen (SCIM_BRIDGE_MESSAGE_COLOR)) == 0) {
                    const char *color_str = attribute_value_str + strlen (SCIM_BRIDGE_MESSAGE_COLOR);
                    const size_t color_str_length = strlen (color_str);

                    if (color_str_length == 9) {
                        int j;
                        for (j = 0; color_val < 0x1000000 && j < 9; ++j) {
                            color_val <<= 8;
                            switch (color_str[j]) {
                                case '0':
                                    color_val += 0;
                                    break;
                                case '1':
                                    color_val += 1;
                                    break;
                                case '2':
                                    color_val += 2;
                                    break;
                                case '3':
                                    color_val += 3;
                                    break;
                                case '4':
                                    color_val += 4;
                                    break;
                                case '5':
                                    color_val += 5;
                                    break;
                                case '6':
                                    color_val += 6;
                                    break;
                                case '7':
                                    color_val += 7;
                                    break;
                                case '8':
                                    color_val += 8;
                                    break;
                                case '9':
                                    color_val += 9;
                                    break;
                                case 'a':
                                case 'A':
                                    color_val += 10;
                                    break;
                                case 'b':
                                case 'B':
                                    color_val += 11;
                                    break;
                                case 'c':
                                case 'C':
                                    color_val += 12;
                                    break;
                                case 'd':
                                case 'D':
                                    color_val += 13;
                                    break;
                                case 'e':
                                case 'E':
                                    color_val += 14;
                                    break;
                                case 'f':
                                case 'F':
                                    color_val += 15;
                                    break;
                                default:
                                    color_val = 0x1000000;
                                    break;
                            }
                        }
                    }
                }

                if (color_val > 0xFFFFFF) {
                    scim_bridge_perrorln ("Invalid string for a color: %s", attribute_value_str);
                    scim_bridge_attribute_set_type (attribute, ATTRIBUTE_NONE);
                    scim_bridge_attribute_set_value (attribute, SCIM_BRIDGE_ATTRIBUTE_DECORATE_NONE);
                } else {
                    const unsigned int red = (color_val && 0xFF0000) >> 16;
                    const unsigned int green = (color_val && 0x00FF00) >> 8;
                    const unsigned int blue = (color_val && 0x0000FF) >> 0;

                    scim_bridge_attribute_set_color (attribute, red, green, blue);
                }
            }
        }

        ScimBridgeClientIMContext *imcontext = scim_bridge_client_find_imcontext (ic_id);
        if (imcontext != NULL) {
            scim_bridge_client_imcontext_set_preedit_attributes (imcontext, attributes, attribute_count);
        } else {
            scim_bridge_perrorln ("No such imcontext: id = %d", ic_id);
        }

        int j;
        for (j = 0; j < attribute_count; ++j) {
            scim_bridge_free_attribute (attributes[j]);
        }
    }

    return RETVAL_SUCCEEDED;
}


static retval_t received_message_set_preedit_cursor_position (const ScimBridgeMessage *message)
{
    const char *ic_id_str = scim_bridge_message_get_argument (message, 0);
    const char *cursor_position_str = scim_bridge_message_get_argument (message, 1);

    scim_bridge_imcontext_id_t ic_id;
    unsigned int cursor_position;
    if (scim_bridge_string_to_int (&ic_id, ic_id_str) || scim_bridge_string_to_uint (&cursor_position, cursor_position_str)) {
        scim_bridge_perrorln ("Invalid arguments for the message: set_preedit_cursor_position (%s, %s)", ic_id_str, cursor_position_str);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'set preedit cursor position' message: ic_id =%u, cursor_position = %u", ic_id, cursor_position);

        ScimBridgeClientIMContext *imcontext = scim_bridge_client_find_imcontext (ic_id);

        if (imcontext != NULL) {
            scim_bridge_client_imcontext_set_preedit_cursor_position (imcontext, cursor_position);
        } else {
            scim_bridge_perrorln ("No such imcontext: id = %d", ic_id);
        }
    }

    return RETVAL_SUCCEEDED;
}


static retval_t received_message_update_preedit (const ScimBridgeMessage *message)
{
    const char *ic_id_str = scim_bridge_message_get_argument (message, 0);

    int ic_id;
    if (scim_bridge_string_to_int (&ic_id, ic_id_str)) {
        scim_bridge_perrorln ("Invalid arguments for the message: update_preedit (%s)", ic_id_str);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'update_preedit' message: ic_id =%u", ic_id);
        ScimBridgeClientIMContext *imcontext = scim_bridge_client_find_imcontext (ic_id);

        if (imcontext != NULL) {
            scim_bridge_client_imcontext_update_preedit (imcontext);
        } else {
            scim_bridge_perrorln ("No such imcontext: id = %d", ic_id);
        }
    }

    return RETVAL_SUCCEEDED;
}


static retval_t received_message_beep (const ScimBridgeMessage *message)
{
    const char *ic_id_str = scim_bridge_message_get_argument (message, 0);

    int ic_id;
    if (scim_bridge_string_to_int (&ic_id, ic_id_str)) {
        scim_bridge_perrorln ("Invalid arguments for the message: beep (%s)", ic_id_str);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'beep' message: ic_id =%u", ic_id);
        ScimBridgeClientIMContext *imcontext = scim_bridge_client_find_imcontext (ic_id);

        if (imcontext != NULL) {
            scim_bridge_client_imcontext_beep (imcontext);
        } else {
            scim_bridge_perrorln ("No such imcontext: id = %d", ic_id);
        }
    }

    return RETVAL_SUCCEEDED;
}


static retval_t received_message_forward_key_event (const ScimBridgeMessage *message)
{
    const char *ic_id_str = scim_bridge_message_get_argument (message, 0);
    const char *key_code_str = scim_bridge_message_get_argument (message, 1);
    const char *key_pressed_str = scim_bridge_message_get_argument (message, 2);

    const int modifier_count = scim_bridge_message_get_argument_count (message) - 3;

    boolean shift_down = FALSE;
    boolean control_down = FALSE;
    boolean alt_down = FALSE;
    boolean meta_down = FALSE;
    boolean super_down = FALSE;
    boolean hyper_down = FALSE;
    boolean caps_lock_down = FALSE;
    boolean num_lock_down = FALSE;
    boolean unknown_down = FALSE;

    int i;
    for (i = 0; i < modifier_count; ++i) {
        const char *modifier_str = scim_bridge_message_get_argument (message, i + 3);

        if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_SHIFT) == 0) {
            shift_down = TRUE;
        } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_CONTROL) == 0) {
            control_down = TRUE;
        } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_ALT) == 0) {
            alt_down = TRUE;
        } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_META) == 0) {
            meta_down = TRUE;
        } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_SUPER) == 0) {
            super_down = TRUE;
        } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_HYPER) == 0) {
            hyper_down = TRUE;
        } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_CAPS_LOCK) == 0) {
            caps_lock_down = TRUE;
        } else if (strcmp (modifier_str, SCIM_BRIDGE_MESSAGE_NUM_LOCK) == 0) {
            num_lock_down = TRUE;
        } else {
            scim_bridge_perrorln ("Unknown modifier: %s", modifier_str);
            unknown_down = TRUE;
        }
    }

    char *modifiers_str = alloca (sizeof (char) * (strlen ("shift+control+alt+meta+super+hyper+caps_lock+num_lock+unknown") + 1));
    size_t modifier_str_index = 0;

    modifiers_str[0] = '\0';
    if (shift_down) {
        strcpy (modifiers_str + modifier_str_index, "shift");
        modifier_str_index += strlen ("shift");
    }
    if (control_down) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+control");
            modifier_str_index += strlen ("+control");
        } else {
            strcpy (modifiers_str + modifier_str_index, "control");
            modifier_str_index += strlen ("control");
        }
    }
    if (alt_down) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+alt");
            modifier_str_index += strlen ("+alt");
        } else {
            strcpy (modifiers_str + modifier_str_index, "alt");
            modifier_str_index += strlen ("alt");
        }
    }
    if (meta_down) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+meta");
            modifier_str_index += strlen ("+meta");
        } else {
            strcpy (modifiers_str + modifier_str_index, "meta");
            modifier_str_index += strlen ("meta");
        }
    }
    if (super_down) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+super");
            modifier_str_index += strlen ("+super");
        } else {
            strcpy (modifiers_str + modifier_str_index, "super");
            modifier_str_index += strlen ("super");
        }
    }
    if (hyper_down) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+hyper");
            modifier_str_index += strlen ("+hyper");
        } else {
            strcpy (modifiers_str + modifier_str_index, "hyper");
            modifier_str_index += strlen ("hyper");
        }
    }
    if (caps_lock_down) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+caps_lock");
            modifier_str_index += strlen ("+caps_lock");
        } else {
            strcpy (modifiers_str + modifier_str_index, "caps_lock");
            modifier_str_index += strlen ("caps_lock");
        }
    }
    if (num_lock_down) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+num_lock");
            modifier_str_index += strlen ("+num_lock");
        } else {
            strcpy (modifiers_str + modifier_str_index, "num_lock");
            modifier_str_index += strlen ("num_lock");
        }
    }
    if (unknown_down) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+unknown");
            modifier_str_index += strlen ("+unknown");
        } else {
            strcpy (modifiers_str + modifier_str_index, "unknown");
            modifier_str_index += strlen ("unknown");
        }
    }

    scim_bridge_imcontext_id_t ic_id;
    unsigned int key_code;
    boolean key_pressed;
    if (scim_bridge_string_to_int (&ic_id, ic_id_str) || scim_bridge_string_to_uint (&key_code, key_code_str)
        || scim_bridge_string_to_boolean (&key_pressed, key_pressed_str)) {

        scim_bridge_perrorln ("Invalid arguments for the message: forward_key_event (%s, %s, %s, %s)", ic_id_str, key_code_str, key_pressed_str, modifiers_str);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'forward_key_event' message: ic_id = %d", ic_id);
        ScimBridgeClientIMContext *imcontext = scim_bridge_client_find_imcontext (ic_id);

        if (imcontext != NULL) {
            ScimBridgeKeyEvent *key_event = scim_bridge_alloc_key_event ();
            scim_bridge_key_event_set_code (key_event, key_code);
            scim_bridge_key_event_set_pressed (key_event, key_pressed);
            scim_bridge_key_event_set_shift_down (key_event, shift_down);
            scim_bridge_key_event_set_control_down (key_event, control_down);
            scim_bridge_key_event_set_alt_down (key_event, alt_down);
            scim_bridge_key_event_set_meta_down (key_event, meta_down);
            scim_bridge_key_event_set_super_down (key_event, super_down);
            scim_bridge_key_event_set_hyper_down (key_event, hyper_down);
            scim_bridge_key_event_set_caps_lock_down (key_event, caps_lock_down);
            scim_bridge_key_event_set_num_lock_down (key_event, num_lock_down);
            scim_bridge_client_imcontext_forward_key_event (imcontext, key_event);
            scim_bridge_free_key_event (key_event);
        } else {
            scim_bridge_perrorln ("No such imcontext: id = %d", ic_id);
        }
    }

    return RETVAL_SUCCEEDED;
}


static retval_t received_message_get_surrounding_text (const ScimBridgeMessage *message)
{
    const char *response_id_str = scim_bridge_message_get_argument (message, 0);
    const char *ic_id_str = scim_bridge_message_get_argument (message, 1);
    const char *before_max_str = scim_bridge_message_get_argument (message, 2);
    const char *after_max_str = scim_bridge_message_get_argument (message, 3);

    unsigned int response_id;
    int ic_id;
    unsigned int before_max;
    unsigned int after_max;

    char *string;
    int cursor_position;

    boolean succeeded = FALSE;
    if (scim_bridge_string_to_uint (&response_id, response_id_str) || scim_bridge_string_to_int (&ic_id, ic_id_str)
        || scim_bridge_string_to_uint (&before_max, before_max_str) || scim_bridge_string_to_uint (&after_max, after_max_str)) {
        scim_bridge_perrorln ("Invalid arguments for the message: get_surrounding_text (%s, %s, %s, %s)", response_id_str, ic_id_str, before_max_str, after_max_str);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'get_surrounding_text' message: ic_id = %d, before <= %u, after <= %u", ic_id, before_max, after_max);
        ScimBridgeClientIMContext *imcontext = scim_bridge_client_find_imcontext (ic_id);

        if (imcontext != NULL) {
            succeeded = scim_bridge_client_imcontext_get_surrounding_text (imcontext, before_max, after_max, &string, &cursor_position);
            if (succeeded) {
                scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "surrounding text = '%s', cursor_position = %d", string, cursor_position);
            } else {
                scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "surrounding text = N/A");
            }
        } else {
            scim_bridge_perrorln ("No such imcontext: id = %d", ic_id);
        }
    }

    ScimBridgeMessage *responsive_message;

    if (succeeded) {
        responsive_message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_SURROUNDING_TEXT_GOTTEN, 4);
        scim_bridge_message_set_argument (responsive_message, 1, SCIM_BRIDGE_MESSAGE_TRUE);

        char *cursor_position_str = alloca (sizeof (char) * (cursor_position / 10 + 2));
        scim_bridge_string_from_uint (&cursor_position_str, cursor_position);
        scim_bridge_message_set_argument (responsive_message, 2, cursor_position_str);

        scim_bridge_message_set_argument (responsive_message, 3, string);
    } else {
        responsive_message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_SURROUNDING_TEXT_GOTTEN, 2);
        scim_bridge_message_set_argument (responsive_message, 1, SCIM_BRIDGE_MESSAGE_FALSE);
    }

    scim_bridge_message_set_argument (responsive_message, 0, response_id_str);

    if (scim_bridge_messenger_send (messenger, responsive_message)) {
        scim_bridge_exception_push_stack ("scim_bridge_client_get_surrounding_text ()");
        connection_lost ();
        scim_bridge_free_message (responsive_message);
        return RETVAL_FAILED;
    }

    scim_bridge_free_message (responsive_message);
    return RETVAL_SUCCEEDED;
}


static retval_t received_message_delete_surrounding_text (const ScimBridgeMessage *message)
{
    const char *response_id_str = scim_bridge_message_get_argument (message, 0);
    const char *ic_id_str = scim_bridge_message_get_argument (message, 1);
    const char *offset_str = scim_bridge_message_get_argument (message, 2);
    const char *length_str = scim_bridge_message_get_argument (message, 3);

    unsigned int response_id;
    int ic_id;
    unsigned int offset;
    unsigned int length;
    boolean need_reset = FALSE;

    boolean succeeded = FALSE;
    if (scim_bridge_string_to_uint (&response_id, response_id_str) || scim_bridge_string_to_int (&ic_id, ic_id_str) || scim_bridge_string_to_uint (&offset, offset_str) || scim_bridge_string_to_uint (&length, length_str)) {
        scim_bridge_perrorln ("Invalid arguments for the message: delete_surrounding_text (%s, %s, %s, %s)", response_id_str, ic_id_str, offset_str, length_str);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 3, "Received 'delete_surrounding_text' message: ic_id = %d, offset = %u, length = %u", ic_id, offset, length);
        ScimBridgeClientIMContext *imcontext = scim_bridge_client_find_imcontext (ic_id);

        if (imcontext != NULL) {
            succeeded = scim_bridge_client_imcontext_delete_surrounding_text (imcontext, offset, length, &need_reset);
        } else {
            scim_bridge_perrorln ("No such imcontext: id = %d", ic_id);
        }
    }

    ScimBridgeMessage *responsive_message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_SURROUNDING_TEXT_DELETED, 3);

    scim_bridge_message_set_argument (responsive_message, 0, response_id_str);
    if (succeeded) {
        scim_bridge_message_set_argument (responsive_message, 1, SCIM_BRIDGE_MESSAGE_TRUE);
    } else {
        scim_bridge_message_set_argument (responsive_message, 1, SCIM_BRIDGE_MESSAGE_FALSE);
    }
    if (need_reset) {
        scim_bridge_message_set_argument (responsive_message, 2, SCIM_BRIDGE_MESSAGE_TRUE);
    } else {
        scim_bridge_message_set_argument (responsive_message, 2, SCIM_BRIDGE_MESSAGE_FALSE);
    }
    if (scim_bridge_messenger_send (messenger, responsive_message)) {
        scim_bridge_exception_push_stack ("scim_bridge_client_delete_surrounding_text ()");
        connection_lost ();
        scim_bridge_free_message (responsive_message);
        return RETVAL_FAILED;
    }

    scim_bridge_free_message (responsive_message);
    return RETVAL_SUCCEEDED;
}


/* Public functions */
retval_t scim_bridge_client_initialize ()
{
    scim_bridge_exception_clear ();

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_initialize");
    if (active) return RETVAL_SUCCEEDED;

    if (open_connection () != 0) {
        scim_bridge_perrorln ("Giveup initializing scim-bridge...orz");
        active = FALSE;
        return RETVAL_FAILED;
    }

    messenger = scim_bridge_alloc_messenger (socket_fd);

    key_event_status = KEY_EVENT_IGNORED;

    pending_imcontext_id = -1;

    active = TRUE;

    return RETVAL_SUCCEEDED;
}


void scim_bridge_client_finalize ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_finalize");
    if (!active) return;

    scim_bridge_free_messenger (messenger);
    messenger = NULL;

    socket_fd = -1;

    active = FALSE;
}


boolean scim_bridge_client_is_active ()
{
    return active;
}


int scim_bridge_client_get_socket_fd ()
{
    if (!active) {
        return -1;
    } else {
        return  socket_fd;
    }
}


retval_t scim_bridge_client_read_and_dispatch ()
{
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 2, "scim_bridge_client_read_and_dispatch");

    scim_bridge_exception_clear ();

    if (!active) {
        scim_bridge_exception_occured (ILLEGAL_STATE_EXCEPTION, "ScimBridge is not active");
        scim_bridge_exception_push_stack ("scim_bridge_client_allocate_imcontext ()");
        return RETVAL_FAILED;
    }

    fd_set readset;
    FD_ZERO (&readset);
    FD_SET (socket_fd, &readset);

    const int select_retval = select (socket_fd + 1, &readset, NULL, NULL, NULL);
    if (select_retval < 0) {
        scim_bridge_exception_occured (IO_EXCEPTION, "An exception occured while polling the message socket: %s", strerror (errno));
        scim_bridge_exception_push_stack ("scim_bridge_client_read_and_dispatch ()");
        return RETVAL_FAILED;
    } else if (select_retval == 0) {
        return RETVAL_SUCCEEDED;
    }

    ScimBridgeMessage *message;
    if (scim_bridge_messenger_receive (messenger, &message)) return RETVAL_FAILED;

    const char *message_header = scim_bridge_message_get_header (message);

    if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_COMMIT) == 0) {
        received_message_commit (message);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_SET_PREEDIT_SHOWN) == 0) {
        received_message_set_preedit_shown (message);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_SET_PREEDIT_STRING) == 0) {
        received_message_set_preedit_string (message);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_SET_PREEDIT_CURSOR_POSITION) == 0) {
        received_message_set_preedit_cursor_position (message);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_SET_PREEDIT_ATTRIBUTES) == 0) {
        received_message_set_preedit_attributes (message);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_UPDATE_PREEDIT) == 0) {
        received_message_update_preedit (message);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_KEY_EVENT_HANDLED) == 0) {
        received_message_key_event_handled (message);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_IMCONTEXT_ALLOCATED) == 0) {
        received_message_imcontext_allocated (message);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_IMCONTEXT_FREED) == 0) {
        received_message_imcontext_freed (message);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_FORWARD_KEY_EVENT) == 0) {
        received_message_forward_key_event (message);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_BEEP) == 0) {
        received_message_beep (message);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_GET_SURROUNDING_TEXT) == 0) {
        received_message_get_surrounding_text (message);
    } else if (strcmp (message_header, SCIM_BRIDGE_MESSAGE_DELETE_SURROUNDING_TEXT) == 0) {
        received_message_delete_surrounding_text (message);
    } else {
        received_message_unknown (message);
    }

    scim_bridge_free_message (message);
    return RETVAL_SUCCEEDED;
}


/* Called from GUI through IMContext */
retval_t scim_bridge_client_alloc_imcontext (ScimBridgeClientIMContext *imcontext)
{
    scim_bridge_exception_clear ();

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_alloc_imcontext");

    if (!active) {
        scim_bridge_exception_occured (ILLEGAL_STATE_EXCEPTION, "ScimBridge is not active");
        scim_bridge_exception_push_stack ("scim_bridge_client_allocate_imcontext ()");
        return RETVAL_FAILED;
    }

    pending_imcontext_id = -1;

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 5, "Sending 'alloc_imcontext' message");
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_ALLOC_IMCONTEXT, 0);

    if (scim_bridge_messenger_send (messenger, message)) {
        scim_bridge_exception_push_stack ("scim_bridge_client_alloc_imcontext ()");
        connection_lost ();
        scim_bridge_free_message (message);
        return RETVAL_FAILED;
    }
    scim_bridge_free_message (message);

    while (pending_imcontext_id == -1) {
        if (scim_bridge_client_read_and_dispatch ()) {
            scim_bridge_exception_push_stack ("scim_bridge_client_alloc_imcontext ()");
            connection_lost ();

            return RETVAL_FAILED;
        }
    }

    if (pending_imcontext_id < 0) {
        scim_bridge_exception_occured (ILLEGAL_STATE_EXCEPTION, "Failed to allocate imcontext");
        scim_bridge_exception_push_stack ("scim_bridge_client_allocate_imcontext ()");
        return RETVAL_FAILED;
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "allocated: id = %d", pending_imcontext_id);
        scim_bridge_client_imcontext_set_id (imcontext, pending_imcontext_id);
        return RETVAL_SUCCEEDED;
    }
}


retval_t scim_bridge_client_free_imcontext (ScimBridgeClientIMContext *imcontext)
{
    scim_bridge_exception_clear ();

    const scim_bridge_imcontext_id_t id = scim_bridge_client_imcontext_get_id (imcontext);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_free_imcontext: ic = %d", id);

    if (!active) {
        scim_bridge_exception_occured (ILLEGAL_STATE_EXCEPTION, "ScimBridge is not active");
        scim_bridge_exception_push_stack ("scim_bridge_client_free_imcontext ()");
        return RETVAL_FAILED;
    }

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 5, "Sending 'free_imcontext' message: ic_id = %d", id);
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_FREE_IMCONTEXT, 1);

    char *ic_id_str;
    scim_bridge_string_from_uint (&ic_id_str, id);
    scim_bridge_message_set_argument (message, 0, ic_id_str);
    free (ic_id_str);

    if (scim_bridge_messenger_send (messenger, message)) {
        scim_bridge_exception_push_stack ("scim_bridge_client_free_imcontext ()");
        connection_lost ();
        scim_bridge_free_message (message);
        return RETVAL_FAILED;
    } else {
        scim_bridge_free_message (message);
    }

    pending_imcontext_id = id;
    
    while (pending_imcontext_id == id) {
        if (scim_bridge_client_read_and_dispatch ()) {
            scim_bridge_exception_push_stack ("scim_bridge_client_free_imcontext ()");
            connection_lost ();

            return RETVAL_FAILED;
        }
    }

    if (pending_imcontext_id == -2) {
        scim_bridge_exception_occured (ILLEGAL_STATE_EXCEPTION, "Failed to free imcontext");
        scim_bridge_exception_push_stack ("scim_bridge_client_free_imcontext ()");
        return RETVAL_FAILED;
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "freed: id = %d", id);
        scim_bridge_client_imcontext_set_id (imcontext, -1);
        return RETVAL_SUCCEEDED;
    }
}


retval_t scim_bridge_client_reset_imcontext (const ScimBridgeClientIMContext *imcontext)
{
    scim_bridge_exception_clear ();

    const scim_bridge_imcontext_id_t id = scim_bridge_client_imcontext_get_id (imcontext);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_reset_imcontext: ic = %d", id);

    if (!active) {
        scim_bridge_exception_occured (ILLEGAL_STATE_EXCEPTION, "ScimBridge is not active");
        scim_bridge_exception_push_stack ("scim_bridge_client_reset_imcontext ()");
        return RETVAL_FAILED;
    }

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 5, "Sending 'reset_imcontext' message: ic_id = %d", id);
    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_RESET_IMCONTEXT, 1);

    char *ic_id_str;
    scim_bridge_string_from_uint (&ic_id_str, id);
    scim_bridge_message_set_argument (message, 0, ic_id_str);
    free (ic_id_str);

    if (scim_bridge_messenger_send (messenger, message)) {
        scim_bridge_exception_push_stack ("scim_bridge_client_reset_imcontext ()");
        connection_lost ();
        scim_bridge_free_message (message);
        return RETVAL_FAILED;
    } else {
        scim_bridge_free_message (message);
        return RETVAL_SUCCEEDED;
    }
}


retval_t scim_bridge_client_key_event_occured (const ScimBridgeClientIMContext *imcontext, const ScimBridgeKeyEvent *key_event, boolean *consumed)
{
    scim_bridge_exception_clear ();

    const scim_bridge_imcontext_id_t id = scim_bridge_client_imcontext_get_id (imcontext);

    const scim_bridge_key_code_t key_code = scim_bridge_key_event_get_code (key_event);

    char *modifiers_str = alloca (sizeof (char) * (strlen ("shift+control+alt+meta+super+hyper+caps_lock+num_lock+unknown") + 1));
    size_t modifier_str_index = 0;
    size_t modifier_count = 0;

    modifiers_str[0] = '\0';
    if (scim_bridge_key_event_is_shift_down (key_event)) {
        strcpy (modifiers_str + modifier_str_index, "shift");
        modifier_str_index += strlen ("shift");
        ++modifier_count;
    }
    if (scim_bridge_key_event_is_control_down (key_event)) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+control");
            modifier_str_index += strlen ("+control");
        } else {
            strcpy (modifiers_str + modifier_str_index, "control");
            modifier_str_index += strlen ("control");
        }
        ++modifier_count;
    }
    if (scim_bridge_key_event_is_alt_down (key_event)) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+alt");
            modifier_str_index += strlen ("+alt");
        } else {
            strcpy (modifiers_str + modifier_str_index, "alt");
            modifier_str_index += strlen ("alt");
        }
        ++modifier_count;
    }
    if (scim_bridge_key_event_is_meta_down (key_event)) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+meta");
            modifier_str_index += strlen ("+meta");
        } else {
            strcpy (modifiers_str + modifier_str_index, "meta");
            modifier_str_index += strlen ("meta");
        }
        ++modifier_count;
    }
    if (scim_bridge_key_event_is_super_down (key_event)) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+super");
            modifier_str_index += strlen ("+super");
        } else {
            strcpy (modifiers_str + modifier_str_index, "super");
            modifier_str_index += strlen ("super");
        }
        ++modifier_count;
    }
    if (scim_bridge_key_event_is_hyper_down (key_event)) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+hyper");
            modifier_str_index += strlen ("+hyper");
        } else {
            strcpy (modifiers_str + modifier_str_index, "hyper");
            modifier_str_index += strlen ("hyper");
        }
        ++modifier_count;
    }
    if (scim_bridge_key_event_is_caps_lock_down (key_event)) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+caps_lock");
            modifier_str_index += strlen ("+caps_lock");
        } else {
            strcpy (modifiers_str + modifier_str_index, "caps_lock");
            modifier_str_index += strlen ("caps_lock");
        }
        ++modifier_count;
    }
    if (scim_bridge_key_event_is_num_lock_down (key_event)) {
        if (modifier_str_index > 0) {
            strcpy (modifiers_str + modifier_str_index, "+num_lock");
            modifier_str_index += strlen ("+num_lock");
        } else {
            strcpy (modifiers_str + modifier_str_index, "num_lock");
            modifier_str_index += strlen ("num_lock");
        }
        ++modifier_count;
    }

    const boolean key_pressed = scim_bridge_key_event_is_pressed (key_event);
    if (key_pressed) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_key_event_occured: ic = %d, key_code = %u, pressed = true, modifiers = %s", id, key_code, modifiers_str);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_key_event_occured: ic = %d, key_code = %u, pressed = false, modifiers = %s", id, key_code, modifiers_str);
    }

    if (!active) {
        scim_bridge_exception_occured (ILLEGAL_STATE_EXCEPTION, "ScimBridge is not active");
        scim_bridge_exception_push_stack ("scim_bridge_client_key_event_occured ()");

        return RETVAL_FAILED;
    }

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 5, "Sending 'key_event_occured' message: ic_id = %d", id);

    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_KEY_EVENT_OCCURED, modifier_count + 3);

    char *imcontext_id_str;
    scim_bridge_string_from_int (&imcontext_id_str, id);
    scim_bridge_message_set_argument (message, 0,imcontext_id_str);

    char *key_code_str;
    scim_bridge_string_from_uint (&key_code_str, scim_bridge_key_event_get_code (key_event));
    scim_bridge_message_set_argument (message, 1, key_code_str);

    char *key_pressed_str;
    scim_bridge_string_from_boolean (&key_pressed_str, scim_bridge_key_event_is_pressed (key_event));
    scim_bridge_message_set_argument (message, 2, key_pressed_str);

    free (imcontext_id_str);
    free (key_code_str);
    free (key_pressed_str);

    size_t arg_index = 3;

    if (scim_bridge_key_event_is_shift_down (key_event)) {
        scim_bridge_message_set_argument (message, arg_index, SCIM_BRIDGE_MESSAGE_SHIFT);
        ++arg_index;
    }
    if (scim_bridge_key_event_is_control_down (key_event)) {
        scim_bridge_message_set_argument (message, arg_index, SCIM_BRIDGE_MESSAGE_CONTROL);
        ++arg_index;
    }
    if (scim_bridge_key_event_is_alt_down (key_event)) {
        scim_bridge_message_set_argument (message, arg_index, SCIM_BRIDGE_MESSAGE_ALT);
        ++arg_index;
    }
    if (scim_bridge_key_event_is_meta_down (key_event)) {
        scim_bridge_message_set_argument (message, arg_index, SCIM_BRIDGE_MESSAGE_META);
        ++arg_index;
    }
    if (scim_bridge_key_event_is_super_down (key_event)) {
        scim_bridge_message_set_argument (message, arg_index, SCIM_BRIDGE_MESSAGE_SUPER);
        ++arg_index;
    }
    if (scim_bridge_key_event_is_hyper_down (key_event)) {
        scim_bridge_message_set_argument (message, arg_index, SCIM_BRIDGE_MESSAGE_HYPER);
        ++arg_index;
    }
    if (scim_bridge_key_event_is_caps_lock_down (key_event)) {
        scim_bridge_message_set_argument (message, arg_index, SCIM_BRIDGE_MESSAGE_CAPS_LOCK);
        ++arg_index;
    }
    if (scim_bridge_key_event_is_num_lock_down (key_event)) {
        scim_bridge_message_set_argument (message, arg_index, SCIM_BRIDGE_MESSAGE_NUM_LOCK);
        ++arg_index;
    }

    key_event_status = KEY_EVENT_PENDING;
    if (scim_bridge_messenger_send (messenger, message)) {
        scim_bridge_exception_push_stack ("scim_bridge_client_key_event_occured ()");
        connection_lost ();
        scim_bridge_free_message (message);
        return RETVAL_FAILED;
    }

    while (key_event_status == KEY_EVENT_PENDING) {
        if (scim_bridge_client_read_and_dispatch ()) {
            scim_bridge_exception_push_stack ("scim_bridge_client_key_event_occured ()");
            connection_lost ();
            return RETVAL_FAILED;
        }
    }

    if (key_event_status == KEY_EVENT_IGNORED) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 3, "The key event was ignored");
        *consumed = FALSE;
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 2, "The key event was consumed");
        *consumed = TRUE;
    }
    return RETVAL_SUCCEEDED;
}


retval_t scim_bridge_client_focus_changed (const ScimBridgeClientIMContext *imcontext, boolean focus_in)
{
    scim_bridge_exception_clear ();

    const scim_bridge_imcontext_id_t id = scim_bridge_client_imcontext_get_id (imcontext);
    if (focus_in) {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_focus_changed: ic = %d, focus_in = TRUE", id);
    } else {
        scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_focus_changed: ic = %d, focus_in = FALSE", id);
    }

    if (!active) {
        scim_bridge_exception_occured (ILLEGAL_STATE_EXCEPTION, "ScimBridge is not active");
        scim_bridge_exception_push_stack ("scim_bridge_client_focus_changed ()");
        return RETVAL_FAILED;
    }

    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_MESSENGER, 5, "Sending 'focus_changed' message: ic_id = %d, focus_in = %s", id, focus_in ? "TRUE":"FALSE");

    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_FOCUS_CHANGED, 2);

    char *ic_id_str;
    scim_bridge_string_from_uint (&ic_id_str, id);
    scim_bridge_message_set_argument (message, 0, ic_id_str);

    char *focus_in_str;
    scim_bridge_string_from_boolean (&focus_in_str, focus_in);
    scim_bridge_message_set_argument (message, 1, focus_in_str);

    free (ic_id_str);
    free (focus_in_str);

    if (scim_bridge_messenger_send (messenger, message)) {
        scim_bridge_exception_push_stack ("scim_bridge_client_focus_changed ()");
        connection_lost ();
        scim_bridge_free_message (message);
        return RETVAL_FAILED;
    } else {
        scim_bridge_free_message (message);
        return RETVAL_SUCCEEDED;
    }
}


retval_t scim_bridge_client_cursor_location_changed (const ScimBridgeClientIMContext *imcontext, int x, int y)
{
    scim_bridge_exception_clear ();

    const scim_bridge_imcontext_id_t id = scim_bridge_client_imcontext_get_id (imcontext);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_cursor_location_changed: ic = %d, x = %d, y = %d", id, x, y);

    if (!active) {
        scim_bridge_exception_occured (ILLEGAL_STATE_EXCEPTION, "ScimBridge is not active");
        scim_bridge_exception_push_stack ("scim_bridge_client_cursor_location_changed ()");
        return RETVAL_FAILED;
    }

    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_CURSOR_LOCATION_CHANGED, 3);

    char *ic_id_str;
    scim_bridge_string_from_uint (&ic_id_str, id);
    scim_bridge_message_set_argument (message, 0, ic_id_str);

    char *x_str;
    scim_bridge_string_from_int (&x_str, x);
    scim_bridge_message_set_argument (message, 1, x_str);

    char *y_str;
    scim_bridge_string_from_uint (&y_str, y);
    scim_bridge_message_set_argument (message, 2, y_str);

    free (ic_id_str);
    free (x_str);
    free (y_str);

    if (scim_bridge_messenger_send (messenger, message)) {
        scim_bridge_exception_push_stack ("scim_bridge_client_cursor_position_changed ()");
        connection_lost ();
        scim_bridge_free_message (message);
        return RETVAL_FAILED;
    } else {
        scim_bridge_free_message (message);
        return RETVAL_SUCCEEDED;
    }
}


retval_t scim_bridge_client_set_preedit_enabled (const ScimBridgeClientIMContext *imcontext, boolean enabled)
{
    scim_bridge_exception_clear ();

    const scim_bridge_imcontext_id_t id = scim_bridge_client_imcontext_get_id (imcontext);
    scim_bridge_pdebugln (SCIM_BRIDGE_DEBUG_CLIENT, 5, "scim_bridge_client_set_preedit_enabled: ic = %d, enabled = %s", id, enabled ? "true":"false");

    if (!active) {
        scim_bridge_exception_occured (ILLEGAL_STATE_EXCEPTION, "ScimBridge is not active");
        scim_bridge_exception_push_stack ("scim_bridge_client_set_preedit_enabled ()");
        return RETVAL_FAILED;
    }

    ScimBridgeMessage *message = scim_bridge_alloc_message (SCIM_BRIDGE_MESSAGE_SET_PREEDIT_ENABLED, 2);

    char *ic_id_str;
    scim_bridge_string_from_uint (&ic_id_str, id);
    scim_bridge_message_set_argument (message, 0, ic_id_str);

    char *enabled_str;
    scim_bridge_string_from_boolean (&enabled_str, enabled);
    scim_bridge_message_set_argument (message, 1, enabled_str);

    free (ic_id_str);
    free (enabled_str);

    if (scim_bridge_messenger_send (messenger, message)) {
        scim_bridge_exception_push_stack ("scim_bridge_client_set_preedit_enabled ()");
        connection_lost ();
        scim_bridge_free_message (message);
        return RETVAL_FAILED;
    } else {
        scim_bridge_free_message (message);
        return RETVAL_SUCCEEDED;
    }
}
