/*
 * SCIM Bridge
 *
 * Copyright (c) 2006 Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 *
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.*
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.See the
 * GNU Lesser General Public License for more details.*
 * You should have received a copy of the GNU Lesser General Public
 * License along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA  02111-1307  USA
 */

/**
 * @file
 * @author Ryo Dairiki <ryo-dairiki@users.sourceforge.net>
 * @brief This header is used for print out error messages.
 */
#ifndef SCIMBRIDGEOUTPUT_H_
#define SCIMBRIDGEOUTPUT_H_

#include "scim-bridge.h"
#include "scim-bridge-debug.h"

#ifdef __cplusplus
extern "C"
{
#endif
    
    /**
     * Print out a message and give a line feed ('\n').
     * The messages will be shown in the stdout.
     *
     * @param format The message in printf style.
     */
    void scim_bridge_println (const char *format,...);

    /**
     * Print out an error message and give a line feed ('\n').
     * The error messages will be shown in the stderr.
     *
     * @param format The message in printf style.
     */
    void scim_bridge_perrorln (const char *format,...);

    /**
     * Print out an debug message and give a line feed ('\n').
     * The error messages will be shown in the stdout.
     * Note, the message will be ignored when the given debug flag has not been stored, 
     * or the given debug level is lower than that of the preference.
     * @see scim_bridge_debug_get_flag ()
     * @see scim_bridge_debug_get_level ()
     *
     * @param flag The debug flag.
     * @param level The debug level.
     * @param format The message in printf style.
     */
    void scim_bridge_pdebugln (scim_bridge_debug_flag_t flag, scim_bridge_debug_level_t level, const char *format,...);

    /**
     * Print out a message without giving a line feed ('\n').
     * The messages will be shown in the stdout.
     *
     * @param format The message in printf style.
     */
    void scim_bridge_print (const char *format,...);

    /**
     * Print out an error message without giving a line feed ('\n').
     * The error messages will be shown in the stderr.
     *
     * @param format The message in printf style.
     */
    void scim_bridge_perror (const char *format,...);

    /**
     * Print out an debug message without giving a line feed ('\n').
     * The error messages will be shown in the stdout.
     * Note, the message will be ignored when the given debug flag has not been stored, 
     * or the given debug level is lower than that of the preference.
     * @see scim_bridge_debug_get_flag ()
     * @see scim_bridge_debug_get_level ()
     *
     * @param flag The debug flag.
     * @param level The debug level.
     * @param format The message in printf style.
     */
    void scim_bridge_pdebug (scim_bridge_debug_flag_t flag, scim_bridge_debug_level_t level, const char *format,...);

#ifdef __cplusplus
}
#endif
#endif                                            /*SCIMBRIDGEOUTPUT_H_*/
