static const char* op_c_source =
"/* This file is an image processing operation for GEGL                        \n"
" * GEGL is free software; you can redistribute it and/or                      \n"
" * modify it under the terms of the GNU Lesser General Public                 \n"
" * License as published by the Free Software Foundation; either               \n"
" * version 3 of the License, or (at your option) any later version.           \n"
" *                                                                            \n"
" * GEGL is distributed in the hope that it will be useful,                    \n"
" * but WITHOUT ANY WARRANTY; without even the implied warranty of             \n"
" * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          \n"
" * Lesser General Public License for more details.                            \n"
" *                                                                            \n"
" * You should have received a copy of the GNU Lesser General Public           \n"
" * License along with GEGL; if not, see <http://www.gnu.org/licenses/>.       \n"
" *                                                                            \n"
" * Copyright 2006 Dominik Ernst <dernst@gmx.de>                               \n"
" * Copyright 2013 Massimo Valentini <mvalentini@src.gnome.org>                \n"
" *                                                                            \n"
" * Recursive Gauss IIR Filter as described by Young / van Vliet               \n"
" * in \"Signal Processing 44 (1995) 139 - 151\"                               \n"
" *                                                                            \n"
" * NOTE: The IIR filter should not be used for radius < 0.5, since it         \n"
" *       becomes very inaccurate.                                             \n"
" */                                                                           \n"
"                                                                              \n"
"#include \"config.h\"                                                         \n"
"#include <glib/gi18n-lib.h>                                                   \n"
"                                                                              \n"
"#ifdef GEGL_PROPERTIES                                                        \n"
"                                                                              \n"
"enum_start (gegl_gblur_1d_policy)                                             \n"
"   enum_value (GEGL_GBLUR_1D_ABYSS_NONE,  \"none\",  N_(\"None\"))            \n"
"   enum_value (GEGL_GBLUR_1D_ABYSS_CLAMP, \"clamp\", N_(\"Clamp\"))           \n"
"   enum_value (GEGL_GBLUR_1D_ABYSS_BLACK, \"black\", N_(\"Black\"))           \n"
"   enum_value (GEGL_GBLUR_1D_ABYSS_WHITE, \"white\", N_(\"White\"))           \n"
"enum_end (GeglGblur1dPolicy)                                                  \n"
"                                                                              \n"
"enum_start (gegl_gblur_1d_filter)                                             \n"
"  enum_value (GEGL_GBLUR_1D_AUTO, \"auto\", N_(\"Auto\"))                     \n"
"  enum_value (GEGL_GBLUR_1D_FIR,  \"fir\",  N_(\"FIR\"))                      \n"
"  enum_value (GEGL_GBLUR_1D_IIR,  \"iir\",  N_(\"IIR\"))                      \n"
"enum_end (GeglGblur1dFilter)                                                  \n"
"                                                                              \n"
"property_double (std_dev, _(\"Size\"), 1.5)                                   \n"
"  description (_(\"Standard deviation (spatial scale factor)\"))              \n"
"  value_range   (0.0, 1500.0)                                                 \n"
"  ui_range      (0.0, 100.0)                                                  \n"
"  ui_gamma      (3.0)                                                         \n"
"                                                                              \n"
"property_enum (orientation, _(\"Orientation\"),                               \n"
"               GeglOrientation, gegl_orientation,                             \n"
"               GEGL_ORIENTATION_HORIZONTAL)                                   \n"
"  description (_(\"The orientation of the blur - hor/ver\"))                  \n"
"                                                                              \n"
"property_enum (filter, _(\"Filter\"),                                         \n"
"               GeglGblur1dFilter, gegl_gblur_1d_filter,                       \n"
"               GEGL_GBLUR_1D_AUTO)                                            \n"
"  description (_(\"How the gaussian kernel is discretized\"))                 \n"
"                                                                              \n"
"property_enum (abyss_policy, _(\"Abyss policy\"), GeglGblur1dPolicy,          \n"
"               gegl_gblur_1d_policy, GEGL_GBLUR_1D_ABYSS_NONE)                \n"
"  description (_(\"How image edges are handled\"))                            \n"
"                                                                              \n"
"property_boolean (clip_extent, _(\"Clip to the input extent\"), TRUE)         \n"
"  description (_(\"Should the output extent be clipped to the input extent\"))\n"
"                                                                              \n"
"#else                                                                         \n"
"                                                                              \n"
"#define GEGL_OP_FILTER                                                        \n"
"#define GEGL_OP_C_SOURCE gblur-1d.c                                           \n"
"                                                                              \n"
"#include \"gegl-op.h\"                                                        \n"
"#include <math.h>                                                             \n"
"#include <stdio.h>                                                            \n"
"                                                                              \n"
"                                                                              \n"
"/**********************************************                               \n"
" *                                                                            \n"
" * Infinite Impulse Response (IIR)                                            \n"
" *                                                                            \n"
" **********************************************/                              \n"
"                                                                              \n"
"static void                                                                   \n"
"iir_young_find_constants (gfloat   sigma,                                     \n"
"                          gdouble *b,                                         \n"
"                          gdouble (*m)[3])                                    \n"
"{                                                                             \n"
"  const gdouble K1 = 2.44413;                                                 \n"
"  const gdouble K2 = 1.4281;                                                  \n"
"  const gdouble K3 = 0.422205;                                                \n"
"  const gdouble q = sigma >= 2.5 ?                                            \n"
"                    0.98711 * sigma - 0.96330 :                               \n"
"                    3.97156 - 4.14554 * sqrt (1 - 0.26891 * sigma);           \n"
"                                                                              \n"
"  const gdouble b0 = 1.57825 + q*(K1 + q*(    K2 + q *     K3));              \n"
"  const gdouble b1 =           q*(K1 + q*(2 * K2 + q * 3 * K3));              \n"
"  const gdouble b2 =         - q*      q*(    K2 + q * 3 * K3);               \n"
"  const gdouble b3 =           q*      q*          q *     K3;                \n"
"                                                                              \n"
"  const gdouble a1 = b1 / b0;                                                 \n"
"  const gdouble a2 = b2 / b0;                                                 \n"
"  const gdouble a3 = b3 / b0;                                                 \n"
"  const gdouble c  = 1. / ((1+a1-a2+a3) * (1+a2+(a1-a3)*a3));                 \n"
"                                                                              \n"
"  m[0][0] = c * (-a3*(a1+a3)-a2 + 1);                                         \n"
"  m[0][1] = c * (a3+a1)*(a2+a3*a1);                                           \n"
"  m[0][2] = c * a3*(a1+a3*a2);                                                \n"
"                                                                              \n"
"  m[1][0] = c * (a1+a3*a2);                                                   \n"
"  m[1][1] = c * (1-a2)*(a2+a3*a1);                                            \n"
"  m[1][2] = c * a3*(1-a3*a1-a3*a3-a2);                                        \n"
"                                                                              \n"
"  m[2][0] = c * (a3*a1+a2+a1*a1-a2*a2);                                       \n"
"  m[2][1] = c * (a1*a2+a3*a2*a2-a1*a3*a3-a3*a3*a3-a3*a2+a3);                  \n"
"  m[2][2] = c * a3*(a1+a3*a2);                                                \n"
"                                                                              \n"
"  b[0] = 1. - (b1 + b2 + b3) / b0;                                            \n"
"  b[1] = a1;                                                                  \n"
"  b[2] = a2;                                                                  \n"
"  b[3] = a3;                                                                  \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"fix_right_boundary (gdouble        *buf,                                      \n"
"                    gdouble       (*m)[3],                                    \n"
"                    const gdouble   uplus)                                    \n"
"{                                                                             \n"
"  gdouble u[3] = { buf[-1] - uplus, buf[-2] - uplus, buf[-3] - uplus };       \n"
"  gint    i, k;                                                               \n"
"                                                                              \n"
"  for (i = 0; i < 3; i++)                                                     \n"
"    {                                                                         \n"
"      gdouble tmp = 0.;                                                       \n"
"                                                                              \n"
"      for (k = 0; k < 3; k++)                                                 \n"
"        tmp += m[i][k] * u[k];                                                \n"
"                                                                              \n"
"      buf[i] = tmp + uplus;                                                   \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static inline void                                                            \n"
"iir_young_blur_1D (gfloat        *buf,                                        \n"
"                   gdouble       *tmp,                                        \n"
"                   const gdouble *b,                                          \n"
"                   gdouble      (*m)[3],                                      \n"
"                   const gint     len,                                        \n"
"                   const gint     nc,                                         \n"
"                   GeglAbyssPolicy policy)                                    \n"
"{                                                                             \n"
"  gfloat  white[4] = { 1, 1, 1, 1 };                                          \n"
"  gfloat  black[4] = { 0, 0, 0, 1 };                                          \n"
"  gfloat  none[4]  = { 0, 0, 0, 0 };                                          \n"
"  gfloat *uplus, *iminus;                                                     \n"
"  gint    i, j, c;                                                            \n"
"                                                                              \n"
"  switch (policy)                                                             \n"
"    {                                                                         \n"
"    case GEGL_ABYSS_CLAMP: default:                                           \n"
"      iminus = &buf[nc * 3]; uplus = &buf[nc * (len + 2)]; break;             \n"
"                                                                              \n"
"    case GEGL_ABYSS_NONE:                                                     \n"
"      iminus = uplus = &none[0]; break;                                       \n"
"                                                                              \n"
"    case GEGL_ABYSS_WHITE:                                                    \n"
"      iminus = uplus = &white[0]; break;                                      \n"
"                                                                              \n"
"    case GEGL_ABYSS_BLACK:                                                    \n"
"      iminus = uplus = &black[nc == 2 ? 2 : 0]; break;                        \n"
"    }                                                                         \n"
"                                                                              \n"
"  for (c = 0; c < nc; ++c)                                                    \n"
"    {                                                                         \n"
"      for (i = 0; i < 3; ++i)                                                 \n"
"        tmp[i] = iminus[c];                                                   \n"
"                                                                              \n"
"      for (i = 3; i < 3 + len; ++i)                                           \n"
"        {                                                                     \n"
"          tmp[i] = buf[nc * i + c] * b[0];                                    \n"
"                                                                              \n"
"          for (j = 1; j < 4; ++j)                                             \n"
"            tmp[i] += b[j] * tmp[i - j];                                      \n"
"        }                                                                     \n"
"                                                                              \n"
"      fix_right_boundary (&tmp[3 + len], m, uplus[c]);                        \n"
"                                                                              \n"
"      for (i = 3 + len - 1; 3 <= i; --i)                                      \n"
"        {                                                                     \n"
"          gdouble tt = 0.;                                                    \n"
"                                                                              \n"
"          for (j = 0; j < 4; ++j)                                             \n"
"            tt += b[j] * tmp[i + j];                                          \n"
"                                                                              \n"
"          buf[nc * i + c] = tmp[i] = tt;                                      \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"iir_young_hor_blur (GeglBuffer          *src,                                 \n"
"                    const GeglRectangle *rect,                                \n"
"                    GeglBuffer          *dst,                                 \n"
"                    const gdouble       *b,                                   \n"
"                    gdouble            (*m)[3],                               \n"
"                    GeglAbyssPolicy      policy,                              \n"
"                    const Babl          *format)                              \n"
"{                                                                             \n"
"  GeglRectangle  cur_row = *rect;                                             \n"
"  const gint     nc = babl_format_get_n_components (format);                  \n"
"  gfloat        *row = g_new (gfloat, (3 + rect->width + 3) * nc);            \n"
"  gdouble       *tmp = g_new (gdouble, (3 + rect->width + 3));                \n"
"  gint           v;                                                           \n"
"                                                                              \n"
"  cur_row.height = 1;                                                         \n"
"                                                                              \n"
"  for (v = 0; v < rect->height; v++)                                          \n"
"    {                                                                         \n"
"      cur_row.y = rect->y + v;                                                \n"
"                                                                              \n"
"      gegl_buffer_get (src, &cur_row, 1.0, format, &row[3 * nc],              \n"
"                       GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);                 \n"
"                                                                              \n"
"      iir_young_blur_1D (row, tmp, b, m, rect->width, nc, policy);            \n"
"                                                                              \n"
"      gegl_buffer_set (dst, &cur_row, 0, format, &row[3 * nc],                \n"
"                       GEGL_AUTO_ROWSTRIDE);                                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (tmp);                                                               \n"
"  g_free (row);                                                               \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"iir_young_ver_blur (GeglBuffer          *src,                                 \n"
"                    const GeglRectangle *rect,                                \n"
"                    GeglBuffer          *dst,                                 \n"
"                    const gdouble       *b,                                   \n"
"                    gdouble            (*m)[3],                               \n"
"                    GeglAbyssPolicy      policy,                              \n"
"                    const Babl          *format)                              \n"
"{                                                                             \n"
"  GeglRectangle  cur_col = *rect;                                             \n"
"  const gint     nc = babl_format_get_n_components (format);                  \n"
"  gfloat        *col = g_new (gfloat, (3 + rect->height + 3) * nc);           \n"
"  gdouble       *tmp = g_new (gdouble, (3 + rect->height + 3));               \n"
"  gint           i;                                                           \n"
"                                                                              \n"
"  cur_col.width = 1;                                                          \n"
"                                                                              \n"
"  for (i = 0; i < rect->width; i++)                                           \n"
"    {                                                                         \n"
"      cur_col.x = rect->x + i;                                                \n"
"                                                                              \n"
"      gegl_buffer_get (src, &cur_col, 1.0, format, &col[3 * nc],              \n"
"                       GEGL_AUTO_ROWSTRIDE, GEGL_ABYSS_NONE);                 \n"
"                                                                              \n"
"      iir_young_blur_1D (col, tmp, b, m, rect->height, nc, policy);           \n"
"                                                                              \n"
"      gegl_buffer_set (dst, &cur_col, 0, format, &col[3 * nc],                \n"
"                       GEGL_AUTO_ROWSTRIDE);                                  \n"
"    }                                                                         \n"
"                                                                              \n"
"  g_free (tmp);                                                               \n"
"  g_free (col);                                                               \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/**********************************************                               \n"
" *                                                                            \n"
" * Finite Impulse Response (FIR)                                              \n"
" *                                                                            \n"
" **********************************************/                              \n"
"                                                                              \n"
"static inline void                                                            \n"
"fir_blur_1D (const gfloat *input,                                             \n"
"                   gfloat *output,                                            \n"
"             const gfloat *cmatrix,                                           \n"
"             const gint    clen,                                              \n"
"             const gint    len,                                               \n"
"             const gint    nc)                                                \n"
"{                                                                             \n"
"  gint i;                                                                     \n"
"  for (i = 0; i < len; i++)                                                   \n"
"    {                                                                         \n"
"      gint c;                                                                 \n"
"      for (c = 0; c < nc; c++)                                                \n"
"        {                                                                     \n"
"          gint   index = i * nc + c;                                          \n"
"          gfloat acc   = 0.0f;                                                \n"
"          gint   m;                                                           \n"
"                                                                              \n"
"          for (m = 0; m < clen; m++)                                          \n"
"            acc += input [index + m * nc] * cmatrix [m];                      \n"
"                                                                              \n"
"          output [index] = acc;                                               \n"
"        }                                                                     \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"fir_hor_blur (GeglBuffer          *src,                                       \n"
"              const GeglRectangle *rect,                                      \n"
"              GeglBuffer          *dst,                                       \n"
"              gfloat              *cmatrix,                                   \n"
"              gint                 clen,                                      \n"
"              GeglAbyssPolicy      policy,                                    \n"
"              const Babl          *format)                                    \n"
"{                                                                             \n"
"  GeglRectangle  cur_row = *rect;                                             \n"
"  GeglRectangle  in_row;                                                      \n"
"  const gint     nc = babl_format_get_n_components (format);                  \n"
"  gfloat        *row;                                                         \n"
"  gfloat        *out;                                                         \n"
"  gint           v;                                                           \n"
"                                                                              \n"
"  cur_row.height = 1;                                                         \n"
"                                                                              \n"
"  in_row         = cur_row;                                                   \n"
"  in_row.width  += clen - 1;                                                  \n"
"  in_row.x      -= clen / 2;                                                  \n"
"                                                                              \n"
"  row = gegl_malloc (sizeof (gfloat) * in_row.width  * nc);                   \n"
"  out = gegl_malloc (sizeof (gfloat) * cur_row.width * nc);                   \n"
"                                                                              \n"
"  for (v = 0; v < rect->height; v++)                                          \n"
"    {                                                                         \n"
"      cur_row.y = in_row.y = rect->y + v;                                     \n"
"                                                                              \n"
"      gegl_buffer_get (src, &in_row, 1.0, format, row, GEGL_AUTO_ROWSTRIDE, policy);\n"
"                                                                              \n"
"      fir_blur_1D (row, out, cmatrix, clen, rect->width, nc);                 \n"
"                                                                              \n"
"      gegl_buffer_set (dst, &cur_row, 0, format, out, GEGL_AUTO_ROWSTRIDE);   \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_free (out);                                                            \n"
"  gegl_free (row);                                                            \n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"fir_ver_blur (GeglBuffer          *src,                                       \n"
"              const GeglRectangle *rect,                                      \n"
"              GeglBuffer          *dst,                                       \n"
"              gfloat              *cmatrix,                                   \n"
"              gint                 clen,                                      \n"
"              GeglAbyssPolicy      policy,                                    \n"
"              const Babl          *format)                                    \n"
"{                                                                             \n"
"  GeglRectangle  cur_col = *rect;                                             \n"
"  GeglRectangle  in_col;                                                      \n"
"  const gint     nc = babl_format_get_n_components (format);                  \n"
"  gfloat        *col;                                                         \n"
"  gfloat        *out;                                                         \n"
"  gint           v;                                                           \n"
"                                                                              \n"
"  cur_col.width = 1;                                                          \n"
"                                                                              \n"
"  in_col         = cur_col;                                                   \n"
"  in_col.height += clen - 1;                                                  \n"
"  in_col.y      -= clen / 2;                                                  \n"
"                                                                              \n"
"  col = gegl_malloc (sizeof (gfloat) * in_col.height  * nc);                  \n"
"  out = gegl_malloc (sizeof (gfloat) * cur_col.height * nc);                  \n"
"                                                                              \n"
"  for (v = 0; v < rect->width; v++)                                           \n"
"    {                                                                         \n"
"      cur_col.x = in_col.x = rect->x + v;                                     \n"
"                                                                              \n"
"      gegl_buffer_get (src, &in_col, 1.0, format, col, GEGL_AUTO_ROWSTRIDE, policy);\n"
"                                                                              \n"
"      fir_blur_1D (col, out, cmatrix, clen, rect->height, nc);                \n"
"                                                                              \n"
"      gegl_buffer_set (dst, &cur_col, 0, format, out, GEGL_AUTO_ROWSTRIDE);   \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_free (out);                                                            \n"
"  gegl_free (col);                                                            \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"#include \"opencl/gegl-cl.h\"                                                 \n"
"#include \"buffer/gegl-buffer-cl-iterator.h\"                                 \n"
"                                                                              \n"
"#include \"opencl/gblur-1d.cl.h\"                                             \n"
"                                                                              \n"
"static GeglClRunData *cl_data = NULL;                                         \n"
"                                                                              \n"
"                                                                              \n"
"static gboolean                                                               \n"
"cl_gaussian_blur (cl_mem                 in_tex,                              \n"
"                  cl_mem                 out_tex,                             \n"
"                  const GeglRectangle   *roi,                                 \n"
"                  cl_mem                 cl_cmatrix,                          \n"
"                  gint                   clen,                                \n"
"                  GeglOrientation        orientation)                         \n"
"{                                                                             \n"
"  cl_int cl_err = 0;                                                          \n"
"  size_t global_ws[2];                                                        \n"
"  gint   kernel_num;                                                          \n"
"                                                                              \n"
"  if (!cl_data)                                                               \n"
"    {                                                                         \n"
"      const char *kernel_name[] = {\"fir_ver_blur\", \"fir_hor_blur\", NULL}; \n"
"      cl_data = gegl_cl_compile_and_build (gblur_1d_cl_source, kernel_name);  \n"
"    }                                                                         \n"
"                                                                              \n"
"  if (!cl_data)                                                               \n"
"    return TRUE;                                                              \n"
"                                                                              \n"
"  if (orientation == GEGL_ORIENTATION_VERTICAL)                               \n"
"    kernel_num = 0;                                                           \n"
"  else                                                                        \n"
"    kernel_num = 1;                                                           \n"
"                                                                              \n"
"  global_ws[0] = roi->width;                                                  \n"
"  global_ws[1] = roi->height;                                                 \n"
"                                                                              \n"
"  cl_err = gegl_cl_set_kernel_args (cl_data->kernel[kernel_num],              \n"
"                                    sizeof(cl_mem), (void*)&in_tex,           \n"
"                                    sizeof(cl_mem), (void*)&out_tex,          \n"
"                                    sizeof(cl_mem), (void*)&cl_cmatrix,       \n"
"                                    sizeof(cl_int), (void*)&clen,             \n"
"                                    NULL);                                    \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clEnqueueNDRangeKernel (gegl_cl_get_command_queue (),         \n"
"                                        cl_data->kernel[kernel_num], 2,       \n"
"                                        NULL, global_ws, NULL,                \n"
"                                        0, NULL, NULL);                       \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_err = gegl_clFinish (gegl_cl_get_command_queue ());                      \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  return FALSE;                                                               \n"
"                                                                              \n"
"error:                                                                        \n"
"  return TRUE;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"fir_cl_process (GeglBuffer            *input,                                 \n"
"                GeglBuffer            *output,                                \n"
"                const GeglRectangle   *result,                                \n"
"                const Babl            *format,                                \n"
"                gfloat                *cmatrix,                               \n"
"                gint                   clen,                                  \n"
"                GeglOrientation        orientation,                           \n"
"                GeglAbyssPolicy        abyss)                                 \n"
"{                                                                             \n"
"  gboolean              err = FALSE;                                          \n"
"  cl_int                cl_err;                                               \n"
"  cl_mem                cl_cmatrix = NULL;                                    \n"
"  GeglBufferClIterator *i;                                                    \n"
"  gint                  read;                                                 \n"
"  gint                  left, right, top, bottom;                             \n"
"                                                                              \n"
"  if (orientation == GEGL_ORIENTATION_HORIZONTAL)                             \n"
"    {                                                                         \n"
"      right = left = clen / 2;                                                \n"
"      top = bottom = 0;                                                       \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      right = left = 0;                                                       \n"
"      top = bottom = clen / 2;                                                \n"
"    }                                                                         \n"
"                                                                              \n"
"  i = gegl_buffer_cl_iterator_new (output,                                    \n"
"                                   result,                                    \n"
"                                   format,                                    \n"
"                                   GEGL_CL_BUFFER_WRITE);                     \n"
"                                                                              \n"
"  read = gegl_buffer_cl_iterator_add_2 (i,                                    \n"
"                                        input,                                \n"
"                                        result,                               \n"
"                                        format,                               \n"
"                                        GEGL_CL_BUFFER_READ,                  \n"
"                                        left, right,                          \n"
"                                        top, bottom,                          \n"
"                                        abyss);                               \n"
"                                                                              \n"
"  cl_cmatrix = gegl_clCreateBuffer (gegl_cl_get_context(),                    \n"
"                                    CL_MEM_COPY_HOST_PTR | CL_MEM_READ_ONLY,  \n"
"                                    clen * sizeof(cl_float), cmatrix, &cl_err);\n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  while (gegl_buffer_cl_iterator_next (i, &err) && !err)                      \n"
"    {                                                                         \n"
"      err = cl_gaussian_blur(i->tex[read],                                    \n"
"                             i->tex[0],                                       \n"
"                             &i->roi[0],                                      \n"
"                             cl_cmatrix,                                      \n"
"                             clen,                                            \n"
"                             orientation);                                    \n"
"                                                                              \n"
"      if (err)                                                                \n"
"        {                                                                     \n"
"          gegl_buffer_cl_iterator_stop (i);                                   \n"
"          break;                                                              \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  cl_err = gegl_clReleaseMemObject (cl_cmatrix);                              \n"
"  CL_CHECK;                                                                   \n"
"                                                                              \n"
"  cl_cmatrix = NULL;                                                          \n"
"                                                                              \n"
"  return !err;                                                                \n"
"                                                                              \n"
"error:                                                                        \n"
"  if (cl_cmatrix)                                                             \n"
"    gegl_clReleaseMemObject (cl_cmatrix);                                     \n"
"                                                                              \n"
"  return FALSE;                                                               \n"
"}                                                                             \n"
"                                                                              \n"
"static gfloat                                                                 \n"
"gaussian_func_1d (gfloat x,                                                   \n"
"                  gfloat sigma)                                               \n"
"{                                                                             \n"
"  return (1.0 / (sigma * sqrt (2.0 * G_PI))) *                                \n"
"          exp (-(x * x) / (2.0 * sigma * sigma));                             \n"
"}                                                                             \n"
"                                                                              \n"
"static gint                                                                   \n"
"fir_calc_convolve_matrix_length (gfloat sigma)                                \n"
"{                                                                             \n"
"#if 1                                                                         \n"
"  /* an arbitrary precision */                                                \n"
"  gint clen = sigma > GEGL_FLOAT_EPSILON ? ceil (sigma * 6.5) : 1;            \n"
"  clen = clen + ((clen + 1) % 2);                                             \n"
"  return clen;                                                                \n"
"#else                                                                         \n"
"  if (sigma > GEGL_FLOAT_EPSILON)                                             \n"
"    {                                                                         \n"
"      gint x = 0;                                                             \n"
"      while (gaussian_func_1d (x, sigma) > 1e-3)                              \n"
"        x++;                                                                  \n"
"      return 2 * x + 1;                                                       \n"
"    }                                                                         \n"
"  else return 1;                                                              \n"
"#endif                                                                        \n"
"}                                                                             \n"
"                                                                              \n"
"static gint                                                                   \n"
"fir_gen_convolve_matrix (gfloat   sigma,                                      \n"
"                         gfloat **cmatrix)                                    \n"
"{                                                                             \n"
"  gint    clen;                                                               \n"
"  gfloat *cmatrix_p;                                                          \n"
"                                                                              \n"
"  clen = fir_calc_convolve_matrix_length (sigma);                             \n"
"                                                                              \n"
"  *cmatrix  = gegl_malloc (sizeof (gfloat) * clen);                           \n"
"  cmatrix_p = *cmatrix;                                                       \n"
"                                                                              \n"
"  if (clen == 1)                                                              \n"
"    {                                                                         \n"
"      cmatrix_p [0] = 1;                                                      \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      gint    i;                                                              \n"
"      gdouble sum = 0;                                                        \n"
"      gint    half_clen = clen / 2;                                           \n"
"                                                                              \n"
"      for (i = 0; i < clen; i++)                                              \n"
"        {                                                                     \n"
"          cmatrix_p [i] = gaussian_func_1d (i - half_clen, sigma);            \n"
"          sum += cmatrix_p [i];                                               \n"
"        }                                                                     \n"
"                                                                              \n"
"      for (i = 0; i < clen; i++)                                              \n"
"        {                                                                     \n"
"          cmatrix_p [i] /= sum;                                               \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return clen;                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglGblur1dFilter                                                      \n"
"filter_disambiguation (GeglGblur1dFilter filter,                              \n"
"                       gfloat            std_dev)                             \n"
"{                                                                             \n"
"  if (filter == GEGL_GBLUR_1D_AUTO)                                           \n"
"    {                                                                         \n"
"      /* Threshold 1.0 is arbitrary */                                        \n"
"      if (std_dev < 1.0)                                                      \n"
"        filter = GEGL_GBLUR_1D_FIR;                                           \n"
"      else                                                                    \n"
"        filter = GEGL_GBLUR_1D_IIR;                                           \n"
"    }                                                                         \n"
"  return filter;                                                              \n"
"}                                                                             \n"
"                                                                              \n"
"                                                                              \n"
"/**********************************************                               \n"
" *                                                                            \n"
" * GEGL Operation API                                                         \n"
" *                                                                            \n"
" **********************************************/                              \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_gblur_1d_prepare (GeglOperation *operation)                              \n"
"{                                                                             \n"
"  const Babl *src_format = gegl_operation_get_source_format (operation, \"input\");\n"
"  const char *format     = \"RaGaBaA float\";                                 \n"
"                                                                              \n"
"  /*                                                                          \n"
"   * FIXME: when the abyss policy is _NONE, the behavior at the edge          \n"
"   *        depends on input format (with or without an alpha component)      \n"
"   */                                                                         \n"
"  if (src_format)                                                             \n"
"    {                                                                         \n"
"      const Babl *model = babl_format_get_model (src_format);                 \n"
"                                                                              \n"
"      if (model == babl_model (\"RGB\") || model == babl_model (\"R'G'B'\"))  \n"
"        format = \"RGB float\";                                               \n"
"      else if (model == babl_model (\"Y\") || model == babl_model (\"Y'\"))   \n"
"        format = \"Y float\";                                                 \n"
"      else if (model == babl_model (\"YA\") || model == babl_model (\"Y'A\") ||\n"
"               model == babl_model (\"YaA\") || model == babl_model (\"Y'aA\"))\n"
"        format = \"YaA float\";                                               \n"
"    }                                                                         \n"
"                                                                              \n"
"  gegl_operation_set_format (operation, \"output\", babl_format (format));    \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"gegl_gblur_1d_enlarge_extent (GeglProperties          *o,                     \n"
"                              const GeglRectangle *input_extent)              \n"
"{                                                                             \n"
"  gint clen = fir_calc_convolve_matrix_length (o->std_dev);                   \n"
"                                                                              \n"
"  GeglRectangle bounding_box = *input_extent;                                 \n"
"                                                                              \n"
"  if (o->orientation == GEGL_ORIENTATION_HORIZONTAL)                          \n"
"    {                                                                         \n"
"      bounding_box.x     -= clen / 2;                                         \n"
"      bounding_box.width += clen - 1;                                         \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      bounding_box.y      -= clen / 2;                                        \n"
"      bounding_box.height += clen - 1;                                        \n"
"    }                                                                         \n"
"                                                                              \n"
"  return bounding_box;                                                        \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"gegl_gblur_1d_get_required_for_output (GeglOperation       *operation,        \n"
"                                       const gchar         *input_pad,        \n"
"                                       const GeglRectangle *output_roi)       \n"
"{                                                                             \n"
"  GeglRectangle        required_for_output = { 0, };                          \n"
"  GeglProperties          *o       = GEGL_PROPERTIES (operation);             \n"
"  GeglGblur1dFilter    filter  = filter_disambiguation (o->filter, o->std_dev);\n"
"                                                                              \n"
"  if (filter == GEGL_GBLUR_1D_IIR)                                            \n"
"    {                                                                         \n"
"      const GeglRectangle *in_rect =                                          \n"
"        gegl_operation_source_get_bounding_box (operation, input_pad);        \n"
"                                                                              \n"
"      if (in_rect)                                                            \n"
"        {                                                                     \n"
"          if (!gegl_rectangle_is_infinite_plane (in_rect))                    \n"
"            {                                                                 \n"
"              required_for_output = *output_roi;                              \n"
"                                                                              \n"
"              if (o->orientation == GEGL_ORIENTATION_HORIZONTAL)              \n"
"                {                                                             \n"
"                  required_for_output.x     = in_rect->x;                     \n"
"                  required_for_output.width = in_rect->width;                 \n"
"                }                                                             \n"
"              else                                                            \n"
"                {                                                             \n"
"                  required_for_output.y      = in_rect->y;                    \n"
"                  required_for_output.height = in_rect->height;               \n"
"                }                                                             \n"
"                                                                              \n"
"              if (!o->clip_extent)                                            \n"
"                required_for_output =                                         \n"
"                  gegl_gblur_1d_enlarge_extent (o, &required_for_output);     \n"
"            }                                                                 \n"
"          /* pass-through case */                                             \n"
"          else                                                                \n"
"            return *output_roi;                                               \n"
"        }                                                                     \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      required_for_output = gegl_gblur_1d_enlarge_extent (o, output_roi);     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return required_for_output;                                                 \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"gegl_gblur_1d_get_bounding_box (GeglOperation *operation)                     \n"
"{                                                                             \n"
"  GeglProperties          *o       = GEGL_PROPERTIES (operation);             \n"
"  const GeglRectangle *in_rect =                                              \n"
"    gegl_operation_source_get_bounding_box (operation, \"input\");            \n"
"                                                                              \n"
"  if (! in_rect)                                                              \n"
"    return *GEGL_RECTANGLE (0, 0, 0, 0);                                      \n"
"                                                                              \n"
"  if (gegl_rectangle_is_infinite_plane (in_rect))                             \n"
"    return *in_rect;                                                          \n"
"                                                                              \n"
"  if (o->clip_extent)                                                         \n"
"    {                                                                         \n"
"      return *in_rect;                                                        \n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      /* We use the FIR convolution length for both the FIR and the IIR case */\n"
"      return gegl_gblur_1d_enlarge_extent (o, in_rect);                       \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglRectangle                                                          \n"
"gegl_gblur_1d_get_cached_region (GeglOperation       *operation,              \n"
"                                 const GeglRectangle *output_roi)             \n"
"{                                                                             \n"
"  GeglRectangle      cached_region;                                           \n"
"  GeglProperties        *o       = GEGL_PROPERTIES (operation);               \n"
"  GeglGblur1dFilter  filter  = filter_disambiguation (o->filter, o->std_dev); \n"
"                                                                              \n"
"  cached_region = *output_roi;                                                \n"
"                                                                              \n"
"  if (filter == GEGL_GBLUR_1D_IIR)                                            \n"
"    {                                                                         \n"
"      const GeglRectangle in_rect =                                           \n"
"        gegl_gblur_1d_get_bounding_box (operation);                           \n"
"                                                                              \n"
"      if (! gegl_rectangle_is_empty (&in_rect) &&                             \n"
"          ! gegl_rectangle_is_infinite_plane (&in_rect))                      \n"
"        {                                                                     \n"
"          GeglProperties *o = GEGL_PROPERTIES (operation);                    \n"
"                                                                              \n"
"          cached_region = *output_roi;                                        \n"
"                                                                              \n"
"          if (o->orientation == GEGL_ORIENTATION_HORIZONTAL)                  \n"
"            {                                                                 \n"
"              cached_region.x     = in_rect.x;                                \n"
"              cached_region.width = in_rect.width;                            \n"
"            }                                                                 \n"
"          else                                                                \n"
"            {                                                                 \n"
"              cached_region.y      = in_rect.y;                               \n"
"              cached_region.height = in_rect.height;                          \n"
"            }                                                                 \n"
"        }                                                                     \n"
"    }                                                                         \n"
"                                                                              \n"
"  return cached_region;                                                       \n"
"}                                                                             \n"
"                                                                              \n"
"static GeglAbyssPolicy                                                        \n"
"to_gegl_policy (GeglGblur1dPolicy policy)                                     \n"
"{                                                                             \n"
"  switch (policy)                                                             \n"
"    {                                                                         \n"
"    case (GEGL_GBLUR_1D_ABYSS_NONE):                                          \n"
"      return GEGL_ABYSS_NONE;                                                 \n"
"      break;                                                                  \n"
"    case (GEGL_GBLUR_1D_ABYSS_CLAMP):                                         \n"
"      return GEGL_ABYSS_CLAMP;                                                \n"
"      break;                                                                  \n"
"    case (GEGL_GBLUR_1D_ABYSS_WHITE):                                         \n"
"      return GEGL_ABYSS_WHITE;                                                \n"
"      break;                                                                  \n"
"    case (GEGL_GBLUR_1D_ABYSS_BLACK):                                         \n"
"      return GEGL_ABYSS_BLACK;                                                \n"
"      break;                                                                  \n"
"    default:                                                                  \n"
"      g_warning (\"gblur-1d: unsupported abyss policy\");                     \n"
"      return GEGL_ABYSS_NONE;                                                 \n"
"    }                                                                         \n"
"}                                                                             \n"
"                                                                              \n"
"static gboolean                                                               \n"
"gegl_gblur_1d_process (GeglOperation       *operation,                        \n"
"                       GeglBuffer          *input,                            \n"
"                       GeglBuffer          *output,                           \n"
"                       const GeglRectangle *result,                           \n"
"                       gint                 level)                            \n"
"{                                                                             \n"
"  GeglProperties *o      = GEGL_PROPERTIES (operation);                       \n"
"  const Babl *format = gegl_operation_get_format (operation, \"output\");     \n"
"                                                                              \n"
"  GeglGblur1dFilter filter;                                                   \n"
"  GeglAbyssPolicy   abyss_policy = to_gegl_policy (o->abyss_policy);          \n"
"                                                                              \n"
"  filter = filter_disambiguation (o->filter, o->std_dev);                     \n"
"                                                                              \n"
"  if (filter == GEGL_GBLUR_1D_IIR)                                            \n"
"    {                                                                         \n"
"      gdouble b[4], m[3][3];                                                  \n"
"                                                                              \n"
"      iir_young_find_constants (o->std_dev, b, m);                            \n"
"                                                                              \n"
"      if (o->orientation == GEGL_ORIENTATION_HORIZONTAL)                      \n"
"        iir_young_hor_blur (input, result, output, b, m, abyss_policy, format);\n"
"      else                                                                    \n"
"        iir_young_ver_blur (input, result, output, b, m, abyss_policy, format);\n"
"    }                                                                         \n"
"  else                                                                        \n"
"    {                                                                         \n"
"      gfloat *cmatrix;                                                        \n"
"      gint    clen;                                                           \n"
"                                                                              \n"
"      clen = fir_gen_convolve_matrix (o->std_dev, &cmatrix);                  \n"
"                                                                              \n"
"      /* FIXME: implement others format cases */                              \n"
"      if (gegl_operation_use_opencl (operation) &&                            \n"
"          format == babl_format (\"RaGaBaA float\"))                          \n"
"        if (fir_cl_process(input, output, result, format,                     \n"
"                           cmatrix, clen, o->orientation, abyss_policy))      \n"
"          return TRUE;                                                        \n"
"                                                                              \n"
"      if (o->orientation == GEGL_ORIENTATION_HORIZONTAL)                      \n"
"        fir_hor_blur (input, result, output, cmatrix, clen, abyss_policy, format);\n"
"      else                                                                    \n"
"        fir_ver_blur (input, result, output, cmatrix, clen, abyss_policy, format);\n"
"                                                                              \n"
"      gegl_free (cmatrix);                                                    \n"
"    }                                                                         \n"
"                                                                              \n"
"  return  TRUE;                                                               \n"
"}                                                                             \n"
"                                                                              \n"
"/* Pass-through when trying to perform IIR on an infinite plane               \n"
" */                                                                           \n"
"static gboolean                                                               \n"
"operation_process (GeglOperation        *operation,                           \n"
"                   GeglOperationContext *context,                             \n"
"                   const gchar          *output_prop,                         \n"
"                   const GeglRectangle  *result,                              \n"
"                   gint                  level)                               \n"
"{                                                                             \n"
"  GeglOperationClass  *operation_class;                                       \n"
"  GeglProperties          *o = GEGL_PROPERTIES (operation);                   \n"
"  GeglGblur1dFilter    filter  = filter_disambiguation (o->filter, o->std_dev);\n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (gegl_op_parent_class);              \n"
"                                                                              \n"
"  if (filter == GEGL_GBLUR_1D_IIR)                                            \n"
"    {                                                                         \n"
"      const GeglRectangle *in_rect =                                          \n"
"        gegl_operation_source_get_bounding_box (operation, \"input\");        \n"
"                                                                              \n"
"      if (in_rect && gegl_rectangle_is_infinite_plane (in_rect))              \n"
"        {                                                                     \n"
"          gpointer in = gegl_operation_context_get_object (context, \"input\");\n"
"          gegl_operation_context_take_object (context, \"output\",            \n"
"                                              g_object_ref (G_OBJECT (in)));  \n"
"          return TRUE;                                                        \n"
"        }                                                                     \n"
"    }                                                                         \n"
"  /* chain up, which will create the needed buffers for our actual            \n"
"   * process function                                                         \n"
"   */                                                                         \n"
"  return operation_class->process (operation, context, output_prop, result,   \n"
"                                   gegl_operation_context_get_level (context));\n"
"}                                                                             \n"
"                                                                              \n"
"static void                                                                   \n"
"gegl_op_class_init (GeglOpClass *klass)                                       \n"
"{                                                                             \n"
"  GeglOperationClass       *operation_class;                                  \n"
"  GeglOperationFilterClass *filter_class;                                     \n"
"                                                                              \n"
"  operation_class = GEGL_OPERATION_CLASS (klass);                             \n"
"  filter_class    = GEGL_OPERATION_FILTER_CLASS (klass);                      \n"
"                                                                              \n"
"  filter_class->process                    = gegl_gblur_1d_process;           \n"
"  operation_class->prepare                 = gegl_gblur_1d_prepare;           \n"
"  operation_class->process                 = operation_process;               \n"
"  operation_class->get_bounding_box        = gegl_gblur_1d_get_bounding_box;  \n"
"  operation_class->get_required_for_output = gegl_gblur_1d_get_required_for_output;\n"
"  operation_class->get_cached_region       = gegl_gblur_1d_get_cached_region; \n"
"  operation_class->opencl_support          = TRUE;                            \n"
"  gegl_operation_class_set_keys (operation_class,                             \n"
"    \"name\",       \"gegl:gblur-1d\",                                        \n"
"    \"categories\", \"hidden:blur\",                                          \n"
"    \"description\",                                                          \n"
"        _(\"Performs an averaging of neighboring pixels with the \"           \n"
"          \"normal distribution as weighting\"),                              \n"
"        NULL);                                                                \n"
"}                                                                             \n"
"                                                                              \n"
"#endif                                                                        \n"
;
