# IndieWeb Library

A Rust library for implementing IndieWeb standards and algorithms.

## Overview

Provides tools for IndieWeb protocols including authentication, publishing, and communication standards.

## Installation

Install from crates.io for stable releases:

```bash
cargo add indieweb
```

Or from Git for the latest development version:

```bash
cargo add --git https://git.sr.ht/~jacky/indieweb-rust indieweb
```

## Modules

- **Algorithms**: Link resolution, authorship detection, representative h-card parsing
- **Standards**: IndieAuth, Micropub, Webmention, WebSub implementations
- **Traits**: Common interfaces for IndieWeb interactions

## Experimental Features

The following features are experimental and may change in future releases. Enable them with Cargo feature flags:

| Feature              | Description                             | Specification                                  |
| -------------------- | --------------------------------------- | ---------------------------------------------- |
| WebSub               | Real-time content notification protocol | [WebSub](https://www.w3.org/TR/websub/)         |
| Microsub             | Feed consumption and interaction API    | [Microsub](https://indieweb.org/Microsub-spec)  |
| Microsub PTD         | Post type discovery for entries         | [PTD](https://indieweb.org/post-type-discovery) |
| Microsub Source      | Timeline filtering by feed source       | [Source Parameter](https://indieweb.org/Microsub-spec#Source_Parameter) |
| Microsub Metadata    | Enhanced feed metadata (name/photo)     | [Feed Metadata](https://indieweb.org/Microsub-spec#Feed_Metadata) |
| Vouch                | Anti-spam extension for Webmention      | [Vouch](https://indieweb.org/Vouch)             |
| Salmention           | Upstream response propagation           | [Salmention](https://indieweb.org/Salmention)   |
| Authorship           | Post authorship resolution algorithm    | [Authorship](https://indieweb.org/authorship)   |
| Micropub Channels    | Channel management extensions           | [Channels](https://indieweb.org/channels)       |
| Micropub Syndication | Content syndication extensions          | [Syndication](https://indieweb.org/syndication) |
| Micropub Relations   | Post relationship extensions            | [Relations](https://indieweb.org/relation)      |

### Usage Examples

```toml
# Enable WebSub support
indieweb = { version = "0.6", features = ["experimental_websub"] }

# Enable multiple experimental features
indieweb = { version = "0.6", features = ["experimental_websub", "experimental_microsub"] }

# Enable all experimental features
indieweb = { version = "0.6", features = ["experimental"] }
```

**Note:** Experimental features may have breaking changes between releases. Use with caution in production applications.

### Microsub Extensions

The Microsub implementation includes several extensions that enhance feed reading capabilities:

#### Post Type Discovery (PTD)
Automatically classifies posts by type (article, note, photo, video, audio, like, reply, etc.):

```rust
use indieweb::standards::microsub::{Entry, Content};

// Entry with PTD-processed kind field
let entry = Entry {
    content: Some(Content {
        text: Some("This is a blog post with a title".to_string()),
        html: None,
    }),
    name: Some("My Blog Post".to_string()),
    // kind will be automatically set to "article"
    ..Default::default()
};
```

#### Source Parameter Filtering
Filter timeline results by specific feed sources:

```rust
use indieweb::standards::microsub::MicrosubServer;

// Get timeline filtered to specific feed
let timeline = server.timeline(
    "my-channel",
    None, None, None, // before, after, limit
    Some("https://example.com/feed.xml") // source filter
).await?;
```

#### Enhanced Feed Metadata
Feeds include automatically populated name and photo fields:

```rust
use indieweb::standards::microsub::MicrosubServer;

// Follow feed with enhanced metadata
let feed = server.follow("my-channel", "https://github.com/user/blog.atom").await?;
// feed.name = Some("GITHUB Feed")
// feed.photo = Some("https://github.com/favicon.ico")
```

## Usage

For local development, add to your Cargo.toml:

```toml
[dependencies]
indieweb = { path = "../path/to/library" }
```

For detailed API usage, see the source code.