/* ====================================================================
 * Copyright (c) 2003-2007, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

// sc
#include "config.h"
#include "DirEntryLvi.h"
#include "DragDropMimeTypes.h"
#include "sublib/StringUtil.h"
#include "svn/DirEntry.h"

// qt
#include <QtGui/QMimeSource>


/*static*/ enum Columns // gcc complains about the static
{
  cName = 0,
  cRevision,
  cSize,
  cProperties,
  cAuthor,
  cTime,
  cNop
};


DirEntryLvi::DirEntryLvi( Q3ListView* parent, svn::DirEntryPtr entry )
: super(parent), _entry(entry), _rename(false)
{
  init();
}

DirEntryLvi::DirEntryLvi( Q3ListViewItem* parent, svn::DirEntryPtr entry )
: super(parent), _entry(entry), _rename(false)
{
  init();
}

void DirEntryLvi::init()
{
  QString url = QString::fromUtf8( _entry->getName().getStr() );
  url.remove( 0, _parentUrl.length()+1 );
  
  setDropEnabled(_entry->isDir());

  if( url.length() == 0 )
  {
    // ie. "."
    return;
  }

  setRenameEnabled( cName, true );
  setDragEnabled(true);
}

void DirEntryLvi::okRename( int col )
{
  // tell the text method to return the new text and not the one
  // from the status.
  _rename = true;
  super::okRename(col);
  _rename = false;
}

QString DirEntryLvi::text( int column ) const
{
  switch( column )
  {
  case cName:
    {
      if( _rename )
      {
        return super::text(cName);
      }
      
      QString url = QString::fromUtf8( (const char*)_entry->getName() );
      url.remove( 0, _parentUrl.length()+1 );

      if( url.length() == 0 )
      {
        return ".";
      }
      return url;

    }
  }

  if( _entry->isNop() )
  {
    return "";
  }

  switch( column )
  {
  case cRevision:
    {
      return QString().sprintf( "%10ld", (unsigned long)_entry->getLastRevision() );
    }
  case cTime:
    {
      //char time[APR_RFC822_DATE_LEN] = {};
      //apr_rfc822_date( time, _entry->getLastDate() );
      char time[APR_CTIME_LEN] = {};
      apr_ctime( time, _entry->getLastDate() );
      return time;
    }
  case cAuthor:
    {
      return QString::fromUtf8(_entry->getLastAuthor().getStr());
    }
  case cSize:
    {
      if( _entry->isDir() )
      {
        return _q("folder");
      }
      else
      {
        char buf[5];
        apr_strfsize( _entry->getSize(), buf );
        return buf;

        //return QString().sprintf( "%10d", (unsigned long)_entry->getSize() );
      }
    }
  case cProperties:
    {
      return _entry->hasProperties() ? _q("yes") : _q("no");
    }
  case cNop:
    {
      return "";
    }
  default:
    {
      return "invalid column";
    }
  }
}


const QString dir("a-");
const QString file("b-");

const QString& DirEntryLvi::getSortPrefix() const
{
  // we want to keep dirs together with dirs and the same for files.
  // So we add a common dir or file prefix to the sort key.

  if( _entry->isDir() )
  {
    return dir;
  }
  else
  {
    return file;
  }
}

QString DirEntryLvi::key( int column, bool ascending ) const
{
  switch( column )
  {
  case cName:
    {
      return getSortPrefix() + text(column);
    }
  case cTime:
    {
      char time[APR_RFC822_DATE_LEN] = {};

      apr_time_exp_t timet;
      apr_time_exp_gmt( &timet, _entry->getLastDate() );

      apr_size_t size;
      apr_strftime( time, &size, APR_RFC822_DATE_LEN-1, "%Y-%m-%d %H:%M:%S", &timet );
      //printf( "date: %s\n", time );
      return /*getSortPrefix()*/ + time;
    }
  case cSize:
    {
      QString s = getSortPrefix() + QString().sprintf( "%010lu", (unsigned long)_entry->getSize() );  
      //printf( "size: %s\n", s.ascii() );
      return s;
    }
  case cProperties:
  case cAuthor:
  case cRevision:
  default:
    {
      return text(column);
    }
  }
}

int DirEntryLvi::compare( Q3ListViewItem* i, int col, bool ascending ) const
{
  return sc::compare( key(col,ascending), i->key(col,ascending) );
}

void DirEntryLvi::setParentUrl( const QString& parentUrl )
{
  _parentUrl = parentUrl;
}

const QString& DirEntryLvi::getParentUrl() const
{
  return _parentUrl;
}

const svn::DirEntryPtr DirEntryLvi::getEntry() const
{
  return _entry;
}

bool DirEntryLvi::acceptDrop( const QMimeSource* mime ) const
{
  return dropEnabled() && mime->provides( ScMimeTypeRpFilesItem );
}
