/* persdeeppathstore.cc
 * This file belongs to Worker, a file manager for UN*X/X11.
 * Copyright (C) 2012-2015 Ralf Hoffmann.
 * You can contact me at: ralf@boomerangsworld.de
 *   or http://www.boomerangsworld.de/worker
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 */

#include "persdeeppathstore.hh"
#include <iostream>
#include <fstream>
#include <sstream>
#include "nwc_fsentry.hh"
#include "aguix/util.h"
#include "datei.h"

PersDeepPathStore::PersDeepPathStore( const std::string &filename ) : m_filename( filename ),
                                                                      m_lastmod( 0 ),
                                                                      m_lastsize( 0 )
{
    m_bg_threading.thread_id = std::thread( [this]() { bg_thread(); } );
}

PersDeepPathStore::~PersDeepPathStore()
{
    m_bg_threading.order_cv.lock();
    m_bg_threading.stop_thread = true;
    m_bg_threading.order_cv.signal();
    m_bg_threading.order_cv.unlock();

    m_bg_threading.thread_id.join();
}

void PersDeepPathStore::storePath( const std::string &path, time_t ts )
{
    m_bg_threading.order_cv.lock();

    m_bg_threading.push_list.push_back( std::make_pair( path, ts ) );

    m_bg_threading.order_cv.signal();

    m_bg_threading.order_cv.unlock();
}

std::list< std::pair< std::string, time_t > > PersDeepPathStore::getPaths()
{
    // get the main lock
    m_bg_threading.order_cv.lock();

    // get the result lock (always successful)
    m_bg_threading.result_cv.lock();

    // get next ID for results
    int id = m_bg_threading.current_result_id + 1;

    // trigger re-read
    m_bg_threading.wanted_result_id = id;

    // wake up thread
    m_bg_threading.order_cv.signal();

    // und release main lock (but keep result lock)
    m_bg_threading.order_cv.unlock();

    // now wait until the result is available
    while ( m_bg_threading.current_result_id != id ) {
        m_bg_threading.result_cv.wait();
    }

    // copy them
    std::list< std::pair< std::string, time_t > > res = m_bg_threading.result_list;

    // and unlock the result lock
    m_bg_threading.result_cv.unlock();

    return res;
}

void PersDeepPathStore::read()
{
    std::string fn = m_filename;
    fn += "-v2";

    std::string lock_file = fn;
    lock_file += ".lck";

    NWC::FSEntry tfe( fn );

    if ( ! tfe.entryExists() ) {
        fn = m_filename;
    }

    NWC::FSEntry fe( fn );

    if ( ! fe.entryExists() ) return;

    if ( fe.isLink() ) {
        if ( fe.stat_dest_lastmod() == m_lastmod &&
             fe.stat_dest_size() == m_lastsize ) return;
    } else {
        if ( fe.stat_lastmod() == m_lastmod &&
             fe.stat_size() == m_lastsize ) return;
    }

    Datei d;

    d.open( lock_file.c_str(),
            "w" );
    d.lock();
    
    std::ifstream ifile( fn.c_str() );
    std::string line;

    m_store.clear();
    
    if ( fe.isLink() ) {
        m_lastmod = fe.stat_dest_lastmod();
        m_lastsize = fe.stat_dest_size();
    } else {
        m_lastmod = fe.stat_lastmod();
        m_lastsize = fe.stat_size();
    }

    if ( ifile.is_open() ) {

        m_store.clear();

        time_t now = time( NULL );

        while ( std::getline( ifile, line ) ) {
            if ( line.empty() ) {
                continue;
            }

            if ( line[0] == '/' ) {
                m_store.storePath( line, now );
            } else {
                std::string::size_type pos;
                pos = line.find( ':', 0 );
        
                if ( pos != std::string::npos ) {
                    std::string ts_str( line, 0, pos );
                    std::string path( line, pos + 1 );
                    time_t ts = 0;

                    if ( AGUIXUtils::convertFromString( ts_str, ts ) ) {
                        m_store.storePath( path, ts );
                    }
                }
            }
        }
    }

    d.unlock();
}

void PersDeepPathStore::write()
{
    std::string fn_v2 = m_filename;
    fn_v2 += "-v2";

    std::string lock_file = fn_v2;
    lock_file += ".lck";

    Datei d;

    d.open( lock_file.c_str(),
            "w" );
    d.lock();

    std::ofstream ofile( fn_v2.c_str() );
    std::list< std::pair< std::string, time_t > > l = m_store.getPaths();

    for ( auto it1 = l.begin();
          it1 != l.end();
          it1++ ) {
        ofile << it1->second << ":";
        ofile << it1->first << std::endl;
    }

    d.unlock();

    m_lastmod = time( NULL );
    m_lastsize = ofile.tellp();
}

void PersDeepPathStore::bg_thread()
{
    m_bg_threading.order_cv.lock();

    while ( ! m_bg_threading.stop_thread ) {
        if ( m_bg_threading.push_list.empty() &&
             m_bg_threading.relocate_list.empty() &&
             m_bg_threading.remove_list.empty() &&
             m_bg_threading.current_result_id == m_bg_threading.wanted_result_id &&
             m_bg_threading.stop_thread == false ) {
            m_bg_threading.order_cv.wait();
        }

        if ( ! m_bg_threading.push_list.empty() ) {
            std::list< std::pair< std::string, time_t > > push_list_copy = m_bg_threading.push_list;

            m_bg_threading.push_list.clear();

            m_bg_threading.order_cv.unlock();

            addPaths( push_list_copy );

            m_bg_threading.order_cv.lock();
        }

        if ( ! m_bg_threading.relocate_list.empty() ) {
            auto relocate_list_copy = m_bg_threading.relocate_list;

            m_bg_threading.relocate_list.clear();

            m_bg_threading.order_cv.unlock();

            processRelocateOrders( relocate_list_copy );

            m_bg_threading.order_cv.lock();
        }

        if ( ! m_bg_threading.remove_list.empty() ) {
            auto remove_list_copy = m_bg_threading.remove_list;

            m_bg_threading.remove_list.clear();

            m_bg_threading.order_cv.unlock();

            removeEntries( remove_list_copy );

            m_bg_threading.order_cv.lock();
        }


        if ( m_bg_threading.current_result_id != m_bg_threading.wanted_result_id ) {
            m_bg_threading.result_cv.lock();

            read();
            m_bg_threading.result_list = m_store.getPaths();

            m_bg_threading.current_result_id = m_bg_threading.wanted_result_id;

            m_bg_threading.result_cv.signal();
            m_bg_threading.result_cv.unlock();
        }
    }
    m_bg_threading.order_cv.unlock();
}

void PersDeepPathStore::addPaths( const std::list< std::pair< std::string, time_t > > &list )
{
    read();

    DeepPathNode::node_change_t changed = DeepPathNode::NODE_UNCHANGED;

    for ( auto &p : list ) {
        m_store.storePath( p.first, p.second, &changed );
    }

    if ( changed != DeepPathNode::NODE_UNCHANGED ) {
        write();
    }
}

void PersDeepPathStore::relocateEntries( const std::string &dest,
                                         const std::string &source,
                                         time_t ts,
                                         bool move )
{
    m_bg_threading.order_cv.lock();

    m_bg_threading.relocate_list.push_back( std::make_tuple( dest, source, ts, move ) );

    m_bg_threading.order_cv.signal();

    m_bg_threading.order_cv.unlock();
}

void PersDeepPathStore::processRelocateOrders( const std::list< std::tuple< std::string, std::string, time_t, bool > > &relocate_list )
{
    read();

    DeepPathNode::node_change_t changed = DeepPathNode::NODE_UNCHANGED;

    for ( auto &e : relocate_list ) {
        m_store.relocateEntries( std::get<0>( e ),
                                 std::get<1>( e ),
                                 std::get<2>( e ),
                                 std::get<3>( e ),
                                 &changed);
    }

    if ( changed != DeepPathNode::NODE_UNCHANGED ) {
        write();
    }
}

void PersDeepPathStore::removeEntry( const std::string &path )
{
    m_bg_threading.order_cv.lock();

    m_bg_threading.remove_list.push_back( path );

    m_bg_threading.order_cv.signal();

    m_bg_threading.order_cv.unlock();
}

void PersDeepPathStore::removeEntries( const std::list< std::string > &paths )
{
    read();

    bool changed = false;

    for ( auto &e : paths ) {
        if ( m_store.removeEntry( e ) ) {
            changed = true;
        }
    }

    if ( changed ) {
        write();
    }
}
