/*
 * CHEST, chess analyst.  For Copyright notice read file "COPYRIGHT".
 *
 * $Source: /home/heiner/ca/chest/RCS/solopt.c,v $
 * $Id: solopt.c,v 3.16 1999/10/01 23:35:56 heiner Exp $
 *
 *	Optimized (shortened) solution.
 *
 * Try to find branches that print equal, and combine them.
 *
 * The basic operation is as follows:
 * For a move list (attacker or defender)
 *  (a) Create uniquely (and compressed) all the subtrees.
 *      This also yields levels.
 *  (b) Sort the temporary constructs (level, move-string)
 *  (c) For a defender list (complete list of legal moves):
 *      (c1) When list length > 1, check whether all subtrees are equal
 *           and not empty.  If so, make one "any" node.
 *      (c2) When list length > 2, find most common non-empty subtree.
 *           Combine them with "else" notation.
 *  (?) Promotion blocking
 *  (d) Starting at the end of the sorted list, create new unique (sub-)lists.
 *
 * FFS: space to wanted column leaving alternatingly '.' and ','
 *	where a move would be.
 */

#include <stdio.h>
#include "types.h"
#include "board.h"
#include "job.h"
#include "analyse.h"
#include "mlsubr.h"
#include "move.h"
#include "move_gen.h"
#include "output.h"
#include "solution.h"		/* fill_conti, sol_analyse */
#include "solopt.h"

#if PROD_LEV
# define SOLOPT_STATS	0
#else
# ifndef SOLOPT_STATS
#  define SOLOPT_STATS	1
# endif
#endif

#undef  THIS_STATS
#define THIS_STATS	SOLOPT_STATS
#include "statsupp.h"


#define SOL_DUMPING	0	/* CF: debugging: dump tree */

#ifndef SOL_USE_TABS
# define SOL_USE_TABS	1	/* CF: whether to use tab for indenting */
#endif

#ifndef SOL_NODES
# define SOL_NODES	30000
#endif

#if (SOL_NODES) >= 65531
# include "/// >>>>> SOL_NODES too large <<<<< ///"
#endif

#if 0				/* CF */
static char	sym_any[]	= "any";
static char	sym_else[]	= "etc";
#else
static char	sym_any[]	= "=*=";
static char	sym_else[]	= "-*-";
#endif

#define MV_CHARS	6

typedef uint16		 SolRef;
typedef unsigned int	rSolRef;

typedef struct SolNode
{
    uint16	sn_refcnt;		/* reference count */
    SolRef	sn_hforw;		/* index link: succ in hash class */
    SolRef	sn_hback;		/* index link: pred in hash class */
    SolRef	sn_next;		/* index link: next brother in list */
    SolRef	sn_down;		/* index link: subtree */
    uint8	sn_level;		/* sort of tree height */
    uint8	sn_hclass;		/* hash class */
    char	sn_move[MV_CHARS];
} SolNode;

#define XNIL	((SolRef)~0UL)
#define XBAD	((SolRef)~1UL)


#if SOLOPT_STATS
static Counter	sc_tree_try;
static Counter	sc_tree_succ;

static Counter	sc_node_search;
static Counter	sc_node_found;

static Counter	sc_any_try;
static Counter	sc_any_succ;
static Counter	sc_else_try;
static Counter	sc_else_succ;

static Counter	sc_cmp_move;
static Counter	sc_cmp_templ;
static Counter	sc_sort_list;

#endif	/* SOLOPT_STATS */


    static void
stuff_move( register char* to, register const char* from )
{
    register int	i;

    while( *from && (*from == ' ') ) {
	++from;
    }
    for( i=0 ; i<MV_CHARS ; ++i ) {
	if( ! (to[i] = from[i]) ) {
	    break;
	}
    }
    while( (i > 0) && (to[i-1] == ' ') ) {
	--i;
    }
    for( ; i < MV_CHARS ; ++i ) {
	to[i] = '\0';
    }
}


    static int
cmp_mov( register const char* str1, register const char* str2 )
{
    register int	i;

    scinc(sc_cmp_move);
    i = 0;
    while( str1[i] == str2[i] ) {
	if( ! str1[i] || (++i >= MV_CHARS) ) {
	    return 0;	/* equal */
	}
    }
    return (str1[i] < str2[i]) ? -1 : 1;
}


    static int
cmp_templ( register SolNode* t1, register SolNode* t2 )
{
    register int	rc;

    scinc(sc_cmp_templ);
    if( t1->sn_level != t2->sn_level ) {
	return (t1->sn_level > t2->sn_level) ? -1 : 1;	/* FFS reverse ? */
    }
    if( (rc = cmp_mov(t1->sn_move, t2->sn_move)) ) {
	return rc;
    }
    if( t1->sn_down != t2->sn_down ) {
	return (t1->sn_down < t2->sn_down) ? -1 : 1;
    }
    return 0;
}


    static SolRef		/* first index into arr */
sort_templ_arr( register SolNode* arr, register int elems )
{
    register SolRef	list;
    register SolRef	curr;
    register SolRef	next;
    register int	i;

    if( elems <= 1 ) {
	list = ((elems == 1) ? 0 : XNIL);
    }else {
	scinc(sc_sort_list);
				/* insertion sort  ... */
	list = elems-1;		/* last element */
	for( i=list ; --i >= 0 ; ) {
	    if( cmp_templ(arr+i, arr+list) <= 0 ) {	/* before first */
		arr[i].sn_next = list;
		list = i;
	    }else {
		curr = list;
		for( ; (next = arr[curr].sn_next) != XNIL ; curr=next ) {
		    if( cmp_templ(arr+i, arr+next) <= 0 ) {
			break;
		    }
		}
		arr[   i].sn_next = next;
		arr[curr].sn_next = i;
	    }
	}
    }
    return list;
}

#if 0
    static uint32
tree_lines( register SolRef root )
{
    register const SolNode*	p;
    register rSolRef		curr;
    register uint32		sum;

    sum = 0;
    for( curr=root ; curr!=XNIL ; curr=p->sn_next ) {
	p = &NODE(curr);
	sum += tree_lines(p->sn_down);
    }
    if( sum < 1 ) sum = 1;	/* FFS: here? */
    return sum;
}

    static int
tree_columns( register SolRef root )
{
    register const SolNode*	p;
    register rSolRef		curr;
    register int		maxcols;
    register int		cand;

    maxcols = 0;
    for( curr=root ; curr!=XNIL ; curr=p->sn_next ) {
	p = &NODE(curr);
	cand = 1 + tree_columns(p->sn_down);
	if( maxcols < cand ) {
	    maxcols = cand;
	}
    }
    return maxcols;
}

    static Bool
tree_cols_is_ge( register SolRef root, register int columns )
{
    register const SolNode*	p;
    register rSolRef		curr;

    if( columns <= 0 ) {
	return TRUE;
    }
    for( curr=root ; curr!=XNIL ; curr=p->sn_next ) {
	p = &NODE(curr);
	if( tree_cols_is_ge(p->sn_down, columns-1) ) {
	    return TRUE;
	}
    }
    return FALSE;	/* no subtree was sufficient */
}

    static Bool
tree_cols_is_le( register SolRef root, register int columns )
{
    register const SolNode*	p;
    register rSolRef		curr;

    if( columns < 0 ) {
	return FALSE;		/* not even the empty tree has negative cols */
    }
    for( curr=root ; curr!=XNIL ; curr=p->sn_next ) {
	p = &NODE(curr);
	if( ! tree_cols_is_le(p->sn_down, columns-1) ) {
	    return FALSE;
	}
    }
    return TRUE;		/* no subtree was larger */
}

    static Bool
tree_cols_is_eq( register SolRef root, register int columns )
{
    register const SolNode*	p;
    register rSolRef		curr;

    if( columns <= 0 ) {
	return (columns == 0) && (root == XNIL);
    }
    for( curr=root ; curr!=XNIL ; curr=p->sn_next ) {
	p = &NODE(curr);
	if( ! tree_cols_is_eq(p->sn_down, columns-1) ) {
	    return FALSE;
	}
    }
    return TRUE;		/* no subtree was different */
}
#endif

/*---------------------------------------------------------------------------*/
/*
 * Allocation and retrieval strategy for SolNode's:
 *	All SolNode-objects are allocated statically in a
 *	global array "sol__node".  References to them are not implemented
 *	via pointers, but by indexes into this array.
 *	Indexes are (uint16), and hence much denser than pointers.
 *	This also limits the total number of objects to 65534
 *	(2 spare indices for NIL/BAD).
 *
 *	There is an allocation index "sol_allo": that one and
 *	all behind are free (not allocated).
 *	There is an explicit free list "sol_free", linking
 *	unused objects via "sn_next".  This is used as stack,
 *	what allows for a singly linked list.
 *
 *	When searching for a (probably old) object, we have to
 *	avoid linear searches within up to say 30K nodes.
 *	Hence we build hash codes, and hash code classes,
 *	and doubly link within the hash classes.
 *	Nodes store just their hash class, in order to find
 *	the hash class anchor.  Hash classes are (uint8),
 *	as such a small entity just fits into a gap in the node.
 */

#define SOL_HBITS	8
#define SOL_HCLASSES	(1 << SOL_HBITS)

I0static SolNode	sol__node[ SOL_NODES ];
static SolRef		sol_free;	/* index to unused (linked list) */
static SolRef		sol_allo;	/* array allocation point */
static SolRef		solhclass[ SOL_HCLASSES ];

#define NODE(sref)	(sol__node[sref])

    static void
rel_solref( SolRef ref )
{
    if( ref < SOL_NODES ) {
	    register SolNode*	p;
	p = &NODE(ref);
	if( (p->sn_refcnt == 0) || (p->sn_refcnt >= SOL_NODES) ) {
	    printf("freeing free solnode %d (%d)\n", ref, p->sn_refcnt);
	    panic( "freeing free solnode");
	}
	if( 0 == (p->sn_refcnt -= 1) ) {
	    rel_solref(p->sn_down);
	    rel_solref(p->sn_next);
					/* remove from hash class list ... */
	    if( p->sn_hforw != XNIL ) {
		NODE(p->sn_hforw).sn_hback = p->sn_hback;
	    }
	    if( p->sn_hback != XNIL ) {
		NODE(p->sn_hback).sn_hforw = p->sn_hforw;
	    }else {
		solhclass[p->sn_hclass] = p->sn_hforw;
	    }
					/* insert into free list ... */
	    p->sn_next = sol_free; sol_free = ref;
	}
    }
}


    static void
rel_templ_list(
    register SolNode*	arr,		/* template vector */
    SolRef		frst)		/* first in list */
{
    register rSolRef	ix;

    /*
     * The direct elements are NOT to be freed (they are static).
     * Only their down-links are dynamic and to be freed.
     */
    for( ix = frst ; ix != XNIL ; ix=arr[ix].sn_next ) {
	rel_solref(arr[ix].sn_down);
    }
}

/*---------------------------------------------------------------------------*/

#if SOL_DUMPING
    static void
sol_dump_node( SolRef curr, int indent )
{
    if( curr != XNIL ) {
	    register SolNode*	p;
	    register int	i;
	p = &NODE(curr);
	i = indent * 4;
	if( i > 40 ) i = 40;
	printf("%*s", i, "");
	printf("%2d|%5u:", indent, (unsigned)curr);
	printf(" %2u=r", (unsigned)p->sn_refcnt);
	printf(" %5u=n", (unsigned)p->sn_next);
	printf(" %5u=d", (unsigned)p->sn_down);
	printf(" %2u=L", (unsigned)p->sn_level);
	printf(" %2x=C", (unsigned)p->sn_hclass);
	printf(" %-.*s", MV_CHARS, p->sn_move);
	printf("\n");
    }
}

    static void
sol_dump_tree( SolRef root, int indent )
{
    register SolNode*	p;
    register rSolRef	curr;
    register int	i;

    for( curr=root ; curr!=XNIL ; curr=p->sn_next ) {
	p = &NODE(curr);
	sol_dump_node(curr, indent);
	if( p->sn_down != XNIL ) {
	    sol_dump_tree(p->sn_down, indent+1);
	}
    }
}
#else	/* ! SOL_DUMPING */
# define sol_dump_node(curr, indent)	/*empty*/
# define sol_dump_tree(root, indent)	/*empty*/
#endif	/* ! SOL_DUMPING */

/*---------------------------------------------------------------------------*/

    static void
try_any(
    register SolNode*	arr,		/* template vector */
    SolRef*		firstp,		/* -> first in list */
    int*		elemsp)		/* -> #(nodes in array) */
{
    register rSolRef	ix;
    register rSolRef	frst;
    register rSolRef	tree;

    if( *elemsp >= 2 ) {
	frst = *firstp;
	tree = arr[frst].sn_down;
	if( tree == XNIL ) {	/* empty does not combine */
	    return;
	}
	scinc(sc_any_try);
	for( ix = arr[frst].sn_next ; ix != XNIL ; ix=arr[ix].sn_next ) {
	    if( arr[ix].sn_down != tree ) {
		break;		/* this one is different */
	    }
	}
	if( ix == XNIL ) {		/* all alike */
	    scinc(sc_any_succ);
	    rel_templ_list(arr, arr[frst].sn_next);	/* all but first */
	    if( frst ) {
		arr[0] = arr[frst];
		frst = 0;
		*firstp = frst;
	    }
	    arr[frst].sn_next = XNIL;
	    stuff_move(arr[frst].sn_move, sym_any);
	    *elemsp = 1;
	}
    }
}


    static void
try_else( 
    register SolNode*	arr,		/* template vector */
    SolRef*		firstp,		/* -> first in list */
    int*		elemsp)		/* -> nodes in array */
{
    register rSolRef	ix;
    register int	i;
    register rSolRef	tree;
    struct {
	SolRef	tree;
	uint16	count;
    }			cnts[MAX_MOVES];
    register int	full;
    register unsigned	bestcnt;
    register rSolRef	best;
    register rSolRef	bestix;
    register SolRef*	lastp;

    if( *elemsp >= 3 ) {
	scinc(sc_else_try);
	full = 0;
	bestcnt = 1;
	for( ix = *firstp ; ix != XNIL ; ix=arr[ix].sn_next ) {
	    tree = arr[ix].sn_down;	/* to be searched */
	    if( tree == XNIL ) {
		continue;		/* empty is not combined */
	    }
	    for( i=0 ; i<full ; ++i ) {
		if( cnts[i].tree == tree ) {
		    break;
		}
	    }
	    if( i < full ) {
		if( (cnts[i].count += 1) > bestcnt ) {
		    bestcnt = cnts[i].count;
		}
	    }else {
		cnts[full].tree = tree;
		cnts[full].count = 1;
		++full;
	    }
	}
	if( bestcnt > 1 ) {		/* something to be saved */
	    scinc(sc_else_succ);
	    best = XNIL;
	    for( i=0 ; i<full ; ++i ) {
		if( cnts[i].count == bestcnt ) {
		    if( (best == XNIL)
		     || (NODE(cnts[i].tree).sn_level > NODE(best).sn_level)
		     ) {
			best = cnts[i].tree;
		    }
		}
	    }
	    /*
	     * Now, we have to eliminate all with the best subtree,
	     * except for one, which we want to become the last.
	     */
	    bestcnt = 0;		/* misuse it */
	    lastp = firstp;
	    bestix = *lastp;		/* shut off gcc-warning */
	    while( (ix = *lastp) != XNIL ) {
		tree = arr[ix].sn_down;
		if( tree == best ) {
		    if( bestcnt++ == 0 ) {	/* first */
			bestix = ix;		/* remember */
			stuff_move(arr[ix].sn_move, sym_else);
		    }else {
			rel_solref(tree);
			--*elemsp;
		    }
		    *lastp = arr[ix].sn_next;	/* skip over this */
		}else {
		    lastp = &(arr[ix].sn_next);
		}
	    }
	    *lastp = bestix;
	    arr[bestix].sn_next = XNIL;		/* now reappended at end */
	}
    }
}


    static SolRef
find_or_make( 
    register SolNode*	templ,		/* such contents are wanted */
    register rSolRef	next)		/* but with this (dangling) sn_next */
{
    register unsigned	hash;
    register SolNode*	p;
    register rSolRef	ix;
    register unsigned	c;

    scinc(sc_node_search);
    hash  = 0;
    hash ^= next ^ templ->sn_down ^ templ->sn_level;
    for( ix=0 ; ix<MV_CHARS ; ++ix ) {
	if( 0 == (c = templ->sn_move[ix]) ) {
	    break;
	}
	hash <<= 1;
	hash ^= c;
    }
    hash ^= (hash >> 8);
    hash  = (uint8) hash;
    for( ix=solhclass[hash] ; ix != XNIL ; ix=p->sn_hforw ) {
	p = &NODE(ix);
	if( p->sn_refcnt ) {
	    if( (p->sn_next  == next)
	     && (p->sn_down  == templ->sn_down)
	     && (p->sn_level == templ->sn_level)
	     && (0 == cmp_mov(p->sn_move, templ->sn_move))
	      ) {
		p->sn_refcnt += 1;
		rel_solref(next);	/* not needed */
		scinc(sc_node_found);
		return ix;
	    }
	}
    }
			/* not found, try to create: */
    if( (ix = sol_free) != XNIL ) {
	sol_free = NODE(ix).sn_next;
    }else {
	if( (ix = sol_allo) >= SOL_NODES ) {
	    return XNIL;		/* cannot create */
	}
	sol_allo += 1;
    }
    p = &NODE(ix);
    *p = *templ;
    p->sn_next   = next;		/* the dangling reference is now used */
    p->sn_refcnt = 1;
    if( templ->sn_down != XNIL ) {	/* copied that reference */
	NODE(templ->sn_down).sn_refcnt += 1;
    }
					/* insert into hash class ... */
    p->sn_hclass = (uint8) hash;
    p->sn_hback = XNIL;
    if( XNIL != (p->sn_hforw = solhclass[hash]) ) {
	NODE(p->sn_hforw).sn_hback = ix;
    }
    solhclass[hash] = ix;
    return ix;		/* made a new one */
}


    static Bool
make_conti( int soldep, int level, SolRef* resp )
{
    SolNode	templ;

    stuff_move(templ.sn_move, fill_conti(soldep));
    templ.sn_level = level;
    templ.sn_next = XNIL;
    templ.sn_down = XNIL;
    return XNIL != (*resp = find_or_make(&templ, XNIL));
}


    static Bool			/* success */
templs_to_list( 
    SolNode*	arr,		/* this template vector */
    SolRef	first,		/* starting with this first internal index */
    SolRef*	resp)		/* shall be converted into this global */
{
    if( first == XNIL ) {
	*resp = XNIL;
    }else {
	    SolRef	rest;	/* holds dangling reference */
	if( ! templs_to_list(arr, arr[first].sn_next, &rest) ) {
	    return FALSE;	/* sorry */
	}
	*resp = find_or_make(arr+first, rest);
	if( *resp == XNIL ) {	/* could not use 'rest' */
	    rel_solref(rest);	/* so release it */
	    return FALSE;	/* sorry */
	}
    }
    return TRUE;		/* lucky we */
}


static Bool	try_mksol(Board*, int, Movelist*, int, SolRef*);

    static Bool			/* success */
try_mkdef(		/* build defenders part of solution tree */
    Board*	bp,		/* current board */
    int		subsoldep,	/* what sub-analyse is to be asked */
    Movelist*	defs,		/* those are to be done (probably all legal) */
    int		sublimdep,	/* passed to "try_mksol": sub-analysis */
    SolRef*	resp)		/* OUT: stuff index here */
{
    register Bool	ok;
    register Move*	dp;
    SolNode		dtempl[MAX_MOVES];
    int			dfull;
    SolRef		dfirst;
    Movelist		subsols;

    ok = TRUE;			/* so far */
    ml_ck_attr(bp, defs);
    dfirst = XNIL; dfull = 0;
    formoves( defs, dp ) {
	if( dfull > 0 ) {
	    dtempl[dfull-1].sn_next = dfull;
	}else {
	    dfirst = dfull;
	}
	stuff_move(dtempl[dfull].sn_move, mvc_L6(bp, dp));
	dtempl[dfull].sn_down = XNIL;
	dtempl[dfull].sn_next = XNIL;
	dtempl[dfull].sn_level = (2*subsoldep) + 1;
	if( subsoldep > 0 ) {	/* follow defender move */
		register int	res;
	    move_execute(bp, dp);
	    res = sol_analyse(bp, subsoldep, &subsols);
	    ok = try_mksol(bp, ANADEP(res), &subsols, sublimdep,
						&(dtempl[dfull].sn_down));
	    dtempl[dfull].sn_level = 2*ANADEP(res) + 1;
	    move_undo(bp);
	} /* follow defender move */
	dfull += 1;
	if( ! ok ) break;
    } /* for defs */
    dfirst = sort_templ_arr(dtempl, dfull);		/* builds linkage */
    if( ok ) {
	try_any(dtempl, &dfirst, &dfull);
	try_else(dtempl, &dfirst, &dfull);
	ok = templs_to_list(dtempl, dfirst, resp);
    }
    /*
     * Anyhow, release defender templates ...
     */
    rel_templ_list(dtempl, dfirst);
    return ok;
}

    static Bool			/* success */
try_mksol(		/* build solution tree */
    Board*	bp,		/* current board */
    int		soldep,		/* what analyse told us */
    Movelist*	atts,		/* those belong to the solution */
    int		limdep,		/* <= 1: replace defender by ... */
    SolRef*	resp)		/* OUT: stuff index here */
{
    register Bool	ok;
    register Move*	ap;
    SolNode		atempl[MAX_MOVES];
    int			afull;
    SolRef		afirst;
    Movelist		defends;

    ok = TRUE;			/* so far */
    scinc(sc_tree_try);
    ml_ck_attr(bp, atts);
    afirst = XNIL; afull = 0;
    formoves( atts, ap ) {
	if( afull > 0 ) {
	    atempl[afull-1].sn_next = afull;
	}else {
	    afirst = afull;
	}
	stuff_move(atempl[afull].sn_move, mvc_L6(bp, ap));
	atempl[afull].sn_down = XNIL;
	atempl[afull].sn_next = XNIL;
	atempl[afull].sn_level = (2*soldep);
	if( (soldep > 1) || (f_jobattr & JTA_2plies) ) {
	    move_execute(bp, ap);
	    if( move_gen(bp, &defends) ) {	/* there are moves */
		if( limdep <= 1 ) {
		    ok = make_conti(soldep, 2*soldep-1, &atempl[afull].sn_down);
		}else {		/* defenders are to be done */
		    ok = try_mkdef(bp, soldep-1, &defends, limdep-1,
						&(atempl[afull].sn_down));
		} /* defenders to be done */
	    } /* there are defends */
	    move_undo(bp);
	} /* follow the attacker move */
	afull += 1;
	if( ! ok ) break;
    } /* for attackers */
    afirst = sort_templ_arr(atempl, afull);		/* builds linkage */
    if( ok ) {
	ok = templs_to_list(atempl, afirst, resp);
    }
    /*
     * Anyhow, release the attacker templates, they hold ref-counts.
     */
    rel_templ_list(atempl, afirst);
    scadd(sc_tree_succ, ok);
    return ok;
}

/*---------------------------------------------------------------------------*/

static int	col__curr	= 0;
static int	col__want	= 0;

#define col_inc_want(i)	(col__want += (i))


    static void
col_sync(void)
{
    if( col__want != col__curr ) {
	if( col__want < col__curr ) {
	    printf("\n"); col__curr = 0;
	}
#if SOL_USE_TABS		/* use tab character at begin of line */
	if( col__curr == 0 ) {
	    while( (col__curr & ~7) < (col__want & ~7) ) {
		printf("\t"); col__curr += 8;
	    }
	}
#endif
	if( col__curr < col__want ) {
	    printf("%*c", col__want - col__curr, ' ');
	    col__curr = col__want;
	}
    }
}

    static void
pr_chr( char c )
{
    if( c ) {
	if( c == '\n' ) {
	    col__curr = 0;
	}else {
	    col__curr += 1;
	}
	printf("%c", c);
    }
}


    static void
pr_mov( register const char* mov )
{
    register int	i;

    for( i=0 ; (i<MV_CHARS) && mov[i] ; ++i ) {
	pr_chr(mov[i]);
    }
}


/*
 * mk_length_vec()
 *	Maximize the length vector for the specified tree.
 */
    static void
mk_length_vec( 
    SolRef		root,		/* this tree */
    register int*	lvec)		/* maximizes these lengthes */
{
    register int	i;
    register SolNode*	p;
    register rSolRef	curr;

    for( curr=root ; curr!=XNIL ; curr=p->sn_next ) {
	p = &NODE(curr);
	mk_length_vec(p->sn_down, lvec+1);
	for( i=0 ; i<MV_CHARS ; ++i ) {
	    if( ! p->sn_move[i] ) {
		break;
	    }
	}
	while( (i > 0) && (p->sn_move[i-1] == ' ') ) {
	    --i;
	    p->sn_move[i] = '\0';
	}
	if( *lvec < i ) {
	    *lvec = i;
	}
    }
}


    static void
pr_sol(
    SolRef	root,		/* print this tree */
    const int*	lvec)		/* with these column widthes */
{
    register SolNode*	p;
    register rSolRef	curr;
    register int	i;

    i = 1 + *lvec;
    for( curr=root ; curr!=XNIL ; curr=p->sn_next ) {
	p = &NODE(curr);
	col_sync();
	pr_chr(' '); pr_mov(p->sn_move);
	if( p->sn_down != XNIL ) {
	    col_inc_want( i);
	    pr_sol(p->sn_down, lvec+1);
	    col_inc_want(-i);
	}
    }
}


    static void
pr_solopt(
    Board*	bp,
    int		depth,
    int		preply,
    Movelist*	solp,
    int		limdep)
{
    {
	    SolRef	root = XNIL;
	    Bool	ok;
	if( preply ) {
	    ok = try_mkdef(bp, depth, solp, limdep, &root);
	}else {
	    ok = try_mksol(bp, depth, solp, limdep, &root);
	}
	if( ok ) {				/* Hurra */
#define MX_lvec	(2*ANA_MANY + 2)
		int		lvec[MX_lvec];
		register int	i;
	    for( i=0 ; i<MX_lvec ; ++i ) {
		lvec[i] = 0;
	    }
	    sol_dump_tree(root, 0);		/* debugging, only */
	    mk_length_vec(root, lvec);
	    pr_sol(root, lvec);
	    rel_solref(root);
	    return;
	}
	rel_solref(root);
    }
    /*
     * Huh, didn't work.  Well, do the first level standard way (print
     * immediately) and then call recursively.
     */
    {
	    register Move*	dp;
	    register Move*	ap;
	    Movelist		defends;
	    Movelist		subsols;
	    int			res;
	ml_ck_attr(bp, solp);
	formoves( solp, ap ) {
	    col_sync(); pr_chr(' '); pr_mov(mvc_L6(bp, ap));
	    col_inc_want(1+MV_CHARS);
	    if( preply ) {	/* really was a def-move: call analyse, now */
		move_execute(bp, ap);
		res = sol_analyse(bp, depth, &subsols);
		pr_solopt(bp, ANADEP(res), 0/*preply*/, &subsols, limdep);
		move_undo(bp);
	    }else if( (depth > 1) || (f_jobattr & JTA_2plies) ) {
		move_execute(bp, ap);
		if( move_gen(bp, &defends) ) {	/* there are moves */
		    if( limdep <= 1 ) {
			col_sync(); pr_chr(' '); pr_mov(fill_conti(depth));
		    }else {		/* really do the defenders */
			ml_ck_attr(bp, &defends);
			formoves( &defends, dp ) {
			    col_sync(); pr_chr(' '); pr_mov(mvc_L6(bp, dp));
			    col_inc_want(1+MV_CHARS);
			    if( depth > 1 ) {	/* follow the defender */
				move_execute(bp, dp);
				res = sol_analyse(bp, depth-1, &subsols);
				pr_solopt(bp, ANADEP(res), 0, &subsols,
								    limdep-1);
				move_undo(bp);
			    } /* follow the defender */
			    col_inc_want(-(1+MV_CHARS));
			}
		    } /* really do the defenders */
		} /* there are defends */
		move_undo(bp);
	    }
	    col_inc_want(-(1+MV_CHARS));
	}
    }
}

/*---------------------------------------------------------------------------*/

    Eximpl void		/*ARGSUSED*/
solopt_stats( Bool doprint )
{
#if SOLOPT_STATS
    if( doprint && f_stats && sc_tree_try ) {
	printf("SolOpt :");
	show_scs(1,0, sc_tree_try,  " tries,");
	show_scs(1,0, sc_tree_succ, " succ");
	printf(" (%.1f%%)\n", percent(sc_tree_succ, sc_tree_try));
	if( sc_node_search ) {
	    printf("SOnode :");
	    show_scs(1,0, sc_node_search, " searched,");
	    show_scs(1,0, sc_node_found,  " found");
	    printf(" (%.1f%%),", percent(sc_node_found, sc_node_search));
	    show_scs(1,0, (Counter)sol_allo, " needed,");
	    show_scs(1,0, (Counter)SOL_NODES, " max");
	    printf("\n");
	}
	if( sc_any_try || sc_else_try ) {
	    printf("SOcombi:");
	    show_scs(1,0, sc_any_try,  " any,");
	    show_scs(1,0, sc_any_succ, " succ");
	    printf(" (%.1f%%)", percent(sc_any_succ, sc_any_try));
	    show_scs(1,0, sc_else_try,  " else,");
	    show_scs(1,0, sc_else_succ, " succ");
	    printf(" (%.1f%%)\n", percent(sc_else_succ, sc_else_try));
	}
	if( sc_sort_list ) {
	    printf("SOsort :");
	    show_scs(1,0, sc_sort_list, " lists,");
	    show_scs(1,0, sc_cmp_templ, " cmpT,");
	    show_scs(1,0, sc_cmp_move,  " cmpMv");
	    printf("\n");
	}
    }
    sc_tree_try    = 0;
    sc_tree_succ   = 0;
    sc_node_search = 0;
    sc_node_found  = 0;
    sc_any_try     = 0;
    sc_any_succ    = 0;
    sc_else_try    = 0;
    sc_else_succ   = 0;
    sc_cmp_move    = 0;
    sc_cmp_templ   = 0;
    sc_sort_list   = 0;
#endif	/* SOLOPT_STATS */
}


/*
 * solopt_print()
 *	The given list of moves is the top level of the solution
 *	for the specified board and the specified depth.
 *	Print the complete solution tree.
 */
    Eximpl void
solopt_print(
    Board*	bp,
    int		depth,
    int		preply,
    Movelist*	solp,
    int		limdepth)
{
    if( depth < 1 ) {
	depth = 1;			/* paranoia FFS:preply */
    }
    col__want = 0;
    col__curr = 0;
    pr_solopt(bp, depth, preply, solp, limdepth);
    col_sync();
}


/*
 * init_solopt()
 *	Initialize this module.
 */
    Eximpl void
init_solopt(void)
{
    register int	i;

    sol_free = XNIL;
    sol_allo = 0;
    for( i=0 ; i<SOL_HCLASSES ; ++i ) {
	solhclass[i] = XNIL;		/* empty hash class (anchor) */
    }
    col__want = 0;
    col__curr = 0;
}
