package mempool_test

import (
	"testing"
	"unsafe"

	"codeberg.org/gruf/go-mempool"
)

func TestPool(t *testing.T) {
	var p mempool.Pool[[]byte]

	if p.Get() != nil {
		t.Fatal("empty pool not empty!")
	}

	s := []byte("hello world")
	p.Put(s)

	if string(p.Get()) != string(s) {
		t.Fatal("failed to get pooled string")
	}

	s2 := []byte("another world")
	p.Put(s2)

	if string(p.Get()) != string(s2) {
		t.Fatal("failed to get second pooled string")
	}

	p.New = func() []byte {
		return []byte("new!")
	}

	p.Reset = func(b []byte) {
		clear(b)
	}

	s3 := p.Get()

	if string(s3) != "new!" {
		t.Fatal("newly allocated string not as expected")
	}

	p.Put(s3)

	if string(p.Get()) != "\x00\x00\x00\x00" {
		t.Fatal("reset function not called on bytes")
	}
}

func TestUnsafePool(t *testing.T) {
	var p mempool.UnsafePool

	if p.Get() != nil {
		t.Fatal("empty pool not empty!")
	}

	s := "hello world"
	ptr := unsafe.Pointer(&s)
	p.Put(ptr)

	if p.Get() != ptr {
		t.Fatal("failed to get pooled string")
	}

	s2 := "another world"
	ptr2 := unsafe.Pointer(&s2)

	// Ensure only 1
	// put causes GC.
	p.DirtyFactor = 1
	p.Put(ptr)
	p.Put(ptr2)

	if p.Get() != ptr2 {
		t.Fatal("failed to get second pooled string")
	}
}
