import './style.scss';

import { useEffect } from 'react';
import QRCode from 'react-qr-code';
import { shallow } from 'zustand/shallow';

import { useI18nContext } from '../../../../i18n/i18n-react';
import { OpenDesktopClientButton } from '../../../../shared/components/Layout/buttons/OpenDesktopClientButton/OpenDesktopClientButton';
import { RenderMarkdown } from '../../../../shared/components/Layout/RenderMarkdown/RenderMarkdown';
import { Button } from '../../../../shared/defguard-ui/components/Layout/Button/Button';
import {
  ButtonSize,
  ButtonStyleVariant,
} from '../../../../shared/defguard-ui/components/Layout/Button/types';
import { Card } from '../../../../shared/defguard-ui/components/Layout/Card/Card';
import { CopyField } from '../../../../shared/defguard-ui/components/Layout/CopyField/CopyField';
import { MessageBox } from '../../../../shared/defguard-ui/components/Layout/MessageBox/MessageBox';
import { isPresent } from '../../../../shared/defguard-ui/utils/isPresent';
import { useClipboard } from '../../../../shared/hooks/useClipboard';
import { externalLink } from '../../../../shared/links';
import { useAddDevicePageStore } from '../../hooks/useAddDevicePageStore';
import { AddDeviceStep } from '../../types';
import { enrollmentToImportToken } from '../../utils/enrollmentToToken';

export const AddDeviceClientConfigurationStep = () => {
  const { LL } = useI18nContext();
  const localLL = LL.addDevicePage.steps.client;
  const clientData = useAddDevicePageStore((s) => s.clientSetup);
  const clientSetup = useAddDevicePageStore((s) => s.clientSetup);
  const tokenValue = useAddDevicePageStore((s) =>
    s.clientSetup
      ? enrollmentToImportToken(s.clientSetup.url, s.clientSetup.token)
      : null,
  );
  const setStep = useAddDevicePageStore((s) => s.setStep, shallow);
  const { writeToClipboard } = useClipboard();

  useEffect(() => {
    if (!isPresent(tokenValue)) {
      setStep(AddDeviceStep.CHOOSE_METHOD);
    }
  }, [setStep, tokenValue]);

  if (!isPresent(tokenValue) || !isPresent(clientData)) return null;

  return (
    <Card id="add-device-client-configuration" shaded>
      <h2>{localLL.title()}</h2>
      {isPresent(clientSetup) && (
        <>
          <MessageBox message={localLL.desktopDeepLinkHelp()} />
          <div className="row desktop-button">
            <OpenDesktopClientButton token={clientSetup.token} url={clientSetup.url} />
          </div>
        </>
      )}
      <MessageBox>
        <RenderMarkdown content={localLL.message()} />
      </MessageBox>
      {/* <CopyField
        label={localLL.labels.mergedToken()}
        value={tokenValue}
        onCopy={(value) => {
          void writeToClipboard(value, localLL.tokenCopy());
        }}
      /> */}
      <CopyField
        label={localLL.labels.url()}
        value={clientData.url}
        onCopy={(value) => {
          void writeToClipboard(value, localLL.tokenCopy());
        }}
      />
      <CopyField
        label={localLL.labels.token()}
        value={clientData.token}
        onCopy={(value) => {
          void writeToClipboard(value, localLL.tokenCopy());
        }}
      />
      <MessageBox message={localLL.qrHelp()} />
      <div className="qr">
        <QRCode value={tokenValue} />
      </div>
      <div className="row">
        <p className="qr-description">{localLL.qrDescription()}</p>
      </div>
      <div className="row">
        <a
          href={externalLink.clientApp.download.android}
          rel="noreferrer noopener"
          target="_blank"
        >
          <svg
            xmlns="http://www.w3.org/2000/svg"
            width="111"
            height="32"
            viewBox="0 0 111 32"
            fill="none"
          >
            <path
              d="M106.782 32H4.2178C1.96241 32 0.115234 30.199 0.115234 28V4C0.115234 1.801 1.96241 -4.76837e-07 4.2178 -4.76837e-07H106.782C109.037 -4.76837e-07 110.884 1.801 110.884 4V28C110.884 30.199 109.037 32 106.782 32Z"
              fill="white"
            />
            <path
              d="M106.782 0.640997C108.681 0.640997 110.227 2.148 110.227 4V28C110.227 29.852 108.681 31.359 106.782 31.359H4.2178C2.31831 31.359 0.77267 29.852 0.77267 28V4C0.77267 2.148 2.31831 0.640997 4.2178 0.640997H106.782ZM106.782 -4.76837e-07H4.2178C1.96241 -4.76837e-07 0.115234 1.801 0.115234 4V28C0.115234 30.199 1.96241 32 4.2178 32H106.782C109.037 32 110.884 30.199 110.884 28V4C110.884 1.801 109.037 -4.76837e-07 106.782 -4.76837e-07Z"
              fill="#E8E8E8"
            />
            <path
              d="M8.06215 6.03098C7.82113 6.27698 7.68164 6.65998 7.68164 7.15598V24.848C7.68164 25.344 7.82113 25.727 8.06215 25.973L8.12164 26.027L18.2898 16.117V15.883L8.12164 5.97298L8.06215 6.03098Z"
              fill="black"
            />
            <path
              d="M22.9061 19.4219L19.5205 16.1169V15.8829L22.9103 12.5779L22.9861 12.6209L27.0005 14.8479C28.1461 15.4799 28.1461 16.5199 27.0005 17.1559L22.9861 19.3789L22.9061 19.4219Z"
              fill="black"
            />
            <path
              d="M22.3711 19.9943L18.9054 16.6153L8.67773 26.5883C9.05825 26.9783 9.67876 27.0253 10.3844 26.6353L22.3711 19.9943Z"
              fill="black"
            />
            <path
              d="M22.3711 12.0056L10.3844 5.36465C9.67876 4.97865 9.05825 5.02564 8.67773 5.41564L18.9054 15.3846L22.3711 12.0056Z"
              fill="black"
            />
            <path
              d="M39.0218 8.19482C39.0218 8.86282 38.8167 9.39782 38.4126 9.79682C37.948 10.2698 37.3428 10.5078 36.6013 10.5078C35.8926 10.5078 35.2875 10.2658 34.791 9.78882C34.2936 9.30482 34.0454 8.71082 34.0454 7.99982C34.0454 7.28882 34.2936 6.69482 34.791 6.21482C35.2875 5.73382 35.8926 5.49182 36.6013 5.49182C36.9541 5.49182 37.2905 5.56282 37.6116 5.69482C37.9316 5.82782 38.1921 6.00782 38.3808 6.22982L37.9521 6.65182C37.6228 6.27282 37.1746 6.08582 36.6013 6.08582C36.0844 6.08582 35.6362 6.26182 35.2557 6.61682C34.8793 6.97282 34.6905 7.43382 34.6905 7.99982C34.6905 8.56582 34.8793 9.03082 35.2557 9.38682C35.6362 9.73782 36.0844 9.91782 36.6013 9.91782C37.15 9.91782 37.6116 9.73782 37.9757 9.38282C38.2157 9.14782 38.3521 8.82382 38.388 8.40982H36.6013V7.83182H38.9849C39.0136 7.95682 39.0218 8.07782 39.0218 8.19482Z"
              fill="black"
              stroke="black"
              strokeWidth="0.0627687"
              strokeMiterlimit="10"
            />
            <path
              d="M42.8035 6.19081H40.5645V7.71081H42.583V8.28881H40.5645V9.80881H42.8035V10.3978H39.9307V5.60181H42.8035V6.19081Z"
              fill="black"
              stroke="black"
              strokeWidth="0.0627687"
              strokeMiterlimit="10"
            />
            <path
              d="M45.472 10.3978H44.8382V6.19081H43.4648V5.60181H46.8464V6.19081H45.472V10.3978Z"
              fill="black"
              stroke="black"
              strokeWidth="0.0627687"
              strokeMiterlimit="10"
            />
            <path
              d="M49.2939 10.3978V5.60181H49.9268V10.3978H49.2939Z"
              fill="black"
              stroke="black"
              strokeWidth="0.0627687"
              strokeMiterlimit="10"
            />
            <path
              d="M52.7318 10.3978H52.1031V6.19081H50.7246V5.60181H54.1102V6.19081H52.7318V10.3978Z"
              fill="black"
              stroke="black"
              strokeWidth="0.0627687"
              strokeMiterlimit="10"
            />
            <path
              d="M60.5121 9.78082C60.027 10.2658 59.426 10.5078 58.7091 10.5078C57.988 10.5078 57.387 10.2658 56.9019 9.78082C56.4178 9.29682 56.1768 8.70282 56.1768 7.99982C56.1768 7.29682 56.4178 6.70282 56.9019 6.21882C57.387 5.73382 57.988 5.49182 58.7091 5.49182C59.4219 5.49182 60.0229 5.73382 60.508 6.22282C60.9963 6.71082 61.2373 7.30082 61.2373 7.99982C61.2373 8.70282 60.9962 9.29682 60.5121 9.78082ZM57.3706 9.37882C57.7357 9.73782 58.1798 9.91782 58.7091 9.91782C59.2342 9.91782 59.6824 9.73782 60.0434 9.37882C60.4075 9.01982 60.5921 8.55882 60.5921 7.99982C60.5921 7.44082 60.4075 6.97982 60.0434 6.62082C59.6824 6.26182 59.2342 6.08182 58.7091 6.08182C58.1798 6.08182 57.7357 6.26182 57.3706 6.62082C57.0065 6.97982 56.8219 7.44082 56.8219 7.99982C56.8219 8.55882 57.0065 9.01982 57.3706 9.37882Z"
              fill="black"
              stroke="black"
              strokeWidth="0.0627687"
              strokeMiterlimit="10"
            />
            <path
              d="M62.1265 10.3978V5.60181H62.8957L65.2875 9.33181H65.3152L65.2875 8.40981V5.60181H65.9203V10.3978H65.2598L62.7552 6.48381H62.7275L62.7552 7.40981V10.3978H62.1265Z"
              fill="black"
              stroke="black"
              strokeWidth="0.0627687"
              strokeMiterlimit="10"
            />
            <path
              d="M56.0208 17.4021C54.0937 17.4021 52.5193 18.8321 52.5193 20.8051C52.5193 22.7621 54.0937 24.2071 56.0208 24.2071C57.9521 24.2071 59.5265 22.7621 59.5265 20.8051C59.5265 18.8321 57.9521 17.4021 56.0208 17.4021ZM56.0208 22.8671C54.9634 22.8671 54.0537 22.0161 54.0537 20.8051C54.0537 19.5781 54.9634 18.7421 56.0208 18.7421C57.0783 18.7421 57.9921 19.5781 57.9921 20.8051C57.9921 22.0161 57.0783 22.8671 56.0208 22.8671ZM48.3809 17.4021C46.4496 17.4021 44.8793 18.8321 44.8793 20.8051C44.8793 22.7621 46.4496 24.2071 48.3809 24.2071C50.3111 24.2071 51.8824 22.7621 51.8824 20.8051C51.8824 18.8321 50.3111 17.4021 48.3809 17.4021ZM48.3809 22.8671C47.3224 22.8671 46.4096 22.0161 46.4096 20.8051C46.4096 19.5781 47.3224 18.7421 48.3809 18.7421C49.4383 18.7421 50.348 19.5781 50.348 20.8051C50.348 22.0161 49.4383 22.8671 48.3809 22.8671ZM39.2896 18.4451V19.8911H42.8311C42.7275 20.6991 42.4506 21.2931 42.026 21.7071C41.5091 22.2071 40.7039 22.7621 39.2896 22.7621C37.1101 22.7621 35.4034 21.0471 35.4034 18.9221C35.4034 16.7971 37.1101 15.0821 39.2896 15.0821C40.468 15.0821 41.3255 15.5311 41.9583 16.1131L43.0034 15.0941C42.1183 14.2701 40.9409 13.6371 39.2896 13.6371C36.3009 13.6371 33.7891 16.0081 33.7891 18.9221C33.7891 21.8361 36.3009 24.2071 39.2896 24.2071C40.905 24.2071 42.1183 23.6911 43.0721 22.7231C44.0496 21.7701 44.3542 20.4301 44.3542 19.3481C44.3542 19.0121 44.3255 18.7031 44.2742 18.4451H39.2896ZM76.4649 19.5661C76.1767 18.8051 75.2875 17.4021 73.4762 17.4021C71.6814 17.4021 70.187 18.7811 70.187 20.8051C70.187 22.7111 71.666 24.2071 73.6485 24.2071C75.2516 24.2071 76.1767 23.2541 76.5573 22.6991L75.3675 21.9261C74.9706 22.4921 74.4301 22.8671 73.6485 22.8671C72.8721 22.8671 72.3152 22.5201 71.9583 21.8361L76.626 19.9531L76.4649 19.5661ZM71.706 20.6991C71.666 19.3871 72.7511 18.7151 73.5285 18.7151C74.1378 18.7151 74.6547 19.0121 74.827 19.4371L71.706 20.6991ZM67.9121 24.0001H69.4465V14.0001H67.9121V24.0001ZM65.3993 18.1601H65.348C65.0034 17.7621 64.346 17.4021 63.5132 17.4021C61.7655 17.4021 60.1675 18.8981 60.1675 20.8161C60.1675 22.7231 61.7655 24.2071 63.5132 24.2071C64.346 24.2071 65.0034 23.8441 65.348 23.4341H65.3993V23.9221C65.3993 25.2231 64.6865 25.9221 63.5367 25.9221C62.5993 25.9221 62.0178 25.2621 61.7778 24.7071L60.4434 25.2501C60.828 26.1521 61.8465 27.2621 63.5367 27.2621C65.3357 27.2621 66.8537 26.2301 66.8537 23.7191V17.6091H65.3993V18.1601ZM63.6444 22.8671C62.587 22.8671 61.7019 22.0041 61.7019 20.8161C61.7019 19.6171 62.587 18.7421 63.6444 18.7421C64.6865 18.7421 65.508 19.6171 65.508 20.8161C65.508 22.0041 64.6865 22.8671 63.6444 22.8671ZM83.6485 14.0001H79.9788V24.0001H81.5091V20.2111H83.6485C85.348 20.2111 87.0147 19.0121 87.0147 17.1051C87.0147 15.1991 85.3439 14.0001 83.6485 14.0001ZM83.6885 18.8201H81.5091V15.3911H83.6885C84.8311 15.3911 85.4834 16.3161 85.4834 17.1051C85.4834 17.8791 84.8311 18.8201 83.6885 18.8201ZM93.148 17.3831C92.0424 17.3831 90.8926 17.8591 90.4198 18.9141L91.7778 19.4691C92.0701 18.9141 92.6075 18.7341 93.1757 18.7341C93.9696 18.7341 94.7747 19.1991 94.787 20.0201V20.1251C94.5101 19.9691 93.9173 19.7381 93.188 19.7381C91.7255 19.7381 90.2352 20.5231 90.2352 21.9881C90.2352 23.3281 91.4331 24.1911 92.7798 24.1911C93.8096 24.1911 94.3778 23.7381 94.7347 23.2111H94.787V23.9841H96.2649V20.1481C96.2649 18.3751 94.907 17.3831 93.148 17.3831ZM92.9634 22.8631C92.4629 22.8631 91.7655 22.6211 91.7655 22.0161C91.7655 21.2421 92.6352 20.9451 93.388 20.9451C94.0619 20.9451 94.3778 21.0901 94.787 21.2811C94.667 22.2071 93.8496 22.8631 92.9634 22.8631ZM101.65 17.6021L99.8906 21.9371H99.8383L98.0198 17.6021H96.3696L99.1019 23.6601L97.5429 27.0311H99.1419L103.352 17.6021H101.65ZM87.8557 24.0001H89.3901V14.0001H87.8557V24.0001Z"
              fill="black"
            />
          </svg>
        </a>
        <a
          href={externalLink.clientApp.download.ios}
          rel="noreferrer noopener"
          target="_blank"
        >
          <svg
            xmlns="http://www.w3.org/2000/svg"
            width="112"
            height="32"
            viewBox="0 0 112 32"
            fill="none"
          >
            <path
              d="M111.268 28.3096C111.268 30.0149 109.855 31.3962 108.107 31.3962H4.66646C2.91949 31.3962 1.50146 30.0149 1.50146 28.3096V3.69445C1.50146 1.98994 2.91949 0.60376 4.66646 0.60376H108.106C109.855 0.60376 111.267 1.98994 111.267 3.69445L111.268 28.3096Z"
              fill="white"
            />
            <path
              d="M107.774 0.641001C109.677 0.641001 111.226 2.148 111.226 4V28C111.226 29.852 109.677 31.359 107.774 31.359H4.99588C3.09243 31.359 1.54357 29.852 1.54357 28V4C1.54357 2.148 3.09243 0.641001 4.99588 0.641001H107.774ZM107.774 1.90735e-06H4.99588C2.73579 1.90735e-06 0.884766 1.801 0.884766 4V28C0.884766 30.199 2.73579 32 4.99588 32H107.774C110.034 32 111.885 30.199 111.885 28V4C111.885 1.801 110.034 1.90735e-06 107.774 1.90735e-06Z"
              fill="#E8E8E8"
            />
            <path
              d="M25.6565 15.8272C25.6326 13.2488 27.8263 11.9944 27.9266 11.936C26.6842 10.1736 24.7586 9.93283 24.0819 9.91363C22.4646 9.74803 20.8958 10.8552 20.0719 10.8552C19.2316 10.8552 17.9629 9.92963 16.5956 9.95683C14.836 9.98323 13.1899 10.9744 12.2871 12.5136C10.424 15.652 11.8135 20.264 13.5986 22.8008C14.4915 24.0432 15.5349 25.4304 16.9006 25.3816C18.2367 25.328 18.7358 24.5528 20.3482 24.5528C21.9458 24.5528 22.4144 25.3816 23.8073 25.3504C25.2412 25.328 26.144 24.1024 27.0057 22.8488C28.0376 21.4248 28.452 20.0224 28.4685 19.9504C28.4348 19.9392 25.6836 18.9176 25.6565 15.8272Z"
              fill="black"
            />
            <path
              d="M23.0254 8.24483C23.744 7.37043 24.2357 6.18083 24.0992 4.97363C23.0591 5.01843 21.7583 5.67363 21.0093 6.52883C20.3466 7.28243 19.7546 8.51763 19.9075 9.67923C21.0759 9.76403 22.2755 9.10563 23.0254 8.24483Z"
              fill="black"
            />
            <path
              d="M41.215 8.00721C41.215 8.94881 40.9248 9.65761 40.3451 10.1336C39.8082 10.5728 39.0452 10.7928 38.0569 10.7928C37.5668 10.7928 37.1475 10.772 36.7964 10.7304V5.58561C37.2544 5.51361 37.7477 5.47681 38.2805 5.47681C39.2219 5.47681 39.9315 5.67601 40.4101 6.07441C40.9461 6.52481 41.215 7.16881 41.215 8.00721ZM40.3065 8.03041C40.3065 7.42001 40.1404 6.95201 39.8082 6.62561C39.476 6.30001 38.9909 6.13681 38.352 6.13681C38.0807 6.13681 37.8497 6.15441 37.6581 6.19121V10.1024C37.7641 10.1184 37.9582 10.1256 38.2402 10.1256C38.8996 10.1256 39.4086 9.94721 39.7671 9.59041C40.1256 9.23361 40.3065 8.71361 40.3065 8.03041Z"
              fill="black"
            />
            <path
              d="M46.0325 8.82951C46.0325 9.40951 45.8623 9.88471 45.5219 10.2575C45.1651 10.6407 44.6923 10.8319 44.102 10.8319C43.533 10.8319 43.0799 10.6487 42.742 10.2807C42.4049 9.91351 42.2363 9.45031 42.2363 8.89191C42.2363 8.30791 42.4098 7.82872 42.7584 7.45672C43.1071 7.08472 43.5757 6.89832 44.1661 6.89832C44.7351 6.89832 45.1922 7.08152 45.5384 7.44872C45.8673 7.80552 46.0325 8.26631 46.0325 8.82951ZM45.1388 8.85671C45.1388 8.50872 45.0615 8.21031 44.9077 7.96151C44.7268 7.66071 44.4695 7.51031 44.1348 7.51031C43.7887 7.51031 43.5256 7.66071 43.3447 7.96151C43.1901 8.21031 43.1136 8.51352 43.1136 8.87191C43.1136 9.21991 43.1909 9.51831 43.3447 9.76711C43.5313 10.0679 43.7912 10.2183 44.1266 10.2183C44.4555 10.2183 44.7137 10.0655 44.8995 9.75911C45.059 9.50552 45.1388 9.20472 45.1388 8.85671Z"
              fill="black"
            />
            <path
              d="M52.4918 6.9751L51.279 10.7463H50.4896L49.9873 9.1087C49.8598 8.6999 49.7562 8.2935 49.6756 7.8903H49.66C49.5852 8.3047 49.4816 8.7103 49.3484 9.1087L48.8148 10.7463H48.0164L46.876 6.9751H47.7615L48.1998 8.7679C48.3058 9.1919 48.393 9.5959 48.4629 9.9783H48.4785C48.5426 9.6631 48.6487 9.2615 48.7983 8.7759L49.3484 6.9759H50.0506L50.5776 8.7375C50.7051 9.1671 50.8087 9.5807 50.8884 9.9791H50.9123C50.9706 9.5911 51.0586 9.1775 51.1754 8.7375L51.6457 6.9759H52.4918V6.9751Z"
              fill="black"
            />
            <path
              d="M56.9589 10.7464H56.0973V8.5864C56.0973 7.9208 55.8374 7.588 55.3161 7.588C55.0604 7.588 54.8541 7.6792 54.6937 7.8624C54.535 8.0456 54.4545 8.2616 54.4545 8.5088V10.7456H53.5928V8.0528C53.5928 7.7216 53.5821 7.3624 53.5615 6.9736H54.3188L54.3591 7.5632H54.3829C54.4832 7.38 54.6329 7.2288 54.8294 7.108C55.0629 6.9672 55.3244 6.896 55.6105 6.896C55.9723 6.896 56.2732 7.0096 56.5125 7.2376C56.8101 7.5168 56.9589 7.9336 56.9589 8.4872V10.7464Z"
              fill="black"
            />
            <path d="M59.335 10.7464H58.4741V5.24475H59.335V10.7464Z" fill="black" />
            <path
              d="M64.4085 8.82951C64.4085 9.40951 64.2383 9.88471 63.8979 10.2575C63.5411 10.6407 63.0675 10.8319 62.4779 10.8319C61.9081 10.8319 61.4551 10.6487 61.118 10.2807C60.7809 9.91351 60.6123 9.45031 60.6123 8.89191C60.6123 8.30791 60.7858 7.82872 61.1344 7.45672C61.483 7.08472 61.9517 6.89832 62.5412 6.89832C63.111 6.89832 63.5674 7.08152 63.9143 7.44872C64.2432 7.80552 64.4085 8.26631 64.4085 8.82951ZM63.5139 8.85671C63.5139 8.50872 63.4366 8.21031 63.2829 7.96151C63.1028 7.66071 62.8446 7.51031 62.5108 7.51031C62.1638 7.51031 61.9007 7.66071 61.7207 7.96151C61.5661 8.21031 61.4896 8.51352 61.4896 8.87191C61.4896 9.21991 61.5669 9.51831 61.7207 9.76711C61.9073 10.0679 62.1671 10.2183 62.5026 10.2183C62.8315 10.2183 63.0888 10.0655 63.2747 9.75911C63.435 9.50552 63.5139 9.20472 63.5139 8.85671Z"
              fill="black"
            />
            <path
              d="M68.5785 10.7463H67.8048L67.7406 10.3119H67.7168C67.452 10.6583 67.0746 10.8319 66.5846 10.8319C66.2187 10.8319 65.9227 10.7175 65.6999 10.4903C65.4976 10.2839 65.3965 10.0271 65.3965 9.72231C65.3965 9.26151 65.5938 8.91031 65.9909 8.66711C66.3873 8.42391 66.9447 8.30471 67.6625 8.31031V8.23992C67.6625 7.74312 67.3945 7.49512 66.8576 7.49512C66.4752 7.49512 66.1381 7.58872 65.8471 7.77432L65.6719 7.22392C66.0321 7.00712 66.4769 6.89832 67.0015 6.89832C68.0144 6.89832 68.5226 7.41832 68.5226 8.45831V9.84711C68.5226 10.2239 68.5415 10.5239 68.5785 10.7463ZM67.6839 9.45031V8.86872C66.7334 8.85272 66.2582 9.10632 66.2582 9.62872C66.2582 9.82551 66.3124 9.97271 66.4234 10.0711C66.5344 10.1695 66.6759 10.2183 66.8444 10.2183C67.0335 10.2183 67.2103 10.1599 67.3715 10.0439C67.5334 9.92711 67.6329 9.77911 67.6699 9.59751C67.679 9.55671 67.6839 9.50711 67.6839 9.45031Z"
              fill="black"
            />
            <path
              d="M73.4749 10.7464H72.7102L72.6699 10.1408H72.6461C72.4019 10.6016 71.9859 10.832 71.4013 10.832C70.9342 10.832 70.5453 10.6536 70.237 10.2968C69.9287 9.93995 69.7749 9.47675 69.7749 8.90795C69.7749 8.29755 69.9418 7.80315 70.2773 7.42555C70.6021 7.07355 71 6.89755 71.4736 6.89755C71.9941 6.89755 72.3583 7.06795 72.5655 7.40955H72.582V5.24475H73.4445V9.73035C73.4445 10.0976 73.4543 10.436 73.4749 10.7464ZM72.582 9.15595V8.52715C72.582 8.41835 72.5737 8.33035 72.5581 8.26315C72.5096 8.06155 72.4052 7.89195 72.2465 7.75515C72.0862 7.61835 71.8929 7.54955 71.6701 7.54955C71.3486 7.54955 71.097 7.67355 70.912 7.92235C70.7287 8.17115 70.6358 8.48875 70.6358 8.87675C70.6358 9.24955 70.7237 9.55195 70.9005 9.78475C71.0872 10.0328 71.3388 10.1568 71.6537 10.1568C71.9365 10.1568 72.1626 10.0536 72.3345 9.84635C72.5006 9.65515 72.582 9.42475 72.582 9.15595Z"
              fill="black"
            />
            <path
              d="M80.8445 8.82951C80.8445 9.40951 80.6743 9.88471 80.3339 10.2575C79.9771 10.6407 79.5051 10.8319 78.914 10.8319C78.3458 10.8319 77.8928 10.6487 77.554 10.2807C77.2169 9.91351 77.0483 9.45031 77.0483 8.89191C77.0483 8.30791 77.2218 7.82872 77.5705 7.45672C77.9191 7.08472 78.3877 6.89832 78.9789 6.89832C79.5471 6.89832 80.0051 7.08152 80.3504 7.44872C80.6793 7.80552 80.8445 8.26631 80.8445 8.82951ZM79.9516 8.85671C79.9516 8.50872 79.8743 8.21031 79.7206 7.96151C79.5388 7.66071 79.2823 7.51031 78.9469 7.51031C78.6015 7.51031 78.3384 7.66071 78.1567 7.96151C78.0021 8.21031 77.9257 8.51352 77.9257 8.87191C77.9257 9.21991 78.0029 9.51831 78.1567 9.76711C78.3433 10.0679 78.6032 10.2183 78.9386 10.2183C79.2675 10.2183 79.5265 10.0655 79.7123 9.75911C79.871 9.50552 79.9516 9.20472 79.9516 8.85671Z"
              fill="black"
            />
            <path
              d="M85.4776 10.7464H84.6168V8.5864C84.6168 7.9208 84.357 7.588 83.8349 7.588C83.5791 7.588 83.3728 7.6792 83.2132 7.8624C83.0537 8.0456 82.974 8.2616 82.974 8.5088V10.7456H82.1115V8.0528C82.1115 7.7216 82.1016 7.3624 82.0811 6.9736H82.8375L82.8778 7.5632H82.9016C83.0028 7.38 83.1524 7.2288 83.3481 7.108C83.5824 6.9672 83.8431 6.896 84.13 6.896C84.491 6.896 84.7919 7.0096 85.0312 7.2376C85.3296 7.5168 85.4776 7.9336 85.4776 8.4872V10.7464Z"
              fill="black"
            />
            <path
              d="M91.2766 7.60321H90.3278V9.43521C90.3278 9.90081 90.4963 10.1336 90.8301 10.1336C90.9847 10.1336 91.113 10.1208 91.2141 10.0944L91.2363 10.7304C91.0661 10.7928 90.8425 10.824 90.567 10.824C90.2266 10.824 89.9619 10.7232 89.7703 10.5216C89.5779 10.32 89.4825 9.98081 89.4825 9.50481V7.60321H88.916V6.97521H89.4825V6.28401L90.3269 6.03601V6.97441H91.2758V7.60321H91.2766Z"
              fill="black"
            />
            <path
              d="M95.8384 10.7464H94.9758V8.60235C94.9758 7.92635 94.716 7.58795 94.1956 7.58795C93.796 7.58795 93.523 7.78395 93.3733 8.17595C93.3478 8.25835 93.333 8.35915 93.333 8.47755V10.7456H92.4722V5.24475H93.333V7.51755H93.3495C93.6208 7.10395 94.0097 6.89755 94.5138 6.89755C94.8706 6.89755 95.1658 7.01115 95.4001 7.23915C95.692 7.52315 95.8384 7.94555 95.8384 8.50395V10.7464Z"
              fill="black"
            />
            <path
              d="M100.544 8.68231C100.544 8.83271 100.533 8.95911 100.512 9.06231H97.9278C97.9393 9.43511 98.0626 9.71911 98.3019 9.91591C98.5206 10.0919 98.8026 10.1799 99.1479 10.1799C99.5303 10.1799 99.8789 10.1207 100.193 10.0015L100.328 10.5839C99.9603 10.7391 99.5278 10.8167 99.0271 10.8167C98.4269 10.8167 97.9541 10.6447 97.612 10.3007C97.2683 9.95671 97.0981 9.49511 97.0981 8.91591C97.0981 8.34711 97.2568 7.87351 97.5767 7.49591C97.9105 7.09271 98.3619 6.89111 98.9317 6.89111C99.4892 6.89111 99.9126 7.09271 100.199 7.49591C100.43 7.81591 100.544 8.21191 100.544 8.68231ZM99.7219 8.46551C99.7284 8.21671 99.6717 8.00231 99.5549 7.82151C99.4053 7.58871 99.1775 7.47191 98.8692 7.47191C98.588 7.47191 98.3586 7.58551 98.1835 7.81351C98.0404 7.99511 97.9557 8.21191 97.9278 8.46551H99.7219Z"
              fill="black"
            />
            <path
              d="M44.993 25.2032H43.1258L42.1029 22.076H38.5476L37.5733 25.2032H35.7554L39.2778 14.5568H41.4534L44.993 25.2032ZM41.7946 20.764L40.8696 17.984C40.7717 17.7 40.5884 17.0312 40.3179 15.9784H40.285C40.1773 16.4312 40.0038 17.1 39.7653 17.984L38.8568 20.764H41.7946Z"
              fill="black"
            />
            <path
              d="M54.0513 21.2704C54.0513 22.576 53.6887 23.608 52.9635 24.3656C52.3139 25.04 51.5073 25.3768 50.5445 25.3768C49.5052 25.3768 48.7586 25.0136 48.3039 24.2872H48.2711V28.3312H46.5181V20.0536C46.5181 19.2328 46.4959 18.3904 46.4531 17.5264H47.9948L48.0926 18.7432H48.1255C48.7101 17.8264 49.5973 17.3688 50.7879 17.3688C51.7186 17.3688 52.4956 17.7264 53.1172 18.4424C53.7405 19.1592 54.0513 20.1016 54.0513 21.2704ZM52.2654 21.3328C52.2654 20.5856 52.0927 19.9696 51.7458 19.4848C51.3667 18.9792 50.8578 18.7264 50.2197 18.7264C49.7872 18.7264 49.3942 18.8672 49.0431 19.1448C48.6912 19.4248 48.461 19.7904 48.3533 20.2432C48.299 20.4544 48.2719 20.6272 48.2719 20.7632V22.0432C48.2719 22.6016 48.4478 23.0728 48.7997 23.4576C49.1517 23.8424 49.6088 24.0344 50.1712 24.0344C50.8315 24.0344 51.3453 23.7864 51.7129 23.292C52.0812 22.7968 52.2654 22.144 52.2654 21.3328Z"
              fill="black"
            />
            <path
              d="M63.1261 21.2704C63.1261 22.576 62.7635 23.608 62.0375 24.3656C61.3888 25.04 60.5822 25.3768 59.6194 25.3768C58.5801 25.3768 57.8335 25.0136 57.3796 24.2872H57.3467V28.3312H55.5938V20.0536C55.5938 19.2328 55.5716 18.3904 55.5288 17.5264H57.0705L57.1683 18.7432H57.2012C57.785 17.8264 58.6722 17.3688 59.8636 17.3688C60.7935 17.3688 61.5705 17.7264 62.1937 18.4424C62.8145 19.1592 63.1261 20.1016 63.1261 21.2704ZM61.3403 21.3328C61.3403 20.5856 61.1668 19.9696 60.8198 19.4848C60.4408 18.9792 59.9335 18.7264 59.2946 18.7264C58.8613 18.7264 58.4691 18.8672 58.1172 19.1448C57.7653 19.4248 57.5359 19.7904 57.4281 20.2432C57.3747 20.4544 57.3467 20.6272 57.3467 20.7632V22.0432C57.3467 22.6016 57.5227 23.0728 57.873 23.4576C58.2249 23.8416 58.682 24.0344 59.2461 24.0344C59.9063 24.0344 60.4202 23.7864 60.7877 23.292C61.1561 22.7968 61.3403 22.144 61.3403 21.3328Z"
              fill="black"
            />
            <path
              d="M73.2726 22.2176C73.2726 23.1232 72.9495 23.86 72.3008 24.4288C71.5879 25.0504 70.5955 25.3608 69.3202 25.3608C68.1428 25.3608 67.1989 25.14 66.4844 24.6976L66.8906 23.276C67.6602 23.7288 68.5046 23.956 69.4246 23.956C70.0849 23.956 70.5988 23.8104 70.968 23.5208C71.3355 23.2312 71.5188 22.8424 71.5188 22.3576C71.5188 21.9256 71.3676 21.5616 71.0642 21.2664C70.7624 20.9712 70.2584 20.6968 69.5546 20.4432C67.6388 19.748 66.6817 18.7296 66.6817 17.3904C66.6817 16.5152 67.0172 15.7976 67.6889 15.2392C68.3582 14.68 69.2512 14.4008 70.3677 14.4008C71.3634 14.4008 72.1906 14.5696 72.8508 14.9064L72.4126 16.2968C71.7959 15.9704 71.0987 15.8072 70.3184 15.8072C69.7017 15.8072 69.2199 15.9552 68.8746 16.2496C68.5827 16.5128 68.4363 16.8336 68.4363 17.2136C68.4363 17.6344 68.6032 17.9824 68.9387 18.256C69.2306 18.5088 69.7609 18.7824 70.5305 19.0776C71.472 19.4464 72.1635 19.8776 72.6083 20.372C73.0515 20.8648 73.2726 21.4816 73.2726 22.2176Z"
              fill="black"
            />
            <path
              d="M79.0682 18.8063H77.136V22.5335C77.136 23.4815 77.4764 23.9551 78.1589 23.9551C78.4721 23.9551 78.732 23.9287 78.9375 23.8759L78.986 25.1711C78.6407 25.2967 78.186 25.3599 77.6228 25.3599C76.9305 25.3599 76.3895 25.1543 75.9989 24.7439C75.61 24.3327 75.4143 23.6431 75.4143 22.6743V18.8047H74.2632V17.5247H75.4143V16.1191L77.136 15.6135V17.5247H79.0682V18.8063Z"
              fill="black"
            />
            <path
              d="M87.7862 21.3015C87.7862 22.4815 87.4392 23.4503 86.7469 24.2079C86.0209 24.9879 85.0572 25.3767 83.856 25.3767C82.6983 25.3767 81.7765 25.0031 81.0892 24.2559C80.4018 23.5087 80.0581 22.5655 80.0581 21.4287C80.0581 20.2391 80.4117 19.2647 81.1212 18.5071C81.8292 17.7487 82.7846 17.3695 83.9859 17.3695C85.1435 17.3695 86.0751 17.7431 86.7781 18.4911C87.4507 19.2167 87.7862 20.1535 87.7862 21.3015ZM85.9674 21.3567C85.9674 20.6487 85.812 20.0415 85.4971 19.5351C85.1296 18.9223 84.6042 18.6167 83.9234 18.6167C83.2187 18.6167 82.6835 18.9231 82.3159 19.5351C82.001 20.0423 81.8456 20.6591 81.8456 21.3887C81.8456 22.0967 82.001 22.7039 82.3159 23.2095C82.695 23.8223 83.2245 24.1279 83.9077 24.1279C84.577 24.1279 85.1024 23.8159 85.4815 23.1935C85.8046 22.6775 85.9674 22.0639 85.9674 21.3567Z"
              fill="black"
            />
            <path
              d="M93.4845 19.0264C93.311 18.9952 93.126 18.9792 92.932 18.9792C92.3153 18.9792 91.8384 19.2056 91.5029 19.6592C91.211 20.0592 91.0647 20.5648 91.0647 21.1752V25.2032H89.3125L89.329 19.944C89.329 19.0592 89.3068 18.2536 89.2632 17.5272H90.7901L90.8542 18.996H90.9027C91.0877 18.4912 91.3796 18.0848 91.7792 17.78C92.1697 17.5056 92.5915 17.3688 93.0462 17.3688C93.2082 17.3688 93.3546 17.38 93.4845 17.4V19.0264Z"
              fill="black"
            />
            <path
              d="M101.324 21.0016C101.324 21.3072 101.304 21.5648 101.26 21.7752H96.0013C96.0219 22.5336 96.276 23.1136 96.7644 23.5136C97.2075 23.8712 97.7806 24.0504 98.4845 24.0504C99.2631 24.0504 99.9735 23.9296 100.612 23.6872L100.887 24.8712C100.14 25.188 99.259 25.3456 98.2419 25.3456C97.0184 25.3456 96.0581 24.9952 95.3592 24.2952C94.6619 23.5952 94.3125 22.6552 94.3125 21.476C94.3125 20.3184 94.6373 19.3544 95.2877 18.5856C95.9685 17.7648 96.8885 17.3544 98.0462 17.3544C99.1833 17.3544 100.044 17.7648 100.629 18.5856C101.092 19.2376 101.324 20.044 101.324 21.0016ZM99.6528 20.5592C99.6643 20.0536 99.55 19.6168 99.3124 19.248C99.009 18.7736 98.5428 18.5368 97.9155 18.5368C97.3424 18.5368 96.8762 18.768 96.5202 19.232C96.2283 19.6008 96.0548 20.0432 96.0013 20.5584H99.6528V20.5592Z"
              fill="black"
            />
          </svg>
        </a>
        <a
          className="external"
          href={externalLink.clientApp.download.desktop}
          target="_blank"
          rel="noopener noreferrer"
        >
          <Button
            text={localLL.desktopDownload()}
            size={ButtonSize.SMALL}
            styleVariant={ButtonStyleVariant.LINK}
          />
        </a>
      </div>
    </Card>
  );
};
