/* ========================================================================== */
/*! \file
 * \brief Test of \c core_get_cancel_lock() implementation
 *
 * Copyright (c) 2015-2024 by the developers. See the LICENSE file for details.
 */


/* ========================================================================== */
/* Include headers */

#include "posix.h"  /* Include this first because of feature test macros */

#include <stdio.h>

#include "conf.h"
#include "core.h"
#include "test.h"
#include "test_cancellock.h"


/* ========================================================================== */
/*! \addtogroup TEST */
/*! @{ */


/* ========================================================================== */
/*! \brief Test \c core_get_cancel_lock() implementation
 *
 * The following cases are tested:
 * - Three different Message-ID/Secret pairs for Cancel-Lock with scheme SHA1
 *
 * \return
 * - \c EXIT_SUCCESS on success
 * - \c EXIT_FAILURE on error
 */

int  test_cancellock(void)
{
#define TS_NUM  (size_t) 3  /* Number of tests */
   static const char*  ts[TS_NUM] =
   {
      "Dummy1MessageID@invalid.invalid",
      "DifferentDummy2MessageID@test.invalid",
      "Dummy3@x.invalid"
   };
   static const char*  rs_sha1[TS_NUM] =
   {
      "sha1:/bPdszYoGg5jN5ND++QigLj0t9I=",
      "sha1:iuGB0LPk4gOkm+P8KWUUvREdOgY=",
      "sha1:NvsbxX7h74kZYytOj2fbYrL3lhY="
   };
   static const char*  secret_sha1[TS_NUM] =
   {
      "This string is used as secret by test system for test 1",
      "This string is used as secret by test system for test 2",
      "Secret3"
   };

   int  res = API_POSIX_EXIT_SUCCESS;
   size_t  i;
   const char*  buf = NULL;
   int  rv;

   /* Test SHA1 algorithm */
   for(i = 0; i < TS_NUM; ++i)
   {
      rv = conf_string_replace(&config[CONF_CANCELKEY], secret_sha1[i]);
      if(rv)
      {
         print_error("Preparation of configuration failed");
         res = API_POSIX_EXIT_FAILURE;
         break;
      }
      buf = core_get_cancel_lock(CORE_CL_SHA1, ts[i]);
      if(NULL == buf)
      {
         print_error("Cancel-Lock generation failed");
         res = API_POSIX_EXIT_FAILURE;
         break;
      }
      if(strcmp(&rs_sha1[i][4], &buf[4]))
      {
         print_error("Cancel-Lock is not correct");
         /* For debugging */
         fprintf(stderr, TEST_TAB "Secret    : \"%s\"\n",
                 config[CONF_CANCELKEY].val.s);
         fprintf(stderr, TEST_TAB "Message-ID: \"%s\"\n", ts[i]);
         fprintf(stderr, TEST_TAB "Result is : \"%s\"\n", buf);
         fprintf(stderr, TEST_TAB "Should be : \"%s\"\n", rs_sha1[i]);
         res = API_POSIX_EXIT_FAILURE;
         break;
      }
      core_free((void*) buf);
   }

   return(res);
}


/*! @} */

/* EOF */
