// Copyright 2019, VIXL authors
// All rights reserved.
//
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
//
//   * Redistributions of source code must retain the above copyright notice,
//     this list of conditions and the following disclaimer.
//   * Redistributions in binary form must reproduce the above copyright notice,
//     this list of conditions and the following disclaimer in the documentation
//     and/or other materials provided with the distribution.
//   * Neither the name of ARM Limited nor the names of its contributors may be
//     used to endorse or promote products derived from this software without
//     specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS CONTRIBUTORS "AS IS" AND
// ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
// WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
// DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE LIABLE
// FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
// DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR
// SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
// CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
// OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

namespace vixl {
namespace aarch64 {

// Recursively construct a uint32_t encoded bit pattern from a string literal.
// The string characters are mapped as two-bit symbols '0'=>0, '1'=>1, 'x'=>2.
// The remaining symbol, 3, is used to mark the end of the pattern, allowing
// its length to be found. For example, the pattern "1x01"_b is encoded in a
// uint32_t as 0b11_01_00_01. The maximum pattern string length is 15
// characters, encoded as 3 in the most significant bits, followed by 15 2-bit
// symbols.
constexpr uint32_t str_to_two_bit_pattern(const char* x, size_t s, uint32_t a) {
  if (s == 0) return a;
  uint32_t r = (x[0] == 'x') ? 2 : (x[0] - '0');
  return str_to_two_bit_pattern(x + 1, s - 1, (a << 2) | r);
}

constexpr uint32_t operator"" _b(const char* x, size_t s) {
  return str_to_two_bit_pattern(x, s, DecodeNode::kEndOfPattern);
}

// This decode table is derived from the AArch64 ISA XML specification,
// available from https://developer.arm.com/products/architecture/a-profile/

// clang-format off
static const DecodeMapping kDecodeMapping[] = {
  { "_ggvlym",
    {13, 12},
    { {"00"_b, "adc_32_addsub_carry"},
    },
  },

  { "_ghmtnl",
    {18, 17},
    { {"0x"_b, "ld1_asisdlsep_r3_r3"},
      {"10"_b, "ld1_asisdlsep_r3_r3"},
      {"11"_b, "ld1_asisdlsep_i3_i3"},
    },
  },

  { "_ghpxms",
    {23, 22},
    { {"01"_b, "fmla_z_p_zzz"},
      {"1x"_b, "fmla_z_p_zzz"},
    },
  },

  { "_ghqqzy",
    {11},
    { {"0"_b, "_qrsxzp"},
    },
  },

  { "_ghrnmz",
    {20, 19, 18, 17, 16, 13, 12, 9, 8, 7, 6, 5},
    { {"000010011111"_b, "xpacd_64z_dp_1src"},
    },
  },

  { "_gjprgr",
    {22, 13, 12},
    { {"000"_b, "ldsmax_64_memop"},
      {"001"_b, "ldsmin_64_memop"},
      {"010"_b, "ldumax_64_memop"},
      {"011"_b, "ldumin_64_memop"},
      {"100"_b, "ldsmaxl_64_memop"},
      {"101"_b, "ldsminl_64_memop"},
      {"110"_b, "ldumaxl_64_memop"},
      {"111"_b, "lduminl_64_memop"},
    },
  },

  { "_gjprmg",
    {11},
    { {"0"_b, "_llpsqq"},
    },
  },

  { "_gjtmjg",
    {23, 22, 20, 19, 13, 11},
    { {"0000x0"_b, "bic_asimdimm_l_hl"},
      {"00x100"_b, "uqshrn_asimdshf_n"},
      {"00x101"_b, "uqrshrn_asimdshf_n"},
      {"010x00"_b, "uqshrn_asimdshf_n"},
      {"010x01"_b, "uqrshrn_asimdshf_n"},
      {"011100"_b, "uqshrn_asimdshf_n"},
      {"011101"_b, "uqrshrn_asimdshf_n"},
      {"0x1000"_b, "uqshrn_asimdshf_n"},
      {"0x1001"_b, "uqrshrn_asimdshf_n"},
    },
  },

  { "_gjxsrn",
    {2, 1, 0},
    { {"000"_b, "_sqttsv"},
    },
  },

  { "_gknljg",
    {11, 10, 9, 8, 7, 6},
    { {"000000"_b, "wfet_only_systeminstrswithreg"},
    },
  },

  { "_gkqhyz",
    {23, 22},
    { {"00"_b, "fmsub_s_floatdp3"},
      {"01"_b, "fmsub_d_floatdp3"},
      {"11"_b, "fmsub_h_floatdp3"},
    },
  },

  { "_glgznt",
    {20, 19, 18, 17, 16, 4, 3},
    { {"0000001"_b, "fcmp_dz_floatcmp"},
      {"0000011"_b, "fcmpe_dz_floatcmp"},
      {"xxxxx00"_b, "fcmp_d_floatcmp"},
      {"xxxxx10"_b, "fcmpe_d_floatcmp"},
    },
  },

  { "_gljqng",
    {22, 13, 12},
    { {"000"_b, "ldsmaxa_32_memop"},
      {"001"_b, "ldsmina_32_memop"},
      {"010"_b, "ldumaxa_32_memop"},
      {"011"_b, "ldumina_32_memop"},
      {"100"_b, "ldsmaxal_32_memop"},
      {"101"_b, "ldsminal_32_memop"},
      {"110"_b, "ldumaxal_32_memop"},
      {"111"_b, "lduminal_32_memop"},
    },
  },

  { "_glkvkr",
    {30},
    { {"0"_b, "adds_32_addsub_shift"},
      {"1"_b, "subs_32_addsub_shift"},
    },
  },

  { "_glpxty",
    {20, 19, 18, 17, 16},
    { {"00010"_b, "scvtf_s32_float2fix"},
      {"00011"_b, "ucvtf_s32_float2fix"},
      {"11000"_b, "fcvtzs_32s_float2fix"},
      {"11001"_b, "fcvtzu_32s_float2fix"},
    },
  },

  { "_gmqyjv",
    {30, 20, 19, 18, 17, 16, 13},
    { {"1111110"_b, "_nvkxzs"},
    },
  },

  { "_gmsmls",
    {13},
    { {"0"_b, "mls_asimdelem_r"},
      {"1"_b, "umlsl_asimdelem_l"},
    },
  },

  { "_gmsqqz",
    {23},
    { {"0"_b, "facge_asimdsame_only"},
      {"1"_b, "facgt_asimdsame_only"},
    },
  },

  { "_gmtjvr",
    {16, 13, 12},
    { {"000"_b, "rev_64_dp_1src"},
      {"001"_b, "cnt_64_dp_1src"},
      {"100"_b, "pacdb_64p_dp_1src"},
      {"101"_b, "autdb_64p_dp_1src"},
      {"110"_b, "_rlxhxz"},
      {"111"_b, "_phjkhr"},
    },
  },

  { "_gngjxr",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "cadd_z_zz"},
      {"00001"_b, "sqcadd_z_zz"},
    },
  },

  { "_gnhjkl",
    {16, 13, 12},
    { {"000"_b, "rbit_64_dp_1src"},
      {"001"_b, "clz_64_dp_1src"},
      {"010"_b, "abs_64_dp_1src"},
      {"100"_b, "pacia_64p_dp_1src"},
      {"101"_b, "autia_64p_dp_1src"},
      {"110"_b, "_yzxjnk"},
      {"111"_b, "_prxyhr"},
    },
  },

  { "_gnpgsg",
    {22},
    { {"0"_b, "str_64_ldst_regoff"},
      {"1"_b, "ldr_64_ldst_regoff"},
    },
  },

  { "_gnqhsl",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0010000"_b, "punpklo_p_p"},
      {"0010001"_b, "punpkhi_p_p"},
      {"xx0xxxx"_b, "zip1_p_pp"},
      {"xx10100"_b, "rev_p_p"},
    },
  },

  { "_gnxrlr",
    {23, 22, 13, 12, 11, 10},
    { {"0011x0"_b, "sudot_asimdelem_d"},
      {"0111x0"_b, "bfdot_asimdelem_e"},
      {"0x1001"_b, "scvtf_asimdshf_c"},
      {"0x1111"_b, "fcvtzs_asimdshf_c"},
      {"1011x0"_b, "usdot_asimdelem_d"},
      {"1111x0"_b, "bfmlal_asimdelem_f"},
      {"xx00x0"_b, "sqdmulh_asimdelem_r"},
      {"xx01x0"_b, "sqrdmulh_asimdelem_r"},
      {"xx10x0"_b, "sdot_asimdelem_d"},
    },
  },

  { "_gplkxy",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "sqneg_asimdmisc_r"},
    },
  },

  { "_gpxltv",
    {23, 18, 17, 16},
    { {"0000"_b, "uqxtnt_z_zz"},
    },
  },

  { "_gqmjys",
    {18, 17},
    { {"0x"_b, "st1_asisdlsop_sx1_r1s"},
      {"10"_b, "st1_asisdlsop_sx1_r1s"},
      {"11"_b, "st1_asisdlsop_s1_i1s"},
    },
  },

  { "_grgrpt",
    {18},
    { {"1"_b, "fmaxv_v_p_z"},
    },
  },

  { "_grjzyl",
    {30},
    { {"0"_b, "bl_only_branch_imm"},
      {"1"_b, "_hjtsgj"},
    },
  },

  { "_grktgm",
    {30, 23, 22, 19},
    { {"1001"_b, "aesd_b_cryptoaes"},
      {"xxx0"_b, "cnt_asimdmisc_r"},
    },
  },

  { "_grmpht",
    {20, 18, 17},
    { {"000"_b, "_mjjhqj"},
    },
  },

  { "_grprpj",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldarb_lr32_ldstexcl"},
    },
  },

  { "_grqsgp",
    {23, 22, 4, 3, 2, 1, 0},
    { {"0000001"_b, "svc_ex_exception"},
      {"0000010"_b, "hvc_ex_exception"},
      {"0000011"_b, "smc_ex_exception"},
      {"0100000"_b, "hlt_ex_exception"},
    },
  },

  { "_grsnms",
    {20, 19, 18, 17, 16},
    { {"00010"_b, "scvtf_h32_float2fix"},
      {"00011"_b, "ucvtf_h32_float2fix"},
      {"11000"_b, "fcvtzs_32h_float2fix"},
      {"11001"_b, "fcvtzu_32h_float2fix"},
    },
  },

  { "_grsslr",
    {30, 23, 22, 11, 10, 4},
    { {"001000"_b, "ccmn_32_condcmp_reg"},
      {"001100"_b, "ccmn_32_condcmp_imm"},
      {"101000"_b, "ccmp_32_condcmp_reg"},
      {"101100"_b, "ccmp_32_condcmp_imm"},
    },
  },

  { "_grvxrm",
    {12},
    { {"0"_b, "st4_asisdlsop_dx4_r4d"},
    },
  },

  { "_gshlgj",
    {30, 23, 22, 13, 12, 11, 10},
    { {"100xxx1"_b, "ins_asimdins_iv_v"},
      {"x00xxx0"_b, "ext_asimdext_only"},
      {"x010001"_b, "fmaxnmp_asimdsamefp16_only"},
      {"x010101"_b, "faddp_asimdsamefp16_only"},
      {"x010111"_b, "fmul_asimdsamefp16_only"},
      {"x011001"_b, "fcmge_asimdsamefp16_only"},
      {"x011011"_b, "facge_asimdsamefp16_only"},
      {"x011101"_b, "fmaxp_asimdsamefp16_only"},
      {"x011111"_b, "fdiv_asimdsamefp16_only"},
      {"x110001"_b, "fminnmp_asimdsamefp16_only"},
      {"x110101"_b, "fabd_asimdsamefp16_only"},
      {"x111001"_b, "fcmgt_asimdsamefp16_only"},
      {"x111011"_b, "facgt_asimdsamefp16_only"},
      {"x111101"_b, "fminp_asimdsamefp16_only"},
    },
  },

  { "_gshrzq",
    {22, 20, 11},
    { {"010"_b, "decb_r_rs"},
      {"110"_b, "dech_r_rs"},
    },
  },

  { "_gsjvmx",
    {12},
    { {"0"_b, "st3_asisdlsop_dx3_r3d"},
    },
  },

  { "_gslmjl",
    {23, 22},
    { {"00"_b, "fcsel_s_floatsel"},
      {"01"_b, "fcsel_d_floatsel"},
      {"11"_b, "fcsel_h_floatsel"},
    },
  },

  { "_gsnnnt",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx00"_b, "stlurb_32_ldapstl_unscaled"},
      {"001xx00"_b, "ldapurb_32_ldapstl_unscaled"},
      {"010xx00"_b, "ldapursb_64_ldapstl_unscaled"},
      {"011xx00"_b, "ldapursb_32_ldapstl_unscaled"},
      {"100xx00"_b, "stlurh_32_ldapstl_unscaled"},
      {"101xx00"_b, "ldapurh_32_ldapstl_unscaled"},
      {"110xx00"_b, "ldapursh_64_ldapstl_unscaled"},
      {"111xx00"_b, "ldapursh_32_ldapstl_unscaled"},
      {"x000001"_b, "cpyfprn_cpy_memcms"},
      {"x000101"_b, "cpyfpwtrn_cpy_memcms"},
      {"x001001"_b, "cpyfprtrn_cpy_memcms"},
      {"x001101"_b, "cpyfptrn_cpy_memcms"},
      {"x010001"_b, "cpyfmrn_cpy_memcms"},
      {"x010101"_b, "cpyfmwtrn_cpy_memcms"},
      {"x011001"_b, "cpyfmrtrn_cpy_memcms"},
      {"x011101"_b, "cpyfmtrn_cpy_memcms"},
      {"x100001"_b, "cpyfern_cpy_memcms"},
      {"x100101"_b, "cpyfewtrn_cpy_memcms"},
      {"x101001"_b, "cpyfertrn_cpy_memcms"},
      {"x101101"_b, "cpyfetrn_cpy_memcms"},
      {"x110001"_b, "sete_set_memcms"},
      {"x110101"_b, "setet_set_memcms"},
      {"x111001"_b, "seten_set_memcms"},
      {"x111101"_b, "setetn_set_memcms"},
    },
  },

  { "_gsvlph",
    {22, 4, 3},
    { {"00x"_b, "prfm_p_ldst_regoff"},
      {"010"_b, "prfm_p_ldst_regoff"},
      {"011"_b, "rprfm_r_ldst_regoff"},
    },
  },

  { "_gtqnvr",
    {30, 23, 22},
    { {"000"_b, "msub_32a_dp_3src"},
    },
  },

  { "_gtsglj",
    {11, 10, 9, 8, 7, 6},
    { {"000001"_b, "tcommit_only_barriers"},
      {"000011"_b, "sb_only_barriers"},
      {"xx1000"_b, "dsb_bon_barriers"},
      {"xxxx10"_b, "dmb_bo_barriers"},
    },
  },

  { "_gtxpgx",
    {30, 23, 13, 4},
    { {"0000"_b, "prfw_i_p_bz_s_x32_scaled"},
      {"0010"_b, "prfd_i_p_bz_s_x32_scaled"},
      {"010x"_b, "ld1h_z_p_bz_s_x32_scaled"},
      {"011x"_b, "ldff1h_z_p_bz_s_x32_scaled"},
      {"1000"_b, "prfw_i_p_bz_d_x32_scaled"},
      {"1010"_b, "prfd_i_p_bz_d_x32_scaled"},
      {"110x"_b, "ld1h_z_p_bz_d_x32_scaled"},
      {"111x"_b, "ldff1h_z_p_bz_d_x32_scaled"},
    },
  },

  { "_gvpvjn",
    {20, 19, 18, 17, 16, 13, 12},
    { {"0000000"_b, "rev_32_dp_1src"},
      {"0000001"_b, "ctz_32_dp_1src"},
    },
  },

  { "_gvxjvz",
    {23, 22, 12},
    { {"000"_b, "_tgvkhm"},
      {"001"_b, "_ktyrgy"},
      {"010"_b, "_gxzgtk"},
      {"011"_b, "_vlxrps"},
      {"110"_b, "_jqrmyp"},
      {"111"_b, "_ssypmm"},
    },
  },

  { "_gxmnkl",
    {23, 22},
    { {"10"_b, "cdot_z_zzzi_s"},
      {"11"_b, "cdot_z_zzzi_d"},
    },
  },

  { "_gxqnph",
    {23, 22, 13, 12, 11, 10},
    { {"0x1001"_b, "ucvtf_asimdshf_c"},
      {"0x1111"_b, "fcvtzu_asimdshf_c"},
      {"1000x0"_b, "fmlsl2_asimdelem_lh"},
      {"xx01x0"_b, "sqrdmlah_asimdelem_r"},
      {"xx10x0"_b, "udot_asimdelem_d"},
      {"xx11x0"_b, "sqrdmlsh_asimdelem_r"},
    },
  },

  { "_gxzgtk",
    {20, 19, 18, 17, 16, 13},
    { {"000000"_b, "fabs_d_floatdp1"},
      {"000010"_b, "fsqrt_d_floatdp1"},
      {"000110"_b, "fcvt_hd_floatdp1"},
      {"001000"_b, "frintp_d_floatdp1"},
      {"001010"_b, "frintz_d_floatdp1"},
      {"001110"_b, "frinti_d_floatdp1"},
      {"010000"_b, "frint32x_d_floatdp1"},
      {"010010"_b, "frint64x_d_floatdp1"},
    },
  },

  { "_gyjphh",
    {30, 23, 22, 11, 10},
    { {"00000"_b, "_plgrmv"},
      {"00001"_b, "_xmxhhg"},
      {"00100"_b, "_lmmjvx"},
      {"00110"_b, "_tmtgqm"},
      {"01100"_b, "_hvmyjz"},
      {"10000"_b, "_mgtxyt"},
      {"10100"_b, "_rkzlpp"},
      {"10110"_b, "_xqrgjj"},
    },
  },

  { "_gyllxt",
    {23},
    { {"0"_b, "_hzkxht"},
    },
  },

  { "_gylmmr",
    {30, 23, 22, 11, 10},
    { {"00010"_b, "str_b_ldst_regoff"},
      {"00110"_b, "ldr_b_ldst_regoff"},
      {"01010"_b, "str_q_ldst_regoff"},
      {"01110"_b, "ldr_q_ldst_regoff"},
      {"10010"_b, "str_h_ldst_regoff"},
      {"10110"_b, "ldr_h_ldst_regoff"},
    },
  },

  { "_gyrjrm",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "cpy_z_p_v"},
      {"00001"_b, "compact_z_p_z"},
      {"00010"_b, "lasta_v_p_z"},
      {"00011"_b, "lastb_v_p_z"},
      {"00100"_b, "revb_z_z"},
      {"00101"_b, "revh_z_z"},
      {"00110"_b, "revw_z_z"},
      {"00111"_b, "rbit_z_p_z"},
      {"01000"_b, "clasta_z_p_zz"},
      {"01001"_b, "clastb_z_p_zz"},
      {"01010"_b, "clasta_v_p_z"},
      {"01011"_b, "clastb_v_p_z"},
      {"01100"_b, "splice_z_p_zz_des"},
      {"01101"_b, "splice_z_p_zz_con"},
    },
  },

  { "_gyrkkz",
    {30, 22, 11},
    { {"000"_b, "_nqjvmr"},
      {"001"_b, "_jjnvrv"},
      {"010"_b, "_yptgjg"},
      {"011"_b, "_vsyjql"},
      {"100"_b, "_lzqxgt"},
      {"110"_b, "_xvrvhv"},
      {"111"_b, "_ptstkz"},
    },
  },

  { "_gyymmx",
    {30, 13, 12},
    { {"000"_b, "stilp_32se_ldiappstilp"},
      {"001"_b, "stilp_32s_ldiappstilp"},
      {"100"_b, "stilp_64ss_ldiappstilp"},
      {"101"_b, "stilp_64s_ldiappstilp"},
    },
  },

  { "_gzgpjp",
    {23},
    { {"0"_b, "fmaxp_asimdsame_only"},
      {"1"_b, "fminp_asimdsame_only"},
    },
  },

  { "_gznrjv",
    {30, 23, 22, 19, 16},
    { {"10010"_b, "aese_b_cryptoaes"},
      {"xxx00"_b, "cls_asimdmisc_r"},
      {"xxx01"_b, "sqxtn_asimdmisc_n"},
    },
  },

  { "_gzpkvm",
    {30, 23, 22, 13, 12, 11, 10},
    { {"0000000"_b, "swpb_32_memop"},
      {"0000100"_b, "rcwclr_64_memop"},
      {"0001000"_b, "rcwswp_64_memop"},
      {"0001100"_b, "rcwset_64_memop"},
      {"000xx10"_b, "strb_32b_ldst_regoff"},
      {"0010000"_b, "swplb_32_memop"},
      {"0010100"_b, "rcwclrl_64_memop"},
      {"0011000"_b, "rcwswpl_64_memop"},
      {"0011100"_b, "rcwsetl_64_memop"},
      {"001xx10"_b, "ldrb_32b_ldst_regoff"},
      {"0100000"_b, "swpab_32_memop"},
      {"0100100"_b, "rcwclra_64_memop"},
      {"0101000"_b, "rcwswpa_64_memop"},
      {"0101100"_b, "rcwseta_64_memop"},
      {"010xx10"_b, "ldrsb_64b_ldst_regoff"},
      {"0110000"_b, "swpalb_32_memop"},
      {"0110100"_b, "rcwclral_64_memop"},
      {"0111000"_b, "rcwswpal_64_memop"},
      {"0111100"_b, "rcwsetal_64_memop"},
      {"011xx10"_b, "ldrsb_32b_ldst_regoff"},
      {"1000000"_b, "swph_32_memop"},
      {"1000100"_b, "rcwsclr_64_memop"},
      {"1001000"_b, "rcwsswp_64_memop"},
      {"1001100"_b, "rcwsset_64_memop"},
      {"100xx10"_b, "strh_32_ldst_regoff"},
      {"1010000"_b, "swplh_32_memop"},
      {"1010100"_b, "rcwsclrl_64_memop"},
      {"1011000"_b, "rcwsswpl_64_memop"},
      {"1011100"_b, "rcwssetl_64_memop"},
      {"101xx10"_b, "ldrh_32_ldst_regoff"},
      {"1100000"_b, "swpah_32_memop"},
      {"1100100"_b, "rcwsclra_64_memop"},
      {"1101000"_b, "rcwsswpa_64_memop"},
      {"1101100"_b, "rcwsseta_64_memop"},
      {"110xx10"_b, "ldrsh_64_ldst_regoff"},
      {"1110000"_b, "swpalh_32_memop"},
      {"1110100"_b, "rcwsclral_64_memop"},
      {"1111000"_b, "rcwsswpal_64_memop"},
      {"1111100"_b, "rcwssetal_64_memop"},
      {"111xx10"_b, "ldrsh_32_ldst_regoff"},
    },
  },

  { "_gzqvnk",
    {23, 12, 4, 3, 2, 1, 0},
    { {"1000000"_b, "ctermeq_rr"},
      {"1010000"_b, "ctermne_rr"},
      {"x10xxxx"_b, "whilewr_p_rr"},
      {"x11xxxx"_b, "whilerw_p_rr"},
    },
  },

  { "_gzrtkk",
    {18, 17},
    { {"0x"_b, "ld1_asisdlsep_r1_r1"},
      {"10"_b, "ld1_asisdlsep_r1_r1"},
      {"11"_b, "ld1_asisdlsep_i1_i1"},
    },
  },

  { "_gzvylr",
    {30, 13},
    { {"00"_b, "_rjyrnt"},
      {"01"_b, "_mzhsrq"},
      {"10"_b, "_prtvjm"},
      {"11"_b, "_zspprz"},
    },
  },

  { "_gzzsgh",
    {18},
    { {"0"_b, "ld3_asisdlso_b3_3b"},
    },
  },

  { "_hgjgpm",
    {30},
    { {"0"_b, "bic_64_log_shift"},
      {"1"_b, "eon_64_log_shift"},
    },
  },

  { "_hgxtqy",
    {30, 23, 22, 13},
    { {"0001"_b, "ldnt1w_z_p_ar_s_x32_unscaled"},
      {"0010"_b, "ld1rsh_z_p_bi_s64"},
      {"0011"_b, "ld1rsh_z_p_bi_s32"},
      {"0110"_b, "ld1rsb_z_p_bi_s64"},
      {"0111"_b, "ld1rsb_z_p_bi_s32"},
      {"1000"_b, "ldnt1sw_z_p_ar_d_64_unscaled"},
      {"1010"_b, "ld1sw_z_p_bz_d_64_unscaled"},
      {"1011"_b, "ldff1sw_z_p_bz_d_64_unscaled"},
    },
  },

  { "_hhkqtn",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "lasta_r_p_z"},
      {"00001"_b, "lastb_r_p_z"},
      {"01000"_b, "cpy_z_p_r"},
      {"10000"_b, "clasta_r_p_z"},
      {"10001"_b, "clastb_r_p_z"},
    },
  },

  { "_hhlmrg",
    {23, 20, 19, 18, 17, 16, 13},
    { {"0000000"_b, "ld2r_asisdlso_r2"},
      {"0000001"_b, "ld4r_asisdlso_r4"},
      {"10xxxx0"_b, "ld2r_asisdlsop_rx2_r"},
      {"10xxxx1"_b, "ld4r_asisdlsop_rx4_r"},
      {"110xxx0"_b, "ld2r_asisdlsop_rx2_r"},
      {"110xxx1"_b, "ld4r_asisdlsop_rx4_r"},
      {"1110xx0"_b, "ld2r_asisdlsop_rx2_r"},
      {"1110xx1"_b, "ld4r_asisdlsop_rx4_r"},
      {"11110x0"_b, "ld2r_asisdlsop_rx2_r"},
      {"11110x1"_b, "ld4r_asisdlsop_rx4_r"},
      {"1111100"_b, "ld2r_asisdlsop_rx2_r"},
      {"1111101"_b, "ld4r_asisdlsop_rx4_r"},
      {"1111110"_b, "ld2r_asisdlsop_r2_i"},
      {"1111111"_b, "ld4r_asisdlsop_r4_i"},
    },
  },

  { "_hhxpjz",
    {18},
    { {"0"_b, "ld2_asisdlso_b2_2b"},
    },
  },

  { "_hhxpyt",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx10"_b, "stlur_b_ldapstl_simd"},
      {"001xx10"_b, "ldapur_b_ldapstl_simd"},
      {"010xx10"_b, "stlur_q_ldapstl_simd"},
      {"011xx10"_b, "ldapur_q_ldapstl_simd"},
      {"100xx10"_b, "stlur_h_ldapstl_simd"},
      {"101xx10"_b, "ldapur_h_ldapstl_simd"},
      {"x000001"_b, "cpyp_cpy_memcms"},
      {"x000101"_b, "cpypwt_cpy_memcms"},
      {"x001001"_b, "cpyprt_cpy_memcms"},
      {"x001101"_b, "cpypt_cpy_memcms"},
      {"x010001"_b, "cpym_cpy_memcms"},
      {"x010101"_b, "cpymwt_cpy_memcms"},
      {"x011001"_b, "cpymrt_cpy_memcms"},
      {"x011101"_b, "cpymt_cpy_memcms"},
      {"x100001"_b, "cpye_cpy_memcms"},
      {"x100101"_b, "cpyewt_cpy_memcms"},
      {"x101001"_b, "cpyert_cpy_memcms"},
      {"x101101"_b, "cpyet_cpy_memcms"},
      {"x110001"_b, "setgp_set_memcms"},
      {"x110101"_b, "setgpt_set_memcms"},
      {"x111001"_b, "setgpn_set_memcms"},
      {"x111101"_b, "setgptn_set_memcms"},
    },
  },

  { "_hjplhs",
    {20, 19, 18, 17, 16, 13, 12},
    { {"1111100"_b, "ldaprb_32l_memop"},
    },
  },

  { "_hjqryy",
    {11, 10, 9, 8, 7, 6},
    { {"000000"_b, "wfit_only_systeminstrswithreg"},
    },
  },

  { "_hjtsgj",
    {23},
    { {"0"_b, "_pnkxsr"},
    },
  },

  { "_hjvkkq",
    {18},
    { {"0"_b, "ld4_asisdlsep_r4_r"},
      {"1"_b, "ld4_asisdlsep_i4_i"},
    },
  },

  { "_hkgzsh",
    {13, 12, 11, 10},
    { {"1111"_b, "_qvzvmq"},
    },
  },

  { "_hkjjsr",
    {12},
    { {"0"_b, "ld1_asisdlsop_dx1_r1d"},
    },
  },

  { "_hkpjqm",
    {30},
    { {"1"_b, "_qgyppr"},
    },
  },

  { "_hkxlsm",
    {18},
    { {"0"_b, "st4_asisdlsop_hx4_r4h"},
      {"1"_b, "st4_asisdlsop_h4_i4h"},
    },
  },

  { "_hkxzqg",
    {2, 1},
    { {"00"_b, "br_64_branch_reg"},
    },
  },

  { "_hljttg",
    {12},
    { {"0"_b, "ld2_asisdlsop_dx2_r2d"},
    },
  },

  { "_hlljqz",
    {30, 23, 22, 11, 10},
    { {"00000"_b, "stur_s_ldst_unscaled"},
      {"00001"_b, "str_s_ldst_immpost"},
      {"00011"_b, "str_s_ldst_immpre"},
      {"00100"_b, "ldur_s_ldst_unscaled"},
      {"00101"_b, "ldr_s_ldst_immpost"},
      {"00111"_b, "ldr_s_ldst_immpre"},
      {"10000"_b, "stur_d_ldst_unscaled"},
      {"10001"_b, "str_d_ldst_immpost"},
      {"10011"_b, "str_d_ldst_immpre"},
      {"10100"_b, "ldur_d_ldst_unscaled"},
      {"10101"_b, "ldr_d_ldst_immpost"},
      {"10111"_b, "ldr_d_ldst_immpre"},
    },
  },

  { "_hlqvmm",
    {20, 19, 18, 17, 16, 13, 12, 9, 8, 7, 6, 5},
    { {"000010011111"_b, "xpaci_64z_dp_1src"},
    },
  },

  { "_hlxmpy",
    {13, 12, 11, 10},
    { {"0000"_b, "umlal_asimddiff_l"},
      {"0001"_b, "sub_asimdsame_only"},
      {"0010"_b, "_hytrnv"},
      {"0011"_b, "cmeq_asimdsame_only"},
      {"0101"_b, "mls_asimdsame_only"},
      {"0110"_b, "_vjhrzl"},
      {"0111"_b, "pmul_asimdsame_only"},
      {"1000"_b, "umlsl_asimddiff_l"},
      {"1001"_b, "umaxp_asimdsame_only"},
      {"1010"_b, "_zpjzst"},
      {"1011"_b, "uminp_asimdsame_only"},
      {"1101"_b, "sqrdmulh_asimdsame_only"},
      {"1110"_b, "_jztlrz"},
    },
  },

  { "_hlypvy",
    {30, 23, 22},
    { {"000"_b, "smaddl_64wa_dp_3src"},
      {"010"_b, "umaddl_64wa_dp_3src"},
    },
  },

  { "_hmgzjl",
    {18},
    { {"0"_b, "st3_asisdlso_h3_3h"},
    },
  },

  { "_hmjrmm",
    {30, 23, 22, 20, 19, 18},
    { {"00xxxx"_b, "add_32_addsub_imm"},
      {"011000"_b, "smax_32_minmax_imm"},
      {"011001"_b, "umax_32u_minmax_imm"},
      {"011010"_b, "smin_32_minmax_imm"},
      {"011011"_b, "umin_32u_minmax_imm"},
      {"10xxxx"_b, "sub_32_addsub_imm"},
    },
  },

  { "_hmpzzg",
    {22, 20, 19, 18, 17, 16},
    { {"111000"_b, "fcmle_asisdmiscfp16_fz"},
      {"111001"_b, "frsqrte_asisdmiscfp16_r"},
      {"x00000"_b, "fcmle_asisdmisc_fz"},
      {"x00001"_b, "frsqrte_asisdmisc_r"},
    },
  },

  { "_hmsgpj",
    {13, 12, 10},
    { {"000"_b, "_hthxvr"},
      {"100"_b, "ptrue_p_s"},
      {"101"_b, "_kkvrzq"},
      {"110"_b, "_xxjrsy"},
    },
  },

  { "_hmtmlq",
    {4},
    { {"0"_b, "nor_p_p_pp_z"},
      {"1"_b, "nand_p_p_pp_z"},
    },
  },

  { "_hnjrmp",
    {4},
    { {"0"_b, "cmplo_p_p_zi"},
      {"1"_b, "cmpls_p_p_zi"},
    },
  },

  { "_hnkyxy",
    {18, 17, 16},
    { {"011"_b, "_ykpgyh"},
    },
  },

  { "_hnsvjh",
    {19},
    { {"0"_b, "_ntjrlg"},
      {"1"_b, "sysl_rc_systeminstrs"},
    },
  },

  { "_hpmvzr",
    {11, 10, 9, 8, 7, 6},
    { {"000000"_b, "yield_hi_hints"},
      {"000001"_b, "wfi_hi_hints"},
      {"000010"_b, "sevl_hi_hints"},
      {"000011"_b, "xpaclri_hi_hints"},
      {"001000"_b, "psb_hc_hints"},
      {"001001"_b, "gcsb_hd_hints"},
      {"001100"_b, "paciasp_hi_hints"},
      {"001101"_b, "pacibsp_hi_hints"},
      {"001110"_b, "autiasp_hi_hints"},
      {"001111"_b, "autibsp_hi_hints"},
      {"0x01xx"_b, "hint_hm_hints"},
      {"0x101x"_b, "hint_hm_hints"},
      {"10x0xx"_b, "hint_hm_hints"},
      {"10x1xx"_b, "hint_hm_hints"},
      {"1101xx"_b, "hint_hm_hints"},
      {"11101x"_b, "hint_hm_hints"},
      {"x100xx"_b, "hint_hm_hints"},
      {"x1100x"_b, "hint_hm_hints"},
      {"x111xx"_b, "hint_hm_hints"},
    },
  },

  { "_hpqkhv",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldaxp_lp64_ldstexcl"},
    },
  },

  { "_hptkrj",
    {30, 22, 13, 12, 11, 10},
    { {"000001"_b, "rmif_only_rmif"},
      {"01xx00"_b, "ccmn_64_condcmp_reg"},
      {"01xx10"_b, "ccmn_64_condcmp_imm"},
      {"11xx00"_b, "ccmp_64_condcmp_reg"},
      {"11xx10"_b, "ccmp_64_condcmp_imm"},
    },
  },

  { "_hqkhsy",
    {12},
    { {"0"_b, "st3_asisdlsop_dx3_r3d"},
    },
  },

  { "_hqkljv",
    {30, 23, 22},
    { {"000"_b, "and_32_log_imm"},
      {"010"_b, "movn_32_movewide"},
      {"100"_b, "eor_32_log_imm"},
      {"110"_b, "movz_32_movewide"},
    },
  },

  { "_hqnsvg",
    {30},
    { {"0"_b, "add_64_addsub_shift"},
      {"1"_b, "sub_64_addsub_shift"},
    },
  },

  { "_hqvhjp",
    {22},
    { {"0"_b, "str_32_ldst_regoff"},
      {"1"_b, "ldr_32_ldst_regoff"},
    },
  },

  { "_hrmsnk",
    {9, 8, 7, 6, 5, 2, 1},
    { {"1111111"_b, "eretaa_64e_branch_reg"},
    },
  },

  { "_hrpkqg",
    {18, 17, 12},
    { {"000"_b, "st4_asisdlso_d4_4d"},
    },
  },

  { "_hrxtnj",
    {30, 23, 22, 13, 12, 11, 10},
    { {"1010000"_b, "sm3partw1_vvv4_cryptosha512_3"},
      {"1010001"_b, "sm3partw2_vvv4_cryptosha512_3"},
      {"1010010"_b, "sm4ekey_vvv4_cryptosha512_3"},
    },
  },

  { "_hrxyts",
    {23, 22, 20, 19, 18, 13},
    { {"00000x"_b, "orr_z_zi"},
      {"01000x"_b, "eor_z_zi"},
      {"10000x"_b, "and_z_zi"},
      {"11000x"_b, "dupm_z_i"},
      {"xx1xx0"_b, "fcpy_z_p_i"},
    },
  },

  { "_hrymnk",
    {18},
    { {"0"_b, "st1_asisdlso_h1_1h"},
    },
  },

  { "_hspyhv",
    {13, 12},
    { {"10"_b, "umax_64_dp_2src"},
    },
  },

  { "_hsrkqt",
    {13, 12, 11, 10},
    { {"0000"_b, "addhn_asimddiff_n"},
      {"0001"_b, "sshl_asimdsame_only"},
      {"0010"_b, "_qtgrzv"},
      {"0011"_b, "sqshl_asimdsame_only"},
      {"0100"_b, "sabal_asimddiff_l"},
      {"0101"_b, "srshl_asimdsame_only"},
      {"0110"_b, "_vhkpvn"},
      {"0111"_b, "sqrshl_asimdsame_only"},
      {"1000"_b, "subhn_asimddiff_n"},
      {"1001"_b, "smax_asimdsame_only"},
      {"1010"_b, "_rgztgm"},
      {"1011"_b, "smin_asimdsame_only"},
      {"1100"_b, "sabdl_asimddiff_l"},
      {"1101"_b, "sabd_asimdsame_only"},
      {"1110"_b, "_grmpht"},
      {"1111"_b, "saba_asimdsame_only"},
    },
  },

  { "_hthxvr",
    {23, 22, 9},
    { {"010"_b, "pfirst_p_p_p"},
    },
  },

  { "_htjmmx",
    {30},
    { {"0"_b, "tbnz_only_testbranch"},
    },
  },

  { "_htkpks",
    {30, 23, 22},
    { {"000"_b, "add_32_addsub_ext"},
      {"100"_b, "sub_32_addsub_ext"},
    },
  },

  { "_htplsj",
    {4},
    { {"0"_b, "cmpeq_p_p_zz"},
      {"1"_b, "cmpne_p_p_zz"},
    },
  },

  { "_htqpks",
    {30, 20, 19, 18, 17, 16, 13},
    { {"000000x"_b, "add_z_zi"},
      {"000001x"_b, "sub_z_zi"},
      {"000011x"_b, "subr_z_zi"},
      {"000100x"_b, "sqadd_z_zi"},
      {"000101x"_b, "uqadd_z_zi"},
      {"000110x"_b, "sqsub_z_zi"},
      {"000111x"_b, "uqsub_z_zi"},
      {"0010000"_b, "smax_z_zi"},
      {"0010010"_b, "umax_z_zi"},
      {"0010100"_b, "smin_z_zi"},
      {"0010110"_b, "umin_z_zi"},
      {"0100000"_b, "mul_z_zi"},
      {"011000x"_b, "dup_z_i"},
      {"0110010"_b, "fdup_z_i"},
      {"1xxxxx0"_b, "fnmad_z_p_zzz"},
      {"1xxxxx1"_b, "fnmsb_z_p_zzz"},
    },
  },

  { "_htrtzz",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx10"_b, "stlur_b_ldapstl_simd"},
      {"001xx10"_b, "ldapur_b_ldapstl_simd"},
      {"010xx10"_b, "stlur_q_ldapstl_simd"},
      {"011xx10"_b, "ldapur_q_ldapstl_simd"},
      {"100xx10"_b, "stlur_h_ldapstl_simd"},
      {"101xx10"_b, "ldapur_h_ldapstl_simd"},
      {"x000001"_b, "cpypwn_cpy_memcms"},
      {"x000101"_b, "cpypwtwn_cpy_memcms"},
      {"x001001"_b, "cpyprtwn_cpy_memcms"},
      {"x001101"_b, "cpyptwn_cpy_memcms"},
      {"x010001"_b, "cpymwn_cpy_memcms"},
      {"x010101"_b, "cpymwtwn_cpy_memcms"},
      {"x011001"_b, "cpymrtwn_cpy_memcms"},
      {"x011101"_b, "cpymtwn_cpy_memcms"},
      {"x100001"_b, "cpyewn_cpy_memcms"},
      {"x100101"_b, "cpyewtwn_cpy_memcms"},
      {"x101001"_b, "cpyertwn_cpy_memcms"},
      {"x101101"_b, "cpyetwn_cpy_memcms"},
      {"x110001"_b, "setgm_set_memcms"},
      {"x110101"_b, "setgmt_set_memcms"},
      {"x111001"_b, "setgmn_set_memcms"},
      {"x111101"_b, "setgmtn_set_memcms"},
    },
  },

  { "_htsjxj",
    {23, 22, 13, 12, 11, 10},
    { {"001010"_b, "pmullb_z_zz_q"},
      {"001011"_b, "pmullt_z_zz_q"},
      {"101010"_b, "pmullb_z_zz"},
      {"101011"_b, "pmullt_z_zz"},
      {"x11010"_b, "pmullb_z_zz"},
      {"x11011"_b, "pmullt_z_zz"},
      {"xx0000"_b, "saddwb_z_zz"},
      {"xx0001"_b, "saddwt_z_zz"},
      {"xx0010"_b, "uaddwb_z_zz"},
      {"xx0011"_b, "uaddwt_z_zz"},
      {"xx0100"_b, "ssubwb_z_zz"},
      {"xx0101"_b, "ssubwt_z_zz"},
      {"xx0110"_b, "usubwb_z_zz"},
      {"xx0111"_b, "usubwt_z_zz"},
      {"xx1000"_b, "sqdmullb_z_zz"},
      {"xx1001"_b, "sqdmullt_z_zz"},
      {"xx1100"_b, "smullb_z_zz"},
      {"xx1101"_b, "smullt_z_zz"},
      {"xx1110"_b, "umullb_z_zz"},
      {"xx1111"_b, "umullt_z_zz"},
    },
  },

  { "_hvhrsq",
    {30, 23, 22},
    { {"000"_b, "str_32_ldst_pos"},
      {"001"_b, "ldr_32_ldst_pos"},
      {"010"_b, "ldrsw_64_ldst_pos"},
      {"100"_b, "str_64_ldst_pos"},
      {"101"_b, "ldr_64_ldst_pos"},
      {"110"_b, "prfm_p_ldst_pos"},
    },
  },

  { "_hvmyjz",
    {13, 12},
    { {"00"_b, "subps_64s_dp_2src"},
    },
  },

  { "_hvnhmh",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx00"_b, "stlurb_32_ldapstl_unscaled"},
      {"001xx00"_b, "ldapurb_32_ldapstl_unscaled"},
      {"010xx00"_b, "ldapursb_64_ldapstl_unscaled"},
      {"011xx00"_b, "ldapursb_32_ldapstl_unscaled"},
      {"100xx00"_b, "stlurh_32_ldapstl_unscaled"},
      {"101xx00"_b, "ldapurh_32_ldapstl_unscaled"},
      {"110xx00"_b, "ldapursh_64_ldapstl_unscaled"},
      {"111xx00"_b, "ldapursh_32_ldapstl_unscaled"},
      {"x000001"_b, "cpyfpwn_cpy_memcms"},
      {"x000101"_b, "cpyfpwtwn_cpy_memcms"},
      {"x001001"_b, "cpyfprtwn_cpy_memcms"},
      {"x001101"_b, "cpyfptwn_cpy_memcms"},
      {"x010001"_b, "cpyfmwn_cpy_memcms"},
      {"x010101"_b, "cpyfmwtwn_cpy_memcms"},
      {"x011001"_b, "cpyfmrtwn_cpy_memcms"},
      {"x011101"_b, "cpyfmtwn_cpy_memcms"},
      {"x100001"_b, "cpyfewn_cpy_memcms"},
      {"x100101"_b, "cpyfewtwn_cpy_memcms"},
      {"x101001"_b, "cpyfertwn_cpy_memcms"},
      {"x101101"_b, "cpyfetwn_cpy_memcms"},
      {"x110001"_b, "setm_set_memcms"},
      {"x110101"_b, "setmt_set_memcms"},
      {"x111001"_b, "setmn_set_memcms"},
      {"x111101"_b, "setmtn_set_memcms"},
    },
  },

  { "_hvrjyt",
    {30, 23, 22},
    { {"000"_b, "sbfm_32m_bitfield"},
      {"010"_b, "extr_32_extract"},
      {"100"_b, "ubfm_32m_bitfield"},
    },
  },

  { "_hvyjnk",
    {11},
    { {"0"_b, "sqrdmulh_z_zzi_h"},
    },
  },

  { "_hxgngr",
    {23, 22, 13},
    { {"100"_b, "fmlsl_asimdelem_lh"},
      {"xx1"_b, "smlsl_asimdelem_l"},
    },
  },

  { "_hxlznn",
    {30, 23, 22, 13},
    { {"0000"_b, "ld1sh_z_p_br_s32"},
      {"0001"_b, "ldff1sh_z_p_br_s32"},
      {"0010"_b, "ld1w_z_p_br_u64"},
      {"0011"_b, "ldff1w_z_p_br_u64"},
      {"0100"_b, "ld1sb_z_p_br_s32"},
      {"0101"_b, "ldff1sb_z_p_br_s32"},
      {"0110"_b, "ld1d_z_p_br_u64"},
      {"0111"_b, "ldff1d_z_p_br_u64"},
      {"1001"_b, "st2w_z_p_br_contiguous"},
      {"1010"_b, "st1w_z_p_br"},
      {"1011"_b, "st4w_z_p_br_contiguous"},
      {"1100"_b, "str_z_bi"},
      {"1101"_b, "st2d_z_p_br_contiguous"},
      {"1110"_b, "st1d_z_p_br"},
      {"1111"_b, "st4d_z_p_br_contiguous"},
    },
  },

  { "_hxrnns",
    {23, 22, 13, 12},
    { {"0000"_b, "fmul_s_floatdp2"},
      {"0001"_b, "fdiv_s_floatdp2"},
      {"0010"_b, "fadd_s_floatdp2"},
      {"0011"_b, "fsub_s_floatdp2"},
      {"0100"_b, "fmul_d_floatdp2"},
      {"0101"_b, "fdiv_d_floatdp2"},
      {"0110"_b, "fadd_d_floatdp2"},
      {"0111"_b, "fsub_d_floatdp2"},
      {"1100"_b, "fmul_h_floatdp2"},
      {"1101"_b, "fdiv_h_floatdp2"},
      {"1110"_b, "fadd_h_floatdp2"},
      {"1111"_b, "fsub_h_floatdp2"},
    },
  },

  { "_hxxqks",
    {23},
    { {"0"_b, "fmla_asimdsame_only"},
      {"1"_b, "fmls_asimdsame_only"},
    },
  },

  { "_hxxxyy",
    {13, 12},
    { {"00"_b, "cpyfm_cpy_memcms"},
      {"01"_b, "cpyfmwt_cpy_memcms"},
      {"10"_b, "cpyfmrt_cpy_memcms"},
      {"11"_b, "cpyfmt_cpy_memcms"},
    },
  },

  { "_hykhmt",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "saddv_r_p_z"},
      {"00001"_b, "uaddv_r_p_z"},
      {"01000"_b, "smaxv_r_p_z"},
      {"01001"_b, "umaxv_r_p_z"},
      {"01010"_b, "sminv_r_p_z"},
      {"01011"_b, "uminv_r_p_z"},
      {"1000x"_b, "movprfx_z_p_z"},
      {"11000"_b, "orv_r_p_z"},
      {"11001"_b, "eorv_r_p_z"},
      {"11010"_b, "andv_r_p_z"},
    },
  },

  { "_hynprk",
    {18},
    { {"0"_b, "st2_asisdlso_h2_2h"},
    },
  },

  { "_hyskth",
    {22},
    { {"0"_b, "str_64_ldst_regoff"},
      {"1"_b, "ldr_64_ldst_regoff"},
    },
  },

  { "_hytrnv",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "frinta_asimdmiscfp16_r"},
      {"0x00001"_b, "frinta_asimdmisc_r"},
      {"xx00000"_b, "cmge_asimdmisc_z"},
    },
  },

  { "_hzkglv",
    {30, 23, 22, 13},
    { {"0000"_b, "ld1b_z_p_br_u8"},
      {"0001"_b, "ldff1b_z_p_br_u8"},
      {"0010"_b, "ld1b_z_p_br_u32"},
      {"0011"_b, "ldff1b_z_p_br_u32"},
      {"0100"_b, "ld1sw_z_p_br_s64"},
      {"0101"_b, "ldff1sw_z_p_br_s64"},
      {"0110"_b, "ld1h_z_p_br_u32"},
      {"0111"_b, "ldff1h_z_p_br_u32"},
      {"1001"_b, "stnt1b_z_p_br_contiguous"},
      {"1011"_b, "st3b_z_p_br_contiguous"},
      {"10x0"_b, "st1b_z_p_br"},
      {"1101"_b, "stnt1h_z_p_br_contiguous"},
      {"1111"_b, "st3h_z_p_br_contiguous"},
      {"11x0"_b, "st1h_z_p_br"},
    },
  },

  { "_hzkxht",
    {22, 20},
    { {"00"_b, "_zrxhzq"},
      {"01"_b, "msr_sr_systemmove"},
      {"10"_b, "_krllsy"},
      {"11"_b, "msrr_sr_systemmovepr"},
    },
  },

  { "_hzsxkp",
    {30, 13},
    { {"00"_b, "_jlrrlt"},
      {"01"_b, "_jrlynj"},
      {"10"_b, "_ghpxms"},
      {"11"_b, "_nyjtng"},
    },
  },

  { "_jggxjz",
    {13, 12},
    { {"00"_b, "cmtst_asisdsame_only"},
    },
  },

  { "_jgklkt",
    {30},
    { {"0"_b, "ldrsw_64_loadlit"},
      {"1"_b, "prfm_p_loadlit"},
    },
  },

  { "_jgmlpk",
    {4},
    { {"0"_b, "match_p_p_zz"},
      {"1"_b, "nmatch_p_p_zz"},
    },
  },

  { "_jgsryt",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldaxrh_lr32_ldstexcl"},
    },
  },

  { "_jgxqzr",
    {13, 12, 11, 10},
    { {"0000"_b, "_xzjvkv"},
      {"0001"_b, "_nqjtqn"},
      {"0011"_b, "_qzmrnj"},
      {"0100"_b, "_xptsns"},
      {"0101"_b, "_qpgxxr"},
      {"0110"_b, "uzp1_asimdperm_only"},
      {"0111"_b, "_rsnvnr"},
      {"1000"_b, "_yszlqj"},
      {"1001"_b, "_lzvxxj"},
      {"1010"_b, "trn1_asimdperm_only"},
      {"1011"_b, "_zmrhxx"},
      {"1100"_b, "_skytvx"},
      {"1101"_b, "_smptxh"},
      {"1110"_b, "zip1_asimdperm_only"},
      {"1111"_b, "_rjvgkl"},
    },
  },

  { "_jgyhrh",
    {4},
    { {"0"_b, "cmplo_p_p_zi"},
      {"1"_b, "cmpls_p_p_zi"},
    },
  },

  { "_jhkkgv",
    {10},
    { {"0"_b, "_qvgtlh"},
    },
  },

  { "_jhllmn",
    {4},
    { {"0"_b, "cmpge_p_p_zz"},
      {"1"_b, "cmpgt_p_p_zz"},
    },
  },

  { "_jhltlz",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldxr_lr64_ldstexcl"},
    },
  },

  { "_jjgpxz",
    {9, 8, 7, 6, 5},
    { {"00000"_b, "fmov_h_floatimm"},
    },
  },

  { "_jjnvrv",
    {20, 19, 18, 17, 16, 13, 12, 4, 3, 2, 1, 0},
    { {"000000001101"_b, "setf8_only_setf"},
    },
  },

  { "_jkkqvy",
    {22, 20, 11},
    { {"100"_b, "uqinch_z_zs"},
      {"101"_b, "uqdech_z_zs"},
      {"110"_b, "dech_z_zs"},
    },
  },

  { "_jkvsxy",
    {30, 23, 22, 13, 12, 11, 10},
    { {"0000010"_b, "rcwcas_c64_rcwcomswap"},
      {"0000011"_b, "rcwcasp_c64_rcwcomswappr"},
      {"0000100"_b, "ldclrp_128_memop_128"},
      {"0001100"_b, "ldsetp_128_memop_128"},
      {"0010010"_b, "rcwcasl_c64_rcwcomswap"},
      {"0010011"_b, "rcwcaspl_c64_rcwcomswappr"},
      {"0010100"_b, "ldclrpl_128_memop_128"},
      {"0011100"_b, "ldsetpl_128_memop_128"},
      {"0100010"_b, "rcwcasa_c64_rcwcomswap"},
      {"0100011"_b, "rcwcaspa_c64_rcwcomswappr"},
      {"0100100"_b, "ldclrpa_128_memop_128"},
      {"0101100"_b, "ldsetpa_128_memop_128"},
      {"0110010"_b, "rcwcasal_c64_rcwcomswap"},
      {"0110011"_b, "rcwcaspal_c64_rcwcomswappr"},
      {"0110100"_b, "ldclrpal_128_memop_128"},
      {"0111100"_b, "ldsetpal_128_memop_128"},
      {"1000010"_b, "rcwscas_c64_rcwcomswap"},
      {"1000011"_b, "rcwscasp_c64_rcwcomswappr"},
      {"1010010"_b, "rcwscasl_c64_rcwcomswap"},
      {"1010011"_b, "rcwscaspl_c64_rcwcomswappr"},
      {"1100010"_b, "rcwscasa_c64_rcwcomswap"},
      {"1100011"_b, "rcwscaspa_c64_rcwcomswappr"},
      {"1110010"_b, "rcwscasal_c64_rcwcomswap"},
      {"1110011"_b, "rcwscaspal_c64_rcwcomswappr"},
    },
  },

  { "_jkvvtp",
    {30, 23, 22},
    { {"100"_b, "bcax_vvv16_crypto4"},
    },
  },

  { "_jkxyvn",
    {23},
    { {"0"_b, "fadd_asimdsame_only"},
      {"1"_b, "fsub_asimdsame_only"},
    },
  },

  { "_jlnjsy",
    {23, 22, 20, 19, 18, 17, 16, 13, 12, 11},
    { {"0011111001"_b, "_ssjnph"},
    },
  },

  { "_jlrrlt",
    {11, 10, 4},
    { {"000"_b, "whilege_p_p_rr"},
      {"001"_b, "whilegt_p_p_rr"},
      {"010"_b, "whilelt_p_p_rr"},
      {"011"_b, "whilele_p_p_rr"},
      {"100"_b, "whilehs_p_p_rr"},
      {"101"_b, "whilehi_p_p_rr"},
      {"110"_b, "whilelo_p_p_rr"},
      {"111"_b, "whilels_p_p_rr"},
    },
  },

  { "_jmvgsp",
    {22, 20, 11},
    { {"100"_b, "sqinch_z_zs"},
      {"101"_b, "sqdech_z_zs"},
      {"110"_b, "inch_z_zs"},
    },
  },

  { "_jmxstz",
    {13, 12, 11, 10},
    { {"0000"_b, "sqdecp_z_p_z"},
      {"0010"_b, "sqdecp_r_p_r_sx"},
      {"0011"_b, "sqdecp_r_p_r_x"},
    },
  },

  { "_jnktqs",
    {18, 17},
    { {"00"_b, "ld1_asisdlso_s1_1s"},
    },
  },

  { "_jnnmjk",
    {23, 22, 20, 19, 16, 13, 12},
    { {"0111100"_b, "fcvtas_asisdmiscfp16_r"},
      {"0111101"_b, "scvtf_asisdmiscfp16_r"},
      {"0x00100"_b, "fcvtas_asisdmisc_r"},
      {"0x00101"_b, "scvtf_asisdmisc_r"},
      {"0x10000"_b, "fmaxnmp_asisdpair_only_h"},
      {"0x10001"_b, "faddp_asisdpair_only_h"},
      {"0x10011"_b, "fmaxp_asisdpair_only_h"},
      {"1111000"_b, "fcmgt_asisdmiscfp16_fz"},
      {"1111001"_b, "fcmeq_asisdmiscfp16_fz"},
      {"1111010"_b, "fcmlt_asisdmiscfp16_fz"},
      {"1111101"_b, "frecpe_asisdmiscfp16_r"},
      {"1111111"_b, "frecpx_asisdmiscfp16_r"},
      {"1x00000"_b, "fcmgt_asisdmisc_fz"},
      {"1x00001"_b, "fcmeq_asisdmisc_fz"},
      {"1x00010"_b, "fcmlt_asisdmisc_fz"},
      {"1x00101"_b, "frecpe_asisdmisc_r"},
      {"1x00111"_b, "frecpx_asisdmisc_r"},
      {"1x10000"_b, "fminnmp_asisdpair_only_h"},
      {"1x10011"_b, "fminp_asisdpair_only_h"},
    },
  },

  { "_jpvmkz",
    {18, 17},
    { {"00"_b, "_jnnmjk"},
    },
  },

  { "_jqhvhn",
    {30, 23, 11, 10},
    { {"0000"_b, "_ygtpyl"},
      {"0010"_b, "_hqvhjp"},
      {"0100"_b, "_xkylhh"},
      {"0110"_b, "_mnxgml"},
      {"1000"_b, "_qyyrqq"},
      {"1001"_b, "ldraa_64_ldst_pac"},
      {"1010"_b, "_kpsnsk"},
      {"1011"_b, "ldraa_64w_ldst_pac"},
      {"1100"_b, "_tyzpxk"},
      {"1101"_b, "ldrab_64_ldst_pac"},
      {"1111"_b, "ldrab_64w_ldst_pac"},
    },
  },

  { "_jqlgts",
    {30, 23, 22},
    { {"000"_b, "str_s_ldst_pos"},
      {"001"_b, "ldr_s_ldst_pos"},
      {"100"_b, "str_d_ldst_pos"},
      {"101"_b, "ldr_d_ldst_pos"},
    },
  },

  { "_jqrmyp",
    {20, 19, 18, 17, 16, 13},
    { {"000000"_b, "fabs_h_floatdp1"},
      {"000010"_b, "fsqrt_h_floatdp1"},
      {"000100"_b, "fcvt_dh_floatdp1"},
      {"001000"_b, "frintp_h_floatdp1"},
      {"001010"_b, "frintz_h_floatdp1"},
      {"001110"_b, "frinti_h_floatdp1"},
    },
  },

  { "_jqsjtj",
    {18},
    { {"0"_b, "st2_asisdlse_r2"},
    },
  },

  { "_jqtksx",
    {30, 23, 22, 13, 12, 11, 10},
    { {"0000000"_b, "ldsmaxb_32_memop"},
      {"0000100"_b, "ldsminb_32_memop"},
      {"0000x10"_b, "strb_32b_ldst_regoff"},
      {"0001000"_b, "ldumaxb_32_memop"},
      {"0001100"_b, "lduminb_32_memop"},
      {"0001x10"_b, "strb_32bl_ldst_regoff"},
      {"0010000"_b, "ldsmaxlb_32_memop"},
      {"0010100"_b, "ldsminlb_32_memop"},
      {"0010x10"_b, "ldrb_32b_ldst_regoff"},
      {"0011000"_b, "ldumaxlb_32_memop"},
      {"0011100"_b, "lduminlb_32_memop"},
      {"0011x10"_b, "ldrb_32bl_ldst_regoff"},
      {"0100000"_b, "ldsmaxab_32_memop"},
      {"0100100"_b, "ldsminab_32_memop"},
      {"0100x10"_b, "ldrsb_64b_ldst_regoff"},
      {"0101000"_b, "ldumaxab_32_memop"},
      {"0101100"_b, "lduminab_32_memop"},
      {"0101x10"_b, "ldrsb_64bl_ldst_regoff"},
      {"0110000"_b, "ldsmaxalb_32_memop"},
      {"0110100"_b, "ldsminalb_32_memop"},
      {"0110x10"_b, "ldrsb_32b_ldst_regoff"},
      {"0111000"_b, "ldumaxalb_32_memop"},
      {"0111100"_b, "lduminalb_32_memop"},
      {"0111x10"_b, "ldrsb_32bl_ldst_regoff"},
      {"1000000"_b, "ldsmaxh_32_memop"},
      {"1000100"_b, "ldsminh_32_memop"},
      {"1001000"_b, "ldumaxh_32_memop"},
      {"1001100"_b, "lduminh_32_memop"},
      {"100xx10"_b, "strh_32_ldst_regoff"},
      {"1010000"_b, "ldsmaxlh_32_memop"},
      {"1010100"_b, "ldsminlh_32_memop"},
      {"1011000"_b, "ldumaxlh_32_memop"},
      {"1011100"_b, "lduminlh_32_memop"},
      {"101xx10"_b, "ldrh_32_ldst_regoff"},
      {"1100000"_b, "ldsmaxah_32_memop"},
      {"1100100"_b, "ldsminah_32_memop"},
      {"1101000"_b, "ldumaxah_32_memop"},
      {"1101100"_b, "lduminah_32_memop"},
      {"110xx10"_b, "ldrsh_64_ldst_regoff"},
      {"1110000"_b, "ldsmaxalh_32_memop"},
      {"1110100"_b, "ldsminalh_32_memop"},
      {"1111000"_b, "ldumaxalh_32_memop"},
      {"1111100"_b, "lduminalh_32_memop"},
      {"111xx10"_b, "ldrsh_32_ldst_regoff"},
    },
  },

  { "_jqvpqx",
    {23, 22},
    { {"00"_b, "fmlal_asimdsame_f"},
      {"10"_b, "fmlsl_asimdsame_f"},
    },
  },

  { "_jqxqql",
    {22, 20, 11},
    { {"000"_b, "uqincw_z_zs"},
      {"001"_b, "uqdecw_z_zs"},
      {"010"_b, "decw_z_zs"},
      {"100"_b, "uqincd_z_zs"},
      {"101"_b, "uqdecd_z_zs"},
      {"110"_b, "decd_z_zs"},
    },
  },

  { "_jrlynj",
    {11, 10},
    { {"00"_b, "_gzqvnk"},
    },
  },

  { "_jrnxzh",
    {12},
    { {"0"_b, "cmla_z_zzz"},
      {"1"_b, "sqrdcmlah_z_zzz"},
    },
  },

  { "_jrqxvn",
    {23, 22, 13, 12, 11, 10},
    { {"000000"_b, "tbl_asimdtbl_l3_3"},
      {"000100"_b, "tbx_asimdtbl_l3_3"},
      {"001000"_b, "tbl_asimdtbl_l4_4"},
      {"001100"_b, "tbx_asimdtbl_l4_4"},
      {"xx0110"_b, "uzp2_asimdperm_only"},
      {"xx1010"_b, "trn2_asimdperm_only"},
      {"xx1110"_b, "zip2_asimdperm_only"},
    },
  },

  { "_jrxtzg",
    {30, 23, 22, 11, 10},
    { {"10001"_b, "stg_64spost_ldsttags"},
      {"10010"_b, "stg_64soffset_ldsttags"},
      {"10011"_b, "stg_64spre_ldsttags"},
      {"10100"_b, "ldg_64loffset_ldsttags"},
      {"10101"_b, "stzg_64spost_ldsttags"},
      {"10110"_b, "stzg_64soffset_ldsttags"},
      {"10111"_b, "stzg_64spre_ldsttags"},
      {"11001"_b, "st2g_64spost_ldsttags"},
      {"11010"_b, "st2g_64soffset_ldsttags"},
      {"11011"_b, "st2g_64spre_ldsttags"},
      {"11101"_b, "stz2g_64spost_ldsttags"},
      {"11110"_b, "stz2g_64soffset_ldsttags"},
      {"11111"_b, "stz2g_64spre_ldsttags"},
    },
  },

  { "_jsqvtn",
    {23, 22, 11, 10},
    { {"0000"_b, "_lnsjqy"},
      {"0001"_b, "stg_64spost_ldsttags"},
      {"0010"_b, "stg_64soffset_ldsttags"},
      {"0011"_b, "stg_64spre_ldsttags"},
      {"0100"_b, "ldg_64loffset_ldsttags"},
      {"0101"_b, "stzg_64spost_ldsttags"},
      {"0110"_b, "stzg_64soffset_ldsttags"},
      {"0111"_b, "stzg_64spre_ldsttags"},
      {"1000"_b, "_myzhml"},
      {"1001"_b, "st2g_64spost_ldsttags"},
      {"1010"_b, "st2g_64soffset_ldsttags"},
      {"1011"_b, "st2g_64spre_ldsttags"},
      {"1100"_b, "_mjstgz"},
      {"1101"_b, "stz2g_64spost_ldsttags"},
      {"1110"_b, "stz2g_64soffset_ldsttags"},
      {"1111"_b, "stz2g_64spre_ldsttags"},
    },
  },

  { "_jvkxtj",
    {30, 23, 22},
    { {"000"_b, "stnp_q_ldstnapair_offs"},
      {"001"_b, "ldnp_q_ldstnapair_offs"},
      {"010"_b, "stp_q_ldstpair_post"},
      {"011"_b, "ldp_q_ldstpair_post"},
    },
  },

  { "_jvnsgt",
    {18},
    { {"0"_b, "ld4_asisdlsop_bx4_r4b"},
      {"1"_b, "ld4_asisdlsop_b4_i4b"},
    },
  },

  { "_jvpjsm",
    {20, 19, 18, 17, 16, 13, 12},
    { {"0000000"_b, "_xrnqyn"},
    },
  },

  { "_jxgpgg",
    {13, 12},
    { {"00"_b, "udiv_64_dp_2src"},
      {"10"_b, "asrv_64_dp_2src"},
    },
  },

  { "_jxgqqz",
    {30},
    { {"0"_b, "cbz_64_compbranch"},
    },
  },

  { "_jxltqm",
    {13, 12},
    { {"01"_b, "sqdmull_asisddiff_only"},
    },
  },

  { "_jxszhy",
    {23, 22, 11},
    { {"000"_b, "_rqhryp"},
    },
  },

  { "_jxyskn",
    {13, 12, 11, 10},
    { {"0000"_b, "uqincp_z_p_z"},
      {"0010"_b, "uqincp_r_p_r_uw"},
      {"0011"_b, "uqincp_r_p_r_x"},
    },
  },

  { "_jymnkk",
    {23, 22, 12, 11, 10},
    { {"01000"_b, "bfdot_z_zzzi"},
      {"100x0"_b, "fmlalb_z_zzzi_s"},
      {"100x1"_b, "fmlalt_z_zzzi_s"},
      {"110x0"_b, "bfmlalb_z_zzzi"},
      {"110x1"_b, "bfmlalt_z_zzzi"},
    },
  },

  { "_jyzhnh",
    {18},
    { {"0"_b, "st1_asisdlsop_hx1_r1h"},
      {"1"_b, "st1_asisdlsop_h1_i1h"},
    },
  },

  { "_jzjvtv",
    {19, 18, 17, 16, 4},
    { {"00000"_b, "brkbs_p_p_p_z"},
    },
  },

  { "_jzkqhn",
    {23, 22, 12, 11, 10},
    { {"10000"_b, "fmlslb_z_zzz"},
      {"10001"_b, "fmlslt_z_zzz"},
    },
  },

  { "_jztlrz",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "fcvtmu_asimdmiscfp16_r"},
      {"0x00001"_b, "fcvtmu_asimdmisc_r"},
      {"1111001"_b, "fcvtzu_asimdmiscfp16_r"},
      {"1x00001"_b, "fcvtzu_asimdmisc_r"},
      {"xx00000"_b, "neg_asimdmisc_r"},
    },
  },

  { "_jztspt",
    {18, 17},
    { {"00"_b, "st4_asisdlso_s4_4s"},
    },
  },

  { "_kgmqkh",
    {30, 23, 22, 13},
    { {"0000"_b, "ld1w_z_p_ai_s"},
      {"0001"_b, "ldff1w_z_p_ai_s"},
      {"0010"_b, "ld1rw_z_p_bi_u32"},
      {"0011"_b, "ld1rw_z_p_bi_u64"},
      {"0110"_b, "ld1rsb_z_p_bi_s16"},
      {"0111"_b, "ld1rd_z_p_bi_u64"},
      {"1000"_b, "ld1w_z_p_ai_d"},
      {"1001"_b, "ldff1w_z_p_ai_d"},
      {"1010"_b, "ld1w_z_p_bz_d_64_scaled"},
      {"1011"_b, "ldff1w_z_p_bz_d_64_scaled"},
      {"1100"_b, "ld1d_z_p_ai_d"},
      {"1101"_b, "ldff1d_z_p_ai_d"},
      {"1110"_b, "ld1d_z_p_bz_d_64_scaled"},
      {"1111"_b, "ldff1d_z_p_bz_d_64_scaled"},
    },
  },

  { "_kgpgly",
    {23, 22, 10},
    { {"100"_b, "smlslb_z_zzzi_s"},
      {"101"_b, "smlslt_z_zzzi_s"},
      {"110"_b, "smlslb_z_zzzi_d"},
      {"111"_b, "smlslt_z_zzzi_d"},
    },
  },

  { "_kgpsjz",
    {13, 12, 11, 10},
    { {"0000"_b, "saddl_asimddiff_l"},
      {"0001"_b, "shadd_asimdsame_only"},
      {"0010"_b, "_rkrlsy"},
      {"0011"_b, "sqadd_asimdsame_only"},
      {"0100"_b, "saddw_asimddiff_w"},
      {"0101"_b, "srhadd_asimdsame_only"},
      {"0110"_b, "_vypgrt"},
      {"0111"_b, "_xygvjp"},
      {"1000"_b, "ssubl_asimddiff_l"},
      {"1001"_b, "shsub_asimdsame_only"},
      {"1010"_b, "_pjhmvy"},
      {"1011"_b, "sqsub_asimdsame_only"},
      {"1100"_b, "ssubw_asimddiff_w"},
      {"1101"_b, "cmgt_asimdsame_only"},
      {"1110"_b, "_ygghnn"},
      {"1111"_b, "cmge_asimdsame_only"},
    },
  },

  { "_kgygky",
    {30, 23, 22},
    { {"000"_b, "sbfm_32m_bitfield"},
      {"100"_b, "ubfm_32m_bitfield"},
    },
  },

  { "_khjvqq",
    {22, 11},
    { {"00"_b, "sqrdmulh_z_zzi_s"},
      {"10"_b, "sqrdmulh_z_zzi_d"},
    },
  },

  { "_khrsgv",
    {22, 20, 19, 13, 12},
    { {"0x100"_b, "sri_asisdshf_r"},
      {"0x101"_b, "sli_asisdshf_r"},
      {"0x110"_b, "sqshlu_asisdshf_r"},
      {"0x111"_b, "uqshl_asisdshf_r"},
      {"10x00"_b, "sri_asisdshf_r"},
      {"10x01"_b, "sli_asisdshf_r"},
      {"10x10"_b, "sqshlu_asisdshf_r"},
      {"10x11"_b, "uqshl_asisdshf_r"},
      {"11100"_b, "sri_asisdshf_r"},
      {"11101"_b, "sli_asisdshf_r"},
      {"11110"_b, "sqshlu_asisdshf_r"},
      {"11111"_b, "uqshl_asisdshf_r"},
      {"x1000"_b, "sri_asisdshf_r"},
      {"x1001"_b, "sli_asisdshf_r"},
      {"x1010"_b, "sqshlu_asisdshf_r"},
      {"x1011"_b, "uqshl_asisdshf_r"},
    },
  },

  { "_khtsmx",
    {18},
    { {"0"_b, "ld4_asisdlsop_hx4_r4h"},
      {"1"_b, "ld4_asisdlsop_h4_i4h"},
    },
  },

  { "_khvvtr",
    {20, 19, 18, 17, 16, 13, 12},
    { {"0000000"_b, "rev16_32_dp_1src"},
      {"0000001"_b, "cls_32_dp_1src"},
    },
  },

  { "_kjpxvh",
    {20, 19, 18},
    { {"000"_b, "_yyrkmn"},
    },
  },

  { "_kjqynn",
    {4},
    { {"0"_b, "cmphs_p_p_zi"},
      {"1"_b, "cmphi_p_p_zi"},
    },
  },

  { "_kjsrkm",
    {18, 17, 16, 13, 12, 11, 10, 9, 8, 7, 4, 3, 2, 1, 0},
    { {"000000000011111"_b, "_zztypv"},
    },
  },

  { "_kkkltp",
    {30},
    { {"1"_b, "_sqkkqy"},
    },
  },

  { "_kkpxth",
    {18},
    { {"0"_b, "ld1_asisdlsop_bx1_r1b"},
      {"1"_b, "ld1_asisdlsop_b1_i1b"},
    },
  },

  { "_kktglv",
    {30, 13, 12},
    { {"000"_b, "_njvkjq"},
      {"001"_b, "_rpzykx"},
      {"010"_b, "_zzvxvh"},
      {"011"_b, "_yqxnzl"},
      {"100"_b, "_gxmnkl"},
      {"110"_b, "_lkxgjy"},
      {"111"_b, "_vjmklj"},
    },
  },

  { "_kktzst",
    {13, 12, 11, 10},
    { {"1111"_b, "frsqrts_asisdsamefp16_only"},
    },
  },

  { "_kkvrzq",
    {23, 22, 9, 8, 7, 6, 5},
    { {"0000000"_b, "pfalse_p"},
    },
  },

  { "_klrksl",
    {30, 23, 22, 19, 16},
    { {"10010"_b, "aesmc_b_cryptoaes"},
      {"x0x01"_b, "fcvtn_asimdmisc_n"},
      {"x1001"_b, "bfcvtn_asimdmisc_4s"},
      {"xxx00"_b, "sadalp_asimdmisc_p"},
    },
  },

  { "_klsmsv",
    {30, 23, 22, 10},
    { {"1001"_b, "ins_asimdins_iv_v"},
      {"x000"_b, "ext_asimdext_only"},
    },
  },

  { "_kltlmp",
    {22, 20, 19, 13, 12},
    { {"0x100"_b, "ushr_asisdshf_r"},
      {"0x101"_b, "usra_asisdshf_r"},
      {"0x110"_b, "urshr_asisdshf_r"},
      {"0x111"_b, "ursra_asisdshf_r"},
      {"10x00"_b, "ushr_asisdshf_r"},
      {"10x01"_b, "usra_asisdshf_r"},
      {"10x10"_b, "urshr_asisdshf_r"},
      {"10x11"_b, "ursra_asisdshf_r"},
      {"11100"_b, "ushr_asisdshf_r"},
      {"11101"_b, "usra_asisdshf_r"},
      {"11110"_b, "urshr_asisdshf_r"},
      {"11111"_b, "ursra_asisdshf_r"},
      {"x1000"_b, "ushr_asisdshf_r"},
      {"x1001"_b, "usra_asisdshf_r"},
      {"x1010"_b, "urshr_asisdshf_r"},
      {"x1011"_b, "ursra_asisdshf_r"},
    },
  },

  { "_klxxgx",
    {20, 19, 18, 17, 16, 13},
    { {"000000"_b, "fmov_s_floatdp1"},
      {"000010"_b, "fneg_s_floatdp1"},
      {"001000"_b, "frintn_s_floatdp1"},
      {"001010"_b, "frintm_s_floatdp1"},
      {"001100"_b, "frinta_s_floatdp1"},
      {"001110"_b, "frintx_s_floatdp1"},
      {"010000"_b, "frint32z_s_floatdp1"},
      {"010010"_b, "frint64z_s_floatdp1"},
    },
  },

  { "_kmqlmz",
    {18},
    { {"0"_b, "st1_asisdlso_b1_1b"},
    },
  },

  { "_knkjnz",
    {30, 23, 22, 20, 13},
    { {"00001"_b, "ld1sh_z_p_bi_s32"},
      {"00011"_b, "ldnf1sh_z_p_bi_s32"},
      {"00101"_b, "ld1w_z_p_bi_u64"},
      {"00111"_b, "ldnf1w_z_p_bi_u64"},
      {"01001"_b, "ld1sb_z_p_bi_s32"},
      {"01011"_b, "ldnf1sb_z_p_bi_s32"},
      {"01101"_b, "ld1d_z_p_bi_u64"},
      {"01111"_b, "ldnf1d_z_p_bi_u64"},
      {"100x0"_b, "st1w_z_p_bz_d_x32_scaled"},
      {"100x1"_b, "st1w_z_p_bz_d_64_scaled"},
      {"101x0"_b, "st1w_z_p_bz_s_x32_scaled"},
      {"101x1"_b, "st1w_z_p_ai_s"},
      {"110x0"_b, "st1d_z_p_bz_d_x32_scaled"},
      {"110x1"_b, "st1d_z_p_bz_d_64_scaled"},
    },
  },

  { "_knpjtt",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldxrh_lr32_ldstexcl"},
    },
  },

  { "_kpgghm",
    {22, 20, 19, 18, 17, 16, 13, 12},
    { {"01111100"_b, "ldapr_64l_memop"},
    },
  },

  { "_kpnlmr",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "clz_asimdmisc_r"},
      {"00001"_b, "uqxtn_asimdmisc_n"},
    },
  },

  { "_kppzvh",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx10"_b, "stlur_s_ldapstl_simd"},
      {"001xx10"_b, "ldapur_s_ldapstl_simd"},
      {"100xx10"_b, "stlur_d_ldapstl_simd"},
      {"101xx10"_b, "ldapur_d_ldapstl_simd"},
      {"x000001"_b, "cpyprn_cpy_memcms"},
      {"x000101"_b, "cpypwtrn_cpy_memcms"},
      {"x001001"_b, "cpyprtrn_cpy_memcms"},
      {"x001101"_b, "cpyptrn_cpy_memcms"},
      {"x010001"_b, "cpymrn_cpy_memcms"},
      {"x010101"_b, "cpymwtrn_cpy_memcms"},
      {"x011001"_b, "cpymrtrn_cpy_memcms"},
      {"x011101"_b, "cpymtrn_cpy_memcms"},
      {"x100001"_b, "cpyern_cpy_memcms"},
      {"x100101"_b, "cpyewtrn_cpy_memcms"},
      {"x101001"_b, "cpyertrn_cpy_memcms"},
      {"x101101"_b, "cpyetrn_cpy_memcms"},
      {"x110001"_b, "setge_set_memcms"},
      {"x110101"_b, "setget_set_memcms"},
      {"x111001"_b, "setgen_set_memcms"},
      {"x111101"_b, "setgetn_set_memcms"},
    },
  },

  { "_kpsnsk",
    {22},
    { {"0"_b, "str_64_ldst_regoff"},
      {"1"_b, "ldr_64_ldst_regoff"},
    },
  },

  { "_kqsqly",
    {18},
    { {"0"_b, "st1_asisdlsep_r2_r2"},
      {"1"_b, "st1_asisdlsep_i2_i2"},
    },
  },

  { "_kqstrr",
    {18, 17, 12},
    { {"000"_b, "st3_asisdlso_d3_3d"},
    },
  },

  { "_kqvljp",
    {18, 17, 16},
    { {"000"_b, "fabd_z_p_zz"},
      {"001"_b, "fscale_z_p_zz"},
      {"010"_b, "fmulx_z_p_zz"},
      {"100"_b, "fdivr_z_p_zz"},
      {"101"_b, "fdiv_z_p_zz"},
    },
  },

  { "_kqzmtr",
    {30, 23, 22, 20, 13},
    { {"00001"_b, "ld1b_z_p_bi_u16"},
      {"00011"_b, "ldnf1b_z_p_bi_u16"},
      {"00101"_b, "ld1b_z_p_bi_u64"},
      {"00111"_b, "ldnf1b_z_p_bi_u64"},
      {"01001"_b, "ld1h_z_p_bi_u16"},
      {"01011"_b, "ldnf1h_z_p_bi_u16"},
      {"01101"_b, "ld1h_z_p_bi_u64"},
      {"01111"_b, "ldnf1h_z_p_bi_u64"},
      {"101x1"_b, "st1b_z_p_ai_s"},
      {"110x0"_b, "st1h_z_p_bz_d_x32_scaled"},
      {"110x1"_b, "st1h_z_p_bz_d_64_scaled"},
      {"111x0"_b, "st1h_z_p_bz_s_x32_scaled"},
      {"111x1"_b, "st1h_z_p_ai_s"},
    },
  },

  { "_krllsy",
    {19},
    { {"1"_b, "sysp_cr_syspairinstrs"},
    },
  },

  { "_krtvhr",
    {12, 10},
    { {"00"_b, "_xvmxrg"},
      {"01"_b, "_mvvngm"},
      {"10"_b, "_mkyyng"},
      {"11"_b, "_vvzsmg"},
    },
  },

  { "_krvxxx",
    {12, 9, 8, 7, 6, 5},
    { {"100000"_b, "_skjqrx"},
    },
  },

  { "_ksgpqz",
    {30},
    { {"1"_b, "_trjmmn"},
    },
  },

  { "_kshtnj",
    {23, 22, 13, 12, 11, 10},
    { {"01x1x0"_b, "fcmla_asimdelem_c_h"},
      {"0x0001"_b, "sri_asimdshf_r"},
      {"0x0101"_b, "sli_asimdshf_r"},
      {"0x1001"_b, "sqshlu_asimdshf_r"},
      {"0x1101"_b, "uqshl_asimdshf_r"},
      {"10x1x0"_b, "fcmla_asimdelem_c_s"},
      {"xx00x0"_b, "mls_asimdelem_r"},
      {"xx10x0"_b, "umlsl_asimdelem_l"},
    },
  },

  { "_kskqmz",
    {20, 19, 18, 17, 16, 13, 12},
    { {"0000011"_b, "sqabs_asisdmisc_r"},
      {"0000100"_b, "sqxtn_asisdmisc_n"},
    },
  },

  { "_ksrkkn",
    {22},
    { {"0"_b, "str_32_ldst_regoff"},
      {"1"_b, "ldr_32_ldst_regoff"},
    },
  },

  { "_kssltr",
    {13, 12, 11, 10},
    { {"0000"_b, "smull_asimddiff_l"},
      {"0001"_b, "_pstgvl"},
      {"0010"_b, "_ztlysk"},
      {"0011"_b, "_hxxqks"},
      {"0100"_b, "sqdmull_asimddiff_l"},
      {"0101"_b, "_jkxyvn"},
      {"0110"_b, "_lvsrnj"},
      {"0111"_b, "_vvgnhm"},
      {"1000"_b, "pmull_asimddiff_l"},
      {"1001"_b, "_skqzyg"},
      {"1010"_b, "_szqlsn"},
      {"1011"_b, "_jqvpqx"},
      {"1101"_b, "_yyvjqv"},
      {"1110"_b, "_xlyppq"},
      {"1111"_b, "_mhljkp"},
    },
  },

  { "_ktngnm",
    {12, 10},
    { {"00"_b, "_hxgngr"},
      {"01"_b, "_ngkgsg"},
      {"10"_b, "_plrggq"},
      {"11"_b, "_kxztps"},
    },
  },

  { "_ktpxrr",
    {30, 23, 22, 13, 12, 11, 10},
    { {"0001111"_b, "casp_cp32_ldstexcl"},
      {"0011111"_b, "caspa_cp32_ldstexcl"},
      {"0101111"_b, "casb_c32_ldstexcl"},
      {"0111111"_b, "casab_c32_ldstexcl"},
      {"1001111"_b, "casp_cp64_ldstexcl"},
      {"1011111"_b, "caspa_cp64_ldstexcl"},
      {"1101111"_b, "cash_c32_ldstexcl"},
      {"1111111"_b, "casah_c32_ldstexcl"},
    },
  },

  { "_ktsgth",
    {23, 22},
    { {"00"_b, "fcsel_s_floatsel"},
      {"01"_b, "fcsel_d_floatsel"},
      {"11"_b, "fcsel_h_floatsel"},
    },
  },

  { "_ktyppm",
    {11, 10},
    { {"00"_b, "asr_z_zw"},
      {"01"_b, "lsr_z_zw"},
      {"11"_b, "lsl_z_zw"},
    },
  },

  { "_ktyrgy",
    {9, 8, 7, 6, 5},
    { {"00000"_b, "fmov_s_floatimm"},
    },
  },

  { "_kvnqhn",
    {22, 20, 11},
    { {"000"_b, "sqincw_r_rs_sx"},
      {"001"_b, "sqdecw_r_rs_sx"},
      {"010"_b, "sqincw_r_rs_x"},
      {"011"_b, "sqdecw_r_rs_x"},
      {"100"_b, "sqincd_r_rs_sx"},
      {"101"_b, "sqdecd_r_rs_sx"},
      {"110"_b, "sqincd_r_rs_x"},
      {"111"_b, "sqdecd_r_rs_x"},
    },
  },

  { "_kxhmlx",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "fcvtnu_asisdmiscfp16_r"},
      {"0x00001"_b, "fcvtnu_asisdmisc_r"},
      {"1111001"_b, "fcvtpu_asisdmiscfp16_r"},
      {"1x00001"_b, "fcvtpu_asisdmisc_r"},
    },
  },

  { "_kxmjsh",
    {20, 19, 18, 17, 16},
    { {"10000"_b, "fmaxp_asisdpair_only_sd"},
    },
  },

  { "_kxmxxm",
    {23},
    { {"0"_b, "fcmge_asimdsame_only"},
      {"1"_b, "fcmgt_asimdsame_only"},
    },
  },

  { "_kxpqhv",
    {30, 23, 22, 11, 10},
    { {"10001"_b, "stg_64spost_ldsttags"},
      {"10010"_b, "stg_64soffset_ldsttags"},
      {"10011"_b, "stg_64spre_ldsttags"},
      {"10100"_b, "ldg_64loffset_ldsttags"},
      {"10101"_b, "stzg_64spost_ldsttags"},
      {"10110"_b, "stzg_64soffset_ldsttags"},
      {"10111"_b, "stzg_64spre_ldsttags"},
      {"11001"_b, "st2g_64spost_ldsttags"},
      {"11010"_b, "st2g_64soffset_ldsttags"},
      {"11011"_b, "st2g_64spre_ldsttags"},
      {"11101"_b, "stz2g_64spost_ldsttags"},
      {"11110"_b, "stz2g_64soffset_ldsttags"},
      {"11111"_b, "stz2g_64spre_ldsttags"},
    },
  },

  { "_kxtqjh",
    {23, 22},
    { {"01"_b, "_mhnlsy"},
      {"10"_b, "xar_vvv2_crypto3_imm6"},
      {"11"_b, "_spxvlt"},
    },
  },

  { "_kxvvkq",
    {30, 23, 13},
    { {"000"_b, "ld1b_z_p_bz_s_x32_unscaled"},
      {"001"_b, "ldff1b_z_p_bz_s_x32_unscaled"},
      {"010"_b, "ld1h_z_p_bz_s_x32_unscaled"},
      {"011"_b, "ldff1h_z_p_bz_s_x32_unscaled"},
      {"100"_b, "ld1b_z_p_bz_d_x32_unscaled"},
      {"101"_b, "ldff1b_z_p_bz_d_x32_unscaled"},
      {"110"_b, "ld1h_z_p_bz_d_x32_unscaled"},
      {"111"_b, "ldff1h_z_p_bz_d_x32_unscaled"},
    },
  },

  { "_kxztps",
    {23, 22, 20, 19, 13, 11},
    { {"0000x0"_b, "orr_asimdimm_l_sl"},
      {"00x100"_b, "shl_asimdshf_r"},
      {"00x110"_b, "sqshl_asimdshf_r"},
      {"010x00"_b, "shl_asimdshf_r"},
      {"010x10"_b, "sqshl_asimdshf_r"},
      {"011100"_b, "shl_asimdshf_r"},
      {"011110"_b, "sqshl_asimdshf_r"},
      {"0x1000"_b, "shl_asimdshf_r"},
      {"0x1010"_b, "sqshl_asimdshf_r"},
    },
  },

  { "_kyhhqt",
    {23, 20, 19, 18, 17, 16, 13},
    { {"0000000"_b, "ld1r_asisdlso_r1"},
      {"0000001"_b, "ld3r_asisdlso_r3"},
      {"10xxxx0"_b, "ld1r_asisdlsop_rx1_r"},
      {"10xxxx1"_b, "ld3r_asisdlsop_rx3_r"},
      {"110xxx0"_b, "ld1r_asisdlsop_rx1_r"},
      {"110xxx1"_b, "ld3r_asisdlsop_rx3_r"},
      {"1110xx0"_b, "ld1r_asisdlsop_rx1_r"},
      {"1110xx1"_b, "ld3r_asisdlsop_rx3_r"},
      {"11110x0"_b, "ld1r_asisdlsop_rx1_r"},
      {"11110x1"_b, "ld3r_asisdlsop_rx3_r"},
      {"1111100"_b, "ld1r_asisdlsop_rx1_r"},
      {"1111101"_b, "ld3r_asisdlsop_rx3_r"},
      {"1111110"_b, "ld1r_asisdlsop_r1_i"},
      {"1111111"_b, "ld3r_asisdlsop_r3_i"},
    },
  },

  { "_kyjxrr",
    {30, 13},
    { {"00"_b, "_qtxpky"},
      {"01"_b, "_hnjrmp"},
      {"11"_b, "_vzjvtv"},
    },
  },

  { "_kynxnz",
    {30, 23, 22, 20, 19},
    { {"0xxxx"_b, "bl_only_branch_imm"},
      {"10001"_b, "sysl_rc_systeminstrs"},
      {"1001x"_b, "mrs_rs_systemmove"},
      {"1011x"_b, "mrrs_rs_systemmovepr"},
    },
  },

  { "_kyspnn",
    {22},
    { {"0"_b, "sqdmullb_z_zzi_s"},
      {"1"_b, "sqdmullb_z_zzi_d"},
    },
  },

  { "_kyxrqg",
    {10},
    { {"0"_b, "uabalb_z_zzz"},
      {"1"_b, "uabalt_z_zzz"},
    },
  },

  { "_kzjxxk",
    {20, 19, 18, 17, 16, 13, 12},
    { {"0000000"_b, "rbit_32_dp_1src"},
      {"0000001"_b, "clz_32_dp_1src"},
      {"0000010"_b, "abs_32_dp_1src"},
    },
  },

  { "_kzksnv",
    {13, 12},
    { {"00"_b, "sqshl_asisdsame_only"},
      {"01"_b, "sqrshl_asisdsame_only"},
    },
  },

  { "_kzmvpk",
    {23, 22, 10},
    { {"100"_b, "smlalb_z_zzzi_s"},
      {"101"_b, "smlalt_z_zzzi_s"},
      {"110"_b, "smlalb_z_zzzi_d"},
      {"111"_b, "smlalt_z_zzzi_d"},
    },
  },

  { "_kzprzt",
    {9, 8, 7, 6, 5, 2, 1},
    { {"1111111"_b, "retaa_64e_branch_reg"},
    },
  },

  { "_kzpyzy",
    {30, 23, 22, 13},
    { {"0000"_b, "ld1sh_z_p_br_s64"},
      {"0001"_b, "ldff1sh_z_p_br_s64"},
      {"0010"_b, "ld1w_z_p_br_u32"},
      {"0011"_b, "ldff1w_z_p_br_u32"},
      {"0100"_b, "ld1sb_z_p_br_s64"},
      {"0101"_b, "ldff1sb_z_p_br_s64"},
      {"0110"_b, "ld1sb_z_p_br_s16"},
      {"0111"_b, "ldff1sb_z_p_br_s16"},
      {"1001"_b, "stnt1w_z_p_br_contiguous"},
      {"1010"_b, "st1w_z_p_br"},
      {"1011"_b, "st3w_z_p_br_contiguous"},
      {"1100"_b, "str_z_bi"},
      {"1101"_b, "stnt1d_z_p_br_contiguous"},
      {"1111"_b, "st3d_z_p_br_contiguous"},
    },
  },

  { "_kzyzrh",
    {16, 13, 12},
    { {"000"_b, "rev16_64_dp_1src"},
      {"001"_b, "cls_64_dp_1src"},
      {"100"_b, "pacib_64p_dp_1src"},
      {"101"_b, "autib_64p_dp_1src"},
      {"110"_b, "_vpyvjr"},
      {"111"_b, "_sntnsm"},
    },
  },

  { "_lgmlmt",
    {18, 17},
    { {"00"_b, "ld3_asisdlse_r3"},
    },
  },

  { "_lgyqpk",
    {18, 17},
    { {"0x"_b, "st2_asisdlsop_sx2_r2s"},
      {"10"_b, "st2_asisdlsop_sx2_r2s"},
      {"11"_b, "st2_asisdlsop_s2_i2s"},
    },
  },

  { "_lgzlyq",
    {30, 23, 11, 10},
    { {"1001"_b, "_kltlmp"},
    },
  },

  { "_ljljkv",
    {30, 23, 22, 13, 12, 11, 10},
    { {"0001100"_b, "and_z_zz"},
      {"0001110"_b, "eor3_z_zzz"},
      {"0001111"_b, "bsl_z_zzz"},
      {"0011100"_b, "orr_z_zz"},
      {"0011110"_b, "bcax_z_zzz"},
      {"0011111"_b, "bsl1n_z_zzz"},
      {"0101100"_b, "eor_z_zz"},
      {"0101111"_b, "bsl2n_z_zzz"},
      {"0111100"_b, "bic_z_zz"},
      {"0111111"_b, "nbsl_z_zzz"},
      {"0xx0000"_b, "add_z_zz"},
      {"0xx0001"_b, "sub_z_zz"},
      {"0xx0100"_b, "sqadd_z_zz"},
      {"0xx0101"_b, "uqadd_z_zz"},
      {"0xx0110"_b, "sqsub_z_zz"},
      {"0xx0111"_b, "uqsub_z_zz"},
      {"0xx1101"_b, "xar_z_zzi"},
      {"10x0010"_b, "mla_z_zzzi_h"},
      {"10x0011"_b, "mls_z_zzzi_h"},
      {"10x0100"_b, "sqrdmlah_z_zzzi_h"},
      {"10x0101"_b, "sqrdmlsh_z_zzzi_h"},
      {"1100000"_b, "sdot_z_zzzi_s"},
      {"1100001"_b, "udot_z_zzzi_s"},
      {"1100010"_b, "mla_z_zzzi_s"},
      {"1100011"_b, "mls_z_zzzi_s"},
      {"1100100"_b, "sqrdmlah_z_zzzi_s"},
      {"1100101"_b, "sqrdmlsh_z_zzzi_s"},
      {"1100110"_b, "usdot_z_zzzi_s"},
      {"1100111"_b, "sudot_z_zzzi_s"},
      {"11010x0"_b, "sqdmlalb_z_zzzi_s"},
      {"11010x1"_b, "sqdmlalt_z_zzzi_s"},
      {"11011x0"_b, "sqdmlslb_z_zzzi_s"},
      {"11011x1"_b, "sqdmlslt_z_zzzi_s"},
      {"1110000"_b, "sdot_z_zzzi_d"},
      {"1110001"_b, "udot_z_zzzi_d"},
      {"1110010"_b, "mla_z_zzzi_d"},
      {"1110011"_b, "mls_z_zzzi_d"},
      {"1110100"_b, "sqrdmlah_z_zzzi_d"},
      {"1110101"_b, "sqrdmlsh_z_zzzi_d"},
      {"11110x0"_b, "sqdmlalb_z_zzzi_d"},
      {"11110x1"_b, "sqdmlalt_z_zzzi_d"},
      {"11111x0"_b, "sqdmlslb_z_zzzi_d"},
      {"11111x1"_b, "sqdmlslt_z_zzzi_d"},
    },
  },

  { "_ljtvgz",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "ucvtf_asimdmiscfp16_r"},
      {"0x00001"_b, "ucvtf_asimdmisc_r"},
      {"1111000"_b, "fcmle_asimdmiscfp16_fz"},
      {"1111001"_b, "frsqrte_asimdmiscfp16_r"},
      {"1x00000"_b, "fcmle_asimdmisc_fz"},
      {"1x00001"_b, "frsqrte_asimdmisc_r"},
    },
  },

  { "_lkpprr",
    {30, 23, 22},
    { {"000"_b, "sbfm_32m_bitfield"},
      {"100"_b, "ubfm_32m_bitfield"},
    },
  },

  { "_lkttgy",
    {10},
    { {"0"_b, "saba_z_zzz"},
      {"1"_b, "uaba_z_zzz"},
    },
  },

  { "_lkxgjy",
    {23, 22},
    { {"10"_b, "cmla_z_zzzi_h"},
      {"11"_b, "cmla_z_zzzi_s"},
    },
  },

  { "_lkzyzv",
    {30},
    { {"0"_b, "bl_only_branch_imm"},
      {"1"_b, "_vgxtvy"},
    },
  },

  { "_lljxgp",
    {1},
    { {"1"_b, "blrabz_64_branch_reg"},
    },
  },

  { "_llpsqq",
    {13, 12, 10},
    { {"001"_b, "_zjjxjl"},
      {"100"_b, "ptrues_p_s"},
      {"110"_b, "_njngkk"},
    },
  },

  { "_llqtkj",
    {18, 17},
    { {"00"_b, "ld2_asisdlso_s2_2s"},
    },
  },

  { "_lltzjg",
    {18, 17, 12},
    { {"0x0"_b, "ld2_asisdlsop_dx2_r2d"},
      {"100"_b, "ld2_asisdlsop_dx2_r2d"},
      {"110"_b, "ld2_asisdlsop_d2_i2d"},
    },
  },

  { "_llvrrk",
    {23, 18, 17, 16},
    { {"0000"_b, "sqxtnb_z_zz"},
    },
  },

  { "_lmmjvx",
    {4},
    { {"0"_b, "ccmn_64_condcmp_reg"},
    },
  },

  { "_lmmkzh",
    {4, 3, 2, 1, 0},
    { {"11111"_b, "_nntvzj"},
    },
  },

  { "_lmyxhr",
    {9, 4},
    { {"00"_b, "_gnqhsl"},
    },
  },

  { "_lnkrzt",
    {18, 4},
    { {"00"_b, "fcmne_p_p_z0"},
    },
  },

  { "_lnmhqq",
    {22, 13, 12},
    { {"000"_b, "ldsmaxa_64_memop"},
      {"001"_b, "ldsmina_64_memop"},
      {"010"_b, "ldumaxa_64_memop"},
      {"011"_b, "ldumina_64_memop"},
      {"100"_b, "ldsmaxal_64_memop"},
      {"101"_b, "ldsminal_64_memop"},
      {"110"_b, "ldumaxal_64_memop"},
      {"111"_b, "lduminal_64_memop"},
    },
  },

  { "_lnntps",
    {30, 11, 10},
    { {"000"_b, "_gvxjvz"},
      {"001"_b, "_ypzllm"},
      {"011"_b, "_gslmjl"},
      {"100"_b, "_jxltqm"},
      {"101"_b, "_shqyqv"},
      {"110"_b, "_jpvmkz"},
      {"111"_b, "_pxnyvl"},
    },
  },

  { "_lnsjqy",
    {20, 19, 18, 17, 16, 13, 12},
    { {"0000000"_b, "stzgm_64bulk_ldsttags"},
    },
  },

  { "_lplpkk",
    {30, 23, 22, 13, 12, 11, 10},
    { {"1101001"_b, "smmla_asimdsame2_g"},
      {"1101011"_b, "usmmla_asimdsame2_g"},
      {"x100111"_b, "usdot_asimdsame2_d"},
      {"xxx0101"_b, "sdot_asimdsame2_d"},
    },
  },

  { "_lplzxv",
    {13, 12, 11, 10},
    { {"0000"_b, "umull_asimddiff_l"},
      {"0001"_b, "_yxgmrs"},
      {"0010"_b, "_vyqxyz"},
      {"0011"_b, "_snzvtt"},
      {"0101"_b, "_svgvjm"},
      {"0110"_b, "_ljtvgz"},
      {"0111"_b, "_snhmgn"},
      {"1001"_b, "_kxmxxm"},
      {"1010"_b, "_nkpyjg"},
      {"1011"_b, "_gmsqqz"},
      {"1101"_b, "_gzgpjp"},
      {"1110"_b, "_nzmqhv"},
      {"1111"_b, "_xgxtlr"},
    },
  },

  { "_lptrlg",
    {13, 12},
    { {"00"_b, "sqadd_asisdsame_only"},
      {"10"_b, "sqsub_asisdsame_only"},
      {"11"_b, "cmge_asisdsame_only"},
    },
  },

  { "_lpzgvs",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "stllr_sl32_ldstexcl"},
    },
  },

  { "_lqjlkj",
    {13, 12},
    { {"00"_b, "cpyfp_cpy_memcms"},
      {"01"_b, "cpyfpwt_cpy_memcms"},
      {"10"_b, "cpyfprt_cpy_memcms"},
      {"11"_b, "cpyfpt_cpy_memcms"},
    },
  },

  { "_lqknkn",
    {18, 17},
    { {"0x"_b, "st4_asisdlsop_sx4_r4s"},
      {"10"_b, "st4_asisdlsop_sx4_r4s"},
      {"11"_b, "st4_asisdlsop_s4_i4s"},
    },
  },

  { "_lqlrxp",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "stlrb_sl32_ldstexcl"},
    },
  },

  { "_lqmksm",
    {30, 23, 22, 20, 13, 4},
    { {"00001x"_b, "ld1row_z_p_bi_u32"},
      {"000x0x"_b, "ld1row_z_p_br_contiguous"},
      {"01001x"_b, "ld1rod_z_p_bi_u64"},
      {"010x0x"_b, "ld1rod_z_p_br_contiguous"},
      {"110x00"_b, "str_p_bi"},
    },
  },

  { "_lrmgmq",
    {30, 23, 22},
    { {"00x"_b, "add_64_addsub_imm"},
      {"010"_b, "addg_64_addsub_immtags"},
      {"10x"_b, "sub_64_addsub_imm"},
      {"110"_b, "subg_64_addsub_immtags"},
    },
  },

  { "_lrntmz",
    {13, 12, 11, 10},
    { {"0000"_b, "saddlb_z_zz"},
      {"0001"_b, "saddlt_z_zz"},
      {"0010"_b, "uaddlb_z_zz"},
      {"0011"_b, "uaddlt_z_zz"},
      {"0100"_b, "ssublb_z_zz"},
      {"0101"_b, "ssublt_z_zz"},
      {"0110"_b, "usublb_z_zz"},
      {"0111"_b, "usublt_z_zz"},
      {"1100"_b, "sabdlb_z_zz"},
      {"1101"_b, "sabdlt_z_zz"},
      {"1110"_b, "uabdlb_z_zz"},
      {"1111"_b, "uabdlt_z_zz"},
    },
  },

  { "_lrptrn",
    {30, 23, 13, 12, 11, 10},
    { {"100001"_b, "sri_asisdshf_r"},
      {"100101"_b, "sli_asisdshf_r"},
      {"101001"_b, "sqshlu_asisdshf_r"},
      {"101101"_b, "uqshl_asisdshf_r"},
    },
  },

  { "_lrqlrg",
    {30},
    { {"1"_b, "_ylhgrh"},
    },
  },

  { "_lspzrv",
    {30, 23, 13},
    { {"000"_b, "ld1sb_z_p_bz_s_x32_unscaled"},
      {"001"_b, "ldff1sb_z_p_bz_s_x32_unscaled"},
      {"010"_b, "ld1sh_z_p_bz_s_x32_unscaled"},
      {"011"_b, "ldff1sh_z_p_bz_s_x32_unscaled"},
      {"100"_b, "ld1sb_z_p_bz_d_x32_unscaled"},
      {"101"_b, "ldff1sb_z_p_bz_d_x32_unscaled"},
      {"110"_b, "ld1sh_z_p_bz_d_x32_unscaled"},
      {"111"_b, "ldff1sh_z_p_bz_d_x32_unscaled"},
    },
  },

  { "_lsqgkk",
    {30},
    { {"1"_b, "_jsqvtn"},
    },
  },

  { "_lssjyz",
    {30},
    { {"1"_b, "_kxtqjh"},
    },
  },

  { "_lszlkq",
    {22, 20, 19, 18, 17, 16, 13, 12},
    { {"01111100"_b, "_xtgmvr"},
    },
  },

  { "_ltrntg",
    {12},
    { {"0"_b, "udot_asimdelem_d"},
      {"1"_b, "sqrdmlsh_asimdelem_r"},
    },
  },

  { "_lvjtlg",
    {30, 11, 10},
    { {"000"_b, "_krvxxx"},
      {"001"_b, "_rpjrhs"},
      {"010"_b, "_tsypsz"},
      {"011"_b, "_ktsgth"},
      {"100"_b, "_yhnqyy"},
      {"101"_b, "_xzqmkv"},
      {"110"_b, "_vxqtkl"},
      {"111"_b, "_jggxjz"},
    },
  },

  { "_lvryvp",
    {30},
    { {"0"_b, "_gkqhyz"},
      {"1"_b, "_nzqxrj"},
    },
  },

  { "_lvsrnj",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "scvtf_asimdmiscfp16_r"},
      {"0x00001"_b, "scvtf_asimdmisc_r"},
      {"1111000"_b, "fcmeq_asimdmiscfp16_fz"},
      {"1111001"_b, "frecpe_asimdmiscfp16_r"},
      {"1x00000"_b, "fcmeq_asimdmisc_fz"},
      {"1x00001"_b, "frecpe_asimdmisc_r"},
    },
  },

  { "_lvszgj",
    {2, 1},
    { {"11"_b, "brabz_64_branch_reg"},
    },
  },

  { "_lxggmz",
    {30},
    { {"0"_b, "b_only_branch_imm"},
    },
  },

  { "_lxhlkx",
    {12, 11, 10},
    { {"000"_b, "ftmad_z_zzi"},
    },
  },

  { "_lxlqks",
    {19},
    { {"1"_b, "sysp_cr_syspairinstrs"},
    },
  },

  { "_lylpyx",
    {10},
    { {"0"_b, "sabalb_z_zzz"},
      {"1"_b, "sabalt_z_zzz"},
    },
  },

  { "_lymhlk",
    {30},
    { {"0"_b, "bl_only_branch_imm"},
      {"1"_b, "_vpgxgk"},
    },
  },

  { "_lynsgm",
    {13},
    { {"0"_b, "_ttplgp"},
    },
  },

  { "_lytkrx",
    {12, 11, 10},
    { {"000"_b, "dup_z_zi"},
      {"010"_b, "tbl_z_zz_2"},
      {"011"_b, "tbx_z_zz"},
      {"100"_b, "tbl_z_zz_1"},
      {"110"_b, "_ylnsvy"},
    },
  },

  { "_lyzhrq",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "fcvtms_asimdmiscfp16_r"},
      {"0x00001"_b, "fcvtms_asimdmisc_r"},
      {"1111001"_b, "fcvtzs_asimdmiscfp16_r"},
      {"1x00001"_b, "fcvtzs_asimdmisc_r"},
      {"xx00000"_b, "abs_asimdmisc_r"},
      {"xx10001"_b, "addv_asimdall_only"},
    },
  },

  { "_lzjyhm",
    {30},
    { {"0"_b, "ldapursw_64_ldapstl_unscaled"},
    },
  },

  { "_lzqxgt",
    {13, 12},
    { {"00"_b, "sbcs_32_addsub_carry"},
    },
  },

  { "_lzvxxj",
    {23, 22},
    { {"01"_b, "fcmeq_asimdsamefp16_only"},
    },
  },

  { "_lzzsyj",
    {18, 17},
    { {"0x"_b, "st3_asisdlsep_r3_r"},
      {"10"_b, "st3_asisdlsep_r3_r"},
      {"11"_b, "st3_asisdlsep_i3_i"},
    },
  },

  { "_mgjhts",
    {13, 12, 10},
    { {"001"_b, "_rvtxys"},
      {"010"_b, "_ppyynh"},
      {"011"_b, "_vvyjmh"},
      {"101"_b, "_rpplns"},
      {"110"_b, "sqdmlal_asisdelem_l"},
      {"111"_b, "_ymmhtq"},
    },
  },

  { "_mgspnm",
    {30, 23},
    { {"00"_b, "orr_64_log_imm"},
      {"10"_b, "ands_64s_log_imm"},
      {"11"_b, "movk_64_movewide"},
    },
  },

  { "_mgtxyt",
    {13, 12},
    { {"00"_b, "sbcs_64_addsub_carry"},
    },
  },

  { "_mhksnq",
    {23, 22, 20, 19, 11},
    { {"00010"_b, "ucvtf_asisdshf_c"},
      {"001x0"_b, "ucvtf_asisdshf_c"},
      {"01xx0"_b, "ucvtf_asisdshf_c"},
    },
  },

  { "_mhljkp",
    {23},
    { {"0"_b, "frecps_asimdsame_only"},
      {"1"_b, "frsqrts_asimdsame_only"},
    },
  },

  { "_mhnlsy",
    {11, 10},
    { {"00"_b, "sm3tt1a_vvv4_crypto3_imm2"},
      {"01"_b, "sm3tt1b_vvv4_crypto3_imm2"},
      {"10"_b, "sm3tt2a_vvv4_crypto3_imm2"},
      {"11"_b, "sm3tt2b_vvv_crypto3_imm2"},
    },
  },

  { "_mhpgjx",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "stlr_sl64_ldstexcl"},
    },
  },

  { "_mhrjvp",
    {30, 13},
    { {"00"_b, "_vxhgzz"},
      {"01"_b, "_lytkrx"},
      {"10"_b, "_rlyvpn"},
      {"11"_b, "_yvptvx"},
    },
  },

  { "_mjjhqj",
    {30, 23, 22, 19, 16},
    { {"10010"_b, "aesimc_b_cryptoaes"},
      {"x0x01"_b, "fcvtl_asimdmisc_l"},
      {"xxx00"_b, "sqabs_asimdmisc_r"},
    },
  },

  { "_mjrlkp",
    {23, 22, 20, 19, 13, 11},
    { {"0000x0"_b, "movi_asimdimm_l_hl"},
      {"00x100"_b, "shrn_asimdshf_n"},
      {"00x101"_b, "rshrn_asimdshf_n"},
      {"00x110"_b, "sshll_asimdshf_l"},
      {"010x00"_b, "shrn_asimdshf_n"},
      {"010x01"_b, "rshrn_asimdshf_n"},
      {"010x10"_b, "sshll_asimdshf_l"},
      {"011100"_b, "shrn_asimdshf_n"},
      {"011101"_b, "rshrn_asimdshf_n"},
      {"011110"_b, "sshll_asimdshf_l"},
      {"0x1000"_b, "shrn_asimdshf_n"},
      {"0x1001"_b, "rshrn_asimdshf_n"},
      {"0x1010"_b, "sshll_asimdshf_l"},
    },
  },

  { "_mjrqhl",
    {18, 17},
    { {"0x"_b, "st3_asisdlsop_sx3_r3s"},
      {"10"_b, "st3_asisdlsop_sx3_r3s"},
      {"11"_b, "st3_asisdlsop_s3_i3s"},
    },
  },

  { "_mjstgz",
    {20, 19, 18, 17, 16, 13, 12},
    { {"0000000"_b, "ldgm_64bulk_ldsttags"},
    },
  },

  { "_mjyhsl",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldxp_lp32_ldstexcl"},
    },
  },

  { "_mkgsly",
    {19, 18, 17, 16, 4},
    { {"00000"_b, "brkas_p_p_p_z"},
      {"10000"_b, "brkns_p_p_pp"},
    },
  },

  { "_mkrgxr",
    {23, 4},
    { {"00"_b, "_hptkrj"},
    },
  },

  { "_mkyyng",
    {23, 22},
    { {"01"_b, "fcmla_asimdelem_c_h"},
      {"10"_b, "fcmla_asimdelem_c_s"},
    },
  },

  { "_mkzysy",
    {30, 23, 22},
    { {"000"_b, "str_b_ldst_pos"},
      {"001"_b, "ldr_b_ldst_pos"},
      {"010"_b, "str_q_ldst_pos"},
      {"011"_b, "ldr_q_ldst_pos"},
      {"100"_b, "str_h_ldst_pos"},
      {"101"_b, "ldr_h_ldst_pos"},
    },
  },

  { "_mlgmqm",
    {18, 17},
    { {"00"_b, "st2_asisdlso_s2_2s"},
    },
  },

  { "_mlxtxs",
    {10},
    { {"0"_b, "ssra_z_zi"},
      {"1"_b, "usra_z_zi"},
    },
  },

  { "_mmgpkx",
    {13, 12},
    { {"11"_b, "cmgt_asisdsame_only"},
    },
  },

  { "_mmxgrt",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "rev32_asimdmisc_r"},
    },
  },

  { "_mnmtql",
    {10},
    { {"0"_b, "srsra_z_zi"},
      {"1"_b, "ursra_z_zi"},
    },
  },

  { "_mntnlr",
    {18},
    { {"0"_b, "ld1_asisdlse_r4_4v"},
    },
  },

  { "_mnxgml",
    {22},
    { {"0"_b, "ldrsw_64_ldst_regoff"},
    },
  },

  { "_mnxgqm",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xxxx"_b, "fnmadd_s_floatdp3"},
      {"001xxxx"_b, "fnmadd_d_floatdp3"},
      {"011xxxx"_b, "fnmadd_h_floatdp3"},
      {"10001x0"_b, "fmls_asisdelem_rh_h"},
      {"10x0101"_b, "shl_asisdshf_r"},
      {"10x1101"_b, "sqshl_asisdshf_r"},
      {"11x01x0"_b, "fmls_asisdelem_r_sd"},
      {"1xx11x0"_b, "sqdmlsl_asisdelem_l"},
    },
  },

  { "_mnzgkx",
    {12},
    { {"0"_b, "st1_asisdlsop_dx1_r1d"},
    },
  },

  { "_mnzzhk",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "stlr_sl32_ldstexcl"},
    },
  },

  { "_mphkpq",
    {12},
    { {"0"_b, "st1_asisdlsop_dx1_r1d"},
    },
  },

  { "_mpstrr",
    {23, 22, 8, 7, 6, 5, 4, 3, 2, 1, 0},
    { {"00000000000"_b, "setffr_f"},
    },
  },

  { "_mpvsng",
    {30},
    { {"0"_b, "_vvtnrv"},
      {"1"_b, "_yykhjv"},
    },
  },

  { "_mpytmv",
    {23, 22, 20, 19, 11},
    { {"00011"_b, "fcvtzu_asisdshf_c"},
      {"001x1"_b, "fcvtzu_asisdshf_c"},
      {"01xx1"_b, "fcvtzu_asisdshf_c"},
    },
  },

  { "_mqljmr",
    {2, 1, 0},
    { {"000"_b, "_rnphqp"},
    },
  },

  { "_mqmrng",
    {9, 8, 7, 6, 5, 2, 1},
    { {"1111100"_b, "eret_64e_branch_reg"},
    },
  },

  { "_mqrzzk",
    {22, 20, 11},
    { {"000"_b, "sqincw_z_zs"},
      {"001"_b, "sqdecw_z_zs"},
      {"010"_b, "incw_z_zs"},
      {"100"_b, "sqincd_z_zs"},
      {"101"_b, "sqdecd_z_zs"},
      {"110"_b, "incd_z_zs"},
    },
  },

  { "_mqssgy",
    {30},
    { {"0"_b, "_slzrtr"},
      {"1"_b, "_nsgxlz"},
    },
  },

  { "_mqtgvk",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx00"_b, "stlurb_32_ldapstl_unscaled"},
      {"001xx00"_b, "ldapurb_32_ldapstl_unscaled"},
      {"010xx00"_b, "ldapursb_64_ldapstl_unscaled"},
      {"011xx00"_b, "ldapursb_32_ldapstl_unscaled"},
      {"100xx00"_b, "stlurh_32_ldapstl_unscaled"},
      {"101xx00"_b, "ldapurh_32_ldapstl_unscaled"},
      {"110xx00"_b, "ldapursh_64_ldapstl_unscaled"},
      {"111xx00"_b, "ldapursh_32_ldapstl_unscaled"},
      {"x000001"_b, "cpyfp_cpy_memcms"},
      {"x000101"_b, "cpyfpwt_cpy_memcms"},
      {"x001001"_b, "cpyfprt_cpy_memcms"},
      {"x001101"_b, "cpyfpt_cpy_memcms"},
      {"x010001"_b, "cpyfm_cpy_memcms"},
      {"x010101"_b, "cpyfmwt_cpy_memcms"},
      {"x011001"_b, "cpyfmrt_cpy_memcms"},
      {"x011101"_b, "cpyfmt_cpy_memcms"},
      {"x100001"_b, "cpyfe_cpy_memcms"},
      {"x100101"_b, "cpyfewt_cpy_memcms"},
      {"x101001"_b, "cpyfert_cpy_memcms"},
      {"x101101"_b, "cpyfet_cpy_memcms"},
      {"x110001"_b, "setp_set_memcms"},
      {"x110101"_b, "setpt_set_memcms"},
      {"x111001"_b, "setpn_set_memcms"},
      {"x111101"_b, "setptn_set_memcms"},
    },
  },

  { "_mrhtxt",
    {23, 22, 20, 9},
    { {"0000"_b, "brkpb_p_p_pp"},
      {"0100"_b, "brkpbs_p_p_pp"},
    },
  },

  { "_mrlpxr",
    {30, 23, 22},
    { {"000"_b, "_vqzsgg"},
      {"001"_b, "_tzjyhy"},
      {"011"_b, "_grsnms"},
      {"100"_b, "_sknvhk"},
      {"101"_b, "_ptqtmp"},
      {"111"_b, "_kktzst"},
    },
  },

  { "_msnshr",
    {23, 22, 13, 12, 11, 10},
    { {"0001x0"_b, "fmls_asimdelem_rh_h"},
      {"0x0101"_b, "shl_asimdshf_r"},
      {"0x1101"_b, "sqshl_asimdshf_r"},
      {"1000x0"_b, "fmlsl_asimdelem_lh"},
      {"1x01x0"_b, "fmls_asimdelem_r_sd"},
      {"xx10x0"_b, "smlsl_asimdelem_l"},
      {"xx11x0"_b, "sqdmlsl_asimdelem_l"},
    },
  },

  { "_msvhjv",
    {9, 8, 7, 6, 5},
    { {"00000"_b, "fmov_d_floatimm"},
    },
  },

  { "_msvjxq",
    {20, 19, 18, 17, 16},
    { {"00001"_b, "sqxtun_asisdmisc_n"},
    },
  },

  { "_msyrjz",
    {13, 12, 11, 10},
    { {"1111"_b, "casal_c64_ldstexcl"},
    },
  },

  { "_mthlnv",
    {18},
    { {"0"_b, "ld1_asisdlsep_r4_r4"},
      {"1"_b, "ld1_asisdlsep_i4_i4"},
    },
  },

  { "_mtkhgz",
    {10},
    { {"0"_b, "sha512su0_vv2_cryptosha512_2"},
      {"1"_b, "sm4e_vv4_cryptosha512_2"},
    },
  },

  { "_mtlxqp",
    {30, 23, 22},
    { {"000"_b, "stnp_64_ldstnapair_offs"},
      {"001"_b, "ldnp_64_ldstnapair_offs"},
      {"010"_b, "stp_64_ldstpair_post"},
      {"011"_b, "ldp_64_ldstpair_post"},
    },
  },

  { "_mtshvn",
    {18},
    { {"0"_b, "ld1_asisdlso_b1_1b"},
    },
  },

  { "_mtzhrn",
    {30, 23, 22, 11, 10, 4},
    { {"001000"_b, "ccmn_64_condcmp_reg"},
      {"001100"_b, "ccmn_64_condcmp_imm"},
      {"101000"_b, "ccmp_64_condcmp_reg"},
      {"101100"_b, "ccmp_64_condcmp_imm"},
    },
  },

  { "_mvqkzv",
    {18, 17, 12},
    { {"000"_b, "st2_asisdlso_d2_2d"},
    },
  },

  { "_mvvngm",
    {23, 22, 20, 19, 13, 11},
    { {"0000x0"_b, "mvni_asimdimm_l_sl"},
      {"00x100"_b, "ushr_asimdshf_r"},
      {"00x110"_b, "urshr_asimdshf_r"},
      {"010x00"_b, "ushr_asimdshf_r"},
      {"010x10"_b, "urshr_asimdshf_r"},
      {"011100"_b, "ushr_asimdshf_r"},
      {"011110"_b, "urshr_asimdshf_r"},
      {"0x1000"_b, "ushr_asimdshf_r"},
      {"0x1010"_b, "urshr_asimdshf_r"},
    },
  },

  { "_mxgykv",
    {19, 18, 17, 16},
    { {"0000"_b, "cntp_r_p_p"},
      {"1000"_b, "_lynsgm"},
      {"1001"_b, "_jxyskn"},
      {"1010"_b, "_jmxstz"},
      {"1011"_b, "_yjzknm"},
      {"1100"_b, "_zmtkvx"},
      {"1101"_b, "_yhmlxk"},
    },
  },

  { "_mxnzst",
    {30},
    { {"0"_b, "_vghjnt"},
      {"1"_b, "_pkqvxk"},
    },
  },

  { "_mxnzyr",
    {19, 16},
    { {"00"_b, "_nhxxmh"},
      {"10"_b, "_qgymsy"},
      {"11"_b, "_gjprmg"},
    },
  },

  { "_mxplnn",
    {30, 23, 22},
    { {"000"_b, "stnp_s_ldstnapair_offs"},
      {"001"_b, "ldnp_s_ldstnapair_offs"},
      {"010"_b, "stp_s_ldstpair_post"},
      {"011"_b, "ldp_s_ldstpair_post"},
      {"100"_b, "stnp_d_ldstnapair_offs"},
      {"101"_b, "ldnp_d_ldstnapair_offs"},
      {"110"_b, "stp_d_ldstpair_post"},
      {"111"_b, "ldp_d_ldstpair_post"},
    },
  },

  { "_mxvjxx",
    {20, 19, 18, 16},
    { {"0000"_b, "_nshjhk"},
    },
  },

  { "_mylphg",
    {30, 13, 4},
    { {"000"_b, "cmpge_p_p_zw"},
      {"001"_b, "cmpgt_p_p_zw"},
      {"010"_b, "cmplt_p_p_zw"},
      {"011"_b, "cmple_p_p_zw"},
      {"1xx"_b, "fcmla_z_p_zzz"},
    },
  },

  { "_myrkmk",
    {16, 13, 12},
    { {"000"_b, "rev32_64_dp_1src"},
      {"001"_b, "ctz_64_dp_1src"},
      {"100"_b, "pacda_64p_dp_1src"},
      {"101"_b, "autda_64p_dp_1src"},
      {"110"_b, "_tnjhxp"},
      {"111"_b, "_qqjtpm"},
    },
  },

  { "_myvqtn",
    {12},
    { {"0"_b, "_yrgzqr"},
    },
  },

  { "_myzhml",
    {20, 19, 18, 17, 16, 13, 12},
    { {"0000000"_b, "stgm_64bulk_ldsttags"},
    },
  },

  { "_mzhsrq",
    {4},
    { {"0"_b, "cmplt_p_p_zi"},
      {"1"_b, "cmple_p_p_zi"},
    },
  },

  { "_mzkxzm",
    {1},
    { {"0"_b, "blr_64_branch_reg"},
    },
  },

  { "_nghmrp",
    {13, 12, 11, 10},
    { {"1111"_b, "casal_c32_ldstexcl"},
    },
  },

  { "_ngkgsg",
    {23, 22, 20, 19, 11},
    { {"00000"_b, "movi_asimdimm_l_sl"},
    },
  },

  { "_ngnxrx",
    {18},
    { {"0"_b, "ld1_asisdlse_r2_2v"},
    },
  },

  { "_ngtlpz",
    {18, 17, 12},
    { {"0x0"_b, "st3_asisdlsop_dx3_r3d"},
      {"100"_b, "st3_asisdlsop_dx3_r3d"},
      {"110"_b, "st3_asisdlsop_d3_i3d"},
    },
  },

  { "_ngttyj",
    {30, 23, 22, 13},
    { {"0000"_b, "ld1b_z_p_br_u16"},
      {"0001"_b, "ldff1b_z_p_br_u16"},
      {"0010"_b, "ld1b_z_p_br_u64"},
      {"0011"_b, "ldff1b_z_p_br_u64"},
      {"0100"_b, "ld1h_z_p_br_u16"},
      {"0101"_b, "ldff1h_z_p_br_u16"},
      {"0110"_b, "ld1h_z_p_br_u64"},
      {"0111"_b, "ldff1h_z_p_br_u64"},
      {"1001"_b, "st2b_z_p_br_contiguous"},
      {"1011"_b, "st4b_z_p_br_contiguous"},
      {"10x0"_b, "st1b_z_p_br"},
      {"1101"_b, "st2h_z_p_br_contiguous"},
      {"1111"_b, "st4h_z_p_br_contiguous"},
      {"11x0"_b, "st1h_z_p_br"},
    },
  },

  { "_ngvqhs",
    {13, 12, 11, 10},
    { {"0001"_b, "ushl_asisdsame_only"},
      {"0010"_b, "_vrxhss"},
      {"0011"_b, "uqshl_asisdsame_only"},
      {"0101"_b, "urshl_asisdsame_only"},
      {"0111"_b, "uqrshl_asisdsame_only"},
      {"1010"_b, "_xprqgs"},
      {"1110"_b, "_yskyrg"},
    },
  },

  { "_ngzyqj",
    {11, 10},
    { {"00"_b, "asr_z_zi"},
      {"01"_b, "lsr_z_zi"},
      {"11"_b, "lsl_z_zi"},
    },
  },

  { "_nhnhzp",
    {23, 22, 20, 19, 17, 16, 13},
    { {"0000000"_b, "_hrymnk"},
      {"0000001"_b, "_hmgzjl"},
      {"0100000"_b, "_nxmgqz"},
      {"0100001"_b, "_ssjrxs"},
      {"100xxx0"_b, "st1_asisdlsop_hx1_r1h"},
      {"100xxx1"_b, "st3_asisdlsop_hx3_r3h"},
      {"1010xx0"_b, "st1_asisdlsop_hx1_r1h"},
      {"1010xx1"_b, "st3_asisdlsop_hx3_r3h"},
      {"10110x0"_b, "st1_asisdlsop_hx1_r1h"},
      {"10110x1"_b, "st3_asisdlsop_hx3_r3h"},
      {"1011100"_b, "st1_asisdlsop_hx1_r1h"},
      {"1011101"_b, "st3_asisdlsop_hx3_r3h"},
      {"1011110"_b, "_jyzhnh"},
      {"1011111"_b, "_qzlvkm"},
      {"110xxx0"_b, "ld1_asisdlsop_hx1_r1h"},
      {"110xxx1"_b, "ld3_asisdlsop_hx3_r3h"},
      {"1110xx0"_b, "ld1_asisdlsop_hx1_r1h"},
      {"1110xx1"_b, "ld3_asisdlsop_hx3_r3h"},
      {"11110x0"_b, "ld1_asisdlsop_hx1_r1h"},
      {"11110x1"_b, "ld3_asisdlsop_hx3_r3h"},
      {"1111100"_b, "ld1_asisdlsop_hx1_r1h"},
      {"1111101"_b, "ld3_asisdlsop_hx3_r3h"},
      {"1111110"_b, "_zmkntq"},
      {"1111111"_b, "_rxhssh"},
    },
  },

  { "_nhrkqm",
    {22, 20, 19, 18, 17, 16},
    { {"111001"_b, "ucvtf_asisdmiscfp16_r"},
      {"x00001"_b, "ucvtf_asisdmisc_r"},
      {"x10000"_b, "faddp_asisdpair_only_sd"},
    },
  },

  { "_nhxxmh",
    {23, 22, 9, 3, 2, 1, 0},
    { {"0100000"_b, "ptest_p_p"},
    },
  },

  { "_njjlxy",
    {30, 23, 22},
    { {"000"_b, "stlxp_sp32_ldstexcl"},
      {"001"_b, "_ymvzyh"},
      {"010"_b, "_nxttqn"},
      {"011"_b, "_nghmrp"},
      {"100"_b, "stlxp_sp64_ldstexcl"},
      {"101"_b, "_hpqkhv"},
      {"110"_b, "_xspjzn"},
      {"111"_b, "_msyrjz"},
    },
  },

  { "_njngkk",
    {23, 22, 9, 8, 7, 6, 5},
    { {"0000000"_b, "rdffr_p_f"},
    },
  },

  { "_njnsqm",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "frintn_asimdmiscfp16_r"},
      {"0x00001"_b, "frintn_asimdmisc_r"},
      {"1111001"_b, "frintp_asimdmiscfp16_r"},
      {"1x00001"_b, "frintp_asimdmisc_r"},
      {"xx00000"_b, "cmgt_asimdmisc_z"},
    },
  },

  { "_njvkjq",
    {11, 10},
    { {"00"_b, "index_z_ii"},
      {"01"_b, "index_z_ri"},
      {"10"_b, "index_z_ir"},
      {"11"_b, "index_z_rr"},
    },
  },

  { "_nklqly",
    {13, 12, 11, 10},
    { {"0000"_b, "sha256h_qqv_cryptosha3"},
      {"0100"_b, "sha256h2_qqv_cryptosha3"},
      {"1000"_b, "sha256su1_vvv_cryptosha3"},
    },
  },

  { "_nklvmv",
    {30, 23, 22, 13, 12, 11, 10},
    { {"1011001"_b, "fcmge_asisdsamefp16_only"},
      {"1011011"_b, "facge_asisdsamefp16_only"},
      {"1110101"_b, "fabd_asisdsamefp16_only"},
      {"1111001"_b, "fcmgt_asisdsamefp16_only"},
      {"1111011"_b, "facgt_asisdsamefp16_only"},
    },
  },

  { "_nklyky",
    {18, 17, 12},
    { {"000"_b, "st1_asisdlso_d1_1d"},
    },
  },

  { "_nkmkvz",
    {18},
    { {"0"_b, "st3_asisdlsop_bx3_r3b"},
      {"1"_b, "st3_asisdlsop_b3_i3b"},
    },
  },

  { "_nknntn",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "fcvtns_asimdmiscfp16_r"},
      {"0x00001"_b, "fcvtns_asimdmisc_r"},
      {"1111001"_b, "fcvtps_asimdmiscfp16_r"},
      {"1x00001"_b, "fcvtps_asimdmisc_r"},
      {"xx00000"_b, "cmlt_asimdmisc_z"},
      {"xx10000"_b, "smaxv_asimdall_only"},
      {"xx10001"_b, "sminv_asimdall_only"},
    },
  },

  { "_nkpyjg",
    {23, 20, 19, 18, 17, 16},
    { {"000001"_b, "frint32x_asimdmisc_r"},
    },
  },

  { "_nktrpj",
    {23, 22, 12},
    { {"001"_b, "sudot_asimdelem_d"},
      {"011"_b, "bfdot_asimdelem_e"},
      {"101"_b, "usdot_asimdelem_d"},
      {"111"_b, "bfmlal_asimdelem_f"},
      {"xx0"_b, "sdot_asimdelem_d"},
    },
  },

  { "_nkxhsy",
    {22, 20, 11},
    { {"000"_b, "cntb_r_s"},
      {"010"_b, "incb_r_rs"},
      {"100"_b, "cnth_r_s"},
      {"110"_b, "inch_r_rs"},
    },
  },

  { "_nkyrpv",
    {30, 23, 13, 12, 11, 10},
    { {"101001"_b, "ucvtf_asisdshf_c"},
      {"101111"_b, "fcvtzu_asisdshf_c"},
      {"1x01x0"_b, "sqrdmlah_asisdelem_r"},
      {"1x11x0"_b, "sqrdmlsh_asisdelem_r"},
    },
  },

  { "_nkyynq",
    {23, 22, 20, 19, 17, 16},
    { {"000010"_b, "scvtf_s32_float2fix"},
      {"000011"_b, "ucvtf_s32_float2fix"},
      {"001100"_b, "fcvtzs_32s_float2fix"},
      {"001101"_b, "fcvtzu_32s_float2fix"},
      {"010010"_b, "scvtf_d32_float2fix"},
      {"010011"_b, "ucvtf_d32_float2fix"},
      {"011100"_b, "fcvtzs_32d_float2fix"},
      {"011101"_b, "fcvtzu_32d_float2fix"},
      {"110010"_b, "scvtf_h32_float2fix"},
      {"110011"_b, "ucvtf_h32_float2fix"},
      {"111100"_b, "fcvtzs_32h_float2fix"},
      {"111101"_b, "fcvtzu_32h_float2fix"},
    },
  },

  { "_nlpmvl",
    {30, 13},
    { {"00"_b, "mad_z_p_zzz"},
      {"01"_b, "msb_z_p_zzz"},
    },
  },

  { "_nlrjsj",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx10"_b, "stlur_s_ldapstl_simd"},
      {"001xx10"_b, "ldapur_s_ldapstl_simd"},
      {"100xx10"_b, "stlur_d_ldapstl_simd"},
      {"101xx10"_b, "ldapur_d_ldapstl_simd"},
      {"x000001"_b, "cpypn_cpy_memcms"},
      {"x000101"_b, "cpypwtn_cpy_memcms"},
      {"x001001"_b, "cpyprtn_cpy_memcms"},
      {"x001101"_b, "cpyptn_cpy_memcms"},
      {"x010001"_b, "cpymn_cpy_memcms"},
      {"x010101"_b, "cpymwtn_cpy_memcms"},
      {"x011001"_b, "cpymrtn_cpy_memcms"},
      {"x011101"_b, "cpymtn_cpy_memcms"},
      {"x100001"_b, "cpyen_cpy_memcms"},
      {"x100101"_b, "cpyewtn_cpy_memcms"},
      {"x101001"_b, "cpyertn_cpy_memcms"},
      {"x101101"_b, "cpyetn_cpy_memcms"},
    },
  },

  { "_nmqrtr",
    {23, 22, 13, 12, 11, 10},
    { {"0001x0"_b, "fmul_asimdelem_rh_h"},
      {"0x0001"_b, "shrn_asimdshf_n"},
      {"0x0011"_b, "rshrn_asimdshf_n"},
      {"0x0101"_b, "sqshrn_asimdshf_n"},
      {"0x0111"_b, "sqrshrn_asimdshf_n"},
      {"0x1001"_b, "sshll_asimdshf_l"},
      {"1x01x0"_b, "fmul_asimdelem_r_sd"},
      {"xx00x0"_b, "mul_asimdelem_r"},
      {"xx10x0"_b, "smull_asimdelem_l"},
      {"xx11x0"_b, "sqdmull_asimdelem_l"},
    },
  },

  { "_nmqskh",
    {23, 22, 20, 19, 16, 13, 12},
    { {"0000000"_b, "_xkznrh"},
      {"0000010"_b, "_svlrvy"},
      {"0000011"_b, "_prmjlz"},
      {"0100000"_b, "_lgmlmt"},
      {"0100010"_b, "_qhpkhm"},
      {"0100011"_b, "_sqlsyr"},
      {"100xx00"_b, "st3_asisdlsep_r3_r"},
      {"100xx10"_b, "st1_asisdlsep_r3_r3"},
      {"100xx11"_b, "st1_asisdlsep_r1_r1"},
      {"1010x00"_b, "st3_asisdlsep_r3_r"},
      {"1010x10"_b, "st1_asisdlsep_r3_r3"},
      {"1010x11"_b, "st1_asisdlsep_r1_r1"},
      {"1011000"_b, "st3_asisdlsep_r3_r"},
      {"1011010"_b, "st1_asisdlsep_r3_r3"},
      {"1011011"_b, "st1_asisdlsep_r1_r1"},
      {"1011100"_b, "_lzzsyj"},
      {"1011110"_b, "_xqvzvl"},
      {"1011111"_b, "_vxrnyh"},
      {"110xx00"_b, "ld3_asisdlsep_r3_r"},
      {"110xx10"_b, "ld1_asisdlsep_r3_r3"},
      {"110xx11"_b, "ld1_asisdlsep_r1_r1"},
      {"1110x00"_b, "ld3_asisdlsep_r3_r"},
      {"1110x10"_b, "ld1_asisdlsep_r3_r3"},
      {"1110x11"_b, "ld1_asisdlsep_r1_r1"},
      {"1111000"_b, "ld3_asisdlsep_r3_r"},
      {"1111010"_b, "ld1_asisdlsep_r3_r3"},
      {"1111011"_b, "ld1_asisdlsep_r1_r1"},
      {"1111100"_b, "_ntxnpq"},
      {"1111110"_b, "_ghmtnl"},
      {"1111111"_b, "_gzrtkk"},
    },
  },

  { "_nnkxgr",
    {11, 10},
    { {"00"_b, "ftssel_z_zz"},
      {"10"_b, "_yhlntp"},
      {"11"_b, "_rsqmgk"},
    },
  },

  { "_nnrtpm",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "stllrb_sl32_ldstexcl"},
    },
  },

  { "_nntvzj",
    {11, 10, 9, 8, 7, 6},
    { {"000000"_b, "nop_hi_hints"},
      {"000001"_b, "wfe_hi_hints"},
      {"000010"_b, "sev_hi_hints"},
      {"000011"_b, "dgh_hi_hints"},
      {"000100"_b, "pacia1716_hi_hints"},
      {"000101"_b, "pacib1716_hi_hints"},
      {"000110"_b, "autia1716_hi_hints"},
      {"000111"_b, "autib1716_hi_hints"},
      {"001000"_b, "esb_hi_hints"},
      {"001001"_b, "tsb_hc_hints"},
      {"001010"_b, "csdb_hi_hints"},
      {"001011"_b, "clrbhb_hi_hints"},
      {"001100"_b, "paciaz_hi_hints"},
      {"001101"_b, "pacibz_hi_hints"},
      {"001110"_b, "autiaz_hi_hints"},
      {"001111"_b, "autibz_hi_hints"},
      {"0100xx"_b, "bti_hb_hints"},
      {"010100"_b, "chkfeat_hi_hints"},
      {"0101x1"_b, "hint_hm_hints"},
      {"01x110"_b, "hint_hm_hints"},
      {"10xxxx"_b, "hint_hm_hints"},
      {"110xxx"_b, "hint_hm_hints"},
      {"111110"_b, "hint_hm_hints"},
      {"x110xx"_b, "hint_hm_hints"},
      {"x1110x"_b, "hint_hm_hints"},
      {"x11111"_b, "hint_hm_hints"},
    },
  },

  { "_nnzhgm",
    {19, 18, 17, 16, 4},
    { {"0000x"_b, "brka_p_p_p"},
      {"10000"_b, "brkn_p_p_pp"},
    },
  },

  { "_npjnlv",
    {20, 19, 18, 17},
    { {"0000"_b, "_kzyzrh"},
    },
  },

  { "_npxkzq",
    {20, 19, 18, 17, 16, 13, 12},
    { {"0000000"_b, "_tykvnx"},
    },
  },

  { "_nqjtqn",
    {23, 22},
    { {"00"_b, "dup_asimdins_dv_v"},
      {"01"_b, "fmaxnm_asimdsamefp16_only"},
      {"11"_b, "fminnm_asimdsamefp16_only"},
    },
  },

  { "_nqjvmr",
    {13, 12},
    { {"00"_b, "adcs_32_addsub_carry"},
    },
  },

  { "_nqkhrv",
    {30, 13},
    { {"10"_b, "fnmla_z_p_zzz"},
      {"11"_b, "fnmls_z_p_zzz"},
    },
  },

  { "_nqlrmv",
    {30, 23, 22},
    { {"000"_b, "bfm_32m_bitfield"},
    },
  },

  { "_nqmnzp",
    {30, 23, 22, 20, 19, 18, 17, 16},
    { {"00000000"_b, "udf_only_perm_undef"},
    },
  },

  { "_nrmlqv",
    {23, 22, 20, 19, 13, 11},
    { {"0000x0"_b, "orr_asimdimm_l_sl"},
      {"00x100"_b, "ssra_asimdshf_r"},
      {"00x110"_b, "srsra_asimdshf_r"},
      {"010x00"_b, "ssra_asimdshf_r"},
      {"010x10"_b, "srsra_asimdshf_r"},
      {"011100"_b, "ssra_asimdshf_r"},
      {"011110"_b, "srsra_asimdshf_r"},
      {"0x1000"_b, "ssra_asimdshf_r"},
      {"0x1010"_b, "srsra_asimdshf_r"},
    },
  },

  { "_nsgvsv",
    {9, 8, 7, 6, 5},
    { {"00000"_b, "fmov_h_floatimm"},
    },
  },

  { "_nsgxlz",
    {13, 12, 10},
    { {"000"_b, "sqdmulh_asisdelem_r"},
      {"010"_b, "sqrdmulh_asisdelem_r"},
      {"101"_b, "_rkjjtp"},
      {"111"_b, "_pzpxxv"},
    },
  },

  { "_nshjhk",
    {17, 9, 8, 7, 6, 5},
    { {"000000"_b, "aesimc_z_z"},
      {"1xxxxx"_b, "aesd_z_zz"},
    },
  },

  { "_nsjhhg",
    {30, 13},
    { {"00"_b, "_jhllmn"},
      {"01"_b, "_htplsj"},
      {"10"_b, "_rztvnl"},
      {"11"_b, "_vgtnjh"},
    },
  },

  { "_ntjpsx",
    {22, 20, 11},
    { {"000"_b, "uqincb_r_rs_uw"},
      {"001"_b, "uqdecb_r_rs_uw"},
      {"010"_b, "uqincb_r_rs_x"},
      {"011"_b, "uqdecb_r_rs_x"},
      {"100"_b, "uqinch_r_rs_uw"},
      {"101"_b, "uqdech_r_rs_uw"},
      {"110"_b, "uqinch_r_rs_x"},
      {"111"_b, "uqdech_r_rs_x"},
    },
  },

  { "_ntjrlg",
    {18, 17, 16, 13, 12, 11, 10, 9, 7, 6, 5},
    { {"01111000011"_b, "_vsslrs"},
    },
  },

  { "_ntxnpq",
    {18, 17},
    { {"0x"_b, "ld3_asisdlsep_r3_r"},
      {"10"_b, "ld3_asisdlsep_r3_r"},
      {"11"_b, "ld3_asisdlsep_i3_i"},
    },
  },

  { "_nvkthr",
    {30, 13},
    { {"00"_b, "_kjqynn"},
      {"01"_b, "_jgyhrh"},
      {"10"_b, "_jymnkk"},
      {"11"_b, "_pqjjsh"},
    },
  },

  { "_nvkxzs",
    {12},
    { {"0"_b, "gcsstr_64_ldst_gcs"},
      {"1"_b, "gcssttr_64_ldst_gcs"},
    },
  },

  { "_nvnjyp",
    {23, 22, 20, 19, 11},
    { {"00000"_b, "mvni_asimdimm_m_sm"},
    },
  },

  { "_nvyxmh",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "add_z_p_zz"},
      {"00001"_b, "sub_z_p_zz"},
      {"00011"_b, "subr_z_p_zz"},
      {"01000"_b, "smax_z_p_zz"},
      {"01001"_b, "umax_z_p_zz"},
      {"01010"_b, "smin_z_p_zz"},
      {"01011"_b, "umin_z_p_zz"},
      {"01100"_b, "sabd_z_p_zz"},
      {"01101"_b, "uabd_z_p_zz"},
      {"10000"_b, "mul_z_p_zz"},
      {"10010"_b, "smulh_z_p_zz"},
      {"10011"_b, "umulh_z_p_zz"},
      {"10100"_b, "sdiv_z_p_zz"},
      {"10101"_b, "udiv_z_p_zz"},
      {"10110"_b, "sdivr_z_p_zz"},
      {"10111"_b, "udivr_z_p_zz"},
      {"11000"_b, "orr_z_p_zz"},
      {"11001"_b, "eor_z_p_zz"},
      {"11010"_b, "and_z_p_zz"},
      {"11011"_b, "bic_z_p_zz"},
    },
  },

  { "_nvzsxn",
    {18, 17, 12},
    { {"000"_b, "stl1_asisdlso_d1"},
    },
  },

  { "_nxjkqs",
    {23, 22, 12, 11, 10},
    { {"0x000"_b, "fmla_z_zzzi_h"},
      {"0x001"_b, "fmls_z_zzzi_h"},
      {"10000"_b, "fmla_z_zzzi_s"},
      {"10001"_b, "fmls_z_zzzi_s"},
      {"101xx"_b, "fcmla_z_zzzi_h"},
      {"11000"_b, "fmla_z_zzzi_d"},
      {"11001"_b, "fmls_z_zzzi_d"},
      {"111xx"_b, "fcmla_z_zzzi_s"},
    },
  },

  { "_nxlmhz",
    {30, 23},
    { {"00"_b, "add_32_addsub_imm"},
      {"10"_b, "sub_32_addsub_imm"},
    },
  },

  { "_nxlsjm",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldxrb_lr32_ldstexcl"},
    },
  },

  { "_nxmgqz",
    {18},
    { {"0"_b, "ld1_asisdlso_h1_1h"},
    },
  },

  { "_nxrqmg",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx00"_b, "stlur_32_ldapstl_unscaled"},
      {"001xx00"_b, "ldapur_32_ldapstl_unscaled"},
      {"010xx00"_b, "ldapursw_64_ldapstl_unscaled"},
      {"100xx00"_b, "stlur_64_ldapstl_unscaled"},
      {"101xx00"_b, "ldapur_64_ldapstl_unscaled"},
      {"x000001"_b, "cpyfpn_cpy_memcms"},
      {"x000101"_b, "cpyfpwtn_cpy_memcms"},
      {"x001001"_b, "cpyfprtn_cpy_memcms"},
      {"x001101"_b, "cpyfptn_cpy_memcms"},
      {"x010001"_b, "cpyfmn_cpy_memcms"},
      {"x010101"_b, "cpyfmwtn_cpy_memcms"},
      {"x011001"_b, "cpyfmrtn_cpy_memcms"},
      {"x011101"_b, "cpyfmtn_cpy_memcms"},
      {"x100001"_b, "cpyfen_cpy_memcms"},
      {"x100101"_b, "cpyfewtn_cpy_memcms"},
      {"x101001"_b, "cpyfertn_cpy_memcms"},
      {"x101101"_b, "cpyfetn_cpy_memcms"},
    },
  },

  { "_nxttqn",
    {13, 12, 11, 10},
    { {"1111"_b, "casl_c32_ldstexcl"},
    },
  },

  { "_nygsjm",
    {18},
    { {"0"_b, "st2_asisdlso_b2_2b"},
    },
  },

  { "_nyjtng",
    {23, 22},
    { {"01"_b, "fmls_z_p_zzz"},
      {"1x"_b, "fmls_z_p_zzz"},
    },
  },

  { "_nynrns",
    {23, 22, 12},
    { {"000"_b, "_klxxgx"},
      {"001"_b, "_pglvnj"},
      {"010"_b, "_pzttrn"},
      {"011"_b, "_svyszp"},
      {"110"_b, "_prrkzv"},
      {"111"_b, "_nsgvsv"},
    },
  },

  { "_nzmqhv",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0x00001"_b, "frint64x_asimdmisc_r"},
      {"0x10000"_b, "fmaxv_asimdall_only_sd"},
      {"1111000"_b, "fneg_asimdmiscfp16_r"},
      {"1111001"_b, "fsqrt_asimdmiscfp16_r"},
      {"1x00000"_b, "fneg_asimdmisc_r"},
      {"1x00001"_b, "fsqrt_asimdmisc_r"},
      {"1x10000"_b, "fminv_asimdall_only_sd"},
    },
  },

  { "_nzqxrj",
    {12},
    { {"1"_b, "_qgvtrn"},
    },
  },

  { "_nzskzl",
    {13, 12, 11, 10},
    { {"0000"_b, "uaddl_asimddiff_l"},
      {"0001"_b, "uhadd_asimdsame_only"},
      {"0010"_b, "_mmxgrt"},
      {"0011"_b, "uqadd_asimdsame_only"},
      {"0100"_b, "uaddw_asimddiff_w"},
      {"0101"_b, "urhadd_asimdsame_only"},
      {"0111"_b, "_yyvnrp"},
      {"1000"_b, "usubl_asimddiff_l"},
      {"1001"_b, "uhsub_asimdsame_only"},
      {"1010"_b, "_vlhkgr"},
      {"1011"_b, "uqsub_asimdsame_only"},
      {"1100"_b, "usubw_asimddiff_w"},
      {"1101"_b, "cmhi_asimdsame_only"},
      {"1110"_b, "_srpptk"},
      {"1111"_b, "cmhs_asimdsame_only"},
    },
  },

  { "_nzvlzt",
    {18},
    { {"0"_b, "st1_asisdlse_r4_4v"},
    },
  },

  { "_pgjjsz",
    {30, 13, 12, 11, 10},
    { {"00000"_b, "_lmyxhr"},
      {"00001"_b, "_tmhlvh"},
      {"00010"_b, "_qvtxpr"},
      {"00011"_b, "_ymkthj"},
      {"00100"_b, "_rhmxyp"},
      {"00101"_b, "_zryvjk"},
      {"01000"_b, "zip1_z_zz"},
      {"01001"_b, "zip2_z_zz"},
      {"01010"_b, "uzp1_z_zz"},
      {"01011"_b, "uzp2_z_zz"},
      {"01100"_b, "trn1_z_zz"},
      {"01101"_b, "trn2_z_zz"},
      {"10000"_b, "_llvrrk"},
      {"10001"_b, "_qyjvqr"},
      {"10010"_b, "_tmtnkq"},
      {"10011"_b, "_gpxltv"},
      {"10100"_b, "_pnlnzt"},
      {"10101"_b, "_pygvrr"},
      {"11000"_b, "addhnb_z_zz"},
      {"11001"_b, "addhnt_z_zz"},
      {"11010"_b, "raddhnb_z_zz"},
      {"11011"_b, "raddhnt_z_zz"},
      {"11100"_b, "subhnb_z_zz"},
      {"11101"_b, "subhnt_z_zz"},
      {"11110"_b, "rsubhnb_z_zz"},
      {"11111"_b, "rsubhnt_z_zz"},
    },
  },

  { "_pglvnj",
    {9, 8, 7, 6, 5},
    { {"00000"_b, "fmov_s_floatimm"},
    },
  },

  { "_pgmlrt",
    {30, 23, 22},
    { {"000"_b, "stxrb_sr32_ldstexcl"},
      {"001"_b, "_nxlsjm"},
      {"010"_b, "_nnrtpm"},
      {"011"_b, "_sksvrn"},
      {"100"_b, "stxrh_sr32_ldstexcl"},
      {"101"_b, "_knpjtt"},
      {"110"_b, "_zqhhlq"},
      {"111"_b, "_xtzykp"},
    },
  },

  { "_pgvjgs",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx10"_b, "stlur_s_ldapstl_simd"},
      {"001xx10"_b, "ldapur_s_ldapstl_simd"},
      {"100xx10"_b, "stlur_d_ldapstl_simd"},
      {"101xx10"_b, "ldapur_d_ldapstl_simd"},
      {"x000001"_b, "cpypwn_cpy_memcms"},
      {"x000101"_b, "cpypwtwn_cpy_memcms"},
      {"x001001"_b, "cpyprtwn_cpy_memcms"},
      {"x001101"_b, "cpyptwn_cpy_memcms"},
      {"x010001"_b, "cpymwn_cpy_memcms"},
      {"x010101"_b, "cpymwtwn_cpy_memcms"},
      {"x011001"_b, "cpymrtwn_cpy_memcms"},
      {"x011101"_b, "cpymtwn_cpy_memcms"},
      {"x100001"_b, "cpyewn_cpy_memcms"},
      {"x100101"_b, "cpyewtwn_cpy_memcms"},
      {"x101001"_b, "cpyertwn_cpy_memcms"},
      {"x101101"_b, "cpyetwn_cpy_memcms"},
      {"x110001"_b, "setgm_set_memcms"},
      {"x110101"_b, "setgmt_set_memcms"},
      {"x111001"_b, "setgmn_set_memcms"},
      {"x111101"_b, "setgmtn_set_memcms"},
    },
  },

  { "_phjkhr",
    {9, 8, 7, 6, 5},
    { {"11111"_b, "autdzb_64z_dp_1src"},
    },
  },

  { "_phktvp",
    {7, 6, 4, 3, 2, 1, 0},
    { {"0111111"_b, "clrex_bn_barriers"},
      {"1011111"_b, "dsb_bo_barriers"},
      {"1111111"_b, "isb_bi_barriers"},
    },
  },

  { "_phpphm",
    {18},
    { {"0"_b, "st4_asisdlso_h4_4h"},
    },
  },

  { "_phrqqx",
    {23, 22, 13},
    { {"100"_b, "fmlal_asimdelem_lh"},
      {"xx1"_b, "smlal_asimdelem_l"},
    },
  },

  { "_phsrlk",
    {23, 22, 13},
    { {"000"_b, "fmla_asimdelem_rh_h"},
      {"1x0"_b, "fmla_asimdelem_r_sd"},
      {"xx1"_b, "sqdmlal_asimdelem_l"},
    },
  },

  { "_phthqj",
    {30, 13},
    { {"00"_b, "_sntyqy"},
      {"01"_b, "_xhlhmh"},
      {"10"_b, "_rtrlts"},
      {"11"_b, "_jzkqhn"},
    },
  },

  { "_phtxqg",
    {13, 10},
    { {"00"_b, "_vrjhtm"},
      {"01"_b, "_spktyg"},
      {"10"_b, "_nktrpj"},
      {"11"_b, "_vzvstm"},
    },
  },

  { "_pjgkjs",
    {18, 17},
    { {"00"_b, "_mxnzyr"},
    },
  },

  { "_pjhmvy",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "saddlp_asimdmisc_p"},
      {"00001"_b, "xtn_asimdmisc_n"},
    },
  },

  { "_pjlnhh",
    {30, 23},
    { {"00"_b, "and_64_log_imm"},
      {"01"_b, "movn_64_movewide"},
      {"10"_b, "eor_64_log_imm"},
      {"11"_b, "movz_64_movewide"},
    },
  },

  { "_pjskhr",
    {18, 17},
    { {"00"_b, "st3_asisdlso_s3_3s"},
    },
  },

  { "_pjvkjz",
    {13, 12},
    { {"00"_b, "sbc_64_addsub_carry"},
    },
  },

  { "_pkjqsy",
    {20, 19, 18, 17, 16, 13, 12},
    { {"0000001"_b, "cnt_32_dp_1src"},
    },
  },

  { "_pkpvmj",
    {13, 12, 11, 10},
    { {"1111"_b, "casa_c64_ldstexcl"},
    },
  },

  { "_pkqvxk",
    {12},
    { {"1"_b, "_ynsytg"},
    },
  },

  { "_pkskpp",
    {30, 23},
    { {"00"_b, "adds_64s_addsub_imm"},
      {"10"_b, "subs_64s_addsub_imm"},
    },
  },

  { "_plgrmv",
    {13, 12},
    { {"00"_b, "adcs_64_addsub_carry"},
    },
  },

  { "_plrggq",
    {23, 22, 13},
    { {"000"_b, "fmls_asimdelem_rh_h"},
      {"1x0"_b, "fmls_asimdelem_r_sd"},
      {"xx1"_b, "sqdmlsl_asimdelem_l"},
    },
  },

  { "_plyhhz",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "cmge_asisdmisc_z"},
    },
  },

  { "_plymgg",
    {18},
    { {"1"_b, "frsqrte_z_z"},
    },
  },

  { "_plytvr",
    {22},
    { {"0"_b, "str_32_ldst_regoff"},
      {"1"_b, "ldr_32_ldst_regoff"},
    },
  },

  { "_plyxlq",
    {30, 18},
    { {"00"_b, "_nkyynq"},
    },
  },

  { "_plzqrv",
    {23, 22, 20, 19, 12, 11, 10},
    { {"00x1001"_b, "sqshrun_asisdshf_n"},
      {"00x1011"_b, "sqrshrun_asisdshf_n"},
      {"00x1101"_b, "uqshrn_asisdshf_n"},
      {"00x1111"_b, "uqrshrn_asisdshf_n"},
      {"00xx1x0"_b, "fmulx_asisdelem_rh_h"},
      {"010x001"_b, "sqshrun_asisdshf_n"},
      {"010x011"_b, "sqrshrun_asisdshf_n"},
      {"010x101"_b, "uqshrn_asisdshf_n"},
      {"010x111"_b, "uqrshrn_asisdshf_n"},
      {"0111001"_b, "sqshrun_asisdshf_n"},
      {"0111011"_b, "sqrshrun_asisdshf_n"},
      {"0111101"_b, "uqshrn_asisdshf_n"},
      {"0111111"_b, "uqrshrn_asisdshf_n"},
      {"0x10001"_b, "sqshrun_asisdshf_n"},
      {"0x10011"_b, "sqrshrun_asisdshf_n"},
      {"0x10101"_b, "uqshrn_asisdshf_n"},
      {"0x10111"_b, "uqrshrn_asisdshf_n"},
      {"1xxx1x0"_b, "fmulx_asisdelem_r_sd"},
    },
  },

  { "_pmpsvs",
    {18, 17, 12},
    { {"000"_b, "ld2_asisdlso_d2_2d"},
    },
  },

  { "_pnkxsr",
    {22, 20},
    { {"00"_b, "_hnsvjh"},
      {"01"_b, "mrs_rs_systemmove"},
      {"11"_b, "mrrs_rs_systemmovepr"},
    },
  },

  { "_pnlnzt",
    {23, 18, 17, 16},
    { {"0000"_b, "sqxtunb_z_zz"},
    },
  },

  { "_ppnssm",
    {30, 13, 12},
    { {"000"_b, "_ktyppm"},
      {"001"_b, "_ngzyqj"},
      {"010"_b, "_yxnslx"},
      {"011"_b, "_nnkxgr"},
      {"100"_b, "_kzmvpk"},
      {"101"_b, "_thrxph"},
      {"110"_b, "_kgpgly"},
      {"111"_b, "_yppszx"},
    },
  },

  { "_pppsmg",
    {30},
    { {"0"_b, "_xyhmgh"},
      {"1"_b, "_rlrjxp"},
    },
  },

  { "_ppvnly",
    {18, 17},
    { {"0x"_b, "ld2_asisdlsop_sx2_r2s"},
      {"10"_b, "ld2_asisdlsop_sx2_r2s"},
      {"11"_b, "ld2_asisdlsop_s2_i2s"},
    },
  },

  { "_ppyynh",
    {23, 22},
    { {"00"_b, "fmla_asisdelem_rh_h"},
      {"1x"_b, "fmla_asisdelem_r_sd"},
    },
  },

  { "_pqjjsh",
    {23, 22, 12, 10},
    { {"1000"_b, "fmlslb_z_zzzi_s"},
      {"1001"_b, "fmlslt_z_zzzi_s"},
    },
  },

  { "_pqmqrg",
    {30, 23, 22},
    { {"000"_b, "stp_s_ldstpair_off"},
      {"001"_b, "ldp_s_ldstpair_off"},
      {"010"_b, "stp_s_ldstpair_pre"},
      {"011"_b, "ldp_s_ldstpair_pre"},
      {"100"_b, "stp_d_ldstpair_off"},
      {"101"_b, "ldp_d_ldstpair_off"},
      {"110"_b, "stp_d_ldstpair_pre"},
      {"111"_b, "ldp_d_ldstpair_pre"},
    },
  },

  { "_pqsvty",
    {13},
    { {"0"_b, "_qqslmv"},
      {"1"_b, "_gjxsrn"},
    },
  },

  { "_prgrzz",
    {30},
    { {"0"_b, "cbnz_32_compbranch"},
    },
  },

  { "_prjzxs",
    {12},
    { {"0"_b, "ld2_asisdlsop_dx2_r2d"},
    },
  },

  { "_prkmty",
    {23, 22, 9},
    { {"000"_b, "brkpa_p_p_pp"},
      {"010"_b, "brkpas_p_p_pp"},
    },
  },

  { "_prmjlz",
    {18, 17},
    { {"00"_b, "st1_asisdlse_r1_1v"},
    },
  },

  { "_prrkzv",
    {20, 19, 18, 17, 16, 13},
    { {"000000"_b, "fmov_h_floatdp1"},
      {"000010"_b, "fneg_h_floatdp1"},
      {"000100"_b, "fcvt_sh_floatdp1"},
      {"001000"_b, "frintn_h_floatdp1"},
      {"001010"_b, "frintm_h_floatdp1"},
      {"001100"_b, "frinta_h_floatdp1"},
      {"001110"_b, "frintx_h_floatdp1"},
    },
  },

  { "_prtvjm",
    {23, 22, 12, 11, 10},
    { {"10000"_b, "fadd_z_zz"},
      {"10001"_b, "fsub_z_zz"},
      {"10010"_b, "fmul_z_zz"},
      {"x1000"_b, "fadd_z_zz"},
      {"x1001"_b, "fsub_z_zz"},
      {"x1010"_b, "fmul_z_zz"},
      {"xx011"_b, "ftsmul_z_zz"},
      {"xx110"_b, "frecps_z_zz"},
      {"xx111"_b, "frsqrts_z_zz"},
    },
  },

  { "_prxyhr",
    {9, 8, 7, 6, 5},
    { {"11111"_b, "autiza_64z_dp_1src"},
    },
  },

  { "_prytjs",
    {18, 4},
    { {"00"_b, "fcmge_p_p_z0"},
      {"01"_b, "fcmgt_p_p_z0"},
    },
  },

  { "_pstgvl",
    {23},
    { {"0"_b, "fmaxnm_asimdsame_only"},
      {"1"_b, "fminnm_asimdsame_only"},
    },
  },

  { "_ptjyqx",
    {13},
    { {"0"_b, "fcmuo_p_p_zz"},
    },
  },

  { "_ptkgrz",
    {22},
    { {"0"_b, "ldrsw_64_ldst_regoff"},
    },
  },

  { "_ptqtmp",
    {13, 12, 11, 10},
    { {"0111"_b, "fmulx_asisdsamefp16_only"},
      {"1001"_b, "fcmeq_asisdsamefp16_only"},
      {"1111"_b, "frecps_asisdsamefp16_only"},
    },
  },

  { "_ptsjnr",
    {30, 20, 19, 18, 17, 16, 13},
    { {"0000000"_b, "asr_z_p_zi"},
      {"0000010"_b, "lsr_z_p_zi"},
      {"0000110"_b, "lsl_z_p_zi"},
      {"0001000"_b, "asrd_z_p_zi"},
      {"0001100"_b, "sqshl_z_p_zi"},
      {"0001110"_b, "uqshl_z_p_zi"},
      {"0011000"_b, "srshr_z_p_zi"},
      {"0011010"_b, "urshr_z_p_zi"},
      {"0011110"_b, "sqshlu_z_p_zi"},
      {"0100000"_b, "asr_z_p_zz"},
      {"0100001"_b, "sxtb_z_p_z"},
      {"0100010"_b, "lsr_z_p_zz"},
      {"0100011"_b, "uxtb_z_p_z"},
      {"0100101"_b, "sxth_z_p_z"},
      {"0100110"_b, "lsl_z_p_zz"},
      {"0100111"_b, "uxth_z_p_z"},
      {"0101000"_b, "asrr_z_p_zz"},
      {"0101001"_b, "sxtw_z_p_z"},
      {"0101010"_b, "lsrr_z_p_zz"},
      {"0101011"_b, "uxtw_z_p_z"},
      {"0101101"_b, "abs_z_p_z"},
      {"0101110"_b, "lslr_z_p_zz"},
      {"0101111"_b, "neg_z_p_z"},
      {"0110000"_b, "asr_z_p_zw"},
      {"0110001"_b, "cls_z_p_z"},
      {"0110010"_b, "lsr_z_p_zw"},
      {"0110011"_b, "clz_z_p_z"},
      {"0110101"_b, "cnt_z_p_z"},
      {"0110110"_b, "lsl_z_p_zw"},
      {"0110111"_b, "cnot_z_p_z"},
      {"0111001"_b, "fabs_z_p_z"},
      {"0111011"_b, "fneg_z_p_z"},
      {"0111101"_b, "not_z_p_z"},
      {"1000001"_b, "urecpe_z_p_z"},
      {"1000011"_b, "ursqrte_z_p_z"},
      {"1000100"_b, "srshl_z_p_zz"},
      {"1000110"_b, "urshl_z_p_zz"},
      {"1001001"_b, "sadalp_z_p_z"},
      {"1001011"_b, "uadalp_z_p_z"},
      {"1001100"_b, "srshlr_z_p_zz"},
      {"1001110"_b, "urshlr_z_p_zz"},
      {"1010000"_b, "sqshl_z_p_zz"},
      {"1010001"_b, "sqabs_z_p_z"},
      {"1010010"_b, "uqshl_z_p_zz"},
      {"1010011"_b, "sqneg_z_p_z"},
      {"1010100"_b, "sqrshl_z_p_zz"},
      {"1010110"_b, "uqrshl_z_p_zz"},
      {"1011000"_b, "sqshlr_z_p_zz"},
      {"1011010"_b, "uqshlr_z_p_zz"},
      {"1011100"_b, "sqrshlr_z_p_zz"},
      {"1011110"_b, "uqrshlr_z_p_zz"},
      {"1100000"_b, "shadd_z_p_zz"},
      {"1100010"_b, "uhadd_z_p_zz"},
      {"1100011"_b, "addp_z_p_zz"},
      {"1100100"_b, "shsub_z_p_zz"},
      {"1100110"_b, "uhsub_z_p_zz"},
      {"1101000"_b, "srhadd_z_p_zz"},
      {"1101001"_b, "smaxp_z_p_zz"},
      {"1101010"_b, "urhadd_z_p_zz"},
      {"1101011"_b, "umaxp_z_p_zz"},
      {"1101100"_b, "shsubr_z_p_zz"},
      {"1101101"_b, "sminp_z_p_zz"},
      {"1101110"_b, "uhsubr_z_p_zz"},
      {"1101111"_b, "uminp_z_p_zz"},
      {"1110000"_b, "sqadd_z_p_zz"},
      {"1110010"_b, "uqadd_z_p_zz"},
      {"1110100"_b, "sqsub_z_p_zz"},
      {"1110110"_b, "uqsub_z_p_zz"},
      {"1111000"_b, "suqadd_z_p_zz"},
      {"1111010"_b, "usqadd_z_p_zz"},
      {"1111100"_b, "sqsubr_z_p_zz"},
      {"1111110"_b, "uqsubr_z_p_zz"},
    },
  },

  { "_ptslzg",
    {30, 23, 22, 13, 4},
    { {"01000"_b, "ldr_p_bi"},
      {"01100"_b, "prfb_i_p_bi_s"},
      {"01110"_b, "prfh_i_p_bi_s"},
      {"10x0x"_b, "ld1sw_z_p_bz_d_x32_scaled"},
      {"10x1x"_b, "ldff1sw_z_p_bz_d_x32_scaled"},
    },
  },

  { "_ptstkz",
    {4},
    { {"0"_b, "ccmp_32_condcmp_imm"},
    },
  },

  { "_ptyynt",
    {13, 12, 11, 10},
    { {"1111"_b, "_stmtkr"},
    },
  },

  { "_pvtyjz",
    {30},
    { {"0"_b, "ldapur_32_ldapstl_unscaled"},
      {"1"_b, "ldapur_64_ldapstl_unscaled"},
    },
  },

  { "_pxnyvl",
    {23, 13, 12},
    { {"001"_b, "fmulx_asisdsame_only"},
      {"011"_b, "frecps_asisdsame_only"},
      {"111"_b, "frsqrts_asisdsame_only"},
    },
  },

  { "_pxvjkp",
    {30},
    { {"0"_b, "bl_only_branch_imm"},
      {"1"_b, "_rmkpsk"},
    },
  },

  { "_pxyrpm",
    {22, 11},
    { {"00"_b, "sqdmulh_z_zzi_s"},
      {"01"_b, "mul_z_zzi_s"},
      {"10"_b, "sqdmulh_z_zzi_d"},
      {"11"_b, "mul_z_zzi_d"},
    },
  },

  { "_pxzvjl",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xxxx"_b, "fnmadd_s_floatdp3"},
      {"001xxxx"_b, "fnmadd_d_floatdp3"},
      {"011xxxx"_b, "fnmadd_h_floatdp3"},
      {"10001x0"_b, "fmla_asisdelem_rh_h"},
      {"10x0001"_b, "sshr_asisdshf_r"},
      {"10x0101"_b, "ssra_asisdshf_r"},
      {"10x1001"_b, "srshr_asisdshf_r"},
      {"10x1101"_b, "srsra_asisdshf_r"},
      {"11x01x0"_b, "fmla_asisdelem_r_sd"},
      {"1xx11x0"_b, "sqdmlal_asisdelem_l"},
    },
  },

  { "_pygvrr",
    {23, 18, 17, 16},
    { {"0000"_b, "sqxtunt_z_zz"},
    },
  },

  { "_pyhrrt",
    {30, 23, 22, 13, 12, 11, 10},
    { {"10001x0"_b, "fmulx_asisdelem_rh_h"},
      {"10x0001"_b, "sqshrun_asisdshf_n"},
      {"10x0011"_b, "sqrshrun_asisdshf_n"},
      {"10x0101"_b, "uqshrn_asisdshf_n"},
      {"10x0111"_b, "uqrshrn_asisdshf_n"},
      {"11x01x0"_b, "fmulx_asisdelem_r_sd"},
    },
  },

  { "_pyjnpz",
    {30, 13},
    { {"00"_b, "_xpqglq"},
      {"10"_b, "_ryrkqt"},
      {"11"_b, "_zjzmvh"},
    },
  },

  { "_pyjtyn",
    {22, 20, 19, 18, 17, 16},
    { {"111001"_b, "fcvtau_asisdmiscfp16_r"},
      {"x00001"_b, "fcvtau_asisdmisc_r"},
      {"x10000"_b, "fmaxnmp_asisdpair_only_sd"},
    },
  },

  { "_pyttkp",
    {30, 13, 12, 11, 10},
    { {"10001"_b, "sqrdmlah_asisdsame2_only"},
      {"10011"_b, "sqrdmlsh_asisdsame2_only"},
    },
  },

  { "_pyvvqx",
    {10},
    { {"0"_b, "_rkrntt"},
    },
  },

  { "_pzpxxv",
    {23, 22, 20, 19, 11},
    { {"00011"_b, "fcvtzs_asisdshf_c"},
      {"001x1"_b, "fcvtzs_asisdshf_c"},
      {"01xx1"_b, "fcvtzs_asisdshf_c"},
    },
  },

  { "_pzttrn",
    {20, 19, 18, 17, 16, 13},
    { {"000000"_b, "fmov_d_floatdp1"},
      {"000010"_b, "fneg_d_floatdp1"},
      {"000100"_b, "fcvt_sd_floatdp1"},
      {"000110"_b, "bfcvt_bs_floatdp1"},
      {"001000"_b, "frintn_d_floatdp1"},
      {"001010"_b, "frintm_d_floatdp1"},
      {"001100"_b, "frinta_d_floatdp1"},
      {"001110"_b, "frintx_d_floatdp1"},
      {"010000"_b, "frint32z_d_floatdp1"},
      {"010010"_b, "frint64z_d_floatdp1"},
    },
  },

  { "_pzzgts",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx10"_b, "stlur_s_ldapstl_simd"},
      {"001xx10"_b, "ldapur_s_ldapstl_simd"},
      {"100xx10"_b, "stlur_d_ldapstl_simd"},
      {"101xx10"_b, "ldapur_d_ldapstl_simd"},
      {"x000001"_b, "cpyp_cpy_memcms"},
      {"x000101"_b, "cpypwt_cpy_memcms"},
      {"x001001"_b, "cpyprt_cpy_memcms"},
      {"x001101"_b, "cpypt_cpy_memcms"},
      {"x010001"_b, "cpym_cpy_memcms"},
      {"x010101"_b, "cpymwt_cpy_memcms"},
      {"x011001"_b, "cpymrt_cpy_memcms"},
      {"x011101"_b, "cpymt_cpy_memcms"},
      {"x100001"_b, "cpye_cpy_memcms"},
      {"x100101"_b, "cpyewt_cpy_memcms"},
      {"x101001"_b, "cpyert_cpy_memcms"},
      {"x101101"_b, "cpyet_cpy_memcms"},
      {"x110001"_b, "setgp_set_memcms"},
      {"x110101"_b, "setgpt_set_memcms"},
      {"x111001"_b, "setgpn_set_memcms"},
      {"x111101"_b, "setgptn_set_memcms"},
    },
  },

  { "_qgqgkx",
    {30, 23, 22},
    { {"000"_b, "adds_32s_addsub_ext"},
      {"100"_b, "subs_32s_addsub_ext"},
    },
  },

  { "_qgshrr",
    {30, 22, 20, 19, 18, 17, 16},
    { {"00xxxxx"_b, "stlxp_sp32_ldstexcl"},
      {"0111111"_b, "ldaxp_lp32_ldstexcl"},
      {"10xxxxx"_b, "stlxp_sp64_ldstexcl"},
      {"1111111"_b, "ldaxp_lp64_ldstexcl"},
    },
  },

  { "_qgsrqq",
    {23, 22},
    { {"00"_b, "fmadd_s_floatdp3"},
      {"01"_b, "fmadd_d_floatdp3"},
      {"11"_b, "fmadd_h_floatdp3"},
    },
  },

  { "_qgvrqy",
    {1},
    { {"1"_b, "blraaz_64_branch_reg"},
    },
  },

  { "_qgvtrn",
    {23, 22, 20, 19, 13, 11, 10},
    { {"00x1001"_b, "sqshrn_asisdshf_n"},
      {"00x1011"_b, "sqrshrn_asisdshf_n"},
      {"00xx0x0"_b, "fmul_asisdelem_rh_h"},
      {"010x001"_b, "sqshrn_asisdshf_n"},
      {"010x011"_b, "sqrshrn_asisdshf_n"},
      {"0111001"_b, "sqshrn_asisdshf_n"},
      {"0111011"_b, "sqrshrn_asisdshf_n"},
      {"0x10001"_b, "sqshrn_asisdshf_n"},
      {"0x10011"_b, "sqrshrn_asisdshf_n"},
      {"1xxx0x0"_b, "fmul_asisdelem_r_sd"},
      {"xxxx1x0"_b, "sqdmull_asisdelem_l"},
    },
  },

  { "_qgymsy",
    {11},
    { {"0"_b, "_hmsgpj"},
    },
  },

  { "_qgyppr",
    {23, 13, 12, 11, 10},
    { {"00010"_b, "_pyjtyn"},
      {"00110"_b, "_nhrkqm"},
      {"01001"_b, "fcmge_asisdsame_only"},
      {"01011"_b, "facge_asisdsame_only"},
      {"01110"_b, "_kxmjsh"},
      {"10010"_b, "_rpjgkh"},
      {"10101"_b, "fabd_asisdsame_only"},
      {"10110"_b, "_hmpzzg"},
      {"11001"_b, "fcmgt_asisdsame_only"},
      {"11011"_b, "facgt_asisdsame_only"},
      {"11110"_b, "_sxsxxt"},
    },
  },

  { "_qhpkhm",
    {18, 17},
    { {"00"_b, "ld1_asisdlse_r3_3v"},
    },
  },

  { "_qhzvvh",
    {30},
    { {"0"_b, "bl_only_branch_imm"},
    },
  },

  { "_qjqrgz",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldarh_lr32_ldstexcl"},
    },
  },

  { "_qjrllr",
    {23, 22, 12},
    { {"000"_b, "_pqsvty"},
      {"001"_b, "_rjrqxt"},
      {"010"_b, "_rnsmjq"},
      {"011"_b, "_msvhjv"},
      {"110"_b, "_rnlxtv"},
      {"111"_b, "_jjgpxz"},
    },
  },

  { "_qjstll",
    {18, 17},
    { {"0x"_b, "ld3_asisdlsop_sx3_r3s"},
      {"10"_b, "ld3_asisdlsop_sx3_r3s"},
      {"11"_b, "ld3_asisdlsop_s3_i3s"},
    },
  },

  { "_qkhrkh",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "cmle_asisdmisc_z"},
    },
  },

  { "_qkrnms",
    {30},
    { {"0"_b, "orr_32_log_shift"},
      {"1"_b, "ands_32_log_shift"},
    },
  },

  { "_qkxmvp",
    {13, 12},
    { {"10"_b, "smin_64_dp_2src"},
    },
  },

  { "_qkzjxm",
    {30, 23, 22, 20, 13},
    { {"00001"_b, "ldnt1w_z_p_bi_contiguous"},
      {"000x0"_b, "ldnt1w_z_p_br_contiguous"},
      {"00101"_b, "ld3w_z_p_bi_contiguous"},
      {"001x0"_b, "ld3w_z_p_br_contiguous"},
      {"01001"_b, "ldnt1d_z_p_bi_contiguous"},
      {"010x0"_b, "ldnt1d_z_p_br_contiguous"},
      {"01101"_b, "ld3d_z_p_bi_contiguous"},
      {"011x0"_b, "ld3d_z_p_br_contiguous"},
      {"10011"_b, "stnt1w_z_p_bi_contiguous"},
      {"100x0"_b, "st1w_z_p_bz_d_x32_unscaled"},
      {"10101"_b, "st1w_z_p_bi"},
      {"10111"_b, "st3w_z_p_bi_contiguous"},
      {"101x0"_b, "st1w_z_p_bz_s_x32_unscaled"},
      {"11011"_b, "stnt1d_z_p_bi_contiguous"},
      {"110x0"_b, "st1d_z_p_bz_d_x32_unscaled"},
      {"11111"_b, "st3d_z_p_bi_contiguous"},
    },
  },

  { "_qljhnp",
    {22},
    { {"0"_b, "sqdmullt_z_zzi_s"},
      {"1"_b, "sqdmullt_z_zzi_d"},
    },
  },

  { "_qlmqyx",
    {18, 17, 12},
    { {"0x0"_b, "ld3_asisdlsop_dx3_r3d"},
      {"100"_b, "ld3_asisdlsop_dx3_r3d"},
      {"110"_b, "ld3_asisdlsop_d3_i3d"},
    },
  },

  { "_qlpnnn",
    {23, 10, 4},
    { {"000"_b, "_vryrnh"},
    },
  },

  { "_qlxksl",
    {30},
    { {"0"_b, "_hrxyts"},
      {"1"_b, "_tytvjk"},
    },
  },

  { "_qlxlxk",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldar_lr32_ldstexcl"},
    },
  },

  { "_qlzvpg",
    {13, 12, 11, 10},
    { {"0000"_b, "raddhn_asimddiff_n"},
      {"0001"_b, "ushl_asimdsame_only"},
      {"0010"_b, "_kpnlmr"},
      {"0011"_b, "uqshl_asimdsame_only"},
      {"0100"_b, "uabal_asimddiff_l"},
      {"0101"_b, "urshl_asimdsame_only"},
      {"0110"_b, "_ssqyrk"},
      {"0111"_b, "uqrshl_asimdsame_only"},
      {"1000"_b, "rsubhn_asimddiff_n"},
      {"1001"_b, "umax_asimdsame_only"},
      {"1010"_b, "_sjlqvg"},
      {"1011"_b, "umin_asimdsame_only"},
      {"1100"_b, "uabdl_asimddiff_l"},
      {"1101"_b, "uabd_asimdsame_only"},
      {"1110"_b, "_gplkxy"},
      {"1111"_b, "uaba_asimdsame_only"},
    },
  },

  { "_qnprqt",
    {4},
    { {"0"_b, "eor_p_p_pp_z"},
      {"1"_b, "sel_p_p_pp"},
    },
  },

  { "_qntrvk",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xxxx"_b, "fnmsub_s_floatdp3"},
      {"001xxxx"_b, "fnmsub_d_floatdp3"},
      {"011xxxx"_b, "fnmsub_h_floatdp3"},
      {"10001x0"_b, "fmul_asisdelem_rh_h"},
      {"10x0101"_b, "sqshrn_asisdshf_n"},
      {"10x0111"_b, "sqrshrn_asisdshf_n"},
      {"11x01x0"_b, "fmul_asisdelem_r_sd"},
      {"1xx11x0"_b, "sqdmull_asisdelem_l"},
    },
  },

  { "_qnysqv",
    {30},
    { {"0"_b, "cbnz_64_compbranch"},
    },
  },

  { "_qpgxxr",
    {23, 22},
    { {"01"_b, "fadd_asimdsamefp16_only"},
      {"11"_b, "fsub_asimdsamefp16_only"},
    },
  },

  { "_qpsryx",
    {30, 23, 22, 11, 10},
    { {"01000"_b, "csel_64_condsel"},
      {"01001"_b, "csinc_64_condsel"},
      {"11000"_b, "csinv_64_condsel"},
      {"11001"_b, "csneg_64_condsel"},
    },
  },

  { "_qpvgnh",
    {30, 23, 22, 20, 13},
    { {"00001"_b, "ld2b_z_p_bi_contiguous"},
      {"000x0"_b, "ld2b_z_p_br_contiguous"},
      {"00101"_b, "ld4b_z_p_bi_contiguous"},
      {"001x0"_b, "ld4b_z_p_br_contiguous"},
      {"01001"_b, "ld2h_z_p_bi_contiguous"},
      {"010x0"_b, "ld2h_z_p_br_contiguous"},
      {"01101"_b, "ld4h_z_p_bi_contiguous"},
      {"011x0"_b, "ld4h_z_p_br_contiguous"},
      {"10011"_b, "st2b_z_p_bi_contiguous"},
      {"10111"_b, "st4b_z_p_bi_contiguous"},
      {"10x01"_b, "st1b_z_p_bi"},
      {"11011"_b, "st2h_z_p_bi_contiguous"},
      {"110x0"_b, "st1h_z_p_bz_d_x32_scaled"},
      {"11111"_b, "st4h_z_p_bi_contiguous"},
      {"111x0"_b, "st1h_z_p_bz_s_x32_scaled"},
      {"11x01"_b, "st1h_z_p_bi"},
    },
  },

  { "_qpyxsv",
    {18},
    { {"0"_b, "ld4_asisdlso_h4_4h"},
    },
  },

  { "_qqjtpm",
    {9, 8, 7, 6, 5},
    { {"11111"_b, "autdza_64z_dp_1src"},
    },
  },

  { "_qqslmv",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "fcvtns_32s_float2int"},
      {"00001"_b, "fcvtnu_32s_float2int"},
      {"00010"_b, "scvtf_s32_float2int"},
      {"00011"_b, "ucvtf_s32_float2int"},
      {"00100"_b, "fcvtas_32s_float2int"},
      {"00101"_b, "fcvtau_32s_float2int"},
      {"00110"_b, "fmov_32s_float2int"},
      {"00111"_b, "fmov_s32_float2int"},
      {"01000"_b, "fcvtps_32s_float2int"},
      {"01001"_b, "fcvtpu_32s_float2int"},
      {"10000"_b, "fcvtms_32s_float2int"},
      {"10001"_b, "fcvtmu_32s_float2int"},
      {"11000"_b, "fcvtzs_32s_float2int"},
      {"11001"_b, "fcvtzu_32s_float2int"},
    },
  },

  { "_qqvgql",
    {4, 3, 2, 1, 0},
    { {"11111"_b, "_gtsglj"},
    },
  },

  { "_qqyryl",
    {30, 23, 22, 13, 4},
    { {"00x0x"_b, "ld1w_z_p_bz_s_x32_unscaled"},
      {"00x1x"_b, "ldff1w_z_p_bz_s_x32_unscaled"},
      {"0100x"_b, "ldr_z_bi"},
      {"01100"_b, "prfw_i_p_bi_s"},
      {"01110"_b, "prfd_i_p_bi_s"},
      {"10x0x"_b, "ld1w_z_p_bz_d_x32_unscaled"},
      {"10x1x"_b, "ldff1w_z_p_bz_d_x32_unscaled"},
      {"11x0x"_b, "ld1d_z_p_bz_d_x32_unscaled"},
      {"11x1x"_b, "ldff1d_z_p_bz_d_x32_unscaled"},
    },
  },

  { "_qrsxzp",
    {23, 22, 20, 19, 16, 13, 10},
    { {"0000000"_b, "_tjnzjl"},
      {"0000001"_b, "_nklyky"},
      {"0000010"_b, "_pjskhr"},
      {"0000011"_b, "_kqstrr"},
      {"0000101"_b, "_nvzsxn"},
      {"0100000"_b, "_jnktqs"},
      {"0100001"_b, "_ttzlqn"},
      {"0100010"_b, "_sxgnmg"},
      {"0100011"_b, "_yqzxvr"},
      {"0100101"_b, "_tvtvkt"},
      {"100xx00"_b, "st1_asisdlsop_sx1_r1s"},
      {"100xx01"_b, "_mnzgkx"},
      {"100xx10"_b, "st3_asisdlsop_sx3_r3s"},
      {"100xx11"_b, "_tjxyky"},
      {"1010x00"_b, "st1_asisdlsop_sx1_r1s"},
      {"1010x01"_b, "_mphkpq"},
      {"1010x10"_b, "st3_asisdlsop_sx3_r3s"},
      {"1010x11"_b, "_hqkhsy"},
      {"1011000"_b, "st1_asisdlsop_sx1_r1s"},
      {"1011001"_b, "_qsszkx"},
      {"1011010"_b, "st3_asisdlsop_sx3_r3s"},
      {"1011011"_b, "_gsjvmx"},
      {"1011100"_b, "_gqmjys"},
      {"1011101"_b, "_qtqrmn"},
      {"1011110"_b, "_mjrqhl"},
      {"1011111"_b, "_ngtlpz"},
      {"110xx00"_b, "ld1_asisdlsop_sx1_r1s"},
      {"110xx01"_b, "_hkjjsr"},
      {"110xx10"_b, "ld3_asisdlsop_sx3_r3s"},
      {"110xx11"_b, "_yryygq"},
      {"1110x00"_b, "ld1_asisdlsop_sx1_r1s"},
      {"1110x01"_b, "_tptqjs"},
      {"1110x10"_b, "ld3_asisdlsop_sx3_r3s"},
      {"1110x11"_b, "_szmyzt"},
      {"1111000"_b, "ld1_asisdlsop_sx1_r1s"},
      {"1111001"_b, "_zxklzp"},
      {"1111010"_b, "ld3_asisdlsop_sx3_r3s"},
      {"1111011"_b, "_qzxgqh"},
      {"1111100"_b, "_yzgthp"},
      {"1111101"_b, "_rgnryt"},
      {"1111110"_b, "_qjstll"},
      {"1111111"_b, "_qlmqyx"},
    },
  },

  { "_qrtjvn",
    {30, 23, 22, 20, 19, 12, 11},
    { {"0000000"_b, "movi_asimdimm_d_ds"},
      {"1000000"_b, "movi_asimdimm_d2_d"},
      {"1000010"_b, "fmov_asimdimm_d2_d"},
      {"x00x100"_b, "ucvtf_asimdshf_c"},
      {"x00x111"_b, "fcvtzu_asimdshf_c"},
      {"x010x00"_b, "ucvtf_asimdshf_c"},
      {"x010x11"_b, "fcvtzu_asimdshf_c"},
      {"x011100"_b, "ucvtf_asimdshf_c"},
      {"x011111"_b, "fcvtzu_asimdshf_c"},
      {"x0x1000"_b, "ucvtf_asimdshf_c"},
      {"x0x1011"_b, "fcvtzu_asimdshf_c"},
    },
  },

  { "_qrygny",
    {30, 23, 22, 20, 13},
    { {"00001"_b, "ld1b_z_p_bi_u8"},
      {"00011"_b, "ldnf1b_z_p_bi_u8"},
      {"00101"_b, "ld1b_z_p_bi_u32"},
      {"00111"_b, "ldnf1b_z_p_bi_u32"},
      {"01001"_b, "ld1sw_z_p_bi_s64"},
      {"01011"_b, "ldnf1sw_z_p_bi_s64"},
      {"01101"_b, "ld1h_z_p_bi_u32"},
      {"01111"_b, "ldnf1h_z_p_bi_u32"},
      {"100x0"_b, "st1b_z_p_bz_d_x32_unscaled"},
      {"100x1"_b, "st1b_z_p_bz_d_64_unscaled"},
      {"101x0"_b, "st1b_z_p_bz_s_x32_unscaled"},
      {"101x1"_b, "st1b_z_p_ai_d"},
      {"110x0"_b, "st1h_z_p_bz_d_x32_unscaled"},
      {"110x1"_b, "st1h_z_p_bz_d_64_unscaled"},
      {"111x0"_b, "st1h_z_p_bz_s_x32_unscaled"},
      {"111x1"_b, "st1h_z_p_ai_d"},
    },
  },

  { "_qsszkx",
    {12},
    { {"0"_b, "st1_asisdlsop_dx1_r1d"},
    },
  },

  { "_qtghgs",
    {22},
    { {"0"_b, "ldrsw_64_ldst_regoff"},
    },
  },

  { "_qtgrzv",
    {20, 18, 17},
    { {"000"_b, "_gznrjv"},
    },
  },

  { "_qtgvlx",
    {23, 22, 20, 19, 17, 16},
    { {"000010"_b, "scvtf_s64_float2fix"},
      {"000011"_b, "ucvtf_s64_float2fix"},
      {"001100"_b, "fcvtzs_64s_float2fix"},
      {"001101"_b, "fcvtzu_64s_float2fix"},
      {"010010"_b, "scvtf_d64_float2fix"},
      {"010011"_b, "ucvtf_d64_float2fix"},
      {"011100"_b, "fcvtzs_64d_float2fix"},
      {"011101"_b, "fcvtzu_64d_float2fix"},
      {"110010"_b, "scvtf_h64_float2fix"},
      {"110011"_b, "ucvtf_h64_float2fix"},
      {"111100"_b, "fcvtzs_64h_float2fix"},
      {"111101"_b, "fcvtzu_64h_float2fix"},
    },
  },

  { "_qtqrmn",
    {18, 17, 12},
    { {"0x0"_b, "st1_asisdlsop_dx1_r1d"},
      {"100"_b, "st1_asisdlsop_dx1_r1d"},
      {"110"_b, "st1_asisdlsop_d1_i1d"},
    },
  },

  { "_qtxlsr",
    {13, 12, 11, 10},
    { {"1111"_b, "cas_c64_ldstexcl"},
    },
  },

  { "_qtxpky",
    {4},
    { {"0"_b, "cmphs_p_p_zi"},
      {"1"_b, "cmphi_p_p_zi"},
    },
  },

  { "_qvgtlh",
    {30, 23, 22, 11},
    { {"0001"_b, "strb_32b_ldst_regoff"},
      {"0011"_b, "ldrb_32b_ldst_regoff"},
      {"0100"_b, "_hjplhs"},
      {"0101"_b, "ldrsb_64b_ldst_regoff"},
      {"0111"_b, "ldrsb_32b_ldst_regoff"},
      {"1001"_b, "strh_32_ldst_regoff"},
      {"1011"_b, "ldrh_32_ldst_regoff"},
      {"1100"_b, "_vrzksz"},
      {"1101"_b, "ldrsh_64_ldst_regoff"},
      {"1111"_b, "ldrsh_32_ldst_regoff"},
    },
  },

  { "_qvjmmq",
    {30},
    { {"0"_b, "b_only_branch_imm"},
    },
  },

  { "_qvlnll",
    {22, 20, 11},
    { {"010"_b, "decw_r_rs"},
      {"110"_b, "decd_r_rs"},
    },
  },

  { "_qvtxpr",
    {20, 9, 4},
    { {"000"_b, "uzp1_p_pp"},
    },
  },

  { "_qvzvmq",
    {30, 23, 22},
    { {"000"_b, "stlxrb_sr32_ldstexcl"},
      {"001"_b, "_ynznxv"},
      {"010"_b, "_lqlrxp"},
      {"011"_b, "_grprpj"},
      {"100"_b, "stlxrh_sr32_ldstexcl"},
      {"101"_b, "_jgsryt"},
      {"110"_b, "_qyrqxp"},
      {"111"_b, "_qjqrgz"},
    },
  },

  { "_qyjvqr",
    {23, 18, 17, 16},
    { {"0000"_b, "sqxtnt_z_zz"},
    },
  },

  { "_qyrqxp",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "stlrh_sl32_ldstexcl"},
    },
  },

  { "_qyyrqq",
    {22, 13, 12},
    { {"000"_b, "swp_64_memop"},
      {"001"_b, "_ymghnh"},
      {"010"_b, "st64bv0_64_memop"},
      {"011"_b, "st64bv_64_memop"},
      {"100"_b, "swpl_64_memop"},
    },
  },

  { "_qzlvkm",
    {18},
    { {"0"_b, "st3_asisdlsop_hx3_r3h"},
      {"1"_b, "st3_asisdlsop_h3_i3h"},
    },
  },

  { "_qzmrnj",
    {23, 22},
    { {"00"_b, "dup_asimdins_dr_r"},
      {"01"_b, "fmla_asimdsamefp16_only"},
      {"11"_b, "fmls_asimdsamefp16_only"},
    },
  },

  { "_qzsyvx",
    {30, 23, 22, 11, 10},
    { {"00010"_b, "str_s_ldst_regoff"},
      {"00110"_b, "ldr_s_ldst_regoff"},
      {"10010"_b, "str_d_ldst_regoff"},
      {"10110"_b, "ldr_d_ldst_regoff"},
    },
  },

  { "_qzxgqh",
    {12},
    { {"0"_b, "ld3_asisdlsop_dx3_r3d"},
    },
  },

  { "_rgnryt",
    {18, 17, 12},
    { {"0x0"_b, "ld1_asisdlsop_dx1_r1d"},
      {"100"_b, "ld1_asisdlsop_dx1_r1d"},
      {"110"_b, "ld1_asisdlsop_d1_i1d"},
    },
  },

  { "_rgxthl",
    {30, 23, 22},
    { {"000"_b, "stxp_sp32_ldstexcl"},
      {"001"_b, "_mjyhsl"},
      {"010"_b, "_vrsjnp"},
      {"011"_b, "_zyxnpz"},
      {"100"_b, "stxp_sp64_ldstexcl"},
      {"101"_b, "_snrzky"},
      {"110"_b, "_qtxlsr"},
      {"111"_b, "_pkpvmj"},
    },
  },

  { "_rgztgm",
    {20, 18, 17},
    { {"000"_b, "_klrksl"},
    },
  },

  { "_rhhrhg",
    {30, 13, 4},
    { {"000"_b, "cmphs_p_p_zw"},
      {"001"_b, "cmphi_p_p_zw"},
      {"010"_b, "cmplo_p_p_zw"},
      {"011"_b, "cmpls_p_p_zw"},
    },
  },

  { "_rhmxyp",
    {20, 9, 4},
    { {"000"_b, "trn1_p_pp"},
    },
  },

  { "_rhpmjz",
    {12, 11},
    { {"00"_b, "incp_z_p_z"},
      {"01"_b, "incp_r_p_r"},
      {"10"_b, "_mpstrr"},
    },
  },

  { "_rjmhxr",
    {30},
    { {"0"_b, "adds_64_addsub_shift"},
      {"1"_b, "subs_64_addsub_shift"},
    },
  },

  { "_rjrqxt",
    {9, 8, 7, 6, 5},
    { {"00000"_b, "fmov_s_floatimm"},
    },
  },

  { "_rjspzr",
    {13, 12},
    { {"00"_b, "udiv_32_dp_2src"},
      {"10"_b, "asrv_32_dp_2src"},
    },
  },

  { "_rjthsm",
    {30, 23, 22},
    { {"001"_b, "sbfm_64m_bitfield"},
      {"101"_b, "ubfm_64m_bitfield"},
    },
  },

  { "_rjvgkl",
    {30, 23, 22, 19, 18, 17, 16},
    { {"000xxxx"_b, "umov_asimdins_w_w"},
      {"1001000"_b, "umov_asimdins_x_x"},
      {"x01xxxx"_b, "frecps_asimdsamefp16_only"},
      {"x11xxxx"_b, "frsqrts_asimdsamefp16_only"},
    },
  },

  { "_rjyrnt",
    {4},
    { {"0"_b, "cmpge_p_p_zi"},
      {"1"_b, "cmpgt_p_p_zi"},
    },
  },

  { "_rjysnh",
    {18, 17, 16, 9, 8, 7, 6},
    { {"0000000"_b, "fadd_z_p_zs"},
      {"0010000"_b, "fsub_z_p_zs"},
      {"0100000"_b, "fmul_z_p_zs"},
      {"0110000"_b, "fsubr_z_p_zs"},
      {"1000000"_b, "fmaxnm_z_p_zs"},
      {"1010000"_b, "fminnm_z_p_zs"},
      {"1100000"_b, "fmax_z_p_zs"},
      {"1110000"_b, "fmin_z_p_zs"},
    },
  },

  { "_rkjjtp",
    {23, 22, 20, 19, 11},
    { {"00010"_b, "scvtf_asisdshf_c"},
      {"001x0"_b, "scvtf_asisdshf_c"},
      {"01xx0"_b, "scvtf_asisdshf_c"},
    },
  },

  { "_rknxlg",
    {12},
    { {"0"_b, "ld4_asisdlsop_dx4_r4d"},
    },
  },

  { "_rkpylh",
    {20, 19, 18, 17, 16},
    { {"00010"_b, "scvtf_d32_float2fix"},
      {"00011"_b, "ucvtf_d32_float2fix"},
      {"11000"_b, "fcvtzs_32d_float2fix"},
      {"11001"_b, "fcvtzu_32d_float2fix"},
    },
  },

  { "_rkrlsy",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "rev64_asimdmisc_r"},
    },
  },

  { "_rkrntt",
    {23, 22, 20, 19, 17, 16, 13},
    { {"0000000"_b, "_hynprk"},
      {"0000001"_b, "_phpphm"},
      {"0100000"_b, "_tlvmlq"},
      {"0100001"_b, "_qpyxsv"},
      {"100xxx0"_b, "st2_asisdlsop_hx2_r2h"},
      {"100xxx1"_b, "st4_asisdlsop_hx4_r4h"},
      {"1010xx0"_b, "st2_asisdlsop_hx2_r2h"},
      {"1010xx1"_b, "st4_asisdlsop_hx4_r4h"},
      {"10110x0"_b, "st2_asisdlsop_hx2_r2h"},
      {"10110x1"_b, "st4_asisdlsop_hx4_r4h"},
      {"1011100"_b, "st2_asisdlsop_hx2_r2h"},
      {"1011101"_b, "st4_asisdlsop_hx4_r4h"},
      {"1011110"_b, "_skmzll"},
      {"1011111"_b, "_hkxlsm"},
      {"110xxx0"_b, "ld2_asisdlsop_hx2_r2h"},
      {"110xxx1"_b, "ld4_asisdlsop_hx4_r4h"},
      {"1110xx0"_b, "ld2_asisdlsop_hx2_r2h"},
      {"1110xx1"_b, "ld4_asisdlsop_hx4_r4h"},
      {"11110x0"_b, "ld2_asisdlsop_hx2_r2h"},
      {"11110x1"_b, "ld4_asisdlsop_hx4_r4h"},
      {"1111100"_b, "ld2_asisdlsop_hx2_r2h"},
      {"1111101"_b, "ld4_asisdlsop_hx4_r4h"},
      {"1111110"_b, "_ykhhqq"},
      {"1111111"_b, "_khtsmx"},
    },
  },

  { "_rkskkv",
    {18},
    { {"1"_b, "fminv_v_p_z"},
    },
  },

  { "_rktqym",
    {30, 23, 22, 13, 12, 11, 10},
    { {"010xx00"_b, "csel_32_condsel"},
      {"010xx01"_b, "csinc_32_condsel"},
      {"0110000"_b, "crc32b_32c_dp_2src"},
      {"0110001"_b, "crc32h_32c_dp_2src"},
      {"0110010"_b, "crc32w_32c_dp_2src"},
      {"0110100"_b, "crc32cb_32c_dp_2src"},
      {"0110101"_b, "crc32ch_32c_dp_2src"},
      {"0110110"_b, "crc32cw_32c_dp_2src"},
      {"0111000"_b, "smax_32_dp_2src"},
      {"0111001"_b, "umax_32_dp_2src"},
      {"0111010"_b, "smin_32_dp_2src"},
      {"0111011"_b, "umin_32_dp_2src"},
      {"110xx00"_b, "csinv_32_condsel"},
      {"110xx01"_b, "csneg_32_condsel"},
    },
  },

  { "_rkxlyj",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx10"_b, "stlur_b_ldapstl_simd"},
      {"001xx10"_b, "ldapur_b_ldapstl_simd"},
      {"010xx10"_b, "stlur_q_ldapstl_simd"},
      {"011xx10"_b, "ldapur_q_ldapstl_simd"},
      {"100xx10"_b, "stlur_h_ldapstl_simd"},
      {"101xx10"_b, "ldapur_h_ldapstl_simd"},
      {"x000001"_b, "cpyprn_cpy_memcms"},
      {"x000101"_b, "cpypwtrn_cpy_memcms"},
      {"x001001"_b, "cpyprtrn_cpy_memcms"},
      {"x001101"_b, "cpyptrn_cpy_memcms"},
      {"x010001"_b, "cpymrn_cpy_memcms"},
      {"x010101"_b, "cpymwtrn_cpy_memcms"},
      {"x011001"_b, "cpymrtrn_cpy_memcms"},
      {"x011101"_b, "cpymtrn_cpy_memcms"},
      {"x100001"_b, "cpyern_cpy_memcms"},
      {"x100101"_b, "cpyewtrn_cpy_memcms"},
      {"x101001"_b, "cpyertrn_cpy_memcms"},
      {"x101101"_b, "cpyetrn_cpy_memcms"},
      {"x110001"_b, "setge_set_memcms"},
      {"x110101"_b, "setget_set_memcms"},
      {"x111001"_b, "setgen_set_memcms"},
      {"x111101"_b, "setgetn_set_memcms"},
    },
  },

  { "_rkzlpp",
    {4},
    { {"0"_b, "ccmp_64_condcmp_reg"},
    },
  },

  { "_rlgtnn",
    {23},
    { {"0"_b, "_sxsgmq"},
    },
  },

  { "_rlpmrx",
    {30},
    { {"0"_b, "_txzxzs"},
      {"1"_b, "_htsjxj"},
    },
  },

  { "_rlrjxp",
    {13, 4},
    { {"00"_b, "fcmge_p_p_zz"},
      {"01"_b, "fcmgt_p_p_zz"},
      {"10"_b, "fcmeq_p_p_zz"},
      {"11"_b, "fcmne_p_p_zz"},
    },
  },

  { "_rlxhxz",
    {9, 8, 7, 6, 5},
    { {"11111"_b, "pacdzb_64z_dp_1src"},
    },
  },

  { "_rlylxh",
    {18},
    { {"0"_b, "ld3_asisdlsop_bx3_r3b"},
      {"1"_b, "ld3_asisdlsop_b3_i3b"},
    },
  },

  { "_rlyvpn",
    {23, 12, 11, 10},
    { {"0000"_b, "sqshrunb_z_zi"},
      {"0001"_b, "sqshrunt_z_zi"},
      {"0010"_b, "sqrshrunb_z_zi"},
      {"0011"_b, "sqrshrunt_z_zi"},
      {"0100"_b, "shrnb_z_zi"},
      {"0101"_b, "shrnt_z_zi"},
      {"0110"_b, "rshrnb_z_zi"},
      {"0111"_b, "rshrnt_z_zi"},
    },
  },

  { "_rmkpsk",
    {23},
    { {"0"_b, "_srkslp"},
    },
  },

  { "_rmmpym",
    {2, 1, 0},
    { {"000"_b, "_glgznt"},
    },
  },

  { "_rmyzpp",
    {20, 19, 18, 17},
    { {"0000"_b, "_gnhjkl"},
    },
  },

  { "_rnlxtv",
    {13},
    { {"0"_b, "_vvgpzq"},
      {"1"_b, "_mqljmr"},
    },
  },

  { "_rnphqp",
    {20, 19, 18, 17, 16, 4, 3},
    { {"0000001"_b, "fcmp_hz_floatcmp"},
      {"0000011"_b, "fcmpe_hz_floatcmp"},
      {"xxxxx00"_b, "fcmp_h_floatcmp"},
      {"xxxxx10"_b, "fcmpe_h_floatcmp"},
    },
  },

  { "_rnqmyp",
    {23, 22, 20, 19, 13, 11},
    { {"0000x0"_b, "mvni_asimdimm_l_sl"},
      {"00x100"_b, "sri_asimdshf_r"},
      {"00x110"_b, "sqshlu_asimdshf_r"},
      {"010x00"_b, "sri_asimdshf_r"},
      {"010x10"_b, "sqshlu_asimdshf_r"},
      {"011100"_b, "sri_asimdshf_r"},
      {"011110"_b, "sqshlu_asimdshf_r"},
      {"0x1000"_b, "sri_asimdshf_r"},
      {"0x1010"_b, "sqshlu_asimdshf_r"},
    },
  },

  { "_rnqtmt",
    {30},
    { {"0"_b, "_zyjjgs"},
      {"1"_b, "_lrntmz"},
    },
  },

  { "_rnsmjq",
    {13},
    { {"0"_b, "_xxqzvy"},
      {"1"_b, "_rmmpym"},
    },
  },

  { "_rpjgkh",
    {22, 20, 19, 18, 17, 16},
    { {"111000"_b, "fcmge_asisdmiscfp16_fz"},
      {"x00000"_b, "fcmge_asisdmisc_fz"},
      {"x10000"_b, "fminnmp_asisdpair_only_sd"},
    },
  },

  { "_rpjrhs",
    {23, 22, 4},
    { {"000"_b, "fccmp_s_floatccmp"},
      {"001"_b, "fccmpe_s_floatccmp"},
      {"010"_b, "fccmp_d_floatccmp"},
      {"011"_b, "fccmpe_d_floatccmp"},
      {"110"_b, "fccmp_h_floatccmp"},
      {"111"_b, "fccmpe_h_floatccmp"},
    },
  },

  { "_rpplns",
    {23, 22, 20, 19, 11},
    { {"00010"_b, "srshr_asisdshf_r"},
      {"001x0"_b, "srshr_asisdshf_r"},
      {"01xx0"_b, "srshr_asisdshf_r"},
    },
  },

  { "_rpzykx",
    {11},
    { {"0"_b, "_svvyrz"},
    },
  },

  { "_rqghyv",
    {30, 23, 22, 11, 10},
    { {"00000"_b, "stur_32_ldst_unscaled"},
      {"00001"_b, "str_32_ldst_immpost"},
      {"00010"_b, "sttr_32_ldst_unpriv"},
      {"00011"_b, "str_32_ldst_immpre"},
      {"00100"_b, "ldur_32_ldst_unscaled"},
      {"00101"_b, "ldr_32_ldst_immpost"},
      {"00110"_b, "ldtr_32_ldst_unpriv"},
      {"00111"_b, "ldr_32_ldst_immpre"},
      {"01000"_b, "ldursw_64_ldst_unscaled"},
      {"01001"_b, "ldrsw_64_ldst_immpost"},
      {"01010"_b, "ldtrsw_64_ldst_unpriv"},
      {"01011"_b, "ldrsw_64_ldst_immpre"},
      {"10000"_b, "stur_64_ldst_unscaled"},
      {"10001"_b, "str_64_ldst_immpost"},
      {"10010"_b, "sttr_64_ldst_unpriv"},
      {"10011"_b, "str_64_ldst_immpre"},
      {"10100"_b, "ldur_64_ldst_unscaled"},
      {"10101"_b, "ldr_64_ldst_immpost"},
      {"10110"_b, "ldtr_64_ldst_unpriv"},
      {"10111"_b, "ldr_64_ldst_immpre"},
      {"11000"_b, "prfum_p_ldst_unscaled"},
    },
  },

  { "_rqhryp",
    {12, 10},
    { {"00"_b, "_kjpxvh"},
      {"01"_b, "_mxvjxx"},
      {"10"_b, "sm4ekey_z_zz"},
      {"11"_b, "rax1_z_zz"},
    },
  },

  { "_rqpjjs",
    {30, 11, 10},
    { {"000"_b, "_qjrllr"},
      {"001"_b, "_xlgxhn"},
      {"010"_b, "_hxrnns"},
      {"011"_b, "_xnhkpk"},
      {"101"_b, "_mmgpkx"},
      {"110"_b, "_vxhjgg"},
      {"111"_b, "_lptrlg"},
    },
  },

  { "_rqzpzq",
    {23, 22, 11, 10, 4, 3, 0},
    { {"0000000"_b, "_hkxzqg"},
      {"0010111"_b, "_zqlzzp"},
      {"0011111"_b, "_lvszgj"},
      {"0100000"_b, "_tmsjzg"},
      {"0110111"_b, "_kzprzt"},
      {"0111111"_b, "_tzsnmy"},
      {"1000000"_b, "_mqmrng"},
      {"1010111"_b, "_hrmsnk"},
      {"1011111"_b, "_tqlrzh"},
    },
  },

  { "_rrkmyl",
    {23, 22, 4},
    { {"000"_b, "fccmp_s_floatccmp"},
      {"001"_b, "fccmpe_s_floatccmp"},
      {"010"_b, "fccmp_d_floatccmp"},
      {"011"_b, "fccmpe_d_floatccmp"},
      {"110"_b, "fccmp_h_floatccmp"},
      {"111"_b, "fccmpe_h_floatccmp"},
    },
  },

  { "_rrvltp",
    {18, 4},
    { {"00"_b, "fcmlt_p_p_z0"},
      {"01"_b, "fcmle_p_p_z0"},
    },
  },

  { "_rshyht",
    {13},
    { {"0"_b, "facge_p_p_zz"},
      {"1"_b, "facgt_p_p_zz"},
    },
  },

  { "_rsjgyk",
    {30, 23, 22, 20, 13},
    { {"00001"_b, "ld2w_z_p_bi_contiguous"},
      {"000x0"_b, "ld2w_z_p_br_contiguous"},
      {"00101"_b, "ld4w_z_p_bi_contiguous"},
      {"001x0"_b, "ld4w_z_p_br_contiguous"},
      {"01001"_b, "ld2d_z_p_bi_contiguous"},
      {"010x0"_b, "ld2d_z_p_br_contiguous"},
      {"01101"_b, "ld4d_z_p_bi_contiguous"},
      {"011x0"_b, "ld4d_z_p_br_contiguous"},
      {"10011"_b, "st2w_z_p_bi_contiguous"},
      {"100x0"_b, "st1w_z_p_bz_d_x32_scaled"},
      {"10101"_b, "st1w_z_p_bi"},
      {"10111"_b, "st4w_z_p_bi_contiguous"},
      {"101x0"_b, "st1w_z_p_bz_s_x32_scaled"},
      {"11011"_b, "st2d_z_p_bi_contiguous"},
      {"110x0"_b, "st1d_z_p_bz_d_x32_scaled"},
      {"11101"_b, "st1d_z_p_bi"},
      {"11111"_b, "st4d_z_p_bi_contiguous"},
    },
  },

  { "_rsmyth",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "stllr_sl64_ldstexcl"},
    },
  },

  { "_rsnvnr",
    {30, 23, 22},
    { {"100"_b, "ins_asimdins_ir_r"},
      {"x01"_b, "fmulx_asimdsamefp16_only"},
    },
  },

  { "_rspmth",
    {18},
    { {"0"_b, "st1_asisdlse_r2_2v"},
    },
  },

  { "_rsqmgk",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0000000"_b, "movprfx_z_z"},
    },
  },

  { "_rsqxrs",
    {30, 23, 22, 11, 10},
    { {"00000"_b, "_ggvlym"},
      {"01000"_b, "csel_32_condsel"},
      {"01001"_b, "csinc_32_condsel"},
      {"01100"_b, "_svvylr"},
      {"01101"_b, "_zmhqmr"},
      {"01110"_b, "_rjspzr"},
      {"01111"_b, "_vpknjg"},
      {"10000"_b, "_rzymmk"},
      {"11000"_b, "csinv_32_condsel"},
      {"11001"_b, "csneg_32_condsel"},
      {"11100"_b, "_kzjxxk"},
      {"11101"_b, "_khvvtr"},
      {"11110"_b, "_gvpvjn"},
      {"11111"_b, "_pkjqsy"},
    },
  },

  { "_rssrty",
    {30, 23, 22, 13, 12, 11, 10},
    { {"1011011"_b, "bfmmla_asimdsame2_e"},
      {"x011111"_b, "bfdot_asimdsame2_d"},
      {"x111111"_b, "bfmlal_asimdsame2_f"},
      {"xxx0xx1"_b, "fcmla_asimdsame2_c"},
      {"xxx1x01"_b, "fcadd_asimdsame2_c"},
    },
  },

  { "_rszgzl",
    {30, 23, 22},
    { {"000"_b, "smsubl_64wa_dp_3src"},
      {"010"_b, "umsubl_64wa_dp_3src"},
    },
  },

  { "_rtlvxq",
    {30, 23, 22},
    { {"000"_b, "madd_32a_dp_3src"},
    },
  },

  { "_rtpztp",
    {22},
    { {"0"_b, "umullb_z_zzi_s"},
      {"1"_b, "umullb_z_zzi_d"},
    },
  },

  { "_rtrlts",
    {23, 22, 12, 11, 10},
    { {"01000"_b, "bfdot_z_zzz"},
      {"10000"_b, "fmlalb_z_zzz"},
      {"10001"_b, "fmlalt_z_zzz"},
      {"11000"_b, "bfmlalb_z_zzz"},
      {"11001"_b, "bfmlalt_z_zzz"},
    },
  },

  { "_rvjkyp",
    {13, 12},
    { {"01"_b, "gmi_64g_dp_2src"},
      {"10"_b, "lsrv_64_dp_2src"},
    },
  },

  { "_rvsylx",
    {18},
    { {"1"_b, "frecpe_z_z"},
    },
  },

  { "_rvtxys",
    {23, 22, 20, 19, 11},
    { {"00010"_b, "sshr_asisdshf_r"},
      {"001x0"_b, "sshr_asisdshf_r"},
      {"01xx0"_b, "sshr_asisdshf_r"},
    },
  },

  { "_rvvshx",
    {23, 22, 13, 12},
    { {"0000"_b, "fmax_s_floatdp2"},
      {"0001"_b, "fmin_s_floatdp2"},
      {"0010"_b, "fmaxnm_s_floatdp2"},
      {"0011"_b, "fminnm_s_floatdp2"},
      {"0100"_b, "fmax_d_floatdp2"},
      {"0101"_b, "fmin_d_floatdp2"},
      {"0110"_b, "fmaxnm_d_floatdp2"},
      {"0111"_b, "fminnm_d_floatdp2"},
      {"1100"_b, "fmax_h_floatdp2"},
      {"1101"_b, "fmin_h_floatdp2"},
      {"1110"_b, "fmaxnm_h_floatdp2"},
      {"1111"_b, "fminnm_h_floatdp2"},
    },
  },

  { "_rxgkjn",
    {30, 23, 22},
    { {"000"_b, "adds_64s_addsub_ext"},
      {"100"_b, "subs_64s_addsub_ext"},
    },
  },

  { "_rxhssh",
    {18},
    { {"0"_b, "ld3_asisdlsop_hx3_r3h"},
      {"1"_b, "ld3_asisdlsop_h3_i3h"},
    },
  },

  { "_rxnnvv",
    {23, 22, 4, 3, 2, 1, 0},
    { {"0000000"_b, "brk_ex_exception"},
      {"0100000"_b, "tcancel_ex_exception"},
      {"1000001"_b, "dcps1_dc_exception"},
      {"1000010"_b, "dcps2_dc_exception"},
      {"1000011"_b, "dcps3_dc_exception"},
    },
  },

  { "_rxsqhv",
    {13, 12},
    { {"00"_b, "adc_64_addsub_carry"},
    },
  },

  { "_rxtklv",
    {30, 18},
    { {"00"_b, "_qtgvlx"},
    },
  },

  { "_rxytqg",
    {30, 23, 22, 20, 19, 18},
    { {"00xxxx"_b, "add_64_addsub_imm"},
      {"011000"_b, "smax_64_minmax_imm"},
      {"011001"_b, "umax_64u_minmax_imm"},
      {"011010"_b, "smin_64_minmax_imm"},
      {"011011"_b, "umin_64u_minmax_imm"},
      {"10xxxx"_b, "sub_64_addsub_imm"},
    },
  },

  { "_ryrkqt",
    {20, 19},
    { {"00"_b, "_tsskys"},
      {"01"_b, "_kqvljp"},
      {"10"_b, "_lxhlkx"},
      {"11"_b, "_rjysnh"},
    },
  },

  { "_rznrqt",
    {22},
    { {"0"_b, "umullt_z_zzi_s"},
      {"1"_b, "umullt_z_zzi_d"},
    },
  },

  { "_rzpqmm",
    {23, 22, 20, 19, 17, 16, 13},
    { {"0000000"_b, "_nygsjm"},
      {"0000001"_b, "_snjmrt"},
      {"0100000"_b, "_hhxpjz"},
      {"0100001"_b, "_tktgvg"},
      {"100xxx0"_b, "st2_asisdlsop_bx2_r2b"},
      {"100xxx1"_b, "st4_asisdlsop_bx4_r4b"},
      {"1010xx0"_b, "st2_asisdlsop_bx2_r2b"},
      {"1010xx1"_b, "st4_asisdlsop_bx4_r4b"},
      {"10110x0"_b, "st2_asisdlsop_bx2_r2b"},
      {"10110x1"_b, "st4_asisdlsop_bx4_r4b"},
      {"1011100"_b, "st2_asisdlsop_bx2_r2b"},
      {"1011101"_b, "st4_asisdlsop_bx4_r4b"},
      {"1011110"_b, "_szjjgk"},
      {"1011111"_b, "_tvgklq"},
      {"110xxx0"_b, "ld2_asisdlsop_bx2_r2b"},
      {"110xxx1"_b, "ld4_asisdlsop_bx4_r4b"},
      {"1110xx0"_b, "ld2_asisdlsop_bx2_r2b"},
      {"1110xx1"_b, "ld4_asisdlsop_bx4_r4b"},
      {"11110x0"_b, "ld2_asisdlsop_bx2_r2b"},
      {"11110x1"_b, "ld4_asisdlsop_bx4_r4b"},
      {"1111100"_b, "ld2_asisdlsop_bx2_r2b"},
      {"1111101"_b, "ld4_asisdlsop_bx4_r4b"},
      {"1111110"_b, "_tzsvyv"},
      {"1111111"_b, "_jvnsgt"},
    },
  },

  { "_rztvnl",
    {20, 19, 18, 17, 16},
    { {"0000x"_b, "fcadd_z_p_zz"},
      {"10000"_b, "faddp_z_p_zz"},
      {"10100"_b, "fmaxnmp_z_p_zz"},
      {"10101"_b, "fminnmp_z_p_zz"},
      {"10110"_b, "fmaxp_z_p_zz"},
      {"10111"_b, "fminp_z_p_zz"},
    },
  },

  { "_rzymmk",
    {13, 12},
    { {"00"_b, "sbc_32_addsub_carry"},
    },
  },

  { "_rzzxsn",
    {30, 13},
    { {"00"_b, "_nvyxmh"},
      {"01"_b, "_hykhmt"},
      {"10"_b, "_yszjsm"},
      {"11"_b, "_jrnxzh"},
    },
  },

  { "_sghgtk",
    {4},
    { {"0"_b, "cmplo_p_p_zi"},
      {"1"_b, "cmpls_p_p_zi"},
    },
  },

  { "_sgmpvp",
    {23, 22, 13},
    { {"000"_b, "fmulx_asimdelem_rh_h"},
      {"1x0"_b, "fmulx_asimdelem_r_sd"},
    },
  },

  { "_shgktt",
    {11},
    { {"0"_b, "_tjjqpx"},
    },
  },

  { "_shgxyq",
    {23, 22, 19, 13, 12},
    { {"00100"_b, "sha1h_ss_cryptosha2"},
      {"00101"_b, "sha1su1_vv_cryptosha2"},
      {"00110"_b, "sha256su0_vv_cryptosha2"},
      {"xx011"_b, "suqadd_asisdmisc_r"},
    },
  },

  { "_shqygv",
    {30, 4},
    { {"00"_b, "_thvxym"},
      {"01"_b, "_mrhtxt"},
      {"10"_b, "_ptjyqx"},
      {"11"_b, "_rshyht"},
    },
  },

  { "_shqyqv",
    {23, 13, 12},
    { {"010"_b, "fcmeq_asisdsame_only"},
    },
  },

  { "_shvqkt",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldlar_lr32_ldstexcl"},
    },
  },

  { "_sjlqvg",
    {23, 20, 19, 18, 17, 16},
    { {"000001"_b, "fcvtxn_asimdmisc_n"},
      {"x00000"_b, "uadalp_asimdmisc_p"},
    },
  },

  { "_sjnspg",
    {4},
    { {"0"_b, "nors_p_p_pp_z"},
      {"1"_b, "nands_p_p_pp_z"},
    },
  },

  { "_sjtrhm",
    {30, 23, 22, 20, 13},
    { {"00001"_b, "ld1rqb_z_p_bi_u8"},
      {"000x0"_b, "ld1rqb_z_p_br_contiguous"},
      {"01001"_b, "ld1rqh_z_p_bi_u16"},
      {"010x0"_b, "ld1rqh_z_p_br_contiguous"},
      {"100x1"_b, "stnt1b_z_p_ar_d_64_unscaled"},
      {"101x1"_b, "stnt1b_z_p_ar_s_x32_unscaled"},
      {"110x1"_b, "stnt1h_z_p_ar_d_64_unscaled"},
      {"111x1"_b, "stnt1h_z_p_ar_s_x32_unscaled"},
    },
  },

  { "_sjvhlq",
    {22},
    { {"0"_b, "smullb_z_zzi_s"},
      {"1"_b, "smullb_z_zzi_d"},
    },
  },

  { "_skjqrx",
    {23, 22},
    { {"00"_b, "fmov_s_floatimm"},
      {"01"_b, "fmov_d_floatimm"},
      {"11"_b, "fmov_h_floatimm"},
    },
  },

  { "_skmzll",
    {18},
    { {"0"_b, "st2_asisdlsop_hx2_r2h"},
      {"1"_b, "st2_asisdlsop_h2_i2h"},
    },
  },

  { "_sknvhk",
    {13, 12, 11, 10},
    { {"0000"_b, "sha1c_qsv_cryptosha3"},
      {"0001"_b, "dup_asisdone_only"},
      {"0100"_b, "sha1p_qsv_cryptosha3"},
      {"1000"_b, "sha1m_qsv_cryptosha3"},
      {"1100"_b, "sha1su0_vvv_cryptosha3"},
    },
  },

  { "_skqzyg",
    {23},
    { {"0"_b, "fcmeq_asimdsame_only"},
    },
  },

  { "_sksvrn",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldlarb_lr32_ldstexcl"},
    },
  },

  { "_skszgm",
    {13, 12, 11, 10},
    { {"1111"_b, "_xzmrlg"},
    },
  },

  { "_skytvx",
    {23, 22},
    { {"00"_b, "tbx_asimdtbl_l2_2"},
    },
  },

  { "_slzrtr",
    {23, 22},
    { {"00"_b, "fmsub_s_floatdp3"},
      {"01"_b, "fmsub_d_floatdp3"},
      {"11"_b, "fmsub_h_floatdp3"},
    },
  },

  { "_slzvjh",
    {30, 23, 22},
    { {"000"_b, "orr_32_log_imm"},
      {"100"_b, "ands_32s_log_imm"},
      {"110"_b, "movk_32_movewide"},
    },
  },

  { "_smmrpj",
    {18},
    { {"0"_b, "fadda_v_p_z"},
    },
  },

  { "_smptxh",
    {23, 22},
    { {"01"_b, "fmax_asimdsamefp16_only"},
      {"11"_b, "fmin_asimdsamefp16_only"},
    },
  },

  { "_smsytm",
    {13},
    { {"0"_b, "mul_asimdelem_r"},
      {"1"_b, "smull_asimdelem_l"},
    },
  },

  { "_snhmgn",
    {23},
    { {"0"_b, "fmul_asimdsame_only"},
    },
  },

  { "_snhzxr",
    {30, 23, 22},
    { {"001"_b, "bfm_64m_bitfield"},
    },
  },

  { "_snjmrt",
    {18},
    { {"0"_b, "st4_asisdlso_b4_4b"},
    },
  },

  { "_snnlgr",
    {23, 22, 20, 19, 13, 11},
    { {"0000x0"_b, "movi_asimdimm_l_sl"},
      {"00x100"_b, "sshr_asimdshf_r"},
      {"00x110"_b, "srshr_asimdshf_r"},
      {"010x00"_b, "sshr_asimdshf_r"},
      {"010x10"_b, "srshr_asimdshf_r"},
      {"011100"_b, "sshr_asimdshf_r"},
      {"011110"_b, "srshr_asimdshf_r"},
      {"0x1000"_b, "sshr_asimdshf_r"},
      {"0x1010"_b, "srshr_asimdshf_r"},
    },
  },

  { "_snrzky",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldxp_lp64_ldstexcl"},
    },
  },

  { "_sntnsm",
    {9, 8, 7, 6, 5},
    { {"11111"_b, "autizb_64z_dp_1src"},
    },
  },

  { "_sntyqy",
    {4},
    { {"0"_b, "cmphs_p_p_zi"},
      {"1"_b, "cmphi_p_p_zi"},
    },
  },

  { "_snvnjz",
    {30, 13},
    { {"10"_b, "_plzqrv"},
    },
  },

  { "_snvzjr",
    {12},
    { {"0"_b, "st2_asisdlsop_dx2_r2d"},
    },
  },

  { "_snzvtt",
    {23, 22},
    { {"00"_b, "fmlal2_asimdsame_f"},
      {"10"_b, "fmlsl2_asimdsame_f"},
    },
  },

  { "_spktyg",
    {23, 22, 20, 19, 11},
    { {"00000"_b, "movi_asimdimm_m_sm"},
    },
  },

  { "_spxvlt",
    {20, 19, 18, 17, 16, 13, 12, 11},
    { {"00000000"_b, "_mtkhgz"},
    },
  },

  { "_sqgjmn",
    {20, 9},
    { {"00"_b, "_mxgykv"},
    },
  },

  { "_sqhxzj",
    {30, 23, 22, 13, 12, 11, 10},
    { {"1010000"_b, "sha512h_qqv_cryptosha512_3"},
      {"1010001"_b, "sha512h2_qqv_cryptosha512_3"},
      {"1010010"_b, "sha512su1_vvv2_cryptosha512_3"},
      {"1010011"_b, "rax1_vvv2_cryptosha512_3"},
    },
  },

  { "_sqkkqy",
    {13, 12, 10},
    { {"010"_b, "sqrdmlah_asisdelem_r"},
      {"101"_b, "_mhksnq"},
      {"110"_b, "sqrdmlsh_asisdelem_r"},
      {"111"_b, "_mpytmv"},
    },
  },

  { "_sqlsyr",
    {18, 17},
    { {"00"_b, "ld1_asisdlse_r1_1v"},
    },
  },

  { "_sqttsv",
    {20, 19, 18, 17, 16, 4, 3},
    { {"0000001"_b, "fcmp_sz_floatcmp"},
      {"0000011"_b, "fcmpe_sz_floatcmp"},
      {"xxxxx00"_b, "fcmp_s_floatcmp"},
      {"xxxxx10"_b, "fcmpe_s_floatcmp"},
    },
  },

  { "_srkslp",
    {22, 20},
    { {"00"_b, "_zvynrg"},
      {"01"_b, "msr_sr_systemmove"},
      {"10"_b, "_lxlqks"},
      {"11"_b, "msrr_sr_systemmovepr"},
    },
  },

  { "_srnkng",
    {18},
    { {"0"_b, "faddv_v_p_z"},
      {"1"_b, "fmaxnmv_v_p_z"},
    },
  },

  { "_srpptk",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "usqadd_asimdmisc_r"},
      {"00001"_b, "shll_asimdmisc_s"},
      {"10000"_b, "uaddlv_asimdall_only"},
    },
  },

  { "_srpqmk",
    {30, 23, 22},
    { {"000"_b, "stp_q_ldstpair_off"},
      {"001"_b, "ldp_q_ldstpair_off"},
      {"010"_b, "stp_q_ldstpair_pre"},
      {"011"_b, "ldp_q_ldstpair_pre"},
    },
  },

  { "_srsrtk",
    {30, 23, 22, 13, 11, 10},
    { {"000010"_b, "str_b_ldst_regoff"},
      {"000110"_b, "str_bl_ldst_regoff"},
      {"001010"_b, "ldr_b_ldst_regoff"},
      {"001110"_b, "ldr_bl_ldst_regoff"},
      {"010x10"_b, "str_q_ldst_regoff"},
      {"011x10"_b, "ldr_q_ldst_regoff"},
      {"100x10"_b, "str_h_ldst_regoff"},
      {"101x10"_b, "ldr_h_ldst_regoff"},
    },
  },

  { "_srttng",
    {23, 22},
    { {"01"_b, "fcmla_asimdelem_c_h"},
      {"10"_b, "fcmla_asimdelem_c_s"},
    },
  },

  { "_ssjnph",
    {10},
    { {"0"_b, "blraa_64p_branch_reg"},
      {"1"_b, "blrab_64p_branch_reg"},
    },
  },

  { "_ssjrxs",
    {18},
    { {"0"_b, "ld3_asisdlso_h3_3h"},
    },
  },

  { "_ssqyrk",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0000000"_b, "not_asimdmisc_r"},
      {"0100000"_b, "rbit_asimdmisc_r"},
    },
  },

  { "_ssvpxz",
    {30, 23, 22},
    { {"000"_b, "stnp_32_ldstnapair_offs"},
      {"001"_b, "ldnp_32_ldstnapair_offs"},
      {"010"_b, "stp_32_ldstpair_post"},
      {"011"_b, "ldp_32_ldstpair_post"},
      {"110"_b, "stgp_64_ldstpair_post"},
      {"111"_b, "ldpsw_64_ldstpair_post"},
    },
  },

  { "_ssypmm",
    {9, 8, 7, 6, 5},
    { {"00000"_b, "fmov_h_floatimm"},
    },
  },

  { "_stlgrr",
    {30, 23, 22, 13, 12, 11, 10},
    { {"0001111"_b, "caspl_cp32_ldstexcl"},
      {"0011111"_b, "caspal_cp32_ldstexcl"},
      {"0101111"_b, "caslb_c32_ldstexcl"},
      {"0111111"_b, "casalb_c32_ldstexcl"},
      {"1001111"_b, "caspl_cp64_ldstexcl"},
      {"1011111"_b, "caspal_cp64_ldstexcl"},
      {"1101111"_b, "caslh_c32_ldstexcl"},
      {"1111111"_b, "casalh_c32_ldstexcl"},
    },
  },

  { "_stmtkr",
    {30, 23, 22},
    { {"000"_b, "stxr_sr32_ldstexcl"},
      {"001"_b, "_zlvjrh"},
      {"010"_b, "_lpzgvs"},
      {"011"_b, "_shvqkt"},
      {"100"_b, "stxr_sr64_ldstexcl"},
      {"101"_b, "_jhltlz"},
      {"110"_b, "_rsmyth"},
      {"111"_b, "_vjtgmx"},
    },
  },

  { "_svgvjm",
    {23},
    { {"0"_b, "faddp_asimdsame_only"},
      {"1"_b, "fabd_asimdsame_only"},
    },
  },

  { "_svlrvy",
    {18, 17},
    { {"00"_b, "st1_asisdlse_r3_3v"},
    },
  },

  { "_svvylr",
    {13, 12},
    { {"10"_b, "lslv_32_dp_2src"},
    },
  },

  { "_svvyrz",
    {23, 22, 20, 19, 18, 17, 16},
    { {"00xxxxx"_b, "addvl_r_ri"},
      {"01xxxxx"_b, "addpl_r_ri"},
      {"1011111"_b, "rdvl_r_i"},
    },
  },

  { "_svyszp",
    {9, 8, 7, 6, 5},
    { {"00000"_b, "fmov_d_floatimm"},
    },
  },

  { "_sxgnmg",
    {18, 17},
    { {"00"_b, "ld3_asisdlso_s3_3s"},
    },
  },

  { "_sxptnh",
    {23, 22, 11, 10},
    { {"0000"_b, "_vmtkqp"},
      {"0001"_b, "_lqjlkj"},
      {"0010"_b, "_gyymmx"},
      {"0011"_b, "_gmqyjv"},
      {"0100"_b, "_pvtyjz"},
      {"0101"_b, "_hxxxyy"},
      {"0110"_b, "_xszmjn"},
      {"1000"_b, "_lzjyhm"},
      {"1001"_b, "_zlkygr"},
      {"1010"_b, "_jvpjsm"},
      {"1101"_b, "_vzyklr"},
      {"1110"_b, "_npxkzq"},
    },
  },

  { "_sxpvym",
    {30, 23, 22, 13},
    { {"0000"_b, "ldnt1sb_z_p_ar_s_x32_unscaled"},
      {"0001"_b, "ldnt1b_z_p_ar_s_x32_unscaled"},
      {"0010"_b, "ld1rb_z_p_bi_u8"},
      {"0011"_b, "ld1rb_z_p_bi_u16"},
      {"0100"_b, "ldnt1sh_z_p_ar_s_x32_unscaled"},
      {"0101"_b, "ldnt1h_z_p_ar_s_x32_unscaled"},
      {"0110"_b, "ld1rsw_z_p_bi_s64"},
      {"0111"_b, "ld1rh_z_p_bi_u16"},
      {"1000"_b, "ldnt1sb_z_p_ar_d_64_unscaled"},
      {"1010"_b, "ld1sb_z_p_bz_d_64_unscaled"},
      {"1011"_b, "ldff1sb_z_p_bz_d_64_unscaled"},
      {"1100"_b, "ldnt1sh_z_p_ar_d_64_unscaled"},
      {"1110"_b, "ld1sh_z_p_bz_d_64_unscaled"},
      {"1111"_b, "ldff1sh_z_p_bz_d_64_unscaled"},
    },
  },

  { "_sxsgmq",
    {30, 22, 20, 19, 18, 17, 16},
    { {"00xxxxx"_b, "stxp_sp32_ldstexcl"},
      {"0111111"_b, "ldxp_lp32_ldstexcl"},
      {"10xxxxx"_b, "stxp_sp64_ldstexcl"},
      {"1111111"_b, "ldxp_lp64_ldstexcl"},
    },
  },

  { "_sxsxxt",
    {20, 19, 18, 17, 16},
    { {"10000"_b, "fminp_asisdpair_only_sd"},
    },
  },

  { "_sylkvm",
    {23, 22, 12},
    { {"100"_b, "fmlsl2_asimdelem_lh"},
      {"xx1"_b, "sqrdmlah_asimdelem_r"},
    },
  },

  { "_syrmmr",
    {18, 4},
    { {"00"_b, "fcmeq_p_p_z0"},
    },
  },

  { "_szgqrr",
    {12, 10},
    { {"00"_b, "_xlyjsz"},
      {"01"_b, "_yppmkl"},
      {"10"_b, "_sgmpvp"},
      {"11"_b, "_gjtmjg"},
    },
  },

  { "_szjjgk",
    {18},
    { {"0"_b, "st2_asisdlsop_bx2_r2b"},
      {"1"_b, "st2_asisdlsop_b2_i2b"},
    },
  },

  { "_szmnhg",
    {12},
    { {"0"_b, "ld2_asisdlsop_dx2_r2d"},
    },
  },

  { "_szmyzt",
    {12},
    { {"0"_b, "ld3_asisdlsop_dx3_r3d"},
    },
  },

  { "_szqlsn",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0x00001"_b, "frint32z_asimdmisc_r"},
      {"1111000"_b, "fcmlt_asimdmiscfp16_fz"},
      {"1x00000"_b, "fcmlt_asimdmisc_fz"},
    },
  },

  { "_sztkhs",
    {30, 23, 22},
    { {"000"_b, "msub_64a_dp_3src"},
    },
  },

  { "_szylpy",
    {22, 12},
    { {"10"_b, "_hhlmrg"},
    },
  },

  { "_szysqh",
    {22, 13, 12},
    { {"000"_b, "ldsmax_32_memop"},
      {"001"_b, "ldsmin_32_memop"},
      {"010"_b, "ldumax_32_memop"},
      {"011"_b, "ldumin_32_memop"},
      {"100"_b, "ldsmaxl_32_memop"},
      {"101"_b, "ldsminl_32_memop"},
      {"110"_b, "ldumaxl_32_memop"},
      {"111"_b, "lduminl_32_memop"},
    },
  },

  { "_tgvkhm",
    {20, 19, 18, 17, 16, 13},
    { {"000000"_b, "fabs_s_floatdp1"},
      {"000010"_b, "fsqrt_s_floatdp1"},
      {"000100"_b, "fcvt_ds_floatdp1"},
      {"000110"_b, "fcvt_hs_floatdp1"},
      {"001000"_b, "frintp_s_floatdp1"},
      {"001010"_b, "frintz_s_floatdp1"},
      {"001110"_b, "frinti_s_floatdp1"},
      {"010000"_b, "frint32x_s_floatdp1"},
      {"010010"_b, "frint64x_s_floatdp1"},
    },
  },

  { "_thkkgx",
    {18},
    { {"1"_b, "fminnmv_v_p_z"},
    },
  },

  { "_thqgrq",
    {13, 12, 11, 10},
    { {"1111"_b, "_pgmlrt"},
    },
  },

  { "_thrxph",
    {23, 22, 10},
    { {"100"_b, "umlalb_z_zzzi_s"},
      {"101"_b, "umlalt_z_zzzi_s"},
      {"110"_b, "umlalb_z_zzzi_d"},
      {"111"_b, "umlalt_z_zzzi_d"},
    },
  },

  { "_thvxym",
    {20},
    { {"0"_b, "_prkmty"},
      {"1"_b, "_pjgkjs"},
    },
  },

  { "_tjjqpx",
    {23, 22, 20, 19, 16, 13, 10},
    { {"0000000"_b, "_mlgmqm"},
      {"0000001"_b, "_mvqkzv"},
      {"0000010"_b, "_jztspt"},
      {"0000011"_b, "_hrpkqg"},
      {"0100000"_b, "_llqtkj"},
      {"0100001"_b, "_pmpsvs"},
      {"0100010"_b, "_vhrkvk"},
      {"0100011"_b, "_xsvpzx"},
      {"100xx00"_b, "st2_asisdlsop_sx2_r2s"},
      {"100xx01"_b, "_ynyqky"},
      {"100xx10"_b, "st4_asisdlsop_sx4_r4s"},
      {"100xx11"_b, "_grvxrm"},
      {"1010x00"_b, "st2_asisdlsop_sx2_r2s"},
      {"1010x01"_b, "_snvzjr"},
      {"1010x10"_b, "st4_asisdlsop_sx4_r4s"},
      {"1010x11"_b, "_xmkysx"},
      {"1011000"_b, "st2_asisdlsop_sx2_r2s"},
      {"1011001"_b, "_xqhxql"},
      {"1011010"_b, "st4_asisdlsop_sx4_r4s"},
      {"1011011"_b, "_ykpqth"},
      {"1011100"_b, "_lgyqpk"},
      {"1011101"_b, "_tplghv"},
      {"1011110"_b, "_lqknkn"},
      {"1011111"_b, "_zprgxt"},
      {"110xx00"_b, "ld2_asisdlsop_sx2_r2s"},
      {"110xx01"_b, "_prjzxs"},
      {"110xx10"_b, "ld4_asisdlsop_sx4_r4s"},
      {"110xx11"_b, "_txsvzz"},
      {"1110x00"_b, "ld2_asisdlsop_sx2_r2s"},
      {"1110x01"_b, "_hljttg"},
      {"1110x10"_b, "ld4_asisdlsop_sx4_r4s"},
      {"1110x11"_b, "_rknxlg"},
      {"1111000"_b, "ld2_asisdlsop_sx2_r2s"},
      {"1111001"_b, "_szmnhg"},
      {"1111010"_b, "ld4_asisdlsop_sx4_r4s"},
      {"1111011"_b, "_tjrtxx"},
      {"1111100"_b, "_ppvnly"},
      {"1111101"_b, "_lltzjg"},
      {"1111110"_b, "_ypsgqz"},
      {"1111111"_b, "_vnrlsj"},
    },
  },

  { "_tjlthk",
    {9, 8, 7, 6, 5, 1},
    { {"111110"_b, "drps_64e_branch_reg"},
    },
  },

  { "_tjnzjl",
    {18, 17},
    { {"00"_b, "st1_asisdlso_s1_1s"},
    },
  },

  { "_tjrtxx",
    {12},
    { {"0"_b, "ld4_asisdlsop_dx4_r4d"},
    },
  },

  { "_tjxhsy",
    {10},
    { {"0"_b, "braa_64p_branch_reg"},
      {"1"_b, "brab_64p_branch_reg"},
    },
  },

  { "_tjxyky",
    {12},
    { {"0"_b, "st3_asisdlsop_dx3_r3d"},
    },
  },

  { "_tjzqnp",
    {30, 23, 22, 20, 13},
    { {"00001"_b, "ldnt1b_z_p_bi_contiguous"},
      {"000x0"_b, "ldnt1b_z_p_br_contiguous"},
      {"00101"_b, "ld3b_z_p_bi_contiguous"},
      {"001x0"_b, "ld3b_z_p_br_contiguous"},
      {"01001"_b, "ldnt1h_z_p_bi_contiguous"},
      {"010x0"_b, "ldnt1h_z_p_br_contiguous"},
      {"01101"_b, "ld3h_z_p_bi_contiguous"},
      {"011x0"_b, "ld3h_z_p_br_contiguous"},
      {"10011"_b, "stnt1b_z_p_bi_contiguous"},
      {"100x0"_b, "st1b_z_p_bz_d_x32_unscaled"},
      {"10111"_b, "st3b_z_p_bi_contiguous"},
      {"101x0"_b, "st1b_z_p_bz_s_x32_unscaled"},
      {"10x01"_b, "st1b_z_p_bi"},
      {"11011"_b, "stnt1h_z_p_bi_contiguous"},
      {"110x0"_b, "st1h_z_p_bz_d_x32_unscaled"},
      {"11111"_b, "st3h_z_p_bi_contiguous"},
      {"111x0"_b, "st1h_z_p_bz_s_x32_unscaled"},
      {"11x01"_b, "st1h_z_p_bi"},
    },
  },

  { "_tkjtgp",
    {30},
    { {"0"_b, "_sqgjmn"},
      {"1"_b, "_ztpryr"},
    },
  },

  { "_tklxhy",
    {18},
    { {"0"_b, "st3_asisdlso_b3_3b"},
    },
  },

  { "_tknqxs",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldaxr_lr64_ldstexcl"},
    },
  },

  { "_tktgvg",
    {18},
    { {"0"_b, "ld4_asisdlso_b4_4b"},
    },
  },

  { "_tlvmlq",
    {18},
    { {"0"_b, "ld2_asisdlso_h2_2h"},
    },
  },

  { "_tmhlvh",
    {20, 9, 4},
    { {"000"_b, "zip2_p_pp"},
    },
  },

  { "_tmsjzg",
    {2, 1},
    { {"00"_b, "ret_64r_branch_reg"},
    },
  },

  { "_tmtgqm",
    {4},
    { {"0"_b, "ccmn_64_condcmp_imm"},
    },
  },

  { "_tmtnkq",
    {23, 18, 17, 16},
    { {"0000"_b, "uqxtnb_z_zz"},
    },
  },

  { "_tnjhxp",
    {9, 8, 7, 6, 5},
    { {"11111"_b, "pacdza_64z_dp_1src"},
    },
  },

  { "_tnngsg",
    {23, 22, 13, 12, 11, 10},
    { {"01x1x0"_b, "fcmla_asimdelem_c_h"},
      {"0x0001"_b, "ushr_asimdshf_r"},
      {"0x0101"_b, "usra_asimdshf_r"},
      {"0x1001"_b, "urshr_asimdshf_r"},
      {"0x1101"_b, "ursra_asimdshf_r"},
      {"10x1x0"_b, "fcmla_asimdelem_c_s"},
      {"xx00x0"_b, "mla_asimdelem_r"},
      {"xx10x0"_b, "umlal_asimdelem_l"},
    },
  },

  { "_tnpjts",
    {30},
    { {"0"_b, "and_64_log_shift"},
      {"1"_b, "eor_64_log_shift"},
    },
  },

  { "_tpkslq",
    {30, 23, 22, 20, 13, 4},
    { {"00001x"_b, "ld1rqw_z_p_bi_u32"},
      {"000x0x"_b, "ld1rqw_z_p_br_contiguous"},
      {"01001x"_b, "ld1rqd_z_p_bi_u64"},
      {"010x0x"_b, "ld1rqd_z_p_br_contiguous"},
      {"100x1x"_b, "stnt1w_z_p_ar_d_64_unscaled"},
      {"101x1x"_b, "stnt1w_z_p_ar_s_x32_unscaled"},
      {"110x00"_b, "str_p_bi"},
      {"110x1x"_b, "stnt1d_z_p_ar_d_64_unscaled"},
    },
  },

  { "_tplghv",
    {18, 17, 12},
    { {"0x0"_b, "st2_asisdlsop_dx2_r2d"},
      {"100"_b, "st2_asisdlsop_dx2_r2d"},
      {"110"_b, "st2_asisdlsop_d2_i2d"},
    },
  },

  { "_tpmqyl",
    {30},
    { {"0"_b, "bl_only_branch_imm"},
      {"1"_b, "_lszlkq"},
    },
  },

  { "_tptqjs",
    {12},
    { {"0"_b, "ld1_asisdlsop_dx1_r1d"},
    },
  },

  { "_tqlrzh",
    {9, 8, 7, 6, 5, 2, 1},
    { {"1111111"_b, "eretab_64e_branch_reg"},
    },
  },

  { "_tqlsyy",
    {30},
    { {"0"_b, "add_32_addsub_shift"},
      {"1"_b, "sub_32_addsub_shift"},
    },
  },

  { "_trjmmn",
    {13, 12, 11, 10},
    { {"0001"_b, "sub_asisdsame_only"},
      {"0010"_b, "_plyhhz"},
      {"0011"_b, "cmeq_asisdsame_only"},
      {"0110"_b, "_qkhrkh"},
      {"1010"_b, "_kxhmlx"},
      {"1101"_b, "sqrdmulh_asisdsame_only"},
      {"1110"_b, "_ytrmvz"},
    },
  },

  { "_tshjsk",
    {18},
    { {"0"_b, "st4_asisdlsep_r4_r"},
      {"1"_b, "st4_asisdlsep_i4_i"},
    },
  },

  { "_tsskys",
    {23, 22, 18, 17, 16},
    { {"01000"_b, "fadd_z_p_zz"},
      {"01001"_b, "fsub_z_p_zz"},
      {"01010"_b, "fmul_z_p_zz"},
      {"01100"_b, "fmaxnm_z_p_zz"},
      {"01101"_b, "fminnm_z_p_zz"},
      {"01110"_b, "fmax_z_p_zz"},
      {"01111"_b, "fmin_z_p_zz"},
      {"1x000"_b, "fadd_z_p_zz"},
      {"1x001"_b, "fsub_z_p_zz"},
      {"1x010"_b, "fmul_z_p_zz"},
      {"1x100"_b, "fmaxnm_z_p_zz"},
      {"1x101"_b, "fminnm_z_p_zz"},
      {"1x110"_b, "fmax_z_p_zz"},
      {"1x111"_b, "fmin_z_p_zz"},
      {"xx011"_b, "fsubr_z_p_zz"},
    },
  },

  { "_tsypsz",
    {23, 22, 13, 12},
    { {"0000"_b, "fnmul_s_floatdp2"},
      {"0100"_b, "fnmul_d_floatdp2"},
      {"1100"_b, "fnmul_h_floatdp2"},
    },
  },

  { "_ttmvpr",
    {30, 23, 22, 20, 19},
    { {"0xxxx"_b, "bl_only_branch_imm"},
      {"10001"_b, "sys_cr_systeminstrs"},
      {"1001x"_b, "msr_sr_systemmove"},
      {"10101"_b, "sysp_cr_syspairinstrs"},
      {"1011x"_b, "msrr_sr_systemmovepr"},
    },
  },

  { "_ttmyrv",
    {30, 11, 10},
    { {"000"_b, "_nynrns"},
      {"001"_b, "_rrkmyl"},
      {"010"_b, "_rvvshx"},
      {"011"_b, "_zlmyjt"},
      {"101"_b, "_yrggjm"},
      {"110"_b, "_kskqmz"},
      {"111"_b, "_kzksnv"},
    },
  },

  { "_ttplgp",
    {12, 11, 10},
    { {"000"_b, "sqincp_z_p_z"},
      {"010"_b, "sqincp_r_p_r_sx"},
      {"011"_b, "sqincp_r_p_r_x"},
      {"100"_b, "_zqmrhp"},
    },
  },

  { "_ttsgkt",
    {12, 10},
    { {"00"_b, "_smsytm"},
      {"01"_b, "_mjrlkp"},
      {"10"_b, "_vjkhhm"},
      {"11"_b, "_ymxjjr"},
    },
  },

  { "_ttzlqn",
    {18, 17, 12},
    { {"000"_b, "ld1_asisdlso_d1_1d"},
    },
  },

  { "_tvgklq",
    {18},
    { {"0"_b, "st4_asisdlsop_bx4_r4b"},
      {"1"_b, "st4_asisdlsop_b4_i4b"},
    },
  },

  { "_tvrlgz",
    {18},
    { {"0"_b, "st1_asisdlsop_bx1_r1b"},
      {"1"_b, "st1_asisdlsop_b1_i1b"},
    },
  },

  { "_tvtvkt",
    {18, 17, 12},
    { {"000"_b, "ldap1_asisdlso_d1"},
    },
  },

  { "_tvyxlr",
    {30},
    { {"0"_b, "bl_only_branch_imm"},
      {"1"_b, "_jlnjsy"},
    },
  },

  { "_txkmvh",
    {18},
    { {"0"_b, "ld2_asisdlse_r2"},
    },
  },

  { "_txsvzz",
    {12},
    { {"0"_b, "ld4_asisdlsop_dx4_r4d"},
    },
  },

  { "_txzxzs",
    {23, 22, 20, 19, 18},
    { {"00000"_b, "orr_z_zi"},
      {"01000"_b, "eor_z_zi"},
      {"10000"_b, "and_z_zi"},
      {"11000"_b, "dupm_z_i"},
      {"xx1xx"_b, "cpy_z_p_i"},
    },
  },

  { "_tykvnx",
    {30},
    { {"0"_b, "ldapr_32l_ldapstl_writeback"},
      {"1"_b, "ldapr_64l_ldapstl_writeback"},
    },
  },

  { "_tymryz",
    {23, 22, 20, 19, 13, 11},
    { {"0000x0"_b, "bic_asimdimm_l_sl"},
      {"00x100"_b, "sli_asimdshf_r"},
      {"00x110"_b, "uqshl_asimdshf_r"},
      {"010x00"_b, "sli_asimdshf_r"},
      {"010x10"_b, "uqshl_asimdshf_r"},
      {"011100"_b, "sli_asimdshf_r"},
      {"011110"_b, "uqshl_asimdshf_r"},
      {"0x1000"_b, "sli_asimdshf_r"},
      {"0x1010"_b, "uqshl_asimdshf_r"},
    },
  },

  { "_tytvjk",
    {13, 12, 11},
    { {"000"_b, "_lylpyx"},
      {"001"_b, "_kyxrqg"},
      {"010"_b, "_zmkqxl"},
      {"011"_b, "_gngjxr"},
      {"100"_b, "_mlxtxs"},
      {"101"_b, "_mnmtql"},
      {"110"_b, "_xmxpnx"},
      {"111"_b, "_lkttgy"},
    },
  },

  { "_tytzpq",
    {30},
    { {"0"_b, "bic_32_log_shift"},
      {"1"_b, "eon_32_log_shift"},
    },
  },

  { "_tyzpxk",
    {22, 13, 12},
    { {"000"_b, "swpa_64_memop"},
      {"100"_b, "swpal_64_memop"},
    },
  },

  { "_tzgtvm",
    {13, 12},
    { {"00"_b, "crc32x_64c_dp_2src"},
      {"01"_b, "crc32cx_64c_dp_2src"},
      {"10"_b, "umin_64_dp_2src"},
    },
  },

  { "_tzjyhy",
    {20, 19, 18, 17, 16},
    { {"00010"_b, "scvtf_d32_float2fix"},
      {"00011"_b, "ucvtf_d32_float2fix"},
      {"11000"_b, "fcvtzs_32d_float2fix"},
      {"11001"_b, "fcvtzu_32d_float2fix"},
    },
  },

  { "_tzrgqq",
    {23, 10},
    { {"00"_b, "_gyrkkz"},
    },
  },

  { "_tzsnmy",
    {9, 8, 7, 6, 5, 2, 1},
    { {"1111111"_b, "retab_64e_branch_reg"},
    },
  },

  { "_tzsvyv",
    {18},
    { {"0"_b, "ld2_asisdlsop_bx2_r2b"},
      {"1"_b, "ld2_asisdlsop_b2_i2b"},
    },
  },

  { "_tzzssm",
    {12, 11, 10},
    { {"000"_b, "histseg_z_zz"},
    },
  },

  { "_vghjnt",
    {23, 22},
    { {"00"_b, "fmadd_s_floatdp3"},
      {"01"_b, "fmadd_d_floatdp3"},
      {"11"_b, "fmadd_h_floatdp3"},
    },
  },

  { "_vgqvys",
    {30, 23, 22},
    { {"000"_b, "stp_32_ldstpair_off"},
      {"001"_b, "ldp_32_ldstpair_off"},
      {"010"_b, "stp_32_ldstpair_pre"},
      {"011"_b, "ldp_32_ldstpair_pre"},
      {"100"_b, "stgp_64_ldstpair_off"},
      {"101"_b, "ldpsw_64_ldstpair_off"},
      {"110"_b, "stgp_64_ldstpair_pre"},
      {"111"_b, "ldpsw_64_ldstpair_pre"},
    },
  },

  { "_vgtnjh",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0001010"_b, "fcvtxnt_z_p_z_d2s"},
      {"1001000"_b, "fcvtnt_z_p_z_s2h"},
      {"1001001"_b, "fcvtlt_z_p_z_h2s"},
      {"1001010"_b, "bfcvtnt_z_p_z_s2bf"},
      {"1101010"_b, "fcvtnt_z_p_z_d2s"},
      {"1101011"_b, "fcvtlt_z_p_z_s2d"},
    },
  },

  { "_vgxtvy",
    {23, 22, 20, 19, 18, 17, 16, 13, 12, 11},
    { {"0011111001"_b, "_tjxhsy"},
    },
  },

  { "_vhkjgh",
    {30, 23, 22, 20, 19, 18},
    { {"00xxxx"_b, "add_64_addsub_imm"},
      {"011000"_b, "smax_64_minmax_imm"},
      {"011001"_b, "umax_64u_minmax_imm"},
      {"011010"_b, "smin_64_minmax_imm"},
      {"011011"_b, "umin_64u_minmax_imm"},
      {"10xxxx"_b, "sub_64_addsub_imm"},
    },
  },

  { "_vhkpvn",
    {20, 18, 17, 16},
    { {"0000"_b, "_grktgm"},
    },
  },

  { "_vhlqpr",
    {30, 22, 11, 10},
    { {"0000"_b, "csel_64_condsel"},
      {"0001"_b, "csinc_64_condsel"},
      {"0100"_b, "_xgqhjv"},
      {"0101"_b, "_hspyhv"},
      {"0110"_b, "_qkxmvp"},
      {"0111"_b, "_tzgtvm"},
      {"1000"_b, "csinv_64_condsel"},
      {"1001"_b, "csneg_64_condsel"},
      {"1100"_b, "_hlqvmm"},
      {"1101"_b, "_ghrnmz"},
    },
  },

  { "_vhrkvk",
    {18, 17},
    { {"00"_b, "ld4_asisdlso_s4_4s"},
    },
  },

  { "_vjhrzl",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "frintx_asimdmiscfp16_r"},
      {"0x00001"_b, "frintx_asimdmisc_r"},
      {"1111001"_b, "frinti_asimdmiscfp16_r"},
      {"1x00001"_b, "frinti_asimdmisc_r"},
      {"xx00000"_b, "cmle_asimdmisc_z"},
    },
  },

  { "_vjkhhm",
    {23, 22, 13},
    { {"000"_b, "fmul_asimdelem_rh_h"},
      {"1x0"_b, "fmul_asimdelem_r_sd"},
      {"xx1"_b, "sqdmull_asimdelem_l"},
    },
  },

  { "_vjmklj",
    {23, 22},
    { {"10"_b, "sqrdcmlah_z_zzzi_h"},
      {"11"_b, "sqrdcmlah_z_zzzi_s"},
    },
  },

  { "_vjtgmx",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldlar_lr64_ldstexcl"},
    },
  },

  { "_vkrkks",
    {30, 23, 22, 13, 4},
    { {"00000"_b, "prfb_i_p_br_s"},
      {"00010"_b, "prfb_i_p_ai_s"},
      {"0010x"_b, "ld1rb_z_p_bi_u32"},
      {"0011x"_b, "ld1rb_z_p_bi_u64"},
      {"01000"_b, "prfh_i_p_br_s"},
      {"01010"_b, "prfh_i_p_ai_s"},
      {"0110x"_b, "ld1rh_z_p_bi_u32"},
      {"0111x"_b, "ld1rh_z_p_bi_u64"},
      {"1000x"_b, "ldnt1b_z_p_ar_d_64_unscaled"},
      {"10010"_b, "prfb_i_p_ai_d"},
      {"1010x"_b, "ld1b_z_p_bz_d_64_unscaled"},
      {"1011x"_b, "ldff1b_z_p_bz_d_64_unscaled"},
      {"1100x"_b, "ldnt1h_z_p_ar_d_64_unscaled"},
      {"11010"_b, "prfh_i_p_ai_d"},
      {"1110x"_b, "ld1h_z_p_bz_d_64_unscaled"},
      {"1111x"_b, "ldff1h_z_p_bz_d_64_unscaled"},
    },
  },

  { "_vkrskv",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx00"_b, "stlur_32_ldapstl_unscaled"},
      {"001xx00"_b, "ldapur_32_ldapstl_unscaled"},
      {"010xx00"_b, "ldapursw_64_ldapstl_unscaled"},
      {"100xx00"_b, "stlur_64_ldapstl_unscaled"},
      {"101xx00"_b, "ldapur_64_ldapstl_unscaled"},
      {"x000001"_b, "cpyfprn_cpy_memcms"},
      {"x000101"_b, "cpyfpwtrn_cpy_memcms"},
      {"x001001"_b, "cpyfprtrn_cpy_memcms"},
      {"x001101"_b, "cpyfptrn_cpy_memcms"},
      {"x010001"_b, "cpyfmrn_cpy_memcms"},
      {"x010101"_b, "cpyfmwtrn_cpy_memcms"},
      {"x011001"_b, "cpyfmrtrn_cpy_memcms"},
      {"x011101"_b, "cpyfmtrn_cpy_memcms"},
      {"x100001"_b, "cpyfern_cpy_memcms"},
      {"x100101"_b, "cpyfewtrn_cpy_memcms"},
      {"x101001"_b, "cpyfertrn_cpy_memcms"},
      {"x101101"_b, "cpyfetrn_cpy_memcms"},
      {"x110001"_b, "sete_set_memcms"},
      {"x110101"_b, "setet_set_memcms"},
      {"x111001"_b, "seten_set_memcms"},
      {"x111101"_b, "setetn_set_memcms"},
    },
  },

  { "_vlhkgr",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "uaddlp_asimdmisc_p"},
      {"00001"_b, "sqxtun_asimdmisc_n"},
    },
  },

  { "_vllmnt",
    {20, 19, 18, 17},
    { {"0000"_b, "_gmtjvr"},
    },
  },

  { "_vlrhpy",
    {30, 23, 22, 13, 4},
    { {"0000x"_b, "ld1sb_z_p_ai_s"},
      {"0001x"_b, "ldff1sb_z_p_ai_s"},
      {"0010x"_b, "ld1rb_z_p_bi_u8"},
      {"0011x"_b, "ld1rb_z_p_bi_u16"},
      {"0100x"_b, "ld1sh_z_p_ai_s"},
      {"0101x"_b, "ldff1sh_z_p_ai_s"},
      {"0110x"_b, "ld1rsw_z_p_bi_s64"},
      {"0111x"_b, "ld1rh_z_p_bi_u16"},
      {"1000x"_b, "ld1sb_z_p_ai_d"},
      {"1001x"_b, "ldff1sb_z_p_ai_d"},
      {"10100"_b, "prfb_i_p_bz_d_64_scaled"},
      {"10110"_b, "prfh_i_p_bz_d_64_scaled"},
      {"1100x"_b, "ld1sh_z_p_ai_d"},
      {"1101x"_b, "ldff1sh_z_p_ai_d"},
      {"1110x"_b, "ld1sh_z_p_bz_d_64_scaled"},
      {"1111x"_b, "ldff1sh_z_p_bz_d_64_scaled"},
    },
  },

  { "_vlxrps",
    {9, 8, 7, 6, 5},
    { {"00000"_b, "fmov_d_floatimm"},
    },
  },

  { "_vmgnhk",
    {30, 23},
    { {"00"_b, "add_64_addsub_imm"},
      {"10"_b, "sub_64_addsub_imm"},
    },
  },

  { "_vmsxgq",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx00"_b, "stlur_32_ldapstl_unscaled"},
      {"001xx00"_b, "ldapur_32_ldapstl_unscaled"},
      {"010xx00"_b, "ldapursw_64_ldapstl_unscaled"},
      {"100xx00"_b, "stlur_64_ldapstl_unscaled"},
      {"101xx00"_b, "ldapur_64_ldapstl_unscaled"},
      {"x000001"_b, "cpyfpwn_cpy_memcms"},
      {"x000101"_b, "cpyfpwtwn_cpy_memcms"},
      {"x001001"_b, "cpyfprtwn_cpy_memcms"},
      {"x001101"_b, "cpyfptwn_cpy_memcms"},
      {"x010001"_b, "cpyfmwn_cpy_memcms"},
      {"x010101"_b, "cpyfmwtwn_cpy_memcms"},
      {"x011001"_b, "cpyfmrtwn_cpy_memcms"},
      {"x011101"_b, "cpyfmtwn_cpy_memcms"},
      {"x100001"_b, "cpyfewn_cpy_memcms"},
      {"x100101"_b, "cpyfewtwn_cpy_memcms"},
      {"x101001"_b, "cpyfertwn_cpy_memcms"},
      {"x101101"_b, "cpyfetwn_cpy_memcms"},
      {"x110001"_b, "setm_set_memcms"},
      {"x110101"_b, "setmt_set_memcms"},
      {"x111001"_b, "setmn_set_memcms"},
      {"x111101"_b, "setmtn_set_memcms"},
    },
  },

  { "_vmtkqp",
    {30},
    { {"0"_b, "stlur_32_ldapstl_unscaled"},
      {"1"_b, "stlur_64_ldapstl_unscaled"},
    },
  },

  { "_vmxzxt",
    {23, 22, 13, 12, 11, 10},
    { {"0001x0"_b, "fmulx_asimdelem_rh_h"},
      {"0x0001"_b, "sqshrun_asimdshf_n"},
      {"0x0011"_b, "sqrshrun_asimdshf_n"},
      {"0x0101"_b, "uqshrn_asimdshf_n"},
      {"0x0111"_b, "uqrshrn_asimdshf_n"},
      {"0x1001"_b, "ushll_asimdshf_l"},
      {"1000x0"_b, "fmlal2_asimdelem_lh"},
      {"1x01x0"_b, "fmulx_asimdelem_r_sd"},
      {"xx10x0"_b, "umull_asimdelem_l"},
    },
  },

  { "_vmyztj",
    {30, 23, 22},
    { {"000"_b, "stp_64_ldstpair_off"},
      {"001"_b, "ldp_64_ldstpair_off"},
      {"010"_b, "stp_64_ldstpair_pre"},
      {"011"_b, "ldp_64_ldstpair_pre"},
    },
  },

  { "_vnggzq",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx10"_b, "stlur_b_ldapstl_simd"},
      {"001xx10"_b, "ldapur_b_ldapstl_simd"},
      {"010xx10"_b, "stlur_q_ldapstl_simd"},
      {"011xx10"_b, "ldapur_q_ldapstl_simd"},
      {"100xx10"_b, "stlur_h_ldapstl_simd"},
      {"101xx10"_b, "ldapur_h_ldapstl_simd"},
      {"x000001"_b, "cpypn_cpy_memcms"},
      {"x000101"_b, "cpypwtn_cpy_memcms"},
      {"x001001"_b, "cpyprtn_cpy_memcms"},
      {"x001101"_b, "cpyptn_cpy_memcms"},
      {"x010001"_b, "cpymn_cpy_memcms"},
      {"x010101"_b, "cpymwtn_cpy_memcms"},
      {"x011001"_b, "cpymrtn_cpy_memcms"},
      {"x011101"_b, "cpymtn_cpy_memcms"},
      {"x100001"_b, "cpyen_cpy_memcms"},
      {"x100101"_b, "cpyewtn_cpy_memcms"},
      {"x101001"_b, "cpyertn_cpy_memcms"},
      {"x101101"_b, "cpyetn_cpy_memcms"},
    },
  },

  { "_vnnjxg",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xxxx"_b, "fnmsub_s_floatdp3"},
      {"001xxxx"_b, "fnmsub_d_floatdp3"},
      {"011xxxx"_b, "fnmsub_h_floatdp3"},
      {"10x1001"_b, "scvtf_asisdshf_c"},
      {"10x1111"_b, "fcvtzs_asisdshf_c"},
      {"1xx00x0"_b, "sqdmulh_asisdelem_r"},
      {"1xx01x0"_b, "sqrdmulh_asisdelem_r"},
    },
  },

  { "_vnrlrk",
    {30},
    { {"0"_b, "orn_64_log_shift"},
      {"1"_b, "bics_64_log_shift"},
    },
  },

  { "_vnrlsj",
    {18, 17, 12},
    { {"0x0"_b, "ld4_asisdlsop_dx4_r4d"},
      {"100"_b, "ld4_asisdlsop_dx4_r4d"},
      {"110"_b, "ld4_asisdlsop_d4_i4d"},
    },
  },

  { "_vnsqhn",
    {30, 23, 11, 10},
    { {"0010"_b, "_plytvr"},
      {"0100"_b, "_zghtll"},
      {"0110"_b, "_ptkgrz"},
      {"1000"_b, "_xksqnh"},
      {"1001"_b, "ldraa_64_ldst_pac"},
      {"1010"_b, "_hyskth"},
      {"1011"_b, "ldraa_64w_ldst_pac"},
      {"1100"_b, "_kpgghm"},
      {"1101"_b, "ldrab_64_ldst_pac"},
      {"1110"_b, "_zxjkmj"},
      {"1111"_b, "ldrab_64w_ldst_pac"},
    },
  },

  { "_vnzkty",
    {30},
    { {"0"_b, "orr_64_log_shift"},
      {"1"_b, "ands_64_log_shift"},
    },
  },

  { "_vpgxgk",
    {20, 19, 18, 17, 16, 13, 12},
    { {"1111100"_b, "_rqzpzq"},
    },
  },

  { "_vpjktn",
    {30, 23, 22},
    { {"000"_b, "madd_64a_dp_3src"},
    },
  },

  { "_vpknjg",
    {13, 12},
    { {"00"_b, "sdiv_32_dp_2src"},
      {"10"_b, "rorv_32_dp_2src"},
    },
  },

  { "_vpmxrj",
    {13},
    { {"0"_b, "histcnt_z_p_zz"},
      {"1"_b, "_jxszhy"},
    },
  },

  { "_vpyvjr",
    {9, 8, 7, 6, 5},
    { {"11111"_b, "pacizb_64z_dp_1src"},
    },
  },

  { "_vqrqjt",
    {30, 23, 22, 11, 10},
    { {"01000"_b, "csel_32_condsel"},
      {"01001"_b, "csinc_32_condsel"},
      {"11000"_b, "csinv_32_condsel"},
      {"11001"_b, "csneg_32_condsel"},
    },
  },

  { "_vqzsgg",
    {20, 19, 18, 17, 16},
    { {"00010"_b, "scvtf_s32_float2fix"},
      {"00011"_b, "ucvtf_s32_float2fix"},
      {"11000"_b, "fcvtzs_32s_float2fix"},
      {"11001"_b, "fcvtzu_32s_float2fix"},
    },
  },

  { "_vrjhtm",
    {12},
    { {"0"_b, "sqdmulh_asimdelem_r"},
      {"1"_b, "sqrdmulh_asimdelem_r"},
    },
  },

  { "_vrsgzg",
    {30, 23, 22, 20, 19, 18},
    { {"00xxxx"_b, "add_64_addsub_imm"},
      {"010xxx"_b, "addg_64_addsub_immtags"},
      {"011000"_b, "smax_64_minmax_imm"},
      {"011001"_b, "umax_64u_minmax_imm"},
      {"011010"_b, "smin_64_minmax_imm"},
      {"011011"_b, "umin_64u_minmax_imm"},
      {"10xxxx"_b, "sub_64_addsub_imm"},
      {"110xxx"_b, "subg_64_addsub_immtags"},
    },
  },

  { "_vrsjnp",
    {13, 12, 11, 10},
    { {"1111"_b, "cas_c32_ldstexcl"},
    },
  },

  { "_vrxhss",
    {20, 19, 18, 17, 16},
    { {"00001"_b, "uqxtn_asisdmisc_n"},
    },
  },

  { "_vryrnh",
    {30, 22, 11},
    { {"001"_b, "_zsgpsn"},
      {"010"_b, "ccmn_32_condcmp_reg"},
      {"011"_b, "ccmn_32_condcmp_imm"},
      {"110"_b, "ccmp_32_condcmp_reg"},
      {"111"_b, "ccmp_32_condcmp_imm"},
    },
  },

  { "_vrzksz",
    {20, 19, 18, 17, 16, 13, 12},
    { {"1111100"_b, "ldaprh_32l_memop"},
    },
  },

  { "_vshynq",
    {30, 23, 22, 11, 10},
    { {"00000"_b, "sturb_32_ldst_unscaled"},
      {"00001"_b, "strb_32_ldst_immpost"},
      {"00010"_b, "sttrb_32_ldst_unpriv"},
      {"00011"_b, "strb_32_ldst_immpre"},
      {"00100"_b, "ldurb_32_ldst_unscaled"},
      {"00101"_b, "ldrb_32_ldst_immpost"},
      {"00110"_b, "ldtrb_32_ldst_unpriv"},
      {"00111"_b, "ldrb_32_ldst_immpre"},
      {"01000"_b, "ldursb_64_ldst_unscaled"},
      {"01001"_b, "ldrsb_64_ldst_immpost"},
      {"01010"_b, "ldtrsb_64_ldst_unpriv"},
      {"01011"_b, "ldrsb_64_ldst_immpre"},
      {"01100"_b, "ldursb_32_ldst_unscaled"},
      {"01101"_b, "ldrsb_32_ldst_immpost"},
      {"01110"_b, "ldtrsb_32_ldst_unpriv"},
      {"01111"_b, "ldrsb_32_ldst_immpre"},
      {"10000"_b, "sturh_32_ldst_unscaled"},
      {"10001"_b, "strh_32_ldst_immpost"},
      {"10010"_b, "sttrh_32_ldst_unpriv"},
      {"10011"_b, "strh_32_ldst_immpre"},
      {"10100"_b, "ldurh_32_ldst_unscaled"},
      {"10101"_b, "ldrh_32_ldst_immpost"},
      {"10110"_b, "ldtrh_32_ldst_unpriv"},
      {"10111"_b, "ldrh_32_ldst_immpre"},
      {"11000"_b, "ldursh_64_ldst_unscaled"},
      {"11001"_b, "ldrsh_64_ldst_immpost"},
      {"11010"_b, "ldtrsh_64_ldst_unpriv"},
      {"11011"_b, "ldrsh_64_ldst_immpre"},
      {"11100"_b, "ldursh_32_ldst_unscaled"},
      {"11101"_b, "ldrsh_32_ldst_immpost"},
      {"11110"_b, "ldtrsh_32_ldst_unpriv"},
      {"11111"_b, "ldrsh_32_ldst_immpre"},
    },
  },

  { "_vsnnms",
    {30, 13, 12, 11, 10},
    { {"00000"_b, "_xzntxr"},
    },
  },

  { "_vsslrs",
    {8},
    { {"0"_b, "tstart_br_systemresult"},
      {"1"_b, "ttest_br_systemresult"},
    },
  },

  { "_vsyjql",
    {4},
    { {"0"_b, "ccmn_32_condcmp_imm"},
    },
  },

  { "_vtgnnl",
    {30},
    { {"0"_b, "_qgsrqq"},
      {"1"_b, "_mgjhts"},
    },
  },

  { "_vtllgt",
    {10},
    { {"0"_b, "_nhnhzp"},
    },
  },

  { "_vtyqhh",
    {30},
    { {"0"_b, "and_32_log_shift"},
      {"1"_b, "eor_32_log_shift"},
    },
  },

  { "_vvgnhm",
    {23},
    { {"0"_b, "fmulx_asimdsame_only"},
    },
  },

  { "_vvgpzq",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "fcvtns_32h_float2int"},
      {"00001"_b, "fcvtnu_32h_float2int"},
      {"00010"_b, "scvtf_h32_float2int"},
      {"00011"_b, "ucvtf_h32_float2int"},
      {"00100"_b, "fcvtas_32h_float2int"},
      {"00101"_b, "fcvtau_32h_float2int"},
      {"00110"_b, "fmov_32h_float2int"},
      {"00111"_b, "fmov_h32_float2int"},
      {"01000"_b, "fcvtps_32h_float2int"},
      {"01001"_b, "fcvtpu_32h_float2int"},
      {"10000"_b, "fcvtms_32h_float2int"},
      {"10001"_b, "fcvtmu_32h_float2int"},
      {"11000"_b, "fcvtzs_32h_float2int"},
      {"11001"_b, "fcvtzu_32h_float2int"},
    },
  },

  { "_vvtnrv",
    {23, 22, 20, 19, 18},
    { {"00000"_b, "orr_z_zi"},
      {"01000"_b, "eor_z_zi"},
      {"10000"_b, "and_z_zi"},
      {"11000"_b, "dupm_z_i"},
    },
  },

  { "_vvxsxt",
    {4},
    { {"0"_b, "ands_p_p_pp_z"},
      {"1"_b, "bics_p_p_pp_z"},
    },
  },

  { "_vvyjmh",
    {23, 22, 20, 19, 11},
    { {"00010"_b, "ssra_asisdshf_r"},
      {"001x0"_b, "ssra_asisdshf_r"},
      {"01xx0"_b, "ssra_asisdshf_r"},
    },
  },

  { "_vvzsmg",
    {23, 22, 20, 19, 13, 11},
    { {"0000x0"_b, "bic_asimdimm_l_sl"},
      {"00x100"_b, "usra_asimdshf_r"},
      {"00x110"_b, "ursra_asimdshf_r"},
      {"010x00"_b, "usra_asimdshf_r"},
      {"010x10"_b, "ursra_asimdshf_r"},
      {"011100"_b, "usra_asimdshf_r"},
      {"011110"_b, "ursra_asimdshf_r"},
      {"0x1000"_b, "usra_asimdshf_r"},
      {"0x1010"_b, "ursra_asimdshf_r"},
    },
  },

  { "_vxhgzz",
    {23, 22, 12, 11, 10},
    { {"00xxx"_b, "ext_z_zi_des"},
      {"01xxx"_b, "ext_z_zi_con"},
      {"10000"_b, "zip1_z_zz_q"},
      {"10001"_b, "zip2_z_zz_q"},
      {"10010"_b, "uzp1_z_zz_q"},
      {"10011"_b, "uzp2_z_zz_q"},
      {"10110"_b, "trn1_z_zz_q"},
      {"10111"_b, "trn2_z_zz_q"},
    },
  },

  { "_vxhjgg",
    {20, 18, 17, 16},
    { {"0000"_b, "_shgxyq"},
    },
  },

  { "_vxlmxz",
    {4, 3, 2, 1, 0},
    { {"11111"_b, "_hpmvzr"},
    },
  },

  { "_vxqtkl",
    {18, 17},
    { {"00"_b, "_zqmvqs"},
    },
  },

  { "_vxrnyh",
    {18, 17},
    { {"0x"_b, "st1_asisdlsep_r1_r1"},
      {"10"_b, "st1_asisdlsep_r1_r1"},
      {"11"_b, "st1_asisdlsep_i1_i1"},
    },
  },

  { "_vxvyyg",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xx00"_b, "stlurb_32_ldapstl_unscaled"},
      {"001xx00"_b, "ldapurb_32_ldapstl_unscaled"},
      {"010xx00"_b, "ldapursb_64_ldapstl_unscaled"},
      {"011xx00"_b, "ldapursb_32_ldapstl_unscaled"},
      {"100xx00"_b, "stlurh_32_ldapstl_unscaled"},
      {"101xx00"_b, "ldapurh_32_ldapstl_unscaled"},
      {"110xx00"_b, "ldapursh_64_ldapstl_unscaled"},
      {"111xx00"_b, "ldapursh_32_ldapstl_unscaled"},
      {"x000001"_b, "cpyfpn_cpy_memcms"},
      {"x000101"_b, "cpyfpwtn_cpy_memcms"},
      {"x001001"_b, "cpyfprtn_cpy_memcms"},
      {"x001101"_b, "cpyfptn_cpy_memcms"},
      {"x010001"_b, "cpyfmn_cpy_memcms"},
      {"x010101"_b, "cpyfmwtn_cpy_memcms"},
      {"x011001"_b, "cpyfmrtn_cpy_memcms"},
      {"x011101"_b, "cpyfmtn_cpy_memcms"},
      {"x100001"_b, "cpyfen_cpy_memcms"},
      {"x100101"_b, "cpyfewtn_cpy_memcms"},
      {"x101001"_b, "cpyfertn_cpy_memcms"},
      {"x101101"_b, "cpyfetn_cpy_memcms"},
    },
  },

  { "_vyjsst",
    {30, 4},
    { {"0x"_b, "b_only_branch_imm"},
      {"10"_b, "b_only_condbranch"},
      {"11"_b, "bc_only_condbranch"},
    },
  },

  { "_vypgrt",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "rev16_asimdmisc_r"},
    },
  },

  { "_vypnss",
    {30},
    { {"0"_b, "orn_32_log_shift"},
      {"1"_b, "bics_32_log_shift"},
    },
  },

  { "_vyqxyz",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "fcvtau_asimdmiscfp16_r"},
      {"0x00001"_b, "fcvtau_asimdmisc_r"},
      {"0x10000"_b, "fmaxnmv_asimdall_only_sd"},
      {"1111000"_b, "fcmge_asimdmiscfp16_fz"},
      {"1x00000"_b, "fcmge_asimdmisc_fz"},
      {"1x00001"_b, "ursqrte_asimdmisc_r"},
      {"1x10000"_b, "fminnmv_asimdall_only_sd"},
    },
  },

  { "_vzjvtv",
    {23, 22, 12, 11, 10},
    { {"01001"_b, "bfmmla_z_zzz"},
      {"10001"_b, "fmmla_z_zzz_s"},
      {"11001"_b, "fmmla_z_zzz_d"},
    },
  },

  { "_vzvstm",
    {23, 22, 20, 19, 12, 11},
    { {"000000"_b, "movi_asimdimm_n_b"},
      {"000010"_b, "fmov_asimdimm_s_s"},
      {"000011"_b, "fmov_asimdimm_h_h"},
      {"00x100"_b, "scvtf_asimdshf_c"},
      {"00x111"_b, "fcvtzs_asimdshf_c"},
      {"010x00"_b, "scvtf_asimdshf_c"},
      {"010x11"_b, "fcvtzs_asimdshf_c"},
      {"011100"_b, "scvtf_asimdshf_c"},
      {"011111"_b, "fcvtzs_asimdshf_c"},
      {"0x1000"_b, "scvtf_asimdshf_c"},
      {"0x1011"_b, "fcvtzs_asimdshf_c"},
    },
  },

  { "_vzyklr",
    {13, 12},
    { {"00"_b, "setp_set_memcms"},
      {"01"_b, "setpt_set_memcms"},
      {"10"_b, "setpn_set_memcms"},
      {"11"_b, "setptn_set_memcms"},
    },
  },

  { "_vzzqhx",
    {12, 10},
    { {"00"_b, "_phrqqx"},
      {"01"_b, "_snnlgr"},
      {"10"_b, "_phsrlk"},
      {"11"_b, "_nrmlqv"},
    },
  },

  { "_xghrjn",
    {20, 19, 18, 17, 16},
    { {"00010"_b, "scvtf_h32_float2fix"},
      {"00011"_b, "ucvtf_h32_float2fix"},
      {"11000"_b, "fcvtzs_32h_float2fix"},
      {"11001"_b, "fcvtzu_32h_float2fix"},
    },
  },

  { "_xgqhjv",
    {13, 12},
    { {"10"_b, "smax_64_dp_2src"},
    },
  },

  { "_xgxtlr",
    {23},
    { {"0"_b, "fdiv_asimdsame_only"},
    },
  },

  { "_xhhqnx",
    {30, 23, 22, 13, 12, 11, 10},
    { {"1101001"_b, "ummla_asimdsame2_g"},
      {"xxx0001"_b, "sqrdmlah_asimdsame2_only"},
      {"xxx0011"_b, "sqrdmlsh_asimdsame2_only"},
      {"xxx0101"_b, "udot_asimdsame2_d"},
    },
  },

  { "_xhktsk",
    {22},
    { {"0"_b, "smullt_z_zzi_s"},
      {"1"_b, "smullt_z_zzi_d"},
    },
  },

  { "_xhlhmh",
    {4},
    { {"0"_b, "cmplo_p_p_zi"},
      {"1"_b, "cmpls_p_p_zi"},
    },
  },

  { "_xhmpmy",
    {4},
    { {"0"_b, "and_p_p_pp_z"},
      {"1"_b, "bic_p_p_pp_z"},
    },
  },

  { "_xjtzgm",
    {30, 23, 22, 11, 10},
    { {"00000"_b, "stur_b_ldst_unscaled"},
      {"00001"_b, "str_b_ldst_immpost"},
      {"00011"_b, "str_b_ldst_immpre"},
      {"00100"_b, "ldur_b_ldst_unscaled"},
      {"00101"_b, "ldr_b_ldst_immpost"},
      {"00111"_b, "ldr_b_ldst_immpre"},
      {"01000"_b, "stur_q_ldst_unscaled"},
      {"01001"_b, "str_q_ldst_immpost"},
      {"01011"_b, "str_q_ldst_immpre"},
      {"01100"_b, "ldur_q_ldst_unscaled"},
      {"01101"_b, "ldr_q_ldst_immpost"},
      {"01111"_b, "ldr_q_ldst_immpre"},
      {"10000"_b, "stur_h_ldst_unscaled"},
      {"10001"_b, "str_h_ldst_immpost"},
      {"10011"_b, "str_h_ldst_immpre"},
      {"10100"_b, "ldur_h_ldst_unscaled"},
      {"10101"_b, "ldr_h_ldst_immpost"},
      {"10111"_b, "ldr_h_ldst_immpre"},
    },
  },

  { "_xksqnh",
    {22, 20, 19, 18, 17, 16, 13, 12},
    { {"01111101"_b, "ld64b_64l_memop"},
    },
  },

  { "_xkylhh",
    {22, 13, 12},
    { {"000"_b, "swpa_32_memop"},
      {"100"_b, "swpal_32_memop"},
    },
  },

  { "_xkznrh",
    {18, 17},
    { {"00"_b, "st3_asisdlse_r3"},
    },
  },

  { "_xlgxhn",
    {23, 22, 4},
    { {"000"_b, "fccmp_s_floatccmp"},
      {"001"_b, "fccmpe_s_floatccmp"},
      {"010"_b, "fccmp_d_floatccmp"},
      {"011"_b, "fccmpe_d_floatccmp"},
      {"110"_b, "fccmp_h_floatccmp"},
      {"111"_b, "fccmpe_h_floatccmp"},
    },
  },

  { "_xlqmhl",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldar_lr64_ldstexcl"},
    },
  },

  { "_xlyjsz",
    {23, 22, 13},
    { {"100"_b, "fmlal2_asimdelem_lh"},
      {"xx1"_b, "umull_asimdelem_l"},
    },
  },

  { "_xlyppq",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0010000"_b, "fmaxv_asimdall_only_h"},
      {"0x00001"_b, "frint64z_asimdmisc_r"},
      {"1010000"_b, "fminv_asimdall_only_h"},
      {"1111000"_b, "fabs_asimdmiscfp16_r"},
      {"1x00000"_b, "fabs_asimdmisc_r"},
    },
  },

  { "_xmkysx",
    {12},
    { {"0"_b, "st4_asisdlsop_dx4_r4d"},
    },
  },

  { "_xmxhhg",
    {13, 12, 4},
    { {"000"_b, "rmif_only_rmif"},
    },
  },

  { "_xmxpnx",
    {10},
    { {"0"_b, "sri_z_zzi"},
      {"1"_b, "sli_z_zzi"},
    },
  },

  { "_xnhkpk",
    {23, 22},
    { {"00"_b, "fcsel_s_floatsel"},
      {"01"_b, "fcsel_d_floatsel"},
      {"11"_b, "fcsel_h_floatsel"},
    },
  },

  { "_xnpyvy",
    {13, 10},
    { {"00"_b, "_sylkvm"},
      {"01"_b, "_nvnjyp"},
      {"10"_b, "_ltrntg"},
      {"11"_b, "_qrtjvn"},
    },
  },

  { "_xnrrsy",
    {18},
    { {"0"_b, "st1_asisdlsep_r4_r4"},
      {"1"_b, "st1_asisdlsep_i4_i4"},
    },
  },

  { "_xnrxym",
    {18},
    { {"0"_b, "ld2_asisdlsep_r2_r"},
      {"1"_b, "ld2_asisdlsep_i2_i"},
    },
  },

  { "_xpqglq",
    {4},
    { {"0"_b, "cmpeq_p_p_zi"},
      {"1"_b, "cmpne_p_p_zi"},
    },
  },

  { "_xprqgs",
    {23, 20, 19, 18, 17, 16},
    { {"000001"_b, "fcvtxn_asisdmisc_n"},
    },
  },

  { "_xptsns",
    {23, 22},
    { {"00"_b, "tbx_asimdtbl_l1_1"},
    },
  },

  { "_xqhxql",
    {12},
    { {"0"_b, "st2_asisdlsop_dx2_r2d"},
    },
  },

  { "_xqrgjj",
    {4},
    { {"0"_b, "ccmp_64_condcmp_imm"},
    },
  },

  { "_xqvzvl",
    {18, 17},
    { {"0x"_b, "st1_asisdlsep_r3_r3"},
      {"10"_b, "st1_asisdlsep_r3_r3"},
      {"11"_b, "st1_asisdlsep_i3_i3"},
    },
  },

  { "_xrkzpn",
    {12},
    { {"0"_b, "_zjqssg"},
    },
  },

  { "_xrnqyn",
    {30},
    { {"0"_b, "stlr_32s_ldapstl_writeback"},
      {"1"_b, "stlr_64s_ldapstl_writeback"},
    },
  },

  { "_xrskrk",
    {22, 12},
    { {"10"_b, "_kyhhqt"},
    },
  },

  { "_xrzqtn",
    {30},
    { {"0"_b, "bl_only_branch_imm"},
      {"1"_b, "_gyllxt"},
    },
  },

  { "_xsgnlv",
    {30, 23, 13, 12, 11, 10},
    { {"100001"_b, "ushr_asisdshf_r"},
      {"100101"_b, "usra_asisdshf_r"},
      {"101001"_b, "urshr_asisdshf_r"},
      {"101101"_b, "ursra_asisdshf_r"},
    },
  },

  { "_xspjzn",
    {13, 12, 11, 10},
    { {"1111"_b, "casl_c64_ldstexcl"},
    },
  },

  { "_xsvpzx",
    {18, 17, 12},
    { {"000"_b, "ld4_asisdlso_d4_4d"},
    },
  },

  { "_xszmjn",
    {30, 13, 12},
    { {"000"_b, "ldiapp_32le_ldiappstilp"},
      {"001"_b, "ldiapp_32l_ldiappstilp"},
      {"100"_b, "ldiapp_64ls_ldiappstilp"},
      {"101"_b, "ldiapp_64l_ldiappstilp"},
    },
  },

  { "_xszqrg",
    {30, 23, 22},
    { {"000"_b, "_glpxty"},
      {"001"_b, "_rkpylh"},
      {"011"_b, "_xghrjn"},
      {"100"_b, "_nklqly"},
    },
  },

  { "_xtgmvr",
    {23, 11, 10, 4, 3, 2, 0},
    { {"0000000"_b, "_mzkxzm"},
      {"0101111"_b, "_qgvrqy"},
      {"0111111"_b, "_lljxgp"},
      {"1000000"_b, "_tjlthk"},
    },
  },

  { "_xtgtyz",
    {19, 18, 17, 16},
    { {"0000"_b, "brkb_p_p_p"},
    },
  },

  { "_xtxyxj",
    {4},
    { {"0"_b, "orr_p_p_pp_z"},
      {"1"_b, "orn_p_p_pp_z"},
    },
  },

  { "_xtzykp",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldlarh_lr32_ldstexcl"},
    },
  },

  { "_xvmxrg",
    {13},
    { {"0"_b, "mla_asimdelem_r"},
      {"1"_b, "umlal_asimdelem_l"},
    },
  },

  { "_xvnyxq",
    {30, 23, 13, 4},
    { {"0000"_b, "prfb_i_p_bz_s_x32_scaled"},
      {"0010"_b, "prfh_i_p_bz_s_x32_scaled"},
      {"010x"_b, "ld1sh_z_p_bz_s_x32_scaled"},
      {"011x"_b, "ldff1sh_z_p_bz_s_x32_scaled"},
      {"1000"_b, "prfb_i_p_bz_d_x32_scaled"},
      {"1010"_b, "prfh_i_p_bz_d_x32_scaled"},
      {"110x"_b, "ld1sh_z_p_bz_d_x32_scaled"},
      {"111x"_b, "ldff1sh_z_p_bz_d_x32_scaled"},
    },
  },

  { "_xvppmm",
    {30, 23, 22, 13, 12, 11, 10},
    { {"0xx0xxx"_b, "mla_z_p_zzz"},
      {"0xx1xxx"_b, "mls_z_p_zzz"},
      {"1101110"_b, "usdot_z_zzz_s"},
      {"1xx0000"_b, "smlalb_z_zzz"},
      {"1xx0001"_b, "smlalt_z_zzz"},
      {"1xx0010"_b, "umlalb_z_zzz"},
      {"1xx0011"_b, "umlalt_z_zzz"},
      {"1xx0100"_b, "smlslb_z_zzz"},
      {"1xx0101"_b, "smlslt_z_zzz"},
      {"1xx0110"_b, "umlslb_z_zzz"},
      {"1xx0111"_b, "umlslt_z_zzz"},
      {"1xx1000"_b, "sqdmlalb_z_zzz"},
      {"1xx1001"_b, "sqdmlalt_z_zzz"},
      {"1xx1010"_b, "sqdmlslb_z_zzz"},
      {"1xx1011"_b, "sqdmlslt_z_zzz"},
      {"1xx1100"_b, "sqrdmlah_z_zzz"},
      {"1xx1101"_b, "sqrdmlsh_z_zzz"},
    },
  },

  { "_xvrvhv",
    {4},
    { {"0"_b, "ccmp_32_condcmp_reg"},
    },
  },

  { "_xxjrsy",
    {23, 22, 9},
    { {"000"_b, "rdffr_p_p_f"},
      {"010"_b, "rdffrs_p_p_f"},
    },
  },

  { "_xxphlt",
    {23},
    { {"0"_b, "_qgshrr"},
    },
  },

  { "_xxqzvy",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "fcvtns_32d_float2int"},
      {"00001"_b, "fcvtnu_32d_float2int"},
      {"00010"_b, "scvtf_d32_float2int"},
      {"00011"_b, "ucvtf_d32_float2int"},
      {"00100"_b, "fcvtas_32d_float2int"},
      {"00101"_b, "fcvtau_32d_float2int"},
      {"01000"_b, "fcvtps_32d_float2int"},
      {"01001"_b, "fcvtpu_32d_float2int"},
      {"10000"_b, "fcvtms_32d_float2int"},
      {"10001"_b, "fcvtmu_32d_float2int"},
      {"11000"_b, "fcvtzs_32d_float2int"},
      {"11001"_b, "fcvtzu_32d_float2int"},
      {"11110"_b, "fjcvtzs_32d_float2int"},
    },
  },

  { "_xygvjp",
    {23, 22},
    { {"00"_b, "and_asimdsame_only"},
      {"01"_b, "bic_asimdsame_only"},
      {"10"_b, "orr_asimdsame_only"},
      {"11"_b, "orn_asimdsame_only"},
    },
  },

  { "_xyhmgh",
    {23, 22, 20, 9},
    { {"0000"_b, "_xhmpmy"},
      {"0001"_b, "_qnprqt"},
      {"0010"_b, "_nnzhgm"},
      {"0100"_b, "_vvxsxt"},
      {"0101"_b, "_yzmjhn"},
      {"0110"_b, "_mkgsly"},
      {"1000"_b, "_xtxyxj"},
      {"1001"_b, "_hmtmlq"},
      {"1010"_b, "_xtgtyz"},
      {"1100"_b, "_yynmjl"},
      {"1101"_b, "_sjnspg"},
      {"1110"_b, "_jzjvtv"},
    },
  },

  { "_xymnxy",
    {30},
    { {"0"_b, "tbz_only_testbranch"},
    },
  },

  { "_xynxhx",
    {30, 23, 22, 11, 10},
    { {"00010"_b, "str_b_ldst_regoff"},
      {"00110"_b, "ldr_b_ldst_regoff"},
      {"01010"_b, "str_q_ldst_regoff"},
      {"01110"_b, "ldr_q_ldst_regoff"},
      {"10010"_b, "str_h_ldst_regoff"},
      {"10110"_b, "ldr_h_ldst_regoff"},
    },
  },

  { "_xzjvkv",
    {23, 22},
    { {"00"_b, "tbl_asimdtbl_l1_1"},
    },
  },

  { "_xzlxjh",
    {30, 23, 22},
    { {"001"_b, "sbfm_64m_bitfield"},
      {"011"_b, "extr_64_extract"},
      {"101"_b, "ubfm_64m_bitfield"},
    },
  },

  { "_xzmrlg",
    {30, 23, 22},
    { {"000"_b, "stlxr_sr32_ldstexcl"},
      {"001"_b, "_zzkgsk"},
      {"010"_b, "_mnzzhk"},
      {"011"_b, "_qlxlxk"},
      {"100"_b, "stlxr_sr64_ldstexcl"},
      {"101"_b, "_tknqxs"},
      {"110"_b, "_mhpgjx"},
      {"111"_b, "_xlqmhl"},
    },
  },

  { "_xznsqh",
    {22, 20, 11},
    { {"000"_b, "cntw_r_s"},
      {"010"_b, "incw_r_rs"},
      {"100"_b, "cntd_r_s"},
      {"110"_b, "incd_r_rs"},
    },
  },

  { "_xzntxr",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0000000"_b, "fcvtns_64s_float2int"},
      {"0000001"_b, "fcvtnu_64s_float2int"},
      {"0000010"_b, "scvtf_s64_float2int"},
      {"0000011"_b, "ucvtf_s64_float2int"},
      {"0000100"_b, "fcvtas_64s_float2int"},
      {"0000101"_b, "fcvtau_64s_float2int"},
      {"0001000"_b, "fcvtps_64s_float2int"},
      {"0001001"_b, "fcvtpu_64s_float2int"},
      {"0010000"_b, "fcvtms_64s_float2int"},
      {"0010001"_b, "fcvtmu_64s_float2int"},
      {"0011000"_b, "fcvtzs_64s_float2int"},
      {"0011001"_b, "fcvtzu_64s_float2int"},
      {"0100000"_b, "fcvtns_64d_float2int"},
      {"0100001"_b, "fcvtnu_64d_float2int"},
      {"0100010"_b, "scvtf_d64_float2int"},
      {"0100011"_b, "ucvtf_d64_float2int"},
      {"0100100"_b, "fcvtas_64d_float2int"},
      {"0100101"_b, "fcvtau_64d_float2int"},
      {"0100110"_b, "fmov_64d_float2int"},
      {"0100111"_b, "fmov_d64_float2int"},
      {"0101000"_b, "fcvtps_64d_float2int"},
      {"0101001"_b, "fcvtpu_64d_float2int"},
      {"0110000"_b, "fcvtms_64d_float2int"},
      {"0110001"_b, "fcvtmu_64d_float2int"},
      {"0111000"_b, "fcvtzs_64d_float2int"},
      {"0111001"_b, "fcvtzu_64d_float2int"},
      {"1001110"_b, "fmov_64vx_float2int"},
      {"1001111"_b, "fmov_v64i_float2int"},
      {"1100000"_b, "fcvtns_64h_float2int"},
      {"1100001"_b, "fcvtnu_64h_float2int"},
      {"1100010"_b, "scvtf_h64_float2int"},
      {"1100011"_b, "ucvtf_h64_float2int"},
      {"1100100"_b, "fcvtas_64h_float2int"},
      {"1100101"_b, "fcvtau_64h_float2int"},
      {"1100110"_b, "fmov_64h_float2int"},
      {"1100111"_b, "fmov_h64_float2int"},
      {"1101000"_b, "fcvtps_64h_float2int"},
      {"1101001"_b, "fcvtpu_64h_float2int"},
      {"1110000"_b, "fcvtms_64h_float2int"},
      {"1110001"_b, "fcvtmu_64h_float2int"},
      {"1111000"_b, "fcvtzs_64h_float2int"},
      {"1111001"_b, "fcvtzu_64h_float2int"},
    },
  },

  { "_xzqmkv",
    {13, 12},
    { {"00"_b, "add_asisdsame_only"},
      {"11"_b, "sqdmulh_asisdsame_only"},
    },
  },

  { "_ygghnn",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "suqadd_asimdmisc_r"},
      {"10000"_b, "saddlv_asimdall_only"},
    },
  },

  { "_ygtpyl",
    {22, 13, 12},
    { {"000"_b, "swp_32_memop"},
      {"100"_b, "swpl_32_memop"},
    },
  },

  { "_yhhsns",
    {20, 19, 18, 17},
    { {"0000"_b, "_myrkmk"},
    },
  },

  { "_yhlntp",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "fexpa_z_z"},
    },
  },

  { "_yhmlxk",
    {13, 12, 11, 10},
    { {"0000"_b, "decp_z_p_z"},
      {"0010"_b, "decp_r_p_r"},
    },
  },

  { "_yhnqyy",
    {13, 12},
    { {"01"_b, "sqdmlal_asisddiff_only"},
      {"11"_b, "sqdmlsl_asisddiff_only"},
    },
  },

  { "_yjktml",
    {30},
    { {"0"_b, "ldr_32_loadlit"},
      {"1"_b, "ldr_64_loadlit"},
    },
  },

  { "_yjmngt",
    {30},
    { {"0"_b, "sel_z_p_zz"},
      {"1"_b, "_vpmxrj"},
    },
  },

  { "_yjnkrn",
    {30},
    { {"0"_b, "bl_only_branch_imm"},
      {"1"_b, "_grqsgp"},
    },
  },

  { "_yjnmkg",
    {30, 23, 11, 10},
    { {"0000"_b, "_szysqh"},
      {"0010"_b, "_ksrkkn"},
      {"0100"_b, "_gljqng"},
      {"0110"_b, "_qtghgs"},
      {"1000"_b, "_gjprgr"},
      {"1001"_b, "ldraa_64_ldst_pac"},
      {"1010"_b, "_gnpgsg"},
      {"1011"_b, "ldraa_64w_ldst_pac"},
      {"1100"_b, "_lnmhqq"},
      {"1101"_b, "ldrab_64_ldst_pac"},
      {"1110"_b, "_gsvlph"},
      {"1111"_b, "ldrab_64w_ldst_pac"},
    },
  },

  { "_yjzknm",
    {13, 12, 11, 10},
    { {"0000"_b, "uqdecp_z_p_z"},
      {"0010"_b, "uqdecp_r_p_r_uw"},
      {"0011"_b, "uqdecp_r_p_r_x"},
    },
  },

  { "_ykhhqq",
    {18},
    { {"0"_b, "ld2_asisdlsop_hx2_r2h"},
      {"1"_b, "ld2_asisdlsop_h2_i2h"},
    },
  },

  { "_ykjhgg",
    {30, 23, 22, 13, 12, 11, 10},
    { {"0000000"_b, "ldaddb_32_memop"},
      {"0000100"_b, "ldclrb_32_memop"},
      {"0001000"_b, "ldeorb_32_memop"},
      {"0001100"_b, "ldsetb_32_memop"},
      {"000xx10"_b, "strb_32b_ldst_regoff"},
      {"0010000"_b, "ldaddlb_32_memop"},
      {"0010100"_b, "ldclrlb_32_memop"},
      {"0011000"_b, "ldeorlb_32_memop"},
      {"0011100"_b, "ldsetlb_32_memop"},
      {"001xx10"_b, "ldrb_32b_ldst_regoff"},
      {"0100000"_b, "ldaddab_32_memop"},
      {"0100100"_b, "ldclrab_32_memop"},
      {"0101000"_b, "ldeorab_32_memop"},
      {"0101100"_b, "ldsetab_32_memop"},
      {"010xx10"_b, "ldrsb_64b_ldst_regoff"},
      {"0110000"_b, "ldaddalb_32_memop"},
      {"0110100"_b, "ldclralb_32_memop"},
      {"0111000"_b, "ldeoralb_32_memop"},
      {"0111100"_b, "ldsetalb_32_memop"},
      {"011xx10"_b, "ldrsb_32b_ldst_regoff"},
      {"1000000"_b, "ldaddh_32_memop"},
      {"1000100"_b, "ldclrh_32_memop"},
      {"1001000"_b, "ldeorh_32_memop"},
      {"1001100"_b, "ldseth_32_memop"},
      {"100xx10"_b, "strh_32_ldst_regoff"},
      {"1010000"_b, "ldaddlh_32_memop"},
      {"1010100"_b, "ldclrlh_32_memop"},
      {"1011000"_b, "ldeorlh_32_memop"},
      {"1011100"_b, "ldsetlh_32_memop"},
      {"101xx10"_b, "ldrh_32_ldst_regoff"},
      {"1100000"_b, "ldaddah_32_memop"},
      {"1100100"_b, "ldclrah_32_memop"},
      {"1101000"_b, "ldeorah_32_memop"},
      {"1101100"_b, "ldsetah_32_memop"},
      {"110xx10"_b, "ldrsh_64_ldst_regoff"},
      {"1110000"_b, "ldaddalh_32_memop"},
      {"1110100"_b, "ldclralh_32_memop"},
      {"1111000"_b, "ldeoralh_32_memop"},
      {"1111100"_b, "ldsetalh_32_memop"},
      {"111xx10"_b, "ldrsh_32_ldst_regoff"},
    },
  },

  { "_ykpgyh",
    {13, 12, 5},
    { {"010"_b, "_gknljg"},
      {"011"_b, "_hjqryy"},
      {"100"_b, "_lmmkzh"},
      {"101"_b, "_vxlmxz"},
      {"110"_b, "_phktvp"},
      {"111"_b, "_qqvgql"},
    },
  },

  { "_ykpqth",
    {12},
    { {"0"_b, "st4_asisdlsop_dx4_r4d"},
    },
  },

  { "_ykptgl",
    {30, 23},
    { {"00"_b, "adds_32s_addsub_imm"},
      {"10"_b, "subs_32s_addsub_imm"},
    },
  },

  { "_ylhgrh",
    {13, 12, 11, 10},
    { {"0011"_b, "uqadd_asisdsame_only"},
      {"1010"_b, "_msvjxq"},
      {"1011"_b, "uqsub_asisdsame_only"},
      {"1101"_b, "cmhi_asisdsame_only"},
      {"1110"_b, "_yzlnrs"},
      {"1111"_b, "cmhs_asisdsame_only"},
    },
  },

  { "_ylnsvy",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "dup_z_r"},
      {"00100"_b, "insr_z_r"},
      {"10000"_b, "sunpklo_z_z"},
      {"10001"_b, "sunpkhi_z_z"},
      {"10010"_b, "uunpklo_z_z"},
      {"10011"_b, "uunpkhi_z_z"},
      {"10100"_b, "insr_z_v"},
      {"11000"_b, "rev_z_z"},
    },
  },

  { "_ymghnh",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "st64b_64l_memop"},
    },
  },

  { "_ymhgxg",
    {30, 13},
    { {"00"_b, "_yrmmmg"},
      {"01"_b, "_sghgtk"},
      {"10"_b, "_nxjkqs"},
      {"11"_b, "_yvyhlh"},
    },
  },

  { "_ymhkrx",
    {30, 23, 22, 13, 4},
    { {"0000x"_b, "ld1b_z_p_ai_s"},
      {"0001x"_b, "ldff1b_z_p_ai_s"},
      {"0010x"_b, "ld1rb_z_p_bi_u32"},
      {"0011x"_b, "ld1rb_z_p_bi_u64"},
      {"0100x"_b, "ld1h_z_p_ai_s"},
      {"0101x"_b, "ldff1h_z_p_ai_s"},
      {"0110x"_b, "ld1rh_z_p_bi_u32"},
      {"0111x"_b, "ld1rh_z_p_bi_u64"},
      {"1000x"_b, "ld1b_z_p_ai_d"},
      {"1001x"_b, "ldff1b_z_p_ai_d"},
      {"10100"_b, "prfw_i_p_bz_d_64_scaled"},
      {"10110"_b, "prfd_i_p_bz_d_64_scaled"},
      {"1100x"_b, "ld1h_z_p_ai_d"},
      {"1101x"_b, "ldff1h_z_p_ai_d"},
      {"1110x"_b, "ld1h_z_p_bz_d_64_scaled"},
      {"1111x"_b, "ldff1h_z_p_bz_d_64_scaled"},
    },
  },

  { "_ymkthj",
    {20, 9, 4},
    { {"000"_b, "uzp2_p_pp"},
    },
  },

  { "_ymmhtq",
    {23, 22, 20, 19, 11},
    { {"00010"_b, "srsra_asisdshf_r"},
      {"001x0"_b, "srsra_asisdshf_r"},
      {"01xx0"_b, "srsra_asisdshf_r"},
    },
  },

  { "_ymszkr",
    {30},
    { {"0"_b, "ldr_q_loadlit"},
    },
  },

  { "_ymtzjg",
    {12, 10},
    { {"00"_b, "_gmsmls"},
      {"01"_b, "_rnqmyp"},
      {"10"_b, "_srttng"},
      {"11"_b, "_tymryz"},
    },
  },

  { "_ymvlzl",
    {18},
    { {"0"_b, "st4_asisdlse_r4"},
    },
  },

  { "_ymvzyh",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldaxp_lp32_ldstexcl"},
    },
  },

  { "_ymxjjr",
    {23, 22, 20, 19, 13, 11},
    { {"0000x0"_b, "orr_asimdimm_l_hl"},
      {"00x100"_b, "sqshrn_asimdshf_n"},
      {"00x101"_b, "sqrshrn_asimdshf_n"},
      {"010x00"_b, "sqshrn_asimdshf_n"},
      {"010x01"_b, "sqrshrn_asimdshf_n"},
      {"011100"_b, "sqshrn_asimdshf_n"},
      {"011101"_b, "sqrshrn_asimdshf_n"},
      {"0x1000"_b, "sqshrn_asimdshf_n"},
      {"0x1001"_b, "sqrshrn_asimdshf_n"},
    },
  },

  { "_ynsytg",
    {23, 22, 20, 19, 13, 11, 10},
    { {"0001001"_b, "shl_asisdshf_r"},
      {"0001101"_b, "sqshl_asisdshf_r"},
      {"001x001"_b, "shl_asisdshf_r"},
      {"001x101"_b, "sqshl_asisdshf_r"},
      {"00xx0x0"_b, "fmls_asisdelem_rh_h"},
      {"01xx001"_b, "shl_asisdshf_r"},
      {"01xx101"_b, "sqshl_asisdshf_r"},
      {"1xxx0x0"_b, "fmls_asisdelem_r_sd"},
      {"xxxx1x0"_b, "sqdmlsl_asisdelem_l"},
    },
  },

  { "_ynyqky",
    {12},
    { {"0"_b, "st2_asisdlsop_dx2_r2d"},
    },
  },

  { "_ynznxv",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldaxrb_lr32_ldstexcl"},
    },
  },

  { "_yppmkl",
    {23, 22, 20, 19, 13, 11},
    { {"0000x0"_b, "mvni_asimdimm_l_hl"},
      {"00x100"_b, "sqshrun_asimdshf_n"},
      {"00x101"_b, "sqrshrun_asimdshf_n"},
      {"00x110"_b, "ushll_asimdshf_l"},
      {"010x00"_b, "sqshrun_asimdshf_n"},
      {"010x01"_b, "sqrshrun_asimdshf_n"},
      {"010x10"_b, "ushll_asimdshf_l"},
      {"011100"_b, "sqshrun_asimdshf_n"},
      {"011101"_b, "sqrshrun_asimdshf_n"},
      {"011110"_b, "ushll_asimdshf_l"},
      {"0x1000"_b, "sqshrun_asimdshf_n"},
      {"0x1001"_b, "sqrshrun_asimdshf_n"},
      {"0x1010"_b, "ushll_asimdshf_l"},
    },
  },

  { "_yppszx",
    {23, 22, 10},
    { {"100"_b, "umlslb_z_zzzi_s"},
      {"101"_b, "umlslt_z_zzzi_s"},
      {"110"_b, "umlslb_z_zzzi_d"},
      {"111"_b, "umlslt_z_zzzi_d"},
    },
  },

  { "_yppyky",
    {30, 13},
    { {"00"_b, "_gyrjrm"},
      {"01"_b, "_hhkqtn"},
      {"10"_b, "_jgmlpk"},
      {"11"_b, "_tzzssm"},
    },
  },

  { "_ypsgqz",
    {18, 17},
    { {"0x"_b, "ld4_asisdlsop_sx4_r4s"},
      {"10"_b, "ld4_asisdlsop_sx4_r4s"},
      {"11"_b, "ld4_asisdlsop_s4_i4s"},
    },
  },

  { "_yptgjg",
    {4},
    { {"0"_b, "ccmn_32_condcmp_reg"},
    },
  },

  { "_yptvyx",
    {30, 23, 22},
    { {"000"_b, "strb_32_ldst_pos"},
      {"001"_b, "ldrb_32_ldst_pos"},
      {"010"_b, "ldrsb_64_ldst_pos"},
      {"011"_b, "ldrsb_32_ldst_pos"},
      {"100"_b, "strh_32_ldst_pos"},
      {"101"_b, "ldrh_32_ldst_pos"},
      {"110"_b, "ldrsh_64_ldst_pos"},
      {"111"_b, "ldrsh_32_ldst_pos"},
    },
  },

  { "_ypzllm",
    {23, 22, 4},
    { {"000"_b, "fccmp_s_floatccmp"},
      {"001"_b, "fccmpe_s_floatccmp"},
      {"010"_b, "fccmp_d_floatccmp"},
      {"011"_b, "fccmpe_d_floatccmp"},
      {"110"_b, "fccmp_h_floatccmp"},
      {"111"_b, "fccmpe_h_floatccmp"},
    },
  },

  { "_yqvqtx",
    {30, 23, 22, 20, 13},
    { {"00001"_b, "ld1rob_z_p_bi_u8"},
      {"000x0"_b, "ld1rob_z_p_br_contiguous"},
      {"01001"_b, "ld1roh_z_p_bi_u16"},
      {"010x0"_b, "ld1roh_z_p_br_contiguous"},
    },
  },

  { "_yqxnzl",
    {11, 10},
    { {"00"_b, "sqdmulh_z_zz"},
      {"01"_b, "sqrdmulh_z_zz"},
    },
  },

  { "_yqzxvr",
    {18, 17, 12},
    { {"000"_b, "ld3_asisdlso_d3_3d"},
    },
  },

  { "_yrggjm",
    {13, 12},
    { {"00"_b, "sshl_asisdsame_only"},
      {"01"_b, "srshl_asisdsame_only"},
    },
  },

  { "_yrgzqr",
    {23, 22, 20, 19, 17, 16, 13},
    { {"0000000"_b, "_ymvlzl"},
      {"0000001"_b, "_nzvlzt"},
      {"0100000"_b, "_zyhgnz"},
      {"0100001"_b, "_mntnlr"},
      {"100xxx0"_b, "st4_asisdlsep_r4_r"},
      {"100xxx1"_b, "st1_asisdlsep_r4_r4"},
      {"1010xx0"_b, "st4_asisdlsep_r4_r"},
      {"1010xx1"_b, "st1_asisdlsep_r4_r4"},
      {"10110x0"_b, "st4_asisdlsep_r4_r"},
      {"10110x1"_b, "st1_asisdlsep_r4_r4"},
      {"1011100"_b, "st4_asisdlsep_r4_r"},
      {"1011101"_b, "st1_asisdlsep_r4_r4"},
      {"1011110"_b, "_tshjsk"},
      {"1011111"_b, "_xnrrsy"},
      {"110xxx0"_b, "ld4_asisdlsep_r4_r"},
      {"110xxx1"_b, "ld1_asisdlsep_r4_r4"},
      {"1110xx0"_b, "ld4_asisdlsep_r4_r"},
      {"1110xx1"_b, "ld1_asisdlsep_r4_r4"},
      {"11110x0"_b, "ld4_asisdlsep_r4_r"},
      {"11110x1"_b, "ld1_asisdlsep_r4_r4"},
      {"1111100"_b, "ld4_asisdlsep_r4_r"},
      {"1111101"_b, "ld1_asisdlsep_r4_r4"},
      {"1111110"_b, "_hjvkkq"},
      {"1111111"_b, "_mthlnv"},
    },
  },

  { "_yrjqql",
    {30},
    { {"0"_b, "cbz_32_compbranch"},
    },
  },

  { "_yrmmmg",
    {4},
    { {"0"_b, "cmphs_p_p_zi"},
      {"1"_b, "cmphi_p_p_zi"},
    },
  },

  { "_yrypnt",
    {30, 23, 11, 10},
    { {"1001"_b, "_khrsgv"},
    },
  },

  { "_yryygq",
    {12},
    { {"0"_b, "ld3_asisdlsop_dx3_r3d"},
    },
  },

  { "_yskyrg",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "sqneg_asisdmisc_r"},
    },
  },

  { "_ysspjx",
    {13, 12},
    { {"00"_b, "sdiv_64_dp_2src"},
      {"10"_b, "rorv_64_dp_2src"},
    },
  },

  { "_yszjsm",
    {12, 11, 10},
    { {"000"_b, "sdot_z_zzz"},
      {"001"_b, "udot_z_zzz"},
      {"010"_b, "sqdmlalbt_z_zzz"},
      {"011"_b, "sqdmlslbt_z_zzz"},
      {"1xx"_b, "cdot_z_zzz"},
    },
  },

  { "_yszlqj",
    {23, 22},
    { {"00"_b, "tbl_asimdtbl_l2_2"},
    },
  },

  { "_ytkjxx",
    {30, 23, 22, 13, 4},
    { {"00x0x"_b, "ld1w_z_p_bz_s_x32_scaled"},
      {"00x1x"_b, "ldff1w_z_p_bz_s_x32_scaled"},
      {"0100x"_b, "ldr_z_bi"},
      {"01100"_b, "prfw_i_p_bi_s"},
      {"01110"_b, "prfd_i_p_bi_s"},
      {"10x0x"_b, "ld1w_z_p_bz_d_x32_scaled"},
      {"10x1x"_b, "ldff1w_z_p_bz_d_x32_scaled"},
      {"11x0x"_b, "ld1d_z_p_bz_d_x32_scaled"},
      {"11x1x"_b, "ldff1d_z_p_bz_d_x32_scaled"},
    },
  },

  { "_ytrmvz",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "fcvtmu_asisdmiscfp16_r"},
      {"0x00001"_b, "fcvtmu_asisdmisc_r"},
      {"1111001"_b, "fcvtzu_asisdmiscfp16_r"},
      {"1x00001"_b, "fcvtzu_asisdmisc_r"},
      {"xx00000"_b, "neg_asisdmisc_r"},
    },
  },

  { "_ytvtqn",
    {30, 23, 22, 20, 13},
    { {"00001"_b, "ld1sh_z_p_bi_s64"},
      {"00011"_b, "ldnf1sh_z_p_bi_s64"},
      {"00101"_b, "ld1w_z_p_bi_u32"},
      {"00111"_b, "ldnf1w_z_p_bi_u32"},
      {"01001"_b, "ld1sb_z_p_bi_s64"},
      {"01011"_b, "ldnf1sb_z_p_bi_s64"},
      {"01101"_b, "ld1sb_z_p_bi_s16"},
      {"01111"_b, "ldnf1sb_z_p_bi_s16"},
      {"100x0"_b, "st1w_z_p_bz_d_x32_unscaled"},
      {"100x1"_b, "st1w_z_p_bz_d_64_unscaled"},
      {"101x0"_b, "st1w_z_p_bz_s_x32_unscaled"},
      {"101x1"_b, "st1w_z_p_ai_d"},
      {"110x0"_b, "st1d_z_p_bz_d_x32_unscaled"},
      {"110x1"_b, "st1d_z_p_bz_d_64_unscaled"},
      {"111x1"_b, "st1d_z_p_ai_d"},
    },
  },

  { "_yvptvx",
    {23, 12, 11, 10},
    { {"0000"_b, "sqshrnb_z_zi"},
      {"0001"_b, "sqshrnt_z_zi"},
      {"0010"_b, "sqrshrnb_z_zi"},
      {"0011"_b, "sqrshrnt_z_zi"},
      {"0100"_b, "uqshrnb_z_zi"},
      {"0101"_b, "uqshrnt_z_zi"},
      {"0110"_b, "uqrshrnb_z_zi"},
      {"0111"_b, "uqrshrnt_z_zi"},
    },
  },

  { "_yvqnyq",
    {23},
    { {"1"_b, "_vhlqpr"},
    },
  },

  { "_yvxkhv",
    {30},
    { {"1"_b, "_ngvqhs"},
    },
  },

  { "_yvyhlh",
    {23, 22, 12, 11, 10},
    { {"0x000"_b, "fmul_z_zzi_h"},
      {"10000"_b, "fmul_z_zzi_s"},
      {"11000"_b, "fmul_z_zzi_d"},
    },
  },

  { "_yxgmrs",
    {23},
    { {"0"_b, "fmaxnmp_asimdsame_only"},
      {"1"_b, "fminnmp_asimdsame_only"},
    },
  },

  { "_yxnslx",
    {23, 22},
    { {"00"_b, "adr_z_az_d_s32_scaled"},
      {"01"_b, "adr_z_az_d_u32_scaled"},
      {"1x"_b, "adr_z_az_sd_same_scaled"},
    },
  },

  { "_yxvttm",
    {30},
    { {"0"_b, "bl_only_branch_imm"},
    },
  },

  { "_yykhjv",
    {23, 22, 13, 12, 11, 10},
    { {"000110"_b, "smmla_z_zzz"},
      {"0x1000"_b, "sshllb_z_zi"},
      {"0x1001"_b, "sshllt_z_zi"},
      {"0x1010"_b, "ushllb_z_zi"},
      {"0x1011"_b, "ushllt_z_zi"},
      {"100110"_b, "usmmla_z_zzz"},
      {"110110"_b, "ummla_z_zzz"},
      {"xx0000"_b, "saddlbt_z_zz"},
      {"xx0010"_b, "ssublbt_z_zz"},
      {"xx0011"_b, "ssubltb_z_zz"},
      {"xx0100"_b, "eorbt_z_zz"},
      {"xx0101"_b, "eortb_z_zz"},
      {"xx1100"_b, "bext_z_zz"},
      {"xx1101"_b, "bdep_z_zz"},
      {"xx1110"_b, "bgrp_z_zz"},
    },
  },

  { "_yynmjl",
    {4},
    { {"0"_b, "orrs_p_p_pp_z"},
      {"1"_b, "orns_p_p_pp_z"},
    },
  },

  { "_yyrkmn",
    {17, 16, 9, 8, 7, 6, 5},
    { {"0000000"_b, "aesmc_z_z"},
      {"10xxxxx"_b, "aese_z_zz"},
      {"11xxxxx"_b, "sm4e_z_zz"},
    },
  },

  { "_yysxts",
    {23, 22, 13, 12, 11, 10},
    { {"0001x0"_b, "fmla_asimdelem_rh_h"},
      {"0x0001"_b, "sshr_asimdshf_r"},
      {"0x0101"_b, "ssra_asimdshf_r"},
      {"0x1001"_b, "srshr_asimdshf_r"},
      {"0x1101"_b, "srsra_asimdshf_r"},
      {"1000x0"_b, "fmlal_asimdelem_lh"},
      {"1x01x0"_b, "fmla_asimdelem_r_sd"},
      {"xx10x0"_b, "smlal_asimdelem_l"},
      {"xx11x0"_b, "sqdmlal_asimdelem_l"},
    },
  },

  { "_yytvxh",
    {30, 23, 22, 13, 4},
    { {"00000"_b, "prfw_i_p_br_s"},
      {"00010"_b, "prfw_i_p_ai_s"},
      {"0010x"_b, "ld1rw_z_p_bi_u32"},
      {"0011x"_b, "ld1rw_z_p_bi_u64"},
      {"01000"_b, "prfd_i_p_br_s"},
      {"01010"_b, "prfd_i_p_ai_s"},
      {"0110x"_b, "ld1rsb_z_p_bi_s16"},
      {"0111x"_b, "ld1rd_z_p_bi_u64"},
      {"1000x"_b, "ldnt1w_z_p_ar_d_64_unscaled"},
      {"10010"_b, "prfw_i_p_ai_d"},
      {"1010x"_b, "ld1w_z_p_bz_d_64_unscaled"},
      {"1011x"_b, "ldff1w_z_p_bz_d_64_unscaled"},
      {"1100x"_b, "ldnt1d_z_p_ar_d_64_unscaled"},
      {"11010"_b, "prfd_i_p_ai_d"},
      {"1110x"_b, "ld1d_z_p_bz_d_64_unscaled"},
      {"1111x"_b, "ldff1d_z_p_bz_d_64_unscaled"},
    },
  },

  { "_yyvjqv",
    {23},
    { {"0"_b, "fmax_asimdsame_only"},
      {"1"_b, "fmin_asimdsame_only"},
    },
  },

  { "_yyvnrp",
    {23, 22},
    { {"00"_b, "eor_asimdsame_only"},
      {"01"_b, "bsl_asimdsame_only"},
      {"10"_b, "bit_asimdsame_only"},
      {"11"_b, "bif_asimdsame_only"},
    },
  },

  { "_yyyshx",
    {30, 13, 4},
    { {"000"_b, "cmphs_p_p_zz"},
      {"001"_b, "cmphi_p_p_zz"},
      {"010"_b, "cmpeq_p_p_zw"},
      {"011"_b, "cmpne_p_p_zw"},
      {"1xx"_b, "fcmla_z_p_zzz"},
    },
  },

  { "_yyyxhk",
    {18},
    { {"0"_b, "ld1_asisdlsep_r2_r2"},
      {"1"_b, "ld1_asisdlsep_i2_i2"},
    },
  },

  { "_yzgthp",
    {18, 17},
    { {"0x"_b, "ld1_asisdlsop_sx1_r1s"},
      {"10"_b, "ld1_asisdlsop_sx1_r1s"},
      {"11"_b, "ld1_asisdlsop_s1_i1s"},
    },
  },

  { "_yzlnrs",
    {20, 19, 18, 17, 16},
    { {"00000"_b, "usqadd_asisdmisc_r"},
    },
  },

  { "_yzmjhn",
    {4},
    { {"0"_b, "eors_p_p_pp_z"},
    },
  },

  { "_yzpszn",
    {30},
    { {"0"_b, "ldr_s_loadlit"},
      {"1"_b, "ldr_d_loadlit"},
    },
  },

  { "_yzqhtj",
    {30, 23, 22, 11, 10},
    { {"00000"_b, "_rxsqhv"},
      {"01000"_b, "csel_64_condsel"},
      {"01001"_b, "csinc_64_condsel"},
      {"01100"_b, "_zqxkxg"},
      {"01101"_b, "_rvjkyp"},
      {"01110"_b, "_jxgpgg"},
      {"01111"_b, "_ysspjx"},
      {"10000"_b, "_pjvkjz"},
      {"11000"_b, "csinv_64_condsel"},
      {"11001"_b, "csneg_64_condsel"},
      {"11100"_b, "_rmyzpp"},
      {"11101"_b, "_npjnlv"},
      {"11110"_b, "_yhhsns"},
      {"11111"_b, "_vllmnt"},
    },
  },

  { "_yzxjnk",
    {9, 8, 7, 6, 5},
    { {"11111"_b, "paciza_64z_dp_1src"},
    },
  },

  { "_zghtll",
    {22, 20, 19, 18, 17, 16, 13, 12},
    { {"01111100"_b, "ldapr_32l_memop"},
    },
  },

  { "_zgljvg",
    {30, 23, 22, 13, 12, 11, 10},
    { {"0000000"_b, "ldadd_32_memop"},
      {"0000100"_b, "ldclr_32_memop"},
      {"0001000"_b, "ldeor_32_memop"},
      {"0001100"_b, "ldset_32_memop"},
      {"000xx10"_b, "str_32_ldst_regoff"},
      {"0010000"_b, "ldaddl_32_memop"},
      {"0010100"_b, "ldclrl_32_memop"},
      {"0011000"_b, "ldeorl_32_memop"},
      {"0011100"_b, "ldsetl_32_memop"},
      {"001xx10"_b, "ldr_32_ldst_regoff"},
      {"0100000"_b, "ldadda_32_memop"},
      {"0100100"_b, "ldclra_32_memop"},
      {"0101000"_b, "ldeora_32_memop"},
      {"0101100"_b, "ldseta_32_memop"},
      {"010xx10"_b, "ldrsw_64_ldst_regoff"},
      {"0110000"_b, "ldaddal_32_memop"},
      {"0110100"_b, "ldclral_32_memop"},
      {"0111000"_b, "ldeoral_32_memop"},
      {"0111100"_b, "ldsetal_32_memop"},
      {"1000000"_b, "ldadd_64_memop"},
      {"1000100"_b, "ldclr_64_memop"},
      {"1001000"_b, "ldeor_64_memop"},
      {"1001100"_b, "ldset_64_memop"},
      {"100xx10"_b, "str_64_ldst_regoff"},
      {"1010000"_b, "ldaddl_64_memop"},
      {"1010100"_b, "ldclrl_64_memop"},
      {"1011000"_b, "ldeorl_64_memop"},
      {"1011100"_b, "ldsetl_64_memop"},
      {"101xx10"_b, "ldr_64_ldst_regoff"},
      {"10xxx01"_b, "ldraa_64_ldst_pac"},
      {"10xxx11"_b, "ldraa_64w_ldst_pac"},
      {"1100000"_b, "ldadda_64_memop"},
      {"1100100"_b, "ldclra_64_memop"},
      {"1101000"_b, "ldeora_64_memop"},
      {"1101100"_b, "ldseta_64_memop"},
      {"1110000"_b, "ldaddal_64_memop"},
      {"1110100"_b, "ldclral_64_memop"},
      {"1111000"_b, "ldeoral_64_memop"},
      {"1111100"_b, "ldsetal_64_memop"},
      {"11xxx01"_b, "ldrab_64_ldst_pac"},
      {"11xxx11"_b, "ldrab_64w_ldst_pac"},
    },
  },

  { "_zjjxjl",
    {9},
    { {"0"_b, "pnext_p_p_p"},
    },
  },

  { "_zjqssg",
    {23, 22, 20, 19, 17, 16, 13},
    { {"0000000"_b, "_jqsjtj"},
      {"0000001"_b, "_rspmth"},
      {"0100000"_b, "_txkmvh"},
      {"0100001"_b, "_ngnxrx"},
      {"100xxx0"_b, "st2_asisdlsep_r2_r"},
      {"100xxx1"_b, "st1_asisdlsep_r2_r2"},
      {"1010xx0"_b, "st2_asisdlsep_r2_r"},
      {"1010xx1"_b, "st1_asisdlsep_r2_r2"},
      {"10110x0"_b, "st2_asisdlsep_r2_r"},
      {"10110x1"_b, "st1_asisdlsep_r2_r2"},
      {"1011100"_b, "st2_asisdlsep_r2_r"},
      {"1011101"_b, "st1_asisdlsep_r2_r2"},
      {"1011110"_b, "_zyzsql"},
      {"1011111"_b, "_kqsqly"},
      {"110xxx0"_b, "ld2_asisdlsep_r2_r"},
      {"110xxx1"_b, "ld1_asisdlsep_r2_r2"},
      {"1110xx0"_b, "ld2_asisdlsep_r2_r"},
      {"1110xx1"_b, "ld1_asisdlsep_r2_r2"},
      {"11110x0"_b, "ld2_asisdlsep_r2_r"},
      {"11110x1"_b, "ld1_asisdlsep_r2_r2"},
      {"1111100"_b, "ld2_asisdlsep_r2_r"},
      {"1111101"_b, "ld1_asisdlsep_r2_r2"},
      {"1111110"_b, "_xnrxym"},
      {"1111111"_b, "_yyyxhk"},
    },
  },

  { "_zjrsrx",
    {30, 23},
    { {"00"_b, "add_64_addsub_imm"},
      {"10"_b, "sub_64_addsub_imm"},
    },
  },

  { "_zjzmvh",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0001010"_b, "fcvtx_z_p_z_d2s"},
      {"0011xx0"_b, "flogb_z_p_z"},
      {"0110010"_b, "scvtf_z_p_z_h2fp16"},
      {"0110011"_b, "ucvtf_z_p_z_h2fp16"},
      {"0110100"_b, "scvtf_z_p_z_w2fp16"},
      {"0110101"_b, "ucvtf_z_p_z_w2fp16"},
      {"0110110"_b, "scvtf_z_p_z_x2fp16"},
      {"0110111"_b, "ucvtf_z_p_z_x2fp16"},
      {"0111010"_b, "fcvtzs_z_p_z_fp162h"},
      {"0111011"_b, "fcvtzu_z_p_z_fp162h"},
      {"0111100"_b, "fcvtzs_z_p_z_fp162w"},
      {"0111101"_b, "fcvtzu_z_p_z_fp162w"},
      {"0111110"_b, "fcvtzs_z_p_z_fp162x"},
      {"0111111"_b, "fcvtzu_z_p_z_fp162x"},
      {"1001000"_b, "fcvt_z_p_z_s2h"},
      {"1001001"_b, "fcvt_z_p_z_h2s"},
      {"1001010"_b, "bfcvt_z_p_z_s2bf"},
      {"1010100"_b, "scvtf_z_p_z_w2s"},
      {"1010101"_b, "ucvtf_z_p_z_w2s"},
      {"1011100"_b, "fcvtzs_z_p_z_s2w"},
      {"1011101"_b, "fcvtzu_z_p_z_s2w"},
      {"1101000"_b, "fcvt_z_p_z_d2h"},
      {"1101001"_b, "fcvt_z_p_z_h2d"},
      {"1101010"_b, "fcvt_z_p_z_d2s"},
      {"1101011"_b, "fcvt_z_p_z_s2d"},
      {"1110000"_b, "scvtf_z_p_z_w2d"},
      {"1110001"_b, "ucvtf_z_p_z_w2d"},
      {"1110100"_b, "scvtf_z_p_z_x2s"},
      {"1110101"_b, "ucvtf_z_p_z_x2s"},
      {"1110110"_b, "scvtf_z_p_z_x2d"},
      {"1110111"_b, "ucvtf_z_p_z_x2d"},
      {"1111000"_b, "fcvtzs_z_p_z_d2w"},
      {"1111001"_b, "fcvtzu_z_p_z_d2w"},
      {"1111100"_b, "fcvtzs_z_p_z_s2x"},
      {"1111101"_b, "fcvtzu_z_p_z_s2x"},
      {"1111110"_b, "fcvtzs_z_p_z_d2x"},
      {"1111111"_b, "fcvtzu_z_p_z_d2x"},
      {"xx00000"_b, "frintn_z_p_z"},
      {"xx00001"_b, "frintp_z_p_z"},
      {"xx00010"_b, "frintm_z_p_z"},
      {"xx00011"_b, "frintz_z_p_z"},
      {"xx00100"_b, "frinta_z_p_z"},
      {"xx00110"_b, "frintx_z_p_z"},
      {"xx00111"_b, "frinti_z_p_z"},
      {"xx01100"_b, "frecpx_z_p_z"},
      {"xx01101"_b, "fsqrt_z_p_z"},
    },
  },

  { "_zkhjsp",
    {11},
    { {"0"_b, "sqdmulh_z_zzi_h"},
      {"1"_b, "mul_z_zzi_h"},
    },
  },

  { "_zlhlqy",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "frintm_asimdmiscfp16_r"},
      {"0x00001"_b, "frintm_asimdmisc_r"},
      {"1111001"_b, "frintz_asimdmiscfp16_r"},
      {"1x00001"_b, "frintz_asimdmisc_r"},
      {"xx00000"_b, "cmeq_asimdmisc_z"},
    },
  },

  { "_zlkygr",
    {13, 12},
    { {"00"_b, "cpyfe_cpy_memcms"},
      {"01"_b, "cpyfewt_cpy_memcms"},
      {"10"_b, "cpyfert_cpy_memcms"},
      {"11"_b, "cpyfet_cpy_memcms"},
    },
  },

  { "_zlmyjt",
    {23, 22},
    { {"00"_b, "fcsel_s_floatsel"},
      {"01"_b, "fcsel_d_floatsel"},
      {"11"_b, "fcsel_h_floatsel"},
    },
  },

  { "_zlqnks",
    {23, 22, 20, 19, 17, 16, 13},
    { {"0000000"_b, "_kmqlmz"},
      {"0000001"_b, "_tklxhy"},
      {"0100000"_b, "_mtshvn"},
      {"0100001"_b, "_gzzsgh"},
      {"100xxx0"_b, "st1_asisdlsop_bx1_r1b"},
      {"100xxx1"_b, "st3_asisdlsop_bx3_r3b"},
      {"1010xx0"_b, "st1_asisdlsop_bx1_r1b"},
      {"1010xx1"_b, "st3_asisdlsop_bx3_r3b"},
      {"10110x0"_b, "st1_asisdlsop_bx1_r1b"},
      {"10110x1"_b, "st3_asisdlsop_bx3_r3b"},
      {"1011100"_b, "st1_asisdlsop_bx1_r1b"},
      {"1011101"_b, "st3_asisdlsop_bx3_r3b"},
      {"1011110"_b, "_tvrlgz"},
      {"1011111"_b, "_nkmkvz"},
      {"110xxx0"_b, "ld1_asisdlsop_bx1_r1b"},
      {"110xxx1"_b, "ld3_asisdlsop_bx3_r3b"},
      {"1110xx0"_b, "ld1_asisdlsop_bx1_r1b"},
      {"1110xx1"_b, "ld3_asisdlsop_bx3_r3b"},
      {"11110x0"_b, "ld1_asisdlsop_bx1_r1b"},
      {"11110x1"_b, "ld3_asisdlsop_bx3_r3b"},
      {"1111100"_b, "ld1_asisdlsop_bx1_r1b"},
      {"1111101"_b, "ld3_asisdlsop_bx3_r3b"},
      {"1111110"_b, "_kkpxth"},
      {"1111111"_b, "_rlylxh"},
    },
  },

  { "_zlvjrh",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldxr_lr32_ldstexcl"},
    },
  },

  { "_zmhqmr",
    {13, 12},
    { {"10"_b, "lsrv_32_dp_2src"},
    },
  },

  { "_zmkntq",
    {18},
    { {"0"_b, "ld1_asisdlsop_hx1_r1h"},
      {"1"_b, "ld1_asisdlsop_h1_i1h"},
    },
  },

  { "_zmkqxl",
    {23, 10},
    { {"00"_b, "adclb_z_zzz"},
      {"01"_b, "adclt_z_zzz"},
      {"10"_b, "sbclb_z_zzz"},
      {"11"_b, "sbclt_z_zzz"},
    },
  },

  { "_zmrhxx",
    {30, 23, 22},
    { {"000"_b, "smov_asimdins_w_w"},
      {"100"_b, "smov_asimdins_x_x"},
    },
  },

  { "_zmtkvx",
    {13, 10},
    { {"00"_b, "_rhpmjz"},
    },
  },

  { "_zpjzst",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0111001"_b, "fcvtnu_asimdmiscfp16_r"},
      {"0x00001"_b, "fcvtnu_asimdmisc_r"},
      {"1111001"_b, "fcvtpu_asimdmiscfp16_r"},
      {"1x00001"_b, "fcvtpu_asimdmisc_r"},
      {"xx10000"_b, "umaxv_asimdall_only"},
      {"xx10001"_b, "uminv_asimdall_only"},
    },
  },

  { "_zprgxt",
    {18, 17, 12},
    { {"0x0"_b, "st4_asisdlsop_dx4_r4d"},
      {"100"_b, "st4_asisdlsop_dx4_r4d"},
      {"110"_b, "st4_asisdlsop_d4_i4d"},
    },
  },

  { "_zpxrnm",
    {30, 23, 22},
    { {"110"_b, "xar_vvv2_crypto3_imm6"},
    },
  },

  { "_zqhhlq",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "stllrh_sl32_ldstexcl"},
    },
  },

  { "_zqjgzz",
    {30, 23, 22},
    { {"000"_b, "add_64_addsub_ext"},
      {"100"_b, "sub_64_addsub_ext"},
    },
  },

  { "_zqlzzp",
    {2, 1},
    { {"11"_b, "braaz_64_branch_reg"},
    },
  },

  { "_zqmrhp",
    {23, 22, 4, 3, 2, 1, 0},
    { {"0000000"_b, "wrffr_f_p"},
    },
  },

  { "_zqmvqs",
    {23, 22, 20, 19, 16, 13, 12},
    { {"0111110"_b, "fcvtns_asisdmiscfp16_r"},
      {"0111111"_b, "fcvtms_asisdmiscfp16_r"},
      {"0x00110"_b, "fcvtns_asisdmisc_r"},
      {"0x00111"_b, "fcvtms_asisdmisc_r"},
      {"1111110"_b, "fcvtps_asisdmiscfp16_r"},
      {"1111111"_b, "fcvtzs_asisdmiscfp16_r"},
      {"1x00110"_b, "fcvtps_asisdmisc_r"},
      {"1x00111"_b, "fcvtzs_asisdmisc_r"},
      {"xx00000"_b, "cmgt_asisdmisc_z"},
      {"xx00001"_b, "cmeq_asisdmisc_z"},
      {"xx00010"_b, "cmlt_asisdmisc_z"},
      {"xx00011"_b, "abs_asisdmisc_r"},
      {"xx10111"_b, "addp_asisdpair_only"},
    },
  },

  { "_zqxkxg",
    {13, 12},
    { {"00"_b, "subp_64s_dp_2src"},
      {"01"_b, "irg_64i_dp_2src"},
      {"10"_b, "lslv_64_dp_2src"},
      {"11"_b, "pacga_64p_dp_2src"},
    },
  },

  { "_zrmgjx",
    {30, 23, 22, 13, 4},
    { {"01000"_b, "ldr_p_bi"},
      {"01100"_b, "prfb_i_p_bi_s"},
      {"01110"_b, "prfh_i_p_bi_s"},
      {"10x0x"_b, "ld1sw_z_p_bz_d_x32_unscaled"},
      {"10x1x"_b, "ldff1sw_z_p_bz_d_x32_unscaled"},
    },
  },

  { "_zrpzss",
    {30, 23, 22, 13, 12, 11, 10},
    { {"0000000"_b, "swpp_128_memop_128"},
      {"0000100"_b, "rcwclrp_128_memop_128"},
      {"0001000"_b, "rcwswpp_128_memop_128"},
      {"0001100"_b, "rcwsetp_128_memop_128"},
      {"0010000"_b, "swppl_128_memop_128"},
      {"0010100"_b, "rcwclrpl_128_memop_128"},
      {"0011000"_b, "rcwswppl_128_memop_128"},
      {"0011100"_b, "rcwsetpl_128_memop_128"},
      {"0100000"_b, "swppa_128_memop_128"},
      {"0100100"_b, "rcwclrpa_128_memop_128"},
      {"0101000"_b, "rcwswppa_128_memop_128"},
      {"0101100"_b, "rcwsetpa_128_memop_128"},
      {"0110000"_b, "swppal_128_memop_128"},
      {"0110100"_b, "rcwclrpal_128_memop_128"},
      {"0111000"_b, "rcwswppal_128_memop_128"},
      {"0111100"_b, "rcwsetpal_128_memop_128"},
      {"1000100"_b, "rcwsclrp_128_memop_128"},
      {"1001000"_b, "rcwsswpp_128_memop_128"},
      {"1001100"_b, "rcwssetp_128_memop_128"},
      {"1010100"_b, "rcwsclrpl_128_memop_128"},
      {"1011000"_b, "rcwsswppl_128_memop_128"},
      {"1011100"_b, "rcwssetpl_128_memop_128"},
      {"1100100"_b, "rcwsclrpa_128_memop_128"},
      {"1101000"_b, "rcwsswppa_128_memop_128"},
      {"1101100"_b, "rcwssetpa_128_memop_128"},
      {"1110100"_b, "rcwsclrpal_128_memop_128"},
      {"1111000"_b, "rcwsswppal_128_memop_128"},
      {"1111100"_b, "rcwssetpal_128_memop_128"},
    },
  },

  { "_zrqtgx",
    {30},
    { {"0"_b, "bl_only_branch_imm"},
      {"1"_b, "_rxnnvv"},
    },
  },

  { "_zrxhzq",
    {19},
    { {"0"_b, "_kjsrkm"},
      {"1"_b, "sys_cr_systeminstrs"},
    },
  },

  { "_zryvjk",
    {20, 9, 4},
    { {"000"_b, "trn2_p_pp"},
    },
  },

  { "_zsgpsn",
    {20, 19, 18, 17, 16, 13, 12, 3, 2, 1, 0},
    { {"00000001101"_b, "setf16_only_setf"},
    },
  },

  { "_zsltyl",
    {22, 20, 11},
    { {"000"_b, "uqincw_r_rs_uw"},
      {"001"_b, "uqdecw_r_rs_uw"},
      {"010"_b, "uqincw_r_rs_x"},
      {"011"_b, "uqdecw_r_rs_x"},
      {"100"_b, "uqincd_r_rs_uw"},
      {"101"_b, "uqdecd_r_rs_uw"},
      {"110"_b, "uqincd_r_rs_x"},
      {"111"_b, "uqdecd_r_rs_x"},
    },
  },

  { "_zspprz",
    {20, 19, 17, 16, 12, 11, 10},
    { {"0000xxx"_b, "_srnkng"},
      {"0001xxx"_b, "_thkkgx"},
      {"0010xxx"_b, "_grgrpt"},
      {"0011xxx"_b, "_rkskkv"},
      {"0110100"_b, "_rvsylx"},
      {"0111100"_b, "_plymgg"},
      {"1000xxx"_b, "_prytjs"},
      {"1001xxx"_b, "_rrvltp"},
      {"1010xxx"_b, "_syrmmr"},
      {"1011xxx"_b, "_lnkrzt"},
      {"1100xxx"_b, "_smmrpj"},
    },
  },

  { "_ztjjnh",
    {30, 23, 22},
    { {"100"_b, "eor3_vvv16_crypto4"},
      {"101"_b, "sm3ss1_vvv4_crypto4"},
      {"110"_b, "xar_vvv2_crypto3_imm6"},
    },
  },

  { "_ztlysk",
    {23, 22, 20, 19, 18, 17, 16},
    { {"0010000"_b, "fmaxnmv_asimdall_only_h"},
      {"0111001"_b, "fcvtas_asimdmiscfp16_r"},
      {"0x00001"_b, "fcvtas_asimdmisc_r"},
      {"1010000"_b, "fminnmv_asimdall_only_h"},
      {"1111000"_b, "fcmgt_asimdmiscfp16_fz"},
      {"1x00000"_b, "fcmgt_asimdmisc_fz"},
      {"1x00001"_b, "urecpe_asimdmisc_r"},
    },
  },

  { "_ztpryr",
    {13},
    { {"0"_b, "fmad_z_p_zzz"},
      {"1"_b, "fmsb_z_p_zzz"},
    },
  },

  { "_ztyqrj",
    {30, 23, 13, 12, 10},
    { {"00000"_b, "_jmvgsp"},
      {"00001"_b, "_jkkqvy"},
      {"00100"_b, "_nkxhsy"},
      {"00101"_b, "_gshrzq"},
      {"00110"_b, "_zvjrlz"},
      {"00111"_b, "_ntjpsx"},
      {"01000"_b, "_mqrzzk"},
      {"01001"_b, "_jqxqql"},
      {"01100"_b, "_xznsqh"},
      {"01101"_b, "_qvlnll"},
      {"01110"_b, "_kvnqhn"},
      {"01111"_b, "_zsltyl"},
      {"10110"_b, "_zkhjsp"},
      {"10111"_b, "_hvyjnk"},
      {"11000"_b, "_sjvhlq"},
      {"11001"_b, "_xhktsk"},
      {"11010"_b, "_rtpztp"},
      {"11011"_b, "_rznrqt"},
      {"11100"_b, "_kyspnn"},
      {"11101"_b, "_qljhnp"},
      {"11110"_b, "_pxyrpm"},
      {"11111"_b, "_khjvqq"},
    },
  },

  { "_zvjrlz",
    {22, 20, 11},
    { {"000"_b, "sqincb_r_rs_sx"},
      {"001"_b, "sqdecb_r_rs_sx"},
      {"010"_b, "sqincb_r_rs_x"},
      {"011"_b, "sqdecb_r_rs_x"},
      {"100"_b, "sqinch_r_rs_sx"},
      {"101"_b, "sqdech_r_rs_sx"},
      {"110"_b, "sqinch_r_rs_x"},
      {"111"_b, "sqdech_r_rs_x"},
    },
  },

  { "_zvvvhr",
    {13, 12, 11, 10},
    { {"0000"_b, "smlal_asimddiff_l"},
      {"0001"_b, "add_asimdsame_only"},
      {"0010"_b, "_njnsqm"},
      {"0011"_b, "cmtst_asimdsame_only"},
      {"0100"_b, "sqdmlal_asimddiff_l"},
      {"0101"_b, "mla_asimdsame_only"},
      {"0110"_b, "_zlhlqy"},
      {"0111"_b, "mul_asimdsame_only"},
      {"1000"_b, "smlsl_asimddiff_l"},
      {"1001"_b, "smaxp_asimdsame_only"},
      {"1010"_b, "_nknntn"},
      {"1011"_b, "sminp_asimdsame_only"},
      {"1100"_b, "sqdmlsl_asimddiff_l"},
      {"1101"_b, "sqdmulh_asimdsame_only"},
      {"1110"_b, "_lyzhrq"},
      {"1111"_b, "addp_asimdsame_only"},
    },
  },

  { "_zvxxjk",
    {30, 23, 22, 13, 12, 11, 10},
    { {"000xxxx"_b, "madd_64a_dp_3src"},
      {"0011111"_b, "smulh_64_dp_3src"},
      {"0111111"_b, "umulh_64_dp_3src"},
    },
  },

  { "_zvynrg",
    {19},
    { {"0"_b, "_hnkyxy"},
      {"1"_b, "sys_cr_systeminstrs"},
    },
  },

  { "_zxjkmj",
    {22, 4, 3},
    { {"00x"_b, "prfm_p_ldst_regoff"},
      {"010"_b, "prfm_p_ldst_regoff"},
      {"011"_b, "rprfm_r_ldst_regoff"},
    },
  },

  { "_zxklzp",
    {12},
    { {"0"_b, "ld1_asisdlsop_dx1_r1d"},
    },
  },

  { "_zxtzmv",
    {30, 23, 22, 13},
    { {"0010"_b, "ld1rsh_z_p_bi_s64"},
      {"0011"_b, "ld1rsh_z_p_bi_s32"},
      {"0110"_b, "ld1rsb_z_p_bi_s64"},
      {"0111"_b, "ld1rsb_z_p_bi_s32"},
      {"1000"_b, "ld1sw_z_p_ai_d"},
      {"1001"_b, "ldff1sw_z_p_ai_d"},
      {"1010"_b, "ld1sw_z_p_bz_d_64_scaled"},
      {"1011"_b, "ldff1sw_z_p_bz_d_64_scaled"},
    },
  },

  { "_zyhgnz",
    {18},
    { {"0"_b, "ld4_asisdlse_r4"},
    },
  },

  { "_zyjjgs",
    {23, 22, 20, 19, 18},
    { {"00000"_b, "orr_z_zi"},
      {"01000"_b, "eor_z_zi"},
      {"10000"_b, "and_z_zi"},
      {"11000"_b, "dupm_z_i"},
      {"xx1xx"_b, "cpy_z_o_i"},
    },
  },

  { "_zyxnpz",
    {13, 12, 11, 10},
    { {"1111"_b, "casa_c32_ldstexcl"},
    },
  },

  { "_zyzsql",
    {18},
    { {"0"_b, "st2_asisdlsep_r2_r"},
      {"1"_b, "st2_asisdlsep_i2_i"},
    },
  },

  { "_zzhnxv",
    {30, 23, 22, 20, 19},
    { {"0xxxx"_b, "bl_only_branch_imm"},
      {"10001"_b, "sysl_rc_systeminstrs"},
      {"1001x"_b, "mrs_rs_systemmove"},
      {"1011x"_b, "mrrs_rs_systemmovepr"},
    },
  },

  { "_zzkgsk",
    {20, 19, 18, 17, 16},
    { {"11111"_b, "ldaxr_lr32_ldstexcl"},
    },
  },

  { "_zztypv",
    {6, 5},
    { {"00"_b, "cfinv_m_pstate"},
      {"01"_b, "xaflag_m_pstate"},
      {"10"_b, "axflag_m_pstate"},
    },
  },

  { "_zzvxvh",
    {23, 22, 11, 10},
    { {"0001"_b, "pmul_z_zz"},
      {"xx00"_b, "mul_z_zz"},
      {"xx10"_b, "smulh_z_zz"},
      {"xx11"_b, "umulh_z_zz"},
    },
  },

  { "Root",
    {31, 29, 28, 27, 26, 25, 24, 21, 15, 14},
    { {"00000000xx"_b, "_nqmnzp"},
      {"0000100000"_b, "_rzzxsn"},
      {"0000100001"_b, "_xvppmm"},
      {"0000100010"_b, "_ptsjnr"},
      {"0000100011"_b, "_nlpmvl"},
      {"0000100100"_b, "_ljljkv"},
      {"0000100101"_b, "_kktglv"},
      {"0000100110"_b, "_ppnssm"},
      {"0000100111"_b, "_ztyqrj"},
      {"0000101000"_b, "_rnqtmt"},
      {"0000101001"_b, "_rlpmrx"},
      {"0000101010"_b, "_mpvsng"},
      {"0000101011"_b, "_qlxksl"},
      {"0000101100"_b, "_mhrjvp"},
      {"0000101101"_b, "_pgjjsz"},
      {"0000101110"_b, "_yppyky"},
      {"0000101111"_b, "_yjmngt"},
      {"0001000001"_b, "_thqgrq"},
      {"0001000011"_b, "_hkgzsh"},
      {"0001000101"_b, "_ktpxrr"},
      {"0001000111"_b, "_stlgrr"},
      {"00010100xx"_b, "_vtyqhh"},
      {"00010101xx"_b, "_tytzpq"},
      {"00010110xx"_b, "_tqlsyy"},
      {"00010111xx"_b, "_htkpks"},
      {"0001100000"_b, "_myvqtn"},
      {"0001100001"_b, "_nmqskh"},
      {"0001100010"_b, "_xrkzpn"},
      {"0001101000"_b, "_zlqnks"},
      {"0001101001"_b, "_vtllgt"},
      {"0001101010"_b, "_ghqqzy"},
      {"0001101011"_b, "_xrskrk"},
      {"0001101100"_b, "_rzpqmm"},
      {"0001101101"_b, "_pyvvqx"},
      {"0001101110"_b, "_shgktt"},
      {"0001101111"_b, "_szylpy"},
      {"0001110000"_b, "_jgxqzr"},
      {"0001110001"_b, "_jrqxvn"},
      {"0001110010"_b, "_lplpkk"},
      {"0001110100"_b, "_kgpsjz"},
      {"0001110101"_b, "_hsrkqt"},
      {"0001110110"_b, "_zvvvhr"},
      {"0001110111"_b, "_kssltr"},
      {"0001111000"_b, "_vzzqhx"},
      {"0001111001"_b, "_ktngnm"},
      {"0001111010"_b, "_ttsgkt"},
      {"0001111011"_b, "_phtxqg"},
      {"0001111100"_b, "_yysxts"},
      {"0001111101"_b, "_msnshr"},
      {"0001111110"_b, "_nmqrtr"},
      {"0001111111"_b, "_gnxrlr"},
      {"00100010xx"_b, "_hmjrmm"},
      {"00100011xx"_b, "_nxlmhz"},
      {"0010010xxx"_b, "_hqkljv"},
      {"001001100x"_b, "_hvrjyt"},
      {"001001110x"_b, "_kgygky"},
      {"0010011x1x"_b, "_lkpprr"},
      {"0010100xxx"_b, "_vyjsst"},
      {"0010110xxx"_b, "_qvjmmq"},
      {"00101x1xxx"_b, "_lxggmz"},
      {"0011000xxx"_b, "_yjktml"},
      {"0011001000"_b, "_mqtgvk"},
      {"0011001001"_b, "_hvnhmh"},
      {"0011001010"_b, "_gsnnnt"},
      {"0011001011"_b, "_vxvyyg"},
      {"0011001100"_b, "_jkvsxy"},
      {"0011001110"_b, "_zrpzss"},
      {"0011010000"_b, "_rsqxrs"},
      {"0011010001"_b, "_rktqym"},
      {"001101001x"_b, "_vqrqjt"},
      {"001101100x"_b, "_rtlvxq"},
      {"001101101x"_b, "_gtqnvr"},
      {"0011100xxx"_b, "_yzpszn"},
      {"0011101000"_b, "_hhxpyt"},
      {"0011101001"_b, "_htrtzz"},
      {"0011101010"_b, "_rkxlyj"},
      {"0011101011"_b, "_vnggzq"},
      {"0011110000"_b, "_mrlpxr"},
      {"0011110001"_b, "_xszqrg"},
      {"001111001x"_b, "_plyxlq"},
      {"0011110100"_b, "_rqpjjs"},
      {"0011110101"_b, "_ttmyrv"},
      {"0011110110"_b, "_lvjtlg"},
      {"0011110111"_b, "_lnntps"},
      {"0011111000"_b, "_vtgnnl"},
      {"0011111001"_b, "_mxnzst"},
      {"0011111010"_b, "_lvryvp"},
      {"0011111011"_b, "_mqssgy"},
      {"0011111100"_b, "_pxzvjl"},
      {"0011111101"_b, "_mnxgqm"},
      {"0011111110"_b, "_qntrvk"},
      {"0011111111"_b, "_vnnjxg"},
      {"0100100000"_b, "_yyyshx"},
      {"0100100001"_b, "_mylphg"},
      {"0100100010"_b, "_nsjhhg"},
      {"0100100011"_b, "_rhhrhg"},
      {"0100100100"_b, "_ymhgxg"},
      {"0100100101"_b, "_nvkthr"},
      {"0100100110"_b, "_phthqj"},
      {"0100100111"_b, "_kyjxrr"},
      {"0100101000"_b, "_gzvylr"},
      {"0100101001"_b, "_pppsmg"},
      {"0100101010"_b, "_pyjnpz"},
      {"0100101011"_b, "_shqygv"},
      {"0100101100"_b, "_hzsxkp"},
      {"0100101101"_b, "_nqkhrv"},
      {"0100101110"_b, "_tkjtgp"},
      {"0100101111"_b, "_htqpks"},
      {"0101000xxx"_b, "_ssvpxz"},
      {"0101001xxx"_b, "_vgqvys"},
      {"01010100xx"_b, "_qkrnms"},
      {"01010101xx"_b, "_vypnss"},
      {"01010110xx"_b, "_glkvkr"},
      {"01010111xx"_b, "_qgqgkx"},
      {"0101100xxx"_b, "_mxplnn"},
      {"0101101xxx"_b, "_pqmqrg"},
      {"0101110000"_b, "_gshlgj"},
      {"0101110001"_b, "_klsmsv"},
      {"0101110010"_b, "_xhhqnx"},
      {"0101110011"_b, "_rssrty"},
      {"0101110100"_b, "_nzskzl"},
      {"0101110101"_b, "_qlzvpg"},
      {"0101110110"_b, "_hlxmpy"},
      {"0101110111"_b, "_lplzxv"},
      {"0101111000"_b, "_krtvhr"},
      {"0101111001"_b, "_ymtzjg"},
      {"0101111010"_b, "_szgqrr"},
      {"0101111011"_b, "_xnpyvy"},
      {"0101111100"_b, "_tnngsg"},
      {"0101111101"_b, "_kshtnj"},
      {"0101111110"_b, "_vmxzxt"},
      {"0101111111"_b, "_gxqnph"},
      {"0110001xxx"_b, "_ykptgl"},
      {"0110010xxx"_b, "_slzvjh"},
      {"0110011xxx"_b, "_nqlrmv"},
      {"0110100xxx"_b, "_yrjqql"},
      {"0110101xxx"_b, "_prgrzz"},
      {"01110000xx"_b, "_vshynq"},
      {"0111000100"_b, "_ykjhgg"},
      {"0111000101"_b, "_jqtksx"},
      {"0111000110"_b, "_gzpkvm"},
      {"0111000111"_b, "_jhkkgv"},
      {"0111001xxx"_b, "_yptvyx"},
      {"0111010000"_b, "_tzrgqq"},
      {"0111010001"_b, "_qlpnnn"},
      {"011101001x"_b, "_grsslr"},
      {"01111000xx"_b, "_xjtzgm"},
      {"0111100101"_b, "_srsrtk"},
      {"0111100111"_b, "_xynxhx"},
      {"01111001x0"_b, "_gylmmr"},
      {"0111101xxx"_b, "_mkzysy"},
      {"0111110000"_b, "_nklvmv"},
      {"0111110010"_b, "_pyttkp"},
      {"0111110100"_b, "_lrqlrg"},
      {"0111110101"_b, "_yvxkhv"},
      {"0111110110"_b, "_ksgpqz"},
      {"0111110111"_b, "_hkpjqm"},
      {"0111111000"_b, "_lgzlyq"},
      {"0111111001"_b, "_yrypnt"},
      {"0111111010"_b, "_snvnjz"},
      {"0111111011"_b, "_kkkltp"},
      {"0111111100"_b, "_xsgnlv"},
      {"0111111101"_b, "_lrptrn"},
      {"0111111110"_b, "_pyhrrt"},
      {"0111111111"_b, "_nkyrpv"},
      {"0x10000xxx"_b, "adr_only_pcreladdr"},
      {"1000100000"_b, "_lspzrv"},
      {"1000100001"_b, "_kxvvkq"},
      {"1000100010"_b, "_sxpvym"},
      {"1000100011"_b, "_vkrkks"},
      {"1000100100"_b, "_xvnyxq"},
      {"1000100101"_b, "_gtxpgx"},
      {"1000100110"_b, "_vlrhpy"},
      {"1000100111"_b, "_ymhkrx"},
      {"1000101000"_b, "_zrmgjx"},
      {"1000101001"_b, "_qqyryl"},
      {"1000101010"_b, "_hgxtqy"},
      {"1000101011"_b, "_yytvxh"},
      {"1000101100"_b, "_ptslzg"},
      {"1000101101"_b, "_ytkjxx"},
      {"1000101110"_b, "_zxtzmv"},
      {"1000101111"_b, "_kgmqkh"},
      {"1001000001"_b, "_ptyynt"},
      {"1001000011"_b, "_skszgm"},
      {"1001000100"_b, "_rlgtnn"},
      {"1001000101"_b, "_rgxthl"},
      {"1001000110"_b, "_xxphlt"},
      {"1001000111"_b, "_njjlxy"},
      {"10010100xx"_b, "_tnpjts"},
      {"10010101xx"_b, "_hgjgpm"},
      {"10010110xx"_b, "_hqnsvg"},
      {"10010111xx"_b, "_zqjgzz"},
      {"100111000x"_b, "_ztjjnh"},
      {"1001110010"_b, "_lssjyz"},
      {"1001110011"_b, "_zpxrnm"},
      {"100111010x"_b, "_jkvvtp"},
      {"1001110110"_b, "_sqhxzj"},
      {"1001110111"_b, "_hrxtnj"},
      {"1010001000"_b, "_vrsgzg"},
      {"1010001010"_b, "_vhkjgh"},
      {"10100010x1"_b, "_rxytqg"},
      {"1010001100"_b, "_lrmgmq"},
      {"1010001110"_b, "_zjrsrx"},
      {"10100011x1"_b, "_vmgnhk"},
      {"1010010xxx"_b, "_pjlnhh"},
      {"10100110xx"_b, "_xzlxjh"},
      {"10100111xx"_b, "_rjthsm"},
      {"10101000xx"_b, "_yjnkrn"},
      {"10101001xx"_b, "_zrqtgx"},
      {"1010101000"_b, "_pxvjkp"},
      {"1010101001"_b, "_xrzqtn"},
      {"101010101x"_b, "_ttmvpr"},
      {"1010101100"_b, "_grjzyl"},
      {"1010101110"_b, "_kynxnz"},
      {"10101011x1"_b, "_zzhnxv"},
      {"1010110000"_b, "_lymhlk"},
      {"1010110100"_b, "_tpmqyl"},
      {"1010111000"_b, "_lkzyzv"},
      {"1010111100"_b, "_tvyxlr"},
      {"101011xx10"_b, "_yxvttm"},
      {"101011xxx1"_b, "_qhzvvh"},
      {"1011000xxx"_b, "_jgklkt"},
      {"1011001000"_b, "_sxptnh"},
      {"1011001001"_b, "_vmsxgq"},
      {"1011001010"_b, "_vkrskv"},
      {"1011001011"_b, "_nxrqmg"},
      {"1011001100"_b, "_lsqgkk"},
      {"1011001110"_b, "_kxpqhv"},
      {"10110011x1"_b, "_jrxtzg"},
      {"1011010000"_b, "_yzqhtj"},
      {"1011010001"_b, "_yvqnyq"},
      {"101101001x"_b, "_qpsryx"},
      {"1011011000"_b, "_vpjktn"},
      {"1011011001"_b, "_zvxxjk"},
      {"101101101x"_b, "_sztkhs"},
      {"101101110x"_b, "_hlypvy"},
      {"101101111x"_b, "_rszgzl"},
      {"1011100xxx"_b, "_ymszkr"},
      {"1011101000"_b, "_pzzgts"},
      {"1011101001"_b, "_pgvjgs"},
      {"1011101010"_b, "_kppzvh"},
      {"1011101011"_b, "_nlrjsj"},
      {"10111100xx"_b, "_rxtklv"},
      {"1011110100"_b, "_vsnnms"},
      {"1100100000"_b, "_sjtrhm"},
      {"1100100001"_b, "_hzkglv"},
      {"1100100010"_b, "_qrygny"},
      {"1100100011"_b, "_tjzqnp"},
      {"1100100100"_b, "_yqvqtx"},
      {"1100100101"_b, "_ngttyj"},
      {"1100100110"_b, "_kqzmtr"},
      {"1100100111"_b, "_qpvgnh"},
      {"1100101000"_b, "_tpkslq"},
      {"1100101001"_b, "_kzpyzy"},
      {"1100101010"_b, "_ytvtqn"},
      {"1100101011"_b, "_qkzjxm"},
      {"1100101100"_b, "_lqmksm"},
      {"1100101101"_b, "_hxlznn"},
      {"1100101110"_b, "_knkjnz"},
      {"1100101111"_b, "_rsjgyk"},
      {"1101000xxx"_b, "_mtlxqp"},
      {"1101001xxx"_b, "_vmyztj"},
      {"11010100xx"_b, "_vnzkty"},
      {"11010101xx"_b, "_vnrlrk"},
      {"11010110xx"_b, "_rjmhxr"},
      {"11010111xx"_b, "_rxgkjn"},
      {"1101100xxx"_b, "_jvkxtj"},
      {"1101101xxx"_b, "_srpqmk"},
      {"1110001xxx"_b, "_pkskpp"},
      {"1110010xxx"_b, "_mgspnm"},
      {"1110011xxx"_b, "_snhzxr"},
      {"1110100xxx"_b, "_jxgqqz"},
      {"1110101xxx"_b, "_qnysqv"},
      {"11110000xx"_b, "_rqghyv"},
      {"1111000100"_b, "_zgljvg"},
      {"1111000101"_b, "_yjnmkg"},
      {"1111000110"_b, "_jqhvhn"},
      {"1111000111"_b, "_vnsqhn"},
      {"1111001xxx"_b, "_hvhrsq"},
      {"1111010000"_b, "_gyjphh"},
      {"1111010010"_b, "_mkrgxr"},
      {"11110100x1"_b, "_mtzhrn"},
      {"11111000xx"_b, "_hlljqz"},
      {"11111001xx"_b, "_qzsyvx"},
      {"1111101xxx"_b, "_jqlgts"},
      {"1x10000xxx"_b, "adrp_only_pcreladdr"},
      {"x110110xxx"_b, "_xymnxy"},
      {"x110111xxx"_b, "_htjmmx"},
    },
  },

};
// clang-format on

}  // namespace aarch64
}  // namespace vixl
