# `rq` Builtins

*[back to index](./README.md)*


`rq` includes a number of additional builtins which augment the library of builtins included with OPA ([see here](https://www.openpolicyagent.org/docs/latest/policy-reference/#built-in-functions)). All of these supplemental builtins are prefixed with `rq.` to avoid any possible collision with those in the standard library.

**NOTE**: builtins prefixed with `x.` are considered experimental, and are liable to change behavior, be renamed, or be removed at any time.

Contents:

* [`rq.abs(path)`](#coderqabspathcode)
* [`rq.base(path)`](#coderqbasepathcode)
* [`rq.chdir(dir)`](#coderqchdirdircode)
* [`rq.convert(amount, from, to)`](#coderqconvertamount-from-tocode)
* [`rq.decode(input, spec)`](#coderqdecodeinput-speccode)
* [`rq.dir(path)`](#coderqdirpathcode)
* [`rq.encode(value, spec)`](#coderqencodevalue-speccode)
* [`rq.error(message)`](#coderqerrormessagecode)
* [`rq.ext(path)`](#coderqextpathcode)
* [`rq.fake(kind)`](#coderqfakekindcode)
* [`rq.getwd()`](#coderqgetwdcode)
* [`rq.joinpath(elements)`](#coderqjoinpathelementscode)
* [`rq.parsedate(date)`](#coderqparsedatedatecode)
* [`rq.quote(s)`](#coderqquotescode)
* [`rq.read(spec)`](#coderqreadspeccode)
* [`rq.run(command, options)`](#coderqruncommand-optionscode)
* [`rq.scriptpath()`](#coderqscriptpathcode)
* [`rq.sfake(kind, symbol)`](#coderqsfakekind-symbolcode)
* [`rq.slug(s)`](#coderqslugscode)
* [`rq.splitpath(path)`](#coderqsplitpathpathcode)
* [`rq.strtonum(s)`](#coderqstrtonumscode)
* [`rq.template(s, data)`](#coderqtemplates-datacode)
* [`rq.tree(path, options)`](#coderqtreepath-optionscode)
* [`rq.unquote(s)`](#coderqunquotescode)
* [`rq.version()`](#coderqversioncode)
* [`rq.write(value, spec)`](#coderqwritevalue-speccode)

## `rq.run(command, options)`

`rq.run` allows you to run shell commands from within a Rego expression.

Parameters:
* `command`: either a string (command with no arguments), or a command and list of arguments as an array of strings
* `options`: object with the following keys
	* (optional) `stdin`: string or object to be written to the standard input of the command; if an object is used, then it is formatted according to `stdin_spec`. If this key is omitted, then the input to the command will be the contents specified by the `stdin_spec`'s file path. If the file path is empty, then the input will be empty. If this key is an object and `stdin_spec` is not provided, then JSON will be used as the output handler.
	* (optional) `stdin_spec`: string or object representing a [DataSpec](./dataspec.md) representing an output handler to use to format the input to the command.
	* (optional) `stdout_spec`: As with `stdin_spec`, but specifies an output handler to be applied to the standard output stream of the command. The file path is always ignored.
	* (optional) `stderr_spec`: As with `stdin_spec`, but specifies an output handler to be applied to the standard error stream of the command. The file path is always ignored.
    * (optional) `timeout`: after this many seconds have elapsed, the command is killed if it has not finished already, a value of `0` explicitly indicates no timeout.

Return format:
* `exitcode`: integer representing the exit code of the command
* `stdout`: either the string representing the command's standard output stream, or an object created by an `rq` input handler if `stdout_spec` if it was provided.
* `stderr`: either the string representing the command's standard error stream, or an object created by an `rq` input handler if `stderr_spec` if it was provided.
* `timedout`: `true` if the command timed out; this field is present in the output if and only if the `timeout` option is provided.

Note that if the command times out, then the `exitcode` will be `-1`, in addition to the `timedout` field being `true`.

Examples:

```plain
$ rq 'rq.run("date", {})'
{
	"exitcode": 0,
	"stderr": "",
	"stdout": "Wed Jul 20 11:31:21 PM EDT 2022\n"
}

$ rq 'rq.run(["date"], {})'
{
	"exitcode": 0,
	"stderr": "",
	"stdout": "Wed Jul 20 11:31:27 PM EDT 2022\n"
}

# The command's output can be parsed to a rich object using any format rq
# knows how to parse as an input.
$ rq 'rq.run(["cat", "sample_data/books.csv"], {"stdout_spec": {"format": "csv", "options": {"csv.headers": "true"}}})'
{
	"exitcode": 0,
	"stderr": "",
	"stdout": [
		{
			"first author": "Thorsten Ball",
			"isbn": "978-3982016115",
			"title": "Writing An Interpreter In Go",
			"year": 2018
		},
		{
			"first author": "Thorsten Ball",
			"isbn": "978-3982016108",
			"title": "Writing A Compiler In Go",
			"year": 2018
		},
		...
	]
}

# DataSpecs can be specified as either strings, or as Rego objects (shown in
# the previous example).
$ rq 'rq.run(["cat", "sample_data/books.csv"], {"stdout_spec": "csv:csv.headers=true:"})'
{
	"exitcode": 0,
	"stderr": "",
	"stdout": [
		{
			"first author": "Thorsten Ball",
			"isbn": "978-3982016115",
			"title": "Writing An Interpreter In Go",
			"year": 2018
		},
		...
	]
}

# `stdin` can be a rich object, to be formatted using an output handler.
$ rq 'rq.run(["jq", ".foo"], {"stdin": {"foo": "bar", "baz": "quux"}, "stdout_spec": "json:"})'
{
	"exitcode": 0,
	"stderr": "",
	"stdout": "bar"
}

# Standard input can also be controlled using DataSpecs.
$ rq 'rq.run(["cat"], {"stdin_spec": "sample_data/books.csv", "stdout_spec": "csv:csv.headers=true:"})'
{
	"exitcode": 0,
	"stderr": "",
	"stdout": [
		{
			"first author": "Thorsten Ball",
			"isbn": "978-3982016115",
			"title": "Writing An Interpreter In Go",
			"year": 2018
		},
		...
	]
}

# By combining both 'stdin_spec' and 'stdout_spec', we can easily interface
# with commands that speak JSON (or other formats rq knows about).
$ rq 'rq.run(["jq", "[.[].title]"], {"stdin_spec": "sample_data/books.json", "stdout_spec": "json:"})'
{
	"exitcode": 0,
	"stderr": "",
	"stdout": [
		"Writing An Interpreter In Go",
		"Writing A Compiler In Go",
		"The Go Programming Language",
		"Hands-On GUI Application Development in Go",
		"Building Cross-Platform GUI Applications with Fyne"
	]
}

# Note that if both 'stdin' and 'stdin_spec' are provided, then the file path
# provided in 'stdin_spec' is ignored, if any.
$ rq 'rq.run(["cat"], {"stdin": {"foo": "bar", "baz": "quux"}, "stdin_spec": "sample_data/books.json", "stdout_spec": "json:"})'
{
	"exitcode": 0,
	"stderr": "",
	"stdout": {
		"baz": "quux",
		"foo": "bar"
	}
}

# The file path provided in the 'std{out,err}_spec' field is ignored; `rq.run`
# cannot be used to write to files.
$ cat output.json
cat: output.json: No such file or directory
$ rq 'rq.run(["jq", ".foo"], {"stdin": {"foo": "bar", "baz": "quux"}, "stdout_spec": "json:./output.json"})'
{
	"exitcode": 0,
	"stderr": "",
	"stdout": "bar"
}
$ cat output.json
cat: output.json: No such file or directory

# As of rq 0.0.8, environment variables can be provided.
$ rq 'rq.run(["sh", "-c", "expr $a + $b"], {"env": {"a": 3, "b": 7}})'
{
	"exitcode": 0,
	"stderr": "",
	"stdout": "10\n"
}
$ rq 'rq.run(["sh", "-c", "echo $greeting, rq!"], {"env": {"greeting": "hello"}})'
{
	"exitcode": 0,
	"stderr": "",
	"stdout": "hello, rq!\n"
}

# As of rq 0.0.11, a timeout can be specified
$ ./build/rq 'rq.run(["sh", "-c", "date; sleep 3; date"], {"timeout": 5, "stdout_spec": "lines:", "stderr_spec": "lines:"})'
{
        "exitcode": 0,
        "stderr": [],
        "stdout": [
                "Wed Jan  8 00:30:22 EST 2025",
                "Wed Jan  8 00:30:25 EST 2025"
        ],
        "timedout": false
}
$ ./build/rq 'rq.run(["sh", "-c", "date; sleep 3; date"], {"timeout": 2, "stdout_spec": "lines:", "stderr_spec": "lines:"})'
{
        "exitcode": -1,
        "stderr": [],
        "stdout": [
                "Wed Jan  8 00:30:28 EST 2025"
        ],
        "timedout": true
}
```

## `rq.tree(path, options)`

`rq.tree` allows you to obtain recursive directory trees.

Parameters:
* `path`: a string path to the top directory to generate a tree for.
* `options`: object with the following keys
	* (optional) `maxdepth`: int indicating maximum depth for files to include in the result. Use -1 for unlimited, which is the default.
	* (optional) `hidden`: bool indicating if hidden files/directories should be included in the result. Use `true` if hidden files should be included. `false` is the default.

Return format:
* Object with keys being absolute paths rooted at `path`, each with these keys:
	* `abs`: absolute path to the file
	* `base`: base name of the file (only the final path element)
	* `depth`: int indicating the depth of the file relative to `path`
	* `dir`: directory of the file (all but the final path element)
	* `ext`: file extension (file basename, not including anything after the final `.`)
	* `is_dir`: bool, set to `true` if the file is a directory
	* `mode`: file permissions string
	* `rel`: relative path from `path` to the file
	* `size`: file size in bytes

**NOTE**: Due to implementation details, `rq.tree()` will still recursively traverse all files below the given path, files exceeding the maximum recursion depth are simply hidden from the output. Likewise for hidden files. Ideally, this will change in the future.

Examples:

```
# List all files in the `./cmd` folder, recursively.
$ rq 'rq.tree("./cmd/", {})'
{
	"/home/cad/f/src/rq/cmd/rq": {
		"abs": "/home/cad/f/src/rq/cmd/rq",
		"base": "rq",
		"depth": 1,
		"dir": "/home/cad/f/src/rq/cmd",
		"ext": "",
		"is_dir": true,
		"mode": "drwxr-xr-x",
		"rel": "rq",
		"size": 7
	},
	"/home/cad/f/src/rq/cmd/rq/commands.go": {
		"abs": "/home/cad/f/src/rq/cmd/rq/commands.go",
		"base": "commands.go",
		"depth": 2,
		"dir": "/home/cad/f/src/rq/cmd/rq",
		"ext": "go",
		"is_dir": false,
		"mode": "-rw-r--r--",
		"rel": "rq/commands.go",
		"size": 8587
	},
	"/home/cad/f/src/rq/cmd/rq/helpers.go": {
		"abs": "/home/cad/f/src/rq/cmd/rq/helpers.go",
		"base": "helpers.go",
		"depth": 2,
		"dir": "/home/cad/f/src/rq/cmd/rq",
		"ext": "go",
		"is_dir": false,
		"mode": "-rw-r--r--",
		"rel": "rq/helpers.go",
		"size": 12655
	},
	"/home/cad/f/src/rq/cmd/rq/helpers_test.go": {
		"abs": "/home/cad/f/src/rq/cmd/rq/helpers_test.go",
		"base": "helpers_test.go",
		"depth": 2,
		"dir": "/home/cad/f/src/rq/cmd/rq",
		"ext": "go",
		"is_dir": false,
		"mode": "-rw-r--r--",
		"rel": "rq/helpers_test.go",
		"size": 1270
	},
	"/home/cad/f/src/rq/cmd/rq/main.go": {
		"abs": "/home/cad/f/src/rq/cmd/rq/main.go",
		"base": "main.go",
		"depth": 2,
		"dir": "/home/cad/f/src/rq/cmd/rq",
		"ext": "go",
		"is_dir": false,
		"mode": "-rw-r--r--",
		"rel": "rq/main.go",
		"size": 5435
	},
	"/home/cad/f/src/rq/cmd/rq/main_test.go": {
		"abs": "/home/cad/f/src/rq/cmd/rq/main_test.go",
		"base": "main_test.go",
		"depth": 2,
		"dir": "/home/cad/f/src/rq/cmd/rq",
		"ext": "go",
		"is_dir": false,
		"mode": "-rw-r--r--",
		"rel": "rq/main_test.go",
		"size": 8804
	}
}


# As before, but limiting the maximum recursion depth to 1.
$ rq 'rq.tree("./cmd", {"maxdepth": 1})'
{
	"/home/cad/f/src/rq/cmd/rq": {
		"abs": "/home/cad/f/src/rq/cmd/rq",
		"base": "rq",
		"depth": 1,
		"dir": "/home/cad/f/src/rq/cmd",
		"ext": "",
		"is_dir": true,
		"mode": "drwxr-xr-x",
		"rel": "rq",
		"size": 7
	}
}

# Include hidden files in the result.
$ rq '{f | f := rq.tree("./", {"maxdepth": 1, "hidden": true})[_]; f.base == ".git"}'
[
	{
		"abs": "/home/cad/f/src/rq/.git",
		"base": ".git",
		"depth": 1,
		"dir": "/home/cad/f/src/rq",
		"ext": "",
		"is_dir": true,
		"mode": "drwxr-xr-x",
		"rel": ".git",
		"size": 15
	}
]
```

# `rq.env()`

`rq.env()` allows you to access environment variables.

Parameters: (none)

Return format:
* Object with string keys corresponding to environment variable names, and string values corresponding to the environment variable's values.

Examples:

```plain
$ rq 'object.filter(rq.env(), ["HOME", "PWD", "USER"])'
{
	"HOME": "/home/cad",
	"PWD": "/home/cad/f/src/rq",
	"USER": "cad"
}
```

# `rq.args()`

`rq.args()` allows you to access command-line arguments from within rq scripts.

**NOTE**: in any mode other than script mode, `rq.args()` returns an empty
array.

Parameters: (none)

Return format:
* Array of string arguments given to the rq script.

Examples:

```plain
# rq.args() as called from a script
$ cat argsdemo.rego
#!/usr/bin/env rq

package argsdemo

args := rq.args()
$ ./argsdemo.rego hello, rq arguments
{
	"argsdemo": {
		"args": [
			"hello,",
			"rq",
			"arguments"
		]
	}
}

# rq.args() as called from a query
$ rq 'rq.args()'
[]
```

## `rq.error(message)`

`rq.error` allows you to force `rq` to exit with an error based on your Rego script.

**WARNING:** With the exception of statements inside of rule bodies, it is generally not possible to grantee the order of execution for Rego statements. `rq.error()` is generally not an appropriate mechanism to prevent `rq` from executing statements with side-effects in the event of an error condition. For example, if you need to run one shell command with `rq.run()` to get some data, and want to run another shell command (with side effects) that depends on the result of the first, `rq.error()` **is** a suitable mechanism to report the error to the end user, but **is not** a suitable mechansim to prevent the second command from being run. **TL;DR: only use this builtin for error reporting, do not rely on it to have side effects.**

Parameters:
* `message`: string which should be shown to the user

Examples:

```plain
$ rq 'rq.error("something went wrong :(")'
something went wrong :(
$ echo $?
1
```

## `rq.encode(value, spec)`

`rq.encode` allows you to encode a Rego value to a string using an `rq` output handler.

Parameters:
* `value`: a Rego value to be encoded to a string.
* `spec`: string or object representing a [DataSpec](./dataspec.md) specifying the format and options to use. The file path component is ignored, and an fatal error will be generated if it is non-empty.

Return format:
* String containing the encoded value.

Examples:

```plain
$ rq 'rq.encode([{"foo": "bar"}, {"foo": "baz"}], "json:")'
"[{\"foo\":\"bar\"},{\"foo\":\"baz\"}]"
$ rq 'rq.encode([{"foo": "bar"}, {"foo": "baz"}], "csv:")'
"foo\nbar\nbaz\n"
$ rq 'rq.encode([{"foo": "bar"}, {"foo": "baz"}], "md-table:")'
"| foo |\n|-----|\n| bar |\n| baz |\n"
$ rq 'rq.encode([{"foo": "bar"}, {"foo": "baz"}], {"format": "csv"})'
"foo\nbar\nbaz\n"
```

## `rq.decode(input, spec)`

`rq.decode` allows you to decode a string to a Rego value using an `rq` input handler.

Parameters:
* `input`: a string containing the value to be decoded.
* `spec`: string or object representing a [DataSpec](./dataspec.md) specifying the format and options to use. The file path component is ignored, and an fatal error will be generated if it is non-empty.

Return format:
* Rego value representing the decoded value.

Examples:

```plain
$ rq 'rq.decode("[{\"foo\":\"bar\"},{\"foo\":\"baz\"}]", "json:")'
[
	{
		"foo": "bar"
	},
	{
		"foo": "baz"
	}
]
$ rq 'rq.decode("foo\nbar\nbaz\n", "csv:")'
[
	[
		"foo"
	],
	[
		"bar"
	],
	[
		"baz"
	]
]
$ rq 'rq.decode("foo\nbar\nbaz\n", "csv:csv.headers=true:")'
[
	{
		"foo": "bar"
	},
	{
		"foo": "baz"
	}
]
$ rq 'rq.decode("foo\nbar\nbaz\n", {"format": "csv", "options": {"csv.headers": true}})'
[
	{
		"foo": "bar"
	},
	{
		"foo": "baz"
	}
]
```

## `rq.write(value, spec)`

`rq.write()` allows you to write to a file on disk using an `rq` output handler.

Parameters:
* `value`: a Rego value to encode and write to disk
* `spec`: string or object representing a [DataSpec](./dataspec.md) specifying the format and options to use, and the path to write the resulting file to.

Return format:
* (none)

Examples:

```plain
$ ls
$ rq 'rq.write({"foo": "bar"}, "yaml:./out.yaml")'
null
$ ls
out.yaml
$ cat out.yaml
foo: bar
$ rq 'rq.write({"foo": "bar"}, {"format": "csv", "file_path": "./out.csv", "options": {"csv.headers": true}})'
null
$ ls
out.csv  out.yaml
$ cat out.csv
foo
bar
```

## `rq.read(spec)`

`rq.read()` allows you to read a file on disk using an `rq` input handler.

Parameters

* `spec`: string or object representing a [DataSpec](./dataspec.md) specifying the format and options to use, and the path from which to read the file.

Return format:
* `result`: a Rego object representing the data read from the file.

Examples:

```plain
$ ls
out.csv  out.yaml
$ cat out.csv
foo
bar
$ cat out.yaml
foo: bar
$ rq 'rq.read("csv:csv.headers=true:./out.csv")'
[
        {
                "foo": "bar"
        }
]
$ rq 'rq.read({"format": "yaml", "file_path": "./out.yaml"})'
{
        "foo": "bar"
}
```

## `rq.parsedate(date)`

`rq.parsedate()` allows you to parse a date string without knowing the format
in advance. Internally, `rq.parsedate()` uses
[`araddon/dateparse`](https://github.com/araddon/dateparse) to automatically
infer the proper format and then parse the string appropriately. If the date
cannot be parsed, then this builtin evaluates to undefined.

Parameters:
* `date`: string containing the date to be parsed

Return format:
* Number of ns representing the date, this return will be suitable for use with other ns-based date builtins in the Rego standard library

Examples:

```plain
$ rq 'rq.parsedate("Fri Jul 03 2015 18:04:07 GMT+0100 (GMT Daylight Time)")'
1435943047000000000
$ rq 'rq.parsedate("2022-12-04")'
1670112000000000000
$ date --iso
2022-12-04
$ rq 'rq.parsedate(rq.run(["date", "--iso"], {}).stdout)'
1669853040000000000
```

## `rq.version()`

`rq.version()` allows you to access version information about `rq`; it exposes the exact same information as the `rq version` subcommand. Information in the version info object is determined using the [`runtime`](https://pkg.go.dev/runtime) package as well as [`debug.ReadBuildInfo()`](https://pkg.go.dev/runtime/debug#ReadBuildInfo).

Parameters: (none)

Return format:
* An object with the following fields:
	* `architecture`: CPU architecture on which the `rq` binary is running
	* `go_version`: Go version with which the `rq` binary was built
	* `modified`: `true` if the running `rq` binary included un-committed changes, otherwise `false`
	* `opa`: OPA version which the `rq` binary was built against
	* `os`: operating system on which the `rq` binary is running
	* `revision`: git commit SHA from which the `rq` binary was built
	* `timestamp`: timestamp of the git commit from which the `rq` binary was built
	* `version`: canonical `rq` version string

Examples:

```plain
$ rq version
architecture: amd64
go_version: go1.18.1
modified: true
opa: 0.48.0
os: linux
revision: 60c6654c9130b10235d1e02c2cad26b7b0a7b652
timestamp: "2023-01-12T03:24:09Z"
version: 0.0.5-RC1 (dev)
```

## `rq.convert(amount, from, to)`

`rq.convert()` can be used to perform unit conversion. The canonical list of available units from the upstream unit conversion library can be found [here](https://godocs.io/github.com/bcicen/go-units#pkg-variables).

Parameters:
* `amount`: number denoting quantity in the base units
* `from`: string indicating the base units
* `to`: string indicating the target units

Return format:
* Number corresponding to `amount` converted from the base units to the target units

Examples:

```plain
$ rq 'rq.convert(1, "feet", "inches")'
12.000000000000002
$ rq 'rq.convert(1, "foot", "inches")'
12.000000000000002
$ rq 'rq.convert(1, "mile", "kilometers")'
1.609344
$ rq 'rq.convert(1, "mile", "km")'
1.609344
$ rq 'rq.convert(1000000, "bytes", "megabytes")'
1
```

A list of available units can be found in [`doc/data/units.json`](./data/units.json). It was produced 2023-03-21 against [github.com/bcicen/go-units v1.0.4](https://github.com/bcicen/go-units/tree/v1.0.4); you may find it to be out of date depending on how far in the future you are reading this. The `units.json` file was produced using the following code snippet:

```go
x := map[string]interface{}{}
for _, u := range units.All() {
	y := map[string]interface{}{}
	y["name"] = u.Name
	if u.Symbol != "" {
		y["symbol"] = u.Symbol
	}
	if u.System() != "" {
		y["system"] = u.System()
	}
	y["names"] = u.Names()
	if u.Quantity != "" {
		y["quantity"] = u.Quantity
	}
	if u.PluralName() != "" {
		y["plural"] = u.PluralName()
	}
	x[u.Name] = y
}
j, err := json.MarshalIndent(x, "", "\t")
if err != nil {
	panic(err)
}
fmt.Printf("%s\n", string(j))
```

## `rq.abs(path)`

`rq.abs()` allows you to determine the absolute version of a given file path. This builtin is a wrapper for [`filepath.Abs()`](https://pkg.go.dev/path/filepath#Abs).

Parameters:
* `path`: string filepath to take the absolute path of

Return format:
* string containing the absolute version of the path

Examples:

```plain
$ pwd
/home/cad/f/src/rq
$ rq 'rq.abs("./README.md")'
"/home/cad/f/src/rq/README.md"
```

## `rq.base(path)`

`rq.base()` returns the last element of the given path. This builtin is a wrapper for [`filepath.Base()`](https://pkg.go.dev/path/filepath#Base).

Parameters:
* `path`: string path to find the base of

Return format:
* string containing the last element of the given path

Examples:

```plain
$ rq 'rq.base("/home/cad/f/src/rq/README.md")'
"README.md"
```

## `rq.ext(path)`

`rq.ext()` returns the extension of the given path. This builtin is a wrapper for [`filepath.Ext()`](https://pkg.go.dev/path/filepath#Ext).

Parameters:
* `path`: string path to find the extension of

Return format:
* string containing the extension for the path, including the `.`

Examples:

```plain
$ rq 'rq.ext("/home/cad/f/src/rq/README.md")'
".md"
```

## `rq.dir(path)`

`rq.ext()` returns all but the last element of the given path. This builtin is a wrapper for [`filepath.Dir()`](https://pkg.go.dev/path/filepath#Dir).

Parameters:
* `path`: string path to find the extension of

Return format:
* string containing the path, sans the last element.

Examples:

```plain
$ rq 'rq.dir("/home/cad/f/src/rq/README.md")'
"/home/cad/f/src/rq"
```

## `rq.splitpath(path)`

`rq.splitpath()` splits the given path into individual components. This builtin is a wrapper for [`filepath.Split()`](https://pkg.go.dev/path/filepath#Split), using iteration to separate out each individual path element.

Parameters:
* `path`: string path to split

Return format:
* list of strings containing the path components

Examples:

```plain
$ rq 'rq.splitpath("/home/cad/f/src/rq/README.md")'
[
        "home",
        "cad",
        "f",
        "src",
        "rq",
        "README.md"
]
```

## `rq.joinpath(elements)`

`rq.joinpath()` joins a list of path elements into a single path. This builtin is a wrapper for [`filepath.Join()`](https://pkg.go.dev/path/filepath#Join).

Parameters:
* `elements`: array of strings to joint into a path

Return format:
* string containing the joined path

Examples:

```plain
$ rq 'rq.joinpath(["home","cad","f","src","rq","README.md"])'
"home/cad/f/src/rq/README.md"
```

## `rq.getwd()`

`rq.getwd()` retrieves the current working directory of `rq`. This builtin is a wrapper for [`os.Getwd()`](https://pkg.go.dev/os#Getwd)

Parameters: (none)

Return format:
* string containing the working directory

Examples:

```plain
$ rq 'rq.getwd()'
"/home/cad/f/src/rq"
```

## `rq.chdir(dir)`

`rq.chdir()` changes the current working directory of `rq`. This builtin is a wrapper for [`os.Chdir()`](https://pkg.go.dev/os#Chdir)

Parameters:
* `dir`: string containing the path to change directories to

Return format:
* the boolean value `true`

Examples:

```plain
$ rq 'rq.getwd(); rq.chdir("/etc"); rq.getwd()'
[
        "/home/cad/f/src/rq",
        true,
        "/etc"
]
```

## `rq.scriptpath()`

`rq.scriptpath()` retrieves the path to the file `rq script` was called on. If used in query mode, it returns the empty string.

Parameters: (none)

Return format:
* string containing the absolute path to the script file

Examples:

```plain
$ pwd
/home/cad/f/src/rq
$ rq 'rq.scriptpath()'
""
$ cat smoketest/rq.scriptpath/testscript
# rq: query data.script.out
# rq: output-format raw

out := rq.scriptpath()
$ rq script ./smoketest/rq.scriptpath/testscript
/home/cad/f/src/rq/smoketest/rq.scriptpath/testscript
```

## `rq.template(s, data)`

`rq.template` allows you to template strings using the Go [template](https://pkg.go.dev/text/template) package. In addition to exposing the data provided via the `data` argument, this builtin also exposes several string manipulation functions, mostly mirroring the ones in OPA.

Included Functions:
* `{{lower s}}` - convert a string to lower case, similar to [`lower()`](https://www.openpolicyagent.org/docs/latest/policy-reference/#builtin-strings-lower)
* `{{upper s}}` - convert a string to upper case, similar to [`upper()`](https://www.openpolicyagent.org/docs/latest/policy-reference/#builtin-strings-upper)
* `{{replace s old new}}` - replace all instance of a non-overlapping string, similar to [`replace()`](https://www.openpolicyagent.org/docs/latest/policy-reference/#builtin-strings-replace)
* `{{reverse s}}` - reverse a string, similar to [`strings.reverse()`](https://www.openpolicyagent.org/docs/latest/policy-reference/#builtin-strings-stringsreverse)
* `{{trim s cutset}}` - remove all leading and trailing characters from the cutset, similar to [`trim()`](https://www.openpolicyagent.org/docs/latest/policy-reference/#builtin-strings-trim)
* `{{trim_left s cutset}}` - remove all leading characters from the cutset, similar to [`trim_left()`](https://www.openpolicyagent.org/docs/latest/policy-reference/#builtin-strings-trim_left)
* `{{trim_right s cutset}}` - remove all trailing characters from the cutset, similar to [`trim_right()`](https://www.openpolicyagent.org/docs/latest/policy-reference/#builtin-strings-trim_right)
* `{{trim_suffix s suffix}}` remove the suffix from the string if it is present, similar to [`trim_suffix()`](https://www.openpolicyagent.org/docs/latest/policy-reference/#builtin-strings-trim_suffix)
* `{{trim_prefix s prefix}}` - remove the prefix from the string if it is present, similar to [`trim_prefix()`](https://www.openpolicyagent.org/docs/latest/policy-reference/#builtin-strings-trim_prefix)
* `{{repeat s count}}` - repeat the string `s` `count` many times, wrapper around [`strings.Repeat()`](https://pkg.go.dev/strings#Repeat)
* `{{sprintf fmt arg1 arg2 ... argn}}` - format the given arguments using the provided format string, wrapper around [`fmt.Sprintf()`](https://pkg.go.dev/fmt#Sprintf)
* `{{fake kind}}` - generate fake data, equivalent to [`rq.fake()`](https://git.sr.ht/~charles/rq/tree/master/item/doc/builtins.md#coderqfakekindcode)
* `{{sfake kind symbol}}` - generate fake data, equivalent to [`rq.sfake()`](https://git.sr.ht/~charles/rq/tree/master/item/doc/builtins.md#coderqsfakekind-symbolcode)
* `{{resub s pattern replacement}}` - replace all matches to the regex `pattern` with the string `replacement`, similar to [`regex.replace()`](https://www.openpolicyagent.org/docs/latest/policy-reference/#builtin-regex-regexreplace)
* `{{quote s}}` - wrap the string `s` in double quotes, escaping Go escape sequences; wrapper around [`strconv.Quote()`](https://pkg.go.dev/strconv#Quote)
* `{{unquote s}}` - unquote the quoted string `s`, wrapper around [`strconv.Unquote()`](https://pkg.go.dev/strconv#Unquote)
* `{{str v}}` - conver the value to a string with reasonable default (use `sprintf` for finer control)

Parameters:
* `s`: the string to template, you an access fields in the data via `{{.fieldname}}`.
* `data`: object containing the template data.

Return format:
* String with the result of applying the template to the string.

Examples:

```
$ rq 'rq.template("hello {{.whom}}", {"whom": "world"})'
"hello world"
$ cat sample_data/books.json
[
    {
        "title": "Writing An Interpreter In Go",
        "authors": ["Thorsten Ball"],
        "isbn": "978-3982016115",
        "year": 2018
    },
    {
        "title": "Writing A Compiler In Go",
        "authors": ["Thorsten Ball"],
        "isbn": "978-3982016108",
        "year": 2018
    },
    {
        "title": "The Go Programming Language",
        "authors": ["Alan A. A. Donovan", "Brian W. Kernighan"],
        "isbn": "978-0134190440",
        "year": 2015
    },
    {
        "title": "Hands-On GUI Application Development in Go",
        "authors": ["Andrew Williams"],
        "isbn": "978-1789138412",
        "year": 2019
    },
    {
        "title": "Building Cross-Platform GUI Applications with Fyne",
        "authors": ["Andrew Williams"],
        "isbn": "1800563167",
        "year": 2021
    }
]
$ rq -I ./sample_data/books.json '{rq.template("{{.title}} has {{.nauthors}} author(s)", d) | b := input[_]; d := object.union(b, {"nauthors": count(b.authors)})}'
[
        "Building Cross-Platform GUI Applications with Fyne has 1 author(s)",
        "Hands-On GUI Application Development in Go has 1 author(s)",
        "The Go Programming Language has 2 author(s)",
        "Writing A Compiler In Go has 1 author(s)",
        "Writing An Interpreter In Go has 1 author(s)"
]
```

## `rq.fake(kind)`

`rq.fake` generates fake data for testing or example purposes. It is a thin wrapper around [jaswdr/faker](https://github.com/jaswdr/faker).

Parameters:
* `kind`: string indicating what type of fake data to generate (to see available values, read the source code of the `FakeOptions.Fake()` function in [`util/fake.go`](https://git.sr.ht/~charles/rq/tree/master/item/util/fake.go).

Return format:
* generated fake value

Examples:

```plain
$ rq 'rq.fake("name")'
"Tina Greenholt"
$ rq 'rq.fake("address")'
"0796 Orn Estate\nSouth Josianneside, VT 61645"
$ rq 'rq.fake("email")'
"hassan.vandervort@hotmail.com"
$ rq 'rq.fake("phone")'
"1-380-270-0406 x55921"
$ rq 'rq.fake("file.absolute_file_path_for_windows")'
"u:\\non\\repellat\\consectetur.sid"
$ rq 'rq.fake("time.rfc3339")'
"1458-05-04T19:03:58-04:56"
```

## `rq.sfake(kind, symbol)`

`rq.sfake` works similarly to `rq.fake()`, except that it also accepts a `symbol` value. When a fake value is generated, it is stored, and subsequent calls with the same `kind` and `symbol` will return that value rather than generating a new one. This can be useful for maintaining referential integrity within generated test data.

**NOTE**: The table of stored fake data used by this function cannot shrink. Thus `rq` will leak memory each time this function is used. In situations where `rq` is expected to be long-running, this function should be used with care.

Parameters:
* `kind`: string indicating what type of fake data to generate (to see available values, read the source code of the `FakeOptions.Fake()` function in [`util/fake.go`](https://git.sr.ht/~charles/rq/tree/master/item/util/fake.go).
* `symbol`: string indicating a unique instance of this kind of fake data.

Return format:
* generated fake value


Examples:

*Observe that the user IDs used throughout the test data are consistent within
each run of the script, even though the generated values differ.*

```plain
$ cat sfake_example.rego
# rq: query data.script.out

out := {
	"user_ids": [
		rq.sfake("user", "u1"),
		rq.sfake("user", "u2"),
		rq.sfake("user", "u3"),
	],
	"profiles": {
		rq.sfake("user", "u1"): {
			"name": rq.sfake("name", "n1"),
			"email": rq.sfake("email", "e1"),
		},
		rq.sfake("user", "u2"): {
			"name": rq.sfake("name", "n2"),
			"email": rq.sfake("email", "e2"),
		},
		rq.sfake("user", "u3"): {
			"name": rq.sfake("name", "n3"),
			"email": rq.sfake("email", "e3"),
		}
	},
	"groups": {
		rq.sfake("user", "u1"): {"admin"} ,
		rq.sfake("user", "u2"): {"op"},
		rq.sfake("user", "u3"): set(),
	}
}
$ rq script sfake_example.rego
{
	"groups": {
		"kellen.rippin": [
			"op"
		],
		"nicolas.addie": [
			"admin"
		],
		"satterfield": []
	},
	"profiles": {
		"kellen.rippin": {
			"email": "micaela.cassin@example.org",
			"name": "Lue Kiehn"
		},
		"nicolas.addie": {
			"email": "brown.kamryn@example.org",
			"name": "Hillary Lemke MD"
		},
		"satterfield": {
			"email": "callie.nolan@example.org",
			"name": "Mireille Jacobs"
		}
	},
	"user_ids": [
		"nicolas.addie",
		"kellen.rippin",
		"satterfield"
	]
}
$ rq script sfake_example.rego
{
	"groups": {
		"dagmar": [
			"admin"
		],
		"esteban.koelpin": [
			"op"
		],
		"spencer.maryjane": []
	},
	"profiles": {
		"dagmar": {
			"email": "wava@example.org",
			"name": "Ms. Vivianne Welch Sr."
		},
		"esteban.koelpin": {
			"email": "klocko.lily@example.org",
			"name": "Logan Breitenberg"
		},
		"spencer.maryjane": {
			"email": "bernita.steuber@example.org",
			"name": "Alberta Runolfsdottir"
		}
	},
	"user_ids": [
		"dagmar",
		"esteban.koelpin",
		"spencer.maryjane"
	]
}
```

## `rq.quote(s)`

`rq.quote()` wraps the string `s` in double quotes, escaping Go escape sequences. It is a wrapper around [`strconv.Quote()`](https://pkg.go.dev/strconv#Quote).

Parameters:
* `s`: the string to quote.

Return format:
* Quoted version of the string `s`.

Examples:

```plain
$ rq --raw 'rq.quote("hello, rq!")'
"hello, rq!"
```

## `rq.unquote(s)`

`rq.unquote()` unquotes the string `s`. It is a wrapper around [`strconv.Unquote()`](https://pkg.go.dev/strconv#Unquote). Note that unlike `strconv.Unquote()`, `rq.unquote()` returns `s` if an error occurs while unquoting.

Parameters:
* `s`: the string to unquote.

Return format:
* Unquoted version of the string `s`.

Examples:

```plain
$ rq --raw 'rq.unquote(`"hello, rq!"`)'
hello, rq!
$ rq --raw 'rq.unquote(`"hello, rq!`)'
"hello, rq!
```

## `rq.slug(s)`

`rq.slug()` creates a URL-friendly slug from the string `s`. It is a wrapper around [mozillazg/go-slugify](https://github.com/mozillazg/go-slugify).

Parameters:
* `s`: the string to slug-ify

Return format:
* URL fiendly version of the string `s`.

Examples:

```plain
$ rq --raw 'rq.slug("Hello, World!")'
hello-world
$ rq --raw 'rq.slug("北京kožušček,abc")'
bei-jing-kozuscek-abc
```

## `rq.strtonum(s)`

**WARNING**: This builtin is an escape hatch for advanced use cases. If you just want to parse a number to a string, you almost certainly want the Rego builtin [`to_number()`](https://www.openpolicyagent.org/docs/latest/policy-reference/#builtin-conversions-to_number). Do not use this builtin if you do not understand arbitrary precision numeric encoding in JSON.

`rq.strtonum()` directly instantiates a Rego `types.Number` value from the given string using `ast.NumberTerm`. This allows for rq programs to dynamically create arbitrary precision numeric values. Note that while the Rego runtime will try preserve the "stringiness" of `json.Number` values, performing math on the resulting value will coerce it to a native Go type such as `float64` or [`big.Float`](https://pkg.go.dev/math/big#Float).

**NOTE**: Performing math on `json.Number` values which are too large to fit in a `big.Float` will cause the Rego runtime to panic, and `rq` to crash.

Parameters:
* `s`: the string to construct the number from

Return format:
* Rego numeric value constructed from the given string

Examples:

```plain
$ rq 'rq.strtonum("1.7976931348623157e+308")'
1.7976931348623157e+308
$ # note that 1.7976931348623157e+308 is math.MaxFloat64
$ rq 'rq.strtonum("1.7976931348623157e+308") * 12345'
2219252174987528731700000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
$ # to_number() coerces to FP64 and cannot represent this value
$ rq 'rq.strtonum("2219252174987528731700000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000") * 1'
2219252174987528731700000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000
# the Rego runtime returns undefined for this expression
$ rq 'to_number("2219252174987528731700000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000000") * 1'
[]
```
