package io

// Some code in this file is taken from
// https://github.com/open-policy-agent/conftest/blob/master/parser/ini/ini.go
// and modified. The original code is Copyright (C) 2019 Gareth Rushgrove and
// is made available via an Apache 2 license which can be retrieved here:
// https://www.apache.org/licenses/LICENSE-2.0
//
// Subsequent modifications are made available under the rq project license,
// which may be viewed in the top-level LICENSE file.

import (
	"encoding/json"
	"fmt"
	"io"

	"git.sr.ht/~charles/rq/util"

	"github.com/go-ini/ini"
)

func init() {
	registerInputHandler("ini", func() InputHandler { return &INIInputHandler{} })
}

func unmarshalINI(p []byte, v interface{}) error {
	cfg, err := ini.Load(p)
	if err != nil {
		return fmt.Errorf("read ini file: %w", err)
	}

	result := make(map[string]map[string]interface{})
	for _, s := range cfg.Sections() {
		sectionName := s.Name()
		if sectionName == "DEFAULT" {
			continue
		}

		result[sectionName] = map[string]interface{}{}
		keysHash := s.KeysHash()
		result[sectionName] = convertINIKeyTypes(keysHash)
	}

	j, err := json.Marshal(result)
	if err != nil {
		return fmt.Errorf("marshal ini to json: %w", err)
	}

	if err := json.Unmarshal(j, v); err != nil {
		return fmt.Errorf("unmarshal ini json: %w", err)
	}

	return nil
}

func convertINIKeyTypes(keysHash map[string]string) map[string]interface{} {
	val := map[string]interface{}{}

	for k, v := range keysHash {
		val[k] = util.StringToValue(v)
	}

	return val
}

// Declare conformance with InputHandler interface.
var _ InputHandler = &INIInputHandler{}

// INIInputHandler handles parsing INI data.
type INIInputHandler struct{}

// Name implements InputHandler.Name().
func (i *INIInputHandler) Name() string {
	return "ini"
}

// Parse implements InputHandler.Parse().
func (i *INIInputHandler) Parse(reader io.Reader) (interface{}, error) {
	inputBytes, err := io.ReadAll(reader)
	if err != nil {
		return nil, err
	}

	var parsed interface{}
	err = unmarshalINI(inputBytes, &parsed)
	if err != nil {
		return nil, err
	}

	return parsed, nil
}

// SetOption implements InputHandler.SetOption().
func (i *INIInputHandler) SetOption(name string, value string) error {
	// The INI handler does not have any options.
	return nil
}
