package util

import (
	"math/rand"
	"testing"
	"time"

	"github.com/stretchr/testify/assert"

	"github.com/jaswdr/faker"
)

func TestFake(t *testing.T) {

	// XXX: commented out test cases seem to be flaky due to (presumably)
	// not respecting the random seed as of 2023-08-22.

	cases := []struct {
		kind        string
		expectError bool
		expect      any
	}{
		{
			kind:        "must_error",
			expectError: true,
		},

		{
			kind:   "address",
			expect: "97 Stanton Parkway Suite 410\nLake Enolastad, UT 38358",
		},
		{
			kind:   "address.building_number",
			expect: "09",
		},
		{
			kind:   "city",
			expect: "Denesikland",
		},
		{
			kind:   "address.city",
			expect: "Denesikland",
		},
		{
			kind:   "address.city_prefix",
			expect: "West",
		},
		{
			kind:   "address.city_suffix",
			expect: "ville",
		},
		{
			kind:   "address.country",
			expect: "Bermuda",
		},
		{
			kind:   "address.country_abbr",
			expect: "SPM",
		},
		{
			kind:   "address.country_code",
			expect: "SM",
		},
		{
			kind:   "address.latitude",
			expect: 10.971788,
		},
		{
			kind:   "address.longitude",
			expect: 10.971788,
		},
		{
			kind:   "address.post_code",
			expect: "09717-8844",
		},
		{
			kind:   "address.secondary_address",
			expect: "Suite 097",
		},
		{
			kind:   "address.state",
			expect: "Maryland",
		},
		{
			kind:   "address.state_abbr",
			expect: "MD",
		},
		{
			kind:   "address.street_address",
			expect: "97 Stanton Parkway Suite 410",
		},
		{
			kind:   "address.street_name",
			expect: "Muller Prairie",
		},
		{
			kind:   "address.street_suffix",
			expect: "Gardens",
		},
		{
			kind:   "app",
			expect: "Strong Appetite",
		},
		{
			kind:   "app.name",
			expect: "Strong Appetite",
		},
		{
			kind:   "app.version",
			expect: "v1.0.9",
		},
		{
			kind:   "beer",
			expect: "Boulevard Tank 7",
		},
		{
			kind:   "beer.alcohol",
			expect: "5.1%",
		},
		{
			kind:   "beer.blg",
			expect: "7.1°Blg",
		},
		{
			kind:   "beer.hop",
			expect: "Rakau Hops",
		},
		{
			kind:   "beer.ibu",
			expect: "40 IBU",
		},
		{
			kind:   "beer.malt",
			expect: "Crystal",
		},
		{
			kind:   "beer.name",
			expect: "Boulevard Tank 7",
		},
		{
			kind:   "beer.style",
			expect: "Barley wine",
		},
		{
			kind:   "binary_string",
			expect: "10011100111100101100001010111001",
		},
		{
			kind:   "binary_string.binary_string",
			expect: "10011100111100101100001010111001",
		},
		{
			kind:   "blood",
			expect: "B-",
		},
		{
			kind:   "blood.name",
			expect: "B-",
		},
		{
			kind:   "bool",
			expect: true,
		},
		{
			kind:   "boolean",
			expect: true,
		},
		{
			kind:   "boolean.bool",
			expect: true,
		},
		{
			kind:   "boolean.bool_int",
			expect: 1,
		},
		{
			kind:   "car",
			expect: "Ranger SuperCrew",
		},
		{
			kind:   "car.category",
			expect: "Wagon",
		},
		{
			kind:   "car.fuel_type",
			expect: "Petrol",
		},
		{
			kind:   "car.maker",
			expect: "Jaguar",
		},
		{
			kind:   "car.model",
			expect: "Ranger SuperCrew",
		},
		{
			kind:   "car.plate",
			expect: "СE4668АM",
		},
		{
			kind:   "car.transmission_gear",
			expect: "Tiptronic",
		},
		{
			kind:   "color",
			expect: "NavajoWhite",
		},
		{
			kind:   "color.css",
			expect: "rgb(75,132,62)",
		},
		{
			kind:   "color.color_name",
			expect: "NavajoWhite",
		},
		{
			kind:   "color.hex",
			expect: "#B4EF15",
		},
		{
			kind:   "color.rgb",
			expect: "75,132,62",
		},
		{
			kind:   "color.rgb_as_array",
			expect: "[75,132,62]",
		},
		{
			kind:   "color.safe_color_name",
			expect: "green",
		},
		{
			kind:   "company",
			expect: "Muller, Muller and Muller",
		},
		{
			kind:   "company.bs",
			expect: "extend sexy metrics",
		},
		{
			kind:   "company.catch_phrase",
			expect: "Synergized web-enabled budgetarymanagement",
		},
		{
			kind:   "company.ein",
			expect: 46,
		},
		{
			kind:   "company.job_title",
			expect: "Motion Picture Projectionist",
		},
		{
			kind:   "company.name",
			expect: "Muller, Muller and Muller",
		},
		{
			kind:   "company.suffix",
			expect: "Ltd",
		},
		{
			kind:   "crypto.bech32_address",
			expect: "bc1kT6zT8vnuSVZdL2Ek77pka29Kbw17K",
		},
		{
			kind:   "crypto.bitcoin_address",
			expect: "13TePnU2hk44aXgh9MEjMqK98F286g1dWg",
		},
		{
			kind:   "crypto.etherium_address",
			expect: "0xu3TePnU2hk44aXgh9MEjMqK98F286g1dWgWq0Hdn",
		},
		{
			kind:   "crypto.p2pkh_address",
			expect: "1kT6zT8vnuSVZdL2Ek77pka29Kbw17K65",
		},
		{
			kind:   "crypto.p2sh_address",
			expect: "3kT6zT8vnuSVZdL2Ek77pka29Kbw17K65",
		},
		{
			kind:   "money",
			expect: "760 ILS",
		},
		{
			kind:   "currency",
			expect: "Syrian Pound SYP",
		},
		{
			kind:   "currency.code",
			expect: "SYP",
		},
		{
			kind:   "currency.country",
			expect: "SYRIAN ARAB REPUBLIC",
		},
		{
			kind:   "currency.currency",
			expect: "Syrian Pound",
		},
		{
			kind:   "currency.currency_and_code",
			expect: "Syrian Pound SYP",
		},
		{
			kind:   "currency.number",
			expect: 760,
		},
		{
			kind:   "dir",
			expect: "/provident",
		},
		{
			kind:   "directory",
			expect: "/provident",
		},
		{
			kind:   "directory.directory",
			expect: "/provident",
		},
		{
			kind:   "directory.drive_letter",
			expect: "r:\\",
		},
		{
			kind:   "directory.unix_directory",
			expect: "/provident",
		},
		{
			kind:   "directory.windows_directory",
			expect: "r:\\natus",
		},
		{
			kind:   "emoji",
			expect: "😣",
		},
		{
			kind:   "emoji.emoji",
			expect: "😣",
		},
		{
			kind:   "emoji.emoji_code",
			expect: ":no_mouth:",
		},
		{
			kind:   "file",
			expect: "/provident/delectus.ics",
		},
		{
			kind:   "file.absolute_file_path",
			expect: "/provident/delectus.ics",
		},
		{
			kind:   "abspath",
			expect: "/provident/delectus.ics",
		},
		{
			kind:   "file.absolute_file_path_unix",
			expect: "/provident/delectus.ics",
		},
		{
			kind:   "file.absolute_file_path_for_unix",
			expect: "/provident/delectus.ics",
		},
		{
			kind:   "file.absolute_file_path_for_windows",
			expect: "r:\\natus\\eius.mp3",
		},
		{
			kind:   "file.extension",
			expect: "java",
		},
		{
			kind:   "file.filename_with_extension",
			expect: "natus.java",
		},
		{
			kind:   "food.fruit",
			expect: "Yuzu",
		},
		{
			kind:   "food.vegetable",
			expect: "Green Pepper",
		},
		{
			kind:   "gamer.tag",
			expect: "Revenge",
		},
		{
			kind:   "gender.abbr",
			expect: "fem",
		},
		{
			kind:   "gender.name",
			expect: "feminine",
		},
		{
			kind:   "genre.name",
			expect: "Vapogender",
		},
		{
			kind:   "hash.md5",
			expect: "7d0e8be0594b542aaa75f730fb13da4e",
		},
		{
			kind:   "hash.sha256",
			expect: "3a9ea6efcfc7874c771506906f0c276891d9dbb7c3e319c09fcd96f6cadf6497",
		},
		{
			kind:   "hash.sha512",
			expect: "25dfeea73bbbfeaa28c6be4752558be8951d22b38814719cbd29672b613721825da5d55baa5efbfdb9adf453e66fddc8481238b488b87c1fa7a86fd9462656aa",
		},
		{
			kind:   "internet.company_email",
			expect: "beer.dorthy@muller_muller_and_muller.ptt.biz",
		},
		{
			kind:   "internet.domain",
			expect: "ruk.com",
		},
		{
			kind:   "email",
			expect: "brown@example.org",
		},
		{
			kind:   "internet.email",
			expect: "muller.sean@yahoo.com",
		},
		{
			kind:   "internet.free_email",
			expect: "muller.sean@hotmail.com",
		},
		{
			kind:   "internet.free_email_domain",
			expect: "hotmail.com",
		},
		{
			kind:   "internet.http_method",
			expect: "TRACE",
		},
		{
			kind:   "ipv4",
			expect: "123.132.62.223",
		},
		{
			kind:   "internet.ipv4",
			expect: "123.132.62.223",
		},
		{
			kind:   "ipv6",
			expect: "4122:6815:1327:6754:3846:3647:2255:2272",
		},
		{
			kind:   "internet.ipv6",
			expect: "4122:6815:1327:6754:3846:3647:2255:2272",
		},
		{
			kind:   "internet.local_ipv4",
			expect: "192.168.132.62",
		},
		{
			kind:   "mac",
			expect: "B4:EF:15:80:39:F7",
		},
		{
			kind:   "internet.mac_address",
			expect: "B4:EF:15:80:39:F7",
		},
		{
			kind:   "password",
			expect: "ianz|{intrp}cdc",
		},
		{
			kind:   "internet.password",
			expect: "ianz|{intrp}cdc",
		},
		{
			kind:   "internet.query",
			expect: "?provident=natus&eveniet=8&id=aut&eligendi=7",
		},
		{
			kind:   "internet.safe_email",
			expect: "brown@example.org",
		},
		{
			kind:   "internet.safe_email_domain",
			expect: "example.org",
		},
		{
			kind:   "internet.slug",
			expect: "kpt-tue",
		},
		{
			kind:   "status_code",
			expect: 304,
		},
		{
			kind:   "internet.status_code",
			expect: 304,
		},
		{
			kind:   "internet.status_code_message",
			expect: "Not Modified",
		},
		{
			kind:   "internet.status_code_with_message",
			expect: "304 Not Modified",
		},
		{
			kind:   "internet.tld",
			expect: "info",
		},
		{
			kind:   "url",
			expect: "http://ukp.com/ezp-tne",
		},
		{
			kind:   "internet.url",
			expect: "http://ukp.com/ezp-tne",
		},
		{
			kind:   "user",
			expect: "brown",
		},
		{
			kind:   "internet.user",
			expect: "brown",
		},
		{
			kind:   "language",
			expect: "Japanese",
		},
		{
			kind:   "language.language",
			expect: "Japanese",
		},
		{
			kind:   "language.language_abbr",
			expect: "sg",
		},
		{
			kind:   "language.programming_language",
			expect: "Go",
		},
		{
			kind:   "lorem",
			expect: "natus delectus eius eveniet voluptate est dolores id aut amet nobis eligendi optio maxime est natus aliquid incidunt in nesciunt omnis fuga id quidem ut et vel. rerum explicabo impedit deserunt aut ut fuga eos fugit dicta accusantium culpa tempora quisquam accusantium sequi repellendus rerum neque et impedit officiis porro neque nemo deserunt sint occaecati sit quia est qui placeat saepe aut voluptas laborum natus doloribus. aperiam non maiores corporis eveniet facilis quo possimus nostrum explicabo et possimus quaerat perspiciatis id perspiciatis qui corrupti omnis sunt dolor explicabo facilis nesciunt quae provident culpa rem beatae deleniti reprehenderit ab et accusantium molestias accusamus eum tenetur deleniti rerum recusandae impedit delectus voluptatem corporis et iusto molestiae qui sed quia rerum repellendus aut facilis et omnis id aliquam ducimus et voluptates quidem unde impedit veniam voluptatem aspernatur omnis beatae nam voluptatem mollitia laborum est delectus autem voluptates quo similique id minus voluptatibus dolore blanditiis omnis optio dolorum dignissimos cupiditate repellendus rerum error unde autem.",
		},
		{
			kind:   "lorem.paragraph",
			expect: "natus delectus eius eveniet voluptate est dolores id aut amet nobis eligendi optio maxime est natus aliquid incidunt in nesciunt omnis fuga id quidem ut et vel. rerum explicabo impedit deserunt aut ut fuga eos fugit dicta accusantium culpa tempora quisquam accusantium sequi repellendus rerum neque et impedit officiis porro neque nemo deserunt sint occaecati sit quia est qui placeat saepe aut voluptas laborum natus doloribus. aperiam non maiores corporis eveniet facilis quo possimus nostrum explicabo et possimus quaerat perspiciatis id perspiciatis qui corrupti omnis sunt dolor explicabo facilis nesciunt quae provident culpa rem beatae deleniti reprehenderit ab et accusantium molestias accusamus eum tenetur deleniti rerum recusandae impedit delectus voluptatem corporis et iusto molestiae qui sed quia rerum repellendus aut facilis et omnis id aliquam ducimus et voluptates quidem unde impedit veniam voluptatem aspernatur omnis beatae nam voluptatem mollitia laborum est delectus autem voluptates quo similique id minus voluptatibus dolore blanditiis omnis optio dolorum dignissimos cupiditate repellendus rerum error unde autem.",
		},
		{
			kind:   "mime_type",
			expect: "audio/webm",
		},
		{
			kind:   "mime_type.mime_type",
			expect: "audio/webm",
		},

		//{
		//        kind:   "music.author",
		//        expect: "Ms. Shaina Rippin",
		//},
		{
			kind:   "music.genre",
			expect: "Noise pop",
		},
		{
			kind:   "music.length",
			expect: time.Duration(445000000000),
		},
		{
			kind:   "music.name",
			expect: "Recognizing Tawdry Jails",
		},

		{
			kind:   "payment.credit_card_expiration",
			expect: "11/22",
		},
		{
			kind:   "payment.credit_card_expiration_date_string",
			expect: "11/22",
		},
		{
			kind:   "payment.credit_card_number",
			expect: "1097178844101245",
		},
		{
			kind:   "cc",
			expect: "1097178844101245",
		},
		{
			kind:   "payment.credit_card_type",
			expect: "Visa",
		},

		{
			kind:   "name",
			expect: "Idell Denesik",
		},
		{
			kind:   "person.name",
			expect: "Idell Denesik",
		},
		{
			kind:   "person.first_name",
			expect: "Alexandro",
		},
		{
			kind:   "person.first_name_female",
			expect: "Jennie",
		},
		{
			kind:   "person.first_name_male",
			expect: "Darien",
		},
		{
			kind:   "person.gender",
			expect: "Female",
		},
		{
			kind:   "person.gender_female",
			expect: "Female",
		},
		{
			kind:   "person.gender_male",
			expect: "Male",
		},
		{
			kind:   "person.last_name",
			expect: "Zboncak",
		},
		{
			kind:   "person.name",
			expect: "Idell Denesik",
		},
		{
			kind:   "person.name_female",
			expect: "Jennie Brown",
		},
		{
			kind:   "person.name_male",
			expect: "Darien Brown",
		},
		{
			kind:   "person.ssn",
			expect: "241734987",
		},
		{
			kind:   "person.suffix",
			expect: "PhD",
		},
		{
			kind:   "person.title",
			expect: "Ms.",
		},
		{
			kind:   "person.title_male",
			expect: "Mr.",
		},
		{
			kind:   "person.title_female",
			expect: "Ms.",
		},
		{
			kind:   "pet.cat",
			expect: "Batman",
		},
		{
			kind:   "pet.dog",
			expect: "Louie",
		},
		{
			kind:   "pet.name",
			expect: "Midnight",
		},
		{
			kind:   "phone.area_code",
			expect: "500",
		},
		{
			kind:   "phone.e164_number",
			expect: "+90971788441",
		},
		{
			kind:   "phone.exchange_code",
			expect: "509",
		},
		{
			kind:   "phone",
			expect: "690.624.5299",
		},
		{
			kind:   "phone.number",
			expect: "690.624.5299",
		},
		{
			kind:   "phone.toll_free_area_code",
			expect: "888",
		},
		{
			kind:   "phone.tool_free_number",
			expect: "855.871.7884",
		},
		{
			kind:   "time.ansic",
			expect: "Wed May  5 00:00:00 1458",
		},
		{
			kind:   "time",
			expect: "1458-05-05T00:00:00Z",
		},
		{
			kind:   "time.ampm",
			expect: "pm",
		},
		{
			kind:   "time.century",
			expect: "III",
		},
		{
			kind:   "time.day_of_month",
			expect: 12,
		},
		{
			kind:   "time.day_of_week",
			expect: time.Weekday(2),
		},
		{
			kind:   "time.iso8601",
			expect: "1458-05-05T00:00:00+000",
		},
		{
			kind:   "time.kitchen",
			expect: "12:00AM",
		},
		{
			kind:   "time.month",
			expect: time.Month(12),
		},
		{
			kind:   "time.month_name",
			expect: "December",
		},

		{
			kind:   "time.rfc3339",
			expect: "1458-05-05T00:00:00Z",
		},
		{
			kind:   "rfc3339",
			expect: "1458-05-05T00:00:00Z",
		},
		{
			kind:   "time.rfc3339_nano",
			expect: "1458-05-05T00:00:00Z",
		},
		{
			kind:   "rfc3339_nano",
			expect: "1458-05-05T00:00:00Z",
		},
		{
			kind:   "time.rfc822",
			expect: "05 May 58 00:00 UTC",
		},
		{
			kind:   "rfc822",
			expect: "05 May 58 00:00 UTC",
		},
		{
			kind:   "time.rfc850",
			expect: "Wednesday, 05-May-58 00:00:00 UTC",
		},
		{
			kind:   "rfc850",
			expect: "Wednesday, 05-May-58 00:00:00 UTC",
		},
		{
			kind:   "time.ruby_date",
			expect: "Wed May 05 00:00:00 +0000 1458",
		},
		{
			kind:   "time.time",
			expect: time.Date(1458, time.May, 5, 0, 0, 0, 0, time.Local),
		},
		{
			kind:   "time.time_between",
			expect: time.Date(2481, time.August, 29, 0, 0, 0, 0, time.Local),
		},
		{
			kind:   "time.timezone",
			expect: "Asia/Kuching",
		},
		{
			kind:   "time.unix_date",
			expect: "Wed May  5 00:00:00 UTC 1458",
		},

		// This test will fail from time to time, because the upper
		// bound for the maximum year is determined based on the
		// current year. See:
		//
		// https://github.com/jaswdr/faker/blob/a0f3f9bd3b0f2fcb2b1ea9750cefba5db97e9718/time.go#L157
		//
		//{
		//        kind:   "time.year",
		//        expect: 2023,
		//},

		//{
		//        kind:   "uuid",
		//        expect: "14ae429a-98f7-4f9f-b004-405cb08faade",
		//},
		//{
		//        kind:   "uuid.v4",
		//        expect: "1eb51ab7-0a01-4f49-8349-4f1ba2136ab0",
		//},

		{
			kind:   "user_agent",
			expect: "Mozilla/4.0 (compatible; MSIE 6.0; ; Linux x86_64) Opera 7.50 [en]",
		},
		{
			kind:   "user_agent.chrome",
			expect: "Mozilla/5.0 (Windows NT 6.1; WOW64) AppleWebKit/535.1 (KHTML, like Gecko) Chrome/14.0.810.0 Safari/535.1",
		},
		{
			kind:   "user_agent.firefox",
			expect: "Mozilla/5.0 (X11; U; Linux i686; en-GB; rv:1.7.7) Gecko/20050414 Firefox/1.0.3",
		},
		{
			kind:   "user_agent.ie",
			expect: "Mozilla/5.0 (compatible; MSIE 8.0; Windows NT 5.1; Trident/4.0; SLCC1; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; .NET CLR 1.1.4322)",
		},
		{
			kind:   "user_agent.internet_explorer",
			expect: "Mozilla/5.0 (compatible; MSIE 8.0; Windows NT 5.1; Trident/4.0; SLCC1; .NET CLR 3.0.4506.2152; .NET CLR 3.5.30729; .NET CLR 1.1.4322)",
		},
		{
			kind:   "user_agent.opera",
			expect: "Mozilla/4.0 (compatible; MSIE 6.0; Windows CE; PPC; 320x320) Opera 8.65 [en]",
		},
		{
			kind:   "user_agent.safari",
			expect: "Mozilla/5.0 (iPad;U;CPU OS 3_2_2 like Mac OS X; en-us) AppleWebKit/531.21.10 (KHTML, like Gecko) Version/4.0.4 Mobile/7B500 Safari/531.21.10",
		},
		{
			kind:   "user_agent.user_agent",
			expect: "Mozilla/4.0 (compatible; MSIE 6.0; ; Linux x86_64) Opera 7.50 [en]",
		},

		//{
		//        kind:   "youtube.embeded_url",
		//        expect: "www.youtube.com/embed/uzNRy7jXZUT",
		//},
		//{
		//        kind:   "youtube.full_url",
		//        expect: "www.youtube.com/watch?v=FIf0iML8tCN",
		//},
		//{
		//        kind:   "youtube.generate_share_url",
		//        expect: "youtu.be/Rogy7gJUCzk",
		//},
		//{
		//        kind:   "youtube.video_id",
		//        expect: "vWLRQ3doSb0",
		//},
	}

	for _, c := range cases {
		opt := GetDefaultFakeOptions()
		opt.Faker = faker.NewWithSeed(rand.NewSource(42))

		// this formatting makes using `make test | awk` more convienient
		//
		// $ make test | awk '/^[ \t]/ && $2 == "kind" {kind=$3} /^[ \t]/ && $1 == "actual" {$1=""; $2=""; printf("{\nkind: \"%s\",\nexpect: %s,\n},\n", kind, $0)}'

		t.Logf("kind %s expectError %t expect %v", c.kind, c.expectError, c.expect)
		actual, err := opt.Fake(c.kind)
		if c.expectError {
			assert.NotNil(t, err)
			continue
		}
		assert.Nil(t, err)
		assert.Equal(t, c.expect, actual)

	}
}
