/*
 * Copyright (C) 2014-2022, AdaCore
 * SPDX-License-Identifier: Apache-2.0
 */

#ifndef LIBADALANG
#define LIBADALANG

#include <stdint.h>

#ifdef __cplusplus
extern "C" {
#endif

/*
 * This type represents a context for all source analysis. This is the first
 * type you need to create to use libadalang. It will contain the results of
 * all analysis, and is the main holder for all the data.
 *
 * You can create several analysis contexts if you need to, which enables you,
 * for example to:
 *
 * * analyze several different projects at the same time;
 *
 * * analyze different parts of the same projects in parallel.
 *
 * In the current design, contexts always keep all of their analysis units
 * allocated. If you need to get this memory released, the only option at your
 * disposal is to destroy your analysis context instance.
 *
 * This structure is partially opaque: some fields are exposed to allow direct
 * access, for performance concerns.
 */
typedef struct
{
   uint64_t serial_number;
} *ada_analysis_context;

/*
 * This type represents the analysis of a single file.
 *
 * This type has strong-reference semantics and is ref-counted. Furthermore, a
 * reference to a unit contains an implicit reference to the context that owns
 * it. This means that keeping a reference to a unit will keep the context and
 * all the unit it contains allocated.
 *
 * This structure is partially opaque: some fields are exposed to allow direct
 * access, for performance concerns.
 */
typedef struct
{
   uint64_t version_number;
} *ada_analysis_unit;

/*
 * Data type for all nodes. Nodes are assembled to make up a tree.  See the
 * node primitives below to inspect such trees.
 *
 * Unlike for contexts and units, this type has weak-reference semantics:
 * keeping a reference to a node has no effect on the decision to keep the unit
 * that it owns allocated. This means that once all references to the context
 * and units related to a node are dropped, the context and its units are
 * deallocated and the node becomes a stale reference: most operations on it
 * will raise a ``Stale_Reference_Error``.
 *
 * Note that since reparsing an analysis unit deallocates all the nodes it
 * contains, this operation makes all reference to these nodes stale as well.
 */
typedef struct ada_base_node__struct *ada_base_node;

/*
 * Kind of AST nodes in parse trees.
 */
typedef enum {
    

        /* ada_node (abstract)  */
        /*
         * Root node class for the Ada syntax tree.
         *
         * Derived nodes: ``ada_abort_node``, ``ada_abstract_node``,
         * ``ada_ada_list``, ``ada_aliased_node``, ``ada_all_node``,
         * ``ada_array_indices``, ``ada_aspect_assoc``, ``ada_aspect_clause``,
         * ``ada_aspect_spec``, ``ada_base_assoc``,
         * ``ada_base_formal_param_holder``, ``ada_base_record_def``,
         * ``ada_basic_assoc``, ``ada_basic_decl``,
         * ``ada_case_stmt_alternative``, ``ada_compilation_unit``,
         * ``ada_component_clause``, ``ada_component_def``,
         * ``ada_constant_node``, ``ada_constraint``, ``ada_declarative_part``,
         * ``ada_elsif_expr_part``, ``ada_elsif_stmt_part``, ``ada_expr``,
         * ``ada_for_loop_iter_filter``, ``ada_format_string_chunk``,
         * ``ada_format_string_tok_node``, ``ada_handled_stmts``,
         * ``ada_interface_kind``, ``ada_iter_type``, ``ada_library_item``,
         * ``ada_limited_node``, ``ada_loop_spec``, ``ada_mode``,
         * ``ada_multi_abstract_state_decl``, ``ada_not_null``,
         * ``ada_null_component_decl``, ``ada_others_designator``,
         * ``ada_overriding_node``, ``ada_params``,
         * ``ada_paren_abstract_state_decl``, ``ada_pp_directive``,
         * ``ada_pp_then_kw``, ``ada_pragma_node``, ``ada_private_node``,
         * ``ada_protected_def``, ``ada_protected_node``, ``ada_quantifier``,
         * ``ada_range_spec``, ``ada_renaming_clause``, ``ada_reverse_node``,
         * ``ada_select_when_part``, ``ada_stmt``, ``ada_subp_kind``,
         * ``ada_subunit``, ``ada_synchronized_node``, ``ada_tagged_node``,
         * ``ada_task_def``, ``ada_type_attributes_repository``,
         * ``ada_type_def``, ``ada_type_expr``,
         * ``ada_unconstrained_array_index``, ``ada_until_node``,
         * ``ada_use_clause``, ``ada_value_sequence``, ``ada_variant_part``,
         * ``ada_variant``, ``ada_with_clause``, ``ada_with_private``
         */
    

        /* abort_node (abstract)  */
        /*
         * Qualifier for the ``abort`` keyword.
         *
         * Derived nodes: ``ada_abort_absent``, ``ada_abort_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_abort_absent = 1,
    

        /*
         * This node type has no derivation.
         */
        ada_abort_present = 2,
    

        /* abstract_node (abstract)  */
        /*
         * Qualifier for the ``abstract`` keyword.
         *
         * Derived nodes: ``ada_abstract_absent``, ``ada_abstract_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_abstract_absent = 3,
    

        /*
         * This node type has no derivation.
         */
        ada_abstract_present = 4,
    

        /* ada_list (abstract)  */
        /*
         * Derived nodes: ``ada_ada_node_list``, ``ada_aspect_assoc_list``,
         * ``ada_base_assoc_list``, ``ada_basic_assoc_list``,
         * ``ada_case_expr_alternative_list``,
         * ``ada_case_stmt_alternative_list``, ``ada_compilation_unit_list``,
         * ``ada_concat_operand_list``, ``ada_contract_case_assoc_list``,
         * ``ada_defining_name_list``, ``ada_discriminant_spec_list``,
         * ``ada_elsif_expr_part_list``, ``ada_elsif_stmt_part_list``,
         * ``ada_enum_literal_decl_list``, ``ada_expr_list``,
         * ``ada_format_string_chunk_list``, ``ada_identifier_list``,
         * ``ada_name_list``, ``ada_param_spec_list``,
         * ``ada_pragma_node_list``, ``ada_select_when_part_list``,
         * ``ada_unconstrained_array_index_list``, ``ada_variant_list``
         */
    

        /*
         * List of AdaNode.
         *
         * This list node can contain one of the following nodes:
         * ``ada_abstract_state_decl``, ``ada_abstract_subp_decl``,
         * ``ada_allocator``, ``ada_array_subcomponent_choice_name``,
         * ``ada_aspect_clause``, ``ada_attribute_ref``,
         * ``ada_base_aggregate``, ``ada_base_subp_body``, ``ada_bin_op``,
         * ``ada_body_stub``, ``ada_call_expr``, ``ada_char_literal``,
         * ``ada_component_clause``, ``ada_component_decl``, ``ada_concat_op``,
         * ``ada_concrete_type_decl``, ``ada_cond_expr``, ``ada_decl_expr``,
         * ``ada_dotted_name``, ``ada_entry_body``, ``ada_entry_decl``,
         * ``ada_error_decl``, ``ada_exception_decl``,
         * ``ada_exception_handler``, ``ada_explicit_deref``,
         * ``ada_format_string_literal``, ``ada_generic_decl``,
         * ``ada_generic_formal``, ``ada_generic_instantiation``,
         * ``ada_generic_renaming_decl``, ``ada_identifier``,
         * ``ada_incomplete_type_decl``, ``ada_membership_expr``,
         * ``ada_null_component_decl``, ``ada_null_literal``,
         * ``ada_num_literal``, ``ada_number_decl``, ``ada_object_decl``,
         * ``ada_others_designator``, ``ada_package_body``,
         * ``ada_package_decl``, ``ada_package_renaming_decl``,
         * ``ada_paren_abstract_state_decl``, ``ada_paren_expr``,
         * ``ada_pragma_node``, ``ada_protected_body``,
         * ``ada_protected_type_decl``, ``ada_qual_expr``,
         * ``ada_quantified_expr``, ``ada_raise_expr``,
         * ``ada_reduce_attribute_ref``, ``ada_single_protected_decl``,
         * ``ada_single_task_decl``, ``ada_stmt``, ``ada_string_literal``,
         * ``ada_subp_decl``, ``ada_subtype_decl``, ``ada_subtype_indication``,
         * ``ada_target_name``, ``ada_task_body``, ``ada_task_type_decl``,
         * ``ada_un_op``, ``ada_update_attribute_ref``, ``ada_use_clause``,
         * ``ada_with_clause``
         *
         * Derived nodes: ``ada_abstract_state_decl_list``,
         * ``ada_alternatives_list``, ``ada_constraint_list``,
         * ``ada_decl_list``, ``ada_stmt_list``
         */
        ada_ada_node_list = 5,
    

        /*
         * List of AbstractStateDecls.
         *
         * This list node can contain one of the following nodes:
         * ``ada_abstract_state_decl``, ``ada_paren_abstract_state_decl``
         *
         * This node type has no derivation.
         */
        ada_abstract_state_decl_list = 6,
    

        /*
         * List of alternatives in a ``when ...`` clause.
         *
         * This list node can contain one of the following nodes:
         * ``ada_allocator``, ``ada_array_subcomponent_choice_name``,
         * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
         * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
         * ``ada_cond_expr``, ``ada_decl_expr``,
         * ``ada_discrete_subtype_indication``, ``ada_dotted_name``,
         * ``ada_explicit_deref``, ``ada_format_string_literal``,
         * ``ada_identifier``, ``ada_membership_expr``, ``ada_null_literal``,
         * ``ada_num_literal``, ``ada_others_designator``, ``ada_paren_expr``,
         * ``ada_qual_expr``, ``ada_quantified_expr``, ``ada_raise_expr``,
         * ``ada_reduce_attribute_ref``, ``ada_string_literal``,
         * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
         *
         * This node type has no derivation.
         */
        ada_alternatives_list = 7,
    

        /*
         * List of constraints.
         *
         * This list node can contain one of the following nodes:
         * ``ada_attribute_ref``, ``ada_bin_op``, ``ada_call_expr``,
         * ``ada_char_literal``, ``ada_dotted_name``, ``ada_explicit_deref``,
         * ``ada_identifier``, ``ada_qual_expr``, ``ada_reduce_attribute_ref``,
         * ``ada_string_literal``, ``ada_subtype_indication``,
         * ``ada_target_name``, ``ada_update_attribute_ref``
         *
         * This node type has no derivation.
         */
        ada_constraint_list = 8,
    

        /*
         * List of declarations.
         *
         * This list node can contain one of the following nodes:
         * ``ada_abstract_subp_decl``, ``ada_aspect_clause``,
         * ``ada_component_decl``, ``ada_entry_decl``, ``ada_expr_function``,
         * ``ada_null_subp_decl``, ``ada_pragma_node``, ``ada_subp_decl``,
         * ``ada_subp_renaming_decl``
         *
         * This node type has no derivation.
         */
        ada_decl_list = 9,
    

        /*
         * List of statements.
         *
         * This list node can contain one of the following nodes:
         * ``ada_pragma_node``, ``ada_stmt``
         *
         * This node type has no derivation.
         */
        ada_stmt_list = 10,
    

        /*
         * List of AspectAssoc.
         *
         * This node type has no derivation.
         */
        ada_aspect_assoc_list = 11,
    

        /*
         * List of BaseAssoc.
         *
         * This node type has no derivation.
         */
        ada_base_assoc_list = 12,
    

        /* basic_assoc_list (abstract)  */
        /*
         * List of BasicAssoc.
         *
         * Derived nodes: ``ada_assoc_list``
         */
    

        /*
         * List of associations.
         *
         * This node type has no derivation.
         */
        ada_assoc_list = 13,
    

        /*
         * List of CaseExprAlternative.
         *
         * This node type has no derivation.
         */
        ada_case_expr_alternative_list = 14,
    

        /*
         * List of CaseStmtAlternative.
         *
         * This node type has no derivation.
         */
        ada_case_stmt_alternative_list = 15,
    

        /*
         * List of CompilationUnit.
         *
         * This node type has no derivation.
         */
        ada_compilation_unit_list = 16,
    

        /*
         * List of ConcatOperand.
         *
         * This node type has no derivation.
         */
        ada_concat_operand_list = 17,
    

        /*
         * List of ContractCaseAssoc.
         *
         * This node type has no derivation.
         */
        ada_contract_case_assoc_list = 18,
    

        /*
         * List of DefiningName.
         *
         * This node type has no derivation.
         */
        ada_defining_name_list = 19,
    

        /*
         * List of DiscriminantSpec.
         *
         * This node type has no derivation.
         */
        ada_discriminant_spec_list = 20,
    

        /*
         * List of ElsifExprPart.
         *
         * This node type has no derivation.
         */
        ada_elsif_expr_part_list = 21,
    

        /*
         * List of ElsifStmtPart.
         *
         * This node type has no derivation.
         */
        ada_elsif_stmt_part_list = 22,
    

        /*
         * List of EnumLiteralDecl.
         *
         * This node type has no derivation.
         */
        ada_enum_literal_decl_list = 23,
    

        /* expr_list (abstract)  */
        /*
         * List of Expr.
         *
         * This list node can contain one of the following nodes:
         * ``ada_allocator``, ``ada_attribute_ref``, ``ada_base_aggregate``,
         * ``ada_bin_op``, ``ada_call_expr``, ``ada_char_literal``,
         * ``ada_concat_op``, ``ada_cond_expr``, ``ada_decl_expr``,
         * ``ada_discrete_subtype_name``, ``ada_dotted_name``,
         * ``ada_explicit_deref``, ``ada_format_string_literal``,
         * ``ada_identifier``, ``ada_null_literal``, ``ada_num_literal``,
         * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
         * ``ada_raise_expr``, ``ada_reduce_attribute_ref``,
         * ``ada_string_literal``, ``ada_target_name``, ``ada_un_op``,
         * ``ada_update_attribute_ref``
         *
         * Derived nodes: ``ada_expr_alternatives_list``
         */
    

        /*
         * List of alternatives in a membership test expression.
         *
         * This list node can contain one of the following nodes:
         * ``ada_allocator``, ``ada_attribute_ref``, ``ada_base_aggregate``,
         * ``ada_bin_op``, ``ada_call_expr``, ``ada_char_literal``,
         * ``ada_concat_op``, ``ada_cond_expr``, ``ada_decl_expr``,
         * ``ada_discrete_subtype_name``, ``ada_dotted_name``,
         * ``ada_explicit_deref``, ``ada_format_string_literal``,
         * ``ada_identifier``, ``ada_null_literal``, ``ada_num_literal``,
         * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
         * ``ada_raise_expr``, ``ada_reduce_attribute_ref``,
         * ``ada_string_literal``, ``ada_target_name``, ``ada_un_op``,
         * ``ada_update_attribute_ref``
         *
         * This node type has no derivation.
         */
        ada_expr_alternatives_list = 24,
    

        /*
         * List of FormatStringChunk.
         *
         * This node type has no derivation.
         */
        ada_format_string_chunk_list = 25,
    

        /* identifier_list (abstract)  */
        /*
         * List of Identifier.
         *
         * Derived nodes: ``ada_discriminant_choice_list``
         */
    

        /*
         * List of discriminant associations.
         *
         * This node type has no derivation.
         */
        ada_discriminant_choice_list = 26,
    

        /*
         * List of Name.
         *
         * This list node can contain one of the following nodes:
         * ``ada_attribute_ref``, ``ada_call_expr``, ``ada_char_literal``,
         * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_identifier``,
         * ``ada_qual_expr``, ``ada_reduce_attribute_ref``,
         * ``ada_string_literal``, ``ada_target_name``,
         * ``ada_update_attribute_ref``
         *
         * Derived nodes: ``ada_parent_list``
         */
        ada_name_list = 27,
    

        /*
         * List of parents in a type declaration.
         *
         * This list node can contain one of the following nodes:
         * ``ada_char_literal``, ``ada_dotted_name``, ``ada_identifier``,
         * ``ada_string_literal``
         *
         * This node type has no derivation.
         */
        ada_parent_list = 28,
    

        /*
         * List of ParamSpec.
         *
         * This node type has no derivation.
         */
        ada_param_spec_list = 29,
    

        /*
         * List of Pragma.
         *
         * This node type has no derivation.
         */
        ada_pragma_node_list = 30,
    

        /*
         * List of SelectWhenPart.
         *
         * This node type has no derivation.
         */
        ada_select_when_part_list = 31,
    

        /*
         * List of UnconstrainedArrayIndex.
         *
         * This node type has no derivation.
         */
        ada_unconstrained_array_index_list = 32,
    

        /*
         * List of Variant.
         *
         * This node type has no derivation.
         */
        ada_variant_list = 33,
    

        /* aliased_node (abstract)  */
        /*
         * Qualifier for the ``aliased`` keyword.
         *
         * Derived nodes: ``ada_aliased_absent``, ``ada_aliased_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_aliased_absent = 34,
    

        /*
         * This node type has no derivation.
         */
        ada_aliased_present = 35,
    

        /* all_node (abstract)  */
        /*
         * Qualifier for the ``all`` keyword.
         *
         * Derived nodes: ``ada_all_absent``, ``ada_all_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_all_absent = 36,
    

        /*
         * This node type has no derivation.
         */
        ada_all_present = 37,
    

        /* array_indices (abstract)  */
        /*
         * Specification for array indexes (:rmlink:`3.6`).
         *
         * Derived nodes: ``ada_constrained_array_indices``,
         * ``ada_unconstrained_array_indices``
         */
    

        /*
         * Constrained specification for array indexes (:rmlink:`3.6`).
         *
         * This node type has no derivation.
         */
        ada_constrained_array_indices = 38,
    

        /*
         * Unconstrained specification for array indexes (:rmlink:`3.6`).
         *
         * This node type has no derivation.
         */
        ada_unconstrained_array_indices = 39,
    

        /*
         * Name/expression association in an aspect.
         *
         * This node type has no derivation.
         */
        ada_aspect_assoc = 40,
    

        /* aspect_clause (abstract)  */
        /*
         * Base class for aspect clauses.
         *
         * Derived nodes: ``ada_at_clause``, ``ada_attribute_def_clause``,
         * ``ada_enum_rep_clause``, ``ada_record_rep_clause``
         */
    

        /*
         * Representation clause (``for .. use at ...;``) (:rmlink:`13.5.1`).
         *
         * This node type has no derivation.
         */
        ada_at_clause = 41,
    

        /*
         * Clause for an attribute definition (``for ...'Attribute use ...;``)
         * (:rmlink:`13.3`).
         *
         * This node type has no derivation.
         */
        ada_attribute_def_clause = 42,
    

        /*
         * Representation clause for enumeration types (:rmlink:`13.4`).
         *
         * This node type has no derivation.
         */
        ada_enum_rep_clause = 43,
    

        /*
         * Representation clause for a record type (:rmlink:`13.5.1`).
         *
         * This node type has no derivation.
         */
        ada_record_rep_clause = 44,
    

        /*
         * List of aspects in a declaration (:rmlink:`13.1.1`).
         *
         * This node type has no derivation.
         */
        ada_aspect_spec = 45,
    

        /* base_assoc (abstract)  */
        /*
         * Abstract class for a key/value association, where the value is an
         * expression.
         *
         * Derived nodes: ``ada_contract_case_assoc``,
         * ``ada_pragma_argument_assoc``
         */
    

        /*
         * Single association for the ``Contract_Case`` aspect.
         *
         * This node type has no derivation.
         */
        ada_contract_case_assoc = 46,
    

        /*
         * Argument association in a pragma.
         *
         * This node type has no derivation.
         */
        ada_pragma_argument_assoc = 47,
    

        /* base_formal_param_holder (abstract)  */
        /*
         * Base class for lists of formal parameters. This is used in every
         * case a list of "formals" can be called or instantiated, so in all
         * the following cases:
         *
         * * Subprogram specifications (and subprogram calls).
         *
         * * Component lists (and aggregates).
         *
         * * Generic formals (and generic instantiations).
         *
         * This allows to share the parameter unpacking/matching logic.
         *
         * This is a Libadalang abstraction that has no existence in the Ada
         * reference manual.
         *
         * Derived nodes: ``ada_base_subp_spec``, ``ada_component_list``,
         * ``ada_discriminant_part``, ``ada_entry_completion_formal_params``,
         * ``ada_generic_formal_part``
         */
    

        /* base_subp_spec (abstract)  */
        /*
         * Base class for subprogram specifications (:rmlink:`6.1`).
         *
         * Derived nodes: ``ada_entry_spec``, ``ada_enum_subp_spec``,
         * ``ada_subp_spec``, ``ada_synthetic_binary_spec``,
         * ``ada_synthetic_unary_spec``
         */
    

        /*
         * Entry specification.
         *
         * This node does not have ARM existence, because in the RM subprogram
         * specifications don't encompass the ad-hoc specifications that happen
         * in entry declarations. Entry declarations are described in
         * :rmlink:`9.5.2`.
         *
         * This node type has no derivation.
         */
        ada_entry_spec = 48,
    

        /*
         * Synthetic node for the abstract subprogram spec of an enum literal.
         *
         * NOTE: This has no existence in the ARM. While enum literals are
         * functions semantically, they're not such syntactically.
         *
         * This node type has no derivation.
         */
        ada_enum_subp_spec = 49,
    

        /*
         * Subprogram specification (:rmlink:`6.1`).
         *
         * This node type has no derivation.
         */
        ada_subp_spec = 50,
    

        /*
         * Synthetic subprogram specification for binary operators.
         *
         * This node type has no derivation.
         */
        ada_synthetic_binary_spec = 51,
    

        /*
         * Synthetic subprogram specification for unary operators.
         *
         * This node type has no derivation.
         */
        ada_synthetic_unary_spec = 52,
    

        /*
         * List of component declarations (:rmlink:`3.8`).
         *
         * This node type has no derivation.
         */
        ada_component_list = 53,
    

        /* discriminant_part (abstract)  */
        /*
         * Specification for discriminants in type declarations.
         *
         * Derived nodes: ``ada_known_discriminant_part``,
         * ``ada_unknown_discriminant_part``
         */
    

        /*
         * Known list of discriminants in type declarations (:rmlink:`3.7`).
         *
         * This node type has no derivation.
         */
        ada_known_discriminant_part = 54,
    

        /*
         * Unknown list of discriminants in type declarations (:rmlink:`3.7`).
         *
         * This node type has no derivation.
         */
        ada_unknown_discriminant_part = 55,
    

        /*
         * Formal parameters for the completion of an ``EntryDecl`` (either an
         * ``EntryBody`` or an ``AcceptStmt``).
         *
         * This node type has no derivation.
         */
        ada_entry_completion_formal_params = 56,
    

        /*
         * List of declaration for generic formals (:rmlink:`12.1`).
         *
         * This node type has no derivation.
         */
        ada_generic_formal_part = 57,
    

        /* base_record_def (abstract)  */
        /*
         * Base class for record definitions (:rmlink:`3.8`).
         *
         * Derived nodes: ``ada_null_record_def``, ``ada_record_def``
         */
    

        /*
         * Record definition for ``null record``.
         *
         * This node type has no derivation.
         */
        ada_null_record_def = 58,
    

        /*
         * Record definition that contains components (``record ... end
         * record``).
         *
         * This node type has no derivation.
         */
        ada_record_def = 59,
    

        /* basic_assoc (abstract)  */
        /*
         * Association of one or several names to an expression.
         *
         * Derived nodes: ``ada_aggregate_assoc``,
         * ``ada_composite_constraint_assoc``, ``ada_iterated_assoc``,
         * ``ada_param_assoc``
         */
    

        /*
         * Association (X => Y) used for aggregates associations
         * (:rmlink:`4.3`).
         *
         * Derived nodes: ``ada_multi_dim_array_assoc``
         */
        ada_aggregate_assoc = 60,
    

        /*
         * Association used for multi-dimension array aggregates.
         *
         * This node type has no derivation.
         */
        ada_multi_dim_array_assoc = 61,
    

        /*
         * Association of discriminant names to an expression
         * (:rmlink:`3.7.1`).
         *
         * This node type has no derivation.
         */
        ada_composite_constraint_assoc = 62,
    

        /*
         * Iterated association (Ada 2020, :rmlink:`4.3.3`).
         *
         * This node type has no derivation.
         */
        ada_iterated_assoc = 63,
    

        /*
         * Association (X => Y) used for parameter associations
         * (:rmlink:`6.4`).
         *
         * This node type has no derivation.
         */
        ada_param_assoc = 64,
    

        /* basic_decl (abstract)  */
        /*
         * Root class for an Ada declaration (:rmlink:`3.1`). A declaration
         * associates a name with a language entity, for example a type or a
         * variable.
         *
         * Derived nodes: ``ada_abstract_state_decl``,
         * ``ada_anonymous_expr_decl``, ``ada_base_formal_param_decl``,
         * ``ada_base_package_decl``, ``ada_base_type_decl``,
         * ``ada_basic_subp_decl``, ``ada_body_node``,
         * ``ada_entry_index_spec``, ``ada_error_decl``,
         * ``ada_exception_decl``, ``ada_exception_handler``,
         * ``ada_for_loop_var_decl``, ``ada_generic_decl``,
         * ``ada_generic_instantiation``, ``ada_generic_renaming_decl``,
         * ``ada_label_decl``, ``ada_named_stmt_decl``, ``ada_number_decl``,
         * ``ada_object_decl``, ``ada_package_renaming_decl``,
         * ``ada_single_protected_decl``, ``ada_single_task_decl``,
         * ``ada_synthetic_object_decl``
         */
    

        /*
         * Contained (directly or indirectly) in an AbstractStateDeclExpr, and
         * is used to represent the BasicDecl associated with the abstract
         * state introduced by the Abstract_State aspect. This node is
         * necessary because all of our name resolution routines expect
         * BasicDecls as environments' values.
         *
         * The only purpose of this node is to populate the env with the
         * abstract state declared through this node, so it can be referred in
         * SPARK aspects such as Global, Depends, Refined_State, etc.
         *
         * This node type has no derivation.
         */
        ada_abstract_state_decl = 65,
    

        /*
         * Represents a anonymous declaration that holds an expression.
         *
         * This is used to store the results of queries such as
         * ``referenced_decl`` called on references to object formals from
         * inside a instantiated generic in order to return the relevant
         * actual.
         *
         * Indeed, ``referenced_decl`` must return a ``BasicDecl``, but actuals
         * of generic instantiations are ``Expr``. This wrapper node is
         * therefore a way to both satisfy the ``BasicDecl`` interface, and
         * provide to the user the expression of the actual through the
         * ``expr`` field.
         *
         * This node type has no derivation.
         */
        ada_anonymous_expr_decl = 66,
    

        /* base_formal_param_decl (abstract)  */
        /*
         * Base class for formal parameter declarations. This is used both for
         * records components and for subprogram parameters.
         *
         * This is a Libadalang abstraction, that has no ARM existence.
         *
         * Derived nodes: ``ada_component_decl``, ``ada_discriminant_spec``,
         * ``ada_generic_formal``, ``ada_param_spec``,
         * ``ada_synthetic_formal_param_decl``
         */
    

        /*
         * Declaration for a component (:rmlink:`3.8`).
         *
         * This node type has no derivation.
         */
        ada_component_decl = 67,
    

        /*
         * Known list of discriminants in type declarations (:rmlink:`3.7`).
         *
         * This node type has no derivation.
         */
        ada_discriminant_spec = 68,
    

        /* generic_formal (abstract)  */
        /*
         * Enclosing declaration for a generic formal. The real declaration is
         * accessible via the ``decl`` field.
         *
         * Derived nodes: ``ada_generic_formal_obj_decl``,
         * ``ada_generic_formal_package``, ``ada_generic_formal_subp_decl``,
         * ``ada_generic_formal_type_decl``
         */
    

        /*
         * Formal declaration for an object.
         *
         * This node type has no derivation.
         */
        ada_generic_formal_obj_decl = 69,
    

        /*
         * Formal declaration for a package (:rmlink:`12.1`).
         *
         * This node type has no derivation.
         */
        ada_generic_formal_package = 70,
    

        /*
         * Formal declaration for a subprogram (:rmlink:`12.1`).
         *
         * This node type has no derivation.
         */
        ada_generic_formal_subp_decl = 71,
    

        /*
         * Formal declaration for a type (:rmlink:`12.1`).
         *
         * This node type has no derivation.
         */
        ada_generic_formal_type_decl = 72,
    

        /*
         * Specification for a parameter (:rmlink:`6.1`).
         *
         * This node type has no derivation.
         */
        ada_param_spec = 73,
    

        /*
         * Synthetic parameter declaration.
         *
         * This node type has no derivation.
         */
        ada_synthetic_formal_param_decl = 74,
    

        /* base_package_decl (abstract)  */
        /*
         * Base class for package declarations. This will be used both for non-
         * generic package declarations (via ``ada_package_decl``) and for
         * generic ones (via ``ada_generic_package_internal``).
         *
         * Derived nodes: ``ada_generic_package_internal``,
         * ``ada_package_decl``
         */
    

        /*
         * This class denotes the internal package contained by a
         * GenericPackageDecl.
         *
         * This node type has no derivation.
         */
        ada_generic_package_internal = 75,
    

        /*
         * Non-generic package declarations (:rmlink:`7.1`).
         *
         * This node type has no derivation.
         */
        ada_package_decl = 76,
    

        /* base_type_decl (abstract)  */
        /*
         * Base class for type declarations. It unifies every kind of type that
         * exists in Ada, including types that have no source existence like
         * classwide types.
         *
         * Derived nodes: ``ada_base_subtype_decl``,
         * ``ada_classwide_type_decl``, ``ada_incomplete_type_decl``,
         * ``ada_protected_type_decl``, ``ada_task_type_decl``,
         * ``ada_type_decl``
         */
    

        /* base_subtype_decl (abstract)  */
        /*
         * Base class for subtype declarations (:rmlink:`3.2.2`).
         *
         * Derived nodes: ``ada_discrete_base_subtype_decl``,
         * ``ada_subtype_decl``
         */
    

        /*
         * Specific ``BaseSubtypeDecl`` synthetic subclass for the base type of
         * scalar types.
         *
         * This node type has no derivation.
         */
        ada_discrete_base_subtype_decl = 77,
    

        /*
         * Subtype declaration (:rmlink:`3.2.2`).
         *
         * This node type has no derivation.
         */
        ada_subtype_decl = 78,
    

        /*
         * Synthetic node (not parsed, generated from a property call). Refers
         * to the classwide type for a given tagged type (:rmlink:`3.4.1`).
         *
         * This node type has no derivation.
         */
        ada_classwide_type_decl = 79,
    

        /*
         * Incomplete declaration for a type (:rmlink:`12.5`).
         *
         * Derived nodes: ``ada_incomplete_formal_type_decl``,
         * ``ada_incomplete_tagged_type_decl``
         */
        ada_incomplete_type_decl = 80,
    

        /*
         * A formal incomplete type declaration.
         *
         * This node type has no derivation.
         */
        ada_incomplete_formal_type_decl = 81,
    

        /*
         * Incomplete declaration for a tagged type.
         *
         * This node type has no derivation.
         */
        ada_incomplete_tagged_type_decl = 82,
    

        /*
         * Declaration for a protected type (:rmlink:`9.4`).
         *
         * This node type has no derivation.
         */
        ada_protected_type_decl = 83,
    

        /*
         * Declaration for a task type (:rmlink:`9.1`).
         *
         * Derived nodes: ``ada_single_task_type_decl``
         */
        ada_task_type_decl = 84,
    

        /*
         * Type declaration for a single task (:rmlink:`9.1`).
         *
         * This node type has no derivation.
         */
        ada_single_task_type_decl = 85,
    

        /* type_decl (abstract)  */
        /*
         * Type declarations that embed a type definition node. Corresponds to
         * the ARM's full type declarations (:rmlink:`3.2.1`).
         *
         * Derived nodes: ``ada_anonymous_type_decl``,
         * ``ada_concrete_type_decl``, ``ada_formal_type_decl``
         */
    

        /*
         * Anonymous type declaration (for anonymous array or access types).
         * This class has no RM existence, and anonymous (sub)types are
         * referred to implicitly in the RM.
         *
         * Derived nodes: ``ada_synth_anonymous_type_decl``
         */
        ada_anonymous_type_decl = 86,
    

        /*
         * Synthetic anonymous type decl. Used to generate anonymous access
         * types.
         *
         * This node type has no derivation.
         */
        ada_synth_anonymous_type_decl = 87,
    

        /*
         * A concrete type declaration.
         *
         * This node type has no derivation.
         */
        ada_concrete_type_decl = 88,
    

        /*
         * A formal type declaration.
         *
         * This node type has no derivation.
         */
        ada_formal_type_decl = 89,
    

        /* basic_subp_decl (abstract)  */
        /*
         * Base class for subprogram declarations.
         *
         * Derived nodes: ``ada_classic_subp_decl``, ``ada_entry_decl``,
         * ``ada_enum_literal_decl``, ``ada_generic_subp_internal``,
         * ``ada_synthetic_subp_decl``
         */
    

        /* classic_subp_decl (abstract)  */
        /*
         * This is an intermediate abstract class for subprogram declarations
         * with a common structure: overriding indicator, ``SubpSpec``,
         * aspects, <other fields>.
         *
         * Derived nodes: ``ada_abstract_subp_decl``, ``ada_formal_subp_decl``,
         * ``ada_subp_decl``
         */
    

        /*
         * Declaration for an abstract subprogram (:rmlink:`3.9.3`).
         *
         * This node type has no derivation.
         */
        ada_abstract_subp_decl = 90,
    

        /* formal_subp_decl (abstract)  */
        /*
         * Formal subprogram declarations, in generic declarations formal parts
         * (:rmlink:`12.6`).
         *
         * Derived nodes: ``ada_abstract_formal_subp_decl``,
         * ``ada_concrete_formal_subp_decl``
         */
    

        /*
         * Formal declaration for an abstract subprogram (:rmlink:`12.6`).
         *
         * This node type has no derivation.
         */
        ada_abstract_formal_subp_decl = 91,
    

        /*
         * Formal declaration for a concrete subprogram (:rmlink:`12.6`).
         *
         * This node type has no derivation.
         */
        ada_concrete_formal_subp_decl = 92,
    

        /*
         * Regular subprogram declaration (:rmlink:`6.1`).
         *
         * This node type has no derivation.
         */
        ada_subp_decl = 93,
    

        /*
         * Entry declaration (:rmlink:`9.4`).
         *
         * This node type has no derivation.
         */
        ada_entry_decl = 94,
    

        /*
         * Declaration for an enumeration literal (:rmlink:`3.5.1`).
         *
         * Derived nodes: ``ada_synthetic_char_enum_lit``
         */
        ada_enum_literal_decl = 95,
    

        /*
         * Synthetic character enum literal declaration.
         *
         * This node type has no derivation.
         */
        ada_synthetic_char_enum_lit = 96,
    

        /*
         * Internal node for generic subprograms.
         *
         * This node type has no derivation.
         */
        ada_generic_subp_internal = 97,
    

        /*
         * Synthetic subprogram declaration.
         *
         * Is used to represent predefined operators. This should also be
         * usable for synthesizing function attributes.
         *
         * This node type has no derivation.
         */
        ada_synthetic_subp_decl = 98,
    

        /* body_node (abstract)  */
        /*
         * Base class for an Ada body (:rmlink:`3.11`). A body is the
         * completion of a declaration.
         *
         * Derived nodes: ``ada_accept_stmt_body``, ``ada_base_subp_body``,
         * ``ada_body_stub``, ``ada_entry_body``, ``ada_package_body``,
         * ``ada_protected_body``, ``ada_task_body``
         */
    

        /*
         * BasicDecl that is always the declaration of an AcceptStmt. This is
         * nested *inside* of the accept statement.
         *
         * This node type has no derivation.
         */
        ada_accept_stmt_body = 99,
    

        /* base_subp_body (abstract)  */
        /*
         * Base class for subprogram bodies (:rmlink:`6.3`).
         *
         * Derived nodes: ``ada_expr_function``, ``ada_null_subp_decl``,
         * ``ada_subp_body``, ``ada_subp_renaming_decl``
         */
    

        /*
         * Expression function (:rmlink:`6.8`).
         *
         * This node type has no derivation.
         */
        ada_expr_function = 100,
    

        /*
         * Declaration for a null subprogram (:rmlink:`6.1`).
         *
         * This node type has no derivation.
         */
        ada_null_subp_decl = 101,
    

        /*
         * Subprogram body(:rmlink:`6.3`) .
         *
         * This node type has no derivation.
         */
        ada_subp_body = 102,
    

        /*
         * Declaration for a subprogram renaming (:rmlink:`8.5.4`).
         *
         * This node type has no derivation.
         */
        ada_subp_renaming_decl = 103,
    

        /* body_stub (abstract)  */
        /*
         * Base class for a body stub (:rmlink:`10.1.3`). A body stub is meant
         * to be completed by .
         *
         * Derived nodes: ``ada_package_body_stub``,
         * ``ada_protected_body_stub``, ``ada_subp_body_stub``,
         * ``ada_task_body_stub``
         */
    

        /*
         * Stub for a package body (``is separate``) (:rmlink:`10.1.3`).
         *
         * This node type has no derivation.
         */
        ada_package_body_stub = 104,
    

        /*
         * Stub for a protected object body (``is separate``)
         * (:rmlink:`10.1.3`).
         *
         * This node type has no derivation.
         */
        ada_protected_body_stub = 105,
    

        /*
         * Stub for a subprogram body (``is separate``) (:rmlink:`10.1.3`).
         *
         * This node type has no derivation.
         */
        ada_subp_body_stub = 106,
    

        /*
         * Stub for a task body (``is separate``) (:rmlink:`10.1.3`).
         *
         * This node type has no derivation.
         */
        ada_task_body_stub = 107,
    

        /*
         * Entry body (:rmlink:`9.5.2`).
         *
         * This node type has no derivation.
         */
        ada_entry_body = 108,
    

        /*
         * Package body (:rmlink:`7.2`).
         *
         * This node type has no derivation.
         */
        ada_package_body = 109,
    

        /*
         * Protected object body (:rmlink:`9.4`).
         *
         * This node type has no derivation.
         */
        ada_protected_body = 110,
    

        /*
         * Task body (:rmlink:`9.1`).
         *
         * This node type has no derivation.
         */
        ada_task_body = 111,
    

        /*
         * Index specification for an entry body (:rmlink:`9.5.2`).
         *
         * This node type has no derivation.
         */
        ada_entry_index_spec = 112,
    

        /*
         * Placeholder node for syntax errors in lists of declarations.
         *
         * This node type has no derivation.
         */
        ada_error_decl = 113,
    

        /*
         * Exception declarations (:rmlink:`11.1`).
         *
         * This node type has no derivation.
         */
        ada_exception_decl = 114,
    

        /*
         * Exception handler (:rmlink:`11.2`).
         *
         * This node type has no derivation.
         */
        ada_exception_handler = 115,
    

        /*
         * Declaration for the controlling variable in a ``for`` loop
         * (:rmlink:`5.5`).
         *
         * This node type has no derivation.
         */
        ada_for_loop_var_decl = 116,
    

        /* generic_decl (abstract)  */
        /*
         * Base class for generic declarations (:rmlink:`12.1`).
         *
         * Derived nodes: ``ada_generic_package_decl``,
         * ``ada_generic_subp_decl``
         */
    

        /*
         * Generic package declaration (:rmlink:`12.1`).
         *
         * This node type has no derivation.
         */
        ada_generic_package_decl = 117,
    

        /*
         * Generic subprogram declaration (:rmlink:`12.1`).
         *
         * This node type has no derivation.
         */
        ada_generic_subp_decl = 118,
    

        /* generic_instantiation (abstract)  */
        /*
         * Instantiations of generics (:rmlink:`12.3`).
         *
         * Derived nodes: ``ada_generic_package_instantiation``,
         * ``ada_generic_subp_instantiation``
         */
    

        /*
         * Instantiations of a generic package.
         *
         * This node type has no derivation.
         */
        ada_generic_package_instantiation = 119,
    

        /*
         * Instantiations of a generic subprogram .
         *
         * This node type has no derivation.
         */
        ada_generic_subp_instantiation = 120,
    

        /* generic_renaming_decl (abstract)  */
        /*
         * Base node for all generic renaming declarations (:rmlink:`8.5.5`).
         *
         * Derived nodes: ``ada_generic_package_renaming_decl``,
         * ``ada_generic_subp_renaming_decl``
         */
    

        /*
         * Declaration for a generic package renaming (:rmlink:`8.5.5`).
         *
         * This node type has no derivation.
         */
        ada_generic_package_renaming_decl = 121,
    

        /*
         * Declaration for a generic subprogram renaming.
         *
         * This node type has no derivation.
         */
        ada_generic_subp_renaming_decl = 122,
    

        /*
         * Declaration for a code label (:rmlink:`5.1`).
         *
         * This node type has no derivation.
         */
        ada_label_decl = 123,
    

        /*
         * BasicDecl that is always the declaration inside a named statement.
         *
         * This node type has no derivation.
         */
        ada_named_stmt_decl = 124,
    

        /*
         * Declaration for a static constant number (:rmlink:`3.3.2`).
         *
         * This node type has no derivation.
         */
        ada_number_decl = 125,
    

        /*
         * Base class for Ada object declarations (:rmlink:`3.3.1`). Ada object
         * declarations are variables/constants declarations that can be
         * declared in any declarative scope.
         *
         * Derived nodes: ``ada_extended_return_stmt_object_decl``,
         * ``ada_no_type_object_renaming_decl``
         */
        ada_object_decl = 126,
    

        /*
         * Object declaration that is part of an extended return statement
         * (:rmlink:`6.5`).
         *
         * This node type has no derivation.
         */
        ada_extended_return_stmt_object_decl = 127,
    

        /*
         * Object declaration without subtype indication. This node has been
         * introduced to cover a special case for ``ObjectDecl``, where
         * ``type_expr`` is made optional (AI12-0275), and therefore cannot fit
         * in an ``ObjectDecl``.
         *
         * This node type has no derivation.
         */
        ada_no_type_object_renaming_decl = 128,
    

        /*
         * Declaration for a package renaming (:rmlink:`8.5.3`).
         *
         * This node type has no derivation.
         */
        ada_package_renaming_decl = 129,
    

        /*
         * Declaration for a single protected object (:rmlink:`9.4`).
         *
         * This node type has no derivation.
         */
        ada_single_protected_decl = 130,
    

        /*
         * Declaration for a single task (:rmlink:`9.1`).
         *
         * This node type has no derivation.
         */
        ada_single_task_decl = 131,
    

        /*
         * SyntheticObjectDecl is a declaration that holds a virtual object.
         * This is for example used in type predicates to refer to an object of
         * the enclosing type, as in:
         *
         * .. code::
         *
         *    subtype Odd is Natural with
         *       Dynamic_Predicate => Odd mod 2 = 1;
         *
         * where we have to create an object named ``Odd``, and of type ``Odd``
         * so that the name in the aspect expression refers to it and can be
         * properly resolved to the type identifier.
         *
         * This node has no existance in the Ada RM, it's only used for
         * internal name resolution purposes.
         *
         * This node type has no derivation.
         */
        ada_synthetic_object_decl = 132,
    

        /*
         * Alternative in a ``case`` statement (``when ... => ...``).
         *
         * This node type has no derivation.
         */
        ada_case_stmt_alternative = 133,
    

        /*
         * Root node for all Ada analysis units (:rmlink:`10.1.1`).
         *
         * This node type has no derivation.
         */
        ada_compilation_unit = 134,
    

        /*
         * Representation clause for a single component (:rmlink:`13.5.1`).
         *
         * This node type has no derivation.
         */
        ada_component_clause = 135,
    

        /*
         * Definition for a component (:rmlink:`3.6`).
         *
         * This node type has no derivation.
         */
        ada_component_def = 136,
    

        /* constant_node (abstract)  */
        /*
         * Qualifier for the ``constant`` keyword.
         *
         * Derived nodes: ``ada_constant_absent``, ``ada_constant_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_constant_absent = 137,
    

        /*
         * This node type has no derivation.
         */
        ada_constant_present = 138,
    

        /* constraint (abstract)  */
        /*
         * Base class for type constraints (:rmlink:`3.2.2`).
         *
         * Derived nodes: ``ada_composite_constraint``,
         * ``ada_delta_constraint``, ``ada_digits_constraint``,
         * ``ada_range_constraint``
         */
    

        /*
         * Constraint for a composite type (:rmlink:`3.6.1`). Due to
         * ambiguities in the Ada grammar, this could be either a list of index
         * constraints, if the owning type is an array type, or a list of
         * discriminant constraints, if the owning type is a discriminated
         * record type.
         *
         * This node type has no derivation.
         */
        ada_composite_constraint = 139,
    

        /*
         * Delta and range type constraint (:rmlink:`J.3`).
         *
         * This node type has no derivation.
         */
        ada_delta_constraint = 140,
    

        /*
         * Digits and range type constraint (:rmlink:`3.5.9`).
         *
         * This node type has no derivation.
         */
        ada_digits_constraint = 141,
    

        /*
         * Range-based type constraint (:rmlink:`3.5`).
         *
         * This node type has no derivation.
         */
        ada_range_constraint = 142,
    

        /*
         * List of declarations (:rmlink:`3.11`).
         *
         * Derived nodes: ``ada_private_part``, ``ada_public_part``
         */
        ada_declarative_part = 143,
    

        /*
         * List of declarations in a private part.
         *
         * This node type has no derivation.
         */
        ada_private_part = 144,
    

        /*
         * List of declarations in a public part.
         *
         * This node type has no derivation.
         */
        ada_public_part = 145,
    

        /*
         * ``elsif`` block, part of an ``if`` expression.
         *
         * This node type has no derivation.
         */
        ada_elsif_expr_part = 146,
    

        /*
         * ``elsif`` part in an ``if`` statement block.
         *
         * This node type has no derivation.
         */
        ada_elsif_stmt_part = 147,
    

        /* expr (abstract)  */
        /*
         * Base class for expressions (:rmlink:`4.4`).
         *
         * Derived nodes: ``ada_abstract_state_decl_expr``, ``ada_allocator``,
         * ``ada_base_aggregate``, ``ada_bin_op``, ``ada_box_expr``,
         * ``ada_case_expr_alternative``, ``ada_concat_op``,
         * ``ada_concat_operand``, ``ada_cond_expr``, ``ada_contract_cases``,
         * ``ada_decl_expr``, ``ada_format_string_literal``,
         * ``ada_membership_expr``, ``ada_name``, ``ada_paren_expr``,
         * ``ada_quantified_expr``, ``ada_raise_expr``, ``ada_un_op``
         */
    

        /*
         * Directly corresponds to the right-hand side of the Abstract_State
         * aspect. Only exists because the RHS of an AspectAssoc must be an
         * expression: the actual logic is in AbstractStateDecl.
         *
         * This node type has no derivation.
         */
        ada_abstract_state_decl_expr = 148,
    

        /*
         * Allocator expression (``new ...``) (:rmlink:`4.8`).
         *
         * This node type has no derivation.
         */
        ada_allocator = 149,
    

        /* base_aggregate (abstract)  */
        /*
         * Base class for aggregates (:rmlink:`4.3`).
         *
         * Derived nodes: ``ada_aggregate``, ``ada_delta_aggregate``,
         * ``ada_null_record_aggregate``
         */
    

        /*
         * Aggregate that is not a ``null record`` aggregate (:rmlink:`4.3`).
         *
         * Derived nodes: ``ada_bracket_aggregate``
         */
        ada_aggregate = 150,
    

        /*
         * Bracket array or container aggregate (Ada 2020, :rmlink:`4.3`).
         *
         * This node type has no derivation.
         */
        ada_bracket_aggregate = 151,
    

        /*
         * Aggregate for delta aggregate (Ada 2022, :rmlink:`4.3`).
         *
         * Derived nodes: ``ada_bracket_delta_aggregate``
         */
        ada_delta_aggregate = 152,
    

        /*
         * Bracket delta aggregate (Ada 2020, :rmlink:`4.3`).
         *
         * This node type has no derivation.
         */
        ada_bracket_delta_aggregate = 153,
    

        /*
         * Aggregate for ``null record`` (:rmlink:`4.3`).
         *
         * This node type has no derivation.
         */
        ada_null_record_aggregate = 154,
    

        /*
         * Binary expression.
         *
         * This encompasses several ARM expressions, because it is used for
         * every binary expression in Ada, all documented in ::rmlink:`4.4`.
         *
         * Derived nodes: ``ada_relation_op``
         */
        ada_bin_op = 155,
    

        /*
         * Binary operation that compares two value, producing a boolean
         * (:rmlink:`4.4`).
         *
         * This node type has no derivation.
         */
        ada_relation_op = 156,
    

        /*
         * Box expression (``<>``).
         *
         * This is not an expression per-se in Ada, but treating it as one
         * helps us keep coherent types in some cases, like aggregates
         * expressions.
         *
         * This node type has no derivation.
         */
        ada_box_expr = 157,
    

        /*
         * Alternative in a ``case`` expression (``when ... => ...``).
         *
         * This node type has no derivation.
         */
        ada_case_expr_alternative = 158,
    

        /*
         * Concatenation expression.
         *
         * Since concatenation expression can be huge in practice, this node
         * handles them as a list of operands rather than a deep tree of binary
         * operators, in order to avoid crashes while parsing of running name
         * resolution on such huge expression.
         *
         * The purpose of this node is to replace the arbitrarily too deep tree
         * of binary operators (which can lead to a stack overflow), as for
         * example with ``"A & B & C & D & E"``:
         *
         * .. code::
         *
         *    BinOp(
         *      Binop(
         *        BinOp(
         *          BinOp(A, &, B), & , C), &, D), &, E)
         *
         * by a single operator, handling a list of operands that can be
         * processed without having to perform deep recursions:
         *
         * .. code::
         *
         *    ConcatOp(A,
         *      ConcatOperand(&, B),
         *      ConcatOperand(&, C),
         *      ConcatOperand(&, D),
         *      ConcatOperand(&, E))
         *
         * This node type has no derivation.
         */
        ada_concat_op = 159,
    

        /*
         * A concatenation operator and its RHS operand.
         *
         * This node is used to represent the tuple ("&", operand) used by the
         * ``ConcatOp`` node to store its ``other_operands`` list.
         *
         * This node type has no derivation.
         */
        ada_concat_operand = 160,
    

        /* cond_expr (abstract)  */
        /*
         * Base class for a conditional expressions (:rmlink:`4.5.7`).
         *
         * Derived nodes: ``ada_case_expr``, ``ada_if_expr``
         */
    

        /*
         * ``case`` expression (:rmlink:`4.5.7`).
         *
         * This node type has no derivation.
         */
        ada_case_expr = 161,
    

        /*
         * ``if`` expression (:rmlink`4.5.7`).
         *
         * This node type has no derivation.
         */
        ada_if_expr = 162,
    

        /*
         * List of associations for the ``Contract_Case`` aspect.
         *
         * Contract cases is a non standard Ada extension that's mainly useful
         * in SPARK. See the SPARK RM for more details.
         *
         * This node type has no derivation.
         */
        ada_contract_cases = 163,
    

        /*
         * Declare expression (Ada 2022, :rmlink:`4.5.9`).
         *
         * This node type has no derivation.
         */
        ada_decl_expr = 164,
    

        /*
         * Interpolated string expression.
         *
         * See :gnat_rm:`string-interpolation` for more details.
         *
         * This node type has no derivation.
         */
        ada_format_string_literal = 165,
    

        /*
         * Represent a membership test (in/not in operators) (:rmlink:`4.4`).
         *
         * Note that we don't consider them as binary operators since multiple
         * expressions on the right hand side are allowed.
         *
         * This node type has no derivation.
         */
        ada_membership_expr = 166,
    

        /* name (abstract)  */
        /*
         * Base class for names (:rmlink:`4.1`).
         *
         * Derived nodes: ``ada_array_subcomponent_choice_name``,
         * ``ada_attribute_ref``, ``ada_call_expr``, ``ada_defining_name``,
         * ``ada_discrete_subtype_name``, ``ada_dotted_name``,
         * ``ada_end_name``, ``ada_explicit_deref``, ``ada_qual_expr``,
         * ``ada_reduce_attribute_ref``, ``ada_single_tok_node``,
         * ``ada_synthetic_identifier``, ``ada_target_name``,
         * ``ada_update_attribute_ref``
         */
    

        /*
         * Name for an array subcomponent choice of a deep delta aggregate.
         *
         * This node type has no derivation.
         */
        ada_array_subcomponent_choice_name = 167,
    

        /*
         * Expression to reference an attribute (:rmlink:`4.1.4`).
         *
         * This node type has no derivation.
         */
        ada_attribute_ref = 168,
    

        /*
         * Represent a syntactic call expression.
         *
         * At the semantic level, this can be either a subprogram call, an
         * array subcomponent access expression, an array slice or a type
         * conversion, all described in :rmlink:`4.1`, except for subprogram
         * call statements, described in :rmlink:`6.4`.
         *
         * This node type has no derivation.
         */
        ada_call_expr = 169,
    

        /*
         * Name that defines an entity (:rmlink:`3.1`).
         *
         * Derived nodes: ``ada_synthetic_defining_name``
         */
        ada_defining_name = 170,
    

        /*
         * Synthetic DefiningName.
         *
         * This node type has no derivation.
         */
        ada_synthetic_defining_name = 171,
    

        /*
         * Subtype name for membership test expressions (:rmlink:`3.6`).
         *
         * This node type has no derivation.
         */
        ada_discrete_subtype_name = 172,
    

        /*
         * Name to select a suffix in a prefix (:rmlink:`4.1.3`).
         *
         * This node type has no derivation.
         */
        ada_dotted_name = 173,
    

        /*
         * self name in ``end ...;`` syntactic constructs.
         *
         * This node type has no derivation.
         */
        ada_end_name = 174,
    

        /*
         * Explicit dereference expression (``.all``) (:rmlink:`4.1`).
         *
         * This node type has no derivation.
         */
        ada_explicit_deref = 175,
    

        /*
         * Qualified expression (``...'(...)``) .(:rmlink:`4.7`).
         *
         * This node type has no derivation.
         */
        ada_qual_expr = 176,
    

        /*
         * Reduction expression (``Reduce`` attribute). Ada 2022, RM 4.5.10.
         *
         * This node type has no derivation.
         */
        ada_reduce_attribute_ref = 177,
    

        /* single_tok_node (abstract)  */
        /*
         * Base class for nodes that are made up of a single token.
         *
         * Derived nodes: ``ada_base_id``, ``ada_null_literal``,
         * ``ada_num_literal``
         */
    

        /* base_id (abstract)  */
        /*
         * Base class for identifiers.
         *
         * Derived nodes: ``ada_char_literal``, ``ada_identifier``, ``ada_op``,
         * ``ada_string_literal``
         */
    

        /*
         * Character literal (:rmlink:`4.1`).
         *
         * This node type has no derivation.
         */
        ada_char_literal = 178,
    

        /*
         * Regular identifier (:rmlink:`2.3`).
         *
         * This node type has no derivation.
         */
        ada_identifier = 179,
    

        /* op (abstract)  */
        /*
         * Operation in a binary expression.
         *
         * Note that the ARM does not consider "double_dot" ("..") as a binary
         * operator, but we process it this way here anyway to keep things
         * simple.
         *
         * Derived nodes: ``ada_op_abs``, ``ada_op_and_then``, ``ada_op_and``,
         * ``ada_op_concat``, ``ada_op_div``, ``ada_op_double_dot``,
         * ``ada_op_eq``, ``ada_op_gt``, ``ada_op_gte``, ``ada_op_in``,
         * ``ada_op_lt``, ``ada_op_lte``, ``ada_op_minus``, ``ada_op_mod``,
         * ``ada_op_mult``, ``ada_op_neq``, ``ada_op_not_in``, ``ada_op_not``,
         * ``ada_op_or_else``, ``ada_op_or``, ``ada_op_plus``, ``ada_op_pow``,
         * ``ada_op_rem``, ``ada_op_xor``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_op_abs = 180,
    

        /*
         * This node type has no derivation.
         */
        ada_op_and = 181,
    

        /*
         * This node type has no derivation.
         */
        ada_op_and_then = 182,
    

        /*
         * This node type has no derivation.
         */
        ada_op_concat = 183,
    

        /*
         * This node type has no derivation.
         */
        ada_op_div = 184,
    

        /*
         * This node type has no derivation.
         */
        ada_op_double_dot = 185,
    

        /*
         * This node type has no derivation.
         */
        ada_op_eq = 186,
    

        /*
         * This node type has no derivation.
         */
        ada_op_gt = 187,
    

        /*
         * This node type has no derivation.
         */
        ada_op_gte = 188,
    

        /*
         * This node type has no derivation.
         */
        ada_op_in = 189,
    

        /*
         * This node type has no derivation.
         */
        ada_op_lt = 190,
    

        /*
         * This node type has no derivation.
         */
        ada_op_lte = 191,
    

        /*
         * This node type has no derivation.
         */
        ada_op_minus = 192,
    

        /*
         * This node type has no derivation.
         */
        ada_op_mod = 193,
    

        /*
         * This node type has no derivation.
         */
        ada_op_mult = 194,
    

        /*
         * This node type has no derivation.
         */
        ada_op_neq = 195,
    

        /*
         * This node type has no derivation.
         */
        ada_op_not = 196,
    

        /*
         * This node type has no derivation.
         */
        ada_op_not_in = 197,
    

        /*
         * This node type has no derivation.
         */
        ada_op_or = 198,
    

        /*
         * This node type has no derivation.
         */
        ada_op_or_else = 199,
    

        /*
         * This node type has no derivation.
         */
        ada_op_plus = 200,
    

        /*
         * This node type has no derivation.
         */
        ada_op_pow = 201,
    

        /*
         * This node type has no derivation.
         */
        ada_op_rem = 202,
    

        /*
         * This node type has no derivation.
         */
        ada_op_xor = 203,
    

        /*
         * String literal (:rmlink:`2.6`).
         *
         * This node type has no derivation.
         */
        ada_string_literal = 204,
    

        /*
         * The ``null`` literal (:rmlink:`4.4`).
         *
         * This node type has no derivation.
         */
        ada_null_literal = 205,
    

        /* num_literal (abstract)  */
        /*
         * Base class for number literals (:rmlink:`2.4`).
         *
         * Derived nodes: ``ada_int_literal``, ``ada_real_literal``
         */
    

        /*
         * Literal for an integer (:rmlink:`2.4`).
         *
         * This node type has no derivation.
         */
        ada_int_literal = 206,
    

        /*
         * Literal for a real number (:rmlink:`2.4`).
         *
         * This node type has no derivation.
         */
        ada_real_literal = 207,
    

        /*
         * Synthetic identifier.
         *
         * This node type has no derivation.
         */
        ada_synthetic_identifier = 208,
    

        /*
         * Name for Ada 2020 ``@`` (:rmlink:`5.2.1`).
         *
         * This node type has no derivation.
         */
        ada_target_name = 209,
    

        /*
         * Reference to the ``Update`` attribute, which is a non standard GNAT
         * attribute.
         *
         * This node type has no derivation.
         */
        ada_update_attribute_ref = 210,
    

        /*
         * Parenthesized expression.
         *
         * This node type has no derivation.
         */
        ada_paren_expr = 211,
    

        /*
         * Quantified expression (:rmlink:`4.5.8`).
         *
         * This node type has no derivation.
         */
        ada_quantified_expr = 212,
    

        /*
         * Expression to raise an exception (:rmlink:`4.4`).
         *
         * This node type has no derivation.
         */
        ada_raise_expr = 213,
    

        /*
         * Unary expression.
         *
         * This encompasses several ARM expressions, because it is used for
         * every unary operator in Ada. Those expressions are all documented in
         * :rmlink:`4.4`.
         *
         * This node type has no derivation.
         */
        ada_un_op = 214,
    

        /*
         * Represent the ``when ...`` filter after a for loop specification.
         * This class has no RM existence, it is used internally to wrap the
         * filtering expression, so as to have a dedicated name resolution
         * entry point for it and make sure it is resolved separatly from the
         * ``ForLoopSpec`` itself (which it cannot influence anyway).
         *
         * This node type has no derivation.
         */
        ada_for_loop_iter_filter = 215,
    

        /*
         * Chunk of a format string literal.
         *
         * This node type has no derivation.
         */
        ada_format_string_chunk = 216,
    

        /* format_string_tok_node (abstract)  */
        /*
         * Node holding a format string token.
         *
         * Derived nodes: ``ada_format_string_tok_end``,
         * ``ada_format_string_tok_mid``, ``ada_format_string_tok_start``
         */
    

        /*
         * Node holding a formatting "end" token.
         *
         * This node type has no derivation.
         */
        ada_format_string_tok_end = 217,
    

        /*
         * Node holding a formatting "middle" token.
         *
         * This node type has no derivation.
         */
        ada_format_string_tok_mid = 218,
    

        /*
         * Node holding a formatting "start" token.
         *
         * Derived nodes: ``ada_format_string_tok_string``
         */
        ada_format_string_tok_start = 219,
    

        /*
         * Node holding a formatting "string" token. This token is used when
         * the corresponding interpolated string doesn't have any expression to
         * expand.
         *
         * This node type has no derivation.
         */
        ada_format_string_tok_string = 220,
    

        /*
         * List of statements, with optional exception handlers
         * (:rmlink:`11.2`).
         *
         * This node type has no derivation.
         */
        ada_handled_stmts = 221,
    

        /* interface_kind (abstract)  */
        /*
         * Kind of interface type.
         *
         * Derived nodes: ``ada_interface_kind_limited``,
         * ``ada_interface_kind_protected``,
         * ``ada_interface_kind_synchronized``, ``ada_interface_kind_task``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_interface_kind_limited = 222,
    

        /*
         * This node type has no derivation.
         */
        ada_interface_kind_protected = 223,
    

        /*
         * This node type has no derivation.
         */
        ada_interface_kind_synchronized = 224,
    

        /*
         * This node type has no derivation.
         */
        ada_interface_kind_task = 225,
    

        /* iter_type (abstract)  */
        /*
         * Iteration type for ``for`` loops.
         *
         * Derived nodes: ``ada_iter_type_in``, ``ada_iter_type_of``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_iter_type_in = 226,
    

        /*
         * This node type has no derivation.
         */
        ada_iter_type_of = 227,
    

        /*
         * Library item in a compilation unit (:rmlink:`10.1.1`).
         *
         * This node type has no derivation.
         */
        ada_library_item = 228,
    

        /* limited_node (abstract)  */
        /*
         * Qualifier for the ``limited`` keyword.
         *
         * Derived nodes: ``ada_limited_absent``, ``ada_limited_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_limited_absent = 229,
    

        /*
         * This node type has no derivation.
         */
        ada_limited_present = 230,
    

        /* loop_spec (abstract)  */
        /*
         * Base class for loop specifications (:rmlink:`5.5`).
         *
         * Derived nodes: ``ada_for_loop_spec``, ``ada_while_loop_spec``
         */
    

        /*
         * Specification for a ``for`` loop (:rmlink:`5.5`).
         *
         * This node type has no derivation.
         */
        ada_for_loop_spec = 231,
    

        /*
         * Specification for a ``while`` loop (:rmlink:`5.5`).
         *
         * This node type has no derivation.
         */
        ada_while_loop_spec = 232,
    

        /* mode (abstract)  */
        /*
         * Syntactic indicators for passing modes in formals (:rmlink:`6.1`).
         *
         * Derived nodes: ``ada_mode_default``, ``ada_mode_in_out``,
         * ``ada_mode_in``, ``ada_mode_out``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_mode_default = 233,
    

        /*
         * This node type has no derivation.
         */
        ada_mode_in = 234,
    

        /*
         * This node type has no derivation.
         */
        ada_mode_in_out = 235,
    

        /*
         * This node type has no derivation.
         */
        ada_mode_out = 236,
    

        /*
         * Node that holds several AbstractStateDecl nodes, which is necessary
         * when the Abstract_State aspect is associated with an aggregate in
         * order to declare a list of abstract states.
         *
         * This node type has no derivation.
         */
        ada_multi_abstract_state_decl = 237,
    

        /* not_null (abstract)  */
        /*
         * Qualifier for the ``not null`` keywords.
         *
         * Derived nodes: ``ada_not_null_absent``, ``ada_not_null_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_not_null_absent = 238,
    

        /*
         * This node type has no derivation.
         */
        ada_not_null_present = 239,
    

        /*
         * Placeholder for the ``null`` in lists of components (:rmlink:`3.8`).
         *
         * This node type has no derivation.
         */
        ada_null_component_decl = 240,
    

        /*
         * ``other`` designator.
         *
         * This node type has no derivation.
         */
        ada_others_designator = 241,
    

        /* overriding_node (abstract)  */
        /*
         * Syntactic indicators for subprogram overriding modes.
         *
         * Derived nodes: ``ada_overriding_not_overriding``,
         * ``ada_overriding_overriding``, ``ada_overriding_unspecified``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_overriding_not_overriding = 242,
    

        /*
         * This node type has no derivation.
         */
        ada_overriding_overriding = 243,
    

        /*
         * This node type has no derivation.
         */
        ada_overriding_unspecified = 244,
    

        /*
         * List of parameter specifications.
         *
         * This node type has no derivation.
         */
        ada_params = 245,
    

        /*
         * Holds an AbstractStateDecl between parentheses. Needed to support
         * the syntax:
         *
         * .. code:: ada
         *
         *    package Pkg
         *        with Abstract_State => (A, (B with Some_Aspect))
         *
         * This node type has no derivation.
         */
        ada_paren_abstract_state_decl = 246,
    

        /* pp_directive (abstract)  */
        /*
         * Base node for all preprocessor directives.
         *
         * Derived nodes: ``ada_pp_else_directive``,
         * ``ada_pp_elsif_directive``, ``ada_pp_end_if_directive``,
         * ``ada_pp_if_directive``
         */
    

        /*
         * ``else`` preprocessor directive.
         *
         * This node type has no derivation.
         */
        ada_pp_else_directive = 247,
    

        /*
         * ``elsif ... [then]`` preprocessor directive.
         *
         * This node type has no derivation.
         */
        ada_pp_elsif_directive = 248,
    

        /*
         * ``end if;`` preprocessor directive.
         *
         * This node type has no derivation.
         */
        ada_pp_end_if_directive = 249,
    

        /*
         * ``if ... [then]`` preprocessor directive.
         *
         * This node type has no derivation.
         */
        ada_pp_if_directive = 250,
    

        /*
         * ``then`` keyword in preprocessor directives.
         *
         * This node type has no derivation.
         */
        ada_pp_then_kw = 251,
    

        /*
         * Class for pragmas (:rmlink:`2.8`). Pragmas are compiler directives,
         * that can be language or compiler defined.
         *
         * This node type has no derivation.
         */
        ada_pragma_node = 252,
    

        /* private_node (abstract)  */
        /*
         * Qualifier for the ``private`` keyword.
         *
         * Derived nodes: ``ada_private_absent``, ``ada_private_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_private_absent = 253,
    

        /*
         * This node type has no derivation.
         */
        ada_private_present = 254,
    

        /*
         * Type definition for a protected object (:rmlink:`9.4`).
         *
         * This node type has no derivation.
         */
        ada_protected_def = 255,
    

        /* protected_node (abstract)  */
        /*
         * Qualifier for the ``protected`` keyword.
         *
         * Derived nodes: ``ada_protected_absent``, ``ada_protected_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_protected_absent = 256,
    

        /*
         * This node type has no derivation.
         */
        ada_protected_present = 257,
    

        /* quantifier (abstract)  */
        /*
         * Type for quantified expressions.
         *
         * Derived nodes: ``ada_quantifier_all``, ``ada_quantifier_some``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_quantifier_all = 258,
    

        /*
         * This node type has no derivation.
         */
        ada_quantifier_some = 259,
    

        /*
         * Range specification (:rmlink:`3.5.7`).
         *
         * This node type has no derivation.
         */
        ada_range_spec = 260,
    

        /*
         * Renaming clause, used everywhere renamings are valid.
         *
         * Derived nodes: ``ada_synthetic_renaming_clause``
         */
        ada_renaming_clause = 261,
    

        /*
         * Synthetic renaming clause. Used to synthesize object decls with
         * renamings. (See to_anonymous_object_decl).
         *
         * This node type has no derivation.
         */
        ada_synthetic_renaming_clause = 262,
    

        /* reverse_node (abstract)  */
        /*
         * Qualifier for the ``reverse`` keyword.
         *
         * Derived nodes: ``ada_reverse_absent``, ``ada_reverse_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_reverse_absent = 263,
    

        /*
         * This node type has no derivation.
         */
        ada_reverse_present = 264,
    

        /*
         * Alternative part in a ``select`` statements block (:rmlink:`9.7`).
         *
         * This node type has no derivation.
         */
        ada_select_when_part = 265,
    

        /* stmt (abstract)  */
        /*
         * Bass class for statements (:rmlink:`5.1`).
         *
         * Derived nodes: ``ada_composite_stmt``, ``ada_error_stmt``,
         * ``ada_simple_stmt``
         */
    

        /* composite_stmt (abstract)  */
        /*
         * Base class for composite statements (:rmlink:`5.1`).
         *
         * Derived nodes: ``ada_accept_stmt``, ``ada_base_loop_stmt``,
         * ``ada_block_stmt``, ``ada_case_stmt``, ``ada_extended_return_stmt``,
         * ``ada_if_stmt``, ``ada_named_stmt``, ``ada_select_stmt``
         */
    

        /*
         * ``accept`` statement (:rmlink:`9.5.2`).
         *
         * Derived nodes: ``ada_accept_stmt_with_stmts``
         */
        ada_accept_stmt = 266,
    

        /*
         * Extended ``accept`` statement (:rmlink:`9.5.2`).
         *
         * This node type has no derivation.
         */
        ada_accept_stmt_with_stmts = 267,
    

        /* base_loop_stmt (abstract)  */
        /*
         * Base class for loop statements (:rmlink:`5.5`).
         *
         * Derived nodes: ``ada_for_loop_stmt``, ``ada_loop_stmt``,
         * ``ada_while_loop_stmt``
         */
    

        /*
         * Statement for ``for`` loops (``for ... loop ... end loop;``)
         * (:rmlink:`5.5`).
         *
         * This node type has no derivation.
         */
        ada_for_loop_stmt = 268,
    

        /*
         * Statement for simple loops (``loop ... end loop;``) (:rmlink:`5.5`).
         *
         * This node type has no derivation.
         */
        ada_loop_stmt = 269,
    

        /*
         * Statement for ``while`` loops (``while ... loop ... end loop;``)
         * (:rmlink:`5.5`).
         *
         * This node type has no derivation.
         */
        ada_while_loop_stmt = 270,
    

        /* block_stmt (abstract)  */
        /*
         * Base class for statement blocks (:rmlink:`5.6`).
         *
         * Derived nodes: ``ada_begin_block``, ``ada_decl_block``
         */
    

        /*
         * Statement block with no declarative part (:rmlink:`5.6`).
         *
         * This node type has no derivation.
         */
        ada_begin_block = 271,
    

        /*
         * Statement block with a declarative part (:rmlink:`5.6`).
         *
         * This node type has no derivation.
         */
        ada_decl_block = 272,
    

        /*
         * ``case`` statement (:rmlink:`5.4`).
         *
         * This node type has no derivation.
         */
        ada_case_stmt = 273,
    

        /*
         * Extended ``return`` statement (:rmlink:`6.5`).
         *
         * This node type has no derivation.
         */
        ada_extended_return_stmt = 274,
    

        /*
         * ``if`` statement block (:rmlink:`5.3`).
         *
         * This node type has no derivation.
         */
        ada_if_stmt = 275,
    

        /*
         * Wrapper class, used for composite statements that can be named
         * (declare blocks, loops). This allows to both have a BasicDecl for
         * the named entity declared, and a CompositeStmt for the statement
         * hierarchy.
         *
         * This node type has no derivation.
         */
        ada_named_stmt = 276,
    

        /*
         * ``select`` statements block (:rmlink:`9.7`).
         *
         * This node type has no derivation.
         */
        ada_select_stmt = 277,
    

        /*
         * Placeholder node for syntax errors in lists of statements.
         *
         * This node type has no derivation.
         */
        ada_error_stmt = 278,
    

        /* simple_stmt (abstract)  */
        /*
         * Base class for simple statements (:rmlink:`5.1`).
         *
         * Derived nodes: ``ada_abort_stmt``, ``ada_assign_stmt``,
         * ``ada_call_stmt``, ``ada_delay_stmt``, ``ada_exit_stmt``,
         * ``ada_goto_stmt``, ``ada_label``, ``ada_null_stmt``,
         * ``ada_raise_stmt``, ``ada_requeue_stmt``, ``ada_return_stmt``,
         * ``ada_simple_decl_stmt``, ``ada_terminate_alternative``
         */
    

        /*
         * ``abort`` statement (:rmlink:`9.8`).
         *
         * This node type has no derivation.
         */
        ada_abort_stmt = 279,
    

        /*
         * Statement for assignments (:rmlink:`5.2`).
         *
         * This node type has no derivation.
         */
        ada_assign_stmt = 280,
    

        /*
         * Statement for entry or procedure calls (:rmlink:`6.4`).
         *
         * This node type has no derivation.
         */
        ada_call_stmt = 281,
    

        /*
         * ``delay`` statement (:rmlink:`9.6`).
         *
         * This node type has no derivation.
         */
        ada_delay_stmt = 282,
    

        /*
         * ``exit`` statement (:rmlink:`5.7`).
         *
         * This node type has no derivation.
         */
        ada_exit_stmt = 283,
    

        /*
         * ``goto`` statement (:rmlink:`5.8`).
         *
         * This node type has no derivation.
         */
        ada_goto_stmt = 284,
    

        /*
         * Statement to declare a code label (:rmlink:`5.1`).
         *
         * This node type has no derivation.
         */
        ada_label = 285,
    

        /*
         * ``null;`` statement (:rmlink:`5.1`).
         *
         * This node type has no derivation.
         */
        ada_null_stmt = 286,
    

        /*
         * ``raise`` statement (:rmlink:`11.3`).
         *
         * This node type has no derivation.
         */
        ada_raise_stmt = 287,
    

        /*
         * ``requeue`` statement (:rmlink:`9.5.4`).
         *
         * This node type has no derivation.
         */
        ada_requeue_stmt = 288,
    

        /*
         * ``return`` statement (:rmlink:`6.5`).
         *
         * This node type has no derivation.
         */
        ada_return_stmt = 289,
    

        /*
         * Statement wrapping a simple object declaration.
         *
         * This node type has no derivation.
         */
        ada_simple_decl_stmt = 290,
    

        /*
         * ``terminate`` alternative in a ``select`` statement (:rmlink:`9.7`).
         *
         * This node type has no derivation.
         */
        ada_terminate_alternative = 291,
    

        /* subp_kind (abstract)  */
        /*
         * Qualifier for a subprogram kind.
         *
         * Derived nodes: ``ada_subp_kind_function``,
         * ``ada_subp_kind_procedure``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_subp_kind_function = 292,
    

        /*
         * This node type has no derivation.
         */
        ada_subp_kind_procedure = 293,
    

        /*
         * Subunit (``separate``) (:rmlink:`10.1.3`).
         *
         * This node type has no derivation.
         */
        ada_subunit = 294,
    

        /* synchronized_node (abstract)  */
        /*
         * Qualifier for the ``synchronized`` keyword.
         *
         * Derived nodes: ``ada_synchronized_absent``,
         * ``ada_synchronized_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_synchronized_absent = 295,
    

        /*
         * This node type has no derivation.
         */
        ada_synchronized_present = 296,
    

        /* tagged_node (abstract)  */
        /*
         * Qualifier for the ``tagged`` keyword.
         *
         * Derived nodes: ``ada_tagged_absent``, ``ada_tagged_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_tagged_absent = 297,
    

        /*
         * This node type has no derivation.
         */
        ada_tagged_present = 298,
    

        /*
         * Type definition for a task type (:rmlink:`9.1`).
         *
         * This node type has no derivation.
         */
        ada_task_def = 299,
    

        /*
         * Synthetic node that contains the lazy fields for the attribute
         * subprograms of a given type. The lazy fields are not directly on the
         * BaseTypeDecl node itself to minimize its size in memory: with this
         * indirection, a type for which no function attribute is ever
         * synthesized will not waste any memory.
         *
         * This node type has no derivation.
         */
        ada_type_attributes_repository = 300,
    

        /* type_def (abstract)  */
        /*
         * Base class for type definitions (:rmlink:`3.2.1`).
         *
         * Derived nodes: ``ada_access_def``, ``ada_array_type_def``,
         * ``ada_derived_type_def``, ``ada_enum_type_def``,
         * ``ada_formal_discrete_type_def``, ``ada_interface_type_def``,
         * ``ada_mod_int_type_def``, ``ada_private_type_def``,
         * ``ada_real_type_def``, ``ada_record_type_def``,
         * ``ada_signed_int_type_def``
         */
    

        /* access_def (abstract)  */
        /*
         * Base class for access type definitions (:rmlink:`3.10`).
         *
         * Derived nodes: ``ada_access_to_subp_def``,
         * ``ada_base_type_access_def``
         */
    

        /*
         * Type definition for accesses to subprograms (:rmlink:`3.10`).
         *
         * This node type has no derivation.
         */
        ada_access_to_subp_def = 301,
    

        /* base_type_access_def (abstract)  */
        /*
         * Base class for access type definitions (:rmlink:`3.10`).
         *
         * Derived nodes: ``ada_anonymous_type_access_def``,
         * ``ada_type_access_def``
         */
    

        /*
         * Synthetic type access, that will directly reference a type decl. It
         * is used to generate synthetic anonymous access types.
         *
         * This node type has no derivation.
         */
        ada_anonymous_type_access_def = 302,
    

        /*
         * Syntactic type definition for accesses.
         *
         * This node type has no derivation.
         */
        ada_type_access_def = 303,
    

        /*
         * Type definition for an array (:rmlink:`3.6`).
         *
         * This node type has no derivation.
         */
        ada_array_type_def = 304,
    

        /*
         * Type definition for a derived type (:rmlink:`3.4`).
         *
         * This node type has no derivation.
         */
        ada_derived_type_def = 305,
    

        /*
         * Type definition for enumerations (:rmlink:`3.5.1`).
         *
         * This node type has no derivation.
         */
        ada_enum_type_def = 306,
    

        /*
         * Type definition for discrete types in generic formals
         * (:rmlink:`12.5.2`).
         *
         * This node type has no derivation.
         */
        ada_formal_discrete_type_def = 307,
    

        /*
         * Type definition for an interface (:rmlink:`3.9.4`).
         *
         * This node type has no derivation.
         */
        ada_interface_type_def = 308,
    

        /*
         * Type definition for a modular integer type (:rmlink:`3.5.4`).
         *
         * This node type has no derivation.
         */
        ada_mod_int_type_def = 309,
    

        /*
         * Type definition for a private type.
         *
         * Libadalang diverges from the ARM here, treating private types like
         * regular type declarations that have an embedded type definition.
         * This type definition hence corresponds to :rmlink:`7.3`.
         *
         * This node type has no derivation.
         */
        ada_private_type_def = 310,
    

        /* real_type_def (abstract)  */
        /*
         * Type definition for real numbers (:rmlink:`3.5.6`).
         *
         * Derived nodes: ``ada_decimal_fixed_point_def``,
         * ``ada_floating_point_def``, ``ada_ordinary_fixed_point_def``
         */
    

        /*
         * Type definition for decimal fixed-point numbers (:rmlink:`3.5.9`).
         *
         * This node type has no derivation.
         */
        ada_decimal_fixed_point_def = 311,
    

        /*
         * Type definition for floating-point numbers (:rmlink:`3.5.7`).
         *
         * This node type has no derivation.
         */
        ada_floating_point_def = 312,
    

        /*
         * Type definition for ordinary fixed-point numbers (:rmlink:`3.5.9`).
         *
         * This node type has no derivation.
         */
        ada_ordinary_fixed_point_def = 313,
    

        /*
         * Type definition for a record (:rmlink:`3.8`).
         *
         * This node type has no derivation.
         */
        ada_record_type_def = 314,
    

        /*
         * Type definition for a signed integer type (:rmlink:`3.5.4`).
         *
         * This node type has no derivation.
         */
        ada_signed_int_type_def = 315,
    

        /* type_expr (abstract)  */
        /*
         * A type expression is an abstract node that embodies the concept of a
         * reference to a type.
         *
         * Since Ada has both subtype_indications and anonymous (inline) type
         * declarations, a type expression contains one or the other.
         *
         * This node has no ARM correspondence.
         *
         * Derived nodes: ``ada_anonymous_type``,
         * ``ada_enum_lit_synth_type_expr``, ``ada_subtype_indication``,
         * ``ada_synthetic_type_expr``
         */
    

        /*
         * Container for inline anonymous array and access types declarations.
         *
         * This node type has no derivation.
         */
        ada_anonymous_type = 316,
    

        /*
         * Synthetic node. Represents the type expression for an enum literal.
         *
         * This node type has no derivation.
         */
        ada_enum_lit_synth_type_expr = 317,
    

        /*
         * Reference to a type by name (:rmlink:`3.2.2`).
         *
         * Derived nodes: ``ada_constrained_subtype_indication``,
         * ``ada_discrete_subtype_indication``
         */
        ada_subtype_indication = 318,
    

        /*
         * Reference to a type with a range constraint.
         *
         * This node type has no derivation.
         */
        ada_constrained_subtype_indication = 319,
    

        /*
         * Reference to a type with a general constraint.
         *
         * This node type has no derivation.
         */
        ada_discrete_subtype_indication = 320,
    

        /*
         * Synthetic type expression. The designated type is already known at
         * instantiation time and is to be given in the ``target_type`` field.
         *
         * This node type has no derivation.
         */
        ada_synthetic_type_expr = 321,
    

        /*
         * List of unconstrained array indexes.
         *
         * This node type has no derivation.
         */
        ada_unconstrained_array_index = 322,
    

        /* until_node (abstract)  */
        /*
         * Qualifier for the ``until`` keyword.
         *
         * Derived nodes: ``ada_until_absent``, ``ada_until_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_until_absent = 323,
    

        /*
         * This node type has no derivation.
         */
        ada_until_present = 324,
    

        /* use_clause (abstract)  */
        /*
         * Base class for use clauses (:rmlink:`10.1.2`).
         *
         * Derived nodes: ``ada_use_package_clause``, ``ada_use_type_clause``
         */
    

        /*
         * Use clause for packages (:rmlink:`8.4`).
         *
         * This node type has no derivation.
         */
        ada_use_package_clause = 325,
    

        /*
         * Use clause for types (:rmlink:`8.4`).
         *
         * This node type has no derivation.
         */
        ada_use_type_clause = 326,
    

        /*
         * The value sequence of a reduction expression (see
         * ``ReduceAttributeRef``). Ada 2022, RM 4.5.10.
         *
         * This node type has no derivation.
         */
        ada_value_sequence = 327,
    

        /*
         * Single variant in a discriminated type record declaration.
         *
         * This corresponds to a ``when ... => ...`` section in a variant part.
         *
         * This node type has no derivation.
         */
        ada_variant = 328,
    

        /*
         * Variant part in a discriminated type record declaration
         * (:rmlink:`3.8.1`).
         *
         * This corresponds to the whole ``case ... is ... end case;`` block.
         *
         * This node type has no derivation.
         */
        ada_variant_part = 329,
    

        /*
         * With clause (:rmlink:`10.1.2`).
         *
         * This node type has no derivation.
         */
        ada_with_clause = 330,
    

        /* with_private (abstract)  */
        /*
         * Qualifier for the ``private`` keyword in ``with private`` record
         * clauses.
         *
         * Derived nodes: ``ada_with_private_absent``,
         * ``ada_with_private_present``
         */
    

        /*
         * This node type has no derivation.
         */
        ada_with_private_absent = 331,
    

        /*
         * This node type has no derivation.
         */
        ada_with_private_present = 332,
} ada_node_kind_enum;

/*
 * Reference to a symbol. Symbols are owned by analysis contexts, so they must
 * not outlive them. This type exists only in the C API, and roughly wraps the
 * corresponding Ada type (an array fat pointer).
 */
typedef struct {
   uint32_t thin_sym;
   void *table;
} ada_symbol_type;

/*
 * Type to contain Unicode text data.
 */
typedef struct {
   int length;
   int ref_count;
   uint32_t content[1];
} *ada_string_type;

/*
 * Data type for env rebindings. For internal use only.
 */
typedef struct ada_env_rebindings_type__struct *ada_env_rebindings_type;

typedef uint8_t ada_bool;

/* Helper data structures for source location handling.  */

/*
 * Location in a source file. Line and column numbers are one-based.
 */
typedef struct {
    uint32_t line;
    uint16_t column;
} ada_source_location;

/*
 * Location of a span of text in a source file.
 */
typedef struct {
    ada_source_location start;
    ada_source_location end;
} ada_source_location_range;


/*
 * String encoded in UTF-32 (native endianness).
 */
typedef struct {
   /*
 * Address for the content of the string.
 */
    uint32_t *chars;
   /*
 * Size of the string (in characters).
 */
    size_t length;

    int is_allocated;
} ada_text;

/*
 * Arbitrarily large integer.
 */
typedef struct ada_big_integer__struct *ada_big_integer;

/*
 * Kind for this token.
 */
typedef enum {
   
      
      ADA_ABORT = 0
      ,
      ADA_ABS = 1
      ,
      ADA_ACCEPT = 2
      ,
      ADA_ACCESS = 3
      ,
      ADA_ALL = 4
      ,
      ADA_AMP = 5
      ,
      ADA_AND = 6
      ,
      ADA_ARRAY = 7
      ,
      ADA_ARROW = 8
      ,
      ADA_ASSIGN = 9
      ,
      ADA_AT = 10
      ,
      ADA_BEGIN = 11
      ,
      ADA_BODY = 12
      ,
      ADA_BRACK_CLOSE = 13
      ,
      ADA_BRACK_OPEN = 14
      ,
      ADA_CASE = 15
      ,
      ADA_CHAR = 16
      ,
      ADA_COLON = 17
      ,
      ADA_COMMA = 18
      ,
      ADA_COMMENT = 19
      ,
      ADA_CONSTANT = 20
      ,
      ADA_DECIMAL = 21
      ,
      ADA_DECLARE = 22
      ,
      ADA_DELAY = 23
      ,
      ADA_DELTA = 24
      ,
      ADA_DIAMOND = 25
      ,
      ADA_DIGITS = 26
      ,
      ADA_DIVIDE = 27
      ,
      ADA_DO = 28
      ,
      ADA_DOT = 29
      ,
      ADA_DOUBLEDOT = 30
      ,
      ADA_ELSE = 31
      ,
      ADA_ELSIF = 32
      ,
      ADA_END = 33
      ,
      ADA_ENTRY = 34
      ,
      ADA_EQUAL = 35
      ,
      ADA_EXCEPTION = 36
      ,
      ADA_EXIT = 37
      ,
      ADA_FOR = 38
      ,
      ADA_FORMAT_STRING_END = 39
      ,
      ADA_FORMAT_STRING_MID = 40
      ,
      ADA_FORMAT_STRING_START = 41
      ,
      ADA_FORMAT_STRING_STRING = 42
      ,
      ADA_FUNCTION = 43
      ,
      ADA_GENERIC = 44
      ,
      ADA_GOTO = 45
      ,
      ADA_GT = 46
      ,
      ADA_GTE = 47
      ,
      ADA_IDENTIFIER = 48
      ,
      ADA_IF = 49
      ,
      ADA_IN = 50
      ,
      ADA_INTEGER = 51
      ,
      ADA_IS = 52
      ,
      ADA_LABEL_END = 53
      ,
      ADA_LABEL_START = 54
      ,
      ADA_LEXING_FAILURE = 55
      ,
      ADA_LIMITED = 56
      ,
      ADA_LOOP = 57
      ,
      ADA_LT = 58
      ,
      ADA_LTE = 59
      ,
      ADA_MINUS = 60
      ,
      ADA_MOD = 61
      ,
      ADA_MULT = 62
      ,
      ADA_NEW = 63
      ,
      ADA_NOT = 64
      ,
      ADA_NOTEQUAL = 65
      ,
      ADA_NULL = 66
      ,
      ADA_OF = 67
      ,
      ADA_OR = 68
      ,
      ADA_OTHERS = 69
      ,
      ADA_OUT = 70
      ,
      ADA_PACKAGE = 71
      ,
      ADA_PAR_CLOSE = 72
      ,
      ADA_PAR_OPEN = 73
      ,
      ADA_PIPE = 74
      ,
      ADA_PLUS = 75
      ,
      ADA_POWER = 76
      ,
      ADA_PRAGMA = 77
      ,
      ADA_PREP_LINE = 78
      ,
      ADA_PRIVATE = 79
      ,
      ADA_PROCEDURE = 80
      ,
      ADA_RAISE = 81
      ,
      ADA_RANGE = 82
      ,
      ADA_RECORD = 83
      ,
      ADA_REM = 84
      ,
      ADA_RENAMES = 85
      ,
      ADA_RETURN = 86
      ,
      ADA_REVERSE = 87
      ,
      ADA_SELECT = 88
      ,
      ADA_SEMICOLON = 89
      ,
      ADA_SEPARATE = 90
      ,
      ADA_STRING = 91
      ,
      ADA_SUBTYPE = 92
      ,
      ADA_TARGET = 93
      ,
      ADA_TASK = 94
      ,
      ADA_TERMINATE = 95
      ,
      ADA_TERMINATION = 96
      ,
      ADA_THEN = 97
      ,
      ADA_TICK = 98
      ,
      ADA_TYPE = 99
      ,
      ADA_USE = 100
      ,
      ADA_WHEN = 101
      ,
      ADA_WHILE = 102
      ,
      ADA_WHITESPACE = 103
      ,
      ADA_WITH = 104
      ,
      ADA_XOR = 105
} ada_token_kind;

typedef struct
{
   uint64_t version;
} *ada_token_data_handler;

/*
 * Reference to a token in an analysis unit.
 */
typedef struct {
    /* Private data associated to this token, including stale reference
       checking data, or NULL if this designates no token.  */
    ada_analysis_context context;
    ada_token_data_handler token_data;

    /* Internal identifiers for this token.  */
    int token_index, trivia_index;
} ada_token;


/*
 * Diagnostic for an analysis unit: cannot open the source file, parsing error,
 * ...
 */
typedef struct {
    ada_source_location_range sloc_range;
    ada_text message;
} ada_diagnostic;

   typedef enum {
      ADA_ANALYSIS_UNIT_KIND_UNIT_SPECIFICATION, ADA_ANALYSIS_UNIT_KIND_UNIT_BODY
   } ada_analysis_unit_kind;
   /*
    * Specify a kind of analysis unit. Specification units provide an interface
    * to the outer world while body units provide an implementation for the
    * corresponding interface.
    */
   typedef enum {
      ADA_LOOKUP_KIND_RECURSIVE, ADA_LOOKUP_KIND_FLAT, ADA_LOOKUP_KIND_MINIMAL
   } ada_lookup_kind;
   /*

    */
   typedef enum {
      ADA_DESIGNATED_ENV_KIND_NONE, ADA_DESIGNATED_ENV_KIND_CURRENT_ENV, ADA_DESIGNATED_ENV_KIND_NAMED_ENV, ADA_DESIGNATED_ENV_KIND_DIRECT_ENV
   } ada_designated_env_kind;
   /*
    * Discriminant for DesignatedEnv structures.
    */
   typedef enum {
      ADA_REF_RESULT_KIND_NO_REF, ADA_REF_RESULT_KIND_PRECISE, ADA_REF_RESULT_KIND_IMPRECISE, ADA_REF_RESULT_KIND_ERROR
   } ada_ref_result_kind;
   /*
    * Kind for the result of a cross reference operation.
    *
    * * ``no_ref`` is for no reference, it is the null value for this enum.
    *
    * * ``precise`` is when the reference result is precise.
    *
    * * ``imprecise`` is when there was an error computing the precise result,
    *   and a result was gotten in an imprecise fashion.
    *
    * * ``error`` is for unrecoverable errors (either there is no imprecise
    *   path for the request you made, or the imprecise path errored out too).
    */
   typedef enum {
      ADA_CALL_EXPR_KIND_CALL, ADA_CALL_EXPR_KIND_ARRAY_SLICE, ADA_CALL_EXPR_KIND_ARRAY_INDEX, ADA_CALL_EXPR_KIND_TYPE_CONVERSION, ADA_CALL_EXPR_KIND_FAMILY_INDEX
   } ada_call_expr_kind;
   /*
    * Kind of CallExpr type.
    *
    * * ``call`` is when the CallExpr is a procedure or function call.
    *
    * * ``array_slice``, ``array_index`` is when the CallExpr is in fact an
    *   array slice or an array subcomponent access expression, respectively.
    *
    * * ``type_conversion`` is when the CallExpr is a type conversion.
    *
    * * ``family_index`` is for entry calls using a family index.
    */
   typedef enum {
      ADA_GRAMMAR_RULE_PARENT_LIST_RULE, ADA_GRAMMAR_RULE_PROTECTED_TYPE_DECL_RULE, ADA_GRAMMAR_RULE_PROTECTED_OP_RULE, ADA_GRAMMAR_RULE_PROTECTED_EL_RULE, ADA_GRAMMAR_RULE_PROTECTED_DEF_RULE, ADA_GRAMMAR_RULE_PROTECTED_DECL_RULE, ADA_GRAMMAR_RULE_TASK_ITEM_RULE, ADA_GRAMMAR_RULE_TASK_DEF_RULE, ADA_GRAMMAR_RULE_TASK_TYPE_DECL_RULE, ADA_GRAMMAR_RULE_SUBTYPE_DECL_RULE, ADA_GRAMMAR_RULE_INTERFACE_TYPE_DEF_RULE, ADA_GRAMMAR_RULE_UNCONSTRAINED_INDEX_RULE, ADA_GRAMMAR_RULE_ARRAY_TYPE_DEF_RULE, ADA_GRAMMAR_RULE_DISCRETE_SUBTYPE_DEFINITION_RULE, ADA_GRAMMAR_RULE_CONSTRAINT_LIST_RULE, ADA_GRAMMAR_RULE_SIGNED_INT_TYPE_DEF_RULE, ADA_GRAMMAR_RULE_MOD_INT_TYPE_DEF_RULE, ADA_GRAMMAR_RULE_DERIVED_TYPE_DEF_RULE, ADA_GRAMMAR_RULE_COMPOSITE_CONSTRAINT_ASSOC_RULE, ADA_GRAMMAR_RULE_COMPOSITE_CONSTRAINT_RULE, ADA_GRAMMAR_RULE_DIGITS_CONSTRAINT_RULE, ADA_GRAMMAR_RULE_DELTA_CONSTRAINT_RULE, ADA_GRAMMAR_RULE_RANGE_CONSTRAINT_RULE, ADA_GRAMMAR_RULE_CONSTRAINT_RULE, ADA_GRAMMAR_RULE_DISCRIMINANT_SPEC_RULE, ADA_GRAMMAR_RULE_DISCR_SPEC_LIST_RULE, ADA_GRAMMAR_RULE_DISCRIMINANT_PART_RULE, ADA_GRAMMAR_RULE_ENUM_LITERAL_DECL_RULE, ADA_GRAMMAR_RULE_FORMAL_DISCRETE_TYPE_DEF_RULE, ADA_GRAMMAR_RULE_RECORD_DEF_RULE, ADA_GRAMMAR_RULE_RANGE_SPEC_RULE, ADA_GRAMMAR_RULE_REAL_TYPE_DEF_RULE, ADA_GRAMMAR_RULE_SEXPR_OR_BOX_RULE, ADA_GRAMMAR_RULE_ORDINARY_FIXED_POINT_DEF_RULE, ADA_GRAMMAR_RULE_DECIMAL_FIXED_POINT_DEF_RULE, ADA_GRAMMAR_RULE_FLOATING_POINT_DEF_RULE, ADA_GRAMMAR_RULE_RECORD_TYPE_DEF_RULE, ADA_GRAMMAR_RULE_ACCESS_DEF_RULE, ADA_GRAMMAR_RULE_ENUM_TYPE_DEF_RULE, ADA_GRAMMAR_RULE_TYPE_DEF_RULE, ADA_GRAMMAR_RULE_VARIANT_RULE, ADA_GRAMMAR_RULE_ANONYMOUS_TYPE_DECL_RULE, ADA_GRAMMAR_RULE_INCOMPLETE_TYPE_DECL_RULE, ADA_GRAMMAR_RULE_TYPE_DECL_RULE, ADA_GRAMMAR_RULE_VARIANT_PART_RULE, ADA_GRAMMAR_RULE_COMPONENT_DEF_RULE, ADA_GRAMMAR_RULE_COMPONENT_ITEM_RULE, ADA_GRAMMAR_RULE_COMPONENT_DECL_RULE, ADA_GRAMMAR_RULE_COMPONENT_LIST_RULE, ADA_GRAMMAR_RULE_GENERIC_DECL_RULE, ADA_GRAMMAR_RULE_GENERIC_FORMAL_PART_RULE, ADA_GRAMMAR_RULE_GENERIC_FORMAL_DECL_RULE, ADA_GRAMMAR_RULE_FORMAL_TYPE_DECL_RULE, ADA_GRAMMAR_RULE_FORMAL_SUBP_DECL_RULE, ADA_GRAMMAR_RULE_RENAMING_CLAUSE_RULE, ADA_GRAMMAR_RULE_GENERIC_RENAMING_DECL_RULE, ADA_GRAMMAR_RULE_GENERIC_INSTANTIATION_RULE, ADA_GRAMMAR_RULE_EXCEPTION_DECL_RULE, ADA_GRAMMAR_RULE_BASIC_DECLS_RULE, ADA_GRAMMAR_RULE_PACKAGE_RENAMING_DECL_RULE, ADA_GRAMMAR_RULE_PACKAGE_DECL_RULE, ADA_GRAMMAR_RULE_BASIC_DECL_RULE, ADA_GRAMMAR_RULE_OBJECT_DECL_RULE, ADA_GRAMMAR_RULE_SUB_OBJECT_DECL_RULE, ADA_GRAMMAR_RULE_NO_TYPE_OBJECT_RENAMING_DECL_RULE, ADA_GRAMMAR_RULE_EXT_RET_STMT_OBJECT_DECL_RULE, ADA_GRAMMAR_RULE_DEFINING_ID_LIST_RULE, ADA_GRAMMAR_RULE_NUMBER_DECL_RULE, ADA_GRAMMAR_RULE_CONTRACT_CASE_ASSOC_RULE, ADA_GRAMMAR_RULE_CONTRACT_CASES_EXPR_RULE, ADA_GRAMMAR_RULE_ABSTRACT_STATE_DECL_RULE, ADA_GRAMMAR_RULE_MULTI_ABSTRACT_STATE_DECL_RULE, ADA_GRAMMAR_RULE_ASPECT_ASSOC_RULE, ADA_GRAMMAR_RULE_ASPECT_SPEC_RULE, ADA_GRAMMAR_RULE_SINGLE_TASK_DECL_RULE, ADA_GRAMMAR_RULE_OVERRIDING_INDICATOR_RULE, ADA_GRAMMAR_RULE_ENTRY_DECL_RULE, ADA_GRAMMAR_RULE_COMPONENT_CLAUSE_RULE, ADA_GRAMMAR_RULE_ASPECT_CLAUSE_RULE, ADA_GRAMMAR_RULE_PARAM_SPEC_RULE, ADA_GRAMMAR_RULE_PARAM_SPECS_RULE, ADA_GRAMMAR_RULE_SUBP_SPEC_RULE, ADA_GRAMMAR_RULE_EXPR_FN_RULE, ADA_GRAMMAR_RULE_NULL_SUBP_DECL_RULE, ADA_GRAMMAR_RULE_ABSTRACT_SUBP_DECL_RULE, ADA_GRAMMAR_RULE_SUBP_RENAMING_DECL_RULE, ADA_GRAMMAR_RULE_SIMPLE_SUBP_DECL_RULE, ADA_GRAMMAR_RULE_SUBP_DECL_RULE, ADA_GRAMMAR_RULE_WITH_CLAUSE_RULE, ADA_GRAMMAR_RULE_CONTEXT_ITEM_RULE, ADA_GRAMMAR_RULE_USE_CLAUSE_RULE, ADA_GRAMMAR_RULE_USE_PACKAGE_CLAUSE_RULE, ADA_GRAMMAR_RULE_USE_TYPE_CLAUSE_RULE, ADA_GRAMMAR_RULE_SUBTYPE_INDICATION_RULE, ADA_GRAMMAR_RULE_DISCRETE_SUBTYPE_INDICATION_RULE, ADA_GRAMMAR_RULE_CONSTRAINED_SUBTYPE_INDICATION_RULE, ADA_GRAMMAR_RULE_TYPE_EXPR_RULE, ADA_GRAMMAR_RULE_ANONYMOUS_TYPE_RULE, ADA_GRAMMAR_RULE_MODE_RULE, ADA_GRAMMAR_RULE_PRAGMA_ARGUMENT_RULE, ADA_GRAMMAR_RULE_PRAGMA_RULE, ADA_GRAMMAR_RULE_SUBUNIT_RULE, ADA_GRAMMAR_RULE_LIBRARY_UNIT_BODY_RULE, ADA_GRAMMAR_RULE_LIBRARY_UNIT_RENAMING_DECL_RULE, ADA_GRAMMAR_RULE_LIBRARY_ITEM_RULE, ADA_GRAMMAR_RULE_COMPILATION_UNIT_RULE, ADA_GRAMMAR_RULE_COMPILATION_RULE, ADA_GRAMMAR_RULE_DECL_PART_RULE, ADA_GRAMMAR_RULE_ENTRY_BODY_RULE, ADA_GRAMMAR_RULE_PROTECTED_BODY_RULE, ADA_GRAMMAR_RULE_PROTECTED_BODY_STUB_RULE, ADA_GRAMMAR_RULE_TASK_BODY_RULE, ADA_GRAMMAR_RULE_TASK_BODY_STUB_RULE, ADA_GRAMMAR_RULE_PACKAGE_BODY_STUB_RULE, ADA_GRAMMAR_RULE_PACKAGE_BODY_RULE, ADA_GRAMMAR_RULE_TERMINATE_ALTERNATIVE_RULE, ADA_GRAMMAR_RULE_SELECT_STMT_RULE, ADA_GRAMMAR_RULE_ACCEPT_STMT_RULE, ADA_GRAMMAR_RULE_CASE_ALT_RULE, ADA_GRAMMAR_RULE_CASE_STMT_RULE, ADA_GRAMMAR_RULE_EXT_RETURN_STMT_RULE, ADA_GRAMMAR_RULE_IBLOCK_STMT_RULE, ADA_GRAMMAR_RULE_BLOCK_STMT_RULE, ADA_GRAMMAR_RULE_WHILE_LOOP_SPEC_RULE, ADA_GRAMMAR_RULE_ILOOP_STMT_RULE, ADA_GRAMMAR_RULE_LOOP_STMT_RULE, ADA_GRAMMAR_RULE_COMPOUND_STMT_RULE, ADA_GRAMMAR_RULE_ELSIF_PART_RULE, ADA_GRAMMAR_RULE_IF_STMT_RULE, ADA_GRAMMAR_RULE_RAISE_STMT_RULE, ADA_GRAMMAR_RULE_DELAY_STMT_RULE, ADA_GRAMMAR_RULE_ABORT_STMT_RULE, ADA_GRAMMAR_RULE_BODY_RULE, ADA_GRAMMAR_RULE_BODY_STUB_RULE, ADA_GRAMMAR_RULE_SUBP_BODY_STUB_RULE, ADA_GRAMMAR_RULE_RECOV_DECL_PART_RULE, ADA_GRAMMAR_RULE_SUBP_BODY_RULE, ADA_GRAMMAR_RULE_HANDLED_STMTS_RULE, ADA_GRAMMAR_RULE_EXCEPTION_HANDLER_RULE, ADA_GRAMMAR_RULE_STMTS_RULE, ADA_GRAMMAR_RULE_LABEL_RULE, ADA_GRAMMAR_RULE_STMT_RULE, ADA_GRAMMAR_RULE_CALL_STMT_RULE, ADA_GRAMMAR_RULE_SIMPLE_STMT_RULE, ADA_GRAMMAR_RULE_SIMPLE_DECL_STMT_RULE, ADA_GRAMMAR_RULE_NULL_STMT_RULE, ADA_GRAMMAR_RULE_ASSIGNMENT_STMT_RULE, ADA_GRAMMAR_RULE_GOTO_STMT_RULE, ADA_GRAMMAR_RULE_EXIT_STMT_RULE, ADA_GRAMMAR_RULE_RETURN_STMT_RULE, ADA_GRAMMAR_RULE_REQUEUE_STMT_RULE, ADA_GRAMMAR_RULE_IDENTIFIER_RULE, ADA_GRAMMAR_RULE_CHAR_LITERAL_RULE, ADA_GRAMMAR_RULE_STRING_LITERAL_RULE, ADA_GRAMMAR_RULE_FORMAT_STRING_LITERAL_RULE, ADA_GRAMMAR_RULE_DEFINING_ID_RULE, ADA_GRAMMAR_RULE_DEC_LITERAL_RULE, ADA_GRAMMAR_RULE_INT_LITERAL_RULE, ADA_GRAMMAR_RULE_NUM_LITERAL_RULE, ADA_GRAMMAR_RULE_NULL_LITERAL_RULE, ADA_GRAMMAR_RULE_ALLOCATOR_RULE, ADA_GRAMMAR_RULE_FOR_LOOP_PARAM_SPEC_RULE, ADA_GRAMMAR_RULE_QUANTIFIED_EXPR_RULE, ADA_GRAMMAR_RULE_CASE_EXPR_RULE, ADA_GRAMMAR_RULE_CASE_EXPR_ALT_RULE, ADA_GRAMMAR_RULE_RAISE_EXPR_RULE, ADA_GRAMMAR_RULE_IF_EXPR_RULE, ADA_GRAMMAR_RULE_CONDITIONAL_EXPR_RULE, ADA_GRAMMAR_RULE_BOX_EXPR_RULE, ADA_GRAMMAR_RULE_OTHERS_DESIGNATOR_RULE, ADA_GRAMMAR_RULE_ITERATED_ASSOC_RULE, ADA_GRAMMAR_RULE_AGGREGATE_ASSOC_RULE, ADA_GRAMMAR_RULE_REGULAR_AGGREGATE_RULE, ADA_GRAMMAR_RULE_BRACKET_AGGREGATE_RULE, ADA_GRAMMAR_RULE_AGGREGATE_RULE, ADA_GRAMMAR_RULE_DIRECT_NAME_RULE, ADA_GRAMMAR_RULE_PARAM_ASSOC_RULE, ADA_GRAMMAR_RULE_CALL_SUFFIX_RULE, ADA_GRAMMAR_RULE_ATTR_SUFFIX_RULE, ADA_GRAMMAR_RULE_QUALIFIED_NAME_RULE, ADA_GRAMMAR_RULE_QUAL_NAME_INTERNAL_RULE, ADA_GRAMMAR_RULE_VALUE_SEQUENCE_RULE, ADA_GRAMMAR_RULE_NAME_RULE, ADA_GRAMMAR_RULE_DEFINING_NAME_RULE, ADA_GRAMMAR_RULE_DIRECT_NAME_OR_TARGET_NAME_RULE, ADA_GRAMMAR_RULE_TARGET_NAME_RULE, ADA_GRAMMAR_RULE_UPDATE_ATTR_AGGREGATE_RULE, ADA_GRAMMAR_RULE_UPDATE_ATTR_CONTENT_RULE, ADA_GRAMMAR_RULE_MULTIDIM_ARRAY_ASSOC_RULE, ADA_GRAMMAR_RULE_SUBTYPE_NAME_RULE, ADA_GRAMMAR_RULE_STATIC_NAME_RULE, ADA_GRAMMAR_RULE_PRIMARY_RULE, ADA_GRAMMAR_RULE_PAREN_EXPR_RULE, ADA_GRAMMAR_RULE_DECLARE_EXPR_RULE, ADA_GRAMMAR_RULE_FACTOR_RULE, ADA_GRAMMAR_RULE_TERM_RULE, ADA_GRAMMAR_RULE_UNOP_TERM_RULE, ADA_GRAMMAR_RULE_ADD_TERM_RULE, ADA_GRAMMAR_RULE_SIMPLE_EXPR_RULE, ADA_GRAMMAR_RULE_BOOLEAN_OP_RULE, ADA_GRAMMAR_RULE_DISCRETE_RANGE_RULE, ADA_GRAMMAR_RULE_ARRAY_SUBCOMPONENT_CHOICE_RULE, ADA_GRAMMAR_RULE_CHOICE_RULE, ADA_GRAMMAR_RULE_CHOICE_LIST_RULE, ADA_GRAMMAR_RULE_DELTA_CHOICE_LIST_RULE, ADA_GRAMMAR_RULE_REL_OP_RULE, ADA_GRAMMAR_RULE_MEMBERSHIP_CHOICE_RULE, ADA_GRAMMAR_RULE_MEMBERSHIP_CHOICE_LIST_RULE, ADA_GRAMMAR_RULE_RELATION_RULE, ADA_GRAMMAR_RULE_EXPR_RULE, ADA_GRAMMAR_RULE_PP_DIRECTIVE_RULE, ADA_GRAMMAR_RULE_PP_THEN_RULE, ADA_GRAMMAR_RULE_PP_EXPR_RULE, ADA_GRAMMAR_RULE_PP_TERM_RULE
   } ada_grammar_rule;
   /*
    * Gramar rule to use for parsing.
    */

#define ada_default_grammar_rule ADA_GRAMMAR_RULE_COMPILATION_RULE

/*
 * Enumerated type describing all possible exceptions that need to be handled
 * in the C bindings.
 */
typedef enum {
      EXCEPTION_FILE_READ_ERROR,
      EXCEPTION_BAD_TYPE_ERROR,
      EXCEPTION_OUT_OF_BOUNDS_ERROR,
      EXCEPTION_INVALID_INPUT,
      EXCEPTION_INVALID_SYMBOL_ERROR,
      EXCEPTION_INVALID_UNIT_NAME_ERROR,
      EXCEPTION_NATIVE_EXCEPTION,
      EXCEPTION_PRECONDITION_FAILURE,
      EXCEPTION_PROPERTY_ERROR,
      EXCEPTION_TEMPLATE_ARGS_ERROR,
      EXCEPTION_TEMPLATE_FORMAT_ERROR,
      EXCEPTION_TEMPLATE_INSTANTIATION_ERROR,
      EXCEPTION_STALE_REFERENCE_ERROR,
      EXCEPTION_SYNTAX_ERROR,
      EXCEPTION_UNKNOWN_CHARSET,
      EXCEPTION_MALFORMED_TREE_ERROR,
      EXCEPTION_PROJECT_ERROR,
      EXCEPTION_UNSUPPORTED_VIEW_ERROR,
} ada_exception_kind;

/*
 * Holder for native exceptions-related information.  Memory management for
 * this and all the fields is handled by the library: one just has to make sure
 * not to keep references to it.
 *
 * .. TODO: For the moment, this structure contains already formatted
 *    information, but depending on possible future Ada runtime improvements,
 *    this might change.
 */
typedef struct {
   /*
 * The kind of this exception.
 */
   ada_exception_kind kind;

   /*
 * Message and context information associated with this exception.
 */
   const char *information;
} ada_exception;

/*
 * Array types incomplete declarations
 */

        

typedef struct ada_internal_aspect_array_record *ada_internal_aspect_array;

        

typedef struct ada_internal_discriminant_values_array_record *ada_internal_discriminant_values_array;

        

typedef struct ada_internal_doc_annotation_array_record *ada_internal_doc_annotation_array;

        

typedef struct ada_node_array_record *ada_node_array;

        

typedef struct ada_internal_logic_context_array_record *ada_internal_logic_context_array;

        

typedef struct ada_internal_param_actual_array_record *ada_internal_param_actual_array;

        

typedef struct ada_internal_ref_result_array_record *ada_internal_ref_result_array;

        

typedef struct ada_internal_shape_array_record *ada_internal_shape_array;

        

typedef struct ada_internal_solver_diagnostic_array_record *ada_internal_solver_diagnostic_array;

        

typedef struct ada_internal_substitution_array_record *ada_internal_substitution_array;

        

typedef struct ada_analysis_unit_array_record *ada_analysis_unit_array;

        

typedef struct ada_symbol_type_array_record *ada_symbol_type_array;


/*
 * Iterator types incomplete declarations
 */

/*
 * An iterator provides a mean to retrieve values one-at-a-time.
 *
 * Currently, each iterator is bound to the analysis context used to create it.
 * Iterators are invalidated as soon as any unit of that analysis is reparsed.
 * Due to the nature of iterators (lazy computations), this invalidation is
 * necessary to avoid use of inconsistent state, such as an iterator trying to
 * use analysis context data that is stale.
 */



typedef void* ada_node_iterator;


        

typedef void* ada_internal_completion_item_iterator;


/*
 * Struct types declarations
 */

        



    typedef struct {
            ada_bool dottable_subp;
            ada_base_node primitive;
            ada_base_node primitive_real_type;
    } ada_internal_metadata;



        



    typedef struct {
            ada_internal_metadata md;
            ada_env_rebindings_type rebindings;
            ada_bool from_rebound;
    } ada_internal_entity_info;



        



    typedef struct {
            ada_base_node node;
            ada_internal_entity_info info;
    } ada_node;



        



    typedef struct {
            ada_bool exists;
            ada_node node;
            ada_node value;
            ada_bool inherited;
    } ada_internal_aspect;



        



    typedef struct {
            ada_node decl;
            ada_bool is_dot_call;
            ada_bool is_visible;
            int weight;
    } ada_internal_completion_item;



        



    typedef struct {
            ada_node low_bound;
            ada_node high_bound;
    } ada_internal_discrete_range;



        



    typedef struct {
            ada_node discriminant;
            ada_node values;
    } ada_internal_discriminant_values;



        



    typedef struct {
            ada_string_type key;
            ada_string_type value;
    } ada_internal_doc_annotation;

    /* Increment the ref-count of all components in "r".  */
    extern void
    ada_internal_doc_annotation_inc_ref(ada_internal_doc_annotation *r);

    /* Decrement the ref-count of all components in "r".  */
    extern void
    ada_internal_doc_annotation_dec_ref(ada_internal_doc_annotation *r);


        



    typedef struct {
            ada_node ref_node;
            ada_node decl_node;
    } ada_internal_logic_context;



        



    typedef struct {
            ada_node param;
            ada_node actual;
    } ada_internal_param_actual;



        



    typedef struct {
            ada_node ref;
            ada_ref_result_kind kind;
    } ada_internal_ref_result;



        



    typedef struct {
            ada_node decl;
            ada_ref_result_kind kind;
    } ada_internal_refd_decl;



        



    typedef struct {
            ada_node def_name;
            ada_ref_result_kind kind;
    } ada_internal_refd_def;



        



    typedef struct {
            ada_node_array components;
            ada_internal_discriminant_values_array discriminants_values;
    } ada_internal_shape;

    /* Increment the ref-count of all components in "r".  */
    extern void
    ada_internal_shape_inc_ref(ada_internal_shape *r);

    /* Decrement the ref-count of all components in "r".  */
    extern void
    ada_internal_shape_dec_ref(ada_internal_shape *r);


        



    typedef struct {
            ada_string_type message_template;
            ada_node_array args;
            ada_base_node location;
            ada_internal_logic_context_array contexts;
            int round;
    } ada_internal_solver_diagnostic;

    /* Increment the ref-count of all components in "r".  */
    extern void
    ada_internal_solver_diagnostic_inc_ref(ada_internal_solver_diagnostic *r);

    /* Decrement the ref-count of all components in "r".  */
    extern void
    ada_internal_solver_diagnostic_dec_ref(ada_internal_solver_diagnostic *r);


        



    typedef struct {
            ada_node from_decl;
            ada_big_integer to_value;
            ada_node value_type;
    } ada_internal_substitution;

    /* Increment the ref-count of all components in "r".  */
    extern void
    ada_internal_substitution_inc_ref(ada_internal_substitution *r);

    /* Decrement the ref-count of all components in "r".  */
    extern void
    ada_internal_substitution_dec_ref(ada_internal_substitution *r);



/*
 * Types for event handler
 */

/*
 * Interface to handle events sent by the analysis context.
 */
typedef struct ada_event_handler__struct *ada_event_handler;

/*
 * Callback that will be called when a unit is requested from the context
 * ``Context``.
 *
 * ``Name`` is the name of the requested unit.
 *
 * ``From`` is the unit from which the unit was requested.
 *
 * ``Found`` indicates whether the requested unit was found or not.
 *
 * ``Is_Not_Found_Error`` indicates whether the fact that the unit was not
 * found is an error or not.
 *
 * .. warning:: The interface of this callback is probably subject to change,
 *    so should be treated as experimental.
 */
typedef void (*ada_event_handler_unit_requested_callback)(
   void *data,
   ada_analysis_context context,
   ada_text *name,
   ada_analysis_unit from,
   ada_bool found,
   ada_bool is_not_found_error
);

/*
 * Callback type for functions that are called when destroying an event
 * handler.
 */
typedef void (*ada_event_handler_destroy_callback)(void *data);

/*
 * Callback that will be called when any unit is parsed from the context
 * ``Context``.
 *
 * ``Unit`` is the resulting unit.
 *
 * ``Reparsed`` indicates whether the unit was reparsed, or whether it was the
 * first parse.
 */
typedef void (*ada_event_handler_unit_parsed_callback)(
   void *data,
   ada_analysis_context context,
   ada_analysis_unit unit,
   ada_bool reparsed
);

/*
 * Types for file readers
 */

/*
 * Interface to override how source files are fetched and decoded.
 */
typedef struct ada_file_reader__struct *ada_file_reader;

/*
 * Callback type for functions that are called when destroying a file reader.
 */
typedef void (*ada_file_reader_destroy_callback)(void *data);

/*
 * Callback type for functions that are called to fetch the decoded source
 * buffer for a requested filename.
 */
typedef void (*ada_file_reader_read_callback)(
   void *data,
   const char *filename,
   const char *charset,
   int read_bom,
   ada_text *buffer,
   ada_diagnostic *diagnostic
);

/*
 * Types for unit providers
 */

/*
 * Interface to fetch analysis units from a name and a unit kind.
 *
 * The unit provider mechanism provides an abstraction which assumes that to
 * any couple (unit name, unit kind) we can associate at most one source file.
 * This means that several couples can be associated to the same source file,
 * but on the other hand, only one one source file can be associated to a
 * couple.
 *
 * This is used to make the semantic analysis able to switch from one analysis
 * units to another.
 *
 * See the documentation of each unit provider for the exact semantics of the
 * unit name/kind information.
 */
typedef struct ada_unit_provider__struct *ada_unit_provider;

/*
 * Types for introspection
 */

/* References to struct/node members.  */
typedef enum {
      ada_member_ref_aspect_exists
        = 1,
      ada_member_ref_aspect_node
        = 2,
      ada_member_ref_aspect_value
        = 3,
      ada_member_ref_aspect_inherited
        = 4,
      ada_member_ref_completion_item_decl
        = 5,
      ada_member_ref_completion_item_is_dot_call
        = 6,
      ada_member_ref_completion_item_is_visible
        = 7,
      ada_member_ref_completion_item_weight
        = 8,
      ada_member_ref_discrete_range_low_bound
        = 9,
      ada_member_ref_discrete_range_high_bound
        = 10,
      ada_member_ref_discriminant_values_discriminant
        = 11,
      ada_member_ref_discriminant_values_values
        = 12,
      ada_member_ref_doc_annotation_key
        = 13,
      ada_member_ref_doc_annotation_value
        = 14,
      ada_member_ref_logic_context_ref_node
        = 15,
      ada_member_ref_logic_context_decl_node
        = 16,
      ada_member_ref_param_actual_param
        = 17,
      ada_member_ref_param_actual_actual
        = 18,
      ada_member_ref_ref_result_ref
        = 19,
      ada_member_ref_ref_result_kind
        = 20,
      ada_member_ref_refd_decl_decl
        = 21,
      ada_member_ref_refd_decl_kind
        = 22,
      ada_member_ref_refd_def_def_name
        = 23,
      ada_member_ref_refd_def_kind
        = 24,
      ada_member_ref_shape_components
        = 25,
      ada_member_ref_shape_discriminants_values
        = 26,
      ada_member_ref_solver_diagnostic_message_template
        = 27,
      ada_member_ref_solver_diagnostic_args
        = 28,
      ada_member_ref_solver_diagnostic_location
        = 29,
      ada_member_ref_solver_diagnostic_contexts
        = 30,
      ada_member_ref_solver_diagnostic_round
        = 31,
      ada_member_ref_substitution_from_decl
        = 32,
      ada_member_ref_substitution_to_value
        = 33,
      ada_member_ref_substitution_value_type
        = 34,
      ada_member_ref_constrained_array_indices_f_list
        = 35,
      ada_member_ref_unconstrained_array_indices_f_types
        = 36,
      ada_member_ref_aspect_assoc_f_id
        = 37,
      ada_member_ref_aspect_assoc_f_expr
        = 38,
      ada_member_ref_at_clause_f_name
        = 39,
      ada_member_ref_at_clause_f_expr
        = 40,
      ada_member_ref_attribute_def_clause_f_attribute_expr
        = 41,
      ada_member_ref_attribute_def_clause_f_expr
        = 42,
      ada_member_ref_enum_rep_clause_f_type_name
        = 43,
      ada_member_ref_enum_rep_clause_f_aggregate
        = 44,
      ada_member_ref_record_rep_clause_f_name
        = 45,
      ada_member_ref_record_rep_clause_f_at_expr
        = 46,
      ada_member_ref_record_rep_clause_f_components
        = 47,
      ada_member_ref_aspect_spec_f_aspect_assocs
        = 48,
      ada_member_ref_contract_case_assoc_f_guard
        = 49,
      ada_member_ref_contract_case_assoc_f_consequence
        = 50,
      ada_member_ref_pragma_argument_assoc_f_name
        = 51,
      ada_member_ref_pragma_argument_assoc_f_expr
        = 52,
      ada_member_ref_entry_spec_f_entry_name
        = 53,
      ada_member_ref_entry_spec_f_family_type
        = 54,
      ada_member_ref_entry_spec_f_entry_params
        = 55,
      ada_member_ref_subp_spec_f_subp_kind
        = 56,
      ada_member_ref_subp_spec_f_subp_name
        = 57,
      ada_member_ref_subp_spec_f_subp_params
        = 58,
      ada_member_ref_subp_spec_f_subp_returns
        = 59,
      ada_member_ref_synthetic_binary_spec_f_left_param
        = 60,
      ada_member_ref_synthetic_binary_spec_f_right_param
        = 61,
      ada_member_ref_synthetic_binary_spec_f_return_type_expr
        = 62,
      ada_member_ref_synthetic_unary_spec_f_right_param
        = 63,
      ada_member_ref_synthetic_unary_spec_f_return_type_expr
        = 64,
      ada_member_ref_component_list_f_components
        = 65,
      ada_member_ref_component_list_f_variant_part
        = 66,
      ada_member_ref_known_discriminant_part_f_discr_specs
        = 67,
      ada_member_ref_entry_completion_formal_params_f_params
        = 68,
      ada_member_ref_generic_formal_part_f_decls
        = 69,
      ada_member_ref_base_record_def_f_components
        = 70,
      ada_member_ref_aggregate_assoc_f_designators
        = 71,
      ada_member_ref_aggregate_assoc_f_r_expr
        = 72,
      ada_member_ref_composite_constraint_assoc_f_ids
        = 73,
      ada_member_ref_composite_constraint_assoc_f_constraint_expr
        = 74,
      ada_member_ref_iterated_assoc_f_spec
        = 75,
      ada_member_ref_iterated_assoc_f_key_expr
        = 76,
      ada_member_ref_iterated_assoc_f_r_expr
        = 77,
      ada_member_ref_param_assoc_f_designator
        = 78,
      ada_member_ref_param_assoc_f_r_expr
        = 79,
      ada_member_ref_basic_decl_f_aspects
        = 80,
      ada_member_ref_abstract_state_decl_f_name
        = 81,
      ada_member_ref_anonymous_expr_decl_f_expr
        = 82,
      ada_member_ref_component_decl_f_ids
        = 83,
      ada_member_ref_component_decl_f_component_def
        = 84,
      ada_member_ref_component_decl_f_default_expr
        = 85,
      ada_member_ref_discriminant_spec_f_ids
        = 86,
      ada_member_ref_discriminant_spec_f_type_expr
        = 87,
      ada_member_ref_discriminant_spec_f_default_expr
        = 88,
      ada_member_ref_generic_formal_f_decl
        = 89,
      ada_member_ref_param_spec_f_ids
        = 90,
      ada_member_ref_param_spec_f_has_aliased
        = 91,
      ada_member_ref_param_spec_f_mode
        = 92,
      ada_member_ref_param_spec_f_type_expr
        = 93,
      ada_member_ref_param_spec_f_default_expr
        = 94,
      ada_member_ref_synthetic_formal_param_decl_f_param_type
        = 95,
      ada_member_ref_base_package_decl_f_package_name
        = 96,
      ada_member_ref_base_package_decl_f_public_part
        = 97,
      ada_member_ref_base_package_decl_f_private_part
        = 98,
      ada_member_ref_base_package_decl_f_end_name
        = 99,
      ada_member_ref_base_type_decl_f_name
        = 100,
      ada_member_ref_subtype_decl_f_subtype
        = 101,
      ada_member_ref_incomplete_type_decl_f_discriminants
        = 102,
      ada_member_ref_incomplete_formal_type_decl_f_is_tagged
        = 103,
      ada_member_ref_incomplete_formal_type_decl_f_default_type
        = 104,
      ada_member_ref_incomplete_tagged_type_decl_f_has_abstract
        = 105,
      ada_member_ref_protected_type_decl_f_discriminants
        = 106,
      ada_member_ref_protected_type_decl_f_interfaces
        = 107,
      ada_member_ref_protected_type_decl_f_definition
        = 108,
      ada_member_ref_task_type_decl_f_discriminants
        = 109,
      ada_member_ref_task_type_decl_f_definition
        = 110,
      ada_member_ref_type_decl_f_discriminants
        = 111,
      ada_member_ref_type_decl_f_type_def
        = 112,
      ada_member_ref_formal_type_decl_f_default_type
        = 113,
      ada_member_ref_classic_subp_decl_f_overriding
        = 114,
      ada_member_ref_classic_subp_decl_f_subp_spec
        = 115,
      ada_member_ref_formal_subp_decl_f_default_expr
        = 116,
      ada_member_ref_entry_decl_f_overriding
        = 117,
      ada_member_ref_entry_decl_f_spec
        = 118,
      ada_member_ref_enum_literal_decl_f_name
        = 119,
      ada_member_ref_generic_subp_internal_f_subp_spec
        = 120,
      ada_member_ref_synthetic_subp_decl_f_spec
        = 121,
      ada_member_ref_accept_stmt_body_f_name
        = 122,
      ada_member_ref_base_subp_body_f_overriding
        = 123,
      ada_member_ref_base_subp_body_f_subp_spec
        = 124,
      ada_member_ref_expr_function_f_expr
        = 125,
      ada_member_ref_subp_body_f_decls
        = 126,
      ada_member_ref_subp_body_f_stmts
        = 127,
      ada_member_ref_subp_body_f_end_name
        = 128,
      ada_member_ref_subp_renaming_decl_f_renames
        = 129,
      ada_member_ref_package_body_stub_f_name
        = 130,
      ada_member_ref_protected_body_stub_f_name
        = 131,
      ada_member_ref_subp_body_stub_f_overriding
        = 132,
      ada_member_ref_subp_body_stub_f_subp_spec
        = 133,
      ada_member_ref_task_body_stub_f_name
        = 134,
      ada_member_ref_entry_body_f_entry_name
        = 135,
      ada_member_ref_entry_body_f_index_spec
        = 136,
      ada_member_ref_entry_body_f_params
        = 137,
      ada_member_ref_entry_body_f_barrier
        = 138,
      ada_member_ref_entry_body_f_decls
        = 139,
      ada_member_ref_entry_body_f_stmts
        = 140,
      ada_member_ref_entry_body_f_end_name
        = 141,
      ada_member_ref_package_body_f_package_name
        = 142,
      ada_member_ref_package_body_f_decls
        = 143,
      ada_member_ref_package_body_f_stmts
        = 144,
      ada_member_ref_package_body_f_end_name
        = 145,
      ada_member_ref_protected_body_f_name
        = 146,
      ada_member_ref_protected_body_f_decls
        = 147,
      ada_member_ref_protected_body_f_end_name
        = 148,
      ada_member_ref_task_body_f_name
        = 149,
      ada_member_ref_task_body_f_decls
        = 150,
      ada_member_ref_task_body_f_stmts
        = 151,
      ada_member_ref_task_body_f_end_name
        = 152,
      ada_member_ref_entry_index_spec_f_id
        = 153,
      ada_member_ref_entry_index_spec_f_subtype
        = 154,
      ada_member_ref_exception_decl_f_ids
        = 155,
      ada_member_ref_exception_decl_f_renames
        = 156,
      ada_member_ref_exception_handler_f_exception_name
        = 157,
      ada_member_ref_exception_handler_f_handled_exceptions
        = 158,
      ada_member_ref_exception_handler_f_stmts
        = 159,
      ada_member_ref_for_loop_var_decl_f_id
        = 160,
      ada_member_ref_for_loop_var_decl_f_id_type
        = 161,
      ada_member_ref_generic_decl_f_formal_part
        = 162,
      ada_member_ref_generic_package_decl_f_package_decl
        = 163,
      ada_member_ref_generic_subp_decl_f_subp_decl
        = 164,
      ada_member_ref_generic_package_instantiation_f_name
        = 165,
      ada_member_ref_generic_package_instantiation_f_generic_pkg_name
        = 166,
      ada_member_ref_generic_package_instantiation_f_params
        = 167,
      ada_member_ref_generic_subp_instantiation_f_overriding
        = 168,
      ada_member_ref_generic_subp_instantiation_f_kind
        = 169,
      ada_member_ref_generic_subp_instantiation_f_subp_name
        = 170,
      ada_member_ref_generic_subp_instantiation_f_generic_subp_name
        = 171,
      ada_member_ref_generic_subp_instantiation_f_params
        = 172,
      ada_member_ref_generic_package_renaming_decl_f_name
        = 173,
      ada_member_ref_generic_package_renaming_decl_f_renames
        = 174,
      ada_member_ref_generic_subp_renaming_decl_f_kind
        = 175,
      ada_member_ref_generic_subp_renaming_decl_f_name
        = 176,
      ada_member_ref_generic_subp_renaming_decl_f_renames
        = 177,
      ada_member_ref_label_decl_f_name
        = 178,
      ada_member_ref_named_stmt_decl_f_name
        = 179,
      ada_member_ref_number_decl_f_ids
        = 180,
      ada_member_ref_number_decl_f_expr
        = 181,
      ada_member_ref_object_decl_f_ids
        = 182,
      ada_member_ref_object_decl_f_has_aliased
        = 183,
      ada_member_ref_object_decl_f_has_constant
        = 184,
      ada_member_ref_object_decl_f_mode
        = 185,
      ada_member_ref_object_decl_f_type_expr
        = 186,
      ada_member_ref_object_decl_f_default_expr
        = 187,
      ada_member_ref_object_decl_f_renaming_clause
        = 188,
      ada_member_ref_package_renaming_decl_f_name
        = 189,
      ada_member_ref_package_renaming_decl_f_renames
        = 190,
      ada_member_ref_single_protected_decl_f_name
        = 191,
      ada_member_ref_single_protected_decl_f_interfaces
        = 192,
      ada_member_ref_single_protected_decl_f_definition
        = 193,
      ada_member_ref_single_task_decl_f_task_type
        = 194,
      ada_member_ref_case_stmt_alternative_f_choices
        = 195,
      ada_member_ref_case_stmt_alternative_f_stmts
        = 196,
      ada_member_ref_compilation_unit_f_prelude
        = 197,
      ada_member_ref_compilation_unit_f_body
        = 198,
      ada_member_ref_compilation_unit_f_pragmas
        = 199,
      ada_member_ref_component_clause_f_id
        = 200,
      ada_member_ref_component_clause_f_position
        = 201,
      ada_member_ref_component_clause_f_range
        = 202,
      ada_member_ref_component_def_f_has_aliased
        = 203,
      ada_member_ref_component_def_f_has_constant
        = 204,
      ada_member_ref_component_def_f_type_expr
        = 205,
      ada_member_ref_composite_constraint_f_constraints
        = 206,
      ada_member_ref_delta_constraint_f_delta
        = 207,
      ada_member_ref_delta_constraint_f_range
        = 208,
      ada_member_ref_digits_constraint_f_digits
        = 209,
      ada_member_ref_digits_constraint_f_range
        = 210,
      ada_member_ref_range_constraint_f_range
        = 211,
      ada_member_ref_declarative_part_f_decls
        = 212,
      ada_member_ref_elsif_expr_part_f_cond_expr
        = 213,
      ada_member_ref_elsif_expr_part_f_then_expr
        = 214,
      ada_member_ref_elsif_stmt_part_f_cond_expr
        = 215,
      ada_member_ref_elsif_stmt_part_f_stmts
        = 216,
      ada_member_ref_abstract_state_decl_expr_f_state_decl
        = 217,
      ada_member_ref_allocator_f_subpool
        = 218,
      ada_member_ref_allocator_f_type_or_expr
        = 219,
      ada_member_ref_base_aggregate_f_ancestor_expr
        = 220,
      ada_member_ref_base_aggregate_f_assocs
        = 221,
      ada_member_ref_bin_op_f_left
        = 222,
      ada_member_ref_bin_op_f_op
        = 223,
      ada_member_ref_bin_op_f_right
        = 224,
      ada_member_ref_case_expr_alternative_f_choices
        = 225,
      ada_member_ref_case_expr_alternative_f_expr
        = 226,
      ada_member_ref_concat_op_f_first_operand
        = 227,
      ada_member_ref_concat_op_f_other_operands
        = 228,
      ada_member_ref_concat_operand_f_operator
        = 229,
      ada_member_ref_concat_operand_f_operand
        = 230,
      ada_member_ref_case_expr_f_expr
        = 231,
      ada_member_ref_case_expr_f_cases
        = 232,
      ada_member_ref_if_expr_f_cond_expr
        = 233,
      ada_member_ref_if_expr_f_then_expr
        = 234,
      ada_member_ref_if_expr_f_alternatives
        = 235,
      ada_member_ref_if_expr_f_else_expr
        = 236,
      ada_member_ref_contract_cases_f_contract_cases
        = 237,
      ada_member_ref_decl_expr_f_decls
        = 238,
      ada_member_ref_decl_expr_f_expr
        = 239,
      ada_member_ref_format_string_literal_f_opening_chunk
        = 240,
      ada_member_ref_format_string_literal_f_mid_exprs
        = 241,
      ada_member_ref_format_string_literal_f_trailing_expr
        = 242,
      ada_member_ref_membership_expr_f_expr
        = 243,
      ada_member_ref_membership_expr_f_op
        = 244,
      ada_member_ref_membership_expr_f_membership_exprs
        = 245,
      ada_member_ref_array_subcomponent_choice_name_f_name
        = 246,
      ada_member_ref_array_subcomponent_choice_name_f_suffix
        = 247,
      ada_member_ref_attribute_ref_f_prefix
        = 248,
      ada_member_ref_attribute_ref_f_attribute
        = 249,
      ada_member_ref_attribute_ref_f_args
        = 250,
      ada_member_ref_call_expr_f_name
        = 251,
      ada_member_ref_call_expr_f_suffix
        = 252,
      ada_member_ref_defining_name_f_name
        = 253,
      ada_member_ref_discrete_subtype_name_f_subtype
        = 254,
      ada_member_ref_dotted_name_f_prefix
        = 255,
      ada_member_ref_dotted_name_f_suffix
        = 256,
      ada_member_ref_end_name_f_name
        = 257,
      ada_member_ref_explicit_deref_f_prefix
        = 258,
      ada_member_ref_qual_expr_f_prefix
        = 259,
      ada_member_ref_qual_expr_f_suffix
        = 260,
      ada_member_ref_reduce_attribute_ref_f_prefix
        = 261,
      ada_member_ref_reduce_attribute_ref_f_attribute
        = 262,
      ada_member_ref_reduce_attribute_ref_f_args
        = 263,
      ada_member_ref_update_attribute_ref_f_prefix
        = 264,
      ada_member_ref_update_attribute_ref_f_attribute
        = 265,
      ada_member_ref_update_attribute_ref_f_values
        = 266,
      ada_member_ref_paren_expr_f_expr
        = 267,
      ada_member_ref_quantified_expr_f_quantifier
        = 268,
      ada_member_ref_quantified_expr_f_loop_spec
        = 269,
      ada_member_ref_quantified_expr_f_expr
        = 270,
      ada_member_ref_raise_expr_f_exception_name
        = 271,
      ada_member_ref_raise_expr_f_error_message
        = 272,
      ada_member_ref_un_op_f_op
        = 273,
      ada_member_ref_un_op_f_expr
        = 274,
      ada_member_ref_for_loop_iter_filter_f_expr
        = 275,
      ada_member_ref_format_string_chunk_f_expr
        = 276,
      ada_member_ref_format_string_chunk_f_string_tok
        = 277,
      ada_member_ref_handled_stmts_f_stmts
        = 278,
      ada_member_ref_handled_stmts_f_exceptions
        = 279,
      ada_member_ref_library_item_f_has_private
        = 280,
      ada_member_ref_library_item_f_item
        = 281,
      ada_member_ref_for_loop_spec_f_var_decl
        = 282,
      ada_member_ref_for_loop_spec_f_loop_type
        = 283,
      ada_member_ref_for_loop_spec_f_has_reverse
        = 284,
      ada_member_ref_for_loop_spec_f_iter_expr
        = 285,
      ada_member_ref_for_loop_spec_f_iter_filter
        = 286,
      ada_member_ref_while_loop_spec_f_expr
        = 287,
      ada_member_ref_multi_abstract_state_decl_f_decls
        = 288,
      ada_member_ref_params_f_params
        = 289,
      ada_member_ref_paren_abstract_state_decl_f_decl
        = 290,
      ada_member_ref_pp_elsif_directive_f_expr
        = 291,
      ada_member_ref_pp_elsif_directive_f_then_kw
        = 292,
      ada_member_ref_pp_if_directive_f_expr
        = 293,
      ada_member_ref_pp_if_directive_f_then_kw
        = 294,
      ada_member_ref_pragma_node_f_id
        = 295,
      ada_member_ref_pragma_node_f_args
        = 296,
      ada_member_ref_protected_def_f_public_part
        = 297,
      ada_member_ref_protected_def_f_private_part
        = 298,
      ada_member_ref_protected_def_f_end_name
        = 299,
      ada_member_ref_range_spec_f_range
        = 300,
      ada_member_ref_renaming_clause_f_renamed_object
        = 301,
      ada_member_ref_select_when_part_f_cond_expr
        = 302,
      ada_member_ref_select_when_part_f_stmts
        = 303,
      ada_member_ref_accept_stmt_f_body_decl
        = 304,
      ada_member_ref_accept_stmt_f_entry_index_expr
        = 305,
      ada_member_ref_accept_stmt_f_params
        = 306,
      ada_member_ref_accept_stmt_with_stmts_f_stmts
        = 307,
      ada_member_ref_accept_stmt_with_stmts_f_end_name
        = 308,
      ada_member_ref_base_loop_stmt_f_spec
        = 309,
      ada_member_ref_base_loop_stmt_f_stmts
        = 310,
      ada_member_ref_base_loop_stmt_f_end_name
        = 311,
      ada_member_ref_begin_block_f_stmts
        = 312,
      ada_member_ref_begin_block_f_end_name
        = 313,
      ada_member_ref_decl_block_f_decls
        = 314,
      ada_member_ref_decl_block_f_stmts
        = 315,
      ada_member_ref_decl_block_f_end_name
        = 316,
      ada_member_ref_case_stmt_f_expr
        = 317,
      ada_member_ref_case_stmt_f_pragmas
        = 318,
      ada_member_ref_case_stmt_f_alternatives
        = 319,
      ada_member_ref_extended_return_stmt_f_decl
        = 320,
      ada_member_ref_extended_return_stmt_f_stmts
        = 321,
      ada_member_ref_if_stmt_f_cond_expr
        = 322,
      ada_member_ref_if_stmt_f_then_stmts
        = 323,
      ada_member_ref_if_stmt_f_alternatives
        = 324,
      ada_member_ref_if_stmt_f_else_stmts
        = 325,
      ada_member_ref_named_stmt_f_decl
        = 326,
      ada_member_ref_named_stmt_f_stmt
        = 327,
      ada_member_ref_select_stmt_f_guards
        = 328,
      ada_member_ref_select_stmt_f_else_stmts
        = 329,
      ada_member_ref_select_stmt_f_abort_stmts
        = 330,
      ada_member_ref_abort_stmt_f_names
        = 331,
      ada_member_ref_assign_stmt_f_dest
        = 332,
      ada_member_ref_assign_stmt_f_expr
        = 333,
      ada_member_ref_call_stmt_f_call
        = 334,
      ada_member_ref_delay_stmt_f_has_until
        = 335,
      ada_member_ref_delay_stmt_f_expr
        = 336,
      ada_member_ref_exit_stmt_f_loop_name
        = 337,
      ada_member_ref_exit_stmt_f_cond_expr
        = 338,
      ada_member_ref_goto_stmt_f_label_name
        = 339,
      ada_member_ref_label_f_decl
        = 340,
      ada_member_ref_raise_stmt_f_exception_name
        = 341,
      ada_member_ref_raise_stmt_f_error_message
        = 342,
      ada_member_ref_requeue_stmt_f_call_name
        = 343,
      ada_member_ref_requeue_stmt_f_has_abort
        = 344,
      ada_member_ref_return_stmt_f_return_expr
        = 345,
      ada_member_ref_simple_decl_stmt_f_decl
        = 346,
      ada_member_ref_subunit_f_name
        = 347,
      ada_member_ref_subunit_f_body
        = 348,
      ada_member_ref_task_def_f_interfaces
        = 349,
      ada_member_ref_task_def_f_public_part
        = 350,
      ada_member_ref_task_def_f_private_part
        = 351,
      ada_member_ref_task_def_f_end_name
        = 352,
      ada_member_ref_access_def_f_has_not_null
        = 353,
      ada_member_ref_access_to_subp_def_f_has_protected
        = 354,
      ada_member_ref_access_to_subp_def_f_subp_spec
        = 355,
      ada_member_ref_anonymous_type_access_def_f_type_decl
        = 356,
      ada_member_ref_type_access_def_f_has_all
        = 357,
      ada_member_ref_type_access_def_f_has_constant
        = 358,
      ada_member_ref_type_access_def_f_subtype_indication
        = 359,
      ada_member_ref_array_type_def_f_indices
        = 360,
      ada_member_ref_array_type_def_f_component_type
        = 361,
      ada_member_ref_derived_type_def_f_has_abstract
        = 362,
      ada_member_ref_derived_type_def_f_has_limited
        = 363,
      ada_member_ref_derived_type_def_f_has_synchronized
        = 364,
      ada_member_ref_derived_type_def_f_subtype_indication
        = 365,
      ada_member_ref_derived_type_def_f_interfaces
        = 366,
      ada_member_ref_derived_type_def_f_record_extension
        = 367,
      ada_member_ref_derived_type_def_f_has_with_private
        = 368,
      ada_member_ref_enum_type_def_f_enum_literals
        = 369,
      ada_member_ref_interface_type_def_f_interface_kind
        = 370,
      ada_member_ref_interface_type_def_f_interfaces
        = 371,
      ada_member_ref_mod_int_type_def_f_expr
        = 372,
      ada_member_ref_private_type_def_f_has_abstract
        = 373,
      ada_member_ref_private_type_def_f_has_tagged
        = 374,
      ada_member_ref_private_type_def_f_has_limited
        = 375,
      ada_member_ref_decimal_fixed_point_def_f_delta
        = 376,
      ada_member_ref_decimal_fixed_point_def_f_digits
        = 377,
      ada_member_ref_decimal_fixed_point_def_f_range
        = 378,
      ada_member_ref_floating_point_def_f_num_digits
        = 379,
      ada_member_ref_floating_point_def_f_range
        = 380,
      ada_member_ref_ordinary_fixed_point_def_f_delta
        = 381,
      ada_member_ref_ordinary_fixed_point_def_f_range
        = 382,
      ada_member_ref_record_type_def_f_has_abstract
        = 383,
      ada_member_ref_record_type_def_f_has_tagged
        = 384,
      ada_member_ref_record_type_def_f_has_limited
        = 385,
      ada_member_ref_record_type_def_f_record_def
        = 386,
      ada_member_ref_signed_int_type_def_f_range
        = 387,
      ada_member_ref_anonymous_type_f_type_decl
        = 388,
      ada_member_ref_subtype_indication_f_has_not_null
        = 389,
      ada_member_ref_subtype_indication_f_name
        = 390,
      ada_member_ref_subtype_indication_f_constraint
        = 391,
      ada_member_ref_synthetic_type_expr_f_target_type
        = 392,
      ada_member_ref_unconstrained_array_index_f_subtype_name
        = 393,
      ada_member_ref_unconstrained_array_index_f_lower_bound
        = 394,
      ada_member_ref_use_package_clause_f_packages
        = 395,
      ada_member_ref_use_type_clause_f_has_all
        = 396,
      ada_member_ref_use_type_clause_f_types
        = 397,
      ada_member_ref_value_sequence_f_iter_assoc
        = 398,
      ada_member_ref_variant_f_choices
        = 399,
      ada_member_ref_variant_f_components
        = 400,
      ada_member_ref_variant_part_f_discr_name
        = 401,
      ada_member_ref_variant_part_f_variant
        = 402,
      ada_member_ref_with_clause_f_has_limited
        = 403,
      ada_member_ref_with_clause_f_has_private
        = 404,
      ada_member_ref_with_clause_f_packages
        = 405,
      ada_member_ref_ada_node_p_declarative_scope
        = 406,
      ada_member_ref_ada_node_p_enclosing_compilation_unit
        = 407,
      ada_member_ref_ada_node_p_get_uninstantiated_node
        = 408,
      ada_member_ref_ada_node_p_complete
        = 409,
      ada_member_ref_ada_node_p_valid_keywords
        = 410,
      ada_member_ref_ada_node_p_generic_instantiations
        = 411,
      ada_member_ref_ada_node_p_semantic_parent
        = 412,
      ada_member_ref_ada_node_p_parent_basic_decl
        = 413,
      ada_member_ref_ada_node_p_has_spark_mode_on
        = 414,
      ada_member_ref_ada_node_p_is_subject_to_proof
        = 415,
      ada_member_ref_ada_node_p_filter_is_imported_by
        = 416,
      ada_member_ref_ada_node_p_resolve_names
        = 417,
      ada_member_ref_ada_node_p_nameres_diagnostics
        = 418,
      ada_member_ref_ada_node_p_standard_unit
        = 419,
      ada_member_ref_ada_node_p_is_keyword
        = 420,
      ada_member_ref_ada_node_p_std_entity
        = 421,
      ada_member_ref_ada_node_p_bool_type
        = 422,
      ada_member_ref_ada_node_p_int_type
        = 423,
      ada_member_ref_ada_node_p_universal_int_type
        = 424,
      ada_member_ref_ada_node_p_universal_real_type
        = 425,
      ada_member_ref_ada_node_p_std_char_type
        = 426,
      ada_member_ref_ada_node_p_std_wide_char_type
        = 427,
      ada_member_ref_ada_node_p_std_wide_wide_char_type
        = 428,
      ada_member_ref_ada_node_p_std_string_type
        = 429,
      ada_member_ref_ada_node_p_std_wide_string_type
        = 430,
      ada_member_ref_ada_node_p_std_wide_wide_string_type
        = 431,
      ada_member_ref_ada_node_p_top_level_decl
        = 432,
      ada_member_ref_ada_node_p_choice_match
        = 433,
      ada_member_ref_ada_node_p_gnat_xref
        = 434,
      ada_member_ref_ada_node_p_spark_mode_aspect
        = 435,
      ada_member_ref_ada_node_p_xref_entry_point
        = 436,
      ada_member_ref_ada_node_p_matching_with_use_clause
        = 437,
      ada_member_ref_parent
        = 438,
      ada_member_ref_parents
        = 439,
      ada_member_ref_children
        = 440,
      ada_member_ref_token_start
        = 441,
      ada_member_ref_token_end
        = 442,
      ada_member_ref_child_index
        = 443,
      ada_member_ref_previous_sibling
        = 444,
      ada_member_ref_next_sibling
        = 445,
      ada_member_ref_unit
        = 446,
      ada_member_ref_is_ghost
        = 447,
      ada_member_ref_full_sloc_image
        = 448,
      ada_member_ref_abort_node_p_as_bool
        = 449,
      ada_member_ref_abstract_node_p_as_bool
        = 450,
      ada_member_ref_assoc_list_p_zip_with_params
        = 451,
      ada_member_ref_aliased_node_p_as_bool
        = 452,
      ada_member_ref_all_node_p_as_bool
        = 453,
      ada_member_ref_aspect_assoc_p_is_ghost_code
        = 454,
      ada_member_ref_enum_rep_clause_p_params
        = 455,
      ada_member_ref_base_assoc_p_assoc_expr
        = 456,
      ada_member_ref_base_formal_param_holder_p_abstract_formal_params
        = 457,
      ada_member_ref_base_formal_param_holder_p_formal_params
        = 458,
      ada_member_ref_base_formal_param_holder_p_nb_min_params
        = 459,
      ada_member_ref_base_formal_param_holder_p_nb_max_params
        = 460,
      ada_member_ref_base_formal_param_holder_p_param_types
        = 461,
      ada_member_ref_base_subp_spec_p_name
        = 462,
      ada_member_ref_base_subp_spec_p_returns
        = 463,
      ada_member_ref_base_subp_spec_p_primitive_subp_types
        = 464,
      ada_member_ref_base_subp_spec_p_primitive_subp_first_type
        = 465,
      ada_member_ref_base_subp_spec_p_primitive_subp_tagged_type
        = 466,
      ada_member_ref_base_subp_spec_p_return_type
        = 467,
      ada_member_ref_base_subp_spec_p_params
        = 468,
      ada_member_ref_basic_assoc_p_get_params
        = 469,
      ada_member_ref_basic_decl_p_is_formal
        = 470,
      ada_member_ref_basic_decl_p_doc_annotations
        = 471,
      ada_member_ref_basic_decl_p_doc
        = 472,
      ada_member_ref_basic_decl_p_canonical_part
        = 473,
      ada_member_ref_basic_decl_p_all_parts
        = 474,
      ada_member_ref_basic_decl_p_get_aspect_assoc
        = 475,
      ada_member_ref_basic_decl_p_get_aspect_spec_expr
        = 476,
      ada_member_ref_basic_decl_p_get_aspect
        = 477,
      ada_member_ref_basic_decl_p_has_aspect
        = 478,
      ada_member_ref_basic_decl_p_get_pragma
        = 479,
      ada_member_ref_basic_decl_p_get_representation_clause
        = 480,
      ada_member_ref_basic_decl_p_get_at_clause
        = 481,
      ada_member_ref_basic_decl_p_get_annotations
        = 482,
      ada_member_ref_basic_decl_p_is_ghost_code
        = 483,
      ada_member_ref_basic_decl_p_is_compilation_unit_root
        = 484,
      ada_member_ref_basic_decl_p_is_visible
        = 485,
      ada_member_ref_basic_decl_p_base_subp_declarations
        = 486,
      ada_member_ref_basic_decl_p_root_subp_declarations
        = 487,
      ada_member_ref_basic_decl_p_find_all_overrides
        = 488,
      ada_member_ref_basic_decl_p_defining_names
        = 489,
      ada_member_ref_basic_decl_p_defining_name
        = 490,
      ada_member_ref_basic_decl_p_subp_spec_or_null
        = 491,
      ada_member_ref_basic_decl_p_is_subprogram
        = 492,
      ada_member_ref_basic_decl_p_is_predefined_operator
        = 493,
      ada_member_ref_basic_decl_p_relative_name
        = 494,
      ada_member_ref_basic_decl_p_relative_name_text
        = 495,
      ada_member_ref_basic_decl_p_body_part_for_decl
        = 496,
      ada_member_ref_basic_decl_p_most_visible_part
        = 497,
      ada_member_ref_basic_decl_p_fully_qualified_name_array
        = 498,
      ada_member_ref_basic_decl_p_fully_qualified_name
        = 499,
      ada_member_ref_basic_decl_p_canonical_fully_qualified_name
        = 500,
      ada_member_ref_basic_decl_p_unique_identifying_name
        = 501,
      ada_member_ref_basic_decl_p_previous_part_for_decl
        = 502,
      ada_member_ref_basic_decl_p_is_static_decl
        = 503,
      ada_member_ref_basic_decl_p_is_imported
        = 504,
      ada_member_ref_basic_decl_p_type_expression
        = 505,
      ada_member_ref_basic_decl_p_next_part_for_decl
        = 506,
      ada_member_ref_basic_decl_p_is_constant_object
        = 507,
      ada_member_ref_anonymous_expr_decl_p_get_formal
        = 508,
      ada_member_ref_base_formal_param_decl_p_formal_type
        = 509,
      ada_member_ref_base_package_decl_p_body_part
        = 510,
      ada_member_ref_base_type_decl_p_base_subtype
        = 511,
      ada_member_ref_base_type_decl_p_attribute_subprogram
        = 512,
      ada_member_ref_base_type_decl_p_private_completion
        = 513,
      ada_member_ref_base_type_decl_p_is_inherited_primitive
        = 514,
      ada_member_ref_base_type_decl_p_get_record_representation_clause
        = 515,
      ada_member_ref_base_type_decl_p_get_enum_representation_clause
        = 516,
      ada_member_ref_base_type_decl_p_get_primitives
        = 517,
      ada_member_ref_base_type_decl_p_is_array_type
        = 518,
      ada_member_ref_base_type_decl_p_find_derived_types
        = 519,
      ada_member_ref_base_type_decl_p_is_scalar_type
        = 520,
      ada_member_ref_base_type_decl_p_specific_type
        = 521,
      ada_member_ref_base_type_decl_p_find_all_derived_types
        = 522,
      ada_member_ref_base_type_decl_p_comp_type
        = 523,
      ada_member_ref_base_type_decl_p_index_type
        = 524,
      ada_member_ref_base_type_decl_p_is_derived_type
        = 525,
      ada_member_ref_base_type_decl_p_is_limited_type
        = 526,
      ada_member_ref_base_type_decl_p_matching_type
        = 527,
      ada_member_ref_base_type_decl_p_previous_part
        = 528,
      ada_member_ref_base_type_decl_p_next_part
        = 529,
      ada_member_ref_base_type_decl_p_full_view
        = 530,
      ada_member_ref_base_type_decl_p_is_definite_subtype
        = 531,
      ada_member_ref_base_type_decl_p_discriminants_list
        = 532,
      ada_member_ref_base_type_decl_p_shapes
        = 533,
      ada_member_ref_base_type_decl_p_is_record_type
        = 534,
      ada_member_ref_base_type_decl_p_is_real_type
        = 535,
      ada_member_ref_base_type_decl_p_is_float_type
        = 536,
      ada_member_ref_base_type_decl_p_is_fixed_point
        = 537,
      ada_member_ref_base_type_decl_p_is_enum_type
        = 538,
      ada_member_ref_base_type_decl_p_is_access_type
        = 539,
      ada_member_ref_base_type_decl_p_is_char_type
        = 540,
      ada_member_ref_base_type_decl_p_classwide_type
        = 541,
      ada_member_ref_base_type_decl_p_discrete_range
        = 542,
      ada_member_ref_base_type_decl_p_is_discrete_type
        = 543,
      ada_member_ref_base_type_decl_p_is_int_type
        = 544,
      ada_member_ref_base_type_decl_p_accessed_type
        = 545,
      ada_member_ref_base_type_decl_p_is_tagged_type
        = 546,
      ada_member_ref_base_type_decl_p_base_type
        = 547,
      ada_member_ref_base_type_decl_p_base_types
        = 548,
      ada_member_ref_base_type_decl_p_is_interface_type
        = 549,
      ada_member_ref_base_type_decl_p_canonical_type
        = 550,
      ada_member_ref_base_type_decl_p_is_private
        = 551,
      ada_member_ref_base_type_decl_p_root_type
        = 552,
      ada_member_ref_base_subtype_decl_p_get_type
        = 553,
      ada_member_ref_basic_subp_decl_p_subp_decl_spec
        = 554,
      ada_member_ref_classic_subp_decl_p_body_part
        = 555,
      ada_member_ref_entry_decl_p_body_part
        = 556,
      ada_member_ref_entry_decl_p_accept_stmts
        = 557,
      ada_member_ref_enum_literal_decl_p_enum_rep
        = 558,
      ada_member_ref_enum_literal_decl_p_enum_type
        = 559,
      ada_member_ref_synthetic_char_enum_lit_p_expr
        = 560,
      ada_member_ref_body_node_p_previous_part
        = 561,
      ada_member_ref_body_node_p_decl_part
        = 562,
      ada_member_ref_body_node_p_subunit_root
        = 563,
      ada_member_ref_body_stub_p_syntactic_fully_qualified_name
        = 564,
      ada_member_ref_generic_package_decl_p_body_part
        = 565,
      ada_member_ref_generic_subp_decl_p_body_part
        = 566,
      ada_member_ref_generic_instantiation_p_designated_generic_decl
        = 567,
      ada_member_ref_generic_instantiation_p_inst_params
        = 568,
      ada_member_ref_generic_subp_instantiation_p_designated_subp
        = 569,
      ada_member_ref_object_decl_p_private_part_decl
        = 570,
      ada_member_ref_object_decl_p_public_part_decl
        = 571,
      ada_member_ref_package_renaming_decl_p_renamed_package
        = 572,
      ada_member_ref_package_renaming_decl_p_final_renamed_package
        = 573,
      ada_member_ref_compilation_unit_p_syntactic_fully_qualified_name
        = 574,
      ada_member_ref_compilation_unit_p_unit_kind
        = 575,
      ada_member_ref_compilation_unit_p_withed_units
        = 576,
      ada_member_ref_compilation_unit_p_imported_units
        = 577,
      ada_member_ref_compilation_unit_p_unit_dependencies
        = 578,
      ada_member_ref_compilation_unit_p_decl
        = 579,
      ada_member_ref_compilation_unit_p_is_preelaborable
        = 580,
      ada_member_ref_compilation_unit_p_other_part
        = 581,
      ada_member_ref_compilation_unit_p_has_restriction
        = 582,
      ada_member_ref_compilation_unit_p_all_config_pragmas
        = 583,
      ada_member_ref_compilation_unit_p_config_pragmas
        = 584,
      ada_member_ref_constant_node_p_as_bool
        = 585,
      ada_member_ref_composite_constraint_p_is_index_constraint
        = 586,
      ada_member_ref_composite_constraint_p_is_discriminant_constraint
        = 587,
      ada_member_ref_composite_constraint_p_discriminant_params
        = 588,
      ada_member_ref_expr_p_expression_type
        = 589,
      ada_member_ref_expr_p_expected_expression_type
        = 590,
      ada_member_ref_expr_p_is_dynamically_tagged
        = 591,
      ada_member_ref_expr_p_is_static_expr
        = 592,
      ada_member_ref_expr_p_eval_as_int
        = 593,
      ada_member_ref_expr_p_eval_as_int_in_env
        = 594,
      ada_member_ref_expr_p_eval_as_string
        = 595,
      ada_member_ref_expr_p_eval_as_string_in_env
        = 596,
      ada_member_ref_expr_p_matching_nodes
        = 597,
      ada_member_ref_expr_p_is_dispatching_call
        = 598,
      ada_member_ref_expr_p_first_corresponding_decl
        = 599,
      ada_member_ref_allocator_p_get_allocated_type
        = 600,
      ada_member_ref_base_aggregate_p_aggregate_params
        = 601,
      ada_member_ref_base_aggregate_p_is_subaggregate
        = 602,
      ada_member_ref_concat_op_p_operands
        = 603,
      ada_member_ref_cond_expr_p_dependent_exprs
        = 604,
      ada_member_ref_name_p_enclosing_defining_name
        = 605,
      ada_member_ref_name_p_is_defining
        = 606,
      ada_member_ref_name_p_name_is
        = 607,
      ada_member_ref_name_p_is_direct_call
        = 608,
      ada_member_ref_name_p_is_access_call
        = 609,
      ada_member_ref_name_p_is_call
        = 610,
      ada_member_ref_name_p_is_dot_call
        = 611,
      ada_member_ref_name_p_failsafe_referenced_def_name
        = 612,
      ada_member_ref_name_p_referenced_defining_name
        = 613,
      ada_member_ref_name_p_all_env_elements
        = 614,
      ada_member_ref_name_p_called_subp_spec
        = 615,
      ada_member_ref_name_p_referenced_decl
        = 616,
      ada_member_ref_name_p_failsafe_referenced_decl
        = 617,
      ada_member_ref_name_p_name_designated_type
        = 618,
      ada_member_ref_name_p_is_static_subtype
        = 619,
      ada_member_ref_name_p_name_matches
        = 620,
      ada_member_ref_name_p_is_operator_name
        = 621,
      ada_member_ref_name_p_is_write_reference
        = 622,
      ada_member_ref_name_p_is_static_call
        = 623,
      ada_member_ref_name_p_call_params
        = 624,
      ada_member_ref_name_p_relative_name
        = 625,
      ada_member_ref_name_p_as_symbol_array
        = 626,
      ada_member_ref_name_p_canonical_text
        = 627,
      ada_member_ref_name_p_is_constant
        = 628,
      ada_member_ref_call_expr_p_kind
        = 629,
      ada_member_ref_call_expr_p_is_array_slice
        = 630,
      ada_member_ref_defining_name_p_canonical_fully_qualified_name
        = 631,
      ada_member_ref_defining_name_p_unique_identifying_name
        = 632,
      ada_member_ref_defining_name_p_fully_qualified_name_array
        = 633,
      ada_member_ref_defining_name_p_fully_qualified_name
        = 634,
      ada_member_ref_defining_name_p_basic_decl
        = 635,
      ada_member_ref_defining_name_p_find_refs
        = 636,
      ada_member_ref_defining_name_p_find_all_references
        = 637,
      ada_member_ref_defining_name_p_find_all_calls
        = 638,
      ada_member_ref_defining_name_p_next_part
        = 639,
      ada_member_ref_defining_name_p_previous_part
        = 640,
      ada_member_ref_defining_name_p_canonical_part
        = 641,
      ada_member_ref_defining_name_p_most_visible_part
        = 642,
      ada_member_ref_defining_name_p_all_parts
        = 643,
      ada_member_ref_defining_name_p_get_aspect
        = 644,
      ada_member_ref_defining_name_p_has_aspect
        = 645,
      ada_member_ref_defining_name_p_get_pragma
        = 646,
      ada_member_ref_defining_name_p_get_representation_clause
        = 647,
      ada_member_ref_defining_name_p_get_at_clause
        = 648,
      ada_member_ref_defining_name_p_get_annotations
        = 649,
      ada_member_ref_defining_name_p_is_imported
        = 650,
      ada_member_ref_defining_name_p_is_ghost_code
        = 651,
      ada_member_ref_end_name_p_basic_decl
        = 652,
      ada_member_ref_char_literal_p_denoted_value
        = 653,
      ada_member_ref_string_literal_p_denoted_value
        = 654,
      ada_member_ref_int_literal_p_denoted_value
        = 655,
      ada_member_ref_limited_node_p_as_bool
        = 656,
      ada_member_ref_not_null_p_as_bool
        = 657,
      ada_member_ref_pragma_node_p_is_ghost_code
        = 658,
      ada_member_ref_pragma_node_p_associated_entities
        = 659,
      ada_member_ref_private_node_p_as_bool
        = 660,
      ada_member_ref_protected_node_p_as_bool
        = 661,
      ada_member_ref_reverse_node_p_as_bool
        = 662,
      ada_member_ref_stmt_p_is_ghost_code
        = 663,
      ada_member_ref_accept_stmt_p_corresponding_entry
        = 664,
      ada_member_ref_subunit_p_body_root
        = 665,
      ada_member_ref_synchronized_node_p_as_bool
        = 666,
      ada_member_ref_tagged_node_p_as_bool
        = 667,
      ada_member_ref_type_expr_p_type_name
        = 668,
      ada_member_ref_type_expr_p_designated_type_decl
        = 669,
      ada_member_ref_type_expr_p_designated_type_decl_from
        = 670,
      ada_member_ref_type_expr_p_subtype_constraint
        = 671,
      ada_member_ref_type_expr_p_discriminant_constraints
        = 672,
      ada_member_ref_type_expr_p_is_definite_subtype
        = 673,
      ada_member_ref_subtype_indication_p_is_static_subtype
        = 674,
      ada_member_ref_until_node_p_as_bool
        = 675,
      ada_member_ref_with_private_p_as_bool
        = 676,
} ada_introspection_member_ref;

/*
 * Types for tree rewriting
 */

/*
 * Handle for an analysis context rewriting session
 */
typedef struct ada_rewriting_handle__struct *ada_rewriting_handle;

/*
 * Handle for the process of rewriting an analysis unit. Such handles are owned
 * by a Rewriting_Handle instance.
 */
typedef struct ada_unit_rewriting_handle__struct *ada_unit_rewriting_handle;

/*
 * Handle for the process of rewriting an AST node. Such handles are owned by a
 * Rewriting_Handle instance.
 */
typedef struct ada_node_rewriting_handle__struct *ada_node_rewriting_handle;

/*
 * Result of applying a rewriting session.
 *
 * On success, ``Success`` is true.
 *
 * On failure, ``Success`` is false, ``Unit`` is set to the unit on which
 * rewriting failed, and ``Diagnostics`` is set to related rewriting errors.
 */
typedef struct {
    int success;
    ada_analysis_unit unit;
    int diagnostics_count;
    ada_diagnostic *diagnostics;
} ada_rewriting_apply_result;

/* All the functions below can potentially raise an exception, so
   ada_get_last_exception must be checked after them even
   before trying to use the returned value.  */


/*
 * Array types declarations
 */

        



/*

 */
struct ada_internal_aspect_array_record {
   int n;
   int ref_count;
   ada_internal_aspect items[1];
};

/* Create a length-sized array.  */
extern ada_internal_aspect_array
ada_internal_aspect_array_create(int length);

/* Increment the ref-count for "a".  */
extern void
ada_internal_aspect_array_inc_ref(ada_internal_aspect_array a);

/* Decrement the ref-count for "a". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_internal_aspect_array_dec_ref(ada_internal_aspect_array a);


        



/*

 */
struct ada_internal_discriminant_values_array_record {
   int n;
   int ref_count;
   ada_internal_discriminant_values items[1];
};

/* Create a length-sized array.  */
extern ada_internal_discriminant_values_array
ada_internal_discriminant_values_array_create(int length);

/* Increment the ref-count for "a".  */
extern void
ada_internal_discriminant_values_array_inc_ref(ada_internal_discriminant_values_array a);

/* Decrement the ref-count for "a". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_internal_discriminant_values_array_dec_ref(ada_internal_discriminant_values_array a);


        



/*

 */
struct ada_internal_doc_annotation_array_record {
   int n;
   int ref_count;
   ada_internal_doc_annotation items[1];
};

/* Create a length-sized array.  */
extern ada_internal_doc_annotation_array
ada_internal_doc_annotation_array_create(int length);

/* Increment the ref-count for "a".  */
extern void
ada_internal_doc_annotation_array_inc_ref(ada_internal_doc_annotation_array a);

/* Decrement the ref-count for "a". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_internal_doc_annotation_array_dec_ref(ada_internal_doc_annotation_array a);


        



/*

 */
struct ada_node_array_record {
   int n;
   int ref_count;
   ada_node items[1];
};

/* Create a length-sized array.  */
extern ada_node_array
ada_node_array_create(int length);

/* Increment the ref-count for "a".  */
extern void
ada_node_array_inc_ref(ada_node_array a);

/* Decrement the ref-count for "a". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_node_array_dec_ref(ada_node_array a);


        



/*

 */
struct ada_internal_logic_context_array_record {
   int n;
   int ref_count;
   ada_internal_logic_context items[1];
};

/* Create a length-sized array.  */
extern ada_internal_logic_context_array
ada_internal_logic_context_array_create(int length);

/* Increment the ref-count for "a".  */
extern void
ada_internal_logic_context_array_inc_ref(ada_internal_logic_context_array a);

/* Decrement the ref-count for "a". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_internal_logic_context_array_dec_ref(ada_internal_logic_context_array a);


        



/*

 */
struct ada_internal_param_actual_array_record {
   int n;
   int ref_count;
   ada_internal_param_actual items[1];
};

/* Create a length-sized array.  */
extern ada_internal_param_actual_array
ada_internal_param_actual_array_create(int length);

/* Increment the ref-count for "a".  */
extern void
ada_internal_param_actual_array_inc_ref(ada_internal_param_actual_array a);

/* Decrement the ref-count for "a". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_internal_param_actual_array_dec_ref(ada_internal_param_actual_array a);


        



/*

 */
struct ada_internal_ref_result_array_record {
   int n;
   int ref_count;
   ada_internal_ref_result items[1];
};

/* Create a length-sized array.  */
extern ada_internal_ref_result_array
ada_internal_ref_result_array_create(int length);

/* Increment the ref-count for "a".  */
extern void
ada_internal_ref_result_array_inc_ref(ada_internal_ref_result_array a);

/* Decrement the ref-count for "a". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_internal_ref_result_array_dec_ref(ada_internal_ref_result_array a);


        



/*

 */
struct ada_internal_shape_array_record {
   int n;
   int ref_count;
   ada_internal_shape items[1];
};

/* Create a length-sized array.  */
extern ada_internal_shape_array
ada_internal_shape_array_create(int length);

/* Increment the ref-count for "a".  */
extern void
ada_internal_shape_array_inc_ref(ada_internal_shape_array a);

/* Decrement the ref-count for "a". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_internal_shape_array_dec_ref(ada_internal_shape_array a);


        



/*

 */
struct ada_internal_solver_diagnostic_array_record {
   int n;
   int ref_count;
   ada_internal_solver_diagnostic items[1];
};

/* Create a length-sized array.  */
extern ada_internal_solver_diagnostic_array
ada_internal_solver_diagnostic_array_create(int length);

/* Increment the ref-count for "a".  */
extern void
ada_internal_solver_diagnostic_array_inc_ref(ada_internal_solver_diagnostic_array a);

/* Decrement the ref-count for "a". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_internal_solver_diagnostic_array_dec_ref(ada_internal_solver_diagnostic_array a);


        



/*

 */
struct ada_internal_substitution_array_record {
   int n;
   int ref_count;
   ada_internal_substitution items[1];
};

/* Create a length-sized array.  */
extern ada_internal_substitution_array
ada_internal_substitution_array_create(int length);

/* Increment the ref-count for "a".  */
extern void
ada_internal_substitution_array_inc_ref(ada_internal_substitution_array a);

/* Decrement the ref-count for "a". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_internal_substitution_array_dec_ref(ada_internal_substitution_array a);


        



/*

 */
struct ada_analysis_unit_array_record {
   int n;
   int ref_count;
   ada_analysis_unit items[1];
};

/* Create a length-sized array.  */
extern ada_analysis_unit_array
ada_analysis_unit_array_create(int length);

/* Increment the ref-count for "a".  */
extern void
ada_analysis_unit_array_inc_ref(ada_analysis_unit_array a);

/* Decrement the ref-count for "a". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_analysis_unit_array_dec_ref(ada_analysis_unit_array a);


        



/*

 */
struct ada_symbol_type_array_record {
   int n;
   int ref_count;
   ada_symbol_type items[1];
};

/* Create a length-sized array.  */
extern ada_symbol_type_array
ada_symbol_type_array_create(int length);

/* Increment the ref-count for "a".  */
extern void
ada_symbol_type_array_inc_ref(ada_symbol_type_array a);

/* Decrement the ref-count for "a". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_symbol_type_array_dec_ref(ada_symbol_type_array a);



/*
 * Iterator types declarations
 */





/*
 * Set the next value from the iterator in the given element pointer. Return
 * ``1`` if successful, otherwise ``0``.
 *
 * This raises a ``Stale_Reference_Error`` exception if the iterator is
 * invalidated.
 */
extern int
ada_node_iterator_next(ada_node_iterator i, ada_node* e);

/* Increment the ref-count for "i".  */
extern void
ada_node_iterator_inc_ref(ada_node_iterator i);

/* Decrement the ref-count for "i". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_node_iterator_dec_ref(ada_node_iterator i);



        



/*
 * Set the next value from the iterator in the given element pointer. Return
 * ``1`` if successful, otherwise ``0``.
 *
 * This raises a ``Stale_Reference_Error`` exception if the iterator is
 * invalidated.
 */
extern int
ada_internal_completion_item_iterator_next(ada_internal_completion_item_iterator i, ada_internal_completion_item* e);

/* Increment the ref-count for "i".  */
extern void
ada_internal_completion_item_iterator_inc_ref(ada_internal_completion_item_iterator i);

/* Decrement the ref-count for "i". This deallocates it if the ref-count drops
   to 0.  */
extern void
ada_internal_completion_item_iterator_dec_ref(ada_internal_completion_item_iterator i);



/*
 * Analysis primitives
 */

/*
 * Allocate a new analysis context.
 */
extern ada_analysis_context
ada_allocate_analysis_context (void);

/*
 * Initialize an analysis context. Must be called right after
 * ``Allocate_Context`` on its result.
 *
 * Having separate primitives for allocation/initialization allows library
 * bindings to have a context wrapper (created between the two calls) ready
 * when callbacks that happen during context initialization (for instance "unit
 * parsed" events).
 */
extern void
ada_initialize_analysis_context(
   ada_analysis_context context,
   const char *charset,
   ada_file_reader file_reader,
   ada_unit_provider unit_provider,
   ada_event_handler event_handler,
   int with_trivia,
   int tab_stop
);

/*
 * Increase the reference count to an analysis context. Return the reference
 * for convenience.
 */
extern ada_analysis_context
ada_context_incref(ada_analysis_context context);

/*
 * Decrease the reference count to an analysis context. Destruction happens
 * when the ref-count reaches 0.
 */
extern void
ada_context_decref(ada_analysis_context context);

/*
 * If the given string is a valid symbol, yield it as a symbol and return true.
 * Otherwise, return false.
 */
extern int
ada_context_symbol(ada_analysis_context context,
                                   ada_text *text,
                                   ada_symbol_type *symbol);

/*
 * Debug helper. Set whether ``Property_Error`` exceptions raised in
 * ``Populate_Lexical_Env`` should be discarded. They are by default.
 */
extern void
ada_context_discard_errors_in_populate_lexical_env(
        ada_analysis_context context,
        int discard);

/*
 * Create a new analysis unit for ``Filename`` or return the existing one if
 * any. If ``Reparse`` is true and the analysis unit already exists, reparse it
 * from ``Filename``.
 *
 * ``Rule`` controls which grammar rule is used to parse the unit.
 *
 * Use ``Charset`` in order to decode the source. If ``Charset`` is empty then
 * use the context's default charset.
 *
 * If any failure occurs, such as file opening, decoding, lexing or parsing
 * failure, return an analysis unit anyway: errors are described as diagnostics
 * of the returned analysis unit.
 */
extern ada_analysis_unit
ada_get_analysis_unit_from_file(
        ada_analysis_context context,
        const char *filename,
        const char *charset,
        int reparse,
        ada_grammar_rule rule);

/*
 * Create a new analysis unit for ``Filename`` or return the existing one if
 * any. Whether the analysis unit already exists or not, (re)parse it from the
 * source code in ``Buffer``.
 *
 * ``Rule`` controls which grammar rule is used to parse the unit.
 *
 * Use ``Charset`` in order to decode the source. If ``Charset`` is empty then
 * use the context's default charset.
 *
 * If any failure occurs, such as file opening, decoding, lexing or parsing
 * failure, return an analysis unit anyway: errors are described as diagnostics
 * of the returned analysis unit.
 */
extern ada_analysis_unit
ada_get_analysis_unit_from_buffer(
        ada_analysis_context context,
        const char *filename,
        const char *charset,
        const char *buffer,
        size_t buffer_size,
        ada_grammar_rule rule);

/*
 * Create a new analysis unit for ``Name``/``Kind`` or return the existing one
 * if any. If ``Reparse`` is true and the analysis unit already exists, reparse
 * it from the on-disk source file.
 *
 * The ``Name`` and ``Kind`` arguments are forwarded directly to query the
 * context's unit provider and get the filename for the returned unit. See the
 * documentation of the relevant unit provider for their exact semantics.
 *
 * Use ``Charset`` in order to decode the source. If ``Charset`` is empty then
 * use the context's default charset.
 *
 * If the unit name cannot be tuned into a file name, return ``NULL``. If any
 * other failure occurs, such as file opening, decoding, lexing or parsing
 * failure, return an analysis unit anyway: errors are described as diagnostics
 * of the returned analysis unit.
 */
extern ada_analysis_unit
ada_get_analysis_unit_from_provider(
        ada_analysis_context context,
        ada_text *name,
        ada_analysis_unit_kind kind,
        const char *charset,
        int reparse);

/*
 * Return the root node for this unit, or ``NULL`` if there is none.
 */
extern void
ada_unit_root(ada_analysis_unit unit,
                              ada_node *result_p);

/*
 * Return a reference to the first token scanned in this unit.
 */
extern void
ada_unit_first_token(ada_analysis_unit unit,
                                     ada_token *token);

/*
 * Return a reference to the last token scanned in this unit.
 */
extern void
ada_unit_last_token(ada_analysis_unit unit,
                                    ada_token *token);

/*
 * Return the number of tokens in this unit.
 */
extern int
ada_unit_token_count(ada_analysis_unit unit);

/*
 * Return the number of trivias in this unit. This is 0 for units that were
 * parsed with trivia analysis disabled.
 */
extern int
ada_unit_trivia_count(ada_analysis_unit unit);

/*
 * Debug helper: output the lexical envs for the given analysis unit.
 */
extern void
ada_unit_dump_lexical_env(ada_analysis_unit unit);

/*
 * Return the filename this unit is associated to.
 *
 * The returned string is dynamically allocated and the caller must free it
 * when done with it.
 */
extern char *
ada_unit_filename(ada_analysis_unit unit);

/*
 * Return the number of diagnostics associated to this unit.
 */
extern unsigned
ada_unit_diagnostic_count(ada_analysis_unit unit);

/*
 * Get the Nth diagnostic in this unit and store it into ``*diagnostic_p``.
 * Return zero on failure (when N is too big).
 */
extern int
ada_unit_diagnostic(ada_analysis_unit unit,
                                    unsigned n,
                                    ada_diagnostic *diagnostic_p);

/*
 * Return the context that owns this unit.
 */
extern ada_analysis_context
ada_unit_context(ada_analysis_unit context);

/*
 * Reparse an analysis unit from the associated file.
 *
 * Use ``Charset`` in order to decode the source. If ``Charset`` is empty then
 * use the context's default charset.
 *
 * If any failure occurs, such as decoding, lexing or parsing failure,
 * diagnostic are emitted to explain what happened.
 */
extern void
ada_unit_reparse_from_file(ada_analysis_unit unit,
                                           const char *charset);

/*
 * Reparse an analysis unit from a buffer.
 *
 * Use ``Charset`` in order to decode the source. If ``Charset`` is empty then
 * use the context's default charset.
 *
 * If any failure occurs, such as decoding, lexing or parsing failure,
 * diagnostic are emitted to explain what happened.
 */
extern void
ada_unit_reparse_from_buffer (ada_analysis_unit unit,
                                              const char *charset,
                                              const char *buffer,
                                              size_t buffer_size);

/*
 * Create lexical environments for this analysis unit, according to the
 * specifications given in the language spec.
 *
 * If not done before, it will be automatically called during semantic
 * analysis. Calling it before enables one to control where the latency occurs.
 *
 * Depending on whether errors are discarded (see
 * ``Discard_Errors_In_Populate_Lexical_Env``), return ``0`` on failure and
 * ``1`` on success.
 */
extern int
ada_unit_populate_lexical_env(
    ada_analysis_unit unit
       , int ple_root_index
);

/*
 * General AST node primitives
 */

/*
 * Create an entity with null entity info for a given node.
 */
extern void
ada_create_bare_entity(
    ada_base_node node,
    ada_node *entity
);

/*
 * Return whether this node is a null node reference.
 */
static inline int
ada_node_is_null(ada_node *node) {
    return node->node == NULL;
}

/*
 * Return the kind of this node.
 */
extern ada_node_kind_enum
ada_node_kind(ada_node *node);

/*
 * Helper for textual dump: return the kind name for this node. The returned
 * string is a copy and thus must be free'd by the caller.
 */
extern void
ada_kind_name(ada_node_kind_enum kind, ada_text *result);

/*
 * Return the analysis unit that owns this node.
 */
extern ada_analysis_unit
ada_node_unit(ada_node *node);

/*
 * Return a hash for the given node.
 */
extern uint32_t
ada_node_hash(ada_node *node);

/*
 * Return whether the two nodes are equivalent.
 */
extern ada_bool
ada_node_is_equivalent(ada_node *l, ada_node *r);

/*
 * Return whether this node is a node that contains only a single token.
 */
extern int
ada_node_is_token_node(ada_node *node);

/*
 * Return whether this node is synthetic.
 */
extern int
ada_node_is_synthetic(ada_node *node);

/*
 * Return a representation of this node as a string.
 */
extern void
ada_node_image(ada_node *node,
                               ada_text *result);

/*
 * Return the source buffer slice corresponding to the text that spans between
 * the first and the last tokens of this node.
 *
 * Note that this returns the empty string for synthetic nodes.
 */
extern void
ada_node_text(ada_node *node,
                              ada_text *text);

/*
 * Return the spanning source location range for this node.
 *
 * Note that this returns the sloc of the parent for synthetic nodes.
 */
extern void
ada_node_sloc_range(ada_node *node,
                                    ada_source_location_range *sloc_range);

/*
 * Return the bottom-most node from in ``Node`` and its children which contains
 * ``Sloc``, or ``NULL`` if there is none.
 */
extern void
ada_lookup_in_node(ada_node *node,
                                   const ada_source_location *sloc,
                                   ada_node *result_p);

/*
 * Return the number of children in this node.
 */
extern unsigned
ada_node_children_count(ada_node *node);

/*
 * Return the Nth child for in this node's fields and store it into
 * ``*child_p``.  Return zero on failure (when ``N`` is too big).
 */
extern int
ada_node_child(ada_node *node,
                               unsigned n,
                               ada_node* child_p);

/*
 * Encode some text using the current locale. The result is dynamically
 * allocated: it is up to the caller to free it when done with it.
 *
 * This is a development helper to make it quick and easy to print token and
 * diagnostic text: it ignores errors (when the locale does not support some
 * characters). Production code should use real conversion routines such as
 * libiconv's in order to deal with UTF-32 texts.
 */
extern char *
ada_text_to_locale_string(ada_text *text);

/*
 * Encode some text to a newly allocated UTF-8 buffer (``bytes``). The size of
 * this buffer is stored in ``length``, and the actual allocated buffer has one
 * extra NUL byte (note that it is valid for the first ``length`` bytes in
 * ``bytes`` to contain NUL bytes).
 */
extern void
ada_text_to_utf8(ada_text *text,
                                 char **bytes,
                                 size_t *length);

/*
 * Decode a UTF-8 buffer (``bytes``, of size ``length``) to a text buffer.
 */
extern void
ada_text_from_utf8(const char *bytes,
                                   size_t length,
                                   ada_text *text);

/*
 * Encode the given character to a newly allocated UTF-8 buffer (``bytes``).
 * The size of this buffer is stored in ``length``.
 */
extern void
ada_char_to_utf8(uint32_t codepoint,
                                 char **bytes,
                                 size_t *length);

/*
 * Decode a UTF-8 buffer (``bytes``, of size ``length``) to a text buffer. Note
 * that the UTF-8 buffer is supposed to contain only one codepoint.
 */
extern void
ada_char_from_utf8(const char *bytes,
                                   size_t length,
                                   uint32_t *codepoint);

/*
 * Encode some string to a newly allocated UTF-8 buffer (``bytes``). The size
 * of this buffer is stored in ``length``, and the actual allocated buffer has
 * one extra NUL byte (note that it is valid for the first ``length`` bytes in
 * ``bytes`` to contain NUL bytes).
 */
extern void
ada_string_to_utf8(ada_string_type string,
                                   char **bytes,
                                   size_t *length);

/*
 * Decode a UTF-8 buffer (``bytes``, of size ``length``) to a string buffer.
 */
extern void
ada_string_from_utf8(const char *bytes,
                                     size_t length,
                                     ada_string_type *string);

/*
 * Free dynamically allocated memory.
 *
 * This is a helper to free objects from dynamic languages.
 */
extern void
ada_free(void *address);

/*
 * If this text object owns the buffer it references, free this buffer.
 *
 * Note that even though this accepts a pointer to a text object, it does not
 * deallocates the text object itself but rather the buffer it references.
 */
extern void
ada_destroy_text(ada_text *text);

/*
 * Return the text associated to this symbol.
 */
extern void
ada_symbol_text(ada_symbol_type *symbol,
                                ada_text *text);

/*
 * Create a big integer from its string representation (in base 10).
 */
extern ada_big_integer
ada_create_big_integer(ada_text *text);

/*
 * Return the string representation (in base 10) of this big integer.
 */
extern void
ada_big_integer_text(ada_big_integer bigint,
                                     ada_text *text);

/*
 * Decrease the reference count for this big integer.
 */
extern void
ada_big_integer_decref(ada_big_integer bigint);

/*
 * Allocate strings to represent the library version number and build date and
 * put them in Version/Build_Date. Callers are expected to call free() on the
 * returned string once done.
 */
extern void
ada_get_versions(char **version, char **build_date);

/*
 * Create a string value from its content (UTF32 with native endianity).
 *
 * Note that the CONTENT buffer argument is copied: the returned value does not
 * contain a reference to it.
 */
extern ada_string_type
ada_create_string(uint32_t *content, int length);

/*
 * Decrease the reference count for this string.
 */
extern void
ada_string_dec_ref(ada_string_type self);

/*
 * Kind-specific AST node primitives
 */

/* All these primitives return their result through an OUT parameter.  They
   return a boolean telling whether the operation was successful (it can fail
   if the node does not have the proper type, for instance).  When an AST node
   is returned, its ref-count is left as-is.  */

        



/*
 * Return the scope of definition of this basic declaration.
 */
extern int ada_ada_node_p_declarative_scope(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the compilation unit containing this node.
 *
 * .. note:: This returns the ``ada_compilation_unit`` node, which is different
 *    from the ``AnalysisUnit``. In particular, an analysis unit can contain
 *    multiple compilation units.
 */
extern int ada_ada_node_p_enclosing_compilation_unit(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Assuming this node comes from an instantiated generic declaration, return
 * its non-instantiated counterpart lying in the generic declaration.
 */
extern int ada_ada_node_p_get_uninstantiated_node(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return possible completions at this point in the file.
 */
extern int ada_ada_node_p_complete(
    ada_node *node,


    ada_internal_completion_item_iterator *value_p
);


        



/*
 * Return the list of keywords that are valid at this point in the file.
 *
 * .. note:: This is work in progress. It will return all keywords for now,
 *    without looking at the context.
 */
extern int ada_ada_node_p_valid_keywords(
    ada_node *node,


    ada_symbol_type_array *value_p
);


        



/*
 * Return the potentially empty list of generic package/subprogram
 * instantiations that led to the creation of this entity. Outer-most
 * instantiations appear last.
 */
extern int ada_ada_node_p_generic_instantiations(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * Return the semantic parent for this node, if applicable, null otherwise.
 *
 * .. note:: A node lying outside of a library item's declaration or subunit's
 *    body does not have a parent environment, meaning that this property will
 *    return null.
 */
extern int ada_ada_node_p_semantic_parent(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the parent basic decl for this node, if applicable, null otherwise.
 *
 * .. note:: If the parent BasicDecl of the given node is a generic
 *    declaration, this call will return the instantiation from which the node
 *    was retrieved instead, if any. This also applies to bodies of generic
 *    declarations.
 *
 * .. note:: When called on a subunit's body, this property will return its
 *    corresponding body stub.
 *
 * .. note:: When called on a node lying outside of a library item's
 *    declaration or subunit's body this property will return null.
 */
extern int ada_ada_node_p_parent_basic_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Returns whether this subprogram has explicitly been set as having
 * ``Spark_Mode`` to ``On``, directly or indirectly.
 *
 * Doesn't include subprograms that can be inferred by GNATprove as being
 * SPARK.
 */
extern int ada_ada_node_p_has_spark_mode_on(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Returns whether this subprogram body is subject to proof in the context of
 * the SPARK/GNATprove tools.
 */
extern int ada_ada_node_p_is_subject_to_proof(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Filters out among the list of given units those that cannot refer to the
 * unit in which this node lies. If transitive is True, the whole transitive
 * closure of imports will be used to find a reference to the unit of this
 * node.
 */
extern int ada_ada_node_p_filter_is_imported_by(
    ada_node *node,

        
        ada_analysis_unit_array
        units,
        
        ada_bool
        transitive,

    ada_analysis_unit_array *value_p
);


        



/*
 * This will resolve names for this node. If the operation is successful, then
 * type_var and ref_var will be bound on appropriate subnodes of the statement.
 */
extern int ada_ada_node_p_resolve_names(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * If name resolution on this xref entry point fails, this returns all the
 * diagnostics that were produced while resolving it.
 */
extern int ada_ada_node_p_nameres_diagnostics(
    ada_node *node,


    ada_internal_solver_diagnostic_array *value_p
);


        



/*
 * Static method. Return the analysis unit corresponding to the Standard
 * package.
 */
extern int ada_ada_node_p_standard_unit(
    ada_node *node,


    ada_analysis_unit *value_p
);


        



/*
 * Static method. Return whether the given token is considered a keyword in the
 * given version of Ada. Supported values for the language version argument
 * are: "Ada_83", "Ada_95", "Ada_2005", "Ada_2012", "Ada_2022".
 */
extern int ada_ada_node_p_is_keyword(
    ada_node *node,

        
        const ada_token*
        token,
        
        const ada_symbol_type*
        language_version,

    ada_bool *value_p
);


        



/*
 * Static property. Return an entity from the standard package with name
 * ``sym``.
 */
extern int ada_ada_node_p_std_entity(
    ada_node *node,

        
        const ada_symbol_type*
        sym,

    ada_node *value_p
);


        



/*
 * Static method. Return the standard Boolean type.
 */
extern int ada_ada_node_p_bool_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Static method. Return the standard Integer type.
 */
extern int ada_ada_node_p_int_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Static method. Return the standard Universal Integer type.
 */
extern int ada_ada_node_p_universal_int_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Static method. Return the standard Universal Real type.
 */
extern int ada_ada_node_p_universal_real_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Static method. Return the standard Character type.
 */
extern int ada_ada_node_p_std_char_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Static method. Return the standard Wide_Character type.
 */
extern int ada_ada_node_p_std_wide_char_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Static method. Return the standard Wide_Wide_Character type.
 */
extern int ada_ada_node_p_std_wide_wide_char_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Static method. Return the standard String type.
 */
extern int ada_ada_node_p_std_string_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Static method. Return the standard Wide_String type.
 */
extern int ada_ada_node_p_std_wide_string_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Static method. Return the standard Wide_Wide_String type.
 */
extern int ada_ada_node_p_std_wide_wide_string_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Static method. Get the top-level decl in ``unit``.  This is the body of a
 * Subunit, or the item of a ``LibraryItem``.
 */
extern int ada_ada_node_p_top_level_decl(
    ada_node *node,

        
        ada_analysis_unit
        unit,

    ada_node *value_p
);


        



/*
 * Assuming that self is a choice expression (such as what can appear in an
 * alternative of a case statement or in the RHS of a membership expression,
 * this property returns whether the given value satisfies it.
 *
 * .. attention:: This is an experimental feature, so even if it is exposed to
 *    allow experiments, it is totally unsupported and the API and behavior are
 *    very likely to change in the future.
 */
extern int ada_ada_node_p_choice_match(
    ada_node *node,

        
        const ada_big_integer*
        value,

    ada_bool *value_p
);


        



/*
 * Return a cross reference from this name to a defining identifier, trying to
 * mimic GNAT's xrefs as much as possible.
 */
extern int ada_ada_node_p_gnat_xref(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Helper for the ``has_spark_mode_on`` and ``is_subject_to_proof`` properties.
 *
 * This property will get the applicable aspect defining the SPARK_Mode for the
 * given node, recursing syntactically and taking into account configuration
 * files.
 *
 * This only implements the base logic for recursing up the tree: nodes that
 * need a specific logic must override it. See for example
 * ``BasicDecl.spark_mode_aspect``.
 */
extern int ada_ada_node_p_spark_mode_aspect(
    ada_node *node,


    ada_internal_aspect *value_p
);


        



/*
 * Designates entities that are entry point for the xref solving
 * infrastructure. If this returns true, then resolve_names can be called on
 * it.
 *
 * .. note:: For convenience, and unlike what is defined in the ARM wrt.
 *    complete contexts for name resolution, ``xref_entry_points`` can be
 *    nested.
 */
extern int ada_ada_node_p_xref_entry_point(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return whether this node is a ``UsePackageClause`` that follows a
 * ``WithClause`` for the same unit.
 */
extern int ada_ada_node_p_matching_with_use_clause(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return the syntactic parent for this node. Return null for the root node.
 */
extern int ada_ada_node_parent(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return an array that contains the lexical parents, this node included iff
 * ``with_self`` is True. Nearer parents are first in the list.
 */
extern int ada_ada_node_parents(
    ada_node *node,

        
        ada_bool
        with_self,

    ada_node_array *value_p
);


        



/*
 * Return an array that contains the direct lexical children.
 *
 * .. warning:: This constructs a whole array every-time you call it, and as
 *    such is less efficient than calling the ``Child`` built-in.
 */
extern int ada_ada_node_children(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * Return the first token used to parse this node.
 */
extern int ada_ada_node_token_start(
    ada_node *node,


    ada_token *value_p
);


        



/*
 * Return the last token used to parse this node.
 */
extern int ada_ada_node_token_end(
    ada_node *node,


    ada_token *value_p
);


        



/*
 * Return the 0-based index for Node in its parent's children.
 */
extern int ada_ada_node_child_index(
    ada_node *node,


    int *value_p
);


        



/*
 * Return the node's previous sibling, or null if there is no such sibling.
 */
extern int ada_ada_node_previous_sibling(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the node's next sibling, or null if there is no such sibling.
 */
extern int ada_ada_node_next_sibling(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the analysis unit owning this node.
 */
extern int ada_ada_node_unit(
    ada_node *node,


    ada_analysis_unit *value_p
);


        



/*
 * Return whether the node is a ghost.
 *
 * Unlike regular nodes, ghost nodes cover no token in the input source: they
 * are logically located instead between two tokens. Both the ``token_start``
 * and the ``token_end`` of all ghost nodes is the token right after this
 * logical position.
 */
extern int ada_ada_node_is_ghost(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return a string containing the filename + the sloc in GNU conformant format.
 * Useful to create diagnostics from a node.
 */
extern int ada_ada_node_full_sloc_image(
    ada_node *node,


    ada_string_type *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_abort_node_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_abstract_node_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Returns an array of pairs, associating formal parameters to actual
 * expressions. The formals to match are retrieved by resolving the call which
 * this AssocList represents the actuals of.
 */
extern int ada_assoc_list_p_zip_with_params(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_internal_param_actual_array *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_aliased_node_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_all_node_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_attribute_ref``, ``ada_bin_op``, ``ada_call_expr``,
 * ``ada_char_literal``, ``ada_dotted_name``, ``ada_explicit_deref``,
 * ``ada_identifier``, ``ada_qual_expr``, ``ada_reduce_attribute_ref``,
 * ``ada_string_literal``, ``ada_subtype_indication``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_constrained_array_indices_f_list(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_unconstrained_array_indices_f_types(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_aspect_assoc_f_id(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes:
 * ``ada_abstract_state_decl_expr``, ``ada_allocator``, ``ada_attribute_ref``,
 * ``ada_base_aggregate``, ``ada_bin_op``, ``ada_call_expr``,
 * ``ada_char_literal``, ``ada_concat_op``, ``ada_cond_expr``,
 * ``ada_contract_cases``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_aspect_assoc_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return whether this aspect is ghost code or not. See SPARK RM 6.9.
 */
extern int ada_aspect_assoc_p_is_ghost_code(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_char_literal``,
 * ``ada_identifier``, ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_at_clause_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_at_clause_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_attribute_def_clause_f_attribute_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_attribute_def_clause_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_identifier``, ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_enum_rep_clause_f_type_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_enum_rep_clause_f_aggregate(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Returns an array of pairs, associating enum literals to representation
 * clause actuals.
 */
extern int ada_enum_rep_clause_p_params(
    ada_node *node,


    ada_internal_param_actual_array *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_identifier``, ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_record_rep_clause_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_null_literal``, ``ada_num_literal``, ``ada_paren_expr``,
 * ``ada_qual_expr``, ``ada_quantified_expr``, ``ada_raise_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_record_rep_clause_f_at_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_component_clause``, ``ada_pragma_node``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_record_rep_clause_f_components(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_aspect_spec_f_aspect_assocs(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Returns the expression side of this assoc node.
 */
extern int ada_base_assoc_p_assoc_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_others_designator``, ``ada_paren_expr``, ``ada_qual_expr``,
 * ``ada_quantified_expr``, ``ada_raise_expr``, ``ada_reduce_attribute_ref``,
 * ``ada_string_literal``, ``ada_target_name``, ``ada_un_op``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_contract_case_assoc_f_guard(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_contract_case_assoc_f_consequence(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_identifier``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_pragma_argument_assoc_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_pragma_argument_assoc_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the list of abstract formal parameters for this holder.
 */
extern int ada_base_formal_param_holder_p_abstract_formal_params(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * Return all parameters as a ``DefiningName`` array. This property doesn't
 * return record discriminants nor variants when called on a record component
 * list.
 */
extern int ada_base_formal_param_holder_p_formal_params(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * Return the minimum number of parameters this subprogram can be called while
 * still being a legal call.
 */
extern int ada_base_formal_param_holder_p_nb_min_params(
    ada_node *node,


    int *value_p
);


        



/*
 * Return the maximum number of parameters this subprogram can be called while
 * still being a legal call.
 */
extern int ada_base_formal_param_holder_p_nb_max_params(
    ada_node *node,


    int *value_p
);


        



/*
 * Returns the type of each parameter of self.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_formal_param_holder_p_param_types(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_node_array *value_p
);


        



/*
 * Syntax property. Return the name of the subprogram defined by this
 * specification.
 */
extern int ada_base_subp_spec_p_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Syntax property. Return the type expression node corresponding to the return
 * of this subprogram spec.
 */
extern int ada_base_subp_spec_p_returns(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the types of which this subprogram is a primitive of.
 */
extern int ada_base_subp_spec_p_primitive_subp_types(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node_array *value_p
);


        



/*
 * Return the first type of which this subprogram is a primitive of.
 */
extern int ada_base_subp_spec_p_primitive_subp_first_type(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * If this subprogram is a primitive for a tagged type, then return this type.
 */
extern int ada_base_subp_spec_p_primitive_subp_tagged_type(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Returns the return type of self, if applicable (e.g. if self is a
 * subprogram). Else, returns null.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_subp_spec_p_return_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_node *value_p
);


        



/*
 * Returns the array of parameters specification for this subprogram spec.
 */
extern int ada_base_subp_spec_p_params(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_entry_spec_f_entry_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_bin_op``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_identifier``,
 * ``ada_qual_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_subtype_indication``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_entry_spec_f_family_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_entry_spec_f_entry_params(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_subp_spec_f_subp_kind(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_subp_spec_f_subp_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_subp_spec_f_subp_params(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_anonymous_type``,
 * ``ada_subtype_indication``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_subp_spec_f_subp_returns(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_synthetic_binary_spec_f_left_param(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_synthetic_binary_spec_f_right_param(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_synthetic_binary_spec_f_return_type_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_synthetic_unary_spec_f_right_param(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_synthetic_unary_spec_f_return_type_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_aspect_clause``, ``ada_component_decl``, ``ada_null_component_decl``,
 * ``ada_pragma_node``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_component_list_f_components(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_component_list_f_variant_part(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_known_discriminant_part_f_discr_specs(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_entry_completion_formal_params_f_params(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_generic_formal``, ``ada_pragma_node``, ``ada_use_clause``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_formal_part_f_decls(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_base_record_def_f_components(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the list of parameters that this association refers to.
 */
extern int ada_basic_assoc_p_get_params(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node_array *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_allocator``, ``ada_array_subcomponent_choice_name``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_discrete_subtype_indication``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_membership_expr``, ``ada_null_literal``,
 * ``ada_num_literal``, ``ada_others_designator``, ``ada_paren_expr``,
 * ``ada_qual_expr``, ``ada_quantified_expr``, ``ada_raise_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_aggregate_assoc_f_designators(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_box_expr``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_concat_op``, ``ada_cond_expr``, ``ada_decl_expr``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_membership_expr``, ``ada_null_literal``,
 * ``ada_num_literal``, ``ada_paren_expr``, ``ada_qual_expr``,
 * ``ada_quantified_expr``, ``ada_raise_expr``, ``ada_reduce_attribute_ref``,
 * ``ada_string_literal``, ``ada_target_name``, ``ada_un_op``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_aggregate_assoc_f_r_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_identifier_list``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_composite_constraint_assoc_f_ids(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_discrete_subtype_indication``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_membership_expr``, ``ada_null_literal``,
 * ``ada_num_literal``, ``ada_paren_expr``, ``ada_qual_expr``,
 * ``ada_quantified_expr``, ``ada_raise_expr``, ``ada_reduce_attribute_ref``,
 * ``ada_string_literal``, ``ada_target_name``, ``ada_un_op``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_composite_constraint_assoc_f_constraint_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_iterated_assoc_f_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_iterated_assoc_f_key_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_iterated_assoc_f_r_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_identifier``,
 * ``ada_others_designator``, ``ada_string_literal``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_param_assoc_f_designator(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_box_expr``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_concat_op``, ``ada_cond_expr``, ``ada_decl_expr``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_membership_expr``, ``ada_null_literal``,
 * ``ada_num_literal``, ``ada_paren_expr``, ``ada_qual_expr``,
 * ``ada_quantified_expr``, ``ada_raise_expr``, ``ada_reduce_attribute_ref``,
 * ``ada_string_literal``, ``ada_target_name``, ``ada_un_op``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_param_assoc_f_r_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_basic_decl_f_aspects(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Whether this decl is the nested decl of a generic formal declaration.
 */
extern int ada_basic_decl_p_is_formal(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return the documentation annotations associated with this decl. Annotations
 * are any comment line of the form:
 *
 * .. code::
 *
 *    --% [annotation_name]: [annotation]
 *
 * Raises a property error if the doc is incorrectly formatted.
 *
 * .. attention:: This is an experimental feature, so even if it is exposed to
 *    allow experiments, it is totally unsupported and the API and behavior are
 *    very likely to change in the future.
 */
extern int ada_basic_decl_p_doc_annotations(
    ada_node *node,


    ada_internal_doc_annotation_array *value_p
);


        



/*
 * Return the documentation associated with this decl. Raises a property error
 * if the doc is incorrectly formatted.
 *
 * .. attention:: This is an experimental feature, so even if it is exposed to
 *    allow experiments, it is totally unsupported and the API and behavior are
 *    very likely to change in the future.
 */
extern int ada_basic_decl_p_doc(
    ada_node *node,


    ada_string_type *value_p
);


        



/*
 * Return the canonical part for this decl. In the case of decls composed of
 * several parts, the canonical part will be the first part.
 */
extern int ada_basic_decl_p_canonical_part(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return all parts that define this entity, sorted from first part to last
 * part.
 */
extern int ada_basic_decl_p_all_parts(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node_array *value_p
);


        



/*
 * Return the aspect with name ``name`` for this entity.
 */
extern int ada_basic_decl_p_get_aspect_assoc(
    ada_node *node,

        
        const ada_symbol_type*
        name,

    ada_node *value_p
);


        



/*
 * Return the expression associated to the aspect with name ``name`` for this
 * entity.
 */
extern int ada_basic_decl_p_get_aspect_spec_expr(
    ada_node *node,

        
        const ada_symbol_type*
        name,

    ada_node *value_p
);


        



/*
 * Return the aspect with name ``name`` associated to this entity.
 *
 * Aspects are properties of entities that can be specified by the Ada program,
 * either via aspect specifications, pragmas, or attributes.
 *
 * See ``DefiningName.P_Get_Aspect`` for more details.
 */
extern int ada_basic_decl_p_get_aspect(
    ada_node *node,

        
        const ada_symbol_type*
        name,
        
        ada_bool
        previous_parts_only,
        
        ada_bool
        imprecise_fallback,

    ada_internal_aspect *value_p
);


        



/*
 * Returns whether the boolean aspect named ``name`` is set on the entity
 * represented by this node.
 *
 * Aspects are properties of entities that can be specified by the Ada program,
 * either via aspect specifications, pragmas, or attributes.
 *
 * "Aspect" is used as in RM terminology (see :rmlink:`13`).
 */
extern int ada_basic_decl_p_has_aspect(
    ada_node *node,

        
        const ada_symbol_type*
        name,
        
        ada_bool
        previous_parts_only,
        
        ada_bool
        imprecise_fallback,

    ada_bool *value_p
);


        



/*
 * Return the pragma with name ``name`` associated to this entity.
 *
 * Please use the ``p_get_aspect`` property instead if you are interested in
 * aspects, i.e. information that can be represented by either aspect
 * specification nodes, pragma nodes or attribute definition nodes.
 */
extern int ada_basic_decl_p_get_pragma(
    ada_node *node,

        
        const ada_symbol_type*
        name,

    ada_node *value_p
);


        



/*
 * Return the representation clause associated to this type decl that defines
 * the given attribute name.
 */
extern int ada_basic_decl_p_get_representation_clause(
    ada_node *node,

        
        const ada_symbol_type*
        name,
        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return the at clause associated to this declaration.
 */
extern int ada_basic_decl_p_get_at_clause(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return all the ``Annotate`` aspects defined on this entity, both through
 * pragmas and aspect specifications. For a type declaration, this also
 * includes all annotations defined on its from a base type, when relevant (the
 * field ``inherited`` will be set for those). See
 * ``DefiningName.P_Get_Annotations`` for more details.
 */
extern int ada_basic_decl_p_get_annotations(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_internal_aspect_array *value_p
);


        



/*
 * Return whether this declaration is ghost code or not. See SPARK RM 6.9.
 */
extern int ada_basic_decl_p_is_ghost_code(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Whether a BasicDecl is the root decl for its unit.
 */
extern int ada_basic_decl_p_is_compilation_unit_root(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return whether this declaration is visible from the point of view of the
 * given ``origin`` node.
 *
 * .. attention:: Only package-level (public or private) declarations are
 *    supported for now.
 */
extern int ada_basic_decl_p_is_visible(
    ada_node *node,

        
        const ada_node*
        from_node,

    ada_bool *value_p
);


        



/*
 * If self declares a primitive subprogram of some tagged type T, return the
 * set of all subprogram declarations that it overrides (including itself).
 *
 * .. note:: for the moment this only works for tagged types. Remains to be
 *    seen if we need to extend it.
 */
extern int ada_basic_decl_p_base_subp_declarations(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node_array *value_p
);


        



/*
 * If self declares a primitive subprogram of some tagged type T, return the
 * root subprogram declarations that it overrides. There can be several, as in
 * the following scenario:
 *
 * * package Root defines the root tagged type T and subprogram Foo.
 *
 * * package Itf defines interface I and abstract subprogram Foo.
 *
 * * package D defines "type U is new Root.T and Itf.I" and an overriding
 *   subprogram Foo.
 *
 * Here, root_subp_declarations of Foo defined in package D will return both
 * Foo from package Root and Foo from package Itf.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_basic_decl_p_root_subp_declarations(
    ada_node *node,

        
        const ada_node*
        origin,
        
        ada_bool
        imprecise_fallback,

    ada_node_array *value_p
);


        



/*
 * If self is the declaration of a primitive of some type T, return the list of
 * all subprogram that override this subprogram among the given units.
 */
extern int ada_basic_decl_p_find_all_overrides(
    ada_node *node,

        
        ada_analysis_unit_array
        units,
        
        ada_bool
        imprecise_fallback,

    ada_node_array *value_p
);


        



/*
 * Get all the names of this basic declaration.
 */
extern int ada_basic_decl_p_defining_names(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * Get the name of this declaration. If this declaration has several names, it
 * will return the first one.
 */
extern int ada_basic_decl_p_defining_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * If self is a Subp, returns the specification of this subprogram.
 *
 * If ``follow_generic`` is True, will also work for instances of
 * ``GenericSubpDecl``.
 */
extern int ada_basic_decl_p_subp_spec_or_null(
    ada_node *node,

        
        ada_bool
        follow_generic,

    ada_node *value_p
);


        



/*
 * Return True if self is a subprogram node in the general sense (which is, an
 * entity that can be called). This includes separates and entries.
 *
 * .. attention: This is a purely syntactic query and will return True for
 *    everything that is a syntactic entity that can be called like a
 *    subprogram in some contexts, even generic formal subprograms for example.
 */
extern int ada_basic_decl_p_is_subprogram(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return whether this declaration corresponds to a subprogram that represents
 * a predefined operator.
 */
extern int ada_basic_decl_p_is_predefined_operator(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return the relative name for self. If self's defining name is ``A.B.C``,
 * return ``C`` as a node.
 */
extern int ada_basic_decl_p_relative_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the relative name for self, as text.
 */
extern int ada_basic_decl_p_relative_name_text(
    ada_node *node,


    ada_symbol_type *value_p
);


        



/*
 * Return the body corresponding to this declaration, if applicable.
 *
 * .. note:: It is not named body_part, subclasses have more precise versions
 *    named body_part and returning a more precise result. Probably, we want to
 *    rename the specific versions, and have the root property be named
 *    body_part. (TODO R925-008)
 */
extern int ada_basic_decl_p_body_part_for_decl(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Given an origin node and the entity represented by self, this property
 * returns the most visible completion of self that can be seen by origin,
 * according to Ada's visibility rules.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_basic_decl_p_most_visible_part(
    ada_node *node,

        
        const ada_node*
        origin,
        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return the fully qualified name corresponding to this declaration, as an
 * array of symbols.
 */
extern int ada_basic_decl_p_fully_qualified_name_array(
    ada_node *node,

        
        ada_bool
        include_profile,

    ada_symbol_type_array *value_p
);


        



/*
 * Return the fully qualified name corresponding to this declaration.
 */
extern int ada_basic_decl_p_fully_qualified_name(
    ada_node *node,


    ada_string_type *value_p
);


        



/*
 * Return a canonical representation of the fully qualified name corresponding
 * to this declaration.
 */
extern int ada_basic_decl_p_canonical_fully_qualified_name(
    ada_node *node,


    ada_string_type *value_p
);


        



/*
 * Return a unique identifying name for this declaration, provided this
 * declaration is a public declaration. In the case of subprograms, this will
 * include the profile.
 *
 * .. attention:: This will only return a unique name for public declarations.
 *    Notably, anything nested in an unnamed declare block won't be handled
 *    correctly.
 */
extern int ada_basic_decl_p_unique_identifying_name(
    ada_node *node,


    ada_string_type *value_p
);


        



/*
 * Return the previous part for this decl, if applicable.
 *
 * .. note:: It is not named previous_part, because BaseTypeDecl has a more
 *    precise version of previous_part that returns a BaseTypeDecl. Probably,
 *    we want to rename the specific versions, and have the root property be
 *    named previous_part. (TODO R925-008)
 */
extern int ada_basic_decl_p_previous_part_for_decl(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return whether this declaration is static.
 */
extern int ada_basic_decl_p_is_static_decl(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_bool *value_p
);


        



/*
 * Whether this declaration is imported from another language.
 */
extern int ada_basic_decl_p_is_imported(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return the type expression for this BasicDecl if applicable, a null
 * otherwise.
 */
extern int ada_basic_decl_p_type_expression(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the next part of this declaration, if applicable.
 *
 * .. note:: It is not named next_part, because BaseTypeDecl has a more precise
 *    version of next_part that returns a BaseTypeDecl. Probably, we want to
 *    rename the specific versions, and have the root property be named
 *    next_part. (TODO R925-008)
 */
extern int ada_basic_decl_p_next_part_for_decl(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return whether this object is constant or not.
 */
extern int ada_basic_decl_p_is_constant_object(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_abstract_state_decl_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the expression wrapped by this declaration.
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_anonymous_expr_decl_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the generic formal object declaration corresponding to this actual.
 */
extern int ada_anonymous_expr_decl_p_get_formal(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return the type for this formal.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_formal_param_decl_p_formal_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_component_decl_f_ids(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_component_decl_f_component_def(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_component_decl_f_default_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_discriminant_spec_f_ids(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_anonymous_type``,
 * ``ada_subtype_indication``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_discriminant_spec_f_type_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_discriminant_spec_f_default_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_expr_function``,
 * ``ada_formal_subp_decl``, ``ada_formal_type_decl``,
 * ``ada_generic_instantiation``, ``ada_incomplete_formal_type_decl``,
 * ``ada_object_decl``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_formal_f_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_param_spec_f_ids(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_param_spec_f_has_aliased(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_param_spec_f_mode(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_anonymous_type``,
 * ``ada_subtype_indication``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_param_spec_f_type_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_param_spec_f_default_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_synthetic_formal_param_decl_f_param_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_base_package_decl_f_package_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_base_package_decl_f_public_part(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_base_package_decl_f_private_part(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_base_package_decl_f_end_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the PackageBody corresponding to this node.
 */
extern int ada_base_package_decl_p_body_part(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_base_type_decl_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * If this type decl is a subtype decl, return the base subtype. If not, return
 * ``self``.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_base_subtype(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_node *value_p
);


        



/*
 * Return the subprogram declaration denoted by this attribute name and defined
 * on this type.
 */
extern int ada_base_type_decl_p_attribute_subprogram(
    ada_node *node,

        
        const ada_symbol_type*
        attr_name,

    ada_node *value_p
);


        



/*
 * Return the private completion for this type, if there is one.
 */
extern int ada_base_type_decl_p_private_completion(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Assuming that P is a primitive of self, return whether the given primitive P
 * is inherited from one of self's parents.
 */
extern int ada_base_type_decl_p_is_inherited_primitive(
    ada_node *node,

        
        const ada_node*
        p,

    ada_bool *value_p
);


        



/*
 * Return the record representation clause associated to this type decl, if
 * applicable (i.e. this type decl defines a record type).
 */
extern int ada_base_type_decl_p_get_record_representation_clause(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return the enum representation clause associated to this type decl, if
 * applicable (i.e. this type decl defines an enum type).
 */
extern int ada_base_type_decl_p_get_enum_representation_clause(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return the list of all primitive operations that are available on this type.
 * If ``only_inherited`` is True, it will only return the primitives that are
 * implicitly inherited by this type, discarding those explicitly defined on
 * this type. Predefined operators are included in the result iff
 * ``include_predefined_operators`` is True. It defaults to False.
 */
extern int ada_base_type_decl_p_get_primitives(
    ada_node *node,

        
        ada_bool
        only_inherited,
        
        ada_bool
        include_predefined_operators,

    ada_node_array *value_p
);


        



/*
 * Return whether this type is an array type.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_array_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Find types derived from self in the given ``root`` and its children.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_find_derived_types(
    ada_node *node,

        
        const ada_node*
        root,
        
        const ada_node*
        origin,
        
        ada_bool
        imprecise_fallback,

    ada_node_array *value_p
);


        



/*
 * Whether type is a scalar type.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_scalar_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Return the specific type under a class-wide type. Consider for example:
 *
 * .. code:: ada
 *
 *    subtype S1 is T'Class
 *    subtype S2 is S1'Class
 *
 * Calling this property on ``S2`` will return ``T``.
 */
extern int ada_base_type_decl_p_specific_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the list of all types that inherit (directly or indirectly) from self
 * among the given units.
 */
extern int ada_base_type_decl_p_find_all_derived_types(
    ada_node *node,

        
        ada_analysis_unit_array
        units,
        
        ada_bool
        imprecise_fallback,

    ada_node_array *value_p
);


        



/*
 * Return the component type of ``self``, if applicable. The component type is
 * the type you'll get if you call a value whose type is ``self``. So it can
 * either be:
 *
 * 1. The component type for an array.
 *
 * 2. The return type for an access to function.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_comp_type(
    ada_node *node,

        
        ada_bool
        is_subscript,
        
        const ada_node*
        origin,

    ada_node *value_p
);


        



/*
 * Return the index type for dimension ``dim`` for this type, if applicable.
 *
 * .. warning:: ``dim`` is 0-based, so the first ``index_type`` is at index 0.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_index_type(
    ada_node *node,

        
        int
        dim,
        
        const ada_node*
        origin,

    ada_node *value_p
);


        



/*
 * Whether self is derived from other_type.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_derived_type(
    ada_node *node,

        
        const ada_node*
        other_type,
        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Return True iff this type is limited, either because it is explicitly marked
 * as such, or because it inherits from a limited type or has a component of a
 * limited type. Also note that protected types and task types are limited by
 * definition. Moreover, note that Ada requires all parts of a type to agree of
 * its limitedness (e.g. the public view of a type must indicate that it is
 * limited if its private completion ends up being limited), hence this
 * property does not require looking at any other part of the type to determine
 * its limitedness, excepted for incomplete type declarations. This implies
 * that for illegal code where several parts don't agree, this property will
 * return the result for the particular view of the type on which this property
 * is called.
 */
extern int ada_base_type_decl_p_is_limited_type(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return whether ``self`` matches ``expected_type``.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_matching_type(
    ada_node *node,

        
        const ada_node*
        expected_type,
        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Returns the previous part for this type decl.
 */
extern int ada_base_type_decl_p_previous_part(
    ada_node *node,

        
        ada_bool
        go_to_incomplete,

    ada_node *value_p
);


        



/*
 * Returns the next part for this type decl.
 *
 * .. note:: Since this property returns a ``BaseTypeDecl``, it cannot be used
 *    to retrieve the next part of ``TaskTypeDecl`` and ``ProtectedTypeDecl``
 *    nodes as their next part is actually a ``Body``. Use
 *    ``BasicDecl.next_part_for_decl`` for those instead.
 */
extern int ada_base_type_decl_p_next_part(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the full completion of this type.
 */
extern int ada_base_type_decl_p_full_view(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Returns whether this is a definite subtype.
 *
 * For convenience, this will return ``False`` for incomplete types, even
 * though the correct answer is more akin to "non applicable".
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_definite_subtype(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Return the list of all discriminants of this type. If this type has no
 * discriminant or only unknown discriminants, an empty list is returned.
 *
 * In order to obtain all the discriminants of an extended type, this property
 * looks on parents, recursively.
 *
 * Extended aggregates can be build from any intermediate parent of an extended
 * type. In that case, this property shouldn't recurse to the root type, but
 * the one used as the aggregate's ancestor, designated by ``stop_recurse_at``.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_discriminants_list(
    ada_node *node,

        
        const ada_node*
        stop_recurse_at,
        
        const ada_node*
        origin,

    ada_node_array *value_p
);


        



/*
 * Must be called on a record (sub-)type declaration. Return all the possible
 * shapes that a value of this record type can take. For example, consider the
 * following record definition:
 *
 * .. code::
 *
 *    type R (A : Integer; B : Integer) is record
 *        X : Integer;
 *        case A is
 *            when 1 .. 10 =>
 *                Y_1 : Integer;
 *                case B is
 *                    when 1 .. 10 =>
 *                        Z_1 : Integer;
 *                    when others => null;
 *                end case;
 *            when 11 .. 20 =>
 *                Y_2 : Integer;
 *                case B is
 *                    when 1 .. 10 =>
 *                        Z_2 : Integer;
 *                    when others => null;
 *                end case;
 *            when others => null;
 *        end case;
 *    end record;
 *
 * For this instance, this property will return the following results:
 *
 * .. code::
 *
 *    [
 *        [X, Y_1, Z_1],
 *        [X, Y_1],
 *        [X, Y_2, Z_2],
 *        [X, Y_2],
 *        [X]
 *    ]
 *
 * .. attention:: This property is inaccurate when called on a record extension
 *    which defines components under a certain condition C, and this same
 *    condition is used to define some components in the parent record: in that
 *    case, any feasible shape will in practice contain either both the
 *    components defined under condition C in the child record and the parent
 *    record, or none of them.However, due to the simplified algorithm we use
 *    here to compute the feasible shapes, we will also return shapes that
 *    include the components of the child record but not the parent record, and
 *    conversely.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_shapes(
    ada_node *node,

        
        ada_bool
        include_discriminants,
        
        const ada_node*
        origin,

    ada_internal_shape_array *value_p
);


        



/*
 * Return whether this type is a record type.
 *
 * .. attention:: Private tagged types extending public tagged records are not
 *    considered as record types.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_record_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Whether type is a real type or not.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_real_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Whether type is a float type or not.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_float_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Whether type is a fixed point type or not.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_fixed_point(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Whether type is an enum type
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_enum_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Whether self is an access type or not
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_access_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Whether type is a character type or not
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_char_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Return the classwide type for this type, if applicable
 */
extern int ada_base_type_decl_p_classwide_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the discrete range for this type decl, if applicable.
 */
extern int ada_base_type_decl_p_discrete_range(
    ada_node *node,


    ada_internal_discrete_range *value_p
);


        



/*
 * Whether type is a discrete type or not.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_discrete_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Whether type is an integer type or not.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_int_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * If this type is an access type, or a type with an Implicit_Dereference
 * aspect, return the type of a dereference of an instance of this type.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_accessed_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_node *value_p
);


        



/*
 * Whether type is tagged or not
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_tagged_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Return the base type entity for this derived type declaration
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_base_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_node *value_p
);


        



/*
 * Return the list of base types for self.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_base_types(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_node_array *value_p
);


        



/*
 * Return True iff this type declaration is an interface definition.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_is_interface_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_bool *value_p
);


        



/*
 * Return the canonical type declaration for this type declaration. For
 * subtypes, it will return the base type declaration.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_canonical_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_node *value_p
);


        



/*
 * Whether node is a private view of corresponding type.
 */
extern int ada_base_type_decl_p_is_private(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return the type that is at the root of the derivation hierarchy (ignoring
 * secondary interfaces derivations for tagged types)
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_type_decl_p_root_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_node *value_p
);


        



/*
 * Get the type for this subtype.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_base_subtype_decl_p_get_type(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_subtype_decl_f_subtype(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_incomplete_type_decl_f_discriminants(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_incomplete_formal_type_decl_f_is_tagged(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_incomplete_formal_type_decl_f_default_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_incomplete_tagged_type_decl_f_has_abstract(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_protected_type_decl_f_discriminants(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_char_literal``, ``ada_dotted_name``, ``ada_identifier``,
 * ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_protected_type_decl_f_interfaces(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_protected_type_decl_f_definition(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_task_type_decl_f_discriminants(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_task_type_decl_f_definition(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_type_decl_f_discriminants(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_type_decl_f_type_def(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_formal_type_decl_f_default_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the specification for this subprogram
 */
extern int ada_basic_subp_decl_p_subp_decl_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_classic_subp_decl_f_overriding(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_classic_subp_decl_f_subp_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the BaseSubpBody corresponding to this node.
 */
extern int ada_classic_subp_decl_p_body_part(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_box_expr``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_identifier``,
 * ``ada_null_literal``, ``ada_qual_expr``, ``ada_reduce_attribute_ref``,
 * ``ada_string_literal``, ``ada_target_name``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_formal_subp_decl_f_default_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_entry_decl_f_overriding(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_entry_decl_f_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the entry body associated to this entry declaration.
 */
extern int ada_entry_decl_p_body_part(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return an array of accept statements corresponding to this entry.
 */
extern int ada_entry_decl_p_accept_stmts(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_enum_literal_decl_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the integer used to encode this enum literal.
 *
 * .. note:: This property is equivalent to GNAT's ``Enum_Rep`` attribute.
 */
extern int ada_enum_literal_decl_p_enum_rep(
    ada_node *node,


    ada_big_integer *value_p
);


        



/*
 * Return the enum type corresponding to this enum literal.
 */
extern int ada_enum_literal_decl_p_enum_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the CharLiteral expression corresponding to this enum literal.
 */
extern int ada_synthetic_char_enum_lit_p_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_subp_internal_f_subp_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes:
 * ``ada_synthetic_binary_spec``, ``ada_synthetic_unary_spec``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_synthetic_subp_decl_f_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the previous part for this body. Might be a declaration or a body
 * stub.
 */
extern int ada_body_node_p_previous_part(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return the decl corresponding to this node if applicable.
 */
extern int ada_body_node_p_decl_part(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * If self is a subunit, return the body in which it is rooted.
 */
extern int ada_body_node_p_subunit_root(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_accept_stmt_body_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_base_subp_body_f_overriding(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_base_subp_body_f_subp_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_base_aggregate``,
 * ``ada_paren_expr``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_expr_function_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_subp_body_f_decls(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_subp_body_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_subp_body_f_end_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_subp_renaming_decl_f_renames(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the syntactic fully qualified name to refer to this body.
 *
 * Note that this can raise a Property_Error when the stub is in an illegal
 * place (too nested, in a declare block, etc.).
 */
extern int ada_body_stub_p_syntactic_fully_qualified_name(
    ada_node *node,


    ada_symbol_type_array *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_package_body_stub_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_protected_body_stub_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_subp_body_stub_f_overriding(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_subp_body_stub_f_subp_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_task_body_stub_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_entry_body_f_entry_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_entry_body_f_index_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_entry_body_f_params(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_entry_body_f_barrier(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_entry_body_f_decls(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_entry_body_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_entry_body_f_end_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_package_body_f_package_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_package_body_f_decls(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_package_body_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_package_body_f_end_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_protected_body_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_protected_body_f_decls(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_protected_body_f_end_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_task_body_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_task_body_f_decls(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_task_body_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_task_body_f_end_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_entry_index_spec_f_id(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_bin_op``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_identifier``,
 * ``ada_qual_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_subtype_indication``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_entry_index_spec_f_subtype(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_exception_decl_f_ids(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_exception_decl_f_renames(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_exception_handler_f_exception_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_attribute_ref``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_identifier``,
 * ``ada_others_designator``, ``ada_qual_expr``, ``ada_reduce_attribute_ref``,
 * ``ada_string_literal``, ``ada_target_name``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_exception_handler_f_handled_exceptions(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_pragma_node``, ``ada_stmt``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_exception_handler_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_for_loop_var_decl_f_id(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_anonymous_type``,
 * ``ada_subtype_indication``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_for_loop_var_decl_f_id_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_decl_f_formal_part(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_package_decl_f_package_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the PackageBody corresponding to this node, or null if there is none.
 */
extern int ada_generic_package_decl_p_body_part(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_subp_decl_f_subp_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the BaseSubpBody corresponding to this node.
 */
extern int ada_generic_subp_decl_p_body_part(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return the generic decl entity designated by this instantiation, including
 * instantiation information. This is equivalent to the expanded generic unit
 * in GNAT.
 */
extern int ada_generic_instantiation_p_designated_generic_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Returns an array of pairs, associating formal parameters to actual or
 * default expressions.
 */
extern int ada_generic_instantiation_p_inst_params(
    ada_node *node,


    ada_internal_param_actual_array *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_package_instantiation_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_identifier``, ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_package_instantiation_f_generic_pkg_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_param_assoc``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_package_instantiation_f_params(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_subp_instantiation_f_overriding(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_subp_instantiation_f_kind(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_subp_instantiation_f_subp_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_identifier``, ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_subp_instantiation_f_generic_subp_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_param_assoc``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_subp_instantiation_f_params(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the subprogram decl designated by this instantiation.
 */
extern int ada_generic_subp_instantiation_p_designated_subp(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_package_renaming_decl_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_package_renaming_decl_f_renames(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_subp_renaming_decl_f_kind(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_subp_renaming_decl_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_generic_subp_renaming_decl_f_renames(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_label_decl_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_named_stmt_decl_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_number_decl_f_ids(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_null_literal``, ``ada_num_literal``, ``ada_paren_expr``,
 * ``ada_qual_expr``, ``ada_quantified_expr``, ``ada_raise_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_number_decl_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_object_decl_f_ids(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_object_decl_f_has_aliased(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_object_decl_f_has_constant(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_object_decl_f_mode(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_anonymous_type``,
 * ``ada_subtype_indication``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_object_decl_f_type_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_object_decl_f_default_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_object_decl_f_renaming_clause(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * If this object decl is the constant completion of an object decl in the
 * public part, return the object decl from the public part.
 */
extern int ada_object_decl_p_private_part_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * If this object decl is the incomplete declaration of a constant in a public
 * part, return its completion in the private part.
 */
extern int ada_object_decl_p_public_part_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_package_renaming_decl_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_package_renaming_decl_f_renames(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the declaration of the package that is renamed by self.
 */
extern int ada_package_renaming_decl_p_renamed_package(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the declaration of the package that is ultimately renamed by self,
 * skipping through all intermediate package renamings.
 */
extern int ada_package_renaming_decl_p_final_renamed_package(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_single_protected_decl_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_char_literal``, ``ada_dotted_name``, ``ada_identifier``,
 * ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_single_protected_decl_f_interfaces(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_single_protected_decl_f_definition(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_single_task_decl_f_task_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_allocator``, ``ada_attribute_ref``, ``ada_base_aggregate``,
 * ``ada_bin_op``, ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_discrete_subtype_indication``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_membership_expr``, ``ada_null_literal``,
 * ``ada_num_literal``, ``ada_others_designator``, ``ada_paren_expr``,
 * ``ada_qual_expr``, ``ada_quantified_expr``, ``ada_raise_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_case_stmt_alternative_f_choices(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_pragma_node``, ``ada_stmt``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_case_stmt_alternative_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * ``with``, ``use`` or ``pragma`` statements.
 *
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_pragma_node``, ``ada_use_clause``, ``ada_with_clause``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_compilation_unit_f_prelude(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_library_item``,
 * ``ada_subunit``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_compilation_unit_f_body(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_compilation_unit_f_pragmas(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the syntactic fully qualified name of this compilation unit.
 */
extern int ada_compilation_unit_p_syntactic_fully_qualified_name(
    ada_node *node,


    ada_symbol_type_array *value_p
);


        



/*
 * Return the kind corresponding to this analysis unit.
 */
extern int ada_compilation_unit_p_unit_kind(
    ada_node *node,


    ada_analysis_unit_kind *value_p
);


        



/*
 * Look for all "with" clauses at the top of this compilation unit and return
 * all the compilation units designated by them. For the complete dependencies
 * list of compilation units, see the ``unit_dependencies`` property. Units
 * imported with a "private with" are included in this list only if
 * ``include_privates`` is True.
 */
extern int ada_compilation_unit_p_withed_units(
    ada_node *node,

        
        ada_bool
        include_privates,

    ada_node_array *value_p
);


        



/*
 * Return all the compilation units that are directly imported by this one.
 * This includes "with"ed units as well as the direct parent unit. Units
 * imported with a "private with" are included in this list only if
 * ``include_privates`` is True.
 */
extern int ada_compilation_unit_p_imported_units(
    ada_node *node,

        
        ada_bool
        include_privates,

    ada_node_array *value_p
);


        



/*
 * Return the list of all the compilation units that are (direct and indirect)
 * dependencies of this one. See the ``withed_units``/``imported_units``
 * properties to only get the direct dependencies of this unit.
 */
extern int ada_compilation_unit_p_unit_dependencies(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * Get the root basic decl defined in this compilation unit.
 */
extern int ada_compilation_unit_p_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Whether this compilation unit is preelaborable or not.
 */
extern int ada_compilation_unit_p_is_preelaborable(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_bool *value_p
);


        



/*
 * If this compilation unit is of kind UnitSpecification, return its
 * corresponding body unit, and conversely.
 */
extern int ada_compilation_unit_p_other_part(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Whether this compilation unit is affected by the restriction with the given
 * name.
 *
 * .. warning:: This property only supports the ``No_Elaboration_Code``
 *    restriction for now.
 */
extern int ada_compilation_unit_p_has_restriction(
    ada_node *node,

        
        const ada_symbol_type*
        name,

    ada_bool *value_p
);


        



/*
 * Return the list of configuration pragmas that apply to the current unit.
 */
extern int ada_compilation_unit_p_all_config_pragmas(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * Return the list of configuration pragmas with the given name that apply to
 * the current unit.
 */
extern int ada_compilation_unit_p_config_pragmas(
    ada_node *node,

        
        const ada_symbol_type*
        name,

    ada_node_array *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_component_clause_f_id(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_null_literal``, ``ada_num_literal``, ``ada_paren_expr``,
 * ``ada_qual_expr``, ``ada_quantified_expr``, ``ada_raise_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_component_clause_f_position(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_component_clause_f_range(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_component_def_f_has_aliased(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_component_def_f_has_constant(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_anonymous_type``,
 * ``ada_subtype_indication``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_component_def_f_type_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_constant_node_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_composite_constraint_assoc``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_composite_constraint_f_constraints(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Whether this composite constraint is an index constraint.
 */
extern int ada_composite_constraint_p_is_index_constraint(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Whether this composite constraint is a discriminant constraint.
 */
extern int ada_composite_constraint_p_is_discriminant_constraint(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Returns an array of pairs, associating each discriminant to its actual or
 * default expression.
 */
extern int ada_composite_constraint_p_discriminant_params(
    ada_node *node,


    ada_internal_param_actual_array *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_null_literal``, ``ada_num_literal``, ``ada_paren_expr``,
 * ``ada_qual_expr``, ``ada_quantified_expr``, ``ada_raise_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_delta_constraint_f_delta(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_delta_constraint_f_range(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_null_literal``, ``ada_num_literal``, ``ada_paren_expr``,
 * ``ada_qual_expr``, ``ada_quantified_expr``, ``ada_raise_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_digits_constraint_f_digits(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_digits_constraint_f_range(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_range_constraint_f_range(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_abstract_subp_decl``, ``ada_aspect_clause``, ``ada_base_subp_body``,
 * ``ada_body_stub``, ``ada_component_decl``, ``ada_concrete_type_decl``,
 * ``ada_entry_body``, ``ada_entry_decl``, ``ada_error_decl``,
 * ``ada_exception_decl``, ``ada_generic_decl``, ``ada_generic_instantiation``,
 * ``ada_generic_renaming_decl``, ``ada_incomplete_type_decl``,
 * ``ada_number_decl``, ``ada_object_decl``, ``ada_package_body``,
 * ``ada_package_decl``, ``ada_package_renaming_decl``, ``ada_pragma_node``,
 * ``ada_protected_body``, ``ada_protected_type_decl``,
 * ``ada_single_protected_decl``, ``ada_single_task_decl``, ``ada_subp_decl``,
 * ``ada_subtype_decl``, ``ada_task_body``, ``ada_task_type_decl``,
 * ``ada_use_clause``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_declarative_part_f_decls(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_elsif_expr_part_f_cond_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_elsif_expr_part_f_then_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_elsif_stmt_part_f_cond_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_pragma_node``, ``ada_stmt``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_elsif_stmt_part_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the declaration corresponding to the type of this expression after
 * name resolution.
 */
extern int ada_expr_p_expression_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the declaration corresponding to the expected type of this expression
 * after name resolution.
 */
extern int ada_expr_p_expected_expression_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Returns whether this expression is dynamically tagged (See :rmlink:`3.9.2`).
 */
extern int ada_expr_p_is_dynamically_tagged(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_bool *value_p
);


        



/*
 * Return whether this expression is static according to the ARM definition of
 * static. See :rmlink:`4.9`.
 */
extern int ada_expr_p_is_static_expr(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_bool *value_p
);


        



/*
 * Statically evaluates self, and returns the value of the evaluation as an
 * integer.
 *
 * .. note:: In order for a call to this not to raise, the expression needs to
 *    be a static expression, as specified in :rmlink:`4.9`. You can verify
 *    whether an expression is static with the ``is_static_expr`` property.
 *
 * .. attention:: This is an experimental feature, so even if it is exposed to
 *    allow experiments, it is totally unsupported and the API and behavior are
 *    very likely to change in the future.
 */
extern int ada_expr_p_eval_as_int(
    ada_node *node,


    ada_big_integer *value_p
);


        



/*
 * Statically evaluates self, and returns the value of the evaluation as an
 * integer. The given environment is used to substitute references to
 * declarations by actual values.
 *
 * .. note:: In order for a call to this not to raise, the expression needs to
 *    be a static expression, as specified in :rmlink:`4.9`. You can verify
 *    whether an expression is static with the ``is_static_expr`` property.
 *
 * .. attention:: This is an experimental feature, so even if it is exposed to
 *    allow experiments, it is totally unsupported and the API and behavior are
 *    very likely to change in the future.
 */
extern int ada_expr_p_eval_as_int_in_env(
    ada_node *node,

        
        ada_internal_substitution_array
        env,

    ada_big_integer *value_p
);


        



/*
 * Statically evaluates self, and returns the value of the evaluation as a
 * string.
 *
 * .. note:: In order for a call to this not to raise, the expression needs to
 *    be a static expression, as specified in :rmlink:`4.9`. You can verify
 *    whether an expression is static with the ``is_static_expr`` property.
 *
 * .. attention:: This is an experimental feature, so even if it is exposed to
 *    allow experiments, it is totally unsupported and the API and behavior are
 *    very likely to change in the future.
 */
extern int ada_expr_p_eval_as_string(
    ada_node *node,


    ada_string_type *value_p
);


        



/*
 * Statically evaluates self, and returns the value of the evaluation as a
 * string. The given environment is used to substitute references to
 * declarations by actual values.
 *
 * .. note:: In order for a call to this not to raise, the expression needs to
 *    be a static expression, as specified in :rmlink:`4.9`. You can verify
 *    whether an expression is static with the ``is_static_expr`` property.
 *
 * .. attention:: This is an experimental feature, so even if it is exposed to
 *    allow experiments, it is totally unsupported and the API and behavior are
 *    very likely to change in the future.
 */
extern int ada_expr_p_eval_as_string_in_env(
    ada_node *node,

        
        ada_internal_substitution_array
        env,

    ada_string_type *value_p
);


        



/*
 * Return the list of AST nodes that can be a match for this expression before
 * overloading analysis.
 */
extern int ada_expr_p_matching_nodes(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * Returns True if this ``Name`` corresponds to a dispatching call, including:
 *
 * * Calls done through subprogram access types.
 *
 * * Calls to dispatching subprograms, in the object-oriented sense.
 *
 * .. note:: This is an experimental feature. There might be some discrepancy
 *    with the GNAT concept of "dispatching call".
 *
 * .. note:: This should only be called on a ``Name`` and ``UnOp`` or a
 *    ``BinOp``.
 *
 * .. attention:: There is a known bug, where the ConcatOp node is not
 *    supported, so calling is_dispatching_call on operators nested inside of a
 *    concat operator will always return false. (Internal TN: VC08-029)
 */
extern int ada_expr_p_is_dispatching_call(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_bool *value_p
);


        



/*
 * Return the first decl that is lexically named like self in self's scope.
 */
extern int ada_expr_p_first_corresponding_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes:
 * ``ada_abstract_state_decl``, ``ada_multi_abstract_state_decl``,
 * ``ada_paren_abstract_state_decl``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_abstract_state_decl_expr_f_state_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_allocator_f_subpool(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_qual_expr``,
 * ``ada_subtype_indication``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_allocator_f_type_or_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the allocated type for this allocator.
 */
extern int ada_allocator_p_get_allocated_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_base_aggregate_f_ancestor_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_aggregate_assoc``, ``ada_iterated_assoc``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_base_aggregate_f_assocs(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Returns an array of pairs, associating formal parameters to actual
 * expressions. See ``zip_with_params``.
 */
extern int ada_base_aggregate_p_aggregate_params(
    ada_node *node,


    ada_internal_param_actual_array *value_p
);


        



/*
 * Return whether this aggregate is actually a subaggregate of a
 * multidimensional array aggregate, as described in :rmlink:`4.3.3`.
 */
extern int ada_base_aggregate_p_is_subaggregate(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_bin_op_f_left(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_op_and_then``,
 * ``ada_op_and``, ``ada_op_div``, ``ada_op_double_dot``, ``ada_op_eq``,
 * ``ada_op_gt``, ``ada_op_gte``, ``ada_op_lt``, ``ada_op_lte``,
 * ``ada_op_minus``, ``ada_op_mod``, ``ada_op_mult``, ``ada_op_neq``,
 * ``ada_op_or_else``, ``ada_op_or``, ``ada_op_plus``, ``ada_op_pow``,
 * ``ada_op_rem``, ``ada_op_xor``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_bin_op_f_op(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_box_expr``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_concat_op``, ``ada_cond_expr``, ``ada_decl_expr``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_membership_expr``, ``ada_null_literal``,
 * ``ada_num_literal``, ``ada_paren_expr``, ``ada_qual_expr``,
 * ``ada_quantified_expr``, ``ada_raise_expr``, ``ada_reduce_attribute_ref``,
 * ``ada_string_literal``, ``ada_target_name``, ``ada_un_op``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_bin_op_f_right(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_allocator``, ``ada_attribute_ref``, ``ada_base_aggregate``,
 * ``ada_bin_op``, ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_discrete_subtype_indication``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_membership_expr``, ``ada_null_literal``,
 * ``ada_num_literal``, ``ada_others_designator``, ``ada_paren_expr``,
 * ``ada_qual_expr``, ``ada_quantified_expr``, ``ada_raise_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_case_expr_alternative_f_choices(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_case_expr_alternative_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_cond_expr``,
 * ``ada_decl_expr``, ``ada_dotted_name``, ``ada_explicit_deref``,
 * ``ada_format_string_literal``, ``ada_identifier``, ``ada_null_literal``,
 * ``ada_num_literal``, ``ada_paren_expr``, ``ada_qual_expr``,
 * ``ada_quantified_expr``, ``ada_raise_expr``, ``ada_reduce_attribute_ref``,
 * ``ada_string_literal``, ``ada_target_name``, ``ada_un_op``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_concat_op_f_first_operand(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_concat_op_f_other_operands(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the operands of this concatenation expression
 */
extern int ada_concat_op_p_operands(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_concat_operand_f_operator(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_cond_expr``,
 * ``ada_decl_expr``, ``ada_dotted_name``, ``ada_explicit_deref``,
 * ``ada_format_string_literal``, ``ada_identifier``, ``ada_null_literal``,
 * ``ada_num_literal``, ``ada_paren_expr``, ``ada_qual_expr``,
 * ``ada_quantified_expr``, ``ada_raise_expr``, ``ada_reduce_attribute_ref``,
 * ``ada_string_literal``, ``ada_target_name``, ``ada_un_op``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_concat_operand_f_operand(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the dependent expressions for this conditional expression.
 */
extern int ada_cond_expr_p_dependent_exprs(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_case_expr_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_case_expr_f_cases(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_if_expr_f_cond_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_if_expr_f_then_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_if_expr_f_alternatives(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_if_expr_f_else_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_contract_cases_f_contract_cases(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_object_decl``, ``ada_pragma_node``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_decl_expr_f_decls(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_decl_expr_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_format_string_literal_f_opening_chunk(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_format_string_literal_f_mid_exprs(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_format_string_literal_f_trailing_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_membership_expr_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_op_in``,
 * ``ada_op_not_in``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_membership_expr_f_op(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_allocator``, ``ada_attribute_ref``, ``ada_base_aggregate``,
 * ``ada_bin_op``, ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_discrete_subtype_name``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_membership_expr_f_membership_exprs(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * If this name is part of a defining name, return the enclosing defining name
 * node.
 */
extern int ada_name_p_enclosing_defining_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return True if this name is part of a defining name.
 */
extern int ada_name_p_is_defining(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Helper. Check that this name matches ``sym``.
 */
extern int ada_name_p_name_is(
    ada_node *node,

        
        const ada_symbol_type*
        sym,

    ada_bool *value_p
);


        



/*
 * Return True iff this name represents a call to a subprogram which is
 * referred by its defining name. (i.e. not through a subprogram access).
 */
extern int ada_name_p_is_direct_call(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return True iff this name represents a call to subprogram through an access
 * type.
 */
extern int ada_name_p_is_access_call(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Returns True if this Name corresponds to a call.
 */
extern int ada_name_p_is_call(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Returns True if this Name corresponds to a dot notation call.
 */
extern int ada_name_p_is_dot_call(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_bool *value_p
);


        



/*
 * Failsafe version of ``referenced_defining_name``. Returns a ``RefdDef``,
 * which can be precise, imprecise, or error.
 */
extern int ada_name_p_failsafe_referenced_def_name(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_internal_refd_def *value_p
);


        



/*
 * Like ``referenced_decl``, but will return the defining identifier for the
 * decl, rather than the basic declaration node itself.
 */
extern int ada_name_p_referenced_defining_name(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return all elements in self's scope that are lexically named like self.
 */
extern int ada_name_p_all_env_elements(
    ada_node *node,

        
        ada_bool
        seq,
        
        const ada_node*
        seq_from,

    ada_node_array *value_p
);


        



/*
 * Return the subprogram specification of the subprogram or subprogram access
 * that is being called by this exact Name, if relevant. Note that when inside
 * an instantiated generic, this will return the spec of the actual subprogram.
 */
extern int ada_name_p_called_subp_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the declaration this node references after name resolution. If
 * imprecise_fallback is True, errors raised during resolution of the xref
 * equation are caught and a fallback mechanism is triggered, which tries to
 * find the referenced declaration in an ad-hoc way.
 */
extern int ada_name_p_referenced_decl(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Failsafe version of ``referenced_decl``. Returns a ``RefdDecl``, which can
 * be precise, imprecise, or error.
 */
extern int ada_name_p_failsafe_referenced_decl(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_internal_refd_decl *value_p
);


        



/*
 * Like SubtypeIndication.designated_type, but on names, since because of Ada's
 * ambiguous grammar, some subtype indications will be parsed as names.
 */
extern int ada_name_p_name_designated_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Returns whether self denotes a static subtype or not.
 */
extern int ada_name_p_is_static_subtype(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_bool *value_p
);


        



/*
 * Return whether two names match each other.
 *
 * This compares the symbol for Identifier and StringLiteral nodes. We consider
 * that there is no match for all other node kinds.
 */
extern int ada_name_p_name_matches(
    ada_node *node,

        
        const ada_node*
        n,

    ada_bool *value_p
);


        



/*
 * Return whether the name that self designates is an operator.
 */
extern int ada_name_p_is_operator_name(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Whether this name is a write reference.
 *
 * For example, ``X`` is a write reference in the following cases:
 *
 * 1. ``X := 2;``
 *
 * 2. ``X (2) := 2;``
 *
 * 3. ``P(F => X)`` where F is declared ``out`` or ``in out``.
 *
 * 4. ``P(F => T (X))`` where F is declared ``out`` or ``in out``
 *
 * 5. ``X'Access``.
 *
 * 6. ``X.C := 2``, ``R.X := 2``
 *
 * 7. ``X.P`` where the formal for X is declared ``out`` or ``in out``.
 *
 * .. note:: This is an experimental feature. There might be some discrepancy
 *    with the GNAT concept of "write reference".
 */
extern int ada_name_p_is_write_reference(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_bool *value_p
);


        



/*
 * Returns True if this Name corresponds to a static non-dispatching call. In
 * other words, this will return True if and only if the target of the call is
 * known statically.
 *
 * .. note:: This is an experimental feature. There might be some discrepancy
 *    with the GNAT concept of "static call".
 */
extern int ada_name_p_is_static_call(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_bool *value_p
);


        



/*
 * Returns an array of pairs, associating formal parameters to actual or
 * default expressions.
 */
extern int ada_name_p_call_params(
    ada_node *node,


    ada_internal_param_actual_array *value_p
);


        



/*
 * Returns the relative name of this instance. For example, for a prefix
 * ``A.B.C``, this will return ``C``.
 */
extern int ada_name_p_relative_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Turn this name into an array of symbols.
 *
 * For instance, a node with name ``A.B.C`` is turned into ``['A', 'B', 'C']``.
 *
 * Only simple name kinds are allowed: Identifier, DottedName and DefiningName.
 * Any other kind will trigger a PreconditionFailure.
 */
extern int ada_name_p_as_symbol_array(
    ada_node *node,


    ada_symbol_type_array *value_p
);


        



/*
 * Return a canonicalized version of this name's text.
 *
 * Only simple name kinds are allowed: Identifier, DottedName and DefiningName.
 * Any other kind will trigger a PreconditionFailure.
 */
extern int ada_name_p_canonical_text(
    ada_node *node,


    ada_symbol_type *value_p
);


        



/*
 * Return whether this name denotes a constant value.
 */
extern int ada_name_p_is_constant(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * This field can contain one of the following nodes:
 * ``ada_array_subcomponent_choice_name``, ``ada_dotted_name``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_array_subcomponent_choice_name_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_basic_assoc_list``, ``ada_bin_op``, ``ada_call_expr``,
 * ``ada_char_literal``, ``ada_discrete_subtype_indication``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_identifier``,
 * ``ada_qual_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_array_subcomponent_choice_name_f_suffix(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_attribute_ref_f_prefix(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_attribute_ref_f_attribute(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_param_assoc``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_attribute_ref_f_args(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_call_expr_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_basic_assoc_list``, ``ada_bin_op``, ``ada_call_expr``,
 * ``ada_char_literal``, ``ada_discrete_subtype_indication``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_identifier``,
 * ``ada_qual_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_call_expr_f_suffix(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return whether this expression is a subprogram call, an array subcomponent
 * access expression, an array slice or a type conversion.
 */
extern int ada_call_expr_p_kind(
    ada_node *node,


    ada_call_expr_kind *value_p
);


        



/*
 * Return whether this CallExpr is actually an access to a slice of the array
 * denoted by the prefix of this CallExpr.
 */
extern int ada_call_expr_p_is_array_slice(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_identifier``, ``ada_string_literal``,
 * ``ada_synthetic_identifier``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_defining_name_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return a canonical representation of the fully qualified name corresponding
 * to this defining name.
 */
extern int ada_defining_name_p_canonical_fully_qualified_name(
    ada_node *node,


    ada_string_type *value_p
);


        



/*
 * Return a unique identifying name for this defining name, provided this
 * declaration is a public declaration. In the case of subprograms, this will
 * include the profile.
 *
 * .. attention:: This will only return a unique name for public declarations.
 *    Notably, anything nested in an unnamed declare block won't be handled
 *    correctly.
 */
extern int ada_defining_name_p_unique_identifying_name(
    ada_node *node,


    ada_string_type *value_p
);


        



/*
 * Return the fully qualified name corresponding to this defining name, as an
 * array of symbols.
 */
extern int ada_defining_name_p_fully_qualified_name_array(
    ada_node *node,


    ada_symbol_type_array *value_p
);


        



/*
 * Return the fully qualified name corresponding to this defining name.
 */
extern int ada_defining_name_p_fully_qualified_name(
    ada_node *node,


    ada_string_type *value_p
);


        



/*
 * Return this DefiningName's basic declaration, discarding internal nodes such
 * as Generic*Internal wrappers.
 */
extern int ada_defining_name_p_basic_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Find all references to this defining name in the given ``root`` and its
 * children.
 */
extern int ada_defining_name_p_find_refs(
    ada_node *node,

        
        const ada_node*
        root,
        
        ada_bool
        imprecise_fallback,

    ada_internal_ref_result_array *value_p
);


        



/*
 * Searches all references to this defining name in the given list of units.
 *
 * If ``follow_renamings`` is True, also this also includes references that
 * ultimately refer to this defining name, by unwinding renaming clauses.
 */
extern int ada_defining_name_p_find_all_references(
    ada_node *node,

        
        ada_analysis_unit_array
        units,
        
        ada_bool
        follow_renamings,
        
        ada_bool
        imprecise_fallback,

    ada_internal_ref_result_array *value_p
);


        



/*
 * Return the list of all possible calls to the subprogram which self is the
 * defining name of.
 *
 * This will return the name corresponding to the call, excluding the
 * parameters if there are any. For instance, it will return ``A`` for the ``A
 * (B)`` call.
 *
 * .. note:: This does not yet support calls done inside generics.
 */
extern int ada_defining_name_p_find_all_calls(
    ada_node *node,

        
        ada_analysis_unit_array
        units,
        
        ada_bool
        follow_renamings,
        
        ada_bool
        imprecise_fallback,

    ada_internal_ref_result_array *value_p
);


        



/*
 * Like ``BasicDecl.next_part_for_decl`` on a defining name
 */
extern int ada_defining_name_p_next_part(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Like ``BasicDecl.previous_part_for_decl`` on a defining name
 */
extern int ada_defining_name_p_previous_part(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Like ``BasicDecl.canonical_part`` on a defining name
 */
extern int ada_defining_name_p_canonical_part(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Given an origin node and the entity represented by self, this property
 * returns the most visible completion of self that can be seen by origin,
 * according to Ada's visibility rules.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_defining_name_p_most_visible_part(
    ada_node *node,

        
        const ada_node*
        origin,
        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return all parts that define this entity, sorted from first part to last
 * part.
 */
extern int ada_defining_name_p_all_parts(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node_array *value_p
);


        



/*
 * Return the aspect with name ``name`` associated to entity that this name
 * defines.
 *
 * Aspects are properties of entities that can be specified by the Ada program,
 * either via aspect specifications, pragmas, or attributes.
 *
 * Note: by default, Libadalang will check if the aspect is defined on any part
 * of the entity. However, the ``previous_parts_only`` parameter can be set to
 * True to limit the search to the current entity and its previous parts in
 * order to comply with visibilily rules. That way, if an aspect is defined on
 * the private part of a type, calling this property on its corresponding
 * public view won't return the aspect unlike the call on the private view.
 *
 * Moreover, since aspects can be inherited, if none was found for the current
 * entity, Libadalang will also search for the aspect on the parents of entity
 * (in that case the ``inherited`` field will be set to ``True`` in the
 * returned result).
 */
extern int ada_defining_name_p_get_aspect(
    ada_node *node,

        
        const ada_symbol_type*
        name,
        
        ada_bool
        previous_parts_only,
        
        ada_bool
        imprecise_fallback,

    ada_internal_aspect *value_p
);


        



/*
 * Returns whether the boolean aspect named ``name`` is set on the entity
 * represented by this node.
 *
 * Note: The ``previous_parts_only`` parameter controls how aspects are
 * retrieved. See ``DefiningName.get_aspect`` for more information.
 *
 * Aspects are properties of entities that can be specified by the Ada program,
 * either via aspect specifications, pragmas, or attributes.
 *
 * "Aspect" is used as in RM terminology (see :rmlink:`13.1`).
 */
extern int ada_defining_name_p_has_aspect(
    ada_node *node,

        
        const ada_symbol_type*
        name,
        
        ada_bool
        previous_parts_only,
        
        ada_bool
        imprecise_fallback,

    ada_bool *value_p
);


        



/*
 * Return the pragma with name ``name`` associated to this entity.
 *
 * Please use the ``p_get_aspect`` property instead if you are interested in
 * aspects, i.e. information that can be represented by either aspect
 * specification nodes, pragma nodes or attribute definition nodes.
 */
extern int ada_defining_name_p_get_pragma(
    ada_node *node,

        
        const ada_symbol_type*
        name,

    ada_node *value_p
);


        



/*
 * Return the representation clause associated to this entity that defines the
 * given attribute name.
 */
extern int ada_defining_name_p_get_representation_clause(
    ada_node *node,

        
        const ada_symbol_type*
        name,
        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return the at clause associated to this entity.
 */
extern int ada_defining_name_p_get_at_clause(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_node *value_p
);


        



/*
 * Return all the ``Annotate`` aspects defined on this entity, both through
 * pragmas and aspect specifications. For a type declaration, this also
 * includes all annotations defined on its base type, when relevant (the field
 * ``inherited`` will be set for those).
 *
 * The ``value`` field of each returned ``Aspect`` will be set to be the
 * identifier that designates the tool which is concerned by the annotation.
 *
 * Note: Libadalang will look for the ``Annotate`` aspects on any part of the
 * entity.
 */
extern int ada_defining_name_p_get_annotations(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_internal_aspect_array *value_p
);


        



/*
 * Whether this entity defined by this name is imported from another language.
 */
extern int ada_defining_name_p_is_imported(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return whether the entity defined by this name is ghost or not. See SPARK RM
 * 6.9.
 */
extern int ada_defining_name_p_is_ghost_code(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_discrete_subtype_name_f_subtype(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes:
 * ``ada_array_subcomponent_choice_name``, ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_dotted_name_f_prefix(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_char_literal``,
 * ``ada_identifier``, ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_dotted_name_f_suffix(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_identifier``, ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_end_name_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Returns this EndName's basic declaration
 */
extern int ada_end_name_p_basic_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_explicit_deref_f_prefix(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_qual_expr_f_prefix(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_base_aggregate``,
 * ``ada_paren_expr``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_qual_expr_f_suffix(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``, ``ada_value_sequence``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_reduce_attribute_ref_f_prefix(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_reduce_attribute_ref_f_attribute(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_param_assoc``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_reduce_attribute_ref_f_args(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the value that this literal denotes.
 */
extern int ada_char_literal_p_denoted_value(
    ada_node *node,


    uint32_t *value_p
);


        



/*
 * Return the value that this literal denotes.
 */
extern int ada_string_literal_p_denoted_value(
    ada_node *node,


    ada_string_type *value_p
);


        



/*
 * Return the value that this literal denotes.
 */
extern int ada_int_literal_p_denoted_value(
    ada_node *node,


    ada_big_integer *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_update_attribute_ref_f_prefix(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_update_attribute_ref_f_attribute(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_update_attribute_ref_f_values(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_paren_expr_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_quantified_expr_f_quantifier(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_quantified_expr_f_loop_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_quantified_expr_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_raise_expr_f_exception_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_raise_expr_f_error_message(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_op_abs``,
 * ``ada_op_minus``, ``ada_op_not``, ``ada_op_plus``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_un_op_f_op(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_cond_expr``,
 * ``ada_decl_expr``, ``ada_dotted_name``, ``ada_explicit_deref``,
 * ``ada_format_string_literal``, ``ada_identifier``, ``ada_null_literal``,
 * ``ada_num_literal``, ``ada_paren_expr``, ``ada_qual_expr``,
 * ``ada_quantified_expr``, ``ada_raise_expr``, ``ada_reduce_attribute_ref``,
 * ``ada_string_literal``, ``ada_target_name``, ``ada_un_op``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_un_op_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_for_loop_iter_filter_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_format_string_chunk_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes:
 * ``ada_format_string_tok_end``, ``ada_format_string_tok_mid``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_format_string_chunk_f_string_tok(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_pragma_node``, ``ada_stmt``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_handled_stmts_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_exception_handler``, ``ada_pragma_node``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_handled_stmts_f_exceptions(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_library_item_f_has_private(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes:
 * ``ada_abstract_subp_decl``, ``ada_base_subp_body``, ``ada_error_decl``,
 * ``ada_generic_decl``, ``ada_generic_instantiation``,
 * ``ada_generic_renaming_decl``, ``ada_package_body``, ``ada_package_decl``,
 * ``ada_package_renaming_decl``, ``ada_subp_decl``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_library_item_f_item(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_limited_node_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_for_loop_spec_f_var_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_for_loop_spec_f_loop_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_for_loop_spec_f_has_reverse(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_bin_op``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_discrete_subtype_indication``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_for_loop_spec_f_iter_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_for_loop_spec_f_iter_filter(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_while_loop_spec_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_abstract_state_decl``, ``ada_paren_abstract_state_decl``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_multi_abstract_state_decl_f_decls(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_not_null_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_params_f_params(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes:
 * ``ada_abstract_state_decl``, ``ada_paren_abstract_state_decl``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_paren_abstract_state_decl_f_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_bin_op``, ``ada_identifier``, ``ada_paren_expr``, ``ada_un_op``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_pp_elsif_directive_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_pp_elsif_directive_f_then_kw(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_bin_op``, ``ada_identifier``, ``ada_paren_expr``, ``ada_un_op``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_pp_if_directive_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_pp_if_directive_f_then_kw(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_pragma_node_f_id(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_pragma_node_f_args(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return whether this pragma is ghost code or not. See SPARK RM 6.9.
 */
extern int ada_pragma_node_p_is_ghost_code(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return an array of ``BasicDecl`` instances associated with this pragma, or
 * an empty array if non applicable.
 */
extern int ada_pragma_node_p_associated_entities(
    ada_node *node,


    ada_node_array *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_private_node_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_protected_def_f_public_part(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_protected_def_f_private_part(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_protected_def_f_end_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_protected_node_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_bin_op``, ``ada_box_expr``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_identifier``,
 * ``ada_qual_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_range_spec_f_range(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_renaming_clause_f_renamed_object(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_reverse_node_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_select_when_part_f_cond_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_pragma_node``, ``ada_stmt``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_select_when_part_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return whether this statement is ghost code or not. See SPARK RM 6.9.
 */
extern int ada_stmt_p_is_ghost_code(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_accept_stmt_f_body_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_accept_stmt_f_entry_index_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_accept_stmt_f_params(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the entry which corresponds to this accept statement.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_accept_stmt_p_corresponding_entry(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_accept_stmt_with_stmts_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_accept_stmt_with_stmts_f_end_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_base_loop_stmt_f_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_pragma_node``, ``ada_stmt``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_base_loop_stmt_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_base_loop_stmt_f_end_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_begin_block_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_begin_block_f_end_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_decl_block_f_decls(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_decl_block_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_decl_block_f_end_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_case_stmt_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_case_stmt_f_pragmas(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_case_stmt_f_alternatives(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_extended_return_stmt_f_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_extended_return_stmt_f_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_if_stmt_f_cond_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_pragma_node``, ``ada_stmt``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_if_stmt_f_then_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_if_stmt_f_alternatives(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_pragma_node``, ``ada_stmt``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_if_stmt_f_else_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_named_stmt_f_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_base_loop_stmt``,
 * ``ada_block_stmt``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_named_stmt_f_stmt(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_select_stmt_f_guards(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_pragma_node``, ``ada_stmt``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_select_stmt_f_else_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_pragma_node``, ``ada_stmt``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_select_stmt_f_abort_stmts(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_attribute_ref``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_identifier``,
 * ``ada_qual_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_abort_stmt_f_names(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_assign_stmt_f_dest(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_assign_stmt_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_call_stmt_f_call(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_delay_stmt_f_has_until(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_delay_stmt_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_identifier``, ``ada_string_literal``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_exit_stmt_f_loop_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_exit_stmt_f_cond_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_identifier``, ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_goto_stmt_f_label_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_label_f_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_raise_stmt_f_exception_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_raise_stmt_f_error_message(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_identifier``, ``ada_qual_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_requeue_stmt_f_call_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_requeue_stmt_f_has_abort(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_membership_expr``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_return_stmt_f_return_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_simple_decl_stmt_f_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_identifier``, ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_subunit_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_package_body``,
 * ``ada_protected_body``, ``ada_subp_body``, ``ada_task_body``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_subunit_f_body(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the body in which this subunit is rooted.
 */
extern int ada_subunit_p_body_root(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_synchronized_node_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_tagged_node_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_char_literal``, ``ada_dotted_name``, ``ada_identifier``,
 * ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_task_def_f_interfaces(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_task_def_f_public_part(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_task_def_f_private_part(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_task_def_f_end_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_access_def_f_has_not_null(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_access_to_subp_def_f_has_protected(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_access_to_subp_def_f_subp_spec(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_anonymous_type_access_def_f_type_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_type_access_def_f_has_all(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_type_access_def_f_has_constant(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_type_access_def_f_subtype_indication(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_array_type_def_f_indices(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_array_type_def_f_component_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_derived_type_def_f_has_abstract(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_derived_type_def_f_has_limited(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_derived_type_def_f_has_synchronized(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_derived_type_def_f_subtype_indication(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_char_literal``, ``ada_dotted_name``, ``ada_identifier``,
 * ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_derived_type_def_f_interfaces(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_derived_type_def_f_record_extension(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_derived_type_def_f_has_with_private(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_enum_type_def_f_enum_literals(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_interface_type_def_f_interface_kind(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_char_literal``, ``ada_dotted_name``, ``ada_identifier``,
 * ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_interface_type_def_f_interfaces(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_box_expr``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_concat_op``, ``ada_cond_expr``, ``ada_decl_expr``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_mod_int_type_def_f_expr(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_private_type_def_f_has_abstract(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_private_type_def_f_has_tagged(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_private_type_def_f_has_limited(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_box_expr``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_concat_op``, ``ada_cond_expr``, ``ada_decl_expr``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_decimal_fixed_point_def_f_delta(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_box_expr``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_concat_op``, ``ada_cond_expr``, ``ada_decl_expr``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_decimal_fixed_point_def_f_digits(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_decimal_fixed_point_def_f_range(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_box_expr``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_concat_op``, ``ada_cond_expr``, ``ada_decl_expr``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_floating_point_def_f_num_digits(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_floating_point_def_f_range(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_box_expr``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_concat_op``, ``ada_cond_expr``, ``ada_decl_expr``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_null_literal``, ``ada_num_literal``,
 * ``ada_paren_expr``, ``ada_qual_expr``, ``ada_quantified_expr``,
 * ``ada_raise_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_ordinary_fixed_point_def_f_delta(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_ordinary_fixed_point_def_f_range(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_record_type_def_f_has_abstract(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_record_type_def_f_has_tagged(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_record_type_def_f_has_limited(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_record_type_def_f_record_def(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_signed_int_type_def_f_range(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the name node for this type expression, if applicable, else null
 */
extern int ada_type_expr_p_type_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Returns the type declaration designated by this type expression.
 */
extern int ada_type_expr_p_designated_type_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return the type declaration designated by this type expression as viewed
 * from the node given by origin_node.
 */
extern int ada_type_expr_p_designated_type_decl_from(
    ada_node *node,

        
        const ada_node*
        origin_node,

    ada_node *value_p
);


        



/*
 * Return the constraint that this type expression defines on its designated
 * subtype, if any.
 *
 * ``Origin``: Origin for this property's request. See :ref:`The origin
 * parameter` for more details
 */
extern int ada_type_expr_p_subtype_constraint(
    ada_node *node,

        
        const ada_node*
        origin,

    ada_node *value_p
);


        



/*
 * If this type expression designates a constrained discriminated type, return
 * an array of pairs, associating each discriminant to its actual or default
 * expression.
 */
extern int ada_type_expr_p_discriminant_constraints(
    ada_node *node,


    ada_internal_param_actual_array *value_p
);


        



/*
 * Returns whether this designates a definite subtype.
 */
extern int ada_type_expr_p_is_definite_subtype(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_anonymous_type_f_type_decl(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_subtype_indication_f_has_not_null(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_char_literal``, ``ada_dotted_name``, ``ada_identifier``,
 * ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_subtype_indication_f_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field may be null even when there are no parsing errors.
 */
extern int ada_subtype_indication_f_constraint(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Returns whether self denotes a static subtype or not (i.e. determinable at
 * compile time, see :rmlink:`4.9`).
 */
extern int ada_subtype_indication_p_is_static_subtype(
    ada_node *node,

        
        ada_bool
        imprecise_fallback,

    ada_bool *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_synthetic_type_expr_f_target_type(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_attribute_ref``,
 * ``ada_char_literal``, ``ada_dotted_name``, ``ada_identifier``,
 * ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_unconstrained_array_index_f_subtype_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field can contain one of the following nodes: ``ada_allocator``,
 * ``ada_attribute_ref``, ``ada_base_aggregate``, ``ada_bin_op``,
 * ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_dotted_name``,
 * ``ada_explicit_deref``, ``ada_format_string_literal``, ``ada_identifier``,
 * ``ada_null_literal``, ``ada_num_literal``, ``ada_paren_expr``,
 * ``ada_qual_expr``, ``ada_quantified_expr``, ``ada_raise_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * This field may be null even when there are no parsing errors.
 */
extern int ada_unconstrained_array_index_f_lower_bound(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_until_node_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_char_literal``, ``ada_dotted_name``, ``ada_identifier``,
 * ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_use_package_clause_f_packages(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_use_type_clause_f_has_all(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_attribute_ref``, ``ada_call_expr``, ``ada_char_literal``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_identifier``,
 * ``ada_qual_expr``, ``ada_reduce_attribute_ref``, ``ada_string_literal``,
 * ``ada_target_name``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_use_type_clause_f_types(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_value_sequence_f_iter_assoc(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_allocator``, ``ada_attribute_ref``, ``ada_base_aggregate``,
 * ``ada_bin_op``, ``ada_call_expr``, ``ada_char_literal``, ``ada_concat_op``,
 * ``ada_cond_expr``, ``ada_decl_expr``, ``ada_discrete_subtype_indication``,
 * ``ada_dotted_name``, ``ada_explicit_deref``, ``ada_format_string_literal``,
 * ``ada_identifier``, ``ada_membership_expr``, ``ada_null_literal``,
 * ``ada_num_literal``, ``ada_others_designator``, ``ada_paren_expr``,
 * ``ada_qual_expr``, ``ada_quantified_expr``, ``ada_raise_expr``,
 * ``ada_reduce_attribute_ref``, ``ada_string_literal``, ``ada_target_name``,
 * ``ada_un_op``, ``ada_update_attribute_ref``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_variant_f_choices(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_variant_f_components(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_variant_part_f_discr_name(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_variant_part_f_variant(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_with_clause_f_has_limited(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * When there are no parsing errors, this field is never null.
 */
extern int ada_with_clause_f_has_private(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * This field contains a list that itself contains one of the following nodes:
 * ``ada_char_literal``, ``ada_dotted_name``, ``ada_identifier``,
 * ``ada_string_literal``
 *
 * When there are no parsing errors, this field is never null.
 */
extern int ada_with_clause_f_packages(
    ada_node *node,


    ada_node *value_p
);


        



/*
 * Return whether this node is present
 */
extern int ada_with_private_p_as_bool(
    ada_node *node,


    ada_bool *value_p
);



/*
 * Event handlers
 */

/*
 * Create an event handler. When done with it, the result must be passed to
 * ``ada_dec_ref_event_handler``.
 *
 * Pass as ``data`` a pointer to hold your private data: it will be passed to
 * all callbacks below.
 *
 * ``destroy`` is a callback that is called by ``ada_dec_ref_event_handler`` to
 * leave a chance to free resources that ``data`` may hold. ``NULL`` can be
 * passed if nothing needs to be done.
 *
 * ``unit_requested`` is a callback that will be called when a unit is
 * requested.
 *
 * .. warning:: Please note that the unit requested callback can be called
 *    *many* times for the same unit, so in all likeliness, those events should
 *    be filtered if they're used to forward diagnostics to the user.
 *
 * ``unit_parsed`` is a callback that will be called when a unit is parsed.
 */
extern ada_event_handler
ada_create_event_handler(
   void *data,
   ada_event_handler_destroy_callback destroy_func,
   ada_event_handler_unit_requested_callback unit_requested_func,
   ada_event_handler_unit_parsed_callback unit_parsed_func
);

/*
 * Release an ownership share for this event handler. This destroys the event
 * handler if there are no shares left.
 */
extern void
ada_dec_ref_event_handler(ada_event_handler self);

/*
 * File readers
 */

/*
 * Create a file reader. When done with it, the result must be passed to
 * ``ada_dec_ref_file_reader``.
 *
 * Pass as ``data`` a pointer to hold your private data: it will be passed to
 * all callbacks below.
 *
 * ``destroy`` is a callback that is called by ``ada_dec_ref_file_reader`` to
 * leave a chance to free resources that ``data`` may hold.
 *
 * ``read`` is a callback. For a given filename/charset and whether to read the
 * BOM (Byte Order Mark), it tries to fetch the contents of the source file,
 * returned in ``Contents``. If there is an error, it must return it in
 * ``Diagnostic`` instead.
 */
extern ada_file_reader
ada_create_file_reader(
   void *data,
   ada_file_reader_destroy_callback destroy_func,
   ada_file_reader_read_callback read_func
);

/*
 * Release an ownership share for this file reader. This destroys the file
 * reader if there are no shares left.
 */
extern void
ada_dec_ref_file_reader(ada_file_reader self);




/*
 * Unit providers
 */

/*
 * Release an ownership share for this unit provider. This destroys the unit
 * provider if there are no shares left.
 */
extern void
ada_dec_ref_unit_provider(void *data);




/*
 * Misc
 */

/*
 * Return exception information for the last error that happened in the current
 * thread. Will be automatically allocated on error and free'd on the next
 * error.
 */
extern const ada_exception *
ada_get_last_exception(void);

/*
 * Return the name of the given exception kind. Callers are responsible for
 * free'ing the result.
 */
extern char *
ada_exception_name(ada_exception_kind kind);

/*
 * Kind for this token.
 */
extern int
ada_token_get_kind(ada_token *token);

/*
 * Return a human-readable name for a token kind.
 *
 * The returned string is dynamically allocated and the caller must free it
 * when done with it.
 *
 * If the given kind is invalid, return ``NULL`` and set the last exception
 * accordingly.
 */
extern char *
ada_token_kind_name(ada_token_kind kind);

/*
 * Return the source location range of the given token.
 */
extern void
ada_token_sloc_range(ada_token *token,
                                     ada_source_location_range *result);

/*
 * Return a reference to the next token in the corresponding analysis unit.
 */
extern void
ada_token_next(ada_token *token,
                               ada_token *next_token);

/*
 * Return a reference to the previous token in the corresponding analysis unit.
 */
extern void
ada_token_previous(ada_token *token,
                                   ada_token *previous_token);

/*
 * Compute the source buffer slice corresponding to the text that spans between
 * the ``First`` and ``Last`` tokens (both included). This yields an empty
 * slice if ``Last`` actually appears before ``First``. Put the result in
 * ``RESULT``.
 *
 * This returns ``0`` if ``First`` and ``Last`` don't belong to the same
 * analysis unit. Return ``1`` if successful.
 */
extern int
ada_token_range_text(ada_token *first,
                                     ada_token *last,
                                     ada_text *result);

/*
 * Return whether ``L`` and ``R`` are structurally equivalent tokens. This
 * means that their position in the stream won't be taken into account, only
 * the kind and text of the token.
 */
extern ada_bool
ada_token_is_equivalent(ada_token *left,
                                        ada_token *right);

/*
 * Tree rewriting
 */

/* ... context rewriting... */

/*
 * Return the rewriting handle associated to Context, or No_Rewriting_Handle if
 * Context is not being rewritten.
 */
extern ada_rewriting_handle
ada_rewriting_context_to_handle(
    ada_analysis_context context
);

/*
 * Return the analysis context associated to Handle
 */
extern ada_analysis_context
ada_rewriting_handle_to_context(
    ada_rewriting_handle handle
);

/*
 * Start a rewriting session for Context.
 *
 * This handle will keep track of all changes to do on Context's analysis
 * units. Once the set of changes is complete, call the Apply procedure to
 * actually update Context. This makes it possible to inspect the "old" Context
 * state while creating the list of changes.
 *
 * There can be only one rewriting session per analysis context, so this will
 * raise an Existing_Rewriting_Handle_Error exception if Context already has a
 * living rewriting session.
 */
extern ada_rewriting_handle
ada_rewriting_start_rewriting(
    ada_analysis_context context
);

/*
 * Discard all modifications registered in Handle and close Handle. This
 * invalidates all related unit/node handles.
 */
extern void
ada_rewriting_abort_rewriting(
    ada_rewriting_handle context
);

/*
 * Apply all modifications to Handle's analysis context. If that worked, close
 * Handle and return (Success => True). Otherwise, reparsing did not work, so
 * keep Handle and its Context unchanged and return details about the error
 * that happened.
 *
 * Note that on success, this invalidates all related unit/node handles.
 */
extern void
ada_rewriting_apply(
    ada_rewriting_handle context,
    ada_rewriting_apply_result *result
);

/*
 * Free the result of the ``Apply`` operation.
 */
extern void
ada_rewriting_free_apply_result(
    ada_rewriting_apply_result *result
);

/*
 * Return the list of unit rewriting handles in the given context handle for
 * units that the Apply primitive will modify.
 *
 * This returns the list as a dynamically allocated NULL-terminated array, that
 * the caller must free when done with it.
 */
extern ada_unit_rewriting_handle *
ada_rewriting_unit_handles(
    ada_rewriting_handle handle
);

/* ... unit rewriting... */

/*
 * Return the rewriting handle corresponding to Unit
 */
extern ada_unit_rewriting_handle
ada_rewriting_unit_to_handle(ada_analysis_unit context);

/*
 * Return the unit corresponding to Handle
 */
extern ada_analysis_unit
ada_rewriting_handle_to_unit(
    ada_unit_rewriting_handle handle
);

/*
 * Return the node handle corresponding to the root of the unit which Handle
 * designates.
 */
extern ada_node_rewriting_handle
ada_rewriting_unit_root(
    ada_unit_rewriting_handle handle
);

/*
 * Set the root node for the unit Handle to Root. This unties the previous root
 * handle. If Root is not No_Node_Rewriting_Handle, this also ties Root to
 * Handle.
 *
 * Root must not already be tied to another analysis unit handle.
 */
extern void
ada_rewriting_unit_set_root(
    ada_unit_rewriting_handle handle,
    ada_node_rewriting_handle root
);

/*
 * Return the text associated to the given unit.
 */
extern void
ada_rewriting_unit_unparse(
    ada_unit_rewriting_handle handle,
    ada_text *result
);

/* ... node rewriting... */

/*
 * Return the rewriting handle corresponding to Node.
 *
 * The owning unit of Node must be free of diagnostics.
 */
extern ada_node_rewriting_handle
ada_rewriting_node_to_handle(ada_base_node context);

/*
 * Return the node which the given rewriting Handle relates to. This can be the
 * null entity if this handle designates a new node.
 */
extern ada_base_node
ada_rewriting_handle_to_node(
    ada_node_rewriting_handle handle
);

/*
 * Return a handle for the rewriting context to which Handle belongs
 */
extern ada_rewriting_handle
ada_rewriting_node_to_context(
    ada_node_rewriting_handle handle
);

/*
 * Turn the given rewritten node Handles designates into text. This is the text
 * that is used in Apply in order to re-create an analysis unit.
 */
extern void
ada_rewriting_node_unparse(
    ada_node_rewriting_handle handle,
    ada_text *result
);

/*
 * Return the kind corresponding to Handle's node
 */
extern ada_node_kind_enum
ada_rewriting_kind(ada_node_rewriting_handle handle);

/*
 * Return a representation of ``Handle`` as a string.
 */
extern void
ada_rewriting_node_image(
    ada_node_rewriting_handle handle,
    ada_text *result
);

/*
 * Return whether this node handle is tied to an analysis unit. If it is not,
 * it can be passed as the Child parameter to Set_Child.
 */
extern int
ada_rewriting_tied(ada_node_rewriting_handle handle);

/*
 * Return a handle for the node that is the parent of Handle's node. This is
 * ``No_Rewriting_Handle`` for a node that is not tied to any tree yet.
 */
extern ada_node_rewriting_handle
ada_rewriting_parent(ada_node_rewriting_handle handle);

/*
 * Return the number of children the node represented by Handle has
 */
extern int
ada_rewriting_children_count(
    ada_node_rewriting_handle handle
);

/*
 * Return the node that is in the syntax ``Field`` for ``Handle``
 */
extern ada_node_rewriting_handle
ada_rewriting_child(
    ada_node_rewriting_handle handle,
    ada_introspection_member_ref field
);

/*
 * Return the list of children for ``Handle``.
 *
 * This returns the list as a dynamically allocated array with ``count``
 * elements.  The caller must free it when done with it.
 */
extern void
ada_rewriting_children(
    ada_node_rewriting_handle handle,
    ada_node_rewriting_handle **children,
    int *count
);

/*
 * If ``Child`` is ``No_Rewriting_Node``, untie the syntax field in ``Handle``
 * corresponding to ``Field``, so it can be attached to another one. Otherwise,
 * ``Child`` must have no parent as it will be tied to ``Handle``'s tree.
 */
extern void
ada_rewriting_set_child(
    ada_node_rewriting_handle handle,
    ada_introspection_member_ref field,
    ada_node_rewriting_handle child
);

/*
 * Return the text associated to the given token node.
 */
extern void
ada_rewriting_text(
    ada_node_rewriting_handle handle,
    ada_text *result
);

/*
 * Override text associated to the given token node.
 */
extern void
ada_rewriting_set_text(
    ada_node_rewriting_handle handle,
    ada_text *text
);

/*
 * If Handle is the root of an analysis unit, untie it and set New_Node as its
 * new root. Otherwise, replace Handle with New_Node in Handle's parent node.
 *
 * Note that: * Handle must be tied to an existing analysis unit handle. *
 * New_Node must not already be tied to another analysis unit handle.
 */
extern void
ada_rewriting_replace(
    ada_node_rewriting_handle handle,
    ada_node_rewriting_handle new_node
);

/*
 * Given a list of node rewriting handles ``H1``, ``H2``, ... ``HN``, replace
 * ``H1`` by ``H2`` in the rewritten tree, replace ``H2`` by ``H3``, etc. and
 * replace ``HN`` by ``H1``.
 *
 * Note that this operation is atomic: if it fails, no replacement is actually
 * performed.
 */
extern void
ada_rewriting_rotate(
    ada_node_rewriting_handle *handles,
    int count
);

/* ... list node rewriting... */

/*
 * Assuming ``Handle`` refers to a list node, return a handle to its first
 * child, or ``No_Node_Rewriting_Handle``` if it has no child node.
 */
extern ada_node_rewriting_handle
ada_rewriting_first_child(
    ada_node_rewriting_handle handle
);

/*
 * Assuming ``Handle`` refers to a list node, return a handle to its last
 * child, or ``No_Node_Rewriting_Handle``` if it has no child node.
 */
extern ada_node_rewriting_handle
ada_rewriting_last_child(
    ada_node_rewriting_handle handle
);

/*
 * Assuming ``Handle`` refers to the child of a list node, return a handle to
 * its next sibling, or ``No_Node_Rewriting_Handle``` if it is the last
 * sibling.
 */
extern ada_node_rewriting_handle
ada_rewriting_next_child(
    ada_node_rewriting_handle handle
);

/*
 * Assuming ``Handle`` refers to the child of a list node, return a handle to
 * its previous sibling, or ``No_Node_Rewriting_Handle``` if it is the first
 * sibling.
 */
extern ada_node_rewriting_handle
ada_rewriting_previous_child(
    ada_node_rewriting_handle handle
);

/*
 * Assuming ``Handle`` refers to the child of a list node, insert
 * ``New_Sibling`` as a new child in this list, right before ``Handle``.
 */
extern void
ada_rewriting_insert_before(
    ada_node_rewriting_handle handle,
    ada_node_rewriting_handle new_sibling
);

/*
 * Assuming ``Handle`` refers to the child of a list node, insert
 * ``New_Sibling`` as a new child in this list, right before ``Handle``.
 */
extern void
ada_rewriting_insert_after(
    ada_node_rewriting_handle handle,
    ada_node_rewriting_handle new_sibling
);

/*
 * Assuming ``Handle`` refers to a list node, insert ``New_Child`` to be the
 * first child in this list.
 */
extern void
ada_rewriting_insert_first(
    ada_node_rewriting_handle handle,
    ada_node_rewriting_handle new_sibling
);

/*
 * Assuming ``Handle`` refers to a list node, insert ``New_Child`` to be the
 * last child in this list.
 */
extern void
ada_rewriting_insert_last(
    ada_node_rewriting_handle handle,
    ada_node_rewriting_handle new_sibling
);

/*
 * Assuming Handle refers to the child of a list node, remove it from that
 * list.
 */
extern void
ada_rewriting_remove_child(
    ada_node_rewriting_handle handle
);

/* ... node creation... */

/*
 * Create a clone of the Handle node tree. The result is not tied to any
 * analysis unit tree.
 */
extern ada_node_rewriting_handle
ada_rewriting_clone(ada_node_rewriting_handle handle);

/*
 * Create a new node of the given Kind, with empty text (for token nodes) or
 * children (for regular nodes).
 */
extern ada_node_rewriting_handle
ada_rewriting_create_node(
    ada_rewriting_handle handle,
    ada_node_kind_enum kind
);

/*
 * Create a new token node with the given Kind and Text
 */
extern ada_node_rewriting_handle
ada_rewriting_create_token_node(
    ada_rewriting_handle handle,
    ada_node_kind_enum kind,
    ada_text *text
);

/*
 * Create a new regular node of the given Kind and assign it the given
 * Children.
 *
 * Except for lists, which can have any number of children, the size of
 * Children must match the number of children associated to the given Kind.
 * Besides, all given children must not be tied.
 */
extern ada_node_rewriting_handle
ada_rewriting_create_regular_node(
    ada_rewriting_handle handle,
    ada_node_kind_enum kind,
    ada_node_rewriting_handle *children,
    int count
);

/*
 * Create a tree of new nodes from the given Template string, replacing
 * placeholders with nodes in Arguments and parsed according to the given
 * grammar Rule.
 */
extern ada_node_rewriting_handle
ada_rewriting_create_from_template(
    ada_rewriting_handle handle,
    ada_text *src_template,
    ada_node_rewriting_handle *arguments,
    int count,
    ada_grammar_rule rule
);


      


/* Handling of string arrays */

/*
 * List of strings.
 */
typedef struct {
   int length;
   const char**c_ptr;
} ada_string_array_ptr_struct;

typedef ada_string_array_ptr_struct *ada_string_array_ptr;

/*
 * Free the given list of strings.
 */
extern void
ada_free_string_array (
   ada_string_array_ptr source_files
);

/* Project handling */

/*
 * Loaded GPR project file.
 */
typedef void *ada_gpr_project;

/*
 * Couple name/value to define a scenario variable for a project.
 */
typedef struct {
   char *name;
   char *value;
} ada_gpr_project_scenario_variable;

/*
 * Load the ``Project_File`` GPR file with the given scenario variables,
 * target, runtime and GPR configuration file (all optional).
 *
 * If ``Ada_Only`` is true, call ``Restrict_Autoconf_To_Languages`` to make GPR
 * only consider the Ada language.
 *
 * On success, set ``Project`` to a newly allocated ``ada_gpr_project``, as
 * well as a possibly empty list of error messages in ``Errors``. Raise an
 * ``Invalid_Project`` exception on failure.
 */
extern void
ada_gpr_project_load(
   const char *project_file,
   const ada_gpr_project_scenario_variable *scenario_vars,
   const char *target,
   const char *runtime,
   const char *config_file,
   int ada_only,
   ada_gpr_project *project,
   ada_string_array_ptr *errors
);

/*
 * Load an implicit project in the current directory. This function uses
 * ``GNATCOLL.Projects.Load_Implicit_Project`` to load the ``_default.gpr``
 * project file.
 */
extern void
ada_gpr_project_load_implicit(
   const char *target,
   const char *runtime,
   const char *config_file,
   ada_gpr_project *project,
   ada_string_array_ptr *errors
);

/*
 * Free resources allocated for ``Self``.
 */
extern void
ada_gpr_project_free (ada_gpr_project self);

/*
 * Create a project provider using the given GPR project ``Self``.
 *
 * If ``Project`` is passed, it must be the name of a sub-project. If the
 * selected project contains conflicting sources, raise an ``Project_Error``
 * exception.
 *
 * The returned unit provider assumes that resources allocated by ``Self`` are
 * kept live: it is the responsibility of the caller to make ``Self`` live at
 * least as long as the returned unit provider.
 */
extern ada_unit_provider
ada_gpr_project_create_unit_provider (
    ada_gpr_project self,
    const char *project
);

/*
 * Compute the list of source files in the given GPR project according to
 * ``Mode`` (whose value maps to positions in the
 * ``Libadalang.Project_Provider.Source_Files_Mode`` enum) and return it.
 */
extern ada_string_array_ptr
ada_gpr_project_source_files (
   ada_gpr_project self,
   int mode,
   const char **projects_data,
   int projects_length
);

/*
 * Try to detect the default charset to use for the given project.
 *
 * Restrict the detection to the subproject ``Project``, or to ``Self``'s root
 * project if left to ``Prj.No_Project``.
 *
 * Note that, as of today, this detection only looks for the ``-gnatW8``
 * compiler switch: other charsets are not supported.
 */
extern char *
ada_gpr_project_default_charset (
    ada_gpr_project self,
    const char *project
);

/*
 * Wrapper around ``Initialize_Context_From_Project`` to initialize ``Context``
 * (an already allocated but not yet initialized analysis context) from
 * ``Self``.
 */
extern void
ada_gpr_project_initialize_context(
   ada_gpr_project self,
   ada_analysis_context context,
   const char *project,
   ada_event_handler event_handler,
   int with_trivia,
   int tab_stop
);

/*
 * Load the project file at ``Project_File`` and return a unit provider that
 * uses it.
 *
 * If ``Project`` is passed, use it to provide units, otherwise, try use the
 * whole project tree.
 *
 * As unit providers must guarantee that there exists at most one source file
 * for each couple (unit name, unit kind), aggregate projects that contains
 * several conflicting units are not supported: trying to load one will yield
 * an error (see below).
 *
 * If not ``NULL``, ``Scenario_Vars`` must point to an array of
 * ``ada_project_scenario_variable`` couples to provide scenario variables for
 * this project. The last element of this array must end with a ``{ NULL, NULL
 * }`` couple.
 *
 * If not ``NULL``, ``target`` and ``runtime`` must point to valid NULL-
 * terminated strings.
 *
 * When done with it, the result must be free'd with
 * ``ada_destroy_unit_provider``.
 *
 * If the requested project is invalid (error while opening the file, error
 * while analysing its syntax, ...), or if it is an unsupported aggregate
 * project, this returns ``NULL``.
 */
extern ada_unit_provider
ada_create_project_unit_provider(
   const char *project_file,
   const char *project,
   const ada_gpr_project_scenario_variable *scenario_vars,
   const char *target,
   const char *runtime
);

/* Auto unit provider */

/*
 * Return a unit provider that knows which compilation units are to be found in
 * the given list of source files.
 *
 * This knowledge is built trying to parse all given input files as Ada source
 * files and listing the compilation units found there. Files that cannot be
 * parsed properly are discarded. If two compilation units are found for the
 * same unit, the first that is found in the given input files is taken and the
 * other ones are discarded.
 *
 * Source files are decoded using the given charset. If it is ``NULL``, the
 * default charset (ISO-8859-1) is used.
 *
 * ``input_files`` must point to a ``NULL``-terminated array of filenames.
 * Once this function returns, this array and the strings it contains can be
 * deallocated.
 *
 * When done with it, the result must be free'd with
 * ``ada_destroy_unit_provider``.
 *
 * .. TODO: Find a way to report discarded source files/compilation units.
 */
extern ada_unit_provider
ada_create_auto_provider(
   const char **input_files,
   const char *charset
);

/* Preprocessor */

/*
 * Load the preprocessor data file at ``Filename`` using, directory names in
 * the ``Path_Data``/``Path_Length`` array to look for for it and the
 * definition files it references.
 *
 * If ``Line_Mode`` is not null, use it to force the line mode for source files
 * on which the preprocessor is enabled.  Forcing the line mode is often needed
 * as the default is to remove lines that contain preprocessor directives and
 * disabled code, which breaks the line number correspondence between original
 * source code and preprocessed one.  Forcing to ``blank_lines`` or
 * ``comment_lines`` preserves this correspondence.
 *
 * Return a file reader that preprocesses sources accordingly.
 */
extern ada_file_reader
ada_create_preprocessor_from_file(
   const char *filename,
   const char **path_data,
   int path_length,
   const int *line_mode
);

/*
 * Create preprocessor data from compiler arguments found in the given GPR
 * project ``Self`` (``-gnatep`` and ``-gnateD`` compiler switches), or from
 * the ``Project`` sub-project (if the argument is passed).
 *
 * If ``Line_Mode`` is not null, use it to force the line mode in each
 * preprocessed source file.
 *
 * Note that this function collects all arguments and returns an approximation
 * from them: it does not replicates exactly gprbuild's behavior. This may
 * raise a ``File_Read_Error`` exception if this fails to read a preprocessor
 * data file and a ``Syntax_Error`` exception if one such file has invalid
 * syntax.
 *
 * The returned file reader assumes that resources allocated by ``Self`` are
 * kept live: it is the responsibility of the caller to make ``Self`` live at
 * least as long as the returned file reader.
 */
extern ada_file_reader
ada_gpr_project_create_preprocessor(
   ada_gpr_project self,
   const char *project,
   const int *line_mode
);

/* Config pragmas */

/*
 * Assign in ``Context`` configuration pragmas files to analysis units as
 * described in ``Global_Pragmas`` (configuration pragmas file that applies to
 * all analysis units, or null) and ``Local_Pragmas`` (mapping that associates
 * an analysis unit to the local configuration pragmas file that applies to
 * it).
 *
 * The ``Local_Pragmas`` mapping is encoded as a NULL-trailing analysis unit
 * array that describes a unit-to-unit mapping: for N associations,
 * ``Local_Pragmas[2 * (N - 1)]`` is they key and ``Local_Pragmas[2 * (N - 1) +
 * 1]`` is the value.
 *
 * This raises a ``Precondition_Failure`` exception if any analysis unit in
 * ``Mapping`` does not belong to ``Context`` or if an analysis unit appears
 * twice as a key in ``Mapping``.
 */
extern void
ada_set_config_pragmas_mapping(
   ada_analysis_context context,
   ada_analysis_unit global_pragmas,
   ada_analysis_unit *local_pragmas
);

/* Target information */

/*
 * Built in types characteristics for a given target platform.
 */
typedef void *ada_target_information;

/*
 * Read target information from ``filename`` and return it. Raise an
 * ``Langkit_Support.Errors.Invalid_Input`` exception if it cannot be read of
 * if there is any trouble decoding it.
 */
extern ada_target_information
ada_target_info_load(const char *filename);

/*
 * Free resources allocated for ``self``.
 */
extern void
ada_target_info_free(ada_target_information self);

/*
 * Assign target information to ``self``. This will affect specific bound
 * values for types defined in the built in Standard package.
 */
extern void
ada_target_info_set(ada_analysis_context self,
                                    ada_target_information info);



#ifdef __cplusplus
}
#endif

#endif
