//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// UFS 読み込み
//

#pragma once

#include "device.h"
#include "ffs_dinode.h"
#include "ffs_fs.h"

class SCSIDisk;

// ディスクラベル (1セクタ目)
// svnweb.freebsd.org/csrg/sys/luna68k/stand/omron_disklabel.h
struct scd_dk_label
{
	char asciilabel[128];	/* for compatibility */
	char pad[512 - (128 + 8*8 + 11*2 + 4)];
	uint16 badchk;			/* checksum of bad track */
	uint32 maxblk;			/* # of total logical block */
	uint16 dtype;			/* disk drive type */
	uint16 ndisk;			/* # of disk drives */
	uint16 ncyl;			/* # of data cylinders */
	uint16 acyl;			/* # of altenate cylinders */
	uint16 nhead;			/* # of heads in this partition */
	uint16 nsect;			/* # of 512 byte sectors per track */
	uint16 bhead;			/* identifies proper label locations */
	uint16 ppart;			/* physical partition # */
	struct {				/* logical partitions */
		uint32 blkno;		/* starting block */
		uint32 nblk;		/* number of blocks */
	} map[8];
	uint16 magic;			/* identifies this label format */
	uint16 cksum;			/* xor checksum of sector */

	static const uint16 DKL_MAGIC = 0xdabe;
};
static_assert(sizeof(scd_dk_label) == 512, "size must be 512");

// オレオレ inode 情報クラス
class inodefile final
{
 public:
	explicit inodefile(Device *parent_);
	~inodefile();

	// ディレクトリエントリからファイルを検索
	uint32 FindFile(const std::string& name);

	uint32 ino {};				// inode 番号
	struct ufs1_dinode di {};	// disk inode 情報 (ゲストエンディアンで保持)
	std::vector<uint8> data {};	// ファイル(またはディレクトリ)本文
 private:
	Device *parent {};			// 親デバイス
};

// ファイルシステムハンドルっぽいもの
class Filesys final
{
 public:
	explicit Filesys(Device *parent_);
	~Filesys();

	// ファイルシステムをオープン
	bool Mount(SCSIDisk *hd_, uint32 start, uint32 size);

	// ファイル名からファイルを探す
	bool OpenFile(inodefile& inode, const std::string& filename);

	// inode で示されるファイルかディレクトリの中身を読み込む
	void ReadData(inodefile& inode);

	std::string errstr {};	// エラーメッセージ

 private:
	// inode 情報を読み込む
	void Readi(inodefile& inode);

	// 指定のブロックを読み出す
	void PeekBlock(void *buf, uint32 b, uint32 len) const;

	Device *parent {};		// 親デバイス
	SCSIDisk *hd {};		// ターゲット SCSI Disk

	// パーティション情報
	uint32 part_start {};	// パーティションの先頭セクタ
	uint32 part_size {};	// パーティションのサイズ(セクタ数)

	// スーパーブロック情報
	// fs の(たぶんほとんどの)整数パラメータは読み込み時にホストエンディアン
	// に変換してあるのでそのままアクセスしてよい。構造体とかまでは全部対応
	// してないので、必要になったら書き足すこと。
	struct fs *fs {};		// 体裁を揃えるためポインタを使いたい
	struct fs fs0 {};		// こっちが実体
};
