//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// MPU 共通部
//

#pragma once

#include "device.h"
#include "message.h"
#include <vector>

class Debugger;
class MainbusDevice;
class Syncer;

// 全 MPU の共通部
class MPUDevice : public Device
{
	using inherited = Device;
 public:
	explicit MPUDevice(uint objid_);
	~MPUDevice() override;

	bool Init() override;
	void PowerOff() override;

	// 現在実行中の命令の PC を取得
	virtual uint32 GetPPC() const = 0;

 protected:
	// 1命令実行イベント
	Event *exec_event {};

	Debugger *debugger {};
};

// メイン MPU (680x0/88xxx0) 共通部
class MainMPUDevice : public MPUDevice
{
	using inherited = MPUDevice;
 protected:
	// CPU の実行状態
	static const uint32 CPU_STATE_NORMAL	= 0;
	static const uint32 CPU_STATE_STOP		= 1;
	static const uint32 CPU_STATE_HALT		= 2;

 public:
	MainMPUDevice();
	~MainMPUDevice() override;

	bool Init() override;

	// 現在の VBR を取得
	virtual uint32 GetVBR() const = 0;

	// 現在アクセス中の論理アドレス、物理アドレスを取得。
	// Laddr == Paddr が MMU オフなのか PA=VA で運用中なのかの区別は
	// ここでは付かない。
	virtual uint32 GetLaddr() const = 0;
	virtual uint32 GetPaddr() const = 0;

	// CPU の実行状態 (uint32) を返す。
	uint32 GetState() const noexcept { return cpu_state; }

	// MPU 名を取得する。("MC68030" みたいなやつ、ログ名とは別)
	virtual const char *GetMPUName() const = 0;

	// MPU クロック [kHz] を取得する。
	uint GetClockSpeed() const { return clock_khz; }

	// MPU クロック [nsec] を取得する。
	uint64 GetClock_nsec() const { return clock_nsec; }

	// 割り込み発生を MPU に通知
	virtual void Interrupt(int level) = 0;

	// デバイスが CI (キャッシュ禁止) をセットする。
	virtual void SetCI() = 0;

	// 例外カウンタ
	std::vector<uint64> excep_counter {};

 protected:
	void TraceMessage(MessageID, uint32);
	virtual void SetTrace(bool enable) = 0;

	// CPU の状態
	void ChangeState(uint32 new_state);

	// CPU の実行状態
	uint32 cpu_state {};

	uint clock_khz {};		// MPU クロック [kHz]
	uint64 clock_nsec {};	// MPU クロック [nsec]

	bool resetting {};		// リセット処理中フラグ

	MainbusDevice *mainbus {};
	Syncer *syncer {};

	// Round Mode (モニタ表示用)
	static const char * const rmstr[4];
};

static inline MainMPUDevice *GetMPUDevice() {
	return Object::GetObject<MainMPUDevice>(OBJ_MPU);
}
