<?php
//
// nono
// Copyright (C) 2022 nono project
// Licensed under nono-license.txt
//

	$inst0table = read_instructions("instructions.txt");
	$funchash = make_funchash($inst0table);

	$insttable = expand_rrr($inst0table, "rrr", "r");
	$insttable = expand_rrr($insttable, "sss", "s");
	$insttable = expand_fff($insttable);
	$insttable = expand_ww($insttable);
	$insttable = expand_zz($insttable);
	$insttable = expand_vvv($insttable);
	$insttable = expand_bbb($insttable);

	$insttable = make($insttable);

	output_switch($insttable);
	output_header();

	exit(0);
?>
<?php
function usage()
{
	global $argv;

	print "Usage: ${argv[0]}\n";
	exit(1);
}

// instructions.txt を読み込んで $insttable*[] 配列にする
function read_instructions($filename)
{
	$fp = fopen($filename, "r");
	if ($fp === false) {
		print "fopen failed: {$filename}\n";
		exit(1);
	}

	$insttable = array();

	while (($line = fgets($fp))) {
		$line = trim(preg_replace("/;.*/", "", $line));
		if ($line == "") {
			continue;
		}

		// 1行はこんな感じ
		// 00rrr100		01	inc_r	INC r
		// CB:00000rrr	01	rlc		RLC r
		// 列は1つ以上のタブで区切られている。
		// $bits … ビットパターン
		// $cpus … CPU 区別
		// $name … 関数名
		// $text … ニーモニック

		$column = preg_split("/\t+/", $line, -1, PREG_SPLIT_NO_EMPTY);
		$bits = $column[0];
		// 1バイト目なら ":" をつけて書式を揃えておく
		if (!preg_match("/:/", $bits)) {
			$bits = ":{$bits}";
		}
		$cpus = $column[1];
		$name = $column[2];
		$text = isset($column[3]) ? $column[3] : strtoupper($name);

		$newinst = array(
			"bits"	=> trim($bits),
			"cpus"	=> trim($cpus),
			"name"	=> trim($name),
			"text"	=> trim($text),
		);

		// $insttable[] に追加。
		$insttable[$bits] = $newinst;
	}
	fclose($fp);

	if (0) {
		print "read_instruction:\n";
		foreach ($insttable as $bits => $inst) {
			printf("%s|%s|%-10s|%s\n",
				$inst["bits"], $inst["cpus"], $inst["name"], $inst["text"]);
		}
		print "\n";
	}

	return $insttable;
}

// 展開前の $insttable から $funchash[] を作成する。
// $funchash は関数名をキーにしたハッシュで
// $funchash = array(
//  "add_hl_ww" => array("00ww1001|01R:\tADD HL,ww",
//                    "DD:00ww1001|01R:\tADD IX,ww"),
//  :
// ); のように、関数名とそれに紐づくニーモニックを集めたもの。
function make_funchash($insttable)
{
	$funchash = array();
	foreach ($insttable as $bits => $inst) {
		// $bits は1ワード目なら : から始まっているので取り除く
		$bit2 = preg_replace("/^:/", "", $bits);
		$name = $inst["name"];
		if (!isset($funchash[$name])) {
			$funchash[$name] = array();
		}
		$funchash[$name][] = "{$bit2}|{$inst["cpus"]}:\t{$inst["text"]}";
	}

	if (0) {
		print "make_funchash:\n";
		foreach ($funchash as $name => $lines) {
			printf("%-11s",  $name);
			foreach ($lines as $str) {
				print " \"{$str}\"";
			}
			print "\n";
		}
		print "\n";
	}

	return $funchash;
}

// $insttable の rrr を展開する。
// regbits は "rrr" か "sss" で、bits 内を置換する文字列。
// regtext は "r" か "s" で、text 内を置換する文字列。
function expand_rrr($insttable, $regbits, $regtext)
{
	$rrr = array(
		"000" => "B",
		"001" => "C",
		"010" => "D",
		"011" => "E",
		"100" => "H",
		"101" => "L",
	//	"110" (HL) は出力しない
		"111" => "A",
	);
	$inst2table = array();
	foreach ($insttable as $bits => $inst) {
		if (preg_match("/{$regbits}/", $bits)) {
			foreach ($rrr as $rbits => $rname) {
				$inst2 = $inst;
				$newbits = preg_replace("/{$regbits}/", $rbits, $bits);
				$inst2["bits"] = $newbits;
				$inst2["text"] =
					preg_replace("/{$regtext}/", $rname, $inst["text"]);
				$inst2table[$newbits] = $inst2;
			}
		} else {
			// 上書きする際は後者優先
			$inst2table[$bits] = $inst;
		}
	}

	// デバッグ表示
	if (0) {
		print "expand_rrr({$regbits})\n";
		foreach ($inst2table as $inst) {
			printf("%s|%-10s|%s\n",
				$inst["bits"], $inst["name"], $inst["text"]);
		}
	}

	return $inst2table;
}

// $insttable の fff を展開する。
function expand_fff($insttable)
{
	$fff = array(
		"000" => "NZ",
		"001" => "NC",
		"010" => "PO",
		"011" => "P",
		"100" => "Z",
		"101" => "C",
		"110" => "PE",
		"111" => "M",
	);

	$inst2table = array();
	foreach ($insttable as $bits => $inst) {
		if (preg_match("/fff/", $bits)) {
			foreach ($fff as $fbits => $fname) {
				$inst2 = $inst;
				$newbits = preg_replace("/fff/", $fbits, $bits);
				$inst2["bits"] = $newbits;
				$inst2["text"] = preg_replace("/f/", $fname, $inst["text"]);
				$inst2table[$newbits] = $inst2;
			}
		} else {
			// 上書きする際は後者優先
			$inst2table[$bits] = $inst;
		}
	}

	// デバッグ表示
	if (0) {
		print "expand_fff\n";
		foreach ($inst2table as $inst) {
			printf("%s|%-10s|%s\n",
				$inst["bits"], $inst["name"], $inst["text"]);
		}
	}

	return $inst2table;
}

// $insttable の ww を展開する。
function expand_ww($insttable)
{
	$ww = array(
		"00" => "BC",
		"01" => "DE",
		"10" => "HL",
		"11" => "SP",
	);

	$inst2table = array();
	foreach ($insttable as $bits => $inst) {
		if (preg_match("/ww/", $bits)) {
			foreach ($ww as $wbits => $wname) {
				$inst2 = $inst;
				$newbits = preg_replace("/ww/", $wbits, $bits);
				$inst2["bits"] = $newbits;
				$inst2["text"] = preg_replace("/ww/", $wname, $inst["text"]);
				$inst2table[$newbits] = $inst2;
			}
		} else {
			// 上書きする際は後者優先
			$inst2table[$bits] = $inst;
		}
	}

	// デバッグ表示
	if (0) {
		print "expand_ww\n";
		foreach ($inst2table as $inst) {
			printf("%s|%-10s|%s\n",
				$inst["bits"], $inst["name"], $inst["text"]);
		}
	}

	return $inst2table;
}

// $insttable の zz を展開する。
function expand_zz($insttable)
{
	$zz = array(
		"00" => "BC",
		"01" => "DE",
		"10" => "HL",
		"11" => "AF",
	);

	$inst2table = array();
	foreach ($insttable as $bits => $inst) {
		if (preg_match("/zz/", $bits)) {
			foreach ($zz as $zbits => $zname) {
				$inst2 = $inst;
				$newbits = preg_replace("/zz/", $zbits, $bits);
				$inst2["bits"] = $newbits;
				$inst2["text"] = preg_replace("/zz/", $zname, $inst["text"]);
				$inst2table[$newbits] = $inst2;
			}
		} else {
			// 上書きする際は後者優先
			$inst2table[$bits] = $inst;
		}
	}

	// デバッグ表示
	if (0) {
		print "expand_zz\n";
		foreach ($inst2table as $inst) {
			printf("%s|%-10s|%s\n",
				$inst["bits"], $inst["name"], $inst["text"]);
		}
	}

	return $inst2table;
}

// $insttable の vvv を展開する。
function expand_vvv($insttable)
{
	$vvv = array(
		"000" => "00H",
		"001" => "10H",
		"010" => "20H",
		"011" => "30H",
		"100" => "08H",
		"101" => "18H",
		"110" => "28H",
		"111" => "38H",
	);

	$inst2table = array();
	foreach ($insttable as $bits => $inst) {
		if (preg_match("/vvv/", $bits)) {
			foreach ($vvv as $vbits => $vname) {
				$inst2 = $inst;
				$newbits = preg_replace("/vvv/", $vbits, $bits);
				$inst2["bits"] = $newbits;
				$inst2["text"] = preg_replace("/v/", $vname, $inst["text"]);
				$inst2table[$newbits] = $inst2;
			}
		} else {
			// 上書きする際は後者優先
			$inst2table[$bits] = $inst;
		}
	}

	// デバッグ表示
	if (0) {
		print "expand_vvv\n";
		foreach ($inst2table as $inst) {
			printf("%s|%-10s|%s\n",
				$inst["bits"], $inst["name"], $inst["text"]);
		}
	}

	return $inst2table;
}

// $insttable の bbb を展開する。
function expand_bbb($insttable)
{
	$inst2table = array();
	foreach ($insttable as $bits => $inst) {
		if (preg_match("/bbb/", $bits)) {
			for ($b = 0; $b < 8; $b++) {
				$inst2 = $inst;
				$nbits = "00" . base_convert($b, 10, 2);
				$nbits = substr($nbits, -3);
				$newbits = preg_replace("/bbb/", $nbits, $bits);
				$inst2["bits"] = $newbits;
				$inst2["text"] = preg_replace("/b/", "{$b}", $inst["text"]);
				$inst2table[$newbits] = $inst2;
			}
		} else {
			// 上書きする際は後者優先
			$inst2table[$bits] = $inst;
		}
	}

	// デバッグ表示
	if (0) {
		print "expand_bbb\n";
		foreach ($inst2table as $inst) {
			printf("%s|%-10s|%s\n",
				$inst["bits"], $inst["name"], $inst["text"]);
		}
	}

	return $inst2table;
}

// ":00000000" => {
//   "bits" => ":00000000",
//   "name" => "nop", ...
// }
// 形式をプレフィックスを1段目にして
// "" => {
//   "00000000" => {
//     "bits" => "00000000",
//     "name" => "nop", ...
//   },
// }
// の形式に組み替えたものを返す。
//
// ついでに namehash[$name] => $name の一覧もグローバル変数で返す。
function make($insttable)
{
	global $argv;
	global $namehash;

	$inst2table = array();

	ksort($insttable);
	foreach ($insttable as $bits => $inst) {
		preg_match("/^([^:]*):(.*)/", $bits, $m);
		$pre  = $m[1];
		$bits = $m[2];

		$inst2 = $inst;
		$inst2["bits"] = $bits;

		if (!isset($inst2table[$pre])) {
			$inst2table[$pre] = array();
		}
		$inst2table[$pre][$bits] = $inst2;
	}

	if (0) {
		print "make:\n";
		foreach ($inst2table as $pre => $table) {
			foreach ($table as $bits => $inst) {
				printf("%5s|%s|%-10s|%s\n",
					$pre, $inst["bits"], $inst["name"], $inst["text"]);
			}
		}
		print "\n";
	}

	// 00 テーブルはすべて埋まっていると分かっているので、ここでチェック。
	// $bits によるハッシュになっているので数だけ数えればいいか?
	if (count($inst2table[""]) != 256) {
		printf("{$argv[0]}: Number of entries in table 00 is wrong: %d\n",
			count($inst2table[""]));
		exit(1);
	}
	// 他のテーブルはすべて埋まってるわけではないので、数えない。

	foreach ($inst2table as $pre => $table) {
		foreach ($table as $bits => $inst) {
			$name = $inst["name"];
			$namehash[$name] = $name;
		}
	}
	ksort($namehash);

	return $inst2table;
}


function output_switch($insttable)
{
	foreach ($insttable as $pre => $table) {
		output_switch_sub($pre, $table);
	}
}

function output_switch_sub($pre, $table)
{
	global $inst0table;
	global $funchash;

	// func2 は [256] => name 形式で、存在してるものだけ抜き出したもの。
	$func2 = array();
	foreach ($table as $bits => $inst) {
		$i = base_convert($bits, 2, 10);
		$i += 0;
		$func2[$i] = $inst["name"];
	}

	$out = "";
	$cpp = "";
	for ($i = 0; $i < 256; $i++) {
		if (!isset($func2[$i])) {
			continue;
		}
		$funcname = $func2[$i];

		$cases = array();
		// この関数名を持つブロックに属する人全員のコメント集める。
		// このブロック自身もここで集計するため $j は $i から始める。
		for ($j = $i; $j < 256; $j++) {
			if (isset($func2[$j]) && $func2[$j] == $funcname) {
				$cases[] = $j;

				// 集計したら消す
				unset($func2[$j]);
			}
		}

		// case 部分の出力。
		// 4個くらいまでなら case 文を羅列、
		// それ以上連続してたら範囲で出力。
		$cont = true;
		for ($k = 1; $k < count($cases); $k++) {
			if ($cases[$k] - $cases[$k - 1] != 1) {
				$cont = false;
				break;
			}
		}
		if ($cont == true && count($cases) > 4) {
			$c = array_shift($cases);
			$bits = decbin8($c);
			$out .= sprintf("\t case 0b{$bits}	// 0x%02x\n", $c);

			$c = array_pop($cases);
			$bits = decbin8($c);
			$out .= sprintf("\t  ... 0b{$bits}:	// 0x%02x\n", $c);
		} else {
			foreach ($cases as $c) {
				$bits = decbin8($c);
				$out .= sprintf("\t case 0b{$bits}:	// 0x%02x\n", $c);
			}
		}
		// 関数
		$out .= "\t\tOP_FUNC({$funcname});\n";
		$out .= "\t\tbreak;\n";

		// ソースの雛形を出力。
		// DD は基本 00 と同じ処理関数を使うので、関数本体の出力は1回のみ。
		if (!isset($funchash[$funcname])) {
			print "$funcname not found\n";
			exit(1);
		}
		if (!isset($funchash[$funcname]["processed"])) {
			// $funchash からこの関数のコメントを生成
			$lines = $funchash[$funcname];
			// プレフィックスの長さを調べる
			$plen = 0;
			foreach ($lines as $line) {
				$m = preg_split("/\|/", $line);
				preg_match("/([A-Z_]+:)?(.*)/", $m[0], $b);
				$plen = max($plen, strlen($b[1]));
			}
			foreach ($lines as $line) {
				$m = preg_split("/\|/", $line);
				preg_match("/([A-Z_]+:)?(.*)/", $m[0], $b);
				$cpp .= sprintf("// %{$plen}s%s: %s\n", $b[1], $b[2], $m[1]);
			}
			$cpp .= "OP_DEF({$funcname})\n";
			$cpp .= "{\n";
			$cpp .= "\tOP_FUNC(unimpl);\n";
			$cpp .= "}\n";
			$cpp .= "\n";

			$funchash[$funcname]["processed"] = true;
		}
	}

	if ($pre == "") {
		$pre = "00";
	} else {
		$pre = strtolower($pre);
	}
	write_file("switch_{$pre}.inc.new", $out);
	write_file("ops_{$pre}.cpp.new", $cpp);
}

// ヘッダを出力
function output_header()
{
	global $namehash;

	$out = "";
	foreach ($namehash as $name) {
		$out .= "OP_PROTO({$name});\n";
	}

	write_file("ops.h.new", $out);
}

// 数値 $num を2進数8桁の文字列にして返す
function decbin8($num)
{
	$str = str_repeat("0", 7) . decbin($num);
	$str = substr($str, -8);
	return $str;
}

// ファイル出力
function write_file($filename, $str)
{
	$fp = fopen($filename, "w");
	fwrite($fp, $str);
	fclose($fp);

	print "Output: {$filename}\n";
}
?>
