//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

#pragma once

#include "disasm.h"
#include "m680x0.h"

// 逆アセンブル
class m680x0disasm final : public DisasmLine
{
	using inherited = DisasmLine;
 public:
	m680x0disasm();
	~m680x0disasm() override;

	// 逆アセンブルを実行
	bool Exec(DebuggerMemoryStream *mem_) override;

	// IOCS コール名を返す。なければ NULL を返す。(分岐履歴から呼ばれる)
	static const char *GetIOCSName(int n) { return x68k_iocscall[n]; }

	// DOS コール名を返す。なければ NULL を返す。(分岐履歴から呼ばれる)
	static const char *GetDOSName(int n) { return x68k_doscall[n]; }

	// FPgen の命令名を返す。なければ "" を返す。(FPU モニタからも呼ばれる)
	static const char *make_fpgen(uint);

 private:
	uint32 pc0 {};			// 命令先頭アドレス

	enum Reg {
		CCR,
		SR,
		PC,
		USP,

		MSP,
		ISP,
		VBR,
		SFC,
		DFC,
		CACR,
		CAAR,

		TC,
		MMUSR,
		SRP,
		CRP,
		TT0,
		TT1,

		FPCR,
		FPSR,
		FPIAR,

		URP,
		ITT0,
		ITT1,
		DTT0,
		DTT1,

		BUSCR,
		PCR,

		MaxReg,
	};

	// L/S/X/P/W/D/B/(Pk) は FMOVE <ea>,FPn の時の並び順。
	enum Size {
		L = 0,	// Long
		S,		// Single
		X,		// eXtended
		P,		// Packed
		W,		// Word
		D,		// Double
		B,		// Byte
		Pk,		// Packed with dynamic k-factor
		Q,		// Quad
		None,
	};

	uint32 fetch2();
	uint32 peek2();			// PC を進めずワードを取得 (IOCSコール用)
	uint32 fetch4();

	std::string make_reg(Reg);

	std::string hex8(uint32);
	std::string hex16(uint32);
	std::string hex32(uint32);
	std::string shex8(uint32);
	std::string shex16(uint32);
	std::string shex32(uint32);
	std::string make_ea();
	std::string make_ea(uint);
	std::string make_eaix(std::string basereg);

	std::string addr2(uint32 addr);	// addr をワードアドレスとして
	std::string addr4(uint32 addr);	// addr をロングワードアドレスとして
	std::string addr2();		// フェッチした値をワードアドレスとして
	std::string addr4();		// フェッチした値をロングワードアドレスとして
	std::string disp2();		// フェッチしたワードディスプレースメント
	std::string disp4();		// フェッチしたロングワードディスプレースメント
	std::string imm1();
	std::string imm2();
	std::string imm4();
	std::string make_dn(uint);
	std::string make_an(uint);
	std::string make_rn(uint);
	std::string make_anin(uint);
	std::string make_anpd(uint);
	std::string make_anpi(uint);
	std::string make_q8();
	std::string make_reglist();
	void make_reglist_half(std::string& s, const char *reg, uint16 mask);
	std::string make_bf();
	std::string make_fc();
	std::string make_movec();
	std::string make_fpn(uint n);
	std::string make_fpcc(uint16 ir);
	uint make_fpctls(std::string& s);
	std::string make_fpnlist();

	static const char * const reg_names[];
	static const char * const size_str[];
	static const char * const conditions[];
	static const char * const fpconditions[];
	static const char * const btst_names[];
	static const char * const fpgen_names[0x3b];
	static const char * const fmovecr_text[];
	static const char * const x68k_doscall[256];
	static const char * const x68k_fpcall[256];
	static const char * const x68k_iocscall[256];
	void ops_cmp2chk2(Size sz);
	void ops_btst_movep();
	void ops_cas(Size sz);
	void ops_cas2(Size sz);
	void ops_moves(Size sz);
	void ops_movep();
	void ops_btst_imm_ea();
	void ops_move(Size sz);
	void ops_movem_ext(Size sz);
	void ops_bcc();
	void ops_sub(Size sz);
	void ops_add(Size sz);
	void ops_eor_cmpm(Size sz);
	void ops_rotate(std::string dir, Size sz);
	void ops_pmove(Reg reg, Size sz);
	void ops_fpgen_fpn_fpn();
	void ops_fpgen_ea_fpn();
	void ops_fmovecr();
	void ops_fmove_fpn_ea();
	void ops_fmovem_ea_ctl();
	void ops_fmovem_ctl_ea();
	void ops_fmovem_ea_fpn();
	void ops_fmovem_fpn_ea();

	std::vector<uint16> ir {};

	std::string name {};
	Size size {};
	std::string src {};
	std::string dst {};

#define OP_PROTO(name)	void __CONCAT(op_,name)()
#include "m680x0ops.h"
#undef OP_PROTO
};
