//
// nono
// Copyright (C) 2021 nono project
// Licensed under nono-license.txt
//

//
// UI スレッドへのメッセージ機構 (GUI 側)
//

#include "wxuimessage.h"
#include <array>
#include <mutex>

// イベントタイプの定義
wxDEFINE_EVENT(NONO_EVT_UIMESSAGE, wxCommandEvent);

static std::array<std::vector<wxEvtHandler *>, UIMessage::ID_MAX> table;
static std::mutex mtx;

// UIMessage を wxWidgets のイベントハンドラに接続する
/*static*/ void
WXUIMessage::Connect(UIMessage::ID id, wxEvtHandler *win,
	wxObjectEventFunction function)
{
	std::lock_guard<std::mutex> lock(mtx);
	win->Connect(id, NONO_EVT_UIMESSAGE, function);
	table[id].push_back(win);
}

// イベントハンドラの接続を解除する
/*static*/ void
WXUIMessage::Disconnect(UIMessage::ID id, wxEvtHandler *win,
	wxObjectEventFunction function)
{
	std::lock_guard<std::mutex> lock(mtx);
	win->Disconnect(NONO_EVT_UIMESSAGE, function);
	for (auto it = table[id].begin(); it != table[id].end(); ++it) {
		if (*it == win) {
			table[id].erase(it);
			break;
		}
	}
}

// UIMessage::Post() で呼ばれるコールバック関数。
/*static*/ void
WXUIMessage::Process(uint id, int arg)
{
	if (id >= UIMessage::ID_MAX) {
		PANIC("Invalid UIMessage %u", id);
	}
	std::lock_guard<std::mutex> lock(mtx);
	for (auto win : table[id]) {
		wxCommandEvent ev(NONO_EVT_UIMESSAGE, id);
		ev.SetInt(arg);
		wxPostEvent(win, ev);
	}
}

// すべてのハンドラが解除されているかチェックする (開発用)
/*static*/ void
WXUIMessage::AssertAllDisconnected()
{
	for (int i = 0; i < table.size(); i++) {
		if (table[i].empty() == false) {
			warnx("UIMessage ID=%d not disconnected", i);
		}
	}
}
