//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// Human68k .r .x .z console emulator
//

#pragma once

#include "device.h"

class MainbusDevice;
class MainRAMDevice;
class MPU680x0Device;

class Human68kDevice : public Device
{
	using inherited = Device;

	static const uint32 default_load_addr = 0x20000;

	static const int FilesCount = 256;

 public:
	// .X ファイルのヘッダ構造
	struct XFileHeader {
		uint8  magic[2];		// +00 識別子 'HU'
		uint8  unused1;
		uint8  loadmode;		// +03 ロードモード
		uint32 base_addr;		// +04 ベースアドレス
		uint32 exec_addr;		// +08 実行開始アドレス
		uint32 text_size;		// +0c テキストサイズ
		uint32 data_size;		// +10 データサイズ
		uint32 bss_size;		// +14 BSS サイズ
		uint32 reloc_size;		// +18 再配置テーブルサイズ
		uint32 symbol_size;		// +1c シンボルサイズ
		uint32 scdline_size;	// +20 SCD 行番号テーブルサイズ
		uint32 scdsym_size;		// +24 SCD シンボルテーブルサイズ
		uint32 scdstr_size;		// +28 SCD 文字列テーブルサイズ
		uint32 unused2[4];
		uint32 module_offset;	// +3c モジュールリストの位置
	} __packed;

	// .Z ファイルのヘッダ構造
	struct ZFileHeader {
		uint16 magic1;			// +00 第一識別子 $601a
		uint32 text_size;		// +02 テキストサイズ
		uint32 data_size;		// +06 データサイズ
		uint32 bss_size;		// +0a BSS サイズ
		uint8  unused[8];
		uint32 base_addr;		// +16 実行開始アドレス
		uint16 magic2;			// +1a 第二識別子 $ffff
	} __packed;

	// Human68k File
	struct File {
		int fd = -1;			// unix fd
		std::string filename {};
	};

	Human68kDevice();
	~Human68kDevice() override;

	bool Init() override;

	bool FLineOp();

	// MSXDOSDevice からも呼ぶ
	static void RequestExit(int code);

 private:
	// Human Emulation API
	int32 OpenFile(uint32 fileaddr, uint16 atr, int mode);
	int32 CloseFile(int32 fileno);
	int32 WriteFile(int32 fileno, uint32 dataaddr, uint32 size);
	void FputsFile(int32 fileno, uint32 dataaddr);
	int GetFileMode(int32 fileno);

	void IOCS();
	void DOS_PRINT(uint32 dataptr);

	bool LoadR(const uint8 *memfile, uint size);
	bool LoadX(const uint8 *memfile, uint size);
	bool LoadZ(const uint8 *memfile, uint size);
	bool LoadELF(const uint8 *memfile, uint size);
	bool LoadMem(uint32 addr, const uint8 *src, uint size);

	bool isext(const char *file, const char *ext);

	const char *human68k_file {};
	std::string human68k_arg {};

	uint32 boot_addr {};		// ローダプログラムアドレス
	uint32 load_addr {};		// ロードアドレス
	uint32 load_size {};		// ロードサイズ

	uint32 base_addr {};		// ベースアドレス(ファイルの値)
								// ベースアドレスはload_addr決定で使用される
	uint32 exec_addr {};		// 実行開始アドレス

	uint32 text_size {};		// テキストセクションサイズ
	uint32 data_size {};		// データセクションサイズ
	uint32 bss_size {};			// BSS セクションサイズ(記録用)

	File Files[FilesCount] {};

	MainbusDevice *mainbus {};
	MainRAMDevice *mainram {};
	MPU680x0Device *mpu680x0 {};
};
