//
// nono
// Copyright (C) 2023 nono project
// Licensed under nono-license.txt
//

// キャッシュ

#pragma once

#include "m680x0.h"
#include <array>
#include <mutex>
#include "fixedqueue.h"

// キャッシュライン
struct m68030CacheLine
{
	// タグは命令キャッシュ/データキャッシュで共通だが、
	// 命令キャッシュなら bit2 が S (FC2)、
	// データキャッシュなら下位3ビットが FC (2,1,0 の順)。
	//    3                   2                   1
	//  1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0 9 8 7 6 5 4 3 2 1 0
	// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
	// |S|1|0|S|               Tag Address (24bit)             |       | (Inst)
	// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
	//
	// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
	// |FC2-0|S|               Tag Address (24bit)             |       | (Data)
	// +-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+-+
	uint32 tag {};
	union {
		uint32 valid_all {};
		uint8 valid[4];
	};
	std::array<uint32, 4> data {};

	bool IsSuper() const	{ return (tag & 0x80000000); }
	uint32 GetFC() const	{ return (tag >> 29); }
	uint32 TagAddr() const	{ return (tag << 4); }
};

// キャッシュの統計情報
struct m68030CacheStat
{
	float hit;	// ヒット率
};

// キャッシュ (命令/データ共通)
class m68030Cache
{
 public:
	// キャッシュイネーブル。
	bool enable {};

	// 凍結なら true。
	bool freeze {};

	// バースト転送可能かつ許可なら true。
	bool burst_enable {};

	// データキャッシュのライトアロケート。
	bool wa {};

	// キャッシュ。
	std::array<m68030CacheLine, 16> line {};

	// 統計情報。
	uint32 stat_fasthit {};		// ファストパスでのヒット回数
	uint32 stat_hit {};			// 通常パスでのヒット回数
	uint32 stat_miss {};		// 通常パスでのミス回数

	// 移動平均用バッファ。
	FixedQueue<m68030CacheStat, 3> hist {};
	std::mutex hist_mtx {};

 public:
	void CalcStat();
};
