//
// nono
// Copyright (C) 2020 nono project
// Licensed under nono-license.txt
//

//
// 基本ヘッダ
//

#pragma once

#include "config-nono.h"
#include <cassert>
#include <cerrno>
#include <cstdarg>
#include <cstdio>
#include <cstdint>
#include <cstdlib>
#include <cstring>
#include <memory>
#include <string>

#include <err.h>
#include <inttypes.h>
#include <unistd.h>
#include <sys/cdefs.h>
#include <sys/types.h>

using uint8		= uint8_t;
using uint16	= uint16_t;
using uint32	= uint32_t;
using uint64	= uint64_t;
using int8		= int8_t;
using int16		= int16_t;
using int32		= int32_t;
using int64		= int64_t;

#if defined(HAVE_BSD_BSD_H)
#include <bsd/bsd.h>
#endif

#include "missing_endian.h"

// For cppcheck...
#if !defined(__CONCAT)
#define __CONCAT(a, b)	a ## b
#endif

#if !defined(__printflike)
#if defined(HAVE___ATTRIBUTE_FORMAT)
# define __printflike(a,b)	__attribute__((__format__(__printf__, (a), (b))))
#else
# define __printflike(a,b)
#endif
#endif

#if !defined(__unreachable)
#if defined(HAVE___BUILTIN_UNREACHABLE)
# define __unreachable()	__builtin_unreachable()
#else
# define __unreachable()	((void)0)
#endif
#endif

// gcc でコンパイラに最適化の条件を与える。
#if defined(HAVE___BUILTIN_ASSUME)
#define __assume(cond)	__builtin_assume(cond)
#else
#define __assume(cond)	if (!(cond)) __unreachable()
#endif

#if !defined(__packed)
#if defined(HAVE___ATTRIBUTE_PACKED)
# define __packed	__attribute__((__packed__))
#else
# error not tested.
#endif
#endif

// FreeBSD(12.1) の __predict_true/false は定義がいまいちで使えない。
// __builtin_expect() があることが分かればどの環境でも自前で定義できるので
// 既存定義は取り消す。
#if defined(HAVE___BUILTIN_EXPECT)
# undef  __predict_true
# undef  __predict_false
# define __predict_true(exp)	__builtin_expect((exp) != 0, 1)
# define __predict_false(exp)	__builtin_expect((exp) != 0, 0)
#elif !defined(__predict_true)
# define __predict_true(exp)	(exp)
# define __predict_false(exp)	(exp)
#endif

#if !defined(__unused)
#if defined(HAVE___ATTRIBUTE_UNUSED)
# define __unused __attribute__((__unused__))
#else
# define __unused
#endif
#endif

#if defined(__clang__)
# define FALLTHROUGH	[[clang::fallthrough]]
#elif defined(HAVE___ATTRIBUTE_FALLTHROUGH)
# define FALLTHROUGH	__attribute__((__fallthrough__))
#else
# define FALLTHROUGH	((void)0)
#endif

#if defined(HAVE___ATTRIBUTE_NO_SANITIZE)
# define NO_SANITIZE(x)	__attribute__((no_sanitize(x)))
#else
# define NO_SANITIZE(x)
#endif

// 64ビット数と2つの32ビット数を扱う共用体。
// 構造はまったく同じだが名前だけ場所に合わせてつけたい。
#if BYTE_ORDER == LITTLE_ENDIAN
#define DEF_UNION64(NAME, Q, H, L)	\
	union NAME {	\
		uint64 Q;	\
		struct { uint32 L, H; };	\
	}
#else
#define DEF_UNION64(NAME, Q, H, L)	\
	union NAME {	\
		uint64 Q;	\
		struct { uint32 H, L; };	\
	}
#endif

DEF_UNION64(union64, q, h, l);

#ifndef countof
#define countof(x)	(sizeof(x) / sizeof(x[0]))
#endif

#ifndef howmany
#define	howmany(x, y)	(((x)+((y)-1))/(y))
#endif

#ifndef roundup
#define roundup(x, y)	((((x)+((y)-1))/(y))*(y))
#endif

#ifndef rounddown
#define rounddown(x, y)	(((x)/(y))*(y))
#endif

// メッセージ付きの assert。
// 本当は後ろに足したいのだがそれは無理なので、せめてそれっぽく1行にしてみる。
#if !defined(NDEBUG)
#define assertmsg(expr, ...)	do {	\
	if (__predict_false(!(expr))) {	\
		fprintf(stderr, __VA_ARGS__);	\
		fprintf(stderr, ": "); \
		assert(expr);	\
	}	\
} while (0)
#else
#define assertmsg(expr, ...)
#endif

// __func__ のようだけどクラス名と関数名だけを表示する。
// __PRETTY_FUNCTION__ が近いけど、あれは戻り値やら引数全部表示するので。
#define __method__ \
	(get_classfunc_name(__PRETTY_FUNCTION__, __FUNCTION__).c_str())
extern std::string get_classfunc_name(const char *pretty, const char *func);

// iconv() の第2引数の型は OS によって違う…
#if defined(HAVE_ICONV_CONST)
#define ICONV(cd, s, slen, d, dlen)	iconv((cd), (s), (slen), (d), (dlen))
#else
#define ICONV(cd, s, slen, d, dlen)	\
	iconv((cd), const_cast<char **>(s), (slen), (d), (dlen))
#endif
