#include <iostream>
#include <filesystem>

#include <podofo/private/PdfDeclarationsPrivate.h>
#include <podofo/podofo.h>
#include <podofo/private/PdfFilterFactory.h>

using namespace std;
using namespace PoDoFo;

namespace
{
    enum class AglType : uint8_t
    {
        AdobeGlyphList = 1,
        AdobeGlyphListNewFonts,
        ZapfDingbatsGlyphList,
    };

    struct AglMapping
    {
        AglType Type;
        unsigned char CodePointCount;
        unsigned short Code;            ///< The unicode code point of the charater, or the index in the s_ligatures lists
    };

    struct AglLigatureInfo
    {
        AglLigatureInfo(unsigned short nameIndex, initializer_list<codepoint>&& codePoints)
            : NameIndex(nameIndex), CodePoints(std::move(codePoints)) {
        }

        unsigned short NameIndex;
        CodePointSpan CodePoints;
    };
}

static void loadMaps();
static void writeString(FileStreamDevice& stream, const string_view& view);

static unique_ptr<FileStreamDevice> s_stream;
static vector<pair<PdfName, AglMapping>> s_aglList;
static unordered_map<string_view, unsigned short> s_aglMap;
static vector<AglLigatureInfo> s_ligatures;

int main()
{
    loadMaps();

    string serialized;
    StringStreamDevice stream(serialized);

    utls::WriteUInt16BE(stream, (uint16_t)s_aglList.size());
    for (auto& pair : s_aglList)
    {
        utls::WriteUInt16BE(stream, (uint16_t)s_aglMap[pair.first.GetString()]);
        stream.Write((char)pair.second.Type);
        stream.Write((char)pair.second.CodePointCount);
        utls::WriteUInt16BE(stream, (uint16_t)pair.second.Code);
    }

    utls::WriteUInt16BE(stream, (uint16_t)s_ligatures.size());
    for (auto& info : s_ligatures)
    {
        utls::WriteUInt16BE(stream, (uint16_t)info.NameIndex);
        unsigned char cpSize = (unsigned char)info.CodePoints.GetSize();
        stream.Write((char)cpSize);
        auto view = info.CodePoints.view();
        for (unsigned char i = 0; i < cpSize; i++)
            utls::WriteUInt16BE(stream, (uint16_t)view[i]);
    }

    auto filter = PdfFilterFactory::Create(PdfFilterType::FlateDecode);
    charbuff compressed;
    filter->EncodeTo(compressed, serialized);

    s_stream.reset(new FileStreamDevice(R"(D:\aglmap.cpp)", FileMode::Create));
    s_stream->Write("static string_view s_aglNames[] = {\n"sv);
    for (auto& pair : s_aglList)
    {
        s_stream->Write("    \"");
        s_stream->Write(pair.first);
        s_stream->Write("\"sv,\n");
    }
    s_stream->Write("}\n");

    writeString(*s_stream, compressed);
    return 0;
}

static string_view s_aglNames[] = {
    "test"sv,
    "ddd"sv,
};

void writeString(FileStreamDevice& stream, const string_view& view)
{
    const int max_line_length = 16;
    std::string line;
    line.push_back('"');

    for (size_t i = 0; i < view.size(); i++)
    {
        // Convert character to octal and append to the line
        line += fmt::format("\\{:03o}", static_cast<unsigned char>(view[i]));

        // If the line reaches the maximum length or it's the last character, write to the output
        if ((i + 1) % max_line_length == 0 || i == view.size() - 1)
        {
            line.append("\"\n");
            stream.Write(line);
            line.clear();
            line.push_back('"');
        }
    }
}

void loadMaps()
{
    // References:
    // https://github.com/adobe-type-tools/agl-aglfn/
    // https://github.com/adobe-type-tools/agl-specification
    s_aglList.emplace_back("A"_n, AglMapping{ (AglType)11, 1, 0x0041 });
    s_aglList.emplace_back("AE"_n, AglMapping{ (AglType)11, 1, 0x00C6 });
    s_aglList.emplace_back("AEacute"_n, AglMapping{ (AglType)3, 1, 0x01FC });
    s_aglList.emplace_back("AEmacron"_n, AglMapping{ (AglType)1, 1, 0x01E2 });
    s_aglList.emplace_back("AEsmall"_n, AglMapping{ (AglType)1, 1, 0xF7E6 });
    s_aglList.emplace_back("Aacute"_n, AglMapping{ (AglType)11, 1, 0x00C1 });
    s_aglList.emplace_back("Aacutesmall"_n, AglMapping{ (AglType)1, 1, 0xF7E1 });
    s_aglList.emplace_back("Abreve"_n, AglMapping{ (AglType)3, 1, 0x0102 });
    s_aglList.emplace_back("Abreveacute"_n, AglMapping{ (AglType)1, 1, 0x1EAE });
    s_aglList.emplace_back("Abrevecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04D0 });
    s_aglList.emplace_back("Abrevedotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EB6 });
    s_aglList.emplace_back("Abrevegrave"_n, AglMapping{ (AglType)1, 1, 0x1EB0 });
    s_aglList.emplace_back("Abrevehookabove"_n, AglMapping{ (AglType)1, 1, 0x1EB2 });
    s_aglList.emplace_back("Abrevetilde"_n, AglMapping{ (AglType)1, 1, 0x1EB4 });
    s_aglList.emplace_back("Acaron"_n, AglMapping{ (AglType)1, 1, 0x01CD });
    s_aglList.emplace_back("Acircle"_n, AglMapping{ (AglType)1, 1, 0x24B6 });
    s_aglList.emplace_back("Acircumflex"_n, AglMapping{ (AglType)11, 1, 0x00C2 });
    s_aglList.emplace_back("Acircumflexacute"_n, AglMapping{ (AglType)1, 1, 0x1EA4 });
    s_aglList.emplace_back("Acircumflexdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EAC });
    s_aglList.emplace_back("Acircumflexgrave"_n, AglMapping{ (AglType)1, 1, 0x1EA6 });
    s_aglList.emplace_back("Acircumflexhookabove"_n, AglMapping{ (AglType)1, 1, 0x1EA8 });
    s_aglList.emplace_back("Acircumflexsmall"_n, AglMapping{ (AglType)1, 1, 0xF7E2 });
    s_aglList.emplace_back("Acircumflextilde"_n, AglMapping{ (AglType)1, 1, 0x1EAA });
    s_aglList.emplace_back("Acute"_n, AglMapping{ (AglType)1, 1, 0xF6C9 });
    s_aglList.emplace_back("Acutesmall"_n, AglMapping{ (AglType)1, 1, 0xF7B4 });
    s_aglList.emplace_back("Acyrillic"_n, AglMapping{ (AglType)1, 1, 0x0410 });
    s_aglList.emplace_back("Adblgrave"_n, AglMapping{ (AglType)1, 1, 0x0200 });
    s_aglList.emplace_back("Adieresis"_n, AglMapping{ (AglType)11, 1, 0x00C4 });
    s_aglList.emplace_back("Adieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04D2 });
    s_aglList.emplace_back("Adieresismacron"_n, AglMapping{ (AglType)1, 1, 0x01DE });
    s_aglList.emplace_back("Adieresissmall"_n, AglMapping{ (AglType)1, 1, 0xF7E4 });
    s_aglList.emplace_back("Adotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EA0 });
    s_aglList.emplace_back("Adotmacron"_n, AglMapping{ (AglType)1, 1, 0x01E0 });
    s_aglList.emplace_back("Agrave"_n, AglMapping{ (AglType)11, 1, 0x00C0 });
    s_aglList.emplace_back("Agravesmall"_n, AglMapping{ (AglType)1, 1, 0xF7E0 });
    s_aglList.emplace_back("Ahookabove"_n, AglMapping{ (AglType)1, 1, 0x1EA2 });
    s_aglList.emplace_back("Aiecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04D4 });
    s_aglList.emplace_back("Ainvertedbreve"_n, AglMapping{ (AglType)1, 1, 0x0202 });
    s_aglList.emplace_back("Alpha"_n, AglMapping{ (AglType)3, 1, 0x0391 });
    s_aglList.emplace_back("Alphatonos"_n, AglMapping{ (AglType)3, 1, 0x0386 });
    s_aglList.emplace_back("Amacron"_n, AglMapping{ (AglType)3, 1, 0x0100 });
    s_aglList.emplace_back("Amonospace"_n, AglMapping{ (AglType)1, 1, 0xFF21 });
    s_aglList.emplace_back("Aogonek"_n, AglMapping{ (AglType)3, 1, 0x0104 });
    s_aglList.emplace_back("Aring"_n, AglMapping{ (AglType)11, 1, 0x00C5 });
    s_aglList.emplace_back("Aringacute"_n, AglMapping{ (AglType)3, 1, 0x01FA });
    s_aglList.emplace_back("Aringbelow"_n, AglMapping{ (AglType)1, 1, 0x1E00 });
    s_aglList.emplace_back("Aringsmall"_n, AglMapping{ (AglType)1, 1, 0xF7E5 });
    s_aglList.emplace_back("Asmall"_n, AglMapping{ (AglType)1, 1, 0xF761 });
    s_aglList.emplace_back("Atilde"_n, AglMapping{ (AglType)11, 1, 0x00C3 });
    s_aglList.emplace_back("Atildesmall"_n, AglMapping{ (AglType)1, 1, 0xF7E3 });
    s_aglList.emplace_back("Aybarmenian"_n, AglMapping{ (AglType)1, 1, 0x0531 });
    s_aglList.emplace_back("B"_n, AglMapping{ (AglType)11, 1, 0x0042 });
    s_aglList.emplace_back("Bcircle"_n, AglMapping{ (AglType)1, 1, 0x24B7 });
    s_aglList.emplace_back("Bdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E02 });
    s_aglList.emplace_back("Bdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E04 });
    s_aglList.emplace_back("Becyrillic"_n, AglMapping{ (AglType)1, 1, 0x0411 });
    s_aglList.emplace_back("Benarmenian"_n, AglMapping{ (AglType)1, 1, 0x0532 });
    s_aglList.emplace_back("Beta"_n, AglMapping{ (AglType)3, 1, 0x0392 });
    s_aglList.emplace_back("Bhook"_n, AglMapping{ (AglType)1, 1, 0x0181 });
    s_aglList.emplace_back("Blinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E06 });
    s_aglList.emplace_back("Bmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF22 });
    s_aglList.emplace_back("Brevesmall"_n, AglMapping{ (AglType)1, 1, 0xF6F4 });
    s_aglList.emplace_back("Bsmall"_n, AglMapping{ (AglType)1, 1, 0xF762 });
    s_aglList.emplace_back("Btopbar"_n, AglMapping{ (AglType)1, 1, 0x0182 });
    s_aglList.emplace_back("C"_n, AglMapping{ (AglType)11, 1, 0x0043 });
    s_aglList.emplace_back("Caarmenian"_n, AglMapping{ (AglType)1, 1, 0x053E });
    s_aglList.emplace_back("Cacute"_n, AglMapping{ (AglType)3, 1, 0x0106 });
    s_aglList.emplace_back("Caron"_n, AglMapping{ (AglType)1, 1, 0xF6CA });
    s_aglList.emplace_back("Caronsmall"_n, AglMapping{ (AglType)1, 1, 0xF6F5 });
    s_aglList.emplace_back("Ccaron"_n, AglMapping{ (AglType)3, 1, 0x010C });
    s_aglList.emplace_back("Ccedilla"_n, AglMapping{ (AglType)11, 1, 0x00C7 });
    s_aglList.emplace_back("Ccedillaacute"_n, AglMapping{ (AglType)1, 1, 0x1E08 });
    s_aglList.emplace_back("Ccedillasmall"_n, AglMapping{ (AglType)1, 1, 0xF7E7 });
    s_aglList.emplace_back("Ccircle"_n, AglMapping{ (AglType)1, 1, 0x24B8 });
    s_aglList.emplace_back("Ccircumflex"_n, AglMapping{ (AglType)3, 1, 0x0108 });
    s_aglList.emplace_back("Cdot"_n, AglMapping{ (AglType)1, 1, 0x010A });
    s_aglList.emplace_back("Cdotaccent"_n, AglMapping{ (AglType)3, 1, 0x010A });
    s_aglList.emplace_back("Cedillasmall"_n, AglMapping{ (AglType)1, 1, 0xF7B8 });
    s_aglList.emplace_back("Chaarmenian"_n, AglMapping{ (AglType)1, 1, 0x0549 });
    s_aglList.emplace_back("Cheabkhasiancyrillic"_n, AglMapping{ (AglType)1, 1, 0x04BC });
    s_aglList.emplace_back("Checyrillic"_n, AglMapping{ (AglType)1, 1, 0x0427 });
    s_aglList.emplace_back("Chedescenderabkhasiancyrillic"_n, AglMapping{ (AglType)1, 1, 0x04BE });
    s_aglList.emplace_back("Chedescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x04B6 });
    s_aglList.emplace_back("Chedieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04F4 });
    s_aglList.emplace_back("Cheharmenian"_n, AglMapping{ (AglType)1, 1, 0x0543 });
    s_aglList.emplace_back("Chekhakassiancyrillic"_n, AglMapping{ (AglType)1, 1, 0x04CB });
    s_aglList.emplace_back("Cheverticalstrokecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04B8 });
    s_aglList.emplace_back("Chi"_n, AglMapping{ (AglType)3, 1, 0x03A7 });
    s_aglList.emplace_back("Chook"_n, AglMapping{ (AglType)1, 1, 0x0187 });
    s_aglList.emplace_back("Circumflexsmall"_n, AglMapping{ (AglType)1, 1, 0xF6F6 });
    s_aglList.emplace_back("Cmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF23 });
    s_aglList.emplace_back("Coarmenian"_n, AglMapping{ (AglType)1, 1, 0x0551 });
    s_aglList.emplace_back("Csmall"_n, AglMapping{ (AglType)1, 1, 0xF763 });
    s_aglList.emplace_back("D"_n, AglMapping{ (AglType)11, 1, 0x0044 });
    s_aglList.emplace_back("DZ"_n, AglMapping{ (AglType)1, 1, 0x01F1 });
    s_aglList.emplace_back("DZcaron"_n, AglMapping{ (AglType)1, 1, 0x01C4 });
    s_aglList.emplace_back("Daarmenian"_n, AglMapping{ (AglType)1, 1, 0x0534 });
    s_aglList.emplace_back("Dafrican"_n, AglMapping{ (AglType)1, 1, 0x0189 });
    s_aglList.emplace_back("Dcaron"_n, AglMapping{ (AglType)3, 1, 0x010E });
    s_aglList.emplace_back("Dcedilla"_n, AglMapping{ (AglType)1, 1, 0x1E10 });
    s_aglList.emplace_back("Dcircle"_n, AglMapping{ (AglType)1, 1, 0x24B9 });
    s_aglList.emplace_back("Dcircumflexbelow"_n, AglMapping{ (AglType)1, 1, 0x1E12 });
    s_aglList.emplace_back("Dcroat"_n, AglMapping{ (AglType)3, 1, 0x0110 });
    s_aglList.emplace_back("Ddotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E0A });
    s_aglList.emplace_back("Ddotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E0C });
    s_aglList.emplace_back("Decyrillic"_n, AglMapping{ (AglType)1, 1, 0x0414 });
    s_aglList.emplace_back("Deicoptic"_n, AglMapping{ (AglType)1, 1, 0x03EE });
    s_aglList.emplace_back("Delta"_n, AglMapping{ (AglType)3, 1, 0x2206 });
    s_aglList.emplace_back("Deltagreek"_n, AglMapping{ (AglType)1, 1, 0x0394 });
    s_aglList.emplace_back("Dhook"_n, AglMapping{ (AglType)1, 1, 0x018A });
    s_aglList.emplace_back("Dieresis"_n, AglMapping{ (AglType)1, 1, 0xF6CB });
    s_aglList.emplace_back("DieresisAcute"_n, AglMapping{ (AglType)1, 1, 0xF6CC });
    s_aglList.emplace_back("DieresisGrave"_n, AglMapping{ (AglType)1, 1, 0xF6CD });
    s_aglList.emplace_back("Dieresissmall"_n, AglMapping{ (AglType)1, 1, 0xF7A8 });
    s_aglList.emplace_back("Digammagreek"_n, AglMapping{ (AglType)1, 1, 0x03DC });
    s_aglList.emplace_back("Djecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0402 });
    s_aglList.emplace_back("Dlinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E0E });
    s_aglList.emplace_back("Dmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF24 });
    s_aglList.emplace_back("Dotaccentsmall"_n, AglMapping{ (AglType)1, 1, 0xF6F7 });
    s_aglList.emplace_back("Dslash"_n, AglMapping{ (AglType)1, 1, 0x0110 });
    s_aglList.emplace_back("Dsmall"_n, AglMapping{ (AglType)1, 1, 0xF764 });
    s_aglList.emplace_back("Dtopbar"_n, AglMapping{ (AglType)1, 1, 0x018B });
    s_aglList.emplace_back("Dz"_n, AglMapping{ (AglType)1, 1, 0x01F2 });
    s_aglList.emplace_back("Dzcaron"_n, AglMapping{ (AglType)1, 1, 0x01C5 });
    s_aglList.emplace_back("Dzeabkhasiancyrillic"_n, AglMapping{ (AglType)1, 1, 0x04E0 });
    s_aglList.emplace_back("Dzecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0405 });
    s_aglList.emplace_back("Dzhecyrillic"_n, AglMapping{ (AglType)1, 1, 0x040F });
    s_aglList.emplace_back("E"_n, AglMapping{ (AglType)11, 1, 0x0045 });
    s_aglList.emplace_back("Eacute"_n, AglMapping{ (AglType)11, 1, 0x00C9 });
    s_aglList.emplace_back("Eacutesmall"_n, AglMapping{ (AglType)1, 1, 0xF7E9 });
    s_aglList.emplace_back("Ebreve"_n, AglMapping{ (AglType)3, 1, 0x0114 });
    s_aglList.emplace_back("Ecaron"_n, AglMapping{ (AglType)3, 1, 0x011A });
    s_aglList.emplace_back("Ecedillabreve"_n, AglMapping{ (AglType)1, 1, 0x1E1C });
    s_aglList.emplace_back("Echarmenian"_n, AglMapping{ (AglType)1, 1, 0x0535 });
    s_aglList.emplace_back("Ecircle"_n, AglMapping{ (AglType)1, 1, 0x24BA });
    s_aglList.emplace_back("Ecircumflex"_n, AglMapping{ (AglType)11, 1, 0x00CA });
    s_aglList.emplace_back("Ecircumflexacute"_n, AglMapping{ (AglType)1, 1, 0x1EBE });
    s_aglList.emplace_back("Ecircumflexbelow"_n, AglMapping{ (AglType)1, 1, 0x1E18 });
    s_aglList.emplace_back("Ecircumflexdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EC6 });
    s_aglList.emplace_back("Ecircumflexgrave"_n, AglMapping{ (AglType)1, 1, 0x1EC0 });
    s_aglList.emplace_back("Ecircumflexhookabove"_n, AglMapping{ (AglType)1, 1, 0x1EC2 });
    s_aglList.emplace_back("Ecircumflexsmall"_n, AglMapping{ (AglType)1, 1, 0xF7EA });
    s_aglList.emplace_back("Ecircumflextilde"_n, AglMapping{ (AglType)1, 1, 0x1EC4 });
    s_aglList.emplace_back("Ecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0404 });
    s_aglList.emplace_back("Edblgrave"_n, AglMapping{ (AglType)1, 1, 0x0204 });
    s_aglList.emplace_back("Edieresis"_n, AglMapping{ (AglType)11, 1, 0x00CB });
    s_aglList.emplace_back("Edieresissmall"_n, AglMapping{ (AglType)1, 1, 0xF7EB });
    s_aglList.emplace_back("Edot"_n, AglMapping{ (AglType)1, 1, 0x0116 });
    s_aglList.emplace_back("Edotaccent"_n, AglMapping{ (AglType)3, 1, 0x0116 });
    s_aglList.emplace_back("Edotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EB8 });
    s_aglList.emplace_back("Efcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0424 });
    s_aglList.emplace_back("Egrave"_n, AglMapping{ (AglType)11, 1, 0x00C8 });
    s_aglList.emplace_back("Egravesmall"_n, AglMapping{ (AglType)1, 1, 0xF7E8 });
    s_aglList.emplace_back("Eharmenian"_n, AglMapping{ (AglType)1, 1, 0x0537 });
    s_aglList.emplace_back("Ehookabove"_n, AglMapping{ (AglType)1, 1, 0x1EBA });
    s_aglList.emplace_back("Eightroman"_n, AglMapping{ (AglType)1, 1, 0x2167 });
    s_aglList.emplace_back("Einvertedbreve"_n, AglMapping{ (AglType)1, 1, 0x0206 });
    s_aglList.emplace_back("Eiotifiedcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0464 });
    s_aglList.emplace_back("Elcyrillic"_n, AglMapping{ (AglType)1, 1, 0x041B });
    s_aglList.emplace_back("Elevenroman"_n, AglMapping{ (AglType)1, 1, 0x216A });
    s_aglList.emplace_back("Emacron"_n, AglMapping{ (AglType)3, 1, 0x0112 });
    s_aglList.emplace_back("Emacronacute"_n, AglMapping{ (AglType)1, 1, 0x1E16 });
    s_aglList.emplace_back("Emacrongrave"_n, AglMapping{ (AglType)1, 1, 0x1E14 });
    s_aglList.emplace_back("Emcyrillic"_n, AglMapping{ (AglType)1, 1, 0x041C });
    s_aglList.emplace_back("Emonospace"_n, AglMapping{ (AglType)1, 1, 0xFF25 });
    s_aglList.emplace_back("Encyrillic"_n, AglMapping{ (AglType)1, 1, 0x041D });
    s_aglList.emplace_back("Endescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x04A2 });
    s_aglList.emplace_back("Eng"_n, AglMapping{ (AglType)3, 1, 0x014A });
    s_aglList.emplace_back("Enghecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04A4 });
    s_aglList.emplace_back("Enhookcyrillic"_n, AglMapping{ (AglType)1, 1, 0x04C7 });
    s_aglList.emplace_back("Eogonek"_n, AglMapping{ (AglType)3, 1, 0x0118 });
    s_aglList.emplace_back("Eopen"_n, AglMapping{ (AglType)1, 1, 0x0190 });
    s_aglList.emplace_back("Epsilon"_n, AglMapping{ (AglType)3, 1, 0x0395 });
    s_aglList.emplace_back("Epsilontonos"_n, AglMapping{ (AglType)3, 1, 0x0388 });
    s_aglList.emplace_back("Ercyrillic"_n, AglMapping{ (AglType)1, 1, 0x0420 });
    s_aglList.emplace_back("Ereversed"_n, AglMapping{ (AglType)1, 1, 0x018E });
    s_aglList.emplace_back("Ereversedcyrillic"_n, AglMapping{ (AglType)1, 1, 0x042D });
    s_aglList.emplace_back("Escyrillic"_n, AglMapping{ (AglType)1, 1, 0x0421 });
    s_aglList.emplace_back("Esdescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x04AA });
    s_aglList.emplace_back("Esh"_n, AglMapping{ (AglType)1, 1, 0x01A9 });
    s_aglList.emplace_back("Esmall"_n, AglMapping{ (AglType)1, 1, 0xF765 });
    s_aglList.emplace_back("Eta"_n, AglMapping{ (AglType)3, 1, 0x0397 });
    s_aglList.emplace_back("Etarmenian"_n, AglMapping{ (AglType)1, 1, 0x0538 });
    s_aglList.emplace_back("Etatonos"_n, AglMapping{ (AglType)3, 1, 0x0389 });
    s_aglList.emplace_back("Eth"_n, AglMapping{ (AglType)11, 1, 0x00D0 });
    s_aglList.emplace_back("Ethsmall"_n, AglMapping{ (AglType)1, 1, 0xF7F0 });
    s_aglList.emplace_back("Etilde"_n, AglMapping{ (AglType)1, 1, 0x1EBC });
    s_aglList.emplace_back("Etildebelow"_n, AglMapping{ (AglType)1, 1, 0x1E1A });
    s_aglList.emplace_back("Euro"_n, AglMapping{ (AglType)11, 1, 0x20AC });
    s_aglList.emplace_back("Ezh"_n, AglMapping{ (AglType)1, 1, 0x01B7 });
    s_aglList.emplace_back("Ezhcaron"_n, AglMapping{ (AglType)1, 1, 0x01EE });
    s_aglList.emplace_back("Ezhreversed"_n, AglMapping{ (AglType)1, 1, 0x01B8 });
    s_aglList.emplace_back("F"_n, AglMapping{ (AglType)11, 1, 0x0046 });
    s_aglList.emplace_back("Fcircle"_n, AglMapping{ (AglType)1, 1, 0x24BB });
    s_aglList.emplace_back("Fdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E1E });
    s_aglList.emplace_back("Feharmenian"_n, AglMapping{ (AglType)1, 1, 0x0556 });
    s_aglList.emplace_back("Feicoptic"_n, AglMapping{ (AglType)1, 1, 0x03E4 });
    s_aglList.emplace_back("Fhook"_n, AglMapping{ (AglType)1, 1, 0x0191 });
    s_aglList.emplace_back("Fitacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0472 });
    s_aglList.emplace_back("Fiveroman"_n, AglMapping{ (AglType)1, 1, 0x2164 });
    s_aglList.emplace_back("Fmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF26 });
    s_aglList.emplace_back("Fourroman"_n, AglMapping{ (AglType)1, 1, 0x2163 });
    s_aglList.emplace_back("Fsmall"_n, AglMapping{ (AglType)1, 1, 0xF766 });
    s_aglList.emplace_back("G"_n, AglMapping{ (AglType)11, 1, 0x0047 });
    s_aglList.emplace_back("GBsquare"_n, AglMapping{ (AglType)1, 1, 0x3387 });
    s_aglList.emplace_back("Gacute"_n, AglMapping{ (AglType)1, 1, 0x01F4 });
    s_aglList.emplace_back("Gamma"_n, AglMapping{ (AglType)3, 1, 0x0393 });
    s_aglList.emplace_back("Gammaafrican"_n, AglMapping{ (AglType)1, 1, 0x0194 });
    s_aglList.emplace_back("Gangiacoptic"_n, AglMapping{ (AglType)1, 1, 0x03EA });
    s_aglList.emplace_back("Gbreve"_n, AglMapping{ (AglType)3, 1, 0x011E });
    s_aglList.emplace_back("Gcaron"_n, AglMapping{ (AglType)3, 1, 0x01E6 });
    s_aglList.emplace_back("Gcedilla"_n, AglMapping{ (AglType)1, 1, 0x0122 });
    s_aglList.emplace_back("Gcircle"_n, AglMapping{ (AglType)1, 1, 0x24BC });
    s_aglList.emplace_back("Gcircumflex"_n, AglMapping{ (AglType)3, 1, 0x011C });
    s_aglList.emplace_back("Gcommaaccent"_n, AglMapping{ (AglType)1, 1, 0x0122 });
    s_aglList.emplace_back("Gdot"_n, AglMapping{ (AglType)1, 1, 0x0120 });
    s_aglList.emplace_back("Gdotaccent"_n, AglMapping{ (AglType)3, 1, 0x0120 });
    s_aglList.emplace_back("Gecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0413 });
    s_aglList.emplace_back("Ghadarmenian"_n, AglMapping{ (AglType)1, 1, 0x0542 });
    s_aglList.emplace_back("Ghemiddlehookcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0494 });
    s_aglList.emplace_back("Ghestrokecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0492 });
    s_aglList.emplace_back("Gheupturncyrillic"_n, AglMapping{ (AglType)1, 1, 0x0490 });
    s_aglList.emplace_back("Ghook"_n, AglMapping{ (AglType)1, 1, 0x0193 });
    s_aglList.emplace_back("Gimarmenian"_n, AglMapping{ (AglType)1, 1, 0x0533 });
    s_aglList.emplace_back("Gjecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0403 });
    s_aglList.emplace_back("Gmacron"_n, AglMapping{ (AglType)1, 1, 0x1E20 });
    s_aglList.emplace_back("Gmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF27 });
    s_aglList.emplace_back("Grave"_n, AglMapping{ (AglType)1, 1, 0xF6CE });
    s_aglList.emplace_back("Gravesmall"_n, AglMapping{ (AglType)1, 1, 0xF760 });
    s_aglList.emplace_back("Gsmall"_n, AglMapping{ (AglType)1, 1, 0xF767 });
    s_aglList.emplace_back("Gsmallhook"_n, AglMapping{ (AglType)1, 1, 0x029B });
    s_aglList.emplace_back("Gstroke"_n, AglMapping{ (AglType)1, 1, 0x01E4 });
    s_aglList.emplace_back("H"_n, AglMapping{ (AglType)11, 1, 0x0048 });
    s_aglList.emplace_back("H18533"_n, AglMapping{ (AglType)3, 1, 0x25CF });
    s_aglList.emplace_back("H18543"_n, AglMapping{ (AglType)3, 1, 0x25AA });
    s_aglList.emplace_back("H18551"_n, AglMapping{ (AglType)3, 1, 0x25AB });
    s_aglList.emplace_back("H22073"_n, AglMapping{ (AglType)3, 1, 0x25A1 });
    s_aglList.emplace_back("HPsquare"_n, AglMapping{ (AglType)1, 1, 0x33CB });
    s_aglList.emplace_back("Haabkhasiancyrillic"_n, AglMapping{ (AglType)1, 1, 0x04A8 });
    s_aglList.emplace_back("Hadescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x04B2 });
    s_aglList.emplace_back("Hardsigncyrillic"_n, AglMapping{ (AglType)1, 1, 0x042A });
    s_aglList.emplace_back("Hbar"_n, AglMapping{ (AglType)3, 1, 0x0126 });
    s_aglList.emplace_back("Hbrevebelow"_n, AglMapping{ (AglType)1, 1, 0x1E2A });
    s_aglList.emplace_back("Hcedilla"_n, AglMapping{ (AglType)1, 1, 0x1E28 });
    s_aglList.emplace_back("Hcircle"_n, AglMapping{ (AglType)1, 1, 0x24BD });
    s_aglList.emplace_back("Hcircumflex"_n, AglMapping{ (AglType)3, 1, 0x0124 });
    s_aglList.emplace_back("Hdieresis"_n, AglMapping{ (AglType)1, 1, 0x1E26 });
    s_aglList.emplace_back("Hdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E22 });
    s_aglList.emplace_back("Hdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E24 });
    s_aglList.emplace_back("Hmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF28 });
    s_aglList.emplace_back("Hoarmenian"_n, AglMapping{ (AglType)1, 1, 0x0540 });
    s_aglList.emplace_back("Horicoptic"_n, AglMapping{ (AglType)1, 1, 0x03E8 });
    s_aglList.emplace_back("Hsmall"_n, AglMapping{ (AglType)1, 1, 0xF768 });
    s_aglList.emplace_back("Hungarumlaut"_n, AglMapping{ (AglType)1, 1, 0xF6CF });
    s_aglList.emplace_back("Hungarumlautsmall"_n, AglMapping{ (AglType)1, 1, 0xF6F8 });
    s_aglList.emplace_back("Hzsquare"_n, AglMapping{ (AglType)1, 1, 0x3390 });
    s_aglList.emplace_back("I"_n, AglMapping{ (AglType)11, 1, 0x0049 });
    s_aglList.emplace_back("IAcyrillic"_n, AglMapping{ (AglType)1, 1, 0x042F });
    s_aglList.emplace_back("IJ"_n, AglMapping{ (AglType)3, 1, 0x0132 });
    s_aglList.emplace_back("IUcyrillic"_n, AglMapping{ (AglType)1, 1, 0x042E });
    s_aglList.emplace_back("Iacute"_n, AglMapping{ (AglType)11, 1, 0x00CD });
    s_aglList.emplace_back("Iacutesmall"_n, AglMapping{ (AglType)1, 1, 0xF7ED });
    s_aglList.emplace_back("Ibreve"_n, AglMapping{ (AglType)3, 1, 0x012C });
    s_aglList.emplace_back("Icaron"_n, AglMapping{ (AglType)1, 1, 0x01CF });
    s_aglList.emplace_back("Icircle"_n, AglMapping{ (AglType)1, 1, 0x24BE });
    s_aglList.emplace_back("Icircumflex"_n, AglMapping{ (AglType)11, 1, 0x00CE });
    s_aglList.emplace_back("Icircumflexsmall"_n, AglMapping{ (AglType)1, 1, 0xF7EE });
    s_aglList.emplace_back("Icyrillic"_n, AglMapping{ (AglType)1, 1, 0x0406 });
    s_aglList.emplace_back("Idblgrave"_n, AglMapping{ (AglType)1, 1, 0x0208 });
    s_aglList.emplace_back("Idieresis"_n, AglMapping{ (AglType)11, 1, 0x00CF });
    s_aglList.emplace_back("Idieresisacute"_n, AglMapping{ (AglType)1, 1, 0x1E2E });
    s_aglList.emplace_back("Idieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04E4 });
    s_aglList.emplace_back("Idieresissmall"_n, AglMapping{ (AglType)1, 1, 0xF7EF });
    s_aglList.emplace_back("Idot"_n, AglMapping{ (AglType)1, 1, 0x0130 });
    s_aglList.emplace_back("Idotaccent"_n, AglMapping{ (AglType)3, 1, 0x0130 });
    s_aglList.emplace_back("Idotbelow"_n, AglMapping{ (AglType)1, 1, 0x1ECA });
    s_aglList.emplace_back("Iebrevecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04D6 });
    s_aglList.emplace_back("Iecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0415 });
    s_aglList.emplace_back("Ifraktur"_n, AglMapping{ (AglType)3, 1, 0x2111 });
    s_aglList.emplace_back("Igrave"_n, AglMapping{ (AglType)11, 1, 0x00CC });
    s_aglList.emplace_back("Igravesmall"_n, AglMapping{ (AglType)1, 1, 0xF7EC });
    s_aglList.emplace_back("Ihookabove"_n, AglMapping{ (AglType)1, 1, 0x1EC8 });
    s_aglList.emplace_back("Iicyrillic"_n, AglMapping{ (AglType)1, 1, 0x0418 });
    s_aglList.emplace_back("Iinvertedbreve"_n, AglMapping{ (AglType)1, 1, 0x020A });
    s_aglList.emplace_back("Iishortcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0419 });
    s_aglList.emplace_back("Imacron"_n, AglMapping{ (AglType)3, 1, 0x012A });
    s_aglList.emplace_back("Imacroncyrillic"_n, AglMapping{ (AglType)1, 1, 0x04E2 });
    s_aglList.emplace_back("Imonospace"_n, AglMapping{ (AglType)1, 1, 0xFF29 });
    s_aglList.emplace_back("Iniarmenian"_n, AglMapping{ (AglType)1, 1, 0x053B });
    s_aglList.emplace_back("Iocyrillic"_n, AglMapping{ (AglType)1, 1, 0x0401 });
    s_aglList.emplace_back("Iogonek"_n, AglMapping{ (AglType)3, 1, 0x012E });
    s_aglList.emplace_back("Iota"_n, AglMapping{ (AglType)3, 1, 0x0399 });
    s_aglList.emplace_back("Iotaafrican"_n, AglMapping{ (AglType)1, 1, 0x0196 });
    s_aglList.emplace_back("Iotadieresis"_n, AglMapping{ (AglType)3, 1, 0x03AA });
    s_aglList.emplace_back("Iotatonos"_n, AglMapping{ (AglType)3, 1, 0x038A });
    s_aglList.emplace_back("Ismall"_n, AglMapping{ (AglType)1, 1, 0xF769 });
    s_aglList.emplace_back("Istroke"_n, AglMapping{ (AglType)1, 1, 0x0197 });
    s_aglList.emplace_back("Itilde"_n, AglMapping{ (AglType)3, 1, 0x0128 });
    s_aglList.emplace_back("Itildebelow"_n, AglMapping{ (AglType)1, 1, 0x1E2C });
    s_aglList.emplace_back("Izhitsacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0474 });
    s_aglList.emplace_back("Izhitsadblgravecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0476 });
    s_aglList.emplace_back("J"_n, AglMapping{ (AglType)11, 1, 0x004A });
    s_aglList.emplace_back("Jaarmenian"_n, AglMapping{ (AglType)1, 1, 0x0541 });
    s_aglList.emplace_back("Jcircle"_n, AglMapping{ (AglType)1, 1, 0x24BF });
    s_aglList.emplace_back("Jcircumflex"_n, AglMapping{ (AglType)3, 1, 0x0134 });
    s_aglList.emplace_back("Jecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0408 });
    s_aglList.emplace_back("Jheharmenian"_n, AglMapping{ (AglType)1, 1, 0x054B });
    s_aglList.emplace_back("Jmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF2A });
    s_aglList.emplace_back("Jsmall"_n, AglMapping{ (AglType)1, 1, 0xF76A });
    s_aglList.emplace_back("K"_n, AglMapping{ (AglType)11, 1, 0x004B });
    s_aglList.emplace_back("KBsquare"_n, AglMapping{ (AglType)1, 1, 0x3385 });
    s_aglList.emplace_back("KKsquare"_n, AglMapping{ (AglType)1, 1, 0x33CD });
    s_aglList.emplace_back("Kabashkircyrillic"_n, AglMapping{ (AglType)1, 1, 0x04A0 });
    s_aglList.emplace_back("Kacute"_n, AglMapping{ (AglType)1, 1, 0x1E30 });
    s_aglList.emplace_back("Kacyrillic"_n, AglMapping{ (AglType)1, 1, 0x041A });
    s_aglList.emplace_back("Kadescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x049A });
    s_aglList.emplace_back("Kahookcyrillic"_n, AglMapping{ (AglType)1, 1, 0x04C3 });
    s_aglList.emplace_back("Kappa"_n, AglMapping{ (AglType)3, 1, 0x039A });
    s_aglList.emplace_back("Kastrokecyrillic"_n, AglMapping{ (AglType)1, 1, 0x049E });
    s_aglList.emplace_back("Kaverticalstrokecyrillic"_n, AglMapping{ (AglType)1, 1, 0x049C });
    s_aglList.emplace_back("Kcaron"_n, AglMapping{ (AglType)1, 1, 0x01E8 });
    s_aglList.emplace_back("Kcedilla"_n, AglMapping{ (AglType)1, 1, 0x0136 });
    s_aglList.emplace_back("Kcircle"_n, AglMapping{ (AglType)1, 1, 0x24C0 });
    s_aglList.emplace_back("Kcommaaccent"_n, AglMapping{ (AglType)1, 1, 0x0136 });
    s_aglList.emplace_back("Kdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E32 });
    s_aglList.emplace_back("Keharmenian"_n, AglMapping{ (AglType)1, 1, 0x0554 });
    s_aglList.emplace_back("Kenarmenian"_n, AglMapping{ (AglType)1, 1, 0x053F });
    s_aglList.emplace_back("Khacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0425 });
    s_aglList.emplace_back("Kheicoptic"_n, AglMapping{ (AglType)1, 1, 0x03E6 });
    s_aglList.emplace_back("Khook"_n, AglMapping{ (AglType)1, 1, 0x0198 });
    s_aglList.emplace_back("Kjecyrillic"_n, AglMapping{ (AglType)1, 1, 0x040C });
    s_aglList.emplace_back("Klinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E34 });
    s_aglList.emplace_back("Kmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF2B });
    s_aglList.emplace_back("Koppacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0480 });
    s_aglList.emplace_back("Koppagreek"_n, AglMapping{ (AglType)1, 1, 0x03DE });
    s_aglList.emplace_back("Ksicyrillic"_n, AglMapping{ (AglType)1, 1, 0x046E });
    s_aglList.emplace_back("Ksmall"_n, AglMapping{ (AglType)1, 1, 0xF76B });
    s_aglList.emplace_back("L"_n, AglMapping{ (AglType)11, 1, 0x004C });
    s_aglList.emplace_back("LJ"_n, AglMapping{ (AglType)1, 1, 0x01C7 });
    s_aglList.emplace_back("LL"_n, AglMapping{ (AglType)1, 1, 0xF6BF });
    s_aglList.emplace_back("Lacute"_n, AglMapping{ (AglType)3, 1, 0x0139 });
    s_aglList.emplace_back("Lambda"_n, AglMapping{ (AglType)3, 1, 0x039B });
    s_aglList.emplace_back("Lcaron"_n, AglMapping{ (AglType)3, 1, 0x013D });
    s_aglList.emplace_back("Lcedilla"_n, AglMapping{ (AglType)1, 1, 0x013B });
    s_aglList.emplace_back("Lcircle"_n, AglMapping{ (AglType)1, 1, 0x24C1 });
    s_aglList.emplace_back("Lcircumflexbelow"_n, AglMapping{ (AglType)1, 1, 0x1E3C });
    s_aglList.emplace_back("Lcommaaccent"_n, AglMapping{ (AglType)1, 1, 0x013B });
    s_aglList.emplace_back("Ldot"_n, AglMapping{ (AglType)3, 1, 0x013F });
    s_aglList.emplace_back("Ldotaccent"_n, AglMapping{ (AglType)1, 1, 0x013F });
    s_aglList.emplace_back("Ldotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E36 });
    s_aglList.emplace_back("Ldotbelowmacron"_n, AglMapping{ (AglType)1, 1, 0x1E38 });
    s_aglList.emplace_back("Liwnarmenian"_n, AglMapping{ (AglType)1, 1, 0x053C });
    s_aglList.emplace_back("Lj"_n, AglMapping{ (AglType)1, 1, 0x01C8 });
    s_aglList.emplace_back("Ljecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0409 });
    s_aglList.emplace_back("Llinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E3A });
    s_aglList.emplace_back("Lmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF2C });
    s_aglList.emplace_back("Lslash"_n, AglMapping{ (AglType)11, 1, 0x0141 });
    s_aglList.emplace_back("Lslashsmall"_n, AglMapping{ (AglType)1, 1, 0xF6F9 });
    s_aglList.emplace_back("Lsmall"_n, AglMapping{ (AglType)1, 1, 0xF76C });
    s_aglList.emplace_back("M"_n, AglMapping{ (AglType)11, 1, 0x004D });
    s_aglList.emplace_back("MBsquare"_n, AglMapping{ (AglType)1, 1, 0x3386 });
    s_aglList.emplace_back("Macron"_n, AglMapping{ (AglType)1, 1, 0xF6D0 });
    s_aglList.emplace_back("Macronsmall"_n, AglMapping{ (AglType)1, 1, 0xF7AF });
    s_aglList.emplace_back("Macute"_n, AglMapping{ (AglType)1, 1, 0x1E3E });
    s_aglList.emplace_back("Mcircle"_n, AglMapping{ (AglType)1, 1, 0x24C2 });
    s_aglList.emplace_back("Mdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E40 });
    s_aglList.emplace_back("Mdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E42 });
    s_aglList.emplace_back("Menarmenian"_n, AglMapping{ (AglType)1, 1, 0x0544 });
    s_aglList.emplace_back("Mmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF2D });
    s_aglList.emplace_back("Msmall"_n, AglMapping{ (AglType)1, 1, 0xF76D });
    s_aglList.emplace_back("Mturned"_n, AglMapping{ (AglType)1, 1, 0x019C });
    s_aglList.emplace_back("Mu"_n, AglMapping{ (AglType)3, 1, 0x039C });
    s_aglList.emplace_back("N"_n, AglMapping{ (AglType)11, 1, 0x004E });
    s_aglList.emplace_back("NJ"_n, AglMapping{ (AglType)1, 1, 0x01CA });
    s_aglList.emplace_back("Nacute"_n, AglMapping{ (AglType)3, 1, 0x0143 });
    s_aglList.emplace_back("Ncaron"_n, AglMapping{ (AglType)3, 1, 0x0147 });
    s_aglList.emplace_back("Ncedilla"_n, AglMapping{ (AglType)1, 1, 0x0145 });
    s_aglList.emplace_back("Ncircle"_n, AglMapping{ (AglType)1, 1, 0x24C3 });
    s_aglList.emplace_back("Ncircumflexbelow"_n, AglMapping{ (AglType)1, 1, 0x1E4A });
    s_aglList.emplace_back("Ncommaaccent"_n, AglMapping{ (AglType)1, 1, 0x0145 });
    s_aglList.emplace_back("Ndotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E44 });
    s_aglList.emplace_back("Ndotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E46 });
    s_aglList.emplace_back("Nhookleft"_n, AglMapping{ (AglType)1, 1, 0x019D });
    s_aglList.emplace_back("Nineroman"_n, AglMapping{ (AglType)1, 1, 0x2168 });
    s_aglList.emplace_back("Nj"_n, AglMapping{ (AglType)1, 1, 0x01CB });
    s_aglList.emplace_back("Njecyrillic"_n, AglMapping{ (AglType)1, 1, 0x040A });
    s_aglList.emplace_back("Nlinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E48 });
    s_aglList.emplace_back("Nmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF2E });
    s_aglList.emplace_back("Nowarmenian"_n, AglMapping{ (AglType)1, 1, 0x0546 });
    s_aglList.emplace_back("Nsmall"_n, AglMapping{ (AglType)1, 1, 0xF76E });
    s_aglList.emplace_back("Ntilde"_n, AglMapping{ (AglType)11, 1, 0x00D1 });
    s_aglList.emplace_back("Ntildesmall"_n, AglMapping{ (AglType)1, 1, 0xF7F1 });
    s_aglList.emplace_back("Nu"_n, AglMapping{ (AglType)3, 1, 0x039D });
    s_aglList.emplace_back("O"_n, AglMapping{ (AglType)11, 1, 0x004F });
    s_aglList.emplace_back("OE"_n, AglMapping{ (AglType)11, 1, 0x0152 });
    s_aglList.emplace_back("OEsmall"_n, AglMapping{ (AglType)1, 1, 0xF6FA });
    s_aglList.emplace_back("Oacute"_n, AglMapping{ (AglType)11, 1, 0x00D3 });
    s_aglList.emplace_back("Oacutesmall"_n, AglMapping{ (AglType)1, 1, 0xF7F3 });
    s_aglList.emplace_back("Obarredcyrillic"_n, AglMapping{ (AglType)1, 1, 0x04E8 });
    s_aglList.emplace_back("Obarreddieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04EA });
    s_aglList.emplace_back("Obreve"_n, AglMapping{ (AglType)3, 1, 0x014E });
    s_aglList.emplace_back("Ocaron"_n, AglMapping{ (AglType)1, 1, 0x01D1 });
    s_aglList.emplace_back("Ocenteredtilde"_n, AglMapping{ (AglType)1, 1, 0x019F });
    s_aglList.emplace_back("Ocircle"_n, AglMapping{ (AglType)1, 1, 0x24C4 });
    s_aglList.emplace_back("Ocircumflex"_n, AglMapping{ (AglType)11, 1, 0x00D4 });
    s_aglList.emplace_back("Ocircumflexacute"_n, AglMapping{ (AglType)1, 1, 0x1ED0 });
    s_aglList.emplace_back("Ocircumflexdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1ED8 });
    s_aglList.emplace_back("Ocircumflexgrave"_n, AglMapping{ (AglType)1, 1, 0x1ED2 });
    s_aglList.emplace_back("Ocircumflexhookabove"_n, AglMapping{ (AglType)1, 1, 0x1ED4 });
    s_aglList.emplace_back("Ocircumflexsmall"_n, AglMapping{ (AglType)1, 1, 0xF7F4 });
    s_aglList.emplace_back("Ocircumflextilde"_n, AglMapping{ (AglType)1, 1, 0x1ED6 });
    s_aglList.emplace_back("Ocyrillic"_n, AglMapping{ (AglType)1, 1, 0x041E });
    s_aglList.emplace_back("Odblacute"_n, AglMapping{ (AglType)1, 1, 0x0150 });
    s_aglList.emplace_back("Odblgrave"_n, AglMapping{ (AglType)1, 1, 0x020C });
    s_aglList.emplace_back("Odieresis"_n, AglMapping{ (AglType)11, 1, 0x00D6 });
    s_aglList.emplace_back("Odieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04E6 });
    s_aglList.emplace_back("Odieresissmall"_n, AglMapping{ (AglType)1, 1, 0xF7F6 });
    s_aglList.emplace_back("Odotbelow"_n, AglMapping{ (AglType)1, 1, 0x1ECC });
    s_aglList.emplace_back("Ogoneksmall"_n, AglMapping{ (AglType)1, 1, 0xF6FB });
    s_aglList.emplace_back("Ograve"_n, AglMapping{ (AglType)11, 1, 0x00D2 });
    s_aglList.emplace_back("Ogravesmall"_n, AglMapping{ (AglType)1, 1, 0xF7F2 });
    s_aglList.emplace_back("Oharmenian"_n, AglMapping{ (AglType)1, 1, 0x0555 });
    s_aglList.emplace_back("Ohm"_n, AglMapping{ (AglType)1, 1, 0x2126 });
    s_aglList.emplace_back("Ohookabove"_n, AglMapping{ (AglType)1, 1, 0x1ECE });
    s_aglList.emplace_back("Ohorn"_n, AglMapping{ (AglType)3, 1, 0x01A0 });
    s_aglList.emplace_back("Ohornacute"_n, AglMapping{ (AglType)1, 1, 0x1EDA });
    s_aglList.emplace_back("Ohorndotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EE2 });
    s_aglList.emplace_back("Ohorngrave"_n, AglMapping{ (AglType)1, 1, 0x1EDC });
    s_aglList.emplace_back("Ohornhookabove"_n, AglMapping{ (AglType)1, 1, 0x1EDE });
    s_aglList.emplace_back("Ohorntilde"_n, AglMapping{ (AglType)1, 1, 0x1EE0 });
    s_aglList.emplace_back("Ohungarumlaut"_n, AglMapping{ (AglType)3, 1, 0x0150 });
    s_aglList.emplace_back("Oi"_n, AglMapping{ (AglType)1, 1, 0x01A2 });
    s_aglList.emplace_back("Oinvertedbreve"_n, AglMapping{ (AglType)1, 1, 0x020E });
    s_aglList.emplace_back("Omacron"_n, AglMapping{ (AglType)3, 1, 0x014C });
    s_aglList.emplace_back("Omacronacute"_n, AglMapping{ (AglType)1, 1, 0x1E52 });
    s_aglList.emplace_back("Omacrongrave"_n, AglMapping{ (AglType)1, 1, 0x1E50 });
    s_aglList.emplace_back("Omega"_n, AglMapping{ (AglType)3, 1, 0x2126 });
    s_aglList.emplace_back("Omegacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0460 });
    s_aglList.emplace_back("Omegagreek"_n, AglMapping{ (AglType)1, 1, 0x03A9 });
    s_aglList.emplace_back("Omegaroundcyrillic"_n, AglMapping{ (AglType)1, 1, 0x047A });
    s_aglList.emplace_back("Omegatitlocyrillic"_n, AglMapping{ (AglType)1, 1, 0x047C });
    s_aglList.emplace_back("Omegatonos"_n, AglMapping{ (AglType)3, 1, 0x038F });
    s_aglList.emplace_back("Omicron"_n, AglMapping{ (AglType)3, 1, 0x039F });
    s_aglList.emplace_back("Omicrontonos"_n, AglMapping{ (AglType)3, 1, 0x038C });
    s_aglList.emplace_back("Omonospace"_n, AglMapping{ (AglType)1, 1, 0xFF2F });
    s_aglList.emplace_back("Oneroman"_n, AglMapping{ (AglType)1, 1, 0x2160 });
    s_aglList.emplace_back("Oogonek"_n, AglMapping{ (AglType)1, 1, 0x01EA });
    s_aglList.emplace_back("Oogonekmacron"_n, AglMapping{ (AglType)1, 1, 0x01EC });
    s_aglList.emplace_back("Oopen"_n, AglMapping{ (AglType)1, 1, 0x0186 });
    s_aglList.emplace_back("Oslash"_n, AglMapping{ (AglType)11, 1, 0x00D8 });
    s_aglList.emplace_back("Oslashacute"_n, AglMapping{ (AglType)3, 1, 0x01FE });
    s_aglList.emplace_back("Oslashsmall"_n, AglMapping{ (AglType)1, 1, 0xF7F8 });
    s_aglList.emplace_back("Osmall"_n, AglMapping{ (AglType)1, 1, 0xF76F });
    s_aglList.emplace_back("Ostrokeacute"_n, AglMapping{ (AglType)1, 1, 0x01FE });
    s_aglList.emplace_back("Otcyrillic"_n, AglMapping{ (AglType)1, 1, 0x047E });
    s_aglList.emplace_back("Otilde"_n, AglMapping{ (AglType)11, 1, 0x00D5 });
    s_aglList.emplace_back("Otildeacute"_n, AglMapping{ (AglType)1, 1, 0x1E4C });
    s_aglList.emplace_back("Otildedieresis"_n, AglMapping{ (AglType)1, 1, 0x1E4E });
    s_aglList.emplace_back("Otildesmall"_n, AglMapping{ (AglType)1, 1, 0xF7F5 });
    s_aglList.emplace_back("P"_n, AglMapping{ (AglType)11, 1, 0x0050 });
    s_aglList.emplace_back("Pacute"_n, AglMapping{ (AglType)1, 1, 0x1E54 });
    s_aglList.emplace_back("Pcircle"_n, AglMapping{ (AglType)1, 1, 0x24C5 });
    s_aglList.emplace_back("Pdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E56 });
    s_aglList.emplace_back("Pecyrillic"_n, AglMapping{ (AglType)1, 1, 0x041F });
    s_aglList.emplace_back("Peharmenian"_n, AglMapping{ (AglType)1, 1, 0x054A });
    s_aglList.emplace_back("Pemiddlehookcyrillic"_n, AglMapping{ (AglType)1, 1, 0x04A6 });
    s_aglList.emplace_back("Phi"_n, AglMapping{ (AglType)3, 1, 0x03A6 });
    s_aglList.emplace_back("Phook"_n, AglMapping{ (AglType)1, 1, 0x01A4 });
    s_aglList.emplace_back("Pi"_n, AglMapping{ (AglType)3, 1, 0x03A0 });
    s_aglList.emplace_back("Piwrarmenian"_n, AglMapping{ (AglType)1, 1, 0x0553 });
    s_aglList.emplace_back("Pmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF30 });
    s_aglList.emplace_back("Psi"_n, AglMapping{ (AglType)3, 1, 0x03A8 });
    s_aglList.emplace_back("Psicyrillic"_n, AglMapping{ (AglType)1, 1, 0x0470 });
    s_aglList.emplace_back("Psmall"_n, AglMapping{ (AglType)1, 1, 0xF770 });
    s_aglList.emplace_back("Q"_n, AglMapping{ (AglType)11, 1, 0x0051 });
    s_aglList.emplace_back("Qcircle"_n, AglMapping{ (AglType)1, 1, 0x24C6 });
    s_aglList.emplace_back("Qmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF31 });
    s_aglList.emplace_back("Qsmall"_n, AglMapping{ (AglType)1, 1, 0xF771 });
    s_aglList.emplace_back("R"_n, AglMapping{ (AglType)11, 1, 0x0052 });
    s_aglList.emplace_back("Raarmenian"_n, AglMapping{ (AglType)1, 1, 0x054C });
    s_aglList.emplace_back("Racute"_n, AglMapping{ (AglType)3, 1, 0x0154 });
    s_aglList.emplace_back("Rcaron"_n, AglMapping{ (AglType)3, 1, 0x0158 });
    s_aglList.emplace_back("Rcedilla"_n, AglMapping{ (AglType)1, 1, 0x0156 });
    s_aglList.emplace_back("Rcircle"_n, AglMapping{ (AglType)1, 1, 0x24C7 });
    s_aglList.emplace_back("Rcommaaccent"_n, AglMapping{ (AglType)1, 1, 0x0156 });
    s_aglList.emplace_back("Rdblgrave"_n, AglMapping{ (AglType)1, 1, 0x0210 });
    s_aglList.emplace_back("Rdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E58 });
    s_aglList.emplace_back("Rdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E5A });
    s_aglList.emplace_back("Rdotbelowmacron"_n, AglMapping{ (AglType)1, 1, 0x1E5C });
    s_aglList.emplace_back("Reharmenian"_n, AglMapping{ (AglType)1, 1, 0x0550 });
    s_aglList.emplace_back("Rfraktur"_n, AglMapping{ (AglType)3, 1, 0x211C });
    s_aglList.emplace_back("Rho"_n, AglMapping{ (AglType)3, 1, 0x03A1 });
    s_aglList.emplace_back("Ringsmall"_n, AglMapping{ (AglType)1, 1, 0xF6FC });
    s_aglList.emplace_back("Rinvertedbreve"_n, AglMapping{ (AglType)1, 1, 0x0212 });
    s_aglList.emplace_back("Rlinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E5E });
    s_aglList.emplace_back("Rmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF32 });
    s_aglList.emplace_back("Rsmall"_n, AglMapping{ (AglType)1, 1, 0xF772 });
    s_aglList.emplace_back("Rsmallinverted"_n, AglMapping{ (AglType)1, 1, 0x0281 });
    s_aglList.emplace_back("Rsmallinvertedsuperior"_n, AglMapping{ (AglType)1, 1, 0x02B6 });
    s_aglList.emplace_back("S"_n, AglMapping{ (AglType)11, 1, 0x0053 });
    s_aglList.emplace_back("SF010000"_n, AglMapping{ (AglType)3, 1, 0x250C });
    s_aglList.emplace_back("SF020000"_n, AglMapping{ (AglType)3, 1, 0x2514 });
    s_aglList.emplace_back("SF030000"_n, AglMapping{ (AglType)3, 1, 0x2510 });
    s_aglList.emplace_back("SF040000"_n, AglMapping{ (AglType)3, 1, 0x2518 });
    s_aglList.emplace_back("SF050000"_n, AglMapping{ (AglType)3, 1, 0x253C });
    s_aglList.emplace_back("SF060000"_n, AglMapping{ (AglType)3, 1, 0x252C });
    s_aglList.emplace_back("SF070000"_n, AglMapping{ (AglType)3, 1, 0x2534 });
    s_aglList.emplace_back("SF080000"_n, AglMapping{ (AglType)3, 1, 0x251C });
    s_aglList.emplace_back("SF090000"_n, AglMapping{ (AglType)3, 1, 0x2524 });
    s_aglList.emplace_back("SF100000"_n, AglMapping{ (AglType)3, 1, 0x2500 });
    s_aglList.emplace_back("SF110000"_n, AglMapping{ (AglType)3, 1, 0x2502 });
    s_aglList.emplace_back("SF190000"_n, AglMapping{ (AglType)3, 1, 0x2561 });
    s_aglList.emplace_back("SF200000"_n, AglMapping{ (AglType)3, 1, 0x2562 });
    s_aglList.emplace_back("SF210000"_n, AglMapping{ (AglType)3, 1, 0x2556 });
    s_aglList.emplace_back("SF220000"_n, AglMapping{ (AglType)3, 1, 0x2555 });
    s_aglList.emplace_back("SF230000"_n, AglMapping{ (AglType)3, 1, 0x2563 });
    s_aglList.emplace_back("SF240000"_n, AglMapping{ (AglType)3, 1, 0x2551 });
    s_aglList.emplace_back("SF250000"_n, AglMapping{ (AglType)3, 1, 0x2557 });
    s_aglList.emplace_back("SF260000"_n, AglMapping{ (AglType)3, 1, 0x255D });
    s_aglList.emplace_back("SF270000"_n, AglMapping{ (AglType)3, 1, 0x255C });
    s_aglList.emplace_back("SF280000"_n, AglMapping{ (AglType)3, 1, 0x255B });
    s_aglList.emplace_back("SF360000"_n, AglMapping{ (AglType)3, 1, 0x255E });
    s_aglList.emplace_back("SF370000"_n, AglMapping{ (AglType)3, 1, 0x255F });
    s_aglList.emplace_back("SF380000"_n, AglMapping{ (AglType)3, 1, 0x255A });
    s_aglList.emplace_back("SF390000"_n, AglMapping{ (AglType)3, 1, 0x2554 });
    s_aglList.emplace_back("SF400000"_n, AglMapping{ (AglType)3, 1, 0x2569 });
    s_aglList.emplace_back("SF410000"_n, AglMapping{ (AglType)3, 1, 0x2566 });
    s_aglList.emplace_back("SF420000"_n, AglMapping{ (AglType)3, 1, 0x2560 });
    s_aglList.emplace_back("SF430000"_n, AglMapping{ (AglType)3, 1, 0x2550 });
    s_aglList.emplace_back("SF440000"_n, AglMapping{ (AglType)3, 1, 0x256C });
    s_aglList.emplace_back("SF450000"_n, AglMapping{ (AglType)3, 1, 0x2567 });
    s_aglList.emplace_back("SF460000"_n, AglMapping{ (AglType)3, 1, 0x2568 });
    s_aglList.emplace_back("SF470000"_n, AglMapping{ (AglType)3, 1, 0x2564 });
    s_aglList.emplace_back("SF480000"_n, AglMapping{ (AglType)3, 1, 0x2565 });
    s_aglList.emplace_back("SF490000"_n, AglMapping{ (AglType)3, 1, 0x2559 });
    s_aglList.emplace_back("SF500000"_n, AglMapping{ (AglType)3, 1, 0x2558 });
    s_aglList.emplace_back("SF510000"_n, AglMapping{ (AglType)3, 1, 0x2552 });
    s_aglList.emplace_back("SF520000"_n, AglMapping{ (AglType)3, 1, 0x2553 });
    s_aglList.emplace_back("SF530000"_n, AglMapping{ (AglType)3, 1, 0x256B });
    s_aglList.emplace_back("SF540000"_n, AglMapping{ (AglType)3, 1, 0x256A });
    s_aglList.emplace_back("Sacute"_n, AglMapping{ (AglType)3, 1, 0x015A });
    s_aglList.emplace_back("Sacutedotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E64 });
    s_aglList.emplace_back("Sampigreek"_n, AglMapping{ (AglType)1, 1, 0x03E0 });
    s_aglList.emplace_back("Scaron"_n, AglMapping{ (AglType)11, 1, 0x0160 });
    s_aglList.emplace_back("Scarondotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E66 });
    s_aglList.emplace_back("Scaronsmall"_n, AglMapping{ (AglType)1, 1, 0xF6FD });
    s_aglList.emplace_back("Scedilla"_n, AglMapping{ (AglType)3, 1, 0x015E });
    s_aglList.emplace_back("Schwa"_n, AglMapping{ (AglType)1, 1, 0x018F });
    s_aglList.emplace_back("Schwacyrillic"_n, AglMapping{ (AglType)1, 1, 0x04D8 });
    s_aglList.emplace_back("Schwadieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04DA });
    s_aglList.emplace_back("Scircle"_n, AglMapping{ (AglType)1, 1, 0x24C8 });
    s_aglList.emplace_back("Scircumflex"_n, AglMapping{ (AglType)3, 1, 0x015C });
    s_aglList.emplace_back("Scommaaccent"_n, AglMapping{ (AglType)1, 1, 0x0218 });
    s_aglList.emplace_back("Sdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E60 });
    s_aglList.emplace_back("Sdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E62 });
    s_aglList.emplace_back("Sdotbelowdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E68 });
    s_aglList.emplace_back("Seharmenian"_n, AglMapping{ (AglType)1, 1, 0x054D });
    s_aglList.emplace_back("Sevenroman"_n, AglMapping{ (AglType)1, 1, 0x2166 });
    s_aglList.emplace_back("Shaarmenian"_n, AglMapping{ (AglType)1, 1, 0x0547 });
    s_aglList.emplace_back("Shacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0428 });
    s_aglList.emplace_back("Shchacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0429 });
    s_aglList.emplace_back("Sheicoptic"_n, AglMapping{ (AglType)1, 1, 0x03E2 });
    s_aglList.emplace_back("Shhacyrillic"_n, AglMapping{ (AglType)1, 1, 0x04BA });
    s_aglList.emplace_back("Shimacoptic"_n, AglMapping{ (AglType)1, 1, 0x03EC });
    s_aglList.emplace_back("Sigma"_n, AglMapping{ (AglType)3, 1, 0x03A3 });
    s_aglList.emplace_back("Sixroman"_n, AglMapping{ (AglType)1, 1, 0x2165 });
    s_aglList.emplace_back("Smonospace"_n, AglMapping{ (AglType)1, 1, 0xFF33 });
    s_aglList.emplace_back("Softsigncyrillic"_n, AglMapping{ (AglType)1, 1, 0x042C });
    s_aglList.emplace_back("Ssmall"_n, AglMapping{ (AglType)1, 1, 0xF773 });
    s_aglList.emplace_back("Stigmagreek"_n, AglMapping{ (AglType)1, 1, 0x03DA });
    s_aglList.emplace_back("T"_n, AglMapping{ (AglType)11, 1, 0x0054 });
    s_aglList.emplace_back("Tau"_n, AglMapping{ (AglType)3, 1, 0x03A4 });
    s_aglList.emplace_back("Tbar"_n, AglMapping{ (AglType)3, 1, 0x0166 });
    s_aglList.emplace_back("Tcaron"_n, AglMapping{ (AglType)3, 1, 0x0164 });
    s_aglList.emplace_back("Tcedilla"_n, AglMapping{ (AglType)1, 1, 0x0162 });
    s_aglList.emplace_back("Tcircle"_n, AglMapping{ (AglType)1, 1, 0x24C9 });
    s_aglList.emplace_back("Tcircumflexbelow"_n, AglMapping{ (AglType)1, 1, 0x1E70 });
    s_aglList.emplace_back("Tcommaaccent"_n, AglMapping{ (AglType)1, 1, 0x0162 });
    s_aglList.emplace_back("Tdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E6A });
    s_aglList.emplace_back("Tdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E6C });
    s_aglList.emplace_back("Tecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0422 });
    s_aglList.emplace_back("Tedescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x04AC });
    s_aglList.emplace_back("Tenroman"_n, AglMapping{ (AglType)1, 1, 0x2169 });
    s_aglList.emplace_back("Tetsecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04B4 });
    s_aglList.emplace_back("Theta"_n, AglMapping{ (AglType)3, 1, 0x0398 });
    s_aglList.emplace_back("Thook"_n, AglMapping{ (AglType)1, 1, 0x01AC });
    s_aglList.emplace_back("Thorn"_n, AglMapping{ (AglType)11, 1, 0x00DE });
    s_aglList.emplace_back("Thornsmall"_n, AglMapping{ (AglType)1, 1, 0xF7FE });
    s_aglList.emplace_back("Threeroman"_n, AglMapping{ (AglType)1, 1, 0x2162 });
    s_aglList.emplace_back("Tildesmall"_n, AglMapping{ (AglType)1, 1, 0xF6FE });
    s_aglList.emplace_back("Tiwnarmenian"_n, AglMapping{ (AglType)1, 1, 0x054F });
    s_aglList.emplace_back("Tlinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E6E });
    s_aglList.emplace_back("Tmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF34 });
    s_aglList.emplace_back("Toarmenian"_n, AglMapping{ (AglType)1, 1, 0x0539 });
    s_aglList.emplace_back("Tonefive"_n, AglMapping{ (AglType)1, 1, 0x01BC });
    s_aglList.emplace_back("Tonesix"_n, AglMapping{ (AglType)1, 1, 0x0184 });
    s_aglList.emplace_back("Tonetwo"_n, AglMapping{ (AglType)1, 1, 0x01A7 });
    s_aglList.emplace_back("Tretroflexhook"_n, AglMapping{ (AglType)1, 1, 0x01AE });
    s_aglList.emplace_back("Tsecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0426 });
    s_aglList.emplace_back("Tshecyrillic"_n, AglMapping{ (AglType)1, 1, 0x040B });
    s_aglList.emplace_back("Tsmall"_n, AglMapping{ (AglType)1, 1, 0xF774 });
    s_aglList.emplace_back("Twelveroman"_n, AglMapping{ (AglType)1, 1, 0x216B });
    s_aglList.emplace_back("Tworoman"_n, AglMapping{ (AglType)1, 1, 0x2161 });
    s_aglList.emplace_back("U"_n, AglMapping{ (AglType)11, 1, 0x0055 });
    s_aglList.emplace_back("Uacute"_n, AglMapping{ (AglType)11, 1, 0x00DA });
    s_aglList.emplace_back("Uacutesmall"_n, AglMapping{ (AglType)1, 1, 0xF7FA });
    s_aglList.emplace_back("Ubreve"_n, AglMapping{ (AglType)3, 1, 0x016C });
    s_aglList.emplace_back("Ucaron"_n, AglMapping{ (AglType)1, 1, 0x01D3 });
    s_aglList.emplace_back("Ucircle"_n, AglMapping{ (AglType)1, 1, 0x24CA });
    s_aglList.emplace_back("Ucircumflex"_n, AglMapping{ (AglType)11, 1, 0x00DB });
    s_aglList.emplace_back("Ucircumflexbelow"_n, AglMapping{ (AglType)1, 1, 0x1E76 });
    s_aglList.emplace_back("Ucircumflexsmall"_n, AglMapping{ (AglType)1, 1, 0xF7FB });
    s_aglList.emplace_back("Ucyrillic"_n, AglMapping{ (AglType)1, 1, 0x0423 });
    s_aglList.emplace_back("Udblacute"_n, AglMapping{ (AglType)1, 1, 0x0170 });
    s_aglList.emplace_back("Udblgrave"_n, AglMapping{ (AglType)1, 1, 0x0214 });
    s_aglList.emplace_back("Udieresis"_n, AglMapping{ (AglType)11, 1, 0x00DC });
    s_aglList.emplace_back("Udieresisacute"_n, AglMapping{ (AglType)1, 1, 0x01D7 });
    s_aglList.emplace_back("Udieresisbelow"_n, AglMapping{ (AglType)1, 1, 0x1E72 });
    s_aglList.emplace_back("Udieresiscaron"_n, AglMapping{ (AglType)1, 1, 0x01D9 });
    s_aglList.emplace_back("Udieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04F0 });
    s_aglList.emplace_back("Udieresisgrave"_n, AglMapping{ (AglType)1, 1, 0x01DB });
    s_aglList.emplace_back("Udieresismacron"_n, AglMapping{ (AglType)1, 1, 0x01D5 });
    s_aglList.emplace_back("Udieresissmall"_n, AglMapping{ (AglType)1, 1, 0xF7FC });
    s_aglList.emplace_back("Udotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EE4 });
    s_aglList.emplace_back("Ugrave"_n, AglMapping{ (AglType)11, 1, 0x00D9 });
    s_aglList.emplace_back("Ugravesmall"_n, AglMapping{ (AglType)1, 1, 0xF7F9 });
    s_aglList.emplace_back("Uhookabove"_n, AglMapping{ (AglType)1, 1, 0x1EE6 });
    s_aglList.emplace_back("Uhorn"_n, AglMapping{ (AglType)3, 1, 0x01AF });
    s_aglList.emplace_back("Uhornacute"_n, AglMapping{ (AglType)1, 1, 0x1EE8 });
    s_aglList.emplace_back("Uhorndotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EF0 });
    s_aglList.emplace_back("Uhorngrave"_n, AglMapping{ (AglType)1, 1, 0x1EEA });
    s_aglList.emplace_back("Uhornhookabove"_n, AglMapping{ (AglType)1, 1, 0x1EEC });
    s_aglList.emplace_back("Uhorntilde"_n, AglMapping{ (AglType)1, 1, 0x1EEE });
    s_aglList.emplace_back("Uhungarumlaut"_n, AglMapping{ (AglType)3, 1, 0x0170 });
    s_aglList.emplace_back("Uhungarumlautcyrillic"_n, AglMapping{ (AglType)1, 1, 0x04F2 });
    s_aglList.emplace_back("Uinvertedbreve"_n, AglMapping{ (AglType)1, 1, 0x0216 });
    s_aglList.emplace_back("Ukcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0478 });
    s_aglList.emplace_back("Umacron"_n, AglMapping{ (AglType)3, 1, 0x016A });
    s_aglList.emplace_back("Umacroncyrillic"_n, AglMapping{ (AglType)1, 1, 0x04EE });
    s_aglList.emplace_back("Umacrondieresis"_n, AglMapping{ (AglType)1, 1, 0x1E7A });
    s_aglList.emplace_back("Umonospace"_n, AglMapping{ (AglType)1, 1, 0xFF35 });
    s_aglList.emplace_back("Uogonek"_n, AglMapping{ (AglType)3, 1, 0x0172 });
    s_aglList.emplace_back("Upsilon"_n, AglMapping{ (AglType)3, 1, 0x03A5 });
    s_aglList.emplace_back("Upsilon1"_n, AglMapping{ (AglType)3, 1, 0x03D2 });
    s_aglList.emplace_back("Upsilonacutehooksymbolgreek"_n, AglMapping{ (AglType)1, 1, 0x03D3 });
    s_aglList.emplace_back("Upsilonafrican"_n, AglMapping{ (AglType)1, 1, 0x01B1 });
    s_aglList.emplace_back("Upsilondieresis"_n, AglMapping{ (AglType)3, 1, 0x03AB });
    s_aglList.emplace_back("Upsilondieresishooksymbolgreek"_n, AglMapping{ (AglType)1, 1, 0x03D4 });
    s_aglList.emplace_back("Upsilonhooksymbol"_n, AglMapping{ (AglType)1, 1, 0x03D2 });
    s_aglList.emplace_back("Upsilontonos"_n, AglMapping{ (AglType)3, 1, 0x038E });
    s_aglList.emplace_back("Uring"_n, AglMapping{ (AglType)3, 1, 0x016E });
    s_aglList.emplace_back("Ushortcyrillic"_n, AglMapping{ (AglType)1, 1, 0x040E });
    s_aglList.emplace_back("Usmall"_n, AglMapping{ (AglType)1, 1, 0xF775 });
    s_aglList.emplace_back("Ustraightcyrillic"_n, AglMapping{ (AglType)1, 1, 0x04AE });
    s_aglList.emplace_back("Ustraightstrokecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04B0 });
    s_aglList.emplace_back("Utilde"_n, AglMapping{ (AglType)3, 1, 0x0168 });
    s_aglList.emplace_back("Utildeacute"_n, AglMapping{ (AglType)1, 1, 0x1E78 });
    s_aglList.emplace_back("Utildebelow"_n, AglMapping{ (AglType)1, 1, 0x1E74 });
    s_aglList.emplace_back("V"_n, AglMapping{ (AglType)11, 1, 0x0056 });
    s_aglList.emplace_back("Vcircle"_n, AglMapping{ (AglType)1, 1, 0x24CB });
    s_aglList.emplace_back("Vdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E7E });
    s_aglList.emplace_back("Vecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0412 });
    s_aglList.emplace_back("Vewarmenian"_n, AglMapping{ (AglType)1, 1, 0x054E });
    s_aglList.emplace_back("Vhook"_n, AglMapping{ (AglType)1, 1, 0x01B2 });
    s_aglList.emplace_back("Vmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF36 });
    s_aglList.emplace_back("Voarmenian"_n, AglMapping{ (AglType)1, 1, 0x0548 });
    s_aglList.emplace_back("Vsmall"_n, AglMapping{ (AglType)1, 1, 0xF776 });
    s_aglList.emplace_back("Vtilde"_n, AglMapping{ (AglType)1, 1, 0x1E7C });
    s_aglList.emplace_back("W"_n, AglMapping{ (AglType)11, 1, 0x0057 });
    s_aglList.emplace_back("Wacute"_n, AglMapping{ (AglType)3, 1, 0x1E82 });
    s_aglList.emplace_back("Wcircle"_n, AglMapping{ (AglType)1, 1, 0x24CC });
    s_aglList.emplace_back("Wcircumflex"_n, AglMapping{ (AglType)3, 1, 0x0174 });
    s_aglList.emplace_back("Wdieresis"_n, AglMapping{ (AglType)3, 1, 0x1E84 });
    s_aglList.emplace_back("Wdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E86 });
    s_aglList.emplace_back("Wdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E88 });
    s_aglList.emplace_back("Wgrave"_n, AglMapping{ (AglType)3, 1, 0x1E80 });
    s_aglList.emplace_back("Wmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF37 });
    s_aglList.emplace_back("Wsmall"_n, AglMapping{ (AglType)1, 1, 0xF777 });
    s_aglList.emplace_back("X"_n, AglMapping{ (AglType)11, 1, 0x0058 });
    s_aglList.emplace_back("Xcircle"_n, AglMapping{ (AglType)1, 1, 0x24CD });
    s_aglList.emplace_back("Xdieresis"_n, AglMapping{ (AglType)1, 1, 0x1E8C });
    s_aglList.emplace_back("Xdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E8A });
    s_aglList.emplace_back("Xeharmenian"_n, AglMapping{ (AglType)1, 1, 0x053D });
    s_aglList.emplace_back("Xi"_n, AglMapping{ (AglType)3, 1, 0x039E });
    s_aglList.emplace_back("Xmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF38 });
    s_aglList.emplace_back("Xsmall"_n, AglMapping{ (AglType)1, 1, 0xF778 });
    s_aglList.emplace_back("Y"_n, AglMapping{ (AglType)11, 1, 0x0059 });
    s_aglList.emplace_back("Yacute"_n, AglMapping{ (AglType)11, 1, 0x00DD });
    s_aglList.emplace_back("Yacutesmall"_n, AglMapping{ (AglType)1, 1, 0xF7FD });
    s_aglList.emplace_back("Yatcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0462 });
    s_aglList.emplace_back("Ycircle"_n, AglMapping{ (AglType)1, 1, 0x24CE });
    s_aglList.emplace_back("Ycircumflex"_n, AglMapping{ (AglType)3, 1, 0x0176 });
    s_aglList.emplace_back("Ydieresis"_n, AglMapping{ (AglType)11, 1, 0x0178 });
    s_aglList.emplace_back("Ydieresissmall"_n, AglMapping{ (AglType)1, 1, 0xF7FF });
    s_aglList.emplace_back("Ydotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E8E });
    s_aglList.emplace_back("Ydotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EF4 });
    s_aglList.emplace_back("Yericyrillic"_n, AglMapping{ (AglType)1, 1, 0x042B });
    s_aglList.emplace_back("Yerudieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04F8 });
    s_aglList.emplace_back("Ygrave"_n, AglMapping{ (AglType)3, 1, 0x1EF2 });
    s_aglList.emplace_back("Yhook"_n, AglMapping{ (AglType)1, 1, 0x01B3 });
    s_aglList.emplace_back("Yhookabove"_n, AglMapping{ (AglType)1, 1, 0x1EF6 });
    s_aglList.emplace_back("Yiarmenian"_n, AglMapping{ (AglType)1, 1, 0x0545 });
    s_aglList.emplace_back("Yicyrillic"_n, AglMapping{ (AglType)1, 1, 0x0407 });
    s_aglList.emplace_back("Yiwnarmenian"_n, AglMapping{ (AglType)1, 1, 0x0552 });
    s_aglList.emplace_back("Ymonospace"_n, AglMapping{ (AglType)1, 1, 0xFF39 });
    s_aglList.emplace_back("Ysmall"_n, AglMapping{ (AglType)1, 1, 0xF779 });
    s_aglList.emplace_back("Ytilde"_n, AglMapping{ (AglType)1, 1, 0x1EF8 });
    s_aglList.emplace_back("Yusbigcyrillic"_n, AglMapping{ (AglType)1, 1, 0x046A });
    s_aglList.emplace_back("Yusbigiotifiedcyrillic"_n, AglMapping{ (AglType)1, 1, 0x046C });
    s_aglList.emplace_back("Yuslittlecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0466 });
    s_aglList.emplace_back("Yuslittleiotifiedcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0468 });
    s_aglList.emplace_back("Z"_n, AglMapping{ (AglType)11, 1, 0x005A });
    s_aglList.emplace_back("Zaarmenian"_n, AglMapping{ (AglType)1, 1, 0x0536 });
    s_aglList.emplace_back("Zacute"_n, AglMapping{ (AglType)3, 1, 0x0179 });
    s_aglList.emplace_back("Zcaron"_n, AglMapping{ (AglType)11, 1, 0x017D });
    s_aglList.emplace_back("Zcaronsmall"_n, AglMapping{ (AglType)1, 1, 0xF6FF });
    s_aglList.emplace_back("Zcircle"_n, AglMapping{ (AglType)1, 1, 0x24CF });
    s_aglList.emplace_back("Zcircumflex"_n, AglMapping{ (AglType)1, 1, 0x1E90 });
    s_aglList.emplace_back("Zdot"_n, AglMapping{ (AglType)1, 1, 0x017B });
    s_aglList.emplace_back("Zdotaccent"_n, AglMapping{ (AglType)3, 1, 0x017B });
    s_aglList.emplace_back("Zdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E92 });
    s_aglList.emplace_back("Zecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0417 });
    s_aglList.emplace_back("Zedescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x0498 });
    s_aglList.emplace_back("Zedieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04DE });
    s_aglList.emplace_back("Zeta"_n, AglMapping{ (AglType)3, 1, 0x0396 });
    s_aglList.emplace_back("Zhearmenian"_n, AglMapping{ (AglType)1, 1, 0x053A });
    s_aglList.emplace_back("Zhebrevecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04C1 });
    s_aglList.emplace_back("Zhecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0416 });
    s_aglList.emplace_back("Zhedescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x0496 });
    s_aglList.emplace_back("Zhedieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04DC });
    s_aglList.emplace_back("Zlinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E94 });
    s_aglList.emplace_back("Zmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF3A });
    s_aglList.emplace_back("Zsmall"_n, AglMapping{ (AglType)1, 1, 0xF77A });
    s_aglList.emplace_back("Zstroke"_n, AglMapping{ (AglType)1, 1, 0x01B5 });
    s_aglList.emplace_back("a"_n, AglMapping{ (AglType)11, 1, 0x0061 });
    s_aglList.emplace_back("aabengali"_n, AglMapping{ (AglType)1, 1, 0x0986 });
    s_aglList.emplace_back("aacute"_n, AglMapping{ (AglType)11, 1, 0x00E1 });
    s_aglList.emplace_back("aadeva"_n, AglMapping{ (AglType)1, 1, 0x0906 });
    s_aglList.emplace_back("aagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A86 });
    s_aglList.emplace_back("aagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A06 });
    s_aglList.emplace_back("aamatragurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A3E });
    s_aglList.emplace_back("aarusquare"_n, AglMapping{ (AglType)1, 1, 0x3303 });
    s_aglList.emplace_back("aavowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09BE });
    s_aglList.emplace_back("aavowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x093E });
    s_aglList.emplace_back("aavowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0ABE });
    s_aglList.emplace_back("abbreviationmarkarmenian"_n, AglMapping{ (AglType)1, 1, 0x055F });
    s_aglList.emplace_back("abbreviationsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0970 });
    s_aglList.emplace_back("abengali"_n, AglMapping{ (AglType)1, 1, 0x0985 });
    s_aglList.emplace_back("abopomofo"_n, AglMapping{ (AglType)1, 1, 0x311A });
    s_aglList.emplace_back("abreve"_n, AglMapping{ (AglType)3, 1, 0x0103 });
    s_aglList.emplace_back("abreveacute"_n, AglMapping{ (AglType)1, 1, 0x1EAF });
    s_aglList.emplace_back("abrevecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04D1 });
    s_aglList.emplace_back("abrevedotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EB7 });
    s_aglList.emplace_back("abrevegrave"_n, AglMapping{ (AglType)1, 1, 0x1EB1 });
    s_aglList.emplace_back("abrevehookabove"_n, AglMapping{ (AglType)1, 1, 0x1EB3 });
    s_aglList.emplace_back("abrevetilde"_n, AglMapping{ (AglType)1, 1, 0x1EB5 });
    s_aglList.emplace_back("acaron"_n, AglMapping{ (AglType)1, 1, 0x01CE });
    s_aglList.emplace_back("acircle"_n, AglMapping{ (AglType)1, 1, 0x24D0 });
    s_aglList.emplace_back("acircumflex"_n, AglMapping{ (AglType)11, 1, 0x00E2 });
    s_aglList.emplace_back("acircumflexacute"_n, AglMapping{ (AglType)1, 1, 0x1EA5 });
    s_aglList.emplace_back("acircumflexdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EAD });
    s_aglList.emplace_back("acircumflexgrave"_n, AglMapping{ (AglType)1, 1, 0x1EA7 });
    s_aglList.emplace_back("acircumflexhookabove"_n, AglMapping{ (AglType)1, 1, 0x1EA9 });
    s_aglList.emplace_back("acircumflextilde"_n, AglMapping{ (AglType)1, 1, 0x1EAB });
    s_aglList.emplace_back("acute"_n, AglMapping{ (AglType)11, 1, 0x00B4 });
    s_aglList.emplace_back("acutebelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0317 });
    s_aglList.emplace_back("acutecmb"_n, AglMapping{ (AglType)1, 1, 0x0301 });
    s_aglList.emplace_back("acutecomb"_n, AglMapping{ (AglType)3, 1, 0x0301 });
    s_aglList.emplace_back("acutedeva"_n, AglMapping{ (AglType)1, 1, 0x0954 });
    s_aglList.emplace_back("acutelowmod"_n, AglMapping{ (AglType)1, 1, 0x02CF });
    s_aglList.emplace_back("acutetonecmb"_n, AglMapping{ (AglType)1, 1, 0x0341 });
    s_aglList.emplace_back("acyrillic"_n, AglMapping{ (AglType)1, 1, 0x0430 });
    s_aglList.emplace_back("adblgrave"_n, AglMapping{ (AglType)1, 1, 0x0201 });
    s_aglList.emplace_back("addakgurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A71 });
    s_aglList.emplace_back("adeva"_n, AglMapping{ (AglType)1, 1, 0x0905 });
    s_aglList.emplace_back("adieresis"_n, AglMapping{ (AglType)11, 1, 0x00E4 });
    s_aglList.emplace_back("adieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04D3 });
    s_aglList.emplace_back("adieresismacron"_n, AglMapping{ (AglType)1, 1, 0x01DF });
    s_aglList.emplace_back("adotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EA1 });
    s_aglList.emplace_back("adotmacron"_n, AglMapping{ (AglType)1, 1, 0x01E1 });
    s_aglList.emplace_back("ae"_n, AglMapping{ (AglType)11, 1, 0x00E6 });
    s_aglList.emplace_back("aeacute"_n, AglMapping{ (AglType)3, 1, 0x01FD });
    s_aglList.emplace_back("aekorean"_n, AglMapping{ (AglType)1, 1, 0x3150 });
    s_aglList.emplace_back("aemacron"_n, AglMapping{ (AglType)1, 1, 0x01E3 });
    s_aglList.emplace_back("afii00208"_n, AglMapping{ (AglType)1, 1, 0x2015 });
    s_aglList.emplace_back("afii08941"_n, AglMapping{ (AglType)1, 1, 0x20A4 });
    s_aglList.emplace_back("afii10017"_n, AglMapping{ (AglType)1, 1, 0x0410 });
    s_aglList.emplace_back("afii10018"_n, AglMapping{ (AglType)1, 1, 0x0411 });
    s_aglList.emplace_back("afii10019"_n, AglMapping{ (AglType)1, 1, 0x0412 });
    s_aglList.emplace_back("afii10020"_n, AglMapping{ (AglType)1, 1, 0x0413 });
    s_aglList.emplace_back("afii10021"_n, AglMapping{ (AglType)1, 1, 0x0414 });
    s_aglList.emplace_back("afii10022"_n, AglMapping{ (AglType)1, 1, 0x0415 });
    s_aglList.emplace_back("afii10023"_n, AglMapping{ (AglType)1, 1, 0x0401 });
    s_aglList.emplace_back("afii10024"_n, AglMapping{ (AglType)1, 1, 0x0416 });
    s_aglList.emplace_back("afii10025"_n, AglMapping{ (AglType)1, 1, 0x0417 });
    s_aglList.emplace_back("afii10026"_n, AglMapping{ (AglType)1, 1, 0x0418 });
    s_aglList.emplace_back("afii10027"_n, AglMapping{ (AglType)1, 1, 0x0419 });
    s_aglList.emplace_back("afii10028"_n, AglMapping{ (AglType)1, 1, 0x041A });
    s_aglList.emplace_back("afii10029"_n, AglMapping{ (AglType)1, 1, 0x041B });
    s_aglList.emplace_back("afii10030"_n, AglMapping{ (AglType)1, 1, 0x041C });
    s_aglList.emplace_back("afii10031"_n, AglMapping{ (AglType)1, 1, 0x041D });
    s_aglList.emplace_back("afii10032"_n, AglMapping{ (AglType)1, 1, 0x041E });
    s_aglList.emplace_back("afii10033"_n, AglMapping{ (AglType)1, 1, 0x041F });
    s_aglList.emplace_back("afii10034"_n, AglMapping{ (AglType)1, 1, 0x0420 });
    s_aglList.emplace_back("afii10035"_n, AglMapping{ (AglType)1, 1, 0x0421 });
    s_aglList.emplace_back("afii10036"_n, AglMapping{ (AglType)1, 1, 0x0422 });
    s_aglList.emplace_back("afii10037"_n, AglMapping{ (AglType)1, 1, 0x0423 });
    s_aglList.emplace_back("afii10038"_n, AglMapping{ (AglType)1, 1, 0x0424 });
    s_aglList.emplace_back("afii10039"_n, AglMapping{ (AglType)1, 1, 0x0425 });
    s_aglList.emplace_back("afii10040"_n, AglMapping{ (AglType)1, 1, 0x0426 });
    s_aglList.emplace_back("afii10041"_n, AglMapping{ (AglType)1, 1, 0x0427 });
    s_aglList.emplace_back("afii10042"_n, AglMapping{ (AglType)1, 1, 0x0428 });
    s_aglList.emplace_back("afii10043"_n, AglMapping{ (AglType)1, 1, 0x0429 });
    s_aglList.emplace_back("afii10044"_n, AglMapping{ (AglType)1, 1, 0x042A });
    s_aglList.emplace_back("afii10045"_n, AglMapping{ (AglType)1, 1, 0x042B });
    s_aglList.emplace_back("afii10046"_n, AglMapping{ (AglType)1, 1, 0x042C });
    s_aglList.emplace_back("afii10047"_n, AglMapping{ (AglType)1, 1, 0x042D });
    s_aglList.emplace_back("afii10048"_n, AglMapping{ (AglType)1, 1, 0x042E });
    s_aglList.emplace_back("afii10049"_n, AglMapping{ (AglType)1, 1, 0x042F });
    s_aglList.emplace_back("afii10050"_n, AglMapping{ (AglType)1, 1, 0x0490 });
    s_aglList.emplace_back("afii10051"_n, AglMapping{ (AglType)1, 1, 0x0402 });
    s_aglList.emplace_back("afii10052"_n, AglMapping{ (AglType)1, 1, 0x0403 });
    s_aglList.emplace_back("afii10053"_n, AglMapping{ (AglType)1, 1, 0x0404 });
    s_aglList.emplace_back("afii10054"_n, AglMapping{ (AglType)1, 1, 0x0405 });
    s_aglList.emplace_back("afii10055"_n, AglMapping{ (AglType)1, 1, 0x0406 });
    s_aglList.emplace_back("afii10056"_n, AglMapping{ (AglType)1, 1, 0x0407 });
    s_aglList.emplace_back("afii10057"_n, AglMapping{ (AglType)1, 1, 0x0408 });
    s_aglList.emplace_back("afii10058"_n, AglMapping{ (AglType)1, 1, 0x0409 });
    s_aglList.emplace_back("afii10059"_n, AglMapping{ (AglType)1, 1, 0x040A });
    s_aglList.emplace_back("afii10060"_n, AglMapping{ (AglType)1, 1, 0x040B });
    s_aglList.emplace_back("afii10061"_n, AglMapping{ (AglType)1, 1, 0x040C });
    s_aglList.emplace_back("afii10062"_n, AglMapping{ (AglType)1, 1, 0x040E });
    s_aglList.emplace_back("afii10063"_n, AglMapping{ (AglType)1, 1, 0xF6C4 });
    s_aglList.emplace_back("afii10064"_n, AglMapping{ (AglType)1, 1, 0xF6C5 });
    s_aglList.emplace_back("afii10065"_n, AglMapping{ (AglType)1, 1, 0x0430 });
    s_aglList.emplace_back("afii10066"_n, AglMapping{ (AglType)1, 1, 0x0431 });
    s_aglList.emplace_back("afii10067"_n, AglMapping{ (AglType)1, 1, 0x0432 });
    s_aglList.emplace_back("afii10068"_n, AglMapping{ (AglType)1, 1, 0x0433 });
    s_aglList.emplace_back("afii10069"_n, AglMapping{ (AglType)1, 1, 0x0434 });
    s_aglList.emplace_back("afii10070"_n, AglMapping{ (AglType)1, 1, 0x0435 });
    s_aglList.emplace_back("afii10071"_n, AglMapping{ (AglType)1, 1, 0x0451 });
    s_aglList.emplace_back("afii10072"_n, AglMapping{ (AglType)1, 1, 0x0436 });
    s_aglList.emplace_back("afii10073"_n, AglMapping{ (AglType)1, 1, 0x0437 });
    s_aglList.emplace_back("afii10074"_n, AglMapping{ (AglType)1, 1, 0x0438 });
    s_aglList.emplace_back("afii10075"_n, AglMapping{ (AglType)1, 1, 0x0439 });
    s_aglList.emplace_back("afii10076"_n, AglMapping{ (AglType)1, 1, 0x043A });
    s_aglList.emplace_back("afii10077"_n, AglMapping{ (AglType)1, 1, 0x043B });
    s_aglList.emplace_back("afii10078"_n, AglMapping{ (AglType)1, 1, 0x043C });
    s_aglList.emplace_back("afii10079"_n, AglMapping{ (AglType)1, 1, 0x043D });
    s_aglList.emplace_back("afii10080"_n, AglMapping{ (AglType)1, 1, 0x043E });
    s_aglList.emplace_back("afii10081"_n, AglMapping{ (AglType)1, 1, 0x043F });
    s_aglList.emplace_back("afii10082"_n, AglMapping{ (AglType)1, 1, 0x0440 });
    s_aglList.emplace_back("afii10083"_n, AglMapping{ (AglType)1, 1, 0x0441 });
    s_aglList.emplace_back("afii10084"_n, AglMapping{ (AglType)1, 1, 0x0442 });
    s_aglList.emplace_back("afii10085"_n, AglMapping{ (AglType)1, 1, 0x0443 });
    s_aglList.emplace_back("afii10086"_n, AglMapping{ (AglType)1, 1, 0x0444 });
    s_aglList.emplace_back("afii10087"_n, AglMapping{ (AglType)1, 1, 0x0445 });
    s_aglList.emplace_back("afii10088"_n, AglMapping{ (AglType)1, 1, 0x0446 });
    s_aglList.emplace_back("afii10089"_n, AglMapping{ (AglType)1, 1, 0x0447 });
    s_aglList.emplace_back("afii10090"_n, AglMapping{ (AglType)1, 1, 0x0448 });
    s_aglList.emplace_back("afii10091"_n, AglMapping{ (AglType)1, 1, 0x0449 });
    s_aglList.emplace_back("afii10092"_n, AglMapping{ (AglType)1, 1, 0x044A });
    s_aglList.emplace_back("afii10093"_n, AglMapping{ (AglType)1, 1, 0x044B });
    s_aglList.emplace_back("afii10094"_n, AglMapping{ (AglType)1, 1, 0x044C });
    s_aglList.emplace_back("afii10095"_n, AglMapping{ (AglType)1, 1, 0x044D });
    s_aglList.emplace_back("afii10096"_n, AglMapping{ (AglType)1, 1, 0x044E });
    s_aglList.emplace_back("afii10097"_n, AglMapping{ (AglType)1, 1, 0x044F });
    s_aglList.emplace_back("afii10098"_n, AglMapping{ (AglType)1, 1, 0x0491 });
    s_aglList.emplace_back("afii10099"_n, AglMapping{ (AglType)1, 1, 0x0452 });
    s_aglList.emplace_back("afii10100"_n, AglMapping{ (AglType)1, 1, 0x0453 });
    s_aglList.emplace_back("afii10101"_n, AglMapping{ (AglType)1, 1, 0x0454 });
    s_aglList.emplace_back("afii10102"_n, AglMapping{ (AglType)1, 1, 0x0455 });
    s_aglList.emplace_back("afii10103"_n, AglMapping{ (AglType)1, 1, 0x0456 });
    s_aglList.emplace_back("afii10104"_n, AglMapping{ (AglType)1, 1, 0x0457 });
    s_aglList.emplace_back("afii10105"_n, AglMapping{ (AglType)1, 1, 0x0458 });
    s_aglList.emplace_back("afii10106"_n, AglMapping{ (AglType)1, 1, 0x0459 });
    s_aglList.emplace_back("afii10107"_n, AglMapping{ (AglType)1, 1, 0x045A });
    s_aglList.emplace_back("afii10108"_n, AglMapping{ (AglType)1, 1, 0x045B });
    s_aglList.emplace_back("afii10109"_n, AglMapping{ (AglType)1, 1, 0x045C });
    s_aglList.emplace_back("afii10110"_n, AglMapping{ (AglType)1, 1, 0x045E });
    s_aglList.emplace_back("afii10145"_n, AglMapping{ (AglType)1, 1, 0x040F });
    s_aglList.emplace_back("afii10146"_n, AglMapping{ (AglType)1, 1, 0x0462 });
    s_aglList.emplace_back("afii10147"_n, AglMapping{ (AglType)1, 1, 0x0472 });
    s_aglList.emplace_back("afii10148"_n, AglMapping{ (AglType)1, 1, 0x0474 });
    s_aglList.emplace_back("afii10192"_n, AglMapping{ (AglType)1, 1, 0xF6C6 });
    s_aglList.emplace_back("afii10193"_n, AglMapping{ (AglType)1, 1, 0x045F });
    s_aglList.emplace_back("afii10194"_n, AglMapping{ (AglType)1, 1, 0x0463 });
    s_aglList.emplace_back("afii10195"_n, AglMapping{ (AglType)1, 1, 0x0473 });
    s_aglList.emplace_back("afii10196"_n, AglMapping{ (AglType)1, 1, 0x0475 });
    s_aglList.emplace_back("afii10831"_n, AglMapping{ (AglType)1, 1, 0xF6C7 });
    s_aglList.emplace_back("afii10832"_n, AglMapping{ (AglType)1, 1, 0xF6C8 });
    s_aglList.emplace_back("afii10846"_n, AglMapping{ (AglType)1, 1, 0x04D9 });
    s_aglList.emplace_back("afii299"_n, AglMapping{ (AglType)1, 1, 0x200E });
    s_aglList.emplace_back("afii300"_n, AglMapping{ (AglType)1, 1, 0x200F });
    s_aglList.emplace_back("afii301"_n, AglMapping{ (AglType)1, 1, 0x200D });
    s_aglList.emplace_back("afii57381"_n, AglMapping{ (AglType)1, 1, 0x066A });
    s_aglList.emplace_back("afii57388"_n, AglMapping{ (AglType)1, 1, 0x060C });
    s_aglList.emplace_back("afii57392"_n, AglMapping{ (AglType)1, 1, 0x0660 });
    s_aglList.emplace_back("afii57393"_n, AglMapping{ (AglType)1, 1, 0x0661 });
    s_aglList.emplace_back("afii57394"_n, AglMapping{ (AglType)1, 1, 0x0662 });
    s_aglList.emplace_back("afii57395"_n, AglMapping{ (AglType)1, 1, 0x0663 });
    s_aglList.emplace_back("afii57396"_n, AglMapping{ (AglType)1, 1, 0x0664 });
    s_aglList.emplace_back("afii57397"_n, AglMapping{ (AglType)1, 1, 0x0665 });
    s_aglList.emplace_back("afii57398"_n, AglMapping{ (AglType)1, 1, 0x0666 });
    s_aglList.emplace_back("afii57399"_n, AglMapping{ (AglType)1, 1, 0x0667 });
    s_aglList.emplace_back("afii57400"_n, AglMapping{ (AglType)1, 1, 0x0668 });
    s_aglList.emplace_back("afii57401"_n, AglMapping{ (AglType)1, 1, 0x0669 });
    s_aglList.emplace_back("afii57403"_n, AglMapping{ (AglType)1, 1, 0x061B });
    s_aglList.emplace_back("afii57407"_n, AglMapping{ (AglType)1, 1, 0x061F });
    s_aglList.emplace_back("afii57409"_n, AglMapping{ (AglType)1, 1, 0x0621 });
    s_aglList.emplace_back("afii57410"_n, AglMapping{ (AglType)1, 1, 0x0622 });
    s_aglList.emplace_back("afii57411"_n, AglMapping{ (AglType)1, 1, 0x0623 });
    s_aglList.emplace_back("afii57412"_n, AglMapping{ (AglType)1, 1, 0x0624 });
    s_aglList.emplace_back("afii57413"_n, AglMapping{ (AglType)1, 1, 0x0625 });
    s_aglList.emplace_back("afii57414"_n, AglMapping{ (AglType)1, 1, 0x0626 });
    s_aglList.emplace_back("afii57415"_n, AglMapping{ (AglType)1, 1, 0x0627 });
    s_aglList.emplace_back("afii57416"_n, AglMapping{ (AglType)1, 1, 0x0628 });
    s_aglList.emplace_back("afii57417"_n, AglMapping{ (AglType)1, 1, 0x0629 });
    s_aglList.emplace_back("afii57418"_n, AglMapping{ (AglType)1, 1, 0x062A });
    s_aglList.emplace_back("afii57419"_n, AglMapping{ (AglType)1, 1, 0x062B });
    s_aglList.emplace_back("afii57420"_n, AglMapping{ (AglType)1, 1, 0x062C });
    s_aglList.emplace_back("afii57421"_n, AglMapping{ (AglType)1, 1, 0x062D });
    s_aglList.emplace_back("afii57422"_n, AglMapping{ (AglType)1, 1, 0x062E });
    s_aglList.emplace_back("afii57423"_n, AglMapping{ (AglType)1, 1, 0x062F });
    s_aglList.emplace_back("afii57424"_n, AglMapping{ (AglType)1, 1, 0x0630 });
    s_aglList.emplace_back("afii57425"_n, AglMapping{ (AglType)1, 1, 0x0631 });
    s_aglList.emplace_back("afii57426"_n, AglMapping{ (AglType)1, 1, 0x0632 });
    s_aglList.emplace_back("afii57427"_n, AglMapping{ (AglType)1, 1, 0x0633 });
    s_aglList.emplace_back("afii57428"_n, AglMapping{ (AglType)1, 1, 0x0634 });
    s_aglList.emplace_back("afii57429"_n, AglMapping{ (AglType)1, 1, 0x0635 });
    s_aglList.emplace_back("afii57430"_n, AglMapping{ (AglType)1, 1, 0x0636 });
    s_aglList.emplace_back("afii57431"_n, AglMapping{ (AglType)1, 1, 0x0637 });
    s_aglList.emplace_back("afii57432"_n, AglMapping{ (AglType)1, 1, 0x0638 });
    s_aglList.emplace_back("afii57433"_n, AglMapping{ (AglType)1, 1, 0x0639 });
    s_aglList.emplace_back("afii57434"_n, AglMapping{ (AglType)1, 1, 0x063A });
    s_aglList.emplace_back("afii57440"_n, AglMapping{ (AglType)1, 1, 0x0640 });
    s_aglList.emplace_back("afii57441"_n, AglMapping{ (AglType)1, 1, 0x0641 });
    s_aglList.emplace_back("afii57442"_n, AglMapping{ (AglType)1, 1, 0x0642 });
    s_aglList.emplace_back("afii57443"_n, AglMapping{ (AglType)1, 1, 0x0643 });
    s_aglList.emplace_back("afii57444"_n, AglMapping{ (AglType)1, 1, 0x0644 });
    s_aglList.emplace_back("afii57445"_n, AglMapping{ (AglType)1, 1, 0x0645 });
    s_aglList.emplace_back("afii57446"_n, AglMapping{ (AglType)1, 1, 0x0646 });
    s_aglList.emplace_back("afii57448"_n, AglMapping{ (AglType)1, 1, 0x0648 });
    s_aglList.emplace_back("afii57449"_n, AglMapping{ (AglType)1, 1, 0x0649 });
    s_aglList.emplace_back("afii57450"_n, AglMapping{ (AglType)1, 1, 0x064A });
    s_aglList.emplace_back("afii57451"_n, AglMapping{ (AglType)1, 1, 0x064B });
    s_aglList.emplace_back("afii57452"_n, AglMapping{ (AglType)1, 1, 0x064C });
    s_aglList.emplace_back("afii57453"_n, AglMapping{ (AglType)1, 1, 0x064D });
    s_aglList.emplace_back("afii57454"_n, AglMapping{ (AglType)1, 1, 0x064E });
    s_aglList.emplace_back("afii57455"_n, AglMapping{ (AglType)1, 1, 0x064F });
    s_aglList.emplace_back("afii57456"_n, AglMapping{ (AglType)1, 1, 0x0650 });
    s_aglList.emplace_back("afii57457"_n, AglMapping{ (AglType)1, 1, 0x0651 });
    s_aglList.emplace_back("afii57458"_n, AglMapping{ (AglType)1, 1, 0x0652 });
    s_aglList.emplace_back("afii57470"_n, AglMapping{ (AglType)1, 1, 0x0647 });
    s_aglList.emplace_back("afii57505"_n, AglMapping{ (AglType)1, 1, 0x06A4 });
    s_aglList.emplace_back("afii57506"_n, AglMapping{ (AglType)1, 1, 0x067E });
    s_aglList.emplace_back("afii57507"_n, AglMapping{ (AglType)1, 1, 0x0686 });
    s_aglList.emplace_back("afii57508"_n, AglMapping{ (AglType)1, 1, 0x0698 });
    s_aglList.emplace_back("afii57509"_n, AglMapping{ (AglType)1, 1, 0x06AF });
    s_aglList.emplace_back("afii57511"_n, AglMapping{ (AglType)1, 1, 0x0679 });
    s_aglList.emplace_back("afii57512"_n, AglMapping{ (AglType)1, 1, 0x0688 });
    s_aglList.emplace_back("afii57513"_n, AglMapping{ (AglType)1, 1, 0x0691 });
    s_aglList.emplace_back("afii57514"_n, AglMapping{ (AglType)1, 1, 0x06BA });
    s_aglList.emplace_back("afii57519"_n, AglMapping{ (AglType)1, 1, 0x06D2 });
    s_aglList.emplace_back("afii57534"_n, AglMapping{ (AglType)1, 1, 0x06D5 });
    s_aglList.emplace_back("afii57636"_n, AglMapping{ (AglType)1, 1, 0x20AA });
    s_aglList.emplace_back("afii57645"_n, AglMapping{ (AglType)1, 1, 0x05BE });
    s_aglList.emplace_back("afii57658"_n, AglMapping{ (AglType)1, 1, 0x05C3 });
    s_aglList.emplace_back("afii57664"_n, AglMapping{ (AglType)1, 1, 0x05D0 });
    s_aglList.emplace_back("afii57665"_n, AglMapping{ (AglType)1, 1, 0x05D1 });
    s_aglList.emplace_back("afii57666"_n, AglMapping{ (AglType)1, 1, 0x05D2 });
    s_aglList.emplace_back("afii57667"_n, AglMapping{ (AglType)1, 1, 0x05D3 });
    s_aglList.emplace_back("afii57668"_n, AglMapping{ (AglType)1, 1, 0x05D4 });
    s_aglList.emplace_back("afii57669"_n, AglMapping{ (AglType)1, 1, 0x05D5 });
    s_aglList.emplace_back("afii57670"_n, AglMapping{ (AglType)1, 1, 0x05D6 });
    s_aglList.emplace_back("afii57671"_n, AglMapping{ (AglType)1, 1, 0x05D7 });
    s_aglList.emplace_back("afii57672"_n, AglMapping{ (AglType)1, 1, 0x05D8 });
    s_aglList.emplace_back("afii57673"_n, AglMapping{ (AglType)1, 1, 0x05D9 });
    s_aglList.emplace_back("afii57674"_n, AglMapping{ (AglType)1, 1, 0x05DA });
    s_aglList.emplace_back("afii57675"_n, AglMapping{ (AglType)1, 1, 0x05DB });
    s_aglList.emplace_back("afii57676"_n, AglMapping{ (AglType)1, 1, 0x05DC });
    s_aglList.emplace_back("afii57677"_n, AglMapping{ (AglType)1, 1, 0x05DD });
    s_aglList.emplace_back("afii57678"_n, AglMapping{ (AglType)1, 1, 0x05DE });
    s_aglList.emplace_back("afii57679"_n, AglMapping{ (AglType)1, 1, 0x05DF });
    s_aglList.emplace_back("afii57680"_n, AglMapping{ (AglType)1, 1, 0x05E0 });
    s_aglList.emplace_back("afii57681"_n, AglMapping{ (AglType)1, 1, 0x05E1 });
    s_aglList.emplace_back("afii57682"_n, AglMapping{ (AglType)1, 1, 0x05E2 });
    s_aglList.emplace_back("afii57683"_n, AglMapping{ (AglType)1, 1, 0x05E3 });
    s_aglList.emplace_back("afii57684"_n, AglMapping{ (AglType)1, 1, 0x05E4 });
    s_aglList.emplace_back("afii57685"_n, AglMapping{ (AglType)1, 1, 0x05E5 });
    s_aglList.emplace_back("afii57686"_n, AglMapping{ (AglType)1, 1, 0x05E6 });
    s_aglList.emplace_back("afii57687"_n, AglMapping{ (AglType)1, 1, 0x05E7 });
    s_aglList.emplace_back("afii57688"_n, AglMapping{ (AglType)1, 1, 0x05E8 });
    s_aglList.emplace_back("afii57689"_n, AglMapping{ (AglType)1, 1, 0x05E9 });
    s_aglList.emplace_back("afii57690"_n, AglMapping{ (AglType)1, 1, 0x05EA });
    s_aglList.emplace_back("afii57694"_n, AglMapping{ (AglType)1, 1, 0xFB2A });
    s_aglList.emplace_back("afii57695"_n, AglMapping{ (AglType)1, 1, 0xFB2B });
    s_aglList.emplace_back("afii57700"_n, AglMapping{ (AglType)1, 1, 0xFB4B });
    s_aglList.emplace_back("afii57705"_n, AglMapping{ (AglType)1, 1, 0xFB1F });
    s_aglList.emplace_back("afii57716"_n, AglMapping{ (AglType)1, 1, 0x05F0 });
    s_aglList.emplace_back("afii57717"_n, AglMapping{ (AglType)1, 1, 0x05F1 });
    s_aglList.emplace_back("afii57718"_n, AglMapping{ (AglType)1, 1, 0x05F2 });
    s_aglList.emplace_back("afii57723"_n, AglMapping{ (AglType)1, 1, 0xFB35 });
    s_aglList.emplace_back("afii57793"_n, AglMapping{ (AglType)1, 1, 0x05B4 });
    s_aglList.emplace_back("afii57794"_n, AglMapping{ (AglType)1, 1, 0x05B5 });
    s_aglList.emplace_back("afii57795"_n, AglMapping{ (AglType)1, 1, 0x05B6 });
    s_aglList.emplace_back("afii57796"_n, AglMapping{ (AglType)1, 1, 0x05BB });
    s_aglList.emplace_back("afii57797"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("afii57798"_n, AglMapping{ (AglType)1, 1, 0x05B7 });
    s_aglList.emplace_back("afii57799"_n, AglMapping{ (AglType)1, 1, 0x05B0 });
    s_aglList.emplace_back("afii57800"_n, AglMapping{ (AglType)1, 1, 0x05B2 });
    s_aglList.emplace_back("afii57801"_n, AglMapping{ (AglType)1, 1, 0x05B1 });
    s_aglList.emplace_back("afii57802"_n, AglMapping{ (AglType)1, 1, 0x05B3 });
    s_aglList.emplace_back("afii57803"_n, AglMapping{ (AglType)1, 1, 0x05C2 });
    s_aglList.emplace_back("afii57804"_n, AglMapping{ (AglType)1, 1, 0x05C1 });
    s_aglList.emplace_back("afii57806"_n, AglMapping{ (AglType)1, 1, 0x05B9 });
    s_aglList.emplace_back("afii57807"_n, AglMapping{ (AglType)1, 1, 0x05BC });
    s_aglList.emplace_back("afii57839"_n, AglMapping{ (AglType)1, 1, 0x05BD });
    s_aglList.emplace_back("afii57841"_n, AglMapping{ (AglType)1, 1, 0x05BF });
    s_aglList.emplace_back("afii57842"_n, AglMapping{ (AglType)1, 1, 0x05C0 });
    s_aglList.emplace_back("afii57929"_n, AglMapping{ (AglType)1, 1, 0x02BC });
    s_aglList.emplace_back("afii61248"_n, AglMapping{ (AglType)1, 1, 0x2105 });
    s_aglList.emplace_back("afii61289"_n, AglMapping{ (AglType)1, 1, 0x2113 });
    s_aglList.emplace_back("afii61352"_n, AglMapping{ (AglType)1, 1, 0x2116 });
    s_aglList.emplace_back("afii61573"_n, AglMapping{ (AglType)1, 1, 0x202C });
    s_aglList.emplace_back("afii61574"_n, AglMapping{ (AglType)1, 1, 0x202D });
    s_aglList.emplace_back("afii61575"_n, AglMapping{ (AglType)1, 1, 0x202E });
    s_aglList.emplace_back("afii61664"_n, AglMapping{ (AglType)1, 1, 0x200C });
    s_aglList.emplace_back("afii63167"_n, AglMapping{ (AglType)1, 1, 0x066D });
    s_aglList.emplace_back("afii64937"_n, AglMapping{ (AglType)1, 1, 0x02BD });
    s_aglList.emplace_back("agrave"_n, AglMapping{ (AglType)11, 1, 0x00E0 });
    s_aglList.emplace_back("agujarati"_n, AglMapping{ (AglType)1, 1, 0x0A85 });
    s_aglList.emplace_back("agurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A05 });
    s_aglList.emplace_back("ahiragana"_n, AglMapping{ (AglType)1, 1, 0x3042 });
    s_aglList.emplace_back("ahookabove"_n, AglMapping{ (AglType)1, 1, 0x1EA3 });
    s_aglList.emplace_back("aibengali"_n, AglMapping{ (AglType)1, 1, 0x0990 });
    s_aglList.emplace_back("aibopomofo"_n, AglMapping{ (AglType)1, 1, 0x311E });
    s_aglList.emplace_back("aideva"_n, AglMapping{ (AglType)1, 1, 0x0910 });
    s_aglList.emplace_back("aiecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04D5 });
    s_aglList.emplace_back("aigujarati"_n, AglMapping{ (AglType)1, 1, 0x0A90 });
    s_aglList.emplace_back("aigurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A10 });
    s_aglList.emplace_back("aimatragurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A48 });
    s_aglList.emplace_back("ainarabic"_n, AglMapping{ (AglType)1, 1, 0x0639 });
    s_aglList.emplace_back("ainfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFECA });
    s_aglList.emplace_back("aininitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFECB });
    s_aglList.emplace_back("ainmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFECC });
    s_aglList.emplace_back("ainvertedbreve"_n, AglMapping{ (AglType)1, 1, 0x0203 });
    s_aglList.emplace_back("aivowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09C8 });
    s_aglList.emplace_back("aivowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0948 });
    s_aglList.emplace_back("aivowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0AC8 });
    s_aglList.emplace_back("akatakana"_n, AglMapping{ (AglType)1, 1, 0x30A2 });
    s_aglList.emplace_back("akatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF71 });
    s_aglList.emplace_back("akorean"_n, AglMapping{ (AglType)1, 1, 0x314F });
    s_aglList.emplace_back("alef"_n, AglMapping{ (AglType)1, 1, 0x05D0 });
    s_aglList.emplace_back("alefarabic"_n, AglMapping{ (AglType)1, 1, 0x0627 });
    s_aglList.emplace_back("alefdageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB30 });
    s_aglList.emplace_back("aleffinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFE8E });
    s_aglList.emplace_back("alefhamzaabovearabic"_n, AglMapping{ (AglType)1, 1, 0x0623 });
    s_aglList.emplace_back("alefhamzaabovefinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFE84 });
    s_aglList.emplace_back("alefhamzabelowarabic"_n, AglMapping{ (AglType)1, 1, 0x0625 });
    s_aglList.emplace_back("alefhamzabelowfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFE88 });
    s_aglList.emplace_back("alefhebrew"_n, AglMapping{ (AglType)1, 1, 0x05D0 });
    s_aglList.emplace_back("aleflamedhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB4F });
    s_aglList.emplace_back("alefmaddaabovearabic"_n, AglMapping{ (AglType)1, 1, 0x0622 });
    s_aglList.emplace_back("alefmaddaabovefinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFE82 });
    s_aglList.emplace_back("alefmaksuraarabic"_n, AglMapping{ (AglType)1, 1, 0x0649 });
    s_aglList.emplace_back("alefmaksurafinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEF0 });
    s_aglList.emplace_back("alefmaksurainitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEF3 });
    s_aglList.emplace_back("alefmaksuramedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEF4 });
    s_aglList.emplace_back("alefpatahhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB2E });
    s_aglList.emplace_back("alefqamatshebrew"_n, AglMapping{ (AglType)1, 1, 0xFB2F });
    s_aglList.emplace_back("aleph"_n, AglMapping{ (AglType)3, 1, 0x2135 });
    s_aglList.emplace_back("allequal"_n, AglMapping{ (AglType)1, 1, 0x224C });
    s_aglList.emplace_back("alpha"_n, AglMapping{ (AglType)3, 1, 0x03B1 });
    s_aglList.emplace_back("alphatonos"_n, AglMapping{ (AglType)3, 1, 0x03AC });
    s_aglList.emplace_back("amacron"_n, AglMapping{ (AglType)3, 1, 0x0101 });
    s_aglList.emplace_back("amonospace"_n, AglMapping{ (AglType)1, 1, 0xFF41 });
    s_aglList.emplace_back("ampersand"_n, AglMapping{ (AglType)11, 1, 0x0026 });
    s_aglList.emplace_back("ampersandmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF06 });
    s_aglList.emplace_back("ampersandsmall"_n, AglMapping{ (AglType)1, 1, 0xF726 });
    s_aglList.emplace_back("amsquare"_n, AglMapping{ (AglType)1, 1, 0x33C2 });
    s_aglList.emplace_back("anbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3122 });
    s_aglList.emplace_back("angbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3124 });
    s_aglList.emplace_back("angkhankhuthai"_n, AglMapping{ (AglType)1, 1, 0x0E5A });
    s_aglList.emplace_back("angle"_n, AglMapping{ (AglType)3, 1, 0x2220 });
    s_aglList.emplace_back("anglebracketleft"_n, AglMapping{ (AglType)1, 1, 0x3008 });
    s_aglList.emplace_back("anglebracketleftvertical"_n, AglMapping{ (AglType)1, 1, 0xFE3F });
    s_aglList.emplace_back("anglebracketright"_n, AglMapping{ (AglType)1, 1, 0x3009 });
    s_aglList.emplace_back("anglebracketrightvertical"_n, AglMapping{ (AglType)1, 1, 0xFE40 });
    s_aglList.emplace_back("angleleft"_n, AglMapping{ (AglType)3, 1, 0x2329 });
    s_aglList.emplace_back("angleright"_n, AglMapping{ (AglType)3, 1, 0x232A });
    s_aglList.emplace_back("angstrom"_n, AglMapping{ (AglType)1, 1, 0x212B });
    s_aglList.emplace_back("anoteleia"_n, AglMapping{ (AglType)3, 1, 0x0387 });
    s_aglList.emplace_back("anudattadeva"_n, AglMapping{ (AglType)1, 1, 0x0952 });
    s_aglList.emplace_back("anusvarabengali"_n, AglMapping{ (AglType)1, 1, 0x0982 });
    s_aglList.emplace_back("anusvaradeva"_n, AglMapping{ (AglType)1, 1, 0x0902 });
    s_aglList.emplace_back("anusvaragujarati"_n, AglMapping{ (AglType)1, 1, 0x0A82 });
    s_aglList.emplace_back("aogonek"_n, AglMapping{ (AglType)3, 1, 0x0105 });
    s_aglList.emplace_back("apaatosquare"_n, AglMapping{ (AglType)1, 1, 0x3300 });
    s_aglList.emplace_back("aparen"_n, AglMapping{ (AglType)1, 1, 0x249C });
    s_aglList.emplace_back("apostrophearmenian"_n, AglMapping{ (AglType)1, 1, 0x055A });
    s_aglList.emplace_back("apostrophemod"_n, AglMapping{ (AglType)1, 1, 0x02BC });
    s_aglList.emplace_back("apple"_n, AglMapping{ (AglType)1, 1, 0xF8FF });
    s_aglList.emplace_back("approaches"_n, AglMapping{ (AglType)1, 1, 0x2250 });
    s_aglList.emplace_back("approxequal"_n, AglMapping{ (AglType)3, 1, 0x2248 });
    s_aglList.emplace_back("approxequalorimage"_n, AglMapping{ (AglType)1, 1, 0x2252 });
    s_aglList.emplace_back("approximatelyequal"_n, AglMapping{ (AglType)1, 1, 0x2245 });
    s_aglList.emplace_back("araeaekorean"_n, AglMapping{ (AglType)1, 1, 0x318E });
    s_aglList.emplace_back("araeakorean"_n, AglMapping{ (AglType)1, 1, 0x318D });
    s_aglList.emplace_back("arc"_n, AglMapping{ (AglType)1, 1, 0x2312 });
    s_aglList.emplace_back("arighthalfring"_n, AglMapping{ (AglType)1, 1, 0x1E9A });
    s_aglList.emplace_back("aring"_n, AglMapping{ (AglType)11, 1, 0x00E5 });
    s_aglList.emplace_back("aringacute"_n, AglMapping{ (AglType)3, 1, 0x01FB });
    s_aglList.emplace_back("aringbelow"_n, AglMapping{ (AglType)1, 1, 0x1E01 });
    s_aglList.emplace_back("arrowboth"_n, AglMapping{ (AglType)3, 1, 0x2194 });
    s_aglList.emplace_back("arrowdashdown"_n, AglMapping{ (AglType)1, 1, 0x21E3 });
    s_aglList.emplace_back("arrowdashleft"_n, AglMapping{ (AglType)1, 1, 0x21E0 });
    s_aglList.emplace_back("arrowdashright"_n, AglMapping{ (AglType)1, 1, 0x21E2 });
    s_aglList.emplace_back("arrowdashup"_n, AglMapping{ (AglType)1, 1, 0x21E1 });
    s_aglList.emplace_back("arrowdblboth"_n, AglMapping{ (AglType)3, 1, 0x21D4 });
    s_aglList.emplace_back("arrowdbldown"_n, AglMapping{ (AglType)3, 1, 0x21D3 });
    s_aglList.emplace_back("arrowdblleft"_n, AglMapping{ (AglType)3, 1, 0x21D0 });
    s_aglList.emplace_back("arrowdblright"_n, AglMapping{ (AglType)3, 1, 0x21D2 });
    s_aglList.emplace_back("arrowdblup"_n, AglMapping{ (AglType)3, 1, 0x21D1 });
    s_aglList.emplace_back("arrowdown"_n, AglMapping{ (AglType)3, 1, 0x2193 });
    s_aglList.emplace_back("arrowdownleft"_n, AglMapping{ (AglType)1, 1, 0x2199 });
    s_aglList.emplace_back("arrowdownright"_n, AglMapping{ (AglType)1, 1, 0x2198 });
    s_aglList.emplace_back("arrowdownwhite"_n, AglMapping{ (AglType)1, 1, 0x21E9 });
    s_aglList.emplace_back("arrowheaddownmod"_n, AglMapping{ (AglType)1, 1, 0x02C5 });
    s_aglList.emplace_back("arrowheadleftmod"_n, AglMapping{ (AglType)1, 1, 0x02C2 });
    s_aglList.emplace_back("arrowheadrightmod"_n, AglMapping{ (AglType)1, 1, 0x02C3 });
    s_aglList.emplace_back("arrowheadupmod"_n, AglMapping{ (AglType)1, 1, 0x02C4 });
    s_aglList.emplace_back("arrowhorizex"_n, AglMapping{ (AglType)1, 1, 0xF8E7 });
    s_aglList.emplace_back("arrowleft"_n, AglMapping{ (AglType)3, 1, 0x2190 });
    s_aglList.emplace_back("arrowleftdbl"_n, AglMapping{ (AglType)1, 1, 0x21D0 });
    s_aglList.emplace_back("arrowleftdblstroke"_n, AglMapping{ (AglType)1, 1, 0x21CD });
    s_aglList.emplace_back("arrowleftoverright"_n, AglMapping{ (AglType)1, 1, 0x21C6 });
    s_aglList.emplace_back("arrowleftwhite"_n, AglMapping{ (AglType)1, 1, 0x21E6 });
    s_aglList.emplace_back("arrowright"_n, AglMapping{ (AglType)3, 1, 0x2192 });
    s_aglList.emplace_back("arrowrightdblstroke"_n, AglMapping{ (AglType)1, 1, 0x21CF });
    s_aglList.emplace_back("arrowrightheavy"_n, AglMapping{ (AglType)1, 1, 0x279E });
    s_aglList.emplace_back("arrowrightoverleft"_n, AglMapping{ (AglType)1, 1, 0x21C4 });
    s_aglList.emplace_back("arrowrightwhite"_n, AglMapping{ (AglType)1, 1, 0x21E8 });
    s_aglList.emplace_back("arrowtableft"_n, AglMapping{ (AglType)1, 1, 0x21E4 });
    s_aglList.emplace_back("arrowtabright"_n, AglMapping{ (AglType)1, 1, 0x21E5 });
    s_aglList.emplace_back("arrowup"_n, AglMapping{ (AglType)3, 1, 0x2191 });
    s_aglList.emplace_back("arrowupdn"_n, AglMapping{ (AglType)3, 1, 0x2195 });
    s_aglList.emplace_back("arrowupdnbse"_n, AglMapping{ (AglType)3, 1, 0x21A8 });
    s_aglList.emplace_back("arrowupdownbase"_n, AglMapping{ (AglType)1, 1, 0x21A8 });
    s_aglList.emplace_back("arrowupleft"_n, AglMapping{ (AglType)1, 1, 0x2196 });
    s_aglList.emplace_back("arrowupleftofdown"_n, AglMapping{ (AglType)1, 1, 0x21C5 });
    s_aglList.emplace_back("arrowupright"_n, AglMapping{ (AglType)1, 1, 0x2197 });
    s_aglList.emplace_back("arrowupwhite"_n, AglMapping{ (AglType)1, 1, 0x21E7 });
    s_aglList.emplace_back("arrowvertex"_n, AglMapping{ (AglType)1, 1, 0xF8E6 });
    s_aglList.emplace_back("asciicircum"_n, AglMapping{ (AglType)11, 1, 0x005E });
    s_aglList.emplace_back("asciicircummonospace"_n, AglMapping{ (AglType)1, 1, 0xFF3E });
    s_aglList.emplace_back("asciitilde"_n, AglMapping{ (AglType)11, 1, 0x007E });
    s_aglList.emplace_back("asciitildemonospace"_n, AglMapping{ (AglType)1, 1, 0xFF5E });
    s_aglList.emplace_back("ascript"_n, AglMapping{ (AglType)1, 1, 0x0251 });
    s_aglList.emplace_back("ascriptturned"_n, AglMapping{ (AglType)1, 1, 0x0252 });
    s_aglList.emplace_back("asmallhiragana"_n, AglMapping{ (AglType)1, 1, 0x3041 });
    s_aglList.emplace_back("asmallkatakana"_n, AglMapping{ (AglType)1, 1, 0x30A1 });
    s_aglList.emplace_back("asmallkatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF67 });
    s_aglList.emplace_back("asterisk"_n, AglMapping{ (AglType)11, 1, 0x002A });
    s_aglList.emplace_back("asteriskaltonearabic"_n, AglMapping{ (AglType)1, 1, 0x066D });
    s_aglList.emplace_back("asteriskarabic"_n, AglMapping{ (AglType)1, 1, 0x066D });
    s_aglList.emplace_back("asteriskmath"_n, AglMapping{ (AglType)3, 1, 0x2217 });
    s_aglList.emplace_back("asteriskmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF0A });
    s_aglList.emplace_back("asterisksmall"_n, AglMapping{ (AglType)1, 1, 0xFE61 });
    s_aglList.emplace_back("asterism"_n, AglMapping{ (AglType)1, 1, 0x2042 });
    s_aglList.emplace_back("asuperior"_n, AglMapping{ (AglType)1, 1, 0xF6E9 });
    s_aglList.emplace_back("asymptoticallyequal"_n, AglMapping{ (AglType)1, 1, 0x2243 });
    s_aglList.emplace_back("at"_n, AglMapping{ (AglType)11, 1, 0x0040 });
    s_aglList.emplace_back("atilde"_n, AglMapping{ (AglType)11, 1, 0x00E3 });
    s_aglList.emplace_back("atmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF20 });
    s_aglList.emplace_back("atsmall"_n, AglMapping{ (AglType)1, 1, 0xFE6B });
    s_aglList.emplace_back("aturned"_n, AglMapping{ (AglType)1, 1, 0x0250 });
    s_aglList.emplace_back("aubengali"_n, AglMapping{ (AglType)1, 1, 0x0994 });
    s_aglList.emplace_back("aubopomofo"_n, AglMapping{ (AglType)1, 1, 0x3120 });
    s_aglList.emplace_back("audeva"_n, AglMapping{ (AglType)1, 1, 0x0914 });
    s_aglList.emplace_back("augujarati"_n, AglMapping{ (AglType)1, 1, 0x0A94 });
    s_aglList.emplace_back("augurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A14 });
    s_aglList.emplace_back("aulengthmarkbengali"_n, AglMapping{ (AglType)1, 1, 0x09D7 });
    s_aglList.emplace_back("aumatragurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A4C });
    s_aglList.emplace_back("auvowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09CC });
    s_aglList.emplace_back("auvowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x094C });
    s_aglList.emplace_back("auvowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0ACC });
    s_aglList.emplace_back("avagrahadeva"_n, AglMapping{ (AglType)1, 1, 0x093D });
    s_aglList.emplace_back("aybarmenian"_n, AglMapping{ (AglType)1, 1, 0x0561 });
    s_aglList.emplace_back("ayin"_n, AglMapping{ (AglType)1, 1, 0x05E2 });
    s_aglList.emplace_back("ayinaltonehebrew"_n, AglMapping{ (AglType)1, 1, 0xFB20 });
    s_aglList.emplace_back("ayinhebrew"_n, AglMapping{ (AglType)1, 1, 0x05E2 });
    s_aglList.emplace_back("b"_n, AglMapping{ (AglType)11, 1, 0x0062 });
    s_aglList.emplace_back("babengali"_n, AglMapping{ (AglType)1, 1, 0x09AC });
    s_aglList.emplace_back("backslash"_n, AglMapping{ (AglType)11, 1, 0x005C });
    s_aglList.emplace_back("backslashmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF3C });
    s_aglList.emplace_back("badeva"_n, AglMapping{ (AglType)1, 1, 0x092C });
    s_aglList.emplace_back("bagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AAC });
    s_aglList.emplace_back("bagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A2C });
    s_aglList.emplace_back("bahiragana"_n, AglMapping{ (AglType)1, 1, 0x3070 });
    s_aglList.emplace_back("bahtthai"_n, AglMapping{ (AglType)1, 1, 0x0E3F });
    s_aglList.emplace_back("bakatakana"_n, AglMapping{ (AglType)1, 1, 0x30D0 });
    s_aglList.emplace_back("bar"_n, AglMapping{ (AglType)11, 1, 0x007C });
    s_aglList.emplace_back("barmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF5C });
    s_aglList.emplace_back("bbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3105 });
    s_aglList.emplace_back("bcircle"_n, AglMapping{ (AglType)1, 1, 0x24D1 });
    s_aglList.emplace_back("bdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E03 });
    s_aglList.emplace_back("bdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E05 });
    s_aglList.emplace_back("beamedsixteenthnotes"_n, AglMapping{ (AglType)1, 1, 0x266C });
    s_aglList.emplace_back("because"_n, AglMapping{ (AglType)1, 1, 0x2235 });
    s_aglList.emplace_back("becyrillic"_n, AglMapping{ (AglType)1, 1, 0x0431 });
    s_aglList.emplace_back("beharabic"_n, AglMapping{ (AglType)1, 1, 0x0628 });
    s_aglList.emplace_back("behfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFE90 });
    s_aglList.emplace_back("behinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFE91 });
    s_aglList.emplace_back("behiragana"_n, AglMapping{ (AglType)1, 1, 0x3079 });
    s_aglList.emplace_back("behmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFE92 });
    s_aglList.emplace_back("behmeeminitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFC9F });
    s_aglList.emplace_back("behmeemisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFC08 });
    s_aglList.emplace_back("behnoonfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFC6D });
    s_aglList.emplace_back("bekatakana"_n, AglMapping{ (AglType)1, 1, 0x30D9 });
    s_aglList.emplace_back("benarmenian"_n, AglMapping{ (AglType)1, 1, 0x0562 });
    s_aglList.emplace_back("bet"_n, AglMapping{ (AglType)1, 1, 0x05D1 });
    s_aglList.emplace_back("beta"_n, AglMapping{ (AglType)3, 1, 0x03B2 });
    s_aglList.emplace_back("betasymbolgreek"_n, AglMapping{ (AglType)1, 1, 0x03D0 });
    s_aglList.emplace_back("betdagesh"_n, AglMapping{ (AglType)1, 1, 0xFB31 });
    s_aglList.emplace_back("betdageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB31 });
    s_aglList.emplace_back("bethebrew"_n, AglMapping{ (AglType)1, 1, 0x05D1 });
    s_aglList.emplace_back("betrafehebrew"_n, AglMapping{ (AglType)1, 1, 0xFB4C });
    s_aglList.emplace_back("bhabengali"_n, AglMapping{ (AglType)1, 1, 0x09AD });
    s_aglList.emplace_back("bhadeva"_n, AglMapping{ (AglType)1, 1, 0x092D });
    s_aglList.emplace_back("bhagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AAD });
    s_aglList.emplace_back("bhagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A2D });
    s_aglList.emplace_back("bhook"_n, AglMapping{ (AglType)1, 1, 0x0253 });
    s_aglList.emplace_back("bihiragana"_n, AglMapping{ (AglType)1, 1, 0x3073 });
    s_aglList.emplace_back("bikatakana"_n, AglMapping{ (AglType)1, 1, 0x30D3 });
    s_aglList.emplace_back("bilabialclick"_n, AglMapping{ (AglType)1, 1, 0x0298 });
    s_aglList.emplace_back("bindigurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A02 });
    s_aglList.emplace_back("birusquare"_n, AglMapping{ (AglType)1, 1, 0x3331 });
    s_aglList.emplace_back("blackcircle"_n, AglMapping{ (AglType)1, 1, 0x25CF });
    s_aglList.emplace_back("blackdiamond"_n, AglMapping{ (AglType)1, 1, 0x25C6 });
    s_aglList.emplace_back("blackdownpointingtriangle"_n, AglMapping{ (AglType)1, 1, 0x25BC });
    s_aglList.emplace_back("blackleftpointingpointer"_n, AglMapping{ (AglType)1, 1, 0x25C4 });
    s_aglList.emplace_back("blackleftpointingtriangle"_n, AglMapping{ (AglType)1, 1, 0x25C0 });
    s_aglList.emplace_back("blacklenticularbracketleft"_n, AglMapping{ (AglType)1, 1, 0x3010 });
    s_aglList.emplace_back("blacklenticularbracketleftvertical"_n, AglMapping{ (AglType)1, 1, 0xFE3B });
    s_aglList.emplace_back("blacklenticularbracketright"_n, AglMapping{ (AglType)1, 1, 0x3011 });
    s_aglList.emplace_back("blacklenticularbracketrightvertical"_n, AglMapping{ (AglType)1, 1, 0xFE3C });
    s_aglList.emplace_back("blacklowerlefttriangle"_n, AglMapping{ (AglType)1, 1, 0x25E3 });
    s_aglList.emplace_back("blacklowerrighttriangle"_n, AglMapping{ (AglType)1, 1, 0x25E2 });
    s_aglList.emplace_back("blackrectangle"_n, AglMapping{ (AglType)1, 1, 0x25AC });
    s_aglList.emplace_back("blackrightpointingpointer"_n, AglMapping{ (AglType)1, 1, 0x25BA });
    s_aglList.emplace_back("blackrightpointingtriangle"_n, AglMapping{ (AglType)1, 1, 0x25B6 });
    s_aglList.emplace_back("blacksmallsquare"_n, AglMapping{ (AglType)1, 1, 0x25AA });
    s_aglList.emplace_back("blacksmilingface"_n, AglMapping{ (AglType)1, 1, 0x263B });
    s_aglList.emplace_back("blacksquare"_n, AglMapping{ (AglType)1, 1, 0x25A0 });
    s_aglList.emplace_back("blackstar"_n, AglMapping{ (AglType)1, 1, 0x2605 });
    s_aglList.emplace_back("blackupperlefttriangle"_n, AglMapping{ (AglType)1, 1, 0x25E4 });
    s_aglList.emplace_back("blackupperrighttriangle"_n, AglMapping{ (AglType)1, 1, 0x25E5 });
    s_aglList.emplace_back("blackuppointingsmalltriangle"_n, AglMapping{ (AglType)1, 1, 0x25B4 });
    s_aglList.emplace_back("blackuppointingtriangle"_n, AglMapping{ (AglType)1, 1, 0x25B2 });
    s_aglList.emplace_back("blank"_n, AglMapping{ (AglType)1, 1, 0x2423 });
    s_aglList.emplace_back("blinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E07 });
    s_aglList.emplace_back("block"_n, AglMapping{ (AglType)3, 1, 0x2588 });
    s_aglList.emplace_back("bmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF42 });
    s_aglList.emplace_back("bobaimaithai"_n, AglMapping{ (AglType)1, 1, 0x0E1A });
    s_aglList.emplace_back("bohiragana"_n, AglMapping{ (AglType)1, 1, 0x307C });
    s_aglList.emplace_back("bokatakana"_n, AglMapping{ (AglType)1, 1, 0x30DC });
    s_aglList.emplace_back("bparen"_n, AglMapping{ (AglType)1, 1, 0x249D });
    s_aglList.emplace_back("bqsquare"_n, AglMapping{ (AglType)1, 1, 0x33C3 });
    s_aglList.emplace_back("braceex"_n, AglMapping{ (AglType)1, 1, 0xF8F4 });
    s_aglList.emplace_back("braceleft"_n, AglMapping{ (AglType)11, 1, 0x007B });
    s_aglList.emplace_back("braceleftbt"_n, AglMapping{ (AglType)1, 1, 0xF8F3 });
    s_aglList.emplace_back("braceleftmid"_n, AglMapping{ (AglType)1, 1, 0xF8F2 });
    s_aglList.emplace_back("braceleftmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF5B });
    s_aglList.emplace_back("braceleftsmall"_n, AglMapping{ (AglType)1, 1, 0xFE5B });
    s_aglList.emplace_back("bracelefttp"_n, AglMapping{ (AglType)1, 1, 0xF8F1 });
    s_aglList.emplace_back("braceleftvertical"_n, AglMapping{ (AglType)1, 1, 0xFE37 });
    s_aglList.emplace_back("braceright"_n, AglMapping{ (AglType)11, 1, 0x007D });
    s_aglList.emplace_back("bracerightbt"_n, AglMapping{ (AglType)1, 1, 0xF8FE });
    s_aglList.emplace_back("bracerightmid"_n, AglMapping{ (AglType)1, 1, 0xF8FD });
    s_aglList.emplace_back("bracerightmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF5D });
    s_aglList.emplace_back("bracerightsmall"_n, AglMapping{ (AglType)1, 1, 0xFE5C });
    s_aglList.emplace_back("bracerighttp"_n, AglMapping{ (AglType)1, 1, 0xF8FC });
    s_aglList.emplace_back("bracerightvertical"_n, AglMapping{ (AglType)1, 1, 0xFE38 });
    s_aglList.emplace_back("bracketleft"_n, AglMapping{ (AglType)11, 1, 0x005B });
    s_aglList.emplace_back("bracketleftbt"_n, AglMapping{ (AglType)1, 1, 0xF8F0 });
    s_aglList.emplace_back("bracketleftex"_n, AglMapping{ (AglType)1, 1, 0xF8EF });
    s_aglList.emplace_back("bracketleftmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF3B });
    s_aglList.emplace_back("bracketlefttp"_n, AglMapping{ (AglType)1, 1, 0xF8EE });
    s_aglList.emplace_back("bracketright"_n, AglMapping{ (AglType)11, 1, 0x005D });
    s_aglList.emplace_back("bracketrightbt"_n, AglMapping{ (AglType)1, 1, 0xF8FB });
    s_aglList.emplace_back("bracketrightex"_n, AglMapping{ (AglType)1, 1, 0xF8FA });
    s_aglList.emplace_back("bracketrightmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF3D });
    s_aglList.emplace_back("bracketrighttp"_n, AglMapping{ (AglType)1, 1, 0xF8F9 });
    s_aglList.emplace_back("breve"_n, AglMapping{ (AglType)11, 1, 0x02D8 });
    s_aglList.emplace_back("brevebelowcmb"_n, AglMapping{ (AglType)1, 1, 0x032E });
    s_aglList.emplace_back("brevecmb"_n, AglMapping{ (AglType)1, 1, 0x0306 });
    s_aglList.emplace_back("breveinvertedbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x032F });
    s_aglList.emplace_back("breveinvertedcmb"_n, AglMapping{ (AglType)1, 1, 0x0311 });
    s_aglList.emplace_back("breveinverteddoublecmb"_n, AglMapping{ (AglType)1, 1, 0x0361 });
    s_aglList.emplace_back("bridgebelowcmb"_n, AglMapping{ (AglType)1, 1, 0x032A });
    s_aglList.emplace_back("bridgeinvertedbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x033A });
    s_aglList.emplace_back("brokenbar"_n, AglMapping{ (AglType)11, 1, 0x00A6 });
    s_aglList.emplace_back("bstroke"_n, AglMapping{ (AglType)1, 1, 0x0180 });
    s_aglList.emplace_back("bsuperior"_n, AglMapping{ (AglType)1, 1, 0xF6EA });
    s_aglList.emplace_back("btopbar"_n, AglMapping{ (AglType)1, 1, 0x0183 });
    s_aglList.emplace_back("buhiragana"_n, AglMapping{ (AglType)1, 1, 0x3076 });
    s_aglList.emplace_back("bukatakana"_n, AglMapping{ (AglType)1, 1, 0x30D6 });
    s_aglList.emplace_back("bullet"_n, AglMapping{ (AglType)11, 1, 0x2022 });
    s_aglList.emplace_back("bulletinverse"_n, AglMapping{ (AglType)1, 1, 0x25D8 });
    s_aglList.emplace_back("bulletoperator"_n, AglMapping{ (AglType)1, 1, 0x2219 });
    s_aglList.emplace_back("bullseye"_n, AglMapping{ (AglType)1, 1, 0x25CE });
    s_aglList.emplace_back("c"_n, AglMapping{ (AglType)11, 1, 0x0063 });
    s_aglList.emplace_back("caarmenian"_n, AglMapping{ (AglType)1, 1, 0x056E });
    s_aglList.emplace_back("cabengali"_n, AglMapping{ (AglType)1, 1, 0x099A });
    s_aglList.emplace_back("cacute"_n, AglMapping{ (AglType)3, 1, 0x0107 });
    s_aglList.emplace_back("cadeva"_n, AglMapping{ (AglType)1, 1, 0x091A });
    s_aglList.emplace_back("cagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A9A });
    s_aglList.emplace_back("cagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A1A });
    s_aglList.emplace_back("calsquare"_n, AglMapping{ (AglType)1, 1, 0x3388 });
    s_aglList.emplace_back("candrabindubengali"_n, AglMapping{ (AglType)1, 1, 0x0981 });
    s_aglList.emplace_back("candrabinducmb"_n, AglMapping{ (AglType)1, 1, 0x0310 });
    s_aglList.emplace_back("candrabindudeva"_n, AglMapping{ (AglType)1, 1, 0x0901 });
    s_aglList.emplace_back("candrabindugujarati"_n, AglMapping{ (AglType)1, 1, 0x0A81 });
    s_aglList.emplace_back("capslock"_n, AglMapping{ (AglType)1, 1, 0x21EA });
    s_aglList.emplace_back("careof"_n, AglMapping{ (AglType)1, 1, 0x2105 });
    s_aglList.emplace_back("caron"_n, AglMapping{ (AglType)11, 1, 0x02C7 });
    s_aglList.emplace_back("caronbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x032C });
    s_aglList.emplace_back("caroncmb"_n, AglMapping{ (AglType)1, 1, 0x030C });
    s_aglList.emplace_back("carriagereturn"_n, AglMapping{ (AglType)3, 1, 0x21B5 });
    s_aglList.emplace_back("cbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3118 });
    s_aglList.emplace_back("ccaron"_n, AglMapping{ (AglType)3, 1, 0x010D });
    s_aglList.emplace_back("ccedilla"_n, AglMapping{ (AglType)11, 1, 0x00E7 });
    s_aglList.emplace_back("ccedillaacute"_n, AglMapping{ (AglType)1, 1, 0x1E09 });
    s_aglList.emplace_back("ccircle"_n, AglMapping{ (AglType)1, 1, 0x24D2 });
    s_aglList.emplace_back("ccircumflex"_n, AglMapping{ (AglType)3, 1, 0x0109 });
    s_aglList.emplace_back("ccurl"_n, AglMapping{ (AglType)1, 1, 0x0255 });
    s_aglList.emplace_back("cdot"_n, AglMapping{ (AglType)1, 1, 0x010B });
    s_aglList.emplace_back("cdotaccent"_n, AglMapping{ (AglType)3, 1, 0x010B });
    s_aglList.emplace_back("cdsquare"_n, AglMapping{ (AglType)1, 1, 0x33C5 });
    s_aglList.emplace_back("cedilla"_n, AglMapping{ (AglType)11, 1, 0x00B8 });
    s_aglList.emplace_back("cedillacmb"_n, AglMapping{ (AglType)1, 1, 0x0327 });
    s_aglList.emplace_back("cent"_n, AglMapping{ (AglType)11, 1, 0x00A2 });
    s_aglList.emplace_back("centigrade"_n, AglMapping{ (AglType)1, 1, 0x2103 });
    s_aglList.emplace_back("centinferior"_n, AglMapping{ (AglType)1, 1, 0xF6DF });
    s_aglList.emplace_back("centmonospace"_n, AglMapping{ (AglType)1, 1, 0xFFE0 });
    s_aglList.emplace_back("centoldstyle"_n, AglMapping{ (AglType)1, 1, 0xF7A2 });
    s_aglList.emplace_back("centsuperior"_n, AglMapping{ (AglType)1, 1, 0xF6E0 });
    s_aglList.emplace_back("chaarmenian"_n, AglMapping{ (AglType)1, 1, 0x0579 });
    s_aglList.emplace_back("chabengali"_n, AglMapping{ (AglType)1, 1, 0x099B });
    s_aglList.emplace_back("chadeva"_n, AglMapping{ (AglType)1, 1, 0x091B });
    s_aglList.emplace_back("chagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A9B });
    s_aglList.emplace_back("chagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A1B });
    s_aglList.emplace_back("chbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3114 });
    s_aglList.emplace_back("cheabkhasiancyrillic"_n, AglMapping{ (AglType)1, 1, 0x04BD });
    s_aglList.emplace_back("checkmark"_n, AglMapping{ (AglType)1, 1, 0x2713 });
    s_aglList.emplace_back("checyrillic"_n, AglMapping{ (AglType)1, 1, 0x0447 });
    s_aglList.emplace_back("chedescenderabkhasiancyrillic"_n, AglMapping{ (AglType)1, 1, 0x04BF });
    s_aglList.emplace_back("chedescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x04B7 });
    s_aglList.emplace_back("chedieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04F5 });
    s_aglList.emplace_back("cheharmenian"_n, AglMapping{ (AglType)1, 1, 0x0573 });
    s_aglList.emplace_back("chekhakassiancyrillic"_n, AglMapping{ (AglType)1, 1, 0x04CC });
    s_aglList.emplace_back("cheverticalstrokecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04B9 });
    s_aglList.emplace_back("chi"_n, AglMapping{ (AglType)3, 1, 0x03C7 });
    s_aglList.emplace_back("chieuchacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3277 });
    s_aglList.emplace_back("chieuchaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3217 });
    s_aglList.emplace_back("chieuchcirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3269 });
    s_aglList.emplace_back("chieuchkorean"_n, AglMapping{ (AglType)1, 1, 0x314A });
    s_aglList.emplace_back("chieuchparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3209 });
    s_aglList.emplace_back("chochangthai"_n, AglMapping{ (AglType)1, 1, 0x0E0A });
    s_aglList.emplace_back("chochanthai"_n, AglMapping{ (AglType)1, 1, 0x0E08 });
    s_aglList.emplace_back("chochingthai"_n, AglMapping{ (AglType)1, 1, 0x0E09 });
    s_aglList.emplace_back("chochoethai"_n, AglMapping{ (AglType)1, 1, 0x0E0C });
    s_aglList.emplace_back("chook"_n, AglMapping{ (AglType)1, 1, 0x0188 });
    s_aglList.emplace_back("cieucacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3276 });
    s_aglList.emplace_back("cieucaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3216 });
    s_aglList.emplace_back("cieuccirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3268 });
    s_aglList.emplace_back("cieuckorean"_n, AglMapping{ (AglType)1, 1, 0x3148 });
    s_aglList.emplace_back("cieucparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3208 });
    s_aglList.emplace_back("cieucuparenkorean"_n, AglMapping{ (AglType)1, 1, 0x321C });
    s_aglList.emplace_back("circle"_n, AglMapping{ (AglType)3, 1, 0x25CB });
    s_aglList.emplace_back("circlemultiply"_n, AglMapping{ (AglType)3, 1, 0x2297 });
    s_aglList.emplace_back("circleot"_n, AglMapping{ (AglType)1, 1, 0x2299 });
    s_aglList.emplace_back("circleplus"_n, AglMapping{ (AglType)3, 1, 0x2295 });
    s_aglList.emplace_back("circlepostalmark"_n, AglMapping{ (AglType)1, 1, 0x3036 });
    s_aglList.emplace_back("circlewithlefthalfblack"_n, AglMapping{ (AglType)1, 1, 0x25D0 });
    s_aglList.emplace_back("circlewithrighthalfblack"_n, AglMapping{ (AglType)1, 1, 0x25D1 });
    s_aglList.emplace_back("circumflex"_n, AglMapping{ (AglType)11, 1, 0x02C6 });
    s_aglList.emplace_back("circumflexbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x032D });
    s_aglList.emplace_back("circumflexcmb"_n, AglMapping{ (AglType)1, 1, 0x0302 });
    s_aglList.emplace_back("clear"_n, AglMapping{ (AglType)1, 1, 0x2327 });
    s_aglList.emplace_back("clickalveolar"_n, AglMapping{ (AglType)1, 1, 0x01C2 });
    s_aglList.emplace_back("clickdental"_n, AglMapping{ (AglType)1, 1, 0x01C0 });
    s_aglList.emplace_back("clicklateral"_n, AglMapping{ (AglType)1, 1, 0x01C1 });
    s_aglList.emplace_back("clickretroflex"_n, AglMapping{ (AglType)1, 1, 0x01C3 });
    s_aglList.emplace_back("club"_n, AglMapping{ (AglType)3, 1, 0x2663 });
    s_aglList.emplace_back("clubsuitblack"_n, AglMapping{ (AglType)1, 1, 0x2663 });
    s_aglList.emplace_back("clubsuitwhite"_n, AglMapping{ (AglType)1, 1, 0x2667 });
    s_aglList.emplace_back("cmcubedsquare"_n, AglMapping{ (AglType)1, 1, 0x33A4 });
    s_aglList.emplace_back("cmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF43 });
    s_aglList.emplace_back("cmsquaredsquare"_n, AglMapping{ (AglType)1, 1, 0x33A0 });
    s_aglList.emplace_back("coarmenian"_n, AglMapping{ (AglType)1, 1, 0x0581 });
    s_aglList.emplace_back("colon"_n, AglMapping{ (AglType)11, 1, 0x003A });
    s_aglList.emplace_back("colonmonetary"_n, AglMapping{ (AglType)3, 1, 0x20A1 });
    s_aglList.emplace_back("colonmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF1A });
    s_aglList.emplace_back("colonsign"_n, AglMapping{ (AglType)1, 1, 0x20A1 });
    s_aglList.emplace_back("colonsmall"_n, AglMapping{ (AglType)1, 1, 0xFE55 });
    s_aglList.emplace_back("colontriangularhalfmod"_n, AglMapping{ (AglType)1, 1, 0x02D1 });
    s_aglList.emplace_back("colontriangularmod"_n, AglMapping{ (AglType)1, 1, 0x02D0 });
    s_aglList.emplace_back("comma"_n, AglMapping{ (AglType)11, 1, 0x002C });
    s_aglList.emplace_back("commaabovecmb"_n, AglMapping{ (AglType)1, 1, 0x0313 });
    s_aglList.emplace_back("commaaboverightcmb"_n, AglMapping{ (AglType)1, 1, 0x0315 });
    s_aglList.emplace_back("commaaccent"_n, AglMapping{ (AglType)1, 1, 0xF6C3 });
    s_aglList.emplace_back("commaarabic"_n, AglMapping{ (AglType)1, 1, 0x060C });
    s_aglList.emplace_back("commaarmenian"_n, AglMapping{ (AglType)1, 1, 0x055D });
    s_aglList.emplace_back("commainferior"_n, AglMapping{ (AglType)1, 1, 0xF6E1 });
    s_aglList.emplace_back("commamonospace"_n, AglMapping{ (AglType)1, 1, 0xFF0C });
    s_aglList.emplace_back("commareversedabovecmb"_n, AglMapping{ (AglType)1, 1, 0x0314 });
    s_aglList.emplace_back("commareversedmod"_n, AglMapping{ (AglType)1, 1, 0x02BD });
    s_aglList.emplace_back("commasmall"_n, AglMapping{ (AglType)1, 1, 0xFE50 });
    s_aglList.emplace_back("commasuperior"_n, AglMapping{ (AglType)1, 1, 0xF6E2 });
    s_aglList.emplace_back("commaturnedabovecmb"_n, AglMapping{ (AglType)1, 1, 0x0312 });
    s_aglList.emplace_back("commaturnedmod"_n, AglMapping{ (AglType)1, 1, 0x02BB });
    s_aglList.emplace_back("compass"_n, AglMapping{ (AglType)1, 1, 0x263C });
    s_aglList.emplace_back("congruent"_n, AglMapping{ (AglType)3, 1, 0x2245 });
    s_aglList.emplace_back("contourintegral"_n, AglMapping{ (AglType)1, 1, 0x222E });
    s_aglList.emplace_back("control"_n, AglMapping{ (AglType)1, 1, 0x2303 });
    s_aglList.emplace_back("controlACK"_n, AglMapping{ (AglType)1, 1, 0x0006 });
    s_aglList.emplace_back("controlBEL"_n, AglMapping{ (AglType)1, 1, 0x0007 });
    s_aglList.emplace_back("controlBS"_n, AglMapping{ (AglType)1, 1, 0x0008 });
    s_aglList.emplace_back("controlCAN"_n, AglMapping{ (AglType)1, 1, 0x0018 });
    s_aglList.emplace_back("controlCR"_n, AglMapping{ (AglType)1, 1, 0x000D });
    s_aglList.emplace_back("controlDC1"_n, AglMapping{ (AglType)1, 1, 0x0011 });
    s_aglList.emplace_back("controlDC2"_n, AglMapping{ (AglType)1, 1, 0x0012 });
    s_aglList.emplace_back("controlDC3"_n, AglMapping{ (AglType)1, 1, 0x0013 });
    s_aglList.emplace_back("controlDC4"_n, AglMapping{ (AglType)1, 1, 0x0014 });
    s_aglList.emplace_back("controlDEL"_n, AglMapping{ (AglType)1, 1, 0x007F });
    s_aglList.emplace_back("controlDLE"_n, AglMapping{ (AglType)1, 1, 0x0010 });
    s_aglList.emplace_back("controlEM"_n, AglMapping{ (AglType)1, 1, 0x0019 });
    s_aglList.emplace_back("controlENQ"_n, AglMapping{ (AglType)1, 1, 0x0005 });
    s_aglList.emplace_back("controlEOT"_n, AglMapping{ (AglType)1, 1, 0x0004 });
    s_aglList.emplace_back("controlESC"_n, AglMapping{ (AglType)1, 1, 0x001B });
    s_aglList.emplace_back("controlETB"_n, AglMapping{ (AglType)1, 1, 0x0017 });
    s_aglList.emplace_back("controlETX"_n, AglMapping{ (AglType)1, 1, 0x0003 });
    s_aglList.emplace_back("controlFF"_n, AglMapping{ (AglType)1, 1, 0x000C });
    s_aglList.emplace_back("controlFS"_n, AglMapping{ (AglType)1, 1, 0x001C });
    s_aglList.emplace_back("controlGS"_n, AglMapping{ (AglType)1, 1, 0x001D });
    s_aglList.emplace_back("controlHT"_n, AglMapping{ (AglType)1, 1, 0x0009 });
    s_aglList.emplace_back("controlLF"_n, AglMapping{ (AglType)1, 1, 0x000A });
    s_aglList.emplace_back("controlNAK"_n, AglMapping{ (AglType)1, 1, 0x0015 });
    s_aglList.emplace_back("controlRS"_n, AglMapping{ (AglType)1, 1, 0x001E });
    s_aglList.emplace_back("controlSI"_n, AglMapping{ (AglType)1, 1, 0x000F });
    s_aglList.emplace_back("controlSO"_n, AglMapping{ (AglType)1, 1, 0x000E });
    s_aglList.emplace_back("controlSOT"_n, AglMapping{ (AglType)1, 1, 0x0002 });
    s_aglList.emplace_back("controlSTX"_n, AglMapping{ (AglType)1, 1, 0x0001 });
    s_aglList.emplace_back("controlSUB"_n, AglMapping{ (AglType)1, 1, 0x001A });
    s_aglList.emplace_back("controlSYN"_n, AglMapping{ (AglType)1, 1, 0x0016 });
    s_aglList.emplace_back("controlUS"_n, AglMapping{ (AglType)1, 1, 0x001F });
    s_aglList.emplace_back("controlVT"_n, AglMapping{ (AglType)1, 1, 0x000B });
    s_aglList.emplace_back("copyright"_n, AglMapping{ (AglType)11, 1, 0x00A9 });
    s_aglList.emplace_back("copyrightsans"_n, AglMapping{ (AglType)1, 1, 0xF8E9 });
    s_aglList.emplace_back("copyrightserif"_n, AglMapping{ (AglType)1, 1, 0xF6D9 });
    s_aglList.emplace_back("cornerbracketleft"_n, AglMapping{ (AglType)1, 1, 0x300C });
    s_aglList.emplace_back("cornerbracketlefthalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF62 });
    s_aglList.emplace_back("cornerbracketleftvertical"_n, AglMapping{ (AglType)1, 1, 0xFE41 });
    s_aglList.emplace_back("cornerbracketright"_n, AglMapping{ (AglType)1, 1, 0x300D });
    s_aglList.emplace_back("cornerbracketrighthalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF63 });
    s_aglList.emplace_back("cornerbracketrightvertical"_n, AglMapping{ (AglType)1, 1, 0xFE42 });
    s_aglList.emplace_back("corporationsquare"_n, AglMapping{ (AglType)1, 1, 0x337F });
    s_aglList.emplace_back("cosquare"_n, AglMapping{ (AglType)1, 1, 0x33C7 });
    s_aglList.emplace_back("coverkgsquare"_n, AglMapping{ (AglType)1, 1, 0x33C6 });
    s_aglList.emplace_back("cparen"_n, AglMapping{ (AglType)1, 1, 0x249E });
    s_aglList.emplace_back("cruzeiro"_n, AglMapping{ (AglType)1, 1, 0x20A2 });
    s_aglList.emplace_back("cstretched"_n, AglMapping{ (AglType)1, 1, 0x0297 });
    s_aglList.emplace_back("curlyand"_n, AglMapping{ (AglType)1, 1, 0x22CF });
    s_aglList.emplace_back("curlyor"_n, AglMapping{ (AglType)1, 1, 0x22CE });
    s_aglList.emplace_back("currency"_n, AglMapping{ (AglType)11, 1, 0x00A4 });
    s_aglList.emplace_back("cyrBreve"_n, AglMapping{ (AglType)1, 1, 0xF6D1 });
    s_aglList.emplace_back("cyrFlex"_n, AglMapping{ (AglType)1, 1, 0xF6D2 });
    s_aglList.emplace_back("cyrbreve"_n, AglMapping{ (AglType)1, 1, 0xF6D4 });
    s_aglList.emplace_back("cyrflex"_n, AglMapping{ (AglType)1, 1, 0xF6D5 });
    s_aglList.emplace_back("d"_n, AglMapping{ (AglType)11, 1, 0x0064 });
    s_aglList.emplace_back("daarmenian"_n, AglMapping{ (AglType)1, 1, 0x0564 });
    s_aglList.emplace_back("dabengali"_n, AglMapping{ (AglType)1, 1, 0x09A6 });
    s_aglList.emplace_back("dadarabic"_n, AglMapping{ (AglType)1, 1, 0x0636 });
    s_aglList.emplace_back("dadeva"_n, AglMapping{ (AglType)1, 1, 0x0926 });
    s_aglList.emplace_back("dadfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEBE });
    s_aglList.emplace_back("dadinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEBF });
    s_aglList.emplace_back("dadmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEC0 });
    s_aglList.emplace_back("dagesh"_n, AglMapping{ (AglType)1, 1, 0x05BC });
    s_aglList.emplace_back("dageshhebrew"_n, AglMapping{ (AglType)1, 1, 0x05BC });
    s_aglList.emplace_back("dagger"_n, AglMapping{ (AglType)11, 1, 0x2020 });
    s_aglList.emplace_back("daggerdbl"_n, AglMapping{ (AglType)11, 1, 0x2021 });
    s_aglList.emplace_back("dagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AA6 });
    s_aglList.emplace_back("dagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A26 });
    s_aglList.emplace_back("dahiragana"_n, AglMapping{ (AglType)1, 1, 0x3060 });
    s_aglList.emplace_back("dakatakana"_n, AglMapping{ (AglType)1, 1, 0x30C0 });
    s_aglList.emplace_back("dalarabic"_n, AglMapping{ (AglType)1, 1, 0x062F });
    s_aglList.emplace_back("dalet"_n, AglMapping{ (AglType)1, 1, 0x05D3 });
    s_aglList.emplace_back("daletdagesh"_n, AglMapping{ (AglType)1, 1, 0xFB33 });
    s_aglList.emplace_back("daletdageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB33 });
    s_aglList.emplace_back("dalethatafpatah"_n, AglMapping{ (AglType)1, 2, 0 });
    s_aglList.emplace_back("dalethatafpatahhebrew"_n, AglMapping{ (AglType)1, 2, 1 });
    s_aglList.emplace_back("dalethatafsegol"_n, AglMapping{ (AglType)1, 2, 2 });
    s_aglList.emplace_back("dalethatafsegolhebrew"_n, AglMapping{ (AglType)1, 2, 3 });
    s_aglList.emplace_back("dalethebrew"_n, AglMapping{ (AglType)1, 1, 0x05D3 });
    s_aglList.emplace_back("dalethiriq"_n, AglMapping{ (AglType)1, 2, 4 });
    s_aglList.emplace_back("dalethiriqhebrew"_n, AglMapping{ (AglType)1, 2, 5 });
    s_aglList.emplace_back("daletholam"_n, AglMapping{ (AglType)1, 2, 6 });
    s_aglList.emplace_back("daletholamhebrew"_n, AglMapping{ (AglType)1, 2, 7 });
    s_aglList.emplace_back("daletpatah"_n, AglMapping{ (AglType)1, 2, 8 });
    s_aglList.emplace_back("daletpatahhebrew"_n, AglMapping{ (AglType)1, 2, 9 });
    s_aglList.emplace_back("daletqamats"_n, AglMapping{ (AglType)1, 2, 10 });
    s_aglList.emplace_back("daletqamatshebrew"_n, AglMapping{ (AglType)1, 2, 11 });
    s_aglList.emplace_back("daletqubuts"_n, AglMapping{ (AglType)1, 2, 12 });
    s_aglList.emplace_back("daletqubutshebrew"_n, AglMapping{ (AglType)1, 2, 13 });
    s_aglList.emplace_back("daletsegol"_n, AglMapping{ (AglType)1, 2, 14 });
    s_aglList.emplace_back("daletsegolhebrew"_n, AglMapping{ (AglType)1, 2, 15 });
    s_aglList.emplace_back("daletsheva"_n, AglMapping{ (AglType)1, 2, 16 });
    s_aglList.emplace_back("daletshevahebrew"_n, AglMapping{ (AglType)1, 2, 17 });
    s_aglList.emplace_back("dalettsere"_n, AglMapping{ (AglType)1, 2, 18 });
    s_aglList.emplace_back("dalettserehebrew"_n, AglMapping{ (AglType)1, 2, 19 });
    s_aglList.emplace_back("dalfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEAA });
    s_aglList.emplace_back("dammaarabic"_n, AglMapping{ (AglType)1, 1, 0x064F });
    s_aglList.emplace_back("dammalowarabic"_n, AglMapping{ (AglType)1, 1, 0x064F });
    s_aglList.emplace_back("dammatanaltonearabic"_n, AglMapping{ (AglType)1, 1, 0x064C });
    s_aglList.emplace_back("dammatanarabic"_n, AglMapping{ (AglType)1, 1, 0x064C });
    s_aglList.emplace_back("danda"_n, AglMapping{ (AglType)1, 1, 0x0964 });
    s_aglList.emplace_back("dargahebrew"_n, AglMapping{ (AglType)1, 1, 0x05A7 });
    s_aglList.emplace_back("dargalefthebrew"_n, AglMapping{ (AglType)1, 1, 0x05A7 });
    s_aglList.emplace_back("dasiapneumatacyrilliccmb"_n, AglMapping{ (AglType)1, 1, 0x0485 });
    s_aglList.emplace_back("dblGrave"_n, AglMapping{ (AglType)1, 1, 0xF6D3 });
    s_aglList.emplace_back("dblanglebracketleft"_n, AglMapping{ (AglType)1, 1, 0x300A });
    s_aglList.emplace_back("dblanglebracketleftvertical"_n, AglMapping{ (AglType)1, 1, 0xFE3D });
    s_aglList.emplace_back("dblanglebracketright"_n, AglMapping{ (AglType)1, 1, 0x300B });
    s_aglList.emplace_back("dblanglebracketrightvertical"_n, AglMapping{ (AglType)1, 1, 0xFE3E });
    s_aglList.emplace_back("dblarchinvertedbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x032B });
    s_aglList.emplace_back("dblarrowleft"_n, AglMapping{ (AglType)1, 1, 0x21D4 });
    s_aglList.emplace_back("dblarrowright"_n, AglMapping{ (AglType)1, 1, 0x21D2 });
    s_aglList.emplace_back("dbldanda"_n, AglMapping{ (AglType)1, 1, 0x0965 });
    s_aglList.emplace_back("dblgrave"_n, AglMapping{ (AglType)1, 1, 0xF6D6 });
    s_aglList.emplace_back("dblgravecmb"_n, AglMapping{ (AglType)1, 1, 0x030F });
    s_aglList.emplace_back("dblintegral"_n, AglMapping{ (AglType)1, 1, 0x222C });
    s_aglList.emplace_back("dbllowline"_n, AglMapping{ (AglType)1, 1, 0x2017 });
    s_aglList.emplace_back("dbllowlinecmb"_n, AglMapping{ (AglType)1, 1, 0x0333 });
    s_aglList.emplace_back("dbloverlinecmb"_n, AglMapping{ (AglType)1, 1, 0x033F });
    s_aglList.emplace_back("dblprimemod"_n, AglMapping{ (AglType)1, 1, 0x02BA });
    s_aglList.emplace_back("dblverticalbar"_n, AglMapping{ (AglType)1, 1, 0x2016 });
    s_aglList.emplace_back("dblverticallineabovecmb"_n, AglMapping{ (AglType)1, 1, 0x030E });
    s_aglList.emplace_back("dbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3109 });
    s_aglList.emplace_back("dbsquare"_n, AglMapping{ (AglType)1, 1, 0x33C8 });
    s_aglList.emplace_back("dcaron"_n, AglMapping{ (AglType)3, 1, 0x010F });
    s_aglList.emplace_back("dcedilla"_n, AglMapping{ (AglType)1, 1, 0x1E11 });
    s_aglList.emplace_back("dcircle"_n, AglMapping{ (AglType)1, 1, 0x24D3 });
    s_aglList.emplace_back("dcircumflexbelow"_n, AglMapping{ (AglType)1, 1, 0x1E13 });
    s_aglList.emplace_back("dcroat"_n, AglMapping{ (AglType)3, 1, 0x0111 });
    s_aglList.emplace_back("ddabengali"_n, AglMapping{ (AglType)1, 1, 0x09A1 });
    s_aglList.emplace_back("ddadeva"_n, AglMapping{ (AglType)1, 1, 0x0921 });
    s_aglList.emplace_back("ddagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AA1 });
    s_aglList.emplace_back("ddagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A21 });
    s_aglList.emplace_back("ddalarabic"_n, AglMapping{ (AglType)1, 1, 0x0688 });
    s_aglList.emplace_back("ddalfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFB89 });
    s_aglList.emplace_back("dddhadeva"_n, AglMapping{ (AglType)1, 1, 0x095C });
    s_aglList.emplace_back("ddhabengali"_n, AglMapping{ (AglType)1, 1, 0x09A2 });
    s_aglList.emplace_back("ddhadeva"_n, AglMapping{ (AglType)1, 1, 0x0922 });
    s_aglList.emplace_back("ddhagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AA2 });
    s_aglList.emplace_back("ddhagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A22 });
    s_aglList.emplace_back("ddotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E0B });
    s_aglList.emplace_back("ddotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E0D });
    s_aglList.emplace_back("decimalseparatorarabic"_n, AglMapping{ (AglType)1, 1, 0x066B });
    s_aglList.emplace_back("decimalseparatorpersian"_n, AglMapping{ (AglType)1, 1, 0x066B });
    s_aglList.emplace_back("decyrillic"_n, AglMapping{ (AglType)1, 1, 0x0434 });
    s_aglList.emplace_back("degree"_n, AglMapping{ (AglType)11, 1, 0x00B0 });
    s_aglList.emplace_back("dehihebrew"_n, AglMapping{ (AglType)1, 1, 0x05AD });
    s_aglList.emplace_back("dehiragana"_n, AglMapping{ (AglType)1, 1, 0x3067 });
    s_aglList.emplace_back("deicoptic"_n, AglMapping{ (AglType)1, 1, 0x03EF });
    s_aglList.emplace_back("dekatakana"_n, AglMapping{ (AglType)1, 1, 0x30C7 });
    s_aglList.emplace_back("deleteleft"_n, AglMapping{ (AglType)1, 1, 0x232B });
    s_aglList.emplace_back("deleteright"_n, AglMapping{ (AglType)1, 1, 0x2326 });
    s_aglList.emplace_back("delta"_n, AglMapping{ (AglType)3, 1, 0x03B4 });
    s_aglList.emplace_back("deltaturned"_n, AglMapping{ (AglType)1, 1, 0x018D });
    s_aglList.emplace_back("denominatorminusonenumeratorbengali"_n, AglMapping{ (AglType)1, 1, 0x09F8 });
    s_aglList.emplace_back("dezh"_n, AglMapping{ (AglType)1, 1, 0x02A4 });
    s_aglList.emplace_back("dhabengali"_n, AglMapping{ (AglType)1, 1, 0x09A7 });
    s_aglList.emplace_back("dhadeva"_n, AglMapping{ (AglType)1, 1, 0x0927 });
    s_aglList.emplace_back("dhagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AA7 });
    s_aglList.emplace_back("dhagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A27 });
    s_aglList.emplace_back("dhook"_n, AglMapping{ (AglType)1, 1, 0x0257 });
    s_aglList.emplace_back("dialytikatonos"_n, AglMapping{ (AglType)1, 1, 0x0385 });
    s_aglList.emplace_back("dialytikatonoscmb"_n, AglMapping{ (AglType)1, 1, 0x0344 });
    s_aglList.emplace_back("diamond"_n, AglMapping{ (AglType)3, 1, 0x2666 });
    s_aglList.emplace_back("diamondsuitwhite"_n, AglMapping{ (AglType)1, 1, 0x2662 });
    s_aglList.emplace_back("dieresis"_n, AglMapping{ (AglType)11, 1, 0x00A8 });
    s_aglList.emplace_back("dieresisacute"_n, AglMapping{ (AglType)1, 1, 0xF6D7 });
    s_aglList.emplace_back("dieresisbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0324 });
    s_aglList.emplace_back("dieresiscmb"_n, AglMapping{ (AglType)1, 1, 0x0308 });
    s_aglList.emplace_back("dieresisgrave"_n, AglMapping{ (AglType)1, 1, 0xF6D8 });
    s_aglList.emplace_back("dieresistonos"_n, AglMapping{ (AglType)3, 1, 0x0385 });
    s_aglList.emplace_back("dihiragana"_n, AglMapping{ (AglType)1, 1, 0x3062 });
    s_aglList.emplace_back("dikatakana"_n, AglMapping{ (AglType)1, 1, 0x30C2 });
    s_aglList.emplace_back("dittomark"_n, AglMapping{ (AglType)1, 1, 0x3003 });
    s_aglList.emplace_back("divide"_n, AglMapping{ (AglType)11, 1, 0x00F7 });
    s_aglList.emplace_back("divides"_n, AglMapping{ (AglType)1, 1, 0x2223 });
    s_aglList.emplace_back("divisionslash"_n, AglMapping{ (AglType)1, 1, 0x2215 });
    s_aglList.emplace_back("djecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0452 });
    s_aglList.emplace_back("dkshade"_n, AglMapping{ (AglType)3, 1, 0x2593 });
    s_aglList.emplace_back("dlinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E0F });
    s_aglList.emplace_back("dlsquare"_n, AglMapping{ (AglType)1, 1, 0x3397 });
    s_aglList.emplace_back("dmacron"_n, AglMapping{ (AglType)1, 1, 0x0111 });
    s_aglList.emplace_back("dmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF44 });
    s_aglList.emplace_back("dnblock"_n, AglMapping{ (AglType)3, 1, 0x2584 });
    s_aglList.emplace_back("dochadathai"_n, AglMapping{ (AglType)1, 1, 0x0E0E });
    s_aglList.emplace_back("dodekthai"_n, AglMapping{ (AglType)1, 1, 0x0E14 });
    s_aglList.emplace_back("dohiragana"_n, AglMapping{ (AglType)1, 1, 0x3069 });
    s_aglList.emplace_back("dokatakana"_n, AglMapping{ (AglType)1, 1, 0x30C9 });
    s_aglList.emplace_back("dollar"_n, AglMapping{ (AglType)11, 1, 0x0024 });
    s_aglList.emplace_back("dollarinferior"_n, AglMapping{ (AglType)1, 1, 0xF6E3 });
    s_aglList.emplace_back("dollarmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF04 });
    s_aglList.emplace_back("dollaroldstyle"_n, AglMapping{ (AglType)1, 1, 0xF724 });
    s_aglList.emplace_back("dollarsmall"_n, AglMapping{ (AglType)1, 1, 0xFE69 });
    s_aglList.emplace_back("dollarsuperior"_n, AglMapping{ (AglType)1, 1, 0xF6E4 });
    s_aglList.emplace_back("dong"_n, AglMapping{ (AglType)3, 1, 0x20AB });
    s_aglList.emplace_back("dorusquare"_n, AglMapping{ (AglType)1, 1, 0x3326 });
    s_aglList.emplace_back("dotaccent"_n, AglMapping{ (AglType)11, 1, 0x02D9 });
    s_aglList.emplace_back("dotaccentcmb"_n, AglMapping{ (AglType)1, 1, 0x0307 });
    s_aglList.emplace_back("dotbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0323 });
    s_aglList.emplace_back("dotbelowcomb"_n, AglMapping{ (AglType)3, 1, 0x0323 });
    s_aglList.emplace_back("dotkatakana"_n, AglMapping{ (AglType)1, 1, 0x30FB });
    s_aglList.emplace_back("dotlessi"_n, AglMapping{ (AglType)11, 1, 0x0131 });
    s_aglList.emplace_back("dotlessj"_n, AglMapping{ (AglType)1, 1, 0xF6BE });
    s_aglList.emplace_back("dotlessjstrokehook"_n, AglMapping{ (AglType)1, 1, 0x0284 });
    s_aglList.emplace_back("dotmath"_n, AglMapping{ (AglType)3, 1, 0x22C5 });
    s_aglList.emplace_back("dottedcircle"_n, AglMapping{ (AglType)1, 1, 0x25CC });
    s_aglList.emplace_back("doubleyodpatah"_n, AglMapping{ (AglType)1, 1, 0xFB1F });
    s_aglList.emplace_back("doubleyodpatahhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB1F });
    s_aglList.emplace_back("downtackbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x031E });
    s_aglList.emplace_back("downtackmod"_n, AglMapping{ (AglType)1, 1, 0x02D5 });
    s_aglList.emplace_back("dparen"_n, AglMapping{ (AglType)1, 1, 0x249F });
    s_aglList.emplace_back("dsuperior"_n, AglMapping{ (AglType)1, 1, 0xF6EB });
    s_aglList.emplace_back("dtail"_n, AglMapping{ (AglType)1, 1, 0x0256 });
    s_aglList.emplace_back("dtopbar"_n, AglMapping{ (AglType)1, 1, 0x018C });
    s_aglList.emplace_back("duhiragana"_n, AglMapping{ (AglType)1, 1, 0x3065 });
    s_aglList.emplace_back("dukatakana"_n, AglMapping{ (AglType)1, 1, 0x30C5 });
    s_aglList.emplace_back("dz"_n, AglMapping{ (AglType)1, 1, 0x01F3 });
    s_aglList.emplace_back("dzaltone"_n, AglMapping{ (AglType)1, 1, 0x02A3 });
    s_aglList.emplace_back("dzcaron"_n, AglMapping{ (AglType)1, 1, 0x01C6 });
    s_aglList.emplace_back("dzcurl"_n, AglMapping{ (AglType)1, 1, 0x02A5 });
    s_aglList.emplace_back("dzeabkhasiancyrillic"_n, AglMapping{ (AglType)1, 1, 0x04E1 });
    s_aglList.emplace_back("dzecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0455 });
    s_aglList.emplace_back("dzhecyrillic"_n, AglMapping{ (AglType)1, 1, 0x045F });
    s_aglList.emplace_back("e"_n, AglMapping{ (AglType)11, 1, 0x0065 });
    s_aglList.emplace_back("eacute"_n, AglMapping{ (AglType)11, 1, 0x00E9 });
    s_aglList.emplace_back("earth"_n, AglMapping{ (AglType)1, 1, 0x2641 });
    s_aglList.emplace_back("ebengali"_n, AglMapping{ (AglType)1, 1, 0x098F });
    s_aglList.emplace_back("ebopomofo"_n, AglMapping{ (AglType)1, 1, 0x311C });
    s_aglList.emplace_back("ebreve"_n, AglMapping{ (AglType)3, 1, 0x0115 });
    s_aglList.emplace_back("ecandradeva"_n, AglMapping{ (AglType)1, 1, 0x090D });
    s_aglList.emplace_back("ecandragujarati"_n, AglMapping{ (AglType)1, 1, 0x0A8D });
    s_aglList.emplace_back("ecandravowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0945 });
    s_aglList.emplace_back("ecandravowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0AC5 });
    s_aglList.emplace_back("ecaron"_n, AglMapping{ (AglType)3, 1, 0x011B });
    s_aglList.emplace_back("ecedillabreve"_n, AglMapping{ (AglType)1, 1, 0x1E1D });
    s_aglList.emplace_back("echarmenian"_n, AglMapping{ (AglType)1, 1, 0x0565 });
    s_aglList.emplace_back("echyiwnarmenian"_n, AglMapping{ (AglType)1, 1, 0x0587 });
    s_aglList.emplace_back("ecircle"_n, AglMapping{ (AglType)1, 1, 0x24D4 });
    s_aglList.emplace_back("ecircumflex"_n, AglMapping{ (AglType)11, 1, 0x00EA });
    s_aglList.emplace_back("ecircumflexacute"_n, AglMapping{ (AglType)1, 1, 0x1EBF });
    s_aglList.emplace_back("ecircumflexbelow"_n, AglMapping{ (AglType)1, 1, 0x1E19 });
    s_aglList.emplace_back("ecircumflexdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EC7 });
    s_aglList.emplace_back("ecircumflexgrave"_n, AglMapping{ (AglType)1, 1, 0x1EC1 });
    s_aglList.emplace_back("ecircumflexhookabove"_n, AglMapping{ (AglType)1, 1, 0x1EC3 });
    s_aglList.emplace_back("ecircumflextilde"_n, AglMapping{ (AglType)1, 1, 0x1EC5 });
    s_aglList.emplace_back("ecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0454 });
    s_aglList.emplace_back("edblgrave"_n, AglMapping{ (AglType)1, 1, 0x0205 });
    s_aglList.emplace_back("edeva"_n, AglMapping{ (AglType)1, 1, 0x090F });
    s_aglList.emplace_back("edieresis"_n, AglMapping{ (AglType)11, 1, 0x00EB });
    s_aglList.emplace_back("edot"_n, AglMapping{ (AglType)1, 1, 0x0117 });
    s_aglList.emplace_back("edotaccent"_n, AglMapping{ (AglType)3, 1, 0x0117 });
    s_aglList.emplace_back("edotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EB9 });
    s_aglList.emplace_back("eegurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A0F });
    s_aglList.emplace_back("eematragurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A47 });
    s_aglList.emplace_back("efcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0444 });
    s_aglList.emplace_back("egrave"_n, AglMapping{ (AglType)11, 1, 0x00E8 });
    s_aglList.emplace_back("egujarati"_n, AglMapping{ (AglType)1, 1, 0x0A8F });
    s_aglList.emplace_back("eharmenian"_n, AglMapping{ (AglType)1, 1, 0x0567 });
    s_aglList.emplace_back("ehbopomofo"_n, AglMapping{ (AglType)1, 1, 0x311D });
    s_aglList.emplace_back("ehiragana"_n, AglMapping{ (AglType)1, 1, 0x3048 });
    s_aglList.emplace_back("ehookabove"_n, AglMapping{ (AglType)1, 1, 0x1EBB });
    s_aglList.emplace_back("eibopomofo"_n, AglMapping{ (AglType)1, 1, 0x311F });
    s_aglList.emplace_back("eight"_n, AglMapping{ (AglType)11, 1, 0x0038 });
    s_aglList.emplace_back("eightarabic"_n, AglMapping{ (AglType)1, 1, 0x0668 });
    s_aglList.emplace_back("eightbengali"_n, AglMapping{ (AglType)1, 1, 0x09EE });
    s_aglList.emplace_back("eightcircle"_n, AglMapping{ (AglType)1, 1, 0x2467 });
    s_aglList.emplace_back("eightcircleinversesansserif"_n, AglMapping{ (AglType)1, 1, 0x2791 });
    s_aglList.emplace_back("eightdeva"_n, AglMapping{ (AglType)1, 1, 0x096E });
    s_aglList.emplace_back("eighteencircle"_n, AglMapping{ (AglType)1, 1, 0x2471 });
    s_aglList.emplace_back("eighteenparen"_n, AglMapping{ (AglType)1, 1, 0x2485 });
    s_aglList.emplace_back("eighteenperiod"_n, AglMapping{ (AglType)1, 1, 0x2499 });
    s_aglList.emplace_back("eightgujarati"_n, AglMapping{ (AglType)1, 1, 0x0AEE });
    s_aglList.emplace_back("eightgurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A6E });
    s_aglList.emplace_back("eighthackarabic"_n, AglMapping{ (AglType)1, 1, 0x0668 });
    s_aglList.emplace_back("eighthangzhou"_n, AglMapping{ (AglType)1, 1, 0x3028 });
    s_aglList.emplace_back("eighthnotebeamed"_n, AglMapping{ (AglType)1, 1, 0x266B });
    s_aglList.emplace_back("eightideographicparen"_n, AglMapping{ (AglType)1, 1, 0x3227 });
    s_aglList.emplace_back("eightinferior"_n, AglMapping{ (AglType)1, 1, 0x2088 });
    s_aglList.emplace_back("eightmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF18 });
    s_aglList.emplace_back("eightoldstyle"_n, AglMapping{ (AglType)1, 1, 0xF738 });
    s_aglList.emplace_back("eightparen"_n, AglMapping{ (AglType)1, 1, 0x247B });
    s_aglList.emplace_back("eightperiod"_n, AglMapping{ (AglType)1, 1, 0x248F });
    s_aglList.emplace_back("eightpersian"_n, AglMapping{ (AglType)1, 1, 0x06F8 });
    s_aglList.emplace_back("eightroman"_n, AglMapping{ (AglType)1, 1, 0x2177 });
    s_aglList.emplace_back("eightsuperior"_n, AglMapping{ (AglType)1, 1, 0x2078 });
    s_aglList.emplace_back("eightthai"_n, AglMapping{ (AglType)1, 1, 0x0E58 });
    s_aglList.emplace_back("einvertedbreve"_n, AglMapping{ (AglType)1, 1, 0x0207 });
    s_aglList.emplace_back("eiotifiedcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0465 });
    s_aglList.emplace_back("ekatakana"_n, AglMapping{ (AglType)1, 1, 0x30A8 });
    s_aglList.emplace_back("ekatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF74 });
    s_aglList.emplace_back("ekonkargurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A74 });
    s_aglList.emplace_back("ekorean"_n, AglMapping{ (AglType)1, 1, 0x3154 });
    s_aglList.emplace_back("elcyrillic"_n, AglMapping{ (AglType)1, 1, 0x043B });
    s_aglList.emplace_back("element"_n, AglMapping{ (AglType)3, 1, 0x2208 });
    s_aglList.emplace_back("elevencircle"_n, AglMapping{ (AglType)1, 1, 0x246A });
    s_aglList.emplace_back("elevenparen"_n, AglMapping{ (AglType)1, 1, 0x247E });
    s_aglList.emplace_back("elevenperiod"_n, AglMapping{ (AglType)1, 1, 0x2492 });
    s_aglList.emplace_back("elevenroman"_n, AglMapping{ (AglType)1, 1, 0x217A });
    s_aglList.emplace_back("ellipsis"_n, AglMapping{ (AglType)11, 1, 0x2026 });
    s_aglList.emplace_back("ellipsisvertical"_n, AglMapping{ (AglType)1, 1, 0x22EE });
    s_aglList.emplace_back("emacron"_n, AglMapping{ (AglType)3, 1, 0x0113 });
    s_aglList.emplace_back("emacronacute"_n, AglMapping{ (AglType)1, 1, 0x1E17 });
    s_aglList.emplace_back("emacrongrave"_n, AglMapping{ (AglType)1, 1, 0x1E15 });
    s_aglList.emplace_back("emcyrillic"_n, AglMapping{ (AglType)1, 1, 0x043C });
    s_aglList.emplace_back("emdash"_n, AglMapping{ (AglType)11, 1, 0x2014 });
    s_aglList.emplace_back("emdashvertical"_n, AglMapping{ (AglType)1, 1, 0xFE31 });
    s_aglList.emplace_back("emonospace"_n, AglMapping{ (AglType)1, 1, 0xFF45 });
    s_aglList.emplace_back("emphasismarkarmenian"_n, AglMapping{ (AglType)1, 1, 0x055B });
    s_aglList.emplace_back("emptyset"_n, AglMapping{ (AglType)3, 1, 0x2205 });
    s_aglList.emplace_back("enbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3123 });
    s_aglList.emplace_back("encyrillic"_n, AglMapping{ (AglType)1, 1, 0x043D });
    s_aglList.emplace_back("endash"_n, AglMapping{ (AglType)11, 1, 0x2013 });
    s_aglList.emplace_back("endashvertical"_n, AglMapping{ (AglType)1, 1, 0xFE32 });
    s_aglList.emplace_back("endescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x04A3 });
    s_aglList.emplace_back("eng"_n, AglMapping{ (AglType)3, 1, 0x014B });
    s_aglList.emplace_back("engbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3125 });
    s_aglList.emplace_back("enghecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04A5 });
    s_aglList.emplace_back("enhookcyrillic"_n, AglMapping{ (AglType)1, 1, 0x04C8 });
    s_aglList.emplace_back("enspace"_n, AglMapping{ (AglType)1, 1, 0x2002 });
    s_aglList.emplace_back("eogonek"_n, AglMapping{ (AglType)3, 1, 0x0119 });
    s_aglList.emplace_back("eokorean"_n, AglMapping{ (AglType)1, 1, 0x3153 });
    s_aglList.emplace_back("eopen"_n, AglMapping{ (AglType)1, 1, 0x025B });
    s_aglList.emplace_back("eopenclosed"_n, AglMapping{ (AglType)1, 1, 0x029A });
    s_aglList.emplace_back("eopenreversed"_n, AglMapping{ (AglType)1, 1, 0x025C });
    s_aglList.emplace_back("eopenreversedclosed"_n, AglMapping{ (AglType)1, 1, 0x025E });
    s_aglList.emplace_back("eopenreversedhook"_n, AglMapping{ (AglType)1, 1, 0x025D });
    s_aglList.emplace_back("eparen"_n, AglMapping{ (AglType)1, 1, 0x24A0 });
    s_aglList.emplace_back("epsilon"_n, AglMapping{ (AglType)3, 1, 0x03B5 });
    s_aglList.emplace_back("epsilontonos"_n, AglMapping{ (AglType)3, 1, 0x03AD });
    s_aglList.emplace_back("equal"_n, AglMapping{ (AglType)11, 1, 0x003D });
    s_aglList.emplace_back("equalmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF1D });
    s_aglList.emplace_back("equalsmall"_n, AglMapping{ (AglType)1, 1, 0xFE66 });
    s_aglList.emplace_back("equalsuperior"_n, AglMapping{ (AglType)1, 1, 0x207C });
    s_aglList.emplace_back("equivalence"_n, AglMapping{ (AglType)3, 1, 0x2261 });
    s_aglList.emplace_back("erbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3126 });
    s_aglList.emplace_back("ercyrillic"_n, AglMapping{ (AglType)1, 1, 0x0440 });
    s_aglList.emplace_back("ereversed"_n, AglMapping{ (AglType)1, 1, 0x0258 });
    s_aglList.emplace_back("ereversedcyrillic"_n, AglMapping{ (AglType)1, 1, 0x044D });
    s_aglList.emplace_back("escyrillic"_n, AglMapping{ (AglType)1, 1, 0x0441 });
    s_aglList.emplace_back("esdescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x04AB });
    s_aglList.emplace_back("esh"_n, AglMapping{ (AglType)1, 1, 0x0283 });
    s_aglList.emplace_back("eshcurl"_n, AglMapping{ (AglType)1, 1, 0x0286 });
    s_aglList.emplace_back("eshortdeva"_n, AglMapping{ (AglType)1, 1, 0x090E });
    s_aglList.emplace_back("eshortvowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0946 });
    s_aglList.emplace_back("eshreversedloop"_n, AglMapping{ (AglType)1, 1, 0x01AA });
    s_aglList.emplace_back("eshsquatreversed"_n, AglMapping{ (AglType)1, 1, 0x0285 });
    s_aglList.emplace_back("esmallhiragana"_n, AglMapping{ (AglType)1, 1, 0x3047 });
    s_aglList.emplace_back("esmallkatakana"_n, AglMapping{ (AglType)1, 1, 0x30A7 });
    s_aglList.emplace_back("esmallkatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF6A });
    s_aglList.emplace_back("estimated"_n, AglMapping{ (AglType)3, 1, 0x212E });
    s_aglList.emplace_back("esuperior"_n, AglMapping{ (AglType)1, 1, 0xF6EC });
    s_aglList.emplace_back("eta"_n, AglMapping{ (AglType)3, 1, 0x03B7 });
    s_aglList.emplace_back("etarmenian"_n, AglMapping{ (AglType)1, 1, 0x0568 });
    s_aglList.emplace_back("etatonos"_n, AglMapping{ (AglType)3, 1, 0x03AE });
    s_aglList.emplace_back("eth"_n, AglMapping{ (AglType)11, 1, 0x00F0 });
    s_aglList.emplace_back("etilde"_n, AglMapping{ (AglType)1, 1, 0x1EBD });
    s_aglList.emplace_back("etildebelow"_n, AglMapping{ (AglType)1, 1, 0x1E1B });
    s_aglList.emplace_back("etnahtafoukhhebrew"_n, AglMapping{ (AglType)1, 1, 0x0591 });
    s_aglList.emplace_back("etnahtafoukhlefthebrew"_n, AglMapping{ (AglType)1, 1, 0x0591 });
    s_aglList.emplace_back("etnahtahebrew"_n, AglMapping{ (AglType)1, 1, 0x0591 });
    s_aglList.emplace_back("etnahtalefthebrew"_n, AglMapping{ (AglType)1, 1, 0x0591 });
    s_aglList.emplace_back("eturned"_n, AglMapping{ (AglType)1, 1, 0x01DD });
    s_aglList.emplace_back("eukorean"_n, AglMapping{ (AglType)1, 1, 0x3161 });
    s_aglList.emplace_back("euro"_n, AglMapping{ (AglType)1, 1, 0x20AC });
    s_aglList.emplace_back("evowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09C7 });
    s_aglList.emplace_back("evowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0947 });
    s_aglList.emplace_back("evowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0AC7 });
    s_aglList.emplace_back("exclam"_n, AglMapping{ (AglType)11, 1, 0x0021 });
    s_aglList.emplace_back("exclamarmenian"_n, AglMapping{ (AglType)1, 1, 0x055C });
    s_aglList.emplace_back("exclamdbl"_n, AglMapping{ (AglType)3, 1, 0x203C });
    s_aglList.emplace_back("exclamdown"_n, AglMapping{ (AglType)11, 1, 0x00A1 });
    s_aglList.emplace_back("exclamdownsmall"_n, AglMapping{ (AglType)1, 1, 0xF7A1 });
    s_aglList.emplace_back("exclammonospace"_n, AglMapping{ (AglType)1, 1, 0xFF01 });
    s_aglList.emplace_back("exclamsmall"_n, AglMapping{ (AglType)1, 1, 0xF721 });
    s_aglList.emplace_back("existential"_n, AglMapping{ (AglType)3, 1, 0x2203 });
    s_aglList.emplace_back("ezh"_n, AglMapping{ (AglType)1, 1, 0x0292 });
    s_aglList.emplace_back("ezhcaron"_n, AglMapping{ (AglType)1, 1, 0x01EF });
    s_aglList.emplace_back("ezhcurl"_n, AglMapping{ (AglType)1, 1, 0x0293 });
    s_aglList.emplace_back("ezhreversed"_n, AglMapping{ (AglType)1, 1, 0x01B9 });
    s_aglList.emplace_back("ezhtail"_n, AglMapping{ (AglType)1, 1, 0x01BA });
    s_aglList.emplace_back("f"_n, AglMapping{ (AglType)11, 1, 0x0066 });
    s_aglList.emplace_back("fadeva"_n, AglMapping{ (AglType)1, 1, 0x095E });
    s_aglList.emplace_back("fagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A5E });
    s_aglList.emplace_back("fahrenheit"_n, AglMapping{ (AglType)1, 1, 0x2109 });
    s_aglList.emplace_back("fathaarabic"_n, AglMapping{ (AglType)1, 1, 0x064E });
    s_aglList.emplace_back("fathalowarabic"_n, AglMapping{ (AglType)1, 1, 0x064E });
    s_aglList.emplace_back("fathatanarabic"_n, AglMapping{ (AglType)1, 1, 0x064B });
    s_aglList.emplace_back("fbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3108 });
    s_aglList.emplace_back("fcircle"_n, AglMapping{ (AglType)1, 1, 0x24D5 });
    s_aglList.emplace_back("fdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E1F });
    s_aglList.emplace_back("feharabic"_n, AglMapping{ (AglType)1, 1, 0x0641 });
    s_aglList.emplace_back("feharmenian"_n, AglMapping{ (AglType)1, 1, 0x0586 });
    s_aglList.emplace_back("fehfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFED2 });
    s_aglList.emplace_back("fehinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFED3 });
    s_aglList.emplace_back("fehmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFED4 });
    s_aglList.emplace_back("feicoptic"_n, AglMapping{ (AglType)1, 1, 0x03E5 });
    s_aglList.emplace_back("female"_n, AglMapping{ (AglType)3, 1, 0x2640 });
    s_aglList.emplace_back("ff"_n, AglMapping{ (AglType)1, 1, 0xFB00 });
    s_aglList.emplace_back("ffi"_n, AglMapping{ (AglType)1, 1, 0xFB03 });
    s_aglList.emplace_back("ffl"_n, AglMapping{ (AglType)1, 1, 0xFB04 });
    s_aglList.emplace_back("fi"_n, AglMapping{ (AglType)9, 1, 0xFB01 });
    s_aglList.emplace_back("fifteencircle"_n, AglMapping{ (AglType)1, 1, 0x246E });
    s_aglList.emplace_back("fifteenparen"_n, AglMapping{ (AglType)1, 1, 0x2482 });
    s_aglList.emplace_back("fifteenperiod"_n, AglMapping{ (AglType)1, 1, 0x2496 });
    s_aglList.emplace_back("figuredash"_n, AglMapping{ (AglType)3, 1, 0x2012 });
    s_aglList.emplace_back("filledbox"_n, AglMapping{ (AglType)3, 1, 0x25A0 });
    s_aglList.emplace_back("filledrect"_n, AglMapping{ (AglType)3, 1, 0x25AC });
    s_aglList.emplace_back("finalkaf"_n, AglMapping{ (AglType)1, 1, 0x05DA });
    s_aglList.emplace_back("finalkafdagesh"_n, AglMapping{ (AglType)1, 1, 0xFB3A });
    s_aglList.emplace_back("finalkafdageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB3A });
    s_aglList.emplace_back("finalkafhebrew"_n, AglMapping{ (AglType)1, 1, 0x05DA });
    s_aglList.emplace_back("finalkafqamats"_n, AglMapping{ (AglType)1, 2, 20 });
    s_aglList.emplace_back("finalkafqamatshebrew"_n, AglMapping{ (AglType)1, 2, 21 });
    s_aglList.emplace_back("finalkafsheva"_n, AglMapping{ (AglType)1, 2, 22 });
    s_aglList.emplace_back("finalkafshevahebrew"_n, AglMapping{ (AglType)1, 2, 23 });
    s_aglList.emplace_back("finalmem"_n, AglMapping{ (AglType)1, 1, 0x05DD });
    s_aglList.emplace_back("finalmemhebrew"_n, AglMapping{ (AglType)1, 1, 0x05DD });
    s_aglList.emplace_back("finalnun"_n, AglMapping{ (AglType)1, 1, 0x05DF });
    s_aglList.emplace_back("finalnunhebrew"_n, AglMapping{ (AglType)1, 1, 0x05DF });
    s_aglList.emplace_back("finalpe"_n, AglMapping{ (AglType)1, 1, 0x05E3 });
    s_aglList.emplace_back("finalpehebrew"_n, AglMapping{ (AglType)1, 1, 0x05E3 });
    s_aglList.emplace_back("finaltsadi"_n, AglMapping{ (AglType)1, 1, 0x05E5 });
    s_aglList.emplace_back("finaltsadihebrew"_n, AglMapping{ (AglType)1, 1, 0x05E5 });
    s_aglList.emplace_back("firsttonechinese"_n, AglMapping{ (AglType)1, 1, 0x02C9 });
    s_aglList.emplace_back("fisheye"_n, AglMapping{ (AglType)1, 1, 0x25C9 });
    s_aglList.emplace_back("fitacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0473 });
    s_aglList.emplace_back("five"_n, AglMapping{ (AglType)11, 1, 0x0035 });
    s_aglList.emplace_back("fivearabic"_n, AglMapping{ (AglType)1, 1, 0x0665 });
    s_aglList.emplace_back("fivebengali"_n, AglMapping{ (AglType)1, 1, 0x09EB });
    s_aglList.emplace_back("fivecircle"_n, AglMapping{ (AglType)1, 1, 0x2464 });
    s_aglList.emplace_back("fivecircleinversesansserif"_n, AglMapping{ (AglType)1, 1, 0x278E });
    s_aglList.emplace_back("fivedeva"_n, AglMapping{ (AglType)1, 1, 0x096B });
    s_aglList.emplace_back("fiveeighths"_n, AglMapping{ (AglType)3, 1, 0x215D });
    s_aglList.emplace_back("fivegujarati"_n, AglMapping{ (AglType)1, 1, 0x0AEB });
    s_aglList.emplace_back("fivegurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A6B });
    s_aglList.emplace_back("fivehackarabic"_n, AglMapping{ (AglType)1, 1, 0x0665 });
    s_aglList.emplace_back("fivehangzhou"_n, AglMapping{ (AglType)1, 1, 0x3025 });
    s_aglList.emplace_back("fiveideographicparen"_n, AglMapping{ (AglType)1, 1, 0x3224 });
    s_aglList.emplace_back("fiveinferior"_n, AglMapping{ (AglType)1, 1, 0x2085 });
    s_aglList.emplace_back("fivemonospace"_n, AglMapping{ (AglType)1, 1, 0xFF15 });
    s_aglList.emplace_back("fiveoldstyle"_n, AglMapping{ (AglType)1, 1, 0xF735 });
    s_aglList.emplace_back("fiveparen"_n, AglMapping{ (AglType)1, 1, 0x2478 });
    s_aglList.emplace_back("fiveperiod"_n, AglMapping{ (AglType)1, 1, 0x248C });
    s_aglList.emplace_back("fivepersian"_n, AglMapping{ (AglType)1, 1, 0x06F5 });
    s_aglList.emplace_back("fiveroman"_n, AglMapping{ (AglType)1, 1, 0x2174 });
    s_aglList.emplace_back("fivesuperior"_n, AglMapping{ (AglType)1, 1, 0x2075 });
    s_aglList.emplace_back("fivethai"_n, AglMapping{ (AglType)1, 1, 0x0E55 });
    s_aglList.emplace_back("fl"_n, AglMapping{ (AglType)9, 1, 0xFB02 });
    s_aglList.emplace_back("florin"_n, AglMapping{ (AglType)11, 1, 0x0192 });
    s_aglList.emplace_back("fmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF46 });
    s_aglList.emplace_back("fmsquare"_n, AglMapping{ (AglType)1, 1, 0x3399 });
    s_aglList.emplace_back("fofanthai"_n, AglMapping{ (AglType)1, 1, 0x0E1F });
    s_aglList.emplace_back("fofathai"_n, AglMapping{ (AglType)1, 1, 0x0E1D });
    s_aglList.emplace_back("fongmanthai"_n, AglMapping{ (AglType)1, 1, 0x0E4F });
    s_aglList.emplace_back("forall"_n, AglMapping{ (AglType)1, 1, 0x2200 });
    s_aglList.emplace_back("four"_n, AglMapping{ (AglType)11, 1, 0x0034 });
    s_aglList.emplace_back("fourarabic"_n, AglMapping{ (AglType)1, 1, 0x0664 });
    s_aglList.emplace_back("fourbengali"_n, AglMapping{ (AglType)1, 1, 0x09EA });
    s_aglList.emplace_back("fourcircle"_n, AglMapping{ (AglType)1, 1, 0x2463 });
    s_aglList.emplace_back("fourcircleinversesansserif"_n, AglMapping{ (AglType)1, 1, 0x278D });
    s_aglList.emplace_back("fourdeva"_n, AglMapping{ (AglType)1, 1, 0x096A });
    s_aglList.emplace_back("fourgujarati"_n, AglMapping{ (AglType)1, 1, 0x0AEA });
    s_aglList.emplace_back("fourgurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A6A });
    s_aglList.emplace_back("fourhackarabic"_n, AglMapping{ (AglType)1, 1, 0x0664 });
    s_aglList.emplace_back("fourhangzhou"_n, AglMapping{ (AglType)1, 1, 0x3024 });
    s_aglList.emplace_back("fourideographicparen"_n, AglMapping{ (AglType)1, 1, 0x3223 });
    s_aglList.emplace_back("fourinferior"_n, AglMapping{ (AglType)1, 1, 0x2084 });
    s_aglList.emplace_back("fourmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF14 });
    s_aglList.emplace_back("fournumeratorbengali"_n, AglMapping{ (AglType)1, 1, 0x09F7 });
    s_aglList.emplace_back("fouroldstyle"_n, AglMapping{ (AglType)1, 1, 0xF734 });
    s_aglList.emplace_back("fourparen"_n, AglMapping{ (AglType)1, 1, 0x2477 });
    s_aglList.emplace_back("fourperiod"_n, AglMapping{ (AglType)1, 1, 0x248B });
    s_aglList.emplace_back("fourpersian"_n, AglMapping{ (AglType)1, 1, 0x06F4 });
    s_aglList.emplace_back("fourroman"_n, AglMapping{ (AglType)1, 1, 0x2173 });
    s_aglList.emplace_back("foursuperior"_n, AglMapping{ (AglType)1, 1, 0x2074 });
    s_aglList.emplace_back("fourteencircle"_n, AglMapping{ (AglType)1, 1, 0x246D });
    s_aglList.emplace_back("fourteenparen"_n, AglMapping{ (AglType)1, 1, 0x2481 });
    s_aglList.emplace_back("fourteenperiod"_n, AglMapping{ (AglType)1, 1, 0x2495 });
    s_aglList.emplace_back("fourthai"_n, AglMapping{ (AglType)1, 1, 0x0E54 });
    s_aglList.emplace_back("fourthtonechinese"_n, AglMapping{ (AglType)1, 1, 0x02CB });
    s_aglList.emplace_back("fparen"_n, AglMapping{ (AglType)1, 1, 0x24A1 });
    s_aglList.emplace_back("fraction"_n, AglMapping{ (AglType)11, 1, 0x2044 });
    s_aglList.emplace_back("franc"_n, AglMapping{ (AglType)3, 1, 0x20A3 });
    s_aglList.emplace_back("g"_n, AglMapping{ (AglType)11, 1, 0x0067 });
    s_aglList.emplace_back("gabengali"_n, AglMapping{ (AglType)1, 1, 0x0997 });
    s_aglList.emplace_back("gacute"_n, AglMapping{ (AglType)1, 1, 0x01F5 });
    s_aglList.emplace_back("gadeva"_n, AglMapping{ (AglType)1, 1, 0x0917 });
    s_aglList.emplace_back("gafarabic"_n, AglMapping{ (AglType)1, 1, 0x06AF });
    s_aglList.emplace_back("gaffinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFB93 });
    s_aglList.emplace_back("gafinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFB94 });
    s_aglList.emplace_back("gafmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFB95 });
    s_aglList.emplace_back("gagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A97 });
    s_aglList.emplace_back("gagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A17 });
    s_aglList.emplace_back("gahiragana"_n, AglMapping{ (AglType)1, 1, 0x304C });
    s_aglList.emplace_back("gakatakana"_n, AglMapping{ (AglType)1, 1, 0x30AC });
    s_aglList.emplace_back("gamma"_n, AglMapping{ (AglType)3, 1, 0x03B3 });
    s_aglList.emplace_back("gammalatinsmall"_n, AglMapping{ (AglType)1, 1, 0x0263 });
    s_aglList.emplace_back("gammasuperior"_n, AglMapping{ (AglType)1, 1, 0x02E0 });
    s_aglList.emplace_back("gangiacoptic"_n, AglMapping{ (AglType)1, 1, 0x03EB });
    s_aglList.emplace_back("gbopomofo"_n, AglMapping{ (AglType)1, 1, 0x310D });
    s_aglList.emplace_back("gbreve"_n, AglMapping{ (AglType)3, 1, 0x011F });
    s_aglList.emplace_back("gcaron"_n, AglMapping{ (AglType)3, 1, 0x01E7 });
    s_aglList.emplace_back("gcedilla"_n, AglMapping{ (AglType)1, 1, 0x0123 });
    s_aglList.emplace_back("gcircle"_n, AglMapping{ (AglType)1, 1, 0x24D6 });
    s_aglList.emplace_back("gcircumflex"_n, AglMapping{ (AglType)3, 1, 0x011D });
    s_aglList.emplace_back("gcommaaccent"_n, AglMapping{ (AglType)1, 1, 0x0123 });
    s_aglList.emplace_back("gdot"_n, AglMapping{ (AglType)1, 1, 0x0121 });
    s_aglList.emplace_back("gdotaccent"_n, AglMapping{ (AglType)3, 1, 0x0121 });
    s_aglList.emplace_back("gecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0433 });
    s_aglList.emplace_back("gehiragana"_n, AglMapping{ (AglType)1, 1, 0x3052 });
    s_aglList.emplace_back("gekatakana"_n, AglMapping{ (AglType)1, 1, 0x30B2 });
    s_aglList.emplace_back("geometricallyequal"_n, AglMapping{ (AglType)1, 1, 0x2251 });
    s_aglList.emplace_back("gereshaccenthebrew"_n, AglMapping{ (AglType)1, 1, 0x059C });
    s_aglList.emplace_back("gereshhebrew"_n, AglMapping{ (AglType)1, 1, 0x05F3 });
    s_aglList.emplace_back("gereshmuqdamhebrew"_n, AglMapping{ (AglType)1, 1, 0x059D });
    s_aglList.emplace_back("germandbls"_n, AglMapping{ (AglType)11, 1, 0x00DF });
    s_aglList.emplace_back("gershayimaccenthebrew"_n, AglMapping{ (AglType)1, 1, 0x059E });
    s_aglList.emplace_back("gershayimhebrew"_n, AglMapping{ (AglType)1, 1, 0x05F4 });
    s_aglList.emplace_back("getamark"_n, AglMapping{ (AglType)1, 1, 0x3013 });
    s_aglList.emplace_back("ghabengali"_n, AglMapping{ (AglType)1, 1, 0x0998 });
    s_aglList.emplace_back("ghadarmenian"_n, AglMapping{ (AglType)1, 1, 0x0572 });
    s_aglList.emplace_back("ghadeva"_n, AglMapping{ (AglType)1, 1, 0x0918 });
    s_aglList.emplace_back("ghagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A98 });
    s_aglList.emplace_back("ghagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A18 });
    s_aglList.emplace_back("ghainarabic"_n, AglMapping{ (AglType)1, 1, 0x063A });
    s_aglList.emplace_back("ghainfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFECE });
    s_aglList.emplace_back("ghaininitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFECF });
    s_aglList.emplace_back("ghainmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFED0 });
    s_aglList.emplace_back("ghemiddlehookcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0495 });
    s_aglList.emplace_back("ghestrokecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0493 });
    s_aglList.emplace_back("gheupturncyrillic"_n, AglMapping{ (AglType)1, 1, 0x0491 });
    s_aglList.emplace_back("ghhadeva"_n, AglMapping{ (AglType)1, 1, 0x095A });
    s_aglList.emplace_back("ghhagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A5A });
    s_aglList.emplace_back("ghook"_n, AglMapping{ (AglType)1, 1, 0x0260 });
    s_aglList.emplace_back("ghzsquare"_n, AglMapping{ (AglType)1, 1, 0x3393 });
    s_aglList.emplace_back("gihiragana"_n, AglMapping{ (AglType)1, 1, 0x304E });
    s_aglList.emplace_back("gikatakana"_n, AglMapping{ (AglType)1, 1, 0x30AE });
    s_aglList.emplace_back("gimarmenian"_n, AglMapping{ (AglType)1, 1, 0x0563 });
    s_aglList.emplace_back("gimel"_n, AglMapping{ (AglType)1, 1, 0x05D2 });
    s_aglList.emplace_back("gimeldagesh"_n, AglMapping{ (AglType)1, 1, 0xFB32 });
    s_aglList.emplace_back("gimeldageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB32 });
    s_aglList.emplace_back("gimelhebrew"_n, AglMapping{ (AglType)1, 1, 0x05D2 });
    s_aglList.emplace_back("gjecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0453 });
    s_aglList.emplace_back("glottalinvertedstroke"_n, AglMapping{ (AglType)1, 1, 0x01BE });
    s_aglList.emplace_back("glottalstop"_n, AglMapping{ (AglType)1, 1, 0x0294 });
    s_aglList.emplace_back("glottalstopinverted"_n, AglMapping{ (AglType)1, 1, 0x0296 });
    s_aglList.emplace_back("glottalstopmod"_n, AglMapping{ (AglType)1, 1, 0x02C0 });
    s_aglList.emplace_back("glottalstopreversed"_n, AglMapping{ (AglType)1, 1, 0x0295 });
    s_aglList.emplace_back("glottalstopreversedmod"_n, AglMapping{ (AglType)1, 1, 0x02C1 });
    s_aglList.emplace_back("glottalstopreversedsuperior"_n, AglMapping{ (AglType)1, 1, 0x02E4 });
    s_aglList.emplace_back("glottalstopstroke"_n, AglMapping{ (AglType)1, 1, 0x02A1 });
    s_aglList.emplace_back("glottalstopstrokereversed"_n, AglMapping{ (AglType)1, 1, 0x02A2 });
    s_aglList.emplace_back("gmacron"_n, AglMapping{ (AglType)1, 1, 0x1E21 });
    s_aglList.emplace_back("gmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF47 });
    s_aglList.emplace_back("gohiragana"_n, AglMapping{ (AglType)1, 1, 0x3054 });
    s_aglList.emplace_back("gokatakana"_n, AglMapping{ (AglType)1, 1, 0x30B4 });
    s_aglList.emplace_back("gparen"_n, AglMapping{ (AglType)1, 1, 0x24A2 });
    s_aglList.emplace_back("gpasquare"_n, AglMapping{ (AglType)1, 1, 0x33AC });
    s_aglList.emplace_back("gradient"_n, AglMapping{ (AglType)3, 1, 0x2207 });
    s_aglList.emplace_back("grave"_n, AglMapping{ (AglType)11, 1, 0x0060 });
    s_aglList.emplace_back("gravebelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0316 });
    s_aglList.emplace_back("gravecmb"_n, AglMapping{ (AglType)1, 1, 0x0300 });
    s_aglList.emplace_back("gravecomb"_n, AglMapping{ (AglType)3, 1, 0x0300 });
    s_aglList.emplace_back("gravedeva"_n, AglMapping{ (AglType)1, 1, 0x0953 });
    s_aglList.emplace_back("gravelowmod"_n, AglMapping{ (AglType)1, 1, 0x02CE });
    s_aglList.emplace_back("gravemonospace"_n, AglMapping{ (AglType)1, 1, 0xFF40 });
    s_aglList.emplace_back("gravetonecmb"_n, AglMapping{ (AglType)1, 1, 0x0340 });
    s_aglList.emplace_back("greater"_n, AglMapping{ (AglType)11, 1, 0x003E });
    s_aglList.emplace_back("greaterequal"_n, AglMapping{ (AglType)3, 1, 0x2265 });
    s_aglList.emplace_back("greaterequalorless"_n, AglMapping{ (AglType)1, 1, 0x22DB });
    s_aglList.emplace_back("greatermonospace"_n, AglMapping{ (AglType)1, 1, 0xFF1E });
    s_aglList.emplace_back("greaterorequivalent"_n, AglMapping{ (AglType)1, 1, 0x2273 });
    s_aglList.emplace_back("greaterorless"_n, AglMapping{ (AglType)1, 1, 0x2277 });
    s_aglList.emplace_back("greateroverequal"_n, AglMapping{ (AglType)1, 1, 0x2267 });
    s_aglList.emplace_back("greatersmall"_n, AglMapping{ (AglType)1, 1, 0xFE65 });
    s_aglList.emplace_back("gscript"_n, AglMapping{ (AglType)1, 1, 0x0261 });
    s_aglList.emplace_back("gstroke"_n, AglMapping{ (AglType)1, 1, 0x01E5 });
    s_aglList.emplace_back("guhiragana"_n, AglMapping{ (AglType)1, 1, 0x3050 });
    s_aglList.emplace_back("guillemotleft"_n, AglMapping{ (AglType)11, 1, 0x00AB });
    s_aglList.emplace_back("guillemotright"_n, AglMapping{ (AglType)11, 1, 0x00BB });
    s_aglList.emplace_back("guilsinglleft"_n, AglMapping{ (AglType)11, 1, 0x2039 });
    s_aglList.emplace_back("guilsinglright"_n, AglMapping{ (AglType)11, 1, 0x203A });
    s_aglList.emplace_back("gukatakana"_n, AglMapping{ (AglType)1, 1, 0x30B0 });
    s_aglList.emplace_back("guramusquare"_n, AglMapping{ (AglType)1, 1, 0x3318 });
    s_aglList.emplace_back("gysquare"_n, AglMapping{ (AglType)1, 1, 0x33C9 });
    s_aglList.emplace_back("h"_n, AglMapping{ (AglType)11, 1, 0x0068 });
    s_aglList.emplace_back("haabkhasiancyrillic"_n, AglMapping{ (AglType)1, 1, 0x04A9 });
    s_aglList.emplace_back("haaltonearabic"_n, AglMapping{ (AglType)1, 1, 0x06C1 });
    s_aglList.emplace_back("habengali"_n, AglMapping{ (AglType)1, 1, 0x09B9 });
    s_aglList.emplace_back("hadescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x04B3 });
    s_aglList.emplace_back("hadeva"_n, AglMapping{ (AglType)1, 1, 0x0939 });
    s_aglList.emplace_back("hagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AB9 });
    s_aglList.emplace_back("hagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A39 });
    s_aglList.emplace_back("haharabic"_n, AglMapping{ (AglType)1, 1, 0x062D });
    s_aglList.emplace_back("hahfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEA2 });
    s_aglList.emplace_back("hahinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEA3 });
    s_aglList.emplace_back("hahiragana"_n, AglMapping{ (AglType)1, 1, 0x306F });
    s_aglList.emplace_back("hahmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEA4 });
    s_aglList.emplace_back("haitusquare"_n, AglMapping{ (AglType)1, 1, 0x332A });
    s_aglList.emplace_back("hakatakana"_n, AglMapping{ (AglType)1, 1, 0x30CF });
    s_aglList.emplace_back("hakatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF8A });
    s_aglList.emplace_back("halantgurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A4D });
    s_aglList.emplace_back("hamzaarabic"_n, AglMapping{ (AglType)1, 1, 0x0621 });
    s_aglList.emplace_back("hamzadammaarabic"_n, AglMapping{ (AglType)1, 2, 24 });
    s_aglList.emplace_back("hamzadammatanarabic"_n, AglMapping{ (AglType)1, 2, 25 });
    s_aglList.emplace_back("hamzafathaarabic"_n, AglMapping{ (AglType)1, 2, 26 });
    s_aglList.emplace_back("hamzafathatanarabic"_n, AglMapping{ (AglType)1, 2, 27 });
    s_aglList.emplace_back("hamzalowarabic"_n, AglMapping{ (AglType)1, 1, 0x0621 });
    s_aglList.emplace_back("hamzalowkasraarabic"_n, AglMapping{ (AglType)1, 2, 28 });
    s_aglList.emplace_back("hamzalowkasratanarabic"_n, AglMapping{ (AglType)1, 2, 29 });
    s_aglList.emplace_back("hamzasukunarabic"_n, AglMapping{ (AglType)1, 2, 30 });
    s_aglList.emplace_back("hangulfiller"_n, AglMapping{ (AglType)1, 1, 0x3164 });
    s_aglList.emplace_back("hardsigncyrillic"_n, AglMapping{ (AglType)1, 1, 0x044A });
    s_aglList.emplace_back("harpoonleftbarbup"_n, AglMapping{ (AglType)1, 1, 0x21BC });
    s_aglList.emplace_back("harpoonrightbarbup"_n, AglMapping{ (AglType)1, 1, 0x21C0 });
    s_aglList.emplace_back("hasquare"_n, AglMapping{ (AglType)1, 1, 0x33CA });
    s_aglList.emplace_back("hatafpatah"_n, AglMapping{ (AglType)1, 1, 0x05B2 });
    s_aglList.emplace_back("hatafpatah16"_n, AglMapping{ (AglType)1, 1, 0x05B2 });
    s_aglList.emplace_back("hatafpatah23"_n, AglMapping{ (AglType)1, 1, 0x05B2 });
    s_aglList.emplace_back("hatafpatah2f"_n, AglMapping{ (AglType)1, 1, 0x05B2 });
    s_aglList.emplace_back("hatafpatahhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B2 });
    s_aglList.emplace_back("hatafpatahnarrowhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B2 });
    s_aglList.emplace_back("hatafpatahquarterhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B2 });
    s_aglList.emplace_back("hatafpatahwidehebrew"_n, AglMapping{ (AglType)1, 1, 0x05B2 });
    s_aglList.emplace_back("hatafqamats"_n, AglMapping{ (AglType)1, 1, 0x05B3 });
    s_aglList.emplace_back("hatafqamats1b"_n, AglMapping{ (AglType)1, 1, 0x05B3 });
    s_aglList.emplace_back("hatafqamats28"_n, AglMapping{ (AglType)1, 1, 0x05B3 });
    s_aglList.emplace_back("hatafqamats34"_n, AglMapping{ (AglType)1, 1, 0x05B3 });
    s_aglList.emplace_back("hatafqamatshebrew"_n, AglMapping{ (AglType)1, 1, 0x05B3 });
    s_aglList.emplace_back("hatafqamatsnarrowhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B3 });
    s_aglList.emplace_back("hatafqamatsquarterhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B3 });
    s_aglList.emplace_back("hatafqamatswidehebrew"_n, AglMapping{ (AglType)1, 1, 0x05B3 });
    s_aglList.emplace_back("hatafsegol"_n, AglMapping{ (AglType)1, 1, 0x05B1 });
    s_aglList.emplace_back("hatafsegol17"_n, AglMapping{ (AglType)1, 1, 0x05B1 });
    s_aglList.emplace_back("hatafsegol24"_n, AglMapping{ (AglType)1, 1, 0x05B1 });
    s_aglList.emplace_back("hatafsegol30"_n, AglMapping{ (AglType)1, 1, 0x05B1 });
    s_aglList.emplace_back("hatafsegolhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B1 });
    s_aglList.emplace_back("hatafsegolnarrowhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B1 });
    s_aglList.emplace_back("hatafsegolquarterhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B1 });
    s_aglList.emplace_back("hatafsegolwidehebrew"_n, AglMapping{ (AglType)1, 1, 0x05B1 });
    s_aglList.emplace_back("hbar"_n, AglMapping{ (AglType)3, 1, 0x0127 });
    s_aglList.emplace_back("hbopomofo"_n, AglMapping{ (AglType)1, 1, 0x310F });
    s_aglList.emplace_back("hbrevebelow"_n, AglMapping{ (AglType)1, 1, 0x1E2B });
    s_aglList.emplace_back("hcedilla"_n, AglMapping{ (AglType)1, 1, 0x1E29 });
    s_aglList.emplace_back("hcircle"_n, AglMapping{ (AglType)1, 1, 0x24D7 });
    s_aglList.emplace_back("hcircumflex"_n, AglMapping{ (AglType)3, 1, 0x0125 });
    s_aglList.emplace_back("hdieresis"_n, AglMapping{ (AglType)1, 1, 0x1E27 });
    s_aglList.emplace_back("hdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E23 });
    s_aglList.emplace_back("hdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E25 });
    s_aglList.emplace_back("he"_n, AglMapping{ (AglType)1, 1, 0x05D4 });
    s_aglList.emplace_back("heart"_n, AglMapping{ (AglType)3, 1, 0x2665 });
    s_aglList.emplace_back("heartsuitblack"_n, AglMapping{ (AglType)1, 1, 0x2665 });
    s_aglList.emplace_back("heartsuitwhite"_n, AglMapping{ (AglType)1, 1, 0x2661 });
    s_aglList.emplace_back("hedagesh"_n, AglMapping{ (AglType)1, 1, 0xFB34 });
    s_aglList.emplace_back("hedageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB34 });
    s_aglList.emplace_back("hehaltonearabic"_n, AglMapping{ (AglType)1, 1, 0x06C1 });
    s_aglList.emplace_back("heharabic"_n, AglMapping{ (AglType)1, 1, 0x0647 });
    s_aglList.emplace_back("hehebrew"_n, AglMapping{ (AglType)1, 1, 0x05D4 });
    s_aglList.emplace_back("hehfinalaltonearabic"_n, AglMapping{ (AglType)1, 1, 0xFBA7 });
    s_aglList.emplace_back("hehfinalalttwoarabic"_n, AglMapping{ (AglType)1, 1, 0xFEEA });
    s_aglList.emplace_back("hehfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEEA });
    s_aglList.emplace_back("hehhamzaabovefinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFBA5 });
    s_aglList.emplace_back("hehhamzaaboveisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFBA4 });
    s_aglList.emplace_back("hehinitialaltonearabic"_n, AglMapping{ (AglType)1, 1, 0xFBA8 });
    s_aglList.emplace_back("hehinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEEB });
    s_aglList.emplace_back("hehiragana"_n, AglMapping{ (AglType)1, 1, 0x3078 });
    s_aglList.emplace_back("hehmedialaltonearabic"_n, AglMapping{ (AglType)1, 1, 0xFBA9 });
    s_aglList.emplace_back("hehmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEEC });
    s_aglList.emplace_back("heiseierasquare"_n, AglMapping{ (AglType)1, 1, 0x337B });
    s_aglList.emplace_back("hekatakana"_n, AglMapping{ (AglType)1, 1, 0x30D8 });
    s_aglList.emplace_back("hekatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF8D });
    s_aglList.emplace_back("hekutaarusquare"_n, AglMapping{ (AglType)1, 1, 0x3336 });
    s_aglList.emplace_back("henghook"_n, AglMapping{ (AglType)1, 1, 0x0267 });
    s_aglList.emplace_back("herutusquare"_n, AglMapping{ (AglType)1, 1, 0x3339 });
    s_aglList.emplace_back("het"_n, AglMapping{ (AglType)1, 1, 0x05D7 });
    s_aglList.emplace_back("hethebrew"_n, AglMapping{ (AglType)1, 1, 0x05D7 });
    s_aglList.emplace_back("hhook"_n, AglMapping{ (AglType)1, 1, 0x0266 });
    s_aglList.emplace_back("hhooksuperior"_n, AglMapping{ (AglType)1, 1, 0x02B1 });
    s_aglList.emplace_back("hieuhacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x327B });
    s_aglList.emplace_back("hieuhaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x321B });
    s_aglList.emplace_back("hieuhcirclekorean"_n, AglMapping{ (AglType)1, 1, 0x326D });
    s_aglList.emplace_back("hieuhkorean"_n, AglMapping{ (AglType)1, 1, 0x314E });
    s_aglList.emplace_back("hieuhparenkorean"_n, AglMapping{ (AglType)1, 1, 0x320D });
    s_aglList.emplace_back("hihiragana"_n, AglMapping{ (AglType)1, 1, 0x3072 });
    s_aglList.emplace_back("hikatakana"_n, AglMapping{ (AglType)1, 1, 0x30D2 });
    s_aglList.emplace_back("hikatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF8B });
    s_aglList.emplace_back("hiriq"_n, AglMapping{ (AglType)1, 1, 0x05B4 });
    s_aglList.emplace_back("hiriq14"_n, AglMapping{ (AglType)1, 1, 0x05B4 });
    s_aglList.emplace_back("hiriq21"_n, AglMapping{ (AglType)1, 1, 0x05B4 });
    s_aglList.emplace_back("hiriq2d"_n, AglMapping{ (AglType)1, 1, 0x05B4 });
    s_aglList.emplace_back("hiriqhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B4 });
    s_aglList.emplace_back("hiriqnarrowhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B4 });
    s_aglList.emplace_back("hiriqquarterhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B4 });
    s_aglList.emplace_back("hiriqwidehebrew"_n, AglMapping{ (AglType)1, 1, 0x05B4 });
    s_aglList.emplace_back("hlinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E96 });
    s_aglList.emplace_back("hmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF48 });
    s_aglList.emplace_back("hoarmenian"_n, AglMapping{ (AglType)1, 1, 0x0570 });
    s_aglList.emplace_back("hohipthai"_n, AglMapping{ (AglType)1, 1, 0x0E2B });
    s_aglList.emplace_back("hohiragana"_n, AglMapping{ (AglType)1, 1, 0x307B });
    s_aglList.emplace_back("hokatakana"_n, AglMapping{ (AglType)1, 1, 0x30DB });
    s_aglList.emplace_back("hokatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF8E });
    s_aglList.emplace_back("holam"_n, AglMapping{ (AglType)1, 1, 0x05B9 });
    s_aglList.emplace_back("holam19"_n, AglMapping{ (AglType)1, 1, 0x05B9 });
    s_aglList.emplace_back("holam26"_n, AglMapping{ (AglType)1, 1, 0x05B9 });
    s_aglList.emplace_back("holam32"_n, AglMapping{ (AglType)1, 1, 0x05B9 });
    s_aglList.emplace_back("holamhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B9 });
    s_aglList.emplace_back("holamnarrowhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B9 });
    s_aglList.emplace_back("holamquarterhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B9 });
    s_aglList.emplace_back("holamwidehebrew"_n, AglMapping{ (AglType)1, 1, 0x05B9 });
    s_aglList.emplace_back("honokhukthai"_n, AglMapping{ (AglType)1, 1, 0x0E2E });
    s_aglList.emplace_back("hookabovecomb"_n, AglMapping{ (AglType)3, 1, 0x0309 });
    s_aglList.emplace_back("hookcmb"_n, AglMapping{ (AglType)1, 1, 0x0309 });
    s_aglList.emplace_back("hookpalatalizedbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0321 });
    s_aglList.emplace_back("hookretroflexbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0322 });
    s_aglList.emplace_back("hoonsquare"_n, AglMapping{ (AglType)1, 1, 0x3342 });
    s_aglList.emplace_back("horicoptic"_n, AglMapping{ (AglType)1, 1, 0x03E9 });
    s_aglList.emplace_back("horizontalbar"_n, AglMapping{ (AglType)1, 1, 0x2015 });
    s_aglList.emplace_back("horncmb"_n, AglMapping{ (AglType)1, 1, 0x031B });
    s_aglList.emplace_back("hotsprings"_n, AglMapping{ (AglType)1, 1, 0x2668 });
    s_aglList.emplace_back("house"_n, AglMapping{ (AglType)3, 1, 0x2302 });
    s_aglList.emplace_back("hparen"_n, AglMapping{ (AglType)1, 1, 0x24A3 });
    s_aglList.emplace_back("hsuperior"_n, AglMapping{ (AglType)1, 1, 0x02B0 });
    s_aglList.emplace_back("hturned"_n, AglMapping{ (AglType)1, 1, 0x0265 });
    s_aglList.emplace_back("huhiragana"_n, AglMapping{ (AglType)1, 1, 0x3075 });
    s_aglList.emplace_back("huiitosquare"_n, AglMapping{ (AglType)1, 1, 0x3333 });
    s_aglList.emplace_back("hukatakana"_n, AglMapping{ (AglType)1, 1, 0x30D5 });
    s_aglList.emplace_back("hukatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF8C });
    s_aglList.emplace_back("hungarumlaut"_n, AglMapping{ (AglType)11, 1, 0x02DD });
    s_aglList.emplace_back("hungarumlautcmb"_n, AglMapping{ (AglType)1, 1, 0x030B });
    s_aglList.emplace_back("hv"_n, AglMapping{ (AglType)1, 1, 0x0195 });
    s_aglList.emplace_back("hyphen"_n, AglMapping{ (AglType)11, 1, 0x002D });
    s_aglList.emplace_back("hypheninferior"_n, AglMapping{ (AglType)1, 1, 0xF6E5 });
    s_aglList.emplace_back("hyphenmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF0D });
    s_aglList.emplace_back("hyphensmall"_n, AglMapping{ (AglType)1, 1, 0xFE63 });
    s_aglList.emplace_back("hyphensuperior"_n, AglMapping{ (AglType)1, 1, 0xF6E6 });
    s_aglList.emplace_back("hyphentwo"_n, AglMapping{ (AglType)1, 1, 0x2010 });
    s_aglList.emplace_back("i"_n, AglMapping{ (AglType)11, 1, 0x0069 });
    s_aglList.emplace_back("iacute"_n, AglMapping{ (AglType)11, 1, 0x00ED });
    s_aglList.emplace_back("iacyrillic"_n, AglMapping{ (AglType)1, 1, 0x044F });
    s_aglList.emplace_back("ibengali"_n, AglMapping{ (AglType)1, 1, 0x0987 });
    s_aglList.emplace_back("ibopomofo"_n, AglMapping{ (AglType)1, 1, 0x3127 });
    s_aglList.emplace_back("ibreve"_n, AglMapping{ (AglType)3, 1, 0x012D });
    s_aglList.emplace_back("icaron"_n, AglMapping{ (AglType)1, 1, 0x01D0 });
    s_aglList.emplace_back("icircle"_n, AglMapping{ (AglType)1, 1, 0x24D8 });
    s_aglList.emplace_back("icircumflex"_n, AglMapping{ (AglType)11, 1, 0x00EE });
    s_aglList.emplace_back("icyrillic"_n, AglMapping{ (AglType)1, 1, 0x0456 });
    s_aglList.emplace_back("idblgrave"_n, AglMapping{ (AglType)1, 1, 0x0209 });
    s_aglList.emplace_back("ideographearthcircle"_n, AglMapping{ (AglType)1, 1, 0x328F });
    s_aglList.emplace_back("ideographfirecircle"_n, AglMapping{ (AglType)1, 1, 0x328B });
    s_aglList.emplace_back("ideographicallianceparen"_n, AglMapping{ (AglType)1, 1, 0x323F });
    s_aglList.emplace_back("ideographiccallparen"_n, AglMapping{ (AglType)1, 1, 0x323A });
    s_aglList.emplace_back("ideographiccentrecircle"_n, AglMapping{ (AglType)1, 1, 0x32A5 });
    s_aglList.emplace_back("ideographicclose"_n, AglMapping{ (AglType)1, 1, 0x3006 });
    s_aglList.emplace_back("ideographiccomma"_n, AglMapping{ (AglType)1, 1, 0x3001 });
    s_aglList.emplace_back("ideographiccommaleft"_n, AglMapping{ (AglType)1, 1, 0xFF64 });
    s_aglList.emplace_back("ideographiccongratulationparen"_n, AglMapping{ (AglType)1, 1, 0x3237 });
    s_aglList.emplace_back("ideographiccorrectcircle"_n, AglMapping{ (AglType)1, 1, 0x32A3 });
    s_aglList.emplace_back("ideographicearthparen"_n, AglMapping{ (AglType)1, 1, 0x322F });
    s_aglList.emplace_back("ideographicenterpriseparen"_n, AglMapping{ (AglType)1, 1, 0x323D });
    s_aglList.emplace_back("ideographicexcellentcircle"_n, AglMapping{ (AglType)1, 1, 0x329D });
    s_aglList.emplace_back("ideographicfestivalparen"_n, AglMapping{ (AglType)1, 1, 0x3240 });
    s_aglList.emplace_back("ideographicfinancialcircle"_n, AglMapping{ (AglType)1, 1, 0x3296 });
    s_aglList.emplace_back("ideographicfinancialparen"_n, AglMapping{ (AglType)1, 1, 0x3236 });
    s_aglList.emplace_back("ideographicfireparen"_n, AglMapping{ (AglType)1, 1, 0x322B });
    s_aglList.emplace_back("ideographichaveparen"_n, AglMapping{ (AglType)1, 1, 0x3232 });
    s_aglList.emplace_back("ideographichighcircle"_n, AglMapping{ (AglType)1, 1, 0x32A4 });
    s_aglList.emplace_back("ideographiciterationmark"_n, AglMapping{ (AglType)1, 1, 0x3005 });
    s_aglList.emplace_back("ideographiclaborcircle"_n, AglMapping{ (AglType)1, 1, 0x3298 });
    s_aglList.emplace_back("ideographiclaborparen"_n, AglMapping{ (AglType)1, 1, 0x3238 });
    s_aglList.emplace_back("ideographicleftcircle"_n, AglMapping{ (AglType)1, 1, 0x32A7 });
    s_aglList.emplace_back("ideographiclowcircle"_n, AglMapping{ (AglType)1, 1, 0x32A6 });
    s_aglList.emplace_back("ideographicmedicinecircle"_n, AglMapping{ (AglType)1, 1, 0x32A9 });
    s_aglList.emplace_back("ideographicmetalparen"_n, AglMapping{ (AglType)1, 1, 0x322E });
    s_aglList.emplace_back("ideographicmoonparen"_n, AglMapping{ (AglType)1, 1, 0x322A });
    s_aglList.emplace_back("ideographicnameparen"_n, AglMapping{ (AglType)1, 1, 0x3234 });
    s_aglList.emplace_back("ideographicperiod"_n, AglMapping{ (AglType)1, 1, 0x3002 });
    s_aglList.emplace_back("ideographicprintcircle"_n, AglMapping{ (AglType)1, 1, 0x329E });
    s_aglList.emplace_back("ideographicreachparen"_n, AglMapping{ (AglType)1, 1, 0x3243 });
    s_aglList.emplace_back("ideographicrepresentparen"_n, AglMapping{ (AglType)1, 1, 0x3239 });
    s_aglList.emplace_back("ideographicresourceparen"_n, AglMapping{ (AglType)1, 1, 0x323E });
    s_aglList.emplace_back("ideographicrightcircle"_n, AglMapping{ (AglType)1, 1, 0x32A8 });
    s_aglList.emplace_back("ideographicsecretcircle"_n, AglMapping{ (AglType)1, 1, 0x3299 });
    s_aglList.emplace_back("ideographicselfparen"_n, AglMapping{ (AglType)1, 1, 0x3242 });
    s_aglList.emplace_back("ideographicsocietyparen"_n, AglMapping{ (AglType)1, 1, 0x3233 });
    s_aglList.emplace_back("ideographicspace"_n, AglMapping{ (AglType)1, 1, 0x3000 });
    s_aglList.emplace_back("ideographicspecialparen"_n, AglMapping{ (AglType)1, 1, 0x3235 });
    s_aglList.emplace_back("ideographicstockparen"_n, AglMapping{ (AglType)1, 1, 0x3231 });
    s_aglList.emplace_back("ideographicstudyparen"_n, AglMapping{ (AglType)1, 1, 0x323B });
    s_aglList.emplace_back("ideographicsunparen"_n, AglMapping{ (AglType)1, 1, 0x3230 });
    s_aglList.emplace_back("ideographicsuperviseparen"_n, AglMapping{ (AglType)1, 1, 0x323C });
    s_aglList.emplace_back("ideographicwaterparen"_n, AglMapping{ (AglType)1, 1, 0x322C });
    s_aglList.emplace_back("ideographicwoodparen"_n, AglMapping{ (AglType)1, 1, 0x322D });
    s_aglList.emplace_back("ideographiczero"_n, AglMapping{ (AglType)1, 1, 0x3007 });
    s_aglList.emplace_back("ideographmetalcircle"_n, AglMapping{ (AglType)1, 1, 0x328E });
    s_aglList.emplace_back("ideographmooncircle"_n, AglMapping{ (AglType)1, 1, 0x328A });
    s_aglList.emplace_back("ideographnamecircle"_n, AglMapping{ (AglType)1, 1, 0x3294 });
    s_aglList.emplace_back("ideographsuncircle"_n, AglMapping{ (AglType)1, 1, 0x3290 });
    s_aglList.emplace_back("ideographwatercircle"_n, AglMapping{ (AglType)1, 1, 0x328C });
    s_aglList.emplace_back("ideographwoodcircle"_n, AglMapping{ (AglType)1, 1, 0x328D });
    s_aglList.emplace_back("ideva"_n, AglMapping{ (AglType)1, 1, 0x0907 });
    s_aglList.emplace_back("idieresis"_n, AglMapping{ (AglType)11, 1, 0x00EF });
    s_aglList.emplace_back("idieresisacute"_n, AglMapping{ (AglType)1, 1, 0x1E2F });
    s_aglList.emplace_back("idieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04E5 });
    s_aglList.emplace_back("idotbelow"_n, AglMapping{ (AglType)1, 1, 0x1ECB });
    s_aglList.emplace_back("iebrevecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04D7 });
    s_aglList.emplace_back("iecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0435 });
    s_aglList.emplace_back("ieungacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3275 });
    s_aglList.emplace_back("ieungaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3215 });
    s_aglList.emplace_back("ieungcirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3267 });
    s_aglList.emplace_back("ieungkorean"_n, AglMapping{ (AglType)1, 1, 0x3147 });
    s_aglList.emplace_back("ieungparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3207 });
    s_aglList.emplace_back("igrave"_n, AglMapping{ (AglType)11, 1, 0x00EC });
    s_aglList.emplace_back("igujarati"_n, AglMapping{ (AglType)1, 1, 0x0A87 });
    s_aglList.emplace_back("igurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A07 });
    s_aglList.emplace_back("ihiragana"_n, AglMapping{ (AglType)1, 1, 0x3044 });
    s_aglList.emplace_back("ihookabove"_n, AglMapping{ (AglType)1, 1, 0x1EC9 });
    s_aglList.emplace_back("iibengali"_n, AglMapping{ (AglType)1, 1, 0x0988 });
    s_aglList.emplace_back("iicyrillic"_n, AglMapping{ (AglType)1, 1, 0x0438 });
    s_aglList.emplace_back("iideva"_n, AglMapping{ (AglType)1, 1, 0x0908 });
    s_aglList.emplace_back("iigujarati"_n, AglMapping{ (AglType)1, 1, 0x0A88 });
    s_aglList.emplace_back("iigurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A08 });
    s_aglList.emplace_back("iimatragurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A40 });
    s_aglList.emplace_back("iinvertedbreve"_n, AglMapping{ (AglType)1, 1, 0x020B });
    s_aglList.emplace_back("iishortcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0439 });
    s_aglList.emplace_back("iivowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09C0 });
    s_aglList.emplace_back("iivowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0940 });
    s_aglList.emplace_back("iivowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0AC0 });
    s_aglList.emplace_back("ij"_n, AglMapping{ (AglType)3, 1, 0x0133 });
    s_aglList.emplace_back("ikatakana"_n, AglMapping{ (AglType)1, 1, 0x30A4 });
    s_aglList.emplace_back("ikatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF72 });
    s_aglList.emplace_back("ikorean"_n, AglMapping{ (AglType)1, 1, 0x3163 });
    s_aglList.emplace_back("ilde"_n, AglMapping{ (AglType)1, 1, 0x02DC });
    s_aglList.emplace_back("iluyhebrew"_n, AglMapping{ (AglType)1, 1, 0x05AC });
    s_aglList.emplace_back("imacron"_n, AglMapping{ (AglType)3, 1, 0x012B });
    s_aglList.emplace_back("imacroncyrillic"_n, AglMapping{ (AglType)1, 1, 0x04E3 });
    s_aglList.emplace_back("imageorapproximatelyequal"_n, AglMapping{ (AglType)1, 1, 0x2253 });
    s_aglList.emplace_back("imatragurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A3F });
    s_aglList.emplace_back("imonospace"_n, AglMapping{ (AglType)1, 1, 0xFF49 });
    s_aglList.emplace_back("increment"_n, AglMapping{ (AglType)1, 1, 0x2206 });
    s_aglList.emplace_back("infinity"_n, AglMapping{ (AglType)3, 1, 0x221E });
    s_aglList.emplace_back("iniarmenian"_n, AglMapping{ (AglType)1, 1, 0x056B });
    s_aglList.emplace_back("integral"_n, AglMapping{ (AglType)3, 1, 0x222B });
    s_aglList.emplace_back("integralbottom"_n, AglMapping{ (AglType)1, 1, 0x2321 });
    s_aglList.emplace_back("integralbt"_n, AglMapping{ (AglType)3, 1, 0x2321 });
    s_aglList.emplace_back("integralex"_n, AglMapping{ (AglType)1, 1, 0xF8F5 });
    s_aglList.emplace_back("integraltop"_n, AglMapping{ (AglType)1, 1, 0x2320 });
    s_aglList.emplace_back("integraltp"_n, AglMapping{ (AglType)3, 1, 0x2320 });
    s_aglList.emplace_back("intersection"_n, AglMapping{ (AglType)3, 1, 0x2229 });
    s_aglList.emplace_back("intisquare"_n, AglMapping{ (AglType)1, 1, 0x3305 });
    s_aglList.emplace_back("invbullet"_n, AglMapping{ (AglType)3, 1, 0x25D8 });
    s_aglList.emplace_back("invcircle"_n, AglMapping{ (AglType)3, 1, 0x25D9 });
    s_aglList.emplace_back("invsmileface"_n, AglMapping{ (AglType)3, 1, 0x263B });
    s_aglList.emplace_back("iocyrillic"_n, AglMapping{ (AglType)1, 1, 0x0451 });
    s_aglList.emplace_back("iogonek"_n, AglMapping{ (AglType)3, 1, 0x012F });
    s_aglList.emplace_back("iota"_n, AglMapping{ (AglType)3, 1, 0x03B9 });
    s_aglList.emplace_back("iotadieresis"_n, AglMapping{ (AglType)3, 1, 0x03CA });
    s_aglList.emplace_back("iotadieresistonos"_n, AglMapping{ (AglType)3, 1, 0x0390 });
    s_aglList.emplace_back("iotalatin"_n, AglMapping{ (AglType)1, 1, 0x0269 });
    s_aglList.emplace_back("iotatonos"_n, AglMapping{ (AglType)3, 1, 0x03AF });
    s_aglList.emplace_back("iparen"_n, AglMapping{ (AglType)1, 1, 0x24A4 });
    s_aglList.emplace_back("irigurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A72 });
    s_aglList.emplace_back("ismallhiragana"_n, AglMapping{ (AglType)1, 1, 0x3043 });
    s_aglList.emplace_back("ismallkatakana"_n, AglMapping{ (AglType)1, 1, 0x30A3 });
    s_aglList.emplace_back("ismallkatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF68 });
    s_aglList.emplace_back("issharbengali"_n, AglMapping{ (AglType)1, 1, 0x09FA });
    s_aglList.emplace_back("istroke"_n, AglMapping{ (AglType)1, 1, 0x0268 });
    s_aglList.emplace_back("isuperior"_n, AglMapping{ (AglType)1, 1, 0xF6ED });
    s_aglList.emplace_back("iterationhiragana"_n, AglMapping{ (AglType)1, 1, 0x309D });
    s_aglList.emplace_back("iterationkatakana"_n, AglMapping{ (AglType)1, 1, 0x30FD });
    s_aglList.emplace_back("itilde"_n, AglMapping{ (AglType)3, 1, 0x0129 });
    s_aglList.emplace_back("itildebelow"_n, AglMapping{ (AglType)1, 1, 0x1E2D });
    s_aglList.emplace_back("iubopomofo"_n, AglMapping{ (AglType)1, 1, 0x3129 });
    s_aglList.emplace_back("iucyrillic"_n, AglMapping{ (AglType)1, 1, 0x044E });
    s_aglList.emplace_back("ivowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09BF });
    s_aglList.emplace_back("ivowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x093F });
    s_aglList.emplace_back("ivowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0ABF });
    s_aglList.emplace_back("izhitsacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0475 });
    s_aglList.emplace_back("izhitsadblgravecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0477 });
    s_aglList.emplace_back("j"_n, AglMapping{ (AglType)11, 1, 0x006A });
    s_aglList.emplace_back("jaarmenian"_n, AglMapping{ (AglType)1, 1, 0x0571 });
    s_aglList.emplace_back("jabengali"_n, AglMapping{ (AglType)1, 1, 0x099C });
    s_aglList.emplace_back("jadeva"_n, AglMapping{ (AglType)1, 1, 0x091C });
    s_aglList.emplace_back("jagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A9C });
    s_aglList.emplace_back("jagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A1C });
    s_aglList.emplace_back("jbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3110 });
    s_aglList.emplace_back("jcaron"_n, AglMapping{ (AglType)1, 1, 0x01F0 });
    s_aglList.emplace_back("jcircle"_n, AglMapping{ (AglType)1, 1, 0x24D9 });
    s_aglList.emplace_back("jcircumflex"_n, AglMapping{ (AglType)3, 1, 0x0135 });
    s_aglList.emplace_back("jcrossedtail"_n, AglMapping{ (AglType)1, 1, 0x029D });
    s_aglList.emplace_back("jdotlessstroke"_n, AglMapping{ (AglType)1, 1, 0x025F });
    s_aglList.emplace_back("jecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0458 });
    s_aglList.emplace_back("jeemarabic"_n, AglMapping{ (AglType)1, 1, 0x062C });
    s_aglList.emplace_back("jeemfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFE9E });
    s_aglList.emplace_back("jeeminitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFE9F });
    s_aglList.emplace_back("jeemmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEA0 });
    s_aglList.emplace_back("jeharabic"_n, AglMapping{ (AglType)1, 1, 0x0698 });
    s_aglList.emplace_back("jehfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFB8B });
    s_aglList.emplace_back("jhabengali"_n, AglMapping{ (AglType)1, 1, 0x099D });
    s_aglList.emplace_back("jhadeva"_n, AglMapping{ (AglType)1, 1, 0x091D });
    s_aglList.emplace_back("jhagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A9D });
    s_aglList.emplace_back("jhagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A1D });
    s_aglList.emplace_back("jheharmenian"_n, AglMapping{ (AglType)1, 1, 0x057B });
    s_aglList.emplace_back("jis"_n, AglMapping{ (AglType)1, 1, 0x3004 });
    s_aglList.emplace_back("jmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF4A });
    s_aglList.emplace_back("jparen"_n, AglMapping{ (AglType)1, 1, 0x24A5 });
    s_aglList.emplace_back("jsuperior"_n, AglMapping{ (AglType)1, 1, 0x02B2 });
    s_aglList.emplace_back("k"_n, AglMapping{ (AglType)11, 1, 0x006B });
    s_aglList.emplace_back("kabashkircyrillic"_n, AglMapping{ (AglType)1, 1, 0x04A1 });
    s_aglList.emplace_back("kabengali"_n, AglMapping{ (AglType)1, 1, 0x0995 });
    s_aglList.emplace_back("kacute"_n, AglMapping{ (AglType)1, 1, 0x1E31 });
    s_aglList.emplace_back("kacyrillic"_n, AglMapping{ (AglType)1, 1, 0x043A });
    s_aglList.emplace_back("kadescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x049B });
    s_aglList.emplace_back("kadeva"_n, AglMapping{ (AglType)1, 1, 0x0915 });
    s_aglList.emplace_back("kaf"_n, AglMapping{ (AglType)1, 1, 0x05DB });
    s_aglList.emplace_back("kafarabic"_n, AglMapping{ (AglType)1, 1, 0x0643 });
    s_aglList.emplace_back("kafdagesh"_n, AglMapping{ (AglType)1, 1, 0xFB3B });
    s_aglList.emplace_back("kafdageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB3B });
    s_aglList.emplace_back("kaffinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEDA });
    s_aglList.emplace_back("kafhebrew"_n, AglMapping{ (AglType)1, 1, 0x05DB });
    s_aglList.emplace_back("kafinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEDB });
    s_aglList.emplace_back("kafmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEDC });
    s_aglList.emplace_back("kafrafehebrew"_n, AglMapping{ (AglType)1, 1, 0xFB4D });
    s_aglList.emplace_back("kagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A95 });
    s_aglList.emplace_back("kagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A15 });
    s_aglList.emplace_back("kahiragana"_n, AglMapping{ (AglType)1, 1, 0x304B });
    s_aglList.emplace_back("kahookcyrillic"_n, AglMapping{ (AglType)1, 1, 0x04C4 });
    s_aglList.emplace_back("kakatakana"_n, AglMapping{ (AglType)1, 1, 0x30AB });
    s_aglList.emplace_back("kakatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF76 });
    s_aglList.emplace_back("kappa"_n, AglMapping{ (AglType)3, 1, 0x03BA });
    s_aglList.emplace_back("kappasymbolgreek"_n, AglMapping{ (AglType)1, 1, 0x03F0 });
    s_aglList.emplace_back("kapyeounmieumkorean"_n, AglMapping{ (AglType)1, 1, 0x3171 });
    s_aglList.emplace_back("kapyeounphieuphkorean"_n, AglMapping{ (AglType)1, 1, 0x3184 });
    s_aglList.emplace_back("kapyeounpieupkorean"_n, AglMapping{ (AglType)1, 1, 0x3178 });
    s_aglList.emplace_back("kapyeounssangpieupkorean"_n, AglMapping{ (AglType)1, 1, 0x3179 });
    s_aglList.emplace_back("karoriisquare"_n, AglMapping{ (AglType)1, 1, 0x330D });
    s_aglList.emplace_back("kashidaautoarabic"_n, AglMapping{ (AglType)1, 1, 0x0640 });
    s_aglList.emplace_back("kashidaautonosidebearingarabic"_n, AglMapping{ (AglType)1, 1, 0x0640 });
    s_aglList.emplace_back("kasmallkatakana"_n, AglMapping{ (AglType)1, 1, 0x30F5 });
    s_aglList.emplace_back("kasquare"_n, AglMapping{ (AglType)1, 1, 0x3384 });
    s_aglList.emplace_back("kasraarabic"_n, AglMapping{ (AglType)1, 1, 0x0650 });
    s_aglList.emplace_back("kasratanarabic"_n, AglMapping{ (AglType)1, 1, 0x064D });
    s_aglList.emplace_back("kastrokecyrillic"_n, AglMapping{ (AglType)1, 1, 0x049F });
    s_aglList.emplace_back("katahiraprolongmarkhalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF70 });
    s_aglList.emplace_back("kaverticalstrokecyrillic"_n, AglMapping{ (AglType)1, 1, 0x049D });
    s_aglList.emplace_back("kbopomofo"_n, AglMapping{ (AglType)1, 1, 0x310E });
    s_aglList.emplace_back("kcalsquare"_n, AglMapping{ (AglType)1, 1, 0x3389 });
    s_aglList.emplace_back("kcaron"_n, AglMapping{ (AglType)1, 1, 0x01E9 });
    s_aglList.emplace_back("kcedilla"_n, AglMapping{ (AglType)1, 1, 0x0137 });
    s_aglList.emplace_back("kcircle"_n, AglMapping{ (AglType)1, 1, 0x24DA });
    s_aglList.emplace_back("kcommaaccent"_n, AglMapping{ (AglType)1, 1, 0x0137 });
    s_aglList.emplace_back("kdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E33 });
    s_aglList.emplace_back("keharmenian"_n, AglMapping{ (AglType)1, 1, 0x0584 });
    s_aglList.emplace_back("kehiragana"_n, AglMapping{ (AglType)1, 1, 0x3051 });
    s_aglList.emplace_back("kekatakana"_n, AglMapping{ (AglType)1, 1, 0x30B1 });
    s_aglList.emplace_back("kekatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF79 });
    s_aglList.emplace_back("kenarmenian"_n, AglMapping{ (AglType)1, 1, 0x056F });
    s_aglList.emplace_back("kesmallkatakana"_n, AglMapping{ (AglType)1, 1, 0x30F6 });
    s_aglList.emplace_back("kgreenlandic"_n, AglMapping{ (AglType)3, 1, 0x0138 });
    s_aglList.emplace_back("khabengali"_n, AglMapping{ (AglType)1, 1, 0x0996 });
    s_aglList.emplace_back("khacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0445 });
    s_aglList.emplace_back("khadeva"_n, AglMapping{ (AglType)1, 1, 0x0916 });
    s_aglList.emplace_back("khagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A96 });
    s_aglList.emplace_back("khagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A16 });
    s_aglList.emplace_back("khaharabic"_n, AglMapping{ (AglType)1, 1, 0x062E });
    s_aglList.emplace_back("khahfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEA6 });
    s_aglList.emplace_back("khahinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEA7 });
    s_aglList.emplace_back("khahmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEA8 });
    s_aglList.emplace_back("kheicoptic"_n, AglMapping{ (AglType)1, 1, 0x03E7 });
    s_aglList.emplace_back("khhadeva"_n, AglMapping{ (AglType)1, 1, 0x0959 });
    s_aglList.emplace_back("khhagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A59 });
    s_aglList.emplace_back("khieukhacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3278 });
    s_aglList.emplace_back("khieukhaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3218 });
    s_aglList.emplace_back("khieukhcirclekorean"_n, AglMapping{ (AglType)1, 1, 0x326A });
    s_aglList.emplace_back("khieukhkorean"_n, AglMapping{ (AglType)1, 1, 0x314B });
    s_aglList.emplace_back("khieukhparenkorean"_n, AglMapping{ (AglType)1, 1, 0x320A });
    s_aglList.emplace_back("khokhaithai"_n, AglMapping{ (AglType)1, 1, 0x0E02 });
    s_aglList.emplace_back("khokhonthai"_n, AglMapping{ (AglType)1, 1, 0x0E05 });
    s_aglList.emplace_back("khokhuatthai"_n, AglMapping{ (AglType)1, 1, 0x0E03 });
    s_aglList.emplace_back("khokhwaithai"_n, AglMapping{ (AglType)1, 1, 0x0E04 });
    s_aglList.emplace_back("khomutthai"_n, AglMapping{ (AglType)1, 1, 0x0E5B });
    s_aglList.emplace_back("khook"_n, AglMapping{ (AglType)1, 1, 0x0199 });
    s_aglList.emplace_back("khorakhangthai"_n, AglMapping{ (AglType)1, 1, 0x0E06 });
    s_aglList.emplace_back("khzsquare"_n, AglMapping{ (AglType)1, 1, 0x3391 });
    s_aglList.emplace_back("kihiragana"_n, AglMapping{ (AglType)1, 1, 0x304D });
    s_aglList.emplace_back("kikatakana"_n, AglMapping{ (AglType)1, 1, 0x30AD });
    s_aglList.emplace_back("kikatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF77 });
    s_aglList.emplace_back("kiroguramusquare"_n, AglMapping{ (AglType)1, 1, 0x3315 });
    s_aglList.emplace_back("kiromeetorusquare"_n, AglMapping{ (AglType)1, 1, 0x3316 });
    s_aglList.emplace_back("kirosquare"_n, AglMapping{ (AglType)1, 1, 0x3314 });
    s_aglList.emplace_back("kiyeokacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x326E });
    s_aglList.emplace_back("kiyeokaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x320E });
    s_aglList.emplace_back("kiyeokcirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3260 });
    s_aglList.emplace_back("kiyeokkorean"_n, AglMapping{ (AglType)1, 1, 0x3131 });
    s_aglList.emplace_back("kiyeokparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3200 });
    s_aglList.emplace_back("kiyeoksioskorean"_n, AglMapping{ (AglType)1, 1, 0x3133 });
    s_aglList.emplace_back("kjecyrillic"_n, AglMapping{ (AglType)1, 1, 0x045C });
    s_aglList.emplace_back("klinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E35 });
    s_aglList.emplace_back("klsquare"_n, AglMapping{ (AglType)1, 1, 0x3398 });
    s_aglList.emplace_back("kmcubedsquare"_n, AglMapping{ (AglType)1, 1, 0x33A6 });
    s_aglList.emplace_back("kmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF4B });
    s_aglList.emplace_back("kmsquaredsquare"_n, AglMapping{ (AglType)1, 1, 0x33A2 });
    s_aglList.emplace_back("kohiragana"_n, AglMapping{ (AglType)1, 1, 0x3053 });
    s_aglList.emplace_back("kohmsquare"_n, AglMapping{ (AglType)1, 1, 0x33C0 });
    s_aglList.emplace_back("kokaithai"_n, AglMapping{ (AglType)1, 1, 0x0E01 });
    s_aglList.emplace_back("kokatakana"_n, AglMapping{ (AglType)1, 1, 0x30B3 });
    s_aglList.emplace_back("kokatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF7A });
    s_aglList.emplace_back("kooposquare"_n, AglMapping{ (AglType)1, 1, 0x331E });
    s_aglList.emplace_back("koppacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0481 });
    s_aglList.emplace_back("koreanstandardsymbol"_n, AglMapping{ (AglType)1, 1, 0x327F });
    s_aglList.emplace_back("koroniscmb"_n, AglMapping{ (AglType)1, 1, 0x0343 });
    s_aglList.emplace_back("kparen"_n, AglMapping{ (AglType)1, 1, 0x24A6 });
    s_aglList.emplace_back("kpasquare"_n, AglMapping{ (AglType)1, 1, 0x33AA });
    s_aglList.emplace_back("ksicyrillic"_n, AglMapping{ (AglType)1, 1, 0x046F });
    s_aglList.emplace_back("ktsquare"_n, AglMapping{ (AglType)1, 1, 0x33CF });
    s_aglList.emplace_back("kturned"_n, AglMapping{ (AglType)1, 1, 0x029E });
    s_aglList.emplace_back("kuhiragana"_n, AglMapping{ (AglType)1, 1, 0x304F });
    s_aglList.emplace_back("kukatakana"_n, AglMapping{ (AglType)1, 1, 0x30AF });
    s_aglList.emplace_back("kukatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF78 });
    s_aglList.emplace_back("kvsquare"_n, AglMapping{ (AglType)1, 1, 0x33B8 });
    s_aglList.emplace_back("kwsquare"_n, AglMapping{ (AglType)1, 1, 0x33BE });
    s_aglList.emplace_back("l"_n, AglMapping{ (AglType)11, 1, 0x006C });
    s_aglList.emplace_back("labengali"_n, AglMapping{ (AglType)1, 1, 0x09B2 });
    s_aglList.emplace_back("lacute"_n, AglMapping{ (AglType)3, 1, 0x013A });
    s_aglList.emplace_back("ladeva"_n, AglMapping{ (AglType)1, 1, 0x0932 });
    s_aglList.emplace_back("lagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AB2 });
    s_aglList.emplace_back("lagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A32 });
    s_aglList.emplace_back("lakkhangyaothai"_n, AglMapping{ (AglType)1, 1, 0x0E45 });
    s_aglList.emplace_back("lamaleffinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEFC });
    s_aglList.emplace_back("lamalefhamzaabovefinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEF8 });
    s_aglList.emplace_back("lamalefhamzaaboveisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFEF7 });
    s_aglList.emplace_back("lamalefhamzabelowfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEFA });
    s_aglList.emplace_back("lamalefhamzabelowisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFEF9 });
    s_aglList.emplace_back("lamalefisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFEFB });
    s_aglList.emplace_back("lamalefmaddaabovefinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEF6 });
    s_aglList.emplace_back("lamalefmaddaaboveisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFEF5 });
    s_aglList.emplace_back("lamarabic"_n, AglMapping{ (AglType)1, 1, 0x0644 });
    s_aglList.emplace_back("lambda"_n, AglMapping{ (AglType)3, 1, 0x03BB });
    s_aglList.emplace_back("lambdastroke"_n, AglMapping{ (AglType)1, 1, 0x019B });
    s_aglList.emplace_back("lamed"_n, AglMapping{ (AglType)1, 1, 0x05DC });
    s_aglList.emplace_back("lameddagesh"_n, AglMapping{ (AglType)1, 1, 0xFB3C });
    s_aglList.emplace_back("lameddageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB3C });
    s_aglList.emplace_back("lamedhebrew"_n, AglMapping{ (AglType)1, 1, 0x05DC });
    s_aglList.emplace_back("lamedholam"_n, AglMapping{ (AglType)1, 2, 31 });
    s_aglList.emplace_back("lamedholamdagesh"_n, AglMapping{ (AglType)1, 3, 32 });
    s_aglList.emplace_back("lamedholamdageshhebrew"_n, AglMapping{ (AglType)1, 3, 33 });
    s_aglList.emplace_back("lamedholamhebrew"_n, AglMapping{ (AglType)1, 2, 34 });
    s_aglList.emplace_back("lamfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEDE });
    s_aglList.emplace_back("lamhahinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFCCA });
    s_aglList.emplace_back("laminitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEDF });
    s_aglList.emplace_back("lamjeeminitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFCC9 });
    s_aglList.emplace_back("lamkhahinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFCCB });
    s_aglList.emplace_back("lamlamhehisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFDF2 });
    s_aglList.emplace_back("lammedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEE0 });
    s_aglList.emplace_back("lammeemhahinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFD88 });
    s_aglList.emplace_back("lammeeminitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFCCC });
    s_aglList.emplace_back("lammeemjeeminitialarabic"_n, AglMapping{ (AglType)1, 3, 35 });
    s_aglList.emplace_back("lammeemkhahinitialarabic"_n, AglMapping{ (AglType)1, 3, 36 });
    s_aglList.emplace_back("largecircle"_n, AglMapping{ (AglType)1, 1, 0x25EF });
    s_aglList.emplace_back("lbar"_n, AglMapping{ (AglType)1, 1, 0x019A });
    s_aglList.emplace_back("lbelt"_n, AglMapping{ (AglType)1, 1, 0x026C });
    s_aglList.emplace_back("lbopomofo"_n, AglMapping{ (AglType)1, 1, 0x310C });
    s_aglList.emplace_back("lcaron"_n, AglMapping{ (AglType)3, 1, 0x013E });
    s_aglList.emplace_back("lcedilla"_n, AglMapping{ (AglType)1, 1, 0x013C });
    s_aglList.emplace_back("lcircle"_n, AglMapping{ (AglType)1, 1, 0x24DB });
    s_aglList.emplace_back("lcircumflexbelow"_n, AglMapping{ (AglType)1, 1, 0x1E3D });
    s_aglList.emplace_back("lcommaaccent"_n, AglMapping{ (AglType)1, 1, 0x013C });
    s_aglList.emplace_back("ldot"_n, AglMapping{ (AglType)3, 1, 0x0140 });
    s_aglList.emplace_back("ldotaccent"_n, AglMapping{ (AglType)1, 1, 0x0140 });
    s_aglList.emplace_back("ldotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E37 });
    s_aglList.emplace_back("ldotbelowmacron"_n, AglMapping{ (AglType)1, 1, 0x1E39 });
    s_aglList.emplace_back("leftangleabovecmb"_n, AglMapping{ (AglType)1, 1, 0x031A });
    s_aglList.emplace_back("lefttackbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0318 });
    s_aglList.emplace_back("less"_n, AglMapping{ (AglType)11, 1, 0x003C });
    s_aglList.emplace_back("lessequal"_n, AglMapping{ (AglType)3, 1, 0x2264 });
    s_aglList.emplace_back("lessequalorgreater"_n, AglMapping{ (AglType)1, 1, 0x22DA });
    s_aglList.emplace_back("lessmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF1C });
    s_aglList.emplace_back("lessorequivalent"_n, AglMapping{ (AglType)1, 1, 0x2272 });
    s_aglList.emplace_back("lessorgreater"_n, AglMapping{ (AglType)1, 1, 0x2276 });
    s_aglList.emplace_back("lessoverequal"_n, AglMapping{ (AglType)1, 1, 0x2266 });
    s_aglList.emplace_back("lesssmall"_n, AglMapping{ (AglType)1, 1, 0xFE64 });
    s_aglList.emplace_back("lezh"_n, AglMapping{ (AglType)1, 1, 0x026E });
    s_aglList.emplace_back("lfblock"_n, AglMapping{ (AglType)3, 1, 0x258C });
    s_aglList.emplace_back("lhookretroflex"_n, AglMapping{ (AglType)1, 1, 0x026D });
    s_aglList.emplace_back("lira"_n, AglMapping{ (AglType)3, 1, 0x20A4 });
    s_aglList.emplace_back("liwnarmenian"_n, AglMapping{ (AglType)1, 1, 0x056C });
    s_aglList.emplace_back("lj"_n, AglMapping{ (AglType)1, 1, 0x01C9 });
    s_aglList.emplace_back("ljecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0459 });
    s_aglList.emplace_back("ll"_n, AglMapping{ (AglType)1, 1, 0xF6C0 });
    s_aglList.emplace_back("lladeva"_n, AglMapping{ (AglType)1, 1, 0x0933 });
    s_aglList.emplace_back("llagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AB3 });
    s_aglList.emplace_back("llinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E3B });
    s_aglList.emplace_back("llladeva"_n, AglMapping{ (AglType)1, 1, 0x0934 });
    s_aglList.emplace_back("llvocalicbengali"_n, AglMapping{ (AglType)1, 1, 0x09E1 });
    s_aglList.emplace_back("llvocalicdeva"_n, AglMapping{ (AglType)1, 1, 0x0961 });
    s_aglList.emplace_back("llvocalicvowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09E3 });
    s_aglList.emplace_back("llvocalicvowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0963 });
    s_aglList.emplace_back("lmiddletilde"_n, AglMapping{ (AglType)1, 1, 0x026B });
    s_aglList.emplace_back("lmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF4C });
    s_aglList.emplace_back("lmsquare"_n, AglMapping{ (AglType)1, 1, 0x33D0 });
    s_aglList.emplace_back("lochulathai"_n, AglMapping{ (AglType)1, 1, 0x0E2C });
    s_aglList.emplace_back("logicaland"_n, AglMapping{ (AglType)3, 1, 0x2227 });
    s_aglList.emplace_back("logicalnot"_n, AglMapping{ (AglType)11, 1, 0x00AC });
    s_aglList.emplace_back("logicalnotreversed"_n, AglMapping{ (AglType)1, 1, 0x2310 });
    s_aglList.emplace_back("logicalor"_n, AglMapping{ (AglType)3, 1, 0x2228 });
    s_aglList.emplace_back("lolingthai"_n, AglMapping{ (AglType)1, 1, 0x0E25 });
    s_aglList.emplace_back("longs"_n, AglMapping{ (AglType)3, 1, 0x017F });
    s_aglList.emplace_back("lowlinecenterline"_n, AglMapping{ (AglType)1, 1, 0xFE4E });
    s_aglList.emplace_back("lowlinecmb"_n, AglMapping{ (AglType)1, 1, 0x0332 });
    s_aglList.emplace_back("lowlinedashed"_n, AglMapping{ (AglType)1, 1, 0xFE4D });
    s_aglList.emplace_back("lozenge"_n, AglMapping{ (AglType)3, 1, 0x25CA });
    s_aglList.emplace_back("lparen"_n, AglMapping{ (AglType)1, 1, 0x24A7 });
    s_aglList.emplace_back("lslash"_n, AglMapping{ (AglType)11, 1, 0x0142 });
    s_aglList.emplace_back("lsquare"_n, AglMapping{ (AglType)1, 1, 0x2113 });
    s_aglList.emplace_back("lsuperior"_n, AglMapping{ (AglType)1, 1, 0xF6EE });
    s_aglList.emplace_back("ltshade"_n, AglMapping{ (AglType)3, 1, 0x2591 });
    s_aglList.emplace_back("luthai"_n, AglMapping{ (AglType)1, 1, 0x0E26 });
    s_aglList.emplace_back("lvocalicbengali"_n, AglMapping{ (AglType)1, 1, 0x098C });
    s_aglList.emplace_back("lvocalicdeva"_n, AglMapping{ (AglType)1, 1, 0x090C });
    s_aglList.emplace_back("lvocalicvowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09E2 });
    s_aglList.emplace_back("lvocalicvowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0962 });
    s_aglList.emplace_back("lxsquare"_n, AglMapping{ (AglType)1, 1, 0x33D3 });
    s_aglList.emplace_back("m"_n, AglMapping{ (AglType)11, 1, 0x006D });
    s_aglList.emplace_back("mabengali"_n, AglMapping{ (AglType)1, 1, 0x09AE });
    s_aglList.emplace_back("macron"_n, AglMapping{ (AglType)11, 1, 0x00AF });
    s_aglList.emplace_back("macronbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0331 });
    s_aglList.emplace_back("macroncmb"_n, AglMapping{ (AglType)1, 1, 0x0304 });
    s_aglList.emplace_back("macronlowmod"_n, AglMapping{ (AglType)1, 1, 0x02CD });
    s_aglList.emplace_back("macronmonospace"_n, AglMapping{ (AglType)1, 1, 0xFFE3 });
    s_aglList.emplace_back("macute"_n, AglMapping{ (AglType)1, 1, 0x1E3F });
    s_aglList.emplace_back("madeva"_n, AglMapping{ (AglType)1, 1, 0x092E });
    s_aglList.emplace_back("magujarati"_n, AglMapping{ (AglType)1, 1, 0x0AAE });
    s_aglList.emplace_back("magurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A2E });
    s_aglList.emplace_back("mahapakhhebrew"_n, AglMapping{ (AglType)1, 1, 0x05A4 });
    s_aglList.emplace_back("mahapakhlefthebrew"_n, AglMapping{ (AglType)1, 1, 0x05A4 });
    s_aglList.emplace_back("mahiragana"_n, AglMapping{ (AglType)1, 1, 0x307E });
    s_aglList.emplace_back("maichattawalowleftthai"_n, AglMapping{ (AglType)1, 1, 0xF895 });
    s_aglList.emplace_back("maichattawalowrightthai"_n, AglMapping{ (AglType)1, 1, 0xF894 });
    s_aglList.emplace_back("maichattawathai"_n, AglMapping{ (AglType)1, 1, 0x0E4B });
    s_aglList.emplace_back("maichattawaupperleftthai"_n, AglMapping{ (AglType)1, 1, 0xF893 });
    s_aglList.emplace_back("maieklowleftthai"_n, AglMapping{ (AglType)1, 1, 0xF88C });
    s_aglList.emplace_back("maieklowrightthai"_n, AglMapping{ (AglType)1, 1, 0xF88B });
    s_aglList.emplace_back("maiekthai"_n, AglMapping{ (AglType)1, 1, 0x0E48 });
    s_aglList.emplace_back("maiekupperleftthai"_n, AglMapping{ (AglType)1, 1, 0xF88A });
    s_aglList.emplace_back("maihanakatleftthai"_n, AglMapping{ (AglType)1, 1, 0xF884 });
    s_aglList.emplace_back("maihanakatthai"_n, AglMapping{ (AglType)1, 1, 0x0E31 });
    s_aglList.emplace_back("maitaikhuleftthai"_n, AglMapping{ (AglType)1, 1, 0xF889 });
    s_aglList.emplace_back("maitaikhuthai"_n, AglMapping{ (AglType)1, 1, 0x0E47 });
    s_aglList.emplace_back("maitholowleftthai"_n, AglMapping{ (AglType)1, 1, 0xF88F });
    s_aglList.emplace_back("maitholowrightthai"_n, AglMapping{ (AglType)1, 1, 0xF88E });
    s_aglList.emplace_back("maithothai"_n, AglMapping{ (AglType)1, 1, 0x0E49 });
    s_aglList.emplace_back("maithoupperleftthai"_n, AglMapping{ (AglType)1, 1, 0xF88D });
    s_aglList.emplace_back("maitrilowleftthai"_n, AglMapping{ (AglType)1, 1, 0xF892 });
    s_aglList.emplace_back("maitrilowrightthai"_n, AglMapping{ (AglType)1, 1, 0xF891 });
    s_aglList.emplace_back("maitrithai"_n, AglMapping{ (AglType)1, 1, 0x0E4A });
    s_aglList.emplace_back("maitriupperleftthai"_n, AglMapping{ (AglType)1, 1, 0xF890 });
    s_aglList.emplace_back("maiyamokthai"_n, AglMapping{ (AglType)1, 1, 0x0E46 });
    s_aglList.emplace_back("makatakana"_n, AglMapping{ (AglType)1, 1, 0x30DE });
    s_aglList.emplace_back("makatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF8F });
    s_aglList.emplace_back("male"_n, AglMapping{ (AglType)3, 1, 0x2642 });
    s_aglList.emplace_back("mansyonsquare"_n, AglMapping{ (AglType)1, 1, 0x3347 });
    s_aglList.emplace_back("maqafhebrew"_n, AglMapping{ (AglType)1, 1, 0x05BE });
    s_aglList.emplace_back("mars"_n, AglMapping{ (AglType)1, 1, 0x2642 });
    s_aglList.emplace_back("masoracirclehebrew"_n, AglMapping{ (AglType)1, 1, 0x05AF });
    s_aglList.emplace_back("masquare"_n, AglMapping{ (AglType)1, 1, 0x3383 });
    s_aglList.emplace_back("mbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3107 });
    s_aglList.emplace_back("mbsquare"_n, AglMapping{ (AglType)1, 1, 0x33D4 });
    s_aglList.emplace_back("mcircle"_n, AglMapping{ (AglType)1, 1, 0x24DC });
    s_aglList.emplace_back("mcubedsquare"_n, AglMapping{ (AglType)1, 1, 0x33A5 });
    s_aglList.emplace_back("mdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E41 });
    s_aglList.emplace_back("mdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E43 });
    s_aglList.emplace_back("meemarabic"_n, AglMapping{ (AglType)1, 1, 0x0645 });
    s_aglList.emplace_back("meemfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEE2 });
    s_aglList.emplace_back("meeminitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEE3 });
    s_aglList.emplace_back("meemmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEE4 });
    s_aglList.emplace_back("meemmeeminitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFCD1 });
    s_aglList.emplace_back("meemmeemisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFC48 });
    s_aglList.emplace_back("meetorusquare"_n, AglMapping{ (AglType)1, 1, 0x334D });
    s_aglList.emplace_back("mehiragana"_n, AglMapping{ (AglType)1, 1, 0x3081 });
    s_aglList.emplace_back("meizierasquare"_n, AglMapping{ (AglType)1, 1, 0x337E });
    s_aglList.emplace_back("mekatakana"_n, AglMapping{ (AglType)1, 1, 0x30E1 });
    s_aglList.emplace_back("mekatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF92 });
    s_aglList.emplace_back("mem"_n, AglMapping{ (AglType)1, 1, 0x05DE });
    s_aglList.emplace_back("memdagesh"_n, AglMapping{ (AglType)1, 1, 0xFB3E });
    s_aglList.emplace_back("memdageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB3E });
    s_aglList.emplace_back("memhebrew"_n, AglMapping{ (AglType)1, 1, 0x05DE });
    s_aglList.emplace_back("menarmenian"_n, AglMapping{ (AglType)1, 1, 0x0574 });
    s_aglList.emplace_back("merkhahebrew"_n, AglMapping{ (AglType)1, 1, 0x05A5 });
    s_aglList.emplace_back("merkhakefulahebrew"_n, AglMapping{ (AglType)1, 1, 0x05A6 });
    s_aglList.emplace_back("merkhakefulalefthebrew"_n, AglMapping{ (AglType)1, 1, 0x05A6 });
    s_aglList.emplace_back("merkhalefthebrew"_n, AglMapping{ (AglType)1, 1, 0x05A5 });
    s_aglList.emplace_back("mhook"_n, AglMapping{ (AglType)1, 1, 0x0271 });
    s_aglList.emplace_back("mhzsquare"_n, AglMapping{ (AglType)1, 1, 0x3392 });
    s_aglList.emplace_back("middledotkatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF65 });
    s_aglList.emplace_back("middot"_n, AglMapping{ (AglType)1, 1, 0x00B7 });
    s_aglList.emplace_back("mieumacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3272 });
    s_aglList.emplace_back("mieumaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3212 });
    s_aglList.emplace_back("mieumcirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3264 });
    s_aglList.emplace_back("mieumkorean"_n, AglMapping{ (AglType)1, 1, 0x3141 });
    s_aglList.emplace_back("mieumpansioskorean"_n, AglMapping{ (AglType)1, 1, 0x3170 });
    s_aglList.emplace_back("mieumparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3204 });
    s_aglList.emplace_back("mieumpieupkorean"_n, AglMapping{ (AglType)1, 1, 0x316E });
    s_aglList.emplace_back("mieumsioskorean"_n, AglMapping{ (AglType)1, 1, 0x316F });
    s_aglList.emplace_back("mihiragana"_n, AglMapping{ (AglType)1, 1, 0x307F });
    s_aglList.emplace_back("mikatakana"_n, AglMapping{ (AglType)1, 1, 0x30DF });
    s_aglList.emplace_back("mikatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF90 });
    s_aglList.emplace_back("minus"_n, AglMapping{ (AglType)11, 1, 0x2212 });
    s_aglList.emplace_back("minusbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0320 });
    s_aglList.emplace_back("minuscircle"_n, AglMapping{ (AglType)1, 1, 0x2296 });
    s_aglList.emplace_back("minusmod"_n, AglMapping{ (AglType)1, 1, 0x02D7 });
    s_aglList.emplace_back("minusplus"_n, AglMapping{ (AglType)1, 1, 0x2213 });
    s_aglList.emplace_back("minute"_n, AglMapping{ (AglType)3, 1, 0x2032 });
    s_aglList.emplace_back("miribaarusquare"_n, AglMapping{ (AglType)1, 1, 0x334A });
    s_aglList.emplace_back("mirisquare"_n, AglMapping{ (AglType)1, 1, 0x3349 });
    s_aglList.emplace_back("mlonglegturned"_n, AglMapping{ (AglType)1, 1, 0x0270 });
    s_aglList.emplace_back("mlsquare"_n, AglMapping{ (AglType)1, 1, 0x3396 });
    s_aglList.emplace_back("mmcubedsquare"_n, AglMapping{ (AglType)1, 1, 0x33A3 });
    s_aglList.emplace_back("mmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF4D });
    s_aglList.emplace_back("mmsquaredsquare"_n, AglMapping{ (AglType)1, 1, 0x339F });
    s_aglList.emplace_back("mohiragana"_n, AglMapping{ (AglType)1, 1, 0x3082 });
    s_aglList.emplace_back("mohmsquare"_n, AglMapping{ (AglType)1, 1, 0x33C1 });
    s_aglList.emplace_back("mokatakana"_n, AglMapping{ (AglType)1, 1, 0x30E2 });
    s_aglList.emplace_back("mokatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF93 });
    s_aglList.emplace_back("molsquare"_n, AglMapping{ (AglType)1, 1, 0x33D6 });
    s_aglList.emplace_back("momathai"_n, AglMapping{ (AglType)1, 1, 0x0E21 });
    s_aglList.emplace_back("moverssquare"_n, AglMapping{ (AglType)1, 1, 0x33A7 });
    s_aglList.emplace_back("moverssquaredsquare"_n, AglMapping{ (AglType)1, 1, 0x33A8 });
    s_aglList.emplace_back("mparen"_n, AglMapping{ (AglType)1, 1, 0x24A8 });
    s_aglList.emplace_back("mpasquare"_n, AglMapping{ (AglType)1, 1, 0x33AB });
    s_aglList.emplace_back("mssquare"_n, AglMapping{ (AglType)1, 1, 0x33B3 });
    s_aglList.emplace_back("msuperior"_n, AglMapping{ (AglType)1, 1, 0xF6EF });
    s_aglList.emplace_back("mturned"_n, AglMapping{ (AglType)1, 1, 0x026F });
    s_aglList.emplace_back("mu"_n, AglMapping{ (AglType)11, 1, 0x00B5 });
    s_aglList.emplace_back("mu1"_n, AglMapping{ (AglType)1, 1, 0x00B5 });
    s_aglList.emplace_back("muasquare"_n, AglMapping{ (AglType)1, 1, 0x3382 });
    s_aglList.emplace_back("muchgreater"_n, AglMapping{ (AglType)1, 1, 0x226B });
    s_aglList.emplace_back("muchless"_n, AglMapping{ (AglType)1, 1, 0x226A });
    s_aglList.emplace_back("mufsquare"_n, AglMapping{ (AglType)1, 1, 0x338C });
    s_aglList.emplace_back("mugreek"_n, AglMapping{ (AglType)1, 1, 0x03BC });
    s_aglList.emplace_back("mugsquare"_n, AglMapping{ (AglType)1, 1, 0x338D });
    s_aglList.emplace_back("muhiragana"_n, AglMapping{ (AglType)1, 1, 0x3080 });
    s_aglList.emplace_back("mukatakana"_n, AglMapping{ (AglType)1, 1, 0x30E0 });
    s_aglList.emplace_back("mukatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF91 });
    s_aglList.emplace_back("mulsquare"_n, AglMapping{ (AglType)1, 1, 0x3395 });
    s_aglList.emplace_back("multiply"_n, AglMapping{ (AglType)11, 1, 0x00D7 });
    s_aglList.emplace_back("mumsquare"_n, AglMapping{ (AglType)1, 1, 0x339B });
    s_aglList.emplace_back("munahhebrew"_n, AglMapping{ (AglType)1, 1, 0x05A3 });
    s_aglList.emplace_back("munahlefthebrew"_n, AglMapping{ (AglType)1, 1, 0x05A3 });
    s_aglList.emplace_back("musicalnote"_n, AglMapping{ (AglType)3, 1, 0x266A });
    s_aglList.emplace_back("musicalnotedbl"_n, AglMapping{ (AglType)3, 1, 0x266B });
    s_aglList.emplace_back("musicflatsign"_n, AglMapping{ (AglType)1, 1, 0x266D });
    s_aglList.emplace_back("musicsharpsign"_n, AglMapping{ (AglType)1, 1, 0x266F });
    s_aglList.emplace_back("mussquare"_n, AglMapping{ (AglType)1, 1, 0x33B2 });
    s_aglList.emplace_back("muvsquare"_n, AglMapping{ (AglType)1, 1, 0x33B6 });
    s_aglList.emplace_back("muwsquare"_n, AglMapping{ (AglType)1, 1, 0x33BC });
    s_aglList.emplace_back("mvmegasquare"_n, AglMapping{ (AglType)1, 1, 0x33B9 });
    s_aglList.emplace_back("mvsquare"_n, AglMapping{ (AglType)1, 1, 0x33B7 });
    s_aglList.emplace_back("mwmegasquare"_n, AglMapping{ (AglType)1, 1, 0x33BF });
    s_aglList.emplace_back("mwsquare"_n, AglMapping{ (AglType)1, 1, 0x33BD });
    s_aglList.emplace_back("n"_n, AglMapping{ (AglType)11, 1, 0x006E });
    s_aglList.emplace_back("nabengali"_n, AglMapping{ (AglType)1, 1, 0x09A8 });
    s_aglList.emplace_back("nabla"_n, AglMapping{ (AglType)1, 1, 0x2207 });
    s_aglList.emplace_back("nacute"_n, AglMapping{ (AglType)3, 1, 0x0144 });
    s_aglList.emplace_back("nadeva"_n, AglMapping{ (AglType)1, 1, 0x0928 });
    s_aglList.emplace_back("nagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AA8 });
    s_aglList.emplace_back("nagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A28 });
    s_aglList.emplace_back("nahiragana"_n, AglMapping{ (AglType)1, 1, 0x306A });
    s_aglList.emplace_back("nakatakana"_n, AglMapping{ (AglType)1, 1, 0x30CA });
    s_aglList.emplace_back("nakatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF85 });
    s_aglList.emplace_back("napostrophe"_n, AglMapping{ (AglType)3, 1, 0x0149 });
    s_aglList.emplace_back("nasquare"_n, AglMapping{ (AglType)1, 1, 0x3381 });
    s_aglList.emplace_back("nbopomofo"_n, AglMapping{ (AglType)1, 1, 0x310B });
    s_aglList.emplace_back("nbspace"_n, AglMapping{ (AglType)1, 1, 0x00A0 });
    s_aglList.emplace_back("ncaron"_n, AglMapping{ (AglType)3, 1, 0x0148 });
    s_aglList.emplace_back("ncedilla"_n, AglMapping{ (AglType)1, 1, 0x0146 });
    s_aglList.emplace_back("ncircle"_n, AglMapping{ (AglType)1, 1, 0x24DD });
    s_aglList.emplace_back("ncircumflexbelow"_n, AglMapping{ (AglType)1, 1, 0x1E4B });
    s_aglList.emplace_back("ncommaaccent"_n, AglMapping{ (AglType)1, 1, 0x0146 });
    s_aglList.emplace_back("ndotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E45 });
    s_aglList.emplace_back("ndotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E47 });
    s_aglList.emplace_back("nehiragana"_n, AglMapping{ (AglType)1, 1, 0x306D });
    s_aglList.emplace_back("nekatakana"_n, AglMapping{ (AglType)1, 1, 0x30CD });
    s_aglList.emplace_back("nekatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF88 });
    s_aglList.emplace_back("newsheqelsign"_n, AglMapping{ (AglType)1, 1, 0x20AA });
    s_aglList.emplace_back("nfsquare"_n, AglMapping{ (AglType)1, 1, 0x338B });
    s_aglList.emplace_back("ngabengali"_n, AglMapping{ (AglType)1, 1, 0x0999 });
    s_aglList.emplace_back("ngadeva"_n, AglMapping{ (AglType)1, 1, 0x0919 });
    s_aglList.emplace_back("ngagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A99 });
    s_aglList.emplace_back("ngagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A19 });
    s_aglList.emplace_back("ngonguthai"_n, AglMapping{ (AglType)1, 1, 0x0E07 });
    s_aglList.emplace_back("nhiragana"_n, AglMapping{ (AglType)1, 1, 0x3093 });
    s_aglList.emplace_back("nhookleft"_n, AglMapping{ (AglType)1, 1, 0x0272 });
    s_aglList.emplace_back("nhookretroflex"_n, AglMapping{ (AglType)1, 1, 0x0273 });
    s_aglList.emplace_back("nieunacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x326F });
    s_aglList.emplace_back("nieunaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x320F });
    s_aglList.emplace_back("nieuncieuckorean"_n, AglMapping{ (AglType)1, 1, 0x3135 });
    s_aglList.emplace_back("nieuncirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3261 });
    s_aglList.emplace_back("nieunhieuhkorean"_n, AglMapping{ (AglType)1, 1, 0x3136 });
    s_aglList.emplace_back("nieunkorean"_n, AglMapping{ (AglType)1, 1, 0x3134 });
    s_aglList.emplace_back("nieunpansioskorean"_n, AglMapping{ (AglType)1, 1, 0x3168 });
    s_aglList.emplace_back("nieunparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3201 });
    s_aglList.emplace_back("nieunsioskorean"_n, AglMapping{ (AglType)1, 1, 0x3167 });
    s_aglList.emplace_back("nieuntikeutkorean"_n, AglMapping{ (AglType)1, 1, 0x3166 });
    s_aglList.emplace_back("nihiragana"_n, AglMapping{ (AglType)1, 1, 0x306B });
    s_aglList.emplace_back("nikatakana"_n, AglMapping{ (AglType)1, 1, 0x30CB });
    s_aglList.emplace_back("nikatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF86 });
    s_aglList.emplace_back("nikhahitleftthai"_n, AglMapping{ (AglType)1, 1, 0xF899 });
    s_aglList.emplace_back("nikhahitthai"_n, AglMapping{ (AglType)1, 1, 0x0E4D });
    s_aglList.emplace_back("nine"_n, AglMapping{ (AglType)11, 1, 0x0039 });
    s_aglList.emplace_back("ninearabic"_n, AglMapping{ (AglType)1, 1, 0x0669 });
    s_aglList.emplace_back("ninebengali"_n, AglMapping{ (AglType)1, 1, 0x09EF });
    s_aglList.emplace_back("ninecircle"_n, AglMapping{ (AglType)1, 1, 0x2468 });
    s_aglList.emplace_back("ninecircleinversesansserif"_n, AglMapping{ (AglType)1, 1, 0x2792 });
    s_aglList.emplace_back("ninedeva"_n, AglMapping{ (AglType)1, 1, 0x096F });
    s_aglList.emplace_back("ninegujarati"_n, AglMapping{ (AglType)1, 1, 0x0AEF });
    s_aglList.emplace_back("ninegurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A6F });
    s_aglList.emplace_back("ninehackarabic"_n, AglMapping{ (AglType)1, 1, 0x0669 });
    s_aglList.emplace_back("ninehangzhou"_n, AglMapping{ (AglType)1, 1, 0x3029 });
    s_aglList.emplace_back("nineideographicparen"_n, AglMapping{ (AglType)1, 1, 0x3228 });
    s_aglList.emplace_back("nineinferior"_n, AglMapping{ (AglType)1, 1, 0x2089 });
    s_aglList.emplace_back("ninemonospace"_n, AglMapping{ (AglType)1, 1, 0xFF19 });
    s_aglList.emplace_back("nineoldstyle"_n, AglMapping{ (AglType)1, 1, 0xF739 });
    s_aglList.emplace_back("nineparen"_n, AglMapping{ (AglType)1, 1, 0x247C });
    s_aglList.emplace_back("nineperiod"_n, AglMapping{ (AglType)1, 1, 0x2490 });
    s_aglList.emplace_back("ninepersian"_n, AglMapping{ (AglType)1, 1, 0x06F9 });
    s_aglList.emplace_back("nineroman"_n, AglMapping{ (AglType)1, 1, 0x2178 });
    s_aglList.emplace_back("ninesuperior"_n, AglMapping{ (AglType)1, 1, 0x2079 });
    s_aglList.emplace_back("nineteencircle"_n, AglMapping{ (AglType)1, 1, 0x2472 });
    s_aglList.emplace_back("nineteenparen"_n, AglMapping{ (AglType)1, 1, 0x2486 });
    s_aglList.emplace_back("nineteenperiod"_n, AglMapping{ (AglType)1, 1, 0x249A });
    s_aglList.emplace_back("ninethai"_n, AglMapping{ (AglType)1, 1, 0x0E59 });
    s_aglList.emplace_back("nj"_n, AglMapping{ (AglType)1, 1, 0x01CC });
    s_aglList.emplace_back("njecyrillic"_n, AglMapping{ (AglType)1, 1, 0x045A });
    s_aglList.emplace_back("nkatakana"_n, AglMapping{ (AglType)1, 1, 0x30F3 });
    s_aglList.emplace_back("nkatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF9D });
    s_aglList.emplace_back("nlegrightlong"_n, AglMapping{ (AglType)1, 1, 0x019E });
    s_aglList.emplace_back("nlinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E49 });
    s_aglList.emplace_back("nmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF4E });
    s_aglList.emplace_back("nmsquare"_n, AglMapping{ (AglType)1, 1, 0x339A });
    s_aglList.emplace_back("nnabengali"_n, AglMapping{ (AglType)1, 1, 0x09A3 });
    s_aglList.emplace_back("nnadeva"_n, AglMapping{ (AglType)1, 1, 0x0923 });
    s_aglList.emplace_back("nnagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AA3 });
    s_aglList.emplace_back("nnagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A23 });
    s_aglList.emplace_back("nnnadeva"_n, AglMapping{ (AglType)1, 1, 0x0929 });
    s_aglList.emplace_back("nohiragana"_n, AglMapping{ (AglType)1, 1, 0x306E });
    s_aglList.emplace_back("nokatakana"_n, AglMapping{ (AglType)1, 1, 0x30CE });
    s_aglList.emplace_back("nokatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF89 });
    s_aglList.emplace_back("nonbreakingspace"_n, AglMapping{ (AglType)1, 1, 0x00A0 });
    s_aglList.emplace_back("nonenthai"_n, AglMapping{ (AglType)1, 1, 0x0E13 });
    s_aglList.emplace_back("nonuthai"_n, AglMapping{ (AglType)1, 1, 0x0E19 });
    s_aglList.emplace_back("noonarabic"_n, AglMapping{ (AglType)1, 1, 0x0646 });
    s_aglList.emplace_back("noonfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEE6 });
    s_aglList.emplace_back("noonghunnaarabic"_n, AglMapping{ (AglType)1, 1, 0x06BA });
    s_aglList.emplace_back("noonghunnafinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFB9F });
    s_aglList.emplace_back("noonhehinitialarabic"_n, AglMapping{ (AglType)1, 2, 37 });
    s_aglList.emplace_back("nooninitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEE7 });
    s_aglList.emplace_back("noonjeeminitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFCD2 });
    s_aglList.emplace_back("noonjeemisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFC4B });
    s_aglList.emplace_back("noonmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEE8 });
    s_aglList.emplace_back("noonmeeminitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFCD5 });
    s_aglList.emplace_back("noonmeemisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFC4E });
    s_aglList.emplace_back("noonnoonfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFC8D });
    s_aglList.emplace_back("notcontains"_n, AglMapping{ (AglType)1, 1, 0x220C });
    s_aglList.emplace_back("notelement"_n, AglMapping{ (AglType)3, 1, 0x2209 });
    s_aglList.emplace_back("notelementof"_n, AglMapping{ (AglType)1, 1, 0x2209 });
    s_aglList.emplace_back("notequal"_n, AglMapping{ (AglType)3, 1, 0x2260 });
    s_aglList.emplace_back("notgreater"_n, AglMapping{ (AglType)1, 1, 0x226F });
    s_aglList.emplace_back("notgreaternorequal"_n, AglMapping{ (AglType)1, 1, 0x2271 });
    s_aglList.emplace_back("notgreaternorless"_n, AglMapping{ (AglType)1, 1, 0x2279 });
    s_aglList.emplace_back("notidentical"_n, AglMapping{ (AglType)1, 1, 0x2262 });
    s_aglList.emplace_back("notless"_n, AglMapping{ (AglType)1, 1, 0x226E });
    s_aglList.emplace_back("notlessnorequal"_n, AglMapping{ (AglType)1, 1, 0x2270 });
    s_aglList.emplace_back("notparallel"_n, AglMapping{ (AglType)1, 1, 0x2226 });
    s_aglList.emplace_back("notprecedes"_n, AglMapping{ (AglType)1, 1, 0x2280 });
    s_aglList.emplace_back("notsubset"_n, AglMapping{ (AglType)3, 1, 0x2284 });
    s_aglList.emplace_back("notsucceeds"_n, AglMapping{ (AglType)1, 1, 0x2281 });
    s_aglList.emplace_back("notsuperset"_n, AglMapping{ (AglType)1, 1, 0x2285 });
    s_aglList.emplace_back("nowarmenian"_n, AglMapping{ (AglType)1, 1, 0x0576 });
    s_aglList.emplace_back("nparen"_n, AglMapping{ (AglType)1, 1, 0x24A9 });
    s_aglList.emplace_back("nssquare"_n, AglMapping{ (AglType)1, 1, 0x33B1 });
    s_aglList.emplace_back("nsuperior"_n, AglMapping{ (AglType)1, 1, 0x207F });
    s_aglList.emplace_back("ntilde"_n, AglMapping{ (AglType)11, 1, 0x00F1 });
    s_aglList.emplace_back("nu"_n, AglMapping{ (AglType)3, 1, 0x03BD });
    s_aglList.emplace_back("nuhiragana"_n, AglMapping{ (AglType)1, 1, 0x306C });
    s_aglList.emplace_back("nukatakana"_n, AglMapping{ (AglType)1, 1, 0x30CC });
    s_aglList.emplace_back("nukatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF87 });
    s_aglList.emplace_back("nuktabengali"_n, AglMapping{ (AglType)1, 1, 0x09BC });
    s_aglList.emplace_back("nuktadeva"_n, AglMapping{ (AglType)1, 1, 0x093C });
    s_aglList.emplace_back("nuktagujarati"_n, AglMapping{ (AglType)1, 1, 0x0ABC });
    s_aglList.emplace_back("nuktagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A3C });
    s_aglList.emplace_back("numbersign"_n, AglMapping{ (AglType)11, 1, 0x0023 });
    s_aglList.emplace_back("numbersignmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF03 });
    s_aglList.emplace_back("numbersignsmall"_n, AglMapping{ (AglType)1, 1, 0xFE5F });
    s_aglList.emplace_back("numeralsigngreek"_n, AglMapping{ (AglType)1, 1, 0x0374 });
    s_aglList.emplace_back("numeralsignlowergreek"_n, AglMapping{ (AglType)1, 1, 0x0375 });
    s_aglList.emplace_back("numero"_n, AglMapping{ (AglType)1, 1, 0x2116 });
    s_aglList.emplace_back("nun"_n, AglMapping{ (AglType)1, 1, 0x05E0 });
    s_aglList.emplace_back("nundagesh"_n, AglMapping{ (AglType)1, 1, 0xFB40 });
    s_aglList.emplace_back("nundageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB40 });
    s_aglList.emplace_back("nunhebrew"_n, AglMapping{ (AglType)1, 1, 0x05E0 });
    s_aglList.emplace_back("nvsquare"_n, AglMapping{ (AglType)1, 1, 0x33B5 });
    s_aglList.emplace_back("nwsquare"_n, AglMapping{ (AglType)1, 1, 0x33BB });
    s_aglList.emplace_back("nyabengali"_n, AglMapping{ (AglType)1, 1, 0x099E });
    s_aglList.emplace_back("nyadeva"_n, AglMapping{ (AglType)1, 1, 0x091E });
    s_aglList.emplace_back("nyagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A9E });
    s_aglList.emplace_back("nyagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A1E });
    s_aglList.emplace_back("o"_n, AglMapping{ (AglType)11, 1, 0x006F });
    s_aglList.emplace_back("oacute"_n, AglMapping{ (AglType)11, 1, 0x00F3 });
    s_aglList.emplace_back("oangthai"_n, AglMapping{ (AglType)1, 1, 0x0E2D });
    s_aglList.emplace_back("obarred"_n, AglMapping{ (AglType)1, 1, 0x0275 });
    s_aglList.emplace_back("obarredcyrillic"_n, AglMapping{ (AglType)1, 1, 0x04E9 });
    s_aglList.emplace_back("obarreddieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04EB });
    s_aglList.emplace_back("obengali"_n, AglMapping{ (AglType)1, 1, 0x0993 });
    s_aglList.emplace_back("obopomofo"_n, AglMapping{ (AglType)1, 1, 0x311B });
    s_aglList.emplace_back("obreve"_n, AglMapping{ (AglType)3, 1, 0x014F });
    s_aglList.emplace_back("ocandradeva"_n, AglMapping{ (AglType)1, 1, 0x0911 });
    s_aglList.emplace_back("ocandragujarati"_n, AglMapping{ (AglType)1, 1, 0x0A91 });
    s_aglList.emplace_back("ocandravowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0949 });
    s_aglList.emplace_back("ocandravowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0AC9 });
    s_aglList.emplace_back("ocaron"_n, AglMapping{ (AglType)1, 1, 0x01D2 });
    s_aglList.emplace_back("ocircle"_n, AglMapping{ (AglType)1, 1, 0x24DE });
    s_aglList.emplace_back("ocircumflex"_n, AglMapping{ (AglType)11, 1, 0x00F4 });
    s_aglList.emplace_back("ocircumflexacute"_n, AglMapping{ (AglType)1, 1, 0x1ED1 });
    s_aglList.emplace_back("ocircumflexdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1ED9 });
    s_aglList.emplace_back("ocircumflexgrave"_n, AglMapping{ (AglType)1, 1, 0x1ED3 });
    s_aglList.emplace_back("ocircumflexhookabove"_n, AglMapping{ (AglType)1, 1, 0x1ED5 });
    s_aglList.emplace_back("ocircumflextilde"_n, AglMapping{ (AglType)1, 1, 0x1ED7 });
    s_aglList.emplace_back("ocyrillic"_n, AglMapping{ (AglType)1, 1, 0x043E });
    s_aglList.emplace_back("odblacute"_n, AglMapping{ (AglType)1, 1, 0x0151 });
    s_aglList.emplace_back("odblgrave"_n, AglMapping{ (AglType)1, 1, 0x020D });
    s_aglList.emplace_back("odeva"_n, AglMapping{ (AglType)1, 1, 0x0913 });
    s_aglList.emplace_back("odieresis"_n, AglMapping{ (AglType)11, 1, 0x00F6 });
    s_aglList.emplace_back("odieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04E7 });
    s_aglList.emplace_back("odotbelow"_n, AglMapping{ (AglType)1, 1, 0x1ECD });
    s_aglList.emplace_back("oe"_n, AglMapping{ (AglType)11, 1, 0x0153 });
    s_aglList.emplace_back("oekorean"_n, AglMapping{ (AglType)1, 1, 0x315A });
    s_aglList.emplace_back("ogonek"_n, AglMapping{ (AglType)11, 1, 0x02DB });
    s_aglList.emplace_back("ogonekcmb"_n, AglMapping{ (AglType)1, 1, 0x0328 });
    s_aglList.emplace_back("ograve"_n, AglMapping{ (AglType)11, 1, 0x00F2 });
    s_aglList.emplace_back("ogujarati"_n, AglMapping{ (AglType)1, 1, 0x0A93 });
    s_aglList.emplace_back("oharmenian"_n, AglMapping{ (AglType)1, 1, 0x0585 });
    s_aglList.emplace_back("ohiragana"_n, AglMapping{ (AglType)1, 1, 0x304A });
    s_aglList.emplace_back("ohookabove"_n, AglMapping{ (AglType)1, 1, 0x1ECF });
    s_aglList.emplace_back("ohorn"_n, AglMapping{ (AglType)3, 1, 0x01A1 });
    s_aglList.emplace_back("ohornacute"_n, AglMapping{ (AglType)1, 1, 0x1EDB });
    s_aglList.emplace_back("ohorndotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EE3 });
    s_aglList.emplace_back("ohorngrave"_n, AglMapping{ (AglType)1, 1, 0x1EDD });
    s_aglList.emplace_back("ohornhookabove"_n, AglMapping{ (AglType)1, 1, 0x1EDF });
    s_aglList.emplace_back("ohorntilde"_n, AglMapping{ (AglType)1, 1, 0x1EE1 });
    s_aglList.emplace_back("ohungarumlaut"_n, AglMapping{ (AglType)3, 1, 0x0151 });
    s_aglList.emplace_back("oi"_n, AglMapping{ (AglType)1, 1, 0x01A3 });
    s_aglList.emplace_back("oinvertedbreve"_n, AglMapping{ (AglType)1, 1, 0x020F });
    s_aglList.emplace_back("okatakana"_n, AglMapping{ (AglType)1, 1, 0x30AA });
    s_aglList.emplace_back("okatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF75 });
    s_aglList.emplace_back("okorean"_n, AglMapping{ (AglType)1, 1, 0x3157 });
    s_aglList.emplace_back("olehebrew"_n, AglMapping{ (AglType)1, 1, 0x05AB });
    s_aglList.emplace_back("omacron"_n, AglMapping{ (AglType)3, 1, 0x014D });
    s_aglList.emplace_back("omacronacute"_n, AglMapping{ (AglType)1, 1, 0x1E53 });
    s_aglList.emplace_back("omacrongrave"_n, AglMapping{ (AglType)1, 1, 0x1E51 });
    s_aglList.emplace_back("omdeva"_n, AglMapping{ (AglType)1, 1, 0x0950 });
    s_aglList.emplace_back("omega"_n, AglMapping{ (AglType)3, 1, 0x03C9 });
    s_aglList.emplace_back("omega1"_n, AglMapping{ (AglType)3, 1, 0x03D6 });
    s_aglList.emplace_back("omegacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0461 });
    s_aglList.emplace_back("omegalatinclosed"_n, AglMapping{ (AglType)1, 1, 0x0277 });
    s_aglList.emplace_back("omegaroundcyrillic"_n, AglMapping{ (AglType)1, 1, 0x047B });
    s_aglList.emplace_back("omegatitlocyrillic"_n, AglMapping{ (AglType)1, 1, 0x047D });
    s_aglList.emplace_back("omegatonos"_n, AglMapping{ (AglType)3, 1, 0x03CE });
    s_aglList.emplace_back("omgujarati"_n, AglMapping{ (AglType)1, 1, 0x0AD0 });
    s_aglList.emplace_back("omicron"_n, AglMapping{ (AglType)3, 1, 0x03BF });
    s_aglList.emplace_back("omicrontonos"_n, AglMapping{ (AglType)3, 1, 0x03CC });
    s_aglList.emplace_back("omonospace"_n, AglMapping{ (AglType)1, 1, 0xFF4F });
    s_aglList.emplace_back("one"_n, AglMapping{ (AglType)11, 1, 0x0031 });
    s_aglList.emplace_back("onearabic"_n, AglMapping{ (AglType)1, 1, 0x0661 });
    s_aglList.emplace_back("onebengali"_n, AglMapping{ (AglType)1, 1, 0x09E7 });
    s_aglList.emplace_back("onecircle"_n, AglMapping{ (AglType)1, 1, 0x2460 });
    s_aglList.emplace_back("onecircleinversesansserif"_n, AglMapping{ (AglType)1, 1, 0x278A });
    s_aglList.emplace_back("onedeva"_n, AglMapping{ (AglType)1, 1, 0x0967 });
    s_aglList.emplace_back("onedotenleader"_n, AglMapping{ (AglType)3, 1, 0x2024 });
    s_aglList.emplace_back("oneeighth"_n, AglMapping{ (AglType)3, 1, 0x215B });
    s_aglList.emplace_back("onefitted"_n, AglMapping{ (AglType)1, 1, 0xF6DC });
    s_aglList.emplace_back("onegujarati"_n, AglMapping{ (AglType)1, 1, 0x0AE7 });
    s_aglList.emplace_back("onegurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A67 });
    s_aglList.emplace_back("onehackarabic"_n, AglMapping{ (AglType)1, 1, 0x0661 });
    s_aglList.emplace_back("onehalf"_n, AglMapping{ (AglType)11, 1, 0x00BD });
    s_aglList.emplace_back("onehangzhou"_n, AglMapping{ (AglType)1, 1, 0x3021 });
    s_aglList.emplace_back("oneideographicparen"_n, AglMapping{ (AglType)1, 1, 0x3220 });
    s_aglList.emplace_back("oneinferior"_n, AglMapping{ (AglType)1, 1, 0x2081 });
    s_aglList.emplace_back("onemonospace"_n, AglMapping{ (AglType)1, 1, 0xFF11 });
    s_aglList.emplace_back("onenumeratorbengali"_n, AglMapping{ (AglType)1, 1, 0x09F4 });
    s_aglList.emplace_back("oneoldstyle"_n, AglMapping{ (AglType)1, 1, 0xF731 });
    s_aglList.emplace_back("oneparen"_n, AglMapping{ (AglType)1, 1, 0x2474 });
    s_aglList.emplace_back("oneperiod"_n, AglMapping{ (AglType)1, 1, 0x2488 });
    s_aglList.emplace_back("onepersian"_n, AglMapping{ (AglType)1, 1, 0x06F1 });
    s_aglList.emplace_back("onequarter"_n, AglMapping{ (AglType)11, 1, 0x00BC });
    s_aglList.emplace_back("oneroman"_n, AglMapping{ (AglType)1, 1, 0x2170 });
    s_aglList.emplace_back("onesuperior"_n, AglMapping{ (AglType)9, 1, 0x00B9 });
    s_aglList.emplace_back("onethai"_n, AglMapping{ (AglType)1, 1, 0x0E51 });
    s_aglList.emplace_back("onethird"_n, AglMapping{ (AglType)3, 1, 0x2153 });
    s_aglList.emplace_back("oogonek"_n, AglMapping{ (AglType)1, 1, 0x01EB });
    s_aglList.emplace_back("oogonekmacron"_n, AglMapping{ (AglType)1, 1, 0x01ED });
    s_aglList.emplace_back("oogurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A13 });
    s_aglList.emplace_back("oomatragurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A4B });
    s_aglList.emplace_back("oopen"_n, AglMapping{ (AglType)1, 1, 0x0254 });
    s_aglList.emplace_back("oparen"_n, AglMapping{ (AglType)1, 1, 0x24AA });
    s_aglList.emplace_back("openbullet"_n, AglMapping{ (AglType)3, 1, 0x25E6 });
    s_aglList.emplace_back("option"_n, AglMapping{ (AglType)1, 1, 0x2325 });
    s_aglList.emplace_back("ordfeminine"_n, AglMapping{ (AglType)11, 1, 0x00AA });
    s_aglList.emplace_back("ordmasculine"_n, AglMapping{ (AglType)11, 1, 0x00BA });
    s_aglList.emplace_back("orthogonal"_n, AglMapping{ (AglType)3, 1, 0x221F });
    s_aglList.emplace_back("oshortdeva"_n, AglMapping{ (AglType)1, 1, 0x0912 });
    s_aglList.emplace_back("oshortvowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x094A });
    s_aglList.emplace_back("oslash"_n, AglMapping{ (AglType)11, 1, 0x00F8 });
    s_aglList.emplace_back("oslashacute"_n, AglMapping{ (AglType)3, 1, 0x01FF });
    s_aglList.emplace_back("osmallhiragana"_n, AglMapping{ (AglType)1, 1, 0x3049 });
    s_aglList.emplace_back("osmallkatakana"_n, AglMapping{ (AglType)1, 1, 0x30A9 });
    s_aglList.emplace_back("osmallkatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF6B });
    s_aglList.emplace_back("ostrokeacute"_n, AglMapping{ (AglType)1, 1, 0x01FF });
    s_aglList.emplace_back("osuperior"_n, AglMapping{ (AglType)1, 1, 0xF6F0 });
    s_aglList.emplace_back("otcyrillic"_n, AglMapping{ (AglType)1, 1, 0x047F });
    s_aglList.emplace_back("otilde"_n, AglMapping{ (AglType)11, 1, 0x00F5 });
    s_aglList.emplace_back("otildeacute"_n, AglMapping{ (AglType)1, 1, 0x1E4D });
    s_aglList.emplace_back("otildedieresis"_n, AglMapping{ (AglType)1, 1, 0x1E4F });
    s_aglList.emplace_back("oubopomofo"_n, AglMapping{ (AglType)1, 1, 0x3121 });
    s_aglList.emplace_back("overline"_n, AglMapping{ (AglType)1, 1, 0x203E });
    s_aglList.emplace_back("overlinecenterline"_n, AglMapping{ (AglType)1, 1, 0xFE4A });
    s_aglList.emplace_back("overlinecmb"_n, AglMapping{ (AglType)1, 1, 0x0305 });
    s_aglList.emplace_back("overlinedashed"_n, AglMapping{ (AglType)1, 1, 0xFE49 });
    s_aglList.emplace_back("overlinedblwavy"_n, AglMapping{ (AglType)1, 1, 0xFE4C });
    s_aglList.emplace_back("overlinewavy"_n, AglMapping{ (AglType)1, 1, 0xFE4B });
    s_aglList.emplace_back("overscore"_n, AglMapping{ (AglType)1, 1, 0x00AF });
    s_aglList.emplace_back("ovowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09CB });
    s_aglList.emplace_back("ovowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x094B });
    s_aglList.emplace_back("ovowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0ACB });
    s_aglList.emplace_back("p"_n, AglMapping{ (AglType)11, 1, 0x0070 });
    s_aglList.emplace_back("paampssquare"_n, AglMapping{ (AglType)1, 1, 0x3380 });
    s_aglList.emplace_back("paasentosquare"_n, AglMapping{ (AglType)1, 1, 0x332B });
    s_aglList.emplace_back("pabengali"_n, AglMapping{ (AglType)1, 1, 0x09AA });
    s_aglList.emplace_back("pacute"_n, AglMapping{ (AglType)1, 1, 0x1E55 });
    s_aglList.emplace_back("padeva"_n, AglMapping{ (AglType)1, 1, 0x092A });
    s_aglList.emplace_back("pagedown"_n, AglMapping{ (AglType)1, 1, 0x21DF });
    s_aglList.emplace_back("pageup"_n, AglMapping{ (AglType)1, 1, 0x21DE });
    s_aglList.emplace_back("pagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AAA });
    s_aglList.emplace_back("pagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A2A });
    s_aglList.emplace_back("pahiragana"_n, AglMapping{ (AglType)1, 1, 0x3071 });
    s_aglList.emplace_back("paiyannoithai"_n, AglMapping{ (AglType)1, 1, 0x0E2F });
    s_aglList.emplace_back("pakatakana"_n, AglMapping{ (AglType)1, 1, 0x30D1 });
    s_aglList.emplace_back("palatalizationcyrilliccmb"_n, AglMapping{ (AglType)1, 1, 0x0484 });
    s_aglList.emplace_back("palochkacyrillic"_n, AglMapping{ (AglType)1, 1, 0x04C0 });
    s_aglList.emplace_back("pansioskorean"_n, AglMapping{ (AglType)1, 1, 0x317F });
    s_aglList.emplace_back("paragraph"_n, AglMapping{ (AglType)11, 1, 0x00B6 });
    s_aglList.emplace_back("parallel"_n, AglMapping{ (AglType)1, 1, 0x2225 });
    s_aglList.emplace_back("parenleft"_n, AglMapping{ (AglType)11, 1, 0x0028 });
    s_aglList.emplace_back("parenleftaltonearabic"_n, AglMapping{ (AglType)1, 1, 0xFD3E });
    s_aglList.emplace_back("parenleftbt"_n, AglMapping{ (AglType)1, 1, 0xF8ED });
    s_aglList.emplace_back("parenleftex"_n, AglMapping{ (AglType)1, 1, 0xF8EC });
    s_aglList.emplace_back("parenleftinferior"_n, AglMapping{ (AglType)1, 1, 0x208D });
    s_aglList.emplace_back("parenleftmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF08 });
    s_aglList.emplace_back("parenleftsmall"_n, AglMapping{ (AglType)1, 1, 0xFE59 });
    s_aglList.emplace_back("parenleftsuperior"_n, AglMapping{ (AglType)1, 1, 0x207D });
    s_aglList.emplace_back("parenlefttp"_n, AglMapping{ (AglType)1, 1, 0xF8EB });
    s_aglList.emplace_back("parenleftvertical"_n, AglMapping{ (AglType)1, 1, 0xFE35 });
    s_aglList.emplace_back("parenright"_n, AglMapping{ (AglType)11, 1, 0x0029 });
    s_aglList.emplace_back("parenrightaltonearabic"_n, AglMapping{ (AglType)1, 1, 0xFD3F });
    s_aglList.emplace_back("parenrightbt"_n, AglMapping{ (AglType)1, 1, 0xF8F8 });
    s_aglList.emplace_back("parenrightex"_n, AglMapping{ (AglType)1, 1, 0xF8F7 });
    s_aglList.emplace_back("parenrightinferior"_n, AglMapping{ (AglType)1, 1, 0x208E });
    s_aglList.emplace_back("parenrightmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF09 });
    s_aglList.emplace_back("parenrightsmall"_n, AglMapping{ (AglType)1, 1, 0xFE5A });
    s_aglList.emplace_back("parenrightsuperior"_n, AglMapping{ (AglType)1, 1, 0x207E });
    s_aglList.emplace_back("parenrighttp"_n, AglMapping{ (AglType)1, 1, 0xF8F6 });
    s_aglList.emplace_back("parenrightvertical"_n, AglMapping{ (AglType)1, 1, 0xFE36 });
    s_aglList.emplace_back("partialdiff"_n, AglMapping{ (AglType)3, 1, 0x2202 });
    s_aglList.emplace_back("paseqhebrew"_n, AglMapping{ (AglType)1, 1, 0x05C0 });
    s_aglList.emplace_back("pashtahebrew"_n, AglMapping{ (AglType)1, 1, 0x0599 });
    s_aglList.emplace_back("pasquare"_n, AglMapping{ (AglType)1, 1, 0x33A9 });
    s_aglList.emplace_back("patah"_n, AglMapping{ (AglType)1, 1, 0x05B7 });
    s_aglList.emplace_back("patah11"_n, AglMapping{ (AglType)1, 1, 0x05B7 });
    s_aglList.emplace_back("patah1d"_n, AglMapping{ (AglType)1, 1, 0x05B7 });
    s_aglList.emplace_back("patah2a"_n, AglMapping{ (AglType)1, 1, 0x05B7 });
    s_aglList.emplace_back("patahhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B7 });
    s_aglList.emplace_back("patahnarrowhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B7 });
    s_aglList.emplace_back("patahquarterhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B7 });
    s_aglList.emplace_back("patahwidehebrew"_n, AglMapping{ (AglType)1, 1, 0x05B7 });
    s_aglList.emplace_back("pazerhebrew"_n, AglMapping{ (AglType)1, 1, 0x05A1 });
    s_aglList.emplace_back("pbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3106 });
    s_aglList.emplace_back("pcircle"_n, AglMapping{ (AglType)1, 1, 0x24DF });
    s_aglList.emplace_back("pdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E57 });
    s_aglList.emplace_back("pe"_n, AglMapping{ (AglType)1, 1, 0x05E4 });
    s_aglList.emplace_back("pecyrillic"_n, AglMapping{ (AglType)1, 1, 0x043F });
    s_aglList.emplace_back("pedagesh"_n, AglMapping{ (AglType)1, 1, 0xFB44 });
    s_aglList.emplace_back("pedageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB44 });
    s_aglList.emplace_back("peezisquare"_n, AglMapping{ (AglType)1, 1, 0x333B });
    s_aglList.emplace_back("pefinaldageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB43 });
    s_aglList.emplace_back("peharabic"_n, AglMapping{ (AglType)1, 1, 0x067E });
    s_aglList.emplace_back("peharmenian"_n, AglMapping{ (AglType)1, 1, 0x057A });
    s_aglList.emplace_back("pehebrew"_n, AglMapping{ (AglType)1, 1, 0x05E4 });
    s_aglList.emplace_back("pehfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFB57 });
    s_aglList.emplace_back("pehinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFB58 });
    s_aglList.emplace_back("pehiragana"_n, AglMapping{ (AglType)1, 1, 0x307A });
    s_aglList.emplace_back("pehmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFB59 });
    s_aglList.emplace_back("pekatakana"_n, AglMapping{ (AglType)1, 1, 0x30DA });
    s_aglList.emplace_back("pemiddlehookcyrillic"_n, AglMapping{ (AglType)1, 1, 0x04A7 });
    s_aglList.emplace_back("perafehebrew"_n, AglMapping{ (AglType)1, 1, 0xFB4E });
    s_aglList.emplace_back("percent"_n, AglMapping{ (AglType)11, 1, 0x0025 });
    s_aglList.emplace_back("percentarabic"_n, AglMapping{ (AglType)1, 1, 0x066A });
    s_aglList.emplace_back("percentmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF05 });
    s_aglList.emplace_back("percentsmall"_n, AglMapping{ (AglType)1, 1, 0xFE6A });
    s_aglList.emplace_back("period"_n, AglMapping{ (AglType)11, 1, 0x002E });
    s_aglList.emplace_back("periodarmenian"_n, AglMapping{ (AglType)1, 1, 0x0589 });
    s_aglList.emplace_back("periodcentered"_n, AglMapping{ (AglType)11, 1, 0x00B7 });
    s_aglList.emplace_back("periodhalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF61 });
    s_aglList.emplace_back("periodinferior"_n, AglMapping{ (AglType)1, 1, 0xF6E7 });
    s_aglList.emplace_back("periodmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF0E });
    s_aglList.emplace_back("periodsmall"_n, AglMapping{ (AglType)1, 1, 0xFE52 });
    s_aglList.emplace_back("periodsuperior"_n, AglMapping{ (AglType)1, 1, 0xF6E8 });
    s_aglList.emplace_back("perispomenigreekcmb"_n, AglMapping{ (AglType)1, 1, 0x0342 });
    s_aglList.emplace_back("perpendicular"_n, AglMapping{ (AglType)3, 1, 0x22A5 });
    s_aglList.emplace_back("perthousand"_n, AglMapping{ (AglType)11, 1, 0x2030 });
    s_aglList.emplace_back("peseta"_n, AglMapping{ (AglType)3, 1, 0x20A7 });
    s_aglList.emplace_back("pfsquare"_n, AglMapping{ (AglType)1, 1, 0x338A });
    s_aglList.emplace_back("phabengali"_n, AglMapping{ (AglType)1, 1, 0x09AB });
    s_aglList.emplace_back("phadeva"_n, AglMapping{ (AglType)1, 1, 0x092B });
    s_aglList.emplace_back("phagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AAB });
    s_aglList.emplace_back("phagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A2B });
    s_aglList.emplace_back("phi"_n, AglMapping{ (AglType)3, 1, 0x03C6 });
    s_aglList.emplace_back("phi1"_n, AglMapping{ (AglType)3, 1, 0x03D5 });
    s_aglList.emplace_back("phieuphacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x327A });
    s_aglList.emplace_back("phieuphaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x321A });
    s_aglList.emplace_back("phieuphcirclekorean"_n, AglMapping{ (AglType)1, 1, 0x326C });
    s_aglList.emplace_back("phieuphkorean"_n, AglMapping{ (AglType)1, 1, 0x314D });
    s_aglList.emplace_back("phieuphparenkorean"_n, AglMapping{ (AglType)1, 1, 0x320C });
    s_aglList.emplace_back("philatin"_n, AglMapping{ (AglType)1, 1, 0x0278 });
    s_aglList.emplace_back("phinthuthai"_n, AglMapping{ (AglType)1, 1, 0x0E3A });
    s_aglList.emplace_back("phisymbolgreek"_n, AglMapping{ (AglType)1, 1, 0x03D5 });
    s_aglList.emplace_back("phook"_n, AglMapping{ (AglType)1, 1, 0x01A5 });
    s_aglList.emplace_back("phophanthai"_n, AglMapping{ (AglType)1, 1, 0x0E1E });
    s_aglList.emplace_back("phophungthai"_n, AglMapping{ (AglType)1, 1, 0x0E1C });
    s_aglList.emplace_back("phosamphaothai"_n, AglMapping{ (AglType)1, 1, 0x0E20 });
    s_aglList.emplace_back("pi"_n, AglMapping{ (AglType)3, 1, 0x03C0 });
    s_aglList.emplace_back("pieupacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3273 });
    s_aglList.emplace_back("pieupaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3213 });
    s_aglList.emplace_back("pieupcieuckorean"_n, AglMapping{ (AglType)1, 1, 0x3176 });
    s_aglList.emplace_back("pieupcirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3265 });
    s_aglList.emplace_back("pieupkiyeokkorean"_n, AglMapping{ (AglType)1, 1, 0x3172 });
    s_aglList.emplace_back("pieupkorean"_n, AglMapping{ (AglType)1, 1, 0x3142 });
    s_aglList.emplace_back("pieupparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3205 });
    s_aglList.emplace_back("pieupsioskiyeokkorean"_n, AglMapping{ (AglType)1, 1, 0x3174 });
    s_aglList.emplace_back("pieupsioskorean"_n, AglMapping{ (AglType)1, 1, 0x3144 });
    s_aglList.emplace_back("pieupsiostikeutkorean"_n, AglMapping{ (AglType)1, 1, 0x3175 });
    s_aglList.emplace_back("pieupthieuthkorean"_n, AglMapping{ (AglType)1, 1, 0x3177 });
    s_aglList.emplace_back("pieuptikeutkorean"_n, AglMapping{ (AglType)1, 1, 0x3173 });
    s_aglList.emplace_back("pihiragana"_n, AglMapping{ (AglType)1, 1, 0x3074 });
    s_aglList.emplace_back("pikatakana"_n, AglMapping{ (AglType)1, 1, 0x30D4 });
    s_aglList.emplace_back("pisymbolgreek"_n, AglMapping{ (AglType)1, 1, 0x03D6 });
    s_aglList.emplace_back("piwrarmenian"_n, AglMapping{ (AglType)1, 1, 0x0583 });
    s_aglList.emplace_back("plus"_n, AglMapping{ (AglType)11, 1, 0x002B });
    s_aglList.emplace_back("plusbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x031F });
    s_aglList.emplace_back("pluscircle"_n, AglMapping{ (AglType)1, 1, 0x2295 });
    s_aglList.emplace_back("plusminus"_n, AglMapping{ (AglType)11, 1, 0x00B1 });
    s_aglList.emplace_back("plusmod"_n, AglMapping{ (AglType)1, 1, 0x02D6 });
    s_aglList.emplace_back("plusmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF0B });
    s_aglList.emplace_back("plussmall"_n, AglMapping{ (AglType)1, 1, 0xFE62 });
    s_aglList.emplace_back("plussuperior"_n, AglMapping{ (AglType)1, 1, 0x207A });
    s_aglList.emplace_back("pmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF50 });
    s_aglList.emplace_back("pmsquare"_n, AglMapping{ (AglType)1, 1, 0x33D8 });
    s_aglList.emplace_back("pohiragana"_n, AglMapping{ (AglType)1, 1, 0x307D });
    s_aglList.emplace_back("pointingindexdownwhite"_n, AglMapping{ (AglType)1, 1, 0x261F });
    s_aglList.emplace_back("pointingindexleftwhite"_n, AglMapping{ (AglType)1, 1, 0x261C });
    s_aglList.emplace_back("pointingindexrightwhite"_n, AglMapping{ (AglType)1, 1, 0x261E });
    s_aglList.emplace_back("pointingindexupwhite"_n, AglMapping{ (AglType)1, 1, 0x261D });
    s_aglList.emplace_back("pokatakana"_n, AglMapping{ (AglType)1, 1, 0x30DD });
    s_aglList.emplace_back("poplathai"_n, AglMapping{ (AglType)1, 1, 0x0E1B });
    s_aglList.emplace_back("postalmark"_n, AglMapping{ (AglType)1, 1, 0x3012 });
    s_aglList.emplace_back("postalmarkface"_n, AglMapping{ (AglType)1, 1, 0x3020 });
    s_aglList.emplace_back("pparen"_n, AglMapping{ (AglType)1, 1, 0x24AB });
    s_aglList.emplace_back("precedes"_n, AglMapping{ (AglType)1, 1, 0x227A });
    s_aglList.emplace_back("prescription"_n, AglMapping{ (AglType)3, 1, 0x211E });
    s_aglList.emplace_back("primemod"_n, AglMapping{ (AglType)1, 1, 0x02B9 });
    s_aglList.emplace_back("primereversed"_n, AglMapping{ (AglType)1, 1, 0x2035 });
    s_aglList.emplace_back("product"_n, AglMapping{ (AglType)3, 1, 0x220F });
    s_aglList.emplace_back("projective"_n, AglMapping{ (AglType)1, 1, 0x2305 });
    s_aglList.emplace_back("prolongedkana"_n, AglMapping{ (AglType)1, 1, 0x30FC });
    s_aglList.emplace_back("propellor"_n, AglMapping{ (AglType)1, 1, 0x2318 });
    s_aglList.emplace_back("propersubset"_n, AglMapping{ (AglType)3, 1, 0x2282 });
    s_aglList.emplace_back("propersuperset"_n, AglMapping{ (AglType)3, 1, 0x2283 });
    s_aglList.emplace_back("proportion"_n, AglMapping{ (AglType)1, 1, 0x2237 });
    s_aglList.emplace_back("proportional"_n, AglMapping{ (AglType)3, 1, 0x221D });
    s_aglList.emplace_back("psi"_n, AglMapping{ (AglType)3, 1, 0x03C8 });
    s_aglList.emplace_back("psicyrillic"_n, AglMapping{ (AglType)1, 1, 0x0471 });
    s_aglList.emplace_back("psilipneumatacyrilliccmb"_n, AglMapping{ (AglType)1, 1, 0x0486 });
    s_aglList.emplace_back("pssquare"_n, AglMapping{ (AglType)1, 1, 0x33B0 });
    s_aglList.emplace_back("puhiragana"_n, AglMapping{ (AglType)1, 1, 0x3077 });
    s_aglList.emplace_back("pukatakana"_n, AglMapping{ (AglType)1, 1, 0x30D7 });
    s_aglList.emplace_back("pvsquare"_n, AglMapping{ (AglType)1, 1, 0x33B4 });
    s_aglList.emplace_back("pwsquare"_n, AglMapping{ (AglType)1, 1, 0x33BA });
    s_aglList.emplace_back("q"_n, AglMapping{ (AglType)11, 1, 0x0071 });
    s_aglList.emplace_back("qadeva"_n, AglMapping{ (AglType)1, 1, 0x0958 });
    s_aglList.emplace_back("qadmahebrew"_n, AglMapping{ (AglType)1, 1, 0x05A8 });
    s_aglList.emplace_back("qafarabic"_n, AglMapping{ (AglType)1, 1, 0x0642 });
    s_aglList.emplace_back("qaffinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFED6 });
    s_aglList.emplace_back("qafinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFED7 });
    s_aglList.emplace_back("qafmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFED8 });
    s_aglList.emplace_back("qamats"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamats10"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamats1a"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamats1c"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamats27"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamats29"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamats33"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamatsde"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamatshebrew"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamatsnarrowhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamatsqatanhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamatsqatannarrowhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamatsqatanquarterhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamatsqatanwidehebrew"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamatsquarterhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qamatswidehebrew"_n, AglMapping{ (AglType)1, 1, 0x05B8 });
    s_aglList.emplace_back("qarneyparahebrew"_n, AglMapping{ (AglType)1, 1, 0x059F });
    s_aglList.emplace_back("qbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3111 });
    s_aglList.emplace_back("qcircle"_n, AglMapping{ (AglType)1, 1, 0x24E0 });
    s_aglList.emplace_back("qhook"_n, AglMapping{ (AglType)1, 1, 0x02A0 });
    s_aglList.emplace_back("qmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF51 });
    s_aglList.emplace_back("qof"_n, AglMapping{ (AglType)1, 1, 0x05E7 });
    s_aglList.emplace_back("qofdagesh"_n, AglMapping{ (AglType)1, 1, 0xFB47 });
    s_aglList.emplace_back("qofdageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB47 });
    s_aglList.emplace_back("qofhatafpatah"_n, AglMapping{ (AglType)1, 2, 38 });
    s_aglList.emplace_back("qofhatafpatahhebrew"_n, AglMapping{ (AglType)1, 2, 39 });
    s_aglList.emplace_back("qofhatafsegol"_n, AglMapping{ (AglType)1, 2, 40 });
    s_aglList.emplace_back("qofhatafsegolhebrew"_n, AglMapping{ (AglType)1, 2, 41 });
    s_aglList.emplace_back("qofhebrew"_n, AglMapping{ (AglType)1, 1, 0x05E7 });
    s_aglList.emplace_back("qofhiriq"_n, AglMapping{ (AglType)1, 2, 42 });
    s_aglList.emplace_back("qofhiriqhebrew"_n, AglMapping{ (AglType)1, 2, 43 });
    s_aglList.emplace_back("qofholam"_n, AglMapping{ (AglType)1, 2, 44 });
    s_aglList.emplace_back("qofholamhebrew"_n, AglMapping{ (AglType)1, 2, 45 });
    s_aglList.emplace_back("qofpatah"_n, AglMapping{ (AglType)1, 2, 46 });
    s_aglList.emplace_back("qofpatahhebrew"_n, AglMapping{ (AglType)1, 2, 47 });
    s_aglList.emplace_back("qofqamats"_n, AglMapping{ (AglType)1, 2, 48 });
    s_aglList.emplace_back("qofqamatshebrew"_n, AglMapping{ (AglType)1, 2, 49 });
    s_aglList.emplace_back("qofqubuts"_n, AglMapping{ (AglType)1, 2, 50 });
    s_aglList.emplace_back("qofqubutshebrew"_n, AglMapping{ (AglType)1, 2, 51 });
    s_aglList.emplace_back("qofsegol"_n, AglMapping{ (AglType)1, 2, 52 });
    s_aglList.emplace_back("qofsegolhebrew"_n, AglMapping{ (AglType)1, 2, 53 });
    s_aglList.emplace_back("qofsheva"_n, AglMapping{ (AglType)1, 2, 54 });
    s_aglList.emplace_back("qofshevahebrew"_n, AglMapping{ (AglType)1, 2, 55 });
    s_aglList.emplace_back("qoftsere"_n, AglMapping{ (AglType)1, 2, 56 });
    s_aglList.emplace_back("qoftserehebrew"_n, AglMapping{ (AglType)1, 2, 57 });
    s_aglList.emplace_back("qparen"_n, AglMapping{ (AglType)1, 1, 0x24AC });
    s_aglList.emplace_back("quarternote"_n, AglMapping{ (AglType)1, 1, 0x2669 });
    s_aglList.emplace_back("qubuts"_n, AglMapping{ (AglType)1, 1, 0x05BB });
    s_aglList.emplace_back("qubuts18"_n, AglMapping{ (AglType)1, 1, 0x05BB });
    s_aglList.emplace_back("qubuts25"_n, AglMapping{ (AglType)1, 1, 0x05BB });
    s_aglList.emplace_back("qubuts31"_n, AglMapping{ (AglType)1, 1, 0x05BB });
    s_aglList.emplace_back("qubutshebrew"_n, AglMapping{ (AglType)1, 1, 0x05BB });
    s_aglList.emplace_back("qubutsnarrowhebrew"_n, AglMapping{ (AglType)1, 1, 0x05BB });
    s_aglList.emplace_back("qubutsquarterhebrew"_n, AglMapping{ (AglType)1, 1, 0x05BB });
    s_aglList.emplace_back("qubutswidehebrew"_n, AglMapping{ (AglType)1, 1, 0x05BB });
    s_aglList.emplace_back("question"_n, AglMapping{ (AglType)11, 1, 0x003F });
    s_aglList.emplace_back("questionarabic"_n, AglMapping{ (AglType)1, 1, 0x061F });
    s_aglList.emplace_back("questionarmenian"_n, AglMapping{ (AglType)1, 1, 0x055E });
    s_aglList.emplace_back("questiondown"_n, AglMapping{ (AglType)11, 1, 0x00BF });
    s_aglList.emplace_back("questiondownsmall"_n, AglMapping{ (AglType)1, 1, 0xF7BF });
    s_aglList.emplace_back("questiongreek"_n, AglMapping{ (AglType)1, 1, 0x037E });
    s_aglList.emplace_back("questionmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF1F });
    s_aglList.emplace_back("questionsmall"_n, AglMapping{ (AglType)1, 1, 0xF73F });
    s_aglList.emplace_back("quotedbl"_n, AglMapping{ (AglType)11, 1, 0x0022 });
    s_aglList.emplace_back("quotedblbase"_n, AglMapping{ (AglType)11, 1, 0x201E });
    s_aglList.emplace_back("quotedblleft"_n, AglMapping{ (AglType)11, 1, 0x201C });
    s_aglList.emplace_back("quotedblmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF02 });
    s_aglList.emplace_back("quotedblprime"_n, AglMapping{ (AglType)1, 1, 0x301E });
    s_aglList.emplace_back("quotedblprimereversed"_n, AglMapping{ (AglType)1, 1, 0x301D });
    s_aglList.emplace_back("quotedblright"_n, AglMapping{ (AglType)11, 1, 0x201D });
    s_aglList.emplace_back("quoteleft"_n, AglMapping{ (AglType)11, 1, 0x2018 });
    s_aglList.emplace_back("quoteleftreversed"_n, AglMapping{ (AglType)1, 1, 0x201B });
    s_aglList.emplace_back("quotereversed"_n, AglMapping{ (AglType)3, 1, 0x201B });
    s_aglList.emplace_back("quoteright"_n, AglMapping{ (AglType)11, 1, 0x2019 });
    s_aglList.emplace_back("quoterightn"_n, AglMapping{ (AglType)1, 1, 0x0149 });
    s_aglList.emplace_back("quotesinglbase"_n, AglMapping{ (AglType)11, 1, 0x201A });
    s_aglList.emplace_back("quotesingle"_n, AglMapping{ (AglType)11, 1, 0x0027 });
    s_aglList.emplace_back("quotesinglemonospace"_n, AglMapping{ (AglType)1, 1, 0xFF07 });
    s_aglList.emplace_back("r"_n, AglMapping{ (AglType)11, 1, 0x0072 });
    s_aglList.emplace_back("raarmenian"_n, AglMapping{ (AglType)1, 1, 0x057C });
    s_aglList.emplace_back("rabengali"_n, AglMapping{ (AglType)1, 1, 0x09B0 });
    s_aglList.emplace_back("racute"_n, AglMapping{ (AglType)3, 1, 0x0155 });
    s_aglList.emplace_back("radeva"_n, AglMapping{ (AglType)1, 1, 0x0930 });
    s_aglList.emplace_back("radical"_n, AglMapping{ (AglType)3, 1, 0x221A });
    s_aglList.emplace_back("radicalex"_n, AglMapping{ (AglType)1, 1, 0xF8E5 });
    s_aglList.emplace_back("radoverssquare"_n, AglMapping{ (AglType)1, 1, 0x33AE });
    s_aglList.emplace_back("radoverssquaredsquare"_n, AglMapping{ (AglType)1, 1, 0x33AF });
    s_aglList.emplace_back("radsquare"_n, AglMapping{ (AglType)1, 1, 0x33AD });
    s_aglList.emplace_back("rafe"_n, AglMapping{ (AglType)1, 1, 0x05BF });
    s_aglList.emplace_back("rafehebrew"_n, AglMapping{ (AglType)1, 1, 0x05BF });
    s_aglList.emplace_back("ragujarati"_n, AglMapping{ (AglType)1, 1, 0x0AB0 });
    s_aglList.emplace_back("ragurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A30 });
    s_aglList.emplace_back("rahiragana"_n, AglMapping{ (AglType)1, 1, 0x3089 });
    s_aglList.emplace_back("rakatakana"_n, AglMapping{ (AglType)1, 1, 0x30E9 });
    s_aglList.emplace_back("rakatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF97 });
    s_aglList.emplace_back("ralowerdiagonalbengali"_n, AglMapping{ (AglType)1, 1, 0x09F1 });
    s_aglList.emplace_back("ramiddlediagonalbengali"_n, AglMapping{ (AglType)1, 1, 0x09F0 });
    s_aglList.emplace_back("ramshorn"_n, AglMapping{ (AglType)1, 1, 0x0264 });
    s_aglList.emplace_back("ratio"_n, AglMapping{ (AglType)1, 1, 0x2236 });
    s_aglList.emplace_back("rbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3116 });
    s_aglList.emplace_back("rcaron"_n, AglMapping{ (AglType)3, 1, 0x0159 });
    s_aglList.emplace_back("rcedilla"_n, AglMapping{ (AglType)1, 1, 0x0157 });
    s_aglList.emplace_back("rcircle"_n, AglMapping{ (AglType)1, 1, 0x24E1 });
    s_aglList.emplace_back("rcommaaccent"_n, AglMapping{ (AglType)1, 1, 0x0157 });
    s_aglList.emplace_back("rdblgrave"_n, AglMapping{ (AglType)1, 1, 0x0211 });
    s_aglList.emplace_back("rdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E59 });
    s_aglList.emplace_back("rdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E5B });
    s_aglList.emplace_back("rdotbelowmacron"_n, AglMapping{ (AglType)1, 1, 0x1E5D });
    s_aglList.emplace_back("referencemark"_n, AglMapping{ (AglType)1, 1, 0x203B });
    s_aglList.emplace_back("reflexsubset"_n, AglMapping{ (AglType)3, 1, 0x2286 });
    s_aglList.emplace_back("reflexsuperset"_n, AglMapping{ (AglType)3, 1, 0x2287 });
    s_aglList.emplace_back("registered"_n, AglMapping{ (AglType)11, 1, 0x00AE });
    s_aglList.emplace_back("registersans"_n, AglMapping{ (AglType)1, 1, 0xF8E8 });
    s_aglList.emplace_back("registerserif"_n, AglMapping{ (AglType)1, 1, 0xF6DA });
    s_aglList.emplace_back("reharabic"_n, AglMapping{ (AglType)1, 1, 0x0631 });
    s_aglList.emplace_back("reharmenian"_n, AglMapping{ (AglType)1, 1, 0x0580 });
    s_aglList.emplace_back("rehfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEAE });
    s_aglList.emplace_back("rehiragana"_n, AglMapping{ (AglType)1, 1, 0x308C });
    s_aglList.emplace_back("rehyehaleflamarabic"_n, AglMapping{ (AglType)1, 4, 58 });
    s_aglList.emplace_back("rekatakana"_n, AglMapping{ (AglType)1, 1, 0x30EC });
    s_aglList.emplace_back("rekatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF9A });
    s_aglList.emplace_back("resh"_n, AglMapping{ (AglType)1, 1, 0x05E8 });
    s_aglList.emplace_back("reshdageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB48 });
    s_aglList.emplace_back("reshhatafpatah"_n, AglMapping{ (AglType)1, 2, 59 });
    s_aglList.emplace_back("reshhatafpatahhebrew"_n, AglMapping{ (AglType)1, 2, 60 });
    s_aglList.emplace_back("reshhatafsegol"_n, AglMapping{ (AglType)1, 2, 61 });
    s_aglList.emplace_back("reshhatafsegolhebrew"_n, AglMapping{ (AglType)1, 2, 62 });
    s_aglList.emplace_back("reshhebrew"_n, AglMapping{ (AglType)1, 1, 0x05E8 });
    s_aglList.emplace_back("reshhiriq"_n, AglMapping{ (AglType)1, 2, 63 });
    s_aglList.emplace_back("reshhiriqhebrew"_n, AglMapping{ (AglType)1, 2, 64 });
    s_aglList.emplace_back("reshholam"_n, AglMapping{ (AglType)1, 2, 65 });
    s_aglList.emplace_back("reshholamhebrew"_n, AglMapping{ (AglType)1, 2, 66 });
    s_aglList.emplace_back("reshpatah"_n, AglMapping{ (AglType)1, 2, 67 });
    s_aglList.emplace_back("reshpatahhebrew"_n, AglMapping{ (AglType)1, 2, 68 });
    s_aglList.emplace_back("reshqamats"_n, AglMapping{ (AglType)1, 2, 69 });
    s_aglList.emplace_back("reshqamatshebrew"_n, AglMapping{ (AglType)1, 2, 70 });
    s_aglList.emplace_back("reshqubuts"_n, AglMapping{ (AglType)1, 2, 71 });
    s_aglList.emplace_back("reshqubutshebrew"_n, AglMapping{ (AglType)1, 2, 72 });
    s_aglList.emplace_back("reshsegol"_n, AglMapping{ (AglType)1, 2, 73 });
    s_aglList.emplace_back("reshsegolhebrew"_n, AglMapping{ (AglType)1, 2, 74 });
    s_aglList.emplace_back("reshsheva"_n, AglMapping{ (AglType)1, 2, 75 });
    s_aglList.emplace_back("reshshevahebrew"_n, AglMapping{ (AglType)1, 2, 76 });
    s_aglList.emplace_back("reshtsere"_n, AglMapping{ (AglType)1, 2, 77 });
    s_aglList.emplace_back("reshtserehebrew"_n, AglMapping{ (AglType)1, 2, 78 });
    s_aglList.emplace_back("reversedtilde"_n, AglMapping{ (AglType)1, 1, 0x223D });
    s_aglList.emplace_back("reviahebrew"_n, AglMapping{ (AglType)1, 1, 0x0597 });
    s_aglList.emplace_back("reviamugrashhebrew"_n, AglMapping{ (AglType)1, 1, 0x0597 });
    s_aglList.emplace_back("revlogicalnot"_n, AglMapping{ (AglType)3, 1, 0x2310 });
    s_aglList.emplace_back("rfishhook"_n, AglMapping{ (AglType)1, 1, 0x027E });
    s_aglList.emplace_back("rfishhookreversed"_n, AglMapping{ (AglType)1, 1, 0x027F });
    s_aglList.emplace_back("rhabengali"_n, AglMapping{ (AglType)1, 1, 0x09DD });
    s_aglList.emplace_back("rhadeva"_n, AglMapping{ (AglType)1, 1, 0x095D });
    s_aglList.emplace_back("rho"_n, AglMapping{ (AglType)3, 1, 0x03C1 });
    s_aglList.emplace_back("rhook"_n, AglMapping{ (AglType)1, 1, 0x027D });
    s_aglList.emplace_back("rhookturned"_n, AglMapping{ (AglType)1, 1, 0x027B });
    s_aglList.emplace_back("rhookturnedsuperior"_n, AglMapping{ (AglType)1, 1, 0x02B5 });
    s_aglList.emplace_back("rhosymbolgreek"_n, AglMapping{ (AglType)1, 1, 0x03F1 });
    s_aglList.emplace_back("rhotichookmod"_n, AglMapping{ (AglType)1, 1, 0x02DE });
    s_aglList.emplace_back("rieulacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3271 });
    s_aglList.emplace_back("rieulaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3211 });
    s_aglList.emplace_back("rieulcirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3263 });
    s_aglList.emplace_back("rieulhieuhkorean"_n, AglMapping{ (AglType)1, 1, 0x3140 });
    s_aglList.emplace_back("rieulkiyeokkorean"_n, AglMapping{ (AglType)1, 1, 0x313A });
    s_aglList.emplace_back("rieulkiyeoksioskorean"_n, AglMapping{ (AglType)1, 1, 0x3169 });
    s_aglList.emplace_back("rieulkorean"_n, AglMapping{ (AglType)1, 1, 0x3139 });
    s_aglList.emplace_back("rieulmieumkorean"_n, AglMapping{ (AglType)1, 1, 0x313B });
    s_aglList.emplace_back("rieulpansioskorean"_n, AglMapping{ (AglType)1, 1, 0x316C });
    s_aglList.emplace_back("rieulparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3203 });
    s_aglList.emplace_back("rieulphieuphkorean"_n, AglMapping{ (AglType)1, 1, 0x313F });
    s_aglList.emplace_back("rieulpieupkorean"_n, AglMapping{ (AglType)1, 1, 0x313C });
    s_aglList.emplace_back("rieulpieupsioskorean"_n, AglMapping{ (AglType)1, 1, 0x316B });
    s_aglList.emplace_back("rieulsioskorean"_n, AglMapping{ (AglType)1, 1, 0x313D });
    s_aglList.emplace_back("rieulthieuthkorean"_n, AglMapping{ (AglType)1, 1, 0x313E });
    s_aglList.emplace_back("rieultikeutkorean"_n, AglMapping{ (AglType)1, 1, 0x316A });
    s_aglList.emplace_back("rieulyeorinhieuhkorean"_n, AglMapping{ (AglType)1, 1, 0x316D });
    s_aglList.emplace_back("rightangle"_n, AglMapping{ (AglType)1, 1, 0x221F });
    s_aglList.emplace_back("righttackbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0319 });
    s_aglList.emplace_back("righttriangle"_n, AglMapping{ (AglType)1, 1, 0x22BF });
    s_aglList.emplace_back("rihiragana"_n, AglMapping{ (AglType)1, 1, 0x308A });
    s_aglList.emplace_back("rikatakana"_n, AglMapping{ (AglType)1, 1, 0x30EA });
    s_aglList.emplace_back("rikatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF98 });
    s_aglList.emplace_back("ring"_n, AglMapping{ (AglType)11, 1, 0x02DA });
    s_aglList.emplace_back("ringbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0325 });
    s_aglList.emplace_back("ringcmb"_n, AglMapping{ (AglType)1, 1, 0x030A });
    s_aglList.emplace_back("ringhalfleft"_n, AglMapping{ (AglType)1, 1, 0x02BF });
    s_aglList.emplace_back("ringhalfleftarmenian"_n, AglMapping{ (AglType)1, 1, 0x0559 });
    s_aglList.emplace_back("ringhalfleftbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x031C });
    s_aglList.emplace_back("ringhalfleftcentered"_n, AglMapping{ (AglType)1, 1, 0x02D3 });
    s_aglList.emplace_back("ringhalfright"_n, AglMapping{ (AglType)1, 1, 0x02BE });
    s_aglList.emplace_back("ringhalfrightbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0339 });
    s_aglList.emplace_back("ringhalfrightcentered"_n, AglMapping{ (AglType)1, 1, 0x02D2 });
    s_aglList.emplace_back("rinvertedbreve"_n, AglMapping{ (AglType)1, 1, 0x0213 });
    s_aglList.emplace_back("rittorusquare"_n, AglMapping{ (AglType)1, 1, 0x3351 });
    s_aglList.emplace_back("rlinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E5F });
    s_aglList.emplace_back("rlongleg"_n, AglMapping{ (AglType)1, 1, 0x027C });
    s_aglList.emplace_back("rlonglegturned"_n, AglMapping{ (AglType)1, 1, 0x027A });
    s_aglList.emplace_back("rmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF52 });
    s_aglList.emplace_back("rohiragana"_n, AglMapping{ (AglType)1, 1, 0x308D });
    s_aglList.emplace_back("rokatakana"_n, AglMapping{ (AglType)1, 1, 0x30ED });
    s_aglList.emplace_back("rokatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF9B });
    s_aglList.emplace_back("roruathai"_n, AglMapping{ (AglType)1, 1, 0x0E23 });
    s_aglList.emplace_back("rparen"_n, AglMapping{ (AglType)1, 1, 0x24AD });
    s_aglList.emplace_back("rrabengali"_n, AglMapping{ (AglType)1, 1, 0x09DC });
    s_aglList.emplace_back("rradeva"_n, AglMapping{ (AglType)1, 1, 0x0931 });
    s_aglList.emplace_back("rragurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A5C });
    s_aglList.emplace_back("rreharabic"_n, AglMapping{ (AglType)1, 1, 0x0691 });
    s_aglList.emplace_back("rrehfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFB8D });
    s_aglList.emplace_back("rrvocalicbengali"_n, AglMapping{ (AglType)1, 1, 0x09E0 });
    s_aglList.emplace_back("rrvocalicdeva"_n, AglMapping{ (AglType)1, 1, 0x0960 });
    s_aglList.emplace_back("rrvocalicgujarati"_n, AglMapping{ (AglType)1, 1, 0x0AE0 });
    s_aglList.emplace_back("rrvocalicvowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09C4 });
    s_aglList.emplace_back("rrvocalicvowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0944 });
    s_aglList.emplace_back("rrvocalicvowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0AC4 });
    s_aglList.emplace_back("rsuperior"_n, AglMapping{ (AglType)1, 1, 0xF6F1 });
    s_aglList.emplace_back("rtblock"_n, AglMapping{ (AglType)3, 1, 0x2590 });
    s_aglList.emplace_back("rturned"_n, AglMapping{ (AglType)1, 1, 0x0279 });
    s_aglList.emplace_back("rturnedsuperior"_n, AglMapping{ (AglType)1, 1, 0x02B4 });
    s_aglList.emplace_back("ruhiragana"_n, AglMapping{ (AglType)1, 1, 0x308B });
    s_aglList.emplace_back("rukatakana"_n, AglMapping{ (AglType)1, 1, 0x30EB });
    s_aglList.emplace_back("rukatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF99 });
    s_aglList.emplace_back("rupeemarkbengali"_n, AglMapping{ (AglType)1, 1, 0x09F2 });
    s_aglList.emplace_back("rupeesignbengali"_n, AglMapping{ (AglType)1, 1, 0x09F3 });
    s_aglList.emplace_back("rupiah"_n, AglMapping{ (AglType)1, 1, 0xF6DD });
    s_aglList.emplace_back("ruthai"_n, AglMapping{ (AglType)1, 1, 0x0E24 });
    s_aglList.emplace_back("rvocalicbengali"_n, AglMapping{ (AglType)1, 1, 0x098B });
    s_aglList.emplace_back("rvocalicdeva"_n, AglMapping{ (AglType)1, 1, 0x090B });
    s_aglList.emplace_back("rvocalicgujarati"_n, AglMapping{ (AglType)1, 1, 0x0A8B });
    s_aglList.emplace_back("rvocalicvowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09C3 });
    s_aglList.emplace_back("rvocalicvowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0943 });
    s_aglList.emplace_back("rvocalicvowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0AC3 });
    s_aglList.emplace_back("s"_n, AglMapping{ (AglType)11, 1, 0x0073 });
    s_aglList.emplace_back("sabengali"_n, AglMapping{ (AglType)1, 1, 0x09B8 });
    s_aglList.emplace_back("sacute"_n, AglMapping{ (AglType)3, 1, 0x015B });
    s_aglList.emplace_back("sacutedotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E65 });
    s_aglList.emplace_back("sadarabic"_n, AglMapping{ (AglType)1, 1, 0x0635 });
    s_aglList.emplace_back("sadeva"_n, AglMapping{ (AglType)1, 1, 0x0938 });
    s_aglList.emplace_back("sadfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEBA });
    s_aglList.emplace_back("sadinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEBB });
    s_aglList.emplace_back("sadmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEBC });
    s_aglList.emplace_back("sagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AB8 });
    s_aglList.emplace_back("sagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A38 });
    s_aglList.emplace_back("sahiragana"_n, AglMapping{ (AglType)1, 1, 0x3055 });
    s_aglList.emplace_back("sakatakana"_n, AglMapping{ (AglType)1, 1, 0x30B5 });
    s_aglList.emplace_back("sakatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF7B });
    s_aglList.emplace_back("sallallahoualayhewasallamarabic"_n, AglMapping{ (AglType)1, 1, 0xFDFA });
    s_aglList.emplace_back("samekh"_n, AglMapping{ (AglType)1, 1, 0x05E1 });
    s_aglList.emplace_back("samekhdagesh"_n, AglMapping{ (AglType)1, 1, 0xFB41 });
    s_aglList.emplace_back("samekhdageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB41 });
    s_aglList.emplace_back("samekhhebrew"_n, AglMapping{ (AglType)1, 1, 0x05E1 });
    s_aglList.emplace_back("saraaathai"_n, AglMapping{ (AglType)1, 1, 0x0E32 });
    s_aglList.emplace_back("saraaethai"_n, AglMapping{ (AglType)1, 1, 0x0E41 });
    s_aglList.emplace_back("saraaimaimalaithai"_n, AglMapping{ (AglType)1, 1, 0x0E44 });
    s_aglList.emplace_back("saraaimaimuanthai"_n, AglMapping{ (AglType)1, 1, 0x0E43 });
    s_aglList.emplace_back("saraamthai"_n, AglMapping{ (AglType)1, 1, 0x0E33 });
    s_aglList.emplace_back("saraathai"_n, AglMapping{ (AglType)1, 1, 0x0E30 });
    s_aglList.emplace_back("saraethai"_n, AglMapping{ (AglType)1, 1, 0x0E40 });
    s_aglList.emplace_back("saraiileftthai"_n, AglMapping{ (AglType)1, 1, 0xF886 });
    s_aglList.emplace_back("saraiithai"_n, AglMapping{ (AglType)1, 1, 0x0E35 });
    s_aglList.emplace_back("saraileftthai"_n, AglMapping{ (AglType)1, 1, 0xF885 });
    s_aglList.emplace_back("saraithai"_n, AglMapping{ (AglType)1, 1, 0x0E34 });
    s_aglList.emplace_back("saraothai"_n, AglMapping{ (AglType)1, 1, 0x0E42 });
    s_aglList.emplace_back("saraueeleftthai"_n, AglMapping{ (AglType)1, 1, 0xF888 });
    s_aglList.emplace_back("saraueethai"_n, AglMapping{ (AglType)1, 1, 0x0E37 });
    s_aglList.emplace_back("saraueleftthai"_n, AglMapping{ (AglType)1, 1, 0xF887 });
    s_aglList.emplace_back("sarauethai"_n, AglMapping{ (AglType)1, 1, 0x0E36 });
    s_aglList.emplace_back("sarauthai"_n, AglMapping{ (AglType)1, 1, 0x0E38 });
    s_aglList.emplace_back("sarauuthai"_n, AglMapping{ (AglType)1, 1, 0x0E39 });
    s_aglList.emplace_back("sbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3119 });
    s_aglList.emplace_back("scaron"_n, AglMapping{ (AglType)11, 1, 0x0161 });
    s_aglList.emplace_back("scarondotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E67 });
    s_aglList.emplace_back("scedilla"_n, AglMapping{ (AglType)3, 1, 0x015F });
    s_aglList.emplace_back("schwa"_n, AglMapping{ (AglType)1, 1, 0x0259 });
    s_aglList.emplace_back("schwacyrillic"_n, AglMapping{ (AglType)1, 1, 0x04D9 });
    s_aglList.emplace_back("schwadieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04DB });
    s_aglList.emplace_back("schwahook"_n, AglMapping{ (AglType)1, 1, 0x025A });
    s_aglList.emplace_back("scircle"_n, AglMapping{ (AglType)1, 1, 0x24E2 });
    s_aglList.emplace_back("scircumflex"_n, AglMapping{ (AglType)3, 1, 0x015D });
    s_aglList.emplace_back("scommaaccent"_n, AglMapping{ (AglType)1, 1, 0x0219 });
    s_aglList.emplace_back("sdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E61 });
    s_aglList.emplace_back("sdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E63 });
    s_aglList.emplace_back("sdotbelowdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E69 });
    s_aglList.emplace_back("seagullbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x033C });
    s_aglList.emplace_back("second"_n, AglMapping{ (AglType)3, 1, 0x2033 });
    s_aglList.emplace_back("secondtonechinese"_n, AglMapping{ (AglType)1, 1, 0x02CA });
    s_aglList.emplace_back("section"_n, AglMapping{ (AglType)11, 1, 0x00A7 });
    s_aglList.emplace_back("seenarabic"_n, AglMapping{ (AglType)1, 1, 0x0633 });
    s_aglList.emplace_back("seenfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEB2 });
    s_aglList.emplace_back("seeninitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEB3 });
    s_aglList.emplace_back("seenmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEB4 });
    s_aglList.emplace_back("segol"_n, AglMapping{ (AglType)1, 1, 0x05B6 });
    s_aglList.emplace_back("segol13"_n, AglMapping{ (AglType)1, 1, 0x05B6 });
    s_aglList.emplace_back("segol1f"_n, AglMapping{ (AglType)1, 1, 0x05B6 });
    s_aglList.emplace_back("segol2c"_n, AglMapping{ (AglType)1, 1, 0x05B6 });
    s_aglList.emplace_back("segolhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B6 });
    s_aglList.emplace_back("segolnarrowhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B6 });
    s_aglList.emplace_back("segolquarterhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B6 });
    s_aglList.emplace_back("segoltahebrew"_n, AglMapping{ (AglType)1, 1, 0x0592 });
    s_aglList.emplace_back("segolwidehebrew"_n, AglMapping{ (AglType)1, 1, 0x05B6 });
    s_aglList.emplace_back("seharmenian"_n, AglMapping{ (AglType)1, 1, 0x057D });
    s_aglList.emplace_back("sehiragana"_n, AglMapping{ (AglType)1, 1, 0x305B });
    s_aglList.emplace_back("sekatakana"_n, AglMapping{ (AglType)1, 1, 0x30BB });
    s_aglList.emplace_back("sekatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF7E });
    s_aglList.emplace_back("semicolon"_n, AglMapping{ (AglType)11, 1, 0x003B });
    s_aglList.emplace_back("semicolonarabic"_n, AglMapping{ (AglType)1, 1, 0x061B });
    s_aglList.emplace_back("semicolonmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF1B });
    s_aglList.emplace_back("semicolonsmall"_n, AglMapping{ (AglType)1, 1, 0xFE54 });
    s_aglList.emplace_back("semivoicedmarkkana"_n, AglMapping{ (AglType)1, 1, 0x309C });
    s_aglList.emplace_back("semivoicedmarkkanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF9F });
    s_aglList.emplace_back("sentisquare"_n, AglMapping{ (AglType)1, 1, 0x3322 });
    s_aglList.emplace_back("sentosquare"_n, AglMapping{ (AglType)1, 1, 0x3323 });
    s_aglList.emplace_back("seven"_n, AglMapping{ (AglType)11, 1, 0x0037 });
    s_aglList.emplace_back("sevenarabic"_n, AglMapping{ (AglType)1, 1, 0x0667 });
    s_aglList.emplace_back("sevenbengali"_n, AglMapping{ (AglType)1, 1, 0x09ED });
    s_aglList.emplace_back("sevencircle"_n, AglMapping{ (AglType)1, 1, 0x2466 });
    s_aglList.emplace_back("sevencircleinversesansserif"_n, AglMapping{ (AglType)1, 1, 0x2790 });
    s_aglList.emplace_back("sevendeva"_n, AglMapping{ (AglType)1, 1, 0x096D });
    s_aglList.emplace_back("seveneighths"_n, AglMapping{ (AglType)3, 1, 0x215E });
    s_aglList.emplace_back("sevengujarati"_n, AglMapping{ (AglType)1, 1, 0x0AED });
    s_aglList.emplace_back("sevengurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A6D });
    s_aglList.emplace_back("sevenhackarabic"_n, AglMapping{ (AglType)1, 1, 0x0667 });
    s_aglList.emplace_back("sevenhangzhou"_n, AglMapping{ (AglType)1, 1, 0x3027 });
    s_aglList.emplace_back("sevenideographicparen"_n, AglMapping{ (AglType)1, 1, 0x3226 });
    s_aglList.emplace_back("seveninferior"_n, AglMapping{ (AglType)1, 1, 0x2087 });
    s_aglList.emplace_back("sevenmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF17 });
    s_aglList.emplace_back("sevenoldstyle"_n, AglMapping{ (AglType)1, 1, 0xF737 });
    s_aglList.emplace_back("sevenparen"_n, AglMapping{ (AglType)1, 1, 0x247A });
    s_aglList.emplace_back("sevenperiod"_n, AglMapping{ (AglType)1, 1, 0x248E });
    s_aglList.emplace_back("sevenpersian"_n, AglMapping{ (AglType)1, 1, 0x06F7 });
    s_aglList.emplace_back("sevenroman"_n, AglMapping{ (AglType)1, 1, 0x2176 });
    s_aglList.emplace_back("sevensuperior"_n, AglMapping{ (AglType)1, 1, 0x2077 });
    s_aglList.emplace_back("seventeencircle"_n, AglMapping{ (AglType)1, 1, 0x2470 });
    s_aglList.emplace_back("seventeenparen"_n, AglMapping{ (AglType)1, 1, 0x2484 });
    s_aglList.emplace_back("seventeenperiod"_n, AglMapping{ (AglType)1, 1, 0x2498 });
    s_aglList.emplace_back("seventhai"_n, AglMapping{ (AglType)1, 1, 0x0E57 });
    s_aglList.emplace_back("sfthyphen"_n, AglMapping{ (AglType)1, 1, 0x00AD });
    s_aglList.emplace_back("shaarmenian"_n, AglMapping{ (AglType)1, 1, 0x0577 });
    s_aglList.emplace_back("shabengali"_n, AglMapping{ (AglType)1, 1, 0x09B6 });
    s_aglList.emplace_back("shacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0448 });
    s_aglList.emplace_back("shaddaarabic"_n, AglMapping{ (AglType)1, 1, 0x0651 });
    s_aglList.emplace_back("shaddadammaarabic"_n, AglMapping{ (AglType)1, 1, 0xFC61 });
    s_aglList.emplace_back("shaddadammatanarabic"_n, AglMapping{ (AglType)1, 1, 0xFC5E });
    s_aglList.emplace_back("shaddafathaarabic"_n, AglMapping{ (AglType)1, 1, 0xFC60 });
    s_aglList.emplace_back("shaddafathatanarabic"_n, AglMapping{ (AglType)1, 2, 79 });
    s_aglList.emplace_back("shaddakasraarabic"_n, AglMapping{ (AglType)1, 1, 0xFC62 });
    s_aglList.emplace_back("shaddakasratanarabic"_n, AglMapping{ (AglType)1, 1, 0xFC5F });
    s_aglList.emplace_back("shade"_n, AglMapping{ (AglType)3, 1, 0x2592 });
    s_aglList.emplace_back("shadedark"_n, AglMapping{ (AglType)1, 1, 0x2593 });
    s_aglList.emplace_back("shadelight"_n, AglMapping{ (AglType)1, 1, 0x2591 });
    s_aglList.emplace_back("shademedium"_n, AglMapping{ (AglType)1, 1, 0x2592 });
    s_aglList.emplace_back("shadeva"_n, AglMapping{ (AglType)1, 1, 0x0936 });
    s_aglList.emplace_back("shagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AB6 });
    s_aglList.emplace_back("shagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A36 });
    s_aglList.emplace_back("shalshelethebrew"_n, AglMapping{ (AglType)1, 1, 0x0593 });
    s_aglList.emplace_back("shbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3115 });
    s_aglList.emplace_back("shchacyrillic"_n, AglMapping{ (AglType)1, 1, 0x0449 });
    s_aglList.emplace_back("sheenarabic"_n, AglMapping{ (AglType)1, 1, 0x0634 });
    s_aglList.emplace_back("sheenfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEB6 });
    s_aglList.emplace_back("sheeninitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEB7 });
    s_aglList.emplace_back("sheenmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEB8 });
    s_aglList.emplace_back("sheicoptic"_n, AglMapping{ (AglType)1, 1, 0x03E3 });
    s_aglList.emplace_back("sheqel"_n, AglMapping{ (AglType)1, 1, 0x20AA });
    s_aglList.emplace_back("sheqelhebrew"_n, AglMapping{ (AglType)1, 1, 0x20AA });
    s_aglList.emplace_back("sheva"_n, AglMapping{ (AglType)1, 1, 0x05B0 });
    s_aglList.emplace_back("sheva115"_n, AglMapping{ (AglType)1, 1, 0x05B0 });
    s_aglList.emplace_back("sheva15"_n, AglMapping{ (AglType)1, 1, 0x05B0 });
    s_aglList.emplace_back("sheva22"_n, AglMapping{ (AglType)1, 1, 0x05B0 });
    s_aglList.emplace_back("sheva2e"_n, AglMapping{ (AglType)1, 1, 0x05B0 });
    s_aglList.emplace_back("shevahebrew"_n, AglMapping{ (AglType)1, 1, 0x05B0 });
    s_aglList.emplace_back("shevanarrowhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B0 });
    s_aglList.emplace_back("shevaquarterhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B0 });
    s_aglList.emplace_back("shevawidehebrew"_n, AglMapping{ (AglType)1, 1, 0x05B0 });
    s_aglList.emplace_back("shhacyrillic"_n, AglMapping{ (AglType)1, 1, 0x04BB });
    s_aglList.emplace_back("shimacoptic"_n, AglMapping{ (AglType)1, 1, 0x03ED });
    s_aglList.emplace_back("shin"_n, AglMapping{ (AglType)1, 1, 0x05E9 });
    s_aglList.emplace_back("shindagesh"_n, AglMapping{ (AglType)1, 1, 0xFB49 });
    s_aglList.emplace_back("shindageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB49 });
    s_aglList.emplace_back("shindageshshindot"_n, AglMapping{ (AglType)1, 1, 0xFB2C });
    s_aglList.emplace_back("shindageshshindothebrew"_n, AglMapping{ (AglType)1, 1, 0xFB2C });
    s_aglList.emplace_back("shindageshsindot"_n, AglMapping{ (AglType)1, 1, 0xFB2D });
    s_aglList.emplace_back("shindageshsindothebrew"_n, AglMapping{ (AglType)1, 1, 0xFB2D });
    s_aglList.emplace_back("shindothebrew"_n, AglMapping{ (AglType)1, 1, 0x05C1 });
    s_aglList.emplace_back("shinhebrew"_n, AglMapping{ (AglType)1, 1, 0x05E9 });
    s_aglList.emplace_back("shinshindot"_n, AglMapping{ (AglType)1, 1, 0xFB2A });
    s_aglList.emplace_back("shinshindothebrew"_n, AglMapping{ (AglType)1, 1, 0xFB2A });
    s_aglList.emplace_back("shinsindot"_n, AglMapping{ (AglType)1, 1, 0xFB2B });
    s_aglList.emplace_back("shinsindothebrew"_n, AglMapping{ (AglType)1, 1, 0xFB2B });
    s_aglList.emplace_back("shook"_n, AglMapping{ (AglType)1, 1, 0x0282 });
    s_aglList.emplace_back("sigma"_n, AglMapping{ (AglType)3, 1, 0x03C3 });
    s_aglList.emplace_back("sigma1"_n, AglMapping{ (AglType)3, 1, 0x03C2 });
    s_aglList.emplace_back("sigmafinal"_n, AglMapping{ (AglType)1, 1, 0x03C2 });
    s_aglList.emplace_back("sigmalunatesymbolgreek"_n, AglMapping{ (AglType)1, 1, 0x03F2 });
    s_aglList.emplace_back("sihiragana"_n, AglMapping{ (AglType)1, 1, 0x3057 });
    s_aglList.emplace_back("sikatakana"_n, AglMapping{ (AglType)1, 1, 0x30B7 });
    s_aglList.emplace_back("sikatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF7C });
    s_aglList.emplace_back("siluqhebrew"_n, AglMapping{ (AglType)1, 1, 0x05BD });
    s_aglList.emplace_back("siluqlefthebrew"_n, AglMapping{ (AglType)1, 1, 0x05BD });
    s_aglList.emplace_back("similar"_n, AglMapping{ (AglType)3, 1, 0x223C });
    s_aglList.emplace_back("sindothebrew"_n, AglMapping{ (AglType)1, 1, 0x05C2 });
    s_aglList.emplace_back("siosacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3274 });
    s_aglList.emplace_back("siosaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3214 });
    s_aglList.emplace_back("sioscieuckorean"_n, AglMapping{ (AglType)1, 1, 0x317E });
    s_aglList.emplace_back("sioscirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3266 });
    s_aglList.emplace_back("sioskiyeokkorean"_n, AglMapping{ (AglType)1, 1, 0x317A });
    s_aglList.emplace_back("sioskorean"_n, AglMapping{ (AglType)1, 1, 0x3145 });
    s_aglList.emplace_back("siosnieunkorean"_n, AglMapping{ (AglType)1, 1, 0x317B });
    s_aglList.emplace_back("siosparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3206 });
    s_aglList.emplace_back("siospieupkorean"_n, AglMapping{ (AglType)1, 1, 0x317D });
    s_aglList.emplace_back("siostikeutkorean"_n, AglMapping{ (AglType)1, 1, 0x317C });
    s_aglList.emplace_back("six"_n, AglMapping{ (AglType)11, 1, 0x0036 });
    s_aglList.emplace_back("sixarabic"_n, AglMapping{ (AglType)1, 1, 0x0666 });
    s_aglList.emplace_back("sixbengali"_n, AglMapping{ (AglType)1, 1, 0x09EC });
    s_aglList.emplace_back("sixcircle"_n, AglMapping{ (AglType)1, 1, 0x2465 });
    s_aglList.emplace_back("sixcircleinversesansserif"_n, AglMapping{ (AglType)1, 1, 0x278F });
    s_aglList.emplace_back("sixdeva"_n, AglMapping{ (AglType)1, 1, 0x096C });
    s_aglList.emplace_back("sixgujarati"_n, AglMapping{ (AglType)1, 1, 0x0AEC });
    s_aglList.emplace_back("sixgurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A6C });
    s_aglList.emplace_back("sixhackarabic"_n, AglMapping{ (AglType)1, 1, 0x0666 });
    s_aglList.emplace_back("sixhangzhou"_n, AglMapping{ (AglType)1, 1, 0x3026 });
    s_aglList.emplace_back("sixideographicparen"_n, AglMapping{ (AglType)1, 1, 0x3225 });
    s_aglList.emplace_back("sixinferior"_n, AglMapping{ (AglType)1, 1, 0x2086 });
    s_aglList.emplace_back("sixmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF16 });
    s_aglList.emplace_back("sixoldstyle"_n, AglMapping{ (AglType)1, 1, 0xF736 });
    s_aglList.emplace_back("sixparen"_n, AglMapping{ (AglType)1, 1, 0x2479 });
    s_aglList.emplace_back("sixperiod"_n, AglMapping{ (AglType)1, 1, 0x248D });
    s_aglList.emplace_back("sixpersian"_n, AglMapping{ (AglType)1, 1, 0x06F6 });
    s_aglList.emplace_back("sixroman"_n, AglMapping{ (AglType)1, 1, 0x2175 });
    s_aglList.emplace_back("sixsuperior"_n, AglMapping{ (AglType)1, 1, 0x2076 });
    s_aglList.emplace_back("sixteencircle"_n, AglMapping{ (AglType)1, 1, 0x246F });
    s_aglList.emplace_back("sixteencurrencydenominatorbengali"_n, AglMapping{ (AglType)1, 1, 0x09F9 });
    s_aglList.emplace_back("sixteenparen"_n, AglMapping{ (AglType)1, 1, 0x2483 });
    s_aglList.emplace_back("sixteenperiod"_n, AglMapping{ (AglType)1, 1, 0x2497 });
    s_aglList.emplace_back("sixthai"_n, AglMapping{ (AglType)1, 1, 0x0E56 });
    s_aglList.emplace_back("slash"_n, AglMapping{ (AglType)11, 1, 0x002F });
    s_aglList.emplace_back("slashmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF0F });
    s_aglList.emplace_back("slong"_n, AglMapping{ (AglType)1, 1, 0x017F });
    s_aglList.emplace_back("slongdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E9B });
    s_aglList.emplace_back("smileface"_n, AglMapping{ (AglType)3, 1, 0x263A });
    s_aglList.emplace_back("smonospace"_n, AglMapping{ (AglType)1, 1, 0xFF53 });
    s_aglList.emplace_back("sofpasuqhebrew"_n, AglMapping{ (AglType)1, 1, 0x05C3 });
    s_aglList.emplace_back("softhyphen"_n, AglMapping{ (AglType)1, 1, 0x00AD });
    s_aglList.emplace_back("softsigncyrillic"_n, AglMapping{ (AglType)1, 1, 0x044C });
    s_aglList.emplace_back("sohiragana"_n, AglMapping{ (AglType)1, 1, 0x305D });
    s_aglList.emplace_back("sokatakana"_n, AglMapping{ (AglType)1, 1, 0x30BD });
    s_aglList.emplace_back("sokatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF7F });
    s_aglList.emplace_back("soliduslongoverlaycmb"_n, AglMapping{ (AglType)1, 1, 0x0338 });
    s_aglList.emplace_back("solidusshortoverlaycmb"_n, AglMapping{ (AglType)1, 1, 0x0337 });
    s_aglList.emplace_back("sorusithai"_n, AglMapping{ (AglType)1, 1, 0x0E29 });
    s_aglList.emplace_back("sosalathai"_n, AglMapping{ (AglType)1, 1, 0x0E28 });
    s_aglList.emplace_back("sosothai"_n, AglMapping{ (AglType)1, 1, 0x0E0B });
    s_aglList.emplace_back("sosuathai"_n, AglMapping{ (AglType)1, 1, 0x0E2A });
    s_aglList.emplace_back("space"_n, AglMapping{ (AglType)11, 1, 0x0020 });
    s_aglList.emplace_back("spacehackarabic"_n, AglMapping{ (AglType)1, 1, 0x0020 });
    s_aglList.emplace_back("spade"_n, AglMapping{ (AglType)3, 1, 0x2660 });
    s_aglList.emplace_back("spadesuitblack"_n, AglMapping{ (AglType)1, 1, 0x2660 });
    s_aglList.emplace_back("spadesuitwhite"_n, AglMapping{ (AglType)1, 1, 0x2664 });
    s_aglList.emplace_back("sparen"_n, AglMapping{ (AglType)1, 1, 0x24AE });
    s_aglList.emplace_back("squarebelowcmb"_n, AglMapping{ (AglType)1, 1, 0x033B });
    s_aglList.emplace_back("squarecc"_n, AglMapping{ (AglType)1, 1, 0x33C4 });
    s_aglList.emplace_back("squarecm"_n, AglMapping{ (AglType)1, 1, 0x339D });
    s_aglList.emplace_back("squarediagonalcrosshatchfill"_n, AglMapping{ (AglType)1, 1, 0x25A9 });
    s_aglList.emplace_back("squarehorizontalfill"_n, AglMapping{ (AglType)1, 1, 0x25A4 });
    s_aglList.emplace_back("squarekg"_n, AglMapping{ (AglType)1, 1, 0x338F });
    s_aglList.emplace_back("squarekm"_n, AglMapping{ (AglType)1, 1, 0x339E });
    s_aglList.emplace_back("squarekmcapital"_n, AglMapping{ (AglType)1, 1, 0x33CE });
    s_aglList.emplace_back("squareln"_n, AglMapping{ (AglType)1, 1, 0x33D1 });
    s_aglList.emplace_back("squarelog"_n, AglMapping{ (AglType)1, 1, 0x33D2 });
    s_aglList.emplace_back("squaremg"_n, AglMapping{ (AglType)1, 1, 0x338E });
    s_aglList.emplace_back("squaremil"_n, AglMapping{ (AglType)1, 1, 0x33D5 });
    s_aglList.emplace_back("squaremm"_n, AglMapping{ (AglType)1, 1, 0x339C });
    s_aglList.emplace_back("squaremsquared"_n, AglMapping{ (AglType)1, 1, 0x33A1 });
    s_aglList.emplace_back("squareorthogonalcrosshatchfill"_n, AglMapping{ (AglType)1, 1, 0x25A6 });
    s_aglList.emplace_back("squareupperlefttolowerrightfill"_n, AglMapping{ (AglType)1, 1, 0x25A7 });
    s_aglList.emplace_back("squareupperrighttolowerleftfill"_n, AglMapping{ (AglType)1, 1, 0x25A8 });
    s_aglList.emplace_back("squareverticalfill"_n, AglMapping{ (AglType)1, 1, 0x25A5 });
    s_aglList.emplace_back("squarewhitewithsmallblack"_n, AglMapping{ (AglType)1, 1, 0x25A3 });
    s_aglList.emplace_back("srsquare"_n, AglMapping{ (AglType)1, 1, 0x33DB });
    s_aglList.emplace_back("ssabengali"_n, AglMapping{ (AglType)1, 1, 0x09B7 });
    s_aglList.emplace_back("ssadeva"_n, AglMapping{ (AglType)1, 1, 0x0937 });
    s_aglList.emplace_back("ssagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AB7 });
    s_aglList.emplace_back("ssangcieuckorean"_n, AglMapping{ (AglType)1, 1, 0x3149 });
    s_aglList.emplace_back("ssanghieuhkorean"_n, AglMapping{ (AglType)1, 1, 0x3185 });
    s_aglList.emplace_back("ssangieungkorean"_n, AglMapping{ (AglType)1, 1, 0x3180 });
    s_aglList.emplace_back("ssangkiyeokkorean"_n, AglMapping{ (AglType)1, 1, 0x3132 });
    s_aglList.emplace_back("ssangnieunkorean"_n, AglMapping{ (AglType)1, 1, 0x3165 });
    s_aglList.emplace_back("ssangpieupkorean"_n, AglMapping{ (AglType)1, 1, 0x3143 });
    s_aglList.emplace_back("ssangsioskorean"_n, AglMapping{ (AglType)1, 1, 0x3146 });
    s_aglList.emplace_back("ssangtikeutkorean"_n, AglMapping{ (AglType)1, 1, 0x3138 });
    s_aglList.emplace_back("ssuperior"_n, AglMapping{ (AglType)1, 1, 0xF6F2 });
    s_aglList.emplace_back("sterling"_n, AglMapping{ (AglType)11, 1, 0x00A3 });
    s_aglList.emplace_back("sterlingmonospace"_n, AglMapping{ (AglType)1, 1, 0xFFE1 });
    s_aglList.emplace_back("strokelongoverlaycmb"_n, AglMapping{ (AglType)1, 1, 0x0336 });
    s_aglList.emplace_back("strokeshortoverlaycmb"_n, AglMapping{ (AglType)1, 1, 0x0335 });
    s_aglList.emplace_back("subset"_n, AglMapping{ (AglType)1, 1, 0x2282 });
    s_aglList.emplace_back("subsetnotequal"_n, AglMapping{ (AglType)1, 1, 0x228A });
    s_aglList.emplace_back("subsetorequal"_n, AglMapping{ (AglType)1, 1, 0x2286 });
    s_aglList.emplace_back("succeeds"_n, AglMapping{ (AglType)1, 1, 0x227B });
    s_aglList.emplace_back("suchthat"_n, AglMapping{ (AglType)3, 1, 0x220B });
    s_aglList.emplace_back("suhiragana"_n, AglMapping{ (AglType)1, 1, 0x3059 });
    s_aglList.emplace_back("sukatakana"_n, AglMapping{ (AglType)1, 1, 0x30B9 });
    s_aglList.emplace_back("sukatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF7D });
    s_aglList.emplace_back("sukunarabic"_n, AglMapping{ (AglType)1, 1, 0x0652 });
    s_aglList.emplace_back("summation"_n, AglMapping{ (AglType)3, 1, 0x2211 });
    s_aglList.emplace_back("sun"_n, AglMapping{ (AglType)3, 1, 0x263C });
    s_aglList.emplace_back("superset"_n, AglMapping{ (AglType)1, 1, 0x2283 });
    s_aglList.emplace_back("supersetnotequal"_n, AglMapping{ (AglType)1, 1, 0x228B });
    s_aglList.emplace_back("supersetorequal"_n, AglMapping{ (AglType)1, 1, 0x2287 });
    s_aglList.emplace_back("svsquare"_n, AglMapping{ (AglType)1, 1, 0x33DC });
    s_aglList.emplace_back("syouwaerasquare"_n, AglMapping{ (AglType)1, 1, 0x337C });
    s_aglList.emplace_back("t"_n, AglMapping{ (AglType)11, 1, 0x0074 });
    s_aglList.emplace_back("tabengali"_n, AglMapping{ (AglType)1, 1, 0x09A4 });
    s_aglList.emplace_back("tackdown"_n, AglMapping{ (AglType)1, 1, 0x22A4 });
    s_aglList.emplace_back("tackleft"_n, AglMapping{ (AglType)1, 1, 0x22A3 });
    s_aglList.emplace_back("tadeva"_n, AglMapping{ (AglType)1, 1, 0x0924 });
    s_aglList.emplace_back("tagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AA4 });
    s_aglList.emplace_back("tagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A24 });
    s_aglList.emplace_back("taharabic"_n, AglMapping{ (AglType)1, 1, 0x0637 });
    s_aglList.emplace_back("tahfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEC2 });
    s_aglList.emplace_back("tahinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEC3 });
    s_aglList.emplace_back("tahiragana"_n, AglMapping{ (AglType)1, 1, 0x305F });
    s_aglList.emplace_back("tahmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEC4 });
    s_aglList.emplace_back("taisyouerasquare"_n, AglMapping{ (AglType)1, 1, 0x337D });
    s_aglList.emplace_back("takatakana"_n, AglMapping{ (AglType)1, 1, 0x30BF });
    s_aglList.emplace_back("takatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF80 });
    s_aglList.emplace_back("tatweelarabic"_n, AglMapping{ (AglType)1, 1, 0x0640 });
    s_aglList.emplace_back("tau"_n, AglMapping{ (AglType)3, 1, 0x03C4 });
    s_aglList.emplace_back("tav"_n, AglMapping{ (AglType)1, 1, 0x05EA });
    s_aglList.emplace_back("tavdages"_n, AglMapping{ (AglType)1, 1, 0xFB4A });
    s_aglList.emplace_back("tavdagesh"_n, AglMapping{ (AglType)1, 1, 0xFB4A });
    s_aglList.emplace_back("tavdageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB4A });
    s_aglList.emplace_back("tavhebrew"_n, AglMapping{ (AglType)1, 1, 0x05EA });
    s_aglList.emplace_back("tbar"_n, AglMapping{ (AglType)3, 1, 0x0167 });
    s_aglList.emplace_back("tbopomofo"_n, AglMapping{ (AglType)1, 1, 0x310A });
    s_aglList.emplace_back("tcaron"_n, AglMapping{ (AglType)3, 1, 0x0165 });
    s_aglList.emplace_back("tccurl"_n, AglMapping{ (AglType)1, 1, 0x02A8 });
    s_aglList.emplace_back("tcedilla"_n, AglMapping{ (AglType)1, 1, 0x0163 });
    s_aglList.emplace_back("tcheharabic"_n, AglMapping{ (AglType)1, 1, 0x0686 });
    s_aglList.emplace_back("tchehfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFB7B });
    s_aglList.emplace_back("tchehinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFB7C });
    s_aglList.emplace_back("tchehmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFB7D });
    s_aglList.emplace_back("tchehmeeminitialarabic"_n, AglMapping{ (AglType)1, 2, 80 });
    s_aglList.emplace_back("tcircle"_n, AglMapping{ (AglType)1, 1, 0x24E3 });
    s_aglList.emplace_back("tcircumflexbelow"_n, AglMapping{ (AglType)1, 1, 0x1E71 });
    s_aglList.emplace_back("tcommaaccent"_n, AglMapping{ (AglType)1, 1, 0x0163 });
    s_aglList.emplace_back("tdieresis"_n, AglMapping{ (AglType)1, 1, 0x1E97 });
    s_aglList.emplace_back("tdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E6B });
    s_aglList.emplace_back("tdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E6D });
    s_aglList.emplace_back("tecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0442 });
    s_aglList.emplace_back("tedescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x04AD });
    s_aglList.emplace_back("teharabic"_n, AglMapping{ (AglType)1, 1, 0x062A });
    s_aglList.emplace_back("tehfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFE96 });
    s_aglList.emplace_back("tehhahinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFCA2 });
    s_aglList.emplace_back("tehhahisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFC0C });
    s_aglList.emplace_back("tehinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFE97 });
    s_aglList.emplace_back("tehiragana"_n, AglMapping{ (AglType)1, 1, 0x3066 });
    s_aglList.emplace_back("tehjeeminitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFCA1 });
    s_aglList.emplace_back("tehjeemisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFC0B });
    s_aglList.emplace_back("tehmarbutaarabic"_n, AglMapping{ (AglType)1, 1, 0x0629 });
    s_aglList.emplace_back("tehmarbutafinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFE94 });
    s_aglList.emplace_back("tehmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFE98 });
    s_aglList.emplace_back("tehmeeminitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFCA4 });
    s_aglList.emplace_back("tehmeemisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFC0E });
    s_aglList.emplace_back("tehnoonfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFC73 });
    s_aglList.emplace_back("tekatakana"_n, AglMapping{ (AglType)1, 1, 0x30C6 });
    s_aglList.emplace_back("tekatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF83 });
    s_aglList.emplace_back("telephone"_n, AglMapping{ (AglType)1, 1, 0x2121 });
    s_aglList.emplace_back("telephoneblack"_n, AglMapping{ (AglType)1, 1, 0x260E });
    s_aglList.emplace_back("telishagedolahebrew"_n, AglMapping{ (AglType)1, 1, 0x05A0 });
    s_aglList.emplace_back("telishaqetanahebrew"_n, AglMapping{ (AglType)1, 1, 0x05A9 });
    s_aglList.emplace_back("tencircle"_n, AglMapping{ (AglType)1, 1, 0x2469 });
    s_aglList.emplace_back("tenideographicparen"_n, AglMapping{ (AglType)1, 1, 0x3229 });
    s_aglList.emplace_back("tenparen"_n, AglMapping{ (AglType)1, 1, 0x247D });
    s_aglList.emplace_back("tenperiod"_n, AglMapping{ (AglType)1, 1, 0x2491 });
    s_aglList.emplace_back("tenroman"_n, AglMapping{ (AglType)1, 1, 0x2179 });
    s_aglList.emplace_back("tesh"_n, AglMapping{ (AglType)1, 1, 0x02A7 });
    s_aglList.emplace_back("tet"_n, AglMapping{ (AglType)1, 1, 0x05D8 });
    s_aglList.emplace_back("tetdagesh"_n, AglMapping{ (AglType)1, 1, 0xFB38 });
    s_aglList.emplace_back("tetdageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB38 });
    s_aglList.emplace_back("tethebrew"_n, AglMapping{ (AglType)1, 1, 0x05D8 });
    s_aglList.emplace_back("tetsecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04B5 });
    s_aglList.emplace_back("tevirhebrew"_n, AglMapping{ (AglType)1, 1, 0x059B });
    s_aglList.emplace_back("tevirlefthebrew"_n, AglMapping{ (AglType)1, 1, 0x059B });
    s_aglList.emplace_back("thabengali"_n, AglMapping{ (AglType)1, 1, 0x09A5 });
    s_aglList.emplace_back("thadeva"_n, AglMapping{ (AglType)1, 1, 0x0925 });
    s_aglList.emplace_back("thagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AA5 });
    s_aglList.emplace_back("thagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A25 });
    s_aglList.emplace_back("thalarabic"_n, AglMapping{ (AglType)1, 1, 0x0630 });
    s_aglList.emplace_back("thalfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEAC });
    s_aglList.emplace_back("thanthakhatlowleftthai"_n, AglMapping{ (AglType)1, 1, 0xF898 });
    s_aglList.emplace_back("thanthakhatlowrightthai"_n, AglMapping{ (AglType)1, 1, 0xF897 });
    s_aglList.emplace_back("thanthakhatthai"_n, AglMapping{ (AglType)1, 1, 0x0E4C });
    s_aglList.emplace_back("thanthakhatupperleftthai"_n, AglMapping{ (AglType)1, 1, 0xF896 });
    s_aglList.emplace_back("theharabic"_n, AglMapping{ (AglType)1, 1, 0x062B });
    s_aglList.emplace_back("thehfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFE9A });
    s_aglList.emplace_back("thehinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFE9B });
    s_aglList.emplace_back("thehmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFE9C });
    s_aglList.emplace_back("thereexists"_n, AglMapping{ (AglType)1, 1, 0x2203 });
    s_aglList.emplace_back("therefore"_n, AglMapping{ (AglType)3, 1, 0x2234 });
    s_aglList.emplace_back("theta"_n, AglMapping{ (AglType)3, 1, 0x03B8 });
    s_aglList.emplace_back("theta1"_n, AglMapping{ (AglType)3, 1, 0x03D1 });
    s_aglList.emplace_back("thetasymbolgreek"_n, AglMapping{ (AglType)1, 1, 0x03D1 });
    s_aglList.emplace_back("thieuthacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3279 });
    s_aglList.emplace_back("thieuthaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3219 });
    s_aglList.emplace_back("thieuthcirclekorean"_n, AglMapping{ (AglType)1, 1, 0x326B });
    s_aglList.emplace_back("thieuthkorean"_n, AglMapping{ (AglType)1, 1, 0x314C });
    s_aglList.emplace_back("thieuthparenkorean"_n, AglMapping{ (AglType)1, 1, 0x320B });
    s_aglList.emplace_back("thirteencircle"_n, AglMapping{ (AglType)1, 1, 0x246C });
    s_aglList.emplace_back("thirteenparen"_n, AglMapping{ (AglType)1, 1, 0x2480 });
    s_aglList.emplace_back("thirteenperiod"_n, AglMapping{ (AglType)1, 1, 0x2494 });
    s_aglList.emplace_back("thonangmonthothai"_n, AglMapping{ (AglType)1, 1, 0x0E11 });
    s_aglList.emplace_back("thook"_n, AglMapping{ (AglType)1, 1, 0x01AD });
    s_aglList.emplace_back("thophuthaothai"_n, AglMapping{ (AglType)1, 1, 0x0E12 });
    s_aglList.emplace_back("thorn"_n, AglMapping{ (AglType)11, 1, 0x00FE });
    s_aglList.emplace_back("thothahanthai"_n, AglMapping{ (AglType)1, 1, 0x0E17 });
    s_aglList.emplace_back("thothanthai"_n, AglMapping{ (AglType)1, 1, 0x0E10 });
    s_aglList.emplace_back("thothongthai"_n, AglMapping{ (AglType)1, 1, 0x0E18 });
    s_aglList.emplace_back("thothungthai"_n, AglMapping{ (AglType)1, 1, 0x0E16 });
    s_aglList.emplace_back("thousandcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0482 });
    s_aglList.emplace_back("thousandsseparatorarabic"_n, AglMapping{ (AglType)1, 1, 0x066C });
    s_aglList.emplace_back("thousandsseparatorpersian"_n, AglMapping{ (AglType)1, 1, 0x066C });
    s_aglList.emplace_back("three"_n, AglMapping{ (AglType)11, 1, 0x0033 });
    s_aglList.emplace_back("threearabic"_n, AglMapping{ (AglType)1, 1, 0x0663 });
    s_aglList.emplace_back("threebengali"_n, AglMapping{ (AglType)1, 1, 0x09E9 });
    s_aglList.emplace_back("threecircle"_n, AglMapping{ (AglType)1, 1, 0x2462 });
    s_aglList.emplace_back("threecircleinversesansserif"_n, AglMapping{ (AglType)1, 1, 0x278C });
    s_aglList.emplace_back("threedeva"_n, AglMapping{ (AglType)1, 1, 0x0969 });
    s_aglList.emplace_back("threeeighths"_n, AglMapping{ (AglType)3, 1, 0x215C });
    s_aglList.emplace_back("threegujarati"_n, AglMapping{ (AglType)1, 1, 0x0AE9 });
    s_aglList.emplace_back("threegurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A69 });
    s_aglList.emplace_back("threehackarabic"_n, AglMapping{ (AglType)1, 1, 0x0663 });
    s_aglList.emplace_back("threehangzhou"_n, AglMapping{ (AglType)1, 1, 0x3023 });
    s_aglList.emplace_back("threeideographicparen"_n, AglMapping{ (AglType)1, 1, 0x3222 });
    s_aglList.emplace_back("threeinferior"_n, AglMapping{ (AglType)1, 1, 0x2083 });
    s_aglList.emplace_back("threemonospace"_n, AglMapping{ (AglType)1, 1, 0xFF13 });
    s_aglList.emplace_back("threenumeratorbengali"_n, AglMapping{ (AglType)1, 1, 0x09F6 });
    s_aglList.emplace_back("threeoldstyle"_n, AglMapping{ (AglType)1, 1, 0xF733 });
    s_aglList.emplace_back("threeparen"_n, AglMapping{ (AglType)1, 1, 0x2476 });
    s_aglList.emplace_back("threeperiod"_n, AglMapping{ (AglType)1, 1, 0x248A });
    s_aglList.emplace_back("threepersian"_n, AglMapping{ (AglType)1, 1, 0x06F3 });
    s_aglList.emplace_back("threequarters"_n, AglMapping{ (AglType)11, 1, 0x00BE });
    s_aglList.emplace_back("threequartersemdash"_n, AglMapping{ (AglType)1, 1, 0xF6DE });
    s_aglList.emplace_back("threeroman"_n, AglMapping{ (AglType)1, 1, 0x2172 });
    s_aglList.emplace_back("threesuperior"_n, AglMapping{ (AglType)9, 1, 0x00B3 });
    s_aglList.emplace_back("threethai"_n, AglMapping{ (AglType)1, 1, 0x0E53 });
    s_aglList.emplace_back("thzsquare"_n, AglMapping{ (AglType)1, 1, 0x3394 });
    s_aglList.emplace_back("tihiragana"_n, AglMapping{ (AglType)1, 1, 0x3061 });
    s_aglList.emplace_back("tikatakana"_n, AglMapping{ (AglType)1, 1, 0x30C1 });
    s_aglList.emplace_back("tikatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF81 });
    s_aglList.emplace_back("tikeutacirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3270 });
    s_aglList.emplace_back("tikeutaparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3210 });
    s_aglList.emplace_back("tikeutcirclekorean"_n, AglMapping{ (AglType)1, 1, 0x3262 });
    s_aglList.emplace_back("tikeutkorean"_n, AglMapping{ (AglType)1, 1, 0x3137 });
    s_aglList.emplace_back("tikeutparenkorean"_n, AglMapping{ (AglType)1, 1, 0x3202 });
    s_aglList.emplace_back("tilde"_n, AglMapping{ (AglType)11, 1, 0x02DC });
    s_aglList.emplace_back("tildebelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0330 });
    s_aglList.emplace_back("tildecmb"_n, AglMapping{ (AglType)1, 1, 0x0303 });
    s_aglList.emplace_back("tildecomb"_n, AglMapping{ (AglType)3, 1, 0x0303 });
    s_aglList.emplace_back("tildedoublecmb"_n, AglMapping{ (AglType)1, 1, 0x0360 });
    s_aglList.emplace_back("tildeoperator"_n, AglMapping{ (AglType)1, 1, 0x223C });
    s_aglList.emplace_back("tildeoverlaycmb"_n, AglMapping{ (AglType)1, 1, 0x0334 });
    s_aglList.emplace_back("tildeverticalcmb"_n, AglMapping{ (AglType)1, 1, 0x033E });
    s_aglList.emplace_back("timescircle"_n, AglMapping{ (AglType)1, 1, 0x2297 });
    s_aglList.emplace_back("tipehahebrew"_n, AglMapping{ (AglType)1, 1, 0x0596 });
    s_aglList.emplace_back("tipehalefthebrew"_n, AglMapping{ (AglType)1, 1, 0x0596 });
    s_aglList.emplace_back("tippigurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A70 });
    s_aglList.emplace_back("titlocyrilliccmb"_n, AglMapping{ (AglType)1, 1, 0x0483 });
    s_aglList.emplace_back("tiwnarmenian"_n, AglMapping{ (AglType)1, 1, 0x057F });
    s_aglList.emplace_back("tlinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E6F });
    s_aglList.emplace_back("tmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF54 });
    s_aglList.emplace_back("toarmenian"_n, AglMapping{ (AglType)1, 1, 0x0569 });
    s_aglList.emplace_back("tohiragana"_n, AglMapping{ (AglType)1, 1, 0x3068 });
    s_aglList.emplace_back("tokatakana"_n, AglMapping{ (AglType)1, 1, 0x30C8 });
    s_aglList.emplace_back("tokatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF84 });
    s_aglList.emplace_back("tonebarextrahighmod"_n, AglMapping{ (AglType)1, 1, 0x02E5 });
    s_aglList.emplace_back("tonebarextralowmod"_n, AglMapping{ (AglType)1, 1, 0x02E9 });
    s_aglList.emplace_back("tonebarhighmod"_n, AglMapping{ (AglType)1, 1, 0x02E6 });
    s_aglList.emplace_back("tonebarlowmod"_n, AglMapping{ (AglType)1, 1, 0x02E8 });
    s_aglList.emplace_back("tonebarmidmod"_n, AglMapping{ (AglType)1, 1, 0x02E7 });
    s_aglList.emplace_back("tonefive"_n, AglMapping{ (AglType)1, 1, 0x01BD });
    s_aglList.emplace_back("tonesix"_n, AglMapping{ (AglType)1, 1, 0x0185 });
    s_aglList.emplace_back("tonetwo"_n, AglMapping{ (AglType)1, 1, 0x01A8 });
    s_aglList.emplace_back("tonos"_n, AglMapping{ (AglType)3, 1, 0x0384 });
    s_aglList.emplace_back("tonsquare"_n, AglMapping{ (AglType)1, 1, 0x3327 });
    s_aglList.emplace_back("topatakthai"_n, AglMapping{ (AglType)1, 1, 0x0E0F });
    s_aglList.emplace_back("tortoiseshellbracketleft"_n, AglMapping{ (AglType)1, 1, 0x3014 });
    s_aglList.emplace_back("tortoiseshellbracketleftsmall"_n, AglMapping{ (AglType)1, 1, 0xFE5D });
    s_aglList.emplace_back("tortoiseshellbracketleftvertical"_n, AglMapping{ (AglType)1, 1, 0xFE39 });
    s_aglList.emplace_back("tortoiseshellbracketright"_n, AglMapping{ (AglType)1, 1, 0x3015 });
    s_aglList.emplace_back("tortoiseshellbracketrightsmall"_n, AglMapping{ (AglType)1, 1, 0xFE5E });
    s_aglList.emplace_back("tortoiseshellbracketrightvertical"_n, AglMapping{ (AglType)1, 1, 0xFE3A });
    s_aglList.emplace_back("totaothai"_n, AglMapping{ (AglType)1, 1, 0x0E15 });
    s_aglList.emplace_back("tpalatalhook"_n, AglMapping{ (AglType)1, 1, 0x01AB });
    s_aglList.emplace_back("tparen"_n, AglMapping{ (AglType)1, 1, 0x24AF });
    s_aglList.emplace_back("trademark"_n, AglMapping{ (AglType)11, 1, 0x2122 });
    s_aglList.emplace_back("trademarksans"_n, AglMapping{ (AglType)1, 1, 0xF8EA });
    s_aglList.emplace_back("trademarkserif"_n, AglMapping{ (AglType)1, 1, 0xF6DB });
    s_aglList.emplace_back("tretroflexhook"_n, AglMapping{ (AglType)1, 1, 0x0288 });
    s_aglList.emplace_back("triagdn"_n, AglMapping{ (AglType)3, 1, 0x25BC });
    s_aglList.emplace_back("triaglf"_n, AglMapping{ (AglType)3, 1, 0x25C4 });
    s_aglList.emplace_back("triagrt"_n, AglMapping{ (AglType)3, 1, 0x25BA });
    s_aglList.emplace_back("triagup"_n, AglMapping{ (AglType)3, 1, 0x25B2 });
    s_aglList.emplace_back("ts"_n, AglMapping{ (AglType)1, 1, 0x02A6 });
    s_aglList.emplace_back("tsadi"_n, AglMapping{ (AglType)1, 1, 0x05E6 });
    s_aglList.emplace_back("tsadidagesh"_n, AglMapping{ (AglType)1, 1, 0xFB46 });
    s_aglList.emplace_back("tsadidageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB46 });
    s_aglList.emplace_back("tsadihebrew"_n, AglMapping{ (AglType)1, 1, 0x05E6 });
    s_aglList.emplace_back("tsecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0446 });
    s_aglList.emplace_back("tsere"_n, AglMapping{ (AglType)1, 1, 0x05B5 });
    s_aglList.emplace_back("tsere12"_n, AglMapping{ (AglType)1, 1, 0x05B5 });
    s_aglList.emplace_back("tsere1e"_n, AglMapping{ (AglType)1, 1, 0x05B5 });
    s_aglList.emplace_back("tsere2b"_n, AglMapping{ (AglType)1, 1, 0x05B5 });
    s_aglList.emplace_back("tserehebrew"_n, AglMapping{ (AglType)1, 1, 0x05B5 });
    s_aglList.emplace_back("tserenarrowhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B5 });
    s_aglList.emplace_back("tserequarterhebrew"_n, AglMapping{ (AglType)1, 1, 0x05B5 });
    s_aglList.emplace_back("tserewidehebrew"_n, AglMapping{ (AglType)1, 1, 0x05B5 });
    s_aglList.emplace_back("tshecyrillic"_n, AglMapping{ (AglType)1, 1, 0x045B });
    s_aglList.emplace_back("tsuperior"_n, AglMapping{ (AglType)1, 1, 0xF6F3 });
    s_aglList.emplace_back("ttabengali"_n, AglMapping{ (AglType)1, 1, 0x099F });
    s_aglList.emplace_back("ttadeva"_n, AglMapping{ (AglType)1, 1, 0x091F });
    s_aglList.emplace_back("ttagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A9F });
    s_aglList.emplace_back("ttagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A1F });
    s_aglList.emplace_back("tteharabic"_n, AglMapping{ (AglType)1, 1, 0x0679 });
    s_aglList.emplace_back("ttehfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFB67 });
    s_aglList.emplace_back("ttehinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFB68 });
    s_aglList.emplace_back("ttehmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFB69 });
    s_aglList.emplace_back("tthabengali"_n, AglMapping{ (AglType)1, 1, 0x09A0 });
    s_aglList.emplace_back("tthadeva"_n, AglMapping{ (AglType)1, 1, 0x0920 });
    s_aglList.emplace_back("tthagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AA0 });
    s_aglList.emplace_back("tthagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A20 });
    s_aglList.emplace_back("tturned"_n, AglMapping{ (AglType)1, 1, 0x0287 });
    s_aglList.emplace_back("tuhiragana"_n, AglMapping{ (AglType)1, 1, 0x3064 });
    s_aglList.emplace_back("tukatakana"_n, AglMapping{ (AglType)1, 1, 0x30C4 });
    s_aglList.emplace_back("tukatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF82 });
    s_aglList.emplace_back("tusmallhiragana"_n, AglMapping{ (AglType)1, 1, 0x3063 });
    s_aglList.emplace_back("tusmallkatakana"_n, AglMapping{ (AglType)1, 1, 0x30C3 });
    s_aglList.emplace_back("tusmallkatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF6F });
    s_aglList.emplace_back("twelvecircle"_n, AglMapping{ (AglType)1, 1, 0x246B });
    s_aglList.emplace_back("twelveparen"_n, AglMapping{ (AglType)1, 1, 0x247F });
    s_aglList.emplace_back("twelveperiod"_n, AglMapping{ (AglType)1, 1, 0x2493 });
    s_aglList.emplace_back("twelveroman"_n, AglMapping{ (AglType)1, 1, 0x217B });
    s_aglList.emplace_back("twentycircle"_n, AglMapping{ (AglType)1, 1, 0x2473 });
    s_aglList.emplace_back("twentyhangzhou"_n, AglMapping{ (AglType)1, 1, 0x5344 });
    s_aglList.emplace_back("twentyparen"_n, AglMapping{ (AglType)1, 1, 0x2487 });
    s_aglList.emplace_back("twentyperiod"_n, AglMapping{ (AglType)1, 1, 0x249B });
    s_aglList.emplace_back("two"_n, AglMapping{ (AglType)11, 1, 0x0032 });
    s_aglList.emplace_back("twoarabic"_n, AglMapping{ (AglType)1, 1, 0x0662 });
    s_aglList.emplace_back("twobengali"_n, AglMapping{ (AglType)1, 1, 0x09E8 });
    s_aglList.emplace_back("twocircle"_n, AglMapping{ (AglType)1, 1, 0x2461 });
    s_aglList.emplace_back("twocircleinversesansserif"_n, AglMapping{ (AglType)1, 1, 0x278B });
    s_aglList.emplace_back("twodeva"_n, AglMapping{ (AglType)1, 1, 0x0968 });
    s_aglList.emplace_back("twodotenleader"_n, AglMapping{ (AglType)3, 1, 0x2025 });
    s_aglList.emplace_back("twodotleader"_n, AglMapping{ (AglType)1, 1, 0x2025 });
    s_aglList.emplace_back("twodotleadervertical"_n, AglMapping{ (AglType)1, 1, 0xFE30 });
    s_aglList.emplace_back("twogujarati"_n, AglMapping{ (AglType)1, 1, 0x0AE8 });
    s_aglList.emplace_back("twogurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A68 });
    s_aglList.emplace_back("twohackarabic"_n, AglMapping{ (AglType)1, 1, 0x0662 });
    s_aglList.emplace_back("twohangzhou"_n, AglMapping{ (AglType)1, 1, 0x3022 });
    s_aglList.emplace_back("twoideographicparen"_n, AglMapping{ (AglType)1, 1, 0x3221 });
    s_aglList.emplace_back("twoinferior"_n, AglMapping{ (AglType)1, 1, 0x2082 });
    s_aglList.emplace_back("twomonospace"_n, AglMapping{ (AglType)1, 1, 0xFF12 });
    s_aglList.emplace_back("twonumeratorbengali"_n, AglMapping{ (AglType)1, 1, 0x09F5 });
    s_aglList.emplace_back("twooldstyle"_n, AglMapping{ (AglType)1, 1, 0xF732 });
    s_aglList.emplace_back("twoparen"_n, AglMapping{ (AglType)1, 1, 0x2475 });
    s_aglList.emplace_back("twoperiod"_n, AglMapping{ (AglType)1, 1, 0x2489 });
    s_aglList.emplace_back("twopersian"_n, AglMapping{ (AglType)1, 1, 0x06F2 });
    s_aglList.emplace_back("tworoman"_n, AglMapping{ (AglType)1, 1, 0x2171 });
    s_aglList.emplace_back("twostroke"_n, AglMapping{ (AglType)1, 1, 0x01BB });
    s_aglList.emplace_back("twosuperior"_n, AglMapping{ (AglType)9, 1, 0x00B2 });
    s_aglList.emplace_back("twothai"_n, AglMapping{ (AglType)1, 1, 0x0E52 });
    s_aglList.emplace_back("twothirds"_n, AglMapping{ (AglType)3, 1, 0x2154 });
    s_aglList.emplace_back("u"_n, AglMapping{ (AglType)11, 1, 0x0075 });
    s_aglList.emplace_back("uacute"_n, AglMapping{ (AglType)11, 1, 0x00FA });
    s_aglList.emplace_back("ubar"_n, AglMapping{ (AglType)1, 1, 0x0289 });
    s_aglList.emplace_back("ubengali"_n, AglMapping{ (AglType)1, 1, 0x0989 });
    s_aglList.emplace_back("ubopomofo"_n, AglMapping{ (AglType)1, 1, 0x3128 });
    s_aglList.emplace_back("ubreve"_n, AglMapping{ (AglType)3, 1, 0x016D });
    s_aglList.emplace_back("ucaron"_n, AglMapping{ (AglType)1, 1, 0x01D4 });
    s_aglList.emplace_back("ucircle"_n, AglMapping{ (AglType)1, 1, 0x24E4 });
    s_aglList.emplace_back("ucircumflex"_n, AglMapping{ (AglType)11, 1, 0x00FB });
    s_aglList.emplace_back("ucircumflexbelow"_n, AglMapping{ (AglType)1, 1, 0x1E77 });
    s_aglList.emplace_back("ucyrillic"_n, AglMapping{ (AglType)1, 1, 0x0443 });
    s_aglList.emplace_back("udattadeva"_n, AglMapping{ (AglType)1, 1, 0x0951 });
    s_aglList.emplace_back("udblacute"_n, AglMapping{ (AglType)1, 1, 0x0171 });
    s_aglList.emplace_back("udblgrave"_n, AglMapping{ (AglType)1, 1, 0x0215 });
    s_aglList.emplace_back("udeva"_n, AglMapping{ (AglType)1, 1, 0x0909 });
    s_aglList.emplace_back("udieresis"_n, AglMapping{ (AglType)11, 1, 0x00FC });
    s_aglList.emplace_back("udieresisacute"_n, AglMapping{ (AglType)1, 1, 0x01D8 });
    s_aglList.emplace_back("udieresisbelow"_n, AglMapping{ (AglType)1, 1, 0x1E73 });
    s_aglList.emplace_back("udieresiscaron"_n, AglMapping{ (AglType)1, 1, 0x01DA });
    s_aglList.emplace_back("udieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04F1 });
    s_aglList.emplace_back("udieresisgrave"_n, AglMapping{ (AglType)1, 1, 0x01DC });
    s_aglList.emplace_back("udieresismacron"_n, AglMapping{ (AglType)1, 1, 0x01D6 });
    s_aglList.emplace_back("udotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EE5 });
    s_aglList.emplace_back("ugrave"_n, AglMapping{ (AglType)11, 1, 0x00F9 });
    s_aglList.emplace_back("ugujarati"_n, AglMapping{ (AglType)1, 1, 0x0A89 });
    s_aglList.emplace_back("ugurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A09 });
    s_aglList.emplace_back("uhiragana"_n, AglMapping{ (AglType)1, 1, 0x3046 });
    s_aglList.emplace_back("uhookabove"_n, AglMapping{ (AglType)1, 1, 0x1EE7 });
    s_aglList.emplace_back("uhorn"_n, AglMapping{ (AglType)3, 1, 0x01B0 });
    s_aglList.emplace_back("uhornacute"_n, AglMapping{ (AglType)1, 1, 0x1EE9 });
    s_aglList.emplace_back("uhorndotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EF1 });
    s_aglList.emplace_back("uhorngrave"_n, AglMapping{ (AglType)1, 1, 0x1EEB });
    s_aglList.emplace_back("uhornhookabove"_n, AglMapping{ (AglType)1, 1, 0x1EED });
    s_aglList.emplace_back("uhorntilde"_n, AglMapping{ (AglType)1, 1, 0x1EEF });
    s_aglList.emplace_back("uhungarumlaut"_n, AglMapping{ (AglType)3, 1, 0x0171 });
    s_aglList.emplace_back("uhungarumlautcyrillic"_n, AglMapping{ (AglType)1, 1, 0x04F3 });
    s_aglList.emplace_back("uinvertedbreve"_n, AglMapping{ (AglType)1, 1, 0x0217 });
    s_aglList.emplace_back("ukatakana"_n, AglMapping{ (AglType)1, 1, 0x30A6 });
    s_aglList.emplace_back("ukatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF73 });
    s_aglList.emplace_back("ukcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0479 });
    s_aglList.emplace_back("ukorean"_n, AglMapping{ (AglType)1, 1, 0x315C });
    s_aglList.emplace_back("umacron"_n, AglMapping{ (AglType)3, 1, 0x016B });
    s_aglList.emplace_back("umacroncyrillic"_n, AglMapping{ (AglType)1, 1, 0x04EF });
    s_aglList.emplace_back("umacrondieresis"_n, AglMapping{ (AglType)1, 1, 0x1E7B });
    s_aglList.emplace_back("umatragurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A41 });
    s_aglList.emplace_back("umonospace"_n, AglMapping{ (AglType)1, 1, 0xFF55 });
    s_aglList.emplace_back("underscore"_n, AglMapping{ (AglType)11, 1, 0x005F });
    s_aglList.emplace_back("underscoredbl"_n, AglMapping{ (AglType)3, 1, 0x2017 });
    s_aglList.emplace_back("underscoremonospace"_n, AglMapping{ (AglType)1, 1, 0xFF3F });
    s_aglList.emplace_back("underscorevertical"_n, AglMapping{ (AglType)1, 1, 0xFE33 });
    s_aglList.emplace_back("underscorewavy"_n, AglMapping{ (AglType)1, 1, 0xFE4F });
    s_aglList.emplace_back("union"_n, AglMapping{ (AglType)3, 1, 0x222A });
    s_aglList.emplace_back("universal"_n, AglMapping{ (AglType)3, 1, 0x2200 });
    s_aglList.emplace_back("uogonek"_n, AglMapping{ (AglType)3, 1, 0x0173 });
    s_aglList.emplace_back("uparen"_n, AglMapping{ (AglType)1, 1, 0x24B0 });
    s_aglList.emplace_back("upblock"_n, AglMapping{ (AglType)3, 1, 0x2580 });
    s_aglList.emplace_back("upperdothebrew"_n, AglMapping{ (AglType)1, 1, 0x05C4 });
    s_aglList.emplace_back("upsilon"_n, AglMapping{ (AglType)3, 1, 0x03C5 });
    s_aglList.emplace_back("upsilondieresis"_n, AglMapping{ (AglType)3, 1, 0x03CB });
    s_aglList.emplace_back("upsilondieresistonos"_n, AglMapping{ (AglType)3, 1, 0x03B0 });
    s_aglList.emplace_back("upsilonlatin"_n, AglMapping{ (AglType)1, 1, 0x028A });
    s_aglList.emplace_back("upsilontonos"_n, AglMapping{ (AglType)3, 1, 0x03CD });
    s_aglList.emplace_back("uptackbelowcmb"_n, AglMapping{ (AglType)1, 1, 0x031D });
    s_aglList.emplace_back("uptackmod"_n, AglMapping{ (AglType)1, 1, 0x02D4 });
    s_aglList.emplace_back("uragurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A73 });
    s_aglList.emplace_back("uring"_n, AglMapping{ (AglType)3, 1, 0x016F });
    s_aglList.emplace_back("ushortcyrillic"_n, AglMapping{ (AglType)1, 1, 0x045E });
    s_aglList.emplace_back("usmallhiragana"_n, AglMapping{ (AglType)1, 1, 0x3045 });
    s_aglList.emplace_back("usmallkatakana"_n, AglMapping{ (AglType)1, 1, 0x30A5 });
    s_aglList.emplace_back("usmallkatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF69 });
    s_aglList.emplace_back("ustraightcyrillic"_n, AglMapping{ (AglType)1, 1, 0x04AF });
    s_aglList.emplace_back("ustraightstrokecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04B1 });
    s_aglList.emplace_back("utilde"_n, AglMapping{ (AglType)3, 1, 0x0169 });
    s_aglList.emplace_back("utildeacute"_n, AglMapping{ (AglType)1, 1, 0x1E79 });
    s_aglList.emplace_back("utildebelow"_n, AglMapping{ (AglType)1, 1, 0x1E75 });
    s_aglList.emplace_back("uubengali"_n, AglMapping{ (AglType)1, 1, 0x098A });
    s_aglList.emplace_back("uudeva"_n, AglMapping{ (AglType)1, 1, 0x090A });
    s_aglList.emplace_back("uugujarati"_n, AglMapping{ (AglType)1, 1, 0x0A8A });
    s_aglList.emplace_back("uugurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A0A });
    s_aglList.emplace_back("uumatragurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A42 });
    s_aglList.emplace_back("uuvowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09C2 });
    s_aglList.emplace_back("uuvowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0942 });
    s_aglList.emplace_back("uuvowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0AC2 });
    s_aglList.emplace_back("uvowelsignbengali"_n, AglMapping{ (AglType)1, 1, 0x09C1 });
    s_aglList.emplace_back("uvowelsigndeva"_n, AglMapping{ (AglType)1, 1, 0x0941 });
    s_aglList.emplace_back("uvowelsigngujarati"_n, AglMapping{ (AglType)1, 1, 0x0AC1 });
    s_aglList.emplace_back("v"_n, AglMapping{ (AglType)11, 1, 0x0076 });
    s_aglList.emplace_back("vadeva"_n, AglMapping{ (AglType)1, 1, 0x0935 });
    s_aglList.emplace_back("vagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AB5 });
    s_aglList.emplace_back("vagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A35 });
    s_aglList.emplace_back("vakatakana"_n, AglMapping{ (AglType)1, 1, 0x30F7 });
    s_aglList.emplace_back("vav"_n, AglMapping{ (AglType)1, 1, 0x05D5 });
    s_aglList.emplace_back("vavdagesh"_n, AglMapping{ (AglType)1, 1, 0xFB35 });
    s_aglList.emplace_back("vavdagesh65"_n, AglMapping{ (AglType)1, 1, 0xFB35 });
    s_aglList.emplace_back("vavdageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB35 });
    s_aglList.emplace_back("vavhebrew"_n, AglMapping{ (AglType)1, 1, 0x05D5 });
    s_aglList.emplace_back("vavholam"_n, AglMapping{ (AglType)1, 1, 0xFB4B });
    s_aglList.emplace_back("vavholamhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB4B });
    s_aglList.emplace_back("vavvavhebrew"_n, AglMapping{ (AglType)1, 1, 0x05F0 });
    s_aglList.emplace_back("vavyodhebrew"_n, AglMapping{ (AglType)1, 1, 0x05F1 });
    s_aglList.emplace_back("vcircle"_n, AglMapping{ (AglType)1, 1, 0x24E5 });
    s_aglList.emplace_back("vdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E7F });
    s_aglList.emplace_back("vecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0432 });
    s_aglList.emplace_back("veharabic"_n, AglMapping{ (AglType)1, 1, 0x06A4 });
    s_aglList.emplace_back("vehfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFB6B });
    s_aglList.emplace_back("vehinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFB6C });
    s_aglList.emplace_back("vehmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFB6D });
    s_aglList.emplace_back("vekatakana"_n, AglMapping{ (AglType)1, 1, 0x30F9 });
    s_aglList.emplace_back("venus"_n, AglMapping{ (AglType)1, 1, 0x2640 });
    s_aglList.emplace_back("verticalbar"_n, AglMapping{ (AglType)1, 1, 0x007C });
    s_aglList.emplace_back("verticallineabovecmb"_n, AglMapping{ (AglType)1, 1, 0x030D });
    s_aglList.emplace_back("verticallinebelowcmb"_n, AglMapping{ (AglType)1, 1, 0x0329 });
    s_aglList.emplace_back("verticallinelowmod"_n, AglMapping{ (AglType)1, 1, 0x02CC });
    s_aglList.emplace_back("verticallinemod"_n, AglMapping{ (AglType)1, 1, 0x02C8 });
    s_aglList.emplace_back("vewarmenian"_n, AglMapping{ (AglType)1, 1, 0x057E });
    s_aglList.emplace_back("vhook"_n, AglMapping{ (AglType)1, 1, 0x028B });
    s_aglList.emplace_back("vikatakana"_n, AglMapping{ (AglType)1, 1, 0x30F8 });
    s_aglList.emplace_back("viramabengali"_n, AglMapping{ (AglType)1, 1, 0x09CD });
    s_aglList.emplace_back("viramadeva"_n, AglMapping{ (AglType)1, 1, 0x094D });
    s_aglList.emplace_back("viramagujarati"_n, AglMapping{ (AglType)1, 1, 0x0ACD });
    s_aglList.emplace_back("visargabengali"_n, AglMapping{ (AglType)1, 1, 0x0983 });
    s_aglList.emplace_back("visargadeva"_n, AglMapping{ (AglType)1, 1, 0x0903 });
    s_aglList.emplace_back("visargagujarati"_n, AglMapping{ (AglType)1, 1, 0x0A83 });
    s_aglList.emplace_back("vmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF56 });
    s_aglList.emplace_back("voarmenian"_n, AglMapping{ (AglType)1, 1, 0x0578 });
    s_aglList.emplace_back("voicediterationhiragana"_n, AglMapping{ (AglType)1, 1, 0x309E });
    s_aglList.emplace_back("voicediterationkatakana"_n, AglMapping{ (AglType)1, 1, 0x30FE });
    s_aglList.emplace_back("voicedmarkkana"_n, AglMapping{ (AglType)1, 1, 0x309B });
    s_aglList.emplace_back("voicedmarkkanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF9E });
    s_aglList.emplace_back("vokatakana"_n, AglMapping{ (AglType)1, 1, 0x30FA });
    s_aglList.emplace_back("vparen"_n, AglMapping{ (AglType)1, 1, 0x24B1 });
    s_aglList.emplace_back("vtilde"_n, AglMapping{ (AglType)1, 1, 0x1E7D });
    s_aglList.emplace_back("vturned"_n, AglMapping{ (AglType)1, 1, 0x028C });
    s_aglList.emplace_back("vuhiragana"_n, AglMapping{ (AglType)1, 1, 0x3094 });
    s_aglList.emplace_back("vukatakana"_n, AglMapping{ (AglType)1, 1, 0x30F4 });
    s_aglList.emplace_back("w"_n, AglMapping{ (AglType)11, 1, 0x0077 });
    s_aglList.emplace_back("wacute"_n, AglMapping{ (AglType)3, 1, 0x1E83 });
    s_aglList.emplace_back("waekorean"_n, AglMapping{ (AglType)1, 1, 0x3159 });
    s_aglList.emplace_back("wahiragana"_n, AglMapping{ (AglType)1, 1, 0x308F });
    s_aglList.emplace_back("wakatakana"_n, AglMapping{ (AglType)1, 1, 0x30EF });
    s_aglList.emplace_back("wakatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF9C });
    s_aglList.emplace_back("wakorean"_n, AglMapping{ (AglType)1, 1, 0x3158 });
    s_aglList.emplace_back("wasmallhiragana"_n, AglMapping{ (AglType)1, 1, 0x308E });
    s_aglList.emplace_back("wasmallkatakana"_n, AglMapping{ (AglType)1, 1, 0x30EE });
    s_aglList.emplace_back("wattosquare"_n, AglMapping{ (AglType)1, 1, 0x3357 });
    s_aglList.emplace_back("wavedash"_n, AglMapping{ (AglType)1, 1, 0x301C });
    s_aglList.emplace_back("wavyunderscorevertical"_n, AglMapping{ (AglType)1, 1, 0xFE34 });
    s_aglList.emplace_back("wawarabic"_n, AglMapping{ (AglType)1, 1, 0x0648 });
    s_aglList.emplace_back("wawfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEEE });
    s_aglList.emplace_back("wawhamzaabovearabic"_n, AglMapping{ (AglType)1, 1, 0x0624 });
    s_aglList.emplace_back("wawhamzaabovefinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFE86 });
    s_aglList.emplace_back("wbsquare"_n, AglMapping{ (AglType)1, 1, 0x33DD });
    s_aglList.emplace_back("wcircle"_n, AglMapping{ (AglType)1, 1, 0x24E6 });
    s_aglList.emplace_back("wcircumflex"_n, AglMapping{ (AglType)3, 1, 0x0175 });
    s_aglList.emplace_back("wdieresis"_n, AglMapping{ (AglType)3, 1, 0x1E85 });
    s_aglList.emplace_back("wdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E87 });
    s_aglList.emplace_back("wdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E89 });
    s_aglList.emplace_back("wehiragana"_n, AglMapping{ (AglType)1, 1, 0x3091 });
    s_aglList.emplace_back("weierstrass"_n, AglMapping{ (AglType)3, 1, 0x2118 });
    s_aglList.emplace_back("wekatakana"_n, AglMapping{ (AglType)1, 1, 0x30F1 });
    s_aglList.emplace_back("wekorean"_n, AglMapping{ (AglType)1, 1, 0x315E });
    s_aglList.emplace_back("weokorean"_n, AglMapping{ (AglType)1, 1, 0x315D });
    s_aglList.emplace_back("wgrave"_n, AglMapping{ (AglType)3, 1, 0x1E81 });
    s_aglList.emplace_back("whitebullet"_n, AglMapping{ (AglType)1, 1, 0x25E6 });
    s_aglList.emplace_back("whitecircle"_n, AglMapping{ (AglType)1, 1, 0x25CB });
    s_aglList.emplace_back("whitecircleinverse"_n, AglMapping{ (AglType)1, 1, 0x25D9 });
    s_aglList.emplace_back("whitecornerbracketleft"_n, AglMapping{ (AglType)1, 1, 0x300E });
    s_aglList.emplace_back("whitecornerbracketleftvertical"_n, AglMapping{ (AglType)1, 1, 0xFE43 });
    s_aglList.emplace_back("whitecornerbracketright"_n, AglMapping{ (AglType)1, 1, 0x300F });
    s_aglList.emplace_back("whitecornerbracketrightvertical"_n, AglMapping{ (AglType)1, 1, 0xFE44 });
    s_aglList.emplace_back("whitediamond"_n, AglMapping{ (AglType)1, 1, 0x25C7 });
    s_aglList.emplace_back("whitediamondcontainingblacksmalldiamond"_n, AglMapping{ (AglType)1, 1, 0x25C8 });
    s_aglList.emplace_back("whitedownpointingsmalltriangle"_n, AglMapping{ (AglType)1, 1, 0x25BF });
    s_aglList.emplace_back("whitedownpointingtriangle"_n, AglMapping{ (AglType)1, 1, 0x25BD });
    s_aglList.emplace_back("whiteleftpointingsmalltriangle"_n, AglMapping{ (AglType)1, 1, 0x25C3 });
    s_aglList.emplace_back("whiteleftpointingtriangle"_n, AglMapping{ (AglType)1, 1, 0x25C1 });
    s_aglList.emplace_back("whitelenticularbracketleft"_n, AglMapping{ (AglType)1, 1, 0x3016 });
    s_aglList.emplace_back("whitelenticularbracketright"_n, AglMapping{ (AglType)1, 1, 0x3017 });
    s_aglList.emplace_back("whiterightpointingsmalltriangle"_n, AglMapping{ (AglType)1, 1, 0x25B9 });
    s_aglList.emplace_back("whiterightpointingtriangle"_n, AglMapping{ (AglType)1, 1, 0x25B7 });
    s_aglList.emplace_back("whitesmallsquare"_n, AglMapping{ (AglType)1, 1, 0x25AB });
    s_aglList.emplace_back("whitesmilingface"_n, AglMapping{ (AglType)1, 1, 0x263A });
    s_aglList.emplace_back("whitesquare"_n, AglMapping{ (AglType)1, 1, 0x25A1 });
    s_aglList.emplace_back("whitestar"_n, AglMapping{ (AglType)1, 1, 0x2606 });
    s_aglList.emplace_back("whitetelephone"_n, AglMapping{ (AglType)1, 1, 0x260F });
    s_aglList.emplace_back("whitetortoiseshellbracketleft"_n, AglMapping{ (AglType)1, 1, 0x3018 });
    s_aglList.emplace_back("whitetortoiseshellbracketright"_n, AglMapping{ (AglType)1, 1, 0x3019 });
    s_aglList.emplace_back("whiteuppointingsmalltriangle"_n, AglMapping{ (AglType)1, 1, 0x25B5 });
    s_aglList.emplace_back("whiteuppointingtriangle"_n, AglMapping{ (AglType)1, 1, 0x25B3 });
    s_aglList.emplace_back("wihiragana"_n, AglMapping{ (AglType)1, 1, 0x3090 });
    s_aglList.emplace_back("wikatakana"_n, AglMapping{ (AglType)1, 1, 0x30F0 });
    s_aglList.emplace_back("wikorean"_n, AglMapping{ (AglType)1, 1, 0x315F });
    s_aglList.emplace_back("wmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF57 });
    s_aglList.emplace_back("wohiragana"_n, AglMapping{ (AglType)1, 1, 0x3092 });
    s_aglList.emplace_back("wokatakana"_n, AglMapping{ (AglType)1, 1, 0x30F2 });
    s_aglList.emplace_back("wokatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF66 });
    s_aglList.emplace_back("won"_n, AglMapping{ (AglType)1, 1, 0x20A9 });
    s_aglList.emplace_back("wonmonospace"_n, AglMapping{ (AglType)1, 1, 0xFFE6 });
    s_aglList.emplace_back("wowaenthai"_n, AglMapping{ (AglType)1, 1, 0x0E27 });
    s_aglList.emplace_back("wparen"_n, AglMapping{ (AglType)1, 1, 0x24B2 });
    s_aglList.emplace_back("wring"_n, AglMapping{ (AglType)1, 1, 0x1E98 });
    s_aglList.emplace_back("wsuperior"_n, AglMapping{ (AglType)1, 1, 0x02B7 });
    s_aglList.emplace_back("wturned"_n, AglMapping{ (AglType)1, 1, 0x028D });
    s_aglList.emplace_back("wynn"_n, AglMapping{ (AglType)1, 1, 0x01BF });
    s_aglList.emplace_back("x"_n, AglMapping{ (AglType)11, 1, 0x0078 });
    s_aglList.emplace_back("xabovecmb"_n, AglMapping{ (AglType)1, 1, 0x033D });
    s_aglList.emplace_back("xbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3112 });
    s_aglList.emplace_back("xcircle"_n, AglMapping{ (AglType)1, 1, 0x24E7 });
    s_aglList.emplace_back("xdieresis"_n, AglMapping{ (AglType)1, 1, 0x1E8D });
    s_aglList.emplace_back("xdotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E8B });
    s_aglList.emplace_back("xeharmenian"_n, AglMapping{ (AglType)1, 1, 0x056D });
    s_aglList.emplace_back("xi"_n, AglMapping{ (AglType)3, 1, 0x03BE });
    s_aglList.emplace_back("xmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF58 });
    s_aglList.emplace_back("xparen"_n, AglMapping{ (AglType)1, 1, 0x24B3 });
    s_aglList.emplace_back("xsuperior"_n, AglMapping{ (AglType)1, 1, 0x02E3 });
    s_aglList.emplace_back("y"_n, AglMapping{ (AglType)11, 1, 0x0079 });
    s_aglList.emplace_back("yaadosquare"_n, AglMapping{ (AglType)1, 1, 0x334E });
    s_aglList.emplace_back("yabengali"_n, AglMapping{ (AglType)1, 1, 0x09AF });
    s_aglList.emplace_back("yacute"_n, AglMapping{ (AglType)11, 1, 0x00FD });
    s_aglList.emplace_back("yadeva"_n, AglMapping{ (AglType)1, 1, 0x092F });
    s_aglList.emplace_back("yaekorean"_n, AglMapping{ (AglType)1, 1, 0x3152 });
    s_aglList.emplace_back("yagujarati"_n, AglMapping{ (AglType)1, 1, 0x0AAF });
    s_aglList.emplace_back("yagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A2F });
    s_aglList.emplace_back("yahiragana"_n, AglMapping{ (AglType)1, 1, 0x3084 });
    s_aglList.emplace_back("yakatakana"_n, AglMapping{ (AglType)1, 1, 0x30E4 });
    s_aglList.emplace_back("yakatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF94 });
    s_aglList.emplace_back("yakorean"_n, AglMapping{ (AglType)1, 1, 0x3151 });
    s_aglList.emplace_back("yamakkanthai"_n, AglMapping{ (AglType)1, 1, 0x0E4E });
    s_aglList.emplace_back("yasmallhiragana"_n, AglMapping{ (AglType)1, 1, 0x3083 });
    s_aglList.emplace_back("yasmallkatakana"_n, AglMapping{ (AglType)1, 1, 0x30E3 });
    s_aglList.emplace_back("yasmallkatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF6C });
    s_aglList.emplace_back("yatcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0463 });
    s_aglList.emplace_back("ycircle"_n, AglMapping{ (AglType)1, 1, 0x24E8 });
    s_aglList.emplace_back("ycircumflex"_n, AglMapping{ (AglType)3, 1, 0x0177 });
    s_aglList.emplace_back("ydieresis"_n, AglMapping{ (AglType)11, 1, 0x00FF });
    s_aglList.emplace_back("ydotaccent"_n, AglMapping{ (AglType)1, 1, 0x1E8F });
    s_aglList.emplace_back("ydotbelow"_n, AglMapping{ (AglType)1, 1, 0x1EF5 });
    s_aglList.emplace_back("yeharabic"_n, AglMapping{ (AglType)1, 1, 0x064A });
    s_aglList.emplace_back("yehbarreearabic"_n, AglMapping{ (AglType)1, 1, 0x06D2 });
    s_aglList.emplace_back("yehbarreefinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFBAF });
    s_aglList.emplace_back("yehfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEF2 });
    s_aglList.emplace_back("yehhamzaabovearabic"_n, AglMapping{ (AglType)1, 1, 0x0626 });
    s_aglList.emplace_back("yehhamzaabovefinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFE8A });
    s_aglList.emplace_back("yehhamzaaboveinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFE8B });
    s_aglList.emplace_back("yehhamzaabovemedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFE8C });
    s_aglList.emplace_back("yehinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEF3 });
    s_aglList.emplace_back("yehmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEF4 });
    s_aglList.emplace_back("yehmeeminitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFCDD });
    s_aglList.emplace_back("yehmeemisolatedarabic"_n, AglMapping{ (AglType)1, 1, 0xFC58 });
    s_aglList.emplace_back("yehnoonfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFC94 });
    s_aglList.emplace_back("yehthreedotsbelowarabic"_n, AglMapping{ (AglType)1, 1, 0x06D1 });
    s_aglList.emplace_back("yekorean"_n, AglMapping{ (AglType)1, 1, 0x3156 });
    s_aglList.emplace_back("yen"_n, AglMapping{ (AglType)11, 1, 0x00A5 });
    s_aglList.emplace_back("yenmonospace"_n, AglMapping{ (AglType)1, 1, 0xFFE5 });
    s_aglList.emplace_back("yeokorean"_n, AglMapping{ (AglType)1, 1, 0x3155 });
    s_aglList.emplace_back("yeorinhieuhkorean"_n, AglMapping{ (AglType)1, 1, 0x3186 });
    s_aglList.emplace_back("yerahbenyomohebrew"_n, AglMapping{ (AglType)1, 1, 0x05AA });
    s_aglList.emplace_back("yerahbenyomolefthebrew"_n, AglMapping{ (AglType)1, 1, 0x05AA });
    s_aglList.emplace_back("yericyrillic"_n, AglMapping{ (AglType)1, 1, 0x044B });
    s_aglList.emplace_back("yerudieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04F9 });
    s_aglList.emplace_back("yesieungkorean"_n, AglMapping{ (AglType)1, 1, 0x3181 });
    s_aglList.emplace_back("yesieungpansioskorean"_n, AglMapping{ (AglType)1, 1, 0x3183 });
    s_aglList.emplace_back("yesieungsioskorean"_n, AglMapping{ (AglType)1, 1, 0x3182 });
    s_aglList.emplace_back("yetivhebrew"_n, AglMapping{ (AglType)1, 1, 0x059A });
    s_aglList.emplace_back("ygrave"_n, AglMapping{ (AglType)3, 1, 0x1EF3 });
    s_aglList.emplace_back("yhook"_n, AglMapping{ (AglType)1, 1, 0x01B4 });
    s_aglList.emplace_back("yhookabove"_n, AglMapping{ (AglType)1, 1, 0x1EF7 });
    s_aglList.emplace_back("yiarmenian"_n, AglMapping{ (AglType)1, 1, 0x0575 });
    s_aglList.emplace_back("yicyrillic"_n, AglMapping{ (AglType)1, 1, 0x0457 });
    s_aglList.emplace_back("yikorean"_n, AglMapping{ (AglType)1, 1, 0x3162 });
    s_aglList.emplace_back("yinyang"_n, AglMapping{ (AglType)1, 1, 0x262F });
    s_aglList.emplace_back("yiwnarmenian"_n, AglMapping{ (AglType)1, 1, 0x0582 });
    s_aglList.emplace_back("ymonospace"_n, AglMapping{ (AglType)1, 1, 0xFF59 });
    s_aglList.emplace_back("yod"_n, AglMapping{ (AglType)1, 1, 0x05D9 });
    s_aglList.emplace_back("yoddagesh"_n, AglMapping{ (AglType)1, 1, 0xFB39 });
    s_aglList.emplace_back("yoddageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB39 });
    s_aglList.emplace_back("yodhebrew"_n, AglMapping{ (AglType)1, 1, 0x05D9 });
    s_aglList.emplace_back("yodyodhebrew"_n, AglMapping{ (AglType)1, 1, 0x05F2 });
    s_aglList.emplace_back("yodyodpatahhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB1F });
    s_aglList.emplace_back("yohiragana"_n, AglMapping{ (AglType)1, 1, 0x3088 });
    s_aglList.emplace_back("yoikorean"_n, AglMapping{ (AglType)1, 1, 0x3189 });
    s_aglList.emplace_back("yokatakana"_n, AglMapping{ (AglType)1, 1, 0x30E8 });
    s_aglList.emplace_back("yokatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF96 });
    s_aglList.emplace_back("yokorean"_n, AglMapping{ (AglType)1, 1, 0x315B });
    s_aglList.emplace_back("yosmallhiragana"_n, AglMapping{ (AglType)1, 1, 0x3087 });
    s_aglList.emplace_back("yosmallkatakana"_n, AglMapping{ (AglType)1, 1, 0x30E7 });
    s_aglList.emplace_back("yosmallkatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF6E });
    s_aglList.emplace_back("yotgreek"_n, AglMapping{ (AglType)1, 1, 0x03F3 });
    s_aglList.emplace_back("yoyaekorean"_n, AglMapping{ (AglType)1, 1, 0x3188 });
    s_aglList.emplace_back("yoyakorean"_n, AglMapping{ (AglType)1, 1, 0x3187 });
    s_aglList.emplace_back("yoyakthai"_n, AglMapping{ (AglType)1, 1, 0x0E22 });
    s_aglList.emplace_back("yoyingthai"_n, AglMapping{ (AglType)1, 1, 0x0E0D });
    s_aglList.emplace_back("yparen"_n, AglMapping{ (AglType)1, 1, 0x24B4 });
    s_aglList.emplace_back("ypogegrammeni"_n, AglMapping{ (AglType)1, 1, 0x037A });
    s_aglList.emplace_back("ypogegrammenigreekcmb"_n, AglMapping{ (AglType)1, 1, 0x0345 });
    s_aglList.emplace_back("yr"_n, AglMapping{ (AglType)1, 1, 0x01A6 });
    s_aglList.emplace_back("yring"_n, AglMapping{ (AglType)1, 1, 0x1E99 });
    s_aglList.emplace_back("ysuperior"_n, AglMapping{ (AglType)1, 1, 0x02B8 });
    s_aglList.emplace_back("ytilde"_n, AglMapping{ (AglType)1, 1, 0x1EF9 });
    s_aglList.emplace_back("yturned"_n, AglMapping{ (AglType)1, 1, 0x028E });
    s_aglList.emplace_back("yuhiragana"_n, AglMapping{ (AglType)1, 1, 0x3086 });
    s_aglList.emplace_back("yuikorean"_n, AglMapping{ (AglType)1, 1, 0x318C });
    s_aglList.emplace_back("yukatakana"_n, AglMapping{ (AglType)1, 1, 0x30E6 });
    s_aglList.emplace_back("yukatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF95 });
    s_aglList.emplace_back("yukorean"_n, AglMapping{ (AglType)1, 1, 0x3160 });
    s_aglList.emplace_back("yusbigcyrillic"_n, AglMapping{ (AglType)1, 1, 0x046B });
    s_aglList.emplace_back("yusbigiotifiedcyrillic"_n, AglMapping{ (AglType)1, 1, 0x046D });
    s_aglList.emplace_back("yuslittlecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0467 });
    s_aglList.emplace_back("yuslittleiotifiedcyrillic"_n, AglMapping{ (AglType)1, 1, 0x0469 });
    s_aglList.emplace_back("yusmallhiragana"_n, AglMapping{ (AglType)1, 1, 0x3085 });
    s_aglList.emplace_back("yusmallkatakana"_n, AglMapping{ (AglType)1, 1, 0x30E5 });
    s_aglList.emplace_back("yusmallkatakanahalfwidth"_n, AglMapping{ (AglType)1, 1, 0xFF6D });
    s_aglList.emplace_back("yuyekorean"_n, AglMapping{ (AglType)1, 1, 0x318B });
    s_aglList.emplace_back("yuyeokorean"_n, AglMapping{ (AglType)1, 1, 0x318A });
    s_aglList.emplace_back("yyabengali"_n, AglMapping{ (AglType)1, 1, 0x09DF });
    s_aglList.emplace_back("yyadeva"_n, AglMapping{ (AglType)1, 1, 0x095F });
    s_aglList.emplace_back("z"_n, AglMapping{ (AglType)11, 1, 0x007A });
    s_aglList.emplace_back("zaarmenian"_n, AglMapping{ (AglType)1, 1, 0x0566 });
    s_aglList.emplace_back("zacute"_n, AglMapping{ (AglType)3, 1, 0x017A });
    s_aglList.emplace_back("zadeva"_n, AglMapping{ (AglType)1, 1, 0x095B });
    s_aglList.emplace_back("zagurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A5B });
    s_aglList.emplace_back("zaharabic"_n, AglMapping{ (AglType)1, 1, 0x0638 });
    s_aglList.emplace_back("zahfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEC6 });
    s_aglList.emplace_back("zahinitialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEC7 });
    s_aglList.emplace_back("zahiragana"_n, AglMapping{ (AglType)1, 1, 0x3056 });
    s_aglList.emplace_back("zahmedialarabic"_n, AglMapping{ (AglType)1, 1, 0xFEC8 });
    s_aglList.emplace_back("zainarabic"_n, AglMapping{ (AglType)1, 1, 0x0632 });
    s_aglList.emplace_back("zainfinalarabic"_n, AglMapping{ (AglType)1, 1, 0xFEB0 });
    s_aglList.emplace_back("zakatakana"_n, AglMapping{ (AglType)1, 1, 0x30B6 });
    s_aglList.emplace_back("zaqefgadolhebrew"_n, AglMapping{ (AglType)1, 1, 0x0595 });
    s_aglList.emplace_back("zaqefqatanhebrew"_n, AglMapping{ (AglType)1, 1, 0x0594 });
    s_aglList.emplace_back("zarqahebrew"_n, AglMapping{ (AglType)1, 1, 0x0598 });
    s_aglList.emplace_back("zayin"_n, AglMapping{ (AglType)1, 1, 0x05D6 });
    s_aglList.emplace_back("zayindagesh"_n, AglMapping{ (AglType)1, 1, 0xFB36 });
    s_aglList.emplace_back("zayindageshhebrew"_n, AglMapping{ (AglType)1, 1, 0xFB36 });
    s_aglList.emplace_back("zayinhebrew"_n, AglMapping{ (AglType)1, 1, 0x05D6 });
    s_aglList.emplace_back("zbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3117 });
    s_aglList.emplace_back("zcaron"_n, AglMapping{ (AglType)11, 1, 0x017E });
    s_aglList.emplace_back("zcircle"_n, AglMapping{ (AglType)1, 1, 0x24E9 });
    s_aglList.emplace_back("zcircumflex"_n, AglMapping{ (AglType)1, 1, 0x1E91 });
    s_aglList.emplace_back("zcurl"_n, AglMapping{ (AglType)1, 1, 0x0291 });
    s_aglList.emplace_back("zdot"_n, AglMapping{ (AglType)1, 1, 0x017C });
    s_aglList.emplace_back("zdotaccent"_n, AglMapping{ (AglType)3, 1, 0x017C });
    s_aglList.emplace_back("zdotbelow"_n, AglMapping{ (AglType)1, 1, 0x1E93 });
    s_aglList.emplace_back("zecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0437 });
    s_aglList.emplace_back("zedescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x0499 });
    s_aglList.emplace_back("zedieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04DF });
    s_aglList.emplace_back("zehiragana"_n, AglMapping{ (AglType)1, 1, 0x305C });
    s_aglList.emplace_back("zekatakana"_n, AglMapping{ (AglType)1, 1, 0x30BC });
    s_aglList.emplace_back("zero"_n, AglMapping{ (AglType)11, 1, 0x0030 });
    s_aglList.emplace_back("zeroarabic"_n, AglMapping{ (AglType)1, 1, 0x0660 });
    s_aglList.emplace_back("zerobengali"_n, AglMapping{ (AglType)1, 1, 0x09E6 });
    s_aglList.emplace_back("zerodeva"_n, AglMapping{ (AglType)1, 1, 0x0966 });
    s_aglList.emplace_back("zerogujarati"_n, AglMapping{ (AglType)1, 1, 0x0AE6 });
    s_aglList.emplace_back("zerogurmukhi"_n, AglMapping{ (AglType)1, 1, 0x0A66 });
    s_aglList.emplace_back("zerohackarabic"_n, AglMapping{ (AglType)1, 1, 0x0660 });
    s_aglList.emplace_back("zeroinferior"_n, AglMapping{ (AglType)1, 1, 0x2080 });
    s_aglList.emplace_back("zeromonospace"_n, AglMapping{ (AglType)1, 1, 0xFF10 });
    s_aglList.emplace_back("zerooldstyle"_n, AglMapping{ (AglType)1, 1, 0xF730 });
    s_aglList.emplace_back("zeropersian"_n, AglMapping{ (AglType)1, 1, 0x06F0 });
    s_aglList.emplace_back("zerosuperior"_n, AglMapping{ (AglType)1, 1, 0x2070 });
    s_aglList.emplace_back("zerothai"_n, AglMapping{ (AglType)1, 1, 0x0E50 });
    s_aglList.emplace_back("zerowidthjoiner"_n, AglMapping{ (AglType)1, 1, 0xFEFF });
    s_aglList.emplace_back("zerowidthnonjoiner"_n, AglMapping{ (AglType)1, 1, 0x200C });
    s_aglList.emplace_back("zerowidthspace"_n, AglMapping{ (AglType)1, 1, 0x200B });
    s_aglList.emplace_back("zeta"_n, AglMapping{ (AglType)3, 1, 0x03B6 });
    s_aglList.emplace_back("zhbopomofo"_n, AglMapping{ (AglType)1, 1, 0x3113 });
    s_aglList.emplace_back("zhearmenian"_n, AglMapping{ (AglType)1, 1, 0x056A });
    s_aglList.emplace_back("zhebrevecyrillic"_n, AglMapping{ (AglType)1, 1, 0x04C2 });
    s_aglList.emplace_back("zhecyrillic"_n, AglMapping{ (AglType)1, 1, 0x0436 });
    s_aglList.emplace_back("zhedescendercyrillic"_n, AglMapping{ (AglType)1, 1, 0x0497 });
    s_aglList.emplace_back("zhedieresiscyrillic"_n, AglMapping{ (AglType)1, 1, 0x04DD });
    s_aglList.emplace_back("zihiragana"_n, AglMapping{ (AglType)1, 1, 0x3058 });
    s_aglList.emplace_back("zikatakana"_n, AglMapping{ (AglType)1, 1, 0x30B8 });
    s_aglList.emplace_back("zinorhebrew"_n, AglMapping{ (AglType)1, 1, 0x05AE });
    s_aglList.emplace_back("zlinebelow"_n, AglMapping{ (AglType)1, 1, 0x1E95 });
    s_aglList.emplace_back("zmonospace"_n, AglMapping{ (AglType)1, 1, 0xFF5A });
    s_aglList.emplace_back("zohiragana"_n, AglMapping{ (AglType)1, 1, 0x305E });
    s_aglList.emplace_back("zokatakana"_n, AglMapping{ (AglType)1, 1, 0x30BE });
    s_aglList.emplace_back("zparen"_n, AglMapping{ (AglType)1, 1, 0x24B5 });
    s_aglList.emplace_back("zretroflexhook"_n, AglMapping{ (AglType)1, 1, 0x0290 });
    s_aglList.emplace_back("zstroke"_n, AglMapping{ (AglType)1, 1, 0x01B6 });
    s_aglList.emplace_back("zuhiragana"_n, AglMapping{ (AglType)1, 1, 0x305A });
    s_aglList.emplace_back("zukatakana"_n, AglMapping{ (AglType)1, 1, 0x30BA });
    s_aglList.emplace_back("a100"_n, AglMapping{ (AglType)4, 1, 0x275E });
    s_aglList.emplace_back("a101"_n, AglMapping{ (AglType)4, 1, 0x2761 });
    s_aglList.emplace_back("a102"_n, AglMapping{ (AglType)4, 1, 0x2762 });
    s_aglList.emplace_back("a103"_n, AglMapping{ (AglType)4, 1, 0x2763 });
    s_aglList.emplace_back("a104"_n, AglMapping{ (AglType)4, 1, 0x2764 });
    s_aglList.emplace_back("a105"_n, AglMapping{ (AglType)4, 1, 0x2710 });
    s_aglList.emplace_back("a106"_n, AglMapping{ (AglType)4, 1, 0x2765 });
    s_aglList.emplace_back("a107"_n, AglMapping{ (AglType)4, 1, 0x2766 });
    s_aglList.emplace_back("a108"_n, AglMapping{ (AglType)4, 1, 0x2767 });
    s_aglList.emplace_back("a109"_n, AglMapping{ (AglType)4, 1, 0x2660 });
    s_aglList.emplace_back("a10"_n, AglMapping{ (AglType)4, 1, 0x2721 });
    s_aglList.emplace_back("a110"_n, AglMapping{ (AglType)4, 1, 0x2665 });
    s_aglList.emplace_back("a111"_n, AglMapping{ (AglType)4, 1, 0x2666 });
    s_aglList.emplace_back("a112"_n, AglMapping{ (AglType)4, 1, 0x2663 });
    s_aglList.emplace_back("a117"_n, AglMapping{ (AglType)4, 1, 0x2709 });
    s_aglList.emplace_back("a118"_n, AglMapping{ (AglType)4, 1, 0x2708 });
    s_aglList.emplace_back("a119"_n, AglMapping{ (AglType)4, 1, 0x2707 });
    s_aglList.emplace_back("a11"_n, AglMapping{ (AglType)4, 1, 0x261B });
    s_aglList.emplace_back("a120"_n, AglMapping{ (AglType)4, 1, 0x2460 });
    s_aglList.emplace_back("a121"_n, AglMapping{ (AglType)4, 1, 0x2461 });
    s_aglList.emplace_back("a122"_n, AglMapping{ (AglType)4, 1, 0x2462 });
    s_aglList.emplace_back("a123"_n, AglMapping{ (AglType)4, 1, 0x2463 });
    s_aglList.emplace_back("a124"_n, AglMapping{ (AglType)4, 1, 0x2464 });
    s_aglList.emplace_back("a125"_n, AglMapping{ (AglType)4, 1, 0x2465 });
    s_aglList.emplace_back("a126"_n, AglMapping{ (AglType)4, 1, 0x2466 });
    s_aglList.emplace_back("a127"_n, AglMapping{ (AglType)4, 1, 0x2467 });
    s_aglList.emplace_back("a128"_n, AglMapping{ (AglType)4, 1, 0x2468 });
    s_aglList.emplace_back("a129"_n, AglMapping{ (AglType)4, 1, 0x2469 });
    s_aglList.emplace_back("a12"_n, AglMapping{ (AglType)4, 1, 0x261E });
    s_aglList.emplace_back("a130"_n, AglMapping{ (AglType)4, 1, 0x2776 });
    s_aglList.emplace_back("a131"_n, AglMapping{ (AglType)4, 1, 0x2777 });
    s_aglList.emplace_back("a132"_n, AglMapping{ (AglType)4, 1, 0x2778 });
    s_aglList.emplace_back("a133"_n, AglMapping{ (AglType)4, 1, 0x2779 });
    s_aglList.emplace_back("a134"_n, AglMapping{ (AglType)4, 1, 0x277A });
    s_aglList.emplace_back("a135"_n, AglMapping{ (AglType)4, 1, 0x277B });
    s_aglList.emplace_back("a136"_n, AglMapping{ (AglType)4, 1, 0x277C });
    s_aglList.emplace_back("a137"_n, AglMapping{ (AglType)4, 1, 0x277D });
    s_aglList.emplace_back("a138"_n, AglMapping{ (AglType)4, 1, 0x277E });
    s_aglList.emplace_back("a139"_n, AglMapping{ (AglType)4, 1, 0x277F });
    s_aglList.emplace_back("a13"_n, AglMapping{ (AglType)4, 1, 0x270C });
    s_aglList.emplace_back("a140"_n, AglMapping{ (AglType)4, 1, 0x2780 });
    s_aglList.emplace_back("a141"_n, AglMapping{ (AglType)4, 1, 0x2781 });
    s_aglList.emplace_back("a142"_n, AglMapping{ (AglType)4, 1, 0x2782 });
    s_aglList.emplace_back("a143"_n, AglMapping{ (AglType)4, 1, 0x2783 });
    s_aglList.emplace_back("a144"_n, AglMapping{ (AglType)4, 1, 0x2784 });
    s_aglList.emplace_back("a145"_n, AglMapping{ (AglType)4, 1, 0x2785 });
    s_aglList.emplace_back("a146"_n, AglMapping{ (AglType)4, 1, 0x2786 });
    s_aglList.emplace_back("a147"_n, AglMapping{ (AglType)4, 1, 0x2787 });
    s_aglList.emplace_back("a148"_n, AglMapping{ (AglType)4, 1, 0x2788 });
    s_aglList.emplace_back("a149"_n, AglMapping{ (AglType)4, 1, 0x2789 });
    s_aglList.emplace_back("a14"_n, AglMapping{ (AglType)4, 1, 0x270D });
    s_aglList.emplace_back("a150"_n, AglMapping{ (AglType)4, 1, 0x278A });
    s_aglList.emplace_back("a151"_n, AglMapping{ (AglType)4, 1, 0x278B });
    s_aglList.emplace_back("a152"_n, AglMapping{ (AglType)4, 1, 0x278C });
    s_aglList.emplace_back("a153"_n, AglMapping{ (AglType)4, 1, 0x278D });
    s_aglList.emplace_back("a154"_n, AglMapping{ (AglType)4, 1, 0x278E });
    s_aglList.emplace_back("a155"_n, AglMapping{ (AglType)4, 1, 0x278F });
    s_aglList.emplace_back("a156"_n, AglMapping{ (AglType)4, 1, 0x2790 });
    s_aglList.emplace_back("a157"_n, AglMapping{ (AglType)4, 1, 0x2791 });
    s_aglList.emplace_back("a158"_n, AglMapping{ (AglType)4, 1, 0x2792 });
    s_aglList.emplace_back("a159"_n, AglMapping{ (AglType)4, 1, 0x2793 });
    s_aglList.emplace_back("a15"_n, AglMapping{ (AglType)4, 1, 0x270E });
    s_aglList.emplace_back("a160"_n, AglMapping{ (AglType)4, 1, 0x2794 });
    s_aglList.emplace_back("a161"_n, AglMapping{ (AglType)4, 1, 0x2192 });
    s_aglList.emplace_back("a162"_n, AglMapping{ (AglType)4, 1, 0x27A3 });
    s_aglList.emplace_back("a163"_n, AglMapping{ (AglType)4, 1, 0x2194 });
    s_aglList.emplace_back("a164"_n, AglMapping{ (AglType)4, 1, 0x2195 });
    s_aglList.emplace_back("a165"_n, AglMapping{ (AglType)4, 1, 0x2799 });
    s_aglList.emplace_back("a166"_n, AglMapping{ (AglType)4, 1, 0x279B });
    s_aglList.emplace_back("a167"_n, AglMapping{ (AglType)4, 1, 0x279C });
    s_aglList.emplace_back("a168"_n, AglMapping{ (AglType)4, 1, 0x279D });
    s_aglList.emplace_back("a169"_n, AglMapping{ (AglType)4, 1, 0x279E });
    s_aglList.emplace_back("a16"_n, AglMapping{ (AglType)4, 1, 0x270F });
    s_aglList.emplace_back("a170"_n, AglMapping{ (AglType)4, 1, 0x279F });
    s_aglList.emplace_back("a171"_n, AglMapping{ (AglType)4, 1, 0x27A0 });
    s_aglList.emplace_back("a172"_n, AglMapping{ (AglType)4, 1, 0x27A1 });
    s_aglList.emplace_back("a173"_n, AglMapping{ (AglType)4, 1, 0x27A2 });
    s_aglList.emplace_back("a174"_n, AglMapping{ (AglType)4, 1, 0x27A4 });
    s_aglList.emplace_back("a175"_n, AglMapping{ (AglType)4, 1, 0x27A5 });
    s_aglList.emplace_back("a176"_n, AglMapping{ (AglType)4, 1, 0x27A6 });
    s_aglList.emplace_back("a177"_n, AglMapping{ (AglType)4, 1, 0x27A7 });
    s_aglList.emplace_back("a178"_n, AglMapping{ (AglType)4, 1, 0x27A8 });
    s_aglList.emplace_back("a179"_n, AglMapping{ (AglType)4, 1, 0x27A9 });
    s_aglList.emplace_back("a17"_n, AglMapping{ (AglType)4, 1, 0x2711 });
    s_aglList.emplace_back("a180"_n, AglMapping{ (AglType)4, 1, 0x27AB });
    s_aglList.emplace_back("a181"_n, AglMapping{ (AglType)4, 1, 0x27AD });
    s_aglList.emplace_back("a182"_n, AglMapping{ (AglType)4, 1, 0x27AF });
    s_aglList.emplace_back("a183"_n, AglMapping{ (AglType)4, 1, 0x27B2 });
    s_aglList.emplace_back("a184"_n, AglMapping{ (AglType)4, 1, 0x27B3 });
    s_aglList.emplace_back("a185"_n, AglMapping{ (AglType)4, 1, 0x27B5 });
    s_aglList.emplace_back("a186"_n, AglMapping{ (AglType)4, 1, 0x27B8 });
    s_aglList.emplace_back("a187"_n, AglMapping{ (AglType)4, 1, 0x27BA });
    s_aglList.emplace_back("a188"_n, AglMapping{ (AglType)4, 1, 0x27BB });
    s_aglList.emplace_back("a189"_n, AglMapping{ (AglType)4, 1, 0x27BC });
    s_aglList.emplace_back("a18"_n, AglMapping{ (AglType)4, 1, 0x2712 });
    s_aglList.emplace_back("a190"_n, AglMapping{ (AglType)4, 1, 0x27BD });
    s_aglList.emplace_back("a191"_n, AglMapping{ (AglType)4, 1, 0x27BE });
    s_aglList.emplace_back("a192"_n, AglMapping{ (AglType)4, 1, 0x279A });
    s_aglList.emplace_back("a193"_n, AglMapping{ (AglType)4, 1, 0x27AA });
    s_aglList.emplace_back("a194"_n, AglMapping{ (AglType)4, 1, 0x27B6 });
    s_aglList.emplace_back("a195"_n, AglMapping{ (AglType)4, 1, 0x27B9 });
    s_aglList.emplace_back("a196"_n, AglMapping{ (AglType)4, 1, 0x2798 });
    s_aglList.emplace_back("a197"_n, AglMapping{ (AglType)4, 1, 0x27B4 });
    s_aglList.emplace_back("a198"_n, AglMapping{ (AglType)4, 1, 0x27B7 });
    s_aglList.emplace_back("a199"_n, AglMapping{ (AglType)4, 1, 0x27AC });
    s_aglList.emplace_back("a19"_n, AglMapping{ (AglType)4, 1, 0x2713 });
    s_aglList.emplace_back("a1"_n, AglMapping{ (AglType)4, 1, 0x2701 });
    s_aglList.emplace_back("a200"_n, AglMapping{ (AglType)4, 1, 0x27AE });
    s_aglList.emplace_back("a201"_n, AglMapping{ (AglType)4, 1, 0x27B1 });
    s_aglList.emplace_back("a202"_n, AglMapping{ (AglType)4, 1, 0x2703 });
    s_aglList.emplace_back("a203"_n, AglMapping{ (AglType)4, 1, 0x2750 });
    s_aglList.emplace_back("a204"_n, AglMapping{ (AglType)4, 1, 0x2752 });
    s_aglList.emplace_back("a205"_n, AglMapping{ (AglType)4, 1, 0x276E });
    s_aglList.emplace_back("a206"_n, AglMapping{ (AglType)4, 1, 0x2770 });
    s_aglList.emplace_back("a20"_n, AglMapping{ (AglType)4, 1, 0x2714 });
    s_aglList.emplace_back("a21"_n, AglMapping{ (AglType)4, 1, 0x2715 });
    s_aglList.emplace_back("a22"_n, AglMapping{ (AglType)4, 1, 0x2716 });
    s_aglList.emplace_back("a23"_n, AglMapping{ (AglType)4, 1, 0x2717 });
    s_aglList.emplace_back("a24"_n, AglMapping{ (AglType)4, 1, 0x2718 });
    s_aglList.emplace_back("a25"_n, AglMapping{ (AglType)4, 1, 0x2719 });
    s_aglList.emplace_back("a26"_n, AglMapping{ (AglType)4, 1, 0x271A });
    s_aglList.emplace_back("a27"_n, AglMapping{ (AglType)4, 1, 0x271B });
    s_aglList.emplace_back("a28"_n, AglMapping{ (AglType)4, 1, 0x271C });
    s_aglList.emplace_back("a29"_n, AglMapping{ (AglType)4, 1, 0x2722 });
    s_aglList.emplace_back("a2"_n, AglMapping{ (AglType)4, 1, 0x2702 });
    s_aglList.emplace_back("a30"_n, AglMapping{ (AglType)4, 1, 0x2723 });
    s_aglList.emplace_back("a31"_n, AglMapping{ (AglType)4, 1, 0x2724 });
    s_aglList.emplace_back("a32"_n, AglMapping{ (AglType)4, 1, 0x2725 });
    s_aglList.emplace_back("a33"_n, AglMapping{ (AglType)4, 1, 0x2726 });
    s_aglList.emplace_back("a34"_n, AglMapping{ (AglType)4, 1, 0x2727 });
    s_aglList.emplace_back("a35"_n, AglMapping{ (AglType)4, 1, 0x2605 });
    s_aglList.emplace_back("a36"_n, AglMapping{ (AglType)4, 1, 0x2729 });
    s_aglList.emplace_back("a37"_n, AglMapping{ (AglType)4, 1, 0x272A });
    s_aglList.emplace_back("a38"_n, AglMapping{ (AglType)4, 1, 0x272B });
    s_aglList.emplace_back("a39"_n, AglMapping{ (AglType)4, 1, 0x272C });
    s_aglList.emplace_back("a3"_n, AglMapping{ (AglType)4, 1, 0x2704 });
    s_aglList.emplace_back("a40"_n, AglMapping{ (AglType)4, 1, 0x272D });
    s_aglList.emplace_back("a41"_n, AglMapping{ (AglType)4, 1, 0x272E });
    s_aglList.emplace_back("a42"_n, AglMapping{ (AglType)4, 1, 0x272F });
    s_aglList.emplace_back("a43"_n, AglMapping{ (AglType)4, 1, 0x2730 });
    s_aglList.emplace_back("a44"_n, AglMapping{ (AglType)4, 1, 0x2731 });
    s_aglList.emplace_back("a45"_n, AglMapping{ (AglType)4, 1, 0x2732 });
    s_aglList.emplace_back("a46"_n, AglMapping{ (AglType)4, 1, 0x2733 });
    s_aglList.emplace_back("a47"_n, AglMapping{ (AglType)4, 1, 0x2734 });
    s_aglList.emplace_back("a48"_n, AglMapping{ (AglType)4, 1, 0x2735 });
    s_aglList.emplace_back("a49"_n, AglMapping{ (AglType)4, 1, 0x2736 });
    s_aglList.emplace_back("a4"_n, AglMapping{ (AglType)4, 1, 0x260E });
    s_aglList.emplace_back("a50"_n, AglMapping{ (AglType)4, 1, 0x2737 });
    s_aglList.emplace_back("a51"_n, AglMapping{ (AglType)4, 1, 0x2738 });
    s_aglList.emplace_back("a52"_n, AglMapping{ (AglType)4, 1, 0x2739 });
    s_aglList.emplace_back("a53"_n, AglMapping{ (AglType)4, 1, 0x273A });
    s_aglList.emplace_back("a54"_n, AglMapping{ (AglType)4, 1, 0x273B });
    s_aglList.emplace_back("a55"_n, AglMapping{ (AglType)4, 1, 0x273C });
    s_aglList.emplace_back("a56"_n, AglMapping{ (AglType)4, 1, 0x273D });
    s_aglList.emplace_back("a57"_n, AglMapping{ (AglType)4, 1, 0x273E });
    s_aglList.emplace_back("a58"_n, AglMapping{ (AglType)4, 1, 0x273F });
    s_aglList.emplace_back("a59"_n, AglMapping{ (AglType)4, 1, 0x2740 });
    s_aglList.emplace_back("a5"_n, AglMapping{ (AglType)4, 1, 0x2706 });
    s_aglList.emplace_back("a60"_n, AglMapping{ (AglType)4, 1, 0x2741 });
    s_aglList.emplace_back("a61"_n, AglMapping{ (AglType)4, 1, 0x2742 });
    s_aglList.emplace_back("a62"_n, AglMapping{ (AglType)4, 1, 0x2743 });
    s_aglList.emplace_back("a63"_n, AglMapping{ (AglType)4, 1, 0x2744 });
    s_aglList.emplace_back("a64"_n, AglMapping{ (AglType)4, 1, 0x2745 });
    s_aglList.emplace_back("a65"_n, AglMapping{ (AglType)4, 1, 0x2746 });
    s_aglList.emplace_back("a66"_n, AglMapping{ (AglType)4, 1, 0x2747 });
    s_aglList.emplace_back("a67"_n, AglMapping{ (AglType)4, 1, 0x2748 });
    s_aglList.emplace_back("a68"_n, AglMapping{ (AglType)4, 1, 0x2749 });
    s_aglList.emplace_back("a69"_n, AglMapping{ (AglType)4, 1, 0x274A });
    s_aglList.emplace_back("a6"_n, AglMapping{ (AglType)4, 1, 0x271D });
    s_aglList.emplace_back("a70"_n, AglMapping{ (AglType)4, 1, 0x274B });
    s_aglList.emplace_back("a71"_n, AglMapping{ (AglType)4, 1, 0x25CF });
    s_aglList.emplace_back("a72"_n, AglMapping{ (AglType)4, 1, 0x274D });
    s_aglList.emplace_back("a73"_n, AglMapping{ (AglType)4, 1, 0x25A0 });
    s_aglList.emplace_back("a74"_n, AglMapping{ (AglType)4, 1, 0x274F });
    s_aglList.emplace_back("a75"_n, AglMapping{ (AglType)4, 1, 0x2751 });
    s_aglList.emplace_back("a76"_n, AglMapping{ (AglType)4, 1, 0x25B2 });
    s_aglList.emplace_back("a77"_n, AglMapping{ (AglType)4, 1, 0x25BC });
    s_aglList.emplace_back("a78"_n, AglMapping{ (AglType)4, 1, 0x25C6 });
    s_aglList.emplace_back("a79"_n, AglMapping{ (AglType)4, 1, 0x2756 });
    s_aglList.emplace_back("a7"_n, AglMapping{ (AglType)4, 1, 0x271E });
    s_aglList.emplace_back("a81"_n, AglMapping{ (AglType)4, 1, 0x25D7 });
    s_aglList.emplace_back("a82"_n, AglMapping{ (AglType)4, 1, 0x2758 });
    s_aglList.emplace_back("a83"_n, AglMapping{ (AglType)4, 1, 0x2759 });
    s_aglList.emplace_back("a84"_n, AglMapping{ (AglType)4, 1, 0x275A });
    s_aglList.emplace_back("a85"_n, AglMapping{ (AglType)4, 1, 0x276F });
    s_aglList.emplace_back("a86"_n, AglMapping{ (AglType)4, 1, 0x2771 });
    s_aglList.emplace_back("a87"_n, AglMapping{ (AglType)4, 1, 0x2772 });
    s_aglList.emplace_back("a88"_n, AglMapping{ (AglType)4, 1, 0x2773 });
    s_aglList.emplace_back("a89"_n, AglMapping{ (AglType)4, 1, 0x2768 });
    s_aglList.emplace_back("a8"_n, AglMapping{ (AglType)4, 1, 0x271F });
    s_aglList.emplace_back("a90"_n, AglMapping{ (AglType)4, 1, 0x2769 });
    s_aglList.emplace_back("a91"_n, AglMapping{ (AglType)4, 1, 0x276C });
    s_aglList.emplace_back("a92"_n, AglMapping{ (AglType)4, 1, 0x276D });
    s_aglList.emplace_back("a93"_n, AglMapping{ (AglType)4, 1, 0x276A });
    s_aglList.emplace_back("a94"_n, AglMapping{ (AglType)4, 1, 0x276B });
    s_aglList.emplace_back("a95"_n, AglMapping{ (AglType)4, 1, 0x2774 });
    s_aglList.emplace_back("a96"_n, AglMapping{ (AglType)4, 1, 0x2775 });
    s_aglList.emplace_back("a97"_n, AglMapping{ (AglType)4, 1, 0x275B });
    s_aglList.emplace_back("a98"_n, AglMapping{ (AglType)4, 1, 0x275C });
    s_aglList.emplace_back("a99"_n, AglMapping{ (AglType)4, 1, 0x275D });
    s_aglList.emplace_back("a9"_n, AglMapping{ (AglType)4, 1, 0x2720 });

    for (unsigned short i = 0; i < s_aglList.size(); i++)
    {
        auto& pair = s_aglList[i];
        s_aglMap[pair.first.GetString()] = i;
    }

    s_ligatures.emplace_back(s_aglMap["dalethatafpatah"], initializer_list<codepoint>{0x05D3, 0x05B2 });
    s_ligatures.emplace_back(s_aglMap["dalethatafpatahhebrew"], initializer_list<codepoint>{0x05D3, 0x05B2 });
    s_ligatures.emplace_back(s_aglMap["dalethatafsegol"], initializer_list<codepoint>{0x05D3, 0x05B1 });
    s_ligatures.emplace_back(s_aglMap["dalethatafsegolhebrew"], initializer_list<codepoint>{0x05D3, 0x05B1 });
    s_ligatures.emplace_back(s_aglMap["dalethiriq"], initializer_list<codepoint>{0x05D3, 0x05B4 });
    s_ligatures.emplace_back(s_aglMap["dalethiriqhebrew"], initializer_list<codepoint>{0x05D3, 0x05B4 });
    s_ligatures.emplace_back(s_aglMap["daletholam"], initializer_list<codepoint>{0x05D3, 0x05B9 });
    s_ligatures.emplace_back(s_aglMap["daletholamhebrew"], initializer_list<codepoint>{0x05D3, 0x05B9 });
    s_ligatures.emplace_back(s_aglMap["daletpatah"], initializer_list<codepoint>{0x05D3, 0x05B7 });
    s_ligatures.emplace_back(s_aglMap["daletpatahhebrew"], initializer_list<codepoint>{0x05D3, 0x05B7 });
    s_ligatures.emplace_back(s_aglMap["daletqamats"], initializer_list<codepoint>{0x05D3, 0x05B8 });
    s_ligatures.emplace_back(s_aglMap["daletqamatshebrew"], initializer_list<codepoint>{0x05D3, 0x05B8 });
    s_ligatures.emplace_back(s_aglMap["daletqubuts"], initializer_list<codepoint>{0x05D3, 0x05BB });
    s_ligatures.emplace_back(s_aglMap["daletqubutshebrew"], initializer_list<codepoint>{0x05D3, 0x05BB });
    s_ligatures.emplace_back(s_aglMap["daletsegol"], initializer_list<codepoint>{0x05D3, 0x05B6 });
    s_ligatures.emplace_back(s_aglMap["daletsegolhebrew"], initializer_list<codepoint>{0x05D3, 0x05B6 });
    s_ligatures.emplace_back(s_aglMap["daletsheva"], initializer_list<codepoint>{0x05D3, 0x05B0 });
    s_ligatures.emplace_back(s_aglMap["daletshevahebrew"], initializer_list<codepoint>{0x05D3, 0x05B0 });
    s_ligatures.emplace_back(s_aglMap["dalettsere"], initializer_list<codepoint>{0x05D3, 0x05B5 });
    s_ligatures.emplace_back(s_aglMap["dalettserehebrew"], initializer_list<codepoint>{0x05D3, 0x05B5 });
    s_ligatures.emplace_back(s_aglMap["finalkafqamats"], initializer_list<codepoint>{0x05DA, 0x05B8 });
    s_ligatures.emplace_back(s_aglMap["finalkafqamatshebrew"], initializer_list<codepoint>{0x05DA, 0x05B8 });
    s_ligatures.emplace_back(s_aglMap["finalkafsheva"], initializer_list<codepoint>{0x05DA, 0x05B0 });
    s_ligatures.emplace_back(s_aglMap["finalkafshevahebrew"], initializer_list<codepoint>{0x05DA, 0x05B0 });
    s_ligatures.emplace_back(s_aglMap["hamzadammaarabic"], initializer_list<codepoint>{0x0621, 0x064F });
    s_ligatures.emplace_back(s_aglMap["hamzadammatanarabic"], initializer_list<codepoint>{0x0621, 0x064C });
    s_ligatures.emplace_back(s_aglMap["hamzafathaarabic"], initializer_list<codepoint>{0x0621, 0x064E });
    s_ligatures.emplace_back(s_aglMap["hamzafathatanarabic"], initializer_list<codepoint>{0x0621, 0x064B });
    s_ligatures.emplace_back(s_aglMap["hamzalowkasraarabic"], initializer_list<codepoint>{0x0621, 0x0650 });
    s_ligatures.emplace_back(s_aglMap["hamzalowkasratanarabic"], initializer_list<codepoint>{0x0621, 0x064D });
    s_ligatures.emplace_back(s_aglMap["hamzasukunarabic"], initializer_list<codepoint>{0x0621, 0x0652 });
    s_ligatures.emplace_back(s_aglMap["lamedholam"], initializer_list<codepoint>{0x05DC, 0x05B9 });
    s_ligatures.emplace_back(s_aglMap["lamedholamdagesh"], initializer_list<codepoint>{0x05DC, 0x05B9, 0x05BC });
    s_ligatures.emplace_back(s_aglMap["lamedholamdageshhebrew"], initializer_list<codepoint>{0x05DC, 0x05B9, 0x05BC });
    s_ligatures.emplace_back(s_aglMap["lamedholamhebrew"], initializer_list<codepoint>{0x05DC, 0x05B9 });
    s_ligatures.emplace_back(s_aglMap["lammeemjeeminitialarabic"], initializer_list<codepoint>{0xFEDF, 0xFEE4, 0xFEA0 });
    s_ligatures.emplace_back(s_aglMap["lammeemkhahinitialarabic"], initializer_list<codepoint>{0xFEDF, 0xFEE4, 0xFEA8 });
    s_ligatures.emplace_back(s_aglMap["noonhehinitialarabic"], initializer_list<codepoint>{0xFEE7, 0xFEEC });
    s_ligatures.emplace_back(s_aglMap["qofhatafpatah"], initializer_list<codepoint>{0x05E7, 0x05B2 });
    s_ligatures.emplace_back(s_aglMap["qofhatafpatahhebrew"], initializer_list<codepoint>{0x05E7, 0x05B2 });
    s_ligatures.emplace_back(s_aglMap["qofhatafsegol"], initializer_list<codepoint>{0x05E7, 0x05B1 });
    s_ligatures.emplace_back(s_aglMap["qofhatafsegolhebrew"], initializer_list<codepoint>{0x05E7, 0x05B1 });
    s_ligatures.emplace_back(s_aglMap["qofhiriq"], initializer_list<codepoint>{0x05E7, 0x05B4 });
    s_ligatures.emplace_back(s_aglMap["qofhiriqhebrew"], initializer_list<codepoint>{0x05E7, 0x05B4 });
    s_ligatures.emplace_back(s_aglMap["qofholam"], initializer_list<codepoint>{0x05E7, 0x05B9 });
    s_ligatures.emplace_back(s_aglMap["qofholamhebrew"], initializer_list<codepoint>{0x05E7, 0x05B9 });
    s_ligatures.emplace_back(s_aglMap["qofpatah"], initializer_list<codepoint>{0x05E7, 0x05B7 });
    s_ligatures.emplace_back(s_aglMap["qofpatahhebrew"], initializer_list<codepoint>{0x05E7, 0x05B7 });
    s_ligatures.emplace_back(s_aglMap["qofqamats"], initializer_list<codepoint>{0x05E7, 0x05B8 });
    s_ligatures.emplace_back(s_aglMap["qofqamatshebrew"], initializer_list<codepoint>{0x05E7, 0x05B8 });
    s_ligatures.emplace_back(s_aglMap["qofqubuts"], initializer_list<codepoint>{0x05E7, 0x05BB });
    s_ligatures.emplace_back(s_aglMap["qofqubutshebrew"], initializer_list<codepoint>{0x05E7, 0x05BB });
    s_ligatures.emplace_back(s_aglMap["qofsegol"], initializer_list<codepoint>{0x05E7, 0x05B6 });
    s_ligatures.emplace_back(s_aglMap["qofsegolhebrew"], initializer_list<codepoint>{0x05E7, 0x05B6 });
    s_ligatures.emplace_back(s_aglMap["qofsheva"], initializer_list<codepoint>{0x05E7, 0x05B0 });
    s_ligatures.emplace_back(s_aglMap["qofshevahebrew"], initializer_list<codepoint>{0x05E7, 0x05B0 });
    s_ligatures.emplace_back(s_aglMap["qoftsere"], initializer_list<codepoint>{0x05E7, 0x05B5 });
    s_ligatures.emplace_back(s_aglMap["qoftserehebrew"], initializer_list<codepoint>{0x05E7, 0x05B5 });
    s_ligatures.emplace_back(s_aglMap["rehyehaleflamarabic"], initializer_list<codepoint>{0x0631, 0xFEF3, 0xFE8E, 0x0644 });
    s_ligatures.emplace_back(s_aglMap["reshhatafpatah"], initializer_list<codepoint>{0x05E8, 0x05B2 });
    s_ligatures.emplace_back(s_aglMap["reshhatafpatahhebrew"], initializer_list<codepoint>{0x05E8, 0x05B2 });
    s_ligatures.emplace_back(s_aglMap["reshhatafsegol"], initializer_list<codepoint>{0x05E8, 0x05B1 });
    s_ligatures.emplace_back(s_aglMap["reshhatafsegolhebrew"], initializer_list<codepoint>{0x05E8, 0x05B1 });
    s_ligatures.emplace_back(s_aglMap["reshhiriq"], initializer_list<codepoint>{0x05E8, 0x05B4 });
    s_ligatures.emplace_back(s_aglMap["reshhiriqhebrew"], initializer_list<codepoint>{0x05E8, 0x05B4 });
    s_ligatures.emplace_back(s_aglMap["reshholam"], initializer_list<codepoint>{0x05E8, 0x05B9 });
    s_ligatures.emplace_back(s_aglMap["reshholamhebrew"], initializer_list<codepoint>{0x05E8, 0x05B9 });
    s_ligatures.emplace_back(s_aglMap["reshpatah"], initializer_list<codepoint>{0x05E8, 0x05B7 });
    s_ligatures.emplace_back(s_aglMap["reshpatahhebrew"], initializer_list<codepoint>{0x05E8, 0x05B7 });
    s_ligatures.emplace_back(s_aglMap["reshqamats"], initializer_list<codepoint>{0x05E8, 0x05B8 });
    s_ligatures.emplace_back(s_aglMap["reshqamatshebrew"], initializer_list<codepoint>{0x05E8, 0x05B8 });
    s_ligatures.emplace_back(s_aglMap["reshqubuts"], initializer_list<codepoint>{0x05E8, 0x05BB });
    s_ligatures.emplace_back(s_aglMap["reshqubutshebrew"], initializer_list<codepoint>{0x05E8, 0x05BB });
    s_ligatures.emplace_back(s_aglMap["reshsegol"], initializer_list<codepoint>{0x05E8, 0x05B6 });
    s_ligatures.emplace_back(s_aglMap["reshsegolhebrew"], initializer_list<codepoint>{0x05E8, 0x05B6 });
    s_ligatures.emplace_back(s_aglMap["reshsheva"], initializer_list<codepoint>{0x05E8, 0x05B0 });
    s_ligatures.emplace_back(s_aglMap["reshshevahebrew"], initializer_list<codepoint>{0x05E8, 0x05B0 });
    s_ligatures.emplace_back(s_aglMap["reshtsere"], initializer_list<codepoint>{0x05E8, 0x05B5 });
    s_ligatures.emplace_back(s_aglMap["reshtserehebrew"], initializer_list<codepoint>{0x05E8, 0x05B5 });
    s_ligatures.emplace_back(s_aglMap["shaddafathatanarabic"], initializer_list<codepoint>{0x0651, 0x064B });
    s_ligatures.emplace_back(s_aglMap["tchehmeeminitialarabic"], initializer_list<codepoint>{0xFB7C, 0xFEE4 });
}
