"""
Stream method tests using actual receiver binary outputs for pyrtcm.rtcmReader

Created on 3 Oct 2020

*** NB: must be saved in UTF-8 format ***

@author: semuadmin
"""

# pylint: disable=line-too-long, invalid-name, missing-docstring, no-member

import os
from io import BufferedReader, BytesIO
import sys
import unittest
from logging import ERROR

from io import StringIO
from pyrtcm import (
    RTCMReader,
    RTCMMessage,
    RTCMParseError,
    RTCMMessageError,
    RTCMStreamError,
    ERR_IGNORE,
    ERR_LOG,
    ERR_RAISE,
)
import pyrtcm.rtcmtypes_core as rtt

DIRNAME = os.path.dirname(__file__)


class StreamTest(unittest.TestCase):
    def setUp(self):
        self.maxDiff = None

        self._raw1005ex = b"\xd3\x00\x13\x3e\xd7\xd3\x02\x02\x98\x0e\xde\xef\x34\xb4\xbd\x62\xac\x09\x41\x98\x6f\x33\x36\x0b\x98"
        self._raw1005 = (
            b"\xd3\x00\x13>\xd0\x00\x03\x8aX\xd9I<\x87/4\x10\x9d\x07\xd6\xafH Z\xd7\xf7"
        )
        self._raw1007 = b"\xd3\x00\x08>\xf4\xd2\x03ABC\xeapo\xc7"
        # 00111110 11110100 11010010 00000011 01000001 01000010 01000011 11101010
        self._raw1065 = (
            b"\xd3\x00\x12B\x91\x81\xc9\x84\x00\x04B\xb8\x88\x008\x80\t\xd0F\x00(\xf0kf"
        )
        self._raw1023 = b'\xd3\x00I?\xf0\x8c\xa9\x02\xe0\xe0(\x80P\x07\x04\x04\x80P\x00{?\xc0/o\xf0\x0b\xee\xfc\x81\xfd\xbfp\x17\xc4\x04\x04\xf9\x80\x00\xdf?\xe8#\xe7\xfb\x02~\x00`o\xc8\x04\x11\xfa\xff\x82\xbf?\xc07\xfc\n\x08\x00\x00\x81@\x10\x00/\xff\xfe\x0bz"\x00\xc1\x16\x92'
        self._payload1007 = self._raw1007[3:-3]

    def tearDown(self):
        pass

    def catchio(self):
        """
        Capture stdout as string.
        """

        self._saved_stdout = sys.stdout
        self._strout = StringIO()
        sys.stdout = self._strout

    def restoreio(self) -> str:
        """
        Return captured output and restore stdout.
        """

        sys.stdout = self._saved_stdout
        return self._strout.getvalue().strip()

    def test1005example(
        self,
    ):  # test sample 1005 given in RTCM standard (with scaling applied)
        EXPECTED_RESULT = "<RTCM(1005, DF002=1005, DF003=2003, DF021=0, DF022=1, DF023=0, DF024=0, DF141=0, DF025=1114104.5999, DF142=0, DF001_1=0, DF026=-4850729.7108000005, DF364=0, DF027=3975521.4643)>"
        msg = RTCMReader.parse(self._raw1005ex)
        self.assertEqual(str(msg), EXPECTED_RESULT)
        self.assertEqual(msg.DF025, 1114104.5999)
        self.assertEqual(msg.DF026, -4850729.7108000005)
        self.assertEqual(msg.DF027, 3975521.4643)

    def test1023(
        self,
    ):  # test sample 1023 given in RTCM standard
        EXPECTED_RESULT = "<RTCM(1023, DF002=1023, DF147=8, DF190=1, DF191=1, DF192=173067.5, DF193=57384.5, DF194=5.0, DF195=7.0, DF196=0.008, DF197=0.009000000000000001, DF198=0.8, DF199_01=0.0, DF200_01=-0.0006000000000000001, DF201_01=-0.002, DF199_02=6e-05, DF200_02=-0.00057, DF201_02=-0.004, DF199_03=0.00015000000000000001, DF200_03=-0.00054, DF201_03=-0.007, DF199_04=0.00021, DF200_04=-0.00057, DF201_04=-0.009000000000000001, DF199_05=6e-05, DF200_05=-0.00045, DF201_05=0.002, DF199_06=0.00012, DF200_06=-0.00039, DF201_06=0.0, DF199_07=0.00018, DF200_07=-0.00039, DF201_07=-0.003, DF199_08=0.00024, DF200_08=-0.00039, DF201_08=-0.005, DF199_09=0.00012, DF200_09=-0.00024, DF201_09=0.003, DF199_10=0.00018, DF200_10=-0.00021, DF201_10=0.001, DF199_11=0.00024, DF200_11=-0.00018, DF201_11=-0.001, DF199_12=0.00030000000000000003, DF200_12=-0.00021, DF201_12=-0.004, DF199_13=0.00018, DF200_13=-3e-05, DF201_13=0.005, DF199_14=0.00024, DF200_14=0.0, DF201_14=0.002, DF199_15=0.00030000000000000003, DF200_15=3e-05, DF201_15=0.0, DF199_16=0.00033, DF200_16=-3e-05, DF201_16=-0.002, DF212=0, DF213=0, DF216=5, DF217=5, DF051=59528)>"
        msg = RTCMReader.parse(self._raw1023)
        self.assertEqual(str(msg), EXPECTED_RESULT)

    def test1230(
        self,
    ):  # test sample 1230
        dirname = os.path.dirname(__file__)
        with open(os.path.join(dirname, "pygpsdata-1230.bin"), "rb") as stream:
            EXPECTED_RESULT = [
                "<RTCM(1230, DF002=1230, DF003=0, DF421=0, DF001_3=0, DF422_1=1, DF422_2=1, DF422_3=1, DF422_4=1, DF423=0.0, DF424=0.0, DF425=0.0, DF426=0.0)>",
                "<RTCM(1230, DF002=1230, DF003=0, DF421=0, DF001_3=0, DF422_1=1, DF422_2=1, DF422_3=1, DF422_4=1, DF423=0.0, DF424=0.0, DF425=0.0, DF426=0.0)>",
                "<RTCM(1230, DF002=1230, DF003=0, DF421=0, DF001_3=0, DF422_1=1, DF422_2=1, DF422_3=1, DF422_4=1, DF423=0.0, DF424=0.0, DF425=0.0, DF426=0.0)>",
                "<RTCM(1230, DF002=1230, DF003=0, DF421=0, DF001_3=0, DF422_1=1, DF422_2=1, DF422_3=1, DF422_4=1, DF423=0.0, DF424=0.0, DF425=0.0, DF426=0.0)>",
                "<RTCM(1230, DF002=1230, DF003=0, DF421=0, DF001_3=0, DF422_1=1, DF422_2=0, DF422_3=1, DF422_4=1, DF423=0.0, DF425=0.0, DF426=0.0)>",
            ]
            rtr = RTCMReader(stream, quitonerror=2)
            i = 0
            for raw, parsed in rtr:
                # print(f"{parsed},")
                self.assertEqual(str(parsed), EXPECTED_RESULT[i])
                i += 1
            self.assertEqual(i, 5)

    def testMSM3(
        self,
    ):  # test sample MSM3 output
        dirname = os.path.dirname(__file__)
        with open(os.path.join(dirname, "pygpsdata-RTCMMSM3.log"), "rb") as stream:
            EXPECTED_RESULT = (
                "<RTCM(1073, DF002=1073, DF003=11, DF004=84967000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=301936338347753472, NSat=8, DF395=1077969152, NSig=4, DF396=3148532923, NCell=20, PRN_01=006, PRN_02=011, PRN_03=012, PRN_04=017, PRN_05=019, PRN_06=020, PRN_07=024, PRN_08=025, DF398_01=0.5908203125, DF398_02=0.9345703125, DF398_03=0.705078125, DF398_04=0.8779296875, DF398_05=0.5947265625, DF398_06=0.5322265625, DF398_07=0.5146484375, DF398_08=0.974609375, CELLPRN_01=006, CELLSIG_01=1C, CELLPRN_02=006, CELLSIG_02=2X, CELLPRN_03=006, CELLSIG_03=5X, CELLPRN_04=011, CELLSIG_04=1C, CELLPRN_05=011, CELLSIG_05=2X, CELLPRN_06=011, CELLSIG_06=5X, CELLPRN_07=012, CELLSIG_07=1C, CELLPRN_08=012, CELLSIG_08=2X, CELLPRN_09=017, CELLSIG_09=1C, CELLPRN_10=017, CELLSIG_10=2X, CELLPRN_11=019, CELLSIG_11=1C, CELLPRN_12=019, CELLSIG_12=2W, CELLPRN_13=020, CELLSIG_13=1C, CELLPRN_14=020, CELLSIG_14=2W, CELLPRN_15=024, CELLSIG_15=1C, CELLPRN_16=024, CELLSIG_16=2X, CELLPRN_17=024, CELLSIG_17=5X, CELLPRN_18=025, CELLSIG_18=1C, CELLPRN_19=025, CELLSIG_19=2X, CELLPRN_20=025, CELLSIG_20=5X, DF400_01=-0.00019592046737670898, DF400_02=-0.00015676021575927734, DF400_03=-0.00015121698379516602, DF400_04=-0.00035965442657470703, DF400_05=-0.0003299713134765625, DF400_06=-0.0003161430358886719, DF400_07=-0.00033777952194213867, DF400_08=-0.00030922889709472656, DF400_09=-1.3649463653564453e-05, DF400_10=3.707408905029297e-05, DF400_11=-0.00016617774963378906, DF400_12=-0.00013810396194458008, DF400_13=0.00023931264877319336, DF400_14=0.00028312206268310547, DF400_15=-0.0002028346061706543, DF400_16=-0.00014156103134155273, DF400_17=-0.00013011693954467773, DF400_18=-0.000519096851348877, DF400_19=-0.0004698038101196289, DF400_20=-0.0004578232765197754, DF401_01=-2.4491921067237854e-05, DF401_02=0.0002934858202934265, DF401_03=0.00013548694550991058, DF401_04=-0.00017477944493293762, DF401_05=-5.480460822582245e-05, DF401_06=-1.714937388896942e-05, DF401_07=-0.00022045522928237915, DF401_08=-0.00045765936374664307, DF401_09=-4.125945270061493e-05, DF401_10=0.0002892687916755676, DF401_11=-4.3375417590141296e-05, DF401_12=-3.8215890526771545e-05, DF401_13=0.00035588257014751434, DF401_14=0.00048608891665935516, DF401_15=-0.0001531168818473816, DF401_16=-5.9857964515686035e-05, DF401_17=-5.963258445262909e-05, DF401_18=-0.0004779081791639328, DF401_19=-0.0004279632121324539, DF401_20=-0.0003839787095785141, DF402_01=15, DF402_02=15, DF402_03=15, DF402_04=15, DF402_05=15, DF402_06=15, DF402_07=15, DF402_08=15, DF402_09=15, DF402_10=15, DF402_11=15, DF402_12=15, DF402_13=15, DF402_14=15, DF402_15=15, DF402_16=15, DF402_17=15, DF402_18=15, DF402_19=15, DF402_20=15, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0)>",
                "<RTCM(1083, DF002=1083, DF003=11, DF416=1, DF034=9349000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=4648805530981105664, NSat=7, DF395=1090519040, NSig=2, DF396=16383, NCell=14, PRN_01=002, PRN_02=009, PRN_03=015, PRN_04=016, PRN_05=017, PRN_06=018, PRN_07=019, DF398_01=0.109375, DF398_02=0.0654296875, DF398_03=0.7080078125, DF398_04=0.49609375, DF398_05=0.1494140625, DF398_06=0.572265625, DF398_07=0.9384765625, CELLPRN_01=002, CELLSIG_01=1C, CELLPRN_02=002, CELLSIG_02=2C, CELLPRN_03=009, CELLSIG_03=1C, CELLPRN_04=009, CELLSIG_04=2C, CELLPRN_05=015, CELLSIG_05=1C, CELLPRN_06=015, CELLSIG_06=2C, CELLPRN_07=016, CELLSIG_07=1C, CELLPRN_08=016, CELLSIG_08=2C, CELLPRN_09=017, CELLSIG_09=1C, CELLPRN_10=017, CELLSIG_10=2C, CELLPRN_11=018, CELLSIG_11=1C, CELLPRN_12=018, CELLSIG_12=2C, CELLPRN_13=019, CELLSIG_13=1C, CELLPRN_14=019, CELLSIG_14=2C, DF400_01=4.8160552978515625e-05, DF400_02=9.763240814208984e-05, DF400_03=-0.00029778480529785156, DF400_04=-0.00025451183319091797, DF400_05=0.00029659271240234375, DF400_06=0.0003286004066467285, DF400_07=-0.0003834962844848633, DF400_08=-0.0003559589385986328, DF400_09=-9.435415267944336e-05, DF400_10=-6.389617919921875e-05, DF400_11=0.00026482343673706055, DF400_12=0.00029337406158447266, DF400_13=0.00018155574798583984, DF400_14=0.00021916627883911133, DF401_01=6.616860628128052e-05, DF401_02=-4.974938929080963e-05, DF401_03=-0.0002772025763988495, DF401_04=-0.00024692341685295105, DF401_05=0.000372542068362236, DF401_06=0.0004738159477710724, DF401_07=-0.00028068386018276215, DF401_08=-0.00020503997802734375, DF401_09=5.776435136795044e-05, DF401_10=9.675323963165283e-05, DF401_11=0.00043816305696964264, DF401_12=0.0005486086010932922, DF401_13=0.00024857930839061737, DF401_14=0.00035401806235313416, DF402_01=12, DF402_02=12, DF402_03=15, DF402_04=15, DF402_05=15, DF402_06=15, DF402_07=15, DF402_08=15, DF402_09=15, DF402_10=15, DF402_11=15, DF402_12=15, DF402_13=15, DF402_14=15, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0)>",
                "<RTCM(1093, DF002=1093, DF003=11, DF248=84967000, DF393=0, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=4643212865354858496, NSat=7, DF395=136318976, NSig=3, DF396=2097151, NCell=21, PRN_01=002, PRN_02=010, PRN_03=011, PRN_04=012, PRN_05=024, PRN_06=025, PRN_07=036, DF398_01=0.90625, DF398_02=0.681640625, DF398_03=0.5947265625, DF398_04=0.978515625, DF398_05=0.1435546875, DF398_06=0.69140625, DF398_07=0.0966796875, CELLPRN_01=002, CELLSIG_01=1X, CELLPRN_02=002, CELLSIG_02=6X, CELLPRN_03=002, CELLSIG_03=8X, CELLPRN_04=010, CELLSIG_04=1X, CELLPRN_05=010, CELLSIG_05=6X, CELLPRN_06=010, CELLSIG_06=8X, CELLPRN_07=011, CELLSIG_07=1X, CELLPRN_08=011, CELLSIG_08=6X, CELLPRN_09=011, CELLSIG_09=8X, CELLPRN_10=012, CELLSIG_10=1X, CELLPRN_11=012, CELLSIG_11=6X, CELLPRN_12=012, CELLSIG_12=8X, CELLPRN_13=024, CELLSIG_13=1X, CELLPRN_14=024, CELLSIG_14=6X, CELLPRN_15=024, CELLSIG_15=8X, CELLPRN_16=025, CELLSIG_16=1X, CELLPRN_17=025, CELLSIG_17=6X, CELLPRN_18=025, CELLSIG_18=8X, CELLPRN_19=036, CELLSIG_19=1X, CELLPRN_20=036, CELLSIG_20=6X, CELLPRN_21=036, CELLSIG_21=8X, DF400_01=0.0004801750183105469, DF400_02=0.0004673600196838379, DF400_03=0.0005109310150146484, DF400_04=-0.00032961368560791016, DF400_05=-0.0003273487091064453, DF400_06=-0.00029224157333374023, DF400_07=-0.0004706382751464844, DF400_08=-0.0004949569702148438, DF400_09=-0.00045502185821533203, DF400_10=-0.00025582313537597656, DF400_11=-0.0002620220184326172, DF400_12=-0.0002136826515197754, DF400_13=-0.00032019615173339844, DF400_14=-0.0003135204315185547, DF400_15=-0.00026530027389526367, DF400_16=-0.000273287296295166, DF400_17=-0.00028705596923828125, DF400_18=-0.00024384260177612305, DF400_19=-0.0003002285957336426, DF400_20=-0.00031071901321411133, DF400_21=-0.00026482343673706055, DF401_01=0.0005859378725290298, DF401_02=0.0006353985518217087, DF401_03=0.0006924550980329514, DF401_04=-0.00018913671374320984, DF401_05=-0.0001242123544216156, DF401_06=-0.00010369531810283661, DF401_07=-0.0003564227372407913, DF401_08=-0.00030716508626937866, DF401_09=-0.00023801438510417938, DF401_10=-0.0001758616417646408, DF401_11=-0.00015475042164325714, DF401_12=-7.56438821554184e-05, DF401_13=-0.0003506764769554138, DF401_14=-0.00036552175879478455, DF401_15=-0.00033131055533885956, DF401_16=-0.0001549702137708664, DF401_17=-0.00010699406266212463, DF401_18=-3.882870078086853e-05, DF401_19=-0.00012858770787715912, DF401_20=-7.558800280094147e-05, DF401_21=-4.972890019416809e-05, DF402_01=15, DF402_02=15, DF402_03=15, DF402_04=15, DF402_05=15, DF402_06=15, DF402_07=15, DF402_08=15, DF402_09=15, DF402_10=15, DF402_11=15, DF402_12=15, DF402_13=15, DF402_14=15, DF402_15=15, DF402_16=15, DF402_17=15, DF402_18=15, DF402_19=15, DF402_20=15, DF402_21=15, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0, DF420_21=0)>",
            )
            rtr = RTCMReader(stream, quitonerror=2)
            i = 0
            for raw, parsed in rtr:
                # print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULT[i])
                i += 1
            self.assertEqual(i, 3)
            self.assertIsInstance(
                rtr.datastream, BufferedReader
            )  # test datastream getter

    def testMIXEDRTCM_SCALE(
        self,
    ):  # test mixed stream of NMEA, UBX & RTCM messages with scaling applied
        EXPECTED_RESULTS = (
            "<RTCM(1005, DF002=1005, DF003=0, DF021=0, DF022=1, DF023=1, DF024=1, DF141=0, DF025=4444030.802800001, DF142=1, DF001_1=0, DF026=3085671.2349, DF364=0, DF027=3366658.256)>",
            "<RTCM(4072, DF002=4072, Not_Yet_Implemented)>",
            "<RTCM(1077, DF002=1077, DF003=0, DF004=204137001, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=760738918298550272, NSat=10, DF395=1073807360, NSig=2, DF396=1044459, NCell=17, PRN_01=005, PRN_02=007, PRN_03=009, PRN_04=013, PRN_05=014, PRN_06=015, PRN_07=017, PRN_08=019, PRN_09=020, PRN_10=030, DF397_01=75, DF397_02=75, DF397_03=81, DF397_04=72, DF397_05=67, DF397_06=80, DF397_07=75, DF397_08=82, DF397_09=75, DF397_10=71, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, ExtSatInfo_06=0, ExtSatInfo_07=0, ExtSatInfo_08=0, ExtSatInfo_09=0, ExtSatInfo_10=0, DF398_01=0.005859375, DF398_02=0.5341796875, DF398_03=0.7626953125, DF398_04=0.138671875, DF398_05=0.5498046875, DF398_06=0.11328125, DF398_07=0.8037109375, DF398_08=0.1025390625, DF398_09=0.521484375, DF398_10=0.345703125, DF399_01=-178, DF399_02=-304, DF399_03=-643, DF399_04=477, DF399_05=-52, DF399_06=645, DF399_07=529, DF399_08=643, DF399_09=-428, DF399_10=-181, CELLPRN_01=005, CELLSIG_01=1C, CELLPRN_02=005, CELLSIG_02=2L, CELLPRN_03=007, CELLSIG_03=1C, CELLPRN_04=007, CELLSIG_04=2L, CELLPRN_05=009, CELLSIG_05=1C, CELLPRN_06=009, CELLSIG_06=2L, CELLPRN_07=013, CELLSIG_07=1C, CELLPRN_08=014, CELLSIG_08=1C, CELLPRN_09=014, CELLSIG_09=2L, CELLPRN_10=015, CELLSIG_10=1C, CELLPRN_11=015, CELLSIG_11=2L, CELLPRN_12=017, CELLSIG_12=1C, CELLPRN_13=017, CELLSIG_13=2L, CELLPRN_14=019, CELLSIG_14=1C, CELLPRN_15=020, CELLSIG_15=1C, CELLPRN_16=030, CELLSIG_16=1C, CELLPRN_17=030, CELLSIG_17=2L, DF405_01=0.00014309026300907135, DF405_02=0.00014183297753334045, DF405_03=0.0003883279860019684, DF405_04=0.00038741156458854675, DF405_05=-0.0004838351160287857, DF405_06=-0.00046883709728717804, DF405_07=0.0003478657454252243, DF405_08=0.0002196934074163437, DF405_09=0.00021521002054214478, DF405_10=-0.00018852390348911285, DF405_11=-0.00018319115042686462, DF405_12=-0.00010087713599205017, DF405_13=-9.844452142715454e-05, DF405_14=0.00047875382006168365, DF405_15=0.00043664872646331787, DF405_16=-0.0003105681389570236, DF405_17=-0.00030865520238876343, DF406_01=0.00014193402603268623, DF406_02=0.00014339853078126907, DF406_03=0.00039040297269821167, DF406_04=0.00038743019104003906, DF406_05=-0.0004843934439122677, DF406_06=-0.00046825408935546875, DF406_07=0.0003473707474768162, DF406_08=0.00021758908405900002, DF406_09=0.00021597417071461678, DF406_10=-0.00018658116459846497, DF406_11=-0.00018350128084421158, DF406_12=-9.993184357881546e-05, DF406_13=-9.724870324134827e-05, DF406_14=0.0004128236323595047, DF406_15=0.0004355977289378643, DF406_16=-0.0003112703561782837, DF406_17=-0.00030898721888661385, DF407_01=341, DF407_02=341, DF407_03=341, DF407_04=341, DF407_05=341, DF407_06=341, DF407_07=341, DF407_08=341, DF407_09=341, DF407_10=341, DF407_11=341, DF407_12=341, DF407_13=341, DF407_14=295, DF407_15=341, DF407_16=341, DF407_17=341, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF408_01=45.0, DF408_02=38.0, DF408_03=43.0, DF408_04=39.0, DF408_05=39.0, DF408_06=37.0, DF408_07=45.0, DF408_08=46.0, DF408_09=46.0, DF408_10=39.0, DF408_11=34.0, DF408_12=45.0, DF408_13=38.0, DF408_14=31.0, DF408_15=45.0, DF408_16=46.0, DF408_17=41.0, DF404_01=-0.9231, DF404_02=-0.9194, DF404_03=-0.8321000000000001, DF404_04=-0.8326, DF404_05=-0.4107, DF404_06=-0.4072, DF404_07=0.2451, DF404_08=-0.0693, DF404_09=-0.0684, DF404_10=0.9390000000000001, DF404_11=0.9417000000000001, DF404_12=0.2384, DF404_13=0.2416, DF404_14=0.6636000000000001, DF404_15=-0.9556, DF404_16=-0.21480000000000002, DF404_17=-0.2174)>",
            "<RTCM(1087, DF002=1087, DF003=0, DF416=2, DF034=42119001, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=4039168114821169152, NSat=7, DF395=1090519040, NSig=2, DF396=16382, NCell=13, PRN_01=003, PRN_02=004, PRN_03=005, PRN_04=013, PRN_05=014, PRN_06=015, PRN_07=023, DF397_01=69, DF397_02=64, DF397_03=73, DF397_04=76, DF397_05=66, DF397_06=70, DF397_07=78, DF419_01=12, DF419_02=13, DF419_03=8, DF419_04=5, DF419_05=0, DF419_06=7, DF419_07=10, DF398_01=0.6337890625, DF398_02=0.3427734375, DF398_03=0.25390625, DF398_04=0.310546875, DF398_05=0.5126953125, DF398_06=0.8271484375, DF398_07=0.8837890625, DF399_01=-665, DF399_02=29, DF399_03=672, DF399_04=-573, DF399_05=-211, DF399_06=312, DF399_07=317, CELLPRN_01=003, CELLSIG_01=1C, CELLPRN_02=003, CELLSIG_02=2C, CELLPRN_03=004, CELLSIG_03=1C, CELLPRN_04=004, CELLSIG_04=2C, CELLPRN_05=005, CELLSIG_05=1C, CELLPRN_06=005, CELLSIG_06=2C, CELLPRN_07=013, CELLSIG_07=1C, CELLPRN_08=013, CELLSIG_08=2C, CELLPRN_09=014, CELLSIG_09=1C, CELLPRN_10=014, CELLSIG_10=2C, CELLPRN_11=015, CELLSIG_11=1C, CELLPRN_12=015, CELLSIG_12=2C, CELLPRN_13=023, CELLSIG_13=1C, DF405_01=0.00024936161935329437, DF405_02=0.0002511627972126007, DF405_03=-4.678964614868164e-05, DF405_04=-5.141831934452057e-05, DF405_05=1.1144205927848816e-05, DF405_06=2.15042382478714e-05, DF405_07=0.00047079287469387054, DF405_08=0.0004794951528310776, DF405_09=-0.0003879182040691376, DF405_10=-0.00037603825330734253, DF405_11=0.0002771839499473572, DF405_12=0.0002871435135602951, DF405_13=-0.00023611821234226227, DF406_01=0.00024937279522418976, DF406_02=0.00025077443569898605, DF406_03=-4.834495484828949e-05, DF406_04=-5.1246024668216705e-05, DF406_05=1.1149328202009201e-05, DF406_06=2.1803192794322968e-05, DF406_07=0.00047026341781020164, DF406_08=0.0004848274402320385, DF406_09=-0.0003876127302646637, DF406_10=-0.0003757951781153679, DF406_11=0.0002778824418783188, DF406_12=0.0002880701795220375, DF406_13=-0.00023698341101408005, DF407_01=341, DF407_02=341, DF407_03=340, DF407_04=340, DF407_05=341, DF407_06=341, DF407_07=340, DF407_08=341, DF407_09=341, DF407_10=341, DF407_11=341, DF407_12=341, DF407_13=340, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF408_01=47.0, DF408_02=40.0, DF408_03=47.0, DF408_04=42.0, DF408_05=47.0, DF408_06=39.0, DF408_07=36.0, DF408_08=33.0, DF408_09=48.0, DF408_10=43.0, DF408_11=48.0, DF408_12=40.0, DF408_13=41.0, DF404_01=-0.8193, DF404_02=-0.8173, DF404_03=0.8539, DF404_04=0.8501000000000001, DF404_05=0.7333000000000001, DF404_06=0.7311000000000001, DF404_07=-0.24930000000000002, DF404_08=-0.2543, DF404_09=-0.21580000000000002, DF404_10=-0.21780000000000002, DF404_11=0.3924, DF404_12=0.3947, DF404_13=0.6146)>",
            "<RTCM(1097, DF002=1097, DF003=0, DF248=204137001, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=216181732825628672, NSat=5, DF395=1073872896, NSig=2, DF396=1023, NCell=10, PRN_01=007, PRN_02=008, PRN_03=021, PRN_04=027, PRN_05=030, DF397_01=79, DF397_02=84, DF397_03=89, DF397_04=78, DF397_05=83, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, DF398_01=0.15625, DF398_02=0.2509765625, DF398_03=0.3544921875, DF398_04=0.37109375, DF398_05=0.259765625, DF399_01=-198, DF399_02=-516, DF399_03=423, DF399_04=63, DF399_05=-384, CELLPRN_01=007, CELLSIG_01=1C, CELLPRN_02=007, CELLSIG_02=7Q, CELLPRN_03=008, CELLSIG_03=1C, CELLPRN_04=008, CELLSIG_04=7Q, CELLPRN_05=021, CELLSIG_05=1C, CELLPRN_06=021, CELLSIG_06=7Q, CELLPRN_07=027, CELLSIG_07=1C, CELLPRN_08=027, CELLSIG_08=7Q, CELLPRN_09=030, CELLSIG_09=1C, CELLPRN_10=030, CELLSIG_10=7Q, DF405_01=-4.5398250222206116e-05, DF405_02=-2.8252601623535156e-05, DF405_03=-0.00034597329795360565, DF405_04=-0.0003268253058195114, DF405_05=0.0004809703677892685, DF405_06=0.0005012489855289459, DF405_07=-0.00013696029782295227, DF405_08=-0.0001260414719581604, DF405_09=-1.8440186977386475e-05, DF405_10=-3.041699528694153e-06, DF406_01=-4.44464385509491e-05, DF406_02=-2.835458144545555e-05, DF406_03=-0.0003525479696691036, DF406_04=-0.0003263736143708229, DF406_05=0.00048203859478235245, DF406_06=0.0005008447915315628, DF406_07=-0.0001375703141093254, DF406_08=-0.00012635625898838043, DF406_09=-1.8037855625152588e-05, DF406_10=-3.2926909625530243e-06, DF407_01=341, DF407_02=341, DF407_03=341, DF407_04=341, DF407_05=341, DF407_06=341, DF407_07=341, DF407_08=341, DF407_09=341, DF407_10=341, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF408_01=46.0, DF408_02=49.0, DF408_03=41.0, DF408_04=43.0, DF408_05=43.0, DF408_06=43.0, DF408_07=45.0, DF408_08=49.0, DF408_09=43.0, DF408_10=47.0, DF404_01=-0.5806, DF404_02=-0.5831000000000001, DF404_03=-0.7947000000000001, DF404_04=-0.7943, DF404_05=0.7243, DF404_06=0.7174, DF404_07=0.5534, DF404_08=0.5545, DF404_09=-0.7726000000000001, DF404_10=-0.7733)>",
            "<RTCM(1127, DF002=1127, DF003=0, DF427=204123001, DF393=0, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=198178247981137920, NSat=10, DF395=1074003968, NSig=2, DF396=387754, NCell=11, PRN_01=007, PRN_02=009, PRN_03=010, PRN_04=020, PRN_05=023, PRN_06=028, PRN_07=032, PRN_08=037, PRN_09=040, PRN_10=043, DF397_01=129, DF397_02=132, DF397_03=126, DF397_04=75, DF397_05=81, DF397_06=84, DF397_07=78, DF397_08=74, DF397_09=130, DF397_10=86, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, ExtSatInfo_06=0, ExtSatInfo_07=0, ExtSatInfo_08=0, ExtSatInfo_09=0, ExtSatInfo_10=0, DF398_01=0.1171875, DF398_02=0.4814453125, DF398_03=0.3095703125, DF398_04=0.7255859375, DF398_05=0.41015625, DF398_06=0.5703125, DF398_07=0.5595703125, DF398_08=0.322265625, DF398_09=0.578125, DF398_10=0.673828125, DF399_01=-130, DF399_02=-58, DF399_03=-81, DF399_04=32, DF399_05=-398, DF399_06=436, DF399_07=-523, DF399_08=-65, DF399_09=-182, DF399_10=79, CELLPRN_01=007, CELLSIG_01=7I, CELLPRN_02=009, CELLSIG_02=7I, CELLPRN_03=010, CELLSIG_03=2I, CELLPRN_04=010, CELLSIG_04=7I, CELLPRN_05=020, CELLSIG_05=2I, CELLPRN_06=023, CELLSIG_06=2I, CELLPRN_07=028, CELLSIG_07=2I, CELLPRN_08=032, CELLSIG_08=2I, CELLPRN_09=037, CELLSIG_09=2I, CELLPRN_10=040, CELLSIG_10=2I, CELLPRN_11=043, CELLSIG_11=2I, DF405_01=-0.0003885403275489807, DF405_02=0.00022730417549610138, DF405_03=0.0004036612808704376, DF405_04=0.00039606913924217224, DF405_05=-0.00016684085130691528, DF405_06=-4.75514680147171e-05, DF405_07=0.0003674682229757309, DF405_08=0.00026629865169525146, DF405_09=-0.0002502594143152237, DF405_10=-0.00011803768575191498, DF405_11=-0.0002937670797109604, DF406_01=-0.0003882073797285557, DF406_02=0.0002264929935336113, DF406_03=0.0004031979478895664, DF406_04=0.0003964221104979515, DF406_05=-0.00016694329679012299, DF406_06=-4.848744720220566e-05, DF406_07=0.00036971503868699074, DF406_08=0.0002654106356203556, DF406_09=-0.00025115441530942917, DF406_10=-0.00011868216097354889, DF406_11=-0.00029495684430003166, DF407_01=341, DF407_02=341, DF407_03=341, DF407_04=341, DF407_05=341, DF407_06=341, DF407_07=341, DF407_08=341, DF407_09=341, DF407_10=341, DF407_11=341, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF408_01=45.0, DF408_02=41.0, DF408_03=42.0, DF408_04=45.0, DF408_05=48.0, DF408_06=46.0, DF408_07=42.0, DF408_08=47.0, DF408_09=48.0, DF408_10=44.0, DF408_11=43.0, DF404_01=-0.5674, DF404_02=-0.612, DF404_03=-0.1384, DF404_04=-0.1332, DF404_05=0.5992000000000001, DF404_06=-0.7312000000000001, DF404_07=0.17320000000000002, DF404_08=-0.4308, DF404_09=-0.5975, DF404_10=-0.6733, DF404_11=0.6122000000000001)>",
            "<RTCM(1230, DF002=1230, DF003=0, DF421=1, DF001_3=0, DF422_1=0, DF422_2=0, DF422_3=0, DF422_4=0)>",
            "<RTCM(1007, DF002=1007, DF003=1234, DF029=3, DF030_01=A, DF030_02=B, DF030_03=C, DF031=234)>",
            "<RTCM(1117, DF002=1117, DF003=0, DF428=385820000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=7061644215716937728, NSat=3, DF395=1073774849, NSig=4, DF396=4095, NCell=12, PRN_01=194, PRN_02=195, PRN_03=199, DF397_01=140, DF397_02=112, DF397_03=133, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, DF398_01=0.171875, DF398_02=0.125, DF398_03=0.8720703125, DF399_01=481, DF399_02=-28, DF399_03=1, CELLPRN_01=194, CELLSIG_01=1C, CELLPRN_02=194, CELLSIG_02=2X, CELLPRN_03=194, CELLSIG_03=5X, CELLPRN_04=194, CELLSIG_04=1X, CELLPRN_05=195, CELLSIG_05=1C, CELLPRN_06=195, CELLSIG_06=2X, CELLPRN_07=195, CELLSIG_07=5X, CELLPRN_08=195, CELLSIG_08=1X, CELLPRN_09=199, CELLSIG_09=1C, CELLPRN_10=199, CELLSIG_10=2X, CELLPRN_11=199, CELLSIG_11=5X, CELLPRN_12=199, CELLSIG_12=1X, DF405_01=0.00022634491324424744, DF405_02=0.00026302412152290344, DF405_03=0.0002453848719596863, DF405_04=0.00023026764392852783, DF405_05=-3.849714994430542e-05, DF405_06=-8.327886462211609e-06, DF405_07=-2.2586435079574585e-05, DF405_08=-3.288686275482178e-05, DF405_09=-9.805895388126373e-05, DF405_10=-7.162615656852722e-05, DF405_11=-8.577480912208557e-05, DF405_12=-9.572319686412811e-05, DF406_01=-0.000219784677028656, DF406_02=0.000500889029353857, DF406_03=-3.950204700231552e-06, DF406_04=0.0002219858579337597, DF406_05=0.0003901217132806778, DF406_06=-0.00017092609778046608, DF406_07=-0.0001668902114033699, DF406_08=-0.0001881415955722332, DF406_09=0.0002841702662408352, DF406_10=-0.0001959027722477913, DF406_11=0.00020570401102304459, DF406_12=0.00017753522843122482, DF407_01=516, DF407_02=462, DF407_03=684, DF407_04=365, DF407_05=648, DF407_06=649, DF407_07=649, DF407_08=649, DF407_09=568, DF407_10=704, DF407_11=704, DF407_12=704, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF408_01=30.3125, DF408_02=35.875, DF408_03=34.5, DF408_04=34.5, DF408_05=48.625, DF408_06=50.5, DF408_07=52.625, DF408_08=51.1875, DF408_09=38.3125, DF408_10=43.6875, DF408_11=45.8125, DF408_12=40.3125, DF404_01=-0.10640000000000001, DF404_02=-1.6384, DF404_03=-1.6384, DF404_04=-1.6384, DF404_05=-0.5875, DF404_06=-1.6384, DF404_07=-1.6384, DF404_08=-1.6384, DF404_09=-0.41300000000000003, DF404_10=-1.6384, DF404_11=-1.6384, DF404_12=-1.6384)>",
            "<RTCM(1059, DF002=1059, DF385=466480, DF391=2, DF388=0, DF413=1, DF414=3, DF415=1, DF387=30, DF068_01=1, DF379_01=2, DF380_01_01=0, DF383_01_01=0.23, DF380_01_02=10, DF383_01_02=0.62, DF068_02=2, DF379_02=2, DF380_02_01=0, DF383_02_01=2.71, DF380_02_02=10, DF383_02_02=2.27, DF068_03=3, DF379_03=2, DF380_03_01=0, DF383_03_01=-1.72, DF380_03_02=10, DF383_03_02=-1.47, DF068_04=4, DF379_04=2, DF380_04_01=0, DF383_04_01=-0.31, DF380_04_02=10, DF383_04_02=-0.19, DF068_05=5, DF379_05=2, DF380_05_01=0, DF383_05_01=0.65, DF380_05_02=10, DF383_05_02=0.93, DF068_06=6, DF379_06=2, DF380_06_01=0, DF383_06_01=-2.05, DF380_06_02=10, DF383_06_02=-1.92, DF068_07=7, DF379_07=2, DF380_07_01=0, DF383_07_01=0.93, DF380_07_02=10, DF383_07_02=1.0, DF068_08=8, DF379_08=2, DF380_08_01=0, DF383_08_01=-1.82, DF380_08_02=10, DF383_08_02=-2.05, DF068_09=9, DF379_09=2, DF380_09_01=0, DF383_09_01=-1.1400000000000001, DF380_09_02=10, DF383_09_02=-1.23, DF068_10=10, DF379_10=2, DF380_10_01=0, DF383_10_01=-1.72, DF380_10_02=10, DF383_10_02=-1.68, DF068_11=11, DF379_11=2, DF380_11_01=0, DF383_11_01=0.4, DF380_11_02=10, DF383_11_02=0.5700000000000001, DF068_12=12, DF379_12=2, DF380_12_01=0, DF383_12_01=1.05, DF380_12_02=10, DF383_12_02=1.32, DF068_13=13, DF379_13=2, DF380_13_01=0, DF383_13_01=0.98, DF380_13_02=10, DF383_13_02=1.06, DF068_14=14, DF379_14=2, DF380_14_01=0, DF383_14_01=0.28, DF380_14_02=10, DF383_14_02=0.45, DF068_15=15, DF379_15=2, DF380_15_01=0, DF383_15_01=0.55, DF380_15_02=10, DF383_15_02=0.91, DF068_16=16, DF379_16=2, DF380_16_01=0, DF383_16_01=1.3900000000000001, DF380_16_02=10, DF383_16_02=0.93, DF068_17=17, DF379_17=2, DF380_17_01=0, DF383_17_01=0.86, DF380_17_02=10, DF383_17_02=1.06, DF068_18=18, DF379_18=2, DF380_18_01=0, DF383_18_01=0.31, DF380_18_02=10, DF383_18_02=0.53, DF068_19=19, DF379_19=2, DF380_19_01=0, DF383_19_01=2.56, DF380_19_02=10, DF383_19_02=1.92, DF068_20=20, DF379_20=2, DF380_20_01=0, DF383_20_01=1.1500000000000001, DF380_20_02=10, DF383_20_02=0.49, DF068_21=22, DF379_21=2, DF380_21_01=0, DF383_21_01=1.26, DF380_21_02=10, DF383_21_02=1.03, DF068_22=23, DF379_22=2, DF380_22_01=0, DF383_22_01=0.31, DF380_22_02=10, DF383_22_02=0.54, DF068_23=24, DF379_23=2, DF380_23_01=0, DF383_23_01=-1.87, DF380_23_02=10, DF383_23_02=-1.57, DF068_24=25, DF379_24=2, DF380_24_01=0, DF383_24_01=-1.92, DF380_24_02=10, DF383_24_02=-2.17, DF068_25=26, DF379_25=2, DF380_25_01=0, DF383_25_01=-2.36, DF380_25_02=10, DF383_25_02=-2.41, DF068_26=27, DF379_26=2, DF380_26_01=0, DF383_26_01=-1.52, DF380_26_02=10, DF383_26_02=-1.54, DF068_27=28, DF379_27=2, DF380_27_01=0, DF383_27_01=0.63, DF380_27_02=10, DF383_27_02=0.75, DF068_28=29, DF379_28=2, DF380_28_01=0, DF383_28_01=0.52, DF380_28_02=10, DF383_28_02=0.8300000000000001, DF068_29=31, DF379_29=2, DF380_29_01=0, DF383_29_01=1.26, DF380_29_02=10, DF383_29_02=1.46, DF068_30=32, DF379_30=2, DF380_30_01=0, DF383_30_01=-1.26, DF380_30_02=10, DF383_30_02=-1.3)>",
            "<RTCM(1060, DF002=1060, DF385=466485, DF391=2, DF388=0, DF375=0, DF413=1, DF414=3, DF415=1, DF387=30, DF068_01=1, DF071_01=99, DF365_01=-1040.3, DF366_01=1451.6000000000001, DF367_01=541.2, DF368_01=-0.251, DF369_01=-0.188, DF370_01=0.076, DF376_01=157.20000000000002, DF377_01=0.0, DF378_01=0.0, DF068_02=2, DF071_02=60, DF365_02=-120.9, DF366_02=2483.2000000000003, DF367_02=1154.4, DF368_02=-0.34400000000000003, DF369_02=0.248, DF370_02=0.28, DF376_02=-4896.7, DF377_02=0.0, DF378_02=0.0, DF068_03=3, DF071_03=12, DF365_03=97.5, DF366_03=-60.0, DF367_03=-1040.4, DF368_03=-0.045, DF369_03=-0.108, DF370_03=0.128, DF376_03=555.5, DF377_03=0.0, DF378_03=0.0, DF068_04=4, DF071_04=43, DF365_04=-576.9, DF366_04=-571.6, DF367_04=-583.2, DF368_04=0.041, DF369_04=-0.108, DF370_04=-0.056, DF376_04=-842.5, DF377_04=0.0, DF378_04=0.0, DF068_05=5, DF071_05=23, DF365_05=-75.2, DF366_05=206.0, DF367_05=224.0, DF368_05=-0.073, DF369_05=0.148, DF370_05=-0.088, DF376_05=-967.0, DF377_05=0.0, DF378_05=0.0, DF068_06=6, DF071_06=65, DF365_06=-43.900000000000006, DF366_06=1134.0, DF367_06=94.0, DF368_06=-0.027, DF369_06=-0.044, DF370_06=0.088, DF376_06=466.70000000000005, DF377_06=0.0, DF378_06=0.0, DF068_07=7, DF071_07=9, DF365_07=409.0, DF366_07=27.6, DF367_07=-652.8000000000001, DF368_07=0.012, DF369_07=-0.464, DF370_07=0.256, DF376_07=-2019.3000000000002, DF377_07=0.0, DF378_07=0.0, DF068_08=8, DF071_08=187, DF365_08=33.1, DF366_08=1139.6000000000001, DF367_08=57.6, DF368_08=-0.12, DF369_08=-0.02, DF370_08=0.264, DF376_08=8.3, DF377_08=0.0, DF378_08=0.0, DF068_09=9, DF071_09=103, DF365_09=412.5, DF366_09=-825.6, DF367_09=734.4000000000001, DF368_09=0.1, DF369_09=-0.24, DF370_09=0.016, DF376_09=1188.6000000000001, DF377_09=0.0, DF378_09=0.0, DF068_10=10, DF071_10=36, DF365_10=-113.0, DF366_10=380.0, DF367_10=22.0, DF368_10=-0.024, DF369_10=0.052000000000000005, DF370_10=0.128, DF376_10=101.0, DF377_10=0.0, DF378_10=0.0, DF068_11=11, DF071_11=213, DF365_11=-890.7, DF366_11=699.6, DF367_11=-357.6, DF368_11=-0.176, DF369_11=-0.14, DF370_11=0.028, DF376_11=-1523.2, DF377_11=0.0, DF378_11=0.0, DF068_12=12, DF071_12=22, DF365_12=-60.5, DF366_12=200.8, DF367_12=119.2, DF368_12=-0.09, DF369_12=-0.116, DF370_12=-0.1, DF376_12=-1996.5, DF377_12=0.0, DF378_12=0.0, DF068_13=13, DF071_13=48, DF365_13=-38.900000000000006, DF366_13=-43.2, DF367_13=-838.8000000000001, DF368_13=-0.073, DF369_13=0.004, DF370_13=0.108, DF376_13=-1263.9, DF377_13=0.0, DF378_13=0.0, DF068_14=14, DF071_14=115, DF365_14=-750.4000000000001, DF366_14=793.6, DF367_14=484.40000000000003, DF368_14=-0.109, DF369_14=-0.468, DF370_14=-0.28, DF376_14=-1075.2, DF377_14=0.0, DF378_14=0.0, DF068_15=15, DF071_15=78, DF365_15=30.200000000000003, DF366_15=-1284.4, DF367_15=-942.0, DF368_15=0.115, DF369_15=0.044, DF370_15=0.052000000000000005, DF376_15=-778.3000000000001, DF377_15=0.0, DF378_15=0.0, DF068_16=16, DF071_16=5, DF365_16=-12.700000000000001, DF366_16=835.2, DF367_16=-676.0, DF368_16=-0.07200000000000001, DF369_16=0.02, DF370_16=-0.008, DF376_16=-2940.9, DF377_16=0.0, DF378_16=0.0, DF068_17=17, DF071_17=43, DF365_17=-83.7, DF366_17=924.8000000000001, DF367_17=346.0, DF368_17=-0.067, DF369_17=-0.116, DF370_17=0.02, DF376_17=-2035.7, DF377_17=0.0, DF378_17=0.0, DF068_18=18, DF071_18=177, DF365_18=-973.4000000000001, DF366_18=39.2, DF367_18=-464.8, DF368_18=-0.095, DF369_18=-0.16, DF370_18=-0.1, DF376_18=-650.4000000000001, DF377_18=0.0, DF378_18=0.0, DF068_19=19, DF071_19=28, DF365_19=-55.800000000000004, DF366_19=1466.8000000000002, DF367_19=709.6, DF368_19=-0.185, DF369_19=-0.24, DF370_19=-0.116, DF376_19=-3958.8, DF377_19=0.0, DF378_19=0.0, DF068_20=20, DF071_20=13, DF365_20=-63.0, DF366_20=63.2, DF367_20=-510.8, DF368_20=-0.027, DF369_20=-0.132, DF370_20=0.008, DF376_20=-2539.7000000000003, DF377_20=0.0, DF378_20=0.0, DF068_21=22, DF071_21=25, DF365_21=34.5, DF366_21=890.4000000000001, DF367_21=-823.6, DF368_21=-0.097, DF369_21=-0.128, DF370_21=-0.08, DF376_21=-2546.6000000000004, DF377_21=0.0, DF378_21=0.0, DF068_22=23, DF071_22=180, DF365_22=-655.3000000000001, DF366_22=-971.6, DF367_22=-117.2, DF368_22=0.077, DF369_22=-0.132, DF370_22=-0.024, DF376_22=-616.7, DF377_22=0.0, DF378_22=0.0, DF068_23=24, DF071_23=126, DF365_23=-680.9000000000001, DF366_23=175.20000000000002, DF367_23=-85.2, DF368_23=0.105, DF369_23=-0.124, DF370_23=-0.108, DF376_23=2253.0, DF377_23=0.0, DF378_23=0.0, DF068_24=25, DF071_24=21, DF365_24=-47.2, DF366_24=99.60000000000001, DF367_24=36.4, DF368_24=-0.07100000000000001, DF369_24=-0.268, DF370_24=-0.064, DF376_24=546.7, DF377_24=0.0, DF378_24=0.0, DF068_25=26, DF071_25=70, DF365_25=-46.800000000000004, DF366_25=1472.8000000000002, DF367_25=-284.8, DF368_25=-0.185, DF369_25=-0.14400000000000002, DF370_25=-0.088, DF376_25=578.5, DF377_25=0.0, DF378_25=0.0, DF068_26=27, DF071_26=33, DF365_26=-201.9, DF366_26=626.0, DF367_26=-18.0, DF368_26=-0.115, DF369_26=0.1, DF370_26=0.048, DF376_26=-519.6, DF377_26=0.0, DF378_26=0.0, DF068_27=28, DF071_27=224, DF365_27=-560.3000000000001, DF366_27=70.8, DF367_27=606.4, DF368_27=0.006, DF369_27=-0.456, DF370_27=0.036000000000000004, DF376_27=-1146.6000000000001, DF377_27=0.0, DF378_27=0.0, DF068_28=29, DF071_28=123, DF365_28=-115.5, DF366_28=-397.6, DF367_28=76.4, DF368_28=0.069, DF369_28=-0.192, DF370_28=-0.164, DF376_28=-1773.6000000000001, DF377_28=0.0, DF378_28=0.0, DF068_29=31, DF071_29=7, DF365_29=-21.0, DF366_29=168.4, DF367_29=171.20000000000002, DF368_29=0.078, DF369_29=-0.12, DF370_29=0.024, DF376_29=-2145.3, DF377_29=0.0, DF378_29=0.0, DF068_30=32, DF071_30=88, DF365_30=-288.7, DF366_30=52.400000000000006, DF367_30=-239.60000000000002, DF368_30=-0.066, DF369_30=0.068, DF370_30=0.044, DF376_30=888.0, DF377_30=0.0, DF378_30=0.0)>",
        )
        dirname = os.path.dirname(__file__)
        with open(os.path.join(dirname, "pygpsdata-RTCM3.log"), "rb") as stream:
            i = 0
            raw = 0
            rtr = RTCMReader(stream)
            for raw, parsed in rtr:
                if raw is not None:
                    # print(f'"{parsed}",')
                    self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                    i += 1
        self.assertEqual(i, 11)

    def testMIXEDRTCM_SCALE_LABELMSM_RINEX(
        self,
    ):  # test mixed stream of NMEA, UBX & RTCM messages with scaling and RINEX labelmsm applied
        EXPECTED_RESULTS = (
            "<RTCM(1005, DF002=1005, DF003=0, DF021=0, DF022=1, DF023=1, DF024=1, DF141=0, DF025=4444030.802800001, DF142=1, DF001_1=0, DF026=3085671.2349, DF364=0, DF027=3366658.256)>",
            "<RTCM(4072, DF002=4072, Not_Yet_Implemented)>",
            "<RTCM(1077, DF002=1077, DF003=0, DF004=204137001, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=760738918298550272, NSat=10, DF395=1073807360, NSig=2, DF396=1044459, NCell=17, PRN_01=005, PRN_02=007, PRN_03=009, PRN_04=013, PRN_05=014, PRN_06=015, PRN_07=017, PRN_08=019, PRN_09=020, PRN_10=030, DF397_01=75, DF397_02=75, DF397_03=81, DF397_04=72, DF397_05=67, DF397_06=80, DF397_07=75, DF397_08=82, DF397_09=75, DF397_10=71, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, ExtSatInfo_06=0, ExtSatInfo_07=0, ExtSatInfo_08=0, ExtSatInfo_09=0, ExtSatInfo_10=0, DF398_01=0.005859375, DF398_02=0.5341796875, DF398_03=0.7626953125, DF398_04=0.138671875, DF398_05=0.5498046875, DF398_06=0.11328125, DF398_07=0.8037109375, DF398_08=0.1025390625, DF398_09=0.521484375, DF398_10=0.345703125, DF399_01=-178, DF399_02=-304, DF399_03=-643, DF399_04=477, DF399_05=-52, DF399_06=645, DF399_07=529, DF399_08=643, DF399_09=-428, DF399_10=-181, CELLPRN_01=005, CELLSIG_01=1C, CELLPRN_02=005, CELLSIG_02=2L, CELLPRN_03=007, CELLSIG_03=1C, CELLPRN_04=007, CELLSIG_04=2L, CELLPRN_05=009, CELLSIG_05=1C, CELLPRN_06=009, CELLSIG_06=2L, CELLPRN_07=013, CELLSIG_07=1C, CELLPRN_08=014, CELLSIG_08=1C, CELLPRN_09=014, CELLSIG_09=2L, CELLPRN_10=015, CELLSIG_10=1C, CELLPRN_11=015, CELLSIG_11=2L, CELLPRN_12=017, CELLSIG_12=1C, CELLPRN_13=017, CELLSIG_13=2L, CELLPRN_14=019, CELLSIG_14=1C, CELLPRN_15=020, CELLSIG_15=1C, CELLPRN_16=030, CELLSIG_16=1C, CELLPRN_17=030, CELLSIG_17=2L, DF405_01=0.00014309026300907135, DF405_02=0.00014183297753334045, DF405_03=0.0003883279860019684, DF405_04=0.00038741156458854675, DF405_05=-0.0004838351160287857, DF405_06=-0.00046883709728717804, DF405_07=0.0003478657454252243, DF405_08=0.0002196934074163437, DF405_09=0.00021521002054214478, DF405_10=-0.00018852390348911285, DF405_11=-0.00018319115042686462, DF405_12=-0.00010087713599205017, DF405_13=-9.844452142715454e-05, DF405_14=0.00047875382006168365, DF405_15=0.00043664872646331787, DF405_16=-0.0003105681389570236, DF405_17=-0.00030865520238876343, DF406_01=0.00014193402603268623, DF406_02=0.00014339853078126907, DF406_03=0.00039040297269821167, DF406_04=0.00038743019104003906, DF406_05=-0.0004843934439122677, DF406_06=-0.00046825408935546875, DF406_07=0.0003473707474768162, DF406_08=0.00021758908405900002, DF406_09=0.00021597417071461678, DF406_10=-0.00018658116459846497, DF406_11=-0.00018350128084421158, DF406_12=-9.993184357881546e-05, DF406_13=-9.724870324134827e-05, DF406_14=0.0004128236323595047, DF406_15=0.0004355977289378643, DF406_16=-0.0003112703561782837, DF406_17=-0.00030898721888661385, DF407_01=341, DF407_02=341, DF407_03=341, DF407_04=341, DF407_05=341, DF407_06=341, DF407_07=341, DF407_08=341, DF407_09=341, DF407_10=341, DF407_11=341, DF407_12=341, DF407_13=341, DF407_14=295, DF407_15=341, DF407_16=341, DF407_17=341, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF408_01=45.0, DF408_02=38.0, DF408_03=43.0, DF408_04=39.0, DF408_05=39.0, DF408_06=37.0, DF408_07=45.0, DF408_08=46.0, DF408_09=46.0, DF408_10=39.0, DF408_11=34.0, DF408_12=45.0, DF408_13=38.0, DF408_14=31.0, DF408_15=45.0, DF408_16=46.0, DF408_17=41.0, DF404_01=-0.9231, DF404_02=-0.9194, DF404_03=-0.8321000000000001, DF404_04=-0.8326, DF404_05=-0.4107, DF404_06=-0.4072, DF404_07=0.2451, DF404_08=-0.0693, DF404_09=-0.0684, DF404_10=0.9390000000000001, DF404_11=0.9417000000000001, DF404_12=0.2384, DF404_13=0.2416, DF404_14=0.6636000000000001, DF404_15=-0.9556, DF404_16=-0.21480000000000002, DF404_17=-0.2174)>",
            "<RTCM(1087, DF002=1087, DF003=0, DF416=2, DF034=42119001, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=4039168114821169152, NSat=7, DF395=1090519040, NSig=2, DF396=16382, NCell=13, PRN_01=003, PRN_02=004, PRN_03=005, PRN_04=013, PRN_05=014, PRN_06=015, PRN_07=023, DF397_01=69, DF397_02=64, DF397_03=73, DF397_04=76, DF397_05=66, DF397_06=70, DF397_07=78, DF419_01=12, DF419_02=13, DF419_03=8, DF419_04=5, DF419_05=0, DF419_06=7, DF419_07=10, DF398_01=0.6337890625, DF398_02=0.3427734375, DF398_03=0.25390625, DF398_04=0.310546875, DF398_05=0.5126953125, DF398_06=0.8271484375, DF398_07=0.8837890625, DF399_01=-665, DF399_02=29, DF399_03=672, DF399_04=-573, DF399_05=-211, DF399_06=312, DF399_07=317, CELLPRN_01=003, CELLSIG_01=1C, CELLPRN_02=003, CELLSIG_02=2C, CELLPRN_03=004, CELLSIG_03=1C, CELLPRN_04=004, CELLSIG_04=2C, CELLPRN_05=005, CELLSIG_05=1C, CELLPRN_06=005, CELLSIG_06=2C, CELLPRN_07=013, CELLSIG_07=1C, CELLPRN_08=013, CELLSIG_08=2C, CELLPRN_09=014, CELLSIG_09=1C, CELLPRN_10=014, CELLSIG_10=2C, CELLPRN_11=015, CELLSIG_11=1C, CELLPRN_12=015, CELLSIG_12=2C, CELLPRN_13=023, CELLSIG_13=1C, DF405_01=0.00024936161935329437, DF405_02=0.0002511627972126007, DF405_03=-4.678964614868164e-05, DF405_04=-5.141831934452057e-05, DF405_05=1.1144205927848816e-05, DF405_06=2.15042382478714e-05, DF405_07=0.00047079287469387054, DF405_08=0.0004794951528310776, DF405_09=-0.0003879182040691376, DF405_10=-0.00037603825330734253, DF405_11=0.0002771839499473572, DF405_12=0.0002871435135602951, DF405_13=-0.00023611821234226227, DF406_01=0.00024937279522418976, DF406_02=0.00025077443569898605, DF406_03=-4.834495484828949e-05, DF406_04=-5.1246024668216705e-05, DF406_05=1.1149328202009201e-05, DF406_06=2.1803192794322968e-05, DF406_07=0.00047026341781020164, DF406_08=0.0004848274402320385, DF406_09=-0.0003876127302646637, DF406_10=-0.0003757951781153679, DF406_11=0.0002778824418783188, DF406_12=0.0002880701795220375, DF406_13=-0.00023698341101408005, DF407_01=341, DF407_02=341, DF407_03=340, DF407_04=340, DF407_05=341, DF407_06=341, DF407_07=340, DF407_08=341, DF407_09=341, DF407_10=341, DF407_11=341, DF407_12=341, DF407_13=340, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF408_01=47.0, DF408_02=40.0, DF408_03=47.0, DF408_04=42.0, DF408_05=47.0, DF408_06=39.0, DF408_07=36.0, DF408_08=33.0, DF408_09=48.0, DF408_10=43.0, DF408_11=48.0, DF408_12=40.0, DF408_13=41.0, DF404_01=-0.8193, DF404_02=-0.8173, DF404_03=0.8539, DF404_04=0.8501000000000001, DF404_05=0.7333000000000001, DF404_06=0.7311000000000001, DF404_07=-0.24930000000000002, DF404_08=-0.2543, DF404_09=-0.21580000000000002, DF404_10=-0.21780000000000002, DF404_11=0.3924, DF404_12=0.3947, DF404_13=0.6146)>",
            "<RTCM(1097, DF002=1097, DF003=0, DF248=204137001, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=216181732825628672, NSat=5, DF395=1073872896, NSig=2, DF396=1023, NCell=10, PRN_01=007, PRN_02=008, PRN_03=021, PRN_04=027, PRN_05=030, DF397_01=79, DF397_02=84, DF397_03=89, DF397_04=78, DF397_05=83, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, DF398_01=0.15625, DF398_02=0.2509765625, DF398_03=0.3544921875, DF398_04=0.37109375, DF398_05=0.259765625, DF399_01=-198, DF399_02=-516, DF399_03=423, DF399_04=63, DF399_05=-384, CELLPRN_01=007, CELLSIG_01=1C, CELLPRN_02=007, CELLSIG_02=7Q, CELLPRN_03=008, CELLSIG_03=1C, CELLPRN_04=008, CELLSIG_04=7Q, CELLPRN_05=021, CELLSIG_05=1C, CELLPRN_06=021, CELLSIG_06=7Q, CELLPRN_07=027, CELLSIG_07=1C, CELLPRN_08=027, CELLSIG_08=7Q, CELLPRN_09=030, CELLSIG_09=1C, CELLPRN_10=030, CELLSIG_10=7Q, DF405_01=-4.5398250222206116e-05, DF405_02=-2.8252601623535156e-05, DF405_03=-0.00034597329795360565, DF405_04=-0.0003268253058195114, DF405_05=0.0004809703677892685, DF405_06=0.0005012489855289459, DF405_07=-0.00013696029782295227, DF405_08=-0.0001260414719581604, DF405_09=-1.8440186977386475e-05, DF405_10=-3.041699528694153e-06, DF406_01=-4.44464385509491e-05, DF406_02=-2.835458144545555e-05, DF406_03=-0.0003525479696691036, DF406_04=-0.0003263736143708229, DF406_05=0.00048203859478235245, DF406_06=0.0005008447915315628, DF406_07=-0.0001375703141093254, DF406_08=-0.00012635625898838043, DF406_09=-1.8037855625152588e-05, DF406_10=-3.2926909625530243e-06, DF407_01=341, DF407_02=341, DF407_03=341, DF407_04=341, DF407_05=341, DF407_06=341, DF407_07=341, DF407_08=341, DF407_09=341, DF407_10=341, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF408_01=46.0, DF408_02=49.0, DF408_03=41.0, DF408_04=43.0, DF408_05=43.0, DF408_06=43.0, DF408_07=45.0, DF408_08=49.0, DF408_09=43.0, DF408_10=47.0, DF404_01=-0.5806, DF404_02=-0.5831000000000001, DF404_03=-0.7947000000000001, DF404_04=-0.7943, DF404_05=0.7243, DF404_06=0.7174, DF404_07=0.5534, DF404_08=0.5545, DF404_09=-0.7726000000000001, DF404_10=-0.7733)>",
            "<RTCM(1127, DF002=1127, DF003=0, DF427=204123001, DF393=0, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=198178247981137920, NSat=10, DF395=1074003968, NSig=2, DF396=387754, NCell=11, PRN_01=007, PRN_02=009, PRN_03=010, PRN_04=020, PRN_05=023, PRN_06=028, PRN_07=032, PRN_08=037, PRN_09=040, PRN_10=043, DF397_01=129, DF397_02=132, DF397_03=126, DF397_04=75, DF397_05=81, DF397_06=84, DF397_07=78, DF397_08=74, DF397_09=130, DF397_10=86, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, ExtSatInfo_06=0, ExtSatInfo_07=0, ExtSatInfo_08=0, ExtSatInfo_09=0, ExtSatInfo_10=0, DF398_01=0.1171875, DF398_02=0.4814453125, DF398_03=0.3095703125, DF398_04=0.7255859375, DF398_05=0.41015625, DF398_06=0.5703125, DF398_07=0.5595703125, DF398_08=0.322265625, DF398_09=0.578125, DF398_10=0.673828125, DF399_01=-130, DF399_02=-58, DF399_03=-81, DF399_04=32, DF399_05=-398, DF399_06=436, DF399_07=-523, DF399_08=-65, DF399_09=-182, DF399_10=79, CELLPRN_01=007, CELLSIG_01=7I, CELLPRN_02=009, CELLSIG_02=7I, CELLPRN_03=010, CELLSIG_03=2I, CELLPRN_04=010, CELLSIG_04=7I, CELLPRN_05=020, CELLSIG_05=2I, CELLPRN_06=023, CELLSIG_06=2I, CELLPRN_07=028, CELLSIG_07=2I, CELLPRN_08=032, CELLSIG_08=2I, CELLPRN_09=037, CELLSIG_09=2I, CELLPRN_10=040, CELLSIG_10=2I, CELLPRN_11=043, CELLSIG_11=2I, DF405_01=-0.0003885403275489807, DF405_02=0.00022730417549610138, DF405_03=0.0004036612808704376, DF405_04=0.00039606913924217224, DF405_05=-0.00016684085130691528, DF405_06=-4.75514680147171e-05, DF405_07=0.0003674682229757309, DF405_08=0.00026629865169525146, DF405_09=-0.0002502594143152237, DF405_10=-0.00011803768575191498, DF405_11=-0.0002937670797109604, DF406_01=-0.0003882073797285557, DF406_02=0.0002264929935336113, DF406_03=0.0004031979478895664, DF406_04=0.0003964221104979515, DF406_05=-0.00016694329679012299, DF406_06=-4.848744720220566e-05, DF406_07=0.00036971503868699074, DF406_08=0.0002654106356203556, DF406_09=-0.00025115441530942917, DF406_10=-0.00011868216097354889, DF406_11=-0.00029495684430003166, DF407_01=341, DF407_02=341, DF407_03=341, DF407_04=341, DF407_05=341, DF407_06=341, DF407_07=341, DF407_08=341, DF407_09=341, DF407_10=341, DF407_11=341, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF408_01=45.0, DF408_02=41.0, DF408_03=42.0, DF408_04=45.0, DF408_05=48.0, DF408_06=46.0, DF408_07=42.0, DF408_08=47.0, DF408_09=48.0, DF408_10=44.0, DF408_11=43.0, DF404_01=-0.5674, DF404_02=-0.612, DF404_03=-0.1384, DF404_04=-0.1332, DF404_05=0.5992000000000001, DF404_06=-0.7312000000000001, DF404_07=0.17320000000000002, DF404_08=-0.4308, DF404_09=-0.5975, DF404_10=-0.6733, DF404_11=0.6122000000000001)>",
            "<RTCM(1230, DF002=1230, DF003=0, DF421=1, DF001_3=0, DF422_1=0, DF422_2=0, DF422_3=0, DF422_4=0)>",
            "<RTCM(1007, DF002=1007, DF003=1234, DF029=3, DF030_01=A, DF030_02=B, DF030_03=C, DF031=234)>",
            "<RTCM(1117, DF002=1117, DF003=0, DF428=385820000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=7061644215716937728, NSat=3, DF395=1073774849, NSig=4, DF396=4095, NCell=12, PRN_01=194, PRN_02=195, PRN_03=199, DF397_01=140, DF397_02=112, DF397_03=133, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, DF398_01=0.171875, DF398_02=0.125, DF398_03=0.8720703125, DF399_01=481, DF399_02=-28, DF399_03=1, CELLPRN_01=194, CELLSIG_01=1C, CELLPRN_02=194, CELLSIG_02=2X, CELLPRN_03=194, CELLSIG_03=5X, CELLPRN_04=194, CELLSIG_04=1X, CELLPRN_05=195, CELLSIG_05=1C, CELLPRN_06=195, CELLSIG_06=2X, CELLPRN_07=195, CELLSIG_07=5X, CELLPRN_08=195, CELLSIG_08=1X, CELLPRN_09=199, CELLSIG_09=1C, CELLPRN_10=199, CELLSIG_10=2X, CELLPRN_11=199, CELLSIG_11=5X, CELLPRN_12=199, CELLSIG_12=1X, DF405_01=0.00022634491324424744, DF405_02=0.00026302412152290344, DF405_03=0.0002453848719596863, DF405_04=0.00023026764392852783, DF405_05=-3.849714994430542e-05, DF405_06=-8.327886462211609e-06, DF405_07=-2.2586435079574585e-05, DF405_08=-3.288686275482178e-05, DF405_09=-9.805895388126373e-05, DF405_10=-7.162615656852722e-05, DF405_11=-8.577480912208557e-05, DF405_12=-9.572319686412811e-05, DF406_01=-0.000219784677028656, DF406_02=0.000500889029353857, DF406_03=-3.950204700231552e-06, DF406_04=0.0002219858579337597, DF406_05=0.0003901217132806778, DF406_06=-0.00017092609778046608, DF406_07=-0.0001668902114033699, DF406_08=-0.0001881415955722332, DF406_09=0.0002841702662408352, DF406_10=-0.0001959027722477913, DF406_11=0.00020570401102304459, DF406_12=0.00017753522843122482, DF407_01=516, DF407_02=462, DF407_03=684, DF407_04=365, DF407_05=648, DF407_06=649, DF407_07=649, DF407_08=649, DF407_09=568, DF407_10=704, DF407_11=704, DF407_12=704, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF408_01=30.3125, DF408_02=35.875, DF408_03=34.5, DF408_04=34.5, DF408_05=48.625, DF408_06=50.5, DF408_07=52.625, DF408_08=51.1875, DF408_09=38.3125, DF408_10=43.6875, DF408_11=45.8125, DF408_12=40.3125, DF404_01=-0.10640000000000001, DF404_02=-1.6384, DF404_03=-1.6384, DF404_04=-1.6384, DF404_05=-0.5875, DF404_06=-1.6384, DF404_07=-1.6384, DF404_08=-1.6384, DF404_09=-0.41300000000000003, DF404_10=-1.6384, DF404_11=-1.6384, DF404_12=-1.6384)>",
            "<RTCM(1059, DF002=1059, DF385=466480, DF391=2, DF388=0, DF413=1, DF414=3, DF415=1, DF387=30, DF068_01=1, DF379_01=2, DF380_01_01=0, DF383_01_01=0.23, DF380_01_02=10, DF383_01_02=0.62, DF068_02=2, DF379_02=2, DF380_02_01=0, DF383_02_01=2.71, DF380_02_02=10, DF383_02_02=2.27, DF068_03=3, DF379_03=2, DF380_03_01=0, DF383_03_01=-1.72, DF380_03_02=10, DF383_03_02=-1.47, DF068_04=4, DF379_04=2, DF380_04_01=0, DF383_04_01=-0.31, DF380_04_02=10, DF383_04_02=-0.19, DF068_05=5, DF379_05=2, DF380_05_01=0, DF383_05_01=0.65, DF380_05_02=10, DF383_05_02=0.93, DF068_06=6, DF379_06=2, DF380_06_01=0, DF383_06_01=-2.05, DF380_06_02=10, DF383_06_02=-1.92, DF068_07=7, DF379_07=2, DF380_07_01=0, DF383_07_01=0.93, DF380_07_02=10, DF383_07_02=1.0, DF068_08=8, DF379_08=2, DF380_08_01=0, DF383_08_01=-1.82, DF380_08_02=10, DF383_08_02=-2.05, DF068_09=9, DF379_09=2, DF380_09_01=0, DF383_09_01=-1.1400000000000001, DF380_09_02=10, DF383_09_02=-1.23, DF068_10=10, DF379_10=2, DF380_10_01=0, DF383_10_01=-1.72, DF380_10_02=10, DF383_10_02=-1.68, DF068_11=11, DF379_11=2, DF380_11_01=0, DF383_11_01=0.4, DF380_11_02=10, DF383_11_02=0.5700000000000001, DF068_12=12, DF379_12=2, DF380_12_01=0, DF383_12_01=1.05, DF380_12_02=10, DF383_12_02=1.32, DF068_13=13, DF379_13=2, DF380_13_01=0, DF383_13_01=0.98, DF380_13_02=10, DF383_13_02=1.06, DF068_14=14, DF379_14=2, DF380_14_01=0, DF383_14_01=0.28, DF380_14_02=10, DF383_14_02=0.45, DF068_15=15, DF379_15=2, DF380_15_01=0, DF383_15_01=0.55, DF380_15_02=10, DF383_15_02=0.91, DF068_16=16, DF379_16=2, DF380_16_01=0, DF383_16_01=1.3900000000000001, DF380_16_02=10, DF383_16_02=0.93, DF068_17=17, DF379_17=2, DF380_17_01=0, DF383_17_01=0.86, DF380_17_02=10, DF383_17_02=1.06, DF068_18=18, DF379_18=2, DF380_18_01=0, DF383_18_01=0.31, DF380_18_02=10, DF383_18_02=0.53, DF068_19=19, DF379_19=2, DF380_19_01=0, DF383_19_01=2.56, DF380_19_02=10, DF383_19_02=1.92, DF068_20=20, DF379_20=2, DF380_20_01=0, DF383_20_01=1.1500000000000001, DF380_20_02=10, DF383_20_02=0.49, DF068_21=22, DF379_21=2, DF380_21_01=0, DF383_21_01=1.26, DF380_21_02=10, DF383_21_02=1.03, DF068_22=23, DF379_22=2, DF380_22_01=0, DF383_22_01=0.31, DF380_22_02=10, DF383_22_02=0.54, DF068_23=24, DF379_23=2, DF380_23_01=0, DF383_23_01=-1.87, DF380_23_02=10, DF383_23_02=-1.57, DF068_24=25, DF379_24=2, DF380_24_01=0, DF383_24_01=-1.92, DF380_24_02=10, DF383_24_02=-2.17, DF068_25=26, DF379_25=2, DF380_25_01=0, DF383_25_01=-2.36, DF380_25_02=10, DF383_25_02=-2.41, DF068_26=27, DF379_26=2, DF380_26_01=0, DF383_26_01=-1.52, DF380_26_02=10, DF383_26_02=-1.54, DF068_27=28, DF379_27=2, DF380_27_01=0, DF383_27_01=0.63, DF380_27_02=10, DF383_27_02=0.75, DF068_28=29, DF379_28=2, DF380_28_01=0, DF383_28_01=0.52, DF380_28_02=10, DF383_28_02=0.8300000000000001, DF068_29=31, DF379_29=2, DF380_29_01=0, DF383_29_01=1.26, DF380_29_02=10, DF383_29_02=1.46, DF068_30=32, DF379_30=2, DF380_30_01=0, DF383_30_01=-1.26, DF380_30_02=10, DF383_30_02=-1.3)>",
            "<RTCM(1060, DF002=1060, DF385=466485, DF391=2, DF388=0, DF375=0, DF413=1, DF414=3, DF415=1, DF387=30, DF068_01=1, DF071_01=99, DF365_01=-1040.3, DF366_01=1451.6000000000001, DF367_01=541.2, DF368_01=-0.251, DF369_01=-0.188, DF370_01=0.076, DF376_01=157.20000000000002, DF377_01=0.0, DF378_01=0.0, DF068_02=2, DF071_02=60, DF365_02=-120.9, DF366_02=2483.2000000000003, DF367_02=1154.4, DF368_02=-0.34400000000000003, DF369_02=0.248, DF370_02=0.28, DF376_02=-4896.7, DF377_02=0.0, DF378_02=0.0, DF068_03=3, DF071_03=12, DF365_03=97.5, DF366_03=-60.0, DF367_03=-1040.4, DF368_03=-0.045, DF369_03=-0.108, DF370_03=0.128, DF376_03=555.5, DF377_03=0.0, DF378_03=0.0, DF068_04=4, DF071_04=43, DF365_04=-576.9, DF366_04=-571.6, DF367_04=-583.2, DF368_04=0.041, DF369_04=-0.108, DF370_04=-0.056, DF376_04=-842.5, DF377_04=0.0, DF378_04=0.0, DF068_05=5, DF071_05=23, DF365_05=-75.2, DF366_05=206.0, DF367_05=224.0, DF368_05=-0.073, DF369_05=0.148, DF370_05=-0.088, DF376_05=-967.0, DF377_05=0.0, DF378_05=0.0, DF068_06=6, DF071_06=65, DF365_06=-43.900000000000006, DF366_06=1134.0, DF367_06=94.0, DF368_06=-0.027, DF369_06=-0.044, DF370_06=0.088, DF376_06=466.70000000000005, DF377_06=0.0, DF378_06=0.0, DF068_07=7, DF071_07=9, DF365_07=409.0, DF366_07=27.6, DF367_07=-652.8000000000001, DF368_07=0.012, DF369_07=-0.464, DF370_07=0.256, DF376_07=-2019.3000000000002, DF377_07=0.0, DF378_07=0.0, DF068_08=8, DF071_08=187, DF365_08=33.1, DF366_08=1139.6000000000001, DF367_08=57.6, DF368_08=-0.12, DF369_08=-0.02, DF370_08=0.264, DF376_08=8.3, DF377_08=0.0, DF378_08=0.0, DF068_09=9, DF071_09=103, DF365_09=412.5, DF366_09=-825.6, DF367_09=734.4000000000001, DF368_09=0.1, DF369_09=-0.24, DF370_09=0.016, DF376_09=1188.6000000000001, DF377_09=0.0, DF378_09=0.0, DF068_10=10, DF071_10=36, DF365_10=-113.0, DF366_10=380.0, DF367_10=22.0, DF368_10=-0.024, DF369_10=0.052000000000000005, DF370_10=0.128, DF376_10=101.0, DF377_10=0.0, DF378_10=0.0, DF068_11=11, DF071_11=213, DF365_11=-890.7, DF366_11=699.6, DF367_11=-357.6, DF368_11=-0.176, DF369_11=-0.14, DF370_11=0.028, DF376_11=-1523.2, DF377_11=0.0, DF378_11=0.0, DF068_12=12, DF071_12=22, DF365_12=-60.5, DF366_12=200.8, DF367_12=119.2, DF368_12=-0.09, DF369_12=-0.116, DF370_12=-0.1, DF376_12=-1996.5, DF377_12=0.0, DF378_12=0.0, DF068_13=13, DF071_13=48, DF365_13=-38.900000000000006, DF366_13=-43.2, DF367_13=-838.8000000000001, DF368_13=-0.073, DF369_13=0.004, DF370_13=0.108, DF376_13=-1263.9, DF377_13=0.0, DF378_13=0.0, DF068_14=14, DF071_14=115, DF365_14=-750.4000000000001, DF366_14=793.6, DF367_14=484.40000000000003, DF368_14=-0.109, DF369_14=-0.468, DF370_14=-0.28, DF376_14=-1075.2, DF377_14=0.0, DF378_14=0.0, DF068_15=15, DF071_15=78, DF365_15=30.200000000000003, DF366_15=-1284.4, DF367_15=-942.0, DF368_15=0.115, DF369_15=0.044, DF370_15=0.052000000000000005, DF376_15=-778.3000000000001, DF377_15=0.0, DF378_15=0.0, DF068_16=16, DF071_16=5, DF365_16=-12.700000000000001, DF366_16=835.2, DF367_16=-676.0, DF368_16=-0.07200000000000001, DF369_16=0.02, DF370_16=-0.008, DF376_16=-2940.9, DF377_16=0.0, DF378_16=0.0, DF068_17=17, DF071_17=43, DF365_17=-83.7, DF366_17=924.8000000000001, DF367_17=346.0, DF368_17=-0.067, DF369_17=-0.116, DF370_17=0.02, DF376_17=-2035.7, DF377_17=0.0, DF378_17=0.0, DF068_18=18, DF071_18=177, DF365_18=-973.4000000000001, DF366_18=39.2, DF367_18=-464.8, DF368_18=-0.095, DF369_18=-0.16, DF370_18=-0.1, DF376_18=-650.4000000000001, DF377_18=0.0, DF378_18=0.0, DF068_19=19, DF071_19=28, DF365_19=-55.800000000000004, DF366_19=1466.8000000000002, DF367_19=709.6, DF368_19=-0.185, DF369_19=-0.24, DF370_19=-0.116, DF376_19=-3958.8, DF377_19=0.0, DF378_19=0.0, DF068_20=20, DF071_20=13, DF365_20=-63.0, DF366_20=63.2, DF367_20=-510.8, DF368_20=-0.027, DF369_20=-0.132, DF370_20=0.008, DF376_20=-2539.7000000000003, DF377_20=0.0, DF378_20=0.0, DF068_21=22, DF071_21=25, DF365_21=34.5, DF366_21=890.4000000000001, DF367_21=-823.6, DF368_21=-0.097, DF369_21=-0.128, DF370_21=-0.08, DF376_21=-2546.6000000000004, DF377_21=0.0, DF378_21=0.0, DF068_22=23, DF071_22=180, DF365_22=-655.3000000000001, DF366_22=-971.6, DF367_22=-117.2, DF368_22=0.077, DF369_22=-0.132, DF370_22=-0.024, DF376_22=-616.7, DF377_22=0.0, DF378_22=0.0, DF068_23=24, DF071_23=126, DF365_23=-680.9000000000001, DF366_23=175.20000000000002, DF367_23=-85.2, DF368_23=0.105, DF369_23=-0.124, DF370_23=-0.108, DF376_23=2253.0, DF377_23=0.0, DF378_23=0.0, DF068_24=25, DF071_24=21, DF365_24=-47.2, DF366_24=99.60000000000001, DF367_24=36.4, DF368_24=-0.07100000000000001, DF369_24=-0.268, DF370_24=-0.064, DF376_24=546.7, DF377_24=0.0, DF378_24=0.0, DF068_25=26, DF071_25=70, DF365_25=-46.800000000000004, DF366_25=1472.8000000000002, DF367_25=-284.8, DF368_25=-0.185, DF369_25=-0.14400000000000002, DF370_25=-0.088, DF376_25=578.5, DF377_25=0.0, DF378_25=0.0, DF068_26=27, DF071_26=33, DF365_26=-201.9, DF366_26=626.0, DF367_26=-18.0, DF368_26=-0.115, DF369_26=0.1, DF370_26=0.048, DF376_26=-519.6, DF377_26=0.0, DF378_26=0.0, DF068_27=28, DF071_27=224, DF365_27=-560.3000000000001, DF366_27=70.8, DF367_27=606.4, DF368_27=0.006, DF369_27=-0.456, DF370_27=0.036000000000000004, DF376_27=-1146.6000000000001, DF377_27=0.0, DF378_27=0.0, DF068_28=29, DF071_28=123, DF365_28=-115.5, DF366_28=-397.6, DF367_28=76.4, DF368_28=0.069, DF369_28=-0.192, DF370_28=-0.164, DF376_28=-1773.6000000000001, DF377_28=0.0, DF378_28=0.0, DF068_29=31, DF071_29=7, DF365_29=-21.0, DF366_29=168.4, DF367_29=171.20000000000002, DF368_29=0.078, DF369_29=-0.12, DF370_29=0.024, DF376_29=-2145.3, DF377_29=0.0, DF378_29=0.0, DF068_30=32, DF071_30=88, DF365_30=-288.7, DF366_30=52.400000000000006, DF367_30=-239.60000000000002, DF368_30=-0.066, DF369_30=0.068, DF370_30=0.044, DF376_30=888.0, DF377_30=0.0, DF378_30=0.0)>",
        )

        dirname = os.path.dirname(__file__)
        with open(os.path.join(dirname, "pygpsdata-RTCM3.log"), "rb") as stream:
            i = 0
            raw = 0
            rtr = RTCMReader(stream, labelmsm=True, quitonerror=rtt.ERR_RAISE)
            for raw, parsed in rtr:
                if raw is not None:
                    # print(f'"{parsed}",')
                    self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                    i += 1
        self.assertEqual(i, 11)

    def testMIXEDRTCM_SCALE_LABELMSM_FREQ(
        self,
    ):  # test mixed stream of NMEA, UBX & RTCM messages with scaling and Frequency Band labelmsm applied
        EXPECTED_RESULTS = (
            "<RTCM(1005, DF002=1005, DF003=0, DF021=0, DF022=1, DF023=1, DF024=1, DF141=0, DF025=4444030.802800001, DF142=1, DF001_1=0, DF026=3085671.2349, DF364=0, DF027=3366658.256)>",
            "<RTCM(4072, DF002=4072, Not_Yet_Implemented)>",
            "<RTCM(1077, DF002=1077, DF003=0, DF004=204137001, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=760738918298550272, NSat=10, DF395=1073807360, NSig=2, DF396=1044459, NCell=17, PRN_01=005, PRN_02=007, PRN_03=009, PRN_04=013, PRN_05=014, PRN_06=015, PRN_07=017, PRN_08=019, PRN_09=020, PRN_10=030, DF397_01=75, DF397_02=75, DF397_03=81, DF397_04=72, DF397_05=67, DF397_06=80, DF397_07=75, DF397_08=82, DF397_09=75, DF397_10=71, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, ExtSatInfo_06=0, ExtSatInfo_07=0, ExtSatInfo_08=0, ExtSatInfo_09=0, ExtSatInfo_10=0, DF398_01=0.005859375, DF398_02=0.5341796875, DF398_03=0.7626953125, DF398_04=0.138671875, DF398_05=0.5498046875, DF398_06=0.11328125, DF398_07=0.8037109375, DF398_08=0.1025390625, DF398_09=0.521484375, DF398_10=0.345703125, DF399_01=-178, DF399_02=-304, DF399_03=-643, DF399_04=477, DF399_05=-52, DF399_06=645, DF399_07=529, DF399_08=643, DF399_09=-428, DF399_10=-181, CELLPRN_01=005, CELLSIG_01=L1, CELLPRN_02=005, CELLSIG_02=L2, CELLPRN_03=007, CELLSIG_03=L1, CELLPRN_04=007, CELLSIG_04=L2, CELLPRN_05=009, CELLSIG_05=L1, CELLPRN_06=009, CELLSIG_06=L2, CELLPRN_07=013, CELLSIG_07=L1, CELLPRN_08=014, CELLSIG_08=L1, CELLPRN_09=014, CELLSIG_09=L2, CELLPRN_10=015, CELLSIG_10=L1, CELLPRN_11=015, CELLSIG_11=L2, CELLPRN_12=017, CELLSIG_12=L1, CELLPRN_13=017, CELLSIG_13=L2, CELLPRN_14=019, CELLSIG_14=L1, CELLPRN_15=020, CELLSIG_15=L1, CELLPRN_16=030, CELLSIG_16=L1, CELLPRN_17=030, CELLSIG_17=L2, DF405_01=0.00014309026300907135, DF405_02=0.00014183297753334045, DF405_03=0.0003883279860019684, DF405_04=0.00038741156458854675, DF405_05=-0.0004838351160287857, DF405_06=-0.00046883709728717804, DF405_07=0.0003478657454252243, DF405_08=0.0002196934074163437, DF405_09=0.00021521002054214478, DF405_10=-0.00018852390348911285, DF405_11=-0.00018319115042686462, DF405_12=-0.00010087713599205017, DF405_13=-9.844452142715454e-05, DF405_14=0.00047875382006168365, DF405_15=0.00043664872646331787, DF405_16=-0.0003105681389570236, DF405_17=-0.00030865520238876343, DF406_01=0.00014193402603268623, DF406_02=0.00014339853078126907, DF406_03=0.00039040297269821167, DF406_04=0.00038743019104003906, DF406_05=-0.0004843934439122677, DF406_06=-0.00046825408935546875, DF406_07=0.0003473707474768162, DF406_08=0.00021758908405900002, DF406_09=0.00021597417071461678, DF406_10=-0.00018658116459846497, DF406_11=-0.00018350128084421158, DF406_12=-9.993184357881546e-05, DF406_13=-9.724870324134827e-05, DF406_14=0.0004128236323595047, DF406_15=0.0004355977289378643, DF406_16=-0.0003112703561782837, DF406_17=-0.00030898721888661385, DF407_01=341, DF407_02=341, DF407_03=341, DF407_04=341, DF407_05=341, DF407_06=341, DF407_07=341, DF407_08=341, DF407_09=341, DF407_10=341, DF407_11=341, DF407_12=341, DF407_13=341, DF407_14=295, DF407_15=341, DF407_16=341, DF407_17=341, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF408_01=45.0, DF408_02=38.0, DF408_03=43.0, DF408_04=39.0, DF408_05=39.0, DF408_06=37.0, DF408_07=45.0, DF408_08=46.0, DF408_09=46.0, DF408_10=39.0, DF408_11=34.0, DF408_12=45.0, DF408_13=38.0, DF408_14=31.0, DF408_15=45.0, DF408_16=46.0, DF408_17=41.0, DF404_01=-0.9231, DF404_02=-0.9194, DF404_03=-0.8321000000000001, DF404_04=-0.8326, DF404_05=-0.4107, DF404_06=-0.4072, DF404_07=0.2451, DF404_08=-0.0693, DF404_09=-0.0684, DF404_10=0.9390000000000001, DF404_11=0.9417000000000001, DF404_12=0.2384, DF404_13=0.2416, DF404_14=0.6636000000000001, DF404_15=-0.9556, DF404_16=-0.21480000000000002, DF404_17=-0.2174)>",
            "<RTCM(1087, DF002=1087, DF003=0, DF416=2, DF034=42119001, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=4039168114821169152, NSat=7, DF395=1090519040, NSig=2, DF396=16382, NCell=13, PRN_01=003, PRN_02=004, PRN_03=005, PRN_04=013, PRN_05=014, PRN_06=015, PRN_07=023, DF397_01=69, DF397_02=64, DF397_03=73, DF397_04=76, DF397_05=66, DF397_06=70, DF397_07=78, DF419_01=12, DF419_02=13, DF419_03=8, DF419_04=5, DF419_05=0, DF419_06=7, DF419_07=10, DF398_01=0.6337890625, DF398_02=0.3427734375, DF398_03=0.25390625, DF398_04=0.310546875, DF398_05=0.5126953125, DF398_06=0.8271484375, DF398_07=0.8837890625, DF399_01=-665, DF399_02=29, DF399_03=672, DF399_04=-573, DF399_05=-211, DF399_06=312, DF399_07=317, CELLPRN_01=003, CELLSIG_01=G1, CELLPRN_02=003, CELLSIG_02=G2, CELLPRN_03=004, CELLSIG_03=G1, CELLPRN_04=004, CELLSIG_04=G2, CELLPRN_05=005, CELLSIG_05=G1, CELLPRN_06=005, CELLSIG_06=G2, CELLPRN_07=013, CELLSIG_07=G1, CELLPRN_08=013, CELLSIG_08=G2, CELLPRN_09=014, CELLSIG_09=G1, CELLPRN_10=014, CELLSIG_10=G2, CELLPRN_11=015, CELLSIG_11=G1, CELLPRN_12=015, CELLSIG_12=G2, CELLPRN_13=023, CELLSIG_13=G1, DF405_01=0.00024936161935329437, DF405_02=0.0002511627972126007, DF405_03=-4.678964614868164e-05, DF405_04=-5.141831934452057e-05, DF405_05=1.1144205927848816e-05, DF405_06=2.15042382478714e-05, DF405_07=0.00047079287469387054, DF405_08=0.0004794951528310776, DF405_09=-0.0003879182040691376, DF405_10=-0.00037603825330734253, DF405_11=0.0002771839499473572, DF405_12=0.0002871435135602951, DF405_13=-0.00023611821234226227, DF406_01=0.00024937279522418976, DF406_02=0.00025077443569898605, DF406_03=-4.834495484828949e-05, DF406_04=-5.1246024668216705e-05, DF406_05=1.1149328202009201e-05, DF406_06=2.1803192794322968e-05, DF406_07=0.00047026341781020164, DF406_08=0.0004848274402320385, DF406_09=-0.0003876127302646637, DF406_10=-0.0003757951781153679, DF406_11=0.0002778824418783188, DF406_12=0.0002880701795220375, DF406_13=-0.00023698341101408005, DF407_01=341, DF407_02=341, DF407_03=340, DF407_04=340, DF407_05=341, DF407_06=341, DF407_07=340, DF407_08=341, DF407_09=341, DF407_10=341, DF407_11=341, DF407_12=341, DF407_13=340, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF408_01=47.0, DF408_02=40.0, DF408_03=47.0, DF408_04=42.0, DF408_05=47.0, DF408_06=39.0, DF408_07=36.0, DF408_08=33.0, DF408_09=48.0, DF408_10=43.0, DF408_11=48.0, DF408_12=40.0, DF408_13=41.0, DF404_01=-0.8193, DF404_02=-0.8173, DF404_03=0.8539, DF404_04=0.8501000000000001, DF404_05=0.7333000000000001, DF404_06=0.7311000000000001, DF404_07=-0.24930000000000002, DF404_08=-0.2543, DF404_09=-0.21580000000000002, DF404_10=-0.21780000000000002, DF404_11=0.3924, DF404_12=0.3947, DF404_13=0.6146)>",
            "<RTCM(1097, DF002=1097, DF003=0, DF248=204137001, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=216181732825628672, NSat=5, DF395=1073872896, NSig=2, DF396=1023, NCell=10, PRN_01=007, PRN_02=008, PRN_03=021, PRN_04=027, PRN_05=030, DF397_01=79, DF397_02=84, DF397_03=89, DF397_04=78, DF397_05=83, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, DF398_01=0.15625, DF398_02=0.2509765625, DF398_03=0.3544921875, DF398_04=0.37109375, DF398_05=0.259765625, DF399_01=-198, DF399_02=-516, DF399_03=423, DF399_04=63, DF399_05=-384, CELLPRN_01=007, CELLSIG_01=E1, CELLPRN_02=007, CELLSIG_02=E5B, CELLPRN_03=008, CELLSIG_03=E1, CELLPRN_04=008, CELLSIG_04=E5B, CELLPRN_05=021, CELLSIG_05=E1, CELLPRN_06=021, CELLSIG_06=E5B, CELLPRN_07=027, CELLSIG_07=E1, CELLPRN_08=027, CELLSIG_08=E5B, CELLPRN_09=030, CELLSIG_09=E1, CELLPRN_10=030, CELLSIG_10=E5B, DF405_01=-4.5398250222206116e-05, DF405_02=-2.8252601623535156e-05, DF405_03=-0.00034597329795360565, DF405_04=-0.0003268253058195114, DF405_05=0.0004809703677892685, DF405_06=0.0005012489855289459, DF405_07=-0.00013696029782295227, DF405_08=-0.0001260414719581604, DF405_09=-1.8440186977386475e-05, DF405_10=-3.041699528694153e-06, DF406_01=-4.44464385509491e-05, DF406_02=-2.835458144545555e-05, DF406_03=-0.0003525479696691036, DF406_04=-0.0003263736143708229, DF406_05=0.00048203859478235245, DF406_06=0.0005008447915315628, DF406_07=-0.0001375703141093254, DF406_08=-0.00012635625898838043, DF406_09=-1.8037855625152588e-05, DF406_10=-3.2926909625530243e-06, DF407_01=341, DF407_02=341, DF407_03=341, DF407_04=341, DF407_05=341, DF407_06=341, DF407_07=341, DF407_08=341, DF407_09=341, DF407_10=341, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF408_01=46.0, DF408_02=49.0, DF408_03=41.0, DF408_04=43.0, DF408_05=43.0, DF408_06=43.0, DF408_07=45.0, DF408_08=49.0, DF408_09=43.0, DF408_10=47.0, DF404_01=-0.5806, DF404_02=-0.5831000000000001, DF404_03=-0.7947000000000001, DF404_04=-0.7943, DF404_05=0.7243, DF404_06=0.7174, DF404_07=0.5534, DF404_08=0.5545, DF404_09=-0.7726000000000001, DF404_10=-0.7733)>",
            "<RTCM(1127, DF002=1127, DF003=0, DF427=204123001, DF393=0, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=198178247981137920, NSat=10, DF395=1074003968, NSig=2, DF396=387754, NCell=11, PRN_01=007, PRN_02=009, PRN_03=010, PRN_04=020, PRN_05=023, PRN_06=028, PRN_07=032, PRN_08=037, PRN_09=040, PRN_10=043, DF397_01=129, DF397_02=132, DF397_03=126, DF397_04=75, DF397_05=81, DF397_06=84, DF397_07=78, DF397_08=74, DF397_09=130, DF397_10=86, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, ExtSatInfo_06=0, ExtSatInfo_07=0, ExtSatInfo_08=0, ExtSatInfo_09=0, ExtSatInfo_10=0, DF398_01=0.1171875, DF398_02=0.4814453125, DF398_03=0.3095703125, DF398_04=0.7255859375, DF398_05=0.41015625, DF398_06=0.5703125, DF398_07=0.5595703125, DF398_08=0.322265625, DF398_09=0.578125, DF398_10=0.673828125, DF399_01=-130, DF399_02=-58, DF399_03=-81, DF399_04=32, DF399_05=-398, DF399_06=436, DF399_07=-523, DF399_08=-65, DF399_09=-182, DF399_10=79, CELLPRN_01=007, CELLSIG_01=B2, CELLPRN_02=009, CELLSIG_02=B2, CELLPRN_03=010, CELLSIG_03=B1, CELLPRN_04=010, CELLSIG_04=B2, CELLPRN_05=020, CELLSIG_05=B1, CELLPRN_06=023, CELLSIG_06=B1, CELLPRN_07=028, CELLSIG_07=B1, CELLPRN_08=032, CELLSIG_08=B1, CELLPRN_09=037, CELLSIG_09=B1, CELLPRN_10=040, CELLSIG_10=B1, CELLPRN_11=043, CELLSIG_11=B1, DF405_01=-0.0003885403275489807, DF405_02=0.00022730417549610138, DF405_03=0.0004036612808704376, DF405_04=0.00039606913924217224, DF405_05=-0.00016684085130691528, DF405_06=-4.75514680147171e-05, DF405_07=0.0003674682229757309, DF405_08=0.00026629865169525146, DF405_09=-0.0002502594143152237, DF405_10=-0.00011803768575191498, DF405_11=-0.0002937670797109604, DF406_01=-0.0003882073797285557, DF406_02=0.0002264929935336113, DF406_03=0.0004031979478895664, DF406_04=0.0003964221104979515, DF406_05=-0.00016694329679012299, DF406_06=-4.848744720220566e-05, DF406_07=0.00036971503868699074, DF406_08=0.0002654106356203556, DF406_09=-0.00025115441530942917, DF406_10=-0.00011868216097354889, DF406_11=-0.00029495684430003166, DF407_01=341, DF407_02=341, DF407_03=341, DF407_04=341, DF407_05=341, DF407_06=341, DF407_07=341, DF407_08=341, DF407_09=341, DF407_10=341, DF407_11=341, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF408_01=45.0, DF408_02=41.0, DF408_03=42.0, DF408_04=45.0, DF408_05=48.0, DF408_06=46.0, DF408_07=42.0, DF408_08=47.0, DF408_09=48.0, DF408_10=44.0, DF408_11=43.0, DF404_01=-0.5674, DF404_02=-0.612, DF404_03=-0.1384, DF404_04=-0.1332, DF404_05=0.5992000000000001, DF404_06=-0.7312000000000001, DF404_07=0.17320000000000002, DF404_08=-0.4308, DF404_09=-0.5975, DF404_10=-0.6733, DF404_11=0.6122000000000001)>",
            "<RTCM(1230, DF002=1230, DF003=0, DF421=1, DF001_3=0, DF422_1=0, DF422_2=0, DF422_3=0, DF422_4=0)>",
            "<RTCM(1007, DF002=1007, DF003=1234, DF029=3, DF030_01=A, DF030_02=B, DF030_03=C, DF031=234)>",
            "<RTCM(1117, DF002=1117, DF003=0, DF428=385820000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=7061644215716937728, NSat=3, DF395=1073774849, NSig=4, DF396=4095, NCell=12, PRN_01=194, PRN_02=195, PRN_03=199, DF397_01=140, DF397_02=112, DF397_03=133, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, DF398_01=0.171875, DF398_02=0.125, DF398_03=0.8720703125, DF399_01=481, DF399_02=-28, DF399_03=1, CELLPRN_01=194, CELLSIG_01=L1, CELLPRN_02=194, CELLSIG_02=L2, CELLPRN_03=194, CELLSIG_03=L5, CELLPRN_04=194, CELLSIG_04=L1, CELLPRN_05=195, CELLSIG_05=L1, CELLPRN_06=195, CELLSIG_06=L2, CELLPRN_07=195, CELLSIG_07=L5, CELLPRN_08=195, CELLSIG_08=L1, CELLPRN_09=199, CELLSIG_09=L1, CELLPRN_10=199, CELLSIG_10=L2, CELLPRN_11=199, CELLSIG_11=L5, CELLPRN_12=199, CELLSIG_12=L1, DF405_01=0.00022634491324424744, DF405_02=0.00026302412152290344, DF405_03=0.0002453848719596863, DF405_04=0.00023026764392852783, DF405_05=-3.849714994430542e-05, DF405_06=-8.327886462211609e-06, DF405_07=-2.2586435079574585e-05, DF405_08=-3.288686275482178e-05, DF405_09=-9.805895388126373e-05, DF405_10=-7.162615656852722e-05, DF405_11=-8.577480912208557e-05, DF405_12=-9.572319686412811e-05, DF406_01=-0.000219784677028656, DF406_02=0.000500889029353857, DF406_03=-3.950204700231552e-06, DF406_04=0.0002219858579337597, DF406_05=0.0003901217132806778, DF406_06=-0.00017092609778046608, DF406_07=-0.0001668902114033699, DF406_08=-0.0001881415955722332, DF406_09=0.0002841702662408352, DF406_10=-0.0001959027722477913, DF406_11=0.00020570401102304459, DF406_12=0.00017753522843122482, DF407_01=516, DF407_02=462, DF407_03=684, DF407_04=365, DF407_05=648, DF407_06=649, DF407_07=649, DF407_08=649, DF407_09=568, DF407_10=704, DF407_11=704, DF407_12=704, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF408_01=30.3125, DF408_02=35.875, DF408_03=34.5, DF408_04=34.5, DF408_05=48.625, DF408_06=50.5, DF408_07=52.625, DF408_08=51.1875, DF408_09=38.3125, DF408_10=43.6875, DF408_11=45.8125, DF408_12=40.3125, DF404_01=-0.10640000000000001, DF404_02=-1.6384, DF404_03=-1.6384, DF404_04=-1.6384, DF404_05=-0.5875, DF404_06=-1.6384, DF404_07=-1.6384, DF404_08=-1.6384, DF404_09=-0.41300000000000003, DF404_10=-1.6384, DF404_11=-1.6384, DF404_12=-1.6384)>",
            "<RTCM(1059, DF002=1059, DF385=466480, DF391=2, DF388=0, DF413=1, DF414=3, DF415=1, DF387=30, DF068_01=1, DF379_01=2, DF380_01_01=0, DF383_01_01=0.23, DF380_01_02=10, DF383_01_02=0.62, DF068_02=2, DF379_02=2, DF380_02_01=0, DF383_02_01=2.71, DF380_02_02=10, DF383_02_02=2.27, DF068_03=3, DF379_03=2, DF380_03_01=0, DF383_03_01=-1.72, DF380_03_02=10, DF383_03_02=-1.47, DF068_04=4, DF379_04=2, DF380_04_01=0, DF383_04_01=-0.31, DF380_04_02=10, DF383_04_02=-0.19, DF068_05=5, DF379_05=2, DF380_05_01=0, DF383_05_01=0.65, DF380_05_02=10, DF383_05_02=0.93, DF068_06=6, DF379_06=2, DF380_06_01=0, DF383_06_01=-2.05, DF380_06_02=10, DF383_06_02=-1.92, DF068_07=7, DF379_07=2, DF380_07_01=0, DF383_07_01=0.93, DF380_07_02=10, DF383_07_02=1.0, DF068_08=8, DF379_08=2, DF380_08_01=0, DF383_08_01=-1.82, DF380_08_02=10, DF383_08_02=-2.05, DF068_09=9, DF379_09=2, DF380_09_01=0, DF383_09_01=-1.1400000000000001, DF380_09_02=10, DF383_09_02=-1.23, DF068_10=10, DF379_10=2, DF380_10_01=0, DF383_10_01=-1.72, DF380_10_02=10, DF383_10_02=-1.68, DF068_11=11, DF379_11=2, DF380_11_01=0, DF383_11_01=0.4, DF380_11_02=10, DF383_11_02=0.5700000000000001, DF068_12=12, DF379_12=2, DF380_12_01=0, DF383_12_01=1.05, DF380_12_02=10, DF383_12_02=1.32, DF068_13=13, DF379_13=2, DF380_13_01=0, DF383_13_01=0.98, DF380_13_02=10, DF383_13_02=1.06, DF068_14=14, DF379_14=2, DF380_14_01=0, DF383_14_01=0.28, DF380_14_02=10, DF383_14_02=0.45, DF068_15=15, DF379_15=2, DF380_15_01=0, DF383_15_01=0.55, DF380_15_02=10, DF383_15_02=0.91, DF068_16=16, DF379_16=2, DF380_16_01=0, DF383_16_01=1.3900000000000001, DF380_16_02=10, DF383_16_02=0.93, DF068_17=17, DF379_17=2, DF380_17_01=0, DF383_17_01=0.86, DF380_17_02=10, DF383_17_02=1.06, DF068_18=18, DF379_18=2, DF380_18_01=0, DF383_18_01=0.31, DF380_18_02=10, DF383_18_02=0.53, DF068_19=19, DF379_19=2, DF380_19_01=0, DF383_19_01=2.56, DF380_19_02=10, DF383_19_02=1.92, DF068_20=20, DF379_20=2, DF380_20_01=0, DF383_20_01=1.1500000000000001, DF380_20_02=10, DF383_20_02=0.49, DF068_21=22, DF379_21=2, DF380_21_01=0, DF383_21_01=1.26, DF380_21_02=10, DF383_21_02=1.03, DF068_22=23, DF379_22=2, DF380_22_01=0, DF383_22_01=0.31, DF380_22_02=10, DF383_22_02=0.54, DF068_23=24, DF379_23=2, DF380_23_01=0, DF383_23_01=-1.87, DF380_23_02=10, DF383_23_02=-1.57, DF068_24=25, DF379_24=2, DF380_24_01=0, DF383_24_01=-1.92, DF380_24_02=10, DF383_24_02=-2.17, DF068_25=26, DF379_25=2, DF380_25_01=0, DF383_25_01=-2.36, DF380_25_02=10, DF383_25_02=-2.41, DF068_26=27, DF379_26=2, DF380_26_01=0, DF383_26_01=-1.52, DF380_26_02=10, DF383_26_02=-1.54, DF068_27=28, DF379_27=2, DF380_27_01=0, DF383_27_01=0.63, DF380_27_02=10, DF383_27_02=0.75, DF068_28=29, DF379_28=2, DF380_28_01=0, DF383_28_01=0.52, DF380_28_02=10, DF383_28_02=0.8300000000000001, DF068_29=31, DF379_29=2, DF380_29_01=0, DF383_29_01=1.26, DF380_29_02=10, DF383_29_02=1.46, DF068_30=32, DF379_30=2, DF380_30_01=0, DF383_30_01=-1.26, DF380_30_02=10, DF383_30_02=-1.3)>",
            "<RTCM(1060, DF002=1060, DF385=466485, DF391=2, DF388=0, DF375=0, DF413=1, DF414=3, DF415=1, DF387=30, DF068_01=1, DF071_01=99, DF365_01=-1040.3, DF366_01=1451.6000000000001, DF367_01=541.2, DF368_01=-0.251, DF369_01=-0.188, DF370_01=0.076, DF376_01=157.20000000000002, DF377_01=0.0, DF378_01=0.0, DF068_02=2, DF071_02=60, DF365_02=-120.9, DF366_02=2483.2000000000003, DF367_02=1154.4, DF368_02=-0.34400000000000003, DF369_02=0.248, DF370_02=0.28, DF376_02=-4896.7, DF377_02=0.0, DF378_02=0.0, DF068_03=3, DF071_03=12, DF365_03=97.5, DF366_03=-60.0, DF367_03=-1040.4, DF368_03=-0.045, DF369_03=-0.108, DF370_03=0.128, DF376_03=555.5, DF377_03=0.0, DF378_03=0.0, DF068_04=4, DF071_04=43, DF365_04=-576.9, DF366_04=-571.6, DF367_04=-583.2, DF368_04=0.041, DF369_04=-0.108, DF370_04=-0.056, DF376_04=-842.5, DF377_04=0.0, DF378_04=0.0, DF068_05=5, DF071_05=23, DF365_05=-75.2, DF366_05=206.0, DF367_05=224.0, DF368_05=-0.073, DF369_05=0.148, DF370_05=-0.088, DF376_05=-967.0, DF377_05=0.0, DF378_05=0.0, DF068_06=6, DF071_06=65, DF365_06=-43.900000000000006, DF366_06=1134.0, DF367_06=94.0, DF368_06=-0.027, DF369_06=-0.044, DF370_06=0.088, DF376_06=466.70000000000005, DF377_06=0.0, DF378_06=0.0, DF068_07=7, DF071_07=9, DF365_07=409.0, DF366_07=27.6, DF367_07=-652.8000000000001, DF368_07=0.012, DF369_07=-0.464, DF370_07=0.256, DF376_07=-2019.3000000000002, DF377_07=0.0, DF378_07=0.0, DF068_08=8, DF071_08=187, DF365_08=33.1, DF366_08=1139.6000000000001, DF367_08=57.6, DF368_08=-0.12, DF369_08=-0.02, DF370_08=0.264, DF376_08=8.3, DF377_08=0.0, DF378_08=0.0, DF068_09=9, DF071_09=103, DF365_09=412.5, DF366_09=-825.6, DF367_09=734.4000000000001, DF368_09=0.1, DF369_09=-0.24, DF370_09=0.016, DF376_09=1188.6000000000001, DF377_09=0.0, DF378_09=0.0, DF068_10=10, DF071_10=36, DF365_10=-113.0, DF366_10=380.0, DF367_10=22.0, DF368_10=-0.024, DF369_10=0.052000000000000005, DF370_10=0.128, DF376_10=101.0, DF377_10=0.0, DF378_10=0.0, DF068_11=11, DF071_11=213, DF365_11=-890.7, DF366_11=699.6, DF367_11=-357.6, DF368_11=-0.176, DF369_11=-0.14, DF370_11=0.028, DF376_11=-1523.2, DF377_11=0.0, DF378_11=0.0, DF068_12=12, DF071_12=22, DF365_12=-60.5, DF366_12=200.8, DF367_12=119.2, DF368_12=-0.09, DF369_12=-0.116, DF370_12=-0.1, DF376_12=-1996.5, DF377_12=0.0, DF378_12=0.0, DF068_13=13, DF071_13=48, DF365_13=-38.900000000000006, DF366_13=-43.2, DF367_13=-838.8000000000001, DF368_13=-0.073, DF369_13=0.004, DF370_13=0.108, DF376_13=-1263.9, DF377_13=0.0, DF378_13=0.0, DF068_14=14, DF071_14=115, DF365_14=-750.4000000000001, DF366_14=793.6, DF367_14=484.40000000000003, DF368_14=-0.109, DF369_14=-0.468, DF370_14=-0.28, DF376_14=-1075.2, DF377_14=0.0, DF378_14=0.0, DF068_15=15, DF071_15=78, DF365_15=30.200000000000003, DF366_15=-1284.4, DF367_15=-942.0, DF368_15=0.115, DF369_15=0.044, DF370_15=0.052000000000000005, DF376_15=-778.3000000000001, DF377_15=0.0, DF378_15=0.0, DF068_16=16, DF071_16=5, DF365_16=-12.700000000000001, DF366_16=835.2, DF367_16=-676.0, DF368_16=-0.07200000000000001, DF369_16=0.02, DF370_16=-0.008, DF376_16=-2940.9, DF377_16=0.0, DF378_16=0.0, DF068_17=17, DF071_17=43, DF365_17=-83.7, DF366_17=924.8000000000001, DF367_17=346.0, DF368_17=-0.067, DF369_17=-0.116, DF370_17=0.02, DF376_17=-2035.7, DF377_17=0.0, DF378_17=0.0, DF068_18=18, DF071_18=177, DF365_18=-973.4000000000001, DF366_18=39.2, DF367_18=-464.8, DF368_18=-0.095, DF369_18=-0.16, DF370_18=-0.1, DF376_18=-650.4000000000001, DF377_18=0.0, DF378_18=0.0, DF068_19=19, DF071_19=28, DF365_19=-55.800000000000004, DF366_19=1466.8000000000002, DF367_19=709.6, DF368_19=-0.185, DF369_19=-0.24, DF370_19=-0.116, DF376_19=-3958.8, DF377_19=0.0, DF378_19=0.0, DF068_20=20, DF071_20=13, DF365_20=-63.0, DF366_20=63.2, DF367_20=-510.8, DF368_20=-0.027, DF369_20=-0.132, DF370_20=0.008, DF376_20=-2539.7000000000003, DF377_20=0.0, DF378_20=0.0, DF068_21=22, DF071_21=25, DF365_21=34.5, DF366_21=890.4000000000001, DF367_21=-823.6, DF368_21=-0.097, DF369_21=-0.128, DF370_21=-0.08, DF376_21=-2546.6000000000004, DF377_21=0.0, DF378_21=0.0, DF068_22=23, DF071_22=180, DF365_22=-655.3000000000001, DF366_22=-971.6, DF367_22=-117.2, DF368_22=0.077, DF369_22=-0.132, DF370_22=-0.024, DF376_22=-616.7, DF377_22=0.0, DF378_22=0.0, DF068_23=24, DF071_23=126, DF365_23=-680.9000000000001, DF366_23=175.20000000000002, DF367_23=-85.2, DF368_23=0.105, DF369_23=-0.124, DF370_23=-0.108, DF376_23=2253.0, DF377_23=0.0, DF378_23=0.0, DF068_24=25, DF071_24=21, DF365_24=-47.2, DF366_24=99.60000000000001, DF367_24=36.4, DF368_24=-0.07100000000000001, DF369_24=-0.268, DF370_24=-0.064, DF376_24=546.7, DF377_24=0.0, DF378_24=0.0, DF068_25=26, DF071_25=70, DF365_25=-46.800000000000004, DF366_25=1472.8000000000002, DF367_25=-284.8, DF368_25=-0.185, DF369_25=-0.14400000000000002, DF370_25=-0.088, DF376_25=578.5, DF377_25=0.0, DF378_25=0.0, DF068_26=27, DF071_26=33, DF365_26=-201.9, DF366_26=626.0, DF367_26=-18.0, DF368_26=-0.115, DF369_26=0.1, DF370_26=0.048, DF376_26=-519.6, DF377_26=0.0, DF378_26=0.0, DF068_27=28, DF071_27=224, DF365_27=-560.3000000000001, DF366_27=70.8, DF367_27=606.4, DF368_27=0.006, DF369_27=-0.456, DF370_27=0.036000000000000004, DF376_27=-1146.6000000000001, DF377_27=0.0, DF378_27=0.0, DF068_28=29, DF071_28=123, DF365_28=-115.5, DF366_28=-397.6, DF367_28=76.4, DF368_28=0.069, DF369_28=-0.192, DF370_28=-0.164, DF376_28=-1773.6000000000001, DF377_28=0.0, DF378_28=0.0, DF068_29=31, DF071_29=7, DF365_29=-21.0, DF366_29=168.4, DF367_29=171.20000000000002, DF368_29=0.078, DF369_29=-0.12, DF370_29=0.024, DF376_29=-2145.3, DF377_29=0.0, DF378_29=0.0, DF068_30=32, DF071_30=88, DF365_30=-288.7, DF366_30=52.400000000000006, DF367_30=-239.60000000000002, DF368_30=-0.066, DF369_30=0.068, DF370_30=0.044, DF376_30=888.0, DF377_30=0.0, DF378_30=0.0)>",
        )

        dirname = os.path.dirname(__file__)
        with open(os.path.join(dirname, "pygpsdata-RTCM3.log"), "rb") as stream:
            i = 0
            raw = 0
            rtr = RTCMReader(stream, labelmsm=2, quitonerror=rtt.ERR_RAISE)
            for raw, parsed in rtr:
                if raw is not None:
                    # print(f'"{parsed}",')
                    self.assertEqual(str(parsed), EXPECTED_RESULTS[i])
                    i += 1
        self.assertEqual(i, 11)

    def testntrip2log(
        self,
    ):  # test using log from NTRIP caster igs-ip.net, mountpoint USCL00CHL0
        EXPECTED_RESULTS = [
            "<RTCM(1003, DF002=1003, DF003=0, DF004=318945000, DF005=1, DF006=11, DF007=0, DF008=0, DF009_01=2, DF010_01=0, DF011_01=282060.0, DF012_01=-30.8545, DF013_01=127, DF016_01=3, DF017_01=8.34, DF018_01=-44.145, DF019_01=127, DF009_02=3, DF010_02=0, DF011_02=6888.18, DF012_02=-13.153, DF013_02=127, DF016_02=3, DF017_02=7.9, DF018_02=-12.911, DF019_02=127, DF009_03=21, DF010_03=0, DF011_03=17235.68, DF012_03=-50.468, DF013_03=127, DF016_03=3, DF017_03=15.4, DF018_03=-68.6825, DF019_03=127, DF009_04=4, DF010_04=0, DF011_04=144748.02, DF012_04=1.4805, DF013_04=127, DF016_04=3, DF017_04=5.5600000000000005, DF018_04=7.611, DF019_04=127, DF009_05=9, DF010_05=0, DF011_05=95354.06, DF012_05=7.7525, DF013_05=127, DF016_05=3, DF017_05=7.82, DF018_05=18.57, DF019_05=127, DF009_06=6, DF010_06=0, DF011_06=13104.98, DF012_06=2.84, DF013_06=127, DF016_06=3, DF017_06=11.5, DF018_06=15.8435, DF019_06=127, DF009_07=19, DF010_07=0, DF011_07=21822.56, DF012_07=-8.8195, DF013_07=127, DF016_07=3, DF017_07=7.2, DF018_07=-13.3185, DF019_07=127, DF009_08=31, DF010_08=0, DF011_08=28651.58, DF012_08=-9.156, DF013_08=127, DF016_08=3, DF017_08=10.68, DF018_08=-3.7505, DF019_08=127, DF009_09=17, DF010_09=0, DF011_09=51315.020000000004, DF012_09=-16.4175, DF013_09=127, DF016_09=3, DF017_09=9.96, DF018_09=-20.1975, DF019_09=127, DF009_10=7, DF010_10=0, DF011_10=155082.74, DF012_10=0.28800000000000003, DF013_10=50, DF016_10=3, DF017_10=18.36, DF018_10=16.5705, DF019_10=41, DF009_11=1, DF010_11=0, DF011_11=173993.44, DF012_11=-10.573, DF013_11=127, DF016_11=3, DF017_11=8.44, DF018_11=-8.8375, DF019_11=127)>",
            "<RTCM(1004, DF002=1004, DF003=0, DF004=318945000, DF005=1, DF006=11, DF007=0, DF008=0, DF009_01=2, DF010_01=0, DF011_01=282060.0, DF012_01=-30.8545, DF013_01=127, DF014_01=75, DF015_01=43.0, DF016_01=3, DF017_01=8.34, DF018_01=-44.145, DF019_01=127, DF020_01=31.25, DF009_02=3, DF010_02=0, DF011_02=6888.18, DF012_02=-13.153, DF013_02=127, DF014_02=70, DF015_02=49.5, DF016_02=3, DF017_02=7.9, DF018_02=-12.911, DF019_02=127, DF020_02=52.0, DF009_03=21, DF010_03=0, DF011_03=17235.68, DF012_03=-50.468, DF013_03=127, DF014_03=79, DF015_03=38.5, DF016_03=3, DF017_03=15.4, DF018_03=-68.6825, DF019_03=127, DF020_03=19.5, DF009_04=4, DF010_04=0, DF011_04=144748.02, DF012_04=1.4805, DF013_04=127, DF014_04=67, DF015_04=49.0, DF016_04=3, DF017_04=5.5600000000000005, DF018_04=7.611, DF019_04=127, DF020_04=43.75, DF009_05=9, DF010_05=0, DF011_05=95354.06, DF012_05=7.7525, DF013_05=127, DF014_05=71, DF015_05=48.0, DF016_05=3, DF017_05=7.82, DF018_05=18.57, DF019_05=127, DF020_05=50.5, DF009_06=6, DF010_06=0, DF011_06=13104.98, DF012_06=2.84, DF013_06=127, DF014_06=78, DF015_06=39.75, DF016_06=3, DF017_06=11.5, DF018_06=15.8435, DF019_06=127, DF020_06=35.75, DF009_07=19, DF010_07=0, DF011_07=21822.56, DF012_07=-8.8195, DF013_07=127, DF014_07=78, DF015_07=40.75, DF016_07=3, DF017_07=7.2, DF018_07=-13.3185, DF019_07=127, DF020_07=27.5, DF009_08=31, DF010_08=0, DF011_08=28651.58, DF012_08=-9.156, DF013_08=127, DF014_08=79, DF015_08=42.5, DF016_08=3, DF017_08=10.68, DF018_08=-3.7505, DF019_08=127, DF020_08=37.0, DF009_09=17, DF010_09=0, DF011_09=51315.020000000004, DF012_09=-16.4175, DF013_09=127, DF014_09=76, DF015_09=40.5, DF016_09=3, DF017_09=9.96, DF018_09=-20.1975, DF019_09=127, DF020_09=34.75, DF009_10=7, DF010_10=0, DF011_10=155082.74, DF012_10=0.28800000000000003, DF013_10=50, DF014_10=83, DF015_10=34.0, DF016_10=3, DF017_10=18.36, DF018_10=16.5705, DF019_10=41, DF020_10=18.75, DF009_11=1, DF010_11=0, DF011_11=173993.44, DF012_11=-10.573, DF013_11=127, DF014_11=68, DF015_11=49.5, DF016_11=3, DF017_11=8.44, DF018_11=-8.8375, DF019_11=127, DF020_11=52.0)>",
            "<RTCM(1005, DF002=1005, DF003=0, DF021=0, DF022=1, DF023=1, DF024=1, DF141=0, DF025=1762489.6191, DF142=1, DF001_1=0, DF026=-5027633.8438, DF364=2, DF027=-3496008.8438000004)>",
            "<RTCM(1006, DF002=1006, DF003=0, DF021=0, DF022=1, DF023=1, DF024=1, DF141=0, DF025=1762489.6191, DF142=1, DF001_1=0, DF026=-5027633.8438, DF364=2, DF027=-3496008.8438000004, DF028=0.034300000000000004)>",
            "<RTCM(1007, DF002=1007, DF003=0, DF029=20, DF030_01=S, DF030_02=E, DF030_03=P, DF030_04=C, DF030_05=H, DF030_06=O, DF030_07=K, DF030_08=E, DF030_09=_, DF030_10=B, DF030_11=3, DF030_12=E, DF030_13=6, DF030_14= , DF030_15= , DF030_16= , DF030_17=S, DF030_18=P, DF030_19=K, DF030_20=E, DF031=0)>",
            "<RTCM(1008, DF002=1008, DF003=0, DF029=20, DF030_01=S, DF030_02=E, DF030_03=P, DF030_04=C, DF030_05=H, DF030_06=O, DF030_07=K, DF030_08=E, DF030_09=_, DF030_10=B, DF030_11=3, DF030_12=E, DF030_13=6, DF030_14= , DF030_15= , DF030_16= , DF030_17=S, DF030_18=P, DF030_19=K, DF030_20=E, DF031=0, DF032=4, DF033_01=5, DF033_02=8, DF033_03=5, DF033_04=6)>",
            "<RTCM(1009, DF002=1009, DF003=0, DF034=70527000, DF005=1, DF035=8, DF036=0, DF037=0, DF038_01=1, DF039_01=0, DF040_01=8, DF041_01=272788.02, DF042_01=11.905, DF043_01=127, DF038_02=22, DF039_02=0, DF040_02=4, DF041_02=168818.0, DF042_02=-19.451, DF043_02=127, DF038_03=24, DF039_03=0, DF040_03=9, DF041_03=295925.82, DF042_03=6.0755, DF043_03=127, DF038_04=8, DF039_04=0, DF040_04=13, DF041_04=298011.32, DF042_04=3.767, DF043_04=127, DF038_05=7, DF039_05=0, DF040_05=12, DF041_05=171808.64, DF042_05=-23.433500000000002, DF043_05=127, DF038_06=23, DF039_06=0, DF040_06=10, DF041_06=485086.12, DF042_06=2.96, DF043_06=127, DF038_07=10, DF039_07=0, DF040_07=0, DF041_07=574364.56, DF042_07=13.8625, DF043_07=127, DF038_08=9, DF039_08=0, DF040_08=5, DF041_08=322329.8, DF042_08=11.52, DF043_08=127)>",
            "<RTCM(1010, DF002=1010, DF003=0, DF034=70527000, DF005=1, DF035=8, DF036=0, DF037=0, DF038_01=1, DF039_01=0, DF040_01=8, DF041_01=272788.02, DF042_01=11.905, DF043_01=127, DF044_01=37, DF045_01=41.5, DF038_02=22, DF039_02=0, DF040_02=4, DF041_02=168818.0, DF042_02=-19.451, DF043_02=127, DF044_02=35, DF045_02=49.75, DF038_03=24, DF039_03=0, DF040_03=9, DF041_03=295925.82, DF042_03=6.0755, DF043_03=127, DF044_03=37, DF045_03=45.75, DF038_04=8, DF039_04=0, DF040_04=13, DF041_04=298011.32, DF042_04=3.767, DF043_04=127, DF044_04=33, DF045_04=52.75, DF038_05=7, DF039_05=0, DF040_05=12, DF041_05=171808.64, DF042_05=-23.433500000000002, DF043_05=127, DF044_05=36, DF045_05=48.25, DF038_06=23, DF039_06=0, DF040_06=10, DF041_06=485086.12, DF042_06=2.96, DF043_06=127, DF044_06=32, DF045_06=50.75, DF038_07=10, DF039_07=0, DF040_07=0, DF041_07=574364.56, DF042_07=13.8625, DF043_07=127, DF044_07=37, DF045_07=39.0, DF038_08=9, DF039_08=0, DF040_08=5, DF041_08=322329.8, DF042_08=11.52, DF043_08=127, DF044_08=37, DF045_08=44.5)>",
            "<RTCM(1011, DF002=1011, DF003=0, DF034=70527000, DF005=1, DF035=8, DF036=0, DF037=0, DF038_01=1, DF039_01=0, DF040_01=8, DF041_01=272788.02, DF042_01=11.905, DF043_01=127, DF046_01=0, DF047_01=15.06, DF048_01=19.2865, DF049_01=105, DF038_02=22, DF039_02=0, DF040_02=4, DF041_02=168818.0, DF042_02=-19.451, DF043_02=127, DF046_02=0, DF047_02=12.02, DF048_02=-16.844, DF049_02=127, DF038_03=24, DF039_03=0, DF040_03=9, DF041_03=295925.82, DF042_03=6.0755, DF043_03=127, DF046_03=0, DF047_03=14.08, DF048_03=27.5955, DF049_03=127, DF038_04=8, DF039_04=0, DF040_04=13, DF041_04=298011.32, DF042_04=3.767, DF043_04=127, DF046_04=0, DF047_04=9.48, DF048_04=16.065, DF049_04=127, DF038_05=7, DF039_05=0, DF040_05=12, DF041_05=171808.64, DF042_05=-23.433500000000002, DF043_05=127, DF046_05=0, DF047_05=13.92, DF048_05=-22.281, DF049_05=127, DF038_06=23, DF039_06=0, DF040_06=10, DF041_06=485086.12, DF042_06=2.96, DF043_06=127, DF046_06=0, DF047_06=-163.84, DF048_06=-262.144, DF049_06=0, DF038_07=10, DF039_07=0, DF040_07=0, DF041_07=574364.56, DF042_07=13.8625, DF043_07=127, DF046_07=0, DF047_07=-163.84, DF048_07=-262.144, DF049_07=0, DF038_08=9, DF039_08=0, DF040_08=5, DF041_08=322329.8, DF042_08=11.52, DF043_08=127, DF046_08=0, DF047_08=18.12, DF048_08=42.786, DF049_08=127)>",
            "<RTCM(1012, DF002=1012, DF003=0, DF034=70527000, DF005=1, DF035=8, DF036=0, DF037=0, DF038_01=1, DF039_01=0, DF040_01=8, DF041_01=272788.02, DF042_01=11.905, DF043_01=127, DF044_01=37, DF045_01=41.5, DF046_01=0, DF047_01=15.06, DF048_01=19.2865, DF049_01=105, DF050_01=35.5, DF038_02=22, DF039_02=0, DF040_02=4, DF041_02=168818.0, DF042_02=-19.451, DF043_02=127, DF044_02=35, DF045_02=49.75, DF046_02=0, DF047_02=12.02, DF048_02=-16.844, DF049_02=127, DF050_02=46.25, DF038_03=24, DF039_03=0, DF040_03=9, DF041_03=295925.82, DF042_03=6.0755, DF043_03=127, DF044_03=37, DF045_03=45.75, DF046_03=0, DF047_03=14.08, DF048_03=27.5955, DF049_03=127, DF050_03=42.75, DF038_04=8, DF039_04=0, DF040_04=13, DF041_04=298011.32, DF042_04=3.767, DF043_04=127, DF044_04=33, DF045_04=52.75, DF046_04=0, DF047_04=9.48, DF048_04=16.065, DF049_04=127, DF050_04=49.5, DF038_05=7, DF039_05=0, DF040_05=12, DF041_05=171808.64, DF042_05=-23.433500000000002, DF043_05=127, DF044_05=36, DF045_05=48.25, DF046_05=0, DF047_05=13.92, DF048_05=-22.281, DF049_05=127, DF050_05=44.75, DF038_06=23, DF039_06=0, DF040_06=10, DF041_06=485086.12, DF042_06=2.96, DF043_06=127, DF044_06=32, DF045_06=50.75, DF046_06=0, DF047_06=-163.84, DF048_06=-262.144, DF049_06=0, DF050_06=0.0, DF038_07=10, DF039_07=0, DF040_07=0, DF041_07=574364.56, DF042_07=13.8625, DF043_07=127, DF044_07=37, DF045_07=39.0, DF046_07=0, DF047_07=-163.84, DF048_07=-262.144, DF049_07=0, DF050_07=0.0, DF038_08=9, DF039_08=0, DF040_08=5, DF041_08=322329.8, DF042_08=11.52, DF043_08=127, DF044_08=37, DF045_08=44.5, DF046_08=0, DF047_08=18.12, DF048_08=42.786, DF049_08=127, DF050_08=41.75)>",
            "<RTCM(1013, DF002=1013, DF003=0, DF051=60382, DF052=59727, DF053=0, DF054=18)>",
            "<RTCM(1019, DF002=1019, DF009=2, DF076=257, DF077=0, DF078=1, DF079=-1.559783413540572e-10, DF071=185, DF081=324000, DF082=0.0, DF083=6.139089236967266e-12, DF084=-0.00047086644917726517, DF085=185, DF086=-117.28125, DF087=1.339799382549245e-09, DF088=0.6883564381860197, DF089=-5.889683961868286e-06, DF090=0.016119434614665806, DF091=8.553266525268555e-06, DF092=5153.713861465454, DF093=324000, DF094=2.421438694000244e-07, DF095=-0.944771918002516, DF096=1.6763806343078613e-08, DF097=0.3080678000114858, DF098=210.3125, DF099=-0.3891187282279134, DF100=-2.476781446603127e-09, DF101=-1.7695128917694092e-08, DF102=0, DF103=0, DF137=0)>",
            "<RTCM(1020, DF002=1020, DF038=9, DF040=5, DF104=1, DF105=1, DF106=1, DF107=2492, DF108=0, DF109=1, DF110=79, DF111=-2.059713363647461, DF112=19637.81884765625, DF113=0.0, DF114=0.8449039459228516, DF115=33.10888671875, DF116=-1.862645149230957e-09, DF117=-2.4976272583007812, DF118=-16217.08740234375, DF119=2.7939677238464355e-09, DF120=1, DF121=2, DF122=3, DF123=0, DF124=-188052, DF125=-4, DF126=0, DF127=1, DF128=5, DF129=73, DF130=1, DF131=1, DF132=73, DF133=-3, DF134=8, DF135=8, DF136=0, DF001_7=0)>",
            "<RTCM(1029, DF002=1029, DF003=0, DF051=60382, DF052=59727, DF138=7, DF139=7, DF140=Unknown)>",
            "<RTCM(1033, DF002=1033, DF003=0, DF029=20, DF030_01=S, DF030_02=E, DF030_03=P, DF030_04=C, DF030_05=H, DF030_06=O, DF030_07=K, DF030_08=E, DF030_09=_, DF030_10=B, DF030_11=3, DF030_12=E, DF030_13=6, DF030_14= , DF030_15= , DF030_16= , DF030_17=S, DF030_18=P, DF030_19=K, DF030_20=E, DF031=0, DF032=4, DF033_01=5, DF033_02=8, DF033_03=5, DF033_04=6, DF227=12, DF228_01=S, DF228_02=E, DF228_03=P, DF228_04=T, DF228_05= , DF228_06=P, DF228_07=O, DF228_08=L, DF228_09=A, DF228_10=R, DF228_11=X, DF228_12=5, DF229=5, DF230_01=5, DF230_02=., DF230_03=5, DF230_04=., DF230_05=0, DF231=7, DF232_01=3, DF232_02=0, DF232_03=7, DF232_04=5, DF232_05=0, DF232_06=2, DF232_07=4)>",
            "<RTCM(1042, DF002=1042, DF488=12, DF489=949, DF490=0, DF491=-1.3505996321327984e-10, DF492=3, DF493=316800, DF494=-1.3552527156068805e-19, DF495=-7.778666599733697e-12, DF496=-0.00021217693574726582, DF497=2, DF498=-102.984375, DF499=1.1275460565229878e-09, DF500=-0.11344346264377236, DF501=-5.0924718379974365e-06, DF502=0.001100340741686523, DF503=4.862435162067413e-06, DF504=5282.629014968872, DF505=316800, DF506=4.0978193283081055e-08, DF507=0.9092594981193542, DF508=-1.862645149230957e-08, DF509=0.31285916129127145, DF510=274.09375, DF511=-0.4671555492095649, DF512=-2.2137101041153073e-09, DF513=2.4000000000000004, DF514=0.4, DF515=0)>",
            "<RTCM(1045, DF002=1045, DF252=3, DF289=1281, DF290=22, DF291=107, DF292=-3.115019353572279e-11, DF293=318000, DF294=0.0, DF295=-2.6716406864579767e-12, DF296=-0.00010003114584833384, DF297=-40.125, DF298=1.1664269550237805e-09, DF299=-0.5413645040243864, DF300=-1.8794089555740356e-06, DF301=0.00022546376567333937, DF302=4.287809133529663e-06, DF303=5440.592414855957, DF304=318000, DF305=-3.166496753692627e-08, DF306=-0.24508476676419377, DF307=-3.166496753692627e-08, DF308=0.3057721094228327, DF309=247.90625, DF310=-0.08484991453588009, DF311=-1.8743548935162835e-09, DF312=3.026798367500305e-09, DF314=0, DF315=0, DF001_7=0)>",
            "<RTCM(1046, DF002=1046, DF252=5, DF289=1281, DF290=22, DF286=107, DF292=-3.1377567211166024e-11, DF293=318000, DF294=0.0, DF295=3.552713678800501e-12, DF296=0.004728707484900951, DF297=-44.1875, DF298=1.169496499642264e-09, DF299=0.06877923710271716, DF300=-1.9818544387817383e-06, DF301=0.00023969111498445272, DF302=4.159286618232727e-06, DF303=5440.592296600342, DF304=318000, DF305=-5.587935447692871e-09, DF306=-0.24508378840982914, DF307=-5.587935447692871e-09, DF308=0.30577638652175665, DF309=248.15625, DF310=-0.446898490190506, DF311=-1.8883383745560423e-09, DF312=4.423782229423523e-09, DF313=4.889443516731262e-09, DF316=0, DF317=0, DF287=0, DF288=0, DF001_2=0)>",
            "<RTCM(1076, DF002=1076, DF003=0, DF004=318945000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=17762381648302702592, NSat=10, DF395=1346437634, NSig=6, DF396=1132932382022422072, NCell=42, PRN_01=001, PRN_02=002, PRN_03=003, PRN_04=004, PRN_05=006, PRN_06=007, PRN_07=009, PRN_08=017, PRN_09=019, PRN_10=021, DF397_01=68, DF397_02=75, DF397_03=70, DF397_04=67, DF397_05=78, DF397_06=83, DF397_07=71, DF397_08=76, DF397_09=78, DF397_10=79, DF398_01=0.580078125, DF398_02=0.9404296875, DF398_03=0.0234375, DF398_04=0.482421875, DF398_05=0.0439453125, DF398_06=0.517578125, DF398_07=0.318359375, DF398_08=0.1708984375, DF398_09=0.0732421875, DF398_10=0.0576171875, CELLPRN_01=001, CELLSIG_01=1C, CELLPRN_02=001, CELLSIG_02=1W, CELLPRN_03=001, CELLSIG_03=2W, CELLPRN_04=001, CELLSIG_04=2L, CELLPRN_05=001, CELLSIG_05=5Q, CELLPRN_06=002, CELLSIG_06=1C, CELLPRN_07=002, CELLSIG_07=1W, CELLPRN_08=002, CELLSIG_08=2W, CELLPRN_09=003, CELLSIG_09=1C, CELLPRN_10=003, CELLSIG_10=1W, CELLPRN_11=003, CELLSIG_11=2W, CELLPRN_12=003, CELLSIG_12=2L, CELLPRN_13=003, CELLSIG_13=5Q, CELLPRN_14=004, CELLSIG_14=1C, CELLPRN_15=004, CELLSIG_15=1W, CELLPRN_16=004, CELLSIG_16=2W, CELLPRN_17=004, CELLSIG_17=2L, CELLPRN_18=004, CELLSIG_18=5Q, CELLPRN_19=004, CELLSIG_19=1L, CELLPRN_20=006, CELLSIG_20=1C, CELLPRN_21=006, CELLSIG_21=1W, CELLPRN_22=006, CELLSIG_22=2W, CELLPRN_23=006, CELLSIG_23=2L, CELLPRN_24=006, CELLSIG_24=5Q, CELLPRN_25=007, CELLSIG_25=1C, CELLPRN_26=007, CELLSIG_26=1W, CELLPRN_27=007, CELLSIG_27=2W, CELLPRN_28=009, CELLSIG_28=1C, CELLPRN_29=009, CELLSIG_29=1W, CELLPRN_30=009, CELLSIG_30=2W, CELLPRN_31=009, CELLSIG_31=2L, CELLPRN_32=009, CELLSIG_32=5Q, CELLPRN_33=017, CELLSIG_33=1C, CELLPRN_34=017, CELLSIG_34=1W, CELLPRN_35=017, CELLSIG_35=2W, CELLPRN_36=017, CELLSIG_36=2L, CELLPRN_37=019, CELLSIG_37=1C, CELLPRN_38=019, CELLSIG_38=1W, CELLPRN_39=019, CELLSIG_39=2W, CELLPRN_40=021, CELLSIG_40=1C, CELLPRN_41=021, CELLSIG_41=1W, CELLPRN_42=021, CELLSIG_42=2W, DF405_01=0.00030150264501571655, DF405_02=0.00030026398599147797, DF405_03=0.00032966770231723785, DF405_04=0.00032999180257320404, DF405_05=0.0003276858478784561, DF405_06=0.0004211794584989548, DF405_07=0.00041723065078258514, DF405_08=0.0004490204155445099, DF405_09=-0.0004609767347574234, DF405_10=-0.0004615709185600281, DF405_11=-0.0004346519708633423, DF405_12=-0.00043250061571598053, DF405_13=-0.00043288618326187134, DF405_14=0.00040554627776145935, DF405_15=0.00040445849299430847, DF405_16=0.0004241038113832474, DF405_17=0.00042439810931682587, DF405_18=0.00043609924614429474, DF405_19=0.0004059355705976486, DF405_20=-0.0002317819744348526, DF405_21=-0.00023279525339603424, DF405_22=-0.00019344687461853027, DF405_23=-0.00019032880663871765, DF405_24=-0.00018449127674102783, DF405_25=-0.0002777855843305588, DF405_26=-0.00027890317142009735, DF405_27=-0.000216558575630188, DF405_28=-0.00029244646430015564, DF405_29=-0.00029423460364341736, DF405_30=-0.000266415998339653, DF405_31=-0.00026514194905757904, DF405_32=-0.0002633947879076004, DF405_33=0.0002700742334127426, DF405_34=0.0002683773636817932, DF405_35=0.0003032628446817398, DF405_36=0.0003052111715078354, DF405_37=-0.00044993869960308075, DF405_38=-0.0004546046257019043, DF405_39=-0.00042591989040374756, DF405_40=-0.00012511573731899261, DF405_41=-0.00012937746942043304, DF405_42=-7.376261055469513e-05, DF406_01=0.0002662511542439461, DF406_02=0.0002662511542439461, DF406_03=0.0002720397897064686, DF406_04=0.00027366820722818375, DF406_05=0.00026657385751605034, DF406_06=0.0003182794898748398, DF406_07=0.0003182794898748398, DF406_08=0.0002739476040005684, DF406_09=-0.0005048788152635098, DF406_10=-0.0005048788152635098, DF406_11=-0.0005040713585913181, DF406_12=-0.0005008145235478878, DF406_13=-0.0005074627697467804, DF406_14=0.00041048694401979446, DF406_15=0.00041048694401979446, DF406_16=0.0004309355281293392, DF406_17=0.0004439675249159336, DF406_18=0.000458661001175642, DF406_19=0.0004257233813405037, DF406_20=-0.00022233091294765472, DF406_21=-0.00022233091294765472, DF406_22=-0.00017895596101880074, DF406_23=-0.0001813839189708233, DF406_24=-0.00017541320994496346, DF406_25=-0.00027682632207870483, DF406_26=-0.00027682632207870483, DF406_27=-0.00022251298651099205, DF406_28=-0.0002666083164513111, DF406_29=-0.0002666083164513111, DF406_30=-0.0002305246889591217, DF406_31=-0.0002223653718829155, DF406_32=-0.00020845001563429832, DF406_33=0.00021528173238039017, DF406_34=0.00021528173238039017, DF406_35=0.0002026725560426712, DF406_36=0.00020838622003793716, DF406_37=-0.0004793815314769745, DF406_38=-0.0004793815314769745, DF406_39=-0.0004943888634443283, DF406_40=-0.0002934904769062996, DF406_41=-0.0002934904769062996, DF406_42=-0.0003542480990290642, DF407_01=638, DF407_02=638, DF407_03=638, DF407_04=638, DF407_05=638, DF407_06=646, DF407_07=646, DF407_08=646, DF407_09=635, DF407_10=635, DF407_11=635, DF407_12=635, DF407_13=635, DF407_14=622, DF407_15=622, DF407_16=622, DF407_17=623, DF407_18=623, DF407_19=623, DF407_20=559, DF407_21=558, DF407_22=558, DF407_23=559, DF407_24=559, DF407_25=391, DF407_26=377, DF407_27=377, DF407_28=582, DF407_29=582, DF407_30=582, DF407_31=584, DF407_32=588, DF407_33=619, DF407_34=619, DF407_35=619, DF407_36=619, DF407_37=604, DF407_38=604, DF407_39=604, DF407_40=649, DF407_41=649, DF407_42=649, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0, DF420_21=0, DF420_22=0, DF420_23=0, DF420_24=0, DF420_25=0, DF420_26=0, DF420_27=0, DF420_28=0, DF420_29=0, DF420_30=0, DF420_31=0, DF420_32=0, DF420_33=0, DF420_34=0, DF420_35=0, DF420_36=0, DF420_37=0, DF420_38=0, DF420_39=0, DF420_40=0, DF420_41=0, DF420_42=0, DF408_01=49.4375, DF408_02=52.0625, DF408_03=52.0625, DF408_04=45.6875, DF408_05=54.1875, DF408_06=43.0, DF408_07=31.3125, DF408_08=31.3125, DF408_09=49.5, DF408_10=51.9375, DF408_11=51.9375, DF408_12=45.6875, DF408_13=53.3125, DF408_14=49.0, DF408_15=43.8125, DF408_16=43.8125, DF408_17=50.0625, DF408_18=54.625, DF408_19=50.5, DF408_20=39.75, DF408_21=35.75, DF408_22=35.75, DF408_23=39.5, DF408_24=44.625, DF408_25=34.125, DF408_26=18.625, DF408_27=18.625, DF408_28=48.0, DF408_29=50.625, DF408_30=50.625, DF408_31=44.6875, DF408_32=52.125, DF408_33=40.5625, DF408_34=34.75, DF408_35=34.75, DF408_36=37.5625, DF408_37=40.875, DF408_38=27.375, DF408_39=27.375, DF408_40=38.4375, DF408_41=19.375, DF408_42=19.375)>",
            "<RTCM(1077, DF002=1077, DF003=0, DF004=318945000, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=17762381648302702592, NSat=10, DF395=1346437634, NSig=6, DF396=1132932382022422072, NCell=42, PRN_01=001, PRN_02=002, PRN_03=003, PRN_04=004, PRN_05=006, PRN_06=007, PRN_07=009, PRN_08=017, PRN_09=019, PRN_10=021, DF397_01=68, DF397_02=76, DF397_03=70, DF397_04=67, DF397_05=78, DF397_06=83, DF397_07=71, DF397_08=76, DF397_09=78, DF397_10=79, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, ExtSatInfo_06=0, ExtSatInfo_07=0, ExtSatInfo_08=0, ExtSatInfo_09=0, ExtSatInfo_10=0, DF398_01=0.939453125, DF398_02=0.2998046875, DF398_03=0.3828125, DF398_04=0.841796875, DF398_05=0.4033203125, DF398_06=0.876953125, DF398_07=0.677734375, DF398_08=0.5302734375, DF398_09=0.4326171875, DF398_10=0.4169921875, DF399_01=299, DF399_02=701, DF399_03=95, DF399_04=152, DF399_05=-632, DF399_06=-765, DF399_07=-309, DF399_08=422, DF399_09=101, DF399_10=805, CELLPRN_01=001, CELLSIG_01=1C, CELLPRN_02=001, CELLSIG_02=1W, CELLPRN_03=001, CELLSIG_03=2W, CELLPRN_04=001, CELLSIG_04=2L, CELLPRN_05=001, CELLSIG_05=5Q, CELLPRN_06=002, CELLSIG_06=1C, CELLPRN_07=002, CELLSIG_07=1W, CELLPRN_08=002, CELLSIG_08=2W, CELLPRN_09=003, CELLSIG_09=1C, CELLPRN_10=003, CELLSIG_10=1W, CELLPRN_11=003, CELLSIG_11=2W, CELLPRN_12=003, CELLSIG_12=2L, CELLPRN_13=003, CELLSIG_13=5Q, CELLPRN_14=004, CELLSIG_14=1C, CELLPRN_15=004, CELLSIG_15=1W, CELLPRN_16=004, CELLSIG_16=2W, CELLPRN_17=004, CELLSIG_17=2L, CELLPRN_18=004, CELLSIG_18=5Q, CELLPRN_19=004, CELLSIG_19=1L, CELLPRN_20=006, CELLSIG_20=1C, CELLPRN_21=006, CELLSIG_21=1W, CELLPRN_22=006, CELLSIG_22=2W, CELLPRN_23=006, CELLSIG_23=2L, CELLPRN_24=006, CELLSIG_24=5Q, CELLPRN_25=007, CELLSIG_25=1C, CELLPRN_26=007, CELLSIG_26=1W, CELLPRN_27=007, CELLSIG_27=2W, CELLPRN_28=009, CELLSIG_28=1C, CELLPRN_29=009, CELLSIG_29=1W, CELLPRN_30=009, CELLSIG_30=2W, CELLPRN_31=009, CELLSIG_31=2L, CELLPRN_32=009, CELLSIG_32=5Q, CELLPRN_33=017, CELLSIG_33=1C, CELLPRN_34=017, CELLSIG_34=1W, CELLPRN_35=017, CELLSIG_35=2W, CELLPRN_36=017, CELLSIG_36=2L, CELLPRN_37=019, CELLSIG_37=1C, CELLPRN_38=019, CELLSIG_38=1W, CELLPRN_39=019, CELLSIG_39=2W, CELLPRN_40=021, CELLSIG_40=1C, CELLPRN_41=021, CELLSIG_41=1W, CELLPRN_42=021, CELLSIG_42=2W, DF405_01=0.0003269463777542114, DF405_02=0.00032570958137512207, DF405_03=0.0003551114350557327, DF405_04=0.0003554355353116989, DF405_05=0.0003531314432621002, DF405_06=0.00044614076614379883, DF405_07=0.0004421919584274292, DF405_08=0.00047398172318935394, DF405_09=-0.00043528899550437927, DF405_10=-0.00043588317930698395, DF405_11=-0.00040896423161029816, DF405_12=-0.0004068128764629364, DF405_13=-0.0004071984440088272, DF405_14=0.00043116696178913116, DF405_15=0.0004300791770219803, DF405_16=0.0004497244954109192, DF405_17=0.0004500187933444977, DF405_18=0.00046171993017196655, DF405_19=0.00043155625462532043, DF405_20=-0.00020522251725196838, DF405_21=-0.00020623579621315002, DF405_22=-0.00016688741743564606, DF405_23=-0.00016376934945583344, DF405_24=-0.00015792995691299438, DF405_25=-0.0002510659396648407, DF405_26=-0.00025218352675437927, DF405_27=-0.0001898389309644699, DF405_28=-0.00026627443730831146, DF405_29=-0.0002680625766515732, DF405_30=-0.00024024397134780884, DF405_31=-0.00023896992206573486, DF405_32=-0.00023722276091575623, DF405_33=0.00029537081718444824, DF405_34=0.0002936720848083496, DF405_35=0.0003285575658082962, DF405_36=0.000330507755279541, DF405_37=-0.0004242565482854843, DF405_38=-0.0004289243370294571, DF405_39=-0.0004002377390861511, DF405_40=-0.0001002773642539978, DF405_41=-0.00010454095900058746, DF405_42=-4.892423748970032e-05, DF406_01=0.00029169488698244095, DF406_02=0.00029169488698244095, DF406_03=0.00029748398810625076, DF406_04=0.00029911240562796593, DF406_05=0.0002920180559158325, DF406_06=0.00034324172884225845, DF406_07=0.00034324172884225845, DF406_08=0.00029890984296798706, DF406_09=-0.0004791910760104656, DF406_10=-0.0004791910760104656, DF406_11=-0.000478383619338274, DF406_12=-0.00047512631863355637, DF406_13=-0.00048177503049373627, DF406_14=0.0004361076280474663, DF406_15=0.0004361076280474663, DF406_16=0.00045655621215701103, DF406_17=0.0004695877432823181, DF406_18=0.0004842812195420265, DF406_19=0.0004513435997068882, DF406_20=-0.0001957709901034832, DF406_21=-0.0001957709901034832, DF406_22=-0.0001523960381746292, DF406_23=-0.00015482446178793907, DF406_24=-0.00014885328710079193, DF406_25=-0.00025010714307427406, DF406_26=-0.00025010714307427406, DF406_27=-0.00019579380750656128, DF406_28=-0.00024043582379817963, DF406_29=-0.00024043582379817963, DF406_30=-0.00020435266196727753, DF406_31=-0.00019619334489107132, DF406_32=-0.00018227798864245415, DF406_33=0.00024057738482952118, DF406_34=0.00024057738482952118, DF406_35=0.00022796867415308952, DF406_36=0.00023368187248706818, DF406_37=-0.00045369984582066536, DF406_38=-0.00045369984582066536, DF406_39=-0.0004687071777880192, DF406_40=-0.0002686530351638794, DF406_41=-0.0002686530351638794, DF406_42=-0.000329410657286644, DF407_01=638, DF407_02=638, DF407_03=638, DF407_04=638, DF407_05=638, DF407_06=646, DF407_07=646, DF407_08=646, DF407_09=635, DF407_10=635, DF407_11=635, DF407_12=635, DF407_13=635, DF407_14=622, DF407_15=622, DF407_16=622, DF407_17=623, DF407_18=623, DF407_19=623, DF407_20=559, DF407_21=558, DF407_22=558, DF407_23=559, DF407_24=559, DF407_25=391, DF407_26=377, DF407_27=377, DF407_28=582, DF407_29=582, DF407_30=582, DF407_31=584, DF407_32=588, DF407_33=619, DF407_34=619, DF407_35=619, DF407_36=619, DF407_37=604, DF407_38=604, DF407_39=604, DF407_40=649, DF407_41=649, DF407_42=649, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0, DF420_21=0, DF420_22=0, DF420_23=0, DF420_24=0, DF420_25=0, DF420_26=0, DF420_27=0, DF420_28=0, DF420_29=0, DF420_30=0, DF420_31=0, DF420_32=0, DF420_33=0, DF420_34=0, DF420_35=0, DF420_36=0, DF420_37=0, DF420_38=0, DF420_39=0, DF420_40=0, DF420_41=0, DF420_42=0, DF408_01=49.4375, DF408_02=52.0625, DF408_03=52.0625, DF408_04=45.6875, DF408_05=54.1875, DF408_06=43.0, DF408_07=31.3125, DF408_08=31.3125, DF408_09=49.5, DF408_10=51.9375, DF408_11=51.9375, DF408_12=45.6875, DF408_13=53.3125, DF408_14=49.0, DF408_15=43.8125, DF408_16=43.8125, DF408_17=50.0625, DF408_18=54.625, DF408_19=50.5, DF408_20=39.75, DF408_21=35.75, DF408_22=35.75, DF408_23=39.5, DF408_24=44.625, DF408_25=34.125, DF408_26=18.625, DF408_27=18.625, DF408_28=48.0, DF408_29=50.625, DF408_30=50.625, DF408_31=44.6875, DF408_32=52.125, DF408_33=40.5625, DF408_34=34.75, DF408_35=34.75, DF408_36=37.5625, DF408_37=40.875, DF408_38=27.375, DF408_39=27.375, DF408_40=38.4375, DF408_41=19.375, DF408_42=19.375, DF404_01=-0.274, DF404_02=-0.274, DF404_03=-0.2741, DF404_04=-0.2722, DF404_05=-0.27290000000000003, DF404_06=-0.1899, DF404_07=-0.1899, DF404_08=-0.192, DF404_09=0.41850000000000004, DF404_10=0.41850000000000004, DF404_11=0.4178, DF404_12=0.40740000000000004, DF404_13=0.4188, DF404_14=-0.27340000000000003, DF404_15=-0.27340000000000003, DF404_16=-0.2728, DF404_17=-0.275, DF404_18=-0.2743, DF404_19=-0.2702, DF404_20=0.0712, DF404_21=0.0712, DF404_22=0.0732, DF404_23=0.0863, DF404_24=0.0383, DF404_25=0.06430000000000001, DF404_26=0.06430000000000001, DF404_27=0.068, DF404_28=0.3657, DF404_29=0.3657, DF404_30=0.366, DF404_31=0.3947, DF404_32=0.37670000000000003, DF404_33=0.4339, DF404_34=0.4339, DF404_35=0.4335, DF404_36=0.454, DF404_37=-0.4262, DF404_38=-0.4262, DF404_39=-0.42710000000000004, DF404_40=-0.21150000000000002, DF404_41=-0.21150000000000002, DF404_42=-0.2124)>",
            "<RTCM(1086, DF002=1086, DF003=0, DF416=3, DF034=70527000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=9493595711078400000, NSat=8, DF395=1635778560, NSig=4, DF396=4294954959, NCell=28, PRN_01=001, PRN_02=007, PRN_03=008, PRN_04=009, PRN_05=010, PRN_06=022, PRN_07=023, PRN_08=024, DF397_01=74, DF397_02=72, DF397_03=66, DF397_04=75, DF397_05=75, DF397_06=70, DF397_07=65, DF397_08=74, DF398_01=0.91015625, DF398_02=0.5732421875, DF398_03=0.994140625, DF398_04=0.0751953125, DF398_05=0.916015625, DF398_06=0.5634765625, DF398_07=0.6181640625, DF398_08=0.9873046875, CELLPRN_01=001, CELLSIG_01=1C, CELLPRN_02=001, CELLSIG_02=1P, CELLPRN_03=001, CELLSIG_03=2C, CELLPRN_04=001, CELLSIG_04=2P, CELLPRN_05=007, CELLSIG_05=1C, CELLPRN_06=007, CELLSIG_06=1P, CELLPRN_07=007, CELLSIG_07=2C, CELLPRN_08=007, CELLSIG_08=2P, CELLPRN_09=008, CELLSIG_09=1C, CELLPRN_10=008, CELLSIG_10=1P, CELLPRN_11=008, CELLSIG_11=2C, CELLPRN_12=008, CELLSIG_12=2P, CELLPRN_13=009, CELLSIG_13=1C, CELLPRN_14=009, CELLSIG_14=1P, CELLPRN_15=009, CELLSIG_15=2C, CELLPRN_16=009, CELLSIG_16=2P, CELLPRN_17=010, CELLSIG_17=1C, CELLPRN_18=010, CELLSIG_18=1P, CELLPRN_19=022, CELLSIG_19=1C, CELLPRN_20=022, CELLSIG_20=1P, CELLPRN_21=022, CELLSIG_21=2C, CELLPRN_22=022, CELLSIG_22=2P, CELLPRN_23=023, CELLSIG_23=1C, CELLPRN_24=023, CELLSIG_24=1P, CELLPRN_25=024, CELLSIG_25=1C, CELLPRN_26=024, CELLSIG_26=1P, CELLPRN_27=024, CELLSIG_27=2C, CELLPRN_28=024, CELLSIG_28=2P, DF405_01=-0.00023334287106990814, DF405_02=-0.00023224949836730957, DF405_03=-0.00018310174345970154, DF405_04=-0.00018067844212055206, DF405_05=-0.0001502353698015213, DF405_06=-0.00015071406960487366, DF405_07=-0.00010378658771514893, DF405_08=-0.00010492466390132904, DF405_09=-8.18353146314621e-05, DF405_10=-7.955357432365417e-05, DF405_11=-5.025044083595276e-05, DF405_12=-5.0496309995651245e-05, DF405_13=-1.8831342458724976e-05, DF405_14=-1.8088147044181824e-05, DF405_15=4.15947288274765e-05, DF405_16=3.8605183362960815e-05, DF405_17=-0.0001417100429534912, DF405_18=-0.0001466795802116394, DF405_19=-0.0003603212535381317, DF405_20=-0.0003593824803829193, DF405_21=-0.0003201998770236969, DF405_22=-0.00031863898038864136, DF405_23=-9.093619883060455e-05, DF405_24=-9.151175618171692e-05, DF405_25=-0.00020238198339939117, DF405_26=-0.00020118243992328644, DF405_27=-0.00015542283654212952, DF405_28=-0.00015581399202346802, DF406_01=-0.00019364943727850914, DF406_02=-0.00021111825481057167, DF406_03=-0.00016902713105082512, DF406_04=-0.00017220014706254005, DF406_05=-0.00022841757163405418, DF406_06=-0.00022093765437602997, DF406_07=-0.0002245735377073288, DF406_08=-0.00021815858781337738, DF406_09=-6.929738447070122e-05, DF406_10=-6.867013871669769e-05, DF406_11=-2.8275419026613235e-05, DF406_12=-3.707828000187874e-05, DF406_13=1.9595958292484283e-05, DF406_14=2.709636464715004e-05, DF406_15=0.00012388639152050018, DF406_16=0.00012632180005311966, DF406_17=-9.543681517243385e-05, DF406_18=-9.918259456753731e-05, DF406_19=-0.00042520975694060326, DF406_20=-0.0004164576530456543, DF406_21=-0.00041651399806141853, DF406_22=-0.0004229256883263588, DF406_23=-8.106138557195663e-05, DF406_24=-8.666981011629105e-05, DF406_25=-0.00018213782459497452, DF406_26=-0.0001752721145749092, DF406_27=-0.0001103552058339119, DF406_28=-0.00012638932093977928, DF407_01=540, DF407_02=541, DF407_03=479, DF407_04=455, DF407_05=641, DF407_06=640, DF407_07=641, DF407_08=641, DF407_09=621, DF407_10=621, DF407_11=621, DF407_12=621, DF407_13=578, DF407_14=578, DF407_15=578, DF407_16=578, DF407_17=549, DF407_18=548, DF407_19=637, DF407_20=638, DF407_21=637, DF407_22=638, DF407_23=612, DF407_24=612, DF407_25=550, DF407_26=549, DF407_27=546, DF407_28=546, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0, DF420_21=0, DF420_22=0, DF420_23=0, DF420_24=0, DF420_25=0, DF420_26=0, DF420_27=0, DF420_28=0, DF408_01=41.5625, DF408_02=40.5, DF408_03=35.5, DF408_04=35.125, DF408_05=48.1875, DF408_06=47.4375, DF408_07=44.8125, DF408_08=45.3125, DF408_09=52.625, DF408_10=52.5, DF408_11=49.5625, DF408_12=49.875, DF408_13=44.4375, DF408_14=43.9375, DF408_15=41.6875, DF408_16=41.125, DF408_17=38.875, DF408_18=38.5625, DF408_19=49.6875, DF408_20=49.5, DF408_21=46.3125, DF408_22=47.0625, DF408_23=50.875, DF408_24=50.75, DF408_25=45.75, DF408_26=44.75, DF408_27=42.8125, DF408_28=42.6875)>",
            "<RTCM(1087, DF002=1087, DF003=0, DF416=3, DF034=70527000, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=9493595711078400000, NSat=8, DF395=1635778560, NSig=4, DF396=4294954959, NCell=28, PRN_01=001, PRN_02=007, PRN_03=008, PRN_04=009, PRN_05=010, PRN_06=022, PRN_07=023, PRN_08=024, DF397_01=75, DF397_02=72, DF397_03=67, DF397_04=75, DF397_05=76, DF397_06=70, DF397_07=65, DF397_08=75, DF419_01=8, DF419_02=12, DF419_03=13, DF419_04=5, DF419_05=0, DF419_06=4, DF419_07=10, DF419_08=9, DF398_01=0.26953125, DF398_02=0.9326171875, DF398_03=0.353515625, DF398_04=0.4345703125, DF398_05=0.275390625, DF398_06=0.9228515625, DF398_07=0.9775390625, DF398_08=0.3466796875, DF399_01=-387, DF399_02=458, DF399_03=80, DF399_04=-117, DF399_05=-560, DF399_06=720, DF399_07=47, DF399_08=-638, CELLPRN_01=001, CELLSIG_01=1C, CELLPRN_02=001, CELLSIG_02=1P, CELLPRN_03=001, CELLSIG_03=2C, CELLPRN_04=001, CELLSIG_04=2P, CELLPRN_05=007, CELLSIG_05=1C, CELLPRN_06=007, CELLSIG_06=1P, CELLPRN_07=007, CELLSIG_07=2C, CELLPRN_08=007, CELLSIG_08=2P, CELLPRN_09=008, CELLSIG_09=1C, CELLPRN_10=008, CELLSIG_10=1P, CELLPRN_11=008, CELLSIG_11=2C, CELLPRN_12=008, CELLSIG_12=2P, CELLPRN_13=009, CELLSIG_13=1C, CELLPRN_14=009, CELLSIG_14=1P, CELLPRN_15=009, CELLSIG_15=2C, CELLPRN_16=009, CELLSIG_16=2P, CELLPRN_17=010, CELLSIG_17=1C, CELLPRN_18=010, CELLSIG_18=1P, CELLPRN_19=022, CELLSIG_19=1C, CELLPRN_20=022, CELLSIG_20=1P, CELLPRN_21=022, CELLSIG_21=2C, CELLPRN_22=022, CELLSIG_22=2P, CELLPRN_23=023, CELLSIG_23=1C, CELLPRN_24=023, CELLSIG_24=1P, CELLPRN_25=024, CELLSIG_25=1C, CELLPRN_26=024, CELLSIG_26=1P, CELLPRN_27=024, CELLSIG_27=2C, CELLPRN_28=024, CELLSIG_28=2P, DF405_01=-0.00020707584917545319, DF405_02=-0.00020598247647285461, DF405_03=-0.00015683472156524658, DF405_04=-0.00015441328287124634, DF405_05=-0.00012498348951339722, DF405_06=-0.00012546032667160034, DF405_07=-7.853470742702484e-05, DF405_08=-7.967278361320496e-05, DF405_09=-5.612894892692566e-05, DF405_10=-5.384907126426697e-05, DF405_11=-2.454407513141632e-05, DF405_12=-2.4791806936264038e-05, DF405_13=7.111579179763794e-06, DF405_14=7.854774594306946e-06, DF405_15=6.753765046596527e-05, DF405_16=6.454810500144958e-05, DF405_17=-0.00011523626744747162, DF405_18=-0.00012020580470561981, DF405_19=-0.00033538229763507843, DF405_20=-0.00033444352447986603, DF405_21=-0.0002952609211206436, DF405_22=-0.0002937000244855881, DF405_23=-6.519071757793427e-05, DF405_24=-6.576627492904663e-05, DF405_25=-0.00017581507563591003, DF405_26=-0.0001746155321598053, DF405_27=-0.00012885592877864838, DF405_28=-0.00012924708425998688, DF406_01=-0.00016738241538405418, DF406_02=-0.00018485169857740402, DF406_03=-0.00014276057481765747, DF406_04=-0.0001459335908293724, DF406_05=-0.00020316476002335548, DF406_06=-0.00019568530842661858, DF406_07=-0.0001993211917579174, DF406_08=-0.000192906241863966, DF406_09=-4.359148442745209e-05, DF406_10=-4.296423867344856e-05, DF406_11=-2.5695189833641052e-06, DF406_12=-1.1372379958629608e-05, DF406_13=4.553887993097305e-05, DF406_14=5.303928628563881e-05, DF406_15=0.00014982931315898895, DF406_16=0.00015226472169160843, DF406_17=-6.896303966641426e-05, DF406_18=-7.270928472280502e-05, DF406_19=-0.00040027033537626266, DF406_20=-0.0003915182314813137, DF406_21=-0.00039157457649707794, DF406_22=-0.0003979862667620182, DF406_23=-5.531543865799904e-05, DF406_24=-6.092386320233345e-05, DF406_25=-0.00015557045117020607, DF406_26=-0.00014870474115014076, DF406_27=-8.378783240914345e-05, DF406_28=-9.982194751501083e-05, DF407_01=540, DF407_02=541, DF407_03=479, DF407_04=455, DF407_05=641, DF407_06=640, DF407_07=641, DF407_08=641, DF407_09=621, DF407_10=621, DF407_11=621, DF407_12=621, DF407_13=578, DF407_14=578, DF407_15=578, DF407_16=578, DF407_17=549, DF407_18=548, DF407_19=637, DF407_20=638, DF407_21=637, DF407_22=638, DF407_23=612, DF407_24=612, DF407_25=550, DF407_26=549, DF407_27=546, DF407_28=546, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0, DF420_21=0, DF420_22=0, DF420_23=0, DF420_24=0, DF420_25=0, DF420_26=0, DF420_27=0, DF420_28=0, DF408_01=41.5625, DF408_02=40.5, DF408_03=35.5, DF408_04=35.125, DF408_05=48.1875, DF408_06=47.4375, DF408_07=44.8125, DF408_08=45.3125, DF408_09=52.625, DF408_10=52.5, DF408_11=49.5625, DF408_12=49.875, DF408_13=44.4375, DF408_14=43.9375, DF408_15=41.6875, DF408_16=41.125, DF408_17=38.875, DF408_18=38.5625, DF408_19=49.6875, DF408_20=49.5, DF408_21=46.3125, DF408_22=47.0625, DF408_23=50.875, DF408_24=50.75, DF408_25=45.75, DF408_26=44.75, DF408_27=42.8125, DF408_28=42.6875, DF404_01=-0.41440000000000005, DF404_02=-0.3942, DF404_03=-0.4222, DF404_04=-0.3925, DF404_05=0.39380000000000004, DF404_06=0.4052, DF404_07=0.4078, DF404_08=0.4469, DF404_09=0.43010000000000004, DF404_10=0.43360000000000004, DF404_11=0.43670000000000003, DF404_12=0.47240000000000004, DF404_13=-0.39480000000000004, DF404_14=-0.38130000000000003, DF404_15=-0.4123, DF404_16=-0.3614, DF404_17=0.10700000000000001, DF404_18=0.1277, DF404_19=-0.1338, DF404_20=-0.13190000000000002, DF404_21=-0.1353, DF404_22=-0.07200000000000001, DF404_23=0.12610000000000002, DF404_24=0.1207, DF404_25=-0.1998, DF404_26=-0.19690000000000002, DF404_27=-0.18660000000000002, DF404_28=-0.1595)>",
            "<RTCM(1096, DF002=1096, DF003=0, DF248=318945000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=2957246475140071424, NSat=7, DF395=1090658816, NSig=5, DF396=34359738367, NCell=35, PRN_01=003, PRN_02=005, PRN_03=008, PRN_04=013, PRN_05=015, PRN_06=018, PRN_07=034, DF397_01=79, DF397_02=87, DF397_03=80, DF397_04=83, DF397_05=77, DF397_06=92, DF397_07=86, DF398_01=0.6171875, DF398_02=0.1396484375, DF398_03=0.4052734375, DF398_04=0.79296875, DF398_05=0.87109375, DF398_06=0.8056640625, DF398_07=0.8779296875, CELLPRN_01=003, CELLSIG_01=1C, CELLPRN_02=003, CELLSIG_02=6C, CELLPRN_03=003, CELLSIG_03=7Q, CELLPRN_04=003, CELLSIG_04=8Q, CELLPRN_05=003, CELLSIG_05=5Q, CELLPRN_06=005, CELLSIG_06=1C, CELLPRN_07=005, CELLSIG_07=6C, CELLPRN_08=005, CELLSIG_08=7Q, CELLPRN_09=005, CELLSIG_09=8Q, CELLPRN_10=005, CELLSIG_10=5Q, CELLPRN_11=008, CELLSIG_11=1C, CELLPRN_12=008, CELLSIG_12=6C, CELLPRN_13=008, CELLSIG_13=7Q, CELLPRN_14=008, CELLSIG_14=8Q, CELLPRN_15=008, CELLSIG_15=5Q, CELLPRN_16=013, CELLSIG_16=1C, CELLPRN_17=013, CELLSIG_17=6C, CELLPRN_18=013, CELLSIG_18=7Q, CELLPRN_19=013, CELLSIG_19=8Q, CELLPRN_20=013, CELLSIG_20=5Q, CELLPRN_21=015, CELLSIG_21=1C, CELLPRN_22=015, CELLSIG_22=6C, CELLPRN_23=015, CELLSIG_23=7Q, CELLPRN_24=015, CELLSIG_24=8Q, CELLPRN_25=015, CELLSIG_25=5Q, CELLPRN_26=018, CELLSIG_26=1C, CELLPRN_27=018, CELLSIG_27=6C, CELLPRN_28=018, CELLSIG_28=7Q, CELLPRN_29=018, CELLSIG_29=8Q, CELLPRN_30=018, CELLSIG_30=5Q, CELLPRN_31=034, CELLSIG_31=1C, CELLPRN_32=034, CELLSIG_32=6C, CELLPRN_33=034, CELLSIG_33=7Q, CELLPRN_34=034, CELLSIG_34=8Q, CELLPRN_35=034, CELLSIG_35=5Q, DF405_01=-0.00029922090470790863, DF405_02=-0.0002804826945066452, DF405_03=-0.0002743639051914215, DF405_04=-0.0002722162753343582, DF405_05=-0.0002680569887161255, DF405_06=2.382323145866394e-06, DF405_07=4.805624485015869e-05, DF405_08=6.577372550964355e-05, DF405_09=7.22687691450119e-05, DF405_10=7.960572838783264e-05, DF405_11=0.0004021245986223221, DF405_12=0.0004198439419269562, DF405_13=0.0004223603755235672, DF405_14=0.0004248451441526413, DF405_15=0.00042909011244773865, DF405_16=8.045509457588196e-05, DF405_17=0.00010089576244354248, DF405_18=0.00011058524250984192, DF405_19=0.00011371448636054993, DF405_20=0.0001178886741399765, DF405_21=0.0002819858491420746, DF405_22=0.0003008861094713211, DF405_23=0.00030783191323280334, DF405_24=0.0003102179616689682, DF405_25=0.00031445547938346863, DF405_26=0.0003162119537591934, DF405_27=0.0003452245146036148, DF405_28=0.0003542546182870865, DF405_29=0.00035850889980793, DF405_30=0.0003632865846157074, DF405_31=0.000373179093003273, DF405_32=0.00041130557656288147, DF405_33=0.00042248331010341644, DF405_34=0.00042534805834293365, DF405_35=0.0004306994378566742, DF406_01=-0.00032781297340989113, DF406_02=-0.0003255079500377178, DF406_03=-0.00032354844734072685, DF406_04=-0.00032388046383857727, DF406_05=-0.00032264692708849907, DF406_06=-0.00018738536164164543, DF406_07=-0.00023872079327702522, DF406_08=-0.00025241170078516006, DF406_09=-0.000255427323281765, DF406_10=-0.0002586361952126026, DF406_11=0.00041880039498209953, DF406_12=0.0004542004317045212, DF406_13=0.00046154437586665154, DF406_14=0.00046989740803837776, DF406_15=0.0004706927575170994, DF406_16=8.848030120134354e-05, DF406_17=0.0001138630323112011, DF406_18=0.00012563355267047882, DF406_19=0.00013115163892507553, DF406_20=0.00013158423826098442, DF406_21=0.0002521807327866554, DF406_22=0.0002468419261276722, DF406_23=0.00024701282382011414, DF406_24=0.00025324057787656784, DF406_25=0.00025270553305745125, DF406_26=0.00022117700427770615, DF406_27=0.00021222885698080063, DF406_28=0.00020291702821850777, DF406_29=0.00020459666848182678, DF406_30=0.00020449282601475716, DF406_31=0.0002513905055820942, DF406_32=0.0002280808985233307, DF406_33=0.00021671084687113762, DF406_34=0.0002219090238213539, DF406_35=0.00021351175382733345, DF407_01=642, DF407_02=643, DF407_03=643, DF407_04=643, DF407_05=643, DF407_06=653, DF407_07=654, DF407_08=654, DF407_09=654, DF407_10=654, DF407_11=617, DF407_12=617, DF407_13=617, DF407_14=617, DF407_15=617, DF407_16=599, DF407_17=600, DF407_18=600, DF407_19=600, DF407_20=600, DF407_21=643, DF407_22=643, DF407_23=643, DF407_24=643, DF407_25=643, DF407_26=647, DF407_27=647, DF407_28=647, DF407_29=647, DF407_30=647, DF407_31=664, DF407_32=667, DF407_33=667, DF407_34=667, DF407_35=667, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0, DF420_21=0, DF420_22=0, DF420_23=0, DF420_24=0, DF420_25=0, DF420_26=0, DF420_27=0, DF420_28=0, DF420_29=0, DF420_30=0, DF420_31=0, DF420_32=0, DF420_33=0, DF420_34=0, DF420_35=0, DF408_01=49.3125, DF408_02=51.5625, DF408_03=52.5625, DF408_04=55.125, DF408_05=51.6875, DF408_06=37.9375, DF408_07=37.5625, DF408_08=39.9375, DF408_09=42.9375, DF408_10=40.125, DF408_11=48.75, DF408_12=51.3125, DF408_13=51.5625, DF408_14=54.1875, DF408_15=50.75, DF408_16=45.75, DF408_17=48.125, DF408_18=48.625, DF408_19=51.5, DF408_20=48.375, DF408_21=49.1875, DF408_22=52.1875, DF408_23=52.5625, DF408_24=55.375, DF408_25=52.1875, DF408_26=41.0, DF408_27=43.1875, DF408_28=43.1875, DF408_29=45.75, DF408_30=42.1875, DF408_31=41.875, DF408_32=44.3125, DF408_33=45.6875, DF408_34=48.3125, DF408_35=44.8125)>",
            "<RTCM(1097, DF002=1097, DF003=0, DF248=318945000, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=2957246475140071424, NSat=7, DF395=1090658816, NSig=5, DF396=34359738367, NCell=35, PRN_01=003, PRN_02=005, PRN_03=008, PRN_04=013, PRN_05=015, PRN_06=018, PRN_07=034, DF397_01=79, DF397_02=87, DF397_03=80, DF397_04=84, DF397_05=78, DF397_06=93, DF397_07=87, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, ExtSatInfo_06=0, ExtSatInfo_07=0, DF398_01=0.9765625, DF398_02=0.4990234375, DF398_03=0.7646484375, DF398_04=0.15234375, DF398_05=0.23046875, DF398_06=0.1650390625, DF398_07=0.2373046875, DF399_01=243, DF399_02=467, DF399_03=-183, DF399_04=-394, DF399_05=119, DF399_06=-2, DF399_07=521, CELLPRN_01=003, CELLSIG_01=1C, CELLPRN_02=003, CELLSIG_02=6C, CELLPRN_03=003, CELLSIG_03=7Q, CELLPRN_04=003, CELLSIG_04=8Q, CELLPRN_05=003, CELLSIG_05=5Q, CELLPRN_06=005, CELLSIG_06=1C, CELLPRN_07=005, CELLSIG_07=6C, CELLPRN_08=005, CELLSIG_08=7Q, CELLPRN_09=005, CELLSIG_09=8Q, CELLPRN_10=005, CELLSIG_10=5Q, CELLPRN_11=008, CELLSIG_11=1C, CELLPRN_12=008, CELLSIG_12=6C, CELLPRN_13=008, CELLSIG_13=7Q, CELLPRN_14=008, CELLSIG_14=8Q, CELLPRN_15=008, CELLSIG_15=5Q, CELLPRN_16=013, CELLSIG_16=1C, CELLPRN_17=013, CELLSIG_17=6C, CELLPRN_18=013, CELLSIG_18=7Q, CELLPRN_19=013, CELLSIG_19=8Q, CELLPRN_20=013, CELLSIG_20=5Q, CELLPRN_21=015, CELLSIG_21=1C, CELLPRN_22=015, CELLSIG_22=6C, CELLPRN_23=015, CELLSIG_23=7Q, CELLPRN_24=015, CELLSIG_24=8Q, CELLPRN_25=015, CELLSIG_25=5Q, CELLPRN_26=018, CELLSIG_26=1C, CELLPRN_27=018, CELLSIG_27=6C, CELLPRN_28=018, CELLSIG_28=7Q, CELLPRN_29=018, CELLSIG_29=8Q, CELLPRN_30=018, CELLSIG_30=5Q, CELLPRN_31=034, CELLSIG_31=1C, CELLPRN_32=034, CELLSIG_32=6C, CELLPRN_33=034, CELLSIG_33=7Q, CELLPRN_34=034, CELLSIG_34=8Q, CELLPRN_35=034, CELLSIG_35=5Q, DF405_01=-0.00027371011674404144, DF405_02=-0.000254971906542778, DF405_03=-0.0002488531172275543, DF405_04=-0.0002467036247253418, DF405_05=-0.0002425462007522583, DF405_06=2.7624890208244324e-05, DF405_07=7.329881191253662e-05, DF405_08=9.101629257202148e-05, DF405_09=9.751133620738983e-05, DF405_10=0.00010484829545021057, DF405_11=0.0004281476140022278, DF405_12=0.0004458669573068619, DF405_13=0.00044838152825832367, DF405_14=0.00045086629688739777, DF405_15=0.0004551112651824951, DF405_16=0.00010672770440578461, DF405_17=0.00012717023491859436, DF405_18=0.0001368597149848938, DF405_19=0.0001399889588356018, DF405_20=0.00014416128396987915, DF405_21=0.00030764564871788025, DF405_22=0.00032654590904712677, DF405_23=0.000333491712808609, DF405_24=0.00033587776124477386, DF405_25=0.0003401152789592743, DF405_26=0.00034201517701148987, DF405_27=0.00037102773785591125, DF405_28=0.00038005784153938293, DF405_29=0.00038431398570537567, DF405_30=0.00038908980786800385, DF405_31=0.00039835646748542786, DF405_32=0.0004364829510450363, DF405_33=0.0004476606845855713, DF405_34=0.0004505254328250885, DF405_35=0.00045587681233882904, DF406_01=-0.00030230171978473663, DF406_02=-0.0002999966964125633, DF406_03=-0.00029803719371557236, DF406_04=-0.0002983692102134228, DF406_05=-0.0002971356734633446, DF406_06=-0.0001621423289179802, DF406_07=-0.00021347776055335999, DF406_08=-0.00022716913372278214, DF406_09=-0.00023018475621938705, DF406_10=-0.00023339316248893738, DF406_11=0.0004448220133781433, DF406_12=0.00048022205010056496, DF406_13=0.0004875659942626953, DF406_14=0.0004959190264344215, DF406_15=0.0004967143759131432, DF406_16=0.00011475430801510811, DF406_17=0.00014013703912496567, DF406_18=0.0001519075594842434, DF406_19=0.0001574256457388401, DF406_20=0.0001578587107360363, DF406_21=0.0002778405323624611, DF406_22=0.00027250172570347786, DF406_23=0.0002726730890572071, DF406_24=0.0002789008431136608, DF406_25=0.0002783653326332569, DF406_26=0.0002469811588525772, DF406_27=0.0002380330115556717, DF406_28=0.00022872164845466614, DF406_29=0.00023040128871798515, DF406_30=0.00023029744625091553, DF406_31=0.00027656788006424904, DF406_32=0.00025325873866677284, DF406_33=0.00024188868701457977, DF406_34=0.00024708686396479607, DF406_35=0.0002386895939707756, DF407_01=642, DF407_02=643, DF407_03=643, DF407_04=643, DF407_05=643, DF407_06=653, DF407_07=654, DF407_08=654, DF407_09=654, DF407_10=654, DF407_11=617, DF407_12=617, DF407_13=617, DF407_14=617, DF407_15=617, DF407_16=599, DF407_17=600, DF407_18=600, DF407_19=600, DF407_20=600, DF407_21=643, DF407_22=643, DF407_23=643, DF407_24=643, DF407_25=643, DF407_26=647, DF407_27=647, DF407_28=647, DF407_29=647, DF407_30=647, DF407_31=664, DF407_32=667, DF407_33=667, DF407_34=667, DF407_35=667, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0, DF420_21=0, DF420_22=0, DF420_23=0, DF420_24=0, DF420_25=0, DF420_26=0, DF420_27=0, DF420_28=0, DF420_29=0, DF420_30=0, DF420_31=0, DF420_32=0, DF420_33=0, DF420_34=0, DF420_35=0, DF408_01=49.3125, DF408_02=51.5625, DF408_03=52.5625, DF408_04=55.125, DF408_05=51.6875, DF408_06=37.9375, DF408_07=37.5625, DF408_08=39.9375, DF408_09=42.9375, DF408_10=40.125, DF408_11=48.75, DF408_12=51.3125, DF408_13=51.5625, DF408_14=54.1875, DF408_15=50.75, DF408_16=45.75, DF408_17=48.125, DF408_18=48.625, DF408_19=51.5, DF408_20=48.375, DF408_21=49.1875, DF408_22=52.1875, DF408_23=52.5625, DF408_24=55.375, DF408_25=52.1875, DF408_26=41.0, DF408_27=43.1875, DF408_28=43.1875, DF408_29=45.75, DF408_30=42.1875, DF408_31=41.875, DF408_32=44.3125, DF408_33=45.6875, DF408_34=48.3125, DF408_35=44.8125, DF404_01=-0.2341, DF404_02=-0.22640000000000002, DF404_03=-0.2381, DF404_04=-0.23450000000000001, DF404_05=-0.2369, DF404_06=-0.2247, DF404_07=-0.2048, DF404_08=-0.24380000000000002, DF404_09=-0.2461, DF404_10=-0.24050000000000002, DF404_11=0.0799, DF404_12=0.0772, DF404_13=0.0862, DF404_14=0.0917, DF404_15=0.0918, DF404_16=0.42660000000000003, DF404_17=0.4289, DF404_18=0.43, DF404_19=0.42460000000000003, DF404_20=0.4153, DF404_21=-0.46480000000000005, DF404_22=-0.4746, DF404_23=-0.4731, DF404_24=-0.46900000000000003, DF404_25=-0.468, DF404_26=0.2884, DF404_27=0.2722, DF404_28=0.27540000000000003, DF404_29=0.2775, DF404_30=0.2833, DF404_31=0.1451, DF404_32=0.1522, DF404_33=0.1306, DF404_34=0.1413, DF404_35=0.1463)>",
            "<RTCM(1106, DF002=1106, DF003=0, DF004=318945000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=4503599660924928, NSat=2, DF395=1073742336, NSig=2, DF396=14, NCell=3, PRN_01=131, PRN_02=158, DF397_01=129, DF397_02=122, DF398_01=0.5390625, DF398_02=0.8984375, CELLPRN_01=131, CELLSIG_01=1C, CELLPRN_02=131, CELLSIG_02=5Q, CELLPRN_03=158, CELLSIG_03=1C, DF405_01=0.0003006812185049057, DF405_02=0.000264560803771019, DF405_03=0.00017976947128772736, DF406_01=0.0002510044723749161, DF406_02=0.0002635270357131958, DF406_03=0.0001455545425415039, DF407_01=704, DF407_02=438, DF407_03=523, DF420_01=0, DF420_02=0, DF420_03=0, DF408_01=40.8125, DF408_02=38.3125, DF408_03=43.5625)>",
            "<RTCM(1107, DF002=1107, DF003=0, DF004=318945000, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=4503599660924928, NSat=2, DF395=1073742336, NSig=2, DF396=14, NCell=3, PRN_01=131, PRN_02=158, DF397_01=129, DF397_02=123, ExtSatInfo_01=0, ExtSatInfo_02=0, DF398_01=0.8984375, DF398_02=0.2578125, DF399_01=0, DF399_02=20, CELLPRN_01=131, CELLSIG_01=1C, CELLPRN_02=131, CELLSIG_02=5Q, CELLPRN_03=158, CELLSIG_03=1C, DF405_01=0.0003264825791120529, DF405_02=0.00029036402702331543, DF405_03=0.0002055484801530838, DF406_01=0.0002768067643046379, DF406_02=0.0002893288619816303, DF406_03=0.00017133262008428574, DF407_01=704, DF407_02=438, DF407_03=523, DF420_01=0, DF420_02=0, DF420_03=0, DF408_01=40.8125, DF408_02=38.3125, DF408_03=43.5625, DF404_01=0.0145, DF404_02=0.002, DF404_03=-0.0407)>",
            "<RTCM(1116, DF002=1116, DF003=0, DF428=318945000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=0, NSat=0, DF395=0, NSig=0, DF396=0, NCell=0)>",
            "<RTCM(1117, DF002=1117, DF003=0, DF428=318945000, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=0, NSat=0, DF395=0, NSig=0, DF396=0, NCell=0)>",
            "<RTCM(1126, DF002=1126, DF003=0, DF427=318931000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=4560809532588160, NSat=11, DF395=1090781184, NSig=3, DF396=8436542902, NCell=23, PRN_01=012, PRN_02=019, PRN_03=020, PRN_04=022, PRN_05=029, PRN_06=035, PRN_07=036, PRN_08=037, PRN_09=044, PRN_10=046, PRN_11=057, DF397_01=88, DF397_02=74, DF397_03=81, DF397_04=80, DF397_05=81, DF397_06=76, DF397_07=85, DF397_08=86, DF397_09=82, DF397_10=81, DF397_11=74, DF398_01=0.2724609375, DF398_02=0.6806640625, DF398_03=0.0146484375, DF398_04=0.1748046875, DF398_05=0.9921875, DF398_06=0.138671875, DF398_07=0.1630859375, DF398_08=0.6259765625, DF398_09=0.541015625, DF398_10=0.150390625, DF398_11=0.076171875, CELLPRN_01=012, CELLSIG_01=2I, CELLPRN_02=012, CELLSIG_02=6I, CELLPRN_03=012, CELLSIG_03=7I, CELLPRN_04=019, CELLSIG_04=2I, CELLPRN_05=019, CELLSIG_05=6I, CELLPRN_06=020, CELLSIG_06=2I, CELLPRN_07=020, CELLSIG_07=6I, CELLPRN_08=022, CELLSIG_08=2I, CELLPRN_09=022, CELLSIG_09=6I, CELLPRN_10=029, CELLSIG_10=2I, CELLPRN_11=029, CELLSIG_11=6I, CELLPRN_12=035, CELLSIG_12=2I, CELLPRN_13=035, CELLSIG_13=6I, CELLPRN_14=036, CELLSIG_14=2I, CELLPRN_15=036, CELLSIG_15=6I, CELLPRN_16=037, CELLSIG_16=2I, CELLPRN_17=037, CELLSIG_17=6I, CELLPRN_18=044, CELLSIG_18=2I, CELLPRN_19=044, CELLSIG_19=6I, CELLPRN_20=046, CELLSIG_20=2I, CELLPRN_21=046, CELLSIG_21=6I, CELLPRN_22=057, CELLSIG_22=2I, CELLPRN_23=057, CELLSIG_23=6I, DF405_01=0.0003019813448190689, DF405_02=0.00033625587821006775, DF405_03=0.00034762173891067505, DF405_04=-0.00036572664976119995, DF405_05=-0.00034845247864723206, DF405_06=0.00020186975598335266, DF405_07=0.00023330748081207275, DF405_08=3.952533006668091e-05, DF405_09=5.9451907873153687e-05, DF405_10=0.0004642121493816376, DF405_11=0.0005089379847049713, DF405_12=-0.0004883836954832077, DF405_13=-0.00045639462769031525, DF405_14=7.497146725654602e-06, DF405_15=8.726678788661957e-05, DF405_16=-0.00043341144919395447, DF405_17=-0.0003605857491493225, DF405_18=0.00012992322444915771, DF405_19=0.00016711466014385223, DF405_20=0.0004669707268476486, DF405_21=0.0004929937422275543, DF405_22=-0.0004192069172859192, DF405_23=2.4376437067985535e-05, DF406_01=0.0002920781262218952, DF406_02=0.00031609833240509033, DF406_03=0.0003256821073591709, DF406_04=-0.00038290582597255707, DF406_05=-0.0003754766657948494, DF406_06=7.67214223742485e-05, DF406_07=3.757607191801071e-05, DF406_08=5.945516750216484e-05, DF406_09=8.559366688132286e-05, DF406_10=0.00034990115091204643, DF406_11=0.00033590663224458694, DF406_12=-0.0005094590596854687, DF406_13=-0.0004914868623018265, DF406_14=3.8984231650829315e-05, DF406_15=0.00013488112017512321, DF406_16=-0.0004383223131299019, DF406_17=-0.00037168897688388824, DF406_18=0.00012395810335874557, DF406_19=0.0001520337536931038, DF406_20=0.0005132271908223629, DF406_21=0.0005561341531574726, DF406_22=-0.00042449310421943665, DF406_23=1.592608168721199e-05, DF407_01=517, DF407_02=519, DF407_03=519, DF407_04=635, DF407_05=635, DF407_06=650, DF407_07=650, DF407_08=590, DF407_09=590, DF407_10=650, DF407_11=650, DF407_12=621, DF407_13=621, DF407_14=552, DF407_15=555, DF407_16=594, DF407_17=594, DF407_18=581, DF407_19=580, DF407_20=587, DF407_21=587, DF407_22=609, DF407_23=609, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0, DF420_21=0, DF420_22=0, DF420_23=0, DF408_01=34.8125, DF408_02=39.5, DF408_03=42.4375, DF408_04=53.25, DF408_05=51.9375, DF408_06=47.1875, DF408_07=45.6875, DF408_08=47.5625, DF408_09=46.4375, DF408_10=44.4375, DF408_11=42.5, DF408_12=50.25, DF408_13=48.0, DF408_14=42.5, DF408_15=40.0, DF408_16=41.4375, DF408_17=40.875, DF408_18=43.4375, DF408_19=41.875, DF408_20=45.6875, DF408_21=44.3125, DF408_22=47.9375, DF408_23=46.75)>",
            "<RTCM(1127, DF002=1127, DF003=0, DF427=318931000, DF393=1, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=4560809532588160, NSat=11, DF395=1090781184, NSig=3, DF396=8436542902, NCell=23, PRN_01=012, PRN_02=019, PRN_03=020, PRN_04=022, PRN_05=029, PRN_06=035, PRN_07=036, PRN_08=037, PRN_09=044, PRN_10=046, PRN_11=057, DF397_01=88, DF397_02=75, DF397_03=81, DF397_04=80, DF397_05=82, DF397_06=76, DF397_07=85, DF397_08=86, DF397_09=82, DF397_10=81, DF397_11=74, ExtSatInfo_01=0, ExtSatInfo_02=0, ExtSatInfo_03=0, ExtSatInfo_04=0, ExtSatInfo_05=0, ExtSatInfo_06=0, ExtSatInfo_07=0, ExtSatInfo_08=0, ExtSatInfo_09=0, ExtSatInfo_10=0, ExtSatInfo_11=0, DF398_01=0.6318359375, DF398_02=0.0400390625, DF398_03=0.3740234375, DF398_04=0.5341796875, DF398_05=0.3525390625, DF398_06=0.498046875, DF398_07=0.5224609375, DF398_08=0.9853515625, DF398_09=0.900390625, DF398_10=0.5107421875, DF398_11=0.435546875, DF399_01=-495, DF399_02=54, DF399_03=360, DF399_04=-309, DF399_05=607, DF399_06=167, DF399_07=-556, DF399_08=134, DF399_09=-305, DF399_10=-331, DF399_11=-63, CELLPRN_01=012, CELLSIG_01=2I, CELLPRN_02=012, CELLSIG_02=6I, CELLPRN_03=012, CELLSIG_03=7I, CELLPRN_04=019, CELLSIG_04=2I, CELLPRN_05=019, CELLSIG_05=6I, CELLPRN_06=020, CELLSIG_06=2I, CELLPRN_07=020, CELLSIG_07=6I, CELLPRN_08=022, CELLSIG_08=2I, CELLPRN_09=022, CELLSIG_09=6I, CELLPRN_10=029, CELLSIG_10=2I, CELLPRN_11=029, CELLSIG_11=6I, CELLPRN_12=035, CELLSIG_12=2I, CELLPRN_13=035, CELLSIG_13=6I, CELLPRN_14=036, CELLSIG_14=2I, CELLPRN_15=036, CELLSIG_15=6I, CELLPRN_16=037, CELLSIG_16=2I, CELLPRN_17=037, CELLSIG_17=6I, CELLPRN_18=044, CELLSIG_18=2I, CELLPRN_19=044, CELLSIG_19=6I, CELLPRN_20=046, CELLSIG_20=2I, CELLPRN_21=046, CELLSIG_21=6I, CELLPRN_22=057, CELLSIG_22=2I, CELLPRN_23=057, CELLSIG_23=6I, DF405_01=0.0003283768892288208, DF405_02=0.00036265142261981964, DF405_03=0.00037401728332042694, DF405_04=-0.0003399904817342758, DF405_05=-0.0003227144479751587, DF405_06=0.00022724084556102753, DF405_07=0.0002586767077445984, DF405_08=6.569735705852509e-05, DF405_09=8.562393486499786e-05, DF405_10=-0.0004872772842645645, DF405_11=-0.00044254958629608154, DF405_12=-0.0004627816379070282, DF405_13=-0.00043079257011413574, DF405_14=3.39653342962265e-05, DF405_15=0.00011373497545719147, DF405_16=-0.0004077702760696411, DF405_17=-0.00033494457602500916, DF405_18=0.00015609152615070343, DF405_19=0.00019328296184539795, DF405_20=-0.00048339366912841797, DF405_21=-0.00045737065374851227, DF405_22=-0.0003933291882276535, DF405_23=5.025416612625122e-05, DF406_01=0.0003184732049703598, DF406_02=0.0003424934111535549, DF406_03=0.0003520771861076355, DF406_04=-0.0003571687266230583, DF406_05=-0.00034973956644535065, DF406_06=0.00010209158062934875, DF406_07=6.294623017311096e-05, DF406_08=8.562719449400902e-05, DF406_09=0.00011176569387316704, DF406_10=-0.0006015864200890064, DF406_11=-0.0006155814044177532, DF406_12=-0.00048385653644800186, DF406_13=-0.00046588433906435966, DF406_14=6.545241922140121e-05, DF406_15=0.00016134930774569511, DF406_16=-0.00041268114000558853, DF406_17=-0.0003460478037595749, DF406_18=0.0001501264050602913, DF406_19=0.0001782020553946495, DF406_20=-0.0004371367394924164, DF406_21=-0.00039422931149601936, DF406_22=-0.00039861444383859634, DF406_23=4.1804276406764984e-05, DF407_01=517, DF407_02=519, DF407_03=519, DF407_04=635, DF407_05=635, DF407_06=650, DF407_07=650, DF407_08=590, DF407_09=590, DF407_10=650, DF407_11=650, DF407_12=621, DF407_13=621, DF407_14=552, DF407_15=555, DF407_16=594, DF407_17=594, DF407_18=581, DF407_19=580, DF407_20=587, DF407_21=587, DF407_22=609, DF407_23=609, DF420_01=0, DF420_02=0, DF420_03=0, DF420_04=0, DF420_05=0, DF420_06=0, DF420_07=0, DF420_08=0, DF420_09=0, DF420_10=0, DF420_11=0, DF420_12=0, DF420_13=0, DF420_14=0, DF420_15=0, DF420_16=0, DF420_17=0, DF420_18=0, DF420_19=0, DF420_20=0, DF420_21=0, DF420_22=0, DF420_23=0, DF408_01=34.8125, DF408_02=39.5, DF408_03=42.4375, DF408_04=53.25, DF408_05=51.9375, DF408_06=47.1875, DF408_07=45.6875, DF408_08=47.5625, DF408_09=46.4375, DF408_10=44.4375, DF408_11=42.5, DF408_12=50.25, DF408_13=48.0, DF408_14=42.5, DF408_15=40.0, DF408_16=41.4375, DF408_17=40.875, DF408_18=43.4375, DF408_19=41.875, DF408_20=45.6875, DF408_21=44.3125, DF408_22=47.9375, DF408_23=46.75, DF404_01=0.3755, DF404_02=0.3229, DF404_03=0.3437, DF404_04=0.226, DF404_05=0.2381, DF404_06=0.4325, DF404_07=0.4294, DF404_08=0.5, DF404_09=0.49570000000000003, DF404_10=0.0668, DF404_11=0.06470000000000001, DF404_12=-0.1562, DF404_13=-0.1524, DF404_14=0.4788, DF404_15=0.4703, DF404_16=0.3794, DF404_17=0.3793, DF404_18=-0.4364, DF404_19=-0.4419, DF404_20=0.1844, DF404_21=0.20120000000000002, DF404_22=-0.394, DF404_23=-0.3874)>",
            "<RTCM(1136, DF002=1136, DF003=0, DF546=318945000, DF393=1, DF409=0, DF001_7=0, DF411=1, DF412=0, DF417=0, DF418=0, DF394=0, NSat=0, DF395=0, NSig=0, DF396=0, NCell=0)>",
            "<RTCM(1137, DF002=1137, DF003=0, DF546=318945000, DF393=0, DF409=0, DF001_7=0, DF411=0, DF412=0, DF417=0, DF418=0, DF394=0, NSat=0, DF395=0, NSig=0, DF396=0, NCell=0)>",
            "<RTCM(1230, DF002=1230, DF003=0, DF421=1, DF001_3=0, DF422_1=1, DF422_2=1, DF422_3=1, DF422_4=1, DF423=0.0, DF424=0.0, DF425=0.0, DF426=0.0)>",
            "<RTCM(1001, DF002=1001, DF003=0, DF004=318946000, DF005=1, DF006=11, DF007=0, DF008=0, DF009_01=2, DF010_01=0, DF011_01=282760.82, DF012_01=-30.853, DF013_01=127, DF009_02=3, DF010_02=0, DF011_02=6983.64, DF012_02=-13.2045, DF013_02=127, DF009_03=21, DF010_03=0, DF011_03=18040.420000000002, DF012_03=-50.4415, DF013_03=127, DF009_04=4, DF010_04=0, DF011_04=144899.76, DF012_04=1.477, DF013_04=127, DF009_05=9, DF010_05=0, DF011_05=95045.46, DF012_05=7.735, DF013_05=127, DF009_06=6, DF010_06=0, DF011_06=12473.12, DF012_06=2.7495000000000003, DF013_06=127, DF009_07=19, DF010_07=0, DF011_07=21923.18, DF012_07=-8.8495, DF013_07=127, DF009_08=31, DF010_08=0, DF011_08=29036.4, DF012_08=-9.074, DF013_08=127, DF009_09=17, DF010_09=0, DF011_09=51737.32, DF012_09=-16.2865, DF013_09=127, DF009_10=7, DF010_10=0, DF011_10=154318.04, DF012_10=0.005, DF013_10=50, DF009_11=1, DF010_11=0, DF011_11=174292.16, DF012_11=-10.537, DF013_11=127)>",
            "<RTCM(1002, DF002=1002, DF003=0, DF004=318946000, DF005=1, DF006=11, DF007=0, DF008=0, DF009_01=2, DF010_01=0, DF011_01=282760.82, DF012_01=-30.853, DF013_01=127, DF014_01=75, DF015_01=43.0, DF009_02=3, DF010_02=0, DF011_02=6983.64, DF012_02=-13.2045, DF013_02=127, DF014_02=70, DF015_02=49.5, DF009_03=21, DF010_03=0, DF011_03=18040.420000000002, DF012_03=-50.4415, DF013_03=127, DF014_03=79, DF015_03=38.25, DF009_04=4, DF010_04=0, DF011_04=144899.76, DF012_04=1.477, DF013_04=127, DF014_04=67, DF015_04=49.0, DF009_05=9, DF010_05=0, DF011_05=95045.46, DF012_05=7.735, DF013_05=127, DF014_05=71, DF015_05=48.0, DF009_06=6, DF010_06=0, DF011_06=12473.12, DF012_06=2.7495000000000003, DF013_06=127, DF014_06=78, DF015_06=39.5, DF009_07=19, DF010_07=0, DF011_07=21923.18, DF012_07=-8.8495, DF013_07=127, DF014_07=78, DF015_07=40.75, DF009_08=31, DF010_08=0, DF011_08=29036.4, DF012_08=-9.074, DF013_08=127, DF014_08=79, DF015_08=42.25, DF009_09=17, DF010_09=0, DF011_09=51737.32, DF012_09=-16.2865, DF013_09=127, DF014_09=76, DF015_09=40.5, DF009_10=7, DF010_10=0, DF011_10=154318.04, DF012_10=0.005, DF013_10=50, DF014_10=83, DF015_10=34.0, DF009_11=1, DF010_11=0, DF011_11=174292.16, DF012_11=-10.537, DF013_11=127, DF014_11=68, DF015_11=49.25)>",
        ]
        dirname = os.path.dirname(__file__)
        with open(
            os.path.join(dirname, "pygpsdata-NTRIP-USCL00CHL0.log"), "rb"
        ) as stream:
            i = 0
            raw = 0
            rtr = RTCMReader(stream, labelmsm=True)
            for raw, parsed in rtr:
                if raw is not None:
                    # print(f'"{parsed}",')
                    self.assertEqual(f"{parsed}", EXPECTED_RESULTS[i])
                    i += 1
        self.assertEqual(i, 35)

    def testigsssr4076(
        self,
    ):  # test 4076 messages using log from NTRIP caster products.igs-ip.net, mountpoint SSRA00CNE1
        EXPECTED_RESULTS = [
            "<RTCM(4076_023, DF002=4076, IDF001=3, IDF002=23, IDF003=30270, IDF004=2, IDF005=1, IDF007=0, IDF008=0, IDF009=0, IDF006=0, IDF010=28, IDF011_01=2, IDF012_01=68, IDF013_01=-424.20000000000005, IDF014_01=926.4000000000001, IDF015_01=-61.6, IDF016_01=-0.125, IDF017_01=-0.396, IDF018_01=-0.196, IDF019_01=-177.5, IDF020_01=0.0, IDF021_01=0.0, IDF011_02=3, IDF012_02=76, IDF013_02=-88.4, IDF014_02=933.6, IDF015_02=58.400000000000006, IDF016_02=-0.027, IDF017_02=-0.256, IDF018_02=0.036000000000000004, IDF019_02=87.80000000000001, IDF020_02=0.0, IDF021_02=0.0, IDF011_03=4, IDF012_03=183, IDF013_03=-646.6, IDF014_03=-1270.0, IDF015_03=1463.6000000000001, IDF016_03=0.152, IDF017_03=-0.47600000000000003, IDF018_03=0.07200000000000001, IDF019_03=-162.60000000000002, IDF020_03=0.0, IDF021_03=0.0, IDF011_04=5, IDF012_04=35, IDF013_04=-39.400000000000006, IDF014_04=-1200.8, IDF015_04=149.6, IDF016_04=0.047, IDF017_04=-0.528, IDF018_04=0.084, IDF019_04=-167.10000000000002, IDF020_04=0.0, IDF021_04=0.0, IDF011_05=6, IDF012_05=97, IDF013_05=-502.1, IDF014_05=983.2, IDF015_05=-749.2, IDF016_05=-0.079, IDF017_05=0.28, IDF018_05=-0.264, IDF019_05=15.0, IDF020_05=0.0, IDF021_05=0.0, IDF011_06=7, IDF012_06=34, IDF013_06=27.400000000000002, IDF014_06=33.6, IDF015_06=-438.40000000000003, IDF016_06=0.037, IDF017_06=0.052000000000000005, IDF018_06=0.07200000000000001, IDF019_06=-233.0, IDF020_06=0.0, IDF021_06=0.0, IDF011_07=8, IDF012_07=139, IDF013_07=13.200000000000001, IDF014_07=59.6, IDF015_07=-265.2, IDF016_07=0.048, IDF017_07=-0.336, IDF018_07=0.004, IDF019_07=306.3, IDF020_07=0.0, IDF021_07=0.0, IDF011_08=9, IDF012_08=111, IDF013_08=89.2, IDF014_08=-168.0, IDF015_08=467.20000000000005, IDF016_08=-0.021, IDF017_08=-0.368, IDF018_08=0.128, IDF019_08=-44.7, IDF020_08=0.0, IDF021_08=0.0, IDF011_09=10, IDF012_09=22, IDF013_09=-9.200000000000001, IDF014_09=232.4, IDF015_09=-310.8, IDF016_09=0.032, IDF017_09=-0.148, IDF018_09=-0.10400000000000001, IDF019_09=30.0, IDF020_09=0.0, IDF021_09=0.0, IDF011_10=11, IDF012_10=104, IDF013_10=-901.4000000000001, IDF014_10=-361.6, IDF015_10=395.20000000000005, IDF016_10=-0.032, IDF017_10=-0.084, IDF018_10=-0.132, IDF019_10=506.8, IDF020_10=0.0, IDF021_10=0.0, IDF011_11=12, IDF012_11=88, IDF013_11=211.8, IDF014_11=-559.2, IDF015_11=-1.2000000000000002, IDF016_11=0.089, IDF017_11=0.096, IDF018_11=-0.036000000000000004, IDF019_11=71.5, IDF020_11=0.0, IDF021_11=0.0, IDF011_12=13, IDF012_12=44, IDF013_12=8.0, IDF014_12=-1099.2, IDF015_12=-533.6, IDF016_12=0.10300000000000001, IDF017_12=-0.164, IDF018_12=-0.008, IDF019_12=66.8, IDF020_12=0.0, IDF021_12=0.0, IDF011_13=14, IDF012_13=20, IDF013_13=-938.5, IDF014_13=690.4000000000001, IDF015_13=-628.8000000000001, IDF016_13=-0.088, IDF017_13=0.20800000000000002, IDF018_13=0.092, IDF019_13=143.0, IDF020_13=0.0, IDF021_13=0.0, IDF011_14=15, IDF012_14=6, IDF013_14=-70.4, IDF014_14=-288.40000000000003, IDF015_14=191.60000000000002, IDF016_14=0.082, IDF017_14=-0.276, IDF018_14=-0.24, IDF019_14=-198.9, IDF020_14=0.0, IDF021_14=0.0, IDF011_15=16, IDF012_15=38, IDF013_15=136.1, IDF014_15=-516.4, IDF015_15=97.2, IDF016_15=0.13, IDF017_15=-0.212, IDF018_15=0.168, IDF019_15=-346.8, IDF020_15=0.0, IDF021_15=0.0, IDF011_16=17, IDF012_16=16, IDF013_16=-106.5, IDF014_16=470.8, IDF015_16=-284.0, IDF016_16=0.007, IDF017_16=-0.256, IDF018_16=-0.116, IDF019_16=252.5, IDF020_16=0.0, IDF021_16=0.0, IDF011_17=18, IDF012_17=74, IDF013_17=-892.2, IDF014_17=779.6, IDF015_17=-375.6, IDF016_17=-0.085, IDF017_17=-0.044, IDF018_17=0.04, IDF019_17=43.900000000000006, IDF020_17=0.0, IDF021_17=0.0, IDF011_18=19, IDF012_18=49, IDF013_18=-130.9, IDF014_18=484.40000000000003, IDF015_18=-432.8, IDF016_18=-0.047, IDF017_18=0.356, IDF018_18=-0.04, IDF019_18=-8.3, IDF020_18=0.0, IDF021_18=0.0, IDF011_19=20, IDF012_19=116, IDF013_19=-32.2, IDF014_19=790.4000000000001, IDF015_19=-644.4000000000001, IDF016_19=-0.159, IDF017_19=-0.26, IDF018_19=-0.02, IDF019_19=-1052.8, IDF020_19=0.0, IDF021_19=0.0, IDF011_20=21, IDF012_20=4, IDF013_20=-247.5, IDF014_20=674.0, IDF015_20=276.8, IDF016_20=-0.10300000000000001, IDF017_20=-0.5, IDF018_20=-0.28800000000000003, IDF019_20=-310.3, IDF020_20=0.0, IDF021_20=0.0, IDF011_21=22, IDF012_21=110, IDF013_21=60.7, IDF014_21=-856.0, IDF015_21=-921.6, IDF016_21=0.151, IDF017_21=-0.06, IDF018_21=-0.18, IDF019_21=8.3, IDF020_21=0.0, IDF021_21=0.0, IDF011_22=23, IDF012_22=106, IDF013_22=-804.7, IDF014_22=-156.4, IDF015_22=-35.6, IDF016_22=0.057, IDF017_22=-0.232, IDF018_22=-0.08, IDF019_22=-75.4, IDF020_22=0.0, IDF021_22=0.0, IDF011_23=24, IDF012_23=33, IDF013_23=-57.900000000000006, IDF014_23=-120.4, IDF015_23=204.0, IDF016_23=-0.017, IDF017_23=-0.196, IDF018_23=-0.02, IDF019_23=-30.1, IDF020_23=0.0, IDF021_23=0.0, IDF011_24=25, IDF012_24=7, IDF013_24=-119.0, IDF014_24=548.8000000000001, IDF015_24=60.800000000000004, IDF016_24=-0.084, IDF017_24=0.06, IDF018_24=-0.14, IDF019_24=-244.70000000000002, IDF020_24=0.0, IDF021_24=0.0, IDF011_25=26, IDF012_25=5, IDF013_25=275.0, IDF014_25=1079.2, IDF015_25=66.8, IDF016_25=-0.096, IDF017_25=0.128, IDF018_25=0.124, IDF019_25=190.4, IDF020_25=0.0, IDF021_25=0.0, IDF011_26=27, IDF012_26=70, IDF013_26=431.6, IDF014_26=-4144.8, IDF015_26=-564.4, IDF016_26=0.41500000000000004, IDF017_26=-0.432, IDF018_26=-0.08, IDF019_26=505.1, IDF020_26=0.0, IDF021_26=0.0, IDF011_27=29, IDF012_27=11, IDF013_27=31.0, IDF014_27=564.8000000000001, IDF015_27=-5.6000000000000005, IDF016_27=-0.14, IDF017_27=-0.196, IDF018_27=-0.044, IDF019_27=13.9, IDF020_27=0.0, IDF021_27=0.0, IDF011_28=30, IDF012_28=4, IDF013_28=-321.3, IDF014_28=2084.8, IDF015_28=-101.60000000000001, IDF016_28=-0.178, IDF017_28=0.20400000000000001, IDF018_28=0.252, IDF019_28=102.30000000000001, IDF020_28=0.0, IDF021_28=0.0)>",
            "<RTCM(4076_043, DF002=4076, IDF001=3, IDF002=43, IDF003=30285, IDF004=2, IDF005=0, IDF007=0, IDF008=0, IDF009=0, IDF006=0, IDF010=15, IDF011_01=1, IDF012_01=45, IDF013_01=183.0, IDF014_01=-1555.2, IDF015_01=397.6, IDF016_01=0.736, IDF017_01=-0.468, IDF018_01=0.864, IDF019_01=-270.8, IDF020_01=0.0, IDF021_01=0.0, IDF011_02=2, IDF012_02=45, IDF013_02=565.2, IDF014_02=82.0, IDF015_02=-280.8, IDF016_02=0.28700000000000003, IDF017_02=0.784, IDF018_02=0.464, IDF019_02=-5643.3, IDF020_02=0.0, IDF021_02=0.0, IDF011_03=3, IDF012_03=45, IDF013_03=701.7, IDF014_03=86.4, IDF015_03=-102.4, IDF016_03=0.93, IDF017_03=0.468, IDF018_03=0.212, IDF019_03=-6226.8, IDF020_03=0.0, IDF021_03=0.0, IDF011_04=4, IDF012_04=45, IDF013_04=950.7, IDF014_04=3424.4, IDF015_04=-2404.4, IDF016_04=0.551, IDF017_04=-0.064, IDF018_04=0.132, IDF019_04=-2579.8, IDF020_04=0.0, IDF021_04=0.0, IDF011_05=7, IDF012_05=43, IDF013_05=2930.8, IDF014_05=-2285.6, IDF015_05=-578.4, IDF016_05=2.101, IDF017_05=-1.008, IDF018_05=-1.168, IDF019_05=-1653.6000000000001, IDF020_05=0.0, IDF021_05=0.0, IDF011_06=8, IDF012_06=45, IDF013_06=140.8, IDF014_06=568.0, IDF015_06=-284.8, IDF016_06=0.336, IDF017_06=-0.076, IDF018_06=-0.164, IDF019_06=-4866.2, IDF020_06=0.0, IDF021_06=0.0, IDF011_07=9, IDF012_07=45, IDF013_07=585.6, IDF014_07=2595.6000000000004, IDF015_07=1876.4, IDF016_07=-0.458, IDF017_07=-0.876, IDF018_07=0.24, IDF019_07=3331.7000000000003, IDF020_07=0.0, IDF021_07=0.0, IDF011_08=14, IDF012_08=45, IDF013_08=-425.1, IDF014_08=3587.2000000000003, IDF015_08=-746.0, IDF016_08=-0.626, IDF017_08=-0.896, IDF018_08=-0.864, IDF019_08=8727.0, IDF020_08=0.0, IDF021_08=0.0, IDF011_09=15, IDF012_09=45, IDF013_09=105.80000000000001, IDF014_09=4213.6, IDF015_09=2320.4, IDF016_09=-1.049, IDF017_09=0.404, IDF018_09=-0.392, IDF019_09=-986.7, IDF020_09=0.0, IDF021_09=0.0, IDF011_10=16, IDF012_10=45, IDF013_10=-36.1, IDF014_10=-48.400000000000006, IDF015_10=460.0, IDF016_10=-1.219, IDF017_10=0.336, IDF018_10=-0.936, IDF019_10=1471.8000000000002, IDF020_10=0.0, IDF021_10=0.0, IDF011_11=17, IDF012_11=45, IDF013_11=-236.8, IDF014_11=1737.2, IDF015_11=1006.8000000000001, IDF016_11=-1.201, IDF017_11=-0.116, IDF018_11=-0.028, IDF019_11=0.0, IDF020_11=0.0, IDF021_11=0.0, IDF011_12=18, IDF012_12=45, IDF013_12=46.7, IDF014_12=5.2, IDF015_12=234.4, IDF016_12=-0.299, IDF017_12=0.54, IDF018_12=0.008, IDF019_12=1741.8000000000002, IDF020_12=0.0, IDF021_12=0.0, IDF011_13=19, IDF012_13=45, IDF013_13=605.0, IDF014_13=-6162.8, IDF015_13=571.2, IDF016_13=-0.28300000000000003, IDF017_13=0.47600000000000003, IDF018_13=-0.312, IDF019_13=167.20000000000002, IDF020_13=0.0, IDF021_13=0.0, IDF011_14=20, IDF012_14=45, IDF013_14=-949.2, IDF014_14=-1745.2, IDF015_14=-75.2, IDF016_14=-1.418, IDF017_14=-0.392, IDF018_14=-0.776, IDF019_14=7723.0, IDF020_14=0.0, IDF021_14=0.0, IDF011_15=24, IDF012_15=45, IDF013_15=70.9, IDF014_15=640.8000000000001, IDF015_15=1585.6000000000001, IDF016_15=-0.66, IDF017_15=0.552, IDF018_15=0.6960000000000001, IDF019_15=5114.1, IDF020_15=0.0, IDF021_15=0.0)>",
            "<RTCM(4076_063, DF002=4076, IDF001=3, IDF002=63, IDF003=30285, IDF004=2, IDF005=0, IDF007=0, IDF008=0, IDF009=0, IDF006=0, IDF010=22, IDF011_01=2, IDF012_01=49, IDF013_01=29.8, IDF014_01=281.6, IDF015_01=-26.400000000000002, IDF016_01=-0.017, IDF017_01=-0.008, IDF018_01=-0.028, IDF019_01=169.5, IDF020_01=0.0, IDF021_01=0.0, IDF011_02=3, IDF012_02=49, IDF013_02=36.300000000000004, IDF014_02=-182.0, IDF015_02=-112.80000000000001, IDF016_02=-0.009000000000000001, IDF017_02=-0.152, IDF018_02=0.028, IDF019_02=-11.700000000000001, IDF020_02=0.0, IDF021_02=0.0, IDF011_03=4, IDF012_03=49, IDF013_03=-50.1, IDF014_03=-188.8, IDF015_03=-44.0, IDF016_03=-0.021, IDF017_03=-0.136, IDF018_03=0.012, IDF019_03=41.7, IDF020_03=0.0, IDF021_03=0.0, IDF011_04=5, IDF012_04=49, IDF013_04=-60.400000000000006, IDF014_04=-307.20000000000005, IDF015_04=29.200000000000003, IDF016_04=0.01, IDF017_04=-0.168, IDF018_04=-0.128, IDF019_04=40.300000000000004, IDF020_04=0.0, IDF021_04=0.0, IDF011_05=7, IDF012_05=49, IDF013_05=7.0, IDF014_05=-310.0, IDF015_05=-198.4, IDF016_05=0.021, IDF017_05=-0.192, IDF018_05=0.04, IDF019_05=-90.4, IDF020_05=0.0, IDF021_05=0.0, IDF011_06=8, IDF012_06=48, IDF013_06=-159.5, IDF014_06=162.8, IDF015_06=71.2, IDF016_06=0.009000000000000001, IDF017_06=-0.076, IDF018_06=0.048, IDF019_06=436.6, IDF020_06=0.0, IDF021_06=0.0, IDF011_07=9, IDF012_07=45, IDF013_07=-2.5, IDF014_07=-411.6, IDF015_07=-48.0, IDF016_07=0.011, IDF017_07=-0.108, IDF018_07=-0.092, IDF019_07=161.10000000000002, IDF020_07=0.0, IDF021_07=0.0, IDF011_08=10, IDF012_08=44, IDF013_08=104.4, IDF014_08=196.8, IDF015_08=447.20000000000005, IDF016_08=-0.066, IDF017_08=0.032, IDF018_08=0.068, IDF019_08=-417.90000000000003, IDF020_08=0.0, IDF021_08=0.0, IDF011_09=11, IDF012_09=49, IDF013_09=507.20000000000005, IDF014_09=146.8, IDF015_09=421.6, IDF016_09=-0.03, IDF017_09=-0.028, IDF018_09=-0.024, IDF019_09=27.6, IDF020_09=0.0, IDF021_09=0.0, IDF011_10=12, IDF012_10=46, IDF013_10=78.2, IDF014_10=118.4, IDF015_10=78.0, IDF016_10=-0.066, IDF017_10=0.04, IDF018_10=0.108, IDF019_10=-182.5, IDF020_10=0.0, IDF021_10=0.0, IDF011_11=13, IDF012_11=49, IDF013_11=-139.20000000000002, IDF014_11=-76.4, IDF015_11=100.80000000000001, IDF016_11=-0.042, IDF017_11=-0.088, IDF018_11=-0.004, IDF019_11=90.80000000000001, IDF020_11=0.0, IDF021_11=0.0, IDF011_12=15, IDF012_12=45, IDF013_12=-126.0, IDF014_12=355.6, IDF015_12=-505.6, IDF016_12=0.007, IDF017_12=-0.04, IDF018_12=-0.08, IDF019_12=-99.30000000000001, IDF020_12=0.0, IDF021_12=0.0, IDF011_13=21, IDF012_13=49, IDF013_13=41.5, IDF014_13=-96.4, IDF015_13=342.0, IDF016_13=0.042, IDF017_13=-0.06, IDF018_13=0.016, IDF019_13=-392.90000000000003, IDF020_13=0.0, IDF021_13=0.0, IDF011_14=24, IDF012_14=49, IDF013_14=175.5, IDF014_14=167.20000000000002, IDF015_14=53.2, IDF016_14=-0.008, IDF017_14=-0.008, IDF018_14=-0.036000000000000004, IDF019_14=10.100000000000001, IDF020_14=0.0, IDF021_14=0.0, IDF011_15=25, IDF012_15=49, IDF013_15=-55.1, IDF014_15=247.60000000000002, IDF015_15=36.0, IDF016_15=-0.036000000000000004, IDF017_15=-0.016, IDF018_15=0.04, IDF019_15=-208.8, IDF020_15=0.0, IDF021_15=0.0, IDF011_16=26, IDF012_16=49, IDF013_16=-0.1, IDF014_16=-83.60000000000001, IDF015_16=-85.60000000000001, IDF016_16=0.036000000000000004, IDF017_16=-0.028, IDF018_16=0.0, IDF019_16=299.0, IDF020_16=0.0, IDF021_16=0.0, IDF011_17=27, IDF012_17=49, IDF013_17=-6.9, IDF014_17=-189.20000000000002, IDF015_17=309.6, IDF016_17=-0.024, IDF017_17=-0.052000000000000005, IDF018_17=-0.004, IDF019_17=-10.100000000000001, IDF020_17=0.0, IDF021_17=0.0, IDF011_18=30, IDF012_18=46, IDF013_18=84.9, IDF014_18=-139.20000000000002, IDF015_18=162.8, IDF016_18=-0.018000000000000002, IDF017_18=-0.056, IDF018_18=0.008, IDF019_18=24.5, IDF020_18=0.0, IDF021_18=0.0, IDF011_19=31, IDF012_19=49, IDF013_19=223.0, IDF014_19=335.6, IDF015_19=48.0, IDF016_19=-0.041, IDF017_19=-0.084, IDF018_19=0.024, IDF019_19=-595.1, IDF020_19=0.0, IDF021_19=0.0, IDF011_20=33, IDF012_20=45, IDF013_20=-80.2, IDF014_20=247.60000000000002, IDF015_20=33.6, IDF016_20=-0.043000000000000003, IDF017_20=0.14400000000000002, IDF018_20=0.004, IDF019_20=133.20000000000002, IDF020_20=0.0, IDF021_20=0.0, IDF011_21=34, IDF012_21=46, IDF013_21=-25.900000000000002, IDF014_21=335.20000000000005, IDF015_21=-239.20000000000002, IDF016_21=0.039, IDF017_21=-0.2, IDF018_21=0.06, IDF019_21=-296.40000000000003, IDF020_21=0.0, IDF021_21=0.0, IDF011_22=36, IDF012_22=45, IDF013_22=51.300000000000004, IDF014_22=579.2, IDF015_22=325.6, IDF016_22=-0.077, IDF017_22=0.10400000000000001, IDF018_22=0.06, IDF019_22=-414.70000000000005, IDF020_22=0.0, IDF021_22=0.0)>",
            "<RTCM(4076_103, DF002=4076, IDF001=3, IDF002=103, IDF003=30285, IDF004=2, IDF005=1, IDF007=0, IDF008=0, IDF009=0, IDF006=0, IDF010=28, IDF011_01=6, IDF012_01=160, IDF013_01=2358.4, IDF014_01=1288.0, IDF015_01=-414.40000000000003, IDF016_01=0.012, IDF017_01=-0.028, IDF018_01=-0.064, IDF019_01=-13392.5, IDF020_01=0.0, IDF021_01=0.0, IDF011_02=7, IDF012_02=160, IDF013_02=2346.1, IDF014_02=1272.8000000000002, IDF015_02=1958.0, IDF016_02=-0.08600000000000001, IDF017_02=0.028, IDF018_02=-0.068, IDF019_02=-18887.600000000002, IDF020_02=0.0, IDF021_02=0.0, IDF011_03=8, IDF012_03=160, IDF013_03=2305.3, IDF014_03=-1462.0, IDF015_03=-1202.8, IDF016_03=0.129, IDF017_03=0.028, IDF018_03=-0.02, IDF019_03=-15925.5, IDF020_03=0.0, IDF021_03=0.0, IDF011_04=10, IDF012_04=160, IDF013_04=2457.6000000000004, IDF014_04=998.8000000000001, IDF015_04=1092.0, IDF016_04=-0.011, IDF017_04=0.016, IDF018_04=-0.04, IDF019_04=-10891.2, IDF020_04=0.0, IDF021_04=0.0, IDF011_05=11, IDF012_05=160, IDF013_05=1025.2, IDF014_05=749.2, IDF015_05=58.400000000000006, IDF016_05=-0.021, IDF017_05=-0.028, IDF018_05=-0.068, IDF019_05=-9539.1, IDF020_05=0.0, IDF021_05=0.0, IDF011_06=12, IDF012_06=160, IDF013_06=2176.2000000000003, IDF014_06=1590.0, IDF015_06=522.0, IDF016_06=-0.134, IDF017_06=-0.07200000000000001, IDF018_06=0.256, IDF019_06=-8621.800000000001, IDF020_06=0.0, IDF021_06=0.0, IDF011_07=13, IDF012_07=160, IDF013_07=2088.7000000000003, IDF014_07=-1024.4, IDF015_07=-125.2, IDF016_07=0.032, IDF017_07=-0.004, IDF018_07=-0.016, IDF019_07=4315.400000000001, IDF020_07=0.0, IDF021_07=0.0, IDF011_08=16, IDF012_08=160, IDF013_08=1696.6000000000001, IDF014_08=1474.0, IDF015_08=-381.20000000000005, IDF016_08=0.08, IDF017_08=-0.188, IDF018_08=-0.028, IDF019_08=-2471.3, IDF020_08=0.0, IDF021_08=0.0, IDF011_09=19, IDF012_09=160, IDF013_09=232.0, IDF014_09=-166.0, IDF015_09=-128.0, IDF016_09=-0.019, IDF017_09=-0.176, IDF018_09=0.02, IDF019_09=-7516.1, IDF020_09=0.0, IDF021_09=0.0, IDF011_10=20, IDF012_10=160, IDF013_10=309.1, IDF014_10=-117.2, IDF015_10=-109.60000000000001, IDF016_10=0.002, IDF017_10=-0.064, IDF018_10=0.056, IDF019_10=-9945.0, IDF020_10=0.0, IDF021_10=0.0, IDF011_11=21, IDF012_11=160, IDF013_11=193.20000000000002, IDF014_11=-165.20000000000002, IDF015_11=-75.2, IDF016_11=-0.022, IDF017_11=-0.196, IDF018_11=-0.092, IDF019_11=-9632.300000000001, IDF020_11=0.0, IDF021_11=0.0, IDF011_12=22, IDF012_12=160, IDF013_12=180.10000000000002, IDF014_12=-182.0, IDF015_12=-67.2, IDF016_12=-0.04, IDF017_12=-0.188, IDF018_12=-0.06, IDF019_12=-11826.0, IDF020_12=0.0, IDF021_12=0.0, IDF011_13=23, IDF012_13=160, IDF013_13=231.20000000000002, IDF014_13=-11.200000000000001, IDF015_13=-149.20000000000002, IDF016_13=0.075, IDF017_13=-0.188, IDF018_13=-0.028, IDF019_13=-19084.9, IDF020_13=0.0, IDF021_13=0.0, IDF011_14=24, IDF012_14=160, IDF013_14=259.3, IDF014_14=348.8, IDF015_14=-154.8, IDF016_14=-0.082, IDF017_14=-0.2, IDF018_14=0.044, IDF019_14=-2691.9, IDF020_14=0.0, IDF021_14=0.0, IDF011_15=25, IDF012_15=160, IDF013_15=-7.0, IDF014_15=-40.400000000000006, IDF015_15=-269.6, IDF016_15=-0.02, IDF017_15=-0.112, IDF018_15=0.016, IDF019_15=1006.1, IDF020_15=0.0, IDF021_15=0.0, IDF011_16=26, IDF012_16=160, IDF013_16=162.20000000000002, IDF014_16=-78.0, IDF015_16=292.40000000000003, IDF016_16=0.002, IDF017_16=-0.096, IDF018_16=0.068, IDF019_16=5049.900000000001, IDF020_16=0.0, IDF021_16=0.0, IDF011_17=27, IDF012_17=160, IDF013_17=-108.80000000000001, IDF014_17=73.2, IDF015_17=-194.4, IDF016_17=0.088, IDF017_17=-0.132, IDF018_17=-0.036000000000000004, IDF019_17=4235.2, IDF020_17=0.0, IDF021_17=0.0, IDF011_18=28, IDF012_18=160, IDF013_18=9.600000000000001, IDF014_18=-131.6, IDF015_18=-295.6, IDF016_18=0.032, IDF017_18=-0.148, IDF018_18=-0.07200000000000001, IDF019_18=5310.5, IDF020_18=0.0, IDF021_18=0.0, IDF011_19=29, IDF012_19=160, IDF013_19=31.6, IDF014_19=-137.20000000000002, IDF015_19=428.8, IDF016_19=-0.047, IDF017_19=-0.168, IDF018_19=0.0, IDF019_19=1380.9, IDF020_19=0.0, IDF021_19=0.0, IDF011_20=30, IDF012_20=160, IDF013_20=-129.5, IDF014_20=2.8000000000000003, IDF015_20=125.2, IDF016_20=-0.023, IDF017_20=-0.1, IDF018_20=0.004, IDF019_20=10199.2, IDF020_20=0.0, IDF021_20=0.0, IDF011_21=32, IDF012_21=160, IDF013_21=259.40000000000003, IDF014_21=-364.0, IDF015_21=-64.8, IDF016_21=0.025, IDF017_21=-0.112, IDF018_21=0.128, IDF019_21=6444.200000000001, IDF020_21=0.0, IDF021_21=0.0, IDF011_22=33, IDF012_22=160, IDF013_22=220.3, IDF014_22=-333.20000000000005, IDF015_22=11.600000000000001, IDF016_22=0.018000000000000002, IDF017_22=-0.224, IDF018_22=0.044, IDF019_22=40092.200000000004, IDF020_22=0.0, IDF021_22=0.0, IDF011_23=34, IDF012_23=160, IDF013_23=19.8, IDF014_23=262.40000000000003, IDF015_23=53.6, IDF016_23=-0.053, IDF017_23=-0.14400000000000002, IDF018_23=-0.028, IDF019_23=7065.5, IDF020_23=0.0, IDF021_23=0.0, IDF011_24=36, IDF012_24=160, IDF013_24=228.3, IDF014_24=-256.8, IDF015_24=304.40000000000003, IDF016_24=0.043000000000000003, IDF017_24=-0.128, IDF018_24=-0.004, IDF019_24=20059.5, IDF020_24=0.0, IDF021_24=0.0, IDF011_25=37, IDF012_25=160, IDF013_25=208.10000000000002, IDF014_25=54.400000000000006, IDF015_25=-167.60000000000002, IDF016_25=0.013000000000000001, IDF017_25=-0.1, IDF018_25=-0.056, IDF019_25=15136.0, IDF020_25=0.0, IDF021_25=0.0, IDF011_26=38, IDF012_26=160, IDF013_26=-47.6, IDF014_26=-528.8000000000001, IDF015_26=39.2, IDF016_26=0.019, IDF017_26=-0.028, IDF018_26=-0.064, IDF019_26=-640.7, IDF020_26=0.0, IDF021_26=0.0, IDF011_27=39, IDF012_27=160, IDF013_27=479.5, IDF014_27=-445.6, IDF015_27=-539.2, IDF016_27=0.005, IDF017_27=-0.152, IDF018_27=-0.052000000000000005, IDF019_27=-4077.4, IDF020_27=0.0, IDF021_27=0.0, IDF011_28=40, IDF012_28=160, IDF013_28=535.1, IDF014_28=385.6, IDF015_28=-88.4, IDF016_28=-0.037, IDF017_28=-0.012, IDF018_28=-0.024, IDF019_28=640.7, IDF020_28=0.0, IDF021_28=0.0)>",
            "<RTCM(4076_201, DF002=4076, IDF001=3, IDF002=201, IDF003=30300, IDF004=6, IDF005=0, IDF007=0, IDF008=0, IDF009=0, IDF041=0.0, IDF035=0, IDF036_01=450, IDF037_01=11, IDF038_01=11, IDF039_01_01=35.36, IDF039_01_02=2.34, IDF039_01_03=-12.18, IDF039_01_04=-2.09, IDF039_01_05=2.91, IDF039_01_06=0.43, IDF039_01_07=-0.23500000000000001, IDF039_01_08=0.9550000000000001, IDF039_01_09=0.17, IDF039_01_10=-0.78, IDF039_01_11=-0.14, IDF039_01_12=0.28, IDF039_01_13=0.14, IDF039_01_14=18.145, IDF039_01_15=0.745, IDF039_01_16=-1.705, IDF039_01_17=-0.6, IDF039_01_18=1.01, IDF039_01_19=-0.645, IDF039_01_20=-0.245, IDF039_01_21=0.39, IDF039_01_22=-0.01, IDF039_01_23=-0.38, IDF039_01_24=0.04, IDF039_01_25=0.23500000000000001, IDF039_01_26=2.5100000000000002, IDF039_01_27=0.61, IDF039_01_28=0.01, IDF039_01_29=-0.24, IDF039_01_30=0.075, IDF039_01_31=0.145, IDF039_01_32=0.07, IDF039_01_33=0.06, IDF039_01_34=-0.035, IDF039_01_35=-0.07, IDF039_01_36=-0.055, IDF039_01_37=-1.25, IDF039_01_38=-0.16, IDF039_01_39=-0.41000000000000003, IDF039_01_40=0.215, IDF039_01_41=0.035, IDF039_01_42=-0.12, IDF039_01_43=-0.04, IDF039_01_44=0.135, IDF039_01_45=0.015, IDF039_01_46=-0.005, IDF039_01_47=-0.22, IDF039_01_48=-0.195, IDF039_01_49=0.445, IDF039_01_50=0.17500000000000002, IDF039_01_51=-0.42, IDF039_01_52=-0.185, IDF039_01_53=0.26, IDF039_01_54=0.06, IDF039_01_55=-0.22, IDF039_01_56=0.095, IDF039_01_57=0.03, IDF039_01_58=-0.1, IDF039_01_59=0.07, IDF039_01_60=0.02, IDF039_01_61=-0.015, IDF039_01_62=-0.005, IDF039_01_63=-0.06, IDF039_01_64=-0.065, IDF039_01_65=0.125, IDF039_01_66=-0.03, IDF039_01_67=-0.035, IDF039_01_68=0.03, IDF039_01_69=0.06, IDF039_01_70=-0.03, IDF039_01_71=0.36, IDF039_01_72=0.07, IDF039_01_73=-0.005, IDF039_01_74=0.045, IDF039_01_75=0.025, IDF039_01_76=-0.03, IDF039_01_77=-0.09, IDF039_01_78=-0.005, IDF039_01_79=0.06, IDF039_01_80=-0.035, IDF039_01_81=-0.05, IDF039_01_82=-0.165, IDF039_01_83=-0.01, IDF039_01_84=0.02, IDF039_01_85=0.04, IDF039_01_86=0.1, IDF039_01_87=0.025, IDF039_01_88=0.05, IDF039_01_89=0.075, IDF039_01_90=0.01, IDF039_01_91=0.005, IDF040_01_01=2.15, IDF040_01_02=0.375, IDF040_01_03=-1.945, IDF040_01_04=-0.12, IDF040_01_05=1.115, IDF040_01_06=0.1, IDF040_01_07=-0.8250000000000001, IDF040_01_08=0.34500000000000003, IDF040_01_09=0.19, IDF040_01_10=-0.375, IDF040_01_11=0.42, IDF040_01_12=0.26, IDF040_01_13=-3.5100000000000002, IDF040_01_14=-0.195, IDF040_01_15=0.99, IDF040_01_16=0.33, IDF040_01_17=-0.495, IDF040_01_18=-0.03, IDF040_01_19=-0.08, IDF040_01_20=-0.06, IDF040_01_21=0.0, IDF040_01_22=-0.215, IDF040_01_23=-0.055, IDF040_01_24=1.36, IDF040_01_25=0.12, IDF040_01_26=-0.615, IDF040_01_27=-0.06, IDF040_01_28=0.28, IDF040_01_29=0.145, IDF040_01_30=-0.095, IDF040_01_31=-0.105, IDF040_01_32=0.08, IDF040_01_33=0.11, IDF040_01_34=0.0, IDF040_01_35=-0.02, IDF040_01_36=-0.21, IDF040_01_37=-0.1, IDF040_01_38=0.035, IDF040_01_39=0.075, IDF040_01_40=-0.1, IDF040_01_41=-0.09, IDF040_01_42=0.055, IDF040_01_43=-0.775, IDF040_01_44=-0.08, IDF040_01_45=0.105, IDF040_01_46=-0.015, IDF040_01_47=-0.14, IDF040_01_48=0.005, IDF040_01_49=0.16, IDF040_01_50=-0.035, IDF040_01_51=-0.03, IDF040_01_52=-0.005, IDF040_01_53=0.05, IDF040_01_54=-0.02, IDF040_01_55=0.085, IDF040_01_56=0.02, IDF040_01_57=-0.025, IDF040_01_58=0.315, IDF040_01_59=0.035, IDF040_01_60=0.01, IDF040_01_61=-0.015, IDF040_01_62=0.015, IDF040_01_63=-0.015, IDF040_01_64=0.06, IDF040_01_65=0.015, IDF040_01_66=-0.075, IDF040_01_67=-0.03, IDF040_01_68=0.05, IDF040_01_69=0.05, IDF040_01_70=-0.025, IDF040_01_71=0.09, IDF040_01_72=0.005, IDF040_01_73=-0.015, IDF040_01_74=-0.03, IDF040_01_75=0.02, IDF040_01_76=-0.06, IDF040_01_77=-0.015, IDF040_01_78=0.04)>",
            "<RTCM(4076_065, DF002=4076, IDF001=3, IDF002=65, IDF003=311120, IDF004=3, IDF005=0, IDF007=1, IDF008=0, IDF009=2, IDF010=23, IDF011_01=2, IDF023_01=4, IDF024_01_01=2, IDF025_01_01=-0.25, IDF024_01_02=6, IDF025_01_02=-0.44, IDF024_01_03=16, IDF025_01_03=0.59, IDF024_01_04=9, IDF025_01_04=-0.62, IDF011_02=3, IDF023_02=4, IDF024_02_01=2, IDF025_02_01=1.03, IDF024_02_02=6, IDF025_02_02=1.85, IDF024_02_03=16, IDF025_02_03=1.1, IDF024_02_04=9, IDF025_02_04=1.84, IDF011_03=4, IDF023_03=4, IDF024_03_01=2, IDF025_03_01=-0.9500000000000001, IDF024_03_02=6, IDF025_03_02=-1.71, IDF024_03_03=16, IDF025_03_03=-0.75, IDF024_03_04=9, IDF025_03_04=-1.69, IDF011_04=5, IDF023_04=4, IDF024_04_01=2, IDF025_04_01=1.1500000000000001, IDF024_04_02=6, IDF025_04_02=2.06, IDF024_04_03=16, IDF025_04_03=0.38, IDF024_04_04=9, IDF025_04_04=2.04, IDF011_05=7, IDF023_05=4, IDF024_05_01=2, IDF025_05_01=1.54, IDF024_05_02=6, IDF025_05_02=2.7600000000000002, IDF024_05_03=16, IDF025_05_03=1.41, IDF024_05_04=9, IDF025_05_04=2.73, IDF011_06=8, IDF023_06=4, IDF024_06_01=2, IDF025_06_01=-1.08, IDF024_06_02=6, IDF025_06_02=-1.93, IDF024_06_03=16, IDF025_06_03=-1.52, IDF024_06_04=9, IDF025_06_04=-1.95, IDF011_07=9, IDF023_07=4, IDF024_07_01=2, IDF025_07_01=0.49, IDF024_07_02=6, IDF025_07_02=0.88, IDF024_07_03=16, IDF025_07_03=0.93, IDF024_07_04=9, IDF025_07_04=0.87, IDF011_08=10, IDF023_08=4, IDF024_08_01=2, IDF025_08_01=0.06, IDF024_08_02=6, IDF025_08_02=0.11, IDF024_08_03=16, IDF025_08_03=1.44, IDF024_08_04=9, IDF025_08_04=0.27, IDF011_09=11, IDF023_09=4, IDF024_09_01=2, IDF025_09_01=-4.09, IDF024_09_02=6, IDF025_09_02=-7.33, IDF024_09_03=16, IDF025_09_03=-6.78, IDF024_09_04=9, IDF025_09_04=-7.19, IDF011_10=12, IDF023_10=4, IDF024_10_01=2, IDF025_10_01=-3.19, IDF024_10_02=6, IDF025_10_02=-5.71, IDF024_10_03=16, IDF025_10_03=-5.73, IDF024_10_04=9, IDF025_10_04=-5.41, IDF011_11=13, IDF023_11=4, IDF024_11_01=2, IDF025_11_01=1.58, IDF024_11_02=6, IDF025_11_02=2.84, IDF024_11_03=16, IDF025_11_03=1.31, IDF024_11_04=9, IDF025_11_04=2.7600000000000002, IDF011_12=15, IDF023_12=4, IDF024_12_01=2, IDF025_12_01=1.98, IDF024_12_02=6, IDF025_12_02=3.5500000000000003, IDF024_12_03=16, IDF025_12_03=2.43, IDF024_12_04=9, IDF025_12_04=3.47, IDF011_13=19, IDF023_13=4, IDF024_13_01=2, IDF025_13_01=-1.07, IDF024_13_02=6, IDF025_13_02=-1.93, IDF024_13_03=16, IDF025_13_03=-1.18, IDF024_13_04=9, IDF025_13_04=-1.85, IDF011_14=21, IDF023_14=4, IDF024_14_01=2, IDF025_14_01=0.6, IDF024_14_02=6, IDF025_14_02=1.07, IDF024_14_03=16, IDF025_14_03=1.9000000000000001, IDF024_14_04=9, IDF025_14_04=1.05, IDF011_15=24, IDF023_15=4, IDF024_15_01=2, IDF025_15_01=-1.02, IDF024_15_02=6, IDF025_15_02=-1.83, IDF024_15_03=16, IDF025_15_03=-0.66, IDF024_15_04=9, IDF025_15_04=-1.9100000000000001, IDF011_16=25, IDF023_16=4, IDF024_16_01=2, IDF025_16_01=1.17, IDF024_16_02=6, IDF025_16_02=2.1, IDF024_16_03=16, IDF025_16_03=0.75, IDF024_16_04=9, IDF025_16_04=2.07, IDF011_17=26, IDF023_17=4, IDF024_17_01=2, IDF025_17_01=-1.36, IDF024_17_02=6, IDF025_17_02=-2.44, IDF024_17_03=16, IDF025_17_03=-0.59, IDF024_17_04=9, IDF025_17_04=-2.44, IDF011_18=27, IDF023_18=4, IDF024_18_01=2, IDF025_18_01=0.63, IDF024_18_02=6, IDF025_18_02=1.1300000000000001, IDF024_18_03=16, IDF025_18_03=2.0300000000000002, IDF024_18_04=9, IDF025_18_04=1.11, IDF011_19=30, IDF023_19=4, IDF024_19_01=2, IDF025_19_01=0.63, IDF024_19_02=6, IDF025_19_02=1.12, IDF024_19_03=16, IDF025_19_03=0.43, IDF024_19_04=9, IDF025_19_04=1.06, IDF011_20=31, IDF023_20=4, IDF024_20_01=2, IDF025_20_01=1.41, IDF024_20_02=6, IDF025_20_02=2.5300000000000002, IDF024_20_03=16, IDF025_20_03=0.49, IDF024_20_04=9, IDF025_20_04=2.5300000000000002, IDF011_21=33, IDF023_21=4, IDF024_21_01=2, IDF025_21_01=-1.07, IDF024_21_02=6, IDF025_21_02=-1.92, IDF024_21_03=16, IDF025_21_03=-0.49, IDF024_21_04=9, IDF025_21_04=-1.92, IDF011_22=34, IDF023_22=4, IDF024_22_01=2, IDF025_22_01=1.41, IDF024_22_02=6, IDF025_22_02=2.54, IDF024_22_03=16, IDF025_22_03=2.29, IDF024_22_04=9, IDF025_22_04=2.65, IDF011_23=36, IDF023_23=4, IDF024_23_01=2, IDF025_23_01=1.83, IDF024_23_02=6, IDF025_23_02=3.2800000000000002, IDF024_23_03=16, IDF025_23_03=2.2800000000000002, IDF024_23_04=9, IDF025_23_04=3.29)>",
            "<RTCM(4076_025, DF002=4076, IDF001=3, IDF002=25, IDF003=311120, IDF004=3, IDF005=0, IDF007=1, IDF008=0, IDF009=2, IDF010=31, IDF011_01=2, IDF023_01=3, IDF024_01_01=0, IDF025_01_01=-3.3200000000000003, IDF024_01_02=2, IDF025_01_02=-3.87, IDF024_01_03=11, IDF025_01_03=-6.37, IDF011_02=3, IDF023_02=7, IDF024_02_01=0, IDF025_02_01=1.85, IDF024_02_02=2, IDF025_02_02=2.23, IDF024_02_03=5, IDF025_02_03=3.87, IDF024_02_04=8, IDF025_02_04=3.88, IDF024_02_05=7, IDF025_02_05=3.8000000000000003, IDF024_02_06=11, IDF025_02_06=3.68, IDF024_02_07=15, IDF025_02_07=1.53, IDF011_03=4, IDF023_03=7, IDF024_03_01=0, IDF025_03_01=-0.01, IDF024_03_02=2, IDF025_03_02=0.2, IDF024_03_03=5, IDF025_03_03=0.14, IDF024_03_04=8, IDF025_03_04=0.13, IDF024_03_05=7, IDF025_03_05=0.15, IDF024_03_06=11, IDF025_03_06=0.34, IDF024_03_07=15, IDF025_03_07=1.71, IDF011_04=5, IDF023_04=6, IDF024_04_01=0, IDF025_04_01=-2.12, IDF024_04_02=2, IDF025_04_02=-1.8900000000000001, IDF024_04_03=5, IDF025_04_03=-2.97, IDF024_04_04=8, IDF025_04_04=-2.97, IDF024_04_05=7, IDF025_04_05=-2.87, IDF024_04_06=11, IDF025_04_06=-3.11, IDF011_05=6, IDF023_05=7, IDF024_05_01=0, IDF025_05_01=2.74, IDF024_05_02=2, IDF025_05_02=3.12, IDF024_05_03=5, IDF025_05_03=5.47, IDF024_05_04=8, IDF025_05_04=5.46, IDF024_05_05=7, IDF025_05_05=5.41, IDF024_05_06=11, IDF025_05_06=5.13, IDF024_05_07=15, IDF025_05_07=3.35, IDF011_06=7, IDF023_06=6, IDF024_06_01=0, IDF025_06_01=-1.82, IDF024_06_02=2, IDF025_06_02=-1.75, IDF024_06_03=5, IDF025_06_03=-2.99, IDF024_06_04=8, IDF025_06_04=-3.0, IDF024_06_05=7, IDF025_06_05=-2.94, IDF024_06_06=11, IDF025_06_06=-2.88, IDF011_07=8, IDF023_07=7, IDF024_07_01=0, IDF025_07_01=3.08, IDF024_07_02=2, IDF025_07_02=3.0100000000000002, IDF024_07_03=5, IDF025_07_03=5.36, IDF024_07_04=8, IDF025_07_04=5.36, IDF024_07_05=7, IDF025_07_05=5.28, IDF024_07_06=11, IDF025_07_06=4.95, IDF024_07_07=15, IDF025_07_07=1.8, IDF011_08=9, IDF023_08=7, IDF024_08_01=0, IDF025_08_01=2.08, IDF024_08_02=2, IDF025_08_02=2.09, IDF024_08_03=5, IDF025_08_03=3.36, IDF024_08_04=8, IDF025_08_04=3.38, IDF024_08_05=7, IDF025_08_05=3.25, IDF024_08_06=11, IDF025_08_06=3.44, IDF024_08_07=15, IDF025_08_07=1.3900000000000001, IDF011_09=10, IDF023_09=7, IDF024_09_01=0, IDF025_09_01=2.2, IDF024_09_02=2, IDF025_09_02=2.29, IDF024_09_03=5, IDF025_09_03=3.34, IDF024_09_04=8, IDF025_09_04=3.34, IDF024_09_05=7, IDF025_09_05=3.23, IDF024_09_06=11, IDF025_09_06=3.7600000000000002, IDF024_09_07=15, IDF025_09_07=-0.23, IDF011_10=11, IDF023_10=7, IDF024_10_01=0, IDF025_10_01=-1.16, IDF024_10_02=2, IDF025_10_02=-0.89, IDF024_10_03=5, IDF025_10_03=-1.6300000000000001, IDF024_10_04=8, IDF025_10_04=-1.61, IDF024_10_05=7, IDF025_10_05=-1.61, IDF024_10_06=11, IDF025_10_06=-1.47, IDF024_10_07=15, IDF025_10_07=0.3, IDF011_11=12, IDF023_11=6, IDF024_11_01=0, IDF025_11_01=-2.17, IDF024_11_02=2, IDF025_11_02=-2.12, IDF024_11_03=5, IDF025_11_03=-3.35, IDF024_11_04=8, IDF025_11_04=-3.34, IDF024_11_05=7, IDF025_11_05=-3.2800000000000002, IDF024_11_06=11, IDF025_11_06=-3.5, IDF011_12=13, IDF023_12=3, IDF024_12_01=0, IDF025_12_01=-1.69, IDF024_12_02=2, IDF025_12_02=-1.68, IDF024_12_03=11, IDF025_12_03=-2.7600000000000002, IDF011_13=14, IDF023_13=7, IDF024_13_01=0, IDF025_13_01=-0.91, IDF024_13_02=2, IDF025_13_02=-0.68, IDF024_13_03=5, IDF025_13_03=-1.23, IDF024_13_04=8, IDF025_13_04=-1.23, IDF024_13_05=7, IDF025_13_05=-1.23, IDF024_13_06=11, IDF025_13_06=-1.12, IDF024_13_07=15, IDF025_13_07=0.73, IDF011_14=15, IDF023_14=6, IDF024_14_01=0, IDF025_14_01=-2.07, IDF024_14_02=2, IDF025_14_02=-1.84, IDF024_14_03=5, IDF025_14_03=-2.96, IDF024_14_04=8, IDF025_14_04=-2.96, IDF024_14_05=7, IDF025_14_05=-2.91, IDF024_14_06=11, IDF025_14_06=-3.0300000000000002, IDF011_15=16, IDF023_15=3, IDF024_15_01=0, IDF025_15_01=-1.34, IDF024_15_02=2, IDF025_15_02=-1.72, IDF024_15_03=11, IDF025_15_03=-2.83, IDF011_16=17, IDF023_16=6, IDF024_16_01=0, IDF025_16_01=-1.7, IDF024_16_02=2, IDF025_16_02=-1.61, IDF024_16_03=5, IDF025_16_03=-2.7600000000000002, IDF024_16_04=8, IDF025_16_04=-2.7600000000000002, IDF024_16_05=7, IDF025_16_05=-2.64, IDF024_16_06=11, IDF025_16_06=-2.65, IDF011_17=18, IDF023_17=7, IDF024_17_01=0, IDF025_17_01=-1.26, IDF024_17_02=2, IDF025_17_02=-1.0, IDF024_17_03=5, IDF025_17_03=-1.74, IDF024_17_04=8, IDF025_17_04=-1.74, IDF024_17_05=7, IDF025_17_05=-1.72, IDF024_17_06=11, IDF025_17_06=-1.6500000000000001, IDF024_17_07=15, IDF025_17_07=0.28, IDF011_18=19, IDF023_18=3, IDF024_18_01=0, IDF025_18_01=-2.15, IDF024_18_02=2, IDF025_18_02=-2.91, IDF024_18_03=11, IDF025_18_03=-4.79, IDF011_19=20, IDF023_19=3, IDF024_19_01=0, IDF025_19_01=-0.38, IDF024_19_02=2, IDF025_19_02=-1.03, IDF024_19_03=11, IDF025_19_03=-1.69, IDF011_20=21, IDF023_20=3, IDF024_20_01=0, IDF025_20_01=-1.1300000000000001, IDF024_20_02=2, IDF025_20_02=-1.76, IDF024_20_03=11, IDF025_20_03=-2.91, IDF011_21=22, IDF023_21=3, IDF024_21_01=0, IDF025_21_01=-1.23, IDF024_21_02=2, IDF025_21_02=-1.56, IDF024_21_03=11, IDF025_21_03=-2.57, IDF011_22=23, IDF023_22=7, IDF024_22_01=0, IDF025_22_01=-1.27, IDF024_22_02=2, IDF025_22_02=-1.03, IDF024_22_03=5, IDF025_22_03=-1.86, IDF024_22_04=8, IDF025_22_04=-1.86, IDF024_22_05=7, IDF025_22_05=-1.83, IDF024_22_06=11, IDF025_22_06=-1.7, IDF024_22_07=15, IDF025_22_07=0.23, IDF011_23=24, IDF023_23=7, IDF024_23_01=0, IDF025_23_01=2.23, IDF024_23_02=2, IDF025_23_02=2.47, IDF024_23_03=5, IDF025_23_03=4.0, IDF024_23_04=8, IDF025_23_04=4.01, IDF024_23_05=7, IDF025_23_05=4.01, IDF024_23_06=11, IDF025_23_06=4.0600000000000005, IDF024_23_07=15, IDF025_23_07=2.35, IDF011_24=25, IDF023_24=7, IDF024_24_01=0, IDF025_24_01=3.62, IDF024_24_02=2, IDF025_24_02=3.3200000000000003, IDF024_24_03=5, IDF025_24_03=5.7700000000000005, IDF024_24_04=8, IDF025_24_04=5.76, IDF024_24_05=7, IDF025_24_05=5.69, IDF024_24_06=11, IDF025_24_06=5.47, IDF024_24_07=15, IDF025_24_07=3.75, IDF011_25=26, IDF023_25=7, IDF024_25_01=0, IDF025_25_01=3.79, IDF024_25_02=2, IDF025_25_02=3.7600000000000002, IDF024_25_03=5, IDF025_25_03=5.99, IDF024_25_04=8, IDF025_25_04=5.97, IDF024_25_05=7, IDF025_25_05=5.98, IDF024_25_06=11, IDF025_25_06=6.19, IDF024_25_07=15, IDF025_25_07=2.77, IDF011_26=27, IDF023_26=7, IDF024_26_01=0, IDF025_26_01=2.27, IDF024_26_02=2, IDF025_26_02=2.16, IDF024_26_03=5, IDF025_26_03=3.62, IDF024_26_04=8, IDF025_26_04=3.62, IDF024_26_05=7, IDF025_26_05=3.63, IDF024_26_06=11, IDF025_26_06=3.56, IDF024_26_07=15, IDF025_26_07=1.08, IDF011_27=28, IDF023_27=7, IDF024_27_01=0, IDF025_27_01=-1.5, IDF024_27_02=2, IDF025_27_02=-1.26, IDF024_27_03=5, IDF025_27_03=-2.19, IDF024_27_04=8, IDF025_27_04=-2.2, IDF024_27_05=7, IDF025_27_05=-2.16, IDF024_27_06=11, IDF025_27_06=-2.07, IDF024_27_07=15, IDF025_27_07=0.12, IDF011_28=29, IDF023_28=6, IDF024_28_01=0, IDF025_28_01=-1.6400000000000001, IDF024_28_02=2, IDF025_28_02=-1.57, IDF024_28_03=5, IDF025_28_03=-2.5300000000000002, IDF024_28_04=8, IDF025_28_04=-2.5300000000000002, IDF024_28_05=7, IDF025_28_05=-2.34, IDF024_28_06=11, IDF025_28_06=-2.58, IDF011_29=30, IDF023_29=7, IDF024_29_01=0, IDF025_29_01=3.27, IDF024_29_02=2, IDF025_29_02=2.99, IDF024_29_03=5, IDF025_29_03=5.08, IDF024_29_04=8, IDF025_29_04=5.07, IDF024_29_05=7, IDF025_29_05=5.08, IDF024_29_06=11, IDF025_29_06=4.92, IDF024_29_07=15, IDF025_29_07=2.73, IDF011_30=31, IDF023_30=6, IDF024_30_01=0, IDF025_30_01=-2.57, IDF024_30_02=2, IDF025_30_02=-2.4, IDF024_30_03=5, IDF025_30_03=-3.8200000000000003, IDF024_30_04=8, IDF025_30_04=-3.83, IDF024_30_05=7, IDF025_30_05=-3.79, IDF024_30_06=11, IDF025_30_06=-3.95, IDF011_31=32, IDF023_31=7, IDF024_31_01=0, IDF025_31_01=1.46, IDF024_31_02=2, IDF025_31_02=1.79, IDF024_31_03=5, IDF025_31_03=3.14, IDF024_31_04=8, IDF025_31_04=3.13, IDF024_31_05=7, IDF025_31_05=3.0300000000000002, IDF024_31_06=11, IDF025_31_06=2.94, IDF024_31_07=15, IDF025_31_07=-0.41000000000000003)>",
            "<RTCM(4076_045, DF002=4076, IDF001=3, IDF002=45, IDF003=311120, IDF004=3, IDF005=0, IDF007=1, IDF008=0, IDF009=2, IDF010=21, IDF011_01=1, IDF023_01=4, IDF024_01_01=0, IDF025_01_01=2.54, IDF024_01_02=1, IDF025_01_02=2.36, IDF024_01_03=2, IDF025_01_03=3.42, IDF024_01_04=3, IDF025_01_04=3.91, IDF011_02=2, IDF023_02=4, IDF024_02_01=0, IDF025_02_01=0.53, IDF024_02_02=1, IDF025_02_02=0.49, IDF024_02_03=2, IDF025_02_03=0.79, IDF024_02_04=3, IDF025_02_04=0.8200000000000001, IDF011_03=3, IDF023_03=4, IDF024_03_01=0, IDF025_03_01=-0.97, IDF024_03_02=1, IDF025_03_02=-0.86, IDF024_03_03=2, IDF025_03_03=-1.3900000000000001, IDF024_03_04=3, IDF025_03_04=-1.43, IDF011_04=4, IDF023_04=4, IDF024_04_01=0, IDF025_04_01=-1.29, IDF024_04_02=1, IDF025_04_02=-1.1, IDF024_04_03=2, IDF025_04_03=-1.86, IDF024_04_04=3, IDF025_04_04=-1.81, IDF011_05=5, IDF023_05=4, IDF024_05_01=0, IDF025_05_01=-0.51, IDF024_05_02=1, IDF025_05_02=-0.28, IDF024_05_03=2, IDF025_05_03=-0.47000000000000003, IDF024_05_04=3, IDF025_05_04=-0.47000000000000003, IDF011_06=7, IDF023_06=4, IDF024_06_01=0, IDF025_06_01=-0.23, IDF024_06_02=1, IDF025_06_02=-0.13, IDF024_06_03=2, IDF025_06_03=-0.13, IDF024_06_04=3, IDF025_06_04=-0.22, IDF011_07=8, IDF023_07=4, IDF024_07_01=0, IDF025_07_01=-1.79, IDF024_07_02=1, IDF025_07_02=-1.5, IDF024_07_03=2, IDF025_07_03=-2.5100000000000002, IDF024_07_04=3, IDF025_07_04=-2.48, IDF011_08=9, IDF023_08=4, IDF024_08_01=0, IDF025_08_01=-1.58, IDF024_08_02=1, IDF025_08_02=-1.37, IDF024_08_03=2, IDF025_08_03=-2.29, IDF024_08_04=3, IDF025_08_04=-2.27, IDF011_09=11, IDF023_09=4, IDF024_09_01=0, IDF025_09_01=-1.93, IDF024_09_02=1, IDF025_09_02=-1.79, IDF024_09_03=2, IDF025_09_03=-2.94, IDF024_09_04=3, IDF025_09_04=-2.96, IDF011_10=12, IDF023_10=4, IDF024_10_01=0, IDF025_10_01=0.6, IDF024_10_02=1, IDF025_10_02=0.75, IDF024_10_03=2, IDF025_10_03=1.08, IDF024_10_04=3, IDF025_10_04=1.24, IDF011_11=13, IDF023_11=4, IDF024_11_01=0, IDF025_11_01=1.52, IDF024_11_02=1, IDF025_11_02=0.92, IDF024_11_03=2, IDF025_11_03=1.43, IDF024_11_04=3, IDF025_11_04=1.53, IDF011_12=14, IDF023_12=4, IDF024_12_01=0, IDF025_12_01=0.8300000000000001, IDF024_12_02=1, IDF025_12_02=0.84, IDF024_12_03=2, IDF025_12_03=1.45, IDF024_12_04=3, IDF025_12_04=1.3900000000000001, IDF011_13=15, IDF023_13=4, IDF024_13_01=0, IDF025_13_01=-0.13, IDF024_13_02=1, IDF025_13_02=0.07, IDF024_13_03=2, IDF025_13_03=0.02, IDF024_13_04=3, IDF025_13_04=0.12, IDF011_14=16, IDF023_14=4, IDF024_14_01=0, IDF025_14_01=0.14, IDF024_14_02=1, IDF025_14_02=0.27, IDF024_14_03=2, IDF025_14_03=0.41000000000000003, IDF024_14_04=3, IDF025_14_04=0.44, IDF011_15=17, IDF023_15=4, IDF024_15_01=0, IDF025_15_01=-0.26, IDF024_15_02=1, IDF025_15_02=-0.23, IDF024_15_03=2, IDF025_15_03=-0.41000000000000003, IDF024_15_04=3, IDF025_15_04=-0.38, IDF011_16=18, IDF023_16=4, IDF024_16_01=0, IDF025_16_01=0.55, IDF024_16_02=1, IDF025_16_02=0.66, IDF024_16_03=2, IDF025_16_03=0.96, IDF024_16_04=3, IDF025_16_04=1.09, IDF011_17=19, IDF023_17=4, IDF024_17_01=0, IDF025_17_01=-0.04, IDF024_17_02=1, IDF025_17_02=-0.28, IDF024_17_03=2, IDF025_17_03=-0.01, IDF024_17_04=3, IDF025_17_04=-0.46, IDF011_18=20, IDF023_18=4, IDF024_18_01=0, IDF025_18_01=2.63, IDF024_18_02=1, IDF025_18_02=2.61, IDF024_18_03=2, IDF025_18_03=4.98, IDF024_18_04=3, IDF025_18_04=4.3100000000000005, IDF011_19=21, IDF023_19=4, IDF024_19_01=0, IDF025_19_01=-0.04, IDF024_19_02=1, IDF025_19_02=0.0, IDF024_19_03=2, IDF025_19_03=-0.13, IDF024_19_04=3, IDF025_19_04=0.0, IDF011_20=22, IDF023_20=4, IDF024_20_01=0, IDF025_20_01=-1.49, IDF024_20_02=1, IDF025_20_02=-1.3800000000000001, IDF024_20_03=2, IDF025_20_03=-2.41, IDF024_20_04=3, IDF025_20_04=-2.29, IDF011_21=24, IDF023_21=4, IDF024_21_01=0, IDF025_21_01=-0.07, IDF024_21_02=1, IDF025_21_02=-0.03, IDF024_21_03=2, IDF025_21_03=-0.08, IDF024_21_04=3, IDF025_21_04=-0.06)>",
            "<RTCM(4076_065, DF002=4076, IDF001=3, IDF002=65, IDF003=311130, IDF004=3, IDF005=0, IDF007=1, IDF008=0, IDF009=2, IDF010=23, IDF011_01=2, IDF023_01=4, IDF024_01_01=2, IDF025_01_01=-0.25, IDF024_01_02=6, IDF025_01_02=-0.44, IDF024_01_03=16, IDF025_01_03=0.59, IDF024_01_04=9, IDF025_01_04=-0.62, IDF011_02=3, IDF023_02=4, IDF024_02_01=2, IDF025_02_01=1.03, IDF024_02_02=6, IDF025_02_02=1.85, IDF024_02_03=16, IDF025_02_03=1.1, IDF024_02_04=9, IDF025_02_04=1.84, IDF011_03=4, IDF023_03=4, IDF024_03_01=2, IDF025_03_01=-0.9500000000000001, IDF024_03_02=6, IDF025_03_02=-1.71, IDF024_03_03=16, IDF025_03_03=-0.75, IDF024_03_04=9, IDF025_03_04=-1.69, IDF011_04=5, IDF023_04=4, IDF024_04_01=2, IDF025_04_01=1.1500000000000001, IDF024_04_02=6, IDF025_04_02=2.06, IDF024_04_03=16, IDF025_04_03=0.38, IDF024_04_04=9, IDF025_04_04=2.04, IDF011_05=7, IDF023_05=4, IDF024_05_01=2, IDF025_05_01=1.54, IDF024_05_02=6, IDF025_05_02=2.7600000000000002, IDF024_05_03=16, IDF025_05_03=1.41, IDF024_05_04=9, IDF025_05_04=2.73, IDF011_06=8, IDF023_06=4, IDF024_06_01=2, IDF025_06_01=-1.08, IDF024_06_02=6, IDF025_06_02=-1.93, IDF024_06_03=16, IDF025_06_03=-1.52, IDF024_06_04=9, IDF025_06_04=-1.95, IDF011_07=9, IDF023_07=4, IDF024_07_01=2, IDF025_07_01=0.49, IDF024_07_02=6, IDF025_07_02=0.88, IDF024_07_03=16, IDF025_07_03=0.93, IDF024_07_04=9, IDF025_07_04=0.87, IDF011_08=10, IDF023_08=4, IDF024_08_01=2, IDF025_08_01=0.06, IDF024_08_02=6, IDF025_08_02=0.11, IDF024_08_03=16, IDF025_08_03=1.44, IDF024_08_04=9, IDF025_08_04=0.27, IDF011_09=11, IDF023_09=4, IDF024_09_01=2, IDF025_09_01=-4.09, IDF024_09_02=6, IDF025_09_02=-7.33, IDF024_09_03=16, IDF025_09_03=-6.78, IDF024_09_04=9, IDF025_09_04=-7.19, IDF011_10=12, IDF023_10=4, IDF024_10_01=2, IDF025_10_01=-3.19, IDF024_10_02=6, IDF025_10_02=-5.71, IDF024_10_03=16, IDF025_10_03=-5.73, IDF024_10_04=9, IDF025_10_04=-5.41, IDF011_11=13, IDF023_11=4, IDF024_11_01=2, IDF025_11_01=1.58, IDF024_11_02=6, IDF025_11_02=2.84, IDF024_11_03=16, IDF025_11_03=1.31, IDF024_11_04=9, IDF025_11_04=2.7600000000000002, IDF011_12=15, IDF023_12=4, IDF024_12_01=2, IDF025_12_01=1.98, IDF024_12_02=6, IDF025_12_02=3.5500000000000003, IDF024_12_03=16, IDF025_12_03=2.43, IDF024_12_04=9, IDF025_12_04=3.47, IDF011_13=19, IDF023_13=4, IDF024_13_01=2, IDF025_13_01=-1.07, IDF024_13_02=6, IDF025_13_02=-1.93, IDF024_13_03=16, IDF025_13_03=-1.18, IDF024_13_04=9, IDF025_13_04=-1.85, IDF011_14=21, IDF023_14=4, IDF024_14_01=2, IDF025_14_01=0.6, IDF024_14_02=6, IDF025_14_02=1.07, IDF024_14_03=16, IDF025_14_03=1.9000000000000001, IDF024_14_04=9, IDF025_14_04=1.05, IDF011_15=24, IDF023_15=4, IDF024_15_01=2, IDF025_15_01=-1.02, IDF024_15_02=6, IDF025_15_02=-1.83, IDF024_15_03=16, IDF025_15_03=-0.66, IDF024_15_04=9, IDF025_15_04=-1.9100000000000001, IDF011_16=25, IDF023_16=4, IDF024_16_01=2, IDF025_16_01=1.17, IDF024_16_02=6, IDF025_16_02=2.1, IDF024_16_03=16, IDF025_16_03=0.75, IDF024_16_04=9, IDF025_16_04=2.07, IDF011_17=26, IDF023_17=4, IDF024_17_01=2, IDF025_17_01=-1.36, IDF024_17_02=6, IDF025_17_02=-2.44, IDF024_17_03=16, IDF025_17_03=-0.59, IDF024_17_04=9, IDF025_17_04=-2.44, IDF011_18=27, IDF023_18=4, IDF024_18_01=2, IDF025_18_01=0.63, IDF024_18_02=6, IDF025_18_02=1.1300000000000001, IDF024_18_03=16, IDF025_18_03=2.0300000000000002, IDF024_18_04=9, IDF025_18_04=1.11, IDF011_19=30, IDF023_19=4, IDF024_19_01=2, IDF025_19_01=0.63, IDF024_19_02=6, IDF025_19_02=1.12, IDF024_19_03=16, IDF025_19_03=0.43, IDF024_19_04=9, IDF025_19_04=1.06, IDF011_20=31, IDF023_20=4, IDF024_20_01=2, IDF025_20_01=1.41, IDF024_20_02=6, IDF025_20_02=2.5300000000000002, IDF024_20_03=16, IDF025_20_03=0.49, IDF024_20_04=9, IDF025_20_04=2.5300000000000002, IDF011_21=33, IDF023_21=4, IDF024_21_01=2, IDF025_21_01=-1.07, IDF024_21_02=6, IDF025_21_02=-1.92, IDF024_21_03=16, IDF025_21_03=-0.49, IDF024_21_04=9, IDF025_21_04=-1.92, IDF011_22=34, IDF023_22=4, IDF024_22_01=2, IDF025_22_01=1.41, IDF024_22_02=6, IDF025_22_02=2.54, IDF024_22_03=16, IDF025_22_03=2.29, IDF024_22_04=9, IDF025_22_04=2.65, IDF011_23=36, IDF023_23=4, IDF024_23_01=2, IDF025_23_01=1.83, IDF024_23_02=6, IDF025_23_02=3.2800000000000002, IDF024_23_03=16, IDF025_23_03=2.2800000000000002, IDF024_23_04=9, IDF025_23_04=3.29)>",
            "<RTCM(4076_025, DF002=4076, IDF001=3, IDF002=25, IDF003=311130, IDF004=3, IDF005=0, IDF007=1, IDF008=0, IDF009=2, IDF010=31, IDF011_01=2, IDF023_01=3, IDF024_01_01=0, IDF025_01_01=-3.3200000000000003, IDF024_01_02=2, IDF025_01_02=-3.87, IDF024_01_03=11, IDF025_01_03=-6.37, IDF011_02=3, IDF023_02=7, IDF024_02_01=0, IDF025_02_01=1.85, IDF024_02_02=2, IDF025_02_02=2.23, IDF024_02_03=5, IDF025_02_03=3.87, IDF024_02_04=8, IDF025_02_04=3.88, IDF024_02_05=7, IDF025_02_05=3.8000000000000003, IDF024_02_06=11, IDF025_02_06=3.68, IDF024_02_07=15, IDF025_02_07=1.53, IDF011_03=4, IDF023_03=7, IDF024_03_01=0, IDF025_03_01=-0.01, IDF024_03_02=2, IDF025_03_02=0.2, IDF024_03_03=5, IDF025_03_03=0.14, IDF024_03_04=8, IDF025_03_04=0.13, IDF024_03_05=7, IDF025_03_05=0.15, IDF024_03_06=11, IDF025_03_06=0.34, IDF024_03_07=15, IDF025_03_07=1.71, IDF011_04=5, IDF023_04=6, IDF024_04_01=0, IDF025_04_01=-2.12, IDF024_04_02=2, IDF025_04_02=-1.8900000000000001, IDF024_04_03=5, IDF025_04_03=-2.97, IDF024_04_04=8, IDF025_04_04=-2.97, IDF024_04_05=7, IDF025_04_05=-2.87, IDF024_04_06=11, IDF025_04_06=-3.11, IDF011_05=6, IDF023_05=7, IDF024_05_01=0, IDF025_05_01=2.74, IDF024_05_02=2, IDF025_05_02=3.12, IDF024_05_03=5, IDF025_05_03=5.47, IDF024_05_04=8, IDF025_05_04=5.46, IDF024_05_05=7, IDF025_05_05=5.41, IDF024_05_06=11, IDF025_05_06=5.13, IDF024_05_07=15, IDF025_05_07=3.35, IDF011_06=7, IDF023_06=6, IDF024_06_01=0, IDF025_06_01=-1.82, IDF024_06_02=2, IDF025_06_02=-1.75, IDF024_06_03=5, IDF025_06_03=-2.99, IDF024_06_04=8, IDF025_06_04=-3.0, IDF024_06_05=7, IDF025_06_05=-2.94, IDF024_06_06=11, IDF025_06_06=-2.88, IDF011_07=8, IDF023_07=7, IDF024_07_01=0, IDF025_07_01=3.08, IDF024_07_02=2, IDF025_07_02=3.0100000000000002, IDF024_07_03=5, IDF025_07_03=5.36, IDF024_07_04=8, IDF025_07_04=5.36, IDF024_07_05=7, IDF025_07_05=5.28, IDF024_07_06=11, IDF025_07_06=4.95, IDF024_07_07=15, IDF025_07_07=1.8, IDF011_08=9, IDF023_08=7, IDF024_08_01=0, IDF025_08_01=2.08, IDF024_08_02=2, IDF025_08_02=2.09, IDF024_08_03=5, IDF025_08_03=3.36, IDF024_08_04=8, IDF025_08_04=3.38, IDF024_08_05=7, IDF025_08_05=3.25, IDF024_08_06=11, IDF025_08_06=3.44, IDF024_08_07=15, IDF025_08_07=1.3900000000000001, IDF011_09=10, IDF023_09=7, IDF024_09_01=0, IDF025_09_01=2.2, IDF024_09_02=2, IDF025_09_02=2.29, IDF024_09_03=5, IDF025_09_03=3.34, IDF024_09_04=8, IDF025_09_04=3.34, IDF024_09_05=7, IDF025_09_05=3.23, IDF024_09_06=11, IDF025_09_06=3.7600000000000002, IDF024_09_07=15, IDF025_09_07=-0.23, IDF011_10=11, IDF023_10=7, IDF024_10_01=0, IDF025_10_01=-1.16, IDF024_10_02=2, IDF025_10_02=-0.89, IDF024_10_03=5, IDF025_10_03=-1.6300000000000001, IDF024_10_04=8, IDF025_10_04=-1.61, IDF024_10_05=7, IDF025_10_05=-1.61, IDF024_10_06=11, IDF025_10_06=-1.47, IDF024_10_07=15, IDF025_10_07=0.3, IDF011_11=12, IDF023_11=6, IDF024_11_01=0, IDF025_11_01=-2.17, IDF024_11_02=2, IDF025_11_02=-2.12, IDF024_11_03=5, IDF025_11_03=-3.35, IDF024_11_04=8, IDF025_11_04=-3.34, IDF024_11_05=7, IDF025_11_05=-3.2800000000000002, IDF024_11_06=11, IDF025_11_06=-3.5, IDF011_12=13, IDF023_12=3, IDF024_12_01=0, IDF025_12_01=-1.69, IDF024_12_02=2, IDF025_12_02=-1.68, IDF024_12_03=11, IDF025_12_03=-2.7600000000000002, IDF011_13=14, IDF023_13=7, IDF024_13_01=0, IDF025_13_01=-0.91, IDF024_13_02=2, IDF025_13_02=-0.68, IDF024_13_03=5, IDF025_13_03=-1.23, IDF024_13_04=8, IDF025_13_04=-1.23, IDF024_13_05=7, IDF025_13_05=-1.23, IDF024_13_06=11, IDF025_13_06=-1.12, IDF024_13_07=15, IDF025_13_07=0.73, IDF011_14=15, IDF023_14=6, IDF024_14_01=0, IDF025_14_01=-2.07, IDF024_14_02=2, IDF025_14_02=-1.84, IDF024_14_03=5, IDF025_14_03=-2.96, IDF024_14_04=8, IDF025_14_04=-2.96, IDF024_14_05=7, IDF025_14_05=-2.91, IDF024_14_06=11, IDF025_14_06=-3.0300000000000002, IDF011_15=16, IDF023_15=3, IDF024_15_01=0, IDF025_15_01=-1.34, IDF024_15_02=2, IDF025_15_02=-1.72, IDF024_15_03=11, IDF025_15_03=-2.83, IDF011_16=17, IDF023_16=6, IDF024_16_01=0, IDF025_16_01=-1.7, IDF024_16_02=2, IDF025_16_02=-1.61, IDF024_16_03=5, IDF025_16_03=-2.7600000000000002, IDF024_16_04=8, IDF025_16_04=-2.7600000000000002, IDF024_16_05=7, IDF025_16_05=-2.64, IDF024_16_06=11, IDF025_16_06=-2.65, IDF011_17=18, IDF023_17=7, IDF024_17_01=0, IDF025_17_01=-1.26, IDF024_17_02=2, IDF025_17_02=-1.0, IDF024_17_03=5, IDF025_17_03=-1.74, IDF024_17_04=8, IDF025_17_04=-1.74, IDF024_17_05=7, IDF025_17_05=-1.72, IDF024_17_06=11, IDF025_17_06=-1.6500000000000001, IDF024_17_07=15, IDF025_17_07=0.28, IDF011_18=19, IDF023_18=3, IDF024_18_01=0, IDF025_18_01=-2.15, IDF024_18_02=2, IDF025_18_02=-2.91, IDF024_18_03=11, IDF025_18_03=-4.79, IDF011_19=20, IDF023_19=3, IDF024_19_01=0, IDF025_19_01=-0.38, IDF024_19_02=2, IDF025_19_02=-1.03, IDF024_19_03=11, IDF025_19_03=-1.69, IDF011_20=21, IDF023_20=3, IDF024_20_01=0, IDF025_20_01=-1.1300000000000001, IDF024_20_02=2, IDF025_20_02=-1.76, IDF024_20_03=11, IDF025_20_03=-2.91, IDF011_21=22, IDF023_21=3, IDF024_21_01=0, IDF025_21_01=-1.23, IDF024_21_02=2, IDF025_21_02=-1.56, IDF024_21_03=11, IDF025_21_03=-2.57, IDF011_22=23, IDF023_22=7, IDF024_22_01=0, IDF025_22_01=-1.27, IDF024_22_02=2, IDF025_22_02=-1.03, IDF024_22_03=5, IDF025_22_03=-1.86, IDF024_22_04=8, IDF025_22_04=-1.86, IDF024_22_05=7, IDF025_22_05=-1.83, IDF024_22_06=11, IDF025_22_06=-1.7, IDF024_22_07=15, IDF025_22_07=0.23, IDF011_23=24, IDF023_23=7, IDF024_23_01=0, IDF025_23_01=2.23, IDF024_23_02=2, IDF025_23_02=2.47, IDF024_23_03=5, IDF025_23_03=4.0, IDF024_23_04=8, IDF025_23_04=4.01, IDF024_23_05=7, IDF025_23_05=4.01, IDF024_23_06=11, IDF025_23_06=4.0600000000000005, IDF024_23_07=15, IDF025_23_07=2.35, IDF011_24=25, IDF023_24=7, IDF024_24_01=0, IDF025_24_01=3.62, IDF024_24_02=2, IDF025_24_02=3.3200000000000003, IDF024_24_03=5, IDF025_24_03=5.7700000000000005, IDF024_24_04=8, IDF025_24_04=5.76, IDF024_24_05=7, IDF025_24_05=5.69, IDF024_24_06=11, IDF025_24_06=5.47, IDF024_24_07=15, IDF025_24_07=3.75, IDF011_25=26, IDF023_25=7, IDF024_25_01=0, IDF025_25_01=3.79, IDF024_25_02=2, IDF025_25_02=3.7600000000000002, IDF024_25_03=5, IDF025_25_03=5.99, IDF024_25_04=8, IDF025_25_04=5.97, IDF024_25_05=7, IDF025_25_05=5.98, IDF024_25_06=11, IDF025_25_06=6.19, IDF024_25_07=15, IDF025_25_07=2.77, IDF011_26=27, IDF023_26=7, IDF024_26_01=0, IDF025_26_01=2.27, IDF024_26_02=2, IDF025_26_02=2.16, IDF024_26_03=5, IDF025_26_03=3.62, IDF024_26_04=8, IDF025_26_04=3.62, IDF024_26_05=7, IDF025_26_05=3.63, IDF024_26_06=11, IDF025_26_06=3.56, IDF024_26_07=15, IDF025_26_07=1.08, IDF011_27=28, IDF023_27=7, IDF024_27_01=0, IDF025_27_01=-1.5, IDF024_27_02=2, IDF025_27_02=-1.26, IDF024_27_03=5, IDF025_27_03=-2.19, IDF024_27_04=8, IDF025_27_04=-2.2, IDF024_27_05=7, IDF025_27_05=-2.16, IDF024_27_06=11, IDF025_27_06=-2.07, IDF024_27_07=15, IDF025_27_07=0.12, IDF011_28=29, IDF023_28=6, IDF024_28_01=0, IDF025_28_01=-1.6400000000000001, IDF024_28_02=2, IDF025_28_02=-1.57, IDF024_28_03=5, IDF025_28_03=-2.5300000000000002, IDF024_28_04=8, IDF025_28_04=-2.5300000000000002, IDF024_28_05=7, IDF025_28_05=-2.34, IDF024_28_06=11, IDF025_28_06=-2.58, IDF011_29=30, IDF023_29=7, IDF024_29_01=0, IDF025_29_01=3.27, IDF024_29_02=2, IDF025_29_02=2.99, IDF024_29_03=5, IDF025_29_03=5.08, IDF024_29_04=8, IDF025_29_04=5.07, IDF024_29_05=7, IDF025_29_05=5.08, IDF024_29_06=11, IDF025_29_06=4.92, IDF024_29_07=15, IDF025_29_07=2.73, IDF011_30=31, IDF023_30=6, IDF024_30_01=0, IDF025_30_01=-2.57, IDF024_30_02=2, IDF025_30_02=-2.4, IDF024_30_03=5, IDF025_30_03=-3.8200000000000003, IDF024_30_04=8, IDF025_30_04=-3.83, IDF024_30_05=7, IDF025_30_05=-3.79, IDF024_30_06=11, IDF025_30_06=-3.95, IDF011_31=32, IDF023_31=7, IDF024_31_01=0, IDF025_31_01=1.46, IDF024_31_02=2, IDF025_31_02=1.79, IDF024_31_03=5, IDF025_31_03=3.14, IDF024_31_04=8, IDF025_31_04=3.13, IDF024_31_05=7, IDF025_31_05=3.0300000000000002, IDF024_31_06=11, IDF025_31_06=2.94, IDF024_31_07=15, IDF025_31_07=-0.41000000000000003)>",
            "<RTCM(4076_045, DF002=4076, IDF001=3, IDF002=45, IDF003=311130, IDF004=3, IDF005=0, IDF007=1, IDF008=0, IDF009=2, IDF010=21, IDF011_01=1, IDF023_01=4, IDF024_01_01=0, IDF025_01_01=2.54, IDF024_01_02=1, IDF025_01_02=2.36, IDF024_01_03=2, IDF025_01_03=3.42, IDF024_01_04=3, IDF025_01_04=3.91, IDF011_02=2, IDF023_02=4, IDF024_02_01=0, IDF025_02_01=0.53, IDF024_02_02=1, IDF025_02_02=0.49, IDF024_02_03=2, IDF025_02_03=0.79, IDF024_02_04=3, IDF025_02_04=0.8200000000000001, IDF011_03=3, IDF023_03=4, IDF024_03_01=0, IDF025_03_01=-0.97, IDF024_03_02=1, IDF025_03_02=-0.86, IDF024_03_03=2, IDF025_03_03=-1.3900000000000001, IDF024_03_04=3, IDF025_03_04=-1.43, IDF011_04=4, IDF023_04=4, IDF024_04_01=0, IDF025_04_01=-1.29, IDF024_04_02=1, IDF025_04_02=-1.1, IDF024_04_03=2, IDF025_04_03=-1.86, IDF024_04_04=3, IDF025_04_04=-1.81, IDF011_05=5, IDF023_05=4, IDF024_05_01=0, IDF025_05_01=-0.51, IDF024_05_02=1, IDF025_05_02=-0.28, IDF024_05_03=2, IDF025_05_03=-0.47000000000000003, IDF024_05_04=3, IDF025_05_04=-0.47000000000000003, IDF011_06=7, IDF023_06=4, IDF024_06_01=0, IDF025_06_01=-0.23, IDF024_06_02=1, IDF025_06_02=-0.13, IDF024_06_03=2, IDF025_06_03=-0.13, IDF024_06_04=3, IDF025_06_04=-0.22, IDF011_07=8, IDF023_07=4, IDF024_07_01=0, IDF025_07_01=-1.79, IDF024_07_02=1, IDF025_07_02=-1.5, IDF024_07_03=2, IDF025_07_03=-2.5100000000000002, IDF024_07_04=3, IDF025_07_04=-2.48, IDF011_08=9, IDF023_08=4, IDF024_08_01=0, IDF025_08_01=-1.58, IDF024_08_02=1, IDF025_08_02=-1.37, IDF024_08_03=2, IDF025_08_03=-2.29, IDF024_08_04=3, IDF025_08_04=-2.27, IDF011_09=11, IDF023_09=4, IDF024_09_01=0, IDF025_09_01=-1.93, IDF024_09_02=1, IDF025_09_02=-1.79, IDF024_09_03=2, IDF025_09_03=-2.94, IDF024_09_04=3, IDF025_09_04=-2.96, IDF011_10=12, IDF023_10=4, IDF024_10_01=0, IDF025_10_01=0.6, IDF024_10_02=1, IDF025_10_02=0.75, IDF024_10_03=2, IDF025_10_03=1.08, IDF024_10_04=3, IDF025_10_04=1.24, IDF011_11=13, IDF023_11=4, IDF024_11_01=0, IDF025_11_01=1.52, IDF024_11_02=1, IDF025_11_02=0.92, IDF024_11_03=2, IDF025_11_03=1.43, IDF024_11_04=3, IDF025_11_04=1.53, IDF011_12=14, IDF023_12=4, IDF024_12_01=0, IDF025_12_01=0.8300000000000001, IDF024_12_02=1, IDF025_12_02=0.84, IDF024_12_03=2, IDF025_12_03=1.45, IDF024_12_04=3, IDF025_12_04=1.3900000000000001, IDF011_13=15, IDF023_13=4, IDF024_13_01=0, IDF025_13_01=-0.13, IDF024_13_02=1, IDF025_13_02=0.07, IDF024_13_03=2, IDF025_13_03=0.02, IDF024_13_04=3, IDF025_13_04=0.12, IDF011_14=16, IDF023_14=4, IDF024_14_01=0, IDF025_14_01=0.14, IDF024_14_02=1, IDF025_14_02=0.27, IDF024_14_03=2, IDF025_14_03=0.41000000000000003, IDF024_14_04=3, IDF025_14_04=0.44, IDF011_15=17, IDF023_15=4, IDF024_15_01=0, IDF025_15_01=-0.26, IDF024_15_02=1, IDF025_15_02=-0.23, IDF024_15_03=2, IDF025_15_03=-0.41000000000000003, IDF024_15_04=3, IDF025_15_04=-0.38, IDF011_16=18, IDF023_16=4, IDF024_16_01=0, IDF025_16_01=0.55, IDF024_16_02=1, IDF025_16_02=0.66, IDF024_16_03=2, IDF025_16_03=0.96, IDF024_16_04=3, IDF025_16_04=1.09, IDF011_17=19, IDF023_17=4, IDF024_17_01=0, IDF025_17_01=-0.04, IDF024_17_02=1, IDF025_17_02=-0.28, IDF024_17_03=2, IDF025_17_03=-0.01, IDF024_17_04=3, IDF025_17_04=-0.46, IDF011_18=20, IDF023_18=4, IDF024_18_01=0, IDF025_18_01=2.63, IDF024_18_02=1, IDF025_18_02=2.61, IDF024_18_03=2, IDF025_18_03=4.98, IDF024_18_04=3, IDF025_18_04=4.3100000000000005, IDF011_19=21, IDF023_19=4, IDF024_19_01=0, IDF025_19_01=-0.04, IDF024_19_02=1, IDF025_19_02=0.0, IDF024_19_03=2, IDF025_19_03=-0.13, IDF024_19_04=3, IDF025_19_04=0.0, IDF011_20=22, IDF023_20=4, IDF024_20_01=0, IDF025_20_01=-1.49, IDF024_20_02=1, IDF025_20_02=-1.3800000000000001, IDF024_20_03=2, IDF025_20_03=-2.41, IDF024_20_04=3, IDF025_20_04=-2.29, IDF011_21=24, IDF023_21=4, IDF024_21_01=0, IDF025_21_01=-0.07, IDF024_21_02=1, IDF025_21_02=-0.03, IDF024_21_03=2, IDF025_21_03=-0.08, IDF024_21_04=3, IDF025_21_04=-0.06)>",
        ]
        dirname = os.path.dirname(__file__)
        with open(os.path.join(dirname, "pygpsdata-NTRIP-4076.log"), "rb") as stream:
            i = 0
            raw = 0
            rtr = RTCMReader(stream, labelmsm=True)
            for raw, parsed in rtr:
                if raw is not None:
                    # print(f'"{parsed}",')
                    self.assertEqual(f"{parsed}", EXPECTED_RESULTS[i])
                    i += 1

    def testigsssr1240_1264(
        self,
    ):  # test 1240-1264 messages using log from NTRIP caster products.igs-ip.net, mountpoint SSRA00CNE1
        EXPECTED_RESULTS = [
            "<RTCM(1259, DF002=1259, DF465=338741, DF391=2, DF388=0, DF413=0, DF414=0, DF415=0, DF387=43, DF488_01=1, DF376_01=-1242.2, DF377_01=0.0, DF378_01=0.0, DF488_02=2, DF376_02=-5083.1, DF377_02=0.0, DF378_02=0.0, DF488_03=3, DF376_03=-5913.5, DF377_03=0.0, DF378_03=0.0, DF488_04=4, DF376_04=-10316.2, DF377_04=0.0, DF378_04=0.0, DF488_05=5, DF376_05=-4307.7, DF377_05=0.0, DF378_05=0.0, DF488_06=6, DF376_06=-11986.1, DF377_06=0.0, DF378_06=0.0, DF488_07=7, DF376_07=-18087.600000000002, DF377_07=0.0, DF378_07=0.0, DF488_08=9, DF376_08=-10871.300000000001, DF377_08=0.0, DF378_08=0.0, DF488_09=10, DF376_09=-9945.2, DF377_09=0.0, DF378_09=0.0, DF488_10=11, DF376_10=-5026.700000000001, DF377_10=0.0, DF378_10=0.0, DF488_11=12, DF376_11=-5382.8, DF377_11=0.0, DF378_11=0.0, DF488_12=13, DF376_12=5986.700000000001, DF377_12=0.0, DF378_12=0.0, DF488_13=14, DF376_13=-8109.3, DF377_13=0.0, DF378_13=0.0, DF488_14=16, DF376_14=-1418.1000000000001, DF377_14=0.0, DF378_14=0.0, DF488_15=19, DF376_15=-6802.3, DF377_15=0.0, DF378_15=0.0, DF488_16=20, DF376_16=-9784.0, DF377_16=0.0, DF378_16=0.0, DF488_17=21, DF376_17=-9120.7, DF377_17=0.0, DF378_17=0.0, DF488_18=22, DF376_18=-11134.900000000001, DF377_18=0.0, DF378_18=0.0, DF488_19=23, DF376_19=-17933.8, DF377_19=0.0, DF378_19=0.0, DF488_20=24, DF376_20=-2614.5, DF377_20=0.0, DF378_20=0.0, DF488_21=25, DF376_21=1244.9, DF377_21=0.0, DF378_21=0.0, DF488_22=26, DF376_22=5326.200000000001, DF377_22=0.0, DF378_22=0.0, DF488_23=27, DF376_23=4410.7, DF377_23=0.0, DF378_23=0.0, DF488_24=28, DF376_24=5580.8, DF377_24=0.0, DF378_24=0.0, DF488_25=29, DF376_25=1374.3000000000002, DF377_25=0.0, DF378_25=0.0, DF488_26=30, DF376_26=9591.9, DF377_26=0.0, DF378_26=0.0, DF488_27=32, DF376_27=7473.5, DF377_27=0.0, DF378_27=0.0, DF488_28=33, DF376_28=40740.700000000004, DF377_28=0.0, DF378_28=0.0, DF488_29=34, DF376_29=7249.3, DF377_29=0.0, DF378_29=0.0, DF488_30=35, DF376_30=5381.8, DF377_30=0.0, DF378_30=0.0, DF488_31=36, DF376_31=20111.5, DF377_31=0.0, DF378_31=0.0, DF488_32=37, DF376_32=16168.5, DF377_32=0.0, DF378_32=0.0, DF488_33=38, DF376_33=-108.10000000000001, DF377_33=0.0, DF378_33=0.0, DF488_34=39, DF376_34=-3994.3, DF377_34=0.0, DF378_34=0.0, DF488_35=40, DF376_35=702.2, DF377_35=0.0, DF378_35=0.0, DF488_36=41, DF376_36=22051.300000000003, DF377_36=0.0, DF378_36=0.0, DF488_37=42, DF376_37=29723.100000000002, DF377_37=0.0, DF378_37=0.0, DF488_38=43, DF376_38=4142.7, DF377_38=0.0, DF378_38=0.0, DF488_39=44, DF376_39=3086.8, DF377_39=0.0, DF378_39=0.0, DF488_40=45, DF376_40=-15423.6, DF377_40=0.0, DF378_40=0.0, DF488_41=48, DF376_41=174.0, DF377_41=0.0, DF378_41=0.0, DF488_42=50, DF376_42=-26273.300000000003, DF377_42=0.0, DF378_42=0.0, DF488_43=62, DF376_43=-3227.1000000000004, DF377_43=0.0, DF378_43=0.0)>",
            "<RTCM(1240, DF002=1240, DF458=338760, DF391=2, DF388=0, DF375=0, DF413=0, DF414=0, DF415=0, DF387=28, DF252_01=2, DF459_01=50, DF365_01=-667.6, DF366_01=435.6, DF367_01=44.400000000000006, DF368_01=-0.057, DF369_01=-0.012, DF370_01=-0.008, DF252_02=3, DF459_02=53, DF365_02=-882.3000000000001, DF366_02=-229.60000000000002, DF367_02=93.60000000000001, DF368_02=-0.004, DF369_02=-0.132, DF370_02=0.012, DF252_03=4, DF459_03=55, DF365_03=-969.7, DF366_03=91.2, DF367_03=137.20000000000002, DF368_03=-0.008, DF369_03=-0.048, DF370_03=0.012, DF252_04=5, DF459_04=55, DF365_04=-995.1, DF366_04=46.0, DF367_04=269.2, DF368_04=-0.022, DF369_04=-0.068, DF370_04=0.012, DF252_05=6, DF459_05=50, DF365_05=-981.2, DF366_05=-278.8, DF367_05=-220.4, DF368_05=0.019, DF369_05=-0.044, DF370_05=0.0, DF252_06=7, DF459_06=55, DF365_06=-866.5, DF366_06=71.60000000000001, DF367_06=-140.8, DF368_06=0.003, DF369_06=-0.04, DF370_06=-0.012, DF252_07=8, DF459_07=55, DF365_07=-719.8000000000001, DF366_07=-152.4, DF367_07=-14.8, DF368_07=-0.005, DF369_07=-0.084, DF370_07=0.028, DF252_08=9, DF459_08=48, DF365_08=-889.0, DF366_08=-235.20000000000002, DF367_08=-92.0, DF368_08=0.022, DF369_08=-0.07200000000000001, DF370_08=0.004, DF252_09=10, DF459_09=55, DF365_09=-1006.1, DF366_09=62.0, DF367_09=148.0, DF368_09=-0.032, DF369_09=-0.148, DF370_09=-0.028, DF252_10=11, DF459_10=48, DF365_10=-898.3000000000001, DF366_10=-71.2, DF367_10=456.8, DF368_10=-0.025, DF369_10=-0.076, DF370_10=-0.04, DF252_11=13, DF459_11=55, DF365_11=-763.1, DF366_11=-16.8, DF367_11=-136.4, DF368_11=0.062, DF369_11=-0.07200000000000001, DF370_11=0.02, DF252_12=14, DF459_12=50, DF365_12=-776.2, DF366_12=734.0, DF367_12=176.8, DF368_12=-0.139, DF369_12=-0.184, DF370_12=-0.052000000000000005, DF252_13=15, DF459_13=55, DF365_13=-491.6, DF366_13=-283.6, DF367_13=-79.2, DF368_13=0.013000000000000001, DF369_13=-0.1, DF370_13=-0.024, DF252_14=16, DF459_14=55, DF365_14=-762.7, DF366_14=-157.20000000000002, DF367_14=278.40000000000003, DF368_14=0.017, DF369_14=-0.08, DF370_14=0.032, DF252_15=18, DF459_15=55, DF365_15=-789.1, DF366_15=18.400000000000002, DF367_15=-207.60000000000002, DF368_15=-0.006, DF369_15=-0.032, DF370_15=-0.012, DF252_16=19, DF459_16=55, DF365_16=-750.7, DF366_16=-213.60000000000002, DF367_16=-78.80000000000001, DF368_16=0.004, DF369_16=-0.068, DF370_16=-0.004, DF252_17=21, DF459_17=55, DF365_17=-886.9000000000001, DF366_17=-346.40000000000003, DF367_17=-510.0, DF368_17=0.03, DF369_17=-0.08, DF370_17=-0.016, DF252_18=23, DF459_18=55, DF365_18=-891.8000000000001, DF366_18=2.0, DF367_18=-232.0, DF368_18=0.018000000000000002, DF369_18=-0.076, DF370_18=0.024, DF252_19=24, DF459_19=50, DF365_19=-877.3000000000001, DF366_19=283.2, DF367_19=96.0, DF368_19=-0.031, DF369_19=-0.016, DF370_19=0.004, DF252_20=25, DF459_20=51, DF365_20=-743.4000000000001, DF366_20=73.2, DF367_20=155.20000000000002, DF368_20=-0.009000000000000001, DF369_20=-0.056, DF370_20=0.02, DF252_21=26, DF459_21=55, DF365_21=-930.4000000000001, DF366_21=-70.0, DF367_21=-257.6, DF368_21=-0.013000000000000001, DF369_21=-0.084, DF370_21=0.016, DF252_22=27, DF459_22=49, DF365_22=-715.3000000000001, DF366_22=254.0, DF367_22=-20.400000000000002, DF368_22=0.001, DF369_22=-0.008, DF370_22=-0.012, DF252_23=29, DF459_23=50, DF365_23=-795.1, DF366_23=-234.4, DF367_23=444.0, DF368_23=0.029, DF369_23=-0.136, DF370_23=0.008, DF252_24=30, DF459_24=55, DF365_24=-585.9, DF366_24=100.4, DF367_24=-5.2, DF368_24=0.012, DF369_24=-0.06, DF370_24=0.012, DF252_25=31, DF459_25=52, DF365_25=-878.7, DF366_25=-436.0, DF367_25=-369.6, DF368_25=0.019, DF369_25=-0.032, DF370_25=0.08, DF252_26=33, DF459_26=49, DF365_26=-820.4000000000001, DF366_26=-638.0, DF367_26=-54.0, DF368_26=0.034, DF369_26=-0.168, DF370_26=0.032, DF252_27=34, DF459_27=49, DF365_27=-833.8000000000001, DF366_27=296.8, DF367_27=457.6, DF368_27=-0.022, DF369_27=0.004, DF370_27=0.056, DF252_28=36, DF459_28=55, DF365_28=-790.3000000000001, DF366_28=299.2, DF367_28=248.8, DF368_28=-0.007, DF369_28=-0.084, DF370_28=0.004)>",
            "<RTCM(1258, DF002=1258, DF465=338746, DF391=2, DF388=0, DF375=0, DF413=0, DF414=0, DF415=0, DF387=43, DF488_01=1, DF470_01=2528, DF471_01=230, DF365_01=-1244.4, DF366_01=810.0, DF367_01=11552.400000000001, DF368_01=-0.002, DF369_01=0.004, DF370_01=-0.112, DF488_02=2, DF470_02=2528, DF471_02=230, DF365_02=-1371.3000000000002, DF366_02=-345.6, DF367_02=11270.0, DF368_02=-0.009000000000000001, DF369_02=-0.044, DF370_02=0.14, DF488_03=3, DF470_03=2528, DF471_03=230, DF365_03=-1130.4, DF366_03=-527.2, DF367_03=3811.6000000000004, DF368_03=0.01, DF369_03=-0.02, DF370_03=-0.1, DF488_04=4, DF470_04=2528, DF471_04=230, DF365_04=-2551.7000000000003, DF366_04=2085.2000000000003, DF367_04=9417.2, DF368_04=-0.026000000000000002, DF369_04=-0.024, DF370_04=-0.112, DF488_05=5, DF470_05=2528, DF471_05=230, DF365_05=-1817.7, DF366_05=392.40000000000003, DF367_05=4182.400000000001, DF368_05=-0.04, DF369_05=-0.048, DF370_05=0.156, DF488_06=6, DF470_06=2528, DF471_06=230, DF365_06=-1555.6000000000001, DF366_06=-24.400000000000002, DF367_06=1133.2, DF368_06=-0.043000000000000003, DF369_06=0.052000000000000005, DF370_06=0.02, DF488_07=7, DF470_07=2528, DF471_07=230, DF365_07=-876.1, DF366_07=-423.6, DF367_07=-1589.6000000000001, DF368_07=0.024, DF369_07=-0.06, DF370_07=0.088, DF488_08=8, DF470_08=2528, DF471_08=230, DF365_08=-1475.2, DF366_08=472.8, DF367_08=1083.2, DF368_08=-0.026000000000000002, DF369_08=-0.112, DF370_08=-0.024, DF488_09=9, DF470_09=2528, DF471_09=230, DF365_09=-1318.0, DF366_09=-275.2, DF367_09=1163.2, DF368_09=-0.04, DF369_09=0.14, DF370_09=-0.044, DF488_10=10, DF470_10=2528, DF471_10=230, DF365_10=-1408.4, DF366_10=-217.20000000000002, DF367_10=-1287.2, DF368_10=-0.001, DF369_10=-0.07200000000000001, DF370_10=0.044, DF488_11=11, DF470_11=2528, DF471_11=230, DF365_11=-1832.8000000000002, DF366_11=-2481.6000000000004, DF367_11=1236.8000000000002, DF368_11=0.17400000000000002, DF369_11=-0.092, DF370_11=-0.048, DF488_12=12, DF470_12=2528, DF471_12=230, DF365_12=-1873.8000000000002, DF366_12=-1399.6000000000001, DF367_12=97.2, DF368_12=0.11800000000000001, DF369_12=-0.12, DF370_12=0.052000000000000005, DF488_13=13, DF470_13=2528, DF471_13=230, DF365_13=-1514.8000000000002, DF366_13=515.6, DF367_13=725.2, DF368_13=-0.026000000000000002, DF369_13=-0.1, DF370_13=0.04, DF488_14=14, DF470_14=2528, DF471_14=230, DF365_14=-1305.2, DF366_14=726.4000000000001, DF367_14=302.40000000000003, DF368_14=-0.057, DF369_14=-0.176, DF370_14=0.096, DF488_15=16, DF470_15=2528, DF471_15=230, DF365_15=-1536.9, DF366_15=351.20000000000005, DF367_15=1909.2, DF368_15=-0.022, DF369_15=0.048, DF370_15=-0.032, DF488_16=19, DF470_16=2528, DF471_16=230, DF365_16=-1261.6000000000001, DF366_16=-220.0, DF367_16=-67.2, DF368_16=0.024, DF369_16=-0.168, DF370_16=0.004, DF488_17=20, DF470_17=2528, DF471_17=230, DF365_17=-1298.2, DF366_17=-95.60000000000001, DF367_17=-222.0, DF368_17=0.002, DF369_17=-0.156, DF370_17=-0.036000000000000004, DF488_18=21, DF470_18=2528, DF471_18=230, DF365_18=-1299.7, DF366_18=-158.4, DF367_18=-42.800000000000004, DF368_18=0.02, DF369_18=-0.156, DF370_18=0.032, DF488_19=22, DF470_19=2528, DF471_19=230, DF365_19=-1286.8000000000002, DF366_19=-75.2, DF367_19=-46.0, DF368_19=0.01, DF369_19=-0.196, DF370_19=0.044, DF488_20=23, DF470_20=2528, DF471_20=230, DF365_20=-1238.5, DF366_20=-568.0, DF367_20=-26.0, DF368_20=0.049, DF369_20=-0.152, DF370_20=-0.004, DF488_21=24, DF470_21=2528, DF471_21=230, DF365_21=-1083.6000000000001, DF366_21=-248.8, DF367_21=-887.2, DF368_21=0.017, DF369_21=-0.052000000000000005, DF370_21=0.064, DF488_22=25, DF470_22=2528, DF471_22=230, DF365_22=-990.1, DF366_22=-2.0, DF367_22=-192.0, DF368_22=-0.011, DF369_22=-0.1, DF370_22=-0.036000000000000004, DF488_23=26, DF470_23=2528, DF471_23=230, DF365_23=-1028.6000000000001, DF366_23=49.2, DF367_23=-268.8, DF368_23=0.006, DF369_23=-0.124, DF370_23=-0.008, DF488_24=27, DF470_24=2528, DF471_24=230, DF365_24=-1107.8, DF366_24=-191.60000000000002, DF367_24=197.20000000000002, DF368_24=0.006, DF369_24=-0.02, DF370_24=-0.012, DF488_25=28, DF470_25=2528, DF471_25=230, DF365_25=-1028.6000000000001, DF366_25=-134.0, DF367_25=314.8, DF368_25=0.013000000000000001, DF369_25=-0.14400000000000002, DF370_25=-0.004, DF488_26=29, DF470_26=2528, DF471_26=230, DF365_26=-1063.6000000000001, DF366_26=-62.0, DF367_26=-251.60000000000002, DF368_26=0.018000000000000002, DF369_26=-0.124, DF370_26=0.008, DF488_27=30, DF470_27=2528, DF471_27=230, DF365_27=-908.6, DF366_27=-207.20000000000002, DF367_27=-36.0, DF368_27=0.011, DF369_27=-0.028, DF370_27=0.028, DF488_28=32, DF470_28=2528, DF471_28=230, DF365_28=-1335.8000000000002, DF366_28=7.6000000000000005, DF367_28=-295.2, DF368_28=0.003, DF369_28=-0.168, DF370_28=-0.028, DF488_29=33, DF470_29=2528, DF471_29=230, DF365_29=-1286.4, DF366_29=-225.60000000000002, DF367_29=-441.20000000000005, DF368_29=0.023, DF369_29=-0.112, DF370_29=0.016, DF488_30=34, DF470_30=2528, DF471_30=230, DF365_30=-1023.3000000000001, DF366_30=-256.0, DF367_30=39.6, DF368_30=0.023, DF369_30=-0.036000000000000004, DF370_30=-0.032, DF488_31=35, DF470_31=2528, DF471_31=230, DF365_31=-948.2, DF366_31=-160.4, DF367_31=-437.6, DF368_31=0.005, DF369_31=-0.116, DF370_31=-0.024, DF488_32=36, DF470_32=2528, DF471_32=230, DF365_32=-1105.8, DF366_32=296.0, DF367_32=-258.8, DF368_32=-0.06, DF369_32=-0.056, DF370_32=0.016, DF488_33=37, DF470_33=2528, DF471_33=230, DF365_33=-1299.6000000000001, DF366_33=-148.0, DF367_33=334.40000000000003, DF368_33=-0.005, DF369_33=-0.06, DF370_33=-0.028, DF488_34=38, DF470_34=2528, DF471_34=230, DF365_34=-1691.7, DF366_34=-373.20000000000005, DF367_34=62.400000000000006, DF368_34=0.026000000000000002, DF369_34=-0.112, DF370_34=-0.028, DF488_35=39, DF470_35=2528, DF471_35=230, DF365_35=-1656.3000000000002, DF366_35=-341.6, DF367_35=316.40000000000003, DF368_35=0.025, DF369_35=-0.088, DF370_35=-0.048, DF488_36=40, DF470_36=2528, DF471_36=230, DF365_36=-1520.2, DF366_36=566.0, DF367_36=-123.2, DF368_36=-0.019, DF369_36=-0.088, DF370_36=0.02, DF488_37=41, DF470_37=2528, DF471_37=230, DF365_37=-858.9000000000001, DF366_37=5.6000000000000005, DF367_37=-467.20000000000005, DF368_37=0.008, DF369_37=-0.14400000000000002, DF370_37=-0.036000000000000004, DF488_38=42, DF470_38=2528, DF471_38=230, DF365_38=-811.0, DF366_38=-268.8, DF367_38=-156.0, DF368_38=0.034, DF369_38=-0.032, DF370_38=0.008, DF488_39=43, DF470_39=2528, DF471_39=230, DF365_39=-1157.5, DF366_39=-23.6, DF367_39=202.0, DF368_39=-0.01, DF369_39=-0.088, DF370_39=-0.048, DF488_40=44, DF470_40=2528, DF471_40=230, DF365_40=-1096.4, DF366_40=99.2, DF367_40=-174.0, DF368_40=-0.001, DF369_40=-0.112, DF370_40=-0.024, DF488_41=45, DF470_41=2528, DF471_41=230, DF365_41=-1191.9, DF366_41=-153.60000000000002, DF367_41=-51.2, DF368_41=0.003, DF369_41=-0.156, DF370_41=0.024, DF488_42=48, DF470_42=2528, DF471_42=230, DF365_42=-1273.2, DF366_42=1111.6000000000001, DF367_42=-330.40000000000003, DF368_42=-0.106, DF369_42=-0.17200000000000001, DF370_42=-0.056, DF488_43=50, DF470_43=2528, DF471_43=230, DF365_43=-1177.9, DF366_43=16.400000000000002, DF367_43=-375.20000000000005, DF368_43=0.011, DF369_43=-0.088, DF370_43=-0.008)>",
            "<RTCM(1241, DF002=1241, DF458=338760, DF391=2, DF388=0, DF413=0, DF414=0, DF415=0, DF387=28, DF252_01=2, DF376_01=-1460.3000000000002, DF377_01=0.0, DF378_01=0.0, DF252_02=3, DF376_02=-1511.5, DF377_02=0.0, DF378_02=0.0, DF252_03=4, DF376_03=-1177.9, DF377_03=0.0, DF378_03=0.0, DF252_04=5, DF376_04=-1480.1000000000001, DF377_04=0.0, DF378_04=0.0, DF252_05=6, DF376_05=-1781.3000000000002, DF377_05=0.0, DF378_05=0.0, DF252_06=7, DF376_06=-1481.3000000000002, DF377_06=0.0, DF378_06=0.0, DF252_07=8, DF376_07=-1310.2, DF377_07=0.0, DF378_07=0.0, DF252_08=9, DF376_08=-1503.7, DF377_08=0.0, DF378_08=0.0, DF252_09=10, DF376_09=-1429.1000000000001, DF377_09=0.0, DF378_09=0.0, DF252_10=11, DF376_10=-1199.8, DF377_10=0.0, DF378_10=0.0, DF252_11=13, DF376_11=-1648.8000000000002, DF377_11=0.0, DF378_11=0.0, DF252_12=14, DF376_12=-1238.6000000000001, DF377_12=0.0, DF378_12=0.0, DF252_13=15, DF376_13=-1828.3000000000002, DF377_13=0.0, DF378_13=0.0, DF252_14=16, DF376_14=-1885.4, DF377_14=0.0, DF378_14=0.0, DF252_15=18, DF376_15=-1384.7, DF377_15=0.0, DF378_15=0.0, DF252_16=19, DF376_16=-1494.1000000000001, DF377_16=0.0, DF378_16=0.0, DF252_17=21, DF376_17=-1596.3000000000002, DF377_17=0.0, DF378_17=0.0, DF252_18=23, DF376_18=-1730.5, DF377_18=0.0, DF378_18=0.0, DF252_19=24, DF376_19=-999.1, DF377_19=0.0, DF378_19=0.0, DF252_20=25, DF376_20=-1702.1000000000001, DF377_20=0.0, DF378_20=0.0, DF252_21=26, DF376_21=-1132.9, DF377_21=0.0, DF378_21=0.0, DF252_22=27, DF376_22=-1500.2, DF377_22=0.0, DF378_22=0.0, DF252_23=29, DF376_23=-1574.6000000000001, DF377_23=0.0, DF378_23=0.0, DF252_24=30, DF376_24=-1590.6000000000001, DF377_24=0.0, DF378_24=0.0, DF252_25=31, DF376_25=-1513.2, DF377_25=0.0, DF378_25=0.0, DF252_26=33, DF376_26=-1276.0, DF377_26=0.0, DF378_26=0.0, DF252_27=34, DF376_27=-2076.5, DF377_27=0.0, DF378_27=0.0, DF252_28=36, DF376_28=-1828.3000000000002, DF377_28=0.0, DF378_28=0.0)>",
            "<RTCM(1242, DF002=1242, DF458=338875, DF391=2, DF388=0, DF413=1, DF414=3, DF415=1, DF387=25, DF252_01=2, DF379_01=1, DF382_01_01=2, DF383_01_01=0.0, DF252_02=3, DF379_02=1, DF382_02_01=2, DF383_02_01=0.0, DF252_03=4, DF379_03=1, DF382_03_01=2, DF383_03_01=0.0, DF252_04=5, DF379_04=1, DF382_04_01=2, DF383_04_01=0.0, DF252_05=6, DF379_05=1, DF382_05_01=2, DF383_05_01=0.0, DF252_06=7, DF379_06=1, DF382_06_01=2, DF383_06_01=0.0, DF252_07=8, DF379_07=1, DF382_07_01=2, DF383_07_01=0.0, DF252_08=9, DF379_08=1, DF382_08_01=2, DF383_08_01=0.0, DF252_09=11, DF379_09=1, DF382_09_01=2, DF383_09_01=0.0, DF252_10=12, DF379_10=1, DF382_10_01=2, DF383_10_01=0.0, DF252_11=13, DF379_11=1, DF382_11_01=2, DF383_11_01=0.0, DF252_12=14, DF379_12=1, DF382_12_01=2, DF383_12_01=0.0, DF252_13=15, DF379_13=1, DF382_13_01=2, DF383_13_01=0.0, DF252_14=18, DF379_14=1, DF382_14_01=2, DF383_14_01=0.0, DF252_15=19, DF379_15=1, DF382_15_01=2, DF383_15_01=0.0, DF252_16=21, DF379_16=1, DF382_16_01=2, DF383_16_01=0.0, DF252_17=24, DF379_17=1, DF382_17_01=2, DF383_17_01=0.0, DF252_18=25, DF379_18=1, DF382_18_01=2, DF383_18_01=0.0, DF252_19=26, DF379_19=1, DF382_19_01=2, DF383_19_01=0.0, DF252_20=27, DF379_20=1, DF382_20_01=2, DF383_20_01=0.0, DF252_21=29, DF379_21=1, DF382_21_01=2, DF383_21_01=0.0, DF252_22=30, DF379_22=1, DF382_22_01=2, DF383_22_01=0.0, DF252_23=31, DF379_23=1, DF382_23_01=2, DF383_23_01=0.0, DF252_24=33, DF379_24=1, DF382_24_01=2, DF383_24_01=0.0, DF252_25=36, DF379_25=1, DF382_25_01=2, DF383_25_01=0.0)>",
            "<RTCM(1260, DF002=1260, DF465=338861, DF391=2, DF388=0, DF413=1, DF414=3, DF415=1, DF387=33, DF488_01=7, DF379_01=0, DF488_02=8, DF379_02=0, DF488_03=9, DF379_03=0, DF488_04=10, DF379_04=0, DF488_05=11, DF379_05=0, DF488_06=12, DF379_06=0, DF488_07=13, DF379_07=0, DF488_08=14, DF379_08=0, DF488_09=16, DF379_09=0, DF488_10=19, DF379_10=0, DF488_11=20, DF379_11=0, DF488_12=21, DF379_12=0, DF488_13=22, DF379_13=0, DF488_14=23, DF379_14=0, DF488_15=24, DF379_15=0, DF488_16=25, DF379_16=0, DF488_17=26, DF379_17=0, DF488_18=27, DF379_18=0, DF488_19=28, DF379_19=0, DF488_20=29, DF379_20=0, DF488_21=30, DF379_21=0, DF488_22=32, DF379_22=0, DF488_23=33, DF379_23=0, DF488_24=34, DF379_24=0, DF488_25=35, DF379_25=0, DF488_26=36, DF379_26=0, DF488_27=38, DF379_27=0, DF488_28=39, DF379_28=0, DF488_29=41, DF379_29=0, DF488_30=42, DF379_30=0, DF488_31=43, DF379_31=0, DF488_32=44, DF379_32=0, DF488_33=45, DF379_33=0)>",
            "<RTCM(1243, DF002=1243, DF458=338880, DF391=2, DF388=0, DF375=0, DF413=1, DF414=3, DF415=1, DF387=25, DF252_01=2, DF459_01=55, DF365_01=-712.6, DF366_01=394.40000000000003, DF367_01=147.6, DF368_01=-0.045, DF369_01=-0.024, DF370_01=-0.028, DF376_01=-817.0, DF377_01=0.0, DF378_01=0.0, DF252_02=3, DF459_02=55, DF365_02=-869.8000000000001, DF366_02=-183.20000000000002, DF367_02=121.2, DF368_02=-0.02, DF369_02=-0.08, DF370_02=0.028, DF376_02=237.10000000000002, DF377_02=0.0, DF378_02=0.0, DF252_03=4, DF459_03=55, DF365_03=-969.9000000000001, DF366_03=250.4, DF367_03=184.0, DF368_03=0.0, DF369_03=-0.088, DF370_03=0.028, DF376_03=53.400000000000006, DF377_03=0.0, DF378_03=0.0, DF252_04=5, DF459_04=55, DF365_04=-1026.2, DF366_04=72.0, DF367_04=298.40000000000003, DF368_04=0.023, DF369_04=-0.08, DF370_04=0.0, DF376_04=244.0, DF377_04=0.0, DF378_04=0.0, DF252_05=6, DF459_05=55, DF365_05=-986.2, DF366_05=-103.2, DF367_05=8.0, DF368_05=-0.024, DF369_05=-0.024, DF370_05=-0.012, DF376_05=-512.8000000000001, DF377_05=0.0, DF378_05=0.0, DF252_06=7, DF459_06=55, DF365_06=-848.6, DF366_06=-27.6, DF367_06=-84.4, DF368_06=0.005, DF369_06=-0.048, DF370_06=0.052000000000000005, DF376_06=-91.5, DF377_06=0.0, DF378_06=0.0, DF252_07=8, DF459_07=55, DF365_07=-742.2, DF366_07=-224.4, DF367_07=-6.0, DF368_07=0.037, DF369_07=-0.064, DF370_07=0.016, DF376_07=-25.3, DF377_07=0.0, DF378_07=0.0, DF252_08=9, DF459_08=55, DF365_08=-894.3000000000001, DF366_08=-33.6, DF367_08=-14.8, DF368_08=-0.055, DF369_08=-0.088, DF370_08=0.004, DF376_08=-336.70000000000005, DF377_08=0.0, DF378_08=0.0, DF252_09=11, DF459_09=55, DF365_09=-887.4000000000001, DF366_09=-199.60000000000002, DF367_09=522.0, DF368_09=0.028, DF369_09=-0.096, DF370_09=0.008, DF376_09=-126.60000000000001, DF377_09=0.0, DF378_09=0.0, DF252_10=12, DF459_10=55, DF365_10=-1142.0, DF366_10=112.80000000000001, DF367_10=234.0, DF368_10=0.023, DF369_10=-0.084, DF370_10=-0.112, DF376_10=1.9000000000000001, DF377_10=0.0, DF378_10=0.0, DF252_11=13, DF459_11=55, DF365_11=-784.0, DF366_11=17.2, DF367_11=-178.4, DF368_11=0.045, DF369_11=-0.152, DF370_11=-0.02, DF376_11=-1412.6000000000001, DF377_11=0.0, DF378_11=0.0, DF252_12=14, DF459_12=50, DF365_12=-785.7, DF366_12=632.0, DF367_12=183.60000000000002, DF368_12=-0.186, DF369_12=-0.14400000000000002, DF370_12=-0.1, DF376_12=-130.70000000000002, DF377_12=0.0, DF378_12=0.0, DF252_13=15, DF459_13=55, DF365_13=-525.9, DF366_13=-316.8, DF367_13=-154.0, DF368_13=0.023, DF369_13=-0.092, DF370_13=-0.04, DF376_13=-1418.9, DF377_13=0.0, DF378_13=0.0, DF252_14=18, DF459_14=55, DF365_14=-851.0, DF366_14=82.80000000000001, DF367_14=-248.8, DF368_14=-0.001, DF369_14=-0.048, DF370_14=-0.028, DF376_14=-1394.4, DF377_14=0.0, DF378_14=0.0, DF252_15=19, DF459_15=55, DF365_15=-693.7, DF366_15=-127.60000000000001, DF367_15=-68.8, DF368_15=-0.041, DF369_15=-0.1, DF370_15=-0.02, DF376_15=-286.7, DF377_15=0.0, DF378_15=0.0, DF252_16=21, DF459_16=55, DF365_16=-928.7, DF366_16=-428.40000000000003, DF367_16=-541.2, DF368_16=0.058, DF369_16=-0.044, DF370_16=-0.008, DF376_16=-1104.6000000000001, DF377_16=0.0, DF378_16=0.0, DF252_17=24, DF459_17=55, DF365_17=-836.2, DF366_17=-322.0, DF367_17=10.0, DF368_17=-0.018000000000000002, DF369_17=-0.068, DF370_17=0.004, DF376_17=-92.0, DF377_17=0.0, DF378_17=0.0, DF252_18=25, DF459_18=51, DF365_18=-719.6, DF366_18=88.80000000000001, DF367_18=165.60000000000002, DF368_18=-0.032, DF369_18=-0.008, DF370_18=-0.036000000000000004, DF376_18=-835.6, DF377_18=0.0, DF378_18=0.0, DF252_19=26, DF459_19=55, DF365_19=-896.0, DF366_19=-21.6, DF367_19=-229.20000000000002, DF368_19=-0.018000000000000002, DF369_19=-0.168, DF370_19=0.004, DF376_19=-720.3000000000001, DF377_19=0.0, DF378_19=0.0, DF252_20=27, DF459_20=49, DF365_20=-775.4000000000001, DF366_20=174.8, DF367_20=-13.200000000000001, DF368_20=-0.036000000000000004, DF369_20=0.028, DF370_20=-0.032, DF376_20=-920.9000000000001, DF377_20=0.0, DF378_20=0.0, DF252_21=29, DF459_21=50, DF365_21=-745.7, DF366_21=-245.20000000000002, DF367_21=459.20000000000005, DF368_21=0.054, DF369_21=-0.12, DF370_21=0.024, DF376_21=-857.8000000000001, DF377_21=0.0, DF378_21=0.0, DF252_22=30, DF459_22=55, DF365_22=-600.0, DF366_22=-24.8, DF367_22=138.4, DF368_22=-0.027, DF369_22=-0.028, DF370_22=-0.012, DF376_22=-1350.3000000000002, DF377_22=0.0, DF378_22=0.0, DF252_23=31, DF459_23=52, DF365_23=-892.3000000000001, DF366_23=-468.0, DF367_23=-441.20000000000005, DF368_23=-0.008, DF369_23=-0.056, DF370_23=0.1, DF376_23=-825.7, DF377_23=0.0, DF378_23=0.0, DF252_24=33, DF459_24=49, DF365_24=-791.2, DF366_24=-593.6, DF367_24=-68.8, DF368_24=0.014, DF369_24=-0.184, DF370_24=0.076, DF376_24=-266.1, DF377_24=0.0, DF378_24=0.0, DF252_25=36, DF459_25=55, DF365_25=-768.2, DF366_25=354.0, DF367_25=208.0, DF368_25=-0.005, DF369_25=-0.088, DF370_25=-0.016, DF376_25=-773.4000000000001, DF377_25=0.0, DF378_25=0.0)>",
            "<RTCM(1261, DF002=1261, DF465=338866, DF391=2, DF388=0, DF375=0, DF413=1, DF414=3, DF415=1, DF387=33, DF488_01=7, DF470_01=0, DF471_01=230, DF365_01=-922.5, DF366_01=-485.6, DF367_01=-1332.0, DF368_01=0.12, DF369_01=-0.196, DF370_01=0.34400000000000003, DF376_01=-14734.800000000001, DF377_01=0.0, DF378_01=0.0, DF488_02=8, DF470_02=0, DF471_02=230, DF365_02=-1482.7, DF366_02=668.0, DF367_02=1237.2, DF368_02=-0.017, DF369_02=-0.10400000000000001, DF370_02=-0.012, DF376_02=-12225.6, DF377_02=0.0, DF378_02=0.0, DF488_03=9, DF470_03=0, DF471_03=230, DF365_03=-1356.3000000000002, DF366_03=-29.6, DF367_03=893.6, DF368_03=0.03, DF369_03=0.076, DF370_03=-0.164, DF376_03=-9385.300000000001, DF377_03=0.0, DF378_03=0.0, DF488_04=10, DF470_04=0, DF471_04=230, DF365_04=-1377.6000000000001, DF366_04=20.0, DF367_04=-1210.0, DF368_04=-0.023, DF369_04=-0.244, DF370_04=0.132, DF376_04=-8813.7, DF377_04=0.0, DF378_04=0.0, DF488_05=11, DF470_05=0, DF471_05=230, DF365_05=-1548.0, DF366_05=-2519.2000000000003, DF367_05=1046.4, DF368_05=0.201, DF369_05=-0.136, DF370_05=-0.032, DF376_05=-4845.5, DF377_05=0.0, DF378_05=0.0, DF488_06=12, DF470_06=0, DF471_06=230, DF365_06=-1782.3000000000002, DF366_06=-1406.0, DF367_06=-109.60000000000001, DF368_06=0.082, DF369_06=-0.148, DF370_06=0.12, DF376_06=-6642.6, DF377_06=0.0, DF378_06=0.0, DF488_07=13, DF470_07=0, DF471_07=230, DF365_07=-1758.3000000000002, DF366_07=532.4, DF367_07=1142.8, DF368_07=-0.007, DF369_07=-0.216, DF370_07=-0.064, DF376_07=7067.200000000001, DF377_07=0.0, DF378_07=0.0, DF488_08=14, DF470_08=0, DF471_08=230, DF365_08=-1207.1000000000001, DF366_08=844.0, DF367_08=320.40000000000003, DF368_08=-0.07, DF369_08=-0.14400000000000002, DF370_08=0.17200000000000001, DF376_08=-7483.0, DF377_08=0.0, DF378_08=0.0, DF488_09=16, DF470_09=0, DF471_09=230, DF365_09=-1514.0, DF366_09=267.2, DF367_09=1732.4, DF368_09=-0.038, DF369_09=-0.084, DF370_09=-0.232, DF376_09=192.60000000000002, DF377_09=0.0, DF378_09=0.0, DF488_10=19, DF470_10=0, DF471_10=230, DF365_10=-1200.4, DF366_10=-155.60000000000002, DF367_10=-172.0, DF368_10=0.033, DF369_10=-0.192, DF370_10=0.068, DF376_10=-7843.5, DF377_10=0.0, DF378_10=0.0, DF488_11=20, DF470_11=0, DF471_11=230, DF365_11=-1184.1000000000001, DF366_11=-116.4, DF367_11=-403.20000000000005, DF368_11=-0.044, DF369_11=-0.128, DF370_11=-0.064, DF376_11=-10971.900000000001, DF377_11=0.0, DF378_11=0.0, DF488_12=21, DF470_12=0, DF471_12=230, DF365_12=-1286.7, DF366_12=-186.0, DF367_12=-4.4, DF368_12=0.003, DF369_12=-0.18, DF370_12=0.04, DF376_12=-10734.6, DF377_12=0.0, DF378_12=0.0, DF488_13=22, DF470_13=0, DF471_13=230, DF365_13=-1222.2, DF366_13=-566.4, DF367_13=29.6, DF368_13=0.048, DF369_13=-0.196, DF370_13=0.068, DF376_13=-11808.7, DF377_13=0.0, DF378_13=0.0, DF488_14=23, DF470_14=0, DF471_14=230, DF365_14=-1281.6000000000001, DF366_14=-533.2, DF367_14=-45.2, DF368_14=0.053, DF369_14=-0.17200000000000001, DF370_14=-0.068, DF376_14=-20092.100000000002, DF377_14=0.0, DF378_14=0.0, DF488_15=24, DF470_15=0, DF471_15=230, DF365_15=-1243.2, DF366_15=40.400000000000006, DF367_15=-672.4000000000001, DF368_15=-0.085, DF369_15=-0.068, DF370_15=0.032, DF376_15=-5294.3, DF377_15=0.0, DF378_15=0.0, DF488_16=25, DF470_16=0, DF471_16=230, DF365_16=-1129.3, DF366_16=191.60000000000002, DF367_16=-21.6, DF368_16=-0.021, DF369_16=-0.14, DF370_16=-0.044, DF376_16=-644.3000000000001, DF377_16=0.0, DF378_16=0.0, DF488_17=26, DF470_17=0, DF471_17=230, DF365_17=-1105.4, DF366_17=-76.0, DF367_17=-268.40000000000003, DF368_17=0.046, DF369_17=-0.132, DF370_17=0.004, DF376_17=3109.9, DF377_17=0.0, DF378_17=0.0, DF488_18=27, DF470_18=0, DF471_18=230, DF365_18=-1161.5, DF366_18=163.60000000000002, DF367_18=280.0, DF368_18=-0.068, DF369_18=-0.028, DF370_18=0.028, DF376_18=1641.0, DF377_18=0.0, DF378_18=0.0, DF488_19=28, DF470_19=0, DF471_19=230, DF365_19=-1078.8, DF366_19=-223.60000000000002, DF367_19=439.6, DF368_19=-0.043000000000000003, DF369_19=-0.184, DF370_19=0.04, DF376_19=3413.9, DF377_19=0.0, DF378_19=0.0, DF488_20=29, DF470_20=0, DF471_20=230, DF365_20=-988.2, DF366_20=14.8, DF367_20=-68.8, DF368_20=0.019, DF369_20=-0.088, DF370_20=-0.012, DF376_20=-2693.7000000000003, DF377_20=0.0, DF378_20=0.0, DF488_21=30, DF470_21=0, DF471_21=230, DF365_21=-888.9000000000001, DF366_21=-230.0, DF367_21=88.80000000000001, DF368_21=-0.089, DF369_21=0.02, DF370_21=-0.092, DF376_21=6448.8, DF377_21=0.0, DF378_21=0.0, DF488_22=32, DF470_22=0, DF471_22=230, DF365_22=-1369.4, DF366_22=96.0, DF367_22=-339.6, DF368_22=-0.024, DF369_22=-0.092, DF370_22=-0.124, DF376_22=5787.5, DF377_22=0.0, DF378_22=0.0, DF488_23=33, DF470_23=0, DF471_23=230, DF365_23=-1220.5, DF366_23=-118.4, DF367_23=-498.8, DF368_23=-0.034, DF369_23=-0.076, DF370_23=0.024, DF376_23=38226.6, DF377_23=0.0, DF378_23=0.0, DF488_24=34, DF470_24=0, DF471_24=230, DF365_24=-1247.0, DF366_24=-132.0, DF367_24=128.0, DF368_24=0.041, DF369_24=-0.052000000000000005, DF370_24=-0.036000000000000004, DF376_24=5288.1, DF377_24=0.0, DF378_24=0.0, DF488_25=35, DF470_25=0, DF471_25=230, DF365_25=-987.5, DF366_25=-70.8, DF367_25=-378.40000000000003, DF368_25=0.032, DF369_25=-0.292, DF370_25=-0.076, DF376_25=2627.3, DF377_25=0.0, DF378_25=0.0, DF488_26=36, DF470_26=0, DF471_26=230, DF365_26=-918.0, DF366_26=258.0, DF367_26=-135.20000000000002, DF368_26=-0.028, DF369_26=-0.12, DF370_26=0.1, DF376_26=17313.8, DF377_26=0.0, DF378_26=0.0, DF488_27=38, DF470_27=0, DF471_27=230, DF365_27=-1737.3000000000002, DF366_27=-394.0, DF367_27=61.6, DF368_27=0.049, DF369_27=-0.08, DF370_27=0.004, DF376_27=-2858.3, DF377_27=0.0, DF378_27=0.0, DF488_28=39, DF470_28=0, DF471_28=230, DF365_28=-1662.5, DF366_28=-187.20000000000002, DF367_28=282.8, DF368_28=0.078, DF369_28=-0.064, DF370_28=-0.148, DF376_28=-7481.200000000001, DF377_28=0.0, DF378_28=0.0, DF488_29=41, DF470_29=0, DF471_29=230, DF365_29=-819.5, DF366_29=-13.600000000000001, DF367_29=-562.0, DF368_29=0.007, DF369_29=-0.128, DF370_29=-0.06, DF376_29=19480.600000000002, DF377_29=0.0, DF378_29=0.0, DF488_30=42, DF470_30=0, DF471_30=230, DF365_30=-798.1, DF366_30=-435.20000000000005, DF367_30=-105.60000000000001, DF368_30=0.061, DF369_30=-0.092, DF370_30=0.04, DF376_30=26858.300000000003, DF377_30=0.0, DF378_30=0.0, DF488_31=43, DF470_31=0, DF471_31=230, DF365_31=-1191.4, DF366_31=-51.2, DF367_31=172.0, DF368_31=0.009000000000000001, DF369_31=-0.096, DF370_31=-0.036000000000000004, DF376_31=2224.1, DF377_31=0.0, DF378_31=0.0, DF488_32=44, DF470_32=0, DF471_32=230, DF365_32=-1170.8, DF366_32=121.2, DF367_32=-198.4, DF368_32=-0.007, DF369_32=-0.148, DF370_32=0.008, DF376_32=278.6, DF377_32=0.0, DF378_32=0.0, DF488_33=45, DF470_33=0, DF471_33=230, DF365_33=-1201.7, DF366_33=173.20000000000002, DF367_33=-78.4, DF368_33=-0.036000000000000004, DF369_33=-0.136, DF370_33=0.06, DF376_33=-17662.7, DF377_33=0.0, DF378_33=0.0)>",
            "<RTCM(1243, DF002=1243, DF458=340045, DF391=2, DF388=0, DF375=0, DF413=0, DF414=0, DF415=0, DF387=24, DF252_01=2, DF459_01=57, DF365_01=90.80000000000001, DF366_01=243.60000000000002, DF367_01=300.40000000000003, DF368_01=0.004, DF369_01=-0.032, DF370_01=-0.024, DF376_01=124.10000000000001, DF377_01=0.0, DF378_01=0.0, DF252_02=3, DF459_02=57, DF365_02=11.5, DF366_02=-69.2, DF367_02=122.4, DF368_02=-0.033, DF369_02=-0.092, DF370_02=-0.032, DF376_02=-49.800000000000004, DF377_02=0.0, DF378_02=0.0, DF252_03=4, DF459_03=55, DF365_03=-69.5, DF366_03=136.4, DF367_03=15.600000000000001, DF368_03=-0.028, DF369_03=-0.084, DF370_03=0.028, DF376_03=285.7, DF377_03=0.0, DF378_03=0.0, DF252_04=5, DF459_04=55, DF365_04=-135.8, DF366_04=89.60000000000001, DF367_04=139.6, DF368_04=-0.063, DF369_04=-0.248, DF370_04=-0.06, DF376_04=22.1, DF377_04=0.0, DF378_04=0.0, DF252_05=6, DF459_05=57, DF365_05=-90.2, DF366_05=-198.4, DF367_05=-196.0, DF368_05=0.024, DF369_05=-0.056, DF370_05=0.116, DF376_05=-177.70000000000002, DF377_05=0.0, DF378_05=0.0, DF252_06=7, DF459_06=55, DF365_06=-68.3, DF366_06=51.2, DF367_06=-258.8, DF368_06=0.064, DF369_06=-0.18, DF370_06=-0.016, DF376_06=83.7, DF377_06=0.0, DF378_06=0.0, DF252_07=8, DF459_07=57, DF365_07=88.80000000000001, DF366_07=-10.4, DF367_07=-139.20000000000002, DF368_07=-0.024, DF369_07=-0.1, DF370_07=-0.024, DF376_07=142.20000000000002, DF377_07=0.0, DF378_07=0.0, DF252_08=9, DF459_08=57, DF365_08=-52.300000000000004, DF366_08=-105.2, DF367_08=-132.8, DF368_08=-0.024, DF369_08=-0.148, DF370_08=0.08, DF376_08=160.0, DF377_08=0.0, DF378_08=0.0, DF252_09=10, DF459_09=57, DF365_09=-208.3, DF366_09=-38.0, DF367_09=241.60000000000002, DF368_09=0.003, DF369_09=-0.156, DF370_09=-0.064, DF376_09=143.1, DF377_09=0.0, DF378_09=0.0, DF252_10=13, DF459_10=57, DF365_10=101.60000000000001, DF366_10=-36.0, DF367_10=-38.800000000000004, DF368_10=0.044, DF369_10=-0.028, DF370_10=0.052000000000000005, DF376_10=-196.20000000000002, DF377_10=0.0, DF378_10=0.0, DF252_11=15, DF459_11=55, DF365_11=217.20000000000002, DF366_11=-350.8, DF367_11=-22.400000000000002, DF368_11=-0.064, DF369_11=0.14, DF370_11=0.076, DF376_11=-358.6, DF377_11=0.0, DF378_11=0.0, DF252_12=16, DF459_12=55, DF365_12=37.2, DF366_12=-231.20000000000002, DF367_12=300.40000000000003, DF368_12=0.0, DF369_12=-0.116, DF370_12=-0.008, DF376_12=-331.6, DF377_12=0.0, DF378_12=0.0, DF252_13=21, DF459_13=55, DF365_13=-15.100000000000001, DF366_13=-255.60000000000002, DF367_13=-322.0, DF368_13=-0.015, DF369_13=-0.084, DF370_13=-0.056, DF376_13=-22.1, DF377_13=0.0, DF378_13=0.0, DF252_14=23, DF459_14=57, DF365_14=-73.2, DF366_14=125.60000000000001, DF367_14=-20.400000000000002, DF368_14=0.013000000000000001, DF369_14=-0.128, DF370_14=0.012, DF376_14=-216.60000000000002, DF377_14=0.0, DF378_14=0.0, DF252_15=24, DF459_15=57, DF365_15=-69.3, DF366_15=-102.80000000000001, DF367_15=81.2, DF368_15=-0.008, DF369_15=-0.16, DF370_15=0.02, DF376_15=510.20000000000005, DF377_15=0.0, DF378_15=0.0, DF252_16=25, DF459_16=51, DF365_16=31.1, DF366_16=49.6, DF367_16=202.4, DF368_16=0.011, DF369_16=-0.076, DF370_16=0.052000000000000005, DF376_16=-62.6, DF377_16=0.0, DF378_16=0.0, DF252_17=26, DF459_17=55, DF365_17=-131.70000000000002, DF366_17=86.4, DF367_17=-115.2, DF368_17=-0.09, DF369_17=-0.064, DF370_17=0.016, DF376_17=340.5, DF377_17=0.0, DF378_17=0.0, DF252_18=27, DF459_18=57, DF365_18=195.8, DF366_18=-80.4, DF367_18=54.0, DF368_18=0.052000000000000005, DF369_18=-0.08, DF370_18=0.02, DF376_18=-117.2, DF377_18=0.0, DF378_18=0.0, DF252_19=29, DF459_19=57, DF365_19=-29.1, DF366_19=-90.4, DF367_19=-60.800000000000004, DF368_19=-0.017, DF369_19=-0.044, DF370_19=-0.044, DF376_19=25.6, DF377_19=0.0, DF378_19=0.0, DF252_20=30, DF459_20=55, DF365_20=171.9, DF366_20=290.8, DF367_20=-22.8, DF368_20=-0.004, DF369_20=-0.044, DF370_20=-0.084, DF376_20=-63.2, DF377_20=0.0, DF378_20=0.0, DF252_21=31, DF459_21=52, DF365_21=-112.9, DF366_21=-332.0, DF367_21=-160.4, DF368_21=0.019, DF369_21=-0.092, DF370_21=0.1, DF376_21=51.2, DF377_21=0.0, DF378_21=0.0, DF252_22=33, DF459_22=57, DF365_22=-36.300000000000004, DF366_22=-441.20000000000005, DF367_22=-264.8, DF368_22=0.085, DF369_22=-0.076, DF370_22=0.07200000000000001, DF376_22=152.1, DF377_22=0.0, DF378_22=0.0, DF252_23=34, DF459_23=57, DF365_23=-146.0, DF366_23=308.40000000000003, DF367_23=298.0, DF368_23=-0.083, DF369_23=-0.012, DF370_23=0.004, DF376_23=-36.2, DF377_23=0.0, DF378_23=0.0, DF252_24=36, DF459_24=55, DF365_24=85.30000000000001, DF366_24=210.0, DF367_24=202.0, DF368_24=0.004, DF369_24=-0.192, DF370_24=0.036000000000000004, DF376_24=-315.90000000000003, DF377_24=0.0, DF378_24=0.0)>",
            "<RTCM(1261, DF002=1261, DF465=340031, DF391=2, DF388=1, DF375=0, DF413=0, DF414=0, DF415=0, DF387=28, DF488_01=6, DF470_01=0, DF471_01=230, DF365_01=1875.3000000000002, DF366_01=-254.0, DF367_01=612.0, DF368_01=0.01, DF369_01=-0.052000000000000005, DF370_01=-0.02, DF376_01=-13309.6, DF377_01=0.0, DF378_01=0.0, DF488_02=7, DF470_02=0, DF471_02=230, DF365_02=2502.7000000000003, DF366_02=-250.0, DF367_02=-2185.6, DF368_02=0.006, DF369_02=-0.044, DF370_02=0.176, DF376_02=-19239.9, DF377_02=0.0, DF378_02=0.0, DF488_03=9, DF470_03=0, DF471_03=230, DF365_03=2091.3, DF366_03=-816.4000000000001, DF367_03=541.2, DF368_03=0.017, DF369_03=0.036000000000000004, DF370_03=0.012, DF376_03=-12061.0, DF377_03=0.0, DF378_03=0.0, DF488_04=10, DF470_04=0, DF471_04=230, DF365_04=2181.4, DF366_04=-290.40000000000003, DF367_04=-1366.4, DF368_04=-0.016, DF369_04=-0.064, DF370_04=0.06, DF376_04=-11231.7, DF377_04=0.0, DF378_04=0.0, DF488_05=11, DF470_05=0, DF471_05=230, DF365_05=167.10000000000002, DF366_05=-1486.8000000000002, DF367_05=1042.0, DF368_05=0.128, DF369_05=-0.264, DF370_05=-0.024, DF376_05=-10400.6, DF377_05=0.0, DF378_05=0.0, DF488_06=12, DF470_06=0, DF471_06=230, DF365_06=198.10000000000002, DF366_06=-736.0, DF367_06=354.8, DF368_06=0.116, DF369_06=-0.216, DF370_06=-0.07200000000000001, DF376_06=-9911.900000000001, DF377_06=0.0, DF378_06=0.0, DF488_07=13, DF470_07=0, DF471_07=230, DF365_07=2008.4, DF366_07=821.6, DF367_07=1040.4, DF368_07=0.017, DF369_07=-0.124, DF370_07=-0.032, DF376_07=4437.2, DF377_07=0.0, DF378_07=0.0, DF488_08=16, DF470_08=0, DF471_08=230, DF365_08=2008.5, DF366_08=38.0, DF367_08=1249.2, DF368_08=-0.032, DF369_08=-0.004, DF370_08=-0.016, DF376_08=-2377.3, DF377_08=0.0, DF378_08=0.0, DF488_09=19, DF470_09=0, DF471_09=230, DF365_09=222.8, DF366_09=-358.40000000000003, DF367_09=70.0, DF368_09=0.03, DF369_09=-0.132, DF370_09=0.044, DF376_09=-8262.800000000001, DF377_09=0.0, DF378_09=0.0, DF488_10=20, DF470_10=0, DF471_10=230, DF365_10=235.60000000000002, DF366_10=-172.0, DF367_10=-118.0, DF368_10=-0.01, DF369_10=-0.184, DF370_10=-0.016, DF376_10=-11301.7, DF377_10=0.0, DF378_10=0.0, DF488_11=21, DF470_11=0, DF471_11=230, DF365_11=320.70000000000005, DF366_11=-382.40000000000003, DF367_11=339.20000000000005, DF368_11=0.042, DF369_11=-0.184, DF370_11=0.044, DF376_11=-10894.800000000001, DF377_11=0.0, DF378_11=0.0, DF488_12=22, DF470_12=0, DF471_12=230, DF365_12=270.8, DF366_12=-216.8, DF367_12=208.4, DF368_12=0.055, DF369_12=-0.16, DF370_12=0.052000000000000005, DF376_12=-12705.0, DF377_12=0.0, DF378_12=0.0, DF488_13=23, DF470_13=0, DF471_13=230, DF365_13=201.20000000000002, DF366_13=-404.40000000000003, DF367_13=-104.80000000000001, DF368_13=0.038, DF369_13=-0.22, DF370_13=0.032, DF376_13=-18624.100000000002, DF377_13=0.0, DF378_13=0.0, DF488_14=24, DF470_14=0, DF471_14=230, DF365_14=285.0, DF366_14=-128.4, DF367_14=-585.2, DF368_14=0.014, DF369_14=-0.12, DF370_14=0.076, DF376_14=-3211.1000000000004, DF377_14=0.0, DF378_14=0.0, DF488_15=25, DF470_15=0, DF471_15=230, DF365_15=-145.9, DF366_15=252.4, DF367_15=-19.200000000000003, DF368_15=-0.042, DF369_15=-0.096, DF370_15=-0.028, DF376_15=979.2, DF377_15=0.0, DF378_15=0.0, DF488_16=26, DF470_16=0, DF471_16=230, DF365_16=14.0, DF366_16=-34.4, DF367_16=-204.4, DF368_16=0.03, DF369_16=-0.108, DF370_16=-0.016, DF376_16=3807.5, DF377_16=0.0, DF378_16=0.0, DF488_17=27, DF470_17=0, DF471_17=230, DF365_17=-71.4, DF366_17=-184.8, DF367_17=181.20000000000002, DF368_17=-0.034, DF369_17=-0.112, DF370_17=0.048, DF376_17=3374.7000000000003, DF377_17=0.0, DF378_17=0.0, DF488_18=28, DF470_18=0, DF471_18=230, DF365_18=-39.1, DF366_18=-196.8, DF367_18=273.6, DF368_18=-0.094, DF369_18=-0.148, DF370_18=0.008, DF376_18=4851.3, DF377_18=0.0, DF378_18=0.0, DF488_19=29, DF470_19=0, DF471_19=230, DF365_19=-36.800000000000004, DF366_19=-87.60000000000001, DF367_19=-92.0, DF368_19=-0.052000000000000005, DF369_19=-0.168, DF370_19=0.056, DF376_19=628.8000000000001, DF377_19=0.0, DF378_19=0.0, DF488_20=30, DF470_20=0, DF471_20=230, DF365_20=77.5, DF366_20=-319.20000000000005, DF367_20=42.0, DF368_20=0.07200000000000001, DF369_20=-0.108, DF370_20=0.056, DF376_20=8612.9, DF377_20=0.0, DF378_20=0.0, DF488_21=32, DF470_21=0, DF471_21=230, DF365_21=260.40000000000003, DF366_21=-63.6, DF367_21=-264.8, DF368_21=0.022, DF369_21=-0.164, DF370_21=-0.004, DF376_21=6052.6, DF377_21=0.0, DF378_21=0.0, DF488_22=33, DF470_22=0, DF471_22=230, DF365_22=223.8, DF366_22=-210.8, DF367_22=-165.60000000000002, DF368_22=0.097, DF369_22=-0.14400000000000002, DF370_22=0.02, DF376_22=39926.3, DF377_22=0.0, DF378_22=0.0, DF488_23=34, DF470_23=0, DF471_23=230, DF365_23=-11.700000000000001, DF366_23=-130.0, DF367_23=6.0, DF368_23=0.038, DF369_23=-0.176, DF370_23=-0.008, DF376_23=6640.700000000001, DF377_23=0.0, DF378_23=0.0, DF488_24=35, DF470_24=0, DF471_24=230, DF365_24=57.0, DF366_24=-151.6, DF367_24=-356.40000000000003, DF368_24=-0.074, DF369_24=-0.124, DF370_24=-0.076, DF376_24=4276.7, DF377_24=0.0, DF378_24=0.0, DF488_25=36, DF470_25=0, DF471_25=230, DF365_25=346.3, DF366_25=84.0, DF367_25=76.0, DF368_25=-0.006, DF369_25=-0.132, DF370_25=0.032, DF376_25=19291.9, DF377_25=0.0, DF378_25=0.0, DF488_26=37, DF470_26=0, DF471_26=230, DF365_26=366.70000000000005, DF366_26=156.8, DF367_26=81.60000000000001, DF368_26=0.016, DF369_26=-0.224, DF370_26=0.04, DF376_26=14505.300000000001, DF377_26=0.0, DF378_26=0.0, DF488_27=38, DF470_27=0, DF471_27=230, DF365_27=62.300000000000004, DF366_27=133.6, DF367_27=77.60000000000001, DF368_27=-0.002, DF369_27=-0.124, DF370_27=0.008, DF376_27=-1097.2, DF377_27=0.0, DF378_27=0.0, DF488_28=39, DF470_28=0, DF471_28=230, DF365_28=469.40000000000003, DF366_28=-364.8, DF367_28=45.2, DF368_28=0.057, DF369_28=-0.168, DF370_28=-0.132, DF376_28=-5422.5, DF377_28=0.0, DF378_28=0.0)>",
            "<RTCM(1242, DF002=1242, DF458=340045, DF391=2, DF388=0, DF413=0, DF414=0, DF415=0, DF387=24, DF252_01=2, DF379_01=4, DF382_01_01=2, DF383_01_01=0.37, DF382_01_02=6, DF383_01_02=0.67, DF382_01_03=16, DF383_01_03=-0.43, DF382_01_04=9, DF383_01_04=0.86, DF252_02=3, DF379_02=4, DF382_02_01=2, DF383_02_01=-0.92, DF382_02_02=6, DF383_02_02=-1.6400000000000001, DF382_02_03=16, DF383_02_03=-0.91, DF382_02_04=9, DF383_02_04=-1.55, DF252_03=4, DF379_03=4, DF382_03_01=2, DF383_03_01=1.01, DF382_03_02=6, DF383_03_02=1.82, DF382_03_03=16, DF383_03_03=0.84, DF382_03_04=9, DF383_03_04=1.82, DF252_04=5, DF379_04=4, DF382_04_01=2, DF383_04_01=-0.99, DF382_04_02=6, DF383_04_02=-1.77, DF382_04_03=16, DF383_04_03=-0.14, DF382_04_04=9, DF383_04_04=-1.73, DF252_05=6, DF379_05=4, DF382_05_01=2, DF383_05_01=-0.92, DF382_05_02=6, DF383_05_02=-1.6500000000000001, DF382_05_03=16, DF383_05_03=-0.48, DF382_05_04=9, DF383_05_04=-1.83, DF252_06=7, DF379_06=4, DF382_06_01=2, DF383_06_01=-1.59, DF382_06_02=6, DF383_06_02=-2.84, DF382_06_03=16, DF383_06_03=-1.5, DF382_06_04=9, DF383_06_04=-2.77, DF252_07=8, DF379_07=4, DF382_07_01=2, DF383_07_01=1.11, DF382_07_02=6, DF383_07_02=1.99, DF382_07_03=16, DF383_07_03=1.57, DF382_07_04=9, DF383_07_04=2.04, DF252_08=9, DF379_08=4, DF382_08_01=2, DF383_08_01=-0.45, DF382_08_02=6, DF383_08_02=-0.8, DF382_08_03=16, DF383_08_03=-0.86, DF382_08_04=9, DF383_08_04=-0.74, DF252_09=10, DF379_09=4, DF382_09_01=2, DF383_09_01=0.03, DF382_09_02=6, DF383_09_02=0.05, DF382_09_03=16, DF383_09_03=-1.33, DF382_09_04=9, DF383_09_04=-0.06, DF252_10=13, DF379_10=4, DF382_10_01=2, DF383_10_01=-1.5, DF382_10_02=6, DF383_10_02=-2.68, DF382_10_03=16, DF383_10_03=-1.19, DF382_10_04=9, DF383_10_04=-2.58, DF252_11=15, DF379_11=4, DF382_11_01=2, DF383_11_01=-1.82, DF382_11_02=6, DF383_11_02=-3.2600000000000002, DF382_11_03=16, DF383_11_03=-2.2, DF382_11_04=9, DF383_11_04=-3.16, DF252_12=16, DF379_12=4, DF382_12_01=2, DF383_12_01=0.86, DF382_12_02=6, DF383_12_02=1.54, DF382_12_03=16, DF383_12_03=1.68, DF382_12_04=9, DF383_12_04=1.17, DF252_13=21, DF379_13=4, DF382_13_01=2, DF383_13_01=-0.56, DF382_13_02=6, DF383_13_02=-1.0, DF382_13_03=16, DF383_13_03=-1.87, DF382_13_04=9, DF383_13_04=-0.92, DF252_14=23, DF379_14=4, DF382_14_01=2, DF383_14_01=-0.55, DF382_14_02=6, DF383_14_02=-0.99, DF382_14_03=16, DF383_14_03=-0.8, DF382_14_04=9, DF383_14_04=-1.19, DF252_15=24, DF379_15=4, DF382_15_01=2, DF383_15_01=1.09, DF382_15_02=6, DF383_15_02=1.95, DF382_15_03=16, DF383_15_03=0.74, DF382_15_04=9, DF383_15_04=2.11, DF252_16=25, DF379_16=4, DF382_16_01=2, DF383_16_01=-1.06, DF382_16_02=6, DF383_16_02=-1.9000000000000001, DF382_16_03=16, DF383_16_03=-0.6, DF382_16_04=9, DF383_16_04=-1.8, DF252_17=26, DF379_17=4, DF382_17_01=2, DF383_17_01=1.32, DF382_17_02=6, DF383_17_02=2.37, DF382_17_03=16, DF383_17_03=0.54, DF382_17_04=9, DF383_17_04=2.43, DF252_18=27, DF379_18=4, DF382_18_01=2, DF383_18_01=-0.6, DF382_18_02=6, DF383_18_02=-1.07, DF382_18_03=16, DF383_18_03=-1.97, DF382_18_04=9, DF383_18_04=-1.0, DF252_19=29, DF379_19=4, DF382_19_01=2, DF383_19_01=0.75, DF382_19_02=6, DF383_19_02=1.34, DF382_19_03=16, DF383_19_03=-0.33, DF382_19_04=9, DF383_19_04=1.19, DF252_20=30, DF379_20=4, DF382_20_01=2, DF383_20_01=-0.55, DF382_20_02=6, DF383_20_02=-0.99, DF382_20_03=16, DF383_20_03=-0.34, DF382_20_04=9, DF383_20_04=-0.9, DF252_21=31, DF379_21=4, DF382_21_01=2, DF383_21_01=-1.34, DF382_21_02=6, DF383_21_02=-2.41, DF382_21_03=16, DF383_21_03=-0.4, DF382_21_04=9, DF383_21_04=-2.38, DF252_22=33, DF379_22=4, DF382_22_01=2, DF383_22_01=1.04, DF382_22_02=6, DF383_22_02=1.87, DF382_22_03=16, DF383_22_03=0.4, DF382_22_04=9, DF383_22_04=1.93, DF252_23=34, DF379_23=4, DF382_23_01=2, DF383_23_01=-1.2, DF382_23_02=6, DF383_23_02=-2.15, DF382_23_03=16, DF383_23_03=-1.95, DF382_23_04=9, DF383_23_04=-2.22, DF252_24=36, DF379_24=4, DF382_24_01=2, DF383_24_01=-1.6600000000000001, DF382_24_02=6, DF383_24_02=-2.98, DF382_24_03=16, DF383_24_03=-2.0100000000000002, DF382_24_04=9, DF383_24_04=-2.95)>",
            "<RTCM(1260, DF002=1260, DF465=340031, DF391=2, DF388=0, DF413=0, DF414=0, DF415=0, DF387=34, DF488_01=6, DF379_01=3, DF467_01_01=0, DF383_01_01=7.390000000000001, DF467_01_02=3, DF383_01_02=11.19, DF467_01_03=6, DF383_01_03=4.2700000000000005, DF488_02=7, DF379_02=3, DF467_02_01=0, DF383_02_01=11.5, DF467_02_02=3, DF383_02_02=17.42, DF467_02_03=6, DF383_02_03=9.55, DF488_03=9, DF379_03=3, DF467_03_01=0, DF383_03_01=6.61, DF467_03_02=3, DF383_03_02=10.01, DF467_03_03=6, DF383_03_03=1.23, DF488_04=10, DF379_04=3, DF467_04_01=0, DF383_04_01=6.58, DF467_04_02=3, DF383_04_02=9.96, DF467_04_03=6, DF383_04_03=1.46, DF488_05=11, DF379_05=3, DF467_05_01=0, DF383_05_01=3.92, DF467_05_02=3, DF383_05_02=5.94, DF467_05_03=6, DF383_05_03=-1.6300000000000001, DF488_06=12, DF379_06=3, DF467_06_01=0, DF383_06_01=3.71, DF467_06_02=3, DF383_06_02=5.61, DF467_06_03=6, DF383_06_03=-1.53, DF488_07=13, DF379_07=3, DF467_07_01=0, DF383_07_01=-3.22, DF467_07_02=3, DF383_07_02=-4.87, DF467_07_03=6, DF383_07_03=-13.19, DF488_08=16, DF379_08=3, DF467_08_01=0, DF383_08_01=0.72, DF467_08_02=3, DF383_08_02=1.09, DF467_08_03=6, DF383_08_03=-7.86, DF488_09=19, DF379_09=4, DF467_09_01=10, DF383_09_01=7.28, DF467_09_02=0, DF383_09_02=4.62, DF467_09_03=13, DF383_09_03=8.38, DF467_09_04=3, DF383_09_04=7.0, DF488_10=20, DF379_10=4, DF467_10_01=10, DF383_10_01=8.63, DF467_10_02=0, DF383_10_02=6.09, DF467_10_03=13, DF383_10_03=10.89, DF467_10_04=3, DF383_10_04=9.22, DF488_11=21, DF379_11=4, DF467_11_01=10, DF383_11_01=8.01, DF467_11_02=0, DF383_11_02=6.1000000000000005, DF467_11_03=13, DF383_11_03=10.85, DF467_11_04=3, DF383_11_04=9.23, DF488_12=22, DF379_12=4, DF467_12_01=10, DF383_12_01=9.33, DF467_12_02=0, DF383_12_02=7.44, DF467_12_03=13, DF383_12_03=14.15, DF467_12_04=3, DF383_12_04=11.27, DF488_13=23, DF379_13=4, DF467_13_01=10, DF383_13_01=14.48, DF467_13_02=0, DF383_13_02=12.0, DF467_13_03=13, DF383_13_03=20.57, DF467_13_04=3, DF383_13_04=18.18, DF488_14=24, DF379_14=4, DF467_14_01=10, DF383_14_01=3.73, DF467_14_02=0, DF383_14_02=1.99, DF467_14_03=13, DF383_14_03=8.16, DF467_14_04=3, DF383_14_04=3.0100000000000002, DF488_15=25, DF379_15=4, DF467_15_01=10, DF383_15_01=1.75, DF467_15_02=0, DF383_15_02=-0.63, DF467_15_03=13, DF383_15_03=3.5, DF467_15_04=3, DF383_15_04=-0.9500000000000001, DF488_16=26, DF379_16=4, DF467_16_01=10, DF383_16_01=-1.33, DF467_16_02=0, DF383_16_02=-3.0100000000000002, DF467_16_03=13, DF383_16_03=-0.22, DF467_16_04=3, DF383_16_04=-4.57, DF488_17=27, DF379_17=4, DF467_17_01=10, DF383_17_01=-0.09, DF467_17_02=0, DF383_17_02=-2.33, DF467_17_03=13, DF383_17_03=0.6, DF467_17_04=3, DF383_17_04=-3.5300000000000002, DF488_18=28, DF379_18=4, DF467_18_01=10, DF383_18_01=-0.19, DF467_18_02=0, DF383_18_02=-2.75, DF467_18_03=13, DF383_18_03=0.09, DF467_18_04=3, DF383_18_04=-4.17, DF488_19=29, DF379_19=4, DF467_19_01=10, DF383_19_01=1.53, DF467_19_02=0, DF383_19_02=-0.32, DF467_19_03=13, DF383_19_03=3.84, DF467_19_04=3, DF383_19_04=-0.49, DF488_20=30, DF379_20=4, DF467_20_01=10, DF383_20_01=-3.96, DF467_20_02=0, DF383_20_02=-6.140000000000001, DF467_20_03=13, DF383_20_03=-4.8100000000000005, DF467_20_04=3, DF383_20_04=-9.3, DF488_21=32, DF379_21=4, DF467_21_01=10, DF383_21_01=-3.56, DF467_21_02=0, DF383_21_02=-5.01, DF467_21_03=13, DF383_21_03=-4.73, DF467_21_04=3, DF383_21_04=-7.59, DF488_22=33, DF379_22=4, DF467_22_01=10, DF383_22_01=-25.1, DF467_22_02=0, DF383_22_02=-27.03, DF467_22_03=13, DF383_22_03=-49.35, DF467_22_04=3, DF383_22_04=-40.94, DF488_23=34, DF379_23=4, DF467_23_01=10, DF383_23_01=-2.24, DF467_23_02=0, DF383_23_02=-4.17, DF467_23_03=13, DF383_23_03=-2.13, DF467_23_04=3, DF383_23_04=-6.32, DF488_24=35, DF379_24=4, DF467_24_01=10, DF383_24_01=-0.02, DF467_24_02=0, DF383_24_02=-2.16, DF467_24_03=13, DF383_24_03=0.43, DF467_24_04=3, DF383_24_04=-3.2600000000000002, DF488_25=36, DF379_25=4, DF467_25_01=10, DF383_25_01=-11.58, DF467_25_02=0, DF383_25_02=-13.290000000000001, DF467_25_03=13, DF383_25_03=-17.39, DF467_25_04=3, DF383_25_04=-20.13, DF488_26=37, DF379_26=4, DF467_26_01=10, DF383_26_01=-8.950000000000001, DF467_26_02=0, DF383_26_02=-10.43, DF467_26_03=13, DF383_26_03=-11.6, DF467_26_04=3, DF383_26_04=-15.790000000000001, DF488_27=38, DF379_27=4, DF467_27_01=10, DF383_27_01=2.62, DF467_27_02=0, DF383_27_02=0.91, DF467_27_03=13, DF383_27_03=4.78, DF467_27_04=3, DF383_27_04=1.3800000000000001, DF488_28=39, DF379_28=4, DF467_28_01=10, DF383_28_01=5.0200000000000005, DF467_28_02=0, DF383_28_02=3.39, DF467_28_03=13, DF383_28_03=8.9, DF467_28_04=3, DF383_28_04=5.13, DF488_29=40, DF379_29=4, DF467_29_01=10, DF383_29_01=2.15, DF467_29_02=0, DF383_29_02=-0.11, DF467_29_03=13, DF383_29_03=3.86, DF467_29_04=3, DF383_29_04=-0.16, DF488_30=41, DF379_30=4, DF467_30_01=10, DF383_30_01=-13.36, DF467_30_02=0, DF383_30_02=-14.870000000000001, DF467_30_03=13, DF383_30_03=-22.55, DF467_30_04=3, DF383_30_04=-22.52, DF488_31=42, DF379_31=4, DF467_31_01=10, DF383_31_01=-18.13, DF467_31_02=0, DF383_31_02=-19.490000000000002, DF467_31_03=13, DF383_31_03=-27.41, DF467_31_04=3, DF383_31_04=-29.52, DF488_32=43, DF379_32=4, DF467_32_01=10, DF383_32_01=-0.25, DF467_32_02=0, DF383_32_02=-2.6, DF467_32_03=13, DF383_32_03=-1.01, DF467_32_04=3, DF383_32_04=-3.94, DF488_33=44, DF379_33=4, DF467_33_01=10, DF383_33_01=1.25, DF467_33_02=0, DF383_33_02=-1.68, DF467_33_03=13, DF383_33_03=0.42, DF467_33_04=3, DF383_33_04=-2.5500000000000003, DF488_34=45, DF379_34=4, DF467_34_01=10, DF383_34_01=11.620000000000001, DF467_34_02=0, DF383_34_02=10.83, DF467_34_03=13, DF383_34_03=20.740000000000002, DF467_34_04=3, DF383_34_04=16.4)>",
            "<RTCM(1264, DF002=1264, DF385=340200, DF391=6, DF388=0, DF413=0, DF414=0, DF415=0, DF478=0.0, DF472=0)>",
        ]
        dirname = os.path.dirname(__file__)
        with open(os.path.join(dirname, "pygpsdata-1240-1264.log"), "rb") as stream:
            i = 0
            raw = 0
            rtr = RTCMReader(stream, labelmsm=True)
            for raw, parsed in rtr:
                if raw is not None:
                    # print(f'"{parsed}",')
                    self.assertEqual(f"{parsed}", EXPECTED_RESULTS[i])
                    i += 1
        self.assertEqual(i, len(EXPECTED_RESULTS))

    def test13001302(
        self,
    ):  # test 1300 / 1302 messages using log from NTRIP caster euref-ip.net, mountpoint EUREF01
        # TODO 1302 message payloads from this data log appears to be truncated (59 bytes) NB: these messages also bomb the BNC 2.13.1 NTRIP client
        EXPECTED_RESULTS = [
            "<RTCM(1300, DF002=1300, DF562=31, DF563=ETRF2000, DF564=115.0)>",
            # "<RTCM(1302, DF002=1302, DF565=31, DF566=ETRF2000(2015), DF567=1, DF149=7, DF568=6, DF569_01=9, DF570_01=*, DF569_02=13, DF570_02=<, DF569_03=23, DF570_03=$, DF569_04=21, DF570_04=4, DF569_05=30, DF570_05=\, DF569_06=18, DF570_06=T)>",
            "<RTCM(1300, DF002=1300, DF562=31, DF563=ETRF2000, DF564=115.0)>",
            # "<RTCM(1302, DF002=1302, DF565=31, DF566=ETRF2000(2015), DF567=1, DF149=7, DF568=6, DF569_01=9, DF570_01=*, DF569_02=13, DF570_02=<, DF569_03=23, DF570_03=$, DF569_04=21, DF570_04=4, DF569_05=30, DF570_05=\, DF569_06=18, DF570_06=T)>",
            "<RTCM(1300, DF002=1300, DF562=31, DF563=ETRF2000, DF564=115.0)>",
            # "<RTCM(1302, DF002=1302, DF565=31, DF566=ETRF2000(2015), DF567=1, DF149=7, DF568=6, DF569_01=9, DF570_01=*, DF569_02=13, DF570_02=<, DF569_03=23, DF570_03=$, DF569_04=21, DF570_04=4, DF569_05=30, DF570_05=\, DF569_06=18, DF570_06=T)>",
            "<RTCM(1300, DF002=1300, DF562=31, DF563=ETRF2000, DF564=115.0)>",
            # "<RTCM(1302, DF002=1302, DF565=31, DF566=ETRF2000(2015), DF567=1, DF149=7, DF568=6, DF569_01=9, DF570_01=*, DF569_02=13, DF570_02=<, DF569_03=23, DF570_03=$, DF569_04=21, DF570_04=4, DF569_05=30, DF570_05=\, DF569_06=18, DF570_06=T)>",
            "<RTCM(1300, DF002=1300, DF562=31, DF563=ETRF2000, DF564=115.0)>",
            # "<RTCM(1302, DF002=1302, DF565=31, DF566=ETRF2000(2015), DF567=1, DF149=7, DF568=6, DF569_01=9, DF570_01=*, DF569_02=13, DF570_02=<, DF569_03=23, DF570_03=$, DF569_04=21, DF570_04=4, DF569_05=30, DF570_05=\, DF569_06=18, DF570_06=T)>",
            "<RTCM(1300, DF002=1300, DF562=31, DF563=ETRF2000, DF564=115.0)>",
            # "<RTCM(1302, DF002=1302, DF565=31, DF566=ETRF2000(2015), DF567=1, DF149=7, DF568=6, DF569_01=9, DF570_01=*, DF569_02=13, DF570_02=<, DF569_03=23, DF570_03=$, DF569_04=21, DF570_04=4, DF569_05=30, DF570_05=\, DF569_06=18, DF570_06=T)>",
        ]
        dirname = os.path.dirname(__file__)
        with open(
            os.path.join(dirname, "pygpsdata-NTRIP-1300-1302.log"), "rb"
        ) as stream:
            i = 0
            raw = 0
            rtr = RTCMReader(stream, labelmsm=True, quitonerror=ERR_LOG)
            for raw, parsed in rtr:
                if raw is not None:
                    if parsed.identity in ("1300"):  # , "1302"):
                        # print(f'"{parsed}",')
                        self.assertEqual(f"{parsed}", EXPECTED_RESULTS[i])
                        i += 1
            self.assertEqual(i, 6)

    def testSerialize(self):  # test serialize()
        payload = self._raw1005[3:-3]
        msg1 = RTCMReader.parse(self._raw1005)
        msg2 = RTCMMessage(payload=payload)
        res = msg1.serialize()
        self.assertEqual(res, self._raw1005)
        res1 = msg2.serialize()
        self.assertEqual(res, self._raw1005)

    def testsetattr(self):  # test immutability
        EXPECTED_ERROR = (
            "Object is immutable. Updates to DF002 not permitted after initialisation."
        )
        with self.assertRaisesRegex(RTCMMessageError, EXPECTED_ERROR):
            msg = RTCMReader.parse(self._raw1005)
            msg.DF002 = 9999

    def testrepr(self):  # test repr, check eval recreates original object
        EXPECTED_RESULT = "RTCMMessage(payload=b'>\\xd0\\x00\\x03\\x8aX\\xd9I<\\x87/4\\x10\\x9d\\x07\\xd6\\xafH ')"
        msg1 = RTCMReader.parse(self._raw1005)
        self.assertEqual(repr(msg1), EXPECTED_RESULT)
        msg2 = eval(repr(msg1))
        self.assertEqual(str(msg1), str(msg2))

    def testpayload(self):  # test payload getter
        msg = RTCMReader.parse(self._raw1005)
        payload = self._raw1005[3:-3]
        self.assertEqual(msg.payload, payload)

    def testnopayload(self):  # test null payload
        EXPECTED_ERROR = "Payload must be specified"
        with self.assertRaisesRegex(RTCMMessageError, EXPECTED_ERROR):
            msg = RTCMMessage()

    def testgroups(self):  # test message with repeating group (1007)
        EXPECTED_RESULT = "<RTCM(1007, DF002=1007, DF003=1234, DF029=3, DF030_01=A, DF030_02=B, DF030_03=C, DF031=234)>"
        msg1 = RTCMMessage(payload=self._payload1007)
        msg2 = RTCMReader.parse(self._raw1007)
        self.assertEqual(str(msg1), EXPECTED_RESULT)
        self.assertEqual(str(msg2), EXPECTED_RESULT)

    def testnestedgroups(self):  # test message with nested repeating group (1059, 1065)
        EXPECTED_RESULT = "<RTCM(1065, DF002=1065, DF386=12345, DF391=3, DF388=0, DF413=1, DF414=1, DF415=1, DF387=2, DF384_01=23, DF379_01=2, DF381_01_01=4, DF383_01_01=0.07, DF381_01_02=2, DF383_01_02=0.09, DF384_02=26, DF379_02=1, DF381_02_01=3, DF383_02_01=0.05)>"
        msg = RTCMReader.parse(self._raw1065)
        self.assertEqual(str(msg), EXPECTED_RESULT)

    def testbadCRC(
        self,
    ):  # test mixed stream of NMEA, UBX & RTCM messages with invalid RTCM CRC
        EXPECTED_ERROR = "RTCM3 message invalid - failed CRC: (.*)"
        with open(
            os.path.join(DIRNAME, "pygpsdata-MIXED-RTCM3BADCRC.log"), "rb"
        ) as stream:
            i = 0
            raw = 0
            rtr = RTCMReader(stream, quitonerror=rtt.ERR_RAISE)
            with self.assertRaisesRegex(RTCMParseError, EXPECTED_ERROR):
                for raw, parsed in rtr:
                    if raw is not None:
                        print(parsed)
                        i += 1

    def testbadCRC2(
        self,
    ):  # test mixed stream of NMEA, UBX & RTCM messages with invalid RTCM CRC
        EXPECTED_ERROR = "I ignored the following error: RTCM3 message invalid - failed CRC: b'Z\\xd7\\xf7'"
        dirname = os.path.dirname(__file__)
        with open(
            os.path.join(dirname, "pygpsdata-MIXED-RTCM3BADCRC.log"), "rb"
        ) as stream:
            self.catchio()
            rtr = RTCMReader(
                stream,
                quitonerror=rtt.ERR_LOG,
                errorhandler=lambda e: print(f"I ignored the following error: {e}"),
            )
            for raw, parsed in rtr:
                if raw is not None:
                    pass
            stream.close()
            output = self.restoreio()
            self.assertEqual(output, EXPECTED_ERROR)

    def testbadCRC3(
        self,
    ):  # test mixed stream of NMEA, UBX & RTCM messages with invalid RTCM CRC
        EXPECTED_ERROR = "RTCM3 message invalid - failed CRC: b'Z\\xd7\\xf7'"
        with self.assertLogs(level=ERROR) as log:
            with open(
                os.path.join(DIRNAME, "pygpsdata-MIXED-RTCM3BADCRC.log"), "rb"
            ) as stream:
                rtr = RTCMReader(
                    stream,
                    quitonerror=rtt.ERR_LOG,
                    errorhandler=None,
                )
                for raw, parsed in rtr:
                    if raw is not None:
                        pass
            self.assertEqual(
                [
                    "ERROR:pyrtcm.rtcmreader:RTCM3 message invalid - failed CRC: b'Z\\xd7\\xf7'"
                ],
                log.output,
            )

    def testNOPARSE(self):
        EXPECTED_RESULTS = [
            b"\xd3\x00\x13>\xd0\x00\x03\x84\x1a\x86\x92\xbf\xb4KK\xf4\xfa\xb7\xdc7b\x8a3\x84y",
            b"\xd3\x00\x15>\xe0\x00\x03\x84\x1a\x86\x92\xbf\xb4KK\xf4\xfa\xb7\xdc7b\x8a\x01W\x1b\xa9\xd6",
            b"\xd3\x00\x19>\xf0\x00\x14SEPCHOKE_B3E6   SPKE\x00\x07nf",
            b"\xd3\x00\x1e?\x00\x00\x14SEPCHOKE_B3E6   SPKE\x00\x045856\xffh\x94",
            b'\xd3\x00H?\x10\x00\x86\x85\x03\x14\x00$4\x07\xbaAt\x0b\xfa\xc2 3\x11=\xa0+\xfb\x04\xb8q\x80\xc0\xbd\xdf\xf9\x06\xb8\xd7U\x80u\xbb\xf8\xe6 \xc5\x18=#\xb7\xfa\xe5\\\x85\xd8\x80\\\x83\xf9@m\x8d%\x01\xb17\xf9"\xbdz\xc2\x81h\x03\xf8\x9f\xc6\xe6',
        ]
        RTCMMESSAGES = [
            b"\xd3\x00\x13>\xd0\x00\x03\x84\x1a\x86\x92\xbf\xb4KK\xf4\xfa\xb7\xdc7b\x8a3\x84y",
            b"\xd3\x00\x15>\xe0\x00\x03\x84\x1a\x86\x92\xbf\xb4KK\xf4\xfa\xb7\xdc7b\x8a\x01W\x1b\xa9\xd6",
            b"\xd3\x00\x19>\xf0\x00\x14SEPCHOKE_B3E6   SPKE\x00\x07nf",
            b"\xd3\x00\x1e?\x00\x00\x14SEPCHOKE_B3E6   SPKE\x00\x045856\xffh\x94",
            b'\xd3\x00H?\x10\x00\x86\x85\x03\x14\x00$4\x07\xbaAt\x0b\xfa\xc2 3\x11=\xa0+\xfb\x04\xb8q\x80\xc0\xbd\xdf\xf9\x06\xb8\xd7U\x80u\xbb\xf8\xe6 \xc5\x18=#\xb7\xfa\xe5\\\x85\xd8\x80\\\x83\xf9@m\x8d%\x01\xb17\xf9"\xbdz\xc2\x81h\x03\xf8\x9f\xc6\xe6',
        ]
        msgb = b""
        for msg in RTCMMESSAGES:
            msgb += msg
        RTCMBYTES = msgb
        stream = BytesIO(RTCMBYTES)
        rtr = RTCMReader(stream, parsed=False)
        i = 0
        for raw, parsed in rtr:
            # print(f"{raw},")
            self.assertEqual(EXPECTED_RESULTS[i], raw)
            i += 1
        self.assertEqual(i, 5)

    def testUNKNOWNHDR(self):
        EXPECTED_RESULTS = [
            "<RTCM(1005, DF002=1005, DF003=0, DF021=0, DF022=1, DF023=1, DF024=1, DF141=0, DF025=1762489.6191, DF142=1, DF001_1=0, DF026=-5027633.8438, DF364=2, DF027=-3496008.8438000004)>",
            "<RTCM(1006, DF002=1006, DF003=0, DF021=0, DF022=1, DF023=1, DF024=1, DF141=0, DF025=1762489.6191, DF142=1, DF001_1=0, DF026=-5027633.8438, DF364=2, DF027=-3496008.8438000004, DF028=0.034300000000000004)>",
            "<RTCM(1007, DF002=1007, DF003=0, DF029=20, DF030_01=S, DF030_02=E, DF030_03=P, DF030_04=C, DF030_05=H, DF030_06=O, DF030_07=K, DF030_08=E, DF030_09=_, DF030_10=B, DF030_11=3, DF030_12=E, DF030_13=6, DF030_14= , DF030_15= , DF030_16= , DF030_17=S, DF030_18=P, DF030_19=K, DF030_20=E, DF031=0)>",
            "<RTCM(1008, DF002=1008, DF003=0, DF029=20, DF030_01=S, DF030_02=E, DF030_03=P, DF030_04=C, DF030_05=H, DF030_06=O, DF030_07=K, DF030_08=E, DF030_09=_, DF030_10=B, DF030_11=3, DF030_12=E, DF030_13=6, DF030_14= , DF030_15= , DF030_16= , DF030_17=S, DF030_18=P, DF030_19=K, DF030_20=E, DF031=0, DF032=4, DF033_01=5, DF033_02=8, DF033_03=5, DF033_04=6)>",
            "<RTCM(1009, DF002=1009, DF003=0, DF034=70527000, DF005=1, DF035=8, DF036=0, DF037=0, DF038_01=1, DF039_01=0, DF040_01=8, DF041_01=272788.02, DF042_01=11.905, DF043_01=127, DF038_02=22, DF039_02=0, DF040_02=4, DF041_02=168818.0, DF042_02=-19.451, DF043_02=127, DF038_03=24, DF039_03=0, DF040_03=9, DF041_03=295925.82, DF042_03=6.0755, DF043_03=127, DF038_04=8, DF039_04=0, DF040_04=13, DF041_04=298011.32, DF042_04=3.767, DF043_04=127, DF038_05=7, DF039_05=0, DF040_05=12, DF041_05=171808.64, DF042_05=-23.433500000000002, DF043_05=127, DF038_06=23, DF039_06=0, DF040_06=10, DF041_06=485086.12, DF042_06=2.96, DF043_06=127, DF038_07=10, DF039_07=0, DF040_07=0, DF041_07=574364.56, DF042_07=13.8625, DF043_07=127, DF038_08=9, DF039_08=0, DF040_08=5, DF041_08=322329.8, DF042_08=11.52, DF043_08=127)>",
        ]
        RTCMMESSAGES = [
            b"\xd3\x00\x13>\xd0\x00\x03\x84\x1a\x86\x92\xbf\xb4KK\xf4\xfa\xb7\xdc7b\x8a3\x84y",
            b"\xd3\x00\x15>\xe0\x00\x03\x84\x1a\x86\x92\xbf\xb4KK\xf4\xfa\xb7\xdc7b\x8a\x01W\x1b\xa9\xd6",
            b"\xd3\x00\x19>\xf0\x00\x14SEPCHOKE_B3E6   SPKE\x00\x07nf",
            b"\xd3\x00\x1e?\x00\x00\x14SEPCHOKE_B3E6   SPKE\x00\x045856\xffh\x94",
            b"\xd3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3",  # not RTCM3, discarded
            b'\xd3\x00H?\x10\x00\x86\x85\x03\x14\x00$4\x07\xbaAt\x0b\xfa\xc2 3\x11=\xa0+\xfb\x04\xb8q\x80\xc0\xbd\xdf\xf9\x06\xb8\xd7U\x80u\xbb\xf8\xe6 \xc5\x18=#\xb7\xfa\xe5\\\x85\xd8\x80\\\x83\xf9@m\x8d%\x01\xb17\xf9"\xbdz\xc2\x81h\x03\xf8\x9f\xc6\xe6',
        ]
        msgb = b""
        for msg in RTCMMESSAGES:
            msgb += msg
        RTCMBYTES = msgb
        stream = BytesIO(RTCMBYTES)
        rtr = RTCMReader(stream)
        i = 0
        for _, parsed in rtr:
            # print(f'"{parsed}",')
            self.assertEqual(EXPECTED_RESULTS[i], str(parsed))
            i += 1
        self.assertEqual(i, 5)

    def testPREMATUREEOF(self):
        RTCMMESSAGES = [
            b"\xd3\x00\x13>\xd0\x00\x03\x84\x1a\x86\x92\xbf\xb4KK\xf4\xfa\xb7\xdc7b\x8a3\x84y",
            b"\xd3\x00\x15>\xe0\x00\x03\x84\x1a\x86\x92\xbf\xb4KK\xf4\xfa\xb7\xdc7b\x8a\x01W\x1b\xa9\xd6",
            b"\xd3\x00\x19>\xf0\x00\x14SEPCHOKE_B3E6   SPKE\x00\x07nf",
            b"\xd3\x00\x1e?\x00\x00\x14SEPCHOKE_B3E6   SPKE\x00\x045856\xffh\x94",
            b"\xd3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3\xa3",  # not RTCM3, discarded
            b"\xd3\x00H?\x10\x00\x86\x85\x03\x14\x00$4\x07\xbaAt\x0b\xfa\xca",
        ]
        msgb = b""
        for msg in RTCMMESSAGES:
            msgb += msg
        RTCMBYTES = msgb
        stream = BytesIO(RTCMBYTES)
        rtr = RTCMReader(stream, quitonerror=ERR_RAISE)
        with self.assertRaisesRegex(
            RTCMStreamError,
            "Serial stream terminated unexpectedly. 72 bytes requested, 17 bytes returned.",
        ):
            for _, parsed in rtr:
                pass
                # print(f'"{parsed}",')

    def testBADHDR_IGNORE(self):  # invalid header in data with quitonerror = 0
        i = 0
        with open(os.path.join(DIRNAME, "pygpsdata-BADHDR.log"), "rb") as stream:
            ubr = RTCMReader(stream, quitonerror=ERR_IGNORE)
            for raw, parsed in ubr:
                i += 1

    def testZeroLengthPayload(self):
        DATA = (
            b"\xd3\x00\x15>\xe0\x00\x03\x84\x1a\x86\x92\xbf\xb4KK\xf4\xfa\xb7\xdc7b\x8a\x01W\x1b\xa9\xd6"
            b"\xd3\x00\x00>\xf0\x00\x14SEPCHOKE_B3E6   SPKE\x00\x07nf"
            b"\xd3\x00\x1e?\x00\x00\x14SEPCHOKE_B3E6   SPKE\x00\x045856\xffh\x94"
        )

        with BytesIO(DATA) as stream:
            with self.assertRaisesRegex(
                RTCMStreamError, "Invalid payload size 0 bytes"
            ):
                ubr = RTCMReader(stream, quitonerror=ERR_RAISE)
                for raw, parsed in ubr:
                    print(parsed)

    def testZeroLengthPayloadIgnore(self):
        DATA = (
            b"\xd3\x00\x15>\xe0\x00\x03\x84\x1a\x86\x92\xbf\xb4KK\xf4\xfa\xb7\xdc7b\x8a\x01W\x1b\xa9\xd6"
            b"\xd3\x00\x00>\xf0\x00\x14SEPCHOKE_B3E6   SPKE\x00\x07nf"
            b"\xd3\x00\x1e?\x00\x00\x14SEPCHOKE_B3E6   SPKE\x00\x045856\xffh\x94"
        )
        EXPECTED_RESULT = [
            "<RTCM(1006, DF002=1006, DF003=0, DF021=0, DF022=1, DF023=1, DF024=1, DF141=0, DF025=1762489.6191, DF142=1, DF001_1=0, DF026=-5027633.8438, DF364=2, DF027=-3496008.8438000004, DF028=0.034300000000000004)>",
            # "Invalid payload size 0 bytes",
            "<RTCM(1008, DF002=1008, DF003=0, DF029=20, DF030_01=S, DF030_02=E, DF030_03=P, DF030_04=C, DF030_05=H, DF030_06=O, DF030_07=K, DF030_08=E, DF030_09=_, DF030_10=B, DF030_11=3, DF030_12=E, DF030_13=6, DF030_14= , DF030_15= , DF030_16= , DF030_17=S, DF030_18=P, DF030_19=K, DF030_20=E, DF031=0, DF032=4, DF033_01=5, DF033_02=8, DF033_03=5, DF033_04=6)>",
        ]
        with BytesIO(DATA) as stream:
            ubr = RTCMReader(stream, quitonerror=ERR_LOG)
            i = 0
            for raw, parsed in ubr:
                print(f'"{parsed}",')
                self.assertEqual(str(parsed), EXPECTED_RESULT[i])
                i += 1
            self.assertEqual(i, 2)


if __name__ == "__main__":
    # import sys;sys.argv = ['', 'Test.testName']
    unittest.main()
