/*  $Id: main.cpp,v 1.2 2024/04/08 03:44:10 sarrazip Exp $
    main.cpp - main() function

    quadrupleback - A video game where intruders must be circled.
    Copyright (C) 2012 Pierre Sarrazin <http://sarrazip.com/>

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License
    as published by the Free Software Foundation; either version 2
    of the License, or (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
    02110-1301, USA.
*/

#include "QuadruplebackEngine.h"

#include <iostream>

#include <stdlib.h>
#include <errno.h>
#include <time.h>

#ifdef __GNUG__
#include <getopt.h>
#endif


#ifdef __GNUG__
static struct option knownOptions[] =
{
    { "help",          no_argument,       NULL, 'h' },
    { "version",       no_argument,       NULL, 'v' },
    { "ms-per-frame",  required_argument, NULL, 'm' },
    { "no-sound",      no_argument,       NULL, 'n' },
    { "rendering",     required_argument, NULL, 'r' },

    { NULL, 0, NULL, 0 }  // marks the end
};


static void
displayVersionNo()
{
    cout << PACKAGE << " " << VERSION << "\n";
}


static void
displayHelp()
{
    cout << "\n";

    displayVersionNo();

    cout <<
"\n"
"Copyright (C) 2003-2010 Pierre Sarrazin <http://sarrazip.com/>\n"
"This program is free software; you may redistribute it under the terms of\n"
"the GNU General Public License.  This program has absolutely no warranty.\n"
    ;

    cout <<
"\n"
"Known options:\n"
"--help             Display this help page and exit\n"
"--version          Display this program's version number and exit\n"
"--ms-per-frame=N   N milliseconds per animation frame [default=55]\n"
"                   Min. 1, max. 1000.  50 ms means 20 frames per second\n"
"--no-sound         Disable sound effects [enabled by default]\n"
"--rendering=X      Specify the type of rendering.\n"
"                   Replace X with 'a' for accelerated, 's' for software.\n"
"\n"
    ;
}
#endif  /* __GNUG__ */


int main(int argc, char *argv[])
{
    /*  Default game parameters:
    */
    int millisecondsPerFrame = 55;
    bool useSound = true;
    bool useAcceleratedRendering = false;
            // Accelerated rendering may draw stray pixels on the snake.
            // Observed with SDL2 2.0.10.

    #ifdef __GNUG__

    /*  Interpret the command-line options:
    */
    int c;
    do
    {
        c = getopt_long(argc, argv, "", knownOptions, NULL);

        switch (c)
        {
        case EOF:
            break;  // nothing to do

        case 'm':
            {
                errno = 0;
                long n = strtol(optarg, NULL, 10);
                if (errno == ERANGE || n < 1 || n > 1000)
                {
                    cout << "Invalid number of ms per frame.\n";
                    displayHelp();
                    return EXIT_FAILURE;
                }

                millisecondsPerFrame = int(n);
            }
            break;

        case 'n':
            useSound = false;
            break;

        case 'r':
            if (!strcmp(optarg, "a"))
                useAcceleratedRendering = true;
            else if (!strcmp(optarg, "s"))
                useAcceleratedRendering = false;
            else
            {
                cout << "Invalid argument for --rendering.\n";
                displayHelp();
                return EXIT_FAILURE;
            }
            break;

        case 'v':
            displayVersionNo();
            return EXIT_SUCCESS;

        case 'h':
            displayHelp();
            return EXIT_SUCCESS;

        default:
            displayHelp();
            return EXIT_FAILURE;
        }
    } while (c != EOF && c != '?');

    #endif  /* __GNUG__ */


    /*  Initialize the random number generator:
    */
    const char *s = getenv("SEED");
    unsigned int seed = (s != NULL ? atol(s) : time(NULL));
    #ifndef NDEBUG
    cerr << "seed = " << seed << "\n";
    cerr << "ms-per-frame = " << millisecondsPerFrame << "\n";
    #endif
    srand(seed);


    try
    {
        QuadruplebackEngine theEngine("Quadrupleback", useSound, useAcceleratedRendering);
        theEngine.run(millisecondsPerFrame);
    }
    catch (const string &s)
    {
        cerr << PACKAGE << ": engine init error: " << s << "\n";
        return EXIT_FAILURE;
    }
    catch (int e)
    {
        cerr << PACKAGE << ": init error # " << e << "\n";
        return EXIT_FAILURE;
    }

    return EXIT_SUCCESS;
}
