//! Integration tests for saorsa-browser
//!
//! These tests verify the overall functionality of the browser application.

use sb::{App, Config};
use std::fs;
use std::path::PathBuf;
use tempfile::TempDir;

#[test]
fn test_app_initialization() {
    let temp_dir = TempDir::new().unwrap();
    let app = App::new(temp_dir.path().to_path_buf()).unwrap();

    // Test basic integration - app should initialize successfully
    assert!(!app.root.to_string_lossy().is_empty());
    assert_eq!(app.root, temp_dir.path().to_path_buf());
}

#[test]
fn test_app_integration_basic_operations() {
    let temp_dir = TempDir::new().unwrap();

    // Create some test files
    fs::write(temp_dir.path().join("test.md"), "# Test").unwrap();
    fs::write(temp_dir.path().join("README.md"), "# README").unwrap();

    let app = App::new(temp_dir.path().to_path_buf()).unwrap();

    // Should complete without panicking
    assert!(!app.left_tree.is_empty() || app.left_tree.is_empty()); // Either way is fine
    assert!(!app.right_tree.is_empty() || app.right_tree.is_empty()); // Either way is fine
}

#[test]
fn test_file_operations() {
    let temp_dir = TempDir::new().unwrap();
    let test_file = temp_dir.path().join("test.md");

    fs::write(&test_file, "# Test Content").unwrap();

    assert!(test_file.exists());

    let content = fs::read_to_string(&test_file).unwrap();
    assert_eq!(content, "# Test Content");
}

#[test]
fn test_directory_structure() {
    let temp_dir = TempDir::new().unwrap();
    let base_path = temp_dir.path();

    // Create nested directory structure
    fs::create_dir(base_path.join("docs")).unwrap();
    fs::create_dir(base_path.join("src")).unwrap();
    fs::create_dir(base_path.join("src/tests")).unwrap();

    fs::write(base_path.join("README.md"), "# Project").unwrap();
    fs::write(base_path.join("docs/guide.md"), "# Guide").unwrap();
    fs::write(base_path.join("src/main.rs"), "fn main() {}").unwrap();
    fs::write(base_path.join("src/tests/test.rs"), "#[test]").unwrap();

    // Verify structure
    assert!(base_path.join("docs").is_dir());
    assert!(base_path.join("src").is_dir());
    assert!(base_path.join("src/tests").is_dir());
    assert!(base_path.join("README.md").is_file());
    assert!(base_path.join("docs/guide.md").is_file());
    assert!(base_path.join("src/main.rs").is_file());
    assert!(base_path.join("src/tests/test.rs").is_file());
}

#[test]
fn test_security_path_validation() {
    let temp_dir = TempDir::new().unwrap();
    let base_path = temp_dir.path();

    // Create the file first
    fs::write(base_path.join("safe_file.md"), "# Safe content").unwrap();

    // Test safe paths - use relative path
    let safe_relative_path = std::path::Path::new("safe_file.md");
    assert!(sb::validate_path(safe_relative_path, base_path).is_ok());

    // Test path traversal attempts - relative path with traversal
    let unsafe_relative_path = std::path::Path::new("../../../etc/passwd");
    assert!(sb::validate_path(unsafe_relative_path, base_path).is_err());
}

#[test]
fn test_file_size_limits() {
    let temp_dir = TempDir::new().unwrap();

    // Create test files with different sizes
    let small_file = temp_dir.path().join("small.txt");
    let medium_file = temp_dir.path().join("medium.txt");

    fs::write(&small_file, "small content").unwrap();
    fs::write(&medium_file, "x".repeat(1024)).unwrap(); // 1KB file

    // Test file size checking with actual files
    assert!(sb::check_file_size(&small_file).is_ok()); // Small file - OK
    assert!(sb::check_file_size(&medium_file).is_ok()); // 1KB file - OK

    // Test non-existent file
    let non_existent = temp_dir.path().join("does_not_exist.txt");
    assert!(sb::check_file_size(&non_existent).is_err()); // Should error
}

#[test]
fn test_markdown_file_detection() {
    let md_files = vec![
        PathBuf::from("test.md"),
        PathBuf::from("README.md"),
        PathBuf::from("CHANGELOG.MD"),
        PathBuf::from("notes.markdown"),
    ];

    for file in md_files {
        let ext = file.extension().and_then(|s| s.to_str()).unwrap_or("");
        assert!(ext.eq_ignore_ascii_case("md") || ext.eq_ignore_ascii_case("markdown"));
    }

    let non_md_files = vec![
        PathBuf::from("test.txt"),
        PathBuf::from("main.rs"),
        PathBuf::from("style.css"),
    ];

    for file in non_md_files {
        let ext = file.extension().and_then(|s| s.to_str()).unwrap_or("");
        assert!(!ext.eq_ignore_ascii_case("md"));
    }
}

#[test]
fn test_code_file_detection() {
    let code_files = vec![
        (PathBuf::from("main.rs"), "rust"),
        (PathBuf::from("app.js"), "javascript"),
        (PathBuf::from("test.ts"), "typescript"),
        (PathBuf::from("script.py"), "python"),
    ];

    for (file, _expected_lang) in code_files {
        let ext = file.extension().and_then(|s| s.to_str());
        assert!(ext.is_some());

        // Verify known code extensions
        let code_extensions = ["rs", "js", "ts", "py", "go", "java", "cpp", "c", "h"];
        assert!(code_extensions.contains(&ext.unwrap()));
    }
}

#[test]
fn test_config_defaults() {
    let _config = Config::default();

    // Test config creation works (specific method tests depend on actual implementation)
    // Config creation succeeded if we get here
}

#[test]
fn test_concurrent_file_access() {
    use std::sync::Arc;
    use std::thread;

    let temp_dir = Arc::new(TempDir::new().unwrap());
    let file_path = temp_dir.path().join("concurrent.txt");

    // Create initial file
    fs::write(&file_path, "Initial content").unwrap();

    let mut handles = vec![];

    // Spawn multiple threads to read the file
    for i in 0..5 {
        let path = file_path.clone();
        let handle = thread::spawn(move || {
            let content = fs::read_to_string(&path).unwrap();
            assert!(!content.is_empty());
            format!("Thread {} read: {}", i, content.len())
        });
        handles.push(handle);
    }

    // Wait for all threads
    for handle in handles {
        let result = handle.join().unwrap();
        assert!(result.contains("read:"));
    }
}

#[test]
fn test_app_focus_management() {
    let temp_dir = TempDir::new().unwrap();
    let app = App::new(temp_dir.path().to_path_buf()).unwrap();

    // Test that focus field is accessible
    let _focus = &app.focus;

    // App should start with some focus
    // We can't predict the exact value, so just test it compiles
}

#[test]
fn test_app_help_toggle() {
    let temp_dir = TempDir::new().unwrap();
    let mut app = App::new(temp_dir.path().to_path_buf()).unwrap();

    // Test help toggle functionality
    let initial_help_state = app.show_help;
    app.toggle_help();
    assert_ne!(app.show_help, initial_help_state);

    // Toggle back
    app.toggle_help();
    assert_eq!(app.show_help, initial_help_state);
}

#[test]
fn test_app_pane_toggle() {
    let temp_dir = TempDir::new().unwrap();
    let mut app = App::new(temp_dir.path().to_path_buf()).unwrap();

    // Test left pane toggle functionality
    let initial_pane_state = app.show_left_pane;
    app.toggle_left_pane();
    assert_ne!(app.show_left_pane, initial_pane_state);

    // Toggle back
    app.toggle_left_pane();
    assert_eq!(app.show_left_pane, initial_pane_state);
}
