/*
 * nut_bool.h - Network UPS Tools boolean type definitions
 *              which should ensure a "nut_bool_t" name with
 *              lower-case values "true" and "false"
 *
 * Inspired by earlier efforts and numerous definitions in NUT codebase.
 * Copyright (C) 2024 Jim Klimov <jimklimov+nut@gmail.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#ifndef NUT_BOOL_H_SEEN
#define NUT_BOOL_H_SEEN 1

#ifdef __cplusplus
/* *INDENT-OFF* */
extern "C" {
/* *INDENT-ON* */
#endif

/* "config.h" is generated by autotools and lacks a header guard, so
 * we use an unambiguously named macro we know we must have, as one.
 * It must be the first header: be sure to know all about system config.
 */
#ifndef NUT_NETVERSION
# include "config.h"
#endif

/* See also https://en.cppreference.com/w/cpp/header/cstdbool for more
 * info about what should be available where per standard approach. */
#ifdef __cplusplus
# if defined HAVE_CSTDBOOL_H || defined HAVE_CSTDBOOL
#  include <cstdbool>
# else
#  ifdef HAVE_STDBOOL_H
#   include <stdbool.h>
#  endif
# endif
#else
/* plain C */
#  ifdef HAVE_STDBOOL_H
#   include <stdbool.h>
#  endif
#endif

/* Is the goal achieved by the system headers or compiler itself,
 * so we can just alias to existing type and its values? */
#if defined __bool_true_false_are_defined && __bool_true_false_are_defined
	typedef bool nut_bool_t;
#elif defined FOUND__BOOL_TYPE && defined HAVE__BOOL_VALUE_LOWERCASE
	typedef FOUND__BOOL_TYPE nut_bool_t;
#elif defined FOUND_BOOL_TYPE && defined HAVE_BOOL_VALUE_LOWERCASE
	typedef FOUND_BOOL_TYPE nut_bool_t;
#elif defined FOUND_BOOLEAN_TYPE && defined HAVE_BOOLEAN_VALUE_LOWERCASE
	typedef FOUND_BOOLEAN_TYPE nut_bool_t;
#elif defined FOUND_BOOL_T_TYPE && defined HAVE_BOOL_T_VALUE_LOWERCASE
	typedef FOUND_BOOL_T_TYPE nut_bool_t;
#else
	/* Need a new type; can we use an enum with lower-case values? */
# if (defined true && defined false) || defined HAVE__BOOL_VALUE_LOWERCASE || defined HAVE_BOOL_VALUE_LOWERCASE || defined HAVE_BOOLEAN_VALUE_LOWERCASE || defined HAVE_BOOL_T_VALUE_LOWERCASE
	/* Lower-case true/false are known */
#  if defined FOUND__BOOL_TYPE
	/* Got a C99 built-in mandated by the standard */
	typedef FOUND__BOOL_TYPE nut_bool_t;
#  else
	typedef int nut_bool_t;
#  endif
# elif defined FOUND__BOOL_VALUE_TRUE && defined FOUND__BOOL_VALUE_FALSE
	typedef enum nut_bool_enum { false = FOUND__BOOL_VALUE_FALSE, true = FOUND__BOOL_VALUE_TRUE } nut_bool_t;
# elif defined FOUND_BOOL_VALUE_TRUE && defined FOUND_BOOL_VALUE_FALSE
	typedef enum nut_bool_enum { false = FOUND_BOOL_VALUE_FALSE, true = FOUND_BOOL_VALUE_TRUE } nut_bool_t;
# elif defined FOUND_BOOLEAN_VALUE_TRUE && defined FOUND_BOOLEAN_VALUE_FALSE
	typedef enum nut_bool_enum { false = FOUND_BOOLEAN_VALUE_FALSE, true = FOUND_BOOLEAN_VALUE_TRUE } nut_bool_t;
# elif defined FOUND_BOOL_T_VALUE_TRUE && defined FOUND_BOOL_T_VALUE_FALSE
	typedef enum nut_bool_enum { false = FOUND_BOOL_T_VALUE_FALSE, true = FOUND_BOOL_T_VALUE_TRUE } nut_bool_t;
# else
	typedef enum nut_bool_enum { false = 0, true = 1 } nut_bool_t;
# endif
#endif

#ifdef __cplusplus
/* *INDENT-OFF* */
}
/* *INDENT-ON* */
#endif

#endif	/* NUT_BOOL_H_SEEN */
