use crate::nntp_client::NntpClient;
use rek2_nntp::post::Article as PostArticle;
use std::fs;
use std::io::Write;
use std::process::Command;
use tempfile::NamedTempFile;

/// Attempts to load a signature from `$SIGNATURE_FILE` or `~/.signature`.
fn load_signature() -> Option<String> {
    // Determine signature path: env SIGNATURE_FILE or ~/.signature
    let sig_path = std::env::var("SIGNATURE_FILE").unwrap_or_else(|_| {
        std::env::var("HOME")
            .map(|h| format!("{}/.signature", h))
            .unwrap_or_else(|_| String::new())
    });
    if sig_path.is_empty() {
        return None;
    }
    match fs::read_to_string(&sig_path) {
        Ok(s) if !s.trim().is_empty() => Some(format!("\n-- \n{}", s)),
        _ => None,
    }
}

/// Builds the quoted reply template
fn build_reply_template(in_reply_to: &str, original_body: &str) -> String {
    let quoted = original_body
        .lines()
        .map(|line| format!("> {}", line))
        .collect::<Vec<_>>()
        .join("\n");

    format!("> [in reply to {}]\n\n{}\n\n", in_reply_to, quoted)
}

/// High-level helper to compose a reply
pub async fn compose_reply(
    client: &mut NntpClient,
    newsgroup: &str,
    article_id: &str,
    subject: &str,
    from: String,
) -> Result<Option<PostArticle>, Box<dyn std::error::Error>> {
    // Fetch original article body
    let original_body = client
        .fetch_article_body(article_id)
        .await
        .unwrap_or_default();

    // Create reply template with quoted body
    let prefill = build_reply_template(&from, &original_body);

    // Write to tempfile
    let mut temp = NamedTempFile::new()?;
    write!(temp, "{}", prefill)?;
    let path = temp.path().to_owned();

    // Launch editor
    let editor = std::env::var("EDITOR").unwrap_or_else(|_| "nano".into());
    Command::new(editor).arg(&path).status()?;

    // Read edited content
    let body = fs::read_to_string(&path)?;

    // If user didn't change or quit
    if body.trim().is_empty() || body.trim() == prefill.trim() {
        return Ok(None);
    }

    // Append signature if available
    let mut full_body = body;
    if let Some(sig) = load_signature() {
        full_body.push_str(&sig);
    }

    // Build and return article
    let article = PostArticle {
        from,
        newsgroups: newsgroup.to_string(),
        subject: format!("Re: {}", subject),
        body: full_body,
        message_id: None,
        references: Some(article_id.to_string()),
    };

    Ok(Some(article))
}

/// Composes a brand-new post.
/// `newsgroup` may be a comma‑separated list (e.g. "group1,group2").
/// `subject` is used exactly as given.
pub async fn compose_post(
    _client: &mut NntpClient,
    newsgroup: &str,
    subject: &str,
    from: String,
) -> Result<Option<PostArticle>, Box<dyn std::error::Error>> {
    // Write an empty buffer to tempfile
    let temp = NamedTempFile::new()?;
    let path = temp.path().to_owned();

    // Launch editor for the user to write the body
    let editor = std::env::var("EDITOR").unwrap_or_else(|_| "nano".into());
    Command::new(editor).arg(&path).status()?;

    // Read edited content
    let body = fs::read_to_string(&path)?;

    // If empty, the user cancelled
    if body.trim().is_empty() {
        return Ok(None);
    }

    // Append signature if available
    let mut full_body = body;
    if let Some(sig) = load_signature() {
        full_body.push_str(&sig);
    }

    // Build and return article — use the subject exactly, and allow multiple newsgroups
    let article = PostArticle {
        from,
        newsgroups: newsgroup.to_string(),
        subject: subject.to_string(),
        body: full_body,
        message_id: None,
        references: None,
    };

    Ok(Some(article))
}
