/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.sidecar.server;

import com.google.inject.Inject;
import com.google.inject.Singleton;
import io.vertx.core.CompositeFuture;
import io.vertx.core.DeploymentOptions;
import io.vertx.core.Future;
import io.vertx.core.Promise;
import io.vertx.core.Vertx;
import io.vertx.core.eventbus.Message;
import io.vertx.core.eventbus.MessageConsumer;
import io.vertx.core.http.HttpServerOptions;
import io.vertx.core.json.JsonArray;
import io.vertx.core.json.JsonObject;
import io.vertx.core.net.SSLOptions;
import io.vertx.core.net.TrafficShapingOptions;
import io.vertx.ext.web.Router;
import java.io.PrintStream;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.stream.Collectors;
import org.apache.cassandra.sidecar.cluster.InstancesMetadata;
import org.apache.cassandra.sidecar.cluster.instance.InstanceMetadata;
import org.apache.cassandra.sidecar.common.utils.Preconditions;
import org.apache.cassandra.sidecar.concurrent.ExecutorPools;
import org.apache.cassandra.sidecar.config.SidecarConfiguration;
import org.apache.cassandra.sidecar.config.SslConfiguration;
import org.apache.cassandra.sidecar.metrics.SidecarMetrics;
import org.apache.cassandra.sidecar.server.HttpServerOptionsProvider;
import org.apache.cassandra.sidecar.server.ServerVerticle;
import org.apache.cassandra.sidecar.server.SidecarServerEvents;
import org.apache.cassandra.sidecar.tasks.PeriodicTaskExecutor;
import org.apache.cassandra.sidecar.utils.SidecarClientProvider;
import org.apache.cassandra.sidecar.utils.SslUtils;
import org.jetbrains.annotations.VisibleForTesting;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@Singleton
public class Server {
    private static final Logger LOGGER = LoggerFactory.getLogger(Server.class);
    protected final Vertx vertx;
    protected final ExecutorPools executorPools;
    protected final PeriodicTaskExecutor periodicTaskExecutor;
    protected final SidecarConfiguration sidecarConfiguration;
    protected final InstancesMetadata instancesMetadata;
    protected final Router router;
    protected final HttpServerOptionsProvider optionsProvider;
    protected final SidecarClientProvider sidecarClientProvider;
    protected final SidecarMetrics metrics;
    protected final List<ServerVerticle> deployedServerVerticles = new CopyOnWriteArrayList<ServerVerticle>();
    private final Set<Integer> cqlReadyInstanceIds = Collections.synchronizedSet(new HashSet());

    @Inject
    public Server(Vertx vertx, SidecarConfiguration sidecarConfiguration, Router router, InstancesMetadata instancesMetadata, ExecutorPools executorPools, PeriodicTaskExecutor periodicTaskExecutor, HttpServerOptionsProvider optionsProvider, SidecarClientProvider sidecarClientProvider, SidecarMetrics metrics) {
        this.vertx = vertx;
        this.executorPools = executorPools;
        this.periodicTaskExecutor = periodicTaskExecutor;
        this.sidecarConfiguration = sidecarConfiguration;
        this.instancesMetadata = instancesMetadata;
        this.router = router;
        this.optionsProvider = optionsProvider;
        this.sidecarClientProvider = sidecarClientProvider;
        this.metrics = metrics;
    }

    public Future<String> start() {
        this.banner(System.out);
        this.validate();
        LOGGER.info("Starting Cassandra Sidecar");
        int serverVerticleCount = this.sidecarConfiguration.serviceConfiguration().serverVerticleInstances();
        Preconditions.checkArgument((serverVerticleCount > 0 ? 1 : 0) != 0, (String)"Server verticle count can not be less than 1");
        LOGGER.debug("Deploying {} verticles to vertx", (Object)serverVerticleCount);
        DeploymentOptions deploymentOptions = new DeploymentOptions().setInstances(serverVerticleCount);
        HttpServerOptions options = this.optionsProvider.apply(this.sidecarConfiguration);
        return this.vertx.deployVerticle(() -> {
            ServerVerticle serverVerticle = new ServerVerticle(this.sidecarConfiguration, this.router, options);
            this.deployedServerVerticles.add(serverVerticle);
            return serverVerticle;
        }, deploymentOptions).compose(this::notifyServerStart);
    }

    public Future<Void> stop(String deploymentId) {
        LOGGER.info("Stopping Cassandra Sidecar");
        this.deployedServerVerticles.clear();
        Objects.requireNonNull(deploymentId, "deploymentId must not be null");
        return this.notifyServerStopping(deploymentId).compose(v -> this.vertx.undeploy(deploymentId)).onSuccess(v -> LOGGER.info("Successfully stopped Cassandra Sidecar"));
    }

    public Future<Void> close() {
        LOGGER.info("Stopping Cassandra Sidecar");
        this.deployedServerVerticles.clear();
        ArrayList<Object> closingFutures = new ArrayList<Object>();
        closingFutures.add(this.notifyServerStopping(null));
        closingFutures.add(Future.future(p -> {
            this.sidecarClientProvider.close();
            p.complete();
        }));
        this.instancesMetadata.instances().forEach(instance -> {
            Promise closingFutureForInstance = Promise.promise();
            this.executorPools.internal().runBlocking(() -> {
                try {
                    instance.delegate().close();
                }
                catch (Exception e) {
                    LOGGER.error("Failed to close delegate", (Throwable)e);
                    closingFutureForInstance.tryFail((Throwable)e);
                }
                finally {
                    closingFutureForInstance.tryComplete(null);
                }
            });
            closingFutures.add(closingFutureForInstance.future());
        });
        return Future.all(closingFutures).onSuccess(ignored -> LOGGER.debug("Closed Cassandra adapters")).transform(v -> {
            LOGGER.debug("Closing PeriodicTaskExecutor");
            return this.periodicTaskExecutor.close();
        }).transform(v -> {
            LOGGER.debug("Closing executor pools");
            return this.executorPools.close();
        }).transform(v -> {
            LOGGER.debug("Closing vertx");
            return this.vertx.close();
        }).onFailure(t -> LOGGER.error("Failed to gracefully shutdown Cassandra Sidecar", t)).onSuccess(f -> LOGGER.info("Successfully stopped Cassandra Sidecar"));
    }

    public Future<CompositeFuture> updateSSLOptions(long timestamp) {
        SSLOptions options = new SSLOptions();
        this.optionsProvider.configureSSLOptions(options, this.sidecarConfiguration.sslConfiguration(), timestamp);
        List updateFutures = this.deployedServerVerticles.stream().map(serverVerticle -> serverVerticle.updateSSLOptions(options)).collect(Collectors.toList());
        return Future.all(updateFutures);
    }

    public void updateTrafficShapingOptions(TrafficShapingOptions options) {
        this.deployedServerVerticles.forEach(serverVerticle -> serverVerticle.updateTrafficShapingOptions(options));
    }

    @VisibleForTesting
    public int actualPort() {
        if (!this.deployedServerVerticles.isEmpty()) {
            return this.deployedServerVerticles.get(0).actualPort();
        }
        throw new IllegalStateException("No deployed server verticles. Maybe server failed to deploy due to port conflict");
    }

    @VisibleForTesting
    public String deploymentId() {
        if (!this.deployedServerVerticles.isEmpty()) {
            return this.deployedServerVerticles.get(0).deploymentID();
        }
        throw new IllegalStateException("No deployed server verticles");
    }

    protected Future<String> notifyServerStart(String deploymentId) {
        LOGGER.info("Successfully started Cassandra Sidecar");
        MessageConsumer cqlReadyConsumer = this.vertx.eventBus().localConsumer(SidecarServerEvents.ON_CASSANDRA_CQL_READY.address());
        cqlReadyConsumer.handler(message -> this.onCqlReady((MessageConsumer<JsonObject>)cqlReadyConsumer, (Message<JsonObject>)message));
        this.vertx.eventBus().publish(SidecarServerEvents.ON_SERVER_START.address(), (Object)deploymentId);
        return Future.succeededFuture((Object)deploymentId);
    }

    protected Future<Void> notifyServerStopping(String deploymentId) {
        this.vertx.eventBus().publish(SidecarServerEvents.ON_SERVER_STOP.address(), (Object)deploymentId);
        return Future.succeededFuture();
    }

    protected void banner(PrintStream out) {
        out.println(" _____                               _              _____ _     _                     \n/  __ \\                             | |            /  ___(_)   | |                    \n| /  \\/ __ _ ___ ___  __ _ _ __   __| |_ __ __ _   \\ `--. _  __| | ___  ___ __ _ _ __ \n| |    / _` / __/ __|/ _` | '_ \\ / _` | '__/ _` |   `--. \\ |/ _` |/ _ \\/ __/ _` | '__|\n| \\__/\\ (_| \\__ \\__ \\ (_| | | | | (_| | | | (_| |  /\\__/ / | (_| |  __/ (_| (_| | |   \n \\____/\\__,_|___/___/\\__,_|_| |_|\\__,_|_|  \\__,_|  \\____/|_|\\__,_|\\___|\\___\\__,_|_|\n                                                                                      \n                                                                                      ");
    }

    protected void validate() {
        SslConfiguration ssl = this.sidecarConfiguration.sslConfiguration();
        if (ssl == null || !ssl.enabled()) {
            return;
        }
        try {
            if (!ssl.isKeystoreConfigured()) {
                throw new IllegalArgumentException("keyStorePath and keyStorePassword must be set if ssl enabled");
            }
            SslUtils.validateSslOpts(ssl.keystore());
            if (ssl.isTrustStoreConfigured()) {
                SslUtils.validateSslOpts(ssl.truststore());
            }
        }
        catch (Exception e) {
            throw new RuntimeException("Invalid keystore parameters for SSL", e);
        }
    }

    protected void onCqlReady(MessageConsumer<JsonObject> cqlReadyConsumer, Message<JsonObject> message) {
        this.cqlReadyInstanceIds.add(((JsonObject)message.body()).getInteger("cassandraInstanceId"));
        boolean isCqlReadyOnAllInstances = this.instancesMetadata.instances().stream().map(InstanceMetadata::id).allMatch(this.cqlReadyInstanceIds::contains);
        if (isCqlReadyOnAllInstances) {
            cqlReadyConsumer.unregister();
            this.notifyAllCassandraCqlAreReady();
            LOGGER.info("CQL is ready for all Cassandra instances. {}", this.cqlReadyInstanceIds);
        }
    }

    protected void notifyAllCassandraCqlAreReady() {
        JsonArray cassandraInstanceIds = new JsonArray();
        this.cqlReadyInstanceIds.forEach(arg_0 -> ((JsonArray)cassandraInstanceIds).add(arg_0));
        JsonObject allReadyMessage = new JsonObject().put("cassandraInstanceIds", (Object)cassandraInstanceIds);
        this.vertx.eventBus().publish(SidecarServerEvents.ON_ALL_CASSANDRA_CQL_READY.address(), (Object)allReadyMessage);
    }
}

