// Licensed to the Apache Software Foundation (ASF) under one
// or more contributor license agreements.  See the NOTICE file
// distributed with this work for additional information
// regarding copyright ownership.  The ASF licenses this file
// to you under the Apache License, Version 2.0 (the
// "License"); you may not use this file except in compliance
// with the License.  You may obtain a copy of the License at
//
//   http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing,
// software distributed under the License is distributed on an
// "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
// KIND, either express or implied.  See the License for the
// specific language governing permissions and limitations
// under the License.

#pragma once

#include <cstdint>
#include <memory>
#include <string>

#include "kudu/gutil/port.h"
#include "kudu/util/net/net_util.h"
#include "kudu/util/status.h"

namespace kudu {

class Subprocess;

namespace sentry {

class MiniSentry {
 public:

  MiniSentry();

  ~MiniSentry();

  // Configures the mini Sentry service to use Kerberos.
  void EnableKerberos(std::string krb5_conf,
                      std::string service_principal,
                      std::string keytab_file);

  // Configures the mini Sentry service to connect to a Hive Metastore instance.
  void EnableHms(std::string hms_uris);

  // Configures the mini Sentry service to store its data in the provided path.
  // If not set, it uses a test-only temporary directory.
  void SetDataRoot(std::string data_root);

  // Configures the mini Sentry service to start with the provided address.
  // If not set, it uses the default Ip and port number.
  void SetAddress(const HostPort& address);

  // Starts the mini Sentry service.
  //
  // If the MiniSentry has already been started and stopped, it will be restarted
  // using the same listening port.
  Status Start() WARN_UNUSED_RESULT;

  // Stops the mini Sentry service.
  Status Stop() WARN_UNUSED_RESULT;

  // Pause the Sentry service.
  Status Pause() WARN_UNUSED_RESULT;

  // Unpause the Sentry service.
  Status Resume() WARN_UNUSED_RESULT;

  // Returns the address of the mini Sentry service. Should only be called after the
  // Sentry service is started.
  HostPort address() const {
    return HostPort(ip_, port_);
  }

  // Returns true when Kerberos is enabled.
  bool IsKerberosEnabled() const {
    return !keytab_file_.empty();
  }

  // Returns true when the HMS is enabled.
  bool IsHmsEnabled() const {
    return !hms_uris_.empty();
  }

 private:

  // Creates a sentry-site.xml for the mini Sentry, and other supporting
  // configuration files.
  Status CreateSentryConfigs(const std::string& tmp_dir) const WARN_UNUSED_RESULT;

  std::unique_ptr<Subprocess> sentry_process_;

  // Port number of the mini Sentry service. Default to 0.
  uint16_t port_ = 0;
  // Ip address of the mini Sentry service. Default to 0.0.0.0.
  std::string ip_ = "0.0.0.0";
  std::string data_root_;

  // Kerberos configuration
  std::string krb5_conf_;
  std::string service_principal_;
  std::string keytab_file_;

  // HMS configuration
  std::string hms_uris_;
};

} // namespace sentry
} // namespace kudu
