/*
 * Decompiled with CFR 0.152.
 */
package org.apache.twill.internal.kafka.client;

import com.google.common.base.Charsets;
import com.google.common.base.Function;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Supplier;
import com.google.common.base.Suppliers;
import com.google.common.base.Throwables;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import com.google.common.primitives.Ints;
import com.google.common.util.concurrent.AbstractIdleService;
import com.google.common.util.concurrent.FutureCallback;
import com.google.common.util.concurrent.Futures;
import com.google.common.util.concurrent.SettableFuture;
import com.google.gson.Gson;
import java.util.Set;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Executor;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicReference;
import org.apache.twill.common.Cancellable;
import org.apache.twill.common.Threads;
import org.apache.twill.kafka.client.BrokerInfo;
import org.apache.twill.kafka.client.BrokerService;
import org.apache.twill.kafka.client.TopicPartition;
import org.apache.twill.zookeeper.NodeChildren;
import org.apache.twill.zookeeper.NodeData;
import org.apache.twill.zookeeper.ZKClient;
import org.apache.zookeeper.KeeperException;
import org.apache.zookeeper.WatchedEvent;
import org.apache.zookeeper.Watcher;
import org.apache.zookeeper.data.Stat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ZKBrokerService
extends AbstractIdleService
implements BrokerService {
    private static final Logger LOG = LoggerFactory.getLogger(ZKBrokerService.class);
    private static final String BROKER_IDS_PATH = "/brokers/ids";
    private static final String BROKER_TOPICS_PATH = "/brokers/topics";
    private static final long FAILURE_RETRY_SECONDS = 5L;
    private static final Gson GSON = new Gson();
    private static final Function<String, BrokerId> BROKER_ID_TRANSFORMER = new Function<String, BrokerId>(){

        public BrokerId apply(String input) {
            return new BrokerId(Integer.parseInt(input));
        }
    };
    private static final Function<BrokerInfo, String> BROKER_INFO_TO_ADDRESS = new Function<BrokerInfo, String>(){

        public String apply(BrokerInfo input) {
            return String.format("%s:%d", input.getHost(), input.getPort());
        }
    };
    private final ZKClient zkClient;
    private final LoadingCache<BrokerId, Supplier<BrokerInfo>> brokerInfos;
    private final LoadingCache<KeyPathTopicPartition, Supplier<PartitionInfo>> partitionInfos;
    private final Set<ListenerExecutor> listeners;
    private ExecutorService executorService;
    private Supplier<Iterable<BrokerInfo>> brokerList;

    public ZKBrokerService(ZKClient zkClient) {
        this.zkClient = zkClient;
        this.brokerInfos = CacheBuilder.newBuilder().build(this.createCacheLoader(new CacheInvalidater<BrokerId>(){

            @Override
            public void invalidate(BrokerId key) {
                ZKBrokerService.this.brokerInfos.invalidate((Object)key);
            }
        }, BrokerInfo.class));
        this.partitionInfos = CacheBuilder.newBuilder().build(this.createCacheLoader(new CacheInvalidater<KeyPathTopicPartition>(){

            @Override
            public void invalidate(KeyPathTopicPartition key) {
                ZKBrokerService.this.partitionInfos.invalidate((Object)key);
            }
        }, PartitionInfo.class));
        this.listeners = Sets.newCopyOnWriteArraySet();
    }

    protected void startUp() throws Exception {
        this.executorService = Executors.newCachedThreadPool(Threads.createDaemonThreadFactory("zk-kafka-broker"));
    }

    protected void shutDown() throws Exception {
        this.executorService.shutdownNow();
    }

    @Override
    public BrokerInfo getLeader(String topic, int partition) {
        Preconditions.checkState((boolean)this.isRunning(), (Object)"BrokerService is not running.");
        PartitionInfo partitionInfo = (PartitionInfo)((Supplier)this.partitionInfos.getUnchecked((Object)new KeyPathTopicPartition(topic, partition))).get();
        return partitionInfo == null ? null : (BrokerInfo)((Supplier)this.brokerInfos.getUnchecked((Object)new BrokerId(partitionInfo.getLeader()))).get();
    }

    @Override
    public synchronized Iterable<BrokerInfo> getBrokers() {
        Preconditions.checkState((boolean)this.isRunning(), (Object)"BrokerService is not running.");
        if (this.brokerList != null) {
            return (Iterable)this.brokerList.get();
        }
        final SettableFuture readerFuture = SettableFuture.create();
        final AtomicReference<ImmutableList> brokers = new AtomicReference<ImmutableList>(ImmutableList.of());
        this.actOnExists(BROKER_IDS_PATH, new Runnable(){

            @Override
            public void run() {
                FutureCallback<NodeChildren> childrenCallback = new FutureCallback<NodeChildren>(){

                    public void onSuccess(NodeChildren result) {
                        try {
                            brokers.set(ImmutableList.copyOf((Iterable)Iterables.transform((Iterable)ZKBrokerService.this.brokerInfos.getAll(Iterables.transform(result.getChildren(), (Function)BROKER_ID_TRANSFORMER)).values(), (Function)Suppliers.supplierFunction())));
                            readerFuture.set(null);
                            for (ListenerExecutor listener : ZKBrokerService.this.listeners) {
                                listener.changed(ZKBrokerService.this);
                            }
                        }
                        catch (ExecutionException e) {
                            readerFuture.setException(e.getCause());
                        }
                    }

                    public void onFailure(Throwable t) {
                        readerFuture.setException(t);
                    }
                };
                Futures.addCallback(ZKBrokerService.this.zkClient.getChildren(ZKBrokerService.BROKER_IDS_PATH, new Watcher((FutureCallback)childrenCallback){
                    final /* synthetic */ FutureCallback val$childrenCallback;
                    {
                        this.val$childrenCallback = futureCallback;
                    }

                    public void process(WatchedEvent event) {
                        if (!ZKBrokerService.this.isRunning()) {
                            return;
                        }
                        if (event.getType() == Watcher.Event.EventType.NodeChildrenChanged) {
                            Futures.addCallback(ZKBrokerService.this.zkClient.getChildren(ZKBrokerService.BROKER_IDS_PATH, this), (FutureCallback)this.val$childrenCallback, (Executor)ZKBrokerService.this.executorService);
                        }
                    }
                }), (FutureCallback)childrenCallback, (Executor)ZKBrokerService.this.executorService);
            }
        }, readerFuture, 5L, TimeUnit.SECONDS);
        this.brokerList = this.createSupplier(brokers);
        try {
            readerFuture.get();
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
        return (Iterable)this.brokerList.get();
    }

    @Override
    public String getBrokerList() {
        return Joiner.on((char)',').join(Iterables.transform(this.getBrokers(), BROKER_INFO_TO_ADDRESS));
    }

    @Override
    public Cancellable addChangeListener(BrokerService.BrokerChangeListener listener, Executor executor) {
        final ListenerExecutor listenerExecutor = new ListenerExecutor(listener, executor);
        this.listeners.add(listenerExecutor);
        return new Cancellable(){

            @Override
            public void cancel() {
                ZKBrokerService.this.listeners.remove(listenerExecutor);
            }
        };
    }

    private <K extends KeyPath, T> CacheLoader<K, Supplier<T>> createCacheLoader(final CacheInvalidater<K> invalidater, final Class<T> resultType) {
        return new CacheLoader<K, Supplier<T>>(){

            public Supplier<T> load(K key) throws Exception {
                final SettableFuture readyFuture = SettableFuture.create();
                final AtomicReference resultValue = new AtomicReference();
                final String path = key.getPath();
                ZKBrokerService.this.actOnExists(path, new Runnable((KeyPath)key){
                    final /* synthetic */ KeyPath val$key;
                    {
                        this.val$key = keyPath;
                    }

                    @Override
                    public void run() {
                        FutureCallback<NodeData> dataCallback = new FutureCallback<NodeData>(){

                            public void onSuccess(NodeData result) {
                                Object value = ZKBrokerService.this.decodeNodeData(result, resultType);
                                resultValue.set(value);
                                readyFuture.set(value);
                            }

                            public void onFailure(Throwable t) {
                                LOG.error("Failed to fetch node data on {}", (Object)path, (Object)t);
                                if (t instanceof KeeperException.NoNodeException) {
                                    resultValue.set(null);
                                    readyFuture.set(null);
                                    return;
                                }
                                invalidater.invalidate(val$key);
                                readyFuture.setException(t);
                            }
                        };
                        Futures.addCallback(ZKBrokerService.this.zkClient.getData(path, new Watcher((FutureCallback)dataCallback){
                            final /* synthetic */ FutureCallback val$dataCallback;
                            {
                                this.val$dataCallback = futureCallback;
                            }

                            public void process(WatchedEvent event) {
                                if (!ZKBrokerService.this.isRunning()) {
                                    return;
                                }
                                if (event.getType() == Watcher.Event.EventType.NodeDataChanged) {
                                    Futures.addCallback(ZKBrokerService.this.zkClient.getData(path, this), (FutureCallback)this.val$dataCallback, (Executor)ZKBrokerService.this.executorService);
                                } else if (event.getType() == Watcher.Event.EventType.NodeDeleted) {
                                    ZKBrokerService.this.brokerInfos.invalidate((Object)val$key);
                                }
                            }
                        }), (FutureCallback)dataCallback, (Executor)ZKBrokerService.this.executorService);
                    }
                }, readyFuture, 5L, TimeUnit.SECONDS);
                readyFuture.get();
                return ZKBrokerService.this.createSupplier(resultValue);
            }
        };
    }

    private <T> T decodeNodeData(NodeData nodeData, Class<T> type) {
        byte[] data;
        byte[] byArray = data = nodeData == null ? null : nodeData.getData();
        if (data == null) {
            return null;
        }
        return GSON.fromJson(new String(data, Charsets.UTF_8), type);
    }

    private void actOnExists(final String path, final Runnable action, final SettableFuture<?> readyFuture, final long retryTime, final TimeUnit retryUnit) {
        Futures.addCallback(this.zkClient.exists(path, new Watcher(){

            public void process(WatchedEvent event) {
                if (!ZKBrokerService.this.isRunning()) {
                    return;
                }
                if (event.getType() == Watcher.Event.EventType.NodeCreated) {
                    action.run();
                }
            }
        }), (FutureCallback)new FutureCallback<Stat>(){

            public void onSuccess(Stat result) {
                if (result != null) {
                    action.run();
                } else {
                    readyFuture.set(null);
                }
            }

            public void onFailure(Throwable t) {
                Thread retryThread = new Thread("zk-broker-service-retry"){

                    @Override
                    public void run() {
                        try {
                            retryUnit.sleep(retryTime);
                            ZKBrokerService.this.actOnExists(path, action, readyFuture, retryTime, retryUnit);
                        }
                        catch (InterruptedException e) {
                            LOG.warn("ZK retry thread interrupted. Action not retried.");
                        }
                    }
                };
                retryThread.setDaemon(true);
                retryThread.start();
            }
        }, (Executor)this.executorService);
    }

    private <T> Supplier<T> createSupplier(final AtomicReference<T> ref) {
        return new Supplier<T>(){

            public T get() {
                return ref.get();
            }
        };
    }

    private static final class ListenerExecutor
    extends BrokerService.BrokerChangeListener {
        private final BrokerService.BrokerChangeListener listener;
        private final Executor executor;

        private ListenerExecutor(BrokerService.BrokerChangeListener listener, Executor executor) {
            this.listener = listener;
            this.executor = executor;
        }

        @Override
        public void changed(final BrokerService brokerService) {
            try {
                this.executor.execute(new Runnable(){

                    @Override
                    public void run() {
                        try {
                            ListenerExecutor.this.listener.changed(brokerService);
                        }
                        catch (Throwable t) {
                            LOG.error("Failure when calling BrokerChangeListener.", t);
                        }
                    }
                });
            }
            catch (Throwable t) {
                LOG.error("Failure when calling BrokerChangeListener.", t);
            }
        }
    }

    private static final class PartitionInfo {
        private int[] isr;
        private int leader;

        private PartitionInfo() {
        }

        private int[] getIsr() {
            return this.isr;
        }

        private int getLeader() {
            return this.leader;
        }
    }

    private static final class KeyPathTopicPartition
    extends TopicPartition
    implements KeyPath {
        private KeyPathTopicPartition(String topic, int partition) {
            super(topic, partition);
        }

        @Override
        public String getPath() {
            return String.format("%s/%s/partitions/%d/state", ZKBrokerService.BROKER_TOPICS_PATH, this.getTopic(), this.getPartition());
        }
    }

    private static final class BrokerId
    implements KeyPath {
        private final int id;

        private BrokerId(int id) {
            this.id = id;
        }

        public boolean equals(Object o) {
            return this == o || o != null && this.getClass() == o.getClass() && this.id == ((BrokerId)o).id;
        }

        public int hashCode() {
            return Ints.hashCode((int)this.id);
        }

        @Override
        public String getPath() {
            return "/brokers/ids/" + this.id;
        }
    }

    private static interface KeyPath {
        public String getPath();
    }

    private static interface CacheInvalidater<T> {
        public void invalidate(T var1);
    }
}

