package org.apache.dvsl;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.util.List;
import java.util.Stack;
import java.util.Map;
import java.util.HashMap;

import java.io.Writer;
import java.io.StringWriter;
import java.io.Reader;

import org.apache.velocity.context.Context;
import org.apache.velocity.app.VelocityEngine;
import org.apache.velocity.VelocityContext;

import org.dom4j.Document;
import org.dom4j.Element;
import org.dom4j.Node;
import org.dom4j.DocumentException;
import org.dom4j.io.SAXReader;

import org.apache.dvsl.dom4j.Dom4jNodeImpl;

/**
 *  <p>
 *   Class responsible for actual transformation
 *  of documents.
 *  </p>
 *
 *  <p>
 *  Note that this class is <em>not</em> threadsafe.
 *  </p>
 *
 *  @author <a href="mailto:geirm@apache.org>Geir Magnusson Jr.</a>
 */
class Transformer implements TransformTool
{
    /**
     *  Instance of VelocityEngine we are currently using.
     *  This must be reset with a stylesheeet change
     */
    private VelocityEngine ve = null;

    /**
     *  SAXReader that we reuse for every document.  Much faster.
     */
    private SAXReader saxReader = null;

    /**
     *  basic context passed to us - can contain tools
     *  and such for use.  Is protected from change via
     *  wrapping
     */
    private Context baseContext;

    /**
     *  context used during processing. Wraps the baseContext
     */
    private DVSLNodeContext currentContext;

    private TemplateHandler templateHandler = null;

    /**
     *  HashMap to hold application values
     */

    private Map appValue = new HashMap();

    /**
     *  Sole public CTOR.  We rely on the caller to give us a
     *  VelocityEngine ready with all macros registered.
     *  The context is the callers context with all tools and
     *  style drek.
     */
    public Transformer(VelocityEngine ve, TemplateHandler th,
                        Context context, Map applicationValues,
                        boolean validate)
    {
        this.ve = ve;
        this.baseContext = context;
        this.templateHandler = th;

        appValue = applicationValues;

        saxReader = new SAXReader(validate);
    }

    /**
     *  "Sealed for your protection."
     */
    private Transformer()
    {
    }

    /**
     *  Method that performs the transformation on
     *  a document
     *
     *  @param reader XML document char stream
     *  @param writer Writer to output transformation to
     */
    long transform(Reader reader, Writer writer)
        throws Exception
    {

        /*
         *  parse the document
         */
        Document document = saxReader.read(reader);

        return transform(document, writer);
    }

    long transform(Document dom4jdoc, Writer writer)
        throws Exception
    {
        /*
         *  wrap the document.  We do this as we let the dom4j package
         *  decide if we have a match against "/", so we need document
         *  to do that
         */

        DVSLNode root = new Dom4jNodeImpl(dom4jdoc);

        return transform(root, writer);
    }

    protected long transform(DVSLNode root, Writer writer)
        throws Exception
    {
        /*
          *  wrap in a context to keep subsequent documents from
          *  interacting with each other
          */

         currentContext = new DVSLNodeContext(baseContext);

         long start = System.currentTimeMillis();

        /*
         *  push 'this' into the context as our TransformTool
         *  and invoke the transformation
         */

        currentContext.put("context", this);

        invoke(root, writer);

        long end = System.currentTimeMillis();

        return end-start;
    }


    private void invoke(DVSLNode element, Writer writer)
        throws Exception
    {
        String[] arr = { };

        currentContext.pushNode(element);

        templateHandler.render(element, currentContext, writer);

        currentContext.popNode();
    }

    public Object get(String key)
    {
        return currentContext.get(key);
    }

    public String applyTemplates(DVSLNode node, String xpath)
        throws Exception
    {
        /*
         *  get the nodes that was asked for
         */

        List nodeset = node.selectNodes(xpath);

        StringWriter sw =  new StringWriter();

        for (int i = 0; i < nodeset.size(); i++)
        {
            DVSLNode n = (DVSLNode) nodeset.get(i);

            invoke(n, sw);
        }

        return sw.toString();
    }

    public String applyTemplates(DVSLNode node)
        throws Exception
    {
        StringWriter sw = new StringWriter();

        invoke(node, sw);

        return sw.toString();
    }

    public String applyTemplates()
        throws Exception
    {
        return applyTemplates(currentContext.peekNode(),
                               "*|@*|text()|comment()|processing-instruction()");
    }

    public String applyTemplates(String path)
        throws Exception
    {
        DVSLNode node = currentContext.peekNode();

        return applyTemplates(node, path);
    }

    public String copy()
    {
        /*
         *  fakie, for now
         */

        DVSLNode node = currentContext.peekNode();

        return node.copy();
    }

    public Object getAppValue(Object key)
    {
        return appValue.get(key);
    }

    public Object putAppValue(Object key, Object value)
    {
        return appValue.put(key, value);
    }

}
