/**
 * Copyright (c) 2021, 2024 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
/*
 * generated by Xtext
 */
package org.eclipse.lsat.setting.teditor.validation

import expressions.Declaration
import expressions.ExpressionsPackage
import expressions.impl.DeclarationRefImpl
import java.util.Collection
import java.util.HashMap
import java.util.HashSet
import machine.Import
import machine.Machine
import machine.MachinePackage
import machine.Peripheral
import machine.Resource
import org.eclipse.emf.common.util.EMap
import org.eclipse.emf.common.util.URI
import org.eclipse.lsat.motioncalculator.MotionException
import org.eclipse.lsat.timing.calculator.MotionCalculatorExtension
import org.eclipse.xtext.EcoreUtil2
import org.eclipse.xtext.validation.Check
import setting.MotionProfileSettings
import setting.MotionSettings
import setting.PhysicalSettings
import setting.SettingPackage
import setting.Settings
import setting.impl.MotionSettingsMapEntryImpl

import static extension org.eclipse.lsat.common.xtend.Queries.*

/** 
 * Custom validation rules. 
 * 
 * see http://www.eclipse.org/Xtext/documentation.html#validation
 */
class SettingValidator extends AbstractSettingValidator {

    public static val INVALID_IMPORT = 'invalidImport'

    @Check
    def checkImportIsValid(Import imp) {
        try {
            val isImportUriValid = EcoreUtil2.isValidUri(imp, URI.createURI(imp.importURI))
            if (!isImportUriValid) {
                error('''The import «imp.importURI» cannot be resolved. Make sure that the name is spelled correctly.''',
                    imp, MachinePackage.Literals.IMPORT__IMPORT_URI, INVALID_IMPORT)
            }
            val isUnderstood = imp.importURI.matches(".*\\.(machine|setting)")
            if (!isUnderstood) {
                error('''Importing «imp.importURI» is not allowed. Only 'machine' and 'setting' files are allowed''',
                    imp, MachinePackage.Literals.IMPORT__IMPORT_URI, INVALID_IMPORT)
            }
        } catch (IllegalArgumentException e) {
            error('''The import «imp.importURI» is not a valid URI.''', imp,
                MachinePackage.Literals.IMPORT__IMPORT_URI, INVALID_IMPORT)
        }
    }

    @Check
    def checkPhysicalSettingsForDuplicates(Settings settings) {
        settings.physicalSettings.groupBy[resource].values.forEach [
            groupBy[peripheral].values.filter[size > 1].flatten.forEach [ duplicate |
                error('''Physical settings have been defined more than once for peripheral «duplicate.fqn». Please remove all duplicate instances.''',
                    settings,
                    SettingPackage.Literals.SETTINGS__PHYSICAL_SETTINGS, settings.physicalSettings.indexOf(duplicate))
            ]
        ]
    }

    @Check
    def checkPhysicalSettingsForDuplicatesinResourceResourceImpl(Settings settings) {
        settings.physicalSettings.groupBy[resource.resource].values.forEach[ 
            groupBy[peripheral].values.filter[groupBy[resource.class].size>1].flatten.forEach[
            error('''Physical settings have been defined for resource and resource item. Settings must be defined on Resource or all Resource items.''',
                settings,
                SettingPackage.Literals.SETTINGS__PHYSICAL_SETTINGS, settings.physicalSettings.indexOf(it))
            ]
        ]
    }

    @Check
    def checkTransitivePhysicalSettingsForDuplicates(Settings settings) {
        settings.imports.forEach [ imp |
            val importedSettings = imp.loadSettings.collect[physicalSettings]
            settings.physicalSettings.filter(ps|importedSettings.hasDuplicate(ps)).forEach [ 
                val errorText = '''Settings for peripheral '«fqn»' are already defined in '«imp.importURI»'. Remove either the import or this definition.'''
                error(errorText, settings,
                    SettingPackage.Literals.SETTINGS__PHYSICAL_SETTINGS, settings.physicalSettings.indexOf(it))
                error(errorText, imp, MachinePackage.Literals.IMPORT__IMPORT_URI)
            ]
        ]
    }

    /** Duplicate if a peripheral setting exists for this resource or the resource parent */
    def hasDuplicate(Iterable<PhysicalSettings> col, PhysicalSettings ps ){
        !col.filter[peripheral==ps.peripheral && (resource==ps.resource || resource==ps.resource.resource)].empty
    }

    @Check
    def checkTransitivePhysicalSettingsForDuplicatesInImports(Settings settings) {
        // collect physical settings per import, the import object is needed when an error occurs
        val importContents = new HashMap<Import, Collection<PhysicalSettings>>
        settings.imports.forEach[importContents.put(it, loadSettings.collect[physicalSettings].toSet)]

        importContents.values.flatten.groupBy[resource].values.forEach [
            groupBy[peripheral].values.filter[size > 1].flatten.forEach [ setting |
                val imp = importContents.entrySet.findFirst[value.contains(setting)].key
                val errorText = '''Duplicate definition of '«setting.fqn»'. Remove import or one of the '«setting.fqn»' definitions.'''
                error(errorText, imp, MachinePackage.Literals.IMPORT__IMPORT_URI)
            ]
        ]
    }

    @Check
    def checkDeclarationsForDuplicates(Settings settings) {
        settings.declarations.groupBy[name].values.filter[size > 1].flatten.forEach[ 
            error('''Declaration '«name»' declared more than once, please remove one''',
                settings,
                SettingPackage.Literals.SETTINGS__DECLARATIONS, settings.declarations.indexOf(it))
            ]
    }

    @Check
    def checkTransitiveDeclarationsForDuplicates(Settings settings) {
        settings.imports.forEach [ imp |
            val importedSettings = imp.loadSettings.collect[declarations]
            settings.declarations.filter(d|importedSettings.hasDuplicate(d)).forEach [ 
                val errorText = '''Declaration '«name»' is already defined in '«imp.importURI»'. Remove either the import or the declaration.'''
                error(errorText, settings,
                    SettingPackage.Literals.SETTINGS__DECLARATIONS, settings.declarations.indexOf(it))
            ]
        ]
    }

    @Check
    def checkTransitiveDeclarationForDuplicatesInImports(Settings settings) {
        // collect physical settings per import, the import object is needed when an error occurs
        val importContents = new HashMap<Import, Collection<Declaration>>
        settings.imports.forEach[importContents.put(it, loadSettings.collect[declarations].toSet)]

        importContents.values.flatten.unique.groupBy[name].values.filter[size > 1].flatten.forEach[d|
                val imp = importContents.entrySet.findFirst[value.contains(d)].key
                val errorText = '''Declaration '«d.name»' declared more than once in imports. Remove import or one of the '«d.name»' declarations.'''
                error(errorText, imp, MachinePackage.Literals.IMPORT__IMPORT_URI)
            ]
    }

    /** Duplicate if a declaration with name exist in the collection */
    def hasDuplicate(Iterable<Declaration> col, Declaration d ){
        !col.filter[name==d.name].empty
    }

    private def loadSettings(Import imp) {
        val result = new HashSet<Settings>(); // copy load because it will get modified
        try {
            val load = imp.load.filter(Settings).toSet
            result.addAll(load);
            load.collect[loadAll].filter(Settings).forEach[result.add(it)]
        } catch (Exception e) {
            // ignore loading of settings it the resource does not exist
        }
        return result;
    }

   /**
    * use impl because of suppressed visibility of hasCircularDependencies
    */
    @Check
    def checkCircularReference(DeclarationRefImpl ref) {
        if(ref.hasCircularDependencies(ref)){
                error('''Circular reference''',ref, ExpressionsPackage.Literals::DECLARATION_REF__DECLARATION)
        }
    }

    @Check
    def checkPhysicalSettingsForResourceComplete(Settings settings) {
        settings.imports.forEach [ machineImport, index |
            val importedPeripherals = machineImport.load.filter(Machine).collect[resources].collect[peripherals].toSet
            val incompletePeripherals = importedPeripherals.filter[peripheral|
            !peripheral.incompleteResourceSettings(settings).empty].toSet
            if (!incompletePeripherals.empty) {
                val text = '''No settings specified for peripheral «FOR peripheral : incompletePeripherals SEPARATOR ', '»«peripheral.incompleteSettingsStr(settings)»«ENDFOR»'''
                warning(text, MachinePackage.Literals::IMPORT_CONTAINER__IMPORTS, index)

            }
        ]
    }

    def incompleteSettingsStr(Peripheral peripheral, Settings settings){
       String.join(',',peripheral.incompleteResourceSettings(settings).map["'"+ fqn +'.'+ peripheral.name +"'"])
    }
    
    def incompleteResourceSettings(Peripheral peripheral, Settings settings){
        peripheral.resource.resourceOrItems.filter[settings.getPhysicalSettings(it, peripheral) === null]
    }
    def resourceOrItems( Resource resource) {
        resource.items.empty ? newArrayList(resource) : resource.items
    }
    

    @Check
    def checkMotionSettingsComplete(PhysicalSettings physicalSettings) {
        val peripheralAxes = physicalSettings.peripheral.type.axes.noSettings(physicalSettings.motionSettings);
        if (!peripheralAxes.isEmpty) {
            warning('''No motion settings specified for axes «FOR axis : peripheralAxes SEPARATOR ', '»«axis.name»«ENDFOR»''',
                SettingPackage.Literals::PHYSICAL_SETTINGS__MOTION_SETTINGS)
        }
    }

    @Check
    def checkTimingSettingsComplete(PhysicalSettings physicalSettings) {
        val peripheralActions = physicalSettings.peripheral.type.actions.noSettings(physicalSettings.timingSettings)
        if (!peripheralActions.isEmpty) {
            warning('''No timing settings specified for actions «FOR action : peripheralActions SEPARATOR ', '»«action.name»«ENDFOR»''',
                SettingPackage.Literals::PHYSICAL_SETTINGS__TIMING_SETTINGS)
        }
    }

    @Check
    def checkProfileSettingsComplete(MotionSettings motionSettings) {
        val peripheral = (motionSettings.entry as MotionSettingsMapEntryImpl).settings.peripheral
        val peripheralProfiles = peripheral.profiles.noSettings(motionSettings.profileSettings)
        if (!peripheralProfiles.isEmpty) {
            warning('''No settings specified for profiles «FOR profile : peripheralProfiles SEPARATOR ', '»«profile.name»«ENDFOR»''',
                SettingPackage.Literals::MOTION_SETTINGS__PROFILE_SETTINGS)
        }
    }

    @Check
    def checkTimingSettingsComplete(MotionSettings motionSettings) {
        val axis = (motionSettings.entry as MotionSettingsMapEntryImpl).key
        val peripheral = (motionSettings.entry as MotionSettingsMapEntryImpl).settings.peripheral
        val peripheralPositions = peripheral.positions.map[getPosition(axis)].noSettings(motionSettings.locationSettings)

        if (!peripheralPositions.isEmpty) {
            warning('''No settings specified for positions «FOR position : peripheralPositions SEPARATOR ', '»«position.name»«ENDFOR»''',
                SettingPackage.Literals::MOTION_SETTINGS__LOCATION_SETTINGS)
        }
    }

    @Check
    def checkMotionProfileSettings(MotionProfileSettings mps) {
        try {
            val MotionCalculatorExtension motionCalculator = MotionCalculatorExtension.selectedMotionCalculator
            val motionProfile = motionCalculator.getMotionProfile(mps.motionProfile)
            if (motionProfile === null) {
                error('''Selected motion calculator «motionCalculator.name» does not support «mps.motionProfile===null? "default ":""»motion profile «mps.motionProfile»''',
                    SettingPackage.Literals::MOTION_PROFILE_SETTINGS__MOTION_PROFILE)
                return
            }

            val requiredParameterKeys = motionProfile.parameters.filter[required].map[key].noSettings(mps.motionArguments)
            if (!requiredParameterKeys.isEmpty) {
                error('''No settings specified for required parameters «FOR parameter : requiredParameterKeys SEPARATOR ', '»«parameter»«ENDFOR»''',
                    SettingPackage.Literals::MOTION_PROFILE_SETTINGS__MOTION_ARGUMENTS)
            }

            val parameterKeys = motionProfile.parameters.map[key]
            mps.motionArguments.forEach [ argument, index |
                if (!parameterKeys.contains(argument.key)) {
                    error('''Unknown parameter «argument.key»  for motion profile «mps.motionProfile»''',
                        SettingPackage.Literals::MOTION_PROFILE_SETTINGS__MOTION_ARGUMENTS, index)
                }
            ]
        } catch (MotionException e) {
            error(e.message, null)
        }
    }
    
    def <T> noSettings(Iterable<T> col, EMap<T,?> settings){
        col.filter[k|settings.keySet.findFirst[it==k]===null].toSet
    }
}
