/*
 * Copyright (c) 2021, 2026 Contributors to the Eclipse Foundation
 *
 * This program and the accompanying materials are made
 * available under the terms of the Eclipse Public License 2.0
 * which is available at https://www.eclipse.org/legal/epl-2.0/
 *
 * SPDX-License-Identifier: EPL-2.0
 */
package org.eclipse.lsat.engine.python.generator;

import static org.eclipse.lsat.common.queries.QueryableIterable.from;

import java.math.BigDecimal;
import java.text.DecimalFormat;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.List;
import java.util.Map;

import org.eclipse.lsat.common.graph.directed.DirectedGraph;
import org.eclipse.lsat.common.graph.directed.Edge;
import org.eclipse.lsat.common.graph.directed.Node;
import org.eclipse.lsat.common.graph.directed.util.DirectedGraphQueries;
import org.eclipse.lsat.common.scheduler.schedule.ScheduledDependency;
import org.eclipse.lsat.common.scheduler.schedule.ScheduledDependencyType;
import org.eclipse.lsat.common.scheduler.schedule.ScheduledTask;
import org.eclipse.lsat.common.util.IterableUtil;
import org.eclipse.lsat.timing.util.AxesLocation;
import org.eclipse.lsat.timing.util.MotionCalculatorHelper;
import org.eclipse.lsat.timing.util.SpecificationException;

import activity.Move;
import machine.IResource;
import machine.SetPoint;
import machine.SymbolicPosition;
import setting.Settings;

/**
 *
 */
public class GeneratorServices {
    private static final DecimalFormat FORMAT = new DecimalFormat("0000");
    private static final SimpleDateFormat DATE_FORMAT = new SimpleDateFormat("dd-MMM-yyyy");

    public static final String format(Integer number) {
        return FORMAT.format(number);
    }

    public static final String currentDate() {
        return DATE_FORMAT.format(new Date());
    }

    public static Map<SetPoint, BigDecimal> getSourceSetPoints(Move move, Settings settings) {
        return getSetPoints(move.getResource(), move.getSourcePosition(), settings);
    }

    public static Map<SetPoint, BigDecimal> getTargetSetPoints(Move move, Settings settings) {
        return getSetPoints(move.getResource(), move.getTargetPosition(), settings);
    }

    private static Map<SetPoint, BigDecimal> getSetPoints(IResource resource, SymbolicPosition position, Settings settings) {
        try {
            MotionCalculatorHelper helper = new MotionCalculatorHelper(settings, null);
            AxesLocation axesLocation = helper.getAxesLocation(resource, position);
            return MotionCalculatorHelper.toSetPoint(position.getPeripheral().getType(), axesLocation);
        } catch (SpecificationException e) {
            return Collections.emptyMap();
        }
    }

    public static BigDecimal getValue(Map<SetPoint, BigDecimal> map, SetPoint setPoint) {
        return map.get(setPoint);
    }

    public static BigDecimal getDelay(ScheduledTask<?> task) {
        return IterableUtil.min(from(task.getIncomingEdges()).collectOne(GeneratorServices::getDelay), BigDecimal.ZERO);
    }

    private static BigDecimal getDelay(Edge edge) {
        if (edge instanceof ScheduledDependency dep && ScheduledDependencyType.SOURCE_NODE_ENDS_BEFORE_TARGET_NODE_STARTS.equals(dep.getType())) {
            return ((ScheduledTask<?>)edge.getTargetNode()).getStartTime().subtract(((ScheduledTask<?>)edge.getSourceNode()).getEndTime());
        }
        return BigDecimal.ZERO;
    }

    public static <N extends Node> List<N> allNodesInReverseTopologicalOrder(DirectedGraph<N, ?> graph) {
        return DirectedGraphQueries.reverseTopologicalOrdering(
              DirectedGraphQueries.allSubNodes(graph),
              Comparator.comparing(Node::getName));
    }
}
