/**
 * Copyright (c) 2021 CEA LIST.
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *  Ansgar Radermacher  ansgar.radermacher@cea.fr
 */
package org.eclipse.papyrus.designer.transformation.tracing.barectf.library;

import java.util.List;
import org.eclipse.core.resources.IMarker;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.designer.deployment.profile.DepProfileResource;
import org.eclipse.papyrus.designer.deployment.profile.Deployment.ConfigurationProperty;
import org.eclipse.papyrus.designer.deployment.profile.Deployment.InstanceConfigurator;
import org.eclipse.papyrus.designer.deployment.profile.Deployment.UseInstanceConfigurator;
import org.eclipse.papyrus.designer.languages.common.base.ElementUtils;
import org.eclipse.papyrus.designer.languages.common.base.StdUriConstants;
import org.eclipse.papyrus.designer.languages.common.base.StringUtils;
import org.eclipse.papyrus.designer.transformation.base.utils.OperationUtils;
import org.eclipse.papyrus.designer.transformation.library.xtend.BehaviorUtil;
import org.eclipse.papyrus.designer.transformation.tracing.barectf.library.ModelQNames;
import org.eclipse.papyrus.designer.transformation.tracing.barectf.library.flatten.Flattener;
import org.eclipse.papyrus.designer.transformation.tracing.barectf.library.flatten.NameType;
import org.eclipse.papyrus.designer.transformation.tracing.library.EventNames;
import org.eclipse.papyrus.designer.transformation.tracing.library.InstanceNameConfigurator;
import org.eclipse.papyrus.designer.transformation.tracing.library.utils.TraceUtils;
import org.eclipse.papyrus.designer.transformation.tracing.library.utils.TracingUriConstants;
import org.eclipse.papyrus.moka.tracepoint.service.TraceActions;
import org.eclipse.papyrus.uml.tools.utils.PackageUtil;
import org.eclipse.papyrus.uml.tools.utils.StereotypeUtil;
import org.eclipse.uml2.uml.Behavior;
import org.eclipse.uml2.uml.BehavioredClassifier;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Dependency;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.Enumeration;
import org.eclipse.uml2.uml.Event;
import org.eclipse.uml2.uml.Interface;
import org.eclipse.uml2.uml.LiteralString;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.Port;
import org.eclipse.uml2.uml.Profile;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.SignalEvent;
import org.eclipse.uml2.uml.State;
import org.eclipse.uml2.uml.Transition;
import org.eclipse.uml2.uml.Trigger;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.UMLPackage;
import org.eclipse.uml2.uml.ValueSpecification;
import org.eclipse.uml2.uml.profile.standard.Create;
import org.eclipse.uml2.uml.profile.standard.Destroy;
import org.eclipse.uml2.uml.util.UMLUtil;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;

@SuppressWarnings("all")
public class InstrumentCTF {
  private static final String prefix = "barectf_default_trace";
  
  /**
   * Return a list of interfaces provided by ports (which are being traced)
   */
  public static void instrument(final List<Element> events) {
    Element[] _clone = ((Element[])Conversions.unwrapArray(events, Element.class)).clone();
    for (final Element event : _clone) {
      if ((event instanceof Port)) {
        final Port port = ((Port) event);
        final org.eclipse.uml2.uml.Class clazz = port.getClass_();
        EList<Interface> _provideds = port.getProvideds();
        for (final Interface intf : _provideds) {
          EList<Operation> _ownedOperations = intf.getOwnedOperations();
          for (final Operation iOp : _ownedOperations) {
            {
              final Operation cOp = OperationUtils.getSameOperation(iOp, clazz);
              if ((cOp != null)) {
                InstrumentCTF.instrumentOperation(cOp, port);
                boolean _contains = events.contains(cOp);
                if (_contains) {
                  events.remove(cOp);
                }
              }
            }
          }
        }
      }
    }
    Element[] _clone_1 = ((Element[])Conversions.unwrapArray(events, Element.class)).clone();
    for (final Element event_1 : _clone_1) {
      {
        if ((event_1 instanceof Transition)) {
          final Transition transition = ((Transition) event_1);
          Behavior effect = transition.getEffect();
          if ((effect == null)) {
            effect = transition.createEffect(null, UMLPackage.eINSTANCE.getOpaqueBehavior());
          }
          String attrStr = "";
          boolean _traceTransitionAction = TraceUtils.traceTransitionAction(transition, TraceActions.TATransition.TriggerValues);
          if (_traceTransitionAction) {
            EList<Trigger> _triggers = transition.getTriggers();
            for (final Trigger trigger : _triggers) {
              Event _event = trigger.getEvent();
              if ((_event instanceof SignalEvent)) {
                Event _event_1 = trigger.getEvent();
                final SignalEvent sigEvent = ((SignalEvent) _event_1);
                final List<NameType> attrList = Flattener.<Property>flattenParams(sigEvent.getSignal().getAttributes());
                String _attrStr = attrStr;
                StringConcatenation _builder = new StringConcatenation();
                {
                  for(final NameType attr : attrList) {
                    _builder.append(", sig.");
                    String _castAndName = InstrumentCTF.castAndName(attr);
                    _builder.append(_castAndName);
                  }
                }
                attrStr = (_attrStr + _builder);
              }
            }
          }
          StringConcatenation _builder_1 = new StringConcatenation();
          _builder_1.append(InstrumentCTF.prefix);
          _builder_1.append("_");
          CharSequence _transitionEventName = EventNames.transitionEventName(transition);
          _builder_1.append(_transitionEventName);
          _builder_1.append("(BareCTFInit::ctx(), instanceName");
          _builder_1.append(attrStr);
          _builder_1.append(");");
          BehaviorUtil.prefixBody(effect, _builder_1.toString());
          BehavioredClassifier _context = transition.containingStateMachine().getContext();
          final org.eclipse.uml2.uml.Class clazz_1 = ((org.eclipse.uml2.uml.Class) _context);
          InstrumentCTF.addDependencies(clazz_1);
          InstrumentCTF.addInstanceName(clazz_1);
        }
        if ((event_1 instanceof State)) {
          final State state = ((State) event_1);
          boolean trace = false;
          boolean _traceStateAction = TraceUtils.traceStateAction(state, TraceActions.TAState.StateEnter);
          if (_traceStateAction) {
            Behavior entry = state.getEntry();
            if ((entry == null)) {
              entry = state.createEntry(null, UMLPackage.eINSTANCE.getOpaqueBehavior());
            }
            StringConcatenation _builder_2 = new StringConcatenation();
            _builder_2.append(InstrumentCTF.prefix);
            _builder_2.append("_");
            CharSequence _enterStateEventName = EventNames.enterStateEventName(state);
            _builder_2.append(_enterStateEventName);
            _builder_2.append("(BareCTFInit::ctx(), instanceName);");
            BehaviorUtil.prefixBody(entry, _builder_2.toString());
            trace = true;
          }
          boolean _traceStateAction_1 = TraceUtils.traceStateAction(state, TraceActions.TAState.StateLeave);
          if (_traceStateAction_1) {
            Behavior exit = state.getExit();
            if ((exit == null)) {
              exit = state.createExit(null, UMLPackage.eINSTANCE.getOpaqueBehavior());
            }
            StringConcatenation _builder_3 = new StringConcatenation();
            _builder_3.append(InstrumentCTF.prefix);
            _builder_3.append("_");
            CharSequence _exitStateEventName = EventNames.exitStateEventName(state);
            _builder_3.append(_exitStateEventName);
            _builder_3.append("(BareCTFInit::ctx(), instanceName);");
            BehaviorUtil.prefixBody(exit, _builder_3.toString());
            trace = true;
          }
          if (trace) {
            BehavioredClassifier _context_1 = state.containingStateMachine().getContext();
            final org.eclipse.uml2.uml.Class clazz_2 = ((org.eclipse.uml2.uml.Class) _context_1);
            InstrumentCTF.addDependencies(clazz_2);
            InstrumentCTF.addInstanceName(clazz_2);
          }
        }
        if ((event_1 instanceof Operation)) {
          final Operation operation = ((Operation) event_1);
          InstrumentCTF.instrumentOperation(operation, null);
        }
        if ((event_1 instanceof org.eclipse.uml2.uml.Class)) {
          final org.eclipse.uml2.uml.Class clazz_3 = ((org.eclipse.uml2.uml.Class) event_1);
          final IMarker marker = TraceUtils.getMarkerForTraceElement(clazz_3);
          boolean _isActionActive = TraceUtils.isActionActive(marker, TraceActions.TraceFeature.Class, TraceActions.TAClass.Creation.ordinal());
          if (_isActionActive) {
            final Function1<Operation, Boolean> _function = new Function1<Operation, Boolean>() {
              @Override
              public Boolean apply(final Operation it) {
                return Boolean.valueOf(StereotypeUtil.isApplied(it, Create.class));
              }
            };
            final Iterable<Operation> constructors = IterableExtensions.<Operation>filter(clazz_3.getOwnedOperations(), _function);
            boolean _isEmpty = IterableExtensions.isEmpty(constructors);
            if (_isEmpty) {
              Operation ctor = clazz_3.createOwnedOperation(clazz_3.getName(), null, null);
              StereotypeUtil.apply(ctor, Create.class);
              BehaviorUtil.createOpaqueBehavior(clazz_3, ctor);
              InstrumentCTF.instrumentOperation(ctor, null);
              events.add(ctor);
            }
          }
          boolean _isActionActive_1 = TraceUtils.isActionActive(marker, TraceActions.TraceFeature.Class, TraceActions.TAClass.Destruction.ordinal());
          if (_isActionActive_1) {
            final Function1<Operation, Boolean> _function_1 = new Function1<Operation, Boolean>() {
              @Override
              public Boolean apply(final Operation it) {
                return Boolean.valueOf(StereotypeUtil.isApplied(it, Destroy.class));
              }
            };
            final Iterable<Operation> destructors = IterableExtensions.<Operation>filter(clazz_3.getOwnedOperations(), _function_1);
            boolean _isEmpty_1 = IterableExtensions.isEmpty(destructors);
            if (_isEmpty_1) {
              String _name = clazz_3.getName();
              String _plus = ("~" + _name);
              Operation dtor = clazz_3.createOwnedOperation(_plus, null, null);
              StereotypeUtil.apply(dtor, Destroy.class);
              BehaviorUtil.createOpaqueBehavior(clazz_3, dtor);
              InstrumentCTF.instrumentOperation(dtor, null);
              events.add(dtor);
            }
          }
        }
      }
    }
  }
  
  /**
   * Instrument an operation taking a prepare hint into account
   */
  public static void instrumentOperation(final Operation operation, final Port port) {
    int _size = operation.getMethods().size();
    boolean _greaterThan = (_size > 0);
    if (_greaterThan) {
      Behavior method = operation.getMethods().get(0);
      final String prepare = TraceUtils.prepareHint(operation);
      StringConcatenation _builder = new StringConcatenation();
      {
        if ((prepare != null)) {
          _builder.append(prepare);
        }
      }
      _builder.newLineIfNotEmpty();
      _builder.append(InstrumentCTF.prefix);
      _builder.append("_");
      CharSequence _operationStartsEventName = EventNames.operationStartsEventName(operation, port);
      _builder.append(_operationStartsEventName);
      _builder.append("(BareCTFInit::ctx()");
      {
        boolean _needInstanceName = TraceUtils.needInstanceName(operation);
        if (_needInstanceName) {
          _builder.append(", instanceName");
        }
      }
      _builder.newLineIfNotEmpty();
      {
        String _paramsHint = TraceUtils.paramsHint(operation);
        boolean _tripleNotEquals = (_paramsHint != null);
        if (_tripleNotEquals) {
          _builder.append("\t");
          _builder.append(", ");
          String _paramsHint_1 = TraceUtils.paramsHint(operation);
          _builder.append(_paramsHint_1, "\t");
          _builder.newLineIfNotEmpty();
          _builder.append("\t\t\t\t");
        } else {
          boolean _traceOpOrPortAction = TraceUtils.traceOpOrPortAction(operation, port, TraceActions.TAOperation.ParameterValues);
          if (_traceOpOrPortAction) {
            final List<NameType> paramList = Flattener.<Parameter>flattenParams(TraceUtils.getInAndInout(operation));
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
            _builder.append("\t");
            {
              for(final NameType param : paramList) {
                _builder.append(", ");
                String _castAndName = InstrumentCTF.castAndName(param);
                _builder.append(_castAndName, "\t\t");
              }
            }
            _builder.newLineIfNotEmpty();
            _builder.append("\t");
          }
        }
      }
      _builder.append(");");
      BehaviorUtil.prefixBody(method, _builder.toString());
      if ((TraceUtils.traceOpOrPortAction(operation, port, TraceActions.TAOperation.MethodEnds) && (operation.getType() == null))) {
        StringConcatenation _builder_1 = new StringConcatenation();
        _builder_1.append(InstrumentCTF.prefix);
        _builder_1.append("_");
        CharSequence _operationEndsEventName = EventNames.operationEndsEventName(operation, port);
        _builder_1.append(_operationEndsEventName);
        _builder_1.append("(BareCTFInit::ctx()");
        {
          boolean _needInstanceName_1 = TraceUtils.needInstanceName(operation);
          if (_needInstanceName_1) {
            _builder_1.append(", instanceName");
          }
        }
        _builder_1.append(");");
        BehaviorUtil.appendBody(method, _builder_1.toString());
      }
      final org.eclipse.uml2.uml.Class clazz = operation.getClass_();
      InstrumentCTF.addDependencies(clazz);
      InstrumentCTF.addInstanceName(clazz);
    }
  }
  
  /**
   * add (usage) dependency to generated classes BareCTFInit and BareCTF,
   * if not already existent
   */
  public static void addDependencies(final org.eclipse.uml2.uml.Class clazz) {
    NamedElement _qualifiedElementFromRS = ElementUtils.getQualifiedElementFromRS(clazz, ModelQNames.BareCTFInit);
    Classifier initCl = ((Classifier) _qualifiedElementFromRS);
    InstrumentCTF.addDependency(clazz, initCl);
    NamedElement _qualifiedElementFromRS_1 = ElementUtils.getQualifiedElementFromRS(clazz, ModelQNames.BareCTF);
    Classifier barectf = ((Classifier) _qualifiedElementFromRS_1);
    InstrumentCTF.addDependency(clazz, barectf);
  }
  
  /**
   * Add an instance name attribute
   */
  public static void addInstanceName(final org.eclipse.uml2.uml.Class clazz) {
    Property instAttrib = clazz.getOwnedAttribute(InstanceNameConfigurator.PROP_INSTANCE_NAME, null);
    if ((instAttrib == null)) {
      NamedElement _qualifiedElementFromRS = ElementUtils.getQualifiedElementFromRS(clazz, "PrimitiveTypes::String");
      Classifier string = ((Classifier) _qualifiedElementFromRS);
      if ((string == null)) {
        PackageUtil.loadPackage(StdUriConstants.UML_PRIM_TYPES_URI, clazz.eResource().getResourceSet());
        NamedElement _qualifiedElementFromRS_1 = ElementUtils.getQualifiedElementFromRS(clazz, "PrimitiveTypes::String");
        string = ((Classifier) _qualifiedElementFromRS_1);
      }
      if ((string != null)) {
        instAttrib = clazz.createOwnedAttribute(InstanceNameConfigurator.PROP_INSTANCE_NAME, string);
        StereotypeUtil.apply(instAttrib, ConfigurationProperty.class);
        ValueSpecification _createDefaultValue = instAttrib.createDefaultValue(null, null, 
          UMLPackage.eINSTANCE.getLiteralString());
        final LiteralString litString = ((LiteralString) _createDefaultValue);
        litString.setValue(StringUtils.quote(clazz.getName()));
      }
    }
    UseInstanceConfigurator useIConfig = StereotypeUtil.<UseInstanceConfigurator>applyApp(clazz, UseInstanceConfigurator.class);
    if ((useIConfig == null)) {
      final org.eclipse.uml2.uml.Package depProfile = PackageUtil.loadPackage(DepProfileResource.PROFILE_PATH_URI, clazz.eResource().getResourceSet());
      if ((depProfile instanceof Profile)) {
        PackageUtil.getRootPackage(clazz).applyProfile(((Profile) depProfile));
        useIConfig = StereotypeUtil.<UseInstanceConfigurator>applyApp(clazz, UseInstanceConfigurator.class);
      } else {
        throw new RuntimeException("Cannot apply deployment profile");
      }
    }
    PackageUtil.loadPackage(TracingUriConstants.TRACING_LIB_URI, clazz.eResource().getResourceSet());
    NamedElement _qualifiedElementFromRS_2 = ElementUtils.getQualifiedElementFromRS(clazz, 
      "tracing::iconfigurators::InstanceNameConfigurator");
    final Classifier iconfigCl = ((Classifier) _qualifiedElementFromRS_2);
    final InstanceConfigurator iconfig = UMLUtil.<InstanceConfigurator>getStereotypeApplication(iconfigCl, InstanceConfigurator.class);
    useIConfig.setConfigurator(iconfig);
  }
  
  /**
   * add dependency to a passed target class, if not already existent
   */
  public static void addDependency(final org.eclipse.uml2.uml.Class clazz, final Classifier target) {
    if ((target != null)) {
      EList<Dependency> _clientDependencies = clazz.getClientDependencies();
      for (final Dependency dependency : _clientDependencies) {
        boolean _contains = dependency.getSuppliers().contains(target);
        if (_contains) {
          return;
        }
      }
      clazz.createDependency(target);
    }
  }
  
  /**
   * Returns the name, in case of an Enumeration prefixed with a type cast.
   * 
   * Background: Enumerations must be cast to uint8 in order to be compatible
   * with CTF enumeration declarations (which are uints with a CTF mapping
   * declaration)
   * @param te a typed element
   */
  public static String castAndName(final NameType nt) {
    final Type type = nt.getType();
    if ((type instanceof Enumeration)) {
      StringConcatenation _builder = new StringConcatenation();
      _builder.append("(uint8_t) ");
      String _name = nt.getName();
      _builder.append(_name);
      return _builder.toString();
    }
    return nt.getName();
  }
}
