/**
 * Copyright (c) 2016 CEA LIST
 * 
 * All rights reserved. This program and the accompanying materials are
 * made available under the terms of the Eclipse Public License 2.0 which
 * accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *   Shuai Li (CEA LIST) <shuai.li@cea.fr> - Initial API and implementation
 *   Van Cam Pham (CEA LIST) <vancam.pham@cea.fr> - Reverse implementation
 */
package org.eclipse.papyrus.designer.languages.cpp.reverse.reverse;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;
import org.eclipse.cdt.core.dom.ast.IASTCaseStatement;
import org.eclipse.cdt.core.dom.ast.IASTCastExpression;
import org.eclipse.cdt.core.dom.ast.IASTCompoundStatement;
import org.eclipse.cdt.core.dom.ast.IASTDeclSpecifier;
import org.eclipse.cdt.core.dom.ast.IASTDeclaration;
import org.eclipse.cdt.core.dom.ast.IASTDeclarationStatement;
import org.eclipse.cdt.core.dom.ast.IASTDeclarator;
import org.eclipse.cdt.core.dom.ast.IASTDefaultStatement;
import org.eclipse.cdt.core.dom.ast.IASTDoStatement;
import org.eclipse.cdt.core.dom.ast.IASTEqualsInitializer;
import org.eclipse.cdt.core.dom.ast.IASTExpression;
import org.eclipse.cdt.core.dom.ast.IASTExpressionList;
import org.eclipse.cdt.core.dom.ast.IASTExpressionStatement;
import org.eclipse.cdt.core.dom.ast.IASTFieldReference;
import org.eclipse.cdt.core.dom.ast.IASTForStatement;
import org.eclipse.cdt.core.dom.ast.IASTFunctionCallExpression;
import org.eclipse.cdt.core.dom.ast.IASTFunctionDefinition;
import org.eclipse.cdt.core.dom.ast.IASTIdExpression;
import org.eclipse.cdt.core.dom.ast.IASTIfStatement;
import org.eclipse.cdt.core.dom.ast.IASTInitializer;
import org.eclipse.cdt.core.dom.ast.IASTInitializerClause;
import org.eclipse.cdt.core.dom.ast.IASTLiteralExpression;
import org.eclipse.cdt.core.dom.ast.IASTName;
import org.eclipse.cdt.core.dom.ast.IASTNode;
import org.eclipse.cdt.core.dom.ast.IASTReturnStatement;
import org.eclipse.cdt.core.dom.ast.IASTSimpleDeclaration;
import org.eclipse.cdt.core.dom.ast.IASTStatement;
import org.eclipse.cdt.core.dom.ast.IASTSwitchStatement;
import org.eclipse.cdt.core.dom.ast.IASTWhileStatement;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTCatchHandler;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTFunctionCallExpression;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTNewExpression;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTRangeBasedForStatement;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTSimpleTypeConstructorExpression;
import org.eclipse.cdt.core.dom.ast.cpp.ICPPASTTryBlockStatement;
import org.eclipse.cdt.core.model.ITranslationUnit;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.papyrus.designer.languages.cpp.reverse.utils.RoundtripCppUtils;
import org.eclipse.uml2.uml.Classifier;
import org.eclipse.uml2.uml.Element;
import org.eclipse.uml2.uml.NamedElement;
import org.eclipse.uml2.uml.Operation;
import org.eclipse.uml2.uml.Parameter;
import org.eclipse.uml2.uml.ParameterDirectionKind;
import org.eclipse.uml2.uml.Property;
import org.eclipse.uml2.uml.Type;
import org.eclipse.uml2.uml.Usage;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

/**
 * Analysis dependencies in a C++ method body
 */
@SuppressWarnings("all")
public class DependencyAnalysis {
  private Operation m_operation;

  private IASTFunctionDefinition m_definition;

  private ITranslationUnit m_itu;

  private Classifier m_classifier;

  private Map<String, Type> localVaribaleList = new HashMap<String, Type>();

  public DependencyAnalysis(final Operation op, final IASTFunctionDefinition definition, final ITranslationUnit itu) {
    this.m_operation = op;
    this.m_definition = definition;
    this.m_itu = itu;
    EObject _eContainer = op.eContainer();
    if ((_eContainer instanceof Classifier)) {
      EObject _eContainer_1 = op.eContainer();
      this.m_classifier = ((Classifier) _eContainer_1);
    }
  }

  /**
   * Analyze the dependencies of a function definition. For each
   * usage of a type, a dependency (Usage) is created in the nearest package.
   * TODO: there is no redundancy check to remove types already exposed in the operation
   * signature or via attribute types
   */
  public void analyzeDependencies() {
    final IASTStatement body = this.m_definition.getBody();
    if ((body != null)) {
      this.analyzeStatement(body);
    }
  }

  /**
   * Analyze the dependencies of a statement within a body
   */
  private void analyzeStatement(final IASTStatement statement) {
    if ((statement == null)) {
      return;
    }
    boolean _matched = false;
    if ((statement instanceof IASTCompoundStatement)) {
      _matched=true;
      this.analyzeStatement(((IASTCompoundStatement) statement));
    }
    if (!_matched) {
      if ((statement instanceof ICPPASTCatchHandler)) {
        _matched=true;
        this.analyzeStatement(((ICPPASTCatchHandler) statement));
      }
    }
    if (!_matched) {
      if ((statement instanceof ICPPASTRangeBasedForStatement)) {
        _matched=true;
        this.analyzeStatement(((ICPPASTRangeBasedForStatement) statement));
      }
    }
    if (!_matched) {
      if ((statement instanceof ICPPASTTryBlockStatement)) {
        _matched=true;
        this.analyzeStatement(((ICPPASTTryBlockStatement) statement));
      }
    }
    if (!_matched) {
      if ((statement instanceof IASTCaseStatement)) {
        _matched=true;
        this.analyzeStatement(((IASTCaseStatement) statement));
      }
    }
    if (!_matched) {
      if ((statement instanceof IASTDeclarationStatement)) {
        _matched=true;
        this.analyzeStatement(((IASTDeclarationStatement) statement));
      }
    }
    if (!_matched) {
      if ((statement instanceof IASTDefaultStatement)) {
        _matched=true;
        this.analyzeStatement(((IASTDefaultStatement) statement));
      }
    }
    if (!_matched) {
      if ((statement instanceof IASTDoStatement)) {
        _matched=true;
        this.analyzeStatement(((IASTDoStatement) statement));
      }
    }
    if (!_matched) {
      if ((statement instanceof IASTExpressionStatement)) {
        _matched=true;
        this.analyzeStatement(((IASTExpressionStatement) statement));
      }
    }
    if (!_matched) {
      if ((statement instanceof IASTForStatement)) {
        _matched=true;
        this.analyzeStatement(((IASTForStatement) statement));
      }
    }
    if (!_matched) {
      if ((statement instanceof IASTIfStatement)) {
        _matched=true;
        this.analyzeStatement(((IASTIfStatement) statement));
      }
    }
    if (!_matched) {
      if ((statement instanceof IASTReturnStatement)) {
        _matched=true;
        this.analyzeStatement(((IASTReturnStatement) statement));
      }
    }
    if (!_matched) {
      if ((statement instanceof IASTSwitchStatement)) {
        _matched=true;
        this.analyzeStatement(((IASTSwitchStatement) statement));
      }
    }
    if (!_matched) {
      if ((statement instanceof IASTWhileStatement)) {
        _matched=true;
        this.analyzeStatement(((IASTWhileStatement) statement));
      }
    }
  }

  private void analyzeStatement(final IASTCompoundStatement statement) {
    IASTStatement[] statements = ((IASTCompoundStatement) statement).getStatements();
    final IASTStatement[] _converted_statements = (IASTStatement[])statements;
    final Consumer<IASTStatement> _function = new Consumer<IASTStatement>() {
      @Override
      public void accept(final IASTStatement it) {
        DependencyAnalysis.this.analyzeStatement(it);
      }
    };
    ((List<IASTStatement>)Conversions.doWrapArray(_converted_statements)).forEach(_function);
  }

  private void analyzeStatement(final ICPPASTCatchHandler statement) {
  }

  private void analyzeStatement(final ICPPASTRangeBasedForStatement statement) {
  }

  private void analyzeStatement(final ICPPASTTryBlockStatement statement) {
  }

  private void analyzeStatement(final IASTCaseStatement statement) {
  }

  private void analyzeStatement(final IASTDeclarationStatement statement) {
    IASTDeclaration _declaration = statement.getDeclaration();
    if ((_declaration instanceof IASTSimpleDeclaration)) {
      IASTDeclaration _declaration_1 = statement.getDeclaration();
      IASTSimpleDeclaration declaration = ((IASTSimpleDeclaration) _declaration_1);
      if ((declaration != null)) {
        IASTDeclSpecifier specifier = declaration.getDeclSpecifier();
        String typeName = ASTUtils.getCppTypeName(specifier);
        boolean _isPrimitiveCppType = RoundtripCppUtils.isPrimitiveCppType(typeName);
        boolean _not = (!_isPrimitiveCppType);
        if (_not) {
          final Type umlType = ReverseUtils.getUMLType(typeName, this.m_itu);
          if ((umlType != null)) {
            final Consumer<IASTDeclarator> _function = new Consumer<IASTDeclarator>() {
              @Override
              public void accept(final IASTDeclarator it) {
                DependencyAnalysis.this.localVaribaleList.put(it.getName().toString(), umlType);
              }
            };
            ((List<IASTDeclarator>)Conversions.doWrapArray(declaration.getDeclarators())).forEach(_function);
            this.createDependency(umlType);
          }
        }
        final Consumer<IASTDeclarator> _function_1 = new Consumer<IASTDeclarator>() {
          @Override
          public void accept(final IASTDeclarator it) {
            IASTInitializer initilizer = it.getInitializer();
            if ((initilizer != null)) {
              if ((initilizer instanceof IASTEqualsInitializer)) {
                IASTEqualsInitializer equalsInitialiser = ((IASTEqualsInitializer) initilizer);
                IASTInitializerClause clause = equalsInitialiser.getInitializerClause();
                if ((clause instanceof IASTExpression)) {
                  DependencyAnalysis.this.analyzeExpression(((IASTExpression)clause));
                }
              }
            }
          }
        };
        ((List<IASTDeclarator>)Conversions.doWrapArray(declaration.getDeclarators())).forEach(_function_1);
      }
    }
  }

  /**
   * create a dependency (Usage) from the classifier that owns the examined operation to the passed supplier. The relationship
   * is created in the nearest package.
   * @param the target of the usage.
   */
  private void createDependency(final NamedElement supplier) {
    Element _owner = this.m_operation.getOwner();
    if ((_owner instanceof NamedElement)) {
      org.eclipse.uml2.uml.Package nearestPack = this.m_operation.getNearestPackage();
      if ((nearestPack != null)) {
        List<Usage> usages = IterableExtensions.<Usage>toList(Iterables.<Usage>filter(nearestPack.getOwnedElements(), Usage.class));
        final Function1<Usage, Boolean> _function = new Function1<Usage, Boolean>() {
          @Override
          public Boolean apply(final Usage it) {
            return Boolean.valueOf((it.getClients().contains(DependencyAnalysis.this.m_operation.getOwner()) && it.getSuppliers().contains(supplier)));
          }
        };
        Usage usage = IterableExtensions.<Usage>head(IterableExtensions.<Usage>filter(usages, _function));
        if ((usage == null)) {
          Element _owner_1 = this.m_operation.getOwner();
          ((NamedElement) _owner_1).createUsage(supplier);
        }
      }
    }
  }

  private void analyzeStatement(final IASTDefaultStatement statement) {
  }

  private void analyzeStatement(final IASTDoStatement statement) {
    this.analyzeStatement(statement.getBody());
  }

  private void analyzeStatement(final IASTExpressionStatement statement) {
    IASTExpression expression = statement.getExpression();
    this.analyzeExpression(expression);
  }

  private Type getTypeOfVariableOrAttribute(final String attributeName) {
    Type type = this.localVaribaleList.get(attributeName);
    if ((type == null)) {
      final Function1<Property, Boolean> _function = new Function1<Property, Boolean>() {
        @Override
        public Boolean apply(final Property it) {
          return Boolean.valueOf(it.getName().equals(attributeName));
        }
      };
      Property attribute = IterableExtensions.<Property>head(IterableExtensions.<Property>filter(this.m_classifier.getAllAttributes(), _function));
      if ((attribute != null)) {
        type = attribute.getType();
      }
    }
    return type;
  }

  private boolean isSameSignature(final Operation op, final List<Type> l2) {
    final Function1<Parameter, Boolean> _function = new Function1<Parameter, Boolean>() {
      @Override
      public Boolean apply(final Parameter it) {
        ParameterDirectionKind _direction = it.getDirection();
        return Boolean.valueOf((!Objects.equal(_direction, ParameterDirectionKind.RETURN_LITERAL)));
      }
    };
    final Function1<Parameter, Type> _function_1 = new Function1<Parameter, Type>() {
      @Override
      public Type apply(final Parameter it) {
        return it.getType();
      }
    };
    List<Type> params = ListExtensions.<Parameter, Type>map(IterableExtensions.<Parameter>toList(IterableExtensions.<Parameter>filter(op.getOwnedParameters(), _function)), _function_1);
    return this.isSameTypeList(params, l2);
  }

  private boolean isSameTypeList(final List<Type> l1, final List<Type> l2) {
    int _size = l1.size();
    int _size_1 = l2.size();
    boolean _notEquals = (_size != _size_1);
    if (_notEquals) {
      return false;
    }
    for (int i = 0; (i < l1.size()); i++) {
      Type _get = l1.get(i);
      Type _get_1 = l2.get(i);
      boolean _notEquals_1 = (!Objects.equal(_get, _get_1));
      if (_notEquals_1) {
        return false;
      }
    }
    return true;
  }

  /**
   * return a UML type from an AST expression
   */
  private Type analyzeExpression(final IASTExpression expression) {
    if ((expression == null)) {
      return null;
    }
    Type ret = null;
    if ((expression instanceof ICPPASTFunctionCallExpression)) {
      IASTExpression nameExpression = ((ICPPASTFunctionCallExpression)expression).getFunctionNameExpression();
      IASTInitializerClause[] arguments = ((ICPPASTFunctionCallExpression)expression).getArguments();
      Parameter returnParam = ((Parameter) null);
      if ((nameExpression instanceof IASTFieldReference)) {
        final IASTName functionName = ((IASTFieldReference)nameExpression).getFieldName();
        final IASTExpression fieldOwner = ((IASTFieldReference)nameExpression).getFieldOwner();
        Classifier targetClass = null;
        if ((fieldOwner instanceof IASTIdExpression)) {
          Type type = this.getTypeOfVariableOrAttribute(((IASTIdExpression)fieldOwner).getName().toString());
          if ((type instanceof Classifier)) {
            targetClass = ((Classifier)type);
          }
        } else {
          if ((fieldOwner instanceof IASTLiteralExpression)) {
            String _rawSignature = ((IASTLiteralExpression)fieldOwner).getRawSignature();
            boolean _equals = Objects.equal(_rawSignature, "this");
            if (_equals) {
              targetClass = this.m_classifier;
            }
          }
        }
        if ((targetClass != null)) {
          final Function1<Operation, Boolean> _function = new Function1<Operation, Boolean>() {
            @Override
            public Boolean apply(final Operation it) {
              return Boolean.valueOf(it.getName().equals(functionName.toString()));
            }
          };
          final List<Operation> sameNameOps = IterableExtensions.<Operation>toList(IterableExtensions.<Operation>filter(targetClass.getAllOperations(), _function));
          int _size = sameNameOps.size();
          boolean _greaterThan = (_size > 1);
          if (_greaterThan) {
            final IASTInitializerClause[] _converted_arguments = (IASTInitializerClause[])arguments;
            final Function1<IASTInitializerClause, Type> _function_1 = new Function1<IASTInitializerClause, Type>() {
              @Override
              public Type apply(final IASTInitializerClause it) {
                return DependencyAnalysis.this.getArgumentType(it);
              }
            };
            final List<Type> currentArgumentTypes = IterableExtensions.<Type>toList(ListExtensions.<IASTInitializerClause, Type>map(((List<IASTInitializerClause>)Conversions.doWrapArray(_converted_arguments)), _function_1));
            final Function1<Operation, Boolean> _function_2 = new Function1<Operation, Boolean>() {
              @Override
              public Boolean apply(final Operation it) {
                return Boolean.valueOf(DependencyAnalysis.this.isSameSignature(it, currentArgumentTypes));
              }
            };
            Operation targetOp = IterableExtensions.<Operation>head(IterableExtensions.<Operation>filter(sameNameOps, _function_2));
            if ((targetOp != null)) {
              this.createDependency(targetOp);
              final Function1<Parameter, Boolean> _function_3 = new Function1<Parameter, Boolean>() {
                @Override
                public Boolean apply(final Parameter it) {
                  ParameterDirectionKind _direction = it.getDirection();
                  return Boolean.valueOf(Objects.equal(_direction, ParameterDirectionKind.RETURN_LITERAL));
                }
              };
              returnParam = IterableExtensions.<Parameter>head(IterableExtensions.<Parameter>filter(targetOp.getOwnedParameters(), _function_3));
            }
          } else {
            Operation _head = IterableExtensions.<Operation>head(sameNameOps);
            boolean _tripleNotEquals = (_head != null);
            if (_tripleNotEquals) {
              this.createDependency(IterableExtensions.<Operation>head(sameNameOps));
              final Function1<Parameter, Boolean> _function_4 = new Function1<Parameter, Boolean>() {
                @Override
                public Boolean apply(final Parameter it) {
                  ParameterDirectionKind _direction = it.getDirection();
                  return Boolean.valueOf(Objects.equal(_direction, ParameterDirectionKind.RETURN_LITERAL));
                }
              };
              returnParam = IterableExtensions.<Parameter>head(IterableExtensions.<Parameter>filter(IterableExtensions.<Operation>head(sameNameOps).getOwnedParameters(), _function_4));
            }
          }
        }
      }
      if ((returnParam != null)) {
        ret = returnParam.getType();
      }
    } else {
      if ((expression instanceof IASTFunctionCallExpression)) {
      } else {
        if ((expression instanceof ICPPASTNewExpression)) {
          String typeName = ASTUtils.getCppTypeName(((ICPPASTNewExpression)expression).getTypeId().getDeclSpecifier());
          ret = ReverseUtils.getUMLType(typeName, this.m_itu);
        } else {
          if ((expression instanceof ICPPASTSimpleTypeConstructorExpression)) {
          } else {
            if ((expression instanceof IASTCastExpression)) {
              String typeName_1 = ASTUtils.getCppTypeName(((IASTCastExpression)expression).getTypeId().getDeclSpecifier());
              ret = ReverseUtils.getUMLType(typeName_1, this.m_itu);
            } else {
              if ((expression instanceof IASTExpressionList)) {
                final Consumer<IASTExpression> _function_5 = new Consumer<IASTExpression>() {
                  @Override
                  public void accept(final IASTExpression it) {
                    DependencyAnalysis.this.analyzeExpression(it);
                  }
                };
                ((List<IASTExpression>)Conversions.doWrapArray(((IASTExpressionList)expression).getExpressions())).forEach(_function_5);
              } else {
                if (((expression.getChildren() != null) && (((List<IASTNode>)Conversions.doWrapArray(expression.getChildren())).size() > 0))) {
                  final Consumer<IASTNode> _function_6 = new Consumer<IASTNode>() {
                    @Override
                    public void accept(final IASTNode it) {
                      if ((it instanceof IASTExpression)) {
                        DependencyAnalysis.this.analyzeExpression(((IASTExpression)it));
                      }
                    }
                  };
                  ((List<IASTNode>)Conversions.doWrapArray(expression.getChildren())).forEach(_function_6);
                }
              }
            }
          }
        }
      }
    }
    return ret;
  }

  private Type getArgumentType(final IASTInitializerClause argument) {
    Type argType = null;
    if ((argument instanceof IASTExpression)) {
      argType = this.analyzeExpression(((IASTExpression)argument));
    }
    return argType;
  }

  private void analyzeStatement(final IASTForStatement statement) {
  }

  private void analyzeStatement(final IASTReturnStatement statement) {
  }

  private void analyzeStatement(final IASTSwitchStatement statement) {
    IASTStatement body = statement.getBody();
    this.analyzeStatement(body);
  }

  private void analyzeStatement(final IASTWhileStatement statement) {
    this.analyzeStatement(statement.getBody());
  }

  private void analyzeStatement(final IASTIfStatement statement) {
    this.analyzeExpression(statement.getConditionExpression());
    IASTStatement _thenClause = statement.getThenClause();
    boolean _tripleNotEquals = (_thenClause != null);
    if (_tripleNotEquals) {
      this.analyzeStatement(statement.getThenClause());
    }
    IASTStatement _elseClause = statement.getElseClause();
    boolean _tripleNotEquals_1 = (_elseClause != null);
    if (_tripleNotEquals_1) {
      this.analyzeStatement(statement.getElseClause());
    }
  }
}
