/**
 * Copyright (c) 2006, 2020 Borland Software Corporation, CEA LIST, Artal and others
 * 
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License 2.0
 * which accompanies this distribution, and is available at
 * https://www.eclipse.org/legal/epl-2.0/
 * 
 * SPDX-License-Identifier: EPL-2.0
 * 
 * Contributors:
 *    Artem Tikhomirov (Borland) - initial API and implementation
 *    Michael Golubev (Montages) - #386838 - migrate to Xtend2
 *    Aurelien Didier (ARTAL) - aurelien.didier51@gmail.com - Bug 569174
 */
package gmfgraph;

import com.google.common.base.Objects;
import com.google.inject.Inject;
import com.google.inject.Singleton;
import gmfgraph.attr.Figure;
import gmfgraph.attr.Shape;
import java.util.Arrays;
import org.eclipse.emf.common.util.EList;
import org.eclipse.papyrus.gmf.gmfgraph.CustomFigure;
import org.eclipse.papyrus.gmf.gmfgraph.InvisibleRectangle;
import org.eclipse.papyrus.gmf.gmfgraph.Label;
import org.eclipse.papyrus.gmf.gmfgraph.Point;
import org.eclipse.papyrus.gmf.gmfgraph.Polyline;
import org.eclipse.papyrus.gmf.gmfgraph.Rectangle2D;
import org.eclipse.papyrus.gmf.gmfgraph.RoundedRectangle;
import org.eclipse.papyrus.gmf.gmfgraph.SVGFigure;
import org.eclipse.papyrus.gmf.gmfgraph.VerticalLabel;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.xbase.lib.Extension;
import xpt.Common;

/**
 * Defines attributes for Figures (for DecorationFigure subclasses see Decoration.xpt)
 */
@Singleton
@SuppressWarnings("all")
public class Attrs {
  @Inject
  private Shape xptShape;
  
  @Inject
  private Figure xptFigure;
  
  @Inject
  private MapMode xptMapMode;
  
  @Inject
  @Extension
  private Common _common;
  
  /**
   * Polymorphic entry point for any Figure's subclass attributes
   */
  protected CharSequence _Init(final org.eclipse.papyrus.gmf.gmfgraph.Figure it, final String figureVarName) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _figureAttrs = this.xptFigure.figureAttrs(it, figureVarName);
    _builder.append(_figureAttrs);
    _builder.newLineIfNotEmpty();
    CharSequence _extraLineBreak = this._common.extraLineBreak();
    _builder.append(_extraLineBreak);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  protected CharSequence _Init(final org.eclipse.papyrus.gmf.gmfgraph.Shape it, final String figureVarName) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _shapeAttrs = this.xptShape.shapeAttrs(it, figureVarName);
    _builder.append(_shapeAttrs);
    _builder.newLineIfNotEmpty();
    CharSequence _figureAttrs = this.xptFigure.figureAttrs(it, figureVarName);
    _builder.append(_figureAttrs);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  protected CharSequence _Init(final Label it, final String figureVarName) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _extraLineBreak = this._common.extraLineBreak();
    _builder.append(_extraLineBreak);
    _builder.newLineIfNotEmpty();
    _builder.append(figureVarName);
    _builder.append(".setText(\"");
    String _text = it.getText();
    _builder.append(_text);
    _builder.append("\");");
    _builder.newLineIfNotEmpty();
    CharSequence _figureAttrs = this.xptFigure.figureAttrs(it, figureVarName);
    _builder.append(_figureAttrs);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  protected CharSequence _Init(final VerticalLabel it, final String figureVarName) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _extraLineBreak = this._common.extraLineBreak();
    _builder.append(_extraLineBreak);
    _builder.newLineIfNotEmpty();
    _builder.append(figureVarName);
    _builder.append(".setText(\"");
    String _text = it.getText();
    _builder.append(_text);
    _builder.append("\");");
    _builder.newLineIfNotEmpty();
    CharSequence _figureAttrs = this.xptFigure.figureAttrs(it, figureVarName);
    _builder.append(_figureAttrs);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  protected CharSequence _Init(final RoundedRectangle it, final String figureVarName) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(figureVarName);
    _builder.append(".setCornerDimensions(new org.eclipse.draw2d.geometry.Dimension(");
    CharSequence _mapMode = this.xptMapMode.mapMode(Integer.valueOf(it.getCornerWidth()));
    _builder.append(_mapMode);
    _builder.append(", ");
    CharSequence _mapMode_1 = this.xptMapMode.mapMode(Integer.valueOf(it.getCornerHeight()));
    _builder.append(_mapMode_1);
    _builder.append("));");
    _builder.newLineIfNotEmpty();
    CharSequence _shapeAttrs = this.xptShape.shapeAttrs(it, figureVarName);
    _builder.append(_shapeAttrs);
    _builder.newLineIfNotEmpty();
    CharSequence _figureAttrs = this.xptFigure.figureAttrs(it, figureVarName);
    _builder.append(_figureAttrs);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  protected CharSequence _Init(final InvisibleRectangle it, final String figureVarName) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(figureVarName);
    _builder.append(".setOutline(false);");
    _builder.newLineIfNotEmpty();
    _builder.append(figureVarName);
    _builder.append(".setFill(false);");
    _builder.newLineIfNotEmpty();
    _builder.append(figureVarName);
    _builder.append(".setOpaque(false);");
    _builder.newLineIfNotEmpty();
    CharSequence _figureAttrs = this.xptFigure.figureAttrs(it, figureVarName);
    _builder.append(_figureAttrs);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  protected CharSequence _Init(final CustomFigure it, final String figureVarName) {
    StringConcatenation _builder = new StringConcatenation();
    CharSequence _figureAttrs = this.xptFigure.figureAttrs(it, figureVarName);
    _builder.append(_figureAttrs);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  protected CharSequence _Init(final Polyline it, final String figureVarName) {
    StringConcatenation _builder = new StringConcatenation();
    {
      EList<Point> _template = it.getTemplate();
      for(final Point pt : _template) {
        _builder.append(figureVarName);
        _builder.append(".addPoint(new org.eclipse.draw2d.geometry.Point(");
        CharSequence _mapMode = this.xptMapMode.mapMode(pt);
        _builder.append(_mapMode);
        _builder.append("));");
        _builder.newLineIfNotEmpty();
      }
    }
    CharSequence _shapeAttrs = this.xptShape.shapeAttrs(it, figureVarName);
    _builder.append(_shapeAttrs);
    _builder.newLineIfNotEmpty();
    CharSequence _figureAttrs = this.xptFigure.figureAttrs(it, figureVarName);
    _builder.append(_figureAttrs);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  protected CharSequence _Init(final SVGFigure it, final String figureVarName) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append(figureVarName);
    _builder.append(".setURI(\"");
    String _documentURI = it.getDocumentURI();
    _builder.append(_documentURI);
    _builder.append("\");");
    _builder.newLineIfNotEmpty();
    {
      boolean _isNoCanvasWidth = it.isNoCanvasWidth();
      if (_isNoCanvasWidth) {
        _builder.append(figureVarName);
        _builder.append(".setSpecifyCanvasWidth(false);");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      boolean _isNoCanvasHeight = it.isNoCanvasHeight();
      if (_isNoCanvasHeight) {
        _builder.append(figureVarName);
        _builder.append(".setSpecifyCanvasHeight(false);");
        _builder.newLineIfNotEmpty();
      }
    }
    {
      Rectangle2D _areaOfInterest = it.getAreaOfInterest();
      boolean _notEquals = (!Objects.equal(_areaOfInterest, null));
      if (_notEquals) {
        _builder.append(figureVarName);
        _builder.append(".setAreaOfInterest(");
        CharSequence _Instance = this.Instance(it.getAreaOfInterest());
        _builder.append(_Instance);
        _builder.append(");");
        _builder.newLineIfNotEmpty();
      }
    }
    CharSequence _figureAttrs = this.xptFigure.figureAttrs(it, figureVarName);
    _builder.append(_figureAttrs);
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence Instance(final Rectangle2D it) {
    StringConcatenation _builder = new StringConcatenation();
    _builder.append("new java.awt.geom.Rectangle2D.Float(");
    float _x = it.getX();
    _builder.append(_x);
    _builder.append("f, ");
    float _y = it.getY();
    _builder.append(_y);
    _builder.append("f, ");
    float _width = it.getWidth();
    _builder.append(_width);
    _builder.append("f, ");
    float _height = it.getHeight();
    _builder.append(_height);
    _builder.append("f)");
    _builder.newLineIfNotEmpty();
    return _builder;
  }
  
  public CharSequence Init(final org.eclipse.papyrus.gmf.gmfgraph.Figure it, final String figureVarName) {
    if (it instanceof Polyline) {
      return _Init((Polyline)it, figureVarName);
    } else if (it instanceof RoundedRectangle) {
      return _Init((RoundedRectangle)it, figureVarName);
    } else if (it instanceof CustomFigure) {
      return _Init((CustomFigure)it, figureVarName);
    } else if (it instanceof InvisibleRectangle) {
      return _Init((InvisibleRectangle)it, figureVarName);
    } else if (it instanceof Label) {
      return _Init((Label)it, figureVarName);
    } else if (it instanceof SVGFigure) {
      return _Init((SVGFigure)it, figureVarName);
    } else if (it instanceof org.eclipse.papyrus.gmf.gmfgraph.Shape) {
      return _Init((org.eclipse.papyrus.gmf.gmfgraph.Shape)it, figureVarName);
    } else if (it instanceof VerticalLabel) {
      return _Init((VerticalLabel)it, figureVarName);
    } else if (it != null) {
      return _Init(it, figureVarName);
    } else {
      throw new IllegalArgumentException("Unhandled parameter types: " +
        Arrays.<Object>asList(it, figureVarName).toString());
    }
  }
}
