/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: LoadAndSaveOWLExample.java,v 1.1 2007/03/18 09:07:03 lzhang Exp $
 */
package org.eclipse.eodm.owl.resource;

import java.io.File;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;

import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.emf.ecore.resource.impl.ResourceSetImpl;
import org.eclipse.eodm.rdf.rdfweb.Document;
import org.eclipse.eodm.rdf.resource.RDFXMLResource;
import org.eclipse.eodm.rdf.resource.RDFXMLResourceFactoryImpl;
import org.eclipse.eodm.util.Triple;

/**
 * An example showing how to load and save EODM RDFS Model.
 * 
 */
public class LoadAndSaveOWLExample {

    public static void main(String[] args) {
        try {
            // load a RDFS ontology
            Document ontology = load();

            ArrayList export = ontology.getComplementalGraph().exportTriples();
            for (int i = 0; i < export.size(); i++) {
                ((Triple) export.get(i)).print();
            }
            System.out.println("----------------------------------");

            // save the RDFS ontology
            save(ontology);
            // OWLXMLSaver.saveToFile(ontology,"testcase/test.rdf");

        } catch (IOException e) {
            e.printStackTrace();
        }
    }

    /**
     * An example of loading a RDF/XML resource to create a EODM RDFS ontology
     * model.
     * 
     * @return a RDFDescription representing the RDFS ontology model
     * @throws IOException
     *             If I/O errors occured or the input is not in RDF/XML syntax.
     */
    protected static Document load() throws IOException {
        // The following is a sample code for loading a RDF/XML content as a
        // EODM RDFS ontology model.
        // It follows the standard way of loading a EMF resource.

        // create a resource set
        ResourceSet resourceSet = new ResourceSetImpl();

        // register the RDF resource factory for all ".rdf" files/URIs;
        // you can also register for ".xml" or even the wild card "*" extension
        // depending
        // on the URI of the RDF/XML content you are going to load;
        resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
                .put(RDFXMLResource.DEFAULT_RDFXML_EXTENSION,
                        new OWLXMLResourceFactoryImpl());

        // get the URI of the RDFS file.
        // If the content is a file, use the following code
        URI fileURI = URI.createFileURI(new File("testcase/food.rdf")
                .getAbsolutePath());

        // create a resource for this URI.
        Resource resource = resourceSet.createResource(fileURI);

        // load the RDF/XML content as a EODM RDFS ontology model
        resource.load(Collections.EMPTY_MAP);

        // obtain the RDFS ontology
        return (Document) resource.getContents().get(0);
    }

    /**
     * An example of saving an EODM RDFS ontology model
     * 
     * @param ontology
     *            the EODM RDFS ontology model to be saved
     * @throws IOException
     *             If I/O errors occurs
     */
    protected static void save(Document ontology) throws IOException {
        // similar to loading, the saving of an EODM RDFS Model also follows the
        // standard way of saving model elements in EMF.

        // create a resource set
        ResourceSet resourceSet = new ResourceSetImpl();

        // register the RDF resource factory for all ".rdf" files;
        // you can also register for ".xml" or even the wild card "*" extension
        // depending
        // on the URI of the RDF/XML content you are going to save;
        resourceSet.getResourceFactoryRegistry().getExtensionToFactoryMap()
                .put(RDFXMLResource.DEFAULT_RDFXML_EXTENSION,
                        new RDFXMLResourceFactoryImpl());

        // create a file URI for the file to save to
        URI fileURI = URI.createFileURI(new File("testcase/OrderType5.rdf")
                .getAbsolutePath());

        // create a resource for this URI
        Resource resource = resourceSet.createResource(fileURI);

        // add RDFS ontology -- a RDFDescription object, to the resource
        resource.getContents().add(ontology);

        // optionally, before save, you can specify an encoding of the saved
        // file
        HashMap options = new HashMap();
        options.put(RDFXMLResource.OPTION_ENCODING, "UTF-8");

        // save the ontology to the file
        resource.save(options);
    }
}
