/*******************************************************************************
 * Copyright (c) 2008, 2009 Oracle. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0, which accompanies this distribution
 * and is available at http://www.eclipse.org/legal/epl-v10.html.
 * 
 * Contributors:
 *     Oracle - initial API and implementation
 ******************************************************************************/
package org.eclipse.jpt.eclipselink.core.internal.resource.java.source;

import org.eclipse.jdt.core.dom.CompilationUnit;
import org.eclipse.jpt.core.internal.resource.java.source.SourceAnnotation;
import org.eclipse.jpt.core.internal.utility.jdt.ConversionDeclarationAnnotationElementAdapter;
import org.eclipse.jpt.core.internal.utility.jdt.MemberIndexedAnnotationAdapter;
import org.eclipse.jpt.core.internal.utility.jdt.NestedIndexedDeclarationAnnotationAdapter;
import org.eclipse.jpt.core.internal.utility.jdt.ShortCircuitAnnotationElementAdapter;
import org.eclipse.jpt.core.resource.java.NestableAnnotation;
import org.eclipse.jpt.core.utility.TextRange;
import org.eclipse.jpt.core.utility.jdt.AnnotationElementAdapter;
import org.eclipse.jpt.core.utility.jdt.DeclarationAnnotationAdapter;
import org.eclipse.jpt.core.utility.jdt.DeclarationAnnotationElementAdapter;
import org.eclipse.jpt.core.utility.jdt.IndexedAnnotationAdapter;
import org.eclipse.jpt.core.utility.jdt.IndexedDeclarationAnnotationAdapter;
import org.eclipse.jpt.core.utility.jdt.Member;
import org.eclipse.jpt.eclipselink.core.resource.java.ConversionValueAnnotation;
import org.eclipse.jpt.eclipselink.core.resource.java.EclipseLinkJPA;
import org.eclipse.jpt.eclipselink.core.resource.java.NestableConversionValueAnnotation;
import org.eclipse.jpt.eclipselink.core.resource.java.ObjectTypeConverterAnnotation;

/**
 * org.eclipse.persistence.annotations.ConversionValue
 */
final class SourceConversionValueAnnotation
	extends SourceAnnotation<Member>
	implements NestableConversionValueAnnotation
{
	private final DeclarationAnnotationElementAdapter<String> dataValueDeclarationAdapter;
	private final AnnotationElementAdapter<String> dataValueAdapter;
	private String dataValue;

	private final DeclarationAnnotationElementAdapter<String> objectValueDeclarationAdapter;
	private final AnnotationElementAdapter<String> objectValueAdapter;
	private String objectValue;


	SourceConversionValueAnnotation(ObjectTypeConverterAnnotation parent, Member member, IndexedDeclarationAnnotationAdapter idaa) {
		super(parent, member, idaa, new MemberIndexedAnnotationAdapter(member, idaa));
		this.dataValueDeclarationAdapter = this.buildDataValueAdapter(idaa);
		this.dataValueAdapter = this.buildAdapter(this.dataValueDeclarationAdapter);
		this.objectValueDeclarationAdapter = this.buildObjectValueAdapter(idaa);
		this.objectValueAdapter = this.buildAdapter(this.objectValueDeclarationAdapter);
	}

	private AnnotationElementAdapter<String> buildAdapter(DeclarationAnnotationElementAdapter<String> daea) {
		return new ShortCircuitAnnotationElementAdapter<String>(this.member, daea);
	}

	private DeclarationAnnotationElementAdapter<String> buildDataValueAdapter(DeclarationAnnotationAdapter declarationAnnotationAdapter) {
		return ConversionDeclarationAnnotationElementAdapter.forStrings(declarationAnnotationAdapter, EclipseLinkJPA.CONVERSION_VALUE__DATA_VALUE, false);
	}

	private DeclarationAnnotationElementAdapter<String> buildObjectValueAdapter(DeclarationAnnotationAdapter declarationAnnotationAdapter) {
		return ConversionDeclarationAnnotationElementAdapter.forStrings(declarationAnnotationAdapter, EclipseLinkJPA.CONVERSION_VALUE__OBJECT_VALUE, false);
	}

	public String getAnnotationName() {
		return ANNOTATION_NAME;
	}

	public void initialize(CompilationUnit astRoot) {
		this.dataValue = this.buildDataValue(astRoot);
		this.objectValue = this.buildObjectValue(astRoot);
	}

	public void update(CompilationUnit astRoot) {
		this.setDataValue(this.buildDataValue(astRoot));
		this.setObjectValue(this.buildObjectValue(astRoot));
	}

	public IndexedAnnotationAdapter getIndexedAnnotationAdapter() {
		return (IndexedAnnotationAdapter) this.annotationAdapter;
	}

	@Override
	public void toString(StringBuilder sb) {
		sb.append(this.dataValue);
		sb.append("=>"); //$NON-NLS-1$
		sb.append(this.objectValue);
	}


	// ********** ConversionValueAnnotation implementation **********

	// ***** data value
	public String getDataValue() {
		return this.dataValue;
	}

	public void setDataValue(String dataValue) {
		if (this.attributeValueHasNotChanged(this.dataValue, dataValue)) {
			return;
		}
		String old = this.dataValue;
		this.dataValue = dataValue;
		this.dataValueAdapter.setValue(dataValue);
		this.firePropertyChanged(DATA_VALUE_PROPERTY, old, dataValue);
	}

	private String buildDataValue(CompilationUnit astRoot) {
		return this.dataValueAdapter.getValue(astRoot);
	}

	public TextRange getDataValueTextRange(CompilationUnit astRoot) {
		return this.getElementTextRange(this.dataValueDeclarationAdapter, astRoot);
	}

	// ***** object value
	public String getObjectValue() {
		return this.objectValue;
	}

	public void setObjectValue(String objectValue) {
		if (this.attributeValueHasNotChanged(this.objectValue, objectValue)) {
			return;
		}
		String old = this.objectValue;
		this.objectValue = objectValue;
		this.objectValueAdapter.setValue(objectValue);
		this.firePropertyChanged(OBJECT_VALUE_PROPERTY, old, objectValue);
	}

	private String buildObjectValue(CompilationUnit astRoot) {
		return this.objectValueAdapter.getValue(astRoot);
	}

	public TextRange getObjectValueTextRange(CompilationUnit astRoot) {
		return this.getElementTextRange(this.objectValueDeclarationAdapter, astRoot);
	}


	//************ NestableAnnotation implementation

	public void initializeFrom(NestableAnnotation oldAnnotation) {
		ConversionValueAnnotation oldConversionValue = (ConversionValueAnnotation) oldAnnotation;
		this.setDataValue(oldConversionValue.getDataValue());
		this.setObjectValue(oldConversionValue.getObjectValue());
	}

	public void moveAnnotation(int newIndex) {
		this.getIndexedAnnotationAdapter().moveAnnotation(newIndex);
	}


	// ********** static methods **********

	static NestableConversionValueAnnotation createConversionValue(ObjectTypeConverterAnnotation parent, Member member, DeclarationAnnotationAdapter daa, int index) {
		return new SourceConversionValueAnnotation(parent, member, buildConversionValueAnnotationAdapter(daa, index));
	}

	private static IndexedDeclarationAnnotationAdapter buildConversionValueAnnotationAdapter(DeclarationAnnotationAdapter daa, int index) {
		return new NestedIndexedDeclarationAnnotationAdapter(daa, EclipseLinkJPA.OBJECT_TYPE_CONVERTER__CONVERSION_VALUES, index, EclipseLinkJPA.CONVERSION_VALUE, false);
	}

}
