package migrate

import (
	"bufio"
	"context"
	"fmt"
	"log/slog"
	"os"
	"path/filepath"

	"git.sr.ht/~whynothugo/ImapGoose/internal/config"
	"git.sr.ht/~whynothugo/ImapGoose/internal/imap"
	"git.sr.ht/~whynothugo/ImapGoose/internal/maildir"
	"git.sr.ht/~whynothugo/ImapGoose/internal/status"
)

// MigrateAccount migrates a single account from v1 to v2 storage format.
func MigrateAccount(ctx context.Context, account config.Account, logger *slog.Logger) error {
	logger = logger.With("account", account.Name)
	logger.Info("Starting migration for account")

	fmt.Fprintf(os.Stderr, "\nWARNING: Ensure that no version of ImapGoose is currently operating on any local Maildir.\n")
	fmt.Fprintf(os.Stderr, "Press Enter to continue...")
	if _, err := bufio.NewReader(os.Stdin).ReadBytes('\n'); err != nil {
		return fmt.Errorf("failed to read confirmation (is stdin available?): %w", err)
	}

	statusRepo := status.New(account.Name)
	if err := statusRepo.Init(); err != nil {
		return fmt.Errorf("failed to initialize status repository: %w", err)
	}
	defer func() {
		_ = statusRepo.Close()
	}()

	version, err := statusRepo.GetVersion()
	if err != nil {
		return fmt.Errorf("failed to get status repository version: %w", err)
	}

	if version == 2 {
		logger.Info("Account already migrated to v2", "account", account.Name)
		return nil
	}

	if version != 1 {
		return fmt.Errorf("unsupported schema version %d (expected 1 or 2)", version)
	}

	if err := account.ResolvePassword(); err != nil {
		return fmt.Errorf("failed to resolve password: %w", err)
	}

	logger.Info("Connecting to IMAP server", "server", account.Server)
	client, err := imap.Connect(ctx, account.Server, account.Username, account.Password, account.Plaintext, nil, logger)
	if err != nil {
		return fmt.Errorf("failed to connect to IMAP server: %w", err)
	}
	defer func() {
		_ = client.Close()
	}()

	mailboxes, err := client.ListMailboxes(ctx)
	if err != nil {
		return fmt.Errorf("failed to list mailboxes: %w", err)
	}

	logger.Info("Discovered mailboxes", "count", len(mailboxes))

	if err := validateMailboxes(mailboxes); err != nil {
		return err
	}

	for _, mailbox := range mailboxes {
		if err := migrateMailbox(account.LocalPath, mailbox, logger); err != nil {
			// Hopefully won't happen to any of the 3 users using ImapGoose
			return fmt.Errorf("failed to migrate mailbox %q: %w — KEEP THE ENTIRE LOG OUTPUT TO UNDERSTAND THE STATE OF THE MIGRATION", mailbox, err)
		}
	}

	if err := statusRepo.SetSchemaVersion(2); err != nil {
		return fmt.Errorf("failed to update schema version: %w", err)
	}

	logger.Info("Migration completed successfully", "account", account.Name)
	return nil
}

// validateMailboxes checks for prohibited names and ambiguous v1 mappings.
func validateMailboxes(mailboxes []string) error {
	for _, mailbox := range mailboxes {
		if err := maildir.ValidateMailboxName(mailbox); err != nil {
			return err
		}
	}

	v1Paths := make(map[string][]string)
	for _, mailbox := range mailboxes {
		v1Path := maildir.GetLegacyMaildirPath("", mailbox)
		v1Paths[v1Path] = append(v1Paths[v1Path], mailbox)
	}

	for v1Path, mailboxList := range v1Paths {
		if len(mailboxList) > 1 {
			return fmt.Errorf("ambiguous v1 mapping detected: mailboxes %v all map to V1 path %q", mailboxList, v1Path)
		}
	}

	return nil
}

// migrateMailbox migrates a single mailbox from v1 to v2 path.
func migrateMailbox(basePath, mailbox string, logger *slog.Logger) error {
	logger.Info("Migrating mailbox", "mailbox", mailbox)
	v1Path := maildir.GetLegacyMaildirPath(basePath, mailbox)
	v2Path := filepath.Join(basePath, mailbox)

	if v1Path == v2Path {
		logger.Debug("v1 and v2 paths identical, no migration needed", "mailbox", mailbox)
		return nil
	}

	if _, err := os.Stat(v1Path); err != nil {
		if os.IsNotExist(err) {
			logger.Debug("Mailbox not present locally, skipping", "mailbox", mailbox)
			return nil
		}
		// No idea what to do here, hopefully this will never happen.
		return fmt.Errorf("failed to stat v1 path %q: %w", v1Path, err)
	}

	if _, err := os.Stat(v2Path); err == nil {
		// This should REALLY not happen; validation should have picked this up.
		return fmt.Errorf("v2 path %q already exists, cannot migrate", v2Path)
	} else if !os.IsNotExist(err) {
		return fmt.Errorf("failed to stat v2 path %q: %w", v2Path, err)
	}

	v2Parent := filepath.Dir(v2Path)
	if err := os.MkdirAll(v2Parent, 0700); err != nil {
		return fmt.Errorf("failed to create parent directory %q: %w", v2Parent, err)
	}

	logger.Info("Migrating mailbox", "mailbox", mailbox, "from", v1Path, "to", v2Path)
	if err := os.Rename(v1Path, v2Path); err != nil {
		return fmt.Errorf("failed to rename %q to %q: %w", v1Path, v2Path, err)
	}

	logger.Info("Successfully migrated mailbox", "mailbox", mailbox)
	return nil
}
