package sync

import (
	"bytes"
	"context"
	"fmt"
	"os"

	"git.sr.ht/~whynothugo/ImapGoose/internal/maildir"
	imap2 "github.com/emersion/go-imap/v2"
)

// downloadMessage downloads a single message from IMAP and stores it locally.
// PRECONDITION: the IMAP mailbox must be selected.
func (s *Syncer) downloadMessage(ctx context.Context, uid imap2.UID) error {
	msg, err := s.client.FetchMessage(ctx, uid)
	if err != nil {
		return fmt.Errorf("failed to fetch message %d: %w", uid, err)
	}

	filename, err := s.maildir.Add(uid, msg.Flags, msg.Body)
	if err != nil {
		return fmt.Errorf("failed to add message %d: %w", uid, err)
	}

	if err := s.status.Add(s.mailbox, uid, filename, msg.Flags); err != nil {
		return fmt.Errorf("failed to update status: %w", err)
	}

	s.logger.Info("Downloaded message", "uid", uid, "size", len(msg.Body))
	return nil
}

// tryDeduplicate attempts to deduplicate a local message using its tentative UID hint.
// Returns true if the message already exists in the remote.
// PRECONDITION: the IMAP mailbox must be selected.
func (s *Syncer) tryDeduplicate(
	ctx context.Context,
	filename string,
	tentativeUID imap2.UID,
	localFlags []imap2.Flag,
	localBody []byte,
) (bool, error) {
	s.logger.Info("Attempting deduplication using U= hint", "tentative_uid", tentativeUID)

	remoteMsg, err := s.client.FetchMessage(ctx, tentativeUID)
	if err != nil {
		s.logger.Debug("Could not fetch remote message with UID.", "tentative_uid", tentativeUID, "error", err)
		return false, nil
	}

	if !bytes.Equal(localBody, remoteMsg.Body) {
		s.logger.Debug("Message body differs from remote.", "tentative_uid", tentativeUID)
		return false, nil
	}

	if !flagsEqual(localFlags, remoteMsg.Flags) {
		// TODO: we likely should consider this the same message,
		// but it's unclear which flags we should keep.
		s.logger.Debug(
			"Message flags differ from remote, uploading as new message",
			"tentative_uid", tentativeUID,
			"local_flags", localFlags,
			"remote_flags", remoteMsg.Flags,
		)
		return false, nil
	}

	if err := s.status.Add(s.mailbox, tentativeUID, filename, remoteMsg.Flags); err != nil {
		return false, fmt.Errorf("failed to add deduplicated message to status: %w", err)
	}

	s.logger.Info("Message exists remotely and is identical", "uid", tentativeUID, "size", len(localBody))
	return true, nil
}

// uploadMessage uploads a single local message to IMAP.
// PRECONDITION: the IMAP mailbox must be selected.
func (s *Syncer) uploadMessage(ctx context.Context, filename string, localMsg maildir.Message) error {
	body, err := s.maildir.ReadFile(localMsg.Filename)
	if err != nil {
		return fmt.Errorf("failed to read local message %s: %w", localMsg.Filename, err)
	}

	// Optimization: If local file has U= (UID hint), check for a duplicate.
	tentativeUID := s.maildir.ExtractTentativeUID(filename)
	if tentativeUID != nil {
		deduplicated, err := s.tryDeduplicate(ctx, filename, *tentativeUID, localMsg.Flags, body)
		if err != nil {
			s.logger.Warn("Duplicate detection failed.", "tentative_uid", *tentativeUID, "error", err)
		} else if deduplicated {
			// All done (tryDeduplicate adds message to status).
			return nil
		}
	}

	newUID, err := s.client.AppendMessage(ctx, s.mailbox, localMsg.Flags, body)
	if err != nil {
		return fmt.Errorf("failed to append message: %w", err)
	}

	if err := s.status.Add(s.mailbox, newUID, filename, localMsg.Flags); err != nil {
		return fmt.Errorf("failed to update status: %w", err)
	}

	s.logger.Info("Uploaded message", "uid", newUID, "size", len(body))
	return nil
}

// updateRemoteFlags updates flags for a single message on IMAP and in status.
// PRECONDITION: the IMAP mailbox must be selected.
func (s *Syncer) updateRemoteFlags(ctx context.Context, uid imap2.UID, newFilename string, oldFlags, newFlags []imap2.Flag) error {
	toAdd := flagDifference(newFlags, oldFlags)
	toRemove := flagDifference(oldFlags, newFlags)

	s.logger.Info("Flag change detected, updating remote", "uid", uid, "new_filename", newFilename)

	resultFlags, err := s.client.UpdateFlags(ctx, uid, toAdd, toRemove)
	if err != nil {
		return fmt.Errorf("failed to update remote flags for UID %d: %w", uid, err)
	}

	// We might have merged local changes into diverging remote changes.
	// Propagate those remote changes back to local, if any.
	if !flagsEqual(resultFlags, newFlags) {
		s.logger.Info("Remote has diverging flags, syncing to local", "result_flags", resultFlags, "expected_flags", newFlags)
		toAddLocal := flagDifference(resultFlags, newFlags)
		toRemoveLocal := flagDifference(newFlags, resultFlags)

		absPath, err := s.maildir.ResolveAbsPath(newFilename)
		if err != nil {
			return fmt.Errorf("failed to resolve absolute path: %w", err)
		}

		finalFilename, finalFlags, err := s.maildir.UpdateFlags(absPath, toAddLocal, toRemoveLocal)
		if err != nil {
			return fmt.Errorf("failed to sync diverging flags to local: %w", err)
		}

		newFilename = finalFilename
		resultFlags = finalFlags
	}

	if err := s.status.UpdateFlags(s.mailbox, uid, newFilename, resultFlags); err != nil {
		return fmt.Errorf("failed to update status flags for UID %d: %w", uid, err)
	}

	s.logger.Info("Updated flags to remote", "uid", uid, "added", toAdd, "removed", toRemove, "result_flags", resultFlags)
	return nil
}

// updateLocalFlags updates flags for a single Maildir message and its status.
// PRECONDITION: the IMAP mailbox must be selected.
func (s *Syncer) updateLocalFlags(ctx context.Context, uid imap2.UID, filename string, oldFlags, newFlags []imap2.Flag) error {
	s.logger.Info("Flag change detected, updating local", "filename", filename)

	toAdd := flagDifference(newFlags, oldFlags)
	toRemove := flagDifference(oldFlags, newFlags)

	absPath, err := s.maildir.ResolveAbsPath(filename)
	if err != nil {
		return fmt.Errorf("failed to resolve absolute path: %w", err)
	}

	newFilename, actualFlags, err := s.maildir.UpdateFlags(absPath, toAdd, toRemove)
	if err != nil {
		return fmt.Errorf("failed to update local flags for UID %d: %w", uid, err)
	}

	// We might have merged remote changes into diverging local changes.
	// Propagate those local changes back to remote, if any.
	if !flagsEqual(actualFlags, newFlags) {
		s.logger.Info("Local has diverging flags, syncing to remote", "actual_flags", actualFlags, "expected_flags", newFlags)
		toAddRemote := flagDifference(actualFlags, newFlags)
		toRemoveRemote := flagDifference(newFlags, actualFlags)

		finalFlags, err := s.client.UpdateFlags(ctx, uid, toAddRemote, toRemoveRemote)
		if err != nil {
			return fmt.Errorf("failed to sync diverging flags to remote: %w", err)
		}

		actualFlags = finalFlags
	}

	if err := s.status.UpdateFlags(s.mailbox, uid, newFilename, actualFlags); err != nil {
		return fmt.Errorf("failed to update status flags for UID %d: %w", uid, err)
	}

	s.logger.Info("Updated flags from remote", "uid", uid, "added", toAdd, "removed", toRemove, "new_filename", newFilename)
	return nil
}

// deleteRemoteMessage deletes a single message from IMAP and removes it from status.
// PRECONDITION: the IMAP mailbox must be selected.
func (s *Syncer) deleteRemoteMessage(ctx context.Context, uid imap2.UID) error {
	if err := s.client.DeleteMessage(ctx, uid); err != nil {
		return fmt.Errorf("failed to delete remote message %d: %w", uid, err)
	}

	if err := s.status.Remove(s.mailbox, uid); err != nil {
		return fmt.Errorf("failed to update status: %w", err)
	}

	s.logger.Info("Deleted remote message", "uid", uid)
	return nil
}

// deleteLocalMessage deletes a message from the filesystem.
// Returns true if the message was actually deleted, false if it was already deleted.
// Does not interact with any IMAP mailbox.
func (s *Syncer) deleteLocalMessage(ctx context.Context, uid imap2.UID) (bool, error) {
	statusMsg, err := s.status.GetByUID(ctx, s.mailbox, uid)
	if err != nil {
		return false, fmt.Errorf("failed to get status for UID %d: %w", uid, err)
	}
	if statusMsg == nil {
		s.logger.Debug("Message was already deleted.", "uid", uid)
		return false, nil
	}
	absPath, err := s.maildir.ResolveAbsPath(statusMsg.Filename)
	if err != nil {
		if !os.IsNotExist(err) {
			return false, fmt.Errorf("failed to resolve absolute path for message %s: %w", statusMsg.Filename, err)
		}
		s.logger.Debug("Local message file already deleted", "uid", uid)
	} else if err := s.maildir.Delete(absPath); err != nil {
		if !os.IsNotExist(err) {
			return false, fmt.Errorf("failed to delete local message %d: %w", uid, err)
		}
		s.logger.Debug("Local message file already deleted", "uid", uid)
	} else {
		s.logger.Info("Deleted local message", "uid", uid)
	}

	if err := s.status.Remove(s.mailbox, uid); err != nil {
		return false, fmt.Errorf("failed to update status: %w", err)
	}
	return true, nil
}
